/**************************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2008 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : Sysc_test.cpp
*  
*	File Description : This file implements the functons for System Controller(SYSCON) Unit test.
*
*
**************************************************************************************/

#include <stdlib.h>

#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"

#include "intc.h"
#include "gpio.h"
#include "sysc.h"
#include "dmc.h"
#include "rtc.h"
#include "wdt.h"
#include "timer.h"
#include "sdhc.h"
//#include "i2s1.h"
#include "keypad.h"
#include "modemif.h"
#include "adcts.h"
#include "sys_timer.h"
#include "otgdev.h"
#include "dma.h"


#ifdef DBG_SYSC
	#define DbgSysc(x) Dbg x
#else
	#define DbgSysc(x) 0;
#endif


extern volatile u32 g_uIntCounter0;
volatile ADCTS_oInform g_oADCTSIn;

///// CMU Test
void SYSCT_TestPllBasic(void);
void SYSCT_TestPllRunningChange(void);
void SYSCT_TestPllLockTime(void);

void SYSCT_TestClkMuxControl(void);
void SYSCT_TestClkDiv(void);
void SYSCT_SelectClkGating(void);
void SYSCT_TestClkGating(void);

void SYSCT_TestClkout(void);

///// PMU Test
void SYSCT_SelectPwrGating(void);
void SYSCT_TestNormalMode(void);

void SYSCT_TestSimplePowerMode(void);

void SYSCT_SetOptimalStableCounter(void);
void SYSCT_RandomAgingPowerMode(void);
void SYSCT_EarlyWakeup(void);
void SYSCT_CheckAgingWakeupReset(void);

void SYSCT_SelectWakeupSource(POWER_MODE ePwrMode);
void SYSCT_SaveDataToExtSram(u32 uOffset);
void SYSCT_VerifyDataFromExtSram(u32 uOffset);
void SYSCT_TestIdleMode(void);
void SYSCT_TestDeepIdleMode(void);
void SYSCT_TestStopMode(void);
void SYSCT_TestDeepStopMode(void);
void SYSCT_TestSleepMode(void);
void SYSCT_TestReset(void);


					// {fout,m,p,s}
static u32 g_aApllSet[][4]={{800,200,6,1},
							{200,200,6,3},		// MHz
							{400,200,6,2},
							{533,533,24,1},
							{540,90,4,1},
							{555,185,8,1},
							{570,95,4,1},
							{585,195,8,1},
							{600,100,4,1},
							{615,205,8,1},
							{630,105,4,1},
							{645,215,8,1},
							{660,110,4,1},
							{667,389,14,1},
							{675,225,8,1},
							{690,115,4,1},
							{710,355,12,1},
							{720,120,4,1},
							{730,365,12,1},
							{750,125,4,1},
							{760,190,6,1},
							{770,385,12,1},
							{775,323,10,1},
							{780,130,4,1},
							{785,327,10,1},
							{790,395,12,1},
							{795,265,8,1},
							{800,200,6,1},
							{805,805,24,1},
							{810,135,4,1},
							{815,815,24,1},
							{820,205,6,1},
							{825,275,8,1},
							{830,415,12,1},
							{833,347,10,1},
							{840,140,4,1},
							{850,425,12,1},
							{860,215,6,1},
							{870,145,4,1},
							{880,220,6,1},
							{890,445,12,1},
							{900,150,4,1},
							{910,455,12,1},
							{930,155,4,1},
							{950,475,12,1},
							{960,160,4,1},
							{970,485,12,1},
							{975,325,8,1},
							{980,245,6,1},
							{985,985,24,1},
							{990,165,4,1},
							{995,995,24,1},
							{1000,250,6,1},
							{1005,335,8,1},
							{1010,505,12,1},
							{1015,423,10,1},
							{1020,170,4,1},
							{1025,427,10,1},
							{1030,944,22,1},
							{800,200,6,1}};
static u32 g_aMpllSet[][4]={{667,667,12,1},
							{133,266,6,3},	// 133 Mhz
							{166,332,6,3},	// 166
							{200,400,6,3},	// 200
							{266,266,6,2},	// 266
							{333,333,6,2},	// 333
							{400,400,6,2},	// 400
							{604,302,6,1},	// 604	
							{624,312,6,1},	// 624
							{644,322,6,1},	// 644
							{664,332,6,1},	// 664
							{667,667,12,1},	// 667
							{684,342,6,1},	// 684
							{704,352,6,1},	// 704
							{724,362,6,1},	// 724
							{744,372,6,1},	// 744
							{1200,300,6,0},	// 1200
							{1328,332,6,0},	// 1328
							{1992,498,6,0},	// 1992
							{667,667,12,1}};	// Back to 667
static u32 g_aEpllSet[][4]={{96000,96,6,2},
							{32750,131,6,4},// KHz
							{45167,271,18,3},
							{48000,96,6,3},
							{49125,131,8,3},
							{50000,100,6,3},
							{67750,271,12,3},
							{73750,295,12,3},
							{80000,160,6,3},
							{84000,84,6,2},
							{96000,96,6,2},
							{144000,144,6,2},
							{192000,96,6,1},
							{288000,144,6,1},
							{96000,96,6,2}};
static u32 g_aVpllSet[][4]={{54000,108,6,3},
							{27000,108,6,4},	// KHz
							{54000,108,6,3},
							{74167,445,18,3},
							{74250,198,8,3},
							{148500,198,8,2},
							{108000,108,6,2},
							{148333,445,18,2},
							{222545,408,22,1},
							{222750,297,16,1},
							{296667,445,18,1},
							{370909,340,22,0},
							{371000,371,24,0},
							{397000,397,24,0},
							{445091,408,22,0},
							{445500,297,16,0},
							{519273,496,22,0},
							{54000,108,6,3}};
						
//////////////////////////////////////////////
// Clock Management Unit (CMU - SYS OFF region)
//////////////////////////////////////////////

void SYSCT_TestPllBasic(void)
{
	u32 uPLL_TYPE;
	u32 uPllOnOffCnt;
	u32 uLockTime=300;
	u32 uPause;
	u32 uRegClkDiv0;
	u32 uRegClkDiv0_bk;
	u32 aApllClkSet[][2] = {{800,0x1310},
							  {1000,0x1410},
							  {667,0x310},
							  {600,0x210},
							  {533,0x210},
							  {400,0x1110},
							  {200,0x0}};		// unit : MHz
	u32 aMpllClkSet[] = {1992,667,400,266,200,166,133};		// unit : MHz
	u32 aEpllClkSet[] = {96000,48000,73728,67738,49152,45158,32768}; // unit : KHz
	u32 aVpllClkSet[] = {54,108};		// unit : MHz

	u32 uPllNum;
	u32 uApll_bk,uMpll_bk,uEpll_bk,uVpll_bk;
	u32 uTemp;

// 1. Backup current PLL setting value & divider value
	SYSC_UpdateClkInform();
	uApll_bk = g_uAPLL;
	uMpll_bk = g_uMPLL;
	uEpll_bk = g_uEPLL;
	uVpll_bk = g_uVPLL;
	uRegClkDiv0_bk = SYSC_GetClkDivReg(0);

// 2. Select PLL for Test
	Disp("[Pll basic test ]\n");
	Disp("[Observe clock signal on clkout pad by using oscilloscope]\n");
	Disp("\nSelect PLL : 0:APLL(D), 1:MPLL, 2:EPLL, 3:VPLL	  : ");
	uPLL_TYPE = UART_GetIntNum();
	if( (uPLL_TYPE<0)||(uPLL_TYPE>3) ) uPLL_TYPE = 0;

	Disp("Pll On/Off repeat Test? [0:1 time(D),	 2:100000 times]");
	uPllOnOffCnt = UART_GetIntNum();
	if(uPllOnOffCnt!=2) uPllOnOffCnt=1;
	else uPllOnOffCnt=100000;

// 3. PLL setting by selected PMS
	switch(uPLL_TYPE) {
		case eAPLL :
			Disp("[Clkout : Apll/8]\n");
			SYSC_SetLockTime(eAPLL,uLockTime);
			SYSC_SetClkOut(eCLKOUT_FOUTAPLL_DIV4,2);

			Disp("\nSelect APLL(MHz) : 800Mhz(D)\n");
			for(uTemp=0;uTemp<(sizeof(aApllClkSet)/8);uTemp++)
				Disp("%d: %d\n",uTemp,aApllClkSet[uTemp][0]);
			
			uPllNum = UART_GetIntNum();
			if( (uPllNum<0)||(uPllNum>(sizeof(aApllClkSet)/8-1)) ) uPllNum = 0;

			uRegClkDiv0 = (uRegClkDiv0_bk & ~(0xffff)) | aApllClkSet[uPllNum][1];

			SYSC_SetClkMuxState(eCLKMUX_APLL_FINPLL);
			SYSC_StopPLL(eAPLL);
			SYSC_SetClkDivReg(0,uRegClkDiv0);
			SYSC_StartPLLbyFout(eAPLL,aApllClkSet[uPllNum][0]*1000000);
			SYSC_SetClkMuxState(eCLKMUX_APLL_FOUTAPLL);
			break;

		case eMPLL :
			Disp("[Clkout : Mpll/8]\n");
			SYSC_SetLockTime(eMPLL,uLockTime);
			SYSC_SetClkOut(eCLKOUT_FOUTMPLL_DIV2,4);

			Disp("\nSelect MPLL(MHz) : 667Mhz(D)\n");
			for(uTemp=0;uTemp<(sizeof(aMpllClkSet)/4);uTemp++)
				Disp("%d: %d\n",uTemp,aMpllClkSet[uTemp]);
			
			uPllNum = UART_GetIntNum();
			if( (uPllNum<0)||(uPllNum>(sizeof(aMpllClkSet)/4-1)) ) uPllNum = 2;
			
			SYSC_SetClkDiv(eCLKDIV_A2M,2);
			SYSC_SetClkMuxState(eCLKMUX_DSYS_SCLKA2M);
			SYSC_SetClkMuxState(eCLKMUX_PSYS_SCLKA2M);
			SYSC_StopPLL(eMPLL);
			SYSC_StartPLLbyFout(eMPLL,aMpllClkSet[uPllNum]*1000000);
			break;

		case eEPLL :
			Disp("[Clkout : Epll]\n");
			SYSC_SetLockTime(eEPLL,uLockTime);
			SYSC_SetClkOut(eCLKOUT_FOUTEPLL,1);

			Disp("\nSelect EPLL(MHz) : 96Mhz(D)\n");
			for(uTemp=0;uTemp<(sizeof(aEpllClkSet)/4);uTemp++)
				Disp("%d: %d\n",uTemp,aEpllClkSet[uTemp]);
			
			uPllNum = UART_GetIntNum();
			if( (uPllNum<0)||(uPllNum>(sizeof(aEpllClkSet)/4-1)) ) uPllNum = 0;
			
			SYSC_StopPLL(eEPLL);
			SYSC_StartPLLbyFout(eEPLL, aEpllClkSet[uPllNum]*1000);
			break;

		case eVPLL :
			Disp("[Clkout : Vpll]\n");
			SYSC_SetLockTime(eVPLL,uLockTime);
			SYSC_SetClkOut(eCLKOUT_FOUTVPLL,1);

			Disp("\nSelect VPLL(MHz) : 54Mhz(D)\n");
			for(uTemp=0;uTemp<(sizeof(aVpllClkSet)/4);uTemp++)
				Disp("%d: %d\n",uTemp,aVpllClkSet[uTemp]);
			
			uPllNum = UART_GetIntNum();
			if( (uPllNum<0)||(uPllNum>(sizeof(aVpllClkSet)/4-1)) ) uPllNum = 0;

			SYSC_StopPLL(eVPLL);
			SYSC_StartPLLbyFout(eVPLL, aVpllClkSet[uPllNum]*1000000);
	}

// 4. Uart channel resetting for changed PLL
	SYSC_UpdateClkInform();
	UART_InitDebugCh(3, 115200);

// 5. PLL ON/OFF Test 100000 times
	while(uPllOnOffCnt--) {
		
		if( (uPause=UART_GetKey())!=0x0 ) {
			if(uPause=='5') {
				Disp("Test count : %d   Pause\n",uPllOnOffCnt);
				UART_Getc();
			}
			else break;
		}

		if ((uPllOnOffCnt%1000)==0)
		{
			DisplayLED(uPllOnOffCnt/100);
			Disp("Test Count : %d\n", uPllOnOffCnt);
			UART_TxEmpty();
		}

		switch(uPLL_TYPE) {
			case eAPLL :
				SYSC_SetClkMuxState(eCLKMUX_APLL_FINPLL);
				SYSC_StopPLL(eAPLL);
				SYSC_StartPLL(eAPLL);
				SYSC_SetClkMuxState(eCLKMUX_APLL_FOUTAPLL);
				break;
			case eMPLL :
				SYSC_StopPLL(eMPLL);
				SYSC_StartPLL(eMPLL);
				break;
			case eEPLL :
				SYSC_StopPLL(eEPLL);
				SYSC_StartPLL(eEPLL);
				break;
			case eVPLL :
				SYSC_StopPLL(eVPLL);
				SYSC_StartPLL(eVPLL);
		}


	}

// 6. Restore PLL setting & divider value
	switch(uPLL_TYPE) {
		case eAPLL :
			SYSC_SetClkMuxState(eCLKMUX_APLL_FINPLL);
			SYSC_StopPLL(eAPLL);
			SYSC_SetClkDivReg(0,uRegClkDiv0_bk);
			SYSC_StartPLLbyFout(eAPLL,uApll_bk);
			SYSC_SetClkMuxState(eCLKMUX_APLL_FOUTAPLL);
			break;

		case eMPLL :
			SYSC_StopPLL(eMPLL);
			SYSC_SetClkMuxState(eCLKMUX_DSYS_SCLKMPLL);
			SYSC_SetClkMuxState(eCLKMUX_PSYS_SCLKMPLL);
			SYSC_SetClkDivReg(0,uRegClkDiv0_bk);
			SYSC_StartPLLbyFout(eMPLL,uMpll_bk);
			break;

		case eEPLL :
			Disp("[Clkout : Epll]\n");
			SYSC_SetLockTime(eEPLL,uLockTime);
			SYSC_SetClkOut(eCLKOUT_FOUTEPLL,1);
			break;

		case eVPLL :
			Disp("[Clkout : Vpll]\n");
			SYSC_SetLockTime(eVPLL,uLockTime);
			SYSC_SetClkOut(eCLKOUT_FOUTVPLL,1);
	}

// 7. Uart channel resetting for original PLL
	SYSC_UpdateClkInform();
	UART_InitDebugCh(3, 115200);
	
}

void SYSCT_TestPllRunningChange(void)
{
	u32 uPLL_TYPE;
	u32 uLockTime =300;
	u32 uPllUpDn;
	int uTemp,uPause;
	u32 uCnt,uAging,uAgingCnt=100000;
	u32 upreFreq;
	u32 uDivCk200,uDivCk166,uDivCk133;
	u32 uNumPllSet;

	// Select items
	Disp("[Pll Running change test ]\n");
	Disp("[You can observe clock signal on clkout pad by using oscilloscope]\n");
	Disp("\nSelect PLL : 0:APLL[D], 1:MPLL, 2:EPLL, 3:VPLL	 : ");
	uPLL_TYPE = UART_GetIntNum();
	if(uPLL_TYPE > 3) uPLL_TYPE=0;

	Disp("\nFreq changing direction 0:Down[D], 1:Up, 2:Random	 : ");
	uPllUpDn = UART_GetIntNum();
	if((uPllUpDn<1)||(uPllUpDn>2)) uPllUpDn=0;

	Disp("\nPll running change 0:1 time sweep[D], 1:Repeat 100000 times	  : ");
	uAging = UART_GetIntNum();
	if(uAging != 1) uAging=0;
	if(uAging==0) uAgingCnt=1;

	Disp("Lock time range: 300 or others\n");
	Disp("Input Lock time[us]: ");
	uLockTime = UART_GetIntNum();

	// Aging loop
	while(uAgingCnt--) {

		// display at each 1000th test 
		if(uAging)
			if ((uAgingCnt%1000)==0)
			{
				DisplayLED(uAgingCnt/100);
				SYSC_UpdateClkInform();
				UART_InitDebugCh(3, 115200);
				UART_TxEmpty();
				Disp("Test Count : %d\n", uAgingCnt);
				UART_TxEmpty();
			}

		// Exit to any key
		if( (uPause=UART_GetKey())!=0x0 ) 
		{
			if(uPause=='5') 
			{
				Disp("Test count : %d   Pause\n",uAgingCnt);
				UART_Getc();
			}
			else break;
		}

		// Pll change for each PLL
		switch(uPLL_TYPE) 
		{
			case eAPLL :
				if(uAging==0) Disp("[Clkout : Apll/8]\n\n");
				SYSC_SetLockTime(eAPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTAPLL_DIV4,2);

				uNumPllSet = (u32)(sizeof(g_aApllSet)/16);
				uCnt = uNumPllSet;

				// loop for setting all MPS values in PLL table 
				while(uCnt--) 
				{
					// up to down, down to up, random
					if(uPllUpDn==1)
						uTemp = uCnt;
					else if(uPllUpDn==2)
						uTemp = rand()%uNumPllSet;
					else
						uTemp = uNumPllSet-uCnt-1;

					// calculate for bus clock
					upreFreq = SYSC_CalPllFout(eAPLL, g_aApllSet[uTemp][1],g_aApllSet[uTemp][2],g_aApllSet[uTemp][3]);
					if((upreFreq%200000000)>3) 
						uDivCk200 = (u32)(upreFreq/200000000+1); // more divider value
					else
						uDivCk200 = (u32)(upreFreq/200000000);

					// change MSYS source clock to MPLL
					if(g_uAPLL>g_uMPLL) 
					{
						SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKMPLL);
						SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,4);
					}
					else 
					{
						SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,4);
						SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKMPLL);
					}

					// APLL change
					SYSC_StopPLL(eAPLL);
					SYSC_StartPLLbyMPS(eAPLL,g_aApllSet[uTemp][1],g_aApllSet[uTemp][2],g_aApllSet[uTemp][3]); // include update source clock
					
					// change MSYS source clock to APLL
					if(g_uAPLL>g_uMPLL) 
					{
						SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,uDivCk200);
						SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKAPLL);
					}
					else 
					{
						SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKAPLL);
						SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,uDivCk200);
					}

					// display result
					if(uAging==0) 
					{
						SYSC_UpdateClkInform();
						UART_InitDebugCh(3, 115200);
						UART_TxEmpty();
						Disp("################ APLL Test Case[%d]################\n",uTemp);
						Disp(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz PCLK_Psys: %.2fMHz\n",
								(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
						Disp(" APLL: %.2fMHz MPLL: %.2fMHz\n EPLL: %.2fMHz		VPLL: %.2fMHz\n\n",
								(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);
						Disp("\n\n");
					}

				}

				break;

			case eMPLL :
				if(uAging==0) Disp("[Clkout : Mpll/4]\n\n");
				SYSC_SetLockTime(eMPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTMPLL_DIV2,4);
				SYSC_SetClkDiv(eCLKDIV_A2M,2);	// APLL out / 2 = 800Mhz / 2 = 400Mhz

				uNumPllSet = (u32)(sizeof(g_aMpllSet)/16);
				uCnt = uNumPllSet;
				while(uCnt--) {
					if(uPllUpDn==1)
						uTemp = uCnt;
					else if(uPllUpDn==2)
						uTemp = rand()%uNumPllSet;
					else
						uTemp = uNumPllSet-uCnt-1;
					
					upreFreq = SYSC_CalPllFout(eMPLL, g_aMpllSet[uTemp][1],g_aMpllSet[uTemp][2],g_aMpllSet[uTemp][3]);
					if((upreFreq%166000000)>3)
						uDivCk166 = (u32)(upreFreq/166000000+1);
					else
						uDivCk166 = (u32)(upreFreq/166000000);

					if((upreFreq%133000000)>1)
						uDivCk133 = (u32)(upreFreq/133000000+1);
					else
						uDivCk133 = (u32)(upreFreq/133000000);
					
					if(g_uMPLL>(g_uAPLL/2)) {
						SYSC_SetClkMuxState(eCLKMUX_DSYS_SCLKA2M);
						SYSC_SetClkMuxState(eCLKMUX_PSYS_SCLKA2M);
						SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,2);
						SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,2);
						}
					else {
						SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,2);
						SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,2);
						SYSC_SetClkMuxState(eCLKMUX_DSYS_SCLKA2M);
						SYSC_SetClkMuxState(eCLKMUX_PSYS_SCLKA2M);
						}

					SYSC_StopPLL(eMPLL);
					SYSC_StartPLLbyMPS(eMPLL,g_aMpllSet[uTemp][1],g_aMpllSet[uTemp][2],g_aMpllSet[uTemp][3]);

					if(upreFreq>(g_uAPLL/2)) {
						SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,uDivCk166);
						SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,uDivCk133);
						SYSC_SetClkMuxState(eCLKMUX_DSYS_SCLKMPLL);
						SYSC_SetClkMuxState(eCLKMUX_PSYS_SCLKMPLL);
						}
					else {
						SYSC_SetClkMuxState(eCLKMUX_DSYS_SCLKMPLL);
						SYSC_SetClkMuxState(eCLKMUX_PSYS_SCLKMPLL);
						SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,uDivCk166);
						SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,uDivCk133);
						}

					if(uAging==0) {
						SYSC_UpdateClkInform();
						UART_InitDebugCh(3, 115200);
						UART_TxEmpty();
						Disp("################ MPLL Test Case[%d]################\n",uTemp);
						Disp(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz	PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz	PCLK_Psys: %.2fMHz\n",
								(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
						Disp(" APLL: %.2fMHz	MPLL: %.2fMHz\n EPLL: %.2fMHz		VPLL: %.2fMHz\n\n",
								(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);
						Disp("\n\n");
					}
				}
				
				break;

			case eEPLL :
				if(uAging==0) Disp("[Clkout : Epll/2]\n");
				SYSC_SetLockTime(eEPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTEPLL,2);
				
				uNumPllSet = (u32)(sizeof(g_aEpllSet)/16);
				uCnt = uNumPllSet;
				while(uCnt--) {
					if(uPllUpDn==1)
						uTemp = uCnt;
					else if(uPllUpDn==2)
						uTemp = rand()%uNumPllSet;
					else
						uTemp = uNumPllSet-uCnt-1;
					
					SYSC_StopPLL(eEPLL);
					SYSC_StartPLLbyMPS(eEPLL,g_aEpllSet[uTemp][1],g_aEpllSet[uTemp][2],g_aEpllSet[uTemp][3]);

					if(uAging==0) {
						SYSC_UpdateClkInform();
						UART_InitDebugCh(3, 115200);
						UART_TxEmpty();
						Disp("################ EPLL Test Case[%d]################\n",uTemp);
						Disp(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz PCLK_Psys: %.2fMHz\n",
								(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
						Disp(" APLL: %.2fMHz MPLL: %.2fMHz\n EPLL: %.2fMHz		VPLL: %.2fMHz\n\n",
								(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);
						Disp("\n\n");
					}
				}
				break;

			case eVPLL :
				if(uAging==0) Disp("[Clkout : Vpll/4]\n");
				SYSC_SetLockTime(eVPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTVPLL,4);

				uNumPllSet = (u32)(sizeof(g_aVpllSet)/16);
				uCnt = uNumPllSet;
				while(uCnt--) {
					if(uPllUpDn==1)
						uTemp = uCnt;
					else if(uPllUpDn==2)
						uTemp = rand()%uNumPllSet;
					else
						uTemp = uNumPllSet-uCnt-1;
					
					SYSC_StopPLL(eVPLL);
					SYSC_StartPLLbyMPS(eVPLL,g_aVpllSet[uTemp][1],g_aVpllSet[uTemp][2],g_aVpllSet[uTemp][3]);

					if(uAging==0) {
						SYSC_UpdateClkInform();
						UART_InitDebugCh(3, 115200);
						UART_TxEmpty();
						Disp("################ VPLL Test Case[%d]################\n",uTemp);
						Disp(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz PCLK_Psys: %.2fMHz\n",
								(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
						Disp(" APLL: %.2fMHz MPLL: %.2fMHz\n EPLL: %.2fMHz		VPLL: %.2fMHz\n\n",
								(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);
						Disp("\n\n");
						}
				
				}
				break;

			default :
				Disp("Select miss...!!");
		}

	}

	Disp("=========== Test case : only changing S-div ===========\n");
	switch(uPLL_TYPE) {
		case eAPLL :
			Disp("APLL FOUT : 800MHz <--> 400MHz\n\n");
			break;
		case eMPLL :
			Disp("MPLL FOUT : 400MHz <--> 200MHz\n\n");
			break;
		case eEPLL :
			Disp("EPLL FOUT : 96MHz <--> 48MHz\n\n");
			break;
		case eVPLL :
			Disp("VPLL FOUT : 108MHz <--> 54MHz\n\n");
			break;
	}
	
	// Loop for S-div changing test
	if(uAging==1) 
		uAgingCnt = 100000;
	else 
		uAgingCnt = 1;
	while(uAgingCnt--) {

		if ((uAgingCnt%1000)==0)
		{
			DisplayLED(uAgingCnt/100);
			SYSC_UpdateClkInform();
			UART_InitDebugCh(3, 115200);
			UART_TxEmpty();
			Disp("Test Count : %d\n", uAgingCnt);
			UART_TxEmpty();
		}
		
		switch(uPLL_TYPE) {
			case eAPLL :
				SYSC_SetLockTime(eAPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTAPLL_DIV4,2);

				SYSC_SetClkMuxState(eCLKMUX_APLL_FINPLL);
				SYSC_StopPLL(eAPLL);
				SYSC_StartPLLbyMPS(eAPLL,200,6,2);
				SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,2);
				SYSC_SetClkMuxState(eCLKMUX_APLL_FOUTAPLL);
				Delay(10);
				SYSC_SetClkMuxState(eCLKMUX_APLL_FINPLL);
				SYSC_StopPLL(eAPLL);
				SYSC_StartPLLbyMPS(eAPLL,200,6,1);
				SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,4);
				SYSC_SetClkMuxState(eCLKMUX_APLL_FOUTAPLL);
				Delay(10);
				break;

			case eMPLL :
				SYSC_SetLockTime(eMPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTMPLL_DIV2,4);

				SYSC_SetClkMuxState(eCLKMUX_MPLL_FINPLL);
				SYSC_StopPLL(eMPLL);
				SYSC_StartPLLbyMPS(eMPLL,400,6,2);
				SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,3);
				SYSC_SetClkMuxState(eCLKMUX_MPLL_FOUTMPLL);
				Delay(10);
				SYSC_SetClkMuxState(eCLKMUX_MPLL_FINPLL);
				SYSC_StopPLL(eMPLL);
				SYSC_StartPLLbyMPS(eMPLL,400,6,3);
				SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,2);
				SYSC_SetClkMuxState(eCLKMUX_MPLL_FOUTMPLL);
				Delay(10);
				break;
			case eEPLL :
				SYSC_SetLockTime(eEPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTEPLL,2);

				SYSC_StopPLL(eEPLL);
				SYSC_StartPLLbyMPS(eEPLL,96,6,3);
				Delay(10);
				SYSC_StopPLL(eEPLL);
				SYSC_StartPLLbyMPS(eEPLL,96,6,2);
				Delay(10);
				break;
			case eVPLL :
				SYSC_SetLockTime(eVPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTVPLL,2);

				SYSC_StopPLL(eVPLL);
				SYSC_StartPLLbyMPS(eVPLL,108,6,2);
				Delay(10);
				SYSC_StopPLL(eVPLL);
				SYSC_StartPLLbyMPS(eVPLL,108,6,3);
				Delay(10);
				break;
		}

		// Exit to any key
		if( (uPause=UART_GetKey())!=0x0 ) {
			if(uPause=='5') {
				Disp("Test count : %d   Pause\n",uAgingCnt);
				UART_Getc();
			}
			else break;
		}
		
	}

	if(uPLL_TYPE==eMPLL) {
		SYSC_SetClkMuxState(eCLKMUX_MPLL_FINPLL);
		SYSC_StopPLL(eMPLL);
		SYSC_StartPLLbyMPS(eMPLL,667,12,1);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,4);
		SYSC_SetClkMuxState(eCLKMUX_MPLL_FOUTMPLL);
	}

	Disp("\n\n");
	Disp("=========== Test case : Setting same PMS value ===========\n");
	switch(uPLL_TYPE) {
		case eAPLL :
			Disp("APLL FOUT : 800MHz setting repeat 100000\n\n");
			break;
		case eMPLL :
			Disp("MPLL FOUT : 400MHz setting repeat 100000\n\n");
			break;
		case eEPLL :
			Disp("EPLL FOUT : 96MHz setting repeat 100000\n\n");
			break;
		case eVPLL :
			Disp("VPLL FOUT : 108MHz setting repeat 100000\n\n");
			break;
	}


	// Loop for Same MPS setting test
	if(uAging==1) 
		uAgingCnt = 100000;
	else 
		uAgingCnt = 1;
	while(uAgingCnt--) {

		if ((uAgingCnt%1000)==0)
		{
			DisplayLED(uAgingCnt/100);
			SYSC_UpdateClkInform();
			UART_InitDebugCh(3, 115200);
			UART_TxEmpty();
			Disp("Test Count : %d\n", uAgingCnt);
			UART_TxEmpty();
		}

		// defualt MPS value : 800, 667, 96, 54
		switch(uPLL_TYPE) {
			case eAPLL :
				SYSC_SetLockTime(eAPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTAPLL_DIV4,2);
				SYSC_StartPLLbyMPS(eAPLL,200,6,1);
				break;

			case eMPLL :
				SYSC_SetLockTime(eMPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTMPLL_DIV2,4);
				SYSC_StartPLLbyMPS(eMPLL,667,12,1);
				Delay(10);
				break;
			case eEPLL :
				SYSC_SetLockTime(eEPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTEPLL,2);
				SYSC_StartPLLbyMPS(eEPLL,96,6,2);
				break;
			case eVPLL :
				SYSC_SetLockTime(eVPLL,uLockTime);
				SYSC_SetClkOut(eCLKOUT_FOUTVPLL,2);
				SYSC_StartPLLbyMPS(eVPLL,108,6,3);
				break;
		}

		// Exit to any key
		if(UART_GetKey()) break;
		
	}

	// reset normal operating frequency & update console
	SYSC_InitSysClk(eINIT_SYSCLK_NORMAL);
	SYSC_UpdateClkInform();
	UART_InitDebugCh(3, 115200);
}


void SYSCT_TestPllLockTime(void)
{
	u32 uPLL_TYPE;
	u32 uLockTime =300;

	Disp("[Lock time test ]\n");
	Disp("[Observe clock signal on clkout pad by using oscilloscope]\n");
  Disp("\nSelect PLL : 0:APLL, 1:MPLL, 2:EPLL, 3:VPLL	: ");
	uPLL_TYPE = UART_GetIntNum();

	Disp("Lock time range: 300 or others\n");
	Disp("Input Lock time[us]: ");
	uLockTime = UART_GetIntNum();

	switch(uPLL_TYPE) {
		case eAPLL :
			Disp("[Clkout : Apll/8]\n");
			SYSC_SetLockTime(eAPLL,uLockTime);
			SYSC_SetClkMuxState(eCLKMUX_APLL_FINPLL);
			SYSC_SetClkOut(eCLKOUT_FOUTAPLL_DIV4,2);
			SYSC_StopPLL(eAPLL);
			Delay(10);
			SYSC_StartPLLbyMPS(eAPLL, 200, 6, 1);	// 800MHz
			SYSC_SetClkMuxState(eCLKMUX_APLL_FOUTAPLL);
			break;
		case eMPLL :
			Disp("[Clkout : Mpll/4]\n");
			SYSC_SetLockTime(eMPLL,uLockTime);
			SYSC_SetClkDiv(eCLKDIV_A2M,2);
			SYSC_SetClkMuxState(eCLKMUX_DSYS_SCLKA2M);
			SYSC_SetClkMuxState(eCLKMUX_PSYS_SCLKA2M);
			SYSC_SetClkOut(eCLKOUT_FOUTMPLL_DIV2,2);
			SYSC_StopPLL(eMPLL);
			Delay(10);
			SYSC_StartPLLbyFout(eMPLL, 667000000);	// 667MHz
			SYSC_SetClkMuxState(eCLKMUX_DSYS_SCLKMPLL);
			SYSC_SetClkMuxState(eCLKMUX_PSYS_SCLKMPLL);
			break;
		case eEPLL :
			Disp("[Clkout : Epll/2]\n");
			SYSC_SetLockTime(eEPLL,uLockTime);
			SYSC_SetClkOut(eCLKOUT_FOUTEPLL,2);
			SYSC_StopPLL(eEPLL);
			Delay(10);
			SYSC_StartPLLbyFout(eEPLL, 96000000);
			break;
		case eVPLL :
			Disp("[Clkout : Vpll/2]\n");
			SYSC_SetLockTime(eVPLL,uLockTime);
			SYSC_SetClkOut(eCLKOUT_FOUTVPLL,2);
			SYSC_StopPLL(eVPLL);
			SYSC_StartPLLbyFout(eVPLL, 54000000);
			break;
		default :
			Disp("Select miss...!!");
	}
}


void SYSCT_TestPllManChg(void)
{
	PLL_TYPE eSelPllType;
	u32 uSelLockTime=300;
	u32 uSelPllValue;
	u32 uSelFreq, uSelM, uSelP, uSelS;
	u32 uSelDivAPLL, uSelDivA2M, uSelDivHclkMsys, uSelDivPclkMsys,
		uSelDivHclkDsys, uSelDivPclkDsys, uSelDivHclkPsys, uSelDivPclkPsys;
	u32 i,j,uTemp;
	u32 uNumPllSet;
	u32 **paPllSet, *pTemp[100];
	float fTemp;

	// Select PLL type
	Disp("[Pll Manual change]\n");
	Disp("[You can observe clock signal on clkout pad by using oscilloscope]\n");
	Disp("\nSelect PLL : 0:APLL[D], 1:MPLL, 2:EPLL, 3:VPLL	 : ");
	eSelPllType = (PLL_TYPE)UART_GetIntNum();
	if (eSelPllType > 3) eSelPllType=0;

	Disp("Lock time range[us] : 300us or others\n");
	Disp("Input Lock time [D=300] : ");
	uSelLockTime = UART_GetIntNum();
	if (uSelLockTime == (-1)) uSelLockTime = 300;
	
	// Get PLL MPS table
	switch(eSelPllType)
	{
		case eMPLL :
			Disp("\n***** MPLL MPS table *****\n");
			uNumPllSet = (u32)(sizeof(g_aMpllSet)/16); // loop size for table array
			for(i=0;i<uNumPllSet;i++)
				pTemp[i] = g_aMpllSet[i];
			paPllSet = pTemp;
			break;
		case eEPLL :
			Disp("\n***** EPLL MPS table *****\n");
			uNumPllSet = (u32)(sizeof(g_aEpllSet)/16); // loop size for table array
			for(i=0;i<uNumPllSet;i++)
				pTemp[i] = g_aEpllSet[i];
			paPllSet = pTemp;
			break;
		case eVPLL :
			Disp("\n***** VPLL MPS table *****\n");
			uNumPllSet = (u32)(sizeof(g_aVpllSet)/16); // loop size for table array
			for(i=0;i<uNumPllSet;i++)
				pTemp[i] = g_aVpllSet[i];
			paPllSet = pTemp;
			break;
		default : // eAPLL
			Disp("\n***** APLL MPS table *****\n");
			uNumPllSet = (u32)(sizeof(g_aApllSet)/16); // loop size for table array
			for(i=0;i<uNumPllSet;i++)
				pTemp[i] = g_aApllSet[i];
			paPllSet = pTemp;
			break;
	}

	for (i=0;i<uNumPllSet;i++)
	{
		if ((eSelPllType==eAPLL)||(eSelPllType==eMPLL))
			Disp(" %2d : [Freq:%4dMHz], [M:%3d], [P:%2d], [S:%1d]\n", i, paPllSet[i][0], paPllSet[i][1], paPllSet[i][2], paPllSet[i][3]);
		else
			Disp(" %2d : [Freq:%6dKHz], [M:%3d], [P:%2d], [S:%1d]\n", i, paPllSet[i][0], paPllSet[i][1], paPllSet[i][2], paPllSet[i][3]);
	}
	Disp(" Press Enter Key only : Enter M.P.S. values manually \n");
	Disp(" Select number : ");
	uSelPllValue = UART_GetIntNum();

	// enter key for manual M.P.S value
	if (uSelPllValue == (-1)) 
	{
		Disp(" You want to select M.P.S manually!!\n");
		Disp(" M value : ");
		uSelM = UART_GetIntNum();
		Disp(" P value : ");
		uSelP = UART_GetIntNum();
		Disp(" S value : ");
		uSelS = UART_GetIntNum();
		uSelFreq = SYSC_CalPllFout(eSelPllType, uSelM, uSelP, uSelS)/1000; // unit KHz
	}		
	else 
	{
		if (uSelPllValue > uNumPllSet) 
			uSelPllValue = 0; // default value

		if ((eSelPllType==eAPLL)||(eSelPllType==eMPLL))
			uSelFreq = paPllSet[uSelPllValue][0]*1000; // KHz
		else
			uSelFreq = paPllSet[uSelPllValue][0];
		
		uSelM = paPllSet[uSelPllValue][1];
		uSelP = paPllSet[uSelPllValue][2];
		uSelS = paPllSet[uSelPllValue][3];		
	}
	Disp("\n***** You select followed values *****\n [Freq:%7dKHz], [M:%3d], [P:%2d], [S:%1d]\n", uSelFreq, uSelM, uSelP, uSelS);

	// Select divider values
	Disp("\n Divider setting for Bus Clocks!! \n");
	uTemp = 0;
	while(uTemp != 1)
	{
		switch(eSelPllType)
		{
		case eAPLL :
			Disp(" DIV_APLL (1~8,[D=1]) : ");
			uSelDivAPLL = UART_GetIntNum();
			if (uSelDivAPLL == (u32)(-1)) uSelDivAPLL = 1;

			Disp(" DIV_A2M (1~8,[D=5]) : ");
			uSelDivA2M = UART_GetIntNum();
			if (uSelDivA2M == (u32)(-1)) uSelDivA2M = 5;

			Disp(" DIV_HCLK_MSYS (1~8,[D=5]) : ");
			uSelDivHclkMsys = UART_GetIntNum();
			if (uSelDivHclkMsys == (u32)(-1)) uSelDivHclkMsys = 5;

			Disp(" DIV_PCLK_MSYS (1~8,[D=2]) : ");
			uSelDivPclkMsys = UART_GetIntNum();
			if (uSelDivPclkMsys == (u32)(-1)) uSelDivPclkMsys = 2;

			fTemp = (float)uSelFreq/1.0e3;
			Disp(" Target Bus Clock\n ARMCLK: %.2fMHz\n SCLK_A2M: %.2fMHz HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n",
				fTemp/uSelDivAPLL, fTemp/uSelDivA2M, fTemp/uSelDivAPLL/uSelDivHclkMsys, fTemp/uSelDivAPLL/uSelDivHclkMsys/uSelDivPclkMsys);

			Disp(" Is it OK? 1.Yes[D]  2.No : ");
			uTemp = UART_GetIntNum();
			if (uTemp != 2) uTemp = 1;
			break;
		case eMPLL :
			Disp(" DIV_HCLK_DSYS (1~8,[D=4]) : ");
			uSelDivHclkDsys = UART_GetIntNum();
			if (uSelDivHclkDsys == (u32)(-1)) uSelDivHclkDsys = 4;

			Disp(" DIV_PCLK_DSYS (1~8,[D=2]) : ");
			uSelDivPclkDsys = UART_GetIntNum();
			if (uSelDivPclkDsys == (u32)(-1)) uSelDivPclkDsys = 2;

			Disp(" DIV_HCLK_PSYS (1~8,[D=5]) : ");
			uSelDivHclkPsys = UART_GetIntNum();
			if (uSelDivHclkPsys == (u32)(-1)) uSelDivHclkPsys = 5;

			Disp(" DIV_PCLK_PSYS (1~8,[D=2]) : ");
			uSelDivPclkPsys = UART_GetIntNum();
			if (uSelDivPclkPsys == (u32)(-1)) uSelDivPclkPsys = 2;

			fTemp = (float)uSelFreq/1.0e3;
			Disp(" Target Bus Clock\n HCLK_Dsys: %.2fMHz PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz PCLK_Psys: %.2fMHz\n",
				fTemp/uSelDivHclkDsys, fTemp/uSelDivHclkDsys/uSelDivPclkDsys, fTemp/uSelDivHclkPsys, fTemp/uSelDivHclkPsys/uSelDivPclkPsys);

			Disp(" Is it OK? 1.Yes[D]  2.No : ");
			uTemp = UART_GetIntNum();
			if (uTemp != 2) uTemp = 1;
			
			break;
		default : // eEPLL, eVPLL
			uTemp = 1;
			break;
		}			
	}

	///////////// Apply selected values /////////////

	//// Change PLL to Fin ==> for EVT0
	// DRAM refresh setting for 24Mhz
	DMC_SetCh(1);
	DMC_SetAutoRefresh(TRUE, 156); // for 20MHz
	DMC_SetCh(0);
	DMC_SetAutoRefresh(TRUE, 156); // for 20MHz
	// Change Mux for selected PLL
	switch(eSelPllType)
	{
	case eAPLL :
		SYSC_SetClkMuxState(eCLKMUX_APLL_FINPLL);
		break;		
	case eMPLL :
		SYSC_SetClkMuxState(eCLKMUX_MPLL_FINPLL);
		break;
	case eEPLL :
		SYSC_SetClkMuxState(eCLKMUX_EPLL_FINPLL);
		break;
	case eVPLL :
		SYSC_SetClkMuxState(eCLKMUX_VPLL_FINVPLL);
		break;
	default : 
		Assert(0);
		break;
	}
	SYSC_StopPLL(eSelPllType); 
	//// for EVT0

	// Set PLL & Dividers
	SYSC_ChangePllnDiv(eSelPllType, uSelM, uSelP, uSelS, uSelLockTime,
		uSelDivAPLL, uSelDivA2M, uSelDivHclkMsys, uSelDivPclkMsys,
		uSelDivHclkDsys, uSelDivPclkDsys, uSelDivHclkPsys, uSelDivPclkPsys);
	Disp(" Set CLKOUT to your selected pll source.\n");
	Disp("  case MPLL : FOUTmpll/20\n");
	Disp("  case EPLL : FOUTepll/10\n");
	Disp("  case VPLL : FOUTvpll/10\n");
	Disp("  case APLL : FOUTapll/8\n ");

	// Display changed clocks
	Disp("\n***** PLL setting changed as follows *****\n");
	Disp(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz	PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz	PCLK_Psys: %.2fMHz\n",
			(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
	Disp(" APLL: %.2fMHz	MPLL: %.2fMHz\n EPLL: %.2fMHz	VPLL: %.2fMHz\n\n",
			(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);
}


void SYSCT_TestDVFS(void)
{
	u32 uSelLv, uSelAging;
	
	///////// user setting //////////
	u32 uTransBase = 0x80000; // 500KB
	u32 uSrcAddr = _DRAM_BaseAddress+0x01100000;
	u32 uDstAddr = _DRAM_BaseAddress+0x01800000;
	/////////////////////////////

	Disp("Select Aging : 1.General(D), 2.Aging\n");
	uSelAging = UART_GetIntNum();
	if ((uSelAging>2)||(uSelAging<1)) uSelAging = 1;

	// Select DVFS level
	Disp("[DVFS level change test]\n");
	Disp("  Level no.		ARMLCK	HCLKM	PCLKM	HCLKD	PCLKD	HCLKP	PCLKP\n" );
	Disp(" 0:Level_0[D] - 	1000 	200 	100.0 	166.8 	83.4 	133.4 	66.7 \n");
	Disp(" 1:Level_1 - 		800 	200 	100.0 	166.8 	83.4 	133.4 	66.7 \n");
	Disp(" 2:Level_2 - 		400 	200 	100.0 	166.8 	83.4 	133.4 	66.7 \n");
	Disp(" 3:Level_3 - 		200 	200 	100.0 	166.8 	83.4 	133.4 	66.7 \n");
	Disp(" 4:Level_4 - 		200 	100 	100.0 	166.8 	83.4 	133.4 	66.7 \n");
	Disp(" 5:Level_5 - 		200 	100 	100.0 	83.4 	41.7 	83.4 	41.7 \n");
	Disp(" 6:Level_6 - 		100 	100 	50.0 	83.4 	41.7 	83.4 	41.7 \n");
	Disp(" 7:Level_7 - 		100 	50.0	50.0 	41.7 	41.7 	41.7 	41.7 \n");
	Disp(" Select DVFS level :");
	uSelLv = UART_GetIntNum();
	if (uSelLv > (DVFS_MAX_LV-1)) uSelLv = 0;

	do
	{
		// random aging test
		if (uSelAging == 2)
		{
			uSelLv = rand()%DVFS_MAX_LV;
			uTransBase = 0x80000*(rand()%5);
		}
		DisplayLED(1);

		// Level setting
		SYSC_ChangeDvfsLv(uSelLv);
		DisplayLED(6);

		// Change Voltage 
//		PMIC_Buck1SetVoltage();
//		PMIC_Buck2SetVoltage();
		DisplayLED(7);
		
		// Display changed clocks
		Disp("\n***** PLL setting changed as follows *****\n");
		Disp(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz	PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz	PCLK_Psys: %.2fMHz\n",
				(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
		Disp(" APLL: %.2fMHz	MPLL: %.2fMHz\n EPLL: %.2fMHz	VPLL: %.2fMHz\n\n",
				(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);

		// Mem to mem test
		Disp(" Mem to mem start !!!");
		memcpy((u32 *)(uDstAddr), (u32 *)(uSrcAddr), uTransBase);
		if ( Compare(uSrcAddr, uDstAddr, uTransBase/4) != true)
			Disp(" ...followed item copy failed...\n");
		else
			Disp(" copy ok!! \n");

		Disp("\n If you want to exit, press enter key now!!\n");
		Delay(10);
		if (UART_GetKey()!=0) break;		
		DisplayLED(8);		
	} 
	while (uSelAging == 2);
}


// [Test Clock Mux Control]
void SYSCT_TestClkMuxControl(void)
{
	u32 uSelMux, uSelMux_sub, uSelSrc, uSelIP, uSelCh;
	u32 uRegSrc0, uRegSrc1, uRegDiv0;
	CLKMUX_STATE_ID eClkMuxId;
	SDHC SDCH;
		
	//// DRAM Parameter setting
	//  - Row, Column and Power timing parameters of the DRAM should be set for max. freq.
	//  - Auto Refresh count should be set for min. freq.
#if 1 // LPDDR
	DMC_SetRowTimingParameters(0xF, 0x3, 0x4, 0x4, 0xE, 0x29); // for 200MHz LPDDR
	DMC_SetDataTimingParameters(0x3, 0x3, 0x4, 0x3, 0x3, 0x3); // for 200MHz
	DMC_SetPowerModeTimingParameters(0x4, 0x18, 0x2, 0x3, 0x3); // for 200MHz
	DMC_SetAutoRefresh(AREF_EN, 0x4E);	// for 12MHz
#endif
	
	SYSC_InitSysClk(eINIT_SYSCLK_MUX);

	// update console
	SYSC_UpdateClkInform();
	OpenConsole(DBG_UART_CH);

	//// Start system clock mux test
	while (1)
	{
		Disp("\nPlease check OM[0] pin(0:XXTI,1:XusbXTI) for MUX_FINPLL output .\n");
		Disp(" 1:MUX_APLL, 2:MUX_MPLL, 3:MUX_EPLL, 4:MUX_VPLL, 5:MUX_FINVPLL\n");
		Disp(" 6:MUX_MSYS, 7:MUX_DSYS, 8:MUX_PSYS\n");
		Disp("Press Enter key only to exit.\n");
		Disp("System bus clock mux test!! Select MUX.\n");
		
		uSelMux = UART_GetIntNum();
		if(uSelMux == (u32)(-1)) break;

		Disp("Select input source!!\n");
		switch (uSelMux)
		{
 		case 2 :
			Disp(" 1(D):FIN_PLL, 2:FOUT_MPLL\n"); // check update clock & UART
			eClkMuxId = eCLKMUX_MPLL_FINPLL;
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 1);
			Disp(" Check CLKOUT HCLKDSYS pad for your selection!! And press any key!! \n");		
			break;
		case 3 :
			Disp(" 1(D):FIN_PLL, 2:FOUT_EPLL\n");
			eClkMuxId = eCLKMUX_EPLL_FINPLL;
			SYSC_SetClkMuxState(eCLKMUX_MMC0_SEPLL);
			SDHC_OpenMediaWithMode(4, SDHC_POLLING_MODE, SDHC_EPLL, 0, SDHC_CHANNEL_0, &SDCH);			
			Disp(" Check MMC0 CLK pad for your selection!! And press any key!! \n");		
			break;
		case 4 :
			Disp(" 1(D):FIN_VPLL, 2:FOUT_VPLL\n");
			eClkMuxId = eCLKMUX_VPLL_FINVPLL;	
			SYSC_SetClkMuxState(eCLKMUX_MMC0_SVPLL);			
			SDHC_OpenMediaWithMode(4, SDHC_POLLING_MODE, SDHC_EPLL, 0, SDHC_CHANNEL_0, &SDCH);	
			Disp(" Check MMC0 CLK pad for your selection!! And press any key!! \n");		
			break;
		case 5 :
			Disp(" 1(D):FIN_PLL, 2:SCLK_HDMI27M\n");
			eClkMuxId = eCLKMUX_VPLLSRC_FINPLL;
			SYSC_SetClkMuxState(eCLKMUX_MMC0_SVPLL);			
			SDHC_OpenMediaWithMode(4, SDHC_POLLING_MODE, SDHC_EPLL, 0, SDHC_CHANNEL_0, &SDCH);	
			Disp(" Check MMC0 CLK pad for your selection!! And press any key!! \n");
			break;
		case 6 :
			Disp(" 1(D):SCLKAPLL, 2:SCLKMPLL\n");
			eClkMuxId = eCLKMUX_MSYS_SCLKAPLL;
			SYSC_SetClkOut(eCLKOUT_HCLKMSYS, 1);
			Disp(" Check CLKOUT HCLKMSYS pad for your selection!! And press any key!! \n");
			break;
		case 7 :
			Disp(" 1(D):SCLKMPLL, 2:SCLKA2M\n");
			eClkMuxId = eCLKMUX_DSYS_SCLKMPLL;
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 1);
			Disp(" Check CLKOUT HCLKDSYS pad for your selection!! And press any key!! \n");
			break;
		case 8 :
			Disp(" 1(D):SCLKMPLL, 2:SCLKA2M\n");
			eClkMuxId = eCLKMUX_PSYS_SCLKMPLL;
			SYSC_SetClkOut(eCLKOUT_HCLKPSYS, 1);
			Disp(" Check CLKOUT HCLKPSYS pad for your selection!! And press any key!! \n");
			break;
 		default :
			Disp(" 1(D):FIN_PLL, 2:FOUT_APLL\n");
			eClkMuxId = eCLKMUX_APLL_FINPLL;
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 1);
			Disp(" Check CLKOUT ARMCLK_DIV4 pad for your selection!! And press any key!! \n");
			break;
		}
		uSelSrc = UART_GetIntNum();
		if (uSelSrc == 2)
			eClkMuxId += 1;

		SYSC_SetClkMuxState(eClkMuxId);

		// update console
		SYSC_UpdateClkInform();
		OpenConsole(DBG_UART_CH);
		Disp(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz  PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz  PCLK_Psys: %.2fMHz\n\n",
					(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
		
		// Display CLK_SRC0 & CLK_SRC1
		uRegSrc0 = SYSC_GetClkMuxReg(0);
		uRegSrc1 = SYSC_GetClkMuxReg(1);
		uRegDiv0 = SYSC_GetClkDivReg(0);
		Disp(" rCLK_SRC0 : 0x%8x \n", uRegSrc0);
		Disp(" rCLK_SRC1 : 0x%8x \n", uRegSrc1);
		Disp(" rCLK_DIV0 : 0x%8x \n", uRegDiv0);
	}

	while (1)
	{
		Disp("\nPlease check OM[0] pin(0:XXTI,1:XusbXTI) for MUX_FINPLL output .\n");
		Disp(" 1:OneNand(MOUTFLASH), 2:OneDRAM(MOUTONEDR), 3:MFC, 4:G3D, 5:HPM\n");
		Disp(" 6:AUDIO, 7:TV, 8:OTHERS\n");
		Disp("Press Enter key only to exit.\n");
		Disp("Select test IP!!\n");
		uSelIP = UART_GetIntNum();
		if(uSelIP == (u32)(-1)) break;

		switch (uSelIP)
		{
		case 1 :
			Disp(" 1(D):MOUT133, 2:MOUT166\n");
			Disp("Select clock source!!\n");
			uSelSrc = UART_GetIntNum();
			eClkMuxId = eCLKMUX_OND_HCLKPSYS;
			Disp(" Check OneNand CLK(HCLKPSYS) pad for your selection!! And press any key!! \n"); 	
			break;
		case 2 :
			Disp(" 1(D):SCLKA2M, 2:SCLKMPLL, 3:SCLKEPLL, 4:SCLKVPLL\n");
			Disp("Select clock source!!\n");
			uSelSrc = UART_GetIntNum();
			Disp(" Check OneDram CLK pad for your selection!! And press any key!! \n");
			eClkMuxId = eCLKMUX_ONEDRAM_SCLKA2M;
			break;
		case 3 :
			Disp(" 1:SCLKA2M, 2:SCLKMPLL, 3:SCLKEPLL, 4:SCLKVPLL\n");
			Disp("Select clock source!!\n");
			uSelSrc = UART_GetIntNum();
			Disp(" Check MFC ??? And press any key!! \n");
			eClkMuxId = eCLKMUX_MFC_SCLKA2M;
			break;
		case 4 :
			Disp(" 1:SCLKA2M, 2:SCLKMPLL, 3:SCLKEPLL, 4:SCLKVPLL\n");
			Disp("Select clock source!!\n");
			uSelSrc = UART_GetIntNum();
			Disp(" Check G3D ??? And press any key!! \n");
			eClkMuxId = eCLKMUX_G3D_SCLKA2M;
			break;
		case 5 :
			Disp(" 1(D):SCLKAPLL, 2:SCLKMPLL\n");
			Disp("Select clock source!!\n");
			uSelSrc = UART_GetIntNum();
			eClkMuxId = eCLKMUX_HPM_SCLKAPLL;
			Disp(" Check HPM ??? And press any key!! \n");
			break;
		case 6 :
			Disp("\nAUDIO\n");
			break;
		case 7 :
			Disp("\nTV\n");			
			break;
		case 8 :
			Disp(" 0:XXTI, 1:XusbXTI, 2:SCLK_HDMI27M, 3:SCLK_USBPHY0, 4:SCLK_USBPHY1\n");
			Disp(" 5:SCLK_HDMIPHY, 6:SCLKMPLL, 7:SCLKEPLL, 8:SCLKVPLL\n");
			Disp("Select clock source!!\n");
			uSelSrc = UART_GetIntNum();

			Disp(" \nSelect IP 1:UART, 2:PWM\n");
			uSelIP = UART_GetIntNum();

			if(uSelIP == 1)
			{
				Disp("\nSelect UART Channel\n");
				uSelCh = UART_GetIntNum();				
				switch(uSelCh)
				{
				case 0 : 
					eClkMuxId = eCLKMUX_UART0_XXTI;
					break;
				case 1 :
					eClkMuxId = eCLKMUX_UART1_XXTI;
					break;
				case 2 :
					eClkMuxId = eCLKMUX_UART2_XXTI;
					break;
				case 3 :
					eClkMuxId = eCLKMUX_UART3_XXTI;
					break;
				default : 
					eClkMuxId = eCLKMUX_UART1_XXTI;
					break;
				}
				eClkMuxId = eClkMuxId + uSelSrc;

				SYSC_SetClkMuxState(eClkMuxId);

				//OpenConsole_GOM(1);	
				Disp("\nChanged Uart Print test OK?\n");
			}
			else if(uSelIP == 2)
			{
				g_uIntCounter0 =0;   
				PWM_stopAll();	// stop timers
	
				INTC_SetVectAddr(NUM_TIMER0,Isr_TIMER0);
				INTC_Enable(NUM_TIMER0);
	
    			Disp("\n[PWM Clock Mux test using Timer0 Interrupt]\n\n");

				eClkMuxId = eCLKMUX_PWM_XXTI;
				eClkMuxId = eClkMuxId + uSelSrc;
				
				PWM_Select(0,255,5,30000,15000,0,0,0,1,0,1);
				SYSC_SetClkDiv(eCLKDIV_PWM, 12);	
				SYSC_SetClkMuxState(eClkMuxId);	
				
				while(1)
				{
				if(g_uIntCounter0 == 1)
				break;
				}
				PWM_stopAll();	 //Stop Timers

	      		Disp("Timer0 Interrupt Test => OK!\n");					
			}
			break;
			
		default :
			Disp(" 1(D):FIN_PLL, 2:FOUT_APLL\n");
			eClkMuxId = eCLKMUX_APLL_FINPLL;
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 1);
			Disp(" Check CLKOUT ARMCLK_DIV4 pad for your selection!! And press any key!! \n");
			break;
		}

		if(uSelIP !=8)
		{
			if (uSelSrc > 2)
				eClkMuxId = eClkMuxId + uSelSrc;
			SYSC_SetClkMuxState(eClkMuxId);
		}
		
		// system init setting	
		SYSC_InitSysClk(eINIT_SYSCLK_NORMAL);
		SYSC_UpdateClkInform();
		OpenConsole(DBG_UART_CH);
	}

	//// Retrurn DRAM parameter setting

}

void SYSCT_TestClkDiv(void)
{

	u32 uSelDiv, uDividerValue=0;
	u32 uRegDiv0,uRegDiv1,uRegDiv2;
	float uClkOutFreq;

	u32 uregtemp;

	//// DRAM Parameter setting
	//  - Row, Column and Power timing parameters of the DRAM should be set for max. freq.
	//  - Auto Refresh count should be set for min. freq.
#if 1 // LPDDR
	DMC_SetRowTimingParameters(0xF, 0x3, 0x4, 0x4, 0xE, 0x29); // for 200MHz LPDDR
	DMC_SetDataTimingParameters(0x3, 0x3, 0x4, 0x3, 0x3, 0x3); // for 200MHz
	DMC_SetPowerModeTimingParameters(0x4, 0x18, 0x2, 0x3, 0x3); // for 200MHz
	DMC_SetAutoRefresh(AREF_EN, 0x4E);	// for 12MHz
#endif
	
	SYSC_InitSysClk(eINIT_SYSCLK_DIV);
	SYSC_UpdateClkInform();
	OpenConsole(DBG_UART_CH);
	Disp(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz  PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz  PCLK_Psys: %.2fMHz\n\n",
						(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
			

	// update console
	SYSC_UpdateClkInform();
	OpenConsole(DBG_UART_CH);
	
	while (1)
	{
		Disp("System bus clock Divider test!!\n");
		Disp(" 1:DIV_APLL(D), 2:DIV_A2M, 3:DIV_HCLK200, 4:DIV_PCLK100\n");
		Disp(" 5:DIV_HCLK166, 6:DIV_PCLK83, 7:DIV_HCLK133, 8:DIV_PCLK66\n");
		Disp("Press Enter key only to exit.\n");
		Disp("Select Divider!!\n");
		uSelDiv = UART_GetIntNum();
		if (uSelDiv == (u32)(-1)) break;

		Disp("Select input source!!\n");
		switch (uSelDiv)
		{
 		case 1 :
			Disp(" DIV_APLL check using ARMCLK/4 port\n"); 
			Disp(" Select a Divider value(1~8)\n"); 
			uDividerValue = UART_GetIntNum();
			SYSC_SetClkDiv(eCLKDIV_APLL, uDividerValue);
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 1);
			Disp("uregtemp = %x\n", SYSC_GetClkDivStatusReg(0));
			while(uregtemp&0x1)
				{
				uregtemp = SYSC_GetClkDivStatusReg(0);
				Disp("uregtemp = %x\n", uregtemp);
				}
			uClkOutFreq = (g_uAPLL/4.)/(uDividerValue);
			Disp("Clock Out frequency = %.2fMHz\n", (uClkOutFreq/1.0e6));
			Disp(" Check CLKOUT pad for your selection!!\n");		
			break;
		case 2 :
			Disp(" DIV_A2M check using HCLKDSYS port\n"); 
			Disp(" Select a Divider value(1~8)\n"); 
			uDividerValue = UART_GetIntNum();
			SYSC_SetClkDiv(eCLKDIV_A2M, uDividerValue);
			SYSC_SetClkDiv(eCLKDIV_HCLKDSYS, 1);
			SYSC_SetClkMuxState(eCLKMUX_DSYS_SCLKA2M);
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 1);
			uClkOutFreq = (g_uAPLL/uDividerValue);
			Disp("Clock Out frequency = %.2fMHz\n", (uClkOutFreq/1.0e6));
			Disp(" Check CLKOUT pad for your selection!!\n");
			break;
		case 3 :
			Disp(" DIV_HCLK_MSYS check using HCLKMSYS port\n"); 
			Disp(" Select a Divider value(1~8)\n"); 
			uDividerValue = UART_GetIntNum();
			SYSC_SetClkDiv(eCLKDIV_HCLKMSYS, uDividerValue);
			SYSC_SetClkOut(eCLKOUT_HCLKMSYS, 1);
			uClkOutFreq = (g_uARMCLK/uDividerValue);
			Disp("Clock Out frequency = %.2fMHz\n", (uClkOutFreq/1.0e6));
			Disp(" Check CLKOUT pad for your selection!!\n");
			break;
		case 4 :
			Disp(" DIV_PCLK_MSYS check using PCLKMSYS port\n"); 
			Disp(" Select a Divider value(1~8)\n"); 
			uDividerValue = UART_GetIntNum();
			SYSC_SetClkDiv(eCLKDIV_PCLKMSYS, uDividerValue);
			SYSC_SetClkOut(eCLKOUT_PCLKMSYS, 1);
			uClkOutFreq = (g_uARMCLK/uDividerValue);
			Disp("Clock Out frequency = %.2fMHz\n", (uClkOutFreq/1.0e6));
			Disp(" Check CLKOUT pad for your selection!!\n");
			break;
		case 5 :
			Disp(" DIV_HCLK_DSYS check using HCLKDSYS port\n"); 
			Disp(" Select a Divider value(1~8)\n"); 
			uDividerValue = UART_GetIntNum();
			SYSC_SetClkDiv(eCLKDIV_HCLKDSYS, uDividerValue);
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 1);
			uClkOutFreq = (g_uMPLL/uDividerValue);
			Disp("Clock Out frequency = %.2fMHz\n", (uClkOutFreq/1.0e6));
			Disp(" Check CLKOUT pad for your selection!!\n");
			break;
		case 6 :
			Disp(" DIV_PCLK_DSYS check using PCLKDSYS port\n"); 
			Disp(" Select a Divider value(1~8)\n"); 
			uDividerValue = UART_GetIntNum();
			SYSC_SetClkDiv(eCLKDIV_PCLKDSYS, uDividerValue);
			SYSC_SetClkOut(eCLKOUT_PCLKDSYS, 1);
			uClkOutFreq = (g_uHclkDsys/uDividerValue);
			Disp("Clock Out frequency = %.2fMHz\n", (uClkOutFreq/1.0e6));
			Disp(" Check CLKOUT pad for your selection!!\n");
			break;
		case 7 :
			Disp(" DIV_HCLK_PSYS check using HCLKPSYS port\n"); 
			Disp(" Select a Divider value(1~8)\n"); 
			uDividerValue = UART_GetIntNum();
			SYSC_SetClkDiv(eCLKDIV_HCLKPSYS, uDividerValue);
			SYSC_SetClkOut(eCLKOUT_HCLKPSYS, 1);
			uClkOutFreq = (g_uMPLL/uDividerValue);
			Disp("Clock Out frequency = %.2fMHz\n", (uClkOutFreq/1.0e6));
			Disp(" Check CLKOUT pad for your selection!! \n");
			break;
		case 8 :
			Disp(" DIV_PCLK_PSYS check using PCLKPSYS port\n"); 
			Disp(" Select a Divider value(1~8)\n"); 
			uDividerValue = UART_GetIntNum();
			SYSC_SetClkDiv(eCLKDIV_PCLKPSYS, uDividerValue);
			SYSC_SetClkOut(eCLKOUT_PCLKPSYS, 1);
			uClkOutFreq = (g_uHclkPsys/uDividerValue);
			Disp("Clock Out frequency = %.2fMHz\n", (uClkOutFreq/1.0e6));
			Disp(" Check CLKOUT pad for your selection!!\n");
			break;
 		default :
			Disp(" DIV_APLL check using ARMCLK/4 port\n"); 
			Disp(" Select a Divider value(1~8)\n"); 
			uDividerValue = UART_GetIntNum();
			SYSC_SetClkDiv(eCLKDIV_APLL, uDividerValue);
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 1);
			uClkOutFreq = (g_uAPLL/4.)/(uDividerValue);
			Disp("Clock Out frequency = %.2fMHz\n", (uClkOutFreq/1.0e6));
			Disp(" Check CLKOUT pad for your selection!!\n");	
			break;
		}

		// Display CLK_SRC0 & CLK_SRC1
		
		uRegDiv0 = SYSC_GetClkDivReg(0);
		uRegDiv1 = SYSC_GetClkDivReg(1);
		uRegDiv2 = SYSC_GetClkDivReg(2);
		Disp(" rCLK_DIV0 : 0x%8x \n", uRegDiv0);
		Disp(" rCLK_DIV1 : 0x%8x \n", uRegDiv1);
		Disp(" rCLK_DIV2 : 0x%8x \n", uRegDiv2);

		Disp(" \nPress any key!!\n");
		while(!UART_Getc());
		
		SYSC_InitSysClk(eINIT_SYSCLK_DIV);
		SYSC_UpdateClkInform();
		OpenConsole(DBG_UART_CH);
	}	
	
	//// Retrurn DRAM parameter setting

}


void SYSCT_SelectClkGating(void)
{
	u32 uSelClkGatingType, uSelClkGatingIp, uSelClkGatingBlock, uSelOnOff;
	u32 uIpReg[5], uBlockReg;
	u32 i;
	bool bOn;
	CLKGATE_IP_TYPE eIpType;
	
	while(1)
	{
		Disp("* Select clock gating type :\n 1.by IP, 2.by Block, 0.Exit(D)\n");
		Disp(" Loop this setting function for selecting several clock gatings until exiting!!\n");
		uSelClkGatingType = UART_GetIntNum();

		if (uSelClkGatingType == 2) // Block gating
		{
			Disp("* Select clock gate type :\n 1.G3D, 2.MFC, 3.IMG, 4.LCD, 5.TV, 6.USB, 7.MEM,\n"); 
			Disp("	8.SECURE, 9.DEBUG, 10.MMC, 11.INTC, 12.ALL, 0.EXIT(D)\n");
			uSelClkGatingBlock = UART_GetIntNum();
			if((uSelClkGatingBlock<1)||(uSelClkGatingBlock>12))
				break;
			Disp("* Select : 1.On(D), 2.Off\n");
			uSelOnOff = UART_GetIntNum();
			if (uSelOnOff ==2) 
				bOn = FALSE;
			else 
				bOn= TRUE;
			
			SYSC_SetClkGatebyBlock(uSelClkGatingBlock-1, bOn);
		}
		else if (uSelClkGatingType == 1) // default IP gating
		{
			Disp("* Select clock gating by IP : \n");
			Disp("   1.CSIS, 2.IPC, 3.ROTATOR, 4.JPEG, 5.FIMC2, 6.FIMC1, 7.FIMC0, 8.MFC, 9.G3D, 10.IMEM, \n");
			Disp("      11.PDMA1, 12.PDMA2, 13.MDMA, 14.DMC1, 15.DMC0 \n");
			Disp("   21.NFCON ,22.SROMC, 23.CFCON, 24.NANDXL, 25.USBHOST, 26.USBOTG, 27.HDMI, 28.TVENC, 29.MIXER, 30.VP\n");
			Disp("      31.DSIM, 32.MIE, 33.FIMD, \n");
			Disp("   41.TZIC3, 42.TZIC2, 43.TZIC1, 44.TZIC0, 45.VIC3, 46.VIC2, 47.VIC1, 48.VIC0, 49.TSI, 50.MMC3, \n");
			Disp("      51.MMC2, 52.MMC1, 53.MMC0, 54.SECJTAG, 55.MODEM, 56.CORESIGHT, 57.SDM, 58.SECSS \n");
			Disp("   61.PCM2, 62.PCM1, 63.PCM0, 64.GPIO, 65.TSADC, 66.PWM, 67.WDT, 68.KEYIF, 69.UART3, 70.UART2, \n");
			Disp("      71.UART1, 72.UART0, 73.SYSTIMER, 74.RTC, 75.SPI2, 76,SPI1, 77.SPI0, 78.IIC_HDMI_PHY, 79.IIC_HDMI_DDC, 80.IIC2, \n");
			Disp("      81.IIC1, 82.IIC0, 83.IIS2, 84.IIS1, 85.IIS0, 86.AC97, 87.SPDIF, \n");
			Disp("   91.TZPC3, 92.TZPC2, 93.TZPC1, 94.TZPC0, 95.SECKEY, 96.IEM_APC, 97.IEM_IEC, 98.CHIP_ID, \n");
			Disp("   99.ALL(D) \n");
			uSelClkGatingIp = UART_GetIntNum();
			switch(uSelClkGatingIp)
			{
			// CLK_GATE_IP0	
			case 1 :
				eIpType = eCLKGATE_IP_CSIS;
				break;
			case 2 :
				eIpType = eCLKGATE_IP_IPC;
				break;
			case 3 :
				eIpType = eCLKGATE_IP_ROTATOR;
				break;
			case 4 :
				eIpType = eCLKGATE_IP_JPEG;
				break;
			case 5 :
				eIpType = eCLKGATE_IP_FIMC2;
				break;
			case 6 :
				eIpType = eCLKGATE_IP_FIMC1;
				break;
			case 7 :
				eIpType = eCLKGATE_IP_FIMC0;
				break;
			case 8 :
				eIpType = eCLKGATE_IP_MFC;
				break;
			case 9 :
				eIpType = eCLKGATE_IP_G3D;
				break;
			case 10 :
				eIpType = eCLKGATE_IP_IMEM;
				break;
			case 11 :
				eIpType = eCLKGATE_IP_PDMA1;
				break;
			case 12 :
				eIpType = eCLKGATE_IP_PDMA0;
				break;
			case 13 :
				eIpType = eCLKGATE_IP_MDMA;
				break;
			case 14 :
				eIpType = eCLKGATE_IP_DMC1;
				break;
			case 15 :
				eIpType = eCLKGATE_IP_DMC0;
				break;
			case 16 :
			case 17 :
			case 18 :
			case 19 :
			case 20 :
				eIpType = 900;
				break;
			// CLK_GATE_IP1
			case 21 :
				eIpType = eCLKGATE_IP_NFCON;
				break;
			case 22 :
				eIpType = eCLKGATE_IP_SROMC;
				break;
			case 23 :
				eIpType = eCLKGATE_IP_CFCON;
				break;
			case 24 :
				eIpType = eCLKGATE_IP_NANDXL;
				break;
			case 25 :
				eIpType = eCLKGATE_IP_USBHOST;
				break;
			case 26 :
				eIpType = eCLKGATE_IP_USBOTG;
				break;
			case 27 :
				eIpType = eCLKGATE_IP_HDMI;
				break;
			case 28 :
				eIpType = eCLKGATE_IP_TVENC;
				break;
			case 29 :
				eIpType = eCLKGATE_IP_MIXER;
				break;
			case 30 :
				eIpType = eCLKGATE_IP_VP;
				break;
			case 31 :
				eIpType = eCLKGATE_IP_DSIM;
				break;
			case 32 :
				eIpType = eCLKGATE_IP_MIE;
				break;
			case 33 :
				eIpType = eCLKGATE_IP_FIMD;
				break;
			case 34 :
			case 35 :
			case 36 :
			case 37 :
			case 38 :
			case 39 :
			case 40 :
				eIpType = 900;
				break;
			// CLK_GATE_IP2
			case 41 :
				eIpType = eCLKGATE_IP_TZIC3;
				break;
			case 42 :
				eIpType = eCLKGATE_IP_TZIC2;
				break;
			case 43 :
				eIpType = eCLKGATE_IP_TZIC1;
				break;
			case 44 :
				eIpType = eCLKGATE_IP_TZIC0;
				break;
			case 45 :
				eIpType = eCLKGATE_IP_VIC3;
				break;
			case 46 :
				eIpType = eCLKGATE_IP_VIC2;
				break;
			case 47 :
				eIpType = eCLKGATE_IP_VIC1;
				break;
			case 48 :
				eIpType = eCLKGATE_IP_VIC0;
				break;
			case 49 :
				eIpType = eCLKGATE_IP_TSI;
				break;
			case 50 :
				eIpType = eCLKGATE_IP_MMC3;
				break;
			case 51 :
				eIpType = eCLKGATE_IP_MMC2;
				break;
			case 52 :
				eIpType = eCLKGATE_IP_MMC1;
				break;
			case 53 :
				eIpType = eCLKGATE_IP_MMC0;
				break;
			case 54 :
				eIpType = eCLKGATE_IP_SECJTAG;
				break;
			case 55 :
				eIpType = eCLKGATE_IP_MODEM;
				break;
			case 56 :
				eIpType = eCLKGATE_IP_CORESIGHT;
				break;
			case 57 :
				eIpType = eCLKGATE_IP_SDM;
				break;
			case 58 :
				eIpType = eCLKGATE_IP_SECSS;
				break;
			case 59 :
			case 60 :
				eIpType = 900;
				break;
			// CLK_GATE_IP3
			case 61 :
				eIpType = eCLKGATE_IP_PCM2;
				break;
			case 62 :
				eIpType = eCLKGATE_IP_PCM1;
				break;
			case 63 :
				eIpType = eCLKGATE_IP_PCM0;
				break;
			case 64 :
				eIpType = eCLKGATE_IP_GPIO;
				break;
			case 65 :
				eIpType = eCLKGATE_IP_TSADC;
				break;
			case 66 :
				eIpType = eCLKGATE_IP_PWM;
				break;
			case 67 :
				eIpType = eCLKGATE_IP_WDT;
				break;
			case 68 :
				eIpType = eCLKGATE_IP_KEYIF;
				break;
			case 69 :
				eIpType = eCLKGATE_IP_UART3;
				break;
			case 70 :
				eIpType = eCLKGATE_IP_UART2;
				break;
			case 71 :
				eIpType = eCLKGATE_IP_UART1;
				break;
			case 72 :
				eIpType = eCLKGATE_IP_UART0;
				break;
			case 73 :
				eIpType = eCLKGATE_IP_SYSTIMER;
				break;
			case 74 :
				eIpType = eCLKGATE_IP_RTC;
				break;
			case 75 :
				eIpType = eCLKGATE_IP_SPI2;
				break;
			case 76 :
				eIpType = eCLKGATE_IP_SPI1;
				break;
			case 77 :
				eIpType = eCLKGATE_IP_SPI0;
				break;
			case 78 :
				eIpType = eCLKGATE_IP_I2CHDMIPHY;
				break;
			case 79 :
				eIpType = eCLKGATE_IP_I2CHDMIDDC;
				break;
			case 80 :
				eIpType = eCLKGATE_IP_I2C2;
				break;
			case 81 :
				eIpType = eCLKGATE_IP_I2C1;
				break;
			case 82 :
				eIpType = eCLKGATE_IP_I2C0;
				break;
			case 83 :
				eIpType = eCLKGATE_IP_I2S2;
				break;
			case 84 :
				eIpType = eCLKGATE_IP_I2S1;
				break;
			case 85 :
				eIpType = eCLKGATE_IP_I2S0;
				break;
			case 86 :
				eIpType = eCLKGATE_IP_AC94;
				break;
			case 87 :
				eIpType = eCLKGATE_IP_SPDIF;
				break;
			case 88 :
			case 89 :
			case 90 :
				eIpType = 900;
				break;
			// CLK_GATE_IP4
			case 91 :
				eIpType = eCLKGATE_IP_TZPC3;
				break;
			case 92 :
				eIpType = eCLKGATE_IP_TZPC2;
				break;
			case 93 :
				eIpType = eCLKGATE_IP_TZPC1;
				break;
			case 94 :
				eIpType = eCLKGATE_IP_TZPC0;
				break;
			case 95 :
				eIpType = eCLKGATE_IP_SECKEY;
				break;
			case 96 :
				eIpType = eCLKGATE_IP_IEM_APC;
				break;
			case 97 :
				eIpType = eCLKGATE_IP_IEM_IEC;
				break;
			case 98 :
				eIpType = eCLKGATE_IP_CHIP_ID;
				break;
			// ALL
			case 99 :
				eIpType = eCLKGATE_IP_ALL;
				break;
		
			}
			Disp("* Select : 1.On(D), 2.Off\n");
			uSelOnOff = UART_GetIntNum();
			if (uSelOnOff ==2) 
				bOn = FALSE;
			else 
				bOn= TRUE;
			
			if (eIpType != 900)
				SYSC_SetClkGatebyIP(eIpType, bOn);
		}
		else 
		{
			return;
		}

		for (i=0;i<=4;i++)
		{
			uIpReg[i] = SYSC_GetClkGateIPReg(i);
			Disp(" Clock gate by IP %1d = 0x%x\n", i, uIpReg[i]);
		}
		uBlockReg = SYSC_GetClkGateBlockReg();
		Disp(" Clock gate by Block = 0x%x\n", uBlockReg);
		
	}
}

void SYSCT_TestClkGating(void)
{
	SYSC_InitSysClk(eINIT_SYSCLK_NORMAL);
	SYSC_UpdateClkInform();
	OpenConsole(DBG_UART_CH);
	
	SYSCT_SelectClkGating();

	// Test each IP
	Disp(" Return to main menu for each IP test..\n");
}


void SYSCT_TestClkout(void)
{
	u32 uSelSrc;
	u32 uDclkCmp, uDclkDiv, uDclkSrcClk;
	bool bOn;

	u8 *parrClkOut[] = {"FOUTAPLL/4","FOUTMPLL/2","FOUTEPLL","FOUTVPLL","SCLK_USBPHY0",
						   "SCLK_USBPHY1","SCLK_HDMIPHY","RTC","RTC_TICK_SRC","HCLK_MSYS",
						   "PCLK_MSYS","HCLK_DSYS","PCLK_DSYS","HCLK_PSYS","PCLK_PSYS",
						   "ARMCLK/4","SCLK_HPM","XXTI","XUSBXTI","DOUT","XXTI_PD","XUSBXTI_PD"};
	u32 uClkSelCnt;
	u32 temp;

	u8 *parrDclkOut[] = {"XXTI","XUSBXTI","SCLK_HDMI27M","SCLK_USBPHY0","SCLK_USBPHY1",
						"SCLK_HDMIPHY","FOUTMPLL/2","SCLKEPLL"};
	u32 uDclkSelCnt;
	
	SYSC_UpdateClkInform();
	Disp(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz  PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz  PCLK_Psys: %.2fMHz\n",
			(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
	Disp(" APLL: %.2fMHz\tMPLL: %.2fMHz\tEPLL: %.2fMHz\tVPLL: %.2fMHz\n\n",
			(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);

	Disp("CLKOUT Divider value : 2\n");
	for(uClkSelCnt=0;uClkSelCnt<22;uClkSelCnt++) {
		Disp("CLKSEL[%02d] : %s\n",uClkSelCnt,parrClkOut[uClkSelCnt]);

		if(uClkSelCnt==eCLKOUT_SCLKUSBPHY0) {
			SYSC_SetClkOut(uClkSelCnt,2);
			SYSC_SetUSBPHYControl(1);
			OTGDEV_InitPhyCon();
		}
		else if(uClkSelCnt==eCLKOUT_SCLKUSBPHY1) {
			SYSC_SetClkOut(uClkSelCnt,2);
			SYSC_SetUSBPHYControl(1);

			temp = Inp32(0xEC100000);
			temp = (temp & ~(0x7<<6))|(0x4<<6);
			Outp32(0xEC100000, temp);

			Outp32(0xEC100004, 0x03);
			Outp32(0xEC100008, 0x4);
			Delay(10);
			Outp32(0xEC100008, 0x0);
			Delay(10);
		}
		else if(uClkSelCnt==eCLKOUT_SCLKHDMIPHY) {
			SYSC_SetClkOut(uClkSelCnt,2);
			SYSC_EnHDMIPHY();
		}
		else if(uClkSelCnt==eCLKOUT_SCLKHPM) {
			SYSC_SetClkOut(uClkSelCnt,2);
			SYSC_SetClkDiv(eCLKDIV_HPM,4); 
		}
		else if(uClkSelCnt==eCLKOUT_DOUT) {
			SYSC_SetClkOut(uClkSelCnt,2);
			SYSC_SetDclkOnOff(1); 
			for(uDclkSelCnt=0;uDclkSelCnt<8;uDclkSelCnt++) {
				Disp("DCLKSEL[%d] : %s\n",uDclkSelCnt,parrDclkOut[uDclkSelCnt]);
				SYSC_SetDclkOut(5,10,uDclkSelCnt);
				Disp("   Duty ratio 50:50\n");	if(UART_Getc()=='0') return;
				
				SYSC_SetDclkOut(1,10,uDclkSelCnt);
				Disp("   Duty ratio 10:90\n");	if(UART_Getc()=='0') return;
				
				SYSC_SetDclkOut(9,10,uDclkSelCnt);
				Disp("   Duty ratio 90:10\n");	if(UART_Getc()=='0') return;
				}
		}
		else if(uClkSelCnt==20) {
			SYSC_SetClkOut(eCLKOUT_XXTI_PD,2);
		}
		else if(uClkSelCnt==21) {
			SYSC_SetClkOut(eCLKOUT_XUSBXTI_PD,2);
		}
		else {
			SYSC_SetClkOut(uClkSelCnt,2);
		}

		if(UART_Getc()=='0') return;

	}

	Disp("CLKOUT DIVVAL MIN/MAX Test\n");
	SYSC_SetClkOut(eCLKOUT_XXTI,1);
	Disp("24MHz/1=24MHz\n");	if(UART_Getc()=='0') return;
	SYSC_SetClkOut(eCLKOUT_XXTI,16);
	Disp("24MHz/16=1.5MHz\n");	if(UART_Getc()=='0') return;

}



//////////////////////////////////////////////
// Power Management Unit (PMU - ALIVE region)
//////////////////////////////////////////////


// [Interrupt service routine for External Int.]
static __irq void Isr_EInt1(void)
{
	DbgSysc(("EINT1 ISR\n"));

	// interrupt clear in EINT SFR
	GPIO_EINT30ClrPend(1);
	INTC_ClearVectAddr();
}

static __irq void Isr_EInt4(void)
{
	DbgSysc(("EINT4 ISR\n"));

	// interrupt clear in EINT SFR
	GPIO_EINT30ClrPend(4);
	INTC_ClearVectAddr();
}

static __irq void Isr_EInt31(void)
{
	DbgSysc(("EINT31 ISR\n"));
	Disp("E31\n");

	// interrupt clear in EINT SFR
	GPIO_EINT33ClrPend(7);
	INTC_ClearVectAddr();
}


// [Interrupt service routine for RTC Alarm]
static __irq void Isr_RtcAlarm(void)
{
	DbgSysc(("RTC Alarm ISR\n"));
	Disp("RA\n");

	// pending clear
	RTC_ClearPending(ALARM); 
//	Outp32(0xE2800030, 2); // RTC alarm pending clear
	INTC_ClearVectAddr();
}

// [Interrupt service routine for RTC Tick]
static __irq void Isr_RtcTick(void)
{
	DbgSysc(("RTC Tick  ISR\n"));

	// for aging test
	DisplayLED(0);
	DisplayLED(2);
	DisplayLED(0);		

	// pending clear
	RTC_ClearPending(TimeTic); 
//	Outp32(0xE2800030, 1); // RTC tick pending clear
	INTC_ClearVectAddr();
}

// [Interrupt service routine for time-out]
static __irq void Isr_SystemTimer(void) 
{
	u32 uTmp0;
	
	DbgSysc(("System Timer ISR\n"));
//	Disp("ST\n");

	// for aging test
	DisplayLED(0);
	DisplayLED(2);
	DisplayLED(0);		

	// temporary ..
	uTmp0 = Inp32(0xE2600020);
	
	if (uTmp0 & (1<<1))
	{
		uTmp0 |= (1<<1);	      //  Interrupt count expired interrupt
	}
	else if (uTmp0 & (1<<2))
	{
		uTmp0 |= (1<<2);	      //  TICNTB write interrupt
		DbgSysc(("\nIsr_TICNTB \n"));
	}
	else if (uTmp0 & (1<<3))
	{
		uTmp0 |= (1<<3);	      //  TFCNTB write interrupt
		DbgSysc(("\nIsr_TFCNTB \n"));
	}
	else if (uTmp0 & (1<<4))
	{
		uTmp0 |= (1<<4);	      //  ICNTB write interrupt
		DbgSysc(("\nIsr_ICNTB \n"));
	}
	else if (uTmp0 & (1<<5))
	{
		uTmp0 |= (1<<5);	      //  TCON write interrupt
		DbgSysc(("\nIsr_TCON \n"));
	}
			
	// pending clear
	// temporary ..
	Outp32(0xE2600020,uTmp0);       //  Timer0 Interrupt Clear register	
	INTC_ClearVectAddr();

}

// [Interrupt service routine for Touch Screen]
static __irq void Isr_Ts0Pendown()
{
	DbgSysc(("Touch screen0 ISR\n"));
	Disp("TS0\n");

	ADCTS_SetADCClearWKInt();
	INTC_ClearVectAddr();
}

// [Interrupt service routine for Touch Screen]
static __irq void Isr_Ts1Pendown()
{
	DbgSysc(("Touch screen1 ISR\n"));
	Disp("TS1\n");

//	ADCTS_SetADCClearWKInt();
	INTC_ClearVectAddr();
}

SDHC SDHC_card[4];
// [Interrupt service routine for MMC0]
static __irq void Isr_Mmc0()
{
	DbgSysc(("MMC0 ISR\n"));
	Disp("M0\n");

//	SDHC_InterruptHandler(&SDHC_card[0]);

	// interrupt pending clear
	Outp8(0xEB00002B, 0xF); // SDHC_WAKEUP
	Outp16(0xEB000030, 0x0); // bit 6,7	
	Outp16(0xEB000030, 0x1C0); // bit 6,7
	INTC_ClearVectAddr();
}

// [Interrupt service routine for MMC1]
static __irq void Isr_Mmc1()
{
	DbgSysc(("MMC1 ISR\n"));
	Disp("M1\n");

//	SDHC_InterruptHandler(&SDHC_card[1]);

	// interrupt pending clear
	Outp8(0xEB10002B, 0xF); // SDHC_WAKEUP
	Outp16(0xEB100030, 0x0); // bit 6,7	
	Outp16(0xEB100030, 0x1C0); // bit 6,7
	INTC_ClearVectAddr();
}

// [Interrupt service routine for MMC2]
static __irq void Isr_Mmc2()
{
	DbgSysc(("MMC2 ISR\n"));
	Disp("M2\n");

//	SDHC_InterruptHandler(&SDHC_card[2]);

	// interrupt pending clear
	Outp8(0xEB20002B, 0xF); // SDHC_WAKEUP
	Outp16(0xEB200030, 0x0); // bit 6,7	
	Outp16(0xEB200030, 0x1C0); // bit 6,7
	INTC_ClearVectAddr();
}

// [Interrupt service routine for MMC3]
static __irq void Isr_Mmc3()
{
	DbgSysc(("MMC3 ISR\n"));
	Disp("M3\n");

//	SDHC_InterruptHandler(&SDHC_card[3]);

	// interrupt pending clear
	Outp8(0xEB30002B, 0xF); // SDHC_WAKEUP
	Outp16(0xEB300030, 0x0); // bit 6,7	
	Outp16(0xEB300030, 0x1C0); // bit 6,7
	INTC_ClearVectAddr();
}

// [Interrupt service routine when KEYIF processing done]
static __irq void Isr_KeyifProcessDone(void)
{
	DbgSysc(("KEYIF ISR\n"));
	Disp("KY\n");

	KEYPAD_STSCLR(0x3fff,0x3fff);		
	INTC_ClearVectAddr();
}

// [Interrupt service routine for MMC3]
static __irq void Isr_PwmTimer4()
{
	u32 uTmp;

	DbgSysc(("PWM Timer ISR\n"));
	Disp("PT4\n");

	// Temporary..
	uTmp = Inp32(0xE2500044); // included interrupt enable bit
	uTmp = (uTmp & ~(0x1f<<5))|(1<<9);
	Outp32(0xE2500044, uTmp);       //  Timer4 Interrupt Clear register
	INTC_ClearVectAddr();
}


// can't test in FPGA
void SYSCT_SelectPwrGating(void)
{
	u32 uSelPwrGatingBlock, uSelOnOff;
	u32 uNormalCfgtReg, uBlockPwrStatReg;
	SUB_BLOCK eBlock;
	bool bOn;
	u32 uSelMTCCnt, i;

	while(1)
	{
		Disp("* Select block power gating :\n 1.MFC, 2.G3D, 3.LCD, 4.TV, 5.CAM, 6.AUDIO, 7.IROM, 8.All, 0.Exit(D)\n");
		Disp(" Loop this setting function until exiting for selecting several blocks\n");
		uSelPwrGatingBlock = UART_GetIntNum();

		if ( (uSelPwrGatingBlock<1)||(uSelPwrGatingBlock>8) ) break;

		Disp("* Select : 1.On(D), 2.Off\n");
		uSelOnOff = UART_GetIntNum();
		if (uSelOnOff ==2) 
			bOn = FALSE;
		else 
			bOn= TRUE;

		Disp("Enter MTC Stable Counter [0~0xF(D)] : \n");
		uSelMTCCnt = UART_GetIntNum();
		if (uSelMTCCnt > 0xF) 
			uSelMTCCnt = 0xF;

		if (uSelMTCCnt==7)
			Disp("TOP block has same counter value~\n ");

		SYSC_SetMtcStableCounter(uSelPwrGatingBlock, uSelMTCCnt); // reset value = 0xF
		SYSC_SetBlockPowerGating(uSelPwrGatingBlock, bOn);

		for (i=0;i<1000;i++)
		{
			if (uSelPwrGatingBlock==6) 
				uSelPwrGatingBlock = 7; // audio bit is 7
			uBlockPwrStatReg = SYSC_GetBlockPowerStatusReg();
			uBlockPwrStatReg = (uBlockPwrStatReg>>uSelPwrGatingBlock)&1;
			if ( uBlockPwrStatReg==bOn )
			{
				Disp(" Transition loop count = %d\n", i);
				break;
			}
		}
		
		uNormalCfgtReg = SYSC_GetNormalModeConfigReg();
		Disp(" Normal mode config = 0x%x\n", uNormalCfgtReg);
		uBlockPwrStatReg = SYSC_GetBlockPowerStatusReg();
		Disp(" Block power status = 0x%x\n", uBlockPwrStatReg);
	}
}


// [Test normal mode config]
void SYSCT_TestNormalMode(void)
{
	u32 uSelMTCCnt;	
	SUB_BLOCK eSubTemp;

	// Clock gating
	SYSCT_SelectClkGating();
	
	// Power gating
	SYSCT_SelectPwrGating();

	// Test each IP
	Disp(" Return to main menu for each IP test..\n");	
}


void SYSCT_TestSimplePowerMode(void)
{
	u32 uSelMode, uSelAging, uSelAgingSrc;
	u32 uInformReg, uAgingCnt;
	u8	ucSelTop, ucSelL2, ucSelOsc;
	bool bEnableOsc = FALSE;
	POWER_MODE ePwrMode;
	WAKEUP_ID eWakeupSrc, eEintWakeup;
	u32 uEintNum;

	// Set wakeup interrupt for GPIO as EINT
#ifdef FPGA
	INTC_SetVectAddr(NUM_EINT1, Isr_EInt1);
	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_1, eGINT);
	GPIO_SetPullUpDownEach(eGPIO_H0, eGPIO_1, eGPUDdis);
	GPIO_SetEint30(1, Both_Edge, eDLYFLT, 0x1f); // EINT1, falling edge, enable filter, filter width
	GPIO_EINT30UnMask(1);
	eEintWakeup = eWAKEUP_EINT1;
	uEintNum = NUM_EINT1;
#else
	INTC_SetVectAddr(NUM_EINT4, Isr_EInt4);
	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_4, eGINT);
	GPIO_SetPullUpDownEach(eGPIO_H0, eGPIO_4, eGPUDdis);
	GPIO_SetEint30(4, Both_Edge, eDLYFLT, 0x1f); // EINT1, falling edge, enable filter, filter width
	GPIO_EINT30UnMask(4);
	eEintWakeup = eWAKEUP_EINT4;
	uEintNum = NUM_EINT4;
#endif
	INTC_SetVectAddr(NUM_RTC_TICK, Isr_RtcTick); // for aging test 

	// Set all wakeup source 
	SYSC_SetWakeupMask(eWAKEUP_ALL , TRUE); // mask all wakeup sources

	uInformReg = SYSC_GetInformReg(5); // init inform reg for power mode & aging
	uInformReg &= ~(0xFF); // clear inform register for aging

	SYSC_SetSleepGPIO(); // use deep(top off/retention) & sleep mode

	// Select mode configurations
	Disp("Select Test mode : 1.Idle(D), 2.DeepIdle, 3.Stop, 4.DeepStop, 5.Sleep, 6.Idle WFI only\n");
	uSelMode = UART_GetIntNum();

	if ( (uSelMode==2)||(uSelMode==4) )
	{
		Disp("Select Top module status(DeepIdle/DeepStop only) : 0.Off, 1.Retention(D), 2.On \n");
		ucSelTop = (u8)UART_GetIntNum();
	}
	else
		ucSelTop = 1;

	if ( (uSelMode==3)||(uSelMode==4)||(uSelMode==5) )
	{
		Disp("Select OSC enable : 1.Enable(D), 2.Disable \n");
		if( UART_GetIntNum() != 2) 
			bEnableOsc = TRUE;
		else 
			bEnableOsc = FALSE;
	}
	
	if ((ucSelTop!=0)) 
		ucSelL2 = 1;
	else 
	{
		ucSelTop = 1;
		ucSelL2 = 0;
	}

	if ((ucSelTop>2)||(ucSelTop<1)) 
		ucSelTop = 1; // default retention

	Disp("Select Aging : 1.General(D), 2.Aging on this mode, 3.Aging on random mode(RTCt), 4.Aging as early wakeup(RTCt)\n");
	uSelAging = UART_GetIntNum();

	Disp("Select Wakeup source : 1.RTC tick(D), 2.EINT\n");
	uSelAgingSrc = UART_GetIntNum();

	// Implement configuration
	switch(uSelMode)
	{
		case 2 :
			ePwrMode = eMODE_DEEPIDLE;
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
			SYSC_SetIdleModeConfig(ucSelTop, ucSelTop, ucSelL2, eCFG_DEEP);
			uInformReg |= INFORM_DIDLE;	// 0x2x = deep idle mode
			break;
		case 3 :
			ePwrMode = eMODE_STOP;
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
			SYSC_SetStopModeConfig(eCFG_ON, (u8)eCFG_RET, ucSelL2, eCFG_NODEEP);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, bEnableOsc);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, bEnableOsc);
			uInformReg |= INFORM_STOP;	// 0x3x = stop mode
			break;
		case 4 :
			ePwrMode = eMODE_DEEPSTOP;
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
			SYSC_SetStopModeConfig(ucSelTop, (u8)eCFG_RET, ucSelL2, eCFG_DEEP);
			SYSC_SetStopMemConfig(eMEMCFG_ALL, ucSelL2);			
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, bEnableOsc);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, bEnableOsc);
			uInformReg |= INFORM_DSTOP;	// 0x4x = deep stop mode
			break;
		case 5 :
			ePwrMode = eMODE_SLEEP;
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, bEnableOsc);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, bEnableOsc);
			uInformReg |= INFORM_SLEEP;	// 0x5x = sleep mode
			break;
		default:
			ePwrMode = eMODE_IDLE;
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
			SYSC_SetIdleModeConfig(eCFG_ON, ucSelTop, ucSelL2, eCFG_NODEEP);
			uInformReg |= INFORM_IDLE;	// 0x1x = Idle mode
			break;
	}

	if (uSelAgingSrc != 2) // defualt RTC tick
	{
		RTC_StartTickTime(CLK_32768Hz, 0x40);
		SYSC_SetWakeupMask(eWAKEUP_RTCTICK, FALSE);
	}
	else
	{
		SYSC_SetWakeupMask(eEintWakeup, FALSE);
	}
	
	// write aging and mode info. to infrom register 
	if (uSelAging==2) // aging test
	{
		uAgingCnt = 0;
		SYSC_SetInformReg(6, uAgingCnt); // count aging test
		SYSC_SetInformReg(5, uInformReg|INFORM_AGING);  
		// Set wakeup source as RTC Tick for Aging Test
		SYSC_SetClkOut(eCLKOUT_TICK, 1); // CLKOUT pin out = RTC tick

		if ( (ePwrMode==eMODE_IDLE)||(ePwrMode==eMODE_STOP) )
		{
			// Set bigger value as RTC tick count than deep & sleep mode
			RTC_StartTickTime(CLK_16384Hz, 0x500);
		}

		if ( (ePwrMode==eMODE_SLEEP)||((ePwrMode==eMODE_DEEPSTOP)&&(ucSelTop==0)) )
			Outp32(0xD0030000, 0x0); // remove iRAM start address for verifying BL1 copy operation
	}
	else if (uSelAging==3) // random mode aging test 
	{
		if ( (ePwrMode==eMODE_IDLE)||(ePwrMode==eMODE_STOP) ) // first sleep mode
		{
			ePwrMode = eMODE_SLEEP;
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, FALSE);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, FALSE);
		}
		SYSC_SetInformReg(6, 0); // count aging test
		SYSC_SetInformReg(5, uInformReg|INFORM_AGING_R);  
	}
	else if (uSelAging==4)
	{
		SYSC_SetInformReg(6, 0); // count aging test
		SYSC_SetInformReg(5, uInformReg|INFORM_AGING_E);  			
	}
	else // general test
	{

		SYSC_SetInformReg(5, uInformReg|INFORM_GENERAL); 
	}

	if ( (ePwrMode==eMODE_IDLE)||(ePwrMode==eMODE_STOP) )
	{
		INTC_Enable(NUM_RTC_TICK); // Set interrupt controller enable
		INTC_Enable(uEintNum); // Set interrupt controller enable
		Disp(" Idle & Stop mode are supported in aging mode only.. (exit manually) \n");
	}

	// Stable counter setting. 
	// OSC stable => when OSC disabled in STOP/DEEPSTOP/SLEEP modes
	// PWR stable => when Internal power off in SLEEP modes
	// Clamp stable => when ARM power off in DEEPIDLE/DEEPSTOP/SLEEP modes
	SYSC_SetOscStableCounter(0xFFF); // reset value = 0xFFFFF
	SYSC_SetPowerStableCounter(0xFFF); // reset value = 0xFFFFF
	SYSC_SetClampStableCounter(TRUE, 0xFF); // reset value = 0x3FF
	SYSC_SetClampStableCounter(FALSE, 0xFF); // reset value = 0x3FF

	Disp(" Check CLKOUT Idle & DeepIdle => ARMCLK/8, Stop & DeepStop & Sleep => HCLKMSYS/2\n");
	Disp(" Press Enter Key to enter selected power mode!! \n");
	UART_GetIntNum();

	while(UART_GetKey() == 0) 
	{
		Disp(" Enter Power mode!!\n");

		// triggering point for debugging
		DisplayLED(0);		
		DisplayLED(1);
		Delay(10);
		DisplayLED(0);	
		
		if ((ePwrMode == eMODE_IDLE)||(ePwrMode == eMODE_STOP))
		{
			if (uSelAging==4) // early wakeup aging test
			{
				// RTC tick config
				RTC_StartTickTime(CLK_32768Hz, 0x2);
				Delay((uAgingCnt+1)%100); 
			}
		}
		
		// Enter power mode
//		SYSC_StartPowerMode(ePwrMode, SW_WFI, USER_CONFIG); // selected power mode, selected enterfing method, Set user config
		if (uSelMode==6)
			SYSC_SetConfigStandbyWFI(eMODE_NORMAL);
		else
			SYSC_SetConfigStandbyWFI(ePwrMode);
		SYSC_DisableSyscInt();
		__wfi();
			
		// loop for Idle & Stop mode
		SYSC_ReleaseRetentionIO(eRET_BLOCK_ALL); // when top off

		Disp("\n Aging Count = %d\n", uAgingCnt);
		uAgingCnt++;

		if (ePwrMode == eMODE_IDLE) 	
			Disp(" IDLE_CFG = 0x%x\n", SYSC_GetIdleModeConfigReg());
		else if (ePwrMode == eMODE_STOP) 
			Disp(" STOP_CFG = 0x%x\n", SYSC_GetStopModeConfigReg());

		eWakeupSrc = (WAKEUP_ID)SYSC_GetWakeupStatus(FALSE);
		Disp(" Wakeup source(bit) = %d\n", (u32)eWakeupSrc);
		if (eWakeupSrc != eWAKEUP_OTHERS)
			SYSC_ClearWakeupStatus(eWakeupSrc);
		else
			Disp(" Wakeup by another wakeup sources!! Not dedicated wakeup source.\n");
		
		Disp("\nPress any key to exit the this test, otherwise Entering same power mode !! \n");	
		Delay(2000); 
		Disp("Enter to same power mode\n");		
	}	

	INTC_Disable(NUM_RTC_TICK); // Set interrupt controller disable
	INTC_Disable(uEintNum); // Set interrupt controller disable
	RTC_TickTimeEnable(false);

	Disp("Return to Normal Mode\n");
}


void SYSCT_SetOptimalStableCounter(void)
{
	// Set optimal stable counters
	SYSC_SetOscFreq(0x6); // 250ns, >200ns
	SYSC_SetOscStableCounter(0xFFFF); // 2.7ms, >2ms @ variable for crystal type
	SYSC_SetPowerStableCounter(0xFFF); // 170us, >50us@8698C
	SYSC_SetMtcStableCounter(eSUB_BLOCK_ALL, 0xF); // 625ns, max value
	SYSC_SetClampStableCounter(TRUE, 0xFF); // Clamp on, 10us, >5us spec.
	SYSC_SetClampStableCounter(FALSE, 0xFF); // Clamp off, 10us, >5us spec.
}


void SYSCT_RandomAgingPowerMode(void)
{
	POWER_MODE eAgingPwrMode;
	u32 uSelMode, uSelAging, uSelMethod;
	u32 uAgingCnt;
	u8	ucSelTop, ucSelL2;
	bool bMainOscEn, bUsbOscEn;

	while(1) 
	{
		Disp("\nPress any key to exit the aging test, otherwise Entering random power mode !! \n");	
		Delay(2000); 
		if (UART_GetKey()) return;

		// RTC tick config
#ifndef EVT0	
		RTC_StartTickTime(CLK_16384Hz, 0x10);
#else // avoid early wakeup	
		RTC_StartTickTime(CLK_1Hz, 0x2); // 2 sec, stable wakeup test
#endif // EVT0
		INTC_SetVectAddr(NUM_RTC_TICK, Isr_RtcTick); // for aging test 

#if 1
		// Set optimal stable counters
		SYSCT_SetOptimalStableCounter();
#endif 

		// Count aging test
		uAgingCnt = SYSC_GetInformReg(6);
		SYSC_SetInformReg(6, uAgingCnt+1);
		Disp(" Aging Count = %d\n\n", uAgingCnt);

		// Select mode configurations
		// Test mode : 1.Idle(D), 2.DeepIdle, 3.Stop, 4.DeepStop, 5.Sleep
		uSelMode =  uAgingCnt*rand() %6;

		// Top module status(DeepIdle/DeepStop only) : 0.Off, 1.Retention(D), 2.On
		ucSelTop = uAgingCnt*rand() %3;

		// OSC enable or disable
#ifndef EVT0		
		bMainOscEn = (uAgingCnt*rand()%3==1) ? TRUE : FALSE;
		bUsbOscEn = (uAgingCnt*rand()%4==1) ? TRUE : FALSE;
#else // avoid wakeup filter error
		bMainOscEn = FALSE;
		bUsbOscEn = FALSE;
#endif // EVT0

		if ((ucSelTop!=0)) 
			ucSelL2 = 1;
		else 
			ucSelL2 = 0;

		if ((ucSelTop>2)||(ucSelTop<1)) 
			ucSelTop = 1; // default retention

		Disp(" Mode num = %d, Top num = %d\n", uSelMode, ucSelTop);

		// Implement configuration
		switch(uSelMode)
		{
			case 2 :
				Disp(" Enter Deep Idle mode!\n");
				eAgingPwrMode = eMODE_DEEPIDLE;
				SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
				SYSC_SetIdleModeConfig(ucSelTop, ucSelTop, ucSelL2, eCFG_DEEP);
				break;
			case 3 :
				Disp(" Enter Stop mode!\n");
				eAgingPwrMode = eMODE_STOP;
				SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
				SYSC_SetStopModeConfig(ucSelTop, (u8)eCFG_RET, ucSelL2, eCFG_NODEEP);
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_MAIN, bMainOscEn);
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_USB, bUsbOscEn);
				break;
			case 4 :
				Disp(" Enter Deep Stop mode!\n");
				eAgingPwrMode = eMODE_DEEPSTOP;
				SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
				SYSC_SetStopModeConfig(ucSelTop, (u8)eCFG_RET, ucSelL2, eCFG_DEEP);
#ifndef EVT0	
				SYSC_SetStopMemConfig(eMEMCFG_ALL, ucSelL2);			
#else // avoid iRom deepstop
				SYSC_SetStopMemConfig(eMEMCFG_ALL, TRUE);			
#endif // EVT0
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_MAIN, bMainOscEn);
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_USB, bUsbOscEn);
				break;
			case 5 :
				Disp(" Enter Sleep mode!\n");
				eAgingPwrMode = eMODE_SLEEP;
				SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_MAIN, bMainOscEn);
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_USB, bUsbOscEn);
				break;
			default:
				Disp(" Enter Idle mode!\n");
				eAgingPwrMode = eMODE_IDLE;
				SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
				SYSC_SetIdleModeConfig(ucSelTop, ucSelTop, ucSelL2, eCFG_NODEEP);
				break;
		}

		// enable interrupt		
		INTC_Enable(NUM_RTC_TICK); // Set interrupt controller enable
#ifdef FPGA
		INTC_Enable(NUM_EINT1); // Set interrupt controller enable
#else
		INTC_Enable(NUM_EINT4); // Set interrupt controller enable
#endif // FPGA
		// triggering point for debugging
		DisplayLED(0);		
		DisplayLED(1);
		Delay(10);
		DisplayLED(0);		

		if (eAgingPwrMode != eMODE_SLEEP)
			SYSC_StartPowerMode(eAgingPwrMode, SW_WFI, USER_CONFIG); // selected power mode, selected enterfing method, Set user config
		else // sleep mode
#ifndef EVT0	
			SYSC_StartPowerMode(eAgingPwrMode, SW_WFI, USER_CONFIG); // selected power mode, selected enterfing method, Set user config
#else // avoid wakeup source fail
			SYSC_StartPowerMode(eAgingPwrMode, HW_WFI, USER_CONFIG); // selected power mode, selected enterfing method, Set user config
#endif // EVT0

		if (eAgingPwrMode == eMODE_IDLE) 	
			Disp(" IDLE_CFG = 0x%x\n", SYSC_GetIdleModeConfigReg());
		else if (eAgingPwrMode == eMODE_STOP) 
			Disp(" STOP_CFG = 0x%x\n", SYSC_GetStopModeConfigReg());		
	}	
}

void SYSCT_EarlyWakeup(void)
{
	u32 uInformReg, uAgingCnt;
	POWER_MODE eAgingPwrMode;
	u8 ucSelTop, ucSelL2;

	while(1) 
	{
		// Waiting for test
		Disp("\nPress any key to exit the aging test, otherwise Entering power mode for early wakeup aging !! \n");	
		Delay(2000);
		if (UART_GetKey()) return;

		uInformReg = SYSC_GetInformReg(5) & (0xF0);

		// Count aging test
		uAgingCnt = SYSC_GetInformReg(6);
		SYSC_SetInformReg(6, uAgingCnt+1);
		Disp(" Aging Count = %d\n\n", uAgingCnt);

		// RTC tick config
		RTC_TickTimeEnable(false);
		RTC_ClkReset(1);
		RTC_ClkReset(0);
		RTC_TickClkSelect(CLK_32768Hz);
		RTC_TickCnt(2);

		// GPIO config
		DisplayLED(0);

		// Clock out as ARMCLK
		SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 1); 

		//	Disp("Select Top module status(DeepIdle/DeepStop only) : 0.Off, 1.Retention(D), 2.On \n");
		ucSelTop = uAgingCnt*rand() %3;

		if ((ucSelTop!=0)) 
			ucSelL2 = 1;
		else 
			ucSelL2 = 0;

		if ((ucSelTop>2)||(ucSelTop<1)) 
			ucSelTop = 1; // default retention

		Disp(" Mode num = 0x%x, Top num = %d\n", uInformReg, ucSelTop);

		// Implement configuration
		switch(uInformReg)
		{
			case INFORM_DIDLE :
				Disp(" Enter Deep Idle mode!\n");
				eAgingPwrMode = eMODE_DEEPIDLE;
				SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
				SYSC_SetIdleModeConfig(ucSelTop, ucSelTop, ucSelL2, eCFG_DEEP);
				break;
			case INFORM_STOP :
				Disp(" Enter Stop mode!\n");
				eAgingPwrMode = eMODE_STOP;
				SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
				SYSC_SetStopModeConfig(eCFG_ON, (u8)eCFG_RET, ucSelL2, eCFG_NODEEP);
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_MAIN, true);
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_USB, true);
				break;
			case INFORM_DSTOP :
				Disp(" Enter Deep Stop mode!\n");
				eAgingPwrMode = eMODE_DEEPSTOP;
				SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
				SYSC_SetStopModeConfig(ucSelTop, (u8)eCFG_RET, ucSelL2, eCFG_DEEP);
				SYSC_SetStopMemConfig(eMEMCFG_ALL, ucSelL2);			
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_MAIN, true);
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_USB, true);
				break;
			case INFORM_SLEEP :
				Disp(" Enter Sleep mode!\n");
				eAgingPwrMode = eMODE_SLEEP;
				SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); 
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_MAIN, true);
				SYSC_SetOscPadOnOff(eAgingPwrMode, eOSC_USB, true);
				Outp32(0xD0030000, 0x0); // remove iRAM start address for test
				break;
			default:
				Disp(" Enter Idle mode!\n");
				eAgingPwrMode = eMODE_IDLE;
				SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
				SYSC_SetIdleModeConfig(ucSelTop, ucSelTop, ucSelL2, eCFG_NODEEP);
				break;
		}

		// enable interrupt		
		INTC_Enable(NUM_RTC_TICK); // Set interrupt controller enable
#ifdef FPGA
		INTC_Enable(NUM_EINT1); // Set interrupt controller enable
#else
		INTC_Enable(NUM_EINT4); // Set interrupt controller enable
#endif // FPGA

		// Enter power mode 
		SYSC_SetConfigStandbyWFI(eAgingPwrMode);
		DisplayLED(1);
		Delay(10);
		DisplayLED(0);		
		RTC_TickTimeEnable(true);
		Delay((uAgingCnt+1)%100); 
		
		SYSC_DisableSyscInt();
		__wfi();
	}
}

void SYSCT_CheckAgingWakeupReset(void)
{
	u32 uIdleCfgReg, uStopCfgReg, uStopMemReg, uRstStatReg, uWakeupStatReg;
	u32 uInformReg, uAgingCnt;
	u32 uDramtemp;
	WAKEUP_ID eWakeupSrc, eEintWakeup;
	u32 uEintNum;

	SYSC_ReleaseRetentionIO(eRET_BLOCK_ALL);
	
	// Display power down mode register information
	uIdleCfgReg = SYSC_GetIdleModeConfigReg();
	uStopCfgReg = SYSC_GetStopModeConfigReg();
	uStopMemReg = SYSC_GetStopMemConfigReg();
	uRstStatReg = SYSC_GetResetStatusReg();
	uWakeupStatReg = SYSC_GetWakeupStatus(TRUE);
	uInformReg = SYSC_GetInformReg(5);
	uAgingCnt = SYSC_GetInformReg(6);
	Disp("\nIDLE_CFG = 0x%x\n", uIdleCfgReg);
	Disp("STOP_CFG = 0x%x\n", uStopCfgReg);
	Disp("STOP_MEM_CFG = 0x%x\n", uStopMemReg);
	Disp("RESET_STAT = 0x%x\n", uRstStatReg);
	Disp("WAKEUP_STAT = 0x%x\n", uWakeupStatReg);
	Disp("INFORM5 = 0x%x\n", uInformReg);
	Disp("INFORM6 = 0x%x\n", uAgingCnt);

	// Clear wakeup status..
	SYSC_ClearWakeupStatus(eWAKEUP_ALL);
	
	// Set wakeup interrupt for GPIO as EINT
#ifdef FPGA
	INTC_SetVectAddr(NUM_EINT1, Isr_EInt1);
	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_1, eGINT);
	GPIO_SetPullUpDownEach(eGPIO_H0, eGPIO_1, eGPUDdis);
	GPIO_SetEint30(1, Both_Edge, eDLYFLT, 0x1f); // EINT1, falling edge, enable filter, filter width
	GPIO_EINT30UnMask(1);
	eEintWakeup = eWAKEUP_EINT1;
	uEintNum = NUM_EINT1;
#else
	INTC_SetVectAddr(NUM_EINT4, Isr_EInt4);
	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_4, eGINT);
	GPIO_SetPullUpDownEach(eGPIO_H0, eGPIO_4, eGPUDdis);
	GPIO_SetEint30(4, Both_Edge, eDLYFLT, 0x1f); // EINT1, falling edge, enable filter, filter width
	GPIO_EINT30UnMask(4);
	eEintWakeup = eWAKEUP_EINT4;
	uEintNum = NUM_EINT4;
#endif
	INTC_SetVectAddr(NUM_RTC_TICK, Isr_RtcTick); // for aging test 

	// enable interrupt		
	INTC_Enable(NUM_RTC_TICK); // Set interrupt controller enable
	INTC_Enable(uEintNum); // Set interrupt controller enable

	// Set all wakeup source 
	SYSC_SetWakeupMask(eWAKEUP_ALL , TRUE); // mask all wakeup sources
	SYSC_SetWakeupMask(eEintWakeup , FALSE); // unmask wakeup sources
	SYSC_SetWakeupMask(eWAKEUP_RTCTICK , FALSE); // unmask wakeup sources

	// GPIO PDN reg. setting in sleep & deep mode
	SYSC_SetSleepGPIO(); // use deep(top off/retention) & sleep mode

	// Set CLKOUT to ARMCLK/4. 
	SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 1); // CLKOUT pin out = ARMCLK/4

	// Check inform 5 register for aging
	uInformReg &= 0xFF;

#if 1
	// Set optimal stable counters
	SYSCT_SetOptimalStableCounter();
#endif 

	// Check aging test for random or early wakeup aging test first..
	if ((uRstStatReg==eRST_STAT_DEEPIDLE_WU)||(uRstStatReg==eRST_STAT_DEEPSTOP_WU)||(uRstStatReg==eRST_STAT_SLEEP_WU))
	{
		if ((uInformReg&0xF) == INFORM_AGING_R)
		{
			SYSCT_RandomAgingPowerMode();
			return;
		}
		
		if ((uInformReg&0xF) == INFORM_AGING_E)
		{
			SYSCT_EarlyWakeup();
			return;
		}
	}
	
	// Check aging test for each wakeup & reset
	switch(uRstStatReg)
	{
	case eRST_STAT_DEEPIDLE_WU :
		Disp("\nDeepIdle wakeup\n");
		if (uInformReg == (INFORM_DIDLE|INFORM_AGING))
		{
			Disp("\nPress any key to exit the aging test, otherwise Entering DeepIdle for aging !! \n");	
			Delay(2000);
			if (UART_GetKey()) break;
			Disp("Enter Deep Idle mode\n");
			SYSC_SetInformReg(6, uAgingCnt+1);
#if 0 // temporary test for stable wakeup
		RTC_StartTickTime(CLK_4Hz, 8);	// 2 sec.
		DisplayLED(0);		
		DisplayLED(1);
		Delay(10);
		DisplayLED(0);		
#endif
			SYSC_StartPowerMode(eMODE_DEEPIDLE, SW_WFI, USER_CONFIG); // use previous user config
		}
		break;
	case eRST_STAT_DEEPSTOP_WU :
		Disp("\nDeepStop wakeup\n");
		if (uInformReg == (INFORM_DSTOP|INFORM_AGING))
		{
			Disp("\nPress any key to exit the aging test, otherwise Entering DeepStop for aging !! \n");	
			Delay(2000);
			if (UART_GetKey()) break;
			Disp("Enter Deep Stop mode\n");
			SYSC_SetInformReg(6, uAgingCnt+1);
#if 0 // temporary test for stable wakeup
		RTC_StartTickTime(CLK_4Hz, 8);
		DisplayLED(0);		
		DisplayLED(1);
		Delay(10);
		DisplayLED(0);		
#endif
			SYSC_StartPowerMode(eMODE_DEEPSTOP, SW_WFI, USER_CONFIG); // use previous user config
		}	
		break;
	case eRST_STAT_SLEEP_WU :
		Disp("\nSleep wakeup\n");
		if (uInformReg == (INFORM_SLEEP|INFORM_AGING))
		{
			Disp("\nPress any key to exit the aging test, otherwise Entering Sleep for aging !! \n");	
			Delay(2000);
			if (UART_GetKey()) break;
			Disp("Enter Sleep mode\n");
			SYSC_SetInformReg(6, uAgingCnt+1);
			Outp32(0xD0030000, 0x0); // remove iRAM start address for test
#if 0 // temporary test for stable wakeup
		RTC_StartTickTime(CLK_4Hz, 8);
		DisplayLED(0);		
		DisplayLED(1);
		Delay(10);
		DisplayLED(0);		
		SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 1); // CLKOUT pin out = HCLK Dsys
#endif
			SYSC_StartPowerMode(eMODE_SLEEP, HW_WFI, USER_CONFIG);  // use previous user config 
		}
		break;
	case eRST_STAT_SW_RST :
		Disp("\nS/W reset up!! \n");
		if (uInformReg == (INFORM_SWRST|INFORM_AGING))
		{
			Disp("\nPress any key to exit the aging test, otherwise Assert S/W reset for aging!! \n");	
			Delay(2000);
			if (UART_GetKey()) break;
			SYSC_SetInformReg(6, uAgingCnt+1);
			SYSC_SwReset();
		}
		break;
	case eRST_STAT_WARM_RST :
		Disp("\nWarm reset up(nWRESET)!! \n");
		if (uInformReg == (INFORM_WARMRST|INFORM_AGING))
		{
			Disp("\nPress any key to exit the aging test, otherwise Push warm reset button or input with pulse generator to warm reset pin for aging!! \n");	
			Delay(2000);
			if (UART_GetKey()) break;
			SYSC_SetInformReg(6, uAgingCnt+1);

			if((uAgingCnt*rand()%3)==1)
			{
				Disp("Bus Down !!\n");
				// bus down situation
				SYSC_SetClkGatebyIP(eCLKGATE_IP_DMC0, 0);
				SYSC_SetClkGatebyIP(eCLKGATE_IP_DMC1, 0);
				uDramtemp = *((u32 *)0x20000000);		
			}
		}	
		break;
	case eRST_STAT_WDT_RST :
		Disp("\nWatchdog Timer reset up!! \n");
		if (uInformReg == (INFORM_WDTRST|INFORM_AGING))
		{
			Disp("\nPress any key to exit the aging test, otherwise Assert Watchdog Timer reset for aging!! \n");	
			Delay(2000);
			if (UART_GetKey()) break;
			SYSC_SetInformReg(6, uAgingCnt+1);
			// watchdog reset 
			WDT_Enable(1);
			WDT_EnableReset(1);			

			if((uAgingCnt*rand()%3)==1)
			{
				Disp("Bus Down !!\n");
				// bus down situation
				SYSC_SetClkGatebyIP(eCLKGATE_IP_DMC0, 0);
				SYSC_SetClkGatebyIP(eCLKGATE_IP_DMC1, 0);
				uDramtemp = *((u32 *)0x20000000);		
			}
		}	
		break;
	case eRST_STAT_HW_RST :
		Disp("\nPower on reset or H/W reset(nRESET)!! \n");
		break;
	default :
		Disp("This reset status(0x%x) is wrong value in RST_STAT register\n");
		break;
	}

}


// [Set Wakeup Source]
void SYSCT_SelectWakeupSource(POWER_MODE ePwrMode)
{
	u32 uSel, i;
	u32 uLastIdx;
	WAKEUP_ID eWakeupMask;
	static u32 uEIntDetectType;
	static u32 uWakeupSrc;

	// Mask all wakeup source
	SYSC_SetWakeupMask(eWAKEUP_ALL , TRUE); // mask all wakeup sources
	// Disable all interrupt source
	INTC_Disable(NUM_ALL);	
	
	while (1)
	{
		// Init uLastIdx
		uLastIdx = 
			(ePwrMode == eMODE_IDLE) ? 16 :
			(ePwrMode == eMODE_DEEPIDLE) ? 16 :
			(ePwrMode == eMODE_STOP) ? 14 :
			(ePwrMode == eMODE_DEEPSTOP) ? 14 :
			(ePwrMode == eMODE_SLEEP) ? 6 : 14;	

		// Select wake up sources.. 
		Disp("Select Wakeup sources:\n");
		Disp("1:EINT1, 2:EINT4, 3:EINT31, 4:RTC Alarm, 5:RTC Tick, 6:Keypad, 7:HDMI CEC\n"); // sleep mode wakeup sources
		if ((ePwrMode == eMODE_STOP) || (ePwrMode == eMODE_DEEPSTOP) ||
			(ePwrMode == eMODE_IDLE) || (ePwrMode == eMODE_DEEPIDLE))
		{
			Disp("8:Sys Timer, 9:Touch0 Pen Down, 10:Touch1 Pen Down, 11:MMC0, 12:MMC1, 13:MMC2, 14:MMC3,");
		}
		if ((ePwrMode == eMODE_IDLE) || (ePwrMode == eMODE_DEEPIDLE))
		{
			Disp("15:I2S0, 16:PWM Timer(Top on) \n");
		}
		Disp("20:All wakeup(D) 0: Exit\n");
		uSel = UART_GetIntNum();

		// decoding selected number
		if (uSel == 0)
		{
			return;
		}
		else if ( (uSel>=1)&&(uSel<=3) ) // EINT1, EINT4 or EINT31
		{
			uWakeupSrc = uSel;
			Disp("Select EINT Detect Type:\n");
			Disp("0: Low Level, 1: High Level, 2: Falling Edge, 3: Rising Edge, 4: Both Edge(D)\n");
			uSel = UART_GetIntNum();
			if (uSel>=0 && uSel<=4)
			{
				uEIntDetectType = uSel;
			}
			else
			{
				uEIntDetectType = 4; // default both edge
			}
		}
		else if ( (uSel>=4)&&(uSel<=uLastIdx) )
		{	
			uWakeupSrc = uSel;
		}
		else
		{
			uSel = 20; // default : all wakeup source
			Disp(" All wakeup sources are unmasked !!\n");
			SYSC_SetWakeupMask(eWAKEUP_ALL , FALSE); // mask all wakeup sources
			uEIntDetectType = 4;
		}
		SYSC_SetInformReg(3, uWakeupSrc);	// save last selected item
		
		// Initializing wakeup source function for selected item.
		if (uSel == 20)
			uWakeupSrc = 1; // init all wakeup source in followed for loop
		else 
			uLastIdx = uWakeupSrc; // once run. change uLastIdx value..

		for (i=uWakeupSrc; i<=uLastIdx ;i++)
		{
			switch(i)
			{
			case 1: // EINT1
				eWakeupMask = eWAKEUP_EINT1;
				GPIO_SetFunctionEach(eGPIO_H0, eGPIO_1, eGINT);
				GPIO_SetPullUpDownEach(eGPIO_H0, eGPIO_1, eGPUDdis);
				GPIO_SetEint30(1, uEIntDetectType, eDLYFLT, 0x1f); // EINT1, falling edge, enable filter, filter width
				GPIO_EINT30UnMask(1);
				INTC_SetVectAddr(NUM_EINT1, Isr_EInt1);
				INTC_Enable(NUM_EINT1); // Set interrupt controller enable
				break;
			case 2: // EINT4
				eWakeupMask = eWAKEUP_EINT4;
				GPIO_SetFunctionEach(eGPIO_H0, eGPIO_4, eGINT);
				GPIO_SetPullUpDownEach(eGPIO_H0, eGPIO_4, eGPUDdis);
				GPIO_SetEint30(4, uEIntDetectType, eDLYFLT, 0x1f); // EINT1, falling edge, enable filter, filter width
				GPIO_EINT30UnMask(4);
				INTC_SetVectAddr(NUM_EINT4, Isr_EInt4);
				INTC_Enable(NUM_EINT4); // Set interrupt controller enable
				break;
			case 3: // EINT31
				eWakeupMask = eWAKEUP_EINT31;
				GPIO_SetFunctionEach(eGPIO_H3, eGPIO_7, eGINT);
				GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_7, eGPUDdis);
				GPIO_SetEint33(7, uEIntDetectType, eDLYFLT, 0x1f); // EINT31, falling edge, enable filter, filter width
				GPIO_EINT33UnMask(7);
				INTC_SetVectAddr(NUM_EINT16_31, Isr_EInt31);
				INTC_Enable(NUM_EINT16_31); // Set interrupt controller enable
				break;
			case 4: // RTC Alarm
				eWakeupMask = eWAKEUP_RTCALARM;
				RTC_TimeInit(InitYEAR, InitMONTH, InitDATE, InitDAY, InitHOUR, InitMIN, InitSEC-5);
				RTC_AlarmTimeInit(AlmYear, AlmMon, AlmDate, AlmHour, AlmMin, AlmSec);
				RTC_AlarmEnable(TRUE, FALSE, FALSE, FALSE, FALSE, FALSE, TRUE);
				INTC_SetVectAddr(NUM_RTC_ALARM,Isr_RtcAlarm);
				INTC_Enable(NUM_RTC_ALARM);
				RTC_Print();
				Disp("After 10 sec., Alarm Interrupt Occur\n");
				break;
			case 5: // RTC Tick
				eWakeupMask = eWAKEUP_RTCTICK;
				RTC_StartTickTime(CLK_4Hz, 8);
				INTC_SetVectAddr(NUM_RTC_TICK, Isr_RtcTick);
				INTC_Enable(NUM_RTC_TICK); 
				break;
			case 6: // Keypad
				eWakeupMask = eWAKEUP_KEY;
				KEYPAD_WakeupInit(); // interrupt enable, handler setting
				break;			
			case 7: // HDMI CEC
				eWakeupMask = eWAKEUP_CEC;
//				INTC_SetVectAddr(NUM_HDMI, Isr_HdmiCec);
				INTC_Enable(NUM_HDMI); 
				break;			
			case 8: // System Timer
				eWakeupMask = eWAKEUP_ST;
				SysTimer_Config(0, 0, 2, 1, 15, 2000, 2000, 10, 1); // general div, integer div, XusbXTI, 1/2 div, 16 prescaler, cnt = 2000, autoreload
				SysTimer_Start();				
				INTC_SetVectAddr(NUM_SYSTIMER, Isr_SystemTimer);
				INTC_Enable(NUM_SYSTIMER); 
				break;
			case 9: // TSIF 0
				eWakeupMask = eWAKEUP_TS0;
				ADCTS_WakeUpInit(0);
				break;
			case 10: // TSIF 1
				eWakeupMask = eWAKEUP_TS1;
				ADCTS_WakeUpInit(1);
				break;
			case 11: // MMC0
				eWakeupMask = eWAKEUP_MMC0; 
				SDHC_card[0].m_eChannel = SDHC_CHANNEL_0;
//				SDHC_OpenMedia(SDHC_EPLL, &SDHC_card[0]);
				SDHC_SetGPIO(SDHC_CHANNEL_0, 4);
				Outp8(0xEB000029, 0xF); // power up
				Outp8(0xEB00002B, 0x7); // SDHC_WAKEUP
				Outp16(0xEB000034, 0x1C0); // SDHC_NORMAL_INT_STAT_ENABLE
				Outp16(0xEB000038, 0x1C0); // SDHC_NORMAL_INT_SIGNAL_ENABLE				
				INTC_SetVectAddr(NUM_HSMMC0, Isr_Mmc0);
				INTC_Enable(NUM_HSMMC0); 

				GPIO_SetPullUpDownAll(eGPIO_G0, 0xAAAA); // all pull-up
//				GPIO_SetConRegPDNAll(eGPIO_G0, 0xFFFF); // all previsous state setting
				GPIO_SetConRegPDNAll(eGPIO_G0, 0xAAAA); // all input setting
				GPIO_SetPullUDPDNAll(eGPIO_G0, 0xAAAA); // all pull-up enable
				break;
			case 12: // MMC1
				eWakeupMask = eWAKEUP_MMC1;
				SDHC_card[1].m_eChannel = SDHC_CHANNEL_1;
//				SDHC_OpenMedia(SDHC_EPLL, &SDHC_card[1]);
				SDHC_SetGPIO(SDHC_CHANNEL_1, 4);
				Outp8(0xEB100029, 0xF); // power up
				Outp8(0xEB10002B, 0x7); // SDHC_WAKEUP
				Outp16(0xEB100034, 0x1C0); // SDHC_NORMAL_INT_STAT_ENABLE
				Outp16(0xEB100038, 0x1C0); // SDHC_NORMAL_INT_SIGNAL_ENABLE				
				INTC_SetVectAddr(NUM_HSMMC1, Isr_Mmc1);
				INTC_Enable(NUM_HSMMC1); 

				GPIO_SetPullUpDownAll(eGPIO_G1, 0xAAAA); // all pull-up
//				GPIO_SetConRegPDNAll(eGPIO_G1, 0xFFFF); // all previsous state setting
				GPIO_SetConRegPDNAll(eGPIO_G1, 0xAAAA); // all input setting
				GPIO_SetPullUDPDNAll(eGPIO_G1, 0xAAAA); // all pull-up enable
				break;
			case 13: // MMC 2
				eWakeupMask = eWAKEUP_MMC2;
				SDHC_card[2].m_eChannel = SDHC_CHANNEL_2;
//				SDHC_OpenMedia(SDHC_EPLL, &SDHC_card[2]);
				SDHC_SetGPIO(SDHC_CHANNEL_2, 4);
				Outp8(0xEB200029, 0xF); // power up
				Outp8(0xEB20002B, 0x7); // SDHC_WAKEUP
				Outp16(0xEB200034, 0x1C0); // SDHC_NORMAL_INT_STAT_ENABLE
				Outp16(0xEB200038, 0x1C0); // SDHC_NORMAL_INT_SIGNAL_ENABLE				
				INTC_SetVectAddr(NUM_HSMMC2, Isr_Mmc2);
				INTC_Enable(NUM_HSMMC2); 

				GPIO_SetPullUpDownAll(eGPIO_G2, 0xAAAA); // all pull-up
//				GPIO_SetConRegPDNAll(eGPIO_G2, 0xFFFF); // all previsous state setting
				GPIO_SetConRegPDNAll(eGPIO_G2, 0xAAAA); // all input setting
				GPIO_SetPullUDPDNAll(eGPIO_G2, 0xAAAA); // all pull-up enable
				break;
			case 14: // MMC 3
				eWakeupMask = eWAKEUP_MMC3;
				SDHC_card[3].m_eChannel = SDHC_CHANNEL_3;
//				SDHC_OpenMedia(SDHC_EPLL, &SDHC_card[3]);
				SDHC_SetGPIO(SDHC_CHANNEL_3, 4);
				Outp8(0xEB300029, 0xF); // power up
				Outp8(0xEB30002B, 0x7); // SDHC_WAKEUP
				Outp16(0xEB300034, 0x1C0); // SDHC_NORMAL_INT_STAT_ENABLE
				Outp16(0xEB300038, 0x1C0); // SDHC_NORMAL_INT_SIGNAL_ENABLE				
				INTC_SetVectAddr(NUM_HSMMC3, Isr_Mmc3);
				INTC_Enable(NUM_HSMMC3); 

				GPIO_SetPullUpDownAll(eGPIO_G3, 0xAAAA); // all pull-up
//				GPIO_SetConRegPDNAll(eGPIO_G3, 0xFFFF); // all previsous state setting
				GPIO_SetConRegPDNAll(eGPIO_G3, 0xAAAA); // all input setting
				GPIO_SetPullUDPDNAll(eGPIO_G3, 0xAAAA); // all pull-up enable
				break;
			case 15: // I2S
				eWakeupMask = eWAKEUP_I2S;
//				INTC_SetVectAddr(NUM_I2S0, Isr_Iis);
				INTC_Enable(NUM_I2S0); 
				break;
			case 16: // PWM Timer (only Idle and DeepIdle(Top on)
				eWakeupMask = eWAKEUP_OTHERS;
				//Dead zone=0,Prescaler1=255(0x0f),Prescaler0=255(0x0f)
				//All interrupt,Mux4=1/2,Mux3=1/4,Mux2=1/8,Mux1=1/16,Mux0=1/16
				// PWM_Select(u32 uTimer, u32 uPrescaler, u32 uDivider, u32 uTCNTB, u32 uTCMPB, u32 uEnDz, u32 uDzlen, u32 uEnDMA, u32 uAutoreload, u32 uEnInverter, u32 uEnInt)
				PWM_Select(4,255,4,30000,15000,0,0,0,1,0,1);	
				INTC_SetVectAddr(NUM_TIMER4 , Isr_PwmTimer4);
				INTC_Enable(NUM_TIMER4); 
				break;
			default:
				Disp(" Wrong wakeup source selected!!\n");
				Assert(0);
				break;
			}	
			SYSC_SetWakeupMask(eWakeupMask, FALSE);
			Disp("Wakeup source=%d\n", i);
		}
	}
	
}


void SYSCT_DisplayRegStatus(void)
{
	u32 uIpReg[5], uBlockReg, uNormalCfgtReg, uBlockPwrStatReg, i;

	for (i=0;i<=4;i++)
	{
		uIpReg[i] = SYSC_GetClkGateIPReg(i);
		Disp(" Clock gate by IP %1d = 0x%x\n", i, uIpReg[i]);
	}
	uBlockReg = SYSC_GetClkGateBlockReg();
	Disp(" Clock gate by Block = 0x%x\n", uBlockReg);
	uNormalCfgtReg = SYSC_GetNormalModeConfigReg();
	Disp(" Normal mode config = 0x%x\n", uNormalCfgtReg);
	uBlockPwrStatReg = SYSC_GetBlockPowerStatusReg();
	Disp(" Block power status = 0x%x\n", uBlockPwrStatReg);
}


void SYSCT_SaveDataToExtSram(u32 uOffset)
{
	u32 uIpReg[5], uBlockReg, uNormalCfgtReg, uBlockPwrStatReg, i;
	u32 uTemp;

	// Save Status Registers
	for (i=0;i<=4;i++)
	{
		uIpReg[i] = SYSC_GetClkGateIPReg(i);
		Outp32(_SMC_BaseAddress+uOffset+i*4, uIpReg[i]); 
	}
	uBlockReg = SYSC_GetClkGateBlockReg();
	Outp32(_SMC_BaseAddress+uOffset+0x20, uBlockReg); 
	uNormalCfgtReg = SYSC_GetNormalModeConfigReg();
	Outp32(_SMC_BaseAddress+uOffset+0x24, uNormalCfgtReg); 
	uBlockPwrStatReg = SYSC_GetBlockPowerStatusReg();
	Outp32(_SMC_BaseAddress+uOffset+0x28, uBlockPwrStatReg); 
	
	// Save the same data to SRAM and DRAM 
	for (i=0;i<0x1000;i++) // 16K bytes
	{
		uTemp = i * rand();
		Outp32(_DRAM0_BaseAddress+0x02400000+uOffset+i*4, uTemp); // DRAM : 0x22400000 + offset
		Outp32(DMAC_CODE_BASE+0x00400000+uOffset+i*4, uTemp); // DRAM : 0x44400000 + offset
		Outp32(_SMC_BaseAddress+0x30+uOffset+i*4, uTemp); // SRAM : 0x80000030 + offset
	}
}

void SYSCT_VerifyDataFromExtSram(u32 uOffset)
{
	u32 uIpReg[5], uBlockReg, uNormalCfgtReg, uBlockPwrStatReg, i;
	u32 uTemp[3];
	bool bVerify=TRUE;

	Disp(" Verify the status backup data on SRAM\n");
	
	// Verify Status Registers
	for (i=0;i<=4;i++)
	{
		uIpReg[i] = Inp32(_SMC_BaseAddress+uOffset+i*4); 
		if (SYSC_GetClkGateIPReg(i) != uIpReg[i])
		{
			Disp(" Different status data Clock gate by IP REG%d!!\n", i);
			bVerify = FALSE;
		}
	}
	uBlockReg = Inp32(_SMC_BaseAddress+uOffset+0x20); 
	if (SYSC_GetClkGateBlockReg() != uBlockReg)
	{
		Disp(" Different status data Clock gate by Block REG!!\n", i);
		bVerify = FALSE;
	}
	uNormalCfgtReg = Inp32(_SMC_BaseAddress+uOffset+0x24); 
	if (SYSC_GetNormalModeConfigReg() != uNormalCfgtReg)
	{
		Disp(" Different status data Normal config REG!!\n", i);
		bVerify = FALSE;
	}
	uBlockPwrStatReg = Inp32(_SMC_BaseAddress+uOffset+0x28); 
	if (SYSC_GetBlockPowerStatusReg() != uBlockPwrStatReg)
	{
		Disp(" Different status data Block power status REG!!\n", i);
		bVerify = FALSE;
	}
	
	// Verify the same data from SRAM and DRAM 
	for (i=0;i<0x1000;i++) // 16K bytes
	{
		uTemp[0] = Inp32(_SMC_BaseAddress+0x30+uOffset+i*4); // SRAM
		uTemp[1] = Inp32(DMAC_CODE_BASE+0x00400000+uOffset+i*4); // DRAM
		uTemp[2] = Inp32(_DRAM0_BaseAddress+0x02400000+uOffset+i*4); // DRAM : 0x22400000 + offset
		if (uTemp[0] != uTemp[1])
		{
			Disp(" Different DRAM1 %dth data!!\n", i);		
			bVerify = FALSE;
			break;
		}
		if (uTemp[0] != uTemp[2])
		{
			Disp(" Different DRAM0 %dth data!!\n", i);		
			bVerify = FALSE;
			break;
		}
	}

	if (bVerify)
		Disp(" OK!!\n");
	else 
		Disp(" FAIL!! Check SROM select switch!!\n");
}


static volatile DMAC oDmacTemp;

static void __irq Isr_DMACopyDone(void)
{
	u32 uInt;

	INTC_Disable(NUM_MDMA);

	DMA_GetIntrSrc(&uInt, &oDmacTemp);
	if (uInt==32)
	{
		Disp("[MDMA FAULT & STOP]");
		DMA_StopDmac(&oDmacTemp);
	}
	else
	{
		DMA_ClearIntPending(uInt, &oDmacTemp);	
		Disp("[D]");
	}
	
	INTC_ClearVectAddr();

	INTC_Enable(NUM_MDMA);
}

void dma_cpy(void) // for bus access situation
{
	INTC_SetVectAddr(NUM_MDMA, Isr_DMACopyDone);

	INTC_Enable(NUM_MDMA);

	DMA_SetCh(DMA_20, &oDmacTemp); // MDMA ch 0
	DMA_InitCh(WORD, SOFTWARE, DMA_M2M, BURST4, &oDmacTemp);  
	DMA_StartCh(_DRAM_BaseAddress+0x5100000, _DRAM_BaseAddress+0x5200000, 0x1000000, &oDmacTemp);	// transfer size by word
}


// [Test entering of multi source to IDLE mode]
void SYSCT_TestIdleMode(void)
{
	u32 uSelAging;	
	u8 ucSelTop;
	u32 uAgingCnt = 0;
		
	// Select mode configurations
	Disp("Select Aging : 1.Aging(D), 2.When DMA operating, 3.Aging WFI only\n");
	uSelAging = UART_GetIntNum();
	if ((uSelAging>3)|(uSelAging<1)) uSelAging = 1;

	// Set wake up source
	SYSCT_SelectWakeupSource(eMODE_IDLE);

	// Clock gating
	SYSCT_SelectClkGating();
	
	// Power gating
	SYSCT_SelectPwrGating();

	// Save status for testing retention
	SYSCT_SaveDataToExtSram(0x1B0000);
	
	SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
	Disp(" Measure a current and check CLKOUT as ARMCLK/8\n");
	Disp(" Press any key to continue\n");
	UART_GetIntNum();


	while(UART_GetKey() == 0) 
	{
		Disp("\n Aging Count = %d\n", uAgingCnt);
		uAgingCnt++;

		if (uSelAging==2) // dma
		{
			// Bus operation using the DMA
			Disp(" DMA operate\n");
			dma_cpy();
		}
		Disp(" Enter IDLE mode\n");

		DisplayLED(0);		
		DisplayLED(1);
		Delay(10);
		DisplayLED(0);		

		// Entering power mode
//		SYSC_EnteringPowerMode(eMODE_IDLE, eCFG_ON, SW_WFI, uSelAging);
		if (uSelAging==3) // skip in WFI only
			SYSC_SetConfigStandbyWFI(eMODE_NORMAL); // rPWR_CFG is ignore
		else
			SYSC_SetConfigStandbyWFI(eMODE_IDLE);
		SYSC_DisableSyscInt();
		__wfi();

		SYSCT_VerifyDataFromExtSram(0x1B0000);
		Disp(" IDLE_CFG = 0x%x\n", SYSC_GetIdleModeConfigReg());
		Disp(" WAKEUP_STAT = 0x%x\n", SYSC_GetWakeupStatus(TRUE));
		Disp(" Exit IDLE mode\n");

		// Clear wakeup status..
		SYSC_ClearWakeupStatus(eWAKEUP_ALL);

		Disp("\nPress any key to exit the this test, otherwise Entering same power mode !! \n");	
		Delay(2000); 
	}	

	INTC_Disable(NUM_ALL);	
	SYSCT_DisplayRegStatus();
}


// [Test entering of multi source to IDLE mode]
void SYSCT_TestDeepIdleMode(void)
{
	u32 uSelAging, uSelClampCnt;	
	u8 ucSelTop, ucSelMem, ucSelL2;
	u32 uInformReg;
	u32 uAgingCnt = 0;

	uInformReg = SYSC_GetInformReg(5); // init inform reg for power mode & aging
	uInformReg &= ~(0xFF); // clear inform register for aging

	if (SYSC_GetResetStatusReg() == eRST_STAT_DEEPIDLE_WU)
		SYSCT_VerifyDataFromExtSram(0x1B2000);

	// Select mode configurations
	Disp("Select Top module status : 1.Retention(D), 2.On \n");
	ucSelTop = UART_GetIntNum();
	if (ucSelTop != 2) ucSelTop = 1;
	Disp("Select Top mem module status : 1.Retention(D), 2.On \n");
	ucSelMem = UART_GetIntNum();
	if (ucSelMem != 2) ucSelMem = 1;
	Disp("Select ARM L2 Cache status : 1.Off, 2.Retention(D) \n");
	ucSelL2 = UART_GetIntNum() - 1; // internally off = 0, retention = 1
	if (ucSelL2 != 0) ucSelL2 = 1;

	Disp("Enter Clamp(ARM) Stable Counter [0~0x3FF(D)] : \n");
	uSelClampCnt = UART_GetIntNum();
	if (uSelClampCnt > 0x3FF) uSelClampCnt = 0x3FF;
	
	Disp("Select Aging : 1.General(D), 2.Aging[RTC tick/EINT only], 3.When DMA operating\n");
	uSelAging = UART_GetIntNum();
	if ((uSelAging>3)||(uSelAging<1)) uSelAging = 1;

	SYSC_SetSleepGPIO(); // use deep(top off/retention) & sleep mode

	// Set wake up source
	SYSCT_SelectWakeupSource(eMODE_DEEPIDLE);

	// Clear wakeup status
	SYSC_ClearWakeupStatus(eWAKEUP_ALL);

	// Clock gating
	SYSCT_SelectClkGating();
	
	// Power gating
	SYSCT_SelectPwrGating();

	// Save status for testing retention
	SYSCT_SaveDataToExtSram(0x1B2000);
	
	SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
	Disp(" Measure a current and check CLKOUT as ARMCLK/8\n");
	Disp(" Press any key to continue\n");
	UART_GetIntNum();

	// Clamp stable => when ARM power off in DEEPIDLE/DEEPSTOP/SLEEP modes
	SYSC_SetClampStableCounter(TRUE, uSelClampCnt); // reset value = 0x3FF
	SYSC_SetClampStableCounter(FALSE, uSelClampCnt); // reset value = 0x3FF

	SYSC_SetIdleModeConfig(ucSelTop, ucSelMem, ucSelL2, eCFG_DEEP);
	uInformReg |= INFORM_DIDLE;	// 0x2x = deep idle mode

	// write aging and mode info. to infrom register 
	if (uSelAging==2) // aging test
	{
		SYSC_SetInformReg(6, 0); // init count aging test
		SYSC_SetInformReg(5, uInformReg|INFORM_AGING);  
	}
	else if (uSelAging==3) // dma
	{
		SYSC_SetInformReg(5, uInformReg|INFORM_GENERAL); 
		// Bus operation using the DMA
		Disp(" DMA operate\n");
		dma_cpy();
	}
	else // general test
	{
		SYSC_SetInformReg(5, uInformReg|INFORM_GENERAL); 
	}

	Disp(" Enter DEEPIDLE mode\n");

	DisplayLED(0);		
	DisplayLED(1);
	Delay(10);
	DisplayLED(0);		

	// Entering power mode
	SYSC_SetConfigStandbyWFI(eMODE_DEEPIDLE);
	SYSC_DisableSyscInt();
	__wfi();
}

// [Test entering of multi source to IDLE mode]
void SYSCT_TestStopMode(void)
{
	u32 uSelAging, uSelOscCnt, uSelOscPad;	
	u8 ucSelTop;
	u32 uAgingCnt = 0;
		
	// Select mode configurations
	Disp("Select OSC pad status : 1.Enable(D), 2.Disable \n");
	uSelOscPad = UART_GetIntNum();
	if (uSelOscPad != 2) 
		uSelOscPad = TRUE;
	else 
		uSelOscPad = FALSE;
	Disp("Enter OSC(Crystal) Stable Counter [0~0xFFFFF(D)] : \n");
	uSelOscCnt = UART_GetIntNum();
	if (uSelOscCnt > 0xFFFFF) uSelOscCnt = 0xFFFFF;

	Disp("Select Aging : 1.Aging(D), 2.When DMA operating\n");
	uSelAging = UART_GetIntNum();
	if ((uSelAging>2)||(uSelAging<1)) uSelAging = 1;

	// Set wake up source
	SYSCT_SelectWakeupSource(eMODE_STOP);

	// Clock gating
	SYSCT_SelectClkGating();
	
	// Power gating
	SYSCT_SelectPwrGating();

	// Save status for testing retention
	SYSCT_SaveDataToExtSram(0x1B4000);
	
	SYSC_SetClkOut(eCLKOUT_HCLKMSYS, 2); // HCLKMSYS/2
	Disp(" Measure a current and check CLKOUT as HCLKMSYS/2\n");
	Disp(" Press any key to continue\n");
	UART_GetIntNum();

	// OSC stable => when OSC disabled in STOP/DEEPSTOP/SLEEP modes
	SYSC_SetOscStableCounter(uSelOscCnt); // reset value = 0xFFFFF

	while(UART_GetKey() == 0) 
	{
		Disp("\n Aging Count = %d\n", uAgingCnt);
		uAgingCnt++;
		
		if (uSelAging==2) // dma
		{
			// Bus operation using the DMA
			Disp(" DMA operate\n");
			dma_cpy();
		}
		
		SYSC_SetStopModeConfig(eCFG_ON, eCFG_RET, eCFG_RET, eCFG_NODEEP);
		SYSC_SetOscPadOnOff(eMODE_STOP, eOSC_MAIN, (bool)uSelOscPad);
		SYSC_SetOscPadOnOff(eMODE_STOP, eOSC_USB, (bool)uSelOscPad);

		Disp(" Enter STOP mode\n");

		DisplayLED(0);		
		DisplayLED(1);
		Delay(10);
		DisplayLED(0);		

		// Entering power mode
		SYSC_SetConfigStandbyWFI(eMODE_STOP);
		SYSC_DisableSyscInt();
		__wfi();

		SYSC_ReleaseRetentionIO(eRET_BLOCK_ALL); // when top off..
		SYSCT_VerifyDataFromExtSram(0x1B4000);
		Disp(" STOP_CFG = 0x%x\n", SYSC_GetStopModeConfigReg());
		Disp(" WAKEUP_STAT = 0x%x\n", SYSC_GetWakeupStatus(TRUE));
		Disp(" Exit STOP mode\n");

		// Clear wakeup status
		SYSC_ClearWakeupStatus(eWAKEUP_ALL);

		Delay(2000); 
	}	

	INTC_Disable(NUM_ALL);	
	SYSCT_DisplayRegStatus();
}

// [Test entering of multi source to IDLE mode]
void SYSCT_TestDeepStopMode(void)
{
	u32 uSelAging, uSelOscPad, uSelStopMem, uSelMemRet, uSelOscCnt, uSelClampCnt;	
	u8 ucSelTop, ucSelL2;
	u32 uInformReg;
	u32 uAgingCnt = 0;

	uInformReg = SYSC_GetInformReg(5); // init inform reg for power mode & aging
	uInformReg &= ~(0xFF); // clear inform register for aging
		
	if (SYSC_GetResetStatusReg() == eRST_STAT_DEEPSTOP_WU)
		SYSCT_VerifyDataFromExtSram(0x1B6000);

	// Select mode configurations
	Disp("Select Top module status : 1.Retention(D), 2.On \n");
	ucSelTop = UART_GetIntNum();
	if (ucSelTop != 2) ucSelTop = 1;
	Disp("Select ARM L2 Cache status : 1.Off, 2.Retention(D) \n");
	ucSelL2 = UART_GetIntNum() - 1; // internally off = 0, retention = 1
	if (ucSelL2 != 0) ucSelL2 = 1;
	while(1)
	{
		Disp("Select Top memory control \n");
		Disp(" 1.IRAM, 2.SECSS, 3.CSSYS, 4.HSMMC, 5.USBOTG, 6.HOSTIF, 7.MODEM, 8.ONENAND, 9.ALL, 0.Exit(D)\n");
		uSelStopMem = UART_GetIntNum();
		if ( (uSelStopMem > 9) || (uSelStopMem < 1) ) 
			break;

		Disp("Select Retention/Off \n");
		Disp(" 1.Retention(D), 2.Off\n");
		uSelMemRet = UART_GetIntNum();
		if (uSelMemRet != 2) 
			uSelMemRet = TRUE;
		else 
			uSelMemRet = FALSE;
		
		SYSC_SetStopMemConfig((CFG_STOP_MEM)uSelStopMem, (bool)uSelMemRet);	
		Disp(" StopMemConfig Reg = 0x%x\n", SYSC_GetStopMemConfigReg());
	}	
	Disp("Select OSC pad status : 1.Enable(D), 2.Disable \n");
	uSelOscPad = UART_GetIntNum();
	if (uSelOscPad != 2) 
		uSelOscPad = TRUE;
	else 
		uSelOscPad = FALSE;
	Disp("Enter OSC(Crystal) Stable Counter [0~0xFFFFF(D)] : \n");
	uSelOscCnt = UART_GetIntNum();
	if (uSelOscCnt > 0xFFFFF) uSelOscCnt = 0xFFFFF;
	Disp("Enter Clamp(ARM) Stable Counter [0~0x3FF(D)] : \n");
	uSelClampCnt = UART_GetIntNum();
	if (uSelClampCnt > 0x3FF) uSelClampCnt = 0x3FF;
	
	Disp("Select Aging : 1.General(D), 2.Aging[RTC tick/EINT only], 3.When DMA operating\n");
	uSelAging = UART_GetIntNum();
	if ((uSelAging>3)||(uSelAging<1)) uSelAging = 1;

	SYSC_SetSleepGPIO(); // use deep(top off/retention) & sleep mode

	// Set wake up source
	SYSCT_SelectWakeupSource(eMODE_DEEPSTOP);

	// Clear wakeup status
	SYSC_ClearWakeupStatus(eWAKEUP_ALL);

	// Clock gating
	SYSCT_SelectClkGating();
	
	// Power gating
	SYSCT_SelectPwrGating();

	// Save status for testing retention
	SYSCT_SaveDataToExtSram(0x1B6000);
	
	SYSC_SetClkOut(eCLKOUT_HCLKMSYS, 2); // HCLKMSYS/2
	Disp(" Measure a current and check CLKOUT as HCLKMSYS/2\n");
	Disp(" Press any key to continue\n");
	UART_GetIntNum();

	// OSC stable => when OSC disabled in STOP/DEEPSTOP/SLEEP modes
	// Clamp stable => when ARM power off in DEEPIDLE/DEEPSTOP/SLEEP modes
	SYSC_SetOscStableCounter(uSelOscCnt); // reset value = 0xFFFFF
	SYSC_SetClampStableCounter(TRUE, uSelClampCnt); // reset value = 0x3FF
	SYSC_SetClampStableCounter(FALSE, uSelClampCnt); // reset value = 0x3FF

	SYSC_SetStopModeConfig(ucSelTop, 1, ucSelL2, eCFG_DEEP);
	SYSC_SetOscPadOnOff(eMODE_DEEPSTOP, eOSC_MAIN, (bool)uSelOscPad);
	SYSC_SetOscPadOnOff(eMODE_DEEPSTOP, eOSC_USB, (bool)uSelOscPad);
	uInformReg |= INFORM_DSTOP;	// 0x2x = deep idle mode

	// write aging and mode info. to infrom register 
	if (uSelAging==2) // aging test
	{
		SYSC_SetInformReg(6, 0); // init count aging test
		SYSC_SetInformReg(5, uInformReg|INFORM_AGING);  
	}
	else if (uSelAging==3) // dma
	{
		SYSC_SetInformReg(5, uInformReg|INFORM_GENERAL); 
		// Bus operation using the DMA
		Disp(" DMA operate\n");
		dma_cpy();
	}
	else // general test
	{

		SYSC_SetInformReg(5, uInformReg|INFORM_GENERAL); 
	}

	Disp(" Enter DEEPSTOP mode\n");

	DisplayLED(0);		
	DisplayLED(1);
	Delay(10);
	DisplayLED(0);	

	// Entering power mode	
	SYSC_SetConfigStandbyWFI(eMODE_DEEPSTOP);
	SYSC_DisableSyscInt();
	__wfi();
}


// [Test entering of multi source to Sleep mode]
void SYSCT_TestSleepMode(void)
{
	u32 uSelMethod, uSelAging, uSelOscPad, uSelOscCnt, uSelPwrCnt, uSelClampCnt;
	WAKEUP_ID eWakeupStatus;
	u32 uInformReg;

	uInformReg = SYSC_GetInformReg(5); // init inform reg for power mode & aging
	uInformReg &= ~(0xFF); // clear inform register for aging

	// Verify saved data			
	if (SYSC_GetResetStatusReg() == eRST_STAT_SLEEP_WU)
		SYSCT_VerifyDataFromExtSram(0x1B8000);

	// Select mode configurations
	Disp("Select OSC pad status : 1.Enable(D), 2.Disable \n");
	uSelOscPad = UART_GetIntNum();
	if (uSelOscPad != 2) 
		uSelOscPad = TRUE;
	else 
		uSelOscPad = FALSE;
	Disp("Enter OSC(Crystal) Stable Counter [0~0xFFFFF(D)] : \n");
	uSelOscCnt = UART_GetIntNum();
	if (uSelOscCnt > 0xFFFFF) uSelOscCnt = 0xFFFFF;
	Disp("Enter Power Stable Counter [0~0xFFFFF(D)] : \n");
	uSelPwrCnt = UART_GetIntNum();
	if (uSelPwrCnt > 0xFFFFF) uSelPwrCnt = 0xFFFFF;
	Disp("Enter Clamp(ARM) Stable Counter [0~0x3FF(D)] : \n");
	uSelClampCnt = UART_GetIntNum();
	if (uSelClampCnt > 0x3FF) uSelClampCnt = 0x3FF;
	
	Disp("Select Aging : 1.General(D), 2.Aging[RTC tick/EINT only], 3.When DMA operating\n");
	uSelAging = UART_GetIntNum();
	if ((uSelAging>3)||(uSelAging<1)) uSelAging = 1;

	eWakeupStatus = SYSC_GetWakeupStatus(FALSE); // get wakeup source as WAKEUP_ID
	
	SYSC_SetSleepGPIO(); // use deep(top off/retention) & sleep mode

	// Set wake up source
	SYSCT_SelectWakeupSource(eMODE_SLEEP);

	// Clear wakeup status
	SYSC_ClearWakeupStatus(eWAKEUP_ALL);

	// Clock gating
	SYSCT_SelectClkGating();
	
	// Power gating
	SYSCT_SelectPwrGating();

	// Save status for testing retention
	SYSCT_SaveDataToExtSram(0x1B8000);
	

	SYSC_SetClkOut(eCLKOUT_HCLKMSYS, 2); // HCLKMSYS/2
	Disp(" Measure a current and check CLKOUT as HCLKMSYS/2\n");
	Disp(" Press any key to continue\n");
	UART_GetIntNum();

	// OSC stable => when OSC disabled in STOP/DEEPSTOP/SLEEP modes
	// PWR stable => when Internal power off in SLEEP modes
	// Clamp stable => when ARM power off in DEEPIDLE/DEEPSTOP/SLEEP modes
	SYSC_SetOscStableCounter(uSelOscCnt); // reset value = 0xFFFFF
	SYSC_SetPowerStableCounter(uSelPwrCnt); // reset value = 0xFFFFF
	SYSC_SetClampStableCounter(TRUE, uSelClampCnt); // reset value = 0x3FF
	SYSC_SetClampStableCounter(FALSE, uSelClampCnt); // reset value = 0x3FF

	SYSC_SetOscPadOnOff(eMODE_SLEEP, eOSC_MAIN, (bool)uSelOscPad);
	SYSC_SetOscPadOnOff(eMODE_SLEEP, eOSC_USB, (bool)uSelOscPad);
	uInformReg |= INFORM_SLEEP;	// 0x2x = deep idle mode

	// write aging and mode info. to infrom register 
	if (uSelAging==2) // aging test
	{
		SYSC_SetInformReg(6, 0); // init count aging test
		SYSC_SetInformReg(5, uInformReg|INFORM_AGING);  
	}
	else if (uSelAging==3) // dma
	{
		SYSC_SetInformReg(5, uInformReg|INFORM_GENERAL); 
		// Bus operation using the DMA
		Disp(" DMA operate\n");
		dma_cpy();
	}
	else // general test
	{

		SYSC_SetInformReg(5, uInformReg|INFORM_GENERAL); 
	}

	Disp(" Enter SLEEP mode\n");

	DisplayLED(0);		
	DisplayLED(1);
	Delay(10);
	DisplayLED(0);	

	Disp(" Check LED...\n");
	// Entering power mode	
#if 0
	// S/W WFI
	SYSC_SetConfigStandbyWFI(eMODE_SLEEP);
	SYSC_DisableSyscInt();
	__wfi();
#else
	// H/W WFI temp, WA for unstable power state
	SYSC_SetConfigStandbyWFI(eMODE_NORMAL); // for HW WFI
	SYSC_DisableSyscInt();
	SYSC_SetPowerMode(eMODE_SLEEP);
#endif	

}


// [Test Resets]
void SYSCT_TestReset(void)
{
	u32 uSelRstType, uSelTestCase, uSelRstatPDN;
	u32 uInformReg;
	u32 uDramtemp;
	RESET_STATUS eRstStatus;
		
	uInformReg = SYSC_GetInformReg(5); // init inform reg for power mode & aging
	uInformReg &= ~(0xFF); // clear inform register for aging

	Disp("Select Reset Type : 1.SW Reset(D), 2.WDT Reset, 3.Warm Reset \n");
	uSelRstType = (u8)UART_GetIntNum();

	Disp("Select Test case : 1.General(D), 2.Aging, 3.When DMA operating, 4.When bus down(except SW reset)\n");
	uSelTestCase = UART_GetIntNum();

	// setting for test case #1
	if (uSelTestCase == 2) // aging
	{
		SYSC_SetInformReg(6, 0); // count aging test
		uInformReg = INFORM_AGING;  
	}
	else if (uSelTestCase == 3) // dma
	{
		uInformReg = INFORM_GENERAL;  
		dma_cpy();
	}
	else
		uInformReg = INFORM_GENERAL;  

	// setting for each reset type
	if (uSelRstType == 2) // WDT reset
	{
		SYSC_SetInformReg(5, uInformReg|INFORM_WDTRST);  
		Disp("WDT reset assert after a moment!!\n");

		WDT_Init(1, DIV16, 0x8000);
		WDT_Enable(1);
		WDT_EnableReset(1);

		// setting for test case #2
		if (uSelTestCase == 4) // bus down
		{
			Disp("Bus down!!\n");
			// bus down situation
			SYSC_SetClkGatebyIP(eCLKGATE_IP_DMC0, 0);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_DMC1, 0);
			uDramtemp = *((u32 *)0x20000000);		
		}
		UART_GetIntNum();
	}
	else if (uSelRstType == 3) // Warm reset
	{
		SYSC_SetInformReg(5, uInformReg|INFORM_WARMRST);  		
		Disp("Press Warm reset button to test Warm reset\n");

		// setting for test case #2
		if (uSelTestCase == 4) // bus down
		{
			Disp("Bus down!!\n");
			// bus down situation
			SYSC_SetClkGatebyIP(eCLKGATE_IP_DMC0, 0);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_DMC1, 0);
			uDramtemp = *((u32 *)0x20000000);		
		}
		
		UART_GetIntNum();
	}
	else // default S/W reset
	{
		SYSC_SetInformReg(5, uInformReg|INFORM_SWRST);  		
		Disp("S/W reset assert!!\n");

		SYSC_SwReset();		
	}			
}


void SYSCT_Test(void)
{
	u32 sel, i;
	u32 x, uRegData;
	u32 uPwrMode, uTestMode;

	const testFuncMenu menu[]=
	{
		SYSCT_TestClkout,					"CLKOUT test",

		SYSCT_TestPllBasic,					"Basic PLL operation",
		SYSCT_TestPllRunningChange,			"PLL running change",
		SYSCT_TestPllLockTime,				"PLL lock time check",
		SYSCT_TestPllManChg,				"PLL manual change",
		SYSCT_TestClkMuxControl,			"Clock Source MUXing control",
		SYSCT_TestClkDiv,					"Clock DIVider control",
		SYSCT_TestClkGating,				"Clock GATE control",
		SYSCT_TestDVFS,						"DVFS control",
		
		SYSCT_TestReset,					"Several RESET functions",

		SYSCT_TestSimplePowerMode,			"Basic Power mode operation check",
		SYSCT_TestNormalMode,				"Power & Clock gating in normal mode",
		SYSCT_TestIdleMode,					"Enter & exit IDLE mode with various config.",
		SYSCT_TestDeepIdleMode,				"Enter & exit DEEPIDLE mode with various config.",
		SYSCT_TestStopMode,					"Enter & exit STOP mode with various config.",
		SYSCT_TestDeepStopMode,				"Enter & exit DEEPSTOP mode with various config.",
		SYSCT_TestSleepMode,				"Enter & exit SLEEP mode with various config.",
		0,									0
	};
		
	// Inform0: bit[15:8] - Power mode, bit[23:16] - Test mode 
	// Inform3: wakeup src, Inform4: index
	uRegData = SYSC_GetInformReg(0); 
	x = (uRegData>>8)&0xffffff;

	// Set drive strength of CLKOUT to Max.
	GPIO_SetDSEach(eETC2, eGPIO_1, 3); // Max. DS
	GPIO_SetSREach(eETC2, eGPIO_1, 0); // Fast SR

	// SRAM setting to CS0
	Outp32(0xE0200260, 0x00444444); // GPJ1 SROM_ADDR_16to21
	Outp32(0xE02002C0, 0x00040000); // GPJ4 SROM_ADDR_16to21
	Outp32(0xE8000000, 0x0000000B); // CS0 - 16bit sram, enable nBE, Byte base address


	while(1)
	{
		Disp("\n\n================== Syscon Function Test =====================\n\n");
		for (i=0; (u32)(menu[i].desc)!=0; i++)
			Disp("%2d: %s\n", i, menu[i].desc);

		Disp("\nSelect the function to test : ");
		sel = UART_GetIntNum();
		Disp("\n");

		if (sel == -1)
			break;
		else if (sel>=0 && sel<(sizeof(menu)/8-1))
			(menu[sel].func)();
	}
}


void SYSCT_TestSimplePowerMode_Temp(u32 PowerMode)
{
	u32 uSelMode, uSelAging;
	u32 uInformReg, uAgingCnt;
	u8	ucSelTop, ucSelL2;
	POWER_MODE ePwrMode;
	WAKEUP_ID eWakeupSrc, eEintWakeup;
	u32 uEintNum;

	// Clear wakeup status..
	SYSC_ClearWakeupStatus(eWAKEUP_ALL);
	
	// Set wakeup interrupt for GPIO as EINT
#ifdef FPGA
	INTC_SetVectAddr(NUM_EINT1, Isr_EInt1);
	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_1, eGINT);
	GPIO_SetPullUpDownEach(eGPIO_H0, eGPIO_1, eGPUDdis);
	GPIO_SetEint30(1, Both_Edge, eDLYFLT, 0x1f); // EINT1, falling edge, enable filter, filter width
	GPIO_EINT30UnMask(1);
	eEintWakeup = eWAKEUP_EINT1;
	uEintNum = NUM_EINT1;
#else
	INTC_SetVectAddr(NUM_EINT4, Isr_EInt4);
	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_4, eGINT);
	GPIO_SetPullUpDownEach(eGPIO_H0, eGPIO_4, eGPUDdis);
	GPIO_SetEint30(4, Both_Edge, eDLYFLT, 0x1f); // EINT1, falling edge, enable filter, filter width
	GPIO_EINT30UnMask(4);
	eEintWakeup = eWAKEUP_EINT4;
	uEintNum = NUM_EINT4;
#endif
	INTC_SetVectAddr(NUM_RTC_TICK, Isr_RtcTick); // for aging test 

	eWakeupSrc = eWAKEUP_EINT;

	// Set all wakeup source 
	SYSC_SetWakeupMask(eWAKEUP_ALL , TRUE); // mask all wakeup sources
	SYSC_SetWakeupMask(eEintWakeup , FALSE); // unmask wakeup sources
	SYSC_SetWakeupMask(eWAKEUP_RTCTICK , FALSE); // unmask wakeup sources
	SYSC_SetWakeupMask(eWAKEUP_CEC , FALSE); // unmask wakeup sources

	SYSC_SetSleepGPIO(); // use deep(top off/retention) & sleep mode

	// Select mode configurations
	Disp("Select Test mode : 1.Idle(D), 2.DeepIdle, 3.Stop, 4.DeepStop, 5.Sleep, 6.S/W reset\n");
	uSelMode = PowerMode;

	Disp("Select Top module status(DeepIdle/DeepStop only) : 0.Off, 1.Retention(D), 2.On \n");
	ucSelTop = 1;

	if ((ucSelTop!=0)) 
		ucSelL2 = 1;
	else 
		ucSelL2 = 0;

	if ((ucSelTop>2)||(ucSelTop<1)) 
		ucSelTop = 1; // default retention
 
	Disp("Select Aging : 1.General(D), 2.Aging\n");
	uSelAging = 1;
	uSelAging = (uSelAging != 2) ? 1 : 2;	// default	General

	// Implement configuration
	switch(uSelMode)
	{
		case 2 :
			ePwrMode = eMODE_DEEPIDLE;
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
			SYSC_SetIdleModeConfig(ucSelTop, ucSelTop, ucSelL2, eCFG_DEEP);
			uInformReg |= INFORM_DIDLE;	// 0x2x = deep idle mode
			break;
		case 3 :
			ePwrMode = eMODE_STOP;
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
			SYSC_SetStopModeConfig(eCFG_ON, (u8)eCFG_RET, ucSelL2, eCFG_NODEEP);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, true);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, true);
			uInformReg |= INFORM_STOP;	// 0x3x = stop mode
			break;
		case 4 :
			ePwrMode = eMODE_DEEPSTOP;
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
			SYSC_SetStopModeConfig(ucSelTop, (u8)eCFG_RET, ucSelL2, eCFG_DEEP);
			SYSC_SetStopMemConfig(eMEMCFG_ALL, ucSelL2);			
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, true);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, true);
			uInformReg |= INFORM_DSTOP;	// 0x4x = deep stop mode
			break;
		case 5 :
			ePwrMode = eMODE_SLEEP;
			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, true);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, true);
			uInformReg |= INFORM_SLEEP;	// 0x5x = sleep mode
			break;
		case 6 :
			SYSC_SwReset();
			break;
		default:
			ePwrMode = eMODE_IDLE;
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
			SYSC_SetIdleModeConfig(ucSelTop, ucSelTop, ucSelL2, eCFG_NODEEP);
			uInformReg |= INFORM_IDLE;	// 0x1x = Idle mode
			break;
	}


	if ( (ePwrMode==eMODE_IDLE)||(ePwrMode==eMODE_STOP) )
	{
		GPIO_EINT30ClrPend(1);
		INTC_Enable(uEintNum); // Set interrupt controller enable
	}

	SYSC_StartPowerMode(ePwrMode, SW_WFI, USER_CONFIG); // selected power mode, selected enterfing method, Set user config

	INTC_Disable(uEintNum); // Set interrupt controller disable

	SYSC_ClearWakeupStatus(eWakeupSrc);

	Disp("Return to Normal Mode\n");
}

