/**************************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2006 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : sysc.c
*  
*	File Description : This file implements the API functons for system controller.
*
*	Author : Joonsoo Jeon
*	Dept. : AP Development Team
*	Created Date : 2009/03/08
*	Version : 0.1 
* 
*	History
*	- Created(Joonsoo Jeon 2009/03/08)
*  
**************************************************************************************/

#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "sysc.h"
#include "dmc.h"
#include "gpio.h"


enum CHIP_ID_SFR		
{
	rPRODUCT_ID 		= (u32)(CHIPID_BASE+0x0000), // RO

	rOPERATING_MODE 	= (u32)(CHIPID_BASE+0x0004), // RO
	rEFUSE_CFG 		= (u32)(CHIPID_BASE+0x0008),
	rEFUSE_CTRL 		= (u32)(CHIPID_BASE+0x000c),
	rSPARE_AREA 		= (u32)(CHIPID_BASE+0x0010), // RO
	rCHIP_ID0 			= (u32)(CHIPID_BASE+0x0014), // RO
	rCHIP_ID1 			= (u32)(CHIPID_BASE+0x0018), // RO
};


enum CMU_SFR		
{
	rAPLL_LOCK 			= SYSCON_BASE+0x0000,
	rMPLL_LOCK 			= SYSCON_BASE+0x0008,
	rEPLL_LOCK 			= SYSCON_BASE+0x0010,
	rVPLL_LOCK 			= SYSCON_BASE+0x0020,
		// [15:0] Lock time.
	rAPLL_CON 			= SYSCON_BASE+0x0100,
	rMPLL_CON 			= SYSCON_BASE+0x0108,
	rEPLL_CON 			= SYSCON_BASE+0x0110,
	rVPLL_CON 			= SYSCON_BASE+0x0120,
		// [31] Enable PLL.
		// [30] PLL is locked.
		// [23:16] M value
		// [13:8] P value
		// [2:0] S value
	rCLK_SRC0 			= SYSCON_BASE+0x0200,
	rCLK_SRC1 			= SYSCON_BASE+0x0204,
	rCLK_SRC2 			= SYSCON_BASE+0x0208,
	rCLK_SRC3 			= SYSCON_BASE+0x020c,
	rCLK_SRC4 			= SYSCON_BASE+0x0210,
	rCLK_SRC5 			= SYSCON_BASE+0x0214,
	rCLK_SRC6 			= SYSCON_BASE+0x0218,
	rCLK_SRC_MASK0 	= SYSCON_BASE+0x0280,
	rCLK_SRC_MASK1 	= SYSCON_BASE+0x0284,
	rCLK_DIV0 			= SYSCON_BASE+0x0300,
	rCLK_DIV1 			= SYSCON_BASE+0x0304,
	rCLK_DIV2 			= SYSCON_BASE+0x0308,
	rCLK_DIV3 			= SYSCON_BASE+0x030c,
	rCLK_DIV4 			= SYSCON_BASE+0x0310,
	rCLK_DIV5 			= SYSCON_BASE+0x0314,
	rCLK_DIV6 			= SYSCON_BASE+0x0318,
	rCLK_DIV7 			= SYSCON_BASE+0x031C,
	rCLK_GATE_IP0 		= SYSCON_BASE+0x0460,
	rCLK_GATE_IP1 		= SYSCON_BASE+0x0464,
	rCLK_GATE_IP2 		= SYSCON_BASE+0x0468,
	rCLK_GATE_IP3 		= SYSCON_BASE+0x046C,
	rCLK_GATE_IP4 		= SYSCON_BASE+0x0470,
	rCLK_GATE_BLOCK 	= SYSCON_BASE+0x0480,
	rCLK_OUT 			= SYSCON_BASE+0x0500,
		// [16:12] CLKSEL
	rCLK_DIV_STAT0		= SYSCON_BASE+0x1000,
	rCLK_DIV_STAT1		= SYSCON_BASE+0x1004,
	rCLK_MUX_STAT0	= SYSCON_BASE+0x1100,
	rCLK_MUX_STAT1	= SYSCON_BASE+0x1104,

// RESET
	rGLOBAL_SW_RESET 	= SYSCON_BASE+0x2000,

// IEM
	rDCGIDX_MAP0 		= SYSCON_BASE+0x3000,
	rDCGIDX_MAP1 		= SYSCON_BASE+0x3004,
	rDCGIDX_MAP2 		= SYSCON_BASE+0x3008,
	rDCGPERF_MAP0		= SYSCON_BASE+0x3020,
	rDCGPERF_MAP1		= SYSCON_BASE+0x3024,
	rDVCIDX_MAP		= SYSCON_BASE+0x3040,
	rFREQ_CPU			= SYSCON_BASE+0x3060,
	rFREQ_DPM			= SYSCON_BASE+0x3064,
	rDVSEMCLK_EN 		= SYSCON_BASE+0x3080,
	rMAXPERF	 		= SYSCON_BASE+0x3084,
	rAPLL_CON_L8 		= SYSCON_BASE+0x3100,
	rAPLL_CON_L7 		= SYSCON_BASE+0x3104,
	rAPLL_CON_L6 		= SYSCON_BASE+0x3108,
	rAPLL_CON_L5 		= SYSCON_BASE+0x310c,
	rAPLL_CON_L4 		= SYSCON_BASE+0x3110,
	rAPLL_CON_L3 		= SYSCON_BASE+0x3114,
	rAPLL_CON_L2 		= SYSCON_BASE+0x3118,
	rAPLL_CON_L1 		= SYSCON_BASE+0x311c,
	rCLKDIV_IEM_L8		= SYSCON_BASE+0x3200,
	rCLKDIV_IEM_L7		= SYSCON_BASE+0x3204,
	rCLKDIV_IEM_L6		= SYSCON_BASE+0x3208,
	rCLKDIV_IEM_L5		= SYSCON_BASE+0x320c,
	rCLKDIV_IEM_L4		= SYSCON_BASE+0x3210,
	rCLKDIV_IEM_L3		= SYSCON_BASE+0x3214,
	rCLKDIV_IEM_L2		= SYSCON_BASE+0x3218,
	rCLKDIV_IEM_L1		= SYSCON_BASE+0x321c,

// MISC
	rONEDRAM_CFG 		= SYSCON_BASE+0x6208,
	rDISPLAY_CONTROL			= SYSCON_BASE+0x7008,
	rAUDIO_ENDIAN_SEL = SYSCON_BASE+0x700C,
	rHDMI_PHY_CON0 	= SYSCON_BASE+0x7208
};


enum PMU_SFR
{
	// Power Manager
	rOSC_CON			= SYSCON_BASE+0x8000,
	rRST_STAT			= SYSCON_BASE+0xA000,
	rPWR_CFG 			= SYSCON_BASE+0xC000,
	rEINT_WAKEUP_MASK	= SYSCON_BASE+0xC004,
	rWAKEUP_MASK		= SYSCON_BASE+0xC008,
	rPWR_MODE			= SYSCON_BASE+0xC00C,
	rNORMAL_CFG		= SYSCON_BASE+0xC010,
	rIDLE_CFG			= SYSCON_BASE+0xC020,
	rSTOP_CFG			= SYSCON_BASE+0xC030,
	rSTOP_MEM_CFG		= SYSCON_BASE+0xC034,
	rSLEEP_CFG			= SYSCON_BASE+0xC040,

	rOSC_FREQ			= SYSCON_BASE+0xC100,
	rOSC_STABLE		= SYSCON_BASE+0xC104,
	rPWR_STABLE		= SYSCON_BASE+0xC108,
	rMTC_STABLE		= SYSCON_BASE+0xC110,
	rCLAMP_STABLE		= SYSCON_BASE+0xC114,

	rWAKEUP_STAT 		= SYSCON_BASE+0xC200,
	rBLK_PWR_STAT		= SYSCON_BASE+0xC204,

	rOTHERS			= SYSCON_BASE+0xE000,
	rOM_STAT			= SYSCON_BASE+0xE100,
	rMIE_CON		 	= SYSCON_BASE+0xE800,
	rHDMI_CON		 	= SYSCON_BASE+0xE804,
	rUSB_CON			= SYSCON_BASE+0xE80C,
	rDAC_CON			= SYSCON_BASE+0xE810,
	rMIPI_PHY_CON		= SYSCON_BASE+0xE814,
	rADC_CON			= SYSCON_BASE+0xE818,
	rPS_HOLD_CON		= SYSCON_BASE+0xE81C,

	rINFORM0 			= SYSCON_BASE+0xF000,
	rINFORM1 			= SYSCON_BASE+0xF004,
	rINFORM2 			= SYSCON_BASE+0xF008,
	rINFORM3 			= SYSCON_BASE+0xF00C,
	rINFORM4 			= SYSCON_BASE+0xF010,
	rINFORM5 			= SYSCON_BASE+0xF014,
	rINFORM6 			= SYSCON_BASE+0xF018,
	rINFORM7 			= SYSCON_BASE+0xF01C,
};


#ifdef DBG_SYSC
	#define DbgSysc(x) Dbg x
#else
	#define DbgSysc(x) 0;
#endif

#ifdef DISP_SYSC_REG
#define syscOutp32(a, d) {Disp("Outp32(\'h%08x, \'h%08x);\n", a, d); Outp32(a, d);}
#define syscInp32(a) {Inp32(a); u32 d2; d2=Inp32(a); Disp("Inp32(\'h%08x); // d=0x%08x\n", a, d2);}
#else
#define syscOutp32 Outp32
#define syscInp32 Inp32
#endif


volatile u32 g_uProductID, g_uPkgMode, g_uMainRev, g_uSubRev; // Chip & Revision inform
volatile u32 g_uAPLL, g_uMPLL, g_uEPLL, g_uVPLL; // SCLKapll,mpll,epll,vpll
volatile u32 g_uARMCLK, g_uHclkMsys, g_uHclkDsys, g_uHclkPsys, g_uPclkMsys, g_uPclkDsys, g_uPclkPsys, g_uOneDRAM; // Core & Bus clocks
// Added by Jang, WJ
volatile u32 g_uSclkA2M, g_uSclkG3D, g_uSclkMFC;
volatile u32 g_uSclkFIMD, g_uSclkMDNIE, g_uSclkMDNIEPWM;
volatile u32 g_uSclkFIMC, g_uSclkFIMC0LCLK, g_uSclkFIMC1LCLK, g_uSclkFIMC2LCLK, g_uSclkCAM0, g_uSclkCAM1;
volatile u32 g_uSclkPIXEL, g_uSclkHDMIPHY, g_uSclkHDMI, g_uSclkMIXER, g_uSclkTVENC;


//////////////////////////////////////////////
// Chip ID
//////////////////////////////////////////////

//////////
// Function Name : SYSC_ReadSystemID
// Function Description : This function reads system ID register
// Input : NONE 
// Output : NONE
// Version : 
void SYSC_ReadSystemID(void)
{
	u32 temp;

	temp = Inp32(rPRODUCT_ID);

	g_uProductID	=	(temp>>12)&0xfffff;
//	Assert(g_uProductID == 0x43110);
	
	g_uPkgMode 	=	(temp>>8)&0x3;
	g_uMainRev	=	(temp>>4)&0xf;
	g_uSubRev 	=	(temp)&0xf;

	return;
}


//////////////////////////////////////////////
// Clock Management Unit (CMU - SYS OFF region)
//////////////////////////////////////////////

//////////
// Function Name : SYSC_GetClkInform
// Function Description : This function gets common clock information
// Input : NONE 
// Output : NONE
// Version : 
void SYSC_UpdateClkInform(void)
{
	// PLLs
	u16 usApllM, usApllP, usApllS, usMpllM, usMpllP, usMpllS, usEpllM, usEpllP, usEpllS, usVpllM, usVpllP, usVpllS;
	// MUXs
	u8 ucMuxApll, ucMuxMpll, ucMuxEpll, ucMuxVpllSrc, ucMuxVpll, ucMuxMsys, ucMuxDsys, ucMuxPsys, ucMuxOneDRAM;
	// DIVIDERs
	u8 ucDivApll, ucDivAM, ucDivOneDRAM,	
		ucDivHclkMsys, ucDivHclkDsys, ucDivHclkPsys, 
		ucDivPclkMsys, ucDivPclkDsys, ucDivPclkPsys;
	
	u32 uTemp, uVpllFin, uMsysMuxClk, uDsysMuxClk, uPsysMuxClk;
 
	////
	// clock division ratio	
	uTemp = Inp32(rCLK_DIV0);
	ucDivApll = uTemp & 0x7;	// ARMCLK
	ucDivAM = (uTemp>>4) & 0x7;
	ucDivHclkMsys = (uTemp>>8) & 0x7;
	ucDivPclkMsys = (uTemp>>12) & 0x7;
	ucDivHclkDsys = (uTemp>>16) & 0xf;
	ucDivPclkDsys = (uTemp>>20) & 0x7;
	ucDivHclkPsys = (uTemp>>24) & 0xf;
	ucDivPclkPsys = (uTemp>>28) & 0x7;

	uTemp = Inp32(rCLK_DIV6);
	ucDivOneDRAM = (uTemp>>28) & 0xf;

	uTemp = Inp32(rAPLL_CON);
	usApllM = (uTemp>>16)&0x3ff;
	usApllP = (uTemp>>8)&0x3f;
	usApllS = (uTemp&0x7);

	uTemp = Inp32(rMPLL_CON);
	usMpllM = (uTemp>>16)&0x3ff;
	usMpllP = (uTemp>>8)&0x3f;
	usMpllS = (uTemp&0x7);

	uTemp = Inp32(rEPLL_CON);
	usEpllM = (uTemp>>16)&0x3ff;
	usEpllP = (uTemp>>8)&0x3f;
	usEpllS = (uTemp&0x7);

	uTemp = Inp32(rVPLL_CON);
	usVpllM = (uTemp>>16)&0x3ff;
	usVpllP = (uTemp>>8)&0x3f;
	usVpllS = (uTemp&0x7);

	////
	// APLL out
	uTemp = Inp32(rCLK_SRC0);
	ucMuxApll = uTemp & 0x1;
	if(ucMuxApll)	//FOUTapll
	{			
#ifndef FPGA	
		g_uAPLL = ((FIN>>(usApllS-1))/usApllP)*usApllM;
#else // fpga
		g_uAPLL = ((FIN>>(usApllS))/usApllP)*usApllM;
#endif
	}
	else	//FIN
	{
		g_uAPLL = FIN;
	}
	
	////
	// MPLL out
	ucMuxMpll = (uTemp>>4) & 0x1;
	if(ucMuxMpll)	//FOUTmpll
	{			
		g_uMPLL = ((FIN>>(usMpllS))/usMpllP)*usMpllM;
	}
	else	//FIN
	{
		g_uMPLL = FIN;
	}

	////
	// EPLL out
	ucMuxEpll = (uTemp>>8) & 0x1;
	if(ucMuxEpll)	//FOUTepll
	{			
		g_uEPLL = ((FIN>>(usEpllS))/usEpllP)*usEpllM;
	}
	else	//FIN
	{
		g_uEPLL = FIN;
	}

	////
	// VPLL out
	ucMuxVpll = (uTemp>>12) & 0x1;
	uTemp = Inp32(rCLK_SRC1);
	ucMuxVpllSrc = (uTemp>>28) & 0x1;
	if (ucMuxVpllSrc)
	{
		uVpllFin = 27000000; // SCLK_HDMI27M : HDMI PHY clock out
	}
	else
	{
		uVpllFin = FIN;
	}
	if(ucMuxVpll)	//FOUTvpll
	{			
		g_uVPLL = ((uVpllFin>>(usVpllS))/usVpllP)*usVpllM;
	}
	else	//FINvpll
	{
		g_uVPLL = uVpllFin;
	}

	////
	// MSYS domain & ARMCLK
	uTemp = Inp32(rCLK_SRC0);
	ucMuxMsys = (uTemp>>16) & 0x1;
	if(ucMuxMsys)	
	{			
		uMsysMuxClk = g_uMPLL;
	}
	else	
	{
		uMsysMuxClk = g_uAPLL;
	}

	g_uARMCLK = uMsysMuxClk/(ucDivApll+1);

	// HCLK MSYS & ACLK200 & DMC1 clock
	g_uHclkMsys = g_uARMCLK/(ucDivHclkMsys+1);

	// PCLK MSYS
	g_uPclkMsys = g_uHclkMsys/(ucDivPclkMsys+1);

	// A2M clock
	g_uSclkA2M = g_uAPLL/(ucDivAM+1);

	
	////
	// DSYS domain
	ucMuxDsys = (uTemp>>20) & 0x1;
	if(ucMuxDsys)	
	{			
		uDsysMuxClk = g_uSclkA2M;
	}
	else	
	{
		uDsysMuxClk = g_uMPLL;
	}

	// HCLK DSYS
	g_uHclkDsys = uDsysMuxClk/(ucDivHclkDsys+1);

	// PCLK DSYS
	g_uPclkDsys = g_uHclkDsys/(ucDivPclkDsys+1);


	////
	// PSYS domain
	ucMuxPsys = (uTemp>>24) & 0x1;
	if(ucMuxPsys)	
	{			
		uPsysMuxClk = g_uSclkA2M;
	}
	else	
	{
		uPsysMuxClk = g_uMPLL;
	}

	// HCLK PSYS
	g_uHclkPsys = uPsysMuxClk/(ucDivHclkPsys+1);

	// PCLK PSYS
	g_uPclkPsys = g_uHclkPsys/(ucDivPclkPsys+1);


	////
	// OneDRAM(DMC0) clock
	uTemp = Inp32(rCLK_SRC6);
	ucMuxOneDRAM = (uTemp>>24) & 0x3;
	switch (ucMuxOneDRAM)
	{
	case 1 : // SCLKMPLL
		g_uOneDRAM = g_uMPLL/(ucDivOneDRAM+1);
		break;
	case 2 : // SCLKEPLL
		g_uOneDRAM = g_uEPLL/(ucDivOneDRAM+1);
		break;
	case 3 : // SCLKVPLL
		g_uOneDRAM = g_uVPLL/(ucDivOneDRAM+1);
		break;
	default : // SCLKA2M
		g_uOneDRAM = g_uSclkA2M/(ucDivOneDRAM+1);
		break;
	}

}

u32	SYSC_GetARMClock(void)
{
	SYSC_UpdateClkInform();
	
	return	g_uARMCLK;
}

// Is lock detected?
// After counter value becomes 0 in LOCK_TIME register, this lock will be detected
bool SYSC_IsLockDetected(PLL_TYPE ePLL)
{
	u32 uRegValue;
	u32 uRegAddr;
	u32 uLockDetect;

	uRegAddr = (ePLL == eAPLL) ? rAPLL_CON : (ePLL == eMPLL) ? rMPLL_CON : (ePLL == eEPLL) ? rEPLL_CON : rVPLL_CON;

	uRegValue = syscInp32(uRegAddr);
	syscOutp32(uRegAddr,uRegValue|(1<<28)); // enable lock detect
	
	uRegValue = syscInp32(uRegAddr);
	uLockDetect = (uRegValue >>29) & 0x1;

	return (uLockDetect) ? true : false;
}

// Set Lock time
// Function Description : This function control APLL, M/H/EPLL Lock Time
// APLL - pll6522x (locking time=300us), M,E,VPLL - pll6545x (locking time=300us)
//				 Fout = (mdiv * Fin) / (pdiv x 2^sdiv)
// Input : 	uLockTime : Max 300us, us(micro second) order
void SYSC_SetLockTime(PLL_TYPE ePLL, u32 uLockTime)
{
	u32 uLockCount;
	u32 uRegAddr;

	uLockCount = (u32)((float)uLockTime/((float)(1.0e6/(float)FIN)));

	uRegAddr = (ePLL == eAPLL) ? rAPLL_LOCK : (ePLL == eMPLL) ? rMPLL_LOCK : (ePLL == eEPLL) ? rEPLL_LOCK : rVPLL_LOCK;

	syscOutp32(uRegAddr, uLockCount);
}

// Get lock time
// uLockTime : Max 100us, us order (Max 100us)
u32 SYSC_GetLockTime(PLL_TYPE ePLL)
{
	u32 uRegAddr;
	u32 uLockTime;

	uRegAddr = (ePLL == eAPLL) ? rAPLL_LOCK : (ePLL == eMPLL) ? rMPLL_LOCK : (ePLL == eEPLL) ? rEPLL_LOCK : rVPLL_LOCK;

	uLockTime = syscInp32(uRegAddr);
	return uLockTime;
}


u32 SYSC_GetPLLReg(PLL_TYPE ePLL)
{
	u32 uRegValue;
	
	switch(ePLL)
	{
		case eAPLL:
			uRegValue = syscInp32(rAPLL_CON);
			break;
		case eMPLL:
			uRegValue = syscInp32(rMPLL_CON);
			break;
		case eEPLL:
			uRegValue = syscInp32(rEPLL_CON);
			break;
		case eVPLL:
			uRegValue = syscInp32(rVPLL_CON);
			break;
		default:
			DbgSysc(("No appropriate PLL source is found in GetPLLReg()\n\x04"));
			Assert(0);
	}

	return uRegValue;
}


void SYSC_SetPLLReg(PLL_TYPE ePLL, u32 uRegValue)
{
	switch(ePLL)
	{
		case eAPLL:
			syscOutp32(rAPLL_CON, uRegValue);
			break;
		case eMPLL:
			syscOutp32(rMPLL_CON, uRegValue);
			break;
		case eEPLL:
			syscOutp32(rEPLL_CON, uRegValue);
			break;
		case eVPLL:
			syscOutp32(rVPLL_CON, uRegValue);
			break;
		default:
			DbgSysc(("No appropriate PLL source is found in StartPllbyReg()\n\x04"));
			Assert(0);
	}
}


// Start PLL (PLL type, divider value(m,p,s))
// Function Description : This function control PLL Output Frequency
// APLL - pll6522x, M,E,VPLL - pll6545x
void SYSC_StartPLLbyMPS(PLL_TYPE ePLL, u32 uMdiv, u32 uPdiv, u32 uSdiv)
{
	u32 uRegValue;
	u32 uRegAddr;
	u32 uFvco;
	u32 uVsel=0;

	switch(ePLL)
	{
		case eAPLL:
#ifndef FPGA
			// Check the Divider Value.
			Assert((uMdiv >= 64) && (uMdiv <= 1023));
			Assert((uPdiv >= 1) && (uPdiv <= 63));
			Assert((uPdiv >= 1) && (uSdiv <= 5));

			// Check the Fvco Range
			uFvco = (2*(FIN/uPdiv)*uMdiv)/1000000;
			Assert((uFvco >= 960) && (uFvco <= 2060));
#endif
			uRegAddr = rAPLL_CON;
			break;

		case eMPLL:
#ifndef FPGA
			// Check the Divider Value.
			Assert((uMdiv >= 64) && (uMdiv <= 1023));
			Assert((uPdiv >= 1) && (uPdiv <= 63));
			Assert((uPdiv >= 0) && (uSdiv <= 5));

			// Check the Fvco Range
			uFvco = ((FIN/uPdiv)*uMdiv)/1000000;
			if ( (uFvco>=1000) && (uFvco<=1400) ) 
				uVsel = 0;
			else if ( (uFvco>1400) && (uFvco<=2000) )
				uVsel = 1;
			else 
				Assert(0);
#endif
			uRegAddr = rMPLL_CON;
			break;
			
		case eEPLL:
		case eVPLL:
#ifndef FPGA
			// Check the Divider Value.
			Assert((uMdiv >= 64) && (uMdiv <= 511));
			Assert((uPdiv >= 1) && (uPdiv <= 63));
			Assert((uPdiv >= 0) && (uSdiv <= 5));

			// Check the Fvco Range
			uFvco = ((FIN/uPdiv)*uMdiv)/1000000;
			if ( (uFvco>=330) && (uFvco<=460) ) 
				uVsel = 0;
			else if ( (uFvco>460) && (uFvco<=660) )
				uVsel = 1;
			else 
				Assert(0);
#endif
			uRegAddr = (ePLL == eEPLL) ? rEPLL_CON : rVPLL_CON;
			break;

		default:
			DbgSysc(("No appropriate PLL source is found in StartPll()\n\x04"));
			Assert(0);
	}

	uRegValue = (0<<31)|(0<<28)|(uVsel<<27)|(uMdiv<<16)|(uPdiv<<8)|(uSdiv<<0);
	syscOutp32(uRegAddr, uRegValue|(1<<31)); 

	// Check PLL locked
	while(!(syscInp32(uRegAddr) & (1<<29))) ;

	// Update bus clock Information
	SYSC_UpdateClkInform();
}


// Start PLL (PLL type, target Fout freq)
// FOUT = MDIV X FIN / (PDIV X 2^SDIV)
// FIN = 12MHz (default)
void SYSC_StartPLLbyFout(PLL_TYPE ePLL, u32 uFout)
{
	u32 uMdiv, uPdiv, uSdiv, uVsel;

#if FIN == 12000000
#ifndef FPGA // silicon
	if (ePLL == eAPLL)
	{
		switch(uFout)
		{
			// for 200
			case 1000*1000000: 
				uMdiv = 250; uPdiv = 3; uSdiv = 1; break;
			case 800*1000000: 
				uMdiv = 200; uPdiv = 3; uSdiv = 1; break;
			case 600*1000000: 
				uMdiv = 100; uPdiv = 2; uSdiv = 1; break;
			case 400*1000000: 
				uMdiv = 200; uPdiv = 3; uSdiv = 2; break;
			case 200*1000000: 
				uMdiv = 200; uPdiv = 3; uSdiv = 3; break;
			// for 166
			case 833*1000000: 
			case 832800000: 
				uMdiv = 347; uPdiv = 5; uSdiv = 1; break;
			case 667*1000000: // target
			case 666857000: // real
				uMdiv = 389; uPdiv = 7; uSdiv = 1; break;
			case 533*1000000: 
				uMdiv = 533; uPdiv = 12; uSdiv = 1; break;

			default:
				Assert(0);
		}
	}
	else if (ePLL == eMPLL)
	{
		switch(uFout)
		{
			// for common 166 & 133
			case 1992*1000000:
				uMdiv = 498; uPdiv = 3; uSdiv = 0; break;	
			case 1328*1000000:
				uMdiv = 332; uPdiv = 3; uSdiv = 0; break;	
			case 1200*1000000:
				uMdiv = 300; uPdiv = 3; uSdiv = 0; break;	
			case 667*1000000:
				uMdiv = 667; uPdiv = 6; uSdiv = 1; break;
			// for 166
			case 333*1000000:
				uMdiv = 333; uPdiv = 3; uSdiv = 2; break;
			case 166*1000000:
				uMdiv = 332; uPdiv = 3; uSdiv = 3; break;
			// for 133
			case 266*1000000:
				uMdiv = 266; uPdiv = 3; uSdiv = 2; break;
			case 133*1000000:
				uMdiv = 266; uPdiv = 3; uSdiv = 3; break;	
			// for 200 & 100
			case 400*1000000:
				uMdiv = 400; uPdiv = 3; uSdiv = 2; break;	
			case 200*1000000:
				uMdiv = 400; uPdiv = 3; uSdiv = 3; break;	
			// for 120.8~148.8
			case 604*1000000:
				uMdiv = 302; uPdiv = 3; uSdiv = 1; break;	
			case 624*1000000:
				uMdiv = 312; uPdiv = 3; uSdiv = 1; break;	
			case 644*1000000:
				uMdiv = 322; uPdiv = 3; uSdiv = 1; break;	
			case 664*1000000:
				uMdiv = 332; uPdiv = 3; uSdiv = 1; break;	
			case 684*1000000:
				uMdiv = 342; uPdiv = 3; uSdiv = 1; break;	
			case 704*1000000:
				uMdiv = 352; uPdiv = 3; uSdiv = 1; break;	
			case 724*1000000:
				uMdiv = 362; uPdiv = 3; uSdiv = 1; break;	
			case 744*1000000:
				uMdiv = 372; uPdiv = 3; uSdiv = 1; break;	

			default:
				Assert(0);
		}
	}
	else if ( (ePLL==eEPLL) ||(ePLL==eVPLL) )
	{
		switch(uFout)
		{
			// for USB
			case 192*1000000:
				uMdiv = 96; uPdiv = 3; uSdiv = 1; break;
			case 144*1000000:
				uMdiv = 144; uPdiv = 3; uSdiv = 2; break;
			case 96*1000000:
				uMdiv = 96; uPdiv = 3; uSdiv = 2; break;
			case 48*1000000:
				uMdiv = 96; uPdiv = 3; uSdiv = 3; break;
			// for I2S
			case 73728000: // target
			case 73750000: // real
				uMdiv = 295; uPdiv = 6; uSdiv = 3; break;
			case 67738000:
			case 67750000:
				uMdiv = 271; uPdiv = 6; uSdiv = 3; break;
//			case 49152000:
			case 49152000:
				uMdiv = 131; uPdiv = 4; uSdiv = 3; break;
			case 45158000:
			case 45167000:
				uMdiv = 271; uPdiv = 9; uSdiv = 3; break;
			case 32768000:
			case 32750000:
				uMdiv = 131; uPdiv = 3; uSdiv = 4; break;
			// for MMC
			case 80*1000000:
				uMdiv = 160; uPdiv = 3; uSdiv = 3; break;
			case 50*1000000:
				uMdiv = 100; uPdiv = 3; uSdiv = 3; break;
			// for TVout @ 12MHz
			case 108*1000000:
				uMdiv = 108; uPdiv = 3; uSdiv = 2; break;
			case 54*1000000:
				uMdiv = 108; uPdiv = 3; uSdiv = 3; break;
			// for MHL @ 12MHz
			case 74250000:
				uMdiv = 198; uPdiv = 4; uSdiv = 3; break;
			case 27027000: // target
			case 27000000: // real
				uMdiv = 108; uPdiv = 3; uSdiv = 4; break;
			default:
				Assert(0);
		}
	}
	else
		Assert(0);
#else // fpga
	switch(uFout)
	{
		case 36*1000000:
			uMdiv = 12; uPdiv = 4; uSdiv = 0; break;
		case 30*1000000:
			uMdiv = 10; uPdiv = 4; uSdiv = 0; break;
		case 24*1000000:
			uMdiv = 12; uPdiv = 3; uSdiv = 1; break;
		case 20*1000000:
			uMdiv = 10; uPdiv = 3; uSdiv = 1; break;
		default:
			Assert(0);
	}
#endif // FPGA
#endif // FIN 12MHz


#if FIN == 24000000
#ifndef FPGA // silicon
	if (ePLL == eAPLL)
	{
		switch(uFout)
		{
			// for 200
			case 1000*1000000: 
				uMdiv = 250; uPdiv = 6; uSdiv = 1; break;
			case 800*1000000: 
				uMdiv = 200; uPdiv = 6; uSdiv = 1; break;
			case 600*1000000: 
				uMdiv = 100; uPdiv = 4; uSdiv = 1; break;
			case 400*1000000: 
				uMdiv = 200; uPdiv = 6; uSdiv = 2; break;
			case 200*1000000: 
				uMdiv = 200; uPdiv = 6; uSdiv = 3; break;
			// for 166
			case 833*1000000: 
			case 832800000: 
				uMdiv = 347; uPdiv = 10; uSdiv = 1; break;
			case 667*1000000: // target
			case 666857000: // real
				uMdiv = 389; uPdiv = 14; uSdiv = 1; break;
			case 533*1000000: 
				uMdiv = 533; uPdiv = 24; uSdiv = 1; break;

			default:
				Assert(0);
		}
	}
	else if (ePLL == eMPLL)
	{
		switch(uFout)
		{
			// for common 166 & 133
			case 1992*1000000:
				uMdiv = 498; uPdiv = 6; uSdiv = 0; break;	
			case 1328*1000000:
				uMdiv = 332; uPdiv = 6; uSdiv = 0; break;	
			case 1200*1000000:
				uMdiv = 300; uPdiv = 6; uSdiv = 0; break;	
			case 667*1000000:
				uMdiv = 667; uPdiv = 12; uSdiv = 1; break;
			// for 166
			case 333*1000000:
				uMdiv = 333; uPdiv = 6; uSdiv = 2; break;
			case 166*1000000:
				uMdiv = 332; uPdiv = 6; uSdiv = 3; break;
			// for 133
			case 266*1000000:
				uMdiv = 266; uPdiv = 6; uSdiv = 2; break;
			case 133*1000000:
				uMdiv = 266; uPdiv = 6; uSdiv = 3; break;	
			// for 200 & 100
			case 400*1000000:
				uMdiv = 400; uPdiv = 6; uSdiv = 2; break;	
			case 200*1000000:
				uMdiv = 400; uPdiv = 6; uSdiv = 3; break;	
			// for 120.8~148.8
			case 604*1000000:
				uMdiv = 302; uPdiv = 6; uSdiv = 1; break;	
			case 624*1000000:
				uMdiv = 312; uPdiv = 6; uSdiv = 1; break;	
			case 644*1000000:
				uMdiv = 322; uPdiv = 6; uSdiv = 1; break;	
			case 664*1000000:
				uMdiv = 332; uPdiv = 6; uSdiv = 1; break;	
			case 684*1000000:
				uMdiv = 342; uPdiv = 6; uSdiv = 1; break;	
			case 704*1000000:
				uMdiv = 352; uPdiv = 6; uSdiv = 1; break;	
			case 724*1000000:
				uMdiv = 362; uPdiv = 6; uSdiv = 1; break;	
			case 744*1000000:
				uMdiv = 372; uPdiv = 6; uSdiv = 1; break;	

			default:
				Assert(0);
		}
	}
	else if ( (ePLL==eEPLL) ||(ePLL==eVPLL) )
	{
		switch(uFout)
		{
			// for USB
			case 192*1000000:
				uMdiv = 96; uPdiv = 6; uSdiv = 1; break;
			case 144*1000000:
				uMdiv = 144; uPdiv = 6; uSdiv = 2; break;
			case 96*1000000:
				uMdiv = 96; uPdiv = 6; uSdiv = 2; break;
			case 48*1000000:
				uMdiv = 96; uPdiv = 6; uSdiv = 3; break;
			// for I2S
			case 73728000: // target
			case 73750000: // real
				uMdiv = 295; uPdiv = 12; uSdiv = 3; break;
			case 67738000:
			case 67750000:
				uMdiv = 271; uPdiv = 12; uSdiv = 3; break;
//			case 49152000:
			case 49152000:
				uMdiv = 131; uPdiv = 8; uSdiv = 3; break;
			case 45158000:
			case 45167000:
				uMdiv = 271; uPdiv = 18; uSdiv = 3; break;
			case 32768000:
			case 32750000:
				uMdiv = 131; uPdiv = 6; uSdiv = 4; break;
			// for MMC
			case 80*1000000:
				uMdiv = 160; uPdiv = 6; uSdiv = 3; break;
			case 50*1000000:
				uMdiv = 100; uPdiv = 6; uSdiv = 3; break;
			// for TVout @ 12MHz
			case 108*1000000:
				uMdiv = 108; uPdiv = 6; uSdiv = 2; break;
			case 54*1000000:
				uMdiv = 108; uPdiv = 6; uSdiv = 3; break;
			// for MHL @ 12MHz
			case 74250000:
				uMdiv = 198; uPdiv = 8; uSdiv = 3; break;
			case 27027000: // target
			case 27000000: // real
				uMdiv = 108; uPdiv = 6; uSdiv = 4; break;
			default:
				Assert(0);
		}
	}
	else
		Assert(0);
#else // fpga
	switch(uFout)
	{
		case 36*1000000:
			uMdiv = 12; uPdiv = 8; uSdiv = 0; break;
		case 30*1000000:
			uMdiv = 10; uPdiv = 8; uSdiv = 0; break;
		case 24*1000000:
			uMdiv = 12; uPdiv = 6; uSdiv = 1; break;
		case 20*1000000:
			uMdiv = 10; uPdiv = 6; uSdiv = 1; break;
		default:
			Assert(0);
	}
#endif // FPGA
#endif // FIN 24MHz

	SYSC_StartPLLbyMPS(ePLL, uMdiv, uPdiv, uSdiv);
}


// Start PLL
void SYSC_StartPLL(PLL_TYPE ePLL)
{
	u32 uRegValue;
	u32 uRegAddr;

	uRegAddr = (ePLL == eAPLL) ? rAPLL_CON : (ePLL == eMPLL) ? rMPLL_CON : (ePLL == eEPLL) ? rEPLL_CON : rVPLL_CON;

	uRegValue = syscInp32(uRegAddr);
	uRegValue = (uRegValue | (0x1U<<31));
	syscOutp32(uRegAddr, uRegValue);
}


// Stop PLL
void SYSC_StopPLL(PLL_TYPE ePLL)
{
	u32 uRegValue;
	u32 uRegAddr;

	uRegAddr = (ePLL == eAPLL) ? rAPLL_CON : (ePLL == eMPLL) ? rMPLL_CON : (ePLL == eEPLL) ? rEPLL_CON : rVPLL_CON;

	uRegValue = syscInp32(uRegAddr);
	uRegValue = (uRegValue & ~(0x1U<<31));
	syscOutp32(uRegAddr, uRegValue);
}

// Calc. PLL FOUT
u32 SYSC_CalPllFout(PLL_TYPE uPLL_TYPE, u32 uMdiv, u32 uPdiv, u32 uSdiv)
{
	u32 uPllFout;
	u32 uFvco;
	u32 uVsel;
	
	switch(uPLL_TYPE) {
		case eAPLL :
			Assert((uMdiv >= 64) && (uMdiv <= 1023));
			Assert((uPdiv >= 1) && (uPdiv <= 63));
			Assert((uSdiv >= 1) && (uSdiv <= 5));

 			uFvco = (2*(FIN/uPdiv)*uMdiv)/1000000;
			Assert((uFvco >= 960) && (uFvco <= 2060));
			uPllFout=((FIN>>(uSdiv-1))/uPdiv)*uMdiv;
			break;

		case eMPLL :
			Assert((uMdiv >= 64) && (uMdiv <= 1023));
			Assert((uPdiv >= 1) && (uPdiv <= 63));
			Assert((uSdiv >= 0) && (uSdiv <= 4));

 			uFvco = ((FIN/uPdiv)*uMdiv)/1000000;
			if ( (uFvco>=1000) && (uFvco<=1400) ) 
				uVsel = 0;
			else if ( (uFvco>1400) && (uFvco<=2000) )
				uVsel = 1;
			else 
				Assert(0);
			
			uPllFout=((FIN>>(uSdiv))/uPdiv)*uMdiv;
			break;
			
		case eEPLL :
		case eVPLL :
			Assert((uMdiv >= 16) && (uMdiv <= 255));
			Assert((uPdiv >= 1) && (uPdiv <= 63));
			Assert((uSdiv >= 0) && (uSdiv <= 4));

 			uFvco = ((FIN/uPdiv)*uMdiv)/1000000;
			if ( (uFvco>=330) && (uFvco<=460) ) 
				uVsel = 0;
			else if ( (uFvco>460) && (uFvco<=660) )
				uVsel = 1;
			else 
				Assert(0);

			uPllFout=((FIN>>(uSdiv))/uPdiv)*uMdiv;
			break;
	}
	
	return uPllFout; // Unit : Hz
}

// Set source clock
void SYSC_SetClkMuxState(CLKMUX_STATE_ID eId)
{
	u32 uRegAddr;
	u32 uRegData;
	u32 uRegBitOffset;
	u32 uRegTemp;

	switch(eId)
	{
		// CLK_SRC0
		case eCLKMUX_OND_HCLKPSYS:
			uRegAddr = rCLK_SRC0; uRegData = 0x0; uRegBitOffset = 28; break;
		case eCLKMUX_OND_HCLKDSYS:
			uRegAddr = rCLK_SRC0; uRegData = 0x1; uRegBitOffset = 28; break;
		case eCLKMUX_PSYS_SCLKMPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x0; uRegBitOffset = 24; break;
		case eCLKMUX_PSYS_SCLKA2M:
			uRegAddr = rCLK_SRC0; uRegData = 0x1; uRegBitOffset = 24; break;
		case eCLKMUX_DSYS_SCLKMPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x0; uRegBitOffset = 20; break;
		case eCLKMUX_DSYS_SCLKA2M:
			uRegAddr = rCLK_SRC0; uRegData = 0x1; uRegBitOffset = 20; break;
		case eCLKMUX_MSYS_SCLKAPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x0; uRegBitOffset = 16; break;
		case eCLKMUX_MSYS_SCLKMPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x1; uRegBitOffset = 16; break;
		case eCLKMUX_VPLL_FINVPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x0; uRegBitOffset = 12; break;
		case eCLKMUX_VPLL_FOUTVPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x1; uRegBitOffset = 12; break;
		case eCLKMUX_EPLL_FINPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x0; uRegBitOffset = 8; break;
		case eCLKMUX_EPLL_FOUTEPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x1; uRegBitOffset = 8; break;
		case eCLKMUX_MPLL_FINPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x0; uRegBitOffset = 4; break;
		case eCLKMUX_MPLL_FOUTMPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x1; uRegBitOffset = 4; break;
		case eCLKMUX_APLL_FINPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x0; uRegBitOffset = 0; break;
		case eCLKMUX_APLL_FOUTAPLL:
			uRegAddr = rCLK_SRC0; uRegData = 0x1; uRegBitOffset = 0; break;

		// CLK_SRC1
		case eCLKMUX_VPLLSRC_FINPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x0; uRegBitOffset = 28; break;
		case eCLKMUX_VPLLSRC_SHDMI27M:
			uRegAddr = rCLK_SRC1; uRegData = 0x1; uRegBitOffset = 28; break;
		case eCLKMUX_CSIS_XXTI:
			uRegAddr = rCLK_SRC1; uRegData = 0x0; uRegBitOffset = 24; break;
		case eCLKMUX_CSIS_XUSBXTI:
			uRegAddr = rCLK_SRC1; uRegData = 0x1; uRegBitOffset = 24; break;
		case eCLKMUX_CSIS_SHDMI27M:
			uRegAddr = rCLK_SRC1; uRegData = 0x2; uRegBitOffset = 24; break;
		case eCLKMUX_CSIS_SUSBPHY0:
			uRegAddr = rCLK_SRC1; uRegData = 0x3; uRegBitOffset = 24; break;
		case eCLKMUX_CSIS_SUSBPHY1:
			uRegAddr = rCLK_SRC1; uRegData = 0x4; uRegBitOffset = 24; break;
		case eCLKMUX_CSIS_SHDMIPHY:
			uRegAddr = rCLK_SRC1; uRegData = 0x5; uRegBitOffset = 24; break;
		case eCLKMUX_CSIS_SMPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x6; uRegBitOffset = 24; break;
		case eCLKMUX_CSIS_SEPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x7; uRegBitOffset = 24; break;
		case eCLKMUX_CSIS_SVPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x8; uRegBitOffset = 24; break;
		case eCLKMUX_FIMD_XXTI:
			uRegAddr = rCLK_SRC1; uRegData = 0x0; uRegBitOffset = 20; break;
		case eCLKMUX_FIMD_XUSBXTI:
			uRegAddr = rCLK_SRC1; uRegData = 0x1; uRegBitOffset = 20; break;
		case eCLKMUX_FIMD_SHDMI27M:
			uRegAddr = rCLK_SRC1; uRegData = 0x2; uRegBitOffset = 20; break;
		case eCLKMUX_FIMD_SUSBPHY0:
			uRegAddr = rCLK_SRC1; uRegData = 0x3; uRegBitOffset = 20; break;
		case eCLKMUX_FIMD_SUSBPHY1:
			uRegAddr = rCLK_SRC1; uRegData = 0x4; uRegBitOffset = 20; break;
		case eCLKMUX_FIMD_SHDMIPHY:
			uRegAddr = rCLK_SRC1; uRegData = 0x5; uRegBitOffset = 20; break;
		case eCLKMUX_FIMD_SMPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x6; uRegBitOffset = 20; break;
		case eCLKMUX_FIMD_SEPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x7; uRegBitOffset = 20; break;
		case eCLKMUX_FIMD_SVPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x8; uRegBitOffset = 20; break;
		case eCLKMUX_CAM1_XXTI:
			uRegAddr = rCLK_SRC1; uRegData = 0x0; uRegBitOffset = 16; break;
		case eCLKMUX_CAM1_XUSBXTI:
			uRegAddr = rCLK_SRC1; uRegData = 0x1; uRegBitOffset = 16; break;
		case eCLKMUX_CAM1_SHDMI27M:
			uRegAddr = rCLK_SRC1; uRegData = 0x2; uRegBitOffset = 16; break;
		case eCLKMUX_CAM1_SUSBPHY0:
			uRegAddr = rCLK_SRC1; uRegData = 0x3; uRegBitOffset = 16; break;
		case eCLKMUX_CAM1_SUSBPHY1:
			uRegAddr = rCLK_SRC1; uRegData = 0x4; uRegBitOffset = 16; break;
		case eCLKMUX_CAM1_SHDMIPHY:
			uRegAddr = rCLK_SRC1; uRegData = 0x5; uRegBitOffset = 16; break;
		case eCLKMUX_CAM1_SMPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x6; uRegBitOffset = 16; break;
		case eCLKMUX_CAM1_SEPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x7; uRegBitOffset = 16; break;
		case eCLKMUX_CAM1_SVPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x8; uRegBitOffset = 16; break;
		case eCLKMUX_CAM0_XXTI:
			uRegAddr = rCLK_SRC1; uRegData = 0x0; uRegBitOffset = 12; break;
		case eCLKMUX_CAM0_XUSBXTI:
			uRegAddr = rCLK_SRC1; uRegData = 0x1; uRegBitOffset = 12; break;
		case eCLKMUX_CAM0_SHDMI27M:
			uRegAddr = rCLK_SRC1; uRegData = 0x2; uRegBitOffset = 12; break;
		case eCLKMUX_CAM0_SUSBPHY0:
			uRegAddr = rCLK_SRC1; uRegData = 0x3; uRegBitOffset = 12; break;
		case eCLKMUX_CAM0_SUSBPHY1:
			uRegAddr = rCLK_SRC1; uRegData = 0x4; uRegBitOffset = 12; break;
		case eCLKMUX_CAM0_SHDMIPHY:
			uRegAddr = rCLK_SRC1; uRegData = 0x5; uRegBitOffset = 12; break;
		case eCLKMUX_CAM0_SMPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x6; uRegBitOffset = 12; break;
		case eCLKMUX_CAM0_SEPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x7; uRegBitOffset = 12; break;
		case eCLKMUX_CAM0_SVPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x8; uRegBitOffset = 12; break;
		case eCLKMUX_DAC_SVPLL:
			uRegAddr = rCLK_SRC1; uRegData = 0x0; uRegBitOffset = 8; break;
		case eCLKMUX_DAC_SHDMIPHY:
			uRegAddr = rCLK_SRC1; uRegData = 0x1; uRegBitOffset = 8; break;
		case eCLKMUX_MIXER_SDAC:
			uRegAddr = rCLK_SRC1; uRegData = 0x0; uRegBitOffset = 4; break;
		case eCLKMUX_MIXER_SHDMI:
			uRegAddr = rCLK_SRC1; uRegData = 0x1; uRegBitOffset = 4; break;
		case eCLKMUX_HDMI_SPIXEL:
			uRegAddr = rCLK_SRC1; uRegData = 0x0; uRegBitOffset = 0; break;
		case eCLKMUX_HDMI_SHDMIPHY:
			uRegAddr = rCLK_SRC1; uRegData = 0x1; uRegBitOffset = 0; break;

		// CLK_SRC2
		case eCLKMUX_MFC_SCLKA2M:
			uRegAddr = rCLK_SRC2; uRegData = 0x0; uRegBitOffset = 4; break;
		case eCLKMUX_MFC_SCLKMPLL:
			uRegAddr = rCLK_SRC2; uRegData = 0x1; uRegBitOffset = 4; break;
		case eCLKMUX_MFC_SCLKEPLL:
			uRegAddr = rCLK_SRC2; uRegData = 0x2; uRegBitOffset = 4; break;
		case eCLKMUX_MFC_SCLKVPLL:
			uRegAddr = rCLK_SRC2; uRegData = 0x3; uRegBitOffset = 4; break;
		case eCLKMUX_G3D_SCLKA2M:
			uRegAddr = rCLK_SRC2; uRegData = 0x0; uRegBitOffset = 0; break;
		case eCLKMUX_G3D_SCLKMPLL:
			uRegAddr = rCLK_SRC2; uRegData = 0x1; uRegBitOffset = 0; break;
		case eCLKMUX_G3D_SCLKEPLL:
			uRegAddr = rCLK_SRC2; uRegData = 0x2; uRegBitOffset = 0; break;
		case eCLKMUX_G3D_SCLKVPLL:
			uRegAddr = rCLK_SRC2; uRegData = 0x3; uRegBitOffset = 0; break;
			
		// CLK_SRC3
		case eCLKMUX_FIMC2LCLK_XXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x0; uRegBitOffset = 20; break;
		case eCLKMUX_FIMC2LCLK_XUSBXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x1; uRegBitOffset = 20; break;
		case eCLKMUX_FIMC2LCLK_SHDMI27M:
			uRegAddr = rCLK_SRC3; uRegData = 0x2; uRegBitOffset = 20; break;
		case eCLKMUX_FIMC2LCLK_SUSBPHY0:
			uRegAddr = rCLK_SRC3; uRegData = 0x3; uRegBitOffset = 20; break;
		case eCLKMUX_FIMC2LCLK_SUSBPHY1:
			uRegAddr = rCLK_SRC3; uRegData = 0x4; uRegBitOffset = 20; break;
		case eCLKMUX_FIMC2LCLK_SHDMIPHY:
			uRegAddr = rCLK_SRC3; uRegData = 0x5; uRegBitOffset = 20; break;
		case eCLKMUX_FIMC2LCLK_SMPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x6; uRegBitOffset = 20; break;
		case eCLKMUX_FIMC2LCLK_SEPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x7; uRegBitOffset = 20; break;
		case eCLKMUX_FIMC2LCLK_SVPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x8; uRegBitOffset = 20; break;
		case eCLKMUX_FIMC1LCLK_XXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x0; uRegBitOffset = 16; break;
		case eCLKMUX_FIMC1LCLK_XUSBXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x1; uRegBitOffset = 16; break;
		case eCLKMUX_FIMC1LCLK_SHDMI27M:
			uRegAddr = rCLK_SRC3; uRegData = 0x2; uRegBitOffset = 16; break;
		case eCLKMUX_FIMC1LCLK_SUSBPHY0:
			uRegAddr = rCLK_SRC3; uRegData = 0x3; uRegBitOffset = 16; break;
		case eCLKMUX_FIMC1LCLK_SUSBPHY1:
			uRegAddr = rCLK_SRC3; uRegData = 0x4; uRegBitOffset = 16; break;
		case eCLKMUX_FIMC1LCLK_SHDMIPHY:
			uRegAddr = rCLK_SRC3; uRegData = 0x5; uRegBitOffset = 16; break;
		case eCLKMUX_FIMC1LCLK_SMPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x6; uRegBitOffset = 16; break;
		case eCLKMUX_FIMC1LCLK_SEPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x7; uRegBitOffset = 16; break;
		case eCLKMUX_FIMC1LCLK_SVPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x8; uRegBitOffset = 16; break;
		case eCLKMUX_FIMC0LCLK_XXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x0; uRegBitOffset = 12; break;
		case eCLKMUX_FIMC0LCLK_XUSBXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x1; uRegBitOffset = 12; break;
		case eCLKMUX_FIMC0LCLK_SHDMI27M:
			uRegAddr = rCLK_SRC3; uRegData = 0x2; uRegBitOffset = 12; break;
		case eCLKMUX_FIMC0LCLK_SUSBPHY0:
			uRegAddr = rCLK_SRC3; uRegData = 0x3; uRegBitOffset = 12; break;
		case eCLKMUX_FIMC0LCLK_SUSBPHY1:
			uRegAddr = rCLK_SRC3; uRegData = 0x4; uRegBitOffset = 12; break;
		case eCLKMUX_FIMC0LCLK_SHDMIPHY:
			uRegAddr = rCLK_SRC3; uRegData = 0x5; uRegBitOffset = 12; break;
		case eCLKMUX_FIMC0LCLK_SMPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x6; uRegBitOffset = 12; break;
		case eCLKMUX_FIMC0LCLK_SEPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x7; uRegBitOffset = 12; break;
		case eCLKMUX_FIMC0LCLK_SVPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x8; uRegBitOffset = 12; break;
		case eCLKMUX_MDNIEPWM_XXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x0; uRegBitOffset = 4; break;
		case eCLKMUX_MDNIEPWM_XUSBXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x1; uRegBitOffset = 4; break;
		case eCLKMUX_MDNIEPWM_SHDMI27M:
			uRegAddr = rCLK_SRC3; uRegData = 0x2; uRegBitOffset = 4; break;
		case eCLKMUX_MDNIEPWM_SUSBPHY0:
			uRegAddr = rCLK_SRC3; uRegData = 0x3; uRegBitOffset = 4; break;
		case eCLKMUX_MDNIEPWM_SUSBPHY1:
			uRegAddr = rCLK_SRC3; uRegData = 0x4; uRegBitOffset = 4; break;
		case eCLKMUX_MDNIEPWM_SHDMIPHY:
			uRegAddr = rCLK_SRC3; uRegData = 0x5; uRegBitOffset = 4; break;
		case eCLKMUX_MDNIEPWM_SMPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x6; uRegBitOffset = 4; break;
		case eCLKMUX_MDNIEPWM_SEPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x7; uRegBitOffset = 4; break;
		case eCLKMUX_MDNIEPWM_SVPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x8; uRegBitOffset = 4; break;
		case eCLKMUX_MDNIE_XXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x0; uRegBitOffset = 0; break;
		case eCLKMUX_MDNIE_XUSBXTI:
			uRegAddr = rCLK_SRC3; uRegData = 0x1; uRegBitOffset = 0; break;
		case eCLKMUX_MDNIE_SHDMI27M:
			uRegAddr = rCLK_SRC3; uRegData = 0x2; uRegBitOffset = 0; break;
		case eCLKMUX_MDNIE_SUSBPHY0:
			uRegAddr = rCLK_SRC3; uRegData = 0x3; uRegBitOffset = 0; break;
		case eCLKMUX_MDNIE_SUSBPHY1:
			uRegAddr = rCLK_SRC3; uRegData = 0x4; uRegBitOffset = 0; break;
		case eCLKMUX_MDNIE_SHDMIPHY:
			uRegAddr = rCLK_SRC3; uRegData = 0x5; uRegBitOffset = 0; break;
		case eCLKMUX_MDNIE_SMPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x6; uRegBitOffset = 0; break;
		case eCLKMUX_MDNIE_SEPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x7; uRegBitOffset = 0; break;
		case eCLKMUX_MDNIE_SVPLL:
			uRegAddr = rCLK_SRC3; uRegData = 0x8; uRegBitOffset = 0; break;

		// CLK_SRC4
		case eCLKMUX_UART3_XXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x0; uRegBitOffset = 28; break;
		case eCLKMUX_UART3_XUSBXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x1; uRegBitOffset = 28; break;
		case eCLKMUX_UART3_SHDMI27M:
			uRegAddr = rCLK_SRC4; uRegData = 0x2; uRegBitOffset = 28; break;
		case eCLKMUX_UART3_SUSBPHY0:
			uRegAddr = rCLK_SRC4; uRegData = 0x3; uRegBitOffset = 28; break;
		case eCLKMUX_UART3_SUSBPHY1:
			uRegAddr = rCLK_SRC4; uRegData = 0x4; uRegBitOffset = 28; break;
		case eCLKMUX_UART3_SHDMIPHY:
			uRegAddr = rCLK_SRC4; uRegData = 0x5; uRegBitOffset = 28; break;
		case eCLKMUX_UART3_SMPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x6; uRegBitOffset = 28; break;
		case eCLKMUX_UART3_SEPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x7; uRegBitOffset = 28; break;
		case eCLKMUX_UART3_SVPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x8; uRegBitOffset = 28; break;
		case eCLKMUX_UART2_XXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x0; uRegBitOffset = 24; break;
		case eCLKMUX_UART2_XUSBXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x1; uRegBitOffset = 24; break;
		case eCLKMUX_UART2_SHDMI27M:
			uRegAddr = rCLK_SRC4; uRegData = 0x2; uRegBitOffset = 24; break;
		case eCLKMUX_UART2_SUSBPHY0:
			uRegAddr = rCLK_SRC4; uRegData = 0x3; uRegBitOffset = 24; break;
		case eCLKMUX_UART2_SUSBPHY1:
			uRegAddr = rCLK_SRC4; uRegData = 0x4; uRegBitOffset = 24; break;
		case eCLKMUX_UART2_SHDMIPHY:
			uRegAddr = rCLK_SRC4; uRegData = 0x5; uRegBitOffset = 24; break;
		case eCLKMUX_UART2_SMPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x6; uRegBitOffset = 24; break;
		case eCLKMUX_UART2_SEPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x7; uRegBitOffset = 24; break;
		case eCLKMUX_UART2_SVPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x8; uRegBitOffset = 24; break;	
		case eCLKMUX_UART1_XXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x0; uRegBitOffset = 20; break;
		case eCLKMUX_UART1_XUSBXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x1; uRegBitOffset = 20; break;
		case eCLKMUX_UART1_SHDMI27M:
			uRegAddr = rCLK_SRC4; uRegData = 0x2; uRegBitOffset = 20; break;
		case eCLKMUX_UART1_SUSBPHY0:
			uRegAddr = rCLK_SRC4; uRegData = 0x3; uRegBitOffset = 20; break;
		case eCLKMUX_UART1_SUSBPHY1:
			uRegAddr = rCLK_SRC4; uRegData = 0x4; uRegBitOffset = 20; break;
		case eCLKMUX_UART1_SHDMIPHY:
			uRegAddr = rCLK_SRC4; uRegData = 0x5; uRegBitOffset = 20; break;
		case eCLKMUX_UART1_SMPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x6; uRegBitOffset = 20; break;
		case eCLKMUX_UART1_SEPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x7; uRegBitOffset = 20; break;
		case eCLKMUX_UART1_SVPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x8; uRegBitOffset = 20; break;
		case eCLKMUX_UART0_XXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x0; uRegBitOffset = 16; break;
		case eCLKMUX_UART0_XUSBXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x1; uRegBitOffset = 16; break;
		case eCLKMUX_UART0_SHDMI27M:
			uRegAddr = rCLK_SRC4; uRegData = 0x2; uRegBitOffset = 16; break;
		case eCLKMUX_UART0_SUSBPHY0:
			uRegAddr = rCLK_SRC4; uRegData = 0x3; uRegBitOffset = 16; break;
		case eCLKMUX_UART0_SUSBPHY1:
			uRegAddr = rCLK_SRC4; uRegData = 0x4; uRegBitOffset = 16; break;
		case eCLKMUX_UART0_SHDMIPHY:
			uRegAddr = rCLK_SRC4; uRegData = 0x5; uRegBitOffset = 16; break;
		case eCLKMUX_UART0_SMPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x6; uRegBitOffset = 16; break;
		case eCLKMUX_UART0_SEPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x7; uRegBitOffset = 16; break;
		case eCLKMUX_UART0_SVPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x8; uRegBitOffset = 16; break;
		case eCLKMUX_MMC3_XXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x0; uRegBitOffset = 12; break;
		case eCLKMUX_MMC3_XUSBXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x1; uRegBitOffset = 12; break;
		case eCLKMUX_MMC3_SHDMI27M:
			uRegAddr = rCLK_SRC4; uRegData = 0x2; uRegBitOffset = 12; break;
		case eCLKMUX_MMC3_SUSBPHY0:
			uRegAddr = rCLK_SRC4; uRegData = 0x3; uRegBitOffset = 12; break;
		case eCLKMUX_MMC3_SUSBPHY1:
			uRegAddr = rCLK_SRC4; uRegData = 0x4; uRegBitOffset = 12; break;
		case eCLKMUX_MMC3_SHDMIPHY:
			uRegAddr = rCLK_SRC4; uRegData = 0x5; uRegBitOffset = 12; break;
		case eCLKMUX_MMC3_SMPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x6; uRegBitOffset = 12; break;
		case eCLKMUX_MMC3_SEPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x7; uRegBitOffset = 12; break;
		case eCLKMUX_MMC3_SVPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x8; uRegBitOffset = 12; break;
		case eCLKMUX_MMC2_XXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x0; uRegBitOffset = 8; break;
		case eCLKMUX_MMC2_XUSBXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x1; uRegBitOffset = 8; break;
		case eCLKMUX_MMC2_SHDMI27M:
			uRegAddr = rCLK_SRC4; uRegData = 0x2; uRegBitOffset = 8; break;
		case eCLKMUX_MMC2_SUSBPHY0:
			uRegAddr = rCLK_SRC4; uRegData = 0x3; uRegBitOffset = 8; break;
		case eCLKMUX_MMC2_SUSBPHY1:
			uRegAddr = rCLK_SRC4; uRegData = 0x4; uRegBitOffset = 8; break;
		case eCLKMUX_MMC2_SHDMIPHY:
			uRegAddr = rCLK_SRC4; uRegData = 0x5; uRegBitOffset = 8; break;
		case eCLKMUX_MMC2_SMPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x6; uRegBitOffset = 8; break;
		case eCLKMUX_MMC2_SEPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x7; uRegBitOffset = 8; break;
		case eCLKMUX_MMC2_SVPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x8; uRegBitOffset = 8; break;
		case eCLKMUX_MMC1_XXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x0; uRegBitOffset = 4; break;
		case eCLKMUX_MMC1_XUSBXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x1; uRegBitOffset = 4; break;
		case eCLKMUX_MMC1_SHDMI27M:
			uRegAddr = rCLK_SRC4; uRegData = 0x2; uRegBitOffset = 4; break;
		case eCLKMUX_MMC1_SUSBPHY0:
			uRegAddr = rCLK_SRC4; uRegData = 0x3; uRegBitOffset = 4; break;
		case eCLKMUX_MMC1_SUSBPHY1:
			uRegAddr = rCLK_SRC4; uRegData = 0x4; uRegBitOffset = 4; break;
		case eCLKMUX_MMC1_SHDMIPHY:
			uRegAddr = rCLK_SRC4; uRegData = 0x5; uRegBitOffset = 4; break;
		case eCLKMUX_MMC1_SMPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x6; uRegBitOffset = 4; break;
		case eCLKMUX_MMC1_SEPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x7; uRegBitOffset = 4; break;
		case eCLKMUX_MMC1_SVPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x8; uRegBitOffset = 4; break;
		case eCLKMUX_MMC0_XXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x0; uRegBitOffset = 0; break;
		case eCLKMUX_MMC0_XUSBXTI:
			uRegAddr = rCLK_SRC4; uRegData = 0x1; uRegBitOffset = 0; break;
		case eCLKMUX_MMC0_SHDMI27M:
			uRegAddr = rCLK_SRC4; uRegData = 0x2; uRegBitOffset = 0; break;
		case eCLKMUX_MMC0_SUSBPHY0:
			uRegAddr = rCLK_SRC4; uRegData = 0x3; uRegBitOffset = 0; break;
		case eCLKMUX_MMC0_SUSBPHY1:
			uRegAddr = rCLK_SRC4; uRegData = 0x4; uRegBitOffset = 0; break;
		case eCLKMUX_MMC0_SHDMIPHY:
			uRegAddr = rCLK_SRC4; uRegData = 0x5; uRegBitOffset = 0; break;
		case eCLKMUX_MMC0_SMPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x6; uRegBitOffset = 0; break;
		case eCLKMUX_MMC0_SEPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x7; uRegBitOffset = 0; break;
		case eCLKMUX_MMC0_SVPLL:
			uRegAddr = rCLK_SRC4; uRegData = 0x8; uRegBitOffset = 0; break;

		// CLK_SRC5
		case eCLKMUX_PWM_XXTI:
			uRegAddr = rCLK_SRC5; uRegData = 0x0; uRegBitOffset = 12; break;
		case eCLKMUX_PWM_XUSBXTI:
			uRegAddr = rCLK_SRC5; uRegData = 0x1; uRegBitOffset = 12; break;
		case eCLKMUX_PWM_SHDMI27M:
			uRegAddr = rCLK_SRC5; uRegData = 0x2; uRegBitOffset = 12; break;
		case eCLKMUX_PWM_SUSBPHY0:
			uRegAddr = rCLK_SRC5; uRegData = 0x3; uRegBitOffset = 12; break;
		case eCLKMUX_PWM_SUSBPHY1:
			uRegAddr = rCLK_SRC5; uRegData = 0x4; uRegBitOffset = 12; break;
		case eCLKMUX_PWM_SHDMIPHY:
			uRegAddr = rCLK_SRC5; uRegData = 0x5; uRegBitOffset = 12; break;
		case eCLKMUX_PWM_SMPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x6; uRegBitOffset = 12; break;
		case eCLKMUX_PWM_SEPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x7; uRegBitOffset = 12; break;
		case eCLKMUX_PWM_SVPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x8; uRegBitOffset = 12; break;
		case eCLKMUX_SPI2_XXTI:
			uRegAddr = rCLK_SRC5; uRegData = 0x0; uRegBitOffset = 8; break;
		case eCLKMUX_SPI2_XUSBXTI:
			uRegAddr = rCLK_SRC5; uRegData = 0x1; uRegBitOffset = 8; break;
		case eCLKMUX_SPI2_SHDMI27M:
			uRegAddr = rCLK_SRC5; uRegData = 0x2; uRegBitOffset = 8; break;
		case eCLKMUX_SPI2_SUSBPHY0:
			uRegAddr = rCLK_SRC5; uRegData = 0x3; uRegBitOffset = 8; break;
		case eCLKMUX_SPI2_SUSBPHY1:
			uRegAddr = rCLK_SRC5; uRegData = 0x4; uRegBitOffset = 8; break;
		case eCLKMUX_SPI2_SHDMIPHY:
			uRegAddr = rCLK_SRC5; uRegData = 0x5; uRegBitOffset = 8; break;
		case eCLKMUX_SPI2_SMPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x6; uRegBitOffset = 8; break;
		case eCLKMUX_SPI2_SEPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x7; uRegBitOffset = 8; break;
		case eCLKMUX_SPI2_SVPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x8; uRegBitOffset = 8; break;
		case eCLKMUX_SPI1_XXTI:
			uRegAddr = rCLK_SRC5; uRegData = 0x0; uRegBitOffset = 4; break;
		case eCLKMUX_SPI1_XUSBXTI:
			uRegAddr = rCLK_SRC5; uRegData = 0x1; uRegBitOffset = 4; break;
		case eCLKMUX_SPI1_SHDMI27M:
			uRegAddr = rCLK_SRC5; uRegData = 0x2; uRegBitOffset = 4; break;
		case eCLKMUX_SPI1_SUSBPHY0:
			uRegAddr = rCLK_SRC5; uRegData = 0x3; uRegBitOffset = 4; break;
		case eCLKMUX_SPI1_SUSBPHY1:
			uRegAddr = rCLK_SRC5; uRegData = 0x4; uRegBitOffset = 4; break;
		case eCLKMUX_SPI1_SHDMIPHY:
			uRegAddr = rCLK_SRC5; uRegData = 0x5; uRegBitOffset = 4; break;
		case eCLKMUX_SPI1_SMPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x6; uRegBitOffset = 4; break;
		case eCLKMUX_SPI1_SEPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x7; uRegBitOffset = 4; break;
		case eCLKMUX_SPI1_SVPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x8; uRegBitOffset = 4; break;
		case eCLKMUX_SPI0_XXTI:
			uRegAddr = rCLK_SRC5; uRegData = 0x0; uRegBitOffset = 0; break;
		case eCLKMUX_SPI0_XUSBXTI:
			uRegAddr = rCLK_SRC5; uRegData = 0x1; uRegBitOffset = 0; break;
		case eCLKMUX_SPI0_SHDMI27M:
			uRegAddr = rCLK_SRC5; uRegData = 0x2; uRegBitOffset = 0; break;
		case eCLKMUX_SPI0_SUSBPHY0:
			uRegAddr = rCLK_SRC5; uRegData = 0x3; uRegBitOffset = 0; break;
		case eCLKMUX_SPI0_SUSBPHY1:
			uRegAddr = rCLK_SRC5; uRegData = 0x4; uRegBitOffset = 0; break;
		case eCLKMUX_SPI0_SHDMIPHY:
			uRegAddr = rCLK_SRC5; uRegData = 0x5; uRegBitOffset = 0; break;
		case eCLKMUX_SPI0_SMPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x6; uRegBitOffset = 0; break;
		case eCLKMUX_SPI0_SEPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x7; uRegBitOffset = 0; break;
		case eCLKMUX_SPI0_SVPLL:
			uRegAddr = rCLK_SRC5; uRegData = 0x8; uRegBitOffset = 0; break;

		// CLK_SRC6
		case eCLKMUX_ONEDRAM_SCLKA2M:
			uRegAddr = rCLK_SRC6; uRegData = 0x0; uRegBitOffset = 24; break;
		case eCLKMUX_ONEDRAM_SCLKMPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x1; uRegBitOffset = 24; break;
		case eCLKMUX_ONEDRAM_SCLKEPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x2; uRegBitOffset = 24; break;
		case eCLKMUX_ONEDRAM_SCLKVPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x3; uRegBitOffset = 24; break;
		case eCLKMUX_PWI_XUSBXTI:
			uRegAddr = rCLK_SRC6; uRegData = 0x1; uRegBitOffset = 20; break;
		case eCLKMUX_PWI_SHDMI27M:
			uRegAddr = rCLK_SRC6; uRegData = 0x2; uRegBitOffset = 20; break;
		case eCLKMUX_PWI_SUSBPHY0:
			uRegAddr = rCLK_SRC6; uRegData = 0x3; uRegBitOffset = 20; break;
		case eCLKMUX_PWI_SUSBPHY1:
			uRegAddr = rCLK_SRC6; uRegData = 0x4; uRegBitOffset = 20; break;
		case eCLKMUX_PWI_SHDMIPHY:
			uRegAddr = rCLK_SRC6; uRegData = 0x5; uRegBitOffset = 20; break;
		case eCLKMUX_PWI_SMPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x6; uRegBitOffset = 20; break;
		case eCLKMUX_PWI_SEPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x7; uRegBitOffset = 20; break;
		case eCLKMUX_PWI_SVPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x8; uRegBitOffset = 20; break;
		case eCLKMUX_HPM_SCLKAPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x0; uRegBitOffset = 16; break;
		case eCLKMUX_HPM_SCLKMPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x1; uRegBitOffset = 16; break;
		case eCLKMUX_SPDIF_SAUDIO0:
			uRegAddr = rCLK_SRC6; uRegData = 0x0; uRegBitOffset = 12; break;
		case eCLKMUX_SPDIF_SAUDIO1:
			uRegAddr = rCLK_SRC6; uRegData = 0x1; uRegBitOffset = 12; break;
		case eCLKMUX_SPDIF_SAUDIO2:
			uRegAddr = rCLK_SRC6; uRegData = 0x2; uRegBitOffset = 12; break;
		case eCLKMUX_AUDIO2_XXTI:
			uRegAddr = rCLK_SRC6; uRegData = 0x0; uRegBitOffset = 8; break;
		case eCLKMUX_AUDIO2_PCMCDCLK0:
			uRegAddr = rCLK_SRC6; uRegData = 0x1; uRegBitOffset = 8; break;
		case eCLKMUX_AUDIO2_SHDMI27M:
			uRegAddr = rCLK_SRC6; uRegData = 0x2; uRegBitOffset = 8; break;
		case eCLKMUX_AUDIO2_SUSBPHY0:
			uRegAddr = rCLK_SRC6; uRegData = 0x3; uRegBitOffset = 8; break;
		case eCLKMUX_AUDIO2_SUSBPHY1:
			uRegAddr = rCLK_SRC6; uRegData = 0x4; uRegBitOffset = 8; break;
		case eCLKMUX_AUDIO2_SHDMIPHY:
			uRegAddr = rCLK_SRC6; uRegData = 0x5; uRegBitOffset = 8; break;
		case eCLKMUX_AUDIO2_SMPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x6; uRegBitOffset = 8; break;
		case eCLKMUX_AUDIO2_SEPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x7; uRegBitOffset = 8; break;
		case eCLKMUX_AUDIO2_SVPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x8; uRegBitOffset = 8; break;
		case eCLKMUX_AUDIO1_I2SCDCLK1:
			uRegAddr = rCLK_SRC6; uRegData = 0x0; uRegBitOffset = 4; break;
		case eCLKMUX_AUDIO1_PCMCDCLK1:
			uRegAddr = rCLK_SRC6; uRegData = 0x1; uRegBitOffset = 4; break;
		case eCLKMUX_AUDIO1_SHDMI27M:
			uRegAddr = rCLK_SRC6; uRegData = 0x2; uRegBitOffset = 4; break;
		case eCLKMUX_AUDIO1_SUSBPHY0:
			uRegAddr = rCLK_SRC6; uRegData = 0x3; uRegBitOffset = 4; break;
		case eCLKMUX_AUDIO1_SUSBPHY1:
			uRegAddr = rCLK_SRC6; uRegData = 0x4; uRegBitOffset = 4; break;
		case eCLKMUX_AUDIO1_SHDMIPHY:
			uRegAddr = rCLK_SRC6; uRegData = 0x5; uRegBitOffset = 4; break;
		case eCLKMUX_AUDIO1_SMPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x6; uRegBitOffset = 4; break;
		case eCLKMUX_AUDIO1_SEPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x7; uRegBitOffset = 4; break;
		case eCLKMUX_AUDIO1_SVPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x8; uRegBitOffset = 4; break;
		case eCLKMUX_AUDIO0_XXTI:
			uRegAddr = rCLK_SRC6; uRegData = 0x0; uRegBitOffset = 0; break;
		case eCLKMUX_AUDIO0_XUSBXTI:
			uRegAddr = rCLK_SRC6; uRegData = 0x1; uRegBitOffset = 0; break;
		case eCLKMUX_AUDIO0_SHDMI27M:
			uRegAddr = rCLK_SRC6; uRegData = 0x2; uRegBitOffset = 0; break;
		case eCLKMUX_AUDIO0_SUSBPHY0:
			uRegAddr = rCLK_SRC6; uRegData = 0x3; uRegBitOffset = 0; break;
		case eCLKMUX_AUDIO0_SUSBPHY1:
			uRegAddr = rCLK_SRC6; uRegData = 0x4; uRegBitOffset = 0; break;
		case eCLKMUX_AUDIO0_SHDMIPHY:
			uRegAddr = rCLK_SRC6; uRegData = 0x5; uRegBitOffset = 0; break;
		case eCLKMUX_AUDIO0_SMPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x6; uRegBitOffset = 0; break;
		case eCLKMUX_AUDIO0_SEPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x7; uRegBitOffset = 0; break;
		case eCLKMUX_AUDIO0_SVPLL:
			uRegAddr = rCLK_SRC6; uRegData = 0x8; uRegBitOffset = 0; break;

		default:
			DbgSysc(("No appropriate source clock is found in SetClkMuxState()\n\x04"));
			while(1);
	}

	uRegTemp = syscInp32(uRegAddr);
	uRegTemp = uRegTemp &~(0xf<<uRegBitOffset) | (uRegData<<uRegBitOffset);
	syscOutp32(uRegAddr, uRegTemp);
}

u32 SYSC_GetClkMuxReg(u32 uRegNum)
{
	u32 uRegTemp;
	Assert(uRegNum<=6);

	uRegTemp = syscInp32(rCLK_SRC0+4*uRegNum);
	return uRegTemp;
}

u32 SYSC_GetClkMuxStatusReg(u32 uRegNum)
{
	u32 uRegTemp;
	Assert(uRegNum<=6);

	uRegTemp = syscInp32(rCLK_MUX_STAT0+4*uRegNum);
	return uRegTemp;
}

// Set clock divider
void SYSC_SetClkDiv(CLKDIV_ID eId, u32 uRatio)
{
	u32 uRegAddr;
	u32 uRegMaskData;
	u32 uRegBitOffset;
	u32 uRegTemp;
	u32 uDivValue;
	u32 uStatRegAddr;
	u32 uStatRegBitOffset;
	u32 uStatRegTemp;

	switch(eId)
	{
		case eCLKDIV_PCLKPSYS: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 7;
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 28; break;
		case eCLKDIV_HCLKPSYS: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 6;
			uRegAddr = rCLK_DIV0; uRegMaskData = 0xf; uRegBitOffset = 24; break;
		case eCLKDIV_PCLKDSYS: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 5;
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 20; break;
		case eCLKDIV_HCLKDSYS: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 4;
			uRegAddr = rCLK_DIV0; uRegMaskData = 0xf; uRegBitOffset = 16; break;
		case eCLKDIV_PCLKMSYS: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 3;
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 12; break;
		case eCLKDIV_HCLKMSYS: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 2;
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 8; break;
		case eCLKDIV_A2M: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 1;
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 4; break;
		case eCLKDIV_APLL: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 0;
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 0; break;

		case eCLKDIV_CSIS: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 15;
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 28; break;
		case eCLKDIV_FIMD: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 13;
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 20; break;
		case eCLKDIV_CAM1:
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 12;
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 16; break;
		case eCLKDIV_CAM0:
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 11;
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 12; break;
		case eCLKDIV_FIMC: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 10;
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 8; break;
		case eCLKDIV_TBLK: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 8;
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_MFC: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 17;
			uRegAddr = rCLK_DIV2; uRegMaskData = 0xf; uRegBitOffset = 4; break;
		case eCLKDIV_G3D: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 16;
			uRegAddr = rCLK_DIV2; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_FIMC2: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 22;
			uRegAddr = rCLK_DIV3; uRegMaskData = 0xf; uRegBitOffset = 20; break;
		case eCLKDIV_FIMC1: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 21;
			uRegAddr = rCLK_DIV3; uRegMaskData = 0xf; uRegBitOffset = 16; break;
		case eCLKDIV_FIMC0: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 20;
			uRegAddr = rCLK_DIV3; uRegMaskData = 0xf; uRegBitOffset = 12; break;
		case eCLKDIV_MDNIEPWM:
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 19;
			uRegAddr = rCLK_DIV3; uRegMaskData = 0x7f; uRegBitOffset = 4; break;
		case eCLKDIV_MDNIE: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 18;
			uRegAddr = rCLK_DIV3; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_UART3: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 31;
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 28; break;
		case eCLKDIV_UART2: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 30;
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 24; break;
		case eCLKDIV_UART1: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 29;
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 20; break;
		case eCLKDIV_UART0: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 28;
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 16; break;
		case eCLKDIV_MMC3: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 27;
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 12; break;
		case eCLKDIV_MMC2: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 26;
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 8; break;
		case eCLKDIV_MMC1: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 25;
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 4; break;
		case eCLKDIV_MMC0: 
			uStatRegAddr = rCLK_DIV_STAT0; uStatRegBitOffset = 24;
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_PWM: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 3;
			uRegAddr = rCLK_DIV5; uRegMaskData = 0xf; uRegBitOffset = 12; break;
		case eCLKDIV_SPI2: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 2;
			uRegAddr = rCLK_DIV5; uRegMaskData = 0xf; uRegBitOffset = 8; break;
		case eCLKDIV_SPI1: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 1;
			uRegAddr = rCLK_DIV5; uRegMaskData = 0xf; uRegBitOffset = 4; break;
		case eCLKDIV_SPI0: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 0;
			uRegAddr = rCLK_DIV5; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_ONEDRAM: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 15;
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 28; break;
		case eCLKDIV_PWI: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 14;
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 24; break;
		case eCLKDIV_HPM: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 13;
			uRegAddr = rCLK_DIV6; uRegMaskData = 0x7; uRegBitOffset = 20; break;
		case eCLKDIV_COPY:
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 12;
			uRegAddr = rCLK_DIV6; uRegMaskData = 0x7; uRegBitOffset = 16; break;
		case eCLKDIV_ONENAND: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 11;
			uRegAddr = rCLK_DIV6; uRegMaskData = 0x7; uRegBitOffset = 12; break;
		case eCLKDIV_AUDIO2: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 10;
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 8; break;
		case eCLKDIV_AUDIO1: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 9;
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 4; break;
		case eCLKDIV_AUDIO0: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 8;
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_DPM: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 17;
			uRegAddr = rCLK_DIV7; uRegMaskData = 0x7f; uRegBitOffset = 8; break;
		case eCLKDIV_DVSEM: 
			uStatRegAddr = rCLK_DIV_STAT1; uStatRegBitOffset = 16;
			uRegAddr = rCLK_DIV7; uRegMaskData = 0x7f; uRegBitOffset = 0; break;
		default:
			DbgSysc(("No appropriate clock divider is found in SetClockDivider()\n\x04"));
			return;
	}

	// CLKOUT = CLKIN / RATIO (RATIO = uRatio + 1)
	uRegTemp = syscInp32(uRegAddr);
	uDivValue = uRatio-1;
	uRegTemp = uRegTemp &~(uRegMaskData<<uRegBitOffset) | (uDivValue<<uRegBitOffset);
	syscOutp32(uRegAddr, uRegTemp); // set divider

//	while ( syscInp32(uStatRegAddr)&(1<<uStatRegBitOffset) ); // 1 is on changing. 0 is stable
	DbgSysc(("Divider changed!\n"));

}

void SYSC_SetClkDivReg(u32 uRegNum, u32 uRegVal)
{
	Assert(uRegNum<=7);

	Outp32(rCLK_DIV0+4*uRegNum,uRegVal);
}

// Get Clock divider
u32 SYSC_GetClkDiv(CLKDIV_ID eId)
{
	u32 uRegAddr;
	u32 uRegMaskData;
	u32 uRegBitOffset;
	u32 uRegTemp;

	switch(eId)
	{
		case eCLKDIV_PCLKPSYS: 
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 28; break;
		case eCLKDIV_HCLKPSYS: 
			uRegAddr = rCLK_DIV0; uRegMaskData = 0xf; uRegBitOffset = 24; break;
		case eCLKDIV_PCLKDSYS: 
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 20; break;
		case eCLKDIV_HCLKDSYS: 
			uRegAddr = rCLK_DIV0; uRegMaskData = 0xf; uRegBitOffset = 16; break;
		case eCLKDIV_PCLKMSYS: 
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 12; break;
		case eCLKDIV_HCLKMSYS: 
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 8; break;
		case eCLKDIV_A2M: 
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 4; break;
		case eCLKDIV_APLL: 
			uRegAddr = rCLK_DIV0; uRegMaskData = 0x7; uRegBitOffset = 0; break;

		case eCLKDIV_CSIS: 
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 28; break;
		case eCLKDIV_FIMD: 
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 20; break;
		case eCLKDIV_CAM1:
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 16; break;
		case eCLKDIV_CAM0:
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 12; break;
		case eCLKDIV_FIMC: 
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 8; break;
		case eCLKDIV_TBLK: 
			uRegAddr = rCLK_DIV1; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_MFC: 
			uRegAddr = rCLK_DIV2; uRegMaskData = 0xf; uRegBitOffset = 4; break;
		case eCLKDIV_G3D: 
			uRegAddr = rCLK_DIV2; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_FIMC2: 
			uRegAddr = rCLK_DIV3; uRegMaskData = 0xf; uRegBitOffset = 20; break;
		case eCLKDIV_FIMC1: 
			uRegAddr = rCLK_DIV3; uRegMaskData = 0xf; uRegBitOffset = 16; break;
		case eCLKDIV_FIMC0: 
			uRegAddr = rCLK_DIV3; uRegMaskData = 0xf; uRegBitOffset = 12; break;
		case eCLKDIV_MDNIEPWM:
			uRegAddr = rCLK_DIV3; uRegMaskData = 0x7f; uRegBitOffset = 4; break;
		case eCLKDIV_MDNIE: 
			uRegAddr = rCLK_DIV3; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_UART3: 
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 28; break;
		case eCLKDIV_UART2: 
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 24; break;
		case eCLKDIV_UART1: 
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 20; break;
		case eCLKDIV_UART0: 
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 16; break;
		case eCLKDIV_MMC3: 
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 12; break;
		case eCLKDIV_MMC2: 
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 8; break;
		case eCLKDIV_MMC1: 
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 4; break;
		case eCLKDIV_MMC0: 
			uRegAddr = rCLK_DIV4; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_PWM: 
			uRegAddr = rCLK_DIV5; uRegMaskData = 0xf; uRegBitOffset = 12; break;
		case eCLKDIV_SPI2: 
			uRegAddr = rCLK_DIV5; uRegMaskData = 0xf; uRegBitOffset = 8; break;
		case eCLKDIV_SPI1: 
			uRegAddr = rCLK_DIV5; uRegMaskData = 0xf; uRegBitOffset = 4; break;
		case eCLKDIV_SPI0: 
			uRegAddr = rCLK_DIV5; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_ONEDRAM: 
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 28; break;
		case eCLKDIV_PWI: 
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 24; break;
		case eCLKDIV_HPM: 
			uRegAddr = rCLK_DIV6; uRegMaskData = 0x7; uRegBitOffset = 20; break;
		case eCLKDIV_COPY:
			uRegAddr = rCLK_DIV6; uRegMaskData = 0x7; uRegBitOffset = 16; break;
		case eCLKDIV_ONENAND: 
			uRegAddr = rCLK_DIV6; uRegMaskData = 0x7; uRegBitOffset = 12; break;
		case eCLKDIV_AUDIO2: 
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 8; break;
		case eCLKDIV_AUDIO1: 
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 4; break;
		case eCLKDIV_AUDIO0: 
			uRegAddr = rCLK_DIV6; uRegMaskData = 0xf; uRegBitOffset = 0; break;

		case eCLKDIV_DPM: 
			uRegAddr = rCLK_DIV7; uRegMaskData = 0x7f; uRegBitOffset = 8; break;
		case eCLKDIV_DVSEM: 
			uRegAddr = rCLK_DIV7; uRegMaskData = 0x7f; uRegBitOffset = 0; break;
		default:
			DbgSysc(("No appropriate clock divider is found in GetClockDivider()\n\x04"));
			return;
	}

	// CLKOUT = CLKIN / RATIO (RATIO = uRatio + 1)
	uRegTemp = syscInp32(uRegAddr);
	return ((uRegTemp>>uRegBitOffset)&uRegMaskData)+1;  // return real divide value. ex) 1/4 -> return 4
}

u32 SYSC_GetClkDivReg(u32 uRegNum)
{
	u32 uRegTemp;
	Assert(uRegNum<=7);

	uRegTemp = syscInp32(rCLK_DIV0+4*uRegNum);
	return uRegTemp;
}

u32 SYSC_GetClkDivStatusReg(u32 uRegNum)
{
	u32 uRegTemp;
	Assert(uRegNum<=7);

	uRegTemp = syscInp32(rCLK_DIV_STAT0+4*uRegNum);
	return uRegTemp;
}

// Set clock gate by IP
void SYSC_SetClkGatebyIP(CLKGATE_IP_TYPE eType, bool bPass) // added by junon
{
	u32 uRegAddr;
	u32 uRegGateOn;
	u32 uRegBitOffset;
	u32 uRegTemp;

	uRegGateOn = (bPass == TRUE) ? 1 : 0;

	switch(eType)
	{
		// CLK_GATE_IP0	
		case eCLKGATE_IP_CSIS:
		case eCLKGATE_IP_IPC:
		case eCLKGATE_IP_ROTATOR:
		case eCLKGATE_IP_JPEG:
		case eCLKGATE_IP_FIMC2: 
		case eCLKGATE_IP_FIMC1: 
		case eCLKGATE_IP_FIMC0: 
		case eCLKGATE_IP_MFC: 
		case eCLKGATE_IP_G3D: 
		case eCLKGATE_IP_IMEM:
		case eCLKGATE_IP_PDMA1:
		case eCLKGATE_IP_PDMA0:
		case eCLKGATE_IP_MDMA:
		case eCLKGATE_IP_DMC1:
		case eCLKGATE_IP_DMC0:
			uRegAddr = rCLK_GATE_IP0; 
			uRegBitOffset = eType;
			break;
		// CLK_GATE_IP1
		case eCLKGATE_IP_NFCON:
		case eCLKGATE_IP_SROMC:
		case eCLKGATE_IP_CFCON:
		case eCLKGATE_IP_NANDXL: 	
		case eCLKGATE_IP_USBHOST:
		case eCLKGATE_IP_USBOTG: 
		case eCLKGATE_IP_HDMI:
		case eCLKGATE_IP_TVENC:
		case eCLKGATE_IP_MIXER:
		case eCLKGATE_IP_VP:
		case eCLKGATE_IP_DSIM:
		case eCLKGATE_IP_MIE: 
		case eCLKGATE_IP_FIMD:
			uRegAddr = rCLK_GATE_IP1;
			uRegBitOffset = (u32)eType - 100;
			break;
		// CLK_GATE_IP2
		case eCLKGATE_IP_TZIC3:
		case eCLKGATE_IP_TZIC2:
		case eCLKGATE_IP_TZIC1:
		case eCLKGATE_IP_TZIC0:
		case eCLKGATE_IP_VIC3:
		case eCLKGATE_IP_VIC2:
		case eCLKGATE_IP_VIC1:
		case eCLKGATE_IP_VIC0:
		case eCLKGATE_IP_TSI:
		case eCLKGATE_IP_MMC3:
		case eCLKGATE_IP_MMC2:
		case eCLKGATE_IP_MMC1:
		case eCLKGATE_IP_MMC0:
		case eCLKGATE_IP_SECJTAG:
		case eCLKGATE_IP_HOSTIF:
		case eCLKGATE_IP_MODEM:
		case eCLKGATE_IP_CORESIGHT:
		case eCLKGATE_IP_SDM:
		case eCLKGATE_IP_SECSS:
			uRegAddr = rCLK_GATE_IP2;
			uRegBitOffset = (u32)eType - 200;
			break;
		// CLK_GATE_IP3
		case eCLKGATE_IP_PCM2:
		case eCLKGATE_IP_PCM1:
		case eCLKGATE_IP_PCM0:
		case eCLKGATE_IP_SYSCON:
		case eCLKGATE_IP_GPIO:
		case eCLKGATE_IP_TSADC:
		case eCLKGATE_IP_PWM:
		case eCLKGATE_IP_WDT:
		case eCLKGATE_IP_KEYIF:
		case eCLKGATE_IP_UART3:
		case eCLKGATE_IP_UART2:
		case eCLKGATE_IP_UART1:
		case eCLKGATE_IP_UART0:
		case eCLKGATE_IP_SYSTIMER:
		case eCLKGATE_IP_RTC:
		case eCLKGATE_IP_SPI2:
		case eCLKGATE_IP_SPI1:
		case eCLKGATE_IP_SPI0:
		case eCLKGATE_IP_I2CHDMIPHY:
		case eCLKGATE_IP_I2CHDMIDDC:
		case eCLKGATE_IP_I2C2:
		case eCLKGATE_IP_I2C1:
		case eCLKGATE_IP_I2C0:
		case eCLKGATE_IP_I2S2:
		case eCLKGATE_IP_I2S1:
		case eCLKGATE_IP_I2S0:
		case eCLKGATE_IP_AC94:
		case eCLKGATE_IP_SPDIF:
			uRegAddr = rCLK_GATE_IP3;
			uRegBitOffset = (u32)eType - 300;
			break;
		// CLK_GATE_IP4
		case eCLKGATE_IP_TZPC3:
		case eCLKGATE_IP_TZPC2:
		case eCLKGATE_IP_TZPC1:
		case eCLKGATE_IP_TZPC0:
		case eCLKGATE_IP_SECKEY:
		case eCLKGATE_IP_IEM_APC:
		case eCLKGATE_IP_IEM_IEC:
		case eCLKGATE_IP_CHIP_ID:
			uRegAddr = rCLK_GATE_IP4;
			uRegBitOffset = (u32)eType - 400;
			break;
		// ALL
		case eCLKGATE_IP_ALL:
			if (uRegGateOn)
			{
				syscOutp32(rCLK_GATE_IP0, 0xFFFFFFFF);
				syscOutp32(rCLK_GATE_IP1, 0xFFFFFFFF);
				syscOutp32(rCLK_GATE_IP2, 0xFFFFFFFF);
				syscOutp32(rCLK_GATE_IP3, 0xFFFFFFFF);
				syscOutp32(rCLK_GATE_IP4, 0xFFFFFFFF);
			}
			else
			{
				syscOutp32(rCLK_GATE_IP0, 0);
				syscOutp32(rCLK_GATE_IP1, 0);
				syscOutp32(rCLK_GATE_IP2, 0);
				syscOutp32(rCLK_GATE_IP3, 0);
				syscOutp32(rCLK_GATE_IP4, 0);
			}
			return;
		default:
			DbgSysc(("No appropriate clock divider is found in SetClockGatebyIP()\n\x04"));
			return;
	}

	uRegTemp = syscInp32(uRegAddr);
	uRegTemp = uRegTemp &~(0x1<<uRegBitOffset) | (uRegGateOn<<uRegBitOffset);
	syscOutp32(uRegAddr, uRegTemp);
}

// Set clock gate by block
void SYSC_SetClkGatebyBlock(CLKGATE_BLOCK_TYPE eType, bool bPass)  // added by junon
{
	u32 uRegGateOn;
	u32 uRegBitOffset;
	u32 uRegTemp;

	uRegGateOn = (bPass == true) ? 1 : 0;
	uRegBitOffset = (u32)eType;

	if (eType == eCLKGATE_BLOCK_ALL)
	{
		if (uRegGateOn)
			syscOutp32(rCLK_GATE_BLOCK, 0xFFFFFFFF);
		else
			syscOutp32(rCLK_GATE_BLOCK, 0x0);			
		return;
	}

	if (eType > 11)
	{
		DbgSysc(("No appropriate clock divider is found in SetClockGatebyBlock()\n\x04"));
		return;
	}

	uRegTemp = syscInp32(rCLK_GATE_BLOCK);
	uRegTemp = uRegTemp &~(0x1<<uRegBitOffset) | (uRegGateOn<<uRegBitOffset);
	syscOutp32(rCLK_GATE_BLOCK, uRegTemp);
}

// Get rCLK_GATE_IP0~4 registers respectively
// uIPRegNum = 0~4
u32 SYSC_GetClkGateIPReg(u32 uIPRegNum)
{
	u32 uRegTemp;
	Assert(uIPRegNum<=4);

	uRegTemp = syscInp32(rCLK_GATE_IP0+4*uIPRegNum);
	return uRegTemp;
}

// Get rCLK_GATE_BLOCK register
u32 SYSC_GetClkGateBlockReg(void)
{
	u32 uRegTemp;

	uRegTemp = syscInp32(rCLK_GATE_BLOCK);
	return uRegTemp;
}


// Set Clock output
void SYSC_SetClkOut(CLKOUT_TYPE eType, u32 uDivVal)
{
	u32 uRegTemp[2];

	Assert(uDivVal <= 16);

	if ( (eType == eCLKOUT_XXTI_PD)||(eType == eCLKOUT_XUSBXTI_PD) )
	{
		// control CLKOUT mux using in power down mode
		uRegTemp[0] = syscInp32(rOTHERS);
		uRegTemp[0] &= ~(0x3<<8); // Clock out signal from SYSCON(rCLKOUT)
		uRegTemp[0] |= ((eType-100)<<8); 
		syscOutp32(rOTHERS, uRegTemp[0]);
	}
	else
	{
		// control CLKOUT mux using in Normal mode
		uRegTemp[0] = syscInp32(rOTHERS);
		uRegTemp[0] &= ~(0x3<<8); // Clock out signal from SYSCON(rCLKOUT)
		syscOutp32(rOTHERS, uRegTemp[0]); 

		// Select clock source to CLKOUT
		uDivVal = uDivVal - 1;
		uRegTemp[1] = syscInp32(rCLK_OUT);
		uRegTemp[1] &= ~((0x1f<<12)|(0xf<<20));
		uRegTemp[1] |= (eType<<12)|(uDivVal<<20);
		syscOutp32(rCLK_OUT, uRegTemp[1]);
	}
}

// Set Dclk output
// uDclkCmp = Low level duration
// High level duration = uDclkDiv - uDclkCmp
// DclkOut = uDclkSrcClk / uDclkDiv
// uDclkSrcClk = DCLK source clock (0:XXTI, 1:XusbXTI, 2:HDMI27M, 3:USBPHY0, 4:USBPHY1, 5:HDMIPHY, 6:MPLL/2, 7:EPLL)
// ex) uDclkCmp = 5, uDclkDiv = 10, uDclkSrcClk = 1(24MHz)   ===> 50:50 duty 2.4MHz
void SYSC_SetDclkOut(u32 uDclkCmp, u32 uDclkDiv, u32 uDclkSrcClk)
{
	u32 uRegTemp;

	uDclkCmp = uDclkCmp - 1;
	uDclkDiv = uDclkDiv - 1;
	
	uRegTemp = syscInp32(rCLK_OUT);
	uRegTemp = uRegTemp &~(0x7ff<<1) | (uDclkCmp<<8) | (uDclkDiv<<4) | (uDclkSrcClk<<1);
	syscOutp32(rCLK_OUT, uRegTemp);
}

// [Set Dclk on/off]
void SYSC_SetDclkOnOff(bool bEnable)
{
	u32 uRegTemp;
	u32 uEnable = (bEnable == true) ? 1 : 0;

	uRegTemp = syscInp32(rCLK_OUT);
	uRegTemp = uRegTemp &~(1<<0) | (uEnable<<0);
	syscOutp32(rCLK_OUT, uRegTemp);
}

//===================================
// MISC in CMU
//===================================

// Software Reset IP
void SYSC_SwReset(void)
{
	syscOutp32(rGLOBAL_SW_RESET, 0x1);
}



// Set USB Control
void SYSC_SetUSBPHYControl(bool bEnable)
{
	u32 uRegTemp;
	u32 uEnable = (bEnable == true) ? 1 : 0;

	uRegTemp = syscInp32(rUSB_CON);
	uRegTemp = uRegTemp &~(3<<0) | (uEnable<<1)| (uEnable<<0);
	syscOutp32(rUSB_CON, uRegTemp);
}



//////////////////////////////////////////////
// Power Management Unit (PMU - ALIVE region)
//////////////////////////////////////////////


u32 SYSC_GetOperatingMode(void)
{
	u32 uRegData;
	u32 uOperatingMode;

	uRegData = syscInp32(rOM_STAT);

	uOperatingMode = uRegData&0x3f;
	return uOperatingMode;
}

u32 SYSC_GetResetStatusReg(void) 
{
	u32 uRegTemp;
	uRegTemp = syscInp32(rRST_STAT);
	return uRegTemp;
}


// [Set configuration of standby wait for interrupt]
// Determines what action is taken when the STANDBYWFI signal is activated by Cortex-A8
// 0: ignore, 1: Enter IDLE mode, 2: Enter STOP mode, 3: Enter SLEEP mode
void SYSC_SetConfigStandbyWFI(POWER_MODE ePwrMode)
{
	u32 uRegTemp;
	u32 uRegBitValue;

	uRegBitValue = 
		(ePwrMode == eMODE_NORMAL) ? 0x0 :
		(ePwrMode == eMODE_IDLE) ? 0x1 :
		(ePwrMode == eMODE_DEEPIDLE) ? 0x1 :
		(ePwrMode == eMODE_STOP) ? 0x2 :
		(ePwrMode == eMODE_DEEPSTOP) ? 0x2 :
		(ePwrMode == eMODE_SLEEP) ? 0x3 : 0xff;
	Assert(uRegBitValue != 0xff);	
	
	uRegTemp = syscInp32(rPWR_CFG);
	
	uRegTemp &= ~(0x3<<8);
	uRegTemp |= (uRegBitValue<<8);

	syscOutp32(rPWR_CFG, uRegTemp);

	DbgSysc(("4. PWR_CFG(0xE010_C000): 0x%x)\n", uRegTemp));
}

// Set before wfi() instruction preventing early wakeup error. 
void SYSC_DisableSyscInt(void)
{
	u32 uRegTemp;

	uRegTemp = syscInp32(rOTHERS);
	syscOutp32(rOTHERS, uRegTemp|(1));	
}

u32 SYSC_GetSyscIntStatus(void)
{
	return ( syscInp32(rOTHERS)&1 );
}

// [Set Usb Osc Pad on/off]
void SYSC_SetOscPadOnOff(POWER_MODE ePwrMode, OSC_CON eOscType, bool bEnable)
{
	u32 uRegTemp;
	u32 uOscType = (eOscType == eOSC_USB) ? 1 : 0;
	u32 uOscClkEnBitValue = (bEnable == true) ? 1 : 0;

	switch(ePwrMode)
	{
		case eMODE_NORMAL:			
		case eMODE_IDLE:
		case eMODE_DEEPIDLE:
			uRegTemp = syscInp32(rOSC_CON);
			uRegTemp &= ~(1<<uOscType);
			syscOutp32(rOSC_CON, uRegTemp|(uOscClkEnBitValue<<uOscType));
			break;
		case eMODE_STOP:
		case eMODE_DEEPSTOP:
			uRegTemp = syscInp32(rSTOP_CFG);
			uRegTemp &= ~(1<<uOscType);
			syscOutp32(rSTOP_CFG, uRegTemp|(uOscClkEnBitValue<<uOscType));
			break;
		case eMODE_SLEEP:
			uRegTemp = syscInp32(rSLEEP_CFG);
			uRegTemp &= ~(1<<uOscType);
			syscOutp32(rSLEEP_CFG, uRegTemp|(uOscClkEnBitValue<<uOscType));
			break;
		default:
			DbgSysc(("Not suppoerted power mode\n"));
			Assert(0);			
	}
}


// [Set wakeup mask]
void SYSC_SetWakeupMask(WAKEUP_ID eWakeupMask, bool bSrcMask)
{
	u32 uRegTemp;
	u32 uRegAddr;
	u32 uRegBitOffset;
	u32 uRegBitValue;

	uRegBitValue = (bSrcMask==true) ? 1 : 0; // unmask=0, mask=1

	switch(eWakeupMask)
	{
		case eWAKEUP_CEC:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 15;
			break;
		case eWAKEUP_ST:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 14;
			break;
		case eWAKEUP_I2S:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 13;
			break;
		case eWAKEUP_MMC3:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 12;
			break;
		case eWAKEUP_MMC2:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 11;
			break;
		case eWAKEUP_MMC1:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 10;
			break;
		case eWAKEUP_MMC0:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 9;
			break;
		case eWAKEUP_KEY:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 5;
			break;
		case eWAKEUP_TS1:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 4;
			break;
		case eWAKEUP_TS0:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 3;
			break;
		case eWAKEUP_RTCTICK:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 2;
			break;
		case eWAKEUP_RTCALARM:
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 1;
			break;
		case eWAKEUP_EINT0:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 0;
			break;
		case eWAKEUP_EINT1:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 1;
			break;
		case eWAKEUP_EINT2:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 2;
			break;
		case eWAKEUP_EINT3:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 3;
			break;
		case eWAKEUP_EINT4:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 4;
			break;
		case eWAKEUP_EINT5:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 5;
			break;
		case eWAKEUP_EINT6:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 6;
			break;
		case eWAKEUP_EINT7:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 7;
			break;
		case eWAKEUP_EINT8:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 8;
			break;
		case eWAKEUP_EINT9:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 9;
			break;
		case eWAKEUP_EINT10:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 10;
			break;
		case eWAKEUP_EINT11:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 11;
			break;	
		case eWAKEUP_EINT12:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 12;
			break;
		case eWAKEUP_EINT13:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 13;
			break;
		case eWAKEUP_EINT14:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 14;
			break;
		case eWAKEUP_EINT15:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 15;
			break;	
		case eWAKEUP_EINT16:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 16;
			break;
		case eWAKEUP_EINT17:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 17;
			break;
		case eWAKEUP_EINT18:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 18;
			break;
		case eWAKEUP_EINT19:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 19;
			break;
		case eWAKEUP_EINT20:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 20;
			break;
		case eWAKEUP_EINT21:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 21;
			break;	
		case eWAKEUP_EINT22:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 22;
			break;
		case eWAKEUP_EINT23:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 23;
			break;
		case eWAKEUP_EINT24:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 24;
			break;
		case eWAKEUP_EINT25:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 25;
			break;	
		case eWAKEUP_EINT26:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 26;
			break;
		case eWAKEUP_EINT27:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 27;
			break;
		case eWAKEUP_EINT28:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 28;
			break;
		case eWAKEUP_EINT29:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 29;
			break;
		case eWAKEUP_EINT30:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 30;
			break;
		case eWAKEUP_EINT31:
			uRegAddr = rEINT_WAKEUP_MASK; uRegBitOffset = 31;
			break;			
		case eWAKEUP_OTHERS: // dummy for test only.. reserved bit..
			uRegAddr = rWAKEUP_MASK; uRegBitOffset = 31;
			break;
		case eWAKEUP_ALL:
			if (uRegBitValue) // mask
			{
				syscOutp32(rWAKEUP_MASK, 0xffff);
				syscOutp32(rEINT_WAKEUP_MASK, 0xffffffff);
			}
			else // unmask
			{
				syscOutp32(rWAKEUP_MASK, 0);
				syscOutp32(rEINT_WAKEUP_MASK, 0);
			}
			return;
		default:
			DbgSysc(("No appropriate wakeup mask is found in SetWakeupMask()\n\x04"));
			Assert(0);
	}

	uRegTemp = syscInp32(uRegAddr);
	uRegTemp = uRegTemp &~(0x1<<uRegBitOffset) | (uRegBitValue<<uRegBitOffset);
	syscOutp32(uRegAddr, uRegTemp);
}

u32 SYSC_GetWakeupStatus(bool bRegValue) // added by junon
{
	u32 uRegTemp;
	u32 uRegAddr;
	u32 uRegBitOffset;
	u32 uRegBitValue;
	u32 i;
	WAKEUP_ID eWakeup;

	uRegTemp = syscInp32(rWAKEUP_STAT);

	if (bRegValue == TRUE) // get register value
		return uRegTemp;
	else // get wakeup source 
	{
		for (i=0;i<=15;i++) // wakeup source bit [15:0]
		{
			uRegBitValue = uRegTemp&(1<<i);
			eWakeup = (WAKEUP_ID)i;
			if (uRegBitValue) // first checked bit..
				break;
			else // didn't get wakeup source..
				eWakeup = eWAKEUP_OTHERS;
		}		
	}	

	return eWakeup;	
}

bool SYSC_ClearWakeupStatus(WAKEUP_ID eWakeup) // added by junon
{
	u32 uRegTemp;
	u32 uRegAddr;
	u32 uRegBitOffset;
	u32 uRegBitValue;
	
	switch(eWakeup)
	{
		case eWAKEUP_CEC:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 15;
			break;
		case eWAKEUP_ST:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 14;
			break;
		case eWAKEUP_I2S:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 13;
			break;
		case eWAKEUP_MMC3:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 12;
			break;
		case eWAKEUP_MMC2:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 11;
			break;
		case eWAKEUP_MMC1:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 10;
			break;
		case eWAKEUP_MMC0:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 9;
			break;
		case eWAKEUP_KEY:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 5;
			break;
		case eWAKEUP_TS1: 
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 4;
			break;
		case eWAKEUP_TS0:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 3;
			break;
		case eWAKEUP_RTCTICK:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 2;
			break;
		case eWAKEUP_RTCALARM:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 1;
			break;
		case eWAKEUP_EINT:
			uRegAddr = rWAKEUP_STAT; uRegBitOffset = 0;
			break;
		case eWAKEUP_ALL:
			syscOutp32(rWAKEUP_STAT, 0xFFFF);
			uRegTemp = syscInp32(rWAKEUP_STAT);
			if (uRegTemp == 0)
				return TRUE;
			else
				return FALSE; 
		default:
			DbgSysc(("No appropriate wakeup mask is found in ClearWakeup()\n\x04"));
			Assert(0);
	}

	// clear
	syscOutp32(uRegAddr, 0x1<<uRegBitOffset);

	// confirm to clear
	uRegTemp = syscInp32(uRegAddr);
	uRegTemp = uRegTemp & (0x1<<uRegBitOffset);

	if (uRegTemp == 0)
		return TRUE;
	else
		return FALSE; 
}


void SYSC_SetPowerMode(POWER_MODE ePwrMode) // temporary
{
	switch(ePwrMode)
	{
		case eMODE_NORMAL:			
			break;
		case eMODE_IDLE:
		case eMODE_DEEPIDLE:
			syscOutp32(rPWR_MODE, (1<<0));
			break;
		case eMODE_STOP:
		case eMODE_DEEPSTOP:
			syscOutp32(rPWR_MODE, (1<<1));
			break;
		case eMODE_SLEEP:
			syscOutp32(rPWR_MODE, (1<<2));
			break;
		default:
			Assert(0);			
	}
}


void SYSC_SetIdleModeConfig(u8 usTopLogicOn, u8 usTopMemOn, u8 usArmL2CacheRet, u8 usDeepMode) // added by junon
{
	u32 uRegData;

	uRegData = (usTopLogicOn<<30) | (usTopMemOn<<28) | (usArmL2CacheRet<<26) | usDeepMode;
	syscOutp32(rIDLE_CFG, uRegData);
}

// [Set stop mode configuration]
// STOP_CFG : Configure power manager at STOP mode
void SYSC_SetStopModeConfig(u8 usTopLogicOn, u8 usTopMemOn, u8 usArmL2CacheRet, u8 usDeepMode) // added by junon
{
	u32 uRegData;
	u32 uDeepOn = (usDeepMode == eCFG_DEEP) ? 0 : 2;

	uRegData = syscInp32(rSTOP_CFG);
	uRegData &= ~(0xff<<24);
	uRegData |= (usTopLogicOn<<30) | (usTopMemOn<<28) | (usArmL2CacheRet<<26) | (uDeepOn<<24);
	syscOutp32(rSTOP_CFG, uRegData);
}

// [Set stop memory configuration]
// STOP_MEM_CFG : Configure memory power at STOP mode
// Each bit is available when TOP_MEMORY_ON field of STOP_CFG is 0
// bRetentionEn - true: retention, false: off 
void SYSC_SetStopMemConfig(CFG_STOP_MEM eStopMem, bool bRetentionEn)
{
	u32 uRegTemp;
	u32 uRegBitOffset;
	u32 uRegBitValue;

	switch(eStopMem)
	{
		case eMEMCFG_ONENAND:
			uRegBitOffset = 8;
			break;
		case eMEMCFG_MODEMIF:
			uRegBitOffset = 7;
			break;
		case eMEMCFG_HOSTIF:
			uRegBitOffset = 6;
			break;
		case eMEMCFG_USBOTG:
			uRegBitOffset = 5;
			break;
		case eMEMCFG_MMC:
			uRegBitOffset = 4;
			break;
		case eMEMCFG_CSSYS:
			uRegBitOffset = 3;
			break;
		case eMEMCFG_SECSS:
			uRegBitOffset = 2;
			break;
		case eMEMCFG_IRAM:
			uRegBitOffset = 1;
			break;					
		case eMEMCFG_ALL:
			uRegTemp = (bRetentionEn == true) ? 0x1FF : 0x0;
			syscOutp32(rSTOP_MEM_CFG, uRegTemp);
			return;
		default:
			DbgSysc(("No appropriate stop mem config is found in SetStopMemConfig()\n\x04"));
			Assert(0);
	}
	
	uRegBitValue = (bRetentionEn) ? 1 : 0;

	uRegTemp = syscInp32(rSTOP_MEM_CFG);
	uRegTemp = uRegTemp &~(0x1<<uRegBitOffset) | (uRegBitValue<<uRegBitOffset);
	syscOutp32(rSTOP_MEM_CFG, uRegTemp);	
}

// [Set all stop memory configuration]
void SYSC_SetAllStopMemConfig(bool bOn)
{
	u32 uRegValue;

	uRegValue = (bOn == true) ? 0xffffffff : 0x0;
	syscOutp32(rSTOP_MEM_CFG, uRegValue);
}

u32 SYSC_GetNormalModeConfigReg(void)
{
	return syscInp32(rNORMAL_CFG);
}

u32 SYSC_GetIdleModeConfigReg(void)
{
	return syscInp32(rIDLE_CFG);
}

u32 SYSC_GetStopModeConfigReg(void)
{
	return syscInp32(rSTOP_CFG);
}

u32 SYSC_GetStopMemConfigReg(void)
{
	return syscInp32(rSTOP_MEM_CFG);
}


void SYSC_SetBlockPowerGating(SUB_BLOCK eBlock, bool bOn) // added by junon
{
	u32 uRegTemp;
	u32 uRegBitOffset;
	u32 uRegBitValue;

	switch(eBlock)
	{
		case eSUB_BLOCK_IROM:
			uRegBitOffset = 20;
			break;
		case eSUB_BLOCK_AUDIO:
			uRegBitOffset = 7;
			break;
		case eSUB_BLOCK_CAM:
			uRegBitOffset = 5;
			break;
		case eSUB_BLOCK_TV:
			uRegBitOffset = 4;
			break;
		case eSUB_BLOCK_LCD:
			uRegBitOffset = 3;
			break;
		case eSUB_BLOCK_G3D:
			uRegBitOffset = 2;
			break;
		case eSUB_BLOCK_MFC:
			uRegBitOffset = 1;
			break;
		case eSUB_BLOCK_ALL:
			uRegTemp = (bOn == true) ? 0xffffffff : 0x0;
			syscOutp32(rNORMAL_CFG, uRegTemp);	
			return;
		default:
			DbgSysc(("No appropriate stop mem config is found in SetStopMemConfig()\n\x04"));
			Assert(0);
	}
	
	uRegBitValue = (bOn) ? 1 : 0;

	uRegTemp = syscInp32(rNORMAL_CFG);
	uRegTemp &= ~(0x1<<uRegBitOffset); 
	uRegTemp |= (uRegBitValue<<uRegBitOffset);
	syscOutp32(rNORMAL_CFG, uRegTemp);	
}

u32 SYSC_GetBlockPowerStatus(SUB_BLOCK eBlock) // added by junon
{
	u32 uRegTemp;
	u32 uRegBitOffset;
	u32 uRegBitValue;

	switch(eBlock)
	{
		case eSUB_BLOCK_AUDIO:
			uRegBitOffset = 7;
			break;
		case eSUB_BLOCK_CAM:
			uRegBitOffset = 5;
			break;
		case eSUB_BLOCK_TV:
			uRegBitOffset = 4;
			break;
		case eSUB_BLOCK_LCD:
			uRegBitOffset = 3;
			break;
		case eSUB_BLOCK_G3D:
			uRegBitOffset = 2;
			break;
		case eSUB_BLOCK_MFC:
			uRegBitOffset = 1;
			break;
		case eSUB_BLOCK_TOP:
			uRegBitOffset = 0;
			break;
		case eSUB_BLOCK_ALL:
			uRegTemp = syscInp32(rBLK_PWR_STAT);
			return uRegTemp;
		default:
			DbgSysc(("No appropriate stop mem config is found in SetStopMemConfig()\n\x04"));
			Assert(0);
	}
	
	uRegTemp = syscInp32(rBLK_PWR_STAT);
	uRegTemp = uRegTemp & (0x1<<uRegBitOffset); 

	if (uRegTemp == 0)
		return 0;
	else
		return 1; 
}

u32 SYSC_GetBlockPowerStatusReg(void)
{
	return syscInp32(rBLK_PWR_STAT);
}


// [Set OSC frequency]
// Oscillator frequency scale counter (OSC_FREQ_VALUE x oscillator_frequency > 200ns)
// The count values vary depending on the size of the logic gates and the number of memories.
void SYSC_SetOscFreq(u32 uFreqValue)
{
	Assert(uFreqValue <= 0xF); // 4 bits
	syscOutp32(rOSC_FREQ, uFreqValue);
}

// [Set OSC stable counter]
// uCntValue max. : 0xFFFFF
void SYSC_SetOscStableCounter(u32 uCntValue)
{
	Assert(uCntValue <= 0xFFFFF); // 20 bits
	syscOutp32(rOSC_STABLE, uCntValue); 
}

// [Set power stable counter]
// uCntValue max. : 0xFFFFF
void SYSC_SetPowerStableCounter(u32 uCntValue)
{
	Assert(uCntValue <= 0xFFFFF); // 20 bits
	syscOutp32(rPWR_STABLE, uCntValue);
}

// [Set memory power stable counter]
// uRegBitValue - the number of external oscillator (or clock) cycles
// uCntValue max. : 0xF
void SYSC_SetMtcStableCounter(SUB_BLOCK eMtcStable, u32 uCntValue)
{
	u32 uRegTemp = 0;
	u32 uRegBitOffset;

	Assert(uCntValue <= 0xF);
	
	switch(eMtcStable)
	{
		case eSUB_BLOCK_AUDIO:
			uRegBitOffset = 28;
			break;
		case eSUB_BLOCK_CAM:
			uRegBitOffset = 20;
			break;
		case eSUB_BLOCK_TV:
			uRegBitOffset = 16;
			break;
		case eSUB_BLOCK_LCD:
			uRegBitOffset = 12;
			break;
		case eSUB_BLOCK_G3D:
			uRegBitOffset = 8;
			break;
		case eSUB_BLOCK_MFC:
			uRegBitOffset = 4;
			break;
		case eSUB_BLOCK_TOP:
		case eSUB_BLOCK_IROM:
			uRegBitOffset = 0;
			break;
		case eSUB_BLOCK_ALL:
			uRegTemp = 
				(uCntValue<<28)|(uCntValue<<20)|(uCntValue<<16)|(uCntValue<<12)|
				(uCntValue<<8)|(uCntValue<<4)|(uCntValue<<0);
			syscOutp32(rMTC_STABLE, uRegTemp);
			return;
		default:
			DbgSysc(("No appropriate MTC domain is found in SYSC_SetMtcStableCounter()\n\x04"));
			Assert(0);
	}

	uRegTemp = syscInp32(rMTC_STABLE);
	uRegTemp = uRegTemp &~(0xf<<uRegBitOffset) | (uCntValue<<uRegBitOffset);
	syscOutp32(rMTC_STABLE, uRegTemp);
}

// [Set clamp counter]
// CLAMP_STABLE is used for control timing 
//  between CLAMP_ON/OFF and POWER_ON/OFF in Core, L2Cache, ETM, and NEON of Cortex-A8.
void SYSC_SetClampStableCounter(bool bClampOn, u32 uCntValue)
{
	u32 uRegTemp;
	u32 uClampOnOffBitOffset;

	Assert(uCntValue <=0x3FF);

	uClampOnOffBitOffset = (bClampOn) ? 0 : 16;
	
	uRegTemp = syscInp32(rCLAMP_STABLE);
	uRegTemp = uRegTemp &~(0x3ff<<uClampOnOffBitOffset) | (uCntValue<<uClampOnOffBitOffset);
	syscOutp32(rCLAMP_STABLE, uRegTemp);
}




//===================================
// MISC in PMU
//===================================

void SYSC_ReleaseRetentionIO(RET_BLOCK eRET)
{
	u32 uRegTemp = 0;
	u32 uRegBitOffset;

	uRegTemp = syscInp32(rOTHERS);
	uRegBitOffset = eRET;
	
	if (eRET == eRET_BLOCK_ALL)
		uRegTemp = uRegTemp &~(0xf<<28) | (0xf<<28);
	else 
		uRegTemp = uRegTemp &~(1<<uRegBitOffset) | (1<<uRegBitOffset);
	syscOutp32(rOTHERS, uRegTemp);
}

// PS hold signal on/off
void SYSC_SetPSHold(bool bOn)
{
	u32 uRegTemp;
	u32 uRegValue;

	uRegValue = (bOn) ? 0x301 : 0;

	syscOutp32(rPS_HOLD_CON, uRegValue);	
}

// [Set information register (0~7)]
void SYSC_SetInformReg(u32 uRegIdx, u32 uRegValue)
{
	Assert(uRegIdx < 8);
	syscOutp32(rINFORM0+uRegIdx*4, uRegValue);
}

// [Get information register (0~7)]
u32 SYSC_GetInformReg(u32 uRegIdx) 
{
	Assert(uRegIdx < 8);
	return syscInp32(rINFORM0+uRegIdx*4);
}



///////////////////////////////////// User Library ///////////////////////////////////////

void SYSC_InitSysClk(INIT_SYSCLK uInitType)
{
 
	//**** New method to change PLL
	if ((uInitType == eINIT_SYSCLK_NB_NORM)|(uInitType == eINIT_SYSCLK_NB_HIGH))
	{
	    SYSC_ChangePllnDiv(eMPLL, 667, 12, 1, 300, 0, 0, 0, 0, 4, 2, 5, 2); // MPLL667, 166:133

		// DRAM refresh setting for 24Mhz
		DMC_ChangeRef(156,156); 

		SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKMPLL); // change ARMCLK source to MPLL
		SYSC_SetClkDiv(eCLKDIV_ONEDRAM, 4);
		SYSC_SetClkMuxState(eCLKMUX_ONEDRAM_SCLKMPLL); // change ARMCLK source to MPLL
		SYSC_UpdateClkInform(); // because mux is changed 

		if (uInitType == eINIT_SYSCLK_NB_NORM)		
			SYSC_ChangePllnDiv(eAPLL, 200, 6, 1, 300, 1, 4, 4, 2, 0, 0, 0, 0); // APLL800, 800:200, return ARMCLK to APLL
		else if (uInitType == eINIT_SYSCLK_NB_HIGH)		
			SYSC_ChangePllnDiv(eAPLL, 250, 6, 1, 300, 1, 5, 5, 2, 0, 0, 0, 0); // APLL1000, 1000:200

	 	SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKAPLL); // change ARMCLK source to APLL
		SYSC_SetClkMuxState(eCLKMUX_ONEDRAM_SCLKA2M); // change ARMCLK source to APLL
		SYSC_SetClkDiv(eCLKDIV_ONEDRAM, 1);

		// Update DRAM refresh counter
		SYSC_UpdateClkInform(); // because mux is changed 
		DMC_ChangeRef((10*(g_uOneDRAM/10000000)*78+9)/10, (10*(g_uHclkMsys/10000000)*78+9)/10); 

	 	return;
	}


	//**** Previous method to change PLL
	//DRAM 24Mhz refresh count
	DMC_ChangeRef(156,156); 

	//// Turn off a PLL sequence
	// - Deselect the output of PLLs
	SYSC_SetClkMuxState(eCLKMUX_APLL_FINPLL);
	SYSC_SetClkMuxState(eCLKMUX_MPLL_FINPLL);
	SYSC_SetClkMuxState(eCLKMUX_EPLL_FINPLL);
	SYSC_SetClkMuxState(eCLKMUX_VPLL_FINVPLL);
	// - Power off PLLs
	SYSC_StopPLL(eAPLL);	
	SYSC_StopPLL(eMPLL);		
	SYSC_StopPLL(eEPLL);		
	SYSC_StopPLL(eVPLL);		

	if (uInitType == eINIT_SYSCLK_NORMAL)
	{
		//// Fix dividers's value for max. operating frequency
#ifndef FPGA // silicon for POP
		// - APLL=800MHz, ARM=800MHz, MSYS=200MHz, MPLL=667MHz, DSYS=166MHz, PSYS=133MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,1);
		SYSC_SetClkDiv(eCLKDIV_A2M,4);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,4);	// for APLL 800MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,4);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,5);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
		SYSC_StartPLLbyFout(eAPLL, 800000000);
		SYSC_StartPLLbyFout(eMPLL, 667000000);
		SYSC_StartPLLbyFout(eEPLL, 96000000);
		SYSC_StartPLLbyFout(eVPLL, 54000000);
#else //fpga

#if 0
		// - APLL=36MHz, ARM=18MHz, MSYS=18MHz, MPLL=30MHz, DSYS=15MHz, PSYS=15MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,2); // 36/2=18MHz for APLL 36MHz
		SYSC_SetClkDiv(eCLKDIV_A2M,1);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,1);	// 18MHz 
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,2);	// 30/2=15Mhz for MPLL 30MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,2);	// 30/2=15Mhz for MPLL 30MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
		SYSC_StartPLLbyMPS(eAPLL, 12, 4, 0);	// 36MHz
		SYSC_StartPLLbyMPS(eMPLL, 10, 4, 0);	// 30MHz
		SYSC_StartPLLbyMPS(eEPLL, 12, 3, 1);	// 24MHz
		SYSC_StartPLLbyMPS(eVPLL, 10, 3, 1);	// 20MHz
#else
		// - APLL=24MHz, ARM=?/MHz, MSYS=?MHz, MPLL=20MHz, DSYS=?MHz, PSYS=?MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,2); // 24/2=12MHz for APLL 20MHz
		SYSC_SetClkDiv(eCLKDIV_A2M,2);
		SYSC_SetClkDiv(eCLKDIV_ONEDRAM,1);
		SYSC_SetClkMuxState(eCLKMUX_ONEDRAM_SCLKA2M);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,1);	
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,2);	// 24/2=12Mhz for MPLL 20MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,2);	// 24/2=12Mhz for MPLL 20MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
		SYSC_StartPLLbyMPS(eAPLL, 12, 3, 1);	// 24MHz
		SYSC_StartPLLbyMPS(eMPLL, 12, 3, 1);	// 24MHz
		SYSC_StartPLLbyMPS(eEPLL, 12, 3, 1);	// 24MHz
		SYSC_StartPLLbyMPS(eVPLL, 10, 3, 1);	// 20MHz
#endif

#endif
	}

	else if (uInitType == eINIT_SYSCLK_HIGH)
	{
		// - APLL=1GHz, ARM=1GHz, MSYS=200MHz, MPLL=667MHz, DSYS=166MHz, PSYS=133MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,1);
		SYSC_SetClkDiv(eCLKDIV_A2M,5);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,5);	// for APLL 1GHz
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,4);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,5);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
		SYSC_StartPLLbyFout(eAPLL, 1000000000);
		SYSC_StartPLLbyFout(eMPLL, 667000000);
		SYSC_StartPLLbyFout(eEPLL, 96000000);
		SYSC_StartPLLbyFout(eVPLL, 54000000);
	}
	else if (uInitType == eINIT_SYSCLK_MID)
	{
		// - APLL=667MHz, ARM=667MHz, MSYS=166MHz, MPLL=667MHz, DSYS=166MHz, PSYS=133MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,1);
		SYSC_SetClkDiv(eCLKDIV_A2M,4);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,4);	// for APLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,4);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,5);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
		SYSC_StartPLLbyFout(eAPLL, 667000000);
		SYSC_StartPLLbyFout(eMPLL, 667000000);
		SYSC_StartPLLbyFout(eEPLL, 96000000);
		SYSC_StartPLLbyFout(eVPLL, 54000000);
	}
	else if (uInitType == eINIT_SYSCLK_LOW)
	{
		// - APLL=533MHz, ARM=533MHz, MSYS=133MHz, MPLL=667MHz, DSYS=133MHz, PSYS=133MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,1);
		SYSC_SetClkDiv(eCLKDIV_A2M,4);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,4);	// for APLL 533MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,5);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,5);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
		SYSC_StartPLLbyFout(eAPLL, 533000000);
		SYSC_StartPLLbyFout(eMPLL, 667000000);
		SYSC_StartPLLbyFout(eEPLL, 96000000);
		SYSC_StartPLLbyFout(eVPLL, 54000000);
	}
	else if (uInitType == eINIT_SYSCLK_VERYLOW)
	{
		// - APLL=600MHz, ARM=300MHz, MSYS=100MHz, MPLL=604MHz, DSYS=101MHz, PSYS=101MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,2); // ARM
		SYSC_SetClkDiv(eCLKDIV_A2M,6);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,3);	// for APLL 600MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,6);	// for MPLL 604MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,6);	// for MPLL 604MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
		SYSC_StartPLLbyFout(eAPLL, 600000000);
		SYSC_StartPLLbyFout(eMPLL, 604000000);
		SYSC_StartPLLbyFout(eEPLL, 96000000);
		SYSC_StartPLLbyFout(eVPLL, 54000000);
	}

	else if (uInitType == eINIT_SYSCLK_MUX)
	{
		//// Fix dividers's value for max. operating frequency
#ifndef FPGA // silicon
		// - APLL=1GHz, ARM=1GHz, MSYS=200MHz, MPLL=667MHz, DSYS=166MHz, PSYS=133MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,1);
		SYSC_SetClkDiv(eCLKDIV_A2M,5);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,5);	// for APLL 1GHz
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,4);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,5);	// for MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
		SYSC_StartPLLbyFout(eAPLL, 1000000000);
		SYSC_StartPLLbyFout(eMPLL, 667000000);
		SYSC_StartPLLbyFout(eEPLL, 96000000);
		SYSC_StartPLLbyFout(eVPLL, 54000000);
#else //fpga
		// - APLL=36MHz, ARM=18MHz, MSYS=18MHz, MPLL=30MHz, DSYS=15MHz, PSYS=15MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,3); // for APLL 36MHz
		SYSC_SetClkDiv(eCLKDIV_A2M,1);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,1);	// for APLL 36MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,2);	// for MPLL 30MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,2);	// for MPLL 30MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
#if 1
		SYSC_StartPLLbyMPS(eAPLL, 12, 4, 0);	// 36MHz
		SYSC_StartPLLbyMPS(eMPLL, 10, 4, 0);	// 30MHz
		SYSC_StartPLLbyMPS(eEPLL, 12, 3, 1);	// 24MHz
		SYSC_StartPLLbyMPS(eVPLL, 10, 3, 1);	// 20MHz
#else
		SYSC_StartPLLbyMPS(eAPLL, 8, 4, 1);	// 12MHz
		SYSC_StartPLLbyMPS(eMPLL, 8, 4, 1);	// 12MHz
		SYSC_StartPLLbyMPS(eEPLL, 8, 4, 1);	// 12MHz
		SYSC_StartPLLbyMPS(eVPLL, 8, 4, 1);	// 12MHz
#endif

#endif		
	}

	else if (uInitType == eINIT_SYSCLK_DIV)
	{
		//// Fix dividers's value for max. operating frequency
#ifndef FPGA // silicon
		//// Fix dividers's value for max. operating frequency
		// - APLL=200MHz, ARM=200MHz, MSYS=200MHz, MPLL=133MHz, DSYS=133MHz, PSYS=133MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,1);
		SYSC_SetClkDiv(eCLKDIV_A2M,1);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,1);	
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,1);	
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,1);	
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
		SYSC_StartPLLbyFout(eAPLL, 200000000);
		SYSC_StartPLLbyFout(eMPLL, 133000000);
		SYSC_StartPLLbyFout(eEPLL, 96000000);
		SYSC_StartPLLbyFout(eVPLL, 54000000);
#else //fpga
		// - APLL=36MHz, ARM=18MHz, MSYS=18MHz, MPLL=30MHz, DSYS=15MHz, PSYS=15MHz
		SYSC_SetClkDiv(eCLKDIV_APLL,3); // for APLL 36MHz
		SYSC_SetClkDiv(eCLKDIV_A2M,1);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS,1);	// for APLL 36MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS,2);	// for MPLL 30MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS,2);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS,2);	// for MPLL 30MHz
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS,2);
#if 1
		SYSC_StartPLLbyMPS(eAPLL, 12, 4, 0);	// 36MHz
		SYSC_StartPLLbyMPS(eMPLL, 10, 4, 0);	// 30MHz
		SYSC_StartPLLbyMPS(eEPLL, 12, 3, 1);	// 24MHz
		SYSC_StartPLLbyMPS(eVPLL, 10, 3, 1);	// 20MHz
#else
		SYSC_StartPLLbyMPS(eAPLL, 8, 4, 1);	// 12MHz
		SYSC_StartPLLbyMPS(eMPLL, 8, 4, 1);	// 12MHz
		SYSC_StartPLLbyMPS(eEPLL, 8, 4, 1);	// 12MHz
		SYSC_StartPLLbyMPS(eVPLL, 8, 4, 1);	// 12MHz
#endif

#endif
	}

	else 
		Assert(0);
	
	//// Turn on a PLL sequence
	// - Power on PLLs
#if 0 // same register value 
	SYSC_StartPLL(eAPLL);	
	SYSC_StartPLL(eMPLL);		
	SYSC_StartPLL(eEPLL);		
	SYSC_StartPLL(eVPLL);		
#endif
	// - Select the output of PLLs
	SYSC_SetClkMuxState(eCLKMUX_APLL_FOUTAPLL);
	SYSC_SetClkMuxState(eCLKMUX_MPLL_FOUTMPLL);
	SYSC_SetClkMuxState(eCLKMUX_EPLL_FOUTEPLL);
	SYSC_SetClkMuxState(eCLKMUX_VPLL_FOUTVPLL);

	// Update DRAM refresh counter & console
	SYSC_UpdateClkInform(); // because mux is changed 
	DMC_ChangeRef((10*(g_uOneDRAM/10000000)*78+9)/10, (10*(g_uHclkMsys/10000000)*78+9)/10); 
}


// uSelDivAPLL, uSelDivA2M, uSelDivHclkMsys, uSelDivPclkMsys are only for APLL
// uSelDivHclkDsys, uSelDivPclkDsys, uSelDivHclkPsys, uSelDivPclkPsys are only for MPLL
void SYSC_ChangePllnDiv(PLL_TYPE eSelPllType, u32 uSelM, u32 uSelP, u32 uSelS, u32 uSelLockTime,
	u32 uSelDivAPLL, u32 uSelDivA2M, u32 uSelDivHclkMsys, u32 uSelDivPclkMsys,
	u32 uSelDivHclkDsys, u32 uSelDivPclkDsys, u32 uSelDivHclkPsys, u32 uSelDivPclkPsys)
{
	u32 uSelFreq; // unit KHz
	
	SYSC_SetLockTime(eSelPllType, uSelLockTime);
	uSelFreq = SYSC_CalPllFout(eSelPllType, uSelM, uSelP, uSelS)/1000; // unit KHz

	// pre DRAM refresh setting for 24Mhz worst case
	if ((eSelPllType == eAPLL)||(eSelPllType == eMPLL))
	{
		DMC_SetCh(1);
		DMC_SetAutoRefresh(TRUE, 156); // for 20MHz
		DMC_SetCh(0);
		DMC_SetAutoRefresh(TRUE, 156); // for 20MHz
	}

	// Change M.P.S values of PLL & Bus divider values
	switch(eSelPllType)
	{
	case eAPLL :
		// Check changed clock speed (up or down)
		// down to up (Div->PLL), up to down (PLL->Div)
		if (g_uAPLL >= uSelFreq*1000) // up to down. 
			SYSC_StartPLLbyMPS(eSelPllType, uSelM, uSelP, uSelS); 
		SYSC_SetClkDiv(eCLKDIV_APLL, uSelDivAPLL);
		SYSC_SetClkDiv(eCLKDIV_A2M, uSelDivA2M);
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS, uSelDivHclkMsys);
		SYSC_SetClkDiv(eCLKDIV_PCLKMSYS, uSelDivPclkMsys);
		if (g_uAPLL < uSelFreq*1000) // down to up. 
			SYSC_StartPLLbyMPS(eSelPllType, uSelM, uSelP, uSelS); 
		// Select Mux to PLL 
		SYSC_SetClkMuxState(eCLKMUX_APLL_FOUTAPLL);
		SYSC_SetClkOut(eCLKOUT_FOUTAPLL_DIV4,2);
		break;
	case eMPLL :
		// Check changed clock speed (up or down)
		// down to up (Div->PLL), up to down (PLL->Div)
		if (g_uMPLL >= uSelFreq*1000) // up to down. 
			SYSC_StartPLLbyMPS(eSelPllType, uSelM, uSelP, uSelS); 
		SYSC_SetClkDiv(eCLKDIV_HCLKDSYS, uSelDivHclkDsys);
		SYSC_SetClkDiv(eCLKDIV_PCLKDSYS, uSelDivPclkDsys);
		SYSC_SetClkDiv(eCLKDIV_HCLKPSYS, uSelDivHclkPsys);
		SYSC_SetClkDiv(eCLKDIV_PCLKPSYS, uSelDivPclkPsys);
		if (g_uMPLL < uSelFreq*1000) // down to up. 
			SYSC_StartPLLbyMPS(eSelPllType, uSelM, uSelP, uSelS); 
		// Select Mux to PLL 
		SYSC_SetClkMuxState(eCLKMUX_MPLL_FOUTMPLL);
		SYSC_SetClkOut(eCLKOUT_FOUTMPLL_DIV2,10);
		break;
	case eEPLL :
		SYSC_StartPLLbyMPS(eSelPllType, uSelM, uSelP, uSelS); 
		// Select Mux to PLL 
		SYSC_SetClkMuxState(eCLKMUX_EPLL_FOUTEPLL);
		SYSC_SetClkOut(eCLKOUT_FOUTEPLL,10);
	case eVPLL :
		SYSC_StartPLLbyMPS(eSelPllType, uSelM, uSelP, uSelS); 
		// Select Mux to PLL 
		SYSC_SetClkMuxState(eCLKMUX_VPLL_FOUTVPLL);
		SYSC_SetClkOut(eCLKOUT_FOUTVPLL,10);
		break;
	default :
		Assert(0);
		break;
	}			
		
	// Update DRAM refresh counter & console
	if ((eSelPllType == eAPLL)||(eSelPllType == eMPLL))
	{
		SYSC_UpdateClkInform(); // because mux is changed 
		DMC_SetCh(1);
		DMC_SetAutoRefresh(TRUE, (10*(g_uHclkMsys/10000000)*78+9)/10); // for Result DMC1 Frequency
		DMC_SetCh(0);
		DMC_SetAutoRefresh(TRUE, (10*(g_uOneDRAM/10000000)*78+9)/10); // for Result DMC0 Frequency
		UART_InitDebugCh(DBG_UART_CH, 115200);
	}
}


void SYSC_SetBusDiv(u32 uSelDivAPLL, u32 uSelDivA2M, u32 uSelDivHclkMsys, u32 uSelDivPclkMsys,
					u32 uSelDivHclkDsys, u32 uSelDivPclkDsys, u32 uSelDivHclkPsys, u32 uSelDivPclkPsys)
{
	u32 uTempReg;

	uTempReg = ((uSelDivAPLL-1)|
				((uSelDivA2M-1)<<4)|
				((uSelDivHclkMsys-1)<<8)|
				((uSelDivPclkMsys-1)<<12)|
				((uSelDivHclkDsys-1)<<16)|
				((uSelDivPclkDsys-1)<<20)|
				((uSelDivHclkPsys-1)<<24)|
				((uSelDivPclkPsys-1)<<28));
	syscOutp32(rCLK_DIV0, uTempReg);
}


volatile static DVFS_LV	g_ePrevDvfs = eDVFS_LV1;

// uSelDivAPLL, uSelDivA2M, uSelDivHclkMsys, uSelDivPclkMsys are only for APLL
// uSelDivHclkDsys, uSelDivPclkDsys, uSelDivHclkPsys, uSelDivPclkPsys are only for MPLL
void SYSC_ChangeDvfsLv(DVFS_LV eDvfs_Lv)
{
	// ARMdiv, MHdiv(A2Mdiv), MPdiv, DHdiv, DPdiv, PHdiv, PPdiv 
	u32 uDVFS_LV[DVFS_MAX_LV][7] =   {{	1,	5,	2,	4,	2,	5,	2}, // 1GHz only
									  {	1,	4,	2,	4,	2,	5,	2}, // 800MHz
									  {	2,	2,	2,	4,	2,	5,	2},
									  {	4,	1,	2,	4,	2,	5,	2},
									  {	4,	2,	1,	4,	2,	5,	2},
									  {	4,	2,	1,	8,	2,	8,	2},
									  {	8,	1,	2,	8,	2,	8,	2},
									  {	8,	2,	1,	16,	1,	16,	1}	
									 };
	u32 uTempReg;

	// DRAM refresh setting for 24Mhz
	DMC_SetCh(1);
	DMC_SetAutoRefresh(TRUE, 156); // for 20MHz
	DMC_SetCh(0);
	DMC_SetAutoRefresh(TRUE, 156); // for 20MHz

	// Change M.P.S values of PLL & Bus divider values
	// Check changed clock speed (up or down)
	// down to up (Div->PLL), up to down (PLL->Div)
	if (eDvfs_Lv == eDVFS_LV0)
	{
		SYSC_SetBusDiv(uDVFS_LV[eDvfs_Lv][0], uDVFS_LV[eDvfs_Lv][1], uDVFS_LV[eDvfs_Lv][1], uDVFS_LV[eDvfs_Lv][2],
					uDVFS_LV[eDvfs_Lv][3], uDVFS_LV[eDvfs_Lv][4], uDVFS_LV[eDvfs_Lv][5], uDVFS_LV[eDvfs_Lv][6]);

		// change method : MPLL 667MHz
		SYSC_SetClkDiv(eCLKDIV_ONEDRAM, 4); // change DMC0 source to MPLL 667MHz - case using A2M
		SYSC_SetClkMuxState(eCLKMUX_ONEDRAM_SCLKMPLL); // change DMC0 source to MPLL - case using A2M
		SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKMPLL); // change ARMCLK source to MPLL
		
		SYSC_StartPLLbyMPS(eAPLL, 250, 6, 1); // 1GHz

		// return to APLL 
		SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKAPLL); // change ARMCLK source to APLL
	}
	else
	{
		if (g_ePrevDvfs == eDVFS_LV0)
		{
			// change method : MPLL 667MHz
			SYSC_SetClkDiv(eCLKDIV_ONEDRAM, 4); // change DMC0 source to MPLL 667MHz - case using A2M
			SYSC_SetClkMuxState(eCLKMUX_ONEDRAM_SCLKMPLL); // change DMC0 source to MPLL - case using A2M
			SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKMPLL); // change ARMCLK source to MPLL
			
			SYSC_StartPLLbyMPS(eAPLL, 200, 6, 1); // 800MHz

			// return to APLL 
			SYSC_SetClkMuxState(eCLKMUX_MSYS_SCLKAPLL); // change ARMCLK source to APLL
		}

		SYSC_SetBusDiv(uDVFS_LV[eDvfs_Lv][0], uDVFS_LV[eDvfs_Lv][1], uDVFS_LV[eDvfs_Lv][1], uDVFS_LV[eDvfs_Lv][2],
					uDVFS_LV[eDvfs_Lv][3], uDVFS_LV[eDvfs_Lv][4], uDVFS_LV[eDvfs_Lv][5], uDVFS_LV[eDvfs_Lv][6]);
	}	

	
#if 0	// high to low
	SYSC_SetClkDiv(eCLKDIV_APLL, uDVFS_LV[eDvfs_Lv][0]);
	SYSC_SetClkDiv(eCLKDIV_A2M, uDVFS_LV[eDvfs_Lv][1]);
	SYSC_SetClkDiv(eCLKDIV_HCLKMSYS, uDVFS_LV[eDvfs_Lv][1]);
	SYSC_SetClkDiv(eCLKDIV_PCLKMSYS, uDVFS_LV[eDvfs_Lv][2]);
	SYSC_SetClkDiv(eCLKDIV_HCLKDSYS, uDVFS_LV[eDvfs_Lv][3]);
	SYSC_SetClkDiv(eCLKDIV_PCLKDSYS, uDVFS_LV[eDvfs_Lv][4]);
	SYSC_SetClkDiv(eCLKDIV_HCLKPSYS, uDVFS_LV[eDvfs_Lv][5]);
	SYSC_SetClkDiv(eCLKDIV_PCLKPSYS, uDVFS_LV[eDvfs_Lv][6]);
#endif

	SYSC_SetClkOut(eCLKOUT_HCLKMSYS,4);
		
	// Update DRAM refresh counter & console
	SYSC_UpdateClkInform(); // because mux is changed 
	DMC_SetCh(1);
	DMC_SetAutoRefresh(TRUE, (10*(g_uHclkMsys/10000000)*78+9)/10); // for Result DMC1 Frequency
	DMC_SetCh(0);
	DMC_SetAutoRefresh(TRUE, (10*(g_uOneDRAM/10000000)*78+9)/10); // for Result DMC0 Frequency
	UART_InitDebugCh(3, 115200);

	g_ePrevDvfs = eDvfs_Lv;
}


// [Start power mode]
// bEnterMethod : Reserved..
// bUserConfig : TRUE-Use user's setting, FALSE-Use Simple following setting
void SYSC_StartPowerMode(POWER_MODE ePwrMode, bool bEnterMethod, bool bUserConfig)	
{
	if (!bUserConfig)	// If user didn't config. for test
	{
		switch(ePwrMode)
		{
			case eMODE_NORMAL:			
				break;
			case eMODE_IDLE: 
				SYSC_SetIdleModeConfig(eCFG_ON, eCFG_ON, eCFG_ON, eCFG_NODEEP); // 0:OFF, 1:Retention, 2:ON, TOP_LOGIC = 1, TOP_MEM = 1, ARM_MEM = 1, DEEP_MODE =0
				break;
			case eMODE_DEEPIDLE:
				SYSC_SetIdleModeConfig(eCFG_ON, eCFG_ON, eCFG_ON, eCFG_DEEP); // TOP_LOGIC = 1, TOP_MEM = 1, ARM_MEM = 1, DEEP_MODE =1
				break;
			case eMODE_STOP:
				SYSC_SetStopModeConfig(eCFG_ON, eCFG_ON, eCFG_ON, eCFG_NODEEP); // TOP_LOGIC = 1, TOP_MEM = 1, ARM_MEM = 1, DEEP_MODE =0
				SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, true);
				SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, true);
				break;
			case eMODE_DEEPSTOP:
				SYSC_SetStopModeConfig(eCFG_ON, eCFG_ON, eCFG_ON, eCFG_DEEP);	// TOP_LOGIC = 0, TOP_MEM = 0, ARM_MEM = 0, DEEP_MODE =1
				SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, true);
				SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, true);
				break;
			case eMODE_SLEEP:
				SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, true);
				SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, true);
				break;
			default:
				Assert(0);			
		}
	}

	if (bEnterMethod==SW_WFI) 
	{
		// S/W WFI only
		SYSC_SetConfigStandbyWFI(ePwrMode);
		SYSC_DisableSyscInt();
#ifndef EVT0	
		do __wfi();
		while ( SYSC_GetSyscIntStatus()==1 );
#else
		__wfi();
#endif
	}
	else
	{
		// H/W WFI temp
		SYSC_SetConfigStandbyWFI(eMODE_NORMAL); // for HW WFI
		SYSC_DisableSyscInt();
		SYSC_SetPowerMode(ePwrMode);
	}
}


// [Simple entering power mode library]
void SYSC_EnteringPowerMode(POWER_MODE ePwrMode, PWRMODE_CFG_ID ucSelTop, u32 uSelMethod, u32 uSelAging)
{
	u32 uInformReg;
	u8	ucSelL2;

	uInformReg = SYSC_GetInformReg(5); // init inform reg for power mode & aging
	uInformReg &= ~(0xFF); // clear inform register for aging

	SYSC_SetSleepGPIO(); // use deep(top off/retention) & sleep mode

	if ((ucSelTop!=0)) 
		ucSelL2 = 1;
	else 
		ucSelL2 = 0;

	if ((ucSelTop>2)||(ucSelTop<1)) 
		ucSelTop = 1; // default retention

	uSelAging = (uSelAging != 2) ? 1 : 2;	// default	General

	// Implement configuration
	switch(ePwrMode)
	{
		case eMODE_DEEPIDLE :
 			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
			SYSC_SetIdleModeConfig(ucSelTop, ucSelTop, ucSelL2, eCFG_DEEP);
			uInformReg |= INFORM_DIDLE;	// 0x2x = deep idle mode
			break;
		case eMODE_STOP :
 			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
			SYSC_SetStopModeConfig(eCFG_ON, (u8)eCFG_RET, ucSelL2, eCFG_NODEEP);
			SYSC_SetStopMemConfig(eMEMCFG_ALL, ucSelL2);			
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, true);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, true);
			uInformReg |= INFORM_STOP;	// 0x3x = stop mode
			break;
		case eMODE_DEEPSTOP :
 			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
			SYSC_SetStopModeConfig(ucSelTop, (u8)eCFG_RET, ucSelL2, eCFG_DEEP);
			SYSC_SetStopMemConfig(eMEMCFG_ALL, ucSelL2);			
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, false);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, false);
			uInformReg |= INFORM_DSTOP;	// 0x4x = deep stop mode
			break;
		case eMODE_SLEEP :
 			SYSC_SetClkOut(eCLKOUT_HCLKDSYS, 2); 
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_MAIN, false);
			SYSC_SetOscPadOnOff(ePwrMode, eOSC_USB, false);
			uInformReg |= INFORM_SLEEP;	// 0x5x = sleep mode
			break;
		default:
			ePwrMode = eMODE_IDLE;
			SYSC_SetClkOut(eCLKOUT_ARMCLK_DIV4, 2); // ARMCLK / 8
			SYSC_SetIdleModeConfig(ucSelTop, ucSelTop, ucSelL2, eCFG_NODEEP);
			uInformReg |= INFORM_IDLE;	// 0x1x = Idle mode
			break;
	}

	// write aging and mode info. to infrom register 
	if (uSelAging==2) // aging test
	{
		SYSC_SetInformReg(6, 0); // init count aging test
		SYSC_SetInformReg(5, uInformReg|INFORM_AGING);  
		if ( (ePwrMode==eMODE_SLEEP)||((ePwrMode==eMODE_DEEPSTOP)&&(ucSelTop==0)) )
			Outp32(0xD0030000, 0x0); // remove iRAM start address for verifying BL1 copy operation
	}
	else // general test
	{
		SYSC_SetInformReg(5, uInformReg|INFORM_GENERAL); 
	}

	if (uSelMethod==SW_WFI) 
	{
		// S/W WFI only
		SYSC_SetConfigStandbyWFI(ePwrMode);
		SYSC_DisableSyscInt();
#ifndef EVT0	
		do __wfi();
		while ( SYSC_GetSyscIntStatus()==1 );
#else
		__wfi();
#endif
	}
	else
	{
		// H/W WFI temp
		SYSC_SetConfigStandbyWFI(eMODE_NORMAL); // for HW WFI
		SYSC_DisableSyscInt();
		SYSC_SetPowerMode(ePwrMode);
	}

}

void SYSC_SetSleepGPIO(void)
{
	// ConRegPDN 00:output 0, 01:output 1, 10:input, 11:previous -- 2n
	// PullUDPDN 00:disable 0, 01:pulldown 1, 10:pullup, 11:res -- 2n

	// set MP0 I/O retention status
	GPIO_SetConRegPDNAll(eGPIO_MP0_1, 0xFFFF); // previous state
	GPIO_SetConRegPDNAll(eGPIO_MP0_2, 0xFFFF); // previous state
	GPIO_SetConRegPDNAll(eGPIO_MP0_3, 0xFFFF); // previous state
	GPIO_SetConRegPDNAll(eGPIO_MP0_4, 0xFFFF); // previous state
	GPIO_SetConRegPDNAll(eGPIO_MP0_5, 0xFFFF); // previous state
	GPIO_SetConRegPDNAll(eGPIO_MP0_6, 0xFFFF); // previous state
	GPIO_SetConRegPDNAll(eGPIO_MP0_7, 0xFFFF); // previous state

	GPIO_SetConRegPDNAll(eGPIO_J1, 0x000); // SROM ADDR 16~21 : set Output L
	GPIO_SetPullUDPDNAll(eGPIO_J1, 0x000); // SROM ADDR 16~21 : Pullup disable
	GPIO_SetConRegPDNAll(eGPIO_J4, 0x0); // SROM ADDR 22 : set Output L
	GPIO_SetPullUDPDNAll(eGPIO_J4, 0x0); // SROM ADDR 22 : Pullup disable

	DMC_PhyDriving(0, 1, 0); // DMC0 data bus All Zero
	DMC_PhyDriving(1, 1, 0); // DMC1 data bus All Zero
	// add nDQS enable
	DMC_SetDQSn(0, true);
	DMC_SetDQSn(1, true);	
	// Automatic release item
	GPIO_SetConRegPDNAll(eGPIO_MP0_3, 0x0FFF); // RnB2,3 Output Low 

	// SPI PDN setting pull up
	GPIO_SetConRegPDNAll(eGPIO_B, 0xAAAA); // SPI0,1
	GPIO_SetPullUDPDNAll(eGPIO_B, 0xAAAA); 

	// MMC PDN setting pull up
	GPIO_SetConRegPDNAll(eGPIO_G0, 0xAAAA); // MMC0
	GPIO_SetPullUDPDNAll(eGPIO_G0, 0xAAAA); 
	GPIO_SetConRegPDNAll(eGPIO_G1, 0xAAAA); // MMC1
	GPIO_SetPullUDPDNAll(eGPIO_G1, 0xAAAA); 
	GPIO_SetConRegPDNAll(eGPIO_G2, 0xAAAA); // MMC2
	GPIO_SetPullUDPDNAll(eGPIO_G2, 0xAAAA); 
	GPIO_SetConRegPDNAll(eGPIO_G3, 0xAAAA); // MMC3
	GPIO_SetPullUDPDNAll(eGPIO_G3, 0xAAAA); 

	// UART PDN setting pull up
	GPIO_SetConRegPDNAll(eGPIO_A1, 0xAAAA); // UART2,3
	GPIO_SetPullUDPDNAll(eGPIO_A1, 0xAAAA); 
	
}



#if 1 // added by jspark for code merge (09.04.27)

/*
/////////////////////////////////////////////////////////////
//Selection Source CLK of Audio IPs(I2S, PCM)
//for EPLL, speed must set before this.
void SYSC_SetAudioPath(AUDIO_CLKNum eClkNum, AUDIO_CLKSRC eClkSrc)
{

//-------------------------------------
//
//this code should be generated for C110
//
//-------------------------------------

//this code is for 6410
#if 0
	if(eClkNum == AUDIO_CLK0)
	{
		if(eClkSrc == eAUDIO_MOUT_EPLL){
			SYSC_ClkSrc(eEPLL_FOUT);
			SYSC_ClkSrc(eAUDIO0_MOUTEPLL);
		}		 		
		else if (eClkSrc == eAUDIO_DOUT_MPLL) SYSC_ClkSrc(eAUDIO0_DOUTMPLL);
		else if (eClkSrc == eAUDIO_FIN) 		SYSC_ClkSrc(eAUDIO0_FINEPLL);
		else if (eClkSrc == eAUDIO_I2SCDCLK)	SYSC_ClkSrc(eAUDIO0_I2SCDCLK0);
		else if (eClkSrc == eAUDIO_PCMCDCLK)	SYSC_ClkSrc(eAUDIO0_PCMCDCLK0);		
		else if (eClkSrc == eAUDIO_MOUT_HPLL) SYSC_ClkSrc(eAUDIO0_MOUTHPLL);		
	} else 
	if(eClkNum == AUDIO_CLK1)
	{
		if(eClkSrc == eAUDIO_MOUT_EPLL){
					SYSC_ClkSrc(eEPLL_FOUT);
			 		SYSC_ClkSrc(eAUDIO1_MOUTEPLL);
		}		 		
		else if (eClkSrc == eAUDIO_DOUT_MPLL) 	SYSC_ClkSrc(eAUDIO1_DOUTMPLL);
		else if (eClkSrc == eAUDIO_FIN) 			SYSC_ClkSrc(eAUDIO1_FINEPLL);
		else if (eClkSrc == eAUDIO_I2SCDCLK)		SYSC_ClkSrc(eAUDIO1_I2SCDCLK1);
		else if (eClkSrc == eAUDIO_PCMCDCLK)		SYSC_ClkSrc(eAUDIO1_PCMCDCLK1);		
		else if (eClkSrc == eAUDIO_MOUT_HPLL) 	SYSC_ClkSrc(eAUDIO1_MOUTHPLL);		
	}
	else 
	if(eClkNum == AUDIO_CLK2)
	{
		if(eClkSrc == eAUDIO_MOUT_EPLL) {
			SYSC_ClkSrc(eEPLL_FOUT);
			SYSC_ClkSrc(eAUDIO2_MOUTEPLL);
		}
		else if (eClkSrc == eAUDIO_DOUT_MPLL) 	SYSC_ClkSrc(eAUDIO2_DOUTMPLL);
		else if (eClkSrc == eAUDIO_FIN) 			SYSC_ClkSrc(eAUDIO2_FINEPLL);
		else if (eClkSrc == eAUDIO_I2SCDCLK)		SYSC_ClkSrc(eAUDIO2_I2SCDCLK2);
		else if (eClkSrc == eAUDIO_MOUT_HPLL) 	SYSC_ClkSrc(eAUDIO2_MOUTHPLL);
	}
#endif	 
}
*/

void SYSC_SetPLL(PLL_TYPE  ePLL, u32 uMdiv, u32 uPdiv, u32 uSdiv, u32 uKdiv)
{
	u32 uRegValue;
	uRegValue = (u32)(((u32)(0x1<<31))|(uMdiv<<16)|(uPdiv<<8)|(uSdiv<<0));
	SYSC_SetPLLReg(ePLL,uRegValue);
}

		
//////////
// Function Name : Audio_SetEPLL
// Function Description : 
// Input : 	RFS
// Output : Output Frequency of EPLL[Hz]
// Version : 
// Example :
//	
//	RFS	: 384fs ( I2SCDCLK/384 = 44.09KHz)
//	BFS : 48fs (RFS*48 = 2.116MHz)
u32 Sys_SetAudioEPLL(u32 uRCLK)
{
	u32 dTempRCLK;
	u32 uFOUTEpll;	
	
	dTempRCLK=uRCLK;	
	uFOUTEpll = 	(32768000%dTempRCLK ==0)?32768000 :
						(45158400%dTempRCLK ==0)?45158400 : 
						(49152000%dTempRCLK ==0)?49152000 :
						(33868800%dTempRCLK ==0)?67738000 : 
						(36864000%dTempRCLK ==0)?73728000 : 0;	
	
	if(uFOUTEpll == 32768000)			SYSC_SetPLL(eEPLL,132,3,4,0);//m,p,s
	else if(uFOUTEpll == 49152000)		SYSC_SetPLL(eEPLL,99,3,3,0);
	else if(uFOUTEpll == 67738000)		SYSC_SetPLL(eEPLL,135,3,3,0);
	else if(uFOUTEpll == 73728000)		SYSC_SetPLL(eEPLL,147,3,3,0);
	else if(uFOUTEpll == 45158400)		SYSC_SetPLL(eEPLL,90,3,3,0);
	else Assert(0);
	
	Delay(100);
			
	return uFOUTEpll;
}
/////////////////////

#endif


//////////
// Function Name : SYSC_RdRSTSTAT
// Function Description : This function read Rest Status Register
//				  
// Input : NONE		  	
//							  
// Output : Status Value
// Version : 
u32 SYSC_RdRSTSTAT(u32 uPRINT)
{
	u32 uRegValue;
	u32 uRstId;

	uRegValue =Inp32(rRST_STAT);

	if(uPRINT == 1)
	UART_Printf("Reset Status Register:	0x%x\n",  uRegValue);

	if( uRegValue & (1<<0))
	{
		if(uPRINT == 1)
		UART_Printf("External reset by XnRESET  \n\n");
		uRstId = 0;
	}
	else if (uRegValue & (1<<1))
	{
		if(uPRINT == 1)
		UART_Printf("Warm reset by XnWRESET \n\n");
		uRstId = 1;
		
	}
	else if (uRegValue & (1<<2))
	{
		if(uPRINT == 1)
		UART_Printf("Watch dog timer reset by WDTRST\n\n");
		uRstId = 2;
		
	}
	else if (uRegValue & (1<<3))
	{
		if(uPRINT == 1)
		UART_Printf("Software reset by SWRESET\n\n");
		uRstId = 3;
		
	}
	else if (uRegValue & (1<<16))
	{
		if(uPRINT == 1)
			UART_Printf("Reset by SLEEP Mode Wake-up \n\n");
		uRstId = 16;		
	}	
	else if (uRegValue & (1<<18))
	{
		if(uPRINT == 1)
		UART_Printf("Reset by DEEP-STOP mode Wake-up \n\n");
		uRstId = 18;
	}
	else if (uRegValue & (1<<19))
	{
		if(uPRINT == 1)
		UART_Printf("Reset by DEEP-IDLE mode Wake-up \n\n");
		uRstId = 19;
	}

	return uRstId;
}

#if 1		// For MM Part
void SYSC_SetBlockPowerON_IP(IP_eID eIP) // added by JWJ
{
	u32 uBlockStatus;

	switch(eIP)
	{
		// L Block
		case eFIMD:
		case eMDNIE:
		case eMIE:
		case eBLKLCD:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_LCD);
			if(uBlockStatus == 0)	// Block Power Off
				{
					SYSC_SetBlockPowerGating(eSUB_BLOCK_LCD, true);
					do
					{
					uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_LCD);
					}
					while(uBlockStatus != 1);
				}
			break;
		// X Block
		case eFIMC0:
		case eFIMC1:
		case eFIMC2:
		case eIPC:
		case eJPEG:
		case eROTATOR:
		case eBLKCAM:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_CAM);
			if(uBlockStatus == 0)	// Block Power Off
				{
					SYSC_SetBlockPowerGating(eSUB_BLOCK_CAM, true);
					do
					{
					uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_CAM);
					}
					while(uBlockStatus != 1);
				}
			break;
		// F Block
		case eMFC:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_MFC);
			if(uBlockStatus == 0)	// Block Power Off
				{
					SYSC_SetBlockPowerGating(eSUB_BLOCK_MFC, true);
					do
					{
						uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_MFC);
					}
					while(uBlockStatus != 1);	
				}
			break;
		// 3D Block
		case eSGX540:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_G3D);
			if(uBlockStatus == 0)	// Block Power Off
			{
				SYSC_SetBlockPowerGating(eSUB_BLOCK_G3D, true);
				do
				{
					uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_G3D);
				}
				while(uBlockStatus != 1);		
			}
			break;
		// T Block
		case eVP:
		case eMIXER:
		case eHDMI:
		case eTVENC:
		case eHDTV:
		case eSDTV:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_TV);
			if(uBlockStatus == 0)	// Block Power Off
				{
					SYSC_SetBlockPowerGating(eSUB_BLOCK_TV, true);
					do
					{
						uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_TV);
					}
					while(uBlockStatus != 1);
				}
			break;
		case eCRYPTO:
			break;	

		case eMM_ALL:
			SYSC_SetBlockPowerGating(eSUB_BLOCK_ALL, true);
			break;
	}

}

void SYSC_SetBlockPowerOFF_IP(IP_eID eIP) // added by JWJ
{
	u32 uBlockStatus;

	switch(eIP)
	{
		// L Block
		case eFIMD:
		case eMDNIE:
		case eMIE:
		case eBLKLCD:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_LCD);
			if(uBlockStatus == 1)	// Block Power ON
				{
					SYSC_SetBlockPowerGating(eSUB_BLOCK_LCD, false);
					do
					{
					uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_LCD);
					}
					while(uBlockStatus != 0);
				}
			break;
		// X Block
		case eFIMC0:
		case eFIMC1:
		case eFIMC2:
		case eIPC:
		case eJPEG:
		case eROTATOR:
		case eBLKCAM:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_CAM);
			if(uBlockStatus == 1)	// Block Power ON
				{
					SYSC_SetBlockPowerGating(eSUB_BLOCK_CAM, false);
					do
					{
					uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_CAM);
					}
					while(uBlockStatus != 0);
				}
			break;
		// F Block
		case eMFC:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_MFC);
			if(uBlockStatus == 1)	// Block Power Off
				{
					SYSC_SetBlockPowerGating(eSUB_BLOCK_MFC, false);
					do
					{
						uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_MFC);
					}
					while(uBlockStatus != 0);	
				}
			break;
		// 3D Block
		case eSGX540:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_G3D);
			if(uBlockStatus == 1)	// Block Power ON
			{
				SYSC_SetBlockPowerGating(eSUB_BLOCK_G3D, false);
				do
				{
					uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_G3D);
				}
				while(uBlockStatus != 0);		
			}
			break;
		// T Block
		case eVP:
		case eMIXER:
		case eHDMI:
		case eTVENC:
		case eHDTV:
		case eSDTV:
			uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_TV);
			if(uBlockStatus == 1)	// Block Power ON
				{
					SYSC_SetBlockPowerGating(eSUB_BLOCK_TV, false);
					do
					{
						uBlockStatus = SYSC_GetBlockPowerStatus(eSUB_BLOCK_TV);
					}
					while(uBlockStatus != 0);
				}
			break;
		case eCRYPTO:
			break;	

		case eMM_ALL:
			SYSC_SetBlockPowerGating(eSUB_BLOCK_CAM, false);
			SYSC_SetBlockPowerGating(eSUB_BLOCK_TV, false);
			SYSC_SetBlockPowerGating(eSUB_BLOCK_LCD, false);
			SYSC_SetBlockPowerGating(eSUB_BLOCK_G3D, false);
			SYSC_SetBlockPowerGating(eSUB_BLOCK_MFC, false);
			break;
	}

}

void SYSC_SetClkON_IP(IP_eID eIP) // added by JWJ
{
	u32 uRegVal;
	//void SYSC_SetClkGatebyIP(CLKGATE_IP_TYPE eType, bool bPass)
	switch(eIP)
	{
		// L Block
		case eFIMD:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~(0x1<<5) |(0x1<<5);
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMD, true);	
			break;
		case eMDNIE:
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x3<<0) |(0x3<<0);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			// Clock Gate IP ?
			break;
		case eMIE:
			// Clock Src Mask ?
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIE, true);
			break;
		case eBLKLCD:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~(0x1<<5) |(0x1<<5);
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x3<<0) |(0x3<<0);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMD, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIE, true);
			break;
		// X Block
		case eFIMC0:		// It need added Item - CAM0
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1<<2 )|(0x1<<2);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC0, true);	
			break;
		case eFIMC1:
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1<<3 )|(0x1<<3);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC1, true);		
			break;
		case eFIMC2:
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1<<4 )|(0x1<<4);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC2, true);			
			break;
		case eIPC:
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1<<3 )|(0x1<<3);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_IPC, true);				
			break;
		case eJPEG:
			//
			SYSC_SetClkGatebyIP(eCLKGATE_IP_JPEG, true);		
			break;
		case eROTATOR:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_ROTATOR, true);	
			break;
		case eBLKCAM:
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x7<<2 )|(0x7<<2);
			syscOutp32(rCLK_SRC_MASK1 ,uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC0, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC1, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC2, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_IPC, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_JPEG, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_ROTATOR, true);
			break;
		// F Block
		case eMFC:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MFC, true);	
			break;
		// 3D Block
		case eSGX540:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_G3D, true);	
			break;
		// T Block
		case eVP:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_VP, true);
			break;
		case eMIXER:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~((0x1<<7)|(0x1<<1) )|((0x1<<7)|(0x1<<1));
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIXER, true);
			break;
		case eHDMI:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~((0x1<<7)|(0x3<<0) )|((0x1<<7)|(0x3<<0));
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_HDMI, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIDDC, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIPHY, true);
			break;
		case eTVENC:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~((0x1<<7)|(0x3<<1) )|((0x1<<7)|(0x3<<1));
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_TVENC, true);
			break;			
		case eHDTV:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~((0x1<<7)|(0x3<<0) )|((0x1<<7)|(0x3<<0));
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_VP, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIXER, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_HDMI, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIDDC, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIPHY, true);
			break;
		case eSDTV:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~((0x1<<7)|(0x3<<1) )|((0x1<<7)|(0x3<<1));
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_VP, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIXER, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_TVENC, true);
			break;
		case eCRYPTO:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_SECKEY, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_SECSS, true);
			break;	

		case eMM_ALL:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~((0x1<<7)|(0x1<<5)|(0x7<<0) )|((0x1<<7)|(0x1<<5)|(0x7<<0));
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1f<<0) |(0x1f<<0);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMD, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIE, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC0, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC1, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC2, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_IPC, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_JPEG, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_ROTATOR, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_VP, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIXER, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_HDMI, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIDDC, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIPHY, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_TVENC, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_SECKEY, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_SECSS, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MFC, true);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_G3D, true);
			
			break;
	}

}

void SYSC_SetClkOFF_IP(IP_eID eIP) // added by JWJ
{
	u32 uRegVal;
	//void SYSC_SetClkGatebyIP(CLKGATE_IP_TYPE eType, bool bPass)
	switch(eIP)
	{
		// L Block
		case eFIMD:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~(0x1<<5) |(0x0<<5);
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMD, false);	
			break;
		case eMDNIE:
			//uRegVal = syscInp32(rCLK_SRC_MASK1);
			//uRegVal = uRegVal &~(0x3<<0) |(0x3<<0);
			//syscOutp32(rCLK_SRC_MASK1, uRegVal);
			// Clock Gate IP ?
			break;
		case eMIE:
			// Clock Src Mask ?
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIE, false);
			break;
		case eBLKLCD:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~(0x1<<5) |(0x0<<5);
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x3<<0) |(0x0<<0);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMD, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIE, false);
			break;
		// X Block
		case eFIMC0:		// It need added Item - CAM0
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1<<2 )|(0x0<<2);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC0, false);	
			break;
		case eFIMC1:
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1<<3 )|(0x0<<3);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC1, false);		
			break;
		case eFIMC2:
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1<<4 )|(0x0<<4);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC2, false);			
			break;
		case eIPC:
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1<<3 )|(0x0<<3);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_IPC, false);				
			break;
		case eJPEG:
			//
			SYSC_SetClkGatebyIP(eCLKGATE_IP_JPEG, false);		
			break;
		case eROTATOR:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_ROTATOR, false);	
			break;
		case eBLKCAM:
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x7<<2 )|(0x0<<2);
			syscOutp32(rCLK_SRC_MASK1 ,uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC0, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC1, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC2, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_IPC, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_JPEG, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_ROTATOR, false);
			break;
		// F Block
		case eMFC:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MFC, false);	
			break;
		// 3D Block
		case eSGX540:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_G3D, false);	
			break;
		// T Block
		case eVP:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_VP, false);
			break;
		case eMIXER:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~(0x1<<1);
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIXER, false);
			break;
		case eHDMI:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~(0x1<<0);
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_HDMI, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIDDC, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIPHY, false);
			break;
		case eTVENC:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~(0x3<<1);
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_TVENC, false);
			break;			
		case eHDTV:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~((0x1<<7)|(0x3<<0) );
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_VP, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIXER, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_HDMI, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIDDC, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIPHY, false);
			break;
		case eSDTV:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~((0x1<<7)|(0x3<<1) );
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_VP, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIXER, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_TVENC, false);
			break;
		case eCRYPTO:
			SYSC_SetClkGatebyIP(eCLKGATE_IP_SECKEY, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_SECSS, false);
			break;	

		case eMM_ALL:
			uRegVal = syscInp32(rCLK_SRC_MASK0);
			uRegVal = uRegVal &~((0x1<<7)|(0x1<<5)|(0x7<<0) )|((0x0<<7)|(0x0<<5)|(0x0<<0));
			syscOutp32(rCLK_SRC_MASK0, uRegVal);
			uRegVal = syscInp32(rCLK_SRC_MASK1);
			uRegVal = uRegVal &~(0x1f<<0) |(0x0<<0);
			syscOutp32(rCLK_SRC_MASK1, uRegVal);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMD, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIE, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC0, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC1, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_FIMC2, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_IPC, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_JPEG, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_ROTATOR, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_VP, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MIXER, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_HDMI, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIDDC, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIPHY, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_TVENC, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_SECKEY, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_SECSS, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_MFC, false);
			SYSC_SetClkGatebyIP(eCLKGATE_IP_G3D, false);
			
			break;
	}

}

void SYSC_EnHDMIPHY (void)
{
	u32 uRegVal;

	uRegVal = syscInp32(rHDMI_CON);
	uRegVal |= (1<<0);
	syscOutp32(rHDMI_CON, uRegVal);
}

void SYSC_DisHDMIPHY (void)
{
	u32 uRegVal;

	uRegVal = syscInp32(rHDMI_CON);
	uRegVal = uRegVal & ~(1<<0);
	syscOutp32(rHDMI_CON, uRegVal);
}

void SYSC_SetDivHDMI(u32 uDiv)
{
	u32 uRegVal;

	uRegVal = syscInp32(rHDMI_CON);
	uRegVal = uRegVal & ~(0x3ff<<16) | (uDiv<<16);
	syscOutp32(rHDMI_CON, uRegVal);
	
}

void SYSC_EnDAC (void)
{
	u32 uRegVal;

	uRegVal = syscInp32(rDAC_CON);
	uRegVal |= (1<<0);
	syscOutp32(rDAC_CON, uRegVal);
}

void SYSC_DisDAC (void)
{
	u32 uRegVal;

	uRegVal = syscInp32(rDAC_CON);
	uRegVal = uRegVal & ~(1<<0);
	syscOutp32(rDAC_CON, uRegVal);
}

void SYSC_EnADC (void)
{
	u32 uRegVal;

	uRegVal = syscInp32(rADC_CON);
	uRegVal |= (1<<0);
	syscOutp32(rADC_CON, uRegVal);
}

void SYSC_DisADC (void)
{
	u32 uRegVal;

	uRegVal = syscInp32(rADC_CON);
	uRegVal = uRegVal & ~(1<<0);
	syscOutp32(rADC_CON, uRegVal);
}

void SYSC_UpdateClkInform_IP(void) 
{
	u32 uTemp;
	u32 uDivA2M;  // CLK_DIV0
	u32 uDivFIMD, uDivCAM0, uDivCAM1, uDivFIMC, uDivTBLK;	// CLK_DIV1
	u32 uDivMFC, uDivG3D;	// CLK_DIV2
	u32 uDivFIMC2LCLK, uDivFIMC1LCLK, uDivFIMC0LCLK, uDivMDNIEPWM, uDivMDNIE; //CLK_DIV3

	u32 uFIMDSEL, uCAM1SEL, uCAM2SEL, uDACSEL, uMIXERSEL, uHDMISEL;
	u32 uMFCSEL, uG3DSEL, uFIMC0LCLKSEL, uFIMC1LCLKSEL, uFIMC2LCLKSEL, uMDNIEPWMSEL, uMDNIESEL;
	u32 uMOUTG3D, uMOUTMFC;
	
	// JPEG : g_uHclkDsys
	// MIE   : Clock of the FIMD
	// IPC   : g_uSclkFIMC1LCLK
	// VP     : g_uHclkDsys
	// ROT   : g_uHclkDsys
	// CRYP : g_uHclkPsys
//volatile u32 , , g_uSclkMFC;
//volatile u32 g_uSclkFIMD, g_uSclkMDNIE, g_uSclkMDNIEPWM;
//volatile u32 g_uSclkFIMC, g_uSclkFIMC0LCLK, g_uSclkFIMC1LCLK, g_uSclkFIMC2LCLK, g_uSclkCAM0, g_uSclkCAM1;
//volatile u32 g_uSclkPIXEL, g_uSclkHDMIPHY, g_uSclkHDMI, g_uSclkMIXER, g_uSclkTVENC;
	
	SYSC_UpdateClkInform();

	// clock division ratio	
	uTemp = syscInp32(rCLK_DIV0);
	uDivA2M = (uTemp>>4) & 0x7;

	uTemp = syscInp32(rCLK_DIV1);
	uDivFIMD = (uTemp>>20) & 0xf;
	uDivCAM0 = (uTemp>>16) & 0xf;
	uDivCAM1 = (uTemp>>12) & 0xf;
	uDivFIMC = (uTemp>>8) & 0xf;
	uDivTBLK = (uTemp>>0) & 0xf;

	uTemp = syscInp32(rCLK_DIV2);
	uDivMFC = (uTemp>>4) & 0xf;
	uDivG3D = (uTemp>>0) & 0xf;

	uTemp = syscInp32(rCLK_DIV3);
	uDivFIMC2LCLK = (uTemp>>20) & 0xf;
	uDivFIMC1LCLK = (uTemp>>16) & 0xf;
	uDivFIMC0LCLK = (uTemp>>12) & 0xf;
	uDivMDNIEPWM = (uTemp>>4) & 0xf;
	uDivMDNIE = (uTemp>>0) & 0xf;

	// Clock Source Select
	uTemp = syscInp32(rCLK_SRC1);
	uFIMDSEL = (uTemp>>20) & 0xf;
	uCAM1SEL = (uTemp>>16) & 0xf;
	uCAM2SEL = (uTemp>>12) & 0xf;
	uDACSEL = (uTemp>>8) & 0x1;
	uMIXERSEL = (uTemp>>4) & 0x1;
	uHDMISEL = (uTemp>>0)&0x1;

	uTemp = syscInp32(rCLK_SRC2);
	uMFCSEL = (uTemp>>4) & 0x3;
	uG3DSEL = (uTemp>>0) & 0x3;
	
	uTemp = syscInp32(rCLK_SRC3);
	uFIMC2LCLKSEL = (uTemp>>20) & 0xf;
	uFIMC1LCLKSEL = (uTemp>>16) & 0xf;
	uFIMC0LCLKSEL = (uTemp>>12) & 0xf;
	uMDNIEPWMSEL = (uTemp>>4) & 0xf;
	uMDNIESEL = (uTemp>>0) & 0xf;

	//00. SCLKA2M 
	g_uSclkA2M = g_uAPLL/(uDivA2M+1);

	//01. g_uSclkG3D
	g_uSclkG3D = uMOUTG3D /(uDivG3D+1);
	


	
}


void SYSC_InitClkForMM(void)
{
	u32 uRegVal;

	SYSC_SetBlockPowerON_IP(eMM_ALL);
	SYSC_SetClkON_IP(eMM_ALL);

	// Set to EPLL/VPLL
	SYSC_SetClkMuxState(eCLKMUX_EPLL_FINPLL);
	SYSC_SetClkMuxState(eCLKMUX_VPLL_FINVPLL);
//	SYSC_StartPLLbyFout(eEPLL, 96000000);
//	SYSC_StartPLLbyFout(eVPLL, 54000000);
	SYSC_SetClkMuxState(eCLKMUX_EPLL_FOUTEPLL);
	SYSC_SetClkMuxState(eCLKMUX_VPLL_FOUTVPLL);

	// Set to HDMIPHY
	// HDMI_PHY_CON0 must be set to 0
	uRegVal = syscInp32(rHDMI_PHY_CON0);
	if(uRegVal != 0) syscOutp32(rHDMI_PHY_CON0, 0x0);

//rb1004
//	SYSC_SetDivHDMI(0x96);
//	SYSC_EnHDMIPHY();
		
	//update system clock information
	//g_uAPLL, g_uMPLL, g_uEPLL, g_uVPLL
	//g_uARMCLK, g_uHclkMsys, g_uPclkMsys
	//g_uHclkDsys, g_uPclkDsys, g_uHclkPsys, g_uPclkPsys
	SYSC_UpdateClkInform();

	//00. SCLKA2M : g_uAPLL /4,  A2M_RATIO(0x3),
	SYSC_SetClkDiv(eCLKDIV_A2M, 4);
	//01. g_uSclkG3D(g_uAPLL/4) : SCLKA2M, G3D_RATIO(0x0) 
	SYSC_SetClkDiv(eCLKDIV_G3D, 1);
	SYSC_SetClkMuxState(eCLKMUX_G3D_SCLKA2M);
	//02. g_uSclkMFC( g_uAPLL/4) : SCLKA2M, MFC_RATIO(0x0)
	SYSC_SetClkDiv(eCLKDIV_MFC, 1);
	SYSC_SetClkMuxState(eCLKMUX_MFC_SCLKA2M);
	//03. g_uSclkFIMD (g_uMPLL/4): SCLKMPLL, FIMD_RATIO(0x3)
	SYSC_SetClkDiv(eCLKDIV_FIMD, 4);
	SYSC_SetClkMuxState(eCLKMUX_FIMD_SMPLL);	
	//04. g_uSclkMDNIE (g_uMPLL/4): SCLKMPLL, MDNIE_RATIO(0x3)
	SYSC_SetClkDiv(eCLKDIV_MDNIE, 4);
	SYSC_SetClkMuxState(eCLKMUX_MDNIE_SMPLL);		
	//05. g_uSclkMDNIEPWM(g_uMPLL/10): SCLKMPLL, MDNIE_PWM_RATIO(0x9)
	SYSC_SetClkDiv(eCLKDIV_MDNIEPWM, 10);
	SYSC_SetClkMuxState(eCLKMUX_MDNIEPWM_SMPLL);
	//06. g_uSclkFIMC(g_uMPLL/4): MOUTDSYS(Default), FIMC_RATIO(0x3)
	SYSC_SetClkDiv(eCLKDIV_FIMC, 4);
	//07. g_uSclkFIMC[2:0]LCLK(g_uMPLL/4): SCLKMPLL, FIMC0_LCLK(0x3)
	SYSC_SetClkDiv(eCLKDIV_FIMC0, 4);
	SYSC_SetClkMuxState(eCLKMUX_FIMC0LCLK_SMPLL);	
	SYSC_SetClkDiv(eCLKDIV_FIMC1, 4);
	SYSC_SetClkMuxState(eCLKMUX_FIMC1LCLK_SMPLL);	
	SYSC_SetClkDiv(eCLKDIV_FIMC2, 4);
	SYSC_SetClkMuxState(eCLKMUX_FIMC2LCLK_SMPLL);	
	//08. g_uSclkCAM[1:0](g_uEPLL/2) : SCLKEPLL, CAM[1:0]_RATIO(0x1)
	SYSC_SetClkDiv(eCLKDIV_CAM0, 4); // 96[MHz] / 4 = 24[MHz]
	SYSC_SetClkMuxState(eCLKMUX_CAM0_SEPLL);	
	SYSC_SetClkDiv(eCLKDIV_CAM1, 4); // 96[MHz] / 4 = 24[MHz]
	SYSC_SetClkMuxState(eCLKMUX_CAM1_SEPLL);	

//rb1004...setting in phy.c
	//09. g_uSclkMIXER : SCLK_HDMI
//	SYSC_SetClkMuxState(eCLKMUX_MIXER_SHDMI);
	//10. g_uSclkHDMI  : SCLK_HDMIPHY
//	SYSC_SetClkMuxState(eCLKMUX_HDMI_SHDMIPHY);
	//11. g_uSclkPIXEL :
	SYSC_SetClkDiv(eCLKDIV_TBLK, 1);
	//12. g_uSclkTVENC: SCLK_VPLL
	SYSC_SetClkMuxState(eCLKMUX_DAC_SVPLL);

	//etc
	// JPEG : g_uHclkDsys
	// MIE   : Clock of the FIMD
	// IPC   : g_uSclkFIMC1LCLK
	// VP     : g_uHclkDsys
	// ROT   : g_uHclkDsys
	// CRYP : g_uHclkPsys
	
}


void SYSC_SetSleepMIE (u32 uSleep)
{
	u32 uRegVal;

	uRegVal = syscInp32(rMIE_CON);
	uRegVal = uRegVal & ~(0x1<<1)|(uSleep<<1);
	syscOutp32(rMIE_CON, uRegVal);
}

void SYSC_SetDispMIE (u32 uDisp)
{
	u32 uRegVal;

	uRegVal = syscInp32(rMIE_CON);
	uRegVal = uRegVal & ~(0x1<<0)|(uDisp<<0);
	syscOutp32(rMIE_CON, uRegVal);
}

void SYSC_SetDispPath(ePath_Disp eSel)
{
	u32 uTmpReg;

	uTmpReg = syscInp32(rDISPLAY_CONTROL);			// need to be changed later (09.01.23 WJ.Kim)	

	if(eSel == eDISP_MDNIE)
		uTmpReg = uTmpReg & ~(0x3) | 0x1 ;
	else if(eSel == eDISP_MIE)
		uTmpReg = uTmpReg & ~(0x3) | 0x0 ;
	else		//bypass
		uTmpReg = uTmpReg & ~(0x3) | 0x2 ;		

	syscOutp32(rDISPLAY_CONTROL, uTmpReg);
}

void SYSC_EnDPHY(DPHY_Block ePhyBlock, u8 bEn)
{
	u32 uTemp = syscInp32(rMIPI_PHY_CON) & ~(0x7);
	if (bEn)
	{
		uTemp |= ((u32) ePhyBlock & 0x7);
	}
	else
	{
		uTemp &= ~((u32) ePhyBlock & 0x7);
	}	
	syscOutp32(rMIPI_PHY_CON,uTemp);
}
#endif

