/**************************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : Sys_timer.c
*  
*	File Description : This file implements the API functons for System Timer
*
*	Author : chansik.Jeon
*	Dept. : AP Development Team
*	Created Date : 2009/02/11
*	Version : 0.0 
* 
*	History"
*	- Created(chansik.Jeon 2009/02/11)

**************************************************************************************/

#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "Sys_timer.h"
#include "intc.h"
#include "sysc.h"
#include "gpio.h"




// Interrupt Handler Global valiable

volatile u32 g_uIntCounter = 0;
volatile u32 g_uTICNTB = 0;
volatile u32 g_uTFCNTB = 0;
volatile u32 g_uICNTB = 0;
volatile u32 g_uTCON = 0;



//////////
// Function Name : SysTimer_UserSelect
// Function Description : This function set up Timers by User's Inputs
// Input :	uTimer [0~4]
//			uPrescaler [1~256]
//			uDivider [0:1/1	1:1/2	2:1/4	3:1/8	4:1/16	5:extCLK]
//			uTCNTB [1~2^31]
//			uTCMPB [1~2^31]
//			uEnDz, uEnDMA, uAutoreload, uEnInverter, uEnInt [0:Disable	1:Enbale]
//			uDzlen [0~255]
// Output : NONE
// Version : v0.1
void SysTimer_UserSelect(void)
{
	s32	sTimer;
	s32	sClock;
	s32	sTickGen_sel;
	s32	sFDIV_sel;
	s32	sPrescaler;
	s32  sDivider;
	s32  sAutoreload;
	s32 	sIntStart;
	s32  sTICNTB;
	s32  sTFCNTB;
	s32  sICNTB;

	float fTimerclk;
	float fTimerfreq;
	float fTickclk;
	float fTickfreq;
	float fIntclk;
	float fIntfreq;
	u32	uTemp0;
	s32  sContinue;
	u32 uOpClk;
	u32 i;

	GPIO_SetPullUpDownEach(eGPIO_H0, eGPIO_1, eGPUDdis); //Pull-up/down disable
	GPIO_SetDataEach(eGPIO_H0, eGPIO_1, eHigh); //High
	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_1, eGPO); // Output setting

	INTC_SetVectAddr(NUM_SYSTIMER,Isr_SysTimer);
	
	INTC_Enable(NUM_SYSTIMER);


	
	  
	//--------------------select clock input-----------------------	
		UART_Printf("\nSelect clock source\n ");
		UART_Printf(" 0: System Reference CLK(FIN),	   1: XrtcXTI,     2: XusbXTI,     3: PCLK  ");
		UART_Printf("[0~3] :");

		sClock = UART_GetIntNum();
		if(sClock == -1)
		{
			sClock = 0;
		}

		switch (sClock) {
			case 0:
				uOpClk = FIN;
				break;
			case 1:
				uOpClk = 32768;
				break;
			case 2:
				uOpClk = FIN;
				break;
			case 3:
				uOpClk = g_uPclkPsys;
				break;
			
			default:
			}

	//--------------------select divider-----------------------	
		UART_Printf("\nSelect Tick Generation\n ");
		UART_Printf(" 0: Integer divider,	   1: Fractional divider  ");
		UART_Printf("[0, 1] :");

		sTickGen_sel = UART_GetIntNum();
		if(sTickGen_sel == -1)
		{
			sTickGen_sel = 0;
		}

		if(sTickGen_sel==1)  //Fractional divider
	       {
			UART_Printf("\nSelect Fractional divider\n ");
			UART_Printf(" 0: General,	   1: RTC clock-dedicated to make 1ms  ");
			UART_Printf("[0, 1] :");

			sFDIV_sel = UART_GetIntNum();
			if(sFDIV_sel == -1)
			{
				sFDIV_sel = 0;
			}
	       }
		
		
	//--------------------select TICNTB-----------------------	
		UART_Printf("\nSelect TICNTB value  : ");
		UART_Printf("[1~0xffffffff] :");
		sTICNTB = UART_GetIntNum();
		if(sTICNTB == -1)
		{
			sTICNTB = 2000;
		}

      //--------------------select TFCNTB-----------------------  

	       if(sTickGen_sel==1)  //Fractional divider
	       {
			UART_Printf("\nSelect TFCNTB value  : ");
			UART_Printf("[1~0xffff] :");
			sTFCNTB = UART_GetIntNum();
			if(sTFCNTB == -1)
			{
				sTFCNTB = 2000;
			}
	       }

		if(sTickGen_sel==0)  //Integer divider
		{

		   //--------------------select dividerMUX---------------------	
			UART_Printf("\nSelect dividerMUX value \n ");
			UART_Printf("0: 1/1     1: 1/2     2: 1/4     3: 1/8     4: 1/16  \n");
			UART_Printf("[0~4] :");
			sDivider = UART_GetIntNum();
			if(sDivider== -1)
			{
				sDivider= 1;
			}

		  //--------------------select Prescaler---------------------	
			UART_Printf("\nSelect Prescaler value \n ");
			UART_Printf("[0~255] :");
			sPrescaler = UART_GetIntNum();
			if(sPrescaler == -1)
			{
				sPrescaler = 1;
			}
		}
		else if(sTickGen_sel==1)  //Fractional divider
		{
			UART_Printf("\n divider MUX & prescaler value should be '0' in Fractional divider \n ");
			sDivider =0;
			sPrescaler=0;

		}
	//--------------------select Interrupt Auto-reload---------------------		
		UART_Printf("\nSelect Interrupt Type\n");
		UART_Printf("0: One-shot mode    1: Interval mode (Auto-reload) \n");
		UART_Printf("[0, 1] :");
		sAutoreload = UART_GetIntNum();
		if(sAutoreload == -1)
		{
			sAutoreload = 1;
		}


	//--------------------select ICNTB-----------------------	
		UART_Printf("\nSelect ICNTB value  : ");
		UART_Printf("[1~0xffff] :");
		sICNTB = UART_GetIntNum();
		if(sICNTB == -1)
		{
			sICNTB = 2000;

		}


	//-----------------------Set SFRs--------------------------

		SysTimer_Reset();

		Int_Start();

		uTemp0 = Inp32(rSYS_TCFG);
		uTemp0 = (uTemp0 & ~(0xffffffff)) |((sFDIV_sel<<15)|(sTickGen_sel<<14)|(sClock<<12)|(sDivider<<8)|(sPrescaler<<0));
		Outp32(rSYS_TCFG,uTemp0);

		uTemp0 = Inp32(rSYS_TICNTB);
		uTemp0 = (uTemp0 & ~(0xffffffff)) |(sTICNTB);
		Outp32(rSYS_TICNTB,uTemp0);
		Check_Status(eTICNTB);
		
		uTemp0 = Inp32(rSYS_TFCNTB);
		uTemp0 = (uTemp0 & ~(0xffffffff)) |(sTFCNTB);
		Outp32(rSYS_TFCNTB,uTemp0);
		Check_Status(eTFCNTB);

		uTemp0 = Inp32(rSYS_ICNTB);
		uTemp0 = (uTemp0 & ~(0xffffffff)) |(sICNTB);
		Outp32(rSYS_ICNTB,uTemp0);
		Check_Status(eICNTB);

		uTemp0 = Inp32(rSYS_TCON);	//setting Autoreload
		uTemp0 = (uTemp0 & ~(1<<5))|(sAutoreload<<5);
		Outp32(rSYS_TCON,uTemp0);
		Check_Status(eTCON);

		Int_Manual_Update();
		Check_Status(eTCON);
			
		SysTimer_Start();
		Check_Status(eTCON);

		GPIO_SetDataEach(eGPIO_H0, eGPIO_1, eLow); //Low
	
		if(sTickGen_sel==0)  //Integer divider
		{
			fTimerfreq = (float)uOpClk/(sPrescaler+1)/(1<<(sDivider));			
			fTimerclk = 1/fTimerfreq;
				
			UART_Printf("OpCLK = %d , sPrescaler = %d, sDivider = %d dd= %d\n",uOpClk,sPrescaler,sDivider,(1<<(sDivider)));
			UART_Printf("Timer Clock = %f sec , Timer Frequency = %f hz\n\n",fTimerclk,fTimerfreq);
			
			fTickfreq = (float)fTimerfreq/(sTICNTB+1);			
			fTickclk = 1/fTickfreq;
			
			fIntfreq = (float)fTickfreq/(sICNTB+1); 		
			fIntclk = 1/fIntfreq;
			
			UART_Printf("TICNTB = %d , sICNTB = %d\n",sTICNTB,sICNTB);		
			UART_Printf("Tick Clock = %f sec , Tick Frequency = %f Hz\n\n",fTickclk,fTickfreq);
			UART_Printf("Int Clock = %f sec , Int Frequency = %f Hz\n\n",fIntclk,fIntfreq);
		}
		
		else if((sTickGen_sel==1) && (sFDIV_sel==0))  //Fractional divider
		{
			fTimerfreq = (float)uOpClk;			
			fTimerclk = 1/fTimerfreq;
				
			UART_Printf("OpCLK = %d , Timer Clock = %f sec , Timer Frequency = %f hz\n\n", uOpClk, fTimerclk,fTimerfreq);
					
			fTickfreq = (float)fTimerfreq/2 /((float)((sTICNTB+1)+(float)sTFCNTB/(float)65536));			
			fTickclk = 1/fTickfreq;
			
			fIntfreq = (float)fTickfreq/(sICNTB+1); 		
			fIntclk = 1/fIntfreq;
			
			UART_Printf("TICNTB = %d , sICNTB = %d\n",sTICNTB,sICNTB);		
			UART_Printf("Tick Clock = %f sec , Tick Frequency = %f Hz\n\n",fTickclk,fTickfreq);
			UART_Printf("Int Clock = %f sec , Int Frequency = %f Hz\n\n",fIntclk,fIntfreq);
		}
		
			
}




void SysTimer_Config(u32 uFDIV_sel, u32 uTickGen_sel, u32 uClock, u32 uDivider, u32 uPrescaler, 
	                                 u32 uTICNTB, u32 uTFCNTB, u32 uICNTB, u32 uAutoreload)
{
	u32 uTemp;
	
	SysTimer_Reset();

	Int_Start();

	uTemp = Inp32(rSYS_TCFG);
	uTemp = (uTemp & ~(0xffffffff)) |((uFDIV_sel<<15)|(uTickGen_sel<<14)|(uClock<<12)|(uDivider<<8)|(uPrescaler<<0));
	Outp32(rSYS_TCFG,uTemp);

	uTemp = Inp32(rSYS_TICNTB);
	uTemp = (uTemp & ~(0xffffffff)) |(uTICNTB);
	Outp32(rSYS_TICNTB,uTemp);
	Check_Status(eTICNTB);
	
	uTemp = Inp32(rSYS_TFCNTB);
	uTemp = (uTemp & ~(0xffffffff)) |(uTFCNTB);
	Outp32(rSYS_TFCNTB,uTemp);
	Check_Status(eTFCNTB);

	uTemp = Inp32(rSYS_ICNTB);
	uTemp = (uTemp & ~(0xffffffff)) |(uICNTB);
	Outp32(rSYS_ICNTB,uTemp);
	Check_Status(eICNTB);

	Int_Manual_Update();
	Check_Status(eTCON);
		
	uTemp = Inp32(rSYS_TCON);	//setting Autoreload
	uTemp = (uTemp & ~(1<<5))|(uAutoreload<<5);
	Outp32(rSYS_TCON,uTemp);
	Check_Status(eTCON);

}
//////////
// Function Name : Check_Status
// Function Description : This function represents status of system timer
// Input : write interrupt status bit
// Output : NONE
// Version : v0.0
void Check_Status(Stimer_eStatus eBitPos)
{
	u32 uTemp;

       while(1)
       {
		uTemp = Inp32(rSYS_INT_CSTAT);
		if( uTemp & (1<<eBitPos)) break;
       }

	 uTemp|=(1<<eBitPos);
	 
	Outp32(rSYS_INT_CSTAT,uTemp);
}


//////////
// Function Name : Int_Control
// Function Description : This function controls the Interrupt of system timer
// Input : Interrupt bit
//            Enable / Disable
// Output : NONE
// Version : v0.0
void Int_Control(Stimer_eInt eBitPos, Int_eOnOff eOnOff )
{
	u32 uTemp;

     	uTemp = Inp32(rSYS_INT_CSTAT);
	uTemp = uTemp & ~ (1<<eBitPos)|(eOnOff<<eBitPos);
    
	Outp32(rSYS_INT_CSTAT, uTemp);
}



//////////
// Function Name : SysTimer_reset
// Function Description : This function resets system timer
// Input : NONE
// Output : NONE
// Version : v0.0
void SysTimer_Reset(void)
{
	u32 uTemp;

	uTemp = Inp32(rSYS_TCFG);
	uTemp = (uTemp & ~(0x1<<16))|(0x1<<16);
		
	Outp32(rSYS_TCFG,uTemp);
}




//////////
// Function Name : SysTimer_Start
// Function Description : This function starts system timer
// Input : NONE
// Output : NONE
// Version : v0.0
void SysTimer_Start(void)
{
	u32 uTemp;

	uTemp = Inp32(rSYS_TCON);
	uTemp = ( uTemp & ~((0x1<<3)|(0x1<<0)))|(0x1<<3)|(0x1<<0);
		
	Outp32(rSYS_TCON,uTemp);
}


//////////
// Function Name : SysTimer_Stop
// Function Description : This function stops system timer
// Input : NONE
// Output : NONE
// Version : v0.0
void SysTimer_Stop(void)
{
	u32 uTemp;

	uTemp = Inp32(rSYS_TCON);
	uTemp &= ~(0x1<<0);
	Outp32(rSYS_TCON,uTemp);
}


//////////
// Function Name : Int_Start
// Function Description : This function starts Interrupt
// Input : NONE
// Output : NONE
// Version : v0.0
void Int_Start(void)
{
	u32 uTemp0, uTemp1;


	uTemp0 = Inp32(rSYS_INT_CSTAT);
	uTemp0 = (uTemp0 & ~((0x1<<6)|(0x1<<0)))|(0x1<<6)|(0x1<<0);
//	uTemp0 = uTemp0 | (0xf<<7);
	Outp32(rSYS_INT_CSTAT,uTemp0);
	
/*
	uTemp1 = Inp32(rSYS_TCON);
	uTemp1 = (uTemp1 & ~(0x1<<3))|(0x1<<3);
		
	Outp32(rSYS_TCON,uTemp1);
	*/
}


//////////
// Function Name : SysTimer_Stop
// Function Description : This function stops Interrupt
// Input : NONE
// Output : NONE
// Version : v0.0
void Int_Stop(void)
{
	u32 uTemp;

	uTemp = Inp32(rSYS_TCON);
	uTemp &= ~(0x1<<3);
		
	Outp32(rSYS_TCON,uTemp);
}




//////////
// Function Name : Int_Manual_Update
// Function Description : Interrupt manual update
// Input : NONE
// Output : NONE
// Version : v0.0
void Int_Manual_Update(void)
{
	u32 uTemp;

	uTemp = Inp32(rSYS_TCON);
	uTemp = (uTemp &~(0x1<<4))|(0x1<<4);
	Outp32(rSYS_TCON,uTemp);



//	uTemp = Inp32(rSYS_TCON);
//	uTemp &= ~(0x1<<4);
//	Outp32(rSYS_TCON,uTemp);
}



//////////
// Function Name : SysTimer_ReadCount
// Function Description : This function read out TICNTO & ICNTO value
// Input : None
// Output : TICNTO & ICNTO value
// Version : v0.0
u32 SysTimer_ReadCount(void)
{
	u32 uTemp0, uTemp1;

	while(1)
	{
		uTemp0 = Inp32(rSYS_TICNTO);	// Tick Integer count
		uTemp1 = Inp32(rSYS_ICNTO);		// Interrupt Count
				
		UART_Printf("Tick Integer Count = 0x%x  , Interrupt Count = 0x%x \n\n",uTemp0, uTemp1);
		if(UART_GetKey()=='x') break;
	}

}



//////////
// Function Name : SysTimer_ChangeIntervalINT
// Function Description : This function change the Interval interrupt at run-time
// Input : None
// Output : None
// Version : v0.0
u32 SysTimer_ChangeIntervalINT(void)
{
	
	s32  sICNTB;
	u32 uTemp0, uTemp1;

	//--------------------select ICNTB-----------------------	
			UART_Printf("\nSelect ICNTB value  : ");
			UART_Printf("[1~0xffff] :");
			sICNTB = UART_GetIntNum();
			if(sICNTB == -1)
			{
				sICNTB = 2000;

			}

			uTemp0 = Inp32(rSYS_ICNTB);
			uTemp0 = (uTemp0 & ~(0xffffffff)) |(sICNTB);
			Outp32(rSYS_ICNTB,uTemp0);
			Check_Status(eICNTB);


			Int_Manual_Update();
			Check_Status(eTCON);

}


u32 SysTimer_ChangeIntervalINT_autoreload(void)
{
	s32  sICNTB;
	u32 uTemp0, uTemp1;

	//--------------------select ICNTB-----------------------	
			UART_Printf("\nSelect ICNTB value  : ");
			UART_Printf("[1~0xffff] :");
			sICNTB = UART_GetIntNum();
			if(sICNTB == -1)
			{
				sICNTB = 2000;

			}

			uTemp0 = Inp32(rSYS_ICNTB);
			uTemp0 = (uTemp0 & ~(0xffffffff)) |(sICNTB);
			Outp32(rSYS_ICNTB,uTemp0);
			Check_Status(eICNTB);
}






//////////
// Function Name : Isr_SysTimer
// Function Description : This function is Interrupt Service Routine of System Timer
// Input : NONE
// Output : NONE (increment of g_uIntCounter value)
// Version : v0.0

void __irq Isr_SysTimer(void)
{
	u32 uTmp0;

//	UART_Printf("\nIsr_systimer \n ");
//	GPIO_SetPullUpDownEach(eGPIO_D0, eGPIO_0, eGPUDdis); //Pull-up/down disable
//	GPIO_SetFunctionEach(eGPIO_D0, eGPIO_0, eGPO); // Output setting

	if((g_uIntCounter&0x1) ==0)
       {
       	GPIO_SetDataEach(eGPIO_H0, eGPIO_1, eHigh); //High	
	}
	else
	{
		GPIO_SetDataEach(eGPIO_H0, eGPIO_1, eLow); //Low
	}

	uTmp0 = Inp32(rSYS_INT_CSTAT);
	
	if (uTmp0 & (1<<1))
	{
		uTmp0|=(1<<1);	      //  Interrupt count expired interrupt
		g_uIntCounter++;
	}
	
	else if (uTmp0 & (1<<2))
	{
		uTmp0|=(1<<2);	      //  TICNTB write interrupt
		g_uTICNTB=1;
		UART_Printf("Isr_TICNTB \n ");
	}
	else if (uTmp0 & (1<<3))
	{
		uTmp0|=(1<<3);	      //  TFCNTB write interrupt
		g_uTFCNTB=1;
		UART_Printf("Isr_TFCNTB \n ");
	}
	else if (uTmp0 & (1<<4))
	{
		uTmp0|=(1<<4);	      //  ICNTB write interrupt
		g_uICNTB=1;
		UART_Printf("Isr_ICNTB \n ");
	}
	else if (uTmp0 & (1<<5))
	{
		uTmp0|=(1<<5);	      //  TCON write interrupt
		g_uTCON=1;
		UART_Printf("Isr_TCON \n ");
	}
	
		
	Outp32(rSYS_INT_CSTAT,uTmp0);       //  Timer0 Interrupt Clear register	

	  	INTC_ClearVectAddr();

}


