/********************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2006 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for verifying functions of the S5PC100.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------
* 
*	File Name : dma.c
*  
*	File Description :
*
*	Author	: cha gyu hwan
*	Dept. : AP Development Team
*	Created Date : 2008/10/17
*	Version : 0.1 
* 
*	History 
*	
*	
*  
********************************************************************************/


#include "system.h"
#include "def.h"
#include "option.h"
#include "library.h"
#include "v210_sfr.h"


//#include "sfr_base.h"
#include "dma.h"


/*
struct DMAC
{
	u32 m_uBaseAddr;
	u32 m_uChAddr;
	u32 m_uIntSrc;
	u32 m_uChNum;
	u32 m_uCodeStAddr;
	u8*  m_pCodeBuf;
	u32 m_uCodeOffset;
	u32 m_uSecondLoopEndOffset;
	u32 m_uSrcOffsetAddr;
	u32 m_uDstOffsetAddr;
	u32 m_uLoopCntPtr0,m_uLoopCntPtr1,m_uLoopCntPtr2;
	u32 m_uFirstLoopEndOffset;
	
	u8 m_bFiniteLoop;
	u32 m_uLoopStOffset[2];

	//DMA_CTR_REG m_oCtrReg;

	DREQ_SRC m_eDreqSrc;
	DATA_SIZE m_eDataSz;
	DMA_TR_DIR m_eDir;
	u32 m_uBurstSz;
	u8 m_uDreqSrc; // 0~31
	u32 m_uProtectionBit;
};
*/

struct DMA_CTR_REG
{
	unsigned uSI		:1; //[0] Transfer size not count
	unsigned uSBSize	:3 ; //[3:1] Source 1 transfer size
	unsigned uSWidth	:4 ; //[7:4] Sourse burst len
	unsigned uSProt		:3 ; //[10:8] Source Protection set 101b=5
	unsigned uSCache	:3 ; //[13:11] Source Cache control
	unsigned uDI 		:1 ; //[14] Destination increment
	unsigned uDBSize	:3 ; //[17:15] Destination 1 transfer size
	unsigned uDWidth	:4 ; //[21:18] Destination burst len
	unsigned uDProt		:3 ; //[24:22] Source Protection set 101b=5
	unsigned uDCache	:3 ; //[27:25] Source Cache control	
	unsigned uESSize	:4 ; //[31:28] endian_swap_size
}m_oCtrReg;

enum DMA_REG
{
	DMA_DS = 0x000, // RO DMA status
		// [9] DNS (0-operate in secure mode, 1-operate in non-secure mode)
		// [8:4] Wakeup event (0-irq0, ..., 31-irq31) (check DMA_CR0's num_events)
		// (when executing DMAWFE instruction, it signals the above interrupt HIGH.)
		// [3:0] DMA status (0-stopped, 1-executing, 2-Cache miss, 3-Updating PC, 4-Waiting for event, 0xf-faulting)
	DMA_DPC = 0x004, // RO DMA Program Counter
		// when boot_from_pc is tied to HIGH, DMA_DPC is updated by boot_addr[31:0].
	DMA_INTEN = 0x020, // RW Interrupt Enable
		// [N] (0 - If executing DMASEV for event N then the DMAC signals event N to all of the threads,
		// 1 - If executing DMASEV for event N then the DMAC sets irq[N] HIGH)
	DMA_ES = 0x024, // [31:0] RO Event Status
		// [N] (0 - Event N is inactive or irq[N] is LOW, 1 - Event N is active or irq[N] is HIGH)
	DMA_INTSTATUS = 0x028, // RO Interrupt Status
		// [N] (0 - Event N is inactive or irq[N] is LOW, 1 - Event N is active or irq[N] is HIGH)
	DMA_INTCLR = 0x02c, // WO Interrupt Clear
		// [N] (0 - The status of irq[N] does not change, 1 - DMAC sets irq[N] LOW if INTEN is enabled, otherwise no change)
	DMA_FSM = 0x030, // RO Fault Status DMA Manager
		// [0] fs_mgr (0-DMA manager thread is not in the Faulting state 1- is in the Faulting state)
	DMA_FSC = 0x034, // RO Fault Status DMA Channel
		// [N] (0 - No fault is present on DMA ch N, 1 - DMA ch N is in the Faulting state)
	DMA_FTM = 0x038, // RO Fault Type DMA Manager
		// [30] dbg_inst -
		// [16] instr_fetch_err
		// [5] mgr_event_err
		// [4] dmago_err
		// [1] operand_invalid
		// [0] undef_instr
	DMA_FTC0 = 0x040, // RO Fault Type for DMA ch 0
	DMA_FTC1 = 0x044, // RO Fault Type for DMA ch 1
	DMA_FTC2 = 0x048, // RO Fault Type for DMA ch 2
	DMA_FTC3 = 0x04c, // RO Fault Type for DMA ch 3
	DMA_FTC4 = 0x050, // RO Fault Type for DMA ch 4
	DMA_FTC5 = 0x054, // RO Fault Type for DMA ch 5
	DMA_FTC6 = 0x058, // RO Fault Type for DMA ch 6
	DMA_FTC7 = 0x05c, // RO Fault Type for DMA ch 7
		// [31] lockup_err
		// [30] dbg_inst
		// [18] data_read_err
		// [17] data_write_err
		// [16] instr_fetch_err
		// [12] mfifo_err
		// [7] ch_rdwr_err
		// [6] ch_periph_err
		// [5] ch_event_err
		// [1] operand_invalid
		// [0] undef_instr
	DMA_CS_BASE = 0x100, // RO Channel Status for DMA ch 0
		// [21] CNS 0-DMA ch operates in the Secure state, 1- in the Non-secure state
		// [15] dmawfp_periph 0 - DMAWFP executed with the p operand not set, 1- with the p operand set
		// [14] dmawfp_b_ns 0 - DMAWFP executed with the s operand set, 1 - with the b operand set
		// [8:4] wakeup number 0 - DMA channel is waiting for event 0, ..., 31 - waiting for event 31
		// [3:0] channel status
	DMA_CPC_BASE = 0x104, // RO Channel Program Counter for DMA ch 0
	DMA_CH0_REG = 0x400,
	DMA_CH1_REG = 0x420,
	DMA_CH2_REG = 0x440,
	DMA_CH3_REG = 0x460,
	DMA_CH4_REG = 0x480,
	DMA_CH5_REG = 0x4A0,
	DMA_CH6_REG = 0x4C0,
	DMA_CH7_REG = 0x4E0,
	DMA_DBGSTATUS = 0xd00, // RO Debug Status
		// [0] dbgstatus 0-Idle, 1-Busy
	DMA_DBGCMD = 0xd04, // WO Debug Command
		// [1:0] dbgcmd 0-execute the instruction that the DBGINST0,1 registers contain
	DMA_DBGINST0 = 0xd08, // WO Debug Instruction 0
		// [31:24] Instruction byte 1
		// [23:16] Instruction byte 0
		// [10:8] Channel number (0~7)
		// [0] Debug thread 0-DMA manager thread, 1-DMA channel (when set to 1, [10:8] will be used
	DMA_DBGINST1 = 0xd0c, // WO Debug Instruction 1
		// [31:24] Instruction byte 5
		// [23:16] Instruction byte 4
		// [15:8] Instruction byte 3
		// [7:0] Instruction byte 2
	DMA_CR0 = 0xe00, // RO Configuration 0
		// [21:17] num_events (0-1 interrupt output,irq[0], ..., 31-32 interrupt output,irq[31:0])
		// [16:12] num_periph_req (0-1 peripheral request i/f, ..., 31-32 peripheral request i/f)
		// [6:4] num_chnls (0-1 DMA ch, ..., 7-8 DMA chs)
		// [2] mgr_ns_at_rst (0-boot_manager_ns was LOW, 1-HIGH)
		// [1] boot_en (0-boot_from_pc was LOW, 1-HIGH)
		// [0] periph_req (0-DMAC does not provide a peripheral request i/f, 1- provide it with num_periph_req)
	DMA_CR1 = 0xe04, // RO Configuration 1
		// [7:4] num_i-cache_lines (0-1 i-cache line, ..., 15-16 i-cache lines)
		// [2:0] i-cache_len (2-4 bytes, 3-8 bytes, 4-16 bytes, 5-32 bytes)
	DMA_CR2 = 0xe08, // RO Configuration 2
		// [31:0] boot_addr Provide the value of boot_addr[31:0] when the DMAC exited from reset
	DMA_CR3 = 0xe0c, // RO Configuration 3
		// [N:0] INS Provide the security state of the interrupt outputs
		// 0-Assign irq[N] to the Secure state
		// 1-Assign irq[N] to the Non-secure state
		// boot_irq_ns input pin
	DMA_CR4 = 0xe10, // RO Configuration 4
		// [N:0] PNS Provide the security state of the peripheral request interface
		// 0-Assign peripheral request interface N to the Secure state
		// 1-Assign peripheral request interface N to the Non-secure state
		// boot_periph_ns input pin
	DMA_CRDn = 0xe14, // RO Configuration Dn
		// [29:20] data_buffer_dep 0-1 line, ..., 1023-1024 lines
		// [19:16] depth of the read queue 0-1 line, ..., 15-16 lines
		// [14:12] read issuing capability 0-1, ..., 7-8
		// [11:8] depth of the write queue 0-1 line, ..., 15-16 lines
		// [6:4] write issuing capability 0-1, ..., 7-8
		// [2:0] data width on AXI bus 2-32bit, 3-64bit, 4-128bit

	// [ Peripheral Identification ]
		// [24] integration config
		// [19:12] designer's ID - 0x41 for ARM
		// [11:0] part number - 0x330 for the DMAC
	DMA_PERI_ID0 = 0xfe0, //[7:0] RO Peripheral ID 0
		// [7:0] part number 0 set to 0x30
	DMA_PERI_ID1 = 0xfe4, //[7:0] RO Peripheral ID 1
		// [7:4] designer 0 set to 0x1
		// [3:0] part number 1 set to 0x3
	DMA_PERI_ID2 = 0xfe8, //[7:0] RO Peripheral ID 2
		// [7:4] revision 0x0 for r0p0
		// [3:0] designer 1 set to 0x4
	DMA_PERI_ID3 = 0xfec, //[0] RO Peripheral ID 3
		// [0] integration config

	// [ PrimeCell Identification ]
	DMA_PCELL_ID0 = 0xff0, //[7:0] RO PrimeCell ID 0
		// [7:0] 0x0d
	DMA_PCELL_ID1 = 0xff4, //[7:0] RO PrimeCell ID 1
		// [7:0] 0xf0
	DMA_PCELL_ID2 = 0xff8, //[7:0] RO PrimeCell ID 2
		// [7:0] 0x05
	DMA_PCELL_ID3 = 0xffc //[7:0] RO PrimeCell ID 3
		// [7:0] 0xb1
};


enum DMA_CH_REG
{
	DMA_CH_SRCADDR = 0x00, //[31:0] RO Source Address for DMA ch n(0~7)
	DMA_CH_DSTADDR = 0x04, //[31:0] RO Destination Address for DMA ch n(0~7)
	DMA_CH_CONTROL = 0x08, //[31:0] RO Channel Control for DMA ch n(0~7)
		// [31:28] endian_swap_size 0-No swap, 1-swap bytes between 16bits, 2-32bits, 3-64bits, 4-128bits
		// [27:25] dst_cache_ctrl
		// [24:22] dst_prot_ctrl
		// [21:18] dst_burst_len 0-1 data transfer, ..., 15-16 data transfer
		// [17:15] dst_burst_size 0-1byte, 1-2bytes, 2-4bytes, 3-8bytes, 4-16bytes, ..., 7-128bytes
		// [14] dst_inc 0-fixed addr burst, 1-incrementing addr burst
		// [13:11] src_cache_ctrl
		// [10:8] src_prot_ctrl
		// [7:4] src_burst_len 0-1 data transfer, ..., 15-16 data transfer
		// [3:1] src_burst_size 0-1byte, 1-2bytes, 2-4bytes, 3-8bytes, 4-16bytes, ..., 7-128bytes
		// [0] src_inc 0-fixed addr burst, 1-incrementing addr burst
	DMA_CH_LoopCount0 = 0x0c, //[7:0] RO Loop counter 0 for DMA ch n(0~7)
		// This reg is updated when DMALPEND is executed.
	DMA_CH_LoopCount1 = 0x10 //[7:0] RO Loop counter 1 for DMA ch n(0~7)
		// This reg is updated when DMALPEND is executed.
};


#ifdef DBG_DMAC
#define DbgDma(x) Dbg x
#else
#define DbgDma(x) 0;
#endif


#define memOutp32(offset, x) Outp32(offset, x)
#define memOutp8(offset, x) Outp8(offset, x)

#define dmaOutp32(offset, x) Outp32(m_uBaseAddr+offset, x)
#define dmaInp32(offset, x) Inp32(m_uBaseAddr+offset, x)

#define dmaChOutp32(offset, x) Outp32(m_uBaseAddr+m_uChAddr+offset, x)
#define dmaChInp32(offset, x) Inp32(m_uBaseAddr+m_uChAddr+offset, x)

#define dmaDispVector(x) 0
#define WRITEWORD(addr, d) \
	((*(volatile u8*)(addr)=(u8)(d&0xff)), (*(volatile u8*)(addr+1)=(u8)((d>>8)&0xff)), \
	(*(volatile u8*)(addr+2)=(u8)((d>>16)&0xff)), (*(volatile u8*)(addr+3)=(u8)((d>>24)&0xff)))

void DMA_SetCh(DMA_CH eCh ,DMAC *sCh)
{
	u32 x;
	u32 uCh;
	
	switch( eCh )
	{
		case DMA_00:case DMA_01:case DMA_02:case DMA_03:
		case DMA_04:case DMA_05:case DMA_06:case DMA_07:
			sCh->m_uBaseAddr = PDMA0_BASE;//m_uIntSrc=INT_DMA_A;
			break;
		case DMA_10:case DMA_11:case DMA_12:case DMA_13:
		case DMA_14:case DMA_15:case DMA_16:case DMA_17:
			sCh->m_uBaseAddr = PDMA1_BASE;//m_uIntSrc=INT_DMA_B;
			break;
		case DMA_20:case DMA_21:case DMA_22:case DMA_23:
		case DMA_24:case DMA_25:case DMA_26:case DMA_27:
			sCh->m_uBaseAddr = SMDMA_BASE;//m_uIntSrc=INT_MDMA;
			break;
//		default : Assert(0);
	}

	switch( eCh )
	{
		case DMA_00:case DMA_10:case DMA_20:
			sCh->m_uChAddr = DMA_CH0_REG; sCh->m_uChNum = 0; x = 1<<0; break;
		case DMA_01:case DMA_11:case DMA_21:
			sCh->m_uChAddr = DMA_CH1_REG; sCh->m_uChNum = 1; x = 1<<1; break;
		case DMA_02:case DMA_12:case DMA_22:
			sCh->m_uChAddr = DMA_CH2_REG; sCh->m_uChNum = 2; x = 1<<2; break;
		case DMA_03:case DMA_13:case DMA_23:
			sCh->m_uChAddr = DMA_CH3_REG; sCh->m_uChNum = 3; x = 1<<3; break;
		case DMA_04:case DMA_14:case DMA_24:
			sCh->m_uChAddr = DMA_CH4_REG; sCh->m_uChNum = 4; x = 1<<4; break;
		case DMA_05:case DMA_15:case DMA_25:
			sCh->m_uChAddr = DMA_CH5_REG; sCh->m_uChNum = 5; x = 1<<5; break;
		case DMA_06:case DMA_16:case DMA_26:
			sCh->m_uChAddr = DMA_CH6_REG; sCh->m_uChNum = 6; x = 1<<6; break;
		case DMA_07:case DMA_17:case DMA_27:
			sCh->m_uChAddr = DMA_CH7_REG; sCh->m_uChNum = 7; x = 1<<7; break;

	}

	uCh = (sCh->m_uBaseAddr==PDMA0_BASE) ? 0 :
		(sCh->m_uBaseAddr==PDMA1_BASE) ? 8 : 16;
	uCh += sCh->m_uChNum;
	sCh->m_uCodeStAddr = DMAC_CODE_BASE+uCh*(DMAC_CODE_SIZE/24);

#ifndef DMAC_NS_MODE
	if(sCh->m_uBaseAddr==SMDMA_BASE)
	{
		sCh->m_uProtectionBit = 0;
	}
	else
	{
		sCh->m_uProtectionBit = 2;
	}
#else
	sCh->m_uProtectionBit = 2;
#endif
}

void DMA_InitCh(DATA_SIZE eDataSz, DREQ_SRC eSrc, DMA_TR_DIR eDir, BURST_MODE eBurst, DMAC *sCh)
{
	u32 sfr;
	u32 uSrcAhb = 0;//((APB_ST<=uSrcAddr)&&(uSrcAddr<=APB_LMT)) ? 1: 0;
	u32 uDstAhb = 1;//((APB_ST<=uDstAddr)&&(uDstAddr<=APB_LMT)) ? 1: 0;
	u32 uNextLliAhb = 0;

	u32 uIsSrcInc = (eDir==DMA_P2M) ? 0 : 1; // 0: p2m or p2p (fixed), 1: m2m or m2p (incremental)
	u32 uIsDstInc = (eDir==DMA_M2P) ? 0 : 1; // 0: m2p, p2p (fixed), 1: m2m, p2m (incremental)

	u32 uBWidth;
	u32 uDataSz;

	if(eSrc == SOFTWARE) // for mem to mem by P2M ,M2P 
	{
		uIsSrcInc =1; //incremental
		uIsDstInc =1; //incremental
	}

	/* Special Case : MODEM I/F */
	if( eSrc == MSM_REQ0 || eSrc == MSM_REQ1 || eSrc == MSM_REQ2 || eSrc == MSM_REQ3 ) // for mem to mem by P2M ,M2P 
	{
		uIsSrcInc = 1; //incremental
		uIsDstInc = 1; //incremental
	}
	
	sCh->m_uCodeOffset = 0;

	sCh->m_eDataSz = eDataSz;
	sCh->m_eDir= eDir;
	sCh->m_eDreqSrc = eSrc;

		
	sfr = Inp32(sCh->m_uBaseAddr+DMA_INTEN);
	
	Outp32(sCh->m_uBaseAddr+DMA_INTEN, sfr);
	Outp32(sCh->m_uBaseAddr+DMA_INTEN, sfr|(1<<sCh->m_uChNum));
	
	switch(eBurst) // Source burst length 1~16
	{
		case SINGLE : sCh->m_uBurstSz = 0x1;uBWidth=0x0;break;
		case BURST4 : sCh->m_uBurstSz = 0x4;uBWidth=0x3;break;
		case BURST8 : sCh->m_uBurstSz = 0x8;uBWidth=0x7;break;
		case BURST16 : sCh->m_uBurstSz = 0x10;uBWidth=0xf;break;
		
		default : //UART_Printf(0);
	}

	switch(eDataSz) // Source data size 1~8
	{
		case BYTE  : sCh->m_eDataSz = 0x1;uDataSz=0x0;break;
		case HWORD : sCh->m_eDataSz = 0x2;uDataSz=0x1;break;
		case WORD  : sCh->m_eDataSz = 0x4;uDataSz=0x2;break;
		case DWORD : sCh->m_eDataSz = 0x8;uDataSz=0x3;break;
		
		default : //UART_Printf(0);
	}

	
			  
	sCh->m_uDreqSrc = eSrc;
	m_oCtrReg.uSI = uIsSrcInc;
	m_oCtrReg.uSBSize = uDataSz;
	m_oCtrReg.uSWidth = uBWidth;
	m_oCtrReg.uSProt = sCh->m_uProtectionBit;
	m_oCtrReg.uSCache = 0;
	m_oCtrReg.uDI = uIsDstInc;
	m_oCtrReg.uDBSize = uDataSz;
	m_oCtrReg.uDWidth = uBWidth;
	m_oCtrReg.uDProt = sCh->m_uProtectionBit;
	m_oCtrReg.uDCache = 0;
	m_oCtrReg.uESSize = 0;

	DMA_EncodeDmaMoveChCtrl(sCh);

	// alloc Src,Dst Addr
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset, 0xbc);
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+1, 0);
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+6, 0xbc);
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+7, 2);
	sCh->m_uSrcOffsetAddr = sCh->m_uCodeStAddr+sCh->m_uCodeOffset+2;
	sCh->m_uDstOffsetAddr = sCh->m_uCodeStAddr+sCh->m_uCodeOffset+8;
	sCh->m_uCodeOffset+=12;

	DMA_GenerateTransferOp(sCh);
}


void DMA_GenerateTransferOp(DMAC *sCh)
{
	switch(sCh->m_eDir)
	{
		case DMA_M2M:
			DMA_EncodeDmaLoop(1, 1,sCh);
			sCh->m_uLoopCntPtr0 = sCh->m_uCodeStAddr+sCh->m_uCodeOffset-1;
			DMA_EncodeDmaLoop(0, 1,sCh);
			sCh->m_uLoopCntPtr1 = sCh->m_uCodeStAddr+sCh->m_uCodeOffset-1;

			DMA_EncodeDmaLoad(sCh);
			DMA_EncodeDmaReadMemBarrier(sCh);   // add by cha 090413
		
			DMA_EncodeDmaStore(sCh);
			DMA_EncodeDmaWriteMemBarrier(sCh);  // add by cha 090413

			DMA_EncodeDmaLoopEnd(0,sCh);
			DMA_EncodeDmaLoopEnd(1,sCh);
			sCh->m_uFirstLoopEndOffset = sCh->m_uCodeOffset;

			DMA_EncodeDmaLoop(0, 1,sCh);
			sCh->m_uLoopCntPtr2 = sCh->m_uCodeStAddr+sCh->m_uCodeOffset-1;

			DMA_EncodeDmaLoad(sCh);
			DMA_EncodeDmaReadMemBarrier(sCh);   // add by cha 090413
			//if (sCh->m_uBurstSz == 0x20)
			//{
			//	DMA_EncodeDmaLoad(sCh);
			//	DMA_EncodeDmaStore(sCh);
			//}
			DMA_EncodeDmaStore(sCh);
			DMA_EncodeDmaWriteMemBarrier(sCh);  // add by cha 090413

			DMA_EncodeDmaLoopEnd(0,sCh);
			break;
		case DMA_M2P:
			DMA_EncodeDmaFlushPeri(sCh);
			DMA_EncodeDmaLoop(1, 1,sCh);
			sCh->m_uLoopCntPtr0 = sCh->m_uCodeStAddr+sCh->m_uCodeOffset-1;
			DMA_EncodeDmaLoop(0, 1,sCh);
			sCh->m_uLoopCntPtr1 = sCh->m_uCodeStAddr+sCh->m_uCodeOffset-1;

			if (sCh->m_eDreqSrc == SOFTWARE)
			{
				DMA_EncodeDmaLoad(sCh);
				DMA_EncodeDmaStore(sCh);       // add by cha 090413
			}
			else
			{
				if( sCh->m_eDreqSrc ==  MSM_REQ0 || sCh->m_eDreqSrc ==  MSM_REQ1 )	// MODEM I/F TX
				{
					DMA_EncodeDmaWaitForPeri(sCh);
					DMA_EncodeDmaLoad(sCh);
					DMA_EncodeDmaStorePeri(sCh);
				}
				else
				{
					DMA_EncodeDmaWaitForPeri(sCh);
					DMA_EncodeDmaLoad(sCh);
					DMA_EncodeDmaStore(sCh);           // add by cha 090413
					DMA_EncodeDmaFlushPeri(sCh);					
				}
			}
			DMA_EncodeDmaLoopEnd(0,sCh);
			DMA_EncodeDmaLoopEnd(1,sCh);

			if( sCh->m_eDreqSrc ==  MSM_REQ0 || sCh->m_eDreqSrc ==  MSM_REQ1 )	// MODEM I/F TX 
				DMA_EncodeDmaFlushPeri(sCh);
						
			sCh->m_uFirstLoopEndOffset = sCh->m_uCodeOffset;

			DMA_EncodeDmaLoop(0, 1,sCh);
			sCh->m_uLoopCntPtr2 = sCh->m_uCodeStAddr+sCh->m_uCodeOffset-1;
			if (sCh->m_eDreqSrc == SOFTWARE)
			{
				DMA_EncodeDmaLoad(sCh);
				DMA_EncodeDmaStore(sCh);       // add by cha 090413
			}
			else
			{
				if( sCh->m_eDreqSrc ==  MSM_REQ0 || sCh->m_eDreqSrc ==  MSM_REQ1 )	// MODEM I/F TX
				{
					DMA_EncodeDmaWaitForPeri(sCh);
					DMA_EncodeDmaLoad(sCh);
					DMA_EncodeDmaStorePeri(sCh);
				}
				else
				{
					DMA_EncodeDmaWaitForPeri(sCh);
					DMA_EncodeDmaLoad(sCh);
					DMA_EncodeDmaStore(sCh);           // add by cha 090413
					DMA_EncodeDmaFlushPeri(sCh);
				}
			}
			DMA_EncodeDmaLoopEnd(0,sCh);

			if( sCh->m_eDreqSrc ==  MSM_REQ0 || sCh->m_eDreqSrc ==  MSM_REQ1 )	// MODEM I/F TX 
				DMA_EncodeDmaFlushPeri(sCh);
			
			break;
		case DMA_P2M:
		  DMA_EncodeDmaFlushPeri(sCh);
			DMA_EncodeDmaLoop(1, 1,sCh);
			sCh->m_uLoopCntPtr0 = sCh->m_uCodeStAddr+sCh->m_uCodeOffset-1;
			DMA_EncodeDmaLoop(0, 1,sCh);
			sCh->m_uLoopCntPtr1 = sCh->m_uCodeStAddr+sCh->m_uCodeOffset-1;
			if (sCh->m_eDreqSrc == SOFTWARE)
			{
				DMA_EncodeDmaLoad(sCh);
				DMA_EncodeDmaStore(sCh);
			}
			else 
			{
				if( sCh->m_eDreqSrc ==  MSM_REQ2 || sCh->m_eDreqSrc ==  MSM_REQ3 )	// MODEM I/F RX
				{
					DMA_EncodeDmaWaitForPeri(sCh);
					DMA_EncodeDmaLoadPeri(sCh);
					DMA_EncodeDmaStore(sCh);
				}
				else
				{
					DMA_EncodeDmaWaitForPeri(sCh);
					DMA_EncodeDmaLoadPeri(sCh);
					DMA_EncodeDmaStore(sCh);
					DMA_EncodeDmaFlushPeri(sCh);
				}
			}
			DMA_EncodeDmaLoopEnd(0,sCh);
			DMA_EncodeDmaLoopEnd(1,sCh);

			if( sCh->m_eDreqSrc ==  MSM_REQ2 || sCh->m_eDreqSrc ==  MSM_REQ3 )	// MODEM I/F RX
				DMA_EncodeDmaFlushPeri(sCh);
			
			sCh->m_uFirstLoopEndOffset = sCh->m_uCodeOffset;

			DMA_EncodeDmaLoop(0, 1,sCh);
			sCh->m_uLoopCntPtr2 = sCh->m_uCodeStAddr+sCh->m_uCodeOffset-1;
			if (sCh->m_eDreqSrc == SOFTWARE)
			{
				DMA_EncodeDmaLoad(sCh);
				DMA_EncodeDmaStore(sCh);
			}
			else 
			{
				if( sCh->m_eDreqSrc ==  MSM_REQ2 || sCh->m_eDreqSrc ==  MSM_REQ3 )	// MODEM I/F RX
				{
					DMA_EncodeDmaWaitForPeri(sCh);
					DMA_EncodeDmaLoadPeri(sCh);
					DMA_EncodeDmaStore(sCh);
				}
				else
				{
					DMA_EncodeDmaWaitForPeri(sCh);
					DMA_EncodeDmaLoadPeri(sCh);
					DMA_EncodeDmaStore(sCh);
					DMA_EncodeDmaFlushPeri(sCh);
				}
			}

			DMA_EncodeDmaFlushPeri(sCh);
			DMA_EncodeDmaLoopEnd(0,sCh);

			if( sCh->m_eDreqSrc ==  MSM_REQ2 || sCh->m_eDreqSrc ==  MSM_REQ3 )	// MODEM I/F RX
				DMA_EncodeDmaFlushPeri(sCh);

			break;
		case DMA_P2P:
		default:
			//UART_Printf(0);
			break;
	}
	sCh->m_uSecondLoopEndOffset=sCh->m_uCodeOffset;
}

void DMA_SetLoopCnt(u32 uCnt, DMAC *sCh)
{
	u8 uIter1 = uCnt/256;
	u8 uIter0 = uCnt%256;
	

	

	DbgDma(("Gen Cnt=%d, i1=%d, i0=%d\n",uCnt, uIter1, uIter0));

	memOutp8(sCh->m_uLoopCntPtr0, (uIter1==0) ? 0 : (uIter1-1));
	memOutp8(sCh->m_uLoopCntPtr1, (uIter1==0) ? (uIter0-1) : 255);
	memOutp8(sCh->m_uLoopCntPtr2, uIter0-1);

	sCh->m_uCodeOffset = ((uIter0==0)||(uIter1==0)) ? sCh->m_uFirstLoopEndOffset:sCh->m_uSecondLoopEndOffset;
}

void DMA_StartCh(u32 uSrcAddr, u32 uDstAddr, u32 uTrByte, DMAC *sCh)
{
	u32 uCnt = uTrByte/sCh->m_uBurstSz/sCh->m_eDataSz;
	u32 uIter0, uIter1;
		u32 uCurCnt;
		u32 uSrcOffsetAddr=sCh->m_uSrcOffsetAddr;
		u32 uDstOffsetAddr=sCh->m_uDstOffsetAddr;
		u32 uLoopPtr0 = sCh->m_uLoopCntPtr0;
		u32 uLoopPtr1 = sCh->m_uLoopCntPtr1;
		u32 uLoopPtr2 = sCh->m_uLoopCntPtr2;
	
	if (uCnt<=DMAC_TRUNIT_SZ)
	{

		uIter1 = uCnt/256;
		uIter0 = uCnt%256;
		WRITEWORD(sCh->m_uSrcOffsetAddr, uSrcAddr);
		WRITEWORD(sCh->m_uDstOffsetAddr, uDstAddr);
		memOutp8(sCh->m_uLoopCntPtr0, (uIter1==0) ? 0 : (uIter1-1));
		memOutp8(sCh->m_uLoopCntPtr1, (uIter1==0) ? (uIter0-1) : 255);
		memOutp8(sCh->m_uLoopCntPtr2, (uIter0==0) ? 255 : uIter0-1);
		sCh->m_uCodeOffset = ((uIter0==0)||(uIter1==0)) ? sCh->m_uFirstLoopEndOffset:sCh->m_uSecondLoopEndOffset;
	}
	else
	{
			
		while(1)
		{
			uCurCnt = (uCnt>=DMAC_TRUNIT_SZ) ? DMAC_TRUNIT_SZ : uCnt;
			uIter1 = uCurCnt/256;
			uIter0 = uCurCnt%256;
			WRITEWORD(sCh->m_uSrcOffsetAddr, uSrcAddr);
			WRITEWORD(sCh->m_uDstOffsetAddr, uDstAddr);
			memOutp8(sCh->m_uLoopCntPtr0, (uIter1==0) ? 0 : (uIter1-1));
			memOutp8(sCh->m_uLoopCntPtr1, (uIter1==0) ? (uIter0-1) : 255);
			memOutp8(sCh->m_uLoopCntPtr2, uIter0-1);
			sCh->m_uCodeOffset = ((uIter0==0)||(uIter1==0)) ? sCh->m_uFirstLoopEndOffset:sCh->m_uSecondLoopEndOffset;

			uSrcAddr += ((sCh->m_eDir==DMA_P2M) ? 0 : (uCurCnt*sCh->m_eDataSz*sCh->m_uBurstSz));
			uDstAddr += ((sCh->m_eDir==DMA_M2P) ? 0 : (uCurCnt*sCh->m_eDataSz*sCh->m_uBurstSz));
			uCnt -= uCurCnt;
			if (uCnt==0)
				break;

			memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset, 0xbc);
			memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+1, 0);
			memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+6, 0xbc);
			memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+7, 2);
			sCh->m_uSrcOffsetAddr = sCh->m_uCodeStAddr+sCh->m_uCodeOffset+2;
			sCh->m_uDstOffsetAddr = sCh->m_uCodeStAddr+sCh->m_uCodeOffset+8;
			sCh->m_uCodeOffset+=12;
			DMA_GenerateTransferOp(sCh);
		}
		sCh->m_uSrcOffsetAddr=uSrcOffsetAddr;
		sCh->m_uDstOffsetAddr=uDstOffsetAddr;
		sCh->m_uLoopCntPtr0 = uLoopPtr0;
		sCh->m_uLoopCntPtr1 = uLoopPtr1;
		sCh->m_uLoopCntPtr2 = uLoopPtr2;
	}
	DMA_EncodeDmaSendEvent(sCh->m_uChNum,sCh); // ch num = 0~7
	DMA_EncodeDmaEnd(sCh);
    
	DMA_Go(sCh);
}

// DMAGO instruction
void DMA_Go(DMAC *sCh)
{
	u32 x;
	u8 uDebugThread=0;
	u8 uSecureMode;
	
	do
	{
		x = Inp32(sCh->m_uBaseAddr+DMA_DBGSTATUS);
	} while ((x&0x1)==0x1);
	dmaDispVector(("d=0xf; while( (d&0x1)==0x1 ) Inp32 (0x%08x, d); // d=0x%08x\n",
		sCh->m_uBaseAddr+DMA_DBGSTATUS, x ));



	uSecureMode = (sCh->m_uProtectionBit==0) ?
		(0xa0|(0<<1)) : // secure mode
		(0xa0|(1<<1)); // non secure mode

	Outp32(sCh->m_uBaseAddr+DMA_DBGINST0, (sCh->m_uChNum<<24)|(uSecureMode<<16)|(sCh->m_uChNum<<8)|(uDebugThread<<0));
	Outp32(sCh->m_uBaseAddr+DMA_DBGINST1, sCh->m_uCodeStAddr);
	Outp32(sCh->m_uBaseAddr+DMA_DBGCMD, 0); // execute the instruction that the DBGINST0,1 registers contain
}


void DMA_StopCh(DMAC *sCh)
{
	u32 x;
	do
	{
		x = Inp32(sCh->m_uBaseAddr+DMA_DBGSTATUS);
	} while ((x&0x1)==0x1);
	dmaDispVector(("d=0xf; while( (d&0x1)==0x1 ) Inp32 (0x%08x, d); // d=0x%08x\n",
		m_uBaseAddr+DMA_DBGSTATUS, x ));

	Outp32(sCh->m_uBaseAddr+DMA_DBGINST0, (0<<24)|(1<<16)|(sCh->m_uChNum<<8)|(1<<0)); // issue instruction by manager thread
	Outp32(sCh->m_uBaseAddr+DMA_DBGINST1, 0);
	Outp32(sCh->m_uBaseAddr+DMA_DBGCMD, 0); // execute the instruction that the DBGINST0,1 registers contain
	
	do
	{
		x = Inp32(sCh->m_uBaseAddr+DMA_DBGSTATUS);
	} while ((x&0x1)==0x1);
	dmaDispVector(("d=0xf; while( (d&0x1)==0x1 ) Inp32 (0x%08x, d); // d=0x%08x\n",
		sCh->m_uBaseAddr+DMA_DBGSTATUS, x ));
}

// Stop Dmac manager thread( Stop Dmac all ch)
void DMA_StopDmac(DMAC *sCh)
{
	u32 x;
	//Stop Channel
	DMA_StopCh(sCh);

	
	do
	{
		x = Inp32(sCh->m_uBaseAddr+DMA_DBGSTATUS);
	} while ((x&0x1)==0x1);
	dmaDispVector(("d=0xf; while( (d&0x1)==0x1 ) Inp32 (0x%08x, d); // d=0x%08x\n",
		sCh->m_uBaseAddr+DMA_DBGSTATUS, x ));

	Outp32(sCh->m_uBaseAddr+DMA_DBGINST0, (0<<24)|(1<<16)|(sCh->m_uChNum<<8)|(0<<0));
	Outp32(sCh->m_uBaseAddr+DMA_DBGINST1, 0);
	Outp32(sCh->m_uBaseAddr+DMA_DBGCMD, 0); // execute the instruction that the DBGINST0,1 registers contain
	do
	{
		x = Inp32(sCh->m_uBaseAddr+DMA_DBGSTATUS);
	} while ((x&0x1)==0x1);
	dmaDispVector(("d=0xf; while( (d&0x1)==0x1 ) Inp32 (0x%08x, d); // d=0x%08x\n",
		sCh->m_uBaseAddr+DMA_DBGSTATUS, x ));
}


u8 DMA_IsTransferDone(DMAC *sCh)
{
	u32 x;
	u8 res;

	x = Inp32(sCh->m_uBaseAddr+DMA_CS_BASE+sCh->m_uChNum*8);


	if ( (x&0xf) == 0 ) // stopped state
		res = true;
	else
		res = false;

	return res;
}

u8  DMA_WaitForTrDone(DMAC *sCh)
{
	u32 d;
	do
	{
		d = Inp32(sCh->m_uBaseAddr+DMA_CS_BASE+sCh->m_uChNum*8);
	}while( (d&0xf) > 0);
	dmaDispVector(("d=0xf; while( (d&0xf)>0 ) Inp32 (0x%08x, d); // d=0x%08x\n",
		sCh->m_uBaseAddr+DMA_CS_BASE+sCh->m_uChNum*8,d ));
	return true;
}

void DMA_ClearIntPending(u32 uIntNum, DMAC *sCh)
{
	
	u32 uClearReg;
	uClearReg = 1<<uIntNum;
	Outp32(sCh->m_uBaseAddr+DMA_INTCLR, uClearReg);
}

void DMA_GetIntrSrc(u32 *uIntNum, DMAC *sCh)
{
	u32 x,i;
  
//	dmaInp32(DMA_INTSTATUS, x);
    x = Inp32(sCh->m_uBaseAddr+DMA_INTSTATUS);
   
	
	*uIntNum=32;
	for( i=0; i<32; i++)
	{
		if ((x>>i)&0x1 == 0x1)
		{
			*uIntNum = i;
			break;
		}
	}
	
}

void DMA_GetCurSrcAddr(u32 uAddr,DMAC *sCh)
{
	uAddr = Inp32(sCh->m_uBaseAddr+sCh->m_uChAddr+DMA_CH_SRCADDR);
}

void DMA_GetCurDstAddr(u32 uAddr, DMAC *sCh)
{
	uAddr = Inp32(sCh->m_uBaseAddr+sCh->m_uChAddr+DMA_CH_DSTADDR);
}

void DMA_ReadPC(u32 uPc, DMAC *sCh)
{
	u32 x;
	x = Inp32(sCh->m_uBaseAddr+DMA_CPC_BASE+sCh->m_uChNum*8);

	uPc = x;

}

// DMAMOV CCR, ...
void DMA_EncodeDmaMoveChCtrl(DMAC *sCh)
{
	u8 uDir = 1; // Channel Control
	u32 x = *(u32*)&m_oCtrReg;
	u8 uInsBytes[6];
	u32 i;

	DbgDma(("CCR x = 0x%x\n", x));

	DMA_EncodeDmaMove(false,0,sCh);    //DMAMOV DAR,0 ..090414 added by cha

	uInsBytes[0] = (u8)(0xbc);
	uInsBytes[1] = (u8)(0x0|uDir);
	uInsBytes[2] = (u8)((x>>0)&0xff);
	uInsBytes[3] = (u8)((x>>8)&0xff);
	uInsBytes[4] = (u8)((x>>16)&0xff);
	uInsBytes[5] = (u8)((x>>24)&0xff);

	for(i=0; i<6; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}
	sCh->m_uCodeOffset += 6;
}

void DMA_EncodeDmaMove(u8 bSrcDir, u32 uStAddr, DMAC *sCh)
{
	u8 uDir = (bSrcDir) ? 0 : 2; // src addr=0, dst addr=2
	u8 uInsBytes[6];
    u32 i;
    
	uInsBytes[0] = (u8)(0xbc);
	uInsBytes[1] = (u8)(0x0|uDir);
	uInsBytes[2] = (u8)((uStAddr>>0)&0xff);
	uInsBytes[3] = (u8)((uStAddr>>8)&0xff);
	uInsBytes[4] = (u8)((uStAddr>>16)&0xff);
	uInsBytes[5] = (u8)((uStAddr>>24)&0xff);

	for(i=0; i<6; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}
	sCh->m_uCodeOffset += 6;
}

// DMALD, DMALDS, DMALDB
void DMA_EncodeDmaLoad(DMAC *sCh)
{
	u8 bs;
	u8 x;
	u8 uInsBytes[1];
	u32 i;

	if (sCh->m_eDir == DMA_M2M|| sCh->m_eDreqSrc==SOFTWARE) // 090423 modified by cha 
	{
		bs=0; x=0;
	}
	else if ((sCh->m_eDir == DMA_M2P) && (sCh->m_uBurstSz == 1)) // single transfer only
	{
		bs=0; x=1;
	}
	else if ((sCh->m_eDir == DMA_M2P) && (sCh->m_uBurstSz != 1)) // burst transfer only
	{
		bs=1; x=1;
	}
	else
	{
		//UART_Printf(0);
	}

	uInsBytes[0] = (u8)(0x04|(bs<<1)|(x<<0));

	for(i=0; i<1; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 1;
}

// DMALDPS, DMALDPB (Load Peripheral)
void DMA_EncodeDmaLoadPeri(DMAC *sCh)
{
	//UART_Printf(sCh->m_eDir == DMA_P2M);

	u8 bs;
	u8 uInsBytes[2];
	u32 i;

	bs = (sCh->m_uBurstSz == 1) ? 0 : 1; // single -> 0, burst -> 1

	uInsBytes[0] = (u8)(0x25|(bs<<1));
	uInsBytes[1] = (u8)(0x00|((sCh->m_uDreqSrc&0x1f)<<3));

	for(i=0; i<2; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 2;
}

// DMAST, DMASTS, DMASTB
void DMA_EncodeDmaStore(DMAC *sCh)
{
	u8 bs;
	u8 x;
	u8 uInsBytes[1];
	u32 i;

	if (sCh->m_eDir == DMA_M2M|| sCh->m_eDreqSrc==SOFTWARE)  // 090423 modified by cha 
	{
		bs=0; x=0;
	}
	else if (sCh->m_uBurstSz == 1) // single transfer only
	{
		bs=0; x=1;
	}
	else if (sCh->m_uBurstSz != 1) // burst transfer only
	{
		bs=1; x=1;
	}
	else
	{
		//UART_Printf(0);
	}

	uInsBytes[0] = (u8)(0x08|(bs<<1)|(x<<0));

	for(i=0; i<1; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 1;
}

// DMASTPS, DMASTPB (Store and notify Peripheral)
void DMA_EncodeDmaStorePeri(DMAC *sCh)
{
	

	u8 bs;
	u8 uInsBytes[2];
	u32 i;
	
	//UART_Printf(sCh->m_eDir == DMA_M2P);

	bs = (sCh->m_uBurstSz == 1) ? 0 : 1; // single -> 0, burst -> 1

	uInsBytes[0] = (u8)(0x29|(bs<<1));
	uInsBytes[1] = (u8)(0x00|((sCh->m_uDreqSrc&0x1f)<<3));

	for(i=0; i<2; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 2;
}

// DMASTZ
void DMA_EncodeDmaStoreZero(DMAC *sCh)
{
	u8 uInsBytes[1];
	u32 i;
	
	uInsBytes[0] = (u8)(0x0c);

	for(i=0; i<1; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 1;
}

// DMALP
void DMA_EncodeDmaLoop(u8 uLoopCnt, u8 uIteration, DMAC *sCh)
{
	
	u8 uInsBytes[2];

	u8 uIter = uIteration-1;

	u32 i;
	
	//UART_Printf(uLoopCnt < 2); // Lc0 or Lc1

	sCh->m_bFiniteLoop = true; // used for DMALPEND
	uInsBytes[0] = (u8)(0x20|(uLoopCnt<<1));
	uInsBytes[1] = (u8)(uIter);

	for(i=0; i<2; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 2;
	sCh->m_uLoopStOffset[uLoopCnt] = sCh->m_uCodeOffset;
}

// DMALPFE	
void DMA_EncodeDmaLoopForever(u8 uLoopCnt, DMAC *sCh)
{
	
	sCh->m_bFiniteLoop = false; // used for DMALPEND
	sCh->m_uLoopStOffset[uLoopCnt] = sCh->m_uCodeOffset;
	//UART_Printf(uLoopCnt < 2); // Lc0 or Lc1
	DMA_EncodeDmaLoopEnd(uLoopCnt,sCh);
}

// DMALPEND, DMALPENDS, DMALPENDB
void DMA_EncodeDmaLoopEnd(u8 uLoopCnt,DMAC *sCh)
{
	

	u8 bs;
	u8 x;
	u8 uInsBytes[2];
	u8 uBwJump;
    u8 nf;
	u32 i;
	
	nf = (sCh->m_bFiniteLoop == true) ? 1 : 0;

	//UART_Printf(uLoopCnt < 2); // Lc0 or Lc1
	
	uBwJump = (u8)(sCh->m_uCodeOffset - sCh->m_uLoopStOffset[uLoopCnt]);
	sCh->m_uLoopStOffset[uLoopCnt] = 0;

	if (sCh->m_eDir == DMA_M2M)
	{
		bs=0; x=0;
	}
	else if (sCh->m_uBurstSz == 1) // single transfer only
	{
		/// temp bs=0; x=1;
		bs=0; x=0;
	}
	else if (sCh->m_uBurstSz != 1) // burst transfer only
	{
		bs=1; x=1;
	}
	else
	{
		//UART_Printf(0);
	}

	uInsBytes[0] = (u8)(0x28|(nf<<4)|(uLoopCnt<<2)|(bs<<1)|x);
	uInsBytes[1] = (u8)(uBwJump);

	//UART_Printf("m_uCodeStAddr = %x\n",m_uCodeStAddr);
	for(i=0; i<2; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 2;
	//UART_Printf("last address = %x",m_uCodeStAddr+m_uCodeOffset+2);
}

// DMAWFP, DMAWFPS, DMAWFPB (Wait For Peripheral)
void DMA_EncodeDmaWaitForPeri(DMAC *sCh)
{
	
	u8 bs;
	u8 p;
	u8 uInsBytes[2];
    u32 i;
    
	//UART_Printf(sCh->m_eDir != DMA_M2M);
	
	if (sCh->m_uBurstSz == 1) // single transfer only
	{
		bs=0; p=0;
	}
	else if (sCh->m_uBurstSz != 1) // burst transfer only
	{
		bs=1; p=0;
	}
	else // both single transfer and burst one
	{
		bs=0; p=1;
	}

	uInsBytes[0] = (u8)(0x30|(bs<<1)|p);
	uInsBytes[1] = (u8)(0x00|((sCh->m_uDreqSrc&0x1f)<<3));

	for(i=0; i<2; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 2;
}

// DMAWFE (Wait For Event)
void DMA_EncodeDmaWaitForEvent(u8 uEventNum, DMAC *sCh) // 0~31
{


	u8 uInsBytes[2];
	u32 i;

	uInsBytes[0] = (u8)(0x36);
	uInsBytes[1] = (u8)((uEventNum<<3)|0x2); // for cache coherency, invalid is issued.

	//UART_Printf(uEventNum < 32);

	for(i=0; i<2; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 2;
}

// DMAFLUSHP (Flush and notify Peripheral)
void DMA_EncodeDmaFlushPeri(DMAC *sCh)
{

	u8 uInsBytes[2];
	u32 i;

	uInsBytes[0] = (u8)(0x35);
	uInsBytes[1] = (u8)(0x00|((sCh->m_uDreqSrc&0x1f)<<3));

	//UART_Printf(sCh->m_eDir != DMA_M2M);

	for(i=0; i<2; i++)
	{
		memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+i, uInsBytes[i]);
	}

	sCh->m_uCodeOffset += 2;
}

// DMAEND
void DMA_EncodeDmaEnd(DMAC *sCh)
{
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset, 0x00);
	sCh->m_uCodeOffset += 1;
}

// DMAADDH (Add Halfword)
void DMA_EncodeDmaAddHalfword(u8 bSrcDir, u16 uStAddr, DMAC *sCh)
{
	u8 uDir = (bSrcDir) ? 0 : 1; // src addr=0, dst addr=1

	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset, (u8)(0x54|(uDir<<1)));
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+1, (u8)((uStAddr>>0)&0xff));
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+2, (u8)((uStAddr>>8)&0xff));
	sCh->m_uCodeOffset += 3;
}

// DMAKILL (Kill)
void DMA_EncodeDmaKill(DMAC *sCh)
{
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset, (u8)(0x01));
	sCh->m_uCodeOffset += 1;
}

// DMANOP (No operation)
void DMA_EncodeDmaNop(DMAC *sCh)
{
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset, (u8)(0x18));
	sCh->m_uCodeOffset += 1;
}


// DMARMB (Read Memory Barrier)
void DMA_EncodeDmaReadMemBarrier(DMAC *sCh)
{
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset, (u8)(0x12));
	sCh->m_uCodeOffset += 1;
}

// DMASEV (Send Event)
void DMA_EncodeDmaSendEvent(u8 uEventNum, DMAC *sCh) // 0~31
{
//	Assert(uEventNum < 32);

	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset, (u8)(0x34));
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset+1, (u8)((uEventNum<<3)|0x0));
	sCh->m_uCodeOffset += 2;
}



// DMAWMB (Write Memory Barrier)
void DMA_EncodeDmaWriteMemBarrier(DMAC *sCh)
{
	memOutp8(sCh->m_uCodeStAddr+sCh->m_uCodeOffset,  (u8)(0x13));
	sCh->m_uCodeOffset += 1;
}

void DMA_AddTrCode(
	DATA_SIZE eDataSz, DREQ_SRC eSrc, DMA_TR_DIR eDir, BURST_MODE eTrMode,
	u32 uSrcAddr, u32 uDstAddr, u32 uDataCnt,DMAC *sCh)
{
	u32 uCnt = uDataCnt/sCh->m_uBurstSz;
	u32 uIter1 = uCnt/256;
	u32 uIter0 = uCnt%256;
	u32 uSrcAhb = 0;//((APB_ST<=uSrcAddr)&&(uSrcAddr<=APB_LMT)) ? 1: 0;
	u32 uDstAhb = 1;//((APB_ST<=uDstAddr)&&(uDstAddr<=APB_LMT)) ? 1: 0;
	u32 uNextLliAhb = 0;
	u32 uDataSz ;

	u32 uIsSrcInc = (eDir==DMA_P2M) ? 0 : 1; // 0: p2m or p2p (fixed), 1: m2m or m2p (incremental)
	u32 uIsDstInc = (eDir==DMA_M2P) ? 0 : 1; // 0: m2p, p2p (fixed), 1: m2m, p2m (incremental)

	u32 uBWidth;
	
	//UART_Printf( (uCnt<=DMAC_TRUNIT_SZ)&&((uDataCnt%m_uBurstSz)==0));

	if(eDataSz == DWORD)
			uDataSz = 3 ;
//	uDataSz = (eDataSz == DWORD) ? 3 :



// m_uCodeOffset = 0;

	sCh->m_eDataSz = eDataSz;
	sCh->m_eDir= eDir;
	sCh->m_eDreqSrc = eSrc;


	//UART_Printf(eDir != DMA_P2P);
	
	switch(eTrMode) // Source burst length 1~16
	{
		case SINGLE : sCh->m_uBurstSz = 0x1;uBWidth=0x0;break;
		case BURST4 : sCh->m_uBurstSz = 0x4;uBWidth=0x3;break;
		case BURST8 : sCh->m_uBurstSz = 0x8;uBWidth=0x7;break;
		case BURST16 : sCh->m_uBurstSz = 0x10;uBWidth=0xf;break;
		case BURST16X2 : sCh->m_uBurstSz = 0x20;uBWidth=0xf;break;
		default : //UART_Printf(0);
	}
		(eDataSz == WORD) ? 2 :
		(eDataSz == HWORD) ? 1 : 0;
	sCh->m_uDreqSrc = eSrc;
	m_oCtrReg.uSI = uIsSrcInc;
	m_oCtrReg.uSBSize = uDataSz;
	m_oCtrReg.uSWidth = uBWidth;
	m_oCtrReg.uSProt = sCh->m_uProtectionBit;
	m_oCtrReg.uSCache = 0;
	m_oCtrReg.uDI = uIsDstInc;
	m_oCtrReg.uDBSize = uDataSz;
	m_oCtrReg.uDWidth = uBWidth;
	m_oCtrReg.uDProt = sCh->m_uProtectionBit;
	m_oCtrReg.uDCache = 0;
	m_oCtrReg.uESSize = 0;

	//////
	DMA_EncodeDmaMoveChCtrl(sCh);

	DMA_EncodeDmaMove(true, uSrcAddr,sCh);
	DMA_EncodeDmaMove(false, uDstAddr,sCh);
	

	DMA_GenerateTransferOp(sCh);


	memOutp8(sCh->m_uLoopCntPtr0, (uIter1==0) ? 0 : (uIter1-1));
	memOutp8(sCh->m_uLoopCntPtr1, (uIter1==0) ? (uIter0-1) : 255);
	memOutp8(sCh->m_uLoopCntPtr2, uIter0-1);
	sCh->m_uCodeOffset = ((uIter0==0)||(uIter1==0)) ?sCh-> m_uFirstLoopEndOffset:sCh->m_uSecondLoopEndOffset;
}

void DMA_AddEventCode(u8 uEventNum, DMAC *sCh)
{
	u32 sfr;
	//dmaInp32(DMA_INTEN, sfr);
	sfr = Inp32(sCh->m_uBaseAddr+DMA_INTEN);
	Outp32(sCh->m_uBaseAddr+DMA_INTEN, sfr|(1<<uEventNum));
	DMA_EncodeDmaSendEvent(uEventNum,sCh);
}
// used for LLI

void DMA_AddTr_StartCh(DMAC *sCh)
{
	DMA_EncodeDmaEnd(sCh);
	DMA_Go(sCh);
}

void DMA_CopyMem(DATA_SIZE eDataSz, BURST_MODE eTrMode, u32 uSrcAddr, u32 uDstAddr, u32 uDataCnt, DMAC *sCh)
{
	u32 uTrCnt;
	u32 uDataBytes ;//= (eDataSz == DWORD) ? 8 :
	
	if(eDataSz == DWORD)
		uDataBytes =8;
	
	sCh->m_uCodeOffset = 0;
	
		(eDataSz == WORD) ? 4 :
		(eDataSz == HWORD) ? 2 : 1;
	while(uDataCnt>0)
	{
		uTrCnt = (uDataCnt>DMAC_TRUNIT_SZ) ? DMAC_TRUNIT_SZ : uDataCnt;
		DMA_AddTrCode(eDataSz, SOFTWARE, DMA_M2M, eTrMode, uSrcAddr, uDstAddr, uTrCnt,sCh);
		uSrcAddr+= uTrCnt*uDataBytes;
		uDstAddr+= uTrCnt*uDataBytes;
		uDataCnt-=uTrCnt;
	}
	DMA_AddTr_StartCh(sCh);
	while( !DMA_IsTransferDone(sCh) ) ;
}


//inline void DMAC::GetCodeStAddr(u32& uCodeAddr)
void DMA_GetCodeStAddr(u32 uCodeAddr, DMAC *sCh)
//void DMA_GetCodeStAddr(u32& uCodeAddr)
{
	uCodeAddr = sCh->m_uCodeStAddr;
}
