/**************************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2008 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : vp.c
*  
*	File Description : This file implements the API functons for Video Processor controller.
*
*	Author : Heemyung.noh
*	Dept. : AP Development Team
*	Version : 0.1 
* 
*	History
*	- Created(Heemyung.noh 2008/12/23)
*  
**************************************************************************************/
#include "system.h" 
#include "util.h"
#include "Library.h"
#include "v210_sfr.h"
#include "intc.h"
#include "vp_v210.h"
#include "vp_table.h"
 
//#define DISP_VP_REG

#ifndef DISP_VP_REG
#define vpOutp32(addr, data) Outp32(addr, data)
#define vpInp32(addr, data)  Inp32_(addr, data)
#else
#define vpOutp32(addr, data) {UART_Printf("Outp32(\'h%08x, \'h%08x);\n", addr, data), Outp32(addr, data);}
#define vpInp32(addr, data) {Inp32_(addr, data); UART_Printf("Inp32(\'h%08x, d); // d=0x%08x\n", addr, data);}
#endif
 
//////////////////////////////////////////////////////////////////////////////
// CSC Coefficient Value
//////////////////////////////////////////////////////////////////////////////

// CSC from BT.601(SD) to BT.709(HD)
#define Y2Y_COEF_601_TO_709    0x400 // 1.0
#define CB2Y_COEF_601_TO_709   0xF87 // about -0.118188  
#define CR2Y_COEF_601_TO_709   0xF27 // about -0.212685

#define Y2CB_COEF_601_TO_709   0x0   // 0
#define CB2CB_COEF_601_TO_709  0x413 // about 1.018640
#define CR2CB_COEF_601_TO_709  0xF8B // about -0.114618

#define Y2CR_COEF_601_TO_709   0x0
#define CB2CR_COEF_601_TO_709  0x04d // about 0.075049
#define CR2CR_COEF_601_TO_709  0x41a // about 1.025327

// CSC from BT.709(HD) to BT.601(SD)
#define Y2Y_COEF_709_TO_601    0x400
#define CB2Y_COEF_709_TO_601   0x068 // about 0.101579
#define CR2Y_COEF_709_TO_601   0x0c9 // about 0.196076

#define Y2CB_COEF_709_TO_601   0x0
#define CB2CB_COEF_709_TO_601  0x3f6 // about 0.989854
#define CR2CB_COEF_709_TO_601  0xF8F // about -0.110653

#define Y2CR_COEF_709_TO_601   0x0
#define CB2CR_COEF_709_TO_601  0xFB6 // about -0.072453
//#define CR2CR_COEF_709_TO_601  0xc11 // about -0.983398
#define CR2CR_COEF_709_TO_601  0x3ef // about -0.983398
/////////////////////////////////////////////////////////////////////////////////

#define TILE_WIDTH   0x40
#define TILE_HEIGHT  0x20
#define MAX_NUM_OF_FRM 34 // according to MFC

//////////////////////////////////////////////
/// Video Processor SFR Map
/////////////////////////////////////////////
typedef enum VP_SFR
{
	rVP_ENABLE              		= VP_BASE+0x000,
	rVP_SRESET              		= VP_BASE+0x004,
	rVP_SHADOW_UPDATA     	= VP_BASE+0x008,
	rVP_FIELD_ID            		= VP_BASE+0x00C,
	rVP_MODE                		= VP_BASE+0x010,
                               
	rVP_IMG_SIZE_Y        	= VP_BASE+0x014,
	rVP_IMG_SIZE_C        	= VP_BASE+0x018,
                               
	rVP_PER_RATE_CTRL       	= VP_BASE+0x01C,
                               
	rVP_TOP_Y_PTR           	= VP_BASE+0x028,
	rVP_BOT_Y_PTR           	= VP_BASE+0x02C,
	rVP_TOP_C_PTR           	= VP_BASE+0x030,
	rVP_BOT_C_PTR           	= VP_BASE+0x034,
                               
	rVP_ENDIAN_MODE         	= VP_BASE+0x03CC,
                               
	rVP_SRC_H_POSITION      	= VP_BASE+0x044,
	rVP_SRC_V_POSITION      	= VP_BASE+0x048,
	rVP_SRC_WIDTH			= VP_BASE+0x04C,
	rVP_SRC_HEIGHT			= VP_BASE+0x050,
                               
	rVP_DST_H_POSITION     	 = VP_BASE+0x054,
	rVP_DST_V_POSITION      	= VP_BASE+0x058,
	rVP_DST_WIDTH           	= VP_BASE+0x05C,
	rVP_DST_HEIGHT          	= VP_BASE+0x060,
                               
	rVP_H_RATIO             		= VP_BASE+0x064,
	rVP_V_RATIO             		= VP_BASE+0x068,
                               
	rVP_POLY8_Y0_LL         	= VP_BASE + 0x6C,
	rVP_POLY8_Y0_LH         	= VP_BASE + 0x70,
	rVP_POLY8_Y0_HL         	= VP_BASE + 0x74,
	rVP_POLY8_Y0_HH         	= VP_BASE + 0x78,
	rVP_POLY8_Y1_LL         	= VP_BASE + 0x7C,
	rVP_POLY8_Y1_LH         	= VP_BASE + 0x80,
	rVP_POLY8_Y1_HL         	= VP_BASE + 0x84,
	rVP_POLY8_Y1_HH         	= VP_BASE + 0x88,
	rVP_POLY8_Y2_LL         	= VP_BASE + 0x8C,
	rVP_POLY8_Y2_LH         	= VP_BASE + 0x90,
	rVP_POLY8_Y2_HL         	= VP_BASE + 0x94,
	rVP_POLY8_Y2_HH         	= VP_BASE + 0x98,
	rVP_POLY8_Y3_LL         	= VP_BASE + 0x9C,
	rVP_POLY8_Y3_LH         	= VP_BASE + 0xA0,
	rVP_POLY8_Y3_HL         	= VP_BASE + 0xA4,
	rVP_POLY8_Y3_HH         	= VP_BASE + 0xA8,
	rVP_POLY4_Y0_LL         	= VP_BASE + 0xEC,
	rVP_POLY4_Y0_LH         	= VP_BASE + 0xF0,
	rVP_POLY4_Y0_HL         	= VP_BASE + 0xF4,
	rVP_POLY4_Y0_HH         	= VP_BASE + 0xF8,
	rVP_POLY4_Y1_LL         	= VP_BASE + 0xFC,
	rVP_POLY4_Y1_LH         	= VP_BASE + 0x100,
	rVP_POLY4_Y1_HL         	= VP_BASE + 0x104,
	rVP_POLY4_Y1_HH         	= VP_BASE + 0x108,
	rVP_POLY4_Y2_LL         	= VP_BASE + 0x10C,
	rVP_POLY4_Y2_LH         	= VP_BASE + 0x110,
	rVP_POLY4_Y2_HL         	= VP_BASE + 0x114,
	rVP_POLY4_Y2_HH         	= VP_BASE + 0x118,
	rVP_POLY4_Y3_LL         	= VP_BASE + 0x11C,
	rVP_POLY4_Y3_LH         	= VP_BASE + 0x120,
	rVP_POLY4_Y3_HL         	= VP_BASE + 0x124,
	rVP_POLY4_Y3_HH         	= VP_BASE + 0x128,
	rVP_POLY4_C0_LL         	= VP_BASE + 0x12C,
	rVP_POLY4_C0_LH         	= VP_BASE + 0x130,
	rVP_POLY4_C0_HL         	= VP_BASE + 0x134,
	rVP_POLY4_C0_HH         	= VP_BASE + 0x138,
	rVP_POLY4_C1_LL         	= VP_BASE + 0x13C,
	rVP_POLY4_C1_LH         	= VP_BASE + 0x140,
	rVP_POLY4_C1_HL         	= VP_BASE + 0x144,
	rVP_POLY4_C1_HH         	= VP_BASE + 0x148,
                               
	rPP_CSC_Y2Y_COEF        	= VP_BASE + 0x1D4,
	rPP_CSC_CB2Y_COEF       	= VP_BASE + 0x1D8,
	rPP_CSC_CR2Y_COEF       	= VP_BASE + 0x1DC,
	rPP_CSC_Y2CB_COEF       	= VP_BASE + 0x1E0,
	rPP_CSC_CB2CB_COEF      	= VP_BASE + 0x1E4,
	// Manual modify
	//rPP_CSC_CR2CB_COEF      	= VP_BASE + 0x1E8,
	rPP_CSC_CR2CB_COEF      	= VP_BASE + 0x1F0,
	rPP_CSC_Y2CR_COEF       	= VP_BASE + 0x1EC,
	//rPP_CSC_CB2CR_COEF      	= VP_BASE + 0x1F0,
	rPP_CSC_CB2CR_COEF      	= VP_BASE + 0x1E8,
	rPP_CSC_CR2CR_COEF      	= VP_BASE + 0x1F4,
                               
	rPP_BYPASS              		= VP_BASE + 0x200,
	rPP_SATURATION          	= VP_BASE + 0x20C,
	rPP_SHARPNESS           	= VP_BASE + 0x210,
                               
	rPP_LINE_EQ0            	= VP_BASE + 0x218,
	rPP_LINE_EQ1            	= VP_BASE + 0x21C,
	rPP_LINE_EQ2            	= VP_BASE + 0x220,
	rPP_LINE_EQ3            	= VP_BASE + 0x224,
	rPP_LINE_EQ4            	= VP_BASE + 0x228,
	rPP_LINE_EQ5            	= VP_BASE + 0x22C,
	rPP_LINE_EQ6            	= VP_BASE + 0x230,
	rPP_LINE_EQ7            	= VP_BASE + 0x234,
                               
	rPP_BRIGHT_OFFSET       	= VP_BASE + 0x238,
	rPP_CSC_EN              		= VP_BASE + 0x23C,
	rPP_VERSION_INFO        	= VP_BASE + 0x3FC
}VP_SFR;

//static oVP_SourceVariable 		oVp_SrcVar;
//static oVP_DestinationVariable 	oVp_DstVar;
static oVP_ControlVariable 		oVp_ConVar;
static oVP_EnhancingVariable	oVp_EnhVar;
oVP_Source 		oVP_SrcInf;
oVP_Destination 	oVP_DstInf;
 
//////////
// Function Name 	: 	VP_InitIP
// Description		:	Initialize the VP 
// Input 			:	oVp_Src - refer to oVP_Source
//					oVp_Dst - refer to oVP_Destination
void VP_InitIP(oVP_Source oVp_Src, oVP_Destination oVp_Dst)
{
	u32 uYSize, uCSize;	
	
	Assert((oVp_Src.eSrcBpp == NV12)||(oVp_Src.eSrcBpp == NV12_IB)||(oVp_Src.eSrcBpp == NV12_IW)
		    ||(oVp_Src.eSrcBpp == TILE_NV12)||(oVp_Src.eSrcBpp == TILE_NV12_IB)||(oVp_Src.eSrcBpp == TILE_NV12_IW));
	Assert((oVp_Src.uSrcStAddr&0x7) == 0); // uSrcStAddr should be multiple of 8
	Assert(oVp_Src.uImgHsz>=8 && oVp_Src.uImgHsz<=8192 && (oVp_Src.uImgHsz%8) == 0); //uImgHsz[2:0] should be 3'b000 for 64bit interface
	Assert(oVp_Src.uImgVsz>=1 && oVp_Src.uImgVsz<=2047);
	Assert(oVp_Src.uSrcXPos+oVp_Src.uSrcHsz <= oVp_Src.uImgHsz);
	Assert(oVp_Src.uSrcYPos+oVp_Src.uSrcVsz <= oVp_Src.uImgVsz);
	Assert(oVp_Src.uSrcNumOfFrm <= MAX_NUM_OF_FRM);

	if (oVp_Dst.eImgSz == SD480)
	{
		Assert(oVp_Dst.uDstXPos+oVp_Dst.uDstHsz <= VP_SD_H_SZ);
		Assert(oVp_Dst.uDstYPos+oVp_Dst.uDstVsz <= VP_SD_V_SZ);
	}
	else // HD720
	{
		Assert(oVp_Dst.uDstXPos+oVp_Dst.uDstHsz <= VP_HD_H_SZ);
		Assert(oVp_Dst.uDstYPos+oVp_Dst.uDstVsz <= VP_HD_V_SZ);
	}
		
///	oVp_Src.uLastFrmBufIdx = oVp_Src.uSrcNumOfFrm-1;

	VP_SWReset();
	VP_EnableIP(0);

	VP_SetPostProcessingOnOff(0);
	VP_SetEndianMode(ENDIAN_LITTLE);
	VP_SetCscOnOff(SD_TO_HD, MINUS16, 0);
	
	//Initialize the Image Information
	VP_CalcImageSize(oVp_Src, &uYSize, &uCSize);
	VP_SetModeAndImgSize(oVp_Src, oVp_Dst, uYSize, uCSize);

	//Initialize Enhancement 
	VP_InitEnhancingParameter();
	VP_SetContrast(oVp_EnhVar.aContrast); 
	VP_SetBrightness(oVp_EnhVar.aBrightness);
	VP_SetBrightOffset(oVp_EnhVar.uBrightOffset);
	VP_SetSaturation(oVp_EnhVar.uSaturation);
	VP_SetSharpness(oVp_EnhVar.eSharpness, oVp_EnhVar.uThHnoise);

	//Initialize the Poly-Phase Filter Coefficients
	VP_SetFilter();

	vpOutp32(rVP_PER_RATE_CTRL, 0x0<<0); // 1pixel/1cycle
	vpOutp32(rVP_SHADOW_UPDATA, 0x1<<0); // Update shadow reg.s
}


//////////
// Function Name 	: 	VP_CalcImageSize
// Description		:	VP Y&C Memory Size Caculation
// Input 			:	oVp_Src - refer to oVP_Source
//					*uYSize - Y(luminance) Memory Size
//					*uCSize - C(Chrominance) Memory Size
void VP_CalcImageSize(oVP_Source oVp_Src, u32 *uYSize, u32 *uCSize)
{
	u32 uY_HSizeByteMinus, uY_VSizeByteMinus;
	u32 uY_HRoundUp, uY_VRoundUp;
	u32 uC_HSizeByteMinus, uC_VSizeByteMinus;
	u32 uC_HRoundUp, uC_VRoundUp;
	u32 uRemain;
	
	if ((oVp_Src.eSrcBpp == TILE_NV12) || (oVp_Src.eSrcBpp == TILE_NV12_IB)|| (oVp_Src.eSrcBpp == TILE_NV12_IW))
	{
		// Calc Y size 
		uY_HSizeByteMinus = oVp_Src.uImgHsz - 1;
		if(oVp_Src.eSrcBpp == TILE_NV12_IB)
			uY_VSizeByteMinus = oVp_Src.uImgVsz/2 - 1;
		else
			uY_VSizeByteMinus = oVp_Src.uImgVsz - 1;
		uY_HRoundUp = ((uY_HSizeByteMinus/16)/8) + 1;
		uY_VRoundUp = ((uY_VSizeByteMinus/16)/4) + 1;

		if((uY_VSizeByteMinus & 0x20) == 0)
		{
			uY_HSizeByteMinus = uY_HSizeByteMinus & 0x7f00;
			uY_HSizeByteMinus = uY_HSizeByteMinus >> 8;
			uY_HSizeByteMinus = uY_HSizeByteMinus & 0x7f;

			uY_VSizeByteMinus = uY_VSizeByteMinus & 0x7fc0;
			uY_VSizeByteMinus = uY_VSizeByteMinus >> 6;
			uY_VSizeByteMinus = uY_VSizeByteMinus & 0x1ff;
			
			*uYSize = uY_HSizeByteMinus + (uY_VSizeByteMinus*uY_HRoundUp) + 1;
		}
		else
		{
			*uYSize = uY_HRoundUp * uY_VRoundUp;
		}
		*uYSize = *uYSize << 13;

		// Calc C(CbCr) size
		uC_HSizeByteMinus = oVp_Src.uImgHsz - 1;
		if(oVp_Src.eSrcBpp == TILE_NV12_IB)
			uC_VSizeByteMinus = oVp_Src.uImgVsz/4 - 1;
		else
			uC_VSizeByteMinus = oVp_Src.uImgVsz/2 - 1;

		uC_HRoundUp = ((uC_HSizeByteMinus/16)/8) + 1;
		uC_VRoundUp = ((uC_VSizeByteMinus/16)/4) + 1;
		
		if((uC_VSizeByteMinus & 0x20) == 0)
		{
			uC_HSizeByteMinus = uC_HSizeByteMinus & 0x7f00;
			uC_HSizeByteMinus = uC_HSizeByteMinus >> 8;
			uC_HSizeByteMinus = uC_HSizeByteMinus & 0x7f;

			uC_VSizeByteMinus = uC_VSizeByteMinus & 0x7fc0;
			uC_VSizeByteMinus = uC_VSizeByteMinus >> 6;
			uC_VSizeByteMinus = uC_VSizeByteMinus & 0x1ff;
			
			*uCSize = uC_HSizeByteMinus + (uC_VSizeByteMinus*uC_HRoundUp) + 1;
		}
		else
		{
			*uCSize = uC_HRoundUp * uC_VRoundUp;
		}
		*uCSize = *uCSize << 13;
	}
	else // NV12 or NV12_IW or NV12_IB
	{
		if(oVp_Src.eSrcBpp == NV12_IB)
		{
			*uYSize = oVp_Src.uImgHsz*oVp_Src.uImgVsz/2;
			*uCSize = oVp_Src.uImgHsz*oVp_Src.uImgVsz/4;
		}
		else
		{
			*uYSize = oVp_Src.uImgHsz*oVp_Src.uImgVsz;
			*uCSize = oVp_Src.uImgHsz*oVp_Src.uImgVsz/2;
		}
	}
}


//////////
// Function Name 	: 	VP_SetModeAndImgSize
// Description		:	VP Mode Setting & Source Image Parameter Setting & Destination Image Parameter Setting.
// Input 			:	oVp_Src - refer to oVP_Source
//					oVp_Dst - refer to oVP_Destination
//					uYSize - Y(luminance) Memory Size
//					uCSize - C(Chrominance) Memory Size
void VP_SetModeAndImgSize(oVP_Source oVp_Src, oVP_Destination oVp_Dst, u32 uYSize, u32 uCSize)
{
	u32 i;
	u32 uFieldIdReg, uWidthOffset, uVRatio;
	
	if ((oVp_Src.eSrcBpp == NV12) || (oVp_Src.eSrcBpp == TILE_NV12))
	{
		if (oVp_Dst.eScanMode == PROGRESSIVE)
		{
			// Set the start address of frame buffer
			for (i=0; i<oVp_Src.uSrcNumOfFrm; i++)
			{
				oVp_Src.uTopYAddr[i] = oVp_Src.uSrcStAddr + (uYSize * i);
				oVp_Src.uTopCAddr[i] = oVp_Src.uSrcStAddr + (uYSize * oVp_Src.uSrcNumOfFrm) + (uCSize * i);				
			}
			uFieldIdReg = 0x0<<0; // Top field
			oVp_ConVar.uModeVal = 
				 (0x0<<5) // Line skip off
				|((oVp_Src.eSrcBpp == NV12) ? 0x0<<4 : 0x1<<4)
				|(0x0<<3) // Use one chroma
				|(0x0<<2) // Field id is defined by user
				|(0x0<<1); // Disable 2D-IPC. In fact, in this mode 2D-IPC is don't care
			
			oVp_ConVar.uScanConversionIdx = 0;
		}
		else // oVp_Dst.eScanMode == INTERLACED
		{
			uFieldIdReg = 0x0<<0; // Top field

			uWidthOffset = (oVp_Src.eSrcBpp == NV12) ? oVp_Src.uImgHsz : TILE_WIDTH;
			for (i=0; i<oVp_Src.uSrcNumOfFrm; i++)
			{
				oVp_Src.uTopYAddr[i] = oVp_Src.uSrcStAddr + (uYSize * i);
				oVp_Src.uBotYAddr[i] = oVp_Src.uSrcStAddr + (uYSize * i) + uWidthOffset ;
				oVp_Src.uTopCAddr[i] = oVp_Src.uSrcStAddr + (uYSize * oVp_Src.uSrcNumOfFrm) + (uCSize * i);
				oVp_Src.uBotCAddr[i] = oVp_Src.uSrcStAddr + (uYSize * oVp_Src.uSrcNumOfFrm) + (uCSize * i) + uWidthOffset;
			}
			oVp_ConVar.uModeVal =
				 (0x1<<5) // Line skip on
				|((oVp_Src.eSrcBpp == NV12) ? 0x0<<4 : 0x1<<4)
				|(0x1<<3) // Use top c and bot c
				|(0x1<<2) // Field id is automatically toggled by V_SYNC
				|(0x0<<1); // Disable 2D-IPC.
			
			oVp_Src.uSrcVsz = oVp_Src.uSrcVsz/2;
			oVp_Src.uSrcYPos = oVp_Src.uSrcYPos/2;
			
			oVp_Dst.uDstVsz = oVp_Dst.uDstVsz/2;			
			oVp_Dst.uDstYPos = oVp_Dst.uDstYPos/2;
			
			oVp_ConVar.uScanConversionIdx = 1;			
		}
	}
	else if((oVp_Src.eSrcBpp == NV12_IB) || (oVp_Src.eSrcBpp == TILE_NV12_IB)) // Interlace bob
	{
		// Set the start address of frame buffer	
		uWidthOffset = (oVp_Src.eSrcBpp == NV12_IB) ? oVp_Src.uImgHsz*oVp_Src.uImgVsz/2 : TILE_WIDTH*TILE_HEIGHT/2;
		//uWidthOffset = oVp_Src.uImgHsz*oVp_Src.uImgVsz/2; 
		
		for (i=0; i<oVp_Src.uSrcNumOfFrm; i++)
		{
		#if 0
			oVp_Src.uTopYAddr[i] = oVp_Src.uSrcStAddr + (uYSize * i);	
			oVp_Src.uBotYAddr[i] = oVp_Src.uSrcStAddr + (uYSize * i) + uWidthOffset;
			oVp_Src.uTopCAddr[i] = oVp_Src.uSrcStAddr + (uYSize * oVp_Src.uSrcNumOfFrm) + (uCSize * i); // uYSize = Top Y size + Bottom Y size
			oVp_Src.uBotCAddr[i] = oVp_Src.uSrcStAddr + (uYSize * oVp_Src.uSrcNumOfFrm) + (uCSize * i) + uWidthOffset/2;
		#else
			oVp_Src.uTopYAddr[i] = oVp_Src.uSrcStAddr + ((uYSize+uCSize) * i);	
			oVp_Src.uTopCAddr[i] = oVp_Src.uSrcStAddr + ((uYSize+uCSize) * i) + uYSize; 
			oVp_Src.uBotYAddr[i] = oVp_Src.uSrcStAddr + ((uYSize+uCSize) * i) + uYSize + uCSize;
			oVp_Src.uBotCAddr[i] = oVp_Src.uSrcStAddr + ((uYSize+uCSize) * i) + uYSize + uCSize + uYSize;
		#endif
		}
		
		if (oVp_Dst.eScanMode == PROGRESSIVE)
		{
			uFieldIdReg = 0x0<<0;
			oVp_ConVar.uModeVal = 
				 (0x0<<5) // Line skip on
				|((oVp_Src.eSrcBpp == NV12_IB) ? 0x0<<4 : 0x1<<4)
				|(0x0<<3) // Use one chroma
				|(0x0<<2) // Field id is automatically toggled by V_SYNC
		//		|(0x1<<2) // Field id is automatically toggled by V_SYNC...must be tested
				|(0x1<<1); // Enable 2D-IPC

			oVp_Src.uSrcVsz = oVp_Src.uSrcVsz/2; 
			oVp_Src.uSrcYPos = oVp_Src.uSrcYPos/2;
			
			oVp_ConVar.uScanConversionIdx = 2;				
		}
		else // oVp_Dst.eScanMode == INTERLACED
		{
			// Top field. In fact, in this mode this value is don't care because FIELD_ID_AUTO_TOGGLLING bit will be enabled		
			uFieldIdReg = 0x0<<0;
			oVp_ConVar.uModeVal =
				 (0x0<<5) // Line skip on
				|((oVp_Src.eSrcBpp == NV12_IB) ? 0x0<<4 : 0x1<<4)
				|(0x1<<3) // Use top c and bot c
				|(0x1<<2) // Field id is automatically toggled by V_SYNC
				|(0x0<<1); // Disable 2D-IPC.	

			oVp_Src.uSrcVsz = oVp_Src.uSrcVsz/2;
			oVp_Src.uSrcYPos = oVp_Src.uSrcYPos/2;
			oVp_Dst.uDstVsz = oVp_Dst.uDstVsz/2;
			oVp_Dst.uDstYPos = oVp_Dst.uDstYPos/2;
			
			oVp_ConVar.uScanConversionIdx = 3;			
		}
	}	
	else if((oVp_Src.eSrcBpp == NV12_IW) || (oVp_Src.eSrcBpp == TILE_NV12_IW)) // Interlace weave
	{
		// Set the start address of frame buffer	
		uWidthOffset = (oVp_Src.eSrcBpp == NV12_IW) ? oVp_Src.uImgHsz : TILE_WIDTH;
		for (i=0; i<oVp_Src.uSrcNumOfFrm; i++)
		{
			oVp_Src.uTopYAddr[i] = oVp_Src.uSrcStAddr + (uYSize * i);	
			oVp_Src.uBotYAddr[i] = oVp_Src.uSrcStAddr + (uYSize * i) + uWidthOffset;
			oVp_Src.uTopCAddr[i] = oVp_Src.uSrcStAddr + (uYSize * oVp_Src.uSrcNumOfFrm) + (uCSize * i); // uYSize = Top Y size + Bottom Y size
			oVp_Src.uBotCAddr[i] = oVp_Src.uSrcStAddr + (uYSize * oVp_Src.uSrcNumOfFrm) + (uCSize * i) + uWidthOffset;
		}
		
		if (oVp_Dst.eScanMode == PROGRESSIVE)
		{
			uFieldIdReg = 0x0<<0;
			oVp_ConVar.uModeVal = 
				 (0x1<<5) // Line skip on
				|((oVp_Src.eSrcBpp == NV12_IW) ? 0x0<<4 : 0x1<<4)
				|(0x0<<3) // Use one chroma
				|(0x0<<2) // Field id is automatically toggled by V_SYNC
		//		|(0x1<<2) // Field id is automatically toggled by V_SYNC...must be tested				
				|(0x1<<1); // Enable 2D-IPC

			oVp_Src.uSrcVsz = oVp_Src.uSrcVsz/2; 
			oVp_Src.uSrcYPos = oVp_Src.uSrcYPos/2;
			
			oVp_ConVar.uScanConversionIdx = 4;				
		}
		else // oVp_Dst.eScanMode == INTERLACED
		{
			// Top field. In fact, in this mode this value is don't care because FIELD_ID_AUTO_TOGGLLING bit will be enabled		
			uFieldIdReg = 0x0<<0;
			oVp_ConVar.uModeVal =
				 (0x1<<5) // Line skip on
				|((oVp_Src.eSrcBpp == NV12_IW) ? 0x0<<4 : 0x1<<4)
				|(0x1<<3) // Use top c and bot c
				|(0x1<<2) // Field id is automatically toggled by V_SYNC
				|(0x0<<1); // Disable 2D-IPC.	

			oVp_Src.uSrcVsz = oVp_Src.uSrcVsz/2;
			oVp_Src.uSrcYPos = oVp_Src.uSrcYPos/2;
			oVp_Dst.uDstVsz = oVp_Dst.uDstVsz/2;
			oVp_Dst.uDstYPos = oVp_Dst.uDstYPos/2;
			
			oVp_ConVar.uScanConversionIdx = 5;			
		}
	}
	else
	{
		Assert(0);
	}

	// Mode Setting
	vpOutp32(rVP_FIELD_ID, uFieldIdReg);
	vpOutp32(rVP_MODE, oVp_ConVar.uModeVal);

	// Setting for source image
	vpOutp32(rVP_IMG_SIZE_Y, (oVp_Src.uImgHsz<<16)|(oVp_Src.uImgVsz<<0));
	vpOutp32(rVP_IMG_SIZE_C, (oVp_Src.uImgHsz<<16)|(oVp_Src.uImgVsz/2<<0));

	if (((oVp_Src.eSrcBpp == NV12) || (oVp_Src.eSrcBpp == TILE_NV12)) && (oVp_Dst.eScanMode == PROGRESSIVE))
	{
		vpOutp32(rVP_TOP_Y_PTR, oVp_Src.uTopYAddr[0]);
		vpOutp32(rVP_TOP_C_PTR, oVp_Src.uTopCAddr[0]); 
	}
	else // VP_IN_INTERLACE_PACKED or VP_IN_INTERLACE_WEAVE
	{
		vpOutp32(rVP_TOP_Y_PTR, oVp_Src.uTopYAddr[0]);
		vpOutp32(rVP_BOT_Y_PTR, oVp_Src.uBotYAddr[0]); 
		vpOutp32(rVP_TOP_C_PTR, oVp_Src.uTopCAddr[0]); 
		vpOutp32(rVP_BOT_C_PTR, oVp_Src.uBotCAddr[0]); 
	}
	
	vpOutp32(rVP_SRC_H_POSITION, (oVp_Src.uSrcXPos<<4)|((u32)oVp_Src.eStep<<0));
	vpOutp32(rVP_SRC_V_POSITION, (oVp_Src.uSrcYPos&0x7ff)<<0);
	
	vpOutp32(rVP_SRC_WIDTH, (oVp_Src.uSrcHsz&0x7ff)<<0);
	vpOutp32(rVP_SRC_HEIGHT, (oVp_Src.uSrcVsz&0x7ff)<<0);
	
	// Set the Destination Image
	vpOutp32(rVP_DST_H_POSITION, (oVp_Dst.uDstXPos&0x7ff)<<0);
	vpOutp32(rVP_DST_V_POSITION, (oVp_Dst.uDstYPos&0x7ff)<<0);	
	vpOutp32(rVP_DST_WIDTH, (oVp_Dst.uDstHsz&0x7ff)<<0);
	vpOutp32(rVP_DST_HEIGHT, (oVp_Dst.uDstVsz&0x7ff)<<0);

	// Set H/V ratio
	vpOutp32(rVP_H_RATIO, (oVp_Src.uSrcHsz<<16)/oVp_Dst.uDstHsz);
	uVRatio = ( (oVp_ConVar.uModeVal&(0x1<<1)) == (0x1<<1) ) ? ((oVp_Src.uSrcVsz<<17)/oVp_Dst.uDstVsz) : ((oVp_Src.uSrcVsz<<16)/oVp_Dst.uDstVsz);
	vpOutp32(rVP_V_RATIO, uVRatio);

	//  Update shadow register
	vpOutp32(rVP_SHADOW_UPDATA, 0x1<<0); 
}


//////////
// Function Name 	: 	VP_SetCroppingSize
// Description		:	Set the Cropping Value of Source Image
// Input 			:	uXPos - Source Horizontal Offset
//					uYPos - Source Vertical Offset
//					uCroppedHsz - Width of Horizontal Source Image Size
//					uCroppedVsz - Width of Vertical Source Image Size
//					eStep - refer to VP_SRC_X_POS_FRACTION_STEP enum
void VP_SetCroppingSize(	u32 uXPos, u32 uYPos, u32 uCroppedHsz, u32 uCroppedVsz, VP_eSRC_X_POS_FRACTION_STEP eStep)
{
	u32 uVRatio;	
	
	Assert(uXPos+uCroppedHsz <= oVP_SrcInf.uImgHsz);
	Assert(uYPos+uCroppedVsz<= oVP_SrcInf.uImgVsz);

	oVP_SrcInf.uSrcXPos = uXPos;
	oVP_SrcInf.uSrcYPos = uYPos;
	oVP_SrcInf.uSrcHsz = uCroppedHsz;
	oVP_SrcInf.uSrcVsz = uCroppedVsz;
	oVP_SrcInf.eStep = eStep;
	
	if ((oVP_SrcInf.eSrcBpp == NV12) || (oVP_SrcInf.eSrcBpp == TILE_NV12))
	{
		if (oVP_DstInf.eScanMode == PROGRESSIVE)
		{
		}
		else // oVp_Dst->eScanMode == INTERLACED
		{
			oVP_SrcInf.uSrcVsz = oVP_SrcInf.uSrcVsz/2;
			oVP_SrcInf.uSrcYPos = oVP_SrcInf.uSrcYPos/2;
		}
	}
	else if((oVP_SrcInf.eSrcBpp == NV12_IB) || (oVP_SrcInf.eSrcBpp == TILE_NV12_IB)) // Interlace bob
	{
		if (oVP_DstInf.eScanMode == PROGRESSIVE)
		{
			oVP_SrcInf.uSrcVsz = oVP_SrcInf.uSrcVsz/2; 
			oVP_SrcInf.uSrcYPos = oVP_SrcInf.uSrcYPos/2;
		}
		else // oVp_Dst->eScanMode == INTERLACED
		{
			oVP_SrcInf.uSrcVsz = oVP_SrcInf.uSrcVsz/2;
			oVP_SrcInf.uSrcYPos = oVP_SrcInf.uSrcYPos/2;
		}
	}	
	else if((oVP_SrcInf.eSrcBpp == NV12_IW) || (oVP_SrcInf.eSrcBpp == TILE_NV12_IW)) // Interlace weave
	{
		if (oVP_DstInf.eScanMode == PROGRESSIVE)
		{
			oVP_SrcInf.uSrcVsz = oVP_SrcInf.uSrcVsz/2; 
			oVP_SrcInf.uSrcYPos = oVP_SrcInf.uSrcYPos/2;
		}
		else // oVp_Dst->eScanMode == INTERLACED
		{
			oVP_SrcInf.uSrcVsz = oVP_SrcInf.uSrcVsz/2;
			oVP_SrcInf.uSrcYPos = oVP_SrcInf.uSrcYPos/2;
		}
	}
	else
	{
		Assert(0);
	}

	vpOutp32(rVP_SRC_H_POSITION, (oVP_SrcInf.uSrcXPos<<4)|((u32)oVP_SrcInf.eStep<<0));
	vpOutp32(rVP_SRC_V_POSITION, (oVP_SrcInf.uSrcYPos&0x7ff)<<0);
	
	vpOutp32(rVP_SRC_WIDTH, (oVP_SrcInf.uSrcHsz&0x7ff)<<0);
	vpOutp32(rVP_SRC_HEIGHT, (oVP_SrcInf.uSrcVsz&0x7ff)<<0);

	// Set H/V ratio
	vpOutp32(rVP_H_RATIO, (oVP_SrcInf.uSrcHsz<<16)/oVP_DstInf.uDstHsz);
	uVRatio = ( (oVp_ConVar.uModeVal&(0x1<<1)) == (0x1<<1) ) ? ((oVP_SrcInf.uSrcVsz<<17)/oVP_DstInf.uDstVsz) : ((oVP_SrcInf.uSrcVsz<<16)/oVP_DstInf.uDstVsz);
	vpOutp32(rVP_V_RATIO, uVRatio);

	VP_SetFilter();

	vpOutp32(rVP_SHADOW_UPDATA, 0x1<<0);	
}


//////////
// Function Name 	: 	SetViewPosition
// Description		:	Set the display position value of destination image
// Input 			:	uXPos - Destination Horizontal Offset
//					uYPos - Destination Vertical Offset
//					uViewHsz - Width of Horizontal Source Image Size
//					uViewVsz - Width of Vertical Source Image Size
void VP_SetViewPosition(u32 uXPos, u32 uYPos, u32 uViewHsz, u32 uViewVsz)
{
	u32 uVRatio;		
	
	if (oVP_DstInf.eImgSz == SD480)
	{
		Assert(uXPos+uViewHsz <= SD480_HSIZE);
		Assert(uYPos+uViewVsz <= SD480_VSIZE);		
	}
	else if (oVP_DstInf.eImgSz == SD576)
	{
		Assert(uXPos+uViewHsz <= SD576_HSIZE);
		Assert(uYPos+uViewVsz <= SD576_VSIZE);		
	}	
	else if (oVP_DstInf.eImgSz == HD720)
	{
		Assert(uXPos+uViewHsz <= HD720_HSIZE);
		Assert(uYPos+uViewVsz <= HD720_VSIZE);		
	}		
	else // HD1080
	{
		Assert(uXPos+uViewHsz <= HD1080_HSIZE);
		Assert(uYPos+uViewVsz <= HD1080_VSIZE);		
	}
	
	oVP_DstInf.uDstXPos = uXPos;
	oVP_DstInf.uDstYPos = uYPos;
	oVP_DstInf.uDstHsz = uViewHsz;
	oVP_DstInf.uDstVsz = uViewVsz;

	if ((oVP_SrcInf.eSrcBpp == NV12) || (oVP_SrcInf.eSrcBpp == TILE_NV12))
	{
		if (oVP_DstInf.eScanMode == PROGRESSIVE)
		{
		}
		else // oVp_Dst->eScanMode == INTERLACED
		{
			oVP_DstInf.uDstVsz = oVP_DstInf.uDstVsz/2;			
			oVP_DstInf.uDstYPos = oVP_DstInf.uDstYPos/2;
		}
	}
	else if((oVP_SrcInf.eSrcBpp == NV12_IB) || (oVP_SrcInf.eSrcBpp == TILE_NV12_IB)) // Interlace bob
	{
		if (oVP_DstInf.eScanMode == PROGRESSIVE)
		{
		}
		else // oVp_Dst->eScanMode == INTERLACED
		{
			oVP_DstInf.uDstVsz = oVP_DstInf.uDstVsz/2;
			oVP_DstInf.uDstYPos = oVP_DstInf.uDstYPos/2;
		}
	}	
	else if((oVP_SrcInf.eSrcBpp == NV12_IW) || (oVP_SrcInf.eSrcBpp == TILE_NV12_IW)) // Interlace weave
	{
		if (oVP_DstInf.eScanMode == PROGRESSIVE)
		{
		}
		else // oVp_Dst->eScanMode == INTERLACED
		{
			oVP_DstInf.uDstVsz = oVP_DstInf.uDstVsz/2;
			oVP_DstInf.uDstYPos = oVP_DstInf.uDstYPos/2;
		}
	}
	else
	{
		Assert(0);
	}

	vpOutp32(rVP_DST_H_POSITION, (oVP_DstInf.uDstXPos&0x7ff)<<0);
	vpOutp32(rVP_DST_V_POSITION, (oVP_DstInf.uDstYPos&0x7ff)<<0);

	vpOutp32(rVP_DST_WIDTH, (oVP_DstInf.uDstHsz&0x7ff)<<0);
	vpOutp32(rVP_DST_HEIGHT, (oVP_DstInf.uDstVsz&0x7ff)<<0);

	// Set H/V ratio
	vpOutp32(rVP_H_RATIO, (oVP_SrcInf.uSrcHsz<<16)/uViewHsz);
	//uVRatio = ( (oVp_ConVar.uModeVal&(0x1<<1)) == (0x1<<1) ) ? ((oVP_SrcInf.uSrcVsz<<17)/uViewVsz) : ((oVP_SrcInf.uSrcVsz<<16)/uViewVsz);
	uVRatio = ( (oVp_ConVar.uModeVal&(0x1<<1)) == (0x1<<1) ) ? ((oVP_SrcInf.uSrcVsz<<17)/oVP_DstInf.uDstVsz) : ((oVP_SrcInf.uSrcVsz<<16)/oVP_DstInf.uDstVsz);
	vpOutp32(rVP_V_RATIO, uVRatio);
	VP_SetFilter();

	vpOutp32(rVP_SHADOW_UPDATA, 0x1<<0); // Update shadow reg.s		
}


//////////
// Function Name 	: 	VP_StartProcessing
// Description		:	Start the VP Processing
// Input 			:	uFrmBufIdx - Frame Buffer Index
void VP_StartProcessing(u32 uFrmBufIdx)
{
	Assert(uFrmBufIdx <= oVP_SrcInf.uLastFrmBufIdx);

	if ((oVP_SrcInf.eSrcBpp == NV12 || TILE_NV12) && (oVP_DstInf.eScanMode == PROGRESSIVE))
	{
		vpOutp32(rVP_TOP_Y_PTR, oVP_SrcInf.uTopYAddr[uFrmBufIdx]);
		vpOutp32(rVP_TOP_C_PTR, oVP_SrcInf.uTopCAddr[uFrmBufIdx]); 
	}
	else // VP_IN_INTERLACE_PACKED or VP_IN_INTERLACE_WEAVE
	{
		vpOutp32(rVP_TOP_Y_PTR, oVP_SrcInf.uTopYAddr[uFrmBufIdx]);
		vpOutp32(rVP_BOT_Y_PTR, oVP_SrcInf.uBotYAddr[uFrmBufIdx]); 
		vpOutp32(rVP_TOP_C_PTR, oVP_SrcInf.uTopCAddr[uFrmBufIdx]); 
		vpOutp32(rVP_BOT_C_PTR, oVP_SrcInf.uBotCAddr[uFrmBufIdx]); 
	}

	vpOutp32(rVP_ENABLE, 0x1<<0);
}


//////////
// Function Name 	: 	VP_IsProcessingDone
// Description		:	Check whether the VP is processing.
// Input 			:	None
// Output		: 	true - Processing
//					false - Processing Done
u8 VP_IsProcessingDone(void)
{
	u32 uEnableReg;
	vpInp32(rVP_ENABLE, uEnableReg);

	return ( uEnableReg&(0x1<<1) == (0x1<<1) ) ? true : false;
}


//////////
// Function Name 	: 	VP_SetActiveFrameBuf
// Description		:	Set the frame buffer address
// Input 			:	uFrmBufIdx - Frame Buffer Index
void VP_SetActiveFrameBuf(u32 uFrmBufIdx)
{
	if ((oVP_SrcInf.eSrcBpp == NV12 || TILE_NV12) && (oVP_DstInf.eScanMode == PROGRESSIVE))
	{
		vpOutp32(rVP_TOP_Y_PTR, oVP_SrcInf.uTopYAddr[uFrmBufIdx]);
		vpOutp32(rVP_TOP_C_PTR, oVP_SrcInf.uTopCAddr[uFrmBufIdx]); 
	}
	else // VP_IN_INTERLACE_PACKED or VP_IN_INTERLACE_WEAVE
	{
		vpOutp32(rVP_TOP_Y_PTR, oVP_SrcInf.uTopYAddr[uFrmBufIdx]);
		vpOutp32(rVP_BOT_Y_PTR, oVP_SrcInf.uBotYAddr[uFrmBufIdx]); 
		vpOutp32(rVP_TOP_C_PTR, oVP_SrcInf.uTopCAddr[uFrmBufIdx]); 
		vpOutp32(rVP_BOT_C_PTR, oVP_SrcInf.uBotCAddr[uFrmBufIdx]); 
	}

	vpOutp32(rVP_SHADOW_UPDATA, 0x1<<0);
}


//////////
// Function Name 	: 	VP_SetPostProcessingOnOff
// Description		:	VP PostProcessing On/Off Selection
// Input 			:	uOnOff - 1(On), 0(Off)
void VP_SetPostProcessingOnOff(u32 uOnOff)
{
	Assert(uOnOff == 1 || uOnOff == 0);
	
	vpOutp32(rPP_BYPASS, (uOnOff==1) ? (0x0<<0) : (0x1<<0) );

	vpOutp32(rVP_SHADOW_UPDATA, 0x1<<0); 	
}


//////////
// Function Name 	: 	VP_SetCscOnOff
// Description		:	VP CSC On/Off Selection
// Input 			:	eCscType - refer to VP_CSC_TYPE
//					eYOffset - refer to VP_Y_OFFSET
//					uOnOff - 1(On), 0(Off)
void VP_SetCscOnOff(VP_eCSC_TYPE eCscType, VP_eY_OFFSET eYOffset, u32 uOnOff)
{
	u32 uTemp;
	
	if(uOnOff == 1)
	{
		if (eCscType == SD_TO_HD)
		{
			vpOutp32(rPP_CSC_Y2Y_COEF, Y2Y_COEF_601_TO_709);
			vpOutp32(rPP_CSC_CB2Y_COEF, CB2Y_COEF_601_TO_709);
			vpOutp32(rPP_CSC_CR2Y_COEF, CR2Y_COEF_601_TO_709);

			vpOutp32(rPP_CSC_Y2CB_COEF, Y2CB_COEF_601_TO_709);
			vpOutp32(rPP_CSC_CB2CB_COEF, CB2CB_COEF_601_TO_709);
			vpOutp32(rPP_CSC_CR2CB_COEF, CR2CB_COEF_601_TO_709);

			vpOutp32(rPP_CSC_Y2CR_COEF, Y2CR_COEF_601_TO_709);
			vpOutp32(rPP_CSC_CB2CR_COEF, CB2CR_COEF_601_TO_709);
			vpOutp32(rPP_CSC_CR2CR_COEF, CR2CR_COEF_601_TO_709);
		}
		else 
		{
			vpOutp32(rPP_CSC_Y2Y_COEF, Y2Y_COEF_709_TO_601);
			vpOutp32(rPP_CSC_CB2Y_COEF, CB2Y_COEF_709_TO_601);
			vpOutp32(rPP_CSC_CR2Y_COEF, CR2Y_COEF_709_TO_601);

			vpOutp32(rPP_CSC_Y2CB_COEF, Y2CB_COEF_709_TO_601);
			vpOutp32(rPP_CSC_CB2CB_COEF, CB2CB_COEF_709_TO_601);
			vpOutp32(rPP_CSC_CR2CB_COEF, CR2CB_COEF_709_TO_601);

			vpOutp32(rPP_CSC_Y2CR_COEF, Y2CR_COEF_709_TO_601);
			vpOutp32(rPP_CSC_CB2CR_COEF, CB2CR_COEF_709_TO_601);
			vpOutp32(rPP_CSC_CR2CR_COEF, CR2CR_COEF_709_TO_601);
		}
		
		vpOutp32(rPP_CSC_EN, (0x1<<0) |	((eYOffset == MINUS16) ? (0x1<<1) : (0x0<<0)) );
	}
	else		// uOnOff == 0
	{
		vpInp32(rPP_CSC_EN, uTemp);
		vpOutp32(rPP_CSC_EN, uTemp & ~(0x1<<0));
	}
	
	vpOutp32(rVP_SHADOW_UPDATA, 0x1<<0); 
}


//////////
// Function Name 	: 	VP_SWReset
// Description		:	VP Software Reset Command
// Input 			:	None
void VP_SWReset(void)
{
	u32 uResetReg; 
	
	do 
	{
		vpInp32(rVP_SRESET, uResetReg);
	} while ( (uResetReg&(0x1<<0)) != (0x0<<0) ); // Wait for this SFR[0] to zero
	
	vpOutp32(rVP_SRESET, 0x1<<0);
}


//////////
// Function Name 	: 	VP_EnableIP
// Description		:	VP On
// Input 			:	uOnOff - 1(On), 0(Off)
void VP_EnableIP(u32 uOnOff)
{
	Assert(uOnOff == 0 || uOnOff == 1);

	vpOutp32(rVP_ENABLE, uOnOff<<0);
}


//////////
// Function Name 	: 	VP_SetEndianMode
// Description		:	VP Source Image Endian Mode Setting
// Input 			:	uLittleBig - ENDIAN_BIG(Big Endian), ENDIAN_LITTLE(Little Endian)
void VP_SetEndianMode(u32 uLittleBig)
{
	Assert(uLittleBig == ENDIAN_BIG || uLittleBig == ENDIAN_LITTLE);

	vpOutp32(rVP_ENDIAN_MODE, uLittleBig<<0);

	vpOutp32(rVP_SHADOW_UPDATA, 0x1<<0); 
}


//////////
// Function Name 	: 	VP_InitEnhancingParameter
// Description		:	Initialize the VP Enhancing Parameter Variable(refer to oVP_EnhancingVariable Structure)
// Input 			:	None
void VP_InitEnhancingParameter(void)
{
	u32 i;
	
	for(i=0 ; i<8 ; i++)
	{
		oVp_EnhVar.aBrightness[i] = 0x0;
		oVp_EnhVar.aContrast[i] = 0x80;
	}

	oVp_EnhVar.uSaturation = 0x80;
	oVp_EnhVar.eSharpness = NO_EFFECT;
	oVp_EnhVar.uThHnoise = 0x5;
	oVp_EnhVar.uBrightOffset = 0x0;
}


//////////
// Function Name 	: 	VP_SetContrast
// Description		:	VP Contrast Setting
// Input 			:	uContrast - LINE_SLOPE Value(unsigned 1.7 format)
void VP_SetContrast(u32 *uContrast)
{
	u32 i, uLineEq[8];

	for(i=0 ; i<8 ; i++)
	{
		vpInp32(rPP_LINE_EQ0 + 4*i, uLineEq[i]);
		uLineEq[i] &= ~(0xFF<<0);
		uLineEq[i] |= ((uContrast[i]&0xFF)<<0);

		vpOutp32(rPP_LINE_EQ0 + 4*i, uLineEq[i]);
	}

	vpOutp32(rVP_SHADOW_UPDATA, 0x1);			
}


//////////
// Function Name 	: 	VP_SetBrightness
// Description		:	VP Brightness Setting
// Input 			:	uBrightness - LINE_INTC Value(unsigned 9.7 format)
void VP_SetBrightness(u32 *uBrightness)
{
	u32 i, uLineEq[8];

	for(i=0 ; i<8 ; i++)
	{
		vpInp32(rPP_LINE_EQ0 + 4*i, uLineEq[i]);
		uLineEq[i] &= ~(0xFFFF<<8);
		uLineEq[i] |= ((uBrightness[i]&0xFFFF)<<8);

		vpOutp32(rPP_LINE_EQ0 + 4*i, uLineEq[i]);
	}

	vpOutp32(rVP_SHADOW_UPDATA, 0x1);			
}


//////////
// Function Name 	: 	VP_SetBrightOffset
// Description		:	Set the Brightness Offset control register for Y
// Input 			:	uOffset - Offset for Y Brightness Value(signed 1.8 format)
void VP_SetBrightOffset(u32 uOffset)
{
	Assert(uOffset <= 0x1FF);

	vpOutp32(rPP_BRIGHT_OFFSET, (uOffset&0x1FF)<<0);
	vpOutp32(rVP_SHADOW_UPDATA, 0x1);		
}


//////////
// Function Name 	: 	VP_SetSaturation
// Description		:	Set the Color Saturation control register
// Input 			:	uSaturation - Color Saturation Factor Value(signed 1.7 format)
void VP_SetSaturation(u32 uSaturation)
{
	Assert(uSaturation <= 0xFF);

	vpOutp32(rPP_SATURATION, (uSaturation&0xff)<<0);
	vpOutp32(rVP_SHADOW_UPDATA, 0x1);	
}


//////////
// Function Name 	: 	VP_SetSharpness
// Description		:	Set the Picture sharpness control register
// Input 			:	eSharpness - Control for the edge enhancement Value
//					uThreshold - Treshold value setting to decide minimum vertical edge value
void VP_SetSharpness(VP_eSHARPNESS eSharpness, u32 uThreshold)
{
	u32 uSharpVal;
	
	Assert(uThreshold <= 0xff);

	uSharpVal = 	(eSharpness == NO_EFFECT) ? (0x0<<0) :
				(eSharpness == MIN_EDGE) ? (0x1<<0) :
				(eSharpness == MODERATE_EDGE) ? (0x2<<0) : 
				(0x3<<0); // MAX_EDGE

	vpOutp32(rPP_SHARPNESS, uSharpVal | ((uThreshold&0xff)<<8));
	vpOutp32(rVP_SHADOW_UPDATA, 0x1);		
}


//////////
// Function Name 	: 	VP_SetFilter
// Description		:	Set the Poly-Phase Filter Coefficients
// Input 			:	None
void VP_SetFilter(void)
{
	u32 uHRatio, uVRatio;
	VP_eFILTER_H_PP eHFilter;
	VP_eFILTER_V_PP eVFilter;

	vpInp32(rVP_H_RATIO, uHRatio );
	vpInp32(rVP_V_RATIO, uVRatio );

	// 	For the real interlace mode, the vertical ratio should be used after divided by 2.
	//	Because in the interlace mode, all the VP output is used for SDOUT display
	//	and it should be the same as one field of the progressive mode. 
	//	Therefore the same filter coefficients should be used for the same the final output video.
	//	When half of the interlace V_RATIO is same as the progressive V_RATIO,
	//	the final output video scale is same. (20051104,ishan)

	//Horizontal Y 8tap 
	//Horizontal C 4tap	
	if (uHRatio <= (0x1<<16))			// 720->720 or zoom in
		eHFilter = VP_PP_H_NORMAL;
	else if (uHRatio <= (0x9<<13))		// 720->640
		eHFilter = VP_PP_H_8_9 ;
	else if(uHRatio <= (0x1<<17))		// 2->1
		eHFilter = VP_PP_H_1_2;
	else if(uHRatio <= (0x3<<16))		// 2->1	
		eHFilter = VP_PP_H_1_3;
	else // 0x800						// 4->1
		eHFilter = VP_PP_H_1_4;

	// Vertical Y 4tap
	if (uVRatio <= (0x1<<16))			// 720->720 or zoom in
		eVFilter = VP_PP_V_NORMAL;
	else if (uVRatio <= (0x3<<15))		//*6->5
		eVFilter = VP_PP_V_5_6;
	else if(uVRatio <= (0x5<<14))		// 4->3
		eVFilter = VP_PP_V_3_4;
	else if (uVRatio <= (0x1<<17))		// 2->1
		eVFilter = VP_PP_V_1_2;
	else if (uVRatio <= (0x3<<16))		// 3->1
		eVFilter = VP_PP_V_1_3;
	else // 0x400						// 4->1
		eVFilter = VP_PP_V_1_4;

	VP_SetPolyPhaseFilterSet( eHFilter, eVFilter );

}


//////////
// Function Name 	: 	VP_SetPolyPhaseFilterSet
// Description		:	Set the Poly-Phase Filter Coefficients
// Input 			:	eHFilter - refer to VP_FILTER_H_PP enum
//					eVFilter - refer to VP_FILTER_V_PP enum
void VP_SetPolyPhaseFilterSet(VP_eFILTER_H_PP eHFilter, VP_eFILTER_V_PP eVFilter)
{
	// Horizontal Y 8-tap
	VP_SetPolyPhaseFilter(rVP_POLY8_Y0_LL, sVp8tapCoef_Y_H+eHFilter*16*8, 8 );
	// Horizontal C 4-tap
	VP_SetPolyPhaseFilter(rVP_POLY4_C0_LL, sVp4tapCoef_C_H+eHFilter*16*4, 4 );
	// Vertical Y 4-tap
	VP_SetPolyPhaseFilter(rVP_POLY4_Y0_LL, sVp4tapCoef_Y_V+eVFilter*16*4, 4 );
}


//////////
// Function Name 	: 	VP_SetPolyPhaseFilter
// Description		:	Set the Poly-Phase Filter Coefficients
// Input 			:	uFilterReg - the Base Address of Poly-Phase Filter Coefficient registers.
//					sFilterCoef - the Base Address of Coefficient value array
//					usTapSz - Tap Size
void VP_SetPolyPhaseFilter(volatile u32 uFilterReg, const s8* sFilterCoef, u16 usTapSz)
{
	u32 i,j;
	u8* ucpFilterCoef;
	u16 usTempTapSz;
	volatile u32 uFilterBaseAddr;
	
	uFilterBaseAddr = uFilterReg;
	ucpFilterCoef = (u8*)sFilterCoef;

	for ( i=0; i<usTapSz; i++ )
	{
		// VP_POLYx_Y0/1(/2/3)_xx Setting
		usTempTapSz= usTapSz-i-1;
		
		for ( j=0; j<4; j++ )
		{
			// VP_POLYx_Yx_LL/LH/HL/HH Setting
			vpOutp32(uFilterBaseAddr, ( (ucpFilterCoef[4*j*usTapSz + usTempTapSz] <<24)
				| (ucpFilterCoef[(4*j+1)*usTapSz + usTempTapSz] <<16)
				| (ucpFilterCoef[(4*j+2)*usTapSz + usTempTapSz] <<8) 
				| (ucpFilterCoef[(4*j+3)*usTapSz + usTempTapSz]) ) );
			uFilterBaseAddr += 4;
		}
	}
}


void VP_SetStartAddress(u32 uTopYAddr, u32 uTopCAddr, u32 uBotYAddr, u32 uBotCAddr)
{
	vpOutp32(rVP_TOP_Y_PTR, uTopYAddr);
	vpOutp32(rVP_BOT_Y_PTR, uBotYAddr); 
	vpOutp32(rVP_TOP_C_PTR, uTopCAddr); 
	vpOutp32(rVP_BOT_C_PTR, uBotCAddr); 

	vpOutp32(rVP_SHADOW_UPDATA, 0x1);		
}


