
#include "system.h"
#include "util.h"
#include "Library.h"
#include "v210_sfr.h"
#include "intc.h"
#include "mixer.h"

#ifndef DISP_MIXER_REG
#define mxOutp32(addr, data) Outp32(addr, data)
#define mxInp32(addr, data)  Inp32_(addr, data)
#else
#define mxOutp32(addr, data) {UART_Printf("Outp32(\'h%08x, \'h%08x);\n", addr, data), Outp32(addr, data);}
#define mxInp32(addr, data) {Inp32_(addr, data); UART_Printf("Inp32(\'h%08x, d); // d=0x%08x\n", addr, data);}
#endif

enum MIXER_SFR
{
	// Mixer global setting
	rMIXER_STATUS         	= MIXER_BASE + 0x00,
	rMIXER_CFG            	= MIXER_BASE + 0x04,
                             
	// Mixer int.        
	rMIXER_INT_EN         	= MIXER_BASE + 0x08,
	rMIXER_INT_STATUS     	= MIXER_BASE + 0x0c,

	// Video & blender configuration
	rMIXER_LAYER_CFG      	= MIXER_BASE + 0x10,
	rMIXER_VIDEO_CFG      	= MIXER_BASE + 0x14,
	rMIXER_VIDEO_LIMITER	= MIXER_BASE + 0x18,

	// Graphic0 layer configuration
	rMIXER_GRAPH0_CFG     	= MIXER_BASE + 0x20,
	rMIXER_GRAPH0_BASE    	= MIXER_BASE + 0x24,
	rMIXER_GRAPH0_SPAN    	= MIXER_BASE + 0x28,
	rMIXER_GRAPH0_SXY     	= MIXER_BASE + 0x2c,
	rMIXER_GRAPH0_WH      	= MIXER_BASE + 0x30,
	rMIXER_GRAPH0_DXY     	= MIXER_BASE + 0x34,
	rMIXER_GRAPH0_BLANK   	= MIXER_BASE + 0x38,

	// Graphic1 layer configuration
	rMIXER_GRAPH1_CFG     	= MIXER_BASE + 0x40,
	rMIXER_GRAPH1_BASE    	= MIXER_BASE + 0x44,
	rMIXER_GRAPH1_SPAN    	= MIXER_BASE + 0x48,
	rMIXER_GRAPH1_SXY     	= MIXER_BASE + 0x4c,
	rMIXER_GRAPH1_WH      	= MIXER_BASE + 0x50,
	rMIXER_GRAPH1_DXY     	= MIXER_BASE + 0x54,
	rMIXER_GRAPH1_BLANK   	= MIXER_BASE + 0x58,

	// Background layer configuration
	rMIXER_BG_CFG         	= MIXER_BASE + 0x60,
	rMIXER_BG_COLOR0      	= MIXER_BASE + 0x64,
	rMIXER_BG_COLOR1      	= MIXER_BASE + 0x68,
	rMIXER_BG_COLOR2      	= MIXER_BASE + 0x6c,	

	// Color space conversion coefficient
	rMIXER_CM_COEFF_Y     	= MIXER_BASE + 0x80,
	rMIXER_CM_COEFF_CB    	= MIXER_BASE + 0x84,
	rMIXER_CM_COEFF_CR    	= MIXER_BASE + 0x88,

	rMIXER_STATUS_S         	= MIXER_BASE + 0x200,
};

static IMG_RESOLUTION g_eImgSz = SD480;
//static TV_STANDARDS g_eTvStrd;
static u32 g_uImgHsz = SD480_HSIZE;
//static u32 g_uImgVsz = SD480_VSIZE;
static u32 g_uDstHsz = SD480_HSIZE;
static u32 g_uDstVsz = SD480_VSIZE;
oMX_Source oMx_SrcInf[2];
oMX_Destination oMx_DstInf[2]; 


void __irq MIXER_IRQHandler(void)
{
	MX_INTR_SRC eIntStatus;
	
	UART_Printf("MIXER_IRQHandler\n");

	MIXER_GetIntrSrc(&eIntStatus);

	switch(eIntStatus)
	{
		case VP_UNDERRUN	:		UART_Printf("Occured VP Under-Run Interrupt!!!!\n");
									//Pending Clear
									mxOutp32(rMIXER_INT_STATUS, 1<<VP_UNDERRUN);
									break;
		case GRP0_UNDERRUN	:		UART_Printf("Occured GRAPHIC0 Under-Run Interrupt!!!!\n");
									//Pending Clear
									mxOutp32(rMIXER_INT_STATUS, 1<<GRP0_UNDERRUN);
									break;
		case GRP1_UNDERRUN	:		UART_Printf("Occured GRAPHIC1 Under-Run Interrupt!!!!\n");
									//Pending Clear
									mxOutp32(rMIXER_INT_STATUS, 1<<GRP1_UNDERRUN);
									break;
		case LINEBUF_UNDERRUN	:	UART_Printf("Occured Line Buffer Under-Run Interrupt!!!!\n");
									//Pending Clear
									mxOutp32(rMIXER_INT_STATUS, 1<<LINEBUF_UNDERRUN);
									break;
		default	:					UART_Printf("Invalid Interrupt!!!!\n");	
									break;

	}

	INTC_ClearVectAddr();
}


//////////
// Function Name 	: 	MIXER_InitIP
// Description		:	Initialize the Mixer 
// Input 			:	eImgSz - refer to oVP_Source
//					eScanMode - refer to SCAN_MODE
//					eStrd - refer to TV_STANDARD
//					eOutSel - refer to MX_OUTSEL
void MIXER_InitIP(IMG_RESOLUTION eImgSz, SCAN_MODE eScanMode, TV_STANDARDS eStrd, MX_OUTSEL eOutSel)
{
	u32 uSfrVal;
	
	Assert(eImgSz==SD480 ||eImgSz==SD576 ||eImgSz==HD720 || eImgSz==HD1080);	//SD576 - PAL

	g_eImgSz = eImgSz;
//	g_eTvStrd = eStrd;

	// Common setting
	uSfrVal = 
		 (0x0<<7) // 16Beat burst mode
		|(0x0<<3) // Little endian
		|(0x1<<2) // Values set by user can be applied to the mixer operation after v_sync detected
		|(0x0<<1) // ARM cannot stop the clock for the mixer
		|(0x0<<0); // Mixer stop
	mxOutp32(rMIXER_STATUS, uSfrVal);

	uSfrVal =
		((eOutSel==TV_OUT)?(0x0<<7) : (0x1<<7))
		|((eImgSz==HD1080)?(0x1<<6) : (0x0<<6))
		|(0x0<<5) // Grph1 is hide
		|(0x0<<4) // Grph0 is hide
		|(0x0<<3) // Video is hide
		|((eScanMode==INTERLACED) ? (0x0<<2) : (0x1<<2)) // INTERLACED or PROGRESSIVE
		|(((eStrd==NTSC_M)||(eStrd==NTSC_443)||(eStrd==NTSC_J)||(eStrd==PAL_M)) ? (0x0<<1) : (0x1<<1)) // NTSC or PAL
		|(((eImgSz==SD480)||(eImgSz==SD576)) ? (0x0<<0) : (0x1<<0)); // SD or HD
	mxOutp32(rMIXER_CFG, uSfrVal);

	// Enable int.
	uSfrVal =
		 (0x1<<10) // Enable VP underflow int.
		|(0x1<<9)  // Enable line buffer of grph. win1 int.
		|(0x1<<8); // Enable line buffer of grph. win0 int.
	mxOutp32(rMIXER_INT_EN, uSfrVal);
	
	// Set video and grph. wins order : grph. win1 > grph. win0 > video layer
	uSfrVal = (0x0<<8) |(0x0<<4) |(0x0<<0); // Disable video/grph0/grph1 window
	mxOutp32(rMIXER_LAYER_CFG, uSfrVal);

	// Video limiter setting & disable
	MIXER_SetVideoLimiterOnOff(0xeb, 0x10, 0xf0, 0x10, 0);
		
	// Background setting
	uSfrVal = 
		 (0x0<<19) // Disable Cr dithering
		|(0x0<<18) // Disable Cb dithering
		|(0x0<<17) // Disable Y dithering
		|(0x0<<16) // Background color will be changed in horizontal direction
		|(0x0<<0); // Start position of second background color
	mxOutp32(rMIXER_BG_CFG, uSfrVal);
	
	// Background color : white
	//MIXER_SetBgColor((180<<16)|(128<<8)|(128<<0), (180<<16)|(128<<8)|(128<<0), (180<<16)|(128<<8)|(128<<0));
	// yellow
	//MIXER_SetBgColor((210<<16)|(16<<8)|(146<<0), (210<<16)|(16<<8)|(146<<0), (210<<16)|(16<<8)|(146<<0));
	// blue
	//MIXER_SetBgColor((41<<16)|(240<<8)|(110<<0), (41<<16)|(240<<8)|(110<<0), (41<<16)|(240<<8)|(110<<0));
	//Cyan
	MIXER_SetBgColor((170<<16)|(166<<8)|(16<<0), (170<<16)|(166<<8)|(16<<0), (170<<16)|(166<<8)|(16<<0));
	
	// CSC coeffiecient setting
	MIXER_SetCscVal(RGB_TO_YUV601_FR);

	// Interrupt Enable
	INTC_SetVectAddr(NUM_MIXER, MIXER_IRQHandler);
	INTC_Enable(NUM_MIXER);		
}


// [Initialize base of IP for Analog TV]
void MIXER_InitForATv(IMG_RESOLUTION eImgSz, SCAN_MODE eScanMode, TV_STANDARDS eStrd)
{
	//MIXER_InitIP(eImgSz, eScanMode, eStrd, TV_OUT);
	MIXER_InitIP(SD480, eScanMode, eStrd, TV_OUT);
	MIXER_SetCscVal(RGB_TO_YUV601_FR);
}


// [Initialize base of IP for Digital TV]
void MIXER_InitForDTv(IMG_RESOLUTION eImgSz, SCAN_MODE eScanMode, MX_RGB_FORMAT eMxCscInRgb)
{
	MX_CSC_TYPE eCscType;
	TV_STANDARDS eTvStand;

///	eCscType = ((eImgSz==SD480) || (eImgSz==SD576) ) ? RGB_TO_YUV601_FR : RGB_TO_YUV709_FR;
	eCscType = 	(eMxCscInRgb==RGB601_0_255 ) ? RGB_TO_YUV601_FR : 
				(eMxCscInRgb==RGB601_16_235) ? RGB_TO_YUV601_LR:
				(eMxCscInRgb==RGB709_0_255) ? RGB_TO_YUV709_FR : RGB_TO_YUV709_LR;

	if(eImgSz == SD480)
		eTvStand = NTSC_M;
	else if(eImgSz == SD576)
		eTvStand = PAL_N;
	else		// In case of HDTV, don't care
		eTvStand = NTSC_M;
		
	///rb1004...modify 090122
//	MIXER_InitIP(eImgSz, PROGRESSIVE, (TV_STANDARDS)0, HDMI_OUT);   // SCAN_MODE and TV_STANDARD argument is garbage in this func.
	MIXER_InitIP(eImgSz, eScanMode, (TV_STANDARDS)eTvStand, HDMI_OUT);   // SCAN_MODE and TV_STANDARD argument is garbage in this func.

	MIXER_SetCscVal(eCscType);
}


// [Start Processing]
void MIXER_StartProcessing(void)
{
	u32 uSfrVal;
	
	mxInp32(rMIXER_STATUS, uSfrVal);
	uSfrVal |= (0x1<<0);
	mxOutp32(rMIXER_STATUS, uSfrVal);
}


void MIXER_SetEndianMode(u32 uEndian)
{
	u32 uSfrVal;

	mxInp32(rMIXER_STATUS, uSfrVal);

	if(uEndian == MX_ENDIAN_BIG)
		uSfrVal |= (0x1<<3);
	else
		uSfrVal &= ~(0x1<<3);

	mxOutp32(rMIXER_STATUS, uSfrVal);
}


// [Stop Processing]
void MIXER_StopProcessing(void)
{
	u32 uSfrVal;
	
	mxInp32(rMIXER_STATUS, uSfrVal);
	uSfrVal &= ~(0x1<<0);
	mxOutp32(rMIXER_STATUS, uSfrVal);

	do
	{
		mxInp32(rMIXER_STATUS, uSfrVal);
	}while(!(uSfrVal & (0x1<<1)));
}


// [Set Order]
void MIXER_SetOrder(MIXER_WIN_ORDER eOrder)
{
	u32 uSfrVal;
	
	uSfrVal = 
		(eOrder == DN_VDW0W1_UP) ? ( (0x1<<0)|(0x2<<4)|(0x3<<8) ) :
		(eOrder == DN_VDW1W0_UP) ? ( (0x1<<0)|(0x3<<4)|(0x2<<8) ) :
		(eOrder == DN_W1VDW0_UP) ? ( (0x2<<0)|(0x3<<4)|(0x1<<8) ) :
		(eOrder == DN_W1W0VD_UP) ? ( (0x3<<0)|(0x2<<4)|(0x1<<8) ) :
		(eOrder == DN_W0W1VD_UP) ? ( (0x3<<0)|(0x1<<4)|(0x2<<8) ) :
		( (0x2<<0)|(0x1<<4)|(0x3<<8) ); // (eOrder == DN_W0VDW1_UP)
	mxOutp32(rMIXER_LAYER_CFG, uSfrVal);
}


// [Set background color]
void MIXER_SetBgColor(u32 uColor1, u32 uColor2, u32 uColor3)
{
	mxOutp32(rMIXER_BG_COLOR0, uColor1);
	mxOutp32(rMIXER_BG_COLOR1, uColor2);
	mxOutp32(rMIXER_BG_COLOR2, uColor3);
}


// [Initialize graphic window]
void MIXER_InitGrphWin(MIXER_GRPH_WIN eWin, oMX_Source oSrcArgs, oMX_Destination oDstArgs)
{
	u32 uBppSel;
	u32 uSfrAddr, uSfrVal; 
	
	Assert(oSrcArgs.uImgHsz>=oSrcArgs.uSrcXPos+oSrcArgs.uSrcHsz);
	Assert(oSrcArgs.uFbStAddr%4 == 0); // Base addr. should be word-aligned
	Assert(oSrcArgs.uImgHsz >= oSrcArgs.uSrcHsz);
	
	if (g_eImgSz == SD480)
	{
		Assert(oSrcArgs.uSrcXPos+oSrcArgs.uSrcHsz <= SD480_HSIZE);			
		Assert(oSrcArgs.uSrcYPos+oSrcArgs.uSrcVsz <= SD480_VSIZE);

		Assert(oDstArgs.uDstXPos+oSrcArgs.uSrcHsz <= SD480_HSIZE);	
		Assert(oDstArgs.uDstYPos+oSrcArgs.uSrcVsz <= SD480_VSIZE);
	}
	else if(g_eImgSz == SD576)
	{
		Assert(oSrcArgs.uSrcXPos+oSrcArgs.uSrcHsz <= SD576_HSIZE);				
		Assert(oSrcArgs.uSrcYPos+oSrcArgs.uSrcVsz <= SD576_VSIZE);

		Assert(oDstArgs.uDstXPos+oSrcArgs.uSrcHsz <= SD576_HSIZE);	
		Assert(oDstArgs.uDstYPos+oSrcArgs.uSrcVsz <= SD576_VSIZE);
	}
	else if(g_eImgSz == HD720)
	{
		// X/Y pos and H/V size of src. constriction		
		Assert(oSrcArgs.uSrcXPos+oSrcArgs.uSrcHsz <= HD720_HSIZE);		
		Assert(oSrcArgs.uSrcYPos+oSrcArgs.uSrcVsz <= HD720_VSIZE);

		// X/Y pos and H/V size of src. constriction		
		Assert(oDstArgs.uDstXPos+oSrcArgs.uSrcHsz <= HD720_HSIZE);		
		Assert(oDstArgs.uDstYPos+oSrcArgs.uSrcVsz <= HD720_VSIZE);
	}
	else		//HD1080
	{
		// X/Y pos and H/V size of src. constriction		
		Assert(oSrcArgs.uSrcXPos+oSrcArgs.uSrcHsz <= HD1080_HSIZE);		
		Assert(oSrcArgs.uSrcYPos+oSrcArgs.uSrcVsz <= HD1080_VSIZE);

		// X/Y pos and H/V size of src. constriction		
		Assert(oDstArgs.uDstXPos+oSrcArgs.uSrcHsz <= HD1080_HSIZE);		
		Assert(oDstArgs.uDstYPos+oSrcArgs.uSrcVsz <= HD1080_VSIZE);
	}

	g_uImgHsz = oSrcArgs.uImgHsz;
	g_uDstHsz = oSrcArgs.uSrcHsz;
	g_uDstVsz = oSrcArgs.uSrcVsz;
	
	uBppSel = 
		(oSrcArgs.eBpp == RGB16) ? (0x4<<8) :
		(oSrcArgs.eBpp == ARGB1555) ? (0x5<<8) :
		(oSrcArgs.eBpp == ARGB4444) ? (0x6<<8) : (0x7<<8); // eBpp == MX_ARGB32_8888
		
	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_CFG : rMIXER_GRAPH1_CFG;		
	uSfrVal =
		 (0x1<<21) // No modification is applied to the input graphic layer's pixel data
		//|( (oSrcArgs.eBpp == ARGB8888) ? (0x1<<20) : (0x0<<20) ) 
		|(0x0<<20)
		|(0x0<<17) // Disable per-plane blending
		//|( (oSrcArgs.eBpp == ARGB8888) ? (0x1<<16) : (0x0<<16) ) 
		|(0x0<<16)
		|uBppSel
		|(0x0<<0); // Alpha value
	mxOutp32(uSfrAddr, uSfrVal);

	// Window order : grph win 0 < grph win 1
	mxInp32(rMIXER_LAYER_CFG, uSfrVal);
	uSfrVal |= (0x1<<4)	|(0x2<<8);
	mxOutp32(rMIXER_LAYER_CFG, uSfrVal);
	
	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_BASE : rMIXER_GRAPH1_BASE;
	mxOutp32(uSfrAddr, oSrcArgs.uFbStAddr);

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_SPAN : rMIXER_GRAPH1_SPAN;
	mxOutp32(uSfrAddr, oSrcArgs.uImgHsz); 

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_WH : rMIXER_GRAPH1_WH;
	mxOutp32(uSfrAddr,	((oSrcArgs.uSrcHsz&0x7ff)<<16)|((oSrcArgs.uSrcVsz&0x7ff)<<0));

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_SXY : rMIXER_GRAPH1_SXY;
	mxOutp32(uSfrAddr, ((oSrcArgs.uSrcXPos&0x7ff)<<16)|((oSrcArgs.uSrcYPos&0x7ff)<<0x0));

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_DXY : rMIXER_GRAPH1_DXY;
	mxOutp32(uSfrAddr, ((oDstArgs.uDstXPos&0x7ff)<<16)|((oDstArgs.uDstYPos&0x7ff)<<0x0));

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_BLANK : rMIXER_GRAPH1_BLANK;
	mxOutp32(uSfrAddr, 0x0);

}


// [Initialize and set grapnic window on]
void MIXER_InitAndSetGrphWinOn(MIXER_GRPH_WIN eWin, oMX_Source oSrcArgs, oMX_Destination oDstArgs)
{
	MIXER_InitGrphWin(eWin, oSrcArgs, oDstArgs);
	MIXER_SetGrphWinOnOff(eWin, 1);	
}


// [Set graphic window on/off]
void MIXER_SetGrphWinOnOff(MIXER_GRPH_WIN eWin, u32 uOnOff)
{
	u32 uCfgVal, uLayerCfgVal;
	
	Assert(uOnOff == 0 || uOnOff == 1);

	mxInp32(rMIXER_CFG, uCfgVal);
	mxInp32(rMIXER_LAYER_CFG, uLayerCfgVal);

	if(uOnOff == 0)
	{
		if (eWin == MX_WIN0)
			uCfgVal = uCfgVal&(~(0x1<<4));
		else
			uCfgVal = uCfgVal&(~(0x1<<5));
	}
	else
	{
		if (eWin == MX_WIN0)
		{
			uCfgVal = uCfgVal|(0x1<<4);
			if(!(uLayerCfgVal&(0xf<<4)))
				uLayerCfgVal |= (0x1<<4);
		}			
		else
		{
			uCfgVal = uCfgVal|(0x1<<5);
			if(!(uLayerCfgVal&(0xf<<8)))
				uLayerCfgVal |= (0x1<<8);
		}
	}

	mxOutp32(rMIXER_CFG,  uCfgVal);
	mxOutp32(rMIXER_LAYER_CFG, uLayerCfgVal);
}


// [Set all graphic windows on/off]
void MIXER_SetAllGrphWinsOnOff(u32 uOnOff)
{
	u32 uSfrVal;
	
	Assert(uOnOff == 0 || uOnOff == 1);	

	mxInp32(rMIXER_CFG, uSfrVal);
	uSfrVal = (uOnOff == 0) ? uSfrVal&(~(0x3<<4)) : uSfrVal|(0x3<<4);		
	mxOutp32(rMIXER_CFG, uSfrVal);

	mxInp32(rMIXER_LAYER_CFG, uSfrVal);
	uSfrVal = (uOnOff==0) ? uSfrVal&(~(0xf<<4))&(~(0xf<<8)) : uSfrVal|((0x1<<4))|((0x1<<8));	
	mxOutp32(rMIXER_LAYER_CFG,  uSfrVal);	
}

// [Set per-pixel blending on/off]
void MIXER_SetPerPixelBlendingOnOff(MIXER_GRPH_WIN eWin, u32 uOnOff)
{
	u32 uSfrAddr, uSfrVal;

	Assert(uOnOff == 0 ||uOnOff == 1);

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_CFG : rMIXER_GRAPH1_CFG;
	mxInp32(uSfrAddr, uSfrVal);
	uSfrVal = (uOnOff == 0) ? (uSfrVal&(~(0x1<<16))) : (uSfrVal|(0x1<<16));
	mxOutp32(uSfrAddr, uSfrVal);
}


void MIXER_SetPerPreMul_PixelBlendingOnOff(MIXER_GRPH_WIN eWin, u32 uOnOff)
{
	u32 uSfrAddr, uSfrVal;

	Assert(uOnOff == 0 ||uOnOff == 1);

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_CFG : rMIXER_GRAPH1_CFG;
	mxInp32(uSfrAddr, uSfrVal);
	uSfrVal = (uOnOff == 0) ? (uSfrVal&(~(0x1<<20))&(~(0x1<<16))) : (uSfrVal|(0x1<20)|(0x1<<16));
	mxOutp32(uSfrAddr, uSfrVal);
}



// [Set per-plane blending on]
void MIXER_SetPerPlaneBlendingOnOff(MIXER_GRPH_WIN eWin, u8 ucAlphaVal, u32 uOnOff)
{
	u32 uSfrAddr, uSfrVal;

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_CFG : rMIXER_GRAPH1_CFG;
	mxInp32(uSfrAddr, uSfrVal);
	if(uOnOff == 1)
	{
		uSfrVal &= ~(0xff<<0);
		uSfrVal |= ((0x1<<17) | (ucAlphaVal<<0));
	}
	else
	{
		uSfrVal &= (~(0x1<<17));
	}
	mxOutp32(uSfrAddr, uSfrVal);
}


// [Set color key on]
void MIXER_SetColorKeyOnOff(MIXER_GRPH_WIN eWin, u32 uColorKeyVal, u32 uOnOff)
{
	u32 uSfrAddr, uSfrVal;

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_CFG : rMIXER_GRAPH1_CFG;
	mxInp32(uSfrAddr, uSfrVal);
	
	uSfrVal = (uOnOff == 0) ? uSfrVal|(0x1<<21) : uSfrVal&(~(0x1<<21));
	mxOutp32(uSfrAddr, uSfrVal);

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_BLANK : rMIXER_GRAPH1_BLANK;
	if(uOnOff == 0)
		mxOutp32(uSfrAddr, 0);
	else
		mxOutp32(uSfrAddr, uColorKeyVal);
}


// [Set video window on/off]
void MIXER_SetVdWinOnOff(u32 uOnOff)
{
	u32 uSfrVal;

	Assert(uOnOff == 0 || uOnOff == 1);

	mxInp32(rMIXER_CFG, uSfrVal);
	uSfrVal = (uOnOff == 0) ? (uSfrVal&(~(0x1<<3))) : (uSfrVal|(0x1<<3));
	mxOutp32(rMIXER_CFG, uSfrVal);

	mxInp32(rMIXER_LAYER_CFG, uSfrVal);
	uSfrVal = (uOnOff == 0) ? (uSfrVal&(~(0xf<<0))) : (uSfrVal|(0x1<<0));
	mxOutp32(rMIXER_LAYER_CFG, uSfrVal);
}


// [Set video blending on]
void MIXER_SetVdBlendingOnOff(u8 ucAlphaVal, u32 uOnOff)
{
	u32 uSfrVal;

	mxInp32(rMIXER_VIDEO_CFG, uSfrVal);
	if(uOnOff == 0)
		uSfrVal &= ~(0x1<<16);
	else
		uSfrVal = (0x1<<16) | (ucAlphaVal<<0);
	mxOutp32(rMIXER_VIDEO_CFG, uSfrVal);
}


// [Set position]
void MIXER_SetPosition(MIXER_GRPH_WIN eWin, u32 uDstXPos, u32 uDstYPos)
{
	u32 uSfrAddr, uSfrVal;
	
	if (g_eImgSz==SD480)
	{
		Assert(uDstXPos+g_uDstHsz <= SD480_HSIZE);
		Assert(uDstYPos+g_uDstVsz <= SD480_VSIZE);		
	}
	else if(g_eImgSz==SD576)
	{
		Assert(uDstXPos+g_uDstHsz <= SD576_HSIZE);
		Assert(uDstYPos+g_uDstVsz <= SD576_VSIZE);		
	}
	else if(g_eImgSz== HD720)
	{
		Assert(uDstXPos+g_uDstHsz <= HD720_HSIZE);
		Assert(uDstYPos+g_uDstVsz <= HD720_VSIZE);		
	}
	else		//HD1080
	{
		Assert(uDstXPos+g_uDstHsz <= HD1080_HSIZE);
		Assert(uDstYPos+g_uDstVsz <= HD1080_VSIZE);		
	}

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_DXY : rMIXER_GRAPH1_DXY;
	uSfrVal = ((uDstXPos&0x7ff)<<16)|((uDstYPos&0x7ff)<<0x0);
	mxOutp32(uSfrAddr, uSfrVal);
}


// [Set cropping]
void MIXER_SetCropping(	MIXER_GRPH_WIN eWin, u32 uSrcXPos, u32 uSrcYPos, u32 uCroppedHsz, u32 uCroppedVsz,
							u32 uDstXPos, u32 uDstYPos)
{
	u32 uSfrAddr, uSfrVal;

	if (g_eImgSz==SD480)
	{
		Assert(uDstXPos+uCroppedHsz <= SD480_HSIZE);
		Assert(uDstYPos+uCroppedVsz <= SD480_VSIZE);		
	}
	else if(g_eImgSz==SD576)
	{
		Assert(uDstXPos+uCroppedHsz <= SD576_HSIZE);
		Assert(uDstYPos+uCroppedVsz <= SD576_VSIZE);		
	}
	else if(g_eImgSz==HD720)
	{
		Assert(uDstXPos+uCroppedHsz <= HD720_HSIZE);
		Assert(uDstYPos+uCroppedVsz <= HD720_VSIZE);		
	}
	else //if(g_eImgSz==HD1080)
	{
		Assert(uDstXPos+uCroppedHsz <= HD1080_HSIZE);
		Assert(uDstYPos+uCroppedVsz <= HD1080_VSIZE);		
	}
		
	Assert(uSrcXPos+uCroppedHsz <= g_uImgHsz);

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_SXY : rMIXER_GRAPH1_SXY;
	mxInp32(uSfrAddr, uSfrVal);
	uSfrVal = ((uSrcXPos&0x7ff)<<16)|((uSrcYPos&0x7ff)<<0x0);
	mxOutp32(uSfrAddr, uSfrVal);

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_WH : rMIXER_GRAPH1_WH;
	mxInp32(uSfrAddr, uSfrVal);
	uSfrVal = ((uCroppedHsz&0x7ff)<<16)|((uCroppedVsz&0x7ff)<<0);
	mxOutp32(uSfrAddr, uSfrVal);

	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_DXY : rMIXER_GRAPH1_DXY;
	mxInp32(uSfrAddr, uSfrVal);
	uSfrVal = ((uDstXPos&0x7ff)<<16)|((uDstYPos&0x7ff)<<0x0);
	mxOutp32(uSfrAddr, uSfrVal);
}


void MIXER_SetCscVal(MX_CSC_TYPE eMxCscType)
{
	if(eMxCscType == RGB_TO_YUV601_LR)
	{	
		mxOutp32(rMIXER_CM_COEFF_Y, (0<<30)|(153<<20)|(300<<10)|(58<<0));
		mxOutp32(rMIXER_CM_COEFF_CB, (936<<20)|(851<<10)|(262<<0));
		mxOutp32(rMIXER_CM_COEFF_CR, (262<<20)|(805<<10)|(982<<0));		
	}
	else if(eMxCscType == RGB_TO_YUV601_FR)
	{
		mxOutp32(rMIXER_CM_COEFF_Y,  (1<<30)|(132<<20)|(258<<10)|(50<<0));
		mxOutp32(rMIXER_CM_COEFF_CB, (949<<20)|(876<<10)|(225<<0));
		mxOutp32(rMIXER_CM_COEFF_CR, (225<<20)|(836<<10)|(988<<0));	
	}
	else if (eMxCscType == RGB_TO_YUV709_LR)
	{
		mxOutp32(rMIXER_CM_COEFF_Y,  (0<<30)|(109<<20)|(366<<10)|(36<<0));
		mxOutp32(rMIXER_CM_COEFF_CB, (964<<20)|(822<<10)|(216<<0));
		mxOutp32(rMIXER_CM_COEFF_CR, (262<<20)|(787<<10)|(1000<<0));
	}
	else // eMxCscType == RGB_TO_YUV709_FR
	{		
		mxOutp32(rMIXER_CM_COEFF_Y,  (1<<30)|(94<<20)|(314<<10)|(32<<0));
		mxOutp32(rMIXER_CM_COEFF_CB, (972<<20)|(851<<10)|(225<<0));
		mxOutp32(rMIXER_CM_COEFF_CR, (225<<20)|(820<<10)|(1004<<0));
	}

	//must be added YCbCr wide range..rb1004
}


void MIXER_GetIntrSrc(MX_INTR_SRC* eIntStat)
{
	u32 uStatus;
		
	mxInp32(rMIXER_INT_STATUS, uStatus);
	UART_Printf("uStatus: 0x%x\n", uStatus);

	if (uStatus & (1<<10))
	{
		*eIntStat = VP_UNDERRUN;
	}
	else if (uStatus & (1<<8))
	{
		*eIntStat = GRP0_UNDERRUN;
	}
	else if (uStatus & (1<<9))
	{
		*eIntStat = GRP1_UNDERRUN;
	}
	else if(uStatus & (1<<0))
	{
		*eIntStat = LINEBUF_UNDERRUN;
	}
	else
	{
		Assert(0);
	}
}

void MIXER_ClearPending(void)
{
	mxOutp32(rMIXER_INT_STATUS, 0x401);
}


void MIXER_SetVideoLimiterOnOff(u32 uY_Upper, u32 uY_Lower, u32 uC_Upper, u32 uC_Lower, u32 uOnOff)
{
	u32 uSfrVal;
	
	Assert( (uY_Upper<=255) && (uY_Lower<=255) && (uC_Upper<=255) && (uC_Lower<=255));
	Assert((uOnOff == 0) || (uOnOff == 1));
	
	mxOutp32(rMIXER_VIDEO_LIMITER, (uY_Upper<<24)|(uY_Lower<<16)|(uC_Upper<<8)|(uC_Lower<<0));

	mxInp32(rMIXER_VIDEO_CFG, uSfrVal );
	uSfrVal = (uSfrVal &~(1<<17))|(uOnOff<<17);
	mxOutp32(rMIXER_VIDEO_CFG, uSfrVal);	
}



void MIXER_SetScalingRatio(MIXER_GRPH_WIN eWin, MX_SCALE_RATIO eScale_H, MX_SCALE_RATIO eScale_V)
{
	u32 uSfrAddr, uSfrVal;
	
	uSfrAddr = (eWin == MX_WIN0) ? rMIXER_GRAPH0_WH : rMIXER_GRAPH1_WH;

	mxInp32(uSfrAddr, uSfrVal);
	
	//uSfrVal = (uSfrVal&(~(0x3<<28))&(~(0x3<<12))) | ((u32)eScale_V<<28) | ((u32)eScale_H<<12);
	uSfrVal = (uSfrVal&(~(0x3<<28))&(~(0x3<<12))) | ((u32)eScale_H<<28) | ((u32)eScale_V<<12);
	mxOutp32(uSfrAddr, uSfrVal);
}


void MIXER_SetRGBColorFormat(MX_RGB_OUT_TYPE eRGBOut, MX_RGB_FORMAT eRGBOutFormat)
{
	u32 uSfrVal;

	mxInp32(rMIXER_CFG, uSfrVal);

	uSfrVal = (uSfrVal & ~(0x7<<8)) | ((u32)eRGBOutFormat<<9) | ((u32)eRGBOut<<8);
	
	mxOutp32(rMIXER_CFG, uSfrVal);
}

