/**
 * File Name : i2c_hdmi_phy.c
 *
 * File Description :
 * This file implements the I2C function for communication between the hdmi link layer and phy layer
 *
 * Author : Hee Myung Noh
 * Dept : AP Development
 * Created Date : 2009/01/22
 * Version : 0.1
 * History 
 *  - Initiate version (Hee Myung Noh 090122)
 */

#include "system.h"
#include "Library.h"
#include "v210_sfr.h"
#include "intc.h"
#include "phy.h"
#include "i2c_hdmi.h"

//#define	DISP_I2C_HDMI_PHY_REG

#ifndef DISP_I2C_HDMI_PHY_REG
#define I2C_HdmiOutp8(addr, data) 	Outp8(addr, data)
#define I2C_HdmiInp8(addr)  			Inp8(addr)
#else
#define I2C_HdmiOutp8(addr, data) 	{UART_Printf("Outp8(\'h%08x, \'h%08x);\n", addr, data); Outp8(addr, data);}
#define I2C_HdmiInp8(addr) 			Inp8(addr)
#endif


static struct {
	s32    state;
	u8 	*buffer;
	s32    bytes;
} i2c_hdmi_context;

typedef enum I2C_HDMI_PHY_SFR
{
	rI2C_HDMI_CON		=	(I2C_HDMI_PHY_BASE + 0x00),
	rI2C_HDMI_STAT		=	(I2C_HDMI_PHY_BASE + 0x04),
	rI2C_HDMI_ADD		=	(I2C_HDMI_PHY_BASE + 0x08),
	rI2C_HDMI_DS		=	(I2C_HDMI_PHY_BASE + 0x0c),
	rI2C_HDMI_LC		=	(I2C_HDMI_PHY_BASE + 0x10)
}I2C_HDMI_PHY_SFR;


static s32 I2CHDMI_Release(void);

/**
 * Polling interrupt 
 *
 * @return  On success, return OK;Otherwise return, ERROR.
 */
static s32 I2CHDMI_InterruptWait(void)
{
	u8 ucStatus, ucReg;
	s32 retval = OK;

	do
	{
		ucStatus = I2C_HdmiInp8(rI2C_HDMI_CON);
		if (ucStatus & I2C_PEND)
		{
			// check status flags
			ucReg = I2C_HdmiInp8(rI2C_HDMI_STAT);

#if 0
			if (ucReg & (I2C_STAT_ARBITRERR | I2C_STAT_ADDRASSLAVE |I2C_STAT_ADDRZERO | I2C_STAT_ACKERROR))
			{
				if (ucReg & I2C_STAT_ARBITRERR)
				{
					UART_Printf("I2C_STAT_ARBITRERR\n");
				}
				else if (ucReg & I2C_STAT_ADDRASSLAVE)
				{
					UART_Printf("I2C_STAT_ADDRASSLAVE\n");
				}
				else if (ucReg & I2C_STAT_ADDRZERO)
				{
					UART_Printf("I2C_STAT_ADDRZERO\n");
				}
				else if (ucReg & I2C_STAT_ACKERROR)
				{
					UART_Printf("I2C_STAT_ACKERROR\n");
				}
				I2CHDMI_Release();
				retval = ERROR;
			}
#endif			
			break;
		}
	}while (1);

	return retval;
}

/**
 *  Initialize I2C library
 */
s32 I2CHDMI_Init(void)
{
	UART_Printf("[I2CHDMI_Init]\n");
	
	// I2C Filter Setting
	I2C_HdmiOutp8(rI2C_HDMI_LC, 0x5);
	
	return OK;
}

/**
 * Read data through I2C
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
s32 I2CHDMI_Read(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	u32 i, uProcess = TRUE;

	// set state
	i2c_hdmi_context.state = STATE_RX_ADDR;

	// set parameters
	i2c_hdmi_context.buffer = pBuffer;
	i2c_hdmi_context.bytes = ucBytes;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_ACK); // 0xE1
	// to make IDSR write-enabled
	I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_ENABLE | I2C_MODE_MRX);
	// put address to shift register
	I2C_HdmiOutp8(rI2C_HDMI_DS, ucAddr & 0xFE);
	// master rx mode + start + enable - "START" sequence generation
	I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_ENABLE | I2C_START | I2C_MODE_MRX); // 0xB0

	while (uProcess)
	{
		//In case of Master RX Mode, Last data is not ack. but set the "ACK not received" bit
		if(i2c_hdmi_context.state != STATE_RX_STOP)
		{
			if (I2CHDMI_InterruptWait() != OK)
			{
				UART_Printf("InterruptWait() failed!!!\n");
				iRetval = ERROR;
				break;
			}
		}

		switch (i2c_hdmi_context.state)
		{
			case STATE_RX_DATA:
				// read byte
				*(i2c_hdmi_context.buffer) = I2C_HdmiInp8(rI2C_HDMI_DS);
				i2c_hdmi_context.buffer++;
				--(i2c_hdmi_context.bytes);
				            
				if (i2c_hdmi_context.bytes == 1)
				{
					i2c_hdmi_context.state = STATE_RX_STOP;

					//rb1004...removed
					//I2C_HdmiOutp8(rI2C_HDMI_LC, I2C_NO_ACK);

					//rb1004
					// clear interrupt pending flag and interrupt clear bit
					//I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK);    
					I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);    
				}
				else
				{
					// clear interrupt pending flag and interrupt clear bit
					I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK);    
					//I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
				}

				//for(i=0 ; i<10000 ; i++);
				break;

			case STATE_RX_ADDR:
				i2c_hdmi_context.state = STATE_RX_DATA;

				if (i2c_hdmi_context.bytes == 1)
				{
					i2c_hdmi_context.state = STATE_RX_STOP;

					//rb1004...removed
					//I2C_HdmiOutp8(rI2C_HDMI_LC, I2C_NO_ACK);
					// clear interrupt pending flag and interrupt clear bit
					I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);  
				}
				else
				{
					// clear interrupt pending flag and interrupt clear bit
					I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK);    
				}
				break;

			case STATE_RX_STOP:
				i2c_hdmi_context.state = STATE_IDLE;

				for(i=0 ; i<10000 ; i++);
				
				*(i2c_hdmi_context.buffer) = I2C_HdmiInp8(rI2C_HDMI_DS);
				
				//rb1004...removed
				//I2C_HdmiOutp8(rI2C_HDMI_LC, I2C_STOP);
				//"STOP" sequence generation
				I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_MODE_MRX|I2C_ENABLE); 
				// clear interrupt pending flag and interrupt clear bit
				I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);

				I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_MODE_MRX);

				while( I2C_HdmiInp8(rI2C_HDMI_STAT) & (1<<5) );
				///for(i=0 ; i<10000 ; i++);	//Wait until the stop condition takes effect
				uProcess = FALSE;
				break;

			case STATE_IDLE:
				default:
				UART_Printf("error state!!!\n");
				iRetval = ERROR;
				uProcess = FALSE;
				break;
		}
	}

	return iRetval;
}

/**
 * Write data through I2C
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
 s32 I2CHDMI_Write(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	u32 uProcess = TRUE;

	// set state
	i2c_hdmi_context.state = STATE_TX_ADDR;

	// set parameters
	i2c_hdmi_context.buffer = pBuffer;
	i2c_hdmi_context.bytes = ucBytes;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_ACK); // 0xE1
	// to make IDSR write-enabled
	I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_ENABLE | I2C_MODE_MTX);
	// put address to shift register
	I2C_HdmiOutp8(rI2C_HDMI_DS, ucAddr & 0xFE);
	// master tx mode + start + enable - "START" sequence generation
	I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_ENABLE | I2C_START | I2C_MODE_MTX); // 0xF0
	
	while (uProcess)
	{
		if (I2CHDMI_InterruptWait() != OK)
		{
			UART_Printf("I2CHDMIInterruptWait() failed!!!\n");
			iRetval = ERROR;
			break;
		}

		switch (i2c_hdmi_context.state)
		{
			case STATE_TX_ADDR:
			case STATE_TX_DATA:
				i2c_hdmi_context.state = STATE_TX_DATA;

				I2C_HdmiOutp8(rI2C_HDMI_DS, *(i2c_hdmi_context.buffer));
///				for(i=0 ; i<50 ; i++);
				
				i2c_hdmi_context.buffer++;
				--(i2c_hdmi_context.bytes);

				if (i2c_hdmi_context.bytes == 0)
				{
					i2c_hdmi_context.state = STATE_TX_STOP;
					I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
				}
				else
					// clear interrupt pending flag and interrupt clear bit
					//I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
					I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK);
				break;

			case STATE_TX_STOP:
				i2c_hdmi_context.state = STATE_IDLE;

				// send stop bit
				I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_MODE_MTX | I2C_ENABLE); 
///				for(i=0 ; i<50 ; i++);
				//I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_MODE_MTX); 
				
				// clear interrupt pending flag and interrupt clear bit
				I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
				//I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK);
				
				 // disable Rx/Tx and set to "not busy" - "STOP" sequence generation
				I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_MODE_MTX); 

				while( I2C_HdmiInp8(rI2C_HDMI_STAT) & (1<<5) );
				
				uProcess = FALSE;
				break;

			case STATE_IDLE:
			default:
				UART_Printf("error state!!!\n");
				iRetval = ERROR;
				uProcess = FALSE;
				break;
		}
	}

	return iRetval;
}

/**
 * Release I2C line
 */
static s32 I2CHDMI_Release(void)
{
	//rb1004...removed
	//I2C_HdmiOutp8(rI2C_HDMI_LC, I2C_STOP);
	// clear interrupt pending flag and interrupt clear bit
	I2C_HdmiOutp8(rI2C_HDMI_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
	// disable 
	I2C_HdmiOutp8(rI2C_HDMI_STAT, I2C_IDLE);

	return OK;
}
