/**
 * File Name : i2c_ddc.c
 *
 * File Description :
 * This file implements the I2C function for EDDC
 *
 * Author : Hee Myung Noh
 * Dept : AP Development
 * Created Date : 2009/01/22
 * Version : 0.1
 * History 
 *  - Initiate version (Hee Myung Noh 090122)
 */

#include "system.h"
#include "Library.h"
#include "v210_sfr.h"
#include "intc.h"
#include "i2c_hdmi.h"
#include "gpio.h"
#include "edid.h"
//#include "i2c_ddc.h"


//#define	DISP_I2C_DDC_REG

#ifndef DISP_I2C_DDC_REG
#define I2C_DdcOutp8(addr, data) 		Outp8(addr, data)
#define I2C_DdcInp8(addr)  			Inp8(addr)
#else
#define I2C_DdcOutp8(addr, data) {UART_Printf("Outp8(\'h%08x, \'h%08x);\n", addr, data), Outp8(addr, data);}
#define I2C_DdcInp8(addr) {Inp8(addr); UART_Printf("Inp8(\'h%08x, d); // d=0x%08x\n", addr, Inp8(addr));}
#endif


///static EventGroup gI2CDDCEventGroup;
///static Mutex gI2CDDCMutex;
#define EVENT_I2C_DDC_NO	  	(0)
#define EVENT_I2C_DDC_DONE  	(1<<0)
#define EVENT_I2C_DDC_ERROR	(1<<1)

static u32 g_DDC_EventStatus = EVENT_I2C_DDC_NO;

static struct 
{
	s32		state;
	u8		*buffer;
	s32		bytes;
	u8		offset;
	u8		addr;
	u8		segnum;
} i2c_ddc_context;

typedef enum I2C_DDC_SFR
{
	rI2C_DDC_CON			=	(I2C_HDMI_DDC_BASE + 0x00),
	rI2C_DDC_STAT			=	(I2C_HDMI_DDC_BASE + 0x04),
	rI2C_DDC_ADD			=	(I2C_HDMI_DDC_BASE + 0x08),
	rI2C_DDC_DS			=	(I2C_HDMI_DDC_BASE + 0x0C),
	rI2C_DDC_LC			=	(I2C_HDMI_DDC_BASE + 0x10)
}I2C_DDC_SFR;

static s32 I2CDDC_Release(void);

/**
 * DDC interrupt service routine
 */
void __irq I2CDDC_IRQHandler(void)
{
	u8 ucStatus;
	u8 ucReg;
	s32 sCont = TRUE;	

	// check interrupt reason
	ucStatus = I2C_DdcInp8(rI2C_DDC_CON);

	if (ucStatus & I2C_PEND)
	{
		// check Status flags
		ucReg = I2C_DdcInp8(rI2C_DDC_STAT);

		while (ucReg & (I2C_STAT_ARBITRERR | I2C_STAT_ADDRASSLAVE |	I2C_STAT_ADDRZERO | I2C_STAT_ACKERROR))
		{
			sCont = FALSE;
			if (ucReg & I2C_STAT_ARBITRERR)
			{
				UART_Printf ("I2C_STAT_ARBITRERR\n");
			}
			else if (ucReg & I2C_STAT_ADDRASSLAVE)
			{
				UART_Printf ("I2C_STAT_ADDRASSLAVE\n");
			}
			else if (ucReg & I2C_STAT_ADDRZERO)
			{
				UART_Printf ("I2C_STAT_ADDRZERO\n");
			}
			else if (ucReg & I2C_STAT_ACKERROR)
			{
				if (	i2c_ddc_context.state == STATE_RX_STOP ||
					i2c_ddc_context.state == STATE_TX_EDDC_SEGADDR ||
					((i2c_ddc_context.state == STATE_TX_EDDC_SEGNUM) && (i2c_ddc_context.segnum == 0)))
				{
					sCont = TRUE;
					break;
				}
				else
				{
					UART_Printf ("I2C_STAT_ACKERROR : %d\n", i2c_ddc_context.state);
				}
			}
			I2CDDC_Release();
			///            EventPost(&gI2CDDCEventGroup, EVENT_I2C_DDC_ERROR);
			g_DDC_EventStatus = EVENT_I2C_DDC_ERROR;
			break;
		}

		if (sCont)
		{
			switch (i2c_ddc_context.state)
			{
				case STATE_TX_EDDC_SEGADDR:
					i2c_ddc_context.state = STATE_TX_EDDC_SEGNUM;

					// write segment number
					I2C_DdcOutp8(rI2C_DDC_DS, i2c_ddc_context.segnum);
					// clear interrupt pending flag and interrupt clear bit
					I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);

					break;

				case STATE_TX_EDDC_SEGNUM:
					i2c_ddc_context.state = STATE_TX_DDC_ADDR;

					// put address to shift register
					I2C_DdcOutp8(rI2C_DDC_DS, i2c_ddc_context.addr & 0xFE);
					// master tx mode + start + enable - "START" sequence generation
					I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE | I2C_START | I2C_MODE_MTX); // 0xF0
					// clear interrupt pending flag and interrupt clear bit
					I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR );

					break;

				case STATE_TX_DDC_ADDR:
					i2c_ddc_context.state = STATE_TX_DDC_OFFSET;

					// write offset
					I2C_DdcOutp8(rI2C_DDC_DS, i2c_ddc_context.offset);
					// clear interrupt pending flag and interrupt clear bit
					I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);

					break;

				case STATE_TX_DDC_OFFSET:
					i2c_ddc_context.state = STATE_RX_DDC_ADDR;

					// put address to shift register
					I2C_DdcOutp8(rI2C_DDC_DS, i2c_ddc_context.addr & 0xFE);
					// master rx mode + start + enable - "START" sequence generation
					I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE | I2C_START | I2C_MODE_MRX); // 0xB0
					// clear interrupt pending flag and interrupt clear bit
					I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR );

					break;

				case STATE_RX_DDC_ADDR:
					i2c_ddc_context.state = STATE_RX_DDC_DATA;

					if (i2c_ddc_context.bytes == 1)
					{
						i2c_ddc_context.state = STATE_RX_STOP;

						//rb1004...removed
						//I2C_DdcOutp8(rI2C_DDC_LC, I2C_NO_ACK);
						// clear interrupt pending flag and interrupt clear bit
						I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR );  
					}
					else
					{
						// clear interrupt pending flag and interrupt clear bit
						I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK );    
					}

					break;

				case STATE_RX_DATA:
				case STATE_RX_DDC_DATA:
					// read byte
					*(i2c_ddc_context.buffer) = I2C_DdcInp8(rI2C_DDC_DS);
					i2c_ddc_context.buffer++;
					--(i2c_ddc_context.bytes);
					        
					if (i2c_ddc_context.bytes == 1)
					{
						i2c_ddc_context.state = STATE_RX_STOP;

						//rb1004...removed
						//I2C_DdcOutp8(rI2C_DDC_LC, I2C_NO_ACK);
						// clear interrupt pending flag and interrupt clear bit
						I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR );    
						//I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK  );    
					}
					else
					{
						// clear interrupt pending flag and interrupt clear bit
						I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK );    
					}

					break;

				case STATE_RX_ADDR:
					i2c_ddc_context.state = STATE_RX_DATA;

					if (i2c_ddc_context.bytes == 1)
					{
						i2c_ddc_context.state = STATE_RX_STOP;

						//rb1004...removed
						//I2C_DdcOutp8(rI2C_DDC_LC, I2C_NO_ACK);
						// clear interrupt pending flag and interrupt clear bit
						I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR );  
					}
					else
					{
						// clear interrupt pending flag and interrupt clear bit
						I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK );    
					}

					break;

				case STATE_TX_ADDR:
				case STATE_TX_DATA:
					i2c_ddc_context.state = STATE_TX_DATA;

					I2C_DdcOutp8(rI2C_DDC_DS, *(i2c_ddc_context.buffer));
					i2c_ddc_context.buffer++;
					--(i2c_ddc_context.bytes);

					if (i2c_ddc_context.bytes == 0)
					{
						i2c_ddc_context.state = STATE_TX_STOP;
						I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
					}
					else
						I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK);
					break;

				case STATE_TX_STOP:
					i2c_ddc_context.state = STATE_IDLE;

					I2C_DdcOutp8(rI2C_DDC_STAT, I2C_MODE_MTX | I2C_ENABLE); 
					
					// clear interrupt pending flag and interrupt clear bit
					I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
					// disable Rx/Tx and set to "not busy" - "STOP" sequence generation
					I2C_DdcOutp8(rI2C_DDC_STAT, I2C_MODE_MTX); // 0xC0

					while( I2C_DdcInp8(rI2C_DDC_STAT) & (1<<5) );

					///rb1004...modify 090206
					///EventPost(&gI2CDDCEventGroup, EVENT_I2C_DDC_DONE);
					g_DDC_EventStatus = EVENT_I2C_DDC_DONE;

					break;

				case STATE_RX_STOP:
					i2c_ddc_context.state = STATE_IDLE;

					*(i2c_ddc_context.buffer) = I2C_DdcInp8(rI2C_DDC_DS);

					//I2C_DdcOutp8(rI2C_DDC_STAT, I2C_MODE_MRX); 
					I2C_DdcOutp8(rI2C_DDC_STAT, I2C_MODE_MRX | I2C_ENABLE); 
					
					// clear interrupt pending flag and interrupt clear bit
					I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
					
					// disable Rx/Tx and set to "not busy" - "STOP" sequence generation
					I2C_DdcOutp8(rI2C_DDC_STAT, I2C_MODE_MRX); // 0x80

					while( I2C_DdcInp8(rI2C_DDC_STAT) & (1<<5) );
					
					g_DDC_EventStatus = EVENT_I2C_DDC_DONE;
					break;

				case STATE_IDLE:
				default:
					UART_Printf ("Unknown state!!!\n");
					g_DDC_EventStatus = EVENT_I2C_DDC_ERROR;
			}
		}
	}
	else
	{
		UART_Printf ("*** UNHANDLED ***\n");
	}

	INTC_ClearVectAddr();
}

/**
 * Initialize DDC library
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
s32 I2CDDC_Init(void)
{
	// I2C DDC Channel GPIO Setting
	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_2, 2);
	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_3, 2);

	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_2, 0);
	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_3, 0);
	
	//rb1004....add..090205
	I2C_DdcOutp8(rI2C_DDC_LC, 0x7);

	INTC_SetVectAddr(NUM_HDMI_I2C, I2CDDC_IRQHandler);
	INTC_Enable(NUM_HDMI_I2C);	

	// initialize controller
	// set clock, disable interrupts
	I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK);
	// disable 
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_IDLE);

	return OK;
}

/**
 * Read data through DDC
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
s32 I2CDDC_Read(u8 addr, u8 bytes, u8 *buffer)
{
	//rb1004...add
	g_DDC_EventStatus = EVENT_I2C_DDC_NO;
	// set state
	i2c_ddc_context.state = STATE_RX_ADDR;

	// set parameters
	i2c_ddc_context.buffer = buffer;
	i2c_ddc_context.bytes = bytes;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_ACK); // 0xE1
	// to make IDSR write-enabled
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE | I2C_MODE_MRX); 
	// put address to shift register
	I2C_DdcOutp8(rI2C_DDC_DS, addr & 0xFE);
	// master rx mode + start + enable - "START" sequence generation
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE | I2C_START | I2C_MODE_MRX); // 0xB0

	//waiting DDC status
	while(g_DDC_EventStatus == EVENT_I2C_DDC_NO);
	
	if (g_DDC_EventStatus == EVENT_I2C_DDC_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}

	return OK;
}

/**
 * Write data through DDC
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
s32 I2CDDC_Write(u8 addr, u8 bytes, u8 *buffer)
{
	//rb1004...add
	g_DDC_EventStatus = EVENT_I2C_DDC_NO;
	// set state
	i2c_ddc_context.state = STATE_TX_ADDR;

	// set parameters
	i2c_ddc_context.buffer = buffer;
	i2c_ddc_context.bytes = bytes;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_ACK); // 0xE1
	// to make IDSR write-enabled
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE | I2C_MODE_MTX); 
	// put address to shift register
	I2C_DdcOutp8(rI2C_DDC_DS, addr & 0xFE);
	// master tx mode + start + enable - "START" sequence generation
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE | I2C_START | I2C_MODE_MTX); // 0xF0

	//waiting DDC status
	while(g_DDC_EventStatus == EVENT_I2C_DDC_NO);
	
	if (g_DDC_EventStatus == EVENT_I2C_DDC_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}

	return OK;
}

/**
 * Process fast I2C read through DDC
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
s32 I2CDDC_ReadHDCP(u8 addr, u8 offset, u8 bytes, u8 *buffer)
{
	//rb1004...add
	g_DDC_EventStatus = EVENT_I2C_DDC_NO;
	// set state
	i2c_ddc_context.state = STATE_TX_DDC_ADDR;

	// set parameters
	i2c_ddc_context.offset = offset;
	i2c_ddc_context.buffer = buffer;
	i2c_ddc_context.bytes = bytes;
	i2c_ddc_context.addr = addr;

	// set clock, enable interrupts and ack generation
	I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_ACK); // 0xE1
	// to make IDSR write-enabled
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE); // 0x10
	// put address to shift register
	I2C_DdcOutp8(rI2C_DDC_DS, addr & 0xFE);

	// master tx mode + start + enable - "START" sequence generation
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE | I2C_START | I2C_MODE_MTX); // 0xF0

	//waiting DDC status
	while(g_DDC_EventStatus == EVENT_I2C_DDC_NO);
	
	if (g_DDC_EventStatus == EVENT_I2C_DDC_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}

	return OK;
}

/**
 * Process E-DDC read.
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
s32 I2CDDC_ReadEDDC(u8 segaddr, u8 segnum, u8 addr, u8 offset, u8 bytes, u8* buffer)
{
	//rb1004...add
	g_DDC_EventStatus = EVENT_I2C_DDC_NO;
	// set state
	i2c_ddc_context.state = STATE_TX_EDDC_SEGADDR;

	// set parameters
	i2c_ddc_context.segnum = segnum;
	i2c_ddc_context.addr = addr;
	i2c_ddc_context.offset = offset;
	i2c_ddc_context.buffer = buffer;
	i2c_ddc_context.bytes = bytes;

	// set clock, enable interrupts and ack generation
	I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_ACK); // 0xE1
	// to make IDSR write-enabled
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE); // 0x10
	// put address to shift register
	I2C_DdcOutp8(rI2C_DDC_DS, segaddr & 0xFE);

	// master tx mode + start + enable - "START" sequence generation
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_ENABLE | I2C_START | I2C_MODE_MTX); // 0xF0

	//waiting DDC status
	while(g_DDC_EventStatus == EVENT_I2C_DDC_NO);
	
	if (g_DDC_EventStatus == EVENT_I2C_DDC_ERROR)
	{
		UART_Printf("ERROR!!!\n");
		return ERROR;
	}

	return OK;
}

/**
 * Release DDC line
 */
static s32 I2CDDC_Release(void)
{
	// clear interrupt pending flag and interrupt clear bit
	I2C_DdcOutp8(rI2C_DDC_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
	// disable 
	I2C_DdcOutp8(rI2C_DDC_STAT, I2C_IDLE);

	return OK;
}


#define EDDC_EDID_ADDRESS       0xA0
#define EDDC_SEGMENT_POINTER    0x60


/**
 * Initialize DDC library
 *
 * @return On success, return OK;Otherwise, return ERROR
 */
s32 EDDC_Init(void)
{
	UART_Printf("[EDDC_Init]\n");
	
	if (I2CDDC_Init() != OK)
	{
		UART_Printf("I2CDDC_Init() failed.\n");
		return ERROR;
	}
	
	ResetEDID();

	return OK;
}

/**
 * Read Data through E-DDC
 *
 * @return On success, return OK;Otherwise, return ERROR
 */
s32 EDDC_Read(u8 segment,u8 offset, u8 bytes, u8 *buffer)
{
	if (I2CDDC_ReadEDDC(EDDC_SEGMENT_POINTER,segment,EDDC_EDID_ADDRESS, offset, bytes, buffer) != OK)
	{
		UART_Printf("I2CDDC_ReadEDDC() failed.\n");
		return ERROR;
	}

	return OK;
}

