/**
 * File Name : hdmi.c
 *
 * File Description :
 * This file implements the hdmi v1.3 in S5PV210 project
 *
 * Author : Hee Myung Noh
 * Dept : AP Development
 * Created Date : 2009/01/22
 * Version : 0.1
 * History 
 *  - Initial version (Hee Myung Noh 090122)
 */


#include "system.h"
#include "Library.h"
#include "v210_sfr.h"
#include "intc.h"
#include "hdmi_reg.h"
#include "hdmi.h"
#include "hdmi_audio1.h"
#include "hdmi_audioreg.h"
#include "phy.h"

// mode 
#define MODE_SEL_HDMI_MODE_MASK			(1<<1)

// pixel repetition
#define HDMI_DOUBLE_PIXEL_REPETITION		(0x01)
#define HDMI_CON_PXL_REP_RATIO_MASK		(1<<1 | 1<<0)              

// pixel limit for syscon1
#define CON1_RGB_LIMIT		(1<<5)
#define CON1_YCBCR_LIMIT 	(1<<6)

// SW HPD
#define HPD_SW_ENABLE			1
#define HPD_SW_DISABLE			0
#define DO_NOT_TRANSMIT		0
#define TRANSMIT_ONCE			1

#define TRANSMIT_EVERY_VSYNC		(1<<1)

// Clear_AVMUTE, Set_AVMUTE
#define GCP_AVMUTE_ON				(1<<0)
#define GCP_AVMUTE_OFF				(1<<4)

// Color Depth
// CD0, CD1, CD2, CD3
#define GCP_CD_NOT_INDICATED		0
#define GCP_CD_24BPP				(1<<2)
#define GCP_CD_30BPP				(1<<0 | 1<<2)
#define GCP_CD_36BPP				(1<<1 | 1<<2)
#define GCP_CD_48BPP				(1<<0 | 1<<1 | 1<<2)

#define GCP_DEFAULT_PHASE			1

// for DC_CONTRAL
#define HDMI_DC_CTL_8 				0
#define HDMI_DC_CTL_10				(1<<0)
#define HDMI_DC_CTL_12				(1<<1)


/////////////////////////////////////
// AVI InfoFrame
/////////////////////////////////////
                        
#define AVI_HEADER					(0x82 + 0x02 + 0x0D)
#define AVI_PACKET_BYTE_LENGTH	13
// AVI PB1
// Y1, Y0 
#define AVI_CS_RGB					0
#define AVI_CS_Y422					(1<<5)
#define AVI_CS_Y444					(1<<6)

// A0 
#define AVI_ACTIVE_INFO_PRESENT	(1<<4)

// B1, B0 
#define AVI_BAR_NOT_VALID			0
#define AVI_BAR_VERT_VALID			(1<<2)
#define AVI_BAR_HORIZ_VALID		(1<<3)
#define AVI_BAR_VALID				(1<<2 | 1<<3)

// S1, S0
#define AVI_SCAN_NO_DATA			0
#define AVI_SCAN_OVER				0
#define AVI_SCAN_UNDER				0

// AVI PB2
// C1, C0
#define AVI_COLORIMETRY_MASK			0xC0 // 0b11000000
#define AVI_COLORIMETRY_NO_DATA		(0x0)
#define AVI_COLORIMETRY_ITU601			(0x40) // SMPTE170M, ITU601
#define AVI_COLORIMETRY_ITU709			(0x80) // ITU709
#define AVI_COLORIMETRY_EXTENDED		(0xC0) // Extended Colorimetry

// M1, M0
#define AVI_PICTURE_ASPECT_4_3			(1<<4)                            
#define AVI_PICTURE_ASPECT_16_9		(1<<5)

// R3, R2, R1, R0
#define AVI_FORMAT_ASPECT_AS_PICTURE		(1<<3)
#define AVI_FORMAT_ASPECT_4_3				(1<<3 | 1<<0)
#define AVI_FORMAT_ASPECT_16_9			(1<<3 | 1<<1)
#define AVI_FORMAT_ASPECT_14_9			(1<<3 | 1<<1 | 1<<0)

// AVI PB3
// ITC
#define AVI_IT_CONTENT						(1<<7)

// EC2, EC1, EC0
#define AVI_COLORIMETRY_EXT_MASK			0x70 // 0b01110000
#define AVI_COLORIMETRY_EXT_xvYCC601		(0x0 << 4)
#define AVI_COLORIMETRY_EXT_xvYCC709		(0x1 << 4)

// Q1, Q0
#define AVI_QUANTIZATION_DEFAULT			0
#define AVI_QUANTIZATION_LIMITED			(1<<2)
#define AVI_QUANTIZATION_FULL				(1<<3)

// SC1, SC0
#define AVI_PICTURE_SCALE_HORIZ			(1<<0)
#define AVI_PICTURE_SCALE_VERT				(1<<1)
#define AVI_PICTURE_SCALE_BOTH			(1<<0 | 1<<1)

// AVI PB5
#define AVI_PIXEL_REPETITION_DOUBLE		(1<<0)

/////////////////////////////////////
// Audio Sample Packet 
/////////////////////////////////////
                        
// ASP_CON
#define ASP_TYPE_MASK				(1<<5 | 1<<6)
#define ASP_LPCM_TYPE				0
#define ASP_SACD_TYPE				(1<<5)
#define ASP_HBR_TYPE				(1<<6)
#define ASP_DST_TYPE				(1<<5 | 1<<6)

#define ASP_MODE_MASK				(1<<4)        
#define ASP_2CH_AUD_MODE			0
#define ASP_MULTICH_AUD_MODE		(1<<4)

// sample present field
#define ASP_SP_MASK				(0x0F)
#define ASP_SP_0					(1<<0)
#define ASP_SP_1					(1<<1)
#define ASP_SP_2					(1<<2)
#define ASP_SP_3					(1<<3)

/////////////////////////////////////
// Audio Clock Regeneration Packet 
/////////////////////////////////////

// ACR_CON
#define ACR_MEASURED_CTS_MODE	0x04

/////////////////////////////////////
// Source Product Description InfoFrame  
/////////////////////////////////////
#define SPD_HEADER00				0x83
#define SPD_HEADER01				0x01
#define SPD_HEADER02				0x19

/// MPEG InfoFrame
#define MPG_INFOFRAME_HEADER			(0x85 + 0x01 + 0x0a)
#define MPG_PACKET_BYTE_LENGTH		0xa


//#define	DISP_HDMI_REG

#ifndef DISP_HDMI_REG
#define HdmiOutp8(addr, data) 	Outp8(addr, data)
#define HdmiInp8(addr)  			Inp8(addr)
#else
#define HdmiOutp8(addr, data) 	{UART_Printf("Outp8(\'h%08x, \'h%08x);\n", addr, data), Outp8(addr, data);}
#define HdmiInp8(addr) 			{Inp8(addr); UART_Printf("Inp8(\'h%08x, d); // d=0x%08x\n", addr, Inp8(addr));}
#endif

static const u32 aACR_N_params[] =
{
	4096,   // 32 KHz
	6272,   // 44 kHz
	12544,  // 88 kHz
	25088,  // 176 kHz
	6144,   // 48 kHz
	12288,  // 96 kHz
	24576   // 192 kHz
};

/////////////////////////////////////
// Audio Info frame 
/////////////////////////////////////

#define AUI_INFOFRAME_HEADER			(0x84 + 0x01 + 0x0A)
//rb1004...modify
//#define AUI_PACKET_BYTE_LENGTH		10
#define AUI_PACKET_BYTE_LENGTH		5

#define AUI_CC_2CH						1
#define AUI_CC_3CH						2
#define AUI_CC_4CH						3
#define AUI_CC_5CH						4
#define AUI_CC_6CH						5
#define AUI_CC_7CH						6
#define AUI_CC_8CH						7

#define HDMI_AUI_SF_MASK				0x1C // 0b00011100
#define HDMI_AUI_SF_SF_32KHZ			(0x01 << 2)
#define HDMI_AUI_SF_SF_44KHZ			(0x02 << 2)
#define HDMI_AUI_SF_SF_88KHZ			(0x04 << 2)
#define HDMI_AUI_SF_SF_176KHZ			(0x06 << 2)
#define HDMI_AUI_SF_SF_48KHZ			(0x03 << 2)
#define HDMI_AUI_SF_SF_96KHZ			(0x05 << 2)
#define HDMI_AUI_SF_SF_192KHZ			(0x07 << 2)

#define HDMI_HPD_ON					(1<<1)
#define HDMI_HPD_OFF					0


#define GAMUT_MAXIMUM_SIZE			28

// Video parameters to set HDMI HW
static const struct params
{
	u16 	vHBlank;          	// H Blank
	u32 	vVBlank;          	// V Blank
	u32 	vHVLine;          	// HVLine
	u32 	vHSYNCGEN;        	// HSYNCGen
	u32 	vVSYNCGEN;		// VSYCGen don't care the frequency (50/60)
	u8  	vAVI_VIC;		// CEA VIC
	u8  	vAVI_VIC_16_9;	// CEA VIC
	u8  	interlaced;		// 0 - non-interlaced, 1 - interlaced
	u8  	repetition;		// pixel repetition
	u8  	polarity;			// VSYNC polarity
	u32 	VBLANK_F;		// VBLANK for Interlaced
	u32 	VSYNCGEN2;		// VSYNC for Interlaced
	u32 	VSYNCGEN3;		//
	PHYFreq PixelClock;	// pixel clock
} 	aVideoParams[] =
	{
		{ 0xA0 , 0x16A0D, 0x32020D, 0x11B80E, 0xA00C , 1 , 1 , 0, 0, 1, 0			, 0			, 0       		, ePHY_FREQ_25_200,  	},  // v640x480p_60Hz   
		{ 0x8A , 0x16A0D, 0x35A20D, 0x11300E, 0x900F , 2 , 3 , 0, 0, 1, 0			, 0			, 0       		, ePHY_FREQ_27_027,  	},  // v720x480p_60Hz
		{ 0x172, 0xF2EE , 0x6722EE, 0x2506C , 0x500A , 4 , 4 , 0, 0, 0, 0			, 0			, 0       		, ePHY_FREQ_74_250, 	},  // v1280x720p_60Hz
		{ 0x118, 0xB232 , 0x898465, 0x20856 , 0x2007 , 5 , 5 , 1, 0, 0, 0x232A49	, 0x234239	, 0x4A44A4	, ePHY_FREQ_74_250, 	},  // v1920x1080i_60Hz
		{ 0x114, 0xB106 , 0x6B420D, 0x128024, 0x4007 , 6 , 7 , 1, 1, 1, 0x10691D	, 0x10A10D	, 0x380380	, ePHY_FREQ_27_027,  	},  // v720x480i_60Hz   
		{ 0x114, 0xB106 , 0x6B4106, 0x128024, 0x4007 , 8 , 9 , 0, 1, 1, 0			, 0       		, 0       		, ePHY_FREQ_27_027,  	},  // v720x240p_60Hz   
		{ 0x228, 0xB106 , 0xD6820D, 0x15084A, 0x4007 , 10, 11, 1, 1, 1, 0x10691D	, 0x10A10D	, 0x700700	, ePHY_FREQ_54_054,  	},  // v2880x480i_60Hz   
		{ 0x228, 0xB106 , 0x6B4106, 0x15084A, 0x4007 , 12, 13, 0, 1, 1, 0			, 0       		, 0       		, ePHY_FREQ_54_054,  	},  // v2880x240p_60Hz   
		{ 0x114, 0x16A0D, 0x6B420D, 0x12681E, 0x900F , 14, 15, 0, 1, 1, 0			, 0			, 0			, ePHY_FREQ_54_054,  	},  // v1440x480p_60Hz
		{ 0x118, 0x16C65, 0x898465, 0x20856 , 0x4009 , 16, 16, 0, 0, 0, 0			, 0			, 0			, ePHY_FREQ_148_500, 	},  // v1920x1080p_60Hz
		
		{ 0x90 , 0x18A71, 0x360271, 0x11280A, 0x500A , 17, 18, 0, 0, 1, 0			, 0			, 0			, ePHY_FREQ_27,      	},  // v720x576p_50Hz
		{ 0x2BC, 0xF2EE , 0x7BC2EE, 0x779B6 , 0x500A , 19, 19, 0, 0, 0, 0			, 0			, 0			, ePHY_FREQ_74_250,  	},  // v1280x720p_50Hz
		{ 0x2D0, 0xB232 , 0xA50465, 0x8EA0E , 0x2007 , 20, 20, 1, 0, 0, 0x232A49	, 0x234239	, 0x738738	, ePHY_FREQ_74_250,  	},  // v1920x1080i_50Hz
		{ 0x120, 0xC138 , 0x6C0271, 0x125016, 0x2005 , 21, 22, 1, 1, 1, 0x138951	, 0x13A13D	, 0x378378	, ePHY_FREQ_27,      	},  // v720x576i_50Hz
		{ 0x120, 0xC138 , 0x6C0138, 0x125016, 0x3006 , 23, 24, 0, 1, 1, 0       		, 0       		, 0       		, ePHY_FREQ_27,      	},  // v720x288p_50Hz
		{ 0x240, 0xC138 , 0xD80271, 0x14A82E, 0x2005 , 25, 26, 1, 1, 1, 0x138951	, 0x13A13D	, 0x6F06F0	, ePHY_FREQ_54,      	},  // v2880x576i_50Hz
		{ 0x240, 0xC138 , 0xD80138, 0x14A82E, 0x2005 , 27, 28, 0, 1, 1, 0       		, 0       		, 0       		, ePHY_FREQ_54,      	},  // v2880x288p_50Hz
		{ 0x120, 0x18A71, 0x6C0271, 0x125816, 0x500A , 29, 30, 0, 1, 1, 0       		, 0       		, 0       		, ePHY_FREQ_54,      	},  // v1440x576p_50Hz
		{ 0x2D0, 0x16C65, 0xA50465, 0x8EA0E , 0x4009 , 31, 31, 0, 0, 0, 0       		, 0       		, 0       		, ePHY_FREQ_148_500, 	},  // v1920x1080p_50Hz
		{ 0x33E, 0x16C65, 0xABE465, 0xAA27C , 0x4009 , 32, 32, 0, 0, 0, 0       		, 0       		, 0       		, ePHY_FREQ_74_250,  	},  // v1920x1080p_24Hz

		{ 0x2D0, 0x16C65, 0xA50465, 0x8EA0E , 0x4009 , 33, 33, 0, 0, 0, 0       		, 0       		, 0       		, ePHY_FREQ_74_250,  	},  // v1920x1080p_25Hz
//rb1004
//		{ 0x2D0, 0x16C65, 0xA50465, 0x8EA0E , 0x4009 , 34, 34, 0, 0, 0, 0       		, 0       		, 0       		, ePHY_FREQ_74_250,  	},  // v1920x1080p_30Hz
		{ 0x118, 0x16C65, 0x898465, 0x20856 , 0x4009 , 34, 34, 0, 0, 0, 0       		, 0       		, 0       		, ePHY_FREQ_74_250,  	},  // v1920x1080p_30Hz
		
		{ 0x228, 0x16A0D, 0xD6820D, 0x14D83E, 0x900F , 35, 36, 0, 0, 1, 0       		, 0       		, 0       		, ePHY_FREQ_108_108, 	},  // v2880x480p_60Hz
		{ 0x240, 0x18A71, 0xD80271, 0x14B82E, 0x500A , 37, 38, 0, 1, 1, 0       		, 0       		, 0       		, ePHY_FREQ_108,     	},  // v2880x576p_50Hz
		{ 0x180, 0x2AA71, 0x9004E2, 0x3181E , 0x1701C, 39, 39, 0, 0, 0, 0x2712C6	, 0x28728F	, 0x4a44a4	, ePHY_FREQ_72,      	},  // v1920x1080i_50Hz(1250)
		{ 0x2D0, 0xB232 , 0xA50465, 0x8EA0E , 0x2007 , 40, 40, 1, 0, 0, 0x232A49	, 0x234239	, 0x738738	, ePHY_FREQ_148_500, 	},  // v1920x1080i_100Hz
		{ 0x2BC, 0xF2EE , 0x7BC2EE, 0x779B6 , 0x500A , 41, 41, 0, 0, 0, 0       		, 0       		, 0       		, ePHY_FREQ_148_500, 	},  // v1280x720p_100Hz
		{ 0x90 , 0x18A71, 0x360271, 0x11280A, 0x500A , 42, 43, 0, 0, 1, 0       		, 0       		, 0       		, ePHY_FREQ_54,      	},  // v720x576p_100Hz
		{ 0x120, 0xC138 , 0x6C0271, 0x125016, 0x2005 , 44, 45, 1, 1, 1, 0x138951	, 0x13A13D	, 0x378378	, ePHY_FREQ_54,      	},  // v720x576i_100Hz
		{ 0x118, 0xB232 , 0x898465, 0x20856 , 0x2007 , 46, 46, 1, 0, 0, 0x232A49	, 0x234239	, 0x4A44A4	, ePHY_FREQ_148_500, 	},  // v1920x1080i_120Hz

		{ 0x172, 0xF2EE , 0x6722EE, 0x2506C , 0x500A , 47, 47, 0, 0, 0, 0       		, 0       		, 0       		, ePHY_FREQ_148_500, 	},  // v1280x720p_120Hz
		{ 0x8A , 0x16A0D, 0x35A20D, 0x11300E, 0x900F , 48, 49, 0, 0, 1, 0       		, 0       		, 0       		, ePHY_FREQ_54_054,  	},  // v720x480p_120Hz
		{ 0x114, 0xB106 , 0x6B420D, 0x128024, 0x4007 , 50, 51, 1, 1, 1, 0x10691D	, 0x10A10D	, 0x380380	, ePHY_FREQ_54_054,  	},  // v720x480i_120Hz
		{ 0x90 , 0x18A71, 0x360271, 0x11280A, 0x500A , 52, 53, 0, 0, 1, 0       		, 0       		, 0       		, ePHY_FREQ_108,     	},  // v720x576p_200Hz
		{ 0x120, 0xC138 , 0x6C0271, 0x125016, 0x2005 , 54, 55, 1, 1, 1, 0x138951	, 0x13A13D	, 0x378378	, ePHY_FREQ_108,     	},  // v720x576i_200Hz
		{ 0x8A , 0x16A0D, 0x35A20D, 0x11300E, 0x900F , 56, 57, 0, 0, 1, 0       		, 0       		, 0       		, ePHY_FREQ_108_108, 	},  // v720x480p_240Hz
		{ 0x114, 0xB106 , 0x6B420D, 0x128024, 0x4007 , 58, 59, 1, 1, 1, 0x10691D	, 0x10A10D	, 0x380380	, ePHY_FREQ_108_108, 	},  // v720x480i_240Hz   
	};

// HDMI Timing generation register setting
static const struct HDMI_TG_param
{
	u16 	tg_H_FSZ;
	u16 	tg_HACT_ST;
	u16 	tg_HACT_SZ;
	u16 	tg_V_FSZ;
	u16 	tg_VSYNC;
	u16	tg_VSYNC2;
	u16	tg_VACT_ST;
	u16	tg_VACT_SZ;
	u16	tg_FIELD_CHG;
	u16	tg_VACT_ST2;	
	u16 	tg_VSYNC_TOP_HDMI;
	u16 	tg_VSYNC_BOT_HDMI;
	u16 	tg_FIELD_TOP_HDMI;
	u16	tg_FIELD_BOT_HDMI;
	u8	tg_MHL_HSYNC_WIDTH;
	u8	tg_MHL_VSYNC_WIDTH;
} oHDMI_TG_Param[]=
	{
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v640x480p_60Hz
		{0x35a	,0x8a	,0x2d0	,0x20d	,0x1	 	,0x233	,0x2d	,0x1e0	,0x233	,0x248	,0x1		,0x1		,0x1		,0x233	,0xf		,0x1	},	//v720x480p_60Hz
		{0x672	,0x172	,0x500	,0x2ee	,0x1		,0x233	,0x1e	,0x2d0	,0x233	,0x248	,0x1		,0x1		,0x1		,0x233	,0xf		,0x1	},	//v1280x720p_60Hz
		{0x898	,0x118	,0x780	,0x465	,0x1		,0x233	,0x16	,0x21c	,0x233	,0x249	,0x1		,0x233	,0x1		,0x233	,0xf		,0x1	},	//v1920x1080i_60Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v720x480i_60Hz			
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v720x240p_60Hz			
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v2880x480i_60Hz		
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v2880x240p_60Hz	
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v1440x480p_60Hz
		{0x898	,0x118	,0x780	,0x465	,0x1		,0x233	,0x2d	,0x438	,0x233	,0x248	,0x1		,0x1		,0x1		,0x233	,0xf		,0x1	},	//v1920x1080p_60Hz

		{0x360	,0x90	,0x2d0	,0x271	,0x1	 	,0x233	,0x31	,0x240	,0x233	,0x248	,0x1		,0x1		,0x1		,0x233	,0xf		,0x1	},	//v720x576p_50Hz
		{0x7bc	,0x2bc	,0x500	,0x2ee	,0x1		,0x233	,0x1e	,0x2d0	,0x233	,0x248	,0x1		,0x1		,0x1		,0x233	,0xf		,0x1	},	//v1280x720p_50Hz		
		{0xa50	,0x2d0	,0x780	,0x465	,0x1		,0x233	,0x16	,0x21c	,0x233	,0x249	,0x1		,0x233	,0x1		,0x233	,0xf		,0x1	},	//1920*1080i 50Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v720x576i_50Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v720x288p_50Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v2880x576i_50Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v2880x288p_50Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v1440x576p_50Hz
		{0xa50	,0x2d0	,0x780	,0x465	,0x1		,0x233	,0x2d	,0x438	,0x233	,0x248	,0x1		,0x1		,0x1		,0x233	,0xf		,0x1	},	//v1920x1080p_50Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v1920x1080p_24Hz

		{0xa50	,0x2d0	,0x780	,0x465	,0x1		,0x233	,0x2d	,0x438	,0x233	,0x248	,0x1		,0x1		,0x1		,0x233	,0xf		,0x1	},	//v1920x1080p_25Hz
		{0x898	,0x118	,0x780	,0x465	,0x1		,0x233	,0x2d	,0x438	,0x233	,0x248	,0x1		,0x1		,0x1		,0x233	,0xf		,0x1	},	//v1920x1080p_30Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v2880x480p_60Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v2880x576p_50Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v1920x1080i_50Hz_1250
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v1920x1080i_100Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v1280x720p_100Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v720x576p_100Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v720x576i_100Hz
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},	//v1920x1080i_120Hz
		
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},
		{0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0		,0	},
		
	};


static s32 HDMI_SetColorDepth(ColorDepth cd);
static s32 HDMI_SetAudioClockRecovery(SamplingFreq freq);
static s32 HDMI_SetAUISampleFreq(SamplingFreq freq);
static s32 HDMI_ResetAUISampleFreq(void);
//static void HDMI_UpdateAVIChecksum(void);
static void HDMI_UpdateAUIChecksum(void);

ISR vpISRTABLE[HDMI_IRQ_TOTAL_NUM];

static u8 aGamutMetaData[28] = {0, };

s32 HDMI_RegisterISR(ISR isr, u8 ucIRQNum)
{
///	UART_Printf("Try to register ISR for IRQ number (%d)\n", ucIRQNum );

	if ( isr == NULL )
	{
		UART_Printf("Invaild ISR\n" );
		return ERROR;
	}

	// check IRQ number
	if ( ucIRQNum > HDMI_IRQ_TOTAL_NUM )
	{
		UART_Printf("ucIRQNum exceeds allowed IRQ number(%d)\n", HDMI_IRQ_TOTAL_NUM );
		return ERROR;
	}

	// check if is the number already registered?
	if ( vpISRTABLE[ucIRQNum] )
	{
		UART_Printf("the %d th ISR is already registered\n", ucIRQNum);
	}

	vpISRTABLE[ucIRQNum] = isr;

///	UART_Printf("Success to register ISR for IRQ number (%d)\n", ucIRQNum);
	return OK;
}



/**
 * HDMI interrupt service routine. @n
 * Process all interrupts related to HDMI HW. @n
 * Check all HDMI subsystem interrupts, and invoke registered interrupt service routine.
 */
void __irq HDMI_IRQHandler(void)
{
	u8 ucIRQState, ucIrqNum, ucIrqEnable;
	
	ucIRQState = HdmiInp8(rHDMI_INTC_FLAG);

	UART_Printf("\n");
	UART_Printf("*** HDMI_IRQHandler() : rHDMI_INTC_FLAG = 0x%02x ***\n", ucIRQState);

	// Check interrupt happened (using OR)
	// Proiority of Interrupt  HDCP> I2C > Audio > CEC ( Not implemented)
	if (ucIRQState) //HDCP IRQ
	{
		// invoke ISR 
		ucIrqNum = 0;
		// check if ISR is null or not
		while(ucIrqNum < HDMI_IRQ_TOTAL_NUM)
		{
			if (ucIRQState & (1<<ucIrqNum))
			{
				if (vpISRTABLE[ucIrqNum]) // valid ISR
				{
					// invocate ISR
					(vpISRTABLE[ucIrqNum])(ucIrqNum);
				}
				else // Invalid ISR
				{
					UART_Printf(" HDMIIRQHandler(); No registered ISR for IRQ[%d]\n", ucIrqNum);
				}
			}
		++ucIrqNum;
		}

///rb1004...removed
		// clear interrupt pending all
///		HdmiOutp8(rHDMI_INTC_FLAG, 0x00);
	}
	else
	{
		UART_Printf("HDMIIRQHandler(); Undefined IRQ happened[%x]\n", ucIRQState);
	}

	// Because of HW Problem, after processing INT we should INT enable again.
	//HdmiOutp8( rHDMI_INTC_CON, 1 << HDMI_IRQ_HDCP | 1 << HDMI_IRQ_GLOBAL );
	
	INTC_ClearVectAddr();

	return;
}

/**
 * Initilize, HDMI library
 * 
 * @return    On Success - return OK;Otherwise, return ERROR.
 */
s32 HDMI_Init(void)
{
	u32 i;
	
	UART_Printf("[HDMI_Init]\n");

	// IRQ handler Init
	for(i=0 ; i<HDMI_IRQ_TOTAL_NUM ; i++)
		vpISRTABLE[i] = 0;
	
	INTC_SetVectAddr(NUM_HDMI, HDMI_IRQHandler);
	INTC_Enable(NUM_HDMI);	

	return OK;
}

/**
 * Enable HDMI output
 * 
 * @return    On Success - return OK; Otherwise, return ERROR.
 */
s32 HDMI_Enable(u32 enable)
{
	u8 ucCurrentState, ucHdmiMode;

	ucHdmiMode = HdmiInp8(rHDMI_MODE_SEL) & MODE_SEL_HDMI_MODE_MASK;
	ucCurrentState = HdmiInp8(rHDMI_CON_0);

	if (enable)
	{
		// in HDMI mode enable AVI, AUD, ASP
		if (ucHdmiMode)
		{
			// setup AVI InfoFrame Packet
			HDMI_UpdateAVIChecksum();

			// set up Audio InfoFrame Packet
			HDMI_UpdateAUIChecksum();

			// every frame
			HdmiOutp8(rHDMI_AVI_CON,TRANSMIT_EVERY_VSYNC);
			HdmiOutp8(rHDMI_AUI_CON,TRANSMIT_EVERY_VSYNC);
			if(HdmiInp8(rHDMI_DC_CONTROL))
			{
				u32 uGcpCon;

				uGcpCon = HdmiInp8(rHDMI_GCP_CON);
				uGcpCon = uGcpCon&(~(3<<0))|TRANSMIT_EVERY_VSYNC;
				
				HdmiOutp8(rHDMI_GCP_CON,uGcpCon);
			}

			//SPD InfoFrame
			HDMI_TransmitSPDInfoFrame();
			HDMI_TransmitACPPacket();
			HDMI_TransmitISRCPacket();
			HDMI_TransmitMPEGInfoFrame();
			
			//ucCurrentState |= (HDMI_SYS_ENABLE | HDMI_ASP_ENABLE | ENCODING_OPTION_ENABLE);
			ucCurrentState |= (HDMI_SYS_ENABLE | HDMI_ASP_ENABLE );
		}
		else // DVI mode
		{
			// do not transmit all packet
			HdmiOutp8(rHDMI_AVI_CON,DO_NOT_TRANSMIT);
			HdmiOutp8(rHDMI_AUI_CON,DO_NOT_TRANSMIT);
			HdmiOutp8(rHDMI_GCP_CON,DO_NOT_TRANSMIT);
			HdmiOutp8(rHDMI_GAMUT_CON,DO_NOT_TRANSMIT);
			HdmiOutp8(rHDMI_ACP_CON,DO_NOT_TRANSMIT);
			HdmiOutp8(rHDMI_ISRC_CON,DO_NOT_TRANSMIT);
			HdmiOutp8(rHDMI_MPG_CON,DO_NOT_TRANSMIT);
			HdmiOutp8(rHDMI_SPD_CON,DO_NOT_TRANSMIT);

			ucCurrentState &= ~HDMI_ASP_ENABLE;    
			ucCurrentState |= HDMI_SYS_ENABLE;
		}    
		
		// set the state
		HdmiOutp8(rHDMI_CON_0, ucCurrentState);	
	}
	else // disable
	{
		u8 ucStatus;
		
		ucCurrentState &= ~(HDMI_SYS_ENABLE | HDMI_ASP_ENABLE);
		HdmiOutp8(rHDMI_CON_0, ucCurrentState);	

		do
		{
			ucStatus = HdmiInp8(rHDMI_CON_0);
		}while(ucStatus & (0x1<<0));
	}

	return OK;
}

/**
 * Set HDMI/DVI mode
 */
s32 HDMI_SetMode(HDMIMode mode)
{
	if (mode == HDMI)
	{
		// set HDMI mode
		HdmiOutp8(rHDMI_MODE_SEL,HDMI_MODE_SEL_HDMI);
		HdmiOutp8(rHDMI_CON_2, SYSCON2_HDMI);
	}
	else
	{
		// set DVI mode
		HdmiOutp8(rHDMI_MODE_SEL,HDMI_MODE_SEL_DVI);
		HdmiOutp8(rHDMI_CON_2, SYSCON2_DVI);
	}

	return OK;
}

/** 
 * Enable/Disable the blue screen mode
 */
s32 HDMI_EnableBlueScreen(u32 enable)
{
	u8 ucReg;

	ucReg = HdmiInp8(rHDMI_CON_0);

	if (enable)
		ucReg |= HDMI_BLUE_SCR_ENABLE;
	else
		ucReg &= ~HDMI_BLUE_SCR_ENABLE;

	HdmiOutp8(rHDMI_CON_0, ucReg);

	return OK;
}

/**
 * Set audio sampling frequency in AUI packet
 */
static s32 HDMI_SetAUISampleFreq(SamplingFreq freq)
{
	u8 ucTemp;

	ucTemp = HdmiInp8(rHDMI_AUI_BYTE2) & ~HDMI_AUI_SF_MASK;

	switch (freq)
	{
		case SF_32KHZ:
			ucTemp |= HDMI_AUI_SF_SF_32KHZ;
			break;

		case SF_44KHZ:
			ucTemp |= HDMI_AUI_SF_SF_44KHZ;
			break;

		case SF_88KHZ:
			ucTemp |= HDMI_AUI_SF_SF_88KHZ;
			break;

		case SF_176KHZ:
			ucTemp |= HDMI_AUI_SF_SF_176KHZ;
			break;

		case SF_48KHZ:
			ucTemp |= HDMI_AUI_SF_SF_48KHZ;
			break;

		case SF_96KHZ:
			ucTemp |= HDMI_AUI_SF_SF_96KHZ;
			break;

		case SF_192KHZ:
			ucTemp |= HDMI_AUI_SF_SF_192KHZ;
			break;

		default:
			return ERROR;
	}

	HdmiOutp8(rHDMI_AUI_BYTE2, ucTemp);

	return OK;
}

/**
 * Enable/Disable Software HPD control
 */
s32 HDMI_SWHPDEnable(u32 enable)
{
	u8 ucReg;

	ucReg = HdmiInp8(rHDMI_HPD);
	ucReg &= ~(1<<0);
	
	if (enable)
	{
		HdmiOutp8(rHDMI_HPD, ucReg |HPD_SW_ENABLE);
	}
	else
	{
		HdmiOutp8(rHDMI_HPD, ucReg);
	}

	return OK;
}

/**
 * Set Software HPD level
 *
 * @param   level   [in]    if 0 - low;othewise, high
 */
s32 HDMI_SetSWHPD(u32 level)
{
	u8 ucReg;
	
	ucReg = HdmiInp8(rHDMI_HPD);
	ucReg &= ~(1<<1);
	
	if (level)
	{
		HdmiOutp8(rHDMI_HPD, ucReg |HDMI_HPD_ON);
	}
	else
	{
		HdmiOutp8(rHDMI_HPD, ucReg);
	}

	return OK;
}

/**
 * Configure HDMI HW base on requested video parameters
 * 
 * @return  On success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_SetVideoMode(VideoFormat mode, ColorDepth cd, PixelAspectRatio pixelRatio)
{
	u8  ucTemp8;
	u16 usTemp16;
	u32 uTemp32, uvSYNC2Temp, uvSYNC3Temp;

	// check if HDMI code support that mode
	if (mode > (sizeof(aVideoParams)/sizeof(struct params)) || (s32)mode < 0 )
	{
		UART_Printf("This video mode is not Supported on current Source Device\n");
		return ERROR;
	}
	
	//rb1004...TG register setting
	HDMI_TG_Setting(mode);

	// set HBlank
	usTemp16 = aVideoParams[mode].vHBlank;
	HdmiOutp8(rHDMI_H_BLANK_0, (u8)(usTemp16&0xff) );
	HdmiOutp8(rHDMI_H_BLANK_1, (u8)(usTemp16 >> 8) );

	// set VBlank
	uTemp32 = aVideoParams[mode].vVBlank;
	HdmiOutp8(rHDMI_V_BLANK_0, (u8)(uTemp32&0xff) );
	HdmiOutp8(rHDMI_V_BLANK_1, (u8)(uTemp32 >> 8) );
	HdmiOutp8(rHDMI_V_BLANK_2, (u8)(uTemp32 >> 16) );

	// set HVLine
	uTemp32 = aVideoParams[mode].vHVLine;
	HdmiOutp8(rHDMI_H_V_LINE_0, (u8)(uTemp32&0xff) );
	HdmiOutp8(rHDMI_H_V_LINE_1, (u8)(uTemp32 >> 8) );
	HdmiOutp8(rHDMI_H_V_LINE_2, (u8)(uTemp32 >> 16) );

	// set VSYNC polarity
	HdmiOutp8(rHDMI_VSYNC_POL, aVideoParams[mode].polarity);

	// set HSyncGen
	uTemp32 = aVideoParams[mode].vHSYNCGEN;
	HdmiOutp8(rHDMI_H_SYNC_GEN_0, (u8)(uTemp32&0xff) );
	HdmiOutp8(rHDMI_H_SYNC_GEN_1, (u8)(uTemp32 >> 8) );
	HdmiOutp8(rHDMI_H_SYNC_GEN_2, (u8)(uTemp32 >> 16) );

	// set VSyncGen1
	uTemp32 = aVideoParams[mode].vVSYNCGEN;
	HdmiOutp8(rHDMI_V_SYNC_GEN1_0, (u8)(uTemp32&0xff) );
	HdmiOutp8(rHDMI_V_SYNC_GEN1_1, (u8)(uTemp32 >> 8) );
	HdmiOutp8(rHDMI_V_SYNC_GEN1_2, (u8)(uTemp32 >> 16) );

///////////////////////////////////////////////////////////
//rb1004....hidden register
///	HdmiOutp8(rHDMI_VACT_END_MG,0x2f);
///////////////////////////////////////////////////////////

	if (aVideoParams[mode].interlaced)
	{
		// set up VBLANK_F, VSYNCGEN2, VSYNCGEN3
		uTemp32 = aVideoParams[mode].VBLANK_F;
		uvSYNC2Temp = aVideoParams[mode].VSYNCGEN2;
		uvSYNC3Temp = aVideoParams[mode].VSYNCGEN3;

		HdmiOutp8(rHDMI_V_BLANK_F_0, (u8)(uTemp32 & 0xff));
		HdmiOutp8(rHDMI_V_BLANK_F_1, (u8)(uTemp32 >> 8));
		HdmiOutp8(rHDMI_V_BLANK_F_2, (u8)(uTemp32 >> 16));

		HdmiOutp8(rHDMI_V_SYNC_GEN2_0, (u8)(uvSYNC2Temp & 0xff));
		HdmiOutp8(rHDMI_V_SYNC_GEN2_1, (u8)(uvSYNC2Temp >> 8));
		HdmiOutp8(rHDMI_V_SYNC_GEN2_2, (u8)(uvSYNC2Temp >> 16));

		HdmiOutp8(rHDMI_V_SYNC_GEN3_0, (u8)(uvSYNC3Temp & 0xff));
		HdmiOutp8(rHDMI_V_SYNC_GEN3_1, (u8)(uvSYNC3Temp >> 8));
		HdmiOutp8(rHDMI_V_SYNC_GEN3_2, (u8)(uvSYNC3Temp >> 16));
	}
	else
	{
		// progressive mode
		HdmiOutp8(rHDMI_V_BLANK_F_0, 0x00);
		HdmiOutp8(rHDMI_V_BLANK_F_1, 0x00);
		HdmiOutp8(rHDMI_V_BLANK_F_2, 0x00);

		HdmiOutp8(rHDMI_V_SYNC_GEN2_0, 0x01);
		HdmiOutp8(rHDMI_V_SYNC_GEN2_1, 0x10);
		HdmiOutp8(rHDMI_V_SYNC_GEN2_2, 0x00);

		HdmiOutp8(rHDMI_V_SYNC_GEN3_0, 0x01);
		HdmiOutp8(rHDMI_V_SYNC_GEN3_1, 0x10);
		HdmiOutp8(rHDMI_V_SYNC_GEN3_2, 0x00);
	}
	
	// set interlaced mode
	HdmiOutp8(rHDMI_INT_PRO_MODE,aVideoParams[mode].interlaced);

	// pixel repetition
	ucTemp8 = HdmiInp8(rHDMI_CON_1);

	// clear 
	ucTemp8 &= ~ HDMI_CON_PXL_REP_RATIO_MASK;

	if (aVideoParams[mode].repetition)
	{
		// set pixel repetition        
		ucTemp8 |= HDMI_DOUBLE_PIXEL_REPETITION;
		// AVI Packet
		HdmiOutp8(rHDMI_AVI_BYTE5,AVI_PIXEL_REPETITION_DOUBLE);
	}
	else // clear pixel repetition
	{
		//AVI Packet 
		HdmiOutp8(rHDMI_AVI_BYTE5,0x00);
	}
	
	// set pixel repetition
	HdmiOutp8(rHDMI_CON_1,ucTemp8 );

///////////////////////////////////////////////////////////
//rb1004....hidden register
#if 0
	// set SEND_START
	HdmiOutp8(rHDMI_SEND_START_0,0xfd);
	HdmiOutp8(rHDMI_SEND_START_1,0x01);
	// set SEND_END
	HdmiOutp8(rHDMI_SEND_END_0,0x0d);
	HdmiOutp8(rHDMI_SEND_END_1,0x3a);
	HdmiOutp8(rHDMI_SEND_END_2,0x08);
#endif	
///////////////////////////////////////////////////////////  

	// set AVI packet VIC
	if (pixelRatio == HDMI_PIXEL_RATIO_16_9)
		HdmiOutp8(rHDMI_AVI_BYTE4, aVideoParams[mode].vAVI_VIC_16_9);
	else
		HdmiOutp8(rHDMI_AVI_BYTE4, aVideoParams[mode].vAVI_VIC);

	// clear
	ucTemp8 = HdmiInp8(rHDMI_AVI_BYTE2) & ~(AVI_PICTURE_ASPECT_4_3 | AVI_PICTURE_ASPECT_16_9);
	if (pixelRatio == HDMI_PIXEL_RATIO_16_9)
	{
		ucTemp8 |= AVI_PICTURE_ASPECT_16_9;
	}
	else
	{
		ucTemp8 |= AVI_PICTURE_ASPECT_4_3;
	}

	// set AVI packet MM
	HdmiOutp8(rHDMI_AVI_BYTE2, ucTemp8);

	// set color depth
	if (HDMI_SetColorDepth(cd) != OK)
	{
		return ERROR;
	}

	//rb1004.....GCP Control : Interlaced check
	if(aVideoParams[mode].interlaced == 1)
	{
		u32 uGcpCon;

		uGcpCon = HdmiInp8(rHDMI_GCP_CON);
		uGcpCon |=  (3<<2);

		HdmiOutp8(rHDMI_GCP_CON, uGcpCon);
	}
	else
	{
		u32 uGcpCon;

		uGcpCon = HdmiInp8(rHDMI_GCP_CON);
		uGcpCon &= (~(3<<2));

		HdmiOutp8(rHDMI_GCP_CON, uGcpCon);
	}
	
	// config Phy
#if 1	
	if (PHYConfig(aVideoParams[mode].PixelClock, cd) == ERROR)
	{
		UART_Printf("PHYConfig() failed.\n");
		return ERROR;
	}	
#endif

	return OK;
}


///rb1004....temporary for test
s32 HDMI_SetPhyFreq(VideoFormat mode, ColorDepth cd)
{
	u32 uTemp32;
	
	if (PHYConfig(aVideoParams[mode].PixelClock, cd) == ERROR)
	{
		UART_Printf("PHYConfig() failed.\n");
		return ERROR;
	}
#if 0	
	else
	{
		u32 i;
		
		// CMU Block : HDMI_SEL(SCLK_HDMI) - 0 : SCLK_PIXEL, 1 : SCLK_HDMIPHY
		uTemp32 = Inp32(0xE0100204);
		uTemp32 |= (1<<0);
		Outp32(0xE0100204, uTemp32);	

		// CMU Block : MIXER_SEL(SCLK_MIXER) - 0 : SCLK_DAC,  1 : SCLK_HDMI
		uTemp32 = Inp32(0xE0100204);
		uTemp32 |= (1<<4);
		Outp32(0xE0100204, uTemp32);

		//HDMI Core Reset
		Outp32(0xFA100020, 0);
		for(i=0 ; i<1000 ; i++);
		Outp32(0xFA100020, 1);		
	}
#endif
	return OK;
}
/**
 * Set requested color-space.
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_SetColorSpace(ColorSpace colorSpace)
{
	u8 ucTempCon0, ucTempAVIYY;
	u8 ucTempBlueScr0, ucTempBlueScr1, ucTempBlueScr2;

	// for HDMI_CON0
///	ucTempCon0 = HdmiInp8(rHDMI_CON_0);
///rb1004...090207... removed because the Mixer support only YCbCr444 output mode
///	 ucTempCon0 &= ~(YCBCR422_ENABLE);

	// for AVI_BYTE1
	ucTempAVIYY = HdmiInp8(rHDMI_AVI_BYTE1);
	// clear YY field
	ucTempAVIYY &= ~(1<<5 | 1<<6);

	// check colorSpace
	switch (colorSpace)
	{
		
		case HDMI_CS_YCBCR444:
			ucTempAVIYY |= AVI_CS_Y444;

			// set black screen value for YCBCR
			ucTempBlueScr0 = 255;
			ucTempBlueScr1 = 29;
			ucTempBlueScr2 = 107;
			break;
			
		case HDMI_CS_RGB:
			// set blue screen value for RGB
			ucTempBlueScr0 = 255;
			ucTempBlueScr1 = 0;
			ucTempBlueScr2 = 0;
			break;
			
		case HDMI_CS_YCBCR422:
			// set HDMI_CON_0 register
			//ucTempCon0 |= (YCBCR422_ENABLE);
			// set AVI Byte1
			ucTempAVIYY |= (AVI_CS_Y422);
			// set black screen value for YCBCR
			ucTempBlueScr0 = 255;
			ucTempBlueScr1 = 29;
			ucTempBlueScr2 = 107;
			break;
		
		default :
			UART_Printf("S5PV210 does not support requested Color Space\n");
			return ERROR;
	}

	// set HDMI_CON_1 register
///	HdmiOutp8(rHDMI_CON_0 , ucTempCon0);
	// set AVI packet
	HdmiOutp8(rHDMI_AVI_BYTE1 , ucTempAVIYY);

	// set blue scr register 
	HdmiOutp8(rHDMI_BLUE_SCREEN_0, ucTempBlueScr0);
	HdmiOutp8(rHDMI_BLUE_SCREEN_1, ucTempBlueScr1);
	HdmiOutp8(rHDMI_BLUE_SCREEN_2, ucTempBlueScr2);

	return OK;
}

/**
 * Set requested pixel limitation
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_SetPixelLimit(PixelLimit mode)
{
	u8 ucTempAVIQQ, ucTempCon1;

	// read and clear
	ucTempAVIQQ = HdmiInp8(rHDMI_AVI_BYTE3);
	ucTempAVIQQ &= ~(1<<3 | 1<<2);

	// read and clear
	ucTempCon1 = HdmiInp8(rHDMI_CON_1);
	ucTempCon1 &= ~(1<<6 | 1<<5);

	// check parameter
	switch (mode)
	{
		case HDMI_FULL_RANGE:
			// set avi packet byte 1 to full range
			ucTempAVIQQ |= AVI_QUANTIZATION_FULL;
			break;
		case HDMI_RGB_LIMIT_RANGE:
			// set HDMI_CON_1 as RGB limited value
			ucTempCon1 |= CON1_RGB_LIMIT;
			//set avi packet byte 1 to limited range
			ucTempAVIQQ |= AVI_QUANTIZATION_LIMITED;
			break;
		case HDMI_YCBCR_LIMIT_RANGE:
			// set HDMI_CON_1 as YCBCR limited value
			ucTempCon1 |= CON1_YCBCR_LIMIT;
			// set avi packet byte 1 to limited range
			ucTempAVIQQ |= AVI_QUANTIZATION_LIMITED;
			break;
		default:
			UART_Printf("HDMI core does not support requested Pixel Limit mode\n");
			return ERROR;
	}

	// set register
	HdmiOutp8(rHDMI_AVI_BYTE3, ucTempAVIQQ);
	HdmiOutp8(rHDMI_CON_1, ucTempCon1);

	return OK;
}

/**
 * Enable AV Mute
 */
s32 HDMI_SetAVMute(u32 enable)
{
	// if HDMI mode
	if (HdmiInp8(rHDMI_MODE_SEL) & HDMI_MODE_SEL_HDMI)
	{
		if (enable)
		{
			HdmiOutp8(rHDMI_GCP_BYTE1,GCP_AVMUTE_ON);
			// HW bug : for clearing AV_MUTE, we should set TRANSMIT_EVERY_VSYNC|TRANSMIT_ONCE(b11) on GCP_CON        
			//HdmiOutp8(rHDMI_GCP_CON,TRANSMIT_EVERY_VSYNC|TRANSMIT_ONCE);
			HdmiOutp8(rHDMI_GCP_CON,TRANSMIT_EVERY_VSYNC);
		}
		else
		{
			HdmiOutp8(rHDMI_GCP_BYTE1,GCP_AVMUTE_OFF);
			// enable GCP
			HdmiOutp8(rHDMI_GCP_CON,TRANSMIT_EVERY_VSYNC);
		}    
	}

	return OK;
}

/**
 * Set requested colorimetry.
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_SetColorimetry(HDMIColorimetry colorimetry)
{
	u8 ucTemp;

	ucTemp = HdmiInp8(rHDMI_AVI_BYTE2) & ~AVI_COLORIMETRY_MASK;

	switch (colorimetry)
	{
		case HDMI_COLORIMETRY_NO_DATA:
			break;

		case HDMI_COLORIMETRY_ITU601:
			ucTemp |= AVI_COLORIMETRY_ITU601;
			break;

		case HDMI_COLORIMETRY_ITU709:
			ucTemp |= AVI_COLORIMETRY_ITU709;
			break;

		case HDMI_COLORIMETRY_EXTENDED:
			ucTemp |= AVI_COLORIMETRY_EXTENDED;
			break;

		default:
			return ERROR;
	}

	HdmiOutp8(rHDMI_AVI_BYTE2, ucTemp);

	return OK;
}

/**
 * Set requested extended colorimetry.
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_SetExtendedColorimetry(HDMIExtendedColorimetry extcolorimetry)
{
	u8 ucTemp;

	ucTemp = HdmiInp8(rHDMI_AVI_BYTE3) & ~AVI_COLORIMETRY_EXT_MASK;

	switch (extcolorimetry)
	{
		case HDMI_COLORIMETRY_EXTENDED_xvYCC601:
			ucTemp |= AVI_COLORIMETRY_EXT_xvYCC601;
			break;

		case HDMI_COLORIMETRY_EXTENDED_xvYCC709:
			ucTemp |= AVI_COLORIMETRY_EXT_xvYCC709;
			break;

		default:
			return ERROR;
	}

	HdmiOutp8(rHDMI_AVI_BYTE3, ucTemp);

	return OK;
}

s32 HDMI_SetGamutMetadataColorSpace(HDMIColorimetry eColormetry)
{
	Assert( (eColormetry == HDMI_COLORIMETRY_ITU601) || (eColormetry == HDMI_COLORIMETRY_ITU709) );

	if(eColormetry == HDMI_COLORIMETRY_ITU601)
		aGamutMetaData[0] = (aGamutMetaData[0]&(~(0x7<<0))) | (0x1<<0);
	else
		aGamutMetaData[0] = (aGamutMetaData[0]&(~(0x7<<0))) | (0x2<<0);

	return OK;
}

/**
 * Set gamut data packet body.
 *
 * @param   pdata   [in]    data to set in gamut data packet body
 *          size    [in]    size of data
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_SetGamutMetadata(u8 *pdata, u8 size)
{
	// profile only P0
	// Packet_Seq = 3
	u32 i;

	if (size > GAMUT_MAXIMUM_SIZE)
	{
		UART_Printf("Metadata too big\n");
		return ERROR;
	}

	// set gamut metadata packet header
	HdmiOutp8(rHDMI_GAMUT_HEADER0, 0x0A); // 0b00001010
	HdmiOutp8(rHDMI_GAMUT_HEADER1, 0x80); // 0b10000000
	HdmiOutp8(rHDMI_GAMUT_HEADER2, 0x30); // 0x00110000

	// reset previous metadata
	for (i=0; i<GAMUT_MAXIMUM_SIZE; i++)
	{
		HdmiOutp8(rHDMI_GAMUT_DATA00 + (i * 4), 0x00);
	}

	pdata[0] |= (0x1<<7);		//Format Flag = 1
	
	// set new metadata
	for (i=0; i<size; i++)
	{
		HdmiOutp8(rHDMI_GAMUT_DATA00 + (i * 4), pdata[i]);
	}

	return OK;
}

/**
 * Enable gamut data packet to transfer
 */
s32 HDMI_EnableGamutMetadata(u32 enable)
{
	// Set Gamut Metadata
	HDMI_SetGamutMetadata(aGamutMetaData, GAMUT_MAXIMUM_SIZE);
			
	// enable/disable Gamut Metadata Packet sending
	if (enable)
	{
		HdmiOutp8(rHDMI_GAMUT_CON, TRANSMIT_EVERY_VSYNC);
	}
	else
	{
		HdmiOutp8(rHDMI_GAMUT_CON, DO_NOT_TRANSMIT);
	}

	return OK;
}

/**
 * Set Audio HW based on requested port
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
 extern u32 gHDMI_SPDIF_Event;
s32 HDMI_SetAudioInputPort(AudioInputPort port)
{
	u32 reg;
	
	switch (port)
	{
		case SPDIF_PORT:
				// ISR install
				if ( HDMI_RegisterISR(HDMI_SPDIF_ISR,HDMI_IRQ_SPDIF) != 0 )
				{
					UART_Printf("fail to register HDMI_SPDIF_ISR Interrupt Service Routine\n\n");
					return ERROR;
				}
				reg = HdmiInp8(rHDMI_INTC_CON);
				HdmiOutp8(rHDMI_INTC_CON, reg | (1<<HDMI_IRQ_SPDIF) | (1<<HDMI_IRQ_GLOBAL));

#ifdef SPDIF_NLPCM  
				HdmiOutp8(rHDMI_SPDIF_IRQ_MASK, HDMI_SPDIF_INT_ALL_EN);
#else
				HdmiOutp8(rHDMI_SPDIF_IRQ_MASK, HDMI_SPDIF_INT_WRONG_SIG|HDMI_SPDIF_INT_STATUS_RECOVERED|HDMI_SPDIF_INT_WRONG_PREAMBLE|HDMI_SPDIF_INT_BUF_OVERFLOW);
#endif				
				// enable I2S
				HdmiOutp8(rHDMI_I2S_CLK_CON, I2S_CLK_CON_ENABLE);
				
				// set mux control
				HdmiOutp8(rHDMI_I2S_IN_MUX_CON, I2S_IN_MUX_SELECT_SPDIF | I2S_IN_MUX_ENABLE);
				HdmiOutp8(rHDMI_I2S_MUX_CH, 0xFF); // enable all channels
				HdmiOutp8(rHDMI_I2S_MUX_CUV, 0x03); // enable CUV from right and left channel
				
				HdmiOutp8(rHDMI_SPDIF_CONFIG_1, SPDIF_CONFIG_1_NOISE_FILTER_3_SAMPLES | SPDIF_CONFIG_1_UVCP_ENABLE | SPDIF_CONFIG_1_WORD_LENGTH_MANUAL | SPDIF_CONFIG_1_ALIGN_32BIT | SPDIF_CONFIG_1_HDMI_2_BURST);
				HdmiOutp8(rHDMI_SPDIF_USER_VALUE_1, 0x0b);			
				HdmiOutp8(rHDMI_SPDIF_CLK_CTRL, 0); // enable clock???
				HdmiOutp8(rHDMI_SPDIF_CLK_CTRL, SPDIF_CLK_CTRL_ENABLE);
				HdmiOutp8(rHDMI_SPDIF_OP_CTRL, 0x01 /* 0b01 */); // detect signal

#ifdef SPDIF_NLPCM  
				// Non-LPCM
				//while((HdmiInp8(rHDMI_SPDIF_IRQ_STATUS) & 0x73) != 0x12);
				while(!(gHDMI_SPDIF_Event & HDMI_SPDIF_INT_HEADER_DET));
				while(!(gHDMI_SPDIF_Event & HDMI_SPDIF_INT_STATUS_RECOVERED));
				
				// Disable ch_status_recovered_ir
				reg = HdmiInp8(rHDMI_SPDIF_IRQ_MASK);
				HdmiOutp8(rHDMI_SPDIF_IRQ_MASK, reg&~HDMI_SPDIF_INT_STATUS_RECOVERED);
#else
				//LPCM          
				//while((HdmiInp8(rHDMI_SPDIF_IRQ_STATUS) & 0x03) != 0x02);
				while(!(gHDMI_SPDIF_Event & HDMI_SPDIF_INT_STATUS_RECOVERED));

				// Disable ch_status_recovered_ir
				reg = HdmiInp8(rHDMI_SPDIF_IRQ_MASK);
				HdmiOutp8(rHDMI_SPDIF_IRQ_MASK, reg&~HDMI_SPDIF_INT_STATUS_RECOVERED);
#endif            

				HdmiOutp8(rHDMI_SPDIF_OP_CTRL, 0x03 /* 0b11 */); // run
				break;

		case I2S_PORT:
				// ISR install
				if ( HDMI_RegisterISR(HDMI_I2S_ISR,HDMI_IRQ_I2S) != 0 )
				{
					UART_Printf("fail to register HDMI_I2S_ISR Interrupt Service Routine\n\n");
					return ERROR;
				}
				reg = HdmiInp8(rHDMI_INTC_CON);
				HdmiOutp8(rHDMI_INTC_CON, reg | (1<<HDMI_IRQ_I2S) | (1<<HDMI_IRQ_GLOBAL));
				
				// enable I2S
				HdmiOutp8(rHDMI_I2S_CLK_CON, I2S_CLK_CON_ENABLE);

				// disable DSD
				HdmiOutp8(rHDMI_I2S_DSD_CON, I2S_DSD_CON_DISABLE);

				// pin select for I2S
				HdmiOutp8(rHDMI_I2S_PIN_SEL_0, I2S_PIN_SEL_AUDIO_6 | (I2S_PIN_SEL_AUDIO_5 << 4));
				HdmiOutp8(rHDMI_I2S_PIN_SEL_1, I2S_PIN_SEL_AUDIO_4 | (I2S_PIN_SEL_AUDIO_3 << 4));
				HdmiOutp8(rHDMI_I2S_PIN_SEL_2, I2S_PIN_SEL_AUDIO_2 | (I2S_PIN_SEL_AUDIO_1 << 4));
				HdmiOutp8(rHDMI_I2S_PIN_SEL_3, I2S_PIN_SEL_AUDIO_0);

				// I2S control 
				HdmiOutp8(rHDMI_I2S_CON_1, I2S_CON_SC_POL_RISING| I2S_CON_CH_POL_LOW);
	//rb1004
	//			HdmiOutp8(rHDMI_I2S_CON_2, I2S_CON_I2S_MODE_BASIC | I2S_CON_DATA_NUM_24 | I2S_CON_BIT_CH_32 | I2S_CON_MSB);
				HdmiOutp8(rHDMI_I2S_CON_2, I2S_CON_I2S_MODE_BASIC | I2S_CON_DATA_NUM_16 | I2S_CON_BIT_CH_32 | I2S_CON_MSB);

				// I2S MUX Control
				HdmiOutp8(rHDMI_I2S_IN_MUX_CON, I2S_IN_MUX_ENABLE | I2S_IN_MUX_CUV_ENABLE | I2S_IN_MUX_SELECT_I2S | I2S_IN_MUX_IN_ENABLE);
				HdmiOutp8(rHDMI_I2S_MUX_CH, 0xFF); // enable all channels
				HdmiOutp8(rHDMI_I2S_MUX_CUV, 0x03); // enable CUV from right and left channel
				break;

		default:
				UART_Printf("Unsupported audio port!\n");
				return ERROR;
	}

	return OK;
}


s32 HDMI_CloseAudioInputPort(AudioInputPort port)
{
	u32 reg;
	
	switch (port)
	{
		case SPDIF_PORT:	HdmiOutp8(rHDMI_SPDIF_IRQ_MASK, 0);		//IRQ Disable
							HdmiOutp8(rHDMI_SPDIF_OP_CTRL, 0x00);
							HdmiOutp8(rHDMI_SPDIF_CLK_CTRL, 0); 
							HdmiOutp8(rHDMI_I2S_CLK_CON, I2S_CLK_CON_DISABLE);
							break;
		case I2S_PORT:		HdmiOutp8(rHDMI_I2S_CLK_CON, I2S_CLK_CON_DISABLE);				
							break;

		default:
				UART_Printf("Unsupported audio port!\n");
				return ERROR;		
	}
}


/**
 * Set HDMI HW related to audio packet.@n
 * Set Audio packet type and ACR/AUI packet.
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_SetAudioStreamType(AudioPacketType type, SamplingFreq freq)
{
    u8 ucTemp;

    // clear ASP_CON type mask
    ucTemp = HdmiInp8(rHDMI_ASP_CON) & ~ASP_TYPE_MASK;
        
    // set mode 
    switch (type)                  
    {
        case ASP:
        {   
            // set L-PCM Mode
            HdmiOutp8(rHDMI_ASP_CON, ucTemp | ASP_LPCM_TYPE);
            // reset SF0...SF2
            HDMI_ResetAUISampleFreq();
            break;
        }
    
        case SACD:
        {
            HdmiOutp8(rHDMI_ASP_CON, ucTemp | ASP_SACD_TYPE);
            HDMI_SetAUISampleFreq(freq);
            break;
        }

        case OneBitAudio:
        {
            HdmiOutp8(rHDMI_ASP_CON, ucTemp | ASP_HBR_TYPE);
            // set SF0...SF2
            HDMI_SetAUISampleFreq(freq);
            break;
        }

        case DST:
        {
            HdmiOutp8(rHDMI_ASP_CON, ucTemp | ASP_DST_TYPE);
            // set SF0...SF2
            HDMI_SetAUISampleFreq(freq);
            break;
        }
        
        default:
            return ERROR;
    }

    HDMI_SetAudioClockRecovery(freq);

    CUVSetSamplingFreq(freq);

    return OK;
}

/**
 * Set avaliable bits per channel in I2S port
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_I2SBitsPerChannel(I2SBitsPerChannel bpc)
{
    u8 ucReg;

    ucReg = HdmiInp8(rHDMI_I2S_CON_2) & ~I2S_CON_DATA_NUM_MASK;

    switch (bpc)    
    {
        case I2S_BPC_16:
            ucReg |= I2S_CON_DATA_NUM_16;
            break;

        case I2S_BPC_20:
            ucReg |= I2S_CON_DATA_NUM_20;
            break;

        case I2S_BPC_24:
            ucReg |= I2S_CON_DATA_NUM_24;
            break;

        default:
            return ERROR;
    }

    HdmiOutp8(rHDMI_I2S_CON_2, ucReg);

    return OK;
}

/**
 * Set ASP packet header and AUI packet body fields related to audio channel number
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_SetAudioNumberOfChannels(ChannelNum chnum)
{
	u8 ucReg;

	ucReg = HdmiInp8(rHDMI_ASP_CON);
	ucReg &= ~ASP_MODE_MASK;
	ucReg &= ~ASP_SP_MASK;

	// set number of audio channels
	switch (chnum)
	{
		case CH_2: 
			HdmiOutp8(rHDMI_AUI_BYTE1, AUI_CC_2CH);
			ucReg |= ASP_2CH_AUD_MODE;
			ucReg |= ASP_SP_0;
			break;

		case CH_3:
			HdmiOutp8(rHDMI_AUI_BYTE1, AUI_CC_3CH);
			ucReg |= ASP_MULTICH_AUD_MODE;
			ucReg |= (ASP_SP_0 | ASP_SP_1);
			break;

		case CH_4:
			HdmiOutp8(rHDMI_AUI_BYTE1, AUI_CC_4CH);
			ucReg |= ASP_MULTICH_AUD_MODE;
			ucReg |= (ASP_SP_0 | ASP_SP_1);
			break;

		case CH_5:
			HdmiOutp8(rHDMI_AUI_BYTE1, AUI_CC_5CH);
			ucReg |= ASP_MULTICH_AUD_MODE;
			ucReg |= (ASP_SP_0 | ASP_SP_1 | ASP_SP_2);
			break;

		case CH_6:
			HdmiOutp8(rHDMI_AUI_BYTE1, AUI_CC_6CH);
			ucReg |= ASP_MULTICH_AUD_MODE;
			ucReg |= (ASP_SP_0 | ASP_SP_1 | ASP_SP_2);
			break;

		case CH_7:
			HdmiOutp8(rHDMI_AUI_BYTE1, AUI_CC_7CH);
			ucReg |= ASP_MULTICH_AUD_MODE;
			ucReg |= (ASP_SP_0 | ASP_SP_1 | ASP_SP_2 | ASP_SP_3);
			break;

		case CH_8:
			HdmiOutp8(rHDMI_AUI_BYTE1, AUI_CC_8CH);
			ucReg |= ASP_MULTICH_AUD_MODE;
			ucReg |= (ASP_SP_0 | ASP_SP_1 | ASP_SP_2 | ASP_SP_3);
			break;

		default:
			UART_Printf("Unsupported number of channels!\n");
			return ERROR;
	}

	HdmiOutp8(rHDMI_ASP_CON, ucReg);

	return OK;  
}

/**
 * Set AUI packet field related to channel allocation information.
 *
 * @return  On Success, return OK;Otherwise, return ERROR.
 */
s32 HDMI_SetChannelAllocation(u8 ucCa)
{
	HdmiOutp8(rHDMI_AUI_BYTE4, ucCa);

	return OK;
}

/**
 * Reset AUI packet field realted to audio sample packet
 */
static s32 HDMI_ResetAUISampleFreq(void)
{
    u8 ucTemp;
	
    ucTemp = HdmiInp8(rHDMI_AUI_BYTE2) & ~HDMI_AUI_SF_MASK;
    HdmiOutp8(rHDMI_AUI_BYTE2, ucTemp);

    return OK;
}

/**
 * Calculate checksum for AVI packet and set it in packet.
 */
void HDMI_UpdateAVIChecksum(void)
{
	u32 i;
	u8 ucChecksum;

	ucChecksum = AVI_HEADER;

	for (i = 0; i < AVI_PACKET_BYTE_LENGTH; i++)
	{
		// read byte and sum
		ucChecksum += HdmiInp8(rHDMI_AVI_BYTE1+4*i);
	}
	HdmiOutp8(rHDMI_AVI_CHECK_SUM, ~ucChecksum + (u8)1);

	return;
}

/**
 * Calculate checksum for AUI packet and set it in packet.
 */
static void HDMI_UpdateAUIChecksum(void)
{
	u32 i;
	u8 ucChecksum;

	ucChecksum = AUI_INFOFRAME_HEADER;

	for (i = 0; i < AUI_PACKET_BYTE_LENGTH; i++)
	{		
		// read byte and sum
		ucChecksum += HdmiInp8(rHDMI_AUI_BYTE1+4*i);
	}
	HdmiOutp8(rHDMI_AUI_CHECK_SUM, ~ucChecksum + (u8)1);

	return;
}

/**
 * Set registers related to color depth.
 */
static s32 HDMI_SetColorDepth(ColorDepth cd)
{
	// if color depth is supported by RX, set GCP packet
	switch (cd)
	{
		case HDMI_CD_48:
			HdmiOutp8(rHDMI_GCP_BYTE2,GCP_CD_48BPP);
			break;
			
		case HDMI_CD_36:
			HdmiOutp8(rHDMI_GCP_BYTE2,GCP_CD_36BPP);
			// set DC register
			HdmiOutp8(rHDMI_DC_CONTROL,HDMI_DC_CTL_12);
			aGamutMetaData[0] = (aGamutMetaData[0]&(~(0x3<<3))) | (0x2<<3);
			aGamutMetaData[1] = 0x00; 
			aGamutMetaData[2] = 0x0f;
			aGamutMetaData[3] = 0xff;
			aGamutMetaData[4] = 0x00;
			aGamutMetaData[5] = 0x0f;
			aGamutMetaData[6] = 0xff;
			aGamutMetaData[7] = 0x00;
			aGamutMetaData[8] = 0x0f;
			aGamutMetaData[9] = 0xff;
			break;
			
		case HDMI_CD_30:
			HdmiOutp8(rHDMI_GCP_BYTE2,GCP_CD_30BPP);
			// set DC register
			HdmiOutp8(rHDMI_DC_CONTROL,HDMI_DC_CTL_10);
			aGamutMetaData[0] = (aGamutMetaData[0]&(~(0x3<<3))) | (0x1<<3);
			aGamutMetaData[1] = 0x00; 
			aGamutMetaData[2] = 0x3f;
			aGamutMetaData[3] = 0xf0;
			aGamutMetaData[4] = 0x03;
			aGamutMetaData[5] = 0xff;
			aGamutMetaData[6] = 0x00;
			aGamutMetaData[7] = 0x3f;
			aGamutMetaData[8] = 0xf0;
			break;
			
		case HDMI_CD_24:
			HdmiOutp8(rHDMI_GCP_BYTE2,GCP_CD_24BPP);
			// set DC register
			HdmiOutp8(rHDMI_DC_CONTROL,HDMI_DC_CTL_8);
			// disable GCP
			HdmiOutp8(rHDMI_GCP_CON,DO_NOT_TRANSMIT);
			aGamutMetaData[0] = (aGamutMetaData[0]&(~(0x3<<3))) | (0x0<<3);
			aGamutMetaData[1] = 0x00; 
			aGamutMetaData[2] = 0xff;
			aGamutMetaData[3] = 0x00;
			aGamutMetaData[4] = 0xff;
			aGamutMetaData[5] = 0x00;
			aGamutMetaData[6] = 0xff;				
			// return 
			return OK;
			
		default:
			UART_Printf("HDMI core does not support requested Deep Color mode\n");
			return ERROR;
	}

	return OK;
}

/**
 * Set Audio clock recovery packet
 */
static s32 HDMI_SetAudioClockRecovery(SamplingFreq freq)
{
	u32 uN;

	// ACR N register
	uN = aACR_N_params[freq];
	HdmiOutp8(rHDMI_ACR_N0, 0xff & uN);
	HdmiOutp8(rHDMI_ACR_N1, (uN >> 8) & 0xff);
	HdmiOutp8(rHDMI_ACR_N2, (uN >> 16) & 0xff);

	// ACR packet (Measured)
	// ACR CON register
	HdmiOutp8(rHDMI_ACR_CON, ACR_MEASURED_CTS_MODE);

	return OK;     
}


static s32 HDMI_TransmitSPDInfoFrame(void)
{
	u32 uSpdCon;
				
	HdmiOutp8(rHDMI_SPD_HEADER0, SPD_HEADER00);
	HdmiOutp8(rHDMI_SPD_HEADER1, SPD_HEADER01);
	HdmiOutp8(rHDMI_SPD_HEADER2, SPD_HEADER02);
	
	HdmiOutp8(rHDMI_SPD_DATA00, 'S');
	HdmiOutp8(rHDMI_SPD_DATA01, 'A');
	HdmiOutp8(rHDMI_SPD_DATA02, 'M');
	HdmiOutp8(rHDMI_SPD_DATA03, 'S');
	HdmiOutp8(rHDMI_SPD_DATA04, 'U');
	HdmiOutp8(rHDMI_SPD_DATA05, 'N');
	HdmiOutp8(rHDMI_SPD_DATA06, 'G');
	HdmiOutp8(rHDMI_SPD_DATA07, '!');

	HdmiOutp8(rHDMI_SPD_DATA08, 'S');
	HdmiOutp8(rHDMI_SPD_DATA09, '5');
	HdmiOutp8(rHDMI_SPD_DATA10, 'P');
	HdmiOutp8(rHDMI_SPD_DATA11, 'C');
	HdmiOutp8(rHDMI_SPD_DATA12, '1');
	HdmiOutp8(rHDMI_SPD_DATA13, '1');
	HdmiOutp8(rHDMI_SPD_DATA14, '0');
	HdmiOutp8(rHDMI_SPD_DATA15, ' ');
	HdmiOutp8(rHDMI_SPD_DATA16, ' ');
	HdmiOutp8(rHDMI_SPD_DATA17, ' ');
	HdmiOutp8(rHDMI_SPD_DATA18, ' ');
	HdmiOutp8(rHDMI_SPD_DATA19, ' ');
	HdmiOutp8(rHDMI_SPD_DATA20, ' ');
	HdmiOutp8(rHDMI_SPD_DATA21, ' ');
	HdmiOutp8(rHDMI_SPD_DATA22, ' ');
	HdmiOutp8(rHDMI_SPD_DATA23, ' ');
	HdmiOutp8(rHDMI_SPD_DATA24, 0x02);
	HdmiOutp8(rHDMI_SPD_DATA25, ' ');
	HdmiOutp8(rHDMI_SPD_DATA26, ' ');
	HdmiOutp8(rHDMI_SPD_DATA27, ' ');

	uSpdCon = HdmiInp8(rHDMI_SPD_CON);
	uSpdCon = uSpdCon&(~(3<<0))|TRANSMIT_EVERY_VSYNC;
	HdmiOutp8(rHDMI_SPD_CON,uSpdCon);
				
	return OK;
}


static s32 HDMI_TransmitACPPacket(void)
{
	u32 uAcpCon;

	HdmiOutp8(rHDMI_ACP_TYPE, 0x00);
	
	uAcpCon = HdmiInp8(rHDMI_ACP_CON);
	uAcpCon = uAcpCon&(~(3<<0))|TRANSMIT_EVERY_VSYNC;
	HdmiOutp8(rHDMI_ACP_CON,uAcpCon);

	return OK;
}


static s32 HDMI_TransmitISRCPacket(void)
{
	u32 uIsrcCon;

	HdmiOutp8(rHDMI_ISRC1_HEADER1, 0x00);
	
	uIsrcCon = HdmiInp8(rHDMI_ISRC_CON);
	uIsrcCon = uIsrcCon&(~(3<<0))|TRANSMIT_EVERY_VSYNC;
	HdmiOutp8(rHDMI_ISRC_CON,uIsrcCon);

	return OK;

}


static s32 HDMI_TransmitMPEGInfoFrame(void)
{
	u32 i, uMpegCon;
	u8 ucChecksum;

	ucChecksum = MPG_INFOFRAME_HEADER;

	for (i = 0; i < MPG_PACKET_BYTE_LENGTH; i++)
	{		
		// read byte and sum
		ucChecksum += HdmiInp8(rHDMI_MPG_BYTE1+4*i);
	}
	HdmiOutp8(rHDMI_MPG_CHECK_SUM, ~ucChecksum + (u8)1);
		
	uMpegCon = HdmiInp8(rHDMI_MPG_CON);
	uMpegCon = uMpegCon&(~(3<<0))|TRANSMIT_EVERY_VSYNC;
	HdmiOutp8(rHDMI_MPG_CON,uMpegCon);

	return OK;
}


s32 HDMI_CoreReset(void)
{
	u32 i;
	
	HdmiOutp8(rHDNI_CORE_RSTOUT, 0x00);
	for(i=0 ; i<1000 ; i++);
	HdmiOutp8(rHDNI_CORE_RSTOUT, 0x01);

	return OK;
}


s32 HDMI_CheckPhyReady(void)
{
	while(!(HdmiInp8(rHDMI_PHY_STATUS) & HDMI_PHY_READY));

	return OK;
}

s32 HDMI_TG_Setting(VideoFormat mode)
{
	u16 	usTemp;
	u8	ucTG_cmd;
	
	// set Horizontal Full Size
	usTemp = oHDMI_TG_Param[mode].tg_H_FSZ;
	HdmiOutp8(rTG_H_FSZ_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_H_FSZ_H, (u8)(usTemp >> 8) );

	//set Horizontal Active Start Position
	usTemp = oHDMI_TG_Param[mode].tg_HACT_ST;
	HdmiOutp8(rTG_HACT_ST_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_HACT_ST_H, (u8)(usTemp >> 8) );

	// set Horizontal Active Size
	usTemp = oHDMI_TG_Param[mode].tg_HACT_SZ;
	HdmiOutp8(rTG_HACT_SZ_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_HACT_SZ_H, (u8)(usTemp >> 8) );	

	// set Vertical Full Size
	usTemp = oHDMI_TG_Param[mode].tg_V_FSZ;
	HdmiOutp8(rTG_V_FSZ_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_V_FSZ_H, (u8)(usTemp >> 8) );

	// set VSYNC Position
	usTemp = oHDMI_TG_Param[mode].tg_VSYNC;
	HdmiOutp8(rTG_VSYNC_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_VSYNC_H, (u8)(usTemp >> 8) );	

	// set Bottom Field VSYNC Position
	usTemp = oHDMI_TG_Param[mode].tg_VSYNC2;
	HdmiOutp8(rTG_VSYNC2_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_VSYNC2_H, (u8)(usTemp >> 8) );		

	// set Vertical Active Start Position
	usTemp = oHDMI_TG_Param[mode].tg_VACT_ST;
	HdmiOutp8(rTG_VACT_ST_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_VACT_ST_H, (u8)(usTemp >> 8) );		

	// set Vertical Active Size
	usTemp = oHDMI_TG_Param[mode].tg_VACT_SZ;
	HdmiOutp8(rTG_VACT_SZ_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_VACT_SZ_H, (u8)(usTemp >> 8) );	

	// set Field Change Position
	usTemp = oHDMI_TG_Param[mode].tg_FIELD_CHG;
	HdmiOutp8(rTG_FIELD_CHG_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_FIELD_CHG_H, (u8)(usTemp >> 8) );

	// set Bottom Field Vertical Active Start Position
	usTemp = oHDMI_TG_Param[mode].tg_VACT_ST2;
	HdmiOutp8(rTG_VACT_ST2_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_VACT_ST2_H, (u8)(usTemp >> 8) );

	// set VSYNC Position for HDMI
	usTemp = oHDMI_TG_Param[mode].tg_VSYNC_TOP_HDMI;
	HdmiOutp8(rTG_VSYNC_TOP_HDMI_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_VSYNC_TOP_HDMI_H, (u8)(usTemp >> 8) );	

	// set Bottom Field VSYNC Position
	usTemp = oHDMI_TG_Param[mode].tg_VSYNC_BOT_HDMI;
	HdmiOutp8(rTG_VSYNC_BOT_HDMI_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_VSYNC_BOT_HDMI_H, (u8)(usTemp >> 8) );	

	// set Top Field Change Position for HDMI
	usTemp = oHDMI_TG_Param[mode].tg_FIELD_TOP_HDMI;
	HdmiOutp8(rTG_FIELD_TOP_HDMI_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_FIELD_TOP_HDMI_H, (u8)(usTemp >> 8) );	

	// set Bottom Field Change Position for HDMI
	usTemp = oHDMI_TG_Param[mode].tg_FIELD_BOT_HDMI;
	HdmiOutp8(rTG_FIELD_BOT_HDMI_L, (u8)(usTemp&0xff) );
	HdmiOutp8(rTG_FIELD_BOT_HDMI_H, (u8)(usTemp >> 8) );	

	ucTG_cmd = HdmiInp8(rTG_CMD);
	if(aVideoParams[mode].interlaced == 1)
		ucTG_cmd |= (1<<1);			//Field Mode enable(interlace mode)
	else
		ucTG_cmd &= ~(1<<1);			//Field Mode disable

	HdmiOutp8(rTG_CMD, ucTG_cmd);
		
	return OK;
}


s32 HDMI_TG_OnOff(u8 ucEnable)
{
	u8 ucTG_cmd;

	ucTG_cmd = HdmiInp8(rTG_CMD);

	if(ucEnable == 1)
		ucTG_cmd |= (1<<0);
	else
		ucTG_cmd &= ~(1<<0);

	HdmiOutp8(rTG_CMD, ucTG_cmd);

	return OK;
}


s32 HDMI_TG_MHL_Setting(VideoFormat mode, u8 ucClockInv)
{
	u8 ucTemp;
	
	// set Top Field Change Position for HDMI
	ucTemp = oHDMI_TG_Param[mode].tg_MHL_HSYNC_WIDTH;
	HdmiOutp8(rMHL_HSYNC_WIDTH, (u8)(ucTemp&0xff) );

	// set Bottom Field Change Position for HDMI
	ucTemp = oHDMI_TG_Param[mode].tg_MHL_VSYNC_WIDTH;
	HdmiOutp8(rMHL_VSYNC_WIDTH, (u8)(ucTemp&0xff) );

	// set RGB Clock nversion for MHL I/F
	HdmiOutp8(rMHL_CLK_INV, ucClockInv&0x1);
	
	return OK;
}


s32 HDMI_MHLEnable(u32 uOnOff)
{
	u8 ucTemp;

	ucTemp = HdmiInp8(rHDMI_CON_0);

	if(uOnOff == 1)
	{
		ucTemp |= (1<<7)|(1<<6);
		HdmiOutp8(rHDMI_CON_0, ucTemp);
	}
	else
	{
		ucTemp = ucTemp&(~((1<<7)|(1<<6)));
		HdmiOutp8(rHDMI_CON_0, ucTemp);
	}
	return OK;
}


s32 HDMI_PHYSetting(VideoFormat mode, ColorDepth cd)
{
	if (PHYConfig(aVideoParams[mode].PixelClock, cd) == ERROR)
	{
		UART_Printf("PHYConfig() failed.\n");
		return ERROR;
	}	

	return OK;
}
