/**
 * File Name : hdcp.c
 *
 * File Description :
 * This file implements the hdcp v1.1 in S5PV210 project
 *
 * Author : Hee Myung Noh
 * Dept : AP Development
 * Created Date : 2009/01/22
 * Version : 0.1
 * History 
 *  - Initial version (Hee Myung Noh 090122)
 */

#include "system.h"
#include "Library.h"
#include "sysc.h"
#include "v210_sfr.h"
#include "intc.h"
#include "hdmi_reg.h"
#include "hdmi.h"
#include "phy.h"
#include "hdcp.h"
#include "i2c_hdmi.h"
#include "timer.h"

#define HDCP_KSV_SIZE					(5)
#define HDCP_AN_SIZE					(8)

#define HDCP_RX_ADDR					(0x74)

// Bcaps
#define HDCP_BCAPS_REPEATER			(6)
#define HDCP_BCAPS_READY				(5)
#define HDCP_BCAPS_OFFSET				(0x40)

// HDCP_RESULT
#define HDCP_RI_NOT_MATCH				(2)
#define HDCP_RI_MATCH					(3)


#define NUM_OF_Ri_COMPARE_RETRIAL		(3)

// Encrypt Enable
#define HDCP_ENC_DIS					(0)
#define HDCP_ENC_EN					(1)

// HDCP Rx device addr
#define HDCP_RX_DEV_ADDR				(0x74)

// Bksv
#define HDCP_BKSV_OFFSET				(0x00)
#define WRITE_ENABLE					(1)

// Aksv
#define HDCP_AKSV_OFFSET				(0x10)

// An
#define HDCP_AN_OFFSET					(0x18)

// Rj
#define HDCP_RX_Ri_OFFSET				(0x08)

// Timer
#define TIMER_EXPIRED					(1)

#define HDCP_BSTATUS_OFFSET			(0x41)
#define HDCP_BSTATUS_SIZE				(2)

// HDCP_CTL
#define HDCP_ENABLE						(0x02)
#define HDCP_TIMEROUT					(0x04)

// CTL2
#define HDCP_REVOCATION_SET			(0x01)
#define HDCP_SHA1_SW_EN				(0x02)
#define HDCP_SHA1_SW_RESULT_READY	(0x04)
#define HDCP_SHA1_SW_RESULT_VALID	(0x08)

// Bstatus
#define MAX_DEVS_EXCEEDED_MASK		(1<<7)
#define MAX_CASCADE_EXCEEDED_MASK	(1<<3)
#define DEVICE_COUNT_MASK				(0x7F)
#define HDMI_MODE_MASK				(1<<4)
#define HDCP_MAX_DEVS					(128)
#define HDCP_MAX_CASCADE				(8)


// KSV_LIST_CON
#define HDCP_KSV_LIST_OFFSET			(0x43)

#define HDCP_KSV_LIST_READ_DONE		(1<<0)
#define HDCP_KSV_LIST_END				(1<<1)
#define HDCP_KSV_LIST_EMPTY			(1<<2)

// SHA-1
#define HDCP_RX_SHA1_OFFSET			(0x20)
#define HDCP_SHA1_SIZE					(20)
#define HDCP_SHA1_VALID_READY			(1<<1)
#define HDCP_SHA1_VALID				(1<<0)

// Statue_EN register
#define HDCP_I2C_INT					(0)
#define HDCP_WATCHDOG_INT				(1)
#define HDCP_AN_WRITE_INT				(2)
#define HDCP_UPDATE_PJ_INT				(3)
#define HDCP_UPDATE_RI_INT				(4)
#define HDCP_AUD_FIFO_OVF_EN			(5)
#define HDCP_AUTHEN_ACK				(7)

#define HDCP_I2C_INT_MASK				(1<<HDCP_I2C_INT)
#define HDCP_WATCHDOG_INT_MASK		(1<<HDCP_WATCHDOG_INT)
#define HDCP_AN_WRITE_INT_MASK		(1<<HDCP_AN_WRITE_INT)
#define HDCP_UPDATE_PJ_INT_MASK		(1<<HDCP_UPDATE_PJ_INT)
#define HDCP_UPDATE_RI_INT_MASK		(1<<HDCP_UPDATE_RI_INT)
#define HDCP_AUD_FIFO_OVF_EN_MASK	(1<<HDCP_AUD_FIFO_OVF_EN)
#define HDCP_AUTHEN_ACK_MASK			(1<<HDCP_AUTHEN_ACK)
#define HDCP_AUTHEN_FAIL				(0)

///rb1004
///static EventGroup gHDCPEvent;

#define EVENT_HDCP_START				0
#define EVENT_HDCP_STOP				1
#define EVENT_HDCP_I2C					2
#define EVENT_HDCP_WATCHDOG			3
#define EVENT_HDCP_AN_WRITE			4
#define EVENT_HDCP_UPDATE_RI			5

// TIMEOUT
#define HDCP_TIMEOUT					100 //ms
#define HDCP_REPEATER_TIMEOUT			5000 // 5sec

//#define	DISP_HDCP_REG

#ifndef DISP_HDCP_REG
#define HdcpOutp8(addr, data) 	Outp8(addr, data)
#define HdcpInp8(addr)  			Inp8(addr)
#else
#define HdcpOutp8(addr, data) 	{UART_Printf("Outp8(\'h%08x, \'h%08x);\n", addr, data), Outp8(addr, data);}
#define HdcpInp8(addr) 			{Inp8(addr); UART_Printf("Inp8(\'h%08x, d); // d=0x%08x\n", addr, Inp8(addr));}
#endif

typedef enum
{
	IDLE = 0,
	NOT_AUTHENTICATED,
	BKSV_READ_DONE,
	AKSV_WRITE_DONE,
	SECOND_AUTHENTICATION,
	THIRD_AUTHENTICATION
}Authen_State;

u32 gHDCPEvent = 0;

static void HDCP_Enable(s32 enable);
static s32 HDCP_StartEncryption(void);
static s32 HDCP_ReadHDCP(u8 offset, u8 length, u8* destAddr);
static s32 HDCP_ReadBksv(u8* outBksv);
static s32 HDCP_ReadBcaps(u8* outBcaps);
static s32 HDCP_ReadBstatus(void);
static void HDCP_SetBksv(u8* inBksv);
static s32 HDCP_ReadAksv(u8* outBksv);
static void HDCP_SetBcaps(u8 Bcaps);
static s32 HDCP_WriteAksv(void);
static s32 HDCP_WriteAn(void);
static s32 HDCP_CompareRi(void);
static s32 HDCP_CheckRepeater(void);
static s32 HDCP_LoadKey(void);

/*
 * Callback for timer
 */
void TimerCallBack(void *vpArg)
{
	 UART_Printf("Timer Expired (LEO)***\n\n");
	 *((u8 *)vpArg) = TIMER_EXPIRED;
}

/**
 *  HDCP interrupt service routine
 */
void HDCP_ISR(u32 IRQNum)
{
	u8 HDCPState = HdcpInp8(rHDMI_STATUS);

	UART_Printf("HDCP State  = 0x%x\n",HDCPState);

	// check state and post event  

	// HDCP_I2C_INT happen
	if (HDCPState & HDCP_I2C_INT_MASK)
	{
		UART_Printf("HDCP_I2C_INT\n");

		// clear HDCP_STATUS
		HdcpOutp8(rHDMI_STATUS,HDCP_I2C_INT_MASK);

		// clear I2C_INT debug reg
		HdcpOutp8(rHDCP_I2C_INT,0x00); 

		// Event posting to HDCP Processing
		///EventPost(&gHDCPEvent, EVENT_HDCP_I2C);
		gHDCPEvent |= 1<<EVENT_HDCP_I2C;
	}

	// HDCP_WATCHDOG_INT
	if (HDCPState & HDCP_WATCHDOG_INT_MASK)
	{
		UART_Printf("HDCP_WDT_INT\n");

		// clear HDCP_STATUS
		HdcpOutp8(rHDMI_STATUS,HDCP_WATCHDOG_INT_MASK);

		// clear I2C_INT debug reg
		HdcpOutp8(rHDCP_WDT_INT,0x00); 

		// Event posting to HDCP Processing
		///EventPost(&gHDCPEvent, EVENT_HDCP_WATCHDOG);
		gHDCPEvent |= 1<<EVENT_HDCP_WATCHDOG;
	}

	// HDCP_AN_INT
	if (HDCPState & HDCP_AN_WRITE_INT_MASK)
	{
		UART_Printf("HDCP_AN_INT\n");

		// clear HDCP_STATUS
		HdcpOutp8(rHDMI_STATUS,HDCP_AN_WRITE_INT_MASK); 

		// clear AN_INT debug reg
		HdcpOutp8(rHDCP_AN_INT,0x00); 

		// Event posting to HDCP Processing
		///EventPost(&gHDCPEvent, EVENT_HDCP_AN_WRITE);
		gHDCPEvent |= 1<<EVENT_HDCP_AN_WRITE;
	}

	// HDCP_RI_INT
	if (HDCPState & HDCP_UPDATE_RI_INT_MASK)
	{
		UART_Printf("HDCP_RI_INT\n");
		    
		// clear HDCP_STATUS
		HdcpOutp8(rHDMI_STATUS,HDCP_UPDATE_RI_INT_MASK); 

		// clear RI_INT debug reg
		HdcpOutp8(rHDCP_RI_INT,0x00); 

		// Event posting to HDCP Processing
		///EventPost(&gHDCPEvent, EVENT_HDCP_UPDATE_RI);
		gHDCPEvent |= 1<<EVENT_HDCP_UPDATE_RI;
	}
}

/**
 * Initialize HDCP libarary.
 *
 * @return  On success, return OK;Otherwise, return ERROR.
 */
s32 HDCP_Init(void)
{
	u8 reg;

	// Init Event
	///EventInit(&gHDCPEvent);

	UART_Printf("[HDCP_Init]\n");

	// ISR install
	if ( HDMI_RegisterISR(HDCP_ISR,HDMI_IRQ_HDCP) != 0 )
	{
		UART_Printf("fail to register HDCP Interrupt Service Routine\n\n");
		return ERROR;
	}

	// SS_INT_ENABLE for HDCP    
	reg = HdcpInp8(rHDMI_INTC_CON);
	HdcpOutp8(rHDMI_INTC_CON, reg | (1<<HDMI_IRQ_HDCP) | (1<<HDMI_IRQ_GLOBAL));

	return OK;
}

/**
 * Start processing HDCP
 */
s32 HDCP_Start(void)
{
	UART_Printf("starting HDCP...\n");

	///EventPost(&gHDCPEvent, EVENT_HDCP_START);
	gHDCPEvent = 1<<EVENT_HDCP_START;
	
	return OK;
}

/**
 * Stop processing HDCP
 */
s32 HDCP_Stop(void)
{
	UART_Printf("stopping HDCP...\n");

	///EventPost(&gHDCPEvent, EVENT_HDCP_STOP);
	gHDCPEvent = 1<<EVENT_HDCP_STOP;

	return OK;
}

/**
 * Process HDCP protocol
 */
void HDCP_Run(void)
{
	u8 state, repeater=0, resetAuth = FALSE;
	u32 pend; 
	s32 iRet, result;
	u8 Bksv[HDCP_KSV_SIZE]; 
	u8 Aksv[HDCP_KSV_SIZE]; 
	u8 Bcaps;
	u8 temp;

	// initial state
	state = IDLE;

	HDCP_Start();
	
	// Start AES
	if ( HDCP_LoadKey() != OK )
	{
		UART_Printf("Efuse ECC fail\n");	
		UART_Getc();
		return;
	}

	while (!UART_GetKey())
	{
		if(gHDCPEvent & ((1<<EVENT_HDCP_START)| (1<<EVENT_HDCP_STOP) |
		               (1<<EVENT_HDCP_I2C) | (1<<EVENT_HDCP_AN_WRITE) | 
		               (1<<EVENT_HDCP_UPDATE_RI) | (1<<EVENT_HDCP_WATCHDOG)) )
		{
			// if STOP Event
			if ( gHDCPEvent & (1<<EVENT_HDCP_STOP))
			{
				UART_Printf("-HDCP_STOP Received\n");

				// change state
				state = IDLE;

				// disable HDCP
				HDCP_Enable(FALSE);

				gHDCPEvent &= (1<<EVENT_HDCP_START);
			}

			// state machine start
			switch (state)
			{
				// if not start yet
				case IDLE:
					UART_Printf("[STATE=IDLE]\n");
					// if START Event
					if (gHDCPEvent & (1<< EVENT_HDCP_START))
					{
						UART_Printf("-HDCP_START Received\n");

						// start HDCP
						HDCP_Enable(TRUE);

						// enable blue screen mode
						HDMI_EnableBlueScreen(TRUE);

						// change state
						state = NOT_AUTHENTICATED;

						//clear pend
						gHDCPEvent &= ~(1<< EVENT_HDCP_START);
					}
					// if other events, do nothing
					else
					{
						// clear all event
						gHDCPEvent = 0;
					}
					break;

				// start, not authenticated yet
				case NOT_AUTHENTICATED:
					UART_Printf("[STATE=NOT_AUTHENTICATED]\n");
					// if I2C Event
					if(gHDCPEvent &(1<<EVENT_HDCP_I2C))
					{
						///HDCP_ReadBstatus();
						// read Bksv,Bcaps and set Bksv, and check repeater and Set Bcaps without repeater bit
///rb1004....modify						
///						if ( ReadBksv(Bksv) == OK && ReadBcaps(&Bcaps) == OK && RevocListFind(Bksv, 1) == OK)
						if ( HDCP_ReadBksv(Bksv) == OK && HDCP_ReadBcaps(&Bcaps) == OK)
						{
///							HDCP_ReadBcaps(&Bcaps);
								
							// check repeater
							repeater = Bcaps & (1<<HDCP_BCAPS_REPEATER);
							                  
							// set bcaps
							HDCP_SetBcaps(Bcaps);
							                
							// set bksv
							HDCP_SetBksv(Bksv);

							//change state
							state = BKSV_READ_DONE;
						}
						// if error, reset processing
						else
						{
							resetAuth = TRUE;
						}
						// clear pend
						///gHDCPEvent = 0;                            
						gHDCPEvent &= ~(1<<EVENT_HDCP_I2C);
					}
					// if other EVENT, reset Auth
					else
					{
						resetAuth = TRUE;
					}
					break;

				// read Bksv/Bcaps done
				case BKSV_READ_DONE:
					UART_Printf("[STATE=BKSV_READ_DONE]\n");
					// if AN Event
					if(gHDCPEvent & (1<<EVENT_HDCP_AN_WRITE))
					{
						// write An/Aksv and ReadBcaps/Bksv
						// For Simplay HD problem, following the spec, Read Bcaps()/Bksv() again
						if ( HDCP_WriteAn() == OK && HDCP_WriteAksv() == OK && HDCP_ReadBcaps(&Bcaps) == OK && HDCP_ReadBksv(Bksv) == OK)
						{
							// wait for 100ms for following spec.
							DelayfrTimer(milli, 100);

							// change state
							state =  AKSV_WRITE_DONE;                           
						}
						// if error, reset_auth
						else
						{
							resetAuth = TRUE;
						}
						// clear pend
						gHDCPEvent &= ~(1<<EVENT_HDCP_AN_WRITE);
					}
					// other event, reset processing
					else
					{
						resetAuth = TRUE;
					}
					break;

				// write An/Aksv/Ainfo done
				case AKSV_WRITE_DONE:
					UART_Printf("[STATE=AKSV_WRITE_DONE]\n");
					// if UPDATE_RI Event
					if(gHDCPEvent & (1<<EVENT_HDCP_UPDATE_RI))
					{
						// check Ri == Rj
						if (HDCP_CompareRi() == OK)
						{
							// set the HDCP result register as match
							HdcpOutp8(rHDCP_CHECK_RESULT,HDCP_RI_MATCH);
							// clear the HDCP result register
							HdcpOutp8(rHDCP_CHECK_RESULT,0x00);

							// if repeater
							if (repeater)
							{
								// change state to start 2nd authentication
								state = SECOND_AUTHENTICATION;
								UART_Printf("!!! Repeater Check Start!!!\n");
							}
							else
							{
								// start Encryption
								if (HDCP_StartEncryption() != OK)
								{
									resetAuth = TRUE;
								}
								else
								{
									// change state;
									state = THIRD_AUTHENTICATION;

									// disable blue screen mode
									HDMI_EnableBlueScreen(FALSE);
								}
							}
						}
						// if error, reset processing
						else
						{
							UART_Printf("Ri != Rj : not matched\n");
							// set the HDCP result register as not match
							HdcpOutp8(rHDCP_CHECK_RESULT,HDCP_RI_NOT_MATCH);
							// clear Result
							HdcpOutp8(rHDCP_CHECK_RESULT,0x00);
							// change state
							state = NOT_AUTHENTICATED;
						}
						// clear pending
						gHDCPEvent &= ~(1<<EVENT_HDCP_UPDATE_RI);
					}
					// if other events, reset processing
					else
					{
						resetAuth = TRUE;
					}
					break;

				// second_authentication
				case SECOND_AUTHENTICATION:
					UART_Printf("[STATE=SECOND_AUTHENTICATION]\n");
					// if WDT Event
					if(gHDCPEvent & (1<<EVENT_HDCP_WATCHDOG))
					{
						UART_Printf("-WATCHDOG INT Happen!!!!\n");

						result = HDCP_CheckRepeater();
						// check repeater
						if ( result == OK)
						{
							// start Encryption
							if (HDCP_StartEncryption() != OK)
							{
								resetAuth = TRUE;
							}
							//change state;
							else
							{
								state = THIRD_AUTHENTICATION;

								// disable blue screen mode
								HDMI_EnableBlueScreen(FALSE);
							}
						}
						else if (result == ILLEGAL_DEVICE)
						{
							// set revocation ksv exist
							HdcpOutp8(rHDCP_CTRL2, HDCP_REVOCATION_SET);
							// clear
							HdcpOutp8(rHDCP_CTRL2, 0x00);
							// change state
							state = NOT_AUTHENTICATED;
						}
						else if (result == REPEATER_NOT_READY)
						{
							// write timerout on hdcp_ctl reg
							temp = HdcpInp8(rHDCP_CTRL1);

							HdcpOutp8(rHDCP_CTRL1, temp|HDCP_TIMEROUT);

							// clear
							HdcpOutp8(rHDCP_CTRL1, temp);
							// change state
							state = NOT_AUTHENTICATED;
						}
						else
						{
							resetAuth = TRUE;
						}
						// clear pending
						gHDCPEvent &= ~(1<<EVENT_HDCP_WATCHDOG);
					}
					// if other event, reset processing
					else
					{
						resetAuth = TRUE;
					}
					break;

				// third authentication
				case THIRD_AUTHENTICATION:
					UART_Printf("[STATE=THIRD_AUTHENTICATION]\n");
					// if UPDATE_RI Event
					if(gHDCPEvent & (1<<EVENT_HDCP_UPDATE_RI))
					{
						// check Ri != Rj
						if (HDCP_CompareRi() == OK)
						{
							// do nothing                            
						}
						else
						{
							// set the HDCP result register as match
							HdcpOutp8(rHDCP_CHECK_RESULT,HDCP_RI_NOT_MATCH);
							// stop sending EESS
							HdcpOutp8(rHDCP_ENC_EN, HDCP_ENC_DIS);
							// clear Result
							HdcpOutp8(rHDCP_CHECK_RESULT,0x00);

							//enable blue screen mode
							HDMI_EnableBlueScreen(TRUE);

							// change state
							state = NOT_AUTHENTICATED;
						}
						// clear pending
						gHDCPEvent &= ~(1<<EVENT_HDCP_UPDATE_RI);
					}
					// if other Event, reset Auth
					else
					{
						resetAuth = TRUE;
					}
					break;
			} //switch

			// reset Authentication
			if (resetAuth)
			{
				// pending clear all without Start and Stop, state initialize, reset HDCP
				gHDCPEvent &= ((1<<EVENT_HDCP_START) | (1<<EVENT_HDCP_STOP));
				if (gHDCPEvent & (1<<EVENT_HDCP_START)) 
					state = IDLE;
				else
					state = NOT_AUTHENTICATED;

				HDCP_Enable(FALSE);
				resetAuth = FALSE;
				HDCP_Enable(TRUE);
				HDMI_EnableBlueScreen(TRUE);              
			}
		}
	} //while(1), wait for event

	
	UART_Printf("HDCP_STOP\n");
	// change state
	state = IDLE;
	// disable HDCP
	HDCP_Enable(FALSE);
	HDMI_EnableBlueScreen(FALSE);
	gHDCPEvent = 0;	
} // HDCPRun()


/**
 * Enable/Disable HDCP HW to work
 */
static void HDCP_Enable(s32 enable)
{
	u8 reg;
	
	if (enable)
	{
		UART_Printf("-HDCP_ENABLE\n");

		// set HDCP_INT enable
		reg = HdcpInp8(rHDMI_STATUS_EN);
		reg |= (HDCP_I2C_INT_MASK | HDCP_WATCHDOG_INT_MASK | HDCP_AN_WRITE_INT_MASK | HDCP_UPDATE_RI_INT_MASK);
		HdcpOutp8( rHDMI_STATUS_EN, reg);

		//HDCP Enable
		HdcpOutp8( rHDCP_CTRL1, HDCP_ENABLE); 		
	}
	else
	{
		UART_Printf("-HDCP_DISABLE\n");

		// set HDCP_INT disable
		reg = HdcpInp8(rHDMI_STATUS_EN);
		reg &= ~(HDCP_I2C_INT_MASK | HDCP_WATCHDOG_INT_MASK | HDCP_AN_WRITE_INT_MASK | HDCP_UPDATE_RI_INT_MASK);
		HdcpOutp8( rHDMI_STATUS_EN, reg);

		// disable HDCP
		HdcpOutp8(rHDCP_ENC_EN, HDCP_ENC_DIS);
		HdcpOutp8(rHDCP_CTRL1, 0x00);
		HdcpOutp8(rHDCP_CTRL2, 0x00);

	        // HPD deassert/assert for initialzing HDCP
	        // disable HPD INT
	        reg = HdcpInp8(rHDMI_INTC_CON);
	        HdcpOutp8(rHDMI_INTC_CON, reg & ~((1<<HDMI_IRQ_HPD_PLUG) | (1<<HDMI_IRQ_HPD_UNPLUG)));        
	        
	        HDMI_SWHPDEnable(TRUE);
	        HDMI_SetSWHPD(FALSE);
	        
	        HDMI_SetSWHPD(TRUE);
	        HDMI_SWHPDEnable(FALSE);
	        
	        // enable HPD INT again
	        HdcpOutp8(rHDMI_INTC_CON,reg);		
	}
}

/**
 * Start sending EESS signal to RX
 *
 * @return  On success, return OK;Otherwise, return ERROR.
 */
static s32 HDCP_StartEncryption(void)
{
	// check for AUTHENTICATION_ACK    
	if (!(HdcpInp8(rHDMI_STATUS) & HDCP_AUTHEN_ACK_MASK))
	{
		UART_Printf("fail to start encryption\n");
		return ERROR;
	}

	// start to send EESS
	HdcpOutp8(rHDCP_ENC_EN, HDCP_ENC_EN);

	return OK;
}

/**
 * Read data from HDCP RX port
 *
 * @return  On success, return OK;Otherwise, return ERROR.
 */
static s32 HDCP_ReadHDCP(u8 offset, u8 length, u8* destAddr)
{
	if (I2CDDC_ReadHDCP((u8)HDCP_RX_DEV_ADDR, offset, length, destAddr) != OK)
	{
		UART_Printf("Fail to Read from HDCP\n");
		return ERROR;
	}

	return OK;
}

/**
 * Read Bksv from HDCP RX port
 * 
 * @return On success, return OK;Otherwise, return ERROR
 */
static s32 HDCP_ReadBksv(u8* outBksv)
{
	s32 i; 

	UART_Printf("-reading Bksv...\n");
	// Read Bksv
	if (HDCP_ReadHDCP(HDCP_BKSV_OFFSET, HDCP_KSV_SIZE, outBksv) != OK)
	{
		return ERROR;
	}

	UART_Printf("-Bksv: 0x");
	for (i=0; i <HDCP_KSV_SIZE; i++)
	{
		UART_Printf("%02x", outBksv[i]);
	}
	UART_Printf("\n");

	return OK;
}


///rb10004....for debugging function
static s32 HDCP_ReadAksv(u8* outBksv)
{
	s32 i; 

	UART_Printf("-reading Aksv...\n");
	// Read Bksv
	if (HDCP_ReadHDCP(HDCP_AKSV_OFFSET, HDCP_KSV_SIZE, outBksv) != OK)
	{
		return ERROR;
	}

	UART_Printf("-Aksv: 0x");
	for (i=0; i <HDCP_KSV_SIZE; i++)
	{
		UART_Printf("%02x", outBksv[i]);
	}
	UART_Printf("\n");

	return OK;
}


/**
 * Read Bcaps from HDCP RX port
 * 
 * @return On success, return OK;Otherwise, return ERROR
 */
static s32 HDCP_ReadBcaps(u8* outBcaps)
{
	UART_Printf("-reading Bcaps...\n");

	// read Bcaps
	if (HDCP_ReadHDCP(HDCP_BCAPS_OFFSET, 1, outBcaps) != OK)
	{
		return ERROR;
	}

	UART_Printf("-Bcaps byte = 0x%x\n", *outBcaps);

	return OK;
}


static s32 HDCP_ReadBstatus(void)
{
	u8 pBstatus[2];
	
	UART_Printf("-reading Bstatus...\n");

	// read Bcaps
	if (HDCP_ReadHDCP(HDCP_BSTATUS_OFFSET, HDCP_BSTATUS_SIZE, pBstatus) != OK)
	{
		return ERROR;
	}

	UART_Printf("-Bstatus byte = 0x%02x%02x\n", pBstatus[1], pBstatus[0]);

	return OK;
}


/**
 * Set Bksv in HDMI HW
 */
static void HDCP_SetBksv(u8* inBksv)
{
	u32 i; 
	u32* pBksv = (u32*)rHDCP_BKSV_0;

	// Write Bksv on Bksv Register
	for ( i = 0 ; i < HDCP_KSV_SIZE ; i++,pBksv++)
	{
		HdcpOutp8((u32)pBksv,inBksv[i]);
	}
}

/**
 * Set Bcaps in HDMI HW
 */
static void HDCP_SetBcaps(u8 Bcaps)
{
	HdcpOutp8(rHDCP_BCAPS, Bcaps);
}

/**
 * Write Aksv on HDCP RX port
 * 
 * @return On success, return OK;Otherwise, return ERROR
 */
static s32 HDCP_WriteAksv(void)
{
	s32 i; 
	u32* pAksv = (u32*)rHDCP_AKSV_0;
	u8   Aksv[HDCP_KSV_SIZE+1];

	// Read Aksv from Register and prepare data to send with offset
	for ( i = 1; i <= sizeof(Aksv); i++,pAksv++)
	{
		Aksv[i] = HdcpInp8((u32) pAksv);
	}
	Aksv[0] = HDCP_AKSV_OFFSET;


	UART_Printf("-Aksv: 0x");
	for (i=1; i<sizeof(Aksv); i++)
	{
		UART_Printf("%02x", Aksv[i]);
	}
	UART_Printf("\n");


	//Write Aksv 
	if (I2CDDC_Write(HDCP_RX_DEV_ADDR, HDCP_KSV_SIZE+1, Aksv) != OK)
	{
		return ERROR;
	}

	return OK;
}

/**
 * Write An on HDCP RX port
 * 
 * @return On success, return OK;Otherwise, return ERROR
 */
static s32 HDCP_WriteAn(void)
{
	s32   i; 
	u32* pAn = (u32*)rHDCP_AN_0;
	u8   An[HDCP_AN_SIZE+1];

	// Read An from Register and prepare data to send with offset
	for (i = 1; i <= sizeof(An); i++,pAn++)
	{
		An[i] = HdcpInp8((u32) pAn);
	}
	An[0] = HDCP_AN_OFFSET;

	UART_Printf("-An: 0x");
	for (i=1; i<sizeof(An); i++)
	{
		UART_Printf("%02x", An[i]);
	}
	UART_Printf("\n");

	//Write An 
	if (I2CDDC_Write(HDCP_RX_DEV_ADDR, HDCP_AN_SIZE+1, An) != OK)
	{
		 return ERROR;
	}

	return OK;
}

/**
 * Compare Ri and Rj
 *
 * @return  If matched, return OK;Otherwise, return ERROR
 */
static s32 HDCP_CompareRi(void)
{
	u32 i;
	u8 Ri[2], Rj[2];

	for(i=0 ; i<HDCP_RI_RETRY ; i++)
	{
		// read Rj from Rx through DDC
		if (HDCP_ReadHDCP(HDCP_RX_Ri_OFFSET, 2, Rj) != OK)
		{
			UART_Printf("-CompareRi() : Fail to Read Ri\n");    
			return ERROR;
		}
		// read Ri
		Ri[0] = HdcpInp8( rHDCP_RI_0 );
		Ri[1] = HdcpInp8( rHDCP_RI_1 );

		UART_Printf("-Ri = 0x%02x%02x, 	Rj = 0x%02x%02x\n", Ri[1], Ri[0], Rj[1], Rj[0]);
	                
		// compare Ri, Rj
		// Match
		// Ri = Rj & Ri !=0 
		if ( (Ri[0] == Rj[0]) && (Ri[1] == Rj[1]) && (Ri[0] | Ri[1]) )
		{
			return OK;
		}
	}
	return ERROR;
}

/**
 * Processing 2nd part of HDCP authentication
 *
 * @return  On success, return OK;Otherwise, return ERROR.
 */
static s32 HDCP_CheckRepeater(void)
{
///rb1004....must modify
#if 0
	u8 deviceCnt, Bstatus[2], timerstate = !TIMER_EXPIRED;
	// prepare maximum 
	u8 KSVlist[HDCP_MAX_DEVS*HDCP_KSV_SIZE];
	u8 RxV[HDCP_SHA1_SIZE];

	u8 Bcaps; s32 i,j;

	// set timer and start
	Timer waitForReady;
	TimerInit(&waitForReady);

	// wait for 5 seconds
	if ( TimerSet(&waitForReady,HDCP_REPEATER_TIMEOUT,0,TIME_RELATIVE,TimerCallBack,(void*)&timerstate,TIMER_ENABLE) )
	{
		UART_Printf("ERROR to Set Timer(LEO)***\n\n");
		return ERROR;
	}

	// wait for Ready bit setting
	while (1)
	{
		if (HDCP_ReadHDCP(HDCP_BCAPS_OFFSET,1,&Bcaps) != OK)
		{
			return ERROR;    
		}
		UART_Printf("Bcaps = 0x%x\n",Bcaps);        
		if ( Bcaps & (1<<HDCP_BCAPS_READY)) // ready
		{
			// disable timer
			TimerDisable(&waitForReady);

			//set Bcaps
			HDCP_SetBcaps(Bcaps);

			break;
		}
		else if ( timerstate == TIMER_EXPIRED) // timer expired
		{
			// return timeout
			return REPEATER_NOT_READY;
		}
		TaskSleep(100,TIME_RELATIVE);
	}


	// read Bstatus
	if (HDCP_ReadHDCP(HDCP_BSTATUS_OFFSET,HDCP_BSTATUS_SIZE,Bstatus) != OK)
	{
		return ERROR;
	}

	// check if Bstatus is OK
	if ( (Bstatus[0] & MAX_DEVS_EXCEEDED_MASK) || (Bstatus[1] & MAX_CASCADE_EXCEEDED_MASK))
	{
		return ERROR;
	}

	UART_Printf("BSTATUS = 0x%02x%02x\n",Bstatus[1],Bstatus[0]);

	// set Bstatus
	HdcpOutp8(rHDCP_BSTATUS_0,Bstatus[0]);
	HdcpOutp8(rHDCP_BSTATUS_1,Bstatus[1]);

	// get total device number(except repeater itself) 
	deviceCnt = Bstatus[0] & DEVICE_COUNT_MASK;

	UART_Printf("Device Cnt = %d\n",deviceCnt);

	// Set ksv list 
	if (deviceCnt == 0)
	{
		// if 0
		// set empty
		HdcpOutp8(rHDCP_KSV_LIST_CON,HDCP_KSV_LIST_EMPTY);
	}
	else // if not 0
	{
		// read KSV Lists
		if (HDCP_ReadHDCP(HDCP_KSV_LIST_OFFSET, deviceCnt*HDCP_KSV_SIZE, KSVlist) != OK)
		{
			UART_Printf("ERROR:fail to read KSV_Lists\n\n");
			return  ERROR;
		}

		UART_Printf("KsvList: ");
		for (i=0; i < (deviceCnt)*HDCP_KSV_SIZE; i++)
		{
			if ( !(i % HDCP_KSV_SIZE) )
				UART_Printf("\n0x");
				UART_Printf("%02x", KSVlist[i]);
		}
		UART_Printf("\n");

		// compare revocation list
		if ( RevocListFind(KSVlist, deviceCnt ) != OK )
		{
			UART_Printf("Set Revocation List\n");
			return ILLEGAL_DEVICE;
		}

		// write KSV_LISTs except last one
		for (j=0; j < deviceCnt; j++ )
		{
			// write one KSV_LIST
			for (i=0; i < HDCP_KSV_SIZE; i++)
			{
				HdcpOutp8(rHDCP_KSV_LIST_0+4*i,KSVlist[j*HDCP_KSV_SIZE + i]);
			}

			// if not last KSV
			if (j != deviceCnt-1)
			{
				// Write done?            
				HdcpOutp8(rHDCP_KSV_LIST_CON,0x00);
				// KSV_LIST_READ_DONE check

				// wait for 100ms
				if ( TimerSet(&waitForReady,HDCP_TIMEOUT,0,TIME_RELATIVE,TimerCallBack,(void*)&timerstate,TIMER_ENABLE) )
				{
					 UART_Printf("ERROR to Set Timer(LEO)***\n\n");
					 return ERROR;
				}
				for (;;)
				{
					if (HdcpInp8(rHDCP_KSV_LIST_CON) & HDCP_KSV_LIST_READ_DONE )
					{
						// if read done, exit loop
						// Disable Timer
						TimerDisable(&waitForReady);
						break;
					}
					else if(timerstate == TIMER_EXPIRED)
					{
						// Disable Timer
						TimerDisable(&waitForReady);
						return ERROR;    
					}    
				}
			}
		}
		// after write last one, start to calculate
		HdcpOutp8(rHDCP_KSV_LIST_CON, HDCP_KSV_LIST_END);
	}

	UART_Printf("\n\n*** Calculate SHA start !!! ***\n\n");

	// read SHA-1 from RX
	if ( HDCP_ReadHDCP(HDCP_RX_SHA1_OFFSET, HDCP_SHA1_SIZE, RxV) != OK)
	{
		return ERROR;
	}

	UART_Printf("SHA_RX: ");
	for (i=0; i < HDCP_SHA1_SIZE; i++)
	{
		if ( (i % 4) )
			UART_Printf(" ");
		else
			UART_Printf("\n");

		UART_Printf("%02x", RxV[i]);
	}
	UART_Printf("\n");

	//write SHA-1 on HW
	for ( i=0; i < HDCP_SHA1_SIZE; i++)
	{
		HdcpOutp8(rHDCP_SHA1_00+4*i,RxV[i]);
	}

	// wait for SHA-1 result
	// wait for 100ms
	if ( TimerSet(&waitForReady,HDCP_TIMEOUT,0,TIME_RELATIVE,TimerCallBack,(void*)&timerstate,TIMER_ENABLE) )
	{
		UART_Printf("ERROR to Set Timer(LEO)***\n\n");
		return ERROR;
	}
	while(1)
	{
		// wait for sha1 valid ready
		if ( HdcpInp8(rHDCP_SHA_RESULT) & (HDCP_SHA1_VALID_READY) )
		{
			// TIMER DISABLE
			TimerDisable(&waitForReady);
			UART_Printf("\n\n*** SHA Result is Ready !!! ***\n\n");
			break;
		}
		else if(timerstate == TIMER_EXPIRED)
		{
			// TIMER DISABLE
			TimerDisable(&waitForReady);
			UART_Printf("\n\n*** SHA Result is not Ready TIME_OUT !!! ***\n\n");
			//time_out
			return TIMEOUT;    
		} 
	}

	// check result
	if ( !(HdcpInp8(rHDCP_SHA_RESULT) & HDCP_SHA1_VALID) )
	{
		UART_Printf("SHA1 result ERROR!!!!\n");
		return ERROR;
	}

	// check clear
	HdcpOutp8(rHDCP_SHA_RESULT,0x00);
	UART_Printf("SHA1 result is Correct!!! Status = %d\n",HdcpInp8(rHDMI_STATUS));
#endif
	return OK;
}


/**
 * Set the timing parameter for load e-fuse key
 */
static s32 HDCP_SetEfuseTiming(void)
{
	u32 uTime, uValue;

	uTime = 1000000000/g_uPclkDsys;

	uValue = EFUSE_ADDR_WIDTH/uTime;
	if(EFUSE_ADDR_WIDTH%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_ADDR_WIDTH, uValue);

	uValue = EFUSE_SIGDEV_ASSERT/uTime;
	if(EFUSE_SIGDEV_ASSERT%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_SIGDEV_ASSERT, uValue);

	uValue = EFUSE_SIGDEV_DEASSERT/uTime;
	if(EFUSE_SIGDEV_DEASSERT%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_SIGDEV_DEASSERT, uValue);

	uValue = EFUSE_PRCHG_ASSERT/uTime;
	if(EFUSE_PRCHG_ASSERT%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_PRCHG_ASSERT, uValue);

	uValue = EFUSE_PRCHG_DEASSERT/uTime;
	if(EFUSE_PRCHG_DEASSERT%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_PRCHG_DEASSERT, uValue);

	uValue = EFUSE_FSET_ASSERT/uTime;
	if(EFUSE_FSET_ASSERT%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_FSET_ASSERT, uValue);

	uValue = EFUSE_FSET_DEASSERT/uTime;
	if(EFUSE_FSET_DEASSERT%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_FSET_DEASSERT, uValue);

	uValue = EFUSE_SENSING/uTime;
	if(EFUSE_SENSING%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_SENSING, uValue);

	uValue = EFUSE_SCK_ASSERT/uTime;
	if(EFUSE_SCK_ASSERT%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_SCK_ASSERT, uValue);

	uValue = EFUSE_SCK_DEASSERT/uTime;
	if(EFUSE_SCK_DEASSERT%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_SCK_DEASSERT, uValue);

	uValue = EFUSE_SDOUT_OFFSET/uTime;
	if(EFUSE_SDOUT_OFFSET%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_SDOUT_OFFSET, uValue);

	uValue = EFUSE_READ_OFFSET/uTime;
	if(EFUSE_READ_OFFSET%uTime)
		uValue += 1;		
	HdcpOutp8(rEFUSE_READ_OFFSET, uValue);

	return OK;
}


/**
 * Start AES decryption for HDCP Private Keys
 */
static s32 HDCP_LoadKey(void)
{
	u8 ucStatus;
	
	HDCP_SetEfuseTiming();

	HdcpOutp8(rEFUSE_CTRL, 0x1);

	do
	{
		ucStatus = HdcpInp8(rEFUSE_STATUS);
	}while(!(ucStatus&EFUSE_ECC_DONE));

	if(HdcpInp8(rEFUSE_STATUS) & EFUSE_ECC_FAIL)
	{
		return ERROR;
	}	

	return OK;
}

