/**
 * File Name : hdcp.c
 *
 * File Description :
 * This file implements the hdcp v1.1 in S5PV210 project
 *
 * Author : Hee Myung Noh
 * Dept : AP Development
 * Created Date : 2009/01/22
 * Version : 0.1
 * History 
 *  - Initial version (Hee Myung Noh 090122)
 */

#include "system.h"
#include "Library.h"
#include "v210_sfr.h"
#include "intc.h"
#include "hdmi_reg.h"
#include "hdmi.h"
#include "phy.h"
#include "hdcp.h"
#include "edid.h"

#define EDID_BLOCK									0
#define SIZE_OF_EDIDBLOCK							128
#define EDID_EXTENSION_NUMBER_POS				0x7E

#define EDID_TIMING_EXT_TAG_ADDR_POS				0
#define EDID_TIMING_EXT_REV_NUMBER				1
#define EDID_DETAILED_TIMING_OFFSET_POS			2
#define EDID_DATA_BLOCK_START_POS				4

// for Extension Data Block
#define EDID_TIMING_EXT_TAG_VAL					0x02
#define EDID_BLOCK_MAP_EXT_TAG_VAL				0xF0

#define EDID_SHORT_AUD_DEC_TAG_VAL				(1<<5)
#define EDID_SHORT_VID_DEC_TAG_VAL				(2<<5)
#define EDID_VSDB_TAG_VAL				    			(3<<5)
#define EDID_SPEAKER_ALLOCATION_TAG_VAL			(4<<5)
#define EDID_VESA_DTC_TAG_VAL						(5<<5)
#define EDID_RESERVED_TAG_VAL						(6<<5)

#define EDID_EXTENDED_TAG_VAL						(7<<5)
#define EDID_EXTENDED_COLORIMETRY_VAL			5
#define EDID_EXTENDED_COLORIMETRY_BLOCK_LEN		3

#define EDID_TAG_CODE_MASK						(1<<7 | 1<<6 | 1<<5) 
#define EDID_DATA_BLOCK_SIZE_MASK				(1<<4 | 1<<3 | 1<<2 | 1<<1 | 1<<0) 

#define EDID_VSDB_MIN_LENGTH_VAL			5

// for SAD
#define EDID_SAD_CODE_MASK				(1<<6 | 1<<5 | 1<<4 | 1<<3)
#define EDID_SAD_CHANNEL_MASK			(1<<2 | 1<<1 | 1<<0)

// for CS
#define EDID_COLOR_SPACE_POS				3
#define EDID_YCBCR444_CS_MASK				(1<<5)
#define EDID_YCBCR422_CS_MASK				(1<<4)

// for Color Depth
#define EDID_DC_48_VAL						(1<<6)
#define EDID_DC_36_VAL						(1<<5)
#define EDID_DC_30_VAL						(1<<4)
#define EDID_DC_YCBCR_VAL					(1<<3)

#define EDID_DC_POS							6
#define EDID_DC_MASK						(EDID_DC_48_VAL | EDID_DC_36_VAL| EDID_DC_30_VAL | EDID_DC_YCBCR_VAL)

// for colorimetry
#define EDID_XVYCC601_MASK					(1<<0)
#define EDID_XVYCC709_MASK					(1<<1)
#define EDID_EXTENDED_MASK				(1<<0|1<<1|1<<2)

// for Established Timings
#define EDID_ET_POS							0x23
#define EDID_ET_640x480p_VAL				0x20

// for DTD
#define EDID_DTD_START_ADDR				0x36
#define EDID_DTD_BYTE_LENGTH				18
#define EDID_DTD_TOTAL_LENGTH				EDID_DTD_BYTE_LENGTH*4

#define SIZE_OF_BYTE						8
#define EDID_DTD_PIXELCLOCK_POS1			0
#define EDID_DTD_PIXELCLOCK_POS2			1

#define EDID_DTD_HBLANK_POS1				3
#define EDID_DTD_HBLANK_POS2				4
#define EDID_DTD_HBLANK_POS2_MASK		0xF

#define EDID_DTD_HACTIVE_POS1				2
#define EDID_DTD_HACTIVE_POS2				4
#define EDID_DTD_HACTIVE_POS2_MASK		0xF0

#define EDID_DTD_VBLANK_POS1				6
#define EDID_DTD_VBLANK_POS2				7
#define EDID_DTD_VBLANK_POS2_MASK		0x0F

#define EDID_DTD_VACTIVE_POS1				5
#define EDID_DTD_VACTIVE_POS2				7
#define EDID_DTD_VACTIVE_POS2_MASK		0xF0

#define EDID_DTD_INTERLACE_POS			17
#define EDID_DTD_INTERLACE_MASK			(1<<7)


// for SVD
#define EDID_SVD_VIC_MASK					0x7F

// for SAD
#define SHORT_AUD_DESCRIPTOR_LPCM		0x01 /* TV Supported Audio format : LPCM */
#define SHORT_AUD_DESCRIPTOR_AC3			0x02 /* TV Supported Audio format : AC3 */
#define SHORT_AUD_DESCRIPTOR_MPEG1		0x04 /* TV Supported Audio format : MPEG1 */
#define SHORT_AUD_DESCRIPTOR_MP3			0x08 /* TV Supported Audio format : MP3 */
#define SHORT_AUD_DESCRIPTOR_MPEG2		0x10 /* TV Supported Audio format : MPEG2 */
#define SHORT_AUD_DESCRIPTOR_AAC			0x20 /* TV Supported Audio format : AAC */
#define SHORT_AUD_DESCRIPTOR_DTS			0x40 /* TV Supported Audio format : DTS */
#define SHORT_AUD_DESCRIPTOR_ATRAC		0x80 /* TV Supported Audio format : ATRAC */

#define EDID_SAD_CODE_MASK				(1<<6 | 1<<5 | 1<<4 | 1<<3)
#define EDID_SAD_CHANNEL_MASK			(1<<2 | 1<<1 | 1<<0)
#define EDID_SAD_192KHZ_MASK				(1<<6)
#define EDID_SAD_176KHZ_MASK				(1<<5)
#define EDID_SAD_96KHZ_MASK				(1<<4)
#define EDID_SAD_88KHZ_MASK				(1<<3)
#define EDID_SAD_48KHZ_MASK				(1<<2)
#define EDID_SAD_44KHZ_MASK				(1<<1)
#define EDID_SAD_32KHZ_MASK				(1<<0)

#define EDID_SAD_WORD_24_MASK			(1<<2)
#define EDID_SAD_WORD_20_MASK			(1<<1)
#define EDID_SAD_WORD_16_MASK 			(1<<0)

// for CEC
#define EDID_CEC_PHYICAL_ADDR				4


typedef enum 
{
	SIZE_OF_EDID = SIZE_OF_EDIDBLOCK * NUM_OF_POSSIBLE_EDID_BLOCK
}EDID_Size;        // unit of extension(128)* up to 4 extension = 512

// buffer for EDID data
static u8 gpEdidBuffer[SIZE_OF_EDID] = {0, };

// EDID video parameter for parsing EDID
static const struct params
{
	u32  InterlaceMode;  // interlace = 1
	u32  HBLANK;
	u32  VBLANK;
	u32  HACTIVE;
	u32  VACTIVE;
	u32  VIC;			//Video ID Code (refer to CEA-861D Table3)
	u32  VIC16_9; 
	u32  PixelClock;
} aEDIDVideoParameters[] =
{
        {0, 160 ,45 ,640    ,480    ,1  ,1  ,   2520,}, // 640x480p@60Hz
        {0, 138 ,45 ,720    ,480    ,2  ,3  ,   2702,}, // 720x480p@60Hz
        {0, 370 ,30 ,1280   ,720    ,4  ,4  ,   7425,}, // 1280x720p@60Hz
        {1, 280 ,22 ,1920   ,540    ,5  ,5  ,   7425,}, // 1920x1080i@60Hz
        {1, 276 ,22 ,1440   ,240    ,6  ,7  ,   2702,}, // 720x480i@60Hz
        {0, 276 ,22 ,1440   ,240    ,8  ,9  ,   2702,}, // 720x240p@60Hz(pixel repetition)
        {1, 552 ,22 ,2880   ,240    ,10 ,11 ,   5405,}, // 2880x480i@60Hz(pixel repetition)
        {0, 552 ,22 ,2880   ,480    ,12 ,13 ,   5405,}, // 2880x480p@60Hz(pixel repetition)
        {0, 276 ,45 ,1440   ,480    ,14 ,15 ,   5405,}, // 1440x480p@60Hz(pixel repetition)
        {0, 280 ,45 ,1920   ,1080   ,16 ,16 ,   14850,},// 1920x1080p@60Hz
        {0, 144 ,49 ,720    ,576    ,17 ,18 ,   2700,}, // 720x576p@50Hz
        {0, 700 ,30 ,1280   ,720    ,19 ,19 ,   7425,}, // 1280x720p@50Hz
        {1, 720 ,22 ,1920   ,540    ,20 ,20 ,   7425,}, // 1920x1080i@50Hz
        {1, 288 ,24 ,1440   ,288    ,21 ,22 ,   2700,}, // 720x576i@50Hz(pixel repetition)
        {0, 288 ,24 ,1440   ,288    ,23 ,24 ,   2700,}, // 720x288p@50Hz(pixel repetition)
        {1, 576 ,24 ,2880   ,288    ,25 ,26 ,   5400,}, // 2880x576i@50Hz(pixel repetition)
        {0, 576 ,24 ,2880   ,288    ,27 ,28 ,   5400,}, // 2880x288p@50Hz(pixel repetition)
        {0, 288 ,49 ,1440   ,576    ,29 ,30 ,   5400,}, // 1440x576p@50Hz(pixel repetition)
        {0, 720 ,45 ,1920   ,1080   ,31 ,31 ,   14850,},    // 1920x1080p@50Hz
        {0, 830 ,45 ,1920   ,1080   ,32 ,32 ,   7425,}, // 1920x1080p@24Hz
        {0, 720 ,45 ,1920   ,1080   ,33 ,33 ,   7425,}, // 1920x1080p@25Hz
        {0, 720 ,45 ,1920   ,1080   ,34 ,34 ,   7425,}, // 1920x1080p@30Hz
        {0, 552 ,45 ,2880   ,480    ,35 ,36 ,   10810,},    // 2880x480p@60Hz(pixel repetition)
        {0, 576 ,49 ,2880   ,576    ,37 ,38 ,   10800,},    // 2880x576p@60Hz(pixel repetition)
        {1, 384 ,85 ,1920   ,540    ,39 ,39 ,   7200,}, // 1920x1080i@50Hz
        {1, 720 ,22 ,1920   ,540    ,40 ,40 ,   14850,},    // 1920x1080i@100Hz
        {0, 700 ,30 ,1280   ,720    ,41 ,41 ,   14850,},    // 1280x720p@100Hz
        {0, 144 ,49 ,720    ,576    ,42 ,43 ,   5400,}, // 720x576p@100Hz
        {1, 288 ,24 ,720    ,288    ,44 ,45 ,   5400,}, // 720x576i@100Hz(pixel repetition)
        {1, 280 ,22 ,1920   ,540    ,46 ,46 ,   14835,},    // 1920x1080i@120Hz
        {0, 370 ,30 ,1280   ,720    ,47 ,47 ,   14835,},    // 1280x720p@120Hz
        {0, 138 ,45 ,720    ,480    ,48 ,49 ,   5405,}, // 720x480p@120Hz
        {1, 276 ,22 ,1440   ,240    ,50 ,51 ,   5405,}, // 720x480i@120Hz
        {0, 144 ,49 ,720    ,576    ,52 ,53 ,   10800,},    // 720x576p@200Hz
        {1, 288 ,24 ,1440   ,288    ,54 ,55 ,   10800,},    // 720x576i@200Hz(pixel repetition)
        {0, 138 ,45 ,720    ,480    ,56 ,57 ,   10800,},    // 720x480p@240Hz
        {1, 276 ,22 ,1440   ,240    ,58 ,59 ,   10800,},    // 720x480i@240Hz
};

static s32 CalcChecksum(unsigned char *buffer, s32 size);
static s32 ReadEDIDBlock(s32 blockNum);
static s32 IsContainVIC(s32 extension, s32 VIC);
static s32 IsColorDepthSupported(ColorDepth colorDepth, s32 YCBCR444);
static s32 GetVSDBStartAddr(s32 extension);
static s32 IsContainVideoDTD(s32 extension, VideoFormat videoMode);
static s32 IsTimingExtension(s32 extension);

/**
 * Reset EDID. When HPD is deasserted, call this.
 */
void ResetEDID(void)
{
    gpEdidBuffer[1] = 0x00;
    gpEdidBuffer[EDID_EXTENSION_NUMBER_POS] = 0x00;
}

/**
 * Check if RX supports HDMI Mode or not 
 *
 * @return  If Rx supports HDMI mode, return OK;Otherwise, return ERROR.
 */ 
s32 IsHDMIModeSupported(void)
{
	s32 extension = 0, i;    

	// read EDID
	if(ReadEDID(NUM_OF_RETRY) != OK)
	{
		UART_Printf("Error : fail to read the EDID: %s, %s\n",__FILE__,__LINE__);
		return ERROR;
	}

	// get number of extension block
	extension = gpEdidBuffer[EDID_EXTENSION_NUMBER_POS];

	UART_Printf("edid extension = %d\n",(s32)extension);


	// check if extension is more than 1, and first extension block is not "block map extension".
	if (extension > 1 && gpEdidBuffer[SIZE_OF_EDIDBLOCK] != EDID_BLOCK_MAP_EXT_TAG_VAL)
	{
		return ERROR;
	}


	// find VSDB(Vendor Specific Data Block)
	for ( i = 1 ; i <= extension ; i++ )
	{
		if ( IsTimingExtension(extension) == OK ) // if it's timing extension block
		{
			if (GetVSDBStartAddr(extension) > 0) // check block
				return OK;
		}
	}
	return ERROR;
}

/**
 * Based on EDID, return whether Rx supports the requested video parameters
 *
 * @return    If it's possible for Rx to support the video mode - return true; otherwise return false
 */
s32 IsVideoModeSupported(VideoFormat videoMode, PixelAspectRatio pixelRatio)
{
	s32 extension = 0,i,vic;    

	// read EDID
	if(ReadEDID(NUM_OF_RETRY) != OK)
	{
		UART_Printf("Error : fail to read the EDID: %s, %s\n",__FILE__,__LINE__);
		return ERROR;
	}

	// check ET(Established Timings) for 640x480p@60Hz
	if ( videoMode == v640x480p_60Hz && (gpEdidBuffer[EDID_ET_POS] & EDID_ET_640x480p_VAL) ) // it support
	{
		return OK;
	}

	// check STI(Standard Timing Identification)
	// do not need


	// check DTD(Detailed Timing Description) of EDID block(0th)
	if ( IsContainVideoDTD(EDID_BLOCK,videoMode) == OK)
	{
		return OK;
	}

	// check EDID Extension

	// get number of extension block
	extension = gpEdidBuffer[EDID_EXTENSION_NUMBER_POS];

	// check if extension is more than 1, and first extension block is not block map.
	if (extension > 1 && gpEdidBuffer[SIZE_OF_EDIDBLOCK] != EDID_BLOCK_MAP_EXT_TAG_VAL)
	{
		return ERROR;
	}     

	// get VIC to find
	vic = (pixelRatio == HDMI_PIXEL_RATIO_16_9) ? aEDIDVideoParameters[videoMode].VIC16_9 : aEDIDVideoParameters[videoMode].VIC;

	// find VSDB
	for ( i = 1 ; i <= extension ; i++ )
	{
		if ( IsTimingExtension(extension) == OK ) // if it's timing block
		{
			if ( IsContainVIC(extension,vic) == OK || IsContainVideoDTD(extension,videoMode) == OK )
			return OK;
		}
	}    

	return ERROR;
}

/**
 * Based on EDID, return whether Rx supports the requested color space and depth.
 *
 * @return If Rx supports them, return OK;Otherwise, return ERROR.
 */
s32 IsColorSpaceAndDepthSupported(ColorSpace colorSpace,ColorDepth colorDepth)
{
	s32 extension = 0, i;    

	// check EDID data is valid or not
	// read EDID
	if(ReadEDID(NUM_OF_RETRY) != OK)
	{
		UART_Printf("Error : fail to read the EDID: %s, %s\n",__FILE__,__LINE__);
		return ERROR;
	}

	// get number of extension block
	extension = gpEdidBuffer[EDID_EXTENSION_NUMBER_POS];

	// check if extension is more than 1, and first extension block is not block map.
	if (extension > 1 && gpEdidBuffer[SIZE_OF_EDIDBLOCK] != EDID_BLOCK_MAP_EXT_TAG_VAL)
	{
		return ERROR;
	}

	// Check color space and depth
	for ( i = 1 ; i <= extension ; i++ )
	{
		if ( IsTimingExtension(extension) == OK )  // if it's timing block
		{
			// read Color Space
			s32 CS = gpEdidBuffer[extension*SIZE_OF_EDIDBLOCK + EDID_COLOR_SPACE_POS];

			switch(colorSpace)
			{
				// YCBCR444
				case HDMI_CS_YCBCR444:
					if ( CS & EDID_YCBCR444_CS_MASK)
					{
						UART_Printf("Sink support YCBCR 444\n");
						return IsColorDepthSupported(colorDepth, TRUE);
					}
					break;
					
				// YCBCR422
				case HDMI_CS_YCBCR422:
					if ( CS & EDID_YCBCR422_CS_MASK)
					{
						UART_Printf("Sink support YCBCR 422\n");
						return IsColorDepthSupported(colorDepth, FALSE);
					}
					break;
				
				// RGB
				default:
					return IsColorDepthSupported(colorDepth, FALSE);
			}
		} // if
	} // for
	return ERROR;
}

/**
 * Based on EDID, return whether Rx supports the requested colorimetry.
 *
 * @param   colorimetry    [in]  Colorimetry \n
 *
 * @return  If Rx supports it, return OK;Otherwise return ERROR.
 */
s32 IsColorimetrySupported(HDMIColorimetry colorimetry)
{
	s32 extension = 0, i;    

	// do not need to parse
	if (colorimetry == HDMI_COLORIMETRY_NO_DATA)
		return OK;

	// read EDID
	if(ReadEDID(NUM_OF_RETRY) != OK)
	{
		UART_Printf("Error : fail to read the EDID: %s, %s\n",__FILE__,__LINE__);
		return ERROR;
	}

	// get number of extension block
	extension = gpEdidBuffer[EDID_EXTENSION_NUMBER_POS];

	// check if extension is more than 1, and first extension block is not block map.
	if (extension > 1 && gpEdidBuffer[SIZE_OF_EDIDBLOCK] != EDID_BLOCK_MAP_EXT_TAG_VAL)
	{
		return ERROR;
	}

	// check colorimetry block
	for ( i = 1 ; i <= extension ; i++ )
	{
		if ( IsTimingExtension(extension) == OK ) // if it's timing block
		{
			// check address
			u32 ExtAddr = extension*SIZE_OF_EDIDBLOCK + EDID_DATA_BLOCK_START_POS;
			u32 EndAddr = extension*SIZE_OF_EDIDBLOCK + gpEdidBuffer[extension*SIZE_OF_EDIDBLOCK + EDID_DETAILED_TIMING_OFFSET_POS];
			u32 tag,blockLen;

			// while 
			while ( ExtAddr < EndAddr )
			{
				// find the block tag and length
				// tag
				tag = gpEdidBuffer[ExtAddr] & EDID_TAG_CODE_MASK;
				// block len
				blockLen = (gpEdidBuffer[ExtAddr] & EDID_DATA_BLOCK_SIZE_MASK) + 1;

				// check if it is colorimetry block
				if (tag == EDID_EXTENDED_TAG_VAL && // extended tag
				gpEdidBuffer[ExtAddr+1] == EDID_EXTENDED_COLORIMETRY_VAL && // colorimetry block
				(blockLen-1) == EDID_EXTENDED_COLORIMETRY_BLOCK_LEN )// check length
				{
					// get supported DC value
					s32 colorimetry = (gpEdidBuffer[ExtAddr + 2]);
					s32 extColorimetry = (gpEdidBuffer[ExtAddr + 3]);
					            
					UART_Printf("EDID colorimetry = %x\n",colorimetry);
					UART_Printf("EDID extColorimetry = %x\n",extColorimetry);

					// check colorDepth
					switch (colorimetry)
					{
						case HDMI_COLORIMETRY_ITU601:
							if (colorimetry & EDID_XVYCC601_MASK)
								return OK;
							break;    
						case HDMI_COLORIMETRY_ITU709:
							if (colorimetry & EDID_XVYCC709_MASK)
								return OK;
							break;
						case HDMI_COLORIMETRY_EXTENDED:
							if (extColorimetry & EDID_EXTENDED_MASK)
								return OK;
							break;
						default:
							break;
					}
					return ERROR;
				} // if VSDB block
				// else find next block
				ExtAddr += blockLen;    
			} // while()
		} // if
	} // for

	return ERROR;
}

/** 
 * Check if Rx supports requested audio parameters.
 *
 * @return  If Rx supports them, return OK;Otherwise, return ERROR.
 */
s32 IsAudioSupported(AudioParameter audioPara)
{
	s32 extension = 0,i;    

	// read EDID
	if(ReadEDID(NUM_OF_RETRY) != OK)
	{
		UART_Printf("Error : fail to read the EDID: %s, %s\n",__FILE__,__LINE__);
		return ERROR;
	}

	// check EDID Extension

	// get number of extension block
	extension = gpEdidBuffer[EDID_EXTENSION_NUMBER_POS];

	// check if extension is more than 1, and first extension block is not block map.
	if (extension > 1 && gpEdidBuffer[SIZE_OF_EDIDBLOCK] != EDID_BLOCK_MAP_EXT_TAG_VAL)
	{
		return ERROR;
	}     

	// find timing block
	for ( i = 1 ; i <= extension ; i++ )
	{
		if ( IsTimingExtension(extension) == OK ) // if it's timing block
		{
			// find Short Audio Description
			u32 StartAddr = extension*SIZE_OF_EDIDBLOCK;
			u32 ExtAddr = StartAddr + EDID_DATA_BLOCK_START_POS;
			u32 tag,blockLen;
			u32 DTDStartAddr = gpEdidBuffer[StartAddr + EDID_DETAILED_TIMING_OFFSET_POS];

			// while
			while ( ExtAddr < StartAddr + DTDStartAddr )
			{
				// find the block tag and length
				// tag
				tag = gpEdidBuffer[ExtAddr] & EDID_TAG_CODE_MASK;
				// block len
				blockLen = (gpEdidBuffer[ExtAddr] & EDID_DATA_BLOCK_SIZE_MASK) + 1;

				UART_Printf("tag = %d\n",tag);
				UART_Printf("blockLen = %d\n",blockLen-1);

				// check if it is short video description
				if (tag == EDID_SHORT_AUD_DEC_TAG_VAL)
				{
					// if so, check SAD
					s32 i;
					s32 audioFormat,channelNum,sampleFreq,wordLen;
					for (i=1; i < blockLen; i+=3)                
					{
						audioFormat = gpEdidBuffer[ExtAddr+i] & EDID_SAD_CODE_MASK;
						channelNum = gpEdidBuffer[ExtAddr+i] & EDID_SAD_CHANNEL_MASK;
						sampleFreq = gpEdidBuffer[ExtAddr+i+1];
						wordLen = gpEdidBuffer[ExtAddr+i+2];

						UART_Printf("EDIDAudioFormatCode = %d\n",audioFormat);
						UART_Printf("EDIDChannelNumber= %d\n",channelNum);
						UART_Printf("EDIDSampleFreq= %d\n",sampleFreq);
						UART_Printf("EDIDWordLeng= %d\n",wordLen);

						// check parameter              
						// check audioFormat
						if ( audioFormat == (audioPara.FormatCode << 3)     &&  // format code 
						channelNum >= (audioPara.MaxChannels - 1)         &&  // channel number
						(sampleFreq & (1<<audioPara.SampleFreq))      ) // sample frequency
						{
							if (audioFormat == LPCM_FORMAT) // check wordLen
							{
								if ( wordLen & (1<<audioPara.WordLength) )
									return OK;
								else
									return ERROR;
							}
							return OK; // if not LPCM
						}
					} // for
				} // if tag
				// else find next block
				ExtAddr += blockLen;    
			} // while()
		} // if
	} // for    

	return ERROR;    
}

/** 
 * Get CEC physical address.
 * 
 * @param   outAddr [out]   CEC physical address
 *
 * @return  On success, return OK;Otherwise, return ERROR.
 */
s32 GetCECPhysicalAddr(u32* outAddr)
{
	s32 extension = 0, i;    
	u32 StartAddr;

	// check EDID data is valid or not
	// read EDID
	if(ReadEDID(NUM_OF_RETRY) != OK)
	{
		UART_Printf("Error : fail to read the EDID: %s, %s\n",__FILE__,__LINE__);
		return ERROR;
	}

	// get number of extension block
	extension = gpEdidBuffer[EDID_EXTENSION_NUMBER_POS];

	// check if extension is more than 1, and first extension block is not block map.
	if (extension > 1 && gpEdidBuffer[SIZE_OF_EDIDBLOCK] != EDID_BLOCK_MAP_EXT_TAG_VAL)
	{
		return ERROR;
	}

	// find VSDB
	for ( i = 1 ; i <= extension ; i++ )
	{
		if ( IsTimingExtension(extension) == OK // if it's timing block
		&& (StartAddr = GetVSDBStartAddr(extension)) > 0 )   // check block
		{
			// get cec physical address
			u32 phyAddr = gpEdidBuffer[StartAddr + EDID_CEC_PHYICAL_ADDR] << 8;
			phyAddr |= gpEdidBuffer[StartAddr + EDID_CEC_PHYICAL_ADDR+1];

			UART_Printf("phyAddr = %x\n",phyAddr);

			*outAddr = phyAddr;

			return OK;
		} // if
	} // for

	return ERROR;
}



/**
 * Calculate checksum. 
 *
 * @return  If checksum is not correct, return ERROR; Otherwise, return OK.
 */ 
static s32 CalcChecksum(unsigned char *buffer, s32 size)
{
	u8 i,sum;

	// calculate checksum
	for (sum = 0, i = 0 ; i < size; i++)
	{
		sum += buffer[i];
	}

	// check checksum
	if (sum != 0)
	{
		return ERROR;
	}

	return OK;
}

/** 
 * Read 'block number'th EDID Blcok 
 * 
 * @return  On success, return OK; Otherwise, return ERROR;
 */ 
static s32 ReadEDIDBlock(s32 blockNum)
{
	u8 ucSegPnt,ucOffset;

	// find segment pointer
	ucSegPnt = (blockNum/2);

	// find offset
	ucOffset = (blockNum%2)*SIZE_OF_EDIDBLOCK;

	// read one block
	if (EDDC_Read(ucSegPnt, ucOffset, (u8)SIZE_OF_EDIDBLOCK, (u8 *)(gpEdidBuffer+blockNum*SIZE_OF_EDIDBLOCK)) != OK)
	{
		UART_Printf("Error : fail to read the EDID %s, %s\n",__FILE__,__LINE__);
		return ERROR;
	}

	if (CalcChecksum(gpEdidBuffer+blockNum*SIZE_OF_EDIDBLOCK, SIZE_OF_EDIDBLOCK) != OK)
	{
		UART_Printf("Error : CheckSum Error : the EDID(%d block) %s, %s\n",blockNum, __FILE__,__LINE__);
		return ERROR;
	}

	// print data for debugging
	for(ucOffset = 0 ; ucOffset < SIZE_OF_EDIDBLOCK ; ucOffset++)
	{
		UART_Printf("0x%02X", gpEdidBuffer[blockNum*SIZE_OF_EDIDBLOCK+ucOffset]);
		if ((ucOffset%16) != 15)
			UART_Printf(" ");
		else
			UART_Printf("\n");
	}

	return OK;
}

/** 
 * Check if 'extension'th block is timing extension block or not
 *
 * @return  If 'extension'th block is timing extension block, return OK;Otherwise return ERROR.
 */
static s32 IsTimingExtension(s32 extension)
{
	if (gpEdidBuffer[extension*SIZE_OF_EDIDBLOCK] == EDID_TIMING_EXT_TAG_VAL 
		&& gpEdidBuffer[extension*SIZE_OF_EDIDBLOCK + EDID_TIMING_EXT_REV_NUMBER] >= 3)
		return OK;
	return ERROR;    
}

/** 
 * Get VSDB(Vendor Specific Data Block) Start offset.
 *
 * @return  If there is VSDB in 'extension'th block, return the offset of VSDB; @n
 *          Otherwise, return 0
 */
static s32 GetVSDBStartAddr(s32 extension)
{
	u32 StartAddr = extension*SIZE_OF_EDIDBLOCK;
	u32 ExtAddr = StartAddr + EDID_DATA_BLOCK_START_POS;
	u32 tag,blockLen;
	u32 DTDStartAddr = gpEdidBuffer[StartAddr + EDID_DETAILED_TIMING_OFFSET_POS];

	// check if there is HDMI VSDB
	while ( ExtAddr < StartAddr + DTDStartAddr )
	{
		// find the block tag and length
		// tag
		tag = gpEdidBuffer[ExtAddr] & EDID_TAG_CODE_MASK;
		// block len
		blockLen = (gpEdidBuffer[ExtAddr] & EDID_DATA_BLOCK_SIZE_MASK) + 1;

		// check if it is HDMI VSDB
		// if so, check identifier value, if it's hdmi vsbd - return HDMI mode            
		if (	tag == EDID_VSDB_TAG_VAL && 
			gpEdidBuffer[ExtAddr+1] == 0x03 &&
			gpEdidBuffer[ExtAddr+2] == 0x0C &&
			gpEdidBuffer[ExtAddr+3] == 0x0 &&
			blockLen > EDID_VSDB_MIN_LENGTH_VAL )
		{
			return ExtAddr;
		}
		// else find next block
		ExtAddr += blockLen;    
	} // while()

	// it means error
	return 0;
}

/** 
 * Find Video DTD(Supported Detail Timing Descriptor -> 18 Bytes Information)
 *
 * @return  If 'extension'th block contains video DTD of 'videoMode', return OK;@n
 *          Otherwise, return ERROR.
 */
static s32 IsContainVideoDTD(s32 extension, VideoFormat videoMode)
{
	u32 i,StartAddr,EndAddr;

	// define StartAddr, EndAddr
	if (extension == EDID_BLOCK)
	{
		StartAddr = EDID_DTD_START_ADDR;
		EndAddr = StartAddr + EDID_DTD_TOTAL_LENGTH;
	}
	else
	{
		StartAddr = gpEdidBuffer[extension*SIZE_OF_EDIDBLOCK + EDID_DETAILED_TIMING_OFFSET_POS];
		EndAddr = gpEdidBuffer[(extension+1)*SIZE_OF_EDIDBLOCK];
	}

	// check DTD(Detailed Timing Description)
	for (i = StartAddr; i < EndAddr; i+= EDID_DTD_BYTE_LENGTH)
	{
		u32 hblank = 0, hactive = 0, vblank = 0, vactive = 0, interlaced = 0, pixelclock = 0;

		// get pixel clock
		pixelclock = (gpEdidBuffer[i+EDID_DTD_PIXELCLOCK_POS2] << SIZE_OF_BYTE);
		pixelclock |= gpEdidBuffer[i+EDID_DTD_PIXELCLOCK_POS1];

		if (!pixelclock)
		{
			continue;
		}

		// get HBLANK value in pixels
		hblank = gpEdidBuffer[i+EDID_DTD_HBLANK_POS2] & EDID_DTD_HBLANK_POS2_MASK;
		hblank <<= SIZE_OF_BYTE; // lower 4 bits
		hblank |= gpEdidBuffer[i+EDID_DTD_HBLANK_POS1];

		// get HACTIVE value in pixels
		hactive = gpEdidBuffer[i+EDID_DTD_HACTIVE_POS2] & EDID_DTD_HACTIVE_POS2_MASK;
		hactive <<= (SIZE_OF_BYTE/2); // upper 4 bits
		hactive |= gpEdidBuffer[i+EDID_DTD_HACTIVE_POS1];

		// get VBLANK value in pixels
		vblank = gpEdidBuffer[i+EDID_DTD_VBLANK_POS2] & EDID_DTD_VBLANK_POS2_MASK;
		vblank <<= SIZE_OF_BYTE; // lower 4 bits
		vblank |= gpEdidBuffer[i+EDID_DTD_VBLANK_POS1];

		// get VACTIVE value in pixels
		vactive = gpEdidBuffer[i+EDID_DTD_VACTIVE_POS2] & EDID_DTD_VACTIVE_POS2_MASK;
		vactive <<= (SIZE_OF_BYTE/2); // upper 4 bits
		vactive |= gpEdidBuffer[i+EDID_DTD_VACTIVE_POS1];

		// get Interlaced Mode Value
		interlaced = (s32)(gpEdidBuffer[i+EDID_DTD_INTERLACE_POS] & EDID_DTD_INTERLACE_MASK);
		if (interlaced) 
			interlaced = 1;

		if (	hblank == aEDIDVideoParameters[videoMode].HBLANK && vblank == aEDIDVideoParameters[videoMode].VBLANK // blank
			&& hactive == aEDIDVideoParameters[videoMode].HACTIVE && vactive == aEDIDVideoParameters[videoMode].VACTIVE ) //line 
		{
			s32 EDIDpixelclock = aEDIDVideoParameters[videoMode].PixelClock;

			///rb1004...090207...why divided by 100 ????
			EDIDpixelclock /= 100; 
			pixelclock /= 100;

			if (pixelclock == EDIDpixelclock)
			{
				UART_Printf("Sink Support the Video mode\n");
				return OK;
			}
		}
	} // for
	return ERROR;    
}

/**
 * Read the EDID from RX.
 *
 * @param    Retry            [in]  retry to read
 *
 * @return   On Success, return OK; otherwise return ERROR
 */
s32 ReadEDID(s32 Retry)
{
	s32 sResult = ERROR;
	u32 uBlockNum = 1, uEdidExtensions = 0;

	// check if it was read before
	if (gpEdidBuffer[1] == 0xFF && gpEdidBuffer[2] == 0xFF)
	{
		UART_Printf("EDID was read before...\n");
		return OK;
	}

	UART_Printf("reading EDID Block 0...\n");

	// read first block
	for ( ; Retry >=0 ; Retry--)
	{
		// Read EDID block (0th block)
		if (ReadEDIDBlock(EDID_BLOCK) == OK)
		{
			sResult = OK;
			break;
		}
	}

	// get number of extension block
	uEdidExtensions = gpEdidBuffer[EDID_EXTENSION_NUMBER_POS];

	// read Extension
	for (; (Retry >= 0) && (uBlockNum <= uEdidExtensions) ; Retry--)
	{
		UART_Printf("reading EDID Block %d...\n", uBlockNum);
		if (ReadEDIDBlock(uBlockNum) == OK) 
		{
			uBlockNum++;
			sResult = OK;
		}
		else
		{
			ResetEDID();
			sResult = ERROR;
		}
	}
	return sResult;
}

/**
 * Check if VIC(Video ID Code...refer to CEA-861-D Table3) is contained in 'extension'th EDID block.
 *
 * @return  On success, return OK;Otherwise return ERROR
 */
static s32 IsContainVIC(s32 extension, s32 VIC)
{
	u32 StartAddr = extension*SIZE_OF_EDIDBLOCK;
	u32 ExtAddr = StartAddr + EDID_DATA_BLOCK_START_POS;
	u32 tag,blockLen;
	u32 DTDStartAddr = gpEdidBuffer[StartAddr + EDID_DETAILED_TIMING_OFFSET_POS];

	// while
	while ( ExtAddr < StartAddr + DTDStartAddr )
	{
		// find the block tag and length
		// tag
		tag = gpEdidBuffer[ExtAddr] & EDID_TAG_CODE_MASK;
		// block len
		blockLen = (gpEdidBuffer[ExtAddr] & EDID_DATA_BLOCK_SIZE_MASK) + 1;

		UART_Printf("tag = %d\n",tag);
		UART_Printf("blockLen = %d\n",blockLen-1);

		// check if it is short video description
		if (tag == EDID_SHORT_VID_DEC_TAG_VAL)
		{
			// if so, check SVD(Short Video Descriptor)
			u32 i;
			
			for (i=1; i < blockLen;i++)                
			{
				UART_Printf("EDID_VIC(Video ID Code) = %d\n",gpEdidBuffer[ExtAddr+i] & EDID_SVD_VIC_MASK);
				UART_Printf("VIC = %d\n",VIC);

				// check VIC with SVDB      
				if (VIC == (gpEdidBuffer[ExtAddr+i] & EDID_SVD_VIC_MASK) )
				{
					UART_Printf("Sink Device supports requested video mode\n");
					return OK;
				}
			} // for
		} // if tag
		// else find next block
		ExtAddr += blockLen;    
	} // while()

	return ERROR;
}

/**
  * Based on EDID, return whether Rx can support the color depth to set or not.
  *
  * @param  colorDepth    [in]  color depth
  *
  * @return If it's possible for Rx to support the color depth - return OK;@n
  *         Otherwise return ERROR.
  */
static s32 IsColorDepthSupported(ColorDepth colorDepth, s32 YCBCR444)
{
	s32 extension = 0, i;
	u32 StartAddr;    
	// check EDID data is valid or not
	// read EDID
	if(ReadEDID(NUM_OF_RETRY) != OK)
	{
		UART_Printf("Error : fail to read the EDID: %s, %s\n",__FILE__,__LINE__);
		return ERROR;
	}

	// get number of extension block
	extension = gpEdidBuffer[EDID_EXTENSION_NUMBER_POS];

	// check if extension is more than 1, and first extension block is not block map.
	if (extension > 1 && gpEdidBuffer[SIZE_OF_EDIDBLOCK] != EDID_BLOCK_MAP_EXT_TAG_VAL)
	{
		return ERROR;
	}

	// find VSDB
	for ( i = 1 ; i <= extension ; i++ )
	{
		if (	IsTimingExtension(extension) == OK // if it's timing block
			&& (StartAddr = GetVSDBStartAddr(extension)) > 0 )   // check block
		{
			// get supported DC value
			// s32 tempDC1 = (s32)(gpEdidBuffer[tempAddr+EDID_DC_POS]);
			u32 uDeepColor = gpEdidBuffer[StartAddr + EDID_DC_POS] & EDID_DC_MASK;	//rb1004...?????????????????????????????/

			UART_Printf("EDID deepColor = %x\n",uDeepColor);

			// check supported DeepColor

			// if YCBCR444
			if (YCBCR444)
			{
				if ( !(uDeepColor & EDID_DC_YCBCR_VAL))
					return ERROR;
			}

			// check colorDepth
			switch (colorDepth)
			{
				case HDMI_CD_48:
					uDeepColor &= EDID_DC_48_VAL;
					break;
				case HDMI_CD_36:
					uDeepColor &= EDID_DC_36_VAL;
					break;
				case HDMI_CD_30:
					uDeepColor &= EDID_DC_30_VAL;
					break;
				case HDMI_CD_24:
					uDeepColor = TRUE;
					break;
				default :
					uDeepColor = FALSE;
			}

			if (uDeepColor)
				return OK;
			else
				return ERROR;
		} // if
	} // for

	return ERROR;
}
