/**
 * File Name : cec.c
 *
 * File Description :
 * This file implements the hdmi v1.3 cec function in S5PV210 project
 *
 * Author : Hee Myung Noh
 * Dept : AP Development
 * Created Date : 2009/01/22
 * Version : 0.1
 * History 
 *  - Initiate version (Hee Myung Noh 090122)
 */

#include "option.h"
#include "system.h"
#include "Library.h"
#include "v210_sfr.h"
#include "intc.h"
#include "hdmi_reg.h"
#include "hdmi.h"
#include "gpio.h"
#include "cec.h"
#include "edid.h"


#define VERSION  								"1.0" 		/* Driver version number */
#define CEC_MINOR 								242 			/* Major 10, Minor 242, /dev/cec */

#define CEC_MESSAGE_BROADCAST_MASK    		0x0F
#define CEC_MESSAGE_BROADCAST         			0x0F
#define CEC_FILTER_THRESHOLD          				0x15

//#define	DISP_CEC_REG

#ifndef DISP_CEC_REG
#define CecOutp8(addr, data) 		Outp8(addr, data)
#define CecInp8(addr)  			Inp8(addr)
#else
#define CecOutp8(addr, data) 		{UART_Printf("Outp8(\'h%08x, \'h%08x);\n", addr, data), Outp8(addr, data);}
#define CecInp8(addr) 			{Inp8(addr); UART_Printf("Inp8(\'h%08x, d); // d=0x%08x\n", addr, Inp8(addr));}
#endif

static struct 
{
	CECDeviceType devtype;
	u8 laddr;
} laddresses[] = {   { CEC_DEVICE_RECODER, 	1  },
				    { CEC_DEVICE_RECODER, 	2  },
				    { CEC_DEVICE_TUNER,   		3  },
				    { CEC_DEVICE_PLAYER,  		4  },
				    { CEC_DEVICE_AUDIO,   		5  },
				    { CEC_DEVICE_TUNER,   		6  },
				    { CEC_DEVICE_TUNER,   		7  },
				    { CEC_DEVICE_PLAYER,  		8  },
				    { CEC_DEVICE_RECODER, 	9  },
				    { CEC_DEVICE_TUNER,   		10 },
				    { CEC_DEVICE_PLAYER,  		11 },
				};

oCec_Rx_struct oCec_Rx;
cec_state g_eCECRxStatus;
cec_state g_eCECTxStatus;

void __irq CEC_IRQ_Handler(void)
{
	u8 flag;
	u32 uStatus;

	uStatus = CecInp8(rCEC_TX_STATUS_0);
	uStatus |= CecInp8(rCEC_TX_STATUS_1) << 8;
	uStatus |= CecInp8(rCEC_RX_STATUS_0) << 16;
	uStatus |= CecInp8(rCEC_RX_STATUS_1) << 24;

	UART_Printf("CEC: status = 0x%04x\n", uStatus);

	if (uStatus & CEC_STATUS_TX_DONE) 
	{
		if (uStatus & CEC_STATUS_TX_ERROR) 
		{
			UART_Printf("CEC: CEC_STATUS_TX_ERROR!\n");
			CEC_SetTxState(STATE_ERROR);
		} 
		else 
		{
			UART_Printf("CEC: CEC_STATUS_TX_DONE!\n");
			CEC_SetTxState(STATE_DONE);
		}
		/* clear interrupt pending bit */
		CecOutp8(rCEC_INTR_CLEAR, CEC_IRQ_TX_DONE | CEC_IRQ_TX_ERROR);
	}

	if (uStatus & CEC_STATUS_RX_DONE) 
	{
		if (uStatus & CEC_STATUS_RX_ERROR) 
		{
			UART_Printf("CEC: CEC_STATUS_RX_ERROR!\n");
			CecOutp8(rCEC_RX_CTRL, CEC_RX_CTRL_RESET); // reset CEC Rx
			CEC_SetRxState(STATE_ERROR);
		} 
		else 
		{
			u32 i=0, uRxSize;

			UART_Printf("CEC: CEC_STATUS_RX_DONE!\n");

			/* copy data from internal buffer */
			uRxSize = uStatus >> 24;

			while (i < uRxSize) 
			{
				oCec_Rx.buffer[i] = CecInp8(rCEC_RX_BUFFER00+ (i*4));
				i++;
			}
			oCec_Rx.size = uRxSize;

			if(uRxSize != 1)
				UART_Printf("CEC Received Opcode : 0x%02x\n", oCec_Rx.buffer[1]); 

			CEC_SetRxState(STATE_DONE);

			CEC_EnableRx();
		}
		/* clear interrupt pending bit */
		CecOutp8(rCEC_INTR_CLEAR, CEC_IRQ_RX_DONE | CEC_IRQ_RX_ERROR);
	}
	
	INTC_ClearVectAddr();
}


u32 CEC_Init(void)
{
	UART_Printf("[CEC_Init]");

	//GPIO muxing selection
	GPIO_SetFunctionEach(eGPIO_H1, eGPIO_4, 4);
	GPIO_SetPullUpDownEach(eGPIO_H1, eGPIO_4, 0);

#if 0	// The CEC Interrupt in HDMI IP
	if ( HDMI_RegisterISR(CEC_ISR,HDMI_IRQ_CEC) != 0 )
	{
		UART_Printf("fail to register CEC Interrupt Service Routine\n\n");
		return ERROR;
	}
#else	// The CEC Interrupt to VIC directly
	INTC_SetVectAddr(NUM_CEC, CEC_IRQ_Handler);
	INTC_Enable(NUM_CEC);
#endif

	return OK;
}


/**
 * @brief CEC interrupt handler
 *
 * Handles interrupt requests from CEC hardware. \n
 * Action depends on current state of CEC hardware.
 */
void CEC_ISR(void)
{
	u8 flag;
	u32 uStatus;

	/* read flag register */
	flag = CecInp8(rHDMI_INTC_FLAG);

	if (!(flag & (1<<HDMI_IRQ_CEC))) 
	{	
		UART_Printf("CEC Interrupt Bit is not set!!\n", uStatus);
		return;
	}
	
	uStatus = CecInp8(rCEC_TX_STATUS_0);
	uStatus |= CecInp8(rCEC_TX_STATUS_1) << 8;
	uStatus |= CecInp8(rCEC_RX_STATUS_0) << 16;
	uStatus |= CecInp8(rCEC_RX_STATUS_1) << 24;

	UART_Printf("CEC: status = 0x%04x\n", uStatus);

	if (uStatus & CEC_STATUS_TX_DONE) 
	{
		if (uStatus & CEC_STATUS_TX_ERROR) 
		{
			UART_Printf("CEC: CEC_STATUS_TX_ERROR!\n");
			CEC_SetTxState(STATE_ERROR);
		} 
		else 
		{
			UART_Printf("CEC: CEC_STATUS_TX_DONE!\n");
			CEC_SetTxState(STATE_DONE);
		}
		/* clear interrupt pending bit */
		CecOutp8(rCEC_INTR_CLEAR, CEC_IRQ_TX_DONE | CEC_IRQ_TX_ERROR);
		///wake_up_interruptible(&cec_tx_struct.waitq);
	}

	if (uStatus & CEC_STATUS_RX_DONE) 
	{
		if (uStatus & CEC_STATUS_RX_ERROR) 
		{
			UART_Printf("CEC: CEC_STATUS_RX_ERROR!\n");
			CecOutp8(rCEC_RX_CTRL, CEC_RX_CTRL_RESET); // reset CEC Rx
			CEC_SetRxState(STATE_ERROR);
		} 
		else 
		{
			u32 i=0, uRxSize;

			UART_Printf("CEC: CEC_STATUS_RX_DONE!\n");

			/* copy data from internal buffer */
			uRxSize = uStatus >> 24;

			while (i < uRxSize) 
			{
				oCec_Rx.buffer[i] = CecInp8(rCEC_RX_BUFFER00+ (i*4));
				i++;
			}
			oCec_Rx.size = uRxSize;

			if(uRxSize != 1)
				UART_Printf("CEC Received Opcode : 0x%02x\n", oCec_Rx.buffer[1]); 

			CEC_SetRxState(STATE_DONE);

			CEC_EnableRx();
		}
		/* clear interrupt pending bit */
		CecOutp8(rCEC_INTR_CLEAR, CEC_IRQ_RX_DONE | CEC_IRQ_RX_ERROR);
	}
}


s32 CEC_Open(void)
{
	UART_Printf("[CEC_Open]\n");

	CecOutp8(rCEC_RX_CTRL, CEC_RX_CTRL_RESET); // reset CEC Rx
	CecOutp8(rCEC_TX_CTRL, CEC_TX_CTRL_RESET); // reset CEC Tx

	CEC_SetDivider();

	CecOutp8(rCEC_FILTER_TH, CEC_FILTER_THRESHOLD); // setup filter
	CecOutp8(rCEC_FILTER_CTRL, 0x0);
	
	//pending clear
	CecOutp8(rCEC_INTR_CLEAR, 	(1<<CEC_IRQ_TX_DONE) | (1<<CEC_IRQ_TX_ERROR)
								| (1<<CEC_IRQ_RX_DONE) | (1<<CEC_IRQ_RX_ERROR) );

//use Direct CEC Interrupt, not CEC in HDMI 
//	CEC_EnableInterrupts();

	CEC_UnmaskTxInterrupts();
	CEC_SetRxState(STATE_RX);
	CEC_UnmaskRxInterrupts();
	CEC_EnableRx();

	return OK;
}

s32 CEC_Close(void)
{
	UART_Printf("[CEC_Close]\n");

	CEC_MaskTxInterrupts();
	CEC_MaskRxInterrupts();
	CEC_DisableInterrupts();

	return OK;
}


s32 CEC_SendMessage(u8 *pBuffer, u32 uCount)
{
	unsigned char reg;
	int i = 0;

	/* check data size */
	if (uCount > CEC_TX_BUFF_SIZE || uCount == 0)
		return -1;

	/* copy packet to hardware buffer */
	while (i < uCount) 
	{
		CecOutp8(rCEC_TX_BUFFER00 + (i*4), pBuffer[i]);
		i++;
	}

	/* set number of bytes to transfer */
	CecOutp8(rCEC_TX_BYTE_NUM, uCount);

	CEC_SetTxState(STATE_TX);

	/* start transfer */
	reg = CecInp8(rCEC_TX_CTRL);
	reg |= CEC_TX_CTRL_START;

	/* if message is broadcast message - set corresponding bit */
	if ((pBuffer[0] & CEC_MESSAGE_BROADCAST_MASK) == CEC_MESSAGE_BROADCAST)
		reg |= CEC_TX_CTRL_BCAST;
	else
		reg &= ~CEC_TX_CTRL_BCAST;

	/* set number of retransmissions */
	reg |= (5<<4);

	CecOutp8(rCEC_TX_CTRL, reg);

	/* wait for interrupt */
	while(g_eCECTxStatus == STATE_TX);

	if(g_eCECTxStatus == STATE_ERROR)
		return -1;

	return uCount;
}



/**
 * Set CEC divider value.
 */
void CEC_SetDivider(void)
{
	u32 uDivRatio, uTemp;
	
	 /// (CEC_DIVISOR) * (clock cycle time) = 0.05ms \n
	uDivRatio  = FIN/160000-1;		//160KHz

	uTemp = Inp32(0xe010e804);
	uTemp = (uTemp & ~(0x3FF<<16)) | (uDivRatio<<16);
	Outp32(0xe010e804, uTemp);
	
	CecOutp8(rCEC_DIVISOR_3, 0x0);
	CecOutp8(rCEC_DIVISOR_2, 0x0);
	CecOutp8(rCEC_DIVISOR_1, 0x0);
	CecOutp8(rCEC_DIVISOR_0, 0x7);
}

/**
 * Enable CEC interrupts
 */
void CEC_EnableInterrupts(void)
{
	unsigned char reg;
	reg = CecInp8(rHDMI_INTC_CON);
	CecOutp8(rHDMI_INTC_CON, reg | (1<<HDMI_IRQ_CEC) | (1<<HDMI_IRQ_GLOBAL));
}

/**
 * Disable CEC interrupts
 */
void CEC_DisableInterrupts(void)
{
	u8 reg;
	reg = CecInp8(rHDMI_INTC_CON);
	CecOutp8(rHDMI_INTC_CON, reg & ~(1<<HDMI_IRQ_CEC) );
}

/**
 * Enable CEC Rx engine
 */
void CEC_EnableRx(void)
{
	unsigned char reg;
	reg = CecInp8(rCEC_RX_CTRL);
	reg |= CEC_RX_CTRL_ENABLE;
	CecOutp8(rCEC_RX_CTRL, reg);
}

/**
 * Mask CEC Rx interrupts
 */
void CEC_MaskRxInterrupts(void)
{
	unsigned char reg;
	reg = CecInp8(rCEC_INTR_MASK);
	reg |= CEC_IRQ_RX_DONE;
	reg |= CEC_IRQ_RX_ERROR;
	CecOutp8(rCEC_INTR_MASK, reg);
}

/**
 * Unmask CEC Rx interrupts
 */
void CEC_UnmaskRxInterrupts(void)
{
	unsigned char reg;
	reg = CecInp8(rCEC_INTR_MASK);
	reg &= ~CEC_IRQ_RX_DONE;
	reg &= ~CEC_IRQ_RX_ERROR;
	CecOutp8(rCEC_INTR_MASK, reg);
}

/**
 * Mask CEC Tx interrupts
 */
void CEC_MaskTxInterrupts(void)
{
	unsigned char reg;
	reg = CecInp8(rCEC_INTR_MASK);
	reg |= CEC_IRQ_TX_DONE;
	reg |= CEC_IRQ_TX_ERROR;
	CecOutp8(rCEC_INTR_MASK, reg);
}

/**
 * Unmask CEC Tx interrupts
 */
void CEC_UnmaskTxInterrupts(void)
{
	unsigned char reg;
	reg = CecInp8(rCEC_INTR_MASK);
	reg &= ~CEC_IRQ_TX_DONE;
	reg &= ~CEC_IRQ_TX_ERROR;
	CecOutp8(rCEC_INTR_MASK, reg);
}

/**
 * Change CEC Tx state to state
 * @param state [in] new CEC Tx state.
 */
void CEC_SetTxState(cec_state eState)
{
	g_eCECTxStatus = eState;
}

/**
 * Change CEC Rx state to @c state.
 * @param state [in] new CEC Rx state.
 */
void CEC_SetRxState(cec_state eState)
{
	g_eCECRxStatus = eState;
}


/**
 * Allocate logical address.
 *
 * @param paddr   [in] CEC device physical address.
 * @param devtype [in] CEC device type.
 *
 * @return new logical address, or 0 if an arror occured.
 */
u8 CEC_AllocLogicalAddress(u32 paddr, CECDeviceType devtype)
{
	u8 laddr = CEC_LADDR_UNREGISTERED;
	s32 i = 0;
	u8 buffer[5];

	if (CEC_SetLogicalAddr(laddr) < 0) 
	{
		UART_Printf("CECSetLogicalAddr() failed!\n");
		return 0;
	}

	if (paddr == CEC_NOT_VALID_PHYSICAL_ADDRESS) 
	{
		return CEC_LADDR_UNREGISTERED;
	}

	/* send "Polling Message" */
	while (i < sizeof(laddresses)/sizeof(laddresses[0])) 
	{
		if (laddresses[i].devtype == devtype) 
		{
			u8 _laddr = laddresses[i].laddr;
			u8 message = ((_laddr << 4) | _laddr);
			
			if (CEC_SendMessage(&message, 1) != 1) 
			{
				laddr = _laddr;
				break;
			}
		}
		i++;
	}

	if (laddr == CEC_LADDR_UNREGISTERED) 
	{
		UART_Printf("All LA addresses in use!!!\n");
		return CEC_LADDR_UNREGISTERED;
	}

	if (CEC_SetLogicalAddr(laddr) < 0) 
	{
		UART_Printf("CECSetLogicalAddr() failed!\n");
		return 0;
	}

	UART_Printf("Logical address = %d\n", laddr);

	/* broadcast "Report Physical Address" */
	buffer[0] = (laddr << 4) | CEC_MSG_BROADCAST;
	buffer[1] = CEC_OPCODE_REPORT_PHYSICAL_ADDRESS;
	buffer[2] = (paddr >> 8) & 0xFF;
	buffer[3] = paddr & 0xFF;
	buffer[4] = devtype;

	if (CEC_SendMessage(buffer, 5) != 5) 
	{
		UART_Printf("CECSendMessage() failed!\n");
		return 0;
	}

	return laddr;
}


/**
 * Set CEC logical address.
 *
 * @return 1 if success, otherwise, return 0.
 */
s32 CEC_SetLogicalAddr(u32 laddr)
{
	CecOutp8(rCEC_LOGIC_ADDR, laddr&0x0F);

	return 1;
}


/**
 * Print CEC frame.
 */
void CEC_PrintFrame(u8 *buffer, u32 size)
{
	if (size > 0) 
	{
		s32 i;
		
		UART_Printf("fsize: %d ", size);
		UART_Printf("frame: ");
		
		for (i = 0; i < size; i++) 
		{
			UART_Printf("0x%02x ", buffer[i]);
		}
		UART_Printf("\n");
	}
}


/**
 * Check CEC message.
 *
 * @param opcode   [in] pointer to buffer address where message will be stored.
 * @param lsrc     [in] buffer size.
 *
 * @return 1 if message should be ignored, otherwise, return 0.
 */
//TODO: not finished
s32 CEC_IgnoreMessage(u8 opcode, u8 lsrc)
{
	s32 retval = 0;

	/* if a message coming from address 15 (unregistered) */
	if (lsrc == CEC_LADDR_UNREGISTERED) 
	{
		switch (opcode) 
		{
			case CEC_OPCODE_DECK_CONTROL:
			case CEC_OPCODE_PLAY:
				retval = 1;
				default:
			break;
		}
	}

	return retval;
}

/**
 * Check CEC message.
 *
 * @param opcode   [in] pointer to buffer address where message will be stored.
 * @param size     [in] message size.
 *
 * @return 0 if message should be ignored, otherwise, return 1.
 */
//TODO: not finished
s32 CEC_CheckMessageSize(u8 opcode, s32 size)
{
	s32 retval = 1;

	switch (opcode) 
	{
		case CEC_OPCODE_PLAY:
		case CEC_OPCODE_DECK_CONTROL:
			if (size != 3) 
				retval = 0;
			break;			
		case CEC_OPCODE_SET_MENU_LANGUAGE:
			if (size != 5) 
				retval = 0;
			break;
		case CEC_OPCODE_FEATURE_ABORT:
			if (size != 4) 
				retval = 0;
			break;
		default:
			break;
	}

	return retval;
}

/**
 * Check CEC message.
 *
 * @param opcode    [in] pointer to buffer address where message will be stored.
 * @param broadcast [in] broadcast/direct message.
 *
 * @return 0 if message should be ignored, otherwise, return 1.
 */
//TODO: not finished
s32 CEC_CheckMessageMode(u8 opcode, s32 broadcast)
{
	s32 retval = 1;

	switch (opcode) 
	{
		case CEC_OPCODE_REQUEST_ACTIVE_SOURCE:
		case CEC_OPCODE_SET_MENU_LANGUAGE:
			if (!broadcast) 
				retval = 0;
			break;
			
		case CEC_OPCODE_GIVE_PHYSICAL_ADDRESS:
		case CEC_OPCODE_DECK_CONTROL:
		case CEC_OPCODE_PLAY:
		case CEC_OPCODE_FEATURE_ABORT:
		case CEC_OPCODE_ABORT:
			if (broadcast) 
				retval = 0;
			break;
		default:
			break;
	}

	return retval;
}

