/**************************************************************************************
* 
*	Project Name : S5PV210 validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File name : mie.c
*
*	File Description : This file implements the API functons for MIE
*
*	Author : Woojin,Kim
*	Dept : AP application
*	Created Date : 2009/02/09
*	Version	: v0.1
*
*	History 
*		- Created(Woojin.Kim 2009/02/09)
*
******************************************************************************/

#include "system.h"
#include "library.h"
#include "util.h"
#include "Lcd.h"
#include "mie.h"
#include "v210_sfr.h"


#if DBG_MIE
#define MieOutp32(offset, x) {Disp("Outp1w(32'h%x, 32'h%x);\n", (MIE_BASE+offset), (x)); Outp32(MIE_BASE+offset, x);}
#define MieInp32(offset, x) {x=Inp32(MIE_BASE+offset); /*Disp("Inp1w(32'h%x, 32'h%x);\n", (LCDC_BASE+offset), (x));*/}
#else
#define MieOutp32(offset, x) Outp32(MIE_BASE+offset, x)
#define MieInp32(offset, x) {x=Inp32(MIE_BASE+offset);}
#endif

enum MIE_SFR
{
// Algorithm Control Registers
	rMIE_CTRL1			= 0x0000,
	rMIE_CTRL2			= 0x0004,
	rMIE_CTRL3			= 0x0008,	//Hidden Spec
	rMIE_STATUS		= 0x000C,	//Hidden Spec
	rMIE_WINHADDR		= 0x0010,
	rMIE_WINVADDR		= 0x0014,
	rMIE_TESTCTRL		= 0x0020,	//Hidden Spec
	rMIE_TESTDI			= 0x0024,	//Hidden Spec
	rMIE_TESTDO		= 0x0028,	//Hidden Sepc
	rMIE_PWMCTRL1		= 0x0030,
	rMIE_PWMCTRL2		= 0x0034,
	rMIE_PWMVIDTCON1	= 0x0038,
	rMIE_PWMVIDTCON2	= 0x003C,
	rMIE_PWMSTATUS1	= 0x0040,
	rMIE_PWMSTATUS2	= 0x0044
};

//MIE_CTRL1
#define HRESOL(n)			(((n)&0x7FF)<<18)
#define VRESOL(n)			(((n)&0x7FF)<<7)
#define MIEOFF				(0<<5)
#define UI_MODE				(1<<5)
#define STILL_MODE			(2<<5)
#define MOVING_MODE		(3<<5)
#define DIMMING_DIS			(0<<4)
#define DIMMING_EN			(1<<4)
#define CAT1					(0<<2)
#define CAT2					(1<<2)
#define CAT4					(2<<2)
#define CAT8					(3<<2)
#define CST32				(0<<0)
#define CST64				(1<<0)
#define CST96				(2<<0)
#define CST128				(3<<0)

//MIE_CTRL2
#define IERC(n)				(((n)&0xFF)<<14)
#define BRRC(n)				(((n)&0xFF)<<6)
#define SERC(n)				(((n)&0x1F)<<1)
#define ORDER_RGB			(0<<0)
#define ORDER_BGR			(1<<0)

//MIE_CTRL3                          
#define DUAL_VCLK			(0<<5)
#define DUAL_VDEN			(1<<5)
#define IVCLK_FALL_EDGE		(0<<4)
#define IVCLK_RISE_EDGE		(1<<4)
#define IHSYNC_NORMAL		(0<<3)
#define IHSYNC_INVERT		(1<<3)
#define IVSYNC_NORMAL		(0<<2)
#define IVSYNC_INVERT		(1<<2)
#define IVDEN_NORMAL		(0<<1)
#define IVDEN_INVERT		(1<<1)
#define MIE_NORMAL			(0<<0)
#define MIE_FIXED			(1<<0)

//MIE_WINHADDR
#define WINHADDR1(n)		(((n)&0x7FF)<<21)
#define WINHADDR0(n)		(((n)&0x7FF)<<0)

//MIE_WINVADDR
#define WINVADDR1(n)		(((n)&0x7FF)<<21)
#define WINVADDR0(n)		(((n)&0x7FF)<<0)


//MIE_TESTCTRL				(HIDDEN SPEC)
#define TESTMODE_OFF		(0<<0)
#define WRITE_CDF_HIST		(2<<0)
#define READ_CDF_HIST		(3<<0)
#define WRITE_TMF_PRE		(4<<0)
#define READ_TMF_PRE		(5<<0)
#define WRITE_TMF_LPF		(6<<0)
#define READ_TMF_LPF		(7<<0)
#define WRITE_TMF_CNT		(8<<0)
#define READ_TMF_CNT		(9<<0)
#define CONT_SIG_GEN		(A<<0)
#define NO_OPERATION		(F<<0)

//MIE_TESTDI				(HIDDEN SPEC)
#define TESTDI(n)				(((n)&0xFFFFF)<<0)

//MIE_TESTDO				(HIDDEN SPEC)
#define TESTDO(n)			(((n)&0xFFFFF)<<0)

//MIE_PWMCTRL1
#define DT_16FRM			(0<<16)
#define DT_24FRM			(1<<16)
#define DT_32FRM			(2<<16)
#define DT_40FRM			(3<<16)
#define DT_48FRM			(4<<16)
#define DT_56FRM			(5<<16)
#define DT_65FRM			(6<<16)
#define DT_72FRM			(7<<16)
#define BLMODEINSLP_OFF	(1<<15)
#define BLMODEINSLP_ON		(0<<15)
#define BRSEL_ON			(1<<14)
#define BRSEL_OFF			(0<<14)
#define BLDIMMSTEP_256		(0<<12)
#define BLDIMMSTEP_128		(1<<12)
#define BLDIMMSTEP_64		(2<<12)
#define BLDIMMSTEP_32		(3<<12)
#define BCFREQSEL(n)		(((n)&0x7F)<<5)
#define BLU_OFF				(0<<4)
#define BLU_ON				(1<<4)
#define DD_OFF				(0<<3)
#define DD_ON				(1<<3)
#define BL_ZERO				(0<<2)
#define BL_ACT				(1<<2)
#define BCMODE_OFF			(0<<0)
#define BCMODE_MANUAL		(1<<0)
#define BCMODE_MIE			(2<<0)
#define BCMODE_MANUALMIE	(3<<0)

//MIE_PWMCTRL2
#define MANBR(n)				(((n)&0xFF)<<8)
#define BRMIN(n)				(((n)&0xFF)<<0)

//MIE_PWMVIDTCON1
#define MIEVBPD(n)					(((n)&0xFF)<<16)
#define MIEVFPD(n)					(((n)&0xFF)<<8)
#define MIEVSPW(n)					(((n)&0xFF)<<0)

//MIE_PWMVIDTCON2
#define MIEHBPD(n)					(((n)&0xFF)<<16)
#define MIEHFPD(n)					(((n)&0xFF)<<8)
#define MIEHSPW(n)					(((n)&0xFF)<<0)

//MIE_PWMSTATUS1
#define FRAMECLKCNT(n)			(((n)&0x3FFFFF)<<0)



//-------------------------------------------------------------------------
// Function Name : MIE_InitGPIOpwm
// Function Description : This function set LCD BackLight as a PWM_MIE signal
// Input : ePWMen ; MIE_PWMon (MIE PWM control on)  MIE_PWMoff (MIE PWM control off)
// Output : none
// Version : v0.0
//-------------------------------------------------------------------------
void MIE_InitGPIOpwm(MIE_PWMENABLE ePWMen)
{
	if(ePWMen == MIE_PWMon)
	{
	GPIO_SetFunctionEach(eGPIO_D0,eGPIO_3, 0x3);	//set GPD0's TOUT3 as PWM_MIE signal
	GPIO_SetPullUpDownAll(eGPIO_D0,0x0);			//set pull-up,down disable
	}
	else
	{
	GPIO_SetFunctionEach(eGPIO_D0,eGPIO_3, 0x1);	//set GPD0's TOUT3 as PWM_MIE signal
	GPIO_SetPullUpDownAll(eGPIO_D0,0x0);			//set pull-up,down disable		
	}
}




//-------------------------------------------------------------------------
// Function Name : MIE_SetOutputSignal
// Function Description : This function select mDNIe's output signal polarity to LCD module
// Input : none
// Output : none
// Version : v0.0
//-------------------------------------------------------------------------
void MIE_SetOutputSignal(void)
{

#if (LCD_MODEL == LCD_LTV350QV_RGB)
	MieOutp32(rMIE_CTRL3, IHSYNC_INVERT | IVSYNC_INVERT); 	

#elif ( LCD_MODEL == LCD_LTS350Q1 )
	MieOutp32(rMIE_CTRL3, IVCLK_FALL_EDGE | IHSYNC_NORMAL | IVSYNC_NORMAL | IVDEN_NORMAL);
	
#elif ( LCD_MODEL == LCD_LTS222Q_CPU || LCD_MODEL == LCD_S6D0139_CPU )
	
#elif ( LCD_MODEL == LCD_LTS222Q_RGB )
	MieOutp32(rMIE_CTRL3, IVCLK_FALL_EDGE | IHSYNC_INVERT | IVSYNC_INVERT | IVDEN_NORMAL);
	
#elif ( LCD_MODEL == LCD_LTS222Q_SRGB)
	MieOutp32(rMIE_CTRL3, IVCLK_FALL_EDGE | IHSYNC_INVERT | IVSYNC_INVERT | IVDEN_NORMAL);

#elif (LCD_MODEL == LCD_LTE480WV_RGB)
	MieOutp32(rMIE_CTRL3,IHSYNC_INVERT | IVSYNC_INVERT | IVDEN_NORMAL);

#elif (LCD_MODEL == LCD_TL2796)
	MieOutp32(rMIE_CTRL3,IVCLK_RISE_EDGE | IHSYNC_INVERT | IVSYNC_INVERT | IVDEN_INVERT);

#else
	Assert(0);
#endif

}


//-------------------------------------------------------------------------
// Function Name : MIE_SelectMode
// Function Description : This function select MIE mode
// Input : eModeSel ; MIE_off, UI(User Interface mode), STILL(Still Image mode), MOVING(Moving Image mode)
//           eCtDimm ; CT_DIMM_DIS (abrupt transition when MIE on/off) , CT_DIMM_EN (smooth transition when MIE on/off)
// Output : none
// Version : v0.0
//-------------------------------------------------------------------------
void MIE_SelectMode(MODE_SEL eModeSel, CTRL_DIMMING eCtDimm)
{
	u32 uTmpReg;
	u32 uTmpVal = 0x0;
	u32 uSfr;


	uTmpVal = uTmpVal |
	(eModeSel == MIE_off) ? MIEOFF :
	(eModeSel == UI) ? UI_MODE :
	(eModeSel == STILL) ? STILL_MODE :
	(eModeSel == MOVING) ? MOVING_MODE : Assert(0);	

									
	uTmpVal = uTmpVal |
	((eCtDimm == CT_DIMM_DIS) ? DIMMING_DIS :
	(eCtDimm == CT_DIMM_EN) ? DIMMING_EN : Assert(0));
	
	MieInp32(rMIE_CTRL1, uTmpReg);

	uTmpReg = (uTmpReg & ~((0x7)<<4)) | uTmpVal ;

	MieOutp32(rMIE_CTRL1,uTmpReg);


}

//---------------------------------------------------------
// Function Name : MIE_SetSize
// Function Description : This function set Output Image Size
// Input : eHSize ; output Horizental Size       eVSize ; output Vertical Size
// Output : none
// Version : v0.0
//---------------------------------------------------------
void MIE_SetSize(u32 uHSize, u32 uVSize)
{
	u32 uTmpReg;

	if (uHSize > 1024 || uVSize > 1024)		// v210 target (1024 x 1024)
		Assert(0);
	
	MieInp32(rMIE_CTRL1, uTmpReg);
	uTmpReg = uTmpReg & ~((0x3fffff)<<7) | HRESOL(uHSize) | VRESOL(uVSize);
	
	MieOutp32(rMIE_CTRL1, uTmpReg);
}

//---------------------------------------------------------
// Function Name : MIE_SetCatCst
// Function Description : This function set CAT and CST of MIE
//                                CAT ; Control signal of Abrupt frame Transition  
//                                CST ; Control signal of Smooth frame Transition
// Input : eCAT ; CAT_1,CAT_2,CAT_4,CAT_8       eCST ; CST_32, CST_64, CST_96, CST_128
// Output : none
// Version : v0.0
//---------------------------------------------------------
void MIE_SetCatCst(CAT_SEL eCAT,CST_SEL eCST)
{
	u32 uTmpReg;
	u32 uTmpVal = 0x0;

	uTmpVal = uTmpVal |
	(eCAT == CAT_1) ? CAT1 :
	(eCAT == CAT_2) ? CAT2 :
	(eCAT == CAT_4) ? CAT4 :
	(eCAT == CAT_8) ? CAT8 : Assert(0);		
									
	uTmpVal = uTmpVal |
	(eCST == CST_32) ? CST32 :
	(eCST == CST_64) ? CST64 :
	(eCST == CST_96) ? CST96 :
	(eCST == CST_128) ? CST128 : Assert(0);		

	
	MieInp32(rMIE_CTRL1,uTmpReg);
	uTmpReg = uTmpReg & ~(0xF) | uTmpVal;
	MieOutp32(rMIE_CTRL1,uTmpReg);		

}

//-------------------------------------------------------------------
// Function Name : MIE_SetIERC
// Function Description : This function set Image Enhancement ratio control signal
// Input : uIEratio ; 0~255
// Output : none
// Version : v0.0
//-------------------------------------------------------------------
void MIE_SetIERC(u32 uIEratio)
{
	u32 uTmpReg;

	if (uIEratio > 255)
		Assert(0);
	
	MieInp32(rMIE_CTRL2,uTmpReg);
	uTmpReg = uTmpReg & ~((0xFF)<<14) | IERC(uIEratio);
	MieOutp32(rMIE_CTRL2,uTmpReg);		

}

//-------------------------------------------------------------------
// Function Name : MIE_SetBRRC
// Function Description : This function set Backlight Reduction ratio control signal
// Input : uBRratio ; 0~255
// Output : none
// Version : v0.0
//-------------------------------------------------------------------
void MIE_SetBRRC(u32 uBRratio)
{
	u32 uTmpReg;

	if (uBRratio > 255)
		Assert(0);


	MieInp32(rMIE_CTRL2,uTmpReg);
	uTmpReg = uTmpReg & ~((0xFF)<<6) | BRRC(uBRratio);
	MieOutp32(rMIE_CTRL2,uTmpReg);		

}

//--------------------------------------------------------------------
// Function Name : MIE_SetSERC
// Function Description : This function set Saturation Enhance ratio control signal
// Input : uSEratio ; 0~31
// Output : none
// Version : v0.0
//--------------------------------------------------------------------
void MIE_SetSERC(u32 uSEratio)
{
	u32 uTmpReg;

	if (uSEratio > 31)
		Assert(0);

	MieInp32(rMIE_CTRL2,uTmpReg);
	uTmpReg = uTmpReg & ~((0x1F)<<1) | SERC(uSEratio);
	MieOutp32(rMIE_CTRL2,uTmpReg);		

}

//---------------------------------------------------------
// Function Name : MDNIE_SetWinPosition
// Function Description : This function set Horizontal and vertical position
// Input : uHSize ; Horizontal size       uVSize ; Vertical size
// Output : none
// Version : v0.0
//---------------------------------------------------------
void MIE_SetWinPosition(u32 uHpSize,u32 uVpSize)
{
	u32 uTmpRegH = 0x0;
	u32 uTmpRegV = 0x0;

	MieOutp32(rMIE_WINHADDR, uTmpRegH | WINHADDR1(uHpSize) | WINHADDR0(0));

	MieOutp32(rMIE_WINVADDR, uTmpRegV | WINVADDR1(uVpSize) | WINVADDR0(0));
	
}

//--------------------------------------------------------------------------
// Function Name : MIE_SetPWMPorch
// Function Description : This function set PWM porch by defined LCD module type in LCD.h
// Input : none
// Output : none
// Version : v0.0
//--------------------------------------------------------------------------
void MIE_SetPWMPorch(void)
{
	u32 uTmpRegH = 0x0;
	u32 uTmpRegV = 0x0;

#if (LCD_MODEL == LCD_LTV350QV_RGB)
	MieOutp32(rMIE_PWMVIDTCON1, uTmpRegH | MIEVBPD(VBPD_LTV350QV) | MIEVFPD(VFPD_LTV350QV) | MIEVSPW(VSPW_LTV350QV));
	MieOutp32(rMIE_PWMVIDTCON2, uTmpRegV | MIEHBPD(HBPD_LTV350QV) | MIEHFPD(HFPD_LTV350QV) | MIEHSPW(HSPW_LTV350QV));


#elif ( LCD_MODEL == LCD_LTS350Q1 )
	MieOutp32(rMIE_PWMVIDTCON1, uTmpRegH | MIEVBPD(VBPD_240320) | MIEVFPD(VFPD_240320) | MIEVSPW(VSPW_240320));
	MieOutp32(rMIE_PWMVIDTCON2, uTmpRegV | MIEHBPD(HBPD_240320) | MIEHFPD(HFPD_240320) | MIEHSPW(HSPW_240320));

#elif ( LCD_MODEL == LCD_LTS222Q_RGB || LCD_LTS222Q_SRGB || LCD_LTS222Q_CPU || LCD_S6D0139_CPU || LCD_S6D66A0_RGB )
	MieOutp32(rMIE_PWMVIDTCON1, uTmpRegH | MIEVBPD(VBPD_LTS222Q) | MIEVFPD(VFPD_LTS222Q) | MIEVSPW(VSPW_LTS222Q));
	MieOutp32(rMIE_PWMVIDTCON2, uTmpRegV | MIEHBPD(HBPD_LTS222Q) | MIEHFPD(HFPD_LTS222Q) | MIEHSPW(HSPW_LTS222Q));

#elif (LCD_MODEL == LCD_LTE480WV_RGB)
	MieOutp32(rMIE_PWMVIDTCON1, uTmpRegH | MIEVBPD(6) | MIEVFPD(4) | MIEVSPW(0));
	MieOutp32(rMIE_PWMVIDTCON2, uTmpRegV | MIEHBPD(12) | MIEHFPD(7) | MIEHSPW(2));

#elif (LCD_MODEL == LCD_TL2796)
	MieOutp32(rMIE_PWMVIDTCON1, uTmpRegH | MIEVBPD(VBPD_TL2796) | MIEVFPD(VFPD_TL2796) | MIEVSPW(VSPW_TL2796));
	MieOutp32(rMIE_PWMVIDTCON2, uTmpRegV | MIEHBPD(HBPD_TL2796) | MIEHFPD(HFPD_TL2796) | MIEHSPW(HSPW_TL2796));

#endif

}

//--------------------------------------------------------------------------
// Function Name : MIE_SetPWMcontrol
// Function Description : This function set PWM control functions
// Input : eBluCon ; BCTRL_OFF/ON      eDdEn ; DDoff/on      eBlEn ; BLoff/on       
//           eBcMode ; MODEOFF/MIEONLY/PHOTOSENSOR/MIEPHOTO
// Output : none
// Version : v0.0
//--------------------------------------------------------------------------
void MIE_SetPWMcontrol(BLU_CONTROL eBluCon,DD_ENABLE eDdEn,BL_ENABLE eBlEn,BCMODE_SEL eBcMode)
{

	u32 uBluConSel = 0x0;
	u32	uDdEnSel = 0x0 ; 
	u32	uBlEnSel = 0x0 ;
	u32	uBcModeSel = 0x0 ;
	u32 uTmpReg = 0x0 ;

	uBluConSel = (eBluCon == BCTRL_OFF ) ? BLU_OFF : BLU_ON ;
	uDdEnSel	= (eDdEn == DDoff ) ? DD_OFF : DD_ON ;
	uBlEnSel = (eBlEn == BLoff ) ? BL_ZERO : BL_ACT ;
	
	uBcModeSel = (eBcMode == MODEOFF ) ? BCMODE_OFF :
				(eBcMode == MANUALONLY ) ? BCMODE_MANUAL :
		             (eBcMode == MIEONLY ) ? BCMODE_MIE :
				(eBcMode == MANUALMIE ) ? BCMODE_MANUALMIE :
										Assert(0);
										
	MieInp32(rMIE_PWMCTRL1, uTmpReg );
	uTmpReg = (uTmpReg & ~(0x1F)) | uBluConSel | uDdEnSel | uBlEnSel | uBcModeSel ;
	MieOutp32(rMIE_PWMCTRL1, uTmpReg);

}


//--------------------------------------------------------------------------
// Function Name : MIE_SetPWMdefault
// Function Description : This function set PWM control functions
// Input : ePwmDm ; PWMDM_256,128,64,32      uPwmFq ; PWM Frequency      uPwmMin ; Minimum PWM width
// Output : none
// Version : v0.0
//--------------------------------------------------------------------------
void MIE_SetPWMdefault(PWM_DM_SEL ePwmDm,u32 uPwmFq,u32 uPwmMin)
{

	u32 uPwmDm = 0x0 ;
	u32 uTmpReg = 0x0 ;


	uPwmDm =
	(ePwmDm == PWMDM_256 ) ? BLDIMMSTEP_256 :
	(ePwmDm == PWMDM_128 ) ? BLDIMMSTEP_128 :
	(ePwmDm == PWMDM_64 ) ? BLDIMMSTEP_64 :
	(ePwmDm == PWMDM_32 ) ? BLDIMMSTEP_32 :	Assert(0);
										
	MieInp32(rMIE_PWMCTRL1, uTmpReg );
	uTmpReg = (uTmpReg & ~(0x3E0)) | uPwmDm | BCFREQSEL(uPwmFq);
	MieOutp32(rMIE_PWMCTRL1, uTmpReg);

	MieInp32(rMIE_PWMCTRL2, uTmpReg );
	uTmpReg = (uTmpReg & ~(0xff)) | BRMIN(uPwmMin);
	MieOutp32(rMIE_PWMCTRL2, uTmpReg);

	

}


//--------------------------------------------------------------------------
// Function Name : MIE_SetPWMManual
// Function Description : This function set PWM Manual mode control values
// Input : eDdEn ; DDon /off      eDtSel ; DT_16 ~ 72       
//           eBrEn ; BRon / off	uManBr ; Brightness value as manual setting
// Output : none
// Version : v0.0
//--------------------------------------------------------------------------
void MIE_SetPWMManual(DD_ENABLE eDdEn, DT_SEL eDtSel, BR_ENABLE eBrEn, u32 uManBr)
{

	u32 uDT = 0x0;
	u32 uBrSel = 0x0;
	u32	uDdEnSel = 0x0 ; 
	u32 uTmpReg = 0x0 ;

	uDdEnSel	= (eDdEn == DDon ) ? DD_ON : DD_OFF ;
	
	uDT =
	(eDtSel == DT_16 ) ? DT_16FRM :
	(eDtSel == DT_24 ) ? DT_24FRM :
	(eDtSel == DT_32 ) ? DT_32FRM :
	(eDtSel == DT_40 ) ? DT_40FRM :
	(eDtSel == DT_48 ) ? DT_48FRM :
	(eDtSel == DT_56 ) ? DT_56FRM :
	(eDtSel == DT_65 ) ? DT_65FRM :	
	(eDtSel == DT_72 ) ? DT_72FRM :	Assert(0);

	uBrSel =	(eBrEn == BRon ) ? BRSEL_ON : BRSEL_OFF ;

	MieInp32(rMIE_PWMCTRL1, uTmpReg );
	uTmpReg = (uTmpReg & ~(0x74008)) | uDT | uBrSel | uDdEnSel ;
	MieOutp32(rMIE_PWMCTRL1, uTmpReg);

	MieInp32(rMIE_PWMCTRL2, uTmpReg);
	uTmpReg = (uTmpReg & ~(0xff00)) | MANBR(uManBr);
	MieOutp32(rMIE_PWMCTRL2, uTmpReg);

}


//--------------------------------------------------------------------------
// Function Name : MIE_GetTMF
// Function Description : This function get current and previous TMF (Hidden spec)
// Input : none 
// Output : uTmpReg ; return MIE_STATUS value
// Version : v0.0
//--------------------------------------------------------------------------
u32 MIE_GetTMF(void)
{
	u32 uTmpReg;

	MieInp32(rMIE_STATUS, uTmpReg);

	return uTmpReg;

}

