/*----------------------------------------------------------------------
 *
 * Filename: mfc.c
 *
 * Contents:  Implemention of "MFCv5"
 *
 * Authors:
 *
 * History:
 *
 * Notes:
 *
 * Copyright (c) 2008 SAMSUNG Electronics.
 *
 *----------------------------------------------------------------------
 */
#include "system.h"
#include <string.h>
#include <stdlib.h>
#include "mfc.h"
#include "v210_sfr.h"

//#define DISP_MFC
//#define DBG_MFC

#ifdef DISP_MFC
	#define mfcDbg(x) Dbg x
#else
	#define mfcDbg(x) 0;
#endif

#ifdef DBG_MFC
#define mfcOutp32(offset, data) {Disp("Outp1w(32'h%x, 32'h%x);\n", (MFC_BASE+offset), (data)); (*(volatile u32 *)(MFC_BASE+offset) = (data)); } 
#define mfcInp32(offset, data) {data=Inp32(MFC_BASE+offset);  Disp("Inp1w(32'h%x, 32'h%x);\n", (MFC_BASE+offset), data); }
#else
#define mfcOutp32(offset, data) Outp32(MFC_BASE+offset, data)
#define mfcInp32(offset, data) {data=Inp32(MFC_BASE+offset);}
#endif



enum MFC_REG
{
	// MFC control reg
	MFC_SW_RESET					= 0x0000, 
	MFC_RISC_HOST_INT				= 0x0008,
 	MFC_HOST2RISC_CMD				= 0x0030,
 	MFC_HOST2RISC_ARG1				= 0x0034,
 	MFC_HOST2RISC_ARG2				= 0x0038,
 	MFC_HOST2RISC_ARG3				= 0x003c,	
  	MFC_HOST2RISC_ARG4				= 0x0040,	
 	MFC_RISC2HOST_CMD				= 0x0044,
 	MFC_RISC2HOST_ARG1				= 0x0048,
 	MFC_RISC2HOST_ARG2				= 0x004c,
	MFC_RISC2HOST_ARG3				= 0x0050,
 	MFC_RISC2HOST_ARG4				= 0x0054,
 	MFC_FW_VERSION					= 0x0058,
 	MFC_DBF_INFO_OUTPUT2			= 0x0068, 	
 	MFC_FW_STATUS					= 0x0080,
 	MFC_MC_DRAMBASE_ADDR_A			= 0x0508,
 	MFC_MC_DRAMBASE_ADDR_B			= 0x050c,
 	MFC_MC_STATUS					= 0x0510,

	MFC_COMMON_BASE_ADDR			= 0x0600,

 	MFC_HSIZE_PX 					= 0x0818,
 	MFC_VSIZE_PX 					= 0x081c,
 	MFC_PROFILE	 					= 0x0830,
 	MFC_PICTURE_STRUCT 				= 0x083c,
 	MFC_LF_CONTROL					= 0x0848,
 	MFC_LF_ALPHA_OFF				= 0x084c,
 	MFC_LF_BETA_OFF					= 0x0850,
 
	MFC_QP_OFFSET					= 0x0c30,
	MFC_QP_OUT_EN					= 0x0c34,

 	MFC_SI_RTN_CHID					= 0x2000,
 	MFC_SI_VRESOL					= 0x2004,	
 	MFC_SI_HRESOL					= 0x2008,	
 	MFC_SI_MIN_NUB_DPB				= 0x200c,	
 	MFC_SI_DISPLAY_Y_ADR 			= 0x2010, 
 	MFC_SI_DISPLAY_C_ADR 			= 0x2014,
 	MFC_SI_DEC_FRM_SIZE				= 0x2018,
 	MFC_SI_DISPLAY_STATUS			= 0x201c,
	MFC_SI_FRAME_TYPE				= 0x2020,
	MFC_SI_DECODE_Y_ADR				= 0x2024,	
	MFC_SI_DECODE_C_ADR				= 0x2028,
	MFC_SI_DECODE_STATUS			= 0x202c,
	
	MFC_SI_CH0_INST_ID				= 0x2040,	
 	MFC_SI_CH0_ES_ADDR				= 0x2044,	
 	MFC_SI_CH0_ES_DEC_UNIT_SIZE		= 0x2048,	
 	MFC_SI_CH0_DESC_ADDR			= 0x204c,
 	MFC_SI_CH0_CPB_SIZE				= 0x2058,
	MFC_SI_CH0_DIVX311_VRESOL		= 0x2050,
	MFC_SI_CH0_DIVX311_HRESOL		= 0x2054,
 	MFC_SI_CH0_DESC_SIZE			= 0x205c,

	MFC_SI_CH1_INST_ID				= 0x2080,
	MFC_SI_CH1_ES_ADDR				= 0x2084,
	MFC_SI_CH1_ES_DEC_UNIT_SIZE		= 0x2088,
	MFC_SI_CH1_DESC_ADDR			= 0x208c,
	MFC_SI_CH1_DIVX311_VRESOL		= 0x2090,
	MFC_SI_CH1_DIVX311_HRESOL		= 0x2094,
	MFC_SI_CH1_CPB_SIZE				= 0x2098,
	MFC_SI_CH1_DESC_SIZE			= 0x209c,	

	// Encoder
	MFC_SI_STREAM_SIZE				= 0x2004,
	MFC_SI_PICTURE_CNT				= 0x2008,
	MFC_SI_WRITE_PTR				= 0x200c,
	MFC_SI_SLICE_TYPE				= 0x2010,

	MFC_SI_ENCODED_Y_ADDR			= 0x2014,
	MFC_SI_ENCODED_C_ADDR			= 0x2018,

 	MFC_SI_CH0_SB_ADDR				= 0x2044,	
 	MFC_SI_CH0_SB_SIZE				= 0x204c,
 	MFC_SI_CH0_CURRENT_Y_ADDR		= 0x2050,
 	MFC_SI_CH0_CURRENT_C_ADDR		= 0x2054,

 	MFC_SI_CH1_SB_ADDR				= 0x2084,	
 	MFC_SI_CH1_SB_SIZE				= 0x208c,
 	MFC_SI_CH1_CURRENT_Y_ADDR		= 0x2090,
 	MFC_SI_CH1_CURRENT_C_ADDR		= 0x2094,

	MFC_SI_CH0_FRAME_INSERTION		= 0x2058,
	MFC_SI_CH1_FRAME_INSERTION		= 0x2098,
	
	// Common
	MFC_SI_CH0_RELEASE_BUFFER		= 0x2060,
	MFC_SI_CH0_HOST_WR_ADR			= 0x2064,
	MFC_SI_CH0_DPB_CONFIG_CTRL		= 0x2068,
	MFC_SI_CH0_ALLOCATED_DPD_SZ		= 0x206c,
	
	MFC_SI_CH1_RELEASE_BUFFER		= 0x20A0,
	MFC_SI_CH1_HOST_WR_ADR			= 0x20A4,
	MFC_SI_CH1_DPB_CONFIG_CTRL		= 0x20A8,
	MFC_SI_CH1_ALLOCATED_DPD_SZ		= 0x20Ac,
	
 	// Encoded Data Formatter Unit Register
	MFC_EDFU_STR_BF_MODE_CTRL		= 0xc05c,
	
	// Encoder Common Register
	MFC_ENC_PIC_TYPE_CTRL			= 0xc504,
	MFC_ENC_B_RECON_WRITE_ON		= 0xc508,
	MFC_ENC_MSLICE_CTRL				= 0xc50c,
	MFC_ENC_MSLICE_MB				= 0xc510,
	MFC_ENC_MSLICE_BYTE				= 0xc514,
	MFC_ENC_CIR_NUM					= 0xc518,
	MFC_ENC_MAP_FOR_CUR				= 0xc51c,
	MFC_ENC_PADDIMG_CTRL			= 0xc520,

	// New Register
	MFC_ENC_CMN_INTRA_BIAS			= 0xc588,
	MFC_ENC_CMN_BI_DIRECT_BIAS		= 0xc58c,	
	
	//9.Rate Control Register
	MFC_RC_CONFIG					= 0xc5a0,
	MFC_RC_FRAME_RATE				= 0xc5a4,
	MFC_RC_BIT_RATE					= 0xc5a8,
	MFC_RC_QBOUND					= 0xc5ac,
	MFC_RC_RPARA					= 0xc5b0,
	MFC_RC_MB_CTRL					= 0xc5b4,
	
	//10.H264 Encoder Register
	H264_ENC_ENTRP_MODE				= 0xd004,
	H264_ENC_NUM_OF_REF				= 0xd010,
	
	H264_ENC_MDINTER_WEIGHT			= 0xd01c,
	H264_ENC_MDINTRA_WEIGHT			= 0xd020,
	H264_ENC_TRANS_8X8_FLAG			= 0xd034,	
	
	MPEG4_ENC_QUART_PXL				= 0xe008,

#if ACOMPANY
	H264_ENC_PRED_MODE_CTRL 		= 0xD0D8,
	H264_ENC_I4X4_I8X8_WEIGHT		= 0xd044,
	H264_ENC_I16X16_WEIGHT			= 0xd048,
#endif
};

enum MFC_SHARED_REG
{
	EXTENEDED_DECODE_STATUS			= 0x0000,
	SET_FRAME_TAG					= 0x0004,
	GET_FRAME_TAG_TOP				= 0x0008,
	GET_FRAME_TAG_BOT				= 0x000C,
	PIC_TIME_TOP					= 0x0010,
	PIC_TIME_BOT					= 0x0014,
	START_BYTE_NUM					= 0x0018,
	CROP_INFO1						= 0x0020,
	CROP_INFO2						= 0x0024,
	EXT_ENC_CONTROL					= 0x0028,
	ENC_PARAM_CHANGE				= 0x002C,
	VOP_TIMING						= 0x0030,
	HEC_PERIOD						= 0x0034,
	METADATA_ENABLE					= 0x0038,
	METADATA_STATUS					= 0x003C,
	METADATA_DISPLAY_INDEX			= 0x0040,
	EXT_METADATA_START_ADDR			= 0x0044,
	PUT_EXTRADATA					= 0x0048,
	EXTRADATA_ADDR					= 0x004C,
	ALLOCATED_LUMA_DPB_SIZE			= 0x0064,
	ALLOCATED_CHROMA_DPB_SIZE		= 0x0068,
	ALLOCATED_MV_SIZE				= 0x006C,
	P_B_FRAME_QP					= 0x0070,
};



MFC oMfc;
MFC_ENC_COFIG oMfcEncConfig[MAX_PROCESS_NUM];
InputParameters oH264ConfigInput;
static MapMpeg4Enc oMpeg4ConfigInput;
static MapH263Enc oH263ConfigInput;

static MapList MapH264Enc[] = {
	
    {"InputFile", &oH264ConfigInput.infile, 1, 0, 0, 0, 0, 1},
	{"FramesToBeEncoded", &oH264ConfigInput.no_frames, 0, 30, 2, 1, 0, 1},

	//// / NORMAL ////////
	{"ProfileIDC", &oH264ConfigInput.ProfileIDC, 0, PROFILE_IDC, 0, 0, 0, 1},
	{"LevelIDC", &oH264ConfigInput.LevelIDC, 0, LEVEL_IDC, 0, 0, 0, 1},
	{"SourceWidth", &oH264ConfigInput.img_width, 0, 176, 1, 32, 1280, 1},
	{"SourceHeight", &oH264ConfigInput.img_height, 0, 144, 1, 32, 720, 1},
	{"IntraPeriod", &oH264ConfigInput.intra_period, 0, 24, 1, 0, 65536, 1},
	{"IDRPeriod", &oH264ConfigInput.IDR_period, 0, 24, 1, 0, 65536, 1},
	{"NumberReferenceFrames", &oH264ConfigInput.num_ref_frames, 0, 1, 1, 1, 16, 1},
	{"NUM_REFS_FOR_P", &oH264ConfigInput.NUM_REFS_FOR_P, 0, 1, 1, 0, 100, 1}, 
	{"SliceMode", &oH264ConfigInput.SliceMode, 0, 0, 1, 0, 4, 1},
	{"SliceArgument", &oH264ConfigInput.SliceArgument, 0, 0, 1, 0, 9999999, 1}, 
	{"NumberBFrames", &oH264ConfigInput.successive_Bframe, 0, 0, 1, 1, 2, 0},
	{"DirectInferenceFlag", &oH264ConfigInput.directInferenceFlag, 0, 1, 1, 0, 1, 0},
	{"LoopFilterDisable", &oH264ConfigInput.LFDisableIdc, 0, 0, 1, 0, 2, 1},
	{"LoopFilterAlphaC0Offset", &oH264ConfigInput.LFAlphaC0Offset, 0, 0, 1, -6, 6, 1},
	{"LoopFilterBetaOffset", &oH264ConfigInput.LFBetaOffset, 0, 0, 1, -6, 6, 1},
	{"SymbolMode", &oH264ConfigInput.symbol_mode, 0, 0, 1, UVLC, CABAC, 1},
	{"PicInterlace", &oH264ConfigInput.PicInterlace, 0, 0, 1, 0, 1, 0},
	{"Transform8x8Mode", &oH264ConfigInput.Transform8x8Mode, 0, 0, 1, 0, 1, 0},
	{"IntraMD8x8Disable", &oH264ConfigInput.IntraMD8x8Disable, 0, 0, 1, 0, 1, 1},
	{"IntraMD4x4Disable", &oH264ConfigInput.IntraMD4x4Disable, 0, 0, 1, 0, 1, 1},
	{"IntraMD4x4Weight", &oH264ConfigInput.IntraMD4x4Weight, 0, 0, 1, 0, 32767, 1},
	{"IntraMD8x8Weight", &oH264ConfigInput.IntraMD8x8Weight, 0, 0, 1, 0, 32767, 1},
	{"IntraMD16x16Weight", &oH264ConfigInput.IntraMD16x16Weight, 0, 0, 1, 0, 32767, 1},
	{"InterSearch16x8", &oH264ConfigInput.InterSearch16x8, 0, 1, 1, 0, 1, 1},
	{"InterSearch8x16", &oH264ConfigInput.InterSearch8x16, 0, 1, 1, 0, 1, 1},
	{"InterSearch8x8", &oH264ConfigInput.InterSearch8x8, 0, 1, 1, 0, 1, 1},
	{"DisableHpelME", &oH264ConfigInput.DisableHpelME, 0, 0, 1, 0, 1, 1},
	{"DisableQpelME", &oH264ConfigInput.DisableQpelME, 0, 0, 1, 0, 1, 1},
	{"MDInterWeightPPS", &oH264ConfigInput.MDInterWeightPPS, 0, 300, 1, 0, 32767, 1},	
	{"MDIntraWeightPPS", &oH264ConfigInput.MDIntraWeightPPS, 0, 170, 1, 0, 32767, 1},	
	{"RandomIntraMBRefresh", &oH264ConfigInput.RandomIntraMBRefresh, 0, 0, 2, 0, 0, 0},
	{"PAD_CTRL_ON", &oH264ConfigInput.pad_ctrl_on, 0, 0, 1, 0, 1, 1},
	{"LUMA_PAD_VAL", &oH264ConfigInput.luma_pad_val, 0, 0, 1, 0, 255, 1},
	{"CB_PAD_VAL", &oH264ConfigInput.cb_pad_val, 0, 0, 1, 0, 255, 1},
	{"CR_PAD_VAL", &oH264ConfigInput.cr_pad_val, 0, 0, 1, 0, 255, 1},

	/////// RC ////////
	{"EnableFRMRateControl", &oH264ConfigInput.enablefrmratecontrol, 0, 0, 1, 0, 1, 1},
	{"EnableMBRateControl", &oH264ConfigInput.enablembratecontrol, 0, 0, 1, 0, 1, 1},
	{"FrameRate", &oH264ConfigInput.FrameRate, 0, INIT_FRAME_RATE, 1, 1, 255, 1},
	{"Bitrate", &oH264ConfigInput.BitRate, 0, 10000, 2, 1000, 0, 1},
	{"FrameQp", &oH264ConfigInput.frame_qp, 0, 24, 1, 0, 51, 1},
	{"P_FrameQp", &oH264ConfigInput.p_frame_qp, 0, 24, 1, 0, 51, 1},
	{"B_FrameQp", &oH264ConfigInput.b_frame_qp, 0, 24, 1, 0, 51, 1},
	{"QSCodeMax", &oH264ConfigInput.qs_code_max, 0, 45, 1, 0, 51, 1},
	{"QSCodeMin", &oH264ConfigInput.qs_code_min, 0, 25, 1, 0, 51, 1},
	{"CBRPeriodRf", &oH264ConfigInput.rf, 0, 2, 1, 0, 65535, 1}, 
	{"DarkDisable", &oH264ConfigInput.dark_disable, 0, 0, 1, 0, 1, 1},   
	{"SmoothDisable", &oH264ConfigInput.smooth_disable, 0, 0, 1, 0, 1, 1},
	{"StaticDisable", &oH264ConfigInput.static_disable, 0, 0, 1, 0, 1, 1},
	{"ActivityDisable", &oH264ConfigInput.act_disable, 0, 0, 1, 0, 1, 1},
	{"ACDIntraMotionEn", &oH264ConfigInput.acd_intra_motion_en, 0, 0, 1, 0, 1, 1},

	{NULL, NULL, -1, 0, 0, 0, 0, 0}
};

int Split(char **str_arr, char *msg, const char *spliter, const char comment)
{
	int len;
	int len_ns;	// length with no space
	int i;
	int ret=1;
	char *comment_pos;	
	char *str_rs;	// remove space
	char *p;
	int num_item;


	comment_pos = strchr(msg, comment);
	if(comment_pos==NULL)
	{
		len = strlen(msg);
	}
	else
	{
		len = (int)(comment_pos-msg);
	}

	// check invalid parameter size
	if(len<5)
	{
		return 0;
	}
		
	str_rs = (char *)malloc(len);
	strcpy(str_rs, "");

	len_ns = 0;
	for(i=0 ; i<len ; i++)
	{
		// remove unused characters
		if(msg[i] != ' ' && msg[i]!='\n' && msg[i]!='\r' && msg[i]!='\t')
		{
			strncat(str_rs, &msg[i], 1);
			len_ns++;
		}
	}
	
	num_item = 0;

	// separate token and value
	if((p = strtok(str_rs, spliter))!=NULL)
	{
		strcpy(str_arr[num_item++], p);
	}

	// check small length parameter (invalid)
	if(len_ns<5 || p==NULL)
		ret = 0;

	while(p!=NULL)
	{
		if((p = strtok(NULL, spliter))!=NULL)
		{
			strcpy(str_arr[num_item++], p);
		}
	}

	if(num_item != 2)
		ret = 0;
		
	free(str_rs);
	
	return ret;

}


MapList *FindItem(char *item_name)
{
	int idx=0;
	
	while(MapH264Enc[idx].TokenName != NULL)
	{			
		if(strcmp(MapH264Enc[idx].TokenName, item_name)==0)
		{
			return &MapH264Enc[idx];
		}
		idx++;
	}

	// nothing matched with current Map list
	return NULL;
}

int Map2H264EncConfig(char **line_item)
{
	MapList *cur_map;

	// line_item[0] is TokenName
	cur_map = FindItem(line_item[0]);
	if(cur_map!=NULL)
	{
		if(cur_map->Type == 0)
		{
			// line_item[1] is Default value, type is integer
			int *cur_value = (int *)cur_map->Place;
			*cur_value = atoi(line_item[1]);
		}
		else if(cur_map->Type == 1)
		{
			// line_item[1] is Default value, type is string
			char *cur_string = (char *)cur_map->Place;
			strcpy(cur_string, line_item[1]);
		}
		return 1;
	}
	else
	{
		// nothing to parse or wrong param
		return 0;
	}
}

static char ucInputFileName[MAX_MSG_SIZE];
static u32 uEncodingParameters[26];


void MFC_ResetIp(void)
{
	u32 i;
	u32 uMcStatus;

	// stop procedure
	mfcOutp32(MFC_SW_RESET, 0x3f7);		// VI reset
	mfcOutp32(MFC_SW_RESET, 0x3f6);		// RISC reset
	mfcOutp32(MFC_SW_RESET, 0x3e2);		// All reset except MC
	Delay(100);						

	// MC status check
	do{
		mfcInp32(MFC_MC_STATUS, uMcStatus);
	}while(uMcStatus & 0x3);

	mfcOutp32(MFC_SW_RESET, 0x0);		// MC reset
	
	// reset procedure
	mfcOutp32(MFC_SW_RESET, 0x3fe);
}

// Get current display framebuffer index
u8 MFC_GetDisplayFbAddr(u32* pDisplayAddrOfY, u32* pDisplayAddrOfC)
{
	u32 uDisplayAddrOfY,uDisplayStatus;
	u32 uDisplayAddrOfC;
	u8 bResult = false;

	mfcInp32(MFC_SI_DISPLAY_STATUS, uDisplayStatus);
	//if(uDisplayStatus&(1<<3))
	//	Disp("[INTERLACE]");
	
	if((uDisplayStatus&0x3) == 1 || (uDisplayStatus&0x3)==2)
	{
		mfcInp32(MFC_SI_DISPLAY_Y_ADR, uDisplayAddrOfY);
		mfcInp32(MFC_SI_DISPLAY_C_ADR, uDisplayAddrOfC);

		*pDisplayAddrOfY = uDisplayAddrOfY<<11;
		*pDisplayAddrOfC = uDisplayAddrOfC<<11;
		
		mfcDbg(("DispOfY=0x%x, DispOfC=0x%0x\n", uDisplayAddrOfY, uDisplayAddrOfC));
		bResult = true;
	}
	return bResult;	
}

void MFC_ClearR2HCmdRtn(void)
{
	mfcOutp32(MFC_RISC_HOST_INT, 0); // clear interrupt
	mfcOutp32(MFC_RISC2HOST_CMD, 0);
}

void MFC_ClearPending(void)
{
	mfcOutp32(MFC_RISC_HOST_INT, 0); // clear interrupt
	mfcOutp32(MFC_RISC2HOST_CMD, 0);
}

void MFC_ClearRtnInstance(void)
{
	mfcOutp32(MFC_SI_RTN_CHID, 0xffff);
}

void MFC_GetRtnInstanceId(u32* pRtnInstId)
{
	mfcInp32(MFC_SI_RTN_CHID, *pRtnInstId);
	//Assert(*pRtnInstId == 0xffff);
}

void MFC_LoadFirmwareIntoIp(u32 uDramABaseAddr, u32 uDramBBaseAddr)
{
	u32 i, x;
	u32 uData;
	u32 uFwStatus;
	u32 uMcStatus;
	u32 uFwLdStatus;
	u32 uFwSize;
	u32 uAllocatedFwBufSZ;
	u32 uIntReg;

	do{
		mfcInp32(MFC_MC_STATUS, uMcStatus);
	} while(uMcStatus != 0); 

	// Reset IP
	MFC_ResetIp(); 

	// Set dram base Addr
	mfcOutp32(MFC_MC_DRAMBASE_ADDR_A, uDramABaseAddr);
	mfcOutp32(MFC_MC_DRAMBASE_ADDR_B, uDramBBaseAddr);
	oMfc.m_uDramABaseAddr = uDramABaseAddr;
	oMfc.m_uDramBBaseAddr = uDramBBaseAddr;
	oMfc.m_uDramBDpbStAddr = uDramBBaseAddr;
	
	// MFC FirmWare Booting
	oMfc.m_bFwReady=false;
	// 1. Initialize registers of stream I/F for decorder
	mfcOutp32(MFC_SI_CH0_INST_ID, 0x0000ffff);
	mfcOutp32(MFC_SI_CH1_INST_ID, 0x0000ffff);

	mfcOutp32(MFC_RISC2HOST_CMD, 0);
	mfcOutp32(MFC_HOST2RISC_CMD, 0);

	// 2. release reset signal to the RISC.
	mfcOutp32(MFC_SW_RESET, 0x3ff);

	// 3. Wait until the F/W is ready to receive commands
	do{
		mfcInp32(MFC_FW_STATUS, uFwStatus);
	} while(uFwStatus != 1); // 0: Not ready, 1: ready
 
	while(!oMfc.m_bFwReady);
	oMfc.m_bFwReady=false;
 
	// 4. Initialize the FirmWare
	oMfc.m_bDecFirstInstance = true;
	
	oMfc.m_bSysInit=false;
	MFC_SendH2RCmd(H2R_CMD_SYS_INIT, 300*1024);	// Host2Risc_cmd_sys_init, FW buf size:300k

	oMfc.m_uFwBufEndAddr = uDramABaseAddr + 300*1024;

	// FwCodeStAddr must be same uDramABaseAddr.	
	oMfc.m_uConTextBufEndAddr = Align(uDramABaseAddr + 10*1024*1024, 2048); // 2MB buffer temp buffer(include Mem Of f/w & Context size)

	while(!oMfc.m_bSysInit);
	oMfc.m_bSysInit=false;
}

void MFC_GetSharedMemRegBufEndAddr(u32 *pSharedMemRegBufEndAddr)
{
	u32 uSharedMemBufEndAddr;
	
	oMfc.m_uShardMemBufAddr[0] = Align(oMfc.m_uConTextBufEndAddr, 2048);
	oMfc.m_uShardMemBufAddr[1]  = Align(oMfc.m_uShardMemBufAddr[0]+1024, 2048);
	uSharedMemBufEndAddr = Align(oMfc.m_uShardMemBufAddr[1]+1024, 2048);
	memset((void *)oMfc.m_uShardMemBufAddr[0], 0, uSharedMemBufEndAddr-oMfc.m_uShardMemBufAddr[0]);
	
	mfcOutp32(MFC_SI_CH0_HOST_WR_ADR, oMfc.m_uShardMemBufAddr[0]-oMfc.m_uDramABaseAddr);
	mfcOutp32(MFC_SI_CH1_HOST_WR_ADR, oMfc.m_uShardMemBufAddr[1]-oMfc.m_uDramABaseAddr);

	*pSharedMemRegBufEndAddr = uSharedMemBufEndAddr;
}

// Notify : uProcessIdx and  uInstanceId are different.
void MFC_InitProcessForDecoding(
				u32 uProcessIdx, VIDEO_CODEC_TYPE eDecCodec, u8 bPixelCacheEn, u8 bMp4DeblkEn, u8 bCrcEn)
{
	u32 i;
	u32 uCpbSize;
	u32 uDescBufSize;
	u32 uDecCodec;
	u32 uRisc2Host;
	u32 uInstanceId;
	u32 uPixelCacheOnOff;
	u32 uMpeg4LfFilterOnOff;
 	u32 uContextAddrOffset;
	u32 uFreeMemBufA, uFreeMemBufB;
	u32 uFirstDecProcessIdx;
	
	oMfc.m_bError = false;
	oMfc.m_uProcessIdx=uProcessIdx;
	oMfc.m_uCmdChType = TWO_CH_CMD;	
	
	uDecCodec = (eDecCodec==H264_DEC) ? 0:
				 (eDecCodec==VC1AP_DEC) ? 1:
				 ((eDecCodec==MPEG4_DEC)||(eDecCodec==XviD_DEC)) ? 2:
				 ((eDecCodec==MPEG1_DEC)||(eDecCodec==MPEG2_DEC)) ? 3:
				 (eDecCodec==H263_DEC) ? 4:
				 (eDecCodec==VC1RCV_DEC) ? 5:
				 (eDecCodec==DivX311_DEC) ? 6:
				 (eDecCodec==DivX412_DEC) ? 7: 
				 (eDecCodec==DivX502_DEC) ? 8:	// DivX502_DEC(Ver 5.00, 5.01, 5.02)
				 (eDecCodec==DivX503_DEC) ? 9: // DivX503_DEC(Ver 5.03 and upper)
				 Assert(0);

	if(eDecCodec==H264_DEC)
		oMfc.m_uConTextBufSize[uProcessIdx] = 600*1024;
	else
		oMfc.m_uConTextBufSize[uProcessIdx] = 10*1024;

	if(uProcessIdx == 0)
		oMfc.m_uConTextBufAddr[uProcessIdx] = oMfc.m_uFwBufEndAddr;
 	else
 		oMfc.m_uConTextBufAddr[uProcessIdx] = oMfc.m_uConTextBufAddr[uProcessIdx-1] + oMfc.m_uConTextBufSize[uProcessIdx-1];

 	uContextAddrOffset = (oMfc.m_uConTextBufAddr[uProcessIdx] - oMfc.m_uDramABaseAddr)>>11; // must be 2K align
	
	// 2.Open Instance
	// Enable pixel cache : default p picture only
	oMfc.m_bOpenInstance=false;
	uPixelCacheOnOff = (bPixelCacheEn) ? 0 : 3;

	if(bCrcEn==false)
		MFC_SendH2RCmdWithFullArg(H2R_CMD_OPEN_INSTANCE, uDecCodec, bPixelCacheEn, uContextAddrOffset, oMfc.m_uConTextBufSize[uProcessIdx]);
	else
		MFC_SendH2RCmdWithFullArg(H2R_CMD_OPEN_INSTANCE, uDecCodec, (1<<31)|bPixelCacheEn, uContextAddrOffset, oMfc.m_uConTextBufSize[uProcessIdx]);

	oMfc.m_bbEnableCrc = bCrcEn;
	
 	// 3. wait until 1(Open Instance Return) 
	while(!oMfc.m_bOpenInstance);
	oMfc.m_bOpenInstance=false; 

	// 4. Get the Open Instance ID.
	mfcInp32(MFC_RISC2HOST_ARG1, uInstanceId);
	oMfc.m_eCodecMode[oMfc.m_uProcessIdx] = eDecCodec;
	oMfc.m_uInstanceId[oMfc.m_uProcessIdx] = uInstanceId;

	// 5. mpeg4 loof filter on/off
	uMpeg4LfFilterOnOff = (bMp4DeblkEn) ? 1 : 0;
	mfcOutp32(MFC_LF_CONTROL, uMpeg4LfFilterOnOff);

	if(uProcessIdx == 0)
		oMfc.m_bDecFirstInstance=true;
}

void MFC_SetDecPixelCacheType(MFC_DEC_PIXEL_CACHE_TYPE ePixelCacheType)
{
	if(ePixelCacheType == P_ONLY)
	{
		mfcOutp32(MFC_HOST2RISC_ARG2, 0);
	}
	else if(ePixelCacheType == B_ONLY)
	{
		mfcOutp32(MFC_HOST2RISC_ARG2, 0x1);
	}
	else if(ePixelCacheType == P_AND_B)
	{
		mfcOutp32(MFC_HOST2RISC_ARG2, 0x2);
	}
	else if(ePixelCacheType == PIX_CACHE_OFF)
	{
		mfcOutp32(MFC_HOST2RISC_ARG2, 0x3);
	}
	else
	{
		Assert(0);
	}
}

void MFC_SetDramBDpbBufStAddr(u32 uDramBDpbBufAddr)
{
	oMfc.m_uDramBDpbStAddr = uDramBDpbBufAddr;
}

// Must be set before Parsing command
void MFC_SetDpbCfg(MFC_CMD_CH_NUM eCmdCh, u8 bIsSliceBaseDec, u8 bEnableCfgDelay, u32 uFrameNumOfDispDelay)
{
	u32 uDpbCfgCtrl;

	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H264_DEC)
		uDpbCfgCtrl = (bIsSliceBaseDec<<31)|(bEnableCfgDelay<<30)|(uFrameNumOfDispDelay<<16);
	else
		uDpbCfgCtrl = (bIsSliceBaseDec<<31)|(0<<30)|(0<<16);
		
	if(eCmdCh == CMD_CH0)
	{
		mfcOutp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uDpbCfgCtrl);
	}
	else
	{
		mfcOutp32(MFC_SI_CH1_DPB_CONFIG_CTRL, uDpbCfgCtrl);
	}
}

void MFC_CalcTiled4x2PlaneSize(u32 uWidth, u32 uHeight, u32* pAlignedPlaneSz)
{
	u32 uAlignedWidth;
	u32 uAlignedHeght;
	
	uAlignedWidth = Align(uWidth, 128);
	uAlignedHeght = Align(uHeight, 32);	

	*pAlignedPlaneSz = Align( (uAlignedWidth*uAlignedHeght), 8*1024 ); // must be aligned to 8KB
}

void MFC_ParseStreamHeader(MFC_CMD_CH_NUM eCmdCh, u32 uCpbBufStAddr, u32 uOneFrameSz)
{
	u32 uRisc2Host;
	u32 uRtnChId;
	u32 uDescBufEndAddrTmp[2];
	u32 uDescEndAddr;
	u32 uH264VertNbMvBaseAddr;
	u32 uH264NbIpBaseAddr;
	u32 uUpNbMvbaseAddr;
	u32 uOTLineBaseAddr;
	u32 uBitPlainBaseAddr[3];
	u32 uNbDcAcBaseAddr;
	u32 uSpBaseAddr;
	u32 uH264VertNbMvBaseAddrReg;
	u32 uH264NbIpBaseAddrReg;
	u32 uNbDcAcBaseAddrReg;
	u32 uUpNbMvbaseAddrReg;
	u32 uOTLineBaseAddrReg;
	u32 uBitPlainBaseAddrReg[3];
	u32 uSharedMemBufAddr;
	u32 uSharedMemBufEndAddr;
	
	oMfc.m_bSeqDone=false;	
	oMfc.m_bFatalErr = false;

	// Cpb, Desc buffer base addr
	oMfc.m_uCpbBaseAddr[0] = Align(uCpbBufStAddr, 2048);
	oMfc.m_uDescBaseAddr[0] = Align(oMfc.m_uCpbBaseAddr[0]+CPB_BUF_SIZE, 2048);
	uDescBufEndAddrTmp[0] = oMfc.m_uDescBaseAddr[0] + DESC_BUF_SIZE;
	oMfc.m_uDescBufEndAddr = uDescBufEndAddrTmp[0];

	if(oMfc.m_uCmdChType == TWO_CH_CMD)
	{
		oMfc.m_uCpbBaseAddr[1] = Align(uDescBufEndAddrTmp[0], 2048);
		oMfc.m_uDescBaseAddr[1] = Align(oMfc.m_uCpbBaseAddr[1]+CPB_BUF_SIZE, 2048);
		uDescBufEndAddrTmp[1]  = oMfc.m_uDescBaseAddr[1] + DESC_BUF_SIZE;
		oMfc.m_uDescBufEndAddr = uDescBufEndAddrTmp[1];
	}

	// Set first framestaddr & framesize(header+one Frame)
	if(eCmdCh == CMD_CH0)
	{
		mfcOutp32(MFC_SI_CH0_ES_ADDR, (oMfc.m_uCpbBaseAddr[0]-oMfc.m_uDramABaseAddr)>>11);
		mfcOutp32(MFC_SI_CH0_DESC_ADDR, (oMfc.m_uDescBaseAddr[0]-oMfc.m_uDramABaseAddr)>>11	);		
		mfcOutp32(MFC_SI_CH0_ES_DEC_UNIT_SIZE, uOneFrameSz);
		mfcOutp32(MFC_SI_CH0_CPB_SIZE, CPB_BUF_SIZE);
		mfcOutp32(MFC_SI_CH0_DESC_SIZE, DESC_BUF_SIZE);
		// Start Parsing (Set instance ID at Ch1)
		mfcOutp32(MFC_SI_CH0_INST_ID, (0x1<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx]);
	}
	else
	{
		mfcOutp32(MFC_SI_CH1_ES_ADDR, (oMfc.m_uCpbBaseAddr[1]-oMfc.m_uDramABaseAddr)>>11);
		mfcOutp32(MFC_SI_CH1_DESC_ADDR, (oMfc.m_uDescBaseAddr[1]-oMfc.m_uDramABaseAddr)>>11	);	
		mfcOutp32(MFC_SI_CH1_ES_DEC_UNIT_SIZE, uOneFrameSz);
		mfcOutp32(MFC_SI_CH1_CPB_SIZE, CPB_BUF_SIZE);
		mfcOutp32(MFC_SI_CH1_DESC_SIZE, DESC_BUF_SIZE);
		// Start Parsing (Set Ch2 instance ID at Ch2)
		mfcOutp32(MFC_SI_CH1_INST_ID, (0x1<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx]);
	} 
	while(!(oMfc.m_bSeqDone || oMfc.m_bError));

#if 0
	if(oMfc.m_bError==false)
		while(!oMfc.m_bSeqDone);
	else
		while(!oMfc.m_bFatalErr);
	oMfc.m_bFatalErr=false;
#endif

	oMfc.m_bSeqDone=false;
	oMfc.m_bError=false;
}


#if 0
void MFC_SetBufAddrOfDecoder(u32 uDecABufStAddr, u32 uDecBBufStAddr)
{
	u32 uDpbCfgCtrl;
	u32 uDramADpbBufStAddr;
	u32 uDramBDpbBufStAddr;
	
	MFC_SetDecCodecBuf(uDecABufStAddr);
	MFC_GetDecCodecBufEndAddr(&uDramADpbBufStAddr);
	
	uDramADpbBufStAddr = Align(uDramADpbBufStAddr, 8*1024);
	uDramADpbBufStAddr =  Align(uDecBBufStAddr, 8*1024);
	
	MFC_SetDpbBuf(oMfc.m_uProcessIdx, uDramADpbBufStAddr, uDramBDpbBufStAddr);

	mfcInp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uDpbCfgCtrl);
	uDpbCfgCtrl &=~ (0xffff);
	uDpbCfgCtrl |=  oMfc.m_uNumOfDpb;
	mfcOutp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uDpbCfgCtrl);

	MFC_InitBuffer(CMD_CH0, oMfc.m_uProcessIdx);
}
#else
void MFC_SetBufAddrOfDecoder(u32 uDecABufStAddr, u32 uDecBBufStAddr)
{
	u32 i;
	u32 uDpbCfgCtrl;
	u32 uNumOfDpb;
	u32 uLumaPlaneSZ;
	u32 uChromaPlaneSZ;
	u32 uMvPlaneSZ;
	u32 uDpbSZ;
	u32 uAlignedW, uAlignedH, uAlignedChromaH, uAlignedMvH;	
	u32 uH264VertNbMvBaseAddr;
	u32 uH264NbIpBaseAddr;
	u32 uSAMvBaseAddr;
	u32 uSAMvBaseAddrReg;
	u32 uSpBaseAddrReg;
	u32 uUpNbMvbaseAddr;
	u32 uOTLineBaseAddr;
	u32 uBitPlainBaseAddr[3];
	u32 uNbDcAcBaseAddr;
	u32 uSpBaseAddr;
	u32 uH264VertNbMvBaseAddrReg;
	u32 uH264NbIpBaseAddrReg;
	u32 uNbDcAcBaseAddrReg;
	u32 uUpNbMvbaseAddrReg;
	u32 uOTLineBaseAddrReg;
	u32 uBitPlainBaseAddrReg[3];
	u32 uCodecBufEndAddr;
	u32 uDramADpbBufStAddr;
	u32 uDramBDpbBufStAddr;

	uH264VertNbMvBaseAddrReg = MFC_COMMON_BASE_ADDR+35*4;
	uH264NbIpBaseAddrReg = MFC_COMMON_BASE_ADDR+36*4;
	uNbDcAcBaseAddrReg = MFC_COMMON_BASE_ADDR+35*4;
	uUpNbMvbaseAddrReg = MFC_COMMON_BASE_ADDR+36*4;
	uSAMvBaseAddrReg = MFC_COMMON_BASE_ADDR+37*4;
	uOTLineBaseAddrReg = MFC_COMMON_BASE_ADDR+38*4;
	uBitPlainBaseAddrReg[2] = MFC_COMMON_BASE_ADDR+39*4;
	uBitPlainBaseAddrReg[1] = MFC_COMMON_BASE_ADDR+40*4;
	uBitPlainBaseAddrReg[0] = MFC_COMMON_BASE_ADDR+41*4;
	uSpBaseAddrReg	 = MFC_COMMON_BASE_ADDR+42*4;

	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H264_DEC)
	{
		// Set VertNbMv Buffer BaseRam addr
		uH264VertNbMvBaseAddr = Align(uDecABufStAddr, 2048);
		mfcOutp32(uH264VertNbMvBaseAddrReg, (uH264VertNbMvBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uH264NbIpBaseAddr = uH264VertNbMvBaseAddr + 16*1024;

		// Set NbForIP Buffer BaserRam reg
		mfcOutp32(uH264NbIpBaseAddrReg, (uH264NbIpBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uCodecBufEndAddr  = uH264NbIpBaseAddr + 32*1024;
	}
	else		// other codec except H264
	{
		uNbDcAcBaseAddr = Align(uDecABufStAddr, 2048);
		mfcOutp32(uNbDcAcBaseAddrReg, (uNbDcAcBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uUpNbMvbaseAddr = uNbDcAcBaseAddr + 16*1024;
		
		mfcOutp32(uUpNbMvbaseAddrReg, (uUpNbMvbaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uSAMvBaseAddr = uUpNbMvbaseAddr + 68*1024;

		mfcOutp32(uSAMvBaseAddrReg, (uSAMvBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uOTLineBaseAddr = uSAMvBaseAddr + 136*1024;

		mfcOutp32(uOTLineBaseAddrReg, (uOTLineBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uBitPlainBaseAddr[2]  = uOTLineBaseAddr + 32*1024;

		mfcOutp32(uBitPlainBaseAddrReg[2], (uBitPlainBaseAddr[2]-oMfc.m_uDramABaseAddr)>>11);
		uBitPlainBaseAddr[1] = uBitPlainBaseAddr[2] + 2*1024;

		mfcOutp32(uBitPlainBaseAddrReg[1], (uBitPlainBaseAddr[1]-oMfc.m_uDramABaseAddr)>>11);
		uBitPlainBaseAddr[0] = uBitPlainBaseAddr[1] + 2*1024;

		mfcOutp32(uBitPlainBaseAddrReg[0], (uBitPlainBaseAddr[0]-oMfc.m_uDramABaseAddr)>>11);
		uSpBaseAddr = uBitPlainBaseAddr[0] + 2*1024;
		
		mfcOutp32(uSpBaseAddrReg, (uSpBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uCodecBufEndAddr = uSpBaseAddr + 68*1024;
	}	

	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] != H264_DEC)
	{
		mfcDbg(("NbDcAcBaseAddr:0x%x \n", uNbDcAcBaseAddr));
		mfcDbg(("UpNbMvbaseAddr:0x%x \n", uUpNbMvbaseAddr));
		mfcDbg(("OTLineBaseAddr:0x%x \n", uOTLineBaseAddr));
		mfcDbg(("BitPlainBaseAddr[2]:0x%x \n", uBitPlainBaseAddr[2]));
		mfcDbg(("BitPlainBaseAddr[1]:0x%x \n", uBitPlainBaseAddr[1]));
		mfcDbg(("BitPlainBaseAddr[0]:0x%x \n", uBitPlainBaseAddr[0]));	
		mfcDbg(("CpbBaseAddr[0]:0x%x\n", oMfc.m_uCpbBaseAddr[0]));
		mfcDbg(("CpbBaseAddr[1]:0x%x\n", oMfc.m_uCpbBaseAddr[1]));		
	}
	else
	{
		mfcDbg(("uH264VertNbMvBaseAddr:0x%x \n", uH264VertNbMvBaseAddr));
		mfcDbg(("H264NbIpBaseAddr:0x%x \n", uH264NbIpBaseAddr));
		mfcDbg(("CpbBaseAddr[0]:0x%x\n", oMfc.m_uCpbBaseAddr[0]));
		mfcDbg(("CpbBaseAddr[1]:0x%x\n", oMfc.m_uCpbBaseAddr[1]));	
	}
	
	uDramADpbBufStAddr = Align(uCodecBufEndAddr, 8*1024);
	uDramBDpbBufStAddr =  Align(uDecBBufStAddr, 8*1024);

	mfcInp32(MFC_SI_MIN_NUB_DPB, uNumOfDpb);
	
	Disp("Num Of Min DPB : %d\n", uNumOfDpb);

#if 1
	uNumOfDpb = uNumOfDpb;
	Disp("Num Of Min DPB + Extra DPB : %d\n", uNumOfDpb);
#else
#if 0
	if(oMfc.m_uProcessIdx==0)
	{
		uNumOfDpb = uNumOfDpb;
		Disp("Num Of Min DPB + Extra DPB : %d\n", uNumOfDpb);
	}
	else
	{
		uNumOfDpb = uNumOfDpb+1;
		Disp("Num Of Min DPB + Extra DPB : %d\n", uNumOfDpb);
	}
#else
	uNumOfDpb = uNumOfDpb+rand()%(32-uNumOfDpb);
	Disp("Num Of Min DPB + Extra DPB : %d\n", uNumOfDpb);
#endif
#endif

	oMfc.m_uNumOfDpb = uNumOfDpb;

	mfcInp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uDpbCfgCtrl);
	uDpbCfgCtrl &=~ (0xffff);
	uDpbCfgCtrl |=  oMfc.m_uNumOfDpb;
	mfcOutp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uDpbCfgCtrl);
	
	MFC_CalcTiled4x2PlaneSize(oMfc.m_uPicHsz[oMfc.m_uProcessIdx], oMfc.m_uPicVsz[oMfc.m_uProcessIdx], &uLumaPlaneSZ);
	MFC_CalcTiled4x2PlaneSize(oMfc.m_uPicHsz[oMfc.m_uProcessIdx], oMfc.m_uPicVsz[oMfc.m_uProcessIdx]/2, &uChromaPlaneSZ);
	MFC_CalcTiled4x2PlaneSize(oMfc.m_uPicHsz[oMfc.m_uProcessIdx], oMfc.m_uPicVsz[oMfc.m_uProcessIdx]/4, &uMvPlaneSZ);

	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H264_DEC)
	{
		uDpbSZ = uLumaPlaneSZ+uChromaPlaneSZ+uMvPlaneSZ;
		for (i=0; i<uNumOfDpb; i++)
		{
			oMfc.m_uFrameAddrY[oMfc.m_uProcessIdx][i] = uDramBDpbBufStAddr + (uLumaPlaneSZ+uMvPlaneSZ)*i;
			mfcOutp32(MFC_COMMON_BASE_ADDR+(64+i)*4, (oMfc.m_uFrameAddrY[oMfc.m_uProcessIdx][i]-oMfc.m_uDramBBaseAddr)>>11);

			oMfc.m_uFrameAddrC[oMfc.m_uProcessIdx][i] = uDramADpbBufStAddr + uChromaPlaneSZ*i;
			mfcOutp32(MFC_COMMON_BASE_ADDR+i*4, (oMfc.m_uFrameAddrC[oMfc.m_uProcessIdx][i]-oMfc.m_uDramABaseAddr)>>11);

			oMfc.m_uFrameAddrMv[oMfc.m_uProcessIdx][i] = uDramBDpbBufStAddr + (uLumaPlaneSZ+uMvPlaneSZ)*i + uLumaPlaneSZ;
			mfcOutp32(MFC_COMMON_BASE_ADDR+(96+i)*4, (oMfc.m_uFrameAddrMv[oMfc.m_uProcessIdx][i]-oMfc.m_uDramBBaseAddr)>>11);

			mfcDbg(("FrameAddrY[%d]:0x%x\n", i, oMfc.m_uFrameAddrY[oMfc.m_uProcessIdx][i]));
			mfcDbg(("FrameAddrC[%d]:0x%x\n", i, oMfc.m_uFrameAddrC[oMfc.m_uProcessIdx][i]));	
			mfcDbg(("FrameAddrMv[%d]:0x%x\n", i, oMfc.m_uFrameAddrMv[oMfc.m_uProcessIdx][i]));	
		}
		
		// Allocate DPB size
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_LUMA_DPB_SIZE, uLumaPlaneSZ);
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_CHROMA_DPB_SIZE, uChromaPlaneSZ);
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_MV_SIZE, uMvPlaneSZ);
	}
	else	// other codec except H264
	{
		uDpbSZ = uLumaPlaneSZ+uChromaPlaneSZ;
		for (i=0; i<uNumOfDpb; i++)
		{
			oMfc.m_uFrameAddrY[oMfc.m_uProcessIdx][i] = uDramBDpbBufStAddr + uLumaPlaneSZ*i;
			mfcOutp32(MFC_COMMON_BASE_ADDR+(64+i)*4, (oMfc.m_uFrameAddrY[oMfc.m_uProcessIdx][i]-oMfc.m_uDramBBaseAddr)>>11);

			oMfc.m_uFrameAddrC[oMfc.m_uProcessIdx][i] = uDramADpbBufStAddr + uChromaPlaneSZ*i;
			mfcOutp32(MFC_COMMON_BASE_ADDR+i*4, (oMfc.m_uFrameAddrC[oMfc.m_uProcessIdx][i]-oMfc.m_uDramABaseAddr)>>11);
			
			mfcDbg(("FrameAddrY[%d]:0x%x\n", i, oMfc.m_uFrameAddrY[oMfc.m_uProcessIdx][i]));
			mfcDbg(("FrameAddrC[%d]:0x%x\n", i, oMfc.m_uFrameAddrC[oMfc.m_uProcessIdx][i]));	
		}

		// Allocate DPB size
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_LUMA_DPB_SIZE, uLumaPlaneSZ);
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_CHROMA_DPB_SIZE, uChromaPlaneSZ);
	}

	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H264_DEC)
	{
		oMfc.m_uADramDpbBufEndAddr[oMfc.m_uProcessIdx]=oMfc.m_uFrameAddrC[oMfc.m_uProcessIdx][0]+uChromaPlaneSZ*uNumOfDpb;
		oMfc.m_uBDramDpbBufEndAddr[oMfc.m_uProcessIdx]=oMfc.m_uFrameAddrY[oMfc.m_uProcessIdx][0]+(uLumaPlaneSZ+uMvPlaneSZ)*uNumOfDpb;
	}
	else
	{
		oMfc.m_uADramDpbBufEndAddr[oMfc.m_uProcessIdx]=oMfc.m_uFrameAddrC[oMfc.m_uProcessIdx][0]+uChromaPlaneSZ*uNumOfDpb;
		oMfc.m_uBDramDpbBufEndAddr[oMfc.m_uProcessIdx]=oMfc.m_uFrameAddrY[oMfc.m_uProcessIdx][0]+uLumaPlaneSZ*uNumOfDpb;
	}	
}
#endif

#if 1 // for single instance
void MFC_ParseStreamHeaderAndSetBuf(MFC_CMD_CH_NUM eCmdCh, u32 uCpbBufStAddr, u32 uOneFrameSz)
{
	u32 uDpbCfgCtrl;
	u32 uFreeMemBufA;
	u32 uFreeMemBufB;
	u32 uDecCodecBufStAddr;
	u32 uDramADpbBufStAddr;
	u32 uDramBDpbBufStAddr;
	u32 uDecABufStAddr;
	u32 uDecBBufStAddr;
	
	oMfc.m_bInitBufferDone = false;

	MFC_ParseStreamHeader(eCmdCh, uCpbBufStAddr, uOneFrameSz); 

#if 0
	uDramBDpbBufStAddr =  oMfc.m_uDramBDpbStAddr;
	uDecCodecBufStAddr = Align(oMfc.m_uDescBufEndAddr, 2*1024);
	MFC_SetDecCodecBuf(uDecCodecBufStAddr);
	MFC_GetDecCodecBufEndAddr(&uDramADpbBufStAddr);
	
	uDramADpbBufStAddr =  Align(uDramADpbBufStAddr, 8*1024);
	
	MFC_SetDpbBuf(oMfc.m_uProcessIdx, uDramADpbBufStAddr, uDramBDpbBufStAddr);
#else
	uDecABufStAddr = Align(oMfc.m_uDescBufEndAddr, 2*1024);
	//uDecBBufStAddr =  oMfc.m_uDramBBaseAddr;
	uDecBBufStAddr = oMfc.m_uDramBDpbStAddr;
	MFC_SetBufAddrOfDecoder(uDecABufStAddr, uDecBBufStAddr);
#endif

#if 0
	mfcInp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uDpbCfgCtrl);
	uDpbCfgCtrl &=~ (0xffff);
	uDpbCfgCtrl |=  oMfc.m_uNumOfDpb;
	mfcOutp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uDpbCfgCtrl);
#endif

	MFC_InitBuffer(CMD_CH0, oMfc.m_uProcessIdx);
}
#else
void MFC_ParseStreamHeaderAndSetBuf(MFC_CMD_CH_NUM eCmdCh, u32 uCpbBufStAddr, u32 uOneFrameSz)
{
	u32 uDpbCfgCtrl;
	u32 uFreeMemBufA;
	u32 uFreeMemBufB;
	u32 uDecCodecBufStAddr;
	u32 uDramADpbBufStAddr;
	u32 uDramBDpbBufStAddr;
	
	oMfc.m_bInitBufferDone = false;

	MFC_ParseStreamHeader(eCmdCh, uCpbBufStAddr, uOneFrameSz);
		
	if(oMfc.m_bDecFirstInstance==true)
	{
		MFC_GetDescriptorBufEndAddr(&uDecCodecBufStAddr);
		uDramBDpbBufStAddr =  oMfc.m_uDramBDpbStAddr;
		oMfc.m_bDecFirstInstance = false;
	}
	else
	{
		MFC_GetDbpBufEndAddr(oMfc.m_uProcessIdx-1, &uFreeMemBufA, &uFreeMemBufB);	 
		uDramBDpbBufStAddr = Align(uFreeMemBufB, 8*1024);
		uDecCodecBufStAddr = uFreeMemBufA;
	}
	uDecCodecBufStAddr = Align(uDecCodecBufStAddr, 2*1024);
	MFC_SetDecCodecBuf(uDecCodecBufStAddr);
	MFC_GetDecCodecBufEndAddr(&uDramADpbBufStAddr);
	uDramADpbBufStAddr =  Align(uDramADpbBufStAddr, 8*1024);

	MFC_SetDpbBuf(oMfc.m_uProcessIdx, uDramADpbBufStAddr, uDramBDpbBufStAddr);

	mfcInp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uDpbCfgCtrl);
	uDpbCfgCtrl &=~ (0xffff);
	uDpbCfgCtrl |=  oMfc.m_uNumOfDpb;
	mfcOutp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uDpbCfgCtrl);

	MFC_InitBuffer(CMD_CH0, oMfc.m_uProcessIdx);
}
 
void MFC_SetDecCodecBuf(u32 uDecCodecStAddr)
{
	u32 uH264VertNbMvBaseAddr;
	u32 uH264NbIpBaseAddr;
	u32 uSAMvBaseAddr;
	u32 uSAMvBaseAddrReg;
	u32 uSpBaseAddrReg;
	u32 uUpNbMvbaseAddr;
	u32 uOTLineBaseAddr;
	u32 uBitPlainBaseAddr[3];
	u32 uNbDcAcBaseAddr;
	u32 uSpBaseAddr;
	u32 uH264VertNbMvBaseAddrReg;
	u32 uH264NbIpBaseAddrReg;
	u32 uNbDcAcBaseAddrReg;
	u32 uUpNbMvbaseAddrReg;
	u32 uOTLineBaseAddrReg;
	u32 uBitPlainBaseAddrReg[3];

	uH264VertNbMvBaseAddrReg = MFC_COMMON_BASE_ADDR+35*4;
	uH264NbIpBaseAddrReg = MFC_COMMON_BASE_ADDR+36*4;
	uNbDcAcBaseAddrReg = MFC_COMMON_BASE_ADDR+35*4;
	uUpNbMvbaseAddrReg = MFC_COMMON_BASE_ADDR+36*4;
	uSAMvBaseAddrReg = MFC_COMMON_BASE_ADDR+37*4;
	uOTLineBaseAddrReg = MFC_COMMON_BASE_ADDR+38*4;
	uBitPlainBaseAddrReg[2] = MFC_COMMON_BASE_ADDR+39*4;
	uBitPlainBaseAddrReg[1] = MFC_COMMON_BASE_ADDR+40*4;
	uBitPlainBaseAddrReg[0] = MFC_COMMON_BASE_ADDR+41*4;
	uSpBaseAddrReg	 = MFC_COMMON_BASE_ADDR+42*4;
	
	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H264_DEC)
	{
		// Set VertNbMv Buffer BaseRam addr
		uH264VertNbMvBaseAddr = Align(uDecCodecStAddr, 2048);
		mfcOutp32(uH264VertNbMvBaseAddrReg, (uH264VertNbMvBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uH264NbIpBaseAddr = uH264VertNbMvBaseAddr + 16*1024;

		// Set NbForIP Buffer BaserRam reg
		mfcOutp32(uH264NbIpBaseAddrReg, (uH264NbIpBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		oMfc.m_uCodecBufEndAddr  = uH264NbIpBaseAddr + 32*1024;
	}
	else		// other codec except H264
	{
		uNbDcAcBaseAddr = Align(uDecCodecStAddr, 2048);
		mfcOutp32(uNbDcAcBaseAddrReg, (uNbDcAcBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uUpNbMvbaseAddr = uNbDcAcBaseAddr + 16*1024;
		
		mfcOutp32(uUpNbMvbaseAddrReg, (uUpNbMvbaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uSAMvBaseAddr = uUpNbMvbaseAddr + 68*1024;

		mfcOutp32(uSAMvBaseAddrReg, (uSAMvBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uOTLineBaseAddr = uSAMvBaseAddr + 136*1024;

		mfcOutp32(uOTLineBaseAddrReg, (uOTLineBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		uBitPlainBaseAddr[2]  = uOTLineBaseAddr + 32*1024;

		mfcOutp32(uBitPlainBaseAddrReg[2], (uBitPlainBaseAddr[2]-oMfc.m_uDramABaseAddr)>>11);
		uBitPlainBaseAddr[1] = uBitPlainBaseAddr[2] + 2*1024;

		mfcOutp32(uBitPlainBaseAddrReg[1], (uBitPlainBaseAddr[1]-oMfc.m_uDramABaseAddr)>>11);
		uBitPlainBaseAddr[0] = uBitPlainBaseAddr[1] + 2*1024;

		mfcOutp32(uBitPlainBaseAddrReg[0], (uBitPlainBaseAddr[0]-oMfc.m_uDramABaseAddr)>>11);
		uSpBaseAddr = uBitPlainBaseAddr[0] + 2*1024;
		
		mfcOutp32(uSpBaseAddrReg, (uSpBaseAddr-oMfc.m_uDramABaseAddr)>>11);
		oMfc.m_uCodecBufEndAddr= uSpBaseAddr + 68*1024;
	}

	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] != H264_DEC)
	{
		mfcDbg(("NbDcAcBaseAddr:0x%x \n", uNbDcAcBaseAddr));
		mfcDbg(("UpNbMvbaseAddr:0x%x \n", uUpNbMvbaseAddr));
		mfcDbg(("OTLineBaseAddr:0x%x \n", uOTLineBaseAddr));
		mfcDbg(("BitPlainBaseAddr[2]:0x%x \n", uBitPlainBaseAddr[2]));
		mfcDbg(("BitPlainBaseAddr[1]:0x%x \n", uBitPlainBaseAddr[1]));
		mfcDbg(("BitPlainBaseAddr[0]:0x%x \n", uBitPlainBaseAddr[0]));	
		mfcDbg(("CpbBaseAddr[0]:0x%x\n", oMfc.m_uCpbBaseAddr[0]));
		mfcDbg(("CpbBaseAddr[1]:0x%x\n", oMfc.m_uCpbBaseAddr[1]));		
	}
	else
	{
		mfcDbg(("uH264VertNbMvBaseAddr:0x%x \n", uH264VertNbMvBaseAddr));
		mfcDbg(("H264NbIpBaseAddr:0x%x \n", uH264NbIpBaseAddr));
		mfcDbg(("CpbBaseAddr[0]:0x%x\n", oMfc.m_uCpbBaseAddr[0]));
		mfcDbg(("CpbBaseAddr[1]:0x%x\n", oMfc.m_uCpbBaseAddr[1]));	
	}
}

void MFC_GetDescriptorBufEndAddr(u32* pDescBufEndAddr)
{
	*pDescBufEndAddr=oMfc.m_uDescBufEndAddr;
}

void MFC_GetDecCodecBufEndAddr(u32* pDecCodecBufEndAddr)
{
	*pDecCodecBufEndAddr = oMfc.m_uCodecBufEndAddr;
}

void MFC_SetDpbBuf(u32 uProcessIdx, u32 uDramADpbBufStAddr, u32 uDramBDpbBufStAddr)
{
	u32 i;
	u32 uNumOfDpb;
	u32 uLumaPlaneSZ;
	u32 uChromaPlaneSZ;
	u32 uMvPlaneSZ;
	u32 uDpbSZ;

	u32 uAlignedW, uAlignedH, uAlignedChromaH, uAlignedMvH;

	mfcInp32(MFC_SI_MIN_NUB_DPB, uNumOfDpb);
	
	Disp("Num Of Dpb : %d\n", uNumOfDpb);

	oMfc.m_uNumOfDpb = uNumOfDpb;

#if 1
	MFC_CalcTiled4x2PlaneSize(oMfc.m_uPicHsz[uProcessIdx], oMfc.m_uPicVsz[uProcessIdx], &uLumaPlaneSZ);
	MFC_CalcTiled4x2PlaneSize(oMfc.m_uPicHsz[uProcessIdx], oMfc.m_uPicVsz[uProcessIdx]/2, &uChromaPlaneSZ);
	MFC_CalcTiled4x2PlaneSize(oMfc.m_uPicHsz[uProcessIdx], oMfc.m_uPicVsz[uProcessIdx]/4, &uMvPlaneSZ);
#else
	uAlignedW = (((oMfc.m_uPicHsz[uProcessIdx])+(128))/(128)*(128));
	uAlignedH = (((oMfc.m_uPicVsz[uProcessIdx])+(32))/(32)*(32));
	uAlignedChromaH = (((oMfc.m_uPicVsz[uProcessIdx]/2)+(32))/(32)*(32));
	uAlignedMvH = (((oMfc.m_uPicVsz[uProcessIdx]/4)+(32))/(32)*(32));

	uLumaPlaneSZ = (((uAlignedW*uAlignedH)+(64*1024-1))/(64*1024)*(64*1024));	
	uChromaPlaneSZ = (((uAlignedW*uAlignedChromaH)+(64*1024-1))/(64*1024)*(64*1024));	
	uMvPlaneSZ = (((uAlignedW*uAlignedMvH)+(64*1024-1))/(64*1024)*(64*1024));
#endif


	if(oMfc.m_eCodecMode[uProcessIdx] == H264_DEC)
	{
		uDpbSZ = uLumaPlaneSZ+uChromaPlaneSZ+uMvPlaneSZ;
		for (i=0; i<uNumOfDpb; i++)
		{
			oMfc.m_uFrameAddrY[uProcessIdx][i] = uDramBDpbBufStAddr + (uLumaPlaneSZ+uMvPlaneSZ)*i;
			mfcOutp32(MFC_COMMON_BASE_ADDR+(64+i)*4, (oMfc.m_uFrameAddrY[uProcessIdx][i]-oMfc.m_uDramBBaseAddr)>>11);

			oMfc.m_uFrameAddrC[uProcessIdx][i] = uDramADpbBufStAddr + uChromaPlaneSZ*i;
			mfcOutp32(MFC_COMMON_BASE_ADDR+i*4, (oMfc.m_uFrameAddrC[uProcessIdx][i]-oMfc.m_uDramABaseAddr)>>11);

			oMfc.m_uFrameAddrMv[uProcessIdx][i] = uDramBDpbBufStAddr + (uLumaPlaneSZ+uMvPlaneSZ)*i + uLumaPlaneSZ;
			mfcOutp32(MFC_COMMON_BASE_ADDR+(96+i)*4, (oMfc.m_uFrameAddrMv[uProcessIdx][i]-oMfc.m_uDramBBaseAddr)>>11);

			mfcDbg(("FrameAddrY[%d]:0x%x\n", i, oMfc.m_uFrameAddrY[uProcessIdx][i]));
			mfcDbg(("FrameAddrC[%d]:0x%x\n", i, oMfc.m_uFrameAddrC[uProcessIdx][i]));	
			mfcDbg(("FrameAddrMv[%d]:0x%x\n", i, oMfc.m_uFrameAddrMv[uProcessIdx][i]));	
		}
		
		// Allocate DPB size
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_LUMA_DPB_SIZE, uLumaPlaneSZ);
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_CHROMA_DPB_SIZE, uChromaPlaneSZ);
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_MV_SIZE, uMvPlaneSZ);
	}
	else	// other codec except H264
	{
		uDpbSZ = uLumaPlaneSZ+uChromaPlaneSZ;
		for (i=0; i<uNumOfDpb; i++)
		{
			oMfc.m_uFrameAddrY[uProcessIdx][i] = uDramBDpbBufStAddr + uLumaPlaneSZ*i;
			mfcOutp32(MFC_COMMON_BASE_ADDR+(64+i)*4, (oMfc.m_uFrameAddrY[uProcessIdx][i]-oMfc.m_uDramBBaseAddr)>>11);

			oMfc.m_uFrameAddrC[uProcessIdx][i] = uDramADpbBufStAddr + uChromaPlaneSZ*i;
			mfcOutp32(MFC_COMMON_BASE_ADDR+i*4, (oMfc.m_uFrameAddrC[uProcessIdx][i]-oMfc.m_uDramABaseAddr)>>11);
			
			mfcDbg(("FrameAddrY[%d]:0x%x\n", i, oMfc.m_uFrameAddrY[uProcessIdx][i]));
			mfcDbg(("FrameAddrC[%d]:0x%x\n", i, oMfc.m_uFrameAddrC[uProcessIdx][i]));	
		}

		// Allocate DPB size
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_LUMA_DPB_SIZE, uLumaPlaneSZ);
		Outp32(oMfc.m_uShardMemBufAddr[0]+ALLOCATED_CHROMA_DPB_SIZE, uChromaPlaneSZ);
	}

	if(oMfc.m_eCodecMode[uProcessIdx] == H264_DEC)
	{
		oMfc.m_uADramDpbBufEndAddr[uProcessIdx]=oMfc.m_uFrameAddrC[uProcessIdx][0]+uChromaPlaneSZ*uNumOfDpb;
		oMfc.m_uBDramDpbBufEndAddr[uProcessIdx]=oMfc.m_uFrameAddrY[uProcessIdx][0]+(uLumaPlaneSZ+uMvPlaneSZ)*uNumOfDpb;
	}
	else
	{
		oMfc.m_uADramDpbBufEndAddr[uProcessIdx]=oMfc.m_uFrameAddrC[uProcessIdx][0]+uChromaPlaneSZ*uNumOfDpb;
		oMfc.m_uBDramDpbBufEndAddr[uProcessIdx]=oMfc.m_uFrameAddrY[uProcessIdx][0]+uLumaPlaneSZ*uNumOfDpb;
	}
}
#endif

void MFC_GetMinNumOfDpbBuf(u32* pNumOfDpbBuf)
{
	u32 uNumOfDpb;

	mfcInp32(MFC_SI_MIN_NUB_DPB, uNumOfDpb);
	
	Disp("Num Of Dpb : %d\n", uNumOfDpb);
	oMfc.m_uNumOfDpb = uNumOfDpb;
	*pNumOfDpbBuf = uNumOfDpb;
}

void MFC_SetNumOfTotalDpbBuf(u32 uExtraDpbNum)
{
	oMfc.m_uNumOfDpb = uExtraDpbNum;
	mfcOutp32(MFC_SI_CH0_DPB_CONFIG_CTRL, uExtraDpbNum);
}

void MFC_GetDbpBufEndAddr(u32 uProcessIdx, u32* pADramDpbBufEndAddr, u32* pBDramDpbBufEndAddr)
{
	*pADramDpbBufEndAddr = oMfc.m_uADramDpbBufEndAddr[uProcessIdx];
	*pBDramDpbBufEndAddr = oMfc.m_uBDramDpbBufEndAddr[uProcessIdx];
}

void MFC_SendH2RCmd(MFC_FACADE_CMD eCmd, u32 uArg)
{
	u32 H2RCmd;

	// wait until h2r command register becomes '0'
	do
	{
		mfcInp32(MFC_HOST2RISC_CMD, H2RCmd);
	} while(H2RCmd != 0x0);
	
	mfcOutp32(MFC_HOST2RISC_ARG1, uArg);
	mfcOutp32(MFC_HOST2RISC_CMD, eCmd);
}

void MFC_SendH2RCmdWithFullArg(MFC_FACADE_CMD eCmd, u32 uArg1, u32 uArg2, u32 uArg3, u32 uArg4)
{
	u32 H2RCmd;

	do
	{
		mfcInp32(MFC_HOST2RISC_CMD, H2RCmd);
	} while(H2RCmd != 0x0);
	
	mfcOutp32(MFC_HOST2RISC_ARG1, uArg1);
	mfcOutp32(MFC_HOST2RISC_ARG2, uArg2);
	mfcOutp32(MFC_HOST2RISC_ARG3, uArg3);
	mfcOutp32(MFC_HOST2RISC_ARG4, uArg4);
	mfcOutp32(MFC_HOST2RISC_CMD, eCmd);
}

void MFC_InitBuffer(MFC_CMD_CH_NUM eCmdCh, u32 uProcessIdx)
{
	oMfc.m_bInitBufferDone=false;

	if(eCmdCh == CMD_CH0)
	{
		mfcOutp32(MFC_SI_CH0_HOST_WR_ADR, oMfc.m_uShardMemBufAddr[0]-oMfc.m_uDramABaseAddr);
		mfcOutp32(MFC_SI_CH0_INST_ID, (0x4<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx] ); 
	}
	else
	{
		mfcOutp32(MFC_SI_CH1_HOST_WR_ADR, oMfc.m_uShardMemBufAddr[1]-oMfc.m_uDramABaseAddr);
		mfcOutp32(MFC_SI_CH1_INST_ID, (0x4<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx] ); 
	}
	while(!oMfc.m_bInitBufferDone);
	oMfc.m_bInitBufferDone = false;
}

void MFC_GetDecodedFrmSize(u32* pConsumedbyteSize)
{
	mfcInp32(MFC_SI_DEC_FRM_SIZE, *pConsumedbyteSize)
}
 
void MFC_StartDecodingOneFrameOfTheBuffer(MFC_CMD_CH_NUM eCmdCh)
{
	u32 uSfr;
	u32 uOneFrameSZ;
	
	oMfc.m_bFrameDone[oMfc.m_uProcessIdx]=false;
	oMfc.m_bSliceDone[oMfc.m_uProcessIdx] =false;
	oMfc.m_bFatalErr = false;

	if(eCmdCh == CMD_CH0)
	{
		mfcOutp32(MFC_SI_CH0_HOST_WR_ADR, oMfc.m_uShardMemBufAddr[0]-oMfc.m_uDramABaseAddr);
		mfcOutp32(MFC_SI_CH0_RELEASE_BUFFER, 0xffffffff);
		mfcInp32(MFC_SI_CH0_ES_DEC_UNIT_SIZE, uOneFrameSZ);
		mfcOutp32(MFC_SI_CH0_ES_ADDR, (oMfc.m_uCpbBaseAddr[0]-oMfc.m_uDramABaseAddr)>>11);
		mfcOutp32(MFC_SI_CH0_DESC_ADDR, (oMfc.m_uDescBaseAddr[0]-oMfc.m_uDramABaseAddr)>>11);	
		mfcOutp32(MFC_SI_CH0_ES_DEC_UNIT_SIZE, uOneFrameSZ);
		mfcOutp32(MFC_SI_CH0_CPB_SIZE, CPB_BUF_SIZE);
		mfcOutp32(MFC_SI_CH0_DESC_SIZE, DESC_BUF_SIZE);
		mfcOutp32(MFC_SI_CH0_INST_ID, (0x2<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx] );
	}
	else
	{
		mfcOutp32(MFC_SI_CH1_HOST_WR_ADR, oMfc.m_uShardMemBufAddr[1]-oMfc.m_uDramABaseAddr);
		mfcOutp32(MFC_SI_CH1_RELEASE_BUFFER, 0xffffffff);
		mfcInp32(MFC_SI_CH1_ES_DEC_UNIT_SIZE, uOneFrameSZ);
		mfcOutp32(MFC_SI_CH1_ES_ADDR, (oMfc.m_uCpbBaseAddr[1]-oMfc.m_uDramABaseAddr)>>11);
		mfcOutp32(MFC_SI_CH1_DESC_ADDR, (oMfc.m_uDescBaseAddr[1]-oMfc.m_uDramABaseAddr)>>11	);	
		mfcOutp32(MFC_SI_CH1_ES_DEC_UNIT_SIZE, uOneFrameSZ);
		mfcOutp32(MFC_SI_CH1_CPB_SIZE, CPB_BUF_SIZE);
		mfcOutp32(MFC_SI_CH1_DESC_SIZE, DESC_BUF_SIZE);
		mfcOutp32(MFC_SI_CH1_INST_ID, (0x2<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx] );
	}		
}

void MFC_StartDecodingOneFrame(MFC_CMD_CH_NUM eCmdCh, u32 uOneFrameSZ, MFC_DEC_TYPE eDecType)
{
	u32 uSfr;
	
	oMfc.m_bFrameDone[oMfc.m_uProcessIdx]=false;
	oMfc.m_bSliceDone[oMfc.m_uProcessIdx] =false;
	oMfc.m_bFatalErr = false;
	
	if(eCmdCh == CMD_CH0)
	{
		mfcOutp32(MFC_SI_CH0_HOST_WR_ADR, oMfc.m_uShardMemBufAddr[0]-oMfc.m_uDramABaseAddr);
		mfcOutp32(MFC_SI_CH0_RELEASE_BUFFER, 0xffffffff);
		mfcOutp32(MFC_SI_CH0_ES_ADDR, (oMfc.m_uCpbBaseAddr[0]-oMfc.m_uDramABaseAddr)>>11);
		mfcOutp32(MFC_SI_CH0_DESC_ADDR, (oMfc.m_uDescBaseAddr[0]-oMfc.m_uDramABaseAddr)>>11);	
		mfcOutp32(MFC_SI_CH0_ES_DEC_UNIT_SIZE, uOneFrameSZ);
		mfcOutp32(MFC_SI_CH0_CPB_SIZE, CPB_BUF_SIZE);
		mfcOutp32(MFC_SI_CH0_DESC_SIZE, DESC_BUF_SIZE);
		mfcOutp32(MFC_SI_CH0_INST_ID, (eDecType<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx]);
	}
	else
	{
		mfcOutp32(MFC_SI_CH1_HOST_WR_ADR, oMfc.m_uShardMemBufAddr[1]-oMfc.m_uDramABaseAddr);
		mfcOutp32(MFC_SI_CH1_RELEASE_BUFFER, 0xffffffff);
		mfcOutp32(MFC_SI_CH1_ES_ADDR, (oMfc.m_uCpbBaseAddr[1]-oMfc.m_uDramABaseAddr)>>11);
		mfcOutp32(MFC_SI_CH1_DESC_ADDR, (oMfc.m_uDescBaseAddr[1]-oMfc.m_uDramABaseAddr)>>11	);	
		mfcOutp32(MFC_SI_CH1_ES_DEC_UNIT_SIZE, uOneFrameSZ);
		mfcOutp32(MFC_SI_CH1_CPB_SIZE, CPB_BUF_SIZE);
		mfcOutp32(MFC_SI_CH1_DESC_SIZE, DESC_BUF_SIZE);
		mfcOutp32(MFC_SI_CH1_INST_ID, (eDecType<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx]);
	}		
}

void MFC_GetErrorType(void)
{
	u32 uErrorType;
	mfcInp32(0x004c, uErrorType);

#if 1
	//if((uErrorType>=1) && (uErrorType>=124))
	if((uErrorType>=0) && (uErrorType>=124))
	{
		oMfc.m_bFatalErr = true;
		Disp("[Fatal Error\n]");
	}
	else if((uErrorType>=151) && (uErrorType>=181))
	{
		oMfc.m_bFatalErr = false;
		Disp("[Warning\n]");
		Assert(0);
	}
	else
		Assert(0);
#else
#endif
}

u8 MFC_IsDecodingFinished(void)
{
	u32 x;

	// check buffer flush.
	mfcInp32(MFC_SI_DISPLAY_STATUS, x);

	if( (x & 0x3) !=0x3 )	
		return false;
	else
		return true;
}

void MFC_GetPicResolution(void)
{
	mfcInp32(MFC_SI_HRESOL, oMfc.m_uPicHsz[oMfc.m_uProcessIdx]);
	mfcInp32(MFC_SI_VRESOL, oMfc.m_uPicVsz[oMfc.m_uProcessIdx]);
}

void MFC_GetPictureType(MFC_PIC_TYPE *ePicType, u8* bInterlace)
{
	u32 uPicType;
	u32 uDispStatus;
	
	mfcInp32(MFC_SI_FRAME_TYPE, uPicType);
	*ePicType = (MFC_PIC_TYPE)uPicType;

	mfcInp32(MFC_SI_DISPLAY_STATUS, uDispStatus);
	if(uDispStatus&(1<<2))
		*bInterlace = true;
	//	Disp("[INTERLACE]");
}

void MFC_GetDispType(MFC_DISP_TYPE *eDispType)
{
	u32 uDispStatus;

	mfcInp32(MFC_SI_DISPLAY_STATUS, uDispStatus);
	uDispStatus = uDispStatus&0x3;

	if(uDispStatus == 0)
		*eDispType = DEC_ONLY;
	else if(uDispStatus == 0x1)
		*eDispType = DEC_AND_DISP;
	else if(uDispStatus == 0x2)
		*eDispType = DISP_ONLY;
	else
		*eDispType = DPB_EMPTY;
}

void MFC_GetDispInfo(MFC_DISP_TYPE *eDispType, RESOL_CHANGE *eResChange, u8* bProgressive)
{
	u32 uDispStatus;
	u32 uDispStatusReg;
	u32 uResInfo;

	mfcInp32(MFC_SI_DISPLAY_STATUS, uDispStatusReg);

	// check display status
	uDispStatus = uDispStatusReg & 0x3;
	
	if(uDispStatus == 0)
		*eDispType = DEC_ONLY;
	else if(uDispStatus == 0x1)
		*eDispType = DEC_AND_DISP;
	else if(uDispStatus == 0x2)
		*eDispType = DISP_ONLY;
	else
		*eDispType = DPB_EMPTY;

	// check whether resolution change or not
	uResInfo = (uDispStatusReg>>4) & 0x3;
	
	if(uResInfo == 1)
		*eResChange = RESOL_INCR;
	else if(uResInfo == 2)
		*eResChange = RESOL_DECR;
	else
		*eResChange = NO_CHANGE;

	if((uDispStatusReg>>3) & 0x1)
		*bProgressive = false;
	else
		*bProgressive = true;
}

u8 MFC_WaitForFrameDone()
{
	u32 i=false;
	u32 a;
	u32 x;
	u32 uRtnChId;
	u32 uRiscRetVal;
	
	while(!(oMfc.m_bFrameDone[oMfc.m_uProcessIdx]||oMfc.m_bError
		||oMfc.m_bSliceDone[oMfc.m_uProcessIdx]||oMfc.m_bEncComplete[oMfc.m_uProcessIdx]
		||oMfc.m_bEdfuInt[oMfc.m_uProcessIdx]));

	if((oMfc.m_bFrameDone[oMfc.m_uProcessIdx] == true)||(oMfc.m_bSliceDone[oMfc.m_uProcessIdx] == true)||
		(oMfc.m_bEncComplete[oMfc.m_uProcessIdx] == true)|| (oMfc.m_bEdfuInt[oMfc.m_uProcessIdx] == true))
		x = true;
	else if((oMfc.m_bError==true) && (oMfc.m_bFatalErr == false))
		x = false;

	oMfc.m_bFrameDone[oMfc.m_uProcessIdx]=false;
	oMfc.m_bSliceDone[oMfc.m_uProcessIdx]=false;
	oMfc.m_bEdfuInt[oMfc.m_uProcessIdx]=false;
	oMfc.m_bError=false;
	
	return x;
}

void MFC_SetSleepOrWakeUpMode(MFC_FACADE_CMD eH2RCmd)
{
	u32 uHost2RiscCmd;

	if(eH2RCmd == H2R_CMD_SLEEP)
	{
		oMfc.m_bSleepDone=false;
		mfcOutp32(MFC_HOST2RISC_CMD, eH2RCmd);
		while(!oMfc.m_bSleepDone);
		oMfc.m_bSleepDone=false;
	}
	else if(eH2RCmd == H2R_CMD_WAKEUP)
	{ 
		// wake up cmd => Risc release => Wait for wakeup return sig.
		oMfc.m_bWakeupDone=false;
		mfcOutp32(MFC_HOST2RISC_CMD, eH2RCmd);
		mfcOutp32(MFC_SW_RESET, 0x3ff);

		while(!oMfc.m_bWakeupDone);
		oMfc.m_bWakeupDone=false;
	}
}

void MFC_CloseInstance(u32 uInstanceId)
{
	u32 uRisc2Host;
	u32 uRtnInstanceId;

	oMfc.m_bCloseInstance=false;
	MFC_SendH2RCmd(H2R_CMD_CLOSE_INSTANCE, uInstanceId);
	 
	while(!oMfc.m_bCloseInstance);
	oMfc.m_bCloseInstance=false; 
}

void MFC_SetProcessIdx(u32 uProcessIdx)
{
	oMfc.m_uProcessIdx = uProcessIdx;
}

void MFC_GetInstanceId(u32* pInstanceId, u32* pProcessIdx)
{
	*pInstanceId = oMfc.m_uInstanceId[oMfc.m_uProcessIdx];
	*pProcessIdx = oMfc.m_uProcessIdx;
}

void MFC_GetProcessIdxFromInstId(u32 uInstanceId, u32* pProcessIdx)
{
	u32 i = 0;

	Assert(uInstanceId!=0xffff);

	while(uInstanceId != oMfc.m_uInstanceId[i])
	{
			i++;
	}
	*pProcessIdx = i;
}

void MFC_GetInstIdFromProcessIdx(u32 uProcessIdx, u32* pInstanceId)
{
	*pInstanceId = oMfc.m_uInstanceId[uProcessIdx];
}

void MFC_GetIntFlag(MFC_FACADE_RESPONSE* eIntFlag)
{
	u32 uRegData;
	
	mfcInp32(MFC_RISC2HOST_CMD, uRegData);
	*eIntFlag = (MFC_FACADE_RESPONSE)(uRegData & 0xff);

	switch(*eIntFlag)
	{
		case R2H_CMD_OPEN_INSTANCE_RET:
			oMfc.m_bOpenInstance = true;
			mfcDbg(("[OPEN]\n"));
			//Disp("[OPEN]\n");
			break;
		case R2H_CMD_CLOSE_INSTANCE_RET:
			oMfc.m_bCloseInstance = true;
			mfcDbg(("[CLOSE]\n"));
			break;
		case R2H_CMD_SEQ_DONE_RET:
			oMfc.m_bSeqDone = true;
			mfcDbg(("[SEQ]\n"));
			//Disp("[SEQ]\n");
			break;
		case R2H_CMD_FRAME_DONE_RET:
			oMfc.m_bFrameDone[oMfc.m_uProcessIdx] = true;
			mfcDbg(("[FRAME]\n"));
			//Disp("[FRAME]\n");
			break;
		case R2H_CMD_SLICE_DONE_RET:
			oMfc.m_bSliceDone[oMfc.m_uProcessIdx] =true;
			mfcDbg(("[SLICE]\n"));
			//Disp("[FRAME]\n");
			break;
		case R2H_CMD_ENC_COMPLETE_RET:
			oMfc.m_bEncComplete[oMfc.m_uProcessIdx]=true;
			mfcDbg(("[ENC COMP]\n"));
			//Disp("[ENC COMP]\n");
			break;	
		case R2H_CMD_SYS_INIT_RET:
			oMfc.m_bSysInit = true;
			mfcDbg(("[SYSINIT]\n"));
			//Disp("SYSINIT]\n");
			 break;		
		case R2H_CMD_FW_STATUS_RET:
			oMfc.m_bFwReady = true;
			mfcDbg(("[FWREADY]\n"));
			//Disp("[FWREADY]\n");
			 break;
		case R2H_CMD_SLEEP_RET:
			oMfc.m_bSleepDone = true;
			mfcDbg(("[SLEEP]\n"));
			break;
		case R2H_CMD_WAKEUP_RET:
			oMfc.m_bWakeupDone = true;
			mfcDbg(("[WAKEUP]\n"));
			break;
		case R2H_CMD_INIT_BUFFER_RET:
			oMfc.m_bInitBufferDone = true;
			mfcDbg(("[BUFFER]\n"));
			//Disp("[BUFFER]\n");
			break;
		case R2H_CMD_EDFU_INT_RET:
			oMfc.m_bEdfuInt[oMfc.m_uProcessIdx] = true;
			 break;	
		case R2H_CMD_EMPTY:
			break;
		case R2H_CMD_ERR_RET:
			oMfc.m_bError = true;
			//MFC_GetErrorType();
			Disp("[ERR]");
			break;
		default:
			break;	
	}
}

u8 MFC_IsEncodingFinished(void)
{
	if(oMfc.m_bEncComplete[oMfc.m_uProcessIdx]==true)
		return true;
	else
		return false;
	oMfc.m_bEncComplete[oMfc.m_uProcessIdx] =  false;
}

u8 MFC_GetPreparedChAndCpbBuf(MFC_CMD_CH_NUM* pChId, u32* pCpbBuf)
{
	u32 i;
	u32 uCh1IdReg;
	u32 uCh2IdReg;

	mfcInp32(MFC_SI_CH0_INST_ID, uCh1IdReg);
	uCh1IdReg = uCh1IdReg&0xffff;
	mfcInp32(MFC_SI_CH1_INST_ID, uCh2IdReg);
	uCh2IdReg = uCh2IdReg&0xffff;

	if(uCh1IdReg == 0xffff)
	{
		*pChId = CMD_CH0;
		*pCpbBuf = oMfc.m_uCpbBaseAddr[0];

		return true;
	}
	else if(uCh2IdReg == 0xffff)
	{
		*pChId = CMD_CH1;
		*pCpbBuf = oMfc.m_uCpbBaseAddr[1];

		return true;
	}
	else
		return false;
}

// Get video real img picture size
void MFC_GetImagePicSize(u32* pPicX, u32* pPicY)
{
	*pPicX = oMfc.m_uPicHsz[oMfc.m_uProcessIdx];
	*pPicY = oMfc.m_uPicVsz[oMfc.m_uProcessIdx];
}

void MFC_GetImageFrameSize(u32* pWidth, u32* pHeight)
{
	*pWidth = (oMfc.m_uPicHsz[oMfc.m_uProcessIdx]+15)/16*16; //Fimc 16's multiple alignment
	*pHeight = oMfc.m_uPicVsz[oMfc.m_uProcessIdx];
	//*pHeight =(oMfc.m_uPicVsz[oMfc.m_uProcessIdx]+15)/16*16;
}

void MFC_SetCmdChNum(MFC_CMD_CH eCmcChNum)
{
	if(eCmcChNum == ONE_CH_CMD)
		oMfc.m_uCmdChType = ONE_CH_CMD;
	else 
		oMfc.m_uCmdChType = TWO_CH_CMD;
}

u8 MFC_GetCodecType(char* fileExt, VIDEO_CODEC_TYPE* eCodecMode)
{
	char pCh[10];
	int i=-1;
	// change alphabet lower case
	do
	{
		i++;
		if(fileExt[i]>='A' && fileExt[i]<='Z')
			pCh[i] = fileExt[i]+32;
		else
			pCh[i] = fileExt[i];
	}
	while (pCh[i]);
	
	if( (strcmp(pCh, "m4v")==0)||(strcmp(pCh, "mp4")==0)||(strcmp(pCh, "mpeg4")==0)||(strcmp(pCh, "bits")==0))
		*eCodecMode = MPEG4_DEC;
	else if( (strcmp(pCh, "264")==0)||(strcmp(pCh, "26l")==0)||(strcmp(pCh, "h264")==0))
		*eCodecMode = H264_DEC;
	else if( (strcmp(pCh, "dvx")==0)||(strcmp(pCh, "divx")==0))
		*eCodecMode = DivX503_DEC;	
#if 0
	else if( (strcmp(pCh, "dvx502")==0)||(strcmp(pCh, "divx502")==0))
		*eCodecMode = DivX502_DEC;
	else if( (strcmp(pCh, "dvx503")==0)||(strcmp(pCh, "divx503")==0))
		*eCodecMode = DivX503_DEC;
#endif

	else if( (strcmp(pCh, "xvd")==0)||(strcmp(pCh, "xvid")==0))
		*eCodecMode = XviD_DEC;
	else if( (strcmp(pCh, "dvx3")==0))
		*eCodecMode = DivX311_DEC;	
	else if( (strcmp(pCh, "dvx4")==0))
		*eCodecMode = DivX412_DEC;	
	else if( (strcmp(pCh, "263")==0)||(strcmp(pCh, "msv")==0))
		*eCodecMode = H263_DEC;
	else if( (strcmp(pCh, "m2v")==0) || (strcmp(pCh, "mp2")==0) || (strcmp(pCh, "mpg")==0) || (strcmp(pCh, "mpeg")==0) || (strcmp(pCh, "mpeg2")==0) || (strcmp(pCh, "es")==0) )
		*eCodecMode = MPEG2_DEC;
	else if( (strcmp(pCh, "rcv")==0) || (strcmp(pCh, "wmv")==0))
		*eCodecMode = VC1RCV_DEC;
	else if	 ( (strcmp(pCh, "vc1")==0))
		*eCodecMode = VC1AP_DEC;
	else
		return false;

	return true;
}

void MFC_SetDivX311ImgSize(u32 Hsize, u32 Vsize)
{
	u32 uMbWidth;
	u32 uMbHeight;

	mfcOutp32(MFC_SI_CH0_DIVX311_HRESOL, Hsize);
	mfcOutp32(MFC_SI_CH0_DIVX311_VRESOL, Vsize);  
}

void MFC_ReadFwVersion(u32 *uFwVersion)
{
	u32 uVersion;

	mfcInp32(MFC_FW_VERSION, uVersion);

	*uFwVersion = uVersion &0x00ffffff;
}

void MFC_SetBufAddrOfEncoder(u32 uEncABufStAddr, u32 uEncBBufStAddr)
{
	u32 i;
	u32 uEncDpbAddrOfY0;
	u32 uEncDpbAddrOfY1;
	u32 uEncDpbAddrOfY2;
	u32 uEncDpbAddrOfY3;
	u32 uEncDpbAddrOfC0;
	u32 uEncDpbAddrOfC1;
	u32 uEncDpbAddrOfC2;
	u32 uEncDpbAddrOfC3;
	u32 uUpMvAddr;
	u32 uCoZeroFlagAddr;
	u32 uUpIntraMdAddr;
	u32 uUpIntraPredAddr;
	u32 uNborInfoMpEncAddr;
	u32 uStreamBufAddr;
	u32 uCurImgSrcAddr;
	u32 uEncDpbAddrOfY0Reg;
	u32 uEncDpbAddrOfY1Reg;
	u32 uEncDpbAddrOfY2Reg;
	u32 uEncDpbAddrOfY3Reg;
	u32 uEncDpbAddrOfC0Reg;
	u32 uEncDpbAddrOfC1Reg;
	u32 uEncDpbAddrOfC2Reg;
	u32 uEncDpbAddrOfC3Reg;
	u32 uUpMvAddrReg;
	u32 uCoZeroFlagAddrReg;
	u32 uUpIntraMdAddrReg;
	u32 uUpIntraPredAddrReg;
	u32 uNborInfoMpEncAddrReg;
	u32 uEncABufEndAddr;
	u32 uEncBBufEndAddr;
	u32 uEncDpbAddrOfY1BufEndAddr;	
	u32 uLumaPlaneSz;
	u32 uChromaPlaneSz;

	u32 uRecBLumaAddrReg;
	u32 uRecBChromaAddrReg;
	u32 uRecBLumaAddr;
	u32 uRecBChromaAddr;

	uEncDpbAddrOfY0Reg = MFC_COMMON_BASE_ADDR+7*4;
	uEncDpbAddrOfY1Reg = MFC_COMMON_BASE_ADDR+8*4;
	uEncDpbAddrOfY2Reg = MFC_COMMON_BASE_ADDR+68*4;
	uEncDpbAddrOfY3Reg = MFC_COMMON_BASE_ADDR+69*4;
	uEncDpbAddrOfC0Reg = MFC_COMMON_BASE_ADDR+64*4;
	uEncDpbAddrOfC1Reg = MFC_COMMON_BASE_ADDR+65*4;
	uEncDpbAddrOfC2Reg = MFC_COMMON_BASE_ADDR+66*4;
	uEncDpbAddrOfC3Reg = MFC_COMMON_BASE_ADDR+67*4;
	uUpMvAddrReg 	   = MFC_COMMON_BASE_ADDR+0*4;
	uCoZeroFlagAddrReg = MFC_COMMON_BASE_ADDR+4*4;
	uUpIntraMdAddrReg 	= MFC_COMMON_BASE_ADDR+2*4;
	uUpIntraPredAddrReg = MFC_COMMON_BASE_ADDR+80*4;
	uNborInfoMpEncAddrReg = MFC_COMMON_BASE_ADDR+1*4; 	// At h.263. ACDC_COEF_BASE_ADDR
	uRecBLumaAddrReg = MFC_COMMON_BASE_ADDR+11*4; 		// reconstruction B-picutre luma address register
	uRecBChromaAddrReg = MFC_COMMON_BASE_ADDR+12*4; 	// reconstruction B-picture chroma address register

#if RECON_B_ENABLE
	// A port common region
	oMfc.m_uStreamBufAddr[oMfc.m_uProcessIdx] = uEncABufStAddr;
	uUpMvAddr = Align(uEncABufStAddr + MFC_STR_U_BUF_SIZE, 2048);
	
	mfcOutp32(uUpMvAddrReg, ((uUpMvAddr-oMfc.m_uDramABaseAddr)>>11));
	uCoZeroFlagAddr = uUpMvAddr + MFC_UPPER_MV_BUF_SIZE;

	mfcOutp32(uCoZeroFlagAddrReg, ((uCoZeroFlagAddr-oMfc.m_uDramABaseAddr)>>11));
	uUpIntraMdAddr = uCoZeroFlagAddr + MFC_DIRECT_COZERO_FLAG_BUF_SIZE;

	mfcOutp32(uUpIntraMdAddrReg, ((uUpIntraMdAddr-oMfc.m_uDramABaseAddr)>>11));
	uNborInfoMpEncAddr = uUpIntraMdAddr + MFC_UPPER_INTRA_MD_BUF_SIZE;
		
	mfcOutp32(uNborInfoMpEncAddrReg, ((uNborInfoMpEncAddr-oMfc.m_uDramABaseAddr)>>11));
	uRecBLumaAddr = uNborInfoMpEncAddr + MFC_NBOR_INFO_MPENC_BUF_SIZE;

	mfcOutp32(uRecBLumaAddrReg, ((uRecBLumaAddr-oMfc.m_uDramABaseAddr)>>11));
	uRecBChromaAddr = uRecBLumaAddr + MFC_REF_Y_BUF_SIZE;

	mfcOutp32(uRecBChromaAddrReg, ((uRecBChromaAddr-oMfc.m_uDramABaseAddr)>>11));
	uEncDpbAddrOfY0 = uRecBChromaAddr + MFC_REF_C_BUF_SIZE;

	mfcOutp32(uEncDpbAddrOfY0Reg, ((uEncDpbAddrOfY0-oMfc.m_uDramABaseAddr)>>11));
	uEncDpbAddrOfY1 = uEncDpbAddrOfY0 + MFC_REF_Y_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfY1Reg, ((uEncDpbAddrOfY1-oMfc.m_uDramABaseAddr)>>11));
	uEncDpbAddrOfY1BufEndAddr = uEncDpbAddrOfY1 + MFC_REF_Y_BUF_SIZE;

	uEncABufEndAddr=uEncDpbAddrOfY1BufEndAddr;

	// B port common region
	mfcOutp32(uUpIntraPredAddrReg, ((uEncBBufStAddr-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfC0 = uEncBBufStAddr + MFC_UPPER_INTRA_PRED_BUF_SIZE;

	mfcOutp32(uEncDpbAddrOfC0Reg, ((uEncDpbAddrOfC0-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfY2 = uEncDpbAddrOfC0 + MFC_REF_C_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfY2Reg, ((uEncDpbAddrOfY2-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfC1 = uEncDpbAddrOfY2 + MFC_REF_Y_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfC1Reg, ((uEncDpbAddrOfC1-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfC2 = uEncDpbAddrOfC1 + MFC_REF_C_BUF_SIZE;
		
	mfcOutp32(uEncDpbAddrOfC2Reg, ((uEncDpbAddrOfC2-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfY3 = uEncDpbAddrOfC2 + MFC_REF_C_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfY3Reg, ((uEncDpbAddrOfY3-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfC3 = uEncDpbAddrOfY3 + MFC_REF_Y_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfC3Reg, ((uEncDpbAddrOfC3-oMfc.m_uDramBBaseAddr)>>11));
	uCurImgSrcAddr = uEncDpbAddrOfC3 + MFC_REF_C_BUF_SIZE;

#else  // B recon disable
	// A port common region
	oMfc.m_uStreamBufAddr[oMfc.m_uProcessIdx] = uEncABufStAddr;
	uUpMvAddr = Align(uEncABufStAddr + MFC_STR_U_BUF_SIZE, 2048);
	
	mfcOutp32(uUpMvAddrReg, ((uUpMvAddr-oMfc.m_uDramABaseAddr)>>11));
	uCoZeroFlagAddr = uUpMvAddr + MFC_UPPER_MV_BUF_SIZE;

	mfcOutp32(uCoZeroFlagAddrReg, ((uCoZeroFlagAddr-oMfc.m_uDramABaseAddr)>>11));
	uUpIntraMdAddr = uCoZeroFlagAddr + MFC_DIRECT_COZERO_FLAG_BUF_SIZE;

	mfcOutp32(uUpIntraMdAddrReg, ((uUpIntraMdAddr-oMfc.m_uDramABaseAddr)>>11));
	uNborInfoMpEncAddr = uUpIntraMdAddr + MFC_UPPER_INTRA_MD_BUF_SIZE;
		
	mfcOutp32(uNborInfoMpEncAddrReg, ((uNborInfoMpEncAddr-oMfc.m_uDramABaseAddr)>>11));
	uEncDpbAddrOfY0 = uNborInfoMpEncAddr + MFC_NBOR_INFO_MPENC_BUF_SIZE;

	mfcOutp32(uEncDpbAddrOfY0Reg, ((uEncDpbAddrOfY0-oMfc.m_uDramABaseAddr)>>11));
	uEncDpbAddrOfY1 = uEncDpbAddrOfY0 + MFC_REF_Y_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfY1Reg, ((uEncDpbAddrOfY1-oMfc.m_uDramABaseAddr)>>11));
	uEncDpbAddrOfY1BufEndAddr = uEncDpbAddrOfY1 + MFC_REF_Y_BUF_SIZE;

	uEncABufEndAddr=uEncDpbAddrOfY1BufEndAddr;

	// B port common region
	mfcOutp32(uUpIntraPredAddrReg, ((uEncBBufStAddr-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfC0 = uEncBBufStAddr + MFC_UPPER_INTRA_PRED_BUF_SIZE;

	mfcOutp32(uEncDpbAddrOfC0Reg, ((uEncDpbAddrOfC0-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfY2 = uEncDpbAddrOfC0 + MFC_REF_C_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfY2Reg, ((uEncDpbAddrOfY2-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfC1 = uEncDpbAddrOfY2 + MFC_REF_Y_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfC1Reg, ((uEncDpbAddrOfC1-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfC2 = uEncDpbAddrOfC1 + MFC_REF_C_BUF_SIZE;
		
	mfcOutp32(uEncDpbAddrOfC2Reg, ((uEncDpbAddrOfC2-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfY3 = uEncDpbAddrOfC2 + MFC_REF_C_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfY3Reg, ((uEncDpbAddrOfY3-oMfc.m_uDramBBaseAddr)>>11));
	uEncDpbAddrOfC3 = uEncDpbAddrOfY3 + MFC_REF_Y_BUF_SIZE;
	
	mfcOutp32(uEncDpbAddrOfC3Reg, ((uEncDpbAddrOfC3-oMfc.m_uDramBBaseAddr)>>11));
	uCurImgSrcAddr = uEncDpbAddrOfC3 + MFC_REF_C_BUF_SIZE;
#endif

	uCurImgSrcAddr = Align(uCurImgSrcAddr, 8*1024);
	
	MFC_CalcTiled4x2PlaneSize(oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth, 
							oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight, &uLumaPlaneSz);
	MFC_CalcTiled4x2PlaneSize(oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth, 
							oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight/2, &uChromaPlaneSz);

	for(i=0; i<4; i++)
	{
		oMfc.m_uCurrLumaAddr[oMfc.m_uProcessIdx][i] = uCurImgSrcAddr + (uLumaPlaneSz+uChromaPlaneSz)*i;
		oMfc.m_uCurrChromaAddr[oMfc.m_uProcessIdx][i] = uCurImgSrcAddr + (uLumaPlaneSz+uChromaPlaneSz)*i + uLumaPlaneSz;
		oMfc.m_uCurrAddrState[oMfc.m_uProcessIdx][i] = 0xffffffff;
	}

	uEncBBufEndAddr = uCurImgSrcAddr + (uLumaPlaneSz+uChromaPlaneSz)*4;
	
	oMfc.m_uEncABufEndAddr = uEncABufEndAddr;
	oMfc.m_uEncBBufEndAddr = uEncBBufEndAddr;
}

void MFC_GetEncBufEndAddr(u32* pEncABufAddr, u32* pEncBBufAddr)
{
	*pEncABufAddr = oMfc.m_uEncABufEndAddr;
	*pEncBBufAddr = oMfc.m_uEncBBufEndAddr;
}

void MFC_SelectEncStandard(u32 uProcessIdx, VIDEO_CODEC_TYPE *eEncStandard)
{
	s32 uSel;
	
	Disp("Enter the Encoding mode (1:H.264_ENC[D],  2:MPEG4_ENC,  3:H263_ENC) ? ");
	uSel = UART_GetIntNum();
	Disp("\n");
	
	*eEncStandard = (uSel==3) ? (H263_ENC) :
				(uSel==2) ? (MPEG4_ENC) : (H264_ENC);
	oMfc.m_eCodecMode[uProcessIdx] = *eEncStandard;
	oMfc.m_uProcessIdx = uProcessIdx;
}

void MFC_SetCodecStandard(u32 uProcessIdx, VIDEO_CODEC_TYPE eStandard)
{
	oMfc.m_eCodecMode[uProcessIdx] = eStandard;
	oMfc.m_uProcessIdx = uProcessIdx;
}

void MFC_ParseConfigFile(u32 uConfigFileBufAddr)
{
	char* pConfigFileStr = (char*)uConfigFileBufAddr;
	char *param_line;
			
	//Disp( "Parsing Config File...START\n");
	
	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H264_ENC)
	{
		char *param_arr[MAX_PARAM_NUM];
		char *h264_para[512];
		u32 uPara_Count;
		u32 i, j=0;

		for(i=0 ; i<MAX_PARAM_NUM ; i++)
			param_arr[i] = (char *)malloc(MAX_MSG_SIZE);

		i = 0;
		h264_para[i] = strtok(pConfigFileStr, "\r\n");
		
		while(1)
		{	
			if(j != 0)
				h264_para[i] = strtok(NULL, "\r\n");
			if(h264_para[i]== NULL)
				break;
			j++;
			if((h264_para[i][0] == ' ')||(h264_para[i][0] == '\t')||(h264_para[i][0] == '#')||(h264_para[i][0] == '\n')||(h264_para[i][0] == '\r'))
				continue;
			i++;
			
		}

		uPara_Count = i-1;

		for (i=0;i<uPara_Count;i++)
		{
			if(Split(&param_arr[0], h264_para[i], "=", '#'))
			{
				Map2H264EncConfig(param_arr);
			}

			if(0==strcmp(param_arr[0], "CompleteSeqParam") ||0==strcmp(param_arr[0], "CompleteFrameParam"))
			{
				Disp("ACK CompleteSeqParam \n");
				break;
			}		
		}

		for(i=0 ; i<MAX_PARAM_NUM ; i++)
			free(param_arr[i]);
			
	}
	
	else if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == MPEG4_ENC)
	{
		char* ptemp;
		char* ptemp2;
		u32 j=0;

		param_line = strtok(pConfigFileStr, "\r\n");

		do
		{		
			if((param_line[0] == '\n')||(param_line[0] == '\r'))
				continue;
				
			ptemp  = strchr(param_line, '\t');
			ptemp2 = strchr(param_line, ' ');
			if(ptemp != NULL)
				*ptemp = NULL;
			if(ptemp2 != NULL)
				*ptemp2 = NULL;
			j++;

			switch (j)
			{
				case 16:	strcpy(oMpeg4ConfigInput.ucInputFileName, param_line);			break;
				case 5:		oMpeg4ConfigInput.m_uProfileLevel			=	atoi(param_line) ;	break;
				case 7:		oMpeg4ConfigInput.m_uNumberBFrames			= 	atoi(param_line) ;	break;
				case 8:		oMpeg4ConfigInput.m_uSourceWidth			= 	atoi(param_line) ;	break;
				case 9:		oMpeg4ConfigInput.m_uSourceHeight			= 	atoi(param_line) ;	break;
				case 123:	oMpeg4ConfigInput.m_uFrameRate				=   atoi(param_line) ;	break;
				case 27:	oMpeg4ConfigInput.m_uFramesToBeEncoded		= 	atoi(param_line) ;	break;
				case 124:	oMpeg4ConfigInput.m_uBitrate				= 	atoi(param_line) ;	break;
				case 42:	oMpeg4ConfigInput.m_uFrameQp				= 	atoi(param_line) ;	break;
				case 43:	oMpeg4ConfigInput.m_uPFrameQp				= 	atoi(param_line) ;	break;
				case 44:	oMpeg4ConfigInput.m_uBFrameQp				= 	atoi(param_line) ;	break;
				case 61:	oMpeg4ConfigInput.m_uDisableQpelME			= 	atoi(param_line) ;	break;
				case 79:	oMpeg4ConfigInput.m_uIDRPeriod				= 	atoi(param_line) ;	break;
				case 96:	oMpeg4ConfigInput.m_uRandomIntraMBRefresh	= 	atoi(param_line) ;	break;
				case 106:	oMpeg4ConfigInput.m_uResyncEnable			= 	atoi(param_line) ;	break;
				case 107:	oMpeg4ConfigInput.m_uResyncType				= 	atoi(param_line) ;	break;
				case 114:	oMpeg4ConfigInput.m_uEnableFRMRateControl	= 	atoi(param_line) ;	break;
				case 115:	oMpeg4ConfigInput.m_uEnableMBRateControl	= 	atoi(param_line) ;	break;
				case 116:	oMpeg4ConfigInput.m_uQSCodeMax				= 	atoi(param_line) ;	break;
				case 117:	oMpeg4ConfigInput.m_uQSCodeMin				= 	atoi(param_line) ;	break;
				case 118:	oMpeg4ConfigInput.m_uCBRPeriodRf			= 	atoi(param_line) ;	break;
				case 119:	oMpeg4ConfigInput.m_uDarkDisable			= 	atoi(param_line) ;	break;
				case 120:	oMpeg4ConfigInput.m_uSmoothDisable			= 	atoi(param_line) ;	break;
				case 121:	oMpeg4ConfigInput.m_uStaticDisable			= 	atoi(param_line) ;	break;
				case 122:	oMpeg4ConfigInput.m_uActivityDisable		= 	atoi(param_line) ;	break;
				case 108:	oMpeg4ConfigInput.m_uSliceArgument1			= 	atoi(param_line) ;	break;
				case 109:	oMpeg4ConfigInput.m_uSliceArgument2			= 	atoi(param_line) ;	break;
				case 125:	oMpeg4ConfigInput.m_uEnableVopTiming		= 	atoi(param_line) ;	break;	
				case 126:	oMpeg4ConfigInput.m_uVopTimeResolution		= 	atoi(param_line) ;	break;
				case 127:	oMpeg4ConfigInput.m_uFrameDelta				= 	atoi(param_line) ;	break;
			}			
		}while((param_line = strtok(NULL, "\r\n")) != NULL);		
	}	
	
	else if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H263_ENC)
	{
		char* ptemp;
		char* ptemp2;
		char* ptemp3;
		int j=0;
		
		param_line = strtok(pConfigFileStr, "\r\n");

		do
		{
			if((param_line[0] == '\n')||(param_line[0] == '\r'))
				continue;
				
			if(j != 5)
			{
				ptemp  = strchr(param_line, '\t');
				ptemp2 = strchr(param_line, ' ');
				if(ptemp != NULL)
					*ptemp = NULL;
				if(ptemp2 != NULL)
					*ptemp2 = NULL;
			}
			else
			{
				ptemp  = strchr(param_line, ' '); *ptemp = NULL;
				oH263ConfigInput.m_uSourceResolution	= 	atoi(param_line) ;
				ptemp = ptemp + 1; 
				ptemp2  = strchr(ptemp, ' '); *ptemp2 = NULL;
				oH263ConfigInput.m_uSourceWidth		=     atoi(ptemp) ;
				ptemp2 = ptemp2+ 1; 
				ptemp3  = strchr(ptemp2, ' '); *ptemp3 = NULL;
				oH263ConfigInput.m_uSourceHeight		=     atoi(ptemp2) ;

			}
			
			j++;

			switch(j)
			{
				case 1:		strcpy(oH263ConfigInput.ucInputFileName, param_line);			break;
				case 6:		oH263ConfigInput.m_uSourceResolution		= 	atoi(param_line) ;	break;
				case 13:	oH263ConfigInput.m_uFrameRate				= 	atoi(param_line) ;	break;
				case 5:		oH263ConfigInput.m_uFramesToBeEncoded		= 	atoi(param_line) ;	break;
				case 10:	oH263ConfigInput.m_uBitrate					=   atoi(param_line) ;	break;
				case 8:		oH263ConfigInput.m_uFrameQp					= 	atoi(param_line) ;	break;
				case 20:	oH263ConfigInput.m_uIDRPeriod				= 	atoi(param_line) ;	break;
				case 38:	oH263ConfigInput.m_uRandomIntraMBRefresh	= 	atoi(param_line) ;	break;
				case 16:	oH263ConfigInput.m_uSliceMode				= 	atoi(param_line) ;	break;
				case 47:	oH263ConfigInput.m_uEnableFRMRateControl	= 	atoi(param_line) ;	break;
				case 48:	oH263ConfigInput.m_uEnableMBRateControl		= 	atoi(param_line) ;	break;
				case 49:	oH263ConfigInput.m_uQSCodeMax				= 	atoi(param_line) ;	break;
				case 50:	oH263ConfigInput.m_uQSCodeMin				= 	atoi(param_line) ;	break;
				case 51:	oH263ConfigInput.m_uCBRPeriodRf				= 	atoi(param_line) ;	break;
			}
		}while((param_line = strtok(NULL, "\r\n")) != NULL);
	}

	//Disp( "Parsing Config File...END\n");
	
}

void MFC_SetConfigParameters(void)
{
	u32 i;
	
	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H264_ENC)
	{		
		strcpy(oMfcEncConfig[oMfc.m_uProcessIdx].m_pInputFile, oH264ConfigInput.infile);
		mfcDbg(("source image[process %d] : %s\n", oMfc.m_uProcessIdx, oMfcEncConfig[oMfc.m_uProcessIdx].m_pInputFile));

		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFramesToBeEncoded		= oH264ConfigInput.no_frames;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth			= oH264ConfigInput.img_width;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight			= oH264ConfigInput.img_height;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uProfileIDC				= oH264ConfigInput.ProfileIDC;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uLevelIDC				= oH264ConfigInput.LevelIDC;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod				= oH264ConfigInput.IDR_period;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberReferenceFrames	= oH264ConfigInput.num_ref_frames;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberRefForPframes	= oH264ConfigInput.NUM_REFS_FOR_P;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceMode				= oH264ConfigInput.SliceMode;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceArgument			= oH264ConfigInput.SliceArgument;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uInterSearch16x8		= oH264ConfigInput.InterSearch16x8;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uInterSearch8x16		= oH264ConfigInput.InterSearch8x16;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uInterSearch8x8			= oH264ConfigInput.InterSearch8x8;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames			= oH264ConfigInput.successive_Bframe;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDirectInferenceFlag	= oH264ConfigInput.directInferenceFlag;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uLoopFilterDisable		= oH264ConfigInput.LFDisableIdc;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_sLoopFilterAlphaC0Offset	= oH264ConfigInput.LFAlphaC0Offset;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_sLoopFilterBetaOffset	= oH264ConfigInput.LFBetaOffset;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSymbolMode				= oH264ConfigInput.symbol_mode;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uPictureInterlace		= oH264ConfigInput.PicInterlace;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uTransform8x8Mode		= oH264ConfigInput.Transform8x8Mode;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDisableHpelME			= oH264ConfigInput.DisableHpelME;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDisableQpelME			= oH264ConfigInput.DisableQpelME;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uMDInterWeightPPS		= oH264ConfigInput.MDInterWeightPPS;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uMDIntraWeightPPS		= oH264ConfigInput.MDIntraWeightPPS;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uRandomIntraMBRefresh	= oH264ConfigInput.RandomIntraMBRefresh;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD4x4Disable		= oH264ConfigInput.IntraMD4x4Disable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD8x8Disable		= oH264ConfigInput.IntraMD8x8Disable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD4x4Weight		= oH264ConfigInput.IntraMD4x4Weight;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD8x8Weight		= oH264ConfigInput.IntraMD8x8Weight;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD16x16Weight		= oH264ConfigInput.IntraMD16x16Weight;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uPadControlOn			= oH264ConfigInput.pad_ctrl_on;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uLumaPadVal				= oH264ConfigInput.luma_pad_val;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uCbPadVal				= oH264ConfigInput.cb_pad_val;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uCrPadVal				= oH264ConfigInput.cr_pad_val;			

		///RC
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uEnableFRMRateControl	= oH264ConfigInput.enablefrmratecontrol;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uEnableMBRateControl	= oH264ConfigInput.enablembratecontrol;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameRate				= oH264ConfigInput.FrameRate;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uBitrate				= oH264ConfigInput.BitRate;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameQp				= oH264ConfigInput.frame_qp;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uBFrameQp				= oH264ConfigInput.b_frame_qp;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uPFrameQp				= oH264ConfigInput.p_frame_qp;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uQSCodeMax				= oH264ConfigInput.qs_code_max;
	  	oMfcEncConfig[oMfc.m_uProcessIdx].m_uQSCodeMin				= oH264ConfigInput.qs_code_min;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uCBRPeriodRf			= oH264ConfigInput.rf;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDarkDisable			= oH264ConfigInput.dark_disable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSmoothDisable			= oH264ConfigInput.smooth_disable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uStaticDisable			= oH264ConfigInput.static_disable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uActivityDisable		= oH264ConfigInput.act_disable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uACDIntraMotionEn		= oH264ConfigInput.acd_intra_motion_en;
	}
	
	else if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == MPEG4_ENC)
	{
		strcpy(oMfcEncConfig[oMfc.m_uProcessIdx].m_pInputFile, oMpeg4ConfigInput.ucInputFileName);
		mfcDbg(("source image[process %d] : %s\n", oMfc.m_uProcessIdx, oMfcEncConfig[oMfc.m_uProcessIdx].m_pInputFile));
    	
    	if(oMpeg4ConfigInput.m_uProfileLevel == 0)
    	{
    		//Disp(" Error> Profile and level indication parameter does not exist. Assumming Simple Profile Level 3\n");
    		oMfcEncConfig[oMfc.m_uProcessIdx].m_uProfileIDC	= 0;
    		oMfcEncConfig[oMfc.m_uProcessIdx].m_uLevelIDC	= 3;    		
    	}
    	else
    	{
    		if(oMpeg4ConfigInput.m_uProfileLevel == 8)
			{
				// SP@L0
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uProfileIDC	= 0;
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uLevelIDC	= 0;
			}
			else if((oMpeg4ConfigInput.m_uProfileLevel&0xf0) == 0)
    		{
    			// simple profile
    			oMfcEncConfig[oMfc.m_uProcessIdx].m_uProfileIDC	= 0;
    			oMfcEncConfig[oMfc.m_uProcessIdx].m_uLevelIDC	= oMpeg4ConfigInput.m_uProfileLevel & 0xf;
    		}
    		else
    		{
    			// Advanced simple profile
    			oMfcEncConfig[oMfc.m_uProcessIdx].m_uProfileIDC	= 1;
    			oMfcEncConfig[oMfc.m_uProcessIdx].m_uLevelIDC	= oMpeg4ConfigInput.m_uProfileLevel & 0xf;
    		}
    	}		
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames			= oMpeg4ConfigInput.m_uNumberBFrames-1 ;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth			= oMpeg4ConfigInput.m_uSourceWidth;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight 			= oMpeg4ConfigInput.m_uSourceHeight;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameRate 				= oMpeg4ConfigInput.m_uFrameRate;  //floating point.?????
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFramesToBeEncoded		= oMpeg4ConfigInput.m_uFramesToBeEncoded+1;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uBitrate 				= oMpeg4ConfigInput.m_uBitrate;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameQp				= oMpeg4ConfigInput.m_uFrameQp;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uPFrameQp				= oMpeg4ConfigInput.m_uPFrameQp;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uBFrameQp				= oMpeg4ConfigInput.m_uPFrameQp;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDisableQpelME			= (!oMpeg4ConfigInput.m_uDisableQpelME)&0x01 ;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod				= oMpeg4ConfigInput.m_uIDRPeriod;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uRandomIntraMBRefresh 	= oMpeg4ConfigInput.m_uRandomIntraMBRefresh;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uResyncEnable			= oMpeg4ConfigInput.m_uResyncEnable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uResyncType				= oMpeg4ConfigInput.m_uResyncType;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uEnableFRMRateControl	= oMpeg4ConfigInput.m_uEnableFRMRateControl;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uEnableMBRateControl	= oMpeg4ConfigInput.m_uEnableMBRateControl;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uQSCodeMax 				= oMpeg4ConfigInput.m_uQSCodeMax;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uQSCodeMin 				= oMpeg4ConfigInput.m_uQSCodeMin;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uCBRPeriodRf			= oMpeg4ConfigInput.m_uCBRPeriodRf;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDarkDisable			= oMpeg4ConfigInput.m_uDarkDisable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSmoothDisable 			= oMpeg4ConfigInput.m_uSmoothDisable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uStaticDisable 			= oMpeg4ConfigInput.m_uStaticDisable;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uActivityDisable		= oMpeg4ConfigInput.m_uActivityDisable;

		oMfcEncConfig[oMfc.m_uProcessIdx].m_uEnableVopTiming 		= oMpeg4ConfigInput.m_uEnableVopTiming;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uVopTimeResolution 		= oMpeg4ConfigInput.m_uVopTimeResolution;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameDelta				= oMpeg4ConfigInput.m_uFrameDelta;

		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uInterSearch16x8		= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uInterSearch8x16		= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uInterSearch8x8			= 1;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD4x4Disable		= 1;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD8x8Disable		= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDisableHpelME			= 0;

		if(oMfcEncConfig[oMfc.m_uProcessIdx].m_uResyncEnable ==0 )
		{
			oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceMode			= 0;
			oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceArgument		= 0;
		}
		else if(oMfcEncConfig[oMfc.m_uProcessIdx].m_uResyncType == 0) 
		{    
			oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceMode			= 1;
			oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceArgument		=oMpeg4ConfigInput.m_uSliceArgument1;
		}
		else
		{    
			oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceMode			= 2;
			oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceArgument		=oMpeg4ConfigInput.m_uSliceArgument2;
		}

		oMfcEncConfig[oMfc.m_uProcessIdx].m_uPictureInterlace			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_sLoopFilterAlphaC0Offset	= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_sLoopFilterBetaOffset		= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSymbolMode					= 0; 
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uTransform8x8Mode			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDirectInferenceFlag		= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uLoopFilterDisable			= 1;

		oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberReferenceFrames		= 1;
		//PList0Reference
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD4x4Weight			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD8x8Weight			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD16x16Weight			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uMDInterWeightPPS			= 300;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uMDIntraWeightPPS			= 170; 

	
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uPadControlOn				= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uLumaPadVal					= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uCbPadVal					= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uCrPadVal					= 0;
			
	}
	
	else if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H263_ENC)
	{	
		strcpy(oMfcEncConfig[oMfc.m_uProcessIdx].m_pInputFile, oH263ConfigInput.ucInputFileName);
		mfcDbg(("source image[process %d] : %s\n", oMfc.m_uProcessIdx, oMfcEncConfig[oMfc.m_uProcessIdx].m_pInputFile));

		switch(oH263ConfigInput.m_uSourceResolution)
		{
			case 1 :  
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth	= 128;
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight	= 96 ;
				break;
			case 2 :	
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth	= 176;
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight	= 144 ;
				break;
			case 3 :
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth	= 352;
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight	= 288 ;
				break;
			case 4 :
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth	= 704;
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight	= 576 ;
				break;
			case 5 :
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth	= 1408;
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight	= 1152 ;
				break;
			case 6 :
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth	= oH263ConfigInput.m_uSourceWidth;
				oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight	= oH263ConfigInput.m_uSourceHeight;
				break;
			default :
				Disp(" image size format ERROR\n");
				break;
		}
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameRate					= oH263ConfigInput.m_uFrameRate;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFramesToBeEncoded			= oH263ConfigInput.m_uFramesToBeEncoded+ 1;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uBitrate 					= oH263ConfigInput.m_uBitrate;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameQp					= oH263ConfigInput.m_uFrameQp;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod					= oH263ConfigInput.m_uIDRPeriod;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uRandomIntraMBRefresh 		= oH263ConfigInput.m_uRandomIntraMBRefresh;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceMode					= oH263ConfigInput.m_uSliceMode;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uEnableFRMRateControl		= oH263ConfigInput.m_uEnableFRMRateControl;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uEnableMBRateControl		= oH263ConfigInput.m_uEnableMBRateControl;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uQSCodeMax 					= oH263ConfigInput.m_uQSCodeMax;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uQSCodeMin 					= oH263ConfigInput.m_uQSCodeMin;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uCBRPeriodRf				= oH263ConfigInput.m_uCBRPeriodRf;

		oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames				= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uInterSearch16x8			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uInterSearch8x16			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uInterSearch8x8				= 1;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD4x4Disable			= 1;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD8x8Disable			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDisableHpelME				= 1;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDisableQpelME				= 1; 

		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSliceArgument				= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDarkDisable 				= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSmoothDisable 				= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uStaticDisable 				= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uActivityDisable			= 0;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uProfileIDC					= 66;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uLevelIDC					= 40;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uPictureInterlace			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_sLoopFilterAlphaC0Offset	= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_sLoopFilterBetaOffset		= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uSymbolMode					= 0;
		
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uTransform8x8Mode			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uDirectInferenceFlag		= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uLoopFilterDisable			= 1;

		oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberReferenceFrames		= 1;
		//PList0Reference
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD4x4Weight			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD8x8Weight			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uIntraMD16x16Weight			= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uMDInterWeightPPS			= 300;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uMDIntraWeightPPS			= 170;

		oMfcEncConfig[oMfc.m_uProcessIdx].m_uPadControlOn				= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uLumaPadVal					= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uCbPadVal					= 0;
		oMfcEncConfig[oMfc.m_uProcessIdx].m_uCrPadVal					= 0;
	}
	else
	{
		Disp("Wrong Encoding Format!!!\n");
	}	
}

void MFC_CalculateTotalFramesToEnc(void)
{
	u32 uIdrPeriod = oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod;
	u32 uNumOfFrames = oMfcEncConfig[oMfc.m_uProcessIdx].m_uFramesToBeEncoded;
	u32 uNumOfBframes = oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames;
	u32 uTotalFrames;

	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx]== H264_ENC)
	{
		if(uIdrPeriod== 0)
		{
			uTotalFrames = uNumOfFrames + (uNumOfFrames-1)*uNumOfBframes;
		}
		else
		{
			u32 uFramesInGOP = uIdrPeriod + (uIdrPeriod-1)*uNumOfBframes;
				
			uTotalFrames = (uNumOfFrames/uIdrPeriod)*uFramesInGOP;

			if((uNumOfFrames%uIdrPeriod) > 0)
			{
				uTotalFrames += uNumOfFrames%uIdrPeriod + (uNumOfFrames%uIdrPeriod-1)*uNumOfBframes;
			}
		}
	}
	else if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == MPEG4_ENC)
	{
		if(uIdrPeriod == 0)
		{
			uTotalFrames = 1 + ((uNumOfFrames-1)/(1+uNumOfBframes))*(1+uNumOfBframes);	// multiple of P_Interval
		}
		else
		{
			u32 uFramesInGOP  = uIdrPeriod + uIdrPeriod*uNumOfBframes;
			uTotalFrames = 1 + ((uNumOfFrames-1)/uFramesInGOP) * uFramesInGOP;
			if(uTotalFrames < uNumOfFrames)
				uTotalFrames += ((uNumOfFrames - uTotalFrames)/(uNumOfBframes +1))*(uNumOfBframes+1);
		}
	}
	else		// H263_ENC
	{
		uTotalFrames = uNumOfFrames;
	}

	oMfcEncConfig[oMfc.m_uProcessIdx].m_uTotalFramesToBeEncoded = uTotalFrames;
}

void MFC_CreateInstance(u32 uProcessIdx, VIDEO_CODEC_TYPE eCodec, u8 bPixelCacheEn, u32* pInstanceId)
{
	u32 uArg, uInstanceId;
	u32 uRisc2Host;
	u32 uPixelCache;
	
	oMfc.m_bOpenInstance=false;
	// send open instance command to RISC processor
	uArg = (oMfc.m_eCodecMode[uProcessIdx]==H264_DEC) ? 0:
				 (oMfc.m_eCodecMode[uProcessIdx]==VC1AP_DEC) ? 1:
				 ((oMfc.m_eCodecMode[uProcessIdx]==MPEG4_DEC)||
				 (oMfc.m_eCodecMode[uProcessIdx]==XviD_DEC)) ? 2:
				 ((oMfc.m_eCodecMode[uProcessIdx]==MPEG1_DEC)||
				 (oMfc.m_eCodecMode[uProcessIdx]==MPEG2_DEC)) ? 3:
				 (oMfc.m_eCodecMode[uProcessIdx]==H263_DEC) ? 4:
				 (oMfc.m_eCodecMode[uProcessIdx]==VC1RCV_DEC) ? 5:
				 (oMfc.m_eCodecMode[uProcessIdx]==DivX311_DEC) ? 6:
				 (oMfc.m_eCodecMode[uProcessIdx]==DivX412_DEC) ? 7:
				 (oMfc.m_eCodecMode[uProcessIdx]==DivX502_DEC) ? 8: // DivX502_DEC(Ver 5.00, 5.01, 5.02)
				 (oMfc.m_eCodecMode[uProcessIdx]==DivX503_DEC) ? 9:	  // DivX503_DEC(Ver 5.03 and upper)
				 (oMfc.m_eCodecMode[uProcessIdx]==H264_ENC) ? 16:
				 (oMfc.m_eCodecMode[uProcessIdx]==MPEG4_ENC) ? 17:
			 	 (oMfc.m_eCodecMode[uProcessIdx]==H263_ENC) ? 18: Assert(0);

	// Enable pixel cache : dec case(default p picture only)
	uPixelCache = (bPixelCacheEn == true) ? 0: 3;
	
	MFC_SendH2RCmdWithFullArg(H2R_CMD_OPEN_INSTANCE, uArg, uPixelCache, 0, 0);
	
	while(!oMfc.m_bOpenInstance);
	oMfc.m_bOpenInstance=false; 

	mfcInp32(MFC_RISC2HOST_ARG1, uInstanceId);

	if(uInstanceId == 255 || uInstanceId  >=  16 ) 
		Disp("Instance ID error\n");

	oMfc.m_uInstanceId[uProcessIdx] = uInstanceId;
	oMfc.m_uProcessIdx=uProcessIdx;
	*pInstanceId = uInstanceId;
}

void MFC_InitProcessForEncoding(u32 uProcessIdx, VIDEO_CODEC_TYPE eEncCodec, u8 bPixelCacheEn, MFC_MB_LINEAR_TILE_MODE eLinearTileMode)
{
	u32 uProfile;
	u32 uInstanceId;
	u32 uRisc2Host;
	VIDEO_CODEC_TYPE eEncCodecType;
	u32 uPixelCache;
 	u32 uContextAddrOffset;
	
	oMfc.m_uProcessIdx = uProcessIdx;
	oMfc.m_bEncComplete[oMfc.m_uProcessIdx]=false;
	oMfc.m_bLastImageEncoding[oMfc.m_uProcessIdx]=true;
	oMfc.m_bFirstImageEncoding[oMfc.m_uProcessIdx]=true;
	
	// enc pixel cache on/off
	uPixelCache = (bPixelCacheEn == true) ? 0: 3;
	
	eEncCodecType = (eEncCodec==H264_ENC) ? 16:
			 		(eEncCodec==MPEG4_ENC) ? 17:
			 		(eEncCodec==H263_ENC) ? 18: Assert(0);

	// allocate context addr & size for each instance
 	if(uProcessIdx == 0)
		oMfc.m_uConTextBufAddr[uProcessIdx] = oMfc.m_uFwBufEndAddr;
 	else
 		oMfc.m_uConTextBufAddr[uProcessIdx] = oMfc.m_uConTextBufAddr[uProcessIdx-1] + 10*1024; // uContextSize

 	uContextAddrOffset = (oMfc.m_uConTextBufAddr[uProcessIdx] - oMfc.m_uDramABaseAddr)>>11;  

	// Open Instance
	oMfc.m_bOpenInstance=false;	
	MFC_SendH2RCmdWithFullArg(H2R_CMD_OPEN_INSTANCE, eEncCodecType, (oMfc.m_bbEnableCrc<<31)|uPixelCache, uContextAddrOffset, 10*1024);
 	
 	// wait until 1(Open Instance Return) 
	while(!oMfc.m_bOpenInstance);
	oMfc.m_bOpenInstance=false; 

	// Get the Open Instance ID.
	mfcInp32(MFC_RISC2HOST_ARG1, uInstanceId);
	oMfc.m_uInstanceId[uProcessIdx] = uInstanceId;	

	// set source image size
	mfcOutp32(MFC_HSIZE_PX, oMfcEncConfig[uProcessIdx].m_uSourceWidth);

	if(oMfcEncConfig[uProcessIdx].m_uPictureInterlace == 1)	// field coding
	{
		mfcOutp32(MFC_VSIZE_PX, oMfcEncConfig[uProcessIdx].m_uSourceHeight/2);
	}
	else
	{
		mfcOutp32(MFC_VSIZE_PX, oMfcEncConfig[uProcessIdx].m_uSourceHeight);
	}

	// profile register
	if(oMfc.m_eCodecMode[uProcessIdx] == H264_ENC)
	{
		u32 uProfile;

		uProfile = 
			(oMfcEncConfig[uProcessIdx].m_uProfileIDC == 66) ? 2 :	// baseline, oMfcEncConfig[oMfc.m_uProcessIdx].m_uProfileIDC == 66
			(oMfcEncConfig[uProcessIdx].m_uProfileIDC == 77) ? 0 :	// main
			1 ;	// high
		//uProfile |= (1<<2);	// [3:2] = Chroma IDC (1 for encoding)
		//uProfile |= (oMfcEncConfig[uProcessIdx].m_uTransform8x8Mode << 4);	// [4] = 8x8 enable flag (don't-care for encoding)
		uProfile |= (oMfcEncConfig[uProcessIdx].m_uLevelIDC << 8);		// [15:8] = level control		
		mfcOutp32(MFC_PROFILE, uProfile);
	}
	else if(oMfc.m_eCodecMode[uProcessIdx] == MPEG4_ENC)
	{		
		mfcOutp32(MFC_PROFILE, (oMfcEncConfig[uProcessIdx].m_uLevelIDC << 8) | oMfcEncConfig[uProcessIdx].m_uProfileIDC );
	}
	else if(oMfc.m_eCodecMode[uProcessIdx] == H263_ENC)
	{
		mfcOutp32(MFC_PROFILE, 0x20);
	}
	else
	{
		Disp("\r\n!!! Un Supported Standard !!! \r\n");
		return;
	}
	
	// picture structure register
	mfcOutp32(MFC_PICTURE_STRUCT, oMfcEncConfig[uProcessIdx].m_uPictureInterlace);

	// loop filter control register
	mfcOutp32(MFC_LF_CONTROL, oMfcEncConfig[uProcessIdx].m_uLoopFilterDisable);

	// loop fileter alpha & beta offset registers
	mfcOutp32(MFC_LF_ALPHA_OFF, oMfcEncConfig[uProcessIdx].m_sLoopFilterAlphaC0Offset * 2);
	mfcOutp32(MFC_LF_BETA_OFF, oMfcEncConfig[uProcessIdx].m_sLoopFilterBetaOffset * 2);
	
	// stream buffer mode control register
	mfcOutp32(MFC_EDFU_STR_BF_MODE_CTRL, 1);

	// stream control register
	//mfcOutp32(MFC_EDFU_SF_EPB_ON_CTRL, 1);	// automatical EPB insertion on

	// picture type control
	mfcOutp32(MFC_ENC_PIC_TYPE_CTRL, (1<<18)|(oMfcEncConfig[uProcessIdx].m_uNumberBFrames << 16)|oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod);

	// Enable recon data write at B-frame
#if RECON_B_ENABLE // At EVT1, RECON_B_WRITE_ON must be disable.
	mfcOutp32(MFC_ENC_B_RECON_WRITE_ON, 1); // 0=disable, 1=enable recon data write at B-frame
#endif
	
	// set registers related to multi-slice
	if(oMfcEncConfig[uProcessIdx].m_uSliceMode == 0)
	{
		mfcOutp32(MFC_ENC_MSLICE_CTRL, 0);
	}
	else if(oMfcEncConfig[uProcessIdx].m_uSliceMode == 1)	// fixed # mb in slice
	{
		mfcOutp32(MFC_ENC_MSLICE_CTRL, (0<<1)|0x1);
		mfcOutp32(MFC_ENC_MSLICE_MB, oMfcEncConfig[uProcessIdx].m_uSliceArgument);
	}
	else if(oMfcEncConfig[uProcessIdx].m_uSliceMode == 2)	// fixed # bytes in slice
	{
		mfcOutp32(MFC_ENC_MSLICE_CTRL, (1<<1)|0x1);
		mfcOutp32(MFC_ENC_MSLICE_BYTE, oMfcEncConfig[uProcessIdx].m_uSliceArgument);
	}
	else
		Assert(0);

	// set Circular intra refresh mb count setting reg.
	mfcOutp32(MFC_ENC_CIR_NUM, oMfcEncConfig[uProcessIdx].m_uRandomIntraMBRefresh);

	// set memory structure of current frame
	mfcOutp32(MFC_ENC_MAP_FOR_CUR, eLinearTileMode);

	// padding value control
	mfcOutp32(MFC_ENC_PADDIMG_CTRL, (oMfcEncConfig[uProcessIdx].m_uPadControlOn<<31)|
										(oMfcEncConfig[uProcessIdx].m_uCrPadVal<<16)|
										(oMfcEncConfig[uProcessIdx].m_uCbPadVal<<8)|
										(oMfcEncConfig[uProcessIdx].m_uLumaPadVal<<0));

	// rate control config.
	mfcOutp32(MFC_RC_CONFIG, (oMfcEncConfig[uProcessIdx].m_uEnableFRMRateControl<<9)|(oMfcEncConfig[uProcessIdx].m_uEnableMBRateControl<<8)|
								(oMfcEncConfig[uProcessIdx].m_uFrameQp<<0));

	// frame rate
	// [091124] MFC_RC_FRAME_RATE must be set regardless of RC value
	if(oMfcEncConfig[uProcessIdx].m_uFrameRate != 0)	// '0' is forbidden
	{
		mfcOutp32(MFC_RC_FRAME_RATE, oMfcEncConfig[uProcessIdx].m_uFrameRate);
	}

	// bit rate
	if(oMfcEncConfig[uProcessIdx].m_uBitrate != 0)	// '0' is forbidden
	{
		mfcOutp32(MFC_RC_BIT_RATE, oMfcEncConfig[uProcessIdx].m_uBitrate);
	}

	// Qunatization Parameter Boundary
	mfcOutp32(MFC_RC_QBOUND, (oMfcEncConfig[uProcessIdx].m_uQSCodeMax<<8)|(oMfcEncConfig[uProcessIdx].m_uQSCodeMin<<0));
	
	// reaction coefficient
	if(oMfcEncConfig[uProcessIdx].m_uEnableFRMRateControl == true)
	{
		mfcOutp32(MFC_RC_RPARA, oMfcEncConfig[uProcessIdx].m_uCBRPeriodRf);
	}

	// macroblock level rate control
	if(oMfcEncConfig[uProcessIdx].m_uEnableMBRateControl == true)
	{
		mfcOutp32(MFC_RC_MB_CTRL, (oMfcEncConfig[uProcessIdx].m_uDarkDisable<<3)|(oMfcEncConfig[uProcessIdx].m_uSmoothDisable<<2)|
								(oMfcEncConfig[uProcessIdx].m_uStaticDisable<<1)|(oMfcEncConfig[uProcessIdx].m_uActivityDisable<<0));
	}

	// P Frame QP and B Frame QP
	Outp32(oMfc.m_uShardMemBufAddr[0]+P_B_FRAME_QP, (oMfcEncConfig[uProcessIdx].m_uBFrameQp<<6)|(oMfcEncConfig[uProcessIdx].m_uPFrameQp<<0));

	// set h.264/mpeg4 encoder registers
	if(oMfc.m_eCodecMode[uProcessIdx] == H264_ENC)
	{
		u32 uNumOfRefsForP;
		 		
		// entropy
		mfcOutp32(H264_ENC_ENTRP_MODE, oMfcEncConfig[uProcessIdx].m_uSymbolMode);

		// num. of ref.
		uNumOfRefsForP = oMfcEncConfig[uProcessIdx].m_uNumberRefForPframes;
		if(uNumOfRefsForP > 2) 
		{
			uNumOfRefsForP = 2;
		}
		if(uNumOfRefsForP > oMfcEncConfig[uProcessIdx].m_uNumberReferenceFrames)
		{
			uNumOfRefsForP = oMfcEncConfig[uProcessIdx].m_uNumberReferenceFrames;
		}

		mfcOutp32(H264_ENC_NUM_OF_REF, (uNumOfRefsForP<<5)|(oMfcEncConfig[uProcessIdx].m_uNumberReferenceFrames));
#if 1
		// inter weighted parameter
		mfcOutp32(H264_ENC_MDINTER_WEIGHT, oMfcEncConfig[uProcessIdx].m_uMDInterWeightPPS);

		// intra weighted parameter
		mfcOutp32(H264_ENC_MDINTRA_WEIGHT, oMfcEncConfig[uProcessIdx].m_uMDIntraWeightPPS);
#endif
		// 8x8 transform enable flag
		mfcOutp32(H264_ENC_TRANS_8X8_FLAG, oMfcEncConfig[uProcessIdx].m_uTransform8x8Mode);

#if ACOMPANY
		mfcOutp32(H264_ENC_PRED_MODE_CTRL, ((oMfcEncConfig[uProcessIdx].m_uIntraMD8x8Disable&0x1)<<6) |
			((oMfcEncConfig[uProcessIdx].m_uIntraMD4x4Disable&0x1)<<5) | ((!(oMfcEncConfig[uProcessIdx].m_uInterSearch8x16)&0x1)<<4) |
			((!(oMfcEncConfig[uProcessIdx].m_uInterSearch16x8)&0x1)<<3) |((!(oMfcEncConfig[uProcessIdx].m_uInterSearch8x8)&0x1)<<2) |
			((oMfcEncConfig[uProcessIdx].m_uDisableHpelME&0x1)<<1) | ((oMfcEncConfig[uProcessIdx].m_uDisableQpelME&0x1)<<0));


		mfcOutp32(H264_ENC_I4X4_I8X8_WEIGHT, ((oMfcEncConfig[uProcessIdx].m_uIntraMD8x8Weight&0x7fff)<<16) |
			((oMfcEncConfig[uProcessIdx].m_uIntraMD4x4Weight&0x7fff)<<0)); 
		mfcOutp32(H264_ENC_I16X16_WEIGHT, oMfcEncConfig[uProcessIdx].m_uIntraMD16x16Weight);
#endif
	}
	else if(oMfc.m_eCodecMode[uProcessIdx] == MPEG4_ENC)
	{
		mfcOutp32(MPEG4_ENC_QUART_PXL, !oMfcEncConfig[uProcessIdx].m_uDisableQpelME);
		
		// vop timing
		Outp32(oMfc.m_uShardMemBufAddr[0]+VOP_TIMING, (oMfcEncConfig[uProcessIdx].m_uEnableVopTiming<<31)|		
														(oMfcEncConfig[uProcessIdx].m_uVopTimeResolution<<16)|
														(0<<0));
#if ACOMPANY
		mfcOutp32(H264_ENC_PRED_MODE_CTRL, 0);
#endif
	}					
	else
	{
#if ACOMPANY
		mfcOutp32(H264_ENC_PRED_MODE_CTRL, 0);
#endif
	}
}

void MFC_GetEncStandard(u32 uProcessIdx, VIDEO_CODEC_TYPE *eStandard)
{
	*eStandard = oMfc.m_eCodecMode[uProcessIdx];
}

void MFC_GetInstanceIdFromIdx(u32 uProcessIdx, u32 *pInstanceId)
{
	*pInstanceId = oMfc.m_uInstanceId[uProcessIdx];
}

void MFC_GetEncFrameSizeCfg(u32* pPicWidth, u32* pPicHeight)
{
	*pPicWidth = oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceWidth;
	*pPicHeight = oMfcEncConfig[oMfc.m_uProcessIdx].m_uSourceHeight;
}

void MFC_GetEncNumOfFramesCfg(u32* pNumOfFrames)
{
	*pNumOfFrames = oMfcEncConfig[oMfc.m_uProcessIdx].m_uFramesToBeEncoded;
}

void MFC_GetEncNumOfTotalFrames(u32* pNumOfTotalFrames)
{
	*pNumOfTotalFrames = oMfcEncConfig[oMfc.m_uProcessIdx].m_uTotalFramesToBeEncoded;
}

void MFC_GetEncPicTypeCfg(u32* pNumberBFrames)
{
	*pNumberBFrames = oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames;
}

void MFC_GetInterlace(u32* pIsInterlace)
{
	*pIsInterlace = oMfcEncConfig[oMfc.m_uProcessIdx].m_uPictureInterlace;
}

void MFC_GetIDRPeriod(u32* pIDR_period)
{
	*pIDR_period = oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod;
}

void MFC_GetBitRate(u32* pBitRate)
{
	*pBitRate = oMfcEncConfig[oMfc.m_uProcessIdx].m_uBitrate;
}

void MFC_GetGOP(u32* pGOP)
{
	*pGOP = oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod + (oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod-1)*oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames;
}

void MFC_GetEncFrameRate(u32* pFrameRate)
{
	*pFrameRate = oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameRate;
}

void MFC_GetSrcImgFileName(char *aFileName)
{
	u32 uLength=0;
	u32 i=0;

	uLength = strlen(oMfcEncConfig[oMfc.m_uProcessIdx].m_pInputFile);

	strcpy(aFileName, oMfcEncConfig[oMfc.m_uProcessIdx].m_pInputFile);
	aFileName[uLength] = NULL;
}

#if 0 //[oMfc.m_uCurrReleaseIdx]
void MFC_GetEncCurrentYCBufAddr(u32 *pCurrLumeAddr, u32 * pCurrChromaAddr)
{	
	*pCurrLumeAddr = oMfc.m_uCurrLumaAddr[oMfc.m_uProcessIdx];
	*pCurrChromaAddr = oMfc.m_uCurrChromaAddr[oMfc.m_uProcessIdx];
}
#else
void MFC_GetEncCurrentYCBufAddr(u32 *pCurrLumeAddr, u32 *pCurrChromaAddr)
{	
	u32 i;
	
//	MFC_ReleaseCurrentYCBuf();
	
	for(i=0; i<4; i++)
	{
		oMfc.m_uCurrReleaseIdx = i;
		if(oMfc.m_uCurrAddrState[oMfc.m_uProcessIdx][oMfc.m_uCurrReleaseIdx]==0xffffffff)
			break;
	}
	oMfc.m_uCurrAddrState[oMfc.m_uProcessIdx][oMfc.m_uCurrReleaseIdx] = 0;

	*pCurrLumeAddr = oMfc.m_uCurrLumaAddr[oMfc.m_uProcessIdx][oMfc.m_uCurrReleaseIdx];
	*pCurrChromaAddr = oMfc.m_uCurrChromaAddr[oMfc.m_uProcessIdx][oMfc.m_uCurrReleaseIdx];
}
#endif


void MFC_GetEncYCBufAddrFromBufIdx(u32 uBufIdx, u32 *pCurrLumeAddr, u32 *pCurrChromaAddr)
{
	*pCurrLumeAddr =  oMfc.m_uCurrLumaAddr[oMfc.m_uProcessIdx][uBufIdx];
	*pCurrChromaAddr =  oMfc.m_uCurrChromaAddr[oMfc.m_uProcessIdx][uBufIdx];
}

void MFC_SetEncCurrentYCBufAddr(u32 uCurrLumeAddr, u32 uCurrChromaAddr)
{
	oMfc.m_uCurrLumaAddr[oMfc.m_uProcessIdx][oMfc.m_uCurrReleaseIdx] = uCurrLumeAddr;
	oMfc.m_uCurrChromaAddr[oMfc.m_uProcessIdx][oMfc.m_uCurrReleaseIdx] = uCurrChromaAddr;
}

void MFC_GetEncStrSzPerFrame(u32 *uEncStrSzPerFrame)
{
	u32 uOutputSize;
	u32 uOneFrameSize;

#if 0
	mfcInp32(MFC_RISC2HOST_ARG3, uOutputSize);
	mfcInp32(MFC_SI_STREAM_SIZE, uOneFrameSize);

	*uEncStrSzPerFrame = uOutputSize;
	
	if((uOneFrameSize > MFC_STR_BUF_SIZE) || (uOneFrameSize > uOutputSize))
	{
		Disp(">>>One Frame Stream Size(%d) is Over Allocated Stream Buffer(%d)\n", uOneFrameSize, MFC_STR_BUF_SIZE);
	}
#else
	mfcInp32(MFC_SI_STREAM_SIZE, uOutputSize);	

	*uEncStrSzPerFrame = uOutputSize;
	
	if((uOutputSize > MFC_STR_BUF_SIZE))
	{
		Disp(">>>One Frame Stream Size(%d) is Over Allocated Stream Buffer(%d)\n", uOutputSize, MFC_STR_BUF_SIZE);
	}
#endif


}

void MFC_GetFrameType(char **pcFrameType)
{
	u32 uFrameType;
	u32 uEncStreamSize;

	mfcInp32(MFC_SI_STREAM_SIZE, uEncStreamSize);
	
	if(uEncStreamSize == 0)
	{
		*pcFrameType = "Not Yet";
		return;
	}

	mfcInp32(MFC_SI_SLICE_TYPE, uFrameType);

	if( uFrameType == 0 ) *pcFrameType = "Not_CODED_FRAME";	
	if( uFrameType == 1 ) *pcFrameType = "I_FRAME";
	if( uFrameType == 2 ) *pcFrameType = "P_FRAME";
	if( uFrameType == 3 ) *pcFrameType = "B_FRAME";
	if( uFrameType == 4 ) *pcFrameType = "SKIPED_FRAME";
	if( uFrameType == 5 ) *pcFrameType = "OTHERS";
}

void MFC_ReleaseCurrentYCBuf(void)
{
	u32 i;
	u32 uCurrLumaAddr;
	u32 uEncStreamSize;
	
	mfcInp32(MFC_SI_STREAM_SIZE, uEncStreamSize);
	mfcInp32(MFC_SI_ENCODED_Y_ADDR, uCurrLumaAddr);

	if(uEncStreamSize != 0)
	{
		for (i=0; i<4; i++)
		{
			if ( (oMfc.m_uCurrLumaAddr[oMfc.m_uProcessIdx][i])==((uCurrLumaAddr<<11)+oMfc.m_uDramBBaseAddr) )
			{
				oMfc.m_uCurrAddrState[oMfc.m_uProcessIdx][i]  = 0xffffffff;
				break;
			}
		}

	}
}

void MFC_ReadEncOneFrameStream(u32 uDstAddr, u32 uSize)
{
	u32 i;
	u32 uSrcAddr;	
	u32 uStreamBufAddr;

	uStreamBufAddr = oMfc.m_uStreamBufAddr[oMfc.m_uProcessIdx];
	
	uSrcAddr = uStreamBufAddr;
	
	for (i=0; i<uSize; i++)
		*(u8 *)(uDstAddr+i) = *(u8 *)(uSrcAddr+i);
}

void MFC_GetEncSteamBufAddr(u32* pStreamBufAddr)
{
	u32 uEncStreamBufAddr;	
	u32 uStreamBufAddr;
	
	uStreamBufAddr = oMfc.m_uStreamBufAddr[oMfc.m_uProcessIdx];
	
	uEncStreamBufAddr = uStreamBufAddr;
	*pStreamBufAddr = uEncStreamBufAddr;
}

void MFC_ContinueEncoding(void)
{
	u32 uStreamBufAddr;
	
 	mfcOutp32(MFC_SI_CH0_SB_ADDR, (oMfc.m_uStreamBufAddr[oMfc.m_uProcessIdx]-oMfc.m_uDramABaseAddr)>>11);
	uStreamBufAddr = (oMfc.m_uStreamBufAddr[oMfc.m_uProcessIdx]-oMfc.m_uDramABaseAddr)>>11;
	MFC_SendH2RCmdWithFullArg(H2R_CMD_CONTINUE_ENC, uStreamBufAddr, 0, 0, 0);
}

void MFC_EncodingOneFrame(MFC_DEC_TYPE eEncType)
{
	u32 uFrameDelta;

	oMfc.m_bFatalErr = false;

	// stream buffer address
	mfcOutp32(MFC_SI_CH0_SB_ADDR, (oMfc.m_uStreamBufAddr[oMfc.m_uProcessIdx]-oMfc.m_uDramABaseAddr)>>11);
	mfcOutp32(MFC_SI_CH0_SB_SIZE, MFC_STR_BUF_SIZE);

	// current frame buffer address
	mfcOutp32(MFC_SI_CH0_CURRENT_Y_ADDR, 
				(oMfc.m_uCurrLumaAddr[oMfc.m_uProcessIdx][oMfc.m_uCurrReleaseIdx] - oMfc.m_uDramBBaseAddr)>>11);	
	mfcOutp32(MFC_SI_CH0_CURRENT_C_ADDR, 
				(oMfc.m_uCurrChromaAddr[oMfc.m_uProcessIdx][oMfc.m_uCurrReleaseIdx] - oMfc.m_uDramBBaseAddr)>>11);

	// set shared mem reg
	mfcOutp32(MFC_SI_CH0_HOST_WR_ADR, oMfc.m_uShardMemBufAddr[0]-oMfc.m_uDramABaseAddr);

	if(eEncType==SEQ_HEADER)
	{
		oMfc.m_bSeqDone = false;
		mfcOutp32(MFC_SI_CH0_INST_ID, (0x1<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx]);
		while(!oMfc.m_bSeqDone);
	}
	else if(eEncType==FRAME)
	{
		if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == MPEG4_ENC)
		{
			if(oMfc.m_bFirstImageEncoding[oMfc.m_uProcessIdx]==true)
			{
				uFrameDelta=0;
				oMfc.m_bFirstImageEncoding[oMfc.m_uProcessIdx]=false;
			}
			else
			{
				uFrameDelta=oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameDelta;
			}
			
			Outp32(oMfc.m_uShardMemBufAddr[0]+VOP_TIMING, (oMfcEncConfig[oMfc.m_uProcessIdx].m_uEnableVopTiming<<31)|		
															(oMfcEncConfig[oMfc.m_uProcessIdx].m_uVopTimeResolution<<16)|
															(uFrameDelta<<0));
		}
		mfcOutp32(MFC_SI_CH0_INST_ID, (0x2<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx]);
	}
	else if(eEncType==LAST_SEQ)
	{
		if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == MPEG4_ENC)
		{		
			if(oMfc.m_bLastImageEncoding[oMfc.m_uProcessIdx]==true)
			{
				uFrameDelta=oMfcEncConfig[oMfc.m_uProcessIdx].m_uFrameDelta;		
				oMfc.m_bLastImageEncoding[oMfc.m_uProcessIdx]=false;
			}
			else
			{
				uFrameDelta=0;
			}
			
			Outp32(oMfc.m_uShardMemBufAddr[0]+VOP_TIMING, (oMfcEncConfig[oMfc.m_uProcessIdx].m_uEnableVopTiming<<31)|		
															(oMfcEncConfig[oMfc.m_uProcessIdx].m_uVopTimeResolution<<16)|
															(uFrameDelta<<0));	
		}
		mfcOutp32(MFC_SI_CH0_INST_ID, (0x3<<16)|oMfc.m_uInstanceId[oMfc.m_uProcessIdx]);
	}
}

void MFC_ConvertYuv420ToNv12OnlyCbCr(u32 uSrcAddrOfCbCr, u32 uPicHsz, u32 uPicVsz, u32 uDstAddrOfCbCr)
{
	u32 i;
	u32 uStAddrOfCb = uSrcAddrOfCbCr;
	u32 uStAddrOfCr = uSrcAddrOfCbCr+uPicHsz*uPicVsz/4;

	for (i=0; i<uPicHsz*uPicVsz/2; i++) // Cb, Cr
	{
		Copy8((i%2==0)? (uStAddrOfCb+i/2) : (uStAddrOfCr+(i-1)/2), uDstAddrOfCbCr+i, 1);
	}
}

static char pConfigFileName[MAX_CONFIGFILE_COUNT][MAX_FILENAME_SIZE];
void MFC_ParseConfigListFile(u32 uConfigListFileBufAddr, u32 *uCount)
{
	char *pConfigListFileStr = (char*)uConfigListFileBufAddr;
	char *param_line = malloc(MAX_FILENAME_SIZE);
	u32 i;

	for(i=0; i<MAX_CONFIGFILE_COUNT; i++)
		pConfigFileName[i][0] = NULL;

	param_line = strtok(pConfigListFileStr, "\r\n");

	i = 0;

	do
	{
		if((param_line[0] == '\n')||(param_line[0] == '\r')||(param_line[0] == '#'))
			continue;
		
		strcpy(pConfigFileName[i++], param_line);
		
	}while((param_line = strtok(NULL, "\r\n")) != NULL);

	*uCount = i;
		
}

void MFC_GetConfigFileFromList(u32 uIdx, char *pFileName)
{
	u32 uLength;

	uLength = strlen(pConfigFileName[uIdx]);
	
	strcpy(pFileName, pConfigFileName[uIdx]);
	
	pFileName[uLength] = NULL;
}

u8 MFC_GeneratedCrcValue(u32* pNumOfCrc)
{
	u8 bIsCrcValue;
	u32 uDecStatus;
	
	mfcInp32(MFC_SI_DECODE_STATUS, uDecStatus);

	if( (uDecStatus>>5) & (0x1) )
		bIsCrcValue = true;
	else
		bIsCrcValue = false;

	if( (uDecStatus>>4) & (0x1) )
		*pNumOfCrc = 4;
	else
		*pNumOfCrc = 2;

	return bIsCrcValue;
}

// old function
void MFC_SearchFrameIdxToEnc(u32 uFrameCount, u32* uFrameIdx)
{	
	u32 uTotalFramesToBeEncoded;
	u32 uIDRPeriod;
	u32 uNumberBFrames;

	MFC_GetEncNumOfTotalFrames(&uTotalFramesToBeEncoded);
	MFC_GetIDRPeriod(&uIDRPeriod);
	MFC_GetEncPicTypeCfg(&uNumberBFrames);
	
	if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H264_ENC)
	{
		if(uNumberBFrames == 0)
		{
			*uFrameIdx = uFrameCount;
		}
		else
		{
			u32 uFramesInGOP;
			u32 GOP_order;
			u32 Pframe_interval;
			
			if(uIDRPeriod == 0)
			{
				uFramesInGOP = uTotalFramesToBeEncoded;
			}
			else
			{
				uFramesInGOP = uIDRPeriod + (uIDRPeriod-1)*uNumberBFrames;	// Total Num(I+P) + Total Num(B)
			}

			GOP_order = uFrameCount % uFramesInGOP;
			Pframe_interval = uNumberBFrames+1;
			
			// Closed GOP
			// exaple 1) uNumberBFrames = 1 & uIDRPeriod = 4
			// picture type	: I B P B P B P I B P B P B P I B P B P B P
			// encoding order	: 0 2 1 4 3 6 5 7 9 8 .....
			//-------------------------------------------------
			// exaple 2) uNumberBFrames = 2 & uIDRPeriod = 4
			// picture type	: I B B P B B P I B B  P B B P I B B P B B P
			// encoding order	: 0 2 3 1 5 6 4 7 9 10 8
			
			if(GOP_order == 0)	// IDR picture
				*uFrameIdx = uFrameCount;
			else if((GOP_order%Pframe_interval) == 1)
				*uFrameIdx = uFrameCount + uNumberBFrames;	// P picture
			else
				*uFrameIdx = uFrameCount - 1;		// first B or second B picture
		}
	}
	else if(oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == MPEG4_ENC)
	{
		if(uNumberBFrames == 0)
		{
			*uFrameIdx = uFrameCount;
		}
		else
		{
			u32 Pframe_interval;

			Pframe_interval = uNumberBFrames+1;
			
			// Open GOP
			// exaple 1) uNumberBFrames = 1 & uIDRPeriod = 3
			// picture type	: I B P B P B I B P B P B I B P B P B I
			// encoding order	: 0 2 1 4 3 6 5 8 7 .....
			//-------------------------------------------------
			// exaple 2) uNumberBFrames = 2 & uIDRPeriod = 3
			// picture type	: I B B P B B P B B  I  B   B  P  B B p B P B B P
			// encoding order	: 0 2 3 1 5 6 4 8  9  7 11 12 10 ...	
			
			if( uFrameCount == 0 )
				*uFrameIdx = uFrameCount;
			else if((uFrameCount %Pframe_interval) == 1)	
				*uFrameIdx = uFrameCount + uNumberBFrames;	// I picture or P picture
			else
				*uFrameIdx = uFrameCount - 1;		// first B or second B picture
		}
	}
	else		// H263_ENC
	{
		*uFrameIdx = uFrameCount;
	}
}

void MFC_FrameIdxToBeLoaded(u32 uFrameCount, s32* sFrameIdx)
{
	s32 GOP = ( oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames +1 )*(oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod-1) + 1;

	if((oMfc.m_eCodecMode[oMfc.m_uProcessIdx] == H264_ENC) && (oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod != 0) && ( oMfcEncConfig[oMfc.m_uProcessIdx].m_uIDRPeriod != 1) ) // H.264 closed GOP case
	{
		if( oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames == 1 ) 
		{
			if( uFrameCount == 0 ) *sFrameIdx = 0;
			else if( (uFrameCount%GOP) == 0)  *sFrameIdx = 1; // next IDR
			else if( (uFrameCount %GOP) == 1 ) *sFrameIdx = 1;
			else if(  ((uFrameCount %GOP)%2)==0 ) *sFrameIdx = -2;
			else *sFrameIdx = 2;
		}
		else if( oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames == 2 )
		{
			if( uFrameCount == 0 ) *sFrameIdx = 0;
			else if( (uFrameCount%GOP) == 0)  *sFrameIdx = 1; // next IDR
			else if( (uFrameCount %GOP) == 1 ) *sFrameIdx = 2;
			else if( ((uFrameCount %GOP)%3)==2 ) *sFrameIdx = -3;
			else if( ((uFrameCount %GOP)%3)==1 ) *sFrameIdx = 3;
			else *sFrameIdx = 0;
		}
		else *sFrameIdx = 0;

	}
	else
	{
		if( oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames == 1 ) 
		{
			if( uFrameCount == 0 ) *sFrameIdx = 0;
			else if( uFrameCount == 1 ) *sFrameIdx = 1;
			else if( (uFrameCount%2)==0 ) *sFrameIdx = -2;
			else *sFrameIdx = 2;
		}
		else if( oMfcEncConfig[oMfc.m_uProcessIdx].m_uNumberBFrames == 2 )
		{
			if( uFrameCount == 0 ) *sFrameIdx = 0;
			else if( uFrameCount == 1 ) *sFrameIdx = 2;
			else if( (uFrameCount%3)==2 ) *sFrameIdx = -3;
			else if( (uFrameCount%3)==1 ) *sFrameIdx = 3;
			else *sFrameIdx = 0;
		}
		else *sFrameIdx = 0;
	}
}


int TILE_4x2_READ(unsigned char* pLinearImgStAddr, unsigned char* pTiledImgStAddr, int I_XSIZE, int I_YSIZE, int x_pos, int y_pos)
{

	int pixel_x_m1, pixel_y_m1;
	int roundup_x, roundup_y;
	int linear_addr0, linear_addr1, bank_addr ;
	int x_addr ;
	int trans_addr, i;
	unsigned int data32 ;
	int tmp =0 ;

	pixel_x_m1 = I_XSIZE -1 ;
	pixel_y_m1 = I_YSIZE -1 ;

	roundup_x = ((pixel_x_m1)/16/8 + 1) ;
	roundup_y = ((pixel_x_m1)/16/4 + 1) ;

	x_addr = x_pos/4;

	if((I_YSIZE <= y_pos+32) && ( y_pos < I_YSIZE) &&
	    (((pixel_y_m1 >> 5) & 0x1) == 0) && (((y_pos >> 5) & 0x1) == 0))
	{
		
		linear_addr0 =  ( ((y_pos  & 0x1f) <<4) |(x_addr & 0xf ) ) ;
		linear_addr1 = (((y_pos >> 6) & 0xff) * roundup_x + ((x_addr >> 6) & 0x3f)) ;
		

		if(  ((x_addr >> 5) & 0x1) ==  ((y_pos >> 5) & 0x1))
			bank_addr = ((x_addr >> 4) & 0x1);
		else 
			bank_addr = 0x2 | ((x_addr >> 4) & 0x1);		
	}
	else
	{
		linear_addr0 =  ( ((y_pos  & 0x1f) <<4) |(x_addr & 0xf ) ) ;
		linear_addr1 = (((y_pos >> 6) & 0xff) * roundup_x + ((x_addr >> 5) & 0x7f)) ;
		

		if(  ((x_addr >> 5) & 0x1) ==  ((y_pos >> 5) & 0x1))
			bank_addr = ((x_addr >> 4) & 0x1);
		else 
			bank_addr = 0x2 | ((x_addr >> 4) & 0x1);		
		
	}		

	linear_addr0 = linear_addr0 << 2 ;

	trans_addr = (linear_addr1 <<13) | (bank_addr << 11) | linear_addr0 ;

	
	return trans_addr; 

}

void Tile2Linear_4x2(unsigned char* pLinearImgStAddr, unsigned char* pTiledImgStAddr, unsigned int I_XSIZE, unsigned int I_YSIZE)
{
	//int pLinearImgStAddr, pTiledImgStAddr,  I_XSIZE, I_YSIZE, 
	int trans_addr  ;
	int i, j,k ;
	unsigned char data8[4] ;
	 
	// .. TILE 4x2 test
	for (i=0; i<I_YSIZE; i=i+16)
	{
		for (j=0; j<I_XSIZE; j=j+16) 
		{
			trans_addr = TILE_4x2_READ(pLinearImgStAddr,   pTiledImgStAddr,   I_XSIZE, I_YSIZE, j, i);
			
	    	for (k=0; k<16; k++)
	    	{
				data8[0] = pTiledImgStAddr[trans_addr + 64*k + 0];
				data8[1] = pTiledImgStAddr[trans_addr + 64*k + 1];
				data8[2] = pTiledImgStAddr[trans_addr + 64*k + 2];
				data8[3] = pTiledImgStAddr[trans_addr + 64*k + 3];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j] = data8[0];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 1] = data8[1];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 2] = data8[2];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 3] = data8[3];	
				
				data8[0] = pTiledImgStAddr[trans_addr + 64*k + 4];
				data8[1] = pTiledImgStAddr[trans_addr + 64*k + 5];
				data8[2] = pTiledImgStAddr[trans_addr + 64*k + 6];
				data8[3] = pTiledImgStAddr[trans_addr + 64*k + 7];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 4] = data8[0];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 5] = data8[1];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 6] = data8[2];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 7] = data8[3];	

				data8[0] = pTiledImgStAddr[trans_addr + 64*k + 8];
				data8[1] = pTiledImgStAddr[trans_addr + 64*k + 9];
				data8[2] = pTiledImgStAddr[trans_addr + 64*k + 10];
				data8[3] = pTiledImgStAddr[trans_addr + 64*k + 11];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 8] = data8[0];	
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 9] = data8[1];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 10] = data8[2];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 11] = data8[3];

				data8[0] = pTiledImgStAddr[trans_addr + 64*k + 12];
				data8[1] = pTiledImgStAddr[trans_addr + 64*k + 13];
				data8[2] = pTiledImgStAddr[trans_addr + 64*k + 14];
				data8[3] = pTiledImgStAddr[trans_addr + 64*k + 15];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 12] = data8[0];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 13] = data8[1];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 14] = data8[2];
				pLinearImgStAddr[i*I_XSIZE  + I_XSIZE * k + j + 15] = data8[3];	
			}
		}
	}
}

void SplitInterveneCbCr(unsigned char* pLinearCBufAddr, u32 uHsize, u32 uVsize, unsigned char* pCbStBufAddr, unsigned char* pCrStBufAddr)
{
	int i;

	for(i=0; i<uHsize*uVsize; i++)
	{
		if((i % 2) == 0)
		{
			// Cb
			pCbStBufAddr[i/2] = pLinearCBufAddr[i];
		}
		else
		{
			// Cr
			pCrStBufAddr[(i-1)/2] = pLinearCBufAddr[i];
		}
	}
}

void ConvertTiled4x2ToLinear(u32 uHsize, u32 uVsize, u32 uLumaSrcAddr, u32 uChromaSrcAddr, u32 uDstImgAddr, u32* uImgEndAddr)
{
	u32 uChromaHsize;
	u32 uChromaVsize;
	u32 uLumaHsize;
	u32 uLumaVsize;
	u8* pLinearChromaBuf;
	u32 uLinearYSize;
	u32 uLinearCSize;
	u32 uLinearCbAddr;
	u32 uLinearCrAddr;
	
	uLumaHsize = uHsize;
	uLumaVsize = uVsize;
	
	uChromaHsize = uLumaHsize;
	uChromaVsize = uLumaVsize/2;

	pLinearChromaBuf = malloc(uHsize * uVsize / 2);

	uLinearYSize = uHsize*uVsize;
	uLinearCSize = uHsize*uVsize/2;
	
	uLinearCbAddr = uDstImgAddr +  uLinearYSize;
	uLinearCrAddr = uLinearCbAddr + uHsize*uVsize/4;
	

	*uImgEndAddr = uLinearCrAddr + uHsize*uVsize/4;
	
	Tile2Linear_4x2((u8*)uDstImgAddr, (u8*)uLumaSrcAddr, uLumaHsize, uLumaVsize); // uImgDstAddr=DstLumaAddr
	Tile2Linear_4x2(pLinearChromaBuf, (u8*)uChromaSrcAddr, uChromaHsize, uChromaVsize);

	SplitInterveneCbCr(pLinearChromaBuf, uChromaHsize, uChromaVsize, (u8*)uLinearCbAddr, (u8*)uLinearCrAddr);

	free(pLinearChromaBuf);
}

