/*---------------------------------------------------------------------------*/
/*                                                                           */
/*          COPYRIGHT 2003-2005 SAMSUNG ELECTRONICS CO., LTD.                */
/*                          ALL RIGHTS RESERVED                              */ 
/*                                                                           */
/*   Permission is hereby granted to licensees of Samsung Electronics        */
/*   Co., Ltd. products to use or abstract this computer program for the     */
/*   sole purpose of implementing a product based on Samsung                 */
/*   Electronics Co., Ltd. products. No other rights to reproduce, use,      */
/*   or disseminate this computer program, whether in part or in whole,      */
/*   are granted.                                                            */
/*                                                                           */
/*   Samsung Electronics Co., Ltd. makes no representation or warranties     */
/*   with respect to the performance of this computer program, and           */
/*   specifically disclaims any responsibility for any damages,              */
/*   special or consequential, connected with the use of this program.       */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/**
* @project 
			MOREX II
* @file
          	morex_util.c
* @module
          	Source file for the utiltity functions of the MOREX II
* @purpose
			This file defines utility functions of the MOREX II 
* @author
            Young Won Yun (youngwon.yun@samsung.com)
* @date
            2005/09/22
* @revision history
            2005/09/22 [YoungWon Yun]: First writing
*/
 
#include "morex_util.h"
 
 
/**
* 
* Compare SingleByte String
*
* @param		szStr1		SingleByte String
				szStr2		SingleByte String
				nSize		Length
* @return		0			Two input strings are equal
				not 0			Two input strings are not equal
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_Strncmp(UINT8 *szStr1, UINT8 *szStr2, UINT32 nSize)
{
	INT32 nRe = 0;
 
	while (nSize-- && !(*szStr1 - *szStr2) && *szStr2)
	{
		szStr1++; szStr2++;
	}
 
 	nRe = *szStr1 - *szStr2;
 
	if (nRe == 0)
	{
		return MOREX_OK;
	}
	else
	{
		return MOREX_ERROR;
	}
}
 
/**
* 
* Compare Multibyte String and Unicode String case insensitive
*
* Example
*
* @param		szStr1		Multibyte String
				szStr2		Unicode String
				nSize		Length
* @return		0			Two input strings are equal
				< 0			Two input strings are not equal
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_Mstricmp(UINT8 *szStr1, UINT8 *szStr2, INT32 nLength)
{
	INT32	i;
	UINT16	*szTmpStr1;
	UINT16	*szTmpStr2;
	
	szTmpStr1 = (UINT16*)szStr1;
	szTmpStr2 = (UINT16*)szStr2;
 
	for (i = 0; i < nLength; i++)
	{
		if (*szTmpStr1 == *szTmpStr2)
		{
			szTmpStr1++;
			szTmpStr2++;
			continue;
		}
		else
		{
			if (((*szTmpStr1 >= 'A') && (*szTmpStr1 <= 'Z')))
			{
				if (*szTmpStr1 == TFFN_TO_Upper(*szTmpStr2))
				{
					szTmpStr1++;
					szTmpStr2++;
					continue;
				}
			}
			if (((*szTmpStr1 >= 'a') && (*szTmpStr1 <= 'z')))
			{
				if (*szTmpStr1 == TFFN_TO_Lower(*szTmpStr2))
				{
					szTmpStr1++;
					szTmpStr2++;
					continue;
				}
			}
			return MOREX_ERROR;
		}		
	}
 
	return MOREX_OK;
}
 
/**
* 
* Calculate SingleByte String Length
*
* @param		szStr		String
* @return		String length
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_Strlen(UINT8 *szStr)
{
	UINT32	nSize = 0;
 
	while (*szStr++)
	{
		nSize++;
	}
 
	return nSize;
}
 
/**
* 
* Calculate MultiByte String Length
*
* @param		szStr		Multibyte String
* @return		String length
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_Mstrlen(UINT8 *szStr)
{
	UINT32	nSize;
	UINT16	*szTmpStr;
 
	szTmpStr = (UINT16 *)szStr;
 
	nSize = 0;
 
	while (szTmpStr[nSize] != 0)
	{
		nSize ++;
	}
 
	return nSize;
}
 
/**
* 
* Change to upper case string
*
* @param		szStr		String
				nLength		String Length
* @return		VOID
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
VOID
TFFN_ToUpperCase(UINT8 *szStr, UINT32 nLength)
{
	UINT32	i;
 
	for (i = 0; i < nLength; i++)
	{
		*szStr = (UINT8)TFFN_TO_Upper(*szStr);
		szStr++;		
	}
 
	return;
}
 
/**
* 
* Convert the argument to upper case letter
*
* @param		nValue		letter to convert
* @return		Converted letter
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_TO_Upper(INT32 nValue)
{
	UINT8 wc;
 
	nValue = nValue & 0xff;
 
	if ((nValue >= 'a') && (nValue <= 'z'))
	{
		wc = 'A' + (nValue - 'a');
	}
	else
	{
		wc = nValue;
	}
 
	return wc;
}
 
/**
* 
* Change to lower case string
*
* @param		szStr		String
				nLength		String Length
* @return		VOID
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
VOID
TFFN_ToLowerCase(UINT8 *szStr, UINT32 nLength)
{
	UINT32	i;
 
	for (i = 0; i < nLength; i++)
	{
		*szStr = (UINT8)TFFN_TO_Lower(*szStr);
		szStr++;		
	}
 
	return;
}
 
/**
* 
* Convert the argument to lower case letter
*
* @param		nValue		letter to convert
* @return		Converted letter
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_TO_Lower(INT32 nValue)
{
	UINT8 wc;
 
	nValue = nValue & 0xff;
 
	if ((nValue >= 'A') && (nValue <= 'Z'))
	{
		wc = 'a' + (nValue - 'A');
	}
	else
	{
		wc = nValue;
	}
 
	return wc;
}
 
/**
* 
* Convert Multibyte String to Singlebyte String
*
* @param		szStr1		Multibyte String
				szStr2		Singlebyte String
* @return		0			Convert success
				< 0			Convert failure
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_Multi2Single(UINT8 *szStr1, UINT8 *szStr2)
{
	UINT32	nSize1, nSize2;
 
	nSize1 = 0;
	nSize2 = 0;
 
	while ((szStr1[nSize1] != 0) || (szStr1[nSize1+1] != 0))
	{
	
#if (MOREX_BYTE_ORDER == MOREX_LITTLE_ENDIAN)	
		szStr2[nSize2] = szStr1[nSize1];
#else
		szStr2[nSize2] = szStr1[nSize1 + 1];
#endif
 
		nSize1 += 2;
		nSize2++;
	}
 
	szStr2[nSize2] = 0;
 
	return MOREX_OK;
}
 
/**
* 
* Convert Singlebyte String to Multibyte String
*
* @param		szStr1		Singlebyte String
				szStr2		Multibyte String
* @return		0			Convert success
				< 0			Convert failure
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_Single2Multi(UINT8 *szStr1, UINT8 *szStr2)
{
	UINT32	nSize1, nSize2;
 
	nSize1 = 0;
	nSize2 = 0;
 
	while (szStr1[nSize1] != 0)
	{
#if (MOREX_BYTE_ORDER == MOREX_LITTLE_ENDIAN)
		szStr2[nSize2] = szStr1[nSize1];
		szStr2[nSize2 + 1] = 0;
#else
		szStr2[nSize2] = 0;
		szStr2[nSize2 + 1] = szStr1[nSize1];
#endif
		nSize2 += 2;
		nSize1++;
	}
 
	szStr2[nSize2] = 0;
	szStr2[nSize2 + 1] = 0;
 
	return MOREX_OK;
}
 
/**
* 
* Return log_2(x)
*
* @param		nValue		Test Value
* @return		bit length	if x is power of 2
				4096		if x is not power of 2
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_Log2(UINT32 nValue)
{
	UINT32	r, i;
	UINT32	nOrg;
 
	r = 4096;
 
	if (nValue == 0)
	{
		return r;
	}
	
	nOrg = nValue;
 
	i = 1;
 
	while ((nValue & 0x00000001) == 0)
	{
		nValue = nValue >> 1;
		i++;
	}
 
	if (nOrg == (0x01U << (i - 1)))
	{
		return (i - 1);
	}
	else
	{
		return r;
	}
}
 
/**
* 
* ceiling device
*
* @param		nNumber		numerator
*				nDenom		denominator
* @return		ceiling device
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
* @date			2005/11/03
*
* @revision history	
*				2005/11/03	[Ahn HyunJu]	first writing
**/
 
UINT32
TFFN_Cdiv(UINT32 nNumber, UINT32 nDenom)
{
	UINT32 nCdiv;
 
	nCdiv = nNumber + nDenom - 1;
	nCdiv = nCdiv / nDenom;
 
	return nCdiv;
}
 
/**
* 
* Print debug message
*
* @param		nNumber		numerator
*				nDenom		denominator
* @return		ceiling device
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
* @date			2005/11/03
*
* @revision history	
*				2005/11/03	[YoungWon Yun]	first writing
**/
VOID
TFFN_Debug_Msg(const INT8 *stMsg, const INT8 *stFileName, INT32 nLineNumber, const INT8 *stFuncName)
{
	TFFN_printf("________________________________________________________\n");
	TFFN_printf("%s\n(%s, %d)\n -> %s\n", stMsg, stFileName, nLineNumber, stFuncName);
	TFFN_printf("________________________________________________________\n");	
}
