/*---------------------------------------------------------------------------*/
/*                                                                           */
/*          COPYRIGHT 2003-2005 SAMSUNG ELECTRONICS CO., LTD.                */
/*                          ALL RIGHTS RESERVED                              */ 
/*                                                                           */
/*   Permission is hereby granted to licensees of Samsung Electronics        */
/*   Co., Ltd. products to use or abstract this computer program for the     */
/*   sole purpose of implementing a product based on Samsung                 */
/*   Electronics Co., Ltd. products. No other rights to reproduce, use,      */
/*   or disseminate this computer program, whether in part or in whole,      */
/*   are granted.                                                            */
/*                                                                           */
/*   Samsung Electronics Co., Ltd. makes no representation or warranties     */
/*   with respect to the performance of this computer program, and           */
/*   specifically disclaims any responsibility for any damages,              */
/*   special or consequential, connected with the use of this program.       */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/**
* @project 
			MOREX II
* @file
          	morex_fat.c
* @module
          	FAT core
* @purpose
			This file defines FAT core of the MOREX II  
* @author
            Young Won Yun (youngwon.yun@samsung.com)
* @date
            2005/09/08
* @revision history
            2005/09/08 [YoungWon Yun]: First writing
*/
#include "morex_fat.h"
#include "morex_util.h"
#include "morex_ddm.h"
#include "morex_iobuffer.h"
#include "string.h"

/************************************************************************/
/*						Global Variable									*/
/************************************************************************/
 
static t_volume_info	stVolumeInfo;			
static BOOL32			bInitialized 	= false;	
 
static UINT32			nCurFATsector 	= 0;		
static BOOL32			bChangeFAT 		= false;		
static UINT32			nTotalSectors 	= 0;

#ifdef	_USE_UCMALLOC
extern UINT32*		gaFatTable;		
extern UINT32*		gaBuffer;			
extern t_dir_entries*	pstDirEntsCache;			
#else
extern UINT32			gaFatTable[128];		
extern UINT32			gaBuffer[128];			
extern t_dir_entries	stDirEntsCache;			
#endif

static INT32 nCurSct = 0;
static INT32 nCurClt = 0;
static INT32 nCurIndex = 0;

 
/************************************************************************/
/*						Extern Variable									*/
/************************************************************************/
 
/************************************************************************/
/*						Local Definition								*/
/************************************************************************/
 
 
 
 
 
 
 
 
 
 
 
 
 
/************************************************************************/
/*						Local typedef									*/
/************************************************************************/
 
/************************************************************************/
/*						Local Functions									*/
/************************************************************************/
#define malloc_io		malloc
#define free_io		free

/**
* 
* @name			_Is_Same_Case_String
*
* @description	This function checks whether all the name characters are same case.
* 
* @param		szName			pointer to the name 
*
* @return		enuCASE_Lower	all characters are lower case
*				enuCASE_Upper	all characters are upper case
*				enuCASE_Mixed	upper case char and lower case char mixed
*				enuCASE_None	all characters are neither lower case or upper case
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Is_Same_Case_String(UINT16* szName)
{	
	UINT16	*tempPC;
	t_case_enum	eStrCase = enuCASE_None;
	t_case_enum	ePrevCase = enuCASE_None;
 
	tempPC = szName;
 
	while (*tempPC != '\0')
	{
		
		if ((*tempPC >= 'a') && (*tempPC <= 'z'))
		{
			eStrCase = enuCASE_Lower;
		}
		else if ((*tempPC >= 'A') && (*tempPC <= 'Z'))
		{
			eStrCase = enuCASE_Upper;
		}
		else
		{
			eStrCase = enuCASE_None;
		}
 
		
		if (ePrevCase == enuCASE_None)
		{
			ePrevCase = eStrCase;
		}
		else if (ePrevCase != eStrCase)
		{
			if (eStrCase != enuCASE_None )
			{
				ePrevCase = enuCASE_Mixed;
				break;
			}
		}
 
		tempPC++;
	}
 
	return ePrevCase;
}
 
/**
* 
* @name			_Sector_Num_Of_Cluster
*
* @description	This function returns logical sector number 
*				of the specified cluster number
* 
* @param		stVolInfo	pointer to volume information
*				nCluster	cluster number 
*
* @return		logical sector number of the cluster
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
UINT32
_Sector_Num_of_Cluster(t_volume_info *stVolInfo, UINT32 nCluster)
{
	if (nCluster == 0)
	{
		if (stVolInfo->nFATType == MOREX_FAT16)
		{
			return (stVolInfo->nROOTAddr);
		}
		else
		{
			return 0;
		}
	}
 
	return (((nCluster - 2) << stVolInfo->nSectorPerCltBits) + stVolInfo->nDATAAddr);
}
 
/**
* 
* @name			_Is_Valid_8_3_Format
*
* @description	This function checks if the name is valid 8.3 format  
* 
* @param		szName			pointer to FileName(encoded in multibyte)
*								the name must not have '.' on the first position and the last position
*				nNameLen		length of the name
*
* @return		MOREX_OK		the name is a valid 8.3 format name
*				MOREX_ERROR		the name is not a valid 8.3 format name
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Is_Valid_8_3_Format(UINT16 *szName, UINT32 *nNameLen)
{
	UINT16	*tempPC;
	INT32	nTempLen, nComparedLen, nDotCount;
	INT32	i;
 
	nDotCount = 0;
	tempPC = szName;
	nTempLen = *nNameLen;
	nComparedLen = 0;
 
 
	if ((nTempLen <= 0) || (nTempLen > 12)) 
	{
		return MOREX_ERROR;
	}
 
 
	for (i = 0; i < nTempLen; i++)
	{
		if (*tempPC != '.')
		{
 
			if (*tempPC == ' ')
			{
				return MOREX_ERROR;
			}
 
			tempPC++;
			nComparedLen++;
		}
		else
		{
			if (nComparedLen == 0)
			{
				return MOREX_ERROR;
			}
			if (nDotCount > 0)
			{
				return MOREX_ERROR;
			}
 
			if (i > 8)
			{
				return MOREX_ERROR;
			}
			
			if ((nTempLen - i) > 4)
			{
				return MOREX_ERROR;
			}
 
			
			nDotCount++;
			nComparedLen++;
			tempPC++;
		}
	}
 
	if ((nDotCount == 0) && (nTempLen > 8))
	{
		return MOREX_ERROR;
	}
 
	return MOREX_OK;
}
/**
* 
* @name			_Get_Long_File_Name
*
* @description	This function gets partial name from a long dir entry
* 
* @param		pstLongDirEnt	pointer to long directory entry
*				szName			pointer to the partial name buffer
*
* @return		MOREX_OK		success
*				MOREX_ERROR		2-byte align error of the szName
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32
_Get_Long_File_Name(t_dir_entry_long *pstLongDirEnt, UINT16 *szName)
{
	UINT16	*tempPC;
	INT32	i;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((UINT32)szName & MOREX_2BYTE_ALIGN_MASK)
	{
		return MOREX_ERROR;
	}
#endif
 
	tempPC = szName;
 
	for (i = 0; i < 10; i += 2)
	{
		*tempPC = (pstLongDirEnt->szName1[i] | (pstLongDirEnt->szName1[i + 1] << 8));
		if (*tempPC == '\0')
		{
			return MOREX_OK;
		}
		tempPC++;
	}
	for (i = 0; i < 12; i += 2)
	{
		*tempPC = (pstLongDirEnt->szName2[i] | (pstLongDirEnt->szName2[i + 1] << 8));
		if (*tempPC == '\0')
		{
			return MOREX_OK;
		}
		tempPC++;
	}
 
	for (i = 0; i < 4; i += 2)
	{
		*tempPC = (pstLongDirEnt->szName3[i] | (pstLongDirEnt->szName3[i + 1] << 8));
		if (*tempPC == '\0')
		{
			return MOREX_OK;
		}
		tempPC++;
	}
 
	*tempPC = '\0';
	return MOREX_OK;
}
 
/**
* 
* @name			_Set_LFN_Entry_Name
*
* @description	This function sets the partial name to a long dir entry
* 
* @param		szName			pointer to the partial name buffer
*				pstLongDirEnt	pointer to long directory entry				
*	
* @return		MOREX_OK		success
*				MOREX_ERROR		error
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Set_LFN_Entry_Name(UINT16 *szName, t_dir_entry_long *stDirLongEnt)
{
	INT32	i;
	UINT16	*tempPC;
	BOOL32	bNullTerminated = false;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((UINT32)szName & MOREX_2BYTE_ALIGN_MASK)
	{
		return MOREX_ERROR;
	}
#endif
 
	tempPC = szName;
 
	for (i = 0; i < 10; i += 2)
	{
		if (bNullTerminated == false)
		{
			if ((*tempPC == '\0'))
			{
				bNullTerminated = true;			
			}
 
			stDirLongEnt->szName1[i] = (*tempPC & 0xff);
			stDirLongEnt->szName1[i + 1] = (*tempPC & 0xff00) >> 8;
			tempPC++;
		}
		else
		{
			stDirLongEnt->szName1[i] = 0xFF;
			stDirLongEnt->szName1[i + 1] = 0xFF;
		}
	}
	for (i = 0; i < 12; i += 2)
	{
		if (bNullTerminated == false)
		{
			if ((*tempPC == '\0'))
			{
				bNullTerminated = true;			
			}
			stDirLongEnt->szName2[i] = (*tempPC & 0xff);
			stDirLongEnt->szName2[i + 1] = (*tempPC & 0xff00) >> 8;
			tempPC++;
		}
		else
		{
			stDirLongEnt->szName2[i] = 0xFF;
			stDirLongEnt->szName2[i + 1] = 0xFF;
		}
	}
    
	for (i = 0; i < 4; i += 2)
	{
		if (bNullTerminated == false)
		{
			if ((*tempPC == '\0'))
			{
				bNullTerminated = true;			
			}
			stDirLongEnt->szName3[i] = (*tempPC & 0xff);
			stDirLongEnt->szName3[i + 1] = (*tempPC & 0xff00) >> 8;
			tempPC++;
		}
		else
		{
			stDirLongEnt->szName3[i] = 0xFF;
			stDirLongEnt->szName3[i + 1] = 0xFF;
		}		
	}
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Set_Dir_Time
*
* @description	This function sets time field in a directory entry
* 
* @param		stDirEnt	pointer to directory entry
*
* @return		
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
VOID
_Set_Dir_Time(t_dir_entry *stDirEnt)
{
	UINT16	nDate;
	UINT16	nTime;
 
	TFFN_get_time(&nDate, &nTime);
 
	stDirEnt->nDirCtrTimeTenth = 0;
 
	stDirEnt->nDirCrtDate = TFFN_BO_uint16(nDate);
	stDirEnt->nDirCrtTime = TFFN_BO_uint16(nTime);
	stDirEnt->nDirLstAccDate = TFFN_BO_uint16(nDate);
	stDirEnt->nDirWrtDate = TFFN_BO_uint16(nDate);
	stDirEnt->nDirWrtTime = TFFN_BO_uint16(nTime);
 
	return;
}
/**
* 
* @name			_Compare_LFNEntry
*
* @description	This function compares the long file name to 
*				long directory entry 
* 
* @param		szName			long file name ( 13 characters )
*				pstLongDirEnt	long directory entry
*
* @return		MOREX_OK		the name and entry are matched
*				MOREX_ERROR		not matched
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Compare_LFNEntry(UINT16 *szName, t_dir_entry_long *pstLongDirEnt)
{
	UINT16	aDirChar[14];
	INT32	nNameLen, nEntryLen;
	INT32	nRe;
 
	nRe = _Get_Long_File_Name(pstLongDirEnt, aDirChar);
	if (nRe < 0)
	{
		return MOREX_ERROR;
	}
 
	nNameLen = TFFN_Mstrlen((UINT8*)szName);
	nEntryLen = TFFN_Mstrlen((UINT8*)aDirChar);
 
	if (nNameLen == 0)
	{
		return MOREX_ERROR;
	}
	else if (nNameLen > 13)
	{
		nNameLen = 13;
	}
 
	if (TFFN_Mstricmp((UINT8*)szName, (UINT8*)aDirChar, (nNameLen > nEntryLen)?nNameLen:nEntryLen) < 0)
	{
		return MOREX_ERROR;
	}
 
	return MOREX_OK;
}
			
/**
* 
* @name			_Valid_Short_Name_Char
*
* @description	This function checks whether the name character 
*				is invalid short name character
* 
* @param		sChar	pointer to a character
*
* @return		MOREX_OK	the character is valid
*				MOREX_ERROR	the character is invalid
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32
_Valid_Short_Name_Char(UINT16 *sChar)
{
	if (*sChar < 0x20)
	{
		return MOREX_ERROR;
	}
	switch (*sChar)
	{
		case 0x22: /* " */
		case 0x2A: /* * */
		case 0x2B: /* + */
		case 0x2C: /* , */		
		
		case 0x2F: /* / */
		case 0x3A: /* : */
		case 0x3B: /* ; */
		case 0x3C: /* < */
		case 0x3D: /* = */
		case 0x3E: /* > */
		case 0x3F: /* ? */
		case 0x5B: /* [ */
		case 0x5C: /* \ */
		case 0x5D: /* ] */
		case 0x7C: /* | */
			return MOREX_ERROR;
	}
 
	return MOREX_OK;
}
/**
* 
* @name			_Valid_Short_Name
*
* @description	This function checks whether the name is valid short name 
* 
* @param		szName	pointer to a name
*
* @return		MOREX_OK	the name is valid short name
*				MOREX_ERROR	the name is invalid short name
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 _Valid_Short_Name(UINT16 *szName)
{
	UINT16	*tempPC;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((UINT32)szName & MOREX_2BYTE_ALIGN_MASK)
	{
		return MOREX_ERROR;
	}
#endif
 
	tempPC = szName;
 
	while (*tempPC != '\0')
	{
		if (_Valid_Short_Name_Char(tempPC++) < 0)
		{
			return MOREX_ERROR;
		}			
	}
	
	return MOREX_OK;
}
 
/**
* 
* @name			_Valid_Long_Name_Char
*
* @description	This function checks whether the name character 
*				is invalid long name character
* 
* @param		sChar	pointer to a character
*
* @return		MOREX_OK	the character is valid
*				MOREX_ERROR	the character is invalid
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 _Valid_Long_Name_Char(UINT16 *sChar)
{
	if (*sChar < 0x20)
	{
		return MOREX_ERROR;
	}
	switch (*sChar)
	{
		case 0x22: /* " */
		case 0x2A: /* * */		
		case 0x2F: /* / */
		case 0x3A: /* : */
		case 0x3C: /* < */
		case 0x3E: /* > */
		case 0x3F: /* ? */
		case 0x5C: /* \ */
		case 0x7C: /* | */
			return MOREX_ERROR;
	}
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Valid_Long_Name
*
* @description	This function checks whether the name is valid long name
* 
* @param		szName	pointer to the name
*
* @return		MOREX_OK	the name is valid long name
*				MOREX_ERROR	the name is invalid long name
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 _Valid_Long_Name(UINT16 *szName)
{
	UINT16	*tempPC;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((UINT32)szName & MOREX_2BYTE_ALIGN_MASK)
	{
		return MOREX_ERROR;
	}
#endif
 
	tempPC = szName;
	while (*tempPC != '\0') 
	{
		if (_Valid_Long_Name_Char(tempPC++) < 0)
		{
			return MOREX_ERROR;
		}	
	}
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Remove_Trailing_Dot_N_Space
*
* @description	This function removes trailing dots and spaces from the name
* 
* @param		szName		pointer to Name(encoded in multibyte)
*				nNameLen	length of the name
*
* @return		MOREX_OK	remove the trailing dots and spaces
*				MOREX_ERROR	the name is not aligned 
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/11/29
*
* @revision history	
*				2005/11/29	[Ahn HyunJu]	first writing
*
**/
INT32
_Remove_Trailing_Dot_N_Space(UINT16 *szName, UINT32 *nNameLen)
{
	UINT16	*tempPC;		
	UINT32	i;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((UINT32)szName & MOREX_2BYTE_ALIGN_MASK)
	{
		return MOREX_ERROR;
	}
#endif
	
	i = *nNameLen;
	tempPC = szName + i - 1;
 
	while ((*tempPC == '.') || (*tempPC == ' '))
	{
		*tempPC = '\0';
		*nNameLen -= 1;
		i--;
		tempPC = szName + i - 1;
	}
 
	if (*nNameLen <= 0)
	{
		return MOREX_ERROR;
	}
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Check_FileName
*
* @description	This function checks whether the name is 
*				a short file name or 
*				a long file name
* 
* @param		szName		pointer to FileName(encoded in multibyte)
*				nNameLen	file name length
*
* @return		0			the name is a short file name 
*				> 0			number of long directory entries 
*				MOREX_ERROR	the name is invalid
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Check_FileName(UINT16 *szName, UINT32 *nNameLen)
{
	INT32	nRe;			
	INT32	nLFNEntryNum;	
	
#if (MOREX_STRICT_CHECK == 1)
	if ((UINT32)szName & MOREX_2BYTE_ALIGN_MASK)
	{
		return MOREX_ERROR;
	}
#endif
	
	nLFNEntryNum = 0;
 
	nRe = _Remove_Trailing_Dot_N_Space(szName, nNameLen);
	if (nRe < 0)
	{
		return MOREX_ERROR;
	}
 
 
	nRe = _Is_Valid_8_3_Format(szName, nNameLen);
	if (nRe < 0)
	{
 
		
		nRe = _Valid_Long_Name(szName);
		if (nRe < 0)
		{
			return MOREX_ERROR;
		}
 
		
		nLFNEntryNum = TFFN_Cdiv(*nNameLen, 13);
		
		return nLFNEntryNum;
	}
 
 
 
	nRe = _Valid_Short_Name(szName);
	if (nRe < 0)
	{
		
		nRe = _Valid_Long_Name(szName);
		if (nRe < 0)
		{
			return MOREX_ERROR;
		}
		nLFNEntryNum = 1;
	}
	return nLFNEntryNum;
}
 
/**
* 
* @name			_Check_Reserved_Name
*
* @description	This function checks whether the name is 
*				a reserved name
* 
* @param		szName		pointer to short file name 
*
* @return		MOREX_OK	the name is not a reserved name
*				MOREX_ERROR	the name is a reserved name
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Check_Reserved_Name(UINT8 *szName)
{
	/* MS-DOS "device special files" */
	UINT8 *psReservedNames[] = {
	"CON     ", "PRN     ", "NUL     ", "AUX     ",
	"COM1    ", "COM2    ", "COM3    ", "COM4    ", "COM5    ",
	"COM6    ", "COM7    ", "COM8    ", "COM9    ",
	"LPT1    ", "LPT2    ", "LPT3    ", "LPT4    ", "LPT5    ",
	"LPT6    ", "LPT7    ", "LPT8    ", "LPT9    ",
	NULL
	};
	UINT32		i;
 
	i = 0;
	while (psReservedNames[i] != NULL)
	{
		if (TFFN_Strncmp(szName, psReservedNames[i], 8) == MOREX_OK)
		{
			return MOREX_ERROR;
		}
		i++;
	}
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Make_8_3_Format
*
* @description	This function makes the short name to 
*				8.3 dos name format in directory entry 
* 
* @param		szName			pointer to FileName(encoded in multibyte)
*								the name must be a valid short file name 
*				sShortFileName	pointer to dos format file name(out)
*
* @return		MOREX_OK		
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Make_8_3_Format(UINT16 *szName, UINT8 *sShortFileName)
{
	INT32	i, j;
	UINT16	*tempPC;		
	UINT8	*tempSFN;		
 
	tempPC = (UINT16*) szName;
	tempSFN = sShortFileName;
 
 
	for (i = 0; i < 8 ; i++)
	{
		if ((*tempPC != '.') && (*tempPC != '\0'))
		{
			*tempSFN = TFFN_TO_Upper((UINT8)*tempPC);
			tempPC++;
			tempSFN++;
			continue;
		}
		else
		{
			for (j = i; j < 8; j++)
			{
				*tempSFN = ' ';
				tempSFN++;
			}
			break;
		}
	}
 
 
	if (*tempPC == '\0')
	{
		*tempSFN++ = ' ';
		*tempSFN++ = ' ';
		*tempSFN++ = ' ';
	}
	else 
	{
		tempPC++;
		for (i = 0; i < 3; i++)
		{
			if (*tempPC == '\0')
			{
				for (j = i; j < 3 ; j++)
				{
					*tempSFN++ = ' ';					
				}
				break;
			}
			else
			{
				*tempSFN = TFFN_TO_Upper((UINT8)*tempPC);				
				tempPC++;
				tempSFN++;
				continue;
			}
		}
	}
	*tempSFN = '\0'; 
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Make_8_3_Format_Lossy
*
* @description	This function makes the long name to 
*				8.3 dos name format in directory entry
*				this function may be used for creat, mkdir operation
* 
* @param		szName			pointer to FileName(encoded in multibyte)
*								the name must be a valid short file name 
*				sShortFileName	pointer to dos format file name(out)
*
* @return		MOREX_OK		
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Make_8_3_Format_Lossy(UINT16 *szName, UINT8 *sShortFileName)
{	
	INT32	i;
	UINT16	*tempName;	
	UINT16	*tempPCLong;	
	UINT32	nLength;		
	UINT32	nDotPos;		
	
	tempName = szName;
 
	while ((*tempName == ' ') || (*tempName == '.'))
	{
		tempName++;
	}
 
	nLength = TFFN_Mstrlen((UINT8*)tempName);	
 
 
	for (nDotPos = nLength-1; nDotPos > 0; nDotPos--)
	{
		if (*(tempName + nDotPos) == '.')
		{
			break;
		}
	}
 
	MOREX_memset(sShortFileName, ' ', 11);
 
 
	if (nDotPos != 0)
	{
		tempPCLong = (UINT16*)(tempName + nDotPos + 1);
		i = 8;
 
		while ((*tempPCLong != '\0') && (i < 11))
		{
			if (*tempPCLong != ' ')
			{
				if (_Valid_Short_Name_Char(tempPCLong) < 0)
				{
					sShortFileName[i++] = '_';
				}
				else
				{
					sShortFileName[i++] = (UINT8)*tempPCLong;
				}			
			}
			tempPCLong++;
		}
	}
	sShortFileName[11] = '\0';
 
 
	tempPCLong = tempName;
	nLength = TFFN_Mstrlen((UINT8*)tempPCLong);
	nLength = nLength - nDotPos;
	i = 0;
 
	while ((nLength > 0) && (i < 8))
	{
		if ((*tempPCLong != ' ') && (*tempPCLong != '.'))
		{
			if (_Valid_Short_Name_Char(tempPCLong) < 0)
			{
				sShortFileName[i++] = '_';
			}
			else
			{
				sShortFileName[i++] = (UINT8)*tempPCLong;
			}
		}
		tempPCLong++;
		nLength--;
	}
 
	TFFN_ToUpperCase(sShortFileName, 11);
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Increase_Directory_Entry_Index
*
* @description	This function increases the directory entry index 
*				in a sector by nIncNum
*				if the increased index is out of current sector or cluster range, 
*				calculate the sector, cluster and read the sector 
*				to the directory entry buffer
* 
* @param		*nIndex		current index in current sector
*				*nCurSct	current sector number
*				*nCurClt	current cluster number
*				stVolInfo	information of the mounted volume
*				nIncNum		increase by nIncNum
*
* @return		MOREX_OK	index is increased
*				MOREX_EOF	end of cluster chain
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Increase_Directory_Entry_Index(UINT32 *nCurIndex,
								UINT32 *nCurSector,
								UINT32 *nCurCluster,
								t_volume_info *stVolInfo,
								UINT32 nIncNum)
{
	UINT32	i;
	UINT32	nIndex; 
	UINT32	nSector; 
	UINT32	nCluster; 
	UINT32	nNextCluster = MOREX_FAT32_END;
	UINT32	nIncSct;
	UINT32	nIncClt;
    
	nIndex = (*nCurIndex) + nIncNum;
	nSector = *nCurSector;
	nCluster = *nCurCluster;
 
 
	if (nIndex > 15)
	{
		
		
		
		nSector += (nIndex >> MOREX_DIR_ENTRY_PER_SECTOR_BITS);	
		nIndex = nIndex & MOREX_DIR_ENTRY_PER_SECTOR_MASK;
 
 
		
		if (nCluster == 0) 
		{
			if (nSector > stVolInfo->nDATAAddr)
			{
				return MOREX_ERROR;
			}
			else if (nSector == stVolInfo->nDATAAddr)
			{
				if (nIndex == 0)
				{
					*nCurIndex = 15;
					*nCurSector = nSector - 1;
					*nCurCluster = nCluster;
					return MOREX_EOF;
				}
				else
				{
					return MOREX_ERROR;
				}
			}
			else
			{
				
			}
		}
		else
		{
			nIncSct = nSector - _Sector_Num_of_Cluster(stVolInfo, nCluster);
			if (nIncSct >= stVolInfo->nSectorPerClt)
			{
				nIncClt = (nIncSct >> stVolInfo->nSectorPerCltBits);
 
				for (i = 0; i < nIncClt ; i++)
				{
					
					_Get_Next_FAT_Value(nCluster, &nNextCluster);
					if (nNextCluster >= stVolInfo->nEndCltMask)
					{
						if (i == (nIncClt - 1))
						{
							if (nIndex == 0)
							{
								nSector = _Sector_Num_of_Cluster(stVolInfo, nCluster) + stVolInfo->nSectorPerClt;
 
								*nCurIndex = 15;
								*nCurSector = nSector - 1;
								*nCurCluster = nCluster;
								return MOREX_EOF;
							}
							else
							{
								return MOREX_ERROR;
							}
						}
						else
						{
							return MOREX_ERROR;
						}						
					}
					nCluster = nNextCluster;				
				}
				nSector = _Sector_Num_of_Cluster(stVolInfo, nCluster) + (nIncSct & (stVolInfo->nSectorPerClt) - 1);
			}
		}
		
	} 
 
	*nCurIndex = nIndex;
	*nCurSector = nSector;
	*nCurCluster = nCluster;
 
	return MOREX_OK;
}
 
 
/**
* 
* @name			_Get_Next_FAT_Value
*
* @description	This function gets the next cluster number from the FAT table
* 
* @param		nCurCluster		current cluster number
*				nNextCluster	pointer to the next cluster number
*
* @return		MOREX_OK	succeed to get next cluster number
*				MOREX_ERROR	failed to read fat table from the device
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Get_Next_FAT_Value(UINT32 nCurCluster, UINT32 *nNextCluster)
{
	INT32	nRe;
	UINT32	nFATsector;
	UINT32	nFATEntOffset;
	UINT32	nIndex;
	UINT8	*pstTempFAT;
 
	nFATsector = stVolumeInfo.nFAT1Addr + (nCurCluster >> stVolumeInfo.nClusterPerSectorBits);
	nFATEntOffset = nCurCluster & (stVolumeInfo.nClusterPerSector - 1);
 
	pstTempFAT = (UINT8*)gaFatTable;
	nIndex = 0;
 
	if (nFATsector != nCurFATsector )
	{
		nRe = TFFN_Read_Sectors(nFATsector, 1, pstTempFAT);
		if ((nRe < 0) || (nRe != 1))
		{
			return MOREX_ERROR;
		}
		else
		{
			nCurFATsector = nFATsector;		
		}		
	}
	
	if (stVolumeInfo.nFATType == MOREX_FAT16)
	{
		nIndex = nFATEntOffset * 2;
		*nNextCluster = (*(pstTempFAT + nIndex) | ( *(pstTempFAT + nIndex + 1) << 8)) & 0x0000ffff;
	}
	else
	{
		nIndex = nFATEntOffset * 4;
		*nNextCluster = (*(pstTempFAT + nIndex) | (*(pstTempFAT + nIndex + 1) << 8) | (*(pstTempFAT + nIndex + 2) << 16) | (*(pstTempFAT + nIndex + 3) << 24)) & 0xffffffff;
	}
	return MOREX_OK;
}
 
/**
* 
* @name			_Read_Dir_Entry
*
* @description	This function reads a sector to directory entries buffer
* 
* @param		nCurCluster		current cluster number
*				nCurSector		current sector number
*
* @return		MOREX_OK	succeed to read directory entries
*				MOREX_ERROR	failed to read directory entries
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32
_Read_Dir_Entry(UINT32 nCurCluster, UINT32 nCurSector)
{
	INT32 nReturn;
 
 #ifdef	_USE_UCMALLOC
 	if ((pstDirEntsCache->nCurCluster == nCurCluster) &&
		(pstDirEntsCache->nCurSector == nCurSector))
	{
		return MOREX_OK;
	}
 
	nReturn = TFFN_Read_Sectors(nCurSector, 1, (UINT8 *)pstDirEntsCache->aDirEnts);
	if (nReturn != 1)
	{
		return MOREX_EIO;
	}
	pstDirEntsCache->bChanged = 0;
	pstDirEntsCache->nCurCluster = nCurCluster;
	pstDirEntsCache->nCurSector = nCurSector;
 #else
	if ((stDirEntsCache.nCurCluster == nCurCluster) &&
		(stDirEntsCache.nCurSector == nCurSector))
	{
		return MOREX_OK;
	}
 
	nReturn = TFFN_Read_Sectors(nCurSector, 1, (UINT8 *)stDirEntsCache.aDirEnts);
	if (nReturn != 1)
	{
		return MOREX_EIO;
	}
	stDirEntsCache.bChanged = 0;
	stDirEntsCache.nCurCluster = nCurCluster;
	stDirEntsCache.nCurSector = nCurSector;
 #endif
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Write_Dir_Entry
*
* @description	This function writes directory entries buffer to the sector
* 
* @param		
*
* @return		MOREX_OK	succeed to write directory entries
*				MOREX_ERROR	failed to write directory entries
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32
_Write_Dir_Entry(VOID)
{
	INT32 nReturn;
#ifdef	_USE_UCMALLOC
	nReturn = TFFN_Write_Sectors(pstDirEntsCache->nCurSector, 1, (UINT8*)pstDirEntsCache->aDirEnts);
	if (nReturn != 1)
	{
		return MOREX_EIO;
	}
 
	pstDirEntsCache->bChanged = 0;
#else
	nReturn = TFFN_Write_Sectors(stDirEntsCache.nCurSector, 1, (UINT8*)stDirEntsCache.aDirEnts);
	if (nReturn != 1)
	{
		return MOREX_EIO;
	}
 
	stDirEntsCache.bChanged = 0;
 #endif
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Get_Dir_Entry1
*
* @description	This function returns one directory entry from the directory entry buffer
* 
* @param		nClusterOffset	offset from the start cluster
*
* @return		pointer to a directory entry 
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
t_dir_entry*
_Get_Dir_Entry1(UINT32 nClusterOffset)
{
	INT32 nIndex;
	
	nIndex = nClusterOffset & 15;
 
#ifdef	_USE_UCMALLOC
	return &pstDirEntsCache->aDirEnts[nIndex];
#else
	return &stDirEntsCache.aDirEnts[nIndex];
#endif
}
 
/**
* 
* @name			_Attach_Number
*
* @description	This function attaches ~ and number to the short file name 
* 
* @param		szDosName		pointer to 8.3 format FileName
*				nNumber			number to be attached 
*
* @return		MOREX_OK		
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Attach_Number(UINT8 *szDosName, UINT32 nNumber)
{
	UINT32	i, nDigit, nDigitPos, nSpacePos;	
	
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
	nDigitPos = 7;	
	nSpacePos = 0;
 
	while (nNumber > 0)
	{
		nDigit = nNumber % 10;
 
		szDosName[nDigitPos--] = nDigit + '0';
 
		nNumber = nNumber / 10;		
	}
 
	szDosName[nDigitPos] = '~';
 
	for (i = 0; i < 7; i++)
	{
		if (szDosName[i] == ' ')
		{
			nSpacePos = i;
			break;
		}
	}
 
	if (nSpacePos != 0)
	{        
		for (i = nDigitPos; i < 8; i++)
		{
			szDosName[nSpacePos++] = szDosName[i];
			szDosName[i] = ' ';
		}
	}
 
	return MOREX_OK;
}
 
/**
* 
* @name			_Get_Attached_Number
*
* @description	This function compares the string and 
*				gets attached number from the directory entry name
* 
* @param		szCreatName			pointer to 8.3 format FileName to be created
*				szDirName			pointer to directory entry's short file name 
*				nAttachedNumber		the attached number			
*
* @return		MOREX_ERROR		two string are not similar
*				> 0				the attached number of the directory entry's name
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Get_Attached_Number(UINT8 *szCreatName, UINT8 *szDirName, UINT32 *nAttachedNumber)
{
	UINT32	nNumber;		
	UINT32	i;
 
	UINT32	nDecimal;		
 
	UINT32	nTildePos;		
  
	nNumber = 0;
	nTildePos = 0;
	nDecimal = 1;
  
	for (i = 0 ; i < 3; i++)
	{
		if (szCreatName[8 + i] != szDirName[8 + i])
		{
			return MOREX_ERROR;
		}
	}
 
 
	for (i = 0; i < 8; i++)
	{
		if (szDirName[i] == '~')
		{
			nTildePos = i;
			break;
		}
	}
 
	if ((nTildePos == 0) || (nTildePos > 6))
	{
		return MOREX_ERROR;
	}
 
 
	for (i = 0; i < nTildePos; i++)
	{
		if(szCreatName[i] != szDirName[i])
		{
			return MOREX_ERROR;
		}
	}
    
 
	for (i = 7; i > nTildePos; i--)
	{
		if (szDirName[i] == ' ')
		{			
			continue;
		}
 
		if ((szDirName[i] < '0') || (szDirName[i] > '9'))
		{
			return MOREX_ERROR;
		}
		nNumber += ((szDirName[i] - 0x30) * nDecimal);
		nDecimal = nDecimal * 10;
	}
 
	*nAttachedNumber = nNumber;
	return MOREX_OK;
}
 
/**
* 
* @name			_Get_Lowest_Zero_Bit
*
* @description	This function finds the lowest zero bit in the bit map buffer
* 
* @param		pBuff	bit map buffer
*				nSize	buffer size in byte
*
* @return		MOREX_ERROR		there is no zero bit
*				> 0				the lowest bit index
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/11/04
*
* @revision history	
*				2005/11/04	[Ahn HyunJu]	first writing
*
**/
INT32 
_Get_Lowest_Zero_Bit(UINT8 *pBuff, UINT32 nSize)
{
	UINT32	i, j;
	UINT32	nLowestFree;
	UINT32	*pTemp, udwIp;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((UINT32)pBuff & MOREX_4BYTE_ALIGN_MASK)
	{
		return MOREX_ERROR;
	}
#endif
	
	pTemp = (UINT32 *) pBuff;
	nSize = nSize >> 2;			
 
 
	for (i = 0; i < nSize; i++)
	{
		
		if (pTemp[i] != (UINT32)0xffffffff)
		{
			break;
		}
	}
	
	if (i != nSize)
	{
		udwIp = TFFN_BO_uint32(pTemp[i]);
 
		
		nLowestFree = 1;
		for (j = 0; j < 32; j++)
		{
			if ((udwIp & nLowestFree) == 0)
			{	
				
				nLowestFree = (i << 5) + j;
				break;
			}
 
			
			nLowestFree <<= 1;
		}
		
		return nLowestFree;
	}
 
	return MOREX_ERROR;	
}
/**
* 
* @name			_Numeric_Tail_Generation
*
* @description	This function make the long name to 
*				8.3 dos name format in directory entry 
*				this function may be used for creat, mkdir operation
* 
* @param		szDosName		pointer to 8.3 format FileName
*				stFileHandler	pointer to FileHandler to be created
*
* @return		MOREX_OK		
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32 
_Numeric_Tail_Generation(UINT8 *szDosName, t_file_handler *stFileHandler)
{	
 
	#define MOREX_SHORT_INDEX_DB_MAX	512			
	#define MOREX_SHORT_INDEX_DB_SIZE	(MOREX_SHORT_INDEX_DB_MAX >> 3)
 
	INT32			nRe;
	INT32			nLowestFree;		
	UINT32			nAttachedNumber	= 0;	
	UINT32			nUpper;
	UINT32			nBase;	
 
	UINT32			nCurClt, nCurSct;	
	UINT32			nIndex;				
	t_dir_entry		*stDirEnt1;		
 
	UINT32			aBufShortDB[MOREX_SHORT_INDEX_DB_SIZE >> 2];
 
	UINT8			*pbShortDB;
    
	pbShortDB = (UINT8*)aBufShortDB;
    
	nBase = 1;
 
again:
 
	nCurClt = stFileHandler->nParentCluster;
	nCurSct = _Sector_Num_of_Cluster(&stVolumeInfo, nCurClt);
	nIndex = 0;
 
	MOREX_memset(pbShortDB, 0x00, MOREX_SHORT_INDEX_DB_SIZE);
	nLowestFree = 0;
	nUpper = MOREX_SHORT_INDEX_DB_MAX;
    
	do
	{
		
		nRe = _Read_Dir_Entry(nCurClt, nCurSct);
		if (nRe < 0)
		{
			return MOREX_EIO;
		}
 
		
		stDirEnt1 = _Get_Dir_Entry1(nIndex);
 
		if (stDirEnt1->aDirName[0] == 0x00)
		{
			goto get_lowest_free;
		}
        
		if (stDirEnt1->aDirName[0] == 0xe5)
		{
			
			nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
			if (nRe < 0)
			{        
				if (nRe == MOREX_EOF)
				{
					goto get_lowest_free;
				}
 
				return MOREX_ERROR;
			}
			
			continue;
		}
 
		if ((stDirEnt1->nDirAttr & MOREX_LONG_NAME_MASK) != MOREX_ATTR_LONG_NAME)
		{	
 
			
			nRe = _Get_Attached_Number(szDosName, &(stDirEnt1->aDirName[0]), &nAttachedNumber);
 
			if (nRe < 0)
			{
			}
			else
			{
				if (nAttachedNumber >= nBase)
				{
					nAttachedNumber -= nBase;
					if (nAttachedNumber >= MOREX_SHORT_INDEX_DB_MAX)
					{
						if (nAttachedNumber >= nUpper)
						{
							nUpper = nAttachedNumber + 1;							
						}
					}
					else
					{
						pbShortDB[(nAttachedNumber >> 3)] |= (1 << (nAttachedNumber & 0x07));						
					}
				}
 
			}
 
			
			nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
			if (nRe < 0)
			{
				if (nRe == MOREX_EOF)
				{
					goto get_lowest_free;
				}
 
				return MOREX_ERROR;
			}
		}
		else
		{
			
			nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
			if (nRe < 0)
			{
				if (nRe == MOREX_EOF)
				{
					goto get_lowest_free;
				}
 
				return MOREX_ERROR;
			}
		} 
 
	} while (1);
 
 
get_lowest_free:
 
 
	nLowestFree = _Get_Lowest_Zero_Bit(pbShortDB, MOREX_SHORT_INDEX_DB_SIZE);
	if (nLowestFree < 0)
	{
		nUpper += nBase;
		if (nUpper >= 999999)
		{
			return MOREX_ERROR;
		}
		
		nBase += MOREX_SHORT_INDEX_DB_MAX;
		goto again;
	}
 
 
	nLowestFree += nBase;
 
	_Attach_Number(szDosName, nLowestFree);
 
	return MOREX_OK;
}
 
/**
* 
* @name			_LFN_Checksum
*
* @description	This function returns an unsigned byte checksum computed on 
*				the 11 bytes short file name
* 
* @param		szShortName		pointer to 8.3 dos format short file name
*
* @return		checksum		
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
UINT8
_LFN_Checksum(UINT8 *szShortName)
{
	UINT8	bSum;		
	UINT8	*tempPC;	
	INT32	i;
 
	bSum = 0;
	tempPC = szShortName;
 
    for (i = 11; i != 0; i--)
    {
        bSum = (UINT8)( ((bSum & 0x01) ? 0x80 : 0) + (bSum >> 1) + (*tempPC++) );
    }
 
    return bSum;
}
 
/**
* 
* @name			_Lookup
*
* @description	This function lookup the entry which has the specified name 
*				from the start cluster
*				if the entry is found, set the file handler
* 
* @param		nStartCluster	start cluster number to lookup
*				szName			name to lookup
*				stFileHandler	pointer to the file handler
*
* @return		
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32
_Lookup(UINT32 nStartCluster, UINT16 *szName, t_file_handler *stFileHandler)
{
	INT32	nRe;
	INT32	i;
	UINT32	nCurClt, nCurSct;		
	UINT32	nIndex;					
	t_dir_entry	*stDirEnt1;			
	UINT8	sShortFileName[16];		
	UINT8	sDirEntFileName[16];	
	
	BOOL32	bFound;					
	BOOL32	bMatch;					
	UINT32	nNameLen;				
	INT32	nLFNEntryNumber;		
	UINT16	*tempPC;				
 
	UINT32	nPrevFreeIndexOffset;	
	UINT32	nFreeIndexOffset;		
	INT32	nFreeEntryNum;			
 
	UINT8		nChkSum;			
 
	nLFNEntryNumber = 0;
	nPrevFreeIndexOffset = 0;
	nFreeIndexOffset = 0;
	nFreeEntryNum = 0;
 
 
	nCurClt = nStartCluster;
	nCurSct = _Sector_Num_of_Cluster(&stVolumeInfo, nCurClt);
	nIndex = 0;
	
	stFileHandler->nParentClusterOffset = 0;
	stFileHandler->nFreeEntryIndexOffset = 0;
	stFileHandler->nFreeEntryNumber = 0;
 
	nNameLen = TFFN_Mstrlen((UINT8*)szName);
 
 
	nLFNEntryNumber = _Check_FileName(szName, &nNameLen);
	if (nLFNEntryNumber < 0)
	{
		
		return MOREX_ERROR;
	}
	else if (nLFNEntryNumber == 0)
	{
		
		
		stFileHandler->nParentCluster = nStartCluster;
		stFileHandler->nLFNEntsNumber = nLFNEntryNumber;
 
		goto SFNLookup;
	}
	else 
	{
		
		stFileHandler->nParentCluster = nStartCluster;
		stFileHandler->nLFNEntsNumber = nLFNEntryNumber;
 
		goto LFNLookup;
	}
	
 
SFNLookup:
// This routine only lookup with short file name
 
 
	_Make_8_3_Format(szName, &sShortFileName[0]);
	if (_Is_Same_Case_String(szName) == enuCASE_Mixed)
	{
		nLFNEntryNumber++;
	}
 
 
	if (sShortFileName[0] == 0xE5)
	{
		sShortFileName[0] = 0x05;
	}
 
 
 
	while (1)
	{
		
		nRe = _Read_Dir_Entry(nCurClt, nCurSct);
		if (nRe < 0)
		{
			return MOREX_EIO;
		}
 
		
		stDirEnt1 = _Get_Dir_Entry1(nIndex);
 
		if (stDirEnt1->aDirName[0] == 0x00)
		{
			
			stFileHandler->nParentClusterStart = nCurClt;			
			stFileHandler->nFreeEntryNumber = nFreeEntryNum;
			return MOREX_ENOTEXIST;
		}
 
		if (stDirEnt1->aDirName[0] == 0xe5)
		{
 
			
			nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
			if (nRe < 0)
			{
				if (nRe == MOREX_EOF)
				{
					
					
					stFileHandler->nParentClusterStart = nCurClt;			
					stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
					return MOREX_EOF;
				}
 
				return MOREX_ERROR;
			}
 
			stFileHandler->nParentClusterOffset += 1;
 
			
			if (nFreeEntryNum < (nLFNEntryNumber + 1))
			{
				nFreeIndexOffset = stFileHandler->nParentClusterOffset - 1;
 
				if ((nPrevFreeIndexOffset == 0) && (nFreeEntryNum == 0))
				{
					nPrevFreeIndexOffset = nFreeIndexOffset;
					stFileHandler->nFreeEntryIndexOffset = nFreeIndexOffset;
					nFreeEntryNum = 1;
				}
				else
				{
 
					
					if ((nFreeIndexOffset - nPrevFreeIndexOffset) == nFreeEntryNum)
					{
 
						
						nFreeEntryNum++;					
					}
					else
					{
 
						
						stFileHandler->nFreeEntryIndexOffset = nFreeIndexOffset;
						nPrevFreeIndexOffset = nFreeIndexOffset;
						nFreeEntryNum = 1;
					}
				}
			}
 
			continue;
		} 
 
		if ((stDirEnt1->nDirAttr & MOREX_LONG_NAME_MASK) != MOREX_ATTR_LONG_NAME)
		{
 
			
 
			MOREX_memcpy(&sDirEntFileName[0], &stDirEnt1->aDirName[0], 11);
 
			sDirEntFileName[11] = '\0';
			
			
			if ((TFFN_Strncmp(&sDirEntFileName[0], &sShortFileName[0], 11)) == 0)
			{
 
				
				
				stFileHandler->nMyCluster = TFFN_BO_uint16(stDirEnt1->nDirFstClusLo) 
											| (TFFN_BO_uint16(stDirEnt1->nDirFstClusHi) << 16);
				stFileHandler->nMyClusterCurrentStart = stFileHandler->nMyCluster;
				stFileHandler->nMyClusterCurrentOffset = 0;
				stFileHandler->nOffset = 0;
				if ((stDirEnt1->nDirAttr & MOREX_ATTR_DIRECTORY) != 0)
				{
					stFileHandler->nIsDir = 1;
				}
				else
				{
					stFileHandler->nIsDir = 0;
				}
				stFileHandler->nSize = TFFN_BO_uint32(stDirEnt1->nDirFileSize);
				stFileHandler->nParentClusterStart = nCurClt;				
				stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
				return MOREX_OK;
			}
 
			
			
 
			nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
			if (nRe < 0)
			{
				if (nRe == MOREX_EOF)
				{	
					
					
					stFileHandler->nParentClusterStart = nCurClt;				
					stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
					return MOREX_EOF;				
				}
 
				return MOREX_ERROR;
			}
 
			
			
			stFileHandler->nParentClusterOffset += 1;
 
		}
		else
		{
 
			
			
			nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
			if (nRe < 0)
			{
				if (nRe == MOREX_EOF)
				{	
					
					
					stFileHandler->nParentClusterStart = nCurClt;				
					stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
					return MOREX_EOF;				
				}
 
				return MOREX_ERROR;
			}
 
			
			
			stFileHandler->nParentClusterOffset += 1;
 
		} 
 
	} 
 
LFNLookup:
// This routine only lookup with long file name
 
	bFound = false;	
	bMatch = false;	
 
 
 
	while (1)
	{
		
		nRe = _Read_Dir_Entry(nCurClt, nCurSct);
		if (nRe < 0)
		{
			return MOREX_EIO;
		}
 
		
		stDirEnt1 = _Get_Dir_Entry1(nIndex);
 
		if (stDirEnt1->aDirName[0] == 0x00)
		{
			
			
			stFileHandler->nParentClusterStart = nCurClt;			
			stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
			return MOREX_ENOTEXIST;
		}
 
		if (stDirEnt1->aDirName[0] == 0xe5)
		{
 
			
			nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
			if (nRe < 0)
			{
				if (nRe == MOREX_EOF)
				{
					
					
					stFileHandler->nParentClusterStart = nCurClt;				
					stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
					return MOREX_EOF;				
				}
 
				return MOREX_ERROR;
			}
 
			stFileHandler->nParentClusterOffset += 1;
 
			
			if (nFreeEntryNum < (nLFNEntryNumber + 1))
			{
				nFreeIndexOffset = stFileHandler->nParentClusterOffset - 1;
 
				if ((nPrevFreeIndexOffset == 0) && (nFreeEntryNum == 0))
				{
					nPrevFreeIndexOffset = nFreeIndexOffset;
					stFileHandler->nFreeEntryIndexOffset = nFreeIndexOffset;
					nFreeEntryNum = 1;
				}
				else
				{
 
					
					if ((nFreeIndexOffset - nPrevFreeIndexOffset) == nFreeEntryNum)
					{
 
						
						nFreeEntryNum++;					
					}
					else
					{
 
						
						stFileHandler->nFreeEntryIndexOffset = nFreeIndexOffset;
						nPrevFreeIndexOffset = nFreeIndexOffset;
						nFreeEntryNum = 1;
					}
				}
			}
 
			continue;
 
		} 
 
		if( (nLFNEntryNumber == (stDirEnt1->aDirName[0] - MOREX_LAST_LONG_ENTRY)) && 
			((stDirEnt1->nDirAttr & MOREX_LONG_NAME_MASK) == MOREX_ATTR_LONG_NAME) )
		{
		 
 
			nChkSum = ((t_dir_entry_long*)stDirEnt1)->nDirChecksum;
 
			
			for (i = nLFNEntryNumber; i > 0 ; i--)
			{
				
				tempPC = (UINT16*)(szName + ((i-1)*13));
 
				if (i == 1) 
				{
					if (_Compare_LFNEntry(tempPC, (t_dir_entry_long*)stDirEnt1) < 0)
					{
						bMatch = false;
					}
					else
					{
						
						bMatch = true;
						bFound = true;
					}
				}
				else
				{
					if (_Compare_LFNEntry(tempPC, (t_dir_entry_long*)stDirEnt1) < 0)
					{
						bMatch = false;
					}
					else
					{
						bMatch = true;
					}
				}
 
				if ((bMatch == true) && (bFound == true))
				{
					
					nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
					if (nRe < 0)
					{
						if (nRe == MOREX_EOF)
						{
							return MOREX_EOF;
						}
 
						return MOREX_ERROR;
					}
 
					nRe = _Read_Dir_Entry(nCurClt, nCurSct);
					if (nRe < 0)
					{
						return MOREX_EIO;
					}
					stDirEnt1 = _Get_Dir_Entry1(nIndex);
 
					if (nChkSum == _LFN_Checksum(stDirEnt1->aDirName))
					{
						
						stFileHandler->nMyCluster = TFFN_BO_uint16(stDirEnt1->nDirFstClusLo) 
							| (TFFN_BO_uint16(stDirEnt1->nDirFstClusHi) << 16);
						stFileHandler->nMyClusterCurrentStart = stFileHandler->nMyCluster;
						stFileHandler->nMyClusterCurrentOffset = 0;
						stFileHandler->nOffset = 0;
						if ((stDirEnt1->nDirAttr & MOREX_ATTR_DIRECTORY) != 0)
						{
							stFileHandler->nIsDir = 1;
						}
						else
						{
							stFileHandler->nIsDir = 0;
						}
						stFileHandler->nSize = TFFN_BO_uint32(stDirEnt1->nDirFileSize);
 
						stFileHandler->nParentClusterStart = nCurClt;					
						stFileHandler->nParentClusterOffset += 1;
						stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
						return MOREX_OK;
					}
					else
					{
						
						nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
						if (nRe < 0)
						{
							if (nRe == MOREX_EOF)
							{
								
								
								
								stFileHandler->nParentClusterStart = nCurClt;						
								stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
								return MOREX_EOF;							
							}
 
							return MOREX_ERROR;
						}
						
						stFileHandler->nParentClusterOffset += 1;
 
						bMatch = false;
						bFound = false;
 
						break;					
					}
				}
				else if (bMatch == true)
				{
					
					nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
					if (nRe < 0)
					{
						if (nRe == MOREX_EOF)
						{
							
							
							
							stFileHandler->nParentClusterStart = nCurClt;						
							stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
							return MOREX_EOF;						
						}
 
						return MOREX_ERROR;
					}
 
					
					
					stFileHandler->nParentClusterOffset += 1;
                    
					nRe = _Read_Dir_Entry(nCurClt, nCurSct);
					if (nRe < 0)
					{
						return MOREX_EIO;
					}
					
					
					stDirEnt1 = _Get_Dir_Entry1(nIndex);
 
					if (nChkSum == ((t_dir_entry_long*)stDirEnt1)->nDirChecksum)
					{
						
						continue;
					}
					else
					{
						
						
						break;
					}
				}
 
				else 
				{
 
					
					nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
					if (nRe < 0)
					{
						if (nRe == MOREX_EOF)
						{
							
							
							
							stFileHandler->nParentClusterStart = nCurClt;						
							stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
							return MOREX_EOF;							
						}
 
						return MOREX_ERROR;
					}
 
					
					stFileHandler->nParentClusterOffset += 1;
 
					break;
 
				} 
 
			} 
 
		}
		else
		{
		 
 
			nRe = _Increase_Directory_Entry_Index(&nIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
			if (nRe < 0)
			{
				if (nRe == MOREX_EOF)
				{
					
					
					
					stFileHandler->nParentClusterStart = nCurClt;					
					stFileHandler->nFreeEntryNumber = nFreeEntryNum;
 
					return MOREX_EOF;							
				}
 
				return MOREX_ERROR;
			}
			
			stFileHandler->nParentClusterOffset += 1;
 
		} 
		  
 
	} 
}
 
/**
* 
* @name			_Path_Resolve
*
* @description	This function resolve the path name and lookup the path component
*
* @param		szPath			pointer to the path name
*				stFileHandler	pointer to the file handler for last component
*				bFlag			flag to open or creat
*								MOREX_OPEN_MODE or MOREX_CREATE_MODE
*
* @return		
*				for MOREX_OPEN_MODE
*					MOREX_OK		path resolution is succeed
*					MOREX_ENOTEXIST	path resolution is failed
*				for MOREX_CREATE_MODE
*					MOREX_OK		path resolution is succeed and the last entry does not exist
*					MOREX_ERROR		path resolution is succeed and the last entry exists				
*					MOREX_EOF		path resolution is succeed and the last entry does not exist the cluster is EOF				
*					MOREX_ENOTEXIST	path resolution is fail				
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32
_Path_Resolve(UINT16 *szPath, t_file_handler *stFileHandler, UINT32 bFlag)
{
 
	INT32	nRe;
	UINT32	nLength;				
	UINT32	nComparedLength;		
	UINT32  nTempLength;		
	UINT16	*tempPath;				
	UINT16	curPC[256], *tempCur;	
	UINT32	nStartCluster;			
	UINT32	nPrevSlash;
 
 
	tempPath = (UINT16 *)szPath;	
	nLength = TFFN_Mstrlen((UINT8*)szPath);
	nComparedLength = 0;
	nStartCluster = stFileHandler->nParentCluster;
	nPrevSlash = 0;
 
	if (nLength > MOREX_PATH_NAME_MAX_LENGTH)
	{
		return MOREX_EINVALID;
	}
 
name_resolve:
 
	if (nComparedLength >= nLength)
	{
		if (nPrevSlash > 0)
		{
			return MOREX_ERROR;
		}
		return MOREX_OK;
	}
 
 
	if (stFileHandler->nIsDir != 1)
	{
		return MOREX_ERROR;
	}
	tempCur = &curPC[0];
	nTempLength = 0;
 
	if (*tempPath != '\0')
	{
 
		while ((*tempPath != '/') && (*tempPath != '\0'))
		{
			
			*tempCur = *tempPath;				
			tempPath++;
			tempCur++;
			nTempLength++;
			nComparedLength++;
		}
 
		if (nTempLength == 0)
		{
			if (nComparedLength == 0)
			{
				return MOREX_ERROR;
			}
			if (nPrevSlash > 0)
			{
				return MOREX_ERROR;
			}
			nPrevSlash++;
			
			tempPath++;
			nComparedLength++;
			goto name_resolve;
		}
		else
		{
			
			curPC[nTempLength] = '\0';	
 
			
			nRe = _Lookup(nStartCluster, &curPC[0], stFileHandler);
			if (nRe < 0)
			{
				if ((*tempPath == '\0') && (bFlag == MOREX_CREATE_MODE))
				{
					if (nRe == MOREX_ENOTEXIST)
					{
						return MOREX_OK;
					}
					else if (nRe == MOREX_EOF)
					{							
						return MOREX_EOF;
					}					
				}	
				
				
				
				return MOREX_ENOTEXIST;
			}
			else
			{
				if ((*tempPath == '\0') && (bFlag == MOREX_CREATE_MODE))
				{
					
					return MOREX_ERROR;
				}
			}
			
			nStartCluster = stFileHandler->nMyCluster;
			nPrevSlash = 0;
			goto name_resolve;
 
		} 
 
	} 
 
	return MOREX_ERROR;
}
 
 
/**
* 
* @name			TFFN_FS_Init
*
* @description	This function initialize the file system
*
* @param		
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32
TFFN_FS_Init(VOID)
{
	INT32	nReturn;
	UINT8	*tempBuffer;
	t_boot_sector_common	*pstBS;
	t_boot_sector_32		*pstBS32;
 
	UINT32	nRootDirSectors;
	UINT32	nClusterCount;
 
	

	if (bInitialized == true)
	{
		return MOREX_OK;
	}
 
	nReturn = TFFN_DDM_Init(&nTotalSectors);
	if (nReturn < 0)
	{
		return MOREX_ERROR;
	}
 
 	tempBuffer = (UINT8*)gaBuffer;
 
	nReturn = TFFN_Read_Sectors(0, 1, tempBuffer);
	if (nReturn < 0)
	{
		return MOREX_ERROR;
	}
 
	pstBS = (t_boot_sector_common*)tempBuffer;
	pstBS32 = (t_boot_sector_32*)(tempBuffer + sizeof(t_boot_sector_common));
 
 
	stVolumeInfo.nSectorPerClt = pstBS->nSectorsPerClus;
	stVolumeInfo.nSectorPerCltBits = TFFN_Log2(pstBS->nSectorsPerClus);
	stVolumeInfo.nReservedSect = TFFN_BO_uint16(pstBS->nRsvdSecCnt);	
 
	nRootDirSectors = (pstBS->nRootEntCnt[0] | (pstBS->nRootEntCnt[1]<<8));
 
	nRootDirSectors = (((nRootDirSectors << 5) + (MOREX_SECTOR_SIZE -1)) >> MOREX_SECTOR_SIZE_BITS);
 
	if (TFFN_BO_uint16(pstBS->nFATSz16) != 0)
	{
		stVolumeInfo.nSectorPerFAT = TFFN_BO_uint16(pstBS->nFATSz16);
	}
	else
	{
		stVolumeInfo.nSectorPerFAT = TFFN_BO_uint32(pstBS32->nFATSz32);
	}
 
	nClusterCount = (pstBS->nTotSec16[0] | (pstBS->nTotSec16[1]<<8));
 
	if (nClusterCount == 0)
	{		
		nClusterCount = TFFN_BO_uint32(pstBS->nTotSec32);
	}
	
	stVolumeInfo.nTotalSectors = nClusterCount;
	
	nClusterCount = (nClusterCount
					- stVolumeInfo.nReservedSect 
					- (pstBS->nNumFATs * stVolumeInfo.nSectorPerFAT)
					- nRootDirSectors) >> stVolumeInfo.nSectorPerCltBits;
 
	if (nClusterCount < 4085)
	{
		
		return MOREX_ERROR;
	}
	else if (nClusterCount < 65525)
	{
		stVolumeInfo.nFATType = MOREX_FAT16;
		stVolumeInfo.nFAT1Addr = stVolumeInfo.nReservedSect;
		stVolumeInfo.nFAT2Addr = stVolumeInfo.nFAT1Addr + stVolumeInfo.nSectorPerFAT;
		stVolumeInfo.nRootCluster = 0;
		stVolumeInfo.nROOTAddr = stVolumeInfo.nFAT2Addr + stVolumeInfo.nSectorPerFAT;
		stVolumeInfo.nFirstRootAddr = stVolumeInfo.nFAT2Addr + stVolumeInfo.nSectorPerFAT;
		stVolumeInfo.nDATAAddr = stVolumeInfo.nROOTAddr + nRootDirSectors;
		stVolumeInfo.nTotalClusters = nClusterCount;
		stVolumeInfo.nEndCltMask = MOREX_FAT16_EOF;
		stVolumeInfo.nClusterPerSector = FAT16_CLUSTER_PER_SECTOR;
		stVolumeInfo.nClusterPerSectorBits = FAT16_CLUSTER_PER_SECTOR_BITS;
	}
	else
	{
		stVolumeInfo.nFATType = MOREX_FAT32;		
		stVolumeInfo.nFAT1Addr = stVolumeInfo.nReservedSect;
		stVolumeInfo.nFAT2Addr = stVolumeInfo.nFAT1Addr + stVolumeInfo.nSectorPerFAT;
		stVolumeInfo.nRootCluster = TFFN_BO_uint32(pstBS32->nRootClus);
		stVolumeInfo.nROOTAddr = stVolumeInfo.nFAT2Addr + stVolumeInfo.nSectorPerFAT;
		stVolumeInfo.nFirstRootAddr = stVolumeInfo.nFAT2Addr + stVolumeInfo.nSectorPerFAT;
		stVolumeInfo.nDATAAddr = stVolumeInfo.nROOTAddr + nRootDirSectors;
		stVolumeInfo.nTotalClusters = nClusterCount;
		stVolumeInfo.nEndCltMask = MOREX_FAT32_EOF;
		stVolumeInfo.nClusterPerSector = FAT32_CLUSTER_PER_SECTOR;
		stVolumeInfo.nClusterPerSectorBits = FAT32_CLUSTER_PER_SECTOR_BITS;
	}
 
#ifdef	_USE_UCMALLOC
	pstDirEntsCache->bChanged = 0;
	pstDirEntsCache->nCurCluster = 0;
	pstDirEntsCache->nCurSector = 0;	
#else
	stDirEntsCache.bChanged = 0;
	stDirEntsCache.nCurCluster = 0;
	stDirEntsCache.nCurSector = 0;	
 #endif
 
	nReturn = TFFN_Read_Sectors(stVolumeInfo.nFAT1Addr, 1, (UINT8*)gaFatTable);
	if ((nReturn < 0) || (nReturn != 1))
	{
		return MOREX_ERROR;
	}
	nCurFATsector = stVolumeInfo.nFAT1Addr;
 
	bInitialized = true;
 
 
 
 
 
 
 
 
	return MOREX_OK;
}
 
/**
* 
* @name			TFFN_FS_Creat
*
* @description	This function creates a file 
*
* @param		szPath			pointer to the file path
*				stFileHandler	pointer to the file handler
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32
TFFN_FS_Creat(UINT16 *szPath, t_file_handler *stFileHandler)
{
	INT32		nReturn;
	UINT32		i, j;
	UINT32		nNeedCluster;
	UINT32		nPathLength;
	UINT32		nNameLength;
	UINT32		nLFNEntsNum;			
	UINT8		sShortFileName[16];		
	UINT16		*szCreatName = NULL;			
	UINT16		*tempPC;				
	UINT32		nRemainedFreeIndex;		
	BOOL32		bFlagEOF;				
	UINT8		nChkSum = 0;				
	UINT32		nCluster, nSector, nIndex, nNextCluster; 
	UINT32		nSlashPos;				
	t_dir_entry			*stDirEnt;		
	t_dir_entry_long	*stDirLongEnt;	
	UINT32		nCurFat;
	UINT32		nParentDirSize;
	UINT32		nNewCls = 0;
 
	nLFNEntsNum = 0;
	bFlagEOF = false;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((bInitialized == false) || (szPath == NULL) || (stFileHandler == NULL))
	{
		return MOREX_ERROR;
	}
 
	tempPC = szPath;
	while (*tempPC != '\0')
	{
		if ((*tempPC < ' ') || (*tempPC > '~'))
		{
			return MOREX_ERROR;
		}
		tempPC++;		
	}
#endif	
 
	stFileHandler->nParentCluster = stVolumeInfo.nRootCluster;
	stFileHandler->nIsDir = 1;
 
	nReturn = _Path_Resolve(szPath, stFileHandler, MOREX_CREATE_MODE);
 
	if (nReturn < 0)
	{
		if (nReturn == MOREX_EOF)
		{
			bFlagEOF = true;
		}
		else
		{
			return MOREX_ERROR;
		}
	}
 
	nSlashPos = 0;
	nPathLength = TFFN_Mstrlen((UINT8*)szPath);
	for (i = nPathLength; i > 0; i--)
	{
		if (*(szPath + i - 1) == '/')
		{
			szCreatName = (UINT16*)(szPath + i);
			nSlashPos = i - 1;
			break;
		}
	}
 
	if (nSlashPos == 0)
	{
		szCreatName = szPath;		
	}
 
	nNameLength = TFFN_Mstrlen((UINT8*)szCreatName);
 
	nReturn = _Remove_Trailing_Dot_N_Space(szCreatName, &nNameLength);
	if (nReturn < 0)
	{
		return MOREX_ERROR;
	}
 
	if (nNameLength > MOREX_FILE_NAME_MAX_LENGTH)
	{
		return MOREX_ERROR;
	}
 
	if (stFileHandler->nLFNEntsNumber < 0)
	{
		return MOREX_ERROR;
	}
	else if (stFileHandler->nLFNEntsNumber == 0)
	{	
		nReturn = _Is_Same_Case_String(szCreatName);
		if (nReturn == enuCASE_Mixed)
		{
			stFileHandler->nLFNEntsNumber = 1;			
		}
 
		_Make_8_3_Format(szCreatName, sShortFileName);
		
		if (sShortFileName[0] == 0xE5)
		{
			sShortFileName[0] = 0x05;
		}
 
		nReturn = _Check_Reserved_Name(sShortFileName);
		if (nReturn < 0)
		{
			return MOREX_ERROR;
		}
	}
	else
	{
		
		_Make_8_3_Format_Lossy(szCreatName, sShortFileName);
 
		if (sShortFileName[0] == 0xE5)
		{
			sShortFileName[0] = 0x05;
		}
 
		nReturn = _Numeric_Tail_Generation(sShortFileName, stFileHandler);
		if (nReturn < 0)
		{
			return MOREX_ERROR;
		}
 
		nReturn = _Check_Reserved_Name(sShortFileName);
		if (nReturn < 0)
		{
			return MOREX_ERROR;
		}
	}
 
	nLFNEntsNum = stFileHandler->nLFNEntsNumber;
 
 
	if (bFlagEOF == true)
	{
		if ((nLFNEntsNum + 1) <= stFileHandler->nFreeEntryNumber)
		{
			
			stFileHandler->nParentClusterOffset = stFileHandler->nFreeEntryIndexOffset;	
			if (stFileHandler->nParentCluster != 0) 
			{
				i = stFileHandler->nFreeEntryIndexOffset / (stVolumeInfo.nSectorPerClt << 4);
				nCluster = stFileHandler->nParentCluster;
				for ( ; i > 0; i--)
				{
					nReturn = _Get_Next_FAT_Value(nCluster, &nNextCluster);
					if ((nReturn < 0) || (nNextCluster >= stVolumeInfo.nEndCltMask))
					{
						return MOREX_ERROR;
					}
					nCluster = nNextCluster;
				}
				stFileHandler->nParentClusterStart = nCluster;
			}
		}
		else
		{
			if ((stFileHandler->nParentCluster == 0) && (stVolumeInfo.nFATType == MOREX_FAT16))
			{
				return MOREX_ERROR;
			}
 
			
			
			nNeedCluster = TFFN_Cdiv((nLFNEntsNum + 1), 
										(stVolumeInfo.nSectorPerClt << MOREX_DIR_ENTRY_PER_SECTOR_BITS));
			nParentDirSize = TFFN_Cdiv((stFileHandler->nParentClusterOffset + 1), 
										(stVolumeInfo.nSectorPerClt << MOREX_DIR_ENTRY_PER_SECTOR_BITS));
 
			nParentDirSize += nNeedCluster;
			nParentDirSize = nParentDirSize << stVolumeInfo.nSectorPerCltBits;
			nParentDirSize = nParentDirSize << MOREX_SECTOR_SIZE_BITS;
 
			if (nParentDirSize > MOREX_DIR_SIZE_MAX)
			{
				return MOREX_ERROR;
			}
 
			nCluster = 2;
			nCurFat = nCurFATsector;
			for (i = 0; i < nNeedCluster; i++)
			{
				nReturn = _Get_Free_FAT_Value(nCluster, &nNextCluster);
				if (nReturn == MOREX_ERROR)
				{
					
					return MOREX_ERROR;
				}
				
				
				if (i == 0)
				{
					nNewCls = nNextCluster;
				}
 
				nReturn = _Update_FAT(nNextCluster, -1, false);
				if (nReturn == MOREX_ERROR)
				{				
					return MOREX_ERROR;
				}
 
				nCluster = nNextCluster;
 
				if ((i != 0) && (nNextCluster == nNewCls) && (i != (INT32)nNeedCluster))
				{
					return MOREX_ERROR;
				}
 
			} 
			nCurFATsector = nCurFat;
 
			nReturn = TFFN_Read_Sectors(nCurFATsector, 1, (UINT8 *)gaFatTable);
			
			if((nReturn < 0) || (nReturn != 1))
			{
				return MOREX_ERROR;
			}
 
			
			nCluster = stFileHandler->nParentClusterStart;
			nReturn = _Allocate_Cluster(nCluster, &nNextCluster, 1);
			if (nReturn < 0)
			{
				return MOREX_ERROR;
			}
 
			nCluster = nNextCluster; 
 
			
			nReturn = _Clean_Cluster(nCluster);
			if (nReturn < 0)
			{
				return MOREX_ERROR;
			}
 
			stFileHandler->nParentClusterStart = nCluster;	
			stFileHandler->nParentClusterOffset++;
		}
	}
	else 
	{
		
		
		if ((stVolumeInfo.nFATType == MOREX_FAT16) && (stFileHandler->nParentCluster == 0))
		{
            nRemainedFreeIndex = 512 - stFileHandler->nParentClusterOffset;
		}
		else
		{
			nRemainedFreeIndex = (stVolumeInfo.nSectorPerClt << 4) 
								- (stFileHandler->nParentClusterOffset & ((stVolumeInfo.nSectorPerClt << 4) - 1));
		}
		
		if (nRemainedFreeIndex >= (nLFNEntsNum + 1))
		{
			
		}
		else if (stFileHandler->nFreeEntryNumber >= (nLFNEntsNum + 1))
		{
			
			
 
			stFileHandler->nParentClusterOffset = stFileHandler->nFreeEntryIndexOffset;
			if (stFileHandler->nParentCluster != 0)
			{
				i = stFileHandler->nFreeEntryIndexOffset / (stVolumeInfo.nSectorPerClt << 4);
				nCluster = stFileHandler->nParentCluster;
				for ( ; i > 0; i--)
				{
					nReturn = _Get_Next_FAT_Value(nCluster, &nNextCluster);
					if ((nReturn < 0) || (nNextCluster >= stVolumeInfo.nEndCltMask))
					{
						return MOREX_ERROR;
					}
					nCluster = nNextCluster;
				}
				stFileHandler->nParentClusterStart = nCluster;
			}
		}
		else
		{
			if ((stFileHandler->nParentCluster == 0) && (stVolumeInfo.nFATType == MOREX_FAT16))
			{
				return MOREX_ERROR;
			}
 
			
 
			
			nNeedCluster = TFFN_Cdiv((nLFNEntsNum + 1 - nRemainedFreeIndex),
										(stVolumeInfo.nSectorPerClt << MOREX_DIR_ENTRY_PER_SECTOR_BITS));
			nParentDirSize = TFFN_Cdiv((stFileHandler->nParentClusterOffset + 1), 
										(stVolumeInfo.nSectorPerClt << MOREX_DIR_ENTRY_PER_SECTOR_BITS));
 
			nParentDirSize += nNeedCluster;
			nParentDirSize = nParentDirSize << stVolumeInfo.nSectorPerCltBits;
			nParentDirSize = nParentDirSize << MOREX_SECTOR_SIZE_BITS;
 
			if (nParentDirSize > MOREX_DIR_SIZE_MAX)
			{
				return MOREX_ERROR;
			}
 
			nCluster = 2;
			nCurFat = nCurFATsector;
			for (i = 0; i < nNeedCluster; i++)
			{
				nReturn = _Get_Free_FAT_Value(nCluster, &nNextCluster);
				if (nReturn == MOREX_ERROR)
				{
					
					return MOREX_ERROR;
				}
 
				
				if (i == 0)
				{
					nNewCls = nNextCluster;
				}
 
				nReturn = _Update_FAT(nNextCluster, -1, false);
				if (nReturn == MOREX_ERROR)
				{				
					return MOREX_ERROR;
				}
 
				nCluster = nNextCluster;
 
				if ((i != 0) && (nNextCluster == nNewCls) && (i != (INT32)nNeedCluster))
				{
					return MOREX_ERROR;
				}
			} 
			nCurFATsector = nCurFat;
 
			nReturn = TFFN_Read_Sectors(nCurFATsector, 1, (UINT8 *)gaFatTable);
			
			if((nReturn < 0) || (nReturn != 1))
			{
				return MOREX_ERROR;
			}
 
			
			nCluster = stFileHandler->nParentClusterStart;
			nReturn = _Allocate_Cluster(nCluster, &nNextCluster, 1);
			if (nReturn < 0)
			{
				return MOREX_ERROR;
			}
 
			
			nReturn = _Clean_Cluster(nNextCluster);
			if (nReturn < 0)
			{
				return MOREX_ERROR;
			}
		}
	} 
 
 
	if (nLFNEntsNum > 0)
	{
		nChkSum = _LFN_Checksum(sShortFileName);
	}
 
	i = nLFNEntsNum;
	nCluster = stFileHandler->nParentClusterStart;
	if ((stVolumeInfo.nFATType == MOREX_FAT16) && (nCluster == 0))
	{
		nSector = stFileHandler->nParentClusterOffset >> MOREX_DIR_ENTRY_PER_SECTOR_BITS;
		nSector = nSector + stVolumeInfo.nROOTAddr;		
	}
	else
	{
		nSector = stFileHandler->nParentClusterOffset & ((stVolumeInfo.nSectorPerClt << 4) - 1);
		nSector = nSector >> 4;
		nSector = _Sector_Num_of_Cluster(&stVolumeInfo, nCluster) + nSector;
	}
	nIndex = stFileHandler->nParentClusterOffset & 15;
 
	while (i > 0)
	{
		nReturn = _Read_Dir_Entry(nCluster, nSector);
		if (nReturn < 0)
		{
			return MOREX_EIO;
		}
 
		for (j = nIndex; j < MOREX_DIR_ENTRY_PER_SECTOR; j++)
		{ 
 
			tempPC = (UINT16*)(szCreatName + 13*(i - 1));
 
#ifdef	_USE_UCMALLOC
			stDirLongEnt = (t_dir_entry_long*)&pstDirEntsCache->aDirEnts[j];
#else
			stDirLongEnt = (t_dir_entry_long*)&stDirEntsCache.aDirEnts[j];
#endif
 
			if (i == nLFNEntsNum)
			{
				stDirLongEnt->nDirOrd = MOREX_LAST_LONG_ENTRY | i;
			}
			else
			{
				stDirLongEnt->nDirOrd = i;
			}
 
			_Set_LFN_Entry_Name(tempPC, stDirLongEnt);
			stDirLongEnt->nDirChecksum = nChkSum;
			stDirLongEnt->nDirAttr = MOREX_ATTR_LONG_NAME;
			stDirLongEnt->nDirLongType = 0;
			stDirLongEnt->nFstClusLo = 0;
			
			stFileHandler->nParentClusterOffset++;
			i--;
 
			if (i < 1)
			{
				if (j == 15)
				{ 
					nReturn = _Write_Dir_Entry();
					if (nReturn < 0)
					{
						return MOREX_EIO;
					}
				}
				nReturn = _Increase_Directory_Entry_Index(&nIndex, &nSector, &nCluster, &stVolumeInfo, j - nIndex + 1);
				if (nReturn < 0)
				{
					if (nReturn == MOREX_EOF)
					{
						
						nReturn = _Allocate_Cluster(nCluster, &nNextCluster, 1);
						if (nReturn < 0)
						{
							return MOREX_ERROR;
						}
						nCluster = nNextCluster;
 
						
						nReturn = _Clean_Cluster(nCluster);
						if (nReturn < 0)
						{
							return MOREX_ERROR;
						}
 
						nSector = _Sector_Num_of_Cluster(&stVolumeInfo, nCluster);
						nIndex = 0;					
					}
					else
					{
						return MOREX_ERROR;
					}
				}
 
				stFileHandler->nParentClusterStart = nCluster;
 
				
				goto write_sfn; 
			}
 
		} 
 
		
		nReturn = _Write_Dir_Entry();
		if (nReturn < 0)
		{
			return MOREX_EIO;
		}
 
		
		nReturn = _Increase_Directory_Entry_Index(&nIndex, &nSector, &nCluster, &stVolumeInfo, MOREX_DIR_ENTRY_PER_SECTOR - nIndex);
		if (nReturn < 0)
		{
			if (nReturn == MOREX_EOF)
			{
				
				nReturn = _Allocate_Cluster(nCluster, &nNextCluster, 1);
				if (nReturn < 0 )
				{
					return MOREX_ERROR;
				}
				nCluster = nNextCluster;
 
				
				nReturn = _Clean_Cluster(nCluster);
				if (nReturn < 0)
				{
					return MOREX_ERROR;
				}
 
				nSector = _Sector_Num_of_Cluster(&stVolumeInfo, nCluster);
				nIndex = 0;					
			}
			else
			{
				return MOREX_ERROR;
			}
		}
		stFileHandler->nMyClusterCurrentStart = nCluster;
        
	} 
 
 
write_sfn:
 
	nReturn = _Read_Dir_Entry(nCluster, nSector);
	if (nReturn < 0)
	{
		return MOREX_EIO;
	}
 
#ifdef	_USE_UCMALLOC
	stDirEnt = &pstDirEntsCache->aDirEnts[nIndex];
#else 
	stDirEnt = &stDirEntsCache.aDirEnts[nIndex];
#endif
	
	MOREX_memcpy(&stDirEnt->aDirName[0], &sShortFileName[0], 11);
 
	stDirEnt->nDirAttr = MOREX_ATTR_ARCHIVE;
	stDirEnt->nDirFileSize = 0;
	stDirEnt->nDirFstClusHi = 0;
	stDirEnt->nDirFstClusLo = 0;
	stDirEnt->nDirNTRes = 0;
	_Set_Dir_Time(stDirEnt);
 
	nReturn = _Write_Dir_Entry();
	if (nReturn < 0)
	{
		return MOREX_EIO;
	}
 
	stFileHandler->nIsDir = 0;
	stFileHandler->nMyCluster = 0;
	stFileHandler->nMyClusterCurrentStart = 0;
	stFileHandler->nMyClusterCurrentOffset = 0;
	stFileHandler->nOffset = 0;
	stFileHandler->nSize = 0;
   
	return MOREX_OK;
}
 
/**
* 
* @name			TFFN_FS_Open
*
* @description	This function opens a file 
*
* @param		szPath			pointer to the file path
*				stFileHandler	pointer to the file handler
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/09/29
*
* @revision history	
*				2005/09/29	[Ahn HyunJu]	first writing
*
**/
INT32
TFFN_FS_Open(UINT16 *szPath, t_file_handler *stFileHandler)
{
	INT32	nReturn;
	UINT16	*tempPC;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((bInitialized == false) || (szPath == NULL) || (stFileHandler == NULL))
	{
		return MOREX_ERROR;
	}
 
	tempPC = szPath;
	while (*tempPC != '\0')
	{
		if ((*tempPC < ' ') || (*tempPC > '~'))
		{
			return MOREX_ERROR;
		}
		tempPC++;		
	}
#endif
    
	stFileHandler->nParentCluster = stVolumeInfo.nRootCluster;
	stFileHandler->nIsDir = 1;
 
	nReturn = _Path_Resolve(szPath, stFileHandler, MOREX_OPEN_MODE);
 
	if (nReturn < 0)
	{
		return MOREX_ERROR;
	}
 
	if (stFileHandler->nIsDir == 1)
	{		
		return MOREX_ERROR;
	}
 
	stFileHandler->nOffset = 0;	
    
	return MOREX_OK;
}
 
/**
* 
* @name			TFFN_FS_Read
*
* @description	This function read data from a file 
*
* @param		stFileHandler	pointer to the file handler
*				nInputSize			data size to read
*				szBuffer		buffer pointer for the read data
*				nReadSize		data size to be read
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		YoungWon Yun (youngwon.yun@samsung.com)	
*
* @date			2005/10/28
*
* @revision history	
*				2005/10/28	[YoungWon Yun] First Writing
*
**/
INT32
TFFN_FS_Read(t_file_handler *stFileHandler, UINT32 nInputSize, UINT32 *szBuffer, UINT32 *nReadSize)
{
	UINT32	nReadTotalSectors, nCheckOption;
	UINT32	nStartSector, nNextCluster;
	UINT32	nNumStartSectors, nNumBodyClusters, nNumEndSectors;
	UINT32	nRSize, nSize;
	INT32	nRe, i, nContinuousCluster, nOffset, nBufferOffset;
	UINT8	*szTempBuff, *szBuff;
	UINT32	nOrigMyClsOffset, nOrigMyClsStart, nOrigOffset;
	BOOL32	bReadFlag = false;
	UINT32 idx;
 
	szTempBuff = (UINT8 *)szBuffer;
	szBuff = (UINT8 *)gaBuffer;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((bInitialized == false) || (szBuffer == NULL) || (nReadSize == NULL) || (stFileHandler == NULL))
	{
		return MOREX_ERROR;
	}
#endif
 
	if (stFileHandler->nIsDir == 1)
	{
		return MOREX_ERROR;
	}
 
 
	if (((nInputSize + stFileHandler->nOffset) > MOREX_MAX_FILE_SIZE) || (nInputSize == 0))
	{
		
		return MOREX_ERROR;
	}
 
 
	nOffset = stFileHandler->nSize - (stFileHandler->nOffset + nInputSize);
 
 
	if (nOffset < 0)
	{
		nSize = stFileHandler->nSize - stFileHandler->nOffset;
	}
	else
	{
		nSize = nInputSize;
	}
 
 
 
 
	nRSize = (stFileHandler->nMyClusterCurrentOffset & MOREX_SECTOR_SIZE_MASK) + nSize;
 
 
	nRSize = TFFN_Cdiv(nRSize, MOREX_SECTOR_SIZE);
 
 
	nOrigMyClsStart = stFileHandler->nMyClusterCurrentStart;
	nOrigMyClsOffset = stFileHandler->nMyClusterCurrentOffset;
	nOrigOffset = stFileHandler->nOffset;
 
 
 
 
	nReadTotalSectors = nRSize;
	
 
	nCheckOption = stFileHandler->nMyClusterCurrentOffset + nSize;
 
 
 
	if (nCheckOption <= (stVolumeInfo.nSectorPerClt * MOREX_SECTOR_SIZE))
	{
		
		nStartSector = stFileHandler->nMyClusterCurrentOffset >> MOREX_SECTOR_SIZE_BITS;
 
		
		nStartSector += _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
 
		nNumStartSectors = nRSize;
 
		nBufferOffset = stFileHandler->nMyClusterCurrentOffset & MOREX_SECTOR_SIZE_MASK;
 
		
		if (nNumStartSectors > 1)
		{
			
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			MOREX_memcpy(szTempBuff, szBuff + nBufferOffset, (MOREX_SECTOR_SIZE - nBufferOffset));
			szTempBuff += (MOREX_SECTOR_SIZE - nBufferOffset);
			nStartSector += 1;
 
			if (nNumStartSectors > 2)
			{
				
				nRe = TFFN_Read_Sectors(nStartSector, (INT32)nNumStartSectors - 2, szTempBuff);
				if (nRe == MOREX_EIO)
				{
					goto ERROR_OUT;
				}
				szTempBuff += ((nNumStartSectors - 2) << MOREX_SECTOR_SIZE_BITS);
				nStartSector += (nNumStartSectors - 2);
			}
 
			
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			if (nNumStartSectors > 2)
			{
				MOREX_memcpy(szTempBuff, szBuff, 
					(nSize - (MOREX_SECTOR_SIZE - nBufferOffset) - ((nNumStartSectors - 2) << MOREX_SECTOR_SIZE_BITS)));
			}
			else
			{
				MOREX_memcpy(szTempBuff, szBuff, (nSize - (MOREX_SECTOR_SIZE - nBufferOffset)));
			}
		}
		else
		{
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
 
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			MOREX_memcpy(szTempBuff, szBuff + nBufferOffset, nSize);
		}
 
		*nReadSize = nSize;
		goto SAVE_INFO;
	}
 
 
 
	else if (nCheckOption <= ((stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS) << 1))
	{
		bReadFlag = true;
 
		goto READ_HEAD;
	}
 
 
	else
	{
		goto READ_HEAD;
	}
 
 
READ_HEAD:
 
 
 
 
 
 
	if (stFileHandler->nMyClusterCurrentOffset == (stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS))
	{
		
		*nReadSize = 0;
 
		nNumStartSectors = 0;
 
		
		if (bReadFlag == true)
		{
			nNumBodyClusters = 0;
			goto READ_TAIL;
		}
		else
		{
			goto READ_BODY;
		}
	}
	else
	{
		nStartSector = stFileHandler->nMyClusterCurrentOffset >> MOREX_SECTOR_SIZE_BITS;
 
		nNumStartSectors = stVolumeInfo.nSectorPerClt - nStartSector;
 
		
		
		nStartSector += _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
 
		nBufferOffset = stFileHandler->nMyClusterCurrentOffset & MOREX_SECTOR_SIZE_MASK;
 
		
		if (nNumStartSectors > 1)
		{
			
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			MOREX_memcpy(szTempBuff, szBuff + nBufferOffset, (MOREX_SECTOR_SIZE - nBufferOffset));
			szTempBuff += (MOREX_SECTOR_SIZE - nBufferOffset);
			nStartSector += 1;
 
			
			nRe = TFFN_Read_Sectors(nStartSector, (INT32)(nNumStartSectors - 1), szTempBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
			szTempBuff += ((nNumStartSectors - 1) << MOREX_SECTOR_SIZE_BITS);
		}
		else
		{
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
 
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			MOREX_memcpy(szTempBuff, szBuff + nBufferOffset, (MOREX_SECTOR_SIZE - nBufferOffset));
			szTempBuff += (stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS) - stFileHandler->nMyClusterCurrentOffset;
		}
 
		*nReadSize = (stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS) - stFileHandler->nMyClusterCurrentOffset;
 
		
		if (bReadFlag == true)
		{
			nNumBodyClusters = 0;
			goto READ_TAIL;
		}
		else
		{
			goto READ_BODY;
		}
	}
 
READ_BODY:
 
 
	nRe = _Get_Next_FAT_Value(stFileHandler->nMyClusterCurrentStart, &nNextCluster);
 
	if (nRe == MOREX_ERROR)
	{
		goto ERROR_OUT;
	}
 
 
	stFileHandler->nMyClusterCurrentStart = nNextCluster;
 
 
 
	nNumBodyClusters = (nReadTotalSectors - nNumStartSectors) >> stVolumeInfo.nSectorPerCltBits;
 
 
	if (((*nReadSize) + ((nNumBodyClusters << stVolumeInfo.nSectorPerCltBits) << MOREX_SECTOR_SIZE_BITS)) > nSize)
	{
		
		nNumBodyClusters -= 1;
	}
 
	if (nNumBodyClusters == 1)
	{
		
		goto CONT;
	}
 
 
	nContinuousCluster = _Check_FAT_Chain_Continuous(nNextCluster, nNumBodyClusters);
 
	if (nContinuousCluster == MOREX_FAT_CHAIN_CONT) 
	{
CONT:
		
		nStartSector = _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);

		for(idx = 0; idx < nNumBodyClusters; idx++)
		{
			nRe = TFFN_Read_Sectors(nStartSector + (idx *stVolumeInfo.nSectorPerClt), stVolumeInfo.nSectorPerClt, szTempBuff);
 
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
 
		
		*nReadSize += (nRe << MOREX_SECTOR_SIZE_BITS);
 
		
			szTempBuff += ((stVolumeInfo.nSectorPerClt) << MOREX_SECTOR_SIZE_BITS);
		}
  
  
  
 
		
		stFileHandler->nMyClusterCurrentStart += (nNumBodyClusters - 1);
	}
	else
	{
		
		if (nContinuousCluster >= 0)
		{
			if (nContinuousCluster >0)
			{
				
				
				nStartSector = _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
				nRe = TFFN_Read_Sectors(nStartSector, (nContinuousCluster << stVolumeInfo.nSectorPerCltBits), szTempBuff);
 
				if (nRe == MOREX_EIO)
				{
					goto ERROR_OUT;
				}
 
				
				*nReadSize += (nRe << MOREX_SECTOR_SIZE_BITS);
 
				
				szTempBuff += ((nContinuousCluster << stVolumeInfo.nSectorPerCltBits) << MOREX_SECTOR_SIZE_BITS);
 
				
				stFileHandler->nMyClusterCurrentStart += (nContinuousCluster - 1);
 
				
				nRe = _Get_Next_FAT_Value(stFileHandler->nMyClusterCurrentStart, &nNextCluster);
				if (nRe == MOREX_ERROR)
				{
					goto ERROR_OUT;
				}
				stFileHandler->nMyClusterCurrentStart = nNextCluster;
				
			}
 
			
			
			for (i = 0; i < (INT32)(nNumBodyClusters - nContinuousCluster); i++)
			{
				nStartSector = _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
				nRe = TFFN_Read_Sectors(nStartSector, stVolumeInfo.nSectorPerClt, szTempBuff);
 
				if (nRe == MOREX_EIO)
				{
					goto ERROR_OUT;
				}
 
				
				*nReadSize += (nRe << MOREX_SECTOR_SIZE_BITS);
 
				
				nRe = _Get_Next_FAT_Value(stFileHandler->nMyClusterCurrentStart, &nNextCluster);
 
				if (nRe == MOREX_ERROR)
				{
					goto ERROR_OUT;
				}
 
				
				if (i != (INT32)(nNumBodyClusters - nContinuousCluster - 1))
				{
					
					stFileHandler->nMyClusterCurrentStart = nNextCluster;
				}
 
				
				szTempBuff +=  (stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS);
			} 
			
		}
		else 
		{
			goto ERROR_OUT;
		}
	}
 
READ_TAIL:
 
 
	nNumEndSectors = nReadTotalSectors - (nNumStartSectors + (nNumBodyClusters << stVolumeInfo.nSectorPerCltBits));
	if (nNumEndSectors == 0)
	{
		goto SAVE_INFO;
	}
 
 
	nRe = _Get_Next_FAT_Value(stFileHandler->nMyClusterCurrentStart, &nNextCluster);
 
	if (nRe == MOREX_ERROR)
	{
		goto ERROR_OUT;
	}
 
 
	stFileHandler->nMyClusterCurrentStart = nNextCluster;
 
 
	nStartSector = _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
	if (nNumEndSectors > 1)
	{
		
		nRe = TFFN_Read_Sectors(nStartSector, nNumEndSectors - 1, szTempBuff);
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
 
		szTempBuff += ((nNumEndSectors - 1) << MOREX_SECTOR_SIZE_BITS);
		*nReadSize += ((nNumEndSectors - 1) << MOREX_SECTOR_SIZE_BITS);
		nStartSector += nNumEndSectors - 1;
 
		
		nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
 
		MOREX_memcpy(szTempBuff, szBuff, (nSize - (*nReadSize)));
	} 
	else
	{
		nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
 
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
 
		MOREX_memcpy(szTempBuff, szBuff, (nSize - (*nReadSize)));
	} 
 
	*nReadSize = nSize;
 
SAVE_INFO:
 
	stFileHandler->nMyClusterCurrentOffset += nSize;
	stFileHandler->nMyClusterCurrentOffset &= ((stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS) - 1);
	if (stFileHandler->nMyClusterCurrentOffset == 0)
	{
		stFileHandler->nMyClusterCurrentOffset = stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS;
	}
 
	stFileHandler->nOffset += nSize;
 
	return MOREX_OK;
 
ERROR_OUT:
 
	stFileHandler->nMyClusterCurrentStart = nOrigMyClsStart;
	stFileHandler->nMyClusterCurrentOffset = nOrigMyClsOffset;
	stFileHandler->nOffset = nOrigOffset;
 
	return MOREX_ERROR;
}
 
/**
* 
* @name			TFFN_FS_Write
*
* @description	This function write data to a file 
*
* @param		stFileHandler	pointer to the file handler
*				nSize			data size to write
*				szBuffer		buffer pointer for the written data
*				nWritten		data size to be written
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		
*				YoungWon Yun	(youngwon.yun@samsung.com)
* @date			
*
* @revision history	
*				
*
**/
INT32
TFFN_FS_Write(t_file_handler *stFileHandler, UINT32 nSize, UINT32 *szBuffer, UINT32 *nWrittenSize)
{
	UINT32	nNewSize, nCurSize, nAllocClusters, nCurCls, nNewCls, nCurRem, nNewRem;
	UINT32	nNextCls, nCheckOption, nNextCluster;
	UINT32	nWriteTotalSectors, nTotalAllocClusters;
	UINT32	nStartSector, nFATsector;
	UINT32	nNumStartSectors, nNumBodyClusters, nNumEndSectors, nRSize;
	INT32	nRe, i, nContinuousCluster, nBufferOffset;
	UINT8	*szTempBuff, *szTempFAT, *szBuff;
	UINT32	nOrigMyClsOffset, nOrigMyClsStart, nOrigOffset;
	BOOL32	bWriteFlag = false;
 
	t_dir_entry	*stDirEnt;
	UINT32	nCurClsPDir, nCurSecPDir, nDirIndex;
 
	szTempBuff = (UINT8 *)szBuffer;
	
	szTempFAT = (UINT8 *)gaFatTable;
 
	szBuff = (UINT8 *)gaBuffer;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((bInitialized == false) || (szBuffer == NULL) || (nWrittenSize == NULL) || (stFileHandler == NULL))
	{
		return MOREX_ERROR;
	}
#endif
 
	if (stFileHandler->nIsDir == 1)
	{
		return MOREX_ERROR;
	}
 
 
 
	nCurSize = stFileHandler->nSize;
	nNewSize = stFileHandler->nOffset + nSize;
 
	if ((nNewSize > MOREX_MAX_FILE_SIZE) || (nSize == 0))
	{
		
		return MOREX_ERROR;
	}
 
 
 
	if (stFileHandler->nSize >= nNewSize)
	{
		
	}
	else
	{
		
		
		
		
		
		nCurCls = (nCurSize >> MOREX_SECTOR_SIZE_BITS) >> stVolumeInfo.nSectorPerCltBits;
		nNewCls = (nNewSize >> MOREX_SECTOR_SIZE_BITS) >> stVolumeInfo.nSectorPerCltBits;
 
 
		
		nCurRem = nCurSize - ((nCurCls << stVolumeInfo.nSectorPerCltBits) << MOREX_SECTOR_SIZE_BITS);
		nNewRem = nNewSize - ((nNewCls << stVolumeInfo.nSectorPerCltBits) << MOREX_SECTOR_SIZE_BITS);
 
		if (nCurSize == 0)
		{
			
			
			
			if (nNewRem == 0)
			{
				nAllocClusters = nNewCls - 1;
			}
			else 
			{
				nAllocClusters = nNewCls;
			}
		}
		else
		{
			
			if (nCurRem == 0)
			{
				nCurCls -= 1;
			}
 
			
			if (nNewRem == 0)
			{
				nNewCls -= 1;
			}
 
			
			
			
			
			
			
			
			
			
			
			nAllocClusters = nNewCls - nCurCls;
		}
 
		
		
		nCurCls = stFileHandler->nMyClusterCurrentStart;
 
		
		
		if (stFileHandler->nSize == 0)
		{
			nTotalAllocClusters = nAllocClusters + 1;
		}
		else
		{
			nTotalAllocClusters = nAllocClusters;
		}
 
		if (nTotalAllocClusters > 0)
		{
			
			
			nFATsector = nCurFATsector;
			for (i = 0; i < (INT32)nTotalAllocClusters; i++)
			{
				nRe = _Get_Free_FAT_Value(nCurCls, &nNextCluster);
				if (nRe == MOREX_ERROR)
				{
					
					return MOREX_ERROR;
				}
 
				
				if (i == 0)
				{
					nNewCls = nNextCluster;
				}
				
				nRe = _Update_FAT(nNextCluster, -1, false);
 
				nCurCls = nNextCluster;
 
				if ((i != 0) && (nNextCluster == nNewCls) && (i != (INT32)nTotalAllocClusters))
				{
					return MOREX_ERROR;
				}
			} 
 
			
			nCurFATsector = nFATsector;
			nRe = TFFN_Read_Sectors(nCurFATsector, 1, szTempFAT);
			if((nRe < 0) || (nRe != 1))
			{
				return MOREX_ERROR;
			}
 
			
			
			
			
			if ((stFileHandler->nSize == 0) && (stFileHandler->nMyCluster == 0))
			{
				
				
				nRe = _Get_Free_FAT_Value(nNewCls, &nNextCls);
 
				if (nRe == MOREX_ERROR)
				{
					
					return MOREX_ERROR;
				}
 
				nRe = _Update_FAT(nNextCls, -1, true);
				if (nRe == MOREX_ERROR)
				{
					return MOREX_ERROR;
				}
 
 
				
				stFileHandler->nMyCluster = nNextCls;
				stFileHandler->nMyClusterCurrentStart = nNextCls;
				stFileHandler->nMyClusterCurrentOffset = 0;
				stFileHandler->nOffset = 0;
				stFileHandler->nSize = 0;
 
				
				
				
			} 
			
 
			
			if (nAllocClusters > 0)
			{
				nRe = _Allocate_Cluster(stFileHandler->nMyClusterCurrentStart, &nNextCls, nAllocClusters);
				if (nRe == MOREX_ERROR)
				{
					return MOREX_ERROR;
				}
			}
		} 
	} 
 
 
 
	MOREX_memset((UINT8 *)gaBuffer, 0, 512);
	szBuff = (UINT8 *)gaBuffer;
 
 
 
 
	nRSize = (stFileHandler->nMyClusterCurrentOffset & MOREX_SECTOR_SIZE_MASK) + nSize;
 
 
	nRSize = TFFN_Cdiv(nRSize, MOREX_SECTOR_SIZE);
 
 
	nOrigMyClsStart = stFileHandler->nMyClusterCurrentStart;
	nOrigMyClsOffset = stFileHandler->nMyClusterCurrentOffset;
	nOrigOffset = stFileHandler->nOffset;
 
 
 
 
 
 
	nWriteTotalSectors = nRSize;
 
	 
	nCheckOption = stFileHandler->nMyClusterCurrentOffset + nSize;
 
 
 
	if (nCheckOption <= (stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS))
	{
		
		nStartSector = stFileHandler->nMyClusterCurrentOffset >> MOREX_SECTOR_SIZE_BITS;
 
		
		nStartSector += _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
 
		nNumStartSectors = nRSize;
 
		nBufferOffset = stFileHandler->nMyClusterCurrentOffset & MOREX_SECTOR_SIZE_MASK;
 
		
		if (nNumStartSectors > 1)
		{
			
			
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
			
			
			MOREX_memcpy(szBuff + nBufferOffset, szTempBuff, (MOREX_SECTOR_SIZE - nBufferOffset));
			nRe = TFFN_Write_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
			szTempBuff += (MOREX_SECTOR_SIZE - nBufferOffset);
			nStartSector += 1;
 
			if (nNumStartSectors > 2)
			{
				
				nRe = TFFN_Write_Sectors(nStartSector, (INT32)nNumStartSectors - 2, szTempBuff);
				if (nRe == MOREX_EIO)
				{
					goto ERROR_OUT;
				}
 
				szTempBuff += ((nNumStartSectors - 2) << MOREX_SECTOR_SIZE_BITS);
				nStartSector += (nNumStartSectors - 2);
			}
 
			
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			
			if (nNumStartSectors > 2)
			{
				MOREX_memcpy(szBuff, szTempBuff, 
					(nSize - (MOREX_SECTOR_SIZE - nBufferOffset) - ((nNumStartSectors - 2) << MOREX_SECTOR_SIZE_BITS)));
			}
			else
			{
				MOREX_memcpy(szBuff, szTempBuff, (nSize - (MOREX_SECTOR_SIZE - nBufferOffset)));
			}
			nRe = TFFN_Write_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
		}
		else
		{
			
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			MOREX_memcpy(szBuff + nBufferOffset, szTempBuff, nSize);
			nRe = TFFN_Write_Sectors(nStartSector, 1, szBuff);
 
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
		}
 
		*nWrittenSize = nSize;
 
		goto SAVE_INFO;
	}
 
 
 
	else if (nCheckOption <= ((stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS) << 1))
	{
		bWriteFlag = true;
 
		goto WRITE_HEAD;
	}
 
 
 
	else
	{
		goto WRITE_HEAD;
	}
 
WRITE_HEAD:
 
 
	MOREX_memset((UINT8 *)gaBuffer, 0, 512);
	szBuff = (UINT8 *)gaBuffer;
 
 
	if (stFileHandler->nMyClusterCurrentOffset == (stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS))
	{
		
		
		
		
 
		
		*nWrittenSize = 0;
		nNumStartSectors = 0;
 
		
		if (bWriteFlag == true)
		{
			nNumBodyClusters = 0;
			goto WRITE_TAIL;
		}
		else
		{
			goto WRITE_BODY;
		}
	}
	else
	{
		nStartSector = stFileHandler->nMyClusterCurrentOffset >> MOREX_SECTOR_SIZE_BITS;
 
		nNumStartSectors = stVolumeInfo.nSectorPerClt - nStartSector;
 
		
		
		nStartSector += _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
 
		nBufferOffset = stFileHandler->nMyClusterCurrentOffset & MOREX_SECTOR_SIZE_MASK;
 
		
		if (nNumStartSectors > 1)
		{
			
			
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			MOREX_memcpy(szBuff + nBufferOffset, szTempBuff, (MOREX_SECTOR_SIZE - nBufferOffset));
			nRe = TFFN_Write_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			szTempBuff += (MOREX_SECTOR_SIZE - nBufferOffset);
			nStartSector += 1;
 
			
			nRe = TFFN_Write_Sectors(nStartSector, (INT32)(nNumStartSectors - 1), szTempBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			szTempBuff += ((nNumStartSectors - 1) << MOREX_SECTOR_SIZE_BITS);
		}
		else
		{
			
			nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			MOREX_memcpy(szBuff + nBufferOffset, szTempBuff, (MOREX_SECTOR_SIZE - nBufferOffset));
			nRe = TFFN_Write_Sectors(nStartSector, 1, szBuff);
 
			if (nRe == MOREX_EIO)
			{
				goto ERROR_OUT;
			}
 
			szTempBuff += ((stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS) - stFileHandler->nMyClusterCurrentOffset);
		}
 
		*nWrittenSize = (stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS) - stFileHandler->nMyClusterCurrentOffset;
 
		
		if (bWriteFlag == true)
		{
			nNumBodyClusters = 0;
			goto WRITE_TAIL;
		}
		else
		{
			goto WRITE_BODY;
		}
	}
 
WRITE_BODY:
 
 
	nRe = _Get_Next_FAT_Value(stFileHandler->nMyClusterCurrentStart, &nNextCluster);
 
	if (nRe == MOREX_ERROR)
	{
		goto ERROR_OUT;
	}
 
 
	stFileHandler->nMyClusterCurrentStart = nNextCluster;
 
 
 
	nNumBodyClusters = (nWriteTotalSectors - nNumStartSectors) >> stVolumeInfo.nSectorPerCltBits;
 
 
	if (((*nWrittenSize) + ((nNumBodyClusters << stVolumeInfo.nSectorPerCltBits) << MOREX_SECTOR_SIZE_BITS)) > nSize)
	{
		
		nNumBodyClusters -= 1;
	}
 
	if (nNumBodyClusters == 1)
	{
		
		goto CONT;
	}
 
 
	nContinuousCluster = _Check_FAT_Chain_Continuous(nNextCluster, nNumBodyClusters);
 
	if (nContinuousCluster == MOREX_FAT_CHAIN_CONT) 
	{
CONT:
		
		nStartSector = _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
		nRe = TFFN_Write_Sectors(nStartSector, (nNumBodyClusters << stVolumeInfo.nSectorPerCltBits), szTempBuff);
 
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
 
		
		*nWrittenSize += (nRe << MOREX_SECTOR_SIZE_BITS);
 
		
		szTempBuff += ((nNumBodyClusters << stVolumeInfo.nSectorPerCltBits) << MOREX_SECTOR_SIZE_BITS);
 
		
		stFileHandler->nMyClusterCurrentStart += (nNumBodyClusters - 1);
	} 
	else
	{
		
		if (nContinuousCluster >= 0)
		{
			if (nContinuousCluster >0)
			{
				
				
				nStartSector = _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
				nRe = TFFN_Write_Sectors(nStartSector, (nContinuousCluster << stVolumeInfo.nSectorPerCltBits), szTempBuff);
				if (nRe == MOREX_EIO)
				{
					goto ERROR_OUT;
				}
 
				
				*nWrittenSize += (nRe << MOREX_SECTOR_SIZE_BITS);
 
				
				szTempBuff += ((nContinuousCluster << stVolumeInfo.nSectorPerCltBits) << MOREX_SECTOR_SIZE_BITS);
 
				
				stFileHandler->nMyClusterCurrentStart += (nContinuousCluster - 1);
 
				
				nRe = _Get_Next_FAT_Value(stFileHandler->nMyClusterCurrentStart, &nNextCluster);
				if (nRe == MOREX_ERROR)
				{
					goto ERROR_OUT;
				}
				stFileHandler->nMyClusterCurrentStart = nNextCluster;
 
				
			}
 
			
			
			for (i = 0; i < (INT32)(nNumBodyClusters - nContinuousCluster); i++)
			{
				nStartSector = _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
				nRe = TFFN_Write_Sectors(nStartSector, stVolumeInfo.nSectorPerClt, szTempBuff);
 
				if (nRe == MOREX_EIO)
				{
					goto ERROR_OUT;
				}
 
				
				*nWrittenSize += (nRe << MOREX_SECTOR_SIZE_BITS);
 
				
				nRe = _Get_Next_FAT_Value(stFileHandler->nMyClusterCurrentStart, &nNextCluster);
 
				if (nRe == MOREX_ERROR)
				{
					goto ERROR_OUT;
				}
 
				
				if (i != (INT32)(nNumBodyClusters - nContinuousCluster - 1))
				{
					
					stFileHandler->nMyClusterCurrentStart = nNextCluster;
				}
 
				
				szTempBuff +=  (stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS);
			}
			
		} 
		else 
		{
			goto ERROR_OUT;
		}
	} 
 
WRITE_TAIL:
	MOREX_memset((UINT8 *)gaBuffer, 0, 512);
	szBuff = (UINT8 *)gaBuffer;
 
 
	nNumEndSectors = nWriteTotalSectors - (nNumStartSectors + (nNumBodyClusters << stVolumeInfo.nSectorPerCltBits));
 
	if (nNumEndSectors == 0)
	{
		goto SAVE_INFO;
	}
 
 
	nRe = _Get_Next_FAT_Value(stFileHandler->nMyClusterCurrentStart, &nNextCluster);
 
	if (nRe == MOREX_ERROR)
	{
		goto ERROR_OUT;
	}
 
 
	stFileHandler->nMyClusterCurrentStart = nNextCluster;
 
 
	nStartSector = _Sector_Num_of_Cluster(&stVolumeInfo, stFileHandler->nMyClusterCurrentStart);
	if (nNumEndSectors > 1)
	{
		
		nRe = TFFN_Write_Sectors(nStartSector, nNumEndSectors - 1, szTempBuff);
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
 
		szTempBuff += ((nNumEndSectors - 1) << MOREX_SECTOR_SIZE_BITS);
		*nWrittenSize += ((nNumEndSectors - 1) << MOREX_SECTOR_SIZE_BITS);
		nStartSector += nNumEndSectors - 1;
 
		
		
		nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
		MOREX_memcpy(szBuff, szTempBuff, (nSize - (*nWrittenSize)));
		nRe = TFFN_Write_Sectors(nStartSector, 1, szBuff);
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
	} 
	else
	{
		
		nRe = TFFN_Read_Sectors(nStartSector, 1, szBuff);
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
 
		MOREX_memcpy(szBuff, szTempBuff, (nSize - (*nWrittenSize)));
		nRe = TFFN_Write_Sectors(nStartSector, 1, szBuff);
 
		if (nRe == MOREX_EIO)
		{
			goto ERROR_OUT;
		}
	} 
 
	*nWrittenSize = nSize;
 
SAVE_INFO:
 
 
	stFileHandler->nMyClusterCurrentOffset += nSize;
	stFileHandler->nMyClusterCurrentOffset &= ((stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS) - 1);
	if (stFileHandler->nMyClusterCurrentOffset == 0)
	{
		stFileHandler->nMyClusterCurrentOffset = stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS;
	}
 
	stFileHandler->nOffset += nSize;
 
 
 
	if (stFileHandler->nOffset <= stFileHandler->nSize)
	{
		goto END;
	}
	else
	{
		
		stFileHandler->nSize = stFileHandler->nOffset;
	}
 
 
	nCurClsPDir = stFileHandler->nParentClusterStart;
	nCurSecPDir = stFileHandler->nParentClusterOffset & ((MOREX_DIR_ENTRY_PER_SECTOR << stVolumeInfo.nSectorPerCltBits) - 1);
	nCurSecPDir = nCurSecPDir >> MOREX_DIR_ENTRY_PER_SECTOR_BITS;
	nCurSecPDir = _Sector_Num_of_Cluster(&stVolumeInfo, nCurClsPDir) + nCurSecPDir;
	nDirIndex = stFileHandler->nParentClusterOffset & MOREX_DIR_ENTRY_PER_SECTOR_MASK;
 
 
	nRe = _Read_Dir_Entry(nCurClsPDir, nCurSecPDir);
	if (nRe < 0)
	{
		return MOREX_ERROR;
	}
	
#ifdef	_USE_UCMALLOC
	stDirEnt = &pstDirEntsCache->aDirEnts[nDirIndex];
#else 
	stDirEnt = &stDirEntsCache.aDirEnts[nDirIndex];
 #endif
 
 
 
	if (stVolumeInfo.nFATType == MOREX_FAT16)
	{
		stDirEnt->nDirFstClusHi = 0;
		stDirEnt->nDirFstClusLo = TFFN_BO_uint16(stFileHandler->nMyCluster);
	}
	else
	{
		stDirEnt->nDirFstClusHi = TFFN_BO_uint16((stFileHandler->nMyCluster & 0xffff0000) >> MOREX_DIR_ENTRY_PER_SECTOR);
		stDirEnt->nDirFstClusLo = TFFN_BO_uint16(stFileHandler->nMyCluster & 0xffff);
	}
 
	stDirEnt->nDirFileSize = TFFN_BO_uint32(stFileHandler->nSize);
 
 
	nRe = _Write_Dir_Entry();
 
	if (nRe == MOREX_EIO)
	{
		return MOREX_ERROR;
	}
 
 
 
END:
	return MOREX_OK;
 
ERROR_OUT:
 
	stFileHandler->nMyClusterCurrentStart = nOrigMyClsStart;
	stFileHandler->nMyClusterCurrentOffset = nOrigMyClsOffset;
	stFileHandler->nOffset = nOrigOffset;
 
	return MOREX_ERROR;
}
 
/**
* 
* Move file pointer
* 
* @param		stFileHandler		File Handler structure
				nOffset				Offset
				nWhence				
									MOREX_SEEK_SET : Offset from start
									MOREX_SEEK_CUR : Offset from the current position
									MOREX_SEEK_END : Offset from the end of file
* @return		0 >=  : Success
				< 0   : Failure
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
TFFN_FS_Lseek(t_file_handler *stFileHandler, INT32 nOffset, INT32 nWhence)
{
	UINT32	nCluster, nRem, nNextCluster;
	INT32	i, nRe, nPos;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((bInitialized == false) || (stFileHandler == NULL))
	{
		return MOREX_ERROR;
	}
#endif
 
    if (stFileHandler->nIsDir == 1)
	{
		return MOREX_ERROR;
	}
 
	if (nWhence == MOREX_SEEK_SET) 
	{
		if (nOffset < 0)
		{
			nPos = 0;
		}
		else
		{
			nPos = nOffset;
		}
	}
	else if (nWhence == MOREX_SEEK_CUR) 
	{
		if (nOffset >= 0)
		{
			nPos = stFileHandler->nOffset + nOffset;
			if (((UINT32)nPos < stFileHandler->nOffset) || ((UINT32)nPos < (UINT32)nOffset))
			{
				nPos = MOREX_MAX_FILE_SIZE;
			}
		}
		else
		{
			nOffset = nOffset * (-1);
			if (stFileHandler->nOffset <= (UINT32)nOffset)
			{
				nPos = 0;
			}
			else
			{
				nPos = stFileHandler->nOffset - nOffset;
			}
		}
	}
	else if (nWhence == MOREX_SEEK_END) 
	{
		if (nOffset >= 0)
		{
			return MOREX_ERROR;
		}
		else
		{
			nPos = stFileHandler->nSize + nOffset;
		}
	}
	else
	{
		return MOREX_ERROR;
	}
 
 
	if ((nPos < 0) || (nPos > (INT32)stFileHandler->nSize))
	{
		return MOREX_ERROR;
	}
 
	if (nPos > MOREX_MAX_FILE_SIZE)
	{
		nPos = MOREX_MAX_FILE_SIZE;
	}
 
	stFileHandler->nOffset = nPos;
 
 
 
 
 
	nCluster = nPos / (stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS);
	
 
	nRem = nPos & ((stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS) - 1);

 
 
	if ((nRem == 0) && (nPos != 0))
	{
		
		
		stFileHandler->nMyClusterCurrentOffset = stVolumeInfo.nSectorPerClt << MOREX_SECTOR_SIZE_BITS;
		nCluster -= 1;
	}
	else
	{
		stFileHandler->nMyClusterCurrentOffset = nRem;
	}
 
 
	stFileHandler->nMyClusterCurrentStart = stFileHandler->nMyCluster;
 
 
 
	if (nCluster > 0)
	{
		for (i = 0; i < (INT32)nCluster; i++)
		{
			
			nNextCluster = 0;
 
			nRe = _Get_Next_FAT_Value(stFileHandler->nMyClusterCurrentStart, &nNextCluster);
 
			if (nRe == MOREX_ERROR)
			{
				return MOREX_ERROR;
			}
 
			stFileHandler->nMyClusterCurrentStart = nNextCluster;
		}
	}
 
	return MOREX_OK;
}
 
/**
* 
* Get Next Free FAT Value
* 
* @param		nStartCluster		Start Cluster
*				nFreeCluster		Pointer to the free cluster
* @return		0		:	Success
				< 0		:	Failure
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32 
_Get_Free_FAT_Value(UINT32 nStartCluster, UINT32 *nFreeCluster)
{
	INT32 nRe;
	UINT32 nFATsector;
	UINT32 nIndex;
	UINT8 *pstFATBuffer;
	INT32 i, j;
 
	pstFATBuffer = (UINT8*)gaFatTable;
 
	for (i = nStartCluster; i <= (INT32)(stVolumeInfo.nTotalClusters + 1); i++)
	{		
		nFATsector = stVolumeInfo.nFAT1Addr + (i >> stVolumeInfo.nClusterPerSectorBits);
		
		if (nFATsector != nCurFATsector)
		{
			
			if (bChangeFAT == true)
			{
				
				
				nRe = TFFN_Write_Sectors(nCurFATsector, 1, pstFATBuffer);
				if (nRe == MOREX_EIO)
				{
					return MOREX_ERROR;
				}
 
				
				
				nRe = TFFN_Write_Sectors((stVolumeInfo.nFAT2Addr + (nCurFATsector - stVolumeInfo.nFAT1Addr)) , 1, pstFATBuffer);
				if (nRe == MOREX_EIO)
				{
					return MOREX_ERROR;
				}
				bChangeFAT = false;
			}
 
			nRe = TFFN_Read_Sectors(nFATsector, 1, pstFATBuffer);
			if ((nRe < 0) || (nRe != 1))
			{
				return MOREX_ERROR;
			}
			else
			{
				nCurFATsector = nFATsector;		
			}		
		}
 
		j = i & (stVolumeInfo.nClusterPerSector - 1);
 
		if (stVolumeInfo.nFATType == MOREX_FAT16)
		{
			nIndex = j * 2;
			*nFreeCluster = ((*(pstFATBuffer + nIndex) | ( *(pstFATBuffer + nIndex + 1) << 8)) & 0x0000ffff);
 
			if (*nFreeCluster == 0)
			{					
				*nFreeCluster = i;
				return MOREX_OK;
			}
		}
		else
		{
			nIndex = j * 4;
			*nFreeCluster = ((*(pstFATBuffer + nIndex) | (*(pstFATBuffer + nIndex + 1) << 8) 
				| (*(pstFATBuffer + nIndex + 2) << 16) | (*(pstFATBuffer + nIndex + 3) << 24)) & 0xffffffff);
 
			if (*nFreeCluster == 0)
			{
				*nFreeCluster = i;
				return MOREX_OK;
			} 
		} 
	}
 
	for (i = 2; i < (INT32)nStartCluster; i++)
	{		
		nFATsector = stVolumeInfo.nFAT1Addr + (i >> stVolumeInfo.nClusterPerSectorBits);
		
		if (nFATsector != nCurFATsector)
		{
			
			if (bChangeFAT == true)
			{
				
				
				nRe = TFFN_Write_Sectors(nCurFATsector, 1, pstFATBuffer);
				if (nRe == MOREX_EIO)
				{
					return MOREX_ERROR;
				}
 
				
				
				nRe = TFFN_Write_Sectors((stVolumeInfo.nFAT2Addr + (nCurFATsector - stVolumeInfo.nFAT1Addr)) , 1, pstFATBuffer);
				if (nRe == MOREX_EIO)
				{
					return MOREX_ERROR;
				}
				bChangeFAT = false;
			}
 
			nRe = TFFN_Read_Sectors(nFATsector, 1, pstFATBuffer);
			if ((nRe < 0) || (nRe != 1))
			{
				return MOREX_ERROR;
			}
			else
			{
				nCurFATsector = nFATsector;		
			}		
		}
 
		j = i & (stVolumeInfo.nClusterPerSector - 1);
 
		if (stVolumeInfo.nFATType == MOREX_FAT16)
		{
			nIndex = j * 2;
			*nFreeCluster = ((*(pstFATBuffer + nIndex) | ( *(pstFATBuffer + nIndex + 1) << 8)) & 0x0000ffff);
 
			if (*nFreeCluster == 0)
			{					
				*nFreeCluster = i;
				return MOREX_OK;
			}
		}
		else
		{
			nIndex = j * 4;
			*nFreeCluster = ((*(pstFATBuffer + nIndex) | (*(pstFATBuffer + nIndex + 1) << 8) 
				| (*(pstFATBuffer + nIndex + 2) << 16) | (*(pstFATBuffer + nIndex + 3) << 24)) & 0xffffffff);
 
			if (*nFreeCluster == 0)
			{
				*nFreeCluster = i;
				return MOREX_OK;
			} 
		} 
	}
 
	return MOREX_ERROR;
}
/**
* 
* Check if FAT Chain is continuous
* 
* @param		nStartCluster			Start Cluster
				nNumBodyClusters			number of Clusters
* @return		0 >						:	Partial Success
				MOREX_ERROR				:	Failure
				MOREX_FAT_CHAIN_CONT	: all clusters are continuous
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
_Check_FAT_Chain_Continuous(UINT32 nStartCluster, UINT32 nNumBodyClusters)
{
	UINT32 nNextCluster, nClusters, i;
 	INT32 nRe;
 
 
	for (i = 0, nClusters = 1; i < nNumBodyClusters - 1; i++, nClusters++)
	{
		
		nRe = _Get_Next_FAT_Value(nStartCluster + i, &nNextCluster);
 
		if (nRe == MOREX_ERROR)
		{
			return MOREX_ERROR;
		}
		else
		{
			
			if ((nNextCluster - (nStartCluster + i)) != 1)
			{
				return nClusters;
			}
		}
	}
 
	return MOREX_FAT_CHAIN_CONT;
}
 
/**
* 
* Check if two sector in the same FAT sector
* 
* @param		nStartCluster			Start Cluster
				nNextCluster			Next Cluster
* @return		0						:	Success
				0 <						:	Failure
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
_Check_Same_FAT(UINT32 nStartCluster, UINT32 nNextCluster)
{
	UINT32 nFATSector1, nFATSector2;
 
	nFATSector1 = stVolumeInfo.nFAT1Addr + (nStartCluster >> stVolumeInfo.nClusterPerSectorBits);
	nFATSector2 = stVolumeInfo.nFAT1Addr + (nNextCluster >> stVolumeInfo.nClusterPerSectorBits);
 
	if (nFATSector1 == nFATSector2)
	{
		return MOREX_OK;
	}
	else
	{
		return MOREX_ERROR;
	}
}
 
/**
* 
* Allocate cluster & Link FAT chain
* 
* @param			nCurCls		Current Cluster
*					nNextCls	Pointer to the next cluster
*					nCls		Number of cluster to allocate
* @return		0 >						:	Success
				MOREX_ERROR				:	Failure
* @author		Young Won Yun (youngwon.yun@samsung.com)
* @version		1.0
* @see			
* @since		1.0
*/
INT32
_Allocate_Cluster(UINT32 nCurCls, UINT32 *nNextCls, UINT32 nCls)
{
	INT32	nRe, i, nNumClusters;
	UINT32	nCurCluster, nNextCluster, nFATsector;
	BOOL32	bWriteFlag = false;
	UINT8	*pstTempFAT;
	UINT32	nNumDataCluster;
 
	pstTempFAT = (UINT8 *)gaFatTable;
 
 
 
 
 
 
 
	nRe = _Get_Free_FAT_Value(nCurCls, &nNextCluster);
	if (nRe == MOREX_ERROR)
	{
		return MOREX_ERROR;
	}
 
 
	*nNextCls = nNextCluster;
 
 
	nRe = _Update_FAT(nNextCluster, -1, bWriteFlag);
	if (nRe == MOREX_ERROR)
	{
		return MOREX_ERROR;
	}
	bChangeFAT = true;
 
 
	if ((INT32) nCls > 1)
	{
		nCurCluster = nNextCluster;
 
		
		
		nNumClusters = (INT32)nCls - 1;
 
		for (i = 0; i < nNumClusters; i++)
		{
			nRe = _Get_Free_FAT_Value(nCurCluster, &nNextCluster);
			if (nRe == MOREX_ERROR)
			{
				return MOREX_ERROR;
			}
 
			
			
			
 
			
			
			
			nRe = _Check_Same_FAT(nCurCluster, nNextCluster);
			if (nRe == MOREX_ERROR)
			{
				nFATsector = stVolumeInfo.nFAT1Addr + 
					(nCurCluster >> stVolumeInfo.nClusterPerSectorBits);
 
				nRe = TFFN_Read_Sectors(nFATsector, 1, pstTempFAT);
				if (nRe == MOREX_EIO)
				{
					
					return MOREX_ERROR;
				}
				bWriteFlag = true;
			}
 
			nRe = _Update_FAT(nCurCluster, nNextCluster, bWriteFlag);
			if (nRe == MOREX_ERROR)
			{
				return MOREX_ERROR;
			}
			
 
 
			
			
			
			nRe = _Check_Same_FAT(nCurCluster, nNextCluster);
			if (nRe == MOREX_ERROR)
			{
				nFATsector = stVolumeInfo.nFAT1Addr + 
					(nNextCluster >> stVolumeInfo.nClusterPerSectorBits);
 
				nRe = TFFN_Read_Sectors(nFATsector, 1, pstTempFAT);
				if (nRe == MOREX_EIO)
				{
					
					return MOREX_ERROR;
				}
				
				
				
				bWriteFlag = false;
			}
 
			nRe = _Update_FAT(nNextCluster, -1, bWriteFlag);
			if (nRe == MOREX_ERROR)
			{
				return MOREX_ERROR;
			}
			bChangeFAT = true;
 
			nCurCluster = nNextCluster;
		} 
	} 
 
 
 
 
 
	if (bChangeFAT == true)
	{
		nRe = TFFN_Write_Sectors(nCurFATsector, 1, pstTempFAT);
		if (nRe == MOREX_EIO)
		{
			return MOREX_ERROR;
		}
 
		
		
		nRe = TFFN_Write_Sectors((stVolumeInfo.nFAT2Addr + (nCurFATsector - stVolumeInfo.nFAT1Addr)) , 1, pstTempFAT);
		if (nRe == MOREX_EIO)
		{
			return MOREX_ERROR;
		}
		bChangeFAT = false;
	}
 
 
 
	nCurCluster = nCurCls;
 
 
	nNumDataCluster = stVolumeInfo.nTotalClusters;
	i = 0;
	while (i < (INT32)nNumDataCluster)
	{
		
		nRe = _Get_Next_FAT_Value(nCurCluster, &nNextCluster);
 
		if (nRe == MOREX_ERROR)
		{
			return MOREX_ERROR;
		}
 
		
		if (stVolumeInfo.nFATType == MOREX_FAT16)
		{
			
			if (nNextCluster >= MOREX_FAT16_EOF)
			{
				
				nNextCluster = nCurCluster;
				break;
			}
		} 
		else
		{
			if (nNextCluster >= MOREX_FAT32_EOF)
			{
				
				nNextCluster = nCurCluster;
				break;
			}
		} 
		
		
		nCurCluster = nNextCluster;
		i++;
	} 
 
 
	if (i >= (INT32)nNumDataCluster)
	{
		return MOREX_ERROR;
	}
 
 
 
 
 
	nRe = _Update_FAT(nCurCluster, (*nNextCls), true);
	if (nRe == MOREX_ERROR)
	{
		return MOREX_ERROR;
	}
 
	return MOREX_OK;
}
 
 
/**
* 
* @name			TFFN_FS_OpenDir
*
* @description	This function opens a directory
*
* @param		szPath			pointer to the directory path
*				stDirHandler	pointer to the directory handler
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/10/14
*
* @revision history	
*				2005/10/14	[Ahn HyunJu]	first writing
*
**/
INT32
TFFN_FS_OpenDir(UINT16 *szPath, t_file_handler *stDirHandler)
{
	INT32	nReturn;
	UINT16	*tempPC;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((bInitialized == false) || (szPath == NULL) || (stDirHandler == NULL))
	{
		return MOREX_ERROR;
	}
 
	tempPC = szPath;
	while (*tempPC != '\0')
	{
		if ((*tempPC < ' ') || (*tempPC > '~'))
		{
			return MOREX_ERROR;
		}
		tempPC++;		
	}
#endif
 
	stDirHandler->nParentCluster = stVolumeInfo.nRootCluster;
	stDirHandler->nIsDir = 1;
 
	nReturn = _Path_Resolve(szPath, stDirHandler, MOREX_OPEN_MODE);
 
	if (nReturn < 0)
	{
		return MOREX_ERROR;
	}
 
	if (stDirHandler->nIsDir == 1)
	{		
		stDirHandler->nOffset = 0;	
		return MOREX_OK;
	}
   
	return MOREX_ERROR;
}
 
/**
* 
* @name			TFFN_FS_Open_With_Hint
*
* @description	This function opens a file in the parent directory
*
* @param		stDirHandler	pointer to the directory handler
*				szPath			pointer to the file path in the parent directory
*				stFileHandler	pointer to the file handler
*				
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		Ahn HyunJu (hyunju.ahn@samsung.com)
*
* @date			2005/10/14
*
* @revision history	
*				2005/10/14	[Ahn HyunJu]	first writing
*
**/
INT32
TFFN_FS_Open_Ex(t_file_handler *stDirHandler, UINT16 *szPath, t_file_handler *stFileHandler)
{
	INT32	nReturn;
	UINT16	*tempPC;
 
#if (MOREX_STRICT_CHECK == 1)
	if ((bInitialized == false) || (szPath == NULL) || (stFileHandler == NULL) || (stDirHandler == NULL))
	{
		return MOREX_ERROR;
	}
 
	tempPC = szPath;
	while (*tempPC != '\0')
	{
		if ((*tempPC < ' ') || (*tempPC > '~'))
		{
			return MOREX_ERROR;
		}
		tempPC++;		
	}
#endif
 
    if (stDirHandler->nIsDir == 0)
	{
		return MOREX_ERROR;
	}
    
	stFileHandler->nParentCluster = stDirHandler->nMyCluster;
	stFileHandler->nIsDir = 1;
 
	nReturn = _Path_Resolve(szPath, stFileHandler, MOREX_OPEN_MODE);
 
	if (nReturn < 0)
	{
		return MOREX_ERROR;
	}
 
	if (stFileHandler->nIsDir == 1)
	{		
		return MOREX_ERROR;
	}
	
	stFileHandler->nOffset = 0;	
	   
	return MOREX_OK;
}
 
/**
* 
* @name			_Clean_Cluster
*
* @description	This function write 0 in the cluster
*
* @param		nCurCluster		cluster
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		YoungWon Yun (youngwon.yun@samsung.com)
*
* @date			2005/10/18
*
* @revision history	
*				2005/10/18	[YoungWon Yun]	first writing
*
**/
INT32
_Clean_Cluster(UINT32 nCluster)
{
	INT32	nRe, i;
	UINT32	nStartSector;
	UINT8	*szBuff;
 
	MOREX_memset((UINT8 *)gaBuffer, 0, 512);
	szBuff = (UINT8 *)gaBuffer;
 
	MOREX_memset(szBuff, 0, MOREX_SECTOR_SIZE);
 
	nStartSector = _Sector_Num_of_Cluster(&stVolumeInfo, nCluster);
 
	for (i = 0; i < (INT32)stVolumeInfo.nSectorPerClt; i++)
	{
 
		nRe = TFFN_Write_Sectors(nStartSector + i, 1, szBuff);
 
		if (nRe == MOREX_EIO)
		{
			return MOREX_ERROR;
		}
	}
	
	return MOREX_OK;
}
 
/**
* 
* @name			_Update_FAT
*
* @description	This function updates FAT table
*
* @param			nCluster		cluster
*					nValue			FAT value to update
*									-1		: if FAT value is FAT End
*									other	: if FAT value is not FAT End
*					bWriteFlag		write flag
*									true	: write FAT table to the storage
*									false	: dont write FAT table to the storage
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		YoungWon Yun (youngwon.yun@samsung.com)
*
* @date			2005/10/18
*
* @revision history	
*				2005/10/18	[YoungWon Yun]	first writing
*
**/
INT32
_Update_FAT(UINT32 nCluster, INT32 nValue, BOOL32 nWriteFlag)
{
	INT32	nRe;
	UINT32	nIndex, nFATsector1, nFATsector2, nFATEntOffset;
	UINT8	*szFAT;
 
	szFAT = (UINT8 *)gaFatTable;
 
 
	if (nValue == -1)
	{
		if (stVolumeInfo.nFATType == MOREX_FAT16)
		{
			nValue = MOREX_FAT16_END;
		}
		else
		{
			nValue = MOREX_FAT32_END;
		}
	}
 
 
	nFATsector1 = stVolumeInfo.nFAT1Addr + (nCluster >> stVolumeInfo.nClusterPerSectorBits);
	nFATEntOffset = nCluster & (stVolumeInfo.nClusterPerSector - 1);
 
	if (stVolumeInfo.nFATType == MOREX_FAT16)
	{
		nIndex = nFATEntOffset << 1;
		MOREX_memset(szFAT + nIndex, nValue >> 0, 1);
		MOREX_memset(szFAT + nIndex + 1, nValue >> 8, 1);	
	}
	else
	{
		nIndex = nFATEntOffset << 2;
		MOREX_memset(szFAT + nIndex, nValue >> 0, 1);
		MOREX_memset(szFAT + nIndex + 1, nValue >> 8, 1);
		MOREX_memset(szFAT + nIndex + 2, nValue >> 16, 1);
		MOREX_memset(szFAT + nIndex + 3, nValue >> 24, 1);
	}
 
	if (nWriteFlag == true)
	{
		nRe = TFFN_Write_Sectors(nFATsector1, 1, szFAT);
		if (nRe == MOREX_EIO)
		{
			return MOREX_ERROR;
		}
	
		
		nFATsector2 = stVolumeInfo.nFAT2Addr + (nCluster >> stVolumeInfo.nClusterPerSectorBits);
 
		nRe = TFFN_Write_Sectors(nFATsector2, 1, szFAT);
		if (nRe == MOREX_EIO)
		{
			return MOREX_ERROR;
		}
	}
	
	return MOREX_OK;
}
 
 
 
 
/**
* 
* @name			TFFN_FS_GetHandleinfo
*
* @description	This function opens a file 
*
* @param		szPath			pointer to the file path
*				stFileHandler	pointer to the file handler
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		hyung tae Park (htae.park@samsung.com)
*
* @date			2008/07/11
*
* @revision history	
*				2008/07/11	[hyung tae Park]	first writing
*
**/
INT32
TFFN_FS_GetHandleinfo(UINT16 *szPath, t_file_handler *stDirHandler)
{
		INT32	nReturn;
		UINT16	*tempPC;
	 
#if (MOREX_STRICT_CHECK == 0)
		if ((bInitialized == false) || (szPath == NULL) || (stDirHandler == NULL))
		{
			return MOREX_ERROR;
		}
	 
		tempPC = szPath;
		while (*tempPC != '\0')
		{
			if ((*tempPC < ' ') || (*tempPC > '~'))
			{
				return MOREX_ERROR;
			}
			tempPC++;		
		}
#endif
	 
		 memset((void*)stDirHandler, 0x0, sizeof(t_file_handler));
		stDirHandler->nParentCluster = stVolumeInfo.nRootCluster;
		stDirHandler->nIsDir = 1;
		stDirHandler->nMyCluster=2;
		stDirHandler->nMyClusterCurrentStart =2;
		
		nReturn = _Path_Resolve(szPath, stDirHandler, MOREX_OPEN_MODE);
	
		if (nReturn < 0)
		{
			return MOREX_ERROR;
		}
	 
		if (stDirHandler->nIsDir == 1)
		{		
			stDirHandler->nOffset = 0;	
			return MOREX_OK;
		}
	   
		return MOREX_ERROR;

} 
 
 
 /**
* 
* @name			_getEntry
*
* @description	This function is to get entry information using file handle 
                and increse file handle by 1
* 
* @param		stFileHandler	file handle information indicating where the entry is
*				stDirEnt1	    entry which we are trying to get by file handle

*
* @return		
*
* @author		hyung tae Park (htae.park@samsung.com)
*
* @date			2008/07/11
*
* @revision history	
*				2008/07/11	[hyung tae Park]	first writing

*
**/
 
 INT32
 _getEntry( t_file_handler *stFileHandler, t_dir_entry *stDirEnt1)
{
	INT32 nRe;

	t_dir_entry *tmpEnt;
	INT8 nCurSctflag = 0;
	INT32 tempIndex = 0;

	nCurClt = stFileHandler->nMyClusterCurrentStart;	  //DIRInfo is located in parentcluter
	nCurSct = _Sector_Num_of_Cluster(&stVolumeInfo, nCurClt);		 //sector can be caluated by volumeInfo(fixed) and cluter
	nCurSct = nCurSct + ((stFileHandler->nMyClusterCurrentOffset)/16); //if offset is bigger than 15, sector ++
	nCurIndex = ((stFileHandler->nMyClusterCurrentOffset)&MOREX_DIR_ENTRY_PER_SECTOR_MASK);   //if offset is bigger than 16, cut off the value


	nRe = _Read_Dir_Entry(nCurClt, nCurSct);
	if (nRe == MOREX_EIO)
	{
		return MOREX_EIO;
	}

	tmpEnt= _Get_Dir_Entry1(nCurIndex);

	MOREX_memcpy(stDirEnt1, tmpEnt, sizeof(t_dir_entry));

	if (stDirEnt1->aDirName[0] == 0x00)
	{
		return MOREX_ENOTEXIST;
	}

	nRe = _Increase_Directory_Entry_Index(&nCurIndex, &nCurSct, &nCurClt, &stVolumeInfo, 1);
	if(nRe == MOREX_EOF)
	{
		if(stDirEnt1->aDirName[0] == 0xE5)
		{
			return MOREX_ENOTEXIST;
		}
		return MOREX_EOF;
	}
	stFileHandler->nMyClusterCurrentOffset += 1;
	stFileHandler->nMyClusterCurrentOffset &= ((MOREX_DIR_ENTRY_PER_SECTOR*stVolumeInfo.nSectorPerClt)-1);
	stFileHandler->nMyClusterCurrentStart = nCurClt;

	return MOREX_OK;

}
 
 

/**
* 
* @name			TFFN_FS_Getlist
*
* @description	This function get file info using file handle information 
*
* @param		stFileHandler	pointer to the file handler
*				filename        pointer to the file name to get
*				size            pointer to the file size to get
*				dirinfo         pointer to the file or directory  to get
*
*
* @return		MOREX_OK		succeed
*				MOREX_ERROR		failed
*
* @author		hyung tae Park (htae.park@samsung.com)
*
* @date			2008/07/11
*
* @revision history	
*				2008/07/11	[hyung tae Park]	first writing

*
**/


INT32
TFFN_FS_Getlist(t_file_handler *stFileHandler, UINT8 *filename, UINT32 *size, UINT32 *dirinfo)
{
	INT32	nReturn = -1;
	UINT16	*tempPC = NULL;
	t_dir_entry *stDirEnt1 = NULL;
	t_dir_entry_long *tmpEtry[63] = {0,};	
	UINT32 tmpDirinfo = NULL;
	INT32 i = 0;
	INT32 j = 0;

	INT8 tmpEntryNum =0;
	INT8 longentryflag = 0;
	INT8 curentry = 0;
	UINT8 tmpfilename[512] = {0,};
	UINT32 currentName = 0;
	UINT32 des = 0, cur=0;
	INT8 CheckE5=0;
	UINT8 TempFileName[11];
	INT8 EOFflag = 0;

	while(EOFflag != 1)  /* To get file info from nand*/
	{
		stDirEnt1 = (t_dir_entry*)malloc_io(sizeof(t_dir_entry));
		if (stDirEnt1 == NULL)
		{
			while(tmpEntryNum--)
			{
				free_io(tmpEtry[tmpEntryNum]);
			}
			return MOREX_ERROR;
		}
		memset(stDirEnt1, 0, sizeof(t_dir_entry));
		nReturn = _getEntry(stFileHandler, stDirEnt1);
		
		if (nReturn == MOREX_ENOTEXIST)
		{
			free_io(stDirEnt1);
			while(tmpEntryNum--)
			{		
				free_io(tmpEtry[tmpEntryNum]);
				return MOREX_ENOTEXIST;
			}
			return MOREX_ENOTEXIST;
		}
		if(nReturn == MOREX_EOF)
		{
			EOFflag = 1;

		}

		if((*((UINT8 *)stDirEnt1)==0xE5)||(*((UINT8 *)stDirEnt1)==0x00))
		{
			free_io(stDirEnt1);
			continue;
		}
		
FILEHANDLE:
		if((stDirEnt1->nDirAttr)== 0x0F)  /*long name buffer input*/ 
		{
			longentryflag= 1;
			tmpEtry[tmpEntryNum] = (t_dir_entry_long *)stDirEnt1;
			tmpEntryNum++;
			continue;
		}
	
		if(longentryflag == 0)     /*short name*/
		{
			MOREX_memcpy(filename,stDirEnt1, 11 );
			*size = stDirEnt1->nDirFileSize;
			tmpDirinfo = stDirEnt1->nDirAttr;
			_fileName2str(filename);
			if ((tmpDirinfo & ATT_DIRECTORY)!=0)
			{
				*dirinfo = ATT_DIR;
			}
			else 
			{	/*only for short file to make correct file name*/

				MOREX_memcpy(TempFileName, (filename+8), 3);
				memset((filename+8),0x0,3);

				if(TempFileName[0]!=NULL)
				{
					for(j =0; j<9; j++)
					{
						if(*(filename+j)==0x20)
						{
							*(filename+j) = '.';
							*(filename+j+1) = TempFileName[0];
							*(filename+j+2) = TempFileName[1];	
							*(filename+j+3) = TempFileName[2];	
							*(filename+j+4) = 0;	
							break;
						}
						else if(j == 8)
						{
							*(filename+j) = '.';
							*(filename+j+1) = TempFileName[0];
							*(filename+j+2) = TempFileName[1];	
							*(filename+j+3) = TempFileName[2];	
							*(filename+j+4) = 0;
							break;
						}
					}
				}
				*dirinfo = ATT_FILE;
			}
			free_io(stDirEnt1);
			while(tmpEntryNum--)
			{
				free_io(tmpEtry[tmpEntryNum]);
			}	

			if (nReturn == MOREX_EOF)
			{
				return MOREX_EOF;
			}
			return MOREX_OK;

		}
		else      /*long name*/                
		{
			
			for(i=tmpEntryNum;i>0;i--)   
			{
				MOREX_memcpy(tmpfilename+currentName,(UINT8*)&(tmpEtry[i-1]->szName1), 10);
				currentName += 10;
				MOREX_memcpy(tmpfilename+currentName,(UINT8*)&(tmpEtry[i-1]->szName2), 12);	
				currentName += 12;
				MOREX_memcpy(tmpfilename+currentName,(UINT8*)&(tmpEtry[i-1]->szName3), 4);
				currentName += 4;					

			}
			do{
					
				tmpfilename[des]=tmpfilename[cur];
				cur++;
				des++;
				if(tmpfilename[cur]==0x0)
				{
					cur++;
					if(tmpfilename[cur]==0x0)
					{
						memset((UINT8*)&tmpfilename[des], 0x0,512-des);
						break;
					}
				}
				else
				{
					tmpfilename[des]=tmpfilename[cur];

				}
				
			}while(1);   /*multi to single name*/
			
			strcpy((char*)filename,(char*) tmpfilename);
			*size = stDirEnt1->nDirFileSize;
			tmpDirinfo = stDirEnt1->nDirAttr;
			
			if ((tmpDirinfo & ATT_DIRECTORY)!=0)
			{
				*dirinfo = ATT_DIR;
			}
			else 
			{
				*dirinfo = ATT_FILE;
			}
			
			free_io(stDirEnt1);
			while(tmpEntryNum--)
			{
				free_io(tmpEtry[tmpEntryNum]);
			}						

			if (nReturn == MOREX_EOF)
			{
		
				return MOREX_EOF;
			}
			return MOREX_OK;


		}	
	}

	return MOREX_EOF;

} 
 
 
 
 void
 _fileName2str(UINT8  * filename)
 {
 
	 UINT8 i=10;
	 while(*(filename+i)==(UINT8)0X20)
	 {
		 *(filename+i)=NULL;
		 i--;
	 }
 
 
 }
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
