#include <system.h>
#include "gpio.h"
#include "smsc911x.h"
#include "smsc911x_macro.h"
#include "protocolstack.h"
#include "burstcopy.h"


/***********************************************************/
/*    System dependent codes                                                            */
/***********************************************************/

#if 1 // for FPGA_v210(FSMDK v210)
#define SMSC_SYSINT 0
#define SMSC_BASE 0xA0000000
unsigned int ioaddr = SMSC_BASE;

#define SROM_BASE 0xe8000000
#define GPIO_BASE 0xe0200000
#define GPH0CON 0xE0200C00
#define EXT_INT30_CON 0xE0200E00
#define EXT_INT30_PEND 0xE0200F40
#define EXT_INT30_FLTCON0 0xE0200E80
#define EXT_INT30_FLTCON1 0xE0200E84
#define EXT_INT30_MASK 0xE0200F00

__irq void SMSC_Interrupt_handler(void)
{	
	u32	status;

	INTC_Disable(SMSC_SYSINT);
	*((volatile u32 *)EXT_INT30_MASK) |= (1<<0); // Ext INT30_0 disable interrupt

	//Disp("Interrupt!!!!!\n");
	smc911x_interrupt();
	
	*((volatile u32 *)EXT_INT30_PEND) = (0xff); // Ext INT30_0 pending clear
	*((volatile u32 *)EXT_INT30_MASK) = ~(1<<0); // Ext INT30_0 enable interrupt

	INTC_ClearVectAddr();
	INTC_Enable(SMSC_SYSINT);	
}

static void SMSC_SetupGPIO()
{
	*((volatile u32 *)EXT_INT30_MASK) = (0xff); // Ext INT30_0 int mask
	*((volatile u32 *)EXT_INT30_PEND) = (0xff); // Ext INT30_0 pending clear
	*((volatile u32 *)GPH0CON) = (0xf<<0);
	*((volatile u32 *)EXT_INT30_CON) = (0x1<<0); //Ext INT30_0set to high level detecting
	*((volatile u32 *)EXT_INT30_PEND) = (0x1); // Ext INT30_0 pending clear
	*((volatile u32 *)EXT_INT30_FLTCON0) = ((0x3<<6) | (0x4<<0)); // Ext _INT30_0 filter enable set '4'
	*((volatile u32 *)EXT_INT30_MASK) = ~(1<<0); // Ext INT30_3 enable interrupt
}

static void SMSC_SetupSROM()
{
	volatile unsigned int *temp;

	// set SROM1 data width to 16bits from 8bits
	temp = (unsigned int *)(SROM_BASE);
	*temp &= ~(0xf<<16);
	*temp |= (0x9<<16);

	// set SROM1 clock counts to maximum delay
	temp = (unsigned int *)(SROM_BASE+0x14);
	//*temp = 0xfffffff0;
	*temp = 0x00000100;

	// set GPIO to nCS4 
	temp = (unsigned int *)(GPIO_BASE+0x2e0);
	*temp &= ~(0xf<<16);
	*temp |= (2<<16);

}
#else  // It's for SMDK6400
unsigned int ioaddr = SMSC_BASE;

static void SMSC_Interrupt_handler()
{	
	u32	status;
	INTC_Disable(INT_EINT1);
	//Disp("SMSC interrupt occured\n");
	smc911x_interrupt();

	// clear interrupt pending in s3c ext. interrupt controller.
	{
		u32 v;
		// ext 10 interrupt pending clear
		v = GPIO_GetValue(EINT0PEND,0);
		v |= (1<<10);
		GPIO_SetValue(EINT0PEND,0,v);
	}	

	// clear interrupt pending in s3c VIC.
	INTC_ClearPending(INT_EINT1);

	//status = SMC_GET_INT();
	//	Disp("status=0x%x\n",status);
	INTC_Enable(INT_EINT1);
	
}

static void SMSC_SetupGPIO()
{
	u32 v;

	// set GPN10 to ext 10 interrupt pin
	v = GPIO_GetValue(GPIO_GPN,GPIO_CONFIG);
	v &= ~(0x3 << 20);
	v |= (0x2 << 20);
	GPIO_SetValue(GPIO_GPN,GPIO_CONFIG,v);

	// enable GPN10's pull-up
	v = GPIO_GetValue(GPIO_GPN,GPIO_PULL);
	v &= ~(0x3 << 20);
	v |= (0x2 << 20);
	GPIO_SetValue(GPIO_GPN,GPIO_PULL,v);

	// set ext 10 interrupt pin to rising edge triggered
	v = GPIO_GetValue(EINT0CON0,0);
	v &= ~(0x7<<20);
	v |= (0x4<<20);
	GPIO_SetValue(EINT0CON0,0,v);

	// set ext 10 interrupt pin's filter
	v = GPIO_GetValue(EINT0FLTCON1,0);
	v &= ~(0x7<<8);
	v |= ((0x1<<14)|(0x1<<15)|(0x4<<8));
	GPIO_SetValue(EINT0FLTCON1,0,v);

	// set ext 10 interrupt mask bit to 0 (enabling interrupt)
	v = GPIO_GetValue(EINT0MASK,0);
	v &= ~(1<<10);
	GPIO_SetValue(EINT0MASK,0,v);

	// ext 10 interrupt pending clear
	v = GPIO_GetValue(EINT0PEND,0);
	v |= (1<<10);
	GPIO_SetValue(EINT0PEND,0,v);
	
}

static void SMSC_SetupSROM()
{
	volatile unsigned int *temp;

	// set SROM1 data width to 16bits from 8bits
	temp = (unsigned int *)(0x70000000);
	*temp |= 0x10;

	// set SROM1 clock counts to maximum delay
	temp = (unsigned int *)(0x70000008);
	*temp = 0xfffffff0;

}

#endif

/****************************************************************************/


/* store this information for the driver.. */
typedef struct {
	/* version/revision of the SMC911x chip */
	u16 version;
	u16 revision;

	/* FIFO sizes */
	int tx_fifo_kb;
	int tx_fifo_size;
	int rx_fifo_size;
	int afc_cfg;

	/* Contains the current active receive/phy mode */
	int ctl_rfduplx;
	int ctl_rspeed;

	u32 msg_enable;
	u32 phy_type;
	int work_pending;

	int tx_throttle;
}SMSC911X_INFO;

SMSC911X_INFO smsc911x_info;

unsigned int transmit_completed;
unsigned int transmit_failed;

static void (*ReceiveFrameHandler)(u8 *data,u16 length)=0;

u8 MACADDR[6];

/* this enables an interrupt in the interrupt mask register */
#define SMC_ENABLE_INT(x) do {				\
	unsigned int  __mask;				\
	unsigned long __flags;				\
	__mask = SMC_GET_INT_EN();			\
	__mask |= (x);					\
	SMC_SET_INT_EN(__mask);				\
} while (0)

/* this disables an interrupt from the interrupt mask register */
#define SMC_DISABLE_INT(x) do {				\
	unsigned int  __mask;				\
	unsigned long __flags;				\
	__mask = SMC_GET_INT_EN();			\
	__mask &= ~(x);					\
	SMC_SET_INT_EN(__mask);				\
} while (0)

static void udelay(unsigned int v)
{
	volatile int temp;
	int i;
	for(i=0;i<v*10;i++)temp++; //it's delay
}

/*
 * this does a soft reset on the device
 */
static void smc911x_reset()
{
	unsigned int reg, timeout=0, resets=1;
	unsigned long flags;

	/*	 Take out of PM setting first */
	if ((SMC_GET_PMT_CTRL() & PMT_CTRL_READY_) == 0) {
		/* Write to the bytetest will take out of powerdown */
		SMC_SET_BYTE_TEST(0);
		timeout=10;
		do {
			udelay(10);
			reg = SMC_GET_PMT_CTRL() & PMT_CTRL_READY_;
		} while ( timeout-- && !reg);
		if (timeout == 0) {
			Disp("fail to PM restore\n");
			return;
		}
	}

	/* Disable all interrupts */
	SMC_SET_INT_EN(0);

	while (resets--) {
		SMC_SET_HW_CFG(HW_CFG_SRST_);
		timeout=10;
		do {
			udelay(10);
			reg = SMC_GET_HW_CFG();
			/* If chip indicates reset timeout then try again */
			if (reg & HW_CFG_SRST_TO_) {
				Disp("fail to reset\n");
				resets++;
				break;
			}
		} while ( timeout-- && (reg & HW_CFG_SRST_));
	}
	if (timeout == 0) {
		Disp("smc911x_reset timeout waiting for reset\n");
		return;
	}

	/* make sure EEPROM has finished loading before setting GPIO_CFG */
	timeout=1000;
	while ( timeout-- && (SMC_GET_E2P_CMD() & E2P_CMD_EPC_BUSY_)) {
		udelay(10);
	}
	if (timeout == 0){
		Disp("smc911x_reset timeout waiting for EEPROM busy\n");
		return;
	}

	/* Initialize interrupts */
	SMC_SET_INT_EN(0);
	SMC_ACK_INT(-1);

	/* Reset the FIFO level and flow control settings */
	SMC_SET_HW_CFG((smsc911x_info.tx_fifo_kb & 0xF) << 16);
//TODO: Figure out what appropriate pause time is
	SMC_SET_FLOW(FLOW_FCPT_ | FLOW_FCEN_);
	SMC_SET_AFC_CFG(smsc911x_info.afc_cfg);

	/* Set to LED outputs */
	#if 1
	SMC_SET_GPIO_CFG(0x70070000);
	#else
	SMC_SET_GPIO_CFG(0x73070000);
	#endif

	/*
	 * Deassert IRQ for 1*10us for edge type interrupts
	 * and drive IRQ pin push-pull
	 */
	 #if 0
	SMC_SET_IRQ_CFG( (1 << 24) | INT_CFG_IRQ_EN_ | INT_CFG_IRQ_TYPE_ );
	#else
	// cglee sets interrupt pin to active high.
	SMC_SET_IRQ_CFG( (1 << 24) | INT_CFG_IRQ_EN_ | INT_CFG_IRQ_TYPE_ |INT_CFG_IRQ_POL_ );
	#endif
}

static void smc911x_drop_pkt()
{
	unsigned int fifo_count, timeout, reg;
	u32 temp;

	fifo_count = SMC_GET_RX_FIFO_INF() & 0xFFFF;
	if (fifo_count <= 4) {
		/* Manually dump the packet data */
		while (fifo_count--)temp=SMC_GET_RX_FIFO();
	} else{
		/* Fast forward through the bad packet */
		SMC_SET_RX_DP_CTRL(RX_DP_CTRL_FFWD_BUSY_);
		timeout=50;
		do {
			udelay(10);
			reg = SMC_GET_RX_DP_CTRL() & RX_DP_CTRL_FFWD_BUSY_;
		} while ( timeout-- && reg);
		if (timeout == 0) {
		}
	}
}

/*
 * This handles a TX status interrupt, which is only called when:
 * - a TX error occurred, or
 * - TX of a packet completed.
 */
static void smc911x_tx()
{
	unsigned int tx_status;

	/* Collect the TX status */
	while (((SMC_GET_TX_FIFO_INF() & TX_FIFO_INF_TSUSED_) >> 16) != 0) {
		tx_status = SMC_GET_TX_STS_FIFO();
		/* count Tx errors, but ignore lost carrier errors when in
		 * full-duplex mode */
		if ((tx_status & TX_STS_ES_) && !(smsc911x_info.ctl_rfduplx &&
		    !(tx_status & 0x00000306))) {
		}
		if (tx_status & TX_STS_MANY_COLL_) {
		} else {
		}
		/* carrier error only has meaning for half-duplex communication */
		if ((tx_status & (TX_STS_LOC_ | TX_STS_NO_CARR_)) &&  !smsc911x_info.ctl_rfduplx) {
		}
		if (tx_status & TX_STS_LATE_COLL_) {
		}
	}
}

static void PrintPacket(u8 *buf, int size)
{
	int i;
	for(i=0;i<size;i++)
	{
		if(i%16==0)Disp("\n");
		Disp("%X ",buf[i]);
	}
	Disp("\n");
}

void SMSC_pull_data(u32 *buf,int len)
{
	volatile u32 *temp;
	volatile u16 *temp16;
	volatile u8 *temp8;
	u16 *buf16;
	u8 *buf8;
	int i;
	u32 loopCount;
	
	temp = (u32 *)(ioaddr);
	temp16 = (u16 *)(ioaddr);
	temp8 = (u8 *)(ioaddr);

	buf16 = (u16 *)buf;
	buf8 = (u8 *)buf;

	#if 1
	loopCount = len/32;
	
	burst8_memcpy_static_src_inc_dst((u32)buf, (u32)ioaddr, loopCount);

	for(i=0;i<(len%32)/4;i++)
	{
		*(buf+i+loopCount*8) = *temp;
	}
	if((len%32)%4)
	{
		*(buf+i+loopCount*8) = *temp;
	}
	
	#else
	for(i=0;i<len/4;i++)
	{
		*(buf+i) = *temp;
	}
	if(len%4)
	{
		*(buf+i) = *temp;
	}
	#endif


	//Disp("----Receving Packet----size:%d",len);
	//PrintPacket((u8 *)buf,len);	
}

static void SMSC_push_data(u32 *buf,int len)
{
	volatile u32 *temp;
	volatile u16 *temp16;
	volatile u8 *temp8;
	u16 *buf16;
	u8 *buf8;
	int i;
	u32 loopCount;
	temp = (u32 *)(ioaddr+0x20);
	temp16 = (u16 *)(ioaddr+0x20);
	temp8 = (u8 *)(ioaddr+0x20);
	buf16 = (u16 *)buf;
	buf8 = (u8 *)buf;

	//Disp("push buf addr=0x%x\n",buf);

	#if 0
	loopCount = len/32;
	burst8_memcpy_inc_src_static_dst((u32)(ioaddr+0x20),(u32)buf,loopCount);
	
	for(i=0;i<(len%32)/4;i++)
	{
		*temp = buf[i+loopCount*8];
	}
	if((len%32)%4){*temp=buf[i+loopCount*8];};
	#else
	for(i=0;i<len/4;i++)
	{
		*temp16 = buf16[i*2];
		*(temp16+1) = buf16[i*2+1];
	}

	if(len%4)
	{
		*temp16 = buf16[i*2];
		*(temp16+1) = buf16[i*2+1];
	}
	#endif
}

static void smc911x_rcv()
{
	unsigned int pkt_len, status;
	unsigned int data[4096];

	status = SMC_GET_RX_STS_FIFO();
	pkt_len = (status & RX_STS_PKT_LEN_) >> 16;
	if (status & RX_STS_ES_) {
		/* Deal with a bad packet */
		if (status & RX_STS_CRC_ERR_){}
		else {
			if (status & RX_STS_LEN_ERR_){}
			if (status & RX_STS_MCAST_){}
		}
		/* Remove the bad packet data from the RX FIFO */
		smc911x_drop_pkt();
	} else {
		/* Receive a valid packet */

		/* Align IP header to 32 bits */

		SMC_SET_RX_CFG(RX_CFG_RX_END_ALGN4_ | (0<<8) & ( RX_CFG_RXDOFF_));
		//SMC_PULL_DATA(data, pkt_len+2+3);
		SMSC_pull_data(data,pkt_len+0+0);
		//Disp("receive packet size=%d\n",pkt_len);
		ReceiveFrameHandler((u8 *)data,pkt_len);
	}
}

static void smc911x_phy_interrupt()
{
}

static void smc911x_hardware_send_pkt(u32 *buf,int size)
{
	unsigned int cmdA, cmdB, len;
	unsigned long flags;
	int i;
	unsigned char *temp;

	/* cmdA {25:24] data alignment [20:16] start offset [10:0] buffer length */
	/* cmdB {31:16] pkt tag [10:0] length */
	len =  size;
	cmdA = ((0 << 16) |
			TX_CMD_A_INT_FIRST_SEG_ | TX_CMD_A_INT_LAST_SEG_ | TX_CMD_A_INT_ON_COMP_ |
			len);

	/* tag is packet length so we can use this in stats update later */
	cmdB = (len  << 16) | (len & 0x7FF);

	SMC_SET_TX_FIFO(cmdA);
	SMC_SET_TX_FIFO(cmdB);

	/* Send pkt via PIO or DMA */
	SMSC_push_data(buf, len);

	#if 0
	temp = (u8 *)buf;
	Disp("Send packet(%d):",len);
	for(i=0;i<len;i++)
	{
		if(i%16==0)Disp("\n");
		Disp("%X ",temp[i]);
	}
	Disp("\n");
	#endif
}

/*
 * Since I am not sure if I will have enough room in the chip's ram
 * to store the packet, I call this routine which either sends it
 * now, or set the card to generates an interrupt when ready
 * for the packet.
 */
static int smc911x_hard_start_xmit(u32 *buf,int size)
{
	unsigned int free;
	unsigned long flags;

	do{
		free = SMC_GET_TX_FIFO_INF() & TX_FIFO_INF_TDFREE_;
	}while(free<size+4);

	smc911x_hardware_send_pkt(buf,size);

	return 0;
}

static int interrupt_test;
void smc911x_interrupt()
{
	struct SMSC911X_INFO *lp = (struct SMSC911X_INFO *)&smsc911x_info;
	unsigned int status, mask, timeout;
	unsigned int rx_overrun=0, cr, pkts;
	unsigned long flags;


	/* Spurious interrupt check */
	if ((SMC_GET_IRQ_CFG() & (INT_CFG_IRQ_INT_ | INT_CFG_IRQ_EN_)) !=
		(INT_CFG_IRQ_INT_ | INT_CFG_IRQ_EN_)) {
		return;
	}

	mask = SMC_GET_INT_EN();
	SMC_SET_INT_EN(0);

	timeout = 8;

	do {
		status = SMC_GET_INT();

		//Disp("status=0x%x mask=0x%x\n",status,mask);

		status &= mask;
		//Disp("masked status=0x%x\n",status);
		if (!status)
			break;

		/* Handle SW interrupt condition */
		if (status & INT_STS_SW_INT_) {
			SMC_ACK_INT(INT_STS_SW_INT_);
			mask &= ~INT_EN_SW_INT_EN_;
			interrupt_test = 1;
		}
		/* Handle various error conditions */
		if (status & INT_STS_RXE_) {
			SMC_ACK_INT(INT_STS_RXE_);
		}
		if (status & INT_STS_RXDFH_INT_) {
			SMC_ACK_INT(INT_STS_RXDFH_INT_);
		 }
		/* Undocumented interrupt-what is the right thing to do here? */
		if (status & INT_STS_RXDF_INT_) {
			SMC_ACK_INT(INT_STS_RXDF_INT_);
		}

		/* Rx Data FIFO exceeds set level */
		if (status & INT_STS_RDFL_) {
			if (IS_REV_A(smsc911x_info.revision)) {
				rx_overrun=1;
				SMC_GET_MAC_CR(cr);
				cr &= ~MAC_CR_RXEN_;
				SMC_SET_MAC_CR(cr);
			}
			SMC_ACK_INT(INT_STS_RDFL_);
		}
		if (status & INT_STS_RDFO_) {
			if (!IS_REV_A(smsc911x_info.revision)) {
				SMC_GET_MAC_CR(cr);
				cr &= ~MAC_CR_RXEN_;
				SMC_SET_MAC_CR(cr);
				rx_overrun=1;
			}
			SMC_ACK_INT(INT_STS_RDFO_);
		}
		/* Handle receive condition */
		if ((status & INT_STS_RSFL_) || rx_overrun) {
			unsigned int fifo;
			fifo = SMC_GET_RX_FIFO_INF();
			pkts = (fifo & RX_FIFO_INF_RXSUSED_) >> 16;
			//Disp("Rx FIFO pkts %d, bytes %d\n",pkts, fifo & 0xFFFF );
			if (pkts != 0) {
				smc911x_rcv();
			}
			SMC_ACK_INT(INT_STS_RSFL_);
		}
		/* Handle transmit FIFO available */
		if (status & INT_STS_TDFA_) {
			SMC_SET_FIFO_TDA(0xFF);
			smsc911x_info.tx_throttle = 0;
			SMC_ACK_INT(INT_STS_TDFA_);
		}
		/* Handle transmit done condition */
		if (status & (INT_STS_TSFL_ | INT_STS_GPT_INT_)) {
			smc911x_tx();
			SMC_SET_GPT_CFG(GPT_CFG_TIMER_EN_ | 10000);
			SMC_ACK_INT(INT_STS_TSFL_);
			SMC_ACK_INT(INT_STS_TSFL_ | INT_STS_GPT_INT_);
		}

		if(status & (INT_STS_TX_IOC_))
		{
			SMC_ACK_INT(INT_STS_TX_IOC_);
			transmit_completed=1;
		}

		/* Handle PHY interrupt condition */
		if (status & INT_STS_PHY_INT_) {
			smc911x_phy_interrupt();
			SMC_ACK_INT(INT_STS_PHY_INT_);
		}
	} while (--timeout);


	/* restore mask state */
	SMC_SET_INT_EN(mask);

	return;
}


void SMSC_Generate_SW_Interrupt()
{
	interrupt_test = 0;
	
	SMC_SET_INT_EN(INT_EN_SW_INT_EN_);

	while(!interrupt_test){}
}


static void smc911x_enable()
{
	unsigned mask, cfg, cr;
	unsigned long flags;

	SMC_SET_MAC_ADDR(MACADDR);

	/* Enable TX */
	cfg = SMC_GET_HW_CFG();
	cfg &= HW_CFG_TX_FIF_SZ_ | 0xFFF;
	cfg |= HW_CFG_SF_;
	SMC_SET_HW_CFG(cfg);
	SMC_SET_FIFO_TDA(0xFF);
	/* Update TX stats on every 64 packets received or every 1 sec */
	SMC_SET_FIFO_TSL(64);
	SMC_SET_GPT_CFG(GPT_CFG_TIMER_EN_ | 10000);

	SMC_GET_MAC_CR(cr);
	cr |= MAC_CR_TXEN_ | MAC_CR_HBDIS_;
	SMC_SET_MAC_CR(cr);
	SMC_SET_TX_CFG(TX_CFG_TX_ON_);

	SMC_SET_RX_CFG((0<<8) & RX_CFG_RXDOFF_);

	/* Turn on receiver and enable RX */
	if (cr & MAC_CR_RXEN_){}

	SMC_SET_MAC_CR( cr | MAC_CR_RXEN_ );

	/* Interrupt on every received packet */
	SMC_SET_FIFO_RSA(0x01);
	SMC_SET_FIFO_RSL(0x00);

	/* now, enable interrupts */
	#if 0
	mask = INT_EN_TDFA_EN_ | INT_EN_TSFL_EN_ | INT_EN_RSFL_EN_ |
		INT_EN_GPT_INT_EN_ | INT_EN_RXDFH_INT_EN_ | INT_EN_RXE_EN_ |
		INT_EN_PHY_INT_EN_;
	#else
	mask = INT_EN_TDFA_EN_ | INT_EN_TSFL_EN_ | INT_EN_RSFL_EN_ |
		 INT_EN_RXDFH_INT_EN_ | INT_EN_RXE_EN_ |
		INT_EN_PHY_INT_EN_ | INT_EN_TIOC_INT_EN_;
	#endif
	if (IS_REV_A(smsc911x_info.revision))
		mask|=INT_EN_RDFL_EN_;
	else {
		mask|=INT_EN_RDFO_EN_;
	}
	//SMC_ACK_INT(0x7fffffff); // gpio interrupt pin should always be cleared.

	//Disp("interrupt enable mask=0x%x\n",mask);
	SMC_ENABLE_INT(mask);
}

int SMSC_Init(u8 *mac_addr)
{
	unsigned int val, chip_id, revision;
	const char *version_string;
	int i;
	u8 stored_mac_addr[6];

	MACADDR[0] = mac_addr[0];
	MACADDR[1] = mac_addr[1];
	MACADDR[2] = mac_addr[2];
	MACADDR[3] = mac_addr[3];
	MACADDR[4] = mac_addr[4];
	MACADDR[5] = mac_addr[5];
	
	SMSC_SetupSROM();
	
	val = SMC_GET_BYTE_TEST();
	Disp("val = 0x%x\n",val);
	if (val != 0x87654321) {
		Disp("It's not 0x87654321. Fail to init smsc\n");
		return 1;
	}

	chip_id = SMC_GET_PN();
	for(i=0;chip_ids[i].id != 0; i++) {
		if (chip_ids[i].id == chip_id) break;
	}
	if (!chip_ids[i].id) {
		Disp("Unknown chip id\n");
		return 1;
	}
	version_string = chip_ids[i].name;
	revision = SMC_GET_REV();
	Disp("SMSC:%s, rev=0x%04x\n",version_string,revision);


	/* fill in some of the fields */
	smsc911x_info.version = chip_ids[i].id;
	smsc911x_info.revision = revision;
	smsc911x_info.tx_fifo_kb =8;
	/* Reverse calculate the RX FIFO size from the TX */
	smsc911x_info.tx_fifo_size=(smsc911x_info.tx_fifo_kb<<10) - 512;
	smsc911x_info.rx_fifo_size= ((0x4000 - 512 - smsc911x_info.tx_fifo_size) / 16) * 15;

	/* Set the automatic flow control values */
	switch(smsc911x_info.tx_fifo_kb) {
		/*
		 *	 AFC_HI is about ((Rx Data Fifo Size)*2/3)/64
		 *	 AFC_LO is AFC_HI/2
		 *	 BACK_DUR is about 5uS*(AFC_LO) rounded down
		 */
		case 2:/* 13440 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x008C46AF;break;
		case 3:/* 12480 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x0082419F;break;
		case 4:/* 11520 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x00783C9F;break;
		case 5:/* 10560 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x006E374F;break;
		case 6:/* 9600 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x0064328F;break;
		case 7:/* 8640 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x005A2D7F;break;
		case 8:/* 7680 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x0050287F;break;
		case 9:/* 6720 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x0046236F;break;
		case 10:/* 5760 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x003C1E6F;break;
		case 11:/* 4800 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x0032195F;break;
		/*
		 *	 AFC_HI is ~1520 bytes less than RX Data Fifo Size
		 *	 AFC_LO is AFC_HI/2
		 *	 BACK_DUR is about 5uS*(AFC_LO) rounded down
		 */
		case 12:/* 3840 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x0024124F;break;
		case 13:/* 2880 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x0015073F;break;
		case 14:/* 1920 Rx Data Fifo Size */
			smsc911x_info.afc_cfg=0x0006032F;break;
		 default:
			smsc911x_info.afc_cfg=0x0050287F;break; //case 8 is default.
			 break;
	}

	SMC_SET_MAC_ADDR(MACADDR);
	
	SMC_GET_MAC_ADDR(stored_mac_addr);
	Disp("MAC:%x-%x-%x-%x-%x-%x\n",stored_mac_addr[0],
		stored_mac_addr[1],stored_mac_addr[2],stored_mac_addr[3],stored_mac_addr[4],stored_mac_addr[5]);
	
	smc911x_reset();
	
	SMSC_SetupGPIO();
	INTC_Init();
	INTC_SetVectAddr(SMSC_SYSINT, SMSC_Interrupt_handler);
	INTC_Enable(SMSC_SYSINT);
	
	smc911x_enable();
	//*((u32 *)EXT_INT30_PEND) = (0x1); // Ext INT30_0 pending clear
	
	return 0;
}

void SMSC_Sender(u32 *buf,u16 len)
{	
	smc911x_hard_start_xmit((u32 *)buf,(int)len);
}

void SMSC_RegisterReceiveHandler(void (*handler)(u8 *data, u16 length))
{
	ReceiveFrameHandler=handler;
}




