/*-------------------------------------------------------------
 * Filename: protocolstackudp.c
 *
 * Contents: Implemention of UDP of TCP/IP
 *
 * Abbreviations:
 *
 * Person Involved: chun gil lee
 *
 * Notes: 
 *
 * History
 *  - First created by chun gil lee
 *
 * Copyright (c) 2008 SAMSUNG Electronics.
 --------------------------------------------------------------*/
 #include "system.h"
#include "ProtocolStack.h"
#include "ProtocolStackUDP.h"

#define DATAGRAM_LENGTH 1400


typedef __packed struct
{
	u16 sourcePort;
	u16 destinationPort;
	u16 length;
	u16 checksum;
}UDP_HEADER;

typedef enum
{
	UDP_DATA=1,DATA_ACK
}UDP_DATAGRAM_TYPE;

typedef __packed struct
{
	u8 datagramType;
	u32 sequenceNum;
	u16 dataSize;
}UDP_DATAGRAM_HEADER;

static u8 MACAddress[6];
static u8 IPAddress[4];
static u16 Port;
static u8 DestIPAddress[4];
static u16 DestPort;
static u32 SequenceNum;
static u32 destSequnceNum;
static u32 ackSequnceNum;
static u32 lastHandledSeqNum=0;
static void (*packet_receiver)(u8 *data,u16 length) = 0;
static void (*packet_receiver_raw)(u8 *data,u16 length,UDP_PACKET_INFO info) = 0;
static u32 datagramBuffer[((DATAGRAM_LENGTH)+sizeof(UDP_HEADER)+sizeof(UDP_DATAGRAM_HEADER))/4+1];

#define SWAP16(x) (u16)((u16)(x<<8)|(u16)(x>>8))

static void dump(u8 *buf,u32 size)
{
	u32 i;
	for(i=0;i<size;i++)
	{
		if(i!=0 && i%16==0)Disp("\n");
		if(buf[i]<0x10)Disp("0x0%x ",buf[i]);
		else Disp("0x%x ",buf[i]);
	}
	Disp("\n");
}

static void UDP_ack(u32 sequnceNum)
{
	u16 chunkLength = sizeof(UDP_HEADER)+sizeof(UDP_DATAGRAM_HEADER);
	u8 *dataBuffer = (u8*)datagramBuffer;
	UDP_HEADER *udpHeader = (UDP_HEADER *)datagramBuffer;
	UDP_DATAGRAM_HEADER *datagramHeader = (UDP_DATAGRAM_HEADER *)(dataBuffer + sizeof(UDP_HEADER));	

	udpHeader->checksum=0;
	udpHeader->destinationPort= SWAP16(DestPort);
	udpHeader->length = SWAP16(chunkLength);
	udpHeader->sourcePort = SWAP16(Port);

	datagramHeader->datagramType = DATA_ACK;
	datagramHeader->sequenceNum = sequnceNum;
	datagramHeader->dataSize = 0;

	PROTOCOL_IPSender(dataBuffer, chunkLength,PROTOCOL_IP_UDP, DestIPAddress, 0);
}

void UDP_ReceiveHandlerRaw(u8 *data,u16 length)
{
	UDP_PACKET_INFO info;
	UDP_DATAGRAM_HEADER *datagramHeader;
	UDP_HEADER *udpHeader = (UDP_HEADER *)data;
	u8 *realData = data+sizeof(UDP_DATAGRAM_HEADER)+sizeof(UDP_HEADER);
	info.receivedSourcePort = SWAP16(udpHeader->sourcePort);
	info.receviedDestinationPort = SWAP16(udpHeader->destinationPort);
	info.receviedLength = SWAP16(udpHeader->length);
	info.receviedChecksum = SWAP16(udpHeader->checksum);	

	datagramHeader = (UDP_DATAGRAM_HEADER *)(data + sizeof(UDP_HEADER));

	if(packet_receiver_raw)packet_receiver_raw(realData,datagramHeader->dataSize,info);
}

void UDP_ReceiveHandler(u8 *data,u16 length)
{
	u16 receivedSourcePort;
	u16 receviedDestinationPort;
	u16 receviedLength;
	u16 receviedChecksum;
	UDP_DATAGRAM_HEADER *datagramHeader;
	UDP_HEADER *udpHeader = (UDP_HEADER *)data;
	u8 *realData = data+sizeof(UDP_DATAGRAM_HEADER)+sizeof(UDP_HEADER);
	receivedSourcePort = SWAP16(udpHeader->sourcePort);
	receviedDestinationPort = SWAP16(udpHeader->destinationPort);
	receviedLength = SWAP16(udpHeader->length);
	receviedChecksum = SWAP16(udpHeader->checksum);	
	
	if(receivedSourcePort==DestPort && receviedDestinationPort==Port)
	{
		datagramHeader = (UDP_DATAGRAM_HEADER *)(data + sizeof(UDP_HEADER));
		if(datagramHeader->datagramType == DATA_ACK)
		{
			//Disp("ACK packet !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n");
			ackSequnceNum = datagramHeader->sequenceNum;
		}
		else if(datagramHeader->datagramType == UDP_DATA)
		{

		//	Disp("Received [%x] [%x] [%x] [%x] [%x] [%x] [%x] [%x] [%x] [%x]\n",
		//		realData[0],realData[1],realData[2],realData[3],realData[4],realData[5],realData[6],realData[7],realData[8],realData[9]);
			
			if(lastHandledSeqNum==datagramHeader->sequenceNum)
			{

			}
			else
			{
				lastHandledSeqNum = datagramHeader->sequenceNum;

				if(packet_receiver)packet_receiver(realData,datagramHeader->dataSize);
				//.. do copy at here ../
			}

			UDP_ack(datagramHeader->sequenceNum);
		}
	}
}

int UDP_InitRaw(u8 *macAddress,u8 *ipAddress,u16 port, u8 *destIPAddress,u16 destPort,void (*receiver)(u8 *data,u16 length,UDP_PACKET_INFO info))
{
	int i;
	
	SequenceNum = 1;
	destSequnceNum = 1;

	packet_receiver_raw = receiver;
	
	for(i=0;i<6;i++)MACAddress[i] = macAddress[i];
	for(i=0;i<4;i++)IPAddress[i] = ipAddress[i];
	for(i=0;i<4;i++)DestIPAddress[i]= destIPAddress[i];
	Port = port;
	DestPort = destPort;

	return true;
}
 
int UDP_Init(u8 *macAddress,u8 *ipAddress,u16 port, u8 *destIPAddress,u16 destPort,void (*receiver)(u8 *data,u16 length))
{
	int i;
	
	SequenceNum = 1;
	destSequnceNum = 1;

	packet_receiver = receiver;
	
	for(i=0;i<6;i++)MACAddress[i] = macAddress[i];
	for(i=0;i<4;i++)IPAddress[i] = ipAddress[i];
	for(i=0;i<4;i++)DestIPAddress[i]= destIPAddress[i];
	Port = port;
	DestPort = destPort;

	return true;
}

u8 *UDP_GetWriteBuffer()
{
	u8 *dataBuffer = (u8*)PROTOCOL_GetIPBuffer();
	return dataBuffer+sizeof(UDP_HEADER)+sizeof(UDP_DATAGRAM_HEADER);
}

void UDP_SendDatagramFast(u16 length)
{
	int i,j;
	u8 *dataBuffer = (u8*)PROTOCOL_GetIPBuffer();
	UDP_HEADER *udpHeader = (UDP_HEADER *)dataBuffer;
	UDP_DATAGRAM_HEADER *datagramHeader = (UDP_DATAGRAM_HEADER *)(dataBuffer + sizeof(UDP_HEADER));
	
	volatile u32 tmp;
	int sendOK=0;
	
	u16 chunkLength;

	datagramHeader->dataSize = length;
	chunkLength = datagramHeader->dataSize+sizeof(UDP_HEADER)+sizeof(UDP_DATAGRAM_HEADER);
		
	udpHeader->checksum=0;
	udpHeader->destinationPort= SWAP16(DestPort);
	udpHeader->length = SWAP16(chunkLength);
	udpHeader->sourcePort = SWAP16(Port);

	datagramHeader->datagramType = UDP_DATA;
	datagramHeader->sequenceNum = SequenceNum;

	//Disp("SendSeq:%d\n",SequenceNum);

	while(1)
	{
		PROTOCOL_IPSenderFast(chunkLength,PROTOCOL_IP_UDP, DestIPAddress, 0);
		sendOK=0;
		for(tmp=0;tmp<0xfffff;tmp++)
		{
			if(ackSequnceNum==SequenceNum)
			{
				sendOK=1;
				break;
			}
		}
		if(sendOK)break;
	}

	SequenceNum++;
	if(SequenceNum>=1000000)SequenceNum=0;

}

void UDP_SendDatagramRaw(u8 *data,u16 length)
{
	int i,j;
	u8 *dataBuffer = (u8*)datagramBuffer;
	UDP_HEADER *udpHeader = (UDP_HEADER *)datagramBuffer;
	UDP_DATAGRAM_HEADER *datagramHeader = (UDP_DATAGRAM_HEADER *)(dataBuffer + sizeof(UDP_HEADER));
	
	volatile u32 tmp;
	int sendOK=0;
	
	u16 chunkLength;
	for(i=0;i<length;i=i+DATAGRAM_LENGTH)
	{

		for(j=0;j<DATAGRAM_LENGTH && j+i<length;j++)
			dataBuffer[sizeof(UDP_HEADER)+sizeof(UDP_DATAGRAM_HEADER)+j] = data[i+j];

		datagramHeader->dataSize = j;
		chunkLength = datagramHeader->dataSize+sizeof(UDP_HEADER)+sizeof(UDP_DATAGRAM_HEADER);
		
		udpHeader->checksum=0;
		udpHeader->destinationPort= SWAP16(DestPort);
		udpHeader->length = SWAP16(chunkLength);
		udpHeader->sourcePort = SWAP16(Port);

		datagramHeader->datagramType = UDP_DATA;
		datagramHeader->sequenceNum = SequenceNum;
		PROTOCOL_IPSender(dataBuffer, chunkLength,PROTOCOL_IP_UDP, DestIPAddress, 0);
	}
}

void UDP_SendDatagram(u8 *data,u16 length)
{
	int i,j;
	u8 *dataBuffer = (u8*)datagramBuffer;
	UDP_HEADER *udpHeader = (UDP_HEADER *)datagramBuffer;
	UDP_DATAGRAM_HEADER *datagramHeader = (UDP_DATAGRAM_HEADER *)(dataBuffer + sizeof(UDP_HEADER));
	
	volatile u32 tmp;
	int sendOK=0;
	
	u16 chunkLength;
	for(i=0;i<length;i=i+DATAGRAM_LENGTH)
	{

		for(j=0;j<DATAGRAM_LENGTH && j+i<length;j++)
			dataBuffer[sizeof(UDP_HEADER)+sizeof(UDP_DATAGRAM_HEADER)+j] = data[i+j];

		datagramHeader->dataSize = j;
		chunkLength = datagramHeader->dataSize+sizeof(UDP_HEADER)+sizeof(UDP_DATAGRAM_HEADER);
		
		udpHeader->checksum=0;
		udpHeader->destinationPort= SWAP16(DestPort);
		udpHeader->length = SWAP16(chunkLength);
		udpHeader->sourcePort = SWAP16(Port);

		datagramHeader->datagramType = UDP_DATA;
		datagramHeader->sequenceNum = SequenceNum;

		while(1)
		{
			//Disp("SendIPSender:chunkLength:%d\n",chunkLength);
			Disp("chunkLength:%d\n",chunkLength);
			PROTOCOL_IPSender(dataBuffer, chunkLength,PROTOCOL_IP_UDP, DestIPAddress, 0);
			sendOK=0;
			for(tmp=0;tmp<0xfffff;tmp++)
			{
				if(ackSequnceNum==SequenceNum)
				{
					sendOK=1;
					break;
				}
			}
			if(sendOK)break;
		}

		SequenceNum++;
		if(SequenceNum>=1000000)SequenceNum=0;
	}
	
}


