/*-------------------------------------------------------------
 * Filename: protocolstack.c
 *
 * Contents: Implemention of Ethernet Frame, ICMP ping of TCP/IP and ARP
 *
 * Abbreviations:
 *
 * Person Involved: chun gil lee
 *
 * Notes: 
 *
 * History
 *  - First created by chun gil lee
 *
 * Copyright (c) 2008 SAMSUNG Electronics.
 --------------------------------------------------------------*/
#include "system.h"
#include "ProtocolStack.h"

static u8 SrcEthAddr[6];
static u8 SrcIPAddr[4];
static void (*Sender)(u8 *data, u16 length);
static void (*UDPHandler)(u8 *data,u16 length) = 0;

static PROTOCOL_ARPTABLE ARPTable[PROTOCOL_MAX_ARP_TABLE];
static u32 ARPTableSeq;

static u16 ICMPSeq;
static u16 IPSeq;
static u32 WriteBuffer[2048/4]; // first 2 bytes are not used to support 4bytes-alinged buffer for upper layer.

static bool ARPGetMAC(u8 *IP,u8 *MAC)
{
	int i;
	for(i=0;i<PROTOCOL_MAX_ARP_TABLE;i++)
	{
		if(ARPTable[i].IsUsed)
		{
		//	ARPDEBUG(("Index[%d]=>%d.%d.%d.%d -- %X-%X-%X-%X-%X-%X\n",i,
		//		ARPTable[i].IP[0],ARPTable[i].IP[1],ARPTable[i].IP[2],ARPTable[i].IP[3],
		//		ARPTable[i].MAC[0],ARPTable[i].MAC[1],ARPTable[i].MAC[2],ARPTable[i].MAC[3],ARPTable[i].MAC[4],ARPTable[i].MAC[5]));
			if(ARPTable[i].IP[0] == IP[0] &&
				ARPTable[i].IP[1] == IP[1] &&
				ARPTable[i].IP[2] == IP[2] &&
				ARPTable[i].IP[3] == IP[3])
			{
			//	ARPDEBUG(("MAC mem addr=0x%x\n",(u32)(ARPTable[i].MAC)));
				MAC[0] = ARPTable[i].MAC[0];
				MAC[1] = ARPTable[i].MAC[1];
				MAC[2] = ARPTable[i].MAC[2];
				MAC[3] = ARPTable[i].MAC[3];
				MAC[4] = ARPTable[i].MAC[4];
				MAC[5] = ARPTable[i].MAC[5];
				return true;
			}
		}
	}
	return false;
}

static void ARPTableUpdate(u8 *IP, u8 *MAC)
{
	bool isFound;
	int i;
	u32 oldest;
	u32 tmp;

	ARPDEBUG(("[ARP]Register IP:%d.%d.%d.%d<=>MAC:%X-%X-%X-%X-%X-%X\n",
	IP[0],IP[1],IP[2],IP[3],MAC[0],MAC[1],MAC[2],MAC[3],MAC[4],MAC[5]));		

	if(IP[0]==0 && IP[1]==0 && IP[2]==0 && IP[3]==0)return;

	for(i=0;i<PROTOCOL_MAX_ARP_TABLE;i++)
	{
		if(ARPTable[i].IsUsed)
		{
			if(ARPTable[i].MAC[0] == MAC[0] &&
				ARPTable[i].MAC[1] == MAC[1] &&
				ARPTable[i].MAC[2] == MAC[2] &&
				ARPTable[i].MAC[3] == MAC[3] &&
				ARPTable[i].MAC[4] == MAC[4] &&
				ARPTable[i].MAC[5] == MAC[5])
			{
				ARPDEBUG(("[ARP]Update ARP Table In [%d]\n",i));
				ARPTable[i].IP[0] = IP[0];
				ARPTable[i].IP[1] = IP[1];
				ARPTable[i].IP[2] = IP[2];
				ARPTable[i].IP[3] = IP[3];
				ARPTable[i].Seq = ARPTableSeq;
				ARPTableSeq++;
				return;				
			}
		}
	}
	
	isFound = false;
	for(i=0;i<PROTOCOL_MAX_ARP_TABLE;i++)
	{
		if(ARPTable[i].IsUsed==false)
		{
			ARPDEBUG(("[ARP]Update ARP Table In [%d]\n",i));
			ARPTable[i].MAC[0] = MAC[0];
			ARPTable[i].MAC[1] = MAC[1];
			ARPTable[i].MAC[2] = MAC[2];
			ARPTable[i].MAC[3] = MAC[3];
			ARPTable[i].MAC[4] = MAC[4];
			ARPTable[i].MAC[5] = MAC[5];

			ARPTable[i].IP[0] = IP[0];
			ARPTable[i].IP[1] = IP[1];
			ARPTable[i].IP[2] = IP[2];
			ARPTable[i].IP[3] = IP[3];

			ARPTable[i].Seq = ARPTableSeq;
			ARPTableSeq++;
			ARPTable[i].IsUsed = true;
			isFound = true;
			break;
		}	
	}
	if(isFound==false) // let's locate the oldest table entry
	{
		for(i=0;i<PROTOCOL_MAX_ARP_TABLE;i++)
		{
			if(i==0)
			{
				tmp = ARPTable[i].Seq;
				oldest = i;
			}
			else{
				if(tmp>ARPTable[i].Seq)
				{
					tmp = ARPTable[i].Seq;
					oldest = i;
				}
			}
		}
		ARPDEBUG(("[ARP]Update ARP Table In [%d]\n",oldest));	
		ARPTable[oldest].MAC[0] = MAC[0];
		ARPTable[oldest].MAC[1] = MAC[1];
		ARPTable[oldest].MAC[2] = MAC[2];
		ARPTable[oldest].MAC[3] = MAC[3];
		ARPTable[oldest].MAC[4] = MAC[4];
		ARPTable[oldest].MAC[5] = MAC[5];

		ARPTable[oldest].IP[0] = IP[0];
		ARPTable[oldest].IP[1] = IP[1];
		ARPTable[oldest].IP[2] = IP[2];
		ARPTable[oldest].IP[3] = IP[3];

		ARPTable[oldest].Seq = ARPTableSeq;
		ARPTableSeq++;
		ARPTable[oldest].IsUsed = true;
	}



}


void PROTOCOL_ICMPHandler(u8 *data,u16 length, PROTOCOL_IPFormat *ipHeader)
{
	u16 *tmp;
	u32 tmp2;
	int i;
	PROTOCOL_ICMPFormat *format = (PROTOCOL_ICMPFormat *)data;
	ICMPDEBUG(("[ICMP]Type=0x%x",format->type));
	switch(format->type)
	{
		case(PROTOCOL_ICMP_ECHO_REPLY):
			ICMPDEBUG(("==>ECHO REPLY\n"));
			ICMPDEBUG(("[ICMP]Code=0x%x\n",format->code));
			ICMPDEBUG(("[ICMP]ID=%d\n",((u16)data[4])<<8 |((u16)data[5])));
			ICMPDEBUG(("[ICMP]Seq=%d\n",((u16)data[6])<<8 |((u16)data[7])));
			ICMPDEBUG(("[ICMP]Length=%d\n",length));
			break;
		case(PROTOCOL_ICMP_ECHO_REQUEST):
			ICMPDEBUG(("==>ECHO REQUEST\n"));
			ICMPDEBUG(("[ICMP]Code=0x%x\n",format->code));
			ICMPDEBUG(("[ICMP]ID=%d\n",((u16)data[4])<<8 |((u16)data[5])));
			ICMPDEBUG(("[ICMP]Seq=%d\n",((u16)data[6])<<8 |((u16)data[7])));
			ICMPDEBUG(("[ICMP]Length=%d\n",length));

			format->type=PROTOCOL_ICMP_ECHO_REPLY;
			format->checksum = 0;
			
			//let's calculate checksum(1's complement)
			tmp = (u16 *)data;
			tmp2=0;
			for(i=0;i<length/2;i++)tmp2+=SWAP16(tmp[i]);
			format->checksum = (u16)tmp2;
			format->checksum += (u16)(tmp2>>16);
			format->checksum = ~format->checksum;
			format->checksum = SWAP16(format->checksum);
			
			PROTOCOL_IPSender(data,length,PROTOCOL_IP_ICMP,ipHeader->SrcIP,ipHeader->ID);
			break;
		default:
			ICMPDEBUG(("==>See Reference\n"));
			break;
	}	
	
}

void PROTOCOL_IPSender(u8 *data,u16 length,u8 Protocol,u8 *DestIP,u16 ID)
{
	u8 buf[2048];
	int i;
	u8 DestMAC[6];
	u16 *tmp;
	u32 tmp2;
	PROTOCOL_IPFormat *format = (PROTOCOL_IPFormat *)buf;
	format->Version_IHL = 4<<4;
	format->Version_IHL |= 5;
	format->TOS = 0;
	format->Length = SWAP16(length+20);
	format->ID = IPSeq;
	IPSeq++;
	format->Flags_Fragment =0;
	format->TTL = 128;
	format->Protocol = Protocol;

	format->SrcIP[0] = SrcIPAddr[0];
	format->SrcIP[1] = SrcIPAddr[1];
	format->SrcIP[2] = SrcIPAddr[2];
	format->SrcIP[3] = SrcIPAddr[3];

	format->DestIP[0] = DestIP[0];
	format->DestIP[1] = DestIP[1];
	format->DestIP[2] = DestIP[2];
	format->DestIP[3] = DestIP[3];

	//let's calculate checksum (1's complment checksum)
	tmp = (u16 *)buf;
	tmp2 = (u32)SWAP16(tmp[0])+(u32)SWAP16(tmp[1])+(u32)SWAP16(tmp[2])+(u32)SWAP16(tmp[3])
		+(u32)SWAP16(tmp[4])+(u32)SWAP16(tmp[6])+(u32)SWAP16(tmp[7])+(u32)SWAP16(tmp[8])+(u32)SWAP16(tmp[9]);
	format->HeaderChecksum = (u16)tmp2;
	format->HeaderChecksum += tmp2>>16;
	format->HeaderChecksum = ~format->HeaderChecksum;
	format->HeaderChecksum = SWAP16(format->HeaderChecksum);

	for(i=0;i<length;i++)buf[20+i]=data[i];

	if(ARPGetMAC(DestIP,DestMAC)==false)
	{
		IPDEBUG(("[IP]Fail to get dest MAC address of IP[%d.%d.%d.%d]\n",
			DestIP[0],DestIP[1],DestIP[2],DestIP[3]));

		PROTOCOL_ARPRequest(DestIP);
		while(1)
		{
			if(ARPGetMAC(DestIP,DestMAC))break;
		}
	}
	PROTOCOL_EthernetSender(buf,length+20, PROTOCOL_ETHERNET_IPv4, DestMAC);
}

u32 *PROTOCOL_GetIPBuffer()
{
	u32 *buf = (u32 *)PROTOCOL_GetEthernetBuffer();
	return &buf[5];
}

void PROTOCOL_IPSenderFast(u16 length,u8 Protocol,u8 *DestIP,u16 ID)
{
	u8 *buf;
	int i;
	u8 DestMAC[6];
	u16 *tmp;
	u32 tmp2;
	PROTOCOL_IPFormat *format;
	buf = (u8 *)PROTOCOL_GetEthernetBuffer();
	format= (PROTOCOL_IPFormat *)buf;
	format->Version_IHL = 4<<4;
	format->Version_IHL |= 5;
	format->TOS = 0;
	format->Length = SWAP16(length+20);
	format->ID = IPSeq;
	IPSeq++;
	format->Flags_Fragment =0;
	format->TTL = 128;
	format->Protocol = Protocol;

	format->SrcIP[0] = SrcIPAddr[0];
	format->SrcIP[1] = SrcIPAddr[1];
	format->SrcIP[2] = SrcIPAddr[2];
	format->SrcIP[3] = SrcIPAddr[3];

	format->DestIP[0] = DestIP[0];
	format->DestIP[1] = DestIP[1];
	format->DestIP[2] = DestIP[2];
	format->DestIP[3] = DestIP[3];

	//let's calculate checksum (1's complment checksum)
	tmp = (u16 *)buf;
	tmp2 = (u32)SWAP16(tmp[0])+(u32)SWAP16(tmp[1])+(u32)SWAP16(tmp[2])+(u32)SWAP16(tmp[3])
		+(u32)SWAP16(tmp[4])+(u32)SWAP16(tmp[6])+(u32)SWAP16(tmp[7])+(u32)SWAP16(tmp[8])+(u32)SWAP16(tmp[9]);
	format->HeaderChecksum = (u16)tmp2;
	format->HeaderChecksum += tmp2>>16;
	format->HeaderChecksum = ~format->HeaderChecksum;
	format->HeaderChecksum = SWAP16(format->HeaderChecksum);

//	for(i=0;i<length;i++)buf[20+i]=data[i];

	#if 0
	Disp("IPFast buf:");
	for(i=0;i<length;i++)
	{
		if(i%16==0)Disp("\n");
		Disp("%X ",buf[i]);
	}
	Disp("\n");
	#endif

	if(ARPGetMAC(DestIP,DestMAC)==false)
	{
		IPDEBUG(("[IP]Fail to get dest MAC address of IP[%d.%d.%d.%d]\n",
			DestIP[0],DestIP[1],DestIP[2],DestIP[3]));

		PROTOCOL_ARPRequest(DestIP);
		while(1)
		{
			if(ARPGetMAC(DestIP,DestMAC))break;
		}
	}
	PROTOCOL_EthernetSenderFast(length+20, PROTOCOL_ETHERNET_IPv4, DestMAC);
}


void PROTOCOL_IPHandler(u8 *data,u16 length,PROTOCOL_ETHERNETFormat *MACFormat)
{
	PROTOCOL_IPFormat *format = ( PROTOCOL_IPFormat *)data;
	IPDEBUG(("[IP]Version=%d\n",(format->Version_IHL & 0xf0)>>4));
	IPDEBUG(("[IP]HeaderLength=%d\n",(format->Version_IHL & 0x0f)*4));
	IPDEBUG(("[IP]TotalLength=%d\n",SWAP16(format->Length)));
	IPDEBUG(("[IP]ID=0x%x\n",SWAP16(format->ID)));
	IPDEBUG(("[IP]TTL=%d\n",format->TTL));
	IPDEBUG(("[IP]SrcAddr=%d.%d.%d.%d\n",format->SrcIP[0],format->SrcIP[1],
		format->SrcIP[2],format->SrcIP[3]));
	IPDEBUG(("[IP]DestAddr=%d.%d.%d.%d\n",format->DestIP[0],format->DestIP[1],
		format->DestIP[2],format->DestIP[3]));

	ARPTableUpdate(format->SrcIP, MACFormat->SrcMAC);

	IPDEBUG(("[IP]Protocol=0x%x\n",format->Protocol));

	if(format->DestIP[0] == SrcIPAddr[0] && format->DestIP[1] == SrcIPAddr[1] &&
		format->DestIP[2] == SrcIPAddr[2] && format->DestIP[3] == SrcIPAddr[3])
	{
	}
	else return;
		
	switch(format->Protocol)
	{
		case(PROTOCOL_IP_ICMP):IPDEBUG(("==>ICMP\n"));
			PROTOCOL_ICMPHandler(&(data[(format->Version_IHL & 0x0f)*4]), (length-(format->Version_IHL & 0x0f)*4),format);
			break;
		case(PROTOCOL_IP_IGMP):IPDEBUG(("==>IGMP\n"));
			break;
		case(PROTOCOL_IP_TCP):IPDEBUG(("==>TCP\n"));
			break;
		case(PROTOCOL_IP_UDP):IPDEBUG(("==>UDP\n"));
			if(UDPHandler)UDPHandler(&(data[(format->Version_IHL & 0x0f)*4]), (length-(format->Version_IHL & 0x0f)*4));
			break;
		default :IPDEBUG(("==> See reference\n"));
			break;
	}
	
	
}


void PROTOCOL_ARPHandler(u8 *data,u16 length)
{
	int i;
	bool isFound;
	u32 oldest;
	u32 tmp;
	u16 tmp16;
	PROTOCOL_ARPFormat *format = (PROTOCOL_ARPFormat *)data;
	ARPDEBUG(("[ARP]HWType=0x%x ProtocolType=0x%x HLEN=0x%x PLEN=0x%x Operation=0x%x\n",
		SWAP16(format->HWType),SWAP16(format->ProtocolType),format->HLEN,format->PLEN,SWAP16(format->Operation)));
	ARPDEBUG(("[ARP]Sender MAC=%X-%X-%X-%X-%X-%X\n",format->SenderMAC[0],format->SenderMAC[1],
		format->SenderMAC[2],format->SenderMAC[3],format->SenderMAC[4],format->SenderMAC[5]));
	ARPDEBUG(("[ARP]Sender IP=%d.%d.%d.%d\n",format->SenderIP[0],format->SenderIP[1],format->SenderIP[2],
		format->SenderIP[3]));
	ARPDEBUG(("[ARP]Target MAC=%X-%X-%X-%X-%X-%X\n",format->TargetMAC[0],format->TargetMAC[1],
		format->TargetMAC[2],format->TargetMAC[3],format->TargetMAC[4],format->TargetMAC[5]));
	ARPDEBUG(("[ARP]Target IP=%d.%d.%d.%d\n",format->TargetIP[0],format->TargetIP[1],
		format->TargetIP[2],format->TargetIP[3]));

	ARPTableUpdate(format->SenderIP,format->SenderMAC);

	if(SWAP16(format->Operation)==1 &&
format->TargetIP[0] == SrcIPAddr[0] && format->TargetIP[1] == SrcIPAddr[1] &&
		format->TargetIP[2] == SrcIPAddr[2] && format->TargetIP[3] == SrcIPAddr[3])
	{
		tmp16 = 2; //reply operation code
		format->Operation=SWAP16(tmp16);

		format->TargetIP[0] = format->SenderIP[0];
		format->TargetIP[1] = format->SenderIP[1];
		format->TargetIP[2] = format->SenderIP[2];
		format->TargetIP[3] = format->SenderIP[3];

		format->TargetMAC[0] = format->SenderMAC[0];
		format->TargetMAC[1] = format->SenderMAC[1];
		format->TargetMAC[2] = format->SenderMAC[2];
		format->TargetMAC[3] = format->SenderMAC[3];
		format->TargetMAC[4] = format->SenderMAC[4];
		format->TargetMAC[5] = format->SenderMAC[5];
		
		format->SenderIP[0] = SrcIPAddr[0];
		format->SenderIP[1] = SrcIPAddr[1];
		format->SenderIP[2] = SrcIPAddr[2];
		format->SenderIP[3] = SrcIPAddr[3];
	
		format->SenderMAC[0] = SrcEthAddr[0];
		format->SenderMAC[1] = SrcEthAddr[1];
		format->SenderMAC[2] = SrcEthAddr[2];
		format->SenderMAC[3] = SrcEthAddr[3];
		format->SenderMAC[4] = SrcEthAddr[4];
		format->SenderMAC[5] = SrcEthAddr[5];
		
		PROTOCOL_EthernetSender((u8*)format, sizeof(PROTOCOL_ARPFormat), PROTOCOL_ETHERNET_ARP, format->TargetMAC);
	}


}

void PROTOCOL_ARPRequest(u8 *targetIPAddress)
{
	PROTOCOL_ARPFormat format;
	u16 tmp;
	u8 targetMAC[6]={0xff,0xff,0xff,0xff,0xff,0xff};
	tmp=0x1;
	format.HWType = SWAP16(tmp);
	tmp=0x800;
	format.ProtocolType = SWAP16(tmp);
	format.HLEN = 0x6;
	format.PLEN =0x4;
	tmp =1;
	format.Operation =SWAP16(tmp);
	format.SenderIP[0] = SrcIPAddr[0];
	format.SenderIP[1] = SrcIPAddr[1];
	format.SenderIP[2] = SrcIPAddr[2];
	format.SenderIP[3] = SrcIPAddr[3];
	format.SenderMAC[0] = SrcEthAddr[0];
	format.SenderMAC[1] = SrcEthAddr[1];
	format.SenderMAC[2] = SrcEthAddr[2];
	format.SenderMAC[3] = SrcEthAddr[3];
	format.SenderMAC[4] = SrcEthAddr[4];
	format.SenderMAC[5] = SrcEthAddr[5];
	format.TargetIP[0] = targetIPAddress[0];
	format.TargetIP[1] = targetIPAddress[1];
	format.TargetIP[2] = targetIPAddress[2];
	format.TargetIP[3] = targetIPAddress[3];
	format.TargetMAC[0] = 0;
	format.TargetMAC[1] = 0;
	format.TargetMAC[2] = 0;
	format.TargetMAC[3] = 0;
	format.TargetMAC[4] = 0;
	format.TargetMAC[5] = 0;

	ARPDEBUG(("[ARP] Reqest IP[%u.%u.%u.%u]\n",targetIPAddress[0],targetIPAddress[1],targetIPAddress[2],targetIPAddress[3]));
	PROTOCOL_EthernetSender((u8*)(&format), sizeof(PROTOCOL_ARPFormat), PROTOCOL_ETHERNET_ARP, targetMAC);
	
}

void PROTOCOL_EthernetHandler(u8 *data, u16 length)
{
	int i;
	u16 frameDataLength;
	PROTOCOL_ETHERNETFormat *format = (PROTOCOL_ETHERNETFormat *)data;
	
	ETHERNETDEBUG(("==================frame incoming====================\n"));
	ETHERNETDEBUG(("[ETH]Dest Addr=%X-%X-%X-%X-%X-%X\n",data[0],data[1],data[2],data[3],data[4],data[5]));
	ETHERNETDEBUG(("[ETH]Src Addr=%X-%X-%X-%X-%X-%X\n",data[6],data[7],data[8],data[9],data[10],data[11]));
	frameDataLength = (u16)(data[12] <<8) + (u16)(data[13] <<0);
	if(frameDataLength>0x05dc)
	{
		switch(frameDataLength)
		{
			case(PROTOCOL_ETHERNET_IPv4):
				ETHERNETDEBUG(("[ETH]Ethernet Frame Type: IPv4\n"));PROTOCOL_IPHandler(&(data[14]),length-14,format);break;
			case(PROTOCOL_ETHERNET_ARP):
				ETHERNETDEBUG(("[ETH]Ethernet Frame Type: ARP\n"));PROTOCOL_ARPHandler(&(data[14]),length-14);break;
			case(PROTOCOL_ETHERNET_IPv6):
				ETHERNETDEBUG(("[ETH]Ethernet Frame Type: IPv6\n"));break;
			default:
				ETHERNETDEBUG(("[ETH]Ethernet Frame Type: See reference 0x%x\n",frameDataLength));break;
		}
	}
	else ETHERNETDEBUG(("[ETH]Ethernet length=0x%x\n",frameDataLength));
	for(i=14;i<length;i++)
	{
		
	}
	ETHERNETDEBUG(("================================================\n"));
}

void PROTOCOL_EthernetSender(u8 *data,u16 length,u16 type,u8 *DestMAC)
{
	int i;
	u8 buf[2048];
	buf[0] = DestMAC[0];
	buf[1] = DestMAC[1];
	buf[2] = DestMAC[2];
	buf[3] = DestMAC[3];
	buf[4] = DestMAC[4];
	buf[5] = DestMAC[5];

	buf[6] = SrcEthAddr[0];
	buf[7] = SrcEthAddr[1];
	buf[8] = SrcEthAddr[2];
	buf[9] = SrcEthAddr[3];
	buf[10] = SrcEthAddr[4];
	buf[11] = SrcEthAddr[5];

	if(SWAP16(type)==PROTOCOL_ETHERNET_NORMAL)
	{
		*((u16 *)(&buf[12])) = length;
	}
	else *((u16 *)(&buf[12])) = SWAP16(type);
	
	for(i=0;i<length;i++)buf[14+i] = data[i];

	ETHERNETDEBUG(("[ETH]Sending Frame Src:%X-%X-%X-%X-%X-%X  to %X-%X-%X-%X-%X-%X\n"
		,SrcEthAddr[0],SrcEthAddr[1],SrcEthAddr[2],SrcEthAddr[3],SrcEthAddr[4],SrcEthAddr[5],
		DestMAC[0],DestMAC[1],DestMAC[2],DestMAC[3],DestMAC[4],DestMAC[5]));
	Sender(buf,length+6+6+2);
	
}

u32 *PROTOCOL_GetEthernetBuffer()
{
	u32 temp;
	temp = (u32)(WriteBuffer);
	return (u32 *)(temp+6+6+4);
}

void PROTOCOL_EthernetSenderFast(u16 length,u16 type,u8 *DestMAC)
{
	int i;
	u8 *buf = (u8 *)WriteBuffer;
	buf[2] = DestMAC[0];
	buf[3] = DestMAC[1];
	buf[4] = DestMAC[2];
	buf[5] = DestMAC[3];
	buf[6] = DestMAC[4];
	buf[7] = DestMAC[5];

	if(SWAP16(type)==PROTOCOL_ETHERNET_NORMAL)
	{
		*((u16 *)(&buf[14])) = length;
	}
	else *((u16 *)(&buf[14])) = SWAP16(type);
	
	//for(i=0;i<length;i++)buf[14+i] = data[i];

	ETHERNETDEBUG(("[ETH]Sending Frame Src:%X-%X-%X-%X-%X-%X  to %X-%X-%X-%X-%X-%X\n"
		,SrcEthAddr[0],SrcEthAddr[1],SrcEthAddr[2],SrcEthAddr[3],SrcEthAddr[4],SrcEthAddr[5],
		DestMAC[0],DestMAC[1],DestMAC[2],DestMAC[3],DestMAC[4],DestMAC[5]));
	Sender(buf+2,length+6+6+2);
	
}

void PROTOCOL_InitStack(u8* srcEthernetAddr,u8 *srcIPAddress,
	void (*sender)(u8 *data, u16 length))
{
	int i;
	u8 *buf = (u8 *)WriteBuffer;
	SrcEthAddr[0] = srcEthernetAddr[0];
	SrcEthAddr[1] = srcEthernetAddr[1];
	SrcEthAddr[2] = srcEthernetAddr[2];
	SrcEthAddr[3] = srcEthernetAddr[3];
	SrcEthAddr[4] = srcEthernetAddr[4];
	SrcEthAddr[5] = srcEthernetAddr[5];

	SrcIPAddr[0] = srcIPAddress[0];
	SrcIPAddr[1] = srcIPAddress[1];
	SrcIPAddr[2] = srcIPAddress[2];
	SrcIPAddr[3] = srcIPAddress[3];

	buf[8] = SrcEthAddr[0];
	buf[9] = SrcEthAddr[1];
	buf[10] = SrcEthAddr[2];
	buf[11] = SrcEthAddr[3];
	buf[12] = SrcEthAddr[4];
	buf[13] = SrcEthAddr[5];
	
	Sender = sender;

	ARPTableSeq=0;
	for(i=0;i<PROTOCOL_MAX_ARP_TABLE;i++)
	{
		ARPTable[i].IsUsed = false;
		ARPTable[i].Seq = ARPTableSeq;
		ARPTableSeq++;
	}
}

void PROTOCOL_InitStackWithUDP(u8* srcEthernetAddr,u8 *srcIPAddress,
	void (*sender)(u8 *data, u16 length),void (*udpHandler)(u8 *data,u16 length))
{
	PROTOCOL_InitStack(srcEthernetAddr,srcIPAddress,sender);
	UDPHandler = udpHandler;
}


void PROTOCOL_FrameHander(u8 *data,u16 length)
{
	PROTOCOL_EthernetHandler(data,length);
}

