/**************************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2008 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : ipc.c
*  
*	File Description : This file implements the API functons for Interlaced to Progressive convertor.
*
*	Author : Won Jung
*	Dept. : AP Development Team
*	Version : 0.1 
* 
*	History
*	- Created(Won Jung 2009/3/3)
*  
**************************************************************************************/

#include <string.h>
#include "def.h"
#include "option.h"
#include "library.h"
#include "intc.h"
#include "util.h"
#include "ipc.h"
#include "ipc_table.h"




#ifdef DISP_FIMC_REG
#define IpcOutp32(offset, x) {u32 a2, d2; a2=IPC_v210_BASE+offset; d2=x; Disp("Outp32(0x%08x, 0x%08x); \n", a2, d2); Outp32(a2, d2);}
#define IpcOutp16(offset, x) {u32 a2, d2; a2=IPC_v210_BASE+offset; d2=x; Disp("Outp16(0x%08x, 0x%08x);\n", a2, d2); Outp16(a2, d2);}
#define IpcOutp8(offset, x) {u32 a2, d2; a2=IPC_v210_BASE+offset; d2=x; Disp("Outp8(0x%08x, 0x%08x);\n", a2, d2); Outp8(a2, d2);}

#define IpcInp32(offset, x) {u32 a2, d2; a2=IPC_v210_BASE+offset; d2=Inp32(a2); x=d2; Disp("d = Inp32(0x%08x); // d=0x%08x\n", a2, d2);}
#define IpcInp16(offset, x) {u32 a2, d2; a2=IPC_v210_BASE+offset; d2=Inp16(a2); x=d2; Disp("d = Inp16(0x%08x); // d=0x%08x\n", a2, d2);}
#define IpcInp8(offset, x) {u32 a2, d2; a2=IPC_v210_BASE+offset; d2=Inp8(a2); x=d2; Disp("d = Inp8(0x%08x); // d=0x%08x\n", a2, ,d2);}

//#define fimcOutp32(offset, x) {u32 a2, d2; a2=g_uBaseAddress+offset; d2=x; Disp("Outp32(\'h%08x, \'h%08x); \/\/ %s(line %d)\n", a2, d2, __FILE__, __LINE__); Outp32(a2, d2);}
//#define fimcInp32(offset, x) {u32 a2, d2; a2=g_uBaseAddress+offset; Inp32(a2, d2); x=d2; Disp("Inp32(\'h%08x, d); \/\/ %s(line %d)// d=0x%08x\n", a2, __FILE__, __LINE__,d2);}
//#define fimcOutp16(offset, x) {u32 a2, d2; a2=g_uBaseAddress+offset; d2=x; Disp("Outp16(\'h%08x, \'h%08x); \/\/ %s(line %d)\n", a2, d2, __FILE__, __LINE__); Outp16(a2, d2);}
//#define fimcInp16(offset, x) {u32 a2, d2; a2=g_uBaseAddress+offset; Inp16(a2, d2); x=d2; Disp("Inp16(\'h%08x, d); \/\/ %s(line %d)// d=0x%08x\n", a2, __FILE__, __LINE__,d2);}
//#define fimcOutp8(offset, x) {u32 a2, d2; a2=g_uBaseAddress+offset; d2=x; Disp("Outp8(\'h%08x, \'h%08x); \/\/ %s(line %d)\n", a2, d2, __FILE__, __LINE__); Outp8(a2, d2);}
//#define fimcInp8(offset, x) {u32 a2, d2; a2=g_uBaseAddress+offset; Inp8(a2, d2); x=d2; Disp("Inp8(\'h%08x, d); \/\/ %s(line %d)// d=0x%08x\n", a2, __FILE__, __LINE__,d2);}
#else
#define IpcOutp32(offset, x) Outp32(IPC_v210_BASE+offset, x)
#define IpcOutp16(offset, x) Outp16(IPC_v210_BASE+offset, x)
#define IpcOutp8(offset, x) Outp8(IPC_v210_BASE+offset, x)

#define IpcInp32(offset, x) {x=Inp32(IPC_v210_BASE+offset);}
#define IpcInp16(offset, x) {x=Inp16(IPC_v210_BASE+offset);}
#define IpcInp8(offset, x) {x=Inp8(IPC_v210_BASE+offset);}
#endif


typedef union
{
	u32 val;
	/* IPC_ENABLE */
	struct{
		u32 IPC_ON:1;
		u32 reserved1:1;
		u32 IPC_ON_S:1;
		u32 reserved2:29;
		}IPC_ENABLE;
	/* IPC_SRESET */
	struct{
		u32 IPC_SRESET:1;
		u32 reserved1:31;
		}IPC_SRESET;
	/* IPC_SHADOW_UPDATE */
	struct{
		u32 IPC_SHADOW_UPDATE:1;
		u32 reserved1:31;
		}IPC_SHADOW_UPDATE;
	/* IPC_FIELD_ID */
	struct{
		u32 IPC_FILED_ID:1;
		u32 reserved1:31;
		}IPC_FIELD_ID;
	/* IPC_MODE */
	struct{
		u32 reserved1:1;
		u32 IPC_2D:1;
		u32 FIELD_ID_AUTO_TOGGLING:1;
		u32 reserved2:3;
		u32 FIELD_ID_SELECTION:1;
		u32 INT_EN:1;
		u32 reserved3:24;
		}IPC_MODE;
	/* IPC_PEL_RATE_CTRL */
	struct{
		u32 IPC_PEL_RATE_CTRL:2;
		u32 reserved:30;
		}IPC_PEL_RATE_CTRL;
	/* IPC_SRC_WIDTH */
	struct{
		u32 IPC_SRC_WIDTH:11;
		u32 reserved1:21;
		}IPC_SRC_WIDTH;
	/* IPC_SRC_HEIGHT */
	struct{
		u32 IPC_SRC_HEIGHT:10;
		u32 reserved1:22;
		}IPC_SRC_HEIGHT;
	/* IPC_DST_WIDTH */
	struct{
		u32 IPC_DST_WIDTH:11;
		u32 reserved1:21;
		}IPC_DST_WIDTH;
	/* IPC_DST_HEIGHT */
	struct{
		u32 IPC_DST_HEIGHT:10;
		u32 reserved1:22;
		}IPC_DST_HEIGHT;
	/* IPC_H_RATIO */
	struct{
		u32 IPC_H_RATIO:19;
		u32 reserved1:13;
		}IPC_H_RATIO;
	/* IPC_V_RATIO */
	struct{
		u32 IPC_V_RATIO:19;
		u32 reserved1:13;
		}IPC_V_RATIO;
	/* IPC_POLY8_Y0_LL */
	struct{
		u32 IPC_poly8_y0_ph3:3;
		u32 reserved1:5;
		u32 IPC_poly8_y0_ph2:3;
		u32 reserved2:5;
		u32 IPC_poly8_y0_ph1:3;
		u32 reserved3:5;
		u32 IPC_poly8_y0_ph0:3;
		u32 reserved4:5;
		}IPC_POLY8_Y0_LL;
	/* IPC_POLY8_Y0_LH */
	struct{
		u32 IPC_poly8_y0_ph7:3;
		u32 reserved1:5;
		u32 IPC_poly8_y0_ph6:3;
		u32 reserved2:5;
		u32 IPC_poly8_y0_ph5:3;
		u32 reserved3:5;
		u32 IPC_poly8_y0_ph4:3;
		u32 reserved4:5;
		}IPC_POLY8_Y0_LH;
	/* IPC_POLY8_Y0_HL */
	struct{
		u32 IPC_poly8_y0_ph11:3;
		u32 reserved1:5;
		u32 IPC_poly8_y0_ph10:3;
		u32 reserved2:5;
		u32 IPC_poly8_y0_ph9:3;
		u32 reserved3:5;
		u32 IPC_poly8_y0_ph8:3;
		u32 reserved4:5;
		}IPC_POLY8_Y0_HL;
	/* IPC_POLY8_Y0_HH */
	struct{
		u32 IPC_poly8_y0_ph15:3;
		u32 reserved1:5;
		u32 IPC_poly8_y0_ph14:3;
		u32 reserved2:5;
		u32 IPC_poly8_y0_ph13:3;
		u32 reserved3:5;
		u32 IPC_poly8_y0_ph12:3;
		u32 reserved4:5;
		}IPC_POLY8_Y0_HH;
	/* IPC_POLY8_Y1_LL */
	struct{
		u32 IPC_poly8_y1_ph3:5;
		u32 reserved1:3;
		u32 IPC_poly8_y1_ph2:5;
		u32 reserved2:3;
		u32 IPC_poly8_y1_ph1:5;
		u32 reserved3:3;
		u32 IPC_poly8_y1_ph0:5;
		u32 reserved4:3;
		}IPC_POLY8_Y1_LL;
	/* IPC_POLY8_Y1_LH */
	struct{
		u32 IPC_poly8_y1_ph7:5;
		u32 reserved1:3;
		u32 IPC_poly8_y1_ph6:5;
		u32 reserved2:3;
		u32 IPC_poly8_y1_ph5:5;
		u32 reserved3:3;
		u32 IPC_poly8_y1_ph4:5;
		u32 reserved4:3;
		}IPC_POLY8_Y1_LH;
	/* IPC_POLY8_Y1_HL */
	struct{
		u32 IPC_poly8_y1_ph11:5;
		u32 reserved1:3;
		u32 IPC_poly8_y1_ph10:5;
		u32 reserved2:3;
		u32 IPC_poly8_y1_ph9:5;
		u32 reserved3:3;
		u32 IPC_poly8_y1_ph8:5;
		u32 reserved4:3;
		}IPC_POLY8_Y1_HL;
	/* IPC_POLY8_Y1_HH */
	struct{
		u32 IPC_poly8_y1_ph15:5;
		u32 reserved1:3;
		u32 IPC_poly8_y1_ph14:5;
		u32 reserved2:3;
		u32 IPC_poly8_y1_ph13:5;
		u32 reserved3:3;
		u32 IPC_poly8_y1_ph12:5;
		u32 reserved4:3;
		}IPC_POLY8_Y1_HH;
	/* IPC_POLY8_Y2_LL */
	struct{
		u32 IPC_poly8_y2_ph3:7;
		u32 reserved1:1;
		u32 IPC_poly8_y2_ph2:7;
		u32 reserved2:1;
		u32 IPC_poly8_y2_ph1:7;
		u32 reserved3:1;
		u32 IPC_poly8_y2_ph0:7;
		u32 reserved4:1;
		}IPC_POLY8_Y2_LL;
	/* IPC_POLY8_Y2_LH */
	struct{
		u32 IPC_poly8_y2_ph7:7;
		u32 reserved1:1;
		u32 IPC_poly8_y2_ph6:7;
		u32 reserved2:1;
		u32 IPC_poly8_y2_ph5:7;
		u32 reserved3:1;
		u32 IPC_poly8_y2_ph4:7;
		u32 reserved4:1;
		}IPC_POLY8_Y2_LH;
	/* IPC_POLY8_Y2_HL */
	struct{
		u32 IPC_poly8_y2_ph11:7;
		u32 reserved1:1;
		u32 IPC_poly8_y2_ph10:7;
		u32 reserved2:1;
		u32 IPC_poly8_y2_ph9:7;
		u32 reserved3:1;
		u32 IPC_poly8_y2_ph8:7;
		u32 reserved4:1;
		}IPC_POLY8_Y2_HL;
	/* IPC_POLY8_Y2_HH */
	struct{
		u32 IPC_poly8_y2_ph15:7;
		u32 reserved1:1;
		u32 IPC_poly8_y2_ph14:7;
		u32 reserved2:1;
		u32 IPC_poly8_y2_ph13:7;
		u32 reserved3:1;
		u32 IPC_poly8_y2_ph12:7;
		u32 reserved4:1;
		}IPC_POLY8_Y2_HH;
	/* IPC_POLY8_Y3_LL */
	struct{
		u32 IPC_poly8_y3_ph3:7;
		u32 reserved1:1;
		u32 IPC_poly8_y3_ph2:7;
		u32 reserved2:1;
		u32 IPC_poly8_y3_ph1:7;
		u32 reserved3:1;
		u32 IPC_poly8_y3_ph0:7;
		u32 reserved4:1;
		}IPC_POLY8_Y3_LL;
	/* IPC_POLY8_Y3_LH */
	struct{
		u32 IPC_poly8_y3_ph7:7;
		u32 reserved1:1;
		u32 IPC_poly8_y3_ph6:7;
		u32 reserved2:1;
		u32 IPC_poly8_y3_ph5:7;
		u32 reserved3:1;
		u32 IPC_poly8_y3_ph4:7;
		u32 reserved4:1;
		}IPC_POLY8_Y3_LH;
	/* IPC_POLY8_Y3_HL */
	struct{
		u32 IPC_poly8_y3_ph11:7;
		u32 reserved1:1;
		u32 IPC_poly8_y3_ph10:7;
		u32 reserved2:1;
		u32 IPC_poly8_y3_ph9:7;
		u32 reserved3:1;
		u32 IPC_poly8_y3_ph8:7;
		u32 reserved4:1;
		}IPC_POLY8_Y3_HL;
	/* IPC_POLY8_Y3_HH */
	struct{
		u32 IPC_poly8_y3_ph15:7;
		u32 reserved1:1;
		u32 IPC_poly8_y3_ph14:7;
		u32 reserved2:1;
		u32 IPC_poly8_y3_ph13:7;
		u32 reserved3:1;
		u32 IPC_poly8_y3_ph12:7;
		u32 reserved4:1;
		}IPC_POLY8_Y3_HH;	
	/* IPC_POLY4_Y0_LL */
	struct{
		u32 IPC_poly4_y0_ph3:6;
		u32 reserved1:2;
		u32 IPC_poly4_y0_ph2:6;
		u32 reserved2:2;
		u32 IPC_poly4_y0_ph1:6;
		u32 reserved3:2;
		u32 IPC_poly4_y0_ph0:6;
		u32 reserved4:2;
		}IPC_POLY4_Y0_LL;
	/* IPC_POLY4_Y0_LH */
	struct{
		u32 IPC_poly4_y0_ph7:6;
		u32 reserved1:2;
		u32 IPC_poly4_y0_ph6:6;
		u32 reserved2:2;
		u32 IPC_poly4_y0_ph5:6;
		u32 reserved3:2;
		u32 IPC_poly4_y0_ph4:6;
		u32 reserved4:2;
		}IPC_POLY4_Y0_LH;
	/* IPC_POLY4_Y0_HL */
	struct{
		u32 IPC_poly4_y0_ph11:6;
		u32 reserved1:2;
		u32 IPC_poly4_y0_ph10:6;
		u32 reserved2:2;
		u32 IPC_poly4_y0_ph9:6;
		u32 reserved3:2;
		u32 IPC_poly4_y0_ph8:6;
		u32 reserved4:2;
		}IPC_POLY4_Y0_HL;
	/* IPC_POLY4_Y0_HH */
	struct{
		u32 IPC_poly4_y0_ph15:6;
		u32 reserved1:2;
		u32 IPC_poly4_y0_ph14:6;
		u32 reserved2:2;
		u32 IPC_poly4_y0_ph13:6;
		u32 reserved3:2;
		u32 IPC_poly4_y0_ph12:6;
		u32 reserved4:2;
		}IPC_POLY4_Y0_HH;
	/* IPC_POLY4_Y1_LL */
	struct{
		u32 IPC_poly4_y1_ph3:7;
		u32 reserved1:1;
		u32 IPC_poly4_y1_ph2:7;
		u32 reserved2:1;
		u32 IPC_poly4_y1_ph1:7;
		u32 reserved3:1;
		u32 IPC_poly4_y1_ph0:7;
		u32 reserved4:1;
		}IPC_POLY4_Y1_LL;
	/* IPC_POLY4_Y1_LH */
	struct{
		u32 IPC_poly4_y1_ph7:7;
		u32 reserved1:1;
		u32 IPC_poly4_y1_ph6:7;
		u32 reserved2:1;
		u32 IPC_poly4_y1_ph5:7;
		u32 reserved3:1;
		u32 IPC_poly4_y1_ph4:7;
		u32 reserved4:1;
		}IPC_POLY4_Y1_LH;
	/* IPC_POLY4_Y1_HL */
	struct{
		u32 IPC_poly4_y1_ph11:7;
		u32 reserved1:1;
		u32 IPC_poly4_y1_ph10:7;
		u32 reserved2:1;
		u32 IPC_poly4_y1_ph9:7;
		u32 reserved3:1;
		u32 IPC_poly4_y1_ph8:7;
		u32 reserved4:1;
		}IPC_POLY4_Y1_HL;
	/* IPC_POLY4_Y1_HH */
	struct{
		u32 IPC_poly4_y1_ph15:7;
		u32 reserved1:1;
		u32 IPC_poly4_y1_ph14:7;
		u32 reserved2:1;
		u32 IPC_poly4_y1_ph13:7;
		u32 reserved3:1;
		u32 IPC_poly4_y1_ph12:7;
		u32 reserved4:1;
		}IPC_POLY4_Y1_HH;
	/* IPC_POLY4_Y2_LL */
	struct{
		u32 IPC_poly4_y2_ph3:7;
		u32 reserved1:1;
		u32 IPC_poly4_y2_ph2:7;
		u32 reserved2:1;
		u32 IPC_poly4_y2_ph1:7;
		u32 reserved3:1;
		u32 IPC_poly4_y2_ph0:7;
		u32 reserved4:1;
		}IPC_POLY4_Y2_LL;
	/* IPC_POLY4_Y2_LH */
	struct{
		u32 IPC_poly4_y2_ph7:7;
		u32 reserved1:1;
		u32 IPC_poly4_y2_ph6:7;
		u32 reserved2:1;
		u32 IPC_poly4_y2_ph5:7;
		u32 reserved3:1;
		u32 IPC_poly4_y2_ph4:7;
		u32 reserved4:1;
		}IPC_POLY4_Y2_LH;
	/* IPC_POLY4_Y2_HL */
	struct{
		u32 IPC_poly4_y2_ph11:7;
		u32 reserved1:1;
		u32 IPC_poly4_y2_ph10:7;
		u32 reserved2:1;
		u32 IPC_poly4_y2_ph9:7;
		u32 reserved3:1;
		u32 IPC_poly4_y2_ph8:7;
		u32 reserved4:1;
		}IPC_POLY4_Y2_HL;
	/* IPC_POLY4_Y2_HH */
	struct{
		u32 IPC_poly4_y2_ph15:7;
		u32 reserved1:1;
		u32 IPC_poly4_y2_ph14:7;
		u32 reserved2:1;
		u32 IPC_poly4_y2_ph13:7;
		u32 reserved3:1;
		u32 IPC_poly4_y2_ph12:7;
		u32 reserved4:1;
		}IPC_POLY4_Y2_HH;
	/* IPC_POLY4_Y3_LL */
	struct{
		u32 IPC_poly4_y3_ph3:6;
		u32 reserved1:2;
		u32 IPC_poly4_y3_ph2:6;
		u32 reserved2:2;
		u32 IPC_poly4_y3_ph1:6;
		u32 reserved3:2;
		u32 IPC_poly4_y3_ph0:6;
		u32 reserved4:2;
		}IPC_POLY4_Y3_LL;
	/* IPC_POLY4_Y3_LH */
	struct{
		u32 IPC_poly4_y3_ph7:6;
		u32 reserved1:2;
		u32 IPC_poly4_y3_ph6:6;
		u32 reserved2:2;
		u32 IPC_poly4_y3_ph5:6;
		u32 reserved3:2;
		u32 IPC_poly4_y3_ph4:6;
		u32 reserved4:2;
		}IPC_POLY4_Y3_LH;
	/* IPC_POLY4_Y3_HL */
	struct{
		u32 IPC_poly4_y3_ph11:6;
		u32 reserved1:2;
		u32 IPC_poly4_y3_ph10:6;
		u32 reserved2:2;
		u32 IPC_poly4_y3_ph9:6;
		u32 reserved3:2;
		u32 IPC_poly4_y3_ph8:6;
		u32 reserved4:2;
		}IPC_POLY4_Y3_HL;
	/* IPC_POLY4_Y3_HH */
	struct{
		u32 IPC_poly4_y3_ph15:6;
		u32 reserved1:2;
		u32 IPC_poly4_y3_ph14:6;
		u32 reserved2:2;
		u32 IPC_poly4_y3_ph13:6;
		u32 reserved3:2;
		u32 IPC_poly4_y3_ph12:6;
		u32 reserved4:2;
		}IPC_POLY4_Y3_HH;	
	/* IPC_POLY4_C0_LL */
	struct{
		u32 IPC_poly4_C0_ph3:7;
		u32 reserved1:1;
		u32 IPC_poly4_C0_ph2:7;
		u32 reserved2:1;
		u32 IPC_poly4_C0_ph1:7;
		u32 reserved3:1;
		u32 IPC_poly4_C0_ph0:7;
		u32 reserved4:1;
		}IPC_POLY4_C0_LL;	
	/* IPC_POLY4_C0_LH */
	struct{
		u32 IPC_poly4_C0_ph7:6;
		u32 reserved1:2;
		u32 IPC_poly4_C0_ph6:6;
		u32 reserved2:2;
		u32 IPC_poly4_C0_ph5:6;
		u32 reserved3:2;
		u32 IPC_poly4_C0_ph4:6;
		u32 reserved4:2;
		}IPC_POLY4_C0_LH;	
	/* IPC_POLY4_C0_HL */
	struct{
		u32 IPC_poly4_C0_ph11:6;
		u32 reserved1:2;
		u32 IPC_poly4_C0_ph10:6;
		u32 reserved2:2;
		u32 IPC_poly4_C0_ph9:6;
		u32 reserved3:2;
		u32 IPC_poly4_C0_ph8:6;
		u32 reserved4:2;
		}IPC_POLY4_C0_HL;	
	/* IPC_POLY4_C0_HH */
	struct{
		u32 IPC_poly4_C0_ph15:5;
		u32 reserved1:3;
		u32 IPC_poly4_C0_ph14:5;
		u32 reserved2:3;
		u32 IPC_poly4_C0_ph13:5;
		u32 reserved3:3;
		u32 IPC_poly4_C0_ph12:5;
		u32 reserved4:3;
		}IPC_POLY4_C0_HH;		
	/* IPC_POLY4_C1_LL */
	struct{
		u32 IPC_poly4_C1_ph3:8;
		u32 IPC_poly4_C1_ph2:8;
		u32 IPC_poly4_C1_ph1:8;
		u32 IPC_poly4_C1_ph0:8;
		}IPC_POLY4_C1_LL;	
	/* IPC_POLY4_C1_LH */
	struct{
		u32 IPC_poly4_C1_ph7:8;
		u32 IPC_poly4_C1_ph6:8;
		u32 IPC_poly4_C1_ph5:8;
		u32 IPC_poly4_C1_ph4:8;
		}IPC_POLY4_C1_LH;	
	/* IPC_POLY4_C1_HL */
	struct{
		u32 IPC_poly4_C1_ph11:8;
		u32 IPC_poly4_C1_ph10:8;
		u32 IPC_poly4_C1_ph9:8;
		u32 IPC_poly4_C1_ph8:8;
		}IPC_POLY4_C1_HL;	
	/* IPC_POLY4_C1_HH */
	struct{
		u32 IPC_poly4_C1_ph15:7;
		u32 reserved1:1;
		u32 IPC_poly4_C1_ph14:7;
		u32 reserved2:1;
		u32 IPC_poly4_C1_ph13:7;
		u32 reserved3:1;
		u32 IPC_poly4_C1_ph12:7;
		u32 reserved4:1;
		}IPC_POLY4_C1_HH;			
	/* PP_BYPASS */
	struct{
		u32 PP_BYPASS:1;
		u32 reserved1:20;
		}PP_BYPASS;
	/* PP_SATURATION */
	struct{
		u32 PP_SATURATION:8;
		u32 reserved1:24;
		}PP_SATURATION;
	/* PP_SHARPNESS */
	struct{
		u32 PP_SHARPNESS:2;
		u32 reserved1:6;
		u32 PP_TH_HNOISE:8;
		u32 reserved2:16;
		}PP_SHARPNESS;
	/* PP_LINE_EQx */
	struct{
		u32 LINE_SLOPE:8;
		u32 LINE_INTC:16;
		u32 reserved1:8;
		}PP_LINE_EQx;
	/* PP_BRIGHT_OFFSET */
	struct{
		u32 PP_BRIGHT_OFFSET:9;
		u32 reserved1:23;
		}PP_BRIGHT_OFFSET;
	/* IPC_VERSION_INFO */
	struct{
		u32 VERSION_INFO:32;
		}IPC_VERSION_INFO;
	
}IPC_REG;

#define IN_SC_MAX_HSZ	1024
#define IN_SC_MAX_VSZ	1024
//#define ON 1
//#define OFF 0

static oIPC_EnhancingVariable oIpc_EnhVar;
static oIPC_ControlVariable 		oIpc_ConVar;

oIPC_Source 		oIpc_SrcInf;
oIPC_Destination 	oIpc_DstInf;
oIPC_ControlVariable oIpc_ConVar;


REGISTER ipc_reg_info[] = 
{ 
	{"IPC_ENABLE",			IPC_ENABLE,			REG_WRITE|REG_READ,0xffffffff,0xffffffff,0x02,		0xffffffff,0},
	{"IPC_SRESET",			IPC_SRESET,			REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_SHADOW_UPDATE",	IPC_SHADOW_UPDATE,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_FIELD_ID",			IPC_FIELD_ID,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_MODE",				IPC_MODE,			REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_PEL_RATE_CTRL",	IPC_PEL_RATE_CTRL,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
//	{"IPC_ENDIAN_MODE",IPC_ENDIAN_MODE,REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,0xffffffff,0},	
	{"IPC_SRC_WIDTH",		IPC_SRC_WIDTH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_SRC_HEIGHT",		IPC_SRC_HEIGHT,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_DST_WIDTH",		IPC_DST_WIDTH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_DST_HEIGHT",		IPC_DST_HEIGHT,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_H_RATIO",			IPC_H_RATIO,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_V_RATIO",			IPC_V_RATIO,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y0_LL",		IPC_POLY8_Y0_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y0_LH",		IPC_POLY8_Y0_LH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y0_HL",		IPC_POLY8_Y0_HL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y0_HH",		IPC_POLY8_Y0_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y1_LL",		IPC_POLY8_Y1_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y1_LH",		IPC_POLY8_Y1_LH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y1_HL",		IPC_POLY8_Y1_HL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0			,0xffffffff,0},
	{"IPC_POLY8_Y1_HH",		IPC_POLY8_Y1_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y2_LL",		IPC_POLY8_Y2_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y2_LH",		IPC_POLY8_Y2_LH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y2_HL",		IPC_POLY8_Y2_HL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y2_HH",		IPC_POLY8_Y2_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y3_LL",		IPC_POLY8_Y3_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y3_LH",		IPC_POLY8_Y3_LH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y3_HL",		IPC_POLY8_Y3_HL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY8_Y3_HH",		IPC_POLY8_Y3_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y0_LL",		IPC_POLY4_Y0_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y0_LH",		IPC_POLY4_Y0_LH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y0_HL",		IPC_POLY4_Y0_HL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y0_HH",		IPC_POLY4_Y0_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y1_LL",		IPC_POLY4_Y1_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y1_LH",		IPC_POLY4_Y1_LH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y1_HL",		IPC_POLY4_Y1_HL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y1_HH",		IPC_POLY4_Y1_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y2_LL",		IPC_POLY4_Y2_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y2_LH",		IPC_POLY4_Y2_LH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y2_HL",		IPC_POLY4_Y2_HL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y2_HH",		IPC_POLY4_Y2_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y3_LL",		IPC_POLY4_Y3_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y3_LH",		IPC_POLY4_Y3_LH,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y3_HL",		IPC_POLY4_Y3_HL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_Y3_HH",		IPC_POLY4_Y3_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_C0_LL",		IPC_POLY4_C0_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_C0_LH",		IPC_POLY4_C0_LH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_C0_HL",		IPC_POLY4_C0_HL,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_C0_HH",		IPC_POLY4_C0_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_C1_LL",		IPC_POLY4_C1_LL,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_C1_LH",		IPC_POLY4_C1_LH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_C1_HL",		IPC_POLY4_C1_HL,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_POLY4_C1_HH",		IPC_POLY4_C1_HH,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_BYPASS",			IPC_BYPASS,			REG_WRITE|REG_READ,0xffffffff,0xffffffff,0x01,		0xffffffff,0},
	{"IPC_PP_SATURATION",	IPC_PP_SATURATION,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0x080,		0xffffffff,0},
	{"IPC_PP_SHARPNESS",	IPC_PP_SHARPNESS,	REG_WRITE|REG_READ,0xffffffff,0xffffffff,0x0500,	0xffffffff,0},
	{"IPC_PP_LINE_EQ0",		IPC_PP_LINE_EQ0,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_PP_LINE_EQ1",		IPC_PP_LINE_EQ1,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_PP_LINE_EQ2",		IPC_PP_LINE_EQ2,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_PP_LINE_EQ3",		IPC_PP_LINE_EQ3,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_PP_LINE_EQ4",		IPC_PP_LINE_EQ4,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_PP_LINE_EQ5",		IPC_PP_LINE_EQ5,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_PP_LINE_EQ6",		IPC_PP_LINE_EQ6,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_PP_LINE_EQ7",		IPC_PP_LINE_EQ7,		REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,			0xffffffff,0},
	{"IPC_PP_BRIGHT_OFFSET",	IPC_PP_BRIGHT_OFFSET,REG_WRITE|REG_READ,0xffffffff,0xffffffff,0,		0xffffffff,0},
	{"IPC_VERSION_INFO",		IPC_VERSION_INFO,	REG_READ,			   0xffffffff,0xffffffff,0xc1100000,0xffffffff,0},
	
	
};

u32 IPC_RegResetValueVerify(void)
{
	u32 i;

	IPC_REG reg;

	for(i=0;i<sizeof(ipc_reg_info)/sizeof(REGISTER);i++)
	{
		if(ipc_reg_info[i].type & REG_READ)
		{
			IpcInp32(ipc_reg_info[i].offset, reg.val);
			Disp("%s correct value=0x%x actual value=0x%x masked=0x%x\n",ipc_reg_info[i].name,ipc_reg_info[i].resetValue,reg.val,
				(reg.val) & (ipc_reg_info[i].resetValueMask));
			if((reg.val&ipc_reg_info[i].resetValueMask)!=ipc_reg_info[i].resetValue)
			{
				Disp("Wrong reset value at %s(0x%x). correct value=0x%x actual value=0x%x\n",
					ipc_reg_info[i].name,ipc_reg_info[i].offset,ipc_reg_info[i].resetValue,reg.val);
				return i;
			}
		}
	}
	return 0xffffffff;
}


void Test_initReg()
{
	if(IPC_RegResetValueVerify()!=0xffffffff)
	{
		Disp("Invalid Reset Reg!!\n");
	}
	else{
		Disp("\n SFR Reset Value Test OK!!\n");
		}
}


//////////
// Function Name 	: 	IPC_InitIP
// Description		:	Initialize the IPC
// Input 			:	oIpc_Src - refer to oIPC_Source
//					oIpc_Dst - refer to oIPC_Destination
void IPC_Init(oIPC_Source oIpc_Src, oIPC_Destination oIpc_Dst, oIPC_ControlVariable oIpc_ConVar)
{
	u32 uYSize, uCSize;	
	
	Assert(oIpc_Src.uImgHsz>=32 && oIpc_Src.uImgHsz<=1024);// && (oIpc_Src.uImgHsz%8) == 0); //uImgHsz[2:0] should be 3'b000 for 64bit interface
	Assert(oIpc_Src.uImgVsz>=4);
		 
	IPC_SWReset();
	IPC_EnableIP(OFF);

	IPC_SetPostProcessingOnOff(OFF);
	//IPC_SetEndianMode(ENDIAN_LITTLE);
		
	//Initialize the Image Information
	//IPC_CalcImageSize(oVp_Src, &uYSize, &uCSize);
	IPC_SetModeAndImgSize(oIpc_Src, oIpc_Dst, oIpc_ConVar);

	//Initialize Enhancement 
	IPC_InitEnhancingParameter();
	IPC_SetContrast(oIpc_EnhVar.aContrast); 
	IPC_SetBrightness(oIpc_EnhVar.aBrightness);
	IPC_SetBrightOffset(oIpc_EnhVar.uBrightOffset);
	IPC_SetSaturation(oIpc_EnhVar.uSaturation);
	IPC_SetSharpness(oIpc_EnhVar.eSharpness, oIpc_EnhVar.uThHnoise);

	//Initialize the Poly-Phase Filter Coefficients
	IPC_SetFilter();

	IpcOutp32(IPC_PEL_RATE_CTRL, 0x0<<0); // 1pixel/1cycle
	IpcOutp32(IPC_SHADOW_UPDATE, 0x1<<0); // Update shadow reg.s
	
}
/*
void IPC_Init(FIMC_DST_st *dstform)
{
	oIpc_SrcInf.eSrcBpp = dstform->DstDataFmt;
	oIpc_SrcInf.uImgHsz = dstform->dst_width;
	oIpc_SrcInf.uImgVsz = dstform->dst_height;
	oIpc_SrcInf.uSrcHsz = dstform->dst_width;
	oIpc_SrcInf.uSrcVsz = dstform->dst_height;

	oIpc_DstInf.eScanMode = PROGRESSIVE;
	oIpc_DstInf.uDstHsz = dstform->dst_width;
	oIpc_DstInf.uDstVsz = dstform->dst_height;
	
	IPC_InitIP(oIpc_SrcInf, oIpc_DstInf);
}
*/
void IPC_InitIp(u32 InHSz, u32 InVSz, _2D_eIPC Ipc2d)
{
	oIpc_SrcInf.uImgHsz = InHSz;
	oIpc_SrcInf.uImgVsz = InVSz;
	oIpc_SrcInf.uSrcHsz = InHSz;
	oIpc_SrcInf.uSrcVsz = InVSz;

	oIpc_DstInf.eScanMode = PROGRESSIVE;
			
	if(Ipc2d == IPC_2D){
		oIpc_DstInf.uDstHsz = InHSz;
		oIpc_DstInf.uDstVsz = InVSz*2;		
		Disp("\n IPC OUT IMAGE SIZE : %d X %d",InHSz,InVSz*2);
		}
	else{
		oIpc_DstInf.uDstHsz = InHSz*2;
		oIpc_DstInf.uDstVsz = InVSz;		
		Disp("\n IPC OUT IMAGE SIZE : %d X %d",InHSz*2,InVSz);
		}
	oIpc_ConVar.uModeVal = Ipc2d;
	
	
	IPC_Init(oIpc_SrcInf, oIpc_DstInf, oIpc_ConVar);
}

void IPC_InitIp_Exp(u32 InHSz, u32 InVSz, u32 OutHSz, u32 OutVSz, _2D_eIPC Ipc2d)
{
	oIpc_SrcInf.uImgHsz = InHSz;
	oIpc_SrcInf.uImgVsz = InVSz;
	oIpc_SrcInf.uSrcHsz = InHSz;
	oIpc_SrcInf.uSrcVsz = InVSz;

	oIpc_DstInf.eScanMode = PROGRESSIVE;			

		oIpc_DstInf.uDstHsz = OutHSz;
		oIpc_DstInf.uDstVsz = OutVSz;		
		Disp("\n IPC OUT IMAGE SIZE : %d X %d",OutHSz,OutVSz);

	oIpc_ConVar.uModeVal = Ipc2d;
	
	
	IPC_Init(oIpc_SrcInf, oIpc_DstInf, oIpc_ConVar);
}


void IPC_ONOFF(IPC_eONOFF eONOFF)
{
	IPC_SetPostProcessingOnOff(eONOFF);
	IPC_EnableIP(eONOFF);

}

void IPC_On(void)
{
	IPC_SetPostProcessingOnOff(ON);
	IPC_EnableIP(ON);

}

void IPC_Off(void)
{
	IPC_SetPostProcessingOnOff(OFF);
	IPC_EnableIP(OFF);

}


//////////
// Function Name 	: 	IPC_SWReset
// Description		:	IPC Software Reset Command
// Input 			:	None
void IPC_SWReset(void)
{
	IPC_REG reg;
	reg.val=0;

	do 
	{
		IpcInp32(IPC_SRESET, reg.val);
	} while ( (reg.IPC_SRESET.IPC_SRESET&(0x1<<0)) != (0x0<<0) ); // Wait for this SFR[0] to zero
	reg.IPC_SRESET.IPC_SRESET = 1;	
	IpcOutp32(IPC_SRESET, reg.val);			//Reset start !!
}


//////////
// Function Name 	: 	IPC_EnableIP
// Description		:	IPC On
// Input 			:	uOnOff - 1(On), 0(Off)
void IPC_EnableIP(u32 uOnOff)
{
	IPC_REG reg;
	reg.val=0;

	Assert(uOnOff == 0 || uOnOff == 1);

	IpcInp32(IPC_ENABLE, reg.val);
	reg.IPC_ENABLE.IPC_ON = uOnOff;
	IpcOutp32(IPC_ENABLE, reg.val);
}

//////////
// Function Name 	: 	IPC_IsProcessingDone
// Description		:	Check whether the IPC is processing.
// Input 			:	None
// Output		: 	true - Processing
//					false - Processing Done
u8 IPC_IsProcessingDone(void)
{
	u32 uEnableReg;
	
	IpcInp32(IPC_ENABLE, uEnableReg);

	return ( uEnableReg&(0x1<<1) == (0x1<<1) ) ? true : false;
}

//////////
// Function Name 	: 	IPC_Set2DIpcOnOff
// Description		:	2D IPC On/Off Selection
// Input 			:	uOnOff - 1(On), 0(Off)
void IPC_Set2DIpcOnOff(u32 uOnOff)
{
	u32 uModeVal;
	
	Assert(uOnOff==1 || uOnOff==0);

	
	IpcInp32(IPC_MODE, uModeVal);
	uModeVal &= ~(0x1<<1);
	uModeVal |= (uOnOff<<1);
	IpcOutp32(IPC_MODE, uModeVal);

	IpcOutp32(IPC_SHADOW_UPDATE, 0x1<<0); // Update shadow reg.s				
}

//////////
// Function Name 	: 	IPC_SetPostProcessingOnOff
// Description		:	IPC PostProcessing On/Off Selection
// Input 			:	uOnOff - 1(On), 0(Off)
void IPC_SetPostProcessingOnOff(u32 uOnOff)
{
	IPC_REG reg;
	reg.val=0;
	
	Assert(uOnOff == 1 || uOnOff == 0);

	IpcInp32(IPC_BYPASS, reg.val);
	uOnOff==1 ? (reg.PP_BYPASS.PP_BYPASS = 0) :( reg.PP_BYPASS.PP_BYPASS = 1);	
	IpcOutp32(IPC_BYPASS, reg.val);

	IpcOutp32(IPC_SHADOW_UPDATE, 0x1);
	
}

//////////
// Function Name 	: 	IPC_FILED_ID_Set
// Description		:	IPC Field id top/bottom Selection
// Input 			:	uFieldIdReg - 0(top), 1(bottom)
void IPC_FILED_ID_Control(IPC_eFIELD_ID id )
{
	IpcOutp32(IPC_FIELD_ID, id);
	//  Update shadow register
	IpcOutp32(IPC_SHADOW_UPDATE, 0x1<<0); 
}

//////////
// Function Name 	: 	IPC_FILED_ID_Mode
// Description		:	IPC Field id signal source Selection, auto toggling define
// Input 			:	sel - 0(Internal generation), 1(camif field signal accept) 
//				:	toggle - 0(define by user), 1(auto toggling)
void IPC_FILED_ID_Mode(IPC_eFIELD_ID_SEL sel,IPC_eFIELD_ID_TOGL toggle )
{
	IPC_REG reg;
	reg.val=0;
	
	IpcInp32(IPC_MODE, reg.val);
	reg.IPC_MODE.FIELD_ID_SELECTION = sel;	
	IpcOutp32(IPC_MODE, reg.val);
	
	IpcInp32(IPC_MODE, reg.val);
	reg.IPC_MODE.FIELD_ID_AUTO_TOGGLING = toggle;
	IpcOutp32(IPC_MODE, reg.val);
	
	//  Update shadow register
	IpcOutp32(IPC_SHADOW_UPDATE, 0x1<<0); 
}

//////////
// Function Name 	: 	IPC_2D_ENABLE
// Description		:	IPC 2D enable/disable Selection
// Input 			:	uOnOff - 0(disable), 1(enable)
void IPC_2D_ENABLE(IPC_eON uOnOff)
{
	IPC_REG reg;
	reg.val = 0;
	
	IpcInp32(IPC_MODE, reg.val);
	reg.IPC_MODE.IPC_2D  = uOnOff;	
	IpcOutp32(IPC_MODE, reg.val);

	//  Update shadow register
	IpcOutp32(IPC_SHADOW_UPDATE, 0x1<<0); 
	
}

//////////
// Function Name 	: 	IPC_SetModeAndImgSize
// Description		:	IPC Mode Setting & Source Image Parameter Setting & Destination Image Parameter Setting.
// Input 			:	oIPC_Src - refer to oVP_Source
//					oIPC_Dst - refer to oVP_Destination

void IPC_SetModeAndImgSize(oIPC_Source oIpc_Src, oIPC_Destination oIpc_Dst, oIPC_ControlVariable oIpc_ConVar)
{
	u32 i;
	u32 uFieldIdReg, uWidthOffset, uVRatio, uHRatio;
	IPC_REG reg;
	reg.val = 0;
	
#if 0
	oIpc_ConVar.uModeVal = 
			(0<<6)		//FIELD ID SELECTION : Internal generation	
			|(1<<2)		//FIELD ID AUTO TOGGLING 
			|(1<<1);		// 2D_IPC enable

#endif

	// Mode Setting
#if 0
	uFieldIdReg = 0x0;
	IpcOutp32(IPC_FIELD_ID, uFieldIdReg);
	IpcOutp32(IPC_MODE, oIpc_ConVar.uModeVal);
#else
		
	IPC_FILED_ID_Control(IPC_BOTTOM_FIELD);
	IPC_FILED_ID_Mode(INTERNAL, BYUSER);
	//IPC_FILED_ID_Mode(INTERNAL, AUTO);
	
	if(oIpc_ConVar.uModeVal == IPC_2D)
		IPC_2D_ENABLE(oIpc_ConVar.uModeVal);	// Enalbed : 2D IPC , Disabled : Horizon Double Scailing
		//IPC_2D_ENABLE(ENABLED);	// Enalbed : 2D IPC , Disabled : Horizon Double Scailing
#endif
	//IPC_2D_ENABLE(ENABLED);	// Enalbed : 2D IPC , Disabled : Horizon Double Scailing
	
	IpcOutp32(IPC_SRC_WIDTH, (oIpc_Src.uSrcHsz&0x7ff)<<0);
	IpcOutp32(IPC_SRC_HEIGHT, (oIpc_Src.uSrcVsz&0x7ff)<<0);
	
	// Set the Destination Image
	IpcOutp32(IPC_DST_WIDTH, (oIpc_Dst.uDstHsz&0x7ff)<<0);
	IpcOutp32(IPC_DST_HEIGHT, (oIpc_Dst.uDstVsz&0x7ff)<<0);
	//IpcOutp32(IPC_DST_HEIGHT, ((oIpc_Dst.uDstVsz+1)&0x7ff)<<0);

	// Set H/V ratio

	//uHRatio = ( oIpc_ConVar.uModeVal== 1 ) ? (0x10000) : (0x08000);
	//uVRatio = 0x10000;

	if(oIpc_ConVar.uModeVal== 0)
	{
		uHRatio = 0x08000;
		uVRatio = 0x10000;
		
	}
	else if(oIpc_ConVar.uModeVal== 1)
	{
		uHRatio = 0x10000;
	uVRatio = 0x10000;
	}
	else{
			uHRatio =  (oIpc_Src.uSrcHsz<<16)/oIpc_Dst.uDstHsz;
			//uVRatio = ( (oIpc_ConVar.uModeVal&(0x1<<1)) == (0x1<<1) ) ? ((oIpc_Src.uSrcVsz<<17)/oIpc_Dst.uDstVsz) : ((oIpc_Src.uSrcVsz<<16)/oIpc_Dst.uDstVsz);
			uVRatio = (oIpc_Src.uSrcVsz<<16)/oIpc_Dst.uDstVsz;
			//uVRatio = (oIpc_Src.uSrcVsz<<17)/oIpc_Dst.uDstVsz;
			
			UART_Printf("\n uHRatio : 0x%x",uHRatio);
			UART_Printf("\n uVRatio : 0x%x",uVRatio);
	}
	// Set H/V ratio	
	
	IpcOutp32(IPC_H_RATIO, uHRatio);	
	IpcOutp32(IPC_V_RATIO, uVRatio);

	//  Update shadow register
	IpcOutp32(IPC_SHADOW_UPDATE, 0x1<<0); 
}

//////////
// Function Name 	: 	IPC_InitEnhancingParameter
// Description		:	Initialize the IPC Enhancing Parameter Variable(refer to oVP_EnhancingVariable Structure)
// Input 			:	None
void IPC_InitEnhancingParameter(void)
{
	u32 i;
	
	for(i=0 ; i<8 ; i++)
	{
		oIpc_EnhVar.aBrightness[i] = 0x0;
		oIpc_EnhVar.aContrast[i] = 0x80;
	}

	oIpc_EnhVar.uSaturation = 0x80;
	oIpc_EnhVar.eSharpness = NO_EFFECT;
	oIpc_EnhVar.uThHnoise = 0x5;
	oIpc_EnhVar.uBrightOffset = 0x0;
}

//////////
// Function Name 	: 	IPC_SetContrast
// Description		:	IPC Contrast Setting
// Input 			:	uContrast - LINE_SLOPE Value(unsigned 1.7 format)
void IPC_SetContrast(u32 *uContrast)
{
	u32 i, uLineEq[8];

	for(i=0 ; i<8 ; i++)
	{
		IpcInp32(IPC_PP_LINE_EQ0 + 4*i, uLineEq[i]);
		uLineEq[i] &= ~(0xFF<<0);
		uLineEq[i] |= ((uContrast[i]&0xFF)<<0);

		IpcOutp32(IPC_PP_LINE_EQ0 + 4*i, uLineEq[i]);
	}

	IpcOutp32(IPC_SHADOW_UPDATE, 0x1);	
}


//////////
// Function Name 	: 	IPC_SetBrightness
// Description		:	IPC Brightness Setting
// Input 			:	uBrightness - LINE_INTC Value(unsigned 9.7 format)
void IPC_SetBrightness(u32 *uBrightness)
{
	u32 i, uLineEq[8];

	for(i=0 ; i<8 ; i++)
	{
		IpcInp32(IPC_PP_LINE_EQ0 + 4*i, uLineEq[i]);
		uLineEq[i] &= ~(0xFFFF<<8);
		uLineEq[i] |= ((uBrightness[i]&0xFFFF)<<8);		//intercept, signed 9.7 format

		IpcOutp32(IPC_PP_LINE_EQ0 + 4*i, uLineEq[i]);
	}

	IpcOutp32(IPC_SHADOW_UPDATE, 0x1);				
}

//////////
// Function Name 	: 	IPC_SetBrightOffset
// Description		:	Set the Brightness Offset control register for Y
// Input 			:	uOffset - Offset for Y Brightness Value(signed 1.8 format)
void IPC_SetBrightOffset(u32 uOffset)
{
	Assert(uOffset <= 0x1FF);

	IpcOutp32(IPC_PP_BRIGHT_OFFSET, (uOffset&0x1FF)<<0);
	
	IpcOutp32(IPC_SHADOW_UPDATE, 0x1); 		
}

//////////
// Function Name 	: 	IPC_SetSaturation
// Description		:	Set the Color Saturation control register
// Input 			:	uSaturation - Color Saturation Factor Value(signed 1.7 format)
void IPC_SetSaturation(u32 uSaturation)
{
	Assert(uSaturation <= 0xFF);

	IpcOutp32(IPC_PP_SATURATION, (uSaturation&0xff)<<0);
	IpcOutp32(IPC_SHADOW_UPDATE, 0x1);	
}


//////////
// Function Name 	: 	IPC_SetSharpness
// Description		:	Set the Picture sharpness control register
// Input 			:	eSharpness - Control for the edge enhancement Value
//					uThreshold - Treshold value setting to decide minimum vertical edge value
void IPC_SetSharpness(IPC_eSHARPNESS eSharpness, u32 uThreshold)
{
	u32 uSharpVal;
	
	Assert(uThreshold <= 0xff);

	uSharpVal = 	(eSharpness == NO_EFFECT) ? (0x0<<0) :
				(eSharpness == MIN_EDGE) ? (0x1<<0) :
				(eSharpness == MODERATE_EDGE) ? (0x2<<0) : 
				(0x3<<0); // MAX_EDGE

	IpcOutp32(IPC_PP_SHARPNESS, uSharpVal | ((uThreshold&0xff)<<8));
	IpcOutp32(IPC_SHADOW_UPDATE, 0x1);		
}


//////////
// Function Name 	: 	IPC_SetFilter
// Description		:	Set the Poly-Phase Filter Coefficients
// Input 			:	None
void IPC_SetFilter(void)
{
	u32 uHRatio, uVRatio;
	IPC_eFILTER_H_PP eHFilter;
	IPC_eFILTER_V_PP eVFilter;

	IpcInp32(IPC_H_RATIO, uHRatio );
	IpcInp32(IPC_V_RATIO, uVRatio );

	// 	For the real interlace mode, the vertical ratio should be used after divided by 2.
	//	Because in the interlace mode, all the IPC output is used for FIMD display
	//	and it should be the same as one field of the progressive mode. 
	//	Therefore the same filter coefficients should be used for the same the final output video.
	//	When half of the interlace V_RATIO is same as the progressive V_RATIO,
	//	the final output video scale is same. (20051104,ishan)

	//Horizontal Y 8tap 
	//Horizontal C 4tap	
	if (uHRatio <= (0x1<<16))			// 720->720 or zoom in
		eHFilter = IPC_PP_H_NORMAL;
	else if (uHRatio <= (0x9<<13))		// 720->640
		eHFilter = IPC_PP_H_8_9 ;
	else if(uHRatio <= (0x1<<17))		// 2->1
		eHFilter = IPC_PP_H_1_2;
	else if(uHRatio <= (0x3<<16))		// 2->1	
		eHFilter = IPC_PP_H_1_3;
	else // 0x800						// 4->1
		eHFilter = IPC_PP_H_1_4;

	// Vertical Y 4tap
	if (uVRatio <= (0x1<<16))			// 720->720 or zoom in
		eVFilter = IPC_PP_V_NORMAL;
	else if (uVRatio <= (0x3<<15))		//*6->5
		eVFilter = IPC_PP_V_5_6;
	else if(uVRatio <= (0x5<<14))		// 4->3
		eVFilter = IPC_PP_V_3_4;
	else if (uVRatio <= (0x1<<17))		// 2->1
		eVFilter = IPC_PP_V_1_2;
	else if (uVRatio <= (0x3<<16))		// 3->1
		eVFilter = IPC_PP_V_1_3;
	else // 0x400						// 4->1
		eVFilter = IPC_PP_V_1_4;

	IPC_SetPolyPhaseFilterSet( eHFilter, eVFilter );

}


//////////
// Function Name 	: 	IPC_SetPolyPhaseFilterSet
// Description		:	Set the Poly-Phase Filter Coefficients
// Input 			:	eHFilter - refer to IPC_FILTER_H_PP enum
//					eVFilter - refer to IPC_FILTER_V_PP enum
void IPC_SetPolyPhaseFilterSet(IPC_eFILTER_H_PP eHFilter, IPC_eFILTER_V_PP eVFilter)
{
	Disp("\nHFilter : %d, VFilter : %d",eHFilter, eVFilter);
	// Horizontal Y 8-tap
	IPC_SetPolyPhaseFilter(IPC_POLY8_Y0_LL, sIpc8tapCoef_Y_H+eHFilter*16*8, 8 );
	// Horizontal C 4-tap
	IPC_SetPolyPhaseFilter(IPC_POLY4_C0_LL, sIpc4tapCoef_C_H+eHFilter*16*4, 4 );
	// Vertical Y 4-tap
	IPC_SetPolyPhaseFilter(IPC_POLY4_Y0_LL, sIpc4tapCoef_Y_V+eVFilter*16*4, 4 );
}


//////////
// Function Name 	: 	IPC_SetPolyPhaseFilter
// Description		:	Set the Poly-Phase Filter Coefficients
// Input 			:	uFilterReg - the Base Address of Poly-Phase Filter Coefficient registers.
//					sFilterCoef - the Base Address of Coefficient value array
//					usTapSz - Tap Size
void IPC_SetPolyPhaseFilter(volatile u32 uFilterReg, const s8* sFilterCoef, u16 usTapSz)
{
	u32 i,j;
	u8* ucpFilterCoef;
	u16 usTempTapSz;
	volatile u32 uFilterBaseAddr;
	
	uFilterBaseAddr = uFilterReg;
	ucpFilterCoef = (u8*)sFilterCoef;

	for ( i=0; i<usTapSz; i++ )
	{
		// VP_POLYx_Y0/1(/2/3)_xx Setting
		usTempTapSz= usTapSz-i-1;
		
		for ( j=0; j<4; j++ )
		{
			// VP_POLYx_Yx_LL/LH/HL/HH Setting
			IpcOutp32(uFilterBaseAddr, ( (ucpFilterCoef[4*j*usTapSz + usTempTapSz] <<24)
				| (ucpFilterCoef[(4*j+1)*usTapSz + usTempTapSz] <<16)
				| (ucpFilterCoef[(4*j+2)*usTapSz + usTempTapSz] <<8) 
				| (ucpFilterCoef[(4*j+3)*usTapSz + usTempTapSz]) ) );
			uFilterBaseAddr += 4;
		}
	}
}


void IPC_SaturationTest(void)
{
	u32 saturation = 0;
//	IPC_SetPostProcessingOnOff(ON);

	Disp("IPC_SaturationTest........................... ");
	for(saturation = 0; saturation<256; saturation +=8)
		{
			IPC_SetSaturation(saturation);
			UART_Getc();
		}

	/* recover reset value */
	IPC_SetSaturation(0x80);
	
}

void IPC_SharpnessTH_HNOISE_Test(void)
{
	u32 sharp = 0, Th_value=0;
//	IPC_SetPostProcessingOnOff(ON);

	Disp("IPC_SharpnessTH_HNOISE_Test........................... ");
	for(sharp = 0; sharp < 4; sharp++)
		{
		for(Th_value = 0 ; Th_value < 256 ; Th_value += 8)
			{
			//format 20, 0~3, reset 0x0
			IPC_SetSharpness(sharp,Th_value);	//to be modified....
			UART_Getc();
		}
	}
	/* recover reset value */
	IPC_SetSharpness(0,0x05);	//to be modified....
}

void IPC_SharpnessTest(void)
{
	u32 sharp = 0;
//	IPC_SetPostProcessingOnOff(ON);

	Disp("IPC_SharpnessTest........................... ");
	for(sharp = 0; sharp < 4; sharp++)
		{
			//format 20, 0~3, reset 0x0
			IPC_SetSharpness(sharp,0x05);	//to be modified....
			UART_Getc();
		}

	/* recover reset value */
	IPC_SetSharpness(0,0x05);	//to be modified....
}

void IPC_TH_HNOISE_Test(void)
{
	u32 Th_value = 0;
//	IPC_SetPostProcessingOnOff(ON);

	Disp("IPC_TH_HNOISE_Test........................... ");
	for(Th_value = 0 ; Th_value < 256 ; Th_value += 8)
		{
			//reset 0x5
			//format 20, 0~3, reset 0x0
			IPC_SetSharpness(0x0,Th_value);	//to be modified..
			UART_Getc();	
		}
	
	/* recover reset value */
	IPC_SetSharpness(0,0x05);	//to be modified....
}

void IPC_ContrastTest(void)
{
	u32 cont = 0, i= 0, contbf[8];
//	IPC_SetPostProcessingOnOff(ON);

	Disp("IPC_ContrastTest........................... ");
	for(cont =0 ; cont < 256 ; cont += 8)
		{	
			for(i = 0 ; i <8 ; i ++)
				contbf[i]=cont;
			
			//format 1.7, 0~2.0, reset 0x80
			IPC_SetContrast(contbf);
			UART_Getc();			
		}
	/* recover reset value */
	for(i = 0 ; i <8 ; i ++)
		contbf[i]=0x80;
				
	IPC_SetContrast(contbf);
	
}

void IPC_Brightness_Test(void)
{
	int bright = 0, i= 0;
	int brightbf[8];
//	IPC_SetPostProcessingOnOff(ON);

	Disp("IPC_Brightness_Test........................... ");
	for(bright = -256 ; bright < 256 ; bright += 8)
		{
			for(i = 0 ; i <8 ; i ++)
				brightbf[i]=bright<<7;
			//format 9.7, -256~255, reset 0x0

			IPC_SetBrightness(brightbf);
			UART_Getc();			
		}
	/* recover reset value */
	for(i = 0 ; i <8 ; i ++)
			brightbf[i]=0x0;
	
	IPC_SetBrightness(brightbf);
}


void IPC_BRIGHTOffsetTest(void)
{
	u32 offset = 0;
	u32 bright_offset;
//	IPC_SetPostProcessingOnOff(ON);

	Disp("Increasing Bright offset........................... ");
	for(offset = 100 ; offset <= 0x1FF; offset += 8)
		{
			//format 9, -256~-1, reset 0x0
			bright_offset = offset;

			IPC_SetBrightOffset(bright_offset);
			UART_Getc();
		}
	for(offset = 0 ; offset <= 0xFF; offset += 8)
		{
			//format 9, 0~255, reset 0x0
			bright_offset = offset;

			IPC_SetBrightOffset(bright_offset);
			UART_Getc();
		}
	Disp("Done");

	IPC_SetBrightOffset(0x0);

}

void IPC_IPC_VERSION_INFO(void)
{
	u32 ver = 0;
		IpcInp32(IPC_VERSION_INFO, ver );
		Disp("IPC Version : %ul\n",ver);
}