
#include <stdio.h>
#include <string.h>

#include "def.h"
#include "option.h"
#include "library.h"
#include "sysc.h"
#include "system.h"

#include "lcd.h"
#include "glib.h"
#include "intc.h"
#include "util.h"
#include "gpio.h"
#include "sysc.h"
#include "sblock.h"
#include "timer.h"
#include "iic.h"
#include "jpeg.h"

#include "fimc.h"
#include "fimc_sfr.h"
//#include "fimc_golden.h"

#include "ipc.h"
#include "csi.h"

static CAMERA eCamera;
static FIMC_CH eCh;
static CIS_MODEL eCisModel;
static CSC_EQUATION eCscEquation;
static YCBCR_RANGE eYCbCrRange;
static LOCAL_IF_TYPE eLocalInCS;

static u32 uLcdHsz, uLcdVsz;

static IMG_FMT eBpp;
static u32 uLcdFbAddr;
static u32 uInFbAddr;
static u32 uOutFbAddr;
static u32 uLcdOutFbAddr;

// =====
static TEST_PATTERN eTestPattern; // Set test pattern

static IMG_FMT eBppInFmt; // Set input image format

static u32 uInImgHsz, uInImgVsz; // Set input image size

// Set cropping region
static u32 uInImgCropStartX, uInImgCropStartY;
static u32 uInImgCropHsz, uInImgCropVsz;

static u32 uOutImgHsz, uOutImgVsz; // Set scaler

static ROT_FLIP eRotFlip; // Set rotator & flip

static IMG_FMT eBppOutFmt; // Set output image format

static u32 uOutDispHsz, uOutDispVsz; // Set output display size

static u32 uOutImgStartX, uOutImgStartY; // Set output offset

static SCAN_MODE eOutScanMode; // Set out scan mode

static DMA_OUT_MODE eDmaOutMode; // Set FIFO-in DMA-out mode

// Set image effect
static IMG_EFF eImgEff;
static u32 uCbVal, uCrVal;

// Set capture frame control
static u8 bCptFrmCtrl;
static u32 uCptSeqPattern, uCptSeqPointer;
// Set capture count control
static u8 bCptCntCtrl;
static u32 uCptCnt;
// =====

static IP_eID eIpId;

static volatile u8 bTimer0Done;
static void __irq Isr_Timer0(void)
{
	u32 uTmp;	

	INTC_Disable(NUM_TIMER0);
	
	uTmp = Inp32(rTINT_CSTAT);
	uTmp = (uTmp & 0x1f)| (1<<5);
	Outp32(rTINT_CSTAT,uTmp);       //  Timer0 Interrupt Clear register

	bTimer0Done = true;

	INTC_Enable(NUM_TIMER0);
	INTC_ClearVectAddr();
}

static volatile u8 bJpegDone;
static void __irq Isr_Jpeg(void)
{
	u32 uStatus;
	JPEG_IRQSTATUS eJpgIrq;

	INTC_Disable(NUM_JPEG);

	JPEG_ReadStatus(&eJpgIrq);
	switch (eJpgIrq)
	{
		case 0x40 : UART_Printf("JPGIRQ : 0x%x   Enc or Dec is OK\n",eJpgIrq); break;	//ERR_HD_PARSING2
		case 0x20 : UART_Printf("JPGIRQ : 0x%x   Error! Syntax or Abnormally\n",eJpgIrq); break;		//ERR_ENC_OR_DEC
	}
	
	bJpegDone = true;
	//UART_Printf("JpegDone: %d\n", bJpegDone); 

	JPEG_Wait_Done();
	JPEG_IntClear();

	INTC_Enable(NUM_JPEG);
	INTC_ClearVectAddr();	
}

#define WaitForJpegDone() \
{ \
	while(!bJpegDone); \
}

static void __irq Isr_Fimd(void)
{
	u32 uIntFlag;

	LCDC_GetIntFlag(&uIntFlag);	
	Disp("[u]");
	LCDC_ClearPending(uIntFlag);	
	INTC_ClearVectAddr();	
}

static u32 uLoopCntForBackPorch;
static u8 volatile bFimdDoneForBackPorch;
static void __irq Isr_FimdForBackPorch(void)
{
	u32 uIntFlag;
	u8 bEnable;
	u32 i;

	Disp("[FRAME]");
	INTC_Disable(NUM_LCD1);
	LCDC_DisableInterrupt();

	LCDC_GetIntFlag(&uIntFlag);
	LCDC_ClearPending(uIntFlag);

	INTC_ClearVectAddr();
	
	FIMC_GetLcdcEnstatus(eCh, &bEnable);
	if(bEnable == false)
	{
		Delay(uLoopCntForBackPorch*1000+1000);
		
		FIMC_SetDinEnvid(eCh, false); // ENVID forced off	
		
		Delay(uLoopCntForBackPorch*1000+1000);	
		
		FIMC_StopProcessing(eCh);		

		bFimdDoneForBackPorch = true;
	}
	else if(bEnable == true)
	{
		INTC_Enable(NUM_LCD1);
		LCDC_EnableInterrupt();

		bFimdDoneForBackPorch = false;
	}
	else
	{
		Assert(0);
	}
}

static void __irq Isr_Csis(void)
{
	u32 uCsisInt;
	u32 uIntStatus;
	u32 uIntMask;

	CSI_ChkIntStatus(&uIntStatus);
	CSI_GetIntMask(&uIntMask);

	uCsisInt = uIntMask&uIntStatus;	

	if(uCsisInt & (0x1<<0))
		Disp("Unknown ID error\n");

	if(uCsisInt & (0x1<<1)) 
		Disp("CRC error\n");

	if(uCsisInt & (0x1<<2))
		Disp("ECC error\n");

	if(uCsisInt & (0x1<<3))
		Disp("Overflow error\n");

	// Datalane 0~3 SOT error
	if(uCsisInt & (0xF<<12))
		Disp("Start of transmission error\n");

	if(uCsisInt & (0x1<<28))
		Disp("Embedded Data : OddAfter\n");

	if(uCsisInt & (0x1<<29))
		Disp("Embedded Data : OddBefore\n");

	if(uCsisInt & (0x1<<30))
		Disp("Embedded Data : EvenAfter\n");

	if(uCsisInt & (0x1u<<31))
		Disp("Embedded Data : EvenBefore\n");	

	CSI_ClearInterrupt(uCsisInt);
	INTC_ClearVectAddr();
}

static u8 volatile bFimcDone;
static void __irq Isr_Fimc(void)
{
	FIMC_INTR_SRC eSrc;
	FIMC_GetIntrSrc(eCh, &eSrc);
	Disp("[i]");
	switch(eSrc)
	{
		case FIMC_SUCCESS:
			bFimcDone = true;
			break;
		case FIFO_Y_OVF:
			bFimcDone = false;
			Disp("\n FIFO Y overflow \n");
			break;
		case FIFO_CB_OVF:
			bFimcDone = false;
			Disp("\n FIFO Cb overflow \n");
			break;
		case FIFO_CR_OVF:
			bFimcDone = false;
			Disp("\n FIFO Cr overflow \n");
			break;
		case ROT_LINE_BUF_OVF:
			bFimcDone = false;
			Disp("\n Rotation line buffer overflow \n");
			break;
		default:
			bFimcDone = false;
			Assert(0);
			break;		
	}

	FIMC_ClearInterrupt(eCh);
	INTC_ClearVectAddr();
}

#ifndef FIMC_POLLING
#define WaitForFimcDone(eCh) \
{ \
	while(!bFimcDone); \
}
#else // polling
#define WaitForFimcDone(eCh) \
{ \
	FIMC_WaitForProcessingDone(eCh); \
}
#endif

// [Initialize variables]
static void InitVariables(void)
{
	eTestPattern=EXTERNAL_CAMERA_INPUT; // Set test pattern

	eBppInFmt=RGB24; // Set input image format

	uInImgHsz=uLcdHsz, uInImgVsz=uLcdVsz; // Set input image size

	// Set cropping region
	uInImgCropStartX=0, uInImgCropStartY=0;
	uInImgCropHsz=uLcdHsz, uInImgCropVsz=uLcdVsz;

	uOutImgHsz=uLcdHsz, uOutImgVsz=uLcdVsz; // Set scaler

	eRotFlip=ROT_NO_FLIP_NO; // Set rotator & flip

	eBppOutFmt=RGB24; // Set output image format

	uOutDispHsz=uLcdHsz, uOutDispVsz=uLcdVsz; // Set output display size

	uOutImgStartX=0, uOutImgStartY=0; // Set output offset

	eOutScanMode = PROGRESSIVE; // Set out scan mode

	eDmaOutMode = PROGRESSIVE_OUT; // Set FIFO-in DMA-out mode

	// Set image effect
	eImgEff=EFF_BYPASS;
	uCbVal=0, uCrVal=0;

	// Set capture frame control
	bCptFrmCtrl=false;
	uCptSeqPattern=0, uCptSeqPointer=0;
	// Set capture count control
	bCptCntCtrl=false;
	uCptCnt=0;
}

// [Select FIMC A]
static void SelFimcA(void)
{
	eCh = FIMC_A;
	//FIMC_SetChA();
	Disp("\n FIMC_A selected! \n");
}

// [Select FIMC B]
static void SelFimcB(void)
{
	eCh = FIMC_B;
	//FIMC_SetChB();	
	Disp("\n FIMC_B selected! \n");
}

// [Select FIMC C]
static void SelFimcC(void)
{
	eCh = FIMC_C;
	//FIMC_SetChC();	
	Disp("\n FIMC_C selected! \n");
}

static void TestSfr(void)
{
	static REGINFO *sRegInfo;
	
	volatile u32 *pAddr;	
	s32 i,j, count , nPattern;
	u32 uRegStatus, uWritePatternWithMask, uReadWithMask;	
	
	u32 uPatternShitfOneBit[] = 
	{
		0x00000001,		0x00000002,		0x00000004,		0x00000008,		0x00000010,		0x00000020,		0x00000040,		0x00000080,
		0x00000100,		0x00000200,		0x00000400,		0x00000800,		0x00001000,		0x00002000,		0x00004000,		0x00008000,
		0x00010000,		0x00020000,		0x00040000,		0x00080000,		0x00100000,		0x00200000,		0x00400000,		0x00800000,
		0x01000000,		0x02000000,		0x04000000,		0x08000000,		0x10000000,		0x20000000,		0x40000000,		0x80000000,
	};
	
	u32 uDefaultBitMask[] =
	{
		0x00000001,		0x00000003,		0x00000007,		0x0000000F,		0x0000001F,		0x0000003F,		0x0000007F,		0x000000FF,
		0x000001FF,		0x000003FF,		0x000007FF,		0x00000FFF,		0x00001FFF,		0x00003FFF,		0x00007FFF,		0x0000FFFF,
		0x0001FFFF,		0x0003FFFF,		0x0007FFFF,		0x000FFFFF,		0x001FFFFF,		0x003FFFFF,		0x007FFFFF,		0x00FFFFFF,
		0x01FFFFFF,		0x03FFFFFF,		0x07FFFFFF,		0x0FFFFFFF,		0x1FFFFFFF,		0x3FFFFFFF,		0x7FFFFFFF,		0xFFFFFFFF,
	};

	switch(eCh)
	{
		case FIMC_A:
			 sRegInfo = sRegInfo_FimcA;
			 count = sizeof(sRegInfo_FimcA)/sizeof(REGINFO);
			 break;
		case FIMC_B:
			 sRegInfo = sRegInfo_FimcB;
			 count = sizeof(sRegInfo_FimcB)/sizeof(REGINFO);
			 break;
		case FIMC_C:
			 sRegInfo = sRegInfo_FimcC;
			 count = sizeof(sRegInfo_FimcC)/sizeof(REGINFO);
			 break;
		default:
			Assert(0);
			break;
	}
	
	for(i=0; i<count; i++)
	{		
		pAddr = (u32*)sRegInfo[i].uAddr;		
		
		if(sRegInfo[i].uFlag == DPDB)
		{		
			if(sRegInfo[i].uRWType == RW)
			{
				for(j=0; j< sRegInfo[i].uBitLen ; j++)
				{			
					uWritePatternWithMask = (uPatternShitfOneBit[j] &uDefaultBitMask[j]);
					*pAddr = uWritePatternWithMask;					
					
					uReadWithMask = (*pAddr & uDefaultBitMask[j]);	
					sRegInfo[i].rValue = uReadWithMask;
					
					uRegStatus = (	uWritePatternWithMask !=uReadWithMask );					
					RPRINTF(sRegInfo[i], uPatternShitfOneBit[j], uRegStatus, j, 1);
				}
			}
		}		
		else if(sRegInfo[i].uFlag == DPPB)
		{		
			nPattern = (sizeof(uPatternShitfOneBit)/sizeof(uPatternShitfOneBit[0]));			
			if(sRegInfo[i].uRWType == RW)
			{
				for(j=0; j<nPattern; j++)
				{
					if(uPatternShitfOneBit[j] & sRegInfo[i].uPrivateBitMask)
					{
						uWritePatternWithMask = (uPatternShitfOneBit[j] & sRegInfo[i].uPrivateBitMask);
						*pAddr = uWritePatternWithMask;
						
						uReadWithMask = (*pAddr & sRegInfo[i].uPrivateBitMask);
						sRegInfo[i].rValue = uReadWithMask;
						
						uRegStatus = (	uWritePatternWithMask != uReadWithMask );												
						RPRINTF(sRegInfo[i], uPatternShitfOneBit[j], uRegStatus, j, 1);					
					}
				}
			}			
		}					
	}
	
	UART_Printf("\n\nTest Complete!\n\n");
}

// [Three stage for DMA-in DMA-out]
static void ThreeStageForDinDout(void)
{
	u32 uTemp;

	// ======================= [The 1st CAMIF IP] =======================

	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uLcdFbAddr, eBpp,
		uInImgHsz, uInImgVsz,
		0, 0, uInImgHsz, uInImgVsz, 0, // Scaler	
		uInFbAddr, eBppInFmt,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);

	// ======================= [The 2nd CAMIF IP] =======================

	FIMC_InitIpForDinDout(eCh,
		uInImgHsz, uInImgVsz, 0,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uInFbAddr, eBppInFmt,
		uOutDispHsz, uOutDispVsz,
		uOutImgStartX, uOutImgStartY, uOutImgHsz, uOutImgVsz, 0, // Scaler
		uOutFbAddr, eBppOutFmt,		
		eRotFlip, eCscEquation, eYCbCrRange, false);
	FIMC_SetImgEffect(eCh, eImgEff, uCbVal, uCrVal, AFTER_SCALING);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);

	// ======================= [The 3rd CAMIF IP] =======================

	// Exception for rotation
	switch(eRotFlip)
	{
		case ROT_NO_FLIP_NO:
		case ROT_NO_FLIP_X:
		case ROT_NO_FLIP_Y:
		case ROT_NO_FLIP_XY:
			break;
		case ROT_90_FLIP_NO:
		case ROT_90_FLIP_X:
		case ROT_90_FLIP_Y:
		case ROT_90_FLIP_XY:
			//u32 uTemp;
			uTemp = uOutImgHsz;
			uOutImgHsz = uOutImgVsz;
			uOutImgVsz = uTemp;
			break;
		default:
			Assert(0);
			break;
	}

	// Clear screen by black color
	memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitIpForDinDout(eCh,
		uOutDispHsz, uOutDispVsz, 0,
		uOutImgStartX, uOutImgStartY, uOutImgHsz, uOutImgVsz,
		uOutFbAddr, eBppOutFmt,
		uOutDispHsz, uOutDispVsz,
		uOutImgStartX, uOutImgStartY, uOutImgHsz, uOutImgVsz, 0, // Scaler		
		uLcdOutFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false); // Image effect

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);	
	FIMC_StopProcessing(eCh);	
}

// [Test DMA-in DMA-out for basic function]
static void TestDinDoutForBasicFunc(void)
{
	u32 i;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	for(i=0; i<2; i++)
	{
		// Generate DMA input image
		GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
		switch(i)
		{
			case 0:
				GLIB_DrawPattern(uLcdHsz, uLcdVsz); 
				break;
			case 1:
				GLIB_DrawPattern2(uLcdHsz, uLcdVsz); 
				break;
			default:
				Assert(0);
				break;
		}
		
		ThreeStageForDinDout();

		Disp("\n Press any key to go next! \n");
		UART_Getc();		
	}

}

// [Test DMA-in DMA-out for each function]
static void TestDinDoutForEachFunc(void)
{
	u32 i, j, k;
	u32 uInImgHVszList[10][2];
	IMG_FMT eImgFmtList[11] = 
	{
		NV12, 
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);	

	// [Image cropping]
	//----------------
	Disp("\n Image cropping \n");	

	InitVariables();

	for(i=0; i<4; i++)
	{
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		switch(i)
		{
			case 0:
				uInImgCropStartX = 0;
				uInImgCropStartY = 0;
				break;
			case 1:
				uInImgCropStartX = uLcdHsz/2;
				uInImgCropStartY = 0;		
				break;
			case 2:
				uInImgCropStartX = 0;
				uInImgCropStartY = uLcdVsz/2;				
				break;
			case 3:
				uInImgCropStartX = uLcdHsz/2;
				uInImgCropStartY = uLcdVsz/2;				
				break;
			default:
				Assert(0);
				break;
		}

		uInImgCropHsz = (uLcdHsz/2)-((uLcdHsz/2)%16); // Set cropping H size
		uInImgCropVsz = (uLcdVsz/2)-((uLcdVsz/2)%16); // Set cropping V size		
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdFbAddr,	eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		
		WaitForFimcDone(eCh);				
		
		FIMC_StopProcessing(eCh);
	
		UART_Getc();	
	}

	// [Image scaling]
	//---------------
	Disp("\n Image scaling \n");

	InitVariables();

	for(i=0; i<5; i++)
	{
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		switch(i)
		{
			case 0:
				uOutImgHsz = (uLcdHsz/5)-((uLcdHsz/5)%16);
				uOutImgVsz = (uLcdVsz/5)-((uLcdVsz/5)%16);
				break;
			case 1:
				uOutImgHsz = (uLcdHsz/4)-((uLcdHsz/4)%16);
				uOutImgVsz = (uLcdVsz/4)-((uLcdVsz/4)%16);				
				break;
			case 2:
				uOutImgHsz = (uLcdHsz/3)-((uLcdHsz/3)%16);
				uOutImgVsz = (uLcdVsz/3)-((uLcdVsz/3)%16);				
				break;
			case 3:
				uOutImgHsz = (uLcdHsz/2)-((uLcdHsz/2)%16);
				uOutImgVsz = (uLcdVsz/2)-((uLcdVsz/2)%16);				
				break;
			case 4:
				uOutImgHsz = (uLcdHsz/1)-((uLcdHsz/1)%16);
				uOutImgVsz = (uLcdVsz/1)-((uLcdVsz/1)%16);				
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr,	eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uOutImgHsz, uOutImgVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		
		WaitForFimcDone(eCh);		
		
		FIMC_StopProcessing(eCh);
	
		UART_Getc();
	}

	for(i=0; i<4; i++)
	{
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		switch(i)
		{
			case 0:
				uInImgCropStartX = 24;
				uInImgCropStartY = 32;
				uInImgCropHsz = (uLcdHsz-(24*2))-((uLcdHsz-(24*2))%16);
				uInImgCropVsz = (uLcdVsz-(32*2))-((uLcdVsz-(32*2))%16);				
				break;
			case 1:
				uInImgCropStartX = 48;
				uInImgCropStartY = 64;
				uInImgCropHsz = (uLcdHsz-(48*2))-((uLcdHsz-(48*2))%16);
				uInImgCropVsz = (uLcdVsz-(64*2))-((uLcdVsz-(64*2))%16);				
				break;
			case 2:
				uInImgCropStartX = 60;
				uInImgCropStartY = 80;
				uInImgCropHsz = (uLcdHsz-(60*2))-((uLcdHsz-(60*2))%16);
				uInImgCropVsz = (uLcdVsz-(80*2))-((uLcdVsz-(80*2))%16);				
				break;
			case 3:
				uInImgCropStartX = 72;
				uInImgCropStartY = 96;
				uInImgCropHsz = (uLcdHsz-(72*2))-((uLcdHsz-(72*2))%16);
				uInImgCropVsz = (uLcdVsz-(96*2))-((uLcdVsz-(96*2))%16);				
				break;
			default:
				Assert(0);
				break;
		}		
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdFbAddr,	eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		
		WaitForFimcDone(eCh);		
		
		FIMC_StopProcessing(eCh);
	
		UART_Getc();
	}

	// [Rotation(Output rotator) and Flip]
	//------------------------------
	Disp("\n Rotation(Output rotator) and Flip \n");

	InitVariables();

	for(i=0; i<8; i++)
	{
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		if(uLcdHsz >= uLcdVsz)
		{
			uOutImgHsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
			uOutImgVsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
		}
		else // uLcdHsz < uLcdVsz
		{
			uOutImgHsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
			uOutImgVsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
		}
		
		switch(i)
		{
			case 0:
				eRotFlip = ROT_NO_FLIP_NO;
				break;
			case 1:
				eRotFlip = ROT_NO_FLIP_X;
				break;
			case 2:
				eRotFlip = ROT_NO_FLIP_Y;
				break;
			case 3:
				eRotFlip = ROT_NO_FLIP_XY;
				break;
			case 4:
				eRotFlip = ROT_90_FLIP_NO;
				break;
			case 5:
				eRotFlip = ROT_90_FLIP_X;
				break;
			case 6:
				eRotFlip = ROT_90_FLIP_Y;
				break;
			case 7:
				eRotFlip = ROT_90_FLIP_XY;
				break;
			default:
				Assert(0);
				break;
		}
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr,	eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uOutImgHsz, uOutImgVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			eRotFlip, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		
		WaitForFimcDone(eCh);		
		
		FIMC_StopProcessing(eCh);
	
		UART_Getc();
	}

	// [Image effect] 
	//--------------	
	Disp("\n Image effect \n");

	InitVariables();

	for(i=0; i<5; i++)
	{
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		switch(i)
		{
			case 0:
				eImgEff = EFF_BYPASS;
				break;
			case 1:
				eImgEff = NEGATIVE;
				break;
			case 2:
				eImgEff = ART_FREEZE;
				break;
			case 3:
				eImgEff = EMBOSSING;
				break;
			case 4:
				eImgEff = SILHOUETTE;
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		
		WaitForFimcDone(eCh);		
		
		FIMC_StopProcessing(eCh);	
		
		UART_Getc();		
	}

	for(i=0; i<5; i++)
	{
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		eImgEff = ARBITRARY_CBCR;

		switch(i)
		{
			case 0:
				uCbVal = 1;
				uCrVal = 1;
				break;
			case 1:
				uCbVal = 128;
				uCrVal = 128;				
				break;
			case 2:
				uCbVal = 254;
				uCrVal = 254;				
				break;
			case 3:
				uCbVal = 1;
				uCrVal = 254;				
				break;
			case 4:
				uCbVal = 254;
				uCrVal = 1;				
				break;
			default:
				Assert(0);
				break;
		}
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr,	eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		
		WaitForFimcDone(eCh);		
		
		FIMC_StopProcessing(eCh);	
		
		UART_Getc();					
	}	

	// [Variable input image size]
	//------------------------
	Disp("\n Variable input image size \n");

	InitVariables();

	switch(eCh)
	{
		case FIMC_A:
		case FIMC_B:
			uInImgHVszList[0][0]=320; uInImgHVszList[0][1]=240;
			uInImgHVszList[1][0]=640; uInImgHVszList[1][1]=480;
			uInImgHVszList[2][0]=800; uInImgHVszList[2][1]=480;
			uInImgHVszList[3][0]=1280; uInImgHVszList[3][1]=720;
			uInImgHVszList[4][0]=1920; uInImgHVszList[4][1]=1080;
			uInImgHVszList[5][0]=2048; uInImgHVszList[5][1]=1536;
			uInImgHVszList[6][0]=2560; uInImgHVszList[6][1]=2048;
			uInImgHVszList[7][0]=3264; uInImgHVszList[7][1]=2448; // 8M
			uInImgHVszList[8][0]=4032; uInImgHVszList[8][1]=3008; // 12M
			uInImgHVszList[9][0]=4224; uInImgHVszList[9][1]=4224;		
			break;
		case FIMC_C:
			uInImgHVszList[0][0]=320; uInImgHVszList[0][1]=240;
			uInImgHVszList[1][0]=640; uInImgHVszList[1][1]=480;
			uInImgHVszList[2][0]=720; uInImgHVszList[2][1]=576;
			uInImgHVszList[3][0]=768; uInImgHVszList[3][1]=576;
			uInImgHVszList[4][0]=800; uInImgHVszList[4][1]=480;
			uInImgHVszList[5][0]=1024; uInImgHVszList[5][1]=768;
			uInImgHVszList[6][0]=1280; uInImgHVszList[6][1]=720;
			uInImgHVszList[7][0]=1440; uInImgHVszList[7][1]=1440;
			uInImgHVszList[8][0]=1600; uInImgHVszList[8][1]=1200;
			uInImgHVszList[9][0]=1920; uInImgHVszList[9][1]=1080;				
			break;
		default:
			Assert(0);
			break;
	}

	for(i=0; i<11; i++)
	{
		for(j=0; j<10; j++)
		{
			eBppInFmt = eImgFmtList[i];
			eBppOutFmt = eBppInFmt;
			
			uInImgHsz = uInImgHVszList[j][0];
			uInImgVsz = uInImgHVszList[j][1];

			uInImgCropHsz = uInImgHsz;
			uInImgCropVsz = uInImgVsz;
					
			ThreeStageForDinDout();

			Disp("\n eBppInOutFmt order num = %d, uInImgHVsz order num = %d \n", i, j);
			Disp(" Hit any key to go to the next step! \n");	
			UART_Getc();		
		}
	}
		
	// [Memory store type and Color format conversion]
	//--------------------------------------	
	Disp("\n Memory store type and Color format conversion \n");

	InitVariables();

	for(i=0; i<11; i++)
	{
		for(j=0; j<11; j++)
		{
			eBppInFmt = eImgFmtList[i];
			eBppOutFmt = eImgFmtList[j];

			ThreeStageForDinDout();
			
			Disp("\n eBppInFmt order num = %d, eBppOutFmt order num = %d \n", i, j);
			Disp(" Hit any key to go to the next step! \n");	
			UART_Getc();		
		}
	}

	// [Variable input image size + Memory store type and Color format conversion]
	//-------------------------------------------------------------
	Disp("\n Variable input image size + Memory store type and Color format conversion \n");
	
	InitVariables();

	for(i=0; i<11; i++)
	{
		for(j=0; j<11; j++)
		{
			for(k=0; k<10; k++)
			{
				eBppInFmt = eImgFmtList[i];
				eBppOutFmt = eImgFmtList[j];

				uInImgHsz = uInImgHVszList[k][0];
				uInImgVsz = uInImgHVszList[k][1];

				uInImgCropHsz = uInImgHsz;
				uInImgCropVsz = uInImgVsz;		

				ThreeStageForDinDout();
				
				Disp("\n eBppInFmt order num = %d, eBppOutFmt order num = %d, InImgHVsz = %d \n", i, j, k);
				Disp(" Hit any key to go to the next step! \n");	
				UART_Getc();		
			}	
		}
	}
	
}

// [Test DMA-in DMA-out for composition]
static void TestDinDoutForComposition(void)
{
	u32 i, j;
	u32 uInImgCropHszList[3];
	u32 uInImgCropVszList[3];
	u32 uInImgCropStartXList[3] = {16, 48, 80};
	u32 uInImgCropStartYList[3] = {24, 56, 80};
	u32 uInImgHVszList[8][2];

	IMG_FMT eImgFmtList[11] = 
	{	
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};
	
	ROT_FLIP eRotFlipList[8] = 
	{
		ROT_NO_FLIP_NO,
		ROT_NO_FLIP_X,
		ROT_NO_FLIP_Y,
		ROT_NO_FLIP_XY,
		ROT_90_FLIP_NO,
		ROT_90_FLIP_X,
		ROT_90_FLIP_Y,
		ROT_90_FLIP_XY
	};

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);	

	// [Composition] 
	//-------------
	Disp("\n Composition \n");

	InitVariables();

	switch(eCh)
	{
		case FIMC_A:
		case FIMC_B:
			uInImgHVszList[0][0]=800; uInImgHVszList[0][1]=480;
			uInImgHVszList[1][0]=1280; uInImgHVszList[1][1]=720;
			uInImgHVszList[2][0]=1920; uInImgHVszList[2][1]=1080;
			uInImgHVszList[3][0]=2048; uInImgHVszList[3][1]=1536;
			uInImgHVszList[4][0]=2560; uInImgHVszList[4][1]=2048;
			uInImgHVszList[5][0]=3264; uInImgHVszList[5][1]=2448; // 8M
			uInImgHVszList[6][0]=4032; uInImgHVszList[6][1]=3008; // 12M
			uInImgHVszList[7][0]=4224; uInImgHVszList[7][1]=4224;		
			break;
		case FIMC_C:
			uInImgHVszList[0][0]=720; uInImgHVszList[0][1]=576;
			uInImgHVszList[1][0]=768; uInImgHVszList[1][1]=576;
			uInImgHVszList[2][0]=800; uInImgHVszList[2][1]=480;
			uInImgHVszList[3][0]=1024; uInImgHVszList[3][1]=768;
			uInImgHVszList[4][0]=1280; uInImgHVszList[4][1]=720;
			uInImgHVszList[5][0]=1440; uInImgHVszList[5][1]=1440;
			uInImgHVszList[6][0]=1600; uInImgHVszList[6][1]=1200;
			uInImgHVszList[7][0]=1920; uInImgHVszList[7][1]=1080;				
			break;
		default:
			Assert(0);
			break;
	}

	uInImgCropHszList[0]=(640-(16*2))-((640-(16*2))%16);
	uInImgCropHszList[1]=(640-(48*2))-((640-(48*2))%16);
	uInImgCropHszList[2]=(640-(80*2))-((640-(80*2))%16);

	uInImgCropVszList[0]=(480-(24*2))-((480-(24*2))%16);
	uInImgCropVszList[1]=(480-(56*2))-((480-(56*2))%16);
	uInImgCropVszList[2]=(480-(80*2))-((480-(80*2))%16);

	for(i=0; i<11; i++)
	{
		for(j=0; j<11; j++)
		{
			uInImgHsz = uInImgHVszList[((11*i)+j)%8][0];
			uInImgVsz = uInImgHVszList[((11*i)+j)%8][1];			
			uInImgCropStartX = uInImgCropStartXList[(((11*i)+j)/3)%3];
			uInImgCropStartY = uInImgCropStartYList[((11*i)+j)%3];
			uInImgCropHsz = uInImgCropHszList[(((11*i)+j)/3)%3];
			uInImgCropVsz = uInImgCropVszList[((11*i)+j)%3];
			eBppInFmt = eImgFmtList[i];

			uOutImgStartX=16, uOutImgStartY=24;			
			if(uLcdHsz >= uLcdVsz)
			{
				uOutImgHsz = uLcdVsz-(uLcdVsz/8)-((uLcdVsz-(uLcdVsz/8))%16);
				uOutImgVsz = uLcdVsz-(uLcdVsz/8)-((uLcdVsz-(uLcdVsz/8))%16);
			}
			else // uLcdHsz < uLcdVsz
			{
				uOutImgHsz = uLcdHsz-(uLcdHsz/8)-((uLcdHsz-(uLcdHsz/8))%16);
				uOutImgVsz = uLcdHsz-(uLcdHsz/8)-((uLcdHsz-(uLcdHsz/8))%16);
			}			
			eBppOutFmt = eImgFmtList[j];

			eRotFlip = eRotFlipList[((11*i)+j)%8];

			ThreeStageForDinDout();

			Disp("\n eBppInFmt order num = %d, eBppOutFmt order num = %d \n", i, j);
			Disp(" H/V=%d, CropX=%d, CropY=%d, CropH=%d, CropV=%d, RotFlip=%d \n",
				((11*i)+j)%8, (((11*i)+j)/3)%3, ((11*i)+j)%3,
				(((11*i)+j)/3)%3, ((11*i)+j)%3, ((11*i)+j)%8);
			Disp("\n Hit any key to go to the next step! \n");	
			UART_Getc();		
		}	
	}
	
}

// [Test DMA-in DMA-out for input DMA line skip]
static void TestDinDoutForInDmaLineSkip(void)
{
	u32 i, uLineSkipVal;
	
	IMG_FMT eImgFmtList[11] = 
	{
		NV12, 
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);	

	for(i=0; i<11; i++)
	{
		//for(uLineSkipVal=0; uLineSkipVal<16; uLineSkipVal++)
		for(uLineSkipVal=0; uLineSkipVal<4; uLineSkipVal++)
		{
			Disp("\n ImgFmtOrder = %d, LineSkipValue = %d \n", i, uLineSkipVal);

			memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
			
			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				0, 0, uLcdHsz, uLcdVsz,
				uLcdFbAddr, eBpp,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
				uInFbAddr, eImgFmtList[i],		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, uLineSkipVal,
				0, 0, uLcdHsz, uLcdVsz/(uLineSkipVal+1),
				uInFbAddr, eImgFmtList[i],
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz/(uLineSkipVal+1), 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);
			
			Disp("\n Hit any key to go next! \n");	
			UART_Getc();				
		}
	}

}

// [Test DMA-in DMA-out for output DMA line skip]
static void TestDinDoutForOutDmaLineSkip(void)
{
	u32 i, uLineSkipVal;
	
	IMG_FMT eImgFmtList[11] = 
	{
		NV12, 
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);	

	for(i=0; i<11; i++)
	{
		//for(uLineSkipVal=0; uLineSkipVal<16; uLineSkipVal++)
		for(uLineSkipVal=0; uLineSkipVal<4; uLineSkipVal++)
		{
			Disp("\n ImgFmtOrder = %d, LineSkipValue = %d \n", i, uLineSkipVal);

			memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
			memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				0, 0, uLcdHsz, uLcdVsz,
				uLcdFbAddr, eBpp,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz/(uLineSkipVal+1), uLineSkipVal, // Scaler	
				uInFbAddr, eImgFmtList[i],		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				0, 0, uLcdHsz, uLcdVsz,
				uInFbAddr, eImgFmtList[i],
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);		
			
			Disp("\n Hit any key to go next! \n");	
			UART_Getc();				
		}
	}
	
}

// [Test DMA-in DMA-out for One2one]
static void TestDinDoutForOne2one(void)
{
	u32 i, j, k;
	u32 uImgHsz, uImgVsz;
	u32 uFileSz;
	u8 bResult;

	IMG_FMT eImgFmtList[11] = 
	{	
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	for(k=0; k<11; k++)
	{
		Disp("\n ===== ImgFmtOrder = %d ===== \n", k);
		
		for(i=0; i<2; i++)
		{
			switch(i)
			{
				case 0:
					if((eCh==FIMC_A)||(eCh==FIMC_B))
					{
						uImgHsz = 4224; // MAIN_SC_IN_MAX_HSZ_A/MAIN_SC_IN_MAX_HSZ_B
						uImgVsz = 16;				
					}
					else if(eCh==FIMC_C)
					{
						uImgHsz = 1920; // MAIN_SC_IN_MAX_HSZ_C
						uImgVsz = 16;				
					}
					else
					{
						Assert(0);
					}
					break;
				case 1:
					uImgHsz = 16;
					uImgVsz = 8192;	
					break;
				default:
					Assert(0);
					break;
			}

			Disp("\n Generating sample image... \n");
			// Generate DMA input image
			GLIB_InitInstance(uLcdFbAddr, uImgHsz, uImgVsz, eBpp);
			GLIB_DrawPattern(uImgHsz, uImgVsz); 		

			FIMC_GetNextFbAddr(eImgFmtList[k], uImgHsz, uImgVsz, 0x0, &uFileSz);
			Disp("\n uFileSz = %d \n", uFileSz);
			memset((void *)uInFbAddr, 0, uFileSz); // Clear screen by black color
			memset((void *)uOutFbAddr, 0, uFileSz); // Clear screen by black color
		
			FIMC_InitIpForDinDout(eCh,
				uImgHsz, uImgVsz, 0,
				0, 0, uImgHsz, uImgVsz,
				uLcdFbAddr, eBpp,
				uImgHsz, uImgVsz,
				0, 0, uImgHsz, uImgVsz, 0, // Scaler		
				uInFbAddr, eImgFmtList[k],		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			// One2one
			FIMC_InitIpForDinDout(eCh,
				uImgHsz, uImgVsz, 0,
				0, 0, uImgHsz, uImgVsz,
				uInFbAddr, eImgFmtList[k],
				uImgHsz, uImgVsz,
				0, 0, uImgHsz, uImgVsz, 0, // Scaler		
				uOutFbAddr, eImgFmtList[k],		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, true);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//-----------------------------

			Disp("\n Comparing source and result... \n");

			for(j=0; j<uFileSz; j++) // MAIN_SC_IN_MAX_HSZ_A/MAIN_SC_IN_MAX_HSZ_B
			{
				bResult = Compare8(uInFbAddr+j, uOutFbAddr+j, 1);

				// For log
				//Disp("\n (0x%x, 0x%x) = (0x%x, 0x%x) \n", 
				//	(uInFbAddr+j), Inp8(uInFbAddr+j), (uOutFbAddr+j), Inp8(uOutFbAddr+j));			
				
				Assert(bResult);	
			}

			Disp("\n Source and result are same: PASS \n");

			//-----------------------------

			memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

			FIMC_InitIpForDinDout(eCh,
				uImgHsz, uImgVsz, 0,
				0, 0, uImgHsz, uImgVsz,
				uOutFbAddr, eImgFmtList[k],
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);
			
			//Disp("\n Hit any key to go next! \n");	
			//UART_Getc();		
		}
	}

	//-----------------------------

	// For MFC conformance test (input: TILE_NV12, output: YC420)
	Disp("\n ===== For MFC conformance test (input: TILE_NV12, output: YC420) ===== \n");
	for(i=0; i<2; i++)
	{
		switch(i)
		{
			case 0:
				if((eCh==FIMC_A)||(eCh==FIMC_B))
				{
					uImgHsz = 4224; // MAIN_SC_IN_MAX_HSZ_A/MAIN_SC_IN_MAX_HSZ_B
					uImgVsz = 16;				
				}
				else if(eCh==FIMC_C)
				{
					uImgHsz = 1920; // MAIN_SC_IN_MAX_HSZ_C
					uImgVsz = 16;				
				}
				else
				{
					Assert(0);
				}
				break;
			case 1:
				uImgHsz = 16;
				uImgVsz = 8192; 
				break;
			default:
				Assert(0);
				break;
		}

		Disp("\n Generating sample image... \n");
		// Generate DMA input image
		GLIB_InitInstance(uLcdFbAddr, uImgHsz, uImgVsz, eBpp);
		GLIB_DrawPattern(uImgHsz, uImgVsz);		

		FIMC_GetNextFbAddr(TILE_NV12, uImgHsz, uImgVsz, 0x0, &uFileSz);
		Disp("\n uFileSz = %d \n", uFileSz);
		memset((void *)uInFbAddr, 0, uFileSz); // Clear screen by black color
		memset((void *)uLcdOutFbAddr, 0, uFileSz); // Clear screen by black color

		FIMC_InitIpForDinDout(eCh,
			uImgHsz, uImgVsz, 0,
			0, 0, uImgHsz, uImgVsz,
			uLcdFbAddr, eBpp,
			uImgHsz, uImgVsz,
			0, 0, uImgHsz, uImgVsz, 0, // Scaler		
			uInFbAddr, TILE_NV12,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);		

		// One2one
		FIMC_InitIpForDinDout(eCh,
			uImgHsz, uImgVsz, 0,
			0, 0, uImgHsz, uImgVsz,
			uInFbAddr, TILE_NV12,
			uImgHsz, uImgVsz,
			0, 0, uImgHsz, uImgVsz, 0, // Scaler		
			uOutFbAddr, YC420,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, true);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);	

		// One2one
		FIMC_InitIpForDinDout(eCh,
			uImgHsz, uImgVsz, 0,
			0, 0, uImgHsz, uImgVsz,
			uOutFbAddr, YC420,
			uImgHsz, uImgVsz,
			0, 0, uImgHsz, uImgVsz, 0, // Scaler		
			uLcdOutFbAddr, TILE_NV12,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, true);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);	

		//-----------------------------

		Disp("\n Comparing source and result... \n");

		for(j=0; j<uFileSz; j++) // MAIN_SC_IN_MAX_HSZ_A/MAIN_SC_IN_MAX_HSZ_B
		{
			bResult = Compare8(uInFbAddr+j, uLcdOutFbAddr+j, 1);

			// For log
			//Disp("\n (0x%x, 0x%x) = (0x%x, 0x%x) \n", 
			//	(uInFbAddr+j), Inp8(uInFbAddr+j), (uLcdOutFbAddr+j), Inp8(uLcdOutFbAddr+j));			
			
			Assert(bResult);	
		}

		Disp("\n Source and result are same: PASS \n");
	}	

}

#if 0
// [Test DMA-in DMA-out for interlace bob out]
static void TestDinDoutForInterlaceBobOut(void)
{
	IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	Disp("\n Generating sample image... \n");
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	for(i=0; i<4; i++)
	{
		for(j=0; j<8; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = uLcdHsz;
					uOutDispVsz = uLcdVsz/2;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uOutDispHsz = uLcdVsz/2;
					uOutDispVsz = uLcdHsz;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;		
					break;
				default:
					Assert(0);
					break;
			}

			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				0, 0, uLcdHsz, uLcdVsz,
				uLcdFbAddr,	eBpp,
				uOutDispHsz, uOutDispVsz,
				0, 0, uOutImgHsz, uOutImgVsz, 0, // Scaler		
				uInFbAddr, eInterlaceImgFmt[i],		
				eRotFlipList[j], eCscEquation, eYCbCrRange, false);
#if 1
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
			WaitForFimcDone(eCh);		
			FIMC_StopProcessing(eCh);
#else
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
			WaitForFimcDone(eCh);		
			FIMC_StopProcessing(eCh);						

			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
			WaitForFimcDone(eCh);		
			FIMC_StopProcessing(eCh);
#endif			
			//--------------------------------------------

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					uInImgVsz = uLcdVsz/2;	
					
					uInImgCropHsz = uLcdHsz; 
					uInImgCropVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uInImgHsz =  uLcdVsz/2;
					uInImgVsz = uLcdHsz;			
					
					uInImgCropHsz =  uLcdVsz/2; 
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			

			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uInFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			// Odd field
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uInFbAddr, &uNextFbAddr);
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uNextFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}
		
}
#endif

// [Test DMA-in FIFO-out for basic function]
static void TestDinFoutForBasicFunc(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	InitVariables();

	for(i=0; i<2; i++)
	{
		// Generate DMA input image
		GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
		switch(i)
		{
			case 0:
				GLIB_DrawPattern(uLcdHsz, uLcdVsz);
				break;
			case 1:
				GLIB_DrawPattern2(uLcdHsz, uLcdVsz);
				break;
			default:
				Assert(0);
				break;
		}		

		FIMC_InitIpForDinFout(eCh,
			uInImgHsz, uInImgVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdFbAddr, eBpp, false,
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();
	}

}

// [Test DMA-in FIFO-out for each function]
static void TestDinFoutForEachFunc(void)
{
	u32 i, j;
	u32 uDispHSz, uDispVSz;
	u32 uInImgHVszList[10][2];

	IMG_FMT eImgFmtList[11] = 
	{	NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	// [Image cropping]
	//----------------
	Disp("\n Image cropping \n");

	InitVariables();
	
	for(i=0; i<4; i++)
	{
		switch(i)
		{
			case 0:
				uInImgCropStartX = 0;
				uInImgCropStartY = 0;
				break;
			case 1:
				uInImgCropStartX = uLcdHsz/2;
				uInImgCropStartY = 0;				
				break;
			case 2:
				uInImgCropStartX = 0;
				uInImgCropStartY = uLcdVsz/2;				
				break;
			case 3:
				uInImgCropStartX = uLcdHsz/2;
				uInImgCropStartY = uLcdVsz/2;				
				break;
			default:
				Assert(0);
				break;
		}

		uInImgCropHsz = (uLcdHsz/2)-((uLcdHsz/2)%16);
		uInImgCropVsz = (uLcdVsz/2)-((uLcdVsz/2)%16);

		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdFbAddr, eBpp, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);
	
		UART_Getc();	
	}

	// [Image scaling]
	//---------------
	Disp("\n Image scaling \n");

	InitVariables();
	
	for(i=0; i<4; i++)
	{
		switch(i)
		{
			case 0:
				uInImgCropStartX = 24;
				uInImgCropStartY = 32;
				uInImgCropHsz = (uLcdHsz-(24*2))-((uLcdHsz-(24*2))%16);
				uInImgCropVsz = (uLcdVsz-(32*2))-((uLcdVsz-(32*2))%16);
				break;
			case 1:
				uInImgCropStartX = 48;
				uInImgCropStartY = 64;
				uInImgCropHsz = (uLcdHsz-(48*2))-((uLcdHsz-(48*2))%16);
				uInImgCropVsz = (uLcdVsz-(64*2))-((uLcdVsz-(64*2))%16);				
				break;
			case 2:
				uInImgCropStartX = 72;
				uInImgCropStartY = 96;
				uInImgCropHsz = (uLcdHsz-(72*2))-((uLcdHsz-(72*2))%16);
				uInImgCropVsz = (uLcdVsz-(96*2))-((uLcdVsz-(96*2))%16);				
				break;
			case 3:
				uInImgCropStartX = 96;
				uInImgCropStartY = 112;
				uInImgCropHsz = (uLcdHsz-(96*2))-((uLcdHsz-(96*2))%16);
				uInImgCropVsz = (uLcdVsz-(112*2))-((uLcdVsz-(112*2))%16);				
				break;
			default:
				Assert(0);
				break;
		}
		
		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdFbAddr, eBpp, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);
	
		UART_Getc();
	}

	// [Rotation(Input rotator) and Flip]
	//-----------------------------
	Disp("\n Rotation(Input rotator) and Flip \n");

	InitVariables();
	
	for(i=0; i<8; i++)
	{
		switch(i)
		{
			case 0:
				eRotFlip = ROT_NO_FLIP_NO;
				break;
			case 1:
				eRotFlip = ROT_NO_FLIP_X;
				break;
			case 2:
				eRotFlip = ROT_NO_FLIP_Y;
				break;
			case 3:
				eRotFlip = ROT_NO_FLIP_XY;
				break;
			case 4:
				eRotFlip = ROT_90_FLIP_NO;
				break;
			case 5:
				eRotFlip = ROT_90_FLIP_X;
				break;
			case 6:
				eRotFlip = ROT_90_FLIP_Y;
				break;
			case 7:
				eRotFlip = ROT_90_FLIP_XY;
				break;
			default:
				Assert(0);
				break;
		}
		
		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			eRotFlip, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);
	
		UART_Getc();		
	}

	// [Image effect]
	//--------------
	Disp("\n Image effect \n");

	InitVariables();

	for(i=0; i<5; i++)
	{
		switch(i)
		{
			case 0:
				eImgEff = EFF_BYPASS;
				break;
			case 1:
				eImgEff = NEGATIVE;
				break;
			case 2:
				eImgEff = ART_FREEZE;
				break;
			case 3:
				eImgEff = EMBOSSING;
				break;
			case 4:
				eImgEff = SILHOUETTE;
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);	

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		UART_Getc();
	}

	for(i=0; i<5; i++)
	{
		eImgEff = ARBITRARY_CBCR;
		
		switch(i)
		{
			case 0:
				uCbVal = 1;
				uCrVal = 1;
				break;
			case 1:
				uCbVal = 128;
				uCrVal = 128;				
				break;
			case 2:
				uCbVal = 254;
				uCrVal = 254;				
				break;
			case 3:
				uCbVal = 1;
				uCrVal = 254;				
				break;
			case 4:
				uCbVal = 254;
				uCrVal = 1;				
				break;
			default:
				Assert(0);
				break;
		}		
		
		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);	

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);	
		
		UART_Getc();					
	}

	// [Memory store type and Color format conversion]
	//-------------------------------------
	Disp("\n Memory store type and Color format conversion \n");

	InitVariables();
	
	for(i=0; i<11; i++)
	{
		eBppInFmt = eImgFmtList[i];

		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler
			uInFbAddr, eBppInFmt,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);		
		FIMC_StopProcessing(eCh);
		
		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uInFbAddr, eBppInFmt, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n eBppInFmt order num = %d \n", i);
		Disp(" Hit any key to go to the next step! \n");	
		UART_Getc();
	}

	// [Variable input image size]
	//-----------------------
	Disp("\n Variable input image size \n");

	InitVariables();

	switch(eCh)
	{
		case FIMC_A:
		case FIMC_B:
			uInImgHVszList[0][0]=320; uInImgHVszList[0][1]=240;
			uInImgHVszList[1][0]=640; uInImgHVszList[1][1]=480;
			uInImgHVszList[2][0]=800; uInImgHVszList[2][1]=480;
			uInImgHVszList[3][0]=1280; uInImgHVszList[3][1]=720;
			uInImgHVszList[4][0]=1920; uInImgHVszList[4][1]=1080;
			uInImgHVszList[5][0]=2048; uInImgHVszList[5][1]=1536;
			uInImgHVszList[6][0]=2560; uInImgHVszList[6][1]=2048;
			uInImgHVszList[7][0]=3264; uInImgHVszList[7][1]=2448; // 8M
			//uInImgHVszList[8][0]=4032; uInImgHVszList[8][1]=3008; // 12M
			uInImgHVszList[8][0]=4096; uInImgHVszList[8][1]=3008; // 12M
			uInImgHVszList[9][0]=4224; uInImgHVszList[9][1]=4224;		
			break;
		case FIMC_C:
			uInImgHVszList[0][0]=320; uInImgHVszList[0][1]=240;
			uInImgHVszList[1][0]=640; uInImgHVszList[1][1]=480;
			uInImgHVszList[2][0]=720; uInImgHVszList[2][1]=576;
			uInImgHVszList[3][0]=768; uInImgHVszList[3][1]=576;
			uInImgHVszList[4][0]=800; uInImgHVszList[4][1]=480;
			uInImgHVszList[5][0]=1024; uInImgHVszList[5][1]=768;
			uInImgHVszList[6][0]=1280; uInImgHVszList[6][1]=720;
			uInImgHVszList[7][0]=1440; uInImgHVszList[7][1]=1440;
			uInImgHVszList[8][0]=1600; uInImgHVszList[8][1]=1200;
			uInImgHVszList[9][0]=1920; uInImgHVszList[9][1]=1080;				
			break;
		default:
			Assert(0);
			break;
	}

	for(i=0; i<11; i++)
	{
		for(j=0; j<10; j++)
		{
			uInImgHsz = uInImgHVszList[j][0];
			uInImgVsz = uInImgHVszList[j][1];

			uInImgCropHsz = uInImgHsz;
			uInImgCropVsz = uInImgVsz;

			eBppInFmt = eImgFmtList[i];

			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				0, 0, uLcdHsz, uLcdVsz,
				uLcdFbAddr, eBpp,
				uInImgHsz, uInImgVsz,
				0, 0, uInImgHsz, uInImgVsz, 0, // Scaler
				uInFbAddr, eBppInFmt,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
			WaitForFimcDone(eCh);				
			FIMC_StopProcessing(eCh);
			
			FIMC_InitIpForDinFout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uInFbAddr, eBppInFmt, false,		
				uDispHSz, uDispVSz, // Fifo out display size
				eLocalInCS, PROGRESSIVE,
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			Disp("\n Press any key to stop! \n");
			UART_Getc();
			FIMC_StopProcessing(eCh);

			Disp("\n eBppInFmt order num = %d, InImgHVsz order num = %d \n", i,j);
			Disp(" Hit any key to go to the next step! \n");	
			UART_Getc();		
		}
	}

}

// [Test DMA-in FIFO-out for composition]
static void TestDinFoutForComposition(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;
	u32 uInImgCropHszList[3];
	u32 uInImgCropVszList[3];
	u32 uInImgCropStartXList[3]={16,32,48};
	u32 uInImgCropStartYList[3]={48,56,80};
	u32 uInImgHVszList[8][2];

	IMG_FMT eImgFmtList[11] = 
	{
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};

	ROT_FLIP eRotFlipList[8] = 
	{
		ROT_NO_FLIP_NO,
		ROT_NO_FLIP_X,
		ROT_NO_FLIP_Y,
		ROT_NO_FLIP_XY,
		ROT_90_FLIP_NO,
		ROT_90_FLIP_X,
		ROT_90_FLIP_Y,
		ROT_90_FLIP_XY
	};

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);
	
	// [Composition]
	//-------------
	Disp("\n Composition \n");	

	InitVariables();

	switch(eCh)
	{
		case FIMC_A:
		case FIMC_B:
			uInImgHVszList[0][0]=800; uInImgHVszList[0][1]=480;
			uInImgHVszList[1][0]=1280; uInImgHVszList[1][1]=720;
			uInImgHVszList[2][0]=1920; uInImgHVszList[2][1]=1080;
			uInImgHVszList[3][0]=2048; uInImgHVszList[3][1]=1536;
			uInImgHVszList[4][0]=2560; uInImgHVszList[4][1]=2048;
			uInImgHVszList[5][0]=3264; uInImgHVszList[5][1]=2448; // 8M
			uInImgHVszList[6][0]=4032; uInImgHVszList[6][1]=3008; // 12M
			uInImgHVszList[7][0]=4224; uInImgHVszList[7][1]=4224;		
			break;
		case FIMC_C:
			uInImgHVszList[0][0]=720; uInImgHVszList[0][1]=576;
			uInImgHVszList[1][0]=768; uInImgHVszList[1][1]=576;
			uInImgHVszList[2][0]=800; uInImgHVszList[2][1]=480;
			uInImgHVszList[3][0]=1024; uInImgHVszList[3][1]=768;
			uInImgHVszList[4][0]=1280; uInImgHVszList[4][1]=720;
			uInImgHVszList[5][0]=1440; uInImgHVszList[5][1]=1440;
			uInImgHVszList[6][0]=1600; uInImgHVszList[6][1]=1200;
			uInImgHVszList[7][0]=1920; uInImgHVszList[7][1]=1080;				
			break;
		default:
			Assert(0);
			break;
	}

	uInImgCropHszList[0]=(640-(16*2))-((640-(16*2))%16);
	uInImgCropHszList[1]=(640-(48*2))-((640-(48*2))%16);
	uInImgCropHszList[2]=(640-(80*2))-((640-(80*2))%16);

	uInImgCropVszList[0]=(480-(24*2))-((480-(24*2))%16);
	uInImgCropVszList[1]=(480-(56*2))-((480-(56*2))%16);
	uInImgCropVszList[2]=(480-(80*2))-((480-(80*2))%16);

	for(i=0; i<11; i++)
	{
		uInImgHsz = uInImgHVszList[i%8][0];
		uInImgVsz = uInImgHVszList[i%8][1];
		
		uInImgCropStartX = uInImgCropStartXList[(i/3)%3];
		uInImgCropStartY = uInImgCropStartYList[i%3];
		uInImgCropHsz = uInImgCropHszList[(i/3)%3];
		uInImgCropVsz = uInImgCropVszList[i%3];

		eBppInFmt = eImgFmtList[i];
		
		eRotFlip = eRotFlipList[i%8];		
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler
			uInFbAddr, eBppInFmt,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);			
		FIMC_StopProcessing(eCh);
		
		FIMC_InitIpForDinFout(eCh,
			uInImgHsz, uInImgVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uInFbAddr, eBppInFmt, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			eRotFlip, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n InImg order num = %d \n", i);
		Disp(" Hit any key to go to the next step! \n");	
		UART_Getc();
	}

}

// [Test DMA-in FIFO-out for MFC multi-frame]
static void TestDinFoutForMfcMultiFrame(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;

	u32 uInImgHSz=320;
	u32 uInImgVSz=240;

	u32 uYAddr[71], uCAddr[71];
	u32 uYSize, uCbSize, uCrSize;
	for(i=0; i<71; i++)
	{
		FIMC_GetYCbCrSize(TILE_NV12, uInImgHSz, uInImgVSz, &uYSize, &uCbSize, &uCrSize);
	
		uYAddr[i] = CODEC_MEM_ST + (uYSize*i) + (uCbSize*i);
		uCAddr[i] = CODEC_MEM_ST + (uYSize*i) + (uCbSize*i) + uYSize;
	}

	// How to? d.load "V:\Components\multimedia\fimc\TILE_NV12_320x240_71frames" 0x21000000
	Disp("\n Load 'TILE_NV12_320x240_71frames' file into '0x%x' address using T32, \n press any key to go next! \n", CODEC_MEM_ST);
	UART_Getc();

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = uLcdHsz;
	uDispVSz = uLcdVsz;
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	InitVariables();

	FIMC_InitIpForDinFout(eCh,
		uInImgHSz, uInImgVSz, 0,
		0, 0, uInImgHSz, uInImgVSz,
		CODEC_MEM_ST, TILE_NV12, false,
		uDispHSz, uDispVSz, // Fifo out display size
		//eLocalInCS, PROGRESSIVE, ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
		eLocalInCS, PROGRESSIVE, ROT_90_FLIP_XY, eCscEquation, eYCbCrRange);
	
	while(GetKey() == 0)
	{
		FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[0], uCAddr[0]);
		Disp("\n Y[0] = 0x%x, C[0] = 0x%x \n", uYAddr[0], uCAddr[0]);
			
		FIMC_StartProcessingForMfc(eCh);

		//UART_Getc();

		for(i=1; i<71; i++)
		{
			FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[i], uCAddr[i]);		
			Disp("\n Y[%d] = 0x%x, C[%d] = 0x%x \n", i, uYAddr[i], i, uCAddr[i]);
			//Delay(0x1500);
			Delay(1000);
			//UART_Getc();
		}

		FIMC_StopProcessing(eCh);
	}
}

// [Test DMA-in FIFO-out for MFC interlaced multi-frame]
static void TestDinFoutForMfcInterlacedMultiFrame(void)
{
	// MEM-(interlaced weaving image) -> (weave in)-FIMC-(interlaced out) -> IPC -> FIMD -> LCDC

	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};
	
	u32 i, uLoopCnt;
	u32 uDispHSz, uDispVSz;

	u32 uInImgHSz=720;
	u32 uInImgVSz=480;

	u32 uYAddr[16], uCAddr[16];
	u32 uYSize, uCbSize, uCrSize;
	for(i=0; i<16; i++)
	{
		FIMC_GetYCbCrSize(TILE_NV12, uInImgHSz, uInImgVSz, &uYSize, &uCbSize, &uCrSize);
	
		uYAddr[i] = CODEC_MEM_ST + (uYSize*i) + (uCbSize*i);
		uCAddr[i] = CODEC_MEM_ST + (uYSize*i) + (uCbSize*i) + uYSize;
	}

	// How to? d.load "V:\Components\multimedia\fimc\TILE_NV12_720x480_16frames_Interlaced" 0x21000000
	Disp("\n Load 'TILE_NV12_720x480_16frames_Interlaced' file into '0x%x' address using T32, \n press any key to go next! \n", CODEC_MEM_ST);
	UART_Getc();

	Assert(eCh == FIMC_B);

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = uLcdHsz;
	uDispVSz = uLcdVsz;
	switch(eCh)
	{
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

#if 1
	// Initialize IPC v1.0
	IPC_InitIp(uDispHSz, uDispVSz/2, IPC_2D);
	IPC_On();	
#endif

	InitVariables();

#if 1
	for(uLoopCnt=0; uLoopCnt<8; uLoopCnt++)
	{
		FIMC_InitIpForDinFout(eCh,
			uInImgHSz, uInImgVSz, 0,
			0, 0, uInImgHSz, uInImgVSz,
			CODEC_MEM_ST, TILE_NV12_IW, false, // weave in
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, INTERLACED, eRotFlipList[uLoopCnt], eCscEquation, eYCbCrRange); // interlace out

		while(GetKey() == 0)
		{
			FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[0], uCAddr[0]);
			Disp("\n Y[0] = 0x%x, C[0] = 0x%x \n", uYAddr[0], uCAddr[0]);

			bFimcDone = false;
			FIMC_StartProcessingForMfc(eCh);	
			
			for(i=1; i<16; i++)
			{
				WaitForFimcDone(eCh);
				bFimcDone = false;
				
				FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[i], uCAddr[i]);
				Disp("\n Y[%d] = 0x%x, C[%d] = 0x%x \n", i, uYAddr[i], i, uCAddr[i]);		
			}

			FIMC_StopProcessing(eCh);
		}
	}
#else // Temp for IPC operation
	FIMC_InitIpForDinFout(eCh,
		uInImgHSz, uInImgVSz, 0,
		0, 0, uInImgHSz, uInImgVSz,
		CODEC_MEM_ST, TILE_NV12, false,
		uDispHSz, uDispVSz, // Fifo out display size
		eLocalInCS, INTERLACED, ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	while(GetKey() == 0)
	{
		FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[0], uCAddr[0]);
		Disp("\n Y[0] = 0x%x, C[0] = 0x%x \n", uYAddr[0], uCAddr[0]);

		bFimcDone = false;
		FIMC_StartProcessingForMfc(eCh);	
		
		for(i=1; i<16; i++)
		{
			WaitForFimcDone(eCh);
			bFimcDone = false;
			
			FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[i], uCAddr[i]);
			Disp("\n Y[%d] = 0x%x, C[%d] = 0x%x \n", i, uYAddr[i], i, uCAddr[i]);		
		}

		FIMC_StopProcessing(eCh);
	}
#endif

#if 1
	IPC_Off();	
#endif	
}

// [Test DMA-in FIFO-out for input DMA line skip]
static void TestDinFoutForInDmaLineSkip(void)
{
	u32 i, uLineSkipVal;
	u32 uDispHSz, uDispVSz;

	IMG_FMT eImgFmtList[11] = 
	{	NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};	

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 

	for(i=0; i<11; i++)
	{
		//for(uLineSkipVal=0; uLineSkipVal<16; uLineSkipVal++)
		for(uLineSkipVal=0; uLineSkipVal<4; uLineSkipVal++)
		{
			Disp("\n ImgFmtOrder = %d, LineSkipValue = %d \n", i, uLineSkipVal);

			// Initialize LCDC v6.0
			LCDC_Stop();
			LCDC_SetAllWinOnOff(0);
			uDispHSz = uLcdHsz;
			uDispVSz = uLcdVsz/(uLineSkipVal+1);
			switch(eCh)
			{
				case FIMC_A:					
					LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 0, 0);
					LCDC_SetWinOnOff(1, WIN0);
					break;
				case FIMC_B:
					LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
					LCDC_SetWinOnOff(1, WIN1);			
					break;
				case FIMC_C:
					LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 0, 0);
					LCDC_SetWinOnOff(1, WIN2);			
					break;
				default:
					Assert(0);
					break;			
			}
			LCDC_EnableAllWinFifoLvlInt(EMPTY);
			LCDC_Start();

			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				0, 0, uLcdHsz, uLcdVsz,
				uLcdFbAddr, eBpp,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0, // Scaler	
				uInFbAddr, eImgFmtList[i],		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);
			
			FIMC_InitIpForDinFout(eCh,
				uLcdHsz, uLcdVsz, uLineSkipVal,
				0, 0, uLcdHsz, uLcdVsz/(uLineSkipVal+1),
				uInFbAddr, eImgFmtList[i], false,
				uDispHSz, uDispVSz, // Fifo out display size
				eLocalInCS, PROGRESSIVE,
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			Disp("\n Press any key to stop! \n");
			UART_Getc();
			FIMC_StopProcessing(eCh);
			
			Disp("\n Hit any key to go next! \n");	
			UART_Getc();	
		}
	}
	
}

// [Test DMA-in FIFO-out for interlace out with IPC]
static void TestDinFoutForInterlaceOutWithIpc(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;
	u32 uInImgCropHszList[3];
	u32 uInImgCropVszList[3];
	u32 uInImgCropStartXList[3]={16,32,48};
	u32 uInImgCropStartYList[3]={48,56,80};
	u32 uInImgHVszList[8][2];

	IMG_FMT eImgFmtList[11] = 
	{
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};

	ROT_FLIP eRotFlipList[8] = 
	{
		ROT_NO_FLIP_NO,
		ROT_NO_FLIP_X,
		ROT_NO_FLIP_Y,
		ROT_NO_FLIP_XY,
		ROT_90_FLIP_NO,
		ROT_90_FLIP_X,
		ROT_90_FLIP_Y,
		ROT_90_FLIP_XY
	};

	Assert(eCh == FIMC_B);

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_B:
			LCDC_InitWinForFifoIn4(LOCAL_IF_YCBCR444, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Initialize IPC v1.0
	IPC_InitIp(uDispHSz, uDispVSz/2, IPC_2D);
	IPC_On();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);
	
	// [Composition]
	//-------------
	Disp("\n Composition with IPC \n");	

	InitVariables();

	switch(eCh)
	{
		case FIMC_B:
			uInImgHVszList[0][0]=800; uInImgHVszList[0][1]=480;
			uInImgHVszList[1][0]=1280; uInImgHVszList[1][1]=720;
			uInImgHVszList[2][0]=1920; uInImgHVszList[2][1]=1080;
			uInImgHVszList[3][0]=2048; uInImgHVszList[3][1]=1536;
			uInImgHVszList[4][0]=2560; uInImgHVszList[4][1]=2048;
			uInImgHVszList[5][0]=3264; uInImgHVszList[5][1]=2448; // 8M
			uInImgHVszList[6][0]=4032; uInImgHVszList[6][1]=3008; // 12M
			uInImgHVszList[7][0]=4224; uInImgHVszList[7][1]=4224;		
			break;
		default:
			Assert(0);
			break;
	}

	uInImgCropHszList[0]=(640-(16*2))-((640-(16*2))%16);
	uInImgCropHszList[1]=(640-(48*2))-((640-(48*2))%16);
	uInImgCropHszList[2]=(640-(80*2))-((640-(80*2))%16);

	uInImgCropVszList[0]=(480-(24*2))-((480-(24*2))%16);
	uInImgCropVszList[1]=(480-(56*2))-((480-(56*2))%16);
	uInImgCropVszList[2]=(480-(80*2))-((480-(80*2))%16);

	for(i=0; i<11; i++)
	{
		uInImgHsz = uInImgHVszList[i%8][0];
		uInImgVsz = uInImgHVszList[i%8][1];
		
		uInImgCropStartX = uInImgCropStartXList[(i/3)%3];
		uInImgCropStartY = uInImgCropStartYList[i%3];
		uInImgCropHsz = uInImgCropHszList[(i/3)%3];
		uInImgCropVsz = uInImgCropVszList[i%3];
		eBppInFmt = eImgFmtList[i];
		
		eRotFlip = eRotFlipList[i%8];		
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler
			uInFbAddr, eBppInFmt,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);			
		FIMC_StopProcessing(eCh);
		
		FIMC_InitIpForDinFout(eCh,
			uInImgHsz, uInImgVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uInFbAddr, eBppInFmt, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			LOCAL_IF_YCBCR444, INTERLACED,
			eRotFlip, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n InImg order num = %d \n", i);
		Disp(" Hit any key to go to the next step! \n");	
		UART_Getc();
	}

	IPC_Off();
}


// [Test DMA-in FIFO-out for forced off]
static void TestDinFoutForForcedOff(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;
	u8 bEnable;

	// FIMD start
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start(); 

	InitVariables();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	FIMC_InitIpForDinFout(eCh,
		uInImgHsz, uInImgVsz, 0,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdFbAddr, eBpp, false,
		uDispHSz, uDispVSz, // Fifo out display size
		eLocalInCS, PROGRESSIVE,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	//for(i=0; i<1000; i++)
	for(i=0; i<10; i++)
	{
		Disp("\n loop count = %d \n", i);
		// FIMD window start
		//Disp("\n Press any key to start FIMD window and local path! \n");
		//UART_Getc();
		switch(eCh) 
		{
			case FIMC_A:
				LCDC_SetLocalEn(LOCAL_EN, WIN0);
				LCDC_SetWinOnOff(1, WIN0);
				break;
			case FIMC_B:
				LCDC_SetLocalEn(LOCAL_EN, WIN1);	
				LCDC_SetWinOnOff(1, WIN1);
				break;
			case FIMC_C:
				LCDC_SetLocalEn(LOCAL_EN, WIN2);	
				LCDC_SetWinOnOff(1, WIN2);
				break;
			default:
				Assert(0);
				break;			
		}

		// FIMC start
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0); 

		Delay(i*1000+1000);

		// FIMD window & local path stop
		//Disp("\n Press any key to stop FIMD window and local path! \n");
		//UART_Getc();
		switch(eCh) 
		{
			case FIMC_A:
				LCDC_SetLocalEn(LOCAL_DIS, WIN0);
				break;
			case FIMC_B:
				LCDC_SetLocalEn(LOCAL_DIS, WIN1);	
				break;
			case FIMC_C:
				LCDC_SetLocalEn(LOCAL_DIS, WIN2);	
				break;
			default:
				Assert(0);
				break;			
		}
		
#if 0
		// FIMC stop
		Disp("\n Press any key to stop FIMC! \n");
		UART_Getc();	
		FIMC_StopProcessing(eCh); 
#else
		Delay(i*1000+1000);

		// FIMC stop
		//Disp("\n Press any key to stop FIMC! \n");
		//UART_Getc();
		do
		{
			FIMC_GetLcdcEnstatus(eCh, &bEnable);
		}
		while(bEnable);
		
		Delay(i*1000+1000);
		
		FIMC_SetDinEnvid(eCh, false); // ENVID forced off	
		
		Delay(i*1000+1000);
		
		FIMC_StopProcessing(eCh);
#endif
	}
	
}

// [Test DMA-in FIFO-out for another forced off]
static void TestDinFoutForAnotherForcedOff(void)
{
	u32 uDispHSz, uDispVSz;
	u8 bEnable;

	// FIMD start
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start(); 

	InitVariables();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	FIMC_InitIpForDinFout(eCh,
		uInImgHsz, uInImgVsz, 0,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdFbAddr, eBpp, false,
		uDispHSz, uDispVSz, // Fifo out display size
		eLocalInCS, PROGRESSIVE,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	for(uLoopCntForBackPorch=0; uLoopCntForBackPorch<10; uLoopCntForBackPorch++)
	{
		Disp("\n loop count = %d \n", uLoopCntForBackPorch);
		// FIMD window start
		//Disp("\n Press any key to start FIMD window and local path! \n");
		//UART_Getc();
		switch(eCh) 
		{
			case FIMC_A:
				LCDC_SetLocalEn(LOCAL_EN, WIN0);
				LCDC_SetWinOnOff(1, WIN0);
				break;
			case FIMC_B:
				LCDC_SetLocalEn(LOCAL_EN, WIN1);	
				LCDC_SetWinOnOff(1, WIN1);
				break;
			case FIMC_C:
				LCDC_SetLocalEn(LOCAL_EN, WIN2);	
				LCDC_SetWinOnOff(1, WIN2);
				break;
			default:
				Assert(0);
				break;			
		}

		// FIMC start
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0); 

		Delay(uLoopCntForBackPorch*1000+1000);

		// FIMD window & local path stop
		//Disp("\n Press any key to stop FIMD window and local path! \n");
		//UART_Getc();
		switch(eCh) 
		{
			case FIMC_A:
				LCDC_SetLocalEn(LOCAL_DIS, WIN0);
				break;
			case FIMC_B:
				LCDC_SetLocalEn(LOCAL_DIS, WIN1);	
				break;
			case FIMC_C:
				LCDC_SetLocalEn(LOCAL_DIS, WIN2);	
				break;
			default:
				Assert(0);
				break;			
		}

		Delay(uLoopCntForBackPorch*1000+1000);

		bFimdDoneForBackPorch = false;
		//INTC_Init();
		INTC_SetVectAddr(NUM_LCD1, Isr_FimdForBackPorch);
		INTC_Enable(NUM_LCD1);
		LCDC_EnableAllWinFrameInt(FINT_BACKPORCH);

		while(!bFimdDoneForBackPorch);
	}
	
}

// [Test DMA-in FIFO-out for checking c100 problem]
static void TestDinFoutForCheckingC100Problem(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;

	IMG_FMT eInImgFmt[3] = {NV12, YC420, TILE_NV12};
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, 528, eBpp);
	GLIB_DrawPattern(uLcdHsz, 528);

	for(i=0; i<3; i++)
	{
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, 528, 0,
			0, 0, uLcdHsz, 528,
			uLcdFbAddr,	eBpp,
			uLcdHsz, 528,
			0, 0, uLcdHsz, 528, 0, // Scaler		
			uInFbAddr, eInImgFmt[i],		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);		
		FIMC_StopProcessing(eCh);
		
		//--------------------------------------------
		
		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, 528, 0,
			0, 0, uLcdHsz, 528,
			uInFbAddr, eInImgFmt[i], false,
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_90_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();	
	}
}

// [Select ITU camera A]
static void SelItuCameraA(void)
{
	eCamera = ITU_CAMERA_A;		
	Disp("\n ITU_CAMERA_A selectd! \n");
}

// [Select ITU camera B]
static void SelItuCameraB(void)
{
	eCamera = ITU_CAMERA_B;		
	Disp("\n ITU_CAMERA_B selectd! \n");
}

// [Test FIFO-in DMA-out for basic function]
static void TestFinDoutForBasicFunc(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
	
	FIMC_InitIpForFinDout(eCh, eCamera,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

	Disp(" Press any key to capture image!\n");

	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		Disp("\n Current frame count is %d\n", uFrmCnt);
	}

	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
}

// [Test FIFO-in DMA-out for each function]
static void TestFinDoutForEachFunc(void)
{
	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;

	IMG_FMT eInImgTypeList[4] = 
	{
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY
	};

	IMG_FMT eImgFmtList[11] = 
	{	
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};	

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
	
	// [Image cropping]
	//----------------
	Disp("\n Image cropping \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	
	uInImgCropHsz = (uFinImgHsz/2)-((uFinImgHsz/2)%16);
	uInImgCropVsz = (uFinImgVsz/2)-((uFinImgVsz/2)%16);	
	uOutImgHsz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uOutImgVsz = (uLcdVsz/2)-((uLcdVsz/2)%16);
	
	FIMC_InitIpForFinDout(eCh, eCamera,
		0, 0, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uOutImgHsz, uOutImgVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
	
	for(i=0; i<4; i++)
	{
		switch(i)
		{
			case 0:
				uInImgCropStartX = 0;
				uInImgCropStartY = 0;
				uOutImgStartX = 0;
				uOutImgStartY = 0;
				break;
			case 1:
				uInImgCropStartX = uFinImgHsz/2;
				uInImgCropStartY = 0;
				uOutImgStartX = uLcdHsz/2;
				uOutImgStartY = 0;
				break;
			case 2:
				uInImgCropStartX = 0;
				uInImgCropStartY = uFinImgVsz/2;
				uOutImgStartX = 0;
				uOutImgStartY = uLcdVsz/2;
				break;
			case 3:
				uInImgCropStartX = uFinImgHsz/2;
				uInImgCropStartY = uFinImgVsz/2;
				uOutImgStartX = uLcdHsz/2;
				uOutImgStartY = uLcdVsz/2;
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_SetFinImgCroppingRegion(eCh,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz);
		FIMC_SetDoutImgOffset(eCh, eBpp, uOutImgStartX, uOutImgStartY);

		if(i==0)
		{
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		}
				
		Disp("\n Order = %d \n", i);
		UART_Getc();
	}
	
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
	UART_Getc();

	// [Image scaling]
	//---------------
	Disp("\n Image scaling \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	
	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region	
	uInImgCropVsz = uFinImgVsz; // Set cropping region	

	FIMC_InitIpForFinDout(eCh, eCamera,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);	

	for(i=0; i<4; i++)
	{
		switch(i)
		{
			case 0:
				uOutImgHsz = 96;
				uOutImgVsz = 96;
				break;
			case 1:
				uOutImgHsz = 144;
				uOutImgVsz = 144;
				break;
			case 2:
				uOutImgHsz = 192;
				uOutImgVsz = 192;
				break;
			case 3:
				uOutImgHsz = (uLcdHsz/1)-((uLcdHsz/1)%16);
				uOutImgVsz = (uLcdVsz/1)-((uLcdVsz/1)%16);
				break;
			default:
				Assert(0);
				break;
		}		

		FIMC_SetFinImgScaler(eCh,uInImgCropHsz, uInImgCropVsz, uOutImgHsz, uOutImgVsz);
		FIMC_SetDoutImgSize(eCh,uOutImgHsz, uOutImgVsz, eBpp, 0);

		FIMC_SetDoutImgOffset(eCh, eBpp, 0, 0);

		if(i==0)
		{
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		}
		
		Disp("\n Order = %d \n", i);
		UART_Getc();
	}

	for(i=0; i<9; i++)
	{
		switch(i)
		{
			case 0:
				uInImgCropStartX = 80;
				uInImgCropStartY = 60;
				uInImgCropHsz = (uFinImgHsz-(80*2))-((uFinImgHsz-(80*2))%16);
				uInImgCropVsz = (uFinImgVsz-(60*2))-((uFinImgVsz-(60*2))%16);
				break;
			case 1:
				uInImgCropStartX = 160;
				uInImgCropStartY = 120;
				uInImgCropHsz = (uFinImgHsz-(160*2))-((uFinImgHsz-(160*2))%16);
				uInImgCropVsz = (uFinImgVsz-(120*2))-((uFinImgVsz-(120*2))%16);
				break;
			case 2:
				uInImgCropStartX = 240;
				uInImgCropStartY = 180;
				uInImgCropHsz = (uFinImgHsz-(240*2))-((uFinImgHsz-(240*2))%16);
				uInImgCropVsz = (uFinImgVsz-(180*2))-((uFinImgVsz-(180*2))%16);
				break;
			case 3:
				uInImgCropStartX = 320;
				uInImgCropStartY = 240;
				uInImgCropHsz = (uFinImgHsz-(320*2))-((uFinImgHsz-(320*2))%16);
				uInImgCropVsz = (uFinImgVsz-(240*2))-((uFinImgVsz-(240*2))%16);
				break;
			case 4:
				uInImgCropStartX = 400;
				uInImgCropStartY = 300;
				uInImgCropHsz = (uFinImgHsz-(400*2))-((uFinImgHsz-(400*2))%16);
				uInImgCropVsz = (uFinImgVsz-(300*2))-((uFinImgVsz-(300*2))%16);
				break;
			case 5:
				uInImgCropStartX = 480;
				uInImgCropStartY = 360;
				uInImgCropHsz = (uFinImgHsz-(480*2))-((uFinImgHsz-(480*2))%16);
				uInImgCropVsz = (uFinImgVsz-(360*2))-((uFinImgVsz-(360*2))%16);
				break;
			case 6:
				uInImgCropStartX = 560;
				uInImgCropStartY = 420;
				uInImgCropHsz = (uFinImgHsz-(560*2))-((uFinImgHsz-(560*2))%16);
				uInImgCropVsz = (uFinImgVsz-(420*2))-((uFinImgVsz-(420*2))%16);
				break;
			case 7:
				uInImgCropStartX = 640;
				uInImgCropStartY = 480;
				uInImgCropHsz = (uFinImgHsz-(640*2))-((uFinImgHsz-(640*2))%16);
				uInImgCropVsz = (uFinImgVsz-(480*2))-((uFinImgVsz-(480*2))%16);
				break;
			case 8:
				uInImgCropStartX = 720;
				uInImgCropStartY = 540;
				uInImgCropHsz = (uFinImgHsz-(720*2))-((uFinImgHsz-(720*2))%16);
				uInImgCropVsz = (uFinImgVsz-(540*2))-((uFinImgVsz-(540*2))%16);
				break;
			default:
				Assert(0);
				break;
		}
		
		FIMC_SetFinImgCroppingRegion(eCh,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz);
		FIMC_SetFinImgScaler(eCh,uInImgCropHsz, uInImgCropVsz, uOutImgHsz, uOutImgVsz);
		
		Disp("\n Order = %d \n", i+4);
		UART_Getc();
	}
	
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
	UART_Getc();

	// [Rotation(Output rotator) and Flip]
	//------------------------------
	Disp("\n Rotation(Output rotator) and Flip \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	
	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region

	if(uLcdHsz >= uLcdVsz)
	{
		uOutImgHsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
		uOutImgVsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
	}
	else // uLcdHsz < uLcdVsz
	{
		uOutImgHsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
		uOutImgVsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
	}
	
	FIMC_InitIpForFinDout(eCh, eCamera,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uOutImgHsz, uOutImgVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	for(i=0; i<8; i++)
	{
		memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		switch(i)
		{
			case 0:
				eRotFlip = ROT_NO_FLIP_NO;
				break;
			case 1:
				eRotFlip = ROT_NO_FLIP_X;
				break;
			case 2:
				eRotFlip = ROT_NO_FLIP_Y;
				break;
			case 3:
				eRotFlip = ROT_NO_FLIP_XY;
				break;
			case 4:
				eRotFlip = ROT_90_FLIP_NO;
				break;
			case 5:
				eRotFlip = ROT_90_FLIP_X;
				break;
			case 6:
				eRotFlip = ROT_90_FLIP_Y;
				break;
			case 7:
				eRotFlip = ROT_90_FLIP_XY;
				break;
			default:
				Assert(0);
				break;
		}
		FIMC_SetOutRotAndFlip(eCh, eRotFlip);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		
		Disp("\n Order = %d \n", i);
		UART_Getc();	
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
	}
	UART_Getc();

	// [Image effect]
	//--------------
	Disp("\n Image effect \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	
	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region

	FIMC_InitIpForFinDout(eCh, eCamera,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	for(i=0; i<5; i++)
	{
		switch(i)
		{
			case 0:
				eImgEff = EFF_BYPASS;
				break;
			case 1:
				eImgEff = NEGATIVE;
				break;
			case 2:
				eImgEff = ART_FREEZE;
				break;
			case 3:
				eImgEff = EMBOSSING;
				break;
			case 4:
				eImgEff = SILHOUETTE;
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);
		
		if(i == 0)
		{
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		}
		
		UART_Getc();
	}	
		
	for(i=0; i<5; i++)
	{
		eImgEff = ARBITRARY_CBCR;

		switch(i)
		{
			case 0:
				uCbVal = 1;
				uCrVal = 1;
				break;
			case 1:
				uCbVal = 128;
				uCrVal = 128;				
				break;
			case 2:
				uCbVal = 254;
				uCrVal = 254;				
				break;
			case 3:
				uCbVal = 1;
				uCrVal = 254;				
				break;
			case 4:
				uCbVal = 254;
				uCrVal = 1; 			
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);			
		UART_Getc();
	}
	
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
	UART_Getc();	

	// [Memory store type and Color format conversion]
	//--------------------------------------
	Disp("\n Memory store type and Color format conversion \n");

	InitVariables();

	for(j=0; j<4; j++)
	{
		Disp("\n Sensor initialization! please wait.......... \n");
		FIMC_InitCamera(eCisModel, eInImgTypeList[j], UXGA); // UXGA:1600x1200	
		FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
		Delay(20000);		
		
		Disp("\n FIFO input image type = %d \n", j);
		Disp("\n Press any to go to the next step!\n");
		UART_Getc();
			
		for(i=0; i<11; i++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();

			memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
			
			// Set cropping region
			uInImgCropStartX = 0;
			uInImgCropStartY = 0;
			uInImgCropHsz = uFinImgHsz;
			uInImgCropVsz = uFinImgVsz;
			
			eBppOutFmt = eImgFmtList[i];			

			FIMC_InitIpForFinDout(eCh, eCamera,
				uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
				uLcdHsz, uLcdVsz,
				0, 0,
				uLcdHsz, uLcdVsz, 0, // Scaler
				uLcdFbAddr, uLcdFbAddr, eBppOutFmt,
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");
			UART_Getc();

			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);
			
			UART_Getc();

			// Set cropping region
			uInImgCropStartX = 0;
			uInImgCropStartY = 0;
			uInImgCropHsz = uLcdHsz;
			uInImgCropVsz = uLcdVsz;	
			eBppInFmt = eImgFmtList[i];			

			memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
			
			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
				uLcdFbAddr,	eBppInFmt,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
			WaitForFimcDone(eCh);				
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n eBppOutFmt order num = %d \n", i);
			Disp("\n Hit any key to go to the next step! \n");
			UART_Getc();
		}
	}

	// [Capture frame control]
	//---------------------	
	Disp("\n Capture frame control \n");

	InitVariables();

	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region
	
	// Set capture frame control
	bCptFrmCtrl = true;
	uCptSeqPattern = 0x10101010, uCptSeqPointer = 0;	

	FIMC_InitIpForFinDout(eCh, eCamera,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, bCptFrmCtrl, uCptSeqPattern, uCptSeqPointer, false, 0);

	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);
#ifndef FIMC_POLLING
	while(!bFimcDone);
#else // polling
	FIMC_WaitForProcessingDoneForCptFrmCtrl(eCh);
#endif
	FIMC_StopProcessing(eCh);

	UART_Getc();

	// Set capture count control
	bCptCntCtrl = true;
	uCptCnt = 10;

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	FIMC_InitIpForFinDout(eCh, eCamera,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, bCptFrmCtrl, uCptSeqPattern, uCptSeqPointer, bCptCntCtrl, uCptCnt);

	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);

#ifndef FIMC_POLLING
	while(!bFimcDone);
#else // polling
	FIMC_WaitForProcessingDoneForCptFrmCtrl(eCh);
#endif
	FIMC_StopProcessing(eCh);	
}

// [Test FIFO-in DMA-out for composition]
static void TestFinDoutForComposition(void)
{
	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uInImgCropHszList[3];
	u32 uInImgCropVszList[3];
	u32 uInImgCropStartXList[3]={48,232,472};
	u32 uInImgCropStartYList[3]={16,152,392};
	
	IMG_FMT eInImgTypeList[4] = {YCBYCR, YCRYCB, CRYCBY, CBYCRY};

	IMG_FMT eImgFmtList[11] = 
	{
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};
	
	ROT_FLIP eRotFlipList[8] = 
	{
		ROT_NO_FLIP_NO,
		ROT_NO_FLIP_X,
		ROT_NO_FLIP_Y,
		ROT_NO_FLIP_XY,
		ROT_90_FLIP_NO,
		ROT_90_FLIP_X,
		ROT_90_FLIP_Y,
		ROT_90_FLIP_XY
	};

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	
	// [Composition] 
	//-------------
	Disp("\n Composition \n");

	InitVariables();
	
	for(j=0; j<4; j++)
	{
		Disp("\n Sensor initialization! please wait..........\n");
		FIMC_InitCamera(eCisModel, eInImgTypeList[j], UXGA); // UXGA:1600x1200		
		FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz); 
		Delay(20000);
		
		uInImgCropHszList[0] = (uFinImgHsz-(48*2))-((uFinImgHsz-(48*2))%16);
		uInImgCropHszList[1] = (uFinImgHsz-(232*2))-((uFinImgHsz-(232*2))%16);
		uInImgCropHszList[2] = (uFinImgHsz-(472*2))-((uFinImgHsz-(472*2))%16);

		uInImgCropVszList[0] = (uFinImgVsz-(16*2))-((uFinImgVsz-(16*2))%16);
		uInImgCropVszList[1] = (uFinImgVsz-(152*2))-((uFinImgVsz-(152*2))%16);
		uInImgCropVszList[2] = (uFinImgVsz-(392*2))-((uFinImgVsz-(392*2))%16);
				
		Disp("\n FIFO input image type = %d \n", j);
		Disp("\n Press any to go to the next step!\n");
		UART_Getc();

		for(i=0; i<11; i++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();

			memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

			// Set cropping region
			uInImgCropStartX = uInImgCropStartXList[(i/3)%3];
			uInImgCropStartY = uInImgCropStartYList[i%3];
			uInImgCropHsz = uInImgCropHszList[(i/3)%3];
			uInImgCropVsz = uInImgCropVszList[i%3];
			uOutImgStartX=16;
			uOutImgStartY=16;

			if(uLcdHsz >= uLcdVsz)
			{
				uOutImgHsz = uLcdVsz-(uLcdVsz/8)-((uLcdVsz-(uLcdVsz/8))%16);
				uOutImgVsz = uLcdVsz-(uLcdVsz/8)-((uLcdVsz-(uLcdVsz/8))%16);
			}
			else // uLcdHsz < uLcdVsz
			{
				uOutImgHsz = uLcdHsz-(uLcdHsz/8)-((uLcdHsz-(uLcdHsz/8))%16);
				uOutImgVsz = uLcdHsz-(uLcdHsz/8)-((uLcdHsz-(uLcdHsz/8))%16);
			}
			
			eBppOutFmt = eImgFmtList[i];
			eRotFlip = eRotFlipList[i%8];			

			FIMC_InitIpForFinDout(eCh, eCamera,
				uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
				uLcdHsz, uLcdVsz,
				uOutImgStartX, uOutImgStartY,
				uOutImgHsz, uOutImgVsz, 0, // Scaler
				uLcdFbAddr, uLcdFbAddr, eBppOutFmt,
				eRotFlip, eCscEquation, eYCbCrRange);

			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");
			UART_Getc();

			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);
			
			UART_Getc();

			// Set cropping region
			memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
			
			uInImgCropStartX = 16;
			uInImgCropStartY = 16;
			switch(eRotFlip)
			{
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					//uInImgCropHsz = 176, uInImgCropVsz = 208;
					//uOutImgHsz=176, uOutImgVsz=208;
					uInImgCropHsz = uOutImgVsz, uInImgCropVsz = uOutImgHsz;
					uOutImgHsz=uOutImgVsz, uOutImgVsz=uOutImgHsz;					
					break;
				default:
					//uInImgCropHsz = 208, uInImgCropVsz = 176;
					uInImgCropHsz = uOutImgHsz, uInImgCropVsz = uOutImgVsz;
					break;
			}		

			eBppInFmt = eImgFmtList[i];
			eBppOutFmt = RGB24;
			eRotFlip = ROT_NO_FLIP_NO;			
			
			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
				uLcdFbAddr,	eBppInFmt,
				uLcdHsz, uLcdVsz,
				uOutImgStartX, uOutImgStartY, uOutImgHsz, uOutImgVsz, 0, // Scaler		
				uLcdOutFbAddr, eBppOutFmt,		
				eRotFlip, eCscEquation, eYCbCrRange, false);

			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
			WaitForFimcDone(eCh);				
			FIMC_StopProcessing(eCh);
			
			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n eBppOutFmt order num = %d \n", i);
			Disp("\n Hit any key to go to the next step! \n");
			UART_Getc();			
		}
	}

}

// [Test FIFO-in DMA-out for test pattern ]
static void TestFinDoutForTestPattern(void)
{
	u32 i;
	
	TEST_PATTERN eTestPatternList[4] = {EXTERNAL_CAMERA_INPUT, COLOR_BAR,
		HORIZONTAL_INCREMENT, VERTICAL_INCREMENT};

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	// [Test pattern] 
	//-------------
	Disp("\n Test pattern \n");	

	Disp("\n Sensor initialization! please wait..........\n");
	FIMC_InitCameraForTestPattern(eCisModel, YCBYCR);
	Delay(20000);
	Disp("\n Press any to go to the next step!\n");
	UART_Getc();		
	
	// Set cropping region
	uInImgCropStartX = 0, uInImgCropStartY = 0;
	uInImgCropHsz = 640, uInImgCropVsz = 480;

	if((uLcdHsz<=640)&&(uLcdVsz<=480))
	{
		uOutImgHsz = uLcdHsz;
		uOutImgVsz = uLcdVsz;
	}
	else
	{
		//uOutImgHsz = 640;
		uOutImgHsz = 480;
		uOutImgVsz = 480;
	}
	
	for(i=1; i<4; i++)
	{		
		eTestPattern = eTestPatternList[i];
		
		memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		FIMC_InitIpForFinDout(eCh, eCamera,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdHsz, uLcdVsz,
			0, 0,
			uOutImgHsz, uOutImgVsz, 0, // Scaler
			uLcdFbAddr, uLcdFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
		FIMC_SetTestPattern(eCh, eTestPattern);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");
		UART_Getc();

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
		
		UART_Getc();	
	}
	
	FIMC_SetTestPattern(eCh, EXTERNAL_CAMERA_INPUT);
}

// [Test FIFO-in DMA-out for double buffering]
static void TestFinDoutForDoubleBuffering(void)
{
	u32 uDoubleBufAddr;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;
	u32 uBufferIdx;
		
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, true); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	LCDC_GetDoubleBufAddr(&uDoubleBufAddr, 1, WIN0);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);
	memset((void *)uDoubleBufAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
	
	FIMC_InitIpForFinDout(eCh, eCamera,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uLcdFbAddr, uDoubleBufAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

	Disp(" Press any key to capture image!\n");

	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		uBufferIdx = uFrmCnt%2;
		LCDC_SetActiveFrameBuf(uBufferIdx, WIN0);
		Disp("\n Current frame count is %d\n", uFrmCnt);
	}

	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
}

// [Test FIFO-in DMA-out for scaler bypass]
static void TestFinDoutForScalerBypass(void)
{
#if 0
	IMG_FMT eInImgTypeList[4] = {YCBYCR, YCRYCB, CRYCBY, CBYCRY};
	IMG_FMT eOutImgTypeList[9] = {NV12, YC420, TILE_NV12,
		YCBYCR, YCRYCB, CRYCBY, CBYCRY, NV16, YC422};

	u32 i, j, k;
	u32 uFinImgHsz, uFinImgVsz;
	
	u32 uFrmCnt;
	
	u32 uDispHSz, uDispVSz;

	u32 RetOfCr, RetOfY0, RetOfCb, RetOfY1;

	// [Scaler bypass]
	//--------------
	Disp("\n Scaler bypass \n");	
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 64, 64);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 64, 64);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 64, 64);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	InitVariables();

	for(i=0; i<4; i++)
	{
		Disp("\n Sensor initialization! please wait..........\n");
		FIMC_InitCamera(eCisModel, eInImgTypeList[i], UXGA); // UXGA:1600x1200
		FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
		Delay(20000);
		
		Disp("\n Press any key to go next!\n");
		UART_Getc();

		for(j=0; j<9; j++)
		{
			FIMC_InitIpForFinDout(eCh, eCamera,
				0, 0, uFinImgHsz, uFinImgVsz,
				uFinImgHsz, uFinImgVsz,
				0, 0, uFinImgHsz, uFinImgVsz, 0,
				uInFbAddr, uInFbAddr, eOutImgTypeList[j],
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

			// ============ Notice !!! ============
			FIMC_StartProcessingForScalerBypass(eCh);
			// ================================
			
			Disp(" Press any key to capture image!\n");
			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}
			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);	

			//----------------

			FIMC_InitIpForDinFout(eCh,
				uFinImgHsz, uFinImgVsz, 0,
				0, 0, uFinImgHsz, uFinImgVsz,
				uInFbAddr, eOutImgTypeList[j], false,		
				uDispHSz, uDispVSz, // Fifo out display size
				eLocalInCS, PROGRESSIVE,
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
			Disp("\n Press any key to stop! \n");
			UART_Getc();
			FIMC_StopProcessing(eCh);
		}
	}

	// [Scaler bypass using golden data]
	//-----------------------------
	Disp("\n Scaler bypass using golden data \n");

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	Disp("\n Test pattern \n");
	
	FIMC_InitCameraForTestPattern(eCisModel, YCBYCR); // Test pattern generates ONLY 'YCBYCR(CISRCFMT[15:14]=00)'

	for( j=0; j<4; j++)
	{
		// Set cropping region
		uInImgCropStartX = 0, uInImgCropStartY = 0;
		uInImgCropHsz = 640, uInImgCropVsz = 480;

		if((uLcdHsz<=640)&&(uLcdVsz<=480))
		{
			uOutImgHsz = uLcdHsz;
			uOutImgVsz = uLcdVsz;
		}
		else
		{
			uOutImgHsz = 640;
			uOutImgVsz = 480;
		}
				
		memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		FIMC_InitIpForFinDout(eCh, eCamera,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uOutImgHsz, uOutImgVsz,
			0, 0,
			uOutImgHsz, uOutImgVsz, 0, // Scaler
			uLcdFbAddr, uLcdFbAddr, eInImgTypeList[j],
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
		FIMC_SetTestPattern(eCh, COLOR_BAR);

		bFimcDone = false;
		// ============ Notice !!! ============		
		FIMC_StartProcessingForScalerBypass(eCh);
		// ================================

		Disp(" Press any key to capture image and compare 'captured image' with 'golden image'! \n");
		UART_Getc();
		Disp(" Wait for a moment, comparing.......... \n");

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		// Compare values
		switch(eInImgTypeList[j])
		{
			case YCBYCR:
				for( k=0; k<153600; k++)
				{
					//Disp("\n Out image format: YCBYCR, k = %d \n", k);
					RetOfCr = Compare8(uLcdFbAddr+(4*k), (u32)(uCrOfColorBar+k), 1);
					RetOfY0 = Compare8(uLcdFbAddr+(4*k)+1, (u32)(uYOfColorBar+(2*k)), 1);
					RetOfCb = Compare8(uLcdFbAddr+(4*k)+2, (u32)(uCbOfColorBar+k), 1);
					RetOfY1 = Compare8(uLcdFbAddr+(4*k)+3, (u32)(uYOfColorBar+(2*k)+1), 1);
					Assert(RetOfCr); Assert(RetOfY0); Assert(RetOfCb);	Assert(RetOfY1);		
				}
				Disp("\n Out image format: YCBYCR: PASS \n");
				break;
			case YCRYCB:
				for( k=0; k<153600; k++)
				{
					//Disp("\n Out image format: YCRYCB, k = %d \n", k);
					RetOfCb = Compare8(uLcdFbAddr+(4*k), (u32)(uCbOfColorBar+k), 1);
					RetOfY0 = Compare8(uLcdFbAddr+(4*k)+1, (u32)(uYOfColorBar+(2*k)), 1);
					RetOfCr = Compare8(uLcdFbAddr+(4*k)+2, (u32)(uCrOfColorBar+k), 1);
					RetOfY1 = Compare8(uLcdFbAddr+(4*k)+3, (u32)(uYOfColorBar+(2*k)+1), 1);
					Assert(RetOfCr); Assert(RetOfY0); Assert(RetOfCb);	Assert(RetOfY1);				
				}
				Disp("\n Out image format: YCRYCB: PASS \n");
				break;
			case CRYCBY:
				for( k=0; k<153600; k++)
				{
					//Disp("\n Out image format: CRYCBY, k = %d \n", k);
					RetOfY0 = Compare8(uLcdFbAddr+(4*k), (u32)(uYOfColorBar+(2*k)), 1);
					RetOfCb = Compare8(uLcdFbAddr+(4*k)+1, (u32)(uCbOfColorBar+k), 1);
					RetOfY1 = Compare8(uLcdFbAddr+(4*k)+2, (u32)(uYOfColorBar+(2*k)+1), 1);
					RetOfCr = Compare8(uLcdFbAddr+(4*k)+3, (u32)(uCrOfColorBar+k), 1);
					Assert(RetOfCr); Assert(RetOfY0); Assert(RetOfCb);	Assert(RetOfY1);					
				}
				Disp("\n Out image format: CRYCBY: PASS \n");
				break;
			case CBYCRY:
				for( k=0; k<153600; k++)
				{
					//Disp("\n Out image format: CBYCRY, k = %d \n", k);
					RetOfY0 = Compare8(uLcdFbAddr+(4*k), (u32)(uYOfColorBar+(2*k)), 1);
					RetOfCr = Compare8(uLcdFbAddr+(4*k)+1, (u32)(uCrOfColorBar+k), 1);
					RetOfY1 = Compare8(uLcdFbAddr+(4*k)+2, (u32)(uYOfColorBar+(2*k)+1), 1);
					RetOfCb = Compare8(uLcdFbAddr+(4*k)+3, (u32)(uCbOfColorBar+k), 1);
					Assert(RetOfCr); Assert(RetOfY0); Assert(RetOfCb);	Assert(RetOfY1);					
				}
				Disp("\n Out image format: CBYCRY: PASS \n");
				break;
			default:
				Assert(0);
				break;
		}

		Disp("\n Press any key to go next! \n");
		UART_Getc();		
	}
		
	FIMC_SetTestPattern(eCh, EXTERNAL_CAMERA_INPUT);
#endif	
}

// [Test FIFO-in DMA-out for interlace bob out]
static void TestFinDoutForInterlaceBobOut(void)
{
	IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	for(i=0; i<4; i++)
	{
		for(j=0; j<8; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = uLcdHsz;
					uOutDispVsz = uLcdVsz/2;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uOutDispHsz = uLcdVsz/2; // Notice!
					uOutDispVsz = uLcdHsz; // Notice!
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;		
					break;
				default:
					Assert(0);
					break;
			}

			FIMC_InitIpForFinDout(eCh, eCamera,
				0, 0, uFinImgHsz, uFinImgVsz,
				uOutDispHsz, uOutDispVsz,
				0, 0, uOutImgHsz, uOutImgVsz, 0,
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt[i],
				eRotFlipList[j], eCscEquation, eYCbCrRange);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//--------------------------------------------

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					uInImgVsz = uLcdVsz/2;		
					
					//uInImgCropHsz = uLcdHsz; 
					uInImgCropHsz = uLcdHsz - (uLcdHsz%16); 
					
					uInImgCropVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uInImgHsz =  uLcdVsz/2;
					uInImgVsz = uLcdHsz;		

					//uInImgCropHsz =  uLcdVsz/2; 
					uInImgCropHsz =  (uLcdVsz/2) - ((uLcdVsz/2)%16);
					
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			

			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uLcdFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			// Odd field
#if 0			
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uLcdFbAddr, &uNextFbAddr);
#else		
			FIMC_GetNextFbAddr(eImgFmt, uInImgHsz, uInImgVsz, uLcdFbAddr, &uNextFbAddr);
#endif

			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uNextFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}
		
}

// [Test FIFO-in DMA-out for interlace weave out]
static void TestFinDoutForInterlaceWeaveOut(void)
{
	// MFC interlaced encoding scenario do NOT need ROTATION
	// CAMERA -> (progressive) -> FIMC -> (interlace, weave out) -> MFC -> MEM
	
	IMG_FMT eInterlaceImgFmt[4] = {RGB16_IW, RGB24_IW, NV12_IW, TILE_NV12_IW};
	ROT_FLIP eRotFlipList[4] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY
		};

	IMG_FMT eImgFmt;

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	for(i=0; i<4; i++)
	{
		for(j=0; j<4; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);
			
			FIMC_InitIpForFinDout(eCh, eCamera,
				0, 0, uFinImgHsz, uFinImgVsz,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0,
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt[i],
				eRotFlipList[j], eCscEquation, eYCbCrRange);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//---------------------------------------------------------------

			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IW:
					eImgFmt = RGB16;
					break;
				case RGB24_IW:
					eImgFmt = RGB24;
					break;
				case NV12_IW:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IW:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}			

			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				0, 0, uLcdHsz, uLcdVsz,
				uLcdFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);		

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}
		
}

// [Test FIFO-in DMA-out for IRQ]
static void TestFinDoutForIrq(void)
{
	u32 i;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;
	
	for(i=0; i<4; i++)
	{	
		// Initialize LCDC v6.0
		LCDC_InitLdi(MAIN);
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
		LCDC_Start();
		LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

		InitVariables();

		memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

		FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
		FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

		FIMC_InitIpForFinDout(eCh, eCamera,
			0, 0, uFinImgHsz, uFinImgVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uLcdFbAddr, uLcdFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		switch(i)
		{
			case 0:
				FIMC_SetFinIrq(eCh,false, false);
				break;
			case 1:
				FIMC_SetFinIrq(eCh,true, false);
				break;
			case 2:
				FIMC_SetFinIrq(eCh,false, true);
				break;
			case 3:
				FIMC_SetFinIrq(eCh,true, true);
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		if(i == 0)
		{
			Disp("\n Press any key to go next! \n");
			UART_Getc();	

			FIMC_StopProcessing(eCh);
		}
		else if((i==1)||(i==3)) // Last IRQ happens when 'bEnableOfFrameStart = true'
		{
			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);			
		}
		else if(i==2)
		{
			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			//bFimcDone = false;
			//FIMC_PreProcessForLastFrmCpt(eCh);
			//WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);			
		}
		else
		{
			Assert(0);
		}
	}

}

// [Test FIFO-in DMA-out for JPEG file input]
static void TestFinDoutForJpegFileInput(void)
{
	u32 uNum;
	IMG_RESOLUTION eSize;

	u32 uDispHSz, uDispVSz;
	u32 uFinImgHsz, uFinImgVsz;

	u32 uRawImgHSz_Dec, uRawImgVSz_Dec;

	u32 i;
	IMG_FMT eJpegDecodedOutImgFmt[2] = {NV12, CRYCBY};

	uInFbAddr = CODEC_MEM_ST + 0x00300000;
	uOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000;
	//uLcdOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000 + 0x00300000;
	uLcdOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000 + 0x00500000;

	// [JPEG file input]
	//--------------
	Disp("\n JPEG file input \n");	

	Disp("\n 0: 4000x3000(CAM_12M) \n 1: 4000x2400(CAM_W10M) \n 2: 3264x2448(CAM_8M) \n 3: 3264x1960(CAM_W6M) \n");
	Disp("\n Select camera JPEG output resolution: ");
	uNum = UART_GetIntNum();
	
	switch(uNum)
	{
		case 0: 
			eSize = CAM_12M; // 4000x3000
			break;	
		case 1: 
			eSize = CAM_W10M; // 4000x2400
			break;			
		case 2: 
			eSize = CAM_8M; // 3264x2448
			break;
		case 3: 
			eSize = CAM_W6M; // 3264x1960
			break;			
		default:
			Assert(0);
			break;			
	}
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4);

	// ================== Notice !!! ==================
	Disp("\n Sensor initialization! please wait..........\n");
	FIMC_InitCameraForJpegInput(CE143Monaco, eSize);	

	FIMC_InitIpForJFinDout(eCh, eCamera, uOutFbAddr, uOutFbAddr);
	bFimcDone = false;	
	FIMC_StartProcessingForJpegInput(eCh);
	
	CAM_SetCE143MonacoToGetJpeg();

	WaitForFimcDone(eCh);
	// ============================================

	for(i=0; i<2; i++)
	{
		//----------------

		memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4);
		
		bJpegDone = false;	
		JPEG_InitDecodingAndStart(JPEG_CLK_ON, JPEG_FULL_CLK, eJpegDecodedOutImgFmt[i], uOutFbAddr, uInFbAddr);	
		WaitForJpegDone();

		JPEG_GetWidthAndHeight(&uRawImgHSz_Dec, &uRawImgVSz_Dec);
		JPEG_GetDecodedImgSz(&uRawImgHSz_Dec, &uRawImgVSz_Dec);
		Disp("\n JpegDecodedImg(Hsz, Vsz) = (%d, %d) \n", uRawImgHSz_Dec, uRawImgVSz_Dec);

		//----------------

		FIMC_InitIpForDinDout(eCh,
			uRawImgHSz_Dec, uRawImgVSz_Dec, 0,
			0, 0, uRawImgHSz_Dec, uRawImgVSz_Dec,
			uInFbAddr, eJpegDecodedOutImgFmt[i],
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,	
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);		
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to stop! \n");
		UART_Getc();
	}

	uInFbAddr = CODEC_MEM_ST + 0x00300000;
	uOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000;
	uLcdOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000 + 0x00300000;
}

// [Test FIFO-in DMA-out for progressive ITU-601/656 input]
static void TestFinDoutForProgressiveItu601And656Input(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	// ======= Set here! =======
#if 0
	CAM_ATTR eCcir = CCIR601;
#else
	CAM_ATTR eCcir = CCIR656;
#endif
	// =====================

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	

	switch(eCcir)
	{
		case CCIR601:
			LCDC_SetITUInterface(ITU_PROGRESSIVE);
			break;
		case CCIR656:
			LCDC_SetITUInterface_656(ITU_PROGRESSIVE);
			break;
		default:
			Assert(0);
			break;
	}	
	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);
	memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4);
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 	

	//FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_InitCameraForFimd(eCcir, PROGRESSIVE_IN_FIFO, YCBYCR, WVGA_PORT);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
	
	FIMC_InitIpForFinDout(eCh, eCamera,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uLcdOutFbAddr, uLcdOutFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

	Disp(" Press any key to capture image!\n");

	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		Disp("\n Current frame count is %d\n", uFrmCnt);
	}

	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	//---------------------------------------------

	LCDC_Stop();
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();	
}

// [Test FIFO-in DMA-out for interlaced ITU-601/656 input]
static void TestFinDoutForInterlacedItu601And656Input(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	// ================= Set here! =================
#if 0
	CAM_ATTR eCcir = CCIR601;
	FIFO_IN_MODE eFinMode = INTERLACE_IN_ITU601_FIELD;
	//FIFO_IN_MODE eFinMode = INTERLACE_IN_ITU601_HSYNC;
#else
	CAM_ATTR eCcir = CCIR656;
	FIFO_IN_MODE eFinMode = INTERLACE_IN_ITU656;
#endif	
	// =========================================
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
	
	switch(eCcir)
	{
		case CCIR601:
			LCDC_SetITUInterface(ITU_INTERLACE);
			break;
		case CCIR656:
			LCDC_SetITUInterface_656(ITU_INTERLACE);
			break;
		default:
			Assert(0);
			break;
	}	
	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);
	memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4);
	memset((void *)uOutFbAddr, 0, uLcdHsz*uLcdVsz*4);
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 	

	//FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_InitCameraForFimd(eCcir, eFinMode, YCBYCR, WVGA_PORT);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
	
	FIMC_InitIpForFinDout(eCh, eCamera,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uInFbAddr, uOutFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

	Disp(" Press any key to capture image!\n");

	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		Disp("\n Current frame count is %d\n", uFrmCnt);
	}

	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	//---------------------------------------------

	// Even field
	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz/2,
		uInFbAddr, eBpp,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
		uLcdOutFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);	
	FIMC_StopProcessing(eCh);	

	// Odd field
	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz/2,
		uOutFbAddr, eBpp,
		uLcdHsz, uLcdVsz,
		0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
		uLcdOutFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);	
	FIMC_StopProcessing(eCh);	

	//---------------------------------------------

	LCDC_Stop();
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);
	LCDC_Start();	
}

// [Test DIRECT-FIFO-in DMA-out for basic function]
static void TestDFinDoutForBasicFunc(void)
{
	IMG_FMT eImgFmt[2] = {NV12, TILE_NV12};

	u32 i;
	u32 uDispHSz, uDispVSz;
	u32 uFrmCnt;

	Assert(eCh == FIMC_C);

	for(i=0; i<2; i++)
	{
		// Initialize LCDC v6.0
		LCDC_InitLdi(MAIN);
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
		LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);

		Disp("\n Drawing sample image! wait for a moment! \n");
		// Generate DMA input image
		GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
		switch(i)
		{
			case 0:
				GLIB_DrawPattern(uLcdHsz, uLcdVsz);
				break;
			case 1:
				GLIB_DrawPattern2(uLcdHsz, uLcdVsz);
				break;
			default:
				Assert(0);
				break;
		}		

		LCDC_Start();

		memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4);

		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//0, 0, uLcdHsz, uLcdVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uInFbAddr, uInFbAddr, eImgFmt[i],
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		Disp(" Press any key to capture image!\n");
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		while(GetKey() == 0)
		{
#ifndef FIMC_POLLING
			bFimcDone = false;
			while(!bFimcDone);
#else // polling
			Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
			Assert(0);
			return;
#endif		
			FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
			//Disp("\n Current frame count is %d\n", uFrmCnt);
			Disp("%d", uFrmCnt);
		}

		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		//----------------------------------------------------

		// Initialize LCDC v6.0
		LCDC_InitLdi(MAIN);
		LCDC_InitBase();
		LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
		uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
		uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
		switch(eCh)
		{
			case FIMC_C:
				LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
				LCDC_SetWinOnOff(1, WIN2);			
				break;
			default:
				Assert(0);
				break;			
		}
		LCDC_EnableAllWinFifoLvlInt(EMPTY);
		LCDC_Start();

		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uInFbAddr, eImgFmt[i], false,
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();
	}
	
}

// [Test DIRECT-FIFO-in DMA-out for each function]
static void TestDFinDoutForEachFunc(void)
{
	u32 i;

	IMG_FMT eImgFmtList[11] = 
	{	
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};	

	Assert(eCh == FIMC_C);

	// Initialize LCDC v6.0 -> To get 'uLcdHsz', 'uLcdVsz'
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	
	Disp("\n Drawing sample image! wait for a moment! \n");
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	// [Image scaling]
	//---------------
	Disp("\n Image scaling \n");

	InitVariables();	

	for(i=0; i<4; i++)
	{
		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);
		LCDC_Start();

		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4);
		
		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//0, 0, uLcdHsz, uLcdVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uLcdOutFbAddr, uLcdOutFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);	
		
		switch(i)
		{
			case 0:
				uOutImgHsz = 96;
				uOutImgVsz = 96;
				break;
			case 1:
				uOutImgHsz = 144;
				uOutImgVsz = 144;
				break;
			case 2:
				uOutImgHsz = 192;
				uOutImgVsz = 192;
				break;
			case 3:
				uOutImgHsz = (uLcdHsz/1)-((uLcdHsz/1)%16);
				uOutImgVsz = (uLcdVsz/1)-((uLcdVsz/1)%16);
				break;
			default:
				Assert(0);
				break;
		}		

		FIMC_SetFinImgScaler(eCh,uInImgCropHsz, uInImgCropVsz, uOutImgHsz, uOutImgVsz);
		FIMC_SetDoutImgSize(eCh,uOutImgHsz, uOutImgVsz, eBpp, 0);

		FIMC_SetDoutImgOffset(eCh, eBpp, 0, 0);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");
		UART_Getc();

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
		LCDC_Start();

		Disp("\n Order = %d \n", i);
		UART_Getc();
	}

	// [Rotation(Output rotator) and Flip]
	//------------------------------
	Disp("\n Rotation(Output rotator) and Flip \n");

	InitVariables();

	for(i=0; i<8; i++)
	{
		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);
		LCDC_Start();

		if(uLcdHsz >= uLcdVsz)
		{
			uOutImgHsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
			uOutImgVsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
		}
		else // uLcdHsz < uLcdVsz
		{
			uOutImgHsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
			uOutImgVsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
		}

		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdHsz, uLcdVsz,
			0, 0,
			uOutImgHsz, uOutImgVsz, 0, // Scaler
			uLcdOutFbAddr, uLcdOutFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);	
		
		switch(i)
		{
			case 0:
				eRotFlip = ROT_NO_FLIP_NO;
				break;
			case 1:
				eRotFlip = ROT_NO_FLIP_X;
				break;
			case 2:
				eRotFlip = ROT_NO_FLIP_Y;
				break;
			case 3:
				eRotFlip = ROT_NO_FLIP_XY;
				break;
			case 4:
				eRotFlip = ROT_90_FLIP_NO;
				break;
			case 5:
				eRotFlip = ROT_90_FLIP_X;
				break;
			case 6:
				eRotFlip = ROT_90_FLIP_Y;
				break;
			case 7:
				eRotFlip = ROT_90_FLIP_XY;
				break;
			default:
				Assert(0);
				break;
		}
		FIMC_SetOutRotAndFlip(eCh, eRotFlip);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");
		UART_Getc();

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
		LCDC_Start();		

		Disp("\n Order = %d \n", i);
		UART_Getc();			
	}

	// [Image effect]
	//--------------
	Disp("\n Image effect \n");

	InitVariables();

	for(i=0; i<5; i++)
	{
		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);
		LCDC_Start();

		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdHsz, uLcdVsz,
			0, 0,
			uLcdHsz, uLcdVsz, 0, // Scaler
			uLcdOutFbAddr, uLcdOutFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
		
		switch(i)
		{
			case 0:
				eImgEff = EFF_BYPASS;
				break;
			case 1:
				eImgEff = NEGATIVE;
				break;
			case 2:
				eImgEff = ART_FREEZE;
				break;
			case 3:
				eImgEff = EMBOSSING;
				break;
			case 4:
				eImgEff = SILHOUETTE;
				break;
			default:
				Assert(0);
				break;
		}
		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);		

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");
		UART_Getc();

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
		LCDC_Start();		

		Disp("\n Order = %d \n", i);
		UART_Getc();
	}	
		
	for(i=0; i<5; i++)
	{
		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);
		LCDC_Start();

		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdHsz, uLcdVsz,
			0, 0,
			uLcdHsz, uLcdVsz, 0, // Scaler
			uLcdOutFbAddr, uLcdOutFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
			
		eImgEff = ARBITRARY_CBCR;
		switch(i)
		{
			case 0:
				uCbVal = 1;
				uCrVal = 1;
				break;
			case 1:
				uCbVal = 128;
				uCrVal = 128;				
				break;
			case 2:
				uCbVal = 254;
				uCrVal = 254;				
				break;
			case 3:
				uCbVal = 1;
				uCrVal = 254;				
				break;
			case 4:
				uCbVal = 254;
				uCrVal = 1; 			
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);	

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");
		UART_Getc();

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
		LCDC_Start();		

		Disp("\n Order = %d \n", i);
		UART_Getc();
	}
	
	// [Memory store type and Color format conversion]
	//--------------------------------------
	Disp("\n Memory store type and Color format conversion \n");

	InitVariables();

	for(i=0; i<11; i++)
	{
		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);
		LCDC_Start();

		memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		eBppOutFmt = eImgFmtList[i];			

		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdHsz, uLcdVsz,
			0, 0,
			uLcdHsz, uLcdVsz, 0, // Scaler
			uInFbAddr, uInFbAddr, eBppOutFmt,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");
		UART_Getc();

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		//----------------------------------------------------------------------
		
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		eBppInFmt = eImgFmtList[i];
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uInFbAddr,	eBppInFmt,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
		WaitForFimcDone(eCh);				
		FIMC_StopProcessing(eCh);

		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
		LCDC_Start();

		Disp("\n eBppOutFmt order num = %d \n", i);
		Disp("\n Hit any key to go to the next step! \n");
		UART_Getc();
	}
	
}

// [Test DIRECT-FIFO-in DMA-out for composition]
static void TestDFinDoutForComposition(void)
{
	u32 i;

	IMG_FMT eImgFmtList[11] = 
	{
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};
	
	ROT_FLIP eRotFlipList[8] = 
	{
		ROT_NO_FLIP_NO,
		ROT_NO_FLIP_X,
		ROT_NO_FLIP_Y,
		ROT_NO_FLIP_XY,
		ROT_90_FLIP_NO,
		ROT_90_FLIP_X,
		ROT_90_FLIP_Y,
		ROT_90_FLIP_XY
	};

	Assert(eCh == FIMC_C);

	// Initialize LCDC v6.0 -> To get 'uLcdHsz', 'uLcdVsz'
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	
	Disp("\n Drawing sample image! wait for a moment! \n");
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);
	
	// [Composition] 
	//-------------
	Disp("\n Composition \n");

	InitVariables();
	
	for(i=0; i<11; i++)
	{
		Disp("\n ImgFmtOrder = %d, RotFlipOrder = %d \n", i, i%8);
	
		// Initialize LCDC v6.0
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);
		LCDC_Start();

		memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		// Set cropping region
		uOutImgStartX=16;
		uOutImgStartY=16;

		if(uLcdHsz >= uLcdVsz)
		{
			uOutImgHsz = uLcdVsz-(uLcdVsz/8)-((uLcdVsz-(uLcdVsz/8))%16);
			uOutImgVsz = uLcdVsz-(uLcdVsz/8)-((uLcdVsz-(uLcdVsz/8))%16);
		}
		else // uLcdHsz < uLcdVsz
		{
			uOutImgHsz = uLcdHsz-(uLcdHsz/8)-((uLcdHsz-(uLcdHsz/8))%16);
			uOutImgVsz = uLcdHsz-(uLcdHsz/8)-((uLcdHsz-(uLcdHsz/8))%16);
		}
		
		eBppOutFmt = eImgFmtList[i];
		eRotFlip = eRotFlipList[i%8];			

		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdHsz, uLcdVsz,
			uOutImgStartX, uOutImgStartY,
			uOutImgHsz, uOutImgVsz, 0, // Scaler
			uInFbAddr, uInFbAddr, eBppOutFmt,
			eRotFlip, eCscEquation, eYCbCrRange);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");
		UART_Getc();

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
		
		UART_Getc();

		// Set cropping region
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		uInImgCropStartX = 16;
		uInImgCropStartY = 16;
		switch(eRotFlip)
		{
			case ROT_90_FLIP_NO:
			case ROT_90_FLIP_X:
			case ROT_90_FLIP_Y:
			case ROT_90_FLIP_XY:
				uInImgCropHsz = uOutImgVsz, uInImgCropVsz = uOutImgHsz;
				uOutImgHsz=uOutImgVsz, uOutImgVsz=uOutImgHsz;					
				break;
			default:
				uInImgCropHsz = uOutImgHsz, uInImgCropVsz = uOutImgVsz;
				break;
		}		

		eBppInFmt = eImgFmtList[i];
		eBppOutFmt = RGB24;
		eRotFlip = ROT_NO_FLIP_NO;			
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uInFbAddr, eBppInFmt,
			uLcdHsz, uLcdVsz,
			uOutImgStartX, uOutImgStartY, uOutImgHsz, uOutImgVsz, 0, // Scaler		
			uLcdOutFbAddr, eBppOutFmt,		
			eRotFlip, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
		WaitForFimcDone(eCh);				
		FIMC_StopProcessing(eCh);
		
		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
		LCDC_Start();

		Disp("\n Hit any key to go next! \n");	
		UART_Getc();			
	}
	
}

// [Test DIRECT-FIFO-in DMA-out for FIMD frame skip]
static void TestDFinDoutForFimdFrameSkip(void)
{
	LCD_WB_SKIP eLCD_WB_SKIP[30] = 
	{
		SKIP_1_1,SKIP_1_2,SKIP_1_3,SKIP_1_4,SKIP_1_5,SKIP_1_6,SKIP_1_7,SKIP_1_8,SKIP_1_9,SKIP_1_10,
		SKIP_1_11,SKIP_1_12,SKIP_1_13,SKIP_1_14,SKIP_1_15,SKIP_1_16,SKIP_1_17,SKIP_1_18,SKIP_1_19,SKIP_1_20,
		SKIP_1_21,SKIP_1_22,SKIP_1_23,SKIP_1_24,SKIP_1_25,SKIP_1_26,SKIP_1_27,SKIP_1_28,SKIP_1_29,SKIP_1_30
	};

	u32 i, j;
	u32 uDispHSz, uDispVSz;
	u32 uFrmCnt;
	u32 uTotalFrameCount;
	IMG_FMT eImgFmt[2] = {NV12, TILE_NV12};

	Assert(eCh == FIMC_C);

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	Disp("\n Drawing sample image! wait for a moment! \n"); 	
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp); // Generate DMA input image
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	for(j=0; j<2; j++)
	{
		for(i=0; i<30; i++)
		{
			// Initialize LCDC v6.0
			LCDC_InitLdi(MAIN);
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
			//LCDC_SetOutputPath(LCD_TV);
			LCDC_SetOutputPath(LCD_TVRGB);
			LCDC_SetWBFrameSkip(eLCD_WB_SKIP[i]);
			LCDC_Start();		
		
			memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4);

			FIMC_InitIpForDFinDout(eCh,
				uLcdHsz, uLcdVsz,
				//0, 0, uLcdHsz, uLcdVsz,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0,
				uInFbAddr, uInFbAddr, eImgFmt[j],
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

			Disp(" Press any key to capture image!\n");
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			uTotalFrameCount = 0;

			PWM_stopAll();
			bTimer0Done = false;
			PWM_Select(0,255,4,1000000,500000,0,0,0,1,0,1);
			
			while(!bTimer0Done)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				uTotalFrameCount = uTotalFrameCount + 1;
				
				//FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				//Disp("\n Current frame count is %d\n", uFrmCnt);
				//Disp("%d", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			Disp("\n\n ===== SKIP_1_%d: uTotalFrameCount = %d, %dx%d=%d  ===== \n\n", 
				i+1, uTotalFrameCount, uTotalFrameCount, i+1, uTotalFrameCount*(i+1));
		}

		//----------------------------------------------------

		// Initialize LCDC v6.0
		LCDC_InitLdi(MAIN);
		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
		LCDC_Start();
		LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

		FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uLcdFbAddr,	eImgFmt[j],
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
		uLcdOutFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);						
		FIMC_StopProcessing(eCh);
	}
	
}

// [Test DIRECT-FIFO-in DMA-out for Fimd Blending]
static void TestDFinDoutForFimdBlending(void)
{
	u32 i;
	u32 j;
	u32 uColor;
	u32 uWin;
	u32 uDispHSz, uDispVSz;
	u32 uFrmCnt;
	u32 uFgAddr;

	Assert(eCh == FIMC_C);

	for(i=0; i<2; i++)
	{
		// Initialize LCDC v6.0
		LCDC_InitLdi(MAIN);
		LCDC_InitBase();
		//LCDC_InitBase_jw(320,240);
		LCDC_Start();

		LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
		
		LCDC_InitWin(eBpp, uLcdHsz, uLcdVsz, 0, 0, uLcdHsz, uLcdVsz,0, 0, uLcdFbAddr, WIN0, false);

		GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
		GLIB_DrawPattern(uLcdHsz, uLcdVsz);
		GLIB_Printf(uLcdHsz/2,0, "WINDOW 0");

		for( j = 1 ; j < 5 ; j++)
		{
			LCDC_GetFbEndAddr(&uFgAddr, (LCD_WINDOW)(j-1));
			LCDC_InitWin(eBpp, uLcdHsz/2, uLcdVsz/2, 0, 0, uLcdHsz/2, uLcdVsz/2,(uLcdHsz/2),(uLcdVsz/8)*j, uFgAddr, (LCD_WINDOW)j, false);
			if(i==1)
			{
				LCDC_SetBlendingType(PER_PLANE, ALPHA0_PLANE, (LCD_WINDOW)j);
				LCDC_SetAlpha(	0x80,0x80,0x80, 0x0, 0x0, 0x0, (LCD_WINDOW)j);	
			}
			GLIB_InitInstance(uFgAddr, uLcdHsz/2, uLcdVsz/2, eBpp);
			//GLIB_DrawPattern(uLcdHsz/2, uLcdVsz/2);
			switch(j)
			{
				case 1: GLIB_ClearFrame(C_RED); break;
				case 2: GLIB_ClearFrame(C_GREEN); break;
				case 3: GLIB_ClearFrame(C_BLUE); break;
				case 4: GLIB_ClearFrame(C_YELLOW); break;
			}
			GLIB_Printf(0,0, "WINDOW %d",j);
		}

		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);

		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//0, 0, uLcdHsz, uLcdVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz/2, uLcdVsz/2, 0,
			uLcdFbAddr, uLcdFbAddr, RGB24,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		uWin=1;
		uColor=0;
		while(GetKey() == 0)
		{
			LCDC_GetFbEndAddr(&uFgAddr, (LCD_WINDOW)(uWin-1));
			GLIB_InitInstance(uFgAddr, uLcdHsz/2, uLcdVsz/2, eBpp);
			GLIB_ClearFrame((COLOR)uColor); 
			GLIB_Printf(uLcdHsz/2 + uColor*10,0, "WINDOW %d",uWin);
			Disp("Window %d color changed\n",uWin);

			uWin++;
			if(uWin>4) uWin=1;
			uColor++;
			if(uColor>7) uColor = 0;
			Delay(1000);			
		}
		
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
	}
	
}

// [Test MIPI-FIFO-in DMA-out for basic function]
static void TestMFinDoutForBasicFunc(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;	
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536
	eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	eHSsettle = 12;	
	//eCSISDataLaneNum = DATALANE1;
	eCSISDataLaneNum = DATALANE2;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);

	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
	
	FIMC_InitIpForMFinDout(eCh,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange,eDataAlign);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

	Disp(" Press any key to capture image!\n");

	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		Disp("\n Current frame count is %d\n", uFrmCnt);
	}

	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	CSI_DisableCSI();
	MIPICAM_Disable();
}

// [Test MIPI-FIFO-in DMA-out for each function]
static void TestMFinDoutForEachFunc(void)
{
	u32 i;
	u32 uFinImgHsz, uFinImgVsz;

	IMG_FMT eImgFmtList[11] = 
	{	
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};	

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536
	//eInSize = HD720; // HD720:1280x720
	//eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	eInSize = HD1080; // HD1080:1920x1080
	eHSsettle = 12;	
	//eCSISDataLaneNum = DATALANE1;	
	eCSISDataLaneNum = DATALANE2;	

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);	
	MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);
	
	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	// [Image cropping]
	//----------------
	Disp("\n Image cropping \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	
	uInImgCropHsz = (uFinImgHsz/2)-((uFinImgHsz/2)%16);
	uInImgCropVsz = (uFinImgVsz/2)-((uFinImgVsz/2)%16); 
	uOutImgHsz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uOutImgVsz = (uLcdVsz/2)-((uLcdVsz/2)%16);
	
	FIMC_InitIpForMFinDout(eCh,
		0, 0, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uOutImgHsz, uOutImgVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);
	
	for(i=0; i<4; i++)
	{
		switch(i)
		{
			case 0:
				uInImgCropStartX = 0;
				uInImgCropStartY = 0;
				uOutImgStartX = 0;
				uOutImgStartY = 0;
				break;
			case 1:
				uInImgCropStartX = uFinImgHsz/2;
				uInImgCropStartY = 0;
				uOutImgStartX = uLcdHsz/2;
				uOutImgStartY = 0;
				break;
			case 2:
				uInImgCropStartX = 0;
				uInImgCropStartY = uFinImgVsz/2;
				uOutImgStartX = 0;
				uOutImgStartY = uLcdVsz/2;
				break;
			case 3:
				uInImgCropStartX = uFinImgHsz/2;
				uInImgCropStartY = uFinImgVsz/2;
				uOutImgStartX = uLcdHsz/2;
				uOutImgStartY = uLcdVsz/2;
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_SetFinImgCroppingRegion(eCh,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz);
		FIMC_SetDoutImgOffset(eCh, eBpp, uOutImgStartX, uOutImgStartY);

		if(i==0)
		{
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		}
				
		Disp("\n Order = %d \n", i);
		UART_Getc();
	}
	
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
	UART_Getc();

	// [Image scaling]
	//---------------
	Disp("\n Image scaling \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	
	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region	
	uInImgCropVsz = uFinImgVsz; // Set cropping region	

	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign); 

	for(i=0; i<4; i++)
	{
		switch(i)
		{
			case 0:
				uOutImgHsz = 96;
				uOutImgVsz = 96;
				break;
			case 1:
				uOutImgHsz = 144;
				uOutImgVsz = 144;
				break;
			case 2:
				uOutImgHsz = 192;
				uOutImgVsz = 192;
				break;
			case 3:
				uOutImgHsz = (uLcdHsz/1)-((uLcdHsz/1)%16);
				uOutImgVsz = (uLcdVsz/1)-((uLcdVsz/1)%16);
				break;
			default:
				Assert(0);
				break;
		}		

		FIMC_SetFinImgScaler(eCh,uInImgCropHsz, uInImgCropVsz, uOutImgHsz, uOutImgVsz);
		FIMC_SetDoutImgSize(eCh,uOutImgHsz, uOutImgVsz, eBpp, 0);

		FIMC_SetDoutImgOffset(eCh, eBpp, 0, 0);

		if(i==0)
		{
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		}
		
		Disp("\n Order = %d \n", i);
		UART_Getc();
	}

	for(i=0; i<9; i++)
	{
#if 0
		switch(i)
		{
			case 0:
				uInImgCropStartX = 80;
				uInImgCropStartY = 60;
				uInImgCropHsz = (uFinImgHsz-(80*2))-((uFinImgHsz-(80*2))%16);
				uInImgCropVsz = (uFinImgVsz-(60*2))-((uFinImgVsz-(60*2))%16);
				break;
			case 1:
				uInImgCropStartX = 160;
				uInImgCropStartY = 120;
				uInImgCropHsz = (uFinImgHsz-(160*2))-((uFinImgHsz-(160*2))%16);
				uInImgCropVsz = (uFinImgVsz-(120*2))-((uFinImgVsz-(120*2))%16);
				break;
			case 2:
				uInImgCropStartX = 240;
				uInImgCropStartY = 180;
				uInImgCropHsz = (uFinImgHsz-(240*2))-((uFinImgHsz-(240*2))%16);
				uInImgCropVsz = (uFinImgVsz-(180*2))-((uFinImgVsz-(180*2))%16);
				break;
			case 3:
				uInImgCropStartX = 320;
				uInImgCropStartY = 240;
				uInImgCropHsz = (uFinImgHsz-(320*2))-((uFinImgHsz-(320*2))%16);
				uInImgCropVsz = (uFinImgVsz-(240*2))-((uFinImgVsz-(240*2))%16);
				break;
			case 4:
				uInImgCropStartX = 400;
				uInImgCropStartY = 300;
				uInImgCropHsz = (uFinImgHsz-(400*2))-((uFinImgHsz-(400*2))%16);
				uInImgCropVsz = (uFinImgVsz-(300*2))-((uFinImgVsz-(300*2))%16);
				break;
			case 5:
				uInImgCropStartX = 480;
				uInImgCropStartY = 360;
				uInImgCropHsz = (uFinImgHsz-(480*2))-((uFinImgHsz-(480*2))%16);
				uInImgCropVsz = (uFinImgVsz-(360*2))-((uFinImgVsz-(360*2))%16);
				break;
			case 6:
				uInImgCropStartX = 560;
				uInImgCropStartY = 420;
				uInImgCropHsz = (uFinImgHsz-(560*2))-((uFinImgHsz-(560*2))%16);
				uInImgCropVsz = (uFinImgVsz-(420*2))-((uFinImgVsz-(420*2))%16);
				break;
			case 7:
				uInImgCropStartX = 640;
				uInImgCropStartY = 480;
				uInImgCropHsz = (uFinImgHsz-(640*2))-((uFinImgHsz-(640*2))%16);
				uInImgCropVsz = (uFinImgVsz-(480*2))-((uFinImgVsz-(480*2))%16);
				break;
			case 8:
				uInImgCropStartX = 720;
				uInImgCropStartY = 540;
				uInImgCropHsz = (uFinImgHsz-(720*2))-((uFinImgHsz-(720*2))%16);
				uInImgCropVsz = (uFinImgVsz-(540*2))-((uFinImgVsz-(540*2))%16);
				break;
			default:
				Assert(0);
				break;
		}
#else
		switch(i)
		{
			case 0:
				uInImgCropStartX = 20;
				uInImgCropStartY = 10;
				uInImgCropHsz = (uFinImgHsz-(20*2))-((uFinImgHsz-(20*2))%16);
				uInImgCropVsz = (uFinImgVsz-(10*2))-((uFinImgVsz-(10*2))%16);
				break;
			case 1:
				uInImgCropStartX = 40;
				uInImgCropStartY = 20;
				uInImgCropHsz = (uFinImgHsz-(40*2))-((uFinImgHsz-(40*2))%16);
				uInImgCropVsz = (uFinImgVsz-(20*2))-((uFinImgVsz-(20*2))%16);
				break;
			case 2:
				uInImgCropStartX = 60;
				uInImgCropStartY = 30;
				uInImgCropHsz = (uFinImgHsz-(60*2))-((uFinImgHsz-(60*2))%16);
				uInImgCropVsz = (uFinImgVsz-(30*2))-((uFinImgVsz-(30*2))%16);
				break;
			case 3:
				uInImgCropStartX = 80;
				uInImgCropStartY = 40;
				uInImgCropHsz = (uFinImgHsz-(80*2))-((uFinImgHsz-(80*2))%16);
				uInImgCropVsz = (uFinImgVsz-(40*2))-((uFinImgVsz-(40*2))%16);
				break;
			case 4:
				uInImgCropStartX = 100;
				uInImgCropStartY = 50;
				uInImgCropHsz = (uFinImgHsz-(100*2))-((uFinImgHsz-(100*2))%16);
				uInImgCropVsz = (uFinImgVsz-(50*2))-((uFinImgVsz-(50*2))%16);
				break;
			case 5:
				uInImgCropStartX = 120;
				uInImgCropStartY = 60;
				uInImgCropHsz = (uFinImgHsz-(120*2))-((uFinImgHsz-(120*2))%16);
				uInImgCropVsz = (uFinImgVsz-(60*2))-((uFinImgVsz-(60*2))%16);
				break;
			case 6:
				uInImgCropStartX = 140;
				uInImgCropStartY = 70;
				uInImgCropHsz = (uFinImgHsz-(140*2))-((uFinImgHsz-(140*2))%16);
				uInImgCropVsz = (uFinImgVsz-(70*2))-((uFinImgVsz-(70*2))%16);
				break;
			case 7:
				uInImgCropStartX = 160;
				uInImgCropStartY = 80;
				uInImgCropHsz = (uFinImgHsz-(160*2))-((uFinImgHsz-(160*2))%16);
				uInImgCropVsz = (uFinImgVsz-(80*2))-((uFinImgVsz-(80*2))%16);
				break;
			case 8:
				uInImgCropStartX = 180;
				uInImgCropStartY = 90;
				uInImgCropHsz = (uFinImgHsz-(180*2))-((uFinImgHsz-(180*2))%16);
				uInImgCropVsz = (uFinImgVsz-(90*2))-((uFinImgVsz-(90*2))%16);
				break;
			default:
				Assert(0);
				break;
		}
#endif
		FIMC_SetFinImgCroppingRegion(eCh,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz);
		FIMC_SetFinImgScaler(eCh,uInImgCropHsz, uInImgCropVsz, uOutImgHsz, uOutImgVsz);
		
		Disp("\n Order = %d \n", i+4);
		UART_Getc();
	}
	
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
	UART_Getc();

	// [Rotation(Output rotator) and Flip]
	//------------------------------
	Disp("\n Rotation(Output rotator) and Flip \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	
	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region

	if(uLcdHsz >= uLcdVsz)
	{
		uOutImgHsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
		uOutImgVsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
	}
	else // uLcdHsz < uLcdVsz
	{
		uOutImgHsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
		uOutImgVsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
	}
	
	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uOutImgHsz, uOutImgVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

	for(i=0; i<8; i++)
	{
		memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		switch(i)
		{
			case 0:
				eRotFlip = ROT_NO_FLIP_NO;
				break;
			case 1:
				eRotFlip = ROT_NO_FLIP_X;
				break;
			case 2:
				eRotFlip = ROT_NO_FLIP_Y;
				break;
			case 3:
				eRotFlip = ROT_NO_FLIP_XY;
				break;
			case 4:
				eRotFlip = ROT_90_FLIP_NO;
				break;
			case 5:
				eRotFlip = ROT_90_FLIP_X;
				break;
			case 6:
				eRotFlip = ROT_90_FLIP_Y;
				break;
			case 7:
				eRotFlip = ROT_90_FLIP_XY;
				break;
			default:
				Assert(0);
				break;
		}
		FIMC_SetOutRotAndFlip(eCh, eRotFlip);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		
		Disp("\n Order = %d \n", i);
		UART_Getc();	
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
	}
	UART_Getc();

	// [Image effect]
	//--------------
	Disp("\n Image effect \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	
	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region

	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

	for(i=0; i<5; i++)
	{
		switch(i)
		{
			case 0:
				eImgEff = EFF_BYPASS;
				break;
			case 1:
				eImgEff = NEGATIVE;
				break;
			case 2:
				eImgEff = ART_FREEZE;
				break;
			case 3:
				eImgEff = EMBOSSING;
				break;
			case 4:
				eImgEff = SILHOUETTE;
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);
		
		if(i == 0)
		{
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		}
		
		UART_Getc();
	}	
		
	for(i=0; i<5; i++)
	{
		eImgEff = ARBITRARY_CBCR;

		switch(i)
		{
			case 0:
				uCbVal = 1;
				uCrVal = 1;
				break;
			case 1:
				uCbVal = 128;
				uCrVal = 128;				
				break;
			case 2:
				uCbVal = 254;
				uCrVal = 254;				
				break;
			case 3:
				uCbVal = 1;
				uCrVal = 254;				
				break;
			case 4:
				uCbVal = 254;
				uCrVal = 1; 			
				break;
			default:
				Assert(0);
				break;
		}

		FIMC_SetImgEffect(eCh,eImgEff, uCbVal, uCrVal, AFTER_SCALING);			
		UART_Getc();
	}
	
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
	UART_Getc();	

	// [Memory store type and Color format conversion]
	//--------------------------------------
	Disp("\n Memory store type and Color format conversion \n");

	InitVariables();
			
	for(i=0; i<11; i++)
	{
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		LCDC_Start();

		memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		// Set cropping region
		uInImgCropStartX = 0;
		uInImgCropStartY = 0;
		uInImgCropHsz = uFinImgHsz;
		uInImgCropVsz = uFinImgVsz;
		
		eBppOutFmt = eImgFmtList[i];			

		FIMC_InitIpForMFinDout(eCh,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdHsz, uLcdVsz,
			0, 0,
			uLcdHsz, uLcdVsz, 0, // Scaler
			uLcdFbAddr, uLcdFbAddr, eBppOutFmt,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");
		UART_Getc();

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
		
		UART_Getc();

		// Set cropping region
		uInImgCropStartX = 0;
		uInImgCropStartY = 0;
		uInImgCropHsz = uLcdHsz;
		uInImgCropVsz = uLcdVsz;	
		eBppInFmt = eImgFmtList[i]; 		

		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdFbAddr, eBppInFmt,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
		WaitForFimcDone(eCh);				
		FIMC_StopProcessing(eCh);

		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
		LCDC_Start();

		Disp("\n eBppOutFmt order num = %d \n", i);
		Disp("\n Hit any key to go to the next step! \n");
		UART_Getc();
	}

	// [Capture frame control]
	//--------------------- 
	Disp("\n Capture frame control \n");

	InitVariables();

	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region
	
	// Set capture frame control
	bCptFrmCtrl = true;
	uCptSeqPattern = 0x10101010, uCptSeqPointer = 0;	

	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, bCptFrmCtrl, uCptSeqPattern, uCptSeqPointer, false, 0);

	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);
#ifndef FIMC_POLLING
	while(!bFimcDone);
#else // polling
	FIMC_WaitForProcessingDoneForCptFrmCtrl(eCh);
#endif
	FIMC_StopProcessing(eCh);

	UART_Getc();

	// Set capture count control
	bCptCntCtrl = true;
	uCptCnt = 10;

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, bCptFrmCtrl, uCptSeqPattern, uCptSeqPointer, bCptCntCtrl, uCptCnt);

	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);

#ifndef FIMC_POLLING
	while(!bFimcDone);
#else // polling
	FIMC_WaitForProcessingDoneForCptFrmCtrl(eCh);
#endif
	FIMC_StopProcessing(eCh);	

	CSI_DisableCSI();
	MIPICAM_Disable();
}

// [Test MIPI-FIFO-in DMA-out for composition]
static void TestMFinDoutForComposition(void)
{
	u32 i;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uInImgCropHszList[3];
	u32 uInImgCropVszList[3];
	u32 uInImgCropStartXList[3]={48,232,472};
	u32 uInImgCropStartYList[3]={16,152,392};
	
	IMG_FMT eImgFmtList[11] = 
	{
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};
	
	ROT_FLIP eRotFlipList[8] = 
	{
		ROT_NO_FLIP_NO,
		ROT_NO_FLIP_X,
		ROT_NO_FLIP_Y,
		ROT_NO_FLIP_XY,
		ROT_90_FLIP_NO,
		ROT_90_FLIP_X,
		ROT_90_FLIP_Y,
		ROT_90_FLIP_XY
	};

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536
	eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	eHSsettle = 12;	
	//eCSISDataLaneNum = DATALANE1;	
	eCSISDataLaneNum = DATALANE2;	

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	
	// [Composition] 
	//-------------
	Disp("\n Composition \n");

	InitVariables();
	
	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);
	
	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	uInImgCropHszList[0] = (uFinImgHsz-(48*2))-((uFinImgHsz-(48*2))%16);
	uInImgCropHszList[1] = (uFinImgHsz-(232*2))-((uFinImgHsz-(232*2))%16);
	uInImgCropHszList[2] = (uFinImgHsz-(472*2))-((uFinImgHsz-(472*2))%16);

	uInImgCropVszList[0] = (uFinImgVsz-(16*2))-((uFinImgVsz-(16*2))%16);
	uInImgCropVszList[1] = (uFinImgVsz-(152*2))-((uFinImgVsz-(152*2))%16);
	uInImgCropVszList[2] = (uFinImgVsz-(392*2))-((uFinImgVsz-(392*2))%16);
			
	for(i=0; i<11; i++)
	{
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		LCDC_Start();

		memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		// Set cropping region
		uInImgCropStartX = uInImgCropStartXList[(i/3)%3];
		uInImgCropStartY = uInImgCropStartYList[i%3];
		uInImgCropHsz = uInImgCropHszList[(i/3)%3];
		uInImgCropVsz = uInImgCropVszList[i%3];
		uOutImgStartX=16;
		uOutImgStartY=16;

		if(uLcdHsz >= uLcdVsz)
		{
			uOutImgHsz = uLcdVsz-(uLcdVsz/8)-((uLcdVsz-(uLcdVsz/8))%16);
			uOutImgVsz = uLcdVsz-(uLcdVsz/8)-((uLcdVsz-(uLcdVsz/8))%16);
		}
		else // uLcdHsz < uLcdVsz
		{
			uOutImgHsz = uLcdHsz-(uLcdHsz/8)-((uLcdHsz-(uLcdHsz/8))%16);
			uOutImgVsz = uLcdHsz-(uLcdHsz/8)-((uLcdHsz-(uLcdHsz/8))%16);
		}
		
		eBppOutFmt = eImgFmtList[i];
		eRotFlip = eRotFlipList[i%8];			

		FIMC_InitIpForMFinDout(eCh,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdHsz, uLcdVsz,
			uOutImgStartX, uOutImgStartY,
			uOutImgHsz, uOutImgVsz, 0, // Scaler
			uLcdFbAddr, uLcdFbAddr, eBppOutFmt,
			eRotFlip, eCscEquation, eYCbCrRange, eDataAlign);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");
		UART_Getc();

		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
		
		UART_Getc();

		// Set cropping region
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		uInImgCropStartX = 16;
		uInImgCropStartY = 16;
		switch(eRotFlip)
		{
			case ROT_90_FLIP_NO:
			case ROT_90_FLIP_X:
			case ROT_90_FLIP_Y:
			case ROT_90_FLIP_XY:
				//uInImgCropHsz = 176, uInImgCropVsz = 208;
				//uOutImgHsz=176, uOutImgVsz=208;
				uInImgCropHsz = uOutImgVsz, uInImgCropVsz = uOutImgHsz;
				uOutImgHsz=uOutImgVsz, uOutImgVsz=uOutImgHsz;					
				break;
			default:
				//uInImgCropHsz = 208, uInImgCropVsz = 176;
				uInImgCropHsz = uOutImgHsz, uInImgCropVsz = uOutImgVsz;
				break;
		}		

		eBppInFmt = eImgFmtList[i];
		eBppOutFmt = RGB24;
		eRotFlip = ROT_NO_FLIP_NO;			
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdFbAddr, eBppInFmt,
			uLcdHsz, uLcdVsz,
			uOutImgStartX, uOutImgStartY, uOutImgHsz, uOutImgVsz, 0, // Scaler		
			uLcdOutFbAddr, eBppOutFmt,		
			eRotFlip, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
		WaitForFimcDone(eCh);				
		FIMC_StopProcessing(eCh);
		
		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
		LCDC_Start();

		Disp("\n eBppOutFmt order num = %d \n", i);
		Disp("\n Hit any key to go to the next step! \n");
		UART_Getc();			
	}	

	CSI_DisableCSI();
	MIPICAM_Disable();
}

// [Test MIPI-FIFO-in DMA-out for double buffering]
static void TestMFinDoutForDoubleBuffering(void)
{
	u32 uDoubleBufAddr;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;
	u32 uBufferIdx;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536
	eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	eHSsettle = 12;	
	//eCSISDataLaneNum = DATALANE1;	
	eCSISDataLaneNum = DATALANE2;	
		
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, true); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	LCDC_GetDoubleBufAddr(&uDoubleBufAddr, 1, WIN0);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);
	memset((void *)uDoubleBufAddr, 0, uLcdHsz*uLcdVsz*4);

	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);
	
	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	FIMC_InitIpForMFinDout(eCh,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uLcdFbAddr, uDoubleBufAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

	Disp(" Press any key to capture image!\n");

	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		uBufferIdx = uFrmCnt%2;
		LCDC_SetActiveFrameBuf(uBufferIdx, WIN0);
		Disp("\n Current frame count is %d\n", uFrmCnt);
	}

	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	CSI_DisableCSI();
	MIPICAM_Disable();
}

// [Test MIPI-FIFO-in DMA-out for scaler bypass]
static void TestMFinDoutForScalerBypass(void)
{
	IMG_FMT eOutImgTypeList[9] = {NV12, YC420, TILE_NV12,
		YCBYCR, YCRYCB, CRYCBY, CBYCRY, NV16, YC422};

	u32 j;
	u32 uFinImgHsz, uFinImgVsz;
	
	u32 uFrmCnt;
	
	u32 uDispHSz, uDispVSz;

	u32 RetOfCr, RetOfY0, RetOfCb, RetOfY1;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536	
	//eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	eInSize = HD1080; // HD1080:1920x1080
	eHSsettle = 12;	
	//eCSISDataLaneNum = DATALANE1;	
	eCSISDataLaneNum = DATALANE2;	

	// [Scaler bypass]
	//--------------
	Disp("\n Scaler bypass \n");	
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 64, 64);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 64, 64);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 64, 64);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	InitVariables();

	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);
	
	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	for(j=0; j<9; j++)
	{
		FIMC_InitIpForMFinDout(eCh,
			0, 0, uFinImgHsz, uFinImgVsz,
			uFinImgHsz, uFinImgVsz,
			0, 0, uFinImgHsz, uFinImgVsz, 0,
			uInFbAddr, uInFbAddr, eOutImgTypeList[j],
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

		// ============ Notice !!! ============
		FIMC_StartProcessingForScalerBypass(eCh);
		// ================================
		
		Disp(" Press any key to capture image!\n");
		while(GetKey() == 0)
		{
#ifndef FIMC_POLLING
			bFimcDone = false;
			while(!bFimcDone);
#else // polling
			Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
			Assert(0);
			return;
#endif		
			FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
			Disp("\n Current frame count is %d\n", uFrmCnt);
		}
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);	

		//----------------

		FIMC_InitIpForDinFout(eCh,
			uFinImgHsz, uFinImgVsz, 0,
			0, 0, uFinImgHsz, uFinImgVsz,
			uInFbAddr, eOutImgTypeList[j], false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);
	}
	
	CSI_DisableCSI();
	MIPICAM_Disable();
}

// [Test MIPI-FIFO-in DMA-out for interlace bob out]
static void TestMFinDoutForInterlaceBobOut(void)
{
	IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536
	//eInSize = HD720; // HD720:1280x720
	eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	//eInSize = HD1080; // HD1080:1920x1080
	eHSsettle = 12;	
	//eCSISDataLaneNum = DATALANE1;	
	eCSISDataLaneNum = DATALANE2;	

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);
	
	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	for(i=0; i<4; i++)
	{
		for(j=0; j<8; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = uLcdHsz;
					uOutDispVsz = uLcdVsz/2;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uOutDispHsz = (uLcdVsz/2) - ((uLcdVsz/2)%16); // Notice!
					uOutDispVsz = uLcdHsz; // Notice!
					
					uOutImgHsz = uLcdHsz;
					//uOutImgVsz = uLcdVsz - 16;
					uOutImgVsz = uLcdVsz - (uLcdVsz%32);
					break;
				default:
					Assert(0);
					break;
			}

			FIMC_InitIpForMFinDout(eCh,				
				0, 0, uFinImgHsz, uFinImgVsz,
				uOutDispHsz, uOutDispVsz,
				0, 0, uOutImgHsz, uOutImgVsz, 0,
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt[i],
				eRotFlipList[j], eCscEquation, eYCbCrRange, eDataAlign);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//--------------------------------------------

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					uInImgVsz = uLcdVsz/2;		
			
					uInImgCropHsz = uLcdHsz - (uLcdHsz%16); 
					uInImgCropVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uInImgHsz = (uLcdVsz/2) - ((uLcdVsz/2)%16);
					uInImgVsz = uLcdHsz;		

					uInImgCropHsz =  (uLcdVsz/2) - ((uLcdVsz/2)%16);
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			

			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uLcdFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,				
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			// Odd field
#if 0			
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uLcdFbAddr, &uNextFbAddr);
#else
			FIMC_GetNextFbAddr(eImgFmt, uInImgHsz, uInImgVsz, uLcdFbAddr, &uNextFbAddr);
#endif
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uNextFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}

	CSI_DisableCSI();
	MIPICAM_Disable();
}

// [Test MIPI-FIFO-in DMA-out for interlace weave out]
static void TestMFinDoutForInterlaceWeaveOut(void)
{
	// MFC interlaced encoding scenario do NOT need ROTATION
	// CAMERA -> (progressive) -> FIMC -> (interlace, weave out) -> MFC -> MEM
	
	IMG_FMT eInterlaceImgFmt[4] = {RGB16_IW, RGB24_IW, NV12_IW, TILE_NV12_IW};
	ROT_FLIP eRotFlipList[4] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY
		};

	IMG_FMT eImgFmt;

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536
	//eInSize = HD720; // HD720:1280x720
	eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	eHSsettle = 12;	
	//eCSISDataLaneNum = DATALANE1;	
	eCSISDataLaneNum = DATALANE2;	

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);
	
	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	for(i=0; i<4; i++)
	{
		for(j=0; j<4; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);
			
			FIMC_InitIpForMFinDout(eCh,
				0, 0, uFinImgHsz, uFinImgVsz,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0,
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt[i],
				eRotFlipList[j], eCscEquation, eYCbCrRange, eDataAlign);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//---------------------------------------------------------------

			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IW:
					eImgFmt = RGB16;
					break;
				case RGB24_IW:
					eImgFmt = RGB24;
					break;
				case NV12_IW:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IW:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}			

			FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
				0, 0, uLcdHsz, uLcdVsz,
				uLcdFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);		

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}

	CSI_DisableCSI();
	MIPICAM_Disable();
}

extern u8 Test_DinDoutForSortingChip(void);
// [Test DMA-in DMA-out for sorting chip]
u8 TestDinDoutForSortingChip(void)
{
	SYSC_InitClkForMM();
	InitBaseHwForFIMC();

	if(Test_DinDoutForSortingChip() == true)
	{
		return true;
	}
	else // Test_DinDoutForSortingChip() == false
	{
		return false;
	}
}

extern u8 Test_DinDoutUsingCModel(void);
// [Test DMA-in DMA-out using c-model]
u8 TestDinDoutUsingCModel(void)
{
	if(Test_DinDoutUsingCModel() == true)
	{
		return true;
	}
	else // Test_DinDoutUsingCModel() == false
	{
		return false;
	}
}

// [Test iteration of DMA-in FIFO-out and DMA-in DMA-out]
static void TestIterationOfDinFoutAndDinDout(void)
{
	u32 uLoopCnt = 0;
	u32 uDispHSz, uDispVSz;
	u32 uRandNum;

	do
	{
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 100)", uLoopCnt);
		
		// Initialize LCDC v6.0
		LCDC_InitLdi(MAIN);
		LCDC_InitBase();
		LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
		uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
		uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
		switch(eCh)
		{
			case FIMC_A:
				LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
				LCDC_SetWinOnOff(1, WIN0);
				break;
			case FIMC_B:
				LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
				LCDC_SetWinOnOff(1, WIN1);			
				break;
			case FIMC_C:
				LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
				LCDC_SetWinOnOff(1, WIN2);			
				break;
			default:
				Assert(0);
				break;			
		}
		LCDC_EnableAllWinFifoLvlInt(EMPTY);
		LCDC_Start();

		// Generate DMA input image
		GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
		GLIB_DrawPattern(uLcdHsz, uLcdVsz);

		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp, false,
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 0		
		Disp("\n Press any key to stop! \n");
		UART_Getc();
#else
		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*100);
		}
#endif
		FIMC_StopProcessing(eCh);

		//----------------------------------------------------

		// Initialize LCDC v6.0
		LCDC_InitLdi(MAIN);
		LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
		LCDC_Start();
		LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

		// Generate DMA input image
		GLIB_InitInstance(uInFbAddr, uLcdHsz, uLcdVsz, eBpp);
		GLIB_DrawPattern2(uLcdHsz, uLcdVsz);	
			
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uInFbAddr,	eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
		WaitForFimcDone(eCh);	
		FIMC_StopProcessing(eCh);

		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*100);
		}		
	}
	while(uLoopCnt < 100);
	
}

// [Test DMA-in DMA-out for power gating]
static void TestDinDoutForPowerGating(void)
{
	u32 uLoopCnt = 0;
	CLKPWR_TEST eCLKPWR_TEST;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);	
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 	

	do
	{
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 1000)", uLoopCnt);
	
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr,	eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);				
		FIMC_StopProcessing(eCh);

		//----------------------------------------------------

		switch(eCh)
		{
			case FIMC_A:
				eIpId = eFIMC0;
				eCLKPWR_TEST = CPTEST_FIMC0;
				break;
			case FIMC_B:
				eIpId = eFIMC1;
				eCLKPWR_TEST = CPTEST_FIMC1;
				break;
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// X block power off & on
		SYSC_SetBlockPowerOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetBlockPowerON_IP(eIpId);
	}
	while(uLoopCnt < 1000);
	
}

// [Test DMA-in DMA-out for clock gating]
static void TestDinDoutForClockGating(void)
{
	u32 uLoopCnt = 0;
	CLKPWR_TEST eCLKPWR_TEST;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);	
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 	

	do
	{
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 1000)", uLoopCnt);
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr,	eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		//----------------------------------------------------

		switch(eCh)
		{
			case FIMC_A:
				eIpId = eFIMC0;
				eCLKPWR_TEST = CPTEST_FIMC0;
				break;
			case FIMC_B:
				eIpId = eFIMC1;
				eCLKPWR_TEST = CPTEST_FIMC1;
				break;
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// Clock off & on
		SYSC_SetClkOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);		
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetClkON_IP(eIpId);		

		//----------------------------------------------------
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);				
		FIMC_StopProcessing(eCh);
	}
	while(uLoopCnt < 1000);
	
}

// [Test DMA-in FIFO-out for power gating]
static void TestDinFoutForPowerGating(void)
{
	u32 uDispHSz, uDispVSz;
	u32 uRandNum, uLoopCnt = 0;
	CLKPWR_TEST eCLKPWR_TEST;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	do
	{
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 100)", uLoopCnt);
			
		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 0		
		Disp("\n Press any key to stop! \n");
		UART_Getc();
#else
		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*1000);
		}
#endif
		FIMC_StopProcessing(eCh);

		//----------------------------------------------------

		switch(eCh)
		{
			case FIMC_A:
				eIpId = eFIMC0;
				eCLKPWR_TEST = CPTEST_FIMC0;
				break;
			case FIMC_B:
				eIpId = eFIMC1;
				eCLKPWR_TEST = CPTEST_FIMC1;
				break;
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// X block power off & on
		SYSC_SetBlockPowerOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetBlockPowerON_IP(eIpId);		
	}
	while(uLoopCnt<100);
		
}

// [Test DMA-in FIFO-out for clock gating]
static void TestDinFoutForClockGating(void)
{
	u32 uDispHSz, uDispVSz;
	u32 uRandNum, uLoopCnt = 0;
	CLKPWR_TEST eCLKPWR_TEST;

	u8 bEnable;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	do
	{
		// FIMD window start
		switch(eCh) 
		{
			case FIMC_A:
				LCDC_SetLocalEn(LOCAL_EN, WIN0);
				LCDC_SetWinOnOff(1, WIN0);
				break;
			case FIMC_B:
				LCDC_SetLocalEn(LOCAL_EN, WIN1);	
				LCDC_SetWinOnOff(1, WIN1);
				break;
			case FIMC_C:
				LCDC_SetLocalEn(LOCAL_EN, WIN2);	
				LCDC_SetWinOnOff(1, WIN2);
				break;
			default:
				Assert(0);
				break;			
		}
		
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 100)", uLoopCnt);
		
		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		//----------------------------------------------------
		
		switch(eCh)
		{
			case FIMC_A:
				eIpId = eFIMC0;
				eCLKPWR_TEST = CPTEST_FIMC0;
				break;
			case FIMC_B:
				eIpId = eFIMC1;
				eCLKPWR_TEST = CPTEST_FIMC1;
				break;
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// Clock off & on
		SYSC_SetClkOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);		
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetClkON_IP(eIpId);		

		//----------------------------------------------------

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 0		
		Disp("\n Press any key to stop! \n");
		UART_Getc();
#else
		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*1000);
		}
#endif

		// FIMD window & local path stop
		switch(eCh) 
		{
			case FIMC_A:
				LCDC_SetLocalEn(LOCAL_DIS, WIN0);
				break;
			case FIMC_B:
				LCDC_SetLocalEn(LOCAL_DIS, WIN1);	
				break;
			case FIMC_C:
				LCDC_SetLocalEn(LOCAL_DIS, WIN2);	
				break;
			default:
				Assert(0);
				break;			
		}

		do
		{
			FIMC_GetLcdcEnstatus(eCh, &bEnable);
		}
		while(bEnable);
		FIMC_SetDinEnvid(eCh, false); // ENVID forced off	
		FIMC_StopProcessing(eCh);
	}
	while(uLoopCnt<100);
	
}

// [Test FIFO-in DMA-out for power gating]
static void TestFinDoutForPowerGating(void)
{
	u32 uFinImgHsz, uFinImgVsz;	
	u32 uRandNum, uLoopCnt = 0;
	CLKPWR_TEST eCLKPWR_TEST;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	do
	{
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 100)", uLoopCnt);
		
		FIMC_InitIpForFinDout(eCh, eCamera,
			0, 0, uFinImgHsz, uFinImgVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uLcdFbAddr, uLcdFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
		
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 0
		Disp(" Press any key to capture image!\n");
		UART_Getc();
#else
		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*1000);
		}
#endif
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		//----------------------------------------------------

		switch(eCh)
		{
			case FIMC_A:
				eIpId = eFIMC0;
				eCLKPWR_TEST = CPTEST_FIMC0;
				break;
			case FIMC_B:
				eIpId = eFIMC1;
				eCLKPWR_TEST = CPTEST_FIMC1;
				break;
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// X block power off & on
		SYSC_SetBlockPowerOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetBlockPowerON_IP(eIpId);
	}
	while(uLoopCnt<100);
	
}

// [Test FIFO-in DMA-out for clock gating]
static void TestFinDoutForClockGating(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uRandNum, uLoopCnt = 0;	
	CLKPWR_TEST eCLKPWR_TEST;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	do
	{
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 100)", uLoopCnt);
		
		FIMC_InitIpForFinDout(eCh, eCamera,
			0, 0, uFinImgHsz, uFinImgVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uLcdFbAddr, uLcdFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		//----------------------------------------------------
		
		switch(eCh)
		{
			case FIMC_A:
				eIpId = eFIMC0;
				eCLKPWR_TEST = CPTEST_FIMC0;
				break;
			case FIMC_B:
				eIpId = eFIMC1;
				eCLKPWR_TEST = CPTEST_FIMC1;
				break;
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// Clock off & on
		SYSC_SetClkOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);		
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetClkON_IP(eIpId);		

		//----------------------------------------------------
		
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 0		
		Disp(" Press any key to capture image!\n");
		UART_Getc();
#else
		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*1000);
		}
#endif
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
	}
	while(uLoopCnt<100);
	
}

// [Test DIRECT-FIFO-in DMA-out for power gating]
static void TestDFinDoutForPowerGating(void)
{
	u32 uDispHSz, uDispVSz;
	u32 uRandNum, uLoopCnt = 0;	
	CLKPWR_TEST eCLKPWR_TEST;

	do
	{
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 100)\n", uLoopCnt);
		
		// Initialize LCDC v6.0
		LCDC_InitLdi(MAIN);
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);

		Disp("\n Drawing sample image! wait for a moment! \n");
		// Generate DMA input image
		GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
		GLIB_DrawPattern(uLcdHsz, uLcdVsz);

		LCDC_Start();

		memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4);

		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//0, 0, uLcdHsz, uLcdVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uInFbAddr, uInFbAddr, NV12,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 0
		Disp(" Press any key to capture image!\n");
		UART_Getc();
#else
		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*1000);
		}
#endif
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		//----------------------------------------------------

		switch(eCh)
		{
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// X block power off & on
		SYSC_SetBlockPowerOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetBlockPowerON_IP(eIpId);		
	}
	while(uLoopCnt<100);
	
}

// [Test DIRECT-FIFO-in DMA-out for clock gating]
static void TestDFinDoutForClockGating(void)
{
	u32 uDispHSz, uDispVSz;
	u32 uRandNum, uLoopCnt = 0;	
	CLKPWR_TEST eCLKPWR_TEST;

	do
	{
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 100)\n", uLoopCnt);
		
		// Initialize LCDC v6.0
		LCDC_InitLdi(MAIN);
		LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
		LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
		//LCDC_SetOutputPath(LCD_TV);
		LCDC_SetOutputPath(LCD_TVRGB);

		Disp("\n Drawing sample image! wait for a moment! \n");
		// Generate DMA input image
		GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
		GLIB_DrawPattern(uLcdHsz, uLcdVsz);

		LCDC_Start();

		memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4);

		FIMC_InitIpForDFinDout(eCh,
			uLcdHsz, uLcdVsz,
			//0, 0, uLcdHsz, uLcdVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uInFbAddr, uInFbAddr, NV12,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		//----------------------------------------------------
		
		switch(eCh)
		{
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// Clock off & on
		SYSC_SetClkOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);		
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetClkON_IP(eIpId);		

		//----------------------------------------------------

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 0
		Disp(" Press any key to capture image!\n");
		UART_Getc();
#else
		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*1000);
		}
#endif
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
	}
	while(uLoopCnt<100);
	
}

// [Test MIPI-FIFO-in DMA-out for power gating]
static void TestMFinDoutForPowerGating(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uRandNum, uLoopCnt = 0;	
	CLKPWR_TEST eCLKPWR_TEST;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536
	eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	eHSsettle = 12;	
	//eCSISDataLaneNum = DATALANE1;	
	eCSISDataLaneNum = DATALANE2;	
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	do
	{
		//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
		MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
		CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);
		MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);
		
		Disp("\n Sensor initialization! please wait.......... \n");
		//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
		FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
		FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
		
		DelayfrTimer(milli, 3000); // for sensor initialization
	
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 100)", uLoopCnt);
		
		FIMC_InitIpForMFinDout(eCh,
			0, 0, uFinImgHsz, uFinImgVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uLcdFbAddr, uLcdFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange,eDataAlign);
		
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 0		
		Disp(" Press any key to capture image!\n");
		UART_Getc();
#else
		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*1000);
		}
#endif
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		CSI_DisableCSI();
		MIPICAM_Disable();

		//----------------------------------------------------

		switch(eCh)
		{
			case FIMC_A:
				eIpId = eFIMC0;
				eCLKPWR_TEST = CPTEST_FIMC0;
				break;
			case FIMC_B:
				eIpId = eFIMC1;
				eCLKPWR_TEST = CPTEST_FIMC1;
				break;
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// X block power off & on
		SYSC_SetBlockPowerOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetBlockPowerON_IP(eIpId);
	}
	while(uLoopCnt<100);
	
}

// [Test MIPI-FIFO-in DMA-out for clock gating]
static void TestMFinDoutForClockGating(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uRandNum, uLoopCnt = 0;	
	CLKPWR_TEST eCLKPWR_TEST;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536
	eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	eHSsettle = 12;	
	//eCSISDataLaneNum = DATALANE1;	
	eCSISDataLaneNum = DATALANE2;	
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);
	
	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
	
	do
	{
		uLoopCnt = uLoopCnt + 1;
		Disp("\n\n uLoopCnt = %d (max: 100)", uLoopCnt);
		
		FIMC_InitIpForMFinDout(eCh,
			0, 0, uFinImgHsz, uFinImgVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uLcdFbAddr, uLcdFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange,eDataAlign);

		//----------------------------------------------------
		
		switch(eCh)
		{
			case FIMC_A:
				eIpId = eFIMC0;
				eCLKPWR_TEST = CPTEST_FIMC0;
				break;
			case FIMC_B:
				eIpId = eFIMC1;
				eCLKPWR_TEST = CPTEST_FIMC1;
				break;
			case FIMC_C:
				eIpId = eFIMC2;
				eCLKPWR_TEST = CPTEST_FIMC2;
				break;
			default:
				Assert(0);
				break;
		}

		// Clock off & on
		SYSC_SetClkOFF_IP(eIpId);
		DelayfrTimer(milli, SBLK_GetRand()%10);			
		AccessOtherRegsInClockPowerOff(eCLKPWR_TEST, 1);
		SYSC_SetClkON_IP(eIpId);	

		//----------------------------------------------------
		
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 0		
		Disp(" Press any key to capture image!\n");
		UART_Getc();
#else
		uRandNum = SBLK_GetRand();
		if((uRandNum==0)||((uRandNum%10)==0))
		{
			DelayfrTimer(milli, 1000);
		}
		else
		{
			DelayfrTimer(milli, (uRandNum%10)*1000);
		}
#endif
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
	}
	while(uLoopCnt<100);

	CSI_DisableCSI();
	MIPICAM_Disable();
}

// [Test power mode]
static void TestPowerMode(void)
{

}

// [Test DMA-in FIFO-out for checking under-run]
static void TestDinFoutForCheckingUnderRun(void)
{
	u32 uInImgHsz = 1920, uInImgVsz = 1088; // Full HD
	//u32 uInImgHsz = 1760, uInImgVsz = 992; // 14
	//u32 uInImgHsz = 1728, uInImgVsz = 976;
	//u32 uInImgHsz = 1712, uInImgVsz = 960; // 0
	//u32 uInImgHsz = 1680, uInImgVsz = 944; // 0
	//u32 uInImgHsz = 1600, uInImgVsz = 896; // 0
	//u32 uInImgHsz = 1440, uInImgVsz = 816;
	//u32 uInImgHsz = 1424, uInImgVsz = 800;
	//u32 uInImgHsz = 1408, uInImgVsz = 784;
	//u32 uInImgHsz = 1360, uInImgVsz = 768;
	//u32 uInImgHsz = 1344, uInImgVsz = 752;
	//u32 uInImgHsz = 1296, uInImgVsz = 736;
	//u32 uInImgHsz = 1280, uInImgVsz = 720; // HD	
	//u32 uInImgHsz = 1264, uInImgVsz = 704;
	//u32 uInImgHsz = 1200, uInImgVsz = 688;
	//u32 uInImgHsz = 1120, uInImgVsz = 640;
	//u32 uInImgHsz = 960, uInImgVsz = 544;
	//u32 uInImgHsz = 944, uInImgVsz = 528;
	//u32 uInImgHsz = 928, uInImgVsz = 528;
	//u32 uInImgHsz = 880, uInImgVsz = 512;
	//u32 uInImgHsz = 800, uInImgVsz = 480; // WVGA	
	//u32 uInImgHsz = 1120, uInImgVsz = 720;
	//u32 uInImgHsz = 960, uInImgVsz = 720;
	//u32 uInImgHsz = 800, uInImgVsz = 720;
	//u32 uInImgHsz = 640, uInImgVsz = 720;
	//u32 uInImgHsz = 560, uInImgVsz = 720;
	//u32 uInImgHsz = 512, uInImgVsz = 720;
	//u32 uInImgHsz = 496, uInImgVsz = 720;
	//u32 uInImgHsz = 480, uInImgVsz = 720;
	//u32 uInImgHsz = 464, uInImgVsz = 720;
	//u32 uInImgHsz = 1280, uInImgVsz = 560;
	//u32 uInImgHsz = 1280, uInImgVsz = 400;
	//u32 uInImgHsz = 1280, uInImgVsz = 384;
	//u32 uInImgHsz = 1280, uInImgVsz = 368;
	//u32 uInImgHsz = 1280, uInImgVsz = 352;
	//u32 uInImgHsz = 1280, uInImgVsz = 320;
	//u32 uInImgHsz = 1280, uInImgVsz = 240;
	//u32 uInImgHsz = 1040, uInImgVsz = 480;
	//u32 uInImgHsz = 1040, uInImgVsz = 480;
	//u32 uInImgHsz = 1008, uInImgVsz = 448;
	//u32 uInImgHsz = 992, uInImgVsz = 432;
	//u32 uInImgHsz = 960, uInImgVsz = 400;
		
	u32 i;
	u32 uDispHSz, uDispVSz;
	ROT_FLIP eRotFlipList[8] = 
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY			
		};	
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = 480;
	uDispVSz = 358;		
	//uDispVSz = 800;		
	//uDispHSz = 800;
	//uDispVSz = 480;	
	//uDispHSz = 1366;
	//uDispVSz = 768;	
	//uDispHSz = 1280;
	//uDispVSz = 1024;	
	//uDispHSz = 1024;
	//uDispVSz = 768;	
	//uDispHSz = 480;
	//uDispVSz = 272;	
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
#if 0	
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);
#else
	GLIB_InitInstance(uLcdFbAddr, 640, 480, eBpp);
	GLIB_DrawPattern(640, 480);
#endif

	for(i=0; i<8; i++)
	{
		FIMC_InitIpForDinDout(eCh,
#if 0
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
#else
			640, 480, 0,
			0, 0, 640, 480,
#endif
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler		
			uInFbAddr, TILE_NV12,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);	
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		FIMC_InitIpForDinFout(eCh,
			uInImgHsz, uInImgVsz, 0,
			0, 0, uInImgHsz, uInImgVsz,
			uInFbAddr, TILE_NV12, false,
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			eRotFlipList[i], eCscEquation, eYCbCrRange);
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();
	}
}

// [Test DMA-in FIFO-out for applying the 1st under-run solution]
static void TestDinFoutForApplyingThe1stUnderRunSol(void)
{
	// Vertical line skip using input DMA line skip 
	
	u32 uInImgHsz = 1920, uInImgVsz = 1088; // Full HD
	//u32 uInImgHsz = 1904, uInImgVsz = 1072;
	//u32 uInImgHsz = 1888, uInImgVsz = 1056;
	//u32 uInImgHsz = 1872, uInImgVsz = 1040;
	//u32 uInImgHsz = 1856, uInImgVsz = 1024;
	//u32 uInImgHsz = 1840, uInImgVsz = 1008;
	//u32 uInImgHsz = 1760, uInImgVsz = 992;
	//u32 uInImgHsz = 1744, uInImgVsz = 976; // 0
	//u32 uInImgHsz = 1680, uInImgVsz = 944; // 0
	//u32 uInImgHsz = 1520, uInImgVsz = 864;
	//u32 uInImgHsz = 1376, uInImgVsz = 784;
	//u32 uInImgHsz = 1360, uInImgVsz = 768;
	//u32 uInImgHsz = 1344, uInImgVsz = 752; // 0
	//u32 uInImgHsz = 1296, uInImgVsz = 752;
	//u32 uInImgHsz = 1280, uInImgVsz = 720; // HD
	//u32 uInImgHsz = 800, uInImgVsz = 480; // WVGA	
		
	u32 i;
	u32 uDispHSz, uDispVSz;
	ROT_FLIP eRotFlipList[8] = 
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY			
		};	
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = 480;
	uDispVSz = 358; 	
	//uDispVSz = 800; 
	//uDispHSz = 800;
	//uDispVSz = 480;	
	//uDispHSz = 1366;
	//uDispVSz = 768;		
	//uDispHSz = 1280;
	//uDispVSz = 1024;	
	//uDispHSz = 1024;
	//uDispVSz = 768;	
	//uDispHSz = 480;
	//uDispVSz = 272;	
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

#if 0	
		GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
		GLIB_DrawPattern(uLcdHsz, uLcdVsz);
#else
		GLIB_InitInstance(uLcdFbAddr, 640, 480, eBpp);
		GLIB_DrawPattern(640, 480);
#endif

	for(i=0; i<8; i++)
	{
		FIMC_InitIpForDinDout(eCh,
#if 0
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
#else
			640, 480, 0,
			0, 0, 640, 480,
#endif
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler		
			uInFbAddr, TILE_NV12,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);	
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		FIMC_InitIpForDinFout(eCh,
			//uInImgHsz, uInImgVsz, 0,
			//0, 0, uInImgHsz, uInImgVsz,			
			uInImgHsz, uInImgVsz, 1,
			0, 0, uInImgHsz, (uInImgVsz/2)-((uInImgVsz/2)%16),
			uInFbAddr, TILE_NV12, false,
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			eRotFlipList[i], eCscEquation, eYCbCrRange);
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();
	}
}

// [Test DMA-in FIFO-out for applying the 2nd under-run solution]
static void TestDinFoutForApplyingThe2ndUnderRunSol(void)
{
	// Input DMA FIFO control using FIFO_CTRL
	
	u32 uInImgHsz = 1920, uInImgVsz = 1088; // Full HD
	//u32 uInImgHsz = 1280, uInImgVsz = 720; // HD
	//u32 uInImgHsz = 800, uInImgVsz = 480; // WVGA	
		
	u32 i;
	u32 uDispHSz, uDispVSz;
	ROT_FLIP eRotFlipList[8] = 
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY			
		};	
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = 480;
	uDispVSz = 358; 
	//uDispVSz = 800;
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	for(i=0; i<8; i++)
	{
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler		
			uInFbAddr, TILE_NV12,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);	
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		FIMC_InitIpForDinFout(eCh,
			uInImgHsz, uInImgVsz, 0,
			0, 0, uInImgHsz, uInImgVsz,		
			uInFbAddr, TILE_NV12, true,
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			eRotFlipList[i], eCscEquation, eYCbCrRange);
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();
	}
}

// [Test DMA-in FIFO-out for applying all the under-run solutions]
static void TestDinFoutForApplyingAllTheUnderRunSols(void)
{
	// Vertical line skip using input DMA line skip + Input DMA FIFO control using FIFO_CTRL
	
	u32 uInImgHsz = 1920, uInImgVsz = 1088; // Full HD
	//u32 uInImgHsz = 1280, uInImgVsz = 720; // HD
	//u32 uInImgHsz = 800, uInImgVsz = 480; // WVGA	
		
	u32 i;
	u32 uDispHSz, uDispVSz;
	ROT_FLIP eRotFlipList[8] = 
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY			
		};	
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = 480;
	uDispVSz = 358; 
	//uDispVSz = 800;
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	for(i=0; i<8; i++)
	{
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler		
			uInFbAddr, TILE_NV12,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);	
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		FIMC_InitIpForDinFout(eCh,
			//uInImgHsz, uInImgVsz, 0,
			//0, 0, uInImgHsz, uInImgVsz,		
			uInImgHsz, uInImgVsz, 1,
			0, 0, uInImgHsz, (uInImgVsz/2)-((uInImgVsz/2)%16),			
			uInFbAddr, TILE_NV12, true,
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			eRotFlipList[i], eCscEquation, eYCbCrRange);
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();
	}
}

// [Test DMA-in FIFO-out for adding BUS load using FIMD]
static void TestDinFoutForAddingBusLoadUsingFIMD(void)
{
	u32 uInImgHsz = 1920, uInImgVsz = 1088; // Full HD
	//u32 uInImgHsz = 1280, uInImgVsz = 720; // HD	
	//u32 uInImgHsz = 800, uInImgVsz = 480; // WVGA	
		
	u32 i;
	u32 uDispHSz, uDispVSz;
	ROT_FLIP eRotFlipList[8] = 
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY			
		};	
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = 480;
	uDispVSz = 358;		
	//uDispVSz = 800;		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN0);
			break;
#if 0			
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
#endif			
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);

	//LCDC_InitWin(RGB24, uLcdHsz, uLcdVsz, 0, 0, uLcdHsz, uLcdVsz, 0, 0, uOutFbAddr, WIN1, false); 	
	//LCDC_SetWinOnOff(1, WIN1);	
	//LCDC_InitWin(RGB24, uLcdHsz, uLcdVsz, 0, 0, uLcdHsz, uLcdVsz, 0, 0, uLcdOutFbAddr, WIN2 false); 	
	//LCDC_SetWinOnOff(1, WIN2);	
	LCDC_InitWin(RGB24, uLcdHsz, uLcdVsz, 0, 0, uLcdHsz, (uLcdVsz-358)-((uLcdVsz-358)%8), 0, 358, uOutFbAddr, WIN3, false); 	
	LCDC_SetWinOnOff(1, WIN3);	
	LCDC_InitWin(RGB24, uLcdHsz, uLcdVsz, 0, 0, uLcdHsz, (uLcdVsz-358)-((uLcdVsz-358)%8), 0, 358, uLcdOutFbAddr, WIN4, false); 	
	LCDC_SetWinOnOff(1, WIN4);	
	//LCDC_SetBlendingType(PER_PLANE, ALPHA0_PLANE, WIN1);
	//LCDC_SetBlendingType(PER_PLANE, ALPHA0_PLANE, WIN2);
	LCDC_SetBlendingType(PER_PLANE, ALPHA0_PLANE, WIN3);
	LCDC_SetBlendingType(PER_PLANE, ALPHA0_PLANE, WIN4);	
	LCDC_SetAlphaClear(ALLWIN);
	//LCDC_SetAlpha(120, 120, 120, 0x0, 0x0, 0x0, WIN1);
	//LCDC_SetAlpha(120, 120, 120, 0x0, 0x0, 0x0, WIN2);
	LCDC_SetAlpha(230, 230, 230, 0x0, 0x0, 0x0, WIN3);
	LCDC_SetAlpha(230, 230, 230, 0x0, 0x0, 0x0, WIN4);

	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	GLIB_InitInstance(uOutFbAddr, uLcdHsz, (uLcdVsz-358)-((uLcdVsz-358)%8), eBpp);
	GLIB_DrawPattern(uLcdHsz, (uLcdVsz-358)-((uLcdVsz-358)%8));

	GLIB_InitInstance(uLcdOutFbAddr, uLcdHsz, (uLcdVsz-358)-((uLcdVsz-358)%8), eBpp);
	GLIB_DrawPattern2(uLcdHsz, (uLcdVsz-358)-((uLcdVsz-358)%8));	

	for(i=0; i<8; i++)
	{
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler		
			uInFbAddr, TILE_NV12,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);	
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		FIMC_InitIpForDinFout(eCh,
			uInImgHsz, uInImgVsz, 0,
			0, 0, uInImgHsz, uInImgVsz,
			uInFbAddr, TILE_NV12, false,
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			eRotFlipList[i], eCscEquation, eYCbCrRange);
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();
	}
}

// [Test DMA-in FIFO-out for applying the 3rd under-run solution]
static void TestDinFoutForApplyingThe3rdUnderRunSol(void)
{
	// Horizontal scale down using IPC
	
	u32 uInImgHsz = 1920, uInImgVsz = 1088; // Full HD
	//u32 uInImgHsz = 1280, uInImgVsz = 720; // HD
	//u32 uInImgHsz = 800, uInImgVsz = 480; // WVGA	

	u32 uMaxInImgHszOfIpc = 1024;
			
	u32 i;
	u32 uDispHSz, uDispVSz;
	ROT_FLIP eRotFlipList[8] = 
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY			
		};	

	Assert(eCh == FIMC_B);
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = 480;
	uDispVSz = 358; 	
	//uDispVSz = 800; 
	//uDispHSz = 800;
	//uDispVSz = 480;	
	switch(eCh)
	{
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	// Initialize IPC v1.0
#if 1
	//IPC_InitIp(uMaxInImgHszOfIpc, uDispVSz, IPC_Scale);
	IPC_InitIp_Exp(uMaxInImgHszOfIpc, uDispVSz, uDispHSz, uDispVSz, IPC_Scale);
	IPC_FILED_ID_Mode(INTERNAL, BYUSER);
	IPC_FILED_ID_Control(IPC_TOP_FIELD);
	IPC_On();		
#endif
	for(i=0; i<8; i++)
	{
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler		
			uInFbAddr, TILE_NV12,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);	
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		FIMC_InitIpForDinFout(eCh,
			uInImgHsz, uInImgVsz, 0,
			0, 0, uInImgHsz, uInImgVsz,		
			uInFbAddr, TILE_NV12, false,
			//uDispHSz, uDispVSz, // Fifo out display size
			uMaxInImgHszOfIpc, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			eRotFlipList[i], eCscEquation, eYCbCrRange);
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();
	}
#if 1
	IPC_Off();
#endif
}

// [Test DMA-in DMA-out for getting operation time]
static void TestDinDoutForGettingOperationTime(void)
{
	u32 uOutCnt, uInCnt, uRotAndFlipCnt;
	u32 uMicroSeconds;

	// Full HD(1920x1088), HD(1280x720), WVGA(800x480)
	u32 uInImgHsz[3] = {1920, 1280, 800};
	u32 uInImgVsz[3] = {1088, 720, 480};

	u32 uOutImgHsz[2] = {480, 480};
	u32 uOutImgVsz[2] = {352, 800};
	u32 uTempOutImgHsz, uTempOutImgVsz;

	ROT_FLIP eRotFlipList[8] = 
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY			
		};		
	
	for(uOutCnt=0; uOutCnt<2; uOutCnt++) // Output image size
	{
		for(uInCnt=0; uInCnt<3; uInCnt++) // Input image size
		{
			for(uRotAndFlipCnt=0; uRotAndFlipCnt<8; uRotAndFlipCnt++) // Rotation & flip
			{
				switch(eRotFlipList[uRotAndFlipCnt])
				{
					case ROT_NO_FLIP_NO:
					case ROT_NO_FLIP_X:
					case ROT_NO_FLIP_Y:
					case ROT_NO_FLIP_XY:
						uTempOutImgHsz = uOutImgHsz[uOutCnt];
						uTempOutImgVsz = uOutImgVsz[uOutCnt];
						break;
					case ROT_90_FLIP_NO:
					case ROT_90_FLIP_X:
					case ROT_90_FLIP_Y:
					case ROT_90_FLIP_XY:
						uTempOutImgHsz = uOutImgVsz[uOutCnt];
						uTempOutImgVsz = uOutImgHsz[uOutCnt];					
						break;
					default:
						Assert(0);
						break;
				}

				FIMC_InitIpForDinDout(eCh,
					uInImgHsz[uInCnt], uInImgVsz[uInCnt], 0,
					0, 0, uInImgHsz[uInCnt], uInImgVsz[uInCnt],
					uLcdFbAddr, TILE_NV12,
					uOutImgHsz[uOutCnt], uOutImgVsz[uOutCnt],
					0, 0, uTempOutImgHsz, uTempOutImgVsz, 0, // Scaler		
					uInFbAddr, eBpp,		
					eRotFlipList[uRotAndFlipCnt], eCscEquation, eYCbCrRange, false);	

				StartTimer(0);
				
				bFimcDone = false;
				FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
				WaitForFimcDone(eCh);
				FIMC_StopProcessing(eCh);					

			   	uMicroSeconds = StopTimer(0);
				Disp("\n uOutCnt = %d, uInCnt = %d, uRotAndFlipCnt = %d, Taken time: %d [usec] \n",
					uOutCnt, uInCnt, uRotAndFlipCnt, uMicroSeconds);				
			}
		}
	}
	
}


static void TestFinDoutForMaxPclk(void)
{
	// 12M monaco camera
	
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	IMG_RESOLUTION eImgResolution;
	u32 uNum;

	u32 uOutImgHsz, uOutImgVsz;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	Disp("\n 0: 1280x720 \n 1: 800x480 \n 2: 720x480 \n 3: 640x480 \n 4: 320x240 \n");
	Disp("\n Select camera output resolution: ");
	uNum = UART_GetIntNum();
	
	switch(uNum)
	{
		case 0: 
			eImgResolution = HD720; // 1280x720		
			break;	
		case 1: 
			eImgResolution = WVGA; // 800x480
			break;			
		case 2: 
			eImgResolution = SD480; // 720x480
			break;
		case 3: 
			eImgResolution = VGA; // 640x480
			break;			
		case 4: 
			eImgResolution = QVGA; // 320x240
			break;
		default:
			Assert(0);
			break;			
	}

	FIMC_InitCamera(CE143Monaco, CBYCRY, eImgResolution);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	if(uFinImgHsz >= uLcdHsz)
	{
		uOutImgHsz = uLcdHsz;
	}
	else
	{
		uOutImgHsz = uFinImgHsz;
	}
	if(uFinImgVsz >= uLcdVsz)
	{
		uOutImgVsz = uLcdVsz;
	}
	else
	{
		uOutImgVsz = uFinImgVsz;
	}
	
	FIMC_InitIpForFinDout(eCh, eCamera,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uOutImgHsz, uOutImgVsz, 0,
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

	Disp(" Press any key to capture image!\n");

	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		Disp("\n Current frame count is %d\n", uFrmCnt);
	}

	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
}

// [Test DMA-in DMA-out for core clock source and divider]
static void TestDinDoutForCoreClkSrcAndDivider(void)
{
	u32 i;

	u32 uClkDivValueOfFimcA[5] = {1, 8, 4, 12, 16};
	CLKMUX_STATE_ID eIdOfFimcA[5] = {
		eCLKMUX_FIMC0LCLK_XXTI,
		eCLKMUX_FIMC0LCLK_XUSBXTI,
		//eCLKMUX_FIMC0LCLK_SHDMI27M,
		//eCLKMUX_FIMC0LCLK_SUSBPHY0,
		//eCLKMUX_FIMC0LCLK_SUSBPHY1,
		//eCLKMUX_FIMC0LCLK_SHDMIPHY,
		eCLKMUX_FIMC0LCLK_SMPLL,
		eCLKMUX_FIMC0LCLK_SEPLL,
		eCLKMUX_FIMC0LCLK_SVPLL};

	u32 uClkDivValueOfFimcB[5] = {1, 8, 4, 12, 16};		
	CLKMUX_STATE_ID eIdOfFimcB[5] = {
		eCLKMUX_FIMC1LCLK_XXTI,
		eCLKMUX_FIMC1LCLK_XUSBXTI,
		//eCLKMUX_FIMC1LCLK_SHDMI27M,
		//eCLKMUX_FIMC1LCLK_SUSBPHY0,
		//eCLKMUX_FIMC1LCLK_SUSBPHY1,
		//eCLKMUX_FIMC1LCLK_SHDMIPHY,
		eCLKMUX_FIMC1LCLK_SMPLL,
		eCLKMUX_FIMC1LCLK_SEPLL,
		eCLKMUX_FIMC1LCLK_SVPLL};

	u32 uClkDivValueOfFimcC[5] = {1, 8, 4, 12, 16};
	CLKMUX_STATE_ID eIdOfFimcC[5] = {
		eCLKMUX_FIMC2LCLK_XXTI,
		eCLKMUX_FIMC2LCLK_XUSBXTI,
		//eCLKMUX_FIMC2LCLK_SHDMI27M,
		//eCLKMUX_FIMC2LCLK_SUSBPHY0,
		//eCLKMUX_FIMC2LCLK_SUSBPHY1,
		//eCLKMUX_FIMC2LCLK_SHDMIPHY,
		eCLKMUX_FIMC2LCLK_SMPLL,
		eCLKMUX_FIMC2LCLK_SEPLL,
		eCLKMUX_FIMC2LCLK_SVPLL};

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	for(i=0; i<5; i++)
	{
		// Generate DMA input image
		GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
		switch(i%2)
		{
			case 0:
				GLIB_DrawPattern(uLcdHsz, uLcdVsz); 
				break;
			case 1:
				GLIB_DrawPattern2(uLcdHsz, uLcdVsz); 
				break;
			default:
				Assert(0);
				break;
		}
		
		switch(eCh)
		{
			case FIMC_A:
				SYSC_SetClkDiv(eCLKDIV_FIMC0, uClkDivValueOfFimcA[i]);
				SYSC_SetClkMuxState(eIdOfFimcA[i]);			
				break;
			case FIMC_B:
				SYSC_SetClkDiv(eCLKDIV_FIMC1, uClkDivValueOfFimcB[i]);
				SYSC_SetClkMuxState(eIdOfFimcB[i]);				
				break;
			case FIMC_C:
				SYSC_SetClkDiv(eCLKDIV_FIMC2, uClkDivValueOfFimcC[i]);
				SYSC_SetClkMuxState(eIdOfFimcC[i]);			
				break;
			default:
				Assert(0);
				break;	
		}
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr,	eBpp,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
			uLcdOutFbAddr, eBpp,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);		
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();		
	}

	SYSC_SetClkDiv(eCLKDIV_FIMC0, 4);
	SYSC_SetClkMuxState(eCLKMUX_FIMC0LCLK_SMPLL);	
	SYSC_SetClkDiv(eCLKDIV_FIMC1, 4);
	SYSC_SetClkMuxState(eCLKMUX_FIMC1LCLK_SMPLL);	
	SYSC_SetClkDiv(eCLKDIV_FIMC2, 4);
	SYSC_SetClkMuxState(eCLKMUX_FIMC2LCLK_SMPLL);
}

// [Test FIFO-in DMA-out for MCLK source and divider]
static void TestFinDoutForMclkSrcAndDivider(void)
{
	u32 i;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uClkDivValueOfCamA[5] = {1, 4, 16, 8, 12};
	CLKMUX_STATE_ID eIdOfCamA[5] = {
		eCLKMUX_CAM0_XXTI,
		eCLKMUX_CAM0_XUSBXTI,
		//eCLKMUX_CAM0_SHDMI27M,
		//eCLKMUX_CAM0_SUSBPHY0,
		//eCLKMUX_CAM0_SUSBPHY1,
		//eCLKMUX_CAM0_SHDMIPHY,
		eCLKMUX_CAM0_SMPLL,
		eCLKMUX_CAM0_SEPLL,
		eCLKMUX_CAM0_SVPLL};

	u32 uClkDivValueOfCamB[5] = {1, 4, 16, 8, 12};		
	CLKMUX_STATE_ID eIdOfCamB[5] = {
		eCLKMUX_CAM1_XXTI,
		eCLKMUX_CAM1_XUSBXTI,
		//eCLKMUX_CAM1_SHDMI27M,
		//eCLKMUX_CAM1_SUSBPHY0,
		//eCLKMUX_CAM1_SUSBPHY1,
		//eCLKMUX_CAM1_SHDMIPHY,
		eCLKMUX_CAM1_SMPLL,
		eCLKMUX_CAM1_SEPLL,
		eCLKMUX_CAM1_SVPLL};

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	for(i=0; i<5; i++)
	{
		switch(eCamera)
		{
			case ITU_CAMERA_A:
				SYSC_SetClkDiv(eCLKDIV_CAM0, uClkDivValueOfCamA[i]);
				SYSC_SetClkMuxState(eIdOfCamA[i]);				
				break;
			case ITU_CAMERA_B:
				SYSC_SetClkDiv(eCLKDIV_CAM1, uClkDivValueOfCamB[i]);
				SYSC_SetClkMuxState(eIdOfCamB[i]);				
				break;
			default:
				Assert(0);
				break;
		}
		
		FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
		FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
		
		FIMC_InitIpForFinDout(eCh, eCamera,
			0, 0, uFinImgHsz, uFinImgVsz,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0,
			uLcdFbAddr, uLcdFbAddr, eBpp,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
	
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

		Disp("\n\n ===== Check MCLK using oscilloscope! ===== \n\n");
		Disp(" Press any key to capture image!\n");
		UART_Getc();
#if 0	
		while(GetKey() == 0)
		{
#ifndef FIMC_POLLING
			bFimcDone = false;
			while(!bFimcDone);
#else // polling
			Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
			Assert(0);
			return;
#endif		
			FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
			Disp("\n Current frame count is %d\n", uFrmCnt);
		}
#endif	
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		Disp("\n Press any key to go next! \n");
		UART_Getc();		
	}

	SYSC_SetClkDiv(eCLKDIV_CAM0, 4); // 96[MHz] / 4 = 24[MHz]
	SYSC_SetClkMuxState(eCLKMUX_CAM0_SEPLL);	
	SYSC_SetClkDiv(eCLKDIV_CAM1, 4); // 96[MHz] / 4 = 24[MHz]
	SYSC_SetClkMuxState(eCLKMUX_CAM1_SEPLL);
}

#if 0
static void TestForDebugging_0(void)
{
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(RGB16, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	//LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	GLIB_InitInstance(uLcdFbAddr, 320, 240, RGB16); 
	GLIB_FillRectangle(0,0,320,240,C_YELLOW);

	FIMC_InitIpForDinDout(eCh,
		320, 240, 0,
		0, 0, 320, 240,
		uLcdFbAddr, RGB565,
		4096, 4096,
		0, 0, 4096, 4096, 0, // Scaler 
		uInFbAddr, CRYCBY,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);  
	FIMC_WaitForProcessingDone(eCh); 
	FIMC_StopProcessing(eCh);

	FIMC_InitIpForDinDout(eCh,
		4096, 4096, 0,
		0, 0, 4096, 4096,
		uInFbAddr, CRYCBY,
		320, 240,
		0, 0, 320, 240, 0, // Scaler 
		uLcdOutFbAddr, RGB565,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);  
	FIMC_WaitForProcessingDone(eCh); 
	FIMC_StopProcessing(eCh);

}

static void TestForDebugging_1(void)
{
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4);

	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp); 
	GLIB_FillRectangle(0,0,uLcdHsz,uLcdVsz,C_YELLOW);

	FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
	                    0, 0, uLcdHsz, uLcdVsz, uLcdFbAddr, eBpp,
	                    //1920, 1080, 0, 0, 1920, 1080, // Scaler 
	                    //1600, 1200, 0, 0, 1600, 1200, // Scaler 
	                    320, 240, 0, 0, 320, 240, 0, // Scaler 
	                    uInFbAddr, NV12, ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);  
	FIMC_WaitForProcessingDone(eCh); 
	FIMC_StopProcessing(eCh);

	FIMC_InitIpForDinDout(eCh,
			      //1920, 1080, 0, 0, 1920, 1080, uInFbAddr, NV12,
			      //1600, 1200, 0, 0, 1600, 1200, uInFbAddr, NV12,
			      320, 240, 0, 0, 0, 320, 240, uInFbAddr, NV12,
	                    320, 240, 32, 32, 160, 112, 0, // Scaler -> NO
	                    //320, 240, 0, 0, 160, 112, // Scaler -> NO
	                    //320, 240, 0, 0, 160, 128, // Scaler -> NO
	                    //320, 240, 0, 0, 176, 112, // Scaler -> NO
	                    //320, 240, 0, 0, 320, 240, // Scaler -> YES	                    
	                    uLcdOutFbAddr, eBpp, ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);  
	FIMC_WaitForProcessingDone(eCh); 
	FIMC_StopProcessing(eCh);
}

static void TestForDebugging_2(void)
{
	//IMG_FMT eImgFmt = CRYCBY;
	IMG_FMT eImgFmt = NV12;
		
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4);

	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp); 
	//GLIB_FillRectangle(0,0,uLcdHsz,uLcdVsz,C_YELLOW);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 

	FIMC_InitIpForDinDout(eCh,
				uLcdHsz, uLcdVsz, 0,
	                    0, 0, uLcdHsz, uLcdVsz,
	                    uLcdFbAddr, eBpp,
	                    400, 288,
	                    0, 0, 400, 288, 0, // Scaler 
	                    uInFbAddr, eImgFmt,
	                    ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);

	FIMC_InitIpForDinDout(eCh,
			      400, 288, 0,
			      0, 0, 400, 288,
			      uInFbAddr, eImgFmt,
	                    800, 480,
	                    0, 0, 320, 240, 0, // Scaler      
	                    uLcdOutFbAddr, RGB16,
	                    ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);
}

static void TestForDebugging_3(void)
{
	u32 i;

	u32 uImgHsz = 240, uImgVsz = 320;
	
	Assert(eCh == FIMC_C);

	// Initialize LCDC v6.0 -> To get 'uLcdHsz', 'uLcdVsz'
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	
	Disp("\n Drawing sample image! wait for a moment! \n");
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uImgHsz, uImgVsz, eBpp);
	GLIB_DrawPattern(uImgHsz, uImgVsz);

	// [Image scaling]
	//---------------
	Disp("\n Image scaling \n");

	InitVariables();	

	// Initialize LCDC v6.0
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	//LCDC_SetOutputPath(LCD_TV);
	LCDC_SetOutputPath(LCD_TVRGB);
	LCDC_Start();

	memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4);
	
	FIMC_InitIpForDFinDout(eCh,
		uImgHsz, uImgVsz,
		//0, 0, uLcdHsz, uLcdVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uImgHsz, uImgVsz, 0,
		uLcdOutFbAddr, uLcdOutFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
	
#if 0	
	switch(i)
	{
		case 0:
			uOutImgHsz = 96;
			uOutImgVsz = 96;
			break;
		case 1:
			uOutImgHsz = 144;
			uOutImgVsz = 144;
			break;
		case 2:
			uOutImgHsz = 192;
			uOutImgVsz = 192;
			break;
		case 3:
			uOutImgHsz = (uLcdHsz/1)-((uLcdHsz/1)%16);
			uOutImgVsz = (uLcdVsz/1)-((uLcdVsz/1)%16);
			break;
		default:
			Assert(0);
			break;
	}
#else
	uOutImgHsz = uImgHsz, uOutImgVsz = uImgVsz;
#endif

	FIMC_SetFinImgScaler(eCh,uImgHsz, uImgVsz, uOutImgHsz, uOutImgVsz);
	FIMC_SetDoutImgSize(eCh,uOutImgHsz, uOutImgVsz, eBpp, 0);

	FIMC_SetDoutImgOffset(eCh, eBpp, 0, 0);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	// Initialize LCDC v6.0
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
}

static void TestForDebugging_4(void)
{
	u32 i;
	u32 uLineSkipVal = 1;
	
	IMG_FMT eImgFmtList = RGB24;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false);
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 

	Disp("\n Press any key to go next! \n");
	UART_Getc();

	// [Output DMA line skip]
	//-------------------

	memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	
	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uLcdFbAddr, eBpp,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz/(uLineSkipVal+1), uLineSkipVal, // Scaler	
		uInFbAddr, eImgFmtList,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	LCDC_InitIp(eBpp, uInFbAddr, WIN0, false); 
	LCDC_Start(); 
	Disp("\n Press any key to go next! \n");
	UART_Getc();
	
	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uInFbAddr, eImgFmtList,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
		uLcdOutFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);	

	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
	LCDC_Start(); 
}

static void TestForDebugging_5(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;
	u32 uFrmCnt;

	Assert(eCh == FIMC_C);

	uLcdHsz = 240, uLcdVsz = 320;
	
	Disp("\n Drawing sample image! wait for a moment! \n");
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 

	// Initialize LCDC v6.0
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	//LCDC_SetOutputPath(LCD_TV);
	LCDC_SetOutputPath(LCD_TVRGB);
	LCDC_Start();

	memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitIpForDFinDout(eCh,
		uLcdHsz, uLcdVsz,
		//0, 0, uLcdHsz, uLcdVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uInFbAddr, uInFbAddr, NV12,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);
	
	// ============ Notice !!! ============
	FIMC_StartProcessingForScalerBypass(eCh);
	// ================================
}

static void TestForDebugging_6(void)
{
	u32 i, j;
	u32 uImgHsz, uImgVsz;
	u32 bResult;

	Assert(eCh == FIMC_C);
		
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	uImgHsz = 8192;
	uImgVsz = 16;

	Disp("\n Generating sample image... \n");
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uImgHsz, uImgVsz, eBpp);
	GLIB_DrawPattern(uImgHsz, uImgVsz); 		
	
	memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	// One2one
	FIMC_InitIpForDinDout(eCh,
		uImgHsz, uImgVsz, 0,
		0, 0, uImgHsz, uImgVsz,
		uLcdFbAddr, eBpp,
		uImgHsz, uImgVsz,
		0, 0, uImgHsz, uImgVsz, 0, // Scaler		
		uInFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, true);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	FIMC_InitIpForDinDout(eCh,
		uImgHsz, uImgVsz, 0,
		0, 0, uImgHsz, uImgVsz,
		uInFbAddr, eBpp,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
		uLcdOutFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
}

static void TestForDebugging_7(void)
{
	u32 uDispHSz, uDispVSz;
	u32 uFrmCnt;

	IMG_FMT eImgFmt = NV12;

	Assert(eCh == FIMC_C);

	// Initialize LCDC v6.0 -> To get 'uLcdHsz', 'uLcdVsz'
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	Disp("\n Drawing sample image! wait for a moment! \n");
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 

	// Initialize LCDC v6.0
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	//LCDC_SetOutputPath(LCD_TV);
	LCDC_SetOutputPath(LCD_TVRGB);
	LCDC_Start();

	memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitIpForDFinDout(eCh,
		uLcdHsz, uLcdVsz,
		//0, 0, uLcdHsz, uLcdVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uInFbAddr, uInFbAddr, eImgFmt,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	bFimcDone = false;
	// ============ Notice !!! ============
	FIMC_StartProcessingForScalerBypass(eCh);
	// ================================
	
	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	//----------------------------------------------------

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	FIMC_InitIpForDinFout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uInFbAddr, eImgFmt, false,
		uDispHSz, uDispVSz, // Fifo out display size
		eLocalInCS, PROGRESSIVE,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
	Disp("\n Press any key to stop! \n");
	UART_Getc();
	FIMC_StopProcessing(eCh);
}

static void TestForDebugging_8(void)
{
	u32 i, j;
	u32 uImgHsz, uImgVsz;
	u32 uLoopSz;
	u32 bResult;

	IMG_FMT eImgFmt = TILE_NV12;

	for(i=0; i<2; i++)
	{
		switch(i)
		{
			case 0:
				if((eCh==FIMC_A)||(eCh==FIMC_B))
				{
					uImgHsz = 4224; // MAIN_SC_IN_MAX_HSZ_A/MAIN_SC_IN_MAX_HSZ_B
					uImgVsz = 16;				
				}
				else if(eCh==FIMC_C)
				{
					uImgHsz = 1920; // MAIN_SC_IN_MAX_HSZ_C
					uImgVsz = 16;				
				}
				else
				{
					Assert(0);
				}
				break;
			case 1:
				uImgHsz = 16;
				uImgVsz = 8192; 
				break;
			default:
				Assert(0);
				break;
		}

		// One2one
		FIMC_InitIpForDinDout(eCh,
			uImgHsz, uImgVsz, 0,
			0, 0, uImgHsz, uImgVsz,
			uInFbAddr, eImgFmt,
			uImgHsz, uImgVsz,
			0, 0, uImgHsz, uImgVsz, 0, // Scaler		
			uOutFbAddr, eImgFmt, 	
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, true);
		
		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		//-----------------------------

		Disp("\n Comparing source and result... \n");

		FIMC_GetNextFbAddr(eImgFmt, uImgHsz, uImgVsz, 0x0, &uLoopSz);
		Disp("\n uLoopSz = %d \n", uLoopSz);

		for(j=0; j<uLoopSz; j++) // MAIN_SC_IN_MAX_HSZ_A/MAIN_SC_IN_MAX_HSZ_B
		{
			bResult = Compare8(uInFbAddr+j, uOutFbAddr+j, 1);

			// For log
			//Disp("\n (0x%x, 0x%x) = (0x%x, 0x%x) \n", 
			//	(uInFbAddr+j), Inp8(uInFbAddr+j), (uOutFbAddr+j), Inp8(uOutFbAddr+j));			
			
			Assert(bResult);	
		}
	}

}

static void TestForDebugging_9(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;
#if 1
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);
#endif
	//FIMC_InitCamera(eCisModel, YCRYCB, QVGA); // QVGA:320x240
	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	FIMC_InitIpForFinDout(eCh, eCamera,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	//FIMC_SetFinIrq(eCh,true, false); // yes
	FIMC_SetFinIrq(eCh,false, true); // no
	//FIMC_SetFinIrq(eCh,true, true); // yes

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#if 1
	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		//FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		//Disp("\n Current frame count is %d\n", uFrmCnt);
		if(bFimcDone == true)
			Disp("\n true \n");
	}
#else
	Disp(" Press any key to capture image!\n");
	UART_Getc();
#endif
	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);
}

static void TestForDebugging_10(void)
{
	u32 i;
	
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	eInSize = QXGA; // QXGA:2048x1536
	//eInSize = HD720; // HD720:1280x720
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);

	Disp("\n Sensor initialization! please wait.......... \n");
	FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	// [Rotation(Output rotator) and Flip]
	//------------------------------
	Disp("\n Rotation(Output rotator) and Flip \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
#if 0	
	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region
#else
	uInImgCropStartX = 0;
	uInImgCropStartY = 0;

	//uInImgCropHsz = 1696; // Max size in 13MHz(system clock)
	//uInImgCropHsz = 1888; // Max size in 16MHz(system clock)
	//uInImgCropHsz = 1952; // Max size in 20MHz(system clock)
	//uInImgCropHsz = 1984; // Max size in 24MHz(system clock)
	uInImgCropHsz = 2048; // In 30MHz(system clock) -> Unstable operation
	
	uInImgCropVsz = 1536; // Set cropping region		
#endif
	if(uLcdHsz >= uLcdVsz)
	{
		uOutImgHsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
		uOutImgVsz = uLcdVsz-(uLcdVsz/10)-((uLcdVsz-(uLcdVsz/10))%16);
	}
	else // uLcdHsz < uLcdVsz
	{
		uOutImgHsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
		uOutImgVsz = uLcdHsz-(uLcdHsz/10)-((uLcdHsz-(uLcdHsz/10))%16);
	}
	
	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uOutImgHsz, uOutImgVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);
#if 0
	for(i=0; i<8; i++)
#else
	i = 4;
#endif
	{	
		memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		
		switch(i)
		{
			case 0:
				eRotFlip = ROT_NO_FLIP_NO;
				break;
			case 1:
				eRotFlip = ROT_NO_FLIP_X;
				break;
			case 2:
				eRotFlip = ROT_NO_FLIP_Y;
				break;
			case 3:
				eRotFlip = ROT_NO_FLIP_XY;
				break;
			case 4:
				eRotFlip = ROT_90_FLIP_NO;
				break;
			case 5:
				eRotFlip = ROT_90_FLIP_X;
				break;
			case 6:
				eRotFlip = ROT_90_FLIP_Y;
				break;
			case 7:
				eRotFlip = ROT_90_FLIP_XY;
				break;
			default:
				Assert(0);
				break;
		}
		FIMC_SetOutRotAndFlip(eCh, eRotFlip);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		
		Disp("\n Order = %d \n", i);
		UART_Getc();	
		FIMC_PreProcessForLastFrmCpt(eCh);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);
	}

}

static void TestForDebugging_11(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;
	
#if 0
	u32 uInImgHSz=320;
	u32 uInImgVSz=240;
#else
	u32 uInImgHSz=800;
	u32 uInImgVSz=480;
#endif

	u32 uYAddr[71], uCAddr[71];
	u32 uYSize, uCbSize, uCrSize;

	for(i=0; i<71; i++)
	{
		FIMC_GetYCbCrSize(RGB24, uInImgHSz, uInImgVSz, &uYSize, &uCbSize, &uCrSize);
	
		uYAddr[i] = CODEC_MEM_ST + (uYSize*i);
		//uCAddr[i] = CODEC_MEM_ST + (uYSize*i) + (uCbSize*i) + uYSize;
	}

	// How to? d.load "V:\Components\multimedia\fimc\TILE_NV12_320x240_71frames" 0x21000000
#if 0	
	Disp("\n Load 'RGB24_320x240_71frames' file into '0x%x' address using T32, \n press any key to go next! \n", CODEC_MEM_ST);
#else
	Disp("\n Load 'RGB24_800x480_50frames' file into '0x%x' address using T32, \n press any key to go next! \n", CODEC_MEM_ST);
#endif
	UART_Getc();

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = uLcdHsz;
	uDispVSz = uLcdVsz;
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	InitVariables();

	FIMC_InitIpForDinFout(eCh,
		uInImgHSz, uInImgVSz, 0,
		0, 0, uInImgHSz, uInImgVSz,
		CODEC_MEM_ST, RGB24, false,
		uDispHSz, uDispVSz, // Fifo out display size
		eLocalInCS, PROGRESSIVE, ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[0], 0);
	Disp("\n Y[0] = 0x%x, C[0] = 0x%x \n", uYAddr[0], 0);
		
	FIMC_StartProcessingForMfc(eCh);

	//UART_Getc();

#if 0
	for(i=1; i<71; i++)
#else
	for(i=1; i<50; i++)
#endif
	{
		FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[i], 0); 	
		Disp("\n Y[%d] = 0x%x, C[%d] = 0x%x \n", i, uYAddr[i], i, 0);
		Delay(0x1500);
		//UART_Getc();
	}

	FIMC_StopProcessing(eCh);
}

static void TestForDebugging_12(void)
{
	//IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	IMG_FMT eInterlaceImgFmt[4] = {NV12_IB, NV12_IB, NV12_IB, NV12_IB};
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	//memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);
	memset((void *)uLcdFbAddr, 0, 720*480*4);

#if 0
	Disp("\n Generating sample image... \n");
	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, 720, 480, eBpp);
	GLIB_DrawPattern(720, 480);
#else
	// How to? d.load "V:\Components\multimedia\fimc\TILE_NV12_320x240_71frames" 0x21000000
	Disp("\n Load 'TILE_NV12_320x240_71frames' file into '0x%x' address using T32, \n press any key to go next! \n", CODEC_MEM_ST);
	UART_Getc();
#endif

	for(i=0; i<4; i++)
	{
		for(j=0; j<8; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = 720;
					uOutDispVsz = 480/2;
					
					uOutImgHsz = 720;
					uOutImgVsz = 480;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uOutDispHsz = uLcdVsz/2;
					uOutDispVsz = uLcdHsz;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;		
					break;
				default:
					Assert(0);
					break;
			}

			FIMC_InitIpForDinDout(eCh,
				720, 480, 0,
				0, 0, 720, 480,
				uLcdFbAddr, NV12,
				uOutDispHsz, uOutDispVsz,
				0, 0, uOutImgHsz, uOutImgVsz, 0, // Scaler		
				uInFbAddr, eInterlaceImgFmt[i], 	
				eRotFlipList[j], eCscEquation, eYCbCrRange, false);
#if 0		
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
			WaitForFimcDone(eCh);		
			FIMC_StopProcessing(eCh);
#else
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
			WaitForFimcDone(eCh);		
			FIMC_StopProcessing(eCh);						

			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
			WaitForFimcDone(eCh);		
			FIMC_StopProcessing(eCh);
#endif			

			// uOutImgAddr: EVEN start address, uNextFbAddr: ODD start address
			FIMC_GetNextFbAddr(NV12, 720, 480, uInFbAddr, &uNextFbAddr);

			Disp("\n start = 0x%x, end = 0x%x \n", uInFbAddr, uNextFbAddr-1);
			
			//--------------------------------------------

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					uInImgVsz = uLcdVsz/2;	
					
					uInImgCropHsz = uLcdHsz; 
					uInImgCropVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uInImgHsz =  uLcdVsz/2;
					uInImgVsz = uLcdHsz;			
					
					uInImgCropHsz =  uLcdVsz/2; 
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			

			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uInFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			// Odd field
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uInFbAddr, &uNextFbAddr);
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uNextFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}
		
}

static void TestForDebugging_13(void)
{
	u32 uNextFbAddr;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	// How to? d.load "d:\720x480_NV12.yuv" 0x21000000
	Disp("\n Load '720x480_NV12.yuv' file into '0x%x' address using T32, \n press any key to go next! \n", uLcdFbAddr);
	UART_Getc();

	FIMC_InitIpForDinDout(eCh,
		720, 480, 0,
		0, 0, 720, 480,
		uLcdFbAddr, NV12,
		720, 480,
		0, 0, 720, 480, 0, // Scaler		
		uInFbAddr, RGB24,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
	WaitForFimcDone(eCh);					
	FIMC_StopProcessing(eCh);	

	//LCDC_InitIp(eBpp, uInFbAddr, WIN0, false); 
	//LCDC_Start();
	//Disp("\n Press any key to go next! \n");
	//UART_Getc();

	//---------------------------
	
	FIMC_InitIpForDinDout(eCh,
		720, 480, 0,
		0, 0, 720, 480,
		uInFbAddr, RGB24,
		720, 480/2,
		0, 0, 720, 480, 0, // Scaler		
		uOutFbAddr, NV12_IB, 	
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);						

	//---------------------------

	// Even field
	FIMC_InitIpForDinDout(eCh,
		720, 480/2, 0,
		0, 0, 720, 480/2,
		uOutFbAddr, NV12,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
		uLcdOutFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);					
	FIMC_StopProcessing(eCh);
	
	// Odd field
	FIMC_GetNextFbAddr(NV12, 720, 480/2, uOutFbAddr, &uNextFbAddr);
	FIMC_InitIpForDinDout(eCh,
		720, 480/2, 0,
		0, 0, 720, 480/2,
		uNextFbAddr, NV12,
		uLcdHsz, uLcdVsz,
		0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
		uLcdOutFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);					
	FIMC_StopProcessing(eCh);	

	FIMC_GetNextFbAddr(NV12, 720, 480, uOutFbAddr, &uNextFbAddr);
	Disp("\n start = 0x%x, end = 0x%x \n", uOutFbAddr, uNextFbAddr - 1);

	//LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
	//LCDC_Start();	
}

static void TestForDebugging_14(void)
{
	u32 uNextFbAddr;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	
#if 0
	// How to? d.load "d:\720x480_NV12.yuv" 0x21000000
	Disp("\n Load '720x480_NV12.yuv' file into '0x%x' address using T32, \n press any key to go next! \n", uLcdFbAddr);
	UART_Getc();

	FIMC_InitIpForDinDout(eCh,
		720, 480, 0,
		0, 0, 720, 480,
		uLcdFbAddr, NV12,
		720, 480,
		0, 0, 720, 480, 0, // Scaler		
		uInFbAddr, RGB24,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
	WaitForFimcDone(eCh);					
	FIMC_StopProcessing(eCh);	

	//LCDC_InitIp(eBpp, uInFbAddr, WIN0, false); 
	//LCDC_Start();
	//Disp("\n Press any key to go next! \n");
	//UART_Getc();

	//---------------------------
	
	FIMC_InitIpForDinDout(eCh,
		720, 480, 0,
		0, 0, 720, 480,
		uInFbAddr, RGB24,
		720, 480/2,
		0, 0, 720, 480, 0, // Scaler		
		uOutFbAddr, NV12_IB, 	
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);		
#else
	// How to? d.load "d:\720x480_NV12_IB_error.yuv" 0x26300000
	Disp("\n Load '720x480_NV12_IB_error.yuv' file into '0x%x' address using T32, \n press any key to go next! \n", uOutFbAddr);
	UART_Getc();
#endif
	//---------------------------

	// Even field
	FIMC_InitIpForDinDout(eCh,
		720, 480/2, 0,
		0, 0, 720, 480/2,
		uOutFbAddr, NV12,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
		uLcdOutFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);					
	FIMC_StopProcessing(eCh);
	
	// Odd field
	FIMC_GetNextFbAddr(NV12, 720, 480/2, uOutFbAddr, &uNextFbAddr);
	FIMC_InitIpForDinDout(eCh,
		720, 480/2, 0,
		0, 0, 720, 480/2,
		uNextFbAddr, NV12,
		uLcdHsz, uLcdVsz,
		0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
		uLcdOutFbAddr, eBpp,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);					
	FIMC_StopProcessing(eCh);	

	//LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
	//LCDC_Start();	
}

static void TestForDebugging_15(void)
{
	u32 i, uLineSkipVal=1;

	IMG_FMT eImgFmt = YC420;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);	

#if 0
	memset((void *)uLcdFbAddr, 0xff, uLcdHsz*uLcdVsz*4); // Clear screen by white color
#else
	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
#endif
	memset((void *)uInFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
	memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uLcdFbAddr, eBpp,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0, // Scaler	
		uInFbAddr, eImgFmt,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uInFbAddr, eImgFmt,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz/(uLineSkipVal+1), uLineSkipVal, // Scaler	
		uLcdOutFbAddr, eImgFmt,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	Disp("\n Check image in T32 (d.image. 0x26600000 800. 480. /YUV420) \n");
	UART_Getc();
}

static void TestForDebugging_15_1(void)
{
	IMG_FMT eImgFmtList[11] = 
	{
		NV12, 
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};

	u32 i;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);	

	for(i=0; i<11; i++)
	{
		memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eImgFmtList[i],
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler	
			uInFbAddr, RGB24,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

#if 1		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uInFbAddr, RGB24,
			uLcdHsz, uLcdVsz,
			0, 0, uLcdHsz, uLcdVsz, 0, // Scaler	
			uLcdOutFbAddr, eImgFmtList[i], 	
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		WaitForFimcDone(eCh);
		FIMC_StopProcessing(eCh);

		Disp("\n Check image in T32 (d.image. 0x26600000 800. 480. /YUV420) \n");
#endif

		Disp("\n Press any key to go next! \n");
		UART_Getc();
	}
	
}

static void TestForDebugging_16(void)
{
	//IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	//IMG_FMT eInterlaceImgFmt = TILE_NV12_IB;
	IMG_FMT eInterlaceImgFmt = NV12_IB;
	
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	//eDataAlign = MIPI_24BIT_ALIGN;
	eFinImgType = CBYCRY;
#if 0
	eInSize = QXGA; // QXGA:2048x1536
#else
	eInSize = HD720; // HD720:1280x720
#endif

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);

	Disp("\n Sensor initialization! please wait.......... \n");
	FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	//for(i=0; i<4; i++)
	{
		//for(j=0; j<8; j++)
		for(j=4; j<8; j++)
		//for(j=4; j<5; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			switch(eInterlaceImgFmt)
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = uLcdHsz;
					uOutDispVsz = uLcdVsz/2;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uOutDispHsz = uLcdVsz/2; // Notice!
					uOutDispVsz = uLcdHsz; // Notice!
					
					uOutImgHsz = uLcdHsz;

					uOutImgVsz = uLcdVsz; // 16 multiple -> X
					//uOutImgVsz = uLcdVsz-16; // 32 multiple -> O
					//uOutImgVsz = uLcdVsz-8; // 8 multiple -> O
					//uOutImgVsz = uLcdVsz-4; // 4 multiple -> O
					//uOutImgVsz = uLcdVsz-2; // 2 multiple -> O
					Disp("\n uOutImgVsz = %d \n", uOutImgVsz);
					break;
				default:
					Assert(0);
					break;
			}

			FIMC_InitIpForMFinDout(eCh,						
				0, 0, uFinImgHsz, uFinImgVsz,
				uOutDispHsz, uOutDispVsz,				
				0, 0, uOutImgHsz, uOutImgVsz, 0,
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt,
				eRotFlipList[j], eCscEquation, eYCbCrRange, eDataAlign);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//--------------------------------------------

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					uInImgVsz = uLcdVsz/2;		
					
					//uInImgCropHsz = uLcdHsz; 
					uInImgCropHsz = uLcdHsz - (uLcdHsz%16); 
					
					uInImgCropVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uInImgHsz =  uLcdVsz/2;
					uInImgVsz = uLcdHsz;		

					//uInImgCropHsz =  uLcdVsz/2; 					
					uInImgCropHsz =  (uLcdVsz/2) - ((uLcdVsz/2)%16);
										
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			

			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uLcdFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,				
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			// Odd field
#if 0			
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uLcdFbAddr, &uNextFbAddr);
#else
			FIMC_GetNextFbAddr(eImgFmt, uInImgHsz, uInImgVsz, uLcdFbAddr, &uNextFbAddr);
#endif
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uNextFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}		
	}

	CSI_DisableCSI();		
}

static void TestForDebugging_17(void)
{
	//IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	IMG_FMT eInterlaceImgFmt = NV12;
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	//eDataAlign = MIPI_24BIT_ALIGN;
	eFinImgType = CBYCRY;
#if 0
	eInSize = QXGA; // QXGA:2048x1536
#else
	eInSize = HD720; // HD720:1280x720
#endif

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);

	Disp("\n Sensor initialization! please wait.......... \n");
	FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	//for(i=0; i<4; i++)
	{
		//for(j=0; j<8; j++)
		for(j=4; j<6; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			switch(eInterlaceImgFmt)
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				case NV12:
					eImgFmt = NV12;
					break;
				default:
					Assert(0);
					break;			
			}

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = uLcdHsz;
					//uOutDispVsz = uLcdVsz/2;
					uOutDispVsz = uLcdVsz;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					//uOutDispHsz = uLcdVsz/2; // Notice!
					uOutDispHsz = uLcdVsz; // Notice!
					uOutDispVsz = uLcdHsz; // Notice!
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;		
					break;
				default:
					Assert(0);
					break;
			}

			FIMC_InitIpForMFinDout(eCh,						
				0, 0, uFinImgHsz, uFinImgVsz,
				uOutDispHsz, uOutDispVsz,
				0, 0, uOutImgHsz, uOutImgVsz, 0,
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt,
				eRotFlipList[j], eCscEquation, eYCbCrRange, eDataAlign);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//--------------------------------------------

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					uInImgVsz = uLcdVsz/2;		
					
					//uInImgCropHsz = uLcdHsz; 
					uInImgCropHsz = uLcdHsz - (uLcdHsz%16); 
					
					uInImgCropVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					//uInImgHsz =  uLcdVsz/2;
					uInImgHsz =  uLcdVsz;
					uInImgVsz = uLcdHsz;		

					//uInImgCropHsz =  uLcdVsz/2; 
					//uInImgCropHsz =  (uLcdVsz/2) - ((uLcdVsz/2)%16);
					uInImgCropHsz =  (uLcdVsz) - ((uLcdVsz)%16);
										
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			

			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uLcdFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,				
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			// Odd field
#if 0			
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uLcdFbAddr, &uNextFbAddr);
#else
			FIMC_GetNextFbAddr(eImgFmt, uInImgHsz, uInImgVsz, uLcdFbAddr, &uNextFbAddr);
#endif
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				//uNextFbAddr, eImgFmt,
				uLcdFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}

	CSI_DisableCSI();		
}

static void TestForDebugging_18(void)
{
	u32 uTemp;

	u32 i;
	u32 uFinImgHsz, uFinImgVsz;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
#if 0
	eInSize = QXGA; // QXGA:2048x1536
#else
	eInSize = HD720; // HD720:1280x720
#endif

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
#if 1
	CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
#else
	CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, 0);
#endif
	Disp("\n Sensor initialization! please wait.......... \n");
	FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	// [Capture frame control]
	//--------------------- 
	Disp("\n Capture frame control \n");

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region
	
	// Set capture frame control
	bCptFrmCtrl = true;
	uCptSeqPattern = 0x10101010, uCptSeqPointer = 0;	

#if 0
	// Set capture count control
	bCptCntCtrl = true;
	uCptCnt = 10;	
#else
	// Set capture count control
	bCptCntCtrl = false;
	uCptCnt = 0;
#endif

	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, bCptFrmCtrl, uCptSeqPattern, uCptSeqPointer, bCptCntCtrl, uCptCnt);

	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);
#ifndef FIMC_POLLING
	while(!bFimcDone);
#else // polling
	FIMC_WaitForProcessingDoneForCptFrmCtrl(eCh);
#endif
	FIMC_StopProcessing(eCh);

	UART_Getc();

#if 1
	//Outp32(0xfb200008, 0x20010080);
	//Outp32(0xfb200008, 0x20010080);


	//========== Software workaround ===========
#if 1
	uTemp = Inp32(0xfb2000c0);
	uTemp = uTemp & (~(0x1<<25));
	Outp32(0xfb2000c0, uTemp); 
#endif
	//===================================


	// Set capture count control
	bCptCntCtrl = true;
	uCptCnt = 10;	

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, bCptFrmCtrl, uCptSeqPattern, uCptSeqPointer, bCptCntCtrl, uCptCnt);

	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);

#ifndef FIMC_POLLING
	while(!bFimcDone);
#else // polling
	FIMC_WaitForProcessingDoneForCptFrmCtrl(eCh);
#endif
	FIMC_StopProcessing(eCh);	
#endif

	CSI_DisableCSI();
}

static void TestForDebugging_18_1(void)
{
	u32 uTemp;

	u32 i;
	u32 uFinImgHsz, uFinImgVsz;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	
	//=================================================================

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	eInSize = HD720; // HD720:1280x720

	CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);

	Disp("\n Sensor initialization! please wait.......... \n");
	FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region
	
	// Set capture frame control
	bCptFrmCtrl = true;
	uCptSeqPattern = 0x10101010, uCptSeqPointer = 0;	

	// Set capture count control
	bCptCntCtrl = false;
	uCptCnt = 0;

	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, bCptFrmCtrl, uCptSeqPattern, uCptSeqPointer, bCptCntCtrl, uCptCnt);

	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);
#ifndef FIMC_POLLING
	while(!bFimcDone);
#else // polling
	FIMC_WaitForProcessingDoneForCptFrmCtrl(eCh);
#endif
	FIMC_StopProcessing(eCh);

	CSI_DisableCSI();

	Disp("\n Reset mipi camera! \n");
	UART_Getc();

	//=================================================================

	//========== Software workaround ===========
#if 1
	uTemp = Inp32(0xfb2000c0);
	uTemp = uTemp & (~(0x1<<25));
	Outp32(0xfb2000c0, uTemp); 
#endif
	//===================================

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);	

	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	eInSize = HD720; // HD720:1280x720

	CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);

	Disp("\n Sensor initialization! please wait.......... \n");
	FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4); // Clear screen by black color

	uInImgCropStartX = 0;
	uInImgCropStartY = 0;
	uInImgCropHsz = uFinImgHsz; // Set cropping region
	uInImgCropVsz = uFinImgVsz; // Set cropping region
	
	// Set capture frame control
	bCptFrmCtrl = true;
	uCptSeqPattern = 0x10101010, uCptSeqPointer = 0;		

	// Set capture count control
	bCptCntCtrl = true;
	uCptCnt = 10;

	FIMC_InitIpForMFinDout(eCh,
		uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
		uLcdHsz, uLcdVsz,
		0, 0,
		uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, eDataAlign);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, bCptFrmCtrl, uCptSeqPattern, uCptSeqPointer, bCptCntCtrl, uCptCnt);

	Disp(" Press any key to capture image!\n");
	UART_Getc();

	FIMC_PreProcessForLastFrmCpt(eCh);

#ifndef FIMC_POLLING
	while(!bFimcDone);
#else // polling
	FIMC_WaitForProcessingDoneForCptFrmCtrl(eCh);
#endif
	FIMC_StopProcessing(eCh);	

	CSI_DisableCSI();
}


static void TestForDebugging_19(void)
{
	//IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	IMG_FMT eInterlaceImgFmt[2] = {NV12_IB, TILE_NV12_IB};
	
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	//for(i=0; i<4; i++)
	for(i=0; i<2; i++)
	{
		//for(j=0; j<8; j++)
		for(j=4; j<8; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = uLcdHsz;
					uOutDispVsz = uLcdVsz/2;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uOutDispHsz = uLcdVsz/2; // Notice!
					uOutDispVsz = uLcdHsz; // Notice!
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;
					break;
				default:
					Assert(0);
					break;
			}

			FIMC_InitIpForFinDout(eCh, eCamera,
				0, 0, uFinImgHsz, uFinImgVsz,
				uOutDispHsz, uOutDispVsz,
				0, 0, uOutImgHsz, uOutImgVsz, 0,
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt[i],
				eRotFlipList[j], eCscEquation, eYCbCrRange);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//--------------------------------------------

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					uInImgVsz = uLcdVsz/2;		
					
					//uInImgCropHsz = uLcdHsz; 
					uInImgCropHsz = uLcdHsz - (uLcdHsz%16); 
					
					uInImgCropVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uInImgHsz =  uLcdVsz/2;
					uInImgVsz = uLcdHsz;		

					//uInImgCropHsz =  uLcdVsz/2; 
					uInImgCropHsz =  (uLcdVsz/2) - ((uLcdVsz/2)%16);
					
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			

			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uLcdFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			// Odd field
#if 0			
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uLcdFbAddr, &uNextFbAddr);
#else		
			FIMC_GetNextFbAddr(eImgFmt, uInImgHsz, uInImgVsz, uLcdFbAddr, &uNextFbAddr);
#endif

			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uNextFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}
		
}

static void TestForDebugging_20(void)
{
	//IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	//IMG_FMT eInterlaceImgFmt[2] = {NV12_IB, TILE_NV12_IB};
	IMG_FMT eInterlaceImgFmt[4] = {RGB16, RGB24, NV12, TILE_NV12};
	
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	//for(i=0; i<4; i++)
	for(i=2; i<4; i++)
	{
		//for(j=0; j<8; j++)
		for(j=4; j<8; j++)
		{
			memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);
			memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4);
		
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
#if 0			
			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}
#endif
			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = uLcdHsz;
					
					uOutDispVsz = uLcdVsz/2;
					//uOutDispVsz = uLcdVsz;
					
					uOutImgHsz = uLcdHsz;
					
					//uOutImgVsz = uLcdVsz;
					uOutImgVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
#if 0					
					uOutDispHsz = uLcdVsz/2; // Notice!
#else
					uOutDispHsz = (uLcdVsz/2) + 8; // Notice!
#endif
					//uOutDispHsz = uLcdVsz; // Notice!
					
					uOutDispVsz = uLcdHsz; // Notice!
					
					uOutImgHsz = uLcdHsz;
					
					//uOutImgVsz = uLcdVsz;
					uOutImgVsz = uLcdVsz/2;
					break;
				default:
					Assert(0);
					break;
			}

			FIMC_InitIpForFinDout(eCh, eCamera,
				0, 0, uFinImgHsz, uFinImgVsz,
				uOutDispHsz, uOutDispVsz,
				0, 0, uOutImgHsz, uOutImgVsz, 0,
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt[i],
				eRotFlipList[j], eCscEquation, eYCbCrRange);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//--------------------------------------------

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					
					uInImgVsz = uLcdVsz/2;		
					//uInImgVsz = uLcdVsz;		
					
					//uInImgCropHsz = uLcdHsz; 
					uInImgCropHsz = uLcdHsz - (uLcdHsz%16); 
					
					uInImgCropVsz = uLcdVsz/2;
					//uInImgCropVsz = uLcdVsz;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
#if 0					
					uInImgHsz =  uLcdVsz/2;
#else
					uInImgHsz =  (uLcdVsz/2) + 8;
#endif
					//uInImgHsz =  uLcdVsz;
					
					uInImgVsz = uLcdHsz;		

					//uInImgCropHsz =  uLcdVsz/2; 
#if 0					
					uInImgCropHsz =  (uLcdVsz/2) - ((uLcdVsz/2)%16);
#else
					uInImgCropHsz =  (uLcdVsz/2+8) - ((uLcdVsz/2+8)%16);
#endif
					//uInImgCropHsz =  (uLcdVsz) - ((uLcdVsz)%16);
					
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			

			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				//uLcdFbAddr, eImgFmt,
				uLcdFbAddr, eInterlaceImgFmt[i],
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				//0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);
#if 0
			// Odd field
#if 0			
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uLcdFbAddr, &uNextFbAddr);
#else		
			FIMC_GetNextFbAddr(eImgFmt, uInImgHsz, uInImgVsz, uLcdFbAddr, &uNextFbAddr);
#endif

			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uNextFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);
#endif
			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}
		
}

static void TestForDebugging_21(void)
{
	u32 i, j;
	u32 uDispHSz, uDispVSz;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	// [Image cropping]
	//----------------
	Disp("\n Image cropping \n");

	InitVariables();
	
	for(i=0; i<2; i++)
	{
		switch(i)
		{
			case 0:
				uInImgCropStartX = 0;
				uInImgCropStartY = 0;
				break;
			case 1:
				uInImgCropStartX = uLcdHsz/2;
				uInImgCropStartY = 0;				
				break;
			case 2:
				uInImgCropStartX = 0;
				uInImgCropStartY = uLcdVsz/2;				
				break;
			case 3:
				uInImgCropStartX = uLcdHsz/2;
				uInImgCropStartY = uLcdVsz/2;				
				break;
			default:
				Assert(0);
				break;
		}

		uInImgCropHsz = (uLcdHsz/2)-((uLcdHsz/2)%16);
		uInImgCropVsz = (uLcdVsz/2)-((uLcdVsz/2)%16);

		FIMC_InitIpForDinFout(eCh,
			uLcdHsz, uLcdVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uLcdFbAddr, eBpp, false,	
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);
	
		UART_Getc();	
	}

}

static void TestForDebugging_22(void)
{
	u32 uDispHSz, uDispVSz;

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	// [Variable input image size]
	//-----------------------
	Disp("\n Variable input image size \n");

	InitVariables();

	uInImgHsz = 320;
	uInImgVsz = 240;

	uInImgCropHsz = uInImgHsz;
	uInImgCropVsz = uInImgVsz;

	eBppInFmt = NV12;

	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uLcdFbAddr, eBpp,
		uInImgHsz, uInImgVsz,
		0, 0, uInImgHsz, uInImgVsz, 0, // Scaler
		uInFbAddr, eBppInFmt,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);			
	WaitForFimcDone(eCh);				
	FIMC_StopProcessing(eCh);
	
	FIMC_InitIpForDinFout(eCh,
		uInImgHsz, uInImgVsz, 0,
		0, 0, uInImgCropHsz, uInImgCropVsz,
		uInFbAddr, eBppInFmt, false,		
		uDispHSz, uDispVSz, // Fifo out display size
		eLocalInCS, PROGRESSIVE,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
}

static void TestForDebugging_23(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;
	u32 uInImgCropHszList[3];
	u32 uInImgCropVszList[3];
	u32 uInImgCropStartXList[3]={16,32,48};
	u32 uInImgCropStartYList[3]={48,56,80};
	u32 uInImgHVszList[8][2];

	IMG_FMT eImgFmtList[11] = 
	{
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};

	ROT_FLIP eRotFlipList[8] = 
	{
		ROT_NO_FLIP_NO,
		ROT_NO_FLIP_X,
		ROT_NO_FLIP_Y,
		ROT_NO_FLIP_XY,
		ROT_90_FLIP_NO,
		ROT_90_FLIP_X,
		ROT_90_FLIP_Y,
		ROT_90_FLIP_XY
	};

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);	
#if 0	
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
#else
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(LOCAL_IF_RGB24, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(LOCAL_IF_RGB24, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(LOCAL_IF_RGB24, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
#endif
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);
	
	// [Composition]
	//-------------
	Disp("\n Composition \n");	

	InitVariables();

	switch(eCh)
	{
#if 0	
		case FIMC_A:
		case FIMC_B:
			uInImgHVszList[0][0]=800; uInImgHVszList[0][1]=480;
			uInImgHVszList[1][0]=1280; uInImgHVszList[1][1]=720;
			uInImgHVszList[2][0]=1920; uInImgHVszList[2][1]=1080;
			uInImgHVszList[3][0]=2048; uInImgHVszList[3][1]=1536;
			uInImgHVszList[4][0]=2560; uInImgHVszList[4][1]=2048;
			uInImgHVszList[5][0]=3264; uInImgHVszList[5][1]=2448; // 8M
			uInImgHVszList[6][0]=4032; uInImgHVszList[6][1]=3008; // 12M
			uInImgHVszList[7][0]=4224; uInImgHVszList[7][1]=4224;		
			break;
#endif			
		case FIMC_A:
		case FIMC_B:
		case FIMC_C:
			uInImgHVszList[0][0]=720; uInImgHVszList[0][1]=576;
			uInImgHVszList[1][0]=768; uInImgHVszList[1][1]=576;
			uInImgHVszList[2][0]=800; uInImgHVszList[2][1]=480;
			uInImgHVszList[3][0]=1024; uInImgHVszList[3][1]=768;
			uInImgHVszList[4][0]=1280; uInImgHVszList[4][1]=720;
			uInImgHVszList[5][0]=1440; uInImgHVszList[5][1]=1440;
			uInImgHVszList[6][0]=1600; uInImgHVszList[6][1]=1200;
			uInImgHVszList[7][0]=1920; uInImgHVszList[7][1]=1080;				
			break;
		default:
			Assert(0);
			break;
	}

	uInImgCropHszList[0]=(640-(16*2))-((640-(16*2))%16);
	uInImgCropHszList[1]=(640-(48*2))-((640-(48*2))%16);
	uInImgCropHszList[2]=(640-(80*2))-((640-(80*2))%16);

	uInImgCropVszList[0]=(480-(24*2))-((480-(24*2))%16);
	uInImgCropVszList[1]=(480-(56*2))-((480-(56*2))%16);
	uInImgCropVszList[2]=(480-(80*2))-((480-(80*2))%16);

#if 0
	for(i=0; i<11; i++)
#else
	for(i=2; i<3; i++)
#endif
	{
		uInImgHsz = uInImgHVszList[i%8][0];
		uInImgVsz = uInImgHVszList[i%8][1];
		
		uInImgCropStartX = uInImgCropStartXList[(i/3)%3];
		uInImgCropStartY = uInImgCropStartYList[i%3];
		uInImgCropHsz = uInImgCropHszList[(i/3)%3];
		uInImgCropVsz = uInImgCropVszList[i%3];

		eBppInFmt = eImgFmtList[i];
		
		eRotFlip = eRotFlipList[i%8];		
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler
			uInFbAddr, eBppInFmt,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);			
		FIMC_StopProcessing(eCh);
		
		FIMC_InitIpForDinFout(eCh,
			uInImgHsz, uInImgVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uInFbAddr, eBppInFmt, false,		
			uDispHSz, uDispVSz, // Fifo out display size
#if 0			
			eLocalInCS, PROGRESSIVE,
#else
			LOCAL_IF_RGB24, PROGRESSIVE,
#endif
			eRotFlip, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);

		Disp("\n InImg order num = %d \n", i);
		Disp(" Hit any key to go to the next step! \n");	
		UART_Getc();
	}

}

static void TestForDebugging_24(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;
	u32 uInImgCropHszList[3];
	u32 uInImgCropVszList[3];
	u32 uInImgCropStartXList[3]={16,32,48};
	u32 uInImgCropStartYList[3]={48,56,80};
	u32 uInImgHVszList[8][2];

	IMG_FMT eImgFmtList[11] = 
	{
		NV12,
		YC420,
		YCBYCR,
		YCRYCB,
		CRYCBY,
		CBYCRY, 
		NV16,
		YC422,
		RGB16,
		RGB24,
		TILE_NV12
	};

	ROT_FLIP eRotFlipList[8] = 
	{
		ROT_NO_FLIP_NO,
		ROT_NO_FLIP_X,
		ROT_NO_FLIP_Y,
		ROT_NO_FLIP_XY,
		ROT_90_FLIP_NO,
		ROT_90_FLIP_X,
		ROT_90_FLIP_Y,
		ROT_90_FLIP_XY
	};

#if 0
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);		
	switch(eCh)
	{
		case FIMC_A:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN0, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN0);
			break;
		case FIMC_B:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN1, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		case FIMC_C:
			LCDC_InitWinForFifoIn4(eLocalInCS, WIN2, uDispHSz, uDispVSz, 32, 32);
			LCDC_SetWinOnOff(1, WIN2);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();
#else
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	uDispHSz = (uLcdHsz/2)-((uLcdHsz/2)%16);
	uDispVSz = (uLcdVsz/2)-((uLcdVsz/2)%16);	
#endif

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);
	
	// [Composition]
	//-------------
	Disp("\n Composition \n");	

	InitVariables();

	switch(eCh)
	{
#if 0	
		case FIMC_A:
		case FIMC_B:
			uInImgHVszList[0][0]=800; uInImgHVszList[0][1]=480;
			uInImgHVszList[1][0]=1280; uInImgHVszList[1][1]=720;
			uInImgHVszList[2][0]=1920; uInImgHVszList[2][1]=1080;
			uInImgHVszList[3][0]=2048; uInImgHVszList[3][1]=1536;
			uInImgHVszList[4][0]=2560; uInImgHVszList[4][1]=2048;
			uInImgHVszList[5][0]=3264; uInImgHVszList[5][1]=2448; // 8M
			uInImgHVszList[6][0]=4032; uInImgHVszList[6][1]=3008; // 12M
			uInImgHVszList[7][0]=4224; uInImgHVszList[7][1]=4224;		
			break;
#endif			
		case FIMC_A:
		case FIMC_B:
		case FIMC_C:
			uInImgHVszList[0][0]=720; uInImgHVszList[0][1]=576;
			uInImgHVszList[1][0]=768; uInImgHVszList[1][1]=576;
			uInImgHVszList[2][0]=800; uInImgHVszList[2][1]=480;
			uInImgHVszList[3][0]=1024; uInImgHVszList[3][1]=768;
			uInImgHVszList[4][0]=1280; uInImgHVszList[4][1]=720;
			uInImgHVszList[5][0]=1440; uInImgHVszList[5][1]=1440;
			uInImgHVszList[6][0]=1600; uInImgHVszList[6][1]=1200;
			uInImgHVszList[7][0]=1920; uInImgHVszList[7][1]=1080;				
			break;
		default:
			Assert(0);
			break;
	}

	uInImgCropHszList[0]=(640-(16*2))-((640-(16*2))%16);
	uInImgCropHszList[1]=(640-(48*2))-((640-(48*2))%16);
	uInImgCropHszList[2]=(640-(80*2))-((640-(80*2))%16);

	uInImgCropVszList[0]=(480-(24*2))-((480-(24*2))%16);
	uInImgCropVszList[1]=(480-(56*2))-((480-(56*2))%16);
	uInImgCropVszList[2]=(480-(80*2))-((480-(80*2))%16);

#if 0
	for(i=0; i<11; i++)
#else
	for(i=2; i<3; i++)
#endif
	{
		memset((void *)uLcdOutFbAddr, 0, uLcdHsz*uLcdVsz*4);
	
		uInImgHsz = uInImgHVszList[i%8][0];
		uInImgVsz = uInImgHVszList[i%8][1];
		
		uInImgCropStartX = uInImgCropStartXList[(i/3)%3];
		uInImgCropStartY = uInImgCropStartYList[i%3];
		uInImgCropHsz = uInImgCropHszList[(i/3)%3];
		uInImgCropVsz = uInImgCropVszList[i%3];

		eBppInFmt = eImgFmtList[i];
		
		eRotFlip = eRotFlipList[i%8];		
		
		FIMC_InitIpForDinDout(eCh,
			uLcdHsz, uLcdVsz, 0,
			0, 0, uLcdHsz, uLcdVsz,
			uLcdFbAddr, eBpp,
			uInImgHsz, uInImgVsz,
			0, 0, uInImgHsz, uInImgVsz, 0, // Scaler
			uInFbAddr, eBppInFmt,		
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);			
		FIMC_StopProcessing(eCh);

#if 0
		FIMC_InitIpForDinFout(eCh,
			uInImgHsz, uInImgVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uInFbAddr, eBppInFmt, false,		
			uDispHSz, uDispVSz, // Fifo out display size
			eLocalInCS, PROGRESSIVE,
			eRotFlip, eCscEquation, eYCbCrRange);

		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
		Disp("\n Press any key to stop! \n");
		UART_Getc();
		FIMC_StopProcessing(eCh);
#else
		FIMC_InitIpForDinDout(eCh,
			uInImgHsz, uInImgVsz, 0,
			uInImgCropStartX, uInImgCropStartY, uInImgCropHsz, uInImgCropVsz,
			uInFbAddr, eBppInFmt,
			uLcdHsz, uLcdVsz,
			32, 32, uDispHSz, uDispVSz, 0, // Scaler
			uLcdOutFbAddr, eBpp,		
			eRotFlip, eCscEquation, eYCbCrRange, false);

		bFimcDone = false;
		FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
		WaitForFimcDone(eCh);			
		FIMC_StopProcessing(eCh);
#endif

		Disp("\n InImg order num = %d \n", i);
		Disp(" Hit any key to go to the next step! \n");	
		UART_Getc();
	}

}

static void TestForDebugging_25(void)
{
	IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};

	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	Assert(eCh == FIMC_C);

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	//temp
	uLcdHsz = 240;
	uLcdVsz = 192;

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	//FIMC_InitCamera(eCisModel, YCRYCB, UXGA); // UXGA:1600x1200
	FIMC_InitCamera(eCisModel, YCRYCB, QVGA); // QVGA:320x240
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	for(i=0; i<4; i++)
	{
		for(j=0; j<8; j++)
		{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
			
			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = uLcdHsz;
					uOutDispVsz = uLcdVsz/2;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uOutDispHsz = uLcdVsz/2; // Notice!
					uOutDispVsz = uLcdHsz; // Notice!
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;		
					break;
				default:
					Assert(0);
					break;
			}

			FIMC_InitIpForFinDout(eCh, eCamera,
				0, 0, uFinImgHsz, uFinImgVsz,
				uOutDispHsz, uOutDispVsz,
				0, 0, uOutImgHsz, uOutImgVsz, 0,
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt[i],
				eRotFlipList[j], eCscEquation, eYCbCrRange);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//--------------------------------------------

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					uInImgVsz = uLcdVsz/2;		
					
					//uInImgCropHsz = uLcdHsz; 
					uInImgCropHsz = uLcdHsz - (uLcdHsz%16); 
					
					uInImgCropVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uInImgHsz =  uLcdVsz/2;
					uInImgVsz = uLcdHsz;		

					//uInImgCropHsz =  uLcdVsz/2; 
					uInImgCropHsz =  (uLcdVsz/2) - ((uLcdVsz/2)%16);
					
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			

			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uLcdFbAddr, eImgFmt,
#if 0				
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
#else
				480, 800,
				0, 0, 480, 800/2, 0, // Scaler
#endif
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			// Odd field
#if 0			
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uLcdFbAddr, &uNextFbAddr);
#else		
			FIMC_GetNextFbAddr(eImgFmt, uInImgHsz, uInImgVsz, uLcdFbAddr, &uNextFbAddr);
#endif

			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uNextFbAddr, eImgFmt,
#if 0				
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
#else
				480, 800,
				0, 800/2, 480, 800/2, 0, // Scaler
#endif
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		}
	}

}

static void TestForDebugging_26(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitBase();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	uDispHSz = uLcdHsz;
	uDispVSz = uLcdVsz;
	switch(eCh)
	{
		case FIMC_B:
			LCDC_InitWinForFifoIn4(LOCAL_IF_RGB24, WIN1, uDispHSz, uDispVSz, 0, 0);
			LCDC_SetWinOnOff(1, WIN1);			
			break;
		default:
			Assert(0);
			break;			
	}
	LCDC_EnableAllWinFifoLvlInt(EMPTY);
	LCDC_Start();

	InitVariables();

	// Generate DMA input image
	GLIB_InitInstance(uLcdFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz);

	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uLcdFbAddr, eBpp,
		1280, 720,
		0, 0, 1280, 720, 0, // Scaler		
		uInFbAddr, NV12,		
		ROT_NO_FLIP_NO, CSC_ITU601, YCBCR_WIDE, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);	
	FIMC_StopProcessing(eCh);

	FIMC_InitIpForDinFout(eCh,
		1280, 720, 0,
		0, 0, 1280, 720,
		uInFbAddr, NV12, false,
		uDispHSz, uDispVSz, // Fifo out display size
		LOCAL_IF_RGB24, PROGRESSIVE,
		ROT_NO_FLIP_NO, CSC_ITU601, YCBCR_WIDE);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
	Disp("\n Press any key to stop! \n");
	UART_Getc();
	FIMC_StopProcessing(eCh);
}

static void TestForDebugging_27(void)
{
	// 12M monaco camera
	
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	IMG_RESOLUTION eImgResolution;
	u32 uNum;

	u32 uOutImgHsz, uOutImgVsz;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	Disp("\n 0: 1280x720 \n 1: 800x480 \n 2: 720x480 \n 3: 640x480 \n 4: 320x240 \n");
	Disp("\n Select camera output resolution: ");
	uNum = UART_GetIntNum();
	
	switch(uNum)
	{
		case 0: 
			eImgResolution = HD720; // 1280x720 	
			break;	
		case 1: 
			eImgResolution = WVGA; // 800x480
			break;			
		case 2: 
			eImgResolution = SD480; // 720x480
			break;
		case 3: 
			eImgResolution = VGA; // 640x480
			break;			
		case 4: 
			eImgResolution = QVGA; // 320x240
			break;
		default:
			Assert(0);
			break;			
	}

	FIMC_InitCamera(CE143Monaco, CBYCRY, eImgResolution);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	if(uFinImgHsz >= uLcdHsz)
	{
		uOutImgHsz = uLcdHsz;
	}
	else
	{
		uOutImgHsz = uFinImgHsz;
	}
	if(uFinImgVsz >= uLcdVsz)
	{
		uOutImgVsz = uLcdVsz;
	}
	else
	{
		uOutImgVsz = uFinImgVsz;
	}
	
	FIMC_InitIpForFinDout(eCh, eCamera,
		0, 0, uFinImgHsz, uFinImgVsz,
#if 0		
		uLcdHsz, uLcdVsz,
		0, 0, uOutImgHsz, uOutImgVsz, 0,
		uLcdFbAddr, uLcdFbAddr, eBpp,
#else
		uFinImgHsz, uFinImgVsz,
		0, 0, uFinImgHsz, uFinImgVsz, 0,
		uLcdFbAddr, uLcdFbAddr, CBYCRY,
#endif		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

#if 0
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);
#else
	// ============ Notice !!! ============
	FIMC_StartProcessingForScalerBypass(eCh);
	// ================================
#endif

	Disp(" Press any key to capture image!\n");

	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		Disp("\n Current frame count is %d\n", uFrmCnt);
	}

	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	FIMC_InitIpForDinDout(eCh,
		uFinImgHsz, uFinImgVsz, 0,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdFbAddr, CBYCRY,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
		uInFbAddr, eBpp,		
		ROT_NO_FLIP_NO, CSC_ITU601, YCBCR_WIDE, false);
	
	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	WaitForFimcDone(eCh);	
	FIMC_StopProcessing(eCh);

	LCDC_InitIp(eBpp, uInFbAddr, WIN0, false); 
	LCDC_Start();
}

static void TestForDebugging_28(void)
{
	// ===== For RGB18 display =====
	
	u32 i;
	u32 uDispHSz, uDispVSz;

	u32 uInImgHSz=320;
	u32 uInImgVSz=240;

	IMG_FMT eOutImgFmt = RGB18;
	//IMG_FMT eOutImgFmt = RGB24;

	u32 uYAddr[71], uCAddr[71];
	u32 uYSize, uCbSize, uCrSize;
	for(i=0; i<71; i++)
	{
		FIMC_GetYCbCrSize(TILE_NV12, uInImgHSz, uInImgVSz, &uYSize, &uCbSize, &uCrSize);
	
		uYAddr[i] = CODEC_MEM_ST + (uYSize*i) + (uCbSize*i);
		uCAddr[i] = CODEC_MEM_ST + (uYSize*i) + (uCbSize*i) + uYSize;
	}

	// How to? d.load "V:\Components\multimedia\fimc\TILE_NV12_320x240_71frames" 0x21000000
	Disp("\n Load 'TILE_NV12_320x240_71frames' file into '0x%x' address using T32, \n press any key to go next! \n", CODEC_MEM_ST);
	UART_Getc();

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
#if 0	
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
#else
	LCDC_InitBase();
	LCDC_InitWin(eOutImgFmt, 1360, 768, 0, 0, 1360, 768,
	//LCDC_InitWin(eOutImgFmt, 320, 240, 0, 0, 320, 240,
		0, 0, uLcdOutFbAddr, WIN0, false);
	LCDC_Start();	
	LCDC_GetFrmSz(&uLcdHsz, &uLcdVsz, WIN0);
#endif

	Disp("\n uLcdHsz = %d, uLcdVsz = %d \n", uLcdHsz, uLcdVsz);

	InitVariables();

	FIMC_InitIpForDinDout(eCh,
		uInImgHSz, uInImgVSz, 0,
		0, 0, uInImgHSz, uInImgVSz,
		CODEC_MEM_ST, TILE_NV12,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0, // Scaler		
#if 0		
		uLcdOutFbAddr, eBpp,		
#else		
		uLcdOutFbAddr, eOutImgFmt,
#endif		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
	
	while(GetKey() == 0)
	{
		for(i=0; i<71; i++)
		{	
			FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[i], uCAddr[i]);
			Disp("\n Y[0] = 0x%x, C[0] = 0x%x \n", uYAddr[i], uCAddr[i]);

			bFimcDone = false;
			FIMC_StartProcessingForMfc(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			Delay(800);
		}
	}
	
}

static void TestForDebugging_29(void)
{
	IMG_FMT eTestImgFmt = CRYCBY;

	uOutFbAddr = uInFbAddr+0x2000000;
	uLcdOutFbAddr = uInFbAddr+0x4000000;
		
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

#if 1
	// How to? d.load "D:\MCC_700Lux_Normal.raw" 0x41300000
	Disp("\n Load 'MCC_700Lux_Normal.raw' file into '0x%x' address using T32, \n press any key to go next! \n", uInFbAddr);
	UART_Getc();
#else
	// Generate DMA input image
	GLIB_InitInstance(uOutFbAddr, uLcdHsz, uLcdVsz, eBpp);
	GLIB_DrawPattern(uLcdHsz, uLcdVsz); 

	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,
		0, 0, uLcdHsz, uLcdVsz,
		uOutFbAddr, eBpp,
		2592, 1944,
		0, 0, 2592, 1944, 0, // Scaler		
		uInFbAddr, CRYCBY,		
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);	
#endif

#if 0
#if 0
	FIMC_InitIpForDinDout(eCh,
		2592, 1944, 0,		
		976, 736, 640, 480,
		uInFbAddr, CRYCBY,	
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0, // Scaler	
		uLcdOutFbAddr, eBpp,	
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
#else	
	FIMC_InitIpForDinDout(eCh,
		2592, 1944, 0,		
		976, 736, 640, 480,		
		uInFbAddr, CRYCBY,
		2592, 1944,
		0, 0, 2592, 1944, 0, // Scaler
		uLcdOutFbAddr, CRYCBY,	
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
#endif

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);	
#else
	FIMC_InitIpForDinDout(eCh,
		2592, 1944, 0,		
		976, 736, 640, 480, 			
		uInFbAddr, eTestImgFmt,
		2592, 1944,
		0, 0, 2592, 1944, 0, // Scaler
		uOutFbAddr, eTestImgFmt,	
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);	

	FIMC_InitIpForDinDout(eCh,
		2592, 1944, 0,		
		0, 0, 2592, 1944, 	
		uOutFbAddr, eTestImgFmt,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdOutFbAddr, eBpp,	
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);
#endif

	uOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000;
	uLcdOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000 + 0x00300000;
}

static u8 volatile bFimcDoneA;
static void __irq Isr_FimcA(void)
{
	FIMC_INTR_SRC eSrc;
	FIMC_GetIntrSrc(FIMC_A, &eSrc);
	Disp("[i]");
	switch(eSrc)
	{
		case FIMC_SUCCESS:
			bFimcDoneA = true;
			break;
		case FIFO_Y_OVF:
			bFimcDoneA = false;
			Disp("\n FIFO Y overflow \n");
			break;
		case FIFO_CB_OVF:
			bFimcDoneA = false;
			Disp("\n FIFO Cb overflow \n");
			break;
		case FIFO_CR_OVF:
			bFimcDoneA = false;
			Disp("\n FIFO Cr overflow \n");
			break;
		case ROT_LINE_BUF_OVF:
			bFimcDoneA = false;
			Disp("\n Rotation line buffer overflow \n");
			break;
		default:
			bFimcDoneA = false;
			Assert(0);
			break;		
	}

	FIMC_ClearInterrupt(FIMC_A);
	INTC_ClearVectAddr();
}

static u8 volatile bFimcDoneB;
static void __irq Isr_FimcB(void)
{
	FIMC_INTR_SRC eSrc;
	FIMC_GetIntrSrc(FIMC_B, &eSrc);
	Disp("[i]");
	switch(eSrc)
	{
		case FIMC_SUCCESS:
			bFimcDoneB = true;
			break;
		case FIFO_Y_OVF:
			bFimcDoneB = false;
			Disp("\n FIFO Y overflow \n");
			break;
		case FIFO_CB_OVF:
			bFimcDoneB = false;
			Disp("\n FIFO Cb overflow \n");
			break;
		case FIFO_CR_OVF:
			bFimcDoneB = false;
			Disp("\n FIFO Cr overflow \n");
			break;
		case ROT_LINE_BUF_OVF:
			bFimcDoneB = false;
			Disp("\n Rotation line buffer overflow \n");
			break;
		default:
			bFimcDoneB = false;
			Assert(0);
			break;		
	}

	FIMC_ClearInterrupt(FIMC_B);
	INTC_ClearVectAddr();
}

static u8 volatile bFimcDoneC;
static void __irq Isr_FimcC(void)
{
	FIMC_INTR_SRC eSrc;
	FIMC_GetIntrSrc(FIMC_C, &eSrc);
	Disp("[i]");
	switch(eSrc)
	{
		case FIMC_SUCCESS:
			bFimcDoneC = true;
			break;
		case FIFO_Y_OVF:
			bFimcDoneC = false;
			Disp("\n FIFO Y overflow \n");
			break;
		case FIFO_CB_OVF:
			bFimcDoneC = false;
			Disp("\n FIFO Cb overflow \n");
			break;
		case FIFO_CR_OVF:
			bFimcDoneC = false;
			Disp("\n FIFO Cr overflow \n");
			break;
		case ROT_LINE_BUF_OVF:
			bFimcDoneC = false;
			Disp("\n Rotation line buffer overflow \n");
			break;
		default:
			bFimcDoneC = false;
			Assert(0);
			break;		
	}

	FIMC_ClearInterrupt(FIMC_C);
	INTC_ClearVectAddr();
}

static void TestForDebugging_30(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	//-----------------------
	u32 i = 0;

	u32 uTestImgHsz;
	u32 uTestImgVsz;	

	IMG_FMT eTestImgFmt;
	//-----------------------	

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;	
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	eInSize = HD1080;
	eHSsettle = 12; 
	//eCSISDataLaneNum = DATALANE1;
	eCSISDataLaneNum = DATALANE2;

	//----------------------------------------

#ifndef FIMC_POLLING	
	INTC_Init(); // Initialize interrupt controller
	INTC_SetVectAddr(NUM_FIMC_A, Isr_FimcA);
	INTC_SetVectAddr(NUM_FIMC_B, Isr_FimcB);
	INTC_SetVectAddr(NUM_FIMC_C, Isr_FimcC);
	INTC_Enable(NUM_FIMC_A);
	INTC_Enable(NUM_FIMC_B);
	INTC_Enable(NUM_FIMC_C);
	
	INTC_SetVectAddr(NUM_LCD0, Isr_Fimd);
	INTC_Enable(NUM_LCD0);

	INTC_SetVectAddr(NUM_MIPI_CSI, Isr_Csis);
	INTC_Enable(NUM_MIPI_CSI);

	INTC_SetVectAddr(NUM_JPEG, Isr_Jpeg);
	INTC_Enable(NUM_JPEG);

	INTC_SetVectAddr(NUM_TIMER0,Isr_Timer0);
	INTC_Enable(NUM_TIMER0);	
#endif

	//----------------------------------------
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	//LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	MIPICAM_Prepare(S5K4EAG, eFinImgType, eInSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(S5K4EAG, eFinImgType, eInSize);

	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	for(;;)
	{
		i = i + 1;
		
		switch(i%6)
		{
			case 0:
				uTestImgHsz = 1920;
				uTestImgVsz = 1080;
				eTestImgFmt = TILE_NV12;
				break;
			case 1:
				uTestImgHsz = 1280;
				uTestImgVsz = 720;		
				eTestImgFmt = TILE_NV12;
				break;
			case 2:
				uTestImgHsz = 720;
				uTestImgVsz = 480;	
				eTestImgFmt = TILE_NV12;
				break;
			case 3:
				uTestImgHsz = 1920;
				uTestImgVsz = 1080;
				eTestImgFmt = TILE_NV12_IW;
				break;
			case 4:
				uTestImgHsz = 1280;
				uTestImgVsz = 720;		
				eTestImgFmt = TILE_NV12_IW;
				break;
			case 5:
				uTestImgHsz = 720;
				uTestImgVsz = 480;	
				eTestImgFmt = TILE_NV12_IW;
				break;				
			default:
				Assert(0);
				break;
		}	

		FIMC_InitIpForMFinDout(FIMC_C,
			0, 0, uFinImgHsz, uFinImgVsz,
			uTestImgHsz, uTestImgVsz,
			0, 0, uTestImgHsz, uTestImgVsz, 0,
			uLcdFbAddr, uLcdFbAddr, eTestImgFmt,
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange,eDataAlign);		
		
		FIMC_StartProcessing(FIMC_C, false, 0, 0, false, 0);

		Disp(" Press any key to capture image!\n");

		while(GetKey() == 0)
		{
#ifndef FIMC_POLLING
			bFimcDoneC = false;
			while(!bFimcDoneC);
#else // polling
			Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
			Assert(0);
			return;
#endif		
			FIMC_GetDoutFrmCnt(FIMC_C, &uFrmCnt);
			Disp("\n Current frame count is %d\n", uFrmCnt);		

			//----------------------------------------

			FIMC_InitIpForDinDout(FIMC_B,
				uTestImgHsz, uTestImgVsz, 0,		
				0, 0, uTestImgHsz, uTestImgVsz,		
				uLcdFbAddr, TILE_NV12,
				uLcdHsz, uLcdVsz,
				0, 0, uLcdHsz, uLcdVsz, 0, // Scaler
				uLcdOutFbAddr, eBpp,	
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);				

			bFimcDoneB = false;
			FIMC_StartProcessing(FIMC_B, false, 0, 0, false, 0);		
			//while(!bFimcDoneB);		
			FIMC_WaitForProcessingDone(FIMC_B);
			FIMC_StopProcessing(FIMC_B);			
		}
#if 1
		bFimcDone = false;
		FIMC_PreProcessForLastFrmCpt(FIMC_C);
		//while(!bFimcDoneC);
		FIMC_WaitForProcessingDone(FIMC_C);
		FIMC_StopProcessing(FIMC_C);		
#endif		
	}

	CSI_DisableCSI();
	MIPICAM_Disable();

	//----------------------------------------

#ifndef FIMC_POLLING	
	INTC_Init(); // Initialize interrupt controller
	INTC_SetVectAddr(NUM_FIMC_A, Isr_Fimc);
	INTC_SetVectAddr(NUM_FIMC_B, Isr_Fimc);
	INTC_SetVectAddr(NUM_FIMC_C, Isr_Fimc);
	INTC_Enable(NUM_FIMC_A);
	INTC_Enable(NUM_FIMC_B);
	INTC_Enable(NUM_FIMC_C);
	
	INTC_SetVectAddr(NUM_LCD0, Isr_Fimd);
	INTC_Enable(NUM_LCD0);

	INTC_SetVectAddr(NUM_MIPI_CSI, Isr_Csis);
	INTC_Enable(NUM_MIPI_CSI);

	INTC_SetVectAddr(NUM_JPEG, Isr_Jpeg);
	INTC_Enable(NUM_JPEG);

	INTC_SetVectAddr(NUM_TIMER0,Isr_Timer0);
	INTC_Enable(NUM_TIMER0);	
#endif	

	//----------------------------------------	
}

static void TestForDebugging_31(void)
{
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	uLcdFbAddr = CODEC_MEM_ST;	
	uInFbAddr = CODEC_MEM_ST + 0x01000000;
	uOutFbAddr = CODEC_MEM_ST + 0x01000000 + 0x01000000;
	uLcdOutFbAddr = CODEC_MEM_ST + 0x01000000 + 0x01000000 + 0x01000000;
	
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	FIMC_InitCamera(eCisModel, CRYCBY, UXGA); // UXGA:1600x1200
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);
	
	FIMC_InitIpForFinDout(eCh, eCamera,
		0, 0, uFinImgHsz, uFinImgVsz,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0,
		uLcdFbAddr, uLcdFbAddr, eBpp,
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

	Disp(" Press any key to capture image!\n");

	while(GetKey() == 0)
	{
#ifndef FIMC_POLLING
		bFimcDone = false;
		while(!bFimcDone);
#else // polling
		Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
		Assert(0);
		return;
#endif		
		FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
		Disp("\n Current frame count is %d\n", uFrmCnt);
	}

	bFimcDone = false;
	FIMC_PreProcessForLastFrmCpt(eCh);
	WaitForFimcDone(eCh);
	FIMC_StopProcessing(eCh);

	FIMC_InitIpForDinDout(eCh,
		uLcdHsz, uLcdVsz, 0,		
		0, 0, uLcdHsz, uLcdVsz, 			
		uLcdFbAddr, eBpp,
		2592, 1944,
		0, 0, 2592, 1944, 0, // Scaler
		uInFbAddr, CRYCBY,	
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);	

	//-----------------------------------------

	FIMC_InitIpForDinDout(eCh,
		2592, 1944, 0,		
		976, 736, 640, 480, 			
		uInFbAddr, CRYCBY,
		2592, 1944,
		0, 0, 2592, 1944, 0, // Scaler
		uOutFbAddr, CRYCBY,	
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);	

	FIMC_InitIpForDinDout(eCh,
		2592, 1944, 0,		
		0, 0, 2592, 1944, 	
		uOutFbAddr, CRYCBY,
		uLcdHsz, uLcdVsz,
		0, 0, uLcdHsz, uLcdVsz, 0, // Scaler
		uLcdOutFbAddr, eBpp,	
		ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	bFimcDone = false;
	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);		
	WaitForFimcDone(eCh);		
	FIMC_StopProcessing(eCh);	

	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
	LCDC_Start();	

	uLcdFbAddr = CODEC_MEM_ST;
	uInFbAddr = CODEC_MEM_ST + 0x00300000;
	uOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000;
	uLcdOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000 + 0x00300000;
}

#endif

#if 0
static void GenerateImage_0(void)
{
	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false);	
	LCDC_Start();
	//LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	GLIB_InitInstance(uLcdFbAddr, 240, 320, RGB16); 
	GLIB_DrawPattern(240, 320); 

	Disp("\n uLcdFbAddr address = 0x%x \n", uLcdFbAddr);

	FIMC_InitIpForDinDout(eCh,
		240, 320, 0,
		0, 0, 240, 320, uLcdFbAddr, RGB16,
		320, 240, 0, 0, 320, 240, 0, // Scaler 
		uLcdOutFbAddr, eBpp, ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);

	FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
	FIMC_WaitForProcessingDone(eCh); 
	FIMC_StopProcessing(eCh);
}

static void GenerateImage_1(void)
{
	u32 i;
	u32 uDispHSz, uDispVSz;

	u32 uInImgHSz=320;
	u32 uInImgVSz=240;

	u32 uYAddr[71], uCAddr[71];
	u32 uYSize, uCbSize, uCrSize;
	for(i=0; i<71; i++)
	{
		FIMC_GetYCbCrSize(TILE_NV12, uInImgHSz, uInImgVSz, &uYSize, &uCbSize, &uCrSize);
	
		uYAddr[i] = CODEC_MEM_ST + (uYSize*i) + (uCbSize*i);
		uCAddr[i] = CODEC_MEM_ST + (uYSize*i) + (uCbSize*i) + uYSize;
	}

	// How to? d.load "V:\Components\multimedia\fimc\TILE_NV12_320x240_71frames" 0x21000000
	Disp("\n Load 'TILE_NV12_320x240_71frames' file into '0x%x' address using T32, \n press any key to go next! \n", CODEC_MEM_ST);
	UART_Getc();

	InitVariables();

#if 0
	for(i=0; i<71; i++)
#else		
	for(i=0; i<50; i++)
#endif		
	{
		FIMC_InitIpForDinDout(eCh,
			uInImgHSz, uInImgVSz, 0,
			0, 0, uInImgHSz, uInImgVSz,
			CODEC_MEM_ST, TILE_NV12,
#if 0
			320, 240,
			0, 0, 320, 240, 0, // Scaler	
			CODEC_MEM_ST+0x1000000+(320*240*4*i), eBpp,	
#else
			800, 480,
			0, 0, 800, 480, 0, // Scaler
			CODEC_MEM_ST+0x1000000+(800*480*4*i), eBpp,	
#endif				
			ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
		
		bFimcDone = false;
		
		FIMC_SetDinImgAddrForMfc(eCh, TILE_NV12, uYAddr[i], uCAddr[i]);
		Disp("\n Y[0] = 0x%x, C[0] = 0x%x \n", uYAddr[i], uCAddr[i]);		
		FIMC_StartProcessingForMfc(eCh);	

		WaitForFimcDone(eCh);	

		FIMC_StopProcessing(eCh);
	}

#if 0
	Disp("\n start = 0x%x, end = 0x%x \n", CODEC_MEM_ST+0x1000000, CODEC_MEM_ST+0x1000000+(320*240*4*71)-1);
#else
	Disp("\n start = 0x%x, end = 0x%x \n", CODEC_MEM_ST+0x1000000, CODEC_MEM_ST+0x1000000+(800*480*4*50)-1);
#endif	
}

static void GenerateImage_2(void)
{
	//IMG_FMT eInterlaceImgFmt[4] = {RGB16_IB, RGB24_IB, NV12_IB, TILE_NV12_IB};
	IMG_FMT eInterlaceImgFmt = TILE_NV12_IB;
#if 0
	ROT_FLIP eRotFlipList[8] =
		{
			ROT_NO_FLIP_NO,
			ROT_NO_FLIP_X,
			ROT_NO_FLIP_Y,
			ROT_NO_FLIP_XY,
			ROT_90_FLIP_NO,
			ROT_90_FLIP_X,
			ROT_90_FLIP_Y,
			ROT_90_FLIP_XY
		};
#else
	ROT_FLIP eRotFlipList = ROT_NO_FLIP_NO;
#endif
	u32 i, j;
	u32 uFinImgHsz, uFinImgVsz;
	u32 uFrmCnt;

	u32 uDispHSz, uDispVSz;
	u32 uNextFbAddr;

	IMG_FMT eImgFmt;

	CSIS_INTV_SETTING eCSISIntvSetting;
	MIPI_DATA_ALIGN eDataAlign;
	IMG_FMT eFinImgType;
	IMG_RESOLUTION eInSize;
	u8 eHSsettle;
	eCSISIntvSetting.CSIS_INTV_Lintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv=0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv=0x0;
	eDataAlign = MIPI_32BIT_ALIGN;
	eFinImgType = CBYCRY;
	//eInSize = QXGA; // QXGA:2048x1536
	//eInSize = HD720; // HD720:1280x720
	//eInSize = USER_DEFINE_SIZE; // USER_DEFINE_SIZE:2592x1944
	eInSize = HD1080; // HD1080:1920x1080
	eHSsettle = 12;	
	eCSISDataLaneNum = DATALANE1;	

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
	LCDC_Start();
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);

	InitVariables();

	memset((void *)uLcdFbAddr, 0, uLcdHsz*uLcdVsz*4);

	//CSI_Start(DATALANE1, eFinImgType, eInSize, eDataAlign, WCLKSRC_EXTCLK, &eCSISIntvSetting, 0);
	CSI_Start(eCSISDataLaneNum, eFinImgType, eInSize, eDataAlign, WCLKSRC_PCLK, &eCSISIntvSetting, eHSsettle);

	Disp("\n Sensor initialization! please wait.......... \n");
	//FIMC_InitCamera(S5K4CAGX, eFinImgType, eInSize);
	FIMC_InitCamera(S5K4EAG, eFinImgType, eInSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);	

	//for(i=0; i<4; i++)
	//{
		//for(j=0; j<8; j++)
		//{
			LCDC_InitIp(eBpp, uLcdFbAddr, WIN0, false); 
			LCDC_Start();
#if 0
			switch(eInterlaceImgFmt[i])
			{
				case RGB16_IB:
					eImgFmt = RGB16;
					break;
				case RGB24_IB:
					eImgFmt = RGB24;
					break;
				case NV12_IB:
					eImgFmt = NV12;
					break;			
				case TILE_NV12_IB:
					eImgFmt = TILE_NV12;
					break;	
				default:
					Assert(0);
					break;			
			}

			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uOutDispHsz = uLcdHsz;
					uOutDispVsz = uLcdVsz/2;
					
					uOutImgHsz = uLcdHsz;
					uOutImgVsz = uLcdVsz;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uOutDispHsz = (uLcdVsz/2) - ((uLcdVsz/2)%16); // Notice!
					uOutDispVsz = uLcdHsz; // Notice!
					
					uOutImgHsz = uLcdHsz;
					//uOutImgVsz = uLcdVsz - 16;
					uOutImgVsz = uLcdVsz - (uLcdVsz%32);
					break;
				default:
					Assert(0);
					break;
			}
#else
			eImgFmt = TILE_NV12;

			uOutDispHsz = 1920;
			uOutDispVsz = 1080/2;

			uOutImgHsz = 1920;
			uOutImgVsz = 1080;
#endif

			FIMC_InitIpForMFinDout(eCh,				
				0, 0, uFinImgHsz, uFinImgVsz,
				uOutDispHsz, uOutDispVsz,
				0, 0, uOutImgHsz, uOutImgVsz, 0,
				//uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt[i],
				uLcdFbAddr, uLcdFbAddr, eInterlaceImgFmt,
				//eRotFlipList[j], eCscEquation, eYCbCrRange, eDataAlign);
				eRotFlipList, eCscEquation, eYCbCrRange, eDataAlign);

			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);

			Disp(" Press any key to capture image!\n");

			while(GetKey() == 0)
			{
#ifndef FIMC_POLLING
				bFimcDone = false;
				while(!bFimcDone);
#else // polling
				Disp("\n You needs INTERRUPT OPERATION to get current frame count\n");
				Assert(0);
				return;
#endif		
				FIMC_GetDoutFrmCnt(eCh, &uFrmCnt);
				Disp("\n Current frame count is %d\n", uFrmCnt);
			}

			bFimcDone = false;
			FIMC_PreProcessForLastFrmCpt(eCh);
			WaitForFimcDone(eCh);
			FIMC_StopProcessing(eCh);

			//--------------------------------------------
#if 0
			switch(eRotFlipList[j])
			{
				case ROT_NO_FLIP_NO:
				case ROT_NO_FLIP_X:
				case ROT_NO_FLIP_Y:
				case ROT_NO_FLIP_XY:
					uInImgHsz = uLcdHsz;
					uInImgVsz = uLcdVsz/2;		
			
					uInImgCropHsz = uLcdHsz - (uLcdHsz%16); 
					uInImgCropVsz = uLcdVsz/2;
					break;
				case ROT_90_FLIP_NO:
				case ROT_90_FLIP_X:
				case ROT_90_FLIP_Y:
				case ROT_90_FLIP_XY:
					uInImgHsz = (uLcdVsz/2) - ((uLcdVsz/2)%16);
					uInImgVsz = uLcdHsz;		

					uInImgCropHsz =  (uLcdVsz/2) - ((uLcdVsz/2)%16);
					uInImgCropVsz = uLcdHsz;	
					break;
				default:
					Assert(0);
					break;
			}			
#else
			uInImgHsz = 1920;
			uInImgVsz = 1080/2;		

			uInImgCropHsz = 1920 - (1920%16); 
			uInImgCropVsz = 1080/2;
#endif
			// Even field
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uLcdFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,				
				0, 0, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			// Odd field
#if 0			
			FIMC_GetNextFbAddr(eImgFmt, uInImgCropHsz, uInImgCropVsz, uLcdFbAddr, &uNextFbAddr);
#else
			FIMC_GetNextFbAddr(eImgFmt, uInImgHsz, uInImgVsz, uLcdFbAddr, &uNextFbAddr);
#endif
			FIMC_InitIpForDinDout(eCh,
				uInImgHsz, uInImgVsz, 0,
				0, 0, uInImgCropHsz, uInImgCropVsz,
				uNextFbAddr, eImgFmt,
				uLcdHsz, uLcdVsz,
				0, uLcdVsz/2, uLcdHsz, uLcdVsz/2, 0, // Scaler		
				uLcdOutFbAddr, eBpp,		
				ROT_NO_FLIP_NO, eCscEquation, eYCbCrRange, false);
			
			bFimcDone = false;
			FIMC_StartProcessing(eCh, false, 0, 0, false, 0);	
			WaitForFimcDone(eCh);					
			FIMC_StopProcessing(eCh);

			LCDC_InitIp(eBpp, uLcdOutFbAddr, WIN0, false); 
			LCDC_Start();

			Disp("\n Press any to go to the next step!\n");
			UART_Getc();
		//}
	//}

	CSI_DisableCSI();		
}
#endif

// [Test FIMC]
void Test_FIMC(void)
{
	int sel, i;

	testFuncMenu menu[] =
	{
		0,											"Exit\n",

		SelFimcA,									"Select FIMC_A",
		SelFimcB,									"Select FIMC_B",
		SelFimcC,									"Select FIMC_C\n",

		TestSfr,										"Test SFR\n",

		TestDinDoutForBasicFunc,					"Test DMA-in DMA-out for basic function",
		TestDinDoutForEachFunc,						"Test DMA-in DMA-out for each function",
		TestDinDoutForComposition,					"Test DMA-in DMA-out for composition",	
		TestDinDoutForInDmaLineSkip,				"Test DMA-in DMA-out for input DMA line skip",
		TestDinDoutForOutDmaLineSkip,				"Test DMA-in DMA-out for output DMA line skip",
		TestDinDoutForOne2one,						"Test DMA-in DMA-out for one2one\n",
		//TestDinDoutForInterlaceBobOut,				"Test DMA-in DMA-out for interlace BOB out\n", -> Spec out
				
		TestDinFoutForBasicFunc,					"Test DMA-in FIFO-out for basic function",
		TestDinFoutForEachFunc,						"Test DMA-in FIFO-out for each function",
		TestDinFoutForComposition,					"Test DMA-in FIFO-out for composition",
		TestDinFoutForMfcMultiFrame,				"Test DMA-in FIFO-out for MFC multi-frame",
		TestDinFoutForMfcInterlacedMultiFrame,		"Test DMA-in FIFO-out for MFC interlaced multi-frame",
		TestDinFoutForInDmaLineSkip,				"Test DMA-in FIFO-out for input DMA line skip",
		TestDinFoutForInterlaceOutWithIpc,			"Test DMA-in FIFO-out for interlace out with IPC",
		TestDinFoutForForcedOff,					"Test DMA-in FIFO-out for forced off",
		TestDinFoutForAnotherForcedOff,				"Test DMA-in FIFO-out for another forced off",
		TestDinFoutForCheckingC100Problem,			"Test DMA-in FIFO-out for checking c100 problem\n",
		
		SelItuCameraA, 								"Select camera A",
		SelItuCameraB, 								"Select camera B",
		TestFinDoutForBasicFunc,					"Test FIFO-in DMA-out for basic function",
		TestFinDoutForEachFunc,						"Test FIFO-in DMA-out for each function",
		TestFinDoutForComposition,					"Test FIFO-in DMA-out for composition",
		TestFinDoutForTestPattern, 					"Test FIFO-in DMA-out for test pattern",
		TestFinDoutForDoubleBuffering,				"Test FIFO-in DMA-out for double buffering",
		TestFinDoutForScalerBypass,					"Test FIFO-in DMA-out for scaler bypass",
		TestFinDoutForInterlaceBobOut,				"Test FIFO-in DMA-out for interlace BOB out",
		TestFinDoutForInterlaceWeaveOut,			"Test FIFO-in DMA-out for interlace WEAVE out",
		TestFinDoutForIrq,							"Test FIFO-in DMA-out for IRQ",
		TestFinDoutForJpegFileInput,					"Test FIFO-in DMA-out for JPEG file input",
		TestFinDoutForProgressiveItu601And656Input,"Test FIFO-in DMA-out for progressive ITU-601/656 input",
		TestFinDoutForInterlacedItu601And656Input,	"Test FIFO-in DMA-out for interlaced ITU-601/656 input\n",
				
		TestDFinDoutForBasicFunc,					"Test DIRECT-FIFO-in DMA-out for basic function",
		TestDFinDoutForEachFunc,					"Test DIRECT-FIFO-in DMA-out for each function",
		TestDFinDoutForComposition,					"Test DIRECT-FIFO-in DMA-out for composition",
		TestDFinDoutForFimdFrameSkip,				"Test DIRECT-FIFO-in DMA-out for FIMD frame skip",
		TestDFinDoutForFimdBlending,				"Test DIRECT-FIFO-in DMA-out for FIMD blending\n",

		TestMFinDoutForBasicFunc,					"Test MIPI-FIFO-in DMA-out for basic function",
		TestMFinDoutForEachFunc,					"Test MIPI-FIFO-in DMA-out for each function",
		TestMFinDoutForComposition,					"Test MIPI-FIFO-in DMA-out for composition",
		TestMFinDoutForDoubleBuffering,				"Test MIPI-FIFO-in DMA-out for double buffering",
		TestMFinDoutForScalerBypass,				"Test MIPI-FIFO-in DMA-out for scaler bypass",
		TestMFinDoutForInterlaceBobOut,				"Test MIPI-FIFO-in DMA-out for interlace BOB out",
		TestMFinDoutForInterlaceWeaveOut,			"Test MIPI-FIFO-in DMA-out for interlace WEAVE out\n",

		//TestDinDoutForSortingChip,					"Test DMA-in DMA-out for sorting chip",
		TestDinDoutUsingCModel,						"Test DMA-in DMA-out using c-model",
		TestIterationOfDinFoutAndDinDout,			"Test iteration of DMA-in FIFO-out and DMA-in DMA-out\n",
	
		TestDinDoutForPowerGating,					"Test DMA-in DMA-out for power gating",
		TestDinDoutForClockGating,					"Test DMA-in DMA-out for clock gating",
		TestDinFoutForPowerGating,					"Test DMA-in FIFO-out for power gating",
		TestDinFoutForClockGating,					"Test DMA-in FIFO-out for clock gating",
		TestFinDoutForPowerGating,					"Test FIFO-in DMA-out for power gating",
		TestFinDoutForClockGating,					"Test FIFO-in DMA-out for clock gating",	
		TestDFinDoutForPowerGating,					"Test DIRECT-FIFO-in DMA-out for power gating",
		TestDFinDoutForClockGating,					"Test DIRECT-FIFO-in DMA-out for clock gating",
		TestMFinDoutForPowerGating,				"Test MIPI-FIFO-in DMA-out for power gating",
		TestMFinDoutForClockGating,					"Test MIPI-FIFO-in DMA-out for clock gating",
		TestPowerMode,								"Test power mode\n",

		TestDinFoutForCheckingUnderRun,			"Test DMA-in FIFO-out for checking under-run",
		TestDinFoutForApplyingThe1stUnderRunSol,	"Test DMA-in FIFO-out for applying the 1st under-run solution",
		TestDinFoutForApplyingThe2ndUnderRunSol,	"Test DMA-in FIFO-out for applying the 2nd under-run solution",
		TestDinFoutForApplyingAllTheUnderRunSols,	"Test DMA-in FIFO-out for applying all the under-run solutions",
		TestDinFoutForAddingBusLoadUsingFIMD,		"Test DMA-in FIFO-out for adding BUS load using FIMD",
		TestDinFoutForApplyingThe3rdUnderRunSol,	"Test DMA-in FIFO-out for applying the 3rd under-run solution\n",

		TestDinDoutForGettingOperationTime,			"Test DMA-in DMA-out for getting operation time",
		TestFinDoutForMaxPclk,						"Test FIFO-in DMA-out for max PCLK",
		TestDinDoutForCoreClkSrcAndDivider,			"Test DMA-in DMA-out for core clock source and divider",
		TestFinDoutForMclkSrcAndDivider,			"Test FIFO-in DMA-out for MCLK source and divider\n",

		//TestForDebugging_0,							"Test for debugging 0",	
		//TestForDebugging_1,							"Test for debugging 1",	
		//TestForDebugging_2,							"Test for debugging 2",
		//TestForDebugging_3,							"Test for debugging 3",
		//TestForDebugging_4,							"Test for debugging 4",
		//TestForDebugging_5,							"Test for debugging 5",
		//TestForDebugging_6,							"Test for debugging 6",
		//TestForDebugging_7,							"Test for debugging 7",
		//TestForDebugging_8,							"Test for debugging 8",
		//TestForDebugging_9,							"Test for debugging 9",
		//TestForDebugging_10,						"Test for debugging 10",
		//TestForDebugging_11,						"Test for debugging 11",
		//TestForDebugging_12,						"Test for debugging 12",
		//TestForDebugging_13,						"Test for debugging 13",
		//TestForDebugging_14,						"Test for debugging 14",
		//TestForDebugging_15,						"Test for debugging 15",
		//TestForDebugging_15_1,						"Test for debugging 15_1",
		//TestForDebugging_16,						"Test for debugging 16",
		//TestForDebugging_17,						"Test for debugging 17",
		//TestForDebugging_18,						"Test for debugging 18",
		//TestForDebugging_18_1,						"Test for debugging 18_1",
		//TestForDebugging_19,						"Test for debugging 19",
		//TestForDebugging_20,						"Test for debugging 20",
		//TestForDebugging_21,						"Test for debugging 21",
		//TestForDebugging_22,						"Test for debugging 22",
		//TestForDebugging_23,						"Test for debugging 23",
		//TestForDebugging_24,						"Test for debugging 24",
		//TestForDebugging_25,						"Test for debugging 25",
		//TestForDebugging_26,						"Test for debugging 26",
		//TestForDebugging_27,						"Test for debugging 27",
		//TestForDebugging_28,						"Test for debugging 28",
		//TestForDebugging_29,						"Test for debugging 29",
		//TestForDebugging_30,						"Test for debugging 30",
		//TestForDebugging_31,						"Test for debugging 31",
		//GenerateImage_0,							"Generate image 0",
		//GenerateImage_1,							"Generate image 1",
		//GenerateImage_2,							"Generate image 2",
		
		0, 0
	};

	// X-block power off & on
	SYSC_SetBlockPowerOFF_IP(eBLKCAM);
	DelayfrTimer(milli, 100);
	SYSC_SetBlockPowerON_IP(eBLKCAM);

	// FIMC clock off & on
	SYSC_SetClkOFF_IP(eFIMC0);
	SYSC_SetClkOFF_IP(eFIMC1);
	SYSC_SetClkOFF_IP(eFIMC2);
	DelayfrTimer(milli, 100);
	SYSC_SetClkON_IP(eFIMC0);
	SYSC_SetClkON_IP(eFIMC1);
	SYSC_SetClkON_IP(eFIMC2);

	//--------------

	SelFimcA(); // Notice!

#if 0
	eCscEquation = CSC_ITU601;	
	eYCbCrRange = YCBCR_WIDE;	
	eLocalInCS = LOCAL_IF_RGB24;
#else
	eCscEquation = CSC_ITU709;
	eYCbCrRange = YCBCR_NARROW;
	eLocalInCS = LOCAL_IF_YCBCR444;
#endif

	SYSC_InitClkForMM();
	InitBaseHwForI2C();
	InitBaseHwForFIMC();
	InitBaseHwForCSIS();
	
	SelItuCameraA();
	eCisModel = S5K3BAF;		
	uMClkOfS5K3BAFB = g_uEPLL/SYSC_GetClkDiv(eCLKDIV_CAM0);
	//uMClkOfS5K3BAFB = g_uEPLL/SYSC_GetClkDiv(eCLKDIV_CAM1);
	Disp("\n\n uMClkOfS5K3BAFB = %d \n\n", uMClkOfS5K3BAFB);
	uFrameRateOfS5K3BAFB = 8; // '4' or '8' or '15'
	
	eBpp = RGB24;
	uLcdFbAddr = CODEC_MEM_ST;
#if 0	
	uInFbAddr = CODEC_MEM_ST + 0x02000000;
	uOutFbAddr = CODEC_MEM_ST + 0x02000000 + 0x02000000;
	uLcdOutFbAddr = CODEC_MEM_ST + 0x02000000 + 0x02000000 + 0x02000000;
#else
	uInFbAddr = CODEC_MEM_ST + 0x00300000;
	uOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000;
	uLcdOutFbAddr = CODEC_MEM_ST + 0x00300000 + 0x05000000 + 0x00300000;
#endif

	SYSC_SetDispPath(eDISP_BYPASS);
//	IEIP_Select(DISP_BYPASS); // mDNIE bypass

	//--------------

#ifndef FIMC_POLLING	
	INTC_Init(); // Initialize interrupt controller
	INTC_SetVectAddr(NUM_FIMC_A, Isr_Fimc);
	INTC_SetVectAddr(NUM_FIMC_B, Isr_Fimc);
	INTC_SetVectAddr(NUM_FIMC_C, Isr_Fimc);
	INTC_Enable(NUM_FIMC_A);
	INTC_Enable(NUM_FIMC_B);
	INTC_Enable(NUM_FIMC_C);
	
	INTC_SetVectAddr(NUM_LCD0, Isr_Fimd);
	INTC_Enable(NUM_LCD0);

	INTC_SetVectAddr(NUM_MIPI_CSI, Isr_Csis);
	INTC_Enable(NUM_MIPI_CSI);

	INTC_SetVectAddr(NUM_JPEG, Isr_Jpeg);
	INTC_Enable(NUM_JPEG);

	INTC_SetVectAddr(NUM_TIMER0,Isr_Timer0);
	INTC_Enable(NUM_TIMER0);	
#endif

	while(true)
	{
		Disp("\n");
		for (i=0; (int)(menu[i].desc)!=0; i++)
		{
			Disp("%2d: %s\n", i, menu[i].desc);
		}

		Disp("\n Select the function to test : ");
		sel = UART_GetIntNum();
		Disp("\n");

		if (sel == 0)
		{
			break;
		}
		else if (sel>0 && sel<(sizeof(menu)/8-1))
		{
			(menu[sel].func) ();
		}
	}
	
}
