#ifndef __NAND_H__
#define __NAND_H__


#ifdef __cplusplus
extern "C" {
#endif

/* Option constants for bizarre disfunctionality and real
*  features
*/
/* Chip can not auto increment pages */
#define NAND_NO_AUTOINCR	0x00000001
/* Buswitdh is 16 bit */
#define NAND_BUSWIDTH_16	0x00000002
/* Device supports partial programming without padding */
#define NAND_NO_PADDING		0x00000004
/* Chip has cache program function */
#define NAND_CACHEPRG		0x00000008
/* Chip has copy back function */
#define NAND_COPYBACK		0x00000010
/* AND Chip which has 4 banks and a confusing page / block
 * assignment. See Renesas datasheet for further information */
#define NAND_IS_AND		0x00000020
/* Chip has a array of 4 pages which can be read without
 * additional ready /busy waits */
#define NAND_4PAGE_ARRAY	0x00000040
/* Chip requires that BBT is periodically rewritten to prevent
 * bits from adjacent blocks from 'leaking' in altering data.
 * This happens with the Renesas AG-AND chips, possibly others.  */
#define BBT_AUTO_REFRESH	0x00000080

/* Options valid for Samsung large page devices */
#define NAND_SAMSUNG_LP_OPTIONS \
	(NAND_NO_PADDING | NAND_CACHEPRG | NAND_COPYBACK)



/*
 * Standard NAND flash commands
 */
#define CMD_READ0				0x00
#define CMD_READ1				0x01

#define CMD_PAGEPROG			0x80
#define CMD_PAGEPROG_2CYCLE		0x10

#define CMD_2PLN_PAGEPRG_1START		0x80	// == CMD_PAGEPROG
#define CMD_2PLN_PAGEPRG_1END		0x11
#define CMD_2PLN_PAGEPRG_2START		0x81
#define CMD_2PLN_PAGEPRG_2END		0x10	// == CMD_PAGEPROG_2CYCLE

#define CMD_READOOB				0x50
#define CMD_ERASE1				0x60
#define CMD_STATUS				0x70
#define CMD_STATUS_MULTI		0x71

#define CMD_READID				0x90
#define CMD_ERASE2				0xd0
#define CMD_RESET				0xff

/* Extended commands for large page devices */
#define CMD_READ0_2CYCLE		0x30
#define CMD_CACHEDPROG				0x15
#define	CMD_RANDOM_DATA_OUT_1ST_CYCLE	0x05
#define	CMD_RANDOM_DATA_OUT_2ND_CYCLE	0xE0
#define	CMD_RANDOM_DATA_IN			0x85



/*
 * NAND Flash Manufacturer ID Codes
 */
#define MAN_ID_TOSHIBA	0x98
#define MAN_ID_SAMSUNG	0xec
#define MAN_ID_FUJITSU	0x04
#define MAN_ID_NATIONAL	0x8f
#define MAN_ID_RENESAS	0x07
#define MAN_ID_STMICRO	0x20
#define MAN_ID_HYNIX    0xad

/**
 * struct nand_flash_dev - NAND Flash Device ID Structure
 *
 * @name:  	Identify the device type
 * @id:   	device ID code
 * @pagesize:  	Pagesize in bytes. Either 256 or 512 or 0
 *		If the pagesize is 0, then the real pagesize
 *		and the eraseize are determined from the
 *		extended id bytes in the chip
 * @erasesize: 	Size of an erase block in the flash device.
 * @chipsize:  	Total chipsize in Mega Bytes
 * @options:	Bitfield to store chip relevant options
 */

/**
 * struct nand_manufacturers - NAND Flash Manufacturer ID Structure
 * @name:	Manufacturer name
 * @id: 	manufacturer ID code of device.
*/
struct nand_manufacturers {
	int id;
	char * name;
};


struct CurNandInfo {
	char *name;
	int id;
	unsigned long pagesize;
	unsigned long chipsize;
	unsigned long erasesize;
	unsigned long options;
};


extern struct nand_flash_dev nand_flash_ids[];
extern struct nand_manufacturers nand_manuf_ids[];


/**
 * struct nand_flash_dev - NAND Flash Device ID Structure
 *
 * @name:  	Identify the device type
 * @id:   	device ID code
 * @pagesize:  	Pagesize in bytes. Either 256 or 512 or 0
 *		If the pagesize is 0, then the real pagesize
 *		and the eraseize are determined from the
 *		extended id bytes in the chip
 * @erasesize: 	Size of an erase block in the flash device.
 * @chipsize:  	Total chipsize in Mega Bytes
 * @options:	Bitfield to store chip relevant options
 */
struct nand_flash_dev {
	char *name;
	int id;
	unsigned long pagesize;
	unsigned long chipsize;
	unsigned long erasesize;
	unsigned long options;
};




#define		NUM_ID				8
#define		SIZE_OF_MAN_NAME	64
#define		SLC_TYPE			0				
#define		MLC_TYPE			1	
#define		N_ROW_ADDRESS		8				





enum NAND_REGS
{
 	rNFCONF        = (NFCON_BASE+0x00),
 	rNFCONT	  = NFCON_BASE+0x04, 
 rNFCMD		  = NFCON_BASE+0x08, 
 rNFADDR	  = NFCON_BASE+0x0C, 
 rNFDATA	  = NFCON_BASE+0x10,
 rNFDATA8	  = NFCON_BASE+0x10,
                                 
 rNFMECCD0	  = NFCON_BASE+0x14,
 rNFMECCD1	  = NFCON_BASE+0x18,
 rNFSECCD	  = NFCON_BASE+0x1C,
 rNFSBLK 	  = NFCON_BASE+0x20,
 rNFEBLK 	  = NFCON_BASE+0x24,
 rNFSTAT 	  = NFCON_BASE+0x28,
 rNFECCERR0	  = NFCON_BASE+0x2C,
 rNFECCERR1	  = NFCON_BASE+0x30, 
 rNFMECC0	  = NFCON_BASE+0x34,  
 rNFMECC1	  = NFCON_BASE+0x38,
 rNFSECC 	  = NFCON_BASE+0x3C,  
 rNFMLCBITPT = NFCON_BASE+0x40, 


 rNFECCCONF	 = NFCON_BASE+0x20000, 
 rNFECCCONT	 = NFCON_BASE+0x20020,  
 rNFECCSTAT	= NFCON_BASE+0x20030,   
 rNFECCSECSTAT	= NFCON_BASE+0x20040,   

 rNFECCPRG0	= NFCON_BASE+0x20090,  
 rNFECCPRG1	= NFCON_BASE+0x20094,   
 rNFECCPRG2	= NFCON_BASE+0x20098,   
 rNFECCPRG3	= NFCON_BASE+0x2009C,  
 rNFECCPRG4	= NFCON_BASE+0x200A0,   
 rNFECCPRG5	= NFCON_BASE+0x200A4,   
 rNFECCPRG6	= NFCON_BASE+0x200A8,  

 rNFECCERL0		= NFCON_BASE+0x200C0,  
 rNFECCERL1		= NFCON_BASE+0x200C4,  
 rNFECCERL2		= NFCON_BASE+0x200C8,  
 rNFECCERL3		= NFCON_BASE+0x200CC,   
 rNFECCERL4		= NFCON_BASE+0x200D0,  
 rNFECCERL5		= NFCON_BASE+0x200D4,   
 rNFECCERL6		= NFCON_BASE+0x200D8,   
 rNFECCERL7		= NFCON_BASE+0x200DC,  
 rNFECCERP0		= NFCON_BASE+0x200F0,  
 rNFECCERP1		= NFCON_BASE+0x200F4,  
 rNFECCERP2		= NFCON_BASE+0x200F8,  
 rNFECCERP3		= NFCON_BASE+0x200FC,  
 rNFECCCON0	= NFCON_BASE+0x20110,
 rNFECCCON1	= NFCON_BASE+0x20114,  
 rNFECCCON2	= NFCON_BASE+0x20118,   
 rNFECCCON3	= NFCON_BASE+0x2011C,  
 rNFECCCON4	= NFCON_BASE+0x20120,    
 rNFECCCON5	= NFCON_BASE+0x20124,     
 rNFECCCON6	= NFCON_BASE+0x20128    

}; 


typedef struct st_NandInfo
{
	u32 uNId;
	u8 uId[NUM_ID];
	char sName[SIZE_OF_MAN_NAME];
	u32 uNandSize;
	u32 uTotalBlockNum;
	u32 uColCycle;     //PageAddr      
	u32 uRowCycle;		//Nth PageAddr  
	u32 uAddrCycle;	//uAddrCycle = uColCycle + uRowCycle
	u32 uIsMLC;
	u32 uAddrCycleNum;
	u32 uBlockShift;
	u8 uRowAddr[N_ROW_ADDRESS];
	u32 u512BytesPerPage;
	
	u32 uCurBlock;
	u32 uCurPage;
	u32 uPagesPerBlock;
	
	// 3rd ID data
	u32 uIntChipNum;
	u32 uCellType;
	u32 uNSimProgPages; //Num of Simultaneously Prgrammed Pages
	u32 uInterleaveProg;//Interleave Program Between multiple chips
	u32 uCacheProg;    //Cache Program
	
	// 4th ID Data
	u32 uPageSize;
	u32 uBlockSize;
	u32 uRedundantAreaSize;
	u32 uOrganization;
	u32 uSerialAccessMin;
		
	// 5th ID Data
	u32 uPlaneNum;
	u32 uPlaneSize;	
		
}NANDINFO;

typedef enum tagECCType
{
	NO_ECC = 0,
	ECC_1BIT = 1,
	ECC_4BIT =4,
	ECC_8BIT =8,
	ECC_12BIT =12,
	ECC_16BIT =16
	
} ECCTYPE;



typedef struct tagErrorList
{
	u32 startno;
	u32 rand_coeff[1];
	u32 madeErrorType;
	u32 checkedErrorType;
	u32 madeErrorByte[16];
	u32 checkedErrorByte[16];
	u32 madeErrorBit[16];
	u32 checkedErrorBit[16];

} ErrorList;




#define TACLS		1	// 1-clk(7.5ns) 
#define TWRPH0		4	// 3-clk(22.5ns)
#define TWRPH1		1	// 2-clk(15ns)  //TACLS+TWRPH0+TWRPH1>=30ns



#define NF_CMD(cmd)			Outp32(rNFCMD, (cmd))
#define NF_ADDR(addr)		Outp32(rNFADDR, (addr))	


#define NF_nFCE3_L()			Outp32(rNFCONT,Inp32(rNFCONT)&~(1<<23))
#define NF_nFCE3_H()		Outp32(rNFCONT,Inp32(rNFCONT)|(1<<23))
#define NF_nFCE2_L()			Outp32(rNFCONT,Inp32(rNFCONT)&~(1<<22))
#define NF_nFCE2_H()		Outp32(rNFCONT,Inp32(rNFCONT)|(1<<22))
#define NF_nFCE1_L()			Outp32(rNFCONT,Inp32(rNFCONT)&~(1<<2))
#define NF_nFCE1_H()		Outp32(rNFCONT,Inp32(rNFCONT)|(1<<2))
#define NF_nFCE0_L()			Outp32(rNFCONT,Inp32(rNFCONT)&~(1<<1))
#define NF_nFCE0_H()		Outp32(rNFCONT,Inp32(rNFCONT)|(1<<1))

#define NF_RDDATA() 			(Inp32(rNFDATA))
#define NF_RDDATA8() 		(Inp8(rNFDATA))
#define NF_RDDATA16() 		(Inp16(rNFDATA))
#define NF_WRDATA(data) 	Outp32(rNFDATA, (data))
#define NF_WRDATA8(data) 	Outp8(rNFDATA, (data))

// RnB Signal
#define NF_CLEAR_RB()    			Outp32(rNFSTAT,Inp32(rNFSTAT)|(1<<4)) // write '1' to clear this bit.
#define NF_CLEAR_RB0()    		Outp32(rNFSTAT,Inp32(rNFSTAT)|(1<<24)) // == NF_CLEAR_RB().
#define NF_CLEAR_RB1()    		Outp32(rNFSTAT,Inp32(rNFSTAT)|(1<<25)) // 
#define NF_CLEAR_RB2()    		Outp32(rNFSTAT,Inp32(rNFSTAT)|(1<<26)) // 
#define NF_CLEAR_RB3()    		Outp32(rNFSTAT,Inp32(rNFSTAT)|(1<<27)) // 

#define NF_DETECT_RB()    	{while(!(Inp32(rNFSTAT)&(1<<4)));}	// 
#define NF_DETECT_RB0()    	{while(!(Inp32(rNFSTAT)&(1<<24)));}	// == NF_DETECT_RB()
#define NF_DETECT_RB1()    	{while(!(Inp32(rNFSTAT)&(1<<25)));}
#define NF_DETECT_RB2()    	{while(!(Inp32(rNFSTAT)&(1<<26)));}
#define NF_DETECT_RB3()    	{while(!(Inp32(rNFSTAT)&(1<<27)));}

#define RNB_INT_ENABLE()		Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9)|(1<<9)); // Enable RnB interrupt
#define RNB_INT_DISABLE()		Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9)|(0<<9)); // Disable RnB interrupt

// soft lock
//#define IS_LOCKED()	    		(Inp32(rNFSTAT)&(1<<5))? (1):(0) 
#define IS_LOCKED()	    		(Inp32(rNFSTAT)&(1<<5))			// check if it is illega access. 
#define ILL_ACC_CLEAR() 	Outp32(rNFSTAT, Inp32(rNFSTAT) | (1<<5));  // clear illegal access status bit
#define ILLACC_INT_ENABLE()			Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10)|(1<<10)); // Enable illegal access interrupt
#define ILLACC_INT_DISABLE()		Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10)|(0<<10)); // Disable illegal access interrupt



// ECC
#define NF_MECC_UnLock()      Outp32(rNFCONT,Inp32(rNFCONT)&~(1<<7))			// for 1/4/8/12/16 bit ECC
#define NF_MECC_Lock()         	Outp32(rNFCONT,Inp32(rNFCONT)|(1<<7))				//
#define NF_SECC_UnLock()       Outp32(rNFCONT,Inp32(rNFCONT)&~(1<<6))			// for 1bit ECC
#define NF_SECC_Lock()         	Outp32(rNFCONT,Inp32(rNFCONT)|(1<<6))				// for 1bit ECC

#define NF_RSTECC()			{Outp32(rNFCONT,Inp32(rNFCONT)|(1<<5)|(1<<4));\
												Outp32(rNFECCCONT,Inp32(rNFECCCONT)|(1<<2));}			// for 1/4/8/12/16bit ECC

// 1/4bit ECC
#define UNSEL_ECC_1_4() 		Outp32(rNFCONF, Inp32(rNFCONF) | (0x3<<23) );   //  unselect Ecc controller 0 (1/4bit Ecc)
#define SEL_ECC4() 		Outp32(rNFCONF, Inp32(rNFCONF) &~(0x3<<23)|(0x2<<23));  	// select 4bit ecc
#define SEL_ECC1() 		Outp32(rNFCONF, Inp32(rNFCONF) &~(0x3<<23)|(0x0<<23));  	// select 1bit ecc

#define ECC4_SET_MSGLEN_512() Outp32(rNFCONF, Inp32(rNFCONF) &~(1<<25)|(0<<25));  	// set 4bit ECC msg length to 512byte
#define ECC4_SET_MSGLEN_24() Outp32(rNFCONF, Inp32(rNFCONF) &~(1<<25)|(1<<25));  	// set 4bit ECC msg length to 24byte

#define ECC4ENC_INT_ENABLE()			Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<13)|(1<<13)); // Enable 4bit Ecc Encoding completion interrupt
#define ECC4ENC_INT_DISABLE()			Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<13)|(0<<13)); // Disable 4bit Ecc Encoding completion interrupt
#define ECC4DEC_INT_ENABLE()			Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<12)|(1<<12)); // Enable 4bit Ecc Decoding completion interrupt
#define ECC4DEC_INT_DISABLE()			Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<12)|(0<<12)); // Disable 4bit Ecc Decoding completion interrupt

#define ECC4_ENC_DONE_CLEAR() 		Outp32(rNFSTAT, Inp32(rNFSTAT) | (1<<7));  // clear 4bit ecc encode done
#define ECC4_DEC_DONE_CLEAR() 		Outp32(rNFSTAT, Inp32(rNFSTAT) | (1<<6));  // clear 4bit ecc decode done

#define ECC4_SETDIR_DEC()			Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<18)|(0<<18)); // set 4bit Ecc direction to Decoding
#define ECC4_SETDIR_ENC()			Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<18)|(1<<18)); // set 4bit Ecc direction to Encoding
#define ECC4_LOOP_UNTIL_ENC_DONE() 	while(!(Inp32(rNFSTAT)&(1<<7))) ;
#define ECC4_LOOP_UNTIL_DEC_DONE() 	while(!(Inp32(rNFSTAT)&(1<<6))) ;
#define ECC4_LOOP_WHILE_DEC_BUSY()	while(!(Inp32(rNFECCERR0)&(1<<31))) ;

// 8/12/16bit ECC
#define UNSEL_ECC_8_12_16() 	Outp32(rNFECCCONF , (Inp32(rNFECCCONF)&~(0xf<<0))|(0<<0));  //  unselect 8/12/16bit ECC 
#define SEL_ECC8() 		Outp32(rNFECCCONF , (Inp32(rNFECCCONF)&~(0xf<<0))|(3<<0));  //  select 8bit ECC 
#define SEL_ECC12() 		Outp32(rNFECCCONF , (Inp32(rNFECCCONF)&~(0xf<<0))|(0x4<<0));  //  select 12bit ECC 
#define SEL_ECC16()	 	Outp32(rNFECCCONF , (Inp32(rNFECCCONF)&~(0xf<<0))|(0x5<<0));  //  select 16bit ECC 

#define ECC8_SET_MSGLEN(msg) 		Outp32(rNFECCCONF , (Inp32(rNFECCCONF)&~(0x3ff<<16))|(((msg)-1)<<16));	 //  set 8/12/16bit ECC message length  to msg

#define ECC8_SETDIR_DEC()			Outp32(rNFECCCONT, Inp32(rNFECCCONT) & ~(1<<16)|(0<<16)); // set 8/12/16bit Ecc direction to Decoding
#define ECC8_SETDIR_ENC()			Outp32(rNFECCCONT, Inp32(rNFECCCONT) & ~(1<<16)|(1<<16)); // set 8/12/16bit Ecc direction to Encoding

#define ECC8ENC_INT_ENABLE()		Outp32(rNFECCCONT, Inp32(rNFECCCONT) & ~(1<<25)|(1<<25)); // Enable 8/12/16bit Ecc Encoding completion interrupt
#define ECC8ENC_INT_DISABLE()		Outp32(rNFECCCONT, Inp32(rNFECCCONT) & ~(1<<25)|(0<<25)); // Disable 8/12/16bit Ecc Encoding completion interrupt
#define ECC8DEC_INT_ENABLE()		Outp32(rNFECCCONT, Inp32(rNFECCCONT) & ~(1<<24)|(1<<24)); // Enable 8/12/16bit Ecc Decoding completion interrupt
#define ECC8DEC_INT_DISABLE()		Outp32(rNFECCCONT, Inp32(rNFECCCONT) & ~(1<<24)|(0<<24)); // Disable 8/12/16bit Ecc Decoding completion interrupt

#define ECC8_ENC_DONE_CLEAR() 		Outp32(rNFECCSTAT, Inp32(rNFECCSTAT) | (1<<25));  // clear 8/12/16bit ecc encode done
#define ECC8_DEC_DONE_CLEAR() 		Outp32(rNFECCSTAT, Inp32(rNFECCSTAT) | (1<<24));  // clear 8/12/16bit ecc decode done

#define ECC8_LOOP_UNTIL_DEC_DONE() 		 while(!(Inp32(rNFECCSTAT)&(1<<24)));
#define ECC8_LOOP_UNTIL_ENC_DONE() 		 while(!(Inp32(rNFECCSTAT)&(1<<25)));
#define ECC8_LOOP_WHILE_DEC_BUSY()		while( Inp32(rNFECCSTAT) & (u32)(1<<31) ) ; // 8bit ECC Decoding Busy Check.

void __irq tNFCon_Int(void);
void InitNFC(void);
u8 ReadNandStatus(void);
void TestNFC(void);
u32 ReadID(void);
u32  ResetNand(void);
int EraseBlock(u32 uBlock);
void PrintSubMessage(void);
u32 TestReadNAND(void);
u32 TestWriteNAND(void);
int WritePage(u8* pWBuf,u32  uBlock, u32 uPage);
u32  NAND_Write2048Page_8BitValid( u8 *pBuffer, u32 uBlock, u32 uPage);
u32 TestEraseBlock(void);
void WriteDramImage2Nand(void);
void ReadNandImage2Dram(void);
u32 SetLock(u32 uStartBlock, u32 uEndBlock, u32 uIsSoftLock);
void Test_Lock(void);
void ClearSoftLock(void);
void SetECC(ECCTYPE ecctype);
u32 TestECC_8bit(void);	


#ifdef __cplusplus
}
#endif

#endif //__NAND_H__

