#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "def.h"
#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "sysc.h"
#include "nand.h"
#include "timer.h"
#include "intc.h"
#include "gpio.h"

#define		__POLLING		(1)
#define		_true		(1)
#define		_false		(0)		
#define 		OK		(1)
#define		FAIL	(0)
#define		_NONCACHE_STARTADDRESS	 (0x21000000)

/*
*	Chip ID list
*
*	Name. ID code, pagesize, chipsize in MegaByte, eraseblock size,
*	options
*
* 	Pagesize; 0, 256, 512
*	0 	get this information from the extended chip ID
+	256	256 Byte page size
*	512	512 Byte page size
*/
struct nand_flash_dev nand_flash_ids[] = {
	{"NAND 1MiB 5V 8-bit", 		0x6e, 256, 1, 0x1000, 0},
	{"NAND 2MiB 5V 8-bit", 		0x64, 256, 2, 0x1000, 0},
	{"NAND 4MiB 5V 8-bit", 		0x6b, 512, 4, 0x2000, 0},
	{"NAND 1MiB 3,3V 8-bit", 	0xe8, 256, 1, 0x1000, 0},
	{"NAND 1MiB 3,3V 8-bit", 	0xec, 256, 1, 0x1000, 0},
	{"NAND 2MiB 3,3V 8-bit", 	0xea, 256, 2, 0x1000, 0},
	{"NAND 4MiB 3,3V 8-bit", 	0xd5, 512, 4, 0x2000, 0},
	{"NAND 4MiB 3,3V 8-bit", 	0xe3, 512, 4, 0x2000, 0},
	{"NAND 4MiB 3,3V 8-bit", 	0xe5, 512, 4, 0x2000, 0},
	{"NAND 8MiB 3,3V 8-bit", 	0xd6, 512, 8, 0x2000, 0},

	{"NAND 8MiB 1,8V 8-bit", 	0x39, 512, 8, 0x2000, 0},
	{"NAND 8MiB 3,3V 8-bit", 	0xe6, 512, 8, 0x2000, 0},
	{"NAND 8MiB 1,8V 16-bit", 	0x49, 512, 8, 0x2000, NAND_BUSWIDTH_16},
	{"NAND 8MiB 3,3V 16-bit", 	0x59, 512, 8, 0x2000, NAND_BUSWIDTH_16},

	{"NAND 16MiB 1,8V 8-bit", 	0x33, 512, 16, 0x4000, 0},
	{"NAND 16MiB 3,3V 8-bit", 	0x73, 512, 16, 0x4000, 0},
	{"NAND 16MiB 1,8V 16-bit", 	0x43, 512, 16, 0x4000, NAND_BUSWIDTH_16},
	{"NAND 16MiB 3,3V 16-bit", 	0x53, 512, 16, 0x4000, NAND_BUSWIDTH_16},

	{"NAND 32MiB 1,8V 8-bit", 	0x35, 512, 32, 0x4000, 0},
	{"NAND 32MiB 3,3V 8-bit", 	0x75, 512, 32, 0x4000, 0},
	{"NAND 32MiB 1,8V 16-bit", 	0x45, 512, 32, 0x4000, NAND_BUSWIDTH_16},
	{"NAND 32MiB 3,3V 16-bit", 	0x55, 512, 32, 0x4000, NAND_BUSWIDTH_16},

	{"NAND 64MiB 1,8V 8-bit", 	0x36, 512, 64, 0x4000, 0},
	{"NAND 64MiB 3,3V 8-bit", 	0x76, 512, 64, 0x4000, 0},
	{"NAND 64MiB 1,8V 16-bit", 	0x46, 512, 64, 0x4000, NAND_BUSWIDTH_16},
	{"NAND 64MiB 3,3V 16-bit", 	0x56, 512, 64, 0x4000, NAND_BUSWIDTH_16},

	{"NAND 128MiB 1,8V 8-bit", 	0x78, 512, 128, 0x4000, 0},
	{"NAND 128MiB 1,8V 8-bit", 	0x39, 512, 128, 0x4000, 0},
	{"NAND 128MiB 3,3V 8-bit", 	0x79, 512, 128, 0x4000, 0},
	{"NAND 128MiB 1,8V 16-bit", 0x72, 512, 128, 0x4000, NAND_BUSWIDTH_16},
	{"NAND 128MiB 1,8V 16-bit", 0x49, 512, 128, 0x4000, NAND_BUSWIDTH_16},
	{"NAND 128MiB 3,3V 16-bit", 0x74, 512, 128, 0x4000, NAND_BUSWIDTH_16},
	{"NAND 128MiB 3,3V 16-bit", 0x59, 512, 128, 0x4000, NAND_BUSWIDTH_16},

	{"NAND 256MiB 3,3V 8-bit", 	0x71, 512, 256, 0x4000, 0},

	/* These are the new chips with large page size. The pagesize
	* and the erasesize is determined from the extended id bytes
	*/
	/*512 Megabit */
	{"NAND 64MiB 1,8V 8-bit", 	0xA2, 0,  64, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 64MiB 3,3V 8-bit", 	0xF2, 0,  64, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 64MiB 1,8V 16-bit", 	0xB2, 0,  64, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},
	{"NAND 64MiB 3,3V 16-bit", 	0xC2, 0,  64, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},

	/* 1 Gigabit */
	{"NAND 128MiB 1,8V 8-bit", 	0xA1, 0, 128, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 128MiB 3,3V 8-bit", 	0xF1, 0, 128, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 128MiB 1,8V 16-bit", 	0xB1, 0, 128, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},
	{"NAND 128MiB 3,3V 16-bit", 	0xC1, 0, 128, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},

	/* 2 Gigabit */
	{"NAND 256MiB 1,8V 8-bit", 	0xAA, 0, 256, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 256MiB 3,3V 8-bit", 	0xDA, 0, 256, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 256MiB 1,8V 16-bit", 	0xBA, 0, 256, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},
	{"NAND 256MiB 3,3V 16-bit", 	0xCA, 0, 256, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},

	/* 4 Gigabit */
	{"NAND 512MiB 1,8V 8-bit", 	0xAC, 0, 512, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 512MiB 3,3V 8-bit", 	0xDC, 0, 512, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 512MiB 1,8V 16-bit", 	0xBC, 0, 512, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},
	{"NAND 512MiB 3,3V 16-bit", 	0xCC, 0, 512, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},

	/* 8 Gigabit */
	{"NAND 1GiB 1,8V 8-bit", 	0xA3, 0, 1024, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 1GiB 3,3V 8-bit", 	0xD3, 0, 1024, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 1GiB 1,8V 16-bit", 	0xB3, 0, 1024, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},
	{"NAND 1GiB 3,3V 16-bit", 	0xC3, 0, 1024, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},

	/* 16 Gigabit */
	{"NAND 2GiB 1,8V 8-bit", 	0xA5, 0, 2048, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 2GiB 3,3V 8-bit", 	0xD5, 0, 2048, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_NO_AUTOINCR},
	{"NAND 2GiB 1,8V 16-bit", 	0xB5, 0, 2048, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},
	{"NAND 2GiB 3,3V 16-bit", 	0xC5, 0, 2048, 0, NAND_SAMSUNG_LP_OPTIONS | NAND_BUSWIDTH_16 | NAND_NO_AUTOINCR},

	/* Renesas AND 1 Gigabit. Those chips do not support extended id and have a strange page/block layout !
	 * The chosen minimum erasesize is 4 * 2 * 2048 = 16384 Byte, as those chips have an array of 4 page planes
	 * 1 block = 2 pages, but due to plane arrangement the blocks 0-3 consists of page 0 + 4,1 + 5, 2 + 6, 3 + 7
	 * Anyway JFFS2 would increase the eraseblock size so we chose a combined one which can be erased in one go
	 * There are more speed improvements for reads and writes possible, but not implemented now
	 */
	{"AND 128MiB 3,3V 8-bit",	0x01, 2048, 128, 0x4000, NAND_IS_AND | NAND_NO_AUTOINCR | NAND_4PAGE_ARRAY | BBT_AUTO_REFRESH},

	{NULL,}
};


static u8 g_uSpareData[128];
//static u8 g_error_byte_pos[];

/*
*	Manufacturer ID list
*/
struct nand_manufacturers nand_manuf_ids[] = {
	{MAN_ID_TOSHIBA, "Toshiba"},
	{MAN_ID_SAMSUNG, "Samsung"},
	{MAN_ID_FUJITSU, "Fujitsu"},
	{MAN_ID_NATIONAL, "National"},
	{MAN_ID_RENESAS, "Renesas"},
	{MAN_ID_STMICRO, "ST Micro"},
    {MAN_ID_HYNIX, "Hynix"},
	{0x0, "Unknown"}
};



	


NANDINFO	sNandInfo;

static int NFConDone;
static int NFECCDecDone;
static int NFECCEncDone;

static ECCTYPE g_ecc_type = ECC_8BIT;
static u32 g_msg_length = 512;  //#define		PAGE_LEN_ECC8		512
static u32 g_ecc_code_length =13;   //#define		VAL_LEN_ECC8		13

// variable for Ecc test
static ErrorList g_err_list[100];
static u32 g_startno, g_rand_coeff[1];
static u32 g_ecc_err_cnt=0;
static u32 g_madeErrorType,g_checkedErrorType;
static u32 g_madeErrorByte[16],g_checkedErrorByte[16];
static u32 g_madeErrorBit[16],g_checkedErrorBit[16];


u32 PrintNandInfo(void)
{
	if(!sNandInfo.uPageSize)
	{
		UART_Printf("\n Error : No Page Size");
	}
	else
	{		
	UART_Printf("\n===========================================================");
	UART_Printf("\n1th ID : 0x%x\t Manufacterer", sNandInfo.uId[0]);
	UART_Printf("\n2nd ID : 0x%x\t Device ID", sNandInfo.uId[1]);
	UART_Printf("\n3rh ID : 0x%x", sNandInfo.uId[2]);
	UART_Printf("\n4th ID : 0x%x", sNandInfo.uId[3]);
	UART_Printf("\n5th ID : 0x%x", sNandInfo.uId[4]);
	UART_Printf("\nNand Size :  %ld MB, \t%d B", sNandInfo.uNandSize>>20, sNandInfo.uNandSize);
	UART_Printf("\nPage Size : %d Bytes", sNandInfo.uPageSize);
	UART_Printf("\nBlock Size : %d KBytpes\n", sNandInfo.uBlockSize >> 10);
	UART_Printf("\n512Bytes Per Page : %d", sNandInfo.u512BytesPerPage);
	
	UART_Printf("\nNand Col Cycle : %d", sNandInfo.uColCycle);
	UART_Printf("\nNand Row Cycle : %d", sNandInfo.uRowCycle);	
	UART_Printf("\nNand Row Addr By Block Size : %d", sNandInfo.uBlockShift);
	UART_Printf("\nPages per Block : %d pages\n", sNandInfo.uPagesPerBlock);
	
	UART_Printf("\nNand Total Address : A%d", sNandInfo.uAddrCycleNum);
	UART_Printf("\nIs MLC ? : %d", sNandInfo.uIsMLC);
	UART_Printf("\nInter Chip Number : %d", sNandInfo.uIntChipNum);
	UART_Printf("\nCell Type : %d Level Cell", sNandInfo.uCellType);
	UART_Printf("\nNumber of Simultaneously Programmed Pages : %d", sNandInfo.uNSimProgPages);
	UART_Printf("\nInterleave Program Between multiple chips : %d", sNandInfo.uInterleaveProg);
	UART_Printf("\nCache Program : %d", sNandInfo.uCacheProg);	
	UART_Printf("\nRedundant Area Size : %d(byte/512bytes)", sNandInfo.uRedundantAreaSize);
	UART_Printf("\nOrganization : x%d", sNandInfo.uOrganization);
	UART_Printf("\nSerial Access Minimum : %d", sNandInfo.uSerialAccessMin);
	UART_Printf("\nPlane Number : %d", sNandInfo.uPlaneNum);
	UART_Printf("\nPlane Size : %dMB", sNandInfo.uPlaneSize>>20);
	UART_Printf("\n===========================================================");
	}
	return 0;	
}


void SetConfiguration(ECCTYPE ecc, u32 add_cycle,u32 row_cycle, u32 pg_size, u32 pg_per_block, u32 total_blk, u32 spare_size)
{
	u32 i,uSize;
	SetECC(ecc);

	sNandInfo.uAddrCycle = add_cycle;
	sNandInfo.uRowCycle = row_cycle;
	sNandInfo.uPageSize = pg_size;
	sNandInfo.u512BytesPerPage = sNandInfo.uPageSize/512 ;
	
	sNandInfo.uColCycle = sNandInfo.uAddrCycle - sNandInfo.uRowCycle ;

	sNandInfo.uPagesPerBlock = pg_per_block;
	uSize = pg_per_block;
	for(i=0; uSize>1; i++)			uSize>>=1;	
	sNandInfo.uBlockShift = i;
	
	sNandInfo.uTotalBlockNum = total_blk;	
	
	sNandInfo.uRedundantAreaSize = spare_size/sNandInfo.u512BytesPerPage;	// spare area for 512byte
	sNandInfo.uNandSize = sNandInfo.uPageSize * sNandInfo.uPagesPerBlock * sNandInfo.uTotalBlockNum;

	// ............................


}

void DisplayConfiguration(void)
{
	UART_Printf("\n====================================\n");
	UART_Printf("Ecc type : %d-bit 	\n",g_ecc_type);
	UART_Printf("Address Cycle : %d \t",sNandInfo.uAddrCycle);
	UART_Printf("(Row : %d \t",sNandInfo.uRowCycle);
	UART_Printf("Column : %d )\n",sNandInfo.uColCycle);
	UART_Printf("one page size : %d byte\n", sNandInfo.uPageSize);
	UART_Printf("spare area size : %d byte\n", sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage);
	UART_Printf("page count per one block : %d \n", sNandInfo.uPagesPerBlock);
	UART_Printf("Total block count : %d \n", sNandInfo.uTotalBlockNum);
	UART_Printf("Nand Size(w/o spare area) : %d byte\n", sNandInfo.uNandSize);
	UART_Printf("Conversion code : 0x%x\t0x%x\t0x%x\t0x%x \n", Inp32(rNFECCCON0),Inp32(rNFECCCON1),Inp32(rNFECCCON2),Inp32(rNFECCCON3));
	UART_Printf("====================================\n");
	// .................
	// ................
	// ............
	


}


void ChangeConfiguration(void)
{
	ECCTYPE ecc;
	u32 add_cycle, row_cycle, pg_size, pg_per_block, total_blk, spare_size, i,j;
	UART_Printf("\n Select number to change..\n");
	UART_Printf("1. Address cycle			2.Row address cycle		3.Page size(byte)\n");
	UART_Printf("4. page count per block		5.Total block count			6.spare size per page(byte)\n");
	UART_Printf("7. Ecc type		              8.conversion code         \n");
	i = UART_GetIntNum();
	switch(i)
	{
		case 1 :
			UART_Printf("\nInput Address cycle(4 or 5)...");
			add_cycle = UART_GetIntNum();
			SetConfiguration( g_ecc_type, add_cycle, sNandInfo.uRowCycle,\
				sNandInfo.uPageSize, sNandInfo.uPagesPerBlock,sNandInfo.uTotalBlockNum,\
				sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage);
			break;
		case 2 :
			UART_Printf("\nInput Row Address cycle...");
			row_cycle = UART_GetIntNum();
			SetConfiguration( g_ecc_type, sNandInfo.uAddrCycle, row_cycle,\
				sNandInfo.uPageSize, sNandInfo.uPagesPerBlock,sNandInfo.uTotalBlockNum,\
				sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage);			
			break;
		case 3 :
			UART_Printf("\nInput page size(byte)...");
			pg_size = UART_GetIntNum();
			SetConfiguration( g_ecc_type, sNandInfo.uAddrCycle, sNandInfo.uRowCycle,\
				pg_size, sNandInfo.uPagesPerBlock,sNandInfo.uTotalBlockNum,\
				sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage);			
			break;
		case 4 :
			UART_Printf("\nInput page count per block...");
			pg_per_block = UART_GetIntNum();
			SetConfiguration( g_ecc_type, sNandInfo.uAddrCycle, sNandInfo.uRowCycle,\
				sNandInfo.uPageSize, pg_per_block,sNandInfo.uTotalBlockNum,\
				sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage);			
			break;
		case 5 :
			UART_Printf("\nInput Total block count...");
			total_blk= UART_GetIntNum();			
			SetConfiguration( g_ecc_type, sNandInfo.uAddrCycle, sNandInfo.uRowCycle,\
				sNandInfo.uPageSize, sNandInfo.uPagesPerBlock,total_blk,\
				sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage);			
			break;
		case 6 :
			UART_Printf("\nInput spare area size per page (byte) ...");
			spare_size = UART_GetIntNum();			
			SetConfiguration( g_ecc_type, sNandInfo.uAddrCycle, sNandInfo.uRowCycle,\
				sNandInfo.uPageSize, sNandInfo.uPagesPerBlock,sNandInfo.uTotalBlockNum,\
				spare_size);						
			break;
		case 7 :
			UART_Printf("\nInput Ecc type... \n");
			UART_Printf("0:No Ecc 	1: 1bit-Ecc 	4: 4bit-Ecc	8: 8bit-Ecc 	12: 12bit-Ecc 	16: 16bit-Ecc	\n");			
			ecc = UART_GetIntNum();	
			SetConfiguration( ecc, sNandInfo.uAddrCycle, sNandInfo.uRowCycle,\
				sNandInfo.uPageSize, sNandInfo.uPagesPerBlock,sNandInfo.uTotalBlockNum,\
				sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage);	
			break;		

		case 8 :

			UART_Printf("\n0:No conversion code(0x0) 	1: 8bit-512	\n");			
			j = UART_GetIntNum();	
		if (j==1)			
			{	
				Outp32(rNFECCCON0, 0xf6d1ae10);  		//  8bit /512byte ECC conversion code.
				Outp32(rNFECCCON1 , 0x3d656c12);
				Outp32(rNFECCCON2 , 0xdb1a8668);
				Outp32(rNFECCCON3 , 0x0000004a);
			}
		else
			{
				Outp32(rNFECCCON0, 0x0);  		//  8bit /512byte ECC conversion code.
				Outp32(rNFECCCON1 , 0x0);
				Outp32(rNFECCCON2 , 0x0);
				Outp32(rNFECCCON3 , 0x0);
			}
			break;			
		default :
			;
	}

	
	// ............................
	// .........................
	// .......................
	

	DisplayConfiguration();
}
	


u32  ResetNand(void)
{
	NF_nFCE0_L();

	NF_CLEAR_RB();
	NF_CMD(CMD_RESET);	//reset command
	NF_DETECT_RB();
	
	NF_nFCE0_H();
	return _true;
}

	/////// for S3C2450		///////////////////////////////////////////////////////////
	//--> [NFCONF]
	// TACLS			[14:12]	CLE&ALE duration = HCLK*TACLS.
	// TWRPH0			[10:8]	TWRPH0 duration = HCLK*(TWRPH0+1)
	// TWRPH1			[6:4]	TWRPH1 duration = HCLK*(TWRPH1+1)
	// PageSize(R)		[3]		NAND memory page size
	//							when [3]==0, 0:512, 1:2048 bytes/page.
	//							when [3]==1, 0:2048, 1:4096 bytes/page by PageSize_Ext
	// PageSize_Ext(R)	[2]		when [2]==0, Small Size Nand Flash
	//							when [2]==1, Large Size Nand Flash
	// AddrCycle(R)		[1]		NAND flash addr size
	//							when [3]==0, 0:3-addr, 1:4-addr.
	//							when [3]==1, 0:4-addr, 1:5-addr.
	// BusWidth(R/W) 	[0]		NAND bus width. 0:8-bit, 1:16-bit.
	
	//--> [NFCONT]
	// ECC Direction	[18]	0:Decoding(4/8 ECC)
	//							1:Encoding(4/8 ECC)
	// Lock-tight		[17]	0:Disable lock, 1:Enable lock.
	// Soft Lock		[16]	0:Disable lock, 1:Enable lock.
	// EnbECCDecINT 	[12]	0:Disable Interrupt,	1:Enable Interrupt (Enable/Disable 4/8ECC decoding Done Interrupt)
	// 8bit Stop		[11]	8-bit ECC en/decoding operation initialization
	// EnablillegalAcINT[10]	Illegal access interupt control. 0:Disable, 1:Enable
	// EnbRnBINT		[9]		RnB interrupt. 0:Disable, 1:Enable
	// RnB_TrandMode	[8]		RnB transition detection config. 0:Low to High, 1:High to Low
	// SpareECCLock		[7]		0:Unlock, 1:Lock
	// MainECCLock		[6]		0:Unlock, 1:Lock
	// InitMECC(W)		[5]		1:Init main area ECC decoder/encoder.
	// InitSECC(W)		[4]		1:Init spare area ECC decoder/encoder.
	// Reg_nCE1			[2]		0:nFCE=0, 1:nFCE=1.
	// Reg_nCE0			[1]		0:nFCE=0, 1:nFCE=1.
	// NANDC Enable		[0]		operating mode. 0:Disable, 1:Enable.
	
	
const testFuncMenu nfc_func[]=
{  	
	ChangeConfiguration,	"Change config  ",
	ReadID,					"Read ID         ",
	ResetNand,				"Nand reset      ",  
	TestEraseBlock,			"Erase Block     ",
	TestReadNAND,			"Read Page       ",  
	TestWriteNAND,			"Write Page      ",  
	TestECC_8bit,			"Test  Ecc	",
	Test_Lock,				"Lock Test       ",
	ClearSoftLock, 			"Clear Soft Lock ",
	WriteDramImage2Nand,	"Write Dram Image",
	ReadNandImage2Dram, 	"Read From Nand  ",
	0,0
};

u32 AutoTestNFC(void)
{
	return 0;
	
}

void NFC_Init(void)
{
	Outp32(rNFCONF , (TACLS<<12)|(TWRPH0<<8)|(TWRPH1<<4)|(0<<3)|(1<<2)|(1<<1)|(0<<0));	
	Outp32(rNFCONT , (0<<18)|(0<<17)|(0<<16)|(0<<10)|(0<<9)|(0<<8)|(0<<7)|(0<<6)|(0x3<<1)|(1<<0));
	Outp32(0xE02002E0,0x22333322);   	// MP0_1CON
	Outp32(0xE0200300,0x00002222);	// MP0_2CON
	Outp32(0xE0200320,0x22222222);	// MP0_3CON
	
	ResetNand();
}

void TestNFC(void)
{
	u32 i;
	s32 uSel;
	oFunctionT_AutoVar oAutoTest;

	//ReadID();
	NFC_Init();
	SetConfiguration(ECC_8BIT, 5,  3, 2048, 64, 2048, 64);
	DisplayConfiguration();

	UART_Printf("\n[NAND Test]\n");
	
	while(1)
	{
		UART_Printf("\n");
		for (i=0; (u32)(nfc_func[i].desc)!=0; i++)
			UART_Printf("%2d: %s\n", i, nfc_func[i].desc);

		UART_Printf("\nSelect the function to test : ");
		uSel =UART_GetIntNum();
		UART_Printf("\n");
		if(uSel == -1) 
			break;

		if (uSel>=0 && uSel<(sizeof(nfc_func)/8-1))
			(nfc_func[uSel].func) ();
	}
}



void SetECC(ECCTYPE ecctype)			// NFCON has two ECC controllers. One is for 1/4bit ECC, and the other is for 8/12/16bit ECC.
{										// If one ECC controller is used then the other should be OFF.
	g_ecc_type = ecctype;
	
	switch (ecctype)
	{
	case NO_ECC:
		UNSEL_ECC_1_4();
		UNSEL_ECC_8_12_16();
		g_ecc_code_length = 0;
		break;
		
	case ECC_1BIT:
		UNSEL_ECC_8_12_16();
		SEL_ECC1();
		g_ecc_code_length = 4;
		break;

	case ECC_4BIT:
		UNSEL_ECC_8_12_16();
		SEL_ECC4();
		g_ecc_code_length = 7;
		break;
		
	case ECC_8BIT:
		UNSEL_ECC_1_4();    			 //  unselect Ecc controller 0 (1/4bit Ecc)
		SEL_ECC8();
		g_ecc_code_length = 13;
		break;

	case ECC_12BIT:
		UNSEL_ECC_1_4();    			 //  unselect Ecc controller 0 (1/4bit Ecc)
		SEL_ECC12();
		g_ecc_code_length = 20;
		break;

	case ECC_16BIT:
		UNSEL_ECC_1_4();    			 //  unselect Ecc controller 0 (1/4bit Ecc)
		SEL_ECC16();
		g_ecc_code_length = 26;
		break;

	default:
		g_ecc_type = NO_ECC;
		UNSEL_ECC_1_4();
		UNSEL_ECC_8_12_16();
		g_ecc_code_length = 0;
			
	}
}

void SetECCMsgLen(int msglen)
{
	g_msg_length = msglen;
	
	if ((g_ecc_type==ECC_4BIT)&&(msglen == 512))
		{	ECC4_SET_MSGLEN_512();}											// ECC 4bit 512byte
	else if ((g_ecc_type==ECC_4BIT)&&(msglen == 24))		
		{ECC4_SET_MSGLEN_24();}												// ECC 4bit 24byte
	else if ((g_ecc_type==ECC_8BIT)||(g_ecc_type==ECC_12BIT)||(g_ecc_type==ECC_16BIT))
		{	 ECC8_SET_MSGLEN(msglen);}											// for 8/12/16bit ECC	(1~512) is allowed.
	else {
			g_ecc_type = NO_ECC;
			 UART_Printf("\n fail to set message length!ECC is changed to no ecc. \n");
		 }

}


u32 ReadID(void)
{
	u8 uId[16], i, uScan;	
	u32 uTemp, uSize;
	
	memset(&sNandInfo,0,sizeof(NANDINFO));
	
	NFC_Init();
	UART_Printf("\nK9F5608 : 2 IDs");
	UART_Printf("\nK9F1208 : 4 IDs");
	UART_Printf("\nK9HBG08U1M / K9MZG08U3M / K9LAG08U0M / K9MCG08U5M : 5 IDs");
	UART_Printf("\nK9GAG08U0M / K9HCG08U5M / K9GAG08B0M / K9LBG08U1M : 5 IDs");	
	UART_Printf("\n\nHow Many IDs? (2/3/4/5/6/7) : ");
		
	sNandInfo.uNId = 5;

	NF_nFCE0_L();
	NF_CMD(CMD_READID);
    NF_ADDR(0x00);

	for (i=0; i<10; i++);//delay

	for(i=0; i<sNandInfo.uNId; i++)	
		sNandInfo.uId[i] = NF_RDDATA8();
				
	NF_nFCE0_H();		
	
	
	if((sNandInfo.uId[0]==0xEC || sNandInfo.uId[0]==0x98) && ((sNandInfo.uId[1]==0x76)||(sNandInfo.uId[1]==0x75)))	sNandInfo.uNId = 4;
	
	if(sNandInfo.uNId < 5)
	{
		sNandInfo.uPageSize = 512;
		sNandInfo.uNandSize = 64*1024*1024;
		sNandInfo.uColCycle = 1;
		if((sNandInfo.uId[1]==0x75))
		{			
			sNandInfo.uRowCycle = 2;
		}
		else
		{			
			sNandInfo.uRowCycle = 3;
		}
		sNandInfo.uBlockSize = 16*1024;
		sNandInfo.u512BytesPerPage = 1;
		sNandInfo.uRedundantAreaSize = 16;
		
		//Scan Cell Type	
		uTemp = (sNandInfo.uId[2]>>2)& 0x03;
		sNandInfo.uCellType = 1;
		for(i=0;i<=uTemp;i++)	sNandInfo.uCellType *= 2;
		sNandInfo.uIsMLC = uTemp;		
	}
	else
	{
		//------> 3rd ID
		uTemp = sNandInfo.uId[2]& 0x03;
		//Scan Internal Chip Number
		sNandInfo.uIntChipNum = 1;					
		for(i=0;i<uTemp;i++)	sNandInfo.uIntChipNum *= 2;				
			
		//Scan Cell Type	
		uTemp = (sNandInfo.uId[2]>>2)& 0x03;
		sNandInfo.uCellType = 1;
		for(i=0;i<=uTemp;i++)	sNandInfo.uCellType *= 2;
		sNandInfo.uIsMLC = uTemp;
			
			
		uTemp = (sNandInfo.uId[2]>>4)& 0x03;
		sNandInfo.uNSimProgPages = 1;
		for(i=0;i<uTemp;i++)	sNandInfo.uNSimProgPages *= 2;				
			
		sNandInfo.uInterleaveProg = (sNandInfo.uId[2]>>6)& 0x01;
		sNandInfo.uCacheProg = (sNandInfo.uId[2]>>7)& 0x01;
		
		//------> 4th ID
		uTemp = sNandInfo.uId[3]& 0x03;
		sNandInfo.uPageSize = 1024;					
		for(i=0;i<uTemp;i++)	sNandInfo.uPageSize *=2;
		sNandInfo.u512BytesPerPage = sNandInfo.uPageSize >> 9;		
					
		uTemp = (sNandInfo.uId[3]>>2)& 0x01;
		if(uTemp)		sNandInfo.uRedundantAreaSize = 16;
		else			sNandInfo.uRedundantAreaSize = 8;
		
		uTemp = (sNandInfo.uId[3]>>4)& 0x03;
		sNandInfo.uBlockSize = 64*1024;
		for(i=0;i<uTemp;i++)	sNandInfo.uBlockSize *=2;
					
		uTemp = (sNandInfo.uId[3]>>6)& 0x01;//TBD
		if(uTemp)		sNandInfo.uOrganization = 16;
		else			sNandInfo.uOrganization = 8;

		//------> 5th ID
		uTemp = (sNandInfo.uId[4]>>2)& 0x03;
		sNandInfo.uPlaneNum = 1;					
		for(i=0;i<uTemp;i++)	sNandInfo.uPlaneNum *= 2;				
			
		uTemp = (sNandInfo.uId[4]>>4)& 0x07;
		sNandInfo.uPlaneSize = (8388608); //64*1024*1024/8
		for(i=0;i<uTemp;i++)	sNandInfo.uPlaneSize *=2;	
		
		sNandInfo.uNandSize = sNandInfo.uPlaneNum*sNandInfo.uPlaneSize;
		
		uTemp = sNandInfo.uPageSize;		
		for(i=0; uTemp>1; i++)		uTemp>>=1;
		uTemp = i;
		sNandInfo.uAddrCycleNum+=uTemp;
		for(i=1; uTemp>8; i++)		uTemp-=8;		
		sNandInfo.uColCycle = i;		
		
		uTemp = sNandInfo.uPageSize;
		uSize = sNandInfo.uNandSize;
		for(i=0; uTemp>1; i++)
		{	
			uTemp>>=1;			
			uSize>>=1;
		}
		for(i=0; uSize>1; i++)			uSize>>=1;
		uSize = i;
		sNandInfo.uAddrCycleNum+=uSize;
		for(i=1; uSize>8; i++)			uSize-=8;
		sNandInfo.uRowCycle = i;			
		
	}	
	sNandInfo.uAddrCycle = sNandInfo.uColCycle + sNandInfo.uRowCycle;	
	uTemp = sNandInfo.uPageSize;
	uSize = sNandInfo.uBlockSize;
	
	for(i=0; uTemp>1; i++)
	{	
		uTemp>>=1;			
		uSize>>=1;
	}
	if(sNandInfo.u512BytesPerPage > 1)  //large block
	{		
		Outp32(rNFCONF, Inp32(rNFCONF) | (1<<2));
		if(sNandInfo.uAddrCycle ==4) //addr cycle : 4
		{
			Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<1)); 
			if(sNandInfo.u512BytesPerPage == 4)
			{
				Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<3)); 
			}
			else
			{
				Outp32(rNFCONF, Inp32(rNFCONF) | (1<<3)); 
			}
		}
		else							//addr cycle : 5
		{
			Outp32(rNFCONF, Inp32(rNFCONF) | (1<<1)); 
			if(sNandInfo.u512BytesPerPage == 4)
			{
				Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<3));
			}
			else
			{
				Outp32(rNFCONF, Inp32(rNFCONF) | (1<<3)); 
			}
		}		
	}
	else  //small block
	{
		Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<2)); 
		if(sNandInfo.uAddrCycle ==3)
		{
			Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<1)); 			//addr cycle : 3
			if(sNandInfo.u512BytesPerPage == 1)
			{
				Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<3)); 
			}
			else
			{
				Outp32(rNFCONF, Inp32(rNFCONF) | (1<<3));
			}
		}
		else
		{
			Outp32(rNFCONF, Inp32(rNFCONF) | (1<<1)); 			//addr cycle : 4
			if(sNandInfo.u512BytesPerPage == 1)
			{
				Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<3)); 
			}
			else
			{
				Outp32(rNFCONF, Inp32(rNFCONF) | (1<<3)); 
			}
		}		
	}
	sNandInfo.uPagesPerBlock = uSize;
	for(i=0; uSize>1; i++)			uSize>>=1;	
	sNandInfo.uBlockShift = i;
	PrintNandInfo();	
	return _true;	
}


u8 ReadNandStatus(void)
{
	// Read status
	u8 ch;
	int i;
	
	NF_nFCE0_L();

	NF_CMD(CMD_STATUS);	
	for(i=0; i<10; i++);
	ch = NF_RDDATA();
	NF_nFCE0_H();

	return ch;
}


#define		DEBUG_MSG		0
void __irq IsrNFC(void)
{
    
    if(DEBUG_MSG)UART_Printf("[INT]");

	INTC_Disable(NUM_NFC);    //rINTMSK|=BIT_NFCON;
  	INTC_ClearSoftInt(NUM_NFC); //ClearPending(BIT_NFCON);  ???????
  	
	if(Inp32(rNFSTAT)&0x20) 
	{
		Outp32(rNFSTAT, Inp32(rNFSTAT) | 0x20);  
		if(DEBUG_MSG)UART_Printf("Illegal Access is detected!!!\n");
	}
	else if(Inp32(rNFSTAT)&0x40) 
	{
		if ((Inp32(rNFECCERR0)&(0x7<<26)) == 0x0)
		{
		   	if(DEBUG_MSG)UART_Printf("ECC OK!\n");
		}
		else 
		{
		   if(DEBUG_MSG)UART_Printf("ECC FAIL!\n");
		   UART_Printf("status0:0x%x|status1:0x%x|bit:0x%x\n", Inp32(rNFECCERR0), Inp32(rNFECCERR1),Inp32(rNFMLCBITPT));
		}	 
		NFECCDecDone=1;
		Outp32(rNFSTAT, Inp32(rNFSTAT) | 0x40);  
		if(DEBUG_MSG)UART_Printf("ECC decoding is completed!!!\n"); 
	}
	else if(Inp32(rNFSTAT)&0x10)
	{
		NFConDone=1;
		Outp32(rNFSTAT, Inp32(rNFSTAT) | 0x10); 
		if(DEBUG_MSG)UART_Printf("RnB is Detected!!!\n");
	}
	 else if(Inp32(rNFSTAT)&0x80) 
	{
		NFECCEncDone=1;
		Outp32(rNFSTAT, Inp32(rNFSTAT) | 0x80); 
	if(DEBUG_MSG)	UART_Printf("ECC Encoding is completed!!!\n"); 
	}
}

//#define		SET_NAND_INT(ISR_FUNC)

int EraseBlock(u32 uBlock)
{
	//u32 blockPage=(block<<5);    
	u32 uCurBlockAddr;// = (uBlock<<sNandInfo.uBlockShift);
	u32 i=0;
	
	uCurBlockAddr = (uBlock<<sNandInfo.uBlockShift);
	//rNFSTAT = rNFSTAT|(1<<6)|(1<<7)|(1<<5)|(1<<4);
	ILL_ACC_CLEAR();
	NF_CLEAR_RB0();
	ECC8_DEC_DONE_CLEAR();
	//Outp32(rNFSTAT, Inp32(rNFSTAT) | ((1<<6)|(1<<5)|(1<<4)));  
#if __POLLING	

#else
  	NFConDone=0;
	RNB_INT_ENABLE();
	ILLACC_INT_ENABLE();
 	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<9)); 
    	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<10)); 
   	INTC_SetVectAddr(NUM_NFC, IsrNFC);
	INTC_Enable(NUM_NFC);
#endif

	NF_nFCE0_L();    
	NF_CMD(CMD_ERASE1);   // Erase one block 1st command, Block Addr:A9-A25
	for(i=0; i<sNandInfo.uRowCycle; i++) //Row Addressing
		NF_ADDR((uCurBlockAddr>>(i<<3))&0xFF);
  
	NF_CLEAR_RB();
	NF_CMD(CMD_ERASE2);	// Erase one blcok 2nd command
#if __POLLING
	NF_DETECT_RB();
#else
    while(NFConDone==0);
	RNB_INT_DISABLE();	//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9));  
	ILLACC_INT_DISABLE(); 	//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10)); // Disable Illegal Access Interrupt
	INTC_Disable(NUM_NFC);
#endif

	if(IS_LOCKED())
	{
		ILL_ACC_CLEAR();	//Outp32(rNFSTAT, Inp32(rNFSTAT) | 0x20); 
		UART_Printf("\nError to erase a block");
		return FAIL;
	}

	NF_CMD(CMD_STATUS);   // Read status command	

	if (NF_RDDATA()&0x1) // Erase error
	{	
		NF_nFCE0_H();
//		rGPGDAT|=(1<<1);
		UART_Printf("[ERASE_ERROR:block#=%d]\n",uBlock);
		//NF8_MarkBadBlock(block); //MGR
		return FAIL;
	}
	else 
	{	
		NF_nFCE0_H();
		return OK;
	}
}



u32 TestEraseBlock(void)
{
	u32 block=0;

	//UART_Printf("SMC(K9S1208V0M) NAND Block erase\n");

	if((ReadNandStatus()&0x80)==0) {
		UART_Printf("Write protected.\n");
		return _false;
	}

	UART_Printf("Block # to erase: ");
	block = UART_GetIntNum();

	
	if(EraseBlock(block)==FAIL) return _false;

	UART_Printf("%d-block erased.\n", block);
	return _true;

}



//////////////////////////////////start of read//////////////////////////////

//////start of read 4bit ecc///
int CorrectECC4Data(u8 *pEncodingBaseAddr)
{	
	unsigned int i,k,total_error,uErrorByte[9],uValidErrorByte[8];//u32 uErrorByte1, uErrorByte2, uErrorByte3, uErrorByte4, uErrorByte5, uErrorByte6, uErrorByte7, uErrorByte8;	
	unsigned char uErrorBit[9];//u32 uErrorBit1, uErrorBit2, uErrorBit3, uErrorBit4, uErrorBit5, uErrorBit6, uErrorBit7, uErrorBit8;
	unsigned int uErrorType;


	// total error: total ecc error     uValidError : which is valid among uErrorByte[n]s.
	total_error=0;
	for(i=0;i<8;i++)
		if( ((Inp32(rNFECCSECSTAT)>>(8+i))&1) == 1)uValidErrorByte[total_error++]=i;

	
	//Error Type // total error bit number or uncorrectable error.
	uErrorType = (Inp32(rNFECCSECSTAT)>>0)&0x1f;
	
	//Error Byte //
	uErrorByte[1] = Inp32(rNFECCERL0)&0x3ff;
	uErrorByte[2] = (Inp32(rNFECCERL0)>>16)&0x3ff;
	
	uErrorByte[3] = (Inp32(rNFECCERL1))&0x3ff;
	uErrorByte[4] = (Inp32(rNFECCERL1)>>16)&0x3ff;	
	
	uErrorByte[5] = (Inp32(rNFECCERL2))&0x3ff;
	uErrorByte[6] = (Inp32(rNFECCERL2)>>16)&0x3ff;
	
	uErrorByte[7] = (Inp32(rNFECCERL3))&0x3ff;
	uErrorByte[8] = (Inp32(rNFECCERL3)>>16)&0x3ff;
	
	// Error Bit //

	uErrorBit[1] = Inp32(rNFECCERP0) &0xff;
	uErrorBit[2] = (Inp32(rNFECCERP0)>>8)&0xff;
	uErrorBit[3] = (Inp32(rNFECCERP0)>>16)&0xff;
	uErrorBit[4] = (Inp32(rNFECCERP0)>>24)&0xff;
	
	uErrorBit[5] = Inp32(rNFECCERP1)&0xff;
	uErrorBit[6] = (Inp32(rNFECCERP1)>>8)&0xff;
	uErrorBit[7] = (Inp32(rNFECCERP1)>>16)&0xff;
	uErrorBit[8] = (Inp32(rNFECCERP1)>>24)&0xff;

	
	
	if(!uErrorType) 
		return 0;
	if(uErrorType > 0x8) return 1;
	
	for(i=1;i<=uErrorType ;i++)	// uErrorType == total_error 
	{
		k= uValidErrorByte[i];
		if(uErrorByte[k] < 512)	
			pEncodingBaseAddr[uErrorByte[k] ]^=uErrorBit[k];
		else
		{
			;
		/*
			if(pNandInfo.uPageSize == 512)
			{
			}
			else if(pNandInfo.uPageSize == 2048)
			{
			}
			else if(pNandInfo.uPageSize == 4096)						
			{
				
			}
			*/
		}			
	}
	return 0;
	
}



int ReadPage4(u8* uRBuf,u32  uBlock, u32 uPage)
{
	int i,m;		
	u8 Addr_Cycle, option2;
	u32 uResult[8]={0,	};
	u8 *pSpare;
	u32 uCurPageAddr, uRet=0;
	pSpare = g_uSpareData;
	
	uCurPageAddr =  uBlock*sNandInfo.uPagesPerBlock + uPage;	 	
	
		//rNFCONF &= ~(0x3<<23)|(1<<23);
	//rNFCONT = ( rNFCONT&~(0x1<<18) ) |(0<<18)|(1<<11)|(1<<10)|(1<<9); // Init NFCONT
	Outp32(rNFSTAT, Inp32(rNFSTAT) | 0x20); 
	Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<18)); 
	Outp32(rNFCONF, Inp32(rNFCONF) & ~(0x3<<23));
	Outp32(rNFCONF, Inp32(rNFCONF) | (1<<23)); 
	Outp32(rNFSTAT, Inp32(rNFSTAT) | (1<<6)|(1<<7)|(1<<5)|(1<<4)); 	
		
	NF_MECC_Lock(); // Main ECC Lock
	NF_nFCE0_L(); // Chip Select Low
	NF_CLEAR_RB(); // RnB Clear
	
#if		__POLLING
#else	
	Outp32(rNFCONT, Inp32(rNFCONT) | (1<<12));  
	Outp32(rNFCONT, Inp32(rNFCONT) | (1<<11)); 
	Outp32(rNFCONT, Inp32(rNFCONT) | (1<<10)); 
	Outp32(rNFCONT, Inp32(rNFCONT) | (1<<9)); 
	
	NFConDone=0;
	NFECCDecDone=0;
	NFECCEncDone=0;

	INTC_SetVectAddr(NUM_NFC, IsrNFC);

	INTC_Enable(NUM_NFC);
 
#endif
	
	Outp32(rNFCONF, ( Inp32(rNFCONF)&~(0x3<<23)&~(0xf<<12)&~(0xf<<8)&~(0xf<<4) ) |(3<<23)|(0x7<<12)|(0x7<<8)|(0x7<<4));   
                                          // 30:reserved 23:unselect ecc0(1/4) TACLS,TWRPH0,1

//	rNFCONT = ( rNFCONT&~(0x1<<18) ) |(0<<18)|(0<<11)|(0<<10)|(0<<9)|(1<<6)|(1<<0); // Init NFCONT
	Outp32(rNFCONT, ( Inp32(rNFCONT)&~(0x1<<18) ) |(0<<18)|(1<<6)|(1<<0)); 
						 // 18:4bit ECC direction 6:SECC lock 0  0:MODE 1-Enable NAND controller
						 
	Outp32(rNFCONT , (Inp32(rNFCONT)&~(0x7<<9) )) ;		// 11:MLCStop 0 10: EnbleillegalAccINT 0 9:EnbRnBINT 0
	
	Outp32(rNFSTAT , Inp32(rNFSTAT)|(1<<6)|(1<<7)|(1<<5)|(1<<4)); 	// clear EncodeDone, DecodeDone,IllegalAccess,RnB_TransDetect


// ----------8/12/16 bit ECC -------- //
	Outp32(rNFECCCONF , (Inp32(rNFECCCONF)&~(0x3ff<<16)&~(0xf<<0))|(511<<16)|(3<<0));		// 8/12/16 ECC configuration
	                            //  16: 512 message length   0: 8bit ECC 
	Outp32(rNFECCCONT , (Inp32(rNFECCCONT)&~(1<<25)&~(1<<24)&~(1<<16))); 		
						// 25:EnbMCLEncINT 0   24:EnbMCLDecInt 0 16:EccDirection 0(read)
	Outp32(rNFECCSTAT , Inp32(rNFECCSTAT)|(1<<25)|(1<<24));  // Clear 25:EncodeDone 24:DecodeDone


 
	
	
	for(i=0; i<sNandInfo.u512BytesPerPage; i++)
	{
		NF_MECC_Lock(); // Main ECC Lock
		NF_CLEAR_RB(); // RnB Clear
		if(!i)
		{			
			NF_CMD(CMD_READ0);	// Read command		
			for(m=0; m<sNandInfo.uColCycle; m++) //Addr Cycle
				NF_ADDR(0);
			for(m=0; m<sNandInfo.uRowCycle; m++)
				NF_ADDR((uCurPageAddr>>(m<<3))&0xFF);	
			if(sNandInfo.uPageSize > 512)
				NF_CMD(CMD_READ0_2CYCLE);	// Read command	
			
#if __POLLING
				NF_DETECT_RB();
				NF_CLEAR_RB();   // RnB Pending Clear
#else
			    while(!NFConDone);
			    NFConDone = 0;				
#endif
		}
		else
		{
			NF_CMD(CMD_RANDOM_DATA_OUT_1ST_CYCLE); // Random Address Access = 512K start
			NF_ADDR(((g_msg_length*i)>>0)&0xFF);
			NF_ADDR(((g_msg_length*i)>>8)&0xFF);
			NF_CMD(CMD_RANDOM_DATA_OUT_2ND_CYCLE);
		}
		
		NF_MECC_UnLock(); // Main ECC Unlock
		NF_RSTECC();    // Initialize ECC
		
		for(m=0;m<g_msg_length;m++)  
			*uRBuf++ = NF_RDDATA8();	// Read Main data
			
		// Spare Area Address Setting.
		if(sNandInfo.uPageSize > 512)
		{			
			NF_CMD(CMD_RANDOM_DATA_OUT_1ST_CYCLE);
			NF_ADDR(((sNandInfo.uPageSize + g_ecc_code_length*i)>>0)&0xFF);
			NF_ADDR(((sNandInfo.uPageSize + g_ecc_code_length*i)>>8)&0xFF);
			NF_CMD(CMD_RANDOM_DATA_OUT_2ND_CYCLE);
		}
		
		
		for(m=0; m<g_ecc_code_length ; m++)
			*pSpare++ = NF_RDDATA8(); // Spare Data Read

		NF_MECC_Lock(); // Main ECC Lock

#if		__POLLING
		while(!(Inp32(rNFECCSTAT)&(1<<24))); 	// Check decoding done 
		Outp32(rNFECCSTAT, Inp32(rNFECCSTAT) | (1<<24));    // Decoding done Clear
#else
		while(!NFECCDecDone);
		NFECCDecDone=0;
#endif		
		while( Inp32(rNFECCSTAT) & (u32)(1<<31) ) ; // 8bit ECC Decoding Busy Check.
		uResult[i] = CorrectECC4Data((uRBuf-g_msg_length));
		//UART_Printf("\nuResult[%d] : %d", i, uResult[i]);
		uRet += uResult[i];
	}
#if		__POLLING	
	if(Inp32(rNFSTAT)&0x20)
	{
		Outp32(rNFSTAT, Inp32(rNFSTAT) | 0x20); 
		UART_Printf("\nError to erase a block");
		NF_nFCE0_H();  // Chip Select is High
		return FAIL;
	}
#else		
	Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9)); 
	Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10));  // Disable Illegal Access Interrupt
	Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<12)); 
#endif
	NF_nFCE0_H();  // Chip Select is High
	INTC_Disable(NUM_NFC);
	
	return uRet;
}



u32 ReadNAND_4BitECC(void)
{
	u32 block=0, page=0;
	u32 i;
	u8 * downPt;
	
	downPt=(u8 *)_NONCACHE_STARTADDRESS;
	for(i=0;i<128;i++)
		g_uSpareData[i]=0xFF;	
	UART_Printf(" NAND(xD-Picture Card) Page Read.\n");

	UART_Printf("Block # to read: ");
	block = UART_GetIntNum();
	UART_Printf("Page # to read: ");
	page = UART_GetIntNum();

	ECC4_SET_MSGLEN_512();		//g_msg_length=512; 
	ECC4_SETDIR_DEC();

	if(ReadPage4(downPt, block, page )) 		
	{
		UART_Printf("Read error.\n");
		return _false;
	}
	else
	{
		UART_Printf("Read OK.\n");	
	}
	
	// Print data.
	UART_Printf("Read data(%d-block,%d-page)\n", block, page);
	UART_Printf("NFMECC0: 0x%x\n", Inp32(rNFMECC0));
	
	for(i=0; i<sNandInfo.uPageSize; i++) 
	{
		if((i%16)==0) UART_Printf("\n%4x: ", i);
		UART_Printf("%02x ", *(u8 *)downPt++);
	}
	
	UART_Printf("\nSpare:");
	for(i=0; i<sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage; i++) 
	{
		if((i%16)==0) 
			UART_Printf("\n%4x: ", i);
	 	UART_Printf("%02x ", g_uSpareData[i]);
	}
	UART_Printf("\n");
	return _true;

}

/////end of read 4bit ecc////////


//////start of read 8bit ecc///
int CorrectECC8Data(u8 *pEncodingBaseAddr)
{	
	unsigned int i,k,total_error,uErrorByte[9],uValidErrorByte[8];//u32 uErrorByte1, uErrorByte2, uErrorByte3, uErrorByte4, uErrorByte5, uErrorByte6, uErrorByte7, uErrorByte8;	
	unsigned char uErrorBit[9];//u32 uErrorBit1, uErrorBit2, uErrorBit3, uErrorBit4, uErrorBit5, uErrorBit6, uErrorBit7, uErrorBit8;
	unsigned int uErrorType;


	// total error: total ecc error     uValidError : which is valid among uErrorByte[n]s.
	total_error=0;
	for(i=0;i<g_ecc_type;i++)
		if( ((Inp32(rNFECCSECSTAT)>>(8+i))&1) == 1)uValidErrorByte[total_error++]=i;

	
	//Error Type // total error bit number or uncorrectable error.
	uErrorType = (Inp32(rNFECCSECSTAT)>>0)&0x1f;
	
	//Error Byte //
	uErrorByte[1] = Inp32(rNFECCERL0)&0x3ff;
	uErrorByte[2] = (Inp32(rNFECCERL0)>>16)&0x3ff;
	
	uErrorByte[3] = (Inp32(rNFECCERL1))&0x3ff;
	uErrorByte[4] = (Inp32(rNFECCERL1)>>16)&0x3ff;	
	
	uErrorByte[5] = (Inp32(rNFECCERL2))&0x3ff;
	uErrorByte[6] = (Inp32(rNFECCERL2)>>16)&0x3ff;
	
	uErrorByte[7] = (Inp32(rNFECCERL3))&0x3ff;
	uErrorByte[8] = (Inp32(rNFECCERL3)>>16)&0x3ff;
	
	// Error Bit //

	uErrorBit[1] = Inp32(rNFECCERP0) &0xff;
	uErrorBit[2] = (Inp32(rNFECCERP0)>>8)&0xff;
	uErrorBit[3] = (Inp32(rNFECCERP0)>>16)&0xff;
	uErrorBit[4] = (Inp32(rNFECCERP0)>>24)&0xff;
	
	uErrorBit[5] = Inp32(rNFECCERP1)&0xff;
	uErrorBit[6] = (Inp32(rNFECCERP1)>>8)&0xff;
	uErrorBit[7] = (Inp32(rNFECCERP1)>>16)&0xff;
	uErrorBit[8] = (Inp32(rNFECCERP1)>>24)&0xff;

	
	
	if(!uErrorType) 
		return 0;

	if(uErrorType > g_ecc_type)
	{	
		UART_Printf("\nUncorrectable Error!!!\n");
		return 1;  //check if error count is less than correctable max error count. 'return 1' means uncorrectable error.
	}
	
	if(!(uErrorType == total_error))     //total error should be the same with error type number.
	{
		UART_Printf("\nECC controller operation error!\n");
		return 1;
	}

	
	for(i=1;i<=uErrorType ;i++)	// uErrorType == total_error 
	{
		k= uValidErrorByte[i];
		
		if(uErrorByte[k] < 512)	
			pEncodingBaseAddr[uErrorByte[k] ]^=uErrorBit[k];
		else
		{
			;
		/*
			if(pNandInfo.uPageSize == 512)
			{
			}
			else if(pNandInfo.uPageSize == 2048)
			{
			}
			else if(pNandInfo.uPageSize == 4096)						
			{
				
			}
		*/
		}			
	}
	return 0;
	
}



int ReadPage8(u8* uRBuf,u32  uBlock, u32 uPage)
{
	int i,m;		
	u8 Addr_Cycle, option2;
	u32 uResult[8]={0,	};
	u8 *pSpare;
	u32 uCurPageAddr, uRet=0;
	pSpare = g_uSpareData;
	
	uCurPageAddr =  uBlock*sNandInfo.uPagesPerBlock + uPage;	 	
	
		//rNFCONF &= ~(0x3<<23)|(1<<23);
	//rNFCONT = ( rNFCONT&~(0x1<<18) ) |(0<<18)|(1<<11)|(1<<10)|(1<<9); // Init NFCONT

	//Outp32(rNFSTAT, Inp32(rNFSTAT) | (1<<6)|(1<<7)|(1<<5)|(1<<4)); 		
	ILL_ACC_CLEAR();
	
	NF_MECC_Lock(); // Main ECC Lock
	NF_nFCE0_L(); // Chip Select Low
	NF_CLEAR_RB(); // RnB Clear
	
	//ECC8DEC_INT_DISABLE();
	//ECC8ENC_INT_DISABLE();
	ECC8_DEC_DONE_CLEAR();
	//ECC8_ENC_DONE_CLEAR();


	
	
#if		__POLLING
#else	
	ECC8DEC_INT_ENABLE();
	ILLACC_INT_ENABLE();
	RNB_INT_ENABLE();
	
	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<12));  
	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<11)); 
	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<10)); 
	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<9)); 
	
	NFConDone=0;
	NFECCDecDone=0;
	NFECCEncDone=0;

	INTC_SetVectAddr(NUM_NFC, IsrNFC);

	INTC_Enable(NUM_NFC);
 
#endif
	
	// Outp32(rNFCONF, ( Inp32(rNFCONF)&~(0x3<<23)&~(0xf<<12)&~(0xf<<8)&~(0xf<<4) ) |(3<<23)|(0x7<<12)|(0x7<<8)|(0x7<<4));   
                                          // 30:reserved 23:unselect ecc0(1/4) TACLS,TWRPH0,1

//	rNFCONT = ( rNFCONT&~(0x1<<18) ) |(0<<18)|(0<<11)|(0<<10)|(0<<9)|(1<<6)|(1<<0); // Init NFCONT
	

	//Outp32(rNFCONT, ( Inp32(rNFCONT)&~(0x1<<18) ) |(0<<18)|(1<<6)|(1<<0)); 
						 // 18:4bit ECC direction 6:SECC lock 0  0:MODE 1-Enable NAND controller
						 
	//Outp32(rNFCONT , (Inp32(rNFCONT)&~(0x7<<9) )) ;		// 11:MLCStop 0 10: EnbleillegalAccINT 0 9:EnbRnBINT 0
	
	//Outp32(rNFSTAT , Inp32(rNFSTAT)|(1<<6)|(1<<7)|(1<<5)|(1<<4)); 	// clear EncodeDone, DecodeDone,IllegalAccess,RnB_TransDetect




	
	//Outp32(rNFECCCONT , (Inp32(rNFECCCONT)&~(1<<25)&~(1<<24)&~(1<<16))); 		
						// 25:EnbMCLEncINT 0   24:EnbMCLDecInt 0 16:EccDirection 0(read)
	//Outp32(rNFECCSTAT , Inp32(rNFECCSTAT)|(1<<25)|(1<<24));  // Clear 25:EncodeDone 24:DecodeDone


 
	
	
	for(i=0; i<sNandInfo.u512BytesPerPage; i++)
	{
		NF_MECC_Lock(); // Main ECC Lock
		NF_CLEAR_RB(); // RnB Clear
		if(!i)
		{			
			NF_CMD(CMD_READ0);	// Read command		
			for(m=0; m<sNandInfo.uColCycle; m++) //Addr Cycle
				NF_ADDR(0);
			for(m=0; m<sNandInfo.uRowCycle; m++)
				NF_ADDR((uCurPageAddr>>(m<<3))&0xFF);	
			if(sNandInfo.uPageSize > 512)
				NF_CMD(CMD_READ0_2CYCLE);	// Read command	
			
#if __POLLING
				NF_DETECT_RB();
				NF_CLEAR_RB();   // RnB Pending Clear
#else
			    while(!NFConDone);
			    NFConDone = 0;				
#endif
		}
		else
		{
			NF_CMD(CMD_RANDOM_DATA_OUT_1ST_CYCLE); // Random Address Access = 512K start
			NF_ADDR(((g_msg_length*i)>>0)&0xFF);
			NF_ADDR(((g_msg_length*i)>>8)&0xFF);
			NF_CMD(CMD_RANDOM_DATA_OUT_2ND_CYCLE);
		}
		
		NF_MECC_UnLock(); // Main ECC Unlock
		NF_RSTECC();    // Initialize ECC
		
		for(m=0;m<g_msg_length;m++)  
			*uRBuf++=NF_RDDATA8();	// Read Main data
			
		// Spare Area Address Setting.
		if(sNandInfo.uPageSize > 512)
		{			
			NF_CMD(CMD_RANDOM_DATA_OUT_1ST_CYCLE);
			NF_ADDR(((sNandInfo.uPageSize + g_ecc_code_length*i)>>0)&0xFF);
			NF_ADDR(((sNandInfo.uPageSize + g_ecc_code_length*i)>>8)&0xFF);
			NF_CMD(CMD_RANDOM_DATA_OUT_2ND_CYCLE);
		}
		
		
		for(m=0; m<g_ecc_code_length ; m++)
			*pSpare++ = NF_RDDATA8(); // Spare Data Read

		NF_MECC_Lock(); // Main ECC Lock

#if		__POLLING
		ECC8_LOOP_UNTIL_DEC_DONE();	// while(!(Inp32(rNFECCSTAT)&(1<<24))); 	// Check decoding done 
		ECC8_DEC_DONE_CLEAR();		//Outp32(rNFECCSTAT, Inp32(rNFECCSTAT) | (1<<24));    // Decoding done Clear
#else
		while(!NFECCDecDone);
		NFECCDecDone=0;
#endif		
		ECC8_LOOP_WHILE_DEC_BUSY(); 	//while( Inp32(rNFECCSTAT) & (u32)(1<<31) ) ; // 8bit ECC Decoding Busy Check.
		uResult[i] = CorrectECC8Data((uRBuf-g_msg_length));
		//UART_Printf("\nuResult[%d] : %d", i, uResult[i]);
		uRet += uResult[i];
	}
#if		__POLLING	
	if(IS_LOCKED())
	{
		ILL_ACC_CLEAR(); 	//Outp32(rNFSTAT, Inp32(rNFSTAT) | 0x20); 
		UART_Printf("\n Accessed locked area!! \n");
		NF_nFCE0_H();  // Chip Select is High
		return FAIL;
	}
#else		
	RNB_INT_DISABLE(); 			//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9)); 
	ILLACC_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10));  // Disable Illegal Access Interrupt
	ECC8DEC_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<12)); 
#endif
	NF_nFCE0_H();  // Chip Select is High
	INTC_Disable(NUM_NFC);
	
	return uRet;
}



u32 ReadNAND_8BitECC(void)
{
	u32 block=0, page=0;
	u32 i;
	u8 * downPt;
	
	downPt=(u8 *)_NONCACHE_STARTADDRESS; 
	for(i=0;i<128;i++)
		g_uSpareData[i]=0xFF;	
	UART_Printf(" NAND(xD-Picture Card) Page Read.\n");

	UART_Printf("Block # to read: ");
	block = UART_GetIntNum();
	UART_Printf("Page # to read: ");
	page = UART_GetIntNum();


	// ----------8/12/16 bit ECC -------- //
	SetECCMsgLen(512);		//g_msg_length=512; ECC8_SET_MSGLEN(g_msg_length);

	ECC8_SETDIR_DEC();

	if(ReadPage8((u8 *)downPt, block, page )) 		
	{
		UART_Printf("Read error.\n");
		return _false;
	}
	else
	{
		UART_Printf("Read OK.\n");	
	}
	
	// Print data.
	UART_Printf("Read data(%d-block,%d-page)\n", block, page);
	UART_Printf("NFECCSTAT: 0x%x\n", Inp32(rNFECCSTAT));
	
	for(i=0; i<sNandInfo.uPageSize; i++) 
	{
		if((i%16)==0) UART_Printf("\n%4x: ", i);
		UART_Printf("%02x ", *(u8 *)downPt++);
	}
	
	UART_Printf("\nSpare:");
	for(i=0; i<sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage; i++) 
	{
		if((i%16)==0) 
			UART_Printf("\n%4x: ", i);
	 	UART_Printf("%02x ", g_uSpareData[i]);
	}
	UART_Printf("\n");
	return _true;

}

/////end of read 8bit ecc////////

u32 TestReadNAND(void)

{

	switch (g_ecc_type)
	{
		case NO_ECC :
		//	ReadNAND_NoECC();
			break;
			
		case ECC_1BIT:
		//	ReadNAND_1BitECC();
			break;
			
		case ECC_4BIT:
			ReadNAND_4BitECC();
			break;
			
		case ECC_8BIT:
			ReadNAND_8BitECC();
			break;
			
		case ECC_12BIT:
		//	ReadNAND_12BitECC();
			break;
			
		case ECC_16BIT:
		//	ReadNAND_16BitECC();
			break;
			
	}
	return;

}

/////////////////////////////////////////end of read ///////////////////////////

/////////////////////////////////////////start of wirte/////////////////////////////////

int WritePage8(u8* pWBuf,u32  uBlock, u32 uPage)
{
	int i=0,m=0,n=0;	
	u8 *pWriteBuffer=pWBuf;	
	u32 u8BitECC[64], uResult[8], uCurPageAddr;
	u8 *p8BitECC = NULL;	
	
	p8BitECC = (u8*)u8BitECC;
	
	//Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<30)|(1<<23)); // System Clock is more than 66Mhz, ECC type is MLC.

	
	NF_CLEAR_RB();
	ILL_ACC_CLEAR();
	ECC8_ENC_DONE_CLEAR();
	
	//ECC8_SET_MSGLEN(g_msg_length);
		

	//	Outp32(rNFCONT, Inp32(rNFCONT) |(1<<18)|(1<<13)|(1<<12)|(1<<11)|(1<<10)|(1<<9)); //ECC for programming.// Enable RnB Interrupt 
	//Outp32(rNFSTAT, Inp32(rNFSTAT) |(1<<6)|(1<<5)|(1<<4)); 
	// Enable Illegal Access Interrupt  // Enable ECC encoding & decoding conpletion interrupt enable  // Enable ECC encoding & decoding conpletion interrupt enable	
        	
	uCurPageAddr =  uBlock*sNandInfo.uPagesPerBlock + uPage;
	
#if		__POLLING

#else
	NFConDone=0;		
	NFECCEncDone=0;

	ECC8ENC_INT_ENABLE();
	ILLACC_INT_ENABLE();
	RNB_INT_ENABLE();
	
	INTC_SetVectAddr(NUM_NFC, IsrNFC);

	INTC_Enable(NUM_NFC);
	
#endif	
	
	
	NF_nFCE0_L(); 			
	NF_CMD(CMD_PAGEPROG);	// Write command		
	for(m=0; m<sNandInfo.uColCycle; m++) //Addr Cycle
		NF_ADDR(0);
	for(m=0; m<sNandInfo.uRowCycle; m++)
		NF_ADDR((uCurPageAddr>>(m<<3))&0xFF);

	
	for(i=0; i<sNandInfo.u512BytesPerPage; i++)
	{		    
	    NF_MECC_UnLock();
	    NF_RSTECC();    
	    	
	    for(m=0;m<512;m++) 
			NF_WRDATA8(*pWriteBuffer++);	// Write one page to NFM from buffer	    
			
	    NF_MECC_Lock();
#if		__POLLING	    
	    ECC8_LOOP_UNTIL_ENC_DONE();	//while(!(Inp32(rNFSTAT)&(1<<7))) ;
	    ECC8_ENC_DONE_CLEAR();		//Outp32(rNFSTAT, Inp32(rNFSTAT) |(1<<7)); 	    	    
#else	    
	    while(!NFECCEncDone);
	    NFECCEncDone=0;
#endif

	       
		u8BitECC[n++] = Inp32(rNFECCPRG0);   		u8BitECC[n++] = Inp32(rNFECCPRG1);
   		u8BitECC[n++] = Inp32(rNFECCPRG2);   		u8BitECC[n++] = Inp32(rNFECCPRG3);   		
	}

	for(i=0, n=0; i<sNandInfo.u512BytesPerPage; i++)//Write Spare Area for 8Bit ECC
	{		
		for(m=0; m<g_ecc_code_length; m++)
		{
			g_uSpareData[m + g_ecc_code_length*i]= p8BitECC[n];			
			NF_WRDATA8(p8BitECC[n++]);
		}
		n+=3;			
	}			

 	NF_CLEAR_RB();
	NF_CMD(CMD_PAGEPROG_2CYCLE);	 // Write 2nd command

#if __POLLING	
	NF_DETECT_RB();
	if(IS_LOCKED()) 
	{	
		ILL_ACC_CLEAR();	//Outp32(rNFSTAT, Inp32(rNFSTAT) |0x20);  
		return FAIL;
	}
#else
while(NFConDone==0);
	NFConDone=0;

	RNB_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9)); 
	ILLACC_INT_DISABLE();	//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10));  // Disable Illegal Access Interrupt
	ECC8ENC_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<12));  // Disable Encoding Done Interrupt
#endif
	
	NF_CMD(CMD_STATUS);   // Read status command       
	for(i=0;i<3;i++);  //twhr=60ns
    
    	if (NF_RDDATA()&0x1) 
    	{// Page write error
       	NF_nFCE0_H();
		UART_Printf("[PROGRAM_ERROR:block#=%d]\n",uBlock);		//NF8_MarkBadBlock(block);
		return FAIL;
    	} 
    	else 
    	{
       	NF_nFCE0_H();
	    	return OK;
	}
}



u32 WriteNAND_8BitECC(void)
{
	u32 block=0, page=0;
	int i, offset;
	u8 * srcPt;
	u32 uRet=0;
	srcPt=(u8 *)(_NONCACHE_STARTADDRESS+0x100000);	
	
//	rGPHCON=rGPHCON&~(0xf<<26)|(0x5<<26);  // GPH13, 14 => OUTPUT
//	rGPHDAT&=~(0x3<<13);		
	for(i=0;i<128;i++)
		g_uSpareData[i]=0xFF;
	UART_Printf("SMC(K9K2G08U) NAND Page Write.\n");	
	
	UART_Printf("You must erase block before you write data!!! \n");	
	UART_Printf("Block # to write: ");
	block = UART_GetIntNum();
	UART_Printf("Page # to write: ");
	page = UART_GetIntNum();
	UART_Printf("offset data(-1:random): ");
	offset = UART_GetIntNum();

#if 0
	srand(0);
#endif

	// Init wdata.
	for(i=0; i<sNandInfo.uPageSize; i++) {
	#if ADS10==TRUE
		if(offset==-1) *srcPt++ = rand()%0xff;
	#else
		if(offset==-1) *srcPt++ = i%0xff;
	#endif
		else *srcPt++ =  i+offset;
	}
    srcPt=(u8 *)(_NONCACHE_STARTADDRESS+0x100000);
	UART_Printf("Write data[%d block, %d page].\n", block, page);

	ECC8_SETDIR_ENC();
	SetECCMsgLen(512);
	
	uRet = WritePage8(srcPt, block, page);
	
	if(uRet==FAIL) 
	{
		UART_Printf("Write Error.\n");
		return _false;
	} else 
	{
		UART_Printf("Write OK.\n");
	}

 	UART_Printf("Write data is");
	for(i=0; i<sNandInfo.uPageSize; i++) {
		if((i%16)==0) UART_Printf("\n%4x: ", i);
		UART_Printf("%02x ", *srcPt++);
	}
	UART_Printf("\n");

	UART_Printf("Spare:");
	for(i=0; i<sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage; i++) {
		if((i%16)==0) UART_Printf("\n%4x: ", i);
	 	UART_Printf("%02x ", g_uSpareData[i]);
	}
	UART_Printf("\n\n");
	return _true;

}

u32 TestWriteNAND(void)

{

	switch (g_ecc_type)
	{
		case NO_ECC :
		//	WriteNAND_NoECC();
			break;
			
		case ECC_1BIT:
		//	WriteNAND_1BitECC();
			break;
			
		case ECC_4BIT:
		//	WriteNAND_4BitECC();
			break;
			
		case ECC_8BIT:
			WriteNAND_8BitECC();
			break;
			
		case ECC_12BIT:
		//	WriteNAND_12BitECC();
			break;
			
		case ECC_16BIT:
		//	WriteNAND_16BitECC();
			break;
			
	}
	return;

}

/////////////////////////////////end of write//////////////////////////////////

////////////////////////////////// start of ECC test //////////////////////////////////

//////start of read 8bit ecc///
u32 Compare_Error(void)
{	
	unsigned int i,k,pos,total_error,uErrorByte[9],uValidErrorByte[8];//u32 uErrorByte1, uErrorByte2, uErrorByte3, uErrorByte4, uErrorByte5, uErrorByte6, uErrorByte7, uErrorByte8;	
	unsigned char uErrorBit[9];//u32 uErrorBit1, uErrorBit2, uErrorBit3, uErrorBit4, uErrorBit5, uErrorBit6, uErrorBit7, uErrorBit8;
	unsigned int uErrorType;


	// total error: total ecc error     uValidError : which is valid among uErrorByte[n]s.
	total_error=0;
	for(i=0;i<g_ecc_type;i++)
		if( ((Inp32(rNFECCSECSTAT)>>(8+i))&1) == 1)uValidErrorByte[total_error++]=i;

	
	//Error Type // total error bit number or uncorrectable error.
	uErrorType = (Inp32(rNFECCSECSTAT)>>0)&0x1f;
	
	//Error Byte //
	uErrorByte[1] = Inp32(rNFECCERL0)&0x3ff;
	uErrorByte[2] = (Inp32(rNFECCERL0)>>16)&0x3ff;
	
	uErrorByte[3] = (Inp32(rNFECCERL1))&0x3ff;
	uErrorByte[4] = (Inp32(rNFECCERL1)>>16)&0x3ff;	
	
	uErrorByte[5] = (Inp32(rNFECCERL2))&0x3ff;
	uErrorByte[6] = (Inp32(rNFECCERL2)>>16)&0x3ff;
	
	uErrorByte[7] = (Inp32(rNFECCERL3))&0x3ff;
	uErrorByte[8] = (Inp32(rNFECCERL3)>>16)&0x3ff;
	
	// Error Bit //

	uErrorBit[1] = Inp32(rNFECCERP0) &0xff;
	uErrorBit[2] = (Inp32(rNFECCERP0)>>8)&0xff;
	uErrorBit[3] = (Inp32(rNFECCERP0)>>16)&0xff;
	uErrorBit[4] = (Inp32(rNFECCERP0)>>24)&0xff;
	
	uErrorBit[5] = Inp32(rNFECCERP1)&0xff;
	uErrorBit[6] = (Inp32(rNFECCERP1)>>8)&0xff;
	uErrorBit[7] = (Inp32(rNFECCERP1)>>16)&0xff;
	uErrorBit[8] = (Inp32(rNFECCERP1)>>24)&0xff;

	
	

	if(!(uErrorType == total_error))     //total error should be the same with error type number.
		{
			UART_Printf("\n%d th error is ECC controller operation error!\n",g_ecc_err_cnt);
			return FAIL;
		}
	//if(uErrorType > g_ecc_type) return 1;  //check if error count is less than correctable max error count. 'return 1' means uncorrectable error.


	g_checkedErrorType = uErrorType;

	pos = 0;
	for(i=1;i<uErrorType ;i++)	// uErrorType == total_error 
	{
		k= uValidErrorByte[i];
		
		// check if errorbyte is equal for g_errortype and check for another uErrorbyte[k]
		g_checkedErrorByte[pos] = uErrorByte[k];
		g_checkedErrorBit[pos] = uErrorBit[k];
		pos++;
	}

	if(uErrorType != g_madeErrorType) return FAIL  ; 
	
	for(i=1;i<uErrorType;i++) 				// compare errorbyte by increasing byte order ,not considering overlapping errorbyte.
	{
		if(g_madeErrorByte[i] != g_checkedErrorByte[i])  return FAIL;
		if(g_madeErrorBit[i] != g_checkedErrorBit[i]) return FAIL;
	}
	
	
	return OK;
	
}



u32 ReadPage8_ecctest(u8* uRBuf,u32  uBlock, u32 uPage)
{
	int i,m;		
	u8 Addr_Cycle, option2;
	//u32 uResult[8]={0,	};
	u8 *pSpare;
	u32 uCurPageAddr, uRet=0;
	pSpare = g_uSpareData;
	
	uCurPageAddr =  uBlock*sNandInfo.uPagesPerBlock + uPage;	 	


	
		//rNFCONF &= ~(0x3<<23)|(1<<23);
	//rNFCONT = ( rNFCONT&~(0x1<<18) ) |(0<<18)|(1<<11)|(1<<10)|(1<<9); // Init NFCONT

	//Outp32(rNFSTAT, Inp32(rNFSTAT) | (1<<6)|(1<<7)|(1<<5)|(1<<4)); 		
	ILL_ACC_CLEAR();
	
	NF_MECC_Lock(); // Main ECC Lock
	NF_nFCE0_L(); // Chip Select Low
	NF_CLEAR_RB(); // RnB Clear
	
	//ECC8DEC_INT_DISABLE();
	//ECC8ENC_INT_DISABLE();
	ECC8_DEC_DONE_CLEAR();
	//ECC8_ENC_DONE_CLEAR();


	
	
#if		__POLLING
#else	
	ECC8DEC_INT_ENABLE();
	ILLACC_INT_ENABLE();
	RNB_INT_ENABLE();
	
	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<12));  
	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<11)); 
	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<10)); 
	//Outp32(rNFCONT, Inp32(rNFCONT) | (1<<9)); 
	
	NFConDone=0;
	NFECCDecDone=0;
	NFECCEncDone=0;

	INTC_SetVectAddr(NUM_NFC, IsrNFC);

	INTC_Enable(NUM_NFC);
 
#endif
	
	// Outp32(rNFCONF, ( Inp32(rNFCONF)&~(0x3<<23)&~(0xf<<12)&~(0xf<<8)&~(0xf<<4) ) |(3<<23)|(0x7<<12)|(0x7<<8)|(0x7<<4));   
                                          // 30:reserved 23:unselect ecc0(1/4) TACLS,TWRPH0,1

//	rNFCONT = ( rNFCONT&~(0x1<<18) ) |(0<<18)|(0<<11)|(0<<10)|(0<<9)|(1<<6)|(1<<0); // Init NFCONT
	

	//Outp32(rNFCONT, ( Inp32(rNFCONT)&~(0x1<<18) ) |(0<<18)|(1<<6)|(1<<0)); 
						 // 18:4bit ECC direction 6:SECC lock 0  0:MODE 1-Enable NAND controller
						 
	//Outp32(rNFCONT , (Inp32(rNFCONT)&~(0x7<<9) )) ;		// 11:MLCStop 0 10: EnbleillegalAccINT 0 9:EnbRnBINT 0
	
	//Outp32(rNFSTAT , Inp32(rNFSTAT)|(1<<6)|(1<<7)|(1<<5)|(1<<4)); 	// clear EncodeDone, DecodeDone,IllegalAccess,RnB_TransDetect




	
	//Outp32(rNFECCCONT , (Inp32(rNFECCCONT)&~(1<<25)&~(1<<24)&~(1<<16))); 		
						// 25:EnbMCLEncINT 0   24:EnbMCLDecInt 0 16:EccDirection 0(read)
	//Outp32(rNFECCSTAT , Inp32(rNFECCSTAT)|(1<<25)|(1<<24));  // Clear 25:EncodeDone 24:DecodeDone


 
	
	
//	for(i=0; i<sNandInfo.u512BytesPerPage; i++)
//	{
		NF_MECC_Lock(); // Main ECC Lock
		NF_CLEAR_RB(); // RnB Clear
	//	if(!i)
	//	{			
			NF_CMD(CMD_READ0);	// Read command		
			for(m=0; m<sNandInfo.uColCycle; m++) //Addr Cycle
				NF_ADDR(0);
			for(m=0; m<sNandInfo.uRowCycle; m++)
				NF_ADDR((uCurPageAddr>>(m<<3))&0xFF);	
			if(sNandInfo.uPageSize > 512)
				NF_CMD(CMD_READ0_2CYCLE);	// Read command	
			
#if __POLLING
				NF_DETECT_RB();
				NF_CLEAR_RB();   // RnB Pending Clear
#else
			    while(!NFConDone);
			    NFConDone = 0;				
#endif
	//	}
	//	else
	//	{
	//		NF_CMD(CMD_RANDOM_DATA_OUT_1ST_CYCLE); // Random Address Access = 512K start
	//		NF_ADDR(((g_msg_length*i)>>0)&0xFF);
	//		NF_ADDR(((g_msg_length*i)>>8)&0xFF);
	//		NF_CMD(CMD_RANDOM_DATA_OUT_2ND_CYCLE)
	//	}
		
		NF_MECC_UnLock(); // Main ECC Unlock
		NF_RSTECC();    // Initialize ECC
		
		for(m=0;m<g_msg_length;m++)  
			*uRBuf++=NF_RDDATA8();	// Read Main data
			
		for(m=0; m<g_ecc_code_length ; m++)
			*pSpare++ = NF_RDDATA8(); // Spare Data Read

		NF_MECC_Lock(); // Main ECC Lock

#if		__POLLING
		ECC8_LOOP_UNTIL_DEC_DONE();	// while(!(Inp32(rNFECCSTAT)&(1<<24))); 	// Check decoding done 
		ECC8_DEC_DONE_CLEAR();		//Outp32(rNFECCSTAT, Inp32(rNFECCSTAT) | (1<<24));    // Decoding done Clear
#else
		while(!NFECCDecDone);
		NFECCDecDone=0;
#endif		
		ECC8_LOOP_WHILE_DEC_BUSY(); 	//while( Inp32(rNFECCSTAT) & (u32)(1<<31) ) ; // 8bit ECC Decoding Busy Check.

		if(Compare_Error()==FAIL)
		{

			g_err_list[g_ecc_err_cnt].startno = g_startno;
			for(i=0;i<2;i++)	g_err_list[g_ecc_err_cnt].rand_coeff[i] = g_rand_coeff[i];
			g_err_list[g_ecc_err_cnt].madeErrorType = g_madeErrorType;
			g_err_list[g_ecc_err_cnt].checkedErrorType = g_checkedErrorType;
			for(i=0;i<16;i++)	g_err_list[g_ecc_err_cnt].madeErrorByte[i] = g_madeErrorByte[i];
			for(i=0;i<16;i++)	g_err_list[g_ecc_err_cnt].checkedErrorByte[i] = g_checkedErrorByte[i];
			for(i=0;i<16;i++)	g_err_list[g_ecc_err_cnt].madeErrorBit[i] = g_madeErrorBit[i];
			for(i=0;i<16;i++)	g_err_list[g_ecc_err_cnt].checkedErrorBit[i] = g_checkedErrorBit[i];
			g_ecc_err_cnt++;
		
		}
		
		//UART_Printf("\nuResult[%d] : %d", i, uResult[i]);
		
//	}
#if		__POLLING	
	if(IS_LOCKED())
	{
		ILL_ACC_CLEAR(); 	//Outp32(rNFSTAT, Inp32(rNFSTAT) | 0x20); 
		UART_Printf("\nError to erase a block");
		NF_nFCE0_H();  // Chip Select is High
		return FAIL;
	}
#else		
	RNB_INT_DISABLE(); 			//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9)); 
	ILLACC_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10));  // Disable Illegal Access Interrupt
	ECC8DEC_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<12)); 
#endif
	NF_nFCE0_H();  // Chip Select is High
	INTC_Disable(NUM_NFC);
	
	return OK;
}







int WritePage8forECCtest(u8* pWBuf,u32  uBlock, u32 uPage)
{
	int i=0,m=0,n=0;	
	u8 *pWriteBuffer=pWBuf;	
	u32 u8BitECC[64], uResult[8], uCurPageAddr;
	u8 *p8BitECC = NULL;	
	
	p8BitECC = (u8*)u8BitECC;
	
	//Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<30)|(1<<23)); // System Clock is more than 66Mhz, ECC type is MLC.

	
	NF_CLEAR_RB();
	ILL_ACC_CLEAR();
	ECC8_ENC_DONE_CLEAR();
	
	//ECC8_SET_MSGLEN(g_msg_length);
		

	//	Outp32(rNFCONT, Inp32(rNFCONT) |(1<<18)|(1<<13)|(1<<12)|(1<<11)|(1<<10)|(1<<9)); //ECC for programming.// Enable RnB Interrupt 
	//Outp32(rNFSTAT, Inp32(rNFSTAT) |(1<<6)|(1<<5)|(1<<4)); 
	// Enable Illegal Access Interrupt  // Enable ECC encoding & decoding conpletion interrupt enable  // Enable ECC encoding & decoding conpletion interrupt enable	
        	
	uCurPageAddr =  uBlock*sNandInfo.uPagesPerBlock + uPage;
	
#if		__POLLING

#else
	NFConDone=0;		
	NFECCEncDone=0;

	ECC8ENC_INT_ENABLE();
	ILLACC_INT_ENABLE();
	RNB_INT_ENABLE();
	
	INTC_SetVectAddr(NUM_NFC, IsrNFC);

	INTC_Enable(NUM_NFC);
	
#endif	
	
	
	NF_nFCE0_L(); 			
	NF_CMD(CMD_PAGEPROG);	// Write command		
	for(m=0; m<sNandInfo.uColCycle; m++) //Addr Cycle
		NF_ADDR(0);
	for(m=0; m<sNandInfo.uRowCycle; m++)
		NF_ADDR((uCurPageAddr>>(m<<3))&0xFF);

	
	//for(i=0; i<sNandInfo.u512BytesPerPage; i++)
	//{		    
	    NF_MECC_UnLock();
	    NF_RSTECC();    
	    	
	    for(m=0;m<512;m++) 
			NF_WRDATA8(*pWriteBuffer++);	// Write one page to NFM from buffer	    
			
	    NF_MECC_Lock();
#if		__POLLING	    
	    ECC8_LOOP_UNTIL_ENC_DONE();	//while(!(Inp32(rNFSTAT)&(1<<7))) ;
	    ECC8_ENC_DONE_CLEAR();		//Outp32(rNFSTAT, Inp32(rNFSTAT) |(1<<7)); 	    	    
#else	    
	    while(!NFECCEncDone);
	    NFECCEncDone=0;
#endif

	       
		u8BitECC[n++] = Inp32(rNFECCPRG0);   		u8BitECC[n++] = Inp32(rNFECCPRG1);
   		u8BitECC[n++] = Inp32(rNFECCPRG2);   		u8BitECC[n++] = Inp32(rNFECCPRG3);   		
	//}

	n=0;
//	for(i=0, n=0; i<sNandInfo.u512BytesPerPage; i++)//Write Spare Area for 8Bit ECC
//	{		
		for(m=0; m<g_ecc_code_length; m++)
		{
			g_uSpareData[m]= p8BitECC[n];
		
			NF_WRDATA8(p8BitECC[n++]);
		}
//		n+=3;			
//	}			

 	NF_CLEAR_RB();
	NF_CMD(CMD_PAGEPROG_2CYCLE);	 // Write 2nd command

#if __POLLING	
	NF_DETECT_RB();
	if(IS_LOCKED()) 
	{	
		ILL_ACC_CLEAR();	//Outp32(rNFSTAT, Inp32(rNFSTAT) |0x20);  
		return FAIL;
	}
#else
while(NFConDone==0);
	NFConDone=0;

	RNB_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9)); 
	ILLACC_INT_DISABLE();	//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10));  // Disable Illegal Access Interrupt
	ECC8ENC_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<12));  // Disable Encoding Done Interrupt
#endif
	
	NF_CMD(CMD_STATUS);   // Read status command       
	for(i=0;i<3;i++);  //twhr=60ns
    
    	if (NF_RDDATA()&0x1) 
    	{// Page write error
       	NF_nFCE0_H();
		UART_Printf("[PROGRAM_ERROR:block#=%d]\n",uBlock);		//NF8_MarkBadBlock(block);
		return FAIL;
    	} 
    	else 
    	{
       	NF_nFCE0_H();
	    	return OK;
	}
}


int WritePage8change(u8* pWBuf,u32  uBlock, u32 uPage)
{

	
	
	
	int i=0,m=0,n=0;	
	u8 *pWriteBuffer=pWBuf;	
	//u32 u8BitECC[64], uResult[8], 
	u32 uCurPageAddr;
	//u8 *p8BitECC = NULL;	
	
	//p8BitECC = (u8*)u8BitECC;
	
	//Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<30)|(1<<23)); // System Clock is more than 66Mhz, ECC type is MLC.

	
	NF_CLEAR_RB();
	ILL_ACC_CLEAR();
	//ECC8_ENC_DONE_CLEAR();
	
	//ECC8_SET_MSGLEN(g_msg_length);
		

	//	Outp32(rNFCONT, Inp32(rNFCONT) |(1<<18)|(1<<13)|(1<<12)|(1<<11)|(1<<10)|(1<<9)); //ECC for programming.// Enable RnB Interrupt 
	//Outp32(rNFSTAT, Inp32(rNFSTAT) |(1<<6)|(1<<5)|(1<<4)); 
	// Enable Illegal Access Interrupt  // Enable ECC encoding & decoding conpletion interrupt enable  // Enable ECC encoding & decoding conpletion interrupt enable	
        	
	uCurPageAddr =  uBlock*sNandInfo.uPagesPerBlock + uPage;
	
#if		__POLLING

#else
	NFConDone=0;		
	//NFECCEncDone=0;

	//ECC8ENC_INT_ENABLE();
	ILLACC_INT_ENABLE();
	RNB_INT_ENABLE();
	
	INTC_SetVectAddr(NUM_NFC, IsrNFC);

	INTC_Enable(NUM_NFC);
	
#endif	
	
	
	NF_nFCE0_L(); 			
	NF_CMD(CMD_PAGEPROG);	// Write command		
	for(m=0; m<sNandInfo.uColCycle; m++) //Addr Cycle
		NF_ADDR(0);
	for(m=0; m<sNandInfo.uRowCycle; m++)
		NF_ADDR((uCurPageAddr>>(m<<3))&0xFF);

	
	//for(i=0; i<sNandInfo.u512BytesPerPage; i++)
	//{		    
	//    NF_MECC_UnLock();
	//    NF_RSTECC();    
	    	
	    for(m=0;m<512;m++) 
			NF_WRDATA8(*pWriteBuffer++);	// Write one page to NFM from buffer	    
			
	//    NF_MECC_Lock();
#if		__POLLING	    
	//    ECC8_LOOP_UNTIL_ENC_DONE();	//while(!(Inp32(rNFSTAT)&(1<<7))) ;
	//    ECC8_ENC_DONE_CLEAR();		//Outp32(rNFSTAT, Inp32(rNFSTAT) |(1<<7)); 	    	    
#else	    
	//    while(!NFECCEncDone);
	//    NFECCEncDone=0;
#endif

	       
	//	u8BitECC[n++] = rNFECCPRG0;   		u8BitECC[n++] = rNFECCPRG1;
   	//	u8BitECC[n++] = rNFECCPRG2;   		u8BitECC[n++] = rNFECCPRG3;   		
	//}

	n=0;
//	for(i=0, n=0; i<sNandInfo.u512BytesPerPage; i++)//Write Spare Area for 8Bit ECC
//	{		
		for(m=0; m<g_ecc_code_length; m++)
		{
			NF_WRDATA8(g_uSpareData[m]);
		}
//		n+=3;			
//	}			

 	NF_CLEAR_RB();
	NF_CMD(CMD_PAGEPROG_2CYCLE);	 // Write 2nd command

#if __POLLING	
	NF_DETECT_RB();
	if(IS_LOCKED()) 
	{	
		ILL_ACC_CLEAR();	//Outp32(rNFSTAT, Inp32(rNFSTAT) |0x20);  
		return FAIL;
	}
#else
while(NFConDone==0);
	NFConDone=0;

	RNB_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9)); 
	ILLACC_INT_DISABLE();	//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10));  // Disable Illegal Access Interrupt
	ECC8ENC_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<12));  // Disable Encoding Done Interrupt
#endif
	
	NF_CMD(CMD_STATUS);   // Read status command       
	for(i=0;i<3;i++);  //twhr=60ns
    
    	if (NF_RDDATA()&0x1) 
    	{// Page write error
       	NF_nFCE0_H();
		UART_Printf("[PROGRAM_ERROR:block#=%d]\n",uBlock);		//NF8_MarkBadBlock(block);
		return FAIL;
    	} 
    	else 
    	{
       	NF_nFCE0_H();
	    	return OK;
	}
}


/////////////
u32 TestECC_8bit(void)
{
	u32 block=0, page=0;
	int i, k,m,offset;
	u8 * srcPt;
	u32 wdata;
	u32 uRet=0;
	u8 ecc_code[13] ;
	srcPt=(u8 *)0x31100000;	

	

	g_ecc_err_cnt = 0;
	
//	rGPHCON=rGPHCON&~(0xf<<26)|(0x5<<26);  // GPH13, 14 => OUTPUT
//	rGPHDAT&=~(0x3<<13);		
	for(i=0;i<128;i++)
		g_uSpareData[i]=0xFF;
	UART_Printf("SMC(K9K2G08U) NAND Page Write.\n");	
	
	UART_Printf("You must erase block before you write data!!! \n");	
	UART_Printf("Block # to write: ");
	block = UART_GetIntNum();
	UART_Printf("Page # to write: ");
	page = UART_GetIntNum();
	UART_Printf("offset data(-1:random): ");
	offset = UART_GetIntNum();

	srand(offset);

	// Init wdata.
	
	g_startno = rand()%0xff;
	g_rand_coeff[0]= rand()%0xff;
	g_rand_coeff[1]= rand()%0xf;

	wdata = g_startno;
	for(i=0; i<512; i++)
	{
		*srcPt++ = (wdata)%0xff;
		wdata = wdata+g_rand_coeff[i%2]*i;
	}
	
    	srcPt=(u8 *)0x31100000;
		
	//UART_Printf("Write data[%d block, %d page].\n", block, page);


	// erase 1 block
	EraseBlock(block);


	
	
	// write with ECC
	ECC8_SETDIR_ENC();
	SetECCMsgLen(512);
	
	uRet = WritePage8forECCtest(srcPt, block, page);
	
	if(uRet==FAIL) 
	{
		UART_Printf("Write Error.\n");
		return _false;
	} else 
	{
		UART_Printf("Write OK.\n");
	}
	/////////////////

	
	// write changed data page + 1
	k=0;
	g_madeErrorType= rand()%0x8;
	for(i=0;i<g_madeErrorType;i++)
	{
		k = k+rand()%(512-8-k);		// make error by increasing byte order, not considering overlapping errorbyte.
		g_madeErrorByte[i]=k;
		m=(0x1<<(rand()%8));
		g_madeErrorBit[i]=m;
		
		*(srcPt+k)^=m;
	}

	uRet = WritePage8change(srcPt, block, page+1);
	
	if(uRet==FAIL) 
	{
		UART_Printf("Write Error.\n");
		return _false;
	} else 
	{
		UART_Printf("Write OK.\n");
	}
	////////////////

	// read data with ecc and compare error
	uRet = ReadPage8_ecctest(srcPt,block,page+1);
	if(uRet==FAIL) 
	{
		UART_Printf("Read Error.\n");
		return _false;
	} else 
	{
		UART_Printf("Read OK.\n");
	}


	////////////////

 	UART_Printf("Write data is");
	for(i=0; i<sNandInfo.uPageSize; i++) {
		if((i%16)==0) UART_Printf("\n%4x: ", i);
		UART_Printf("%02x ", *srcPt++);
	}
	UART_Printf("\n");

	UART_Printf("Spare:");
	for(i=0; i<sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage; i++) {
		if((i%16)==0) UART_Printf("\n%4x: ", i);
	 	UART_Printf("%02x ", g_uSpareData[i]);
	}
	UART_Printf("\n\n");
	return _true;

}




///////////////////////////////// end of ECC test ///////////////////////////////////

u32 SetLock(u32 uStartBlock, u32 uEndBlock, u32 uIsSoftLock)
{
	Outp32(rNFSBLK, (uStartBlock<<sNandInfo.uBlockShift));
	Outp32(rNFEBLK, (uEndBlock<<sNandInfo.uBlockShift));
	if(uIsSoftLock)
	{
		Outp32(rNFCONT, Inp32(rNFCONT) |(1<<16));  
        UART_Printf("Software Locked\n ");
	}
	else
	{
		Outp32(rNFCONT, Inp32(rNFCONT) |(1<<17)); 
        UART_Printf("Lock-tight: To clear Lock-tight, reset S3C2450!!!\n ");
	}
}

void Test_Lock(void)
{
    u32 uOption;
    u32 uSBlock, uEBlock;
	UART_Printf("(x-D Card) NAND Lock Test !!!\n");
	UART_Printf("Select Lock type, Lock-tight(0)/Softlock(1) : ");

	uOption=UART_GetIntNum();
			
	UART_Printf("\nEnter programmable start block address ");
	uSBlock = UART_GetIntNum();
	UART_Printf("Enter programmable end block address ");
    uEBlock = UART_GetIntNum();

	SetLock(uSBlock, uEBlock, uOption);		
    UART_Printf("%d block ~ %d block are Programmable\n ", uSBlock, (uEBlock));
}







#if 0
u32 uTargetBlock;
u32 uTargetSize;
u32 uSrcAddr;

void GetTargetInfo(void)
{
	u32 no_block, no_page, no_byte;
	
	UART_Printf("\nAvailable target block number: 0~2047\n");
	UART_Printf("Input target block number:");
    uTargetBlock=UART_GetIntNum();	// Block number(0~4095)
    //if(uTargetSize==0)
    {
 #if 0
    	UART_Printf("Input target size(0x4000*n):");
    	uTargetSize=UART_GetIntNum();	// Total byte size
#else
 	  	UART_Printf("Input program file size(bytes): ");
    	uTargetSize=UART_GetIntNum();	// Total byte size
    	#endif
    }
	
	no_block = (u32)((uTargetSize/sNandInfo.uPageSize)/sNandInfo.uPagesPerBlock);
	no_page = (u32)((uTargetSize/sNandInfo.uPageSize)%sNandInfo.uPagesPerBlock);
	no_byte = (u32)(uTargetSize%sNandInfo.uPageSize);
	UART_Printf("File:%d[%d-block,%d-page,%d-bytes].\n", uTargetSize, no_block, no_page, no_byte);
}

#endif





void WriteDramImage2Nand(void)
{

//    unsigned long interrupt_reservoir;
	int i;
	int programError=0;
	u8 *srcPt,*saveSrcPt;
	u32 blockIndex;

	static volatile u32 srcAddress;
	static u32 targetBlock;	    // Block number (0 ~ 4095)
	u32 targetSize;	    // Total byte size 

	u32 j;
	u32 uSrcAddr = 0x21000000;//CODEC_MEM_ST+0x100000;
	u8 buffer;
	u32 checkSum=0;
	u32 dataLength = (8*1024)-4;


	UART_Printf("\n[ NAND Flash writing program ]\n");
	UART_Printf("The program buffer: 0x21000000~0x21ffffff(16MB)\n");

//	NF8_Init();

//   rINTMSK = BIT_ALLMSK; 	
    	srcAddress=(0x21000000); 

	for(i=0;i<dataLength;i+=1)
	{
		buffer = Inp8(uSrcAddr+i);
		checkSum = checkSum + buffer;	// Check Sum.
	}
	Outp32(uSrcAddr+i, checkSum);
	Disp(" uSrcAddr : %x, checkSum :%x ",uSrcAddr+i, checkSum);


    	UART_Printf("\nAvailable target block number: 0~2047\n");
	UART_Printf("Input target block number:");
    	targetBlock=UART_GetIntNum();	// Block number(0~4095)

	#if 0
	UART_Printf("Input target size(0x4000*n):");
	targetSize=UART_GetIntNum();	// Total byte size
	#else
	UART_Printf("Input program file size(bytes): ");
	targetSize=UART_GetIntNum();	// Total byte size
	#endif

	
	UART_Printf("File:%d[%d-block,%d-page,%d-bytes].\n", targetSize,(u32)((targetSize/2048)/64), (u32)((targetSize/2048)%64), (u32)(targetSize%2048));
	
    srcPt=(u8 *)srcAddress;
    blockIndex=targetBlock;
    
    ECC8_SETDIR_ENC();
    SetECCMsgLen(512);


	                          	

	

     while(1) {
        saveSrcPt=srcPt;	

#if 0
		if(NF8_IsBadBlock(blockIndex)==FAIL) {
		    blockIndex++;   // for next block
		    continue;
		}
#endif

		if(EraseBlock(blockIndex)==FAIL) {
		    blockIndex++;   // for next block
		    continue;
		}
		UART_Printf(".");		//	if(i==0)  UART_Printf(".");
		// After 1-Block erase, Write 1-Block(32 pages).
		for(i=0;i<sNandInfo.uPagesPerBlock;i++) {
		
		    if(WritePage8(srcPt,blockIndex,i)==FAIL) {// block num, page num, buffer
		        programError=1;
		        break;
			}

#if 0
		    if(ReadPage(srcPt, blockIndex,i)) {
				UART_Printf("ECC Error(block=%d,page=%d!!!\n",blockIndex,i);
		    }
#endif
			
		srcPt+=sNandInfo.uPageSize;	// Increase buffer addr one pase size
		
			//UART_Printf("\b\b\b\b\b\b\b\b%04d,%02d]", blockIndex, i);
		if((u32)srcPt>=(srcAddress+targetSize)) // Check end of buffer
		break;	// Exit for loop
		}
		
		if(programError==1) {
		    blockIndex++;
		    srcPt=saveSrcPt;
		    programError=0;
		    continue;
		}

		if((u32)srcPt>=(srcAddress+targetSize)) break;	// Exit while loop

		blockIndex++;
		
    }
}

//////////// start of interleaving operation ////////////////


int WritePage8_interleave(u8* pWBuf,u32  uBlock, u32 uPage)
{
	int i=0,m=0,n=0;	
	u8 *pWriteBuffer=pWBuf;	
	u32 u8BitECC[64], uResult[8], uCurPageAddr;
	u8 *p8BitECC = NULL;	
	
	p8BitECC = (u8*)u8BitECC;
	
	//Outp32(rNFCONF, Inp32(rNFCONF) & ~(1<<30)|(1<<23)); // System Clock is more than 66Mhz, ECC type is MLC.

	
	NF_CLEAR_RB();
	ILL_ACC_CLEAR();
	ECC8_ENC_DONE_CLEAR();
	
	//ECC8_SET_MSGLEN(g_msg_length);
		

	//	Outp32(rNFCONT, Inp32(rNFCONT) |(1<<18)|(1<<13)|(1<<12)|(1<<11)|(1<<10)|(1<<9)); //ECC for programming.// Enable RnB Interrupt 
	//Outp32(rNFSTAT, Inp32(rNFSTAT) |(1<<6)|(1<<5)|(1<<4)); 
	// Enable Illegal Access Interrupt  // Enable ECC encoding & decoding conpletion interrupt enable  // Enable ECC encoding & decoding conpletion interrupt enable	
        	
	uCurPageAddr =  uBlock*sNandInfo.uPagesPerBlock + uPage;
	
#if		__POLLING

#else
	NFConDone=0;		
	NFECCEncDone=0;

	ECC8ENC_INT_ENABLE();
	ILLACC_INT_ENABLE();
	RNB_INT_ENABLE();
	
	INTC_SetVectAddr(NUM_NFC, IsrNFC);

	INTC_Enable(NUM_NFC);
	
#endif	
	
	
	NF_nFCE0_L(); 			
	NF_CMD(CMD_PAGEPROG);	// Write command		
	for(m=0; m<sNandInfo.uColCycle; m++) //Addr Cycle
		NF_ADDR(0);
	for(m=0; m<sNandInfo.uRowCycle; m++)
		NF_ADDR((uCurPageAddr>>(m<<3))&0xFF);

	
	for(i=0; i<sNandInfo.u512BytesPerPage; i++)
	{		    
	    NF_MECC_UnLock();
	    NF_RSTECC();    
	    	
	    for(m=0;m<512;m++) 
			NF_WRDATA8(*pWriteBuffer++);	// Write one page to NFM from buffer	    
			
	    NF_MECC_Lock();
		
#if		__POLLING	    
	    ECC8_LOOP_UNTIL_ENC_DONE();	//while(!(Inp32(rNFSTAT)&(1<<7))) ;
	    ECC8_ENC_DONE_CLEAR();		//Outp32(rNFSTAT, Inp32(rNFSTAT) |(1<<7)); 	    	    
#else	    
	    while(!NFECCEncDone);
	    NFECCEncDone=0;
#endif

	       
		u8BitECC[n++] = rNFECCPRG0;   		u8BitECC[n++] = rNFECCPRG1;
   		u8BitECC[n++] = rNFECCPRG2;   		u8BitECC[n++] = rNFECCPRG3;   		
	}

	for(i=0, n=0; i<sNandInfo.u512BytesPerPage; i++)//Write Spare Area for 8Bit ECC
	{		
		for(m=0; m<g_ecc_code_length; m++)
		{
			g_uSpareData[m + g_ecc_code_length*i]= p8BitECC[n];			
			NF_WRDATA8(p8BitECC[n++]);
		}
		n+=3;			
	}			

 	NF_CLEAR_RB0();
	NF_CMD(CMD_PAGEPROG_2CYCLE);	 // Write 2nd command
	NF_nFCE0_H();
//////////////////

	NF_nFCE1_L(); 			
	NF_CMD(CMD_PAGEPROG);	// Write command		
	for(m=0; m<sNandInfo.uColCycle; m++) //Addr Cycle
		NF_ADDR(0);
	for(m=0; m<sNandInfo.uRowCycle; m++)
		NF_ADDR((uCurPageAddr>>(m<<3))&0xFF);

	
	for(i=0; i<sNandInfo.u512BytesPerPage; i++)
	{		    
	    NF_MECC_UnLock();
	    NF_RSTECC();    
	    	
	    for(m=0;m<512;m++) 
			NF_WRDATA8(*pWriteBuffer++);	// Write one page to NFM from buffer	    
			
	    NF_MECC_Lock();
		
#if		__POLLING	    
	    ECC8_LOOP_UNTIL_ENC_DONE();	//while(!(Inp32(rNFSTAT)&(1<<7))) ;
	    ECC8_ENC_DONE_CLEAR();		//Outp32(rNFSTAT, Inp32(rNFSTAT) |(1<<7)); 	    	    
#else	    
	    while(!NFECCEncDone);
	    NFECCEncDone=0;
#endif

	       
		u8BitECC[n++] = rNFECCPRG0;   		u8BitECC[n++] = rNFECCPRG1;
   		u8BitECC[n++] = rNFECCPRG2;   		u8BitECC[n++] = rNFECCPRG3;   		
	}

	for(i=0, n=0; i<sNandInfo.u512BytesPerPage; i++)//Write Spare Area for 8Bit ECC
	{		
		for(m=0; m<g_ecc_code_length; m++)
		{
			g_uSpareData[m + g_ecc_code_length*i]= p8BitECC[n];			
			NF_WRDATA8(p8BitECC[n++]);
		}
		n+=3;			
	}			

 	NF_CLEAR_RB();
	NF_CMD(CMD_PAGEPROG_2CYCLE);	 // Write 2nd command
	NF_nFCE1_H();







////////////
#if __POLLING	
	NF_DETECT_RB();
	if(IS_LOCKED()) 
	{	
		ILL_ACC_CLEAR();	//Outp32(rNFSTAT, Inp32(rNFSTAT) |0x20);  
		return FAIL;
	}
#else
while(NFConDone==0);
	NFConDone=0;

	RNB_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<9)); 
	ILLACC_INT_DISABLE();	//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<10));  // Disable Illegal Access Interrupt
	ECC8ENC_INT_DISABLE();		//Outp32(rNFCONT, Inp32(rNFCONT) & ~(1<<12));  // Disable Encoding Done Interrupt
#endif
	
	NF_CMD(CMD_STATUS);   // Read status command       
	for(i=0;i<3;i++);  //twhr=60ns
    
    	if (NF_RDDATA()&0x1) 
    	{// Page write error
       	NF_nFCE0_H();
		UART_Printf("[PROGRAM_ERROR:block#=%d]\n",uBlock);		//NF8_MarkBadBlock(block);
		return FAIL;
    	} 
    	else 
    	{
       	NF_nFCE0_H();
	    	return OK;
	}
}



u32 WriteNAND_8BitECC_interleave(void)
{
	u32 block=0, page=0;
	int i, offset;
	u8 * srcPt;
	u32 uRet=0;
	srcPt=(u8 *)(_NONCACHE_STARTADDRESS+0x100000);	
	
//	rGPHCON=rGPHCON&~(0xf<<26)|(0x5<<26);  // GPH13, 14 => OUTPUT
//	rGPHDAT&=~(0x3<<13);		
	for(i=0;i<128;i++)
		g_uSpareData[i]=0xFF;
	UART_Printf("SMC(K9K2G08U) NAND Page Write.\n");	
	
	UART_Printf("You must erase block before you write data!!! \n");	
	UART_Printf("Block # to write: ");
	block = UART_GetIntNum();
	UART_Printf("Page # to write: ");
	page = UART_GetIntNum();
	UART_Printf("offset data(-1:random): ");
	offset = UART_GetIntNum();

#if 0
	srand(0);
#endif

	// Init wdata.
	for(i=0; i<2*sNandInfo.uPageSize; i++) {
	#if ADS10==TRUE
		if(offset==-1) *srcPt++ = rand()%0xff;
	#else
		if(offset==-1) *srcPt++ = i%0xff;
	#endif
		else *srcPt++ =  i+offset;
	}
    srcPt=(u8 *)(_NONCACHE_STARTADDRESS+0x100000);
	UART_Printf("Write data[%d block, %d page].\n", block, page);

	ECC8_SETDIR_ENC();
	SetECCMsgLen(512);
	
	uRet = WritePage8_interleave(srcPt, block, page);
	
	if(uRet==FAIL) 
	{
		UART_Printf("Write Error.\n");
		return _false;
	} else 
	{
		UART_Printf("Write OK.\n");
	}

 	UART_Printf("Write data is");
	for(i=0; i<sNandInfo.uPageSize; i++) {
		if((i%16)==0) UART_Printf("\n%4x: ", i);
		UART_Printf("%02x ", *srcPt++);
	}
	UART_Printf("\n");

	UART_Printf("Spare:");
	for(i=0; i<sNandInfo.uRedundantAreaSize*sNandInfo.u512BytesPerPage; i++) {
		if((i%16)==0) UART_Printf("\n%4x: ", i);
	 	UART_Printf("%02x ", g_uSpareData[i]);
	}
	UART_Printf("\n\n");
	return _true;

}


/////////// end of interleaving operation //////////////////



void ReadNandImage2Dram(void)
{

//    unsigned long interrupt_reservoir;
    int i;
    int programError=0;
	u8 *srcPt,*saveSrcPt;
	u32 *rd_pt, *wr_pt;	
	u32 blockIndex;

	
	static volatile u32 srcAddress;
	static volatile u32 cmpAddress;		
	static u32 targetBlock;	    // Block number (0 ~ 4095)
	u32 targetSize;	    // Total byte size 


  	UART_Printf("\n[x-D Card NAND Flash writing program]\n");
    UART_Printf("The program buffer: 0x22000000~0x22ffffff(16MB)\n");

//	NF8_Init();

 //   rINTMSK = BIT_ALLMSK; 	
    srcAddress=_NONCACHE_STARTADDRESS + 0x1000000; 
    cmpAddress=_NONCACHE_STARTADDRESS; 

    	UART_Printf("\nAvailable target block number: 0~2047\n");
	UART_Printf("Input target block number:");
    	targetBlock=UART_GetIntNum();	// Block number(0~4095)
    	//if(targetSize==0)
    	//{
    		#if 0
    		UART_Printf("Input target size(0x4000*n):");
    		targetSize=UART_GetIntNum();	// Total byte size
		#else
   		UART_Printf("Input program file size(bytes): ");
    		targetSize=UART_GetIntNum();	// Total byte size
    		#endif
    	//}
	
	UART_Printf("File:%d[%d-block,%d-page,%d-bytes].\n", targetSize,(u32)((targetSize/2048)/64), (u32)((targetSize/2048)%64), (u32)(targetSize%2048));

   
	
    srcPt=(u8 *)srcAddress;
    blockIndex=targetBlock;


	SetECCMsgLen(512);		//g_msg_length=512; ECC8_SET_MSGLEN(g_msg_length);

	ECC8_SETDIR_DEC();


	                          	

    

     while(1) {
        saveSrcPt=srcPt;	


		// After 1-Block erase, Write 1-Block(32 pages).
		for(i=0;i<sNandInfo.uPagesPerBlock;i++) 
		{		
		    

#if 1
		    if(ReadPage8(srcPt, blockIndex,i)) {
				//UART_Printf("ECC Error(block=%d,page=%d!!!\n",blockIndex,i);
		    }
#endif
			
		srcPt+=sNandInfo.uPageSize;	// Increase buffer addr one pase size
		if(i==0)  UART_Printf("*");
			//UART_Printf("\b\b\b\b\b\b\b\b%04d,%02d]", blockIndex, i);
		if((u32)srcPt>=(srcAddress+targetSize)) // Check end of buffer
		break;	// Exit for loop
		}
		
		if(programError==1) {
		    blockIndex++;
		    srcPt=saveSrcPt;
		    programError=0;
		    continue;
		}

		if((u32)srcPt>=(srcAddress+targetSize)) break;	// Exit while loop

		blockIndex++;
		}

	////// compare WriteDramImage2NAND buffer and ReadNandImage2Dram buffer
	 rd_pt = (u32 *)srcAddress;		// readNand2Dram buffer
	 wr_pt = (u32 *)cmpAddress;  	// writeDram2Nand buffer
	 
	for(i=0;i<targetSize/4;i++)
	{
		if(*(rd_pt++) != *(wr_pt++))
		{
			UART_Printf("\n0x%x th compare fail!\n",i*4);
		}
	}


	 


    
}


void ClearSoftLock(void)
{
    	UART_Printf("xD-Card NAND SoftUnLock Test !!!\n");
	
	Outp32(rNFSBLK, 0x0);
	Outp32(rNFEBLK, 0x0);

	Outp32(rNFCONT, Inp32(rNFCONT)&~(1<<16)); 
	if(Inp32(rNFCONT)&(1<<17)){
		Outp32(rNFCONT, Inp32(rNFCONT)&~(1<<17)); 
		UART_Printf("Lock-tight\n ");
		UART_Printf("You can't unlock Protected blocks !!!\n ");
		UART_Printf("%d block ~ %d block are Programmable\n ", (Inp32(rNFSBLK)>>5), ((Inp32(rNFEBLK)>>5)-1));
	}
       else UART_Printf("All blocks are Programmable\n ");
}
