/**************************************************************************************
* 
*	Project Name : S5PC100 Validation
*
*	Copyright 2007 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PC100.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : dmc_test.c
*  
*	File Description : This file implements test functons for low power DRAM
*                      controller. (LPCON)
*
*	Author : Joonsoo Jeon
*	Dept. : AP Development Team
*	Created Date : 2007/11/13
*	Version : 0.1 
* 
*	History
*	- Created(Joonsoo Jeon 2007/11/05)
*  
**************************************************************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "def.h"
#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "intc.h"
#include "system.h"
#include "dmc.h"
#include "memcpy_lib.h"
#include "gpio.h"
#include "uart\uart.h"
#include "sysc.h"

////////////////////////////////////// user setting ///////////////////////////////////
#define USE_DMA		1
#define USE_TIMER	1
#define USE_MDMA    1
#define USE_PDMA0   0
#define USE_PDMA1   0
////////////////////////////////////////////////////////////////////////////////////

#define CHIP_SEL CHIP_0

#if USE_DRAM == LPDDR
#define CHIP0_SEL CHIP_BOTH
#define CHIP1_SEL CHIP_0
#elif USE_DRAM == DDR2
#define CHIP0_SEL CHIP_0
#define CHIP1_SEL CHIP_0
#endif // POP

#if USE_DMA
	#include "dma.h"
#endif // USE_DMA
#if USE_TIMER
	#include "timer.h"
#endif // USE_TIMER


#define DRAM_SIZE	(_DRAM_LimitAddress-_DRAM_BaseAddress)	// 128MB(DDR,LPDDR) : 0x4000000, 256MB(DDR2) : 0x10000000
#define DRAM0_SIZE	(_DRAM0_LimitAddress-_DRAM0_BaseAddress)	// 128MB(DDR,LPDDR) : 0x4000000, 256MB(DDR2) : 0x10000000
#define DRAM1_SIZE	(_DRAM1_LimitAddress-_DRAM1_BaseAddress)	// 128MB(DDR,LPDDR) : 0x4000000, 256MB(DDR2) : 0x10000000

// OneDram Address Setting
#define ONEDRAM_B_PORT_SEMAPHORE_ADDR (0x25fff800) //v210  DMC CS0 
#define ONEDRAM_B_PORT_SHARED0_BANK (0x25000000)
#define ONEDRAM_B_PORT_SHARED1_BANK (0x25800000)
#define ONEDRAM_B_PORT_DEDICATED_BANK (0x20000000)
#define	MailBox_AB			(0x25fff820)	// MailBox_AB(Transfer A to B), Bport Read
#define	MailBox_BA			(0x25fff840)	// MailBox_BA(Transfer B to A), Bport Write
#define	Check_AB			(0x25fff8A0)	// Check_AB  (Transfer A to B), Aport Read

// initialize function
u32 _InitDmc(void); // implementation by assembly in lpdmc_init.s
void Drive_Strength(void); //memory Strenth 
void Slew_Rate(void);  // Memory Slew Rate    

void DMCT_Sel_InitDMC(void); //DMC initialize 

// sub function for test
u32 DMCT_CheckInterface(u32 uBlock_size, u32 uOffset); // start point & end point - 0x10000
void DMCT_CheckInOutData(u32 uInterval, u32 uLoopNum); // data input and output check function
void DMCT_CheckBurst(u32 uBurst); // support 2,4,8,16 only
u32 DMCT_MeasureTimeCount(u8 ucChipNum, u8 ucMemcpyType, u8 ucBurstLength, u8 ucMapMethod, u8 ucOutOfOrder, u32 uTransSize, u32 uDstAddr, u32 uSrcAddr );

// memory controller function test
void DMCT_DramInterfaceTest(void);
void DMCT_BasicFuncTest(void);
void DMCT_PowerDownModeTest(void);
void DMCT_PowerDownModeAgingTest(void);
void DMCT_CommandTest(void);
void DMCT_QosTest(void);
void DMCT_QosAgingTest(void);
void DMCT_MemcpyPerfTest(void);
void DMCT_MemcpyManualTest(void);
void DMCT_Ebi2DmcAgingTest(void);
void DMCT_CacheManualTest(void);
void DMCT_memset(void);
void DMCT_MemcpyPerfTest2(void);

// memory Channel Select 
void DMCT_SetBaseAddress(void);
// Dual(only 1Port) Basic Function Test 
void DMCT_Sel_BasicFuctionTest(void);
// Dual(only 1Port) Commnand Test
void DMCT_Sel_CommandTest(void);
// Dual(only 1Port) PowerDownModeTest Test
void DMCT_Sel_PowerDownModeTest(void);
// Dual(only 1Port) QosTest
void DMCT_Sel_QosTest(void);
// Dual(only 1Port) MemcpyPerfTest
void DMCT_Sel_MemcpyPerfTest(void);
// Dual(only 1Port) Memcpy Manual Test
void DMCT_Sel_MemcpyManualTest(void);
// Dual(only 1Port) Cache Manual Test
void DMCT_Sel_CacheManualTest(void);
// Dual(only 1Port) Burst Test
void DMCT_Sel_TestBurst(void);
// Dual(only 1Port) Write & Verify (32bit-Single)
void DMCT_Sel_TestSingle(void);
// Dual(only 1Port) Write & Verify (Single-March)
void DMCT_Sel_TestMarch(void);
// Dual(only 1Port) Power down mode aging test
void DMCT_Sel_PowerDownModeAgingTest(void);
// Dual(only 1Port) QoS aging test
void DMCT_Sel_QosAgingTest(void);
// Dual(only 1Port) DMCAging Test
void DMCT_Sel_Ebi2DmcAgingTest(void);
void Sel_MemoryTest_4burstWrite_4burstRead_Aging(void);
void Sel_MemoryTest_xburstWriteRead_Aging(void);
void DMCT_Set_Port(void);
u8 DMCT_DramInterfaceAutoTest(void);
void DMCT_PhyDriving(void);
void DMCT_Qosclean(void);
void DMCT_PowerModeCurrent(void);
void DMCT_Dmc1ClkChangeAging(void);
void DMCT_Dmc0ClkChangeAging(void);
void DMCT_SetQosParameter(void);
void DMC_DLL_Value(void);


//A-Port SMDK AutoTest Code
u8 DMCT_OnedramAportCtypeSMDK_autotest(void);
u8 DMCT_OnedramAportAtypeSMDK_autotest(void);
u8 DMCT_OnedramAportAtypeSMDKInt_autotest(void); 
void SignalSemaphoreStatus(ONEDRAM_SEMAPHORE eSem);
void WaitSemaphoreStatus(ONEDRAM_SEMAPHORE eSem);
void __irq Isr_OneDram(void);
volatile int  g_APortInt;
volatile int  g_UseInt=0;
volatile int  g_Refcount;



// general write & read data library & test from SA
static u32 BurstOfSequence(u32 sa, u32 ea);
static void BurstOfSequenceWoVerifying(u32 sa, u32 ea);
static u32 BurstOfZigZag(u32 sa, u32 ea, u32 wr);
static void BurstOfZigZagWoVerifying(u32 sa, u32 ea, u32 wr);
static u32 March(u32 sa, u32 ea, u32 wr);
static void TestSingle(void);
static void TestBurst(void);
static void TestMarch(void);
// general write & read data library & test from APP
void MemoryTest_4burstWrite_4burstRead_Aging(void);
void MemoryTest_xburstWriteRead_Aging(void);

void dma_memcpy(u32 uDstAddr, u32 uSrcAddr, u32 uTransSize, u8 ucBurst);
void DMCT_Compare(u32 uD0area,u32 uD1area,u32 uCompareSize);

void SYS_DIV_AGING(void);

// global variable for write & read data library
static volatile u32 uStAddr1;
static volatile u32 uStAddr2;
static volatile u32 uLimitAddr1;
static volatile u32 uLimitAddr2;

// global variable for DRAM type
static volatile u32 g_uDramType;
extern u32 uDmcCh;
static u32 DRAM_BaseAddress;
static u32 Mem_Size;

/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////// Test Menu ///////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////

void DMC_Test(void)
{
	u32 uCountFunc=0;
	s32 iSel=0;
	u32 addr;


	const testFuncMenu g_aDMCTestFunc[]=
	{

        DMCT_Set_Port,                              "DMC Select Port",
		DMC_InitDmc,							"Initialize DMC [except DRAM code region]",
		Drive_Strength,								"Drive Strength: Min 0 Max 3",
		DMCT_DramInterfaceTest,     				"Check DRAM interface",

		DMCT_Sel_BasicFuctionTest,				"Basic function test [except DRAM code region]",
		DMCT_Sel_CommandTest,							"Send Command test [except DRAM code region]",
		DMCT_Sel_PowerDownModeTest,						"Power down mode test",
		DMCT_Sel_QosTest,								"QoS test",
		DMCT_MemcpyPerfTest2,							"Performance(Normal,Interleaved,BL4,ARM memcpy, 1MByte)",
		DMCT_Sel_MemcpyPerfTest,						"Memcpy performance test",
		DMCT_Sel_MemcpyManualTest,						"Memcpy manually",
		DMCT_Sel_CacheManualTest,						"Cache manually test",

		DMCT_Sel_TestBurst,              			"Write & Verify (Burst)",
		DMCT_Sel_TestSingle,             					"Write & Verify (32bit-Single)",
		DMCT_Sel_TestMarch,              					"Write & Verify (Single-March)",

		DMCT_Qosclean,								"Phy Control 1 Qos Cleaning ",
		DMCT_Sel_PowerDownModeAgingTest,			"Power down mode aging test",
		DMCT_Sel_QosAgingTest,						"QoS aging test",
		DMCT_PowerModeCurrent,						"Power Mode vs Current measurement",
		DMCT_Sel_Ebi2DmcAgingTest,					"EBI(nRCS0) to DMC memcpy aging",
		DMCT_PhyDriving,							"Phy Driving Test",
		DMCT_Dmc0ClkChangeAging,							"DMC0 OneDram Div Setting",
		DMCT_Dmc1ClkChangeAging,							"DMC1 HCLK_MSYS Div Setting",
		Sel_MemoryTest_4burstWrite_4burstRead_Aging,"4 Burst W/R aging [except DRAM code region]",
		Sel_MemoryTest_xburstWriteRead_Aging,		"X Burst W/R aging [except DRAM code region]",
		DMCT_memset,                                "Memset Test",
		DMCT_OnedramAportAtypeSMDK_autotest,        "OneDram A-port Test ",
		DMCT_OnedramAportAtypeSMDKInt_autotest,		"Auto Test (Interrupt)",
		DMCT_SetQosParameter,						"DMC (fast)Qos Setting",
		SYS_DIV_AGING,								"ARM ClK Aging Test.800/400/200",
		DMC_DLL_Value,								"DLL C,F lock value check",
		0, 0
	};

// Default DMC Port0 Select
	uDmcCh = 0;
	DRAM_BaseAddress = _DRAM_BaseAddress;
	Mem_Size = DRAM_SIZE;
//	DMCT_SetBaseAddress();

	
//	uStAddr1  = DRAM_BaseAddress+0x1000000;
//	uLimitAddr1 = DRAM_BaseAddress+0x1080000;

//	uStAddr2 =  _DRAM_BaseAddress+0x2000000;
//	uLimitAddr2 = _DRAM_BaseAddress+0x2080000;

	UART_Printf(" Enter the stAddr (0x%08x): ", uStAddr1);
	addr = UART_GetIntNum();
	uStAddr1 = (addr != -1) ? addr: uStAddr1;
	UART_Printf(" Enter the endAddr (0x%08x): ", uLimitAddr1);
	addr = UART_GetIntNum();
	uLimitAddr1 = (addr != -1) ? addr: uLimitAddr1;

	UART_Printf("\n");


	while(1)
	{
		UART_Printf("\n\n================== DMC Function Test =====================\n\n");
	//	g_uDramType = DMC_GetUsingDramType();

		for (uCountFunc=0; (u32)(g_aDMCTestFunc[uCountFunc].desc)!=0; uCountFunc++)
			UART_Printf("%2d: %s\n", uCountFunc, g_aDMCTestFunc[uCountFunc].desc);
		UART_Printf("(0x%08x - 0x%08x)\n", uStAddr1, uLimitAddr1);

		UART_Printf("\nSelect the function to test : ");
		iSel = UART_GetIntNum();
		UART_Printf("\n");
		if(iSel == -1) 
			break;

		if (iSel>=0 && iSel<(sizeof(g_aDMCTestFunc)/8-1))
			(g_aDMCTestFunc[iSel].func) ();
	}

}


/////////////////////////////////////////////////////////////////////////////////
///////////////////////////// Sub functions for Test ////////////////////////////
/////////////////////////////////////////////////////////////////////////////////

//////////
// Function Name : CheckDramInterface
// Function Description : Check connectivity of DRAM interface
// Input : Block size
// Output : u32 result (1b'x:around start address, 2b'x:around end address)
// Version : v0.0
u32 DMCT_CheckInterface(u32 uBlock_size, u32 uOffset)
{

	u32 uCData[5] = { 0x00000000, 0xffffffff, 0x55555555, 0xaaaaaaaa, 0x12345a5a };
	u32 uRData;
	u32 *upWriteAddr, *upReadAddr;
	u32 i, j, loop_size, result=0;

	loop_size = uBlock_size / sizeof(uCData);
	
	// From start address + 0x1000000  About block size
	upWriteAddr = (u32 *)(DRAM_BaseAddress+uOffset);
	upReadAddr = (u32 *)(DRAM_BaseAddress+uOffset);
	for (i=0;i<loop_size;i++)
	{
		for (j=0; j<sizeof(uCData)/sizeof(u32); j++)
			*(upWriteAddr++) = uCData[j];
		
		for (j=0; j<sizeof(uCData)/sizeof(u32); j++)
		{
			uRData = *(upReadAddr++);
			if (uRData != uCData[j]) 
			{
				result &= ~(1); 
				break;
			}
			result |= (1);
		}
	}

	// From start address + DRAM SIZE - block size + uOffset  About +block size
	upWriteAddr = (u32 *)(DRAM_BaseAddress+Mem_Size/2-uBlock_size-uOffset);
	upReadAddr = (u32 *)(DRAM_BaseAddress+Mem_Size/2-uBlock_size-uOffset);
	for (i=0;i<loop_size;i++)
	{
		for (j=0; j<sizeof(uCData)/sizeof(u32); j++)
			*(upWriteAddr++) = uCData[j];
		
		for (j=0; j<sizeof(uCData)/sizeof(u32); j++)
		{
			uRData = *(upReadAddr++);
			if (uRData != uCData[j]) 
			{
				result &= ~(1<<1); 
				break;
			}
			result |= (1<<1);
		}
	}

	return result;
}



//////////
// Function Name : DMCT_CheckBurst
// Function Description : data input and output check function
// Input : 	u32 uBurst - word burst value (2,4,8 or 16)
// Output : none
// Version : v0.0

void DMCT_CheckBurst(u32 uBurst) // support 2,4,8,16 only
{
	u32 uTempStatus, uTempEmpty, uBfactor;
	
	switch (uBurst)
	{
		case 2 : uBfactor = DMC_BURST2;
			break;
		case 4 : uBfactor = DMC_BURST4;
			break;
		case 8 : uBfactor = DMC_BURST8;
			break;
		case 16 : uBfactor = DMC_BURST16;
			break;
		default : uBfactor = DMC_BURST4; uBurst = 4;
			break;
	}
	 
	Disp(" burst %2d : press enter & measure transfer timing \n", uBurst);
	UART_GetIntNum(); // wait for pressing enter key

	do 
		uTempStatus = DMC_GetBankStatus(0);
	while (uTempStatus!=0 && uTempStatus!=0x33333333); // exit during idle or selfrefresh mode of memory
	do 
		uTempEmpty = DMC_GetQueueEmpty(0);
	while (uTempEmpty == 0);
	
	DMC_ChangeBurstLength(uBfactor); // setting Con BL, 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16
	if(g_uDramType==DT_DDR2) 
		DMC_SendCommand(CHIP_SEL, CMD_MRS, 0, 0x440|(uBfactor)); // DDR2, setting MRS BL [2:0], 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16
	else if (g_uDramType==DT_LPDDR2)
		DMC_SendCommand(CHIP_SEL, CMD_MRS, 0, 0x40|(uBfactor)); // LPDDR2, setting MRS BL [2:0], 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16
	else // DDR, LPDDR
		DMC_SendCommand(CHIP_SEL, CMD_MRS, 0, 0x30|(uBfactor)); // DDR, setting MRS BL [2:0], 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16

	dma_memcpy(DRAM_BaseAddress+0x900000, DRAM_BaseAddress+0x100000, 0x80000, uBfactor-1);		
//	BurstOfSequence(DRAM_BaseAddress+0x100000, DRAM_BaseAddress+0x100100); // burst read & burst write
}



//////////
// Function Name : Drive_Strength
// Function Description : PORT GROUP MP1,2 Drive Stregth REGISTER
// Input : 	Drive Stregth Value
// Output : none
// Version : v0.0

void Drive_Strength(void) // Default 0XAAAA
{
	u32 uTemp=0,uTemp_strength,uTemp_slewrate;
	u32 uDrv,uPort,uSlew;
    u32 Strength_Addr;
    u32 uMP1_0DRV = 0xE02003CC;
#if 0
	u32 uMP1_1DRV = 0xE02003CC;
    u32 uMP1_2DRV = 0xE02003CC;
    u32 uMP1_3DRV = 0xE02003CC;
    u32 uMP1_4DRV = 0xE02003CC;
    u32 uMP1_5DRV = 0xE02003CC;
    u32 uMP1_6DRV = 0xE02003CC;
    u32 uMP1_7DRV = 0xE02003CC;
#endif
	
	u32 uMP2_0DRV = 0xE02004EC;
do
	{
		Disp("\npress enter Memory Strength Value ::");
		uTemp_strength = UART_GetIntNum();
	}
while(uTemp_strength< 0 || uTemp_strength > 3);

do
	{
		Disp("\npress enter Memory Slewrate Value(0:Fast,1:slew) ::");
		uTemp_slewrate = UART_GetIntNum();	
	}
while(uTemp_slewrate< 0 || uTemp_slewrate > 1);

	switch (uTemp_strength)
	{
		case 0 : uDrv = 0x0000;
			break;
		case 1 : uDrv = 0x5555;
			break;
		case 2 : uDrv = 0xAAAA;
			break;
		case 3 : uDrv = 0xFFFF;
			break;
		default : uDrv = 0xAAAA;
			break;
	}
	switch (uTemp_slewrate)
	{
		case 0 : uSlew = 0x00;
			break;
		case 1 : uSlew = 0x11;
			break;
		default : uSlew = 0x00;
			break;
	}


for(uPort=0; uPort<9; uPort++)
	{
       if(uDmcCh ==0)
       	{
       	Strength_Addr = (uMP1_0DRV + uPort*0x20);
    	Outp32( Strength_Addr , uTemp =(uDrv<<0|uSlew<<16)) ; 
       	}
	   else
	   	{
		 Strength_Addr = (uMP2_0DRV + uPort*0x20);
		 Outp32( Strength_Addr , uTemp = (uDrv<<0|uSlew<<16)) ; 
	    }
	}
}

//////////
// Function Name : Slew_Rate
// Function Description : PORT GROUP MP1,2 Slew_Rate REGISTER
// Input : 	Drive Stregth Value
// Output : none
// Version : v0.0

void Slew_Rate(void) // Default 0XFF
{
	u32 uTemp,uTemp_slewrate;
	u32 uSlew,uPort;
    u32 Strength_Addr;
    u32 MP1_0DRV = 0xE02003CC;
	u32 MP2_0DRV = 0xE02004EC;

do
	{
		Disp("press enter Memory Slew Rate Value. \n");
		uTemp_slewrate = UART_GetIntNum();
	}
while(uTemp_slewrate< 0 || uTemp_slewrate > 1);
	
	switch (uTemp_slewrate)
	{
		case 0 : uSlew = 0x00;
			break;
		case 1 : uSlew = 0xFF;
			break;
		default : uSlew = 0xFF;
			break;
	}

for(uPort=0; uPort<9; uPort++)
	{
		if(uDmcCh == 0)
			{
   		Strength_Addr = (MP1_0DRV + uPort*0x20);
    	Outp32( Strength_Addr , (uTemp = uTemp &(0x00<<16) | (uSlew<<16))) ; 
			}
		else
			{
	    Strength_Addr = (MP2_0DRV + uPort*0x20);
    	Outp32( Strength_Addr , (uTemp = uTemp &(0x00<<16) | (uSlew<<16))) ; 
			}
	}

}


//////////
// Function Name : DMCT_CheckInOutData
// Function Description : data input and output check function
// Input : 	u32 uInterval - additional address interval per single test
//			u32 uDataNum - Loop count
// Output : u32 result (1b'x:around start address, 2b'x:around end address)
// Version : v0.0
void DMCT_CheckInOutData(u32 uInterval, u32 uLoopNum)
{
	u32 uInputData[8];
	u32 i;
	u32 uFail = 0;
	
	for (i=0;i<uLoopNum;i++)
	{
		Outp32(DRAM_BaseAddress+i*uInterval, i*2); 
		uInputData[i] = Inp32(DRAM_BaseAddress+i*uInterval);
		if (uInputData[i] != i*2) 
		{
			uFail |= 1; // if it could be occured at once
			Disp(" data in/out fail !! \n"); 
		}
	}
	
	if (uFail != 1)
		Disp(" data in/out pass !! \n");
}


void dma_memcpy(u32 uDstAddr, u32 uSrcAddr, u32 uTransSize, u8 ucBurst) // for DMA mem copy test in polling mode
{
#ifdef USE_DMA
	static DMAC oDmacTemp;

	DMA_SetCh(DMA_20, &oDmacTemp); // MDMA ch 0
	DMA_InitCh(WORD, SOFTWARE, DMA_M2M, ucBurst, &oDmacTemp);  
	DMA_StartCh(uSrcAddr, uDstAddr, uTransSize/4, &oDmacTemp);	// transfer size by word

	while (!DMA_IsTransferDone(&oDmacTemp)); 
	DMA_StopCh(&oDmacTemp);
#endif
}

//////////
// Function Name : DMCT_MeasurePerformance
// Function Description : data input and output check function
// Input : 	u8 ucChipNum
//  		u8 ucBurstLength : Burst mode 
//  		u8 ucMapMethod : Address mapping 
//  		u8 ucOutOfOrder : Out of order scheduling function 
//  		u32 uTransSize : Transfer size
// Output : u32 the time required (unit:usec)
// Version : v0.0
u32 DMCT_MeasureTimeCount(u8 ucChipNum, u8 ucMemcpyType, u8 ucBurstLength, u8 ucMapMethod, u8 ucOutOfOrder, u32 uTransSize, u32 uDstAddr, u32 uSrcAddr )
{
	u32 uTempEmpty, uTimeCount;
	u32 i;

	if(uDmcCh==2)
	{
		do 
		uTempEmpty = DMC_GetQueueEmpty(1);
	    while (uTempEmpty == 0);
	}
	else
	{
		do 
		uTempEmpty = DMC_GetQueueEmpty(0);
	    while (uTempEmpty == 0);
	}

	// cache disable for stable MRS.
//	 CoDisableICache();
//	 CoDisableDCache();

	DMC_ChangeBurstLength(ucBurstLength); // setting Con BL, 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16
    DMC_ChangeMapMethod(ucChipNum, ucMapMethod);
	DMC_SetEnhancedUtility(0xff, 1, ucOutOfOrder);

//	UART_Printf(" Bank status1 (0x%08x): \n", DMC_GetBankStatus(ucChipNum));
//	DMC_SendCommand(ucChipNum, CMD_NOP, 0, 0);
	DMC_SendCommand(ucChipNum, CMD_PALL, 0, 0x400); // command 1, address set
//	UART_Printf(" Bank status2 (0x%08x): \n", DMC_GetBankStatus(ucChipNum));

	if(g_uDramType==DT_DDR2) 
		DMC_SendCommand(ucChipNum, CMD_MRS, 0, 0x440|(ucBurstLength)); // DDR2, setting MRS BL [2:0], 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16
	else if (g_uDramType==DT_LPDDR2)
		DMC_SendCommand(ucChipNum, CMD_MRS, 0, 0x40|(ucBurstLength)); // LPDDR2, setting MRS BL [2:0], 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16
	else // DDR, LPDDR
		DMC_SendCommand(ucChipNum, CMD_MRS, 0, 0x30|(ucBurstLength)); // DDR, setting MRS BL [2:0], 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16
//	UART_Printf(" Bank status3 (0x%08x): \n", DMC_GetBankStatus(ucChipNum));

	// enable cache.
//	CoEnableDCache();
//	CoEnableICache();

	for (i=0;i<64;i++)
	{
		*(u32 *)(uSrcAddr+i*4) = rand() * i;
		*(u32 *)(uSrcAddr+uTransSize-i*4) = rand() * i;
	}
#ifdef USE_TIMER
	DisplayLED(0x5);
	StartTimer(2);
#endif //USE_TIMER
	switch(ucMemcpyType)
	{
		case 0 : 
			_burst2_memcpy((u32 *)(uDstAddr), (u32 *)(uSrcAddr), uTransSize);
			break;
		case 1 : 
			_burst4_memcpy((u32 *)(uDstAddr), (u32 *)(uSrcAddr), uTransSize);
			break;
		case 2 : 
			_burst8_memcpy((u32 *)(uDstAddr), (u32 *)(uSrcAddr), uTransSize);
			break;
		case 3 : // memcpy in standard library
			memcpy((u32 *)(uDstAddr), (u32 *)(uSrcAddr), uTransSize);
			break;
		case 4 : // 4 burst
			dma_memcpy(uDstAddr, uSrcAddr, uTransSize, BURST4);
			break;
		case 5 : // 8 burst
			dma_memcpy(uDstAddr, uSrcAddr, uTransSize, BURST8);	// for MDMA
			break;	   			
		case 6 : // 16 burst
			dma_memcpy(uDstAddr, uSrcAddr, uTransSize, BURST16);	// not support in C100
			break;	   			
		case 7 : // 64 burst
			dma_memcpy(uDstAddr, uSrcAddr, uTransSize, BURST64);	// not support in C100
			break;	   			
		case 8 : // 256 burst 
			dma_memcpy(uDstAddr, uSrcAddr, uTransSize, BURST128);
			break;
		default : 
			break;	   			
	}

#ifdef USE_TIMER
	uTimeCount = StopTimer(2); // u second
	DisplayLED(0x0);
#endif //USE_TIMER
	
	if ( Compare(uSrcAddr, uDstAddr, uTransSize/4) != true)
		Disp(" ...followed item copy failed...\n");
	return uTimeCount;
}


/////////////////////////////////////////////////////////////////////////////////
/////////////////////// Memory Controller Test Functions ////////////////////////
/////////////////////////////////////////////////////////////////////////////////

void DMCT_DramInterfaceTest(void)
{
	u32 res;

		res = DMCT_CheckInterface(0x1000, 0x1000000);
		if (res == 3)
		{
		Disp(" DMC[%d] PORT Memory interface : pass\n\n",uDmcCh);
		}
		else 
		Disp(" DMC[%d] PORT Memory interface : fail\n\n",uDmcCh);

}	

////////// for v210(2port) basic function Test 
void DMCT_Sel_BasicFuctionTest(void)
{

	if( uDmcCh== 0||uDmcCh==2)
	{
	// 1. Init Dmc0
	Disp("DMC0 Basic Function Test\n");
	DMC_SetCh(0);
	DMCT_SetBaseAddress();
	DMCT_BasicFuncTest();

	}
	else if(uDmcCh==1)
	{
	// 2. Init DMC1
	Disp("DMC1 Basic Function Test\n");
	DMC_SetCh(1);
	DMCT_SetBaseAddress();
	DMCT_BasicFuncTest();
	}
	else 
		Assert(0);

}



/// for basic function test
void DMCT_BasicFuncTest(void)	// caution : you should do this test during code is not running in DRAM
{
	Disp(" Caution : you should do this test during code is not running in DRAM\n");
	Disp(" Are you ready for test? (y/n)\n");
	if (getchar() == 'n') return; 

	DMC_SetPowerDownModes(0,0,0,0,0); // clear memory power down mode.

	// ** burst type test, DDR2 & LPDDR2 are supported only burst 4
	// * burst 2
	DMCT_CheckBurst(2);
	// * burst 8
	DMCT_CheckBurst(8);
	// * burst 16
	DMCT_CheckBurst(16);
	// * burst 4
	DMCT_CheckBurst(4); // last test item as burst 4 

		// * linear mapping
		Disp("\n linear mapping : press enter & measure bank address");
		UART_GetIntNum(); // wait for pressing enter key

//		DMC_ChangeMapMethod(CHIP_SEL, MAP_LINEAR); // chip 0, linear mapping
		DMC_ChangeMapMethod(CHIP1_SEL, MAP_LINEAR);
		

		if((g_uDramType==DT_DDR2)||(g_uDramType==DT_LPDDR2)) 
			DMCT_CheckInOutData(0x2000000, 8); // in case, 256MB 8bank.. 1 bank = 32MB
		else // DDR, LPDDR
			DMCT_CheckInOutData(0x2000000, 4); // in case, 128MB/16-bitx2 4bank.. 1 bank = 32MB
	//		DMCT_CheckInOutData(0x1000000, 4); // in case, 64MB 4bank.. 1 bank = 16MB
	if(uDmcCh ==2)
		{

			Disp("\n OneDram 2bank interleave mapping : press enter & measure bank address");
			UART_GetIntNum(); // wait for pressing enter key
//			DMC_ChangeMapMethod(CHIP_SEL, MAP_MIXED1); // chip 0, interleave mapping
			DMC_ChangeMapMethod(1, MAP_MIXED1); // chip 1, 2bank interleave mapping
			if((g_uDramType==DT_DDR2)||(g_uDramType==DT_LPDDR2)) 
				DMCT_CheckInOutData(0x1000, 8); // in case, 256MB 8bank. column 10 bits. 1 bank = 32MB
			else // DDR, LPDDR
				DMCT_CheckInOutData(0x1000, 4); // in case, 128MB/16-bitx2 4bank. column 10 bits. 1 bank = 16MB
		//		DMCT_CheckInOutData(0x800, 4); // in case, 64MB 4bank. column 9 bits. 1 bank = 16MB
	// ** address mapping mode test in DDR2 256MB
		}
	else
		{
	// * interleave mapping, change bank address A11, A12
	Disp("\n interleave mapping : press enter & measure bank address");
	UART_GetIntNum(); // wait for pressing enter key
	DMC_ChangeMapMethod(CHIP_SEL, MAP_INTERLEAVED); // chip 0, interleave mapping
	if((g_uDramType==DT_DDR2)||(g_uDramType==DT_LPDDR2)) 
		DMCT_CheckInOutData(0x1000, 8); // in case, 256MB 8bank. column 10 bits. 1 bank = 32MB
	else // DDR, LPDDR
		DMCT_CheckInOutData(0x1000, 4); // in case, 128MB/16-bitx2 4bank. column 10 bits. 1 bank = 16MB
//		DMCT_CheckInOutData(0x800, 4); // in case, 64MB 4bank. column 9 bits. 1 bank = 16MB
		}
}


/// for Power down mode test
void DMCT_PowerDownModeTest(void)
{
	u32 uBankInterval, uBankCount;
	
	if((g_uDramType==DT_DDR2)||(g_uDramType==DT_LPDDR2)) // in case, 256MB 8bank.. 1 bank = 32MB
	{
		uBankInterval=(uDmcCh==0)?0x2000000:0x80000000;
		uBankCount = 8;
	}
	else // DDR, LPDDR. in case, 128MB 4bank.. 1 bank = 16MB
	{
		uBankInterval=(uDmcCh==0)?0x2000000:0x2000000;
		uBankCount = 4;
	}	

	// change address mapping method to linear mapping 
	DMC_ChangeMapMethod(CHIP_SEL, MAP_LINEAR);
	// disable all power down mode for checking successive
	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);

	// * check dynamic clock control(clock stop) in LPDDR, LPDDR2 only
	Disp("\n Check dynamic clock control (clock stop)\n : press enter & measure SCLK or nSCLK");
	UART_GetIntNum(); // wait for pressing enter key
	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_EN);
	DMCT_CheckInOutData(uBankInterval, uBankCount); 

	// * check precharge policy per bank
	Disp("\n Check close page precharge policy per bank\n : press enter & measure A10 bit");
	UART_GetIntNum();
	DMC_SetPrechPolicy(0xff, 0xff); // close page policy for all bank ,u8 ucCHIP_SELPrechPolicy, u8 ucChip1PrechPolicy
									// check automatically performs precharging(A10 bit) after Read or Write
	DMCT_CheckInOutData(uBankInterval, uBankCount); 
	
	// * force precharge test	
	Disp("\n Check open page precharge policy per bank (count 3)\n : press enter & measure precharge command");
	UART_GetIntNum();
	DMC_SetPrechPolicy(0x00, 0x00); // open page policy for all bank ,u8 ucCHIP_SELPrechPolicy, u8 ucChip1PrechPolicy
	DMC_SetPowerDownCounter(3,0,0);	// set only precharge count & check count
	DMC_SetPowerDownModes(FORCE_PRE_EN, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, FORCE_PRE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
	DMCT_CheckInOutData(uBankInterval, uBankCount); 
									 
	// * Dynamic power down test
	Disp("\n Check dynamic active power down (count 63)\n : press enter & measure CKE low");
	UART_GetIntNum();
	DMC_SetPowerDownCounter(1,0x3f,0);	// set precharge count & Dynamic power down count
	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
	DMCT_CheckInOutData(uBankInterval, uBankCount); 

	Disp("\n Check dynamic precharge power down (count 63)\n : press enter & measure CKE low");
	UART_GetIntNum();
	DMC_SetPowerDownCounter(1,0x3f,0);	// set precharge count & Dynamic power down count
	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_DIS, FORCE_PRE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
	DMCT_CheckInOutData(uBankInterval, uBankCount); 

	// * Dynamic self refresh test
	Disp("\n Check dynamic self refresh (count 255)\n : press enter & measure entering self-refresh");
	UART_GetIntNum();
	DMC_SetPowerDownCounter(1,3,0xff);	// set precharge, Dynamic power down & Dynamic self refresh count
	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_EN, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
	DMCT_CheckInOutData(uBankInterval, uBankCount); 

	// * Enable all power down function 
	Disp("\n Enable all power down function (count 1,6,1023)\n : press enter & measure timing");
	UART_GetIntNum();
	DMC_SetPowerDownCounter(1,6,0x3ff);	// set precharge, Dynamic power down & Dynamic self refresh count
	DMC_SetPowerDownModes(FORCE_PRE_EN, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_EN, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_EN);
	DMCT_CheckInOutData(uBankInterval, uBankCount); 

	// disable all power down mode 
	Disp("\n Test end");
	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
}

void DMCT_PowerDownModeAgingTest(void)
{
	u32 ret;
	
	// ** Dynamic self refresh test
	DMC_SetPowerDownCounter(1,0x2,0x3);	// set precharge count,Dynamic power down count & Dynamic self refresh count
	DMC_SetPowerDownModes(FORCE_PRE_EN, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_EN, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_EN);
	DMCT_CheckInOutData(0x2000000, 1); // in case, 256MB 8bank.. 1 bank = 32MB	

	// * Memcpy
	Disp(" Power down mode aging test using Memcpy. Press ESC key to exit \n");
	while(UART_GetKey()!=0x1B) // ESC key
	{
		ret = BurstOfSequence(DRAM_BaseAddress, DRAM_BaseAddress+Mem_Size/256); // burst read & burst write
		ret ? Disp(" bs: pass\n"): Disp(" bs: fail\n");
		ret = BurstOfZigZag(DRAM_BaseAddress+Mem_Size/256, DRAM_BaseAddress+Mem_Size*2/256, 0x11223344); // burst read & burst write
		ret ? Disp(" bz: pass\n"): Disp(" bz: fail\n");
		ret = March(DRAM_BaseAddress+Mem_Size/256, DRAM_BaseAddress+Mem_Size*3/256, 0x55aa55aa); // burst read & burst write
		ret ? Disp(" ma: pass\n"): Disp(" ma: fail\n");
		ret = BurstOfSequence(DRAM_BaseAddress+Mem_Size/256, DRAM_BaseAddress+Mem_Size*4/256); // burst read & burst write
		ret ? Disp(" b2: pass\n"): Disp(" b2: fail\n");
		Disp("Press ESC key to exit \n");
	}
	
	// * MFC operating
	Disp(" Power down mode aging test using MFC operating \n");
	UART_GetIntNum(); // wait for pressing enter key
	// implement this..
}



/// for Command test
void DMCT_CommandTest(void)
{
	u32 uTempEmpty;
	u32 uCnt = 0;
	static DMAC oDmacTemp;
	
#ifdef USE_DMA
	// * Queue empty test
	// DMA setting for queue empty test 
	DMA_SetCh(DMA_20, &oDmacTemp); // MDMA ch 0
	DMA_InitCh(WORD, SOFTWARE, DMA_M2M, BURST4, &oDmacTemp);  
	DMA_StartCh(DRAM_BaseAddress, DRAM_BaseAddress+Mem_Size/2, Mem_Size/32/4, &oDmacTemp);	// transfer size by word
#endif // USE_DMA

	do 
	{
		uCnt++; // default value is '1'
		uTempEmpty = DMC_GetQueueEmpty(0);
	}
	while (uTempEmpty == 0); // while queue is not empty 

#ifdef USE_DMA
	DMA_StopCh(&oDmacTemp);
	Disp(" Queue Empty count : %d\n", uCnt);
#endif
	
	// * Send command test, need to measure with equipment
	if(g_uDramType==DT_DDR2) // DDR2
	{
		Disp(" MRS : mode register setting command\n");
		UART_GetIntNum(); // wait for pressing enter key
		DMC_SendCommand(CHIP_SEL, CMD_MRS, 0, 0x442); // DDR2, setting MRS BL [2:0], 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16
		Disp(" EMRS1 : extended mode register 1 setting command\n");
		UART_GetIntNum();
		DMC_SendCommand(CHIP_SEL, CMD_MRS, 1, 0x10400);
		Disp(" EMRS2 : extended mode register 2 setting command\n");
		UART_GetIntNum();
		DMC_SendCommand(CHIP_SEL, CMD_MRS, 2, 0x20000);
		Disp(" EMRS3 : extended mode register 3 setting command\n");
		UART_GetIntNum();
		DMC_SendCommand(CHIP_SEL, CMD_MRS, 3, 0x30000);
	}
	else if(g_uDramType==DT_LPDDR2) // LPDDR2
	{
		// need to implement MRS
		
		Disp(" MRR : mode register reading command\n");
		UART_GetIntNum();
		DMC_SendCommand(CHIP_SEL, CMD_MRR, 0, 0); // MRR is supported only LPDDR2
	}
	else // DDR, LPDDR
	{
		Disp(" MRS : mode register setting command\n");
		UART_GetIntNum(); // wait for pressing enter key
		DMC_SendCommand(CHIP_SEL, CMD_MRS, 0, 0x31); // DDR, setting MRS BL [2:0], 1=>bl2, 2=>bl4, 3=>bl8, 4=>bl16
		Disp(" EMRS : extended mode register setting command\n");
		UART_GetIntNum();
		DMC_SendCommand(CHIP_SEL, CMD_MRS, 2, 0x39);
	}
	Disp(" PALL : precharge all command\n");
	UART_GetIntNum();
	DMC_SendCommand(CHIP_SEL, CMD_PALL, 0, 0x400); // command 1, address set
									 // u8 ucChipNum, u8 ucCmd, u8 ucCmdBank, u16 ucCmdAddr
	Disp(" PRE : per bank precharge command \n");
	UART_GetIntNum();
	DMC_SendCommand(CHIP_SEL, CMD_PRE, 2, 0);

	Disp(" DPD : deep power down command\n");
	UART_GetIntNum();
	DMC_SendCommand(CHIP_SEL, CMD_DPD, 0, 0);
	Disp(" NOP : exit from active/precharge power down or deep power down command \n");
	UART_GetIntNum();
	DMC_SendCommand(CHIP_SEL, CMD_NOP, 0, 0);

	Disp(" REFS : self refresh command \n");
	UART_GetIntNum();
	DMC_SendCommand(CHIP_SEL, CMD_REFS, 0, 0);
	Disp(" REFSX : exit from self refresh command\n");
	UART_GetIntNum();
	DMC_SendCommand(CHIP_SEL, CMD_REFSX, 0, 0);

	Disp(" CKEL : active/precharge power down command \n");
	UART_GetIntNum();
	DMC_SendCommand(CHIP_SEL, CMD_CKEL, 0, 0);
	Disp(" NOP : exit from active/precharge power down or deep power down command \n");
	UART_GetIntNum();
	DMC_SendCommand(CHIP_SEL, CMD_NOP, 0, 0);


	Disp(" REFA : auto refresh command \n");
	UART_GetIntNum();
	DMC_SendCommand(CHIP_SEL, CMD_REFA, 0, 0);
}


/// for Qos test
#ifdef USE_TIMER
static volatile u32 uElapTime[3];
#endif //USE_TIMER

#ifdef USE_DMA
static volatile DMAC oDmac0, oDmac1, oDmac2;
static volatile u8 bDmaDone[3];

static void __irq Isr_PDma0Done(void)
{
	u32 uInt;
	
	INTC_Disable(NUM_PDMA0);

	DMA_GetIntrSrc(&uInt, &oDmac0);
	if (uInt==32)
	{
		Disp("[PDMA0 FAULT & STOP]");
		DMA_StopDmac(&oDmac0);
	}
	else
	{
		DMA_ClearIntPending(uInt, &oDmac0);	
		bDmaDone[0] = true;
	}
	
//	Disp("\t Occur PDMA0 done!!! \n\n");
	INTC_ClearVectAddr();

	INTC_Enable(NUM_PDMA0);

#ifdef USE_TIMER
	uElapTime[0] = StopTimer(0);
#endif // USE_TIMER
}

static void __irq Isr_PDma1Done(void)
{
	u32 uInt;

	INTC_Disable(NUM_PDMA1);

	DMA_GetIntrSrc(&uInt, &oDmac1);
	if (uInt==32)
	{
		Disp("[PDMA1 FAULT & STOP]");
		DMA_StopDmac(&oDmac1);
	}
	else
	{
		DMA_ClearIntPending(uInt, &oDmac1);	
		bDmaDone[1] = true;
	}
//	Disp("\t Occur PDMA1 done!!! \n\n");
	INTC_ClearVectAddr();

	INTC_Enable(NUM_PDMA1);

#ifdef USE_TIMER
	uElapTime[1] = StopTimer(1);
#endif // USE_TIMER
}

static void __irq Isr_MDmaDone(void)
{
	u32 uInt;

	INTC_Disable(NUM_MDMA);

	DMA_GetIntrSrc(&uInt, &oDmac2);
	if (uInt==32)
	{
		Disp("[MDMA FAULT & STOP]");
		DMA_StopDmac(&oDmac2);
	}
	else
	{
		DMA_ClearIntPending(uInt, &oDmac2);	
		bDmaDone[2] = true;
	}
//	Disp("\t Occur MDMA done!!! \n\n");
	INTC_ClearVectAddr();

	INTC_Enable(NUM_MDMA);

#ifdef USE_TIMER
	uElapTime[2] = StopTimer(2);
#endif // USE_TIMER
}
#endif // USE_DMA

static void SetNonSecure(void)	// S3C24C0 is no secure area.
{
	Outp32(0xF1500804,0xff );
	Outp32(0xF1500810,0xff );
	Outp32(0xF150081C,0xff );
	Outp32(0xFAD00804,0xff );
	Outp32(0xFAD00810,0xff );
	Outp32(0xFAD0081C,0xff );
	Outp32(0xFAD00828,0xff );
	Outp32(0xE0600804,0xff );
	Outp32(0xE0600810,0xff );
	Outp32(0xE060081C,0xff );
	Outp32(0xE0600828,0xff );
	//Outp32(0xE1C00804,0xff );
	//Outp32(0xE1C00810,0xff );
	//Outp32(0xE1C0081C,0xff );
	//Outp32(0xE1C00828,0xff );

}

void DMCT_QosTest(void)
{
#ifdef USE_DMA
	u32 uCh, uT,i;
	u16 QosCount[3];
	u32 uTransSize = 0x1000;
	u32 uStepsize=0x00800000;
	u32 uQosDMAMask[3], uQosDMAID[3];
	u32 uQosEN;
	u32 uSrcAddr;
	
	///////// user setting //////////
//	u32  = DRAM_BaseAddress+0x01000000;
	/////////////////////////////
#if 0
	Disp("\n Select Qos Enable(0) or Disable(1)::   " ); 
	uQosEN=	UART_GetIntNum();
#endif 
	uQosEN=0;

	if(uDmcCh==0)
		{
		uQosDMAMask[0] = 0x43FF; //DMC0 pdma0 mask
		uQosDMAMask[1] = 0x43FF; //DMC0 pdma1 mask
		uQosDMAMask[2] = 0x7C3F; //DMC0 mdam  mask


		uQosDMAID[0]= 0x1F2 ; //DMC0 pdma0 bus id
		uQosDMAID[1]= 0x3F2 ; //DMC0 pdma1 bus id
		uQosDMAID[2]= 0x3A ;  //DMC0 mdma  bus id
		uSrcAddr  = 0x22000000;		
		}
	else
		{
		uQosDMAMask[0] = 0x43FF; //DMC1 pdma0 mask
		uQosDMAMask[1] = 0x43FF; //DMC1 pdma1 mask
		uQosDMAMask[2] = 0x7C3F; //DMC1 mdam  mask

		uQosDMAID[0]= 0x1F3 ; //DMC1 pdma0 bus id
		uQosDMAID[1]= 0x3F3 ; //DMC1 pdma1 bus id
		uQosDMAID[2]= 0x3B ;  //DMC1 mdma  bus id
		uSrcAddr  = 0x42000000;		
		}

		
	// need to enable Out of order scheduling for using QoS func. & setting defualt queue count
	DMC_SetEnhancedUtility(0xf, 1, OUT_ORDER_EN); // Queue count 

	// QoS setting clear.   ch number, count, en/disable, mask, id
	for (uCh=0;uCh<16;uCh++)
		DMC_SetQosConfig(uCh, 0xfff, QOS_DIS, 0xff, 0xff); // max. count 0xfff
	// disable all power down mode 
	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);

//	INTC_Init();
//// Write data source random, destination zero
	for (i=uSrcAddr; i<uSrcAddr+uTransSize; i+=4 )	
		Outp32( i, rand() );
	for (i=uSrcAddr+uStepsize; i<uSrcAddr+uStepsize+uTransSize; i+=4 )	
		Outp32( i, 0 );
	for (i=uSrcAddr+uStepsize*2; i<uSrcAddr+uStepsize*2+uTransSize; i+=4 )	
		Outp32( i, rand() );
	for (i=uSrcAddr+uStepsize*3; i<uSrcAddr+uStepsize*3+uTransSize; i+=4 )	
		Outp32( i, 0 );
	for (i=uSrcAddr+uStepsize*4; i<uSrcAddr+uStepsize*4+uTransSize; i+=4 )	
		Outp32( i, rand() );
	for (i=uSrcAddr+uStepsize*5; i<uSrcAddr+uStepsize*5+uTransSize; i+=4 )	
		Outp32( i, 0 );
/////////////////////////////////////

	INTC_SetVectAddr(NUM_MDMA, Isr_MDmaDone);
	INTC_SetVectAddr(NUM_PDMA0, Isr_PDma0Done);
	INTC_SetVectAddr(NUM_PDMA1, Isr_PDma1Done);

	INTC_Enable(NUM_PDMA0);
	INTC_Enable(NUM_PDMA1);
	INTC_Enable(NUM_MDMA);

	for (uCh=0;uCh<14;uCh++)	// channel set 0,1,2 ~ 13,14,15
	{						
		QosCount[0] = 0+(uCh*315);
		QosCount[1] = 4095-(uCh*315);
		QosCount[2] = 104 -(uCh*8);
			    
		if(uQosEN == 0)
			{
		DMC_SetQosConfig(uCh, QosCount[0], QOS_EN, uQosDMAMask[0], uQosDMAID[0]); // ARID of PDMA0	
		DMC_SetQosConfig(uCh+1, QosCount[1], QOS_EN, uQosDMAMask[1], uQosDMAID[1]); // ARID of PDMA1	 
		DMC_SetQosConfig(uCh+2, QosCount[2], QOS_EN, uQosDMAMask[2], uQosDMAID[2]); // ARID of MDMA	 
			}
#if 0   //MDMA Test 
		DMC_SetQosConfig(uCh, QosCount[0], QOS_EN, 0x7f0f, 0x001); // ARID of MDMA
		DMC_SetQosConfig(uCh+1, QosCount[0], QOS_EN, 0x0f0f, 0x001); //ARID of MDMA	
		DMC_SetQosConfig(uCh+2, QosCount[2], QOS_EN, 0x0f0f, 0x001); // ARID of MDMA
#endif

		if (uCh>0)
		{
			DMC_SetQosConfig(uCh-1, 0x0, QOS_DIS, 0x0, 0x0); // clear previous channel
		}
		
//		for (uT=1;uT<33;uT*=2) // change transfer size
			uT = 10;
			bDmaDone[0] = false;
			bDmaDone[1] = false;
			bDmaDone[2] = false;


			SetNonSecure();
			
			DMA_SetCh(DMA_00, &oDmac0); // PDMA0 ch 0
			DMA_InitCh(BYTE, SOFTWARE, DMA_M2P, SINGLE, &oDmac0);  
			StartTimer(0);
			DMA_StartCh(uSrcAddr, uSrcAddr+uStepsize, uTransSize, &oDmac0);	// transfer size by word

			DMA_SetCh(DMA_10, &oDmac1); // PDMA1 ch 0
			DMA_InitCh(BYTE, SOFTWARE, DMA_M2P, SINGLE, &oDmac1);  
			StartTimer(1);
			DMA_StartCh(uSrcAddr+uStepsize*2, uSrcAddr+uStepsize*3, uTransSize, &oDmac1);	// transfer size by word

			DMA_SetCh(DMA_20, &oDmac2); // MDMA ch 0
//			DMA_InitCh(WORD, SOFTWARE, DMA_M2M, BURST4, &oDmac2);  
			DMA_InitCh(DWORD, SOFTWARE, DMA_M2M, BURST16, &oDmac2);  
			StartTimer(2);
			DMA_StartCh(uSrcAddr+uStepsize*4, uSrcAddr+uStepsize*5, uTransSize, &oDmac2);	// transfer size by word
//			DMA_StartCh(uSrcAddr+Mem_Size*4/16, uSrcAddr+Mem_Size*5/16, uTransSize/2, &oDmac2);	// transfer size by word

			// making outstanding procedure with ARM
			_burst2_memcpy((u32 *)(uSrcAddr+uStepsize*6), (u32 *)(uSrcAddr+uStepsize*7), uTransSize);

			// end DMA
			while( !(bDmaDone[0] & bDmaDone[1] & bDmaDone[2]) );

			DMA_StopCh(&oDmac0);
		if ( Compare(uSrcAddr+uStepsize, uSrcAddr+uStepsize*1, uTransSize/4) != true)
				Disp(" ...followed item copy failed...\n");
			Disp(" PDMA0 - Transfer size : 0x%7x, Elapsed time : %8d usec, Performance : %6.2f MByte/sec, QoS count : %3d \n"
			  , uTransSize, uElapTime[0], (float)(uTransSize*4/(float)uElapTime[0]),  QosCount[0]);

			DMA_StopCh(&oDmac1);
		if ( Compare(uSrcAddr+uStepsize*2, uSrcAddr+uStepsize*3, uTransSize/4) != true)
				Disp(" ...followed item copy failed...\n");
			Disp(" PDMA1 - Transfer size : 0x%7x, Elapsed time : %8d usec, Performance : %6.2f MByte/sec, QoS count : %3d \n"
			  , uTransSize, uElapTime[1], (float)(uTransSize*4/(float)uElapTime[1]),  QosCount[1]);


			DMA_StopCh(&oDmac2);
		if ( Compare(uSrcAddr+uStepsize*4, uSrcAddr+uStepsize*5, uTransSize/4) != true)
			Disp(" ...followed item copy failed...\n");
			Disp(" MDMA  - Transfer size : 0x%7x, Elapsed time : %8d usec, Performance : %6.2f MByte/sec, QoS count : %3d \n"
			  , uTransSize, uElapTime[2], (float)(uTransSize*4/(float)uElapTime[2]),  QosCount[2]);
			Disp(" QOS CH : %d\n",uCh);
		
	}
	
	// Exit function
	// QoS setting clear.   ch number, count, en/disable, mask, id
	for (uCh=0;uCh<16;uCh++)
		DMC_SetQosConfig(uCh, 0xfff, QOS_DIS, 0x0, 0x0); // max. count 0xfff
//	LCDC_Stop();
#endif // USE_DMA	
}


/// for memcpy perf test
void DMCT_MemcpyPerfTest(void)
{
	u32 uTimeCount;
	u32 uPDM,uOO,uLi,uBl,uMc,uTr;
    u8  Addr_map;

	///////// user setting //////////
	u32 uTransBase = 0x80000; // 500KB
	u32 uSrcAddr = DRAM_BaseAddress+0x00100000;
	u32 uDstAddr = DRAM_BaseAddress+0x00800000;
//	u32 uDstAddr = DRAM_BaseAddress+Mem_Size/4;
	/////////////////////////////
	
	// ** need to Timing parameter optimization
	// ** need to QoS setting
	// ** need to Cache usage
	Disp(" ID numbers order : Power down mode, Out of order, Address mapping, Burst mode, Memcpy type, Transfer size\n");

	Addr_map = (uDmcCh==0)? 2: 1;// Address mapping method limit
		
	// ** check performance for each related functions and power down modes
	for (uPDM=0;uPDM<1;uPDM++) // each power down mode 
	{
		switch(uPDM) // selecting mode
		{
			case 0 : // * normal mode - all power down mode disable
				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				break;
			case 1 : // * clock control mode
				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_EN);
				break;
			case 2 : // * close page policy precharge mode
				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				DMC_SetPrechPolicy(0xff, 0xff); // close page policy for all bank ,u8 ucCHIP_SELPrechPolicy, u8 ucChip1PrechPolicy
				break;
			case 3 : // * open page policy & timeout precharge mode
				DMC_SetPrechPolicy(0x00, 0x00); // open page policy for all bank ,u8 ucCHIP_SELPrechPolicy, u8 ucChip1PrechPolicy
				DMC_SetPowerDownCounter(3,0,0);	// set only precharge count & check count
				DMC_SetPowerDownModes(FORCE_PRE_EN, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, FORCE_PRE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				break;
			case 4 : // * dynamic power down mode
				DMC_SetPowerDownCounter(0,1,0);	// set precharge count & Dynamic power down count
				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				break;
			case 5 : // * dynamic self refresh mode
				DMC_SetPowerDownCounter(0,0,2);	// set precharge, Dynamic power down & Dynamic self refresh count
				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_EN, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				break;
			case 6 : // * applied all power down mode
				DMC_SetPowerDownCounter(1,2,3);	// set precharge, Dynamic power down & Dynamic self refresh count
				DMC_SetPowerDownModes(FORCE_PRE_EN, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_EN, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_EN);
				break;
		}

		// measure time count for setting each function 
		for (uOO=1;uOO<2;uOO++) // out of order schedule. 0:disable, 1:enable
			for (uLi=0;uLi<Addr_map+1;uLi+=Addr_map) // address mapping method. 0:linear, 1:interleaved,2:Mixed1
				for (uBl=2;uBl<3;uBl++) // burst option. 1:bl2, 2:bl4, 3:bl8, 4:bl16
					for (uMc=0;uMc<1;uMc++) // memcpy type. 0:b2mem, 1:b4mem, 2:b8mem, 3:memcpy, 4:d4mem, 5:d8mem, 6:d16mem, 7:d64mem, 8:d256mem
						for (uTr=1;uTr<2;uTr*=2) // transfer size. 500KB * n. n=1,2,4,8,16	
						{   
							if(uDmcCh==2)
							{
							uTimeCount = DMCT_MeasureTimeCount(1, uMc, uBl, uLi, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
							Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
						    }
                            else
							{
							uTimeCount = DMCT_MeasureTimeCount(0, uMc, uBl, uLi, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
							Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
							}
						}
							
	}
	
	// disable all power down mode 
	Disp("\n Test end");
	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
}


/// for memcpy perf test
void DMCT_MemcpyPerfTest2(void)
{
	u32 uTimeCount;
	u32 uPDM,uOO,uLi,uBl,uMc,uTr;
    u8  Addr_map;

	///////// user setting //////////
	u32 uTransBase = 0x80000; // 500KB
	u32 uSrcAddr = DRAM_BaseAddress+0x01100000;
	u32 uDstAddr = DRAM_BaseAddress+0x01800000;
//	u32 uDstAddr = DRAM_BaseAddress+Mem_Size/4;
	/////////////////////////////
	
	// ** need to Timing parameter optimization
	// ** need to QoS setting
	// ** need to Cache usage
	do
	{
	Disp(" Select Memory Type 1::CP H type , 2::MCP A or C Type \n");
	Addr_map = UART_GetIntNum();
	}
	while(Addr_map<1 | Addr_map>2);

	
//	Addr_map = (uDmcCh==0)? 2: 1;// Address mapping method limit.
	Disp(" ID numbers order : Power down mode, Out of order, Address mapping, Burst mode, Memcpy type, Transfer size\n");
		
	// ** check performance for each related functions and power down modes
	for (uPDM=0;uPDM<1;uPDM++) // each power down mode 
	{
		switch(uPDM) // selecting mode
		{
			case 0 : // * normal mode - all power down mode disable
//				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				break;
			case 1 : // * clock control mode
				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_EN);
				break;
			case 2 : // * close page policy precharge mode
				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				DMC_SetPrechPolicy(0xff, 0xff); // close page policy for all bank ,u8 ucCHIP_SELPrechPolicy, u8 ucChip1PrechPolicy
				break;
			case 3 : // * open page policy & timeout precharge mode
				DMC_SetPrechPolicy(0x00, 0x00); // open page policy for all bank ,u8 ucCHIP_SELPrechPolicy, u8 ucChip1PrechPolicy
				DMC_SetPowerDownCounter(3,0,0);	// set only precharge count & check count
				DMC_SetPowerDownModes(FORCE_PRE_EN, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, FORCE_PRE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				break;
			case 4 : // * dynamic power down mode
				DMC_SetPowerDownCounter(0,1,0);	// set precharge count & Dynamic power down count
				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				break;
			case 5 : // * dynamic self refresh mode
				DMC_SetPowerDownCounter(0,0,2);	// set precharge, Dynamic power down & Dynamic self refresh count
				DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_EN, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
				break;
			case 6 : // * applied all power down mode
				DMC_SetPowerDownCounter(1,2,3);	// set precharge, Dynamic power down & Dynamic self refresh count
				DMC_SetPowerDownModes(FORCE_PRE_EN, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_EN, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_EN);
				break;
		}

		// measure time count for setting each function 
		for (uOO=1;uOO<2;uOO++) // out of order schedule. 0:disable, 1:enable
			{
			if(uDmcCh==0)
				{
				for (uLi=0;uLi<Addr_map+1;uLi+=Addr_map) // address mapping method. 0:linear, 1:interleaved,2:Mixed1
					for (uBl=1;uBl<5;uBl++) // burst option. 1:bl2, 2:bl4, 3:bl8, 4:bl16
						for (uMc=0;uMc<7;uMc++) // memcpy type. 0:b2mem, 1:b4mem, 2:b8mem, 3:memcpy, 4:d4mem, 5:d8mem, 6:d16mem, 7:d64mem, 8:d256mem
							for (uTr=1;uTr<17;uTr*=2) // transfer size. 500KB * n. n=1,2,4,8,16	
							{   
								uTimeCount = DMCT_MeasureTimeCount(0, uMc, uBl, uLi, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
								Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
							}
						}
			else
				{
				for (uLi=0;uLi<2;uLi+=1) // address mapping method. 0:linear, 1:interleaved,2:Mixed1			
					for (uBl=1;uBl<5;uBl++) // burst option. 1:bl2, 2:bl4, 3:bl8, 4:bl16
						for (uMc=0;uMc<7;uMc++) // memcpy type. 0:b2mem, 1:b4mem, 2:b8mem, 3:memcpy, 4:d4mem, 5:d8mem, 6:d16mem, 7:d64mem, 8:d256mem
							for (uTr=1;uTr<17;uTr*=2) // transfer size. 500KB * n. n=1,2,4,8,16 
								{
								uTimeCount = DMCT_MeasureTimeCount(0, uMc, uBl, uLi, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
								Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
								}
							}
						}
				}
	// disable all power down mode 
	Disp("\n Test end");
//	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
}



void DMCT_MemcpyManualTest(void)
{
	u32 uTimeCount, temp, uCS;
	u32 uPDM,uOO,uLi,uBl,uMc,uTr;
	u32 u_PDM,u_OO,u_Li,u_Bl,u_Mc,u_Tr;
    u8  Addr_map;
	///////// default setting //////////
	u32 uTransBase = 0x80000; // 500KB
	u32 uSrcAddr = DRAM_BaseAddress+0x00100000;
	u32 uDstAddr = DRAM_BaseAddress+0x00800000;
	///////////////////////////////

	uPDM = 0;
	UART_Printf(" Enter source address (0x%08x): ", uSrcAddr);
	temp = UART_GetIntNum();
	uSrcAddr = (temp != -1) ? temp: uSrcAddr;
	UART_Printf(" Enter destination address (0x%08x): ", uDstAddr);
	temp = UART_GetIntNum();
	uDstAddr = (temp != -1) ? temp: uDstAddr;
	UART_Printf(" Enter the data size (0x%08x): ", uTransBase);
	temp = UART_GetIntNum();
	uTransBase = (temp != -1) ? temp: uTransBase;

#if 0 

	UART_Printf("\n Enter out of order(0~1) : ");
	uOO = UART_GetIntNum();
	UART_Printf("\n Enter Address mapping method(0~2) : ");
	uLi = UART_GetIntNum();
	UART_Printf("\n Enter Burst option(1:bl2, 2:bl4, 3:bl8, 4:bl16) : ");
	uBl = UART_GetIntNum();
	UART_Printf("\n 0:b2mem, 1:b4mem, 2:b8mem, 3:memcpy, 4:d4mem, 5:d8mem, 6:d16mem, 7:d64mem, 8:d256mem \n Enter memcpy type : ");
	uMc = UART_GetIntNum();
	UART_Printf("\n Enter transfer size(1,2,4,8,16) : ");
	uTr = UART_GetIntNum();


	// * normal mode - all power down mode disable
//	DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);

	Disp(" ID : Power down mode, Out of order, Add mapping, Burst mode, Memcpy type, Transfer size\n");

	Addr_map = (uDmcCh==2)? 3: 2;// Address mapping method limit.
	
	// memcpy test
	// measure time count for setting each function 
	for (uOO;uOO<2;uOO++) // out of order schedule. 0:disable, 1:enable
		for (uLi;uLi<Addr_map;uLi+=Addr_map-1) // address mapping method. 0:linear, 1:interleaved, 2:MixedMode
			for (uBl;uBl<5;uBl++) // burst option. 1:bl2, 2:bl4, 3:bl8, 4:bl16
				for (uMc;uMc<9;uMc++) // // memcpy type. 0:b2mem, 1:b4mem, 2:b8mem, 3:memcpy, 4:d4mem, 5:d8mem, 6:d16mem, 7:d64mem, 8:d256mem
					for (uTr;uTr<17;uTr*=2) // transfer size. 500KB * n. n=1,2,4,8,16	
					{
					if(uDmcCh==2)
						{
						uTimeCount = DMCT_MeasureTimeCount(1, uMc, uBl, uLi, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
						Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
					    }
                        else
						{
						uTimeCount = DMCT_MeasureTimeCount(0, uMc, uBl, uLi, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
						Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
						}
					}
#endif

#if 1	// normal mode memcpy test
    // ** check performance for each related functions and power down modes
	uPDM = 0;
	uOO = 1;
	uLi = 0;
	uMc = 4;
	uTr = 1;
	uBl=2;

		uTimeCount = DMCT_MeasureTimeCount(CHIP_SEL, uMc, uBl, uLi, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
		Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));

	if(uDmcCh==2)
        	{
		uTimeCount = DMCT_MeasureTimeCount(CHIP_SEL, uMc, uBl, uLi+2, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
		Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi+2,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
        	}
	else
	     	{
		uTimeCount = DMCT_MeasureTimeCount(CHIP_SEL, uMc, uBl, uLi+1, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
		Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi+1,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
        	}

#endif



#if 0	
	for (uBl=1;uBl<5;uBl++) // burst option. 1:bl2, 2:bl4, 3:bl8, 4:bl16
	{
		Disp(" ARM memcpy test\n");
		UART_GetIntNum();
		uTimeCount = DMCT_MeasureTimeCount(CHIP_SEL, uMc, uBl, uLi, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
		Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
	}

	uMc = 3;
	for (uBl=1;uBl<5;uBl++) // burst option. 1:bl2, 2:bl4, 3:bl8, 4:bl16
	{
		Disp(" DMA memcpy test\n");
		UART_GetIntNum();
		uTimeCount = DMCT_MeasureTimeCount(CHIP_SEL, uMc, uBl, uLi, uOO, uTransBase*uTr, uDstAddr, uSrcAddr);
		Disp(" ID: %1d %1d %1d %1d %1d %2d, Elapsed time : %8d usec, Performance : %6.2f MByte/sec \n", uPDM,uOO,uLi,uBl,uMc,uTr,uTimeCount, (float)(uTransBase*uTr/(float)uTimeCount));
	}
#endif


}
void DMCT_SetQosParameter(void)
{
	DMC_SetQosParameter(v210_DMC0, eFIMD_W0_TM,true,0x20, true, 0xf0);

}




void DMCT_Ebi2DmcAgingTest(void)
{
	u32 temp;
	///////// default setting //////////
	u32 uTransSize = 0x80000; // 500KB
	u32 uSrcAddr = DRAM_BaseAddress+0x80000;
//	u32 uSrcAddr = 0x80080000;
	u32 uDstAddr = DRAM_BaseAddress+Mem_Size/4;
	///////////////////////////////
	
	UART_Printf(" Enter source address (0x%08x): ", uSrcAddr);
	temp = UART_GetIntNum();
	uSrcAddr = (temp != -1) ? temp: uSrcAddr;
	UART_Printf(" Enter destination address (0x%08x): ", uDstAddr);
	temp = UART_GetIntNum();
	uDstAddr = (temp != -1) ? temp: uDstAddr;
	UART_Printf(" Enter the data size (0x%08x): ", uTransSize);
	temp = UART_GetIntNum();
	uTransSize = (temp != -1) ? temp: uTransSize;
	
	while(UART_GetKey()!=0x1B) // ESC key
	{
		Disp(" EBI(nRCS0) to DMC memcpy Test start!! Press ESC key to exit. Then it will be exited at test end point. \n");
		_burst2_memcpy((u32 *)(uDstAddr), (u32 *)(uSrcAddr), uTransSize);
		_burst4_memcpy((u32 *)(uDstAddr), (u32 *)(uSrcAddr), uTransSize);
		_burst8_memcpy((u32 *)(uDstAddr), (u32 *)(uSrcAddr), uTransSize);
		dma_memcpy(uDstAddr, uSrcAddr, uTransSize, BURST4);
		dma_memcpy(uDstAddr, uSrcAddr, uTransSize, BURST8);
	}
}


static volatile u32 uCacheData1[32];
static volatile u32 uCacheData2[32];
//volatile u32 uCacheData3[32];
//volatile u32 uCacheData4[32];

void DMCT_CacheManualTest(void)
{
	u32 temp, i;
	volatile u32 uCacheData5[32];
	volatile u32 uCacheData6[32];
	
	///////// default setting //////////
	u32 uTransBase = 0x100; 
	u32 uSrcAddr = _SMC_BaseAddress+0x00100000;
	u32 uDstAddr = _SMC_BaseAddress+0x00120000;
	///////////////////////////////

	UART_Printf(" Enter source address (0x%08x): ", uSrcAddr);
	temp = UART_GetIntNum();
	uSrcAddr = (temp != -1) ? temp: uSrcAddr;
	UART_Printf(" Enter destination address (0x%08x): ", uDstAddr);
	temp = UART_GetIntNum();
	uDstAddr = (temp != -1) ? temp: uDstAddr;
	UART_Printf(" Enter the data size (0x%08x): ", uTransBase);
	temp = UART_GetIntNum();
	uTransBase = (temp != -1) ? temp: uTransBase;

	for (i=0;i<uTransBase;i++)
		*(u32 *)(uSrcAddr+4+i*8) = i+0x8000;

	for (i=0;i<uTransBase;i++)
		*(u32 *)(uDstAddr+12+i*12) = i+0x4000;

	for (i=0;i<15;i++)
	{
		uCacheData1[i*2] = i+0x100;
		uCacheData2[i*2+1] = i+0x200;
//		uCacheData3[i*2] = i+0x300;
//		uCacheData4[i*2+1] = i+0x400;
//		uCacheData5[i*2] = i+0x500;
//		uCacheData6[i*2+1] = i+0x600;
	}
	
}


/////////////////////////////////////////////////////////////////////////////////
///////////////////////////// Read / Write library //////////////////////////////
/////////////////////////////////////////////////////////////////////////////////

static u32 BurstOfSequence(u32 sa, u32 ea)
{
__asm
	{
	mov     r11, sa //	changed from r0 to sa for ARM V6
	mov     r9, #1

loop1:
	add     r2, r9 ,#0x10001
	add     r3, r2 ,#0x10001
	add     r4, r3 ,#0x10001
	add     r5, r4 ,#0x10001

	add     r6, r5 ,#0x10001
	add     r7, r6 ,#0x10001
	add     r8, r7 ,#0x10001
	add     r9, r8 ,#0x10001

	stmia   r11, {r2-r9}

	add     r11, r11, #0x20
	cmp     r11, ea
	bne     loop1

	mov     r11, sa
	mov     r10, #1

loop2:
	ldmia   r11, {r2-r9}

	add	    r10, r10, #0x10001
	cmp     r2, r10
	bne     err
	add     r10, r10, #0x10001
	cmp     r3, r10
	bne     err
	add     r10, r10, #0x10001
	cmp     r4, r10
	bne     err
	add     r10, r10, #0x10001
	cmp     r5, r10
	bne     err
	add     r10, r10, #0x10001
	cmp     r6, r10
	bne     err
	add     r10, r10, #0x10001
	cmp     r7, r10
	bne     err
	add     r10, r10, #0x10001
	cmp     r8, r10
	bne     err
	add     r10, r10, #0x10001
	cmp     r9, r10
	bne     err

	add     r11, r11, #0x20
	cmp     r11, ea
	bne     loop2
	b       ok
	}
err:
	return 0;
ok:
	return 1;
}


static void BurstOfSequenceWoVerifying(u32 sa, u32 ea)
{
__asm
	{
	mov     r11, sa //	changed from r0 to sa for ARM V6
	mov     r9, #1

loop1:
	add     r2, r9 ,#0x20002
	add     r3, r2 ,#0x20002
	add     r4, r3 ,#0x20002
	add     r5, r4 ,#0x20002

	add     r6, r5 ,#0x20002
	add     r7, r6 ,#0x20002
	add     r8, r7 ,#0x20002
	add     r9, r8 ,#0x20002

	stmia   r11, {r2-r9}

	add     r11, r11, #0x20
	cmp     r11, ea
	bne     loop1

	mov     r11, sa
	mov     r10, #0

loop2:
	ldmia   r11, {r2-r9}

	add     r11, r11, #0x20
	cmp     r11, ea
	bne     loop2
	}
}


static u32 BurstOfZigZag(u32 sa, u32 ea, u32 wr)
{
__asm
	{
	mov     r11, sa //	changed from r0 to sa for ARM V6

loop1:
	mov     r4, wr
	mvn     r5, wr
	mov     r6, wr
	mvn     r7, wr
	mov     r8, wr
	mvn     r9, wr
	mov     r10, wr
	mvn     r12, wr

	stmia   r11, {r4-r10,r12}

	add     r11, r11, #0x20
	cmp     r11, ea
	bne     loop1

	mov     r11, sa

loop2:
	ldmia   r11, {r4-r10,r12}

	cmp     r4, wr
	bne     err
	cmp     r6, wr
	bne     err
	cmp     r8, wr
	bne     err
	cmp     r10, wr
	bne     err

	mvn     wr, wr

	cmp     r5, wr
	bne     err
	cmp     r7, wr
	bne     err
	cmp     r9, wr
	bne     err
	cmp     r12, wr
	bne     err

	mvn     wr, wr

	add     r11, r11, #0x20
	cmp     r11, ea
	bne     loop2
	b       ok
	}

err:
	return 0;
ok:
	return 1;
}


void BurstOfZigZagWoVerifying(u32 sa, u32 ea, u32 wr)
{
__asm
	{
	mov     r11, sa //	changed from r0 to sa for ARM V6

loop1:
	mov     r3, wr
	mvn     r4, wr
	mov     r5, wr
	mvn     r6, wr
	mov     r7, wr
	mvn     r8, wr
	mov     r9, wr
	mvn     r10, wr

	stmia   r11, {r3-r10}

	add     r11, r11, #0x20
	cmp     r11, ea
	bne     loop1

	mov     r11, sa

loop2:
	ldmia   r11, {r3-r10}

	add     r11, r11, #0x20
	cmp     r11, ea
	bne     loop2
	}
}

static u32 March(u32 sa, u32 ea, u32 wr)
{
	u32 rd, i;
	u32 space;

	space = ea - sa;

	// Increase
	//----------

	for (i=0; i<space; i+=4) {
		*(u32 *)(sa+i) = wr;
		wr = ~wr;
	}

	for (i=0; i<space; i+=4) {
		if ((rd = *(u32 *)(sa+i)) != wr) {
			Disp(" %08x: %08x -> %08x\n", sa+i, wr, rd);
			return 0;
		}
		wr = ~wr;
	}

	// Decrease
	//-----------

	for (i=4; i<=space; i+=4) {
		*(u32 *)(sa+space-i) = wr;
		wr = ~wr;
	}

	for (i=4; i<=space; i+=4) {
		if ((rd = *(u32 *)(sa+space-i)) != wr) {
			Disp(" %08x: %08x -> %08x\n", sa+i, wr, rd);
			return 0;
			
		}
		wr = ~wr;
	}

	return 1;

}


static void TestSingle(void)
{
	u32 uErrCnt, wr, rd, addr;

	wr = 0x00000001;
	uErrCnt = 0;
	for (addr=uStAddr1; addr<uLimitAddr1; addr+=4)
	{
		*(volatile u32*)addr = wr;
		rd = *(volatile u32*)addr;

		if (wr!= rd) {
			Disp (" %08x = %08x -> %08x \n", addr, wr, rd);
			if (++uErrCnt >= 20) {
				Disp (" Too many errors.\n");
				break;
			}
		}

		wr = wr+0x20002;
	}

	Disp(" %d errors\n\n", uErrCnt);
}

static void TestBurst(void)
{
	u32 ret;
	u32 sa = uStAddr1;
	u32 ea = uLimitAddr1;

	Assert(sa%0x20 == 0);

	ret = BurstOfSequence(sa, ea);
	ret ? Disp(" 1a: pass\n"): Disp(" 1a: fail\n");

	ret = BurstOfZigZag(sa, ea, 0x00000000);
	ret ? Disp(" 2a: pass\n"): Disp(" 2a: fail\n");

	ret = BurstOfZigZag(sa, ea, 0x0000ffff);
	ret ? Disp(" 2b: pass\n"): Disp(" 2b: fail\n");

	ret = BurstOfZigZag(sa, ea, 0x00ff00ff);
	ret ? Disp(" 2c: pass\n"): Disp(" 2c: fail\n");

	ret = BurstOfZigZag(sa, ea, 0x33333333);
	ret ? Disp(" 2d: pass\n"): Disp(" 2d: fail\n");

	ret = BurstOfZigZag(sa, ea, 0x55555555);
	ret ? Disp(" 2e: pass\n"): Disp(" 2e: fail\n");

	Disp("\n");
}

static void TestMarch(void)
{
	u32 sa = uStAddr1;
	u32 ea = uLimitAddr1;
	u32 ret;

	ret = March(sa, ea, 0x00000000);
	ret ? Disp(" 3a: pass\n"): Disp(" 3a: fail\n");

	ret = March(sa, ea, 0x0000ffff);
	ret ? Disp(" 3b: pass\n"): Disp(" 3b: fail\n");

	ret = March(sa, ea, 0x00ff00ff);
	ret ? Disp(" 3c: pass\n"): Disp(" 3c: fail\n");

	ret = March(sa, ea, 0x33333333);
	ret ? Disp(" 3d: pass\n"): Disp(" 3d: fail\n");

	ret = March(sa, ea, 0x55555555);
	ret ? Disp(" 3e: pass\n"): Disp(" 3e: fail\n");
}


/////////////////////////////////////////////////////////////////////////////////
///////////////////////////// Memcpy test functions /////////////////////////////
/////////////////////////////////////////////////////////////////////////////////

#define DP_FF						(0)
#define DP_00						(1)
#define DP_5A						(2)
#define DP_A5						(3)
// DP_[31:24]_[23:0]
#define DP_RAN_RAN					(4)
#define DP_RAN_00					(5)
#define DP_00_RAN					(6)
#define DP_RAN_FF					(10)

#define DP_PRE_RAN					(7)
#define DP_RAN_PRE					(8)
#define DP_PRE_PRE					(9)

void MemoryTest_4burstWrite_4burstRead_Aging(void)
{
	volatile u32 *Pt_Base, *Pt;
	volatile u32 *Data_Base, *Data, *Read_Data, *Read_Data_Base;
	u32 Error_Cnt=0, Error_Cnt_Low=0, Error_Addr_08=0, Error_Addr_0C=0;
	u32 i, j, k, m, test_size, random_offset;
	u32 uLoopCount, uPosition;
	u8 aDataPattern[][4] = 	{ 
								{DP_RAN_RAN,	DP_RAN_RAN,		DP_RAN_RAN,		DP_RAN_RAN},

								{DP_FF, 		DP_FF, 			DP_FF, 			DP_FF},
								{DP_00,			DP_00,			DP_00,			DP_00},
								{DP_FF,			DP_00,			DP_FF,			DP_00},
								{DP_00,			DP_FF,			DP_00,			DP_FF},

								{DP_FF,			DP_FF,			DP_00,			DP_00},
								{DP_00,			DP_FF,			DP_FF,			DP_00},
								{DP_00,			DP_00,			DP_FF,			DP_FF},
								
								{DP_5A,			DP_5A,			DP_5A,			DP_5A},
								{DP_A5,			DP_A5,			DP_A5,			DP_A5},
								{DP_5A,			DP_A5,			DP_5A,			DP_A5},
								{DP_A5,			DP_5A,			DP_A5,			DP_5A},

								{DP_A5,			DP_A5,			DP_5A,			DP_5A},
								{DP_5A,			DP_A5,			DP_A5,			DP_5A},
								{DP_5A,			DP_5A,			DP_A5,			DP_A5},								
								
							};
	
	Disp("Input the Random Offset Value : ");
//	random_offset = UART_GetIntNum();
	random_offset = 0;

	uLoopCount = (u32)sizeof(aDataPattern)/(u32)sizeof(aDataPattern[0]);
	
	test_size = 0x1000;	
	Pt_Base=(volatile u32 *)(DRAM_BaseAddress + 0x1000000);
	Data_Base = (u32 *)malloc(test_size);
	Read_Data_Base = (u32 *)malloc(test_size);
	
	Data = Data_Base;
	Read_Data = Read_Data_Base;
	j=0;
	
	Disp("\nMemory Test(0x%08x-0x%08x)\n",Pt_Base,Pt_Base+test_size/4);//??


	for(k=0 ; k<uLoopCount ; k++)
//	for(k=0 ; k<1 ; k++)

	{
		Error_Cnt = 0;
		Error_Cnt_Low = 0;
		Error_Addr_08 = 0;
		Error_Addr_0C = 0;
		
		for(m=0 ; m<4 ; m++)
//		for(m=0 ; m<1 ; m++)

		{
			if(aDataPattern[k][m] == DP_FF)
				Disp("0xFFFFFFFF, ");
			else if(aDataPattern[k][m] == DP_00)
				Disp("0x00000000, ");
			else if(aDataPattern[k][m] == DP_5A)
				Disp("0x5A5A5A5A, ");
			else if(aDataPattern[k][m] == DP_A5)
				Disp("0xA5A5A5A5, ");
			else if(aDataPattern[k][m] == DP_RAN_RAN)
				Disp("DP_RAN_RAN, ");
			else if(aDataPattern[k][m] == DP_RAN_00)
				Disp("DP_RAN_00, ");
			else if(aDataPattern[k][m] == DP_RAN_FF)
				Disp("DP_RAN_FF, ");
			else if(aDataPattern[k][m] == DP_00_RAN)
				Disp("DP_00_RAN, ");
			else if(aDataPattern[k][m] == DP_RAN_PRE)
				Disp("DP_RAN_PRE, ");
			else if(aDataPattern[k][m] == DP_PRE_RAN)
				Disp("DP_PRE_RAN, ");
			else if(aDataPattern[k][m] == DP_PRE_PRE)
				Disp("DP_PRE_PRE, ");
			else
				Disp("others, ");
		}
		Disp("\b\b      ");
		
		for(j=0 ; j<300 ; j++)		// count * 1M byte test.
		{
			Disp(".");
			srand(random_offset+j);
			
			Data = Data_Base;
			Pt_Base=(u32 *)(DRAM_BaseAddress + 0x1000000);
			Pt= Pt_Base;
			
			for(i=0 ; i<test_size/4 ; i++)
			{
				uPosition = i%4;
				
				if(aDataPattern[k][uPosition] == DP_FF)
					*Data = 0xFFFFFFFF;
				else if(aDataPattern[k][uPosition] == DP_00)
					*Data = 0x0;
				else if(aDataPattern[k][uPosition] == DP_5A)
					*Data = 0x5A5A5A5A;
				else if(aDataPattern[k][uPosition] == DP_A5)
					*Data = 0xA5A5A5A5;
				else if(aDataPattern[k][uPosition] == DP_RAN_RAN)
					*Data = rand();
				else if(aDataPattern[k][uPosition] == DP_RAN_00)
					*Data = rand()&0xFF000000;
				else if(aDataPattern[k][uPosition] == DP_RAN_FF)
					*Data = rand()|0x00FFFFFF;
				else if(aDataPattern[k][uPosition] == DP_00_RAN)
					*Data = rand()&0x00FFFFFF;
				else if(aDataPattern[k][uPosition] == DP_RAN_PRE)
					*Data = (*(Data-1)&0x00FFFFFF) |  (rand()&0xFF000000);
				else if(aDataPattern[k][uPosition] == DP_PRE_RAN)
					*Data = (*(Data-1)&0xFF000000) |  (rand()&0x00FFFFFF);
				else if(aDataPattern[k][uPosition] == DP_PRE_PRE)
					*Data = *(Data-1);
				else
					*Data = 0;

				Data++;
			}


			Data = Data_Base;

			_burst4_memcpy((u32 *)Pt, (u32 *)Data, test_size);
			
			Data = Data_Base;
			Pt= Pt_Base;
			Read_Data = Read_Data_Base;

#if 1

			_burst4_memcpy((u32 *)Read_Data, (u32 *)Pt, test_size);
			for(i=0 ; i<test_size/4 ; i++)
			{
				if(*Read_Data != *Data)
				{
	//				Disp("Error [Addr : 0x%08x] [Data-2 : 0x%08x, Data-1 : 0x%08x, Data : 0x%08x, Data+1 : 0x%08x, RData : 0x%08x]\n", Pt, *(Data-2), *(Data-1), *(Data), *(Data+1), *Read_Data);
	//				Disp("[Previous Data : 0x%02x, WData : 0x%02x, RData : 0x%02x]\n", (*(Data-2)&0xff000000)>>24, (*Data&0xff000000)>>24, (*Read_Data&0xff000000)>>24);
	//				getchar();
					if((*Read_Data&0xff000000) != (*Data&0xff000000))
						Error_Cnt++;
					else
						Error_Cnt_Low++;

					if(((u32)Pt&0xF) == 0x08)
						Error_Addr_08++;
					else if(((u32)Pt&0xF) == 0x0C)
						Error_Addr_0C++; 
				}
				Read_Data++;
				Data++;
				Pt++;
			}
			
			Data = Data_Base;
			Pt= Pt_Base;
			Read_Data = Read_Data_Base;
#endif				
			for(i=0 ; i<test_size/4 ; i++)
			{
				*Data++ = 0;
				*Read_Data++ = 0;
				*Pt++ = 0;
			}		
		}

		Disp("===>Error Count : %d, %d (0x08 : %d,  0x0C : %d)\n", Error_Cnt, Error_Cnt_Low, Error_Addr_08, Error_Addr_0C);
		
	}


	Disp("end\n");
	
	free((void *)Data_Base);
		Disp("free data base\n");
	free((void *)Read_Data_Base);
		Disp("free Read_Data_Base\n");
}

void MemoryTest_xburstWriteRead_Aging(void)
{
	volatile u32 *Pt_Base, *Pt;
	volatile u32 *Data_Base, *Data, *Read_Data, *Read_Data_Base;
	u32 Error_Cnt=0;
	u32 i, j, k, test_size, random_offset, uError;

	Disp("Input the Random Offset Value : ");
//	random_offset = UART_GetIntNum();
	random_offset = 0;

	test_size = 0x10000;	
	Pt_Base=(u32 *)(DRAM_BaseAddress + 0x1000000);
	Data_Base = (u32 *)malloc(test_size);
	Read_Data_Base = (u32 *)malloc(test_size);
		
///////////////////////////////////////////////////////////////////////////////
	Data = Data_Base;
	Read_Data = Read_Data_Base;
	j=0;

	Disp("Single Write & Single Read\n ");
	uError = 0;
	for(k=0 ; k<1000 ; k++)
	{
		Disp(".");
		j++;
		Error_Cnt = 0;

		srand(random_offset+j);
		Data = Data_Base;
		Pt_Base=(u32 *)(DRAM_BaseAddress + 0x1000000);
		Pt= Pt_Base;
		
		for(i=0 ; i<test_size/4 ; i++)
		{
			*Data = rand();
			Data++;
		}

		Data = Data_Base;

		for(i=0 ; i<test_size/4 ; i++)
		{
	   		*Pt=*Data;
			Pt++;
			Data++;
		}
		Data = Data_Base;
		Pt= Pt_Base;
		
		for(i=0 ; i<test_size/4 ; i++)
		{
			if(*Pt != *Data)
			{
				Disp("Error [Addr : 0x%08x] [WData : 0x%08x, RData : 0x%08x]\n", Pt, *Data, *Pt);
				Disp("Count : %d\n", k);
				Error_Cnt++; 
				uError = 1;
				break;
			}
			Pt++;
			Data++;
		}

		Data = Data_Base;
		Pt= Pt_Base;
		Read_Data = Read_Data_Base;
		for(i=0 ; i<test_size/4 ; i++)
		{
			*Data++ = 0;
			*Read_Data++ = 0;
			*Pt++ = 0;
		}	

		if(uError == 1)
			break;
		Disp(".");
		if((k%10)==9)
			Disp("\b\b\b\b\b\b\b\b\b\b");
	}
	if(uError == 0)
		Disp(" => OK\n");

///////////////////////////////////////////////////////////////////////////////	
	Data = Data_Base;
	Read_Data = Read_Data_Base;
	j=0;

	Disp("2burst Write & 2burst Read\n");
	uError = 0;
	for(k=0 ; k<1000 ; k++)
	{
		Disp(".");
		j++;
		Error_Cnt = 0;

		srand(random_offset+j);
		Data = Data_Base;
		Pt_Base=(u32 *)(DRAM_BaseAddress + 0x1000000);
		Pt= Pt_Base;
		
		for(i=0 ; i<test_size/4 ; i++)
		{
			*Data = rand();
			Data++;
		}

		Data = Data_Base;

		_burst2_memcpy((u32 *)Pt, (u32 *)Data, test_size);
		
		Data = Data_Base;
		Pt= Pt_Base;
		Read_Data = Read_Data_Base;

		_burst2_memcpy((u32 *)Read_Data, (u32 *)Pt, test_size);
		for(i=0 ; i<test_size/4 ; i++)
		{
			if(*Read_Data != *Data)
			{
				Disp("Error [Addr : 0x%08x] [Data-2 : 0x%08x, Data-1 : 0x%08x, Data : 0x%08x, Data+1 : 0x%08x, RData : 0x%08x]\n", Pt, *(Data-2), *(Data-1), *(Data), *(Data+1), *Read_Data);
				Disp("Count : %d\n", k);
				Error_Cnt++; 
				uError = 1;
				break;
			}
			Read_Data++;
			Data++;
			Pt++;
		}

		Data = Data_Base;
		Pt= Pt_Base;
		Read_Data = Read_Data_Base;
		for(i=0 ; i<test_size/4 ; i++)
		{
			*Data++ = 0;
			*Read_Data++ = 0;
			*Pt++ = 0;
		}	

		if(uError == 1)
			break;
		Disp(".");
		if((k%10)==9)
			Disp("\b\b\b\b\b\b\b\b\b\b");
	}
	if(uError == 0)
		Disp(" => OK\n");


///////////////////////////////////////////////////////////////////////////////	

	Data = Data_Base;
	Read_Data = Read_Data_Base;
	j=0;

	Disp("4burst Write & 4burst Read\n");
	uError = 0;
	for(k=0 ; k<1000 ; k++)
	{
		Disp(".");
		j++;
		Error_Cnt = 0;

		srand(random_offset+j);
		Data = Data_Base;
		Pt_Base=(u32*)(DRAM_BaseAddress + 0x1000000);
		Pt= Pt_Base;
		
		for(i=0 ; i<test_size/4 ; i++)
		{
			*Data = rand();
			Data++;
		}

		Data = Data_Base;

		_burst4_memcpy((u32 *)Pt, (u32 *)Data, test_size);
		
		Data = Data_Base;
		Pt= Pt_Base;
		Read_Data = Read_Data_Base;

		_burst4_memcpy((u32 *)Read_Data, (u32 *)Pt, test_size);
		for(i=0 ; i<test_size/4 ; i++)
		{
			if(*Read_Data != *Data)
			{
				Disp("Error [Addr : 0x%08x] [Data-2 : 0x%08x, Data-1 : 0x%08x, Data : 0x%08x, Data+1 : 0x%08x, RData : 0x%08x]\n", Pt, *(Data-2), *(Data-1), *(Data), *(Data+1), *Read_Data);
				Disp("Count : %d\n", k);
				Error_Cnt++; 
				uError = 1;
				break;
			}
			Read_Data++;
			Data++;
			Pt++;
		}

		Data = Data_Base;
		Pt= Pt_Base;
		Read_Data = Read_Data_Base;
		for(i=0 ; i<test_size/4 ; i++)
		{
			*Data++ = 0;
			*Read_Data++ = 0;
			*Pt++ = 0;
		}	

		if(uError == 1)
			break;
		Disp(".");
		if((k%10)==9)
			Disp("\b\b\b\b\b\b\b\b\b\b");
	}
	if(uError == 0)
		Disp(" => OK\n");

///////////////////////////////////////////////////////////////////////////////	
	Data = Data_Base;
	Read_Data = Read_Data_Base;
	j=0;

	Disp("8burst Write & 8burst Read\n ");
	uError = 0;
	for(k=0 ; k<1000 ; k++)
	{
		Disp(".");
		j++;
		Error_Cnt = 0;

		srand(random_offset+j);
		Data = Data_Base;
		Pt_Base=(u32 *)(DRAM_BaseAddress + 0x1000000);
		Pt= Pt_Base;
		
		for(i=0 ; i<test_size/4 ; i++)
		{
			*Data = rand();
			Data++;
		}

		Data = Data_Base;

		_burst8_memcpy((u32 *)Pt, (u32 *)Data, test_size);
		
		Data = Data_Base;
		Pt= Pt_Base;
		Read_Data = Read_Data_Base;

		_burst8_memcpy((u32 *)Read_Data, (u32 *)Pt, test_size);
		for(i=0 ; i<test_size/4 ; i++)
		{
			if(*Read_Data != *Data)
			{
				Disp("Error [Addr : 0x%08x] [Data-2 : 0x%08x, Data-1 : 0x%08x, Data : 0x%08x, Data+1 : 0x%08x, RData : 0x%08x]\n", Pt, *(Data-2), *(Data-1), *(Data), *(Data+1), *Read_Data);
				Disp("Count : %d\n", k);
				Error_Cnt++; 
				uError = 1;
				break;
			}
			Read_Data++;
			Data++;
			Pt++;
		}

		Data = Data_Base;
		Pt= Pt_Base;
		Read_Data = Read_Data_Base;
		for(i=0 ; i<test_size/4 ; i++)
		{
			*Data++ = 0;
			*Read_Data++ = 0;
			*Pt++ = 0;
		}	

		if(uError == 1)
			break;
		Disp(".");
		if((k%10)==9)
			Disp("\b\b\b\b\b\b\b\b\b\b");
	}


	if(uError == 0)
		Disp(" => OK\n");

	free((void *)Data_Base);	
	free((void *)Read_Data_Base);
}



void DMCT_SetBaseAddress(void)
{
	s32 uSel=uDmcCh;

	if (uSel==0)
	{
		DRAM_BaseAddress=_DRAM0_BaseAddress;
		uStAddr1  = DRAM_BaseAddress+0x02100000;
		uLimitAddr1 = DRAM_BaseAddress+0x03000000;
		Mem_Size = DRAM_SIZE;
		
	}
	else if(uSel==1)
	{
		DRAM_BaseAddress=_DRAM1_BaseAddress;
		uStAddr1  = DRAM_BaseAddress+0x02100000;
		uLimitAddr1 = DRAM_BaseAddress+0x03000000;
		Mem_Size = DRAM_SIZE;
		
	}
	else if(uSel ==2)
		{
		DRAM_BaseAddress=_DRAM01_BaseAddress;
		uStAddr1  = DRAM_BaseAddress+0x02100000;
		uLimitAddr1 = DRAM_BaseAddress+0x03000000;
		Mem_Size = DRAM_SIZE;
		}
	else
		Disp("Invalid DMC Channel!\n");

}


void DMCT_Sel_CommandTest(void)
{

	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port CommandTest\n",uDmcCh);
	DMCT_CommandTest();

}

void DMCT_Sel_PowerDownModeTest(void)
{
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port PowerDownModeTest\n",uDmcCh);
	DMCT_PowerDownModeTest();
}

void DMCT_Sel_QosTest(void)
{

	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port QosTest\n",uDmcCh);
	DMCT_QosTest();
}

void DMCT_Sel_MemcpyPerfTest(void)
{
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port MemcpyPerfTest\n",uDmcCh);
	DMCT_MemcpyPerfTest();
}

void DMCT_Sel_MemcpyManualTest(void)
{
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port Memcopy Manaul Test\n",uDmcCh);
	DMCT_MemcpyManualTest();

}

void DMCT_Sel_CacheManualTest(void)
{
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port Cache manually test\n",uDmcCh);
 	DMCT_CacheManualTest();
}

void DMCT_Sel_TestBurst(void)
{

	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port Write & Verify (Burst)\n",uDmcCh);
	TestBurst();
}

void DMCT_Sel_TestSingle(void)
{
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port Write & Verify (32bit-Single)Test\n",uDmcCh);
	TestSingle();
}

void DMCT_Sel_TestMarch(void)
{

DMC_SetCh(uDmcCh);
DMCT_SetBaseAddress();
Disp("DMC%d Port Write & Verify (Single-March)\n",uDmcCh);
TestMarch();
}


void DMCT_Sel_PowerDownModeAgingTest(void)
{
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port Power down mode aging test\n",uDmcCh);
	DMCT_PowerDownModeAgingTest();

}

void DMCT_Sel_QosAgingTest(void)
{
	while(UART_GetKey()!=0x1B) // ESC key
	{
		Disp(" QoS Aging Test start!! Press ESC key to exit. Then it will be exited at test end point. \n");
		uDmcCh = (++uDmcCh)%2;
		DMC_SetCh(uDmcCh);
		DMCT_SetBaseAddress();
		Disp("DMC%d Port QoS aging test\n",uDmcCh);
		DMCT_QosTest();
	}
}

void DMCT_PhyDriving(void)
{
	u8 ucDrven, ucType;
	

	do{
	Disp("\n DMC Phy driving Test [Enable: 1], [Disable 0]:: ");
	ucDrven = UART_GetIntNum();}
	while(ucDrven<0 & ucDrven>1);

	if(ucDrven ==1)
		{	
		do{
		Disp("\n DMC Phy driving Type [Drive All to zero: 0], [Pull down all : 1] ::  ");
		ucType = UART_GetIntNum();}
		while(ucType<0 & ucType>1);
		}
	else
		ucType =1;
	
	if(uDmcCh==0)
	DMC_PhyDriving(0,ucDrven,ucType);
	else
	DMC_PhyDriving(1,ucDrven,ucType);

}
void DMCT_Qosclean(void)
{
	u8 ucCtrlShiftc, ucCtrloffsetc,ucCtrloffsetd ;
	u32 uRes;
	
	Disp("\n DMC Phy Qos Cleaning  Test  ");

	for(ucCtrloffsetd=0;ucCtrloffsetd<0x80;ucCtrloffsetd++)
			for(ucCtrloffsetc=0;ucCtrloffsetc<0x80;ucCtrloffsetc++)
					for(ucCtrlShiftc=0;ucCtrlShiftc<0x8;ucCtrlShiftc++)
						{
						DMC_QosClean(uDmcCh,ucCtrloffsetd,ucCtrloffsetc,ucCtrlShiftc);
	
						uRes = DMCT_CheckInterface(0x1000, 0x1000000);
						if (uRes == 3)
						{
//						Disp(" DMC[%d] PORT Memory interface : pass\n\n",uDmcCh);
						}
						else 
							{
						Disp("\n uDmcCh:%d, ucCtrlShiftc:%d, ucCtrloffsetc:%d,ucCtrloffsetd:%d",uDmcCh,ucCtrlShiftc, ucCtrloffsetc,ucCtrloffsetd);
						if(ucCtrlShiftc==4)break; 
						}
	}
}

void DMCT_CompareWrite(u32 uD1area, u32 uD0area, u32 uTransize)
{
	_burst8_memcpy((u32 *)(uD1area), (u32 *)(uD0area), uTransize);
}

void DMCT_Compare(u32 uD0area,u32 uD1area,u32 uCompareSize)
{

if ( Compare(uD0area, uD1area, uCompareSize/4) == true)
	Disp(" DMC0,1 Compare OK\n");
else
	Disp(" ...followed item copy failed...\n");
	
}

void __irq Isr_TQ(void)
{
	u32 uEIntPend;
		
	//uEIntPend = GPIO_GetEINT0PendingStatus();	// temp
	//UART_Printf("\nISR uEIntPend = %x",uEIntPend);

	INTC_Disable(NUM_EINT11);
	GPIO_EINT31ClrPend(4);    //EINT11 Pending Clear-EINT Group30~33=Group0
	g_Refcount = 1;           //
	UART_Printf("Occurred A Port Interrupt\n");

	INTC_ClearVectAddr(); 
	INTC_Enable(NUM_EINT11);	
}

#if 0
void DMCT_ChagneRefAging(void)
{
	u32 uRes;
	u32 uDMC0Tref,uDMC1Tref;

	g_Refcount =0;
	
	Disp("\n DMC Auto Refresh Chang Test  ");


	DMC_ChangeRef(u32 uDMC0Tref,u32 uDMC1Tref);

	INTC_SetVectAddr(NUM_EINT11, Isr_TQ);
	GPIO_SetEint11(1, Falling_Edge, eDisFLT, 0x1f); // EINT11, falling edge, enable filter, filter width
	GPIO_SetPullUpDownEach(eGPIO_H1, eGPIO_3, 0x10);
	GPIO_SetFunctionEach(eGPIO_H1, eGPIO_3, eGINT);
	GPIO_EINT30UnMask(NUM_EINT11);


	while(  )
		{
		if(g_Refcount==1)
			{
			DMC_ChangeRef(uDMC0Tref, uDMC1Tref);
			}
		uRes = DMCT_CheckInterface(0x1000, 0x1000000);
		if (uRes == 3)
			{
			//Disp(" DMC[%d] PORT Memory interface : pass\n\n",uDmcCh);
			}
		else 
		Disp("\n DMC[%d] PORT Memory interface : pass\n");
		}


	}
}

#endif

enum Mem_Clk
{
	// Power Manager
	DMC0_66 			= 66,
	DMC0_100			= 100,
	DMC0_133 			= 133,
	DMC0_166			= 166,
	DMC0_200			= 200,
	DMC1_66 			= 66,
	DMC1_100			= 100,
	DMC1_133			= 133,
	DMC1_166			= 166,
	DMC1_200			= 200
};
enum Mem_ref
{
	DMC0_66_ref			=0x202,
	DMC0_100_ref 		=0x30C,
	DMC0_133_ref 		=0x40D,
	DMC0_166_ref 		=0x50E,
	DMC0_200_ref 		=0x618
};

#if 0
void DMCT_ChangeCLKaging(void)
{
	u8 uTempEmpty,ucFre;
	u16 usArefInterval;
	u32 uDmc0div, uDmc1div;
	u8 ucRefcount;


		g_pDmcRegs = (DMC_REGS *)DMC0_BASE;
		DMC_SetAutoRefresh(true, DMC0_66_ref);

		do 
		uTempEmpty = DMC_GetQueueEmpty(0);		
		while (uTempEmpty == 0);

		//1-3)Down Clock 

		for(ucRefcount=3;ucRefcount<14;ucRefcount++)
	    DMC_DownChangeClk(eCLKDIV_ONEDRAM,ucRefcount,uDmc1Mux,u32 uDmc1Div );


	
		g_pDmcRegs = (DMC_REGS *)DMC1_BASE;
			DMC_SetAutoRefresh(true, DMC0_66_ref);
			
		
	//1-2)Queue Empty Wait 






		}
	else
		
	




}

#endif

void DMCT_Sel_Ebi2DmcAgingTest(void)
{
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port EBI(nRCS0) to DMC memcpy aging\n",uDmcCh);
	DMCT_Ebi2DmcAgingTest();
}


void Sel_MemoryTest_4burstWrite_4burstRead_Aging(void)
{
while(1)
	{
	uDmcCh =0;
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port 4 Burst W/R aging [except DRAM code region]\n",0);
	MemoryTest_4burstWrite_4burstRead_Aging();

	uDmcCh =1;
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port 4 Burst W/R aging [except DRAM code region]\n",1);
	MemoryTest_4burstWrite_4burstRead_Aging();
	}
}

void Sel_MemoryTest_xburstWriteRead_Aging(void)		
{

while(1)
	{
	uDmcCh =0;
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port x Burst W/R aging [except DRAM code region]\n",0);
	MemoryTest_xburstWriteRead_Aging();

	uDmcCh =1;
	DMC_SetCh(uDmcCh);
	DMCT_SetBaseAddress();
	Disp("DMC%d Port x Burst W/R aging [except DRAM code region]\n",1);
	MemoryTest_xburstWriteRead_Aging();
	}
}

void DMCT_Set_Port(void)
{
	s32 x;

	do {
	Disp("[Enter] Exit\n");
	Disp("[0]:: Setting to DMC0 PORT\n");
	Disp("[1]:: Setting to DMC1 PORT\n");
//	Disp("[2]:: Setting to DMC0 CS1(OneDram) PORT\n");	
	Disp("Select DMC Port ::   ");

	x = UART_GetIntNum();
		} while(x<0 || x>1);

	DMC_SetCh(x);
	uDmcCh=x;
	DMCT_SetBaseAddress();

}


void DMCT_memset(void)
{
int i;

for( i=0; i<128*1024; i++)
{
	if(i==16380)
	{
		UART_Printf("\n\n 16380 \n");
	}
	if(i%1024==0)
	UART_Printf("[%d]",i/1024);	
	memset((void*)(0x40000000 + i*1024),0x0,1024);
//if(i%1024==0)
//	UART_Printf("\n\n %d Mbyte is tested!!\n\n",i/1024);
}
}


u8 DMCT_DramInterfaceAutoTest(void)
{
	u8 uResult0;
	u8 uResult1;
	u32 uOffset = 0x01000000; //
	u32 uPortSize = 0x20000000;

	uDmcCh = 0;
	DRAM_BaseAddress = _DRAM0_BaseAddress;
	Mem_Size = DRAM0_SIZE;	
	//1) DMC0 Port Test 
		uResult0 = DMCT_CheckInterface(0x1000, uOffset);
		if (uResult0 == 3)
			{
			UART_Printf(" DMC[0] PORT Memory interface : pass\n");
			}
		else 
			{
			UART_Printf(" DMC[0] PORT Memory interface : fail\n");
 			return false;
		}

	uDmcCh = 1;
	DRAM_BaseAddress = _DRAM1_BaseAddress;
	Mem_Size = DRAM1_SIZE;	

	//2) DMC1 Port Test 
		uResult0 = DMCT_CheckInterface(0x1000, uOffset);
		if (uResult0 == 3)
			{
			UART_Printf(" DMC[1] PORT Memory interface : pass\n\n");
			}
		else 
			{
			UART_Printf(" DMC[1 PORT Memory interface : fail\n\n");
 			return false;
		}
		if(uResult0 ==3) return true;


}	

///////// Semaphore Test Case ///////////////////////////
void __irq Isr_OneDram(void)
{
	u32 uEIntPend;
	
	//uEIntPend = GPIO_GetEINT0PendingStatus();	// temp
	//UART_Printf("\nISR uEIntPend = %x",uEIntPend);

	INTC_Disable(NUM_EINT11);
	GPIO_EINT31ClrPend(4);    //EINT11 Pending Clear-EINT Group30~33=Group0
	g_APortInt = 1;           //
	UART_Printf("Occurred A Port Interrupt\n");

	INTC_ClearVectAddr(); 
	INTC_Enable(NUM_EINT11);	
}




void SignalSemaphoreStatus(ONEDRAM_SEMAPHORE eSem)
{
	u32 uStat;
	if ( eSem == ONEDRAM_A_PORT )
		uStat = 0;
	else // ONEDRAM_B_PORT
		uStat = 1;
	Outp32(ONEDRAM_B_PORT_SEMAPHORE_ADDR, uStat);
}

void WaitSemaphoreStatus(ONEDRAM_SEMAPHORE eSem)
{
	u32 uStat;
	u32 x;

	if ( eSem == ONEDRAM_A_PORT )
		uStat = 0;
	else // ONEDRAM_B_PORT
		uStat = 1;

	do {
		x= Inp32(ONEDRAM_B_PORT_SEMAPHORE_ADDR);
	} while ( x!=uStat );			
}

//// Board : SMDK , A-Port SDR X32

u8 DMCT_OnedramAportAtypeSMDK_autotest(void) 
{	
	u32 TEST_COUNT = 0x400;
	u32 uSrcAddr = CODEC_MEM_ST;
	u32 uDstAddr = CODEC_MEM_ST+(TEST_COUNT*4);
	u32 x = 0x12345678;
	u32 i;
	u32 y,uIsOK;
	

//DMC0 B-Port Read/Write Test
	DMC_SetCh(0);
	DMCT_SetBaseAddress();
	Disp("DMC0 Port Write & Verify (32bit-Single)Test\n");
	TestSingle();


//OneDram Shared Area Read/Write Test

	for(  i=0; i<TEST_COUNT; i++)
	{
		Outp32( uSrcAddr+(i*4), x );
		Outp32( ONEDRAM_B_PORT_SHARED0_BANK+(i*4), x++ );
	}
	SignalSemaphoreStatus(ONEDRAM_A_PORT);
 	Outp32(MailBox_BA,TEST_COUNT);
 	   
	WaitSemaphoreStatus(ONEDRAM_B_PORT);

#if 0
		for( i=0; i<TEST_COUNT; i++)
		{
			y = Inp32( ONEDRAM_B_PORT_SHARED1_BANK+(i*4));
			Outp32( uDstAddr+(i*4), y-1 ); 
		}
#endif

//	uIsOK = Compare8( (void*)uSrcAddr, (void*)uDstAddr, (ONEDRAM_B_PORT*4) );
	uIsOK = Compare32( (void*)uSrcAddr, (void*)ONEDRAM_B_PORT_SHARED1_BANK, TEST_COUNT/16 );	

	if( uIsOK == 0 )
		{
		Disp(" Fail to Compare\n");
		return false;
		}
	else
		{
		Disp(" MCP A-Type A_Port Test OK\n");	
		return true;
		}
}

u8 DMCT_OnedramAportAtypeSMDKInt_autotest(void) 
{	
	u32 TEST_COUNT = 0x400;
	u32 uSrcAddr = CODEC_MEM_ST;
	u32 uDstAddr = CODEC_MEM_ST+(TEST_COUNT*4);
	u32 x = 0x12345678;
	u32 i;
	u32 y,uIsOK;
     

    ///////////////////// Interrupt Setting/////////////////
	g_APortInt =0;

	INTC_SetVectAddr(NUM_EINT11, Isr_OneDram);
	GPIO_SetEint11(1, Falling_Edge, eDisFLT, 0x1f); // EINT11, falling edge, enable filter, filter width
	GPIO_SetPullUpDownEach(eGPIO_H1, eGPIO_3, 0x10);
	GPIO_SetFunctionEach(eGPIO_H1, eGPIO_3, eGINT);
	GPIO_EINT30UnMask(NUM_EINT11);
	///////////////// Interrupt End///////////////////////
	
	for(  i=0; i<TEST_COUNT; i++)
	{
		Outp32( uSrcAddr+(i*4), x );
		Outp32( ONEDRAM_B_PORT_SHARED0_BANK+(i*4), x++ );
	}
	SignalSemaphoreStatus(ONEDRAM_A_PORT);
  	Outp32(MailBox_BA,TEST_COUNT);
#if 0
	for( i=0; i<TEST_COUNT; i++)
	{
		y = Inp32( ONEDRAM_B_PORT_SHARED1_BANK+(i*4));
		Outp32( uDstAddr+(i*4), y-1 ); 
	}
#endif

	WaitSemaphoreStatus(ONEDRAM_B_PORT);

//	uIsOK = Compare8( (void*)uSrcAddr, (void*)uDstAddr, (ONEDRAM_B_PORT*4) );
	uIsOK = Compare32( (void*)ONEDRAM_B_PORT_SHARED0_BANK, (void*)ONEDRAM_B_PORT_SHARED1_BANK, TEST_COUNT/4 );	
	if( uIsOK == 0 )
		{
		Disp(" Fail to Compare\n");
		return false;
		}
	else
		{
		Disp(" MCP A-Type A_Port Test OK\n");	
		return true;
		}
}

//// Board : SMDK , A-Port DDR X16


u8 DMCT_OnedramAportCtypeSMDK_autotest(void) 
{	
		u32 TEST_COUNT = 0x400;
		u32 uSrcAddr = CODEC_MEM_ST;
		u32 uDstAddr = CODEC_MEM_ST+(TEST_COUNT*4);
		u32 x = 0x12345678; //0xAA000055;
		u32 i;
		u32 y,uIsOK;
		 
	
		///////////////////// Interrupt Setting/////////////////
		g_APortInt =0;
	
		INTC_SetVectAddr(NUM_EINT11, Isr_OneDram);
		GPIO_SetEint11(1, Falling_Edge, eDisFLT, 0x1f); // EINT11, falling edge, enable filter, filter width
		GPIO_SetPullUpDownEach(eGPIO_H1, eGPIO_3, 0x10);
		GPIO_SetFunctionEach(eGPIO_H1, eGPIO_3, eGINT);
		GPIO_EINT30UnMask(NUM_EINT11);
		///////////////// Interrupt End///////////////////////
		
		for(  i=0; i<TEST_COUNT; i++)
		{
			Outp32( uSrcAddr+(i*4), x+0x5555 );
			Outp32( ONEDRAM_B_PORT_SHARED0_BANK+(i*4), x+0x5555 );
		}
		SignalSemaphoreStatus(ONEDRAM_A_PORT);
		Outp32(MailBox_BA,TEST_COUNT);
#if 0
		for( i=0; i<TEST_COUNT; i++)
		{
			y = Inp32( ONEDRAM_B_PORT_SHARED1_BANK+(i*4));
			Outp32( uDstAddr+(i*4), y-1 ); 
		}
#endif

		WaitSemaphoreStatus(ONEDRAM_B_PORT);


		uIsOK = Compare32( (void*)ONEDRAM_B_PORT_SHARED0_BANK, (void*)ONEDRAM_B_PORT_SHARED1_BANK, TEST_COUNT/8 );	
		if( uIsOK == 0 )
			{
			Disp(" Fail to Compare\n");
			return false;
			}
		else
			{
			Disp(" MCP A-Type A_Port Test OK\n");	
			return true;
			}


}
void DMCT_PowerModeCurrent(void)
{
	u32 uBankInterval, uBankCount;
	u32 uTransSize = 0x400000;
	u32 uDstAddr, uSrcAddr;
	
	u8 ucPowerMode=0; 
	u32 uTempEmpty;
	u8 ucDMC_CH,ucAddressMap;
	
	// normal,all zero, all Pull-down, DCC,Auto Pre, time out, Dynamic Pre Powerdown, Dynamic force, Dynamic Self

	if(uDmcCh==0){	uSrcAddr=0x23000000, uDstAddr=uSrcAddr+uTransSize;
					ucDMC_CH=v210_DMC0,  ucAddressMap=MAP_MIXED1;}
	else		 {	uSrcAddr=0x43000000, uDstAddr=uSrcAddr+uTransSize;
					ucDMC_CH=v210_DMC1; ucAddressMap=MAP_INTERLEAVED; }
	
	for(ucPowerMode=0;ucPowerMode<9;ucPowerMode++)
		{
			switch(ucPowerMode)
			{
			case 0:
					
					DMC_SetCh(ucDMC_CH);
					DMC_ChangeMapMethod(CHIP_0, ucAddressMap);
					DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
					DMC_PhyDriving(ucDMC_CH,PHY_DRV_DIS,DRV_ALL_ZERO);
					Disp("CASE :(All Disable) Setting Complete\n");
					while(!UART_GetKey()){}
					Disp("CASE : All disable, DMA 16 burst\n Enter any KB\n");
					while(!UART_GetKey())
					{
						DMCT_MeasureTimeCount(CHIP_0,6,2,ucAddressMap,TRUE,uTransSize,uDstAddr, uSrcAddr );
					}
					break;
			case 1:
					
					DMC_SetCh(ucDMC_CH);
					DMC_ChangeMapMethod(CHIP_0, ucAddressMap);
					DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
					DMC_PhyDriving(ucDMC_CH,PHY_DRV_EN,DRV_ALL_ZERO);
					Disp("CASE :(Phy Drive En- All Zero) Setting Complete\n");
					while(!UART_GetKey()){}
					Disp("CASE : All Dis except PHY_DRV ALL Zero, DMA 16 burst\n Enter any KB\n");
					while(!UART_GetKey())
					{
						DMCT_MeasureTimeCount(CHIP_0,6,2,ucAddressMap,TRUE,uTransSize,uDstAddr, uSrcAddr );
					}
					break;
			case 2:
					
					DMC_SetCh(ucDMC_CH);
					DMC_ChangeMapMethod(CHIP_0, ucAddressMap);
					DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
					DMC_PhyDriving(ucDMC_CH,PHY_DRV_EN,DRV_PULL_DOWN);
					Disp("CASE :(Phy Drive En- pull down) Setting Complete\n");
					while(!UART_GetKey()){}					
					Disp("CASE : All Dis except PHY_DRV PULL Down, DMA 16 burst\n Enter any KB\n");
					while(!UART_GetKey())
					{
						DMCT_MeasureTimeCount(CHIP_0,6,2,ucAddressMap,TRUE,uTransSize,uDstAddr, uSrcAddr );
					}
					break;
			case 3:
					
					DMC_SetCh(ucDMC_CH);
					DMC_ChangeMapMethod(CHIP_0, ucAddressMap);
					DMC_SetPowerDownModes(FORCE_PRE_DIS, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_EN);
					DMC_PhyDriving(ucDMC_CH,PHY_DRV_EN,DRV_ALL_ZERO);
					Disp("CASE :(Clock Control) Setting Complete\n");
					while(!UART_GetKey()){}
					Disp("CASE : All Dis except CLK Control EN, DMA 16 burst\n Enter any KB\n");
					while(!UART_GetKey())
					{
						DMCT_MeasureTimeCount(CHIP_0,6,2,ucAddressMap,TRUE,uTransSize,uDstAddr, uSrcAddr );
					}
					break;
			case 4:
					
					DMC_SetCh(ucDMC_CH);
					DMC_ChangeMapMethod(CHIP_SEL, ucAddressMap);
					DMC_SetPowerDownModes(TimeOut_PreDis, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
					DMC_PhyDriving(ucDMC_CH,PHY_DRV_EN,DRV_ALL_ZERO);
					DMC_SetPrechPolicy(0xff, 0xff);// Auto Precharge
					Disp("CASE :(Auto precharge) Setting Complete\n");
					while(!UART_GetKey()){}

					Disp("CASE :(Auto precharge),DMA 16 burst\n Enter any KB\n");
					while(!UART_GetKey())
					{
						DMCT_MeasureTimeCount(CHIP_0,6,2,ucAddressMap,TRUE,uTransSize,uDstAddr, uSrcAddr );
					}
					break;
			case 5:
					
					DMC_SetCh(ucDMC_CH);
					DMC_ChangeMapMethod(CHIP_SEL, ucAddressMap);
					DMC_SetPowerDownModes(TimeOut_PreEn, DYN_POWER_DOWND_DIS, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
					DMC_PhyDriving(ucDMC_CH,PHY_DRV_EN,DRV_ALL_ZERO);
					DMC_SetPrechPolicy(0x00, 0x00);// Open Page Precharge
					Disp("CASE :(Time Out precharge) Setting Complete\n");
					while(!UART_GetKey()){}

					Disp("CASE :(Time Out precharge),DMA 16 burst\n Enter any KB\n");
					while(!UART_GetKey())
					{
						DMCT_MeasureTimeCount(CHIP_0,6,2,ucAddressMap,TRUE,uTransSize,uDstAddr, uSrcAddr );
					}
					break;
			case 6:
					
					DMC_SetCh(ucDMC_CH);
					DMC_ChangeMapMethod(CHIP_SEL, ucAddressMap);
					DMC_SetPowerDownModes(TimeOut_PreDis, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_DIS, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
					DMC_PhyDriving(ucDMC_CH,PHY_DRV_EN,DRV_ALL_ZERO);
					DMC_SetPrechPolicy(0x00, 0x00);// Open Page Precharge
					Disp("CASE :(Active/Precharge) Setting Complete\n");
					while(!UART_GetKey()){}

					Disp("CASE :(Active/Precharge),DMA 16 burst\n Enter any KB\n");
					while(!UART_GetKey())
					{
						DMCT_MeasureTimeCount(CHIP_0,6,2,ucAddressMap,TRUE,uTransSize,uDstAddr, uSrcAddr );
					}
					break;	
			case 7:
					
					DMC_SetCh(ucDMC_CH);
					DMC_ChangeMapMethod(CHIP_SEL, ucAddressMap);
					DMC_SetPowerDownModes(TimeOut_PreDis, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_DIS, FORCE_PRE_POWER_DOWN, DYN_CLK_CONTROL_DIS);
					DMC_PhyDriving(ucDMC_CH,PHY_DRV_EN,DRV_ALL_ZERO);
					DMC_SetPrechPolicy(0x00, 0x00);// Open Page Precharge
					Disp("CASE :(Dynamic force precharge) Setting Complete\n");
					while(!UART_GetKey()){}

					Disp("CASE :(Active/Precharge),DMA 16 burst\n Enter any KB\n");
					while(!UART_GetKey())
					{
						DMCT_MeasureTimeCount(CHIP_0,6,2,ucAddressMap,TRUE,uTransSize,uDstAddr, uSrcAddr );
					}
					break;						
			case 8:
					
					DMC_SetCh(ucDMC_CH);
					DMC_ChangeMapMethod(CHIP_0, ucAddressMap);
					DMC_SetPowerDownModes(TimeOut_PreEn, DYN_POWER_DOWND_EN, DYN_SELF_REFRESH_EN, ACTIVE_POWER_DOWN, DYN_CLK_CONTROL_EN);
					DMC_PhyDriving(ucDMC_CH,PHY_DRV_EN,DRV_ALL_ZERO);
					Disp("CASE :Self Refresh(Only) Setting Complete\n");
					while(!UART_GetKey()){}
			
					do 
					{
					uTempEmpty = DMC_GetQueueEmpty(0);
					}
					while (uTempEmpty == 0); // while queue is not empty 
					Disp("CASE :Self Refresh Enable, Memory Deep Power Down\n");
					DMC_SendCommand(CHIP_0, CMD_DPD, 0, 0); //Memory Deep Power Down
					while(!UART_GetKey())
					{
					//	DMCT_MeasureTimeCount(v210_DMC0,6,2,MAP_MIXED1,TRUE,uTransSize,uDstAddr, uSrcAddr );
					}
					DMC_SendCommand(CHIP_0, CMD_NOP, 0, 0); //Memory NOP
					break;

			default:
			Disp(" Wrong wakeup source selected!!\n");
			Assert(0);
			break;
				}

		}
}

void DMCT_Dmc1ClkChangeAging(void)
{
u8 ucHCLK_MSYS_RATIO,uTempEmpty;

//	UART_Printf("Select Mclk_Msys_Ratio :0 ~ 7\n ");

//	do
//		ucHCLK_MSYS_RATIO = UART_GetIntNum();
//	while(ucHCLK_MSYS_RATIO<0 | ucHCLK_MSYS_RATIO>7);
while(true)
   for(	ucHCLK_MSYS_RATIO=0;ucHCLK_MSYS_RATIO<8;ucHCLK_MSYS_RATIO++)
   	{
	   	
		if(ucHCLK_MSYS_RATIO==0)
		{
			SYSC_SetClkDiv(eCLKDIV_APLL,4);
			Disp("ARM Speed 200MHz Setting\n");
		}
		if(ucHCLK_MSYS_RATIO>0 & ucHCLK_MSYS_RATIO<3)
		{
			SYSC_SetClkDiv(eCLKDIV_APLL,2);
			Disp("ARM Speed 400MHz Setting\n");
		}

	//		Disp("ARM Speed 800MHz Setting\n");

		DMC_SetCh(v210_DMC1);
		do 
		{
		uTempEmpty = DMC_GetQueueEmpty(0);
		}
		while (uTempEmpty == 0); // while queue is not empty 
		
		SYSC_SetClkDiv(eCLKDIV_HCLKMSYS, ucHCLK_MSYS_RATIO+1); //200Mhz@800 ARMCLK / (HCLK_MSYS_RATIO + 1)

	DMCT_MemcpyPerfTest2();		
   }

}

void DMCT_Dmc0ClkChangeAging(void)
{
u8 ucDIV_ONEDRAM,uTempEmpty;

//	UART_Printf("Select Mclk_Msys_Ratio :0 ~ 7\n ");

//	do
//		ucHCLK_MSYS_RATIO = UART_GetIntNum();
//	while(ucHCLK_MSYS_RATIO<0 | ucHCLK_MSYS_RATIO>7);
while(true)
   for(	ucDIV_ONEDRAM=0;ucDIV_ONEDRAM<16;ucDIV_ONEDRAM++)
   	{
	   	
		if(ucDIV_ONEDRAM==0|ucDIV_ONEDRAM==1)
		{
			SYSC_SetClkDiv(eCLKDIV_A2M,4);
			Disp("ARM Speed 200MHz Setting and DIV OneDRAM(%d) \n",ucDIV_ONEDRAM);
		}
		else
		Disp("ARM Speed 800MHz Setting and DIV OneDRAM(%d) \n",ucDIV_ONEDRAM);

		DMC_SetCh(v210_DMC0);
		do 
		{
		uTempEmpty = DMC_GetQueueEmpty(0);
		}
		while (uTempEmpty == 0); // while queue is not empty 
		
		SYSC_SetClkDiv(eCLKDIV_ONEDRAM, ucDIV_ONEDRAM+1); //200Mhz@800 ARMCLK / (HCLK_MSYS_RATIO + 1)

	DMCT_MemcpyPerfTest();		
   }

}


// Set clock divider
void SYS_DIV_AGING()
{
	u32 uRegAddrA2M,uRegAddrHCLK,uRegAddrAPLL;
	u32 uRegMaskDataA2M,uRegMaskDataHCLK,uRegMaskDataAPLL;
	u32 uRegBitOffsetA2M,uRegBitOffsetHCLK,uRegBitOffsetAPLL;
	u32 uRegTemp;
	u32 uDivValueA2M,uDivValueHCLK,uDivValueAPLL;
	u32 uStatRegAddr=0xE0100300;
	u32 uStatRegBitOffset;
	u32 uStatRegTemp;
	u32 i,uTemp;
	int iDmcCH;
	Disp(" Select aging time :  \n");
	uTemp = UART_GetIntNum();

	
	for(i=0;i<uTemp;i++)
	{

	Disp(" Aging Timing cycle : %d\n",i);
//1)// 800Mh/200Mhz		
	//case eCLKDIV_A2M: 
	//	uStatRegBitOffset = 1;
		uDivValueA2M = 3;  uRegMaskDataA2M = 0x7; uRegBitOffsetA2M = 4;
	//case eCLKDIV_HCLKMSYS: 
	//	uStatRegBitOffset = 2;
		uDivValueHCLK = 3; uRegMaskDataHCLK = 0x7; uRegBitOffsetHCLK = 8;
	//	uStatRegBitOffset = 0;
		uDivValueAPLL = 0; uRegMaskDataAPLL = 0x7; uRegBitOffsetAPLL = 0;
	// CLKOUT = CLKIN / RATIO (RATIO = uRatio + 1)

	dma_memcpy(0x51000000, 0x21000000, 0x10000000, BURST8); // for MDMA



				uRegTemp = Inp32(uStatRegAddr);
				Disp("0xE010_3000 value : %x\n",uRegTemp);
				uRegTemp = uRegTemp &~(uRegMaskDataA2M<<uRegBitOffsetA2M) &~(uRegMaskDataHCLK<<uRegBitOffsetHCLK) &~(uRegMaskDataAPLL<<uRegBitOffsetAPLL); 
				uRegTemp = uRegTemp | (uDivValueA2M<<uRegBitOffsetA2M)| (uDivValueHCLK<<uRegBitOffsetHCLK)| (uDivValueAPLL<<uRegBitOffsetAPLL);
				Outp32(uStatRegAddr, uRegTemp); // set divider

				uRegTemp = Inp32(uStatRegAddr);
				Disp("0xE010_3000 value : %x\n",uRegTemp);
				//while (Inp32(uStatRegAddr)&(1<<uStatRegBitOffset) ); // 1 is on changing. 0 is stable

				SYSC_UpdateClkInform();
				
				UART_Printf(" System Info : Product ID [%x], Pkg Type [%x]\n", g_uProductID, g_uPkgMode);
				UART_Printf("				Main Revision [%d], Sub Revision [%d]\n", g_uMainRev, g_uSubRev);
				UART_Printf(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz  PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz  PCLK_Psys: %.2fMHz\n\n",
						(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
				UART_Printf(" APLL: %.2fMHz MPLL: %.2fMHz\n EPLL: %.2fMHz	VPLL: %.2fMHz\n\n",
						(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);

				for(iDmcCH=0;iDmcCH<0;iDmcCH++)
					{
					DMC_SetCh(iDmcCH);
					Disp("DMC channel : %d\n",iDmcCH);
					DMCT_SetBaseAddress();
//					TestBurst();
					TestSingle();
//					TestMarch();
					}
	
	//2)// 400Mh/200Mhz

	//case eCLKDIV_A2M: 
	//	uStatRegBitOffset = 1;
		uDivValueA2M = 3;  uRegMaskDataA2M = 0x7; uRegBitOffsetA2M = 4;
	//case eCLKDIV_HCLKMSYS: 
	//	uStatRegBitOffset = 2;
		uDivValueHCLK = 1; uRegMaskDataHCLK = 0x7; uRegBitOffsetHCLK = 8;
	//	uStatRegBitOffset = 0;
		uDivValueAPLL = 1; uRegMaskDataAPLL = 0x7; uRegBitOffsetAPLL = 0;
	// CLKOUT = CLKIN / RATIO (RATIO = uRatio + 1)




				uRegTemp = Inp32(uStatRegAddr);
				Disp("0xE010_3000 value : %x\n",uRegTemp);
				uRegTemp = uRegTemp &~(uRegMaskDataA2M<<uRegBitOffsetA2M) &~(uRegMaskDataHCLK<<uRegBitOffsetHCLK) &~(uRegMaskDataAPLL<<uRegBitOffsetAPLL); 
				uRegTemp = uRegTemp | (uDivValueA2M<<uRegBitOffsetA2M)| (uDivValueHCLK<<uRegBitOffsetHCLK)| (uDivValueAPLL<<uRegBitOffsetAPLL);
				Outp32(uStatRegAddr, uRegTemp); // set divider

				uRegTemp = Inp32(uStatRegAddr);
				Disp("0xE010_3000 value : %x\n",uRegTemp);
			//	while (Inp32(uStatRegAddr)&(1<<uStatRegBitOffset) ); // 1 is on changing. 0 is stable

				SYSC_UpdateClkInform();
				UART_Printf(" System Info : Product ID [%x], Pkg Type [%x]\n", g_uProductID, g_uPkgMode);
				UART_Printf("				Main Revision [%d], Sub Revision [%d]\n", g_uMainRev, g_uSubRev);
				UART_Printf(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz  PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz  PCLK_Psys: %.2fMHz\n\n",
						(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
				UART_Printf(" APLL: %.2fMHz MPLL: %.2fMHz\n EPLL: %.2fMHz	VPLL: %.2fMHz\n\n",
						(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);
				
								for(iDmcCH=0;iDmcCH<0;iDmcCH++)
									{
									DMC_SetCh(iDmcCH);
									Disp("DMC channel : %d\n",iDmcCH);
									DMCT_SetBaseAddress();
				//					TestBurst();
									TestSingle();
				//					TestMarch();
									}


//3)// 200Mh/200Mhz
		//case eCLKDIV_A2M: 
	//	uStatRegBitOffset = 1;
		uDivValueA2M = 3;  uRegMaskDataA2M = 0x7; uRegBitOffsetA2M = 4;
	//case eCLKDIV_HCLKMSYS: 
	//	uStatRegBitOffset = 2;
		uDivValueHCLK = 0; uRegMaskDataHCLK = 0x7; uRegBitOffsetHCLK = 8;
	//	uStatRegBitOffset = 0;
		uDivValueAPLL = 3; uRegMaskDataAPLL = 0x7; uRegBitOffsetAPLL = 0;
	// CLKOUT = CLKIN / RATIO (RATIO = uRatio + 1)




				uRegTemp = Inp32(uStatRegAddr);
				Disp("0xE010_3000 value : %x\n",uRegTemp);
				uRegTemp = uRegTemp &~(uRegMaskDataA2M<<uRegBitOffsetA2M) &~(uRegMaskDataHCLK<<uRegBitOffsetHCLK) &~(uRegMaskDataAPLL<<uRegBitOffsetAPLL); 
				uRegTemp = uRegTemp | (uDivValueA2M<<uRegBitOffsetA2M)| (uDivValueHCLK<<uRegBitOffsetHCLK)| (uDivValueAPLL<<uRegBitOffsetAPLL);
				Outp32(uStatRegAddr, uRegTemp); // set divider

				uRegTemp = Inp32(uStatRegAddr);
				Disp("0xE010_3000 value : %x\n",uRegTemp);
				//while (Inp32(uStatRegAddr)&(1<<uStatRegBitOffset) ); // 1 is on changing. 0 is stable

				SYSC_UpdateClkInform();
				UART_Printf(" System Info : Product ID [%x], Pkg Type [%x]\n", g_uProductID, g_uPkgMode);
				UART_Printf("				Main Revision [%d], Sub Revision [%d]\n", g_uMainRev, g_uSubRev);
				UART_Printf(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz	PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz  PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz  PCLK_Psys: %.2fMHz\n\n",
						(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
				UART_Printf(" APLL: %.2fMHz MPLL: %.2fMHz\n EPLL: %.2fMHz	VPLL: %.2fMHz\n\n",
						(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);

								for(iDmcCH=0;iDmcCH<0;iDmcCH++)
									{
									DMC_SetCh(iDmcCH);
									Disp("DMC channel : %d\n",iDmcCH);
									DMCT_SetBaseAddress();
				//					TestBurst();
									TestSingle();
				//					TestMarch();
									}

	}


}

void DMC_DLL_Value()
{

	u32 uTempPhyStatus,uTempPhyCtrl0;
	u32 i,j;
	int uC_lock,uF_lock;


for(i=0;i<200;i++)
	{
	(i<100)? DMC_SetCh(0): DMC_SetCh(1);

	uTempPhyStatus = DMC_GetPhyStatus();
	uC_lock = (uTempPhyStatus>>6);
	uF_lock = (uTempPhyStatus>>4)&0x3;


	Disp("DMC DLL : C lock %d \t",uC_lock);
	Disp("DMC DLL : F lock %d \n",uF_lock);
	}

}

