/**************************************************************************************
* 
*	Project Name : S5PC100 Validation
*
*	Copyright 2007 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PC100.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : dmc.c
*  
*	File Description : This file implements the API functons for low power DRAM
*                      controller. (DMC)
*
*	Author : Joonsoo Jeon
*	Dept. : AP Development Team
*	Created Date : 2007/11/13
*	Version : 0.1 
* 
*	History
*	- Created(Joonsoo Jeon 2007/11/05)
*  
**************************************************************************************/


#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "dmc.h"
#include "intc.h"
#include "dma.h"
#include "sysc.h"

///// For initializing DMC
#define LPDDR	1
#define LPDDR2	2
#define DDR		3	// not supported in v210
#define DDR2	4

///// For initializing DMC
#define MCP_AC_TYPE	1
#define MCP_H_TYPE	2


/////////////////////////////// user setting part //////////////////////////////
#ifndef USE_DRAM // default seeting is DDR
#define USE_DRAM 	DDR2		// 1:LPDDR, 2:LPDDR2, 3:DDR, 4:DDR2
#endif
#define DRAM0_FREQ	200000000    //166Mhz
#define DRAM1_FREQ	200000000    //166Mhz

#define ENABLE_PHY_DLL   TRUE
//extern u32 g_HCLKD0;
///////////////////////////////////////////////////////////////////////////

#define DRAM0_KHZ			(DRAM0_FREQ/1000)
#define DRAM0_MHZ			(DRAM0_KHZ/1000)
#define DRAM0_tCK			((10*1000000/DRAM0_KHZ)+9)/10 // for round, (x*10+9)/10
				// @12MHz -> 83ns, @166MHz -> 6ns, @200MHz -> 5ns
#define DRAM1_KHZ			(DRAM1_FREQ/1000)
#define DRAM1_MHZ			(DRAM1_KHZ/1000)
#define DRAM1_tCK			((10*1000000/DRAM1_KHZ)+9)/10 // for round, (x*10+9)/10



//--------------------------------------------------------------------
// DMC_PHY_CONTROL0 register : PHY control 0 register
//--------------------------------------------------------------------
#define CTRL_FORCE			0x82		//	DLL force - max.0xff
#define CTRL_INC			0x10		//	Increase amount - max.0xff
#define CTRL_START_POINT		0x10		//	DLL lock start point - max.0xff
#define CTRL_FNC_FB			0			// Function Feedback Test
#if	(USE_DRAM == DDR) ||(USE_DRAM == LPDDR)	
#define CTRL_DDR			1		// 	reset value of control output 
#define CTRL_DFDQS			1       //    single DQS setting
#else
#define CTRL_DDR			1		// 	reset value of control output 
#define CTRL_DFDQS			1       //    differential DQS setting
#endif

#define CTRL_HALF			0		// 	DLL low speed
#define CTRL_DLL_ON			0		// 	DLL on
#define CTRL_START			0		// 	DLL start

//--------------------------------------------------------------------
// DMC_PHY_CONTROL1 register : PHY control 1 register
//-------------------------------------------------------------------
#define DQS_DELAY			0		// shiftc + dqs delay clock <- DDR2, LPDDR2 upper 333MHz 
#define CTRL_OFFSETD			0		//	offset for debug - max.0x7f
#define	DRV_TYPE			0		// Driving type of bidirectional pins in ildel state
#define CTRL_OFFSETC			4		//	gate offset - max.0x7f 
#define CTRL_REF				8		// 0~0xF, reference count for DLL lock confirmation
#define FD_RESYNC			0		// force dll resync.
#if(USE_DRAM == LPDDR2||USE_DRAM == LPDDR)
#define CTRL_SHIFTC			4		//	gateout signal delay - 0:T/128, 1:T/64, 2:T/32, 3:T/16, 4:T/8, 5:T/4, 6:T/2, 7:T
#else
#define CTRL_SHIFTC			6		
#endif
//--------------------------------------------------------------------
// DMC_PHY_CONTROL2 register : PHY control 2 register
//--------------------------------------------------------------------
#define CTRL_OFFSET3			0		//	offset for debug - max.0x7f
#define CTRL_OFFSET2			0		//	offset for debug - max.0x7f
#define CTRL_OFFSET1			0		//	offset for debug - max.0x7f
#define CTRL_OFFSET0			0		//	offset for debug - max.0x7f

u32 uDmcCh;


#if 0
//===================================================================
// Memory Type
//===================================================================
#define K4X1G163PC              0       // LPDDR2 1Gbit  *2ea
#define K4T2G084QA              1       // DDR2 2Gbit    *4ea
//
#define KAC00F00XM              2		// MCP 2g1g1g    A-port:DDRx16,B-port:DDRx32
#define K4T1G164QE              3       // DDR2 1Gbit    *2ea
//===================================================================

//===================================================================
// DMC Memory(LPDDR/DDR2)
//===================================================================
#if (USE_DRAM == MCP_AC_TYPE)
#define DRAM0_MODEL             KAC00F00XM
#define DRAM1_MODEL             K4X1G163PC
#endif
#if(USE_DRAM == MCP_H_TYPE)
#define DRAM0_MODEL             K4T1G164QE
#define DRAM1_MODEL             K4T2G084QA
#endif
//===================================================================


#endif 

////////  Function Implementation

// #define ENABLE_PHY_DLL		


#ifdef DBG_DMC
	#define DbgDMC(x) Dbg x
#else
	#define DbgDMC(x) 0;
#endif

#ifdef DISP_DMC_REG
#define dmcOutp32(a, d) {u32 a2, d2; a2=a; d2=d; Disp("Outp32(\'h%08x, \'h%08x);\n", a2, d2); Outp32(a2, d2);}
#define dmcInp32(a) {u32 a2, d2; a2=a; d2=Inp32(a2); Disp("Inp32(\'h%08x); // d=0x%08x\n", a2, d2);}
#else
#define dmcOutp32 Outp32
#define dmcInp32 Inp32
#endif


///// dmc SFRs...
typedef struct tag_DMC_REGS
{
	u32 rDMC_CON_CONTROL;		// DMC_SFR_BASE + 0x00000000
        // [27:16] : Default timeout cycles (n aclk cycles)
        // [15:12] : Read data fetch cycles (n mclk cycles)
        // [10]    : DQ swap
        // [7]     : PHY driving
        // [6]     : Read cycle gap for two different chips
        // [5]     : Auto refresh counter
        // [4]     : Out of order scheduling
        // [3:1]   : Clock ratio of bus clock to memory clock[aclk:mclk] (0=1:1, 3-2:1)
        // [0]     : Multi clock operation (0=sync, 1=async)
	u32 rDMC_MEM_CONTROL;		// DMC_SFR_BASE + 0x00000004
        // [22:20] : Memory burst length (1=2, 2=4, 3=8, 4=16)
        // [19:16] : Number of Memory chips (0=1chip, 1=2chips)
        // [15:12] : Width of Memory data bus (1=16bit, 2=32bit)
        // [11:8]  : Type of Memory (1=LPDDR, 2=LPDDR2, 3=DDR, 4=DDR2)
        // [7:6]   : Additional latency for PALL
        // [5]     : Dynamic self refresh
        // [4]     : Force precharge
        // [3:2]   : Type of dynamic power down
        // [1]     : Dynamic power down
        // [0]     : Dynamic clock control
	u32 rDMC_MEM_CONFIG0;		// DMC_SFR_BASE + 0x00000008
        // [31:24] : AXI base address
        // [23:16] : AXI base address mask
        // [15:12] : Address mapping methid (0=bank/row/col, 1=row/bank/col)
        // [11:8]  : Number of column address bits (0=7bits, 1=8bits, 2=9bits, 3=10bits, ... 5=12bits)
        // [7:4]   : Number of row address bits (0=12bits, 1=13bits, 2=14bits, 3=15bits)
        // [3:0]   : Number of banks (0=1banks, 1=2banks, 2=4banks, 3=8banks)
	u32 rDMC_MEM_CONFIG1;		// DMC_SFR_BASE + 0x0000000c
	u32 rDMC_DIRECT_CMD;		// DMC_SFR_BASE + 0x00000010
        // [27:24] : Type of direct command
        // [20]    : The chip number to send the direct command to
        // [18:16] : Related bank address when issuing a direct command
        // [14:0]  : Related address value when issuing a direct command
	u32 rDMC_PRECH_CONFIG;		// DMC_SFR_BASE + 0x00000014
        // [31:24] : Force precharge
        // [15:8]  : Memory chip1 precharge bank selective policy (1=close page policy)
        // [7:0]   : Memory chip0 precharge bank selective policy (1=close page policy)
	u32 rDMC_PHY_CONTROL0;	 	// DMC_SFR_BASE + 0x00000018
        // [31:24] : DLL force delay
        // [23:16] : DLL delay increment
        // [15:8]  : DLL lock start point
        // [4]     : DLL CRC
        // [3]     : Reset value
        // [2]     : DLL low speed
        // [1]     : DLL on
        // [0]     : DLL start
 	u32 rDMC_PHY_CONTROL1;	 	// DMC_SFR_BASE + 0x0000001c
	u32 rDMC_PHY_CONTROL2;	 	// DMC_SFR_BASE + 0x00000020
	u32 reserved1;
	u32 rDMC_PWRDN_CONFIG;	 	// DMC_SFR_BASE + 0x00000028
        // [31:16] : Number of cycles for dynamic self refresh entry
        // [7:0]   : Number of cycles for dynamic power down entry
	u32 reserved2;
	u32 rDMC_TIMING_AREF;		// DMC_SFR_BASE + 0x00000030
	u32 rDMC_TIMING_ROW;		// DMC_SFR_BASE + 0x00000034
	u32 rDMC_TIMING_DATA;		// DMC_SFR_BASE + 0x00000038
	u32 rDMC_TIMING_POWER;		// DMC_SFR_BASE + 0x0000003c
	u32 rDMC_PHY_STATUS0;		// DMC_SFR_BASE + 0x00000040
	u32 rDMC_PHY_STATUS1;		// DMC_SFR_BASE + 0x00000044
	u32 rDMC_CHIP0_STATUS;		// DMC_SFR_BASE + 0x00000048
	u32 rDMC_CHIP1_STATUS;	 	// DMC_SFR_BASE + 0x0000004c
	u32 rDMC_AREF_STATUS;		// DMC_SFR_BASE + 0x00000050
	u32 reserved3[3];	
	u32 rDMC_QOS_CONTROL0;		// DMC_SFR_BASE + 0x00000060
	u32 rDMC_QOS_CONFIG0;		// DMC_SFR_BASE + 0x00000064
	u32 rDMC_QOS_CONTROL1;	 	// DMC_SFR_BASE + 0x00000068
	u32 rDMC_QOS_CONFIG1;		// DMC_SFR_BASE + 0x0000006c
	u32 rDMC_QOS_CONTROL2;		// DMC_SFR_BASE + 0x00000070
	u32 rDMC_QOS_CONFIG2;		// DMC_SFR_BASE + 0x00000074
	u32 rDMC_QOS_CONTROL3;		// DMC_SFR_BASE + 0x00000078
	u32 rDMC_QOS_CONFIG3;		// DMC_SFR_BASE + 0x0000007c
	u32 rDMC_QOS_CONTROL4;		// DMC_SFR_BASE + 0x00000080
	u32 rDMC_QOS_CONFIG4;		// DMC_SFR_BASE + 0x00000084
	u32 rDMC_QOS_CONTROL5;		// DMC_SFR_BASE + 0x00000088
	u32 rDMC_QOS_CONFIG5;		// DMC_SFR_BASE + 0x0000008c
	u32 rDMC_QOS_CONTROL6;		// DMC_SFR_BASE + 0x00000090
	u32 rDMC_QOS_CONFIG6;		// DMC_SFR_BASE + 0x00000094
	u32 rDMC_QOS_CONTROL7;		// DMC_SFR_BASE + 0x00000098
	u32 rDMC_QOS_CONFIG7;		// DMC_SFR_BASE + 0x0000009c
	u32 rDMC_QOS_CONTROL8;		// DMC_SFR_BASE + 0x000000A0
	u32 rDMC_QOS_CONFI8;		// DMC_SFR_BASE + 0x000000A4
	u32 rDMC_QOS_CONTROL9;	 	// DMC_SFR_BASE + 0x000000A8
	u32 rDMC_QOS_CONFIG9;		// DMC_SFR_BASE + 0x000000Ac
	u32 rDMC_QOS_CONTROL10;		// DMC_SFR_BASE + 0x000000B0
	u32 rDMC_QOS_CONFIG10;		// DMC_SFR_BASE + 0x000000B4
	u32 rDMC_QOS_CONTROL11;		// DMC_SFR_BASE + 0x000000B8
	u32 rDMC_QOS_CONFIG11;		// DMC_SFR_BASE + 0x000000Bc
	u32 rDMC_QOS_CONTROL12;		// DMC_SFR_BASE + 0x000000C0
	u32 rDMC_QOS_CONFIG12;		// DMC_SFR_BASE + 0x000000C4
	u32 rDMC_QOS_CONTROL13;		// DMC_SFR_BASE + 0x000000C8
	u32 rDMC_QOS_CONFIG13;		// DMC_SFR_BASE + 0x000000Cc
	u32 rDMC_QOS_CONTROL14;		// DMC_SFR_BASE + 0x000000D0
	u32 rDMC_QOS_CONFIG14;		// DMC_SFR_BASE + 0x000000D4
	u32 rDMC_QOS_CONTROL15;		// DMC_SFR_BASE + 0x000000D8
	u32 rDMC_QOS_CONFIG15;		// DMC_SFR_BASE + 0x000000Dc
} DMC_REGS;

static volatile DMC_REGS *g_pDmcRegs = (DMC_REGS *)DMC0_BASE;
//static volatile DMC_REGS *g_pDmc1Regs = (DMC_REGS *)DMC1_BASE;

void DMC_SetCh(s32 uCh)
{
	if (uCh==0||uCh==2)
	{
		g_pDmcRegs = (DMC_REGS *)DMC0_BASE;

	}
	else if (uCh==1)
	{
		g_pDmcRegs = (DMC_REGS *)DMC1_BASE;
	}
	else 		
		Assert(0);

}


void DMC_InitDmc(void)
{
	u8 iMemType;

	Disp("Start DMC[%d] initial\n", uDmcCh);
	Disp("Select Memory Type\n");
	Disp("0) MCP_A or MCP_C Type \n 1)MCP_H Type \n 2)DMC-DDR2 512MB \n 3)DMC1-DDR2 1024MB(evt0)");
	Disp("\nSelect the Memory Type : ");
	iMemType = UART_GetIntNum();

	//========================================================
	//[KAC00F00XM] MCP(LPDDR+OneDram)2G1G1G Memory Setting
	// CHIP_BASE: 0x20(0x2000_0000), CHIP_MASK: 0xF8(1G,128MByte), CHIP_MAP : 0(0:Linear,1:Interleaved,2:Mixed)
	// Bank: 4bank, Row: A0~A13, Column: A0~A8
	// tRFC 72ns, tRRD 12ns, tRP 18ns, tRCD 18ns, tRC 60ns, tRAS 42ns
	// tWR 12ns, trtp , tWTR 1tCK, tCKE 2tCK, tMRD 2tCK
	//=======================================================

	if(iMemType==0)
		{
		Disp("MCP_A or MCP_C Type initial\n");
		DMC_Init_McpACtype();
		}
	//========================================================
	//[K4X1G163PC] LPDDR Memory Setting
	// CHIP_BASE: 0x40(0x4000_0000), CHIP_MASK: 0xF0(1G*2,128MByte), CHIP_MAP : 0(0:Linear,1:Interleaved,2:Mixed)
	// Bank: 4bank, Row: A0~A13, Column: A0~A9
	// tRFC 72ns, tRRD 12ns, tRP 18ns, tRCD 18ns, tRC 60ns, tRAS 42ns
	// tWR 12ns, tWR 12ns, trtp , tWTR 1tCK, tCKE 2tCK, tMRD 2tCK
	//========================================================

	else if(iMemType==1)
		{
		Disp("MCP_H Type(mDDR 1G *2) initialized\n");
		DMC_Init_McpHtype();	
		}
	else if(iMemType==2)
		{
		Disp("[DMC%d]DDR2 1G*4 initialized\n",uDmcCh);
		DMC_Init_K4T1G084QQ();	
		}
	else if(iMemType==3)
		{
		Disp("[DMC1]DDR2 2G*4 initialized\n");
		DMC_Init_K4T2G084QA();	
		}

		else
		Assert(0);

#if 0		
	//========================================================
	//[KAC00F00XM] MCP(LPDDR+OneDram)2G1G1G Memory Setting
	// CHIP_BASE: 0x20(0x2000_0000), CHIP_MASK: 0xF8(1G,128MByte), CHIP_MAP : 0(0:Linear,1:Interleaved,2:Mixed)
	// Bank: 4bank, Row: A0~A13, Column: A0~A8
	// tRFC 72ns, tRRD 12ns, tRP 18ns, tRCD 18ns, tRC 60ns, tRAS 42ns
	// tWR 12ns, trtp , tWTR 1tCK, tCKE 2tCK, tMRD 2tCK
	//=======================================================
	if(DRAM0_MODEL == KAC00F00XM && g_pDmcRegs == (DMC_REGS *)DMC0_BASE)
	{
		   Disp("KAC00F00XM Mem initial\n");
	   DMC_Init_KAC00F00XM();
	   Disp("KAC00F00XM Mem initial\n");
	}

	//========================================================
	//[K4X1G163PC] LPDDR Memory Setting
	// CHIP_BASE: 0x40(0x4000_0000), CHIP_MASK: 0xF0(1G*2,128MByte), CHIP_MAP : 0(0:Linear,1:Interleaved,2:Mixed)
	// Bank: 4bank, Row: A0~A13, Column: A0~A9
	// tRFC 72ns, tRRD 12ns, tRP 18ns, tRCD 18ns, tRC 60ns, tRAS 42ns
	// tWR 12ns, tWR 12ns, trtp , tWTR 1tCK, tCKE 2tCK, tMRD 2tCK
	//========================================================
	else if(DRAM1_MODEL == K4X1G163PC&& g_pDmcRegs == (DMC_REGS *)DMC1_BASE)
	{
		     Disp("K4X1G163PC Mem initial\n");
		DMC_Init_K4X1G163PC();
	     Disp("K4X1G163PC Mem initial\n");
	}

	//========================================================
	//[K4T1G164QE] DDR2 Memory Setting
	// CHIP_BASE: 0x20(0x2000_0000), CHIP_MASK: 0xF0(1G*2ea,256MByte), CHIP_MAP : 0(0:Linear,1:Interleaved,2:Mixed)
	// Bank: 8bank, Row: A0~A12, Column: A0~A9
	// tRFC 127.5ns, tRRD 10ns, tRP 15ns, tRCD 15ns, tRC 60ns, tRAS 45ns
	// tWR 15ns, trtp 7.5ns, tFAW 50ns(four Activate Window for 2KB Page Size product), tWTR 7.5ns
	// tWTR 2tCK, tCL 3tCK, tWL 3tCK, tRL 3tCK, tXP 2tCK, tCKE 3tCK, tMRD 2tCK
	//========================================================
	else if(DRAM0_MODEL == K4T1G164QE&& g_pDmcRegs == (DMC_REGS *)DMC0_BASE)
	{

		 	     Disp("K4T1G164QE Mem initial\n");
	     DMC_Init_K4T1G164QE();
	     Disp("K4T1G164QE Mem initial\n");

	}

	//========================================================
	//[K4T2G084QA] DDR2 Memory Setting
	// CHIP_BASE: 0x40(0x4000_0000), CHIP_MASK: 0xF0(1G,128MByte), CHIP_MAP : 0(0:Linear,1:Interleaved,2:Mixed)
	// Bank: 8bank, Row: A0~A14, Column: A0~A9
	// tRFC 195ns, tRRD 10ns, tRP 15ns, tRCD 15ns, tRC 60ns, tRAS 45ns
	// tWR 15ns, trtp 7.5ns, tWTR 7.5ns, tFAW 50ns(four Activate Window for 2KB Page Size product)
	// tWTR 2tCK, tCL 3tCK, tWL 3tCK, tRL 3tCK, tXP 2tCK, tCKE 3tCK, tMRD 2tCK
	//========================================================
	else if(DRAM1_MODEL == K4T2G084QA && g_pDmcRegs == (DMC_REGS *)DMC1_BASE)
	{

			     Disp("K4T2G084QA Mem initial\n");
		DMC_Init_K4T2G084QA();
			     Disp("K4T2G084QA Mem initialized\n");
	}
	else
		Assert(0);

#endif
}



//////////
// Function Name : SetDmcControl
// Function Description : DMC Control register setting function
// Input : 	u8 ucDqSwap -> 1h'x
//			u8 ucPhyDrvEn -> 1h'x
//			u8 ucChipReadGap -> 1h'x
//			u8 ucClkRatio -> 3h'x
// Output : none
// Version : v0.0
/*

void DMC_SetDmcControl(u8 ucDqSwap, u8 ucPhyDrvEn, u8 ucChipReadGap, u8 ucClkRatio)
{
	u32 uDmcControl;
	
	uDmcControl = dmcInp32(&g_pDmcRegs->rDMC_CON_CONTROL);
	uDmcControl &= ~((1<<10)|(3<<6)|(0xf));
	uDmcControl |= (ucDqSwap<<10) | (ucPhyDrvEn<<7) | (ucChipReadGap<<6) | (ucClkRatio<<1);
	dmcOutp32(&g_pDmcRegs->rDMC_CON_CONTROL, uDmcControl);			
}
*/

void DMC_SetDmcControl(u8 ucQosFastEn, u8 ucDqSwap, u8 ucPhyDrvEn, u8 ucChipReadGap, u8 ucClkRatio)
{
	u32 uDmcControl;
	
	uDmcControl = dmcInp32(&g_pDmcRegs->rDMC_CON_CONTROL);
	uDmcControl &= ~((3<<10)|(3<<6)|(0xf));
	uDmcControl |=(ucQosFastEn<<11)|(ucDqSwap<<10) | (ucPhyDrvEn<<7) | (ucChipReadGap<<6) | (ucClkRatio<<1);
	dmcOutp32(&g_pDmcRegs->rDMC_CON_CONTROL, uDmcControl);			
}


//////////
// Function Name : DMC_SetEnhancedUtility
// Function Description : setting function for Enhancement utility
// Input : 	u16 usQToutCnt -> 12h'x 
//			u8 ucReadFetchDly -> 4h'x
//			u8 ucOutOfOder -> 1h'x
// Output : none
// Version : v0.0
void DMC_SetEnhancedUtility(u16 usQToutCnt, u8 ucReadFetchDly, u8 ucOutOfOder)
{
	u32 uDmcControl;

	uDmcControl = dmcInp32(&g_pDmcRegs->rDMC_CON_CONTROL);
	uDmcControl &= ~((0xfff<<16)|(0xf<<12)|(1<<4));
	uDmcControl |= (usQToutCnt<<16) | (ucReadFetchDly<<12) | (ucOutOfOder<<4);
	dmcOutp32(&g_pDmcRegs->rDMC_CON_CONTROL, uDmcControl);			
}


//////////
// Function Name : DMC_SetMemControl
// Function Description : setting function for Memory interface control 
// Input : 	u8 ucBurstLength -> 3h'x
//			u8 ucChips -> 4h'x
//			u8 ucMemType -> 4h'x
//			u8 ucAddLatencyPall	-> 2h'x		
// Output : none
// Version : v0.0
void DMC_SetMemControl(u8 ucBurstLength, u8 ucChips, u8 ucMemType, u8 ucAddLatencyPall)
{
	u32 uMemControl;

	uMemControl = dmcInp32(&g_pDmcRegs->rDMC_MEM_CONTROL);
	uMemControl &= ~((7<<20)|(0xf<<16)|(0xf<<8)|(3<<6));
	uMemControl |= (ucBurstLength<<20) | (ucChips<<16) | (ucMemType<<8) | (ucAddLatencyPall<<6);
	dmcOutp32(&g_pDmcRegs->rDMC_MEM_CONTROL, uMemControl);			
}


//////////
// Function Name : DMC_SetMemAddrConfig
// Function Description : setting function for Memory control
// Input : 	u8 ucChipNum -> the number of chip : 0 or 1
//			u8 ucBaseAddr -> 8h'x
//			u8 ucBaseMask -> 8h'x 
//			u8 ucMapMethod -> 4h'x
//			u8 ucColAddr -> 4h'x
//			u8 ucRowAddr ->4h'x
//			u8 ucBanks -> 4h'x
// Output : none
// Version : v0.0
void DMC_SetMemAddrConfig(u8 ucChipNum, u8 ucBaseAddr, u8 ucBaseMask, u8 ucMapMethod, u8 ucColAddr, u8 ucRowAddr, u8 ucBanks)
{
	u32 uMemConfig;
	u32 *puTemp;
	
	puTemp = &g_pDmcRegs->rDMC_MEM_CONFIG0 + ucChipNum;
	
	if (ucChipNum > 1) return;

	uMemConfig = dmcInp32(puTemp);
	uMemConfig &= ~((0xff<<24)|(0xff<<16)|(0xf<<12)|(0xf<<8)|(0xf<<4)|(0xf));
	uMemConfig |= (ucBaseAddr<<24) | (ucBaseMask<<16) | (ucMapMethod<<12) | (ucColAddr<<8) | (ucRowAddr<<4) | (ucBanks);
	dmcOutp32(puTemp, uMemConfig);			
}


//////////
// Function Name : DMC_ChangeBurstLength
// Function Description : burst length setting function
// Input : 	u8 ucBurstLength -> 3h'x
// Output : none
// Version : v0.0
void DMC_ChangeBurstLength(u8 ucBurstLength)
{
	u32 uMemControl;

	uMemControl = dmcInp32(&g_pDmcRegs->rDMC_MEM_CONTROL);
	uMemControl &= ~(7<<20);
	uMemControl |= (ucBurstLength<<20);
	dmcOutp32(&g_pDmcRegs->rDMC_MEM_CONTROL, uMemControl);			
}


//////////
// Function Name : DMC_ChangeMapMethod
// Function Description : Map method setting function
// Input : 	u8 ucChipNum -> the number of chip : 0 or 1
//			u8 ucMapMethod -> 1h'x
// Output : none
// Version : v0.0
void DMC_ChangeMapMethod(u8 ucChipNum, u8 ucMapMethod)
{
	u32 uMemConfig;
	u32 *puTemp;
	
	if (ucChipNum > 2) return;
	
	puTemp = &g_pDmcRegs->rDMC_MEM_CONFIG0 + ucChipNum;
	uMemConfig = dmcInp32(puTemp);
	uMemConfig &= ~(0xf<<12);
	uMemConfig |= (ucMapMethod<<12);
	dmcOutp32(puTemp, uMemConfig);			
}


//////////
// Function Name : DMC_SendCommand
// Function Description : send command to memory using DirectCmd register
// Input : 	u8 ucChipNum -> the number of chip : 0 or 1
//			u8 ucCmd -> 1h'x
//			u8 ucCmdBank -> 3h'x
//			u16 ucCmdAddr -> 15h'x
// Output : none
// Version : v0.0
void DMC_SendCommand(u8 ucChipNum, u8 ucCmd, u8 ucCmdBank, u16 ucCmdAddr)
{
	u32 uDirectCmd, uQueueEmpty;
	
	// check that queue is empty before sending command.
	if ( (ucChipNum == CHIP_0) || (ucChipNum == CHIP_1) ) // until queue of chip 0 or 1 is empty : using single Chip(CS).
	{
		do uQueueEmpty = dmcInp32(&g_pDmcRegs->rDMC_CON_CONTROL);
		while ( (uQueueEmpty&(1<<(8+ucChipNum)))==0 );
		uDirectCmd = (ucChipNum<<20) | (ucCmd<<24) | (ucCmdBank<<16) | (ucCmdAddr);
		dmcOutp32(&g_pDmcRegs->rDMC_DIRECT_CMD, uDirectCmd);
	}
	else if (ucChipNum == CHIP_BOTH) // until queue of both chip
	{
		do uQueueEmpty = dmcInp32(&g_pDmcRegs->rDMC_CON_CONTROL);
		while ( (uQueueEmpty&(3<<8))==0 );
		uDirectCmd = (0<<20) | (ucCmd<<24) | (ucCmdBank<<16) | (ucCmdAddr);
		dmcOutp32(&g_pDmcRegs->rDMC_DIRECT_CMD, uDirectCmd);
		uDirectCmd = (1<<20) | (ucCmd<<24) | (ucCmdBank<<16) | (ucCmdAddr);
		dmcOutp32(&g_pDmcRegs->rDMC_DIRECT_CMD, uDirectCmd);
	}
	else // exception
		return;	

}


//////////
// Function Name : DMC_GetBankStatus
// Function Description : check current status of memory
// Input : 	u8 ucChipNum -> the number of chip : 0 or 1
// Output : u32 status value
//				bank7[31:28] .... bank0[3:0] =  using 4 bits for each bank
//				0x0 : idle (precharged), 
//				0x1 : MRS/EMRS,
//				0x2 : deep power down,
//				0x3 : self refresh,
//				0x4 : auto refresh,
//				0x5 : precharge power down,
//				0x6 : row active, 
//				0x7 : active power down,
//				0x8 : write,
//				0x9 : write with auto precharge,
//				0xA : read,
//				0xB : read with auto precharge,
//				0xC : burst stop,
//				0xD : precharging,
//				0xE : MRR,
//				0xF: reserved
// Version : v0.0

u32 DMC_GetBankStatus(u8 ucChipNum)
{
	u32 uStatus;
	u32 *puTemp;
	
	if (ucChipNum > 1) return 0xffffffff;

	puTemp = &g_pDmcRegs->rDMC_CHIP0_STATUS + ucChipNum;

	uStatus = dmcInp32(puTemp);
	return uStatus;
}



u32 DMC_GetPhyStatus(void)
{
	u32 uStatus;
	u32 *puTemp;
	
	do { uStatus = dmcInp32( &g_pDmcRegs->rDMC_PHY_STATUS0); } while( (uStatus &0x4) != 0x4 ); // wait until PLL lock.

//	puTemp = &g_pDmcRegs->rDMC_PHY_STATUS0;
//	uStatus = dmcInp32(puTemp);
	return uStatus;
}

//////////
// Function Name : DMC_GetQueueEmpty
// Function Description : Auto refresh setting function
// Input : 	u8 ucChipNum -> the number of chip : 0 or 1
// Output : u32 uEmpty -> 1:empty, 0:not empty
// Version : v0.0
u32 DMC_GetQueueEmpty(u8 ucChipNum)
{
	u32 uChipEmpty[2] = {0, 0};
	u32 uConControl;
	
	uConControl = dmcInp32(&g_pDmcRegs->rDMC_CON_CONTROL);
	uChipEmpty[0] = (uConControl>>8)&(1);
	uChipEmpty[1] = (uConControl>>9)&(1);
	
	if (ucChipNum == 0) 
		return uChipEmpty[0];
	else if (ucChipNum == 1) 
		return uChipEmpty[1];
	else 
		return (u32)(-1);
}


//////////
// Function Name : DMC_GetUsingDramType
// Function Description : get type of current DRAM
// Input :  none
// Output : u32 uType - 1=LPDDR, 2=LPDDR2, 3=DDR, 4=DDR2
// Version : v0.0
u32 DMC_GetUsingDramType(void)
{
	u32 uType;
		
	uType = dmcInp32(&g_pDmcRegs->rDMC_MEM_CONTROL);
	uType = (uType>>8) & 0xf;
	
	return uType;
}


//////////
// Function Name : SetAutoRefresh
// Function Description : Auto refresh setting function
// Input : 	u8 ucArefEn -> 1h'x
//			u16 usArefInterval ->16h'x
// Output : none
// Version : v0.0
void DMC_SetAutoRefresh(u8 ucArefEn, u16 usArefInterval)
{
	u32 uDmcControl;
	u32 uTimingAref;
	
	uDmcControl = dmcInp32(&g_pDmcRegs->rDMC_CON_CONTROL);
	
	uDmcControl &= ~(1<<5); 
	uDmcControl |= (ucArefEn<<5);
	uTimingAref = (usArefInterval);
	
	dmcOutp32(&g_pDmcRegs->rDMC_CON_CONTROL, uDmcControl);
	dmcOutp32(&g_pDmcRegs->rDMC_TIMING_AREF, uTimingAref);
}


//////////
// Function Name : DMC_SetRowTimingParameters
// Function Description : setting function for Row timing
// Input : 	u8 ucTrfc -> 8h'x
//			u8 ucTrrd -> 4h'x
//			u8 ucTrp -> 4h'x 
//			u8 ucTrcd -> 4h'x
//			u8 ucTrc -> 6h'x
//			u8 ucTras ->6h'x
// Output : none
// Version : v0.0
void DMC_SetRowTimingParameters(u8 ucTrfc, u8 ucTrrd, u8 ucTrp, u8 ucTrcd, u8 ucTrc, u8 ucTras)
{
	u32 uRowTiming;
	
	uRowTiming = (ucTrfc<<24) | (ucTrrd<<20) | (ucTrp<<16) | (ucTrcd<<12) | (ucTrc<<6) | (ucTras);
	dmcOutp32(&g_pDmcRegs->rDMC_TIMING_ROW, uRowTiming);
}


//////////
// Function Name : DMC_SetDataTimingParameters
// Function Description : setting function for Data timing
// Input : 	u8 ucTwtr -> 4h'x
//			u8 ucTwr -> 4h'x 
//			u8 ucTrtp -> 4h'x
//			u8 ucCl -> 4h'x
//			u8 ucWl ->4h'x
//			u8 ucRl -> 4h'x
// Output : none
// Version : v0.0
void DMC_SetDataTimingParameters(u8 ucTwtr, u8 ucTwr, u8 ucTrtp, u8 ucCl, u8 ucWl, u8 ucRl)
{
	u32 uDataTiming;
	
	uDataTiming = (ucTwtr<<28) | (ucTwr<<24) | (ucTrtp<<50) | (ucCl<<16) | (ucWl<<8) | (ucRl);
	dmcOutp32(&g_pDmcRegs->rDMC_TIMING_DATA, uDataTiming);
}


//////////
// Function Name : DMC_SetPowerModeTimingParameters
// Function Description : setting function for Power mode timing
// Input : 	u8 ucBaseAddr -> 6h'x
//			u8 ucBaseMask -> 8h'x 
//			u8 ucMapMethod -> 8h'x
//			u8 ucColAddr -> 4h'x
//			u8 ucRowAddr ->4h'x
// Output : none
// Version : v0.0
void DMC_SetPowerModeTimingParameters(u8 ucTfaw, u8 ucTxsr, u8 ucTxp, u8 ucTcke, u8 ucTmrd)
{
	u32 uPowerTiming;
	
	uPowerTiming = (ucTfaw<<24) | (ucTxsr<<16) | (ucTxp<<8) | (ucTcke<<4) | (ucTmrd);
	dmcOutp32(&g_pDmcRegs->rDMC_TIMING_POWER, uPowerTiming);
}


//////////
// Function Name : DMC_SetPrechPolicy
// Function Description : setting function for bank prechage policy
// Input : 	u8 ucChip0PrechPolicy -> 8h'x
//			u8 ucChip1PrechPolicy -> 8h'x
// Output : none
// Version : v0.0
void DMC_SetPrechPolicy(u8 ucChip0PrechPolicy, u8 ucChip1PrechPolicy)
{
	u32 uPrechPolicy;
	
	uPrechPolicy = dmcInp32(&g_pDmcRegs->rDMC_PRECH_CONFIG);	
	uPrechPolicy &= ~((0xff<<8)|(0xff));
	uPrechPolicy |= (ucChip1PrechPolicy<<8) | (ucChip0PrechPolicy);
	dmcOutp32(&g_pDmcRegs->rDMC_PRECH_CONFIG, uPrechPolicy);
}


//////////
// Function Name : DMC_SetPowerDownCounter
// Function Description : setting function for counter of each memory power down mode (3 modes)
// Input : 	u8 ucForcePrechCnt -> 8h'x
//			u8 ucDPwrDnCnt -> 8h'x
//			u16 usDSrefCnt -> 16h'x
// Output : none
// Version : v0.0
void DMC_SetPowerDownCounter(u8 ucForcePrechCnt, u8 ucDPwrDnCnt, u16 usDSrefCnt)
{
	u32 uPrechConfig;
	u32 uPwndnConfig;
	
	uPrechConfig = dmcInp32(&g_pDmcRegs->rDMC_PRECH_CONFIG);
	
	uPrechConfig &= ~(0xff<<24);
	uPrechConfig |= (ucForcePrechCnt<<24);
	uPwndnConfig = (usDSrefCnt<<16) | (ucDPwrDnCnt);
	
	dmcOutp32(&g_pDmcRegs->rDMC_PRECH_CONFIG, uPrechConfig);
	dmcOutp32(&g_pDmcRegs->rDMC_PWRDN_CONFIG, uPwndnConfig);
}

	
//////////
// Function Name : DMC_SetPowerDownModes
// Function Description : setting function for control of memory power down mode 
// Input : 	u8 ucForcePrechEn -> 1h'x
//			u8 ucDPwrDnEn -> 1h'x
//			u8 ucDSrefEn -> 1h'x 
//			u8 ucDPwrDnType -> 2h'x
//			u8 ucClkStop -> 1h'x
// Output : none
// Version : v0.0
void DMC_SetPowerDownModes(u8 ucForcePrechEn, u8 ucDPwrDnEn, u8 ucDSrefEn, u8 ucDPwrDnType, u8 ucClkStop)
{
	u32 uMemControl;
	
	uMemControl = dmcInp32(&g_pDmcRegs->rDMC_MEM_CONTROL);
	uMemControl &= ~((1<<4)|(1<<1)|(1<<5)|(3<<2)|(1));
	uMemControl |= (ucForcePrechEn<<4) | (ucDPwrDnEn<<1) | (ucDSrefEn<<5) | (ucDPwrDnType<<2) | (ucClkStop);
	dmcOutp32(&g_pDmcRegs->rDMC_MEM_CONTROL, uMemControl);
}



//////////
// Function Name : DMC_SetQosConfig
// Function Description : setting function for Memory control
// Input : 	
//
// Output : none
// Version : v0.0
void DMC_SetQosConfig(u8 ucQosNum, u16 usQosCnt, u8 ucQosEn, u16 usQosMask, u16 usQosId)
{
	u32 uQosControl;
	u32 uQosConfig;
	u32 *puTemp0, *puTemp1;
	
	puTemp0 = &g_pDmcRegs->rDMC_QOS_CONTROL0 + ucQosNum*2;
	puTemp1 = &g_pDmcRegs->rDMC_QOS_CONFIG0 + ucQosNum*2;
	
	uQosControl = (usQosCnt<<16) | (ucQosEn);
	uQosConfig = (usQosMask<<16) | (usQosId);
	dmcOutp32(puTemp0, uQosControl);
	dmcOutp32(puTemp1, uQosConfig);
}

static u32 g_aDmc0QosId[][3] = {	{0, 0x000, 0x7F87},{1, 0x001, 0x7E1F},{2, 0x011, 0x7E1F},{3, 0x002, 0x61FF},
									{4, 0x042, 0x61FF},{5, 0x082, 0x61FF},{6, 0x0C2, 0x7FFF},{7, 0x102, 0x7FFF},
									{8, 0x012, 0x7FFF},{9, 0x092, 0x7FFF},{10, 0x052, 0x7FFF},{11, 0x0D2, 0x7FFF},
									{12, 0x152, 0x7FFF},{13, 0x022, 0x787F},{14, 0x062, 0x7FFF},{15, 0x0E2, 0x7FFF},
									{16, 0x032, 0x7FFF},{17, 0x072, 0x7FFF},{18, 0x0B2, 0x7FFF},{19, 0x0F2, 0x7FFF},
									{20, 0x132, 0x7FFF},{21, 0x172, 0x7FFF},{22, 0x1B2, 0x7FFF},{23, 0x1F2, 0x43FF},
									{24, 0x3F2, 0x43FF},{25, 0x03A, 0x7C3F} 									
								};

static u32 g_aDmc1QosId[][3] = {	{0, 0x000, 0x7F87},{1, 0x00C, 0x7E1F},{2, 0x014, 0x7E1F},
									{3, 0x00B, 0x61FF},{4, 0x04B, 0x61FF},{5, 0x08B, 0x61FF},
									{6, 0x0CB, 0x7FFF},{7, 0x10B, 0x7FFF},{8, 0x01B, 0x7FFF},
									{9, 0x09B, 0x7FFF},{10, 0x05B, 0x7FFF},{11, 0x0DB, 0x7FFF},
									{12, 0x15B, 0x7FFF},{13, 0x02B, 0x787F},{14, 0x06B, 0x7FFF},
									{15, 0x0EB, 0x7FFF},{16, 0x033, 0x7FFF},{17, 0x073, 0x7FFF},
									{18, 0x0B3, 0x7FFF},{19, 0x0F3, 0x7FFF},{20, 0x133, 0x7FFF},
									{21, 0x173, 0x7FFF},{22, 0x1B3, 0x7FFF},{23, 0x1F3, 0x43FF},
									{24, 0x3F3, 0x43FF},{25, 0x03B, 0x7C3F} 									
								};
 

//////////
// Function Name : DMC_SetQosConfig
// Function Description : setting function for Memory control
// Input : 	
//
// Output : none
// Version : v0.0
void DMC_SetQosParameter(u8 ucDmcCh, TRANS_MASTER eMaster,u8 ucFastQosEn,u16 usFastQosCnt, u8 ucQosEn, u16 usQosCnt)
{
	u32 uQosControl;
	u32 uQosConfig;
	u32 uConControl;
	u32 *puTemp0, *puTemp1,*puConTemp;
	u8 ucQosTempIndex;
	
	if(ucDmcCh==0)
		g_pDmcRegs = (DMC_REGS *)DMC0_BASE;
	else
		g_pDmcRegs = (DMC_REGS *)DMC1_BASE;	

	if(eMaster==eFIMD_W0_TM) 			ucQosTempIndex=4;
	else if(eMaster==eFIMD_W1_TM)		ucQosTempIndex=5;
	else if(eMaster==eFIMD_W2_TM) 		ucQosTempIndex=6;
	else if(eMaster==eFIMD_W3_TM) 		ucQosTempIndex=7;
	else if(eMaster==eFIMD_W4_TM) 		ucQosTempIndex=8;
	else if(eMaster==eFIMC0_TM) 		ucQosTempIndex=9;
	else if(eMaster==eFIMC1_TM)	 		ucQosTempIndex=10;
	else if(eMaster==eFIMC2_TM) 		ucQosTempIndex=11;
	else if(eMaster==eVP_TM) 			ucQosTempIndex=13;
	else if(eMaster==eMIXER_GRP0_TM) 	ucQosTempIndex=14;
	else if(eMaster==eMIXER_GRP1_TM) 	ucQosTempIndex=15;
	else 							 	ucQosTempIndex=0;


	puConTemp = &g_pDmcRegs->rDMC_CON_CONTROL;
	puTemp0   = &g_pDmcRegs->rDMC_QOS_CONTROL0 + ucQosTempIndex*2;
	puTemp1   = &g_pDmcRegs->rDMC_QOS_CONFIG0 + ucQosTempIndex*2;

	uConControl = dmcInp32(&g_pDmcRegs->rDMC_CON_CONTROL);
	uConControl &= ~(1<<11);
	uConControl |= (ucFastQosEn<<11);
	uQosControl = (usQosCnt<<16)|(usFastQosCnt<<4) | (ucQosEn);

	if (ucDmcCh==0)
		uQosConfig = (g_aDmc0QosId[eMaster][2]<<16) | (g_aDmc0QosId[eMaster][1]);
	else
		uQosConfig = (g_aDmc1QosId[eMaster][2]<<16) | (g_aDmc1QosId[eMaster][1]);
		
	dmcOutp32(puConTemp, uConControl);	
	dmcOutp32(puTemp0, uQosControl);
	dmcOutp32(puTemp1, uQosConfig);
}



void DMC_ChangeDramParameter(u32 uDMC , u32 uParam)
{

}


void DMC_PreChangeDramParameter(void)
{
	
}

void DMC_PhyDriving(u8 ucDmcCh, u8 ucDrven, u8 ucType)
{
	u32 uConControl;
	u32 uPhyControl1;
	u32 uTimingData;
	u32 uCas;
	
	if(ucDmcCh ==0)
		g_pDmcRegs = (DMC_REGS *)DMC0_BASE;
	else
		g_pDmcRegs = (DMC_REGS *)DMC1_BASE;

	
	uConControl  = dmcInp32(&g_pDmcRegs->rDMC_CON_CONTROL);
	uPhyControl1 = dmcInp32(&g_pDmcRegs->rDMC_PHY_CONTROL1);	
	uTimingData  = dmcInp32(&g_pDmcRegs->rDMC_TIMING_DATA);

	uCas = ((uTimingData>>16) & 0xf);
	if(uCas == 2)
		{
		uConControl  = ((uConControl & (~(0x1<<7)))| (ucDrven<<7));
		uPhyControl1 = ((uPhyControl1 & (~(0x1<<15))) | (1<<15));// Cas2,Read Data Latency 2 case Pulldown all
		}
	else
		{
		uConControl  = ((uConControl & (~(0x1<<7)))| (ucDrven<<7));
		uPhyControl1 = ((uPhyControl1 & (~(0x1<<15))) | (ucType<<15));
		}		
	dmcOutp32(&g_pDmcRegs->rDMC_PHY_CONTROL1, uPhyControl1);
	dmcOutp32(&g_pDmcRegs->rDMC_CON_CONTROL, uConControl);

}


void DMC_QosClean(u8 ucDmcCh,u8 ucOffsetd,u8 ucOffsetc,u8 ucShiftc)
{

	u32 uPhyControl1;
	u32 uTimingData;
	u32 uCas;
	
	if(ucDmcCh ==0)
		g_pDmcRegs = (DMC_REGS *)DMC0_BASE;
	else
		g_pDmcRegs = (DMC_REGS *)DMC1_BASE;

	
	uPhyControl1 = dmcInp32(&g_pDmcRegs->rDMC_PHY_CONTROL1);	
	uPhyControl1 = ((uPhyControl1 & 0xFF8080F8) | (ucOffsetd<<16)|(ucOffsetc<<8)|(ucShiftc<<0));
	
	dmcOutp32(&g_pDmcRegs->rDMC_PHY_CONTROL1, uPhyControl1);

}

void DMC_ChangeRef(u32 uDMC0Tref,u32 uDMC1Tref)
{

	u32 uTimingData;
	
		g_pDmcRegs = (DMC_REGS *)DMC0_BASE;
	uTimingData = dmcInp32(&g_pDmcRegs->rDMC_TIMING_AREF);	
	uTimingData = ((uTimingData & 0x0) | uDMC0Tref);
	dmcOutp32(&g_pDmcRegs->rDMC_TIMING_AREF, uTimingData);

		g_pDmcRegs = (DMC_REGS *)DMC1_BASE;
	uTimingData = dmcInp32(&g_pDmcRegs->rDMC_TIMING_AREF);	
	uTimingData = ((uTimingData & 0x0) | uDMC1Tref);
	dmcOutp32(&g_pDmcRegs->rDMC_TIMING_AREF, uTimingData);
}

void DMC_DownChangeClk(u32 uDmc0Mux,u32 uDmc0Div,u32 uDmc1Mux,u32 uDmc1Div )
{
	
	SYSC_SetClkMuxState(uDmc0Mux);
	SYSC_SetClkDiv(eCLKDIV_ONEDRAM, uDmc0Div);
//	SYSC_SetClkMuxState(uDmc1Mux);              //Default APLL
	SYSC_SetClkDiv(eCLKDIV_HCLKMSYS, uDmc1Div); //200Mhz@800 ARMCLK / (HCLK_MSYS_RATIO + 1)
}

void DMC_HclkChangeClk(u32 uDmc0Div)
{
	
            //Default APLL

}

void DMC_SetDQSn(u8 ucDmcCh, u8 ucDQSn)
{
	u32 uPhyControl0;

	if(ucDmcCh==0)
	g_pDmcRegs = (DMC_REGS *)DMC0_BASE;
	else
	g_pDmcRegs = (DMC_REGS *)DMC1_BASE;	

	uPhyControl0 = dmcInp32(&g_pDmcRegs->rDMC_PHY_CONTROL0);	
	uPhyControl0 = ((uPhyControl0 & ~(0x8)) | ucDQSn<<3);
	dmcOutp32(&g_pDmcRegs->rDMC_PHY_CONTROL0, uPhyControl0);	


}

void DMC_DriveStrength(u32 uDrv, u32 uSlew) // Default 0XAAAA
{
	u32 uTemp=0,uTemp_strength,uTemp_slewrate;
	u32 uPort;
    u32 Strength_Addr;
    u32 uMP1_0DRV = 0xE02003CC;
	u32 uMP2_0DRV = 0xE02004EC;

	uTemp_slewrate = uSlew;
	uTemp_strength = uDrv;

	switch (uTemp_strength)
	{
		case 0 : uDrv = 0x0000;
			break;
		case 1 : uDrv = 0x5555;
			break;
		case 2 : uDrv = 0xAAAA;
			break;
		case 3 : uDrv = 0xFFFF;
			break;
		default : uDrv = 0xAAAA;
			break;
	}
	switch (uTemp_slewrate)
	{
		case 0 : uSlew = 0x00;
			break;
		case 1 : uSlew = 0x11;
			break;
		default : uSlew = 0x00;
			break;
	}


for(uPort=0; uPort<9; uPort++)
	{
       if(uDmcCh ==0)
       	{
       	Strength_Addr = (uMP1_0DRV + uPort*0x20);
    	Outp32( Strength_Addr , uTemp =(uDrv<<0|uSlew<<16)) ; 
       	}
	   else
	   	{
		 Strength_Addr = (uMP2_0DRV + uPort*0x20);
		 Outp32( Strength_Addr , uTemp = (uDrv<<0|uSlew<<16)) ; 
	    }
	}
}

//========================================================
//[KAC007008M , ] MCP(LPDDR+OneDram)4G1G1G Memory Setting
// CHIP_BASE: 0x20(0x2000_0000), CHIP_MASK: 0xF8(1G,128MByte), CHIP_MAP : 0(0:Linear,1:Interleaved,2:Mixed)
// Bank: 4bank, Row: A0~A13, Column: A0~A8
// tRFC 72ns, tRRD 12ns, tRP 18ns, tRCD 18ns, tRC 60ns, tRAS 42ns
// tWR 12ns, trtp , tWTR 1tCK, tCKE 2tCK, tMRD 2tCK
//=======================================================
void DMC_Init_McpACtype(void) // MCP 2g1g1g    A-port:DDRx16,B-port:DDRx32
{
		u32 x, y, i;
		u32 DRAM_KHZ, DRAM_tCK;
		u32 trfc,trrd,trp,trcd,trc,tras,twtr,twr,trtp,tcl,twl,trl,tfaw,txsr,txp,tcke,tmrd, tref;
		

		if(uDmcCh ==0)
			{
				DRAM_KHZ = DRAM0_KHZ;
				DRAM_tCK = DRAM0_tCK;

				trfc = (10*(DRAM_KHZ*120)/1000000+9)/10; //	auto refresh to active - max.0xff
				trrd = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	active bank A to B delay - max.0xf
				trp  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+9)/10; //	precharge comm. period - max.0xf
				trcd = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+9)/10; //	active to read or write delay - max.oxf
				trc  = (10*(DRAM_KHZ*60)/1000000+9)/10; //	active to active period - max.0x3f
				tras = (10*(DRAM_KHZ*42)/1000000+9)/10; //	active to precharge comm. period - max.3f
				twtr = (10*(DRAM_KHZ*1*DRAM_tCK)/1000000+9)/10; //	internal write to read comm. delay - max.0xf
				twr  = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	write recovery time - max.0xf
				trtp = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+9)/10; //	internal read to precharge comm. delay - max.0xf
				tcl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10;//	CAS latency - max.0xf
				twl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	write data latency - max.0xf
				trl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	read data latency - max.0xf
				tfaw = (10*(DRAM_KHZ*20)/1000000+9)/10; //	four activate window - max.0x3f
				txsr = (10*(DRAM_KHZ*120)/1000000+9)/10;	//	self refresh exit power down to next valid comm. delay - max.0xff
				txp  = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10;	//	exit power down to next valid comm. delay - max.0xff
				tcke = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	CKE min. pulse width - max.0xf
				tmrd = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	mode register set comm. period - max.0xf
				tref = (10*(DRAM_KHZ/10000)*78+9)/10;
			}
		else if(uDmcCh ==1)
			{
				DRAM_KHZ = DRAM1_KHZ;
				DRAM_tCK = DRAM1_tCK;
		
				trfc = (10*(DRAM_KHZ*120)/1000000+9)/10; // auto refresh to active - max.0xff
				trrd = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	active bank A to B delay - max.0xf
				trp  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+9)/10; //	precharge comm. period - max.0xf
				trcd = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+9)/10; //	active to read or write delay - max.oxf
				trc  = (10*(DRAM_KHZ*55)/1000000+9)/10; //	active to active period - max.0x3f
				tras = (10*(DRAM_KHZ*40)/1000000+9)/10; //	active to precharge comm. period - max.3f
				twtr = (10*(DRAM_KHZ*1*DRAM_tCK)/1000000+9)/10; //	internal write to read comm. delay - max.0xf
				twr  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+9)/10; //	write recovery time - max.0xf
				trtp = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+9)/10; //	internal read to precharge comm. delay - max.0xf
				tcl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10;//	CAS latency - max.0xf
				twl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	write data latency - max.0xf
				trl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	read data latency - max.0xf
				tfaw = (10*(DRAM_KHZ*20)/1000000+9)/10; //	four activate window - max.0x3f
				txsr = (10*(DRAM_KHZ*120)/1000000+9)/10;	//	self refresh exit power down to next valid comm. delay - max.0xff
				txp  = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	exit power down to next valid comm. delay - max.0xff
				tcke = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	CKE min. pulse width - max.0xf
				tmrd = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	mode register set comm. period - max.0xf	
				tref = (10*(DRAM_KHZ/10000)*78+9)/10;
				}
	//[DMC PHY Initialize]
	// PHY control reg - DLL Setting
#ifdef  ENABLE_PHY_DLL
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, ((CTRL_FORCE<<24)|(CTRL_INC<<16)|(CTRL_START_POINT<<8)|(0<<4)|(CTRL_DDR<<3)|(CTRL_HALF<<2)|(CTRL_DLL_ON<<1)|(CTRL_START)) );//evt0 Diff DQS enable
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, ((CTRL_OFFSETD<<16)|(DRV_TYPE<<15)|(CTRL_OFFSETC<<8)|(CTRL_REF<<4)|(CTRL_SHIFTC)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, ((CTRL_OFFSET3<<24)|(CTRL_OFFSET2<<16)|(CTRL_OFFSET1<<8)|(CTRL_OFFSET0)) );
	
		x = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<1)) ); // DLL on
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<0)) ); // DLL start
	
		do { x = dmcInp32( &g_pDmcRegs->rDMC_PHY_STATUS0); } while( (x&0x4) != 0x4 ); // wait until PLL lock.
	
		x = (x>>6) & (0xff); // get locked value of DLL
		y = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y = y & ~(0xff<<24) | (x<<24)) ) ; // force lock value for DLL off
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y &= ~(1<<1)) ); // DLL off
	
#else

		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, 0x0 );
#endif


	//1)Memory Setting
	
		//	TIMEOUT_CNT<<0xFF0, RD_FETCH<<1, QOS_FAST_EN<<0, DQ_SWAP<<0, PRE_PRE_EN<<0, PRE_ACT_EN:0, DRV_EN:0, CTC_RTR_GAP_EN:0, AREF_EN:0, OUT_OF:4, CLK_RATIO<<1; // auto refresh off											
		//	BL<<20, NUM_CHIP<<16, MEM_WIDTH<<12,MEM_TYPE<<8)|(ADD_LAT_PALL<<6,DSREF_EN<<5, FP_EN<<4, DPWRDN_TYPE<<2, DPWRDN_EN<<1, CLK_STOP_EN;
		//	CHIP_BASE0<<24, CHIP_MASK0<<16, CHIP_MAP0<<12, CHIP_COL0<<8,CHIP_ROW0<<4,CHIP_BANK0;
		//	CHIP_BASE1<<24, CHIP_MASK1<<16, CHIP_MAP1<<12,CHIP_COL1<<8,CHIP_ROW1<<4,CHIP_BANK1
		//	FP_CNT<<24,CHIP1_POLICY<<8,CHIP0_POLICY)
		//	DSREF_CYC<<16, DPWRDN_CYC	
		//	T_REFI(t_REFI < t_refi*T, 7.8us*100Mhz=780(0x30c)), 7.8us*20Mhz =0x9c
		//	T_RFC<<24,T_RRD<<20,T_RP<<16,T_RCD<<12,T_RC<<6,T_RAS
		//	T_WTR<<28,T_WR<<24,T_RTP<<20,T_CL<<16,T_WL<<8,T_RL
		//	T_FAW<<24,T_XSR<<16,T_XP<<8,T_CKE<<4,T_MRD
		

	
	
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, ((0xFF0<<16)|(1<<12)|(0<<11)|(0<<10)|(0<<9)|(0<<8)|(1<<7)|(0<<6)|(0<<5)|(1<<4)|(0<<1)|(0)) ); // auto refresh off
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, ((2<<20)|(1<<16)|(2<<12)|(1<<8)|(0<<6)|(0<<5)|(0<<4)|(0<<2)|(0<<1)|(0)) );
		if(uDmcCh ==0)
			{
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG0, ((0x20<<24)|(0xF8<<16)|(2<<12)|(2<<8)|(2<<4)|(2)));
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG1, ((0x30<<24)|(0xF8<<16)|(2<<12)|(2<<8)|(2<<4)|(2)) );
			}
		else if(uDmcCh ==1)
			{
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG0, ((0x40<<24)|(0xF8<<16)|(1<<12)|(2<<8)|(2<<4)|(2)));
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG1, ((0x50<<24)|(0xF8<<16)|(1<<12)|(2<<8)|(2<<4)|(2)) );
			}

		dmcOutp32( &g_pDmcRegs->rDMC_PRECH_CONFIG, ((0xFF<<24)|(0<<8)|(0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PWRDN_CONFIG, ((0xFFFF<<16)|(0xFF<<0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_AREF, (tref) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_ROW, ((trfc<<24)|(trrd<<20)|(trp<<16)|(trcd<<12)|(trc<<6)|(tras)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_DATA, ((twtr<<28)|(twr<<24)|(trtp<<20)|(tcl<<16)|(twl<<8)|(trl)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_POWER, ((tfaw<<24)|(txsr<<16)|(txp<<8)|(tcke<<4)|(tmrd)));
	
	//2) initialize QoS 
	for ( i=0; i<32; i+=2 )
		dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONTROL0+i, 0x00ff0000 );
	for ( i=0; i<32; i+=2 )
		dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONFIG0+i, 0x00000000 );
	
	//3) Direct command
		//Chip0 
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07000000 ); // chip0 Deselect (NOP)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01000000 ); // chip0 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00000032 ); // chip0 MRS (CL=3, BL=4)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00020000 ); // chip0 EMRS
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07000000 ); // chip0 Deselect (NOP)
		//Chip1 
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07100000 ); // chip1 Deselect(NOP)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01100000 ); // chip1 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00100032 ); // chip1 MRS, CL=3, BL=4
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00120000 ); // chip1   EMRS
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07100000 ); // chip1 Deselect(NOP)
	
	//4) change control for using memory.
		x = dmcInp32( &g_pDmcRegs->rDMC_CON_CONTROL);
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, (x |= 1<<5) );
		x = dmcInp32( &g_pDmcRegs->rDMC_MEM_CONTROL);
		x &=~0xff;
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, (x |= (1<<4)|(1<<1)|(1<<0))); //Timeout Precharge, DPD(Active/Pre), DCC enable

	
}

//========================================================
//[KAT00200DM] MCP(LPDDR+LPDDR)4G1G1G Memory Setting
// CHIP_BASE: 0x20(0x2000_0000), CHIP_MASK: 0xF8(1G,128MByte), CHIP_MAP : 1(0:Linear,1:Interleaved,2:Mixed)
// Bank: 4bank, Row: A0~A13, Column: A0~A8
// tRFC 72ns, tRRD 12ns, tRP 18ns, tRCD 18ns, tRC 60ns, tRAS 42ns
// tWR 12ns, trtp , tWTR 1tCK, tCKE 2tCK, tMRD 2tCK
//=======================================================
void DMC_Init_McpHtype(void) // MCP 1g1g1g    A-port:DDRx16,B-port:DDRx32
{
		u32 x, y, i;
		u32 DRAM_KHZ, DRAM_tCK;
		u32 trfc,trrd,trp,trcd,trc,tras,twtr,twr,trtp,tcl,twl,trl,tfaw,txsr,txp,tcke,tmrd, tref;


				DRAM_KHZ = DRAM1_KHZ;
				DRAM_tCK = DRAM1_tCK;

		trfc = (10*(DRAM_KHZ*72)/1000000+9)/10; //	auto refresh to active - max.0xff
		trrd = (10*(DRAM_KHZ*12)/1000000+9)/10; //	active bank A to B delay - max.0xf
		trp  = (10*(DRAM_KHZ*18)/1000000+9)/10; //	precharge comm. period - max.0xf
		trcd = (10*(DRAM_KHZ*18)/1000000+9)/10; //	active to read or write delay - max.oxf
		trc  = (10*(DRAM_KHZ*60)/1000000+9)/10; //	active to active period - max.0x3f
		tras = (10*(DRAM_KHZ*42)/1000000+9)/10; //	active to precharge comm. period - max.3f
		twtr = (10*(DRAM_KHZ*1*DRAM_tCK)/1000000+9)/10; //	internal write to read comm. delay - max.0xf
		twr  = (10*(DRAM_KHZ*12)/1000000+9)/10; //	write recovery time - max.0xf
		trtp = (10*(DRAM_KHZ*18)/1000000+9)/10; //	internal read to precharge comm. delay - max.0xf
		tcl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10;//	CAS latency - max.0xf
		twl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	write data latency - max.0xf
		trl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	read data latency - max.0xf
		tfaw = (10*(DRAM_KHZ*20)/1000000+9)/10; //	four activate window - max.0x3f
		txsr = (10*(DRAM_KHZ*120)/1000000+9)/10;	//	self refresh exit power down to next valid comm. delay - max.0xff
		txp  = (10*(DRAM_KHZ*120)/1000000+9)/10;	//	exit power down to next valid comm. delay - max.0xff
		tcke = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	CKE min. pulse width - max.0xf
		tmrd = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	mode register set comm. period - max.0xf	
		tref = (10*(DRAM_KHZ/10000)*78+9)/10;
		
	//[DMC PHY Initialize]
	// PHY control reg - DLL Setting
#ifdef  ENABLE_PHY_DLL
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, ((CTRL_FORCE<<24)|(CTRL_INC<<16)|(CTRL_START_POINT<<8)|(0<<4)|(CTRL_DDR<<3)|(CTRL_HALF<<2)|(CTRL_DLL_ON<<1)|(CTRL_START)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, ((CTRL_OFFSETD<<16)|(DRV_TYPE<<15)|(CTRL_OFFSETC<<8)|(CTRL_REF<<4)|(CTRL_SHIFTC)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, ((CTRL_OFFSET3<<24)|(CTRL_OFFSET2<<16)|(CTRL_OFFSET1<<8)|(CTRL_OFFSET0)) );
	
		x = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<1)) ); // DLL on
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<0)) ); // DLL start
	
		do { x = dmcInp32( &g_pDmcRegs->rDMC_PHY_STATUS0); } while( (x&0x4) != 0x4 ); // wait until PLL lock.
	
		x = (x>>6) & (0xff); // get locked value of DLL
		y = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y = y & ~(0xff<<24) | (x<<24)) ) ; // force lock value for DLL off
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y &= ~(1<<1)) ); // DLL off
	
#else
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, 0x0 );
#endif


	//1)Memory Setting
	
		//	TIMEOUT_CNT<<0xFF0, RD_FETCH<<1, QOS_FAST_EN<<0, DQ_SWAP<<0, PRE_PRE_EN<<0, PRE_ACT_EN:0, DRV_EN:0, CTC_RTR_GAP_EN:0, AREF_EN:0, OUT_OF:4, CLK_RATIO<<1; // auto refresh off											
		//	BL<<20, NUM_CHIP<<16, MEM_WIDTH<<12,MEM_TYPE<<8)|(ADD_LAT_PALL<<6,DSREF_EN<<5, FP_EN<<4, DPWRDN_TYPE<<2, DPWRDN_EN<<1, CLK_STOP_EN;
		//	CHIP_BASE0<<24, CHIP_MASK0<<16, CHIP_MAP0<<12, CHIP_COL0<<8,CHIP_ROW0<<4,CHIP_BANK0;
		//	CHIP_BASE1<<24, CHIP_MASK1<<16, CHIP_MAP1<<12,CHIP_COL1<<8,CHIP_ROW1<<4,CHIP_BANK1
		//	FP_CNT<<24,CHIP1_POLICY<<8,CHIP0_POLICY)
		//	DSREF_CYC<<16, DPWRDN_CYC	
		//	T_REFI(t_REFI < t_refi*T, 7.8us*100Mhz=780(0x30c)), 7.8us*20Mhz =0x9c
		//	T_RFC<<24,T_RRD<<20,T_RP<<16,T_RCD<<12,T_RC<<6,T_RAS
		//	T_WTR<<28,T_WR<<24,T_RTP<<20,T_CL<<16,T_WL<<8,T_RL
		//	T_FAW<<24,T_XSR<<16,T_XP<<8,T_CKE<<4,T_MRD
		

	
	
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, ((0xFF0<<16)|(1<<12)|(0<<11)|(0<<10)|(0<<9)|(0<<8)|(1<<7)|(0<<6)|(0<<5)|(1<<4)|(0<<1)|(0)) ); // auto refresh off
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, ((2<<20)|(1<<16)|(2<<12)|(1<<8)|(0<<6)|(0<<5)|(0<<4)|(0<<2)|(0<<1)|(0)) );
		if(uDmcCh ==0)
			{
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG0, ((0x20<<24)|(0xF8<<16)|(1<<12)|(3<<8)|(1<<4)|(2)));
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG1, ((0x30<<24)|(0xF8<<16)|(1<<12)|(3<<8)|(1<<4)|(2)) );
			}
		else if(uDmcCh ==1)
			{
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG0, ((0x40<<24)|(0xF8<<16)|(1<<12)|(3<<8)|(1<<4)|(2)));
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG1, ((0x50<<24)|(0xF8<<16)|(1<<12)|(3<<8)|(1<<4)|(2)) );
			}

		dmcOutp32( &g_pDmcRegs->rDMC_PRECH_CONFIG, ((0xFF<<24)|(0<<8)|(0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PWRDN_CONFIG, ((0xFFFF<<16)|(0xFF<<0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_AREF, (tref) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_ROW, ((trfc<<24)|(trrd<<20)|(trp<<16)|(trcd<<12)|(trc<<6)|(tras)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_DATA, ((twtr<<28)|(twr<<24)|(trtp<<20)|(tcl<<16)|(twl<<8)|(trl)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_POWER, ((tfaw<<24)|(txsr<<16)|(txp<<8)|(tcke<<4)|(tmrd)));
	
	//2) initialize QoS 
	for ( i=0; i<32; i+=2 )
		dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONTROL0+i, 0x00ff0000 );
	for ( i=0; i<32; i+=2 )
		dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONFIG0+i, 0x00000000 );
	
	//3) Direct command
		//Chip0 
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07000000 ); // chip0 Deselect (NOP)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01000000 ); // chip0 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00000032 ); // chip0 MRS (CL=3, BL=4)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00020000 ); // chip0 EMRS
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07000000 ); // chip0 Deselect (NOP)
		//Chip1 
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07100000 ); // chip1 Deselect(NOP)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01100000 ); // chip1 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00100032 ); // chip1 MRS, CL=3, BL=4
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00120000 ); // chip1   EMRS
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07100000 ); // chip1 Deselect(NOP)
	
	//4) change control for using memory.
		x = dmcInp32( &g_pDmcRegs->rDMC_CON_CONTROL);
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, (x |= 1<<5) );
		x = dmcInp32( &g_pDmcRegs->rDMC_MEM_CONTROL);
		x &=~0xff;
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, (x |= (1<<4)|(1<<1)|(1<<0))); //Timeout Precharge, DPD(Active/Pre), DCC enable
	
}


void DMC_Assem_Init_McpHtype(void) // MCP 1g1g1g    A-port:DDRx16,B-port:DDRx32
{
		u32 x, y, i,uCh;
		u32 DRAM_KHZ, DRAM_tCK;
		u32 trfc,trrd,trp,trcd,trc,tras,twtr,twr,trtp,tcl,twl,trl,tfaw,txsr,txp,tcke,tmrd, tref;


				DRAM_KHZ = 200000;
				DRAM_tCK = ((10*1000000/DRAM_KHZ)+9)/10;

		trfc = (10*(DRAM_KHZ*72)/1000000+9)/10; //	auto refresh to active - max.0xff
		trrd = (10*(DRAM_KHZ*12)/1000000+9)/10; //	active bank A to B delay - max.0xf
		trp  = (10*(DRAM_KHZ*18)/1000000+9)/10; //	precharge comm. period - max.0xf
		trcd = (10*(DRAM_KHZ*18)/1000000+9)/10; //	active to read or write delay - max.oxf
		trc  = (10*(DRAM_KHZ*60)/1000000+9)/10; //	active to active period - max.0x3f
		tras = (10*(DRAM_KHZ*42)/1000000+9)/10; //	active to precharge comm. period - max.3f
		twtr = (10*(DRAM_KHZ*1*DRAM_tCK)/1000000+9)/10; //	internal write to read comm. delay - max.0xf
		twr  = (10*(DRAM_KHZ*12)/1000000+9)/10; //	write recovery time - max.0xf
		trtp = (10*(DRAM_KHZ*18)/1000000+9)/10; //	internal read to precharge comm. delay - max.0xf
		tcl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10;//	CAS latency - max.0xf
		twl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	write data latency - max.0xf
		trl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	read data latency - max.0xf
		tfaw = (10*(DRAM_KHZ*20)/1000000+9)/10; //	four activate window - max.0x3f
		txsr = (10*(DRAM_KHZ*120)/1000000+9)/10;	//	self refresh exit power down to next valid comm. delay - max.0xff
		txp  = (10*(DRAM_KHZ*120)/1000000+9)/10;	//	exit power down to next valid comm. delay - max.0xff
		tcke = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	CKE min. pulse width - max.0xf
		tmrd = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	mode register set comm. period - max.0xf	
		tref = (10*(DRAM_KHZ/10000)*78+9)/10;

		for(uCh=0;uCh<2;uCh++)
			{
		switch (uCh){
				case 0 : g_pDmcRegs = (DMC_REGS *)DMC0_BASE;break;
				case 1 : g_pDmcRegs = (DMC_REGS *)DMC1_BASE;break;
			}
	//[DMC PHY Initialize]
	// PHY control reg - DLL Setting
#ifdef  ENABLE_PHY_DLL
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, ((CTRL_FORCE<<24)|(CTRL_INC<<16)|(CTRL_START_POINT<<8)|(0<<4)|(CTRL_DDR<<3)|(CTRL_HALF<<2)|(CTRL_DLL_ON<<1)|(CTRL_START)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, ((CTRL_OFFSETD<<16)|(DRV_TYPE<<15)|(CTRL_OFFSETC<<8)|(CTRL_REF<<4)|(CTRL_SHIFTC)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, ((CTRL_OFFSET3<<24)|(CTRL_OFFSET2<<16)|(CTRL_OFFSET1<<8)|(CTRL_OFFSET0)) );
	
		x = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<1)) ); // DLL on
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<0)) ); // DLL start
	
		do { x = dmcInp32( &g_pDmcRegs->rDMC_PHY_STATUS0); } while( (x&0x4) != 0x4 ); // wait until PLL lock.
	
		x = (x>>6) & (0xff); // get locked value of DLL
		y = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y = y & ~(0xff<<24) | (x<<24)) ) ; // force lock value for DLL off
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y &= ~(1<<1)) ); // DLL off
	
#else
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, 0x0 );
#endif


	//1)Memory Setting
	
		//	TIMEOUT_CNT<<0xFF0, RD_FETCH<<1, QOS_FAST_EN<<0, DQ_SWAP<<0, PRE_PRE_EN<<0, PRE_ACT_EN:0, DRV_EN:0, CTC_RTR_GAP_EN:0, AREF_EN:0, OUT_OF:4, CLK_RATIO<<1; // auto refresh off											
		//	BL<<20, NUM_CHIP<<16, MEM_WIDTH<<12,MEM_TYPE<<8)|(ADD_LAT_PALL<<6,DSREF_EN<<5, FP_EN<<4, DPWRDN_TYPE<<2, DPWRDN_EN<<1, CLK_STOP_EN;
		//	CHIP_BASE0<<24, CHIP_MASK0<<16, CHIP_MAP0<<12, CHIP_COL0<<8,CHIP_ROW0<<4,CHIP_BANK0;
		//	CHIP_BASE1<<24, CHIP_MASK1<<16, CHIP_MAP1<<12,CHIP_COL1<<8,CHIP_ROW1<<4,CHIP_BANK1
		//	FP_CNT<<24,CHIP1_POLICY<<8,CHIP0_POLICY)
		//	DSREF_CYC<<16, DPWRDN_CYC	
		//	T_REFI(t_REFI < t_refi*T, 7.8us*100Mhz=780(0x30c)), 7.8us*20Mhz =0x9c
		//	T_RFC<<24,T_RRD<<20,T_RP<<16,T_RCD<<12,T_RC<<6,T_RAS
		//	T_WTR<<28,T_WR<<24,T_RTP<<20,T_CL<<16,T_WL<<8,T_RL
		//	T_FAW<<24,T_XSR<<16,T_XP<<8,T_CKE<<4,T_MRD
		

	
	
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, ((0xFF0<<16)|(1<<12)|(0<<11)|(0<<10)|(0<<9)|(0<<8)|(1<<7)|(0<<6)|(0<<5)|(1<<4)|(0<<1)|(0)) ); // auto refresh off
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, ((2<<20)|(1<<16)|(2<<12)|(1<<8)|(0<<6)|(0<<5)|(0<<4)|(0<<2)|(0<<1)|(0)) );
		if(uCh ==0)
			{
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG0, ((0x20<<24)|(0xF8<<16)|(1<<12)|(3<<8)|(1<<4)|(2)));
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG1, ((0x30<<24)|(0xF8<<16)|(1<<12)|(3<<8)|(1<<4)|(2)) );
			}
		else if(uCh ==1)
			{
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG0, ((0x40<<24)|(0xF8<<16)|(1<<12)|(3<<8)|(1<<4)|(2)));
			dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG1, ((0x50<<24)|(0xF8<<16)|(1<<12)|(3<<8)|(1<<4)|(2)) );
			}

		dmcOutp32( &g_pDmcRegs->rDMC_PRECH_CONFIG, ((0xFF<<24)|(0<<8)|(0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PWRDN_CONFIG, ((0xFFFF<<16)|(0xFF<<0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_AREF, (tref) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_ROW, ((trfc<<24)|(trrd<<20)|(trp<<16)|(trcd<<12)|(trc<<6)|(tras)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_DATA, ((twtr<<28)|(twr<<24)|(trtp<<20)|(tcl<<16)|(twl<<8)|(trl)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_POWER, ((tfaw<<24)|(txsr<<16)|(txp<<8)|(tcke<<4)|(tmrd)));
	
	//2) initialize QoS 
	for ( i=0; i<32; i+=2 )
		dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONTROL0+i, 0x00ff0000 );
	for ( i=0; i<32; i+=2 )
		dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONFIG0+i, 0x00000000 );
	
	//3) Direct command
		//Chip0 
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07000000 ); // chip0 Deselect (NOP)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01000000 ); // chip0 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00000032 ); // chip0 MRS (CL=3, BL=4)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00020000 ); // chip0 EMRS
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07000000 ); // chip0 Deselect (NOP)
		//Chip1 
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07100000 ); // chip1 Deselect(NOP)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01100000 ); // chip1 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00100032 ); // chip1 MRS, CL=3, BL=4
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00120000 ); // chip1   EMRS
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07100000 ); // chip1 Deselect(NOP)
	
	//4) change control for using memory.
		x = dmcInp32( &g_pDmcRegs->rDMC_CON_CONTROL);
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, (x |= 1<<5) );
		x = dmcInp32( &g_pDmcRegs->rDMC_MEM_CONTROL);
		x &=~0xff;
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, (x |= (1<<4)|(1<<1)|(1<<0))); //Timeout Precharge, DPD(Active/Pre), DCC enable
	
	}
}

//========================================================
//[K4T2G084QA] DDR2 Memory Setting
// CHIP_BASE: 0x40(0x4000_0000), CHIP_MASK: 0xF0(1G,128MByte), CHIP_MAP : 0(0:Linear,1:Interleaved,2:Mixed)
// Bank: 8bank, Row: A0~A14, Column: A0~A9
// tRFC 195ns, tRRD 10ns, tRP 15ns, tRCD 15ns, tRC 60ns, tRAS 45ns
// tWR 15ns, trtp 7.5ns, tWTR 7.5ns, tFAW 50ns(four Activate Window for 2KB Page Size product)
// tWTR 2tCK, tCL 3tCK, tWL 3tCK, tRL 3tCK, tXP 2tCK, tCKE 3tCK, tMRD 2tCK
//========================================================

void DMC_Init_K4T2G084QA(void) // DDR2 2Gbit    *4ea
{
	u32 x, y, i;
	u32 DRAM_KHZ, DRAM_tCK;
	u32 trfc,trrd,trp,trcd,trc,tras,twtr,twr,trtp,tcl,twl,trl,tfaw,txsr,txp,tcke,tmrd, tref;
	
	
			DRAM_KHZ = DRAM1_KHZ;
			DRAM_tCK = DRAM1_tCK;

	
	 trfc = (10*(DRAM_KHZ*195)/1000000+9)/10; // auto refresh to active - max.0xff
	 trrd = (10*(DRAM_KHZ*8)/1000000+9)/10; //	active bank A to B delay - max.0xf
	 trp  = (10*(DRAM_KHZ*15)/1000000+9)/10; //	precharge comm. period - max.0xf
	 trcd = (10*(DRAM_KHZ*15)/1000000+9)/10; //	active to read or write delay - max.oxf
	 trc  = (10*(DRAM_KHZ*60)/1000000+9)/10; //	active to active period - max.0x3f
	 tras = (10*(DRAM_KHZ*45)/1000000+9)/10; //	active to precharge comm. period - max.3f
	 twtr = (10*(DRAM_KHZ*8)/1000000+9)/10; //	internal write to read comm. delay - max.0xf
	 twr  = (10*(DRAM_KHZ*15)/1000000+9)/10; //	write recovery time - max.0xf
	 trtp = (10*(DRAM_KHZ*8)/1000000+9)/10; //	internal read to precharge comm. delay - max.0xf
	 tcl  = (10*(DRAM_KHZ*4*DRAM_tCK)/1000000+5)/10;//	CAS latency - max.0xf
	 twl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	write data latency - max.0xf
	 trl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	read data latency - max.0xf
	 tfaw = (10*(DRAM_KHZ*38)/1000000+9)/10; //	four activate window - max.0x3f
	 txsr = (10*(DRAM_KHZ*200*DRAM_tCK)/1000000+9)/10;	//	self refresh exit power down to next valid comm. delay - max.0xff
	 txp  = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	exit power down to next valid comm. delay - max.0xff
	 tcke = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+9)/10; //	CKE min. pulse width - max.0xf
	 tmrd = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	mode register set comm. period - max.0xf

	
	//[DMC PHY Initialize]
	// PHY control reg - DLL Setting
#ifdef  ENABLE_PHY_DLL
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, ((CTRL_FORCE<<24)|(CTRL_INC<<16)|(CTRL_START_POINT<<8)|(0<<4)|(CTRL_DDR<<3)|(CTRL_HALF<<2)|(CTRL_DLL_ON<<1)|(CTRL_START)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, ((CTRL_OFFSETD<<16)|(CTRL_OFFSETC<<8)|(CTRL_REF<<4)|(CTRL_SHIFTC)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, ((CTRL_OFFSET3<<24)|(CTRL_OFFSET2<<16)|(CTRL_OFFSET1<<8)|(CTRL_OFFSET0)) );
	
		x = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<1)) ); // DLL on
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<0)) ); // DLL start
	
		do { x = dmcInp32( &g_pDmcRegs->rDMC_PHY_STATUS0); } while( (x&0x4) != 0x4 ); // wait until PLL lock.
	
		x = (x>>6) & (0xff); // get locked value of DLL
		y = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y = y & ~(0xff<<24) | (x<<24)) ) ; // force lock value for DLL off
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y &= ~(1<<1)) ); // DLL off
	
#else
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, 0x0 );
#endif


	//1)Memory Setting

		//	TIMEOUT_CNT<<0xFF0, RD_FETCH<<1, QOS_FAST_EN<<0, DQ_SWAP<<0, PRE_PRE_EN<<0, PRE_ACT_EN:0, DRV_EN:0, CTC_RTR_GAP_EN:0, AREF_EN:0, OUT_OF:4, CLK_RATIO<<1; // auto refresh off											
		//	BL<<20, NUM_CHIP<<16, MEM_WIDTH<<12,MEM_TYPE<<8)|(ADD_LAT_PALL<<6,DSREF_EN<<5, FP_EN<<4, DPWRDN_TYPE<<2, DPWRDN_EN<<1, CLK_STOP_EN;
		//	CHIP_BASE0<<24, CHIP_MASK0<<16, CHIP_MAP0<<12, CHIP_COL0<<8,CHIP_ROW0<<4,CHIP_BANK0;
		//	CHIP_BASE1<<24, CHIP_MASK1<<16, CHIP_MAP1<<12,CHIP_COL1<<8,CHIP_ROW1<<4,CHIP_BANK1
		//	FP_CNT<<24,CHIP1_POLICY<<8,CHIP0_POLICY)
		//	DSREF_CYC<<16, DPWRDN_CYC	
		//	T_REFI(t_REFI < t_refi*T, 7.8us*100Mhz=780(0x30c))
		//	T_RFC<<24,T_RRD<<20,T_RP<<16,T_RCD<<12,T_RC<<6,T_RAS
		//	T_WTR<<28,T_WR<<24,T_RTP<<20,T_CL<<16,T_WL<<8,T_RL
		//	T_FAW<<24,T_XSR<<16,T_XP<<8,T_CKE<<4,T_MRD
		
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, ((0xFFF<<16)|(1<<12)|(0<<11)|(0<<10)|(0<<9)|(0<<8)|(0<<7)|(0<<6)|(0<<5)|(1<<4)|(0<<1)|(0)) ); // auto refresh off
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, ((2<<20)|(0<<16)|(2<<12)|(4<<8)|(0<<6)|(0<<5)|(0<<4)|(0<<2)|(0<<1)|(0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG0, ((0x40<<24)|(0xE0<<16)|(0<<12)|(3<<8)|(3<<4)|(3)));
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG1, ((0xFF<<24)|(0xE0<<16)|(0<<12)|(3<<8)|(3<<4)|(3)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PRECH_CONFIG, ((0xFF<<24)|(0<<8)|(0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PWRDN_CONFIG, ((0xFFFF<<16)|(0xFF)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_AREF, (0x618) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_ROW, ((trfc<<24)|(trrd<<20)|(trp<<16)|(trcd<<12)|(trc<<6)|(tras)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_DATA, ((twtr<<28)|(twr<<24)|(trtp<<20)|(tcl<<16)|(twl<<8)|(trl)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_POWER, ((tfaw<<24)|(txsr<<16)|(txp<<8)|(tcke<<4)|(tmrd)));
	
	//2) initialize QoS 
		for ( i=0; i<32; i+=2 )
			dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONTROL0+i, 0x00ff0000 );
		for ( i=0; i<32; i+=2 )
			dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONFIG0+i, 0x00000000 );
	
	//3) Direct command
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07000000 ); // chip0 Deselect(NOP)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01000000 ); // chip0 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00020000 ); // chip0 EMRS2
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00030000 ); // chip0 EMRS3
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00010400 ); // chip0 EMRS1 (MEM DLL on, DQS# disable)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00000542 ); // chip0 MRS (MEM DLL reset)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01000000 ); // chip0 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00000442 ); // chip0 MRS (MEM DLL unreset), BL=4
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00010780 ); // chip0 EMRS1 (OCD default)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00010400 ); // chip0 EMRS1 (OCD exit)

		x = dmcInp32( &g_pDmcRegs->rDMC_CON_CONTROL);
		if((x &=0x10000)!=0)
			{
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07100000 ); // chip1 Deselect(NOP)
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01100000 ); // chip1 PALL
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00120000 ); // chip1 EMRS2
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00130000 ); // chip1 EMRS3
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00110400 ); // chip1 EMRS1 (MEM DLL on, DQS# disable)
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00100542 ); // chip1 MRS (MEM DLL reset)
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01100000 ); // chip1 PALL
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00100442 ); // chip1 MRS (MEM DLL unreset), BL=4
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00110780 ); // chip1 EMRS1 (OCD default)
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00110400 ); // chip1 EMRS1 (OCD exit)
			}
	//4) change control for using memory.
		x = dmcInp32( &g_pDmcRegs->rDMC_CON_CONTROL);
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, (x |= 1<<5) );
		x = dmcInp32( &g_pDmcRegs->rDMC_MEM_CONTROL);
		x &=~0xff;
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, (x |= (1<<4)|(1<<1)|(1<<0))); //Timeout Precharge, DPD(Active/Pre), DCC enable
	

	
}

//========================================================
//[K4T1G164QE] DDR2 Memory Setting
// CHIP_BASE: 0x20(0x2000_0000), CHIP_MASK: 0xF0(1G*2ea,256MByte), CHIP_MAP : 0(0:Linear,1:Interleaved,2:Mixed)
// Bank: 8bank, Row: A0~A12, Column: A0~A9
// tRFC 127.5ns, tRRD 10ns, tRP 15ns, tRCD 15ns, tRC 60ns, tRAS 45ns
// tWR 15ns, trtp 7.5ns, tFAW 50ns(four Activate Window for 2KB Page Size product), tWTR 7.5ns
// tWTR 2tCK, tCL 3tCK, tWL 3tCK, tRL 3tCK, tXP 2tCK, tCKE 3tCK, tMRD 2tCK
//========================================================

void DMC_Init_K4T1G084QQ(void) //DDR2 1Gbit *2ea 
{
	u32 x, y, i;
	u32 DRAM_KHZ, DRAM_tCK;
	u32 trfc,trrd,trp,trcd,trc,tras,twtr,twr,trtp,tcl,twl,trl,tfaw,txsr,txp,tcke,tmrd, tref;
	
	
			DRAM_KHZ = DRAM1_KHZ;
			DRAM_tCK = DRAM1_tCK;

	
		 trfc = (10*(DRAM_KHZ*128)/1000000+9)/10; // auto refresh to active - max.0xff
		 trrd = (10*(DRAM_KHZ*8)/1000000+9)/10; //	active bank A to B delay - max.0xf
		 trp  = (10*(DRAM_KHZ*15)/1000000+9)/10; //	precharge comm. period - max.0xf
		 trcd = (10*(DRAM_KHZ*15)/1000000+9)/10; //	active to read or write delay - max.oxf
		 trc  = (10*(DRAM_KHZ*60)/1000000+9)/10; //	active to active period - max.0x3f
		 tras = (10*(DRAM_KHZ*45)/1000000+9)/10; //	active to precharge comm. period - max.3f
		 twtr = (10*(DRAM_KHZ*8)/1000000+9)/10; //	internal write to read comm. delay - max.0xf
		 twr  = (10*(DRAM_KHZ*15)/1000000+9)/10; //	write recovery time - max.0xf
		 trtp = (10*(DRAM_KHZ*8)/1000000+9)/10; //	internal read to precharge comm. delay - max.0xf
		 tcl  = (10*(DRAM_KHZ*4*DRAM_tCK)/1000000+5)/10;//	CAS latency - max.0xf
		 twl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	write data latency - max.0xf
		 trl  = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+5)/10; //	read data latency - max.0xf
		 tfaw = (10*(DRAM_KHZ*38)/1000000+9)/10; //	four activate window - max.0x3f
		 txsr = (10*(DRAM_KHZ*200*DRAM_tCK)/1000000+9)/10;	//	self refresh exit power down to next valid comm. delay - max.0xff
		 txp  = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	exit power down to next valid comm. delay - max.0xff
		 tcke = (10*(DRAM_KHZ*3*DRAM_tCK)/1000000+9)/10; //	CKE min. pulse width - max.0xf
		 tmrd = (10*(DRAM_KHZ*2*DRAM_tCK)/1000000+9)/10; //	mode register set comm. period - max.0xf

	//[DMC PHY Initialize]
	// PHY control reg - DLL Setting
#ifdef  ENABLE_PHY_DLL
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, ((CTRL_FORCE<<24)|(CTRL_INC<<16)|(CTRL_START_POINT<<8)|(0<<4)|(CTRL_DDR<<3)|(CTRL_HALF<<2)|(CTRL_DLL_ON<<1)|(CTRL_START)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, ((CTRL_OFFSETD<<16)|(CTRL_OFFSETC<<8)|(CTRL_REF<<4)|(CTRL_SHIFTC)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, ((CTRL_OFFSET3<<24)|(CTRL_OFFSET2<<16)|(CTRL_OFFSET1<<8)|(CTRL_OFFSET0)) );
	
		x = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<1)) ); // DLL on
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (x |= (1<<0)) ); // DLL start
	
		do { x = dmcInp32( &g_pDmcRegs->rDMC_PHY_STATUS0); } while( (x&0x4) != 0x4 ); // wait until PLL lock.
	
		x = (x>>6) & (0xff); // get locked value of DLL
		y = dmcInp32( &g_pDmcRegs->rDMC_PHY_CONTROL0);
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y = y & ~(0xff<<24) | (x<<24)) ) ; // force lock value for DLL off
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, (y &= ~(1<<1)) ); // DLL off
	
#else
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL0, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL1, 0x0 );
		dmcOutp32( &g_pDmcRegs->rDMC_PHY_CONTROL2, 0x0 );
#endif


	//1)Memory Setting
		//	TIMEOUT_CNT<<0xFF0, RD_FETCH<<1, QOS_FAST_EN<<0, DQ_SWAP<<0, PRE_PRE_EN<<0, PRE_ACT_EN:0, DRV_EN:0, CTC_RTR_GAP_EN:0, AREF_EN:0, OUT_OF:4, CLK_RATIO<<1; // auto refresh off											
		//	BL<<20, NUM_CHIP<<16, MEM_WIDTH<<12,MEM_TYPE<<8)|(ADD_LAT_PALL<<6,DSREF_EN<<5, FP_EN<<4, DPWRDN_TYPE<<2, DPWRDN_EN<<1, CLK_STOP_EN;
		//	CHIP_BASE0<<24, CHIP_MASK0<<16, CHIP_MAP0<<12, CHIP_COL0<<8,CHIP_ROW0<<4,CHIP_BANK0;
		//	CHIP_BASE1<<24, CHIP_MASK1<<16, CHIP_MAP1<<12,CHIP_COL1<<8,CHIP_ROW1<<4,CHIP_BANK1
		//	FP_CNT<<24,CHIP1_POLICY<<8,CHIP0_POLICY)
		//	DSREF_CYC<<16, DPWRDN_CYC	
		//	T_REFI(t_REFI < t_refi*T, 7.8us*100Mhz=780(0x30c))
		//	T_RFC<<24,T_RRD<<20,T_RP<<16,T_RCD<<12,T_RC<<6,T_RAS
		//	T_WTR<<28,T_WR<<24,T_RTP<<20,T_CL<<16,T_WL<<8,T_RL
		//	T_FAW<<24,T_XSR<<16,T_XP<<8,T_CKE<<4,T_MRD


	
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, ((0xFFF<<16)|(1<<12)|(0<<11)|(0<<10)|(0<<9)|(0<<8)|(0<<7)|(0<<6)|(0<<5)|(1<<4)|(0<<1)|(0)) ); // auto refresh off
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, ((2<<20)|(0<<16)|(2<<12)|(4<<8)|(0<<6)|(0<<5)|(0<<4)|(0<<2)|(0<<1)|(0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG0, ((0x20<<24)|(0xE0<<16)|(0<<12)|(3<<8)|(2<<4)|(3)));
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONFIG1, ((0xff<<24)|(0xE0<<16)|(0<<12)|(3<<8)|(2<<4)|(3)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PRECH_CONFIG, ((0xFF<<24)|(0<<8)|(0)) );
		dmcOutp32( &g_pDmcRegs->rDMC_PWRDN_CONFIG, ((0xFFFF<<16)|(0xFF)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_AREF, (0x618) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_ROW, ((trfc<<24)|(trrd<<20)|(trp<<16)|(trcd<<12)|(trc<<6)|(tras)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_DATA, ((twtr<<28)|(twr<<24)|(trtp<<20)|(tcl<<16)|(twl<<8)|(trl)) );
		dmcOutp32( &g_pDmcRegs->rDMC_TIMING_POWER, ((tfaw<<24)|(txsr<<16)|(txp<<8)|(tcke<<4)|(tmrd)));
	
	//2) initialize QoS 
		for ( i=0; i<32; i+=2 )
			dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONTROL0+i, 0x00ff0000 );
		for ( i=0; i<32; i+=2 )
			dmcOutp32( &g_pDmcRegs->rDMC_QOS_CONFIG0+i, 0x00000000 );
	
	//3) Direct command
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07000000 ); // chip0 Deselect(NOP)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01000000 ); // chip0 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00020000 ); // chip0 EMRS2
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00030000 ); // chip0 EMRS3
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00010400 ); // chip0 EMRS1 (MEM DLL on, DQS# disable)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00000542 ); // chip0 MRS (MEM DLL reset)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01000000 ); // chip0 PALL
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05000000 ); // chip0 REFA
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00000442 ); // chip0 MRS (MEM DLL unreset), BL=4
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00010780 ); // chip0 EMRS1 (OCD default)
		dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00010400 ); // chip0 EMRS1 (OCD exit)

		x = dmcInp32( &g_pDmcRegs->rDMC_CON_CONTROL);
		if((x &=0x10000)!=0)
			{	
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x07100000 ); // chip1 Deselect(NOP)
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01100000 ); // chip1 PALL
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00120000 ); // chip1 EMRS2
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00130000 ); // chip1 EMRS3
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00110400 ); // chip1 EMRS1 (MEM DLL on, DQS# disable)
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00100542 ); // chip1 MRS (MEM DLL reset)
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x01100000 ); // chip1 PALL
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x05100000 ); // chip1 REFA
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00100442 ); // chip1 MRS (MEM DLL unreset), BL=4
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00110780 ); // chip1 EMRS1 (OCD default)
			dmcOutp32( &g_pDmcRegs->rDMC_DIRECT_CMD, 0x00110400 ); // chip1 EMRS1 (OCD exit)
			}
	//4) change control for using memory.
		x = dmcInp32( &g_pDmcRegs->rDMC_CON_CONTROL);
		dmcOutp32( &g_pDmcRegs->rDMC_CON_CONTROL, (x |= 1<<5) );
		x = dmcInp32( &g_pDmcRegs->rDMC_MEM_CONTROL);
		x &=~0xff;
		dmcOutp32( &g_pDmcRegs->rDMC_MEM_CONTROL, (x |= (1<<4)|(1<<1)|(1<<0))); //Timeout Precharge, DPD(Active/Pre), DCC enable
	

}




