/*======================================================================

 Project Name : S3C24C0 verification project

 Copyright 2006 by Samsung Electronics, Inc.
 All rights reserved.

 Project Description :
 This software is only for verifying functions of the S3C24C0. 
 Anybody can use this code without our permission.

 File Name    : AM29F800.c
 Description  : Program AMD AM29F800 Flash memory 
  Author	   : Kijun.Kim
  Dept		   : AP
  Created Date : 2009.03.18
  Version	   : 0.0
 
  History
 - Created(Kijun.Kim 2009/03/18, Referenced 2450)

		
=======================================================================*/


#include "System.h"
#include "Am29f800.h"


// The addr parameter : WORD mode(16bit), so called in AMD specification.
#define _WR(addr,data)	*((volatile unsigned short *)(addr<<1))=(unsigned short)data
#define _RD(addr)	( *((volatile unsigned short *)(addr<<1)) )
#define _RESET()	_WR(0x0,0xf0f0)
#define BADDR2WADDR(addr)   (addr>>1)

int AM29F800_ProgFlash(unsigned int realAddr,unsigned short data);
void AM29F800_EraseSector(int targetAddr);
void AM29F800_EraseAll(void);
int AM29F800_CheckId(void);
int BlankCheck(int targetAddr,int targetSize);
int _WAIT(void);
  
static void InputTargetAddr(void);
 
static unsigned int srcAddress;
static unsigned int targetAddress;
static unsigned int targetSize;

u32 UartDownProgramSize;

void ProgramAM29F800(void)
{
    int i, iSectorAddr=0;   

    InputTargetAddr();
    
    srcAddress=UartDownAddress;
    	

    UART_Printf("[Check AM29LVxxx]\n");
    switch(AM29F800_CheckId())  //04.01.12 junon

    {
	case 1 :
	    UART_Printf("This device is AM29LV200BB!\n");	
	    break;
	case 2 :
	    UART_Printf("This device is AM29LV400BB!\n");	
	    break;
	case 3 :
	    UART_Printf("This device is AM29LV800BB!\n");	
	    break;
	case 4 :
	    UART_Printf("This device is AM29LV160DB!\n");	
	    break;
	case 5 :
	    UART_Printf("This device is 29LV160D!\n");	
	    break;
	case 6 :
		UART_Printf("This device is AM29LV320D!\n");	
		break;
	case 7 :
		UART_Printf("This device is AM29LV641D!\n");	
		break;
	case 8 :
		UART_Printf("This device is S29GL064N!\n");	
		break;
    case 100 :
	    UART_Printf("This device is maybe AM29LVxxxXB compatible, but we can use only bottom type!!\n");	
	    break;
   	default:
	    UART_Printf("ID Check Error or Not supported IC !!!\n");
	    return;
    }

    UART_Printf("\nErase the sector:0x%x.\n",targetAddress);

    if(targetAddress<0x4000) iSectorAddr=0x0;
    else if(targetAddress<0x6000) iSectorAddr=0x4000;
    else if(targetAddress<0x8000) iSectorAddr=0x6000;
    else if(targetAddress<0x10000) iSectorAddr=0x8000; 
    else 
    {
    	for (i=2; i<256; i++) // until..16MB
    		if (targetAddress < i*0x10000) 
    		{ 
    			iSectorAddr = (i-1)*0x10000; 
    			break;
    		}
    }

    while (iSectorAddr < targetAddress+targetSize)
    {
	AM29F800_EraseSector(iSectorAddr);
    	if (iSectorAddr == 0x0 ) iSectorAddr = 0x4000;
    	else if (iSectorAddr == 0x4000 ) iSectorAddr = 0x6000;
    	else if (iSectorAddr == 0x6000 ) iSectorAddr = 0x8000;
    	else if (iSectorAddr == 0x8000 ) iSectorAddr = 0x10000;    		
    	else iSectorAddr += 0x10000;
    }

    if(!BlankCheck(targetAddress, targetSize))
    {
		UART_Printf("Blank Check Error!!!\n");
		return;
    }

    UART_Printf("\nStart of the data writing.\n");

    for(i=0x0;i<targetSize;i+=2) 
    {
        AM29F800_ProgFlash(  i+targetAddress,*( (unsigned short *)(srcAddress+i) )  );
        if((i%0x1000)==0)UART_Printf("%x ",i);
    }
    UART_Printf("\nEnd of the data writing!!!\n");

    _RESET();

    UART_Printf("\nVerifying Start.\n");
    for(i=0x0;i<targetSize;i+=2) 
    {
        if(*( (unsigned short *)(i+targetAddress) )!=*( (unsigned short *)(srcAddress+i) )  )
	{    
	    UART_Printf("%x=verify error\n",i+targetAddress);
	    return;
	}
        if((i%0x1000)==0)UART_Printf("%x ",i);
    }
    UART_Printf("\nVerifying End!!!\n");

    UART_Printf("Do you want another programming without additional download? [y/N]\n");
    if(UART_Getc()=='y')ProgramAM29F800();
}

		  
static void InputTargetAddr(void)
{
    static unsigned int nextTargetAddr=0;
    
    UART_Printf("[AM29F800 Writing Program]\n");

    UART_Printf("\nCAUTION: Check AM29LV800 BYTE#(47) pin is connected to VDD.\n");

    UART_Printf("\nSource size:0h~%xh\n",UartDownProgramSize);
    UART_Printf("\nAvailable Target/Source Address:\n"); 
    UART_Printf("    0x0, 0x4000, 0x6000, 0x8000,0x10000,0x20000,0x30000,0x40000,\n");
    UART_Printf("0x50000,0x60000,0x70000,0x80000,0x90000,0xa0000,0xb0000,0xc0000,\n");
    UART_Printf("0xd0000,0xe0000,0xf0000\n");

    UART_Printf("Input target start address among above addresses[0x%x]:",nextTargetAddr);
    targetAddress=UART_GetIntNum();
    if(targetAddress==(unsigned int)(-1))targetAddress=nextTargetAddr;

    UART_Printf("Input target data size [0x%x]:",UartDownProgramSize);
    targetSize=UART_GetIntNum();
    if(targetSize==(unsigned int)(-1)) targetSize=UartDownProgramSize;

    UART_Printf("target start address=0x%x\n",targetAddress);
    UART_Printf("target data size=0x%x\n",targetSize);   
}

								 
int AM29F800_CheckId(void)
{
    unsigned short manId,devId;

    _RESET();
    
    _WR(0x555,0xaaaa);
    _WR(0x2aa,0x5555);
    _WR(0x555,0x9090);
    manId=_RD(0x0);

    _RESET(); // New 5V AM29F800 needs this command. 
    _WR(0x555,0xaaaa);
    _WR(0x2aa,0x5555);
    _WR(0x555,0x9090);
    devId=_RD(0x1);

    _RESET();   

    UART_Printf("Manufacture ID(0x0001)=%4x, Device ID(0x22xx)=%4x\n",manId,devId);

    if(manId==0x0001 && devId==0x22bf) return AM29LV200B;
    else if(manId==0x0001 && devId==0x22ba) return AM29LV400B;
    else if(manId==0x0001 && devId==0x225b) return AM29LV800B;
    else if(manId==0x0001 && devId==0x2249) return AM29LV160D;
    else if(manId==0x0004 && devId==0x2249) return _29LV160D; // fairchild        
    else if(manId==0x0001 && devId==0x22f9) return AM29LV320D;
    else if(manId==0x0001 && devId==0x22d7) return AM29LV641D;
    else if(manId==0x0001 && devId==0x227e) return S29GL064N; // spansion
    else if((devId&0xff00)==0x2200) return 100; // maybe AM29LVxxx series, but we can use only bottom type        
    else return 0;
}


void AM29F800_EraseSector(int targetAddr)
{
    UART_Printf("Erasing a sector 0x%x is started!\n", targetAddr);

    _RESET();

    _WR(0x555,0xaaaa);
    _WR(0x2aa,0x5555);
    _WR(0x555,0x8080);
    _WR(0x555,0xaaaa);
    _WR(0x2aa,0x5555);
    _WR(BADDR2WADDR(targetAddr),0x3030);
    _WAIT();
    _RESET();
}


void AM29F800_EraseAll(void) // It takes too many time to erase all
{
    UART_Printf("Erasing a chip is started!\n");

    _RESET();

    _WR(0x555,0xaaaa);
    _WR(0x2aa,0x5555);
    _WR(0x555,0x8080);
    _WR(0x555,0xaaaa);
    _WR(0x2aa,0x5555);
    _WR(0x555,0x1010);
    _WAIT();
    _RESET();

    UART_Printf("Erasing device is completed!\n");
}


int BlankCheck(int targetAddr,int targetSize)
{
    int i,j;
    for(i=0;i<targetSize;i+=2)
    {
		j=*((unsigned short *)(i+targetAddr));
		if( j!=0xffff)
		{
		    UART_Printf("Error!! [Addr:0x%x, Data:0x%x]\n",(i+targetAddr),j);
		    return 0;
		}
    }
    return 1;
}
		   

int _WAIT(void) //Check if the bit6 toggle ends.
{
	volatile unsigned short flashStatus,old;

	old=*((volatile unsigned short *)0x0);

	while(1)
	{
	    flashStatus=*((volatile unsigned short *)0x0);
        if( (old&0x40) == (flashStatus&0x40) )break;
        if( flashStatus&0x20 )
	    {
			old=*((volatile unsigned short *)0x0);
			flashStatus=*((volatile unsigned short *)0x0);
			if( (old&0x40) == (flashStatus&0x40) )
			    return 0;
			else return 1;
	    }
	    old=flashStatus;
	}
	return 1;
}
				

int AM29F800_ProgFlash(unsigned int realAddr,unsigned short data)
{
	volatile unsigned short *tempPt;

        tempPt=(volatile unsigned short *)realAddr;
        _WR(0x555,0xaaaa);
        _WR(0x2aa,0x5555);
        _WR(0x555,0xa0a0);
        *tempPt=data;
        
	return _WAIT();

}


#define	FlashBuffer	0x22000000

int ProgramAM29F800_R(void)
{
    volatile u32 *puRdBuf;
    int i, iSectorAddr=0;   


	targetAddress = 0x800000;
	targetSize = UartDownProgramSize;
    
    srcAddress=UartDownAddress;
    	
    UART_Printf("[Check AM29LVxxx]\n");
    switch(AM29F800_CheckId())  //04.01.12 junon

    {
	case 1 :
	    UART_Printf("This device is AM29LV200BB!\n");	
	    break;
	case 2 :
	    UART_Printf("This device is AM29LV400BB!\n");	
	    break;
	case 3 :
	    UART_Printf("This device is AM29LV800BB!\n");	
	    break;
	case 4 :
	    UART_Printf("This device is AM29LV160DB!\n");	
	    break;
	case 5 :
	    UART_Printf("This device is 29LV160D!\n");	
	    break;
	case 6 :
		UART_Printf("This device is AM29LV320D!\n");	
		break;
	case 7 :
		UART_Printf("This device is AM29LV641D!\n");	
		break;
	case 8 :
		UART_Printf("This device is S29GL064N!\n");	
		break;
    case 100 :
	    UART_Printf("This device is maybe AM29LVxxxXB compatible, but we can use only bottom type!!\n");	
	    break;
   	default:
	    UART_Printf("ID Check Error or Not supported IC !!!\n");
	    return;
    }

    UART_Printf("\nErase the sector:0x%x.\n",targetAddress);

    if(targetAddress<0x4000) iSectorAddr=0x0;
    else if(targetAddress<0x6000) iSectorAddr=0x4000;
    else if(targetAddress<0x8000) iSectorAddr=0x6000;
    else if(targetAddress<0x10000) iSectorAddr=0x8000; 
    else 
    {
    	for (i=2; i<256; i++) // until..16MB
    		if (targetAddress < i*0x10000) 
    		{ 
    			iSectorAddr = (i-1)*0x10000; 
    			break;
    		}
    }

    while (iSectorAddr < targetAddress+targetSize)
    {
	AM29F800_EraseSector(iSectorAddr);
    	if (iSectorAddr == 0x0 ) iSectorAddr = 0x4000;
    	else if (iSectorAddr == 0x4000 ) iSectorAddr = 0x6000;
    	else if (iSectorAddr == 0x6000 ) iSectorAddr = 0x8000;
    	else if (iSectorAddr == 0x8000 ) iSectorAddr = 0x10000;    		
    	else iSectorAddr += 0x10000;
    }

    if(!BlankCheck(targetAddress, targetSize))
    {
		UART_Printf("Blank Check Error!!!\n");
		return;
    }

    UART_Printf("\nStart of the data writing.\n");

    for(i=0x0;i<targetSize;i+=2) 
    {
        AM29F800_ProgFlash(  i+targetAddress,*( (unsigned short *)(srcAddress+i) )  );
        if((i%0x1000)==0)UART_Printf("%x ",i);
    }
    UART_Printf("\nEnd of the data writing!!!\n");

    _RESET();

    UART_Printf("\nVerifying Start.\n");
    for(i=0x0;i<targetSize;i+=2) 
    {
        if(*( (unsigned short *)(i+targetAddress) )!=*( (unsigned short *)(srcAddress+i) )  )
	{    
	    UART_Printf("%x=verify error\n",i+targetAddress);
	    return 0;;
	}
        if((i%0x1000)==0)UART_Printf("%x ",i);
    }
    UART_Printf("\nVerifying End!!!\n");
	return 1;

}

void ProgramAM29F800_Repeat(void)
{
	int i=0;
	int cnt=0;
	while(cnt<100){
		if (ProgramAM29F800_R()==0) i++;
		UART_Printf("===================================\n");
		UART_Printf("=== COUNT NUM => %d\n", cnt++);
		UART_Printf("===================================\n");
		}
	UART_Printf("READ/WRITE ERROR # = %d\n",i);
}

