/**************************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : otgdev_test.c
*  
*	File Description : This file implements the test main functon.
*
*	Author : Haksoo,Kim
*	Dept. : AP Development Team
*	Created Date : 2007/01/23
*	Version : 0.1 
* 
*	History
*	- Created(Haksoo,Kim 2007/01/23)
*  
**************************************************************************************/



#include <stdio.h>

#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "sysc.h"
#include "intc.h"
#include "otgdev.h"
#include "gpio.h"
#include "timer.h"
#include "nand.h"
//#include "am29f800.h"

USB_OPMODE eOpMode = USB_DMA;
USB_DESCMODE eDmaMode = USB_NON_DESC_DMA;
USB_SPEED eSpeed = USB_HIGH;
u8 download_run=false;
u32 tempDownloadAddress;
void (*run)(void);
u8 bStop_Download_Run=false;
u8 g_bSuspendResume=false;
u8 g_bOtgMon=false;

//////////
// Function Name : Isr_UsbOtg
// Function Description : USB OTG ISR
// Input : NONE
// Output : NONE
// Version : 
static void __irq Isr_UsbOtg(void)
{
	INTC_Disable(NUM_OTG);

	OTGDEV_HandleEvent();
	
	INTC_Enable(NUM_OTG);
	
	INTC_ClearVectAddr();
}


//////////
// Function Name : Download_Only
// Function Description : This function downloads a image file thru DNW.
//						You can configure the address to downlad in DNW "Configuration-Options" menu.
// Input : NONE
// Output : NONE
// Version : 
static void Download_Only(void)
{
	u32 uDownAddr, uDownFileSize, pDownPt, i;
	u32 uLapseTime=0;

	OTGDEV_ClearDownFileInfo();

	if ((eOpMode == USB_DMA) && (eDmaMode == USB_DESC_DMA))
		OTGDEV_PrepareDmaDesc(eOpMode);
	else
		OTGDEV_SetOpMode(eOpMode); 
	
	if(download_run==false)
	{	
	    Disp("Enter a new temporary download address(0x...):");
	    tempDownloadAddress=(u32)UART_GetIntNum();
	    if(tempDownloadAddress==0xffffffff)
	    {
	    	tempDownloadAddress=DefaultDownloadAddress;
	    }
	    Disp("The temporary download address is 0x%x.\n\n",tempDownloadAddress);		
	}

	Disp("Select a file to download in DNW\n");
	Disp("If you want to quit, press any key\n");

	i = 0;

	do
	{
		OTGDEV_GetDownFileInfo(&uDownAddr, &uDownFileSize, &pDownPt);
		
		if(GetKey() != 0)
		{
			if (download_run==true)
				bStop_Download_Run = true;
			
			return;
		}      
       
		if(i%0x100000==0)
			DisplayLED(0x6);
		if(i%0x100000==0x80000)
			DisplayLED(0x9);
		i++;		
	}while((uDownAddr == 0) ||(uDownFileSize == 0));

	StartTimer(0);
	
	Disp("\n[ADDRESS:%xh,TOTAL:%d(0x%x)]\n", uDownAddr, uDownFileSize, (uDownFileSize-10));

	do
	{
		OTGDEV_GetDownFileInfo(&uDownAddr, &uDownFileSize, &pDownPt);

		if(GetKey() == 'x')
		{
			if (download_run==true)
				bStop_Download_Run = true;
			
			return;
		} 
	}while((pDownPt - uDownAddr)<(uDownFileSize - 8));

	uLapseTime = StopTimer(0);
	
	if(uLapseTime == 0)
	{
		uLapseTime = 1;
	}
	
	if(eSpeed == USB_FULL)
	{
		Disp("(%3.3fKB/s,%3.3fs)\n",((float)uDownFileSize/uLapseTime*1000000/1024.),((float)uLapseTime/1000000));
	}
	else
	{
		Disp("(%3.3fMB/s,%3.3fs)\n",((float)uDownFileSize/uLapseTime*1000000/1024/1024.),((float)uLapseTime/1000000));
	}

	OTGDEV_VerifyChecksum();
	
	download_run=false;
	
}

//////////
// Function Name : Download_Run
// Function Description : This function downloads a image file thru DNW 
//						and then jumps to the start address of the downloaded image
// Input : NONE
// Output : NONE
// Version : 
void Download_Run(void)
{
	u32 uDownAddr, uDownFileSize, pDownPt, i;
	
	download_run=true;

	Download_Only();
	
	if(bStop_Download_Run==true)
		return;
	
	OTGDEV_GetDownFileInfo(&uDownAddr, &uDownFileSize, &pDownPt);

	run=(void (*)(void))uDownAddr;

	for(i=0;i<INT_LIMIT;i++)
	{
		INTC_Disable(i);
	}
    
	// [ Disable both caches ]
	CoDisableICache();
	CoDisableDCache();
	CoInvalidateICache();

	CoDisableMmu();
	CoInvalidateDTlb();

	run();	
		
}

//////////
// Function Name : Upload_Only
// Function Description : This function uploads a image file thru DNW
//						You can configure the address and the size to upload in DNW "Configuration-Options" menu.
// Input : NONE
// Output : NONE
// Version : 
void Upload_Only()
{
	u32 uUpAddr, uUpFileSize, pUpPt, i;
	u32 uLapseTime=0;

	OTGDEV_ClearDownFileInfo();
	OTGDEV_ClearUpFileInfo();

	if ((eOpMode == USB_DMA) && (eDmaMode == USB_DESC_DMA))
		OTGDEV_PrepareDmaDesc(eOpMode);
	else
		OTGDEV_SetOpMode(eOpMode); 

	Disp("\nConfigure the address and the size of the file to upload in DNW\n");
	Disp("If you want to quit, press any key\n");

	i = 0;

	do
	{
		OTGDEV_GetUpFileInfo(&uUpAddr, &uUpFileSize, &pUpPt);
	
		if(GetKey() != 0)
		{
			return;
		}      
       
		if(i%0x100000==0)
			DisplayLED(0x6);
		if(i%0x100000==0x80000)
			DisplayLED(0x9);
		i++;		
	}while((uUpAddr == 0) || (uUpFileSize == 0));

	StartTimer(0);
	
	Disp("\n[ADDRESS:%xh,TOTAL:%d]\n", uUpAddr, uUpFileSize);

	do
	{
		OTGDEV_GetUpFileInfo(&uUpAddr, &uUpFileSize, &pUpPt);

		if(GetKey() == 'x')
		{
			return;
		} 
	}while((pUpPt - uUpAddr)<uUpFileSize);

	uLapseTime = StopTimer(0);
	
	if(uLapseTime == 0)
	{
		uLapseTime = 1;
	}
	
	if(eSpeed == USB_FULL)
	{
		Disp("(%3.3fKB/s,%3.3fs)\n",((float)uUpFileSize/uLapseTime*1000000/1024.),((float)uLapseTime/1000000));
	}
	else
	{
		Disp("(%3.3fMB/s,%3.3fs)\n",((float)uUpFileSize/uLapseTime*1000000/1024/1024.),((float)uLapseTime/1000000));
	}	
}

//////////
// Function Name : Select_OpMode
// Function Description : This function selects an operation mode of USB OTG of CPU or DMA mode.
// Input : NONE
// Output : NONE
// Version : 
void Select_OpMode(void)
{
	int iSel, i=0;
	USB_SPEED eUsbSpeed;
	
	Disp(" Current Op Mode : ");
	if(eOpMode == USB_CPU)
	{
		Disp("CPU mode\n");
	}
	else if(eOpMode== USB_DMA)
	{
		Disp("DMA mode\n");
	}

	Disp(" Enter the op. mode (0: CPU_MODE, 1: Non-Descriptor DMA_MODE, 2: Descriptor DMA_MODE) : ");
	iSel = UART_GetIntNum();

	if (iSel != -1) 
	{
		if (iSel == 0)
			eOpMode = USB_CPU;
		else if (iSel == 1)
		{
			eOpMode = USB_DMA;
			eDmaMode = USB_NON_DESC_DMA;
		}
		else if (iSel == 2)
		{
			eOpMode = USB_DMA;
			eDmaMode = USB_DESC_DMA;
		}
		else
			Disp("Invalid selection\n");
	}

	OTGDEV_InitOtg(eSpeed);
	
	while(1)
	{
		if(OTGDEV_IsUsbOtgSetConfiguration()==true)
		{
			Disp("\n!!! USB host is connected !!!\n");
			Disp(" - Bulk In EP : %d\n",BULK_IN_EP);
			Disp(" - Bulk Out EP : %d\n",BULK_OUT_EP);
			
			OTGDEV_CheckEnumeratedSpeed(&eUsbSpeed);
			Disp(" - Speed : ");
			if(eUsbSpeed == USB_HIGH)
			{
				Disp("High\n");
				DisplayLED(0xf);
			}
			else if(eUsbSpeed == USB_FULL)
			{
				Disp("Full\n");
				DisplayLED(0x6);
			}

			Disp(" - Op Mode : ");
			if(eOpMode== USB_DMA){
				if (eDmaMode == USB_DESC_DMA)
					Disp("Descriptor DMA mode\n");
				else
					Disp("Non-Descriptor DMA mode\n");
			}
			else if(eOpMode == USB_CPU)
			{
				Disp("CPU mode\n");
			}
			break;
		}
		
		if(i%0x800000==0)
			DisplayLED(0x8);
		if(i%0x800000==0x200000)
			DisplayLED(0x4);
		if(i%0x800000==0x400000)
			DisplayLED(0x2);
		if(i%0x800000==0x600000)
			DisplayLED(0x1);
		i++;
	}
	
}

//////////
// Function Name : Select_Speed
// Function Description : This function selects high- or full-speed.
// Input : NONE
// Output : NONE
// Version : 
void Select_Speed(void)
{
	int iSel, i=0;
	USB_SPEED eUsbSpeed;
	
	Disp(" Current Op Mode : ");
	if(eSpeed == USB_HIGH)
	{
		Disp("HIGH_SPEED\n");
	}
	else if(eSpeed== USB_FULL)
	{
		Disp("FULL_SPEED\n");
	}

	Disp(" Enter the speed mode (0: HIGH_SPEED, 1: FULL_SPEED) : ");
	iSel = UART_GetIntNum();

	if (iSel != -1) 
	{
		if (iSel == 0)
			eSpeed = USB_HIGH;
		else if (iSel == 1)
			eSpeed = USB_FULL;
		else
			Disp("Invalid selection\n");
	}

	OTGDEV_InitOtg(eSpeed);
	
	while(1)
	{
		if(OTGDEV_IsUsbOtgSetConfiguration()==true)
		{
			Disp("\n!!! USB host is connected !!!\n");
			Disp(" - Bulk In EP : %d\n",BULK_IN_EP);
			Disp(" - Bulk Out EP : %d\n",BULK_OUT_EP);
			
			OTGDEV_CheckEnumeratedSpeed(&eUsbSpeed);
			Disp(" - Speed : ");
			if(eUsbSpeed == USB_HIGH)
			{
				Disp("High\n");
				DisplayLED(0xf);
			}
			else if(eUsbSpeed == USB_FULL)
			{
				Disp("Full\n");
				DisplayLED(0x6);
			}

			Disp(" - Op Mode : ");
			if(eOpMode== USB_DMA){
				if (eDmaMode == USB_DESC_DMA)
					Disp("Descriptor DMA mode\n");
				else
					Disp("Non-Descriptor DMA mode\n");
			}
			else if(eOpMode == USB_CPU)
			{
				Disp("CPU mode\n");
			}
			break;
		}
		
		if(i%0x800000==0)
			DisplayLED(0x8);
		if(i%0x800000==0x200000)
			DisplayLED(0x4);
		if(i%0x800000==0x400000)
			DisplayLED(0x2);
		if(i%0x800000==0x600000)
			DisplayLED(0x1);
		i++;
	}
	
}

//////////
// Function Name : Suspend_Resume
// Function Description : This function enables or disables Suspend & Resume option.
// Input : NONE
// Output : NONE
// Version : 
void Suspend_Resume(void)
{
	s32 sSel;
	
	Disp("Suspend & Resume Display On/Off\n");
	Disp(" 0 : off   1 : on\n");
	sSel = UART_GetIntNum();
	if(sSel==0)
	{
		g_bSuspendResume = false;
	}
	else if(sSel==1)
	{
		g_bSuspendResume = true;
		
		Disp("clk48Mhz_ohci On/Off\n");
		Disp(" 0 : off   1 : on[D]\n");
		sSel = UART_GetIntNum();

		if (sSel == 0)
			OTGDEV_Control48MhzClock(0);
		else
			OTGDEV_Control48MhzClock(1);
	}
	else
	{
		Disp("You entered wrong number\n");
	}
	
	return;	
	
}

//////////
// Function Name : Check_SuspendCurrent
// Function Description : 
// Input : NONE
// Output : NONE
// Version : 
void Check_SuspendCurrent(void)
{
	u8 ucKey=0;
	USB_SPEED eUsbSpeed;
	u32 i=0;
		
	Disp("OTG PHY will enter suspend mode(clk48Mhz_ohci is available).\n");
	Disp("After entering suspend mode, OTG will not work\n");
	
	////	
	OTGDEV_ForceSuspend(1);
	////	
	
	Disp("\nMeasure current through Vdd_otgi & Vdd_otg\n");	
	Disp("If you want to stop clk48Mhz_ohci, press 'x' key\n");
	while(ucKey!='x')
	{
		ucKey = UART_GetKey();
	}
	ucKey = 0;
	
	////
	OTGDEV_ForceSuspend(0);
	SYSC_SetUSBPHYControl(false);	//mask usb signal
	//SYSC_SetOscPadOnOff(eMODE_NORMAL, eOSC_USB, false);	//disable OTG clock pad
	////	
	
	Disp("\nMeasure current through Vdd_otgi & Vdd_otg\n");	
	Disp("If you want to have OTG exited from suspend mode, press 'x' key\n");
	while(ucKey!='x')
	{
		ucKey = UART_GetKey();
	}
	
	////
	OTGDEV_InitOtg(eSpeed);
	////
	
	Disp("USB host is not connected yet.\n");
	Disp("Waiting for USB host connection.\n");
	
	while(1)
	{
		if(OTGDEV_IsUsbOtgSetConfiguration()==true)
		{
			Disp("\n!!! USB host is connected !!!\n");
			Disp(" - Bulk In EP : %d\n",BULK_IN_EP);
			Disp(" - Bulk Out EP : %d\n",BULK_OUT_EP);
			
			OTGDEV_CheckEnumeratedSpeed(&eUsbSpeed);
			Disp(" - Speed : ");
			if(eUsbSpeed == USB_HIGH)
			{
				Disp("High\n");
				DisplayLED(0xf);
			}
			else if(eUsbSpeed == USB_FULL)
			{
				Disp("Full\n");
				DisplayLED(0x6);
			}

			Disp(" - Op Mode : ");
			if(eOpMode== USB_DMA)
			{
				Disp("DMA mode\n");
			}
			else if(eOpMode == USB_CPU)
			{
				Disp("CPU mode\n");
			}
			break;
		}
		
		if(i%0x800000==0)
			DisplayLED(0x8);
		if(i%0x800000==0x200000)
			DisplayLED(0x4);
		if(i%0x800000==0x400000)
			DisplayLED(0x2);
		if(i%0x800000==0x600000)
			DisplayLED(0x1);
		i++;
	}	
	
	return;	
	
}

//////////
// Function Name : main
// Function Desctiption : main function of the monitor pgm.
// Input : NONE
// Output : NONE
// Version :
void Test_OTGDEV(void)
{
	u32 iSel, i=0;
	USB_SPEED eUsbSpeed;
	
	const testFuncMenu menu[]=
	{
		0,						"Exit",
		Download_Run,	       	"Download & Run",
		Download_Only,			"Donwload Only",
		Upload_Only,			"Upload Only",
		Select_OpMode,			"Select Op Mode",
		Select_Speed,			"Select Speed(High/Full)",
		Suspend_Resume,		"Suspend & Resume Display On/Off",
		Check_SuspendCurrent,	"Check Suspend Current",
		0,                      0
	};
	
	Disp("USB host is not connected yet.\n");
	Disp("Waiting for USB host connection.\n");	
	
	INTC_SetVectAddr(NUM_OTG, Isr_UsbOtg);
	INTC_Enable(NUM_OTG);	
	
	OTGDEV_InitOtg(eSpeed);	
	
	while(1)
	{
		if(OTGDEV_IsUsbOtgSetConfiguration()==true)
		{
			Disp("\n!!! USB host is connected !!!\n");
			Disp(" - Bulk In EP : %d\n",BULK_IN_EP);
			Disp(" - Bulk Out EP : %d\n",BULK_OUT_EP);
			
			OTGDEV_CheckEnumeratedSpeed(&eUsbSpeed);
			Disp(" - Speed : ");
			if(eUsbSpeed == USB_HIGH)
			{
				Disp("High\n");
				DisplayLED(0xf);
			}
			else if(eUsbSpeed == USB_FULL)
			{
				Disp("Full\n");
				DisplayLED(0x6);
			}

			Disp(" - Op Mode : ");
			if(eOpMode== USB_DMA){
				if (eDmaMode == USB_DESC_DMA)
					Disp("Descriptor DMA mode\n");
				else
					Disp("Non-Descriptor DMA mode\n");
			}
			else if(eOpMode == USB_CPU)
			{
				Disp("CPU mode\n");
			}
			break;
		}
		
		if(i%0x800000==0)
			DisplayLED(0x8);
		if(i%0x800000==0x200000)
			DisplayLED(0x4);
		if(i%0x800000==0x400000)
			DisplayLED(0x2);
		if(i%0x800000==0x600000)
			DisplayLED(0x1);
		i++;
	}

	if (g_bOtgMon==true)
	{	
		Disp("\nDownload & Run is selected\n\n");
		Download_Run();
		
		download_run = false;
		bStop_Download_Run = false;
	}

	while(1)
	{
		Disp("\n");
		for (i=0; (int)(menu[i].desc)!=0; i++)
			Disp("%2d: %s\n", i, menu[i].desc);

		Disp("\nSelect the function to test : ");
		iSel = UART_GetIntNum();
		Disp("\n");
		if (iSel == 0)
		{
			Disp("USB connection will be disconnected...\n");
			OTGDEV_SetSoftDisconnect();			
			break;
		}
		else if (iSel>0 && iSel<(sizeof(menu)/8-1))
			(menu[iSel].func)();
	}
}

