
/**************************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2006 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : usbhost.c
*  
*	File Description : This file implements the API functon for USB HOST.
*
*	Author : Haksoo,Kim
*	Dept. : AP Development Team
*	Created Date : 2007/01/16
*	Version : 0.1 
* 
*	History
*	- Create (Haksoo,Kim 2009/07/15)
**************************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "usbhost.h"
#include "sysc.h"
#include "intc.h"

 
#define DBG_USBHOST
#ifdef DBG_USBHOST
	#define DbgUsbHost	UART_Printf
#else
	#define DbgUsbHost(...)
#endif
//#define DBG_USBHOST_0
#ifdef DBG_USBHOST_0
	#define DbgUsbHost0	UART_Printf
#else
	#define DbgUsbHost0(...)
#endif

#if 1
#define uhOutp32(addr, data)					Outp32(addr, data)
#define uhInp32(addr)  						Inp32(addr)
#define uhInp32_(addr, data)					Inp32_(addr, data)
#define uhSet32(addr, data)					Outp32(addr, Inp32(addr)|data)
#define uhShiftOutp32(addr, data, shift)		Outp32(addr, (data<<shift))
#define uhShiftSet32(addr, data, shift)			Outp32(addr, Inp32(addr)|(data<<shift))
#define uhMaskSet32(addr, data, shift, width)	Outp32(addr, Inp32(addr)&~((2<<(shift+width))-(2<<shift))|(data<<shift))
#else
#define uhOutp32(addr, data)	{UART_Printf("Outp32(\'h%08x, \'h%08x);\n", addr, data), Outp32(addr, data);}
#define uhInp32_(addr, data)	{Inp32_(addr, data); UART_Printf("Inp32(\'h%08x, d); // d=0x%08x\n", addr, data);}
#endif

#define UsbHostAlign(x, alignbyte) (((x)+(alignbyte)-1)/(alignbyte)*(alignbyte))

#define USBHOST_DATA_BASE				CODEC_MEM_ST


//==========================
// USB HOST EHCI REGISTERS
//==========================
#define USBHOST_EHCI_OP_BASE	0xEC200010
enum USBHOST_EHCI_SFR
{
	// implemented capability register
	rUH_EHCI_HCCPBASE					= USBHOST_EHCI_BASE+0x0000,
	rUH_EHCI_HCSPARAMS				= USBHOST_EHCI_BASE+0x0004,
	rUH_EHCI_HCCPARAMS				= USBHOST_EHCI_BASE+0x0008,

	// implemented operational register
	rUH_EHCI_OP_USBCMD				= USBHOST_EHCI_OP_BASE+0x0000,
	rUH_EHCI_OP_USBSTS				= USBHOST_EHCI_OP_BASE+0x0004,
	rUH_EHCI_OP_USBINTR				= USBHOST_EHCI_OP_BASE+0x0008,
	rUH_EHCI_OP_FRINDEX				= USBHOST_EHCI_OP_BASE+0x000C,
	rUH_EHCI_OP_CTRLDSSEGMENT		= USBHOST_EHCI_OP_BASE+0x0010,
	rUH_EHCI_OP_PERIODICLISTBASE		= USBHOST_EHCI_OP_BASE+0x0014,
	rUH_EHCI_OP_ASYNCLISTADDR		= USBHOST_EHCI_OP_BASE+0x0018,

	// implemented auxiliary register
	rUH_EHCI_OP_CONFIGFLAG			= USBHOST_EHCI_OP_BASE+0x0040,
	rUH_EHCI_OP_PORTSC				= USBHOST_EHCI_OP_BASE+0x0044,

	// implemented specific register
	rUH_EHCI_OP_INSNREG00			= USBHOST_EHCI_OP_BASE+0x0080,
	rUH_EHCI_OP_INSNREG01			= USBHOST_EHCI_OP_BASE+0x0084,
	rUH_EHCI_OP_INSNREG02			= USBHOST_EHCI_OP_BASE+0x0088,
	rUH_EHCI_OP_INSNREG03			= USBHOST_EHCI_OP_BASE+0x008C,
	rUH_EHCI_OP_INSNREG04			= USBHOST_EHCI_OP_BASE+0x0090,
	rUH_EHCI_OP_INSNREG05			= USBHOST_EHCI_OP_BASE+0x0094
};

//==========================
// USB HOST OHCI REGISTERS
//==========================
enum USBHOST_OHCI_SFR
{
	// control & status partition
	rUH_OHCI_HcRevision				= USBHOST_OHCI_BASE+0x0000,
	rUH_OHCI_HcControl					= USBHOST_OHCI_BASE+0x0004,
	rUH_OHCI_HcCommandStatus			= USBHOST_OHCI_BASE+0x0008,
	rUH_OHCI_HcInterruptStatus			= USBHOST_OHCI_BASE+0x000C,
	rUH_OHCI_HcInterruptEnable			= USBHOST_OHCI_BASE+0x0010,
	rUH_OHCI_HcInterruptDisable		= USBHOST_OHCI_BASE+0x0014,

	// memory pointer partition
	rUH_OHCI_HcHCCA					= USBHOST_OHCI_BASE+0x0018,
	rUH_OHCI_HcPeriodCurrentED		= USBHOST_OHCI_BASE+0x001C,
	rUH_OHCI_HcControlHeadED			= USBHOST_OHCI_BASE+0x0020,
	rUH_OHCI_HcControlCurrentED		= USBHOST_OHCI_BASE+0x0024,
	rUH_OHCI_HcBulkHeadED				= USBHOST_OHCI_BASE+0x0028,
	rUH_OHCI_HcBulkCurrentED			= USBHOST_OHCI_BASE+0x002C,
	rUH_OHCI_HcDoneHead				= USBHOST_OHCI_BASE+0x0030,

	// frame counter partition
	rUH_OHCI_HcFmInterval				= USBHOST_OHCI_BASE+0x0034,
	rUH_OHCI_HcFmRemaining			= USBHOST_OHCI_BASE+0x0038,
	rUH_OHCI_HcFmNumber				= USBHOST_OHCI_BASE+0x003C,
	rUH_OHCI_HcPeriodicStart			= USBHOST_OHCI_BASE+0x0040,
	rUH_OHCI_HcLSThreshold			= USBHOST_OHCI_BASE+0x0044,

	// root hub partition
	rUH_OHCI_HcRhDescriptorA			= USBHOST_OHCI_BASE+0x0048,
	rUH_OHCI_HcRhDescriptorB			= USBHOST_OHCI_BASE+0x004C,
	rUH_OHCI_HcRhStatus				= USBHOST_OHCI_BASE+0x0050,
	rUH_OHCI_HcRhPortStatus			= USBHOST_OHCI_BASE+0x0054
};


//==========================
// USB PHY CONTROL REGISTERS
//==========================
enum USBPHY_CON_SFR
{
	rUPHY_CON_PHYPWR					= USBPHY_CON_BASE+0x0000,
	rUPHY_CON_PHYCLK					= USBPHY_CON_BASE+0x0004,
	rUPHY_CON_RSTCON					= USBPHY_CON_BASE+0x0008,
	rUPHY_CON_PHYTUNE0				= USBPHY_CON_BASE+0x0020,
	rUPHY_CON_PHYTUNE1				= USBPHY_CON_BASE+0x0024,
};


//=====================================================================
// definitions related to Standard Device Requests
#define SETUP_DATA_SIZE	8

// Standard bmRequestType (direction)
// #define DEVICE_bmREQUEST_TYPE(oDeviceRequest)  ((m_poDeviceRequest->bmRequestType) & 0x80)
enum DEV_REQUEST_DIRECTION
{
	HOST_TO_DEVICE				= 0x00,
	DEVICE_TO_HOST				= 0x80
};

// Standard bmRequestType (Type)
// #define DEVICE_bmREQUEST_TYPE(oDeviceRequest)  ((m_poDeviceRequest->bmRequestType) & 0x60)
enum DEV_REQUEST_TYPE
{
	STANDARD_TYPE               = 0x00,
	CLASS_TYPE                  = 0x20,
	VENDOR_TYPE                 = 0x40,
	RESERVED_TYPE               = 0x60
};

// Standard bmRequestType (Recipient)
// #define DEVICE_bmREQUEST_RECIPIENT(oDeviceRequest)  ((m_poDeviceRequest->bmRequestType) & 0x07)
enum DEV_REQUEST_RECIPIENT
{
	DEVICE_RECIPIENT			= 0,
	INTERFACE_RECIPIENT			= 1,
	ENDPOINT_RECIPIENT			= 2,
	OTHER_RECIPIENT				= 3
};

// Standard bRequest codes
enum STANDARD_REQUEST_CODE
{
	STANDARD_GET_STATUS         = 0,
	STANDARD_CLEAR_FEATURE      = 1,
	STANDARD_RESERVED_1         = 2,
	STANDARD_SET_FEATURE        = 3,
	STANDARD_RESERVED_2         = 4,
	STANDARD_SET_ADDRESS        = 5,
	STANDARD_GET_DESCRIPTOR     = 6,
	STANDARD_SET_DESCRIPTOR     = 7,
	STANDARD_GET_CONFIGURATION  = 8,
	STANDARD_SET_CONFIGURATION  = 9,
	STANDARD_GET_INTERFACE      = 10,
	STANDARD_SET_INTERFACE      = 11,
	STANDARD_SYNCH_FRAME        = 12
};


// Descriptor types
enum DESCRIPTOR_TYPE
{
	DEVICE_DESCRIPTOR           = 1,
	CONFIGURATION_DESCRIPTOR    = 2,
	STRING_DESCRIPTOR           = 3,
	INTERFACE_DESCRIPTOR        = 4,
	ENDPOINT_DESCRIPTOR         = 5,
	DEVICE_QUALIFIER            = 6,
	OTHER_SPEED_CONFIGURATION   = 7,
	INTERFACE_POWER				= 8
};

// configuration descriptor: bmAttributes
enum CONFIG_ATTRIBUTES
{
	CONF_ATTR_DEFAULT       	= 0x80, // in Spec 1.0, it was BUSPOWERED bit.
	CONF_ATTR_REMOTE_WAKEUP 	= 0x20,
	CONF_ATTR_SELFPOWERED   	= 0x40
};

// endpoint descriptor
enum ENDPOINT_ATTRIBUTES
{
	EP_ADDR_IN              = 0x80,
	EP_ADDR_OUT             = 0x00,

	EP_ATTR_CONTROL         = 0x0,
	EP_ATTR_ISOCHRONOUS     = 0x1,
	EP_ATTR_BULK            = 0x2,
	EP_ATTR_INTERRUPT       = 0x3
};

// Descriptor size
enum DESCRIPTOR_SIZE
{
	DEVICE_DESC_SIZE            = 18,
	STRING_DESC0_SIZE           = 4,
	STRING_DESC1_SIZE           = 22,
	STRING_DESC2_SIZE           = 44,
	CONFIG_DESC_SIZE            = 9,
	INTERFACE_DESC_SIZE         = 9,
	ENDPOINT_DESC_SIZE          = 7,
	DEVICE_QUALIFIER_SIZE       = 10,
	OTHER_SPEED_CFG_SIZE 		= 9
};


//=====================================================================
// 
// definition for qTD
enum USB_PID_CODE
{
	USB_OUT_TOKEN, USB_IN_TOKEN, USB_SETUP_TOKEN
};

enum USB_qTD_STATUS
{
	USB_qTD_INACTIVE, USB_qTD_ACTIVE=0x80
};

enum USB_HBW_PIPE_MULT
{
	ZERO_TR_uFRAME, ONE_TR_uFRAME, TWO_TR_uFRAME, THREE_TR_uFRAME
};

//=====================================================================
//definitions related to CSR setting
// . EHCI registers

// USBCMD
#define EHCI_CMD_HC_RESET				(0x1<<1)
#define EHCI_CMD_RUN					(0x1<<0)
#define EHCI_CMD_STOP					(0x0<<0)

// USBSTS
#define EHCI_STS_HALTED				(0x1<<12)

// USBINTR
#define EHCI_INTR_ASYNC_ADVANCE		(0x1<<5)		// ?
#define EHCI_INTR_HOST_SYSTEM_ERROR	(0x1<<4)		// system error (eg, pci master abort, pci parity error...)
#define EHCI_INTR_FRAME_LIST_ROLLOVER	(0x1<<3)		// frame list index roll over from maximum value to zero
#define EHCI_INTR_PORT_CHANGE			(0x1<<2)		// port change detect
#define EHCI_INTR_USB_ERROR			(0x1<<1)		// usb transaction completion with an error
#define EHCI_INTR_USB_COMPLETION		(0x1<<0)		// usb transaction completion

// CONFIGFLAG
#define EHCI_CONFIGFLAG_ROUTE_EHCI	(0x1<<0)
#define EHCI_CONFIGFLAG_ROUTE_OHCI	(0x0<<0)



//=====================================================================
// global varibles used in several functions
USBHOST oUsbHost;
USBDEVICE oUsbDevice;

u8 g_bCurrentConnectStatus = false;
u8 g_bEhciPortEnabled = false;
u8 g_bOhciPortEnabled = false;
volatile u8 g_bUsbTrCompletion = false;
USB_PORT_OWNER g_eUsbPortOwner = EHCI_OWNER;

u32 g_uUsbHostDataMem = USBHOST_DATA_BASE;

// related to EHCI
#define NUM_OF_QH			1
#define NUM_OF_qTD			3
#define QTD_BUF_MAX_SIZE	(5*4096)	// 20K
#define QTD_PAGE_MAX_SIZE	(4096)
#define NUM_OF_Bulk_qTD	2560	// 50MB/(5*4K)
ehci_qh_t_ptr ehci_qh_ptr[NUM_OF_QH];
ehci_qtd_t_ptr ehci_qtd_ptr[NUM_OF_qTD+NUM_OF_Bulk_qTD];
DEVICE_REQUEST_PTR ehci_setup_data_ptr;
u8 *ehci_in_data_ptr;
u8 *ehci_out_data_ptr;
//
////

// related to EHCI
#define OHCI_HCCA_SIZE		256
u32 *ohci_hcca_base;
#define NUM_OF_ED			1
#define NUM_OF_gTD			3
#define GTD_BUF_MAX_SIZE	(8*1024)	// 8K
#define NUM_OF_Bulk_gTD	6400	// 50MB/8K
ohci_ed_t_ptr ohci_ed_ptr[NUM_OF_ED];
ohci_gtd_t_ptr ohci_gtd_ptr[NUM_OF_gTD+NUM_OF_Bulk_gTD];
DEVICE_REQUEST_PTR ohci_setup_data_ptr;
u8 *ohci_in_data_ptr;
u8 *ohci_out_data_ptr;
u32 last_td_addr;
//
////

u8 ucBulkOutDt = 0;
u8 ucBulkInDt = 0;


//////////
// Function Name : USBHOST_Init
// Function Desctiption : This function initializes USB HOST.
// Input : NONE
// Output : NONE
// Version :
void USBHOST_Init(void)
{			
	INTC_Disable(NUM_UHOST);
	
	SYSC_SetOscPadOnOff(eMODE_NORMAL, eOSC_USB, true);	//enable usb clock pad
	SYSC_SetUSBPHYControl(true);	//unmask usb signal

	USBHOST_InitPhyCon();
	
	g_uUsbHostDataMem = USBHOST_DATA_BASE;
	g_bCurrentConnectStatus = false;
	ucBulkOutDt = 0;
	ucBulkInDt = 0;
	
	USBHOST_InitEhciCon();
	
	INTC_Enable(NUM_UHOST);
	
}


//////////
// Function Name : USBHOST_HandleEvent
// Function Desctiption : This function handles USB HOST Interrupts.
// Input : NONE
// Output : NONE
// Version :
// USBSTS
void USBHOST_HandleEvent(void)
{
	ehci_portsc_t ehci_portsc;
	
	uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);

	// ehci has port ownership
	if (g_eUsbPortOwner == EHCI_OWNER)
		USBHOST_HandleEhciEvent();

	// ohci has port ovwership
	else
		USBHOST_HandleOhciEvent();
	
}


//////////
// Function Name : USBHOST_HandleEhciEvent
// Function Desctiption : This function handles USB HOST EHCI Interrupts.
// Input : NONE
// Output : NONE
// Version :
// USBSTS
void USBHOST_HandleEhciEvent(void)
{
	ehci_usbsts_t ehci_usbsts;
	ehci_usbintr_t ehci_usbintr;

	// . check usbsts reg. & interrupt enable reg.
	uhInp32_(rUH_EHCI_OP_USBSTS, ehci_usbsts.data);
	uhInp32_(rUH_EHCI_OP_USBINTR, ehci_usbintr.data);

	// . filter enabled interrupt sources
	ehci_usbsts.data &= ehci_usbintr.data;

	// . clear interrupt
	uhSet32(rUH_EHCI_OP_USBSTS, ehci_usbsts.data);	// clear interrupt

	if (ehci_usbsts.b.usbint)
	{
		DbgUsbHost("\n[EHCI]USB Interrupt...(EHCI_STATE=%d)\n", oUsbHost.m_eEhciState);

		g_bUsbTrCompletion = true;

		g_uUsbHostDataMem = USBHOST_DATA_BASE;

		USBHOST_EhciDisableAsyncSched();

		switch(oUsbHost.m_eEhciState)
		{
			case EHCI_STATE_GD_DEV_BEFORE:
				Copy8((u32)ehci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescDevice, DEVICE_DESC_SIZE);
				break;

			case EHCI_STATE_SET_ADDR: 
				oUsbDevice.m_usDevAddr = oUsbHost.m_usDevAddr;
				break;

			case EHCI_STATE_GD_DEV_AFTER:
				Copy8((u32)ehci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescDevice, DEVICE_DESC_SIZE);
				break;

			case EHCI_STATE_GD_CFG:
				Copy8((u32)ehci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescConfig, CONFIG_DESC_SIZE);
				Copy8((u32)ehci_in_data_ptr+CONFIG_DESC_SIZE, (u32)&oUsbDevice.m_oDesc.oDescInterface, 
					(((oUsbDevice.m_oDesc.oDescConfig.wTotalLengthH<<8)|oUsbDevice.m_oDesc.oDescConfig.wTotalLengthL)-CONFIG_DESC_SIZE));
				break;

			case EHCI_STATE_GD_STR0:
				Copy8((u32)ehci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescStr0, 2);
				oUsbDevice.m_oDesc.oDescStr0.pString= (u8 *)malloc(STRING_DESC0_SIZE-2);
				Copy8((u32)ehci_in_data_ptr+2, (u32)oUsbDevice.m_oDesc.oDescStr0.pString, 
					STRING_DESC0_SIZE-2);
				break;

			case EHCI_STATE_GD_STR1:
				Copy8((u32)ehci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescStr1, 2);
				oUsbDevice.m_oDesc.oDescStr1.pString= (u8 *)malloc(oUsbDevice.m_oDesc.oDescStr1.bLength-2);
				Copy8((u32)ehci_in_data_ptr+2, (u32)oUsbDevice.m_oDesc.oDescStr1.pString, 
					oUsbDevice.m_oDesc.oDescStr1.bLength-2);
				break;

			case EHCI_STATE_GD_STR2:
				Copy8((u32)ehci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescStr2, 2);
				oUsbDevice.m_oDesc.oDescStr2.pString= (u8 *)malloc(oUsbDevice.m_oDesc.oDescStr2.bLength-2);
				Copy8((u32)ehci_in_data_ptr+2, (u32)oUsbDevice.m_oDesc.oDescStr2.pString, 
					oUsbDevice.m_oDesc.oDescStr2.bLength-2);
				break;

			case EHCI_STATE_GD_STR3:
				Copy8((u32)ehci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescStr3, 2);
				oUsbDevice.m_oDesc.oDescStr3.pString= (u8 *)malloc(oUsbDevice.m_oDesc.oDescStr3.bLength-2);
				Copy8((u32)ehci_in_data_ptr+2, (u32)oUsbDevice.m_oDesc.oDescStr3.pString, 
					oUsbDevice.m_oDesc.oDescStr3.bLength-2);
				break;

			case EHCI_STATE_SET_CFG:
				oUsbHost.m_eEhciState = EHCI_STATE_CONFIGURED;
				break;

			case EHCI_STATE_CONFIGURED:
				break;

			default:
				Assert(0);
				break;
		}
	}
	if (ehci_usbsts.b.usberrint)
	{
		UART_Printf("\n[EHCI]USB Error Interrupt...(EHCI_STATE=%d)\n", oUsbHost.m_eEhciState);

		g_uUsbHostDataMem = USBHOST_DATA_BASE;

		USBHOST_EhciDisableAsyncSched();
	}
	if (ehci_usbsts.b.p_chng_detect)
	{
		ehci_portsc_t ehci_portsc;

		DbgUsbHost("\n[EHCI]Port Change Interrupt...\n");

		uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);

		if (ehci_portsc.b.con_sts_chng)
		{
			uhShiftSet32(rUH_EHCI_OP_PORTSC, 1, 1);
			if (g_bCurrentConnectStatus != (u8)ehci_portsc.b.cur_con_sts)
			{
				g_bCurrentConnectStatus = (u8)ehci_portsc.b.cur_con_sts;

				if (g_bCurrentConnectStatus)
				{
					UART_Printf("\n>>>Device has been attached....\n");
					oUsbHost.m_eEhciState = EHCI_STATE_ATTACHED;
				}
				else
				{
					UART_Printf("\n>>>Device has been removed...\n");
					oUsbHost.m_eEhciState = EHCI_STATE_INIT;
				}
			}
		}
		if (ehci_portsc.b.p_en_dis_chng)
		{
			// For the root hub, this bit gets set to '1' only when a port is disabled due to the appropriate conditions existing at the EOF2 point
			// (refer to the EHCI spec. p29)
			uhShiftSet32(rUH_EHCI_OP_PORTSC, 1, 3);
			if (g_bEhciPortEnabled != (u8)ehci_portsc.b.p_en_dis)
			{
				g_bEhciPortEnabled = (u8)ehci_portsc.b.p_en_dis;

				if (g_bEhciPortEnabled)
				{
					DbgUsbHost("Port has been enabled....\n");
					oUsbHost.m_eEhciState = EHCI_STATE_READY;
				}
				else
				{
					DbgUsbHost("Port has been disabled...\n");
					oUsbHost.m_eEhciState = EHCI_STATE_ATTACHED;
				}
			}
		}		
		if (ehci_portsc.b.over_current_chng)
		{
			uhShiftSet32(rUH_EHCI_OP_PORTSC, 1, 5);
			if (ehci_portsc.b.over_current)
			{
				DbgUsbHost("Over Current Active....\n");
			}
			else
			{
				DbgUsbHost("Over Current Inactive....\n");
			}
		}		
		if (g_eUsbPortOwner != (USB_PORT_OWNER)ehci_portsc.b.p_owner)
		{
			g_eUsbPortOwner = (USB_PORT_OWNER)ehci_portsc.b.p_owner;

			if (g_eUsbPortOwner)
			{
				DbgUsbHost("OHCI has port ownership....\n");
			}
			else
			{
				DbgUsbHost("EHCI has port ownership...\n");
			}
		}
	}
	if (ehci_usbsts.b.frm_lst_rollover)
	{
		DbgUsbHost("\n[EHCI]Frame List Rollover Interrupt...\n");
	}
	if (ehci_usbsts.b.host_sys_error)
	{
		DbgUsbHost("\n[EHCI]Host System Error Interrupt...\n");
	}
	if (ehci_usbsts.b.intr_async_adv)
	{
		DbgUsbHost("\n[EHCI]Async Advance Interrupt...\n");
	}
}


//////////
// Function Name : USBHOST_HandleOhciEvent
// Function Desctiption : This function handles USB HOST OHCI Interrupts.
// Input : NONE
// Output : NONE
// Version :
// USBSTS
void USBHOST_HandleOhciEvent(void)
{
	ohci_interruptstatus_t ohci_interruptstatus;
	ohci_interruptenable_t ohci_interruptenable;

	// . check usbsts reg. & interrupt enable reg.
	uhInp32_(rUH_OHCI_HcInterruptStatus, ohci_interruptstatus.data);
	uhInp32_(rUH_OHCI_HcInterruptEnable, ohci_interruptenable.data);

	// . filter enabled interrupt sources
	ohci_interruptstatus.data &= ohci_interruptenable.data;

	// . clear interrupt
	uhOutp32(rUH_OHCI_HcInterruptStatus, ohci_interruptstatus.data);
	
	if (ohci_interruptstatus.b.scheduling_overrun)
	{
		UART_Printf("\n[OHCI]Scheduling Overrun Interrupt...(OHCI_STATE=%d)\n", oUsbHost.m_eOhciState);
	}
	if (ohci_interruptstatus.b.writeback_donehead)
	{
		u8 bResult;

		bResult = (*(u32 *)ohci_hcca_base[0x84/4])>>28;

		// check completion code
		if (bResult != 0)
		{
			UART_Printf("Error!!!(Completion Code is 0x%04x)\n", bResult);
		}

		// check whether if last transfer (by comparing write_done_head addr of HCCA with last td addr)
		if (last_td_addr == ohci_hcca_base[0x84/4])
		{
			DbgUsbHost("\n[OHCI]Writeback DoneHead Interrupt...(OHCI_STATE=%d)\n", oUsbHost.m_eOhciState);
			
			g_bUsbTrCompletion = true;		

			g_uUsbHostDataMem = USBHOST_DATA_BASE + OHCI_HCCA_SIZE;
			
			// control transfer				
			if (oUsbHost.m_eOhciState != OHCI_STATE_CONFIGURED)
			{
				USBHOST_OhciDisableList(1);
			}
			// bulk transfer
			else
			{
				USBHOST_OhciDisableList(0);
			}		
		}
		else
		{
			DbgUsbHost(".");
		}
		
		switch(oUsbHost.m_eOhciState)
		{
			case OHCI_STATE_GD_DEV_BEFORE:
				Copy8((u32)ohci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescDevice, DEVICE_DESC_SIZE);
				break;

			case OHCI_STATE_SET_ADDR: 
				oUsbDevice.m_usDevAddr = oUsbHost.m_usDevAddr;
				break;

			case OHCI_STATE_GD_DEV_AFTER:
				Copy8((u32)ohci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescDevice, DEVICE_DESC_SIZE);
				break;

			case OHCI_STATE_GD_CFG:
				Copy8((u32)ohci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescConfig, CONFIG_DESC_SIZE);
				Copy8((u32)ohci_in_data_ptr+CONFIG_DESC_SIZE, (u32)&oUsbDevice.m_oDesc.oDescInterface, 
					(((oUsbDevice.m_oDesc.oDescConfig.wTotalLengthH<<8)|oUsbDevice.m_oDesc.oDescConfig.wTotalLengthL)-CONFIG_DESC_SIZE));
				break;

			case OHCI_STATE_GD_STR0:
				Copy8((u32)ohci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescStr0, 2);
				oUsbDevice.m_oDesc.oDescStr0.pString= (u8 *)malloc(STRING_DESC0_SIZE-2);
				Copy8((u32)ohci_in_data_ptr+2, (u32)oUsbDevice.m_oDesc.oDescStr0.pString, 
					STRING_DESC0_SIZE-2);
				break;

			case OHCI_STATE_GD_STR1:
				Copy8((u32)ohci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescStr1, 2);
				oUsbDevice.m_oDesc.oDescStr1.pString= (u8 *)malloc(oUsbDevice.m_oDesc.oDescStr1.bLength-2);
				Copy8((u32)ohci_in_data_ptr+2, (u32)oUsbDevice.m_oDesc.oDescStr1.pString, 
					oUsbDevice.m_oDesc.oDescStr1.bLength-2);
				break;

			case OHCI_STATE_GD_STR2:
				Copy8((u32)ohci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescStr2, 2);
				oUsbDevice.m_oDesc.oDescStr2.pString= (u8 *)malloc(oUsbDevice.m_oDesc.oDescStr2.bLength-2);
				Copy8((u32)ohci_in_data_ptr+2, (u32)oUsbDevice.m_oDesc.oDescStr2.pString, 
					oUsbDevice.m_oDesc.oDescStr2.bLength-2);
				break;

			case OHCI_STATE_GD_STR3:
				Copy8((u32)ohci_in_data_ptr, (u32)&oUsbDevice.m_oDesc.oDescStr3, 2);
				oUsbDevice.m_oDesc.oDescStr3.pString= (u8 *)malloc(oUsbDevice.m_oDesc.oDescStr3.bLength-2);
				Copy8((u32)ohci_in_data_ptr+2, (u32)oUsbDevice.m_oDesc.oDescStr3.pString, 
					oUsbDevice.m_oDesc.oDescStr3.bLength-2);
				break;

			case OHCI_STATE_SET_CFG:
				oUsbHost.m_eOhciState = OHCI_STATE_CONFIGURED;
				break;

			case OHCI_STATE_CONFIGURED:
				break;

			default:
				Assert(0);
				break;
		}
	}
	if (ohci_interruptstatus.b.start_of_frame)
	{
		DbgUsbHost("\n[OHCI]Start of Frame Interrupt...\n");
	}
	if (ohci_interruptstatus.b.resume_detected)
	{
		DbgUsbHost("\n[OHCI]Resume Detected Interrupt...\n");
	}
	if (ohci_interruptstatus.b.unrecoverable_error)
	{
		UART_Printf("\n[OHCI]Unrecoverable Error Interrupt...\n");
	}
	if (ohci_interruptstatus.b.frame_number_overflow)
	{
		DbgUsbHost("\n[OHCI]Frame Number Overflow Interrupt...\n");
	}
	if (ohci_interruptstatus.b.root_hub_status_change)
	{
		ohci_rhstatus_t ohci_rhstatus;
		ohci_rhportstatus1_t ohci_rhportstatus1;
		
		DbgUsbHost("\n[OHCI]Root Hub Status Change Interrupt...\n");

		uhInp32_(rUH_OHCI_HcRhStatus, ohci_rhstatus.data);
		uhInp32_(rUH_OHCI_HcRhPortStatus, ohci_rhportstatus1.data);

		if (ohci_rhstatus.b.overcurrent_indicator_change)
		{
			// must use uhShiftOutp32 instead of uhShiftSet32, in order to set only corresponding bit
			uhShiftOutp32(rUH_OHCI_HcRhStatus, 1, 17);	// clearing this bit
			if (ohci_rhstatus.b.overcurrent_indicator)
			{
				DbgUsbHost("Over Current Active....\n");
			}
			else
			{
				DbgUsbHost("Over Current Inactive....\n");
			}
		}

		if (ohci_rhportstatus1.b.connect_status_change)
		{
			// must use uhShiftOutp32 instead of uhShiftSet32, in order to set only corresponding bit
			uhShiftOutp32(rUH_OHCI_HcRhPortStatus, 1, 16);	// clearing this bit
			if (g_bCurrentConnectStatus != (u8)ohci_rhportstatus1.b.ccs_cpe)
			{
				g_bCurrentConnectStatus = (u8)ohci_rhportstatus1.b.ccs_cpe;

				if (g_bCurrentConnectStatus)
				{
					UART_Printf("\n>>>Device has been attached....\n");
					oUsbHost.m_eOhciState = OHCI_STATE_ATTACHED;
				}
				else
				{
					UART_Printf("\n>>>Device has been removed...\n");
					oUsbHost.m_eOhciState = OHCI_STATE_INIT;
				}
			}
		}
		if (ohci_rhportstatus1.b.port_enable_status_change)
		{
			// must use uhShiftOutp32 instead of uhShiftSet32, in order to set only corresponding bit
			uhShiftOutp32(rUH_OHCI_HcRhPortStatus, 1, 17);	// clearing this bit
			DbgUsbHost("Port Enable Status Change Interrupt...\n");
			
			if (g_bOhciPortEnabled != (u8)ohci_rhportstatus1.b.pes_spe)
			{
				g_bOhciPortEnabled = (u8)ohci_rhportstatus1.b.pes_spe;

				if (g_bOhciPortEnabled)
				{
					DbgUsbHost("Port has been enabled....\n");
					oUsbHost.m_eOhciState = OHCI_STATE_READY;
				}
				else
				{
					DbgUsbHost("Port has been disabled...\n");
					oUsbHost.m_eOhciState = OHCI_STATE_ATTACHED;
				}
			}
		}
		if (ohci_rhportstatus1.b.port_suspend_status_change)
		{
			// must use uhShiftOutp32 instead of uhShiftSet32, in order to set only corresponding bit
			uhShiftOutp32(rUH_OHCI_HcRhPortStatus, 1, 18);	// clearing this bit
		}
		if (ohci_rhportstatus1.b.port_overcurrent_indicator_change)
		{
			// must use uhShiftOutp32 instead of uhShiftSet32, in order to set only corresponding bit
			uhShiftOutp32(rUH_OHCI_HcRhPortStatus, 1, 19);	// clearing this bit
		}
		if (ohci_rhportstatus1.b.port_reset_status_change)
		{
			// must use uhShiftOutp32 instead of uhShiftSet32, in order to set only corresponding bit
			uhShiftOutp32(rUH_OHCI_HcRhPortStatus, 1, 20);	// clearing this bit
			DbgUsbHost("Reset Status Change Interrupt...\n");

			if ((g_bOhciPortEnabled == false)&&(ohci_rhportstatus1.b.pes_spe == 1))
			{
				DbgUsbHost("Port has been enabled....\n");
				g_bOhciPortEnabled = true;
				oUsbHost.m_eOhciState = OHCI_STATE_READY;
			}
			
		}		
	}
	if (ohci_interruptstatus.b.ownership_change)
	{
		DbgUsbHost("\n[OHCI]Ownership Change Interrupt...\n");
	}
}

//////////
// Function Name : USBHOST_InitPhyCon
// Function Desctiption : This function initializes USB HOST PHY.
// Input : NONE
// Output : NONE
// Version :
void USBHOST_InitPhyCon(void)
{
	// . phy-power on 
	//---------------
	uhOutp32(rUPHY_CON_PHYPWR, uhInp32(rUPHY_CON_PHYPWR)&~(0x7<<6)|(0x4<<6));

	// . set clock source for PLL
	//-----------------------
	//uhOutp32(rUPHY_CON_PHYCLK, uhInp32(rUPHY_CON_PHYCLK)&~(0x3<<0)|(0x0<<0));	//48MHz
	uhOutp32(rUPHY_CON_PHYCLK, uhInp32(rUPHY_CON_PHYCLK)&~(0x3<<0)|(0x3<<0));	//24MHz
	//uhOutp32(rUPHY_CON_PHYCLK, uhInp32(rUPHY_CON_PHYCLK)&~(0x3<<0)|(0x2<<0));	//12MHz

	// . reset and release after 10us
	//--------------------------
	uhOutp32(rUPHY_CON_RSTCON, uhInp32(rUPHY_CON_RSTCON)&~(0x3<<3)|(0x1<<3));
	Delay(10);
	uhOutp32(rUPHY_CON_RSTCON, uhInp32(rUPHY_CON_RSTCON)&~(0x3<<3));
	Delay(10);	

	// . tune phy
	//-----------
	//uhOutp32(rUPHY_CON_PHYTUNE1, 0x919B3);
	
}


//////////
// Function Name : USBHOST_InitEhciCon
// Function Desctiption : This function initializes USB HOST EHCI Controller.
// Input : NONE
// Output : NONE
// Version :
#define USBHOST_AHB_INCR16	0x80000
#define USBHOST_AHB_INCR8		0x40000
#define USBHOST_AHB_INCR4		0x20000
#define USBHOST_AHB_INCRx		0xe0000
#define USBHOST_AHB_SINGLE	0x00000
void USBHOST_InitEhciCon(void)
{
	ehci_portsc_t ehci_portsc;
	ehci_usbcmd_t ehci_usbcmd;
	ehci_usbintr_t ehci_usbintr;
	ehci_configflag_t ehci_configflag;

	oUsbHost.m_eEhciState = EHCI_STATE_INIT;

	// . set AHB master i/f type (invalid at EVT0)
	//------------------------------------
	uhOutp32(rUH_EHCI_OP_INSNREG00, USBHOST_AHB_SINGLE);

	// . program packet buffer threshold
	//------------------------------
	uhOutp32(rUH_EHCI_OP_INSNREG01, 0x00800040);	// out:512, in:256
	

	// . program CTRLDSSEGMENT regsiter
	//---------------------------------
	// this register is not used when not using 64-bit addressing

	
	// . stop the host controller 
	//-----------------------
	uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	ehci_usbcmd.b.run_stop = 0;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	USBHOST_WaitEhciHcHalted();
	
	
	// . reset the ehci controller & wait until reset operation is completed
	//--------------------------------------------------------
	ehci_usbcmd.b.hc_reset = 1;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	do
	{
		uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	} while(ehci_usbcmd.b.hc_reset);


	// . program USBINTR with appropriat values
	//---------------------------------------
	ehci_usbintr.data = 0;
	ehci_usbintr.b.usbint_en= 1;
	ehci_usbintr.b.usberrint_en = 1;
	ehci_usbintr.b.p_chng_int_en = 1;
	uhOutp32(rUH_EHCI_OP_USBINTR, ehci_usbintr.data);


	// . Write the base address of the periodic frame list to the PERIODICLISTBASE register
	//------------------------------------------------------------------------
	uhOutp32(rUH_EHCI_OP_PERIODICLISTBASE, 0);	// must be modified


	// . set port power-on
	//------------------
	uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);
	ehci_portsc.b.p_power = 1;
	uhOutp32(rUH_EHCI_OP_PORTSC, ehci_portsc.data);


	// . start Host controller
	//--------------------
	USBHOST_WaitEhciHcHalted();
	
	ehci_usbcmd.b.run_stop = 1;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);


	// . route port ownership to the EHCI controller
	//---------------------------------------
	ehci_configflag.data = 0;
	ehci_configflag.b.cf = 1;
	uhOutp32(rUH_EHCI_OP_CONFIGFLAG, ehci_configflag.data);
	
}


//////////
// Function Name : USBHOST_InitOhciCon
// Function Desctiption : This function initializes USB HOST OHCI Controller.
// Input : NONE
// Output : NONE
// Version :
void USBHOST_InitOhciCon(void)
{
	ohci_control_t ohci_control;
	ohci_commandstatus_t ohci_commandstatus;
	ohci_fminterval_t ohci_fminterval;
	ohci_interruptenable_t ohci_interruptenable;
	ohci_rhstatus_t ohci_rhstatus;
	ohci_rhportstatus1_t ohci_rhportstatus;
	u16 i;
	
	oUsbHost.m_eOhciState = OHCI_STATE_INIT;
	 g_bCurrentConnectStatus = false;
	
	// . take ownership 
	//-----------------------
	uhInp32_(rUH_OHCI_HcCommandStatus, ohci_commandstatus.data);
	ohci_commandstatus.b.ownership_change_request = 1;
	uhOutp32(rUH_OHCI_HcCommandStatus, ohci_commandstatus.data);

	// . wait until changeover
	//---------------------
	do
	{
		Delay(1);
		uhInp32_(rUH_OHCI_HcCommandStatus, ohci_commandstatus.data);
	}while(ohci_commandstatus.b.ownership_change_request == 1);

	// . write frame interval register
	//---------------------------
	uhInp32_(rUH_OHCI_HcFmInterval, ohci_fminterval.data);
	ohci_fminterval.b.fs_largest_data_packet = (256*8);
	uhOutp32(rUH_OHCI_HcFmInterval, ohci_fminterval.data);
	


	// . allocate HCCA and then write the addr to HcHCCA register
	//---------------------------------------------------
	ohci_hcca_base = (u32 *)USBHOST_AllocateDataStructure(OHCI_HCCA_SIZE, 256);
	for (i=0;i<(OHCI_HCCA_SIZE/4);i++)
	{
		ohci_hcca_base[i] = 0;
	}
	uhOutp32(rUH_OHCI_HcHCCA, (u32)ohci_hcca_base);
	
	

	// . enable interrupts
	//------------------
	ohci_interruptenable.data = 0;
	ohci_interruptenable.b.writeback_donehead = 1;
	ohci_interruptenable.b.root_hub_status_change = 1;
	ohci_interruptenable.b.master_interrupt_enable = 1;
	uhOutp32(rUH_OHCI_HcInterruptEnable, ohci_interruptenable.data);

	// . set hc to operational state
	//-------------------------
	uhInp32_(rUH_OHCI_HcControl, ohci_control.data);
	ohci_control.b.hc_function_state = OHCI_USB_OPERATIONAL;
	uhOutp32(rUH_OHCI_HcControl, ohci_control.data);

	// . set global power
	//-----------------
	uhInp32_(rUH_OHCI_HcRhStatus, ohci_rhstatus.data);
	ohci_rhstatus.b.lpsc_sgp = 1;
	uhOutp32(rUH_OHCI_HcRhStatus, ohci_rhstatus.data);

	// . wait until connect is detected
	//----------------------------
	USBHOST_OhciWaitEhciStateChange(OHCI_STATE_ATTACHED);	
	
}


//////////
// Function Name : USBHOST_WaitEhciHcHalted
// Function Desctiption : This function waits until host controller is halted.
// Input : NONE
// Output : NONE
// Version :
void USBHOST_WaitEhciHcHalted(void)
{
	ehci_usbsts_t ehci_usbsts;

	do
	{
		uhInp32_(rUH_EHCI_OP_USBSTS, ehci_usbsts.data);
		if (ehci_usbsts.b.hc_halted)
			break;
	}while(1);
}

//////////
// Function Name : USBHOST_WaitCableInsertion
// Function Desctiption : This function waits until usb cable is inserted.
// Input : NONE
// Output : NONE
// Version :
void USBHOST_WaitCableInsertion(void)
{
	u8 ucFirst=1;

	do
	{
		Delay(10);
		
		if (g_bCurrentConnectStatus == true)
		{
			break;
		}

		else if (ucFirst == 1)
		{
			UART_Printf("\n>>>Insert an USB cable into the connector!\n");
			ucFirst = 0;
		}
	}while(1);
}

//////////
// Function Name : USBHOST_EhciPortReset
// Function Desctiption : This function starts the usb bus reset sequence and then check the device speed.
// Input : NONE
// Output : eUsbSpeed, device speed
// Version :
void USBHOST_EhciPortReset(USB_EP_SPEED *eUsbEpSpeed)
{
	ehci_portsc_t ehci_portsc;
	
	g_bEhciPortEnabled = false;

	uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);

	// . start port reset
	ehci_portsc.b.p_reset = 1;
	ehci_portsc.b.p_en_dis = 0;

	uhOutp32(rUH_EHCI_OP_PORTSC, ehci_portsc.data);

	// . wait until reset is completed	
	Delay(100);	// normally 10ms (10~20ms)
	uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);
	ehci_portsc.b.p_reset = 0;
	uhOutp32(rUH_EHCI_OP_PORTSC, ehci_portsc.data);

	// . check whether high speed or not	
	Delay(10);
	uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);

	if ((ehci_portsc.b.p_en_dis==1) && (ehci_portsc.b.p_reset ==0))
	{
		*eUsbEpSpeed = USB_EP_HIGH_SPEED;
		g_bEhciPortEnabled = true;
		oUsbHost.m_eEhciState = EHCI_STATE_READY;
	}
	else
	{
		g_bEhciPortEnabled = false;
		*eUsbEpSpeed = USB_EP_FULL_SPEED;
	}
}
//////////
// Function Name : USBHOST_OhciPortReset
// Function Desctiption : This function starts the usb bus reset sequence and then check the port is enabled.
// Input : NONE
// Output : NONE
// Version :
void USBHOST_OhciPortReset(void)
{
	ohci_rhportstatus1_t ohci_rhportstatus;
	
	g_bOhciPortEnabled = false;

	// . set port reset
	uhShiftSet32(rUH_OHCI_HcRhPortStatus, 1, 4);

	// . wait until port is enabled
	USBHOST_OhciWaitEhciStateChange(OHCI_STATE_READY);
	
}


//////////
// Function Name : USBHOST_PortOwnerChange
// Function Desctiption : This function changes port ownershipt.
// Input : NONE
// Output : eUsbSpeed, device speed
// Version :
void USBHOST_PortOwnerChange(USB_PORT_OWNER ePortOwner)
{
	if (ePortOwner == OHCI_OWNER)
	{
		ehci_portsc_t ehci_portsc;
		
		// . change port ownership from ehci to ohci
		uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);
		ehci_portsc.b.p_owner = 1;
		uhOutp32(rUH_EHCI_OP_PORTSC, ehci_portsc.data);

		// . wait until port ownership gets changed to OHCI controller
		while(g_eUsbPortOwner != OHCI_OWNER);

		Delay(1);
		
	}
	else
	{
	}
}

//////////
// Function Name : USBHOST_EhciWaitEhciStateChange
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
#define USBHOST_MAX_WAITCOUNT	10000		// in unit of 100us, 1sec
void USBHOST_EhciWaitEhciStateChange(EHCI_STATE eEhciState)
{
	u32 uWaitCount = 0;
	
	while(1)
	{
		if (oUsbHost.m_eEhciState == eEhciState)
		{
			break;
		}
		else
		{			
			Delay(1);
			uWaitCount++;
}

		if (uWaitCount > USBHOST_MAX_WAITCOUNT)
		{
			Assert(0);
		}
	}
}

//////////
// Function Name : USBHOST_OhciWaitEhciStateChange
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_OhciWaitEhciStateChange(OHCI_STATE eOhciState)
{
	u32 uWaitCount = 0;
	
	while(1)
	{
		if (oUsbHost.m_eOhciState == eOhciState)
		{
			break;
		}
		else
		{			
			Delay(1);
			uWaitCount++;
		}

		if (uWaitCount > USBHOST_MAX_WAITCOUNT)
		{
			Assert(0);
		}
	}
}

//////////
// Function Name : USBHOST_WaitUsbTrCompetion
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_WaitUsbTrCompetion(u8 ucControlXfer)
{
	u32 uWaitCount = 0;
	
	while(1)
	{
		if (g_bUsbTrCompletion == true)
		{
			g_bUsbTrCompletion = false;
			break;
		}
		else
		{			
			Delay(1);
			uWaitCount++;
		}

		if ((ucControlXfer == true)&&(uWaitCount > USBHOST_MAX_WAITCOUNT))
		{
			Assert(0);
		}
	}
}

//////////
// Function Name : USBHOST_EhciEnableAsyncSched
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_EhciEnableAsyncSched(void)
{
	ehci_usbcmd_t ehci_usbcmd;
	ehci_usbsts_t ehci_usbsts;

	// wait until async schedule is disabled
	do
	{
		uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);		
		uhInp32_(rUH_EHCI_OP_USBSTS, ehci_usbsts.data);
		if ((ehci_usbcmd.b.async_sched_en == 0)&&(ehci_usbsts.b.async_sched_sts == 0))
			break;
	}while(1);
	
	ehci_usbcmd.b.async_sched_en = 1;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
}

//////////
// Function Name : USBHOST_EhciDisableAsyncSched
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_EhciDisableAsyncSched(void)
{
	ehci_usbcmd_t ehci_usbcmd;
	ehci_usbsts_t ehci_usbsts;

	uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);	
	
	ehci_usbcmd.b.async_sched_en = 0;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	
	// wait until async schedule is disabled
	do
	{
		uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);		
		uhInp32_(rUH_EHCI_OP_USBSTS, ehci_usbsts.data);
		if ((ehci_usbcmd.b.async_sched_en == 0)&&(ehci_usbsts.b.async_sched_sts == 0))
			break;
	}while(1);
}

//////////
// Function Name : USBHOST_EhciEnableAsyncSched
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_OhciEnableList(u8 bControl, u32 uListAddr, u32 uLastTdAddr)
{
	ohci_commandstatus_t ohci_commandstatus;
	ohci_control_t ohci_control;
	
	last_td_addr = uLastTdAddr;

	if (bControl == 1)
	{
		uhInp32_(rUH_OHCI_HcCommandStatus, ohci_commandstatus.data);
		ohci_commandstatus.b.control_list_filled = 1;
		uhOutp32(rUH_OHCI_HcCommandStatus, ohci_commandstatus.data);

		uhOutp32(rUH_OHCI_HcControlHeadED, uListAddr);
		
		uhInp32_(rUH_OHCI_HcControl, ohci_control.data);
		ohci_control.b.control_list_enable = 1;
		uhOutp32(rUH_OHCI_HcControl, ohci_control.data);		
	}
	else
	{
		uhInp32_(rUH_OHCI_HcCommandStatus, ohci_commandstatus.data);
		ohci_commandstatus.b.bulk_list_filled = 1;
		uhOutp32(rUH_OHCI_HcCommandStatus, ohci_commandstatus.data);

		uhOutp32(rUH_OHCI_HcBulkHeadED, uListAddr);
		
		uhInp32_(rUH_OHCI_HcControl, ohci_control.data);
		ohci_control.b.bulk_list_enable = 1;
		uhOutp32(rUH_OHCI_HcControl, ohci_control.data);	
	}

}

//////////
// Function Name : USBHOST_OhciDisableList
// Function Desctiption : This function 
// Input : bControl
// Output : NONE
// Version :
void USBHOST_OhciDisableList(u8 bControl)
{
	ohci_control_t ohci_control;

	if (bControl == 1)
	{		
		uhInp32_(rUH_OHCI_HcControl, ohci_control.data);
		ohci_control.b.control_list_enable = 0;
		uhOutp32(rUH_OHCI_HcControl, ohci_control.data);		
	}
	else
	{		
		uhInp32_(rUH_OHCI_HcControl, ohci_control.data);
		ohci_control.b.bulk_list_enable = 0;
		uhOutp32(rUH_OHCI_HcControl, ohci_control.data);	
	}

}

//////////
// Function Name : USBHOST_EhciGetDeviceDescriptor
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_EhciGetDeviceDescriptor(u16 usDevAddr)
{
	u32 i;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_QH;i++)
	{
		ehci_qh_ptr[i] = (ehci_qh_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qh_t), 32);
	}
	for(i=0;i<NUM_OF_qTD;i++)
	{
		ehci_qtd_ptr[i] = (ehci_qtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qtd_t), 32);
	}

	ehci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);
	ehci_in_data_ptr = (u8 *)USBHOST_AllocateDataStructure(DEVICE_DESC_SIZE, 4096);

	// . fill QH
	//---------
	// >> DWORD0
	ehci_qh_ptr[0]->next_qh_ptr.data = 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.t= 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.typ = 1;	// queue head
	ehci_qh_ptr[0]->next_qh_ptr.b.qhlp = (u32)ehci_qh_ptr[0]>>5;
	// >> DWORD1
	ehci_qh_ptr[0]->info1.data = 0;
	ehci_qh_ptr[0]->info1.b.dev_addr = usDevAddr;
	ehci_qh_ptr[0]->info1.b.i= 0;
	ehci_qh_ptr[0]->info1.b.ep_num = 0;
	ehci_qh_ptr[0]->info1.b.eps = USB_EP_HIGH_SPEED;
	ehci_qh_ptr[0]->info1.b.dtc = 1;
	ehci_qh_ptr[0]->info1.b.h = 1;	// ?khs
	if (usDevAddr == 0)
	{
		ehci_qh_ptr[0]->info1.b.max_pkt_sz= HIGH_SPEED_CONTROL_PKT_SIZE;
	}
	else
	{
		ehci_qh_ptr[0]->info1.b.max_pkt_sz= oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	}
	ehci_qh_ptr[0]->info1.b.c = 0;
	ehci_qh_ptr[0]->info1.b.rl = 0;
	// >> DWORD2
	ehci_qh_ptr[0]->info2.data = 0;
	ehci_qh_ptr[0]->info2.b.mult = ONE_TR_uFRAME;
	// >> DWORD3
	ehci_qh_ptr[0]->current_qtd_ptr = 0;
	// >> DWORD4
	ehci_qh_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[0];
	// >> DWORD5
	ehci_qh_ptr[0]->alt_next_qtd_ptr = 1;	// invalid
	// >> DWORD6~11
	ehci_qh_ptr[0]->qtd_token = 0;
	ehci_qh_ptr[0]->buf_ptr0 = 0;
	ehci_qh_ptr[0]->buf_ptr1 = 0;
	ehci_qh_ptr[0]->buf_ptr2 = 0;
	ehci_qh_ptr[0]->buf_ptr3 = 0;
	ehci_qh_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ehci_setup_data_ptr->bmRequestType = DEVICE_TO_HOST | STANDARD_TYPE | DEVICE_RECIPIENT;
	ehci_setup_data_ptr->bRequest = STANDARD_GET_DESCRIPTOR;
	ehci_setup_data_ptr->wValue_L= 0;
	ehci_setup_data_ptr->wValue_H = DEVICE_DESCRIPTOR;
	ehci_setup_data_ptr->wIndex_L = 0;
	ehci_setup_data_ptr->wIndex_H = 0;
	ehci_setup_data_ptr->wLength_L = HIGH_SPEED_CONTROL_PKT_SIZE;
	ehci_setup_data_ptr->wLength_H = 0;
	// qTD[0]
	ehci_qtd_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[1];
	ehci_qtd_ptr[0]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[0]->qtd_token.data = 0;
	ehci_qtd_ptr[0]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[0]->qtd_token.b.pid = USB_SETUP_TOKEN;
	ehci_qtd_ptr[0]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[0]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[0]->qtd_token.b.ioc = 0;
	ehci_qtd_ptr[0]->qtd_token.b.xfer_total_bytes = SETUP_DATA_SIZE;
	ehci_qtd_ptr[0]->qtd_token.b.data_toggle = 0;
	ehci_qtd_ptr[0]->buf_ptr0 = (u32)ehci_setup_data_ptr;		// should wite the addr of setup packet data
	ehci_qtd_ptr[0]->buf_ptr1 = 0;
	ehci_qtd_ptr[0]->buf_ptr2 = 0;
	ehci_qtd_ptr[0]->buf_ptr3 = 0;
	ehci_qtd_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[1] for IN data
	//--------------------------
	ehci_qtd_ptr[1]->next_qtd_ptr = (u32)ehci_qtd_ptr[2];
	ehci_qtd_ptr[1]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[1]->qtd_token.data = 0;
	ehci_qtd_ptr[1]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[1]->qtd_token.b.pid = USB_IN_TOKEN;
	ehci_qtd_ptr[1]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[1]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[1]->qtd_token.b.ioc = 0;
	ehci_qtd_ptr[1]->qtd_token.b.xfer_total_bytes = DEVICE_DESC_SIZE;
	ehci_qtd_ptr[1]->qtd_token.b.data_toggle = 1;
	ehci_qtd_ptr[1]->buf_ptr0 = (u32)ehci_in_data_ptr;		// should wite the addr of IN data packet
	ehci_qtd_ptr[1]->buf_ptr1 = 0;
	ehci_qtd_ptr[1]->buf_ptr2 = 0;
	ehci_qtd_ptr[1]->buf_ptr3 = 0;
	ehci_qtd_ptr[1]->buf_ptr4 = 0;
	
	// . fill qTD[2] for OUT status
	//--------------------------
	ehci_qtd_ptr[2]->next_qtd_ptr = 1;	// invalid
	ehci_qtd_ptr[2]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[2]->qtd_token.data = 0;
	ehci_qtd_ptr[2]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[2]->qtd_token.b.pid = USB_OUT_TOKEN;
	ehci_qtd_ptr[2]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[2]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[2]->qtd_token.b.ioc = 1;
	ehci_qtd_ptr[2]->qtd_token.b.xfer_total_bytes = 0;
	ehci_qtd_ptr[2]->qtd_token.b.data_toggle = 1;
	ehci_qtd_ptr[2]->buf_ptr0 = 0;
	ehci_qtd_ptr[2]->buf_ptr1 = 0;
	ehci_qtd_ptr[2]->buf_ptr2 = 0;
	ehci_qtd_ptr[2]->buf_ptr3 = 0;
	ehci_qtd_ptr[2]->buf_ptr4 = 0;

	// . update Async Schedule & enable Async Schedule
	//--------------------------------------------	
	uhOutp32(rUH_EHCI_OP_ASYNCLISTADDR, (u32)ehci_qh_ptr[0]&0xffffffe0);

	USBHOST_EhciEnableAsyncSched();
	
}

//////////
// Function Name : USBHOST_OhciGetDeviceDescriptor
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_OhciGetDeviceDescriptor(u16 usDevAddr)
{
	u32 i;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_ED;i++)
	{
		ohci_ed_ptr[i] = (ohci_ed_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_ed_t), 32);
	}
	for(i=0;i<NUM_OF_gTD;i++)
	{
		ohci_gtd_ptr[i] = (ohci_gtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_gtd_t), 32);
	}

	ohci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);
	ohci_in_data_ptr = (u8 *)USBHOST_AllocateDataStructure(DEVICE_DESC_SIZE, 4096);

	// . fill ED
	//---------
	ohci_ed_ptr[0]->ed_info.data = 0;
	ohci_ed_ptr[0]->ed_info.b.func_addr = usDevAddr;
	ohci_ed_ptr[0]->ed_info.b.ep_num = 0;
	ohci_ed_ptr[0]->ed_info.b.direction = 0;
	ohci_ed_ptr[0]->ed_info.b.speed = 0;	// full-speed
	ohci_ed_ptr[0]->ed_info.b.format = 0;	// gTD
	if (usDevAddr == 0)
	{
		ohci_ed_ptr[0]->ed_info.b.mps = FULL_SPEED_CONTROL_PKT_SIZE;
	}
	else
	{
		ohci_ed_ptr[0]->ed_info.b.mps = oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	}
	
	ohci_ed_ptr[0]->tail_td_ptr = 0;
	
	ohci_ed_ptr[0]->head_td_ptr = (u32)ohci_gtd_ptr[0];
	
	ohci_ed_ptr[0]->next_ed = 0;

	
	// . fill gTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ohci_setup_data_ptr->bmRequestType = DEVICE_TO_HOST | STANDARD_TYPE | DEVICE_RECIPIENT;
	ohci_setup_data_ptr->bRequest = STANDARD_GET_DESCRIPTOR;
	ohci_setup_data_ptr->wValue_L= 0;
	ohci_setup_data_ptr->wValue_H = DEVICE_DESCRIPTOR;
	ohci_setup_data_ptr->wIndex_L = 0;
	ohci_setup_data_ptr->wIndex_H = 0;
	ohci_setup_data_ptr->wLength_L = FULL_SPEED_CONTROL_PKT_SIZE;
	ohci_setup_data_ptr->wLength_H = 0;
	
	// gTD[0]
	ohci_gtd_ptr[0]->gtd_info.data = 0;
	ohci_gtd_ptr[0]->gtd_info.b.buffer_rounding = 0;
	ohci_gtd_ptr[0]->gtd_info.b.pid = 0;		// SETUP
	ohci_gtd_ptr[0]->gtd_info.b.data_toggle = 2;	// DATA0
	
	ohci_gtd_ptr[0]->current_buf_ptr = (u32)ohci_setup_data_ptr;

	ohci_gtd_ptr[0]->next_td = (u32)ohci_gtd_ptr[1];

	ohci_gtd_ptr[0]->buffer_end = (u32)ohci_setup_data_ptr + SETUP_DATA_SIZE-1;
	
	// gTD[1]
	ohci_gtd_ptr[1]->gtd_info.data = 0;
	ohci_gtd_ptr[1]->gtd_info.b.buffer_rounding = 1;	// may be smaller than the defined buffer
	ohci_gtd_ptr[1]->gtd_info.b.pid = 2;		// IN
	ohci_gtd_ptr[1]->gtd_info.b.data_toggle = 3;	// DATA1
	
	ohci_gtd_ptr[1]->current_buf_ptr = (u32)ohci_in_data_ptr;

	ohci_gtd_ptr[1]->next_td = (u32)ohci_gtd_ptr[2];

	ohci_gtd_ptr[1]->buffer_end = (u32)ohci_in_data_ptr + DEVICE_DESC_SIZE-1;
	
	// gTD[2]
	ohci_gtd_ptr[2]->gtd_info.data = 0;
	ohci_gtd_ptr[2]->gtd_info.b.buffer_rounding = 1;	// may be smaller than the defined buffer
	ohci_gtd_ptr[2]->gtd_info.b.pid = 1;		// OUT
	ohci_gtd_ptr[2]->gtd_info.b.data_toggle = 3;	// DATA1
	
	ohci_gtd_ptr[2]->current_buf_ptr = 0;

	ohci_gtd_ptr[2]->next_td = 0;

	ohci_gtd_ptr[2]->buffer_end = 0;


	USBHOST_OhciEnableList(1, (u32)ohci_ed_ptr[0], (u32)ohci_gtd_ptr[2]);
	
}


//////////
// Function Name : USBHOST_EhciSetAddress
// Function Desctiption : This function 
// Input : usDevAddr
// Output : NONE
// Version :
void USBHOST_EhciSetAddress(u16 usDevAddr)
{
	u32 i;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_QH;i++)
	{
		ehci_qh_ptr[i] = (ehci_qh_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qh_t), 32);
	}
	for(i=0;i<NUM_OF_qTD;i++)
	{
		ehci_qtd_ptr[i] = (ehci_qtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qtd_t), 32);
	}

	ehci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);

	// . fill QH
	//---------
	// >> DWORD0
	ehci_qh_ptr[0]->next_qh_ptr.data = 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.t= 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.typ = 1;	// queue head
	ehci_qh_ptr[0]->next_qh_ptr.b.qhlp = (u32)ehci_qh_ptr[0]>>5;
	// >> DWORD1
	ehci_qh_ptr[0]->info1.data = 0;
	ehci_qh_ptr[0]->info1.b.dev_addr = 0;
	ehci_qh_ptr[0]->info1.b.i= 0;
	ehci_qh_ptr[0]->info1.b.ep_num = 0;
	ehci_qh_ptr[0]->info1.b.eps = USB_EP_HIGH_SPEED;
	ehci_qh_ptr[0]->info1.b.dtc = 1;
	ehci_qh_ptr[0]->info1.b.h = 1;	// ?khs
	ehci_qh_ptr[0]->info1.b.max_pkt_sz= oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	ehci_qh_ptr[0]->info1.b.c = 0;
	ehci_qh_ptr[0]->info1.b.rl = 0;
	// >> DWORD2
	ehci_qh_ptr[0]->info2.data = 0;
	ehci_qh_ptr[0]->info2.b.mult = ONE_TR_uFRAME;
	// >> DWORD3
	ehci_qh_ptr[0]->current_qtd_ptr = 0;
	// >> DWORD4
	ehci_qh_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[0];
	// >> DWORD5
	ehci_qh_ptr[0]->alt_next_qtd_ptr = 1;	// invalid
	// >> DWORD6~11
	ehci_qh_ptr[0]->qtd_token = 0;
	ehci_qh_ptr[0]->buf_ptr0 = 0;
	ehci_qh_ptr[0]->buf_ptr1 = 0;
	ehci_qh_ptr[0]->buf_ptr2 = 0;
	ehci_qh_ptr[0]->buf_ptr3 = 0;
	ehci_qh_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ehci_setup_data_ptr->bmRequestType = HOST_TO_DEVICE| STANDARD_TYPE | DEVICE_RECIPIENT;
	ehci_setup_data_ptr->bRequest = STANDARD_SET_ADDRESS;
	ehci_setup_data_ptr->wValue_L= usDevAddr;
	ehci_setup_data_ptr->wValue_H = 0;
	ehci_setup_data_ptr->wIndex_L = 0;
	ehci_setup_data_ptr->wIndex_H = 0;
	ehci_setup_data_ptr->wLength_L = 0;
	ehci_setup_data_ptr->wLength_H = 0;
	// qTD[0]
	ehci_qtd_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[1];
	ehci_qtd_ptr[0]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[0]->qtd_token.data = 0;
	ehci_qtd_ptr[0]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[0]->qtd_token.b.pid = USB_SETUP_TOKEN;
	ehci_qtd_ptr[0]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[0]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[0]->qtd_token.b.ioc = 0;
	ehci_qtd_ptr[0]->qtd_token.b.xfer_total_bytes = SETUP_DATA_SIZE;
	ehci_qtd_ptr[0]->qtd_token.b.data_toggle = 0;
	ehci_qtd_ptr[0]->buf_ptr0 = (u32)ehci_setup_data_ptr;		// should wite the addr of setup packet data
	ehci_qtd_ptr[0]->buf_ptr1 = 0;
	ehci_qtd_ptr[0]->buf_ptr2 = 0;
	ehci_qtd_ptr[0]->buf_ptr3 = 0;
	ehci_qtd_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[1] for IN data
	//--------------------------
	ehci_qtd_ptr[1]->next_qtd_ptr = 1;
	ehci_qtd_ptr[1]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[1]->qtd_token.data = 0;
	ehci_qtd_ptr[1]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[1]->qtd_token.b.pid = USB_IN_TOKEN;
	ehci_qtd_ptr[1]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[1]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[1]->qtd_token.b.ioc = 1;
	ehci_qtd_ptr[1]->qtd_token.b.xfer_total_bytes = 0;
	ehci_qtd_ptr[1]->qtd_token.b.data_toggle = 1;
	ehci_qtd_ptr[1]->buf_ptr0 = 0;		// should wite the addr of IN data packet
	ehci_qtd_ptr[1]->buf_ptr1 = 0;
	ehci_qtd_ptr[1]->buf_ptr2 = 0;
	ehci_qtd_ptr[1]->buf_ptr3 = 0;
	ehci_qtd_ptr[1]->buf_ptr4 = 0;

	// . update Async Schedule & enable Async Schedule
	//--------------------------------------------
	uhOutp32(rUH_EHCI_OP_ASYNCLISTADDR, (u32)ehci_qh_ptr[0]&0xffffffe0);

	USBHOST_EhciEnableAsyncSched();
	
}


//////////
// Function Name : USBHOST_OhciSetAddress
// Function Desctiption : This function 
// Input : usDevAddr
// Output : NONE
// Version :
void USBHOST_OhciSetAddress(u16 usDevAddr)
{
	u32 i;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_ED;i++)
	{
		ohci_ed_ptr[i] = (ohci_ed_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_ed_t), 32);
	}
	for(i=0;i<NUM_OF_gTD;i++)
	{
		ohci_gtd_ptr[i] = (ohci_gtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_gtd_t), 32);
	}

	ohci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);

	// . fill ED
	//---------
	ohci_ed_ptr[0]->ed_info.data = 0;
	ohci_ed_ptr[0]->ed_info.b.func_addr = 0;
	ohci_ed_ptr[0]->ed_info.b.ep_num = 0;
	ohci_ed_ptr[0]->ed_info.b.direction = 0;
	ohci_ed_ptr[0]->ed_info.b.speed = 0;	// full-speed
	ohci_ed_ptr[0]->ed_info.b.format = 0;	// gTD
	if (usDevAddr == 0)
	{
		ohci_ed_ptr[0]->ed_info.b.mps = FULL_SPEED_CONTROL_PKT_SIZE;
	}
	else
	{
		ohci_ed_ptr[0]->ed_info.b.mps = oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	}
	
	ohci_ed_ptr[0]->tail_td_ptr = 0;
	
	ohci_ed_ptr[0]->head_td_ptr = (u32)ohci_gtd_ptr[0];
	
	ohci_ed_ptr[0]->next_ed = 0;
	
	
	// . fill gTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ohci_setup_data_ptr->bmRequestType = HOST_TO_DEVICE| STANDARD_TYPE | DEVICE_RECIPIENT;
	ohci_setup_data_ptr->bRequest = STANDARD_SET_ADDRESS;
	ohci_setup_data_ptr->wValue_L= usDevAddr;
	ohci_setup_data_ptr->wValue_H = 0;
	ohci_setup_data_ptr->wIndex_L = 0;
	ohci_setup_data_ptr->wIndex_H = 0;
	ohci_setup_data_ptr->wLength_L = 0;
	ohci_setup_data_ptr->wLength_H = 0;
	
	// gTD[0]
	ohci_gtd_ptr[0]->gtd_info.data = 0;
	ohci_gtd_ptr[0]->gtd_info.b.buffer_rounding = 0;
	ohci_gtd_ptr[0]->gtd_info.b.pid = 0;		// SETUP
	ohci_gtd_ptr[0]->gtd_info.b.data_toggle = 2;	// DATA0
	
	ohci_gtd_ptr[0]->current_buf_ptr = (u32)ohci_setup_data_ptr;

	ohci_gtd_ptr[0]->next_td = (u32)ohci_gtd_ptr[1];

	ohci_gtd_ptr[0]->buffer_end = (u32)ohci_setup_data_ptr + SETUP_DATA_SIZE-1;
	
	// gTD[1]
	ohci_gtd_ptr[1]->gtd_info.data = 0;
	ohci_gtd_ptr[1]->gtd_info.b.buffer_rounding = 1;	// may be smaller than the defined buffer
	ohci_gtd_ptr[1]->gtd_info.b.pid = 2;		// IN
	ohci_gtd_ptr[1]->gtd_info.b.data_toggle = 3;	// DATA1
	
	ohci_gtd_ptr[1]->current_buf_ptr = 0;

	ohci_gtd_ptr[1]->next_td = 0;

	ohci_gtd_ptr[1]->buffer_end = 0;


	USBHOST_OhciEnableList(1, (u32)ohci_ed_ptr[0], (u32)ohci_gtd_ptr[1]);
	
}

//////////
// Function Name : USBHOST_EhciGetConfigDescriptor
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_EhciGetConfigDescriptor(void)
{
	u32 i;
	u32 uRequestConfigSize = 0xff;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_QH;i++)
	{
		ehci_qh_ptr[i] = (ehci_qh_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qh_t), 32);
	}
	for(i=0;i<NUM_OF_qTD;i++)
	{
		ehci_qtd_ptr[i] = (ehci_qtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qtd_t), 32);
	}

	ehci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);
	ehci_in_data_ptr = (u8 *)USBHOST_AllocateDataStructure(uRequestConfigSize, 4096);

	// . fill QH
	//---------
	// >> DWORD0
	ehci_qh_ptr[0]->next_qh_ptr.data = 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.t= 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.typ = 1;	// queue head
	ehci_qh_ptr[0]->next_qh_ptr.b.qhlp = (u32)ehci_qh_ptr[0]>>5;
	// >> DWORD1
	ehci_qh_ptr[0]->info1.data = 0;
	ehci_qh_ptr[0]->info1.b.dev_addr = oUsbDevice.m_usDevAddr;
	ehci_qh_ptr[0]->info1.b.i= 0;
	ehci_qh_ptr[0]->info1.b.ep_num = 0;
	ehci_qh_ptr[0]->info1.b.eps = USB_EP_HIGH_SPEED;
	ehci_qh_ptr[0]->info1.b.dtc = 1;
	ehci_qh_ptr[0]->info1.b.h = 1;	// ?khs
	ehci_qh_ptr[0]->info1.b.max_pkt_sz= oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	ehci_qh_ptr[0]->info1.b.c = 0;
	ehci_qh_ptr[0]->info1.b.rl = 0;
	// >> DWORD2
	ehci_qh_ptr[0]->info2.data = 0;
	ehci_qh_ptr[0]->info2.b.mult = ONE_TR_uFRAME;
	// >> DWORD3
	ehci_qh_ptr[0]->current_qtd_ptr = 0;
	// >> DWORD4
	ehci_qh_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[0];
	// >> DWORD5
	ehci_qh_ptr[0]->alt_next_qtd_ptr = 1;	// invalid
	// >> DWORD6~11
	ehci_qh_ptr[0]->qtd_token = 0;
	ehci_qh_ptr[0]->buf_ptr0 = 0;
	ehci_qh_ptr[0]->buf_ptr1 = 0;
	ehci_qh_ptr[0]->buf_ptr2 = 0;
	ehci_qh_ptr[0]->buf_ptr3 = 0;
	ehci_qh_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ehci_setup_data_ptr->bmRequestType = DEVICE_TO_HOST | STANDARD_TYPE | DEVICE_RECIPIENT;
	ehci_setup_data_ptr->bRequest = STANDARD_GET_DESCRIPTOR;
	ehci_setup_data_ptr->wValue_L= 0;
	ehci_setup_data_ptr->wValue_H = CONFIGURATION_DESCRIPTOR;
	ehci_setup_data_ptr->wIndex_L = 0;
	ehci_setup_data_ptr->wIndex_H = 0;
	ehci_setup_data_ptr->wLength_L = uRequestConfigSize;
	ehci_setup_data_ptr->wLength_H = 0;
	// qTD[0]
	ehci_qtd_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[1];
	ehci_qtd_ptr[0]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[0]->qtd_token.data = 0;
	ehci_qtd_ptr[0]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[0]->qtd_token.b.pid = USB_SETUP_TOKEN;
	ehci_qtd_ptr[0]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[0]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[0]->qtd_token.b.ioc = 0;
	ehci_qtd_ptr[0]->qtd_token.b.xfer_total_bytes = SETUP_DATA_SIZE;
	ehci_qtd_ptr[0]->qtd_token.b.data_toggle = 0;
	ehci_qtd_ptr[0]->buf_ptr0 = (u32)ehci_setup_data_ptr;		// should wite the addr of setup packet data
	ehci_qtd_ptr[0]->buf_ptr1 = 0;
	ehci_qtd_ptr[0]->buf_ptr2 = 0;
	ehci_qtd_ptr[0]->buf_ptr3 = 0;
	ehci_qtd_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[1] for IN data
	//--------------------------
	ehci_qtd_ptr[1]->next_qtd_ptr = (u32)ehci_qtd_ptr[2];
	ehci_qtd_ptr[1]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[1]->qtd_token.data = 0;
	ehci_qtd_ptr[1]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[1]->qtd_token.b.pid = USB_IN_TOKEN;
	ehci_qtd_ptr[1]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[1]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[1]->qtd_token.b.ioc = 0;
	ehci_qtd_ptr[1]->qtd_token.b.xfer_total_bytes = uRequestConfigSize;
	ehci_qtd_ptr[1]->qtd_token.b.data_toggle = 1;
	ehci_qtd_ptr[1]->buf_ptr0 = (u32)ehci_in_data_ptr;		// should wite the addr of IN data packet
	ehci_qtd_ptr[1]->buf_ptr1 = 0;
	ehci_qtd_ptr[1]->buf_ptr2 = 0;
	ehci_qtd_ptr[1]->buf_ptr3 = 0;
	ehci_qtd_ptr[1]->buf_ptr4 = 0;
	
	// . fill qTD[2] for OUT status
	//--------------------------
	ehci_qtd_ptr[2]->next_qtd_ptr = 1;	// invalid
	ehci_qtd_ptr[2]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[2]->qtd_token.data = 0;
	ehci_qtd_ptr[2]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[2]->qtd_token.b.pid = USB_OUT_TOKEN;
	ehci_qtd_ptr[2]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[2]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[2]->qtd_token.b.ioc = 1;
	ehci_qtd_ptr[2]->qtd_token.b.xfer_total_bytes = 0;
	ehci_qtd_ptr[2]->qtd_token.b.data_toggle = 1;
	ehci_qtd_ptr[2]->buf_ptr0 = 0;
	ehci_qtd_ptr[2]->buf_ptr1 = 0;
	ehci_qtd_ptr[2]->buf_ptr2 = 0;
	ehci_qtd_ptr[2]->buf_ptr3 = 0;
	ehci_qtd_ptr[2]->buf_ptr4 = 0;

	// . update Async Schedule & enable Async Schedule
	//--------------------------------------------	
	uhOutp32(rUH_EHCI_OP_ASYNCLISTADDR, (u32)ehci_qh_ptr[0]&0xffffffe0);

	USBHOST_EhciEnableAsyncSched();
	
}

//////////
// Function Name : USBHOST_OhciGetConfigDescriptor
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_OhciGetConfigDescriptor(void)
{
	u32 i;
	u32 uRequestConfigSize = 0xff;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_ED;i++)
	{
		ohci_ed_ptr[i] = (ohci_ed_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_ed_t), 32);
	}
	for(i=0;i<NUM_OF_gTD;i++)
	{
		ohci_gtd_ptr[i] = (ohci_gtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_gtd_t), 32);
	}

	ohci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);
	ohci_in_data_ptr = (u8 *)USBHOST_AllocateDataStructure(uRequestConfigSize, 4096);

	// . fill ED
	//---------
	ohci_ed_ptr[0]->ed_info.data = 0;
	ohci_ed_ptr[0]->ed_info.b.func_addr = oUsbDevice.m_usDevAddr;
	ohci_ed_ptr[0]->ed_info.b.ep_num = 0;
	ohci_ed_ptr[0]->ed_info.b.direction = 0;
	ohci_ed_ptr[0]->ed_info.b.speed = 0;	// full-speed
	ohci_ed_ptr[0]->ed_info.b.format = 0;	// gTD
	ohci_ed_ptr[0]->ed_info.b.mps = oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	
	ohci_ed_ptr[0]->tail_td_ptr = 0;
	
	ohci_ed_ptr[0]->head_td_ptr = (u32)ohci_gtd_ptr[0];
	
	ohci_ed_ptr[0]->next_ed = 0;

	
	// . fill gTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ohci_setup_data_ptr->bmRequestType = DEVICE_TO_HOST | STANDARD_TYPE | DEVICE_RECIPIENT;
	ohci_setup_data_ptr->bRequest = STANDARD_GET_DESCRIPTOR;
	ohci_setup_data_ptr->wValue_L= 0;
	ohci_setup_data_ptr->wValue_H = CONFIGURATION_DESCRIPTOR;
	ohci_setup_data_ptr->wIndex_L = 0;
	ohci_setup_data_ptr->wIndex_H = 0;
	ohci_setup_data_ptr->wLength_L = uRequestConfigSize;
	ohci_setup_data_ptr->wLength_H = 0;
	
	// gTD[0]
	ohci_gtd_ptr[0]->gtd_info.data = 0;
	ohci_gtd_ptr[0]->gtd_info.b.buffer_rounding = 0;
	ohci_gtd_ptr[0]->gtd_info.b.pid = 0;		// SETUP
	ohci_gtd_ptr[0]->gtd_info.b.data_toggle = 2;	// DATA0
	
	ohci_gtd_ptr[0]->current_buf_ptr = (u32)ohci_setup_data_ptr;

	ohci_gtd_ptr[0]->next_td = (u32)ohci_gtd_ptr[1];

	ohci_gtd_ptr[0]->buffer_end = (u32)ohci_setup_data_ptr + SETUP_DATA_SIZE-1;
	
	// gTD[1]
	ohci_gtd_ptr[1]->gtd_info.data = 0;
	ohci_gtd_ptr[1]->gtd_info.b.buffer_rounding = 1;	// may be smaller than the defined buffer
	ohci_gtd_ptr[1]->gtd_info.b.pid = 2;		// IN
	ohci_gtd_ptr[1]->gtd_info.b.data_toggle = 3;	// DATA1
	
	ohci_gtd_ptr[1]->current_buf_ptr = (u32)ohci_in_data_ptr;

	ohci_gtd_ptr[1]->next_td = (u32)ohci_gtd_ptr[2];

	ohci_gtd_ptr[1]->buffer_end = (u32)ohci_in_data_ptr + uRequestConfigSize-1;
	
	// gTD[2]
	ohci_gtd_ptr[2]->gtd_info.data = 0;
	ohci_gtd_ptr[2]->gtd_info.b.buffer_rounding = 1;	// may be smaller than the defined buffer
	ohci_gtd_ptr[2]->gtd_info.b.pid = 1;		// OUT
	ohci_gtd_ptr[2]->gtd_info.b.data_toggle = 3;	// DATA1
	
	ohci_gtd_ptr[2]->current_buf_ptr = 0;

	ohci_gtd_ptr[2]->next_td = 0;

	ohci_gtd_ptr[2]->buffer_end = 0;


	USBHOST_OhciEnableList(1, (u32)ohci_ed_ptr[0], (u32)ohci_gtd_ptr[2]);
	
}

//////////
// Function Name : USBHOST_EhciGetStringDescriptor
// Function Desctiption : This function 
// Input : ucIndx
// Output : NONE
// Version :
void USBHOST_EhciGetStringDescriptor(u8 ucIndex)
{
	u32 i;
	u32 uRequestStringSize = 0xff;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_QH;i++)
	{
		ehci_qh_ptr[i] = (ehci_qh_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qh_t), 32);
	}
	for(i=0;i<NUM_OF_qTD;i++)
	{
		ehci_qtd_ptr[i] = (ehci_qtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qtd_t), 32);
	}

	ehci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);
	ehci_in_data_ptr = (u8 *)USBHOST_AllocateDataStructure(uRequestStringSize, 4096);

	// . fill QH
	//---------
	// >> DWORD0
	ehci_qh_ptr[0]->next_qh_ptr.data = 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.t= 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.typ = 1;	// queue head
	ehci_qh_ptr[0]->next_qh_ptr.b.qhlp = (u32)ehci_qh_ptr[0]>>5;
	// >> DWORD1
	ehci_qh_ptr[0]->info1.data = 0;
	ehci_qh_ptr[0]->info1.b.dev_addr = oUsbDevice.m_usDevAddr;
	ehci_qh_ptr[0]->info1.b.i= 0;
	ehci_qh_ptr[0]->info1.b.ep_num = 0;
	ehci_qh_ptr[0]->info1.b.eps = USB_EP_HIGH_SPEED;
	ehci_qh_ptr[0]->info1.b.dtc = 1;
	ehci_qh_ptr[0]->info1.b.h = 1;	// ?khs
	ehci_qh_ptr[0]->info1.b.max_pkt_sz= oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	ehci_qh_ptr[0]->info1.b.c = 0;
	ehci_qh_ptr[0]->info1.b.rl = 0;
	// >> DWORD2
	ehci_qh_ptr[0]->info2.data = 0;
	ehci_qh_ptr[0]->info2.b.mult = ONE_TR_uFRAME;
	// >> DWORD3
	ehci_qh_ptr[0]->current_qtd_ptr = 0;
	// >> DWORD4
	ehci_qh_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[0];
	// >> DWORD5
	ehci_qh_ptr[0]->alt_next_qtd_ptr = 1;	// invalid
	// >> DWORD6~11
	ehci_qh_ptr[0]->qtd_token = 0;
	ehci_qh_ptr[0]->buf_ptr0 = 0;
	ehci_qh_ptr[0]->buf_ptr1 = 0;
	ehci_qh_ptr[0]->buf_ptr2 = 0;
	ehci_qh_ptr[0]->buf_ptr3 = 0;
	ehci_qh_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ehci_setup_data_ptr->bmRequestType = DEVICE_TO_HOST | STANDARD_TYPE | DEVICE_RECIPIENT;
	ehci_setup_data_ptr->bRequest = STANDARD_GET_DESCRIPTOR;
	ehci_setup_data_ptr->wValue_L= ucIndex;
	ehci_setup_data_ptr->wValue_H = STRING_DESCRIPTOR;
	if (ucIndex == 0)
	{
		ehci_setup_data_ptr->wIndex_L = 0;
		ehci_setup_data_ptr->wIndex_H = 0;
	}
	else
	{
		ehci_setup_data_ptr->wIndex_L = oUsbDevice.m_oDesc.oDescStr0.pString[0];
		ehci_setup_data_ptr->wIndex_H = oUsbDevice.m_oDesc.oDescStr0.pString[1];
	}
	ehci_setup_data_ptr->wLength_L = uRequestStringSize;
	ehci_setup_data_ptr->wLength_H = 0;
	// qTD[0]
	ehci_qtd_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[1];
	ehci_qtd_ptr[0]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[0]->qtd_token.data = 0;
	ehci_qtd_ptr[0]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[0]->qtd_token.b.pid = USB_SETUP_TOKEN;
	ehci_qtd_ptr[0]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[0]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[0]->qtd_token.b.ioc = 0;
	ehci_qtd_ptr[0]->qtd_token.b.xfer_total_bytes = SETUP_DATA_SIZE;
	ehci_qtd_ptr[0]->qtd_token.b.data_toggle = 0;
	ehci_qtd_ptr[0]->buf_ptr0 = (u32)ehci_setup_data_ptr;		// should wite the addr of setup packet data
	ehci_qtd_ptr[0]->buf_ptr1 = 0;
	ehci_qtd_ptr[0]->buf_ptr2 = 0;
	ehci_qtd_ptr[0]->buf_ptr3 = 0;
	ehci_qtd_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[1] for IN data
	//--------------------------
	ehci_qtd_ptr[1]->next_qtd_ptr = (u32)ehci_qtd_ptr[2];
	ehci_qtd_ptr[1]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[1]->qtd_token.data = 0;
	ehci_qtd_ptr[1]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[1]->qtd_token.b.pid = USB_IN_TOKEN;
	ehci_qtd_ptr[1]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[1]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[1]->qtd_token.b.ioc = 0;
	ehci_qtd_ptr[1]->qtd_token.b.xfer_total_bytes = uRequestStringSize;
	ehci_qtd_ptr[1]->qtd_token.b.data_toggle = 1;
	ehci_qtd_ptr[1]->buf_ptr0 = (u32)ehci_in_data_ptr;		// should wite the addr of IN data packet
	ehci_qtd_ptr[1]->buf_ptr1 = 0;
	ehci_qtd_ptr[1]->buf_ptr2 = 0;
	ehci_qtd_ptr[1]->buf_ptr3 = 0;
	ehci_qtd_ptr[1]->buf_ptr4 = 0;
	
	// . fill qTD[2] for OUT status
	//--------------------------
	ehci_qtd_ptr[2]->next_qtd_ptr = 1;	// invalid
	ehci_qtd_ptr[2]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[2]->qtd_token.data = 0;
	ehci_qtd_ptr[2]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[2]->qtd_token.b.pid = USB_OUT_TOKEN;
	ehci_qtd_ptr[2]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[2]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[2]->qtd_token.b.ioc = 1;
	ehci_qtd_ptr[2]->qtd_token.b.xfer_total_bytes = 0;
	ehci_qtd_ptr[2]->qtd_token.b.data_toggle = 1;
	ehci_qtd_ptr[2]->buf_ptr0 = 0;
	ehci_qtd_ptr[2]->buf_ptr1 = 0;
	ehci_qtd_ptr[2]->buf_ptr2 = 0;
	ehci_qtd_ptr[2]->buf_ptr3 = 0;
	ehci_qtd_ptr[2]->buf_ptr4 = 0;

	// . update Async Schedule & enable Async Schedule
	//--------------------------------------------	
	uhOutp32(rUH_EHCI_OP_ASYNCLISTADDR, (u32)ehci_qh_ptr[0]&0xffffffe0);

	USBHOST_EhciEnableAsyncSched();
	
}

//////////
// Function Name : USBHOST_OhciGetStringDescriptor
// Function Desctiption : This function 
// Input : ucIndx
// Output : NONE
// Version :
void USBHOST_OhciGetStringDescriptor(u8 ucIndex)
{
	u32 i;
	u32 uRequestStringSize = 0xff;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_ED;i++)
	{
		ohci_ed_ptr[i] = (ohci_ed_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_ed_t), 32);
	}
	for(i=0;i<NUM_OF_gTD;i++)
	{
		ohci_gtd_ptr[i] = (ohci_gtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_gtd_t), 32);
	}

	ohci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);
	ohci_in_data_ptr = (u8 *)USBHOST_AllocateDataStructure(uRequestStringSize, 4096);

	// . fill ED
	//---------
	ohci_ed_ptr[0]->ed_info.data = 0;
	ohci_ed_ptr[0]->ed_info.b.func_addr = oUsbDevice.m_usDevAddr;
	ohci_ed_ptr[0]->ed_info.b.ep_num = 0;
	ohci_ed_ptr[0]->ed_info.b.direction = 0;
	ohci_ed_ptr[0]->ed_info.b.speed = 0;	// full-speed
	ohci_ed_ptr[0]->ed_info.b.format = 0;	// gTD
	ohci_ed_ptr[0]->ed_info.b.mps = oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	
	ohci_ed_ptr[0]->tail_td_ptr = 0;
	
	ohci_ed_ptr[0]->head_td_ptr = (u32)ohci_gtd_ptr[0];
	
	ohci_ed_ptr[0]->next_ed = 0;

	
	// . fill gTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ohci_setup_data_ptr->bmRequestType = DEVICE_TO_HOST | STANDARD_TYPE | DEVICE_RECIPIENT;
	ohci_setup_data_ptr->bRequest = STANDARD_GET_DESCRIPTOR;
	ohci_setup_data_ptr->wValue_L= ucIndex;
	ohci_setup_data_ptr->wValue_H = STRING_DESCRIPTOR;
	if (ucIndex == 0)
	{
		ohci_setup_data_ptr->wIndex_L = 0;
		ohci_setup_data_ptr->wIndex_H = 0;
	}
	else
	{
		ohci_setup_data_ptr->wIndex_L = oUsbDevice.m_oDesc.oDescStr0.pString[0];
		ohci_setup_data_ptr->wIndex_H = oUsbDevice.m_oDesc.oDescStr0.pString[1];
	}
	ohci_setup_data_ptr->wLength_L = uRequestStringSize;
	ohci_setup_data_ptr->wLength_H = 0;
	
	// gTD[0]
	ohci_gtd_ptr[0]->gtd_info.data = 0;
	ohci_gtd_ptr[0]->gtd_info.b.buffer_rounding = 0;
	ohci_gtd_ptr[0]->gtd_info.b.pid = 0;		// SETUP
	ohci_gtd_ptr[0]->gtd_info.b.data_toggle = 2;	// DATA0
	
	ohci_gtd_ptr[0]->current_buf_ptr = (u32)ohci_setup_data_ptr;

	ohci_gtd_ptr[0]->next_td = (u32)ohci_gtd_ptr[1];

	ohci_gtd_ptr[0]->buffer_end = (u32)ohci_setup_data_ptr + SETUP_DATA_SIZE-1;
	
	// gTD[1]
	ohci_gtd_ptr[1]->gtd_info.data = 0;
	ohci_gtd_ptr[1]->gtd_info.b.buffer_rounding = 1;	// may be smaller than the defined buffer
	ohci_gtd_ptr[1]->gtd_info.b.pid = 2;		// IN
	ohci_gtd_ptr[1]->gtd_info.b.data_toggle = 3;	// DATA1
	
	ohci_gtd_ptr[1]->current_buf_ptr = (u32)ohci_in_data_ptr;

	ohci_gtd_ptr[1]->next_td = (u32)ohci_gtd_ptr[2];

	ohci_gtd_ptr[1]->buffer_end = (u32)ohci_in_data_ptr + uRequestStringSize-1;
	
	// gTD[2]
	ohci_gtd_ptr[2]->gtd_info.data = 0;
	ohci_gtd_ptr[2]->gtd_info.b.buffer_rounding = 1;	// may be smaller than the defined buffer
	ohci_gtd_ptr[2]->gtd_info.b.pid = 1;		// OUT
	ohci_gtd_ptr[2]->gtd_info.b.data_toggle = 3;	// DATA1
	
	ohci_gtd_ptr[2]->current_buf_ptr = 0;

	ohci_gtd_ptr[2]->next_td = 0;

	ohci_gtd_ptr[2]->buffer_end = 0;


	USBHOST_OhciEnableList(1, (u32)ohci_ed_ptr[0], (u32)ohci_gtd_ptr[2]);
}


//////////
// Function Name : USBHOST_EhciSetConfiguration
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_EhciSetConfiguration(void)
{
	u32 i;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_QH;i++)
	{
		ehci_qh_ptr[i] = (ehci_qh_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qh_t), 32);
	}
	for(i=0;i<NUM_OF_qTD;i++)
	{
		ehci_qtd_ptr[i] = (ehci_qtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qtd_t), 32);
	}

	ehci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);

	// . fill QH
	//---------
	// >> DWORD0
	ehci_qh_ptr[0]->next_qh_ptr.data = 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.t= 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.typ = 1;	// queue head
	ehci_qh_ptr[0]->next_qh_ptr.b.qhlp = (u32)ehci_qh_ptr[0]>>5;
	// >> DWORD1
	ehci_qh_ptr[0]->info1.data = 0;
	ehci_qh_ptr[0]->info1.b.dev_addr = oUsbDevice.m_usDevAddr;
	ehci_qh_ptr[0]->info1.b.i= 0;
	ehci_qh_ptr[0]->info1.b.ep_num = 0;
	ehci_qh_ptr[0]->info1.b.eps = USB_EP_HIGH_SPEED;
	ehci_qh_ptr[0]->info1.b.dtc = 1;
	ehci_qh_ptr[0]->info1.b.h = 1;	// ?khs
	ehci_qh_ptr[0]->info1.b.max_pkt_sz= oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	ehci_qh_ptr[0]->info1.b.c = 0;
	ehci_qh_ptr[0]->info1.b.rl = 0;
	// >> DWORD2
	ehci_qh_ptr[0]->info2.data = 0;
	ehci_qh_ptr[0]->info2.b.mult = ONE_TR_uFRAME;
	// >> DWORD3
	ehci_qh_ptr[0]->current_qtd_ptr = 0;
	// >> DWORD4
	ehci_qh_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[0];
	// >> DWORD5
	ehci_qh_ptr[0]->alt_next_qtd_ptr = 1;	// invalid
	// >> DWORD6~11
	ehci_qh_ptr[0]->qtd_token = 0;
	ehci_qh_ptr[0]->buf_ptr0 = 0;
	ehci_qh_ptr[0]->buf_ptr1 = 0;
	ehci_qh_ptr[0]->buf_ptr2 = 0;
	ehci_qh_ptr[0]->buf_ptr3 = 0;
	ehci_qh_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ehci_setup_data_ptr->bmRequestType = HOST_TO_DEVICE| STANDARD_TYPE | DEVICE_RECIPIENT;
	ehci_setup_data_ptr->bRequest = STANDARD_SET_CONFIGURATION;
	ehci_setup_data_ptr->wValue_L= oUsbDevice.m_oDesc.oDescConfig.bConfigurationValue;
	ehci_setup_data_ptr->wValue_H = 0;
	ehci_setup_data_ptr->wIndex_L = 0;
	ehci_setup_data_ptr->wIndex_H = 0;
	ehci_setup_data_ptr->wLength_L = 0;
	ehci_setup_data_ptr->wLength_H = 0;
	// qTD[0]
	ehci_qtd_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[1];
	ehci_qtd_ptr[0]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[0]->qtd_token.data = 0;
	ehci_qtd_ptr[0]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[0]->qtd_token.b.pid = USB_SETUP_TOKEN;
	ehci_qtd_ptr[0]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[0]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[0]->qtd_token.b.ioc = 0;
	ehci_qtd_ptr[0]->qtd_token.b.xfer_total_bytes = SETUP_DATA_SIZE;
	ehci_qtd_ptr[0]->qtd_token.b.data_toggle = 0;
	ehci_qtd_ptr[0]->buf_ptr0 = (u32)ehci_setup_data_ptr;		// should wite the addr of setup packet data
	ehci_qtd_ptr[0]->buf_ptr1 = 0;
	ehci_qtd_ptr[0]->buf_ptr2 = 0;
	ehci_qtd_ptr[0]->buf_ptr3 = 0;
	ehci_qtd_ptr[0]->buf_ptr4 = 0;
	
	// . fill qTD[1] for IN data
	//--------------------------
	ehci_qtd_ptr[1]->next_qtd_ptr = 1;
	ehci_qtd_ptr[1]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[1]->qtd_token.data = 0;
	ehci_qtd_ptr[1]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[1]->qtd_token.b.pid = USB_IN_TOKEN;
	ehci_qtd_ptr[1]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[1]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[1]->qtd_token.b.ioc = 1;
	ehci_qtd_ptr[1]->qtd_token.b.xfer_total_bytes = 0;
	ehci_qtd_ptr[1]->qtd_token.b.data_toggle = 1;
	ehci_qtd_ptr[1]->buf_ptr0 = 0;		// should wite the addr of IN data packet
	ehci_qtd_ptr[1]->buf_ptr1 = 0;
	ehci_qtd_ptr[1]->buf_ptr2 = 0;
	ehci_qtd_ptr[1]->buf_ptr3 = 0;
	ehci_qtd_ptr[1]->buf_ptr4 = 0;

	// . update Async Schedule & enable Async Schedule
	//--------------------------------------------
	uhOutp32(rUH_EHCI_OP_ASYNCLISTADDR, (u32)ehci_qh_ptr[0]&0xffffffe0);

	USBHOST_EhciEnableAsyncSched();
	
}


//////////
// Function Name : USBHOST_OhciSetConfiguration
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_OhciSetConfiguration(void)
{
	u32 i;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_ED;i++)
	{
		ohci_ed_ptr[i] = (ohci_ed_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_ed_t), 32);
	}
	for(i=0;i<NUM_OF_gTD;i++)
	{
		ohci_gtd_ptr[i] = (ohci_gtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_gtd_t), 32);
	}

	ohci_setup_data_ptr = (DEVICE_REQUEST_PTR)USBHOST_AllocateDataStructure(SETUP_DATA_SIZE, 4096);

	// . fill ED
	//---------
	ohci_ed_ptr[0]->ed_info.data = 0;
	ohci_ed_ptr[0]->ed_info.b.func_addr = oUsbDevice.m_usDevAddr;
	ohci_ed_ptr[0]->ed_info.b.ep_num = 0;
	ohci_ed_ptr[0]->ed_info.b.direction = 0;
	ohci_ed_ptr[0]->ed_info.b.speed = 0;	// full-speed
	ohci_ed_ptr[0]->ed_info.b.format = 0;	// gTD
	ohci_ed_ptr[0]->ed_info.b.mps = oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;
	
	ohci_ed_ptr[0]->tail_td_ptr = 0;
	
	ohci_ed_ptr[0]->head_td_ptr = (u32)ohci_gtd_ptr[0];
	
	ohci_ed_ptr[0]->next_ed = 0;
	
	
	// . fill gTD[0] for SETUP Packet
	//--------------------------
	// setup data
	ohci_setup_data_ptr->bmRequestType = HOST_TO_DEVICE| STANDARD_TYPE | DEVICE_RECIPIENT;
	ohci_setup_data_ptr->bRequest = STANDARD_SET_CONFIGURATION;
	ohci_setup_data_ptr->wValue_L= oUsbDevice.m_oDesc.oDescConfig.bConfigurationValue;
	ohci_setup_data_ptr->wValue_H = 0;
	ohci_setup_data_ptr->wIndex_L = 0;
	ohci_setup_data_ptr->wIndex_H = 0;
	ohci_setup_data_ptr->wLength_L = 0;
	ohci_setup_data_ptr->wLength_H = 0;
	
	// gTD[0]
	ohci_gtd_ptr[0]->gtd_info.data = 0;
	ohci_gtd_ptr[0]->gtd_info.b.buffer_rounding = 0;
	ohci_gtd_ptr[0]->gtd_info.b.pid = 0;		// SETUP
	ohci_gtd_ptr[0]->gtd_info.b.data_toggle = 2;	// DATA0
	
	ohci_gtd_ptr[0]->current_buf_ptr = (u32)ohci_setup_data_ptr;

	ohci_gtd_ptr[0]->next_td = (u32)ohci_gtd_ptr[1];

	ohci_gtd_ptr[0]->buffer_end = (u32)ohci_setup_data_ptr + SETUP_DATA_SIZE-1;
	
	// gTD[1]
	ohci_gtd_ptr[1]->gtd_info.data = 0;
	ohci_gtd_ptr[1]->gtd_info.b.buffer_rounding = 1;	// may be smaller than the defined buffer
	ohci_gtd_ptr[1]->gtd_info.b.pid = 2;		// IN
	ohci_gtd_ptr[1]->gtd_info.b.data_toggle = 3;	// DATA1
	
	ohci_gtd_ptr[1]->current_buf_ptr = 0;

	ohci_gtd_ptr[1]->next_td = 0;

	ohci_gtd_ptr[1]->buffer_end = 0;


	USBHOST_OhciEnableList(1, (u32)ohci_ed_ptr[0], (u32)ohci_gtd_ptr[1]);
	
}


//////////
// Function Name : USBHOST_EhciEnumerate
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_EhciEnumerate(void)
{
	USBHOST_EhciWaitEhciStateChange(EHCI_STATE_READY);

	// . Get Device Descriptor with Addr 0
	//-------------------------------
	oUsbHost.m_eEhciState = EHCI_STATE_GD_DEV_BEFORE;
	USBHOST_EhciGetDeviceDescriptor(0);
	USBHOST_WaitUsbTrCompetion(true);

	// . Set Address
	//--------------
	oUsbHost.m_eEhciState = EHCI_STATE_SET_ADDR;
	oUsbHost.m_usDevAddr = 1;
	USBHOST_EhciSetAddress(oUsbHost.m_usDevAddr);
	USBHOST_WaitUsbTrCompetion(true);

	// . Get Device Descriptor with setting Addr
	//-----------------------------------
	oUsbHost.m_eEhciState = EHCI_STATE_GD_DEV_AFTER;
	USBHOST_EhciGetDeviceDescriptor(oUsbHost.m_usDevAddr);
	USBHOST_WaitUsbTrCompetion(true);

	// . Get Configuration Descriptor
	//---------------------------
	oUsbHost.m_eEhciState = EHCI_STATE_GD_CFG;
	USBHOST_EhciGetConfigDescriptor();
	USBHOST_WaitUsbTrCompetion(true);

	// . (optional) Get String Descriptor, LANGID
	//------------------------------------
	oUsbHost.m_eEhciState = EHCI_STATE_GD_STR0;
	USBHOST_EhciGetStringDescriptor(0);
	USBHOST_WaitUsbTrCompetion(true);

	// . (optional) Get String Descriptor, Manufacturer
	//-----------------------------------------
	if (oUsbDevice.m_oDesc.oDescDevice.iManufacturer != 0)
	{
		switch (oUsbDevice.m_oDesc.oDescDevice.iManufacturer)
		{
			case 1:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR1;
				break;

			case 2:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR2;
				break;

			case 3:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR3;
				break;

			default:
				UART_Printf("More String Descriptor is needed\n");
				Assert(0);
				break;
		}
		USBHOST_EhciGetStringDescriptor(oUsbDevice.m_oDesc.oDescDevice.iManufacturer);
		USBHOST_WaitUsbTrCompetion(true);
	}

	// . (optional) Get String Descriptor, Product Name
	//------------------------------------------
	if (oUsbDevice.m_oDesc.oDescDevice.iProduct != 0)
	{
		switch (oUsbDevice.m_oDesc.oDescDevice.iProduct)
		{
			case 1:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR1;
				break;

			case 2:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR2;
				break;

			case 3:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR3;
				break;

			default:
				UART_Printf("More String Descriptor is needed\n");
				Assert(0);
				break;
		}
		USBHOST_EhciGetStringDescriptor(oUsbDevice.m_oDesc.oDescDevice.iProduct);
		USBHOST_WaitUsbTrCompetion(true);
	}

	// . (optional) Get String Descriptor, Serial Number
	//------------------------------------------
	if (oUsbDevice.m_oDesc.oDescDevice.iSerialNumber!= 0)
	{
		switch (oUsbDevice.m_oDesc.oDescDevice.iSerialNumber)
		{
			case 1:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR1;
				break;

			case 2:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR2;
				break;

			case 3:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR3;
				break;

			default:
				UART_Printf("More String Descriptor is needed\n");
				Assert(0);
				break;
		}
		USBHOST_EhciGetStringDescriptor(oUsbDevice.m_oDesc.oDescDevice.iSerialNumber);
		USBHOST_WaitUsbTrCompetion(true);
	}


	// . Set Configuration
	//--------------------
	oUsbHost.m_eEhciState = EHCI_STATE_SET_CFG;
	USBHOST_EhciSetConfiguration();
	USBHOST_WaitUsbTrCompetion(true);
	
}

//////////
// Function Name : USBHOST_OhciEnumerate
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_OhciEnumerate(void)
{
	USBHOST_OhciWaitEhciStateChange(OHCI_STATE_READY);

	// . Get Device Descriptor with Addr 0
	//-------------------------------
	oUsbHost.m_eOhciState = OHCI_STATE_GD_DEV_BEFORE;
	USBHOST_OhciGetDeviceDescriptor(0);
	USBHOST_WaitUsbTrCompetion(true);

	// . Set Address
	//--------------
	oUsbHost.m_eOhciState = OHCI_STATE_SET_ADDR;
	oUsbHost.m_usDevAddr = 1;
	USBHOST_OhciSetAddress(oUsbHost.m_usDevAddr);
	USBHOST_WaitUsbTrCompetion(true);
	
	// . Get Device Descriptor with setting Addr
	//-----------------------------------
	oUsbHost.m_eOhciState = OHCI_STATE_GD_DEV_AFTER;
	USBHOST_OhciGetDeviceDescriptor(oUsbHost.m_usDevAddr);
	USBHOST_WaitUsbTrCompetion(true);

	// . Get Configuration Descriptor
	//---------------------------
	oUsbHost.m_eOhciState = OHCI_STATE_GD_CFG;
	USBHOST_OhciGetConfigDescriptor();
	USBHOST_WaitUsbTrCompetion(true);

	// . (optional) Get String Descriptor, LANGID
	//------------------------------------
	oUsbHost.m_eOhciState = OHCI_STATE_GD_STR0;
	USBHOST_OhciGetStringDescriptor(0);
	USBHOST_WaitUsbTrCompetion(true);

	// . (optional) Get String Descriptor, Manufacturer
	//-----------------------------------------
	if (oUsbDevice.m_oDesc.oDescDevice.iManufacturer != 0)
	{
		switch (oUsbDevice.m_oDesc.oDescDevice.iManufacturer)
		{
			case 1:
				oUsbHost.m_eOhciState = OHCI_STATE_GD_STR1;
				break;

			case 2:
				oUsbHost.m_eOhciState = OHCI_STATE_GD_STR2;
				break;

			case 3:
				oUsbHost.m_eOhciState = OHCI_STATE_GD_STR3;
				break;

			default:
				UART_Printf("More String Descriptor is needed\n");
				Assert(0);
				break;
		}
		USBHOST_OhciGetStringDescriptor(oUsbDevice.m_oDesc.oDescDevice.iManufacturer);
		USBHOST_WaitUsbTrCompetion(true);
	}

	// . (optional) Get String Descriptor, Product Name
	//------------------------------------------
	if (oUsbDevice.m_oDesc.oDescDevice.iProduct != 0)
	{
		switch (oUsbDevice.m_oDesc.oDescDevice.iProduct)
		{
			case 1:
				oUsbHost.m_eOhciState = OHCI_STATE_GD_STR1;
				break;

			case 2:
				oUsbHost.m_eOhciState = OHCI_STATE_GD_STR2;
				break;

			case 3:
				oUsbHost.m_eOhciState = OHCI_STATE_GD_STR3;
				break;

			default:
				UART_Printf("More String Descriptor is needed\n");
				Assert(0);
				break;
		}
		USBHOST_OhciGetStringDescriptor(oUsbDevice.m_oDesc.oDescDevice.iProduct);
		USBHOST_WaitUsbTrCompetion(true);
	}

	// . (optional) Get String Descriptor, Serial Number
	//------------------------------------------
	if (oUsbDevice.m_oDesc.oDescDevice.iSerialNumber!= 0)
	{
		switch (oUsbDevice.m_oDesc.oDescDevice.iSerialNumber)
		{
			case 1:
				oUsbHost.m_eOhciState = OHCI_STATE_GD_STR1;
				break;

			case 2:
				oUsbHost.m_eOhciState = OHCI_STATE_GD_STR2;
				break;

			case 3:
				oUsbHost.m_eOhciState = OHCI_STATE_GD_STR3;
				break;

			default:
				UART_Printf("More String Descriptor is needed\n");
				Assert(0);
				break;
		}
		USBHOST_OhciGetStringDescriptor(oUsbDevice.m_oDesc.oDescDevice.iSerialNumber);
		USBHOST_WaitUsbTrCompetion(true);
	}

	// . Set Configuration
	//--------------------
	oUsbHost.m_eOhciState = OHCI_STATE_SET_CFG;
	USBHOST_OhciSetConfiguration();
	USBHOST_WaitUsbTrCompetion(true);
}



//////////
// Function Name : USBHOST_AllocateDataStructure
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
u32 USBHOST_AllocateDataStructure(u32 uSize, u32 uAlign)
{
	u32 uMemAddr;
	
	uMemAddr = UsbHostAlign(g_uUsbHostDataMem, uAlign);
	g_uUsbHostDataMem = uMemAddr+uSize;

	return uMemAddr;
}


//////////
// Function Name : USBHOST_CheckDeviceEnumerated
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
u8 USBHOST_CheckDeviceEnumerated(void)
{

	if ((oUsbHost.m_eEhciState == EHCI_STATE_CONFIGURED) || (oUsbHost.m_eOhciState == OHCI_STATE_CONFIGURED))
		return true;

	else
		return false;
}


//////////
// Function Name : USBHOST_CheckSamsungAP
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
u8 USBHOST_CheckSamsungAP(void)
{	
	if (oUsbDevice.m_oDesc.oDescDevice.idVendorL==0xE8 && oUsbDevice.m_oDesc.oDescDevice.idVendorH==0x04 &&
		oUsbDevice.m_oDesc.oDescDevice.idProductL==0x34 && oUsbDevice.m_oDesc.oDescDevice.idProductH==0x12)
		return true;
		
	else
		return false;
}


//////////
// Function Name : USBHOST_ParseDeviceInform
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_ParseDeviceInform(void)
{
	u8 i, ucNumEp = oUsbDevice.m_oDesc.oDescInterface.bNumEndpoints;
	USB_ENDPOINT_DESCRIPTOR *p_oDescEp;
	
	oUsbDevice.m_ucControlEPMaxPktSize = oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0;

	for(i=0;i<ucNumEp;i++)
	{	
		// . Endpoint Descriptor
		//-------------------------	
		p_oDescEp = &oUsbDevice.m_oDesc.oDescEp0 + i;

		if ((p_oDescEp->bmAttributes &0x3) == EP_ATTR_BULK)
		{
			oUsbDevice.m_usBulkEPMaxPktSize = p_oDescEp->wMaxPacketSizeH<<8|p_oDescEp->wMaxPacketSizeL;
			
			if (p_oDescEp->bEndpointAddress & 0x80)	// in
			{
				oUsbDevice.m_ucBulkInEpNum= p_oDescEp->bEndpointAddress & 0xff;
			}
			else	// out
			{
				oUsbDevice.m_ucBulkOutEpNum = p_oDescEp->bEndpointAddress & 0xff;
			}
		}
		
	}	
	
}


//////////
// Function Name : USBHOST_DisplayDeviceInform
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_DisplayDeviceInform(void)
{
	u8 i, ucNumEp = oUsbDevice.m_oDesc.oDescInterface.bNumEndpoints;
	USB_ENDPOINT_DESCRIPTOR *p_oDescEp;
	
	UART_Printf("\n=======================================================\n");
	
	// . Device Descriptor
	//------------------
	UART_Printf("*Device Descriptor\n");
	UART_Printf(" >>bcdUSB:                    0x%04x\n",oUsbDevice.m_oDesc.oDescDevice.bcdUSBH<<8|oUsbDevice.m_oDesc.oDescDevice.bcdUSBL);
	UART_Printf(" >>bDeviceClass:                0x%02x\n",oUsbDevice.m_oDesc.oDescDevice.bDeviceClass);
	UART_Printf(" >>bDeviceSubClass:             0x%02x\n",oUsbDevice.m_oDesc.oDescDevice.bDeviceSubClass);
	UART_Printf(" >>bDeviceProtocol:             0x%02x\n",oUsbDevice.m_oDesc.oDescDevice.bDeviceProtocol);
	UART_Printf(" >>bMaxPacketSize0:             0x%02x\n",oUsbDevice.m_oDesc.oDescDevice.bMaxPacketSize0);	
	UART_Printf(" >>idVendor:                  0x%04x\n",oUsbDevice.m_oDesc.oDescDevice.idVendorH<<8|oUsbDevice.m_oDesc.oDescDevice.idVendorL);
	UART_Printf(" >>idProduct:                 0x%04x\n",oUsbDevice.m_oDesc.oDescDevice.idProductH<<8|oUsbDevice.m_oDesc.oDescDevice.idProductL);
	UART_Printf(" >>bcdDevice:                 0x%04x\n",oUsbDevice.m_oDesc.oDescDevice.bcdDeviceH<<8|oUsbDevice.m_oDesc.oDescDevice.bcdDeviceL);
	
	UART_Printf(" >>iManufacturer:               0x%02x",oUsbDevice.m_oDesc.oDescDevice.iManufacturer);
	if (oUsbDevice.m_oDesc.oDescDevice.iManufacturer != 0)
	{
		u8 i;
		USB_STRING_DESCRIPTOR *p_oDescManufacturer;
		
		p_oDescManufacturer= &oUsbDevice.m_oDesc.oDescStr0+oUsbDevice.m_oDesc.oDescDevice.iManufacturer;

		UART_Printf("  \"");

		for(i=0;i<(p_oDescManufacturer->bLength-2);i+=2)
		{
			UART_Printf("%c",p_oDescManufacturer->pString[i]);
		}
		UART_Printf("\"");
	}
	UART_Printf("\n");
	
	UART_Printf(" >>iProduct:                    0x%02x",oUsbDevice.m_oDesc.oDescDevice.iProduct);
	if (oUsbDevice.m_oDesc.oDescDevice.iProduct!= 0)
	{
		u8 i;
		USB_STRING_DESCRIPTOR *p_oDescProduct;
		
		p_oDescProduct= &oUsbDevice.m_oDesc.oDescStr0+oUsbDevice.m_oDesc.oDescDevice.iProduct;

		UART_Printf("  \"");

		for(i=0;i<(p_oDescProduct->bLength-2);i+=2)
		{
			UART_Printf("%c",p_oDescProduct->pString[i]);
		}
		UART_Printf("\"");
	}
	UART_Printf("\n");
	
	UART_Printf(" >>iSerialNumber:               0x%02x",oUsbDevice.m_oDesc.oDescDevice.iSerialNumber);
	if (oUsbDevice.m_oDesc.oDescDevice.iSerialNumber!= 0)
	{
		u8 i;
		USB_STRING_DESCRIPTOR *p_oDescSerial;
		
		p_oDescSerial= &oUsbDevice.m_oDesc.oDescStr0+oUsbDevice.m_oDesc.oDescDevice.iSerialNumber;

		UART_Printf("  \"");

		for(i=0;i<(p_oDescSerial->bLength-2);i+=2)
		{
			UART_Printf("%c",p_oDescSerial->pString[i]);
		}
		UART_Printf("\"");
	}
	UART_Printf("\n");
	
	UART_Printf(" >>bNumConfigurations:          0x%02x\n",oUsbDevice.m_oDesc.oDescDevice.bNumConfigurations);

	
	// . Configuration Descriptor
	//-------------------------	
	UART_Printf("-------------------------------------------------------\n");
	UART_Printf("*Configuration Descriptor\n");
	UART_Printf(" >>wTotalLength:              0x%04x\n", oUsbDevice.m_oDesc.oDescConfig.wTotalLengthH<<8|oUsbDevice.m_oDesc.oDescConfig.wTotalLengthL);
	UART_Printf(" >>bNumInterfaces:              0x%02x\n", oUsbDevice.m_oDesc.oDescConfig.bNumInterfaces);
	UART_Printf(" >>bConfigurationValue:         0x%02x\n", oUsbDevice.m_oDesc.oDescConfig.bConfigurationValue);
	UART_Printf(" >>iConfiguration:              0x%02x\n", oUsbDevice.m_oDesc.oDescConfig.iConfiguration);
	UART_Printf(" >>bmAttributes:                0x%02x\n", oUsbDevice.m_oDesc.oDescConfig.bmAttributes);
	UART_Printf(" >>MaxPower:                    0x%02x\n", oUsbDevice.m_oDesc.oDescConfig.maxPower);

	
	// . Interface Descriptor
	//-------------------------	
	UART_Printf("-------------------------------------------------------\n");
	UART_Printf("*Interface Descriptor\n");
	UART_Printf(" >>bInterfaceNumber:            0x%02x\n", oUsbDevice.m_oDesc.oDescInterface.bInterfaceNumber);
	UART_Printf(" >>bAlternateSetting:           0x%02x\n", oUsbDevice.m_oDesc.oDescInterface.bAlternateSetting);
	UART_Printf(" >>bNumEndpoints:               0x%02x\n", oUsbDevice.m_oDesc.oDescInterface.bNumEndpoints);
	UART_Printf(" >>bInterfaceClass:             0x%02x\n", oUsbDevice.m_oDesc.oDescInterface.bInterfaceClass);
	UART_Printf(" >>bInterfaceSubClass:          0x%02x\n", oUsbDevice.m_oDesc.oDescInterface.bInterfaceSubClass);
	UART_Printf(" >>bInterfaceProtocol:          0x%02x\n", oUsbDevice.m_oDesc.oDescInterface.bInterfaceProtocol);
	UART_Printf(" >>iInterface:                  0x%02x\n", oUsbDevice.m_oDesc.oDescInterface.iInterface);

	for(i=0;i<ucNumEp;i++)
	{	
		// . Endpoint Descriptor
		//-------------------------	
		p_oDescEp = &oUsbDevice.m_oDesc.oDescEp0 + i;
		UART_Printf("-------------------------------------------------------\n");
		UART_Printf("*Endpoint Descriptor\n");
		
		UART_Printf(" >>bEndpointAddress:            0x%02x  ", p_oDescEp->bEndpointAddress);
		if (p_oDescEp->bEndpointAddress & 0x80)
		{
			UART_Printf("IN\n");
		}
		else
		{
			UART_Printf("OUT\n");
		}

		UART_Printf(" >>Transfer Type:               ");
		switch(p_oDescEp->bmAttributes &0x3)
		{
			case EP_ATTR_CONTROL:
				UART_Printf("Control\n");
				break;
				
			case EP_ATTR_ISOCHRONOUS:
				UART_Printf("Isochronous\n");
				break;
				
			case EP_ATTR_BULK:
				UART_Printf("Bulk\n");
				break;
				
			case EP_ATTR_INTERRUPT:
				UART_Printf("Interrupt\n");
				break;				
		}

		UART_Printf(" >>wMaxPacketSize:            0x%04x (%d)\n", p_oDescEp->wMaxPacketSizeH<<8|p_oDescEp->wMaxPacketSizeL, p_oDescEp->wMaxPacketSizeH<<8|p_oDescEp->wMaxPacketSizeL);
		UART_Printf(" >>bInterval:                   0x%02x\n", p_oDescEp->bInterval);
		
	}
	
	UART_Printf("=======================================================\n");
	
	
}


//////////
// Function Name : USBHOST_EhciXferBulkOutData
// Function Desctiption : This function 
// Input : uAddr, uSize
// Output : NONE
// Version :
void USBHOST_EhciXferBulkOutData(u32 uAddr, u32 uSize)
{
	u32 i, j=0;
	u32 uTotalLength = uSize+10;	// uSize + addr(4) + size(4) + cs(2)
	u16 ucNumOfqTd, ucTemp;
	u8 *pTestAddr;
	u16 usCS = 0;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_QH;i++)
	{
		ehci_qh_ptr[i] = (ehci_qh_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qh_t), 32);
	}

	ucNumOfqTd = uTotalLength/QTD_BUF_MAX_SIZE;
	if ((uTotalLength%QTD_BUF_MAX_SIZE)!=0)
	{
		ucNumOfqTd++;
	}
	for(i=0;i<ucNumOfqTd;i++)
	{
		ehci_qtd_ptr[i] = (ehci_qtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qtd_t), 32);
	}

	ehci_out_data_ptr = (u8 *)USBHOST_AllocateDataStructure(uTotalLength, 4096);
	
	
	// . fill test data
	//-------------
	*(u32 *)(ehci_out_data_ptr+0) = uAddr;
	*(u32 *)(ehci_out_data_ptr+4) = uTotalLength;
	pTestAddr = (ehci_out_data_ptr + 8);

	for (i=0;i<uSize;i++)
	{
		if ((i%oUsbDevice.m_usBulkEPMaxPktSize) == 0)
		{
			j++;
			*(pTestAddr+i) = j&0xff;
			usCS += j&0xff;
		}
		else
		{
			*(pTestAddr+i) = i&0xff;
			usCS += i&0xff;
		}
	}

	*(u8 *)(pTestAddr + uSize) = (u8)usCS;	
	*(u8 *)(pTestAddr + uSize+1) = (u8)(usCS>>8);

	// . fill QH
	//---------
	// >> DWORD0
	ehci_qh_ptr[0]->next_qh_ptr.data = 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.t= 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.typ = 1;	// queue head
	ehci_qh_ptr[0]->next_qh_ptr.b.qhlp = (u32)ehci_qh_ptr[0]>>5;
	// >> DWORD1
	ehci_qh_ptr[0]->info1.data = 0;
	ehci_qh_ptr[0]->info1.b.dev_addr = oUsbDevice.m_usDevAddr;
	ehci_qh_ptr[0]->info1.b.i= 0;
	ehci_qh_ptr[0]->info1.b.ep_num = oUsbDevice.m_ucBulkOutEpNum;
	ehci_qh_ptr[0]->info1.b.eps = USB_EP_HIGH_SPEED;
	ehci_qh_ptr[0]->info1.b.dtc = 1;
	ehci_qh_ptr[0]->info1.b.h = 1;	// ?khs
	ehci_qh_ptr[0]->info1.b.max_pkt_sz= oUsbDevice.m_usBulkEPMaxPktSize;
	ehci_qh_ptr[0]->info1.b.c = 0;
	ehci_qh_ptr[0]->info1.b.rl = 0;
	// >> DWORD2
	ehci_qh_ptr[0]->info2.data = 0;
	ehci_qh_ptr[0]->info2.b.mult = ONE_TR_uFRAME;
	// >> DWORD3
	ehci_qh_ptr[0]->current_qtd_ptr = 0;
	// >> DWORD4
	ehci_qh_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[0];
	// >> DWORD5
	ehci_qh_ptr[0]->alt_next_qtd_ptr = 1;	// invalid
	// >> DWORD6~11
	ehci_qh_ptr[0]->qtd_token = 0;
	ehci_qh_ptr[0]->buf_ptr0 = 0;
	ehci_qh_ptr[0]->buf_ptr1 = 0;
	ehci_qh_ptr[0]->buf_ptr2 = 0;
	ehci_qh_ptr[0]->buf_ptr3 = 0;
	ehci_qh_ptr[0]->buf_ptr4 = 0;

	
	// . fill qTD[x] except last qTD
	//------------------------
	for(i=0;i<(ucNumOfqTd-1);i++)
	{
		ehci_qtd_ptr[i]->next_qtd_ptr = (u32)ehci_qtd_ptr[i+1];
		ehci_qtd_ptr[i]->alt_next_qtd_prt = 1;	// invalid
		ehci_qtd_ptr[i]->qtd_token.data = 0;
		ehci_qtd_ptr[i]->qtd_token.b.sts = USB_qTD_ACTIVE;
		ehci_qtd_ptr[i]->qtd_token.b.pid = USB_OUT_TOKEN;
		ehci_qtd_ptr[i]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
		ehci_qtd_ptr[i]->qtd_token.b.c_page = 0;
		ehci_qtd_ptr[i]->qtd_token.b.ioc = 0;
		ehci_qtd_ptr[i]->qtd_token.b.xfer_total_bytes = QTD_BUF_MAX_SIZE;	// 5*4K
		ehci_qtd_ptr[i]->qtd_token.b.data_toggle = ucBulkOutDt;
		ehci_qtd_ptr[i]->buf_ptr0 = (u32)(ehci_out_data_ptr+QTD_BUF_MAX_SIZE*i+0);
		ehci_qtd_ptr[i]->buf_ptr1 = (u32)(ehci_out_data_ptr+QTD_BUF_MAX_SIZE*i+QTD_PAGE_MAX_SIZE*1);
		ehci_qtd_ptr[i]->buf_ptr2 = (u32)(ehci_out_data_ptr+QTD_BUF_MAX_SIZE*i+QTD_PAGE_MAX_SIZE*2);
		ehci_qtd_ptr[i]->buf_ptr3 = (u32)(ehci_out_data_ptr+QTD_BUF_MAX_SIZE*i+QTD_PAGE_MAX_SIZE*3);
		ehci_qtd_ptr[i]->buf_ptr4 = (u32)(ehci_out_data_ptr+QTD_BUF_MAX_SIZE*i+QTD_PAGE_MAX_SIZE*4);
	}
	
	// . fill last qTD
	//------------
	ehci_qtd_ptr[ucNumOfqTd-1]->next_qtd_ptr =  1;	// invalid
	ehci_qtd_ptr[ucNumOfqTd-1]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.data = 0;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.pid = USB_OUT_TOKEN;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.ioc = 1;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.xfer_total_bytes = uTotalLength-QTD_BUF_MAX_SIZE*(ucNumOfqTd-1);
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.data_toggle = ucBulkOutDt;

	ucTemp = (uTotalLength-QTD_BUF_MAX_SIZE*(ucNumOfqTd-1))/QTD_PAGE_MAX_SIZE;	// calculate buffer count
	if (((uTotalLength-QTD_BUF_MAX_SIZE*(ucNumOfqTd-1))%QTD_PAGE_MAX_SIZE)!=0)
	{
		ucTemp++;
	}
	for(i=0;i<ucTemp;i++)	// fill the buffer pointer
	{
		*(u32 *)(&(ehci_qtd_ptr[ucNumOfqTd-1]->buf_ptr0) + i) = (u32)(ehci_out_data_ptr+QTD_BUF_MAX_SIZE*(ucNumOfqTd-1)+QTD_PAGE_MAX_SIZE*i);
	}
	for(i=ucTemp;i<5;i++)
	{
		*(u32 *)(&(ehci_qtd_ptr[ucNumOfqTd-1]->buf_ptr0) + i) = 0;
	}

	// . update Async Schedule & enable Async Schedule
	//--------------------------------------------	
	uhOutp32(rUH_EHCI_OP_ASYNCLISTADDR, (u32)ehci_qh_ptr[0]&0xffffffe0);

	USBHOST_EhciEnableAsyncSched();
	
}

//////////
// Function Name : USBHOST_OhciXferBulkOutData
// Function Desctiption : This function 
// Input : uAddr, uSize
// Output : NONE
// Version :
void USBHOST_OhciXferBulkOutData(u32 uAddr, u32 uSize)
{
	u32 i, j=0;
	u32 uTotalLength = uSize+10;	// uSize + addr(4) + size(4) + cs(2)
	u16 ucNumOfgTd, ucTemp;
	u8 *pTestAddr;
	u16 usCS = 0;
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_ED;i++)
	{
		ohci_ed_ptr[i] = (ohci_ed_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_ed_t), 32);
	}

	ucNumOfgTd = uTotalLength/GTD_BUF_MAX_SIZE;
	if ((uTotalLength%GTD_BUF_MAX_SIZE)!=0)
	{
		ucNumOfgTd++;
	}
	
	for(i=0;i<ucNumOfgTd;i++)
	{
		ohci_gtd_ptr[i] = (ohci_gtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_gtd_t), 32);
	}

	ohci_out_data_ptr = (u8 *)USBHOST_AllocateDataStructure(uTotalLength, 4096);
	
	// . fill test data
	//-------------
	*(u32 *)(ohci_out_data_ptr+0) = uAddr;
	*(u32 *)(ohci_out_data_ptr+4) = uTotalLength;
	pTestAddr = (ohci_out_data_ptr + 8);

	for (i=0;i<uSize;i++)
	{
		if ((i%oUsbDevice.m_usBulkEPMaxPktSize) == 0)
		{
			j++;
			*(pTestAddr+i) = j&0xff;
			usCS += j&0xff;
		}
		else
		{
			*(pTestAddr+i) = i&0xff;
			usCS += i&0xff;
		}
	}

	*(u8 *)(pTestAddr + uSize) = (u8)usCS;	
	*(u8 *)(pTestAddr + uSize+1) = (u8)(usCS>>8);
	

	// . fill ED
	//---------
	ohci_ed_ptr[0]->ed_info.data = 0;
	ohci_ed_ptr[0]->ed_info.b.func_addr = oUsbDevice.m_usDevAddr;
	ohci_ed_ptr[0]->ed_info.b.ep_num = oUsbDevice.m_ucBulkOutEpNum;
	ohci_ed_ptr[0]->ed_info.b.direction = 0;
	ohci_ed_ptr[0]->ed_info.b.speed = 0;	// full-speed
	ohci_ed_ptr[0]->ed_info.b.format = 0;	// gTD
	ohci_ed_ptr[0]->ed_info.b.mps = oUsbDevice.m_usBulkEPMaxPktSize;
	
	ohci_ed_ptr[0]->tail_td_ptr = 0;
	
	ohci_ed_ptr[0]->head_td_ptr = (u32)ohci_gtd_ptr[0];
	
	ohci_ed_ptr[0]->next_ed = 0;
	

	// . fill gTD[x] except last gTD
	//------------------------
	for(i=0;i<(ucNumOfgTd-1);i++)
	{
		ohci_gtd_ptr[i]->gtd_info.data = 0;
		ohci_gtd_ptr[i]->gtd_info.b.buffer_rounding = 0;
		ohci_gtd_ptr[i]->gtd_info.b.pid = 1;		// OUT
		ohci_gtd_ptr[i]->gtd_info.b.data_toggle = 0x2|ucBulkOutDt;
		
		ohci_gtd_ptr[i]->current_buf_ptr = (u32)(ohci_out_data_ptr+GTD_BUF_MAX_SIZE*i);

		ohci_gtd_ptr[i]->next_td = (u32)ohci_gtd_ptr[i+1];

		ohci_gtd_ptr[i]->buffer_end = (u32)(ohci_out_data_ptr + GTD_BUF_MAX_SIZE*(i+1)-1);
	}
	

	// . fill the last gTD
	ohci_gtd_ptr[ucNumOfgTd-1]->gtd_info.data = 0;
	ohci_gtd_ptr[ucNumOfgTd-1]->gtd_info.b.buffer_rounding = 0;
	ohci_gtd_ptr[ucNumOfgTd-1]->gtd_info.b.pid = 1;		// OUT
	ohci_gtd_ptr[ucNumOfgTd-1]->gtd_info.b.data_toggle = 0x2|ucBulkOutDt;
	
	ohci_gtd_ptr[ucNumOfgTd-1]->current_buf_ptr = (u32)(ohci_out_data_ptr+GTD_BUF_MAX_SIZE*(ucNumOfgTd-1));

	ohci_gtd_ptr[ucNumOfgTd-1]->next_td = 0;

	ohci_gtd_ptr[ucNumOfgTd-1]->buffer_end = (u32)ohci_out_data_ptr + uTotalLength-1;


	USBHOST_OhciEnableList(0, (u32)ohci_ed_ptr[0], (u32)ohci_gtd_ptr[ucNumOfgTd-1]);
	
}


//////////
// Function Name : USBHOST_EhciXferBulkInData
// Function Desctiption : This function 
// Input : uAddr, uSize
// Output : NONE
// Version :
void USBHOST_EhciXferBulkInData(u32 uAddr, u32 uSize)
{
	u32 i;
	u16 ucNumOfqTd, ucTemp;
	u8 *pTestAddr;
	u16 usCS = 0;

	//================================================
	// first bulk out packet
	//=======================
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_QH;i++)
	{
		ehci_qh_ptr[0] = (ehci_qh_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qh_t), 32);
	}
	ehci_qtd_ptr[0] = (ehci_qtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qtd_t), 32);
	
	ehci_out_data_ptr = (u8 *)USBHOST_AllocateDataStructure(10, 4096);		// addr(4)+size(4)+flag(2)

	
	// . fill test data
	//-------------
	*(u32 *)(ehci_out_data_ptr+0) = uAddr;
	*(u32 *)(ehci_out_data_ptr+4) = uSize;
	*(u16 *)(ehci_out_data_ptr+8) = 0x0001;
	

	// . fill QH
	//---------
	// >> DWORD0
	ehci_qh_ptr[0]->next_qh_ptr.data = 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.t= 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.typ = 1;	// queue head
	ehci_qh_ptr[0]->next_qh_ptr.b.qhlp = (u32)ehci_qh_ptr[0]>>5;
	// >> DWORD1
	ehci_qh_ptr[0]->info1.data = 0;
	ehci_qh_ptr[0]->info1.b.dev_addr = oUsbDevice.m_usDevAddr;
	ehci_qh_ptr[0]->info1.b.i= 0;
	ehci_qh_ptr[0]->info1.b.ep_num = oUsbDevice.m_ucBulkOutEpNum;
	ehci_qh_ptr[0]->info1.b.eps = USB_EP_HIGH_SPEED;
	ehci_qh_ptr[0]->info1.b.dtc = 1;
	ehci_qh_ptr[0]->info1.b.h = 1;	// ?khs
	ehci_qh_ptr[0]->info1.b.max_pkt_sz= oUsbDevice.m_usBulkEPMaxPktSize;
	ehci_qh_ptr[0]->info1.b.c = 0;
	ehci_qh_ptr[0]->info1.b.rl = 0;
	// >> DWORD2
	ehci_qh_ptr[0]->info2.data = 0;
	ehci_qh_ptr[0]->info2.b.mult = ONE_TR_uFRAME;
	// >> DWORD3
	ehci_qh_ptr[0]->current_qtd_ptr = 0;
	// >> DWORD4
	ehci_qh_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[0];
	// >> DWORD5
	ehci_qh_ptr[0]->alt_next_qtd_ptr = 1;	// invalid
	// >> DWORD6~11
	ehci_qh_ptr[0]->qtd_token = 0;
	ehci_qh_ptr[0]->buf_ptr0 = 0;
	ehci_qh_ptr[0]->buf_ptr1 = 0;
	ehci_qh_ptr[0]->buf_ptr2 = 0;
	ehci_qh_ptr[0]->buf_ptr3 = 0;
	ehci_qh_ptr[0]->buf_ptr4 = 0;

	
	// . fill qTD
	//------------
	ehci_qtd_ptr[0]->next_qtd_ptr =  1;	// invalid
	ehci_qtd_ptr[0]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[0]->qtd_token.data = 0;
	ehci_qtd_ptr[0]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[0]->qtd_token.b.pid = USB_OUT_TOKEN;
	ehci_qtd_ptr[0]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[0]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[0]->qtd_token.b.ioc = 1;
	ehci_qtd_ptr[0]->qtd_token.b.xfer_total_bytes = 10;
	ehci_qtd_ptr[0]->qtd_token.b.data_toggle = ucBulkOutDt;
	ehci_qtd_ptr[0]->buf_ptr0 = (u32)ehci_out_data_ptr;	
	ehci_qtd_ptr[0]->buf_ptr1 = 0;
	ehci_qtd_ptr[0]->buf_ptr2 = 0;
	ehci_qtd_ptr[0]->buf_ptr3 = 0;
	ehci_qtd_ptr[0]->buf_ptr4 = 0;

	// . update Async Schedule & enable Async Schedule
	//--------------------------------------------	
	uhOutp32(rUH_EHCI_OP_ASYNCLISTADDR, (u32)ehci_qh_ptr[0]&0xffffffe0);

	USBHOST_EhciEnableAsyncSched();
	
	USBHOST_WaitUsbTrCompetion(false);




	//================================================
	// original bulk in packet
	//=======================	
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_QH;i++)
	{
		ehci_qh_ptr[i] = (ehci_qh_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qh_t), 32);
	}

	ucNumOfqTd = uSize/QTD_BUF_MAX_SIZE;
	if ((uSize%QTD_BUF_MAX_SIZE)!=0)
	{
		ucNumOfqTd++;
	}
	for(i=0;i<ucNumOfqTd;i++)
	{
		ehci_qtd_ptr[i] = (ehci_qtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ehci_qtd_t), 32);
	}

	ehci_in_data_ptr = (u8 *)USBHOST_AllocateDataStructure(uSize, 4096);	
	

	// . fill QH
	//---------
	// >> DWORD0
	ehci_qh_ptr[0]->next_qh_ptr.data = 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.t= 0;
	ehci_qh_ptr[0]->next_qh_ptr.b.typ = 1;	// queue head
	ehci_qh_ptr[0]->next_qh_ptr.b.qhlp = (u32)ehci_qh_ptr[0]>>5;
	// >> DWORD1
	ehci_qh_ptr[0]->info1.data = 0;
	ehci_qh_ptr[0]->info1.b.dev_addr = oUsbDevice.m_usDevAddr;
	ehci_qh_ptr[0]->info1.b.i= 0;
	ehci_qh_ptr[0]->info1.b.ep_num = oUsbDevice.m_ucBulkInEpNum;
	ehci_qh_ptr[0]->info1.b.eps = USB_EP_HIGH_SPEED;
	ehci_qh_ptr[0]->info1.b.dtc = 1;
	ehci_qh_ptr[0]->info1.b.h = 1;	// ?khs
	ehci_qh_ptr[0]->info1.b.max_pkt_sz= oUsbDevice.m_usBulkEPMaxPktSize;
	ehci_qh_ptr[0]->info1.b.c = 0;
	ehci_qh_ptr[0]->info1.b.rl = 0;
	// >> DWORD2
	ehci_qh_ptr[0]->info2.data = 0;
	ehci_qh_ptr[0]->info2.b.mult = ONE_TR_uFRAME;
	// >> DWORD3
	ehci_qh_ptr[0]->current_qtd_ptr = 0;
	// >> DWORD4
	ehci_qh_ptr[0]->next_qtd_ptr = (u32)ehci_qtd_ptr[0];
	// >> DWORD5
	ehci_qh_ptr[0]->alt_next_qtd_ptr = 1;	// invalid
	// >> DWORD6~11
	ehci_qh_ptr[0]->qtd_token = 0;
	ehci_qh_ptr[0]->buf_ptr0 = 0;
	ehci_qh_ptr[0]->buf_ptr1 = 0;
	ehci_qh_ptr[0]->buf_ptr2 = 0;
	ehci_qh_ptr[0]->buf_ptr3 = 0;
	ehci_qh_ptr[0]->buf_ptr4 = 0;

	
	// . fill qTD[x] except last qTD
	//------------------------
	for(i=0;i<(ucNumOfqTd-1);i++)
	{
		ehci_qtd_ptr[i]->next_qtd_ptr = (u32)ehci_qtd_ptr[i+1];
		ehci_qtd_ptr[i]->alt_next_qtd_prt = 1;	// invalid
		ehci_qtd_ptr[i]->qtd_token.data = 0;
		ehci_qtd_ptr[i]->qtd_token.b.sts = USB_qTD_ACTIVE;
		ehci_qtd_ptr[i]->qtd_token.b.pid = USB_IN_TOKEN;
		ehci_qtd_ptr[i]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
		ehci_qtd_ptr[i]->qtd_token.b.c_page = 0;
		ehci_qtd_ptr[i]->qtd_token.b.ioc = 0;
		ehci_qtd_ptr[i]->qtd_token.b.xfer_total_bytes = QTD_BUF_MAX_SIZE;	// 5*4K
		ehci_qtd_ptr[i]->qtd_token.b.data_toggle = ucBulkInDt;
		ehci_qtd_ptr[i]->buf_ptr0 = (u32)(ehci_in_data_ptr+QTD_BUF_MAX_SIZE*i+0);
		ehci_qtd_ptr[i]->buf_ptr1 = (u32)(ehci_in_data_ptr+QTD_BUF_MAX_SIZE*i+QTD_PAGE_MAX_SIZE*1);
		ehci_qtd_ptr[i]->buf_ptr2 = (u32)(ehci_in_data_ptr+QTD_BUF_MAX_SIZE*i+QTD_PAGE_MAX_SIZE*2);
		ehci_qtd_ptr[i]->buf_ptr3 = (u32)(ehci_in_data_ptr+QTD_BUF_MAX_SIZE*i+QTD_PAGE_MAX_SIZE*3);
		ehci_qtd_ptr[i]->buf_ptr4 = (u32)(ehci_in_data_ptr+QTD_BUF_MAX_SIZE*i+QTD_PAGE_MAX_SIZE*4);
	}
	
	// . fill last qTD
	//------------
	ehci_qtd_ptr[ucNumOfqTd-1]->next_qtd_ptr =  1;	// invalid
	ehci_qtd_ptr[ucNumOfqTd-1]->alt_next_qtd_prt = 1;	// invalid
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.data = 0;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.sts = USB_qTD_ACTIVE;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.pid = USB_IN_TOKEN;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.cerr = 0;	// no limit on the retries of this qTD
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.c_page = 0;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.ioc = 1;
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.xfer_total_bytes = uSize-QTD_BUF_MAX_SIZE*(ucNumOfqTd-1);
	ehci_qtd_ptr[ucNumOfqTd-1]->qtd_token.b.data_toggle = ucBulkInDt;

	ucTemp = (uSize-QTD_BUF_MAX_SIZE*(ucNumOfqTd-1))/QTD_PAGE_MAX_SIZE;	// calculate buffer count
	if (((uSize-QTD_BUF_MAX_SIZE*(ucNumOfqTd-1))%QTD_PAGE_MAX_SIZE)!=0)
	{
		ucTemp++;
	}
	for(i=0;i<ucTemp;i++)	// fill the buffer pointer
	{
		*(u32 *)(&(ehci_qtd_ptr[ucNumOfqTd-1]->buf_ptr0) + i) = (u32)(ehci_in_data_ptr+QTD_BUF_MAX_SIZE*(ucNumOfqTd-1)+QTD_PAGE_MAX_SIZE*i);
	}
	for(i=ucTemp;i<5;i++)
	{
		*(u32 *)(&(ehci_qtd_ptr[ucNumOfqTd-1]->buf_ptr0) + i) = 0;
	}

	// . update Async Schedule & enable Async Schedule
	//--------------------------------------------	
	uhOutp32(rUH_EHCI_OP_ASYNCLISTADDR, (u32)ehci_qh_ptr[0]&0xffffffe0);

	USBHOST_EhciEnableAsyncSched();
	
}

//////////
// Function Name : USBHOST_OhciXferBulkInData
// Function Desctiption : This function 
// Input : uAddr, uSize
// Output : NONE
// Version :
void USBHOST_OhciXferBulkInData(u32 uAddr, u32 uSize)
{
	u32 i;
	u16 ucNumOfgTd, ucTemp;
	u8 *pTestAddr;
	u16 usCS = 0;

	//================================================
	// first bulk out packet
	//=======================
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_ED;i++)
	{
		ohci_ed_ptr[i] = (ohci_ed_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_ed_t), 32);
	}
	
	ohci_gtd_ptr[0] = (ohci_gtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_gtd_t), 32);

	ohci_out_data_ptr = (u8 *)USBHOST_AllocateDataStructure(10, 4096);

	// . fill test data
	//-------------
	*(u32 *)(ohci_out_data_ptr+0) = uAddr;
	*(u32 *)(ohci_out_data_ptr+4) = uSize;
	*(u16 *)(ohci_out_data_ptr+8) = 0x0001;

	// . fill ED
	//---------
	ohci_ed_ptr[0]->ed_info.data = 0;
	ohci_ed_ptr[0]->ed_info.b.func_addr = oUsbDevice.m_usDevAddr;
	ohci_ed_ptr[0]->ed_info.b.ep_num = oUsbDevice.m_ucBulkOutEpNum;
	ohci_ed_ptr[0]->ed_info.b.direction = 0;	// get direction from TD
	ohci_ed_ptr[0]->ed_info.b.speed = 0;	// full-speed
	ohci_ed_ptr[0]->ed_info.b.format = 0;	// gTD
	ohci_ed_ptr[0]->ed_info.b.mps = oUsbDevice.m_usBulkEPMaxPktSize;
	
	ohci_ed_ptr[0]->tail_td_ptr = 0;
	
	ohci_ed_ptr[0]->head_td_ptr = (u32)ohci_gtd_ptr[0];
	
	ohci_ed_ptr[0]->next_ed = 0;	

	// gTD[0]
	ohci_gtd_ptr[0]->gtd_info.data = 0;
	ohci_gtd_ptr[0]->gtd_info.b.buffer_rounding = 0;
	ohci_gtd_ptr[0]->gtd_info.b.pid = 1;		// OUT
	ohci_gtd_ptr[0]->gtd_info.b.data_toggle = 0x2|ucBulkOutDt;
	
	ohci_gtd_ptr[0]->current_buf_ptr = (u32)ohci_out_data_ptr;

	ohci_gtd_ptr[0]->next_td = 0;

	ohci_gtd_ptr[0]->buffer_end = (u32)ohci_out_data_ptr + 10-1;

	USBHOST_OhciEnableList(0, (u32)ohci_ed_ptr[0], (u32)ohci_gtd_ptr[0]);
	
	USBHOST_WaitUsbTrCompetion(false);
	


	//================================================
	// original bulk in packet
	//=======================	
	
	// . allocate memory for QH & qTD
	//-------------------------------
	for (i=0;i<NUM_OF_ED;i++)
	{
		ohci_ed_ptr[i] = (ohci_ed_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_ed_t), 32);
	}

	ucNumOfgTd = uSize/GTD_BUF_MAX_SIZE;
	if ((uSize%GTD_BUF_MAX_SIZE)!=0)
	{
		ucNumOfgTd++;
	}
	for(i=0;i<ucNumOfgTd;i++)
	{
		ohci_gtd_ptr[i] = (ohci_gtd_t_ptr)USBHOST_AllocateDataStructure(sizeof(ohci_gtd_t), 32);
	}

	ohci_in_data_ptr = (u8 *)USBHOST_AllocateDataStructure(uSize, 4096);	

	// . fill ED
	//---------
	ohci_ed_ptr[0]->ed_info.data = 0;
	ohci_ed_ptr[0]->ed_info.b.func_addr = oUsbDevice.m_usDevAddr;
	ohci_ed_ptr[0]->ed_info.b.ep_num = oUsbDevice.m_ucBulkInEpNum;
	ohci_ed_ptr[0]->ed_info.b.direction = 0;	// get direction from TD
	ohci_ed_ptr[0]->ed_info.b.speed = 0;	// full-speed
	ohci_ed_ptr[0]->ed_info.b.format = 0;	// gTD
	ohci_ed_ptr[0]->ed_info.b.mps = oUsbDevice.m_usBulkEPMaxPktSize;
	
	ohci_ed_ptr[0]->tail_td_ptr = 0;
	
	ohci_ed_ptr[0]->head_td_ptr = (u32)ohci_gtd_ptr[0];
	
	ohci_ed_ptr[0]->next_ed = 0;	

	// . fill gTD[x] except last gTD
	//------------------------
	for(i=0;i<(ucNumOfgTd-1);i++)
	{
		ohci_gtd_ptr[i]->gtd_info.data = 0;
		ohci_gtd_ptr[i]->gtd_info.b.buffer_rounding = 0;
		ohci_gtd_ptr[i]->gtd_info.b.pid = 2;		// IN
		ohci_gtd_ptr[i]->gtd_info.b.data_toggle = 0x2|ucBulkInDt;
		
		ohci_gtd_ptr[i]->current_buf_ptr = (u32)(ohci_in_data_ptr+GTD_BUF_MAX_SIZE*i);

		ohci_gtd_ptr[i]->next_td = (u32)ohci_gtd_ptr[i+1];

		ohci_gtd_ptr[i]->buffer_end = (u32)(ohci_in_data_ptr + GTD_BUF_MAX_SIZE*(i+1)-1);
	}

	// . fill the last gTD
	ohci_gtd_ptr[ucNumOfgTd-1]->gtd_info.data = 0;
	ohci_gtd_ptr[ucNumOfgTd-1]->gtd_info.b.buffer_rounding = 0;
	ohci_gtd_ptr[ucNumOfgTd-1]->gtd_info.b.pid = 2;		// IN
	ohci_gtd_ptr[ucNumOfgTd-1]->gtd_info.b.data_toggle = 0x2|ucBulkInDt;
	
	ohci_gtd_ptr[ucNumOfgTd-1]->current_buf_ptr = (u32)(ohci_in_data_ptr+GTD_BUF_MAX_SIZE*(ucNumOfgTd-1));

	ohci_gtd_ptr[ucNumOfgTd-1]->next_td = 0;

	ohci_gtd_ptr[ucNumOfgTd-1]->buffer_end = (u32)(ohci_in_data_ptr + uSize-1);

	USBHOST_OhciEnableList(0, (u32)ohci_ed_ptr[0], (u32)ohci_gtd_ptr[ucNumOfgTd-1]);
		
}



//////////
// Function Name : USBHOST_VerifyBulkInData
// Function Desctiption : This function 
// Input : uSize
// Output : NONE
// Version :
void USBHOST_VerifyBulkInData(u32 uSize)
{
	u8 *pTestAddr;
	u32 i, j=0;

	if (oUsbHost.m_eEhciState == EHCI_STATE_CONFIGURED)
	{
		pTestAddr = ehci_in_data_ptr;
	}
	else if (oUsbHost.m_eOhciState == OHCI_STATE_CONFIGURED)
	{
		pTestAddr = ohci_in_data_ptr;
	}
	else
	{
		Assert(0);
	}
	
	for (i=0;i<uSize;i++)
	{
		if ((i%oUsbDevice.m_usBulkEPMaxPktSize) == 0)
		{
			j++;

			if (*pTestAddr++ != (j&0xff))
			{
				UART_Printf("\nVerifying of Bulk In Data is FAIL @0x%08x\n", i);
				return;
			}
		}
		else
		{
			if (*pTestAddr++ != (i&0xff))
			{
				UART_Printf("\nVerifying of Bulk In Data is FAIL @0x%08x\n", i);
				return;
			}
		}
	}
	
	UART_Printf("\nVerifying of Bulk In Data is OK!!!\n", i);
}




//////////
// Function Name : USBHOST_TestBulkOut
// Function Desctiption : This function 
// Input : uAddr, uSize
// Output : NONE
// Version :
void USBHOST_TestBulkOut(u32 uAddr, u32 uSize)
{	
	DbgUsbHost("\nINSNREG00=0x%08x\n", uhInp32(rUH_EHCI_OP_INSNREG00));
	DbgUsbHost("\nINSNREG01=0x%08x\n", uhInp32(rUH_EHCI_OP_INSNREG01));
	
	USBHOST_ParseDeviceInform();

	if (oUsbHost.m_eEhciState == EHCI_STATE_CONFIGURED)
	{
		USBHOST_EhciXferBulkOutData(uAddr, uSize);
	}
	else if (oUsbHost.m_eOhciState == OHCI_STATE_CONFIGURED)
	{
		USBHOST_OhciXferBulkOutData(uAddr, uSize);
	}
	else
	{
		Assert(0);
	}
	
	USBHOST_WaitUsbTrCompetion(false);

	// . update data toggle of bulk out ep
	//------------------------------
	if ((((uSize+10)%(oUsbDevice.m_usBulkEPMaxPktSize*2))>0) && 
		(((uSize+10)%(oUsbDevice.m_usBulkEPMaxPktSize*2))<=oUsbDevice.m_usBulkEPMaxPktSize))
	{
		ucBulkOutDt = !ucBulkOutDt;
	}
}


//////////
// Function Name : USBHOST_TestBulkIn
// Function Desctiption : This function 
// Input : uAddr, uSize
// Output : NONE
// Version :
void USBHOST_TestBulkIn(u32 uAddr, u32 uSize)
{
	DbgUsbHost("\nINSNREG00=0x%08x\n", uhInp32(rUH_EHCI_OP_INSNREG00));
	DbgUsbHost("\nINSNREG01=0x%08x\n", uhInp32(rUH_EHCI_OP_INSNREG01));
	
	USBHOST_ParseDeviceInform();

	if (oUsbHost.m_eEhciState == EHCI_STATE_CONFIGURED)
	{
		USBHOST_EhciXferBulkInData(uAddr, uSize);
	}
	else if (oUsbHost.m_eOhciState == OHCI_STATE_CONFIGURED)
	{
		USBHOST_OhciXferBulkInData(uAddr, uSize);
	}
	else
	{
		Assert(0);
	}
	
	USBHOST_WaitUsbTrCompetion(false);

	USBHOST_VerifyBulkInData(uSize);

	// . update data toggle of bulk in/out ep
	//---------------------------------
	ucBulkOutDt = !ucBulkOutDt;
	if (((uSize%(oUsbDevice.m_usBulkEPMaxPktSize*2))>0) && 
		((uSize%(oUsbDevice.m_usBulkEPMaxPktSize*2))<=oUsbDevice.m_usBulkEPMaxPktSize))
	{
		ucBulkInDt = !ucBulkInDt;
	}
}


//////////
// Function Name : USBHOST_EnterTestMode
// Function Desctiption : This function 
// Input : eTestMode
// Output : NONE
// Version :
void USBHOST_EnterTestMode(USB_PORT_TEST eTestMode)
{
	ehci_portsc_t ehci_portsc;
	ehci_usbcmd_t ehci_usbcmd;

	Assert(eTestMode>TEST_MODE_DISABLED && eTestMode <=TEST_FORCE_ENABLE);
	
	
	// . disable all schedules
	//----------------------
	uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	ehci_usbcmd.b.periodic_sched_en = 0;
	ehci_usbcmd.b.async_sched_en = 0;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);


	// . place all enabled root ports into the suspended state
	//------------------------------------------------
	uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);
	ehci_portsc.b.p_suspend = 1;
	uhOutp32(rUH_EHCI_OP_PORTSC, ehci_portsc.data);

	
	// . stop host controller
	//----------------------
	uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	ehci_usbcmd.b.run_stop = 0;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	USBHOST_WaitEhciHcHalted();

	
	// . set port test control field in the port
	//------------------------------------------------
	uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);
	ehci_portsc.b.p_test_ctrl= eTestMode;
	uhOutp32(rUH_EHCI_OP_PORTSC, ehci_portsc.data);

}


//////////
// Function Name : USBHOST_ExitTestMode
// Function Desctiption : This function 
// Input : NONE
// Output : NONE
// Version :
void USBHOST_ExitTestMode(void)
{
	ehci_usbcmd_t ehci_usbcmd;
	
	
	// . ensure ths host controller is halted
	//---------------------------------
	USBHOST_WaitEhciHcHalted();


	// . exit test mode by resetting host controller
	//---------------------------------------
	uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	ehci_usbcmd.b.hc_reset = 1;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);

	do
	{
		uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	} while(ehci_usbcmd.b.hc_reset);

}



//===================================================================
// AUTO TEST

u8 USBHOST_WaitEhciHcHalted_autotest(void)
{
	ehci_usbsts_t ehci_usbsts;
	u32 uWaitCount = 0;

	do
	{
		uhInp32_(rUH_EHCI_OP_USBSTS, ehci_usbsts.data);
		if (ehci_usbsts.b.hc_halted)
		{
			break;
		}
		else
		{		
			Delay(1);
			uWaitCount++;
		}

		if (uWaitCount > USBHOST_MAX_WAITCOUNT)
		{
			return false;
		}
	}while(1);

	return true;
}


u8 USBHOST_InitEhciCon_autotest(void)
{
	ehci_portsc_t ehci_portsc;
	ehci_usbcmd_t ehci_usbcmd;
	ehci_usbintr_t ehci_usbintr;
	ehci_configflag_t ehci_configflag;
	u32 uWaitCount = 0;

	// . program CTRLDSSEGMENT regsiter
	//---------------------------------
	// this register is not used when not using 64-bit addressing

	
	// . stop the host controller 
	//-----------------------
	uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	ehci_usbcmd.b.run_stop = 0;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	if (USBHOST_WaitEhciHcHalted_autotest() != true)
	{
		return false;
	}
	
	
	// . reset the ehci controller & wait until reset operation is completed
	//--------------------------------------------------------
	ehci_usbcmd.b.hc_reset = 1;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);
	do
	{
		uhInp32_(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);

		Delay(1);
		uWaitCount++;

		if (uWaitCount > USBHOST_MAX_WAITCOUNT)
		{
			return false;
		}
	} while(ehci_usbcmd.b.hc_reset);


	// . program USBINTR with appropriat values
	//---------------------------------------
	ehci_usbintr.data = 0;
	ehci_usbintr.b.usbint_en= 1;
	ehci_usbintr.b.usberrint_en = 1;
	ehci_usbintr.b.p_chng_int_en = 1;
	uhOutp32(rUH_EHCI_OP_USBINTR, ehci_usbintr.data);


	// . Write the base address of the periodic frame list to the PERIODICLISTBASE register
	//------------------------------------------------------------------------
	uhOutp32(rUH_EHCI_OP_PERIODICLISTBASE, 0);	// must be modified


	// . set port power-on
	//------------------
	uhInp32_(rUH_EHCI_OP_PORTSC, ehci_portsc.data);
	ehci_portsc.b.p_power = 1;
	uhOutp32(rUH_EHCI_OP_PORTSC, ehci_portsc.data);


	// . start Host controller
	//--------------------
	if (USBHOST_WaitEhciHcHalted_autotest() != true)
	{
		return false;
	}
	
	ehci_usbcmd.b.run_stop = 1;
	uhOutp32(rUH_EHCI_OP_USBCMD, ehci_usbcmd.data);


	// . route port ownership to the EHCI controller
	//---------------------------------------
	ehci_configflag.data = 0;
	ehci_configflag.b.cf = 1;
	uhOutp32(rUH_EHCI_OP_CONFIGFLAG, ehci_configflag.data);

	return true;
	
}


u8 USBHOST_Init_autotest(void)
{		
	INTC_Disable(NUM_UHOST);
	
	SYSC_SetOscPadOnOff(eMODE_NORMAL, eOSC_USB, true);	//enable OTG clock pad
	SYSC_SetUSBPHYControl(true);	//unmask usb signal

	USBHOST_InitPhyCon();
	
	g_uUsbHostDataMem = USBHOST_DATA_BASE;
	oUsbHost.m_eEhciState = EHCI_STATE_INIT;
	g_bCurrentConnectStatus = false;
	
	if (USBHOST_InitEhciCon_autotest() != true)
	{
		return false;

	}
	
	INTC_Enable(NUM_UHOST);

	return true;	
}


u8 USBHOST_WaitUsbTrCompetion_autotest()
{
	u32 uWaitCount = 0;
	
	while(1)
	{
		if (g_bUsbTrCompletion == true)
		{
			g_bUsbTrCompletion = false;
			break;
		}
		else
		{			
			Delay(1);
			uWaitCount++;
		}

		if (uWaitCount > USBHOST_MAX_WAITCOUNT)
		{
			return false;
		}
	}

	return true;
}

u8 USBHOST_EhciEnumerate_autotest(void)
{
	USBHOST_EhciWaitEhciStateChange(EHCI_STATE_READY);

	// . Get Device Descriptor with Addr 0
	//-------------------------------
	oUsbHost.m_eEhciState = EHCI_STATE_GD_DEV_BEFORE;
	USBHOST_EhciGetDeviceDescriptor(0);
	if (USBHOST_WaitUsbTrCompetion_autotest() != true)
	{
		return false;
	}

	// . Set Address
	//--------------
	oUsbHost.m_eEhciState = EHCI_STATE_SET_ADDR;
	oUsbHost.m_usDevAddr = 1;
	USBHOST_EhciSetAddress(oUsbHost.m_usDevAddr);
	if (USBHOST_WaitUsbTrCompetion_autotest() != true)
	{
		return false;
	}

	// . Get Device Descriptor with setting Addr
	//-----------------------------------
	oUsbHost.m_eEhciState = EHCI_STATE_GD_DEV_AFTER;
	USBHOST_EhciGetDeviceDescriptor(oUsbHost.m_usDevAddr);
	if (USBHOST_WaitUsbTrCompetion_autotest() != true)
	{
		return false;
	}

	// . Get Configuration Descriptor
	//---------------------------
	oUsbHost.m_eEhciState = EHCI_STATE_GD_CFG;
	USBHOST_EhciGetConfigDescriptor();
	if (USBHOST_WaitUsbTrCompetion_autotest() != true)
	{
		return false;
	}

	// . (optional) Get String Descriptor, LANGID
	//------------------------------------
	oUsbHost.m_eEhciState = EHCI_STATE_GD_STR0;
	USBHOST_EhciGetStringDescriptor(0);
	if (USBHOST_WaitUsbTrCompetion_autotest() != true)
	{
		return false;
	}

	// . (optional) Get String Descriptor, Manufacturer
	//-----------------------------------------
	if (oUsbDevice.m_oDesc.oDescDevice.iManufacturer != 0)
	{
		switch (oUsbDevice.m_oDesc.oDescDevice.iManufacturer)
		{
			case 1:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR1;
				break;

			case 2:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR2;
				break;

			case 3:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR3;
				break;

			default:
				UART_Printf("More String Descriptor is needed\n");
				Assert(0);
				break;
		}
		USBHOST_EhciGetStringDescriptor(oUsbDevice.m_oDesc.oDescDevice.iManufacturer);
		if (USBHOST_WaitUsbTrCompetion_autotest() != true)
		{
			return false;
		}
	}

	// . (optional) Get String Descriptor, Product Name
	//------------------------------------------
	if (oUsbDevice.m_oDesc.oDescDevice.iProduct != 0)
	{
		switch (oUsbDevice.m_oDesc.oDescDevice.iProduct)
		{
			case 1:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR1;
				break;

			case 2:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR2;
				break;

			case 3:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR3;
				break;

			default:
				UART_Printf("More String Descriptor is needed\n");
				Assert(0);
				break;
		}
		USBHOST_EhciGetStringDescriptor(oUsbDevice.m_oDesc.oDescDevice.iProduct);
		if (USBHOST_WaitUsbTrCompetion_autotest() != true)
		{
			return false;
		}
	}

	// . (optional) Get String Descriptor, Serial Number
	//------------------------------------------
	if (oUsbDevice.m_oDesc.oDescDevice.iSerialNumber!= 0)
	{
		switch (oUsbDevice.m_oDesc.oDescDevice.iSerialNumber)
		{
			case 1:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR1;
				break;

			case 2:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR2;
				break;

			case 3:
				oUsbHost.m_eEhciState = EHCI_STATE_GD_STR3;
				break;

			default:
				UART_Printf("More String Descriptor is needed\n");
				Assert(0);
				break;
		}
		USBHOST_EhciGetStringDescriptor(oUsbDevice.m_oDesc.oDescDevice.iSerialNumber);
		if (USBHOST_WaitUsbTrCompetion_autotest() != true)
		{
			return false;
		}
	}


	// . Set Configuration
	//--------------------
	oUsbHost.m_eEhciState = EHCI_STATE_SET_CFG;
	USBHOST_EhciSetConfiguration();
	if (USBHOST_WaitUsbTrCompetion_autotest() != true)
	{
		return false;
	}

	return true;
	
}

