#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "def.h"
#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "sysc.h"
#include "gpio.h"
#include "dma.h"
#include "intc.h"
#include "tsi.h"

#define TSI_BUFFER_BASE	CODEC_MEM_ST
#define DST_BUFFER_BASE	(CODEC_MEM_ST+0x01000000)
#define STREAM_PACKET_NUM	3			// STREAM_PACKET_NUM is less than 348.   test : 54
#define LOOP_COUNT		10

u32 uStreamPacketCount, uDoneTsDma;
sync_det_mode eSel_Save;
extern TSI_SIGNAL TsiSignal;
u8 uTsiOddEven, uUsedDubbleBuffer;
	
extern void Test_TxTsi(void);
	
static void ProcessIntFifoFull(void)
{
	TSI_InitIp(TSI_BUFFER_BASE, TsiSignal.TSI_PACKET_NUM);
	TSI_EnableClkTimeoutInt();
	TSI_EnablePidFilter();
	TSI_SetSyncDetMode(eSel_Save);
	TSI_EnablePidFilter();	
	TSI_ClockOn();
}

static void __irq Isr_TSI(void)
{
	u32 uRead; 
	INTC_Disable(NUM_TSI);
	INTC_ClearVectAddr();

	//UART_Printf("\nTSI Interrupt happend!!\n");

	uRead = TSI_ReadIntFlag();

	if(uRead&(1<<0))
	{
//		UART_Printf("ts_error_flag interrupt is generated!\n");
		UART_Printf("ts_error\n");
		TSI_ClearIntFlag(ts_error_flag);
	}

	if(uRead&(1<<1))
	{
		UART_Printf("ts_error_timeout_flag interrupt is generated!\n");
		TSI_ClearIntFlag(ts_error_timeout_flag);
	}

	if(uRead&(1<<2))
	{
		UART_Printf("psof_flag interrupt is generated!\n");
		TSI_ClearIntFlag(psof_flag);
	}
	
	if(uRead&(1<<3))
	{
		UART_Printf("psuf_error\n");	
		TSI_ClearIntFlag(psuf_flag);
	}

	if(uRead&(1<<4))
	{
		UART_Printf("sync_mismatch\n");	
		TSI_ClearIntFlag(sync_mismatch_flag);
	}

	if(uRead&(1<<5))
	{
	//	UART_Printf("int_fifo_full_flag interrupt is generated!\n");
		UART_Printf(".");
		TSI_ClearIntFlag(int_fifo_full_flag);
		ProcessIntFifoFull();
	}

	if(uRead&(1<<6))
	{
		//UART_Printf("output_buffer_full_flag interrupt is generated!\n");
		if (uUsedDubbleBuffer)
		{
			if(uTsiOddEven)
			{
				TSI_WriteRegTsBase(DST_BUFFER_BASE);
				uTsiOddEven = 0;
			}
			else
			{
				TSI_WriteRegTsBase(TSI_BUFFER_BASE);	
				uTsiOddEven = 1;			
			}
		}
		UART_Printf("*");
		TSI_ClearIntFlag(output_buffer_full_flag);
		--uStreamPacketCount;
	}

	if(uRead&(1<<7))
	{
		UART_Printf("@\n");
	//	UART_Printf("dma_complete_flag interrupt is generated!\n");
		TSI_ClearIntFlag(dma_complete_flag);
	}

	INTC_Enable(NUM_TSI);
}

static void CalculateStreamSize(u32 uLoopCount)
{
	u32 temp;

	temp = 47*4*TsiSignal.TSI_PACKET_NUM*uLoopCount;

	UART_Printf("Transport Stream Size: %dbyte\n", temp);
	temp = temp + TSI_BUFFER_BASE -1;
	UART_Printf("The end addres of Transport Stream : %xh\n", temp);
}

static void CalculateStreamSizeForDoubleBuffer(u32 uLoopCount)
{
	u32 temp;

	temp = 47*4*TsiSignal.TSI_PACKET_NUM*uLoopCount;

	UART_Printf("Transport Stream Size: %dbyte\n", temp);
	temp = temp + DST_BUFFER_BASE -1;
	UART_Printf("The end addres of Transport Stream : %xh\n", temp);
}

static void TestReset(void)
{
	TSI_Reset();
}

static void ClearTsiBuffer(void)
{
	u32 i;

	for(i=0; i<188*54; i++)
	*(u8 *)(TSI_BUFFER_BASE+i) = 0;

	for(i=0; i<188*54; i++)
	*(u8 *)(DST_BUFFER_BASE+i) = 0;	
}

static void TurnOffClock(void)
{
	TSI_ClockOff();
}

static void SimpleTsiTest(void)
{
	sync_det_mode eSel;

	uStreamPacketCount = 1;
	
	while(1)
	{
		UART_Printf("Select sync detecting mode : \n");
		UART_Printf("0: using TS_SYNC(detecting consecutive 8-bit)\n");
		UART_Printf("1: using TS_SYNC(detecting only 1-bit)\n");
		UART_Printf("2: using sync byte (0x47)\n");
		eSel = (sync_det_mode)UART_GetIntNum();
		if(eSel == TS_SYNC_8_bit||eSel==TS_SYNC_1_bit||eSel==SYNC_byte_47h)
			break;
		else
			UART_Printf("Wrong number!\n");	
	}

	ClearTsiBuffer();
	INTC_Init();
	INTC_SetVectAddr(NUM_TSI, Isr_TSI);
	INTC_Enable(NUM_TSI);	

	TSI_InitIp(TSI_BUFFER_BASE, TsiSignal.TSI_PACKET_NUM);
	TSI_EnableClkTimeoutInt();
	eSel_Save = eSel;	
	TSI_SetSyncDetMode(eSel);
	TSI_ClockOn();
	while(uStreamPacketCount > 0);
	TSI_ClockOff();

	CalculateStreamSize(1);	
}

static void PidFilterModeTest(void)
{
	sync_det_mode eSel;

	uStreamPacketCount = 1;
	
	while(1)
	{
		UART_Printf("Select sync detecting mode : \n");
		UART_Printf("0: using TS_SYNC(detecting consecutive 8-bit)\n");
		UART_Printf("1: using TS_SYNC(detecting only 1-bit)\n");
		UART_Printf("2: using sync byte (0x47)\n");
		eSel = (sync_det_mode)UART_GetIntNum();
		if(eSel == TS_SYNC_8_bit||eSel==TS_SYNC_1_bit||eSel==SYNC_byte_47h)
			break;
		else
			UART_Printf("Wrong number!\n");	
	}

	ClearTsiBuffer();
	INTC_Init();
	INTC_SetVectAddr(NUM_TSI, Isr_TSI);
	INTC_Enable(NUM_TSI);	

	TSI_InitIp(TSI_BUFFER_BASE, TsiSignal.TSI_PACKET_NUM);
	TSI_EnableClkTimeoutInt();
	TSI_EnablePidFilter();
	eSel_Save = eSel;	
	TSI_SetSyncDetMode(eSel);
	TSI_ClockOn();
	while(uStreamPacketCount > 0);
	TSI_ClockOff();

	CalculateStreamSize(1);	
}


static void AutoResetTest(void)
{
	TSI_Reset();
	TSI_AfterResetReadAllReg();
	UART_Printf("Auto Reset Test is completed, if it is no error message, Auto Reset Test is OK.\n");
}

static void ReadAllRegister(void)
{
	TSI_ReadAllReg();
}

static void AutoRegReadWriteTest(void)
{
	TSI_WriteAllReg();
	TSI_AfterWriteAllRegReadAllReg();
	UART_Printf("Auto Register R/W Test is completed, if it is no error message, Auto Register R/W Test is OK.\n");
}

static void SpecialRegisterTest(void)
{
	int i, sel;
	const testFuncMenu menu[] =
	{
		0,    		         	  		"Exit",
		AutoResetTest,			"Auto Reset Test",		
		AutoRegReadWriteTest,		"Auto Register Read Write Test",				
		ReadAllRegister,			"Read all register",			
		0, 0
	};
 
	while(1)
	{
		UART_Printf("\n");
		for (i=0; (int)(menu[i].desc)!=0; i++)
			UART_Printf("%2d: %s\n", i, menu[i].desc);

		UART_Printf("\nSelect the function to test : ");
		sel = UART_GetIntNum();
		UART_Printf("\n");
		if(sel==0)break;
		if (sel>0 && sel<(sizeof(menu)/8-1))
			(menu[sel].func) ();
	}


}

static void SetTsiCfg(void)
{
	u32 sel;

	UART_Printf("Now, TSI signal setting value.\n");
	if(TsiSignal.TSI_CLK==eRising_Edge) UART_Printf("TS_CLK : Rising_Edge\n");
	else	UART_Printf("TS_CLK : Falling_Edge\n");
	if(TsiSignal.TSI_SYNC==eActive_Low) UART_Printf("TS_SYNC : Active_Low\n");
	else	UART_Printf("TS_SYNC : Active_High\n");
	if(TsiSignal.TSI_VALID==eActive_Low) UART_Printf("TS_VALID : Active_Low\n");
	else	UART_Printf("TS_VALID : Active_High\n");
	if(TsiSignal.TSI_ERROR==eActive_Low) UART_Printf("TS_ERROR : Active_Low\n");
	else	UART_Printf("TS_ERROR : Active_High\n");
	UART_Printf("TSI Packet num : %d\n", TsiSignal.TSI_PACKET_NUM);	
	UART_Printf("Skip: 0, No skip: 1\n");
	sel = UART_GetIntNum();

	if(sel==1)
	{

	while(1)
	{
		UART_Printf("TS_CLK : \n");
		UART_Printf("0: Skip,	1: Rising_Edge,	2: Falling_Edge\n");
		sel =UART_GetIntNum();
		if(sel==0) break;
		else if(sel==1)	{TsiSignal.TSI_CLK = eRising_Edge; break;}	
		else if(sel==2)	{TsiSignal.TSI_CLK = eFalling_Edge; break;}	
		else	UART_Printf("Wrong number!\n");	
	}
	while(1)
	{
		UART_Printf("TS_SYNC : \n");
		UART_Printf("0: Skip,	1: Active_Low,	2: Active_High\n");
		sel =UART_GetIntNum();
		if(sel==0) break;
		else if(sel==1)	{TsiSignal.TSI_SYNC = eActive_Low; break;}	
		else if(sel==2)	{TsiSignal.TSI_SYNC = eActive_High; break;}	
		else	UART_Printf("Wrong number!\n");	
	}
	while(1)
	{
		UART_Printf("TS_VALID : \n");
		UART_Printf("0: Skip,	1: Active_Low,	2: Active_High\n");
		sel =UART_GetIntNum();
		if(sel==0) break;
		else if(sel==1)	{TsiSignal.TSI_VALID = eActive_Low; break;}	
		else if(sel==2)	{TsiSignal.TSI_VALID = eActive_High; break;}	
		else	UART_Printf("Wrong number!\n");	
	}
	while(1)
	{
		UART_Printf("TS_ERROR : \n");
		UART_Printf("0: Skip,	1: Active_Low,	2: Active_High\n");
		sel =UART_GetIntNum();
		if(sel==0) break;
		else if(sel==1)	{TsiSignal.TSI_ERROR = eActive_Low; break;}	
		else if(sel==2)	{TsiSignal.TSI_ERROR = eActive_High; break;}	
		else	UART_Printf("Wrong number!\n");	
	}

	while(1)
	{
		UART_Printf("TSI Packet num : \n");
		UART_Printf("0: Skip,	1: change \n");
		sel =UART_GetIntNum();
		if(sel==0) break;
		else if(sel==1)	
		{			
			TsiSignal.TSI_PACKET_NUM= UART_GetIntNum(); break;
		}	
		else	UART_Printf("Wrong number!\n");	
	}	

	UART_Printf("Now, TSI signal changing the setting value.\n");
	if(TsiSignal.TSI_CLK==eRising_Edge) UART_Printf("TS_CLK : Rising_Edge\n");
	else	UART_Printf("TS_CLK : Falling_Edge\n");
	if(TsiSignal.TSI_SYNC==eActive_Low) UART_Printf("TS_SYNC : Active_Low\n");
	else	UART_Printf("TS_SYNC : Active_High\n");
	if(TsiSignal.TSI_VALID==eActive_Low) UART_Printf("TS_VALID : Active_Low\n");
	else	UART_Printf("TS_VALID : Active_High\n");
	if(TsiSignal.TSI_ERROR==eActive_Low) UART_Printf("TS_ERROR : Active_Low\n");
	else	UART_Printf("TS_ERROR : Active_High\n");
	UART_Printf("TSI Packet num : %d\n", TsiSignal.TSI_PACKET_NUM);	
	}
}


static void ReceiveTsByCpu(void)
{
	sync_det_mode eSel;
	u32 uLoopCnt,uSrcAddr,uDstAddr;

//	uStreamPacketCount = STREAM_PACKET_NUM;
	uStreamPacketCount = 1;	
	uSrcAddr = TSI_BUFFER_BASE;
	uDstAddr 	= DST_BUFFER_BASE;
	while(1)
	{
		UART_Printf("Select sync detecting mode : \n");
		UART_Printf("0: using TS_SYNC(detecting consecutive 8-bit)\n");
		UART_Printf("1: using TS_SYNC(detecting only 1-bit)\n");
		UART_Printf("2: using sync byte (0x47)\n");
		eSel = (sync_det_mode)UART_GetIntNum();
		if(eSel == TS_SYNC_8_bit||eSel==TS_SYNC_1_bit||eSel==SYNC_byte_47h)
			break;
		else
			UART_Printf("Wrong number!\n");	
	}

	ClearTsiBuffer();

	INTC_Init();
	INTC_SetVectAddr(NUM_TSI, Isr_TSI);
	INTC_Enable(NUM_TSI);	

	TSI_InitIp(TSI_BUFFER_BASE, TsiSignal.TSI_PACKET_NUM);	
	TSI_EnableClkTimeoutInt();
//	TSI_EnablePidFilter();	
	eSel_Save = eSel;
	TSI_SetSyncDetMode(eSel);
	TSI_ClockOn();

	for(uLoopCnt=0; uLoopCnt<LOOP_COUNT; uLoopCnt++)
	{
		while(uStreamPacketCount > 0);

		TSI_TransferStream(TSI_BUFFER_BASE, uDstAddr, TsiSignal.TSI_PACKET_NUM);

		uDstAddr = uDstAddr + 47*4*TsiSignal.TSI_PACKET_NUM;
		uStreamPacketCount = TsiSignal.TSI_PACKET_NUM;
	}

	TSI_ClockOff();	
	CalculateStreamSize(LOOP_COUNT);	
}

static void SetTsiSignalByDefault(void)
{
#ifndef USING_CHANNEL_BOARD
	TsiSignal.TSI_CLK = eRising_Edge;
	TsiSignal.TSI_SYNC = eActive_High;
	TsiSignal.TSI_VALID = eActive_Low;
	TsiSignal.TSI_ERROR = eActive_High;
	//TsiSignal.TSI_PACKET_NUM = 4;			// If 4-beat burst, 22 is error
	TsiSignal.TSI_PACKET_NUM = 54;			// If 4-beat burst, 22 is error
#else
	// channel B/D setting
	TsiSignal.TSI_CLK = eRising_Edge;
	TsiSignal.TSI_SYNC = eActive_High;		//8bit sync
	TsiSignal.TSI_VALID = eActive_Low;
	TsiSignal.TSI_ERROR = eActive_High;
	TsiSignal.TSI_PACKET_NUM = 4;			// If 4-beat burst, 22 is error
	//TsiSignal.TSI_PACKET_NUM = 54;			// If 4-beat burst, 22 is error
#endif 
}

static void IntegerMaker(void)
{
	u32 i;

	for(i=0; i<47*4*54; i++)
	*(u8 *)(TSI_BUFFER_BASE+i) = i%255;	
}

static void DoubleBufferTest(void)
{
	sync_det_mode eSel;

	uStreamPacketCount = 4;
	uTsiOddEven =0;
	uUsedDubbleBuffer = true;
	
	while(1)
	{
		UART_Printf("Select sync detecting mode : \n");
		UART_Printf("0: using TS_SYNC(detecting consecutive 8-bit)\n");
		UART_Printf("1: using TS_SYNC(detecting only 1-bit)\n");
		UART_Printf("2: using sync byte (0x47)\n");
		eSel = (sync_det_mode)UART_GetIntNum();
		if(eSel == TS_SYNC_8_bit||eSel==TS_SYNC_1_bit||eSel==SYNC_byte_47h)
			break;
		else
			UART_Printf("Wrong number!\n");	
	}

	ClearTsiBuffer();
	INTC_Init();
	INTC_SetVectAddr(NUM_TSI, Isr_TSI);
	INTC_Enable(NUM_TSI);	

	TSI_InitIp(TSI_BUFFER_BASE, TsiSignal.TSI_PACKET_NUM);
//	TSI_InitIp(TSI_BUFFER_BASE, TsiSignal.TSI_PACKET_NUM/2);
	TSI_EnableClkTimeoutInt();
//	TSI_EnablePidFilter();
	eSel_Save = eSel;	
	TSI_SetSyncDetMode(eSel);
	TSI_ClockOn();	
	TSI_WriteRegTsBase(DST_BUFFER_BASE);	
	while(uStreamPacketCount > 0);
	TSI_ClockOff();

	CalculateStreamSize(1);	
	CalculateStreamSizeForDoubleBuffer(1);
}

static void TsiAgingTest(void)
{
	sync_det_mode eSel;

	uStreamPacketCount = 1000;
	
	while(1)
	{
		UART_Printf("Select sync detecting mode : \n");
		UART_Printf("0: using TS_SYNC(detecting consecutive 8-bit)\n");
		UART_Printf("1: using TS_SYNC(detecting only 1-bit)\n");
		UART_Printf("2: using sync byte (0x47)\n");
		eSel = (sync_det_mode)UART_GetIntNum();
		if(eSel == TS_SYNC_8_bit||eSel==TS_SYNC_1_bit||eSel==SYNC_byte_47h)
			break;
		else
			UART_Printf("Wrong number!\n");	
	}

	ClearTsiBuffer();
	INTC_Init();
	INTC_SetVectAddr(NUM_TSI, Isr_TSI);
	INTC_Enable(NUM_TSI);	

	TSI_InitIp(TSI_BUFFER_BASE, TsiSignal.TSI_PACKET_NUM);
	TSI_EnableClkTimeoutInt();
//	TSI_EnablePidFilter();
	eSel_Save = eSel;	
	TSI_SetSyncDetMode(eSel);
	TSI_ClockOn();
	while(uStreamPacketCount > 0);
	TSI_ClockOff();

	CalculateStreamSize(1);	
}

void Test_TSI(void)
{
	int i, sel;
	const testFuncMenu menu[] =
	{
		0,    		         	  	"Exit",
		SpecialRegisterTest,	"Special Registers test",						
		ClearTsiBuffer,		"Clear TSI Buffer",			
		TestReset,			"TSI Reset test",
		SetTsiCfg,			"Set TSI Config",		
		SimpleTsiTest,		"Simple TSI test",
		ReceiveTsByCpu,		"Receive transport stream by cpu",		
		DoubleBufferTest,		"Double Buffer Test",	
		PidFilterModeTest,		"Pid Filter Mode Test",			
		TsiAgingTest,			"TSI Aging Test",
		TurnOffClock,			"Turn off TSI clock",	
		IntegerMaker,			"Integer Maker",	
		
		Test_TxTsi,			"TSI[Tx] For Testing TSI[Rx]",	
		
		0, 0
	};

	///  GPIO Setting for SPDIF 
	TSI_SetGpio();

	SetTsiSignalByDefault();
 
	while(1)
	{
		uUsedDubbleBuffer = false;
		UART_Printf("\n");
		for (i=0; (int)(menu[i].desc)!=0; i++)
			UART_Printf("%2d: %s\n", i, menu[i].desc);

		UART_Printf("\nSelect the function to test : ");
		sel = UART_GetIntNum();
		UART_Printf("\n");
		if(sel==0)break;
		if (sel>0 && sel<(sizeof(menu)/8-1))
			(menu[sel].func) ();
	}
}


