/**************************************************************************************
*
*	Project Name : S5PV210 FPGA Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for verifying functions of the S5PV210 FPGA
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------
* 
*	File Name : spi_test.c
*  
*	File Description : This file implements the API functons for SPI
*
*	Author	: Park,HeeSang
*	Dept. : AP Development Team
*	Created Date : 2009/08/31
*	Version : 0.3
*
*	Author	: Jongseok,Park
*	Dept. : AP Development Team
*	Created Date : 2009/04/16
*	Version : 0.2
*
*	Author : Sung-Hyun, Na
*	Dept. : AP Development Team
*	Created Date : 2008/11/03
*	Version : 0.1 
* 
*	History
*	- Version 0.1
*           ->Created(Sung-Hyun, Na 2008/11/03) for S5PC100  Validation
*	- Version 0.2
*           -> Modified(JongSeok,Park 2009/03/25) for S5PV210 FPGA verification
*	- Version 0.3
*           -> Modified(Park,Heesang 2009/08/31) for S5PV210 Validation.
**************************************************************************************/


#include <stdio.h>
#include "string.h"
#include "def.h"
#include "option.h"
#include "library.h"
#include "system.h"
#include "spi.h"
#include "sysc.h"
#include "timer.h"

#include "system.h"	/* Just refer to global variables such as g_uAPLL, g_uMPLL, etc. */

typedef struct {
	s32 (*func)(SPI_CtrlInfo_st *pCtrlInfo); 
	const char *desc;	
} SPI_TestFuncMenu;


s32 SPI_PollingTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_InterruptTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_DMATest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_Loopback_Polling_Test(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Loopback_Interrupt_Test(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Loopback_DMA_Test(SPI_CtrlInfo_st *pCtrlInfo);

void SPI_SetPLL(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Consol_AutoTest(SPI_CtrlInfo_st *pCtrlInfo);

const SPI_TestFuncMenu gSPI_Test_TopMenu[]= 
{
	SPI_SetPort0Info,				"Set Port0 Info",
	SPI_SetPort1Info,				"Set Port1 Info",
	SPI_SetPort2Info,				"Set Port2 Info\n",

	SPI_PollingTest,					"Polling Test",
	SPI_InterruptTest,				"Interrupt Test",
	SPI_DMATest,					"DMA Test\n",

	SPI_Loopback_Polling_Test,		"Loopback Polling Test",
	SPI_Loopback_Interrupt_Test,	"Loopback Interrupt Test",
	SPI_Loopback_DMA_Test,		"Loopback DMA Test\n",

	SPI_SFR_test,					"SFR Test",
	SPI_Consol_AutoTest,			"Consol Auto Test\n",

	SPI_SetPLL,						"Set PLL(APLL/MPLL/EPLL/VPLL)\n",

	NULL,							"Exit"
};




void SPI_Test(void)
{
	s32				MenuEntryCnt;
	s32				MenuIndx;
	s32				SelNum;
	s32				TestResult = 0;
	
	SPI_SetDefaultSystemEnv();		/* Set Clock, TZPC */
	SPI_InitDefaultValue(&gSPI_CtrlInfo);	/* Set Default Setting */
	SPI_Init(&gSPI_CtrlInfo);

	UART_Printf("\n\n====================================================");
	UART_Printf("\n   Welcome to SPI Test World!");
	UART_Printf("\n====================================================\n");

	{
		SPI_PortInfo_st	*pSPIPort0;
		SPI_PortInfo_st	*pSPIPort1;
		SPI_PortInfo_st	*pSPIPort2;

		pSPIPort0 = &gSPI_CtrlInfo.m_SPIPortDefaultInfo[SPI_PORT0];
		pSPIPort1 = &gSPI_CtrlInfo.m_SPIPortDefaultInfo[SPI_PORT1];
		pSPIPort2 = &gSPI_CtrlInfo.m_SPIPortDefaultInfo[SPI_PORT2];
	}

	MenuEntryCnt = sizeof(gSPI_Test_TopMenu)/sizeof(SPI_TestFuncMenu);
	while(1)
	{
		/* Display Top Menu */
		UART_Printf("\n====================================================");
		UART_Printf("\n   SPI Top Menu");
		UART_Printf("\n====================================================\n");
		for(MenuIndx = 0; MenuIndx<MenuEntryCnt; MenuIndx++)
		{   
	   		UART_Printf("%2d.%s", MenuIndx, gSPI_Test_TopMenu[MenuIndx].desc);
			UART_Printf("\n");
		}
		UART_Printf("\n====================================================");

		UART_Printf("\nSelect: ");
		SelNum = UART_GetIntNum();
		//if(SelNum <= 0 || SelNum >= MenuEntryCnt)
		if(SelNum == MenuEntryCnt - 1)
		{
			break;
		}

		if(SelNum >= 0 && SelNum < MenuEntryCnt - 1)
		{
			if(gSPI_Test_TopMenu[SelNum].func != NULL)
			{
				TestResult = (gSPI_Test_TopMenu[SelNum].func)(&gSPI_CtrlInfo);
			}
		}
	}
}


const SPI_TestFuncMenu gSPI_Test_PollingTestMenu[]= 
{
	SPI_SetPort0Info,					"Set Port0 Info",
	SPI_SetPort1Info,					"Set Port1 Info",
	SPI_SetPort2Info,					"Set Port2 Info\n",

	SPI_Port0_PollingMasterTxTest,		"Port0 Polling MasterTx Test",
	SPI_Port1_PollingMasterTxTest,		"Port1 Polling MasterTx Test",
	SPI_Port2_PollingMasterTxTest,		"Port2 Polling MasterTx Test\n",

	SPI_Port0_PollingMasterRxTest,		"Port0 Polling MasterRx Test",
	SPI_Port1_PollingMasterRxTest,		"Port1 Polling MasterRx Test",
	SPI_Port2_PollingMasterRxTest,		"Port2 Polling MasterRx Test\n",

	SPI_Port0_PollingSlaveTxTest,		"Port0 Polling SlaveTx Test",
	SPI_Port1_PollingSlaveTxTest,		"Port1 Polling SlaveTx Test",
	SPI_Port2_PollingSlaveTxTest,		"Port2 Polling SlaveTx Test\n",

	SPI_Port0_PollingSlaveRxTest,		"Port0 Polling SlaveRx Test",
	SPI_Port1_PollingSlaveRxTest,		"Port1 Polling SlaveRx Test",
	SPI_Port2_PollingSlaveRxTest,		"Port2 Polling SlaveRx Test\n",

	SPI_Port0_PollingMasterTxRxTest,	"Port0 Polling MasterTxRx Test",
	SPI_Port1_PollingMasterTxRxTest,	"Port1 Polling MasterTxRx Test",
	SPI_Port2_PollingMasterTxRxTest,	"Port2 Polling MasterTxRx Test\n",

	SPI_Port0_PollingSlaveTxRxTest,		"Port0 Polling SlaveTxRx Test",
	SPI_Port1_PollingSlaveTxRxTest,		"Port1 Polling SlaveTxRx Test",
	SPI_Port2_PollingSlaveTxRxTest,		"Port2 Polling SlaveTxRx Test\n",

	SPI_SetPLL,							"Set PLL(APLL/MPLL/EPLL/VPLL)\n",

	NULL,								"Exit",
};



s32 SPI_PollingTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	s32				MenuEntryCnt;
	s32				MenuIndx;
	s32				SelNum;
	s32				TestResult = 0;
	
	MenuEntryCnt = sizeof(gSPI_Test_PollingTestMenu)/sizeof(SPI_TestFuncMenu);
	while(1)
	{
		/* Display Top Menu */
		UART_Printf("\n====================================================");
		UART_Printf("\n   SPI Polling Test");
		UART_Printf("\n====================================================\n");
		for(MenuIndx = 0; MenuIndx<MenuEntryCnt; MenuIndx++)
		{   
	   		UART_Printf("%2d.%s", MenuIndx, gSPI_Test_PollingTestMenu[MenuIndx].desc);
			UART_Printf("\n");
		}
		UART_Printf("\n====================================================");

		UART_Printf("\nSelect: ");
		SelNum = UART_GetIntNum();
		//if(SelNum <= 0 || SelNum >= MenuEntryCnt)
		if(SelNum == MenuEntryCnt - 1)
		{
			break;
		}

		if(SelNum >= 0 && SelNum < MenuEntryCnt - 1)
		{
			if(gSPI_Test_PollingTestMenu[SelNum].func != NULL)
			{
				TestResult = (gSPI_Test_PollingTestMenu[SelNum].func)(pCtrlInfo);
			}
		}
	}
}



s32 SPI_PortA_PollingMasterTxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	s32					Status;

	pPortA->m_OpMode = SPI_MASTER;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = FALSE;


	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	SPI_PollingMasterTx(pPortA);


	SPI_FreeNCBuf(pTxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}



s32 SPI_Port0_PollingMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_PollingMasterTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_PollingMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_PollingMasterTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port2_PollingMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_PollingMasterTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_PortA_PollingMasterRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;

	pPortA->m_OpMode = SPI_MASTER;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = FALSE;
	pPortA->m_EnRxChannel = TRUE;


	SPI_OpenPort(pPortA);

	pRxBufInfo = &pPortA->m_RxBufNode;

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_PollingMasterRx(pPortA);

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port0_PollingMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_PollingMasterRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_PollingMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_PollingMasterRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port2_PollingMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_PollingMasterRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_PortA_PollingSlaveTxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	s32					Status;

	pPortA->m_OpMode = SPI_SLAVE;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = FALSE;


	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	SPI_PollingSlaveTx(pPortA);

	SPI_FreeNCBuf(pTxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port0_PollingSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_PollingSlaveTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_PollingSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_PollingSlaveTxTest(pPortA);

	return SPI_NO_ERROR;
}



s32 SPI_Port2_PollingSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_PollingSlaveTxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_PortA_PollingSlaveRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;

	pPortA->m_OpMode = SPI_SLAVE;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = FALSE;
	pPortA->m_EnRxChannel = TRUE;


	SPI_OpenPort(pPortA);

	pRxBufInfo = &pPortA->m_RxBufNode;

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_EnChannel(pPortA, pPortA->m_EnTxChannel, pPortA->m_EnRxChannel);
	SPI_PollingSlaveRx(pPortA);

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port0_PollingSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_PollingSlaveRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port1_PollingSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_PollingSlaveRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port2_PollingSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_PollingSlaveRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_PortA_PollingMasterTxRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;

	pPortA->m_OpMode = SPI_MASTER;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = TRUE;


	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;
	pRxBufInfo = &pPortA->m_RxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_PollingMasterTxRx(pPortA);

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}



s32 SPI_Port0_PollingMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_PollingMasterTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_PollingMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_PollingMasterTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port2_PollingMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_PollingMasterTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_PortA_PollingSlaveTxRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;

	pPortA->m_OpMode = SPI_SLAVE;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = TRUE;

	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;
	pRxBufInfo = &pPortA->m_RxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);
	
	SPI_PollingSlaveTx(pPortA);
	SPI_PollingSlaveRx(pPortA);

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port0_PollingSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_PollingSlaveTxRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port1_PollingSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_PollingSlaveTxRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port2_PollingSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_PollingSlaveTxRxTest(pPortA);

	return SPI_NO_ERROR;
}



const SPI_TestFuncMenu gSPI_Test_InterruptTestMenu[]= 
{
	SPI_SetPort0Info,					"Set Port0 Info",
	SPI_SetPort1Info,					"Set Port1 Info",
	SPI_SetPort2Info,					"Set Port2 Info\n",

	SPI_Port0_InterruptMasterTxTest,	"Port0 Interrupt MasterTx Test",
	SPI_Port1_InterruptMasterTxTest,	"Port1 Interrupt MasterTx Test",
	SPI_Port2_InterruptMasterTxTest,	"Port2 Interrupt MasterTx Test\n",

	SPI_Port0_InterruptMasterRxTest,	"Port0 Interrupt MasterRx Test",
	SPI_Port1_InterruptMasterRxTest,	"Port1 Interrupt MasterRx Test",
	SPI_Port2_InterruptMasterRxTest,	"Port2 Interrupt MasterRx Test\n",

	SPI_Port0_InterruptSlaveTxTest,		"Port0 Interrupt SlaveTx Test",
	SPI_Port1_InterruptSlaveTxTest,		"Port1 Interrupt SlaveTx Test",
	SPI_Port2_InterruptSlaveTxTest,		"Port2 Interrupt SlaveTx Test\n",

	SPI_Port0_InterruptSlaveRxTest,		"Port0 Interrupt SlaveRx Test",
	SPI_Port1_InterruptSlaveRxTest,		"Port1 Interrupt SlaveRx Test",
	SPI_Port2_InterruptSlaveRxTest,		"Port2 Interrupt SlaveRx Test\n",

	SPI_Port0_InterruptMasterTxRxTest,	"Port0 Interrupt MasterTxRx Test",
	SPI_Port1_InterruptMasterTxRxTest,	"Port1 Interrupt MasterTxRx Test",
	SPI_Port2_InterruptMasterTxRxTest,	"Port2 Interrupt MasterTxRx Test\n",

	SPI_Port0_InterruptSlaveTxRxTest,	"Port0 Interrupt SlaveTxRx Test",
	SPI_Port1_InterruptSlaveTxRxTest,	"Port1 Interrupt SlaveTxRx Test",
	SPI_Port2_InterruptSlaveTxRxTest,	"Port2 Interrupt SlaveTxRx Test\n",

	SPI_SetPLL,							"Set PLL(APLL/MPLL/EPLL/VPLL)\n",

	NULL,								"Exit",
};



s32 SPI_InterruptTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	s32				MenuEntryCnt;
	s32				MenuIndx;
	s32				SelNum;
	s32				TestResult = 0;
	
	MenuEntryCnt = sizeof(gSPI_Test_InterruptTestMenu)/sizeof(SPI_TestFuncMenu);
	while(1)
	{
		/* Display Top Menu */
		UART_Printf("\n====================================================");
		UART_Printf("\n   SPI Interrupt Test");
		UART_Printf("\n====================================================\n");
		for(MenuIndx = 0; MenuIndx<MenuEntryCnt; MenuIndx++)
		{   
	   		UART_Printf("%2d.%s", MenuIndx, gSPI_Test_InterruptTestMenu[MenuIndx].desc);
			UART_Printf("\n");
		}
		UART_Printf("\n====================================================");

		UART_Printf("\nSelect: ");
		SelNum = UART_GetIntNum();
		//if(SelNum <= 0 || SelNum >= MenuEntryCnt)
		if(SelNum == MenuEntryCnt - 1)
		{
			break;
		}

		if(SelNum >= 0 && SelNum < MenuEntryCnt - 1)
		{
			if(gSPI_Test_InterruptTestMenu[SelNum].func != NULL)
			{
				TestResult = (gSPI_Test_InterruptTestMenu[SelNum].func)(pCtrlInfo);
			}
		}
	}
}


s32 SPI_PortA_InterruptMasterTxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	
	pPortA->m_OpMode = SPI_MASTER;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = FALSE;


	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	SPI_InterruptMasterTx(pPortA, SPI_MasterTxCompleteCallback, (u32)&TxDone);

	while(!UART_GetKey())
	{
		if(TxDone)
		{
			break;
		}
	}

	SPI_FreeNCBuf(pTxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}



s32 SPI_Port0_InterruptMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_InterruptMasterTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_InterruptMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_InterruptMasterTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port2_InterruptMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_InterruptMasterTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_PortA_InterruptMasterRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					RxDone = FALSE;

	pPortA->m_OpMode = SPI_MASTER;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = FALSE;
	pPortA->m_EnRxChannel = TRUE;


	SPI_OpenPort(pPortA);

	pRxBufInfo = &pPortA->m_RxBufNode;

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_InterruptMasterRx(pPortA, SPI_MasterRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(RxDone)
		{
			break;
		}
	}

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port0_InterruptMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_InterruptMasterRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_InterruptMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_InterruptMasterRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port2_InterruptMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_InterruptMasterRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_PortA_InterruptSlaveTxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	
	pPortA->m_OpMode = SPI_SLAVE;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = FALSE;


	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	SPI_InterruptSlaveTx(pPortA, SPI_SlaveTxCompleteCallback, (u32)&TxDone);

	while(!UART_GetKey())
	{
		if(TxDone)
		{
			break;
		}
	}

	SPI_FreeNCBuf(pTxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port0_InterruptSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_InterruptSlaveTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_InterruptSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_InterruptSlaveTxTest(pPortA);

	return SPI_NO_ERROR;
}



s32 SPI_Port2_InterruptSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_InterruptSlaveTxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_PortA_InterruptSlaveRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					RxDone = FALSE;
	
	pPortA->m_OpMode = SPI_SLAVE;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = FALSE;
	pPortA->m_EnRxChannel = TRUE;


	SPI_OpenPort(pPortA);

	pRxBufInfo = &pPortA->m_RxBufNode;

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_InterruptSlaveRx(pPortA, SPI_SlaveRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(RxDone)
		{
			break;
		}
	}

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port0_InterruptSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_InterruptSlaveRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port1_InterruptSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_InterruptSlaveRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port2_InterruptSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_InterruptSlaveRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_PortA_InterruptMasterTxRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	bool					RxDone = FALSE;

	pPortA->m_OpMode = SPI_MASTER;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = TRUE;


	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;
	pRxBufInfo = &pPortA->m_RxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_InterruptMasterTxRx(pPortA, SPI_MasterTxCompleteCallback, (u32)&TxDone,
		SPI_MasterRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(TxDone && RxDone)
		{
			break;
		}
	}

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}



s32 SPI_Port0_InterruptMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_InterruptMasterTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_InterruptMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_InterruptMasterTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port2_InterruptMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_InterruptMasterTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_PortA_InterruptSlaveTxRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	bool					RxDone = FALSE;

	pPortA->m_OpMode = SPI_SLAVE;

	pPortA->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortA->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = TRUE;

	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;
	pRxBufInfo = &pPortA->m_RxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);
	
	SPI_InterruptSlaveTxRx(pPortA, SPI_SlaveTxCompleteCallback, (u32)&TxDone,
		SPI_SlaveRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(TxDone && RxDone)
		{
			break;
		}
	}

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port0_InterruptSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_InterruptSlaveTxRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port1_InterruptSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_InterruptSlaveTxRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port2_InterruptSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_InterruptSlaveTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


const SPI_TestFuncMenu gSPI_Test_DMATestMenu[]= 
{
	SPI_SetPort0Info,					"Set Port0 Info",
	SPI_SetPort1Info,					"Set Port1 Info",
	SPI_SetPort2Info,					"Set Port2 Info\n",

	SPI_Port0_DMAMasterTxTest,			"Port0 DMA MasterTx Test",
	SPI_Port1_DMAMasterTxTest,			"Port1 DMA MasterTx Test",
	SPI_Port2_DMAMasterTxTest,			"Port2 DMA MasterTx Test\n",

	SPI_Port0_DMAMasterRxTest,			"Port0 DMA MasterRx Test",
	SPI_Port1_DMAMasterRxTest,			"Port1 DMA MasterRx Test",
	SPI_Port2_DMAMasterRxTest,			"Port2 DMA MasterRx Test\n",

	SPI_Port0_DMASlaveTxTest,			"Port0 DMA SlaveTx Test",
	SPI_Port1_DMASlaveTxTest,			"Port1 DMA SlaveTx Test",
	SPI_Port2_DMASlaveTxTest,			"Port2 DMA SlaveTx Test\n",

	SPI_Port0_DMASlaveRxTest,			"Port0 DMA SlaveRx Test",
	SPI_Port1_DMASlaveRxTest,			"Port1 DMA SlaveRx Test",
	SPI_Port2_DMASlaveRxTest,			"Port2 DMA SlaveRx Test\n",

	SPI_Port0_DMAMasterTxRxTest,		"Port0 DMA MasterTxRx Test",
	SPI_Port1_DMAMasterTxRxTest,		"Port1 DMA MasterTxRx Test",
	SPI_Port2_DMAMasterTxRxTest,		"Port2 DMA MasterTxRx Test\n",

	SPI_Port0_DMASlaveTxRxTest,		"Port0 DMA SlaveTxRx Test",
	SPI_Port1_DMASlaveTxRxTest,		"Port1 DMA SlaveTxRx Test",
	SPI_Port2_DMASlaveTxRxTest,		"Port2 DMA SlaveTxRx Test\n",

	SPI_SetPLL,							"Set PLL(APLL/MPLL/EPLL/VPLL)\n",

	NULL,								"Exit",
};



s32 SPI_DMATest(SPI_CtrlInfo_st *pCtrlInfo)
{
	s32				MenuEntryCnt;
	s32				MenuIndx;
	s32				SelNum;
	s32				TestResult = 0;
	
	MenuEntryCnt = sizeof(gSPI_Test_DMATestMenu)/sizeof(SPI_TestFuncMenu);
	while(1)
	{
		/* Display Top Menu */
		UART_Printf("\n====================================================");
		UART_Printf("\n   SPI DMA Test");
		UART_Printf("\n====================================================\n");
		for(MenuIndx = 0; MenuIndx<MenuEntryCnt; MenuIndx++)
		{   
	   		UART_Printf("%2d.%s", MenuIndx, gSPI_Test_DMATestMenu[MenuIndx].desc);
			UART_Printf("\n");
		}
		UART_Printf("\n====================================================");

		UART_Printf("\nSelect: ");
		SelNum = UART_GetIntNum();
		//if(SelNum <= 0 || SelNum >= MenuEntryCnt)
		if(SelNum == MenuEntryCnt - 1)
		{
			break;
		}

		if(SelNum >= 0 && SelNum < MenuEntryCnt - 1)
		{
			if(gSPI_Test_DMATestMenu[SelNum].func != NULL)
			{
				TestResult = (gSPI_Test_DMATestMenu[SelNum].func)(pCtrlInfo);
			}
		}
	}
}


s32 SPI_PortA_DMAMasterTxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	
	pPortA->m_OpMode = SPI_MASTER;

	pPortA->m_RxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */
	pPortA->m_TxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = FALSE;


	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	SPI_DmaMasterTx(pPortA, SPI_MasterTxCompleteCallback, (u32)&TxDone);

	while(!UART_GetKey())
	{
		if(TxDone)
		{
			break;
		}
	}

	SPI_FreeNCBuf(pTxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}



s32 SPI_Port0_DMAMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_DMAMasterTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_DMAMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_DMAMasterTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port2_DMAMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_DMAMasterTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_PortA_DMAMasterRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					RxDone = FALSE;

	pPortA->m_OpMode = SPI_MASTER;

	pPortA->m_RxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */
	pPortA->m_TxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */

	pPortA->m_EnTxChannel = FALSE;
	pPortA->m_EnRxChannel = TRUE;


	SPI_OpenPort(pPortA);

	pRxBufInfo = &pPortA->m_RxBufNode;

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_DmaMasterRx(pPortA, SPI_MasterRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(RxDone)
		{
			break;
		}
	}

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port0_DMAMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_DMAMasterRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_DMAMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_DMAMasterRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port2_DMAMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_DMAMasterRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_PortA_DMASlaveTxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	
	pPortA->m_OpMode = SPI_SLAVE;

	pPortA->m_RxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */
	pPortA->m_TxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = FALSE;


	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	SPI_DmaSlaveTx(pPortA, SPI_SlaveTxCompleteCallback, (u32)&TxDone);

	while(!UART_GetKey())
	{
		if(TxDone)
		{
			break;
		}
	}

	SPI_FreeNCBuf(pTxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port0_DMASlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_DMASlaveTxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_DMASlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_DMASlaveTxTest(pPortA);

	return SPI_NO_ERROR;
}



s32 SPI_Port2_DMASlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_DMASlaveTxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_PortA_DMASlaveRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					RxDone = FALSE;
	
	pPortA->m_OpMode = SPI_SLAVE;

	pPortA->m_RxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */
	pPortA->m_TxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */

	pPortA->m_EnTxChannel = FALSE;
	pPortA->m_EnRxChannel = TRUE;


	SPI_OpenPort(pPortA);

	pRxBufInfo = &pPortA->m_RxBufNode;

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_DmaSlaveRx(pPortA, SPI_SlaveRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(RxDone)
		{
			break;
		}
	}

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port0_DMASlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_DMASlaveRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port1_DMASlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_DMASlaveRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port2_DMASlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_DMASlaveRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_PortA_DMAMasterTxRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	bool					RxDone = FALSE;

	pPortA->m_OpMode = SPI_MASTER;

	pPortA->m_RxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */
	pPortA->m_TxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = TRUE;


	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;
	pRxBufInfo = &pPortA->m_RxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_DmaMasterTxRx(pPortA, SPI_MasterTxCompleteCallback, (u32)&TxDone,
		SPI_MasterRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(TxDone && RxDone)
		{
			break;
		}
	}

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}



s32 SPI_Port0_DMAMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_DMAMasterTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port1_DMAMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_DMAMasterTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port2_DMAMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_DMAMasterTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_PortA_DMASlaveTxRxTest(SPI_PortInfo_st *pPortA)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	bool					RxDone = FALSE;

	pPortA->m_OpMode = SPI_SLAVE;

	pPortA->m_RxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */
	pPortA->m_TxTransferMode = SPI_DMA_MODE;		/* Set DMA Mode */

	pPortA->m_EnTxChannel = TRUE;
	pPortA->m_EnRxChannel = TRUE;

	SPI_OpenPort(pPortA);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pPortA->m_TxBufNode;
	pRxBufInfo = &pPortA->m_RxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pPortA->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufRandInit(pTxBufInfo, pPortA->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);

	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pPortA->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);
	
	SPI_DmaSlaveTxRx(pPortA, SPI_SlaveTxCompleteCallback, (u32)&TxDone,
		SPI_SlaveRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(TxDone && RxDone)
		{
			break;
		}
	}

	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pPortA);

	return SPI_NO_ERROR;
}


s32 SPI_Port0_DMASlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];

	Status = SPI_PortA_DMASlaveTxRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port1_DMASlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_PortA_DMASlaveTxRxTest(pPortA);

	return SPI_NO_ERROR;
}

s32 SPI_Port2_DMASlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortA;
	s32					Status;

	pPortA = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_PortA_DMASlaveTxRxTest(pPortA);

	return SPI_NO_ERROR;
}


const SPI_TestFuncMenu gSPI_Test_Loopback_PollingTest_Menu[]= 
{
	SPI_SetPort0Info,						"Set Port0 Info",
	SPI_SetPort1Info,						"Set Port1 Info",
	SPI_SetPort2Info,						"Set Port2 Info\n",

	SPI_Loopback_Port0Port1PollingTest,		"Loopback Polling Test : Port0 - Port1",
	SPI_Loopback_Port0Port2PollingTest,		"Loopback Polling Test : Port0 - Port2",
	SPI_Loopback_Port1Port2PollingTest,		"Loopback Polling Test : Port1 - Port2\n",

	SPI_SetPLL,								"Set PLL(APLL/MPLL/EPLL/VPLL)\n",

	NULL,									"Exit",
};

s32 SPI_Loopback_Polling_Test(SPI_CtrlInfo_st *pCtrlInfo)
{
	s32				MenuEntryCnt;
	s32				MenuIndx;
	s32				SelNum;
	s32				TestResult = 0;
	
	MenuEntryCnt = sizeof(gSPI_Test_Loopback_PollingTest_Menu)/sizeof(SPI_TestFuncMenu);
	while(1)
	{
		/* Display Top Menu */
		UART_Printf("\n====================================================");
		UART_Printf("\n   SPI Loopback Polling Test");
		UART_Printf("\n====================================================\n");
		for(MenuIndx = 0; MenuIndx<MenuEntryCnt; MenuIndx++)
		{   
	   		UART_Printf("%2d.%s", MenuIndx, gSPI_Test_Loopback_PollingTest_Menu[MenuIndx].desc);
			UART_Printf("\n");
		}
		UART_Printf("\n====================================================");

		UART_Printf("\nSelect: ");
		SelNum = UART_GetIntNum();
		//if(SelNum <= 0 || SelNum >= MenuEntryCnt)
		if(SelNum == MenuEntryCnt - 1)
		{
			break;
		}

		if(SelNum >= 0 && SelNum < MenuEntryCnt - 1)
		{
			if(gSPI_Test_Loopback_PollingTest_Menu[SelNum].func != NULL)
			{
				TestResult = (gSPI_Test_Loopback_PollingTest_Menu[SelNum].func)(pCtrlInfo);
			}
		}
	}
}


const SPI_TestFuncMenu gSPI_Test_Loopback_InterruptTest_Menu[]= 
{
	SPI_SetPort0Info,							"Set Port0 Info",
	SPI_SetPort1Info,							"Set Port1 Info",
	SPI_SetPort2Info,							"Set Port2 Info\n",

	SPI_Loopback_Port0Port1InterruptTest,		"Loopback Interrupt Test : Port0 - Port1",
	SPI_Loopback_Port0Port2InterruptTest,		"Loopback Interrupt Test : Port0 - Port2",
	SPI_Loopback_Port1Port2InterruptTest,		"Loopback Interrupt Test : Port1 - Port2\n",

	SPI_SetPLL,									"Set PLL(APLL/MPLL/EPLL/VPLL)\n",

	NULL,										"Exit",
};
s32 SPI_Loopback_Interrupt_Test(SPI_CtrlInfo_st *pCtrlInfo)
{
	s32				MenuEntryCnt;
	s32				MenuIndx;
	s32				SelNum;
	s32				TestResult = 0;
	
	MenuEntryCnt = sizeof(gSPI_Test_Loopback_InterruptTest_Menu)/sizeof(SPI_TestFuncMenu);
	while(1)
	{
		/* Display Top Menu */
		UART_Printf("\n====================================================");
		UART_Printf("\n   SPI Loopback Interrupt Test");
		UART_Printf("\n====================================================\n");
		for(MenuIndx = 0; MenuIndx<MenuEntryCnt; MenuIndx++)
		{   
	   		UART_Printf("%2d.%s", MenuIndx, gSPI_Test_Loopback_InterruptTest_Menu[MenuIndx].desc);
			UART_Printf("\n");
		}
		UART_Printf("\n====================================================");

		UART_Printf("\nSelect: ");
		SelNum = UART_GetIntNum();
		//if(SelNum <= 0 || SelNum >= MenuEntryCnt)
		if(SelNum == MenuEntryCnt - 1)
		{
			break;
		}

		if(SelNum >= 0 && SelNum < MenuEntryCnt - 1)
		{
			if(gSPI_Test_Loopback_InterruptTest_Menu[SelNum].func != NULL)
			{
				TestResult = (gSPI_Test_Loopback_InterruptTest_Menu[SelNum].func)(pCtrlInfo);
			}
		}
	}
}

const SPI_TestFuncMenu gSPI_Test_Loopback_DMATest_Menu[]= 
{
	SPI_SetPort0Info,						"Set Port0 Info",
	SPI_SetPort1Info,						"Set Port1 Info",
	SPI_SetPort2Info,						"Set Port2 Info\n",

	SPI_Loopback_Port0Port1DMATest,		"Loopback DMA Test : Port0 - Port1",
	SPI_Loopback_Port0Port2DMATest,		"Loopback DMA Test : Port0 - Port2",
	SPI_Loopback_Port1Port2DMATest,		"Loopback DMA Test : Port1 - Port2\n",

	SPI_SetPLL,								"Set PLL(APLL/MPLL/EPLL/VPLL)\n",

	NULL,									"Exit",
};
s32 SPI_Loopback_DMA_Test(SPI_CtrlInfo_st *pCtrlInfo)
{
	s32				MenuEntryCnt;
	s32				MenuIndx;
	s32				SelNum;
	s32				TestResult = 0;
	
	MenuEntryCnt = sizeof(gSPI_Test_Loopback_DMATest_Menu)/sizeof(SPI_TestFuncMenu);
	while(1)
	{
		/* Display Top Menu */
		UART_Printf("\n====================================================");
		UART_Printf("\n   SPI Loopback DMA Test");
		UART_Printf("\n====================================================\n");
		for(MenuIndx = 0; MenuIndx<MenuEntryCnt; MenuIndx++)
		{   
	   		UART_Printf("%2d.%s", MenuIndx, gSPI_Test_Loopback_DMATest_Menu[MenuIndx].desc);
			UART_Printf("\n");
		}
		UART_Printf("\n====================================================");

		UART_Printf("\nSelect: ");
		SelNum = UART_GetIntNum();
		//if(SelNum <= 0 || SelNum >= MenuEntryCnt)
		if(SelNum == MenuEntryCnt - 1)
		{
			break;
		}

		if(SelNum >= 0 && SelNum < MenuEntryCnt - 1)
		{
			if(gSPI_Test_Loopback_DMATest_Menu[SelNum].func != NULL)
			{
				TestResult = (gSPI_Test_Loopback_DMATest_Menu[SelNum].func)(pCtrlInfo);
			}
		}
	}
}


s32 SPI_Loopback_PollingMasterTxSlaveRxTest(SPI_PortInfo_st *pMaster, SPI_PortInfo_st *pSlave)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;

	if( !(pMaster->m_EnTxChannel == TRUE && pMaster->m_EnRxChannel != TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pMaster);
		SPI_DisplayVar_EnRxChannel(pMaster);
		
		return SPI_ERROR;
	}

	if( !(pSlave->m_EnTxChannel != TRUE && pSlave->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pSlave);
		SPI_DisplayVar_EnRxChannel(pSlave);

		
		return SPI_ERROR;
	}

	SPI_OpenPort(pMaster);
	SPI_OpenPort(pSlave);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pMaster->m_TxBufNode;
	pRxBufInfo = &pSlave->m_RxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pMaster->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pMaster->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pSlave->m_RxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	/* Tx Data size should be less than Tx FIFO size in polling mode. */
	SPI_EnChannel(pSlave, pSlave->m_EnTxChannel, pSlave->m_EnRxChannel);
	SPI_PollingMasterTx(pMaster);
	SPI_PollingSlaveRx(pSlave);

	Status = SPI_CompareData(pTxBufInfo->m_pBufPtr, pRxBufInfo->m_pBufPtr, pTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pMaster);
	SPI_ClosePort(pSlave);

	return Status;
}


s32 SPI_Loopback_PollingMasterRxSlaveTxTest(SPI_PortInfo_st *pMaster, SPI_PortInfo_st *pSlave)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;

	if( !(pMaster->m_EnTxChannel != TRUE && pMaster->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pMaster);
		SPI_DisplayVar_EnRxChannel(pMaster);
		
		return SPI_ERROR;
	}

	if( !(pSlave->m_EnTxChannel == TRUE && pSlave->m_EnRxChannel != TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pSlave);
		SPI_DisplayVar_EnRxChannel(pSlave);
		
		return SPI_ERROR;
	}

	SPI_OpenPort(pMaster);
	SPI_OpenPort(pSlave);

	/* Prepare buffer for Master Tx and Slave Rx */
	pRxBufInfo = &pMaster->m_RxBufNode;
	pTxBufInfo = &pSlave->m_TxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pSlave->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pSlave->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pMaster->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	/* Tx Data size should be less than Tx FIFO size in polling mode. */
	SPI_PollingSlaveTx(pSlave);
	SPI_PollingMasterRx(pMaster);

	Status = SPI_CompareData(pTxBufInfo->m_pBufPtr, pRxBufInfo->m_pBufPtr, pTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pMaster);
	SPI_ClosePort(pSlave);

	return Status;
}


s32 SPI_Loopback_PollingMasterTxRxSlaveTxRxTest(SPI_PortInfo_st *pMaster, SPI_PortInfo_st *pSlave)
{
	SPI_BufAllocInfo_st	*pMasterTxBufInfo;
	SPI_BufAllocInfo_st	*pMasterRxBufInfo;
	SPI_BufAllocInfo_st	*pSlaveTxBufInfo;
	SPI_BufAllocInfo_st	*pSlaveRxBufInfo;
	s32					Status;
	s32					Compare1;
	s32					Compare2;

	if( !(pMaster->m_EnTxChannel == TRUE && pMaster->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		//SPI_DisplayPortInfo_Channel(pMaster);
		SPI_DisplayVar_EnTxChannel(pMaster);
		SPI_DisplayVar_EnRxChannel(pMaster);
		
		return SPI_ERROR;
	}

	if( !(pSlave->m_EnTxChannel == TRUE && pSlave->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		//SPI_DisplayPortInfo_Channel(pSlave);
		SPI_DisplayVar_EnTxChannel(pSlave);
		SPI_DisplayVar_EnRxChannel(pSlave);
		
		return SPI_ERROR;
	}

	SPI_OpenPort(pMaster);
	SPI_OpenPort(pSlave);

	/* Prepare buffer for Master Tx and Slave Rx */
	pMasterTxBufInfo = &pMaster->m_TxBufNode;
	pMasterRxBufInfo = &pMaster->m_RxBufNode;

	pMasterTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pMasterTxBufInfo, pMaster->m_TxTransferSize);
	if(pMasterTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pMasterTxBufInfo, pMaster->m_TxTransferSize);
	SPI_NCBufDisplayHex(pMasterTxBufInfo, TRUE);
	
	pMasterRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pMasterRxBufInfo, pMaster->m_RxTransferSize);
	if(pMasterRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pMasterRxBufInfo);

	pSlaveTxBufInfo = &pSlave->m_TxBufNode;
	pSlaveRxBufInfo = &pSlave->m_RxBufNode;

	pSlaveTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pSlaveTxBufInfo, pSlave->m_TxTransferSize);
	if(pSlaveTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pSlaveTxBufInfo, pSlave->m_TxTransferSize);
	SPI_NCBufDisplayHex(pSlaveTxBufInfo, TRUE);
	
	pSlaveRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pSlaveRxBufInfo, pSlave->m_RxTransferSize);
	if(pSlaveRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pSlaveRxBufInfo);


	/* Tx Data size should be less than Tx FIFO size in polling mode. */
	SPI_PollingSlaveTx(pSlave);
	SPI_PollingMasterTxRx(pMaster);
	SPI_PollingSlaveRx(pSlave);

	Compare1 = SPI_CompareData(pMasterTxBufInfo->m_pBufPtr, pSlaveRxBufInfo->m_pBufPtr, pMasterTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pSlaveRxBufInfo, TRUE);

	Compare2 = SPI_CompareData(pSlaveTxBufInfo->m_pBufPtr, pMasterRxBufInfo->m_pBufPtr, pSlaveTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pMasterRxBufInfo, TRUE);

	SPI_FreeNCBuf(pMasterTxBufInfo);
	SPI_FreeNCBuf(pMasterRxBufInfo);
	SPI_FreeNCBuf(pSlaveTxBufInfo);
	SPI_FreeNCBuf(pSlaveRxBufInfo);

	SPI_ClosePort(pMaster);
	SPI_ClosePort(pSlave);

	if(Compare1 != SPI_NO_ERROR || Compare2 != SPI_NO_ERROR)
	{
		return SPI_ERROR;
	}
	else
	{
		return SPI_NO_ERROR;
	}
}



s32 SPI_Loopback_PortAPortBPollingTest(SPI_PortInfo_st *pPortAInfo, SPI_PortInfo_st *pPortBInfo)
{
	SPI_PortInfo_st		*pMasterPort;
	SPI_PortInfo_st		*pSlavePort;
	SPI_PortInfo_st		*pTxPort;
	SPI_PortInfo_st		*pRxPort;
	s32					Status;

	Status = SPI_CheckLoopbackMasterAndChannel(pPortAInfo, pPortBInfo, &pMasterPort, &pSlavePort, &pTxPort, &pRxPort);
	if(Status != SPI_NO_ERROR)
	{
		return Status;
	}

	pPortAInfo->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortAInfo->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortBInfo->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrrupt Mode */
	pPortBInfo->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrrupt Mode */

	if(pMasterPort->m_EnTxChannel == TRUE)
	{
		if(pMasterPort->m_EnRxChannel == TRUE)
		{
			SPI_Loopback_PollingMasterTxRxSlaveTxRxTest(pMasterPort, pSlavePort);
		}
		else
		{
			SPI_Loopback_PollingMasterTxSlaveRxTest(pMasterPort, pSlavePort);
		}
	}
	else
	{
		if(pMasterPort->m_EnRxChannel == TRUE)
		{
			SPI_Loopback_PollingMasterRxSlaveTxTest(pMasterPort, pSlavePort);
		}
		else
		{
			SPI_ASSERT();
		}
	}
	
	return SPI_NO_ERROR;
}


s32 SPI_Loopback_Port0Port1PollingTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	pPortAInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];
	pPortBInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_Loopback_PortAPortBPollingTest(pPortAInfo, pPortBInfo);

	return Status;
}

s32 SPI_Loopback_Port0Port2PollingTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	pPortAInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];
	pPortBInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_Loopback_PortAPortBPollingTest(pPortAInfo, pPortBInfo);

	return Status;
}

s32 SPI_Loopback_Port1Port2PollingTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	pPortAInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];
	pPortBInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_Loopback_PortAPortBPollingTest(pPortAInfo, pPortBInfo);

	return Status;
}

void SPI_MasterRxCompleteCallback(u32 var0)
{
	bool *pComplete = (bool *)var0;

	(*pComplete) = TRUE;

	if(SPI_IsInterruptPrintValid())
	{
		UART_Printf("Master RxComplete\n");
	}
}

void SPI_MasterTxCompleteCallback(u32 var0)
{
	bool *pComplete = (bool *)var0;

	(*pComplete) = TRUE;

	if(SPI_IsInterruptPrintValid())
	{
		UART_Printf("Master TxComplete\n");
	}
}

void SPI_SlaveRxCompleteCallback(u32 var0)
{
	bool *pComplete = (bool *)var0;

	(*pComplete) = TRUE;

	if(SPI_IsInterruptPrintValid())
	{
		UART_Printf("Slave RxComplete\n");
	}
}

void SPI_SlaveTxCompleteCallback(u32 var0)
{
	bool *pComplete = (bool *)var0;

	(*pComplete) = TRUE;

	if(SPI_IsInterruptPrintValid())
	{
		UART_Printf("Slave TxComplete\n");
	}
}

s32 SPI_Loopback_InterruptMasterTxSlaveRxTest(SPI_PortInfo_st *pMaster, SPI_PortInfo_st *pSlave)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	bool					RxDone = FALSE;

	if( !(pMaster->m_EnTxChannel == TRUE && pMaster->m_EnRxChannel != TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pMaster);
		SPI_DisplayVar_EnRxChannel(pMaster);
		
		return SPI_ERROR;
	}

	if( !(pSlave->m_EnTxChannel != TRUE && pSlave->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pSlave);
		SPI_DisplayVar_EnRxChannel(pSlave);
		
		return SPI_ERROR;
	}

	SPI_OpenPort(pMaster);
	SPI_OpenPort(pSlave);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pMaster->m_TxBufNode;
	pRxBufInfo = &pSlave->m_RxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pMaster->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pMaster->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pSlave->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_InterruptSlaveRx(pSlave, SPI_SlaveRxCompleteCallback, (u32)&RxDone);
	SPI_InterruptMasterTx(pMaster, SPI_MasterTxCompleteCallback, (u32)&TxDone);

	while(!UART_GetKey())
	{
		if(TxDone && RxDone)
		{
			break;
		}
	}

	Status = SPI_CompareData(pTxBufInfo->m_pBufPtr, pRxBufInfo->m_pBufPtr, pTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pMaster);
	SPI_ClosePort(pSlave);

	return Status;
}


s32 SPI_Loopback_InterruptMasterRxSlaveTxTest(SPI_PortInfo_st *pMaster, SPI_PortInfo_st *pSlave)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	bool					RxDone = FALSE;

	if( !(pMaster->m_EnTxChannel != TRUE && pMaster->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pMaster);
		SPI_DisplayVar_EnRxChannel(pMaster);
		
		return SPI_ERROR;
	}

	if( !(pSlave->m_EnTxChannel == TRUE && pSlave->m_EnRxChannel != TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pSlave);
		SPI_DisplayVar_EnRxChannel(pSlave);
		
		return SPI_ERROR;
	}

	SPI_OpenPort(pMaster);
	SPI_OpenPort(pSlave);

	/* Prepare buffer for Master Tx and Slave Rx */
	pRxBufInfo = &pMaster->m_RxBufNode;
	pTxBufInfo = &pSlave->m_TxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pSlave->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pSlave->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pMaster->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	/* Tx Data size should be less than Tx FIFO size in polling mode. */
	SPI_InterruptSlaveTx(pSlave, SPI_SlaveTxCompleteCallback, (u32)&TxDone);
	SPI_InterruptMasterRx(pMaster, SPI_MasterRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(TxDone && RxDone)
		{
			break;
		}
	}

	Status = SPI_CompareData(pTxBufInfo->m_pBufPtr, pRxBufInfo->m_pBufPtr, pTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pMaster);
	SPI_ClosePort(pSlave);

	return Status;
}


s32 SPI_Loopback_InterruptMasterTxRxSlaveTxRxTest(SPI_PortInfo_st *pMaster, SPI_PortInfo_st *pSlave)
{
	SPI_BufAllocInfo_st	*pMasterTxBufInfo;
	SPI_BufAllocInfo_st	*pMasterRxBufInfo;
	SPI_BufAllocInfo_st	*pSlaveTxBufInfo;
	SPI_BufAllocInfo_st	*pSlaveRxBufInfo;
	s32					Status;
	bool					SlaveTxDone = FALSE;
	bool					SlaveRxDone = FALSE;
	bool					MasterTxDone = FALSE;
	bool					MasterRxDone = FALSE;
	s32					Compare1;
	s32					Compare2;

	if( !(pMaster->m_EnTxChannel == TRUE && pMaster->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pMaster);
		SPI_DisplayVar_EnRxChannel(pMaster);
		
		return SPI_ERROR;
	}

	if( !(pSlave->m_EnTxChannel == TRUE && pSlave->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pSlave);
		SPI_DisplayVar_EnRxChannel(pSlave);
		
		return SPI_ERROR;
	}

	SPI_OpenPort(pMaster);
	SPI_OpenPort(pSlave);

	/* Prepare buffer for Master Tx and Slave Rx */
	pMasterTxBufInfo = &pMaster->m_TxBufNode;
	pMasterRxBufInfo = &pMaster->m_RxBufNode;

	pMasterTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pMasterTxBufInfo, pMaster->m_TxTransferSize);
	if(pMasterTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pMasterTxBufInfo, pMaster->m_TxTransferSize);
	SPI_NCBufDisplayHex(pMasterTxBufInfo, TRUE);
	
	pMasterRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pMasterRxBufInfo, pMaster->m_RxTransferSize);
	if(pMasterRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pMasterRxBufInfo);

	pSlaveTxBufInfo = &pSlave->m_TxBufNode;
	pSlaveRxBufInfo = &pSlave->m_RxBufNode;

	pSlaveTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pSlaveTxBufInfo, pSlave->m_TxTransferSize);
	if(pSlaveTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pSlaveTxBufInfo, pSlave->m_TxTransferSize);
	SPI_NCBufDisplayHex(pSlaveTxBufInfo, TRUE);
	
	pSlaveRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pSlaveRxBufInfo, pSlave->m_RxTransferSize);
	if(pSlaveRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pSlaveRxBufInfo);

	/* Tx Data size should be less than Tx FIFO size in polling mode. */
	SPI_InterruptSlaveTxRx(pSlave, SPI_SlaveTxCompleteCallback, (u32)&SlaveTxDone,
		SPI_SlaveRxCompleteCallback, (u32)&SlaveRxDone);
	SPI_InterruptMasterTxRx(pMaster, SPI_MasterTxCompleteCallback, (u32)&MasterTxDone,
		SPI_MasterRxCompleteCallback, (u32)&MasterRxDone);

	while(!UART_GetKey())
	{
		if(SlaveTxDone && SlaveRxDone && MasterTxDone && MasterRxDone)
		{
			break;
		}
	}

	Compare1 = SPI_CompareData(pSlaveTxBufInfo->m_pBufPtr, pMasterRxBufInfo->m_pBufPtr, pSlaveTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pMasterRxBufInfo, TRUE);

	Compare2 = SPI_CompareData(pMasterTxBufInfo->m_pBufPtr, pSlaveRxBufInfo->m_pBufPtr, pMasterTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pSlaveRxBufInfo, TRUE);

	SPI_FreeNCBuf(pMasterTxBufInfo);
	SPI_FreeNCBuf(pMasterRxBufInfo);
	SPI_FreeNCBuf(pSlaveTxBufInfo);
	SPI_FreeNCBuf(pSlaveRxBufInfo);

	SPI_ClosePort(pMaster);
	SPI_ClosePort(pSlave);

	if(Compare1 != SPI_NO_ERROR || Compare2 != SPI_NO_ERROR)
	{
		return SPI_ERROR;
	}
	else
	{
		return SPI_NO_ERROR;
	}
}


s32 SPI_Loopback_PortAPortBInterruptTest(SPI_PortInfo_st *pPortAInfo, SPI_PortInfo_st *pPortBInfo)
{
	SPI_PortInfo_st		*pMasterPort;
	SPI_PortInfo_st		*pSlavePort;
	SPI_PortInfo_st		*pTxPort;
	SPI_PortInfo_st		*pRxPort;
	s32					Status;

	Status = SPI_CheckLoopbackMasterAndChannel(pPortAInfo, pPortBInfo, &pMasterPort, &pSlavePort, &pTxPort, &pRxPort);
	if(Status != SPI_NO_ERROR)
	{
		return Status;
	}

	pPortAInfo->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortAInfo->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrupt Mode */
	pPortBInfo->m_RxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrrupt Mode */
	pPortBInfo->m_TxTransferMode = SPI_INTERRUPT_MODE;		/* Set Interrrupt Mode */

	if(pMasterPort->m_EnTxChannel == TRUE)
	{
		if(pMasterPort->m_EnRxChannel == TRUE)
		{
			SPI_Loopback_InterruptMasterTxRxSlaveTxRxTest(pMasterPort, pSlavePort);
		}
		else
		{
			SPI_Loopback_InterruptMasterTxSlaveRxTest(pMasterPort, pSlavePort);
		}
	}
	else
	{
		if(pMasterPort->m_EnRxChannel == TRUE)
		{
			SPI_Loopback_InterruptMasterRxSlaveTxTest(pMasterPort, pSlavePort);
		}
		else
		{
			SPI_ASSERT();
		}
	}

	return SPI_NO_ERROR;
}


s32 SPI_Loopback_Port0Port1InterruptTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	pPortAInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];
	pPortBInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_Loopback_PortAPortBInterruptTest(pPortAInfo, pPortBInfo);

	return Status;
}

s32 SPI_Loopback_Port0Port2InterruptTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	pPortAInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];
	pPortBInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_Loopback_PortAPortBInterruptTest(pPortAInfo, pPortBInfo);

	return Status;
}

s32 SPI_Loopback_Port1Port2InterruptTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	pPortAInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];
	pPortBInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_Loopback_PortAPortBInterruptTest(pPortAInfo, pPortBInfo);

	return Status;
}


s32 SPI_Loopback_DmaMasterTxSlaveRxTest(SPI_PortInfo_st *pMaster, SPI_PortInfo_st *pSlave)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	bool					RxDone = FALSE;

	if( !(pMaster->m_EnTxChannel == TRUE && pMaster->m_EnRxChannel != TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pMaster);
		SPI_DisplayVar_EnRxChannel(pMaster);
		
		return SPI_ERROR;
	}

	if( !(pSlave->m_EnTxChannel != TRUE && pSlave->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pSlave);
		SPI_DisplayVar_EnRxChannel(pSlave);
		
		return SPI_ERROR;
	}

	SPI_OpenPort(pMaster);
	SPI_OpenPort(pSlave);

	/* Prepare buffer for Master Tx and Slave Rx */
	pTxBufInfo = &pMaster->m_TxBufNode;
	pRxBufInfo = &pSlave->m_RxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pMaster->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pMaster->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pSlave->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	SPI_DmaSlaveRx(pSlave, SPI_SlaveRxCompleteCallback, (u32)&RxDone);
	SPI_DmaMasterTx(pMaster, SPI_MasterTxCompleteCallback, (u32)&TxDone);

	while(!UART_GetKey())
	{
		if(TxDone && RxDone)
		{
			break;
		}
	}

	Status = SPI_CompareData(pTxBufInfo->m_pBufPtr, pRxBufInfo->m_pBufPtr, pTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pMaster);
	SPI_ClosePort(pSlave);

	return Status;
}


s32 SPI_Loopback_DmaMasterRxSlaveTxTest(SPI_PortInfo_st *pMaster, SPI_PortInfo_st *pSlave)
{
	SPI_BufAllocInfo_st	*pTxBufInfo;
	SPI_BufAllocInfo_st	*pRxBufInfo;
	s32					Status;
	bool					TxDone = FALSE;
	bool					RxDone = FALSE;

	if( !(pMaster->m_EnTxChannel != TRUE && pMaster->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pMaster);
		SPI_DisplayVar_EnRxChannel(pMaster);
		
		return SPI_ERROR;
	}

	if( !(pSlave->m_EnTxChannel == TRUE && pSlave->m_EnRxChannel != TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pSlave);
		SPI_DisplayVar_EnRxChannel(pSlave);
		
		return SPI_ERROR;
	}

	SPI_OpenPort(pMaster);
	SPI_OpenPort(pSlave);

	/* Prepare buffer for Master Tx and Slave Rx */
	pRxBufInfo = &pMaster->m_RxBufNode;
	pTxBufInfo = &pSlave->m_TxBufNode;

	pTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pTxBufInfo, pSlave->m_TxTransferSize);
	if(pTxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}

	SPI_NCBufRandInit(pTxBufInfo, pSlave->m_TxTransferSize);
	SPI_NCBufDisplayHex(pTxBufInfo, TRUE);
	
	pRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pRxBufInfo, pMaster->m_RxTransferSize);
	if(pRxBufInfo->m_pBufPtr == NULL)
	{
		SPI_ASSERT();
	}
	SPI_NCBufZeroInit(pRxBufInfo);

	/* Tx Data size should be less than Tx FIFO size in polling mode. */
	SPI_DmaSlaveTx(pSlave, SPI_SlaveTxCompleteCallback, (u32)&TxDone);
	SPI_DmaMasterRx(pMaster, SPI_MasterRxCompleteCallback, (u32)&RxDone);

	while(!UART_GetKey())
	{
		if(TxDone && RxDone)
		{
			break;
		}
	}

	Status = SPI_CompareData(pTxBufInfo->m_pBufPtr, pRxBufInfo->m_pBufPtr, pTxBufInfo->m_ReqDataSize);
	SPI_NCBufDisplayHex(pRxBufInfo, TRUE);

	SPI_FreeNCBuf(pTxBufInfo);
	SPI_FreeNCBuf(pRxBufInfo);

	SPI_ClosePort(pMaster);
	SPI_ClosePort(pSlave);

	return Status;
}


s32 SPI_Loopback_DmaMasterTxRxSlaveTxRxTest(SPI_PortInfo_st *pMaster, SPI_PortInfo_st *pSlave)
{
	SPI_BufAllocInfo_st	*pMasterTxBufInfo;
	SPI_BufAllocInfo_st	*pMasterRxBufInfo;
	SPI_BufAllocInfo_st	*pSlaveTxBufInfo;
	SPI_BufAllocInfo_st	*pSlaveRxBufInfo;
	s32					Status = SPI_ERROR;
	bool					SlaveTxDone = FALSE;
	bool					SlaveRxDone = FALSE;
	bool					MasterTxDone = FALSE;
	bool					MasterRxDone = FALSE;
	int					LoopCnt;
	s32					Compare1 = SPI_NO_ERROR;
	s32					Compare2 = SPI_NO_ERROR;

	if( !(pMaster->m_EnTxChannel == TRUE && pMaster->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pMaster);
		SPI_DisplayVar_EnRxChannel(pMaster);
		
		return SPI_ERROR;
	}

	if( !(pSlave->m_EnTxChannel == TRUE && pSlave->m_EnRxChannel == TRUE) )
	{
		SPI_DisplayError();
		SPI_DisplayVar_EnTxChannel(pSlave);
		SPI_DisplayVar_EnRxChannel(pSlave);
		
		return SPI_ERROR;
	}

	SPI_OpenPort(pMaster);
	SPI_OpenPort(pSlave);

	//for(LoopCnt=0; LoopCnt<2; LoopCnt++)
	{
		/* Prepare buffer for Master Tx and Slave Rx */
		pMasterTxBufInfo = &pMaster->m_TxBufNode;
		pMasterRxBufInfo = &pMaster->m_RxBufNode;

		pMasterTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pMasterTxBufInfo, pMaster->m_TxTransferSize);
		if(pMasterTxBufInfo->m_pBufPtr == NULL)
		{
			SPI_ASSERT();
		}

		SPI_NCBufRandInit(pMasterTxBufInfo, pMaster->m_TxTransferSize);
		SPI_NCBufDisplayHex(pMasterTxBufInfo, TRUE);
		
		pMasterRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pMasterRxBufInfo, pMaster->m_RxTransferSize);
		if(pMasterRxBufInfo->m_pBufPtr == NULL)
		{
			SPI_ASSERT();
		}
		SPI_NCBufZeroInit(pMasterRxBufInfo);

		pSlaveTxBufInfo = &pSlave->m_TxBufNode;
		pSlaveRxBufInfo = &pSlave->m_RxBufNode;

		pSlaveTxBufInfo->m_pBufPtr = SPI_GetNCBuf(pSlaveTxBufInfo, pSlave->m_TxTransferSize);
		if(pSlaveTxBufInfo->m_pBufPtr == NULL)
		{
			SPI_ASSERT();
		}

		SPI_NCBufRandInit(pSlaveTxBufInfo, pSlave->m_TxTransferSize);
		SPI_NCBufDisplayHex(pSlaveTxBufInfo, TRUE);
		
		pSlaveRxBufInfo->m_pBufPtr = SPI_GetNCBuf(pSlaveRxBufInfo, pSlave->m_RxTransferSize);
		if(pSlaveRxBufInfo->m_pBufPtr == NULL)
		{
			SPI_ASSERT();
		}
		SPI_NCBufZeroInit(pSlaveRxBufInfo);

		/* Tx Data size should be less than Tx FIFO size in polling mode. */
		SPI_DmaSlaveTxRx(pSlave, SPI_SlaveTxCompleteCallback, (u32)&SlaveTxDone,
			SPI_SlaveRxCompleteCallback, (u32)&SlaveRxDone);
		SPI_DmaMasterTxRx(pMaster, SPI_MasterTxCompleteCallback, (u32)&MasterTxDone,
			SPI_MasterRxCompleteCallback, (u32)&MasterRxDone);

		while(!UART_GetKey())
		{
			if(SlaveTxDone && SlaveRxDone && MasterTxDone && MasterRxDone)
			{
				break;
			}
		}

		Compare1 = SPI_CompareData(pSlaveTxBufInfo->m_pBufPtr, pMasterRxBufInfo->m_pBufPtr, pSlaveTxBufInfo->m_ReqDataSize);
		SPI_NCBufDisplayHex(pMasterRxBufInfo, TRUE);

		Compare2 = SPI_CompareData(pMasterTxBufInfo->m_pBufPtr, pSlaveRxBufInfo->m_pBufPtr, pMasterTxBufInfo->m_ReqDataSize);
		SPI_NCBufDisplayHex(pSlaveRxBufInfo, TRUE);

		SPI_FreeNCBuf(pMasterTxBufInfo);
		SPI_FreeNCBuf(pMasterRxBufInfo);
		SPI_FreeNCBuf(pSlaveTxBufInfo);
		SPI_FreeNCBuf(pSlaveRxBufInfo);
	}

	SPI_ClosePort(pMaster);
	SPI_ClosePort(pSlave);

	if(Compare1 != SPI_NO_ERROR || Compare2 != SPI_NO_ERROR)
	{
		return SPI_ERROR;
	}
	else
	{
		return SPI_NO_ERROR;
	}
}


s32 SPI_Loopback_PortAPortBDMATest(SPI_PortInfo_st *pPortAInfo, SPI_PortInfo_st *pPortBInfo)
{
	SPI_PortInfo_st		*pMasterPort;
	SPI_PortInfo_st		*pSlavePort;
	SPI_PortInfo_st		*pTxPort;
	SPI_PortInfo_st		*pRxPort;
	s32					Status = SPI_ERROR;

	Status = SPI_CheckLoopbackMasterAndChannel(pPortAInfo, pPortBInfo, &pMasterPort, &pSlavePort, &pTxPort, &pRxPort);
	if(Status != SPI_NO_ERROR)
	{
		return Status;
	}

	pPortAInfo->m_RxTransferMode = SPI_DMA_MODE;		/* Set Interrupt Mode */
	pPortAInfo->m_TxTransferMode = SPI_DMA_MODE;		/* Set Interrupt Mode */
	pPortBInfo->m_RxTransferMode = SPI_DMA_MODE;		/* Set Interrrupt Mode */
	pPortBInfo->m_TxTransferMode = SPI_DMA_MODE;			/* Set Interrrupt Mode */

	if(pMasterPort->m_EnTxChannel == TRUE)
	{
		if(pMasterPort->m_EnRxChannel == TRUE)
		{
			Status = SPI_Loopback_DmaMasterTxRxSlaveTxRxTest(pMasterPort, pSlavePort);
		}
		else
		{
			Status = SPI_Loopback_DmaMasterTxSlaveRxTest(pMasterPort, pSlavePort);
		}
	}
	else
	{
		if(pMasterPort->m_EnRxChannel == TRUE)
		{
			Status = SPI_Loopback_DmaMasterRxSlaveTxTest(pMasterPort, pSlavePort);
		}
		else
		{
			SPI_ASSERT();
		}
	}

	return Status;
}

s32 SPI_Loopback_Port0Port1DMATest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	pPortAInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];
	pPortBInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];

	Status = SPI_Loopback_PortAPortBDMATest(pPortAInfo, pPortBInfo);

	return Status;
}

s32 SPI_Loopback_Port0Port2DMATest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	pPortAInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT0];
	pPortBInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_Loopback_PortAPortBDMATest(pPortAInfo, pPortBInfo);

	return Status;
}

s32 SPI_Loopback_Port1Port2DMATest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	pPortAInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT1];
	pPortBInfo = &pCtrlInfo->m_SPIPortTestInfo[SPI_PORT2];

	Status = SPI_Loopback_PortAPortBDMATest(pPortAInfo, pPortBInfo);

	return Status;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////
#define SPI_MIN_ERROR_CNT 0x50
#define SPI_DATA_COUNT 0x50
//////////
// File Name : SPI_CompareData
// File Description : data compare function.
// Input : source data1, source data2, data length
// Output : NONE.
// Version : 
int SPI_CompareData(u8* data1, u8* data2, u32 bytes)
{
	u32 i;
	int errorCnt=0;
	
	#if 0
	// find gargage offset.
	for(i=0;i<SPI_MIN_ERROR_CNT;i++) {
		if( *data1 == *(data2+i) ) {
			UART_Printf( "\nGarbage Offset : [%d]\n" , i );
			data2=data2+i;
			break;
		}
	}
	#endif /* 0 */
	
	for (i=0; i<bytes; i++)
	{
		if(*data1 != *data2 )
		{
			if (errorCnt==0)
			{
				UART_Printf("\n%08x=%02x <-> %08x=%02x\n", data1, *data1, data2, *data2);
			}
			errorCnt++;
		}
		data1++;
		data2++;
	}
	if(errorCnt==0) {
		UART_Printf("\nData Compare Ok");
	}
	else {
		UART_Printf( "\nError Count: %d", errorCnt );
	}

	#if 0
	return errorCnt;
	#else
	if(errorCnt== 0)
	{
		return SPI_NO_ERROR;
	}
	else
	{
		return SPI_ERROR;
	}
	#endif /* 0 */
}


typedef void (*SPI_fnDisplayInfo_t)(SPI_PortInfo_st *pPortInfo);
typedef void (*SPI_fnSetInfo_t)(SPI_PortInfo_st *pPortInfo);

typedef struct {
	SPI_fnDisplayInfo_t	m_DisplaySPIInfo;
	SPI_fnSetInfo_t		m_SetSPIInfo;
} SPI_SetInfoEntry_st;

SPI_SetInfoEntry_st gSPI_SetInfoEntry[] = 
{
	{SPI_DisplayVar_EnInterruptPrint, 	SPI_SetVar_EnInterruptPrint},
	{SPI_DisplayVar_EnSWReset, 			SPI_SetVar_EnSWReset},
	{SPI_DisplayVar_LoadDefault, 		SPI_SetVar_LoadDefault},
	{SPI_DisplayVar_SWReset, 			SPI_SetVar_SWReset},
	{SPI_DisplayVar_ReadRegister, 		SPI_SetVar_ReadRegister},
	{SPI_DisplayVar_EnHighSpeed, 		SPI_SetVar_EnHighSpeed},
	{SPI_DisplayVar_OpMode, 			SPI_SetVar_OpMode},
	{SPI_DisplayVar_CPOL, 				SPI_SetVar_CPOL},
	{SPI_DisplayVar_CPHA, 				SPI_SetVar_CPHA},
	{SPI_DisplayVar_EnRxChannel, 		SPI_SetVar_EnRxChannel},
	{SPI_DisplayVar_EnTxChannel, 		SPI_SetVar_EnTxChannel},
	{SPI_DisplayVar_IsUseExtClk,		SPI_SetVar_IsUseExtClk},
	{SPI_DisplayVar_EnClk, 				SPI_SetVar_EnClk},
	{SPI_DisplayVar_ExtClkSrc, 			SPI_SetVar_ExtClkSrc},
	{SPI_DisplayVar_ExtClkSrcDiv, 		SPI_SetVar_ExtClkSrcDiv},
	{SPI_DisplayVar_OpClock, 			NULL},
	{SPI_DisplayVar_SpiCLK, 				SPI_SetVar_SpiCLK},
	{SPI_DisplayVar_ChWidth, 			SPI_SetVar_ChWidth},
	{SPI_DisplayVar_TrailingCnt, 			SPI_SetVar_TrailingCnt},
	{SPI_DisplayVar_BusWidth, 			SPI_SetVar_BusWidth},
	{SPI_DisplayVar_RxTriggerLevel, 		SPI_SetVar_RxTriggerLevel},
	{SPI_DisplayVar_TxTriggerLevel, 		SPI_SetVar_TxTriggerLevel},
	{SPI_DisplayVar_RxTransferMode, 	SPI_SetVar_RxTransferMode},
	{SPI_DisplayVar_TxTransferMode, 	SPI_SetVar_TxTransferMode},
	{SPI_DisplayVar_DMAType, 			SPI_SetVar_DMAType},
	{SPI_DisplayVar_TxTransferSize, 		SPI_SetVar_TxTransferSize},
	{SPI_DisplayVar_RxTransferSize, 		SPI_SetVar_RxTransferSize},
	{SPI_DisplayVar_NCSTimeCnt, 		SPI_SetVar_NCSTimeCnt},
	{SPI_DisplayVar_EnAutoCS, 			SPI_SetVar_EnAutoCS},
	{SPI_DisplayVar_EnTrailingInt, 		SPI_SetVar_EnTrailingInt},
	{SPI_DisplayVar_EnRxOverrunInt, 	SPI_SetVar_EnRxOverrunInt},
	{SPI_DisplayVar_EnRxUnderrunInt, 	SPI_SetVar_EnRxUnderrunInt},
	{SPI_DisplayVar_EnTxOverrunInt, 	SPI_SetVar_EnTxOverrunInt},
	{SPI_DisplayVar_EnTxUnderrunInt, 	SPI_SetVar_EnTxUnderrunInt},
	{SPI_DisplayVar_EnRxHwordSwap, 	SPI_SetVar_EnRxHwordSwap},
	{SPI_DisplayVar_EnRxByteSwap, 		SPI_SetVar_EnRxByteSwap},
	{SPI_DisplayVar_EnRxBitSwap, 		SPI_SetVar_EnRxBitSwap},
	{SPI_DisplayVar_EnRxSwap, 			SPI_SetVar_EnRxSwap},
	{SPI_DisplayVar_EnTxHwordSwap, 	SPI_SetVar_EnTxHwordSwap},
	{SPI_DisplayVar_EnTxByteSwap, 		SPI_SetVar_EnTxByteSwap},
	{SPI_DisplayVar_EnTxBitSwap, 		SPI_SetVar_EnTxBitSwap},
	{SPI_DisplayVar_EnTxSwap, 			SPI_SetVar_EnTxSwap},
	{SPI_DisplayVar_FBClkSel, 			SPI_SetVar_FBClkSel},
	{SPI_DisplayVar_DrvStrength, 		SPI_SetVar_DrvStrength},
	{SPI_DisplayVar_RxDMANum, 		NULL},
	{SPI_DisplayVar_TxDMANum, 		NULL},
	{SPI_DisplayVar_RxDMACh, 			SPI_SetVar_RxDMACh},
	{SPI_DisplayVar_TxDMACh, 			SPI_SetVar_TxDMACh}
};


void SPI_SetDefaultInfo(SPI_PortInfo_st *pPortInfo)
{
	s32					MenuEntryCnt;
	s32					MenuIndx;
	s32					SelNum;
	SPI_SetInfoEntry_st	*pSetInfoEntry;

	MenuEntryCnt = sizeof(gSPI_SetInfoEntry)/sizeof(SPI_SetInfoEntry_st);
	while(1)
	{
		UART_Printf("\n====================================================");
		UART_Printf("\n   SPI%d Parameter Setting", pPortInfo->m_SPIPortIdx);
		UART_Printf("\n====================================================\n");
		for(MenuIndx = 0; MenuIndx<MenuEntryCnt; MenuIndx++)
		{
			pSetInfoEntry = &gSPI_SetInfoEntry[MenuIndx];

			UART_Printf("\n%2d. ", MenuIndx);
			if(pSetInfoEntry->m_DisplaySPIInfo != NULL)
			{
				(*pSetInfoEntry->m_DisplaySPIInfo)(pPortInfo);
			}

			if(pSetInfoEntry->m_SetSPIInfo == NULL)
			{
				UART_Printf("\t(Display Only)");
			}
		}
		UART_Printf("\n%2d. Exit", MenuIndx);
		UART_Printf("\n====================================================");

		UART_Printf("\nSelect: ");
		SelNum = UART_GetIntNum();
		//if(SelNum <= 0 || SelNum >= MenuEntryCnt)
		if(SelNum == MenuEntryCnt)	/* Out ouf entry range */
		{
			break;
		}

		if(SelNum >= 0 && SelNum < MenuEntryCnt)
		{
			pSetInfoEntry = &gSPI_SetInfoEntry[SelNum];
			if(pSetInfoEntry->m_SetSPIInfo != NULL)
			{
				(*pSetInfoEntry->m_SetSPIInfo)(pPortInfo);
			}
		}
	}
}


s32 SPI_SetPort0Info(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortInfo;
	pPortInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT0];
	
	SPI_SetDefaultInfo(pPortInfo);
}

s32 SPI_SetPort1Info(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortInfo;
	pPortInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT1];
	
	SPI_SetDefaultInfo(pPortInfo);
}

s32 SPI_SetPort2Info(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPI_PortInfo_st		*pPortInfo;
	pPortInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT2];
	
	SPI_SetDefaultInfo(pPortInfo);
}


u8 SPIAuto_MasterTxSlaveRxDMA(void)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	SPI_SetDefaultSystemEnv();		/* Set Clock, TZPC */
	SPI_InitDefaultValue(&gSPI_CtrlInfo);	/* Set Default Setting */
	SPI_Init(&gSPI_CtrlInfo);

	pPortAInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT0];
	pPortAInfo->m_OpMode = SPI_MASTER;
	pPortAInfo->m_EnTxChannel = TRUE;
	pPortAInfo->m_EnRxChannel = FALSE;
	pPortBInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT1];
	pPortBInfo->m_OpMode = SPI_SLAVE;
	pPortBInfo->m_EnTxChannel = FALSE;
	pPortBInfo->m_EnRxChannel = TRUE;

	Status = SPI_Loopback_PortAPortBDMATest(pPortAInfo, pPortBInfo);
	if(Status == SPI_NO_ERROR)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}


u8 SPIAuto_MasterRxSlaveTxDMA(void)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	SPI_SetDefaultSystemEnv();		/* Set Clock, TZPC */
	SPI_InitDefaultValue(&gSPI_CtrlInfo);	/* Set Default Setting */
	SPI_Init(&gSPI_CtrlInfo);

	pPortAInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT0];
	pPortAInfo->m_OpMode = SPI_MASTER;
	pPortAInfo->m_EnTxChannel = FALSE;
	pPortAInfo->m_EnRxChannel = TRUE;
	pPortBInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT1];
	pPortBInfo->m_OpMode = SPI_SLAVE;
	pPortBInfo->m_EnTxChannel = TRUE;
	pPortBInfo->m_EnRxChannel = FALSE;

	Status = SPI_Loopback_PortAPortBDMATest(pPortAInfo, pPortBInfo);
	if(Status == SPI_NO_ERROR)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}


u8 SPIAuto_SlaveRxMasterTxDMA(void)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	SPI_SetDefaultSystemEnv();		/* Set Clock, TZPC */
	SPI_InitDefaultValue(&gSPI_CtrlInfo);	/* Set Default Setting */
	SPI_Init(&gSPI_CtrlInfo);

	pPortAInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT1];
	pPortAInfo->m_OpMode = SPI_MASTER;
	pPortAInfo->m_EnTxChannel = TRUE;
	pPortAInfo->m_EnRxChannel = FALSE;

	pPortBInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT0];
	pPortBInfo->m_OpMode = SPI_SLAVE;
	pPortBInfo->m_EnTxChannel = FALSE;
	pPortBInfo->m_EnRxChannel = TRUE;

	Status = SPI_Loopback_PortAPortBDMATest(pPortAInfo, pPortBInfo);
	if(Status == SPI_NO_ERROR)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}


u8 SPIAuto_SlaveTxMasterRxDMA(void)
{
	SPI_PortInfo_st		*pPortAInfo;
	SPI_PortInfo_st		*pPortBInfo;
	s32					Status;

	SPI_SetDefaultSystemEnv();		/* Set Clock, TZPC */
	SPI_InitDefaultValue(&gSPI_CtrlInfo);	/* Set Default Setting */
	SPI_Init(&gSPI_CtrlInfo);

	pPortAInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT1];
	pPortAInfo->m_OpMode = SPI_MASTER;
	pPortAInfo->m_EnTxChannel = FALSE;
	pPortAInfo->m_EnRxChannel = TRUE;

	pPortBInfo = &gSPI_CtrlInfo.m_SPIPortTestInfo[SPI_PORT0];
	pPortBInfo->m_OpMode = SPI_SLAVE;
	pPortBInfo->m_EnTxChannel = TRUE;
	pPortBInfo->m_EnRxChannel = FALSE;

	Status = SPI_Loopback_PortAPortBDMATest(pPortAInfo, pPortBInfo);
	if(Status == SPI_NO_ERROR)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

s32 SPI_Consol_AutoTest(SPI_CtrlInfo_st *pCtrlInfo)
{
	SPIAuto_MasterTxSlaveRxDMA();		/* Port0 Master Tx, Port1 Slave Rx */
	SPIAuto_MasterRxSlaveTxDMA();		/* Port0 Master Rx, Port1 Slave Tx */
	SPIAuto_SlaveRxMasterTxDMA();		/* Port0 Slave Rx, Port1 Master Tx */
	SPIAuto_SlaveTxMasterRxDMA();		/* Port0 Slave Tx, Port1 Master Rx */

	return SPI_NO_ERROR;
}



typedef struct SPI_PLL_PMS_Entry_struct {
	float		m_Fout;
	u32		m_PDIV;
	u32		m_MDIV;
	u32		m_SDIV;
	bool		m_IsVSELHigh;
} SPI_PLL_PMSEntry_st;

#define	SPI_PLL_MAX_VALID_ENTRY			20
#define	SPI_PLL_MAX_ACCURATE_ENTRY		5

typedef struct SPI_PLL_control_info_struct{
	SPI_PLL_PMSEntry_st		m_ValidPMSEntry[SPI_PLL_MAX_VALID_ENTRY];
	u32					m_ValidPMSEntryLimit;
	u32					m_PDIVMin;
	u32					m_PDIVMax;
	u32					m_MDIVMin;
	u32					m_MDIVMax;
	u32					m_SDIVMin;
	u32					m_SDIVMax;
	float					m_AccurateRate;		/* 1%, 0.1%, 0.01%, etc */
	float					m_AccurateEntry[SPI_PLL_MAX_ACCURATE_ENTRY];
	u32					m_AccurateEntryLimit;
	u32					m_ValidPMSEntryCnt;
	bool					m_EnSameTargetClk;	/* Only check the PMS for the exactly same Target clock */
	bool					m_EnVSELHigh;
	//float					m_FrefMin;
	//float					m_FrefMax;
	//float					m_FVCOLowMin;
	//float					m_FVCOLowMax;
	//float					m_FVCOHighMin;
	//float					m_FVCOHighMax;
	//float					m_FoutMin;
	//float					m_FoutMax;
} SPI_PLL_CtrlInfo_st;

typedef s32 (*SPI_PLL_GetCalcFreq_t)(SPI_PLL_CtrlInfo_st *pPLLCtrlInfo, float *pFreq, u32 PDIV, u32 MDIV, u32 SDIV, bool *pIsVSELHigh);

s32 SPI_PLL_CalculateFreq_APLL(SPI_PLL_CtrlInfo_st *pPLLCtrlInfo, float *pFreq, u32 PDIV, u32 MDIV, u32 SDIV, bool *pIsVSELHigh)
{
	float		FINClk;
	float		FRef;
	float		FVCO;
	float		Fout;
	u32		SDivExpo;

	FINClk = (float)FIN;

	if(PDIV == 0)
	{
		UART_ASSERT();
	}

	FRef = FINClk/PDIV;
	if(FRef < 1.0e6 || FRef > 1.0e7)	/* FRef should be 3MHz ~ 6MHz in S5PV210. */
	{
		return UART_ERROR;
	}

	//FVCO = MDIV * FINClk / PDIV;
	FVCO = 2 * MDIV * FINClk / PDIV;

	if(FVCO >= 9.6e8 && FVCO <= 2.06e9)	/* FVCO should be 1GHz ~ 1.4GHz in S5PV210 low voltage. */
	{
		*pIsVSELHigh = FALSE;
	}
	#if 0
	else if(FVCO >= 1.4e9 && FVCO <= 2.0e9)	/* FVCO should be 1GHz ~ 1.4GHz in S5PV210 high voltage. */
	{
		if(pPLLCtrlInfo->m_EnVSELHigh == TRUE)
		{
			*pIsVSELHigh = TRUE;
		}
		else
		{
			return UART_ERROR;
		}
	}
	#endif /* 0 */
	else
	{
		return UART_ERROR;
	}
	
	//Fout = MDIV * FINClk / (PDIV * (2^(SDIV-1)));
	SDivExpo = 1 << (SDIV-1);
	Fout = MDIV * FINClk / (PDIV * SDivExpo);
	#if 0
	if(Fout > 3.2e7 || Fout > 2.0e9)			/* FOUT sould be 32Mhz ~ 2GHz */
	{
		return UART_ERROR;
	}
	#endif /* 0 */
	
	*pFreq = Fout;

	return UART_NO_ERROR;
}

s32 SPI_PLL_CalculateFreq_MPLL(SPI_PLL_CtrlInfo_st *pPLLCtrlInfo, float *pFreq, u32 PDIV, u32 MDIV, u32 SDIV, bool *pIsVSELHigh)
{
	float		FINClk;
	float		FRef;
	float		FVCO;
	float		Fout;
	u32		SDivExpo;

	FINClk = (float)FIN;

	if(PDIV == 0)
	{
		UART_ASSERT();
	}

	FRef = FINClk/PDIV;
	if(FRef < 3.0e6 || FRef > 6.0e6)	/* FRef should be 3MHz ~ 6MHz in S5PV210. */
	{
		return UART_ERROR;
	}

	FVCO = MDIV * FINClk / PDIV;

	if(FVCO >= 1.0e9 && FVCO <= 1.4e9)	/* FVCO should be 1GHz ~ 1.4GHz in S5PV210 low voltage. */
	{
		*pIsVSELHigh = FALSE;
	}
	else if(FVCO >= 1.4e9 && FVCO <= 2.0e9)	/* FVCO should be 1GHz ~ 1.4GHz in S5PV210 high voltage. */
	{
		if(pPLLCtrlInfo->m_EnVSELHigh == TRUE)
		{
			*pIsVSELHigh = TRUE;
		}
		else
		{
			return UART_ERROR;
		}
	}
	else
	{
		return UART_ERROR;
	}
	
	//Fout = MDIV * FINClk / (PDIV * (2^SDIV));
	SDivExpo = 1 << SDIV;
	Fout = MDIV * FINClk / (PDIV * SDivExpo);
	if(Fout < 3.2e7 || Fout > 2.0e9)			/* FOUT sould be 32Mhz ~ 2GHz */
	{
		return UART_ERROR;
	}
	
	*pFreq = Fout;

	return UART_NO_ERROR;
}

s32 SPI_PLL_CalculateFreq_EPLL(SPI_PLL_CtrlInfo_st *pPLLCtrlInfo, float *pFreq, u32 PDIV, u32 MDIV, u32 SDIV, bool *pIsVSELHigh)
{
	float		FINClk;
	float		FRef;
	float		FVCO;
	float		Fout;
	u32		SDivExpo;

	FINClk = (float)FIN;

	if(PDIV == 0)
	{
		UART_ASSERT();
	}

	FRef = FINClk/PDIV;
	if(FRef < 2.0e6 || FRef > 6.0e6)	/* FRef should be 3MHz ~ 6MHz in S5PV210. */
	{
		return UART_ERROR;
	}

	FVCO = MDIV * FINClk / PDIV;

	if(FVCO >= 3.3e8 && FVCO <= 4.6e8)	/* FVCO should be 1GHz ~ 1.4GHz in S5PV210 low voltage. */
	{
		*pIsVSELHigh = FALSE;
	}
	else if(FVCO >= 4.6e8 && FVCO <= 6.6e8)	/* FVCO should be 1GHz ~ 1.4GHz in S5PV210 high voltage. */
	{
		if(pPLLCtrlInfo->m_EnVSELHigh == TRUE)
		{
			*pIsVSELHigh = TRUE;
		}
		else
		{
			return UART_ERROR;
		}
	}
	else
	{
		return UART_ERROR;
	}

	//Fout = MDIV * FINClk / (PDIV * (2^SDIV));
	SDivExpo = 1 << SDIV;
	Fout = MDIV * FINClk / (PDIV * SDivExpo);
	if(Fout < 1.0e7 || Fout > 6.0e8)			/* FOUT sould be 32Mhz ~ 2GHz */
	{
		return UART_ERROR;
	}
	
	*pFreq = Fout;

	return UART_NO_ERROR;
}


s32 SPI_PLL_CalculateFreq_VPLL(SPI_PLL_CtrlInfo_st *pPLLCtrlInfo, float *pFreq, u32 PDIV, u32 MDIV, u32 SDIV, bool *pIsVSELHigh)
{
	return SPI_PLL_CalculateFreq_EPLL(pPLLCtrlInfo, pFreq, PDIV, MDIV, SDIV, pIsVSELHigh);	/* same to EPLL in S5PV210 */
}

s32 SPI_PLL_GetPMSValue(char *pPLLName, SPI_PLL_CtrlInfo_st *pPLLCtrlInfo, 
	SPI_PLL_GetCalcFreq_t GetCalcFreqFunction, float TargetClk, u32 *pPDIV, u32 *pMDIV, u32 *pSDIV)
{
	SPI_PLL_PMSEntry_st	*pPMSEntry;
	float		AccuracyMin;
	float		AccuracyMax;
	s32		MaxPmsEntry;
	s32		PmsEntryIdx;
	s32		MaxAccuracy;
	s32		AccuracyIdx;
	float		CalculatedClk;
	u32		Epll_PDIV;
	u32		EPll_MDIV;
	u32		EPll_SDIV;
	bool		IsHighVSel;
	s32		Status;
	s32		SelNum;
	u32		LocalCnt;

	MaxPmsEntry = pPLLCtrlInfo->m_ValidPMSEntryLimit;
	MaxAccuracy = pPLLCtrlInfo->m_AccurateEntryLimit;
	for(LocalCnt = 0; LocalCnt < MaxAccuracy; LocalCnt++)
	{
		pPLLCtrlInfo->m_AccurateEntry[LocalCnt] = LocalCnt * pPLLCtrlInfo->m_AccurateRate;
	}
	
	memset(pPLLCtrlInfo->m_ValidPMSEntry, 0x00, sizeof(pPLLCtrlInfo->m_ValidPMSEntry));

	PmsEntryIdx = 0;
	pPLLCtrlInfo->m_ValidPMSEntryCnt = 0;

	UART_Printf("\nWait a moment (a few seconds)....");

	/* same to the TargetClk */
	for(Epll_PDIV = pPLLCtrlInfo->m_PDIVMin; Epll_PDIV <= pPLLCtrlInfo->m_PDIVMax; Epll_PDIV++)
	{
		for(EPll_MDIV = pPLLCtrlInfo->m_MDIVMin; EPll_MDIV <= pPLLCtrlInfo->m_MDIVMax; EPll_MDIV++)
		{
			for(EPll_SDIV = pPLLCtrlInfo->m_SDIVMin; EPll_SDIV <= pPLLCtrlInfo->m_SDIVMax; EPll_SDIV++)
			{
				Status = (*GetCalcFreqFunction)(pPLLCtrlInfo, &CalculatedClk, Epll_PDIV, EPll_MDIV, EPll_SDIV, &IsHighVSel);
				if(Status == UART_NO_ERROR)
				{
					if(CalculatedClk == TargetClk)
					{
						if(PmsEntryIdx < MaxPmsEntry)
						{
							pPMSEntry = &pPLLCtrlInfo->m_ValidPMSEntry[PmsEntryIdx];
							pPMSEntry->m_Fout = CalculatedClk;
							pPMSEntry->m_PDIV = Epll_PDIV;
							pPMSEntry->m_MDIV = EPll_MDIV;
							pPMSEntry->m_SDIV = EPll_SDIV;
							pPMSEntry->m_IsVSELHigh = IsHighVSel;
							PmsEntryIdx++;
						}
					}
				}
			}
		}
	}

	/* not same to the TargetClk but sometimes useful. */
	if(pPLLCtrlInfo->m_EnSameTargetClk != TRUE || pPLLCtrlInfo->m_ValidPMSEntryCnt == 0)	/* a exactly same to the TargetClk is not found. */
	{
		for(AccuracyIdx = 1; AccuracyIdx < MaxAccuracy; AccuracyIdx++)
		{
			AccuracyMin = TargetClk * pPLLCtrlInfo->m_AccurateEntry[AccuracyIdx-1];
			AccuracyMax = TargetClk * pPLLCtrlInfo->m_AccurateEntry[AccuracyIdx];

			for(Epll_PDIV = pPLLCtrlInfo->m_PDIVMin; Epll_PDIV <= pPLLCtrlInfo->m_PDIVMax; Epll_PDIV++)
			{
				for(EPll_MDIV = pPLLCtrlInfo->m_MDIVMin; EPll_MDIV <= pPLLCtrlInfo->m_MDIVMax; EPll_MDIV++)
				{
					for(EPll_SDIV = pPLLCtrlInfo->m_SDIVMin; EPll_SDIV <= pPLLCtrlInfo->m_SDIVMax; EPll_SDIV++)
					{
						Status = (*GetCalcFreqFunction)(pPLLCtrlInfo, &CalculatedClk, Epll_PDIV, EPll_MDIV, EPll_SDIV, &IsHighVSel);
						if(Status == UART_NO_ERROR)
						{
							if( (CalculatedClk >= TargetClk - AccuracyMax && CalculatedClk < TargetClk - AccuracyMin) ||
								(CalculatedClk > TargetClk + AccuracyMin && CalculatedClk <= TargetClk + AccuracyMax) )
							{
								if(PmsEntryIdx < MaxPmsEntry)
								{
									pPMSEntry = &pPLLCtrlInfo->m_ValidPMSEntry[PmsEntryIdx];
									pPMSEntry->m_Fout = CalculatedClk;
									pPMSEntry->m_PDIV = Epll_PDIV;
									pPMSEntry->m_MDIV = EPll_MDIV;
									pPMSEntry->m_SDIV = EPll_SDIV;
									pPMSEntry->m_IsVSELHigh = IsHighVSel;
									PmsEntryIdx++;
								}
								else
								{
									goto PMS_Found_Done;
								}
							}
						}
					}
				}
			}
		}
	}

PMS_Found_Done:

	UART_Printf("\n====================================================");
	UART_Printf("\n%s Valid PMS Value", pPLLName);
	UART_Printf("\n====================================================\n");

	pPLLCtrlInfo->m_ValidPMSEntryCnt = PmsEntryIdx;
	for(PmsEntryIdx = 0; PmsEntryIdx < pPLLCtrlInfo->m_ValidPMSEntryCnt; PmsEntryIdx++)
	{
		pPMSEntry = &pPLLCtrlInfo->m_ValidPMSEntry[PmsEntryIdx];
		
		UART_Printf("\n%2d. PDIV(%2d) MDIV(%4d) SDIV(%1d) %s : %9.4f MHz", PmsEntryIdx,
			pPMSEntry->m_PDIV, pPMSEntry->m_MDIV, pPMSEntry->m_SDIV, 
			pPMSEntry->m_IsVSELHigh ? "HIGH" : "LOW ", pPMSEntry->m_Fout/1.0e6);
	}

	UART_Printf("\n%2d. Exit", PmsEntryIdx);
	UART_Printf("\n====================================================");

	UART_Printf("\nSelect: ");
	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum >= pPLLCtrlInfo->m_ValidPMSEntryCnt)	/* Out ouf entry range */
	{
		return UART_ERROR;
	}

	pPMSEntry = &pPLLCtrlInfo->m_ValidPMSEntry[SelNum];
	*pPDIV = pPMSEntry->m_PDIV;
	*pMDIV = pPMSEntry->m_MDIV;
	*pSDIV = pPMSEntry->m_SDIV;

	return UART_NO_ERROR;
}


void SPI_DisplayPMS_APLL(void)
{
	UART_Printf("APLL(%.2fMHz)", (float)g_uAPLL/1.0e6);
}

void SPI_DisplayPMS_MPLL(void)
{
	UART_Printf("MPLL(%.2fMHz)", (float)g_uMPLL/1.0e6);
}

void SPI_DisplayPMS_EPLL(void)
{
	UART_Printf("EPLL(%.2fMHz)", (float)g_uEPLL/1.0e6);
}

void SPI_DisplayPMS_VPLL(void)
{
	UART_Printf("VPLL(%.2fMHz)", (float)g_uVPLL/1.0e6);
}


void SPI_SetPMS_APLL(void)
{
	SPI_PLL_CtrlInfo_st		PLLCtrlInfo;
	float		TargetClock;
	u32		PDIV;
	u32		MDIV;
	u32		SDIV;
	s32		Status;
	s32		SelNum;
	float		ClockUnit;

	UART_Printf("\nInput Target Clock (960M ~ 2060M) : ");
	UART_Printf("\nInput Target Clock unit");
	UART_Printf("\n");

	UART_Printf("\n0.Hz");
	UART_Printf("\n1.KHz");
	UART_Printf("\n2.MHz(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");
	SelNum = UART_GetIntNum();
	switch(SelNum)
	{
		case 0:
			ClockUnit = 1.0;
			break;
		case 1:
			ClockUnit = 1.0e3;
			break;
		case 2:
		default:
			ClockUnit = 1.0e6;
			break;
	}

	UART_Printf("\nInput Target Clock : ");
	SelNum = UART_GetIntNum();
	if(SelNum <= 0)
	{
		return;
	}

	TargetClock = (float)SelNum;
	TargetClock *= ClockUnit;
	if(TargetClock < 9.6e7 || TargetClock > 2.06e9)
	{
		UART_Printf("\nInValid Clock Frequency : %9.4f MHz", TargetClock / 1.0e6);
		
		return;
	}

	memset(&PLLCtrlInfo, 0x00, sizeof(SPI_PLL_CtrlInfo_st));
	
	PLLCtrlInfo.m_ValidPMSEntryLimit = SPI_PLL_MAX_VALID_ENTRY;
	PLLCtrlInfo.m_AccurateEntryLimit = SPI_PLL_MAX_ACCURATE_ENTRY;
	PLLCtrlInfo.m_AccurateRate = 0.01;		/* 1 % degree */
	PLLCtrlInfo.m_PDIVMin = 1;
	PLLCtrlInfo.m_PDIVMax = 63;
	PLLCtrlInfo.m_MDIVMin = 64;
	PLLCtrlInfo.m_MDIVMax = 1023;
	PLLCtrlInfo.m_SDIVMin = 1;
	PLLCtrlInfo.m_SDIVMax = 5;
	PLLCtrlInfo.m_EnSameTargetClk = FALSE;
	PLLCtrlInfo.m_EnVSELHigh = FALSE;

	Status = SPI_PLL_GetPMSValue("APLL", &PLLCtrlInfo, SPI_PLL_CalculateFreq_APLL, TargetClock, &PDIV, &MDIV, &SDIV);
	if(Status == UART_NO_ERROR)
	{
		UART_Printf("\nPDIV(%d), MDIV(%d), SDIV(%d) but NOT applied to the system.", PDIV, MDIV, SDIV);

		#if 0
		SYSC_StartPLLbyMPS(eAPLL, MDIV, PDIV, SDIV);
		while(!SYSC_IsLockDetected(eAPLL))
		{
			/* Wait until PLL is locked. */
		}
		
		SYSC_UpdateClkInform();
		#endif /* 0 */
	}
}

void SPI_SetPMS_MPLL(void)
{
	SPI_PLL_CtrlInfo_st		PLLCtrlInfo;
	float		TargetClock;
	u32		PDIV;
	u32		MDIV;
	u32		SDIV;
	s32		Status;
	s32		SelNum;
	float		ClockUnit;

	UART_Printf("\nInput Target Clock (32M ~ 2G) : ");
	UART_Printf("\nInput Target Clock unit");
	UART_Printf("\n");

	UART_Printf("\n0.Hz");
	UART_Printf("\n1.KHz");
	UART_Printf("\n2.MHz(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");
	SelNum = UART_GetIntNum();
	switch(SelNum)
	{
		case 0:
			ClockUnit = 1.0;
			break;
		case 1:
			ClockUnit = 1.0e3;
			break;
		case 2:
		default:
			ClockUnit = 1.0e6;
			break;
	}

	UART_Printf("\nInput Target Clock : ");
	SelNum = UART_GetIntNum();
	if(SelNum <= 0)
	{
		return;
	}

	TargetClock = (float)SelNum;
	TargetClock *= ClockUnit;
	if(TargetClock < 3.2e7 || TargetClock > 2.0e9)
	{
		UART_Printf("\nInValid Clock Frequency : %9.4f MHz", TargetClock / 1.0e6);
		
		return;
	}

	memset(&PLLCtrlInfo, 0x00, sizeof(SPI_PLL_CtrlInfo_st));
	
	PLLCtrlInfo.m_ValidPMSEntryLimit = SPI_PLL_MAX_VALID_ENTRY;
	PLLCtrlInfo.m_AccurateEntryLimit = SPI_PLL_MAX_ACCURATE_ENTRY;
	PLLCtrlInfo.m_AccurateRate = 0.01;		/* 1 % degree */
	PLLCtrlInfo.m_PDIVMin = 1;
	PLLCtrlInfo.m_PDIVMax = 63;
	PLLCtrlInfo.m_MDIVMin = 64;
	PLLCtrlInfo.m_MDIVMax = 1023;
	PLLCtrlInfo.m_SDIVMin = 0;
	PLLCtrlInfo.m_SDIVMax = 4;
	PLLCtrlInfo.m_EnSameTargetClk = FALSE;
	PLLCtrlInfo.m_EnVSELHigh = FALSE;

	Status = SPI_PLL_GetPMSValue("MPLL", &PLLCtrlInfo, SPI_PLL_CalculateFreq_MPLL, TargetClock, &PDIV, &MDIV, &SDIV);
	if(Status == UART_NO_ERROR)
	{
		UART_Printf("\nPDIV(%d), MDIV(%d), SDIV(%d)", PDIV, MDIV, SDIV);
		
		SYSC_StartPLLbyMPS(eMPLL, MDIV, PDIV, SDIV);
		while(!SYSC_IsLockDetected(eMPLL))
		{
			/* Wait until PLL is locked. */
		}
		
		SYSC_UpdateClkInform();
	}
}


void SPI_SetPMS_EPLL(void)
{
	SPI_PLL_CtrlInfo_st		PLLCtrlInfo;
	float		TargetClock;
	u32		PDIV;
	u32		MDIV;
	u32		SDIV;
	s32		Status;
	s32		SelNum;
	float		ClockUnit;

	UART_Printf("\nValid Clock Range(10M ~ 600M)");
	UART_Printf("\nInput Target Clock unit");
	UART_Printf("\n");

	UART_Printf("\n0.Hz");
	UART_Printf("\n1.KHz");
	UART_Printf("\n2.MHz(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");
	SelNum = UART_GetIntNum();
	switch(SelNum)
	{
		case 0:
			ClockUnit = 1.0;
			break;
		case 1:
			ClockUnit = 1.0e3;
			break;
		case 2:
		default:
			ClockUnit = 1.0e6;
			break;
	}

	UART_Printf("\nInput Target Clock : ");
	SelNum = UART_GetIntNum();
	if(SelNum <= 0)
	{
		return;
	}

	TargetClock = (float)SelNum;
	TargetClock *= ClockUnit;
	if(TargetClock < 1.0e7 || TargetClock > 6.0e8)
	{
		UART_Printf("\nInValid Clock Frequency : %9.4f MHz", TargetClock / 1.0e6);
		
		return;
	}

	memset(&PLLCtrlInfo, 0x00, sizeof(SPI_PLL_CtrlInfo_st));
	
	PLLCtrlInfo.m_ValidPMSEntryLimit = SPI_PLL_MAX_VALID_ENTRY;
	PLLCtrlInfo.m_AccurateEntryLimit = SPI_PLL_MAX_ACCURATE_ENTRY;
	PLLCtrlInfo.m_AccurateRate = 0.01;		/* 1 % degree */
	PLLCtrlInfo.m_PDIVMin = 1;
	PLLCtrlInfo.m_PDIVMax = 63;
	PLLCtrlInfo.m_MDIVMin = 16;
	PLLCtrlInfo.m_MDIVMax = 255;
	PLLCtrlInfo.m_SDIVMin = 0;
	PLLCtrlInfo.m_SDIVMax = 4;
	PLLCtrlInfo.m_EnSameTargetClk = FALSE;
	PLLCtrlInfo.m_EnVSELHigh = FALSE;

	Status = SPI_PLL_GetPMSValue("EPLL", &PLLCtrlInfo, SPI_PLL_CalculateFreq_EPLL, TargetClock, &PDIV, &MDIV, &SDIV);
	if(Status == UART_NO_ERROR)
	{
		UART_Printf("\nPDIV(%d), MDIV(%d), SDIV(%d)", PDIV, MDIV, SDIV);
		
		SYSC_StartPLLbyMPS(eEPLL, MDIV, PDIV, SDIV);
		while(!SYSC_IsLockDetected(eEPLL))
		{
			/* Wait until PLL is locked. */
		}
		
		SYSC_UpdateClkInform();
	}
}

void SPI_SetPMS_VPLL(void)
{
	SPI_PLL_CtrlInfo_st		PLLCtrlInfo;
	float		TargetClock;
	u32		PDIV;
	u32		MDIV;
	u32		SDIV;
	s32		Status;
	s32		SelNum;
	float		ClockUnit;

	UART_Printf("\nValid Clock Range(10M ~ 600M)");
	UART_Printf("\nInput Target Clock unit");
	UART_Printf("\n");

	UART_Printf("\n0.Hz");
	UART_Printf("\n1.KHz");
	UART_Printf("\n2.MHz(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");
	SelNum = UART_GetIntNum();
	switch(SelNum)
	{
		case 0:
			ClockUnit = 1.0;
			break;
		case 1:
			ClockUnit = 1.0e3;
			break;
		case 2:
		default:
			ClockUnit = 1.0e6;
			break;
	}

	UART_Printf("\nInput Target Clock : ");
	SelNum = UART_GetIntNum();
	if(SelNum <= 0)
	{
		return;
	}

	TargetClock = (float)SelNum;
	TargetClock *= ClockUnit;
	if(TargetClock < 1.0e7 || TargetClock > 6.0e8)
	{
		UART_Printf("\nInValid Clock Frequency : %9.4f MHz", TargetClock / 1.0e6);
		
		return;
	}

	memset(&PLLCtrlInfo, 0x00, sizeof(SPI_PLL_CtrlInfo_st));
	
	PLLCtrlInfo.m_ValidPMSEntryLimit = SPI_PLL_MAX_VALID_ENTRY;
	PLLCtrlInfo.m_AccurateEntryLimit = SPI_PLL_MAX_ACCURATE_ENTRY;
	PLLCtrlInfo.m_AccurateRate = 0.01;		/* 1 % degree */
	PLLCtrlInfo.m_PDIVMin = 1;
	PLLCtrlInfo.m_PDIVMax = 63;
	PLLCtrlInfo.m_MDIVMin = 16;
	PLLCtrlInfo.m_MDIVMax = 255;
	PLLCtrlInfo.m_SDIVMin = 0;
	PLLCtrlInfo.m_SDIVMax = 4;
	PLLCtrlInfo.m_EnSameTargetClk = FALSE;
	PLLCtrlInfo.m_EnVSELHigh = FALSE;

	Status = SPI_PLL_GetPMSValue("VPLL", &PLLCtrlInfo, SPI_PLL_CalculateFreq_VPLL, TargetClock, &PDIV, &MDIV, &SDIV);
	if(Status == UART_NO_ERROR)
	{
		UART_Printf("\nPDIV(%d), MDIV(%d), SDIV(%d)", PDIV, MDIV, SDIV);
		
		SYSC_StartPLLbyMPS(eVPLL, MDIV, PDIV, SDIV);
		while(!SYSC_IsLockDetected(eVPLL))
		{
			/* Wait until PLL is locked. */
		}
		
		SYSC_UpdateClkInform();
	}
}


typedef void (*SPI_fnDisplaySPI_PLL_t)(void);
typedef void (*SPI_fnSetSPI_PLL_t)(void);

typedef struct {
	SPI_fnDisplaySPI_PLL_t	m_DisplayPLL;
	SPI_fnSetSPI_PLL_t		m_SetPLL;
} SPI_SetPLLEntry_st;

SPI_SetPLLEntry_st gSPI_SetPLLEntry[] = 
{
	{SPI_DisplayPMS_APLL, SPI_SetPMS_APLL},
	{SPI_DisplayPMS_MPLL, SPI_SetPMS_MPLL},
	{SPI_DisplayPMS_EPLL, SPI_SetPMS_EPLL},
	{SPI_DisplayPMS_VPLL, SPI_SetPMS_VPLL}
};


void SPI_SetPLL(SPI_CtrlInfo_st *pCtrlInfo)
{
	s32					MenuEntryCnt;
	s32					MenuIndx;
	s32					SelNum;
	SPI_SetPLLEntry_st	*pSetPLLEntry;

	MenuEntryCnt = sizeof(gSPI_SetPLLEntry)/sizeof(SPI_SetPLLEntry_st);
	while(1)
	{
		UART_Printf("\n====================================================");
		UART_Printf("\n   SET APLL/MPLL/EPLL/VPLL");
		UART_Printf("\n====================================================\n");
		for(MenuIndx = 0; MenuIndx<MenuEntryCnt; MenuIndx++)
		{
			pSetPLLEntry = &gSPI_SetPLLEntry[MenuIndx];

			UART_Printf("\n%2d. ", MenuIndx);
			if(pSetPLLEntry->m_DisplayPLL != NULL)
			{
				(*pSetPLLEntry->m_DisplayPLL)();
			}

			if(pSetPLLEntry->m_SetPLL == NULL)
			{
				UART_Printf("\t(Display Only)");
			}
		}
		UART_Printf("\n%2d. Exit", MenuIndx);
		UART_Printf("\n====================================================");

		UART_Printf("\nSelect: ");
		SelNum = UART_GetIntNum();
		//if(SelNum <= 0 || SelNum >= MenuEntryCnt)
		if(SelNum == MenuEntryCnt)	/* Out ouf entry range */
		{
			break;
		}

		if(SelNum >= 0 && SelNum < MenuEntryCnt)
		{
			pSetPLLEntry = &gSPI_SetPLLEntry[SelNum];
			if(pSetPLLEntry->m_SetPLL != NULL)
			{
				(*pSetPLLEntry->m_SetPLL)();
			}
		}
	}
}


