/**************************************************************************************
*
*	Project Name : S5PV210 Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for verifying functions of the S5PV210
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------
* 
*	File Name : spi.h
*  
*	File Description : This file implements the API functons for SPI
*
*	Author	: Park, HeeSang
*	Dept. : AP Development Team
*	Created Date : 2009/08/31
*	Version : 0.3
*
*	Author	: Jongseok,Park
*	Dept. : AP Development Team
*	Created Date : 2009/04/16
*	Version : 0.2
*
*	Author : Sung-Hyun, Na
*	Dept. : AP Development Team
*	Created Date : 2008/11/03
*	Version : 0.1 
* 
*	History
*	- Version 0.1
*           ->Created(Sung-Hyun, Na 2008/11/03) for S5PC100  Validation
*	- Version 0.2
*           -> Modified(JongSeok,Park 2009/03/25) for S5PV210 FPGA verification
*	- Version 0.3
*           -> Modified(Park,Heesang 2009/08/31) for S5PV210 Validation.
**************************************************************************************/



#ifndef __SPI_H__
#define __SPI_h__

#include "dma.h"

#define	MAX_SPI_PORT				3
#define	MAX_SPI_DMA_CHANNEL		16

#define	SPI_BUFFER_SIZE			8		// for debugging
//#define	SPI_BUFFER_SIZE			32		// for debugging


#define	SPI_DEBUG					1

#if SPI_DEBUG
#define	SPI_ASSERT()	{						\
							int LoopCnt = 0;		\
							while(LoopCnt <10)	\
							{					\
								LoopCnt = 0;		\
							}					\
						}
#else
#define	SPI_ASSERT()
#endif /* SPI_DEBUG */

#define	SPI_NO_ERROR				0
#define	SPI_ERROR					(-1)

#define	SPI_ENABLE					TRUE
#define	SPI_DISABLE				FALSE

#define	SPI_BUF						(0x41000000)
#define	SPI_BUF_SIZE				(0x100000)		/* 1MB */


// SPI SFR Control Structure
typedef struct _SPI_SFR 
{
	u32 CH_CFG;			// 0x00 spi configuration register
	u32 CLK_CFG;			// 0x04 clock configuration register
	u32 MODE_CFG;			// 0x08 spi fifo control register
	u32 CS_REG;				// 0x0C slave selection signal
	u32 SPI_INT_EN;			// 0x10 spi interrupt enable register
	u32 SPI_STATUS;			// 0x14 spi status register
	u32 SPI_TX_DATA;		// 0x18 spi tx data register
	u32 SPI_RX_DATA;		// 0x1C spi rx data register
	u32 PACKET_CNT_REG;	// 0x20 count how many data master gets.
	u32 PENDING_CLR_REG;	// 0x24 pending clear register
	u32 SWAP_CFG;			// 0x28 swap config register
	u32 FB_CLK_SEL;			// 0x2C feedback clock config register
} SPI_SFR;

//SPI Port
typedef enum
{
	SPI_PORT0 = 0,
	SPI_PORT1 = 1,
	SPI_PORT2 = 2
}SPI_PORT;

// SPI Master/Slave Selection
typedef enum {
	SPI_MASTER = 0,
	SPI_SLAVE,
} SPI_OpMode;

// SPI Clock Polarity.
typedef enum {
	SPI_ACTIVE_HIGH = 0,
	SPI_ACTIVE_LOW = 1,
} SPI_CPOL;

// SPI Clock Phase
typedef enum {
	SPI_FORMAT_A = 0,
	SPI_FORMAT_B = 1,
} SPI_CPHA;

// SPI Clock Source Selection
typedef enum {
	SPI_PCLK = 0,
	SPI_EXT_CLK = 1
} SPI_clock_selection;

// SPI DMA mode Structure.
typedef enum {
	SPI_DMA_SINGLE = 0,
	SPI_DMA_4_BURST
} SPI_DMA_type;

//SPI Interrupt Condition
typedef enum{
	SPI_INT_TRAILING = (1<<6),
 	SPI_INT_RX_OVERRUN = (1<<5),
	SPI_INT_RX_UNDERRUN = (1<<4),
	SPI_INT_TX_OVERRUN = (1<<3),
	SPI_INT_TX_UNDERRUN = (1<<2),
	SPI_INT_RX_FIFORDY = (1<<1),
	SPI_INT_TX_FIFORDY = (1<<0)
} SPI_IntrModeBit_et;

typedef enum{
	SPI_INT_CLR_TX_UNDERRUN = (1<<4),
	SPI_INT_CLR_TX_OVERRUN = (1<<3),
	SPI_INT_CLR_RX_UNDERRUN = (1<<2),
 	SPI_INT_CLR_RX_OVERRUN = (1<<1),
	SPI_INT_CLR_TRAILING = (1<<0)
}SPI_IntClrModeBit_et;

typedef enum{
	SPI_STATUS_TX_DONE = (1<<25),
	SPI_STATUS_TRAILING_BYTE = (1<<24),
 	SPI_STATUS_RX_OVERRUN = (1<<5),
	SPI_STATUS_RX_UNDERRUN = (1<<4),
	SPI_STATUS_TX_OVERRUN = (1<<3),
	SPI_STATUS_TX_UNDERRUN = (1<<2),
	SPI_STATUS_RX_FIFORDY = (1<<1),
	SPI_STATUS_TX_FIFORDY = (1<<0)
}SPI_StatusBit_et;


// SPI Transfer Data Type.
typedef enum {
	SPI_BYTE=0,
	SPI_HWORD=1,
	SPI_WORD=2
} SPI_transfer_data_type;


// SPI Feedback Clock Delay Value.
typedef enum {
	//New CLock Feedback (for RX) Schem 
	SPI_PHASE_DELAY0 = 0x00,	
	SPI_PHASE_DELAY90 = 0x01,	
	SPI_PHASE_DELAY180 = 0x02,	
	SPI_PHASE_DELAY270 = 0x03,		

	//Old Schem (hidden[2])
	SPI_0NS_DELAY = (0x00+4),	
	SPI_3NS_DELAY = (0x01+4),	
	SPI_6NS_DELAY = (0x02+4),
	SPI_9NS_DELAY = (0x03+4)
} SPI_feedbackClock;

// SPI Transfer mode
typedef enum {
	SPI_INTERRUPT_MODE = 0,
	SPI_DMA_MODE = 1
} SPI_transfer_mode;

typedef enum {
	SPI_EXT_CLKSRC_XXTI=0,
	SPI_EXT_CLKSRC_XUSBXTI=1,
	SPI_EXT_CLKSRC_HDMI27M=2,
	SPI_EXT_CLKSRC_USBPHY0=3,
	SPI_EXT_CLKSRC_USBPHY1=4,
	SPI_EXT_CLKSRC_HDMIPHY=5,
	SPI_EXT_CLKSRC_MPLL=6,
	SPI_EXT_CLKSRC_EPLL=7,
	SPI_EXT_CLKSRC_VPLL=8
} SPI_EXT_CLKSRC_et;


typedef enum {
	SPI_NSSOUT_ACTIVE = 0,
	SPI_NSSOUT_INACTIVE = 1
} SPI_NSSOUT_et;

typedef enum
{
	SPI_DMA0 = 0,
	SPI_DMA1
} SPI_DMA_et;


typedef struct {
	u32		m_StartMemAddr;
	u32		m_TotalSize;
	u32		m_CurMemAddr;		/* I'll use the Non Cacheable Memory Area as Ring-Buffer. */
} SPI_NCBufCtrlInfo_st;


typedef struct {
	u8 				*m_pBufPtr;			/* Start of buffer pointer */
	u8 				*m_pCurPtr;			/* Current buffer pointer */
	u32				m_BufSize;			/* Buffer size */	
	u32				m_ReqDataSize;		/* Requested data size */
	u32				m_RemainDataSize;	/* not treated, Remain data size */
} SPI_BufAllocInfo_st;


typedef void (*SPI_IntHandler_t)(void) __irq;
typedef void (*SPI_Callback_t)(u32 Value2);

typedef struct {
	SPI_PORT				m_SPIPortIdx;			/* SPI0, SPI1, SPI2 */
	SPI_SFR					*m_SPISFRBase;			/* SFR Base Register */

	//CH_CFG
	bool						m_EnHighSpeed;			/* Enable High Speed in Salve Tx */
	SPI_OpMode				m_OpMode;				/* Master or Slave Mode */
	SPI_CPOL				m_CPOL;				/* CPOL */
	SPI_CPHA				m_CPHA;				/* CPHA */
	bool						m_EnRxChannel;			/* Enable Tx Channel */
	bool						m_EnTxChannel;			/* Enalbe Rx Channel */
	
	//CLK_CFG
	SPI_clock_selection		m_IsUseExtClk;			/* PCLK or EXT_CLK */
	bool						m_EnClk;				/* Clock Enable/Disable */
	u32						m_ClkDivValue;			/* SPI Scaler : clock-out division rate */
	SPI_EXT_CLKSRC_et		m_ExtClkSrc;			/* XTI, USBXTI, ... */
	u32						m_ExtClkSrcDiv;			/* 1 ~ 16 */
	u32						m_OpClock;				/* Opeation clock */
	u32						m_TargetSPICLK;		/* Target Clock */

	//MODE_CFG
	SPI_transfer_data_type		m_ChWidth;				/* BYTE, HWORD, WORD */
	u32 						m_TraillingCnt;			/* Trailing Bytes threshold value in Rx FIFO */
	SPI_transfer_data_type		m_BusWidth;			/* BYTE, HWORD, WORD */
	u32						m_RxReadyLevel;		/* Rx FIFO trigger level */
	u32						m_TxReadyLevel;		/* Tx FIFO trigger level */
	SPI_transfer_mode			m_RxTransferMode;		/* Interrupt mode, DMA mode */
	SPI_transfer_mode			m_TxTransferMode;		/* Interrupt mode, DMA mode */
	u32						m_MaxFIFOSize;			/* Port0 : 256 bytes, Port1/2 : 64 bytes */
	u32						m_FifoLevelGrid;		/* Port0 : 4 bytes, Port1/2 : 1 byte */
	SPI_DMA_type			m_DMAType;			/* Single or 4Burst */

	u32						m_TxTransferSize;
	u32						m_RxTransferSize;
	
	//CS_REG
	u32						m_NCSTimeCnt;			/* NCS Time Count */
	bool						m_EnAutoCS;			/* Manual or Auto Chip Selection */
	//SPI_NSSOUT_et			m_NSSOut;				/* Active or Inactive in Manual mode Chip Selection */

	//INT_EN
	bool						m_EnTrailingInt;			/* Enable Trailing Interrupt */
	bool						m_EnRxOverrunInt;		/* Enable Rx Overrun Interrupt */ 
	bool						m_EnRxUnderrunInt;		/* Enable Rx Underrun Interrupt */ 
	bool						m_EnTxOverrunInt;		/* Enable Tx Overrun Interrupt */ 
	bool						m_EnTxUnderrunInt;		/* Enable Tx Underrun Interrupt */ 
	//bool					m_EnRxFifoReadyInt;		/* Enable Rx FIFO Ready Interrupt */ 
	//bool					m_EnTxFifoReadyInt;		/* Enable Tx FIFO Ready Interrupt */ 
	
	//PACKET_CNT_REG
	bool						m_EnPacketCnt;
	u32						m_PacketCntValue;

	//SWAP_CFG
	bool						m_EnRxHWordSwap;
	bool						m_EnRxByteSwap;
	bool						m_EnRxBitSwap;
	bool						m_EnRxSwap;
	bool						m_EnTxHWordSwap;
	bool						m_EnTxByteSwap;
	bool						m_EnTxBitSwap;
	bool						m_EnTxSwap;
	
	//FB_CLK_SEL
	SPI_feedbackClock			m_FBClkSel;				/* Select Feedback Clock */
	
	// Drive Strength
	u32						m_DrvStrength;			/* Drive Strength */
	bool						m_EnSWReset;		/* Reset in InitializeREG() */

	//bool					m_EnSPIInt;
	u32						m_SPIIntNum;			/* store interrupt Number */
	SPI_IntHandler_t			m_fnISR_SPI;			/* SPI interrupt handler. */

	SPI_BufAllocInfo_st		m_RxBufNode;
	SPI_BufAllocInfo_st		m_TxBufNode;

	u32						m_TxDMANum;
	u32						m_RxDMANum;
	
	DMA_CH					m_TxDmaCh;
	DMA_CH					m_RxDmaCh;
	DREQ_SRC				m_DmaTxReqSrc;
	DREQ_SRC				m_DmaRxReqSrc;
	DMAC					m_SPITxDma;				/* TX DMA Request Descriptor */
	DMAC					m_SPIRxDma;				/* RX DMA Request Descriptor */

	SPI_Callback_t			m_RxDMACompleteCallback;
	u32						m_RxDMACompleteCallback_Value;

	SPI_Callback_t			m_TxDMACompleteCallback;
	u32						m_TxDMACompleteCallback_Value;

	SPI_Callback_t			m_TrailingINT_Callback;
	u32						m_TrailingINT_Value;

	SPI_Callback_t			m_RxOverrunINT_Callback;
	u32						m_RxOverrunINT_Value;

	SPI_Callback_t			m_RxUnderrunINT_Callback;
	u32						m_RxUnderrunINT_Value;

	SPI_Callback_t			m_TxOverrunINT_Callback;
	u32						m_TxOverrunINT_Value;

	SPI_Callback_t			m_TxUnderrunINT_Callback;
	u32						m_TxUnderrunINT_Value;

	SPI_Callback_t			m_RxFIFORdyINT_Callback;
	u32						m_RxFIFORdyINT_Value;

	SPI_Callback_t			m_TxFIFORdyINT_Callback;
	u32						m_TxFIFORdyINT_Value;
	
	SPI_Callback_t			m_RxIntCompleteCallback;
	u32						m_RxIntCompleteCallback_Value;

	SPI_Callback_t			m_TxIntCompleteCallback;
	u32						m_TxIntCompleteCallback_Value;
} SPI_PortInfo_st;


typedef struct {
	SPI_PortInfo_st			m_SPIPortDefaultInfo[MAX_SPI_PORT];		/* default value */
	SPI_PortInfo_st			m_SPIPortTestInfo[MAX_SPI_PORT];		/* Reference value */

	SPI_NCBufCtrlInfo_st		m_NCBufCtrlInfo;
	bool						m_EnPrintInISR;							/* Enable print function in ISR */
} SPI_CtrlInfo_st;


extern SPI_CtrlInfo_st	gSPI_CtrlInfo;



s32 SPI_CalcTargetClk(SPI_clock_selection ClockSelect, u32 TargetClk,
	SPI_EXT_CLKSRC_et ExtClkSrc, u32 ExtClkSrcDiv, u32 *pOpClock, u32 *pClockDivValue);
s32 SPI_UpdateSPIClk(SPI_PortInfo_st *pPortInfo);
void SPI_GetExtClkHz(SPI_EXT_CLKSRC_et ExtClkSrc, u32 *pExtClkHz);

void SPI_SetIsrFunc(SPI_PortInfo_st *pPortInfo, u32 uIntrMode, void (*fnIsrFunc)(u32), u32 uIsrVar);
void SPI_IsrRxFifoRdy(u32 uVar);
void SPI_IsrTxFifoRdy(u32 uVar);
void SPI_IsrTxUnder(u32 uVar);
void SPI_IsrTxOver(u32 uVar);
void SPI_IsrRxOver(u32 uVar);
void SPI_IsrRxUnder(u32 uVar);
void SPI_IsrRxTrailing(u32 uVar);
void SPI_interruptHandler( SPI_PortInfo_st* pPortInfo, int interruptNumber);
void __irq Isr_SPI0( void );
void __irq Isr_SPI1( void );
void __irq Isr_SPI2( void );

void __irq SPI_DMA0Handler( void );
void __irq SPI_DMA1Handler( void );

s32 SPI_OpenPort(SPI_PortInfo_st *pPortInfo);
void SPI_ClosePort(SPI_PortInfo_st *pPortInfo);

void SPI_GPIOPortSet(SPI_PORT ePort, u32 DrvStrength);


void SPI_SWReset(SPI_PortInfo_st *pPortInfo);
void SPI_EnChannel(SPI_PortInfo_st *pPortInfo, bool EnTxChannel, bool EnRxChannel);
	
void SPI_SetClkSource(SPI_PortInfo_st *pPortInfo, SPI_clock_selection eClkSel);
void SPI_ChipSelect(SPI_PortInfo_st *pPortInfo, SPI_NSSOUT_et  NSSOut);
void SPI_EnIntr(SPI_PortInfo_st *pPortInfo, u32 uIntrMode, u8 usEn);
void SPI_EnPktCnt(SPI_PortInfo_st *pPortInfo, bool En, u32 CntVal);
void SPI_ClrPending(SPI_PortInfo_st *pPortInfo, SPI_IntClrModeBit_et IntClrBit);

void SPI_SFR_test(SPI_CtrlInfo_st *pCtrlInfo);

void SPI_Init(SPI_CtrlInfo_st *pSPICtrlInfo);
void SPI_SetDefaultSystemEnv(void);
void SPI_SetNonSecure(void);
bool SPI_IsInterruptPrintValid(void);
void SPI_InitDefaultValue(SPI_CtrlInfo_st *pSPICtrlInfo);

void SPI_SetSFRBase(SPI_PortInfo_st *pPortInfo);

s32 SPI_CheckLoopbackMasterAndChannel(SPI_PortInfo_st *pPortAInfo, SPI_PortInfo_st *pPortBInfo,
	SPI_PortInfo_st **pMasterPort, SPI_PortInfo_st **pSlavePort, SPI_PortInfo_st **pTxPort, SPI_PortInfo_st **pRxPort);

void SPI_DisplayError(void);



s32 SPI_SetPort0Info(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_SetPort1Info(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_SetPort2Info(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_PollingMasterTxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_PollingMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_PollingMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_PollingMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_PollingMasterRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_PollingMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_PollingMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_PollingMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_PollingSlaveTxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_PollingSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_PollingSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_PollingSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_PollingSlaveRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_PollingSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_PollingSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_PollingSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_PollingMasterTxRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_PollingMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_PollingMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_PollingMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_PollingSlaveTxRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_PollingSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_PollingSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_PollingSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);

void SPI_MasterRxCompleteCallback(u32 var0);
void SPI_MasterTxCompleteCallback(u32 var0);
void SPI_SlaveRxCompleteCallback(u32 var0);
void SPI_SlaveTxCompleteCallback(u32 var0);

s32 SPI_PortA_InterruptMasterTxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_InterruptMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_InterruptMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_InterruptMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_InterruptMasterRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_InterruptMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_InterruptMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_InterruptMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_InterruptSlaveTxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_InterruptSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_InterruptSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_InterruptSlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_InterruptSlaveRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_InterruptSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_InterruptSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_InterruptSlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_InterruptMasterTxRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_InterruptMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_InterruptMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_InterruptMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_InterruptSlaveTxRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_InterruptSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_InterruptSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_InterruptSlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);


s32 SPI_PortA_DMAMasterTxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_DMAMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_DMAMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_DMAMasterTxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_DMAMasterRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_DMAMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_DMAMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_DMAMasterRxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_DMASlaveTxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_DMASlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_DMASlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_DMASlaveTxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_DMASlaveRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_DMASlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_DMASlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_DMASlaveRxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_DMAMasterTxRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_DMAMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_DMAMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_DMAMasterTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PortA_DMASlaveTxRxTest(SPI_PortInfo_st *pPortA);
s32 SPI_Port0_DMASlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port1_DMASlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Port2_DMASlaveTxRxTest(SPI_CtrlInfo_st *pCtrlInfo);


s32 SPI_Loopback_PortAPortBPollingTest(SPI_PortInfo_st *pPortAInfo, SPI_PortInfo_st *pPortBInfo);
s32 SPI_Loopback_Port0Port1PollingTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Loopback_Port0Port2PollingTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Loopback_Port1Port2PollingTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_Loopback_PortAPortBInterruptTest(SPI_PortInfo_st *pPortAInfo, SPI_PortInfo_st *pPortBInfo);
s32 SPI_Loopback_Port0Port1InterruptTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Loopback_Port0Port2InterruptTest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Loopback_Port1Port2InterruptTest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_Loopback_PortAPortBDMATest(SPI_PortInfo_st *pPortAInfo, SPI_PortInfo_st *pPortBInfo);
s32 SPI_Loopback_Port0Port1DMATest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Loopback_Port0Port2DMATest(SPI_CtrlInfo_st *pCtrlInfo);
s32 SPI_Loopback_Port1Port2DMATest(SPI_CtrlInfo_st *pCtrlInfo);

s32 SPI_PollingMasterTx(SPI_PortInfo_st *pPortInfo);
s32 SPI_PollingSlaveRx(SPI_PortInfo_st *pPortInfo);
s32 SPI_PollingMasterRx(SPI_PortInfo_st *pPortInfo);
s32 SPI_PollingSlaveTx(SPI_PortInfo_st *pPortInfo);
s32 SPI_PollingMasterTxRx(SPI_PortInfo_st *pPortInfo);
s32 SPI_PollingSlaveTxRx(SPI_PortInfo_st *pPortInfo);
s32 SPI_IntTxFifoRdy_MasterTx(SPI_PortInfo_st *pPortInfo);

s32 SPI_InterruptMasterTx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar);
void SPI_IntRxFifoRdy_MasterRx(SPI_PortInfo_st *pPortInfo);
s32 SPI_InterruptMasterRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar);

s32 SPI_InterruptMasterTxRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t TxcallbackFn, u32 TxCallBackVar,
	SPI_Callback_t RxcallbackFn, u32 RxCallBackVar);
void SPI_IntTxFifoRdy_SlaveTx(SPI_PortInfo_st *pPortInfo);
s32 SPI_InterruptSlaveTx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar);

s32 SPI_IntRxFifoRdy_SlaveRx(SPI_PortInfo_st *pPortInfo);
s32 SPI_InterruptSlaveRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar);
s32 SPI_InterruptSlaveTxRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t TxcallbackFn, u32 TxCallBackVar,
	SPI_Callback_t RxcallbackFn, u32 RxCallBackVar);

s32 SPI_DmaMasterTx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar);
s32 SPI_DmaMasterRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar);
s32 SPI_DmaMasterTxRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t TxcallbackFn, u32 TxCallBackVar,
	SPI_Callback_t RxcallbackFn, u32 RxCallBackVar);

s32 SPI_DmaSlaveTx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar);
s32 SPI_DmaSlaveRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar);
s32 SPI_DmaSlaveTxRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t TxcallbackFn, u32 TxCallBackVar,
	SPI_Callback_t RxcallbackFn, u32 RxCallBackVar);


int SPI_CompareData(u8* data1, u8* data2, u32 bytes);

void SPI_CreateNCBufCtrl(SPI_NCBufCtrlInfo_st *pNCBufCtrlInfo, u32 StartAddr, u32 TotalSize);
void SPI_DeleteNCBufCtrl(UART_NCBufCtrlInfo_st *pNCBufCtrlInfo);

u8 *SPI_GetNCBuf(SPI_BufAllocInfo_st *pBufAllocInfo, u32 BufSize);
int SPI_FreeNCBuf(SPI_BufAllocInfo_st *pBufAllocInfo);
int SPI_IncNCValidDataSize(SPI_BufAllocInfo_st *pBufAllocInfo, u32 Size);
int SPI_NCBufPatternInit(SPI_BufAllocInfo_st *pBufAllocInfo, char *pPattern, int PatternSize);
int SPI_NCBufRandInit(SPI_BufAllocInfo_st *pBufAllocInfo, u32 BufSize);
int SPI_NCBufZeroInit(SPI_BufAllocInfo_st *pBufAllocInfo);
void SPI_NCBufDisplay(SPI_BufAllocInfo_st *pBufAllocInfo, bool EnDisplay);
void SPI_NCBufDisplayHex(SPI_BufAllocInfo_st *pBufAllocInfo, bool EnDisplay);

int SPI_BufPatternInit(void *pBuf, u32 BufSize, char *pPattern, int PatternSize);
int SPI_BufRandInit(void *pBuf, u32 BufSize);
int SPI_BufZeroInit(void *pBuf, u32 BufSize);
void SPI_BufDisplay(u8 *pBuf, u32 BufSize, bool EnDisplay);
void SPI_BufDisplayHex(u8 *pBuf, u32 BufSize, bool EnDisplay);

void SPI_DisplayVar_EnHighSpeed(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_OpMode(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_CPOL(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_CPHA(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnRxChannel(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnTxChannel(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_IsUseExtClk(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnClk(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_ExtClkSrc(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_ExtClkSrcDiv(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_OpClock(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_ChWidth(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_TrailingCnt(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_BusWidth(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_RxTriggerLevel(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_TxTriggerLevel(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_RxTransferMode(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_TxTransferMode(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_DMAType(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_TxTransferSize(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_RxTransferSize(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_NCSTimeCnt(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnAutoCS(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnTrailingInt(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnRxOverrunInt(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnRxOverrunInt(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnRxUnderrunInt(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnTxOverrunInt(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnTxUnderrunInt(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnRxHwordSwap(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnRxByteSwap(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnRxBitSwap(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnRxSwap(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnTxHwordSwap(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnTxByteSwap(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnTxBitSwap(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnTxSwap(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_FBClkSel(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_DrvStrength(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_SpiCLK(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_RxDMANum(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_TxDMANum(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_RxDMACh(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_TxDMACh(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_LoadDefault(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnSWReset(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_SWReset(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_ReadRegister(SPI_PortInfo_st *pPortInfo);
void SPI_DisplayVar_EnInterruptPrint(SPI_PortInfo_st *pPortInfo);

void SPI_SetVar_EnHighSpeed(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_OpMode(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_CPOL(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_CPHA(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnRxChannel(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnTxChannel(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_IsUseExtClk(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnClk(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_SpiCLK(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_ExtClkSrc(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_ExtClkSrcDiv(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_ChWidth(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_TrailingCnt(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_BusWidth(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_RxTriggerLevel(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_TxTriggerLevel(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_RxTransferMode(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_TxTransferMode(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_DMAType(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_TxTransferSize(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_RxTransferSize(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_NCSTimeCnt(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnAutoCS(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnTrailingInt(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnRxOverrunInt(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnRxUnderrunInt(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnTxOverrunInt(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnTxUnderrunInt(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnRxHwordSwap(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnRxByteSwap(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnRxBitSwap(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnRxSwap(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnTxHwordSwap(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnTxByteSwap(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnTxBitSwap(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnTxSwap(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_FBClkSel(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_DrvStrength(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_RxDMACh(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_TxDMACh(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_LoadDefault(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnSWReset(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_SWReset(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_ReadRegister(SPI_PortInfo_st *pPortInfo);
void SPI_SetVar_EnInterruptPrint(SPI_PortInfo_st *pPortInfo);

void SPI_LEDOn(u32 BitNum);
void SPI_LEDOff(u32 BitNum);


#endif


