/**************************************************************************************
*
*	Project Name : S5PV210 FPGA Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for verifying functions of the S5PV210 FPGA
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------
* 
*	File Name : spi.c
*  
*	File Description : This file implements the API functons for SPI
*
*	Author	: Park,HeeSang
*	Dept. : AP Development Team
*	Created Date : 2009/08/31
*	Version : 0.3
*
*	Author	: Jongseok,Park
*	Dept. : AP Development Team
*	Created Date : 2009/04/16
*	Version : 0.2
*
*	Author : Sung-Hyun, Na
*	Dept. : AP Development Team
*	Created Date : 2008/11/03
*	Version : 0.1 
* 
*	History
*	- Version 0.1
*           ->Created(Sung-Hyun, Na 2008/11/03) for S5PV210  Validation
*	- Version 0.2
*           -> Modified(JongSeok,Park 2009/03/25) for S5PV210 FPGA verification
*	- Version 0.3
*           -> Modified(Park,Heesang 2009/08/31) for S5PV210 Validation.
**************************************************************************************/




#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "def.h"
#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "intc.h"
#include "gpio.h"
#include "sysc.h"
#include "spi.h"
#include "dma.h"


SPI_CtrlInfo_st	gSPI_CtrlInfo;

static SPI_PortInfo_st	*gSPI_SPIIsrPortInfo[MAX_SPI_PORT];
static SPI_PortInfo_st	*gSPI_DMAIsrPortInfo[MAX_SPI_DMA_CHANNEL];


#include "SfrReadWriten.h"

REGINFOn		sRegInfoSPI0[] = 
{ 
    //01234567890123                 
	{"rCH_CFG0 ",			SPI0_BASE+0x0, 	7-1, 	RW, 0x0 , DPPB, 0,0},		
	{"rCLK_CFG0 ",			SPI0_BASE+0x4,	10-1, 	RW, 0x0  , DPDB, 0,0},	
	{"rMODE_CFG0 ",			SPI0_BASE+0x8, 	31-1,	RW, 0x0  , DPPB, 0,0},		
	{"rCS_REG0 ",			SPI0_BASE+0xc,	10-1, 	RW, 0x1  , DPDB, 0,0},	
	{"rSPI_INT_EN0 ",			SPI0_BASE+0x10,	7-1, 	WO, 0x0  , DPDB, 0,0},	
	{"rSPI_STATUS0 ",			SPI0_BASE+0x14,	26-1,	RO, 0x0  , DPDB, 0,0},	
	{"rSPI_TX_DATA0 ",		SPI0_BASE+0x18,	32-1, 	WO, 0x0  , DPDB, 0,0},	
	{"rSPI_RX_DATA0 ",		SPI0_BASE+0x1c,	32-1, 	RO, 0x0  , DPDB, 0,0},		
	{"rPACKET_CNT_REG0 ",	SPI0_BASE+0x20,	17-1, 	RW, 0x0  , DPDB, 0,0},		
	{"rPENDING_CLR_REG0 ",	SPI0_BASE+0x24,	5-1, 	RW, 0x0  , DPDB, 0,0},			
	{"rSWAP_CFG0 ",			SPI0_BASE+0x28,	8-1, 	RW, 0x0  , DPDB, 0,0},			
	{"rFB_CLK_SEL0 ",			SPI0_BASE+0x2c,	2-1, 	RW, 0x0  , DPDB, 0,0}			
};

REGINFOn		sRegInfoSPI1[] = 
{ 
    //01234567890123                 
	{"rCH_CFG1 ",			SPI1_BASE+0x0, 	7-1, 	RW, 0x0 , DPPB, 0,0},		
	{"rCLK_CFG1 ",			SPI1_BASE+0x4,	10-1, 	RW, 0x0  , DPDB, 0,0},	
	{"rMODE_CFG1 ",			SPI1_BASE+0x8, 	31-1,	RW, 0x0  , DPPB, 0,0},		
	{"rCS_REG1 ",			SPI1_BASE+0xc,	10-1, 	RW, 0x1  , DPDB, 0,0},	
	{"rSPI_INT_EN1 ",			SPI1_BASE+0x10,	7-1, 	WO, 0x0  , DPDB, 0,0},	
	{"rSPI_STATUS1 ",			SPI1_BASE+0x14,	26-1,	RO, 0x0  , DPDB, 0,0},	
	{"rSPI_TX_DATA1 ",		SPI1_BASE+0x18,	32-1, 	WO, 0x0  , DPDB, 0,0},	
	{"rSPI_RX_DATA1 ",		SPI1_BASE+0x1c,	32-1, 	RO, 0x0  , DPDB, 0,0},		
	{"rPACKET_CNT_REG1 ",	SPI1_BASE+0x20,	17-1, 	RW, 0x0  , DPDB, 0,0},		
	{"rPENDING_CLR_REG1 ",	SPI1_BASE+0x24,	5-1, 	RW, 0x0  , DPDB, 0,0},			
	{"rSWAP_CFG1 ",			SPI1_BASE+0x28,	8-1, 	RW, 0x0  , DPDB, 0,0},			
	{"rFB_CLK_SEL1 ",			SPI1_BASE+0x2c,	2-1, 	RW, 0x0  , DPDB, 0,0}		
};

REGINFOn		sRegInfoSPI2[] = 
{ 
    //01234567890123                 
	{"rCH_CFG2 ",			SPI2_BASE+0x0, 	7-1, 	RW, 0x0 , DPPB, 0,0},		
	{"rCLK_CFG2 ",			SPI2_BASE+0x4,	10-1, 	RW, 0x0  , DPDB, 0,0},	
	{"rMODE_CFG2 ",			SPI2_BASE+0x8, 	31-1,	RW, 0x0  , DPPB, 0,0},		
	{"rCS_REG2 ",			SPI2_BASE+0xc,	10-1, 	RW, 0x1  , DPDB, 0,0},	
	{"rSPI_INT_EN2 ",			SPI2_BASE+0x10,	7-1, 	WO, 0x0  , DPDB, 0,0},	
	{"rSPI_STATUS2 ",			SPI2_BASE+0x14,	26-1,	RO, 0x0  , DPDB, 0,0},	
	{"rSPI_TX_DATA2 ",		SPI2_BASE+0x18,	32-1, 	WO, 0x0  , DPDB, 0,0},	
	{"rSPI_RX_DATA2 ",		SPI2_BASE+0x1c,	32-1, 	RO, 0x0  , DPDB, 0,0},		
	{"rPACKET_CNT_REG2 ",	SPI2_BASE+0x20,	17-1, 	RW, 0x0  , DPDB, 0,0},		
	{"rPENDING_CLR_REG2 ",	SPI2_BASE+0x24,	5-1, 	RW, 0x0  , DPDB, 0,0},			
	{"rSWAP_CFG2 ",			SPI2_BASE+0x28,	8-1, 	RW, 0x0  , DPDB, 0,0},			
	{"rFB_CLK_SEL2 ",			SPI2_BASE+0x2c,	2-1, 	RW, 0x0  , DPDB, 0,0}		
};


//should execute at first
void SPI_SFR_test(SPI_CtrlInfo_st *pCtrlInfo)
{
	TestSFRn(sRegInfoSPI0, sizeof(sRegInfoSPI0)/sizeof(REGINFOn));
	TestSFRn(sRegInfoSPI1, sizeof(sRegInfoSPI1)/sizeof(REGINFOn));
	TestSFRn(sRegInfoSPI2, sizeof(sRegInfoSPI2)/sizeof(REGINFOn));
}


void SPI_InitializeREG(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR			*pSPIRegs;
	u32 				TempReg;

	pSPIRegs = pPortInfo->m_SPISFRBase;


	/*******************************************************
		SETTING SEQUENCE OF SPECIAL FUNCTION REGISTER
	*******************************************************/
	/*

	1. Set Transfer Type. (CPOL & CPHA set)
	2. Set Clock configuration register.
	3. Set SPI MODE configuration register.
	4. Set SPI INT_EN register.
	5. Set PACKET_CNT_REG register if necessary.
	6. Set Tx or Rx Channel on.
	7. Set nSSout low to start Tx or Rx operation.
	    a. Set nSSout Bit to low, then start Tx data writing.
	    b. If auto chip selection bit is set, nSSout is controlled automatically.
	*/

	/* CH_CFG */
	Outp32(&pSPIRegs->CH_CFG, (pPortInfo->m_EnHighSpeed<<6) | (pPortInfo->m_OpMode<<4) |
		(pPortInfo->m_CPOL<<3) | (pPortInfo->m_CPHA<<2) | (0<<1) |
		(0<<0) );

	/* CLK_CFG */
	Outp32(&pSPIRegs->CLK_CFG, (pPortInfo->m_IsUseExtClk<<9) | (pPortInfo->m_EnClk<<8) |
		(pPortInfo->m_ClkDivValue<<0) );

	/* MODE_CFG */
	Outp32(&pSPIRegs->MODE_CFG, (pPortInfo->m_ChWidth<<29) | (pPortInfo->m_TraillingCnt<<19) |
		(pPortInfo->m_BusWidth<<17) | (pPortInfo->m_RxReadyLevel<<11) | (pPortInfo->m_TxReadyLevel<<5) |
		(pPortInfo->m_RxTransferMode<<2) | (pPortInfo->m_TxTransferMode<<1) | (pPortInfo->m_DMAType<<0) );

	/* CS_REG */
	Outp32(&pSPIRegs->CS_REG, (pPortInfo->m_NCSTimeCnt<<4) | (pPortInfo->m_EnAutoCS<<1) |
		(SPI_NSSOUT_INACTIVE<<0) );

	/* SPI_INT_EN */
	Outp32(&pSPIRegs->SPI_INT_EN, (pPortInfo->m_EnTrailingInt<<6) | (pPortInfo->m_EnRxOverrunInt<<5) |
		(pPortInfo->m_EnRxUnderrunInt<<4) | (pPortInfo->m_EnTxOverrunInt<<3) | (pPortInfo->m_EnTxUnderrunInt<<2));
		/* Except RxFIFOReadyInt and TxFIFOReadyInt. */

	/* PACKET_CNT_REG */
	Outp32(&pSPIRegs->PACKET_CNT_REG, (0<<6) | (0<<0) );	/* Reset Value : Disable the Packet Count to prevent the Master Rx Operation. */

	/* SWAP_CFG */
	Outp32(&pSPIRegs->SWAP_CFG, (pPortInfo->m_EnRxHWordSwap<<7) | (pPortInfo->m_EnRxByteSwap<<6) |
		(pPortInfo->m_EnRxBitSwap<<5) | (pPortInfo->m_EnRxSwap<<4) | (pPortInfo->m_EnTxHWordSwap<<3) |
		(pPortInfo->m_EnTxByteSwap<<2) | (pPortInfo->m_EnTxBitSwap<<1) | (pPortInfo->m_EnTxSwap<<0) );

	Outp32(&pSPIRegs->FB_CLK_SEL, (pPortInfo->m_FBClkSel<<0) );

	#if 0	/* Move to SPI_EnChannel() */
	if(pPortInfo->m_EnSWReset)
	{
		SPI_SWReset(pPortInfo);
	}
	#endif /* 0*/
}



//////////
// File Name : SPI_OpenPort
// File Description : This function create certain spi channel.
// Input : channel number, SPI_OpMode, CPOL, CPHA, SPI_clock_selection, clock frequency, channel size, bus size, DMA Type
// Output : SPI_Infor
// Version : 
//SPI_Infor * SPI_OpenPort(SPI_PORT ePort, SPI_Configuration* pConfiguration )
s32 SPI_OpenPort(SPI_PortInfo_st *pPortInfo)
{
	u32	OpClock;
	u32	ClockDivValue;
	
	SPI_GPIOPortSet(pPortInfo->m_SPIPortIdx, pPortInfo->m_DrvStrength);

	SPI_SetIsrFunc(pPortInfo, SPI_INT_TRAILING, SPI_IsrRxTrailing, (u32) pPortInfo);
	SPI_SetIsrFunc(pPortInfo, SPI_INT_RX_OVERRUN, SPI_IsrRxOver, (u32) pPortInfo);
	SPI_SetIsrFunc(pPortInfo, SPI_INT_RX_UNDERRUN, SPI_IsrRxUnder, (u32) pPortInfo);
	SPI_SetIsrFunc(pPortInfo, SPI_INT_TX_OVERRUN, SPI_IsrTxOver, (u32) pPortInfo);
	SPI_SetIsrFunc(pPortInfo, SPI_INT_TX_UNDERRUN, SPI_IsrTxUnder, (u32) pPortInfo);

	if(pPortInfo->m_OpMode == SPI_SLAVE && pPortInfo->m_EnTxChannel == TRUE)
	{
		/* In S5PV210 SPI User manual, this bit(INT_EN_TX_UNDERRUN) must be clear first after turning on slave Tx path. */
		pPortInfo->m_EnTxUnderrunInt = FALSE;
	}

	SPI_InitializeREG(pPortInfo);

	gSPI_SPIIsrPortInfo[pPortInfo->m_SPIPortIdx] = pPortInfo;
	gSPI_DMAIsrPortInfo[pPortInfo->m_TxDmaCh] = pPortInfo;
	gSPI_DMAIsrPortInfo[pPortInfo->m_RxDmaCh] = pPortInfo;

	INTC_SetVectAddr(NUM_PDMA0, SPI_DMA0Handler);
	INTC_SetVectAddr(NUM_PDMA1, SPI_DMA1Handler);

	/* Clear all garbage interrupt source */
	SPI_ClrPending(pPortInfo, (SPI_INT_CLR_TRAILING| SPI_INT_CLR_RX_OVERRUN| SPI_INT_CLR_RX_UNDERRUN |
		SPI_INT_CLR_TX_OVERRUN | SPI_INT_CLR_TX_UNDERRUN));

	INTC_Enable(pPortInfo->m_TxDMANum);
	INTC_Enable(pPortInfo->m_RxDMANum);

	#if 1
	INTC_SetVectAddr( pPortInfo->m_SPIIntNum, pPortInfo->m_fnISR_SPI);
	INTC_Enable( pPortInfo->m_SPIIntNum);
	#endif /* 0 */

	return SPI_NO_ERROR;
}

//////////
// File Name : SPI_close
// File Description : SPI_Infor close.
// Input : SPI_Infor
// Output : NONE.
// Version : 
#if 0
void SPI_close(SPI_PortInfo_st *pPortInfo)
{
}
#endif /* 0 */

void SPI_ClosePort(SPI_PortInfo_st *pPortInfo)
{
	INTC_Disable(pPortInfo->m_SPIIntNum);

	gSPI_SPIIsrPortInfo[pPortInfo->m_SPIPortIdx] = NULL;

	if(pPortInfo->m_TxTransferMode == SPI_DMA_MODE)
	{
		if(pPortInfo->m_EnTxChannel == TRUE)
		{
			DMA_StopDmac(&pPortInfo->m_SPITxDma);
		}	
	}

	if(pPortInfo->m_RxTransferMode == SPI_DMA_MODE)
	{
		if(pPortInfo->m_EnRxChannel == TRUE)
		{
			DMA_StopDmac(&pPortInfo->m_SPIRxDma);
		}
	}

	gSPI_DMAIsrPortInfo[pPortInfo->m_TxDmaCh] = NULL;
	gSPI_DMAIsrPortInfo[pPortInfo->m_RxDmaCh] = NULL;
}



//////////
// File Name : SPI_GPIOPortSet
// File Description : This function set GPIO fit on certain channel.
// Input : NONE
// Output : NONE.
// Version : 
void SPI_GPIOPortSet(SPI_PORT ePort, u32 DrvStrength)
{
	if ( ePort == SPI_PORT0 )
	{
		GPIO_SetFunctionEach( eGPIO_B, eGPIO_0, 0x2 );	// SPI0_CLK
		GPIO_SetFunctionEach( eGPIO_B, eGPIO_1, 0x2 );	// SPI0_CS

		GPIO_SetFunctionEach( eGPIO_B, eGPIO_2, 0x2 );	// SPI0_MISO
		GPIO_SetFunctionEach( eGPIO_B, eGPIO_3, 0x2 );	// SPI0_MOSI

		GPIO_SetPullUpDownEach( eGPIO_B, eGPIO_0, 0 );	
		GPIO_SetPullUpDownEach( eGPIO_B, eGPIO_1, 0 );	
		GPIO_SetPullUpDownEach( eGPIO_B, eGPIO_2, 0 );	
		GPIO_SetPullUpDownEach( eGPIO_B, eGPIO_3, 0 );	

		GPIO_SetDSEach(eGPIO_B, eGPIO_0, DrvStrength);
		GPIO_SetDSEach(eGPIO_B, eGPIO_1, DrvStrength);
		GPIO_SetDSEach(eGPIO_B, eGPIO_2, DrvStrength);
		GPIO_SetDSEach(eGPIO_B, eGPIO_3, DrvStrength);
	}
	else if ( ePort == SPI_PORT1 )
	{	 
		GPIO_SetFunctionEach( eGPIO_B, eGPIO_4, 0x2 );	// SPI1_CLK
		GPIO_SetFunctionEach( eGPIO_B, eGPIO_5, 0x2 );	// SPI1_CS
		GPIO_SetFunctionEach( eGPIO_B, eGPIO_6, 0x2 );	// SPI1_MISO
		GPIO_SetFunctionEach( eGPIO_B, eGPIO_7, 0x2 );	// SPI1_MOSI
	
		GPIO_SetPullUpDownEach( eGPIO_B, eGPIO_4, 0 );
		GPIO_SetPullUpDownEach( eGPIO_B, eGPIO_5, 0 );
		GPIO_SetPullUpDownEach( eGPIO_B, eGPIO_6, 0 );
		GPIO_SetPullUpDownEach( eGPIO_B, eGPIO_7, 0 );

		GPIO_SetDSEach(eGPIO_B, eGPIO_4, DrvStrength);
		GPIO_SetDSEach(eGPIO_B, eGPIO_5, DrvStrength);
		GPIO_SetDSEach(eGPIO_B, eGPIO_6, DrvStrength);
		GPIO_SetDSEach(eGPIO_B, eGPIO_7, DrvStrength);
	}
	else if ( ePort == SPI_PORT2 )
	{
		GPIO_SetFunctionEach( eGPIO_G2, eGPIO_0, 0x3 );	// SPI2_CLK
		GPIO_SetFunctionEach( eGPIO_G2, eGPIO_1, 0x3 );	// SPI2_CS
		GPIO_SetFunctionEach( eGPIO_G2, eGPIO_2, 0x3 );	// SPI2_MISO
		GPIO_SetFunctionEach( eGPIO_G2, eGPIO_3, 0x3 );	// SPI2_MOSI

		GPIO_SetPullUpDownEach( eGPIO_G2, eGPIO_0, 0 );
		GPIO_SetPullUpDownEach( eGPIO_G2, eGPIO_1, 0 );
		GPIO_SetPullUpDownEach( eGPIO_G2, eGPIO_2, 0 );
		GPIO_SetPullUpDownEach( eGPIO_G2, eGPIO_3, 0 );	

		GPIO_SetDSEach(eGPIO_G2, eGPIO_0, DrvStrength);
		GPIO_SetDSEach(eGPIO_G2, eGPIO_1, DrvStrength);
		GPIO_SetDSEach(eGPIO_G2, eGPIO_2, DrvStrength);
		GPIO_SetDSEach(eGPIO_G2, eGPIO_3, DrvStrength);
	}
}

	
void SPI_GetExtClkHz(SPI_EXT_CLKSRC_et ExtClkSrc, u32 *pExtClkHz)
{
	u32		ExtClkHz;

	ExtClkHz = 0;
	
	switch(ExtClkSrc)
	{
		case SPI_EXT_CLKSRC_XXTI:
			ExtClkHz = FIN;
			break;
		case SPI_EXT_CLKSRC_XUSBXTI:
			ExtClkHz = FIN;
			break;
		case SPI_EXT_CLKSRC_HDMI27M:
			ExtClkHz = CLK27M;
			break;
		case SPI_EXT_CLKSRC_USBPHY0:
			ExtClkHz = FIN;
			break;
		case SPI_EXT_CLKSRC_USBPHY1:
			ExtClkHz = FIN;
			break;
		case SPI_EXT_CLKSRC_HDMIPHY:
			ExtClkHz = CLK27M;;
			break;
		case SPI_EXT_CLKSRC_MPLL:
			ExtClkHz = g_uMPLL;
			break;
		case SPI_EXT_CLKSRC_EPLL:
			ExtClkHz = g_uEPLL;
			break;
		case SPI_EXT_CLKSRC_VPLL:
			ExtClkHz = g_uVPLL;
			break;
		default:
			SPI_ASSERT();
			break;
			
	}

	*pExtClkHz = ExtClkHz;
}

s32 SPI_CalcTargetClk(SPI_clock_selection ClockSelect, u32 TargetClk,
	SPI_EXT_CLKSRC_et ExtClkSrc, u32 ExtClkSrcDiv, u32 *pOpClock, u32 *pClockDivValue)
{
	u32		ucPrescaler;
	u32		uSetClock;
	double	dSourceClock;
	u32		ExtClkHz;

	switch(ClockSelect)
	{
		case SPI_PCLK:
			dSourceClock = g_uPclkPsys;
			break;

		case SPI_EXT_CLK:
			SPI_GetExtClkHz(ExtClkSrc, &ExtClkHz);
			if(ExtClkSrcDiv == 0)
			{
				SPI_ASSERT();
			}
			dSourceClock = ExtClkHz/ExtClkSrcDiv;
			break;

		default:
			SPI_ASSERT();
			break;
	}

	if(dSourceClock < TargetClk * 2)
	{
		UART_Printf("\nError : OpClk(%.2f MHz), TargetClk(%.2f MHz)", dSourceClock/1.0e6, TargetClk/1.0e6);

		UART_Printf("\n\nPress any key to continue....\n\n");

		while(!UART_GetKey())
		{
		}
		
		return SPI_ERROR;
	}

	ucPrescaler = (u32)(((double)dSourceClock)/((double)2*(double)TargetClk) - (double)1);
	ucPrescaler &= 0xFF;	/* low 8 bits */
	uSetClock = (u32) dSourceClock / (2*(ucPrescaler+1));
	if(uSetClock > TargetClk)
	{
		ucPrescaler++;
	}

	*pOpClock = dSourceClock;
	*pClockDivValue = ucPrescaler;

	return SPI_NO_ERROR;
}


s32 SPI_SelectExtClkSrc(SPI_PortInfo_st *pPortInfo)
{
	s32 Status;

	#if 0
	Status = SPI_CalcTargetClk(pPortInfo->m_IsUseExtClk, pPortInfo->m_TargetSPICLK, pPortInfo->m_ExtClkSrc,
		pPortInfo->m_ExtClkSrcDiv, &pPortInfo->m_OpClock, &pPortInfo->m_ClkDivValue);
	if(Status != SPI_ERROR)
	{
		return Status;
	}
	#endif /* 0 */

	switch(pPortInfo->m_SPIPortIdx)
	{
		case SPI_PORT0:
			SYSC_SetClkMuxState(eCLKMUX_SPI0_XXTI + pPortInfo->m_ExtClkSrc);
			SYSC_SetClkDiv(eCLKDIV_UART0, pPortInfo->m_ExtClkSrcDiv);
			break;
		case SPI_PORT1:
			SYSC_SetClkMuxState(eCLKMUX_SPI1_XXTI + pPortInfo->m_ExtClkSrc);
			SYSC_SetClkDiv(eCLKDIV_UART1, pPortInfo->m_ExtClkSrcDiv);
			break;
		case SPI_PORT2:
			SYSC_SetClkMuxState(eCLKMUX_SPI2_XXTI + pPortInfo->m_ExtClkSrc);
			SYSC_SetClkDiv(eCLKDIV_UART2, pPortInfo->m_ExtClkSrcDiv);
			break;

		default:
			SPI_ASSERT();
			break;
	}
}




s32 SPI_PollingMasterTx(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pTxBufNode;
	//u32				TempReg;


	//UART_Printf("\nIf you want to quit, press any key...");

	pTxBufNode = &pPortInfo->m_TxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	SPI_EnPktCnt(pPortInfo, SPI_ENABLE, pTxBufNode->m_ReqDataSize);	/* Enable Master Rx */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	//TempReg = Inp32(&pSFRBase->SPI_STATUS);
	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(pTxBufNode->m_RemainDataSize > 0)
		{
			#if 0
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}
			#endif /* 0 */

			//SPI_SWReset(pPortInfo);

			//SPI_EnPktCnt(pPortInfo, SPI_ENABLE, 1);	/* Enable Master Rx */
			//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);
			
			if(Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
				}

				Outp32(&pSFRBase->SPI_TX_DATA, *pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr++;
				pTxBufNode->m_RemainDataSize--;

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
				}
			}

			#if 0
			while(!(Inp32(&pSFRBase->SPI_STATUS) & 0x01))	/* Wait until Tx compledted. */
			{
				/* Wait forever */
			}
			#else
			//Delay(1);
			#endif /* 0 */

			#if 0
			/* NOT necessary */
			//SPI_EnChannel(pPortInfo, FALSE, FALSE);
			//SPI_EnPktCnt(pPortInfo, SPI_DISABLE, 0);	/* Enable Master Rx */
			#endif /* 0 */
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(pTxBufNode->m_RemainDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", pTxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pTxBufNode->m_RemainDataSize > 0)
		{
			#if 0
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}
			#endif /* 0 */

			//SPI_EnPktCnt(pPortInfo, SPI_ENABLE, 1);	/* Enable Master Rx */
			//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);
			if(Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
				}

				Outp32(&pSFRBase->SPI_TX_DATA, *(u16 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u16);
				pTxBufNode->m_RemainDataSize -= sizeof(u16);

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
				}
			}
			//SPI_EnChannel(pPortInfo, FALSE, FALSE);
			//SPI_EnPktCnt(pPortInfo, SPI_DISABLE, 0);	/* Disable Master Rx */
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(pTxBufNode->m_RemainDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", pTxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pTxBufNode->m_RemainDataSize > 0)
		{
			#if 0
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}
			#endif /* 0 */

			//SPI_EnPktCnt(pPortInfo, SPI_ENABLE, 1);	/* Enable Master Rx */
			//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);
			if(Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
				}

				Outp32(&pSFRBase->SPI_TX_DATA, *(u32 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u32);
				pTxBufNode->m_RemainDataSize -= sizeof(u32);

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
				}
			}
			//SPI_EnChannel(pPortInfo, FALSE, FALSE);
			//SPI_EnPktCnt(pPortInfo, SPI_DISABLE, 0);	/* Enable Master Rx */
		}
	}
	else
	{
		SPI_ASSERT();
	}
}



s32 SPI_PollingSlaveRx(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pRxBufNode;
	//u32				TempReg;

	
	//UART_Printf("\nIf you want to quit, press any key...");

	pRxBufNode = &pPortInfo->m_RxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	//TempReg = Inp32(&pSFRBase->SPI_STATUS);
	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(pRxBufNode->m_RemainDataSize > 0)
		{
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}

			#if 0
			if(Inp32(&pSFRBase->SPI_STATUS) & 0x02)	/* Rx FIFO Ready */
			#else
			if(Inp32(&pSFRBase->SPI_STATUS) & 0xFF800)	/* RX_FIFO_LVL [23:15]*/
			#endif /* 0 */
			{
				*pRxBufNode->m_pCurPtr = Inp32(&pSFRBase->SPI_RX_DATA);
				pRxBufNode->m_pCurPtr++;
				pRxBufNode->m_RemainDataSize--;
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(pRxBufNode->m_RemainDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nRxx Size(%d) isn't aligned by 2", pRxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pRxBufNode->m_RemainDataSize > 0)
		{
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}

			#if 0
			if(Inp32(&pSFRBase->SPI_STATUS) & 0x02)	/* Rx FIFO Ready */
			#else
			if(Inp32(&pSFRBase->SPI_STATUS) & 0xFF800)	/* RX_FIFO_LVL [23:15]*/
			#endif /* 0 */
			{
				*(u16 *)pRxBufNode->m_pCurPtr = (u16)Inp32(&pSFRBase->SPI_RX_DATA);
				pRxBufNode->m_pCurPtr += sizeof(u16);
				pRxBufNode->m_RemainDataSize -= sizeof(u16);
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(pRxBufNode->m_RemainDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nRx Size(%d) isn't aligned by 2", pRxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pRxBufNode->m_RemainDataSize > 0)
		{
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}

			#if 0
			if(Inp32(&pSFRBase->SPI_STATUS) & 0x02)	/* Rx FIFO Ready */
			#else
			if(Inp32(&pSFRBase->SPI_STATUS) & 0xFF800)	/* RX_FIFO_LVL [23:15]*/
			#endif /* 0 */
			{
				*(u32 *)pRxBufNode->m_pCurPtr = (u32)Inp32(&pSFRBase->SPI_RX_DATA);
				pRxBufNode->m_pCurPtr += sizeof(u32);
				pRxBufNode->m_RemainDataSize -= sizeof(u32);
			}
		}
	}
	else
	{
		SPI_ASSERT();
	}
}


s32 SPI_PollingMasterRx(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pRxBufNode;
	//u32					TempReg;

	
	//UART_Printf("\nIf you want to quit, press any key...");

	pRxBufNode = &pPortInfo->m_RxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	SPI_EnPktCnt(pPortInfo, SPI_ENABLE, pPortInfo->m_RxBufNode.m_ReqDataSize);	/* Enable Master Rx */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	//TempReg = Inp32(&pSFRBase->SPI_STATUS);
	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(pRxBufNode->m_RemainDataSize > 0)
		{
			#if 0
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}
			#endif /* 0 */

			//SPI_EnPktCnt(pPortInfo, SPI_ENABLE, 1);	/* Enable Master Rx */
			//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);
	
			if(  pPortInfo->m_EnAutoCS == FALSE  )
			{
				SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
			}

			#if 0
			while(!(Inp32(&pSFRBase->SPI_STATUS) & 0x02))	/* wait until Rx FIFO Ready */
			#else
			while(!(Inp32(&pSFRBase->SPI_STATUS) & 0xFF800))	/* RX_FIFO_LVL[23:15] */
			#endif /* 0 */
			{
				if(UART_GetKey())	/* forcely break by key input. */
				{
					break;
				}
			}

			if(pRxBufNode->m_RemainDataSize > 0)	/* To skip this routine when Trailing bytes interrupt */
			{
				*pRxBufNode->m_pCurPtr = Inp32(&pSFRBase->SPI_RX_DATA);
				pRxBufNode->m_pCurPtr++;
				pRxBufNode->m_RemainDataSize--;

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
				}

				//SPI_EnChannel(pPortInfo, FALSE, FALSE);
				//SPI_EnPktCnt(pPortInfo, SPI_DISABLE, 0);	/* Enable Master Rx */
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(pRxBufNode->m_RemainDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nRxx Size(%d) isn't aligned by 2", pRxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pRxBufNode->m_RemainDataSize > 0)
		{
			#if 0
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}
			#endif /* 0 */

			//SPI_EnPktCnt(pPortInfo, SPI_ENABLE, 1);	/* Enable Master Rx */
			//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

			if(  pPortInfo->m_EnAutoCS == FALSE  )
			{
				SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
			}

			#if 0
			while(!(Inp32(&pSFRBase->SPI_STATUS) & 0x02))	/* wait until Rx FIFO Ready */
			#else
			while(!(Inp32(&pSFRBase->SPI_STATUS) & 0xFF800))	/* RX_FIFO_LVL[23:15] */
			#endif /* 0 */
			{
				if(UART_GetKey())	/* forcely break by key input. */
				{
					break;
				}
			}

			if(pRxBufNode->m_RemainDataSize > 0)	/* To skip this routine when Trailing bytes interrupt */
			{
				*(u16 *)pRxBufNode->m_pCurPtr = (u16)Inp32(&pSFRBase->SPI_RX_DATA);
				pRxBufNode->m_pCurPtr += sizeof(u16);
				pRxBufNode->m_RemainDataSize -= sizeof(u16);

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
				}

				//SPI_EnChannel(pPortInfo, FALSE, FALSE);
				//SPI_EnPktCnt(pPortInfo, SPI_DISABLE, 0);	/* Enable Master Rx */
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(pRxBufNode->m_RemainDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nRx Size(%d) isn't aligned by 2", pRxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pRxBufNode->m_RemainDataSize > 0)
		{
			#if 0
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}
			#endif /* 0 */

			//SPI_EnPktCnt(pPortInfo, SPI_ENABLE, 1);	/* Enable Master Rx */
			//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

			if(  pPortInfo->m_EnAutoCS == FALSE  )
			{
				SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
			}

			#if 0
			while(!(Inp32(&pSFRBase->SPI_STATUS) & 0x02))	/* wait until Rx FIFO Ready */
			#else
			while(!(Inp32(&pSFRBase->SPI_STATUS) & 0xFF800))	/* RX_FIFO_LVL[23:15] */
			#endif /* 0 */
			{
				if(UART_GetKey())	/* forcely break by key input. */
				{
					break;
				}
			}

			if(pRxBufNode->m_RemainDataSize > 0)	/* To skip this routine when Trailing bytes interrupt */
			{
				*(u32 *)pRxBufNode->m_pCurPtr = (u32)Inp32(&pSFRBase->SPI_RX_DATA);
				pRxBufNode->m_pCurPtr += sizeof(u32);
				pRxBufNode->m_RemainDataSize -= sizeof(u32);

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
				}

				//SPI_EnChannel(pPortInfo, FALSE, FALSE);
				//SPI_EnPktCnt(pPortInfo, SPI_DISABLE, 0);	/* Enable Master Rx */
			}
		}
	}
	else
	{
		SPI_ASSERT();
	}
}


s32 SPI_PollingSlaveTx(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pTxBufNode;
	//u32					TempReg;


	//UART_Printf("\nIf you want to quit, press any key...");

	pTxBufNode = &pPortInfo->m_TxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	//TempReg = Inp32(&pSFRBase->SPI_STATUS);
	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(pTxBufNode->m_RemainDataSize > 0)
		{
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}

			if(Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				Outp32(&pSFRBase->SPI_TX_DATA, *pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr++;
				pTxBufNode->m_RemainDataSize--;
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(pTxBufNode->m_RemainDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", pTxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pTxBufNode->m_RemainDataSize > 0)
		{
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}

			if(Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				Outp32(&pSFRBase->SPI_TX_DATA, *(u16 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u16);
				pTxBufNode->m_RemainDataSize -= sizeof(u16);
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(pTxBufNode->m_RemainDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", pTxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pTxBufNode->m_RemainDataSize > 0)
		{
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}

			if(Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				Outp32(&pSFRBase->SPI_TX_DATA, *(u32 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u32);
				pTxBufNode->m_RemainDataSize -= sizeof(u32);
			}
		}
	}
	else
	{
		SPI_ASSERT();
	}
}




s32 SPI_PollingMasterTxRx(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pTxBufNode;
	SPI_BufAllocInfo_st	*pRxBufNode;
	//u32					TempReg;


	//UART_Printf("\nIf you want to quit, press any key...");

	pTxBufNode = &pPortInfo->m_TxBufNode;
	pRxBufNode = &pPortInfo->m_RxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	SPI_EnPktCnt(pPortInfo, SPI_ENABLE, pTxBufNode->m_ReqDataSize);	/* Enable Master Rx */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	//TempReg = Inp32(&pSFRBase->SPI_STATUS);
	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(pTxBufNode->m_RemainDataSize > 0)
		{
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}

			if(Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				//SPI_EnPktCnt(pPortInfo, SPI_ENABLE, 1);	/* Enable Master Rx */
				//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
				}

				Outp32(&pSFRBase->SPI_TX_DATA, *pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr++;
				pTxBufNode->m_RemainDataSize--;

				#if 0
				while(!(Inp32(&pSFRBase->SPI_STATUS) & 0x02))	/* wait until Rx FIFO Ready */
				#else
				while(!(Inp32(&pSFRBase->SPI_STATUS) & 0xFF800))	/* RX_FIFO_LVL[23:15] */
				#endif /* 0 */
				{
				}

				*pRxBufNode->m_pCurPtr = Inp32(&pSFRBase->SPI_RX_DATA);
				pRxBufNode->m_pCurPtr++;
				pRxBufNode->m_RemainDataSize--;

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
				}

				//SPI_EnChannel(pPortInfo, FALSE, FALSE);
				//SPI_EnPktCnt(pPortInfo, SPI_DISABLE, 0);	/* Enable Master Rx */
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(pTxBufNode->m_RemainDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", pTxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pTxBufNode->m_RemainDataSize > 0)
		{
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}

			if(Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
				}

				//SPI_EnPktCnt(pPortInfo, SPI_ENABLE, 1);	/* Enable Master Rx */
				//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

				Outp32(&pSFRBase->SPI_TX_DATA, *(u16 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u16);
				pTxBufNode->m_RemainDataSize -= sizeof(u16);

				while(!Inp32(&pSFRBase->SPI_STATUS) & 0x02)	/* wait until Rx FIFO Ready */
				{
				}

				*(u16 *)pRxBufNode->m_pCurPtr = (u16)Inp32(&pSFRBase->SPI_RX_DATA);
				pRxBufNode->m_pCurPtr += sizeof(u16);
				pRxBufNode->m_RemainDataSize -= sizeof(u16);

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
				}

				//SPI_EnChannel(pPortInfo, FALSE, FALSE);
				//SPI_EnPktCnt(pPortInfo, SPI_DISABLE, 0);	/* Enable Master Rx */
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(pTxBufNode->m_RemainDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", pTxBufNode->m_RemainDataSize);
			
			return SPI_ERROR;
		}

		while(pTxBufNode->m_RemainDataSize > 0)
		{
			if(UART_GetKey())	/* forcely break by key input. */
			{
				break;
			}

			if(Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
				}

				//SPI_EnPktCnt(pPortInfo, SPI_ENABLE, 1);	/* Enable Master Rx */
				//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

				Outp32(&pSFRBase->SPI_TX_DATA, *(u32 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u32);
				pTxBufNode->m_RemainDataSize -= sizeof(u32);

				while(!Inp32(&pSFRBase->SPI_STATUS) & 0x02)	/* wait until Rx FIFO Ready */
				{
				}
				*(u32 *)pRxBufNode->m_pCurPtr = (u32)Inp32(&pSFRBase->SPI_RX_DATA);
				pRxBufNode->m_pCurPtr += sizeof(u32);
				pRxBufNode->m_RemainDataSize -= sizeof(u32);

				if(  pPortInfo->m_EnAutoCS == FALSE  )
				{
					SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
				}

				//SPI_EnChannel(pPortInfo, FALSE, FALSE);
				//SPI_EnPktCnt(pPortInfo, SPI_DISABLE, 0);	/* Enable Master Rx */
			}
		}
	}
	else
	{
		SPI_ASSERT();
	}
}

s32 SPI_PollingSlaveTxRx(SPI_PortInfo_st *pPortInfo)
{
	/* This function should be replayed by SPI_PollingSlaveTx and SPI_PollingSlaveRx respectively.
	     Slave Tx is first, Master Tx/Rs is second and Slave Rx is third. */
	     
	SPI_ASSERT();
}

s32 SPI_IntTxFifoRdy_MasterTx(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pTxBufNode;
	u32					TxFifoCnt;
	u32					TxDataSize;
	//u32				TempReg;

	//SPI_LEDOn(6);

	pTxBufNode = &pPortInfo->m_TxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	TxFifoCnt = Inp32(&pSFRBase->SPI_STATUS);
	TxFifoCnt >>= 6;
	TxFifoCnt &= 0x1FF;
	TxDataSize = pPortInfo->m_MaxFIFOSize - TxFifoCnt * pPortInfo->m_FifoLevelGrid;
	if(pTxBufNode->m_RemainDataSize < TxDataSize)
	{
		TxDataSize = pTxBufNode->m_RemainDataSize;
	}

	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(TxDataSize > 0)
		{
			//if( Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				Outp32(&pSFRBase->SPI_TX_DATA, *pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr++;
				pTxBufNode->m_RemainDataSize--;
				TxDataSize--;
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(TxDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", TxDataSize);
			
			return SPI_ERROR;
		}

		while(TxDataSize > 0)
		{
			//if( Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				Outp32(&pSFRBase->SPI_TX_DATA, *(u16 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u16);
				pTxBufNode->m_RemainDataSize -= sizeof(u16);
				TxDataSize -= sizeof(u16);
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(TxDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", TxDataSize);
			
			return SPI_ERROR;
		}

		while(TxDataSize > 0)
		{
			if( Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				Outp32(&pSFRBase->SPI_TX_DATA, *(u32 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u32);
				pTxBufNode->m_RemainDataSize -= sizeof(u32);
				TxDataSize -= sizeof(u32);
			}
		}
	}
	else
	{
		SPI_ASSERT();
	}

	#if 1
	/* Disable Interrupt */
	if(pTxBufNode->m_RemainDataSize == 0)
	{
		SPI_EnIntr(pPortInfo, SPI_INT_TX_FIFORDY, SPI_DISABLE);
		if(  pPortInfo->m_EnAutoCS != TRUE  )
		{
			SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
		}

		if(pPortInfo->m_TxIntCompleteCallback != NULL)
		{
			(*pPortInfo->m_TxIntCompleteCallback)(pPortInfo->m_TxIntCompleteCallback_Value);
		}
	}
	#else
	SPI_EnIntr(pPortInfo, SPI_INT_TX_FIFORDY, SPI_DISABLE);
	#endif /* 0 */

	//SPI_LEDOff(6);
}


s32 SPI_InterruptMasterTx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar)
{
	if( !(pPortInfo->m_EnTxChannel == TRUE && pPortInfo->m_EnRxChannel != TRUE) )
	{
		SPI_ASSERT();
	}
	
	pPortInfo->m_TxIntCompleteCallback = callbackFn;
	pPortInfo->m_TxIntCompleteCallback_Value = CallBackVar;

	SPI_SetIsrFunc(pPortInfo, SPI_INT_TX_FIFORDY, SPI_IntTxFifoRdy_MasterTx, (u32)pPortInfo);	
	SPI_EnIntr(pPortInfo, SPI_INT_TX_FIFORDY, SPI_ENABLE);

	#if 0
	INTC_SetVectAddr( pPortInfo->m_SPIIntNum, pPortInfo->m_fnISR_SPI);
	INTC_Enable( pPortInfo->m_SPIIntNum);
	#endif /* 0 */

	SPI_EnPktCnt(pPortInfo, SPI_ENABLE, pPortInfo->m_RxBufNode.m_ReqDataSize);	/* Enable Master Rx */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	if(  pPortInfo->m_EnAutoCS != TRUE  )
	{
		SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
	}

	#if 0
	INTC_SetVectAddr( pPortInfo->m_SPIIntNum, pPortInfo->m_fnISR_SPI);
	INTC_Enable( pPortInfo->m_SPIIntNum);
	#endif /* 0 */
}


void SPI_IntRxFifoRdy_MasterRx(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pRxBufNode;
	u32					RxFifoCnt;
	u32					RxDataSize;

	pRxBufNode = &pPortInfo->m_RxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	RxFifoCnt = Inp32(&pSFRBase->SPI_STATUS);
	RxFifoCnt >>= 15;
	RxFifoCnt &= 0x1FF;
	RxDataSize = RxFifoCnt;
	if(pRxBufNode->m_RemainDataSize < RxFifoCnt)
	{
		RxDataSize = pRxBufNode->m_RemainDataSize;
	}

	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(RxDataSize > 0)
		{
			*pRxBufNode->m_pCurPtr = Inp32(&pSFRBase->SPI_RX_DATA);
			pRxBufNode->m_pCurPtr++;
			pRxBufNode->m_RemainDataSize--;
			RxDataSize--;
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(RxDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nRxx Size(%d) isn't aligned by 2", RxDataSize);
			
			return;
		}

		while(RxDataSize > 0)
		{
			*(u16 *)pRxBufNode->m_pCurPtr = (u16)Inp32(&pSFRBase->SPI_RX_DATA);
			pRxBufNode->m_pCurPtr += sizeof(u16);
			pRxBufNode->m_RemainDataSize -= sizeof(u16);
			RxDataSize--;
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(RxDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nRx Size(%d) isn't aligned by 2", RxDataSize);
			
			return;
		}

		while(RxDataSize > 0)
		{
			*(u32 *)pRxBufNode->m_pCurPtr = (u32)Inp32(&pSFRBase->SPI_RX_DATA);
			pRxBufNode->m_pCurPtr += sizeof(u32);
			pRxBufNode->m_RemainDataSize -= sizeof(u32);
		}
	}
	else
	{
		SPI_ASSERT();
	}

	/* Disable Interrupt */
	if(pRxBufNode->m_RemainDataSize == 0)
	{
		SPI_EnIntr(pPortInfo, SPI_INT_RX_FIFORDY, SPI_DISABLE);
		if(  pPortInfo->m_EnAutoCS != TRUE  )
		{
			SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
		}

		if(pPortInfo->m_RxIntCompleteCallback != NULL)
		{
			(*pPortInfo->m_RxIntCompleteCallback)(pPortInfo->m_RxIntCompleteCallback_Value);
		}
	}
}



s32 SPI_InterruptMasterRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar)
{
	if( !(pPortInfo->m_EnTxChannel != TRUE && pPortInfo->m_EnRxChannel == TRUE) )
	{
		SPI_ASSERT();
	}

	pPortInfo->m_RxIntCompleteCallback = callbackFn;
	pPortInfo->m_RxIntCompleteCallback_Value = CallBackVar;

	SPI_SetIsrFunc(pPortInfo, SPI_INT_RX_FIFORDY, SPI_IntRxFifoRdy_MasterRx, (u32)pPortInfo);	
	SPI_EnIntr(pPortInfo, SPI_INT_RX_FIFORDY, SPI_ENABLE);

	#if 0
	INTC_SetVectAddr( pPortInfo->m_SPIIntNum, pPortInfo->m_fnISR_SPI);
	INTC_Enable( pPortInfo->m_SPIIntNum);
	#endif /* 0 */

	SPI_EnPktCnt(pPortInfo, SPI_ENABLE, pPortInfo->m_RxBufNode.m_ReqDataSize);
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	if(  pPortInfo->m_EnAutoCS == FALSE  )
	{
		SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
	}
}



s32 SPI_InterruptMasterTxRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t TxcallbackFn, u32 TxCallBackVar,
	SPI_Callback_t RxcallbackFn, u32 RxCallBackVar)
{
	if( !(pPortInfo->m_EnTxChannel == TRUE && pPortInfo->m_EnRxChannel == TRUE) )
	{
		SPI_ASSERT();
	}

	pPortInfo->m_TxIntCompleteCallback = TxcallbackFn;
	pPortInfo->m_TxIntCompleteCallback_Value = TxCallBackVar;

	pPortInfo->m_RxIntCompleteCallback = RxcallbackFn;
	pPortInfo->m_RxIntCompleteCallback_Value = RxCallBackVar;

	SPI_SetIsrFunc(pPortInfo, SPI_INT_RX_FIFORDY, SPI_IntRxFifoRdy_MasterRx, (u32)pPortInfo);	
	SPI_EnIntr(pPortInfo, SPI_INT_RX_FIFORDY, SPI_ENABLE);

	SPI_SetIsrFunc(pPortInfo, SPI_INT_TX_FIFORDY, SPI_IntTxFifoRdy_MasterTx, (u32)pPortInfo);	
	SPI_EnIntr(pPortInfo, SPI_INT_TX_FIFORDY, SPI_ENABLE);

	#if 0
	INTC_SetVectAddr( pPortInfo->m_SPIIntNum, pPortInfo->m_fnISR_SPI);
	INTC_Enable( pPortInfo->m_SPIIntNum);
	#endif /* 0 */

	SPI_EnPktCnt(pPortInfo, SPI_ENABLE, pPortInfo->m_RxBufNode.m_ReqDataSize);	/* Enable Master Rx */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	if(  pPortInfo->m_EnAutoCS == FALSE  )
	{
		SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
	}
}



void SPI_IntTxFifoRdy_SlaveTx(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pTxBufNode;
	u32					TxFifoCnt;
	u32					TxDataSize;
	//u32				TempReg;

	pTxBufNode = &pPortInfo->m_TxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	TxFifoCnt = Inp32(&pSFRBase->SPI_STATUS);
	TxFifoCnt >>= 6;
	//TxFifoCnt &= 0x3F;
	TxFifoCnt &= 0x1FF;
	TxDataSize = pPortInfo->m_MaxFIFOSize - TxFifoCnt * pPortInfo->m_FifoLevelGrid;
	if(pTxBufNode->m_RemainDataSize < TxDataSize)
	{
		TxDataSize = pTxBufNode->m_RemainDataSize;
	}

	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(TxDataSize > 0)
		{
			//if( Inp32(&pSFRBase->SPI_STATUS) & 0x01)		/* Tx FIFO Ready but not working in SlaveTx*/
			{
				Outp32(&pSFRBase->SPI_TX_DATA, *pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr++;
				pTxBufNode->m_RemainDataSize--;
				TxDataSize--;
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(TxDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", TxDataSize);
			
			return;
		}

		while(TxDataSize > 0)
		{
			//if( Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				Outp32(&pSFRBase->SPI_TX_DATA, *(u16 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u16);
				pTxBufNode->m_RemainDataSize -= sizeof(u16);
				TxDataSize -= sizeof(u16);
			}
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(TxDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nTx Size(%d) isn't aligned by 2", TxDataSize);
			
			return;
		}

		while(TxDataSize > 0)
		{
			//if( Inp32(&pSFRBase->SPI_STATUS) & 0x01)	/* Tx FIFO Ready */
			{
				Outp32(&pSFRBase->SPI_TX_DATA, *(u32 *)pTxBufNode->m_pCurPtr);
				pTxBufNode->m_pCurPtr += sizeof(u32);
				pTxBufNode->m_RemainDataSize -= sizeof(u32);
				TxDataSize -= sizeof(u32);
			}
		}
	}
	else
	{
		SPI_ASSERT();
	}

	/* Disable Interrupt */
	if(pTxBufNode->m_RemainDataSize == 0)
	{
		SPI_EnIntr(pPortInfo, SPI_INT_TX_FIFORDY, SPI_DISABLE);

		if(pPortInfo->m_TxIntCompleteCallback != NULL)
		{
			(*pPortInfo->m_TxIntCompleteCallback)(pPortInfo->m_TxIntCompleteCallback_Value);
		}
	}
}


s32 SPI_InterruptSlaveTx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar)
{
	if( !(pPortInfo->m_EnTxChannel == TRUE && pPortInfo->m_EnRxChannel != TRUE) )
	{
		SPI_ASSERT();
	}
	
	pPortInfo->m_TxIntCompleteCallback = callbackFn;
	pPortInfo->m_TxIntCompleteCallback_Value = CallBackVar;

	SPI_SetIsrFunc(pPortInfo, SPI_INT_TX_FIFORDY, SPI_IntTxFifoRdy_SlaveTx, (u32)pPortInfo);	
	SPI_EnIntr(pPortInfo, SPI_INT_TX_FIFORDY, SPI_ENABLE);

	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	#if 0
	INTC_SetVectAddr( pPortInfo->m_SPIIntNum, pPortInfo->m_fnISR_SPI);
	INTC_Enable( pPortInfo->m_SPIIntNum);
	#endif /* 0 */
}



s32 SPI_IntRxFifoRdy_SlaveRx(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pRxBufNode;
	u32					RxFifoCnt;
	u32					RxDataSize;

	pRxBufNode = &pPortInfo->m_RxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	RxFifoCnt = Inp32(&pSFRBase->SPI_STATUS);

	//UART_Printf("\nSTATUS:0x%08x)\n", RxFifoCnt);

	RxFifoCnt >>= 15;
	RxFifoCnt &= 0x1FF;
	RxDataSize = RxFifoCnt;
	if(pRxBufNode->m_RemainDataSize < RxFifoCnt)
	{
		RxDataSize = pRxBufNode->m_RemainDataSize;
	}

	//UART_Printf("(%d)", RxDataSize);

	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(RxDataSize > 0)
		{
			*pRxBufNode->m_pCurPtr = Inp32(&pSFRBase->SPI_RX_DATA);
			pRxBufNode->m_pCurPtr++;
			pRxBufNode->m_RemainDataSize--;
			RxDataSize--;
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(RxDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nRxx Size(%d) isn't aligned by 2", RxDataSize);
			
			return SPI_ERROR;
		}

		while(RxDataSize > 0)
		{
			*(u16 *)pRxBufNode->m_pCurPtr = (u16)Inp32(&pSFRBase->SPI_RX_DATA);
			pRxBufNode->m_pCurPtr += sizeof(u16);
			pRxBufNode->m_RemainDataSize -= sizeof(u16);
			RxDataSize--;
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(RxDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nRx Size(%d) isn't aligned by 2", RxDataSize);
			
			return SPI_ERROR;
		}

		while(RxDataSize > 0)
		{
			*(u32 *)pRxBufNode->m_pCurPtr = (u32)Inp32(&pSFRBase->SPI_RX_DATA);
			pRxBufNode->m_pCurPtr += sizeof(u32);
			pRxBufNode->m_RemainDataSize -= sizeof(u32);
		}
	}
	else
	{
		SPI_ASSERT();
	}

	/* Disable Interrupt */
	if(pRxBufNode->m_RemainDataSize == 0)
	{
		SPI_EnIntr(pPortInfo, SPI_INT_RX_FIFORDY, SPI_DISABLE);

		if(pPortInfo->m_RxIntCompleteCallback != NULL)
		{
			(*pPortInfo->m_RxIntCompleteCallback)(pPortInfo->m_RxIntCompleteCallback_Value);
		}
	}
}



s32 SPI_InterruptSlaveRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar)
{
	if( !(pPortInfo->m_EnTxChannel != TRUE && pPortInfo->m_EnRxChannel == TRUE) )
	{
		SPI_ASSERT();
	}

	pPortInfo->m_RxIntCompleteCallback = callbackFn;
	pPortInfo->m_RxIntCompleteCallback_Value = CallBackVar;

	SPI_SetIsrFunc(pPortInfo, SPI_INT_RX_FIFORDY, SPI_IntRxFifoRdy_SlaveRx, (u32)pPortInfo);	
	SPI_EnIntr(pPortInfo, SPI_INT_RX_FIFORDY, SPI_ENABLE);

	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	#if 0
	INTC_SetVectAddr( pPortInfo->m_SPIIntNum, pPortInfo->m_fnISR_SPI);
	INTC_Enable( pPortInfo->m_SPIIntNum);
	#endif /* 0 */
}

s32 SPI_InterruptSlaveTxRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t TxcallbackFn, u32 TxCallBackVar,
	SPI_Callback_t RxcallbackFn, u32 RxCallBackVar)
{
	if( !(pPortInfo->m_EnTxChannel == TRUE && pPortInfo->m_EnRxChannel == TRUE) )
	{
		SPI_ASSERT();
	}

	pPortInfo->m_TxIntCompleteCallback = TxcallbackFn;
	pPortInfo->m_TxIntCompleteCallback_Value = TxCallBackVar;

	pPortInfo->m_RxIntCompleteCallback = RxcallbackFn;
	pPortInfo->m_RxIntCompleteCallback_Value = RxCallBackVar;

	SPI_SetIsrFunc(pPortInfo, SPI_INT_RX_FIFORDY, SPI_IntRxFifoRdy_MasterRx, (u32)pPortInfo);	
	SPI_EnIntr(pPortInfo, SPI_INT_RX_FIFORDY, SPI_ENABLE);

	SPI_SetIsrFunc(pPortInfo, SPI_INT_TX_FIFORDY, SPI_IntTxFifoRdy_MasterTx, (u32)pPortInfo);	
	SPI_EnIntr(pPortInfo, SPI_INT_TX_FIFORDY, SPI_ENABLE);

	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	#if 0
	INTC_SetVectAddr( pPortInfo->m_SPIIntNum, pPortInfo->m_fnISR_SPI);
	INTC_Enable( pPortInfo->m_SPIIntNum);
	#endif /* 0 */
}



s32 SPI_DmaMasterTx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar)
{
	u32		SrcAddr;
	u32		DstAddr;
	
	if( !(pPortInfo->m_EnTxChannel == TRUE && pPortInfo->m_EnRxChannel != TRUE) )
	{
		SPI_ASSERT();
	}
	
	pPortInfo->m_TxDMACompleteCallback = callbackFn;
	pPortInfo->m_TxDMACompleteCallback_Value = CallBackVar;

	SPI_EnPktCnt(pPortInfo, SPI_ENABLE, pPortInfo->m_TxBufNode.m_ReqDataSize);
	/* NOTICE : SPI_EnChannel should precede the DMA settings. */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	if(  pPortInfo->m_EnAutoCS != TRUE  )
	{
		SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
	}

	/* Enable Tx DMA */
	DMA_SetCh(pPortInfo->m_TxDmaCh, &pPortInfo->m_SPITxDma);
	DMA_InitCh(pPortInfo->m_BusWidth , pPortInfo->m_DmaTxReqSrc, DMA_M2P, 
		pPortInfo->m_DMAType, &pPortInfo->m_SPITxDma);
	DstAddr = (u32)(&pPortInfo->m_SPISFRBase->SPI_TX_DATA);
	SrcAddr = (u32)pPortInfo->m_TxBufNode.m_pCurPtr;
	DMA_StartCh(SrcAddr, DstAddr, pPortInfo->m_TxBufNode.m_ReqDataSize, &pPortInfo->m_SPITxDma);

	#if 0
	INTC_Enable(pPortInfo->m_TxDMANum);
	#endif /* 0 */
}


s32 SPI_DmaMasterRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar)
{
	u32		SrcAddr;
	u32		DstAddr;
	
	if( !(pPortInfo->m_EnTxChannel != TRUE && pPortInfo->m_EnRxChannel == TRUE) )
	{
		SPI_ASSERT();
	}
	
	pPortInfo->m_RxDMACompleteCallback = callbackFn;
	pPortInfo->m_RxDMACompleteCallback_Value = CallBackVar;

	SPI_EnPktCnt(pPortInfo, SPI_ENABLE, pPortInfo->m_RxBufNode.m_ReqDataSize);
	/* NOTICE : SPI_EnChannel should precede the DMA settings. */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	if(  pPortInfo->m_EnAutoCS != TRUE  )
	{
		SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
	}

	/* Enable Tx DMA */
	DMA_SetCh(pPortInfo->m_RxDmaCh, &pPortInfo->m_SPIRxDma);
	DMA_InitCh(pPortInfo->m_BusWidth , pPortInfo->m_DmaRxReqSrc, DMA_P2M, 
		pPortInfo->m_DMAType, &pPortInfo->m_SPIRxDma);
	DstAddr = (u32)pPortInfo->m_RxBufNode.m_pCurPtr;
	SrcAddr = (u32)(&pPortInfo->m_SPISFRBase->SPI_RX_DATA);
	DMA_StartCh(SrcAddr, DstAddr, pPortInfo->m_RxBufNode.m_ReqDataSize, &pPortInfo->m_SPIRxDma);

	#if 0
	INTC_Enable(pPortInfo->m_RxDMANum);
	#endif /* 0 */
}


s32 SPI_DmaMasterTxRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t TxcallbackFn, u32 TxCallBackVar,
	SPI_Callback_t RxcallbackFn, u32 RxCallBackVar)
{
	u32		SrcAddr;
	u32		DstAddr;
	
	if( !(pPortInfo->m_EnTxChannel == TRUE && pPortInfo->m_EnRxChannel == TRUE) )
	{
		SPI_ASSERT();
	}
	
	pPortInfo->m_TxDMACompleteCallback = TxcallbackFn;
	pPortInfo->m_TxDMACompleteCallback_Value = TxCallBackVar;
	pPortInfo->m_RxDMACompleteCallback = RxcallbackFn;
	pPortInfo->m_RxDMACompleteCallback_Value = RxCallBackVar;

	SPI_EnPktCnt(pPortInfo, SPI_ENABLE, pPortInfo->m_RxBufNode.m_ReqDataSize);		/* Enable Master Tx */
	/* NOTICE : SPI_EnChannel should precede the DMA settings. */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	if(  pPortInfo->m_EnAutoCS != TRUE  )
	{
		SPI_ChipSelect(pPortInfo, SPI_NSSOUT_ACTIVE);
	}

	/* Enable Tx DMA */
	DMA_SetCh(pPortInfo->m_TxDmaCh, &pPortInfo->m_SPITxDma);
	DMA_InitCh(pPortInfo->m_BusWidth , pPortInfo->m_DmaTxReqSrc, DMA_M2P, 
		pPortInfo->m_DMAType, &pPortInfo->m_SPITxDma);
	DstAddr = (u32)(&pPortInfo->m_SPISFRBase->SPI_TX_DATA);
	SrcAddr = (u32)pPortInfo->m_TxBufNode.m_pCurPtr;
	DMA_StartCh(SrcAddr, DstAddr, pPortInfo->m_TxBufNode.m_ReqDataSize, &pPortInfo->m_SPITxDma);

	/* Enable Rx DMA */
	DMA_SetCh(pPortInfo->m_RxDmaCh, &pPortInfo->m_SPIRxDma);
	DMA_InitCh(pPortInfo->m_BusWidth , pPortInfo->m_DmaRxReqSrc, DMA_P2M, 
		pPortInfo->m_DMAType, &pPortInfo->m_SPIRxDma);
	DstAddr = (u32)pPortInfo->m_RxBufNode.m_pCurPtr;
	SrcAddr = (u32)(&pPortInfo->m_SPISFRBase->SPI_RX_DATA);
	DMA_StartCh(SrcAddr, DstAddr, pPortInfo->m_RxBufNode.m_ReqDataSize, &pPortInfo->m_SPIRxDma);

	#if 0
	INTC_Enable(pPortInfo->m_TxDMANum);
	INTC_Enable(pPortInfo->m_RxDMANum);
	#endif /* 0 */
}


s32 SPI_DmaSlaveTx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar)
{
	u32		SrcAddr;
	u32		DstAddr;
	
	if( !(pPortInfo->m_EnTxChannel == TRUE && pPortInfo->m_EnRxChannel != TRUE) )
	{
		SPI_ASSERT();
	}
	
	pPortInfo->m_TxDMACompleteCallback = callbackFn;
	pPortInfo->m_TxDMACompleteCallback_Value = CallBackVar;

	/* NOTICE : SPI_EnChannel should precede the DMA settings. */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	/* Enable Tx DMA */
	DMA_SetCh(pPortInfo->m_TxDmaCh, &pPortInfo->m_SPITxDma);
	DMA_InitCh(pPortInfo->m_BusWidth , pPortInfo->m_DmaTxReqSrc, DMA_M2P, 
		pPortInfo->m_DMAType, &pPortInfo->m_SPITxDma);
	DstAddr = (u32)(&pPortInfo->m_SPISFRBase->SPI_TX_DATA);
	SrcAddr = (u32)pPortInfo->m_TxBufNode.m_pCurPtr;
	DMA_StartCh(SrcAddr, DstAddr, pPortInfo->m_TxBufNode.m_ReqDataSize, &pPortInfo->m_SPITxDma);

	#if 0
	INTC_Enable(pPortInfo->m_TxDMANum);
	#endif /* 0 */
}

s32 SPI_DmaSlaveRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t callbackFn, u32 CallBackVar)
{
	u32		SrcAddr;
	u32		DstAddr;
	
	if( !(pPortInfo->m_EnTxChannel != TRUE && pPortInfo->m_EnRxChannel == TRUE) )
	{
		SPI_ASSERT();
	}
	
	pPortInfo->m_RxDMACompleteCallback = callbackFn;
	pPortInfo->m_RxDMACompleteCallback_Value = CallBackVar;


	/* NOTICE : SPI_EnChannel should precede the DMA settings. */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	/* Enable Rx DMA */
	DMA_SetCh(pPortInfo->m_RxDmaCh, &pPortInfo->m_SPIRxDma);
	DMA_InitCh(pPortInfo->m_BusWidth , pPortInfo->m_DmaRxReqSrc, DMA_P2M, 
		pPortInfo->m_DMAType, &pPortInfo->m_SPIRxDma);
	DstAddr = (u32)pPortInfo->m_RxBufNode.m_pCurPtr;
	SrcAddr = (u32)(&pPortInfo->m_SPISFRBase->SPI_RX_DATA);
	DMA_StartCh(SrcAddr, DstAddr, pPortInfo->m_RxBufNode.m_ReqDataSize, &pPortInfo->m_SPIRxDma);

	#if 0
	INTC_Enable(pPortInfo->m_RxDMANum);
	#endif /* 0 */

	//SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);
}


s32 SPI_DmaSlaveTxRx(SPI_PortInfo_st *pPortInfo, SPI_Callback_t TxcallbackFn, u32 TxCallBackVar,
	SPI_Callback_t RxcallbackFn, u32 RxCallBackVar)
{
	u32		SrcAddr;
	u32		DstAddr;
	
	if( !(pPortInfo->m_EnTxChannel == TRUE && pPortInfo->m_EnRxChannel == TRUE) )
	{
		SPI_ASSERT();
	}
	
	pPortInfo->m_TxDMACompleteCallback = TxcallbackFn;
	pPortInfo->m_TxDMACompleteCallback_Value = TxCallBackVar;
	pPortInfo->m_RxDMACompleteCallback = RxcallbackFn;
	pPortInfo->m_RxDMACompleteCallback_Value = RxCallBackVar;

	/* NOTICE : SPI_EnChannel should precede the DMA settings. */
	SPI_EnChannel(pPortInfo, pPortInfo->m_EnTxChannel, pPortInfo->m_EnRxChannel);

	/* Enable Tx DMA */
	DMA_SetCh(pPortInfo->m_TxDmaCh, &pPortInfo->m_SPITxDma);
	DMA_InitCh(pPortInfo->m_BusWidth , pPortInfo->m_DmaTxReqSrc, DMA_M2P, 
		pPortInfo->m_DMAType, &pPortInfo->m_SPITxDma);
	DstAddr = (u32)(&pPortInfo->m_SPISFRBase->SPI_TX_DATA);
	SrcAddr = (u32)pPortInfo->m_TxBufNode.m_pCurPtr;
	DMA_StartCh(SrcAddr, DstAddr, pPortInfo->m_TxBufNode.m_ReqDataSize, &pPortInfo->m_SPITxDma);

	/* Enable Rx DMA */
	DMA_SetCh(pPortInfo->m_RxDmaCh, &pPortInfo->m_SPIRxDma);
	DMA_InitCh(pPortInfo->m_BusWidth , pPortInfo->m_DmaRxReqSrc, DMA_P2M, 
		pPortInfo->m_DMAType, &pPortInfo->m_SPIRxDma);
	DstAddr = (u32)pPortInfo->m_RxBufNode.m_pCurPtr;
	SrcAddr = (u32)(&pPortInfo->m_SPISFRBase->SPI_RX_DATA);
	DMA_StartCh(SrcAddr, DstAddr, pPortInfo->m_RxBufNode.m_ReqDataSize, &pPortInfo->m_SPIRxDma);

	#if 0
	INTC_Enable(pPortInfo->m_TxDMANum);
	INTC_Enable(pPortInfo->m_RxDMANum);
	#endif /* 0 */
}


//////////
// File Name : SPI_interruptHandler
// File Description : This function is interrupt handler for transmission and error handling.
// Input : SPI_Infor, SPI VIC number.
// Output : NONE.
// Version : 
void SPI_IsrRxFifoRdy(u32 uVar)
{
	/* This function is not called, and maybe replaced another function. */
}

//////////
// File Name : SPI_interruptHandler
// File Description : This function is interrupt handler for transmission and error handling.
// Input : SPI_Infor, SPI VIC number.
// Output : NONE.
// Version : 
void SPI_IsrTxFifoRdy(u32 uVar)
{
	/* This function is not called, and maybe replaced another function. */
}

//////////
// File Name : SPI_interruptHandler
// File Description : This function is interrupt handler for transmission and error handling.
// Input : SPI_Infor, SPI VIC number.
// Output : NONE.
// Version : 
void SPI_IsrTxUnder(u32 uVar)
{
	SPI_PortInfo_st *pPortInfo = (SPI_PortInfo_st *) uVar;
	Disp("\n[ISR]SPI Port%d Tx Fifo Under Run!!\n",pPortInfo->m_SPIPortIdx);

	//SPI_ASSERT();
	
	SPI_ClrPending(pPortInfo, SPI_INT_CLR_TX_UNDERRUN);
	
	//SPI_EnIntr(pPortInfo, SPI_INT_TX_UNDERRUN, SPI_DISABLE);	/* for debugging */
}

//////////
// File Name : SPI_interruptHandler
// File Description : This function is interrupt handler for transmission and error handling.
// Input : SPI_Infor, SPI VIC number.
// Output : NONE.
// Version : 
void SPI_IsrTxOver(u32 uVar)
{
	SPI_PortInfo_st *pPortInfo = (SPI_PortInfo_st *) uVar;
	Disp("\n[ISR]SPI Port%d Tx Fifo Over Run!!\n",pPortInfo->m_SPIPortIdx);

	//SPI_ASSERT();

	SPI_ClrPending(pPortInfo, SPI_INT_CLR_TX_OVERRUN);
}

//////////
// File Name : SPI_interruptHandler
// File Description : This function is interrupt handler for transmission and error handling.
// Input : SPI_Infor, SPI VIC number.
// Output : NONE.
// Version : 
void SPI_IsrRxOver(u32 uVar)
{
	SPI_PortInfo_st *pPortInfo = (SPI_PortInfo_st *) uVar;
	Disp("\n[ISR]SPI Port%d Rx Fifo Over  Run!!\n",pPortInfo->m_SPIPortIdx);

	//SPI_ASSERT();

	SPI_ClrPending(pPortInfo, SPI_INT_CLR_RX_OVERRUN);
	
	#if 1
	SPI_SWReset(pPortInfo);
	#endif
}

//////////
// File Name : SPI_interruptHandler
// File Description : This function is interrupt handler for transmission and error handling.
// Input : SPI_Infor, SPI VIC number.
// Output : NONE.
// Version : 
void SPI_IsrRxUnder(u32 uVar)
{
	SPI_PortInfo_st *pPortInfo = (SPI_PortInfo_st *) uVar;
	Disp("\n[ISR]SPI Port%d Rx Fifo UnderRun!!\n",pPortInfo->m_SPIPortIdx);

	//SPI_ASSERT();

	SPI_ClrPending(pPortInfo, SPI_INT_CLR_RX_UNDERRUN);
}


//////////
// File Name : SPI_interruptHandler
// File Description : This function is interrupt handler for transmission and error handling.
// Input : SPI_Infor, SPI VIC number.
// Output : NONE.
// Version : 

void SPI_IsrRxTrailing(u32 uVar)
{
	SPI_SFR				*pSFRBase;
	SPI_BufAllocInfo_st	*pRxBufNode;
	u32					RxFifoCnt;
	u32					RxDataSize;
	bool					IsValidTrailingBytes;

	SPI_PortInfo_st *pPortInfo = (SPI_PortInfo_st *) uVar;
	Disp("\n[ISR]SPI Port%d Rx Trailing! \n",pPortInfo->m_SPIPortIdx);

	pRxBufNode = &pPortInfo->m_RxBufNode;
	pSFRBase = pPortInfo->m_SPISFRBase;

	RxFifoCnt = Inp32(&pSFRBase->SPI_STATUS);
	RxFifoCnt >>= 15;
	RxFifoCnt &= 0x1FF;
	RxDataSize = RxFifoCnt;
	if(pRxBufNode->m_RemainDataSize < RxFifoCnt)
	{
		RxDataSize = pRxBufNode->m_RemainDataSize;
	}

	if(RxDataSize > 0)
	{
		IsValidTrailingBytes = TRUE;
	}
	else
	{
		IsValidTrailingBytes = FALSE;
	}

	if(pPortInfo->m_ChWidth == SPI_BYTE)
	{
		while(RxDataSize > 0)
		{
			*pRxBufNode->m_pCurPtr = Inp32(&pSFRBase->SPI_RX_DATA);
			pRxBufNode->m_pCurPtr++;
			pRxBufNode->m_RemainDataSize--;
			RxDataSize--;
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_HWORD)
	{
		if(RxDataSize % sizeof(u16) != 0)
		{
			UART_Printf("\nRxx Size(%d) isn't aligned by 2", RxDataSize);
			
			return;
		}

		while(RxDataSize > 0)
		{
			*(u16 *)pRxBufNode->m_pCurPtr = (u16)Inp32(&pSFRBase->SPI_RX_DATA);
			pRxBufNode->m_pCurPtr += sizeof(u16);
			pRxBufNode->m_RemainDataSize -= sizeof(u16);
			RxDataSize--;
		}
	}
	else if(pPortInfo->m_ChWidth == SPI_WORD)
	{
		if(RxDataSize % sizeof(u32) != 0)
		{
			UART_Printf("\nRx Size(%d) isn't aligned by 2", RxDataSize);
			
			return;
		}

		while(RxDataSize > 0)
		{
			*(u32 *)pRxBufNode->m_pCurPtr = (u32)Inp32(&pSFRBase->SPI_RX_DATA);
			pRxBufNode->m_pCurPtr += sizeof(u32);
			pRxBufNode->m_RemainDataSize -= sizeof(u32);
		}
	}
	else
	{
		SPI_ASSERT();
	}

	if(IsValidTrailingBytes== TRUE)
	{
		SPI_EnIntr(pPortInfo, SPI_INT_TRAILING, SPI_DISABLE);
		SPI_EnIntr(pPortInfo, SPI_INT_RX_FIFORDY, SPI_DISABLE);		/* Disalbe RX_FIFO_RDY also */
		if(  pPortInfo->m_EnAutoCS != TRUE  )
		{
			SPI_ChipSelect(pPortInfo, SPI_NSSOUT_INACTIVE);
		}

		if(pPortInfo->m_RxIntCompleteCallback != NULL)
		{
			(*pPortInfo->m_RxIntCompleteCallback)(pPortInfo->m_RxIntCompleteCallback_Value);
		}
	}

	SPI_SWReset(pPortInfo);

	//SPI_EnIntr(pPortInfo, SPI_INT_TRAILING, SPI_DISABLE);	/* for debugging */

	SPI_ClrPending(pPortInfo, SPI_INT_CLR_TRAILING);
}


//////////
// File Name : SPI_interruptHandler
// File Description : This function is interrupt handler for transmission and error handling.
// Input : SPI_Infor, SPI VIC number.
// Output : NONE.
// Version : 
void SPI_interruptHandler( SPI_PortInfo_st* pPortInfo, int interruptNumber)
{
	SPI_SFR		*pSFRBase;
	u32			IntEnabled;
	u32			IntStatus;

	//SPI_LEDOn(4);

	//UART_Printf("\nISR_SPI%d : (%d)\n", pPortInfo->m_SPIPortIdx, interruptNumber);
	pSFRBase = pPortInfo->m_SPISFRBase;

	IntEnabled = Inp32(&pSFRBase->SPI_INT_EN);
	IntStatus = Inp32(&pSFRBase->SPI_STATUS);

	//UART_Printf("\nSTATUS:(0x%08x)\n", IntStatus);

	IntStatus &= IntEnabled;	/* If Intrrupt is not enabled, igonore that bit. */
	
	if(IntStatus & SPI_STATUS_RX_FIFORDY)	/* RX_FIFO_RDY */
	{
		if(pPortInfo->m_RxFIFORdyINT_Callback != NULL)
		{
			(*pPortInfo->m_RxFIFORdyINT_Callback)(pPortInfo->m_RxFIFORdyINT_Value);
		}
	}

	if(IntStatus & SPI_STATUS_TX_FIFORDY)	/* TX_FIFO_RDY */
	{
		if(pPortInfo->m_TxFIFORdyINT_Callback != NULL)
		{
			(*pPortInfo->m_TxFIFORdyINT_Callback)(pPortInfo->m_TxFIFORdyINT_Value);
		}
	}

	if (IntStatus & SPI_STATUS_RX_OVERRUN)
	{
		if(pPortInfo->m_RxOverrunINT_Callback != NULL)
		{
			(*pPortInfo->m_RxOverrunINT_Callback)(pPortInfo->m_RxOverrunINT_Value);
		}
	}
	if (IntStatus & SPI_STATUS_RX_UNDERRUN)
	{
		if(pPortInfo->m_RxUnderrunINT_Callback != NULL)
		{
			(*pPortInfo->m_RxUnderrunINT_Callback)(pPortInfo->m_RxUnderrunINT_Value);
		}
	}

	if (IntStatus & SPI_STATUS_TX_UNDERRUN)
	{
		if(pPortInfo->m_TxUnderrunINT_Callback != NULL)
		{
			(*pPortInfo->m_TxUnderrunINT_Callback)(pPortInfo->m_TxUnderrunINT_Value);
		}
	}
	
	if (IntStatus & SPI_STATUS_TX_OVERRUN)
	{
		if(pPortInfo->m_TxOverrunINT_Callback != NULL)
		{
			(*pPortInfo->m_TxOverrunINT_Callback)(pPortInfo->m_TxOverrunINT_Value);
		}
	}

	/* Trailing Byte interrupt always occur. so this interrupt source needs to be checked after RX_FIFO_READY. */
	if( (Inp32(&pSFRBase->SPI_STATUS) & SPI_STATUS_TRAILING_BYTE) && (IntEnabled & SPI_INT_TRAILING) )
	{
		if(pPortInfo->m_TrailingINT_Callback != NULL)
		{
			(*pPortInfo->m_TrailingINT_Callback)(pPortInfo->m_TrailingINT_Value);
		}
	}

	INTC_ClearVectAddr();

	//SPI_LEDOff(4);

}

void SPI_LEDOn(u32 BitNum)
{
	u32	TempReg;
	
	TempReg = GPIO_GetDataAll(eGPIO_H2);
	TempReg |= (1<<BitNum);
	GPIO_SetDataAll(eGPIO_H2, TempReg);
}

void SPI_LEDOff(u32 BitNum)
{
	u32	TempReg;

	TempReg = GPIO_GetDataAll(eGPIO_H2);
	TempReg &= ~(1<< BitNum);
	DisplayLED(TempReg);
	GPIO_SetDataAll(eGPIO_H2, TempReg);
}


//////////
// File Name : Isr_SPI0
// File Description : Interrupt Handler for channel 0
// Input : NONE
// Output : NONE.
// Version : 
void __irq Isr_SPI0( void )
{
	SPI_PortInfo_st* pPortInfo = gSPI_SPIIsrPortInfo[SPI_PORT0];

	if(SPI_IsInterruptPrintValid())
	{
		UART_Printf("\nIsr_SPI0 ");
	}

	//SPI_LEDOn(4);
	
	SPI_interruptHandler(pPortInfo, NUM_SPI0);

	//SPI_LEDOff(4);
 	//INTC_ClearVectAddr();
}

//////////
// File Name : Isr_SPI1
// File Description : Interrupt Handler for channel 1
// Input : NONE
// Output : NONE.
// Version : 
void __irq Isr_SPI1( void ) 
{
	SPI_PortInfo_st* pPortInfo = gSPI_SPIIsrPortInfo[SPI_PORT1];

	if(SPI_IsInterruptPrintValid())
	{
		UART_Printf("\nIsr_SPI1 ");
	}

	//SPI_LEDOn(5);

	SPI_interruptHandler(pPortInfo, NUM_SPI1);

	//SPI_LEDOff(5);

 	//INTC_ClearVectAddr();
}

//////////
// File Name : Isr_SPI1
// File Description : Interrupt Handler for channel 1
// Input : NONE
// Output : NONE.
// Version : 
void __irq Isr_SPI2( void ) 
{
	SPI_PortInfo_st* pPortInfo = gSPI_SPIIsrPortInfo[SPI_PORT2];

	if(SPI_IsInterruptPrintValid())
	{
		UART_Printf("\nIsr_SPI2 ");
	}
	
	SPI_interruptHandler(pPortInfo, NUM_SPI2);

 	//INTC_ClearVectAddr();
}



void __irq SPI_DMA0Handler( void )
{
	DMA_CH		DMACh;
	DMAC		DMAControl;
	SPI_PortInfo_st *pPortInfo;

	//DMAControl = g_oSPIInfor[0].m_SPITxDma;	/* Set dummy value for base address PDMA0_BASE */
	DMAControl.m_uBaseAddr = PDMA0_BASE;

	//INTC_Disable(NUM_PDMA0);

	DMA_GetIntrSrc((u32 *)&DMACh, &DMAControl);
	DMA_ClearIntPending((u32)DMACh, &DMAControl);

	if(SPI_IsInterruptPrintValid())
	{
		UART_Printf("\n[DMA0] CH(%d)\n", DMACh);
	}

	pPortInfo = gSPI_DMAIsrPortInfo[DMACh];

	if(pPortInfo != NULL)
	{
		if(DMACh == pPortInfo->m_TxDmaCh)
		{
			if(pPortInfo->m_TxDMACompleteCallback !=NULL)
			{
				(*pPortInfo->m_TxDMACompleteCallback)(pPortInfo->m_TxDMACompleteCallback_Value);
			}
		}
		else
		{
			if(pPortInfo->m_RxDMACompleteCallback !=NULL)
			{
				(*pPortInfo->m_RxDMACompleteCallback)(pPortInfo->m_RxDMACompleteCallback_Value);
			}
		}
	}
	else
	{
		SPI_ASSERT();
	}

	INTC_ClearVectAddr();
}

void __irq SPI_DMA1Handler( void )
{
	DMA_CH		DMACh;
	DMAC		DMAControl;
	SPI_PortInfo_st *pPortInfo;

	//DMAControl = g_oSPIInfor[0].m_SPITxDma;	/* Set dummy value for base address PDMA0_BASE */
	DMAControl.m_uBaseAddr = PDMA1_BASE;

	//INTC_Disable(NUM_PDMA0);

	DMA_GetIntrSrc((u32 *)&DMACh, &DMAControl);
	DMA_ClearIntPending((u32)DMACh, &DMAControl);

	if(SPI_IsInterruptPrintValid())
	{
		UART_Printf("\n[DMA1] %d", DMACh);
	}

	pPortInfo = gSPI_DMAIsrPortInfo[DMA_10 + DMACh];	/* DMA_10 is the first DMA channel of DMA1. */

	if(pPortInfo != NULL)
	{
		if((DMA_10 + DMACh) == pPortInfo->m_TxDmaCh)
		{
			if(pPortInfo->m_TxDMACompleteCallback !=NULL)
			{
				(*pPortInfo->m_TxDMACompleteCallback)(pPortInfo->m_TxDMACompleteCallback_Value);
			}
		}
		else
		{
			if(pPortInfo->m_RxDMACompleteCallback !=NULL)
			{
				(*pPortInfo->m_RxDMACompleteCallback)(pPortInfo->m_RxDMACompleteCallback_Value);
			}
		}
	}
	else
	{
		SPI_ASSERT();
	}

	INTC_ClearVectAddr();
}



//////////
// File Name : SPI_SetIsrFunc
// File Description : This function set function called in ISR certain interrupt condition
// Input : SPI_Infor 
// Output : NONE.
// Version : 
void SPI_SetIsrFunc(SPI_PortInfo_st *pPortInfo, u32 uIntrMode, void (*fnIsrFunc)(u32), u32 uIsrVar)
{
	if ( uIntrMode & SPI_INT_TRAILING )					
	{
		pPortInfo->m_TrailingINT_Callback = fnIsrFunc;
		pPortInfo->m_TrailingINT_Value = uIsrVar;
	}
	if ( uIntrMode & SPI_INT_RX_OVERRUN )
	{
		pPortInfo->m_RxOverrunINT_Callback = fnIsrFunc;
		pPortInfo->m_RxOverrunINT_Value = uIsrVar;
	}
	if ( uIntrMode & SPI_INT_RX_UNDERRUN )
	{
		pPortInfo->m_RxUnderrunINT_Callback = fnIsrFunc;
		pPortInfo->m_RxUnderrunINT_Value = uIsrVar;
	}
	if ( uIntrMode & SPI_INT_TX_OVERRUN )
	{
		pPortInfo->m_TxOverrunINT_Callback = fnIsrFunc;
		pPortInfo->m_TxOverrunINT_Value = uIsrVar;
	}
	if ( uIntrMode & SPI_INT_TX_UNDERRUN )
	{
		pPortInfo->m_TxUnderrunINT_Callback = fnIsrFunc;
		pPortInfo->m_TxUnderrunINT_Value = uIsrVar;
	}
	if ( uIntrMode & SPI_INT_RX_FIFORDY )
	{
		pPortInfo->m_RxFIFORdyINT_Callback = fnIsrFunc;
		pPortInfo->m_RxFIFORdyINT_Value = uIsrVar;
	}
	if ( uIntrMode & SPI_INT_TX_FIFORDY )
	{
		pPortInfo->m_TxFIFORdyINT_Callback = fnIsrFunc;
		pPortInfo->m_TxFIFORdyINT_Value = uIsrVar;
	}
}



//////////
// Function Name : SPI_SWReset
// Function Description : This function reset Rx/Tx Fifo Data and SPI_STATUS.
// Input : 	ePort - SPI_Infor Number 
// Output :	None
// Version : v0.1
void SPI_SWReset(SPI_PortInfo_st *pPortInfo)
{
	SPI_SFR		*pSFRReg;
	u32 			TempReg;

	pSFRReg = pPortInfo->m_SPISFRBase;

	TempReg = Inp32(&pSFRReg->CH_CFG);
	
	Outp32(&pSFRReg->CH_CFG, TempReg | (1 << 5));
	//Delay(10);
	Outp32(&pSFRReg->CH_CFG, TempReg & ~(1 << 5));
}


//////////
// Function Name : SPI_EnChannel
// Function Description : This function Enable/Disable each channel.
// Input : 	ePort - SPI Port Number 
//			eCh - Tx/Rx Channel
//			usEn - Enable(1)/Diable(0)
// Output :	None
// Version : v0.1
void SPI_EnChannel(SPI_PortInfo_st *pPortInfo, bool EnTxChannel, bool EnRxChannel)
{
	SPI_SFR		*pSFRReg;
	u32 			TempReg;

	pSFRReg = pPortInfo->m_SPISFRBase;

	TempReg = Inp32(&pSFRReg->CH_CFG);
	TempReg &= ~0x03;
	TempReg |= (EnRxChannel << 1) | (EnTxChannel << 0);

	/* S/W Reset */
	/* if SWReset is not called, I'm not sure the correct operation.
	    so I strongly recommend you should do the SW_Reset before Enable_Channel. */
	if(pPortInfo->m_EnSWReset)
	{
		SPI_SWReset(pPortInfo);
	}
	
	Outp32(&pSFRReg->CH_CFG, TempReg);
}


//////////
// File Name : SPI_ChipSelect
// File Description : This function reset certain spi channel.
// Input : SPI_Infor 
// Output : NONE.
// Version : 
void SPI_ChipSelect(SPI_PortInfo_st *pPortInfo, SPI_NSSOUT_et  NSSOut)
{
	SPI_SFR		*pSFRReg;
	u32 			TempReg;

	pSFRReg = pPortInfo->m_SPISFRBase;

	TempReg = Inp32(pSFRReg->CS_REG);
	TempReg &= ~1;
	TempReg |= NSSOut;
	
	Outp32(&pSFRReg->CS_REG, TempReg);
}



/*---------------------------------- APIs of SPI_INT_EN ---------------------------------*/
//////////
// File Name : SPI_EnIntr
// File Description : This function reset certain spi channel.
// Input : SPI_Infor 
// Output : NONE.
// Version : 
void SPI_EnIntr(SPI_PortInfo_st *pPortInfo, u32 IntrMode, bool En)
{
	SPI_SFR		*pSFRBase;
	u32			TempReg;

	pSFRBase = pPortInfo->m_SPISFRBase;

	TempReg = Inp32(&pSFRBase->SPI_INT_EN);
	if( En )
	{
		TempReg |= IntrMode;		
	}
	else 
	{
		TempReg &= ~(IntrMode);
	}

	Outp32(&pSFRBase->SPI_INT_EN, TempReg);	
}





//////////
// File Name : SPI_IsFifoRdy
// File Description : This function reset certain spi channel.
// Input : SPI_Infor 
// Output : NONE.
// Version : 
void SPI_EnPktCnt(SPI_PortInfo_st *pPortInfo, bool En, u32 CntVal)
{
	SPI_SFR		*pSFRBase;
	u32			TempReg;

	pSFRBase = pPortInfo->m_SPISFRBase;

	TempReg = (En<<16) | (CntVal<<0);
	Outp32(&pSFRBase->PACKET_CNT_REG, TempReg);
}

/*---------------------------------- APIs of PENDING_CLR_REG ---------------------------------*/
//////////
// File Name : SPI_ClrPending
// File Description : This function reset certain spi channel.
// Input : SPI_Infor 
// Output : NONE.
// Version : 
void SPI_ClrPending(SPI_PortInfo_st *pPortInfo, SPI_IntClrModeBit_et IntClrBit)
{
	SPI_SFR		*pSFRBase;
	u32			TempReg;

	pSFRBase = pPortInfo->m_SPISFRBase;

	TempReg = IntClrBit & 0x1F;
	Outp32(&pSFRBase->PENDING_CLR_REG, TempReg);
}


void SPI_Init(SPI_CtrlInfo_st *pSPICtrlInfo)
{
	int LoopCnt;

	for(LoopCnt = 0; LoopCnt < MAX_SPI_PORT; LoopCnt++)
	{
		//gSPI_SPIIsrPortInfo[LoopCnt] = pSPICtrlInfo->m_SPIPortTestInfo[LoopCnt];
		gSPI_SPIIsrPortInfo[LoopCnt] = NULL;
	}

	for(LoopCnt = 0; LoopCnt < MAX_SPI_DMA_CHANNEL; LoopCnt++)
	{
		//gSPI_SPIIsrPortInfo[LoopCnt] = pSPICtrlInfo->m_SPIPortTestInfo[LoopCnt];
		gSPI_DMAIsrPortInfo[LoopCnt] = NULL;
		//gSPI_DMA1IsrPortInfo[LoopCnt] = NULL;
	}
}

void SPI_SetDefaultSystemEnv(void)
{
	SPI_SetNonSecure();

	InitLED();	/* for debugging */
}

void SPI_SetNonSecure(void)
{
	Outp32(0xF1500804,0xff );
	Outp32(0xF1500810,0xff );
	Outp32(0xF150081C,0xff );
	Outp32(0xFAD00804,0xff );
	Outp32(0xFAD00810,0xff );
	Outp32(0xFAD0081C,0xff );
	Outp32(0xFAD00828,0xff );
	Outp32(0xE0600804,0xff );
	Outp32(0xE0600810,0xff );
	Outp32(0xE060081C,0xff );
	Outp32(0xE0600828,0xff );
	//Outp32(0xE1C00804,0xff );
	//Outp32(0xE1C00810,0xff );
	//Outp32(0xE1C0081C,0xff );
	//Outp32(0xE1C00828,0xff );

}


bool SPI_IsInterruptPrintValid(void)
{
	return gSPI_CtrlInfo.m_EnPrintInISR;
}


void SPI_InitDefaultValue(SPI_CtrlInfo_st *pSPICtrlInfo)
{
	SPI_PortInfo_st	*pPortInfo;
	SPI_PORT		LoopCnt;
	DMA_CH			DmaCh0Cnt;
	DMA_CH			DmaCh1Cnt;

	memset(pSPICtrlInfo, 0x00, sizeof(SPI_CtrlInfo_st));

	/* Create Non-Cachable Area Control. */
	SPI_CreateNCBufCtrl(&pSPICtrlInfo->m_NCBufCtrlInfo, SPI_BUF, SPI_BUF_SIZE);
	SPI_BufZeroInit((u8 *)SPI_BUF, SPI_BUF_SIZE);

	pSPICtrlInfo->m_EnPrintInISR = TRUE;

	DmaCh0Cnt = DMA_00;
	DmaCh1Cnt = DMA_10;

	for(LoopCnt=0; LoopCnt<MAX_SPI_PORT; LoopCnt++)
	{
		pPortInfo = &pSPICtrlInfo->m_SPIPortDefaultInfo[LoopCnt];
		pPortInfo->m_SPIPortIdx = LoopCnt;
		
		SPI_SetSFRBase(pPortInfo);

		pPortInfo->m_EnHighSpeed = FALSE;
		//pPortInfo->m_EnHighSpeed = TRUE;
		if(pPortInfo->m_SPIPortIdx == SPI_PORT0)
		{
			#if 1
			pPortInfo->m_OpMode = SPI_MASTER;
			//pPortInfo->m_OpMode = SPI_SLAVE;
			pPortInfo->m_EnRxChannel = FALSE;
			//pPortInfo->m_EnRxChannel = TRUE;
			pPortInfo->m_EnTxChannel = TRUE;
			#else
			pPortInfo->m_OpMode = SPI_MASTER;
			//pPortInfo->m_OpMode = SPI_SLAVE;
			pPortInfo->m_EnRxChannel = TRUE;
			pPortInfo->m_EnTxChannel = FALSE;
			#endif /* 1 */
			
			pPortInfo->m_MaxFIFOSize = 256;				/* bytes */
			pPortInfo->m_FifoLevelGrid = 4;					/* 4 bytes grid */

			pPortInfo->m_TxReadyLevel = 1;
			pPortInfo->m_RxReadyLevel = 1;
		}
		else
		{
			#if 1
			pPortInfo->m_OpMode = SPI_SLAVE;
			//pPortInfo->m_OpMode = SPI_MASTER;
			pPortInfo->m_EnRxChannel = TRUE;
			pPortInfo->m_EnTxChannel = FALSE;
			//pPortInfo->m_EnTxChannel = TRUE;
			#else
			pPortInfo->m_OpMode = SPI_SLAVE;
			//pPortInfo->m_OpMode = SPI_MASTER;
			pPortInfo->m_EnRxChannel = FALSE;
			pPortInfo->m_EnTxChannel = TRUE;
			#endif /* 1 */
			
			pPortInfo->m_MaxFIFOSize = 64;		/* bytes */
			pPortInfo->m_FifoLevelGrid = 1;		/* 1 byte grid */

			pPortInfo->m_TxReadyLevel = 4;
			pPortInfo->m_RxReadyLevel = 4;
			//pPortInfo->m_RxReadyLevel = 60;
		}

		pPortInfo->m_TxTransferSize = SPI_BUFFER_SIZE;	/* bytes */
		pPortInfo->m_RxTransferSize = SPI_BUFFER_SIZE;	/* bytes */

		//pPortInfo->m_CPOL = SPI_ACTIVE_HIGH;
		pPortInfo->m_CPOL = SPI_ACTIVE_LOW;
		pPortInfo->m_CPHA = SPI_FORMAT_A;
		//pPortInfo->m_CPHA = SPI_FORMAT_B;
		
		//pPortInfo->m_IsUseExtClk = SPI_PCLK;
		pPortInfo->m_IsUseExtClk = SPI_EXT_CLK;
		pPortInfo->m_EnClk = TRUE;			/* Enable SPI clock */
		//pPortInfo->m_ExtClkSrc = SPI_EXT_CLKSRC_XXTI;
		//pPortInfo->m_ExtClkSrc = SPI_EXT_CLKSRC_MPLL;
		pPortInfo->m_ExtClkSrc = SPI_EXT_CLKSRC_EPLL;
		pPortInfo->m_ExtClkSrcDiv = 1;		/* from 1 */

		pPortInfo->m_TargetSPICLK = 1*1000*1000;	/* 1 MHz */
		//pPortInfo->m_TargetSPICLK = 25*1000*1000;	/* 25 MHz */
		//pPortInfo->m_TargetSPICLK = 48*1000*1000;	/* 48 MHz */
		//pPortInfo->m_TargetSPICLK = 50*1000*1000;	/* 50 MHz */

		SPI_CalcTargetClk(pPortInfo->m_IsUseExtClk, pPortInfo->m_TargetSPICLK,
			pPortInfo->m_ExtClkSrc, pPortInfo->m_ExtClkSrcDiv, &pPortInfo->m_OpClock, 
			&pPortInfo->m_ClkDivValue);
		SPI_SelectExtClkSrc(pPortInfo);

		#if 1
		pPortInfo->m_ChWidth = SPI_BYTE;
		pPortInfo->m_BusWidth = SPI_BYTE;
		#else
		pPortInfo->m_ChWidth = SPI_WORD;
		pPortInfo->m_BusWidth = SPI_WORD;
		#endif /* 0 */

		pPortInfo->m_TraillingCnt = pPortInfo->m_RxReadyLevel * pPortInfo->m_FifoLevelGrid;

		//SPI_SetSlaveSelectControl(pPortInfo, 0, TRUE);	/* NCS_TIME_COUNT : 0, Auto Select */
		pPortInfo->m_NCSTimeCnt = 0;
		pPortInfo->m_EnAutoCS = TRUE;

		pPortInfo->m_EnTrailingInt = TRUE;
		//pPortInfo->m_EnTrailingInt = FALSE;
		pPortInfo->m_EnRxOverrunInt = TRUE;
		pPortInfo->m_EnRxUnderrunInt = TRUE;
		pPortInfo->m_EnTxOverrunInt = TRUE;
		pPortInfo->m_EnTxUnderrunInt = TRUE;

		pPortInfo->m_EnRxSwap = FALSE;
		pPortInfo->m_EnTxSwap = FALSE;

		pPortInfo->m_FBClkSel = SPI_PHASE_DELAY0;
		//pPortInfo->m_FBClkSel = SPI_PHASE_DELAY270;

		pPortInfo->m_EnPacketCnt = FALSE;
		pPortInfo->m_PacketCntValue = 0;
		
		pPortInfo->m_DrvStrength = 0x03;
		//pPortInfo->m_DrvStrength = 0x01;

		#if 1
		pPortInfo->m_EnSWReset = TRUE;	/* Recommended in normal case */
		#else
		pPortInfo->m_EnSWReset = FALSE;	/* Just for debugging */
		#endif /* 1 */

		pPortInfo->m_TxTransferMode = SPI_DMA_MODE;		/* Polling, Interrupt, or DMA mode. */
		pPortInfo->m_RxTransferMode = SPI_DMA_MODE;		/* Polling, Interrupt, or DMA mode. */
		//pPortInfo->m_DMAType = SPI_DMA_4_BURST;			/* SINGLE or 4-Burst */
		pPortInfo->m_DMAType = SPI_DMA_SINGLE;			/* SINGLE or 4-Burst */

		pPortInfo->m_RxDMANum = NUM_PDMA0;
		pPortInfo->m_TxDMANum = NUM_PDMA0;
		if(pPortInfo->m_RxDMANum == NUM_PDMA0)
		{
			pPortInfo->m_RxDmaCh = DmaCh0Cnt++;
		}
		else if(pPortInfo->m_RxDMANum == NUM_PDMA1)
		{
			pPortInfo->m_RxDmaCh = DmaCh1Cnt++;
		}
		else
		{
			SPI_ASSERT();
		}

		if(pPortInfo->m_TxDMANum == NUM_PDMA0)
		{
			pPortInfo->m_TxDmaCh = DmaCh0Cnt++;
		}
		else if(pPortInfo->m_TxDMANum == NUM_PDMA1)
		{
			pPortInfo->m_TxDmaCh = DmaCh1Cnt++;
		}
		else
		{
			SPI_ASSERT();
		}

		pPortInfo->m_TrailingINT_Callback = NULL;
		pPortInfo->m_RxOverrunINT_Callback = NULL;
		pPortInfo->m_RxOverrunINT_Callback = NULL;
		pPortInfo->m_TxOverrunINT_Callback = NULL;
		pPortInfo->m_TxUnderrunINT_Callback = NULL;
		pPortInfo->m_RxFIFORdyINT_Callback = NULL;
		pPortInfo->m_TxFIFORdyINT_Callback = NULL;

		pPortInfo->m_TrailingINT_Value = 0;
		pPortInfo->m_RxOverrunINT_Value = 0;
		pPortInfo->m_RxUnderrunINT_Value = 0;
		pPortInfo->m_RxOverrunINT_Value = 0;
		pPortInfo->m_TxUnderrunINT_Value = 0;
		pPortInfo->m_RxFIFORdyINT_Value = 0;
		pPortInfo->m_TxFIFORdyINT_Value = 0;
		
		pPortInfo->m_RxIntCompleteCallback = NULL;
		pPortInfo->m_TxIntCompleteCallback = NULL;

		pPortInfo->m_TxIntCompleteCallback = 0;
		pPortInfo->m_TxIntCompleteCallback_Value = 0;

		switch(pPortInfo->m_SPIPortIdx)
		{
			case SPI_PORT0:
				pPortInfo->m_SPIIntNum = NUM_SPI0;
				pPortInfo->m_fnISR_SPI = Isr_SPI0;
				pPortInfo->m_DmaTxReqSrc = SPI0_TX;
				pPortInfo->m_DmaRxReqSrc = SPI0_RX;
				break;

			case SPI_PORT1:
				pPortInfo->m_SPIIntNum = NUM_SPI1;
				pPortInfo->m_fnISR_SPI = Isr_SPI1;
				pPortInfo->m_DmaTxReqSrc = SPI1_TX;
				pPortInfo->m_DmaRxReqSrc = SPI1_RX;
				break;
				
			case SPI_PORT2:
				pPortInfo->m_SPIIntNum = NUM_SPI2;
				pPortInfo->m_fnISR_SPI = Isr_SPI2;
				pPortInfo->m_DmaTxReqSrc = SPI2_TX;
				pPortInfo->m_DmaRxReqSrc = SPI2_RX;
				break;

			default:
				SPI_ASSERT();
				break;
		}
	}

	for(LoopCnt=0; LoopCnt<MAX_SPI_PORT; LoopCnt++)
	{
		//pSPICtrlInfo->m_SPIPortTestInfo[LoopCnt] = pSPICtrlInfo->m_SPIPortDefaultInfo[LoopCnt];
		memcpy(&pSPICtrlInfo->m_SPIPortTestInfo[LoopCnt], &pSPICtrlInfo->m_SPIPortDefaultInfo[LoopCnt], sizeof(SPI_PortInfo_st));
	}
}

void SPI_SetSFRBase(SPI_PortInfo_st *pPortInfo)
{
	switch(pPortInfo->m_SPIPortIdx)
	{
		case SPI_PORT0:
			pPortInfo->m_SPISFRBase = (SPI_SFR *)SPI0_BASE;
			break;

		case SPI_PORT1:
			pPortInfo->m_SPISFRBase = (SPI_SFR *)SPI1_BASE;
			break;

		case SPI_PORT2:
			pPortInfo->m_SPISFRBase = (SPI_SFR *)SPI2_BASE;
			break;

		default:
			SPI_ASSERT();
			break;
	}
}


s32 SPI_CheckLoopbackMasterAndChannel(SPI_PortInfo_st *pPortAInfo, SPI_PortInfo_st *pPortBInfo,
	SPI_PortInfo_st **pMasterPort, SPI_PortInfo_st **pSlavePort, SPI_PortInfo_st **pTxPort, 
	SPI_PortInfo_st **pRxPort)
{
	if(pPortAInfo->m_OpMode == SPI_MASTER)
	{
		if(pPortBInfo->m_OpMode == SPI_MASTER)
		{
			SPI_DisplayError();
			SPI_DisplayVar_OpMode(pPortAInfo);
			SPI_DisplayVar_OpMode(pPortBInfo);
			
			return SPI_ERROR;
		}
		else
		{
			/* Port0 Master, Port1 Slave */
			*pMasterPort = pPortAInfo;
			*pSlavePort = pPortBInfo;
		}
		
	}
	else
	{
		if(pPortBInfo->m_OpMode == SPI_SLAVE)
		{
			SPI_DisplayError();
			SPI_DisplayVar_OpMode(pPortAInfo);
			SPI_DisplayVar_OpMode(pPortBInfo);
			
			return SPI_ERROR;
		}
		else
		{
			/* Port0 Slave, Port1 Master */
			*pMasterPort = pPortBInfo;
			*pSlavePort = pPortAInfo;
		}
	}

	if(pPortAInfo->m_EnTxChannel == TRUE)
	{
		if(pPortBInfo->m_EnRxChannel == TRUE)
		{
			*pTxPort = pPortAInfo;
			*pRxPort = pPortBInfo;
		}
		else
		{
			SPI_DisplayError();
			//SPI_DisplayPortInfo_Channel(pPortAInfo);
			SPI_DisplayVar_EnTxChannel(pPortAInfo);
			SPI_DisplayVar_EnRxChannel(pPortAInfo);
			//SPI_DisplayPortInfo_Channel(pPortBInfo);
			SPI_DisplayVar_EnTxChannel(pPortBInfo);
			SPI_DisplayVar_EnRxChannel(pPortBInfo);

			return SPI_ERROR;
		}
	}

	if(pPortAInfo->m_EnRxChannel == TRUE)
	{
		if(pPortBInfo->m_EnTxChannel == TRUE)
		{
			*pTxPort = pPortBInfo;
			*pRxPort = pPortAInfo;
		}
		else
		{
			SPI_DisplayError();
			//SPI_DisplayPortInfo_Channel(pPortAInfo);
			SPI_DisplayVar_EnTxChannel(pPortAInfo);
			SPI_DisplayVar_EnRxChannel(pPortAInfo);
			//SPI_DisplayPortInfo_Channel(pPortBInfo);
			SPI_DisplayVar_EnTxChannel(pPortBInfo);
			SPI_DisplayVar_EnRxChannel(pPortBInfo);

			return SPI_ERROR;
		}
	}

	return SPI_NO_ERROR;
}


void SPI_DisplayError(void)
{
	UART_Printf("\nError.");
}


void SPI_CreateNCBufCtrl(SPI_NCBufCtrlInfo_st *pNCBufCtrlInfo, u32 StartAddr, u32 TotalSize)
{
	pNCBufCtrlInfo->m_StartMemAddr = StartAddr;
	pNCBufCtrlInfo->m_CurMemAddr = StartAddr;
	pNCBufCtrlInfo->m_TotalSize = TotalSize;
}



void SPI_DeleteNCBufCtrl(UART_NCBufCtrlInfo_st *pNCBufCtrlInfo)
{
	/* Do nothing */
}





u8 *SPI_GetNCBuf(SPI_BufAllocInfo_st *pBufAllocInfo, u32 BufSize)
{
	SPI_NCBufCtrlInfo_st *pNCBufCtrlInfo;

	pNCBufCtrlInfo = &gSPI_CtrlInfo.m_NCBufCtrlInfo;
	if(BufSize > pNCBufCtrlInfo->m_TotalSize)
	{
		UART_ASSERT();
		
		return NULL;
	}
	
	if(pNCBufCtrlInfo->m_CurMemAddr + BufSize < pNCBufCtrlInfo->m_StartMemAddr + pNCBufCtrlInfo->m_TotalSize)
	{
		pBufAllocInfo->m_pBufPtr = (u8 *)pNCBufCtrlInfo->m_CurMemAddr;
		pBufAllocInfo->m_pCurPtr = (u8 *)pNCBufCtrlInfo->m_CurMemAddr;
		pNCBufCtrlInfo->m_CurMemAddr  += BufSize;
	}
	else
	{
		/* because ring buffer */
		pBufAllocInfo->m_pBufPtr = (u8 *)pNCBufCtrlInfo->m_StartMemAddr;
		pBufAllocInfo->m_pCurPtr = (u8 *)pNCBufCtrlInfo->m_StartMemAddr;
		pNCBufCtrlInfo->m_CurMemAddr = pNCBufCtrlInfo->m_StartMemAddr + BufSize;
	}
	pBufAllocInfo->m_BufSize = BufSize;

	return (u8 *)pBufAllocInfo->m_pBufPtr;
}




int SPI_FreeNCBuf(SPI_BufAllocInfo_st *pBufAllocInfo)
{
	/* NOT implemented because I'll just use the non cachable memory as Ring-Buffer. */
	
	return UART_NO_ERROR;
}


int SPI_NCBufPatternInit(SPI_BufAllocInfo_st *pBufAllocInfo, char *pPattern, int PatternSize)
{
	void		*pBuf;
	u32 		BufSize;
	u32		LocalLoop;
	char		*pDst;

	pBuf = pBufAllocInfo->m_pBufPtr;
	BufSize = pBufAllocInfo->m_BufSize;
	if(BufSize == 0 || PatternSize == 0)
	{
		UART_ASSERT();
		
		return UART_ERROR;
	}

	pDst = pBuf;

	pBufAllocInfo->m_ReqDataSize = BufSize;
	pBufAllocInfo->m_RemainDataSize = BufSize;
	if(PatternSize > BufSize)
	{
		PatternSize = BufSize;
	}
	while(BufSize>0)
	{
		LocalLoop = 0;
		while(LocalLoop < PatternSize)
		{
			*pDst = *(pPattern + LocalLoop);
			pDst++;
			LocalLoop++;
			BufSize--;
		}
	}

	return UART_NO_ERROR;
}



int SPI_NCBufRandInit(SPI_BufAllocInfo_st *pBufAllocInfo, u32 BufSize)
{
 	u8 *pBufPtr;

	pBufPtr = pBufAllocInfo->m_pBufPtr;
 	if(BufSize == 0)
	{
		UART_ASSERT();
		
		return UART_ERROR;
	}

	pBufAllocInfo->m_ReqDataSize = BufSize;
	pBufAllocInfo->m_RemainDataSize = BufSize;
	while(BufSize>0)
	{
		*pBufPtr = (u8)(rand() & 0xFF);
		pBufPtr++;
		BufSize--;
	}

	return UART_NO_ERROR;
}




int SPI_NCBufZeroInit(SPI_BufAllocInfo_st *pBufAllocInfo)
{
	void *pBuf;
	u32 BufSize;

	pBuf = pBufAllocInfo->m_pBufPtr;
	BufSize = pBufAllocInfo->m_BufSize;
	if(BufSize == 0)
	{
		UART_ASSERT();
		
		return UART_ERROR;
	}

	memset(pBuf, 0x00, BufSize);
	pBufAllocInfo->m_ReqDataSize = BufSize;
	pBufAllocInfo->m_RemainDataSize = BufSize;
	
	return UART_NO_ERROR;
}

void SPI_NCBufDisplay(SPI_BufAllocInfo_st *pBufAllocInfo, bool EnDisplay)
{
	u8 *pBuf;
	u32 BufSize;
	u32	LoopCnt;

	if(EnDisplay != TRUE)
	{
		return;
	}

	pBuf = pBufAllocInfo->m_pBufPtr;
	BufSize = pBufAllocInfo->m_ReqDataSize;
	for(LoopCnt=0; LoopCnt<BufSize; LoopCnt++)
	{
		UART_Printf("%c", *pBuf++);
	}
}

void SPI_NCBufDisplayHex(SPI_BufAllocInfo_st *pBufAllocInfo, bool EnDisplay)
{
	u8 *pBuf;
	u32 BufSize;
	u32	LoopCnt;

	if(EnDisplay != TRUE)
	{
		return;
	}

	UART_Printf("\n");
	pBuf = pBufAllocInfo->m_pBufPtr;
	BufSize = pBufAllocInfo->m_ReqDataSize;
	for(LoopCnt=0; LoopCnt<BufSize; LoopCnt++)
	{
		UART_Printf("%02x ", *pBuf++);
	}
}


int SPI_BufPatternInit(void *pBuf, u32 BufSize, char *pPattern, int PatternSize)
{
	u32		LocalLoop;
	char		*pDst;

	if(BufSize == 0 || PatternSize == 0)
	{
		UART_ASSERT();
		
		return UART_ERROR;
	}

	pDst = pBuf;
	if(PatternSize > BufSize)
	{
		PatternSize = BufSize;
	}

	while(BufSize>0)
	{
		LocalLoop = 0;
		while(LocalLoop < PatternSize)
		{
			*pDst = *(pPattern + LocalLoop);
			pDst++;
			LocalLoop++;
			BufSize--;
		}
	}

	return UART_NO_ERROR;
}



int SPI_BufRandInit(void *pBuf, u32 BufSize)
{
	u8 *pBufPtr;

	pBufPtr = pBuf;
	if(BufSize == 0)
	{
		UART_ASSERT();
		
		return UART_ERROR;
	}

	while(BufSize>0)
	{
		*pBufPtr = (u8)(rand() & 0xFF);
		pBufPtr++;
		BufSize--;
	}

	return UART_NO_ERROR;
}




int SPI_BufZeroInit(void *pBuf, u32 BufSize)
{
	if(BufSize == 0)
	{
		UART_ASSERT();
		
		return UART_ERROR;
	}

	memset(pBuf, 0x00, BufSize);

	return UART_NO_ERROR;
}

void SPI_BufDisplay(u8 *pBuf, u32 BufSize, bool EnDisplay)
{
	u32	LoopCnt;

	if(EnDisplay != TRUE)
	{
		return;
	}

	UART_Printf("\n");
	for(LoopCnt=0; LoopCnt<BufSize; LoopCnt++)
	{
		UART_Printf("%c", *pBuf++);
	}
}


void SPI_BufDisplayHex(u8 *pBuf, u32 BufSize, bool EnDisplay)
{
	u32	LoopCnt;

	if(EnDisplay != TRUE)
	{
		return;
	}

	UART_Printf("\n");
	for(LoopCnt=0; LoopCnt<BufSize; LoopCnt++)
	{
		UART_Printf("%02x ", *pBuf++);
	}
}

void SPI_DisplayVar_EnHighSpeed(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnHighSpeed == TRUE)
	{
		UART_Printf("Enable High Speed : ON");
	}
	else
	{
		UART_Printf("Enable High Speed : OFF");
	}
}


void SPI_DisplayVar_OpMode(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_OpMode == SPI_MASTER)
	{
		UART_Printf("Operation Mode : Master");
	}
	else
	{
		UART_Printf("Operation Mode : Slave");
	}
}


void SPI_DisplayVar_CPOL(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_CPOL == SPI_ACTIVE_HIGH)
	{
		UART_Printf("CPOL : Active HIGH");
	}
	else
	{
		UART_Printf("CPOL : Active LOW");
	}
}


void SPI_DisplayVar_CPHA(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_CPHA == SPI_FORMAT_A)
	{
		UART_Printf("CPHA : FORMAT_A");
	}
	else
	{
		UART_Printf("CPHA : FORMAT_B");
	}
}


void SPI_DisplayVar_EnRxChannel(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnRxChannel == TRUE)
	{
		UART_Printf("RxChannel : ON");
	}
	else
	{
		UART_Printf("RxChannel : OFF");
	}
}


void SPI_DisplayVar_EnTxChannel(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnTxChannel == TRUE)
	{
		UART_Printf("TxChannel : ON");
	}
	else
	{
		UART_Printf("TxChannel : OFF");
	}
}



void SPI_DisplayVar_IsUseExtClk(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_IsUseExtClk == SPI_PCLK)
	{
		UART_Printf("ExtClock : PCLK");
	}
	else
	{
		UART_Printf("ExtClock : EXT_CLK");
	}
}

void SPI_DisplayVar_EnClk(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnClk == TRUE)
	{
		UART_Printf("Enable Clk : ON");
	}
	else
	{
		UART_Printf("Enable Clk : OFF");
	}
}



void SPI_DisplayVar_SpiCLK(SPI_PortInfo_st *pPortInfo)
{
	float			RealSpiCLK;

	RealSpiCLK = (float)pPortInfo->m_OpClock/(2 *(pPortInfo->m_ClkDivValue+1));
	
	UART_Printf("Real SPI_CLK : %.2f MHz, Target SPI_CLK : %.2f MHz, PreScale(%d)", RealSpiCLK/1.0e6, 
		(float)pPortInfo->m_TargetSPICLK/1.0e6, pPortInfo->m_ClkDivValue);
}


void SPI_DisplayVar_ExtClkSrc(SPI_PortInfo_st *pPortInfo)
{
	switch(pPortInfo->m_ExtClkSrc)
	{
		case SPI_EXT_CLKSRC_XXTI:
			UART_Printf("EXT_CLK_SRC : XTI");
			break;
		case SPI_EXT_CLKSRC_XUSBXTI:
			UART_Printf("EXT_CLK_SRC : USBXTI");
			break;
		case SPI_EXT_CLKSRC_HDMI27M:
			UART_Printf("EXT_CLK_SRC : HDMI27M");
			break;
		case SPI_EXT_CLKSRC_USBPHY0:
			UART_Printf("EXT_CLK_SRC : PHY0");
			break;
		case SPI_EXT_CLKSRC_USBPHY1:
			UART_Printf("EXT_CLK_SRC : PHY1");
			break;
		case SPI_EXT_CLKSRC_HDMIPHY:
			UART_Printf("EXT_CLK_SRC : HDMIPHY");
			break;
		case SPI_EXT_CLKSRC_MPLL:
			UART_Printf("EXT_CLK_SRC : MPLL(%.2fMHz)", (float)g_uMPLL/1.0e6);
			break;
		case SPI_EXT_CLKSRC_EPLL:
			UART_Printf("EXT_CLK_SRC : EPLL(%.2fMHz)", (float)g_uEPLL/1.0e6);
			break;
		case SPI_EXT_CLKSRC_VPLL:
			UART_Printf("EXT_CLK_SRC : VPLL(%.2fMHz)", (float)g_uVPLL/1.0e6);
			break;
		default:
			UART_ASSERT();
			break;
	}
}


void SPI_DisplayVar_ExtClkSrcDiv(SPI_PortInfo_st *pPortInfo)
{
	SPI_CalcTargetClk(pPortInfo->m_IsUseExtClk, pPortInfo->m_TargetSPICLK,
		pPortInfo->m_ExtClkSrc, pPortInfo->m_ExtClkSrcDiv, &pPortInfo->m_OpClock, 
		&pPortInfo->m_ClkDivValue);

	UART_Printf("Ext Clock DIVIDE (from 1) : %d", pPortInfo->m_ExtClkSrcDiv);
}

void SPI_DisplayVar_OpClock(SPI_PortInfo_st *pPortInfo)
{
	/* Update Opclock */
	SPI_CalcTargetClk(pPortInfo->m_IsUseExtClk, pPortInfo->m_TargetSPICLK,
		pPortInfo->m_ExtClkSrc, pPortInfo->m_ExtClkSrcDiv, &pPortInfo->m_OpClock, 
		&pPortInfo->m_ClkDivValue);

	UART_Printf("OpClock(%.2fMHz)", (float)pPortInfo->m_OpClock/1.0e6);
}


void SPI_DisplayVar_ChWidth(SPI_PortInfo_st *pPortInfo)
{
	switch(pPortInfo->m_ChWidth)
	{
		case SPI_BYTE:
			UART_Printf("Channel Width : BYTE");
			break;
		case SPI_HWORD:
			UART_Printf("Channel Width : HWORD");
			break;
		case SPI_WORD:
			UART_Printf("Channel Width : WORD");
			break;

		default:
			UART_ASSERT();
			break;
	}
}

void SPI_DisplayVar_TrailingCnt(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Trailing Count : %d", pPortInfo->m_TraillingCnt);
}


void SPI_DisplayVar_BusWidth(SPI_PortInfo_st *pPortInfo)
{
	switch(pPortInfo->m_BusWidth)
	{
		case SPI_BYTE:
			UART_Printf("Bus Width : BYTE");
			break;
		case SPI_HWORD:
			UART_Printf("Bus Width : HWORD");
			break;
		case SPI_WORD:
			UART_Printf("Bus Width : WORD");
			break;

		default:
			UART_ASSERT();
			break;
	}
}

void SPI_DisplayVar_RxTriggerLevel(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Rx FIFO Trigger Level : %d(%d bytes)", pPortInfo->m_RxReadyLevel, 
		pPortInfo->m_RxReadyLevel * pPortInfo->m_FifoLevelGrid);
}


void SPI_DisplayVar_TxTriggerLevel(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Tx FIFO Trigger Level : %d(%d bytes)", pPortInfo->m_TxReadyLevel, 
		pPortInfo->m_TxReadyLevel * pPortInfo->m_FifoLevelGrid);
}


void SPI_DisplayVar_RxTransferMode(SPI_PortInfo_st *pPortInfo)
{
	switch(pPortInfo->m_RxTransferMode)
	{
		case SPI_INTERRUPT_MODE:
			UART_Printf("Rx Transfer Mode : INTERRUP");
			break;
		case SPI_DMA_MODE:
			UART_Printf("Rx Transfer Mode : DMA");
			break;

		default:
			UART_ASSERT();
			break;
	}
}

void SPI_DisplayVar_TxTransferMode(SPI_PortInfo_st *pPortInfo)
{
	switch(pPortInfo->m_TxTransferMode)
	{
		case SPI_INTERRUPT_MODE:
			UART_Printf("Tx Transfer Mode : INTERRUP");
			break;
		case SPI_DMA_MODE:
			UART_Printf("Tx Transfer Mode : DMA");
			break;

		default:
			UART_ASSERT();
			break;
	}
}

void SPI_DisplayVar_DMAType(SPI_PortInfo_st *pPortInfo)
{
	switch(pPortInfo->m_DMAType)
	{
		case SPI_DMA_SINGLE:
			UART_Printf("DMA (burst) Type : SINGLE");
			break;
		case SPI_DMA_4_BURST:
			UART_Printf("DMA (burst) Type : 4-BURST");
			break;

		default:
			UART_ASSERT();
			break;
	}
}


void SPI_DisplayVar_TxTransferSize(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Tx Data Transfer Size : %d byte(s)", pPortInfo->m_TxTransferSize);
}


void SPI_DisplayVar_RxTransferSize(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Rx Data Transfer Size : %d byte(s)", pPortInfo->m_RxTransferSize);
}



void SPI_DisplayVar_NCSTimeCnt(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("NCSTimeoutCnt : %d", pPortInfo->m_NCSTimeCnt);
}


void SPI_DisplayVar_EnAutoCS(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnAutoCS == TRUE)
	{
		UART_Printf("Enable Auto CS : ON");
	}
	else
	{
		UART_Printf("Enable Auto CS : OFF");
	}
}


void SPI_DisplayVar_EnTrailingInt(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnTrailingInt == TRUE)
	{
		UART_Printf("Trailing Interrupt : ON");
	}
	else
	{
		UART_Printf("Trailing Interrupt : OFF");
	}
}


void SPI_DisplayVar_EnRxOverrunInt(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnRxOverrunInt == TRUE)
	{
		UART_Printf("RxOverrun Interrupt : ON");
	}
	else
	{
		UART_Printf("RxOverrun Interrupt : OFF");
	}
}

void SPI_DisplayVar_EnRxUnderrunInt(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnRxUnderrunInt == TRUE)
	{
		UART_Printf("RxUnderrun Interrupt : ON");
	}
	else
	{
		UART_Printf("RxUnderrun Interrupt : OFF");
	}
}



void SPI_DisplayVar_EnTxOverrunInt(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnTxOverrunInt == TRUE)
	{
		UART_Printf("TxOverrun Interrupt : ON");
	}
	else
	{
		UART_Printf("TxOverrun Interrupt : OFF");
	}
}

void SPI_DisplayVar_EnTxUnderrunInt(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnTxUnderrunInt == TRUE)
	{
		UART_Printf("TxUnderrun Interrupt : ON");
	}
	else
	{
		UART_Printf("TxUnderrun Interrupt : OFF");
	}
}


void SPI_DisplayVar_EnRxHwordSwap(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnRxHWordSwap == TRUE)
	{
		UART_Printf("Rx HWORD Swap : ON");
	}
	else
	{
		UART_Printf("Rx HWORD Swap : OFF");
	}
}


void SPI_DisplayVar_EnRxByteSwap(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnRxByteSwap == TRUE)
	{
		UART_Printf("Rx HWORD Swap : ON");
	}
	else
	{
		UART_Printf("Rx HWORD Swap : OFF");
	}
}


void SPI_DisplayVar_EnRxBitSwap(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnRxBitSwap == TRUE)
	{
		UART_Printf("Rx Bit Swap : ON");
	}
	else
	{
		UART_Printf("Rx Bit Swap : OFF");
	}
}


void SPI_DisplayVar_EnRxSwap(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnRxSwap == TRUE)
	{
		UART_Printf("Enable Rx Swap : ON");
	}
	else
	{
		UART_Printf("Enable Rx Swap : OFF");
	}
}


void SPI_DisplayVar_EnTxHwordSwap(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnTxHWordSwap == TRUE)
	{
		UART_Printf("Tx HWORD Swap : ON");
	}
	else
	{
		UART_Printf("Tx HWORD Swap : OFF");
	}
}



void SPI_DisplayVar_EnTxByteSwap(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnTxByteSwap == TRUE)
	{
		UART_Printf("Tx HWORD Swap : ON");
	}
	else
	{
		UART_Printf("Tx HWORD Swap : OFF");
	}
}


void SPI_DisplayVar_EnTxBitSwap(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnTxBitSwap == TRUE)
	{
		UART_Printf("Tx Bit Swap : ON");
	}
	else
	{
		UART_Printf("Tx Bit Swap : OFF");
	}
}


void SPI_DisplayVar_EnTxSwap(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_EnTxSwap == TRUE)
	{
		UART_Printf("Enable Tx Swap : ON");
	}
	else
	{
		UART_Printf("Enable Tx Swap : OFF");
	}
}

void SPI_DisplayVar_FBClkSel(SPI_PortInfo_st *pPortInfo)
{
	switch(pPortInfo->m_FBClkSel)
	{
		case SPI_PHASE_DELAY0:
			UART_Printf("FB Clock Select : PHASE 0 degree");
			break;
		case SPI_PHASE_DELAY90:
			UART_Printf("FB Clock Select : PHASE 90 degree");
			break;
		case SPI_PHASE_DELAY180:
			UART_Printf("FB Clock Select : PHASE 180 degree");
			break;
		case SPI_PHASE_DELAY270:
			UART_Printf("FB Clock Select : PHASE 270 degree");
			break;
		case SPI_0NS_DELAY:
			UART_Printf("FB Clock Select : PHASE 0ns delay");
			break;
		case SPI_3NS_DELAY:
			UART_Printf("FB Clock Select : PHASE 3ns delay");
			break;
		case SPI_6NS_DELAY:
			UART_Printf("FB Clock Select : PHASE 6ns delay");
			break;
		case SPI_9NS_DELAY:
			UART_Printf("FB Clock Select : PHASE 9ns delay");
			break;

		default:
			UART_ASSERT();
			break;
	}
}

void SPI_DisplayVar_DrvStrength(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Drv Strength : %d", pPortInfo->m_DrvStrength);
}



void SPI_DisplayVar_RxDMANum(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_RxDMANum == NUM_PDMA0)
	{
		UART_Printf("Rx DMA : PDMA0");
	}
	else if(pPortInfo->m_RxDMANum == NUM_PDMA1)
	{
		UART_Printf("Rx DMA : PDMA1");
	}
	else
	{
		SPI_ASSERT();
	}
}

void SPI_DisplayVar_TxDMANum(SPI_PortInfo_st *pPortInfo)
{
	if(pPortInfo->m_TxDMANum == NUM_PDMA0)
	{
		UART_Printf("Tx DMA : PDMA0");
	}
	else if(pPortInfo->m_TxDMANum == NUM_PDMA1)
	{
		UART_Printf("Tx DMA : PDMA1");
	}
	else
	{
		SPI_ASSERT();
	}
}

void SPI_DisplayVar_RxDMACh(SPI_PortInfo_st *pPortInfo)
{

	switch(pPortInfo->m_RxDmaCh)
	{
		case DMA_00:
			UART_Printf("Rx DMA_CH : DMA_00");
			break;
		case DMA_01:
			UART_Printf("Rx DMA_CH : DMA_01");
			break;
		case DMA_02:
			UART_Printf("Rx DMA_CH : DMA_02");
			break;
		case DMA_03:
			UART_Printf("Rx DMA_CH : DMA_03");
			break;
		case DMA_04:
			UART_Printf("Rx DMA_CH : DMA_04");
			break;
		case DMA_05:
			UART_Printf("Rx DMA_CH : DMA_05");
			break;
		case DMA_06:
			UART_Printf("Rx DMA_CH : DMA_06");
			break;
		case DMA_07:
			UART_Printf("Rx DMA_CH : DMA_07");
			break;
		case DMA_10:
			UART_Printf("Rx DMA_CH : DMA_10");
			break;
		case DMA_11:
			UART_Printf("Rx DMA_CH : DMA_11");
			break;
		case DMA_12:
			UART_Printf("Rx DMA_CH : DMA_12");
			break;
		case DMA_13:
			UART_Printf("Rx DMA_CH : DMA_13");
			break;
		case DMA_14:
			UART_Printf("Rx DMA_CH : DMA_14");
			break;
		case DMA_15:
			UART_Printf("Rx DMA_CH : DMA_15");
			break;
		case DMA_16:
			UART_Printf("Rx DMA_CH : DMA_16");
			break;
		case DMA_17:
			UART_Printf("Rx DMA_CH : DMA_17");
			break;
		default:
			UART_ASSERT();
			break;
	}
}


void SPI_DisplayVar_TxDMACh(SPI_PortInfo_st *pPortInfo)
{
	switch(pPortInfo->m_TxDmaCh)
	{
		case DMA_00:
			UART_Printf("Tx DMA_CH : DMA_00");
			break;
		case DMA_01:
			UART_Printf("Tx DMA_CH : DMA_01");
			break;
		case DMA_02:
			UART_Printf("Tx DMA_CH : DMA_02");
			break;
		case DMA_03:
			UART_Printf("Tx DMA_CH : DMA_03");
			break;
		case DMA_04:
			UART_Printf("Tx DMA_CH : DMA_04");
			break;
		case DMA_05:
			UART_Printf("Tx DMA_CH : DMA_05");
			break;
		case DMA_06:
			UART_Printf("Tx DMA_CH : DMA_06");
			break;
		case DMA_07:
			UART_Printf("Tx DMA_CH : DMA_07");
			break;
		case DMA_10:
			UART_Printf("Tx DMA_CH : DMA_10");
			break;
		case DMA_11:
			UART_Printf("Tx DMA_CH : DMA_11");
			break;
		case DMA_12:
			UART_Printf("Tx DMA_CH : DMA_12");
			break;
		case DMA_13:
			UART_Printf("Tx DMA_CH : DMA_13");
			break;
		case DMA_14:
			UART_Printf("Tx DMA_CH : DMA_14");
			break;
		case DMA_15:
			UART_Printf("Tx DMA_CH : DMA_15");
			break;
		case DMA_16:
			UART_Printf("Tx DMA_CH : DMA_16");
			break;
		case DMA_17:
			UART_Printf("Tx DMA_CH : DMA_17");
			break;
		default:
			UART_ASSERT();
			break;
	}
}


void SPI_DisplayVar_LoadDefault(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Load Default Setting...");
}

void SPI_DisplayVar_EnSWReset(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Enable S/W Reset : %s", pPortInfo->m_EnSWReset ? "ON" : "OFF");
}

void SPI_DisplayVar_SWReset(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("S/W Reset");
}

void SPI_DisplayVar_ReadRegister(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Read Register");
}

void SPI_DisplayVar_EnInterruptPrint(SPI_PortInfo_st *pPortInfo)
{
	UART_Printf("Enable Interrupt Print : %s", gSPI_CtrlInfo.m_EnPrintInISR ? "ON" : "OFF");
}


void SPI_SetVar_EnHighSpeed(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnHighSpeed : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnHighSpeed = (u32)SelNum;
}



void SPI_SetVar_OpMode(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nOperation Mode : ");
	UART_Printf("\n");

	UART_Printf("\n0.Master(D)");
	UART_Printf("\n1.Slave");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_OpMode = (u32)SelNum;
}


void SPI_SetVar_CPOL(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nCPOL : ");
	UART_Printf("\n");

	UART_Printf("\n0.Active HIGH");
	UART_Printf("\n1.Active LOW(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}
	pPortInfo->m_CPOL = (u32)SelNum;
}


void SPI_SetVar_CPHA(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nCPHA : ");
	UART_Printf("\n");

	UART_Printf("\n0.FORMAT_A(D)");
	UART_Printf("\n1.FORMAT_B");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_CPHA = (u32)SelNum;
}


void SPI_SetVar_EnRxChannel(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable RxChannel : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnRxChannel = (u32)SelNum;
}


void SPI_SetVar_EnTxChannel(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable TxChannel : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnTxChannel = (u32)SelNum;
}


void SPI_SetVar_IsUseExtClk(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;
	s32				Status;

	UART_Printf("\nOperation Clock : ");
	UART_Printf("\n");

	UART_Printf("\n0.PCLK(D)");
	UART_Printf("\n1.EXT_CLK");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}

	Status = SPI_CalcTargetClk(SelNum, pPortInfo->m_TargetSPICLK,
		pPortInfo->m_ExtClkSrc, pPortInfo->m_ExtClkSrcDiv, &pPortInfo->m_OpClock, 
		&pPortInfo->m_ClkDivValue);
	if(Status == SPI_NO_ERROR)
	{
		pPortInfo->m_IsUseExtClk = (u32)SelNum;
		SPI_SelectExtClkSrc(pPortInfo);
	}
}


void SPI_SetVar_EnClk(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nClock Enable : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF");
	UART_Printf("\n1.ON(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}

	pPortInfo->m_EnClk = (u32)SelNum;
}


void SPI_SetVar_SpiCLK(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;
	s32				Status;

	UART_Printf("\nTarget SPI_CLK (1 ~ 50000000) : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 1 || SelNum > 50000000)
	{
		SelNum = 1;
	}

	Status = SPI_CalcTargetClk(pPortInfo->m_IsUseExtClk, SelNum,
		pPortInfo->m_ExtClkSrc, pPortInfo->m_ExtClkSrcDiv, &pPortInfo->m_OpClock, 
		&pPortInfo->m_ClkDivValue);
	if(Status == SPI_NO_ERROR)
	{
		pPortInfo->m_TargetSPICLK = (u32)SelNum;
		SPI_SelectExtClkSrc(pPortInfo);
	}
}


void SPI_SetVar_ExtClkSrc(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;
	s32				Status;

	UART_Printf("\nEXT_CLK_SRC : ");
	UART_Printf("\n");
	
	UART_Printf("\n0.XXTI");
	UART_Printf("\n1.USBXTI[D]");
	UART_Printf("\n2.HDMI27M");
	UART_Printf("\n4.USBPHY1");
	UART_Printf("\n5.HDMYPHY");
	UART_Printf("\n6.MPLL(%.2fMHz)", (float)g_uMPLL/1.0e6);
	UART_Printf("\n7.EPLL(%.2fMHz)", (float)g_uEPLL/1.0e6);
	UART_Printf("\n8.VPLL(%.2fMHz)", (float)g_uVPLL/1.0e6);
	UART_Printf("\n");
	
	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 8)
	{
		SelNum = 1;
	}

	Status = SPI_CalcTargetClk(pPortInfo->m_IsUseExtClk, pPortInfo->m_TargetSPICLK,
		SelNum, pPortInfo->m_ExtClkSrcDiv, &pPortInfo->m_OpClock, 
		&pPortInfo->m_ClkDivValue);
	if(Status == SPI_NO_ERROR)
	{
		pPortInfo->m_ExtClkSrc = (u32)SelNum;
		SPI_SelectExtClkSrc(pPortInfo);
	}
}


void SPI_SetVar_ExtClkSrcDiv(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;
	s32				Status;

	UART_Printf("\nBaudRate (1[D] ~ 8) : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 8)
	{
		SelNum = 1;
	}

	Status = SPI_CalcTargetClk(pPortInfo->m_IsUseExtClk, pPortInfo->m_TargetSPICLK,
		pPortInfo->m_ExtClkSrc, SelNum, &pPortInfo->m_OpClock, 
		&pPortInfo->m_ClkDivValue);
	if(Status == SPI_NO_ERROR)
	{
		pPortInfo->m_ExtClkSrcDiv = (u32)SelNum;
		SPI_SelectExtClkSrc(pPortInfo);
	}
}


void SPI_SetVar_ChWidth(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nChannel Width : ");
	UART_Printf("\n");
	
	UART_Printf("\n0.BYTE");
	UART_Printf("\n1.HWORD");
	UART_Printf("\n2.WORD[D]");
	UART_Printf("\n");
	
	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 2)
	{
		SelNum = 2;
	}

	pPortInfo->m_ChWidth = (u32)SelNum;
}


void SPI_SetVar_TrailingCnt(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nBaudRate (0[D] ~ 1024) : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1024)
	{
		SelNum = 0;
	}

	pPortInfo->m_TraillingCnt = (u32)SelNum;
}


void SPI_SetVar_BusWidth(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nChannel Width : ");
	UART_Printf("\n");
	
	UART_Printf("\n0.BYTE");
	UART_Printf("\n1.HWORD");
	UART_Printf("\n2.WORD[D]");
	UART_Printf("\n");
	
	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 2)
	{
		SelNum = 2;
	}

	pPortInfo->m_BusWidth = (u32)SelNum;
}


void SPI_SetVar_RxTriggerLevel(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nRx Trigger Level (0..1[D]..64) : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 64)
	{
		SelNum = 1;
	}

	pPortInfo->m_RxReadyLevel = (u32)SelNum;
}


void SPI_SetVar_TxTriggerLevel(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nTx Trigger Level (0..1[D]..64) : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 64)
	{
		SelNum = 1;
	}

	pPortInfo->m_TxReadyLevel = (u32)SelNum;
}


void SPI_SetVar_RxTransferMode(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nRx Transfer Mode : ");
	UART_Printf("\n");
	
	UART_Printf("\n0.Interrupt Mode");
	UART_Printf("\n1.DMA[D]");
	UART_Printf("\n");
	
	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}

	pPortInfo->m_RxTransferMode = (u32)SelNum;
}


void SPI_SetVar_TxTransferMode(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nTx Transfer Mode : ");
	UART_Printf("\n");
	
	UART_Printf("\n0.Interrupt Mode");
	UART_Printf("\n1.DMA[D]");
	UART_Printf("\n");
	
	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}

	pPortInfo->m_TxTransferMode = (u32)SelNum;
}


void SPI_SetVar_DMAType(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nDMA (burst) Type: ");
	UART_Printf("\n");
	
	UART_Printf("\n0.SINGLE");
	UART_Printf("\n1.4-Burst[D]");
	UART_Printf("\n");
	
	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}

	pPortInfo->m_DMAType = (u32)SelNum;
}

void SPI_SetVar_TxTransferSize(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nTx Data Transfer Size (1..8[D]..65536  byte(s)) : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 65536)
	{
		SelNum = 8;
	}

	pPortInfo->m_TxTransferSize = (u32)SelNum;
}


void SPI_SetVar_RxTransferSize(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nRx Data Transfer Size (1..8[D]..65536  byte(s)) : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 65536)
	{
		SelNum = 8;
	}

	pPortInfo->m_RxTransferSize = (u32)SelNum;
}




void SPI_SetVar_NCSTimeCnt(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nBaudRate (0[D] ~ 64) : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 64)
	{
		SelNum = 0;
	}

	pPortInfo->m_NCSTimeCnt = (u32)SelNum;
}


void SPI_SetVar_EnAutoCS(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nAuto CS(Chip Select) : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF");
	UART_Printf("\n1.ON(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}
	pPortInfo->m_EnAutoCS = (u32)SelNum;
}


void SPI_SetVar_EnTrailingInt(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Trailing Interrupt : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF");
	UART_Printf("\n1.ON(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}
	pPortInfo->m_EnTrailingInt = (u32)SelNum;
}


void SPI_SetVar_EnRxOverrunInt(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable RxOverrun Interrupt : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF");
	UART_Printf("\n1.ON(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}
	pPortInfo->m_EnRxOverrunInt = (u32)SelNum;
}


void SPI_SetVar_EnRxUnderrunInt(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable RxUnderrun Interrupt : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF");
	UART_Printf("\n1.ON(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}
	pPortInfo->m_EnRxUnderrunInt = (u32)SelNum;
}


void SPI_SetVar_EnTxOverrunInt(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable TxOverrun Interrupt : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF");
	UART_Printf("\n1.ON(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}
	pPortInfo->m_EnTxOverrunInt = (u32)SelNum;
}


void SPI_SetVar_EnTxUnderrunInt(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable RxUnderrun Interrupt : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF");
	UART_Printf("\n1.ON(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}
	pPortInfo->m_EnTxUnderrunInt = (u32)SelNum;
}


void SPI_SetVar_EnRxHwordSwap(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Tx HWORD Swap : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnRxHWordSwap = (u32)SelNum;
}


void SPI_SetVar_EnRxByteSwap(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Tx BYTE Swap : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnRxByteSwap = (u32)SelNum;
}


void SPI_SetVar_EnRxBitSwap(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Tx Bit Swap : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnRxBitSwap = (u32)SelNum;
}


void SPI_SetVar_EnRxSwap(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Rx Swap : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnRxSwap = (u32)SelNum;
}



void SPI_SetVar_EnTxHwordSwap(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Tx HWORD Swap : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnTxHWordSwap = (u32)SelNum;
}


void SPI_SetVar_EnTxByteSwap(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Tx BYTE Swap : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnTxByteSwap = (u32)SelNum;
}


void SPI_SetVar_EnTxBitSwap(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Tx Bit Swap : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnTxBitSwap = (u32)SelNum;
}


void SPI_SetVar_EnTxSwap(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Tx Swap : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF(D)");
	UART_Printf("\n1.ON");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}
	pPortInfo->m_EnTxSwap = (u32)SelNum;
}


void SPI_SetVar_FBClkSel(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Tx Swap : ");
	UART_Printf("\n");

	UART_Printf("\n0.PHASE 0 Degree(D)");
	UART_Printf("\n1.PHASE 90 Degree");
	UART_Printf("\n2.PHASE 180 Degree");
	UART_Printf("\n3.PHASE 270 Degree");
	UART_Printf("\n4.0ns Delay");
	UART_Printf("\n5.3ns Delay");
	UART_Printf("\n6.6ns Delay");
	UART_Printf("\n7.9ns Delay");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 7)
	{
		SelNum = 0;
	}
	pPortInfo->m_FBClkSel = (u32)SelNum;
}


void SPI_SetVar_DrvStrength(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nDrive Strength (0 ~ 3[D]) : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 3)
	{
		SelNum = 3;
	}

	pPortInfo->m_DrvStrength = (u32)SelNum;
}


void SPI_SetVar_RxDMACh(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nRX DMA_CH : ");
	UART_Printf("\n");
	
	UART_Printf("\n 0.DMA_00[D]");
	UART_Printf("\n 1.DMA_01");
	UART_Printf("\n 2.DMA_02");
	UART_Printf("\n 3.DMA_03");
	UART_Printf("\n 4.DMA_04");
	UART_Printf("\n 5.DMA_05");
	UART_Printf("\n 6.DMA_06");
	UART_Printf("\n 7.DMA_07");
	UART_Printf("\n");

	UART_Printf("\n 8.DMA_10");
	UART_Printf("\n 9.DMA_11");
	UART_Printf("\n10.DMA_12");
	UART_Printf("\n11.DMA_13");
	UART_Printf("\n12.DMA_14");
	UART_Printf("\n13.DMA_15");
	UART_Printf("\n14.DMA_16");
	UART_Printf("\n15.DMA_17");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 15)
	{
		SelNum = 0;
	}
	
	pPortInfo->m_RxDmaCh = (u32)SelNum;
	if(SelNum < 8)
	{
		pPortInfo->m_RxDMANum = NUM_PDMA0;
	}
	else
	{
		pPortInfo->m_RxDMANum = NUM_PDMA1;
	}
}



void SPI_SetVar_TxDMACh(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nRX DMA_CH : ");
	UART_Printf("\n");
	
	UART_Printf("\n 0.DMA_00");
	UART_Printf("\n 1.DMA_01[D]");
	UART_Printf("\n 2.DMA_02");
	UART_Printf("\n 3.DMA_03");
	UART_Printf("\n 4.DMA_04");
	UART_Printf("\n 5.DMA_05");
	UART_Printf("\n 6.DMA_06");
	UART_Printf("\n 7.DMA_07");
	UART_Printf("\n");

	UART_Printf("\n 8.DMA_10");
	UART_Printf("\n 9.DMA_11");
	UART_Printf("\n10.DMA_12");
	UART_Printf("\n11.DMA_13");
	UART_Printf("\n12.DMA_14");
	UART_Printf("\n13.DMA_15");
	UART_Printf("\n14.DMA_16");
	UART_Printf("\n15.DMA_17");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 15)
	{
		SelNum = 0;
	}
	
	pPortInfo->m_TxDmaCh = (u32)SelNum;
	if(SelNum < 8)
	{
		pPortInfo->m_TxDMANum = NUM_PDMA0;
	}
	else
	{
		pPortInfo->m_TxDMANum = NUM_PDMA1;
	}
}

void SPI_SetVar_LoadDefault(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nAre you sure? : ");
	UART_Printf("\n");

	UART_Printf("\n0.NO(D)");
	UART_Printf("\n1.YES");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 0;
	}

	if(SelNum == 0)
	{
		/* Do nothing */
		
		return;
	}

	switch(pPortInfo->m_SPIPortIdx)
	{
		case SPI_PORT0:
			*pPortInfo = gSPI_CtrlInfo.m_SPIPortDefaultInfo[SPI_PORT0];
			break;
			
		case SPI_PORT1:
			*pPortInfo = gSPI_CtrlInfo.m_SPIPortDefaultInfo[SPI_PORT1];
			break;

		case SPI_PORT2:
			*pPortInfo = gSPI_CtrlInfo.m_SPIPortDefaultInfo[SPI_PORT2];
			break;

		default:
			UART_ASSERT();
			break;
	}
}


void SPI_SetVar_EnSWReset(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable S/W Reset : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF");
	UART_Printf("\n1.ON(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}

	pPortInfo->m_EnSWReset = SelNum;
}


void SPI_SetVar_SWReset(SPI_PortInfo_st *pPortInfo)
{
	SPI_SWReset(pPortInfo);
}


void SPI_SetVar_ReadRegister(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;
	s32				Status;

	while(1)
	{
		UART_Printf("\nRead Register : ");
		UART_Printf("\n");
		
		UART_Printf("\n0.CH_CFG           (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x00);
		UART_Printf("\n1.CLK_CFG          (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x04);
		UART_Printf("\n2.MODE_CFG         (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x08);
		UART_Printf("\n3.CS_REG           (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x0C);
		UART_Printf("\n4.SPI_INT_EN       (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x10);
		UART_Printf("\n5.SPI_STATUS       (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x14);
		UART_Printf("\n6.TX_DATA          (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x18);
		UART_Printf("\n7.RX_DATA          (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x1C);
		UART_Printf("\n8.PACKET_CNT_REG   (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x20);
		UART_Printf("\n9.PENDING_CLR_REG  (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x24);
		UART_Printf("\n10.SWAP_CFG        (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x28);
		UART_Printf("\n11.FB_CLK_SEL      (%08X)",  (u32)pPortInfo->m_SPISFRBase + 0x2C);
		UART_Printf("\n12.ALL\n");
		UART_Printf("\n13.Exit");
		UART_Printf("\n");
		
		UART_Printf("\nChoose : ");

		SelNum = UART_GetIntNum();
		if(SelNum == 13)
		{
			/* Do nothing */

			return;
		}

		if(SelNum < 0 || SelNum > 12)
		{
			break;
		}

		if(SelNum >=0 && SelNum <= 11)
		{
			UART_Printf("\n%08X : %08X",  (u32)pPortInfo->m_SPISFRBase + SelNum * 4, 
				*(volatile u32 *)((u32)pPortInfo->m_SPISFRBase + SelNum * 4));
		}
		else
		{
			int LoopCnt;

			for(LoopCnt=0; LoopCnt < 12; LoopCnt++)
			{
				UART_Printf("\n%08X : %08X",  (u32)pPortInfo->m_SPISFRBase + LoopCnt * 4, 
					*(volatile u32 *)((u32)pPortInfo->m_SPISFRBase + LoopCnt * 4));
			}
		}

		UART_Printf("\n\nPress any key to continue....\n\n");

		while(!UART_GetKey())
		{
		}

	}
}

void SPI_SetVar_EnInterruptPrint(SPI_PortInfo_st *pPortInfo)
{
	s32				SelNum;

	UART_Printf("\nEnable Interrupt Print : ");
	UART_Printf("\n");

	UART_Printf("\n0.OFF");
	UART_Printf("\n1.ON(D)");
	UART_Printf("\n");

	UART_Printf("\nChoose : ");

	SelNum = UART_GetIntNum();
	if(SelNum < 0 || SelNum > 1)
	{
		SelNum = 1;
	}

	gSPI_CtrlInfo.m_EnPrintInISR = SelNum;
}


