#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "def.h"
#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "sysc.h"
#include "gpio.h"
#include "dma.h"
#include "intc.h"
#include "spdif.h"
#include "wavefile.h"
#include "ac3file.h"

extern void TZPC_NonSecureInit(void);
void MakeWaveFilePutHeader(void);
void Make24BitTo32BitForTest(void);

#define WAVE_BUF_START		CODEC_MEM_ST
#define PCM_BUF_START		(WAVE_BUF_START+WAVFILE_HEADER_SZ)
#define AC3_BUF_START		CODEC_MEM_ST

static u32 uPcmSize;
static u32 uAc3Size;
static u8 IntPlayingDone; 
CLK_SEL eUsedAudioMainClk;
FS_SEL eUsedAudioFs;

void __irq Isr_SPDIF(void)
{
#if 1	 // for Interrupt 
	u32 uRead; 
	INTC_Disable(NUM_SPDIF);
	INTC_ClearVectAddr();

	//UART_Printf("\nSPDIF Interrupt happend!!\n");

	uRead = SPDIF_ReadIntFlag();

	if(uRead&(1<<7))
	{
		UART_Printf("stream_end_flag interrupt is generated!\n");
		SPDIF_ClearIntFlag(stream_end_flag);
	}

	if(uRead&(1<<9))
	{
		UART_Printf("buffer_empty_flag interrupt is generated!\n");
		SPDIF_ClearIntFlag(buffer_empty_flag);
	}

	if(uRead&(1<<11))
	{
		UART_Printf("user_data_flag interrupt is generated!\n");
		SPDIF_ClearIntFlag(user_data_flag);
	}
	
	if(uRead&(1<<16))
	{
		UART_Printf("fifo_level_flag interrupt is generated!\n");
		SPDIF_ClearIntFlag(fifo_level_flag);
	}

	INTC_Enable(NUM_SPDIF);
#else	// for User Data Interrupt
	u32 uRead; 
	INTC_Disable(NUM_SPDIF);
	INTC_ClearVectAddr();

	SPDIF_IntHandler();

	INTC_Enable(NUM_SPDIF);
#endif

}

static void TestPlayingWavFileByCpu(void)
{
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;

	UART_Printf("\nTest start!\n");

	/// 1. Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);
	
	/// 1.2 Protection Code, 768fs 
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}

	/// 2. SPDIF Main Clock Setting, Syscon
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	{
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);
	}
	
	/// 3. Wave file playing start by CPU
	SPDIF_StartPlayingPcmByCpu(PCM_BUF_START, uPcmSize, PCM, uChannelType, uSampleRate, uBitsPerSample);		//size: uPcmSize*2 byte

	UART_Printf("\nTest end!\n");
}

static void TestPlayingWavFileByDma(void)
{
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;

	UART_Printf("\nTest start!\n");

	/// 1. Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}

	/// 2. SPDIF Main Clock Setting, Syscon
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);
	
	/// 3. Wave file playing start by DMA
	SPDIF_StartPlayingPcmByDma(PCM_BUF_START, uPcmSize, PCM, uChannelType, uSampleRate, uBitsPerSample);		//size: uPcmSize*2 byte

	UART_Printf("\nTest end!\n");
}

static void TestPlayingAc3FileByCpu(void)
{
	u32 uSyncword, uSampleRate, uBitRate, uFrameSizeCode, Save_Ac3Size;	

	UART_Printf("\nTest start!\n");

	//uAc3Size = 0xfffff*2;
	uAc3Size = 0x9b000;
	
	/// 1. AC3 file parsing
	AC3FILE_ParseHeader(AC3_BUF_START, &uSyncword, &uSampleRate, &uBitRate, &uFrameSizeCode);

	UART_Printf("\nSyncword: 0x%04x\n", uSyncword);
	UART_Printf("SampleRate(Hz): %d\n", uSampleRate);
	UART_Printf("BitRate(kbps): %d\n", uBitRate);
	UART_Printf("FrameSizeCode: %d\n", uFrameSizeCode);
	Save_Ac3Size = uAc3Size;
	UART_Printf("Ac3FileSize : %d\n", Save_Ac3Size);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}
	
	/// 2. SPDIF Main Clock Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	/// 3. AC3 file playing start by DMA
	SPDIF_StartPlayingAc3ByCpu(AC3_BUF_START, Save_Ac3Size, AC3, uSampleRate, uBitRate, uFrameSizeCode);		//size 0xfffff*2 byte
	
	UART_Printf("\nTest end!\n");
}

static void TestPlayingAc3FileByDma(void)
{
	u32 uSyncword, uSampleRate, uBitRate, uFrameSizeCode, Save_Ac3Size;	

	UART_Printf("\nTest start!\n");

	//uAc3Size = 0xfffff*2;
	uAc3Size = 0x9b000;
	
	/// 1. AC3 file parsing
	AC3FILE_ParseHeader(AC3_BUF_START, &uSyncword, &uSampleRate, &uBitRate, &uFrameSizeCode);

	UART_Printf("\nSyncword: 0x%04x\n", uSyncword);
	UART_Printf("SampleRate(Hz): %d\n", uSampleRate);
	UART_Printf("BitRate(kbps): %d\n", uBitRate);
	UART_Printf("FrameSizeCode: %d\n", uFrameSizeCode);
	Save_Ac3Size = uAc3Size;
	UART_Printf("Ac3FileSize : %d\n", Save_Ac3Size);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}
	
	/// 2. SPDIF Main Clock Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	/// 3. AC3 file playing start by DMA
	SPDIF_StartPlayingAc3ByDma(AC3_BUF_START, Save_Ac3Size, AC3, uSampleRate, uBitRate, uFrameSizeCode);		//size 0xfffff*2 byte
	
	UART_Printf("\nTest end!\n");
}

static void SpdifAgingTest(void)
{
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;

	UART_Printf("\nTest start!\n");

	/// 1. Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	/// 2. SPDIF Main Clock Setting, Syscon
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);
	
	/// 3. Wave file playing start by DMA
	UART_Printf("If you want to exit, please press any key\n");	
	while(!UART_GetKey())
	{
		SPDIF_StartPlayingPcmByDma(PCM_BUF_START, uPcmSize, PCM, uChannelType, uSampleRate, uBitsPerSample);		//size: uPcmSize*2 byte
	}

	UART_Printf("\nTest end!\n");
}

static void ClearSpdifBuffer(void)
{
	u32 i;

	for(i=0; i<188*54; i++)
	*(u8 *)(PCM_BUF_START+i) = 0;

	for(i=0; i<188*54; i++)
	*(u8 *)(PCM_BUF_START+0x01000000+i) = 0;	
}

#if 0
void Make24BitTo32BitForTest(void)
{
	u32 *pSrcAddr, *pTrgAddr, uHeaderSize, uTempPcmSize;
	u8 *p8BitSrcAddr;
	u32 uChannelType, uSampleRate, uBitsPerSample;	

	//pPcmAddr = (u32*)PCM_BUF_START;
	pSrcAddr = (u32*)(PCM_BUF_START);
	pTrgAddr =  (u32*)(PCM_BUF_START+0x01000000);
		
	uHeaderSize = 44;

	//ClearSpdifBuffer();

	/// 1. Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	if(uPcmSize == 0)
		uPcmSize = 2582876;

	WAVEFILE_PutHeader(WAVE_BUF_START, uPcmSize, uChannelType, uSampleRate, uBitsPerSample);

	/// 2. src addr => trg addr copy
	uTempPcmSize = uPcmSize;

	while (uTempPcmSize/4 > 0 )
	{
		*pTrgAddr++ = *pSrcAddr++;
		uTempPcmSize -= 4;	
	}

	/// 3. 24bit => 32bit change
	uTempPcmSize = uPcmSize;	
	pSrcAddr = (u32*)(PCM_BUF_START);
	pTrgAddr =  (u32*)(PCM_BUF_START+0x01000000);

	uTempPcmSize += uTempPcmSize/3;

	while (uTempPcmSize/4 > 0 )
	{
		*((u8*)pSrcAddr)++ = *((u8*)pTrgAddr)++;
		*((u8*)pSrcAddr)++ = *((u8*)pTrgAddr)++;
		*((u8*)pSrcAddr)++ = *((u8*)pTrgAddr)++;
		*((u8*)pSrcAddr)++ = 0x00;		
		uTempPcmSize -= 4;			
	}

}
#endif
#if 0
void MakeWaveFilePutHeader(u32 uPcmSize, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample)
{
	WAVEFILE_PutHeader(WAVE_BUF_START, uPcmSize, uChannelType, uSampleRate, uBitsPerSample);
/*
	Disp("\nChannelType: %d\n", uChannelType);
	Disp("SampleRate: %d\n", uSampleRate);
	Disp("BitsPerSample: %d\n", uBitsPerSample);
	Disp("PcmSize: %d\n", uPcmSize);
*/
}
#endif 

static void TestInterruptByCpu(void)
{
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;

	UART_Printf("\nTest start!\n");

	/// 1. Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	/// 2. SPDIF Main Clock Setting, Syscon
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	INTC_Init();
	INTC_SetVectAddr(NUM_SPDIF, Isr_SPDIF);
	INTC_Enable(NUM_SPDIF);	
	
	/// 3. Wave file playing start by Cpu
	SPDIF_TestInterruptByCpu(PCM_BUF_START, uPcmSize, PCM, uChannelType, uSampleRate, uBitsPerSample);		//size: uPcmSize*2 byte

	UART_Printf("\nTest end!\n");
}

static void TestInterruptByDma(void)
{
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;

	UART_Printf("\nTest start!\n");

	/// 1. Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	/// 2. SPDIF Main Clock Setting, Syscon
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	INTC_Init();
	INTC_SetVectAddr(NUM_SPDIF, Isr_SPDIF);
	INTC_Enable(NUM_SPDIF);	
	
	/// 3. Wave file playing start by DMA
	SPDIF_TestInterruptByDma(PCM_BUF_START, uPcmSize, PCM, uChannelType, uSampleRate, uBitsPerSample);		//size: uPcmSize*2 byte

	UART_Printf("\nTest end!\n");
}

/*************************************************************************
*
*	Preamble Information for AC3
*	Pa = 0xF872, Pb = 0x4E1F, Pc = 0x0001, Pd = 0x3000 = 1536*8
*
*
**************************************************************************/

void * func_spdif_test[][2]=
{	
	//  									"123456789012345678901"
	0,  				  		         	  	"Exit",
	(void *)TestPlayingWavFileByCpu,		"TestPlayingWavFileByCpu",	
	(void *)TestPlayingWavFileByDma,		"TestPlayingWavFileByDma",
	(void *)TestPlayingAc3FileByCpu,			"TestPlayingAc3FileByCpu",
	(void *)TestPlayingAc3FileByDma,		"TestPlayingAc3FileByDma",
	(void *)SpdifAgingTest,					"SpdifAgingTest",
//	(void *)Make24BitTo32BitForTest,		"Make24BitTo32BitForTest",
	(void *)TestInterruptByCpu,				"TestInterruptByCpu",	
	(void *)TestInterruptByDma,				"TestInterruptByDma",		
	0,0
};

void Test_SPDIF(void)
{
	int i;
	
	while(1)
	{
	
		i=0;
		
		/// 1. GPIO Setting for SPDIF 
		SPDIF_SetGpio(1);
		
		/// 2. Select SPDIF Main clock 
		eUsedAudioMainClk = INTERNAL_CLK;		
		eUsedAudioFs = FS_512;

		if (eUsedAudioMainClk == EXTERNAL_CLK)
			UART_Printf("\nUsed External Clock!");
		else if (eUsedAudioMainClk == INTERNAL_CLK)
			UART_Printf("\nUsed Internal Clock!");	

		/// 3. TZPC NonSecure Init
		TZPC_NonSecureInit();

		UART_Printf(" Befoe excute SPDIF Playign test,\n");
		UART_Printf("Upload the sound file of playing at the Memory(addr: 0x%08x).\n", WAVE_BUF_START);

			
		UART_Printf("\n\n================== SPDIF Function Test =====================\n\n");
		
		while(1)
		{   //UART_Printflay menu
			UART_Printf("%2d:%s",i,func_spdif_test[i][1]);
			i++;
			if((int)(func_spdif_test[i][0])==0)
			{
				UART_Printf("\n");
				break;
			}
			if((i%1)==0)
			UART_Printf("\n");
		}
		UART_Printf("\n==========================================================\n");
		UART_Printf("Select #Item or Press Enter key to exit : ");
		i = UART_GetIntNum(); 
		//if(i==-1) break;		// return.
		if(i==0)break;
		if(i>=0 && (i<((sizeof(func_spdif_test)-1)/8)) )	// select and execute...
			( (void (*)(void)) (func_spdif_test[i][0]) )();

	}	
	
}

