/*******************************************************************************

	File Name: spdif.c
	Description: S5PV210 SPDIF Controller Function Test Code

   	Version:  
   	History:
             R0.0(2009): HakSong.Kim First draft

        Memo:
                
********************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "def.h"
#include "option.h"
#include "util.h"
#include "v210_sfr.h"
#include "system.h"
#include "intc.h"
#include "dma.h"
#include "gpio.h"

#include "sysc.h"
#include "spdif.h"

#define DBG_SPDIF

#ifndef DBG_SPDIF
#define spdDbg(x)    0
#else
#define spdDbg(x) Dbg x
#endif

#ifndef DISP_SPDIF_REG
#define spdOutp32(a, d) Outp32(a, d)
#define spdInp32(a, d)  Inp32(a, d)
#define spdOutp16(a, d) Outp16(a, d)
#define spdInp16(a, d)  Inp16(a, d)
#define spdOutp8(a, d) Outp8(a, d)
#define spdInp8(a, d)  Inp8(a, d)
#define spdDispVector(x) 0
#else
#define spdOutp32(a, d) (Disp("Outp32(\'h%08x, \'h%08x);\n", a, d), Outp32(a, d))
#define spdInp32(a, d) (Disp("Inp32(\'h%08x, d); ", a), Inp32(a, d), Dbg("\/\/ d=0x%08x\n", d))
#define spdOutp16(a, d) Disp("Outp32(\'h%08x, \'h%08x);\n", a, d), Outp16(a, d)
#define spdInp16(a, d) (Disp("Inp32(\'h%08x, d); ", a), Inp16(a, d), Dbg("\/\/ d=0x%08x\n", d))
#define spdOutp8(a, d) Disp("Outp32(\'h%08x, \'h%08x);\n", a, d), Outp8(a, d)
#define spdInp8(a, d) (Disp("Inp32(\'h%08x, d); ", a), Inp8(a, d), Dbg("\/\/ d=0x%08x\n", d))
#define spdDispVector(x) Dbg x
#endif

/******************************************************************************
*					SPDIF Register									
 ******************************************************************************/
enum SPDIF_SFR
{
	SPDCLKCON	=	SPDIF_BASE + 0x00,		// Clock control register
	SPDCON		=	SPDIF_BASE + 0x04,		// Control register
	SPDBSTAS	=	SPDIF_BASE + 0x08,		// Burst status register
	SPDCSTAS	=	SPDIF_BASE + 0x0C,		// Channel status register
	SPDDAT		=	SPDIF_BASE + 0x10,		// SPDIFOUT data register
	SPDCNT		=	SPDIF_BASE + 0x14,		// Repetition count register
	SPDBSTAS_SHD	=	SPDIF_BASE + 0x18,		// Shadowed Burst Status Register 
	SPDCNT_SHD		=	SPDIF_BASE + 0x1C,		// Shadowed Repetition count register
	USERBIT1	=	SPDIF_BASE + 0x20,		// Subcode Q1 ~ Q32
	USERBIT2	=	SPDIF_BASE + 0x24,		// Subcode Q33 ~ Q64
	USERBIT3	=	SPDIF_BASE + 0x28,		// Subcode Q65 ~ Q96
	USERBIT1_SHD	=	SPDIF_BASE + 0x2C,		// Shadowed Register Userbit1
	USERBIT2_SHD	=	SPDIF_BASE + 0x30,		// Shadowed Register Userbit2
	USERBIT3_SHD	=	SPDIF_BASE + 0x34,		// Shadowed Register Userbit3
	VERSION_INFO	=	SPDIF_BASE + 0x38,		// RTL version information	
};

// clock controller Reg Definition
#define rEPLL_LOCK		0xE0100010
#define rEPLL_CON		0xE0100110
#define rCLK_SRC0		0xE0100200
#define rCLK_SRC6		0xE0100218
#define rCLK_SRC_MASK0	0xE0100280
#define rCLK_GATE_IP3			0xE010046C
#define rCLK_DIV6		0xE0100318

// Global value
volatile DMAC	oSPDIFDma0;
u32 uMain_Clk_Freq;
volatile u16 *gP16PcmAddr;
extern CLK_SEL eUsedAudioMainClk;

/******************************************************************************
*					SPDIF I/O Functions									
 ******************************************************************************/
void SPDIF_InitIpForPcm(CODEC_TYPE eData, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample)
{
	u32 uEndianFormat, uRequestMainAudioClk, uSamplingFreq,uBitsPerSpl; 

	/// 1. Data type Select
	if(eData == AC3)
	uEndianFormat = 3;		
	else if(eData == PCM)
	uEndianFormat = 0;
	else 
	uEndianFormat = 0;

	/// 2. Main Audio Clcok Freq Select
	
	/// 3. SampleRate Select	
	if(uSampleRate == 44100)
	uSamplingFreq = 0x0;
	else if(uSampleRate == 48000)
	uSamplingFreq = 0x2;
	else if(uSampleRate == 32000)
	uSamplingFreq = 0x3;
	else if(uSampleRate == 96000)
	uSamplingFreq = 0xa;
	else
	Assert(0);

	/// 4. Bit per Sample Select (16bit,20bit,24bit ) 
	if(uBitsPerSample==16)
	uBitsPerSpl = 0;	
	else if(uBitsPerSample==20)
	uBitsPerSpl = 1;	
	else if(uBitsPerSample==24)
	uBitsPerSpl = 2;	
	else
	Assert(0);		

	/// 5. SPDIF Register Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		spdOutp32(SPDCON, uEndianFormat<<13|0<<12|1<<11|0<<10|1<<9|1<<8|1<<7|1<<6|0<<5|uSamplingFreq<<3|uBitsPerSpl<<1|1<<0);
	else if (eUsedAudioMainClk == INTERNAL_CLK)	
		spdOutp32(SPDCON, uEndianFormat<<13|0<<12|1<<11|0<<10|1<<9|1<<8|1<<7|1<<6|0<<5|uMain_Clk_Freq<<3|uBitsPerSpl<<1|1<<0);
	
	//SPDIFOUT Burst Status Register
	// ES_size<<16, AC3<<0
	spdOutp32(SPDBSTAS, (768*16)<<16|0<<13|0<<8|0<<7|1<<0);		
	
	// SPDIFOUT Channel Status Register
	// Level3<<28, 44.1Khz<<24, CD<<8, Mode_0<<6, default<<3, No_cop<<2, Lin_PCM<<1, Con_format<<0
	#if 1  //Consumer format
	spdOutp32(SPDCSTAS, 1<<28|uSamplingFreq<<24|0<<20|0<<16|1<<8|0<<6|0<<3|1<<2|0<<1|0<<0);		
	#else // Professional format
	spdOutp32(SPDCSTAS, 1<<28|uSamplingFreq<<24|0<<20|0<<16|1<<8|0<<6|0<<3|1<<2|0<<1|1<<0);	
	#endif 

	// SPDIFOUT Repitition Count register
	spdOutp32(SPDCNT, 1536*2);		// Repitition count according to data type. This bit is valid only for stream data.
	
	// User Data Register
	spdOutp32(USERBIT1, 0x11111111);		// USERBIT1: Q1 ~ Q32
	spdOutp32(USERBIT2, 0x22222222);		// USERBIT2: Q33 ~ Q64
	spdOutp32(USERBIT3, 0x33333333);		// USERBIT3: Q65 ~ Q96
}

void SPDIF_InitIpForAc3(u32 uAc3DataSize, CODEC_TYPE eData, u32 uSampleRate, u32 uBitRate, u32 uFrameSizeCode)
{
	u32 uEndianFormat, uRealMainAudioClk, uRequestMainAudioClk, uMainAudioClkFrq, uSamplingFreq; 
	u32 i;

	/// 1. Data type Select
	if(eData == AC3)
	uEndianFormat = 3;		
	else if(eData == PCM)
	uEndianFormat = 0;
	else 
	uEndianFormat = 0;

	/// 2. Main Audio Clcok Freq Select
	uRequestMainAudioClk = uSampleRate*uBitRate;
		
	/// 3. SampleRate Select	
	if(uSampleRate == 44100)
	uSamplingFreq = 0x0;
	else if(uSampleRate == 48000)
	uSamplingFreq = 0x2;
	else if(uSampleRate == 32000)
	uSamplingFreq = 0x3;
	else if(uSampleRate == 96000)
	uSamplingFreq = 0xa;
	else
	Assert(0);

	/// 4. BitRate Select 
	if(uBitRate == 256)
	uMainAudioClkFrq = 0x0;
	else if(uBitRate == 384)
	uMainAudioClkFrq = 0x1;
	else if(uBitRate == 512)
	uMainAudioClkFrq = 0x2;
	else
	Assert(0);

	// SPDIF Control Register
	spdOutp32(SPDCON, uEndianFormat<<13|0<<12|1<<11|0<<10|1<<9|0<<8|1<<7|0<<6|0<<5|uMainAudioClkFrq<<3|0<<1|0<<0);
	
	//SPDIFOUT Burst Status Register
	// ES_size<<16, AC3<<0
	spdOutp32(SPDBSTAS, (uFrameSizeCode*16)<<16|0<<13|0<<8|0<<7|1<<0);		
	
	// SPDIFOUT Channel Status Register
	// Level3<<28, 44.1Khz<<24, CD<<8, Mode_0<<6, default<<3, No_cop<<2, Lin_PCM<<1, Con_format<<0
	spdOutp32(SPDCSTAS, 1<<28|uSamplingFreq<<24|0<<20|0<<16|1<<8|0<<6|0<<3|1<<2|1<<1|0<<0);		
	
	// SPDIFOUT Repitition Count register
	spdOutp32(SPDCNT, 1536*2);		// Repitition count according to data type. This bit is valid only for stream data.
}

void SPDIF_StartPlayingPcmByCpu(u32 uCurPcmAddr, u32 uPcmDataSize, CODEC_TYPE eData, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample)
{
	volatile u32 uRead;
	u32 wavdatasize;
	volatile u32 uSfr;
	u32 *pPcmAddr;
	u16 *p16PcmAddr;
	u32 uConReg;

	/// 1. SPDIF Software Reset
	spdOutp32(SPDCON, 1<<5);		

	/// 2. SPDIF(PCM) register Initialize
	SPDIF_TurnOnSpdifoutClock();				// SPDIF Power On	

	if (eUsedAudioMainClk == EXTERNAL_CLK)
		SPDIF_SelectMainClock(EXTERNAL_CLK);		// SPDIF Main Audio clock select	for FPGA
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SelectMainClock(INTERNAL_CLK);				

	SPDIF_InitIpForPcm(eData, uChannelType, uSampleRate, uBitsPerSample);	// SPDIF Reg Initialize 

	uRead = Inp32(SPDCON);
	uRead = (uRead &~((7<<19)|(3<<17))) |((7<<19)|(1<<17)); 	// 19[FIFO Level], 17[Polling mode]
	spdOutp32(SPDCON, uRead);		// Polling transfer mode

	/// 3. Wave file playing start by CPU 
	pPcmAddr = (u32*)uCurPcmAddr;
	p16PcmAddr = (u16*)uCurPcmAddr;

	if(uBitsPerSample == 24)
	{
		while (uPcmDataSize > 0)
		{
			spdOutp32(SPDDAT, *pPcmAddr++);
			uPcmDataSize -= 4;				
		
			do {
				uSfr = Inp32(SPDCON);
			} while (uSfr&(1<<26));			// FIFO 	
		
		}
	}	
	else if (uBitsPerSample == 16)
	{
		spdDbg(("\nSPDIF PCM play polling!\n"));
	
		while (uPcmDataSize > 0)
		{
			spdOutp32(SPDDAT, *p16PcmAddr++);
			uPcmDataSize -= 2;				
		
			do {
				uSfr = Inp32(SPDCON);
			} while (uSfr&(1<<26));			// FIFO 			
		}
		
	}	
	else
		Assert(0);

	/// 4. SPDIF Clock Power off
	SPDIF_TurnOffSpdifoutClock();
}

void SPDIF_StartPlayingPcmByDma(u32 uCurPcmAddr, u32 uPcmDataSize, CODEC_TYPE eData, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample)
{
	// uCurPcmAddr : PCM file start address.
	// uPcmDataSize : PCM file Size(byte)
	
	u32 wavdatasize;
	u32 dma_num;
	DATA_SIZE eDataSize;
	BURST_MODE eBurstMode;
	volatile u32 uRead;

	spdDbg(("\nSPDIF PCM play DMA!\n"));

	/// 1. Wave file size calculation
	if(uBitsPerSample == 24)
	{
		eDataSize = WORD;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 16)
	{
		eDataSize = HWORD;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 8)
	{
		eDataSize = BYTE;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else
		Assert(0);

	/// 2. SPDIF Software Reset
	spdOutp32(SPDCON, 1<<5);		
	
	/// 3. SPDIF(PCM) register Initialize
	SPDIF_TurnOnSpdifoutClock();				// SPDIF Power On
	
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		SPDIF_SelectMainClock(EXTERNAL_CLK);		// SPDIF Main Audio clock select	for FPGA
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SelectMainClock(INTERNAL_CLK);				

	SPDIF_InitIpForPcm(eData, uChannelType, uSampleRate, uBitsPerSample);	// SPDIF Reg Initialize 

	uRead = Inp32(SPDCON);
	uRead = (uRead &~((7<<19)|(3<<17))) |((7<<19)|(0<<17)); 	// DMA transfer mode 
	spdOutp32(SPDCON, uRead);		

	/// 3. Wave file playing start by DMA
	DMA_SetCh(DMA_00, &oSPDIFDma0);//ok. peri 0
	DMA_InitCh(eDataSize , SPDIF, DMA_M2P, eBurstMode, &oSPDIFDma0);
	DMA_StartCh(uCurPcmAddr, SPDDAT, wavdatasize, &oSPDIFDma0);
	while(!DMA_IsTransferDone(&oSPDIFDma0));

	/// 4. SPDIF Clock Power off
	SPDIF_TurnOffSpdifoutClock();
	DMA_StopCh(&oSPDIFDma0);
}

void SPDIF_StartPlayingPcmByDmaForInterrupt(u32 uCurPcmAddr, u32 uPcmDataSize, CODEC_TYPE eData, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample)
{
	// uCurPcmAddr : PCM file start address.
	// uPcmDataSize : PCM file Size(byte)
	
	u32 wavdatasize;
	u32 dma_num;
	DATA_SIZE eDataSize;
	BURST_MODE eBurstMode;
	volatile u32 uRead;

	spdDbg(("\nSPDIF PCM play DMA!\n"));

	/// 1. Wave file size calculation
	if(uBitsPerSample == 24)
	{
		eDataSize = WORD;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 16)
	{
		eDataSize = HWORD;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 8)
	{
		eDataSize = BYTE;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else
		Assert(0);

	/// 2. SPDIF Software Reset
	spdOutp32(SPDCON, 1<<5);		
	
	/// 3. SPDIF(PCM) register Initialize
	SPDIF_TurnOnSpdifoutClock();				// SPDIF Power On
	
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		SPDIF_SelectMainClock(EXTERNAL_CLK);		// SPDIF Main Audio clock select	for FPGA
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SelectMainClock(INTERNAL_CLK);				

	SPDIF_InitIpForPcm(eData, uChannelType, uSampleRate, uBitsPerSample);	// SPDIF Reg Initialize 

	uRead = Inp32(SPDCON);
	uRead = (uRead &~((7<<19)|(3<<17))) |((7<<19)|(0<<17)); 	// DMA transfer mode 
	spdOutp32(SPDCON, uRead);		

	/// 3. Wave file playing start by DMA
	DMA_SetCh(DMA_00, &oSPDIFDma0);//ok. peri 0
	DMA_InitCh(eDataSize , SPDIF, DMA_M2P, eBurstMode, &oSPDIFDma0);
	DMA_StartCh(uCurPcmAddr, SPDDAT, wavdatasize, &oSPDIFDma0);
}

void SPDIF_TestInterruptByCpu(u32 uCurPcmAddr, u32 uPcmDataSize, CODEC_TYPE eData, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample)
{
	volatile u32 uRead;
	u32 wavdatasize;
	volatile u32 uSfr;
	u32 *pPcmAddr;
	u16 *p16PcmAddr;
	u32 uConReg;

	/// 1. SPDIF Software Reset
	spdOutp32(SPDCON, 1<<5);		

	/// 2. SPDIF(PCM) register Initialize
	SPDIF_TurnOnSpdifoutClock();				// SPDIF Power On	

	if (eUsedAudioMainClk == EXTERNAL_CLK)
		SPDIF_SelectMainClock(EXTERNAL_CLK);		// SPDIF Main Audio clock select	for FPGA
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SelectMainClock(INTERNAL_CLK);				

	SPDIF_InitIpForPcm(eData, uChannelType, uSampleRate, uBitsPerSample);	// SPDIF Reg Initialize 

	/// 3. Interrupt Enable
	uRead = Inp32(SPDCON);
	uRead = (uRead &~((7<<19)|(3<<17)|(1<<15)|(1<<10)|(1<<8)|(1<<6))) |((2<<19)|(1<<17)|(0<<15)|(1<<10)|(1<<8)|(1<<6)); 	
	spdOutp32(SPDCON, uRead);		

	/// 4. Wave file playing start by CPU 
	pPcmAddr = (u32*)uCurPcmAddr;
	p16PcmAddr = (u16*)uCurPcmAddr;

	if(uBitsPerSample == 24)
	{
		while (uPcmDataSize > 0)
		{
			spdOutp32(SPDDAT, *pPcmAddr++);
			uPcmDataSize -= 4;				
		
			do {
				uSfr = Inp32(SPDCON);
			} while (uSfr&(1<<26));			// FIFO 	
		
		}
	}	
	else if (uBitsPerSample == 16)
	{
		spdDbg(("\nSPDIF PCM play polling!\n"));
	
		while (uPcmDataSize > 0)
		{
			spdOutp32(SPDDAT, *p16PcmAddr++);
			uPcmDataSize -= 2;				
		
			do {
				uSfr = Inp32(SPDCON);
			} while (uSfr&(1<<26));			// FIFO 			
		}
		
	}	
	else
		Assert(0);

	/// 5. Interrupt Disable
	uRead = Inp32(SPDCON);
	uRead = (uRead &~((7<<19)|(3<<17)|(1<<15)|(1<<10)|(1<<8)|(1<<6))) |((7<<19)|(1<<17)|(0<<15)|(0<<10)|(0<<8)|(0<<6)); 	
	spdOutp32(SPDCON, uRead);		

	/// 6. SPDIF Clock Power off
	SPDIF_TurnOffSpdifoutClock();
}

void SPDIF_TestInterruptByDma(u32 uCurPcmAddr, u32 uPcmDataSize, CODEC_TYPE eData, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample)
{
	// uCurPcmAddr : PCM file start address.
	// uPcmDataSize : PCM file Size(byte)
	
	u32 wavdatasize;
	u32 dma_num;
	DATA_SIZE eDataSize;
	BURST_MODE eBurstMode;	
	volatile u32 uRead;

	spdDbg(("\nSPDIF PCM play DMA!\n"));


	/// 1. Wave file size calculation
	if(uBitsPerSample == 24)
	{
		eDataSize = WORD;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 16)
	{
		eDataSize = HWORD;
		eBurstMode = SINGLE;
			wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 8)
	{
		eDataSize = BYTE;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else
		Assert(0);
	/// 2. SPDIF Software Reset
	spdOutp32(SPDCON, 1<<5);		
	
	/// 3. SPDIF(PCM) register Initialize
	SPDIF_TurnOnSpdifoutClock();				// SPDIF Power On
	
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		SPDIF_SelectMainClock(EXTERNAL_CLK);		// SPDIF Main Audio clock select	for FPGA
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SelectMainClock(INTERNAL_CLK);				

	SPDIF_InitIpForPcm(eData, uChannelType, uSampleRate, uBitsPerSample);	// SPDIF Reg Initialize 

	uRead = Inp32(SPDCON);
	uRead = (uRead &~((7<<19)|(3<<17))) |((7<<19)|(0<<17)); 	// DMA transfer mode 
	spdOutp32(SPDCON, uRead);		

	/// 4. Interrupt Enable
	uRead = Inp32(SPDCON);
	uRead = (uRead &~((1<<15)|(1<<10)|(1<<8)|(1<<6))) |((0<<15)|(0<<10)|(1<<8)|(1<<6)); 	
	spdOutp32(SPDCON, uRead);		

	/// 5. Wave file playing start by DMA
	DMA_SetCh(DMA_00, &oSPDIFDma0);//ok. peri 0
	DMA_InitCh(HWORD , SPDIF, DMA_M2P, SINGLE, &oSPDIFDma0);
	DMA_StartCh(uCurPcmAddr, SPDDAT, wavdatasize, &oSPDIFDma0);
	while(!DMA_IsTransferDone(&oSPDIFDma0));

	/// 6. Interrupt Disable
	uRead = Inp32(SPDCON);
	uRead = (uRead &~((7<<19)|(3<<17)|(1<<15)|(1<<10)|(1<<8)|(1<<6))) |((7<<19)|(1<<17)|(0<<15)|(0<<10)|(0<<8)|(0<<6)); 	
	spdOutp32(SPDCON, uRead);		
	
	/// 7. SPDIF Clock Power off
	SPDIF_TurnOffSpdifoutClock();

	DMA_StopCh(&oSPDIFDma0);
}

void SPDIF_StartPlayingAc3ByCpu(u32 uCurAc3Addr, u32 uAc3DataSize, CODEC_TYPE eData, u32 uSampleRate, u32 uBitRate, u32 uFrameSizeCode)
{
	volatile u32 uRead;
	u32 wavdatasize;
	volatile u32 uSfr;
	u32 *pPcmAddr;
	u16 *p16PcmAddr;
	u32 uConReg;
	u32 ac3datasize;
	u32 uBitsPerSample;

	/// 1. SPDIF Reg Setting
	SPDIF_TurnOnSpdifoutClock();	

	if (eUsedAudioMainClk == EXTERNAL_CLK)
		SPDIF_SelectMainClock(EXTERNAL_CLK);		// SPDIF Main Audio clock select	for FPGA
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SelectMainClock(INTERNAL_CLK);				
	
	SPDIF_InitIpForAc3(uAc3DataSize, eData, uSampleRate, uBitRate, uFrameSizeCode);	

	uRead = Inp32(SPDCON);
	uRead = (uRead &~((7<<19)|(3<<17))) |((7<<19)|(1<<17)); 	// 19[FIFO Level], 17[Polling mode]
	spdOutp32(SPDCON, uRead);		// Polling transfer mode

	/// 2. Wave file playing start by CPU 
	pPcmAddr = (u32*)uCurAc3Addr;
	p16PcmAddr = (u16*)uCurAc3Addr;
	uBitsPerSample = 16;
	if(uBitsPerSample == 24)
	{
		while (uAc3DataSize > 0)
		{
			spdOutp32(SPDDAT, *pPcmAddr++);
			uAc3DataSize -= 4;				
		
			do {
				uSfr = Inp32(SPDCON);
			} while (uSfr&(1<<26));			// FIFO 	
		
		}
	}	
	else if (uBitsPerSample == 16)
	{
		spdDbg(("\nSPDIF PCM play polling!\n"));
	
		while (uAc3DataSize > 0)
		{
			spdOutp32(SPDDAT, *p16PcmAddr++);
			uAc3DataSize -= 2;				
		
			do {
				uSfr = Inp32(SPDCON);
			} while (uSfr&(1<<26));			// FIFO 			
		}
		
	}	
	else
		Assert(0);

	/// 3. SPDIF Clock Power off
	SPDIF_TurnOffSpdifoutClock();
}

void SPDIF_StartPlayingAc3ByDma(u32 uCurAc3Addr, u32 uAc3DataSize, CODEC_TYPE eData, u32 uSampleRate, u32 uBitRate, u32 uFrameSizeCode)
{
	u32 ac3datasize;

	/// 1. Wave file size calculation
	ac3datasize = uAc3DataSize;
	
	/// 2. SPDIF Reg Setting
	SPDIF_TurnOnSpdifoutClock();	

	if (eUsedAudioMainClk == EXTERNAL_CLK)
		SPDIF_SelectMainClock(EXTERNAL_CLK);		// SPDIF Main Audio clock select	for FPGA
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SelectMainClock(INTERNAL_CLK);				
	
	SPDIF_InitIpForAc3(uAc3DataSize, eData, uSampleRate, uBitRate, uFrameSizeCode);	

	/// 3. DMA Setting
	DMA_SetCh(DMA_00, &oSPDIFDma0);//ok. peri 0
	DMA_InitCh(HWORD , SPDIF, DMA_M2P, SINGLE, &oSPDIFDma0);
	DMA_StartCh(uCurAc3Addr, SPDDAT, ac3datasize, &oSPDIFDma0);
	while(!DMA_IsTransferDone(&oSPDIFDma0));

	/// 4. SPDIF Clock Power off
	SPDIF_TurnOffSpdifoutClock();
	DMA_StopCh(&oSPDIFDma0);
}

void SPDIF_StartPlayingAc3ByDmaForInterrupt(u32 uCurAc3Addr, u32 uAc3DataSize, CODEC_TYPE eData, u32 uSampleRate, u32 uBitRate, u32 uFrameSizeCode)
{
	u32 ac3datasize;

	/// 1. Wave file size calculation
	ac3datasize = uAc3DataSize;
	
	/// 2. SPDIF Reg Setting
	SPDIF_TurnOnSpdifoutClock();	

	if (eUsedAudioMainClk == EXTERNAL_CLK)
		SPDIF_SelectMainClock(EXTERNAL_CLK);		// SPDIF Main Audio clock select	for FPGA
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SelectMainClock(INTERNAL_CLK);				
	
	SPDIF_InitIpForAc3(uAc3DataSize, eData, uSampleRate, uBitRate, uFrameSizeCode);	

	/// 3. DMA Setting
	DMA_SetCh(DMA_00, &oSPDIFDma0);//ok. peri 0
	DMA_InitCh(HWORD , SPDIF, DMA_M2P, SINGLE, &oSPDIFDma0);
	DMA_StartCh(uCurAc3Addr, SPDDAT, ac3datasize, &oSPDIFDma0);
}

void SPDIF_TurnOnSpdifoutClock(void)
{
	volatile u32 uRead;

	uRead = Inp32(SPDCLKCON);
	uRead = (uRead &~(1<<0)) | (1<<0); 
	spdOutp32(SPDCLKCON, uRead);		// SPDIFOUT clcok power on
}

void SPDIF_TurnOffSpdifoutClock(void)
{
	volatile u32 uRead;

	uRead = Inp32(SPDCLKCON);
	uRead = (uRead &~(1<<0)) | (0<<0); 
	spdOutp32(SPDCLKCON, uRead);		// SPDIFOUT clcok power off
}

void SPDIF_SelectMainClock(CLK_SEL eSpdifMainClock)
{
	volatile u32 uRead;

	uRead = Inp32(SPDCLKCON);
	uRead = (uRead &~(1<<2)) | (eSpdifMainClock<<2); 
	spdOutp32(SPDCLKCON, uRead);		
}

void SPDIF_IntHandler(void)
{
	volatile u32 uRead;
	u32 temp,temp1,temp2,temp3;

	temp = Inp32(SPDCON);		// SPDCON register Read

	// Check user data interrupt status
	if(temp & 1<<11)
	{
		spdDbg(("\nUser Data Interrupt!\n"));
		temp1 = Inp32(USERBIT1_SHD);
		temp2 = Inp32(USERBIT2_SHD);
		temp3 = Inp32(USERBIT3_SHD);
		spdDbg(("Shadowed User data = %x, %x, %x\n",temp1,temp2,temp3));
		temp1 = Inp32(USERBIT1);
		temp2 = Inp32(USERBIT2);
		temp3 = Inp32(USERBIT3);		
		spdOutp32(USERBIT1, temp1+1);
		spdOutp32(USERBIT2, temp2+1);
		spdOutp32(USERBIT3, temp3+1);
		temp |= 1<<11;
	}

	// Check Buffer Empty Interrupt Status
	if(temp & 1<<9)
	{
		spdDbg(("\nBuffer Empty Interrupt!\n"));		
		uRead = Inp32(SPDCLKCON);
		uRead = (uRead &~(1<<0)) | (0<<0); 
		spdOutp32(SPDCLKCON, uRead);		// SPDIFOUT clcok power off
		temp |= 1<<9;
	}

	// Check Stream End Interrupt Status
	if(temp & 1<<7)
	{
		spdDbg(("\nStream End Interrupt!!\n"));
		temp |= 1<<7;
	}

	spdOutp32(SPDCON, temp);		// SPDCON register write
}

u32 SPDIF_IsFifoFull(void)
{
	volatile u32 uSfr;
	uSfr = Inp32(SPDCON);
	return (uSfr&(0x10<<22));
}

void SPDIF_SetGpio(u32 uPort)
{
	Outp32(0xe0200080, 0x33);		// GPC1.[0] => SPDIF_0_OUT, GPC1.[1] => SPDIF_EXTCLK
	//Outp32(0xe020008c, 0x33);		// GPIO Drive stregth
}

u32 SPDIF_ReadIntFlag(void)
{
	volatile u32 uRead;

	uRead = Inp32(SPDCON);

	return uRead; 
}

void SPDIF_ClearIntFlag(spdif_int_flag eSpdifIntFlag)
{
	volatile u32 uRead;

	uRead = Inp32(SPDCON);	
	uRead = (uRead&~(1<<eSpdifIntFlag))|(1<<eSpdifIntFlag);	
	spdOutp32(SPDCON, uRead);	
}

void SPDIF_SetAudioClock(u32 uSampleRate, FS_SEL eFsSel)
{
	volatile u32 uRead;
	u32 uTrg_Fout, uVSEL=0, uP_Value=0, uM_Value=0, uS_Value=0;
	u32 uCal, uInp_Clk, uDiv, uFsVal;

	/// 1. Clock Cal
	if (eFsSel == FS_256)
		uFsVal = 256;
	else if (eFsSel == FS_384)
		uFsVal = 384;
	else if (eFsSel == FS_512)
		uFsVal = 512;
	else 
		Assert(0);
		
	uTrg_Fout = uSampleRate*uFsVal;   
	uMain_Clk_Freq = 	eFsSel;

	spdDbg(("uTrg_Fout: %d\n",uTrg_Fout));

	/// 2. Target FOUT(MHz), EPLL Table
	switch(uTrg_Fout)
	{
		case	 48000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 96;	uS_Value = 3; 
			if(uTrg_Fout== 48000000) uDiv=0; 
			break;	
		case	 96000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 96;	uS_Value = 2; 
			if(uTrg_Fout== 96000000) uDiv=0; 
			break;	
		case	 144000000:	
			uVSEL=1;	uP_Value = 6;	uM_Value = 144;	uS_Value = 2; 
			if(uTrg_Fout== 144000000) uDiv=0; 
			break;	
		case	 192000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 96;	uS_Value = 1; 
			if(uTrg_Fout== 192000000) uDiv=0; 
			break;		
		case	 288000000:	
			uVSEL=1;	uP_Value = 6;	uM_Value = 144;	uS_Value = 1; 
			if(uTrg_Fout== 288000000) uDiv=0; 
			break;		
		case	 84000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 84;	uS_Value = 2; 
			if(uTrg_Fout== 84000000) uDiv=0; 
			break;	
		case	 50000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 100;	uS_Value = 3; 
			if(uTrg_Fout== 50000000) uDiv=0; 
			break;	
		case	 80000000:	
			uVSEL=1;	uP_Value = 6;	uM_Value = 160;	uS_Value = 3; 
			if(uTrg_Fout== 80000000) uDiv=0; 
			break;
		case 16384000:	
		case 8192000:	
		case	 32768000:	
			uVSEL=1;	uP_Value = 6;	uM_Value = 131;	uS_Value = 4; 
			if (uTrg_Fout== 32768000) uDiv=0; 
			else if (uTrg_Fout== 8192000) uDiv=3; 
			else if (uTrg_Fout== 16384000) uDiv=1; 
			break;
		case 12288000:	
		case 24576000:	
		case 49152000:
			uVSEL=0;	uP_Value = 8;	uM_Value = 131;	uS_Value = 3; 
			if(uTrg_Fout== 49152000) uDiv=0; 
			else if(uTrg_Fout == 24576000) uDiv=1; 
			else if(uTrg_Fout == 12288000) uDiv=3; 			
			break;	
		case 16934400:			
		case 67737600:
			uVSEL=1;	uP_Value = 12;	uM_Value = 271;	uS_Value = 3; 
			if (uTrg_Fout== 67738000) uDiv=0; 
			else if (uTrg_Fout== 16934400) uDiv=3; 
			break;
		case 36864000:
		case 18432000:
		case 73728000:
			uVSEL=1;	uP_Value = 12;	uM_Value = 295;	uS_Value = 3; 
			if (uTrg_Fout== 73728000) uDiv=0;
			else if (uTrg_Fout== 18432000) uDiv=3; 
			else if (uTrg_Fout== 36864000) uDiv=1; 			
			break;	
		case 11289600:	
		case 22579200:	
		case 45158400:
			uVSEL=0;	uP_Value = 18;	uM_Value = 271;	uS_Value = 3; 
			if (uTrg_Fout== 45158400) uDiv=0; 
			else if (uTrg_Fout== 22579200) uDiv=1;
			else if (uTrg_Fout== 11289600) uDiv=3;
			break;	
		default:
			Assert(0); break;
	}

	uRead = Inp32(rCLK_SRC_MASK0);
	uRead = (uRead & ~(1<<27)) | (1<<27);		// Mask output clock of MUXSPDIF enable 
	Outp32(rCLK_SRC_MASK0, uRead);	

	uRead = Inp32(rCLK_GATE_IP3);
	uRead = (uRead & ~((3<<28)|(7<<4)|(1<<0))) | ((3<<28)|(7<<4)|(1<<0));		// CLK_PCM0/PCM1 must pass 
	Outp32(rCLK_GATE_IP3, uRead);	

	///3.  EPLL CLK Settting
	///3.1  CLK_SRC0, FOUT(EPLL)
	uRead = Inp32(rCLK_SRC0);
	uRead = (uRead & ~(1<<8)) | (1<<8);
	Outp32(rCLK_SRC0, uRead);	

	/// 3.2 Mux
	#if 1		// SCLK_AUDIO_0
	uRead = Inp32(rCLK_SRC6);
	uRead = (uRead & ~((3<<12)|(0xf<<0))) | ((0<<12)|(0x7<<0));		// SCLK_AUDIO0, SCLK_EPLL
	Outp32(rCLK_SRC6, uRead);	
	#endif
	#if 0		// SCLK_AUDIO_1
	uRead = Inp32(rCLK_SRC6);
	uRead = (uRead & ~((3<<12)|(0xf<<4))) | ((1<<12)|(0x7<<4));		// SCLK_AUDIO1, SCLK_EPLL
	Outp32(rCLK_SRC6, uRead);	
	#endif
	#if 0		// SCLK_AUDIO_2
	uRead = Inp32(rCLK_SRC6);
	uRead = (uRead & ~((3<<12)|(0xf<<8))) | ((2<<12)|(0x7<<8));		// SCLK_AUDIO2, SCLK_EPLL
	Outp32(rCLK_SRC6, uRead);	
	#endif
	
	/// 3.3 EPLL Lock Time, 300us
	uRead = Inp32(rEPLL_LOCK);
	uRead = (uRead & ~(0xffff<<0)) | (0xe10<<0);
	Outp32(rEPLL_LOCK, uRead);	

	/// 3.4 EPLL control setting
	uRead = Inp32(rEPLL_CON);
	uRead = (uRead & ~((1<<31)|(1<<27)|(0x3ff<<16)|(0x3f<<8)|(7<<0))) | ((1<<31)|(uVSEL<<27)|(uM_Value<<16)|(uP_Value<<8)|(uS_Value<<0));
	Outp32(rEPLL_CON, uRead);	

	while(!(Inp32(rEPLL_CON)>>29)&0x1);		// lock time is done

	/// 3.5 DIV Audio_0 (1~16), If it is 0, it is bypass.
	#if 1		// SCLK_AUDIO_0
	uRead = Inp32(rCLK_DIV6);
	uRead = (uRead & ~(0xf<<0)) | (uDiv<<0);
	Outp32(rCLK_DIV6, uRead);	
	#endif
	#if 0		// SCLK_AUDIO_1
	uRead = Inp32(rCLK_DIV6);
	uRead = (uRead & ~(0xf<<4)) | (uDiv<<4);
	Outp32(rCLK_DIV6, uRead);	
	#endif
	#if 0		// SCLK_AUDIO_2
	uRead = Inp32(rCLK_DIV6);
	uRead = (uRead & ~(0xf<<8)) | (uDiv<<8);
	Outp32(rCLK_DIV6, uRead);	
	#endif 		
}

