/**************************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2006 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : sdhc.c
*	File Description : This file implements the API functon for High Speed MMC.
*	Author : Youngmin.Kim
*	Dept. : AP Development Team
*	Created Date : 08.OCT.2005
*	Version : 0.2 
* 
*	History
*	 1) 1st Made
*      2) add SDIO, SDHC, CE-ATA interface and code compaction by youngbo.song
*  
**************************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "def.h"
#include "option.h"
#include "library.h"

#include "util.h"
#include "system.h"
#include "intc.h"
#include "gpio.h"
#include "sdhc.h"
#include "sysc.h"


#undef FEATURE_SDHC_PREDEFINED
#undef FEATURE_SDHC_INTERRUPT_CONTROL_MODE
#define FEATURE_SDHC_HIGH_CAPACITY_CARD_SUPPORT
#undef FEATURE_SDHC_HIGH_SPEED_SUPPORT

#define PullUp_Enable

// HighSpeed mode separator
#define SDHC_MMC_HIGH_SPEED_CLOCK 20000000
#define SDHC_SD_HIGH_SPEED_CLOCK 25000000

//////////
// File Name : SDHC_SetBlockCountReg (Inline Macro)
// File Description : This function set block count register.
// Input : SDHC, block count 
// Output : NONE.
#define SDHC_SetBlockCountReg( sCh, uBlkCnt) \
	Outp16( (sCh)->m_uBaseAddr + SDHC_BLK_COUNT, (uBlkCnt) );

//////////
// File Name : SDHC_SetSystemAddressReg (Inline Macro)
// File Description : This function set DMA start address.
// Input : SDHC, start address.
// Output : NONE.
#define SDHC_SetSystemAddressReg( sCh, SysAddr) \
	Outp32( (sCh)->m_uBaseAddr + SDHC_SYS_ADDR, (SysAddr) );

//////////
// File Name : SDHC_SetBlockSizeReg (Inline Macro)
// File Description : This function set block size and buffer size.
// Input : SDHC, DMA buffer boundary, One block size.
// Output : NONE.
#define SDHC_SetBlockSizeReg( sCh, uDmaBufBoundary, uBlkSize ) \
	Outp16( (sCh)->m_uBaseAddr + SDHC_BLK_SIZE, (((uDmaBufBoundary)<<12)|(uBlkSize)) );


#ifndef FEATURE_ASYNC_CLOCK_DOMAIN_SUPPORT

#ifdef FEATURE_SDHC_INTERRUPT_CONTROL_MODE
//////////
// File Name : SDHC_INT_WAIT_CLEAR (Inline Macro)
// File Description : Interrupt wait and clear.
// Input : SDHC, interrupt bit, timeout loop count 
// Output : NONE.	// 0x7F000000 youngbo.song
#define SDHC_INT_WAIT_CLEAR(sCh,bit,loop) \
	loop=0xF000000; \
	while ( !((sCh)->m_sNormalStatus & (1<<bit) ) ) { \
		if ( --loop == 0 ) { \
			UART_Printf( "***********Time out Error : bit : %d, Line:%d \n", bit, __LINE__ ); \
			break;	} } \
	(sCh)->m_sNormalStatus &= ~(1<<bit); \
	Outp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT, (1<<bit) ); \
	while( Inp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT ) & (1<<bit) );
#else
//////////
// File Name : SDHC_INT_WAIT_CLEAR (Inline Macro)
// File Description : Interrupt wait and clear.
// Input : SDHC, interrupt bit, timeout loop count 
// Output : NONE.	// 0x7F000000 youngbo.song
#define SDHC_INT_WAIT_CLEAR(sCh,bit,loop) \
	loop=0xF000000; \
	while ( !(Inp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT ) & (1<<bit) ) ) { \
		if ( --loop == 0 ) { \
			UART_Printf( "***********Time out Error : bit : %d, Line:%d \n", bit, __LINE__ ); \
			break;	} } \
	do { Outp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT, (1<<bit) ); \
	} while( Inp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT ) & (1<<bit) );
#endif

#else

#define SDHC_INT_WAIT_CLEAR(sCh,bit,loop) \
	loop=0xF000000; \
	while ( !(Inp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT ) & (1<<bit) ) ) { \
		if ( --loop == 0 ) { \
			UART_Printf( "***********Time out Error : bit : %d, Line:%d \n", bit, __LINE__ ); \
			break;	} } \
	Outp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT, (1<<bit) ); \
	if( Inp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT ) & (1<<bit) ) { \
		UART_Printf("Status do not cleared********************\n"); \
	}

#endif



//////////
// File Name : SDHC_INT_CLEAR (Inline Macro)
// File Description : Interrupt clear.
// Input : SDHC, interrupt bit, timeout loop count 
// Output : NONE.	// 0x7F000000 youngbo.song
#define SDHC_ERROR_INT_CLEAR(sCh,bit) \
	Outp16( (sCh)->m_uBaseAddr + SDHC_ERROR_INT_STAT, (1<<bit) ); \
	while( Inp16( (sCh)->m_uBaseAddr + SDHC_ERROR_INT_STAT ) & (1<<bit) );


//////////
// File Name : SDHC_INT_CLEAR (Inline Macro)
// File Description : Interrupt clear.
// Input : SDHC, interrupt bit, timeout loop count 
// Output : NONE.	// 0x7F000000 youngbo.song
//#define SDHC_NORMAL_INT_CLEAR(sCh,bit) \
//	Outp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT, (1<<bit) ); \
//	while( Inp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT ) & (1<<bit) );



//
// [7:6] Command Type
// [5]  Data Present Select
// [4] Command Index Check Enable
// [3] CRC Check Enable
// [1:0] Response Type Select
const unsigned char SDHC_cmd_sfr_data[] = {
	(unsigned char)((0<<4)|(0<<3)|(0<<0)),	// RES_NO_TYPE
	(unsigned char)((1<<4)|(1<<3)|(2<<0)),	// RES_R1_TYPE,
	(unsigned char)((1<<4)|(1<<3)|(3<<0)),	// RES_R1B_TYPE,
	(unsigned char)((0<<4)|(1<<3)|(1<<0)),	// RES_R2_TYPE,
	(unsigned char)((0<<4)|(0<<3)|(2<<0)),	// RES_R3_TYPE,
	(unsigned char)((0<<4)|(0<<3)|(2<<0)),	// RES_R4_TYPE,
	(unsigned char)((1<<4)|(1<<3)|(2<<0)),	// RES_R5_TYPE,
	(unsigned char)((1<<4)|(1<<3)|(2<<0)),	// RES_R6_TYPE,	// check need.
	(unsigned char)((1<<4)|(1<<3)|(2<<0)),	// RES_R7_TYPE,	// check need.
};

SDHC* SDHC_curr_card[SDHC_CHANNEL_CNT];
unsigned int SDHC_global_card_size;
static u8 SDHC_globl_testBuffer[512];



//////////
// File Name : SDHC_DMAIntHandler
// File Description : This function is interrupt service routine for Common DMA Handling.
// Input : NONE.
// Output : NONE.
void SDHC_DMAIntHandler(SDHC * sCh, u32 intIndex)
{
	u32 i;
	// youngbo.song
//	SDHC_InterruptInspector(sCh);
//	sdDbg(("\nISR rHM_NORINTSTS = %x", Inp16(sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT ) ));
//	sdDbg(("block size :%d \n" , Inp16( sCh->m_uBaseAddr+SDHC_BLK_COUNT ) ));
	if ( Inp16(sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT) & SDHC_BLOCKGAP_EVENT_SIG_INT_EN )
	{
		Disp("\n isr DMAInt0 \n");
		Disp("\n Block Gap Event \n");		
		Inp16(sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT) = SDHC_BLOCKGAP_EVENT_SIG_INT_EN;
	}
	
	if( ( Inp16(sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT) & SDHC_TRANSFERCOMPLETE_SIG_INT_EN ) )
		// When SDIO suspend mode, there is transfercomplete event, but transfer not yet completed.
//		&& ( Inp16( sCh->m_uBaseAddr+SDHC_BLK_COUNT) == 0 ) )
	{
//		sdDbg(("\nTransfer Complete\n"));
//		sdDbg(("T"));
		SDHC_INT_WAIT_CLEAR( sCh, 1, i);	//SDHC_TRANSFERCOMPLETE_STS_INT_EN 
		sCh->m_uRemainBlock=0;

		INTC_Disable(intIndex);
	}
	
	if ( Inp16(sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT) & SDHC_DMA_SIG_INT_EN )
	{
//		sdDbg((" DMA buffer boundary is detected!\n"));
//		sdDbg(("B"));
		SDHC_INT_WAIT_CLEAR( sCh, 3, i);	//SDHC_TRANSFERCOMPLETE_STS_INT_EN
		SDHC_SetSystemAddressReg(sCh, Inp32(sCh->m_uBaseAddr+SDHC_SYS_ADDR) );
	}

	if ( Inp16(sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT) & SDHC_CCS_INTERRUPT_STATUS_EN )
	{
		SDHC_INT_WAIT_CLEAR( sCh, 9, i);	//SDHC_CCS_INTERRUPT_STATUS_EN
		sCh->m_uCCSResponse=0;
//		sdDbg(("SDHC_CCS_INTERRUPT_STATUS_EN----------------\n"));
//		sdDbg(("C"));
	}
}




////////// ////  /////////  ///////  ////////// ///////////
// File Name : SDHC_IssueCommand52
// File Description : This function issue SDIO command.
// Input : SDHC, command Index, Argument, Command Type, Response Type, Command Type
// Output : Command Result.
u32 SDHC_IssueCommand52( SDHC* sCh, u16 uCmd, u32 uArg, SDHC_CommandType cType, SDHC_ResponseType rType, u32 cmdType )
{
	u16 sfrData;
	u32 Loop;
	
	while( Inp32( sCh->m_uBaseAddr+SDHC_PRESENT_STAT ) & 0x1 );	// Check CommandInhibit_CMD

	//while( Inp32( sCh->m_uBaseAddr+SDHC_PRESENT_STAT ) & (1<<1) );// Check CommandInhibit_DAT

	sfrData = (uCmd<<8) | SDHC_cmd_sfr_data[ rType ];
	if ( cType == SDHC_CMD_ADTC_TYPE )
	{
		sfrData |= (1<<5);
	}
	
	// command Type.
	sfrData |= (cmdType<<6);
	
	// argument setting.
	Outp32( sCh->m_uBaseAddr+SDHC_ARG, uArg);
	
	Outp16( sCh->m_uBaseAddr+SDHC_COMMAND, sfrData);

	// Transfer Complete and Command Complete. - SDHC_COMMANDCOMPLETE_STS_INT_EN
	SDHC_INT_WAIT_CLEAR( sCh, 0, Loop);

	// Error Status Check - reduce too much error message.
	if ( (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT ) & (1<<15)) && !(uCmd==1||uCmd==55||uCmd==41) )
	{
		Disp("Command = %d, Error Stat = %x\n", Inp16( sCh->m_uBaseAddr+SDHC_COMMAND )>>8,
			Inp16( sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT ) );
		return FALSE;
	}
	
	return Inp32(sCh->m_uBaseAddr+SDHC_RSP0);;
}




//////////
// File Name : SDHC_ReadBlocksSuspend
// File Description : Suspend / Resume Feature Test function.
// Input : start block, block count, source buffer address, SDHC channel
// Output : Success or Failure
bool SDHC_ReadBlocksSuspend(u32 uStBlock, u16 uBlocks, u32 uBufAddr, SDHC* sCh)
{
	u32 i;
	u32 BackupAddress;
	u16 BackupBlockSize;
	u16 BackupBlockCnt;
	u32 BackupArgument;
	u16 BackupTransMode;
	u32 Writed_Count;
	
	if(sCh->m_eTransMode == SDHC_BYTE_MODE)
	{
		uStBlock = uStBlock<<9;	//	 uStBlock * 512;
	}
	else

	sCh->m_uRemainBlock = uBlocks;	// number of blocks.
	sCh->m_uBufferPtr = (u32*)uBufAddr;

	if ( !SDHC_WaitForCard2TransferState( sCh ) ) //added by HEAD. 2009.04.25
		return FALSE;

	SDHC_SetSystemAddressReg(sCh, uBufAddr);// AHB System Address For Write
	SDHC_SetTransferModeReg(1, 1, 1, 1, 1, sCh ); // Read

	SDHC_SetBlockSizeReg(sCh, 7, 512); // Maximum DMA Buffer Size, Block Size
	SDHC_SetBlockCountReg(sCh, uBlocks); // Block Numbers to Write

	// Blockgap Interrupt Enable
	//Outp8( sCh->m_uBaseAddr+SDHC_BLOCKGAP_CTRL, (1<<3) );

	INTC_SetVectAddr( sCh->m_ucIntChannelNum, sCh->m_fDmaFn );
	INTC_Enable( sCh->m_ucIntChannelNum );
	
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
		SDHC_READWAIT_SIG_INT_EN| //added by HEAD. 2009.04.27
		SDHC_BUFFER_READREADY_SIG_INT_EN|
		SDHC_BUFFER_WRITEREADY_SIG_INT_EN|
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN|
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN|
		SDHC_COMMANDCOMPLETE_SIG_INT_EN );

	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		SDHC_READWAIT_SIG_INT_EN| //added by HEAD. 2009.04.27
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN| //  ּ ó ?
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN);

	// Added by HEAD. 2009.04.25
	Disp("\n\n ______ [ Original Value] _____________ ");
	Disp("\n\t [00] SYS_ADDR    : 0x%X", Inp32(sCh->m_uBaseAddr+SDHC_SYS_ADDR) );
	Disp("\n\t [04] Block Size  : 0x%X", Inp16(sCh->m_uBaseAddr+SDHC_BLK_SIZE) );
	Disp("\n\t [06] Block Cnt   : 0x%X\n", Inp16(sCh->m_uBaseAddr+SDHC_BLK_COUNT) );

	
	// Multi Block Only
	if ( !SDHC_IssueCommand( sCh, 18, uStBlock, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE ))
	{
		return FALSE;
	}

/*------------------------------------------------------------------------------------------
	SD Host Controller Spec.
		3.112 Suspend/Resume
------------------------------------------------------------------------------------------ */

//[1], Set Stop at Block Gap Request
	Outp8( sCh->m_uBaseAddr+SDHC_BLOCKGAP_CTRL, (1<<3)|(1<<0) );

	//SDHC_INT_WAIT_CLEAR( sCh, 2, i );	// waiting Block Gap

	// wait until transfer complete.
	while( sCh->m_uRemainBlock != 0 );

	//	Delay(wait);
	Delay(1000); //added by HEAD. 2009.04.25

/*-------------------- [ Suspend mode ] ------------------------------------------ */	

//[5], Issue Suspend Command
	if ( !SDHC_IssueCommand52( sCh, 12, 0, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE, 1/* suspend */ ) )
	{
		return FALSE;
	}

	// Backup Status Register
	Disp( "status : %x , error status : %x\n" , Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT),
						Inp16( sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT) );
	
	Disp( "DMA Position : %x, SourcePosition:%x, Difference : %x\n", Inp32( sCh->m_uBaseAddr+SDHC_SYS_ADDR ),
		uBufAddr, Inp32( sCh->m_uBaseAddr+SDHC_SYS_ADDR )-uBufAddr );
	
//[7], Save Register (000-00Dh)
	BackupAddress = Inp32( sCh->m_uBaseAddr+SDHC_SYS_ADDR );
	BackupBlockSize = Inp16( sCh->m_uBaseAddr+SDHC_BLK_SIZE);
	BackupBlockCnt = Inp16( sCh->m_uBaseAddr+SDHC_BLK_COUNT);
	BackupArgument = Inp32( sCh->m_uBaseAddr+SDHC_ARG );
	BackupTransMode = Inp16( sCh->m_uBaseAddr+SDHC_TRANS_MODE);
	Writed_Count = uBlocks - BackupBlockCnt;

//[9], Clr Stop at Block Gap Request
	Outp8( sCh->m_uBaseAddr+SDHC_BLOCKGAP_CTRL, 0/*(1<<3)*/ );

	// Added by HEAD. 2009.04.25
	Disp("\n\n ______ [ Saved Value] _____________ ");
	Disp("\n\t [00] SYS_ADDR    : 0x%X", BackupAddress);
	Disp("\n\t [04] Block Size  : 0x%X", BackupBlockSize);
	Disp("\n\t [06] Block Cnt   : 0x%X\n", BackupBlockCnt);
	//Disp("\n\t [00] SYS_ADDR    : 0x%X", Inp32(sCh->m_uBaseAddr+SDHC_SYS_ADDR) );
	//Disp("\n\t [04] Block Size  : 0x%X", Inp16(sCh->m_uBaseAddr+SDHC_BLK_SIZE) );
	//Disp("\n\t [06] Block Cnt   : 0x%X\n", Inp16(sCh->m_uBaseAddr+SDHC_BLK_COUNT) );



	Delay(1000); //added by HEAD. 2009.04.25


/*-------------------- [ Resume mode ] ------------------------------------------ */

	sCh->m_uRemainBlock = BackupBlockCnt;

	if(sCh->m_eTransMode == SDHC_BYTE_MODE)
	{
		uStBlock = uStBlock<<9;	//	 uStBlock * 512;
	}
	else

	if ( !SDHC_WaitForCard2TransferState( sCh ) ) //added by HEAD. 2009.04.25
		return FALSE;

	SDHC_SetSystemAddressReg(sCh, (u32)BackupAddress);// AHB System Address For Write

	//SDHC_SetTransferModeReg( 1, 0, 1, 1, (sCh->m_eOpMode!=SDHC_DMA_MODE)?(0):(1), sCh ); //original code
	SDHC_SetTransferModeReg( 1, 1, 1, 1, 1, sCh );  // Read

	SDHC_SetBlockSizeReg(sCh, 7, 512); // Maximum DMA Buffer Size, Block Size
	SDHC_SetBlockCountReg(sCh, BackupBlockCnt); // Block Numbers to Write

	INTC_SetVectAddr( sCh->m_ucIntChannelNum, sCh->m_fDmaFn );
	INTC_Enable( sCh->m_ucIntChannelNum );

	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
		SDHC_READWAIT_SIG_INT_EN| //added by HEAD. 2009.04.27
		SDHC_BUFFER_READREADY_SIG_INT_EN|
		SDHC_BUFFER_WRITEREADY_SIG_INT_EN|
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN|
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN|
		SDHC_COMMANDCOMPLETE_SIG_INT_EN );
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		SDHC_READWAIT_SIG_INT_EN| //added by HEAD. 2009.04.27
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN| //  ּ ó ?
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN);

	Delay(1000);
	
// [1], Restore Register(000-00Dh)
	Outp32( sCh->m_uBaseAddr+SDHC_SYS_ADDR, BackupAddress);
	Outp16( sCh->m_uBaseAddr+SDHC_BLK_SIZE, BackupBlockSize);
	Outp16( sCh->m_uBaseAddr+SDHC_BLK_COUNT, BackupBlockCnt);
	Outp32( sCh->m_uBaseAddr+SDHC_ARG, BackupArgument );
	Outp16( sCh->m_uBaseAddr+SDHC_TRANS_MODE, BackupTransMode);

// [2], Issue Resume Command	
	if ( !SDHC_IssueCommand52( sCh, 25, uStBlock+Writed_Count, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE, 2/* Resume */ ) )
	{
		return FALSE;
	}

// [3] Check Data Flag ???
	while( sCh->m_uRemainBlock != 0 );

// [4] Set Software Reset for DAT line
	Outp8( sCh->m_uBaseAddr+ SDHC_SOFTWARE_RESET,  0x4);
	Disp("\n Software Reset For DAT Line \n");

	// Added by HEAD. 2009.04.25
	Disp("\n\n ______ [ Final Value] _____________ ");
	Disp("\n\t [00] SYS_ADDR    : 0x%X", Inp32(sCh->m_uBaseAddr+SDHC_SYS_ADDR) );
	Disp("\n\t [04] Block Size  : 0x%X", Inp16(sCh->m_uBaseAddr+SDHC_BLK_SIZE) );
	Disp("\n\t [06] Block Cnt   : 0x%X\n", Inp16(sCh->m_uBaseAddr+SDHC_BLK_COUNT) );
		
	// staus clear.
//	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT,
//		Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT)|SDHC_DMA_SIG_INT_EN );

	return TRUE;
}







//////////
// File Name : SDHC_WriteBlocksSuspend
// File Description : Suspend / Resume Feature Test function.
// Input : start block, block count, source buffer address, SDHC channel
// Output : Success or Failure
bool SDHC_WriteBlocksSuspend(u32 uStBlock, u16 uBlocks, u32 uBufAddr, SDHC* sCh)
{
	u32 i;
	u32 BackupAddress;
	u16 BackupBlockSize;
	u16 BackupBlockCnt;
	u32 BackupArgument;
	u16 BackupTransMode;
	u32 Writed_Count;
	
	if(sCh->m_eTransMode == SDHC_BYTE_MODE)
	{
		uStBlock = uStBlock<<9;	//	 uStBlock * 512;
	}
	else

	sCh->m_uRemainBlock = uBlocks;	// number of blocks.
	sCh->m_uBufferPtr = (u32*)uBufAddr;

	if ( !SDHC_WaitForCard2TransferState( sCh ) ) //added by HEAD. 2009.04.25
		return FALSE;

	SDHC_SetSystemAddressReg(sCh, uBufAddr);// AHB System Address For Write
	SDHC_SetTransferModeReg( (uBlocks==1)?(0):(1), 0, (uBlocks==1)?(0):(1), 1, 1, sCh ); // transfer mode 

	SDHC_SetBlockSizeReg(sCh, 7, 512); // Maximum DMA Buffer Size, Block Size
	SDHC_SetBlockCountReg(sCh, uBlocks); // Block Numbers to Write

	// Blockgap Interrupt Enable
	//Outp8( sCh->m_uBaseAddr+SDHC_BLOCKGAP_CTRL, (1<<3) );

	INTC_SetVectAddr( sCh->m_ucIntChannelNum, sCh->m_fDmaFn );
	INTC_Enable( sCh->m_ucIntChannelNum );
	
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
		SDHC_BUFFER_READREADY_SIG_INT_EN|
		SDHC_BUFFER_WRITEREADY_SIG_INT_EN|
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN|
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN|
		SDHC_COMMANDCOMPLETE_SIG_INT_EN );

	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN| //  ּ ó ?
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN);

	// Added by HEAD. 2009.04.25
	Disp("\n\n ______ [ Original Value] _____________ ");
	Disp("\n\t [00] SYS_ADDR    : 0x%X", Inp32(sCh->m_uBaseAddr+SDHC_SYS_ADDR) );
	Disp("\n\t [04] Block Size  : 0x%X", Inp16(sCh->m_uBaseAddr+SDHC_BLK_SIZE) );
	Disp("\n\t [06] Block Cnt   : 0x%X\n", Inp16(sCh->m_uBaseAddr+SDHC_BLK_COUNT) );

	
	// Multi Block Only
	if ( !SDHC_IssueCommand( sCh, 25, uStBlock, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE ))
	{
		return FALSE;
	}

/*------------------------------------------------------------------------------------------
	SD Host Controller Spec.
		3.112 Suspend/Resume
------------------------------------------------------------------------------------------ */

//[1], Set Stop at Block Gap Request
	Outp8( sCh->m_uBaseAddr+SDHC_BLOCKGAP_CTRL, (1<<3)|(1<<0) );

	//SDHC_INT_WAIT_CLEAR( sCh, 2, i );	// waiting Block Gap

	// wait until transfer complete.
	while( sCh->m_uRemainBlock != 0 );

	//	Delay(wait);
	Delay(1000); //added by HEAD. 2009.04.25

/*-------------------- [ Suspend mode ] ------------------------------------------ */	

//[5], Issue Suspend Command
	if ( !SDHC_IssueCommand52( sCh, 12, 0, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE, 1/* suspend */ ) )
	{
		return FALSE;
	}

	// Backup Status Register
	Disp( "status : %x , error status : %x\n" , Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT),
						Inp16( sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT) );
	
	Disp( "DMA Position : %x, SourcePosition:%x, Difference : %x\n", Inp32( sCh->m_uBaseAddr+SDHC_SYS_ADDR ),
		uBufAddr, Inp32( sCh->m_uBaseAddr+SDHC_SYS_ADDR )-uBufAddr );
	
//[7], Save Register (000-00Dh)
	BackupAddress = Inp32( sCh->m_uBaseAddr+SDHC_SYS_ADDR );
	BackupBlockSize = Inp16( sCh->m_uBaseAddr+SDHC_BLK_SIZE);
	BackupBlockCnt = Inp16( sCh->m_uBaseAddr+SDHC_BLK_COUNT);
	BackupArgument = Inp32( sCh->m_uBaseAddr+SDHC_ARG );
	BackupTransMode = Inp16( sCh->m_uBaseAddr+SDHC_TRANS_MODE);
	Writed_Count = uBlocks - BackupBlockCnt;

//[9], Clr Stop at Block Gap Request
	Outp8( sCh->m_uBaseAddr+SDHC_BLOCKGAP_CTRL, 0/*(1<<3)*/ );

	// Added by HEAD. 2009.04.25
	Disp("\n\n ______ [ Saved Value] _____________ ");
	Disp("\n\t [00] SYS_ADDR    : 0x%X", BackupAddress);
	Disp("\n\t [04] Block Size  : 0x%X", BackupBlockSize);
	Disp("\n\t [06] Block Cnt   : 0x%X\n", BackupBlockCnt);
	//Disp("\n\t [00] SYS_ADDR    : 0x%X", Inp32(sCh->m_uBaseAddr+SDHC_SYS_ADDR) );
	//Disp("\n\t [04] Block Size  : 0x%X", Inp16(sCh->m_uBaseAddr+SDHC_BLK_SIZE) );
	//Disp("\n\t [06] Block Cnt   : 0x%X\n", Inp16(sCh->m_uBaseAddr+SDHC_BLK_COUNT) );



	Delay(1000); //added by HEAD. 2009.04.25


/*-------------------- [ Resume mode ] ------------------------------------------ */

	sCh->m_uRemainBlock = BackupBlockCnt;

	if(sCh->m_eTransMode == SDHC_BYTE_MODE)
	{
		uStBlock = uStBlock<<9;	//	 uStBlock * 512;
	}
	else

	if ( !SDHC_WaitForCard2TransferState( sCh ) ) //added by HEAD. 2009.04.25
		return FALSE;

	SDHC_SetSystemAddressReg(sCh, (u32)BackupAddress);// AHB System Address For Write

	//SDHC_SetTransferModeReg( 1, 0, 1, 1, (sCh->m_eOpMode!=SDHC_DMA_MODE)?(0):(1), sCh ); //original code
	SDHC_SetTransferModeReg( 1, 0, 1, 1, 1, sCh ); 

	SDHC_SetBlockSizeReg(sCh, 7, 512); // Maximum DMA Buffer Size, Block Size
	SDHC_SetBlockCountReg(sCh, BackupBlockCnt); // Block Numbers to Write

	INTC_SetVectAddr( sCh->m_ucIntChannelNum, sCh->m_fDmaFn );
	INTC_Enable( sCh->m_ucIntChannelNum );

	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
		SDHC_BUFFER_READREADY_SIG_INT_EN|
		SDHC_BUFFER_WRITEREADY_SIG_INT_EN|
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN|
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN|
		SDHC_COMMANDCOMPLETE_SIG_INT_EN );
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN| //  ּ ó ?
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN);

	Delay(1000);
	
// [1], Restore Register(000-00Dh)
	Outp32( sCh->m_uBaseAddr+SDHC_SYS_ADDR, BackupAddress);
	Outp16( sCh->m_uBaseAddr+SDHC_BLK_SIZE, BackupBlockSize);
	Outp16( sCh->m_uBaseAddr+SDHC_BLK_COUNT, BackupBlockCnt);
	Outp32( sCh->m_uBaseAddr+SDHC_ARG, BackupArgument );
	Outp16( sCh->m_uBaseAddr+SDHC_TRANS_MODE, BackupTransMode);

// [2], Issue Resume Command	
	if ( !SDHC_IssueCommand52( sCh, 25, uStBlock+Writed_Count, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE, 2/* Resume */ ) )
	{
		return FALSE;
	}

// [3] Check Data Flag ???
	while( sCh->m_uRemainBlock != 0 );

// [4] Set Software Reset for DAT line
	Outp8( sCh->m_uBaseAddr+ SDHC_SOFTWARE_RESET,  0x4);
	Disp("\n Software Reset For DAT Line \n");

	// Added by HEAD. 2009.04.25
	Disp("\n\n ______ [ Final Value] _____________ ");
	Disp("\n\t [00] SYS_ADDR    : 0x%X", Inp32(sCh->m_uBaseAddr+SDHC_SYS_ADDR) );
	Disp("\n\t [04] Block Size  : 0x%X", Inp16(sCh->m_uBaseAddr+SDHC_BLK_SIZE) );
	Disp("\n\t [06] Block Cnt   : 0x%X\n", Inp16(sCh->m_uBaseAddr+SDHC_BLK_COUNT) );
		
	// staus clear.
//	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT,
//		Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT)|SDHC_DMA_SIG_INT_EN );

	return TRUE;
}





//////////
// File Name : SDHC_CMD53_RW
// File Description : Data Read or Write Transaction Only for SDIO.
// Input : ReadWrite : 0 - read / 1 - write.
//            uStBlock : start block address, uBlocks : data block size, uBufaddr : Source/Target Buffer Address, SDHC channel
// Output : success or failure
bool SDHC_CMD53_RW(u32 ReadWrite, u32 uStBlock, u16 uBlocks, u32 uBufAddr, SDHC* sCh)
{
	u32 temp;
	
	if(sCh->m_eTransMode == SDHC_BYTE_MODE)
		uStBlock = uStBlock<<9;	//	 uStBlock * 512;

	sCh->m_uRemainBlock = uBlocks;	// number of blocks.
	sCh->m_uBufferPtr = (u32*)uBufAddr;

	// test Set Address.
	//SDHC_setSDIOAddress(sCh, 0); // ???

	//INTC_SetVectAddr( sCh->m_ucIntChannelNum, SDHC_CMD52_DMA_Handler );
	//INTC_Enable( sCh->m_ucIntChannelNum );
	
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
//		Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE)|
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN|
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN|
		SDHC_READWAIT_SIG_INT_EN|
		SDHC_COMMANDCOMPLETE_SIG_INT_EN );
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		SDHC_READWAIT_SIG_INT_EN|
		SDHC_BLOCKGAP_EVENT_SIG_INT_EN|
		SDHC_TRANSFERCOMPLETE_SIG_INT_EN);

	SDHC_SetSystemAddressReg(sCh, (u32)uBufAddr);// AHB System Address For Write

	SDHC_SetBlockSizeReg(sCh, 7, 512); // Maximum DMA Buffer Size, Block Size
	SDHC_SetBlockCountReg(sCh, uBlocks); // Block Numbers to Write
	SDHC_SetTransferModeReg( (uBlocks==1)?(0):(1), 1, 0, 1, 1, sCh );

	Outp32(sCh->m_uBaseAddr+SDHC_CONTROL2, Inp32(sCh->m_uBaseAddr+SDHC_CONTROL2)|(1<<7));
	// Setting Blockgap controll
	Outp8( sCh->m_uBaseAddr+SDHC_BLOCKGAP_CTRL, (1<<3) );	// Blockgap Interrupt Test for SDIO suspend/resume.
//	Outp8( sCh->m_uBaseAddr+SDHC_BLOCKGAP_CTRL, (1<<3)|(1<<2)|(1<<1)|(1) );	// Normal Interrupt Test.
	
	// read or Write
	if ( !SDHC_IssueCommand( sCh, 53, ReadWrite<<31|1<<28|1<<27|1<<26|0x0<<9|0<<0, SDHC_CMD_ADTC_TYPE, SDHC_RES_R5_TYPE ) ) {
		return FALSE;
	}
	temp = Inp32( sCh->m_uBaseAddr+SDHC_RSP0 );
	Disp( "CMD53 Status Bit: %x\n", temp);
	while( sCh->m_uRemainBlock != 0 );	// wait until transfer done from DMA interrupt handler.
	
//	SDHC_INT_WAIT_CLEAR( sCh, 1, i );	// wait until transfer done.
	return TRUE;
}



//////////
// File Name : SDHC_INT_CLEAR (Inline Macro)
// File Description : Interrupt clear.
// Input : SDHC, interrupt bit, timeout loop count 
// Output : NONE.	// 0x7F000000 youngbo.song
// change from MACRO to FUNCTION by dusang.kim
void SDHC_NORMAL_INT_CLEAR(SDHC * sCh, u32 bit)
{
	Outp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT, (1<<bit) );
	while( Inp16( (sCh)->m_uBaseAddr + SDHC_NORMAL_INT_STAT ) & (1<<bit) );
}


void SDHC_ErrorInterruptHandler(SDHC* sCh)
{
	unsigned short status;

	status = sCh->m_sErrorStatus = Inp16(sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT);
	if( status & SDHC_AUTO_CMD12_ERROR ) {
		UART_Printf("SDHC_AUTO_CMD12_ERROR\n");
		SDHC_ERROR_INT_CLEAR(sCh, 8);
	}
	if( status & SDHC_CURRENT_LIMIT_ERROR ) {
		UART_Printf("SDHC_CURRENT_LIMIT_ERROR\n");
		SDHC_ERROR_INT_CLEAR(sCh, 7);
	}
	if( status & SDHC_DATA_END_BIT_ERROR ) {
		UART_Printf("SDHC_DATA_END_BIT_ERROR\n");
		SDHC_ERROR_INT_CLEAR(sCh, 6);
	}
	if( status & SDHC_DATA_CRC_ERROR ) {
		UART_Printf("SDHC_DATA_CRC_ERROR\n");

		SDHC_ERROR_INT_CLEAR(sCh, 5);
	}
	if( status & SDHC_DATA_TIME_OUT_ERROR ) {
		UART_Printf("SDHC_DATA_TIME_OUT_ERROR\n");

		
		SDHC_ERROR_INT_CLEAR(sCh, 4);
	}
	if( status & SDHC_CMD_INDEX_ERROR ) {
		UART_Printf("SDHC_CMD_INDEX_ERROR\n");
		SDHC_ERROR_INT_CLEAR(sCh, 3);
	}
	if( status & SDHC_CMD_END_BIT_ERROR ) {
		UART_Printf("SDHC_CMD_END_BIT_ERROR\n");
		SDHC_ERROR_INT_CLEAR(sCh, 2);
	}
	if( status & SDHC_CMD_CRC_ERROR ) {
		UART_Printf("SDHC_CMD_CRC_ERROR\n");
		SDHC_ERROR_INT_CLEAR(sCh, 1);
	}
	if( status & SDHC_CMD_TIMEOUT_ERROR ) {
//		UART_Printf("SDHC_CMD_TIMEOUT_ERROR\n");
		SDHC_ERROR_INT_CLEAR(sCh, 0);
	}
}


//////////
// File Name : SDHC_DMAIntHandler
// File Description : This function is interrupt service routine for Common DMA Handling.
// Input : NONE.
// Output : NONE.
void SDHC_InterruptHandler(SDHC * sCh)
{
	unsigned short status;
	u8 wakeup_stat;

	sCh->m_sNormalStatus |= status  = Inp16(sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT);
	if( status & SDHC_ERROR_INTERRUPT_EN )
	{
		Disp("\n Occur Interrupt : Error Interrupt \n");
		SDHC_ErrorInterruptHandler(sCh);
		SDHC_NORMAL_INT_CLEAR(sCh, 15);
	}
	
	if( status & SDHC_SD_ADDRESS_INT3_EN )
	{
		Disp("\n Occur Interrupt : SD Address INT 3 \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 14 );
	}
	
	if( status & SDHC_SD_ADDRESS_INT2_EN )
	{
		Disp("\n Occur Interrupt : SD Address INT 2 \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 13 );
	}
	
	if( status & SDHC_SD_ADDRESS_INT1_EN )
	{
		Disp("\n Occur Interrupt : SD Address INT 1 \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 12 );
	}
	
	if( status & SDHC_SD_ADDRESS_INT0_EN )
	{
		Disp("\n Occur Interrupt : SD Address INT 0 \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 11 );
	}
	
	if( status & SDHC_READWAIT_SIG_INT_EN )
	{
		Disp("\n Occur Interrupt : Read Wait Interrupt \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 10 );
	}
	
	if( status & SDHC_CCS_INTERRUPT_STATUS_EN )
	{
		Disp("\n Occur Interrupt : CCS Interrupt (CE_ATA) \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 9 );
		sCh->m_uCCSResponse=0;
	}
	
	if( status & SDHC_CARD_SIG_INT_EN ) // SDIO Interrupt. ???
	{
		Disp("\n Occur Interrupt : Card Interrupt\n");
		SDHC_NORMAL_INT_CLEAR(sCh, 8 );
	}
	
	if( status & SDHC_CARD_REMOVAL_SIG_INT_EN )
	{
		Disp("\n Occur Interrupt : Card Removed\n");
		SDHC_NORMAL_INT_CLEAR(sCh, 7 );
	}
	
	if( status & SDHC_CARD_INSERT_SIG_INT_EN )
	{
		Disp("\n Occur Interrupt : Card Insered\n");
		SDHC_NORMAL_INT_CLEAR(sCh, 6 );
	}
	
	if( status & SDHC_BUFFER_READREADY_SIG_INT_EN )
	{
		Disp("\n Occur Interrupt : Buffer Read Ready \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 5 );
		SDHC_ReadOneBlock( sCh );
	}
	
	if( status & SDHC_BUFFER_WRITEREADY_SIG_INT_EN )
	{
		Disp("\n Occur Interrupt : Buffer Write Ready \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 4 );
		SDHC_WriteOneBlock( sCh );
	}
	
	if( status & SDHC_DMA_SIG_INT_EN )
	{
		Disp("\n Occur Interrupt : DMA INT \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 3 );
		SDHC_SetSystemAddressReg(sCh, Inp32(sCh->m_uBaseAddr+SDHC_SYS_ADDR) );
	}
	
	if( status & SDHC_BLOCKGAP_EVENT_SIG_INT_EN )
	{
		Disp("\n Occur Interrupt : Block Gap. \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 2 );
	}

	if( status & SDHC_TRANSFERCOMPLETE_SIG_INT_EN )
	{
		sCh->m_uRemainBlock=0;
		Disp("\n Occur Interrupt : Transfer Complete. \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 1 );
	}
	
	if( status & SDHC_COMMANDCOMPLETE_SIG_INT_EN ) // Command Complete.
	{
		Disp("\n Occur Interrupt : Command Complete. \n");
		SDHC_NORMAL_INT_CLEAR(sCh, 0 );
	}

	wakeup_stat  = Inp8(sCh->m_uBaseAddr+SDHC_WAKEUP_CTRL);
	if( wakeup_stat & (1<<3) ) // Wakeup Interrupt 
	{
		Disp("\n Occur Wakeup Interrupt.  \n");
		Outp8(sCh->m_uBaseAddr+SDHC_WAKEUP_CTRL,0x0); // disable wakeup event
		Outp8(sCh->m_uBaseAddr+SDHC_WAKEUP_CTRL,0x8); // clear wakeup event
	}

}


//////////
// File Name : SDHC_DMADone
// File Description : DMA done interrupt handler for channel 0
// Input : NONE.
// Output : NONE.
void __irq SDHC_DMAInt0(void)
{
	Disp("\n irq SDHC_DMAInt0 \n");
	SDHC_DMAIntHandler( SDHC_curr_card[SDHC_CHANNEL_0], NUM_HSMMC0);
	INTC_ClearVectAddr();
}


void __irq SDHC_DMAInt1(void)
{
	Disp("\n irq SDHC_DMAInt1 \n");
	SDHC_DMAIntHandler( SDHC_curr_card[SDHC_CHANNEL_1], NUM_HSMMC1);
	INTC_ClearVectAddr();
}
void __irq SDHC_DMAInt2(void)
{
	Disp("\n irq SDHC_DMAInt2 \n");
	SDHC_DMAIntHandler( SDHC_curr_card[SDHC_CHANNEL_2], NUM_HSMMC2);
	INTC_ClearVectAddr();
}


void __irq SDHC_DMAInt3(void)
{
	Disp("\n irq SDHC_DMAInt3 \n");
	SDHC_DMAIntHandler( SDHC_curr_card[SDHC_CHANNEL_3], NUM_HSMMC3);
	INTC_ClearVectAddr();
}
//////////
// File Name : SDHC_DMADone
// File Description : Interrupt handler for channel 0
// Input : NONE.
// Output : NONE.
void __irq SDHC_ISR0(void)
{
	Disp("\n irq SDHC_ISR0 \n");
	SDHC_InterruptHandler( SDHC_curr_card[SDHC_CHANNEL_0]);
	INTC_ClearVectAddr();
}

//////////
// File Name : SDHC_ISR1
// File Description : Interrupt handler for channel 1
// Input : NONE.
// Output : NONE.
void __irq SDHC_ISR1(void) {
	Disp("\n irq SDHC_ISR1 \n");
	SDHC_InterruptHandler( SDHC_curr_card[SDHC_CHANNEL_1]);
	INTC_ClearVectAddr();
}

//////////
// File Name : SDHC_ISR2
// File Description : Interrupt handler for channel 2
// Input : NONE.
// Output : NONE.
void __irq SDHC_ISR2(void) {
	Disp("\n irq SDHC_ISR2 \n");
	SDHC_InterruptHandler( SDHC_curr_card[SDHC_CHANNEL_2]);
	INTC_ClearVectAddr();
}


//////////
// File Name : SDHC_ISR3
// File Description : Interrupt handler for channel 3
// Input : NONE.
// Output : NONE.
void __irq SDHC_ISR3(void) {
	Disp("\n irq SDHC_ISR3 \n");
	SDHC_InterruptHandler( SDHC_curr_card[SDHC_CHANNEL_3]);
	INTC_ClearVectAddr();
}


//////////
// File Name : SDHC_InitCh
// File Description : Initialize channel information.
// Input : SDHC Channel Number, SDHC pointer..
// Output : NONE.
void SDHC_InitCh(SDHC_channel eCh, SDHC *sCh)
{


	sCh->m_eChannel = eCh;
	
	SDHC_curr_card[sCh->m_eChannel]=sCh;	// Pointer...

	
	if ( eCh == SDHC_CHANNEL_0 ) {		// Channel 0
		sCh->m_uBaseAddr = (u8*)HSMMC0_BASE;
		sCh->m_fIntFn = SDHC_ISR0;
		sCh->m_fDmaFn = SDHC_DMAInt0;
		sCh->m_ucIntChannelNum = NUM_HSMMC0;
	}
	else if ( eCh == SDHC_CHANNEL_1 ) {	// Channel 1
		sCh->m_uBaseAddr = (u8*)HSMMC1_BASE;
		sCh->m_fIntFn = SDHC_ISR1;
		sCh->m_fDmaFn = SDHC_DMAInt1;		
		sCh->m_ucIntChannelNum = NUM_HSMMC1;
	}
	else if ( eCh == SDHC_CHANNEL_2 ) {	// channel 2
		sCh->m_uBaseAddr = (u8*)HSMMC2_BASE;
		sCh->m_fIntFn = SDHC_ISR2;
		sCh->m_fDmaFn = SDHC_DMAInt2;
		sCh->m_ucIntChannelNum = NUM_HSMMC2;
//		sCh->m_ucIntChannelNum = NUM_SPI1;
	}
	else if ( eCh == SDHC_CHANNEL_3 ) {	// channel 3
		sCh->m_uBaseAddr = (u8*)HSMMC3_BASE;
		sCh->m_fIntFn = SDHC_ISR3;
		sCh->m_fDmaFn = SDHC_DMAInt3;
		sCh->m_ucIntChannelNum = NUM_HSMMC3;
		
	
	}
	else {
		Assert(0);
	}
}


/**
 * act2  |  act1  | symbol
 *   0    |    0    | NOP
 *   0    |    1    | reserved
 *   1    |    0    | transfer data
 *   1    |    1    | Link
 * valid : valid = 1 -> Indicates this line of descriptor is effective. If valid=0 generate ADMA Error Interrupt and stop ADMA to prevent runaway.
 * end : end = 1 -> Indicates to end of descriptor. The transfer Complete Interrupt is generated when the operation of the descriptor line is completed.
 * Int : Interrupt = 1 -> generates DMA Interrupt When the Operation of the descriptor line is completed.
 */
typedef struct _SDHC_ADMA2_DescriptorEntry {
//	u32 address;
//	u16 length;
//	u8 reserved;
//	u8 attr;
	u8 attr;
	u8 reserved;
	u16 length;	
	u32 address;

} SDHC_ADMA2_DescriptorEntry;
unsigned int kekeke1;
unsigned int kekeke2;
//SDHC_ADMA2_DescriptorEntry ADMA2List[4096];
SDHC_ADMA2_DescriptorEntry* ADMA2List;

void SDHC_GenerateADMA2Descriptor( SDHC* sCh, u32 sysAddr, u16 linkSize, int linkCount) {
	int index=0;

	ADMA2List = (SDHC_ADMA2_DescriptorEntry*)0x41000000;
	
#if 1		// NORMAL Type
#if 0
	for(;linkCount>0;linkCount--) {
		ADMA2List[index].address = (u32)(sysAddr + linkSize*index);
		ADMA2List[index].length = linkSize;
		if( linkCount>1) {	
			// next transfer
			ADMA2List[index].attr = (0x2<<4) | (0<<2) | (0<<1) | (1<<0);	// interrupt and last index check.
		}
		else {
			// end
			ADMA2List[index].attr = (0x2<<4) | (1<<2) | (1<<1) | (1<<0);	// interrupt and last index check.
		}

		index++;
	}
#else
	for(;linkCount>0;linkCount--) {
		ADMA2List->address = (u32)(sysAddr + linkSize*index);
		ADMA2List->length = linkSize;
		ADMA2List->reserved= 0;
		ADMA2List->attr= 0;
		if( linkCount>1) {	
			// next transfer
			ADMA2List->attr = (0x2<<4) | (0<<2) | (0<<1) | (1<<0);	// interrupt and last index check.
		}
		else {
			// end
			ADMA2List->attr = (0x2<<4) | (0<<2) | (1<<1) | (1<<0);	// interrupt and last index check.
		}

		ADMA2List++;
		index++;
	}
#endif
#else		// Complexed type.
	for(;linkCount>0;linkCount--) {
		ADMA2List[index].address = (u32)(sysAddr + linkSize*count);
		ADMA2List[index].length = linkSize;
		ADMA2List[index].attr.act2 = 1;		// transfer.
		ADMA2List[index].attr.act1 = 0;
		ADMA2List[index].attr.interrupt = (linkCount>1)?(0):(1);	// interrupt..;
		ADMA2List[index].attr.end = (linkCount>1)?(0):(1);	// last index check.
		ADMA2List[index].attr.valid = 1;
		index++;
		
		ADMA2List[index].address = (u32)(&ADMA2List[index+1]);
		ADMA2List[index].length = 0;	// Don't care.
		ADMA2List[index].attr.act2 = 1;	// next link.
		ADMA2List[index].attr.act1 = 1;
		ADMA2List[index].attr.interrupt = 0;	// interrupt..;
		ADMA2List[index].attr.end = 0;	// last index check.
		ADMA2List[index].attr.valid = 1;
		index++;
	}			
#endif
	// ADMA2 Mode Selection.
	Outp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL, ( Inp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL ) & ~(0x3<<3) ) | (0x2<<3) );
	// ADMA2 System Address Selection.
	//Outp32( sCh->m_uBaseAddr + SDHC_ADMA_SYSTEM_ADDRESS, (u32)ADMA2List);
	Outp32( sCh->m_uBaseAddr + SDHC_ADMA_SYSTEM_ADDRESS, (u32)0x41000000);
}

//////////
// File Name : SDHC_OpenMedia
// File Description : Initialize channel information.
// Input : SDHC Clock Source, SDHC pointer.
// Output : Success or Failure.
u8 SDHC_OpenMedia(SDHC_clockSource eClkSrc, SDHC* sCh)
{
	SDHC_SpeedMode speed;
	u32 uSrcFreq;
	u32 uOperFreq,uIdFreq;
	u32 temp;

	sCh->m_eClockSource = eClkSrc;
	sCh->m_ucHostCtrlReg = 0;
	sCh->m_usClkCtrlReg = 0;
	if ( sCh->m_ucBandwidth == 0 )
		sCh->m_ucBandwidth = 4;		// bit width.

	// GPIO Setting.
	SDHC_SetGPIO(sCh->m_eChannel, sCh->m_ucBandwidth);
	
	if ( eClkSrc == SDHC_HCLK || eClkSrc == 0 ) {
		//uSrcFreq = g_HCLK;
		uSrcFreq = g_uHclkPsys; //g_HCLKD0;
	}
	
	else if ( eClkSrc == SDHC_EPLL ) {
		//	uSrcFreq = Inp32(0x7E00F014);	// temp <- EPLL CON0 setting
		
		//	((uSrcFreq>>16)&0xff) //M
		//	((uSrcFreq>>8)&0x3f) //P
		//	(uSrcFreq&0x07) //S
		//   ignore K value
		//	uSrcFreq = (u32)(((float)FIN)*((uSrcFreq>>16)&0xff)/(((uSrcFreq>>8)&0x3f)*(1<<(uSrcFreq&0x07))));

//		UART_Printf("Choose Operating clock: 1. 16MHz   2. 20MHz   3. 32.75MHz    4. 48MHz\n" );
//		temp = UART_GetIntNum();
		temp = 2;

		switch(temp)
		{
			case 1:
				SYSC_StartPLLbyFout(eEPLL, 32750000);	sCh->m_uClockDivision = 1;break;		// 16MHz
			case 2:
				SYSC_StartPLLbyFout(eEPLL, 80000000);	sCh->m_uClockDivision = 2;break;		// 20MHz
			case 3:
				SYSC_StartPLLbyFout(eEPLL, 32750000);	sCh->m_uClockDivision = 0;break;		// 32.75MHz		
			case 4:
				SYSC_StartPLLbyFout(eEPLL, 96000000);	sCh->m_uClockDivision = 1;break;		// 48MHz					
			default:
				SYSC_StartPLLbyFout(eEPLL, 80000000);	sCh->m_uClockDivision = 2;break;		// 20MHz			
	//	SYSC_StartPLLbyFout(eEPLL, 67750000);			// EPLL 67.75MHz  
	//	SYSC_StartPLLbyFout(eEPLL, 32750000);			// EPLL 32.75MHz 
		}

		SYSC_UpdateClkInform();
		uSrcFreq = g_uEPLL;       // sclk
	}
	
	else if ( eClkSrc == SDHC_EXTCLK ) {
		uSrcFreq = 24000000;	// 24MHz External crystal
	}

	else
	{
		Assert(FALSE);
	}

	//	sdhc_test.c ϴ clock division  
	//	SDHC_descriptor.m_uClockDivision = ?;	// clock division
	
	if ( sCh->m_uClockDivision == 0 ) {
		uOperFreq = uSrcFreq;
	}
	else {
		uOperFreq = uSrcFreq / (sCh->m_uClockDivision*2);
	}

	uIdFreq = uSrcFreq / (256);    // Identification clock is set in SDHC_Set_initclock()
	UART_Printf("Identification Freq = %dKHz , WorkingFreq = %dMHz\n",	uIdFreq/1000, uOperFreq/1000000 );

	
	SDHC_ResetController(sCh);

	// Identify Card
	if (!SDHC_IdentifyCard(sCh))
		return FALSE;

	// card Selection
	if ( !SDHC_IssueCommand( sCh, 7, (u32)(sCh->m_uRca<<16), SDHC_CMD_AC_TYPE, SDHC_RES_R1B_TYPE ) )
		return FALSE;

	if ( sCh->m_eCardType == SDHC_MMC_CARD ) {
		SDHC_ReadMMCExtCSD( sCh );
	}

	//  SpeedMode
		UART_Printf("Card speed mode setting...\n");
	if (sCh->m_eCardType == SDHC_MMC_CARD || sCh->m_eCardType == SDHC_CE_ATA_CARD ) {
		if (sCh->m_ucSpecVer >= 4) {
			speed = ( uOperFreq > SDHC_MMC_HIGH_SPEED_CLOCK) ? (1) : (0);
			if (!SDHC_SetMmcSpeedMode(speed, sCh)) {
				return FALSE;
			}
		}
		else{ // old version
			//Assert(uOperFreq<=SDHC_MMC_HIGH_SPEED_CLOCK); // Error! Old version MMC card can not support working frequency higher than 20MHz");
			UART_Printf("Error! Old version MMC card can not support working frequency higher than 20MHz\n");
			return FALSE;
		}
	}
	else if ( sCh->m_eCardType == SDHC_SD_CARD ) {
		SDHC_GetSdScr(sCh);
		if (sCh->m_ucSpecVer==1||sCh->m_ucSpecVer==2) {
			speed = ( uOperFreq > SDHC_SD_HIGH_SPEED_CLOCK) ? (1) : (0);
			if (!SDHC_SetSdCardSpeedMode(speed, sCh)) {
				return FALSE;
			}
		}
		else{
			UART_Printf("Error! Old version SD card can not support working frequency higher than 25MHz\n");
			return FALSE;
			
			//Assert(uOperFreq<=SDHC_SD_HIGH_SPEED_CLOCK); // Error! Old version SD card can not support working frequency higher than 25MHz");
		}
	}
	// host controller speed setting.
//	speed = ( uOperFreq > SDHC_MMC_HIGH_SPEED_CLOCK) ? (SDHC_HIGH_SPEED) : (SDHC_NORMAL_SPEED);
//	SDHC_SetHostCtrlSpeedMode( speed, sCh );

	SDHC_SetSdClockOnOff(0, sCh); // If the sd clock is to be changed, you need to stop sd-clock.
	SDHC_SetSdClock(sCh->m_uClockDivision, sCh, speed, uOperFreq);

	


	UART_Printf("Clock is changed to %dMHz\n",	uOperFreq/1000000 );

	// After a card was selected, then the bus width can be changed.
	UART_Printf("Card Data width setting...\n");

	if (!SDHC_SetDataTransferWidth( sCh))
		return FALSE;

	if (!SDHC_WaitForCard2TransferState(sCh))
		return FALSE;

	// CMD16 can not be set while card is in programming state.
	if (!SDHC_IssueCommand(sCh, 16, 512, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE ) ) // Set the block size
		return FALSE;

	// youngbo.song
	Outp32( sCh->m_uBaseAddr+SDHC_CONTROL2, Inp32(sCh->m_uBaseAddr+SDHC_CONTROL2)|(1<<8)|(2<<9)|(1<<28));
	return TRUE;
}

//////////
// File Name : SDHC_OpenMediaWithMode
// File Description : SD/MMC Channel initialize with parameter.
// Input : Bus width, Operation mode, Clock source, Clock divider, channel Number, Information Structuer pointer.
// Output : Success or Failure.
u8 SDHC_OpenMediaWithMode(u32 uBusWidth, SDHC_operation eOpMode, SDHC_clockSource eClkSrc,
	u32 uSdClkDivisor, SDHC_channel channel, SDHC* sCh)
{
	sCh->m_eOpMode = eOpMode;
	sCh->m_ucBandwidth = uBusWidth;
	sCh->m_uClockDivision = uSdClkDivisor;

	// channel initialize.
	SDHC_InitCh(channel, sCh);


	//INTC_SetVectAddr(sCh->m_eChannel, sCh->m_fIntFn);
	//INTC_Enable(sCh->m_ucIntChannelNum);
	
	if ( !SDHC_OpenMedia(eClkSrc, sCh ) )
		return FALSE;

	return TRUE;
}

//////////
// File Name : SDHC_CloseMedia
// File Description : This function close media session.
// Input : SDHC
// Output : NONE.
u8 SDHC_IdentifyCard(SDHC* sCh)
{

//	SDHC_SetHostCtrlSpeedMode(SDHC_NORMAL_SPEED, sCh);
	SDHC_Set_InitClock( sCh );	// initial Clock Setting.  // feedback timing control

//	Outp8( sCh->m_uBaseAddr+SDHC_TIMEOUT_CTRL, 
//		(Inp8(sCh->m_uBaseAddr+SDHC_TIMEOUT_CTRL)&(0xF<<4))|0xe);	// Timeout setting.

	Outp8( sCh->m_uBaseAddr+SDHC_TIMEOUT_CTRL, 
		(Inp8(sCh->m_uBaseAddr+SDHC_TIMEOUT_CTRL)&(0xF<<4))|0xe);	// Timeout setting.

	



	
#ifdef FEATURE_SDHC_INTERRUPT_CONTROL_MODE
	SDHC_SetSdhcInterruptEnable(0x3FF, 0xFF, 0x3FF, 0xFF, sCh); // original code, modify by HEAD 2009.04.20
	//SDHC_SetSdhcInterruptEnable(0x23, 0x0, 0x22, 0x0, sCh);
	//SDHC_SetSdhcInterruptEnable(0x3FF, 0xFF, 0x3FE, 0xFF, sCh);	
#else
        //(uNormalIntStatusEn, uErrorIntStatusEn, uNormalIntSigEn, uErrorIntSigEn, SDHC* sCh)
	 //SDHC_SetSdhcInterruptEnable(0x3F7, 0xFF, 0x0, 0x0, sCh); // original code, modify by HEAD 2009.04.20
//	SDHC_SetSdhcInterruptEnable(0x23, 0x0, 0x0, 0x0, sCh);
	SDHC_SetSdhcInterruptEnable(0xffff, 0xffff, 0x0, 0x0, sCh);
#endif
	// Check card OCR(Operation Condition Register)
	if (SDHC_SetSDOCR(sCh))
		sCh->m_eCardType = SDHC_SD_CARD;
	else if (SDHC_SetMmcOcr(sCh))
		sCh->m_eCardType = SDHC_MMC_CARD;
	else
		return FALSE;

	// Check the attached card and place the card in the IDENT state rHM_RSPREG0
	SDHC_IssueCommand( sCh, 2, 0, SDHC_CMD_BCR_TYPE, SDHC_RES_R2_TYPE );

	UART_Printf("- Product Name : %c%c%c%c%c%c\n",((Inp32(sCh->m_uBaseAddr+SDHC_RSP2)>>24)&0xFF),
		((Inp32(sCh->m_uBaseAddr+SDHC_RSP2)>>16)&0xFF),
		((Inp32(sCh->m_uBaseAddr+SDHC_RSP2)>>8)&0xFF),
		(Inp32(sCh->m_uBaseAddr+SDHC_RSP2)&0xFF),
		((Inp32(sCh->m_uBaseAddr+SDHC_RSP1)>>24)&0xFF),
		((Inp32(sCh->m_uBaseAddr+SDHC_RSP1)>>16)&0xFF));

	// Send RCA(Relative Card Address). It places the card in the STBY state
	sCh->m_uRca= (sCh->m_eCardType==SDHC_MMC_CARD) ? (1): (0);
	
	SDHC_IssueCommand( sCh, 3, sCh->m_uRca<<16, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE );
	if( sCh->m_eCardType == SDHC_SD_CARD)
	{
		sCh->m_uRca = (Inp32(sCh->m_uBaseAddr+SDHC_RSP0)>>16)&0xFFFF;
		UART_Printf("=>  RCA=0x%x\n", sCh->m_uRca);
	}

	//Send CSD
	SDHC_IssueCommand( sCh, 9, sCh->m_uRca<<16, SDHC_CMD_AC_TYPE, SDHC_RES_R2_TYPE );
	SDHC_DisplayCardInformation(sCh);

	return true;
}

//////////
// File Name : SDHC_CloseMedia
// File Description : This function close media session.
// Input : SDHC
// Output : NONE.
void SDHC_CloseMedia(SDHC* sCh)
{
	SDHC_SetSdClockOnOff(FALSE, sCh); // SDCLK Disable
}

//////////
// File Name : SDHC_WriteOneBlock
// File Description : This function writes one block data by CPU transmission mode.
// Input : SDHC( assert buffer pointer and remain data length.)
// Output : NONE.
void SDHC_WriteOneBlock( SDHC* sCh )
{
	u32* source_Ptr = sCh->m_uBufferPtr;
	int block_size;
	int i;

	block_size = Inp16( (sCh)->m_uBaseAddr + SDHC_BLK_SIZE ) & 0xFFF;
	
	SDHC_INT_WAIT_CLEAR( sCh, 4, i );
	
	block_size = (block_size+3) >> 2;	// block_size = (block_size+3) / 4;
	for(i=block_size; i>0; i--)	//512/4
	{
		Outp32( sCh->m_uBaseAddr+SDHC_BUF_DAT_PORT, *source_Ptr++);
	}
	sCh->m_uRemainBlock--;
	sCh->m_uBufferPtr = source_Ptr;
}

//////////
// File Name : SDHC_ReadOneBlock
// File Description : This function reads one block data by CPU transmission mode.
// Input : SDHC( assert buffer pointer and remain data length.)
// Output : NONE.
void SDHC_ReadOneBlock( SDHC* sCh )
{
	u32* target_Ptr = sCh->m_uBufferPtr;
	int block_size;
	int i;

	block_size = Inp16( (sCh)->m_uBaseAddr + SDHC_BLK_SIZE ) & 0xFFF;
	
	SDHC_INT_WAIT_CLEAR( sCh, 5, i );
		
	block_size = (block_size+3) >> 2;	// block_size = (block_size+3) / 4;
	for(i=block_size; i>0; i--)
	{
		*target_Ptr++ = Inp32( sCh->m_uBaseAddr+SDHC_BUF_DAT_PORT );
	}

	sCh->m_uRemainBlock--;
	sCh->m_uBufferPtr = target_Ptr;
}


//////////
// File Name : SDHC_WriteBlocks
// File Description : This function writes user-data common usage.
// Input : start block, block count, source buffer address, SDHC channel
// Output : Success or Failure
u8 SDHC_WriteBlocks(u32 uStBlock, u16 uBlocks, u32 uBufAddr, SDHC* sCh)
{
	u32 ignore,i;
	s32 temp;
	u32 sdhc_time;
	if(sCh->m_eTransMode == SDHC_BYTE_MODE)
		uStBlock = uStBlock<<9;	//	 uStBlock * 512;

	sCh->m_uRemainBlock = uBlocks;	// number of blocks.
	sCh->m_uBufferPtr = (u32*)uBufAddr;

	
	
	if ( !SDHC_WaitForCard2TransferState( sCh ) )
		return FALSE;

	SDHC_SetBlockSizeReg(sCh, 7, 512); // Maximum DMA Buffer Size, Block Size //original code
	//SDHC_SetBlockSizeReg(sCh, 0, 512); // Maximum DMA Buffer Size, Block Size	
	SDHC_SetBlockCountReg(sCh, uBlocks); // Block Numbers to Write 

	if( sCh->m_eOpMode == SDHC_SDMA_MODE ) {

		// not using buffer boundary
		Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE)) &(~(SDHC_DMA_SIG_INT_EN)));  

		SDHC_SetSystemAddressReg(sCh, uBufAddr);// AHB System Address For Write
		// SDMA mode Selection
		Outp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL, ( Inp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL ) & ~(0x3<<3) ) | (0x0<<3) );
		SDHC_SetTransferModeReg( (uBlocks==1)?(0):(1), 0, (uBlocks==1)?(0):(1), 1, 1, sCh ); // transfer mode 
	}
	else if( sCh->m_eOpMode == SDHC_ADMA2_MODE ) {

		// not using buffer boundary
		Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE)) &(~(SDHC_DMA_SIG_INT_EN)));  

		// ADMA mode Selection
		Outp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL, ( Inp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL ) & ~(0x3<<3) ) | (0x2<<3) );		
		SDHC_GenerateADMA2Descriptor(sCh, uBufAddr, 512, uBlocks);
		SDHC_SetTransferModeReg( (uBlocks==1)?(0):(1), 0, (uBlocks==1)?(0):(1), 1, 1, sCh ); // transfer mode 
	}
	else {
		// Interrupt or Polling mode.
	
		SDHC_SetTransferModeReg( (uBlocks==1)?(0):(1), 0, (uBlocks==1)?(0):(1), 1, 0, sCh ); // transfer mode 
	}


//	Outp32(0xE02000A0, Inp32(0xE02000A0)&~(0xf<<4)|(0x1<<4)); 	// GPD0CON[1]     --> OUTPUT
//	Outp32(0xE02000A4, Inp32(0xE02000A4)&~(0x1<<1)|(0x1<<1)); 	// GPD0DAT[1]     --> HIGH

	// test
//UART_Getc();


	if (uBlocks == 1)
	{
		StartTimer(2);
//		Outp32(0xE02000A4, ~(0x1<<1)); 	// GPD0DAT[1]     --> LOW

		if ( !SDHC_IssueCommand( sCh, 24, uStBlock, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE ))
		{
			return FALSE;
		}
	}

	else
	{
		StartTimer(2);
//		Outp32(0xE02000A4, ~(0x1<<1)); 	// GPD0DAT[1]     --> LOW
		if ( !SDHC_IssueCommand( sCh, 25, uStBlock, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE ))
		{
			return FALSE;
		}
	}


	if( sCh->m_eOpMode == SDHC_SDMA_MODE ) {

			//  total data > buffer boundary  case should be handled..
			//.....
			//

			SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );
			sdhc_time = StopTimer(2);
	}
	else if( sCh->m_eOpMode == SDHC_ADMA2_MODE ) {

			i=0;
			temp=0;
		//	UART_Printf("%x\n",sCh->m_uBaseAddr +SDHC_ADMA_ERROR);
			while(!temp){
				temp = (Inp32(sCh->m_uBaseAddr +SDHC_ADMA_ERROR))&(1<<10);
				if(i==10000000)return FALSE;
				i++;
			}

//			SDHC_INT_WAIT_CLEAR( sCh, 3, ignore );
			sdhc_time = StopTimer(2);
	SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );
	}
	else if( sCh->m_eOpMode == SDHC_POLLING_MODE ) {
		while(sCh->m_uRemainBlock != 0 ) {
			// Wait for buffer write ready... - SDHC_BUFFER_WRITEREADY_STS_INT_EN
			//SDHC_INT_WAIT_CLEAR( sCh, 4, temp );
			//if ( 100000 > temp )	UART_Printf( "time:%d\n", 20000000-temp);
			SDHC_WriteOneBlock( sCh );

		}
	
		sdhc_time = StopTimer(2);
		SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );
	}
	else if( sCh->m_eOpMode == SDHC_INTERRUPT_MODE ) {

	//// buffer write ready interrupt enable
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE)) |SDHC_BUFFER_WRITEREADY_SIG_INT_EN);

	INTC_SetVectAddr( sCh->m_ucIntChannelNum, sCh->m_fIntFn );
	INTC_Enable( sCh->m_ucIntChannelNum );

	
		while(sCh->m_uRemainBlock != 0); // wait for write all blocks
		sdhc_time = StopTimer(2);
		SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );

	//// buffer write ready interrupt disable
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE)) &(~(SDHC_BUFFER_WRITEREADY_SIG_INT_EN)));	
	INTC_Disable( sCh->m_ucIntChannelNum );		
	
	}
	else {
		Assert("Not support mode");
	}

	// wait for transfer complete.
//	SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );
//	Outp32(0xE02000A4, (0x1<<1)); 	// GPD0DAT[1]     --> HIGH	

	UART_Printf("WriteTime: %dus (  Kbyte), Data rate= %fKbyte/sec\n",sdhc_time,(float)(uBlocks<<9)/sdhc_time*1000);
	sCh->m_uRemainBlock = 0;

	return TRUE;
}


//////////
// File Name : SDHC_ReadBlocks
// File Description : This function reads user-data common usage.
// Input : start block, block count, target buffer address, SDHC channel
// Output : Success or Failure
u8 SDHC_ReadBlocks(u32 uStBlock, u16 uBlocks, u32 uBufAddr, SDHC* sCh)
{
	u32 ignore,i;
	u32 temp;
	u32 sdhc_time;
	if(sCh->m_eTransMode == SDHC_BYTE_MODE)
		uStBlock = uStBlock<<9;//*512;
		
	sCh->m_uRemainBlock = uBlocks;
	sCh->m_uBufferPtr=(u32*)uBufAddr;

	if ( !SDHC_WaitForCard2TransferState( sCh ) )
	{
		UART_Printf("SDHC_RSP0:0x%x\n",Inp32( sCh->m_uBaseAddr+SDHC_RSP0));
		return FALSE;
		}

	SDHC_SetBlockSizeReg(sCh, 7, 512); // Maximum DMA Buffer Size, Block Size
	SDHC_SetBlockCountReg(sCh, uBlocks); // Block Numbers to Write

	if ( sCh->m_eOpMode == SDHC_SDMA_MODE ) {

		// not using buffer boundary
		Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE)) &(~(SDHC_DMA_SIG_INT_EN)));  
		
		// SDMA mode Selection
		Outp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL, ( Inp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL ) & ~(0x3<<3) ) | (0x0<<3) );		
		SDHC_SetSystemAddressReg(sCh, uBufAddr);// AHB System Address For Write
		SDHC_SetTransferModeReg((uBlocks==1)?(0):(1), 1, (uBlocks==1)?(0):(1), 1, 1, sCh ); //Transfer mode setting		
	}
	else if ( sCh->m_eOpMode == SDHC_ADMA2_MODE ) {

		// not using buffer boundary
		Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE)) &(~(SDHC_DMA_SIG_INT_EN)));  
		
		// ADMA mode Selection
		Outp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL, ( Inp8( sCh->m_uBaseAddr + SDHC_HOST_CTRL ) & ~(0x3<<3) ) | (0x2<<3) );		
		SDHC_GenerateADMA2Descriptor(sCh, uBufAddr, 512, uBlocks);
		SDHC_SetTransferModeReg((uBlocks==1)?(0):(1), 1, (uBlocks==1)?(0):(1), 1, 1, sCh ); //Transfer mode setting
	}
	else {
		SDHC_SetTransferModeReg((uBlocks==1)?(0):(1), 1, (uBlocks==1)?(0):(1), 1, 0, sCh ); //Transfer mode setting
	}

//	Outp32(0xE02000A0, Inp32(0xE02000A0)&~(0xf<<4)|(0x1<<4)); 	// GPD0CON[1]     --> OUTPUT
//	Outp32(0xE02000A4, Inp32(0xE02000A4)&~(0x1<<1)|(0x1<<1)); 	// GPD0DAT[1]     --> HIGH



	if (uBlocks == 1) { // CMD17: Single-Read
			StartTimer(2);	
//			Outp32(0xE02000A4, ~(0x1<<1)); 	// GPD0DAT[1]     --> LOW
		if ( !SDHC_IssueCommand( sCh, 17, uStBlock, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE )) {


			return FALSE;
		}
	}
	else { // CMD18: Multi-Read
		StartTimer(2);
//		Outp32(0xE02000A4, ~(0x1<<1)); 	// GPD0DAT[1]     --> LOW
		if( !SDHC_IssueCommand( sCh, 18, uStBlock, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE )) {

			return FALSE;
		}
	}

	if( sCh->m_eOpMode == SDHC_SDMA_MODE  )
	{

		//  total data > buffer boundary  case should be handled..
		//.....
		//
		SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );
	
		sdhc_time = StopTimer(2);
			
	}
	else if(  sCh->m_eOpMode == SDHC_ADMA2_MODE )
	{
	//	temp=SDHC_INT_WAIT_CLEAR( sCh, 3, temp );
		i=0;
		temp =0;
		while(!temp ){
			temp = (Inp32(sCh->m_uBaseAddr +SDHC_ADMA_ERROR))&(1<<10);
			if(i==10000000)return FALSE;
			i++;
		}
		sdhc_time = StopTimer(2);
		SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );
			
	}
	else if( sCh->m_eOpMode == SDHC_POLLING_MODE ) {
		while(sCh->m_uRemainBlock != 0 ) {

				// Wait for buffer read ready...- SDHC_BUFFER_READREADY_STS_INT_EN
		//	SDHC_INT_WAIT_CLEAR( sCh, 5, temp );
		//	if ( 100000 > temp  )	UART_Printf( "time:%d\n", 20000000-temp);
	
			SDHC_ReadOneBlock( sCh );
		}
		sdhc_time = StopTimer(2);
		SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );
	}
	else if( sCh->m_eOpMode == SDHC_INTERRUPT_MODE ) {

	//// buffer read ready interrupt enable
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE)) |SDHC_BUFFER_READREADY_SIG_INT_EN);
	
	INTC_SetVectAddr( sCh->m_ucIntChannelNum, sCh->m_fIntFn );
	INTC_Enable( sCh->m_ucIntChannelNum );
	
		while(sCh->m_uRemainBlock != 0);		// wait for reading all blocks
	sdhc_time = StopTimer(2);
	SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );
	//// buffer read ready interrupt disable
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE)) &(~(SDHC_BUFFER_WRITEREADY_SIG_INT_EN)));	
	INTC_Disable( sCh->m_ucIntChannelNum );		


	}
	else {
		Assert("Not support mode");
	}

	// wait for transfer complete.
//	SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );
//	Outp32(0xE02000A4, (0x1<<1)); 	// GPD0DAT[1]     --> HIGH	

	UART_Printf("ReadTime: %dus (  Kbyte), Data rate= %fKbyte/sec\n",sdhc_time,(float)(uBlocks<<9)/sdhc_time*1000);	
	sCh->m_uRemainBlock = 0;
	
	return TRUE;	// block_cnt * 512
}



//////////
// File Name : SDHC_WaitForCard2TransferState
// File Description : Get error source data. 
// Input : SDHC channel
// Output : Success or Failure
u8 SDHC_WaitForCard2TransferState(SDHC* sCh) {
	u32 uStatus;

	// do until programming status.
	do
	{
		if ( !SDHC_IssueCommand( sCh, 13, sCh->m_uRca<<16, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE) )
		{
			return FALSE;
		}
		//Disp("\n\t HEAD!!\n");
		uStatus = (Inp32( sCh->m_uBaseAddr+SDHC_RSP0)>>9) & 0xf;
	}
	while(uStatus==7);

	return (uStatus==4) ? TRUE : FALSE;
}

//////////
// File Name : SDHC_ResetController
// File Description : Reset SDHC Controller.
// Input : SDHC channel
// Output : NONE.
void SDHC_ResetController(SDHC* sCh) {
	Outp8( sCh->m_uBaseAddr+SDHC_SOFTWARE_RESET, 0x3);
}

//////////
// File Name : SDHC_IssueCommand
// File Description : This function issue sd/mmc command.
// Input : SDHC, command Index, Argument, Command Type, Response Type
// Output : Command Result.
u8 SDHC_IssueCommand( SDHC* sCh, u16 uCmd, u32 uArg, SDHC_CommandType cType, SDHC_ResponseType rType ) {
	u16 sfrData;
	u32 Loop;

	while( Inp32( sCh->m_uBaseAddr+SDHC_PRESENT_STAT ) & 0x1 );	// Check CommandInhibit_CMD

//	if ( rType == SDHC_RES_R1B_TYPE ) {
//		while( Inp32( sCh->m_uBaseAddr+SDHC_PRESENT_STAT ) & (1<<1) );// Check CommandInhibit_DAT
//	}

	sfrData = (uCmd<<8) | SDHC_cmd_sfr_data[ rType ];
	if ( cType == SDHC_CMD_ADTC_TYPE ) {
		sfrData |= (1<<5);
	}
	if ( uCmd == 12 ) { // check abort bit when stop command.
		sfrData |= (3<<6);
	}
	// argument setting.
	Outp32( sCh->m_uBaseAddr+SDHC_ARG, uArg);
	
	Outp16( sCh->m_uBaseAddr+SDHC_COMMAND, sfrData);

	// Command Complete. - SDHC_COMMANDCOMPLETE_STS_INT_EN
	// ?????????
	SDHC_INT_WAIT_CLEAR( sCh, 0, Loop);

	// newly added 2009.10.22				///in  r1b case,   wait until card release data line(during card busy)...
	if ( rType == SDHC_RES_R1B_TYPE ) {
	while( Inp32( sCh->m_uBaseAddr+SDHC_PRESENT_STAT ) & (1<<1) );// Check CommandInhibit_DAT
	}
	/////


	// Error Status Check - reduce too much error message.
	if ( (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT ) & (1<<15)) && !(uCmd==1||uCmd==55||uCmd==41) ) {
		UART_Printf("Command = %d, Error Stat = %x\n", Inp16( sCh->m_uBaseAddr+SDHC_COMMAND )>>8,
			Inp16( sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT ) );
		Outp16( sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT, Inp16( sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT ) );
		return FALSE;
	}

	return TRUE;
}

//////////
// File Name : SDHC_GetSdSCR
// File Description : Setting sd card SCR infomation.
// Input : SDHC, temp buffer for gather sd infomation up to 512 byte.
// Output : success of failure.
u8 SDHC_GetSdScr(SDHC* sCh) {
	u32 buffer[2];
	u32 ignore = 0;

	if(!SDHC_IssueCommand( sCh, 16, 8, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE ) )
		return FALSE;

	SDHC_SetBlockSizeReg( sCh, 7, 8);
	SDHC_SetBlockCountReg( sCh, 1);
	SDHC_SetTransferModeReg(0, 1, 0, 0, 0, sCh);
	sCh->m_uRemainBlock = 1;
	sCh->m_uBufferPtr = buffer;

	// CMD55 (For ACMD)
	if (!SDHC_IssueCommand( sCh, 55, sCh->m_uRca<<16, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE ) )
		return FALSE;

	// Acmd51 - Send SCR
	if(!SDHC_IssueCommand( sCh, 51, 0, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE ) )
		return FALSE;

#ifdef FEATURE_SDHC_INTERRUPT_CONTROL_MODE
	while( sCh->m_uRemainBlock != 0 )
		ignore++;
#else
	SDHC_ReadOneBlock( sCh );
	//SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );	// SDHC_TRANSFERCOMPLETE_STS_INT_EN
#endif
	SDHC_INT_WAIT_CLEAR( sCh, 1, ignore );	// SDHC_TRANSFERCOMPLETE_STS_INT_EN // original position. 2009.04.20

// Transfer mode is determined by capacity register at OCR setting.
//	sCh->m_eTransMode = SDHC_BYTE_MODE;
	if ((*buffer&0xf) == 0x0)
		sCh->m_ucSpecVer = 0; // Version 1.0 ~ 1.01
	else if ((*buffer&0xf) == 0x1)
		sCh->m_ucSpecVer = 1; // Version 1.10, support cmd6
	else if((*buffer&0xf) == 0x2) {
		sCh->m_ucSpecVer = 2; // Version 2.0 support cmd6 and cmd8
//		sCh->m_eTransMode = SDHC_BLOCK_MODE;
	}
	else {
		sCh->m_ucSpecVer = 0; // Error... Deda
	}

	UART_Printf("SDSpecVer=%d\n", sCh->m_ucSpecVer);
	return TRUE;
}


//////////
// File Name : SDHC_ReadMMCExtCSD
// File Description : Read mmc extended CSD Register.
// Input : SDHC channel
// Output : success of failure.
u8 SDHC_ReadMMCExtCSD(SDHC* sCh) {
	u32 S_CMD_SET, uHsTiming, uBusMode;
//	u8 buffer[512]; // - to do - move to another memory. - too much stack.
	u8 *buffer;
	u32 ignore;

//	SDHC_SetSystemAddressReg(sCh, (u32)(&SDHC_globl_testBuffer[0]));// AHB System Address For Write
//	buffer = &SDHC_globl_testBuffer[0];
	
	SDHC_SetBlockSizeReg(sCh, 7, 512); // Maximum DMA Buffer Size, Block Size
	SDHC_SetBlockCountReg(sCh, 1); // Block Numbers to Write
	SDHC_SetTransferModeReg(0, 1, 0, 1, 0, sCh );
	sCh->m_uRemainBlock = 1;
	sCh->m_uBufferPtr = (u32*)SDHC_globl_testBuffer;
	buffer = (u8*)sCh->m_uBufferPtr;
	
	if ( sCh->m_ucSpecVer < 4 ) {
		UART_Printf("SKIP SDHC_ReadMMCExtCSD\n");
		return TRUE;
	}
	if(!SDHC_IssueCommand( sCh, 8, 0, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE ))	// cmd8			
		return FALSE;
	
#ifdef FEATURE_SDHC_INTERRUPT_CONTROL_MODE
	while( sCh->m_uRemainBlock != 0 )
		ignore++;
#else
	SDHC_ReadOneBlock( sCh );
#endif
	SDHC_INT_WAIT_CLEAR(sCh, 1, ignore );
	
	S_CMD_SET = buffer[504];
	uHsTiming = buffer[185];
	uBusMode = buffer[183];
	
	UART_Printf( "uHsTiming : %d\n", uHsTiming );
	UART_Printf( "uBusMode : %d\n", uBusMode );
	UART_Printf( "Support Commmand Set : %d\n", buffer[504] );
	UART_Printf( "sector count : %d\n", *((u32*)&buffer[212]) );
	UART_Printf( "min Write performance 8 @52 : %d \n", buffer[210] );
	UART_Printf( "min Read performance 8 @52 : %d \n", buffer[209] );
	UART_Printf( "min Write performance 8 @26 : %d \n", buffer[208] );
	UART_Printf( "min Read performance 8 @26 : %d \n", buffer[207] );
	UART_Printf( "min Write performance 4 @26 : %d \n", buffer[206] );
	UART_Printf( "min Read performance 4 @26 : %d \n", buffer[205] );
	UART_Printf( "Card Type : %d \n", buffer[196] );
	UART_Printf( "CSD structure Version : %d \n", buffer[194] );
	UART_Printf( "CSD Revision : %d\n", buffer[192] );
	UART_Printf( "Command Set : %d\n", buffer[191] );
	UART_Printf( "Command Set Revision : %d\n", buffer[189] );
	UART_Printf( "Power Class :%d \n", buffer[187] );
	UART_Printf( "High Speed Interface Timing : %d\n", buffer[185] );
	UART_Printf( "Bus width timing : %d\n", buffer[183] );
	UART_Printf( "Erased memory content : %d\n", buffer[181] );
	
	SDHC_global_card_size = *((u32*)&buffer[212]);
	UART_Printf("SDHC_global_card_size(high capacity)=%dMByte\n", SDHC_global_card_size*512/1024/1024);
	
	if(S_CMD_SET & (1<<4)) {
		UART_Printf("\n========CE ATA Card detect========\n");
		sCh->m_eCardType = SDHC_CE_ATA_CARD;
	}
	else if(S_CMD_SET & (1<<2))
		UART_Printf("\n========Content Protection SecureMMC Card detect========\n");
	else if(S_CMD_SET & (1<<1))
		UART_Printf("\n========SecureMMC Card detect========\n");
	else if(S_CMD_SET & (1<<0))
		UART_Printf("\n========Standard MMC Card detect========\n");

	return TRUE;
}


//////////
// File Name : SDHC_ClearErrInterruptStatus
// File Description : Clear error interrupt status register.
// Input : SDHC channel
// Output : NONE.
void SDHC_ClearErrInterruptStatus(SDHC* sCh) {
	u16 usSfr, usSfr1;
	usSfr = Inp16(sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT);

	while (usSfr&(0x1<<15)) {
		usSfr1 = Inp16(sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT);
		Outp16(sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT, usSfr1);
		usSfr = Inp16(sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT);
	}
}

//////////
// File Name : SDHC_SetMmcOcr
// File Description : Get MMC OCR Register from MMC Card.
// Input : SDHC channel
// Output : success or failure.
u8 SDHC_SetMmcOcr(SDHC* sCh)
{
	u32 i, OCR;

	// Place all cards in the idle state.
	if (!SDHC_IssueCommand( sCh, 0, 0, SDHC_CMD_BC_TYPE, SDHC_RES_NO_TYPE ) )
		return FALSE;

	for (i=0; i<2000; i++)
	{
#if 1	// for New Movinand 2007.3.29		
		SDHC_IssueCommand( sCh, 1, 0x40FF8080, SDHC_CMD_BCR_TYPE, SDHC_RES_R3_TYPE ); // (Ocr:2.7V~3.6V)
//		SDHC_IssueCommand( sCh, 1, 0x40FF8000, SDHC_CMD_BCR_TYPE, SDHC_RES_R3_TYPE ); // (Ocr:2.7V~3.6V)
//		SDHC_IssueCommand( sCh, 1, 0x40000080, SDHC_CMD_BCR_TYPE, SDHC_RES_R3_TYPE ); // (Ocr:1.7V~1.95V)		
		OCR = Inp32(sCh->m_uBaseAddr+SDHC_RSP0);
#else
		SDHC_IssueCommand( sCh, 1, 0, SDHC_CMD_BCR_TYPE, SDHC_RES_R3_TYPE );
		OCR = Inp32(sCh->m_uBaseAddr+SDHC_RSP0)|(1<<30);
		SDHC_IssueCommand( sCh, 1, OCR, SDHC_CMD_BCR_TYPE, SDHC_RES_R3_TYPE );
#endif
		if (Inp32(sCh->m_uBaseAddr+SDHC_RSP0)&(unsigned int)(0x1<<31))
		{
			if(OCR & (1<<7))
				UART_Printf("Voltage range : 1.65V ~ 1.95V\n");
			if(OCR & (1<<21))
				UART_Printf("Voltage range: 2.7V ~ 3.4V\n");
			else if(OCR & (1<<20))
				UART_Printf("Voltage range: 2.7V ~ 3.3V\n");
			else if(OCR & (1<<19))
				UART_Printf("Voltage range: 2.7V ~ 3.2V\n");
			else if(OCR & (1<<18))
				UART_Printf("Voltage range: 2.7V ~ 3.1V\n");
			else
				continue;

			if ( Inp32(sCh->m_uBaseAddr+SDHC_RSP0) &(unsigned int)(0x1<<30) ) {
				UART_Printf("block mode\n");
				sCh->m_ucSpecVer=4;	// default spec version
				sCh->m_eTransMode=SDHC_BLOCK_MODE;
			}
			else {
				UART_Printf("byte mode\n");
				sCh->m_ucSpecVer=1;	// default spec version
				sCh->m_eTransMode=SDHC_BYTE_MODE;
			}
			sCh->m_eCardType= SDHC_MMC_CARD;
			// If the current card is SD card, then there's time out error, need to be cleared.
			SDHC_ClearErrInterruptStatus(sCh);
			return TRUE;
		}
	}

	// If the current card is SD card, then there's time out error, need to be cleared.
	SDHC_ClearErrInterruptStatus(sCh);
	return false;
}


//////////
// File Name : SDHC_SetSDOCR
// File Description : Get SD OCR Register from SD Card.
// Input : SDHC channel
// Output : success or failure.
u8 SDHC_SetSDOCR(SDHC* sCh)
{
	u32 i, OCR;

	// Place all cards in the idle state.
	if (!SDHC_IssueCommand( sCh, 0, 0, SDHC_CMD_BC_TYPE, SDHC_RES_NO_TYPE ) )
		return FALSE;
#ifdef FEATURE_SDHC_HIGH_CAPACITY_CARD_SUPPORT
	if ( SDHC_IssueCommand( sCh, 8, (0x1<<8)|(0xaa), SDHC_CMD_BCR_TYPE, SDHC_RES_R7_TYPE ) == TRUE ) {
		UART_Printf ( "CMD8 return TRUE\n" );
	}
	// Normal SD has Command Response Error.
	SDHC_ClearErrInterruptStatus(sCh);
#endif

	// for  500  ????????????????????????????????????????????????????
	for(i=0; i<2000; i++)
	{
		// CMD55 (For ACMD)
		SDHC_IssueCommand( sCh, 55, 0, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE );
		// (Ocr:2.7V~3.6V)
		SDHC_IssueCommand( sCh, 41, 0x40ff8000, SDHC_CMD_BCR_TYPE, SDHC_RES_R3_TYPE );

		if (Inp32(sCh->m_uBaseAddr+SDHC_RSP0)&(unsigned int)(0x1<<31))
		{
			OCR = Inp32( sCh->m_uBaseAddr+SDHC_RSP0);
			//UART_Printf("\nrHM_RSPREG0=%x",rHM_RSPREG0);
			if(OCR & (1<<7))
				UART_Printf("Voltage range : 1.65V ~ 1.95V\n");
			if(OCR & (1<<21))
				UART_Printf("Voltage range: 2.7V ~ 3.4V\n");
			else if(OCR & (1<<20))
				UART_Printf("Voltage range: 2.7V ~ 3.3V\n");
			else if(OCR & (1<<21))
				UART_Printf("Voltage range: 2.7V ~ 3.4V\n");
			else if(OCR & (1<<23))
				UART_Printf("Voltage range: 2.7V ~ 3.6V\n");

			if(OCR&(0x1<<30)) {
				sCh->m_eTransMode = SDHC_BLOCK_MODE;
				UART_Printf("High Capacity Card\n");
			}
			else {
				sCh->m_eTransMode = SDHC_BYTE_MODE;
				UART_Printf("Byte mode\n");
			}
			// Normal SD has Command Response Error.
			SDHC_ClearErrInterruptStatus(sCh);

			sCh->m_eCardType = SDHC_SD_CARD;
			return TRUE;
		}
	}
	// The current card is MMC card, then there's time out error, need to be cleared.
	SDHC_ClearErrInterruptStatus(sCh);
	return FALSE;
}

void SDHC_Set_InitClock( SDHC* sCh ) {

	u32 OutEdgeInv, temp, tx_on, rx_on,tx_delay,rx_delay;
//	SYSC_SetLockTime( eEPLL, 300);
//	SYSC_CtrlCLKOUT(eCLKOUT_EPLLOUT, 0);

	//modify by HEAD. 02/16/2009.
	//SYSC_SetPLL(eEPLL, 30, 3, 2, 0);	

	//Outp32( 0x7E00F01C, 7 );
	
//	SYSC_SetPLL(eEPLL, 100, 6, 2, 0);		// EPLL => 100MHz
//	SYSC_StartPLLbyFout(eEPLL, 80000000);		// move to openmedia()
/*
	UART_Printf("Input OutEdgeInv.... 0:normal  1:Inversion  \n");
	OutEdgeInv = UART_GetIntNum();
	UART_Printf("select Tx Feedback on/off.... 0:off  1:on  \n");
	tx_on = UART_GetIntNum();
	UART_Printf("select Rx Feedback on/off.... 0:off  1:on  \n");
	rx_on = UART_GetIntNum();
	UART_Printf("select Tx Feedback delay.... 0 ~ 3  \n");
	tx_delay= UART_GetIntNum();
	UART_Printf("select Rx Feedback delay.... 0 ~ 3  \n");
	rx_delay= UART_GetIntNum();
*/	
	OutEdgeInv = 0;
	tx_on = 0;
	rx_on = 1;
	tx_delay= 0;
	rx_delay= 0;
	
	SDHC_SetHostOutEdgeMode(OutEdgeInv, sCh);

	Outp32(0xE0100210,Inp32(0xE0100210)&~(0xffff)|0x7777);   ///  SYSCON CLK_SRC4 : SCLKEPLL
	Outp32(0xE0100310,Inp32(0xE0100310)&~(0xffff));   /// SYSCON CLK_DIV4 : 0  bypass.
	
	Outp32( sCh->m_uBaseAddr+SDHC_CONTROL2, (Inp32( sCh->m_uBaseAddr+SDHC_CONTROL2)&~(3<<4))
		|(tx_on<<15)|(rx_on<<14)|(0x1<<30)|(0x1<<8)|(SDHC_EPLL<<4)|(1<<1) );  // ENCMDCNFMSK,ENCLKOUTHOLD,ENCLKOUTMSKCON 
	Outp32( sCh->m_uBaseAddr+SDHC_CONTROL3, (((tx_delay&2)>>1)<<31) | ((tx_delay&1)<<23) | (((rx_delay&2)>>1)<<15) | ((rx_delay&1)<<7) );

	// SDCLK Value Setting + Internal Clock Enable
	Outp16( sCh->m_uBaseAddr+SDHC_CLK_CTRL,
		(Inp16(sCh->m_uBaseAddr+SDHC_CLK_CTRL)&(~(0xff<<8)))|(0x80<<8)|(1<<0) );      // division 256 for identify mode

	// CheckInternalClockStable
	while (!(Inp16( sCh->m_uBaseAddr+SDHC_CLK_CTRL )&0x2));

	SDHC_SetSdClockOnOff( TRUE, sCh);
	UART_Printf("HOST_CTRL = %x. (bit2-OutEdgeINV)\n",Inp8(sCh->m_uBaseAddr+SDHC_HOST_CTRL) );
	UART_Printf("CONTROL2 = %x (bit15-TxEN, bit14-RxEN)\n",Inp32( sCh->m_uBaseAddr+SDHC_CONTROL2 ));
	UART_Printf("CONTROL3 = %x (bit31,23-TxDelay, bit15,7-RxDelay)\n",Inp32( sCh->m_uBaseAddr+SDHC_CONTROL3 ));
	UART_Printf("CLKCON = %x\n",Inp16( sCh->m_uBaseAddr+SDHC_CLK_CTRL ));
}


//////////
// File Name : SDHC_SetSDOCR
// File Description : Get SD OCR Register from SD Card.
// Input : SDHC channel
// Output : success or failure.
void SDHC_SetSdClock(u32 uDivisor, SDHC* sCh, SDHC_SpeedMode speed, u32 workingFreq )
{
#if 0
	// SD : Setup Time 6ns, Hold Time 2ns
	if ( sCh->m_eCardType == SDHC_SD_CARD || sCh->m_eCardType == SDHC_SDIO_CARD ) {
		// original
		//Outp32( sCh->m_uBaseAddr+SDHC_CONTROL2, (1<<30)|(0x0<<15)|(0x0<<14)|(0x1<<8)|(sCh->m_eClockSource<<4) );
		//Outp32( sCh->m_uBaseAddr+SDHC_CONTROL3, (0<<31)|(0<<23)|(0<<15)|(0<<7) );
		// modify by HEAD. 2009/02/09
		Outp32( sCh->m_uBaseAddr+SDHC_CONTROL2, (1<<30)|(0x0<<15)|(0x0<<14)|(0x1<<8)|(sCh->m_eClockSource<<4) );
		Outp32( sCh->m_uBaseAddr+SDHC_CONTROL3, (0<<31)|(0<<23)|(1<<15)|(1<<7) );
	}
	// MMC : setup time : 5ns, hold time 5ns
	else if ( sCh->m_eCardType == SDHC_MMC_CARD || sCh->m_eCardType == SDHC_CE_ATA_CARD ) {
		Outp32( sCh->m_uBaseAddr+SDHC_CONTROL2, (1<<30)|(0x0<<15)|(0x0<<14)|(0x1<<8)|(sCh->m_eClockSource<<4) );
		Outp32( sCh->m_uBaseAddr+SDHC_CONTROL3, (0<<31)|(0<<23)|(0<<15)|(1<<7) );	// for movinand card.
	}
	else {
		Assert( "Not support card type");
	}
#endif

	// SDCLK Value Setting + Internal Clock Enable
	Outp16( sCh->m_uBaseAddr+SDHC_CLK_CTRL,
		(Inp16(sCh->m_uBaseAddr+SDHC_CLK_CTRL)&(~(0xff<<8)))|(uDivisor<<8)|(1<<0) );

	// CheckInternalClockStable
	while (!(Inp16( sCh->m_uBaseAddr+SDHC_CLK_CTRL )&0x2));

	SDHC_SetSdClockOnOff( TRUE, sCh);
//	UART_Printf("rHM_CONTROL2 = %x\n",Inp32( sCh->m_uBaseAddr+SDHC_CONTROL2 ));
//	UART_Printf("rHM_CLKCON = %x\n",Inp16( sCh->m_uBaseAddr+SDHC_CLK_CTRL ));
}

//////////
// File Name : SDHC_SetSdClockOnOff
// File Description : On/Off Sd Clock
// Input : On or Off, SDHC Channel
// Output : NONE.
void SDHC_SetSdClockOnOff(u8 uOnOff, SDHC* sCh)
{
//#ifdef FEATURE_SDHC_HIGH_SPEED_SUPPORT
	if (uOnOff == FALSE)
	{
		Outp16( sCh->m_uBaseAddr+SDHC_CLK_CTRL,
			Inp16(sCh->m_uBaseAddr+SDHC_CLK_CTRL)&(~(1<<2)) );	// SD Clock disable
	}
	else
//#endif
	{
		Outp16( sCh->m_uBaseAddr+SDHC_CLK_CTRL,
			Inp16(sCh->m_uBaseAddr+SDHC_CLK_CTRL)|(1<<2) );		// SD Clock enable
		while ( !( Inp16( sCh->m_uBaseAddr+SDHC_CLK_CTRL ) & (1<<3) ) );// SDHC_clockSource is Stable
	}
}

//////////
// File Name : SDHC_SetDataTransferWidth
// File Description : Set SD/MMC Host and Card data transfer width.
// Input : SDHC Channel
// Output : success or failure.
u8 SDHC_SetDataTransferWidth( SDHC* sCh)
{
	u32 uArg=0;

	SDHC_SetSdhcCardIntEnable(FALSE, sCh); // Disable sd card interrupt

	// bandwidth check
	if ( sCh->m_ucBandwidth == 8 ) {
		if ( sCh->m_eCardType == SDHC_SD_CARD ) {
			sCh->m_ucBandwidth = 4;
		}
	}
	else if ( !(sCh->m_ucBandwidth == 1||sCh->m_ucBandwidth == 4) )
		sCh->m_ucBandwidth = 1;		// default bandwidth = 1

	// SDHC_SD_CARD
	if ( sCh->m_eCardType == SDHC_SD_CARD ) {
		// Application Command.
		if (!SDHC_IssueCommand( sCh, 55, sCh->m_uRca<<16, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE) )
			return false;

		// ACMD6 - Set Bus Width. 0->1bit, 2->4bit
		if( !SDHC_IssueCommand( sCh, 6, (sCh->m_ucBandwidth==1)?(0):(2), SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE ) )
			return FALSE;
	}
	else if ( sCh->m_eCardType == SDHC_MMC_CARD || sCh->m_eCardType == SDHC_CE_ATA_CARD  ) {
		if (sCh->m_ucSpecVer >= 4) // It is for a newest MMC Card
		{
			uArg = (3<<24)|(183<<16);
			if (sCh->m_ucBandwidth==1)
				uArg |= (0<<8); //  1-bit bus
			else if (sCh->m_ucBandwidth==4)
				uArg |= (1<<8); //  4-bit bus
			else
				uArg |= (2<<8); // 8-bit bus
			// Refer to p.37~38, p.53~54 & p.83 of "MMC Card system Spec. ver4.0"
			// Refer to Lee's Spec and Add for 8-bit mode
			if( !SDHC_IssueCommand( sCh, 6, uArg, SDHC_CMD_AC_TYPE, SDHC_RES_R1B_TYPE ))
				return FALSE;
		}
		else
			sCh->m_ucBandwidth = 1;	// 1 bit bus.
	}


	Outp8( sCh->m_uBaseAddr+SDHC_HOST_CTRL,
		Inp8( sCh->m_uBaseAddr+SDHC_HOST_CTRL)& ~((1<<5)|(1<<1)) );
	if( sCh->m_ucBandwidth == 8 ) {
		Outp8( sCh->m_uBaseAddr+SDHC_HOST_CTRL,
			Inp8( sCh->m_uBaseAddr+SDHC_HOST_CTRL)|(1<<5) ); // 8 bit bus mode.
	}
	else if( sCh->m_ucBandwidth == 4 ) {
		Outp8( sCh->m_uBaseAddr+SDHC_HOST_CTRL,
			Inp8( sCh->m_uBaseAddr+SDHC_HOST_CTRL)|(1<<1) ); // 4 bit bus mode.
	}

	UART_Printf("Set Data width : %dbit\n",sCh->m_ucBandwidth);
	
	SDHC_SetSdhcCardIntEnable(TRUE, sCh);  // Enable sd card interrupt

	return TRUE;
}

//////////
// File Name : SDHC_SetHostCtrlSpeedMode
// File Description : Set SD/MMC Host Speed
// Input : Speed mode, SDHC Channel
// Output : NONE
void SDHC_SetHostOutEdgeMode(u32 outedge, SDHC* sCh)
{
// Host controller tx works at the failling edge.
	if ( outedge == 1)
	{
		// original
		//Outp8( sCh->m_uBaseAddr+SDHC_HOST_CTRL,
		//	Inp8(sCh->m_uBaseAddr+SDHC_HOST_CTRL)|(1<<2) );	// High Speed mode.
		// modify by HEAD
		Outp8( sCh->m_uBaseAddr+SDHC_HOST_CTRL,
			Inp8(sCh->m_uBaseAddr+SDHC_HOST_CTRL)&~(1<<2)|(1<<2) );	// Out Edge Inversion
	}
	
	else
	{
		Outp8( sCh->m_uBaseAddr+SDHC_HOST_CTRL,
			Inp8(sCh->m_uBaseAddr+SDHC_HOST_CTRL)&~(1<<2) );	// Normal 
	}
}

//////////
// File Name : SDHC_SetTransferModeReg
// File Description : Set host controller transfer mode.
// Input : multi block, read or write, auto command 12 on/off, block count enable or not, DMA enable or not, SDHC Channel
// Output : NONE
void SDHC_SetTransferModeReg(u32 MultiBlk, u32 DataDirection, u32 AutoCmd12En, u32 BlockCntEn, u32 DmaEn, SDHC* sCh)
{
	sCh->m_usTransMode = (u16)((MultiBlk<<5)|(DataDirection<<4)|(AutoCmd12En<<2)|(BlockCntEn<<1)|(DmaEn<<0));
	Outp16( sCh->m_uBaseAddr+SDHC_TRANS_MODE, sCh->m_usTransMode);
}

//////////
// File Name : SDHC_SetSdhcInterruptEnable
// File Description : Setting normal and error interrupt.
// Input : Normal interrupt, Error interrupt, SDHC
// Output : NONE.
void SDHC_SetSdhcInterruptEnable(u16 uNormalIntStatusEn, u16 uErrorIntStatusEn, u16 uNormalIntSigEn, u16 uErrorIntSigEn, SDHC* sCh)
{
	SDHC_ClearErrInterruptStatus(sCh);
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE, uNormalIntStatusEn);
	Outp16( sCh->m_uBaseAddr+SDHC_ERROR_INT_STAT_ENABLE, uErrorIntStatusEn);
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE, uNormalIntSigEn);
	Outp16( sCh->m_uBaseAddr+SDHC_ERROR_INT_SIGNAL_ENABLE, uErrorIntSigEn);
}

//////////
// File Name : SDHC_SetSdhcCardIntEnable
// File Description : Setting normal and error interrupt.
// Input : SDHC, Normal interrupt, Error interrupt.
// Output : NONE.
void SDHC_SetSdhcCardIntEnable(u8 ucTemp, SDHC* sCh)
{
	u16 usSfr;

	usSfr = Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE);
	usSfr = usSfr & 0xFEFF;
	usSfr |= (ucTemp<<8);
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE, usSfr );
}

//////////
// File Name : SDHC_SetSdCardSpeedMode
// File Description : Setting speed mode inside SD card.
// Input : SDHC, SDHC_SpeedMode, temp buffer for operating up to 512 byte.
// Output : NONE.
u8 SDHC_SetSdCardSpeedMode(SDHC_SpeedMode eSpeedMode, SDHC* sCh)
{
	u32 buffer[16];
	u32 uArg = 0;
	u32 ignore;

	// CMD16
	if(!SDHC_IssueCommand( sCh, 16, 64, SDHC_CMD_AC_TYPE, SDHC_RES_R1_TYPE ) ) {
		UART_Printf("CMD16 fail\n");
		return FALSE;
	}

	SDHC_SetBlockSizeReg(sCh, 7, 64);
	SDHC_SetBlockCountReg(sCh, 1);
	SDHC_SetTransferModeReg(0, 1, 0, 0, 0,sCh);
	sCh->m_uRemainBlock = 1;
	sCh->m_uBufferPtr = buffer;

	// High Speed = 1, Normal Speed = 0
	uArg = (0x1<<31)|(0xffff<<8)|(eSpeedMode);
	UART_Printf("Set SD Card to speed mode %d.\n",eSpeedMode);
	if( !SDHC_IssueCommand( sCh, 6, uArg, SDHC_CMD_ADTC_TYPE, SDHC_RES_R1_TYPE ) ) {
		UART_Printf("CMD6 fail\n");
		return FALSE;
	}

#ifdef FEATURE_SDHC_INTERRUPT_CONTROL_MODE
	while( sCh->m_uRemainBlock != 0 )
		ignore++;
#else
	SDHC_ReadOneBlock( sCh );
	//SDHC_INT_WAIT_CLEAR( sCh, 1, ignore ); // SDHC_TRANSFERCOMPLETE_STS_INT_EN
#endif
	SDHC_INT_WAIT_CLEAR( sCh, 1, ignore ); // SDHC_TRANSFERCOMPLETE_STS_INT_EN // original position. 2009.04.20
	
	if ( buffer[3] & (1<<9) ) { // Function Group 1 <- access mode.
		UART_Printf( "This Media support high speed mode.\n" );
	}
	else {
		UART_Printf( "This Media can't support high speed mode.\n" );
	}
	return TRUE;
}

//////////
// File Name : SDHC_SetMmcSpeedMode
// File Description : Setting speed mode inside MMC card.
// Input : SDHC, SDHC_SpeedMode
// Output : NONE.
u8 SDHC_SetMmcSpeedMode(SDHC_SpeedMode eSpeedMode, SDHC* sCh)
{
	UART_Printf("Set MMC Card to speed mode %d.\n",eSpeedMode);
	return SDHC_IssueCommand( sCh, 6, (3<<24)|(185<<16)|(eSpeedMode<<8), SDHC_CMD_AC_TYPE, SDHC_RES_R1B_TYPE);
}


//////////
// File Name : SDHC_DisplayCardInformation
// File Description : Display and set card CSD information after CSD command.
// Input : SDHC.
// Output : NONE.
void SDHC_DisplayCardInformation(SDHC * sCh)
{
	u32 CardSize, OneBlockSize,BlockCnt;
	
	if(sCh->m_eCardType == SDHC_MMC_CARD)
	{
		sCh->m_ucSpecVer=(Inp32( sCh->m_uBaseAddr+SDHC_RSP3 )>>18)& 0xF;
		
		UART_Printf("=>  m_ucSpecVer=%d\n", sCh->m_ucSpecVer);
	}

	sCh->m_sReadBlockLen = (u16)((Inp32( sCh->m_uBaseAddr+SDHC_RSP2 )>>8) & 0xf);
	sCh->m_sReadBlockPartial = (u16)((Inp32( sCh->m_uBaseAddr+SDHC_RSP2 )>>7) & 0x1);
	sCh->m_sCSize = (u16)(((Inp32( sCh->m_uBaseAddr+SDHC_RSP2 ) & 0x3) << 10) | ((Inp32( sCh->m_uBaseAddr+SDHC_RSP1 ) >> 22) & 0x3ff));
	sCh->m_sCSizeMult = (u16)((Inp32( sCh->m_uBaseAddr+SDHC_RSP1 )>>7)&0x7);
	
	CardSize = ((u32)(1<<sCh->m_sReadBlockLen))*(sCh->m_sCSize+1)*(1<<(sCh->m_sCSizeMult+2))/1048576;
	OneBlockSize = (1<<sCh->m_sReadBlockLen);
	SDHC_global_card_size = ((u32)(1<<sCh->m_sReadBlockLen))*(sCh->m_sCSize+1)*(1<<(sCh->m_sCSizeMult+2))/512;

	if(((Inp32( sCh->m_uBaseAddr+SDHC_RSP3 )>>22) & 0x3) == 1)
	{
		BlockCnt = ((Inp32( sCh->m_uBaseAddr+SDHC_RSP1 )>>8) & 0x3FFFFF);
		SDHC_global_card_size = (BlockCnt+1)<<10;
		UART_Printf("SDHC_global_card_size=%d\n", SDHC_global_card_size*512/1024/1024);
	}
	
	UART_Printf("READ_BL_LEN: %d\n",sCh->m_sReadBlockLen);
	UART_Printf("READ_BL_PARTIAL: %d\n",sCh->m_sReadBlockPartial);
	UART_Printf("C_SIZE: %d\n",sCh->m_sCSize);
	UART_Printf("C_SIZE_MULT: %d\n",sCh->m_sCSizeMult);

	UART_Printf("One Block Size: %dByte\n",OneBlockSize);
	UART_Printf("Total Card Size: %dMByte\n",CardSize+1);
}


//////////
// File Name : SDHC_SetGPIO
// File Description : Setting GPIO for 6400 high speed MMC.
// Input : Channel Number, Channel Line width.
// Output : success or failure.
u8 SDHC_SetGPIO(SDHC_channel channelNum, int channel_width)
{
	if( channelNum == SDHC_CHANNEL_0 )	// Channel 0 
	{

		GPIO_SetFunctionEach(eGPIO_G0, eGPIO_0, 0x2); // SD_0_CLK
		GPIO_SetFunctionEach(eGPIO_G0, eGPIO_1, 0x2); // SD_0_CMD
		GPIO_SetFunctionEach(eGPIO_G0, eGPIO_2, 0x2); // SD_0_CDn
	
		GPIO_SetFunctionEach(eGPIO_G0, eGPIO_3, 0x2); // SD_0_DATA[0]
		GPIO_SetFunctionEach(eGPIO_G0, eGPIO_4, 0x2); // SD_0_DATA[1]
		GPIO_SetFunctionEach(eGPIO_G0, eGPIO_5, 0x2); // SD_0_DATA[2]
		GPIO_SetFunctionEach(eGPIO_G0, eGPIO_6, 0x2); // SD_0_DATA[3]
	
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_3, 0x3); // SD_0_DATA[4]
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_4, 0x3); // SD_0_DATA[5]
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_5, 0x3); // SD_0_DATA[6]
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_6, 0x3); // SD_0_DATA[7]

	}

	if( channelNum == SDHC_CHANNEL_1 )	// Channel 1 
	{

		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_0, 0x2); // SD_1_CLK
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_1, 0x2); // SD_1_CMD
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_2, 0x2); // SD_1_CDn
	
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_3, 0x2); // SD_1_DATA[0]
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_4, 0x2); // SD_1_DATA[1]
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_5, 0x2); // SD_1_DATA[2]
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_6, 0x2); // SD_1_DATA[3]
	

	}
	if( channelNum == SDHC_CHANNEL_2 )	// Channel 2 
	{

		GPIO_SetFunctionEach(eGPIO_G2, eGPIO_0, 0x2); // SD_2_CLK
		GPIO_SetFunctionEach(eGPIO_G2, eGPIO_1, 0x2); // SD_2_CMD
		GPIO_SetFunctionEach(eGPIO_G2, eGPIO_2, 0x2); // SD_2_CDn
	
		GPIO_SetFunctionEach(eGPIO_G2, eGPIO_3, 0x2); // SD_2_DATA[0]
		GPIO_SetFunctionEach(eGPIO_G2, eGPIO_4, 0x2); // SD_2_DATA[1]
		GPIO_SetFunctionEach(eGPIO_G2, eGPIO_5, 0x2); // SD_2_DATA[2]
		GPIO_SetFunctionEach(eGPIO_G2, eGPIO_6, 0x2); // SD_2_DATA[3]
	
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_3, 0x3); // SD_2_DATA[4]
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_4, 0x3); // SD_2_DATA[5]
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_5, 0x3); // SD_2_DATA[6]
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_6, 0x3); // SD_2_DATA[7]

	}

	if( channelNum == SDHC_CHANNEL_3 )	// Channel 3 
	{

		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_0, 0x2); // SD_3_CLK
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_1, 0x2); // SD_3_CMD
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_2, 0x2); // SD_3_CDn
	
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_3, 0x2); // SD_3_DATA[0]
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_4, 0x2); // SD_3_DATA[1]
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_5, 0x2); // SD_3_DATA[2]
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_6, 0x2); // SD_3_DATA[3]
	

	}
#if 0
	// CLK pull up down disable
	GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_0, eGPUDdis); 	// SD_0_CLK
	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_0, eGPUDdis); 	// SD_1_CLK
	GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_0, eGPUDdis); 	// SD_2_CLK
	GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_0, eGPUDdis); 	// SD_3_CLK	

	//  ch0 pull up
	GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_0, eGPUDdis); 	// SD_0_CLK
	GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_1, eGPUen); 		// SD_0_CMD
	GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_2, eGPUen);		// SD_0_CDn
	
	GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_3, eGPUen); 		// SD_0_DATA[0]
	GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_4, eGPUen);		// SD_0_DATA[1]
	GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_5, eGPUen);		// SD_0_DATA[2]
	GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_6, eGPUen);		// SD_0_DATA[3]
	
//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_3, eGPUen); 		// SD_0_DATA[4]
//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_4, eGPUen);		// SD_0_DATA[5]
//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_5, eGPUen);		// SD_0_DATA[6]	
//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_6, eGPUen);		// SD_0_DATA[7]	

	//  ch1 pull up
	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_0, eGPUDdis); 	// SD_1_CLK
	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_1, eGPUen); 		// SD_1_CMD
	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_2, eGPUen);		// SD_1_CDn
	
	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_3, eGPUen); 		// SD_1_DATA[0]
	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_4, eGPUen);		// SD_1_DATA[1]
	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_5, eGPUen);		// SD_1_DATA[2]
	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_6, eGPUen);		// SD_1_DATA[3]


	// ch2 pull up
	GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_0, eGPUDdis); 	// SD_2_CLK
	GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_1, eGPUen);		// SD_2_CMD
	GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_2, eGPUen);		// SD_2_CDn	

	GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_3, eGPUen);  		// SD_2_DATA[0]
	GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_4, eGPUen);		// SD_2_DATA[1]
	GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_5, eGPUen);		// SD_2_DATA[2]	
	GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_6, eGPUen);		// SD_2_DATA[3]	

	// ch3 pull up
	GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_0, eGPUDdis); 	// SD_3_CLK
	GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_1, eGPUen);		// SD_3_CMD
	GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_2, eGPUen);		// SD_3_CDn	

	GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_3, eGPUen);  		// SD_3_DATA[0]
	GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_4, eGPUen);		// SD_3_DATA[1]
	GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_5, eGPUen);		// SD_3_DATA[2]	
	GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_6, eGPUen);		// SD_3_DATA[3]	


	UART_Printf("SD GPIO is set to Pull-up\n");

	////////   drive strength  MAX   ////  
//	GPIO_SetDRVRegAll(eGPIO_MP0_1, 0xffff);
//	GPIO_SetDRVRegAll(eGPIO_MP0_2, 0xffff);
//	GPIO_SetDRVRegAll(eGPIO_MP0_3, 0xffff);
//	GPIO_SetDRVRegAll(eGPIO_MP0_4, 0xffff);
#endif	

	return TRUE;
}
//////////
// File Name : SDHC_CMD_LOW
// File Description : Setting CMD to GPIO and LOW --> HIGH
// Input : Channel Number
// Output : void
void SDHC_CMD_LO_HI(SDHC_channel channelNum)
{
	if( channelNum == SDHC_CHANNEL_0 )	// Channel 0 
	{

		GPIO_SetFunctionEach(eGPIO_G0, eGPIO_1, 0x1); // output
		GPIO_SetDataEach(eGPIO_G0, eGPIO_1, 0x0);
		Delay(10);
		GPIO_SetDataEach(eGPIO_G0, eGPIO_1, 0x1);
		
		GPIO_SetFunctionEach(eGPIO_G0, eGPIO_1, 0x2); // SD_0_CMD

	}

	if( channelNum == SDHC_CHANNEL_1 )	// Channel 1 
	{
		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_1, 0x1); // output
		GPIO_SetDataEach(eGPIO_G1, eGPIO_1, 0x0);
		Delay(10);
		GPIO_SetDataEach(eGPIO_G1, eGPIO_1, 0x1);

		GPIO_SetFunctionEach(eGPIO_G1, eGPIO_1, 0x2); // SD_1_CMD
	

	}
	if( channelNum == SDHC_CHANNEL_2 )	// Channel 2 
	{
		GPIO_SetFunctionEach(eGPIO_G2, eGPIO_1, 0x1); // output
		GPIO_SetDataEach(eGPIO_G2, eGPIO_1, 0x0);
		Delay(10);
		GPIO_SetDataEach(eGPIO_G2, eGPIO_1, 0x1);
		
		GPIO_SetFunctionEach(eGPIO_G2, eGPIO_1, 0x2); // SD_2_CMD

	}

	if( channelNum == SDHC_CHANNEL_3 )	// Channel 3 
	{
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_1, 0x1); // output
		GPIO_SetDataEach(eGPIO_G3, eGPIO_1, 0x0);
		Delay(10);
		GPIO_SetDataEach(eGPIO_G3, eGPIO_1, 0x1);
		
		GPIO_SetFunctionEach(eGPIO_G3, eGPIO_1, 0x2); // SD_3_CMD

	}


	
}

u8 SDHC_GPIO_PullUPDown(u32 ePuD)
{
//	u32 temp;

	// CLK pull up down disable
	GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_0, eGPUDdis); 	// SD_0_CLK
	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_0, eGPUDdis); 	// SD_1_CLK
	GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_0, eGPUDdis); 	// SD_2_CLK
	GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_0, eGPUDdis); 	// SD_3_CLK	

//	UART_Printf("0. pull updown disable  1. pull up \n");
//      temp =	UART_GetIntNum(void);
	   

	
	if(ePuD == eGPUDdis)
	{
		//  ch0 pull up
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_0, eGPUDdis); 	// SD_0_CLK
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_1, eGPUDdis); 		// SD_0_CMD
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_2, eGPUDdis);		// SD_0_CDn
		
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_3, eGPUDdis); 		// SD_0_DATA[0]
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_4, eGPUDdis);		// SD_0_DATA[1]
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_5, eGPUDdis);		// SD_0_DATA[2]
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_6, eGPUDdis);		// SD_0_DATA[3]
		
	//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_3, eGPUDdis); 		// SD_0_DATA[4]
	//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_4, eGPUDdis);		// SD_0_DATA[5]
	//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_5, eGPUDdis);		// SD_0_DATA[6]	
	//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_6, eGPUDdis);		// SD_0_DATA[7]	

		//  ch1 pull up
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_0, eGPUDdis); 	// SD_1_CLK
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_1, eGPUDdis); 		// SD_1_CMD
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_2, eGPUDdis);		// SD_1_CDn
		
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_3, eGPUDdis); 		// SD_1_DATA[0]
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_4, eGPUDdis);		// SD_1_DATA[1]
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_5, eGPUDdis);		// SD_1_DATA[2]
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_6, eGPUDdis);		// SD_1_DATA[3]


		// ch2 pull up
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_0, eGPUDdis); 	// SD_2_CLK
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_1, eGPUDdis);		// SD_2_CMD
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_2, eGPUDdis);		// SD_2_CDn	

		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_3, eGPUDdis);  		// SD_2_DATA[0]
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_4, eGPUDdis);		// SD_2_DATA[1]
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_5, eGPUDdis);		// SD_2_DATA[2]	
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_6, eGPUDdis);		// SD_2_DATA[3]	

		// ch3 pull up
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_0, eGPUDdis); 	// SD_3_CLK
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_1, eGPUDdis);		// SD_3_CMD
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_2, eGPUDdis);		// SD_3_CDn	

		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_3, eGPUDdis);  		// SD_3_DATA[0]
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_4, eGPUDdis);		// SD_3_DATA[1]
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_5, eGPUDdis);		// SD_3_DATA[2]	
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_6, eGPUDdis);		// SD_3_DATA[3]	
		UART_Printf("Internal Pull-up is disabled..\n");

	}
	else
	{
		//  ch0 pull up
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_0, eGPUDdis); 	// SD_0_CLK
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_1, eGPUen); 		// SD_0_CMD
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_2, eGPUen);		// SD_0_CDn
		
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_3, eGPUen); 		// SD_0_DATA[0]
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_4, eGPUen);		// SD_0_DATA[1]
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_5, eGPUen);		// SD_0_DATA[2]
		GPIO_SetPullUpDownEach(eGPIO_G0, eGPIO_6, eGPUen);		// SD_0_DATA[3]
		
	//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_3, eGPUen); 		// SD_0_DATA[4]
	//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_4, eGPUen);		// SD_0_DATA[5]
	//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_5, eGPUen);		// SD_0_DATA[6]	
	//	GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_6, eGPUen);		// SD_0_DATA[7]	

		//  ch1 pull up
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_0, eGPUDdis); 	// SD_1_CLK
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_1, eGPUen); 		// SD_1_CMD
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_2, eGPUen);		// SD_1_CDn
		
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_3, eGPUen); 		// SD_1_DATA[0]
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_4, eGPUen);		// SD_1_DATA[1]
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_5, eGPUen);		// SD_1_DATA[2]
		GPIO_SetPullUpDownEach(eGPIO_G1, eGPIO_6, eGPUen);		// SD_1_DATA[3]


		// ch2 pull up
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_0, eGPUDdis); 	// SD_2_CLK
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_1, eGPUen);		// SD_2_CMD
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_2, eGPUen);		// SD_2_CDn	

		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_3, eGPUen);  		// SD_2_DATA[0]
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_4, eGPUen);		// SD_2_DATA[1]
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_5, eGPUen);		// SD_2_DATA[2]	
		GPIO_SetPullUpDownEach(eGPIO_G2, eGPIO_6, eGPUen);		// SD_2_DATA[3]	

		// ch3 pull up
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_0, eGPUDdis); 	// SD_3_CLK
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_1, eGPUen);		// SD_3_CMD
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_2, eGPUen);		// SD_3_CDn	

		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_3, eGPUen);  		// SD_3_DATA[0]
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_4, eGPUen);		// SD_3_DATA[1]
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_5, eGPUen);		// SD_3_DATA[2]	
		GPIO_SetPullUpDownEach(eGPIO_G3, eGPIO_6, eGPUen);		// SD_3_DATA[3]	

		UART_Printf("Internal Pull-up is enabled..\n");

	}
	

}

void SDHC_GPIO_DrvStr(void)
{
	u32 drv_str,slew,i,drv_str_reg,slew_reg;


//	UART_Printf("Driving strength ? 0. 1x   1. 3x    2. 2x   3. 4x    \n");
//       drv_str =	UART_GetIntNum();
       drv_str =	3;
	   
//	UART_Printf("Slew rate ? 0. Fast   1. Slow \n");
//       slew =	UART_GetIntNum();
       slew =	0;

	drv_str_reg=0;
 	for(i = 0;i<8;i++)
	drv_str_reg =	(drv_str<<(i<<1))|drv_str_reg ;	

	slew_reg=0;
 	for(i = 0;i<8;i++)
	slew_reg = (slew<<(i+16))|slew_reg ;	

	Outp32(0xE02001AC,(slew_reg|drv_str_reg));  		// SD 0
	Outp32(0xE02001CC,(slew_reg|drv_str_reg));  		// SD 1
	Outp32(0xE02001EC,(slew_reg|drv_str_reg));  		// SD 2
	Outp32(0xE020020C,(slew_reg|drv_str_reg));  		// SD 3	
   //	GPIO_SetDRVRegAll(eGPIO_G0,(slew_reg)|(drv_str_reg));		// SD 0
   //	GPIO_SetDRVRegAll(eGPIO_G1,(slew_reg)|(drv_str_reg));		// SD 1
   //	GPIO_SetDRVRegAll(eGPIO_G2,(slew_reg)|(drv_str_reg));	 	// SD 2
   //	GPIO_SetDRVRegAll(eGPIO_G3,(slew_reg)|(drv_str_reg));		// SD 3
	
}
void SDHC_WakeUp_Init(SDHC_channel eSDch,SDHC *sCh)
{

	SDHC_SetGPIO(sCh->m_eChannel, sCh->m_ucBandwidth);
	
	//SDHC_GPIO_DrvStr();
	Outp32(0xE02001AC,0xffff);  		// SD 0
	Outp32(0xE02001CC,0xffff);  		// SD 1
	Outp32(0xE02001EC,0xffff);  		// SD 2
	Outp32(0xE020020C,0xffff);  		// SD 3	
	
	sCh->m_ucBandwidth = 4;
	sCh->m_eChannel = eSDch;
	sCh->m_uClockDivision = 2;

	SDHC_SetGPIO(sCh->m_eChannel, sCh->m_ucBandwidth);
	
	SDHC_ResetController(sCh);
	SDHC_InitCh(sCh->m_eChannel, sCh);
	SDHC_SetSdClockOnOff(0, sCh); // If the sd clock is to be changed, you need to stop sd-clock.
	SDHC_SetSdClock(sCh->m_uClockDivision, sCh, 0,24000000);
	
	Outp8(sCh->m_uBaseAddr + SDHC_WAKEUP_CTRL, 0xF);  // wakeup clear and enable
	

	///  interrupt  status enable
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_STAT_ENABLE)) 
		        |SDHC_CARD_SIG_INT_EN|SDHC_CARD_INSERT_SIG_INT_EN|SDHC_CARD_REMOVAL_SIG_INT_EN);


	//// interrupt signal enable
	Outp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE,
		        (Inp16( sCh->m_uBaseAddr+SDHC_NORMAL_INT_SIGNAL_ENABLE)) 
		        |SDHC_CARD_SIG_INT_EN|SDHC_CARD_INSERT_SIG_INT_EN|SDHC_CARD_REMOVAL_SIG_INT_EN);

	

	INTC_SetVectAddr( sCh->m_ucIntChannelNum, sCh->m_fIntFn);  ///   
	INTC_Enable( sCh->m_ucIntChannelNum );


}




