/**
 * File Name : i2c_hdmi_phy.c
 *
 * File Description :
 * This file implements the I2C function for communication between the hdmi link layer and phy layer
 *
 * Author : Hee Myung Noh
 * Dept : AP Development
 * Created Date : 2009/01/22
 * Version : 0.1
 * History 
 *  - Initiate version (Hee Myung Noh 090122)
 */

#include "system.h"
#include "Library.h"
#include "v210_sfr.h"
#include "intc.h"
//#include "phy.h"
#include "i2c_pmic.h"
#include "gpio.h"
#include "iic.h"

//#define	DISP_I2C_pmic_PHY_REG

#ifndef DISP_I2C_pmic_PHY_REG
#define I2C_pmicOutp8(addr, data) 	Outp8(addr, data)
#define I2C_pmicInp8(addr)  			Inp8(addr)
#else
#define I2C_pmicOutp8(addr, data) 	{UART_Printf("Outp8(\'h%08x, \'h%08x);\n", addr, data); Outp8(addr, data);}
#define I2C_pmicInp8(addr) 			Inp8(addr)
#endif


static struct {
	s32    state;
	u8 	*buffer;
	s32    bytes;
} i2c_pmic_context;

typedef enum I2C_pmic_SFR
{
	rI2C_pmic_CON		=	(I2C2_BASE + 0x00),
	rI2C_pmic_STAT		=	(I2C2_BASE + 0x04),
	rI2C_pmic_ADD		=	(I2C2_BASE + 0x08),
	rI2C_pmic_DS		=	(I2C2_BASE + 0x0c),
	rI2C_pmic_LC		=	(I2C2_BASE + 0x10)
}I2C_pmic_SFR;


static s32 I2Cpmic_Release(void);

/**
 * Polling interrupt 
 *
 * @return  On success, return OK;Otherwise return, ERROR.
 */
static s32 I2Cpmic_InterruptWait(void)
{
	u8 ucStatus, ucReg;
	s32 retval = OK;
	u8 i=0;

	do
	{
		ucStatus = I2C_pmicInp8(rI2C_pmic_CON);
		if (ucStatus & I2C_PEND)
		{
			// check status flags
			ucReg = I2C_pmicInp8(rI2C_pmic_STAT);

			if (ucReg & (I2C_STAT_ARBITRERR | I2C_STAT_ADDRASSLAVE |I2C_STAT_ADDRZERO | I2C_STAT_ACKERROR))
			{
				if (ucReg & I2C_STAT_ARBITRERR)
				{
					UART_Printf("I2C_STAT_ARBITRERR\n");
				}
				else if (ucReg & I2C_STAT_ADDRASSLAVE)
				{
					UART_Printf("I2C_STAT_ADDRASSLAVE\n");
				}
				else if (ucReg & I2C_STAT_ADDRZERO)
				{
					UART_Printf("I2C_STAT_ADDRZERO\n");
				}
				else if (ucReg & I2C_STAT_ACKERROR)
				{
					UART_Printf("I2C_STAT_ACKERROR\n");
				}
				I2Cpmic_Release();
				retval = ERROR;
			}
			break;
		}
	}while (1);

	return retval;
}

/**
 *  Initialize I2C library
 */
s32 I2Cpmic_Init(void)
{
	// I2C CH2 GPIO Setting
	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_4, 2);	// GPIO I2C setting
	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_5, 2);
	
	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_4, 0);	// Pull-Up/Down Disable
	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_5, 0);	// Pull-Up/Down Disable
	
	// I2C Filter Setting
	I2C_pmicOutp8(rI2C_pmic_LC, 0x5);

	GPIO_SetFunctionEach(eGPIO_H1, eGPIO_6, 1);
	GPIO_SetFunctionEach(eGPIO_H1, eGPIO_7, 1); // PMIC Set0,1 Pin(xeint14,15 output setting)

	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_4, 1); // PMIC Set2 Pin(xeint4 output setting)
	
	return OK;
}

/**
* {I2Cpmic_DeInit}
*
* @brief			Init 
* 
* @param[in]	(u32 ch)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2Cpmic_DeInit( void )
{
	I2C_Outp8(rI2C_pmic_STAT, 0);// I2C bus data output disable(Rx/Tx)

	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_4, 0);
	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_5, 0);
	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_4, 0);
	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_5, 0);

	GPIO_SetFunctionEach(eGPIO_H1, eGPIO_6, 0);
	GPIO_SetFunctionEach(eGPIO_H1, eGPIO_7, 0); // PMIC Set0,1 Pin(xeint14,15 output setting)
	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_4, 0); // PMIC Set2 Pin(xeint4 output setting)	
	
	return OK;
}

/**
 * Read data through I2C
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
s32 I2Cpmic_Read(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	u32 i, uProcess = TRUE;

#if 1
	u32 uTmp1 = 0;
	
	uTmp1 = I2C_Inp32(rI2C_pmic_STAT);
	
	while( (uTmp1 & I2C_START ) || (uTmp1 & I2C_STAT_ARBITRERR) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp1 = I2C_pmicInp8(rI2C_pmic_STAT);
	}
#endif

	// set state
	i2c_pmic_context.state = STATE_RX_ADDR;

	// set parameters
	i2c_pmic_context.buffer = pBuffer;
	i2c_pmic_context.bytes = ucBytes;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_ACK); // 0xE1
	// to make IDSR write-enabled
	I2C_pmicOutp8(rI2C_pmic_STAT, I2C_ENABLE | I2C_MODE_MRX);
	// put address to shift register
	I2C_pmicOutp8(rI2C_pmic_DS, ucAddr & 0xFE);
	// master rx mode + start + enable - "START" sequence generation
	I2C_pmicOutp8(rI2C_pmic_STAT, I2C_ENABLE | I2C_START | I2C_MODE_MRX); // 0xB0

	while (uProcess)
	{
		//In case of Master RX Mode, Last data is not ack. but set the "ACK not received" bit
		if(i2c_pmic_context.state != STATE_RX_STOP)
		{
			if (I2Cpmic_InterruptWait() != OK)
			{
				UART_Printf("InterruptWait() failed!!!\n");
				iRetval = ERROR;
				break;
			}
		}

		switch (i2c_pmic_context.state)
		{
			case STATE_RX_DATA:
				// read byte
				*(i2c_pmic_context.buffer) = I2C_pmicInp8(rI2C_pmic_DS);
				i2c_pmic_context.buffer++;
				--(i2c_pmic_context.bytes);
				            
				if (i2c_pmic_context.bytes == 1)
				{
					i2c_pmic_context.state = STATE_RX_STOP;

					// clear interrupt pending flag and interrupt clear bit
					I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);    
				}
				else
				{
					// clear interrupt pending flag and interrupt clear bit
					I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK);    
				}
				break;

			case STATE_RX_ADDR:
				i2c_pmic_context.state = STATE_RX_DATA;

				if (i2c_pmic_context.bytes == 1)
				{
					i2c_pmic_context.state = STATE_RX_STOP;

					// clear interrupt pending flag and interrupt clear bit
					I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);  
				}
				else
				{
					// clear interrupt pending flag and interrupt clear bit
					I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK);    
				}
				break;

			case STATE_RX_STOP:
				i2c_pmic_context.state = STATE_IDLE;

				Delay(35);
				
				*(i2c_pmic_context.buffer) = I2C_pmicInp8(rI2C_pmic_DS);
				
				//"STOP" sequence generation
				I2C_pmicOutp8(rI2C_pmic_STAT, I2C_MODE_MRX | I2C_ENABLE); 
				// clear interrupt pending flag and interrupt clear bit
				I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);

				while( I2C_pmicInp8(rI2C_pmic_STAT) & I2C_START );	//Wait until the stop condition takes effect
				uProcess = FALSE;
				break;

			case STATE_IDLE:
			default:
				UART_Printf("error state!!!\n");
				iRetval = ERROR;
				uProcess = FALSE;
				break;
		}
	}

	return iRetval;
}

/**
 * Write data through I2C
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
 s32 I2CPMIC_Write(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	u32 uProcess = TRUE;

	u8 uTmp = 0;
	
#if 1
	u32 uTmp1 = 0;
	
	uTmp1 = I2C_Inp32(rI2C_pmic_STAT);
	
	while( (uTmp1 & I2C_START ) || (uTmp1 & I2C_STAT_ARBITRERR) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp1 = I2C_pmicInp8(rI2C_pmic_STAT);
	}
#endif

	// set state
	i2c_pmic_context.state = STATE_TX_ADDR;

	// set parameters
	i2c_pmic_context.buffer = pBuffer;
	i2c_pmic_context.bytes = ucBytes;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_ACK); // 0xE1
	// to make IDSR write-enabled
	I2C_pmicOutp8(rI2C_pmic_STAT, I2C_ENABLE | I2C_MODE_MTX);
	// put address to shift register
	I2C_pmicOutp8(rI2C_pmic_DS, ucAddr & 0xFE);
	// master tx mode + start + enable - "START" sequence generation
	uTmp = I2C_pmicInp8(rI2C_pmic_STAT);
	I2C_pmicOutp8(rI2C_pmic_STAT, uTmp | I2C_START); // 0xF0
	
	while (uProcess)
	{
		if (I2Cpmic_InterruptWait() != OK)
		{
			UART_Printf("I2CpmicInterruptWait() failed!!!\n");
			iRetval = ERROR;
		
			break;
		}

	
		switch (i2c_pmic_context.state)
		{
			case STATE_TX_ADDR:
			case STATE_TX_DATA:
				i2c_pmic_context.state = STATE_TX_DATA;

				I2C_pmicOutp8(rI2C_pmic_DS, *(i2c_pmic_context.buffer));
				
				i2c_pmic_context.buffer++;
				--(i2c_pmic_context.bytes);

				if (i2c_pmic_context.bytes == 0)
				{
					i2c_pmic_context.state = STATE_TX_STOP;
					I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
				}
				else
				{
					// clear interrupt pending flag and interrupt clear bit
					I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR | I2C_ACK);
				}
				break;

			case STATE_TX_STOP:
				i2c_pmic_context.state = STATE_IDLE;

				// send stop bit
				I2C_pmicOutp8(rI2C_pmic_STAT, I2C_MODE_MTX | I2C_ENABLE); 
				
				// clear interrupt pending flag and interrupt clear bit
				I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);
				
				while( I2C_pmicInp8(rI2C_pmic_STAT) & I2C_START );
				
				uProcess = FALSE;
				break;

			case STATE_IDLE:
			default:
				UART_Printf("error state!!!\n");
				iRetval = ERROR;
				uProcess = FALSE;
				break;
		}
	}

	return iRetval;
}

/**
 * Release I2C line
 */
static s32 I2Cpmic_Release(void)
{
	// clear interrupt pending flag and interrupt clear bit
	I2C_pmicOutp8(rI2C_pmic_CON, I2C_CLK | I2C_INT | I2C_INT_CLEAR);

	// disable 
	I2C_pmicOutp8(rI2C_pmic_STAT, I2C_IDLE);

	return OK;
}
