//===================================================================
// NAME		: modemif_test.c	
// DESC		: S3C6410 DPSRAM test 
// History	: 01.03.2007 ver 0.0 by JAE GEUN. SONG
// 			: 03.06.2008 ver 0.1 by Sun il, Roe	for S3C6410
//			:				Modem DMA functions are added.
//			:				Modem DMA means DMA transaction between DPSRAM and DRAM in AP side.
//===================================================================

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "option.h"
#include "v210_sfr.h"
#include "def.h"
#include "intc.h"
#include "library.h"
#include "sysc.h"
#include "modemif.h"
#include "gpio.h"
#include "timer.h"
#include "dma.h"

/*******************************************************************************
* GLOBAL VARIABLES
*******************************************************************************/
DMAC oAPTx0;
DMAC oAPRx0;

u8	ucModemDMAPort;

volatile int	g_DmaDoneMTx;
volatile int	g_DmaDoneMRx;
volatile int	g_DmaDoneATx;
volatile int	g_DmaDoneARx;


REGINFO	sRegInfoModemIF[] = 
{
	// MODEM IF REG
	{"rINT2AP	",		MODEMIF_BASE + 0x8000, 32,	RW, DPPB, 0x00003FFE, 0x00003FFE},
	{"rINT2MSM	", 		MODEMIF_BASE + 0x8004, 32,	RW, DPPB, 0x00003FFC, 0x00003FFC},
	{"rMIFCON	", 		MODEMIF_BASE + 0x8008, 32,	RW, DPPB, 0x001F000C, 0x00100008},
	{"rMIFPCON	",		MODEMIF_BASE + 0x800C, 32,	RW, DPPB, 0x00000050, 0x00000000}, 
	{"rMSMINTCLR	", 	MODEMIF_BASE + 0x8010, 32,	WO, DPPB, 0x00000000, 0x00000000},
	{"rDMA_TX_ADR ", 	MODEMIF_BASE + 0x8014, 32, 	RW, DPPB, 0x3FFF3FFF, 0x17FE13FE},
	{"rDMA_RX_ADR ",	MODEMIF_BASE + 0x8018, 32, 	RW, DPPB, 0x3FFF3FFF, 0x1FFE1BFE},
};

/*******************************************************************************
* EXTERNS
*******************************************************************************/
extern volatile int g_ApDataCnt;
extern volatile int g_MsmDataCnt;

extern volatile int g_APDMAReadReady;		// DMA Write Done flag of AP side
extern volatile int g_ApDMAReadReadyReply;
extern volatile int g_APDMAReadDone;			// DMA Read Done flag of AP side
extern volatile int g_ApDMAReadDoneReply;

extern volatile int g_ModemDMAReadReady;			// Read Done flag of Modem side
extern volatile int g_ModemDMAReadReadyReply;
extern volatile int g_ModemDMAReadDone;			// Read Done flag of Modem side
extern volatile int g_ModemDMAReadDoneReply;

extern volatile int g_APDMAWriteReady;		// DMA Write Done flag of AP side
extern volatile int g_ApDMAWriteReadyReply;
extern volatile int g_APDMAWriteDone;		// DMA Write Done flag of AP side
extern volatile int g_ApDMAWriteDoneReply;

extern volatile int g_ModemDMAWriteReady;
extern volatile int g_ModemDMAWriteReadyReply;
extern volatile int g_ModemDMAWriteDone;		// Write Done flag of Modem side
extern volatile int g_ModemDMAWriteDoneReply;

extern volatile int g_ApReadyReply;	
extern volatile int g_APReadDone;				// Read Done flag of AP side
extern volatile int g_ApReadDoneReply;

extern volatile int g_ModemReadyReply;		// Modem Ready Reply flag from Modem to AP
extern volatile int g_ModemReadDone;
extern volatile int g_ModemReadDoneReply;	// Read Done Reply flag from Modem to AP

extern volatile int g_ApWriteReadyReply;
extern volatile int g_ApWriteDone;
extern volatile int g_ApWriteDoneReply;

extern volatile int g_ModemWriteReadyReply;
extern volatile int g_ModemWriteDone;
extern volatile int g_ModemWriteDoneReply;

/*******************************************************************************
* FUNCTION PROTOTYPES
*******************************************************************************/
//////////
// Function Name : TestSFRModemIF
// Function Description : This function test pattern for IIC SFR write/read
// Input : NONE
// Output : None
// Version : v0.0
void TestSFRModemIF(void)	
{
	volatile u32 *pAddr;	
	s32 i,j, count = sizeof(sRegInfoModemIF)/sizeof(REGINFO), nPattern;
	u32 uRegStatus, uWritePatternWithMask, uReadWithMask;	
	
	u32 uPatternShitfOneBit[] = 
	{
		0x00000001,		0x00000002,		0x00000004,		0x00000008,		0x00000010,		0x00000020,		0x00000040,		0x00000080,
		0x00000100,		0x00000200,		0x00000400,		0x00000800,		0x00001000,		0x00002000,		0x00004000,		0x00008000,
		0x00010000,		0x00020000,		0x00040000,		0x00080000,		0x00100000,		0x00200000,		0x00400000,		0x00800000,
		0x01000000,		0x02000000,		0x04000000,		0x08000000,		0x10000000,		0x20000000,		0x40000000,		0x80000000,
	};
	
	u32 uDefaultBitMask[] =
	{
		0x00000001,		0x00000003,		0x00000007,		0x0000000F,		0x0000001F,		0x0000003F,		0x0000007F,		0x000000FF,
		0x000001FF,		0x000003FF,		0x000007FF,		0x00000FFF,		0x00001FFF,		0x00003FFF,		0x00007FFF,		0x0000FFFF,
		0x0001FFFF,		0x0003FFFF,		0x0007FFFF,		0x000FFFFF,		0x001FFFFF,		0x003FFFFF,		0x007FFFFF,		0x00FFFFFF,
		0x01FFFFFF,		0x03FFFFFF,		0x07FFFFFF,		0x0FFFFFFF,		0x1FFFFFFF,		0x3FFFFFFF,		0x7FFFFFFF,		0xFFFFFFFF,
	};

	
	for(i=0; i<count; i++)
	{		
		pAddr = (u32*)sRegInfoModemIF[i].uAddr;

		if(sRegInfoModemIF[i].uFlag == DPDB)
		{		
			if(sRegInfoModemIF[i].uRWType == RW)
			{
				for(j=0; j< sRegInfoModemIF[i].uBitLen ; j++)
				{			
					uWritePatternWithMask = (uPatternShitfOneBit[j] & uDefaultBitMask[j]);
					*pAddr = uWritePatternWithMask;					
					
					uReadWithMask = (*pAddr & uDefaultBitMask[j]);	
					sRegInfoModemIF[i].rValue = uReadWithMask;

					uRegStatus = (uWritePatternWithMask != uReadWithMask );					
					RPRINTF(sRegInfoModemIF[i], uPatternShitfOneBit[j], uRegStatus, j, 1);

					/* After the register is tested, reset the register for the next register test */
					*pAddr = 0x0;
				}
			}
		}
		else if(sRegInfoModemIF[i].uFlag == DPPB)
		{		
			nPattern = (sizeof(uPatternShitfOneBit) / sizeof(uPatternShitfOneBit[0]));			
			if(sRegInfoModemIF[i].uRWType == RW)
			{
				for(j=0; j<nPattern; j++)
				{
					if(uPatternShitfOneBit[j] & sRegInfoModemIF[i].uPrivateBitMask)
					{
						uWritePatternWithMask = (uPatternShitfOneBit[j] & sRegInfoModemIF[i].uPrivateBitMask);
						*pAddr = uWritePatternWithMask;
						
						uReadWithMask = (*pAddr & sRegInfoModemIF[i].uPrivateBitMask);
						sRegInfoModemIF[i].rValue = uReadWithMask;
						
						uRegStatus = (uWritePatternWithMask != uReadWithMask );												
						RPRINTF(sRegInfoModemIF[i], uPatternShitfOneBit[j], uRegStatus, j, 1);

						/* After the register is tested, reset the register for the next register test */
						*pAddr = 0x0;
					}
				}
			}
		}
	}
	
	UART_Printf("\n\n[MODEM IF SFR Read/Write Test Complete!]\n\n");
	
	return ;
}

//AP DMA ISR
void __irq Dma0Done_APDMATx(void)
{
	// for S5PV210
	u32 uDmaIntStatus = 0;
 
	INTC_Disable(NUM_PDMA1);

	MODEM_PRINTF("[AP DMA TX INT]\n");
	
	// Get DMA INT Num among 32 interrupt sources
	DMA_GetIntrSrc(&uDmaIntStatus, &oAPTx0);
	if ( uDmaIntStatus == 32 )	// ERROR
	{
		MODEM_PRINTF("[AP DMA TX FAULT]\n");
		DMA_StopDmac(&oAPTx0);
	}
	else if ( uDmaIntStatus == 0 )	// OK
	{
		// Clear DMA Int pending bits
		DMA_ClearIntPending(uDmaIntStatus, &oAPTx0);

		g_APDMAWriteReady = 1;
	}

	INTC_ClearVectAddr();
	
	INTC_Enable(NUM_PDMA1);
}

//AP DMA ISR
void __irq Dma0Done_APDMARx(void)
{
	// for S5PV210
	u32 uDmaIntStatus = 0;

	INTC_Disable(NUM_PDMA1);
	
	MODEM_PRINTF("[AP DMA RX INT]\n");
		
	DMA_GetIntrSrc( &uDmaIntStatus, &oAPRx0);
	if ( uDmaIntStatus == 32 )	// ERROR
	{
		MODEM_PRINTF("[AP DMA TX FAULT]\n");
		DMA_StopDmac(&oAPTx0);
	}
	else if ( uDmaIntStatus == 0 )	// OK
	{		
	  	DMA_ClearIntPending(uDmaIntStatus, &oAPRx0);

		g_APDMAReadReady = 1;
	}
	
	INTC_ClearVectAddr();

	INTC_Enable(NUM_PDMA1);
}

void __irq Isr_AP_DMA_TX_Done(void)
{
	// for S5PV210
	u32 uDmaIntStatus = 0;
 
	INTC_Disable(NUM_PDMA1);

//	MODEM_PRINTF("[AP DMA TX INT]\n");
		
	// Get DMA INT Num among 32 interrupt sources
	DMA_GetIntrSrc(&uDmaIntStatus, &oAPTx0);
	if ( uDmaIntStatus == 32 )	// ERROR
	{
		MODEM_PRINTF("[AP DMA TX FAULT]\n");
		DMA_StopDmac(&oAPTx0);
	}
	else if ( uDmaIntStatus == 0 )	// OK
	{
		// Clear DMA Int pending bits
		DMA_ClearIntPending(uDmaIntStatus, &oAPTx0);
		g_APDMAWriteReady = 1;
	}

	INTC_ClearVectAddr();
	
	INTC_Enable(NUM_PDMA1);
}

void __irq Isr_AP_DMA_RX_Done(void)
{
	// for S5PV210
	u32 uDmaIntStatus = 0;

//	MODEM_PRINTF("[AP DMA RX INT]\n");
	
	DMA_GetIntrSrc( &uDmaIntStatus, &oAPRx0);
	if ( uDmaIntStatus == 32 )	// ERROR
	{
		MODEM_PRINTF("[AP DMA TX FAULT]\n");
		DMA_StopDmac(&oAPTx0);
	}
	else if ( uDmaIntStatus == 0 )	// OK
	{		
	  	DMA_ClearIntPending(uDmaIntStatus, &oAPRx0);		
		g_APDMAReadReady = 1;
	}
	
	INTC_ClearVectAddr();
}

//======================= Interrupt Test Functions ===========================
void Test_AP_TX_INTR(void)
{
	//Message
	MODEM_PRINTF("AP TX Interrupt Test\n");

	g_ModemReadDoneReply = 0;
	
	// Modem i/f setting
	MODEMIF_AP_Init(true, true, true);

	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_AP_TX_INT);

	while(*(volatile u16 *)(DPSRAM_BASE_AP + INT2MSM_ADR) != 0x0);		// busy waiting to clear INT2MSM register
	
	MODEM_PRINTF("AP TX Interrupt Test OK !!\n");
}

void Test_MODEM_TX_INTR(void)
{
	MODEM_PRINTF("MODEM TX to Interrupt Test\n");
	
	MODEMIF_MODEM_Init(true, true);
	
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_INT);
	
	// to AP message "STOP"
	//MODEMIF_Outp16(DPSRAM_BASE_MODEM+INT2AP_ADR, 0x1100);
	MODEM_PRINTF("MODEM TX Interrupt Test OK !!\n");
}

void Test_MODEM_SetClear_INTR(void)
{
	u16 usInt2AP = 0;
	u16 usTemp = 0;
	
	MODEMIF_MODEM_Init(true, true);

	usTemp = (MODEMIF_Inp16(DPSRAM_BASE_MODEM+INT2MSM_ADR))&0xffff;
	MODEM_PRINTF("[ReadValue:0x%x]\n", usTemp);
	MODEMIF_Outp16(DPSRAM_BASE_MODEM+INT2MSM_ADR, usTemp);

	Getc();
	MODEM_PRINTF(" > input interrupt set value(0x0~0xFFFF)\n");
	usInt2AP = (u16)UART_GetIntNum();
	usTemp = (MODEMIF_Inp16(DPSRAM_BASE_MODEM+INT2MSM_ADR))&0xffff;
	MODEM_PRINTF("[ReadValue:0x%x, InputValue:0x%x]\n", usTemp,usInt2AP);
	MODEMIF_Outp16(DPSRAM_BASE_MODEM+INT2MSM_ADR, usInt2AP);
}

//======================= Normal Operational Functions ===========================
/**
* {Test_AP_TX}
*
* @brief			Test AP Tx -> Modem Rx.
* 
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_AP_TX(void)
{
	u32 length,i;
	u16	*txdata = NULL;
	u16 *aptx_buf = NULL;

	g_ModemReadyReply = 0;
	g_ModemReadDoneReply = 0;

	//Message
	MODEM_PRINTF(" [AP TX TEST]\n");

	// Modem i/f setting
	MODEMIF_AP_Init(true, true, true);
	
	txdata = (u16 *)(AP_TX_BUFFER);

	// AP TX Buffer data
	for(length=0 ; length < (AP_TX_BUFLEN*BLK_COUNT) / 2; length++)
	{
		*(txdata + length) = length;
	}

	MODEM_PRINTF("\n------------------------\n");
	MODEM_PRINTF("<--- 0x%x(AP to Modem)\n", MODEM_AP_TX_INITIAL);
	/* 1. Interrupt request to MSM */
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_AP_TX_INITIAL);

	
	/* 2. wait til Interrupt request to MSM is accepted */
	while( !g_ModemReadyReply );


	for(i = 0 ; i < BLK_COUNT ; i++)
	{		
		// TX Buffer pointer setting
		txdata = (u16 *)(AP_TX_BUFFER + ( i * AP_TX_BUFLEN ) );
		aptx_buf = (u16 *)(AP_TX_Queue);

		for (length = 0 ; length < AP_TX_BUFLEN ; length++)
		{
			*(u16 *)(aptx_buf ++) = *(txdata++);
		}
		
		MODEM_PRINTF("\n------------------------\n");
		MODEM_PRINTF("[%dth Data Transfer Attempt]\n", i+1);
		MODEM_PRINTF("<--- 0x%x(AP to Modem)\n", MODEM_AP_TX_DATA_XFER);	
		/* 5. Interrupt request to MSM */
		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_AP_TX_DATA_XFER);
		
		/* 6. wait til Interrupt request to MSM is accepted */
		while( !g_ModemReadDoneReply );

		g_ModemReadDoneReply = 0;
		
		MODEM_PRINTF("[%dth Data Transfer Success]\n", i+1);
	}

	MODEM_PRINTF("\n------------------------\n");	
	MODEM_PRINTF("<--- 0x%x(AP to Modem)\n", MODEM_AP_TX_XFER_DONE);
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_AP_TX_XFER_DONE);
	MODEM_PRINTF("AP TX END\n");
}

void Test_AP_RX(void)
{
	u16 length = 0;
	u16	*txdata = NULL;

	u16 i=0;
	u16 *rx_data = NULL;
	u16	*rx_buf = NULL;
	
	g_ApDataCnt = 0;
	g_APReadDone = 0;

	g_ModemWriteReadyReply = 0;
	
	txdata = (u16 *)(DPSRAM_BASE_AP);

	// Clear DPSRAM Buffer
	for(length = 0 ; length < DPSRAM_SIZE; length++)
	{
		*(txdata + length) = 0;
	}
	
	MODEMIF_AP_Init(true, true, true);
	MODEM_PRINTF("[AP RX START... AP is awaiting data from MODEM]\n");

	while( !g_ModemWriteReadyReply );
	
	MODEM_PRINTF(" 0x%x(AP to Modem) --->\n", MODEM_MSM_TX_INITIAL);
	/* 3. Interrupt request to MSM */
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_TX_INITIAL);


	for(i = 0 ; i < BLK_COUNT ; i++)
	{
		while( !g_ModemWriteDoneReply );
		
		g_ModemWriteDoneReply = 0;
		
		rx_buf = (u16 *)(AP_RX_BUFFER + (g_ApDataCnt*MODEM_TX_BUFLEN));
		rx_data = (u16 *)(AP_RX_Queue);

		for (length = 0 ; length < MODEM_TX_BUFLEN ; length++)
			*(u16 *)(rx_buf++) = *(rx_data++);

		g_ApDataCnt++;
		
		MODEM_PRINTF("[%dth Data Received OK]\n", i+1);
		
		MODEM_PRINTF(" 0x%x(AP to Modem) --->\n", MODEM_MSM_TX_DATA_XFER);
		/* 5. Interrupt request to MSM */
		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_TX_DATA_XFER);

		MODEM_PRINTF("[%dth Data Received Ack]\n", i+1);
	}
	
	while( !((g_ApDataCnt == BLK_COUNT) && (g_APReadDone == 1)) );
	
	MODEM_PRINTF("cnt = 0x%x, dma_cnt = 0x%x\n", g_ApDataCnt, BLK_COUNT);
	MODEM_PRINTF("[AP RX END]\n");
}

void Test_MODEM_TX(void)
{
	u32 length,i;
	u16 *txdata = NULL;
	u16 *modemtx_buf = NULL;

	g_ApReadyReply	 = 0;
	g_ApReadDoneReply = 0;
	
	MODEM_PRINTF("MODEM TX \n");
		
	MODEMIF_MODEM_Init(true, true);

	txdata = (u16 *)(MSM_TX_BUFFER);
	
	// MODEM TX Buffer data
	for(length = 0 ; length < (BLK_COUNT * MODEM_TX_BUFLEN) / 2; length++)
	{
		*(txdata + length) = length;
	}

	MODEM_PRINTF("<--- 0x%x (MSM to AP)\n", MODEM_MSM_TX_INITIAL);
	MODEM_PRINTF(" ------------------------\n");
	/* 1. Interrupt request to MODEM I/F */
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_INITIAL);
	
	/* 2. wait til Interrupt request to MODEM I/F is accepted */
	while( !g_ApReadyReply );

	for(i = 0 ; i < BLK_COUNT ; i++)
	{
		// TX Buffer pointer setting
		txdata = (u16 *)( MSM_TX_BUFFER + (i * MODEM_TX_BUFLEN) );
		modemtx_buf = (u16 *)(MSM_TX_Queue);

		for (length = 0 ; length < MODEM_TX_BUFLEN ; length++)
		{
			*(u16 *)(modemtx_buf++) = *(txdata++);
		}

		MODEM_PRINTF("\n------------------------\n");
		MODEM_PRINTF("[%dth Data Transfer Attempt]\n", i+1);		
		MODEM_PRINTF("<--- 0x%x (MSM to AP)\n", MODEM_MSM_TX_DATA_XFER);
		/* 4. Interrupt request to MODEM I/F */
		MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_DATA_XFER);

		while( !g_ApReadDoneReply );
		
		g_ApReadDoneReply = 0;
		
		MODEM_PRINTF("[%dth Data Transfer Success]\n", i+1);
	}

	MODEM_PRINTF("<--- 0x%x(MSM to AP)\n", MODEM_MSM_TX_XFER_DONE);
	MODEM_PRINTF(" ------------------------\n");
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_XFER_DONE);
	MODEM_PRINTF("MODEM TX END\n");
}

void Test_MODEM_RX(void)
{
	int i =0;	
	u16 length = 0;
	u16 *rx_data = NULL;
	u16	*rx_buf = NULL;
	
	g_MsmDataCnt = 0;
	
	g_ApWriteDoneReply = 0;
	g_ApWriteReadyReply = 0;
	g_ApWriteDone = 0;

	MODEM_PRINTF("[MODEM RX TEST]\n");	
	
	MODEMIF_MODEM_Init(true, true);
	
	MODEM_PRINTF("[MSM WAITING...]\n");	

	while( !g_ApWriteReadyReply );

	MODEM_PRINTF(" ------------------------\n");
	MODEM_PRINTF(" 0x%x(Modem to AP) --->\n", MODEM_AP_TX_INITIAL);
	/* 3. Interrupt request to MODEM I/F */
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_AP_TX_INITIAL);

	
	for(i = 0 ; i < BLK_COUNT; i++)
	{
		/* 6. wait til Interrupt request to MSM is accepted */
		while( !g_ApWriteDoneReply );

		g_ApWriteDoneReply = 0;
		
		rx_buf = (u16 *)(MSM_RX_BUFFER + (g_MsmDataCnt*AP_TX_BUFLEN));
		rx_data = (u16 *)(MSM_RX_Queue);

		for (length = 0 ; length < AP_TX_BUFLEN ; length++)
			*(u16 *)(rx_buf++) = *(rx_data++);

		g_MsmDataCnt++;
		MODEM_PRINTF("[%dth Data Received OK]\n", i+1);

		MODEM_PRINTF(" 0x%x(Modem to AP) --->\n", MODEM_AP_TX_DATA_XFER);
		/* 7. Interrupt request to MODEM I/F */
		MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_AP_TX_DATA_XFER);

		MODEM_PRINTF("[%dth Data Received Ack]\n", i+1);
	}
	
	/* 11. wait til all trasnsactions are done */
	while( !((g_MsmDataCnt == BLK_COUNT) && (g_ApWriteDone == 1)) );
	
	MODEM_PRINTF("cnt = 0x%x, dma_cnt = 0x%x\n", g_MsmDataCnt, BLK_COUNT);
	MODEM_PRINTF("[MODEM RX END]\n");	
}

//======================= Normal Operational Functions ===========================
static void MODEMIF_DMA_SetNonSecure(void)
{
	/* TZPC 0 : DMC0, DMC1, VIC */
	MODEMIF_Outp32(0xF1500804,0xff);
	MODEMIF_Outp32(0xF1500810,0xff);
	MODEMIF_Outp32(0xF150081C,0xff);
	
	MODEMIF_Outp32(0xFAD00804,0xff);
	MODEMIF_Outp32(0xFAD00810,0xff);
	MODEMIF_Outp32(0xFAD0081C,0xff);
	MODEMIF_Outp32(0xFAD00828,0xff);
	
	MODEMIF_Outp32(0xE0600804,0xff);
	MODEMIF_Outp32(0xE0600810,0xff);
	MODEMIF_Outp32(0xE060081C,0xff);
	MODEMIF_Outp32(0xE0600828,0xff);

	/* TZPC 3 : G BLK */
	MODEMIF_Outp32(0xE1C00804,0xff);
	MODEMIF_Outp32(0xE1C00810,0xff);
	MODEMIF_Outp32(0xE1C0081C,0xff);
	MODEMIF_Outp32(0xE1C00828,0xff);
}

void Test_AP_DMA_TX(void)
{
	u16 i = 0;
	u16 length = 0;
	u16	*txdata = NULL;
	u16	*rxdata = NULL;
	
	g_DmaDoneATx = 0;
	g_APDMAWriteReady = 0;
	g_APDMAWriteDone = 0;

	//Message
	MODEM_PRINTF("[AP DMA TX]\n");

	UART_Printf("\n['Test_MODEM_DMA_RX' MUST be called in MSM B'd side first,");
	UART_Printf("\nPress any key when it's ready]");
	while( !GetKey() );
	
	
	MODEM_PRINTF("\n > Select  AP's modem DMA port(0~1) ");
	ucModemDMAPort = (u8)UART_GetIntNum();
	MODEM_PRINTF("\n");
	if( (ucModemDMAPort != 0) && (ucModemDMAPort != 1) )
	{
		MODEM_PRINTF(" --> Error!! You have entered invalid DMA port number!!\n");
		return;
	}
	
	// Modem i/f setting
	MODEMIF_AP_Init(true,true, true);

	// AP DMA TX Buffer data : SRAM
	txdata = (u16 *)(AP_TX_BUFFER);
	for(length=0 ; length < (AP_DMA_TX_BUFLEN * DMA_COUNT) / 2; length++)
	{
		*(txdata + length) = length;
	}
	
	// AP DMA RX Buffer data : DPSRAM
	rxdata = (u16 *)(AP_TX_Queue);
	for(length=0 ; length < (AP_DMA_TX_BUFLEN) / 2; length++)
	{
		*(rxdata + length) = 0;
	}
	
	// Modem I/F DMA setup
	MODEMIF_SET_APDMA(MODEMIF_AP_TX, (MODEMIF_DMA_REQ_NUM)(MODEMIF_TX_0 + ucModemDMAPort), 1);	// Enable AP DMA TX0 Request
	MODEMIF_SET_DMA_REQ_ADDR(MODEMIF_AP_TX, (MODEMIF_DMA_REQ_NUM)(MODEMIF_TX_0 + ucModemDMAPort), DMA_TX_ADDR0 + ucModemDMAPort*2);

	/* set to non-secure */
	MODEMIF_DMA_SetNonSecure();
	
	// for S5PV210
	// Enable PDMA1 interrupt for MSM_REQ 0 ~ 3
	INTC_SetVectAddr(NUM_PDMA1,  Dma0Done_APDMATx);
	INTC_Enable(NUM_PDMA1);

	// DMA start
	MODEM_PRINTF("[AP DMA TX START]\n");
	
	for( i = 0 ; i < DMA_COUNT; i++ )
	{
		// DMA Init & Setup
		DMA_SetCh(DMA_10, &oAPTx0);
		DMA_InitCh(HWORD, (DREQ_SRC)(MSM_REQ0 + ucModemDMAPort), DMA_M2P, SINGLE, &oAPTx0);

		// DMA Start
		DMA_StartCh(AP_TX_BUFFER + (g_DmaDoneATx*AP_DMA_TX_BUFLEN), AP_TX_Queue, AP_DMA_TX_BUFLEN, &oAPTx0);
		
		MODEM_PRINTF("\n ------------------------\n");
		MODEM_PRINTF("<--- 0x%x(AP to Modem)\n", MODEM_AP_DMA_TX_DATA_XFER);
		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_AP_DMA_TX_DATA_XFER);
		MODEM_PRINTF("[%dth DMA Data Transfer Request]\n", i+1);
		
		while( !g_APDMAWriteReady );

		g_APDMAWriteReady = 0;
		
		g_DmaDoneATx++;
		MODEM_PRINTF("[%dth DMA Data Transfer Success]\n", i+1);
	}

	MODEM_PRINTF("\n------------------------\n");	
	MODEM_PRINTF("[DMA Data Transfer Done Request]\n");
	MODEM_PRINTF("<--- 0x%x(AP to Modem)\n", MODEM_AP_DMA_TX_XFER_DONE);
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_AP_DMA_TX_XFER_DONE);

	while( !( (g_DmaDoneATx == DMA_COUNT) && (g_APDMAWriteDone == 1) ) );
	
	MODEM_PRINTF("cnt = 0x%x, dma_cnt = 0x%x\n", g_DmaDoneATx, DMA_COUNT);
	MODEM_PRINTF("[AP DMA TX END]\n");
}

void Test_AP_DMA_RX(void)
{
	u16 i = 0;
	
	g_DmaDoneARx = 0;
	g_APDMAReadReady = 0;
	g_APDMAReadDone = 0;

	UART_Printf("\n['Test_MODEM_DMA_TX' MUST be called in MSM B'd side first,");
	UART_Printf("\nPress any key when it's ready]");
	while( !GetKey() );
	
	MODEM_PRINTF("\n > Select  AP's modem DMA port(0~1) ");
	ucModemDMAPort = (u8)UART_GetIntNum();
	MODEM_PRINTF("\n");
	if( (ucModemDMAPort != 0) && (ucModemDMAPort != 1) )
	{
		MODEM_PRINTF(" --> Error!! You have entered invalid DMA port number!!\n");
		return;
	}

	// Modem i/f setting
	MODEMIF_AP_Init(true, true, true);

	// Modem I/F DMA setup
	MODEMIF_SET_DMA_REQ_ADDR(MODEMIF_AP_RX, (MODEMIF_DMA_REQ_NUM)(MODEMIF_RX_0 + ucModemDMAPort), DMA_RX_ADDR0 + ucModemDMAPort*2);
	MODEMIF_SET_APDMA(MODEMIF_AP_RX, (MODEMIF_DMA_REQ_NUM)(MODEMIF_RX_0 + ucModemDMAPort), 1);	// Enable AP DMA RX0 Request

	/* set to non-secure */
	MODEMIF_DMA_SetNonSecure();
	
	// for S5PV210
	// Enable PDMA1 interrupt for MSM_REQ 0 ~ 3
	INTC_SetVectAddr(NUM_PDMA1,  Dma0Done_APDMARx);
	INTC_Enable(NUM_PDMA1);


	// DMA start
	MODEM_PRINTF("[AP DMA RX START]\n");

	for( i = 0 ; i < DMA_COUNT; i++ )
	{
		// DMA Init & Setup
		DMA_SetCh(DMA_10, &oAPRx0);
		DMA_InitCh(HWORD, (DREQ_SRC)(MSM_REQ2 + ucModemDMAPort), DMA_P2M, SINGLE, &oAPRx0);

		// DMA Start
		DMA_StartCh(AP_RX_Queue, AP_RX_BUFFER + (g_DmaDoneARx*MODEM_DMA_TX_BUFLEN), MODEM_DMA_TX_BUFLEN, &oAPRx0);
		
//		MODEM_PRINTF("\n ------------------------\n");
//		MODEM_PRINTF("[AP DMA RX TO MSM]\n");
//		MODEM_PRINTF("<--- 0x%x(AP to Modem)\n", MODEM_MSM_DMA_TX_DATA_XFER);

		/* 2. Interrupt request to MODEM I/F */
		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_DMA_TX_DATA_XFER);

		while( !g_APDMAReadReady );

		g_APDMAReadReady = 0;

		g_DmaDoneARx++;
		MODEM_PRINTF("[%dth DMA Data Transfer Success]\n", i+1);
	}

	MODEM_PRINTF("\n------------------------\n");	
	MODEM_PRINTF("<--- 0x%x(AP to Modem)\n", MODEM_MSM_DMA_TX_XFER_DONE);
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_DMA_TX_XFER_DONE);
	
	while( !((g_DmaDoneARx == DMA_COUNT) && (g_APDMAReadDone == 1)) );

	MODEM_PRINTF("cnt = 0x%x, dma_cnt = 0x%x\n", g_DmaDoneARx, DMA_COUNT);
	MODEM_PRINTF("[AP DMA RX END]\n");
}

void Test_MODEM_DMA_TX(void)
{
	u16 i = 0;
	u16 temp = 0;
	u16	length = 0;
	u16	*txdata = NULL;

	g_DmaDoneMTx = 0;
	g_ModemDMAWriteReady = 0;
	g_ModemDMAWriteDone = 0;
	
	MODEM_PRINTF("MODEM DMA TX TEST!!!\n");	
	
	MODEM_PRINTF("\n > Select  AP's modem DMA port(0~1) ");
	ucModemDMAPort = (u8)UART_GetIntNum();
	MODEM_PRINTF("\n");
	if( (ucModemDMAPort != 0) && (ucModemDMAPort !=1) )
	{
		MODEM_PRINTF(" --> Error!! You have entered invalid DMA port number!!\n");
		return;
	}

	// MSM TX Buffer data
	txdata = (u16 *)(MSM_TX_BUFFER);
	for(length = 0 ; length < DMA_RX_ADDR0 / 2; length++)
	{
		*(txdata + length) = length;
	}

	// Modem i/f setting
	MODEMIF_MODEM_Init(true, true);

	MODEM_PRINTF("[MODEM DMA TX WAITING...]\n");

	for( i = 0; i < DMA_COUNT; i++ )
	{
		while( !g_ModemDMAWriteReady );

		g_ModemDMAWriteReady = 0;

		/* Trigger DMA */
		MODEM_PRINTF("\n ------------------------ \n");
		MODEM_PRINTF("[MSM Triggered %dth AP DMA RX]\n", i+1);
		MODEMIF_Outp16(DPSRAM_BASE_MODEM + DMA_RX_ADDR0 + ucModemDMAPort*2, 0x5A5A);
		temp = (MODEMIF_Inp16(DPSRAM_BASE_MODEM + DMA_RX_ADDR0 + ucModemDMAPort*2));
		MODEM_PRINTF("[DMA_RX_ADDR0 : 0x%x]\n", temp);
		
		g_DmaDoneMTx++;
	}
	
	while( !((g_DmaDoneMTx == DMA_COUNT) && (g_ModemDMAWriteDone == 1)) );
	
	MODEM_PRINTF("\n ------------------------\n");
	MODEM_PRINTF("<--- 0x%x(MSM to AP)\n", MODEM_MSM_DMA_TX_XFER_DONE);
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_DMA_TX_XFER_DONE);
	MODEM_PRINTF("cnt = 0x%x, dma_cnt = 0x%x\n", g_DmaDoneMTx, DMA_COUNT);
	MODEM_PRINTF("[MODEM DMA TX END]\n");
}

void Test_MODEM_DMA_RX(void)
{
	u16 i = 0;
	u16 temp = 0;
	
	g_DmaDoneMRx = 0;
	g_ModemDMAReadReady = 0;
	g_ModemDMAReadDone = 0;

	MODEM_PRINTF("[MODEM DMA RX TEST]\n");

	MODEM_PRINTF("\n > Select  AP's modem DMA port(0~1) ");
	ucModemDMAPort = (u8)UART_GetIntNum();
	MODEM_PRINTF("\n");
	if( (ucModemDMAPort != 0) && (ucModemDMAPort != 1) )
	{
		MODEM_PRINTF(" --> Error!! You have entered invalid DMA port number!!\n");
		return;
	}

	// Modem i/f setting
	MODEMIF_MODEM_Init(true, true);
	
	MODEM_PRINTF("[MODEM DMA RX WAITING...]\n");

	for( i = 0; i < DMA_COUNT; i++ )
	{
		while( !g_ModemDMAReadReady );

		g_ModemDMAReadReady = 0;
				
		/* Trigger DMA */
		MODEM_PRINTF("\n ------------------------ \n");
		MODEM_PRINTF("[MSM Triggered %dth AP DMA TX]\n", i+1);
		temp = (MODEMIF_Inp16(DPSRAM_BASE_MODEM + DMA_TX_ADDR0 + ucModemDMAPort*2));
		MODEM_PRINTF("[DMA_TX_ADDR0 : 0x%x]\n", temp);

		g_DmaDoneMRx++;
	}
	
	while( !((g_DmaDoneMRx == DMA_COUNT) && (g_ModemDMAReadDone == 1)) );

	MODEM_PRINTF("\n ------------------------\n");
	MODEM_PRINTF("[DMA Data Transfer Done Ack]\n");
	MODEM_PRINTF("<--- 0x%x (MSM to AP)\n", MODEM_AP_DMA_TX_XFER_DONE);
	/* 1. Interrupt request to MODEM I/F */
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_AP_DMA_TX_XFER_DONE);
	
	MODEM_PRINTF("cnt = 0x%x, dma_cnt = 0x%x\n", g_DmaDoneMRx, DMA_COUNT);
	MODEM_PRINTF("[MODEM DMA RX END]\n");
}

//======================= DPSRAM Test Functions ===========================
void Test_Display_MEM(void)
{
	u32 i, length, add, size;

	MODEM_PRINTF("\nMEM DUMP Address : ");
	add = UART_GetIntNum();
	MODEM_PRINTF("add = 0x%x\n", add);

	MODEM_PRINTF("\nMEM DUMP Size : ");
	size = UART_GetIntNum();
	MODEM_PRINTF("size = 0x%x\n", size);

	MODEM_PRINTF("Press any key and DPSRAM R/W test start!!!\n");
	Getc();
	
	for (length = 0 ; length <= size ; length+=4)
	{
		i = MODEMIF_Inp32(add + length);
		MODEM_PRINTF("%08x    ", i);
	}

	MODEM_PRINTF("\nDPSRAM Test........OK!!!\n");
}

void Test_DPSRAM(void)
{
	// ALL DPSRAM TEST
	u32	i = 0xFFFFFFFF;
	u32	length = 0;

	// Modem i/f setting
	MODEMIF_MODEM_Init(true, true);

	MODEM_PRINTF("Press any key and DPSRAM R/W test start!!!\n");
	Getc();

//	for(i=0; i <= 0xFFFFFFFF; i++)
	{	
		for (length = 0 ; length <= 0x1FFE; length+=4)
		{
			*(unsigned int *)(DPSRAM_BASE_AP + length) = i;

			if(*(unsigned int *)(DPSRAM_BASE_AP + length) != i )
				MODEM_PRINTF("Error: %08x\n", DPSRAM_BASE_AP + length);
			else
				*(unsigned int *)(DPSRAM_BASE_AP + length) = 0;
		}
		UART_Printf("%08x ", i);
	}

	MODEM_PRINTF("DPSRAM Test........OK!!!\n");

}

void Test_DPSRAM_OneTime(void)
{
	// ALL DPSRAM TEST
	u32	i = 0;
	u32	length = 0;

	// Modem i/f setting
	MODEMIF_MODEM_Init(true, true);

	MODEM_PRINTF("Press any key and DPSRAM R/W test through ROM I/F start!!!\n");
	Getc();
	

	i = 0xFFFFFFFF;
	for (length = 0 ; length <= 0x1FFE ; length+=4)
	{
		*(unsigned int *)(DPSRAM_BASE_MODEM + length) = i;

		if(*(unsigned int *)(DPSRAM_BASE_MODEM + length) != i)
			MODEM_PRINTF("Error: %08x\n", DPSRAM_BASE_MODEM + length);
		else
			*(unsigned int *)(DPSRAM_BASE_MODEM + length) = 0;
	}

	MODEM_PRINTF("DPSRAM Test........OK!!!\n");

}

/**
* {Test_MODEM_Rx_CLK_Gating}
*
* @brief			This function tests is Clock gating control.
* @conditions	
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
static
u8	Test_MODEM_Rx_CLK_Gating(void)
{
	u8	ret = TRUE;
	
	UART_Printf("\n[MODEM Rx Clock Gating Control Test with AP Tx]\n");

	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, TRUE);
	/* It should work cause CLK Gating is on */	
	Test_MODEM_RX();
	
	UART_Printf("\n[CLK_MODEM OFF (CLK_GATE_IP2)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, FALSE);
	Delay(100);

	/* It shouldn't work cause CLK Gating is off */
//	Test_MODEM_RX();
	
	UART_Printf("\n[CLK_MODEM ON (CLK_GATE_IP2)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, TRUE);

	/* It should work cause CLK Gating is on */	
	Test_MODEM_RX();

	UART_Printf("\n[MODEM Rx Clock Gating Control Test Complete]\n");
	
	return	ret;
}

/**
* {Test_MODEM_Tx_CLK_Gating}
*
* @brief			This function tests is Clock gating control.
* @conditions	
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
static
u8	Test_MODEM_Tx_CLK_Gating(void)
{
	u8	ret = TRUE;
	
	UART_Printf("\n[MODEM Tx Clock Gating Control Test with AP Rx]\n");

	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, TRUE);
	/* It shouldn work cause CLK Gating is on */
	Test_MODEM_TX();
	
	UART_Printf("\n[CLK_MODEM OFF (CLK_GATE_IP2)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, FALSE);
	Delay(100);

	/* It shouldn't work cause CLK Gating is off */
//	Test_MODEM_TX();
	
	UART_Printf("\n[CLK_MODEM ON (CLK_GATE_IP2)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, TRUE);

	/* It shouldn work cause CLK Gating is on */
	Test_MODEM_TX();

	UART_Printf("\n[MODEM Tx Clock Gating Control Test Complete]\n");
	
	return	ret;
}

/**
* {Test_AP_Rx_CLK_Gating}
*
* @brief			This function tests is Clock gating control.
* @conditions	
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
static
u8	Test_AP_Rx_CLK_Gating(void)
{
	u8	ret = TRUE;
	
	UART_Printf("\n[AP Rx Clock Gating Control Test with MODEM Tx]\n");

	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, TRUE);
	/* It should work cause CLK Gating is on */
	Test_AP_RX();
	
	UART_Printf("\n[CLK_MODEM OFF (CLK_GATE_IP2)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, FALSE);
	Delay(100);
	
	/* It shouldn't work cause CLK Gating is off */
//	Test_AP_RX();
	
	UART_Printf("\n[CLK_MODEM ON (CLK_GATE_IP2)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, TRUE);

	/* It should work cause CLK Gating is on */	
	Test_AP_RX();

	UART_Printf("\n[AP Rx Clock Gating Control Test Complete]\n");
	
	return	ret;
}

/**
* {Test_AP_Tx_CLK_Gating}
*
* @brief			This function tests is Clock gating control.
* @conditions	
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
static
u8	Test_AP_Tx_CLK_Gating(void)
{
	u8	ret = TRUE;
	
	UART_Printf("\n[AP Tx Clock Gating Control Test with MODEM Rx]\n");

	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, TRUE);
	/* It should work cause CLK Gating is on */
	Test_AP_TX();
	
	UART_Printf("\n[CLK_MODEM OFF (CLK_GATE_IP2)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, FALSE);
	Delay(100);

	/* It shouldn't work cause CLK Gating is off */
//	Test_AP_TX();

	UART_Printf("\n[CLK_MODEM ON (CLK_GATE_IP2)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_MODEM, TRUE);

	/* It should work cause CLK Gating is on */	
	Test_AP_TX();

	UART_Printf("\n[AP Tx Clock Gating Control Test Complete]\n");
	
	return	ret;
}

//======================= Concurrent & Performance Functions ===========================//

/**
* {Test_AP_TX_Performance}
*
* @brief			Test MSM RX <- AP TX.
* @description	Test Max Write performance of MODEM I/F. \n
*				Use 'Test_MODEM_RX_AP_Write_Performane' in MSM B'd side.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_AP_TX_Performance(void)
{
	u32 length,i;
	u16	*txdata = NULL;
	u16 *aptx_buf = NULL;

	g_ModemReadyReply = 0;
	g_ModemReadDoneReply = 0;
	
	//Message
	MODEM_PRINTF("[AP TX PERFORMANCE TEST]\n");
	
	txdata = (u16 *)(AP_TX_BUFFER);

	// AP TX Buffer data
	for(length=0 ; length < (AP_TX_BUFLEN*BLK_COUNT); length++)
	{
		*(txdata + length) = length;
	}

	// Modem i/f setting
	MODEMIF_AP_Init(true, true, true);

	MODEM_PRINTF("[WAITING...]");
	
	/* 2. wait til Interrupt request to MSM is accepted */
	while( !g_ModemReadyReply );

	MODEM_PRINTF("<--- 0x%x (MSM to AP)\n", MODEM_MSM_RX_AP_TX_INITIAL);
	MODEM_PRINTF(" ------------------------\n");

	/* 1. Interrupt request to MODEM I/F */
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_RX_AP_TX_INITIAL);


	for( i = 0 ; i < BLK_COUNT ; i++)
	{		
		while( !g_ModemReadDoneReply );

		/* 6. wait til Interrupt request to MSM is accepted */
		g_ModemReadDoneReply = 0;

		// TX Buffer pointer setting
		txdata = (u16 *)(AP_TX_BUFFER + ( i * AP_TX_BUFLEN ) );
		aptx_buf = (u16 *)(AP_TX_Queue);

		for (length = 0 ; length < AP_TX_BUFLEN / 2; length++)
		{
			*(u16 *)(aptx_buf ++) = *(txdata++);
		}

//		MODEM_PRINTF(" ------------------------\n");		
//		MODEM_PRINTF("<--- 0x%x(AP to Modem)\n", MODEM_MSM_RX_AP_TX_DATA_XFER);		
		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_RX_AP_TX_DATA_XFER);
		
		MODEM_PRINTF(" Block Count = %d\n", i+1);
	}

	MODEM_PRINTF("<--- 0x%x(MSM to AP)\n", MODEM_MSM_RX_AP_TX_XFER_DONE);
	MODEM_PRINTF(" ------------------------\n");
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_RX_AP_TX_XFER_DONE);

	MODEM_PRINTF("[AP TX PERFORMANCE TEST END]\n");
}

/**
* {Test_MODEM_RX_AP_TX_Performance}
*
* @brief			Test MSM RX <- AP TX.
* @description	Test Max Write performance of MODEM I/F. \n
				Use 'Test_AP_TX_Performance' in AP B'd side.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_MODEM_RX_AP_TX_Performance(void)
{
	u16 length = 0;
	u16 i = 0;
	u16	*rxdata = NULL;
	u16 *rx_buf = NULL;

	u32 uElapsedTime = 0;

	g_ApWriteReadyReply = 0;
	g_ApWriteDoneReply = 0;
	g_ApWriteDone = 0;
	
	MODEM_PRINTF("MODEM RX TEST!!!\n");
	
	UART_Printf("\n['Test_AP_TX_Performance' MUST be called in AP B'd side first");
	UART_Printf("\nPress any key when it's ready]");
	while( !GetKey() );
	

	MODEMIF_MODEM_Init(true, true);

	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_RX_AP_TX_INITIAL);
	
	while( !g_ApWriteReadyReply );
	
	
	for( i = 0 ; i < BLK_COUNT ; i++)
	{
//		MODEM_PRINTF(" ------------------------\n");
//		MODEM_PRINTF("<--- 0x%x(MSM to AP)\n", MODEM_MSM_RX_AP_TX_DATA_XFER);
		MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_RX_AP_TX_DATA_XFER);

		/* Start Timer */
		StartTimer(0);
		
		while( !g_ApWriteDoneReply );
		
		g_ApWriteDoneReply = 0;

		/* Stop Timer */
		uElapsedTime = StopTimer(0);
		MODEM_PRINTF("(%3.1f Mbps, %d us)\n", ((float) 8 * AP_TX_BUFLEN / uElapsedTime), uElapsedTime);
		MODEM_PRINTF(" Blk Count = %d\n", i+1);
	}

	while( !g_ApWriteDone );
	
	MODEM_PRINTF("[MODEM RX AP WRITE TEST END]\n");
}

/**
* {Test_AP_RX_Performance}
*
* @brief			Test MSM TX -> AP RX.
* @description	Test Max Read performance of MODEM I/F. \n
*				Use 'Test_MODEM_TX_AP_RX_Performance' in MSM B'd side.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_AP_RX_Performance(void)
{
	u32 length,i;

	u16 *rx_buf = NULL;
	u16	*rx_data = NULL;
		
	g_ModemWriteReadyReply = 0;
	g_ModemWriteDoneReply = 0;
	g_ModemWriteDone = 0;
	
	//Message
	MODEM_PRINTF("[AP RX PERFORMANCE TEST]\n");
	
	// Modem i/f setting
	MODEMIF_AP_Init(true, true, true);

	MODEM_PRINTF("[WAITING...]");
	
	/* 2. wait til Interrupt request to MSM is accepted */
	while( !g_ModemWriteReadyReply );

	MODEM_PRINTF("<--- 0x%x (MSM to AP)\n", MODEM_MSM_TX_AP_RX_INITIAL);
	MODEM_PRINTF(" ------------------------\n");
	/* 1. Interrupt request to MODEM I/F */
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_TX_AP_RX_INITIAL);


	for( i = 0 ; i < BLK_COUNT ; i++)
	{		
		while( !g_ModemWriteDoneReply );

		/* 6. wait til Interrupt request to MSM is accepted */
		g_ModemWriteDoneReply = 0;

		// RX Buffer pointer setting
		rx_buf = (u16 *)(AP_RX_BUFFER + ( i * MODEM_TX_BUFLEN) );
		rx_data = (u16 *)(AP_RX_Queue);

		for (length = 0 ; length < MODEM_TX_BUFLEN / 2; length++)
			*(u16 *)(rx_buf++) = *(rx_data++);

		//MODEM_PRINTF(" 0x%x(AP to Modem) --->\n", MODEM_MSM_TX_AP_RX_DATA_XFER);
		//MODEM_PRINTF(" ------------------------\n");
		//MODEM_PRINTF("[AP DATA READING...]\n");
		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_TX_AP_RX_DATA_XFER);
	
		MODEM_PRINTF(" Block Count = %d\n", i+1);
	}

	while( !g_ModemWriteDone );
	
	MODEM_PRINTF("[AP TX PERFORMANCE TEST END]\n");
}	
/**
* {Test_MODEM_TX_AP_RX_Performance}
*
* @brief			Test MSM TX -> AP RX.
* @description	Test Max Read performance of MODEM I/F.\n
*				Use 'Test_AP_RX' in AP side B'd.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_MODEM_TX_AP_RX_Performance(void)
{
	u16 length = 0;
	u16 i = 0;
	u16	*txdata = NULL;
	u16 *modemtx_buf = NULL;

	u32 uElapsedTime = 0;

	g_ApReadyReply	 = 0;

	MODEM_PRINTF("MODEM TX TEST!!!\n");
	
	UART_Printf("\n['Test_AP_RX' MUST be called in AP B'd side first");
	UART_Printf("\nPress any key when it's ready]");
	while( !GetKey() );

	txdata = (u16 *)(MSM_TX_BUFFER);

	// MODEM TX Buffer data
	for(length = 0 ; length < (BLK_COUNT * MODEM_TX_BUFLEN); length++)
	{
		*(txdata + length) = length;
	}
	
	MODEMIF_MODEM_Init(true, true);

	MODEM_PRINTF(" ------------------------\n");
	MODEM_PRINTF("<--- 0x%x (MSM to AP)\n", MODEM_MSM_TX_AP_RX_INITIAL);
	/* 1. Interrupt request to MODEM I/F */
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_AP_RX_INITIAL);

	/* 2. wait til Interrupt request to MODEM I/F is accepted */
	while( !g_ApReadyReply );

	for( i = 0 ; i < BLK_COUNT ; i++)
	{		
		// TX Buffer pointer setting
		txdata = (u16 *)( MSM_TX_BUFFER + (i * MODEM_TX_BUFLEN) );
		modemtx_buf = (u16 *)(MSM_TX_Queue);

		for (length = 0 ; length < MODEM_TX_BUFLEN / 2; length++)
		{
			*(u16 *)(modemtx_buf ++) = *(txdata++);
		}
		
//		MODEM_PRINTF("<--- 0x%x(MSM to AP)\n", MODEM_MSM_TX_DATA_XFER);
//		MODEM_PRINTF(" ------------------------\n");

		MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_AP_RX_DATA_XFER);

		/* Start Timer */
		StartTimer(0);
		
		while( !g_ApReadDoneReply );

		g_ApReadDoneReply = 0;

		/* Stop Timer */
		uElapsedTime = StopTimer(0);
		
		MODEM_PRINTF("(%3.1f Mbps, %d us)\n", ((float)( 8 * MODEM_TX_BUFLEN )/ (float)uElapsedTime), uElapsedTime);
		MODEM_PRINTF(" Block Count = %d\n", i+1);
	}

	MODEM_PRINTF("<--- 0x%x(MSM to AP)\n", MODEM_MSM_TX_AP_RX_XFER_DONE);
	MODEM_PRINTF(" ------------------------\n");
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_AP_RX_XFER_DONE);
	MODEM_PRINTF("MODEM TX END\n");
}

/**
* {Test_AP_DMA_TX_Performance}
*
* @brief			Test MSM RX -> AP DMA TX.
* @description	Test Max DMA Write performance of MODEM I/F.\n
*				Use 'Test_MODEM_RX_AP_DMA_TX_Performance' in MSM side B'd.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_AP_DMA_TX_Performance(void)
{
	int i = 0;
	
	g_DmaDoneATx = 0;

	g_APDMAWriteReady = 0;
	g_APDMAWriteDone = 0;
	g_ApDMAWriteDoneReply = 0;

	MODEM_PRINTF("\n['Test_MODEM_RX_AP_DMA_TX_Performance' MUST be called in MSM B'd side first,");
	MODEM_PRINTF("\n  Press any key when it's ready]");
	while( !GetKey() );

	MODEM_PRINTF("\n > Select  AP's modem DMA port(0~1) ");
	ucModemDMAPort = (u8)UART_GetIntNum();
//	ucModemDMAPort = 0;
	MODEM_PRINTF("\n");

	if( (ucModemDMAPort != 0) && (ucModemDMAPort != 1) )
	{
		MODEM_PRINTF(" --> Error!! You have entered invalid DMA port number!!\n");
		return;
	}

	// Modem i/f setting
	MODEMIF_AP_Init(true, true, true);

	// Modem I/F DMA setup
	MODEMIF_SET_APDMA(MODEMIF_AP_TX, (MODEMIF_DMA_REQ_NUM)(MODEMIF_TX_0 + ucModemDMAPort), 1);	// Enable AP DMA TX0 Request
	MODEMIF_SET_DMA_REQ_ADDR(MODEMIF_AP_TX, (MODEMIF_DMA_REQ_NUM)(MODEMIF_TX_0 + ucModemDMAPort), DMA_TX_ADDR0 + ucModemDMAPort*2);

	/* set to non-secure */
	MODEMIF_DMA_SetNonSecure();
	
	// for S5PV210
	// Enable PDMA1 interrupt for MSM_REQ 0 ~ 3
	INTC_SetVectAddr(NUM_PDMA1,  Isr_AP_DMA_TX_Done);
	INTC_Enable(NUM_PDMA1);

	// DMA Init & Setup
	DMA_SetCh(DMA_10, &oAPTx0);	
	DMA_InitCh(HWORD, (DREQ_SRC)(MSM_REQ0 + ucModemDMAPort), DMA_M2P, SINGLE, &oAPTx0);
	
	// DMA start
	MODEM_PRINTF("[AP DMA TX START]\n");

	for(i = 0; i < DMA_COUNT; i++ )
	{
		/* 1. DMA Start */
		DMA_StartCh(AP_TX_BUFFER, AP_TX_Queue, AP_DMA_TX_BUFLEN, &oAPTx0);

//		MODEM_PRINTF("\n------------------------\n");
//		MODEM_PRINTF("[AP DMA TX WAITING...]\n");

		/* 2. Interrupt request to MODEM I/F */
		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_RX_AP_DMA_TX_READY);

		while( !g_APDMAWriteReady );

		g_APDMAWriteReady = 0;

		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_RX_AP_DMA_TX_XFER_DONE);
		
		while( !g_ApDMAWriteDoneReply );
		
		g_ApDMAWriteDoneReply = 0;

		g_DmaDoneATx++;
//		MODEM_PRINTF("[%dth DMA Data Transfer Success]\n", i+1);
		MODEM_PRINTF("[%dth DMA]\n", i+1);
	}
	
	MODEM_PRINTF("\n ------------------------\n");
	MODEM_PRINTF("0x%x(AP to MSM) ---> \n", MODEM_MSM_RX_AP_DMA_TX_END);
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_RX_AP_DMA_TX_END);
	
	MODEM_PRINTF("cnt = 0x%x, dma_cnt = 0x%x\n", g_DmaDoneATx, DMA_COUNT);
	MODEM_PRINTF("[AP DMA TX END]\n");
}	

/**
* {Test_MODEM_RX_AP_DMA_TX_Performance}
*
* @brief			Test MSM RX -> AP DMA TX.
* @description	Test Max DMA Write performance of MODEM I/F.\n
*				Use 'Test_AP_DMA_TX_Performance' in AP side B'd.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_MODEM_RX_AP_DMA_TX_Performance(void)
{
	u16 length = 0;
	u16 i = 0;
	u16	*rxdata = NULL;
	u16 *modemrx_buf = NULL;
	u16 temp = 0;
	
	u32 uElapsedTime = 0;

	g_DmaDoneMRx = 0;
	
	g_ModemDMAReadReady = 0;
	g_ModemDMAReadReadyReply = 0;
	g_ModemDMAReadDone = 0;
	g_ModemDMAReadDoneReply = 0;

	MODEM_PRINTF("MODEM RX TEST!!!\n");
	
	MODEM_PRINTF("\n > Select  AP's modem DMA port(0~1) ");
	ucModemDMAPort = (u8)UART_GetIntNum();
	MODEM_PRINTF("\n");
	if( (ucModemDMAPort != 0) && (ucModemDMAPort != 1) )
	{
		MODEM_PRINTF(" --> Error!! You have entered invalid DMA port number!!\n");
		return;
	}
	
	MODEMIF_MODEM_Init(true, true);

	// MSM RX stands by
	MODEM_PRINTF("[MSM RX STANDING BY]\n");

	for(i=0 ; i<DMA_COUNT ; i++)
	{
		while( !g_ModemDMAReadReady );

		g_ModemDMAReadReady = 0;
		
		// TX Buffer pointer setting
		rxdata = (u16 *)(MSM_RX_Queue + AP_DMA_TX_BUFLEN);
		modemrx_buf = (u16 *)( MSM_RX_BUFFER );
		
		for (length = 0 ; length < AP_DMA_TX_BUFLEN / 2; length++)
		{
			*(u16 *)(modemrx_buf ++) = *(rxdata++);
		}

//		MODEM_PRINTF("[MSM Request DMA]\n");
		
		/* Start Timer */
		StartTimer(0);

		/* 3. Request to DMA */
		temp = MODEMIF_Inp16(DPSRAM_BASE_MODEM + DMA_TX_ADDR0 + ucModemDMAPort*2);
		
		while( !g_ModemDMAReadDone );

		g_ModemDMAReadDone = 0;
		
		/* Stop Timer */
		uElapsedTime = StopTimer(0);
		
		MODEM_PRINTF("(%3.1f Mbps, %d us)\n", ((float) 8 * AP_DMA_TX_BUFLEN / uElapsedTime), uElapsedTime);
		MODEM_PRINTF(" Block Count = %d\n", i+1);

		g_DmaDoneMRx++;
		
		MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_RX_AP_DMA_TX_READY);
		
	}

	while( !((g_DmaDoneMRx == DMA_COUNT) && (g_ModemDMAReadDoneReply == 1)) );

	MODEM_PRINTF("MODEM RX DMA PERFORMANCE TEST END\n");
}

/**
* {Test_AP_DMA_RX_Performance}
*
* @brief			Test MSM TX -> AP RX.
* @description	Test Max DMA Read performance of MODEM I/F.\n
*				Use 'Test_MODEM_TX_AP_DMA_RX_Performance' in MSM side B'd.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_AP_DMA_RX_Performance(void)
{
	int i = 0;
	
	g_DmaDoneARx = 0;
	
	g_ApDMAReadReadyReply = 0;
	g_APDMAReadReady = 0;
	g_APDMAReadDone = 0;
	g_ApDMAReadDoneReply = 0;
	
	MODEM_PRINTF("\n > Select  AP's modem DMA port(0~1) ");
	ucModemDMAPort = (u8)UART_GetIntNum();
	MODEM_PRINTF("\n");
	if( (ucModemDMAPort != 0) && (ucModemDMAPort != 1) )
	{
		MODEM_PRINTF(" --> Error!! You have entered invalid DMA port number!!\n");
		return;
	}

	// Modem i/f setting
	MODEMIF_AP_Init(true, true, true);

	// Modem I/F DMA setup
	
	MODEMIF_SET_DMA_REQ_ADDR(MODEMIF_AP_RX, (MODEMIF_DMA_REQ_NUM)(MODEMIF_RX_0 + ucModemDMAPort), DMA_RX_ADDR0 + ucModemDMAPort*2);
	MODEMIF_SET_APDMA(MODEMIF_AP_RX, (MODEMIF_DMA_REQ_NUM)(MODEMIF_RX_0 + ucModemDMAPort), 1);	// Enable AP DMA RX0 Request

	/* set to non-secure */
	MODEMIF_DMA_SetNonSecure();
	
	// for S5PV210
	// Enable PDMA1 interrupt for MSM_REQ 0 ~ 3
	INTC_SetVectAddr(NUM_PDMA1,  Isr_AP_DMA_RX_Done);
	INTC_Enable(NUM_PDMA1);

	// DMA Init & Setup
	DMA_SetCh(DMA_10, &oAPRx0);
	DMA_InitCh(HWORD, (DREQ_SRC)(MSM_REQ2 + ucModemDMAPort), DMA_P2M, SINGLE, &oAPRx0);

	// AP DMA RX stands by
	MODEM_PRINTF("[AP DMA RX STANDING BY]\n");

	// RX stand by...
	while( !g_ApDMAReadReadyReply );

	// DMA start
	MODEM_PRINTF("[AP DMA RX START]\n");

	for( i = 0 ; i < DMA_COUNT; i++ )
	{
		/* 1. DMA Start */
		DMA_StartCh(AP_RX_Queue, AP_RX_BUFFER, MODEM_DMA_TX_BUFLEN, &oAPRx0);

//		MODEM_PRINTF("\n------------------------\n");
//		MODEM_PRINTF("[AP DMA RX WAITING...]");
		
		/* 2. Interrupt request to MODEM I/F */
		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_TX_AP_DMA_RX_READY);

		while( !g_APDMAReadReady );

		g_APDMAReadReady = 0;

		MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_MSM_TX_AP_DMA_RX_XFER_DONE);
		while( !g_ApDMAReadDoneReply );
		
		g_ApDMAReadDoneReply = 0;
		
		g_DmaDoneARx++;		
		MODEM_PRINTF("[%dth DMA Data Transfer Success]\n", i+1);
	}
	
	while( !((g_DmaDoneARx == DMA_COUNT) && (g_APDMAReadDone == 1)) );

	MODEM_PRINTF("cnt = 0x%x, dma_cnt = 0x%x\n", g_DmaDoneARx, DMA_COUNT);
	MODEM_PRINTF("[AP DMA RX END]\n");
}	

/**
* {Test_MODEM_TX_AP_DMA_RX_Performance}
*
* @brief			Test MSM TX -> AP RX.
* @description	Test Max DMA Read performance of MODEM I/F.\n
*				Use 'Test_AP_DMA_RX_Performance' in AP side B'd.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_MODEM_TX_AP_DMA_RX_Performance(void)
{
	u16 length = 0;
	u16 i = 0;
	u16	*txdata = NULL;
	u16 *modemtx_buf = NULL;

	u32 uElapsedTime = 0;

	g_DmaDoneMTx = 0;
	
	g_ModemDMAWriteReady = 0;
	g_ModemDMAWriteReadyReply = 0;
	g_ModemDMAWriteDone = 0;
	g_ModemDMAWriteDoneReply = 0;

	MODEM_PRINTF("MODEM TX TEST!!!\n");
	
	MODEM_PRINTF("\n['Test_AP_DMA_RX_Performance' MUST be called in AP B'd side first,");
	MODEM_PRINTF("\n  Press any key when it's ready]");
	while( !GetKey() );

	MODEM_PRINTF("\n > Select  AP's modem DMA port(0~1) ");
	ucModemDMAPort = (u8)UART_GetIntNum();
	MODEM_PRINTF("\n");
	if( (ucModemDMAPort != 0) && (ucModemDMAPort != 1) )
	{
		MODEM_PRINTF(" --> Error!! You have entered invalid DMA port number!!\n");
		return;
	}

	// MODEM TX Buffer data
	txdata = (u16 *)(MSM_TX_BUFFER);
	
	for(length = 0 ; length < MODEM_DMA_TX_BUFLEN / 2; length++)
	{
		*(txdata + length) = length;
	}
	
	MODEMIF_MODEM_Init(true, true);

	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_AP_DMA_RX_INITIAL);

	for(i=0 ; i<DMA_COUNT ; i++)
	{
		while( !g_ModemDMAWriteReady );

		g_ModemDMAWriteReady = 0;
		
		// TX Buffer pointer setting
		txdata = (u16 *)( MSM_TX_BUFFER + MODEM_DMA_TX_BUFLEN );
		modemtx_buf = (u16 *)(MSM_TX_Queue);

		for (length = 0 ; length < MODEM_DMA_TX_BUFLEN /2 ; length++)
		{
			*(u16 *)(modemtx_buf ++) = *(txdata++);
		}

//		MODEM_PRINTF("[MSM Request DMA]\n");
		
		/* Start Timer */
		StartTimer(0);

		/* 3. Request to DMA */
		MODEMIF_Outp16(DPSRAM_BASE_MODEM + DMA_RX_ADDR0 + ucModemDMAPort*2, MODEM_MSM_TX_AP_DMA_RX_DATA_XFER);
		
		while( !g_ModemDMAWriteDoneReply );

		g_ModemDMAWriteDoneReply = 0;
		
		/* Stop Timer */
		uElapsedTime = StopTimer(0);
		
		MODEM_PRINTF("(%3.1f Mbps, %d us)\n", ((float) 8 * MODEM_DMA_TX_BUFLEN / uElapsedTime), uElapsedTime);
		MODEM_PRINTF(" Block Count = %d\n", i+1);

		g_DmaDoneMTx++;

		MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_AP_DMA_RX_READY);
		
	}
	
	MODEM_PRINTF("\n ------------------------\n");
	MODEM_PRINTF("<--- 0x%x(MSM to AP)\n", MODEM_MSM_TX_AP_DMA_RX_END);
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TX_AP_DMA_RX_END);

	MODEM_PRINTF("cnt = 0x%x, dma_cnt = 0x%x\n", g_DmaDoneMTx, DMA_COUNT);
	MODEM_PRINTF("MODEM DMA TX PERFORMANCE TEST END\n");
}

/**
* {Test_AP_Max_Read_Write_Performance}
*
* @brief			Test Max Read/Write Performance of MODEM I/F.\n
* @description	Test Max Read/Write Performance of MODEM I/F.\n
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_AP_Max_Read_Write_Performance(void)
{
	u32 length,i;

	u16	*txdata = NULL;
	u16 *aptx_buf = NULL;

	u16 *rx_buf = NULL;
	u16	*rx_data = NULL;

	u32 uElapsedTime = 0;
	
	//Message
	MODEM_PRINTF("[MODEM I/F MAX WRITE PERFORMANCE TEST]\n");

	// WRITE Buffer data	Init
	txdata = (u16 *)(AP_TX_BUFFER);
	for(length = 0 ; length < (AP_TX_BUFLEN * BLK_COUNT); length++)
	{
		*(txdata + length) = length;
	}
	
	for( i = 0 ; i < BLK_COUNT ; i++)
	{
		// TX Buffer pointer setting
		txdata = (u16 *)(AP_TX_BUFFER + ( i * AP_TX_BUFLEN ) );
		aptx_buf = (u16 *)(AP_TX_Queue);

		/* Start Timer */
		StartTimer(0);

		for (length = 0 ; length < AP_TX_BUFLEN / 2; length++)
		{
			*(u16 *)(aptx_buf ++) = *(txdata++);
		}
		
		/* Stop Timer */
		uElapsedTime = StopTimer(0);
		MODEM_PRINTF("(%3.1f Mbps, %d us)\n", ((float) 8 * AP_TX_BUFLEN / uElapsedTime), uElapsedTime);
		MODEM_PRINTF(" Block Count = %d\n", i+1);
	}
	MODEM_PRINTF("[MODEM I/F MAX WRITE PERFORMANCE TEST END]\n");

	//Message
	MODEM_PRINTF("[MODEM I/F MAX READ PERFORMANCE TEST]\n");

	// READ Buffer data Init
	rx_data = (u16 *)(AP_RX_Queue);
	for(length = 0 ; length < (MODEM_TX_BUFLEN * BLK_COUNT); length++)
	{
		*(rx_data + length) = length;
	}
	
	for( i = 0 ; i < BLK_COUNT ; i++)
	{
		// RX Buffer pointer setting
		rx_buf = (u16 *)(AP_RX_BUFFER + ( i * MODEM_TX_BUFLEN) );
		rx_data = (u16 *)(AP_RX_Queue);

		/* Start Timer */
		StartTimer(0);

		for (length = 0 ; length < MODEM_TX_BUFLEN / 2; length++)
			*(u16 *)(rx_buf++) = *(rx_data++);
		
		/* Stop Timer */
		uElapsedTime = StopTimer(0);
		MODEM_PRINTF("(%3.1f Mbps, %d us)\n", ((float) 8 * MODEM_TX_BUFLEN / uElapsedTime), uElapsedTime);
		MODEM_PRINTF(" Block Count = %d\n", i+1);
	}

	//Message
	MODEM_PRINTF("[MODEM I/F MAX READ PERFORMANCE TEST END]\n");
	
}

/**
* {Test_AP_CommandToMSMToReadDPSRAM}
*
* @brief			Concurrent Test.
* @description	AP commands to read the data in DPSRAM to MODEM.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_AP_CommandToMSMToReadDPSRAM(void)
{
	u16 length = 0;
	
	u16	*txdata = NULL;
	u16 *aptx_buf = NULL;

	u16 *rx_buf = NULL;
	u16 *rx_data = NULL;
	
	g_ModemReadyReply = 0;

	//Message
	MODEM_PRINTF("[MODEM_AP_TO_MSM_TO_READ]\n");

	// TX Buffer pointer setting
	txdata = (u16 *)(AP_TX_BUFFER);
	aptx_buf = (u16 *)(AP_TX_Queue);

	for (length = 0 ; length < AP_TX_BUFLEN ; length++)
	{
		*(u16 *)(aptx_buf ++) = *(txdata++);
	}

	// Modem i/f setting
	MODEMIF_AP_Init(true, true, true);

	MODEM_PRINTF("['Test_MSM_CommandToAPToReadDPSRAM' MUST be called in MSM B'd Side at the same time ");
	MODEM_PRINTF("Press any key when it's ready]");
	while( !GetKey() );
	
	/* 1. Interrupt request to MSM */
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_AP_TO_MSM_TO_READ);

	while( !g_ModemReadyReply );
	
	rx_buf = (u16 *)(AP_RX_BUFFER);
	rx_data = (u16 *)(AP_RX_Queue);

	for (length = 0 ; length < MODEM_TX_BUFLEN ; length++)
	*(u16 *)(rx_buf++) = *(rx_data++);

	//Message
	MODEM_PRINTF("[MODEM_AP_TO_MSM_TO_READ DONE]\n");
}	

/**
* {Test_Modem_CommandsToReadToAP}
*
* @brief			Concurrent Test.
* @description	Modem commands to read the data in DPSRAM to AP.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_MSM_CommandToAPToReadDPSRAM(void)
{
	u16 length = 0;
	
	u16	*txdata = NULL;
	u16 *modemtx_buf = NULL;

	u16 *rx_buf = NULL;
	u16 *rx_data = NULL;
	
	g_ApReadyReply	 = 0;

	//Message
	MODEM_PRINTF("[MODEM_MSM_TO_AP_TO_READ]\n");


	// MODEM TX Buffer pointer setting
	txdata = (u16 *)(MSM_TX_BUFFER);
	modemtx_buf = (u16 *)(MSM_TX_Queue);

	for (length = 0 ; length < MODEM_TX_BUFLEN; length++)
	{
		*(u16 *)(modemtx_buf ++) = *(txdata++);
	}
	
	MODEMIF_MODEM_Init(true, true);

	MODEM_PRINTF("['Test_AP_CommandToMSMToReadDPSRAM' MUST be called in AP B'd Side at the same time ");
	MODEM_PRINTF("Press any key when it's ready]");
	while( !GetKey() );
	
	/* 1. Interrupt request to MODEM I/F */
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TO_AP_TO_READ);
	
	/* 2. wait til Interrupt request to MODEM I/F is accepted */
	while( !g_ApReadyReply );

	rx_buf = (u16 *)(MSM_RX_BUFFER);
	rx_data = (u16 *)(MSM_RX_Queue);

	for (length = 0 ; length < AP_TX_BUFLEN ; length++)
	*(u16 *)(rx_buf++) = *(rx_data++);
	
	MODEM_PRINTF("MODEM TX END\n");
}	

/**
* {Test_AP_CommandsToWriteToModem}
*
* @brief			Concurrent Test.
* @description	AP commands to write the data in DPSRAM to MODEM.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_AP_CommandToMSMToWriteDPSRAM(void)
{
	u16 length = 0;
	
	u16	*txdata = NULL;
	u16 *aptx_buf = NULL;

	u16 *rx_buf = NULL;
	u16 *rx_data = NULL;
	
	g_ModemReadyReply = 0;

	//Message
	MODEM_PRINTF("[MODEM_AP_TO_MSM_TO_WRITE]\n");
	
	// Modem i/f setting
	MODEMIF_AP_Init(true, true, true);

	MODEM_PRINTF("['Test_MSM_CommandToAPToWriteDPSRAM' MUST be called in MSM B'd Side at the same time ");
	MODEM_PRINTF("Press any key when it's ready]");
	while( !GetKey() );
		
	/* 1. Interrupt request to MSM */
	MODEMIF_Outp16(DPSRAM_BASE_AP + INT2MSM_ADR, MODEM_AP_TO_MSM_TO_WRITE);

	/* 2. wait til Interrupt request to MSM is accepted */
	while( !g_ModemReadyReply );

	// TX Buffer pointer setting
	txdata = (u16 *)(AP_TX_BUFFER);
	aptx_buf = (u16 *)(AP_TX_Queue);

	for (length = 0 ; length < AP_TX_BUFLEN ; length++)
	{
		*(u16 *)(aptx_buf ++) = *(txdata++);
	}
	
	//Message
	MODEM_PRINTF("[MODEM_AP_TO_MODEM_TO_READ DONE]\n");
}	

/**
* {Test_Modem_CommandsToWriteToAP}
*
* @brief			Concurrent Test.
* @description	Modem commands to write the data in DPSRAM to AP.
* @param[in]	(void)
* @return		void.
* @version		2009.07.17
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void Test_MSM_CommandToAPToWriteDPSRAM(void)
{
	u16 length = 0;
	
	u16 *rx_buf = NULL;
	u16 *rx_data = NULL;

	g_ApReadyReply	 = 0;
	
	//Message
	MODEM_PRINTF("[MODEM_MSM_TO_AP_TO_WRITE]\n");
	
	MODEMIF_MODEM_Init(true, true);

	MODEM_PRINTF("['Test_AP_CommandToMSMToWriteDPSRAM' MUST be called in AP B'd Side at the same time ");
	MODEM_PRINTF("Press any key when it's ready]");
	while( !GetKey() );
	
	/* 1. Interrupt request to MODEM I/F */
	MODEMIF_Outp16(DPSRAM_BASE_MODEM + INT2AP_ADR, MODEM_MSM_TO_AP_TO_WRITE);
	
	/* 2. wait til Interrupt request to MODEM I/F is accepted */
	while( !g_ApReadyReply );

	rx_buf = (u16 *)(MSM_TX_Queue);
	rx_data = (u16 *)(MSM_TX_BUFFER);

	for (length = 0 ; length < AP_TX_BUFLEN ; length++)
		*(u16 *)(rx_buf++) = *(rx_data++);
			
	MODEM_PRINTF("MODEM TX END\n");
}	

//======================= Concurrent & Performance Functions ===========================//

//////////////////////////////////////////////////////////////////////////////
void MODEM_Test(void)
{
	int i=0, nSel;

	const testFuncMenu modemif_function[]=
	{
		0,                          			"Exit",
		TestSFRModemIF,			"MODEM IF SFR TEST",
		Test_AP_TX_INTR,			"Test XmsmIRQn Interrupt signal",
		Test_MODEM_TX_INTR,		"Test MODEM TX to test AP's Modem Interrupt	",
		Test_MODEM_SetClear_INTR,	"Test AP's Interrupt Set & Clear in Modem side",
		0,							"---------------------------------------",

		/* TX / RX TEST */
		Test_AP_TX,					"Test AP TX	",
		Test_MODEM_RX,				"Test MODEM RX    ",

		Test_AP_RX,					"Test AP RX	",
		Test_MODEM_TX,				"Test MODEM TX	",
		0,							"---------------------------------------",

		/* DMA TX / RX TEST */
		Test_AP_DMA_TX,			"Test AP TX using DMA	",
		Test_MODEM_DMA_RX,	"Test MODEM RX using DMA   ",

		Test_AP_DMA_RX,			"Test AP RX using DMA	",
		Test_MODEM_DMA_TX,	"Test MODEM TX using DMA	",
		0,							"---------------------------------------",

		/* PERFORMANCE TEST */
		Test_AP_TX_Performance,				"Test AP TX Performance in AP B'd Side",
		Test_MODEM_RX_AP_TX_Performance,	"Test AP TX Performance in MSM B'd Side",

		Test_AP_RX_Performance,				"Test AP RX Performance in AP B'd Side",
		Test_MODEM_TX_AP_RX_Performance,	"Test AP RX Performance in MSM B'd Side",

		Test_AP_DMA_TX_Performance,					"Test AP DMA TX Performance in AP B'd Side",
		Test_MODEM_RX_AP_DMA_TX_Performance,	"Test AP DMA TX Performance in MSM B'd Side",

		Test_AP_DMA_RX_Performance,					"Test AP DMA RX Performance in AP B'd Side",
		Test_MODEM_TX_AP_DMA_RX_Performance,	"Test AP DMA RX Performance in MSM B'd Side",
		0,							"---------------------------------------",

		/* MAX READ/WRITE PERFORMANCE TEST */
		Test_AP_Max_Read_Write_Performance,		"Test Modem I/F Max Read/Write Performance in AP B'd Side",
		
		/* DPSRAM R / W TEST */
		Test_Display_MEM,			"Dump Memory	",
		Test_DPSRAM,				"Test DPSRAM (AP side)	",
		Test_DPSRAM_OneTime,		"Test DPSRAM (Modem side)	",
		0,							"---------------------------------------",

		/* SIMULTANEOUS DPSRAM ACCESS TEST */
		Test_AP_CommandToMSMToReadDPSRAM,	"AP commands to MSM to read DPSRAM",
		Test_MSM_CommandToAPToReadDPSRAM,	"MSM commands to AP to read DPSRAM",
		
		Test_AP_CommandToMSMToWriteDPSRAM,	"AP commands to MSM to write DPSRAM",
		Test_MSM_CommandToAPToWriteDPSRAM,	"MSM commands to AP to write DPSRAM",
		0,							"---------------------------------------",

		/* CLOCK GATING TEST */
		Test_AP_Tx_CLK_Gating,				"Test AP TX CLK Gating ",
		Test_MODEM_Rx_CLK_Gating,		"Test MODEM RX CLK Gating	 ",
		
		Test_AP_Rx_CLK_Gating,				"Test AP RX CLK Gating ",
		Test_MODEM_Tx_CLK_Gating,		"Test MODEM TX CLK Gating	 ",
		0,							0,
	};
	
	while (1)
	{
		MODEM_PRINTF("\n");
		for (i=0; (int)(modemif_function[i].desc)!=0; i++)
		{
			MODEM_PRINTF("%2d: %s\n", i,modemif_function[i].desc);
		}

		MODEM_PRINTF("==========================================================\n");			
		MODEM_PRINTF("\nSelect the function to test : ");

		nSel = UART_GetIntNum();

		MODEM_PRINTF("\n");

		if (nSel == 0)
			break;
		if (nSel>0 && nSel<(sizeof(modemif_function)/8-1))
			 (modemif_function[nSel].func)();
	}		
}

