/**************************************************************************************
*
*	Project Name : S5PV210 FPGA Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for verifying functions of the S5PV210 FPGA
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------
* 
*	File Name : keypad_test.c
*  
*	File Description : This file implements the API functons for KEYPAD
*
*	Author	: Jongseok,Park
*	Dept. : AP Development Team
*	Created Date : 2009/04/16
*	Version : 0.2
*
*	Author : Sung-Hyun, Na
*	Dept. : AP Development Team
*	Created Date : 2008/11/03
*	Version : 0.1 
* 
*	History
*	- Version 0.1
*           ->Created(Sung-Hyun, Na 2008/11/03) for S5PC100  Validation
*	- Version 0.2
*           -> Modified(JongSeok,Park 2009/03/25) for S5PV210 FPGA verification
**************************************************************************************/


/* Include Files */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "keypad.h"
#include "def.h"
#include "option.h"
#include "library.h"
#include "system.h"
#include "sysc.h"
#include "intc.h" 
#include "gpio.h"



typedef struct testFuncKEYPAD {
	bool (*func)(KEYIF_Configuration *); 
	const char *desc;
} testFuncKEYPAD;

bool KEYPAD_Configuration(KEYIF_Configuration* pConfiguration);
void KEYPAD_DispTestCon(KEYIF_Configuration* pConfiguration);
void KEYPADConf_Port(KEYIF_Configuration* pConfiguration);
void KEYPAD_Sel_Wakeup(KEYIF_Configuration* pConfiguration);
void KEYPADConf_FilterCNT(KEYIF_Configuration* pConfiguration);
void KEYPADConf_DBFilter(KEYIF_Configuration* pConfiguration);
void KEYPADConf_INTCond(KEYIF_Configuration* pConfiguration);
void KEYPADConf_DFCNT(KEYIF_Configuration* pConfiguration);

void KEYPAD_Test(void);
//bool KEYPAD_HighToLowScan(KEYIF_Configuration* pConfiguration);
void KEYPAD_LowToHighScan(KEYIF_Configuration* pConfiguration);



void KEYPAD_Test(void)
{
	u32 uCountFunc=0;
	s32 iSel=0;
	KEYIF_Configuration oConfiguration;
	const testFuncKEYPAD g_aKEYPADTestFunc[] =
	{	
		{ 0,								"Exit"},
		{ KEYPAD_SFR_Test,				"SFR Test"},
		{ KEYPAD_Configuration,			"Configuration"},
		{ KEYPAD_HighToLowScan,		"High To Low Edge(Pull-Up) Scan Test"},
		{ KEYPAD_Set8by8,				"Set 8x8"},
		{ KEYPAD_Set14by8,				"Set 14x8"},
		//KEYPAD_LowToHighScan,			"Low to High(Pull-Down) Scan Test\n",
		//KEYPAD_KeySTOPWakeUpTest,		"Stop Wake Up Test",
		//KEYPAD_KeySLEEPWakeUpTest,	"Sleep Wake Up Test",
		{0,0}
	};
	oConfiguration = KEYPAD_DefaultConfiguration();

	if(gKEYPAD_Is8by8Mode != TRUE)
	{
		oConfiguration.m_eColPort		= KeyPort1;
		oConfiguration.m_eRowPort		= KeyPort1;
	}

	KEYPAD_ClosePort(KeyPort0, KeyPort0);
	KEYPAD_ClosePort(KeyPort1, KeyPort1);
	
	while(1)
	{
		KEYPAD_DispTestCon(&oConfiguration);			
		UART_Printf("\n\n================== KEYPAD Function Test =====================\n\n");	
		for (uCountFunc=0; (u32)(g_aKEYPADTestFunc[uCountFunc].desc)!=0; uCountFunc++)
			UART_Printf("%2d: %s\n", uCountFunc, g_aKEYPADTestFunc[uCountFunc].desc);

		UART_Printf("\nSelect the function to test : ");
		iSel =UART_GetIntNum();
		UART_Printf("\n");
		
		if(iSel == 0) 
			break;
		if (iSel>=0 && iSel<(sizeof(g_aKEYPADTestFunc)/8-1))
			(g_aKEYPADTestFunc[iSel].func) (&oConfiguration);		
	}
}	

//////////
// Function Name : KEYPAD_HighToLowScan
// Function Description : This function test KEYPAD
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPAD_CallBack(u32 uVar)
{
	u32* pSet = (u32 *) uVar;
	*pSet = TRUE;
}

//////////
// Function Name : KEYPAD_HighToLowScan
// Function Description : This function test KEYPAD
// Input : NONE
// Output : NONE
// Version : v0.1
bool KEYPAD_HighToLowScan(KEYIF_Configuration* pConfiguration)
{
	volatile u8 ucDone = 0;
	u32 uCnt = 0;
	u32 uLEDCnt = 0;
	KEYPAD_Create(*pConfiguration, KEYPAD_CallBack, (u32) (&ucDone));
	Disp("\nKeyPad High to Low Edge Scan\n");
	Disp("Key Pressed : High to Low, 		Key Released : Low to High\n");
	Disp("Press Any KeyPad .......\n");
	Disp("If you want to exit, please press any key\n");
	do
	{
		uCnt++;

		#if 1 //jspark
		if (uCnt%0x800==0)
		{
			DisplayLED(uLEDCnt);		
			uLEDCnt++; 
			if (uLEDCnt>7) uLEDCnt=0;
		}	
		#else
		if ( uCnt % 800000 < 400000)	
			DisplayLED(0xe);				
		else 
			DisplayLED(0xf);		
		#endif
		
		if(ucDone == TRUE)
		{
			#if 0
			INTC_Disable(NUM_KEYPAD);
			Disp("========================================\n");
			#if(KEYHW == KEY_Falling_Press)
	 		KEYPAD_ScanFallingHi_Z();
			//KEYPAD_FallingScan();
			#elif(KEYHW == KEY_Rising_Press)
			KEYPAD_RisingScan();
			#else
			#endif
			#endif /* 0 */
			ucDone = 0;
			INTC_Enable(NUM_KEYPAD);
		}	
	}while(!UART_GetKey());
	DisplayLED(0x0);		
	KEYPAD_Close();

	return AUTOTEST_OK;		/* for no warning */
}


bool KEYPAD_Set8by8(KEYIF_Configuration* pConfiguration)
{

	gKEYPAD_Is8by8Mode = TRUE;
	KEYPAD_ROWNUM = 8;
	
	pConfiguration->m_eColPort		= KeyPort0;
	pConfiguration->m_eRowPort		= KeyPort0;

	return AUTOTEST_OK;		/* for no warning */
}

bool KEYPAD_Set14by8(KEYIF_Configuration* pConfiguration)
{

	gKEYPAD_Is8by8Mode = FALSE;
	KEYPAD_ROWNUM = 14;
	
	pConfiguration->m_eColPort		= KeyPort1;
	pConfiguration->m_eRowPort		= KeyPort1;

	return AUTOTEST_OK;		/* for no warning */
}


//////////
// Function Name : KEYPAD_LowToHighScan
// Function Description : This function test KEYPAD
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPAD_LowToHighScan(KEYIF_Configuration* pConfiguration)
{
	volatile u8 ucDone = 0;
	KEYPAD_Create(*pConfiguration, KEYPAD_CallBack, (u32)(&ucDone));
	Disp("\nKeyPad High to Low Edge Scan\n");
	Disp("Key Pressed : Low to High, 		Key Released : High to Low\n");
	Disp("Press Any KeyPad .......\n");
	Disp("If you want to exit, please press any key\n");
	do
	{
		if(ucDone == TRUE)
		{
			Disp("\nKeyPad Scan\n");			
			UART_Printf("Press Any KeyPad .......\n");
			Disp("If you want to exit, please press any key\n");
			ucDone = 0;
		}		
	}while(!UART_GetKey());
	INTC_Disable(NUM_KEYPAD);
	KEYPAD_Close();								//Close KEYPAD GPIO Port
}

//////////
// Function Name : KEYPAD_DispTestCon
// Function Description : This function diplay test condition of KeyPad.
// Input : KEYIF_Configuration : Keypad Inforamation
// Output : NONE
// Version : v0.1
void KEYPAD_DispTestCon(KEYIF_Configuration* pConfiguration)
{
	UART_Printf("\n================== KEYPAD Configuration =====================\n\n");	
	UART_Printf("Key Port Column : ");
	if( pConfiguration->m_eColPort == KeyPort0 )
		UART_Printf(" 0(EINT16~23)\n");
	else
		UART_Printf(" 1(MSM Addr0, data0~6)\n");

	UART_Printf("Key Port Row : ");
	if( pConfiguration->m_eRowPort == KeyPort0 )
		UART_Printf(" 0(EINT24~31)\n");
	else
		UART_Printf(" 1(MSM data7~15, ConSignals)\n");
	
	UART_Printf("Internal Register Status : ");
	if (pConfiguration->m_ePullUpDn == eGPUDdis)
		Disp("Disable\n");
	else if (pConfiguration->m_ePullUpDn == eGPDen)
		Disp("Pull Down\n");
	else if (pConfiguration->m_ePullUpDn == eGPUen)
		Disp("Pull Up\n");
	UART_Printf("Interrupt Condition :  ");
	if( pConfiguration->m_eIntSource == Key_Falling )
		UART_Printf(" Falling Edge\n");
	else if( pConfiguration->m_eIntSource == Key_Rising)
		UART_Printf(" Rising Edge\n");
	else
		UART_Printf(" Both Edge\n");
	UART_Printf("Debouce Filter : ");
	if( pConfiguration->m_eDFilterSource == DF_RTCCLK )
		UART_Printf(" RTC CLK\n");
	else
		UART_Printf(" OSC CLK\n");
	UART_Printf("Filter Enable : ");
	if( pConfiguration->m_uFilterEn == true )
		UART_Printf(" Enable\n");
	else
		UART_Printf(" Disable\n");
	UART_Printf("Filter Value : %d\n",pConfiguration->m_uFilterVal);
	UART_Printf("Wake UP source : ");
	if( pConfiguration->m_uWakeup == 1 )
		UART_Printf(" Enable\n");
	else
		UART_Printf(" Disable\n");	
}


//////////
// Function Name : KEYPAD_Configuration
// Function Description : This function change KeyPad Configuration
// Input : KEYIF_Configuration : Keypad Inforamation
// Output : NONE
// Version : v0.1
bool KEYPAD_Configuration(KEYIF_Configuration* pConfiguration)
{
	u32 uCountFunc=0;
	s32 uSel;
	const testFuncKEYPAD aKEYPADConfigFunc[] =
	{
		KEYPADConf_Port,					"Select KeyPad Port\n",
		
		KEYPAD_Sel_Wakeup,				"Wake Up source",	
		KEYPADConf_FilterCNT,				"Filter Source Clock Counter",				
		KEYPADConf_DBFilter,				"Debounce Filter",	
		KEYPADConf_INTCond,				"Interrupt Source\n",
		
		KEYPADConf_DFCNT,				"Fiter Counter Value\n",
		0,0
	};
	UART_Printf("Selecet Configuration\n");
	UART_Printf("\n\n================== KeyPad Configuration =====================\n\n");
	while(1)
	{
		for (uCountFunc=0; (u32)(aKEYPADConfigFunc[uCountFunc].desc)!=0; uCountFunc++)
			UART_Printf("%2d: %s\n", uCountFunc, aKEYPADConfigFunc[uCountFunc].desc);
		UART_Printf("\nSelect the function to configure : ");
		uSel =UART_GetIntNum();
		UART_Printf("\n");
		if(uSel == (u32)-1)
			break;
		if (uSel>=0 && uSel<(sizeof(aKEYPADConfigFunc)/8-1))
			(aKEYPADConfigFunc[uSel].func) (pConfiguration);		
	}

	return AUTOTEST_OK;		/* for no warning */
}

//////////
// Function Name : KEYPADConf_Port
// Function Description : This function Select Key Pad Port
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPADConf_Port(KEYIF_Configuration* pConfiguration)
{
	u32 uSelRow, uSelCol, uIntLevel;		

	do
	{
		UART_Printf("\nSelect KEY Port Row");
		UART_Printf("\n0 : KEY PORT 0(EINT24~E31)			1 : KEY PORT 1(MSM)\n");
		uSelRow = UART_GetIntNum(); 
	}while(!(uSelRow == 0 || uSelRow == 1));
	do
	{
		UART_Printf("\nSelect KEY Port Column");
		UART_Printf("\n0 : KEY PORT 0(EINT16~E23)			1 : KEY PORT 1(MSM)\n");
		uSelCol = UART_GetIntNum(); 
	}while(!(uSelCol == 0 || uSelCol == 1));

	pConfiguration->m_eRowPort = (KEYPORT) uSelRow;
	pConfiguration->m_eColPort = (KEYPORT) uSelCol;

	do
	{
		UART_Printf("\nEnalbe Row Port Internal Pull Up, Pull Down : ");
		UART_Printf("\n0 : Disable	1 : Pull Down Enable	2 : Pull Up Enable\n");
		uIntLevel = UART_GetIntNum(); 
	}while(!(uIntLevel >= 0 || uIntLevel <= 2));
	pConfiguration->m_ePullUpDn = (GPIO_ePUD) uIntLevel;	
}


//////////
// Function Name : KEYPAD_Sel_Wakeup
// Function Description : This function Select Wake-UP 
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPAD_Sel_Wakeup(KEYIF_Configuration* pConfiguration)
{
	u32 uSel;	

	do{
		UART_Printf("\nEnable Wake Up Source");
		UART_Printf("\n0 : Disable[D],    1 : Enable\n");
		uSel = UART_GetIntNum(); 
	}while(!(uSel == 0 || uSel == 1));
	pConfiguration->m_uWakeup = uSel;	
}

//////////
// Function Name : KEYPADConf_FilterCNT
// Function Description : This function Select Filter Source Clock.
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPADConf_FilterCNT(KEYIF_Configuration* pConfiguration)
{
	u32 uSel = 0;
	do{
		UART_Printf("\nEnable the Filter Source Clock Counter?\n ");
		UART_Printf("\n0 : Disable(RTCCLK),    1 : Enable(OSCIN)[D]\n");
		uSel = UART_GetIntNum(); 	
	}while(!(uSel == 0 || uSel == 1));	
	pConfiguration->m_eDFilterSource = (DFSOURCE) uSel;	
}

//////////
// Function Name : KEYPADConf_INTCond
// Function Description : This function Select Filter Interrupt Condition
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPADConf_INTCond(KEYIF_Configuration* pConfiguration)
{
	u32 uSel = 0;		
	do{
		UART_Printf("\nSelect the KeyPad Interrupt Type : ");
		UART_Printf("\n1 : Falling Edge[D],    2 : Rising Edge,    3 : Both Edge \n");
		uSel = UART_GetIntNum(); 
	}while(!(uSel >= 1 || uSel <= 3));
	pConfiguration->m_eIntSource = (KEY_INTSOURCE) (uSel);	
}

//////////
// Function Name : KEYPADConf_FilterCNT
// Function Description : This function Enabel/Disable De-Bounce Fiter of KEYPAD
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPADConf_DBFilter(KEYIF_Configuration* pConfiguration)
{
	u32 uSel = 0;	
	do{
		UART_Printf("\nEnable the Debounce Filter?\n ");
		UART_Printf("\n0 : Disable,    1 : Enable[D]\n");
		uSel = UART_GetIntNum(); 	
	}while(!(uSel == 0 || uSel == 1));
	pConfiguration->m_uFilterEn= uSel;	
}

//////////
// Function Name : KEYPADConf_DFCNT
// Function Description : This function change Filter Counter Value
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPADConf_DFCNT(KEYIF_Configuration* pConfiguration)
{
	s32 uSel = 0;	
	while(1)
	{
		UART_Printf("\nSelect Debouce Filter Counter Value(0 ~ 1023) : ");
		uSel = UART_GetIntNum(); 
		if(uSel >= 0 && uSel<=1023)				//Default Falling Edge Type
		{
			pConfiguration->m_uFilterVal = uSel;
			break;
		}
		else 		
			UART_Printf("\nError!!!  It is not range of Debouce Filter CNT.\n");		
	}		
}


#define KEYPAD_NORMAL_TEST		0

#if KEYPAD_NORMAL_TEST
#define KEYPAD_MAX_KEY_INPUT		10
#else
//#define KEYPAD_MAX_KEY_INPUT		2
#define KEYPAD_MAX_KEY_INPUT		1
#endif

typedef struct
{
	int	Row;	/* Row */
	int	Col;	/* Col */
}KEYPAD_TESTVECTOR;

#if KEYPAD_NORMAL_TEST
const KEYPAD_TESTVECTOR gKEYPAD_KeyInputVector[KEYPAD_MAX_KEY_INPUT] =
{
	1,1,	/* 1st */
	2,2,
	3,3,
	4,4,
	5,5,
	6,6,
	7,7,
	8,8,
	
	8,1,
	1,8, /* 10th */
};
#else
const KEYPAD_TESTVECTOR gKEYPAD_KeyInputVector[KEYPAD_MAX_KEY_INPUT] =
{
	#if 0
	2,2,	/* 1st */
	3,3	/* 2nd */
	#endif /* 0 */
	2,2
};
#endif /* 0 */

static volatile bool	gKEYPAD_InterruptDetected = FALSE;
static volatile int	gKEYPAD_KeyInputVectorIndex = 0;
static volatile int	gKEYPAD_KeyInputErrorCnt = 0;

extern bool gKEYIF_IsKeyPressed;
extern u8 gKEYIF_Row;
extern u8 gKEYIF_Col;
extern bool gKEYIF_IsKeyEventDetected;


bool KEYPAD_KeyInpuTest(void)
{
	//volatile u8 ucDone = 0;
	u32 uCnt = 0;
	u32 uLEDCnt = 0;
	KEYIF_Configuration oConfiguration;
	//bool IsKeyEventDetected;
	//bool IsKeyPressed;
	u8 Row;
	u8 Col;
	u8 SWNum;

	gKEYPAD_InterruptDetected = FALSE;	/* not used */
	gKEYPAD_KeyInputVectorIndex = 0;
	gKEYPAD_KeyInputErrorCnt = 0;

	gKEYIF_IsKeyEventDetected = FALSE;		

	oConfiguration = KEYPAD_DefaultConfiguration();
	KEYPAD_ClosePort(KeyPort0, KeyPort0);
	KEYPAD_ClosePort(KeyPort1, KeyPort1);		

	KEYPAD_Create(oConfiguration, KEYPAD_CallBack, (u32) (&gKEYPAD_InterruptDetected));
	//Disp("\nKeyPad High to Low Edge Scan\n");
	//Disp("Key Pressed : High to Low, 		Key Released : Low to High\n");
	//Disp("Press Any KeyPad .......\n");
	//Disp("If you want to exit, please press any key\n");

	#if KEYPAD_NORMAL_TEST
	AUTO_preparing(-1,"After closing this message, Press KEYPAD\n\nas order printed on the KEYPAD : (1,1)~(8,8) (8,1) (1,8)");
	#else
	//AUTO_preparing(-1,"After closing this message, Press KEYPAD\n\nas order printed on the KEYPAD : (2,2) (3,3)");
	AUTO_preparing(-1,"After closing this message, Press KEYPAD\n\nas order printed on the KEYPAD : (2,2)");
	#endif /* 0 */

	while(gKEYPAD_KeyInputErrorCnt < 3 && gKEYPAD_KeyInputVectorIndex < KEYPAD_MAX_KEY_INPUT)
	{
		#if 0
		uCnt++;

		#if 1 //jspark
		if (uCnt%0x800==0)
		{
			DisplayLED(uLEDCnt);		
			uLEDCnt++; 
			if (uLEDCnt>7) uLEDCnt=0;
		}	
		#else
		if ( uCnt % 800000 < 400000)	
			DisplayLED(0xe);				
		else 
			DisplayLED(0xf);		
		#endif
		#endif /* 0 */
		
		//if(gKEYPAD_InterruptDetected == TRUE)
		if(gKEYIF_IsKeyEventDetected == TRUE)
		{
			Row = gKEYIF_Row;
			Col = gKEYIF_Col;
			//INTC_Disable(NUM_KEYPAD);
			//Disp("========================================\n");
			#if 0
			#if(KEYHW == KEY_Falling_Press)
	 		KEYPAD_ScanFallingHi_Z();
			//KEYPAD_FallingScan();
			#elif(KEYHW == KEY_Rising_Press)
			KEYPAD_RisingScan();
			#else
			#endif	
			ucDone = 0;
			#else
			//IsKeyEventDetected = KEYPAD_GetCheckedKeyEvent_8by8(&IsKeyPressed, &Row, &Col);
			//if(IsKeyEventDetected == TRUE && IsKeyPressed == TRUE)
			{
				if(Row == gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Row && 
					Col == gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Col)
				{
					gKEYPAD_KeyInputVectorIndex++;
					if(gKEYPAD_KeyInputVectorIndex < KEYPAD_MAX_KEY_INPUT)
					{
						SWNum = (gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Col - 1) * 8 + 
								gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Row;
						AUTO_preparing(-1,"OK! PRESS Next KEY : SW%d(%d, %d)", SWNum,
							gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Row,
							gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Col);
					}
				}
				else
				{
					gKEYPAD_KeyInputErrorCnt++;
					//AUTO_preparing(-1,"Please PRESS AGAIN KEYPAD test Number %d", gKEYPAD_KeyInputVectorIndex+1);
					SWNum = (gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Col - 1) * 8 + 
							gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Row;
					AUTO_preparing(-1,"ERROR! PRESS AGAIN : SW%d(%d, %d)", SWNum,
						gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Row,
						gKEYPAD_KeyInputVector[gKEYPAD_KeyInputVectorIndex].Col);
				}
			}
			#endif /* 0 */

			//gKEYPAD_InterruptDetected = FALSE;
			
			//INTC_Enable(NUM_KEYPAD);
			gKEYIF_IsKeyEventDetected = FALSE;
		}	
	}

	//DisplayLED(0x0);		
	KEYPAD_Close();

	if(gKEYPAD_KeyInputVectorIndex == KEYPAD_MAX_KEY_INPUT)
	{
		return AUTOTEST_OK;
	}
	else
	{
		return AUTOTEST_FAIL;
	}
}


bool KEYPAD_AutoTest(void)
{
	//return KEYPAD_SFR_Test();
	return KEYPAD_KeyInpuTest();
}


