/**************************************************************************************
*
*	Project Name : S5PV210 FPGA Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for verifying functions of the S5PV210 FPGA
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------
* 
*	File Name : keypad.c
*  
*	File Description : This file implements the API functons for KEYPAD
*
*	Author	: Jongseok,Park
*	Dept. : AP Development Team
*	Created Date : 2009/04/16
*	Version : 0.2
*
*	Author : Sung-Hyun, Na
*	Dept. : AP Development Team
*	Created Date : 2008/11/03
*	Version : 0.1 
* 
*	History
*	- Version 0.1
*           ->Created(Sung-Hyun, Na 2008/11/03) for S5PC100  Validation
*	- Version 0.2
*           -> Modified(JongSeok,Park 2009/03/25) for S5PV210 FPGA verification
**************************************************************************************/



#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "option.h"
#include "library.h"
#include "v210_sfr.h"

#include "gpio.h"
#include "intc.h"
#include "timer.h"

#include "keypad.h"
#include "SfrReadWriten.h"

#define KEYIFOutp32(offset, x) 			Outp32(KEYPADIF_BASE+offset, x)
#define KEYIFInp32(offset) 				Inp32(KEYPADIF_BASE+offset)

static KEYPAD_INFOR*	g_pKEYInfor;

bool	gKEYPAD_Is8by8Mode = TRUE;
u32 KEYPAD_ROWNUM=8;


REGINFOn		sRegInfoKEYIIF[] = 
{ 
    //01234567890123                 
	{"rKEYIFCON ",	KEYPADIF_BASE+0x0, 	5-1, 	RW, 0x0, DPDB, 0,0},		
	{"rKEYIFSTSCLR ",	KEYPADIF_BASE+0x4,	30-1,	RW, 0x0  , DPDB, 0,0},	
	{"rKEYIFCOL ",	KEYPADIF_BASE+0x8, 	16-1,	RW, 0xff00  , DPDB, 0,0},		
	{"rKEYIFROW ",	KEYPADIF_BASE+0xc,	14-1, 	RO, 0x0  , DPDB, 0,0},	
	{"rKEYIFFC ",		KEYPADIF_BASE+0x10,	10-1, 	RW, 0x0  , DPDB, 0,0}	
};


//////////
// Function Name : KEYPAD_GetInfor
// Function Description : This function return address of KEYPAD Information structure.
// Input : 	None
// Output : NONE
// Version : v0.1
KEYPAD_INFOR* KEYPAD_GetInfor(void)
{
	return (g_pKEYInfor);
}

//////////
// Function Name : KEYPAD_DefaultConfiguration
// Function Description : This function make default configuration of SMDK6410.
// Input : NONE
// Output : KEYIF_Configuration : Keypad Inforamation
// Version : v0.1
KEYIF_Configuration KEYPAD_DefaultConfiguration(void)
{
	KEYIF_Configuration oConfiguration;	
	#if 1							// For SMDK6410
	oConfiguration.m_eColPort		= KeyPort0;
	oConfiguration.m_eRowPort		= KeyPort0;	
	oConfiguration.m_ePullUpDn		= eGPUen;
	oConfiguration.m_eDFilterSource 	= DF_OSC;
	oConfiguration.m_uWakeup		= ENABLE;
//	oConfiguration.m_eIntSource	= Key_Rising;
	oConfiguration.m_eIntSource	= Key_Falling;
	oConfiguration.m_uFilterEn 		= true;
	
	oConfiguration.m_uColEn	 	= 0x0;
	//oConfiguration.m_uColEn	 	= 0xff; //COL : High-Z output (̻...)
	
	oConfiguration.m_uFilterVal 		= 0x3ff;	
	#else							// For Test
	oConfiguration.m_eColPort		= KeyPort0;
	oConfiguration.m_eRowPort		= KeyPort0;	
	oConfiguration.m_ePullUpDn		= eGPUDdis;
	oConfiguration.m_eDFilterSource 	= DF_OSC;
	oConfiguration.m_uWakeup		= true;
	oConfiguration.m_eIntSource		= Key_Falling;
	oConfiguration.m_uFilterEn 		= false;
	oConfiguration.m_uColEn	 		= 0x0;				
	oConfiguration.m_uFilterVal 		= 0x3ff;	
	#endif
	return oConfiguration;
}

//////////
// Function Name : KEYPAD_Create
// Function Description : This function Create KEYPAD IP 
// Input : 	pConfiguration - Configuration Structure
// 			callbackFn - Call Back Function on ISR of KEYPAD
//			uCallBackVar - parameter of Call Back Function on ISR of KEYPAD
// Output : NONE
// Version : v0.1
void KEYPAD_Create(KEYIF_Configuration pConfiguration, void (*callbackFn)(u32), u32 uCallBackVar)
{
	KEYPAD_INFOR* pInfor = (KEYPAD_INFOR*) malloc(sizeof(KEYPAD_INFOR));		
	u32 uCntCol, uCntRow;
	
	g_pKEYInfor = pInfor;
	
	memcpy((u32 *)(&pInfor->m_oConfiguration), (u32 *)(&pConfiguration), sizeof(KEYIF_Configuration));			
	pInfor->m_fnCallback		= callbackFn;
	pInfor->m_uCallbackVar	= uCallBackVar;			
	for ( uCntCol = 0; uCntCol < KEYPAD_COLNUM; uCntCol++)
	{
		for ( uCntRow = 0; uCntRow < KEYPAD_ROWNUM; uCntRow++)
		{
			pInfor->m_fnKeyISR[uCntCol][uCntRow] = NULL;
			pInfor->m_uVarKeyISR[uCntCol][uCntRow] = NULL;
		}
	}	
	KEYPAD_SetKEYIFCON(pConfiguration.m_uWakeup, pConfiguration.m_eDFilterSource, pConfiguration.m_uFilterEn, pConfiguration.m_eIntSource);	//Set KEYPAD
	KEYPAD_SetDFCNT(pConfiguration.m_uFilterVal);		//Set Debouncing Fileter Counter	
	KEYPAD_EnableCol(pConfiguration.m_uColEn);
	
	#if(KEYHW == KEY_Falling_Press)
	KEYPAD_ColumnOut(0x00);						//Column(Output value) Low
	#else
	KEYPAD_ColumnOut(0xff);							//Column(Output value) High	
	#endif
	
	KEYPAD_OpenPort(pConfiguration.m_eRowPort, pConfiguration.m_eColPort, pConfiguration.m_ePullUpDn);			//Open KEYPAD GPIO Port
	KEYPAD_STSCLR(0x3FFF,0x3FFF);							//Key Interrupt Clear
	
	INTC_ClearVectAddr();	
	INTC_SetVectAddr(NUM_KEYPAD,Isr_KEYPAD);
	INTC_Enable(NUM_KEYPAD);
	CalibrateDelay();
}

//////////
// Function Name : KEYPAD_Close
// Function Description : This function Close KEYPAD Ip
// Input : 	NONE
// Output : NONE
// Version : v0.1
void KEYPAD_Close(void)
{
	INTC_Disable(NUM_KEYPAD);
	KEYPAD_ClosePort(g_pKEYInfor->m_oConfiguration.m_eRowPort, g_pKEYInfor->m_oConfiguration.m_eColPort);	//Close KEYPAD GPIO Port
	free(g_pKEYInfor);
	g_pKEYInfor = NULL;	 
}

//////////
// Function Name : KEYPAD_SetKeyFn
// Function Description : This function Set CallBack Function each Key
// Input : 	usCol - Row Port Selection
// 			usRow - Column Port Selection
//			fnKey - CallBack Function 
//			uVar - CallBack Function Variable
// Output : NONE
// Version : v0.1
void KEYPAD_SetKeyFn(u8 usCol, u8 usRow, void(*fnKey)(u32), u32 uVar)
{
	if( usCol >= KEYPAD_COLNUM || usRow >= KEYPAD_ROWNUM )
		return;
	else
	{
		KEYPAD_INFOR* pInfor = KEYPAD_GetInfor();
		pInfor->m_fnKeyISR[usCol][usRow] = fnKey;
		pInfor->m_uVarKeyISR[usCol][usRow] = uVar;
	}
}

bool gKEYIF_IsKeyPressed;
u8 gKEYIF_Row;
u8 gKEYIF_Col;
bool gKEYIF_IsKeyEventDetected;


/*---------------------------------- APIs of KEY Interrupt --------------------------------*/
//////////
// Function Name : Isr_KEYPAD
// Function Description : This function is Interrupt Service Routine of KEYPAD
//									Following KEYPAD Scanning procedure
// Input : NONE
// Output : NONE
// Version : v0.1
void __irq Isr_KEYPAD(void)
{
	KEYPAD_INFOR* pInfor = KEYPAD_GetInfor();

	Disp("[Isr_KEYPAD]\n");
	
	INTC_Disable(NUM_KEYPAD);	
	//KEYPAD_STSCLR(0x3fff,0x3fff);	

	if(gKEYPAD_Is8by8Mode == TRUE)
	{
		gKEYIF_IsKeyEventDetected = KEYPAD_GetCheckedKeyEvent_8by8(&gKEYIF_IsKeyPressed, &gKEYIF_Row, &gKEYIF_Col);
	}
	else
	{
		gKEYIF_IsKeyEventDetected = KEYPAD_GetCheckedKeyEvent_14by8(&gKEYIF_IsKeyPressed, &gKEYIF_Row, &gKEYIF_Col);
	}
	
	if ( pInfor->m_fnCallback != NULL )
		pInfor->m_fnCallback(pInfor->m_uCallbackVar);	
	KEYPAD_STSCLR(0x3fff,0x3fff);	
	INTC_Enable(NUM_KEYPAD);	
 }


//////////
// Function Name : KEYPAD_FallingScan
// Function Description : This function scan KEYPad for Key Pressed when Falling Edge and Key Released
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPAD_FallingScan(void)
{
	u32 urKEYIFSTSCLR, urKEYIFROW;
	u32 uCol, uRow;
	u32 uColScan, uRowScan;
	urKEYIFSTSCLR = KEYPAD_ROW();		
	INTC_ClearVectAddr();
	if ( urKEYIFSTSCLR & 0xff )										//Key Pressed
	{
		for ( uCol = 0; uCol < KEYPAD_COLNUM ; uCol++)
		{
			uColScan = 0xff ^ (1 << uCol);
			KEYPAD_ColumnOut(uColScan);			
			Delay(KEY_ScanDelay);											//Must have Delay. 
			urKEYIFROW = KEYPAD_ROW();
			if (  urKEYIFROW != 0xff)
			{
				KEYPAD_STSCLR(0x3fff,0x3fff);						//Unwanted Interrupt Source is made. it must clear!						
				for(uRow = 0;uRow < KEYPAD_ROWNUM; uRow++)	//Scan X-Axis (COLUMN)
				{
					uRowScan = 0xff ^(1 << uRow);
					if ( (urKEYIFROW | uRowScan) != 0xff)
					{
						KEYPAD_INFOR* pInfor = KEYPAD_GetInfor();
						if (pInfor->m_fnKeyISR[uCol][uRow] != NULL )
						{
							pInfor->m_fnKeyISR[uCol][uRow](pInfor->m_uVarKeyISR[uCol][uRow]);
						}
						else
						{
							Disp("[%d, %d]\n", uCol, uRow);
							Disp("%d\n", (uCol * 8 + uRow));
						}						
					}				
				}				
			}
		}		
	}
	else if ( urKEYIFSTSCLR & 0xff00 )		//Key Released
	{		
		Disp("Some Key is Released!!\n");		
	}		
	KEYPAD_ColumnOut(0x00);
	Delay(100);
	KEYPAD_STSCLR(0x3fff,0x3fff);	
}

//////////
// Function Name : KEYPAD_ScanFallingHi_Z
// Function Description : This function scan KEYPad for Key Pressed when Falling Edge and Key Released
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPAD_ScanFallingHi_Z(void)
{
	u32 urKEYIFSTSCLR, urKEYIFROW;
	u32 uCol, uRow;
	u32 uColScan, uRowScan;
	urKEYIFSTSCLR = KEYPAD_ROW();	
	//UART_Printf("Read Interrupt Pending Register, Value is 0x%x\n", urKEYIFSTSCLR);
	KEYPAD_STSCLR(0x3fff,0x3fff);	
	INTC_ClearVectAddr();
	if ( urKEYIFSTSCLR & 0xff )										//Key Pressed
	{
		for ( uCol = 0; uCol < KEYPAD_COLNUM ; uCol++)
		{
			uColScan = 0xff ^ (1 << uCol);
			KEYPAD_EnableCol(uColScan);							//Col output Hi-Z enable.		
			Delay(KEY_ScanDelay);								//Must have Delay. 
			urKEYIFROW = KEYPAD_ROW();
			if (  urKEYIFROW != 0xff)
			{
				KEYPAD_STSCLR(0x3fff,0x3fff);						//Unwanted Interrupt Source is made. it must clear!						
				for(uRow = 0;uRow < KEYPAD_ROWNUM; uRow++)	//Scan X-Axis (COLUMN)
				{
					uRowScan = 0xff ^(1 << uRow);
					if ( (urKEYIFROW | uRowScan) != 0xff)
					{
						KEYPAD_INFOR* pInfor = KEYPAD_GetInfor();
						if (pInfor->m_fnKeyISR[uCol][uRow] != NULL )
						{
							pInfor->m_fnKeyISR[uCol][uRow](pInfor->m_uVarKeyISR[uCol][uRow]);
						}
						else
						{
							Disp("[%d, %d]\n", uCol, uRow);
							Disp("%d\n", (uCol * 8 + uRow));
						}						
					}				
				}				
			}
		}		
	}
	else if ( urKEYIFSTSCLR & 0xff00 )		//Key Released
	{		
		Disp("Some Key is Released!!\n");		
	}		
	KEYPAD_EnableCol(0x00);			
	Delay(KEY_ScanDelay);
	KEYPAD_STSCLR(0x3fff,0x3fff);	
}


//////////
// Function Name : KEYPAD_RisingScan
// Function Description : 
// Input : NONE
// Output : NONE
// Version : v0.1
void KEYPAD_RisingScan(void)
{
	u32 urKEYIFSTSCLR, urKEYIFROW;
	u32 uCol, uRow;
	u32 uColScan, uRowScan;
	urKEYIFSTSCLR = KEYPAD_ROW();	
	//UART_Printf("Read Interrupt Pending Register, Value is 0x%x\n", urKEYIFSTSCLR);
	KEYPAD_STSCLR(0x3fff,0x3fff);	
	INTC_ClearVectAddr();
	if ( urKEYIFSTSCLR & 0xff )				//Key Released
	{
		Disp("Some Key is Released!!\n");
	}
	else if ( urKEYIFSTSCLR & 0xff00 )		//Key Pressed
	{
		for ( uCol = 0; uCol < KEYPAD_COLNUM ; uCol++)
		{
			uColScan = (1 << uCol);
			KEYPAD_ColumnOut(uColScan);
			Delay(KEY_ScanDelay);					// Must have Delay. 
			urKEYIFROW = KEYPAD_ROW();
			if ( KEYPAD_ROW() != 0x0)
			{
				KEYPAD_STSCLR(0x3fff,0x3fff);											//Unwanted Interrupt Source is made. it must clear!						
				for(uRow = 0;uRow < KEYPAD_ROWNUM; uRow++)						//Scan X-Axis (COLUMN)
				{
					uRowScan = (1 << uRow);
					if ( (urKEYIFROW & uRowScan) != 0x0)
					{
						KEYPAD_INFOR* pInfor = KEYPAD_GetInfor();
						if (pInfor->m_fnKeyISR[uCol][uRow] != NULL )
						{
							pInfor->m_fnKeyISR[uCol][uRow](pInfor->m_uVarKeyISR[uCol][uRow]);
						}
						else
						{
							Disp("[%d, %d]\n", uCol, uRow);
							Disp("%d\n", (uCol * 8 + uRow));
						}									
					}				
				}				
			}
		}
	}
	KEYPAD_ColumnOut(0xff);
	Delay(100);		
	KEYPAD_STSCLR(0x3fff,0x3fff);	
}
/*---------------------------------- APIs of rKEYIF PORT ---------------------------------*/
//////////
// Function Name : KEYPAD_OpenPort
// Function Description : This function Selects KEYPAD Port between GPK and GPH, GPN and sets up them(HOST I/F)
// Input : 	eRowPort - Row Port Selection
// 			eColPort - Column Port Selection
//			usRowPullEn - Row Pull -Up Enable/Disabel
// Output : NONE
// Version : v0.1
void KEYPAD_OpenPort(KEYPORT eRowPort, KEYPORT eColPort, GPIO_ePUD eRowPullUDEn)
{
	if (eColPort == KeyPort0)		//Column Port : EINT(16~23)  I/F (Power Domain is ?)
	{		
		//Key Pad Column	Host I/F
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_0, 3);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_1, 3);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_2, 3);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_3, 3);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_4, 3);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_5, 3);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_6, 3);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_7, 3);	

		//Pull Up/Down Disable
		GPIO_SetPullUpDownEach(eGPIO_H2, eGPIO_0, 0);
		GPIO_SetPullUpDownEach(eGPIO_H2, eGPIO_1, 0);
		GPIO_SetPullUpDownEach(eGPIO_H2, eGPIO_2, 0);
		GPIO_SetPullUpDownEach(eGPIO_H2, eGPIO_3, 0);
		GPIO_SetPullUpDownEach(eGPIO_H2, eGPIO_4, 0);
		GPIO_SetPullUpDownEach(eGPIO_H2, eGPIO_5, 0);
		GPIO_SetPullUpDownEach(eGPIO_H2, eGPIO_6, 0);
		GPIO_SetPullUpDownEach(eGPIO_H2, eGPIO_7, 0);	

		UART_Printf("Key Pad Column Port : EINT(16~24)  Init!!!\n");
	}
	else 						//Column Port : MMC (Power Domain is VDD_MMC)
	{
		//Key Pad Column		MMC
		GPIO_SetFunctionEach(eGPIO_J1,eGPIO_5,3);  //col 0
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_0,3);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_1,3);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_2,3);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_3,3);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_4,3);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_5,3);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_6,3); //col 7

		//Pull Up/Down Disable
		GPIO_SetPullUpDownEach(eGPIO_J1,eGPIO_5,0);
		GPIO_SetPullUpDownEach(eGPIO_J2,eGPIO_0,0);		
		GPIO_SetPullUpDownEach(eGPIO_J2,eGPIO_1,0);
		GPIO_SetPullUpDownEach(eGPIO_J2,eGPIO_2,0);
		GPIO_SetPullUpDownEach(eGPIO_J2,eGPIO_3,0);
		GPIO_SetPullUpDownEach(eGPIO_J2,eGPIO_4,0);
		GPIO_SetPullUpDownEach(eGPIO_J2,eGPIO_5,0);
		GPIO_SetPullUpDownEach(eGPIO_J2,eGPIO_6,0);
		UART_Printf("Key Pad Column Port : MSM Port  Init!!!\n");
	}
	

	if (eRowPort == KeyPort0)		//Row Port : EINT(24~31)
	{		
		//Key Pad Row		Host I/F
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_0, 3);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_1, 3);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_2, 3);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_3, 3);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_4, 3);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_5, 3);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_6, 3);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_7, 3);			

		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_0, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_1, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_2, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_3, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_4, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_5, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_6, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_7, eRowPullUDEn);	

		UART_Printf("Key Pad Row Port : EINT(24~31) Port Init!!!(Pullup value : 0x%x)\n",eRowPullUDEn);	
	}			
	else 		//Row Port :  MMC (Power Domain is VDD_MMC)
	{
		//Key Pad Row		 	EINT 
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_7,3);  // row 0
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_0,3);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_1,3);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_2,3);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_3,3);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_4,3);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_5,3);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_6,3);  //row 7
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_7,3);  //row 8
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_0,3);		
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_1,3);
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_2,3);
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_3,3);
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_4,3);  //row 13

		GPIO_SetPullUpDownEach(eGPIO_J2, eGPIO_7, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_0, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_1, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_2, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_3, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_4, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_5, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_6, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_7, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_0, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_1, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_2, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_3, eRowPullUDEn);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_4, eRowPullUDEn);
												
		UART_Printf("Key Pad Row Port : MSM Port(for ROW0- ROM13)  Init!!!\n");
	}
	Delay(100);	
}

//////////
// Function Name : KEYPAD_ClosePort
// Function Description : This function close KEYPAD Port between GPK and GPH, GPN and sets up them(HOST I/F)
// Input : 	eRowPort - Row Port Selection
// 			eColPort - Column Port Selection
// Output : NONE
// Version : v0.1
void KEYPAD_ClosePort(KEYPORT eRowPort, KEYPORT eColPort)
{
	if (eColPort == KeyPort0)		//KeyPort0
	{		
		//Key Pad Column	EINT(16-23)
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_0, 0x0);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_1, 0x0);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_2, 0x0);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_3, 0x0);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_4, 0x0);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_5, 0x0);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_6, 0x0);
		GPIO_SetFunctionEach(eGPIO_H2, eGPIO_7, 0x0);			
	}
	else 						//Column Port : MMC (Power Domain is VDD_MMC)
	{
		//Key Pad Column		MMC
		GPIO_SetFunctionEach(eGPIO_J1,eGPIO_5,0);		
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_0,0);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_1,0);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_2,0);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_3,0);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_4,0);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_5,0);
		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_6,0);
	}
	
	if (eRowPort == KeyPort0)		//KeyPort0
	{
		//Key Pad Row		EINT(24-31)
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_0, 0);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_1, 0);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_2, 0);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_3, 0);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_4, 0);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_5, 0);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_6, 0);
		GPIO_SetFunctionEach(eGPIO_H3, eGPIO_7, 0);	
		
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_0, 0x0);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_1, 0x0);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_2, 0x0);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_3, 0x0);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_4, 0x0);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_5, 0x0);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_6, 0x0);
		GPIO_SetPullUpDownEach(eGPIO_H3, eGPIO_7, 0x0);		
	}
	else 		//Row Port : EINT(Power Domain is VDD_SYS)
	{
		//Key Pad Row		 	EINT
 		GPIO_SetFunctionEach(eGPIO_J2,eGPIO_5,0);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_0,0);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_1,0);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_2,0);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_3,0);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_4,0);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_5,0);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_6,0);
		GPIO_SetFunctionEach(eGPIO_J3,eGPIO_7,0);
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_0,0);
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_1,0);
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_2,0);
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_3,0);						
		GPIO_SetFunctionEach(eGPIO_J4,eGPIO_4,0);
		
		GPIO_SetPullUpDownEach(eGPIO_J2, eGPIO_5, 0);		
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_0, 0);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_1, 0);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_2, 0);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_3, 0);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_4, 0);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_5, 0);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_6, 0);
		GPIO_SetPullUpDownEach(eGPIO_J3, eGPIO_7, 0);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_0, 0);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_1, 0);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_2, 0);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_3, 0);
		GPIO_SetPullUpDownEach(eGPIO_J4, eGPIO_4, 0);								
	}
	Delay(100);
}

/*---------------------------------- APIs of rKEYCON ---------------------------------*/
//////////
// Function Name : KEYPAD_SetKEYIFCON
// Function Description : This function Selects KEYPAD Port between GPK and GPH, GPN and sets up them(HOST I/F)
// Input : 	eRowPort - Row Port Selection
// 			eColPort - Column Port Selection
//			usRowPullEn - Row Pull -Up Enable/Disabel
// Output : NONE
// Version : v0.1
void KEYPAD_SetKEYIFCON(u32 uWakeUpEn, u32 uDFSource, u32 uFilterEn, KEY_INTSOURCE uIntrSrc)
{
	u32 uKEYIFCON = (uWakeUpEn << 4) | (uDFSource << 3) | (uFilterEn << 2) | ((u32) uIntrSrc << 0);
	KEYIFOutp32(rKEYIFCON, uKEYIFCON);	
}

//////////
// Function Name : KEYPAD_EnableWakeup
// Function Description : This function enables Stop/Idle mode wakeup
// Input : uEnWakeup [ 0: disabe	1: Low lvl wake up		2: Rising Edge wake up		4.Filling Edge wake up]
// Output : NONE
// Version : v0.1
void KEYPAD_EnableWakeup(u32 uSel)
{
	u32 uTemp = KEYIFInp32(rKEYIFCON);
	if (uSel == 1)
		uTemp |= KEYPAD_EN_WakeupINT;	
	else
		uTemp &= ~KEYPAD_EN_WakeupINT;		
	KEYIFOutp32(rKEYIFCON,uTemp);
}

//////////
// Function Name : KEYPAD_EnableDFFC
// Function Description : This function enables debouncing filter & filter clock
// Input : uEnDF [ 0: disable	1: enable]
// 		  uEnFC [ 0: disable	1: enable]
// Output : NONE
// Version : v0.1
void KEYPAD_EnableFilterCNT(u32 uSel)
{
	u32 uTemp	= KEYIFInp32(rKEYIFCON);
	uTemp = KEYIFInp32(rKEYIFCON);
	if (uSel == 0)
	{
		uTemp &= ~KEYPAD_EN_FilterCounter;
	}
	else
	{
		uTemp |= KEYPAD_EN_FilterCounter;
	}	
	KEYIFOutp32(rKEYIFCON,uTemp);
}

//////////
// Function Name : KEYPAD_EnableDFFC
// Function Description : This function enables debouncing filter & filter clock
// Input : uEnDF [ 0: disable	1: enable]
// 		  uEnFC [ 0: disable	1: enable]
// Output : NONE
// Version : v0.1
void KEYPAD_EnableDBFilter(u32 uSel)
{
	u32 uTemp = KEYIFInp32(rKEYIFCON);
	if (uSel == 0)
		uTemp &= ~KEYPAD_EN_DebounceFilter;
	else
		uTemp |= KEYPAD_EN_DebounceFilter;
	KEYIFOutp32(rKEYIFCON, uTemp);
}

//////////
// Function Name : KEYPAD_SelectINT
// Function Description : This function selects Interrupt type
// Input : uSelect [1:Falling edge INT    2:Rising edge INT    3:Both edge INT]
// Output : NONE
// Version : v0.1
void KEYPAD_SelectINT(KEY_INTSOURCE eSel, u32 uEnable)
{
	u32 uTemp;	
	uTemp = KEYIFInp32(rKEYIFCON);	
	if ( uEnable == TRUE ) 
	{
		if(eSel == Key_Rising)				//Default Falling Edge Type
		{
			uTemp |= KEYPAD_EN_RisingINT;
		}
		else if (eSel == Key_Both)
		{
			uTemp |= KEYPAD_EN_FallingINT | KEYPAD_EN_RisingINT;	
		}
		else
		{
			uTemp |= KEYPAD_EN_FallingINT;
		}	
	}
	else
	{
		if(eSel == Key_Rising)				//Default Falling Edge Type
		{
			uTemp &= ~KEYPAD_EN_RisingINT;
		}
		else if (eSel == Key_Both)
		{
			uTemp &= ~KEYPAD_EN_FallingINT & ~KEYPAD_EN_RisingINT;	
		}
		else
		{
			uTemp &= ~KEYPAD_EN_FallingINT;
		}	
	}	
	KEYIFOutp32(rKEYIFCON, uTemp);	
}
/*---------------------------------- APIs of rKEYIFSTSCLR ------------------------------*/
//////////
// Function Name : KEYPAD_STSCLR
// Function Description : This function checks Interrupt status and clear register
// Input : P_INT [Pressed interrupt is cleared when write data is '1']
// 		  R_INT [Pressed interrupt is cleared when write data is '1']
// Output : NONE
// Version : v0.1
void KEYPAD_STSCLR(u32 ucP_INT, u32 ucR_INT)
{
	KEYIFOutp32(rKEYIFSTSCLR, ucP_INT|ucR_INT<<16);
	
}

//////////
// Function Name : KEYPAD_STSCLRCon
// Function Description : This function clear Interrupt status by Interrupt condition
// Input : 	eInt : Interrupt Condition
// 		  	ucClr Clear Position
// Output : NONE
// Version : v0.1
void KEYPAD_STSCLRCon(KEY_INTSOURCE eInt, u8 ucClr)
{
	u32 urKEYIFSTSCLR = KEYIFInp32(rKEYIFSTSCLR) & 0xffff;
	if (eInt == Key_Falling)
		urKEYIFSTSCLR |= ucClr;
	else if (eInt == Key_Rising)
		urKEYIFSTSCLR |= (ucClr << 8);
	else if (eInt == Key_Both)
		urKEYIFSTSCLR |= (ucClr << 8) | ucClr;
	KEYIFOutp32(rKEYIFSTSCLR, urKEYIFSTSCLR);	
}

//////////
// Function Name : KEYPAD_STSCLR
// Function Description : This function checks Interrupt status and clear register
// Input : None 		  
// Output : Key Interrupt Status Register Value(0x7E00A004)
// Version : v0.1

u32 KEYIFSTSCLR_GetReg(void)
{
	return KEYIFInp32(rKEYIFSTSCLR);
}
/*---------------------------------- APIs of rKEYIFCOL ---------------------------------*/
// Function Name : KEYPAD_COL
// Function Description : This function checks COL data output and enable tri-state buffer
// Input : COL [ column data output]
// 		  COLEnable [0: tri-state buffer enable	1: disable]
// Output : NONE
// Version : v0.1
void KEYPAD_COL(u8 ucCOL,u8 ucCOLLEnable)
{
	KEYIFOutp32(rKEYIFCOL,ucCOL|ucCOLLEnable<<8);
	
}

// Function Name : KEYPAD_ColumnOut
// Function Description : This function output Coulumn data in KEYIFCOL.
// Input : COL [ column data output]
// Output : NONE
// Version : v0.1
void KEYPAD_ColumnOut(u8 ucCol)
{
	u32 urKEYIFCOL = KEYIFInp32(rKEYIFCOL) & ~0xff;
	urKEYIFCOL |= (ucCol << 0);
	KEYIFOutp32(rKEYIFCOL, urKEYIFCOL);
}

// Function Name : KEYPAD_EnableCol
// Function Description : This function enable Column output(Enable Tristate Buffer)
// Input :   COLEnable [0: tri-state buffer enable	1: disable]
// Output : NONE
// Version : v0.1
void KEYPAD_EnableCol(u8 ucColen)
{
	u32 urKEYIFCOL = KEYIFInp32(rKEYIFCOL) & ~(0xff << 8);
	urKEYIFCOL |= (ucColen << 8);
	KEYIFOutp32(rKEYIFCOL, urKEYIFCOL);
	
}

// Function Name : KEYPAD_DisableCol
// Function Description : This function make one column to low and others High-Z.
// Input :   ucCol : column number.
// Output : NONE
// Version : v0.1
void KEYPAD_DisableCol(u8 ucCol)
{
	u32 urKEYIFCOL;

	//urKEYIFCOL = KEYIFInp32(rKEYIFCOL);

	urKEYIFCOL = (1<< (ucCol + 8)) & 0xFF00;	/* ucCol Output is low and others are High-Z. */
	KEYIFOutp32(rKEYIFCOL, urKEYIFCOL);
}

/*---------------------------------- APIs of rKEYIFROW --------------------------------*/
//////////
// Function Name : KEYPAD_ROW
// Function Description : This function reads ROW data input
// Input : NONE
// Output : rKEYIFROW value(u32)
// Version : v0.1

u32 KEYPAD_ROW(void)
{
#if 1 //jspark modified
//	return KEYIFInp32(rKEYIFROW) & 0x3FFF;  //KEYIFROW[13..0]
	#if 0
	return KEYIFInp32(rKEYIFROW) & 0x00FF;  
	                                                                   
	#else
	u32	IntValue;
	
	IntValue = *(u32 *)0xE160000C;
	if(gKEYPAD_Is8by8Mode == TRUE)
	{
		IntValue &= 0xFF;
	}
	else
	{
		IntValue &= 0x3FFF;
	}

	return IntValue;
	#endif /* 0 */
#else
	return KEYIFInp32(rKEYIFROW);
#endif
}
/*---------------------------------- APIs of rKEYIFFC ----------------------------------*/
//////////
// Function Name : KEYPAD_SetDFCNT
// Function Description : This function set Debounce Filter Counter Value;
// Input : uCntValue - Counter Value
// Output : NONE
// Version : v0.1
u8 KEYPAD_SetDFCNT(s32 uCntValue)
{
	if(uCntValue >= 0 && uCntValue<=1023)				
	{		
		KEYIFOutp32(rKEYIFFC, uCntValue);
		return TRUE;
	}
	else 		
	{
		UART_Printf("\nError!!!  It is not range of Debouce Filter CNT.\n");
		return FALSE;
	}	
}
//////////


bool KEYPAD_GetCheckedKeyEvent_8by8(bool *pIsKeyPressed, u8 *pRow, u8 *pCol)
{
	u32 urKEYIFSTSCLR, urKEYIFROW;
	u32 uCol, uRow;
	u32 uColScan;
	u32 uRowScan;
	bool IsKeyEventDetected = FALSE;

	urKEYIFSTSCLR = KEYPAD_ROW();	
	//UART_Printf("Read Interrupt Pending Register, Value is 0x%x\n", urKEYIFSTSCLR);
	KEYPAD_STSCLR(0x3fff,0x3fff);	
	INTC_ClearVectAddr();
	if ( urKEYIFSTSCLR & 0xff )										//Key Pressed
	{
		//IsKeyEventDetected = TRUE;
		
		for ( uCol = 0; uCol < KEYPAD_COLNUM ; uCol++)
		{
			uColScan = 0xff ^ (1 << uCol);
			KEYPAD_EnableCol(uColScan);							//Col output Hi-Z enable.		
			//KEYPAD_DisableCol(uCol);							// Col is low and others are High-Z.
			Delay(KEY_ScanDelay);								//Must have Delay. 
			urKEYIFROW = KEYPAD_ROW();
			if (  urKEYIFROW != 0xff)
			{
				KEYPAD_STSCLR(0x3fff,0x3fff);						//Unwanted Interrupt Source is made. it must clear!						
				for(uRow = 0;uRow < KEYPAD_ROWNUM; uRow++)	//Scan X-Axis (COLUMN)
				{
					//uRowScan = 0xff ^(1 << uRow);
					uRowScan = (1 << uRow);
					//if ( (urKEYIFROW | uRowScan) != 0xff)
					if ( (urKEYIFROW | uRowScan) == 0xff)
					{
						KEYPAD_INFOR* pInfor = KEYPAD_GetInfor();
						if (pInfor->m_fnKeyISR[uCol][uRow] != NULL )
						{
							pInfor->m_fnKeyISR[uCol][uRow](pInfor->m_uVarKeyISR[uCol][uRow]);
						}
						else
						{
							Disp("[%d, %d]\n", uCol+1, uRow+1);
							Disp("%d\n", (uCol * 8 + uRow + 1));

							/* check just one key press */
							*pIsKeyPressed = TRUE;
							*pRow = uRow + 1;
							*pCol = uCol + 1;

							IsKeyEventDetected = TRUE;
							//break;
							//goto Keypad_Detected;
						}						
					}				
				}				
			}
		}		
	}
	else if ( urKEYIFSTSCLR & 0xff00 )		//Key Released
	{
		//IsKeyEventDetected = TRUE;
		IsKeyEventDetected = FALSE;
		Disp("Some Key is Released!!\n");		
	}
	else
	{
		IsKeyEventDetected = FALSE;
	}

Keypad_Detected:

	KEYPAD_EnableCol(0x00);			
	Delay(KEY_ScanDelay);
	KEYPAD_STSCLR(0x3fff,0x3fff);

	return IsKeyEventDetected;
}


bool KEYPAD_GetCheckedKeyEvent_14by8(bool *pIsKeyPressed, u8 *pRow, u8 *pCol)
{
	u32 urKEYIFSTSCLR, urKEYIFROW;
	u32 uCol, uRow;
	u32 uColScan;
	u32 uRowScan;
	bool IsKeyEventDetected = FALSE;

	urKEYIFSTSCLR = KEYPAD_ROW();	
	//UART_Printf("Read Interrupt Pending Register, Value is 0x%x\n", urKEYIFSTSCLR);
	KEYPAD_STSCLR(0x3fff,0x3fff);	
	INTC_ClearVectAddr();
	if ( urKEYIFSTSCLR & 0x3fff )										//Key Pressed
	{
		//IsKeyEventDetected = TRUE;
		
		for ( uCol = 0; uCol < KEYPAD_COLNUM ; uCol++)
		{
			uColScan = 0xff ^ (1 << uCol);
			KEYPAD_EnableCol(uColScan);							//Col output Hi-Z enable.		
			//KEYPAD_DisableCol(uCol);							// Col is low and others are High-Z.
			Delay(KEY_ScanDelay);								//Must have Delay. 
			urKEYIFROW = KEYPAD_ROW();
			if (  urKEYIFROW != 0x3fff)
			{
				KEYPAD_STSCLR(0x3fff,0x3fff);						//Unwanted Interrupt Source is made. it must clear!						
				for(uRow = 0;uRow < KEYPAD_ROWNUM; uRow++)	//Scan X-Axis (COLUMN)
				{
					//uRowScan = 0xff ^(1 << uRow);
					uRowScan = (1 << uRow);
					//if ( (urKEYIFROW | uRowScan) != 0xff)
					if ( (urKEYIFROW | uRowScan) == 0x3fff)
					{
						KEYPAD_INFOR* pInfor = KEYPAD_GetInfor();
						if (pInfor->m_fnKeyISR[uCol][uRow] != NULL )
						{
							pInfor->m_fnKeyISR[uCol][uRow](pInfor->m_uVarKeyISR[uCol][uRow]);
						}
						else
						{
							Disp("[%d, %d]\n", uCol+1, uRow+1);
							Disp("%d\n", (uCol * 14 + uRow + 1));

							/* check just one key press */
							*pIsKeyPressed = TRUE;
							*pRow = uRow + 1;
							*pCol = uCol + 1;

							IsKeyEventDetected = TRUE;
							//break;
							//goto Keypad_Detected;
						}						
					}				
				}				
			}
		}		
	}
	else if ( urKEYIFSTSCLR & 0xff00 )		//Key Released
	{
		//IsKeyEventDetected = TRUE;
		IsKeyEventDetected = FALSE;
		Disp("Some Key is Released!!\n");		
	}
	else
	{
		IsKeyEventDetected = FALSE;
	}

Keypad_Detected:

	KEYPAD_EnableCol(0x00);			
	Delay(KEY_ScanDelay);
	KEYPAD_STSCLR(0x3fff,0x3fff);

	return IsKeyEventDetected;
}


//should execute at first
bool KEYPAD_SFR_Test(void)
{
	bool	InternalSatus;

	InternalSatus = TestSFRn(sRegInfoKEYIIF, sizeof(sRegInfoKEYIIF)/sizeof(REGINFOn));
	if(InternalSatus == 0)
	{
		/* Error Case */

		return AUTOTEST_FAIL;
	}
	else
	{
		/* No Error Case */
		
		return AUTOTEST_OK;
	}
}


void KEYPAD_WakeupInit(void)
{
	KEYIF_Configuration oConfiguration;

	oConfiguration = KEYPAD_DefaultConfiguration();
	KEYPAD_ClosePort(KeyPort0, KeyPort0);
	KEYPAD_ClosePort(KeyPort1, KeyPort1);		

	KEYPAD_Create(oConfiguration, NULL, (u32)NULL);

	#if 1	/* Work-around for S5PV210 EVT0 chip bug. */
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_0, 1);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_1, 1);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_2, 1);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_3, 1);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_4, 1);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_5, 1);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_6, 1);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_7, 1);	

	GPIO_SetDataEach(eGPIO_H2, eGPIO_0, 0);
	GPIO_SetDataEach(eGPIO_H2, eGPIO_1, 0);
	GPIO_SetDataEach(eGPIO_H2, eGPIO_2, 0);
	GPIO_SetDataEach(eGPIO_H2, eGPIO_3, 0);
	GPIO_SetDataEach(eGPIO_H2, eGPIO_4, 0);
	GPIO_SetDataEach(eGPIO_H2, eGPIO_5, 0);
	GPIO_SetDataEach(eGPIO_H2, eGPIO_6, 0);
	GPIO_SetDataEach(eGPIO_H2, eGPIO_7, 0);	
	#endif /* 0 */
}


