/**************************************************************************************
*
*	Project Name : S5PV210 FPGA Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for verifying functions of the S5PV210 FPGA
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------
* 
*	File Name : iic_test.c
*  
*	File Description : This file implements the API functons for IIC
*
*	Author	: Jongseok,Park
*	Dept. : AP Development Team
*	Created Date : 2009/04/16
*	Version : 0.0 
*
*	History
*	- Version 0.1  : (2009/04/16) 
*         -> Created
*	- Version 0.2  : (2009/07/10) 
		-> Updated
**************************************************************************************/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "iic.h"
#include "def.h"
#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "gpio.h"
#include "sysc.h"
#include "intc.h"
#include "sysc.h"


/*******************************************************************************
* DEFINES
*******************************************************************************/
#if 1 //jspark for v210
#define COMPAdd1	0x24000000
#define COMPAdd2	0x24100000
#else
#define COMPAdd1	0x51200000
#define COMPAdd2	0x51300000
#endif

/*******************************************************************************
* ENUM
*******************************************************************************/
enum{
	DBG_No		=	0,
	DBG_Basic	=	1,
	DBG_Step		=	2,			
	DBG_Interrupt	=	3
};	

static int IIC_DEBUG_LEVEL  =	DBG_Basic;//DBG_Step; 
#define IIC_DBG_PRINT(debug_level, args...)   do {  \
															if(debug_level <= IIC_DEBUG_LEVEL)  \
																UART_Printf(args); \
																	}while(0)

/*
*	EEPROM Setting Condition for FPGA is different from the one for SMDK
*	EEPROM in FPGA :  One is KS24C080C for Ch0, the Other is S524AD0XD1 for Ch2.
*	EEPROM in SMDK :  One is S524AD0XD1 for Ch0 Only.
*	2007.07.17
*/
#ifdef v210_I2C_FPGA
#define KS24C080C_SLAVEADDRESS		0xA0	// FPGA
/* Refer to S524AD0XD1 data sheet & SMDK Base B'd Schematics as well... 2007.07.17 */
#define S524AD0XD1_SLAVEADDRESS		0xAE	// FPGA
#else
#define S524AD0XD1_SLAVEADDRESS		0xA0	// SMDK
#endif


#define 	STOP_NO_GEN	0
#define 	STOP_GEN		1

#define	PATTERN_TEST	1

/*******************************************************************************
* EXTERNS
*******************************************************************************/
extern st_IIC_REG IIC_REG[3];
extern u32 IIC_SFR_List[]; 


/*******************************************************************************
* GLOBAL VARIABLES
*******************************************************************************/	
static	u8 testpattern[256] = {
0x89,0x50,0x4E,0x47,0x0D,0x0A,0x1A,0x0A,0x00,0x00,0x00,0x0D,0x49,0x48,0x44,0x52,
0x00,0x00,0x0B,0x8E,0x00,0x00,0x03,0x2C,0x08,0x06,0x00,0x00,0x00,0xA1,0xE0,0x1D,
0x34,0x00,0x00,0x00,0x04,0x73,0x42,0x49,0x54,0x08,0x08,0x08,0x08,0x7C,0x08,0x64,
0x88,0x00,0x00,0x20,0x00,0x49,0x44,0x41,0x54,0x78,0x9C,0xEC,0xDD,0x6B,0x90,0x63,
0xE9,0x79,0x1F,0xF6,0xE7,0x00,0x98,0x0B,0x97,0xBB,0xCB,0xDB,0x52,0x22,0x25,0x72,
0x49,0x91,0x12,0x45,0x4A,0xA4,0xAC,0x5B,0x44,0x5B,0xA4,0x4C,0x59,0xB1,0xA4,0x58,
0x54,0x1C,0xC9,0x4A,0xA5,0x9C,0xB2,0x52,0x4A,0xAA,0x92,0x92,0x92,0x0F,0x4E,0x95,
0x97,0xE4,0x2E,0xF7,0xCA,0xDD,0x99,0xD9,0xD9,0xD9,0xB9,0xEC,0xF4,0x05,0x40,0xA3,
0xBF,0x5F,0xD5,0xD6,0x74,0xE3,0x1C,0x9C,0xF3,0xE0,0xBC,0xE7,0x05,0xCE,0xCE,0xFC, 
0x80,0x05,0x55,0x5A,0x76,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0xCC,0x97,0xE0,0x38,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,
0x8A,0x13,0x1C,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x71,		
0x00,0xCC,0x97,0xE0,0x38,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,
0x8A,0x13,0x1C,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x71,		
0x00,0xCC,0x97,0xE0,0x38,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,
0x8A,0x13,0x1C,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x71		
};


REGINFO	sRegInfoIIC[] = 
{
	// I2C Channel 0
	{"rI2CCON0	",			I2C0_BASE+0x00, 8,	RW, DPPB, 0xEF, 0x00},
	{"rI2CSTAT0	", 			I2C0_BASE+0x04, 8,	RW, DPPB, 0xD0, 0x00},
	{"rI2CADD0	", 			I2C0_BASE+0x08, 8,	RW, DPPB, 0xFE, 0x00},
	{"rI2CCDS0	",			I2C0_BASE+0x0c, 8,	RW, DPDB, 0, 0x00}, 
	{"rI2CLC0	", 			I2C0_BASE+0x10, 3,	RW, DPDB, 0, 0x00},
	 
	{"rI2CCON_HDMI_PHY ", 	I2C_HDMI_PHY_BASE+0x00, 8, RW, DPPB, 0xEF, 0x00},
	{"rI2CSTAT_HDMI_PHY ",	I2C_HDMI_PHY_BASE+0x04, 8, RW, DPPB, 0xD0, 0x00},
	{"rI2CADD_HDMI_PHY ",	I2C_HDMI_PHY_BASE+0x08, 8, RW, DPPB, 0xFE, 0x00},
	{"rI2CDS_HDMI_PHY ",	I2C_HDMI_PHY_BASE+0x0C, 8, RW, DPDB, 0, 0x00},
	{"rI2CLC_HDMI_PHY ",	I2C_HDMI_PHY_BASE+0x10, 3, RW, DPDB, 0, 0x00},
	
	{"rI2CCON_HDMI_DDC ",	I2C_HDMI_DDC_BASE+0x00, 8, RW, DPPB, 0xEF, 0x00},
	{"rI2CSTAT_HDMI_DDC ",	I2C_HDMI_DDC_BASE+0x04, 8, RW, DPPB, 0xD0, 0x00},
	{"rI2CADD_HDMI_DDC ",	I2C_HDMI_DDC_BASE+0x08, 8, RW, DPPB, 0xFE, 0x00},
	{"rI2CDS_HDMI_DDC ",	I2C_HDMI_DDC_BASE+0x0C, 8, RW, DPDB, 0, 0x00},
	{"rI2CLC_HDMI_DDC ",	I2C_HDMI_DDC_BASE+0x10, 3, RW, DPDB, 0, 0x00},
	
	// I2C Channel 2 for PMIC
	{"rI2CCON2	", 			I2C2_BASE+0x00, 8,	RW, DPPB, 0xEF, 0x00},
	{"rI2CSTAT2	", 			I2C2_BASE+0x04, 8,	RW, DPPB, 0xD0, 0x00},
	{"rI2CADD2 ", 			I2C2_BASE+0x08, 8,	RW, DPPB, 0xFE, 0x00},
	{"rI2CCDS2 ", 			I2C2_BASE+0x0c, 8,	RW, DPDB, 0, 0x00}, 
	{"rI2CLC2	",			I2C2_BASE+0x10, 3,	RW, DPDB, 0, 0x00}, 
};


/*******************************************************************************
* FUNCTION PROTOTYPES
*******************************************************************************/
void SFR_DefaultIIC(void);
void SFR_TestIIC(void);

// Channel 0
void IIC0_BTest(void);		// Edited by SOP on 2008/03/03
void IIC0_MasterTxP(void);
void IIC0_SlaveRxP(void);
void IIC0_MasterRxP(void);
void IIC0_SlaveTxP(void);
void IIC0_SlaveTxInt(void);
void IIC0_SlaveRxInt(void);
void IIC0_CLKDIVTest(void);	// Added by SOP on 2008/04/03
void IIC0_CLKGateTest(void);	// Added by SOP on 2008/04/03
void IIC0_CLKDIV(u32 uHCLKx2_DIVN_R, u32 uPCLK_DIVN_R, u32 uFreq);	// Added by SOP on 2008/04/03

// Channel 2
void IIC2_BTest(void);		// Added by SOP on 2008/03/03
void IIC2_MasterTxP(void);
void IIC2_SlaveRxP(void);
void IIC2_MasterRxP(void);
void IIC2_SlaveTxP(void);
void IIC2_SlaveTxInt(void);
void IIC2_SlaveRxInt(void);
void IIC2_CLKDIVTest(void);	// Added by SOP on 2008/04/03
void IIC2_CLKGateTest(void);	// Added by SOP on 2008/04/04
void IIC2_CLKDIV(u32 uHCLKx2_DIVN_R, u32 uPCLK_DIVN_R, u32 uFreq);	// Added by SOP on 2008/04/03

u8 IIC0_AutoTest(void);				// Added by SOP on 2008/04/22
u8 IIC2_AutoTest(void);			// Added by SOP on 2008/04/22
void IIC0_WM8990Test( void);		// Addes by SOP on 2008/09/25
void IIC0_AutoTest_Polling(void); //added by jspark on 2009/03/18
void IIC_CrossTest_Polling(void); //added by jspark on 2009/03/18
u32 IIC0_Read_Polling(u32 ch, u32 SlaveAddr,u8* data, u32 RxSize,u32 StopGen);

//////////////////////////////////////////////////////////////////////////////////////////////
/*******************************************************************************
* EXTERNS
*******************************************************************************/
extern bool 	gbI2C_CH0_Performance_Test;

/*******************************************************************************
* FUNCTION PROTOTYPES
*******************************************************************************/
// Channel 2 : PMIC
u8	I2C2_Slave_Rx_Polling(void);
u8 	I2C2_Slave_Tx_Polling(void);
u8 	I2C2_Slave_Rx_Interrupt(void);
u8	I2C2_Slave_Tx_Interrupt(void);

u8	I2C2_Master_Rx_Polling(void);
u8	I2C2_Master_Tx_Polling(void);
u8	I2C2_Master_Rx_Interrupt(void);
u8	I2C2_Master_Tx_Interrupt(void);

// Channel 1 : HDMI_DDC
u8	I2C1_Slave_Rx_Polling(void);
u8	I2C1_Slave_Tx_Polling(void);
u8	I2C1_Slave_Rx_Interrupt(void);
u8	I2C1_Slave_Tx_Interrupt(void);

u8	I2C1_Master_Rx_Polling(void);
u8	I2C1_Master_Tx_Polling(void);
u8	I2C1_Master_Rx_Interrupt(void);
u8	I2C1_Master_Tx_Interrupt(void);

// Channel 0 : COMMON
u8	I2C0_Slave_Rx_Polling(void);
u8 	I2C0_Slave_Tx_Polling(void);
u8 	I2C0_Slave_Rx_Interrupt(void);
u8 	I2C0_Slave_Tx_Interrupt(void);

u8 	I2C0_Master_Rx_Polling(void);
u8	I2C0_Master_Tx_Polling(void);
u8	I2C0_Master_Rx_Interrupt(void);
u8	I2C0_Master_Tx_Interrupt(void);

u8	I2C_Slave_Rx_Polling(u32 ch);
u8	I2C_Slave_Tx_Polling(u32 ch);
u8	I2C_Slave_Rx_Interrupt(u32 ch);
u8	I2C_Slave_Tx_Interrupt(u32 ch);

u8	I2C_Master_Rx_Polling(u32 ch);
u8	I2C_Master_Tx_Polling(u32 ch);
u8	I2C_Master_Rx_Interrupt(u32 ch);
u8 	I2C_Master_Tx_Interrupt(u32 ch);

#ifdef v210_I2C_FPGA
u8	I2C_CH0_CH2_CrossTest_Polling(void);
#else
u8	I2C_CH1_CH0_CrossTest_Polling(void);
u8	I2C_CH0_CH1_CrossTest_Polling(void);
u8	I2C_CH0_CH1_CrossTest_Interrupt(void);
#endif

u8	I2C_CH0_CH2_AutoTest(void);

#ifndef v210_I2C_FPGA
u8 I2C_Bus_Arbitration_Test(void);
#endif

u8 I2C_CH0_Performance_Test(void);

u8 I2C_CH0_Aging_Test(void);
/*******************************************************************************
* GLOBAL VARIABLES
*******************************************************************************/


/*******************************************************************************
* FUNCTION PROTOTYPES
*******************************************************************************/
void IIC_Test(void)
{
	u32 uCountFunc=0;
	s32 iSel=0;

	const testFuncMenu g_aIICTestFunc[] =
	{	
		SFR_TestIIC,				"IIC SFR R/W test",

#if 0
		// Channel 0 : COMMON
		IIC0_BTest,				"IIC0 MasterTx/Rx from EEPROM (KS24C080C) by Interrupt)",
		IIC0_MasterTxP,			"IIC0 Master Tx (B'd to B'd) by Polling",
		IIC0_SlaveRxP,			"IIC0 Slave  Rx (B'd to B'd) by Polling",
		IIC0_MasterRxP,			"IIC0 Master Rx (B'd to B'd) by Polling",
		IIC0_SlaveTxP,			"IIC0 Slave  Tx (B'd to B'd) by Polling",
		IIC0_SlaveTxInt,			"IIC0 Slave  Tx (B'd to B'd) by Interrupt",	
		IIC0_SlaveRxInt,			"IIC0 Slave  Rx (B'd to B'd) by Interrupt",
		IIC0_AutoTest,		  		"[*]IIC0 Auto Test (Master TX/RX) from EEPROM (KS24C080C) by Interrupt",
		IIC0_AutoTest_Polling,	"[*]IIC0 Auto Test (Master TX/RX) from EEPROM (KS24C080C) by Polling",
		IIC_CrossTest_Polling,	"[*]IIC0 MasterTX / IIC1 SlaveRX  CrossTest by Polling",	
		IIC0_WM8990Test,			"IIC0 from Audio Codec (WM8990) by Interrupt\n",	

		// Channel 2 : PMIC
#ifdef v210_I2C_FPGA
		IIC2_BTest,				"IIC2 MasterTx/Rx from EEPROM (S524AD0XD1) by Interrupt)",
#endif
		IIC2_MasterTxP,			"IIC2 Master Tx (B'd to B'd) by Polling",
		IIC2_SlaveRxP,			"IIC2 Slave  Rx (B'd to B'd) by Polling",
		IIC2_MasterRxP,			"IIC2 Master Rx (B'd to B'd) by Polling",
		IIC2_SlaveTxP,			"IIC2 Slave  Tx (B'd to B'd) by Polling",
		IIC2_SlaveTxInt,			"IIC2 Slave  Tx (B'd to B'd) by Interrupt",	
		IIC2_SlaveRxInt,			"IIC2 Slave  Rx (B'd to B'd) by Interrupt",
#ifdef v210_I2C_FPGA	
		IIC2_AutoTest,				"[*]IIC2 Auto Test (Master TX/RX) from EEPROM (S524AD0XD1) by Interrupt",
#endif
#endif/* 0 */
////////////////////////////////////////////////////////////////////////////////////////////////////////

	// Channel 0 : COMMON
		I2C0_Master_Tx_Polling,	"I2C0 Master Tx (B'd to B'd) by Polling",
		I2C0_Master_Tx_Interrupt,	"I2C0 Master Tx (B'd to B'd) by Interrupt",
		I2C0_Slave_Rx_Polling,	"I2C0 Slave  Rx (B'd to B'd) by Polling",
		I2C0_Slave_Rx_Interrupt,	"I2C0 Slave  Rx (B'd to B'd) by Interrupt",

		I2C0_Master_Rx_Polling,	"I2C0 Master Rx (B'd to B'd) by Polling",
		I2C0_Master_Rx_Interrupt,"I2C0 Master Rx (B'd to B'd) by Interrupt",
		I2C0_Slave_Tx_Polling,	"I2C0 Slave  Tx (B'd to B'd) by Polling",
		I2C0_Slave_Tx_Interrupt,	"I2C0 Slave  Tx (B'd to B'd) by Interrupt",

#ifndef v210_I2C_FPGA	
		I2C0_CLKDIVTest,			"I2C0 Clock Divider Test",
		I2C0_CLKGateTest,		"I2C0 Clock Gating (On/Off) Test",
#endif
		0,							"---------------------------------------",
		
	// Channel 1 : HDMI_DDC
		I2C1_Master_Tx_Polling,	"I2C1 Master Tx (B'd to B'd) by Polling",
		I2C1_Master_Tx_Interrupt,	"I2C1 Master Tx (B'd to B'd) by Interrupt",
		I2C1_Slave_Rx_Polling,	"I2C1 Slave  Rx (B'd to B'd) by Polling",
		I2C1_Slave_Rx_Interrupt,	"I2C1 Slave  Rx (B'd to B'd) by Interrupt",

		I2C1_Master_Rx_Polling,	"I2C1 Master Rx (B'd to B'd) by Polling",
		I2C1_Master_Rx_Interrupt,"I2C1 Master Rx (B'd to B'd) by Interrupt",
		I2C1_Slave_Tx_Polling,	"I2C1 Slave  Tx (B'd to B'd) by Polling",
		I2C1_Slave_Tx_Interrupt,	"I2C1 Slave  Tx (B'd to B'd) by Interrupt",

#ifndef v210_I2C_FPGA		
		I2C1_CLKDIVTest,			"I2C1 Clock Divider Test",
		I2C1_CLKGateTest,		"I2C1 Clock Gating (On/Off) Test",
#endif
		0,							"---------------------------------------",
		
	// Channel 2 : PMIC
		I2C2_Master_Tx_Polling,	"I2C2 Master Tx (B'd to B'd) by Polling",
		I2C2_Master_Tx_Interrupt,	"I2C2 Master Tx (B'd to B'd) by Interrupt",
		I2C2_Slave_Rx_Polling,	"I2C2 Slave  Rx (B'd to B'd) by Polling",
		I2C2_Slave_Rx_Interrupt,	"I2C2 Slave  Rx (B'd to B'd) by Interrupt",

		I2C2_Master_Rx_Polling,	"I2C2 Master Rx (B'd to B'd) by Polling",
		I2C2_Master_Rx_Interrupt,"I2C2 Master Rx (B'd to B'd) by Interrupt",
		I2C2_Slave_Tx_Polling,	"I2C2 Slave  Tx (B'd to B'd) by Polling",
		I2C2_Slave_Tx_Interrupt,	"I2C2 Slave  Tx (B'd to B'd) by Interrupt",
		0,							"---------------------------------------",
		
#ifdef v210_I2C_FPGA
		// Channel 0 & 2
		I2C_CH0_CH2_CrossTest_Polling,	"[*]I2C0 Master TX / I2C2 Slave RX  CrossTest by Polling",
#else
		// Channel 0 & 1
		I2C_CH0_CH1_CrossTest_Polling,	"[*]I2C0 Master TX / I2C1 Slave RX  CrossTest by Polling",
		I2C_CH0_CH1_CrossTest_Interrupt,	"[*]I2C0 Master TX / I2C1 Slave RX  CrossTest by Interrupt",
#endif
		0,							"---------------------------------------",

		I2C_CH0_CH2_AutoTest,	"[*]I2C CH0 & CH2 Auto Test (Master Tx/Rx) from EEPROM(S524AD0XD1) by Interrupt & Polling",
		0,							"---------------------------------------",

#ifndef v210_I2C_FPGA
		I2C_Bus_Arbitration_Test,		  	"I2C CH0 & CH1 Bus Arbitration Test from EEPROM(S524AD0XD1) by Polling",
#endif
		0,							"---------------------------------------",

#ifndef v210_I2C_FPGA		
		I2C_CH0_Performance_Test,	"[*]I2C CH0 Performance Test (Master Tx/Rx) from EEPROM(S524AD0XD1) by Polling",
#endif
		0,							"---------------------------------------",

		I2C_CH0_Aging_Test, 		  	"I2C CH 0 Aging Test",
		0,0
	};

	while(1)
	{
		UART_Printf("\n\n================== IIC Function Test (2009/03/23) =====================\n\n");	
		
		for (uCountFunc=0; (u32)(g_aIICTestFunc[uCountFunc].desc)!=0; uCountFunc++)
			UART_Printf("%2d: %s\n", uCountFunc, g_aIICTestFunc[uCountFunc].desc);

		UART_Printf("\nSelect the function to test : ");
		iSel = UART_GetIntNum();
		UART_Printf("\n");
		if(iSel == -1) 
			break;

		if (iSel>=0 && iSel<(sizeof(g_aIICTestFunc)/8-1))
			(g_aIICTestFunc[iSel].func) ();
	}
}	


//////////
// Function Name : TestSFRIIC
// Function Description : This function test pattern for IIC SFR write/read
// Input : NONE
// Output : None
// Version : v0.0
void TestSFRIIC(void)	
{
	volatile u32 *pAddr;	
	s32 i,j, count = sizeof(sRegInfoIIC)/sizeof(REGINFO), nPattern;
	u32 uRegStatus, uWritePatternWithMask, uReadWithMask;	
	
	u32 uPatternShitfOneBit[] = 
	{
		0x00000001,		0x00000002,		0x00000004,		0x00000008,		0x00000010,		0x00000020,		0x00000040,		0x00000080,
//		0x00000100,		0x00000200,		0x00000400,		0x00000800,		0x00001000,		0x00002000,		0x00004000,		0x00008000,
//		0x00010000,		0x00020000,		0x00040000,		0x00080000,		0x00100000,		0x00200000,		0x00400000,		0x00800000,
//		0x01000000,		0x02000000,		0x04000000,		0x08000000,		0x10000000,		0x20000000,		0x40000000,		0x80000000,
	};
	
	u32 uDefaultBitMask[] =
	{
		0x00000001,		0x00000003,		0x00000007,		0x0000000F,		0x0000001F,		0x0000003F,		0x0000007F,		0x000000FF,
//		0x000001FF,		0x000003FF,		0x000007FF,		0x00000FFF,		0x00001FFF,		0x00003FFF,		0x00007FFF,		0x0000FFFF,
//		0x0001FFFF,		0x0003FFFF,		0x0007FFFF,		0x000FFFFF,		0x001FFFFF,		0x003FFFFF,		0x007FFFFF,		0x00FFFFFF,
//		0x01FFFFFF,		0x03FFFFFF,		0x07FFFFFF,		0x0FFFFFFF,		0x1FFFFFFF,		0x3FFFFFFF,		0x7FFFFFFF,		0xFFFFFFFF,
	};

	
	for(i=0; i<count; i++)
	{		
		pAddr = (u32*)sRegInfoIIC[i].uAddr;		

		if ( (i == 2) || (i == 7) || (i == 12) || (i == 17) )  //When you check Address Regiser, you should Set Serial Output to 0.
		{
			I2C_Outp8(rI2CSTAT0, I2CSTAT_DISABLE);
			I2C_Outp8(rI2CSTAT_HDMI_PHY, I2CSTAT_DISABLE);	
			I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_DISABLE);
			I2C_Outp8(rI2CSTAT2, I2CSTAT_DISABLE);
		}
		else if ( (i == 3) || (i == 8) || (i == 13) || (i == 18) ) //When you check DataShift Regiser, you should Set Serial Output to 1.
		{
			I2C_Outp8(rI2CSTAT0, I2CSTAT_ENABLE); 
			I2C_Outp8(rI2CSTAT_HDMI_PHY, I2CSTAT_ENABLE);	
			I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_ENABLE);
			I2C_Outp8(rI2CSTAT2, I2CSTAT_ENABLE);
		}
		else if ( (i == 4) || (i == 9) || (i == 14) || (i == 19) )    //Clear Serial ouput : when serial output  set.. You cannot check IICCON register
		{
			I2C_Outp8(rI2CSTAT0, I2CSTAT_DISABLE);
			I2C_Outp8(rI2CSTAT_HDMI_PHY, I2CSTAT_DISABLE);	
			I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_DISABLE);
			I2C_Outp8(rI2CSTAT2, I2CSTAT_DISABLE);
		}

		
		if(sRegInfoIIC[i].uFlag == DPDB)
		{		
			if(sRegInfoIIC[i].uRWType == RW)
			{
				for(j=0; j< sRegInfoIIC[i].uBitLen ; j++)
				{			
					uWritePatternWithMask = (uPatternShitfOneBit[j] & uDefaultBitMask[j]);
					*pAddr = uWritePatternWithMask;					
					
					uReadWithMask = (*pAddr & uDefaultBitMask[j]);	
					sRegInfoIIC[i].rValue = uReadWithMask;

					uRegStatus = (uWritePatternWithMask != uReadWithMask );					
					RPRINTF(sRegInfoIIC[i], uPatternShitfOneBit[j], uRegStatus, j, 1);

					/* After the register is tested, reset the register for the next register test */
					*pAddr = 0x0;
				}
			}
		}
		else if(sRegInfoIIC[i].uFlag == DPPB)
		{		
			nPattern = (sizeof(uPatternShitfOneBit) / sizeof(uPatternShitfOneBit[0]));			
			if(sRegInfoIIC[i].uRWType == RW)
			{
				for(j=0; j<nPattern; j++)
				{
					if(uPatternShitfOneBit[j] & sRegInfoIIC[i].uPrivateBitMask)
					{
						uWritePatternWithMask = (uPatternShitfOneBit[j] & sRegInfoIIC[i].uPrivateBitMask);
						*pAddr = uWritePatternWithMask;
						
						uReadWithMask = (*pAddr & sRegInfoIIC[i].uPrivateBitMask);
						sRegInfoIIC[i].rValue = uReadWithMask;
						
						uRegStatus = (uWritePatternWithMask != uReadWithMask );												
						RPRINTF(sRegInfoIIC[i], uPatternShitfOneBit[j], uRegStatus, j, 1);

						/* After the register is tested, reset the register for the next register test */
						*pAddr = 0x0;
					}
				}
			}
		}
	}
	
	UART_Printf("\n\nIIC SFR Read/Write Test Complete!\n\n");
	
	return ;
}


//////////
// Function Name : SFR_TestIIC
// Function Description : This function tests SFR by write and read out from SFR
// Input : NONE
// Output : None
// Version : v0.0
void SFR_TestIIC(void)
{
	TestSFRIIC();
}

//////////
// Function Name : IIC0_Btest
// Function Description : This function tests IIC with 7-bit addresses by write and read out from EEPROM
// Input : NONE
// Output : Print out Data which readed out from EEPROM
// Version : v0.1
void IIC0_BTest( void)
{
	u32 uCnt, uCntErr=0, uLoop=0, uFixcnt=0;
	u8 cData;
	u8 *pCmp;
	u8 *pComp;
	u32 uMin=0x0;
	//u32 uMax=0x100;
	u32 uMax=0x3e0;			// KS24C080C: 1KB (8Kbit) = 0x3e8	

	UART_Printf("[ IIC0 Test using Serial EEPROM (KS24C080C) ]\n");

	IIC0_Open(400000);	

	while(!GetKey())
	{	
		UART_Printf("\nWrite (0x%x ~ 0x%x) and read back from EEPROM (KS24C080C)\n", uMin, (uMax-1));
		pCmp = (u8 *)COMPAdd1;
		pComp = (u8 *)COMPAdd2;	

		for(uCnt=uMin;uCnt<uMax;uCnt++) 
		{
			cData=0;
			*pComp++ = uCnt;
			
			#ifdef v210_I2C_FPGA
			IIC0_Write(KS24C080C_SLAVEADDRESS, uCnt, uCnt);
			IIC0_Read(KS24C080C_SLAVEADDRESS+1,  uCnt, &cData);
			#else
			IIC0_Write(S524AD0XD1_SLAVEADDRESS, uCnt, uCnt);
			IIC0_Read(S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData);
			#endif
			
			while((u8) uCnt != cData)				// If Bus Arbitration failed, So recovery Work Around by SOP on 2008/05/27
			{
				//UART_Printf("\n\n**** Retry Write and Read ***\n\n");		
				#ifdef v210_I2C_FPGA
				IIC0_Write(KS24C080C_SLAVEADDRESS, uCnt, uCnt);
				IIC0_Read(KS24C080C_SLAVEADDRESS+1,  uCnt, &cData);
				#else
				IIC0_Write(S524AD0XD1_SLAVEADDRESS, uCnt, uCnt);
				IIC0_Read(S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData);
				#endif

				uFixcnt++;
			}

			UART_Printf("%02x ", cData);
			*pCmp++ = cData;
			if ((uCnt&0xf)==0xf) UART_Printf("\n");
		}

		pCmp = (u8 *)COMPAdd1;
		pComp = (u8 *)COMPAdd2;

		for (uCnt=uMin; uCnt<uMax; uCnt++)
		{
			if(*pCmp != *pComp)
			{
				uCntErr++;
				UART_Printf("Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x\n", uCntErr,uCnt,*pComp,*pCmp);			
			}			
			
			pCmp++;
			pComp++;
		}
		uLoop++;
	}
	IIC0_Close();
	
	if (uCntErr == 0)
	{
		UART_Printf("\nPass: %d Looping (%d Data Repaired)\n", uLoop, uFixcnt);	
	}
	else
	{
		UART_Printf("\n *** Fail: %d count !!!\n", uCntErr);
		//Pause();
	}
}

//////////
// Function Name : IIC1_Btest
// Function Description : This function tests IIC with 7-bit addresses by write and read out from EEPROM
// Input : NONE
// Output : Print out Data which readed out from EEPROM
// Version : v0.1
void IIC2_BTest( void)		//Added by SOP on 2008/03/04	
{
	u32 uCnt, uCntErr=0, uLoop=0, uFixcnt=0;
	u8 cData;
	u8 *pCmp;
	u8 *pComp;
	u32 uMin=0x0;
	//u32 uMax=0x500;			
	u32 uMax=0x3e00;			// S524AD0XD1: 16KB (128Kbit) = 0x3e80	

	UART_Printf("[ IIC1 Test using Serial EEPROM (S524AD0XD1) ]\n");

	IIC2_Open(400000);	
	
	while(!GetKey())
	{	
		UART_Printf("\nWrite (0x%x ~ 0x%x) and read back from EEPROM (S524AD0XD1)\n", uMin, (uMax-1));
		pCmp = (u8 *)COMPAdd1;
		pComp = (u8 *)COMPAdd2;	

		for(uCnt=uMin;uCnt<uMax;uCnt++) 
		{
			cData=0;
			*pComp++ = uCnt;
			IIC2_Write(S524AD0XD1_SLAVEADDRESS, uCnt, uCnt);
			IIC2_Read(S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData);
			
			while((u8) uCnt != cData)				// If Bus Arbitration failed, So recovery Work Around by SOP on 2008/05/27
			{
				//UART_Printf("\n\n**** Retry Write and Read ***\n\n");		
				IIC2_Write(S524AD0XD1_SLAVEADDRESS, uCnt, uCnt);
				IIC2_Read(S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData);
				uFixcnt++;
			}
			
			UART_Printf("%02x ", cData);
			*pCmp++ = cData;
			if ((uCnt&0xf)==0xf) UART_Printf("\n");
		}

		pCmp = (u8 *)COMPAdd1;
		pComp = (u8 *)COMPAdd2;

		for (uCnt=uMin; uCnt<uMax; uCnt++)
		{
			if(*pCmp != *pComp)
			{
				uCntErr++;
				UART_Printf("Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x\n", uCntErr,uCnt,*pComp,*pCmp);			
			}			
			
			pCmp++;
			pComp++;
		}
		uLoop++;
	}
	IIC2_Close();
	
	if (uCntErr == 0)
	{
		UART_Printf("\nPass: %d Looping (%d Data Repaired)\n", uLoop, uFixcnt);	
	}
	else
	{
		UART_Printf("\n *** Fail: %d count !!!\n", uCntErr);
		//Pause();
	}
}

u32 IIC_ACK_Check(void)
{
	u32 uCnt=0;
	
	while(!IIC_CheckAck()){
		if (uCnt++>0x10000000) {
			UART_Printf("\n[Warning] ACK Checktime Overflow..!!!!");
			return false;
		}
	}	
	return true;
}

u8 Device_ACK_Check(u8 cSlaveAddr)
{
	u32 uTmp1;
	
	uTmp1 = I2C_Inp32(rIICCON0);
	uTmp1 |= ((1<<7) | (1<<5));			// ACK Enable / Interrupt enable 
	uTmp1 &=~(1<<4);					// Clear pending bit to resume	
	I2C_Outp8(rIICCON0,uTmp1);				// Ack generation Enable
			
	I2C_Outp8(rIICDS0,cSlaveAddr);
	I2C_Outp8(rIICSTAT0,0xf0);				// Master Tx Start.

	if (!IIC_ACK_Check()) {
		IIC_DBG_PRINT(DBG_Basic,"\n-> Device Connection check FAIL!!!");
		 return false;
	}	
	I2C_Outp8(rIICSTAT0,0xd0);				// Stop Master Tx condition, ACK flag clear

	//device which has cSlaveaddr is exist~!!
	IIC_DBG_PRINT(DBG_Basic,"\n-> Device Connection check GOOD!!!!.");
	return true;		
}
	



u8 IIC0_Write_EEPROM(u8 cSlaveAddr, u8 cAddr, u8 cData)
{
	u8 cD[2];

	cD[0]= cAddr;
	cD[1]= cData;
	
	IIC_Set_StopSignal(true);
	IIC0_SetWrite(cSlaveAddr, cD, 2);	

	return true;
}


//for EEPRPOM random address Byte read 
u32 IIC0_Read_EEPROM(u8 cSlaveAddr,u8 cAddr,u8 *cData)
{
	IIC_Set_StopSignal(true);
	IIC0_SetWrite(cSlaveAddr, &cAddr, 1);			// following EEPROM random address access procedure	
	IIC0_SetRead(cSlaveAddr, cData, 1);
	return true;
}

#if 0
u8 IIC0_AutoTest(void)				// Modified by jspark at 2009/02/25
{
	u32 uCnt, uCntErr=0;
	u8 cData[3];
	u8 *pCmp;
	u8 *pComp;
	u32 uMin=0x0;
	u32 uMax=0x100;			// 256 byte  ?? (whole size is  1KB)
	u32 iSel;


	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;	
	
	//Inital Buffer data
	for(uCnt=uMin;uCnt<uMax;uCnt++)
	{
		*pCmp++=0;	*pComp++=0;				
	}

	//Inital Buffer index
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;

	IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(KS24C080) Verify Test with Interrupt mode");
	IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");
	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	if ( (iSel =UART_GetIntNum() ) > 500)
		iSel = 500; //MAX 500Khz
		
	IIC0_Open(iSel * 1000);	

	Device_ACK_Check(KS24C080C_SLAVEADDRESS);
	

	IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");
	for(uCnt=uMin;uCnt<uMax;uCnt++)  
	{
		*pCmp++ = uCnt;		
		IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt,uCnt);
		IIC0_Write_EEPROM(KS24C080C_SLAVEADDRESS, uCnt, uCnt);  	//assume address is same data.
		Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
	}

		
	IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
	for(uCnt=uMin;uCnt<uMax;uCnt++)  
	{
		IIC0_Read_EEPROM(KS24C080C_SLAVEADDRESS+1,  uCnt, &cData[0]);
		*pComp++ = cData[0];
		
		IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0],cData[1],cData[2]);				
	}



	IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;
	for (uCnt=uMin; uCnt<uMax; uCnt++)
	{
		if(*pCmp != *pComp)
		{
			uCntErr++;
			IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
		}			
		
		pCmp++;
		pComp++;
	}

	IIC0_Close();
	
	if (uCntErr == 0)
	{
		IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!");	
		return true;	
	}
	else
	{
		IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
		return false;		
	}
	
}
#else
u8 I2C_Ch0_Master_Interrupt_EEPROM_AutoTest(void);
u8 I2C_Ch0_Master_Polling_EEPROM_AutoTest(void);
u8 I2C_Ch0_Master_Interrupt_EEPROM_Test(void);
u8 I2C_Ch0_Master_Polling_EEPROM_Test(void);

#ifdef v210_I2C_FPGA
u8	I2C_Ch2_Master_Interrupt_EEPROM_AutoTest(void);
u8	I2C_Ch2_Master_Polling_EEPROM_AutoTest(void);
u8	I2C_Ch2_Master_Interrupt_EEPROM_Test(void);
u8	I2C_Ch2_Master_Polling_EEPROM_Test(void);
#else
u8	I2C_Ch1_Master_Interrupt_EEPROM_AutoTest(void);
u8	I2C_Ch1_Master_Polling_EEPROM_AutoTest(void);
u8	I2C_Ch1_Master_Interrupt_EEPROM_Test(void);
u8	I2C_Ch1_Master_Polling_EEPROM_Test(void);
#endif/* #ifdef v210_I2C_FPGA */

/**
* {IIC0_AutoTest}
*
* @brief			Ch0 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	void
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8 IIC0_AutoTest(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Ch0_Master_Polling_EEPROM_Test();
	return	ret;
}
#endif/* 0 */

u8 Device2_ACK_Check(u8 cSlaveAddr)
{
	u32 uTmp1;
	
	uTmp1 = I2C_Inp32(rIICCON2);
	uTmp1 |= ((1<<7) | (1<<5));			// ACK Enable / Interrupt enable 
	uTmp1 &=~(1<<4);					// Clear pending bit to resume	
	I2C_Outp8(rIICCON2,uTmp1);				// Ack generation Enable
			
	I2C_Outp8(rIICDS2,cSlaveAddr);
	I2C_Outp8(rIICSTAT2,0xf0);				// Master Tx Start.

	if (!IIC_ACK_Check()) {
		IIC_DBG_PRINT(DBG_Basic,"\n-> Device Connection check FAIL!!!");
		 return false;
	}	
	I2C_Outp8(rIICSTAT2,0xd0);				// Stop Master Tx condition, ACK flag clear

	//device which has cSlaveaddr is exist~!!
	IIC_DBG_PRINT(DBG_Basic,"\n-> Device Connection check GOOD!!!!.");
	return true;		
}
	


//u8 IIC1_Write_EEPROM(u8 cSlaveAddr, u16 cAddr, u8 cData)
u8 IIC2_Write_EEPROM(u8 cSlaveAddr, u8 cAddr, u8 cData)
{
	u8 cD[3];

#if 1
	cD[0]=(cAddr & 0xff00) >> 8;			//MSB First Address
	cD[1]=cAddr & 0xff;					//LSB Last Address
	cD[2]=cData;
	
	IIC_Set_StopSignal(true);
	IIC2_SetWrite(cSlaveAddr, cD, 3);	
#else
//KS24C080C_SLAVEADDRESS control with IIC1
	cD[0]= cAddr;
	cD[1]= cData;
	
	IIC_Set_StopSignal(true);
	IIC2_SetWrite(cSlaveAddr, cD, 2);	

#endif
	return true;
}




//for EEPRPOM random address Byte read 
//u32 IIC1_Read_EEPROM(u8 cSlaveAddr,u16 cAddr,u8 *cData)
u32 IIC2_Read_EEPROM(u8 cSlaveAddr,u8 cAddr,u8 *cData)
{
	u8 cD[2];


	cD[0]=(cAddr & 0xff00) >> 8;			//MSB First Address
	cD[1]=cAddr & 0xff;					//LSB Last Address 
	
	IIC_Set_StopSignal(true);
#if 1	
	IIC2_SetWrite( cSlaveAddr, &cD[0], 2);			// following EEPROM random address access procedure	
	IIC2_SetRead( cSlaveAddr, cData, 1);	
#else
//KS24C080C_SLAVEADDRESS control with IIC1
	IIC2_SetWrite(cSlaveAddr, &cAddr, 1);			// following EEPROM random address access procedure	
	IIC2_SetRead(cSlaveAddr, cData, 1);
#endif	

	
	return true;
}

#ifdef v210_I2C_FPGA
//////////
// Function Name : IIC1_AutoTest
// Function Description : This function tests IIC for AutoTest. (S524AD0XD1 128Kbit)
// Input : NONE
// Output : True (Test success). False (Test fail)
// Version : v0.1
u8 IIC2_AutoTest(void)	// Modified by jspark at 2009/02/25
{
	u32 uCnt, uCntErr=0;
	u8 cData[3];
	u8 *pCmp;
	u8 *pComp;
	u32 uMin=0x0;
	u32 uMax=0x100;			// 256 byte  ?? (whole size is  1KB)
	
	s32 iclkmode,iclkprescale;

	for (iclkmode=1; iclkmode>=0; iclkmode--)
	{
		for (iclkprescale=15; iclkprescale>=0; iclkprescale--)
		{
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;	

			//Inital Buffer data
			for(uCnt=uMin; uCnt<uMax; uCnt++)
			{
				*pCmp++=0;	*pComp++=0;
			}

			//Inital Buffer index
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;

			//KS24C080C_SLAVEADDRESS
			//S524AD0XD1_SLAVEADDRESS

			IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(S524AD0XD1) Verify Test with Interrupt mode");
			IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");

			IIC2_Open_new(iclkmode,iclkprescale);
			Device2_ACK_Check(S524AD0XD1_SLAVEADDRESS);


			IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");
			for(uCnt=uMin;uCnt<uMax;uCnt++)  
			{
				*pCmp++ = uCnt;		
				IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt,uCnt);
				IIC2_Write_EEPROM(S524AD0XD1_SLAVEADDRESS, uCnt, uCnt);  	//assume address is same data.
				Delay(50);  //for EEPROM internal writing cycle max 5ms
			}

			IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
			for(uCnt=uMin; uCnt<uMax; uCnt++)  
			{
				IIC2_Read_EEPROM(S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);		
				*pComp++ = cData[0];
				IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0],cData[1],cData[2]);				
			}

			IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;
			for (uCnt=uMin; uCnt<uMax; uCnt++)
			{
				if(*pCmp != *pComp)
				{
					uCntErr++;
					IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pCmp,*pComp);			
				}			

				pCmp++;
				pComp++;
			}

			IIC2_Close();

			if (uCntErr == 0)
			{
				IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!");	
				//return true;
			}
			else
			{
				IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
				return false;	
			}
		}
	}

	return true;	
}
#endif/* #ifdef v210_I2C_FPGA */

//Only ACK check...
void ACK_Check_Polling(u32 ch)
{
	u32 uTmp,uCnt=0;
		
	uTmp = I2C_Inp32(IIC_REG[ch].CON);	
	while((uTmp&0x10) != 0x10)  //Interrupt Pending (ACK check)
	{
		uTmp = I2C_Inp32(IIC_REG[ch].CON);
		uCnt++;
		if (uCnt > 0x10000) {
			UART_Printf("\n[Warning] ACK Checktime Overflow..!!!!(Polling)");
			break;
		}
	}
}

void Pending_Bit_Clear(u32 ch)
{
	u32 uTmp;
	uTmp = I2C_Inp32(IIC_REG[ch].CON);
	uTmp &= ~(1<<4);				//	Clear pending bit to resume
	I2C_Outp8(IIC_REG[ch].CON,uTmp);			
}
		
void StopGen_Polling(u32 ch)
{
	u32 uTmp;
	
	I2C_Outp8(IIC_REG[ch].STAT,0xd0);		//	Stop Master Tx condition, ACK flag clear
				
	Pending_Bit_Clear(ch);
}

u8 Device_ACK_Check_Polling(u32 ch, u8 cSlaveAddr)
{
	u32 uTmp;
	
	//Wait until IIC bus is free.
	uTmp = I2C_Inp32(IIC_REG[ch].STAT);
	while(uTmp&(1<<5))					
	uTmp = I2C_Inp32(IIC_REG[ch].STAT);

	uTmp = I2C_Inp32(IIC_REG[ch].CON);
	uTmp &= ~ (1<<7);			// ACK Disable	
	I2C_Outp8(IIC_REG[ch].CON,uTmp);		
	
	I2C_Outp8(IIC_REG[ch].DS,cSlaveAddr);
	I2C_Outp8(IIC_REG[ch].STAT,0xf0);		// Master Tx Start.

	ACK_Check_Polling(ch);
	StopGen_Polling(ch);

	return true;		
}

//for WM8580 Register reading
u8 IIC0_Read_DeviceRegBytePolling(u32 ch, u8 SlaveAddr,u8 DeviceReg)
{
	u32 uTmp,i=0;
	u8  uData;
	//Wait until IIC bus is free.
	uTmp = I2C_Inp32(IIC_REG[ch].STAT);
	while(uTmp&(1<<5))					
	uTmp = I2C_Inp32(IIC_REG[ch].STAT);

/*
	uTmp = I2C_Inp32(IIC_REG[ch].CON);
	uTmp &= ~ (1<<7);					// ACK Disable	
	I2C_Outp8(IIC_REG[ch].CON,uTmp);		
//Write device Slave Address
	I2C_Outp8(IIC_REG[ch].DS,SlaveAddr); 	//for Device Slave address
	I2C_Outp8(IIC_REG[ch].STAT,0xf0);		// Master Tx Start. (*address output)
	ACK_Check_Polling(ch);
	
//Write Register Address
//	I2C_Outp8(IIC_REG[ch].DS,(DeviceReg<<1) | 0x1);
	I2C_Outp8(IIC_REG[ch].DS,DeviceReg);
	Pending_Bit_Clear(ch);	  		   	//data TX go.. (*register output)
	ACK_Check_Polling(ch);     			//ACK check

       StopGen_Polling(ch);
	Delay(1); //why? (write time margin)
	*/


//Rx Start signal
	uTmp = I2C_Inp32(IIC_REG[ch].CON);
	uTmp |=  (1<<7);					// ACK Enable
	I2C_Outp8(IIC_REG[ch].CON,uTmp);		


	I2C_Outp8(IIC_REG[ch].DS,SlaveAddr+1); 	//for Device Slave address	
	I2C_Outp8(IIC_REG[ch].STAT,0xb0);		// Master Rx Start. (*address output)	
	ACK_Check_Polling(ch);
	Delay(1); //why? (write time margin)
	Pending_Bit_Clear(ch);  				//RX ready~!! Next Reading Start

//Rx Start signal	 
	 for(;;)
	{
		if(I2C_Inp8(IIC_REG[ch].CON)&0x10)  // ACK generation Done?
		{
			uData= I2C_Inp32(IIC_REG[ch].DS);		//Read data
			Pending_Bit_Clear(ch);  				//RX ready~!! Next Reading Start
			break;
		}
	}	

	I2C_Outp8(IIC_REG[ch].STAT,0x90);		//	Stop Master Rx condition
	Pending_Bit_Clear(ch);

	return uData;
}

u32 IIC0_Read_Polling(u32 ch, u32 SlaveAddr,u8* data, u32 RxSize,u32 StopGen)
{
	u32 uTmp,i=0;
	u8  uData;

	//Wait until IIC bus is free.
	uTmp = I2C_Inp32(IIC_REG[ch].STAT);
	while(uTmp&(1<<5))					
	uTmp = I2C_Inp32(IIC_REG[ch].STAT);

	
	uTmp = I2C_Inp32(IIC_REG[ch].CON);
	uTmp |= (1<<7);			// ACK Enable
	I2C_Outp8(IIC_REG[ch].CON,uTmp);		
	
	I2C_Outp8(IIC_REG[ch].DS, SlaveAddr+1);
	I2C_Outp8(IIC_REG[ch].STAT, 0xb0);		// Master Rx Start.

	ACK_Check_Polling(ch);
	Pending_Bit_Clear(ch);  		//RX ready~!! Next Reading Start

	for(;;)
	{
		if(I2C_Inp8(IIC_REG[ch].CON)&0x10)  // ACK generation Done?
		{
			uData= I2C_Inp32(IIC_REG[ch].DS);
			*data++ = uData; 			//after data received, we should pending bit clear
		
			 Pending_Bit_Clear(ch);  //RX ready~!!
			 
			//UART_Printf("\n (IIC_Read_Polling) [%d] -> data : 0x%x",i,uData);
			 UART_Printf(".");
		
			i++;
			if ( i >= RxSize)
				break;
		}	
	}
	
	if (StopGen)	
	{
		//StopGen_Polling(ch);
		I2C_Outp8(IIC_REG[ch].STAT,0x90);		//	Stop Master Rx condition
		Pending_Bit_Clear(ch);
	}
		
	return true;
}

u32 IIC0_Write_Polling(u32 ch, u32 SlaveAddr,u8* data, u32 TxSize,u32 StopGen)
{
	u32 uTmp,i;
	
	//Wait until IIC bus is free.
	uTmp = I2C_Inp32(IIC_REG[ch].STAT);
	while(uTmp&(1<<5))					
	uTmp = I2C_Inp32(IIC_REG[ch].STAT);

	uTmp = I2C_Inp32(IIC_REG[ch].CON);
	uTmp &= ~ (1<<7);			// ACK Disable	
	I2C_Outp8(IIC_REG[ch].CON,uTmp);		
	//UART_Printf("\n (IIC_Write_Polling) IICCON0 : 0x%x", I2C_Inp32(rIICCON0));
	
	I2C_Outp8(IIC_REG[ch].DS,SlaveAddr);
	I2C_Outp8(IIC_REG[ch].STAT,0xf0);		// Master Tx Start.

	ACK_Check_Polling(ch);
	
	for(i=0;i<TxSize;i++)
	{
		I2C_Outp8(IIC_REG[ch].DS,*data++);
		Pending_Bit_Clear(ch);		  //data TX go..
		ACK_Check_Polling(ch);  		//ACK check
	}
	
	if (StopGen)
		StopGen_Polling(ch);
		
		return true;			
}

void IIC0_AutoTest_Polling(void)
{
	u32	i,ErrCnt=0;
	u8	data[256+4];
	
	UART_Printf("\n[ (JSPARK) IIC0 Master Tx/RX Polling Test with EEPROM(KS24C080C)]");
	IIC0_OpenPolling(400000);
	
//----------------------
//Data Writing to EEPROM
//----------------------
	UART_Printf("\n (JSPARK) EEPROM 256 byte Writing....\n");
	for( i = 0; i < 256; i++)
	{
		data[0]=i;	//EEPROM Address
		#if PATTERN_TEST
		data[1] = testpattern[i]; //Pattern Data
		#else
		data[1] = i;       //EEPROM Data		
		#endif
		
		#ifdef v210_I2C_FPGA
		IIC0_Write_Polling(IIC_CH0, KS24C080C_SLAVEADDRESS,&data[0], 2,STOP_GEN);
		#else
		IIC0_Write_Polling(IIC_CH0, S524AD0XD1_SLAVEADDRESS,&data[0], 2,STOP_GEN);
		#endif
		
		Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)	      
		
		UART_Printf(".");
	}


//----------------------
//Data Reading to EEPROM
//----------------------
	UART_Printf("\n (JSPARK) EEPROM 256 byte Reading.... \n");
	//Read Start Address Setting
	data[0] = 0;  //EEPROM Word Address

	#ifdef v210_I2C_FPGA
	IIC0_Write_Polling(IIC_CH0, KS24C080C_SLAVEADDRESS,&data[0], 1,STOP_GEN);
	#else
	IIC0_Write_Polling(IIC_CH0, S524AD0XD1_SLAVEADDRESS,&data[0], 2,STOP_GEN);
	#endif

	Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)	      

	for( i = 0; i < 256; i++)
		data[i] = 0;	//data inital
	//Sequentail 256 byte reading from EEPROM
	#ifdef v210_I2C_FPGA
	IIC0_Read_Polling(IIC_CH0, KS24C080C_SLAVEADDRESS,&data[0], 256, STOP_GEN);
	#else
	IIC0_Read_Polling(IIC_CH0, S524AD0XD1_SLAVEADDRESS,&data[0], 256, STOP_GEN);
	#endif
	
	UART_Printf("\n (JSPARK) Comparing....");
	for( i = 0 ; i < 256 ; i++)
	{
		#if PATTERN_TEST
		if ( data[i] != testpattern[i] )
		#else	
		if ( data[i] != i )
		#endif
		{
			ErrCnt++;
			#if PATTERN_TEST
	   		UART_Printf("\n Error pattern (0x%x) ==>  (0x%x)",testpattern[i],data[i]);			
			#else
	   		UART_Printf("\n Error pattern (0x%x) ==>  (0x%x)",i,data[i]);				   		
			#endif
		}
	}

	if (ErrCnt)		
		UART_Printf("\n ==> TEST Error !!!");
	else
		UART_Printf("\n ==> TEST Good !!!");

	IIC0_Close();	
}

void IIC_SlaveAddressMatch_Polling(u32 ch)
{
	u32	uTmp;
	u8		uData;
	
	uTmp = I2C_Inp32(IIC_REG[ch].STAT);
	while( !(uTmp&(1<<2) ))				//wait Address Match (for Slave RX)		
		uTmp = I2C_Inp32(IIC_REG[ch].STAT);

	uData=I2C_Inp8(IIC_REG[ch].DS);			//Read data from IICDS
	
	Pending_Bit_Clear(ch);	
}

//////////
// Function Name : IIC_MTx_SRx_CrossTest_Polling
// Function Description : This function do Master TX->SlaveRx mode by polling operation
//                                  with tx_ch,rx_ch
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Rx]
// Output : NONE
// Version : v0.1
// -------------------------
// Master TX -->   Slave RX 
void	IIC_MTx_SRx_CrossTest_Polling(u32 tx_ch, u32 rx_ch)
{
	u32	uTmp,i,ErrCnt=0;
	u8 SlaveAddr,RxDone=0;
	u8 RxData[256+4];

	IIC_sfr_mapping_Init(tx_ch);
	IIC_sfr_mapping_Init(rx_ch);

	SlaveAddr = 0x54;  //assume channel 2 Slave address is 0x55
	
//------------------------------------
//Ch 2  for Slave RX
	uTmp = I2C_Inp32(IIC_REG[rx_ch].STAT);
	while(uTmp&(1<<5))						//	Wait until IIC bus is free.
		uTmp = I2C_Inp32(IIC_REG[rx_ch].STAT);			

	uTmp = I2C_Inp32(IIC_REG[rx_ch].CON);
	uTmp |= (1<<7);
	I2C_Outp8(IIC_REG[rx_ch].CON, uTmp);		//	Ack generation Enable

	uTmp = I2C_Inp32(IIC_REG[rx_ch].STAT);
	uTmp &= ~(1<<4);
	I2C_Outp8(IIC_REG[rx_ch].STAT, uTmp);		// Disable Rx/Tx for setting SlaveAddr
	I2C_Outp8(IIC_REG[rx_ch].ADD,  SlaveAddr);
	I2C_Outp8(IIC_REG[rx_ch].STAT, 0x10);		//	Slave Rx Start.
//------------------------------------
	
//------------------------------------
//Ch 0  for Master TX
	//Wait until IIC bus is free.
	uTmp = I2C_Inp32(IIC_REG[tx_ch].STAT);
	while(uTmp&(1<<5))					
		uTmp = I2C_Inp32(IIC_REG[tx_ch].STAT);

	uTmp = I2C_Inp32(IIC_REG[tx_ch].CON);	
	uTmp &= ~ (1<<7);								// ACK Disable
	I2C_Outp8(IIC_REG[tx_ch].CON,uTmp);
	
	I2C_Outp8(IIC_REG[tx_ch].DS,  SlaveAddr);
	I2C_Outp8(IIC_REG[tx_ch].STAT, 0xf0);		// Master Tx Start.
//------------------------------------
	Delay (1);								//this is critical point.. why?

	IIC_SlaveAddressMatch_Polling(rx_ch);	// RX address matching

	Pending_Bit_Clear(tx_ch);
	ACK_Check_Polling(tx_ch);
//------------------------------------	

	for(i=0;i<256;i++)
	{
		//CH0 Master TX
		I2C_Outp8(IIC_REG[tx_ch].DS, testpattern[i]); 

		Pending_Bit_Clear(rx_ch);		//	CH2 : RX : Clear pending bit to resume  (޴ RX pending clear ؾ غ .  (  ᱹ TX  ϰ .  )
		Pending_Bit_Clear(tx_ch);		//	CH0 : TX : Clear pending bit to resume
	
		//CH2 Slave RX check
		RxDone = 0;
		while( !RxDone )
		{
			if( I2C_Inp8(IIC_REG[rx_ch].CON) & 0x10 ) 
			{
				RxData[i] = I2C_Inp8(IIC_REG[rx_ch].DS);
				RxDone = true;
				Pending_Bit_Clear(rx_ch);		//	RX : Clear pending bit to resume
			}
		}

		//CH0 ACk detect?
		ACK_Check_Polling(tx_ch);

		UART_Printf("\r Master TX (0x%x) ==> ", testpattern[i]);
		UART_Printf(" Slave RX (0x%x) ", RxData[i]);
		//while (GetKey() == 0) ;			
	}


	//CH0 Master TX Stop
	StopGen_Polling(tx_ch);

	//CH2 Slave RS stop
	I2C_Outp8(IIC_REG[rx_ch].STAT, 0x00);		//	Stop Master Tx condition, ACK flag clear
	Pending_Bit_Clear(rx_ch);				//	RX : Clear pending bit to resume
	
	//CH0,CH1 ALL close..
	IIC0_Close();	
	IIC2_Close();		


	for (i=0;i<256;i++)
	{
		if (RxData[i] != testpattern[i])
		{
			ErrCnt++;
	   		UART_Printf("\n Error pattern (0x%x) ==>  (0x%x)",testpattern[i],RxData[i]);			
		}			
	}

	if (ErrCnt)		
		UART_Printf("\n ==> TEST Error !!!");
	else
		UART_Printf("\n ==> TEST Good !!!");
}



//////////
// Function Name : IIC_CrossTest_Polling
// Function Description : This function do Master TX->SlaveRx mode by polling operation
//                                  with tx_ch,rx_ch
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Rx]
// Output : NONE
// Version : v0.1
// -------------------------
// Master TX -->   Slave RX 
void IIC_CrossTest_Polling(void)
{
	UART_Printf("\n\n\n");
	UART_Printf("\n (JSPARK) IIC_CrossTest_Polling : CH0 Master TX --> CH2 Slave RX");
	UART_Printf("\n (JSPARK) CH0 CH2  ؾմϴ. غǸ ƹŰ ..");
	while(!GetKey());
	IIC0_OpenPolling(400000);
	IIC2_OpenPolling(400000);
	IIC_MTx_SRx_CrossTest_Polling(IIC_CH0, IIC_CH2);


	UART_Printf("\n\n\n");	
	UART_Printf("\n (JSPARK) IIC_CrossTest_Polling : CH2 Master TX --> CH0 Slave RX");
	UART_Printf("\n (JSPARK) CH2 CH0  ؾմϴ. غǸ ƹŰ ..");
	while(!GetKey());
	IIC0_OpenPolling(400000);
	IIC2_OpenPolling(400000);
	IIC_MTx_SRx_CrossTest_Polling(IIC_CH2, IIC_CH0);
}

//////////
// Function Name : IIC0_MasterTxP
// Function Description : This function tests polling mode of the MasterTx by B'd to B'd
// Input : NONE
// Output : Transmit data to SlaveRx Board
// Version : v0.1
// note) to test this function, SlaveRx should be ready
void IIC0_MasterTxP(void)
{
	u32 sCnt;
	u8 cData[100];
	static u8 cSlaveAddr = 0xd0;	

	UART_Printf("\n[ IIC0 Master Tx Test (Board to Board) ]\n");
	UART_Printf("Connect IIC related signals (IICSCL0, IICSDA0) between IIC0 Master TX Board and IIC0 Slave RX Board.\n");
	UART_Printf("Are you excution IIC0 Slave Rx (B'd to B'd) by Polling(or Interrupt) of IIC0 Slave Rx Board ?\n");	
	IIC0_OpenPolling(400000);
	//Pause();

	for(sCnt=0;sCnt<100;sCnt++)
		cData[sCnt]=(u8)sCnt;

	IIC0_MasterWrP(cSlaveAddr, cData);
	UART_Printf("\n*** Completed transmit data.\n");	
	IIC0_Close();
}


//////////
// Function Name : IIC1_MasterTxP
// Function Description : This function tests polling mode of the MasterTx by B'd to B'd
// Input : NONE
// Output : Transmit data to SlaveRx Board
// Version : v0.1
// note) to test this function, SlaveRx should be ready
void IIC2_MasterTxP(void)		//Added by SOP on 2008/03/04	
{
	u32 sCnt;
	u8 cData[100];
	static u8 cSlaveAddr = 0xd0;	

	UART_Printf("\n[ IIC1 Master Tx Test (Board to Board) ]\n");
	UART_Printf("Connect IIC related signals (IICSCL1, IICSDA1) between IIC1 Master TX Board and IIC1 Slave RX Board.\n");
	UART_Printf("Are you excution IIC1 Slave Rx (B'd to B'd) by Polling(or Interrupt) of IIC1 Slave Rx Board ?\n");	
	IIC2_OpenPolling(400000);
	//Pause();

	for(sCnt=0;sCnt<100;sCnt++)
		cData[sCnt]=(u8)sCnt;

	IIC2_MasterWrP(cSlaveAddr, cData);
	UART_Printf("\n*** Completed transmit data.\n");	
	IIC2_Close();
}


//////////
// Function Name : IIC0_MasterRxP
// Function Description : This function tests polling mode of the MasterRx by B'd to B'd
// Input : NONE
// Output : Print out Data which readed out from SlaveTx Board
// Version : v0.1
// note ) to test this function , SlaveTx should be ready
void IIC0_MasterRxP(void)
{
	s32 cCnt;
	static u8 cData[256];
	static u8 cSlaveAddr = 0xc0;	

	UART_Printf("[ IIC0 Master Rx Test (Board to Board) ]\n");
	IIC0_OpenPolling(400000);

	for(cCnt=0;cCnt<101;cCnt++)
		cData[cCnt]=0;

	UART_Printf("Connect IIC0 related signals (IICSCL0, IICSDA0) between IIC0 Master RX Board and IIC0 Slave TX Board.\n");
	UART_Printf("At first, SlaveTx should set up before MasterRx. Make sure that SlaveTx is set up.\n");	
	//Pause();
	
	IIC0_MasterRdP(cSlaveAddr,cData);

	UART_Printf("Received Address: 0x%02x\n\n",(u32)cSlaveAddr);
	for(cCnt=1;cCnt<101;cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data.\n");	
	IIC0_Close();
}


//////////
// Function Name : IIC1_MasterRxP
// Function Description : This function tests polling mode of the MasterRx by B'd to B'd
// Input : NONE
// Output : Print out Data which readed out from SlaveTx Board
// Version : v0.1
// note ) to test this function , SlaveTx should be ready
void IIC2_MasterRxP(void)		//Added by SOP on 2008/03/04	
{
	s32 cCnt;
	static u8 cData[256];
	static u8 cSlaveAddr = 0xc0;	

	UART_Printf("[ IIC1 Master Rx Test (Board to Board) ]\n");
	IIC2_OpenPolling(400000);

	for(cCnt=0;cCnt<101;cCnt++)
		cData[cCnt]=0;

	UART_Printf("Connect IIC1 related signals (IICSCL1, IICSDA1) between IIC1 Master RX Board and IIC1 Slave TX Board.\n");
	UART_Printf("At first, SlaveTx should set up before MasterRx. Make sure that SlaveTx is set up.\n");	
	//Pause();
	
	IIC2_MasterRdP(cSlaveAddr,cData);

	UART_Printf("Received Address: 0x%02x\n\n",(u32)cSlaveAddr);
	for(cCnt=1;cCnt<101;cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data.\n");	
	IIC2_Close();
}


//////////
// Function Name : IIC0_SlaveTxP
// Function Description : This function tests polling mode of the SlaveTx by B'd to B'd
// Input : NONE
// Output : Transmit data to MasterRx Board
// Version : v0.1
void IIC0_SlaveTxP(void)
{
	u32 sCnt;
	u8 cData[100];
	static u8 cSlaveAddr = 0xc0;

	UART_Printf("[ IIC0 Slave Tx Test (Board to Board) ]\n");
	UART_Printf("Connect IIC0 related signals (IICSCL0, IICSDA0) between IIC0 Master RX Board and IIC0 Slave TX Board.\n");
	IIC0_OpenPolling(400000);

	for(sCnt=0;sCnt<100;sCnt++)
		cData[sCnt]=(u8)sCnt;

	IIC0_SlaveWrP(&cSlaveAddr, cData);
	UART_Printf("\n*** Completed transmit data.\n");		
	IIC0_Close();
}


//////////
// Function Name : IIC1_SlaveTxP
// Function Description : This function tests polling mode of the SlaveTx by B'd to B'd
// Input : NONE
// Output : Transmit data to MasterRx Board
// Version : v0.1
void IIC2_SlaveTxP(void)		//Added by SOP on 2008/03/04	
{
	u32 sCnt;
	u8 cData[100];
	static u8 cSlaveAddr = 0xc0;

	UART_Printf("[ IIC1 Slave Tx Test (Board to Board) ]\n");
	UART_Printf("Connect IIC1 related signals (IICSCL1, IICSDA1) between IIC1 Master RX Board and IIC1 Slave TX Board.\n");
	IIC2_OpenPolling(400000);

	for(sCnt=0;sCnt<100;sCnt++)
		cData[sCnt]=(u8)sCnt;

	IIC2_SlaveWrP(&cSlaveAddr, cData);
	UART_Printf("\n*** Completed transmit data.\n");		
	IIC2_Close();
}


//////////
// Function Name : IIC0_SlaveRxP
// Function Description : This function tests polling mode of the SlaveRx by B'd to B'd connection
// Input : NONE
// Output : Print out Data which readed out from MasterTx Board
// Version : v0.1
void IIC0_SlaveRxP(void)
{
	s32 cCnt;
	static u8 cData[256];
	static u8 cSlaveAddr = 0xd0;

	UART_Printf("\n[ IIC0 Slave Rx Test (Board to Board) ]\n");
	IIC0_OpenPolling(400000);	

	for(cCnt=0;cCnt<101;cCnt++)
		cData[cCnt]=0;

	UART_Printf("Connect IIC0 related signals (IICSCL0, IICSDA0) between IIC0 Master Tx (Polling) Board and IIC0 Slave RX Board.\n");
	IIC0_SlaveRdP(&cSlaveAddr,cData);
	cSlaveAddr = cData[0];			// Added by SOP on 2008/05/23
	
	UART_Printf("*** Received Address: 0x%02x\n\n",(u32)cSlaveAddr);
	
	for(cCnt=1;cCnt<101;cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data\n");
	IIC0_Close();
}


//////////
// Function Name : IIC1_SlaveRxP
// Function Description : This function tests polling mode of the SlaveRx by B'd to B'd connection
// Input : NONE
// Output : Print out Data which readed out from MasterTx Board
// Version : v0.1
void IIC2_SlaveRxP(void)		//Added by SOP on 2008/03/04	
{
	s32 cCnt;
	static u8 cData[256];
	static u8 cSlaveAddr = 0xd0;

	UART_Printf("\n[ IIC1 Slave Rx Test (Board to Board) ]\n");
	IIC2_OpenPolling(400000);	

	for(cCnt=0;cCnt<101;cCnt++)
		cData[cCnt]=0;

	UART_Printf("Connect IIC1 related signals (IICSCL1, IICSDA1) between IIC1 Master TX Board and IIC1 Slave RX Board.\n");
	IIC2_SlaveRdP(&cSlaveAddr,cData);
	cSlaveAddr = cData[0];			// Added by SOP on 2008/05/23
	
	UART_Printf("*** Received Address: 0x%02x\n\n",(u32)cSlaveAddr);
	
	for(cCnt=1;cCnt<101;cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data\n");
	IIC2_Close();
}


//////////
// Function Name : IIC0_SlaveTxInt
// Function Description : This function tests Interrupt mode of the SlaveTx by B'd to B'd
// Input : NONE
// Output : Transmit data to MasterRx Board
// Version : v0.1
void IIC0_SlaveTxInt(void)
{
	u32 sCnt;
	u8 cData[100];
	static u8 cSlaveAddr= 0xc0;

	UART_Printf("\n[ IIC0 Slave Tx (Interrupt Mode) Test (Board to Board) ]\n");
	UART_Printf("Connect IIC0 related signals (IICSCL0, IICSDA0) between IIC0 Master RX(Polling) Board and IIC0 Slave TX Board.\n");
	IIC0_Open(400000);					// Interrupt mode, Edited by SOP on 2008/03/03

	for(sCnt=0;sCnt<100;sCnt++)
		cData[sCnt]=(u8)sCnt;

	IIC0_SlaveWrInt(&cSlaveAddr, cData);
	UART_Printf("\n*** Completed transmit data\n");	
	IIC0_Close();
}


//////////
// Function Name : IIC1_SlaveTxInt
// Function Description : This function tests Interrupt mode of the SlaveTx by B'd to B'd
// Input : NONE
// Output : Transmit data to MasterRx Board
// Version : v0.1
void IIC2_SlaveTxInt(void)		//Added by SOP on 2008/03/04	
{
	u32 sCnt;
	u8 cData[100];
	static u8 cSlaveAddr= 0xc0;

	UART_Printf("\n[ IIC1 Slave Tx (Interrupt Mode) Test (Board to Board) ]\n");
	UART_Printf("Connect IIC1 related signals (IICSCL1, IICSDA1) between IIC1 Master RX(Polling) Board and IIC1 Slave TX Board.\n");
	IIC2_Open(400000);					// Interrupt mode, Edited by SOP on 2008/03/03

	for(sCnt=0;sCnt<100;sCnt++)
		cData[sCnt]=(u8)sCnt;

	IIC2_SlaveWrInt(&cSlaveAddr, cData);
	UART_Printf("\n*** Completed transmit data\n");	
	IIC2_Close();
}


//////////
// Function Name : IIC0_SlaveRxInt
// Function Description : This function tests Interrupt mode of the SlaveRx by B'd to B'd connection
// Input : NONE
// Output : Print out Data which readed out from MasterTxP Board
// Version : v0.1
void IIC0_SlaveRxInt(void)
{
	s32 cCnt;
	static u8 cData[256];
	static u8 cSlaveAddr = 0xd0;
	
    	UART_Printf("\n[ IIC0 Slave Rx Test (Board to Board) ]\n");
	IIC0_Open(400000);					//Interrupt mode, Edited by SOP on 2008/03/03	
   
	for(cCnt=0;cCnt<101;cCnt++)
		cData[cCnt]=0;

    	UART_Printf("Connect IIC0 related signals (IICSCL0, IICSDA0) between IIC0 Master TX(Polling) Board and IIC0 Slave RX Board.\n");
	IIC0_SlaveRdInt(&cSlaveAddr,cData);
	cSlaveAddr = cData[0];
	
	UART_Printf("Received Addrress: 0x%02x\n\n",(u32)cSlaveAddr);
	
	for(cCnt=1;cCnt<101;cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data\n");
	IIC0_Close();
}


//////////
// Function Name : IIC1_SlaveRxInt
// Function Description : This function tests Interrupt mode of the SlaveRx by B'd to B'd connection
// Input : NONE
// Output : Print out Data which readed out from MasterTxP Board
// Version : v0.1
void IIC2_SlaveRxInt(void)		//Added by SOP on 2008/03/04	
{
	s32 cCnt;
	static u8 cData[256];
	static u8 cSlaveAddr = 0xd0;
	
   	UART_Printf("\n[ IIC1 Slave Rx Test (Board to Board) ]\n");
	IIC2_Open(400000);					//Interrupt mode, Edited by SOP on 2008/03/03	
   
	for(cCnt=0;cCnt<101;cCnt++)
		cData[cCnt]=0;

    	UART_Printf("Connect IIC1 related signals (IICSCL1, IICSDA1) between IIC1 Master TX(Polling) Board and IIC1 Slave RX Board.\n");
	IIC2_SlaveRdInt(&cSlaveAddr,cData);
	cSlaveAddr = cData[0];
	
	UART_Printf("Received Addrress: 0x%02x\n\n",(u32)cSlaveAddr);
	
	for(cCnt=1;cCnt<101;cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data\n");
	IIC2_Close();
}

//////////
// Function Name : IIC0_WM8990Test
// Function Description : This function tests IIC with 7-bit addresses by write and read out from WM8990 (Add/Data 16-bit)
// Input : NONE
// Output : Print out Data which readed out from WM8990
// Version : v0.1
// Date: 2008/09/25
void IIC0_WM8990Test( void)					// Wolfson WM8990
{
	u16 cData;
	u8 uWDeviceID=0x34;		// WM8990	
	u8 uRDeviceID=0x35;	
	//u8 uWDeviceID=0x36;		// WM8990	
	//u8 uRDeviceID=0x37;

	UART_Printf("[ IIC0 Test using Audio Codec (WM8990) ]\n");

	IIC0_Open(400000);

	UART_Printf("\n\n");
	IIC0_Write16(uWDeviceID, 0x02, 0xaaaa);	
	IIC0_Write16(uWDeviceID, 0x03, 0xf0f0);		
	IIC0_Write16(uWDeviceID, 0x17, 0x1fff);		

	cData = IIC0_Read16(uRDeviceID,  0x00);		// Read: Address=0x00, Data = 0x8990
	UART_Printf("Read Data (Expected 0x8990): 0x%04x\n", cData);			
	cData = IIC0_Read16(uRDeviceID,  0x02);		
	UART_Printf("If Write=0xaaaa, Read Data (Expected 0xaaa2): 0x%04x\n", cData);
	cData = IIC0_Read16(uRDeviceID,  0x03);		
	UART_Printf("If Write=0xf0f0, Read Data (Expected 0x30f0): 0x%04x\n", cData);	
	cData = IIC0_Read16(uRDeviceID,  0x17);	
	UART_Printf("If Write=0x1fff, Read Data (Expected 0x1fff): 0x%04x\n", cData);
	//Pause();	
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef v210_I2C_FPGA
/**
* {I2C1_CLKDIV}
*
* @brief			This function tests is Clock Divider control.
* @conditions	
* @param[in]	HCLKX2, PCLK Divider Value
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_CLKDIV( u32 uDividerValue )
{
	u8	ret = TRUE;
	
	Assert( uDividerValue > 1 );
	if( uDividerValue == 1 )
		uDividerValue = 2;
	
	SYSC_SetClkDiv(eCLKDIV_PCLKDSYS, uDividerValue);
	UART_InitDebugCh(3, 115200);
	Delay(100);

	UART_Printf("\nPCLK Divider Value: %d\n", uDividerValue);
	ClockInform();

	ret = I2C_CH1_CH0_CrossTest_Polling();
	
	Delay(100);
	ClockInform();
	
	return	ret;
}

/**
* {I2C1_CLKDIVTest}
*
* @brief			This function tests is Clock divider control.
* @conditions	
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_CLKDIVTest(void)
{
	u8	ret = TRUE;
	
	u32 uDividerValue = 2;
	u32	uDividerRange = 8;
	
	UART_Printf("[PCLKDSYS Divider Range(2~8)]\n");
	UART_Printf("[I2C1 Clock Divider Control Test using Cross Test with Ch 0]\n");
	ClockInform();

	/* 
		// PCLK PSYS = HCLK PSYS / 2
		g_uPclkPsys = g_uHclkPsys / 2
		g_uPclkPsys = g_uHclkPsys / Divider Range(2~8)
		refer to 'SYSC_UpdateClkInform'		
	*/
	for( uDividerValue = 2; uDividerValue <= uDividerRange; uDividerValue++ )
	{
		ret = I2C1_CLKDIV(uDividerValue);
		if( ret == FALSE )
			break;
	}

	/* restore clock setting */
	SYSC_SetClkDiv(eCLKDIV_PCLKDSYS, 1);
	UART_InitDebugCh(3, 115200);
	
	return	ret;
}

/**
* {I2C1_CLKGateTest}
*
* @brief			This function tests is Clock gating control.
* @conditions	
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_CLKGateTest(void)
{
	u8	ret = TRUE;
	
	UART_Printf("\n[I2C1 Clock Gating Control Test using Cross Test with CH 0]\n");

	ret = I2C1_CLKDIV(2);
	if( ret == FALSE )
	{
		UART_Printf("\n[I2C1 CLOCK DIVIDER TEST FAIL]\n");
		return	ret;
	}
	
	UART_Printf("\n[I2C1 OFF (CLK_GATE_IP3)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIDDC, FALSE);
	Delay(100);
	Pause();
	
	UART_Printf("\n[I2C1 ON (CLK_GATE_IP3)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_I2CHDMIDDC, TRUE);
	Delay(100);
	
	ret = I2C1_CLKDIV(4);
	if( ret == FALSE )
	{
		UART_Printf("\n[I2C2 CLOCK DIVIDER TEST FAIL]\n");
		return	ret;
	}

	return	ret;
}
#endif

/**
* {I2C0_CLKDIV}
*
* @brief			This function tests is Clock Divider control.
* @conditions	
* @param[in]	HCLKX2, PCLK Divider Value
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_CLKDIV( u32 uDividerValue )
{
	u8	ret = TRUE;
	
	Assert( uDividerValue > 1 );
	if( uDividerValue == 1 )
		uDividerValue = 2;
	
	SYSC_SetClkDiv(eCLKDIV_PCLKPSYS, uDividerValue);
	UART_InitDebugCh(3, 115200);
	Delay(100);

	UART_Printf("\nPCLK Divider Value: %d\n", uDividerValue);
	ClockInform();

	ret = I2C_Ch0_Master_Interrupt_EEPROM_AutoTest();
	
	Delay(100);
	ClockInform();
	
	return	ret;
}

/**
* {I2C0_CLKDIVTest}
*
* @brief			This function tests is Clock divider control.
* @conditions	
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_CLKDIVTest(void)
{
	u8	ret = TRUE;
	
	u32 uDividerValue = 2;
	u32	uDividerRange = 8;
	
	UART_Printf("[PCLKPSYS Divider Range(2~8)]\n");
	UART_Printf("[I2C0 Clock Divider Control Test using Serial EEPROM]\n");
	ClockInform();

	/* 
		// PCLK PSYS = HCLK PSYS / 2
		g_uPclkPsys = g_uHclkPsys / 2
		g_uPclkPsys = g_uHclkPsys / Divider Range(2~8)
		refer to 'SYSC_UpdateClkInform'		
	*/
	for( uDividerValue = 2; uDividerValue <= uDividerRange; uDividerValue++ )
	{
		ret = I2C0_CLKDIV(uDividerValue);
		if( ret == FALSE )
			break;
	}

	/* restore clock setting */
	SYSC_SetClkDiv(eCLKDIV_PCLKPSYS, 1);
	UART_InitDebugCh(3, 115200);
	
	return	ret;
}

/**
* {I2C0_CLKGateTest}
*
* @brief			This function tests is Clock gating control.
* @conditions	
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_CLKGateTest(void)
{
	u8	ret = TRUE;
	
	UART_Printf("\n[I2C0 Clock Gating Control Test using Serial EEPROM]\n");

	ret = I2C0_CLKDIV(2);
	if( ret == FALSE )
	{
		UART_Printf("\n[I2C0 CLOCK DIVIDER TEST FAIL]\n");
		return	ret;
	}
	
	UART_Printf("\n[I2C0 OFF (CLK_GATE_IP3)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_I2C0, FALSE);
	Delay(100);
	Pause();
	
	UART_Printf("\n[I2C0 ON (CLK_GATE_IP3)]\n");
	SYSC_SetClkGatebyIP(eCLKGATE_IP_I2C0, TRUE);
	Delay(100);
	
	ret = I2C0_CLKDIV(4);
	if( ret == FALSE )
	{
		UART_Printf("\n[I2C0 CLOCK DIVIDER TEST FAIL]\n");
		return	ret;
	}
	
	return	ret;
}

/**
* {I2C0_Slave_Read_Polling}
*
* @brief			This function tests polling mode of the SlaveRx by B'd to B'd.
* @conditions	 to test this function , Master Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Slave_Rx_Polling(u32 ch)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	s32 cCnt;
	static u8 cData[256] = {0,};
	static u8 cSlaveAddr = 0xD0;

	UART_Printf("\n[ I2C0 Slave Rx Test (Board to Board) ]\n");
	UART_Printf("\n[Have the other B'd set with Master Tx mode, Address: 0xD0]\n");
	
	// initialize
	gI2c_CH_Index = ch;
	I2C_GPIO_Open(ch);
	I2C_SFR_Mapping_Init(ch);
	I2C_Initialize_Controller(ch);
	
	UART_Printf("Connect I2C0 related signals (I2CSCL0, I2CSDA0) between I2C0 Master Tx (Polling) Board and I2C0 Slave RX Board.\n");

	// Slave Interrupt Rx
	iRetval = I2C_Slave_Polling_Read(ch, cSlaveAddr, cData, 101);
	if( iRetval != OK )
		ret = FALSE;
	
	cSlaveAddr = cData[0];
	
	UART_Printf("*** Received Address: 0x%02x\n\n",(u32)cSlaveAddr);
	
	for(cCnt = 1; cCnt < 101; cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data\n");

	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);
	
	return	ret;
}

/**
* {I2C0_Slave_Write_Polling}
*
* @brief			This function tests polling mode of the SlaveTx by B'd to B'd.
* @conditions	 to test this function , Master Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Slave_Tx_Polling(u32 ch)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	u32 sCnt;
	static u8 cData[100] = {0,};
	static u8 cSlaveAddr = 0xC0;

	UART_Printf("[ I2C0 Slave Tx Test (Board to Board) ]\n");
	UART_Printf("Connect I2C0 related signals (I2CSCL0, I2CSDA0) between I2C0 Master RX Board and I2C0 Slave TX Board.\n");
	UART_Printf("\n[Have the other B'd set with Master Rx mode, Address: 0xC0]\n");
	
	// initialize
	gI2c_CH_Index = ch;
	I2C_GPIO_Open(ch);
	I2C_SFR_Mapping_Init(ch);
	I2C_Initialize_Controller(ch);
	
	for(sCnt = 0; sCnt < 100; sCnt++)
		cData[sCnt] = (u8)sCnt;

	// Slave Tx Polling
	iRetval = I2C_Slave_Polling_Write(ch, cSlaveAddr, cData, 100);
	if( iRetval != OK )
		ret = FALSE;
	
	UART_Printf("\n*** Completed transmit data.\n");

	
	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);	
	
	return	ret;
}

/**
* {I2C0_Slave_Read_Interrupt}
*
* @brief			This function tests Interrupt mode of the SlaveRx by B'd to B'd.
* @conditions	 to test this function , Master Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Slave_Rx_Interrupt(u32 ch)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	s32 cCnt;
	static u8 cData[256] = {0,};
	static u8 cSlaveAddr = 0xD0;
	
    	UART_Printf("\n[I2C0 Slave Interrupt Rx Test (Board to Board) ]\n");
	UART_Printf("\n[Have the other B'd set with Master Tx mode, Address: 0xD0]\n");
	
	// initialize
	gI2c_CH_Index = ch;
	I2C_GPIO_Open(ch);
	I2C_Register_Interrupt(ch);
	I2C_SFR_Mapping_Init(ch);
	I2C_Initialize_Controller(ch);


    	UART_Printf("Connect I2C0 related signals (I2CSCL0, I2CSDA0) between I2C0 Master TX(Polling) Board and I2C0 Slave RX Board.\n");

	// Slave Interrupt Rx
	iRetval = I2C_Slave_Interrupt_Read(ch, cSlaveAddr, cData, 101);
	if( iRetval != OK )
		ret = FALSE;
	
	cSlaveAddr = cData[0];
	
	UART_Printf("Received Addrress: 0x%02x\n\n",(u32)cSlaveAddr);
	
	for(cCnt = 1; cCnt < 101; cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data\n");

	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);
	
	return	ret;
}

/**
* {I2C0_Slave_Write_Interrupt}
*
* @brief			This function tests Interrupt mode of the SlaveTx by B'd to B'd.
* @conditions	 to test this function , Master Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Slave_Tx_Interrupt(u32 ch)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	u32 sCnt;
	static u8 cData[100] = {0,};
	static u8 cSlaveAddr = 0xC0;

	UART_Printf("\n[ I2C0 Slave Tx (Interrupt Mode) Test (Board to Board) ]\n");
	UART_Printf("Connect I2C0 related signals (I2CSCL0, I2CSDA0) between I2C0 Master RX(Polling) Board and I2C0 Slave TX Board.\n");
	UART_Printf("\n[Have the other B'd set with Master Rx mode, Address: 0xC0]\n");
	
	// initialize
	gI2c_CH_Index = ch;
	I2C_GPIO_Open(ch);
	I2C_Register_Interrupt(ch);
	I2C_SFR_Mapping_Init(ch);
	I2C_Initialize_Controller(ch);

	
	for(sCnt = 0; sCnt < 100; sCnt++)
		cData[sCnt] = (u8)sCnt;

	// Slave Tx Interrupt
	iRetval = I2C_Slave_Interrupt_Write(ch, cSlaveAddr, cData, 100);
	if( iRetval != OK )
		ret = FALSE;
	
	UART_Printf("\n*** Completed transmit data\n");
	
	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);

	return	ret;
}

/**
* {I2C0_Master_Read_Polling}
*
* @brief			This function tests polling mode of the Master Rx by B'd to B'd.
* @conditions	 to test this function , Slave Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Master_Rx_Polling(u32 ch)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	s32 cCnt;
	static u8 cData[256] = {0,};
	static u8 cSlaveAddr = 0xC0;

	UART_Printf("[ I2C0 Master Rx Test (Board to Board) ]\n");
	UART_Printf("\n[Have the other B'd set with Slave Tx mode, Address: 0xC0]\n");
	
	// initialize
	gI2c_CH_Index = ch;
	I2C_GPIO_Open(ch);
	I2C_SFR_Mapping_Init(ch);
	I2C_Initialize_Controller(ch);

	
	UART_Printf("Connect I2C0 related signals (I2CSCL0, I2CSDA0) between I2C0 Master RX Board and I2C0 Slave TX Board.\n");
	UART_Printf("At first, SlaveTx should set up before MasterRx. Make sure that SlaveTx is set up.\n");	
	
	// Master Tx polling
	iRetval = I2C_Master_Polling_Read(ch, cSlaveAddr, cData, 101);
	if( iRetval != OK )
		ret = FALSE;

	cSlaveAddr = cData[0];
	
	UART_Printf("Received Address: 0x%02x\n\n",(u32)cSlaveAddr);
	
	for(cCnt = 1; cCnt < 101; cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data.\n");	

	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);

	return	ret;
}

/**
* {I2C0_Master_Write_Polling}
*
* @brief			This function tests polling mode of the Master Tx by B'd to B'd.
* @conditions	 to test this function, Slave Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Master_Tx_Polling(u32 ch)
{		
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	u32 sCnt;
	static u8 cData[100] = {0,};
	static u8 cSlaveAddr = 0xD0;

	UART_Printf("\n[I2C0 Master Tx Test (Board to Board) ]\n");
	UART_Printf("\n[Connect I2C related signals (I2CSCL0, I2CSDA0) between I2C0 Master TX Board and I2C0 Slave RX Board]\n");
	UART_Printf("\n[Have the other B'd set with Slave Rx mode, Address: 0xD0]\n");

	// initialize
	gI2c_CH_Index = ch;
	I2C_GPIO_Open(ch);
	I2C_SFR_Mapping_Init(ch);
	I2C_Initialize_Controller(ch);


	for(sCnt = 0; sCnt < 100; sCnt++)
		cData[sCnt] = (u8)sCnt;
	
	// Master Tx polling
	iRetval = I2C_Master_Polling_Write(ch, cSlaveAddr, cData, 100);
	if( iRetval != OK )
		ret = FALSE;

	UART_Printf("\n*** Completed transmit data.\n");


	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);

	return	ret;
}

/**
* {I2C_Master_Rx_Interrupt}
*
* @brief			This function tests Interrupt mode of the Master Rx by B'd to B'd.
* @conditions	 to test this function , Slave Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Master_Rx_Interrupt(u32 ch)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	s32 cCnt;
	static u8 cData[256] = {0,};
	static u8 cSlaveAddr = 0xC0;
	
    	UART_Printf("\n[I2C0 Master Interrupt Rx Test (Board to Board) ]\n");
	UART_Printf("\n[Have the other B'd set with Slave Tx mode, Address: 0xC0]\n");
	
	// initialize
	gI2c_CH_Index = ch;
	I2C_GPIO_Open(ch);
	I2C_Register_Interrupt(ch);
	I2C_SFR_Mapping_Init(ch);
	I2C_Initialize_Controller(ch);

	
	for(cCnt = 0; cCnt < 101; cCnt++)
		cData[cCnt] = 0;

    	UART_Printf("Connect I2C0 related signals (I2CSCL0, I2CSDA0) between I2C0 Master TX(Polling) Board and I2C0 Slave RX Board.\n");

	// Master Interrupt Rx
	iRetval = I2C_Master_Interrupt_Read(ch, cSlaveAddr, cData, 101);
	if( iRetval != OK )
		ret = FALSE;
	
	cSlaveAddr = cData[0];
	
	UART_Printf("Received Addrress: 0x%02x\n\n",(u32)cSlaveAddr);
	
	for(cCnt = 1; cCnt < 101; cCnt++)
		UART_Printf("Received Data: %03d\n",(u32)cData[cCnt]);

	UART_Printf("\n*** Completed receive data\n");

	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);
	
	return	ret;
}

/**
* {I2C_Master_Tx_Interrupt}
*
* @brief			This function tests Interrupt mode of the MasterTx by B'd to B'd.
* @conditions	 to test this function, Slave Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Master_Tx_Interrupt(u32 ch)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	u32 sCnt;
	static u8 cData[100] = {0,};
	static u8 cSlaveAddr = 0xD0;

	UART_Printf("\n[ I2C0 Master Tx (Interrupt Mode) Test (Board to Board) ]\n");
	UART_Printf("Connect I2C0 related signals (I2CSCL0, I2CSDA0) between I2C0 Master RX(Polling) Board and I2C0 Slave TX Board.\n");
	UART_Printf("\n[Have the other B'd set with Master Tx mode, Address: 0xD0]\n");
	
	// initialize
	gI2c_CH_Index = ch;
	I2C_GPIO_Open(ch);
	I2C_Register_Interrupt(ch);
	I2C_SFR_Mapping_Init(ch);
	I2C_Initialize_Controller(ch);

	
	for(sCnt = 0; sCnt < 100; sCnt++)
		cData[sCnt] = (u8)sCnt;

	// Master Tx Interrupt
	iRetval = I2C_Master_Interrupt_Write(ch, cSlaveAddr, cData, 100);
	if( iRetval != OK )
		ret = FALSE;
	
	UART_Printf("\n*** Completed transmit data\n");
	
	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);

	return	ret;
}

/**
* {I2C2_Slave_Rx_Polling}
*
* @brief			This function tests polling mode of the SlaveRx by B'd to B'd.
* @conditions	 to test this function , Master Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C2_Slave_Rx_Polling(void)
{
	u8	ret = TRUE;

	ret = I2C_Slave_Rx_Polling(I2C_CH2);
	return	ret;
}

/**
* {I2C2_Slave_Tx_Polling}
*
* @brief			This function tests polling mode of the SlaveTx by B'd to B'd.
* @conditions	 to test this function , Master Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C2_Slave_Tx_Polling(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Slave_Tx_Polling(I2C_CH2);
	return	ret;
}

/**
* {I2C2_Slave_Rx_Interrupt}
*
* @brief			This function tests Interrupt mode of the SlaveRx by B'd to B'd.
* @conditions	 to test this function , Master Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C2_Slave_Rx_Interrupt(void)
{
	u8	ret = TRUE;

	ret = I2C_Slave_Rx_Interrupt(I2C_CH2);
	return	ret;
}

/**
* {I2C2_Slave_Tx_Interrupt}
*
* @brief			This function tests Interrupt mode of the SlaveTx by B'd to B'd.
* @conditions	 to test this function , Master Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C2_Slave_Tx_Interrupt(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Slave_Tx_Interrupt(I2C_CH2);
	return	ret;
}

/**
* {I2C2_Master_Rx_Polling}
*
* @brief			This function tests polling mode of the Master Rx by B'd to B'd.
* @conditions	 to test this function , Slave Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C2_Master_Rx_Polling(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Rx_Polling(I2C_CH2);
	return	ret;	
}

/**
* {I2C2_Master_Tx_Polling}
*
* @brief			This function tests polling mode of the Master Tx by B'd to B'd.
* @conditions	 to test this function, Slave Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C2_Master_Tx_Polling(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Tx_Polling(I2C_CH2);
	return	ret;
}

/**
* {I2C2_Master_Rx_Interrupt}
*
* @brief			This function tests polling mode of the Master Rx by B'd to B'd.
* @conditions	 to test this function , Slave Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C2_Master_Rx_Interrupt(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Rx_Interrupt(I2C_CH2);
	return	ret;	
}

/**
* {I2C2_Master_Tx_Interrupt}
*
* @brief			This function tests polling mode of the Master Tx by B'd to B'd.
* @conditions	 to test this function, Slave Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C2_Master_Tx_Interrupt(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Tx_Interrupt(I2C_CH2);
	return	ret;
}

/**
* {I2C1_Slave_Rx_Polling}
*
* @brief			This function tests polling mode of the SlaveRx by B'd to B'd.
* @conditions	 to test this function , Master Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_Slave_Rx_Polling(void)
{
	u8	ret = TRUE;

	ret = I2C_Slave_Rx_Polling(I2C_CH_HDMI_DDC);
	return	ret;
}

/**
* {I2C1_Slave_Tx_Polling}
*
* @brief			This function tests polling mode of the SlaveTx by B'd to B'd.
* @conditions	 to test this function , Master Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_Slave_Tx_Polling(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Slave_Tx_Polling(I2C_CH_HDMI_DDC);
	return	ret;
}

/**
* {I2C1_Slave_Rx_Interrupt}
*
* @brief			This function tests Interrupt mode of the SlaveRx by B'd to B'd.
* @conditions	 to test this function , Master Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_Slave_Rx_Interrupt(void)
{
	u8	ret = TRUE;

	ret = I2C_Slave_Rx_Interrupt(I2C_CH_HDMI_DDC);
	return	ret;
}

/**
* {I2C1_Slave_Tx_Interrupt}
*
* @brief			This function tests Interrupt mode of the SlaveTx by B'd to B'd.
* @conditions	 to test this function , Master Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_Slave_Tx_Interrupt(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Slave_Tx_Interrupt(I2C_CH_HDMI_DDC);
	return	ret;
}

/**
* {I2C1_Master_Rx_Polling}
*
* @brief			This function tests polling mode of the Master Rx by B'd to B'd.
* @conditions	 to test this function , Slave Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_Master_Rx_Polling(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Rx_Polling(I2C_CH_HDMI_DDC);
	return	ret;	
}

/**
* {I2C1_Master_Tx_Polling}
*
* @brief			This function tests polling mode of the Master Tx by B'd to B'd.
* @conditions	 to test this function, Slave Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_Master_Tx_Polling(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Tx_Polling(I2C_CH_HDMI_DDC);
	return	ret;
}

/**
* {I2C1_Master_Rx_Interrupt}
*
* @brief			This function tests polling mode of the Master Rx by B'd to B'd.
* @conditions	 to test this function , Slave Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_Master_Rx_Interrupt(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Rx_Interrupt(I2C_CH_HDMI_DDC);
	return	ret;	
}

/**
* {I2C1_Master_Tx_Interrupt}
*
* @brief			This function tests polling mode of the Master Tx by B'd to B'd.
* @conditions	 to test this function, Slave Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C1_Master_Tx_Interrupt(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Tx_Interrupt(I2C_CH_HDMI_DDC);
	return	ret;
}

/**
* {I2C0_Slave_Rx_Polling}
*
* @brief			This function tests polling mode of the SlaveRx by B'd to B'd.
* @conditions	 to test this function , Master Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_Slave_Rx_Polling(void)
{
	u8	ret = TRUE;

	ret = I2C_Slave_Rx_Polling(I2C_CH0);
	return	ret;
}

/**
* {I2C0_Slave_Tx_Polling}
*
* @brief			This function tests polling mode of the SlaveTx by B'd to B'd.
* @conditions	 to test this function , Master Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_Slave_Tx_Polling(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Slave_Tx_Polling(I2C_CH0);
	return	ret;
}

/**
* {I2C0_Slave_Rx_Interrupt}
*
* @brief			This function tests Interrupt mode of the SlaveRx by B'd to B'd.
* @conditions	 to test this function , Master Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_Slave_Rx_Interrupt(void)
{
	u8	ret = TRUE;

	ret = I2C_Slave_Rx_Interrupt(I2C_CH0);
	return	ret;
}

/**
* {I2C0_Slave_Tx_Interrupt}
*
* @brief			This function tests Interrupt mode of the SlaveTx by B'd to B'd.
* @conditions	 to test this function , Master Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_Slave_Tx_Interrupt(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Slave_Tx_Interrupt(I2C_CH0);
	return	ret;
}

/**
* {I2C0_Master_Rx_Polling}
*
* @brief			This function tests polling mode of the Master Rx by B'd to B'd.
* @conditions	 to test this function , Slave Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_Master_Rx_Polling(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Rx_Polling(I2C_CH0);
	return	ret;	
}

/**
* {I2C0_Master_Tx_Polling}
*
* @brief			This function tests polling mode of the Master Tx by B'd to B'd.
* @conditions	 to test this function, Slave Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_Master_Tx_Polling(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Tx_Polling(I2C_CH0);
	return	ret;
}

/**
* {I2C0_Master_Rx_Interrupt}
*
* @brief			This function tests polling mode of the Master Rx by B'd to B'd.
* @conditions	 to test this function , Slave Tx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_Master_Rx_Interrupt(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Rx_Interrupt(I2C_CH0);
	return	ret;	
}

/**
* {I2C0_Master_Tx_Interrupt}
*
* @brief			This function tests polling mode of the Master Tx by B'd to B'd.
* @conditions	 to test this function, Slave Rx should be ready.
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C0_Master_Tx_Interrupt(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Master_Tx_Interrupt(I2C_CH0);
	return	ret;
}

/**
* {I2C_Ch0_Master_Interrupt_EEPROM_AutoTest}
*
* @brief			Ch0 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch0_Master_Interrupt_EEPROM_AutoTest(void)
{
	s32 iRetval = OK;
	
	u8	ret = TRUE;
	
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E0;	// KS24C080C: 1KB (8Kbit) = 0x3e8

	s32	iclkmode = 0;
	s32	iclkprescale = 0;
	s32	iclkInitValue = 0;

	s32	iRetValue = OK;
	
	for( iclkmode = 1; iclkmode >= 0; iclkmode-- )
	{
		/* If I2CCON[6] = 0, I2CCON[3:0] = 0x0 or 0x1 is not available */
		if( iclkmode == 0 )
			iclkInitValue = 2;
		else
			iclkInitValue = 0;
		
		for( iclkprescale = 15; iclkprescale >= iclkInitValue; iclkprescale-- )
		{
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;	
			
			//Inital Buffer data
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++=0;	*pComp++=0;
			}

			//Inital Buffer index
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;

			IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(KS24C080) Verify Test with Interrupt mode");
			IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...");


			gI2c_CH_Index = I2C_CH0;
			I2C_GPIO_Open(gI2c_CH_Index);
			I2C_Register_Interrupt(gI2c_CH_Index);
			I2C_SFR_Mapping_Init(gI2c_CH_Index);
			I2C_Initialize_Controller(gI2c_CH_Index);
			
			iRetValue = I2C_ClockPrescaler(gI2c_CH_Index, iclkmode, iclkprescale);
			if( iRetValue == ERROR )
			{
				goto EXCEED_MAX_TX_CLOCK;
			}
						
			IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++ = uCnt;
				IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);

				#ifdef v210_I2C_FPGA
				iRetval = I2C_Ch0_Master_Interrupt_Write_EEPROM(gI2c_CH_Index, KS24C080C_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
				#else
				iRetval = I2C_Ch2_Master_Interrupt_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
				#endif
	
				if( iRetval != OK )
					break;

				Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
				UART_Printf(".");
			}

			IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				#ifdef v210_I2C_FPGA
				iRetval = I2C_Ch0_Master_Interrupt_Read_EEPROM(gI2c_CH_Index, KS24C080C_SLAVEADDRESS+1,  uCnt, &cData[0]);
				#else
				iRetval = I2C_Ch2_Master_Interrupt_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
				#endif
				
				if( iRetval != OK )
					break;

				*pComp++ = cData[0];
				
				IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
			}

			IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;
			for (uCnt = uMin; uCnt < uMax; uCnt++)
			{
				if(*pCmp != *pComp)
				{
					uCntErr++;
					IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
				}			
				
				pCmp++;
				pComp++;
			}

EXCEED_MAX_TX_CLOCK:

			I2C_DeInitialize_Controller(gI2c_CH_Index);
			I2C_UnRegister_Interrupt(gI2c_CH_Index);
			I2C_GPIO_Close(gI2c_CH_Index);

				
			if (uCntErr == 0)
			{
				IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
				ret = TRUE;
			}
			else
			{
				IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
				ret = FALSE;
			}
			
			if( ret == FALSE )
				return	ret;
		}
	}

	return	ret;
}

/**
* {I2C_Ch0_Master_Polling_EEPROM_AutoTest}
*
* @brief			Ch0 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch0_Master_Polling_EEPROM_AutoTest(void)
{
	s32 iRetval = OK;
	
	u8	ret = TRUE;
		
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E0;	// KS24C080C: 1KB (8Kbit) = 0x3e8

	s32	iclkmode,	iclkprescale;
	s32	iclkInitValue;

	s32	iRetValue = OK;
	
	for (iclkmode = 1; iclkmode >= 0; iclkmode--)
	{
		/* If I2CCON[6] = 0, I2CCON[3:0] = 0x0 or 0x1 is not available */
		if( iclkmode == 0 )
			iclkInitValue = 2;
		else
			iclkInitValue = 0;
		
		for (iclkprescale = 15; iclkprescale >= iclkInitValue; iclkprescale--)
		{
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;	
			
			//Inital Buffer data
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++=0;	*pComp++=0;
			}

			//Inital Buffer index
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;

			IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(KS24C080) Verify Test with Polling mode");
			IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");
				

			gI2c_CH_Index = I2C_CH0;
			I2C_GPIO_Open(gI2c_CH_Index);
			I2C_SFR_Mapping_Init(gI2c_CH_Index);
			I2C_Initialize_Controller(gI2c_CH_Index);

			iRetValue = I2C_ClockPrescaler(gI2c_CH_Index, iclkmode, iclkprescale);
			if( iRetValue == ERROR )
			{
				goto EXCEED_MAX_TX_CLOCK;
			}
						
			IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++ = uCnt;
				IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
				
				#ifdef v210_I2C_FPGA
				iRetval = I2C_Ch0_Master_Polling_Write_EEPROM(gI2c_CH_Index, KS24C080C_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
				#else
				iRetval = I2C_Ch2_Master_Polling_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
				#endif
				
				if( iRetval != OK )
					break;
				
				Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
				UART_Printf(".");
			}

			IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				#ifdef v210_I2C_FPGA
				iRetval = I2C_Ch0_Master_Polling_Read_EEPROM(gI2c_CH_Index, KS24C080C_SLAVEADDRESS+1,  uCnt, &cData[0]);
				#else
				iRetval = I2C_Ch2_Master_Polling_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
				#endif

				if( iRetval != OK )
					break;

				*pComp++ = cData[0];
				
				IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
			}

			IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;
			for (uCnt = uMin; uCnt < uMax; uCnt++)
			{
				if(*pCmp != *pComp)
				{
					uCntErr++;
					IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
				}			
				
				pCmp++;
				pComp++;
			}

EXCEED_MAX_TX_CLOCK:

			I2C_DeInitialize_Controller(gI2c_CH_Index);
			I2C_UnRegister_Interrupt(gI2c_CH_Index);
			I2C_GPIO_Close(gI2c_CH_Index);

			
			if (uCntErr == 0)
			{
				IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!");	
				ret = TRUE;
			}
			else
			{
				IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
				ret = FALSE;
			}
			
			if( ret == FALSE )
				return	ret;
		}
	}

	return	ret;
}



/**
* {I2C_Ch0_Master_Interrupt_EEPROM_Test}
*
* @brief			Ch0 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch0_Master_Interrupt_EEPROM_Test(void)
{
	s32 iRetval = OK;

	u8	ret = TRUE;
	
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E0;	// KS24C080C: 1KB (8Kbit) = 0x3e8

	u32 iSel;
	
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;	
	
	//Inital Buffer data
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++=0;	*pComp++=0;				
	}

	//Inital Buffer index
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;

	IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(KS24C080) Verify Test with Interrupt mode");
	IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");


	gI2c_CH_Index = I2C_CH0;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_Register_Interrupt(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);
	
#if 0
	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");	
	if ( (iSel = UART_GetIntNum() ) > 500)
		iSel = 500; //MAX 500Khz
		
	I2C_TxClockPrescaler(gI2c_CH_Index, iSel * 1000);
#endif

//----------------------
//Data Writing to EEPROM
//----------------------	
	IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++ = uCnt;
		IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
		
		#ifdef v210_I2C_FPGA
		iRetval = I2C_Ch0_Master_Interrupt_Write_EEPROM(gI2c_CH_Index, KS24C080C_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
		#else
		iRetval = I2C_Ch2_Master_Interrupt_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
		#endif
		
		if( iRetval != OK )
			break;
		
		Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
		UART_Printf(".");
	}

//----------------------
//Data Reading to EEPROM
//----------------------
	IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		#ifdef v210_I2C_FPGA
		iRetval = I2C_Ch0_Master_Interrupt_Read_EEPROM(gI2c_CH_Index, KS24C080C_SLAVEADDRESS+1,  uCnt, &cData[0]);
		#else
		iRetval = I2C_Ch2_Master_Interrupt_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
		#endif

		if( iRetval != OK )
			break;
		
		*pComp++ = cData[0];
		
		IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
	}

	IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;
	for (uCnt = uMin; uCnt < uMax; uCnt++)
	{
		if(*pCmp != *pComp)
		{
			uCntErr++;
			IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
		}			
		
		pCmp++;
		pComp++;
	}


	I2C_DeInitialize_Controller(gI2c_CH_Index);
	I2C_UnRegister_Interrupt(gI2c_CH_Index);
	I2C_GPIO_Close(gI2c_CH_Index);

	
	if (uCntErr == 0)
	{
		IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
		ret = TRUE;
	}
	else
	{
		IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
		ret = FALSE;
	}

	return	ret;
}

/**
* {I2C_Ch0_Master_Polling_EEPROM_AutoTest}
*
* @brief			Ch0 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch0_Master_Polling_EEPROM_Test(void)
{
	s32 iRetval = OK;
	
	u8	ret = TRUE;
		
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E0;	// KS24C080C: 1KB (8Kbit) = 0x3e8

	u32 iSel;
	
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;	
	
	//Inital Buffer data
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++=0;	*pComp++=0;
	}

	//Inital Buffer index
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;

	IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(KS24C080) Verify Test with Polling mode");
	IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");

	
	gI2c_CH_Index = I2C_CH0;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);
	
#if 0
	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	if ( (iSel = UART_GetIntNum() ) > 500)
		iSel = 500; //MAX 500Khz
		
	I2C_TxClockPrescaler(gI2c_CH_Index, iSel * 1000);
	I2C_SDAClockDelay(gI2c_CH_Index, 0);
#endif

	IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++ = uCnt;
		IIC_DBG_PRINT(DBG_Step,"\n Writing... (0x%x) <= 0x%x", uCnt, uCnt);
		
		#ifdef v210_I2C_FPGA
		iRetval = I2C_Ch0_Master_Polling_Write_EEPROM(gI2c_CH_Index, KS24C080C_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
		#else
		iRetval = I2C_Ch2_Master_Polling_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.		
		#endif
		
		if( iRetval != OK )
			break;
		
		Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
		UART_Printf(".");
	}

	IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		#ifdef v210_I2C_FPGA
		iRetval = I2C_Ch0_Master_Polling_Read_EEPROM(gI2c_CH_Index, KS24C080C_SLAVEADDRESS+1,  uCnt, &cData[0]);
		#else
		iRetval = I2C_Ch2_Master_Polling_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
		#endif

		if( iRetval != OK )
			break;
		
		*pComp++ = cData[0];
		
		IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => 0x%x 0x%x 0x%x", uCnt, cData[0], cData[1], cData[2]);				
	}

	IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;
	for (uCnt = uMin; uCnt < uMax; uCnt++)
	{
		if(*pCmp != *pComp)
		{
			uCntErr++;
			IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
		}			
		
		pCmp++;
		pComp++;
	}


	I2C_DeInitialize_Controller(gI2c_CH_Index);
	I2C_UnRegister_Interrupt(gI2c_CH_Index);
	I2C_GPIO_Close(gI2c_CH_Index);
	
	
	if (uCntErr == 0)
	{
		IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
		ret = TRUE;
	}
	else
	{
		IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
		ret = FALSE;
	}

	return	ret;
}

#ifdef v210_I2C_FPGA
/**
* {I2C_Ch2_Master_Interrupt_EEPROM_AutoTest}
*
* @brief			Ch0 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch2_Master_Interrupt_EEPROM_AutoTest(void)
{
	s32 iRetval = OK;

	u8	ret = TRUE;
	
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E00;	// S524AD0XD1: 16KB (128Kbit) = 0x3e80	

	s32	iclkmode,	iclkprescale;
	s32	iclkInitValue;

	s32	iRetValue = OK;
	
	for( iclkmode = 1; iclkmode >= 0; iclkmode-- )
	{
		/* If I2CCON[6] = 0, I2CCON[3:0] = 0x0 or 0x1 is not available */
		if( iclkmode == 0 )
			iclkInitValue = 2;
		else
			iclkInitValue = 0;
		
		for( iclkprescale = 15; iclkprescale >= iclkInitValue; iclkprescale-- )
		{
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;	
			
			//Inital Buffer data
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++=0;	*pComp++=0;
			}

			//Inital Buffer index
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;

			IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(S524AD0XD1) Verify Test with Interrupt mode");
			IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");
				

			gI2c_CH_Index = I2C_CH2;
			I2C_GPIO_Open(gI2c_CH_Index);
			I2C_Register_Interrupt(gI2c_CH_Index);			
			I2C_SFR_Mapping_Init(gI2c_CH_Index);
			I2C_Initialize_Controller(gI2c_CH_Index);
			
			iRetValue = I2C_ClockPrescaler(gI2c_CH_Index, iclkmode, iclkprescale);
			if( iRetValue == ERROR )
			{
				goto EXCEED_MAX_TX_CLOCK;
			}
						
			IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++ = uCnt;
				IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
				iRetval = I2C_Ch2_Master_Interrupt_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
				if( iRetval != OK)
					break;
				
				Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
			}

			IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				iRetval = I2C_Ch2_Master_Interrupt_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
				if( iRetval != OK)
					break;
				*pComp++ = cData[0];
				
				IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
			}

			IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;
			for (uCnt = uMin; uCnt < uMax; uCnt++)
			{
				if(*pCmp != *pComp)
				{
					uCntErr++;
					IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
				}			
				
				pCmp++;
				pComp++;
			}

EXCEED_MAX_TX_CLOCK:

			I2C_DeInitialize_Controller(gI2c_CH_Index);
			I2C_UnRegister_Interrupt(gI2c_CH_Index);
			I2C_GPIO_Close(gI2c_CH_Index);


			if (uCntErr == 0)
			{
				IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
				ret = TRUE;
			}
			else
			{
				IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
				ret = FALSE;
			}
			
			if( ret == FALSE )
				return	ret;
		}
	}

	return	ret;
}

/**
* {I2C_Ch2_Master_Polling_EEPROM_AutoTest}
*
* @brief			Ch0 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch2_Master_Polling_EEPROM_AutoTest(void)
{
	s32 iRetval = OK;
	
	u8	ret = TRUE;
		
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E00;	// S524AD0XD1: 16KB (128Kbit) = 0x3e80

	s32	iclkmode,	iclkprescale;
	s32	iclkInitValue;

	s32	iRetValue = OK;
	
	for (iclkmode = 1; iclkmode >= 0; iclkmode--)
	{
		/* If I2CCON[6] = 0, I2CCON[3:0] = 0x0 or 0x1 is not available */
		if( iclkmode == 0 )
			iclkInitValue = 2;
		else
			iclkInitValue = 0;
		
		for (iclkprescale = 15; iclkprescale >= iclkInitValue; iclkprescale--)
		{
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;	
			
			//Inital Buffer data
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++=0;	*pComp++=0;
			}

			//Inital Buffer index
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;

			IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(S524AD0XD1) Verify Test with Polling mode");
			IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");
				

			gI2c_CH_Index = I2C_CH2;
			I2C_GPIO_Open(gI2c_CH_Index);
			I2C_SFR_Mapping_Init(gI2c_CH_Index);
			I2C_Initialize_Controller(gI2c_CH_Index);

			iRetValue = I2C_ClockPrescaler(gI2c_CH_Index, iclkmode, iclkprescale);
			if( iRetValue == ERROR )
			{
				goto EXCEED_MAX_TX_CLOCK;
			}
						
			IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++ = uCnt;
				IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
				iRetval = I2C_Ch2_Master_Polling_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
				if( iRetval != OK )
					break;
				
				Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
			}

			IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				iRetval = I2C_Ch2_Master_Polling_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
				if( iRetval != OK )
					break;

				*pComp++ = cData[0];
				
				IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
			}

			IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;
			for (uCnt = uMin; uCnt < uMax; uCnt++)
			{
				if(*pCmp != *pComp)
				{
					uCntErr++;
					IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
				}			
				
				pCmp++;
				pComp++;
			}

EXCEED_MAX_TX_CLOCK:

			I2C_DeInitialize_Controller(gI2c_CH_Index);
			I2C_UnRegister_Interrupt(gI2c_CH_Index);
			I2C_GPIO_Close(gI2c_CH_Index);

			
			if (uCntErr == 0)
			{
				IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
				ret = TRUE;
			}
			else
			{
				IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
				ret = FALSE;
			}

			if( ret == FALSE )
				return	ret;
		}
	}

	return	ret;
}


/**
* {I2C_Ch2_Master_Interrupt_EEPROM_Test}
*
* @brief			Ch2 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch2_Master_Interrupt_EEPROM_Test(void)
{
	s32 iRetval = OK;
	
	u8	ret = TRUE;
	
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E00;	// S524AD0XD1: 16KB (128Kbit) = 0x3e80	

	u32 iSel;

	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;	
	
	//Inital Buffer data
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++=0;	*pComp++=0;				
	}

	//Inital Buffer index
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;

	IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(S524AD0XD1) Verify Test with Interrupt mode");
	IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");


	gI2c_CH_Index = I2C_CH2;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_Register_Interrupt(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);
	
#if 0
	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	if ( (iSel = UART_GetIntNum() ) > 500)
		iSel = 500; //MAX 500Khz
		
	I2C_TxClockPrescaler(gI2c_CH_Index, iSel * 1000);
#endif

	
	IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++ = uCnt;
		IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
		iRetval = I2C_Ch2_Master_Interrupt_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
		if( iRetval != OK )
			break;
		
		Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
	}

	IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		iRetval = I2C_Ch2_Master_Interrupt_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1, (u8) uCnt, &cData[0]);
		if( iRetval != OK ) 
			break;
				
		*pComp++ = cData[0];
		
		IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
	}

	IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;
	for (uCnt = uMin; uCnt < uMax; uCnt++)
	{
		if(*pCmp != *pComp)
		{
			uCntErr++;
			IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
		}			
		
		pCmp++;
		pComp++;
	}


	I2C_DeInitialize_Controller(gI2c_CH_Index);
	I2C_UnRegister_Interrupt(gI2c_CH_Index);
	I2C_GPIO_Close(gI2c_CH_Index);

		
	if( uCntErr == 0 )
	{
		IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
		ret = TRUE;
	}
	else
	{
		IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
		ret = FALSE;
	}

	return	ret;
}

/**
* {I2C_Ch2_Master_Polling_EEPROM_Test}
*
* @brief			Ch2 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch2_Master_Polling_EEPROM_Test(void)
{
	s32 iRetval = OK;
	
	u8	ret = TRUE;
		
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E00;	// S524AD0XD1: 16KB (128Kbit) = 0x3e80

	u32 iSel;

	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;	
	
	//Inital Buffer data
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++=0;	*pComp++=0;
	}

	//Inital Buffer index
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;

	IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(S524AD0XD1) Verify Test with Polling mode");
	IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");
	

	gI2c_CH_Index = I2C_CH2;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);

#if 0
	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	if ( (iSel = UART_GetIntNum() ) > 500)
		iSel = 500; //MAX 500Khz
		
	I2C_TxClockPrescaler(gI2c_CH_Index, iSel * 1000);
#endif

	
	IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++ = uCnt;
		IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
		iRetval = I2C_Ch2_Master_Polling_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
		if( iRetval != OK )
			break;
		
		Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
	}

	IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		iRetval = I2C_Ch2_Master_Polling_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
		if( iRetval != OK )
			break;
		
		*pComp++ = cData[0];
		
		IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
	}

	IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;
	for (uCnt = uMin; uCnt < uMax; uCnt++)
	{
		if(*pCmp != *pComp)
		{
			uCntErr++;
			IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
		}			
		
		pCmp++;
		pComp++;
	}


	I2C_DeInitialize_Controller(gI2c_CH_Index);
	I2C_UnRegister_Interrupt(gI2c_CH_Index);
	I2C_GPIO_Close(gI2c_CH_Index);
	

	if (uCntErr == 0)
	{
		IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
		ret = TRUE;
	}
	else
	{
		IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
		ret = FALSE;
	}

	return	ret;
}
#else
/**
* {I2C_Ch1_Master_Interrupt_EEPROM_AutoTest}
*
* @brief			Ch1 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch1_Master_Interrupt_EEPROM_AutoTest(void)
{
	s32 iRetval = OK;

	u8	ret = TRUE;
	
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E00;	// S524AD0XD1: 16KB (128Kbit) = 0x3e80	

	s32	iclkmode,	iclkprescale;
	s32	iclkInitValue;

	s32	iRetValue = OK;


	UART_Printf("\n[CH0 CH1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	
	for( iclkmode = 1; iclkmode >= 0; iclkmode-- )
	{
		/* If I2CCON[6] = 0, I2CCON[3:0] = 0x0 or 0x1 is not available */
		if( iclkmode == 0 )
			iclkInitValue = 2;
		else
			iclkInitValue = 0;
		
		for( iclkprescale = 15; iclkprescale >= iclkInitValue; iclkprescale-- )
		{
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;	
			
			//Inital Buffer data
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++=0;	*pComp++=0;
			}

			//Inital Buffer index
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;

			IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(S524AD0XD1) Verify Test with Interrupt mode");
			IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");
				

			gI2c_CH_Index = I2C_CH_HDMI_DDC;
			I2C_GPIO_Open(gI2c_CH_Index);
			I2C_Register_Interrupt(gI2c_CH_Index);			
			I2C_SFR_Mapping_Init(gI2c_CH_Index);
			I2C_Initialize_Controller(gI2c_CH_Index);
			
			iRetValue = I2C_ClockPrescaler(gI2c_CH_Index, iclkmode, iclkprescale);
			if( iRetValue == ERROR )
			{
				goto EXCEED_MAX_TX_CLOCK;
			}
						
			IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++ = uCnt;
				IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
				iRetval = I2C_Ch2_Master_Interrupt_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
				if( iRetval != OK)
					break;
				
				Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
			}

			IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				iRetval = I2C_Ch2_Master_Interrupt_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
				if( iRetval != OK)
					break;
				*pComp++ = cData[0];
				
				IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
			}

			IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;
			for (uCnt = uMin; uCnt < uMax; uCnt++)
			{
				if(*pCmp != *pComp)
				{
					uCntErr++;
					IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
				}			
				
				pCmp++;
				pComp++;
			}

EXCEED_MAX_TX_CLOCK:

			I2C_DeInitialize_Controller(gI2c_CH_Index);
			I2C_UnRegister_Interrupt(gI2c_CH_Index);
			I2C_GPIO_Close(gI2c_CH_Index);


			if (uCntErr == 0)
			{
				IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
				ret = TRUE;
			}
			else
			{
				IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
				ret = FALSE;
			}
			
			if( ret == FALSE )
				return	ret;
		}
	}

	return	ret;
}

/**
* {I2C_Ch1_Master_Polling_EEPROM_AutoTest}
*
* @brief			Ch1 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch1_Master_Polling_EEPROM_AutoTest(void)
{
	s32 iRetval = OK;
	
	u8	ret = TRUE;
		
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E00;	// S524AD0XD1: 16KB (128Kbit) = 0x3e80

	s32	iclkmode,	iclkprescale;
	s32	iclkInitValue;

	s32	iRetValue = OK;


	UART_Printf("\n[CH0 CH1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	
	for (iclkmode = 1; iclkmode >= 0; iclkmode--)
	{
		/* If I2CCON[6] = 0, I2CCON[3:0] = 0x0 or 0x1 is not available */
		if( iclkmode == 0 )
			iclkInitValue = 2;
		else
			iclkInitValue = 0;
		
		for (iclkprescale = 15; iclkprescale >= iclkInitValue; iclkprescale--)
		{
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;	
			
			//Inital Buffer data
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++=0;	*pComp++=0;
			}

			//Inital Buffer index
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;

			IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(S524AD0XD1) Verify Test with Polling mode");
			IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");
				

			gI2c_CH_Index = I2C_CH_HDMI_DDC;
			I2C_GPIO_Open(gI2c_CH_Index);
			I2C_SFR_Mapping_Init(gI2c_CH_Index);
			I2C_Initialize_Controller(gI2c_CH_Index);

			iRetValue = I2C_ClockPrescaler(gI2c_CH_Index, iclkmode, iclkprescale);
			if( iRetValue == ERROR )
			{
				goto EXCEED_MAX_TX_CLOCK;
			}
						
			IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				*pCmp++ = uCnt;
				IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
				iRetval = I2C_Ch2_Master_Polling_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
				if( iRetval != OK )
					break;
				
				Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
			}

			IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
			for(uCnt = uMin; uCnt < uMax; uCnt++)
			{
				iRetval = I2C_Ch2_Master_Polling_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
				if( iRetval != OK )
					break;

				*pComp++ = cData[0];
				
				IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
			}

			IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
			pCmp = (u8 *)COMPAdd1;
			pComp = (u8 *)COMPAdd2;
			for (uCnt = uMin; uCnt < uMax; uCnt++)
			{
				if(*pCmp != *pComp)
				{
					uCntErr++;
					IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
				}			
				
				pCmp++;
				pComp++;
			}

EXCEED_MAX_TX_CLOCK:

			I2C_DeInitialize_Controller(gI2c_CH_Index);
			I2C_UnRegister_Interrupt(gI2c_CH_Index);
			I2C_GPIO_Close(gI2c_CH_Index);

			
			if (uCntErr == 0)
			{
				IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
				ret = TRUE;
			}
			else
			{
				IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
				ret = FALSE;
			}

			if( ret == FALSE )
				return	ret;
		}
	}

	return	ret;
}

/**
* {I2C_Ch1_Master_Interrupt_EEPROM_Test}
*
* @brief			Ch1 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch1_Master_Interrupt_EEPROM_Test(void)
{
	s32 iRetval = OK;
	
	u8	ret = TRUE;
	
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E00;	// S524AD0XD1: 16KB (128Kbit) = 0x3e80	

	u32 iSel;


	UART_Printf("\n[CH0 CH1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());


	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;	
	
	//Inital Buffer data
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++=0;	*pComp++=0;				
	}

	//Inital Buffer index
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;

	IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(S524AD0XD1) Verify Test with Interrupt mode");
	IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");


	gI2c_CH_Index = I2C_CH_HDMI_DDC;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_Register_Interrupt(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);
	
#if 0
	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	if ( (iSel = UART_GetIntNum() ) > 500)
		iSel = 500; //MAX 500Khz
		
	I2C_TxClockPrescaler(gI2c_CH_Index, iSel * 1000);
#endif

	
	IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++ = uCnt;
		IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
		iRetval = I2C_Ch2_Master_Interrupt_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
		if( iRetval != OK )
			break;
		
		Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
	}

	IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		iRetval = I2C_Ch2_Master_Interrupt_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1, (u8) uCnt, &cData[0]);
		if( iRetval != OK ) 
			break;
				
		*pComp++ = cData[0];
		
		IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
	}

	IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;
	for (uCnt = uMin; uCnt < uMax; uCnt++)
	{
		if(*pCmp != *pComp)
		{
			uCntErr++;
			IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
		}			
		
		pCmp++;
		pComp++;
	}


	I2C_DeInitialize_Controller(gI2c_CH_Index);
	I2C_UnRegister_Interrupt(gI2c_CH_Index);
	I2C_GPIO_Close(gI2c_CH_Index);

		
	if( uCntErr == 0 )
	{
		IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
		ret = TRUE;
	}
	else
	{
		IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
		ret = FALSE;
	}

	return	ret;
}

/**
* {I2C_Ch1_Master_Polling_EEPROM_Test}
*
* @brief			Ch1 Master Tx/Rx Interrupt Test in EEPROM
* 
* @param[in]	(void)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_Ch1_Master_Polling_EEPROM_Test(void)
{
	s32 iRetval = OK;
	
	u8	ret = TRUE;
		
	u32 uCnt, uCntErr=0;
	u8 cData[3] = {0,};
	u8 *pCmp;
	u8 *pComp;
	
	u32 uMin = 0x0;
	u32 uMax = 0x10;	// 16 byte for Sorting 2007.07.17	
//	u32 uMax = 0x100;	// 256 byte  ?? (whole size is  1KB)
//	u32 uMax = 0x3E00;	// S524AD0XD1: 16KB (128Kbit) = 0x3e80

	u32 iSel;

#if 0
	UART_Printf("\n[CH0 CH1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());
#endif/* 0 */

	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;	
	
	//Inital Buffer data
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++=0;	*pComp++=0;
	}

	//Inital Buffer index
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;

	IIC_DBG_PRINT(DBG_Basic,"\nEEPROM(S524AD0XD1) Verify Test with Polling mode");
	IIC_DBG_PRINT(DBG_Basic,"\n-256byte Write/Read/Compare test...\n");
	

	gI2c_CH_Index = I2C_CH_HDMI_DDC;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);

#if 0
	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	if ( (iSel = UART_GetIntNum() ) > 500)
		iSel = 500; //MAX 500Khz
		
	I2C_TxClockPrescaler(gI2c_CH_Index, iSel * 1000);
#endif

	
	IIC_DBG_PRINT(DBG_Basic,"\n1. EEPROM Writing.....");	
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		*pCmp++ = uCnt;
		IIC_DBG_PRINT(DBG_Step,"\n Writing... (%x) <= %x",uCnt, uCnt);
		iRetval = I2C_Ch2_Master_Polling_Write_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS, (u8) uCnt, (s32) uCnt);//assume address is same data.
		if( iRetval != OK )
			break;
		
		Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
	}

	IIC_DBG_PRINT(DBG_Basic,"\n2. EEPROM Reading....");
	for(uCnt = uMin; uCnt < uMax; uCnt++)
	{
		iRetval = I2C_Ch2_Master_Polling_Read_EEPROM(gI2c_CH_Index, S524AD0XD1_SLAVEADDRESS+1,  uCnt, &cData[0]);
		if( iRetval != OK )
			break;
		
		*pComp++ = cData[0];
		
		IIC_DBG_PRINT(DBG_Step,"\nReading... (%x) => %x %x %x  ",uCnt,cData[0], cData[1], cData[2]);				
	}

	IIC_DBG_PRINT(DBG_Basic,"\n3. EEPROM Data comparing.....");
	pCmp = (u8 *)COMPAdd1;
	pComp = (u8 *)COMPAdd2;
	for (uCnt = uMin; uCnt < uMax; uCnt++)
	{
		if(*pCmp != *pComp)
		{
			uCntErr++;
			IIC_DBG_PRINT(DBG_Basic,"\n  Error %03d [Address= %6d]: Write= 0x%02x  ,  Read= 0x%02x", uCntErr,uCnt,*pComp,*pCmp);			
		}			
		
		pCmp++;
		pComp++;
	}


	I2C_DeInitialize_Controller(gI2c_CH_Index);
	I2C_UnRegister_Interrupt(gI2c_CH_Index);
	I2C_GPIO_Close(gI2c_CH_Index);
	

	if (uCntErr == 0)
	{
		IIC_DBG_PRINT(DBG_Basic,"\nڡڡ Passed...GOOD !!!\n");	
		ret = TRUE;
	}
	else
	{
		IIC_DBG_PRINT(DBG_Basic,"\n ڡڡ Fail: %d Data !!!\n", uCntErr);
		ret = FALSE;
	}

	return	ret;
}
#endif/* #ifdef v210_I2C_FPGA */

/**
* {I2C_Open}
*
* @brief			This function Set up I2CCON with user's input frequency which determines uClkValue
* 				(without ISR set up,for polling mode)
*				GPIO ec..	
* @param[in]	u32 ch, u32 iSel
* @return		void
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Open( u32 ch, u32 iSel )
{
	s32	iRetValue = OK;
	
	I2C_GPIO_Open(ch);
	I2C_SFR_Mapping_Init(ch);
	I2C_Initialize_Controller(ch);
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	iRetValue = I2C_TxClockPrescaler(ch, iSel * 1000);
	return	iRetValue;
}

/**
* {I2C_Close}
*
* @brief			Deinitialize I2CCON & Unregister Interrupt Handler & GPIO
* @param[in]	u32 ch
* @return		void
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void I2C_Close( u32 ch )
{
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);
}

/**
* {I2C_SRx_MTx_CrossTest_Interrupt}
*
* @brief			This function do Slave Rx Interrupt <- Master Tx polling mode operation with tx_ch,rx_ch.
* @ description 	Ch 0 Slave Rx Interrupt <-- Ch 1 Master Tx polling.
* @param[in]	(u32 tx_ch, u32 rx_ch)
* @return		s32
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_SRx_MTx_CrossTest_Interrupt(u32 tx_ch, u32 rx_ch)
{
	s32 iRetval = OK;
	
	u32	uTmp, i, ErrCnt = 0;
	u8 SlaveAddr, RxDone = 0;
	static u8 RxData[256] = {0,};

	SlaveAddr = 0x54;  //assume channel 2 Slave address is 0x55

//------------------------------------
// Set Init
	gI2c_Tx_Ch = tx_ch;
	gI2c_Rx_Ch = rx_ch;
	
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_SLAVE_RX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = &testpattern[0];
	I2C_Context.Rx_buffer = &RxData[0];
	I2C_Context.bytes = 256;	
	
//------------------------------------
// Slave RX
	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	while( uTmp & I2CSTAT_START )				//	Wait until I2C bus is free.
		uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);			

	uTmp = I2C_Inp32(I2C_REG[rx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[rx_ch].CON, uTmp);		

	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp &= ~(I2CSTAT_ENABLE);	// Disable Rx/Tx for setting Slave Address Reg
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[rx_ch].ADD,  SlaveAddr);

	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp |= (I2CSTAT_SLAVE_RX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);	
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp | I2CSTAT_START);	//	Slave Rx Start.
//------------------------------------

	
//------------------------------------
// Master TX
	//Wait until I2C bus is free.
	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	while( uTmp & I2CSTAT_START )
		uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);

	uTmp = I2C_Inp32(I2C_REG[tx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[tx_ch].CON, uTmp);

	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp |= I2CSTAT_ENABLE;	// Enable Rx/Tx for setting Data Shift Reg
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].DS,  SlaveAddr);
	
	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp |= (I2CSTAT_MASTER_TX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp | I2CSTAT_START);	// Master Tx Start.
//------------------------------------
	
	// wait til Tx Interrupt request is done.
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}
//------------------------------------


	for( i = 0; i < 256; i++)
	{
		if (RxData[i] != testpattern[i])
		{
			ErrCnt++;
	   		UART_Printf("\n Error pattern (0x%x) ==>  (0x%x)",testpattern[i], RxData[i]);			
		}
		else
		{
			UART_Printf("\r Slave RX (0x%x) ", RxData[i]);
			UART_Printf(" Master TX (0x%x) ==> ", testpattern[i]);
		}
	}

	if (ErrCnt)		
		UART_Printf("\n ==> TEST Error !!!");
	else
		UART_Printf("\n ==> TEST Good !!!");

	return	iRetval;
}

/**
* {I2C_STx_MRx_CrossTest_Interrupt}
*
* @brief			This function do Slave Tx Interrupt <- Master Rx polling mode operation with tx_ch,rx_ch.
* @ description 	Ch 0 Slave Tx Interrupt <-- Ch 1 Master Rx polling.
* @param[in]	(u32 tx_ch, u32 rx_ch)
* @return		s32
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_STx_MRx_CrossTest_Interrupt(u32 tx_ch, u32 rx_ch)
{
	s32 iRetval = OK;
	
	u32	uTmp, i, ErrCnt = 0;
	u8 SlaveAddr, RxDone = 0;
	static u8 RxData[256+4] = {0,};

	SlaveAddr = 0x54;  //assume channel 2 Slave address is 0x55

//------------------------------------
// Set Init
	gI2c_Tx_Ch = tx_ch;
	gI2c_Rx_Ch = rx_ch;
	
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_SLAVE_TX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = &testpattern[0];
	I2C_Context.Rx_buffer = &RxData[0];
	I2C_Context.bytes = 256;
	
//------------------------------------
// Slave TX
	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	while( uTmp & I2CSTAT_START )				//	Wait until I2C bus is free.
		uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);			

	uTmp = I2C_Inp32(I2C_REG[tx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[tx_ch].CON, uTmp);		

	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp &= ~(I2CSTAT_ENABLE);	// Disable Rx/Tx for setting Slave Address Reg
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].ADD,  SlaveAddr);

	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp |= (I2CSTAT_SLAVE_TX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp | I2CSTAT_ENABLE);	//	Slave Tx Start.
//------------------------------------
	
//------------------------------------
// Master RX
	//Wait until I2C bus is free.
	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	while( uTmp & I2CSTAT_START )
		uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);

	uTmp = I2C_Inp32(I2C_REG[rx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[rx_ch].CON, uTmp);

	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp |= I2CSTAT_ENABLE;	// Enable Rx/Tx for setting Data Shift Reg
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[rx_ch].DS,  SlaveAddr);
	
	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp |= (I2CSTAT_MASTER_RX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp | I2CSTAT_START);	// Master Rx Start.
//------------------------------------

	Delay(100);//wait till Master I2CDS transmitted.
	
	// wait til Tx Interrupt request is done.
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}
	
//------------------------------------

	for( i = 0; i < 256; i++)
	{
		if (RxData[i] != testpattern[i])
		{
			ErrCnt++;
	   		UART_Printf("\n Error pattern (0x%x) ==>  (0x%x)",testpattern[i], RxData[i]);			
		}
		else 
		{
			UART_Printf("\r Slave TX (0x%x) ==> ", testpattern[i]);
			UART_Printf(" Master RX (0x%x) ", RxData[i]);
		}
	}

	if (ErrCnt)		
		UART_Printf("\n ==> TEST Error !!!\n");
	else
		UART_Printf("\n ==> TEST Good !!!\n");

	return	iRetval;
}	

/**
* {I2C_MRx_STx_CrossTest_Interrupt}
*
* @brief			This function do Master Rx Interrupt <- Slave Tx polling mode operation with tx_ch,rx_ch.
* @ description 	Ch 0 Master Rx Interrupt <-- Ch 1 Slave Tx polling.
* @param[in]	(u32 tx_ch, u32 rx_ch)
* @return		s32
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_MRx_STx_CrossTest_Interrupt(u32 tx_ch, u32 rx_ch)
{
	s32 iRetval = OK;
	
	u32	uTmp, i, ErrCnt = 0;
	u8 SlaveAddr, RxDone = 0;
	static u8 RxData[256+4] = {0,};

	SlaveAddr = 0x54;  //assume channel 2 Slave address is 0x55

//------------------------------------
// Set Init
	gI2c_Tx_Ch = tx_ch;
	gI2c_Rx_Ch = rx_ch;
	
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_MASTER_RX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = &testpattern[0];
	I2C_Context.Rx_buffer = &RxData[0];
	I2C_Context.bytes = 256;
	
//------------------------------------
// Slave TX
	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	while( uTmp & I2CSTAT_START )				//	Wait until I2C bus is free.
		uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);			

	uTmp = I2C_Inp32(I2C_REG[tx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[tx_ch].CON, uTmp);		

	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp &= ~(I2CSTAT_ENABLE);	// Disable Rx/Tx for setting Slave Address Reg
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].ADD,  SlaveAddr);

	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp |= (I2CSTAT_SLAVE_TX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp | I2CSTAT_ENABLE);	//	Slave Tx Start.
//------------------------------------
	
//------------------------------------
// Master RX
	//Wait until I2C bus is free.
	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	while( uTmp & I2CSTAT_START )
		uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);

	uTmp = I2C_Inp32(I2C_REG[rx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[rx_ch].CON, uTmp);

	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp |= I2CSTAT_ENABLE;	// Enable Rx/Tx for setting Data Shift Reg
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[rx_ch].DS,  SlaveAddr);
	
	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp |= (I2CSTAT_MASTER_RX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp | I2CSTAT_START);	// Master Rx Start.
//------------------------------------

	// wait til Tx Interrupt request is done.
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}
	
//------------------------------------

	for( i = 0; i < 256; i++)
	{
		if (RxData[i] != testpattern[i])
		{
			ErrCnt++;
	   		UART_Printf("\n Error pattern (0x%x) ==>  (0x%x)",testpattern[i], RxData[i]);			
		}
		else 
		{
			UART_Printf("\r Master RX (0x%x) ", RxData[i]);
			UART_Printf(" Slave TX (0x%x) ==> ", testpattern[i]);
		}
	}

	if (ErrCnt)		
		UART_Printf("\n ==> TEST Error !!!\n");
	else
		UART_Printf("\n ==> TEST Good !!!\n");

	return	iRetval;
}	

/**
* {I2C_MTx_SRx_CrossTest_Interrupt}
*
* @brief			This function do Master TX -> Slave Rx mode by interrupt operation with tx_ch,rx_ch.
* 				Master TX --> Slave RX.
* @param[in]	(u32 tx_ch, u32 rx_ch)
* @return		s32
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_MTx_SRx_CrossTest_Interrupt(u32 tx_ch, u32 rx_ch)
{
	s32 iRetval = OK;
	
	u32	uTmp, i, ErrCnt = 0;
	u8 SlaveAddr, RxDone = 0;
	static u8 RxData[256] = {0,};

	SlaveAddr = 0x54;  //assume channel 2 Slave address is 0x55

//------------------------------------
// Set Init
	gI2c_Tx_Ch = tx_ch;
	gI2c_Rx_Ch = rx_ch;
	
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_MASTER_TX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = &testpattern[0];
	I2C_Context.Rx_buffer = &RxData[0];
	I2C_Context.bytes = 256;	
	
//------------------------------------
// Slave RX
	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	while( uTmp & I2CSTAT_START )				//	Wait until I2C bus is free.
		uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);			

	uTmp = I2C_Inp32(I2C_REG[rx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[rx_ch].CON, uTmp);		

	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp &= ~(I2CSTAT_ENABLE);	// Disable Rx/Tx for setting Slave Address Reg
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[rx_ch].ADD,  SlaveAddr);

	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp |= (I2CSTAT_SLAVE_RX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);	
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp | I2CSTAT_START);	//	Slave Rx Start.
//------------------------------------

	
//------------------------------------
// Master TX
	//Wait until I2C bus is free.
	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	while( uTmp & I2CSTAT_START )
		uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);

	uTmp = I2C_Inp32(I2C_REG[tx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[tx_ch].CON, uTmp);

	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp |= I2CSTAT_ENABLE;	// Enable Rx/Tx for setting Data Shift Reg
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].DS,  SlaveAddr);
	
	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp |= (I2CSTAT_MASTER_TX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp | I2CSTAT_START);	// Master Tx Start.
//------------------------------------
	
	// wait til Tx Interrupt request is done.
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}
//------------------------------------


	for( i = 0; i < 256; i++)
	{
		if (RxData[i] != testpattern[i])
		{
			ErrCnt++;
	   		UART_Printf("\n Error pattern (0x%x) ==>  (0x%x)",testpattern[i],RxData[i]);			
		}
		else
		{
			UART_Printf("\r Master TX (0x%x) ==> ", testpattern[i]);
			UART_Printf(" Slave RX (0x%x) ", RxData[i]);
		}
	}

	if (ErrCnt)		
		UART_Printf("\n ==> TEST Error !!!");
	else
		UART_Printf("\n ==> TEST Good !!!");

	return	iRetval;
}

/**
* {I2C_MRx_STx_CrossTest_Polling}
*
* @brief			This function do Slave TX -> Master Rx mode by polling operation with tx_ch,rx_ch.
* 				Slave TX --> Master RX.
* @param[in]	(u32 tx_ch, u32 rx_ch)
* @return		s32
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_MRx_STx_CrossTest_Polling(u32 tx_ch, u32 rx_ch)
{
	s32 iRetval = OK;
	
	u32	uTmp, i, ErrCnt = 0;
	u8 SlaveAddr, RxDone = 0;
	u8 RxData[256+4] = {0,};

	SlaveAddr = 0x54;  //assume channel 2 Slave address is 0x55
	
//------------------------------------
// Slave TX
	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	while( uTmp & I2CSTAT_START )				//	Wait until I2C bus is free.
		uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);			

	uTmp = I2C_Inp32(I2C_REG[tx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[tx_ch].CON, uTmp);		

	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp &= ~(I2CSTAT_ENABLE);	// Disable Rx/Tx for setting Slave Address Reg
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].ADD,  SlaveAddr);

	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp |= (I2CSTAT_SLAVE_TX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp | I2CSTAT_ENABLE);	//	Slave Tx Start.
//------------------------------------
	
//------------------------------------
// Master RX
	//Wait until I2C bus is free.
	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	while( uTmp & I2CSTAT_START )
		uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);

	uTmp = I2C_Inp32(I2C_REG[rx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[rx_ch].CON, uTmp);

	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp |= I2CSTAT_ENABLE;	// Enable Rx/Tx for setting Data Shift Reg
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[rx_ch].DS,  SlaveAddr);
	
	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp |= (I2CSTAT_MASTER_RX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp | I2CSTAT_START);	// Master Rx Start.
//------------------------------------
	Delay(10);

	I2C_SlaveAddressMatch_Polling(tx_ch);	// Slave TX address matching
	
	I2C_Pending_Bit_Clear(rx_ch);	// Master RX : Clear pending bit to resume.
	I2C_ACK_Check_Polling(rx_ch); 	// Master RX
//------------------------------------

	for(i = 0; i < 256; i++)
	{
		// Slave TX
		I2C_Outp8(I2C_REG[tx_ch].DS, testpattern[i]); 
		
		I2C_Pending_Bit_Clear(rx_ch);		// RX : Clear pending bit to resume  (޴ RX pending clear ؾ غ .(ᱹ TX  ϰ .)
		I2C_Pending_Bit_Clear(tx_ch);		// TX : Clear pending bit to resume

		Delay(10);
		
		// Master RX check
		RxDone = 0;
		while( !RxDone )
		{
			if( I2C_Inp8(I2C_REG[rx_ch].CON) & I2CCON_INT_PEND ) 
			{
				RxData[i] = I2C_Inp8(I2C_REG[rx_ch].DS);
				RxDone = true;
				I2C_Pending_Bit_Clear(rx_ch);	//	RX : Clear pending bit to resume
			}
		}

		// ACk detect?
		I2C_ACK_Check_Polling(tx_ch);

		UART_Printf("\r Slave TX (0x%x) ==> ", testpattern[i]);
		UART_Printf(" Master RX (0x%x) ", RxData[i]);
	}


	// Master RX Stop
	I2C_Outp8(I2C_REG[rx_ch].STAT, I2CSTAT_MASTER_RX_MODE | I2CSTAT_ENABLE );	//	0xD0 : Stop Master Rx condition, ACK flag clear
	I2C_Pending_Bit_Clear(rx_ch);
	
	// Slave Tx stop
	I2C_Outp8(I2C_REG[tx_ch].STAT, I2CSTAT_IDLE);		//	Stop Slave Tx condition, ACK flag clear
	I2C_Pending_Bit_Clear(tx_ch);	//	RX : Clear pending bit to resume


	for( i = 0; i < 256; i++)
	{
		if (RxData[i] != testpattern[i])
		{
			ErrCnt++;
	   		UART_Printf("\n Error pattern (0x%x) ==>  (0x%x)",testpattern[i], RxData[i]);			
		}			
	}

	if (ErrCnt)		
		UART_Printf("\n ==> TEST Error !!!\n");
	else
		UART_Printf("\n ==> TEST Good !!!\n");

	return	iRetval;
}

/**
* {I2C_MTx_SRx_CrossTest_Polling}
*
* @brief			This function do Master TX -> Slave Rx mode by polling operation with tx_ch,rx_ch.
* 				Master TX --> Slave RX.
* @param[in]	(u32 tx_ch, u32 rx_ch)
* @return		s32
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_MTx_SRx_CrossTest_Polling(u32 tx_ch, u32 rx_ch)
{
	s32 iRetval = OK;
	
	u32	uTmp, i, ErrCnt = 0;
	u8 SlaveAddr, RxDone = 0;
	u8 RxData[256+4] = {0,};

	SlaveAddr = 0x54;  //assume channel 2 Slave address is 0x55
	
//------------------------------------
// Slave RX
	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	while( uTmp & I2CSTAT_START )				//	Wait until I2C bus is free.
		uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);			

	uTmp = I2C_Inp32(I2C_REG[rx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[rx_ch].CON, uTmp);		

	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp &= ~(I2CSTAT_ENABLE);	// Disable Rx/Tx for setting Slave Address Reg
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[rx_ch].ADD,  SlaveAddr);

	uTmp = I2C_Inp32(I2C_REG[rx_ch].STAT);
	uTmp |= (I2CSTAT_SLAVE_RX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp);	
	I2C_Outp8(I2C_REG[rx_ch].STAT, uTmp | I2CSTAT_START);	//	Slave Rx Start.
//------------------------------------
	
//------------------------------------
// Master TX
	//Wait until I2C bus is free.
	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	while( uTmp & I2CSTAT_START )
		uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);

	uTmp = I2C_Inp32(I2C_REG[tx_ch].CON);
	uTmp |= (I2CCON_ACK_ENABLE | I2CCON_INT_ENABLE);	//	Enable ACK & Interrupt
	I2C_Outp8(I2C_REG[tx_ch].CON, uTmp);

	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp |= I2CSTAT_ENABLE;	// Enable Rx/Tx for setting Data Shift Reg
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].DS,  SlaveAddr);
	
	uTmp = I2C_Inp32(I2C_REG[tx_ch].STAT);
	uTmp |= (I2CSTAT_MASTER_TX_MODE | I2CSTAT_ENABLE);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp);
	I2C_Outp8(I2C_REG[tx_ch].STAT, uTmp | I2CSTAT_START);	// Master Tx Start.
//------------------------------------
	Delay(10);

	I2C_SlaveAddressMatch_Polling(rx_ch);	// Slave RX address matching

	I2C_Pending_Bit_Clear(tx_ch); //	Master TX : Clear pending bit to resume	
	I2C_ACK_Check_Polling(tx_ch); // Master TX :
//------------------------------------

	for(i = 0; i < 256; i++)
	{
		// Master TX
		I2C_Outp8(I2C_REG[tx_ch].DS, testpattern[i]); 
		
		I2C_Pending_Bit_Clear(rx_ch);	//	RX : Clear pending bit to resume (޴ RX pending clear ؾ غ .(ᱹ TX  ϰ )
		I2C_Pending_Bit_Clear(tx_ch);	//	TX : Clear pending bit to resume

		// Slave RX check
		RxDone = 0;
		while( !RxDone )
		{
			if( I2C_Inp8(I2C_REG[rx_ch].CON) & I2CCON_INT_PEND ) 
			{
				RxData[i] = I2C_Inp8(I2C_REG[rx_ch].DS);
				RxDone = true;
				I2C_Pending_Bit_Clear(rx_ch);	//	RX : Clear pending bit to resume
			}
		}

		// ACk detect?
		I2C_ACK_Check_Polling(tx_ch);

		UART_Printf("\r Master TX (0x%x) ==> ", testpattern[i]);
		UART_Printf(" Slave RX (0x%x) ", RxData[i]);
	}


	// Master TX Stop
	I2C_Outp8(I2C_REG[tx_ch].STAT, I2CSTAT_MASTER_TX_MODE | I2CSTAT_ENABLE );	//	0xD0 : Stop Master Tx condition, ACK flag clear
	I2C_Pending_Bit_Clear(tx_ch);
	
	// Slave RS stop
	I2C_Outp8(I2C_REG[rx_ch].STAT, I2CSTAT_IDLE);		//	Stop Master Tx condition, ACK flag clear
	I2C_Pending_Bit_Clear(rx_ch);		//	RX : Clear pending bit to resume


	for( i = 0; i < 256; i++)
	{
		if (RxData[i] != testpattern[i])
		{
			ErrCnt++;
	   		UART_Printf("\n Error pattern (0x%x) ==>  (0x%x)",testpattern[i],RxData[i]);			
		}			
	}

	if (ErrCnt)		
		UART_Printf("\n ==> TEST Error !!!");
	else
		UART_Printf("\n ==> TEST Good !!!");

	return	iRetval;
}

/**
* {I2C_CH1_CH0_CrossTest_Polling}
*
* @brief			This function do Master TX->SlaveRx mode by polling operation with tx_ch,rx_ch.
* @description	CH 1 Master TX -->  CH 0 Slave RX Only.
* @param[in]	void
* @return		u8
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_CH1_CH0_CrossTest_Polling(void)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	u32 iSel;

	u32 uTx_Ch;
	u32 uRx_Ch;

	s32	iRetValue = OK;
	
	UART_Printf("\n\n\n");
	UART_Printf("\n[I2C_CH1_CH0_CrossTest_Polling : CH1 Master TX --> CH0 Slave RX");
	UART_Printf("\n[CH1 CH0  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz


	uTx_Ch = I2C_CH_HDMI_DDC;
	uRx_Ch = I2C_CH0;
	
	// initialize
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY1;
	}
	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY1;
	}
	
	// Ch 1 Tx <-> Ch 0 Rx
	iRetval = I2C_MTx_SRx_CrossTest_Polling(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY1:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
		
	return	ret;
}

#ifdef v210_I2C_FPGA
/**
* {I2C_CH0_CH2_CrossTest_Polling}
*
* @brief			This function do Master TX->SlaveRx mode by polling operation with tx_ch,rx_ch.
* @description	Ch 0 Master TX -->  Ch 2 Slave RX & vice versa.
* @param[in]	void
* @return		u8
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_CH0_CH2_CrossTest_Polling(void)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	u32 iSel;

	u32 uTx_Ch;
	u32 uRx_Ch;

	s32	iRetValue = OK;
	
	UART_Printf("\n\n\n");
	UART_Printf("\n[I2C_CH0_CH2_CrossTest_Polling : CH0 Master TX --> CH2 Slave RX");
	UART_Printf("\n[CH0 CH2  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH0;
	uRx_Ch = I2C_CH2;
	
	// initialize
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY1;
	}
	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY1;
	}
	
	// Ch0 Tx <-> Ch20 Rx
	iRetval = I2C_MTx_SRx_CrossTest_Polling(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY1:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}

	UART_Printf("\n\n\n");	
	UART_Printf("\n[I2C_CH0_CH2_CrossTest_Polling : CH2 Master TX --> CH0 Slave RX]");
	UART_Printf("\n[CH2 CH0  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());


	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ)
		iSel = I2C_MAX_FREQ; //MAX 500Khz


	uTx_Ch = I2C_CH2;
	uRx_Ch = I2C_CH0;

	// initialize
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY2;
	}
	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY2;
	}
	
	// Ch2 Tx <-> Ch0 Rx
	iRetval = I2C_MTx_SRx_CrossTest_Polling(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY2:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	return	ret;
}
#else
/**
* {I2C_CH0_CH1_CrossTest_Polling}
*
* @brief			This function do Master TX->Slave Rx mode by polling operation with tx_ch, rx_ch.
* @description	CH 0 Master TX -->  CH 1 Slave RX & vice versa.
* @param[in]	void
* @return		u8
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_CH0_CH1_CrossTest_Polling(void)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	u32 iSel;

	u32 uTx_Ch;
	u32 uRx_Ch;

	s32	iRetValue = OK;


/* CASE 1 : CH 0 Master Tx -> CH 1 Slave Rx */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH0 Master TX --> CH1 Slave RX]");
	UART_Printf("\n[CH0 CH1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH0;
	uRx_Ch = I2C_CH_HDMI_DDC;
	
	// Tx Channel Initialization
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY1;
	}

	// Rx Channel Initialization	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY1;
	}
	
	// CH 0 Tx <-> CH 1 Rx
	iRetval = I2C_MTx_SRx_CrossTest_Polling(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY1:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}


/* CASE 2 : CH 1 Master Tx -> CH 0 Slave Rx */
	UART_Printf("\n\n\n");	
	UART_Printf("\n[CH 1 Master TX --> CH 0 Slave RX]");
	UART_Printf("\n[CH1 CH0  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());


	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ)
		iSel = I2C_MAX_FREQ; //MAX 500Khz


	uTx_Ch = I2C_CH_HDMI_DDC;
	uRx_Ch = I2C_CH0;

	// Tx Channel Initialization
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY2;
	}

	// Rx Channel Initialization	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY2;
	}
	
	// CH 1 Tx <-> CH 0 Rx
	iRetval = I2C_MTx_SRx_CrossTest_Polling(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY2:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);

	if( ret == FALSE)
	{
		return	ret;
	}

	
/* CASE 3 : CH 0 Slave Tx -> CH 1 Master Rx */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH 0 Slave TX --> CH 1 Master RX]");
	UART_Printf("\n[CH 0 CH1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH0;
	uRx_Ch = I2C_CH_HDMI_DDC;
	
	// Tx Channel Initialization
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY3;
	}

	// Rx Channel Initialization	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY3;
	}
	
	// CH 0 Tx <-> CH 1 Rx
	iRetval = I2C_MRx_STx_CrossTest_Polling(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY3:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}

/* CASE 4 : CH 1 Slave Tx -> CH 0 Master Rx */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH 1 Slave TX --> CH 0 Master RX]");
	UART_Printf("\n[CH 1 CH 0  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH_HDMI_DDC;
	uRx_Ch = I2C_CH0;
	
	// Tx Channel Initialization
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY4;
	}

	// Rx Channel Initialization	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY4;
	}
	
	// CH 1 Tx <-> CH 0 Rx
	iRetval = I2C_MRx_STx_CrossTest_Polling(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY4:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}	
	return	ret;
}

/**
* {I2C_CH0_CH1_CrossTest_Interrupt}
*
* @brief			This function do Master Tx Interrupt -> Slave Rx Polling mode operation with tx_ch, rx_ch.
* @description	CH 0 Master TX Interrupt --> CH 1 Slave RX Polling \n
				CH 1 Master TX Interrupt --> CH 0 Slave RX Polling.
* @param[in]	void
* @return		u8
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8	I2C_CH0_CH1_CrossTest_Interrupt(void)
{
	u8	ret = TRUE;
	s32 iRetval = OK;
	
	u32 iSel;

	u32 uTx_Ch;
	u32 uRx_Ch;

	s32	iRetValue = OK;


/* CASE 1 : CH 0 Master Tx Interrupt -> CH 1 Slave Rx Polling */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH 0 Master TX Interrupt --> CH 1 Slave RX Polling]");
	UART_Printf("\n[CH 0 CH 1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH0;
	uRx_Ch = I2C_CH_HDMI_DDC;
	
	// Tx Channel Initialization : Interrupt Handler will be registered later.
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY1;
	}
	
	// Register Master Tx Interrupt Handler.
	INTC_SetVectAddr(NUM_I2C, I2C_MTx_IRQHandler);
	INTC_Enable(NUM_I2C);
	
	// Rx Channel Initialization : Interrupt Handler will be registered later.
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY1;
	}
	
	// CH 0 Tx <-> CH 1 Rx
	iRetval = I2C_MTx_SRx_CrossTest_Interrupt(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY1:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}

/* CASE 2 : CH 1 Master Tx Interrupt -> CH 0 Slave Rx Polling */
	UART_Printf("\n\n\n");	
	UART_Printf("\n[CH1 Master TX Interrupt --> CH0 Slave RX Polling]");
	UART_Printf("\n[CH1 CH0  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());


	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ)
		iSel = I2C_MAX_FREQ; //MAX 500Khz


	uTx_Ch = I2C_CH_HDMI_DDC;
	uRx_Ch = I2C_CH0;

	// Tx Channel Initialization : Interrupt Handler will be registered later.
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY2;
	}

	// Register Master Tx Interrupt Handler.
	INTC_SetVectAddr(NUM_HDMI_I2C, I2C_MTx_IRQHandler);
	INTC_Enable(NUM_HDMI_I2C);

	// Rx Channel Initialization : Interrupt Handler will be registered later.
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY2;
	}
	
	// CH 1 Tx <-> CH 0 Rx
	iRetval = I2C_MTx_SRx_CrossTest_Interrupt(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY2:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);


/* CASE 3 : CH 0 Master Rx Interrupt -> CH 1 Slave Tx Polling */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH 0 Master Rx Interrupt --> CH 1 Slave Tx Polling]");
	UART_Printf("\n[CH 0 CH1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH_HDMI_DDC;
	uRx_Ch = I2C_CH0;
	
	// Tx Channel Initialization
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY3;
	}

	// Register Master Rx Interrupt Handler.
	INTC_SetVectAddr(NUM_I2C, I2C_MRx_IRQHandler);
	INTC_Enable(NUM_I2C);
	
	// Rx Channel Initialization	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY3;
	}
	
	// CH 0 Rx <-> CH 1 Tx
	iRetval = I2C_MRx_STx_CrossTest_Interrupt(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY3:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}

/* CASE 4 : CH 1 Master Rx Interrupt -> CH 0 Slave Tx Polling */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH 1 Master Rx Interrupt --> CH 0 Slave Tx Polling]");
	UART_Printf("\n[CH 1 CH 0  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH0;
	uRx_Ch = I2C_CH_HDMI_DDC;
	
	// Tx Channel Initialization
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY4;
	}

	// Register Master Rx Interrupt Handler.
	INTC_SetVectAddr(NUM_HDMI_I2C, I2C_MRx_IRQHandler);
	INTC_Enable(NUM_HDMI_I2C);
	
	// Rx Channel Initialization	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY4;
	}
	
	// CH 0 Tx <-> CH 1 Rx
	iRetval = I2C_MRx_STx_CrossTest_Interrupt(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY4:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}

/* CASE 5 : CH 0 Slave Tx Interrupt -> CH 1 Master Rx Polling */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH 0 Slave Tx Interrupt --> CH 1 Master Rx Polling]");
	UART_Printf("\n[CH 0 CH 1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH0;
	uRx_Ch = I2C_CH_HDMI_DDC;
	
	// Tx Channel Initialization
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY5;
	}

	// Register Master Rx Interrupt Handler.
	INTC_SetVectAddr(NUM_I2C, I2C_STx_IRQHandler);
	INTC_Enable(NUM_I2C);
	
	// Rx Channel Initialization	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY5;
	}
	
	// CH 0 Tx <-> CH 1 Rx
	iRetval = I2C_STx_MRx_CrossTest_Interrupt(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY5:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}

/* CASE 6 : CH 1 Slave Tx Interrupt -> CH 0 Master Rx Polling */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH 1 Slave Tx Interrupt --> CH 0 Master Rx Polling]");
	UART_Printf("\n[CH 1 CH 0  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH_HDMI_DDC;
	uRx_Ch = I2C_CH0;
	
	// Tx Channel Initialization
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY6;
	}

	// Register Master Rx Interrupt Handler.
	INTC_SetVectAddr(NUM_HDMI_I2C, I2C_STx_IRQHandler);
	INTC_Enable(NUM_HDMI_I2C);
	
	// Rx Channel Initialization	
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY6;
	}
	
	// CH 1 Tx <-> CH 0 Rx
	iRetval = I2C_STx_MRx_CrossTest_Interrupt(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY6:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}

/* CASE 7 : CH 0 Slave Rx Interrupt -> CH 1 Master Tx Polling */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH 0 Slave Rx Interrupt --> CH 1 Master Tx Polling]");
	UART_Printf("\n[CH 0 CH 1  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH_HDMI_DDC;
	uRx_Ch = I2C_CH0;
	
	// Tx Channel Initialization : Interrupt Handler will be registered later.
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY7;
	}
	
	// Register Slave Rx Interrupt Handler.
	INTC_SetVectAddr(NUM_I2C, I2C_SRx_IRQHandler);
	INTC_Enable(NUM_I2C);
	
	// Rx Channel Initialization : Interrupt Handler will be registered later.
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY7;
	}
	
	// CH 0 Rx <-> CH 1 Tx
	iRetval = I2C_SRx_MTx_CrossTest_Interrupt(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY7:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}

/* CASE 8 : CH 1 Slave Rx Interrupt -> CH 0 Master Tx Polling */
	UART_Printf("\n\n\n");
	UART_Printf("\n[CH 1 Slave Rx Interrupt --> CH 0 Master Tx Polling]");
	UART_Printf("\n[CH 1 CH 0  ؾմϴ. غǸ ƹŰ ..]");
	while(!GetKey());

	IIC_DBG_PRINT(DBG_Basic,"\n-Enter SCL clock (Khz) : ");
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if ( (iSel = UART_GetIntNum() ) > I2C_MAX_FREQ )
		iSel = I2C_MAX_FREQ; //MAX 500Khz

	uTx_Ch = I2C_CH0;
	uRx_Ch = I2C_CH_HDMI_DDC;
	
	// Tx Channel Initialization : Interrupt Handler will be registered later.
	iRetValue = I2C_Open(uTx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY8;
	}
	
	// Register Slave Rx Interrupt Handler.
	INTC_SetVectAddr(NUM_HDMI_I2C, I2C_SRx_IRQHandler);
	INTC_Enable(NUM_HDMI_I2C);
	
	// Rx Channel Initialization : Interrupt Handler will be registered later.
	iRetValue = I2C_Open(uRx_Ch, iSel);
	if( iRetValue != OK )
	{
		ret = FALSE;
		goto I2C_EMERGENCY8;
	}
	
	// CH 1 Rx <-> CH 0 Tx
	iRetval = I2C_SRx_MTx_CrossTest_Interrupt(uTx_Ch, uRx_Ch);
	if( iRetval != OK )
	{
		ret = FALSE;
	}

I2C_EMERGENCY8:
	
	// deinitialize
	I2C_Close(uRx_Ch);
	I2C_Close(uTx_Ch);
	
	if( ret == FALSE)
	{
		return	ret;
	}

	return	ret;
}	
#endif/* #ifdef v210_I2C_FPGA */

/**
* {I2C_CH0_CH2_AutoTest}
*
* @brief			CH 0 & CH 2 Master Tx/Rx Interrupt & Polling Test in EEPROM.
* 
* @param[in]	void
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8 I2C_CH0_CH2_AutoTest(void)
{
	u8	ret = TRUE;
	
	ret = I2C_Ch0_Master_Interrupt_EEPROM_Test();
	ret = I2C_Ch0_Master_Polling_EEPROM_Test();
#ifdef v210_I2C_FPGA
	ret = I2C_Ch2_Master_Interrupt_EEPROM_Test();
	ret = I2C_Ch2_Master_Polling_EEPROM_Test();
#else
	ret = I2C_Ch1_Master_Interrupt_EEPROM_Test();
	ret = I2C_Ch1_Master_Polling_EEPROM_Test();
#endif

	ret = I2C_Ch0_Master_Interrupt_EEPROM_AutoTest();
	ret = I2C_Ch0_Master_Polling_EEPROM_AutoTest();
#ifdef v210_I2C_FPGA
	ret = I2C_Ch2_Master_Interrupt_EEPROM_AutoTest();
	ret = I2C_Ch2_Master_Polling_EEPROM_AutoTest();
#else
	ret = I2C_Ch1_Master_Interrupt_EEPROM_AutoTest();
	ret = I2C_Ch1_Master_Polling_EEPROM_AutoTest();
#endif

	return	ret;
}

#ifndef v210_I2C_FPGA
/**
* {I2C_Bus_ArbitrationTest}
*
* @brief			CH 0 & CH 1 Bus Arbitration Test from EEPROM.
* @description	Case #1 : If a master with a SDA HIGH level detects the other master with a SDA active LOW level,\n
						 it does not initiate a data transfer.\n
				Case #2 : If the masters simultaneously lower the SDA line, each master evaluates the mastership by slave address.\n		 
* @param[in]	void
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8 I2C_Bus_Arbitration_Test(void)
{
	u8	ret = TRUE;
	
	u32 ch0 = I2C_CH0;				// Channel 0 : I2C_CH0
	u32	ch1 = I2C_CH_HDMI_DDC;	// Channel 1 : I2C_CH_HDMI_DDC
	
	u8	cSlaveAddr = S524AD0XD1_SLAVEADDRESS;

	u8	uTmp;
	u8	ucStatus;
	u8	ucReg;
	s32	retval = OK;

	u32 ch = 0;

	bool bI2C_IntPend = FALSE;

	u32 uCnt = 0;	
	u32 uMin = 0x0;
	u32 uMax = 0x10;

	s32 iRetval = OK;

	u8	uTemp0;
	u8	uTemp1;

	u32 ch_OK = 0;

	u8	Test_SlaveAddr;
	
// Channel 0	
	I2C_Open(ch0, 10);

	uTmp = I2C_Inp32(I2C_REG[ch0].STAT);
	while( (uTmp & I2CSTAT_START ) || (uTmp & I2CSTAT_ARBITER) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp = I2C_Inp8(I2C_REG[ch0].STAT);
	}

	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch0].CON, (I2C_Inp8(I2C_REG[ch0].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_ACK_ENABLE);
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch0].STAT, I2CSTAT_ENABLE | I2CSTAT_MASTER_TX_MODE);	// 0xD0
	// put address to shift register
	I2C_Outp8(I2C_REG[ch0].DS, cSlaveAddr & 0xFE);	// 0xA0 : right slave address( 0 bit is high priority, 1 bit is low priority )


// Channel 1
	I2C_Open(ch1, 10);

	uTmp = I2C_Inp32(I2C_REG[ch1].STAT);
	while( (uTmp & I2CSTAT_START ) || (uTmp & I2CSTAT_ARBITER) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp = I2C_Inp8(I2C_REG[ch1].STAT);
	}

	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch1].CON, (I2C_Inp8(I2C_REG[ch1].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_ACK_ENABLE);
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch1].STAT, I2CSTAT_ENABLE | I2CSTAT_MASTER_TX_MODE);	// 0xD0
	// put address to shift register 
	// Test Case #1 : 0xE0
	// Test Case #2 : 0x80
	
	//Test_SlaveAddr = 0xE0;
	Test_SlaveAddr = 0x80;
	I2C_Outp8(I2C_REG[ch1].DS, Test_SlaveAddr & 0xFE);	// wrong slave address( 0 bit is high priority, 1 bit is low priority )


	/* Case #1, Case #2, one of two should happen */
	uTemp0 = I2C_Inp32(I2C_REG[ch0].STAT);
	uTemp1 = I2C_Inp32(I2C_REG[ch1].STAT);
	
// Channel 0
	// master tx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch0].STAT, uTemp0 | I2CSTAT_START ); // 0xF0
// Channel 1
	// master tx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch1].STAT, uTemp1 | I2CSTAT_START ); // 0xF0

	do
	{
		for(ch = I2C_CH0; ch <= I2C_CH_HDMI_DDC; ch++)
		{
			ucStatus = I2C_Inp8(I2C_REG[ch].CON);
			if (ucStatus & I2CCON_INT_PEND)
			{
				// check status flags
				ucReg = I2C_Inp8(I2C_REG[ch].STAT);

				if (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR) )
				{
					if (ucReg & I2CSTAT_ARBITER)
					{
						UART_Printf("\n [I2C CH %d : ARBITRATION FAIL]\n", ch);
						retval = OK;
					}
					else if (ucReg & I2CSTAT_ACK_ERROR)
					{
						UART_Printf("\n [I2C CH %d : ACK ERROR]\n", ch);
						retval = ERROR;
						
						Assert(ch == I2C_CH_HDMI_DDC);						
						if( ch == I2C_CH_HDMI_DDC )
							bI2C_IntPend = TRUE;
					}
					
					I2C_Release(ch);
					I2C_Close(ch);
					
					if( ch == I2C_CH0 )
						ch_OK = I2C_CH_HDMI_DDC;
					else
						ch_OK = I2C_CH0;					
				}	
				else
				{
					bI2C_IntPend = TRUE;
				}
			}
		}

		if( bI2C_IntPend == TRUE )
			break;
	}while(1);
	
	
	if( retval == OK )
	{
		// send stop bit
		I2C_Outp8(I2C_REG[ch_OK].STAT, I2CSTAT_MASTER_TX_MODE| I2CSTAT_ENABLE); //0xD0	
		// clear interrupt pending bit
		I2C_Outp8(I2C_REG[ch_OK].CON, (I2C_Inp8(I2C_REG[ch_OK].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR);
		// wait till the stop condition takes effect.
		while( I2C_Inp8(I2C_REG[ch_OK].STAT) & I2CSTAT_START );
		
		
		for(uCnt = uMin; uCnt < uMax; uCnt++)
		{
			IIC_DBG_PRINT(DBG_Basic,"\n CH %d : Writing into EEPROM... (%x) <= %x",ch_OK, uCnt, uCnt);
			iRetval = I2C_Ch2_Master_Polling_Write_EEPROM(ch_OK, cSlaveAddr, (u8) uCnt, (s32) uCnt);
			if( iRetval != OK )
				break;
			
			Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
		}
	}
	
	for(ch = I2C_CH0; ch <= I2C_CH_HDMI_DDC; ch++)
	{
		// send stop bit
		I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_MASTER_TX_MODE| I2CSTAT_ENABLE); //0xD0	
		// clear interrupt pending bit
		I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR);
		// wait till the stop condition takes effect.
		while( I2C_Inp8(I2C_REG[ch].STAT) & I2CSTAT_START );
	}		
	
	I2C_Close(ch0);
	I2C_Close(ch1);

	IIC_DBG_PRINT(DBG_Basic,"\n [I2C BUS ARBITRATION TEST COMPLETE !!! ]\n");
	
	return	ret;
}
#endif/* #ifndef v210_I2C_FPGA */

/**
* {I2C_CH0_Performance_Test}
*
* @brief			CH 0 Read / Write Performance Test in EEPROM in 100KHz & 400KHz.
* 
* @param[in]	void
* @return		u8 
* @version		2009.09.04
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8 I2C_CH0_Performance_Test(void)
{
	u8	ret = TRUE;
	
	gbI2C_CH0_Performance_Test = TRUE;
	
	ret = I2C_Ch0_Master_Polling_EEPROM_Test();

	gbI2C_CH0_Performance_Test = FALSE;
	return	ret;
}

/**
* {I2C_CH0_Aging_Test}
*
* @brief			CH 0 Read / Write Aging Test in EEPROM.
* 
* @param[in]	void
* @return		u8 
* @version		2009.09.29
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
u8 I2C_CH0_Aging_Test(void)
{
	u8	ret = TRUE;

	while(1)
	{
		ret = I2C_Ch0_Master_Polling_EEPROM_Test();
		if( ret != OK )
			break;
	};
	
	return	ret;
}
/*******************************************************************************
 * End of file Iic_test.c
 ******************************************************************************/ 
