/**************************************************************************************
*
*	Project Name : S5PV210 FPGA Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for verifying functions of the S5PV210 FPGA
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------
* 
*	File Name : iic.c
*  
*	File Description : This file implements the API functons for IIC
*
*	Author	: Jongseok,Park
*	Dept. : AP Development Team
*	Created Date : 2009/04/16
*	Version : 0.0 
*
*	History
*	- Version 0.1  : (2009/04/16) 
*         -> Created
**************************************************************************************/

#include "iic.h"
#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "gpio.h"
#include "intc.h"
#include "def.h"
#include "sysc.h"

/*******************************************************************************
* DEFINES
*******************************************************************************/

/*******************************************************************************
* GLOBAL VARIABLES
*******************************************************************************/
static volatile u8 *g_PcIIC_BUFFER;
static volatile u32 g_uIIC_PT;
static volatile u32 g_uIIC_DATALEN;	
static volatile u32 g_cIIC_STAT0;
static volatile u32 g_cIIC_SlaveRxDone;
static volatile u32 g_cIIC_SlaveTxDone;

static volatile u32 g_cIIC_AckCheck;	//jspark
static volatile u32 g_cIIC_StopCheck; 	//jspark
static volatile u32 g_uIIC_StopGeneration;

st_IIC_REG IIC_REG[3];
u32 	IIC_SFR_List[5*4] = { 
	rIICCON0, 			rIICSTAT0, 				rIICADD0,  			rIICDS0, 			rIICLC0,
	rIICCON_HDMI_DDC,	rIICSTAT_HDMI_DDC,	rIICADD_HDMI_DDC,	rIICDS_HDMI_DDC,	rIICLC_HDMI_DDC,
	rIICCON2, 			rIICSTAT2, 				rIICADD2,  			rIICDS2, 			rIICLC2,
	rIICCON_HDMI_PHY,	rIICSTAT_HDMI_PHY,		rIICADD_HDMI_PHY,	rIICDS_HDMI_PHY,	rIICLC_HDMI_PHY
};

//  .. ٸ function ; ϳ...Ƿ  for v210 FPGA (20093.22 )
static u32 	g_PCLK = 20000000;
//static u32 	g_HCLK=20000000;
//static u32 	g_HCLKx2=20000000*2;


u32		gI2c_CH_Index = 0;
st_IIC_REG I2C_REG[I2C_CH_NUM_END];
u32 	I2C_SFR_List[5*I2C_CH_NUM_END] = {
	/* CH 0 */
	rI2CCON0, 			rI2CSTAT0, 				rI2CADD0,  			rI2CDS0, 			rI2CLC0,
	/* HDMI PHY */
	rI2CCON_HDMI_DDC,	rI2CSTAT_HDMI_DDC,	rI2CADD_HDMI_DDC,	rI2CDS_HDMI_DDC,	rI2CLC_HDMI_DDC,
	/* CH 2 */
	rI2CCON2, 			rI2CSTAT2, 				rI2CADD2,  			rI2CDS2, 			rI2CLC2,
	/* HDMI DDC */	
	rI2CCON_HDMI_PHY,	rI2CSTAT_HDMI_PHY,	rI2CADD_HDMI_PHY,	rI2CDS_HDMI_PHY,	rI2CLC_HDMI_PHY
};

/*******************************************************************************
* FUNCTION PROTOTYPES
*******************************************************************************/
void IIC_sfr_mapping_Init(u32 ch)
{
	//IIC register struct Inital
	IIC_REG[ch].CON = IIC_SFR_List[ch*5 + 0];
	IIC_REG[ch].STAT = IIC_SFR_List[ch*5+ 1];
	IIC_REG[ch].ADD	= IIC_SFR_List[ch*5+ 2];
	IIC_REG[ch].DS	= IIC_SFR_List[ch*5+ 3];
	IIC_REG[ch].LC	= IIC_SFR_List[ch*5+ 4];
}

//////////
// Function Name : Isr_IIC
// Function Description : This function is Interrupt Service Routine of IIC
//					  when interrupt occurs, check IICSTAT, find mode and operate it
// Input : NONE
// Output : NONE
// Version : v0.1
void __irq Isr_IIC0( void)
{
	u32 uTemp0 = 0;
	u8 cCnt;

	g_cIIC_STAT0 = I2C_Inp32(rIICSTAT0);

	switch( (g_cIIC_STAT0>>6)&0x3) 
	{
		case SlaveRX	:	
			//UART_Printf("SlaveRx (Interrupt Mode) g_uIIC_PT Count: %d\n",g_uIIC_PT);			

			if(g_uIIC_PT<101)		// Edited by SOP on 2008/05/23 				
			{
				//UART_Printf("SlaveRx (Interrupt Mode) g_uIIC_PT Count: %d\n",g_uIIC_PT);
				g_PcIIC_BUFFER[g_uIIC_PT++]=Inp8(rIICDS0);
				Delay(100);
				
				uTemp0 = I2C_Inp32(rIICCON0);
				uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
				uTemp0 &= ~(1<<7);			// Disable Ack generation, Added by SOP on 04/11/2008
				Outp32(rIICCON0,uTemp0);
				
				if(g_uIIC_PT>100) 
				{
					I2C_Outp8(rIICSTAT0,0x0);			// Stop Int
					
					uTemp0 = I2C_Inp32(rIICCON0);
					uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
					Outp32(rIICCON0,uTemp0);

					g_cIIC_SlaveRxDone = 1;

					Delay(1);					//	wait until Stop Condition is in effect
					
					//UART_Printf("*** IIC0 ISR SlaveRX\n");		// For Debugging	
					break;
				}
				break;
			}
			
		case SlaveTX	:		
			if(g_uIIC_PT>100)
			{
				Outp32(rIICSTAT0,0xd0);		//	Stop Master Tx condition, ACK flag clear

				uTemp0 = I2C_Inp32(rIICCON0);
				uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
				Outp32(rIICCON0,uTemp0);

				g_cIIC_SlaveTxDone = 1;

				Delay(1);					//	wait until Stop Condition is in effect
				break;
			}

			I2C_Outp8(rIICDS0,g_PcIIC_BUFFER[g_uIIC_PT++]);
			for(cCnt=0;cCnt<10;cCnt++);	//	for setup time (rising edge of IICSCL)
				Delay(100);
			
			uTemp0 = I2C_Inp32(rIICCON0);
			uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
			Outp32(rIICCON0,uTemp0);
			
			//UART_Printf("*** IIC0 ISR SlaveTX\n");		// For Debugging				
			break;

		case MasterRX:
			//UART_Printf("*** Entering IIC0 ISR MasterRX\n");		// For Debugging

			//UART_Printf("["); //jspark for test						
			if (g_uIIC_PT>0) 	 //ó slave address ACK̹Ƿ.. ̰  Ƕ ..
			{
				g_PcIIC_BUFFER[g_uIIC_PT-1] = I2C_Inp32(rIICDS0);
				//UART_Printf("D"); //jspark for test						
			}
		
			if (g_uIIC_PT >= g_uIIC_DATALEN)	 //	STOP condition
			{
				//uTemp0 = I2C_Inp32(rIICCON0);
				//uTemp0 &= ~(1<<7);			// 	Disable Ack generation
				//Outp32(rIICCON0,uTemp0);	// 
				Outp32(rIICSTAT0,0x90);		//	Stop Master Rx condition
			}
			else if (g_uIIC_PT==(g_uIIC_DATALEN-1))
			{
				//slave address   next data reading NO ACK߻ϱ  ̸ disableŴ
				// Disable Ack generation (for EEPROM reading)			
				Outp32(rIICCON0,I2C_Inp32(rIICCON0)&( ~(1<<7)));	
			}
			
			g_uIIC_PT++;
			g_cIIC_AckCheck ++;		
			
			uTemp0 = I2C_Inp32(rIICCON0);
			uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
			Outp32(rIICCON0,uTemp0);
			//UART_Printf("]"); //jspark for test			
			break;


		case MasterTX:
			
			if (g_uIIC_PT<g_uIIC_DATALEN)
			{
				Outp32(rIICDS0,g_PcIIC_BUFFER[g_uIIC_PT]);
			}
			else
			{
				uTemp0 = I2C_Inp32(rIICCON0);
				uTemp0 &= ~(1<<7);			// Disable Ack generation
				Outp32(rIICCON0,uTemp0);	// 
			
				if (g_uIIC_StopGeneration)
				{
					//No data to TX => we shoud stop here..
					Outp32(rIICSTAT0,0xd0);		// Stop Master Tx condition, ACK flag clear
					//UART_Printf("Stop"); //jspark for test					
					g_cIIC_StopCheck++;
				}
				else
				{
				 	//Master TX stop, without STOP signal
					Outp32(rIICSTAT0,0xc0);		// Master Tx disable
				}	
			}				
			
			g_uIIC_PT++; 
			g_cIIC_AckCheck++;
			//UART_Printf("A(%d)",g_uIIC_PT); //jspark for test								

			//Clear pending bit to resume
			Outp32(rIICCON0, I2C_Inp32(rIICCON0)&(~(1<<4)) );
			
			//UART_Printf("*** Ended IIC ISR MasterTX (rIICCON0 =>0x28 or a8): 0x%02x\n", uTemp0);		// For Debugging			

			//Outp32(rIICSTAT0,0xf0);				// Master Tx Start.				
			break;
	}

	g_cIIC_STAT0 &= 0xf;
	INTC_ClearVectAddr();
}

//////////
// Function Name : Isr_IIC1
// Function Description : This function is Interrupt Service Routine of IIC
//					  when interrupt occurs, check IICSTAT, find mode and operate it
// Input : NONE
// Output : NONE
// Version : v0.1
void __irq Isr_IIC2( void)
{
	u32 uTemp0 = 0;
	u8 cCnt;

	//UART_Printf("(*2)");
	
	g_cIIC_STAT0 = I2C_Inp32(rIICSTAT2);
	
	switch( (g_cIIC_STAT0>>6)&0x3) 
	{
		case SlaveRX:
			//UART_Printf("SlaveRx (Interrupt Mode) g_uIIC_PT Count: %d\n",g_uIIC_PT);			

			if(g_uIIC_PT<101)	// Edited by SOP on 2008/05/23  				
			{
				//UART_Printf("SlaveRx (Interrupt Mode) g_uIIC_PT Count: %d\n",g_uIIC_PT);				
				g_PcIIC_BUFFER[g_uIIC_PT++]=Inp8(rIICDS2);
				Delay(100);
				
				uTemp0 = I2C_Inp32(rIICCON2);
				uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
				uTemp0 &= ~(1<<7);			// Disable Ack generation, Added by SOP on 04/11/2008				
				Outp32(rIICCON2,uTemp0);
				
				if(g_uIIC_PT>100)
				{
					I2C_Outp8(rIICSTAT2,0x0);			// Stop Int
					
					uTemp0 = I2C_Inp32(rIICCON2);
					uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
					Outp32(rIICCON2,uTemp0);

					g_cIIC_SlaveRxDone = 1;

					Delay(1);					//	wait until Stop Condition is in effect
					
					//UART_Printf("*** IIC1 ISR SlaveRX\n");		// For Debugging	
					break;
				}					
				break;
			}

		case SlaveTX:
			if(g_uIIC_PT>100)
			{
				Outp32(rIICSTAT2,0xd0);		//	Stop Master Tx condition, ACK flag clear

				uTemp0 = I2C_Inp32(rIICCON2);
				uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
				Outp32(rIICCON2,uTemp0);

				g_cIIC_SlaveTxDone = 1;

				Delay(1);					//	wait until Stop Condition is in effect
				break;
			}

			I2C_Outp8(rIICDS2,g_PcIIC_BUFFER[g_uIIC_PT++]);
			for(cCnt=0;cCnt<10;cCnt++);	//	for setup time (rising edge of IICSCL)
			Delay(100);
			
			uTemp0 = I2C_Inp32(rIICCON2);
			uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
			Outp32(rIICCON2,uTemp0);
			
			//UART_Printf("*** IIC1 ISR SlaveTX\n");		// For Debugging				
			break;

		case MasterRX:
			//UART_Printf("*** Entering IIC1 ISR MasterRX\n");		// For Debugging

			//UART_Printf("["); //jspark for test						
			if (g_uIIC_PT>0) 	 //ó slave address ACK̹Ƿ.. ̰  Ƕ ..
			{
				g_PcIIC_BUFFER[g_uIIC_PT-1] = I2C_Inp32(rIICDS2);
				//UART_Printf("D"); //jspark for test						
			}
		
			if (g_uIIC_PT >= g_uIIC_DATALEN)	 //	STOP condition
			{
				//uTemp0 = I2C_Inp32(rIICCON0);
				//uTemp0 &= ~(1<<7);			// 	Disable Ack generation
				//Outp32(rIICCON0,uTemp0);	// 
				Outp32(rIICSTAT2,0x90);		//	Stop Master Rx condition
			}
			else if (g_uIIC_PT==(g_uIIC_DATALEN-1))
			{
				//slave address   next data reading NO ACK߻ϱ  ̸ disableŴ
				// Disable Ack generation (for EEPROM reading)			
				Outp32(rIICCON2,I2C_Inp32(rIICCON2)&( ~(1<<7)));	
			}
			
			g_uIIC_PT++;
			g_cIIC_AckCheck ++;		
			
			uTemp0 = I2C_Inp32(rIICCON2);
			uTemp0 &= ~(1<<4);			//	Clear pending bit to resume
			Outp32(rIICCON2,uTemp0);
			//UART_Printf("]"); //jspark for test			
			break;


		case MasterTX:

			if (g_uIIC_PT<g_uIIC_DATALEN)
			{
				Outp32(rIICDS2,g_PcIIC_BUFFER[g_uIIC_PT]);
			}
			else
			{
				uTemp0 = I2C_Inp32(rIICCON2);
				uTemp0 &= ~(1<<7);			// Disable Ack generation
				Outp32(rIICCON2,uTemp0);	// 
			
				if (g_uIIC_StopGeneration)
				{
					//No data to TX => we shoud stop here..
					Outp32(rIICSTAT2,0xd0);		// Stop Master Tx condition, ACK flag clear
					//UART_Printf("Stop"); //jspark for test					
					g_cIIC_StopCheck++;
				}
				else
				{
				  	//Master TX stop, without STOP signal
					Outp32(rIICSTAT2,0xc0);		// Master Tx disable
				}				
			}
				
			
			g_uIIC_PT++; 
			g_cIIC_AckCheck++;
			//UART_Printf("A(%d)",g_uIIC_PT); //jspark for test								

			//Clear pending bit to resume
			Outp32(rIICCON2, I2C_Inp32(rIICCON2)&(~(1<<4)) );
			break;
	}

	g_cIIC_STAT0	&=0xf;
	INTC_ClearVectAddr();
}



//////////
// Function Name : IIC_Open
// Function Description : This function Set up VIC & IICCON with user's input frequency which determines uClkValue
// Input : ufreq	ufreq(Hz) = PCLK/16/uClkValue
// Output : NONE
// Version : v0.1
void IIC0_Open(u32 ufreq)		//	Hz order. freq(Hz) = PCLK/16/clk_divider
{
	u32	uSelClkSrc;
	u32	uClkValue;

	UART_Printf("\n*** IIC0 Setting Frequency: %d Hz\n",ufreq);

	INTC_SetVectAddr(NUM_I2C,Isr_IIC0);
	INTC_Enable(NUM_I2C);

	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_0,2);
	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_1,2);	
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_0,2); //pull up enable
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_1,2);

	if ((((g_PCLK>>4)/ufreq)-1)>0xf) 
	{
		uSelClkSrc	=	1;
		uClkValue	=	((g_PCLK>>9)/ufreq) -1;		//	PCLK/512/freq
		UART_Printf("*** PCLK= %.2fMHz,  SelClkSrc= %d (IICCLK=PCLK/512), ClkDivideValue= %d\n",(float)g_PCLK/1.0e6,uSelClkSrc,uClkValue);		
		UART_Printf("*** IICCLK= %d Hz,  IICSCL (Tx clock)=  %d Hz\n",(g_PCLK/512),((g_PCLK/512)/(uClkValue+1)));			
	} 
	else 
	{
		uSelClkSrc	=	0;
		uClkValue	=	((g_PCLK>>4)/ufreq) -1;		//	PCLK/16/freq
		UART_Printf("*** PCLK= %.2fMHz,  SelClkSrc= %d (IICCLK=PCLK/16), ClkDivideValue= %d\n",(float)g_PCLK/1.0e6,uSelClkSrc,uClkValue);
		UART_Printf("*** IICCLK= %d Hz,  IICSCL (Tx clock)=  %d Hz\n",(g_PCLK/16),((g_PCLK/16)/(uClkValue+1)));		
	}

	//Prescaler IICCLK=PCLK/16, Enable interrupt, Transmit clock value Tx clock=IICCLK/16
	Outp32(rIICCON0,(uSelClkSrc<<6) | (1<<5) | (uClkValue&0xf));
	Outp32(rIICADD0,0xc0);		//	Slave address = [7:1]

	Outp32(rIICSTAT0,0x10);		//	IIC bus data output enable(Rx/Tx)
	Outp32(rIICLC0,0x4);			//	SDA Filter Enable,delayed 0clks
	//Outp32(rIICLC0,0x5);		//	SDA Filter enable,delayed 5clks
	//Outp32(rIICLC0,0x6);		//	SDA Filter enable,delayed 10clks
	//Outp32(rIICLC0,0x7);		//	SDA Filter enable,delayed 15clks	

	 //jspark adding
	IIC_sfr_mapping_Init(IIC_CH0);	
}


//////////
// Function Name : IIC1_Open
// Function Description : This function Set up VIC & IICCON with user's input frequency which determines uClkValue
// Input : ufreq	ufreq(Hz) = PCLK/16/uClkValue
// Output : NONE
// Version : v0.1
void IIC2_Open( u32 ufreq)		//	Hz order. freq(Hz) = PCLK/16/clk_divider
{
	u32	uSelClkSrc;
	u32	uClkValue;

	UART_Printf("\n*** IIC1 Setting Frequency: %d Hz\n",ufreq);
	
	INTC_SetVectAddr(NUM_I2C_PMIC,Isr_IIC2);
	INTC_Enable(NUM_I2C_PMIC);

	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_4,2);
	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_5,2);	
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_4,2); //pull up enable
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_5,2);
	
	//Pause();		// For debugging
	
	if ((((g_PCLK>>4)/ufreq)-1)>0xf) 
	{
		uSelClkSrc	=	1;
		uClkValue	=	((g_PCLK>>9)/ufreq) -1;		//	PCLK/512/freq
		//uClkValue	=	((g_PCLK/512)/ufreq) -1;		//	PCLK/512/freq		
		UART_Printf("*** PCLK= %.2fMHz,  SelClkSrc= %d (IICCLK=PCLK/512)   ,ClkDivideValue= %d\n",(float)g_PCLK/1.0e6,uSelClkSrc,uClkValue);		
		UART_Printf("*** IICCLK= %d Hz,  IICSCL (Tx clock)=  %d Hz\n",(g_PCLK/512),((g_PCLK/512)/(uClkValue+1)));			
	} 
	else 
	{
		uSelClkSrc	=	0;
		uClkValue	=	((g_PCLK>>4)/ufreq) -1;		//	PCLK/16/freq
		//uClkValue	=	((g_PCLK/16)/ufreq) -1;		//	PCLK/16/freq		
		UART_Printf("*** PCLK= %.2fMHz,  SelClkSrc= %d (IICCLK=PCLK/16)   ,ClkDivideValue= %d\n",(float)g_PCLK/1.0e6,uSelClkSrc,uClkValue);		
		UART_Printf("*** IICCLK= %d Hz,  IICSCL (Tx clock)=  %d Hz\n",(g_PCLK/16),((g_PCLK/16)/(uClkValue+1)));		
	}

	Outp32(rIICCON2,(uSelClkSrc<<6) | (1<<5) | (uClkValue&0xf));
	Outp32(rIICADD2,0xc0);		//	Slave address = [7:1]
	Outp32(rIICSTAT2,0x10);		//	IIC bus data output enable(Rx/Tx)
	Outp32(rIICLC2,0x4);			//	SDA Filter Enable,delayed 0clks
	//Outp32(rIICLC1,0x5);		//	SDA Filter enable,delayed 5clks
	//Outp32(rIICLC1,0x6);		//	SDA Filter enable,delayed 10clks
	//Outp32(rIICLC1,0x7);		//	SDA Filter enable,delayed 15clks		


	 //jspark adding
	IIC_sfr_mapping_Init(IIC_CH2);
}


//////////
// Function Name : IIC_OpenPolling
// Function Description : This function Set up IICCON with user's input frequency which determines uClkValue
//                                  (without ISR set up,for polling mode)
// Input : ufreq	ufreq(Hz) = PCLK/16/uClkValue
// Output : NONE
// Version : v0.1
void IIC0_OpenPolling( u32 ufreq)		//	Hz order. freq(Hz) = PCLK/16/clk_divider
{
	u32	uSelClkSrc;
	u32	uClkValue;

	UART_Printf("\n*** IIC0 Setting Frequency: %d Hz\n",ufreq);

	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_0,2);
	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_1,2);	
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_0,2); //pull up enable
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_1,2);

	if ((((g_PCLK>>4)/ufreq)-1)>0xf) 
	{
		uSelClkSrc	=	1;
		uClkValue	=	((g_PCLK>>9)/ufreq) -1;		//	PCLK/512/freq
		UART_Printf("*** PCLK= %.2fMHz,  SelClkSrc= %d (IICCLK=PCLK/512)   ,ClkDivideValue= %d\n",(float)g_PCLK/1.0e6,uSelClkSrc,uClkValue);				
	} 
	else 
	{
		uSelClkSrc	=	0;
		uClkValue	=	((g_PCLK>>4)/ufreq) -1;		//	PCLK/16/freq
		UART_Printf("*** PCLK= %.2fMHz,  SelClkSrc= %d (IICCLK=PCLK/16)   ,ClkDivideValue= %d\n",(float)g_PCLK/1.0e6,uSelClkSrc,uClkValue);
	}

	UART_Printf("*** IICCLK= %d Hz,  IICSCL (Tx clock)=  %d Hz\n",(g_PCLK/16),((g_PCLK/16)/(uClkValue+1)));
		//Prescaler IICCLK=PCLK/16, Enable interrupt, Transmit clock value Tx clock=IICCLK/16
	Outp32(rIICCON0,(uSelClkSrc<<6) | (1<<5) | (uClkValue&0xf));
	Outp32(rIICADD0,0xc0);		//	Slave address = [7:1]
	Outp32(rIICSTAT0,0x10);		//	IIC bus data output enable(Rx/Tx)
	Outp32(rIICLC0,0x7);			//	SDA Filter Enable, delay 15 clocks

	 //jspark adding
	IIC_sfr_mapping_Init(IIC_CH0);
}

//////////
// Function Name : IIC1_OpenPolling
// Function Description : This function Set up IICCON with user's input frequency which determines uClkValue
//                                  (without ISR set up,for polling mode)
// Input : ufreq	ufreq(Hz) = PCLK/16/uClkValue
// Output : NONE
// Version : v0.1
void IIC2_OpenPolling( u32 ufreq)		//	Hz order. freq(Hz) = PCLK/16/clk_divider
{
	u32	uSelClkSrc;
	u32	uClkValue;

	UART_Printf("\n*** IIC1 Setting Frequency: %d Hz\n",ufreq);

	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_4, 2);
	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_5, 2);	
	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_4, 2); //pull up enable
	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_5, 2);
	

	if ((((g_PCLK>>4)/ufreq)-1)>0xf) 
	{
		uSelClkSrc	=	1;
		uClkValue	=	((g_PCLK>>9)/ufreq) -1;		//	PCLK/512/freq
		UART_Printf("*** PCLK= %.2fMHz,  SelClkSrc= %d (IICCLK=PCLK/512)   ,ClkDivideValue= %d\n",(float)g_PCLK/1.0e6,uSelClkSrc,uClkValue);		
	} 
	else 
	{
		uSelClkSrc	=	0;
		uClkValue	=	((g_PCLK>>4)/ufreq) -1;		//	PCLK/16/freq
		UART_Printf("*** PCLK= %.2fMHz,  SelClkSrc= %d (IICCLK=PCLK/16)   ,ClkDivideValue= %d\n",(float)g_PCLK/1.0e6,uSelClkSrc,uClkValue);		
	}
	
	UART_Printf("*** IICCLK= %d Hz,  IICSCL (Tx clock)=  %d Hz\n",(g_PCLK/16),((g_PCLK/16)/(uClkValue+1)));
		//Prescaler IICCLK=PCLK/16, Enable interrupt, Transmit clock value Tx clock=IICCLK/16
	Outp32(rIICCON2,(uSelClkSrc<<6) | (1<<5) | (uClkValue&0xf));
	Outp32(rIICADD2,0xc0);		//	Slave address = [7:1]
	Outp32(rIICSTAT2,0x10);		//	IIC bus data output enable(Rx/Tx)
	Outp32(rIICLC2,0x7);			//	SDA Filter disable

	 //jspark adding
	IIC_sfr_mapping_Init(IIC_CH2);
}


//////////
// Function Name : IIC_Close
// Function Description : This function disable IIC
// Input : NONE
// Output : NONE
// Version : v0.1
void IIC0_Close(void)
{
	INTC_Disable(NUM_I2C);
	Outp32(rIICSTAT0,0x0);                		//	IIC bus data output disable(Rx/Tx)
	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_0,0);
	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_1,0);	
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_0,0);
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_1,0);
}

//////////
// Function Name : IIC1_Close
// Function Description : This function disable IIC
// Input : NONE
// Output : NONE
// Version : v0.1
void IIC2_Close(void)
{
	INTC_Disable(NUM_I2C_PMIC);
	Outp32(rIICSTAT2,0x0);                		//	IIC bus data output disable(Rx/Tx)

	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_4, 0);
	GPIO_SetFunctionEach(eGPIO_D1, eGPIO_5, 0);	
	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_4, 0);
	GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_5, 0);
}



//////////
// Function Name : IIC_SetWrite
// Function Description : This function sets up write mode with 7-bit addresses
// Input :  cSlaveAddr [8bit SlaveDeviceAddress], 
//		  pData[Data which you want to write], 
//		  uDataLen [Data Length]
// Output : NONE
// Version : v0.1
void IIC0_SetWrite( u8 cSlaveAddr,  u8 *pData, u32 uDataLen)
{
	u32 uTmp1;

	uTmp1 = I2C_Inp32(rIICSTAT0);

	while((uTmp1&(1<<5)) || (uTmp1&(1<<3)))	//	Wait until IIC bus is free or Arbitration successful.
	{
		uTmp1 = I2C_Inp32(rIICSTAT0);
	}

	g_PcIIC_BUFFER = pData;
	g_uIIC_PT	 = 0;
	g_uIIC_DATALEN = uDataLen;

	g_cIIC_AckCheck=0;

	uTmp1 = I2C_Inp32(rIICCON0);
	uTmp1 |= (1<<7);						// ACK Enable
	Outp32(rIICCON0,uTmp1);				// Ack generation Enable
	Outp32(rIICDS0,cSlaveAddr);
	Outp32(rIICSTAT0,0xf0);				// Master Tx Start.

	IIC_Wait();		// Added by SOP on 2008/04/10
}


//////////
// Function Name : IIC1_SetWrite
// Function Description : This function sets up write mode with 7-bit addresses
// Input :  cSlaveAddr [8bit SlaveDeviceAddress], 
//		  pData[Data which you want to write], 
//		  uDataLen [Data Length]
// Output : NONE
// Version : v0.1
void IIC2_SetWrite( u8 cSlaveAddr,  u8 *pData, u32 uDataLen)
{
	u32 uTmp1;

	uTmp1 = I2C_Inp32(rIICSTAT2);
	while((uTmp1&(1<<5)) || (uTmp1&(1<<3)))	//	Wait until IIC bus is free or Arbitration successful.
	{
		uTmp1 = I2C_Inp32(rIICSTAT2);
	}

	g_PcIIC_BUFFER	=	pData;
	g_uIIC_PT		=	0;
	g_uIIC_DATALEN	=	uDataLen;

	uTmp1 = I2C_Inp32(rIICCON2);
	uTmp1 |= (1<<7);						// ACK Enable
	Outp32(rIICCON2,uTmp1);				// Ack generation Enable
	Outp32(rIICDS2,cSlaveAddr);
	Outp32(rIICSTAT2,0xf0);				// Master Tx Start.
	IIC_Wait();		// Added by SOP on 2008/04/10
}


//////////
// Function Name : IIC_SetRead
// Function Description : This function sets up Read mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		  pData[Data which you want to read], 
//		  uDataLen [Data Length]
// Output : NONE
// Version : v0.1
void IIC0_SetRead(  u8 cSlaveAddr,  u8 *pData, u32 uDataLen)
{
	u32 uTmp0;
	u32 uTmp3;
	
	uTmp0= I2C_Inp32(rIICSTAT0);
	
	while((uTmp0&(1<<5)) || (uTmp0&(1<<3)))	//	Wait until IIC bus is free.
	{
		uTmp0 = I2C_Inp32(rIICSTAT0);	
	}	

	g_PcIIC_BUFFER	=	pData;
	g_uIIC_PT		=	0;
	g_uIIC_DATALEN	=	uDataLen;

	uTmp3 = I2C_Inp32(rIICCON0);
	uTmp3 |= (1<<7);
	Outp32(rIICCON0,uTmp3);				//	Ack generation Enable
	Outp32(rIICDS0,cSlaveAddr);

	Outp32(rIICSTAT0,0xB0);				//	Master Rx Start.

	IIC_Wait();							// Added by SOP on 2008/04/10
}



//////////
// Function Name : IIC1_SetRead
// Function Description : This function sets up Read mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		  pData[Data which you want to read], 
//		  uDataLen [Data Length]
// Output : NONE
// Version : v0.1
void IIC2_SetRead(  u8 cSlaveAddr,  u8 *pData, u32 uDataLen)
{
	u32 uTmp0;
	u32 uTmp3;

	uTmp0= I2C_Inp32(rIICSTAT2);
	
	while((uTmp0&(1<<5)) || (uTmp0&(1<<3)))	//	Wait until IIC bus is free.
	{
		uTmp0 = I2C_Inp32(rIICSTAT2);	
	}	

	g_PcIIC_BUFFER	=	pData;
	g_uIIC_PT		=	0;
	g_uIIC_DATALEN	=	uDataLen;

	uTmp3 = I2C_Inp32(rIICCON2);
	uTmp3 |= (1<<7);					//	Ack generation Enable
	Outp32(rIICCON2,uTmp3);			
	Outp32(rIICDS2,cSlaveAddr);

	Outp32(rIICSTAT2,0xB0);				//	Master Rx Start.

	IIC_Wait();							// Added by SOP on 2008/04/10
}


//////////
// Function Name : IIC_Wait
// Function Description : This function waits until the command takes effect
//											But not for IIC bus free
// Input : NONE 
// Output : NONE
// Version : v0.1
void IIC_Wait( void)						//	Waiting for the command takes effect.
{										//	But not for IIC bus free.
	u32	loopcnt=0;
	while(g_uIIC_PT<=g_uIIC_DATALEN)
	{
		loopcnt++;
		if (loopcnt > 0x00800000) { 
			UART_Printf("***ڡڡڡ IIC WAIT ERROR ڡڡڡ\n");
			break;
			}
	};
}

//////////
// Function Name : IIC_Status
// Function Description : This function returns IIC Status Register value at last interrupt occur
// Input : NONE
// Output : NONE
// Version : v0.1
u8 IIC_Status( void)						//	Return IIC Status Register value at last interrupt occur.
{
	return	g_cIIC_STAT0;
}

//////////
// Function Name : IIC_Write
// Function Description : This function STARTs up write mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 cAddr[8bit Address where you want to write], 
//		 pData[Data which you want to write]
// Output : NONE
// Version : v0.1
void IIC0_Write(u8 cSlaveAddr, u8 cAddr, u8 cData)
{
	u8 cD[2];
	
	cD[0]=cAddr;
	cD[1]=cData;


#if 0 //jspark test	
	do					//	Polling for an ACK signal from Serial EEPROM.
	{					
		Delay(30);		// Need to time for EEPROM
		IIC0_SetWrite(cSlaveAddr, NULL, 0);
	} while(IIC_Status()&0x1);
#else
		Delay(30);		// Need to time for EEPROM
#endif

	IIC0_SetWrite(cSlaveAddr, cD, 2);	
}


//////////
// Function Name : IIC_Write16
// Function Description : This function STARTs up write mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 cAddr[8bit Address where you want to write], 
//		 pData[16bit Data which you want to write]
// Output : NONE
// Version : v0.1
void IIC0_Write16(u8 cSlaveAddr, u8 cAddr, u16 cData)
{
	u8 cD[3];

	while(!GetKey()); 

	cD[0]=cAddr;
	cD[1]=(cData & 0xff00) >> 8;			//MSB First Data
	cD[2]=cData & 0xff;					//LSB Last Data
#if 0 //jspark test	
	do
	{									//	Polling for an ACK signal from SerialEEPROM.
		IIC_SetWrite(cSlaveAddr, NULL, 0);
		IIC_Wait();
	}while(IIC_Status()&0x1);
#endif	
	IIC0_SetWrite(cSlaveAddr, cD, 3);
}

//////////
// Function Name : IIC_WriteA16D16
// Function Description : This function STARTs up write mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 cAddr[16bit Address where you want to write], 
//		 pData[16bit Data which you want to write]
// Output : NONE
// Version : v0.1
// Date: 2008/06/16
void IIC0_WriteA16D16(u8 cSlaveAddr, u16 cAddr, u16 cData)
{
	u8 cD[4];

	cD[0]=(cAddr & 0xff00) >> 8;			//MSB First Address
	cD[1]=cAddr & 0xff;					//LSB Last Address	
	cD[2]=(cData & 0xff00) >> 8;			//MSB First Data
	cD[3]=cData & 0xff;					//LSB Last Data

	do
	{									//	Polling for an ACK signal from SerialEEPROM.
		IIC0_SetWrite(cSlaveAddr, NULL, 0);
		IIC_Wait();
	}while(IIC_Status()&0x1);
	
	IIC0_SetWrite(cSlaveAddr, cD, 4);
}

//////////
// Function Name : IIC1_Write
// Function Description : This function STARTs up write mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 cAddr[8bit Address where you want to write], 
//		 pData[Data which you want to write]
// Output : NONE
// Version : v0.1
void IIC2_Write(u8 cSlaveAddr, u16 cAddr, u8 cData)
{
	u8 cD[3];

	cD[0]=(cAddr & 0xff00) >> 8;			//MSB First Address
	cD[1]=cAddr & 0xff;					//LSB Last Address
	cD[2]=cData;

	do					//	Polling for an ACK signal from Serial EEPROM.
	{					
		Delay(30);		// Need to time for EEPROM
		IIC2_SetWrite(cSlaveAddr, NULL, 0);
	} while(IIC_Status()&0x1);

	IIC2_SetWrite(cSlaveAddr, cD, 3);	
}


//////////
// Function Name : IIC_Read
// Function Description : This function STARTs up read mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 cAddr [8bit Address where you want to read], 
//		 cData [pointer of Data which you want to read]
// Output : NONE
// Version : v0.1
void IIC0_Read(u8 cSlaveAddr,u8 cAddr,u8 *cData)
{
	do		//	Polling for an ACK signal from SerialEEPROM.
	{					
		IIC0_SetWrite(cSlaveAddr, NULL, 0);
	} while(IIC_Status()&0x1);

	IIC0_SetWrite( cSlaveAddr, &cAddr, 1);			// following EEPROM random address access procedure	
	IIC0_SetRead( cSlaveAddr, cData, 1);
}

//////////
// Function Name : IIC_Read16
// Function Description : This function STARTs up read mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 cAddr [8bit Address where you want to read], 
// Output : u16 : Data which you want to read
// Version : v0.1
// Date: 2008/09/25

u16 IIC0_Read16(u8 cSlaveAddr, u8 cAddr)
//void IIC_Read16(u8 cSlaveAddr, u8 cAddr, u16 *cData)
{
	u8 cData[2] = {0,0};

	while(!GetKey()); 

#if 0 //jspark test	
	do		//	Polling for an ACK signal from Slave Device.
	{		
		IIC_SetWrite(cSlaveAddr, NULL, 0);
	} while(IIC_Status()&0x1);
#endif

	IIC0_SetWrite( cSlaveAddr, &cAddr, 1);	
	Delay(100);				
	IIC0_SetRead( cSlaveAddr, &cData[0], 2);	
	return (((0xff & cData[0]) << 8) | ((0xff & cData[1]) << 0));
}


//////////
// Function Name : IIC_ReadA16D16
// Function Description : This function STARTs up read mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 cAddr [16bit Address where you want to read], 
//		 cData [pointer of Data which you want to read]
// Output : NONE
// Version : v0.1
// Date: 2008/06/16 ~ 2008/07/22
void IIC0_ReadA16D16(u8 cSlaveAddr, u16 cAddr, u8 *cData)
{
	u8 cD[2];

	cD[0]=(cAddr & 0xff00) >> 8;			//MSB First Address
	cD[1]=cAddr & 0xff;					//LSB Last Address

	do		//	Polling for an ACK signal from Slave Device.
	{		
		IIC0_SetWrite(cSlaveAddr, NULL, 0);
	} while(IIC_Status()&0x1);

	IIC0_SetWrite( (cSlaveAddr&0xfe), cD, 2);			
	IIC0_SetRead( cSlaveAddr, cData, 2);		
}


//////////
// Function Name : IIC_Read
// Function Description : This function STARTs up read mode with 7-bit addresses
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 cAddr [16bit Address where you want to read], 
//		 cData [pointer of Data which you want to read]
// Output : NONE
// Version : v0.1
void IIC2_Read(u8 cSlaveAddr, u16 cAddr, u8 *cData)
{
	u8 cD[2];
	
	cD[0]=(cAddr & 0xff00) >> 8;			//MSB First Address
	cD[1]=cAddr & 0xff;					//LSB Last Address

	do		//	Polling for an ACK signal from SerialEEPROM.
	{		
		IIC2_SetWrite(cSlaveAddr, NULL, 0);
	} while(IIC_Status()&0x1);

//	IIC1_SetWrite( cSlaveAddr, cD, 2);			// following EEPROM random address access procedure	
	IIC2_SetWrite( cSlaveAddr, &cD[0], 2);			// following EEPROM random address access procedure	
	IIC2_SetRead( cSlaveAddr, cData, 1);	
}


//////////
// Function Name : IIC_MasterWrP
// Function Description : This function do MasterTx mode by polling operation
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Tx]
// Output : NONE
// Version : v0.1
void IIC0_MasterWrP(u8 cSlaveAddr,u8 * pData)
{
	u32 uTmp0;
	u32 uTmp1;
	u8 cCnt;
	s32 sDcnt = 100;
	u32 uPT = 0;
	
	uTmp0 = I2C_Inp32(rIICSTAT0);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT0);			

	uTmp1 = I2C_Inp32(rIICCON0);
	uTmp1 |= (1<<7);
	Outp32(rIICCON0,uTmp1);				//	Ack generation Enable
	Outp32(rIICDS0,cSlaveAddr);
	Outp32(rIICSTAT0,0xf0);				//	Master Tx Start.

	while(!(sDcnt == -1))
	{
		if(Inp8(rIICCON0)&0x10)			//until Int Pending
		{
			if((sDcnt--) == 0)
			{
				Outp32(rIICSTAT0,0xd0);		//	Stop Master Tx condition, ACK flag clear
				
				uTmp0 = I2C_Inp32(rIICCON0);
				uTmp0 &= ~(1<<4);			//	Clear pending bit to resume
				Outp32(rIICCON0,uTmp0);

				Delay(1);					//	wait until Stop Condition is in effect
				break;
			}

			I2C_Outp8(rIICDS0,pData[uPT++]);
			for(cCnt=0;cCnt<10;cCnt++);		//	for setup time (rising edge of IICSCL)
			
			uTmp0 = I2C_Inp32(rIICCON0);
			uTmp0 &= ~(1<<4);				//	Clear pending bit to resume
			Outp32(rIICCON0,uTmp0);				
		}
	}
}


//////////
// Function Name : IIC1_MasterWrP
// Function Description : This function do MasterTx mode by polling operation
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Tx]
// Output : NONE
// Version : v0.1
void IIC2_MasterWrP(u8 cSlaveAddr,u8 * pData)
{
	u32 uTmp0;
	u32 uTmp1;
	u8 cCnt;
	s32 sDcnt = 100;
	u32 uPT = 0;
	
	uTmp0 = I2C_Inp32(rIICSTAT2);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT2);			

	uTmp1 = I2C_Inp32(rIICCON2);
	uTmp1 |= (1<<7);
	Outp32(rIICCON2,uTmp1);				//	Ack generation Enable
	Outp32(rIICDS2,cSlaveAddr);
	Outp32(rIICSTAT2,0xf0);				//	Master Tx Start.

	while(!(sDcnt == -1))
	{
		if(Inp8(rIICCON2)&0x10)			//until Int Pending
		{
			if((sDcnt--) == 0)
			{
				Outp32(rIICSTAT2,0xd0);		//	Stop Master Tx condition, ACK flag clear
				
				uTmp0 = I2C_Inp32(rIICCON2);
				uTmp0 &= ~(1<<4);			//	Clear pending bit to resume
				Outp32(rIICCON2,uTmp0);

				Delay(1);					//	wait until Stop Condition is in effect
				break;
			}

			I2C_Outp8(rIICDS2,pData[uPT++]);
			for(cCnt=0;cCnt<10;cCnt++);		//	for setup time (rising edge of IICSCL)
			
			uTmp0 = I2C_Inp32(rIICCON2);
			uTmp0 &= ~(1<<4);				//	Clear pending bit to resume
			Outp32(rIICCON2,uTmp0);				
		}
	}
}


//////////
// Function Name : IIC_MasterRdP
// Function Description : This function do MasterRd mode by polling operation
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Rx]
// Output : NONE
// Version : v0.1
void IIC0_MasterRdP(u8 cSlaveAddr,u8 * pData)
{
	u32 uTmp0;
	u32 uTmp1;
	u8 cCnt;
	
	uTmp0 = I2C_Inp32(rIICSTAT0);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT0);			

	uTmp1 = I2C_Inp32(rIICCON0);
	uTmp1 |= (1<<7);
	Outp32(rIICCON0,uTmp1);				//	Ack generation Enable

	Outp32(rIICDS0,cSlaveAddr);

	Outp32(rIICSTAT0,0xB0);				//	Master Rx Start.

	while((Inp8(rIICSTAT0)&0x1));

	cCnt=0;
	
	while(cCnt<101) 
	{
		if(Inp8(rIICCON0)&0x10) 
		{
			pData[cCnt]=Inp8(rIICDS0);
			cCnt++;
			
			uTmp0 = I2C_Inp32(rIICCON0);
			uTmp0 &= ~(1<<4);			//	Clear pending bit to resume
			Outp32(rIICCON0,uTmp0);				
		}
	}
	I2C_Outp8(rIICSTAT0,0x90);				// stop bit gen.
}


//////////
// Function Name : IIC1_MasterRdP
// Function Description : This function do MasterRd mode by polling operation
// Input : cSlaveAddr [8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Rx]
// Output : NONE
// Version : v0.1
void IIC2_MasterRdP(u8 cSlaveAddr,u8 * pData)
{
	u32 uTmp0;
	u32 uTmp1;
	u8 cCnt;
	
	uTmp0 = I2C_Inp32(rIICSTAT2);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT2);			

	uTmp1 = I2C_Inp32(rIICCON2);
	uTmp1 |= (1<<7);
	Outp32(rIICCON2,uTmp1);				//	Ack generation Enable

	Outp32(rIICDS2,cSlaveAddr);

	Outp32(rIICSTAT2,0xB0);				//	Master Rx Start.

	while((Inp8(rIICSTAT2)&0x1));

	cCnt=0;
	
	while(cCnt<101) 
	{
		if(Inp8(rIICCON2)&0x10) 
		{
			pData[cCnt]=Inp8(rIICDS2);
			cCnt++;
			
			uTmp0 = I2C_Inp32(rIICCON2);
			uTmp0 &= ~(1<<4);			//	Clear pending bit to resume
			Outp32(rIICCON2,uTmp0);				
		}
	}
	I2C_Outp8(rIICSTAT2,0x90);				// stop bit gen.
}


//////////
// Function Name : IIC_SlaveRdP
// Function Description : This function do SlaveRx mode by polling operation
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Rx]
// Output : NONE
// Version : v0.1
void IIC0_SlaveRdP(u8 *pSlaveAddr,u8 *pData)
{
	u32 uTmp0;
	u32 uTmp1;
	u8 cCnt;
	
	uTmp0 = I2C_Inp32(rIICSTAT0);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT0);			

	uTmp1 = I2C_Inp32(rIICCON0);
	uTmp1 |= (1<<7);
	Outp32(rIICCON0,uTmp1);				//	Ack generation Enable

	uTmp0 = I2C_Inp32(rIICSTAT0);
	uTmp0 &= ~(1<<4);
	Outp32(rIICSTAT0,uTmp0);				// Disable Rx/Tx for setting SlaveAddr
	I2C_Outp8(rIICADD0,*pSlaveAddr);
	Outp32(rIICSTAT0,0x10);				//	Slave Rx Start.
	UART_Printf("Wait for Slave Addr\n");
	
	cCnt=0;
	
	while(cCnt<101) 
	{
		if(Inp8(rIICCON0)&0x10) 
		{
			pData[cCnt]=Inp8(rIICDS0);
			cCnt++;
			
			uTmp0 = I2C_Inp32(rIICCON0);
			uTmp0 &= ~(1<<4);			//	Clear pending bit to resume
			Outp32(rIICCON0,uTmp0);		
		}
	}
	*pSlaveAddr = pData[0];
	I2C_Outp8(rIICSTAT0,0x0);
}


//////////
// Function Name : IIC1_SlaveRdP
// Function Description : This function do SlaveRx mode by polling operation
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Rx]
// Output : NONE
// Version : v0.1
void IIC2_SlaveRdP(u8 *pSlaveAddr,u8 *pData)
{
	u32 uTmp0;
	u32 uTmp1;
	u8 cCnt;
	
	uTmp0 = I2C_Inp32(rIICSTAT2);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT2);			

	uTmp1 = I2C_Inp32(rIICCON2);
	uTmp1 |= (1<<7);
	Outp32(rIICCON2,uTmp1);				//	Ack generation Enable

	uTmp0 = I2C_Inp32(rIICSTAT2);
	uTmp0 &= ~(1<<4);
	Outp32(rIICSTAT2,uTmp0);				// Disable Rx/Tx for setting SlaveAddr
	I2C_Outp8(rIICADD2,*pSlaveAddr);
	Outp32(rIICSTAT2,0x10);				//	Slave Rx Start.
	UART_Printf("Wait for Slave Addr\n");
	
	cCnt=0;

	while(cCnt<101) 
	{
		if(Inp8(rIICCON2)&0x10) 
		{
			pData[cCnt]=Inp8(rIICDS2);
			cCnt++;
			
			uTmp0 = I2C_Inp32(rIICCON2);
			uTmp0 &= ~(1<<4);			//	Clear pending bit to resume
			Outp32(rIICCON2,uTmp0);		
		}
	}
	*pSlaveAddr = pData[0];
	I2C_Outp8(rIICSTAT2,0x0);
}


//////////
// Function Name : IIC_SlaveWrP
// Function Description : This function do SlaveTx mode by polling operation
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Tx]
// Output : NONE
// Version : v0.1
void IIC0_SlaveWrP(u8 *pSlaveAddr,u8 *pData)
{
	u32 uTmp0;
	u32 uTmp1;
	u8 cCnt;
	s32 sDcnt = 100;
	u32 uPT = 0;

	uTmp0 = I2C_Inp32(rIICSTAT0);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT0);			

	uTmp1 = I2C_Inp32(rIICCON0);
	uTmp1 |= (1<<7);
	Outp32(rIICCON0,uTmp1);				//	Ack generation Enable

	uTmp0 = I2C_Inp32(rIICSTAT0);
	uTmp0 &=~(1<<4);
	Outp32(rIICSTAT0,uTmp0);				// Disable Rx/Tx for setting SlaveAddr

	I2C_Outp8(rIICADD0,*pSlaveAddr);

	Outp32(rIICSTAT0,0x50);				//	Slave Tx Start.

	while(!(sDcnt == -1))
	{
		if(Inp8(rIICCON0)&0x10)			//until Int Pending
		{
			if((sDcnt--) == 0)
			{
				Outp32(rIICSTAT0,0xd0);		//	Stop Master Tx condition, ACK flag clear
				
				uTmp0 = I2C_Inp32(rIICCON0);
				uTmp0 &= ~(1<<4);			//	Clear pending bit to resume
				Outp32(rIICCON0,uTmp0);

				Delay(1);					//	wait until Stop Condition is in effect
				break;
			}

			I2C_Outp8(rIICDS0,pData[uPT++]);
			for(cCnt=0;cCnt<10;cCnt++);		//	for setup time (rising edge of IICSCL)
			
			uTmp0 = I2C_Inp32(rIICCON0);
			uTmp0 &= ~(1<<4);				//	Clear pending bit to resume
			Outp32(rIICCON0,uTmp0);				
		}
	}
}

//////////
// Function Name : IIC1_SlaveWrP
// Function Description : This function do SlaveTx mode by polling operation
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Tx]
// Output : NONE
// Version : v0.1
void IIC2_SlaveWrP(u8 *pSlaveAddr,u8 *pData)
{
	u32 uTmp0;
	u32 uTmp1;
	u8 cCnt;
	s32 sDcnt = 100;
	u32 uPT = 0;

	uTmp0 = I2C_Inp32(rIICSTAT2);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT2);			

	uTmp1 = I2C_Inp32(rIICCON2);
	uTmp1 |= (1<<7);
	Outp32(rIICCON2,uTmp1);				//	Ack generation Enable

	uTmp0 = I2C_Inp32(rIICSTAT2);
	uTmp0 &=~(1<<4);
	Outp32(rIICSTAT2,uTmp0);				// Disable Rx/Tx for setting SlaveAddr

	I2C_Outp8(rIICADD2,*pSlaveAddr);

	Outp32(rIICSTAT2,0x50);				//	Slave Tx Start.


	while(!(sDcnt == -1))
	{
		if(Inp8(rIICCON2)&0x10)			//until Int Pending
		{
			if((sDcnt--) == 0)
			{
				Outp32(rIICSTAT2,0xd0);		//	Stop Master Tx condition, ACK flag clear
				
				uTmp0 = I2C_Inp32(rIICCON2);
				uTmp0 &= ~(1<<4);			//	Clear pending bit to resume
				Outp32(rIICCON2,uTmp0);

				Delay(1);					//	wait until Stop Condition is in effect
				break;
			}

			I2C_Outp8(rIICDS2,pData[uPT++]);
			for(cCnt=0;cCnt<10;cCnt++);		//	for setup time (rising edge of IICSCL)
			
			uTmp0 = I2C_Inp32(rIICCON2);
			uTmp0 &= ~(1<<4);				//	Clear pending bit to resume
			Outp32(rIICCON2,uTmp0);				
		}
	}
}


//////////
// Function Name : IIC_SlaveRdInt
// Function Description : This function do SlaveRx mode by Interrupt operation
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Rx]
// Output : NONE
// Version : v0.1
void IIC0_SlaveRdInt(u8 *pSlaveAddr,u8 *pData)
{
	u32 uTmp0;
	u32 uTmp1;
	
	g_PcIIC_BUFFER	=	pData;
	g_uIIC_PT		=	0;
	g_cIIC_SlaveRxDone = 0;

	uTmp0 = I2C_Inp32(rIICSTAT0);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT0);			

	uTmp1 = I2C_Inp32(rIICCON0);
	uTmp1 |= (1<<7);
	Outp32(rIICCON0,uTmp1);				//	Ack generation Enable

	uTmp0 = I2C_Inp32(rIICSTAT0);
	uTmp0 &= ~(1<<4);
	Outp32(rIICSTAT0,uTmp0);				// Disable Rx/Tx for setting SlaveAddr
	I2C_Outp8(rIICADD0,*pSlaveAddr);
	Outp32(rIICSTAT0,0x10);				//	Slave Rx Start.
	while(!(g_cIIC_SlaveRxDone));
}


//////////
// Function Name : IIC1_SlaveRdInt
// Function Description : This function do SlaveRx mode by Interrupt operation
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Rx]
// Output : NONE
// Version : v0.1
void IIC2_SlaveRdInt(u8 *pSlaveAddr,u8 *pData)
{
	u32 uTmp0;
	u32 uTmp1;
	
	g_PcIIC_BUFFER	=	pData;
	g_uIIC_PT		=	0;
	g_cIIC_SlaveRxDone = 0;

	uTmp0 = I2C_Inp32(rIICSTAT2);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT2);			

	uTmp1 = I2C_Inp32(rIICCON2);
	uTmp1 |= (1<<7);
	Outp32(rIICCON2,uTmp1);				//	Ack generation Enable

	uTmp0 = I2C_Inp32(rIICSTAT2);
	uTmp0 &= ~(1<<4);
	Outp32(rIICSTAT2,uTmp0);				// Disable Rx/Tx for setting SlaveAddr
	I2C_Outp8(rIICADD2,*pSlaveAddr);
	Outp32(rIICSTAT2,0x10);				//	Slave Rx Start.
	while(!(g_cIIC_SlaveRxDone));
}


//////////
// Function Name : IIC_SlaveWrInt
// Function Description : This function do SlaveTx mode by Interrupt operation
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Tx]
// Output : NONE
// Version : v0.1
void IIC0_SlaveWrInt(u8 *pSlaveAddr,u8 *pData)
{
	u32 uTmp0;
	u32 uTmp1;
	
	g_PcIIC_BUFFER	=	pData;
	g_uIIC_PT		=	0;
	g_cIIC_SlaveTxDone = 0;

	uTmp0 = I2C_Inp32(rIICSTAT0);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT0);			

	uTmp1 = I2C_Inp32(rIICCON0);
	uTmp1 |= (1<<7);
	Outp32(rIICCON0,uTmp1);				//	Ack generation Enable

	uTmp0 = I2C_Inp32(rIICSTAT0);
	uTmp0 &=~(1<<4);
	Outp32(rIICSTAT0,uTmp0);				// Disable Rx/Tx for setting SlaveAddr
	I2C_Outp8(rIICADD0,*pSlaveAddr);
	Outp32(rIICSTAT0,0x50);				//	Slave Tx Start.
	while(!(g_cIIC_SlaveTxDone));
}


//////////
// Function Name : IIC1_SlaveWrInt
// Function Description : This function do SlaveTx mode by Interrupt operation
// Input : pSlaveAddr [pointer of 8bit SlaveDeviceAddress], 
//		 pData [pointer of Data which you want to Tx]
// Output : NONE
// Version : v0.1
void IIC2_SlaveWrInt(u8 *pSlaveAddr,u8 *pData)
{
	u32 uTmp0;
	u32 uTmp1;
	
	g_PcIIC_BUFFER	=	pData;
	g_uIIC_PT		=	0;
	g_cIIC_SlaveTxDone = 0;

	uTmp0 = I2C_Inp32(rIICSTAT2);
	while(uTmp0&(1<<5))					//	Wait until IIC bus is free.
		uTmp0 = I2C_Inp32(rIICSTAT2);			

	uTmp1 = I2C_Inp32(rIICCON2);
	uTmp1 |= (1<<7);
	Outp32(rIICCON2,uTmp1);				//	Ack generation Enable

	uTmp0 = I2C_Inp32(rIICSTAT2);
	uTmp0 &=~(1<<4);
	Outp32(rIICSTAT2,uTmp0);				// Disable Rx/Tx for setting SlaveAddr
	I2C_Outp8(rIICADD2,*pSlaveAddr);
	Outp32(rIICSTAT2,0x50);				//	Slave Tx Start.
	while(!(g_cIIC_SlaveTxDone));
}

//////////
// Function Name : IIC Stop flag check
// Function Description : This function do check STOP signal generation flag
// Input : void
// Output : g_cIIC_StopCheck
// Version : v0.1
u32 IIC_CheckStop(void)
{
	return g_cIIC_StopCheck;
}	
//////////
// Function Name : IIC ACK flag check
// Function Description : This function do check ACK signal generation flag
// Input : void
// Output : g_cIIC_AckCheck
// Version : v0.1
u32 IIC_CheckAck(void)
{
	return g_cIIC_AckCheck;
}	

u32 IIC_Set_StopSignal(u8 uStop)
{
	if (uStop !=0 )
		g_uIIC_StopGeneration = true;
	else
		g_uIIC_StopGeneration = false;
				
	return true;				
}	


//////////
// Function Name : IIC1_Open
// Function Description : This function Set up VIC & IICCON with user's input frequency which determines uClkValue
// Input : ufreq	ufreq(Hz) = PCLK/PCLmode/uClkValue
//				PLCKmode : 1: 512,  0:16	
// 				 Prescaler : 0~15
// Output : NONE
// Version : v0.1
void IIC2_Open_new( u32 PLCKmode, u32 Prescaler)		//	IIC clock table index
{
	//u32	uSelClkSrc;
	//u32	uClkValue;
	u32 	uTmp;

	UART_Printf("\n*** IIC1 Setting Frequency\n");
	
	INTC_SetVectAddr(NUM_I2C_PMIC,Isr_IIC2);
	INTC_Enable(NUM_I2C_PMIC);

	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_4,2);
	GPIO_SetFunctionEach(eGPIO_D1,eGPIO_5,2);	
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_4,2);
	GPIO_SetPullUpDownEach(eGPIO_D1,eGPIO_5,2);


	Outp32(rIICCON2,(PLCKmode<<6) | (1<<5) | (Prescaler&0xf));
	Outp32(rIICADD2,0xc0);		//	Slave address = [7:1]
	Outp32(rIICSTAT2,0x10);		//	IIC bus data output enable(Rx/Tx)
	Outp32(rIICLC2,0x4);			//	SDA Filter Enable,delayed 0clks
	//Outp32(rIICLC1,0x5);		//	SDA Filter enable,delayed 5clks
	//Outp32(rIICLC1,0x6);		//	SDA Filter enable,delayed 10clks
	//Outp32(rIICLC1,0x7);		//	SDA Filter enable,delayed 15clks		

	if (PLCKmode==0) //PCLK/16
		uTmp=16;
	else if (PLCKmode==1) //PCLK/512
		uTmp=512;
	else
		uTmp=0;		
	
	UART_Printf("*** PCLK= %.2fMHz,  IICCLK=PCLK/%d   ,ClkDivideValue= %d\n",(float)g_PCLK/1.0e6,uTmp,Prescaler);			
	UART_Printf("*** IICSCL (Tx clock)=  %d Hz\n",((g_PCLK/uTmp)/(Prescaler+1)));		
}

/**
* {ClockInform}
*
* @brief			Update 'g_uPclkPsys'.
* @description	Update 'g_uPclkPsys'. 
* @param[in]	VOID
* @return		VOID 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void ClockInform(void)
{
	SYSC_UpdateClkInform();
	UART_Printf(" APLL: %.2fMHz  MPLL: %.2fMHz  EPLL: %.2fMHz  VPLL: %.2fMHz\n",
			(float)g_uAPLL/1.0e6, (float)g_uMPLL/1.0e6, (float)g_uEPLL/1.0e6, (float)g_uVPLL/1.0e6);
	UART_Printf(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz  PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz  PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz  PCLK_Psys: %.2fMHz\n\n",
			(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);
}
/*******************************************************************************
 * End of file Iic.c
 ******************************************************************************/
/////////////////////////////////////////////////////////////////////////////////////////////

/*******************************************************************************
* GLOBAL VARIABLES
*******************************************************************************/
u32 gI2C_EventStatus = EVENT_I2C_NO;
I2C_Context_st	I2C_Context;

bool 	gbI2C_CH0_Performance_Test = FALSE;

/*******************************************************************************
* FUNCTION PROTOTYPES
*******************************************************************************/
/**
* {I2C_SFR_Mapping_Init}
*
* @brief			Mapping 'I2C_SFR_List' into 'I2C_REG' by channel Index.
* 
* @param[in]	(u32 ch)
* @return		void 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/ 
void I2C_SFR_Mapping_Init(u32 ch)
{
	//I2C register struct Inital
	I2C_REG[ch].CON 	= I2C_SFR_List[ch*5 + 0];
	I2C_REG[ch].STAT	= I2C_SFR_List[ch*5+ 1];
	I2C_REG[ch].ADD		= I2C_SFR_List[ch*5+ 2];
	I2C_REG[ch].DS		= I2C_SFR_List[ch*5+ 3];
	I2C_REG[ch].LC		= I2C_SFR_List[ch*5+ 4];
}

/**
* {I2C_Release}
*
* @brief			Clear interrupt pending bit & disable.
* 
* @param[in]	(u32 ch)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/ 
s32 I2C_Release(u32 ch)
{
	// clear interrupt pending flag and interrupt clear bit
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
	// disable 
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_IDLE);

	return OK;
}


/**
* {I2C_IRQHandler}
*
* @brief			Interrupt Handler.
* 
* @param[in]	VOID
* @return		VOID 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void __irq I2C_IRQHandler(void)
{
	u8 ucStatus;
	u8 ucReg;
	s32 sCont = TRUE;	
	u32 uTmp;
		
	// check interrupt reason
	ucStatus = I2C_Inp8(I2C_REG[gI2c_CH_Index].CON);

	if(ucStatus & I2CCON_INT_PEND)
	{
		// check Status flags
		ucReg = I2C_Inp8(I2C_REG[gI2c_CH_Index].STAT);

		while (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_SLAVE | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR) )
		{
			sCont = FALSE;
			if (ucReg & I2CSTAT_ARBITER)
			{
				UART_Printf ("I2CSTAT_ARBITER\n");
			}
			else if (ucReg & I2CSTAT_ADDR_SLAVE)
			{
				UART_Printf ("I2CSTAT_ADDR_SLAVE\n");
				sCont = TRUE;
				break;
			}
			else if (ucReg & I2CSTAT_ADDR_ZERO)
			{
				UART_Printf ("I2CSTAT_ADDR_ZERO\n");
			}
			else if (ucReg & I2CSTAT_ACK_ERROR)
			{
				if (	I2C_Context.eState == I2C_MASTER_TX_STOP_STATE				||
					I2C_Context.eState == I2C_MASTER_RX_STOP_STATE				||
					I2C_Context.eState == I2C_SLAVE_TX_STOP_STATE					||
					I2C_Context.eState == I2C_SLAVE_RX_STOP_STATE					||
					
					I2C_Context.eState == I2C_MASTER_TX_EDDC_SEGADDR_STATE	||
					( (I2C_Context.eState == I2C_MASTER_TX_EDDC_SEGNUM_STATE)	&& (I2C_Context.segnum == 0) ) )
				{
					sCont = TRUE;
					break;
				}
				else
				{
					UART_Printf ("I2C_STAT_ACKERROR : %d\n", I2C_Context.eState);
				}
			}

			I2C_Release(gI2c_CH_Index);

			gI2C_EventStatus = EVENT_I2C_ERROR;
			break;
		}

		if (sCont)
		{
			switch (I2C_Context.eState)
			{
				case I2C_MASTER_TX_EDDC_SEGADDR_STATE:
					I2C_Context.eState = I2C_MASTER_TX_EDDC_SEGNUM_STATE;

					// write segment number
					I2C_Outp8(I2C_REG[gI2c_CH_Index].DS, I2C_Context.segnum);
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
					break;

				case I2C_MASTER_TX_EDDC_SEGNUM_STATE:
					I2C_Context.eState = I2C_MASTER_TX_DDC_ADDR_STATE;

					// put address to shift register
					I2C_Outp8(I2C_REG[gI2c_CH_Index].DS, I2C_Context.addr & 0xFE);
					// master tx mode + start + enable - "START" sequence generation
					I2C_Outp8(I2C_REG[gI2c_CH_Index].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_TX_MODE); // 0xF0
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );
					break;

				case I2C_MASTER_TX_DDC_ADDR_STATE:
					I2C_Context.eState = I2C_MASTER_TX_DDC_OFFSET_STATE;

					// write offset
					I2C_Outp8(I2C_REG[gI2c_CH_Index].DS, I2C_Context.offset);
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
					break;

				case I2C_MASTER_TX_DDC_OFFSET_STATE:
					I2C_Context.eState = I2C_MASTER_RX_DDC_ADDR_STATE;

					// put address to shift register
					I2C_Outp8(I2C_REG[gI2c_CH_Index].DS, I2C_Context.addr & 0xFE);
					// master rx mode + start + enable - "START" sequence generation
					I2C_Outp8(I2C_REG[gI2c_CH_Index].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_RX_MODE); // 0xB0
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );
					break;
					
				case I2C_MASTER_TX_ADDR_STATE:
				case I2C_MASTER_TX_DATA_STATE:					
					I2C_Context.eState = I2C_MASTER_TX_DATA_STATE;

					I2C_Outp8(I2C_REG[gI2c_CH_Index].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;
					--(I2C_Context.bytes);
			
					if(I2C_Context.bytes > 0)
					{
						//Clear pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE); // 0xE0
					}
					else
					{
						I2C_Context.eState = I2C_MASTER_TX_STOP_STATE;
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR); // 0x60
					}
					break;

				case I2C_MASTER_TX_STOP_STATE:
					I2C_Context.eState = I2C_IDLE_STATE;

					I2C_Outp8(I2C_REG[gI2c_CH_Index].STAT, I2CSTAT_MASTER_TX_MODE | I2CSTAT_ENABLE); // 0xD0
					
					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);// 0x60

					// wait til the stop condition takes effect.
					while( I2C_Inp8(I2C_REG[gI2c_CH_Index].STAT) & I2CSTAT_START );

					gI2C_EventStatus = EVENT_I2C_DONE;
					break;

				case I2C_MASTER_RX_DDC_ADDR_STATE:
					I2C_Context.eState = I2C_MASTER_RX_DDC_DATA_STATE;

					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;

						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );  
					}
					else
					{
						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
					}
					break;
					
				case I2C_MASTER_RX_ADDR_STATE:
					I2C_Context.eState = I2C_MASTER_RX_DATA_STATE;

					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;

						// clear interrupt pending bit  to resume
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );
					}
					else
					{
						// clear interrupt pending bit  to resume
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
					}
					break;
					
				case I2C_MASTER_RX_DATA_STATE:
				case I2C_MASTER_RX_DDC_DATA_STATE:
					// read byte
					*(I2C_Context.buffer) = I2C_Inp8(I2C_REG[gI2c_CH_Index].DS);
					I2C_Context.buffer++;
					--(I2C_Context.bytes);
					        
					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;
						
						//slave address   next data reading NO ACK߻ϱ  ̸ disableŴ
						// Disable Ack generation (for EEPROM reading)							
						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR ); //0x60
					}
					else
					{
						// clear interrupt pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE ); // 0xE0
					}
					break;

				case I2C_MASTER_RX_STOP_STATE:
					I2C_Context.eState = I2C_IDLE_STATE;

					*(I2C_Context.buffer) = I2C_Inp8(I2C_REG[gI2c_CH_Index].DS);

					I2C_Outp8(I2C_REG[gI2c_CH_Index].STAT, I2CSTAT_MASTER_RX_MODE | I2CSTAT_ENABLE); //0x90
					
					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
					
					// wait til the stop condition takes effect.
					while( I2C_Inp8(I2C_REG[gI2c_CH_Index].STAT) & I2CSTAT_START );
					
					gI2C_EventStatus = EVENT_I2C_DONE;
					break;

				case I2C_SLAVE_TX_ADDR_STATE:
				case I2C_SLAVE_TX_DATA_STATE:
					I2C_Context.eState = I2C_SLAVE_TX_DATA_STATE;

					I2C_Outp8(I2C_REG[gI2c_CH_Index].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;
					--(I2C_Context.bytes);
			
					if(I2C_Context.bytes > 0)
					{
						//Clear pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE); // 0xE0
					}
					else
					{
						I2C_Context.eState = I2C_SLAVE_TX_STOP_STATE;
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR); // 0x60
					}
					break;

				case I2C_SLAVE_TX_STOP_STATE:
					I2C_Context.eState = I2C_IDLE_STATE;

					// Slave Rx Stop
					I2C_Outp8(I2C_REG[gI2c_CH_Index].STAT, I2CSTAT_IDLE);	 //	Stop Slave Tx condition
					//I2C_Pending_Bit_Clear(gI2c_CH_Index);	//	Tx : Clear pending bit
					uTmp = I2C_Inp32(I2C_REG[gI2c_CH_Index].CON);
					uTmp &= ~(I2CCON_INT_PEND);		//	Clear pending bit to resume
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, uTmp);
					
					// wait til the stop condition takes effect.
					while( I2C_Inp8(I2C_REG[gI2c_CH_Index].STAT) & I2CSTAT_START );

					gI2C_EventStatus = EVENT_I2C_DONE;
					break;
					
				case I2C_SLAVE_RX_ADDR_STATE:
					I2C_Context.eState = I2C_SLAVE_RX_DATA_STATE;

					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_SLAVE_RX_STOP_STATE;

						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR );
					}
					else
					{
						// clear interrupt pending bit  to resume
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
					}
					break;
					
				case I2C_SLAVE_RX_DATA_STATE:
					// read byte
					*(I2C_Context.buffer) = I2C_Inp8(I2C_REG[gI2c_CH_Index].DS);
					I2C_Context.buffer++;
					--(I2C_Context.bytes);
					        
					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_SLAVE_RX_STOP_STATE;
						
						//slave address   next data reading NO ACK߻ϱ  ̸ disableŴ
						// Disable Ack generation (for EEPROM reading)
						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR ); //0x60
					}
					else
					{
						// clear interrupt pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE ); // 0xE0
					}
					break;
					
				case I2C_SLAVE_RX_STOP_STATE:
					I2C_Context.eState = I2C_IDLE_STATE;

					*(I2C_Context.buffer) = I2C_Inp8(I2C_REG[gI2c_CH_Index].DS);

					// Slave Rx Stop
					I2C_Outp8(I2C_REG[gI2c_CH_Index].STAT, I2CSTAT_IDLE);	 //	Stop Slave Rx condition

					//I2C_Pending_Bit_Clear(gI2c_CH_Index);	//	Rx : Clear pending bit			
					uTmp = I2C_Inp32(I2C_REG[gI2c_CH_Index].CON);
					uTmp &= ~(I2CCON_INT_PEND);		//	Clear pending bit to resume
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, uTmp);

					// wait til the stop condition takes effect.
					while( I2C_Inp8(I2C_REG[gI2c_CH_Index].STAT) & I2CSTAT_START );
					
					gI2C_EventStatus = EVENT_I2C_DONE;					
					break;
					
				case I2C_IDLE_STATE:
				default:
					UART_Printf ("Unknown state!!!\n");
					gI2C_EventStatus = EVENT_I2C_ERROR;
			}
		}
	}
	else
	{
		UART_Printf ("*** UNHANDLED ***\n");
	}

	INTC_ClearVectAddr();
}

/**
* {I2C_Polling_InterruptWait}
*
* @brief			Interrupt wait
* 
* @param[in]	(u32 ch)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Polling_InterruptWait(u32 ch)
{
	u8 ucStatus, ucReg;
	s32 retval = OK;

	do
	{
		ucStatus = I2C_Inp8(I2C_REG[ch].CON);
		if (ucStatus & I2CCON_INT_PEND)
		{
			// check status flags
			ucReg = I2C_Inp8(I2C_REG[ch].STAT);

			if (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_SLAVE | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR) )
			{
				if (ucReg & I2CSTAT_ARBITER)
				{
					UART_Printf("I2CSTAT_ARBITER\n");
				}
				else if (ucReg & I2CSTAT_ADDR_SLAVE)
				{
					UART_Printf("I2CSTAT_ADDR_SLAVE\n");
					retval = OK;
					break;
				}
				else if (ucReg & I2CSTAT_ADDR_ZERO)
				{
					UART_Printf("I2CSTAT_ADDR_ZERO\n");
				}
				else if (ucReg & I2CSTAT_ACK_ERROR)
				{
					UART_Printf("I2CSTAT_ACK_ERROR\n");
				}
				
				I2C_Release(ch);
				retval = ERROR;
			}
			break;
		}
	}while (1);

	return retval;
}

/**
* {I2C_Slave_Polling_Read}
*
* @brief			Read data through I2C 
* 
* @param[in]	( u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Slave_Polling_Read( u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
{
	s32 iRetval = OK;
	u32 i, uProcess = TRUE;

	u32 uTmp1 = 0;
	
	uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);

	while( (uTmp1 & I2CSTAT_START ) || (uTmp1 & I2CSTAT_ARBITER) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);
	}
	
	// set state
	I2C_Context.eState = I2C_SLAVE_RX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_SLAVE_RX_MODE);
	// put address to shift register
	I2C_Outp8(I2C_REG[ch].DS, cSlaveAddr & 0xFE);
	// master rx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_SLAVE_RX_MODE); // 0x30

	while (uProcess)
	{
		//In case of Master RX Mode, Last data is not ack. but set the "ACK not received" bit
		if(I2C_Context.eState != I2C_SLAVE_RX_STOP_STATE )
		{
			if( OK != I2C_Polling_InterruptWait(ch) )
			{
				UART_Printf("I2C_Polling_InterruptWait() failed!!!\n");
				iRetval = ERROR;
				break;
			}
		}

		switch (I2C_Context.eState)
		{
			case I2C_SLAVE_RX_ADDR_STATE:
				I2C_Context.eState = I2C_SLAVE_RX_DATA_STATE;

				if (I2C_Context.bytes == 1)
				{
					I2C_Context.eState = I2C_SLAVE_RX_STOP_STATE;

					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
				}
				else
				{
					// clear interrupt pending bit to resume
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
				}
				break;

			case I2C_SLAVE_RX_DATA_STATE:
				// read byte
				*(I2C_Context.buffer) = I2C_Inp8(I2C_REG[ch].DS);
				I2C_Context.buffer++;
				--(I2C_Context.bytes);
				            
				if (I2C_Context.bytes == 1)
				{
					I2C_Context.eState = I2C_SLAVE_RX_STOP_STATE;

					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
				}
				else
				{
					// clear interrupt pending bit to resume
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
				}
				break;

			case I2C_SLAVE_RX_STOP_STATE:
				I2C_Context.eState = I2C_IDLE_STATE;

				//for(i=0 ; i<10000 ; i++);
				Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
				
				*(I2C_Context.buffer) = I2C_Inp8(I2C_REG[ch].DS);				

				//"STOP" sequence generation
				I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_SLAVE_RX_MODE | I2CSTAT_ENABLE); 
				// clear interrupt pending flag and interrupt clear bit
				I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);

				//Wait until the stop condition takes effect
				while( I2C_Inp8(I2C_REG[ch].STAT) & I2CSTAT_START );
				uProcess = FALSE;
				break;

			case I2C_IDLE_STATE:
			default:
				UART_Printf("error state!!!\n");
				iRetval = ERROR;
				uProcess = FALSE;
				break;
		}
	}

	return iRetval;
}

/**
* {I2C_Slave_Polling_Write}
*
* @brief			Write data through I2C 
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
 s32 I2C_Slave_Polling_Write(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
{
	s32 iRetval = OK;
	u32 uProcess = TRUE;

	u32 uTmp1 = 0;
	
	uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);

	while( (uTmp1 & I2CSTAT_START ) || (uTmp1 & I2CSTAT_ARBITER) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);
	}
	
	// set state
	I2C_Context.eState = I2C_SLAVE_TX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_SLAVE_TX_MODE); // 0x50
	// put address to shift register
	I2C_Outp8(I2C_REG[ch].DS, cSlaveAddr & 0xFE);
	// master tx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_SLAVE_TX_MODE); // 0x70
	
	while (uProcess)
	{
		if( OK != I2C_Polling_InterruptWait(ch) )
		{
			UART_Printf("InterruptWait() failed!!!\n");
			iRetval = ERROR;
			break;
		}

		switch (I2C_Context.eState)
		{
			case I2C_SLAVE_TX_ADDR_STATE:
			case I2C_SLAVE_TX_DATA_STATE:
				I2C_Context.eState = I2C_SLAVE_TX_DATA_STATE;

				I2C_Outp8(I2C_REG[ch].DS, *(I2C_Context.buffer));
				
				I2C_Context.buffer++;
				--(I2C_Context.bytes);

				if (I2C_Context.bytes == 0)
				{
					I2C_Context.eState = I2C_SLAVE_TX_STOP_STATE;
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR);
				}
				else
				{
					// clear interrupt pending bit to resume
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
				}
				break;

			case I2C_SLAVE_TX_STOP_STATE:
				I2C_Context.eState = I2C_IDLE_STATE;

				// send stop bit
				I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_SLAVE_TX_MODE| I2CSTAT_ENABLE); //0x50
				
				// clear interrupt pending bit
				I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR);

				// wait till the stop condition takes effect.
				while( I2C_Inp8(I2C_REG[ch].STAT) & I2CSTAT_START );
				
				uProcess = FALSE;
				break;

			case I2C_IDLE_STATE:
			default:
				UART_Printf("error state!!!\n");
				iRetval = ERROR;
				uProcess = FALSE;
				break;
		}
	}

	return iRetval;
}

/**
* {I2C_Slave_Interrupt_Read}
*
* @brief			Read data through I2C 
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Slave_Interrupt_Read(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
{
	u32 uTmp1 = 0;
	
	uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);

	while( (uTmp1 & I2CSTAT_START ) || (uTmp1 & I2CSTAT_ARBITER) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);
	}
	
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_SLAVE_RX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;

	/* initiate transfer */
	
	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_SLAVE_RX_MODE); // 0x30
	// put address to shift register
	I2C_Outp8(I2C_REG[ch].DS, cSlaveAddr & 0xFE);
	// master rx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_SLAVE_RX_MODE); // 0x30

	//waiting DDC status
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}

	return OK;
}

/**
* {I2C_Slave_Interrupt_Write}
*
* @brief			Write data through I2C 
* 
* @param[in]	( u32 ch, u8 cSlaveAddr,  u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Slave_Interrupt_Write( u32 ch, u8 cSlaveAddr,  u8 *pData, s32 uDataLen)
{
	u32 uTmp1;
	
	uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);

	while( (uTmp1 & I2CSTAT_START ) || (uTmp1 & I2CSTAT_ARBITER) )	//	Wait until IIC bus is free or Arbitration successful.
	{
		uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);
	}
	

	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_SLAVE_TX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;


	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_ACK_ENABLE); // 0xE0
	// Disable Rx/Tx for setting SlaveAddr
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_DISABLE | I2CSTAT_SLAVE_TX_MODE); // 0x40
	// write slave address to address register
	I2C_Outp8(I2C_REG[ch].ADD, cSlaveAddr & 0xFE);
	// slave tx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_SLAVE_TX_MODE); // 0x70

	//waiting DDC status
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}

	return OK;
}

/**
* {I2C_Master_Polling_Read}
*
* @brief			Read data through I2C 
* 
* @param[in]	( u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Master_Polling_Read( u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
{
	s32 iRetval = OK;
	u32 i, uProcess = TRUE;

	u32 uElapsedTime = 0;

	u32 uTmp1 = 0;
	
	uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);

	while( (uTmp1 & I2CSTAT_START ) || (uTmp1 & I2CSTAT_ARBITER) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);
	}
	
	// set state
	I2C_Context.eState = I2C_MASTER_RX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_MASTER_RX_MODE); // 0xB0
	// put address to shift register
	I2C_Outp8(I2C_REG[ch].DS, cSlaveAddr & 0xFE);	

	if( gbI2C_CH0_Performance_Test == TRUE )
	{
		UART_Printf("\nREAD\n");
		StartTimer(0);
	}
	
	
	// master rx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_RX_MODE); // 0xB0
	
	while (uProcess)
	{
		//In case of Master RX Mode, Last data is not ack. but set the "ACK not received" bit
		if(I2C_Context.eState != I2C_MASTER_RX_STOP_STATE )
		{
			if( OK != I2C_Polling_InterruptWait(ch) )
			{
				UART_Printf("InterruptWait() failed!!!\n");
				iRetval = ERROR;
				break;
			}
		}

		switch (I2C_Context.eState)
		{
			case I2C_MASTER_RX_ADDR_STATE:

				if( gbI2C_CH0_Performance_Test == TRUE )
				{
					uElapsedTime = StopTimer(0);
					UART_Printf("(%3.1fKBps,%3.1fKHz)\n",((float)1000 / uElapsedTime ), ((float)1000 * 9/* 9 bits */ / uElapsedTime));
				}

				I2C_Context.eState = I2C_MASTER_RX_DATA_STATE;

				if (I2C_Context.bytes == 1)
				{
					I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;

					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
				}
				else
				{
					if( gbI2C_CH0_Performance_Test == TRUE )
					{
						UART_Printf("\nREAD\n");
						StartTimer(0);
					}

					// clear interrupt pending bit to resume
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
				}
				break;

			case I2C_MASTER_RX_DATA_STATE:

				if( gbI2C_CH0_Performance_Test == TRUE )
				{
					uElapsedTime = StopTimer(0);
					UART_Printf("(%3.1fKBps,%3.1fKHz)\n",((float)1000 / uElapsedTime), ((float)1000 * 9/* 9 bits */ / uElapsedTime));
				}
				
				// read byte
				*(I2C_Context.buffer) = I2C_Inp8(I2C_REG[ch].DS);
				I2C_Context.buffer++;
				--(I2C_Context.bytes);
				            
				if (I2C_Context.bytes == 1)
				{
					I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;

					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
				}
				else
				{
					if( gbI2C_CH0_Performance_Test == TRUE )
					{
						UART_Printf("\nREAD\n");
						StartTimer(0);
					}

					// clear interrupt pending bit to resume
					I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
				}
				break;

			case I2C_MASTER_RX_STOP_STATE:
				I2C_Context.eState = I2C_IDLE_STATE;

				//for(i=0 ; i<10000 ; i++);
				Delay(35);  //for EEPROM internal writing cycle max 10ms(typical 3.5ms)
				
				*(I2C_Context.buffer) = I2C_Inp8(I2C_REG[ch].DS);				

				//"STOP" sequence generation
				I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_MASTER_RX_MODE | I2CSTAT_ENABLE); 
				// clear interrupt pending flag and interrupt clear bit
				I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);

				//Wait until the stop condition takes effect
				while( I2C_Inp8(I2C_REG[ch].STAT) & I2CSTAT_START );
				uProcess = FALSE;
				break;

			case I2C_IDLE_STATE:
			default:
				UART_Printf("error state!!!\n");
				iRetval = ERROR;
				uProcess = FALSE;
				break;
		}
	}

	return iRetval;
}

/**
* {I2C_Master_Polling_Write}
*
* @brief			Write data through I2C 
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
 s32 I2C_Master_Polling_Write(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
{
	s32 iRetval = OK;
	u32 uProcess = TRUE;

	u32 uElapsedTime = 0;

	u32 uTmp1 = 0;
	
	uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);

	while( (uTmp1 & I2CSTAT_START ) || (uTmp1 & I2CSTAT_ARBITER) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);
	}
	
	// set state
	I2C_Context.eState = I2C_MASTER_TX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;

	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_MASTER_TX_MODE);	// 0xD0
	// put address to shift register
	I2C_Outp8(I2C_REG[ch].DS, cSlaveAddr & 0xFE);

	if( gbI2C_CH0_Performance_Test == TRUE )
	{
		UART_Printf("\nWRITE\n");
		StartTimer(0);
	}

	// master tx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_TX_MODE); // 0xF0
	
	while (uProcess)
	{
		if( OK != I2C_Polling_InterruptWait(ch) )
		{
			UART_Printf("I2C_Polling_InterruptWait() failed!!!\n");
			iRetval = ERROR;
			break;
		}

		switch (I2C_Context.eState)
		{
			case I2C_MASTER_TX_ADDR_STATE:
			case I2C_MASTER_TX_DATA_STATE:

				if( gbI2C_CH0_Performance_Test == TRUE )
				{
					uElapsedTime = StopTimer(0);
					UART_Printf("(%3.1fKBps,%3.1fKHz)\n",((float)1000 / uElapsedTime ), ((float)1000 * 9/* 9 bits */ / uElapsedTime ));
				}

				I2C_Context.eState = I2C_MASTER_TX_DATA_STATE;

				I2C_Outp8(I2C_REG[ch].DS, *(I2C_Context.buffer));
				
				I2C_Context.buffer++;
				--(I2C_Context.bytes);

				if(I2C_Context.bytes > 0)
				{
					if( gbI2C_CH0_Performance_Test == TRUE )
					{
						StartTimer(0);
					}

					//Clear pending bit to resume
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE); // 0xE0
				}
				else
				{
					I2C_Context.eState = I2C_MASTER_TX_STOP_STATE;
					I2C_Outp8(I2C_REG[gI2c_CH_Index].CON, (I2C_Inp8(I2C_REG[gI2c_CH_Index].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR); // 0x60
				}
				break;

			case I2C_MASTER_TX_STOP_STATE:
				I2C_Context.eState = I2C_IDLE_STATE;

				// send stop bit
				I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_MASTER_TX_MODE| I2CSTAT_ENABLE); //0xD0
				
				// clear interrupt pending bit
				I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR);

				// wait till the stop condition takes effect.
				while( I2C_Inp8(I2C_REG[ch].STAT) & I2CSTAT_START );
				
				uProcess = FALSE;
				break;

			case I2C_IDLE_STATE:
			default:
				UART_Printf("error state!!!\n");
				iRetval = ERROR;
				uProcess = FALSE;
				break;
		}
	}

	return iRetval;
}

/**
* {I2C_Master_Interrupt_ReadEDDC}
*
* @brief			Read data through I2C 
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Master_Interrupt_ReadEDDC(u32 ch, u8 segaddr, u8 segnum, u8 cSlaveAddr, u8 offset, u8 *pData, s32 uDataLen)
{
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_MASTER_TX_EDDC_SEGADDR_STATE;

	// set parameters
	I2C_Context.segnum = segnum;
	I2C_Context.addr = cSlaveAddr;
	I2C_Context.offset = offset;
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;
	
	/* initiate transfer */
	
	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_MASTER_RX_MODE); // 0xB0
	// put address to shift register
	I2C_Outp8(I2C_REG[ch].DS, cSlaveAddr & 0xFE);
	// master rx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_RX_MODE); // 0xB0

	//waiting DDC status
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}

	return OK;
}	
/**
* {I2C_Master_Interrupt_ReadHDCP}
*
* @brief			Read data through I2C 
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Master_Interrupt_ReadHDCP(u32 ch, u8 cSlaveAddr, u8 offset, u8 *pData, s32 uDataLen)
{
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_MASTER_TX_DDC_ADDR_STATE;

	// set parameters
	I2C_Context.offset = offset;
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;
	I2C_Context.addr = cSlaveAddr;
	
	/* initiate transfer */
	
	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_MASTER_RX_MODE); // 0xB0
	// put address to shift register
	I2C_Outp8(I2C_REG[ch].DS, cSlaveAddr & 0xFE);
	// master rx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_RX_MODE); // 0xB0

	//waiting DDC status
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}

	return OK;
}

/**
* {I2C_Master_Interrupt_Read}
*
* @brief			Read data through I2C 
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Master_Interrupt_Read(u32 ch, u8 cSlaveAddr, u8 *pData, s32 uDataLen)
{
	u32 uElapsedTime = 0;

	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_MASTER_RX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;

	/* initiate transfer */
	
	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_MASTER_RX_MODE); // 0xB0
	// put address to shift register
	I2C_Outp8(I2C_REG[ch].DS, cSlaveAddr & 0xFE);
	// master rx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_RX_MODE); // 0xB0

	//waiting DDC status
	while(gI2C_EventStatus == EVENT_I2C_NO);

	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}

	return OK;
}

/**
* {I2C_Master_Interrupt_Write}
*
* @brief			Write data through I2C 
* 
* @param[in]	( u32 ch, u8 cSlaveAddr,  u8 *pData, s32 uDataLen)
* @return		s32 On success, return OK;Otherwise return ERROR.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Master_Interrupt_Write( u32 ch, u8 cSlaveAddr,  u8 *pData, s32 uDataLen)
{
	u32 uTmp1 = 0;
	
	uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);

	while( (uTmp1 & I2CSTAT_START ) || (uTmp1 & I2CSTAT_ARBITER) )	//	Wait until I2C bus is free or Arbitration successful.
	{
		uTmp1 = I2C_Inp32(I2C_REG[ch].STAT);
	}
	
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_MASTER_TX_ADDR_STATE;

	// set parameters
	I2C_Context.buffer = pData;
	I2C_Context.bytes = uDataLen;


	/* initiate transfer */

	// set clock, enable interrupts and ack generation
	I2C_Outp8(I2C_REG[ch].CON, (I2C_Inp8(I2C_REG[ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_ACK_ENABLE); // 0xE0
	// to make IDSR write-enabled
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_MASTER_TX_MODE); // 0xD0
	// put address to shift register
	I2C_Outp8(I2C_REG[ch].DS, cSlaveAddr & 0xFE);
	// master tx mode + start + enable - "START" sequence generation
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_TX_MODE); // 0xF0

	
	//waiting status
	while(gI2C_EventStatus == EVENT_I2C_NO);

	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}
	
	return OK;
}

/**
* {I2C_Ch2_Master_Polling_Read_EEPROM}
*
* @brief			EEPRPOM random address Byte read.
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 *cData)
* @return		u32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Ch2_Master_Polling_Read_EEPROM(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 *cData)
{
	s32	iRetval = OK;
	s32 uDataLen = 0;
	
	u8 cD[2];

	cD[0] = (cAddr & 0xff00) >> 8;	//MSB First Address
	cD[1] = cAddr & 0xff;				//LSB Last Address 	

	/* set read address into eeprom to read */
	iRetval = I2C_Master_Polling_Write(ch, cSlaveAddr, &cD[0], 2);// following EEPROM random address access procedure
	if( iRetval != OK )
		return	iRetval;

#ifndef v210_I2C_FPGA
	//for EEPROM internal writing cycle max 10ms(typical 3.5ms)
	Delay(35); // SMDK 2009.07.17
#endif

	if( gbI2C_CH0_Performance_Test == TRUE )
	{
		uDataLen = 2;
	}
	else
	{
		uDataLen = 1;
	}
	
	/* read data in eeprom address set by above */
	iRetval = I2C_Master_Polling_Read(ch, cSlaveAddr, cData, uDataLen);

	return iRetval;
}

/**
* {I2C_Ch2_Master_Polling_Write_EEPROM}
*
* @brief			EEPRPOM random address 2 Bytes Write.
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 cData)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Ch2_Master_Polling_Write_EEPROM(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 cData)
{
	s32 iRetval = OK;
	
	u8 cD[3];

	cD[0] = (cAddr & 0xff00) >> 8;	//MSB First Address
	cD[1] = cAddr & 0xff;				//LSB Last Address
	cD[2] = cData;
	
	iRetval = I2C_Master_Polling_Write(ch, cSlaveAddr, cD, 3);

#ifndef v210_I2C_FPGA
	//for EEPROM internal writing cycle max 10ms(typical 3.5ms)
	Delay(35); // SMDK 2009.07.17
#endif
	
	return iRetval;
}

/**
* {I2C_Master_Polling_Read_EEPROM}
*
* @brief			EEPRPOM random address Byte read.
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 *cData)
* @return		u32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Ch0_Master_Polling_Read_EEPROM(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 *cData)
{
	s32 iRetval = OK;

	/* set read address into eeprom to read */
	iRetval = I2C_Master_Polling_Write(ch, cSlaveAddr, &cAddr, 1);// following EEPROM random address access procedure
	if( iRetval != OK )
		return	iRetval;
	
	/* read data in eeprom address set by above */
	iRetval = I2C_Master_Polling_Read(ch, cSlaveAddr, cData, 1);
	return iRetval;
}

/**
* {I2C_Master_Polling_Write_EEPROM}
*
* @brief			EEPRPOM random address 2 Bytes Write.
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 cData)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Ch0_Master_Polling_Write_EEPROM(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 cData)
{
	s32 iRetval = OK;
	
	u8 cD[2];

	cD[0]= cAddr;
	cD[1]= cData;
	
	iRetval = I2C_Master_Polling_Write(ch, cSlaveAddr, cD, 2);

	return iRetval;
}

/**
* {I2C_Ch2_Master_Interrupt_Read_EEPROM}
*
* @brief			EEPRPOM random address Byte read.
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 *cData)
* @return		u32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Ch2_Master_Interrupt_Read_EEPROM(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 *cData)
{
	s32	iRetval = OK;
	s32 uDataLen = 0;
	
	u8 cD[2];

	cD[0] = (cAddr & 0xff00) >> 8;	//MSB First Address
	cD[1] = cAddr & 0xff;				//LSB Last Address 
	
	/* set read address into eeprom to read */
	iRetval = I2C_Master_Interrupt_Write(ch, cSlaveAddr, &cD[0], 2);// following EEPROM random address access procedure	
	if( iRetval != OK )
		return	iRetval;

#ifndef v210_I2C_FPGA
	//for EEPROM internal writing cycle max 10ms(typical 3.5ms)
	Delay(35); // SMDK 2009.07.17
#endif
	
	/* read data in eeprom address set by above */
	iRetval = I2C_Master_Interrupt_Read(ch, cSlaveAddr, cData, 1);
	
	return iRetval;
}

/**
* {I2C_Ch2_Master_Interrupt_Write_EEPROM}
*
* @brief			EEPRPOM random address 2 Bytes Write.
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 cData)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Ch2_Master_Interrupt_Write_EEPROM(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 cData)
{
	s32 iRetval = OK;
	
	u8 cD[3];

	cD[0] = (cAddr & 0xff00) >> 8;		//MSB First Address
	cD[1] = cAddr & 0xff;					//LSB Last Address
	cD[2] = cData;
	
	iRetval = I2C_Master_Interrupt_Write(ch, cSlaveAddr, cD, 3);

#ifndef v210_I2C_FPGA
	//for EEPROM internal writing cycle max 10ms(typical 3.5ms)
	Delay(35); // SMDK 2009.07.17
#endif

	return iRetval;
}

/**
* {I2C_Ch0_Master_Interrupt_Read_EEPROM}
*
* @brief			EEPRPOM random address Byte read.
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 *cData)
* @return		u32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Ch0_Master_Interrupt_Read_EEPROM(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 *cData)
{
	s32	iRetval = OK;
	
	/* set read address into eeprom to read */
	iRetval = I2C_Master_Interrupt_Write(ch, cSlaveAddr, &cAddr, 1);// following EEPROM random address access procedure	
	if( iRetval != OK )
		return	iRetval;
	
	/* read data in eeprom address set by above */
	iRetval = I2C_Master_Interrupt_Read(ch, cSlaveAddr, cData, 1);
	
	return	iRetval;
}

/**
* {I2C_Master_Interrupt_Write_EEPROM}
*
* @brief			EEPRPOM random address 2 Bytes Write.
* 
* @param[in]	(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 cData)
* @return		u8 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Ch0_Master_Interrupt_Write_EEPROM(u32 ch, u8 cSlaveAddr, u8 cAddr, u8 cData)
{
	s32 iRetval = OK;
	
	u8 cD[2];

	cD[0]= cAddr;
	cD[1]= cData;
	
	iRetval = I2C_Master_Interrupt_Write(ch, cSlaveAddr, cD, 2);

	return iRetval;
}

/**
* {I2C_SDAClockDelay}
*
* @brief			SDA line is delayed as clock time(PCLK).
* 				00 : 0 clock, 	01: 5 clocks.
*				10 : 10 clock, 	11: 15 clocks.
* @param[in]	(u32 ch, u32 PLCKmode, u32 Prescaler).
* @return		VOID.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void	I2C_SDAClockDelay( u32 ch, u8 sda_clk_delay )
{
	I2C_Outp8(I2C_REG[ch].LC,  I2CLC_FILTER_ENABLE | (sda_clk_delay & 0x03) );
}

/**
* {I2C_ClockPrescaler}
*
* @brief			Set I2CCON[6] & I2CCON[3:0].
* @description	If 'Tx_Clock' is greater than 500 KHz, it returns error.
				The marginal Speed is given up to 500 KHz for test, Guaranteed Max Speed is 400 KHz though.
* @param[in]	(u32 ch, u32 PLCKmode, u32 Prescaler)
* @return		s32.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_ClockPrescaler(u32 ch, u32 PLCKmode, u32 Prescaler)
{
	s32	iRetValue = OK;
	
	u32 I2CCLK = 0;
	u32 Tx_Clock = 0;
	
	I2C_Outp8(I2C_REG[ch].CON, (PLCKmode<<6) | (Prescaler&0xF));

	if( ch == I2C_CH0 || ch == I2C_CH2 )
	{
		if( PLCKmode == 0 )
			I2CCLK = g_uPclkPsys / 16;
		else
			I2CCLK = g_uPclkPsys / 512;
	}
	else if( ch == I2C_CH_HDMI_DDC )
	{
		if( PLCKmode == 0 )
			I2CCLK = g_uPclkDsys / 16;
		else
			I2CCLK = g_uPclkDsys / 512;		
	}
	else
		Assert(0);
	
	Tx_Clock = I2CCLK / ( Prescaler + 1 );
	
	if( ch == I2C_CH0 || ch == I2C_CH2 )
		UART_Printf("\n[*** PCLK= %.2f MHz, I2CCLK = %.2f MHz, I2CSCL(Tx clock) = %.2f KHz]",(float)g_uPclkPsys/1.0e6, I2CCLK/1.0e6, Tx_Clock/1.0e3);
	else if( ch == I2C_CH_HDMI_DDC )
		UART_Printf("\n[*** PCLK= %.2f MHz, I2CCLK = %.2f MHz, I2CSCL(Tx clock) = %.2f KHz]",(float)g_uPclkDsys/1.0e6, I2CCLK/1.0e6, Tx_Clock/1.0e3);
	
	/* If 'Tx_Clock' exceed 500 KHz, we can't guarantee it works properly */
	if( Tx_Clock > I2C_MAX_FREQ )
		iRetValue = ERROR;
	
	return	iRetValue;
}

/**
* {I2C_Ch0_TxClockPrescaler}
*
* @brief			Given frequency, determine I2CCON[6] & I2CCON[3:0].
* @description	If 'Tx_Clock' is greater than 500 KHz, it returns error.
				The marginal Speed is given up to 500 KHz for test, Guaranteed Max Speed is 400 KHz though.
* @param[in]	(u32 ch, u32 ufreq)
* @return		VOID.
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_TxClockPrescaler(u32 ch, u32 ufreq)
{
	s32	iRetValue = OK;
	
	u32	uSelClkSrc;
	u32	uClkValue;

	UART_Printf("\n*** I2C0 Setting Frequency: %d Hz\n", ufreq);

	/* Update 'g_uPclkPsys' */
	ClockInform();

	if( ch == I2C_CH0 || ch == I2C_CH2 )
	{
		if ( ( ( (g_uPclkPsys>>4) / ufreq) - 1) > 0xF )
		{
			uSelClkSrc = 1;
			uClkValue =	((g_uPclkPsys>>9) / ufreq) - 1;		//	PCLK/512/freq
			UART_Printf("*** PCLK = %.2fMHz, SelClkSrc= %d (I2CCLK=PCLK/512), ClkDivideValue= %d\n",(float)g_uPclkPsys/1.0e6,uSelClkSrc,uClkValue);		
			UART_Printf("*** I2CCLK = %d Hz, I2CSCL(Tx clock) = %d Hz\n",(g_uPclkPsys/512),((g_uPclkPsys/512)/(uClkValue+1)));
		}
		else 
		{
			uSelClkSrc = 0;
			uClkValue = ( (g_uPclkPsys>>4) / ufreq) - 1;		//	PCLK/16/freq
			UART_Printf("*** PCLK = %.2fMHz, SelClkSrc = %d (I2CCLK=PCLK/16), ClkDivideValue= %d\n",(float)g_uPclkPsys/1.0e6,uSelClkSrc,uClkValue);
			UART_Printf("*** I2CCLK = %d Hz, I2CSCL(Tx clock) = %d Hz\n",(g_uPclkPsys/16),((g_uPclkPsys/16)/(uClkValue+1)));
		}
	}
	else if( I2C_CH_HDMI_DDC )
	{
		if ( ( ( (g_uPclkDsys>>4) / ufreq) - 1) > 0xF )
		{
			uSelClkSrc = 1;
			uClkValue =	((g_uPclkDsys>>9) / ufreq) - 1;		//	PCLK/512/freq
			UART_Printf("*** PCLK = %.2fMHz, SelClkSrc= %d (I2CCLK=PCLK/512), ClkDivideValue= %d\n",(float)g_uPclkDsys/1.0e6,uSelClkSrc,uClkValue);		
			UART_Printf("*** I2CCLK = %d Hz, I2CSCL(Tx clock) = %d Hz\n",(g_uPclkDsys/512),((g_uPclkDsys/512)/(uClkValue+1)));
		}
		else 
		{
			uSelClkSrc = 0;
			uClkValue = ( (g_uPclkDsys>>4) / ufreq) - 1;		//	PCLK/16/freq
			UART_Printf("*** PCLK = %.2fMHz, SelClkSrc = %d (I2CCLK=PCLK/16), ClkDivideValue= %d\n",(float)g_uPclkDsys/1.0e6,uSelClkSrc,uClkValue);
			UART_Printf("*** I2CCLK = %d Hz, I2CSCL(Tx clock) = %d Hz\n",(g_uPclkDsys/16),((g_uPclkDsys/16)/(uClkValue+1)));
		}			
	}
	else
		Assert(0);
	
	//Prescaler I2CCLK = PCLK/16, Transmit clock value Tx clock = I2CCLK/16
	iRetValue = I2C_ClockPrescaler(ch, uSelClkSrc, uClkValue);
	/* If 'Tx_Clock' exceed 400 KHz, we can't guarantee it works properly */
	
	return	iRetValue;
}


/**
* {I2C_Ch0_Master_Interrupt_Init}
*
* @brief			Init 
* 
* @param[in]	(u32 ch)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_DeInitialize_Controller(u32 ch)
{
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_IDLE);// I2C bus data output disable(Rx/Tx)

	return OK;
}

/**
* {I2C_Ch0_Master_Interrupt_Init}
*
* @brief			Init 
* 
* @param[in]	(u32 ch)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Initialize_Controller(u32 ch)
{
	// I2C Filter Setting
	I2C_Outp8(I2C_REG[ch].LC, I2CLC_FILTER_ENABLE);

	// initialize controller
	// set clock, disable interrupts
	I2C_Outp8(I2C_REG[ch].CON, I2CCON_CLK);
	// disable 
	I2C_Outp8(I2C_REG[ch].STAT, I2CSTAT_IDLE);

	return OK;
}

/**
* {I2C_GPIO_Close}
*
* @brief			This function disable I2C 
* 
* @param[in]	(u32 ch)
* @return		VOID 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void I2C_GPIO_Close(u32 ch)
{
	switch( ch )
	{
		case I2C_CH0:
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_0, 0);
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_1, 0);	
			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_0, 0);
			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_1, 0);
			break;
			
		case I2C_CH2:
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_4, 0);
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_5, 0);
			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_4, 0);
			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_5, 0);
			break;
			
		case I2C_CH_HDMI_DDC:
			break;
			
		case I2C_CH_HDMI_PHY:
			break;
			
		default:
			break;
	}
	
}

/**
* {I2C_GPIO_Open}
*
* @brief			Init 
* 
* @param[in]	(u32 ch)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_GPIO_Open(u32 ch)
{
	switch( ch )
	{
		case I2C_CH0:
			// I2C CH0 GPIO Setting
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_0, 2);	// GPD1CON[0] -> I2C0_SDA
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_1, 2);	// GPD1CON[1] -> I2C0_SCL

			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_0, 0);	// Pull-Up/Down Disable
			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_1, 0);	// Pull-Up/Down Disable		
			break;
			
		case I2C_CH2:
			// I2C CH2 GPIO Setting
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_4, 2);	// GPD1CON[4] -> I2C2_SDA
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_5, 2);	// GPD1CON[5] -> I2C2_SCL
			
			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_4, 0);	// Pull-Up/Down Disable
			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_5, 0);	// Pull-Up/Down Disable

			// PMIC GPIO Setting
			GPIO_SetFunctionEach(eGPIO_H1, eGPIO_6, 1);
			GPIO_SetFunctionEach(eGPIO_H1, eGPIO_7, 1); // PMIC Set0,1 Pin(xeint14,15 output setting)
			GPIO_SetFunctionEach(eGPIO_H0, eGPIO_4, 1); // PMIC Set2 Pin(xeint4 output setting)
			break;
			
		case I2C_CH_HDMI_DDC:
			// I2C DDC Channel GPIO Setting
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_2, 2);	// GPD1CON[2] -> I2C0_SDA
			GPIO_SetFunctionEach(eGPIO_D1, eGPIO_3, 2);	// GPD1CON[3] -> I2C0_SCL

			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_2, 0);	// Pull-Up/Down Disable
			GPIO_SetPullUpDownEach(eGPIO_D1, eGPIO_3, 0);	// Pull-Up/Down Disable		
			break;
			
		case I2C_CH_HDMI_PHY:
			break;
						
		default:
			break;
	}

	return OK;
}

/**
* {I2C_UnRegister_Interrupt}
*
* @brief			Init 
* 
* @param[in]	(u32 ch)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_UnRegister_Interrupt(u32 ch)
{
	switch( ch )
	{
		case I2C_CH0:
			INTC_Disable(NUM_I2C);
			break;
			
		case I2C_CH2:
			INTC_Disable(NUM_I2C_PMIC);
			break;
			
		case I2C_CH_HDMI_DDC:
			INTC_Disable(NUM_HDMI_I2C);			
			break;
			
		case I2C_CH_HDMI_PHY:
			break;
					
		default:
			break;
	}
	
	return OK;
}

/**
* {I2C_Register_Interrupt}
*
* @brief			Init 
* 
* @param[in]	(u32 ch)
* @return		s32 
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_Register_Interrupt(u32 ch)
{
	switch( ch )
	{
		case I2C_CH0:
			INTC_SetVectAddr(NUM_I2C, I2C_IRQHandler);
			INTC_Enable(NUM_I2C);
			break;
			
		case I2C_CH2:
			INTC_SetVectAddr(NUM_I2C_PMIC, I2C_IRQHandler);
			INTC_Enable(NUM_I2C_PMIC);	
			break;
			
		case I2C_CH_HDMI_DDC:
			INTC_SetVectAddr(NUM_HDMI_I2C, I2C_IRQHandler);
			INTC_Enable(NUM_HDMI_I2C);
			break;
			
		case I2C_CH_HDMI_PHY:
			INTC_SetVectAddr(NUM_I2C_HDMI, I2C_IRQHandler);
			INTC_Enable(NUM_I2C_HDMI);
			break;
						
		default:
			break;
	}
	
	return OK;
}


/*******************************************************************************
* GLOBAL VARIABLES
*******************************************************************************/
/* For LoopBack Test between Channel 0 and Channel 2 in Evaluation B'd : 2009.07.25 */
u32		gI2c_Tx_Ch = 0;
u32		gI2c_Rx_Ch = 0;
/* For LoopBack Test between Channel 0 and Channel 2 in Evaluation B'd : 2009.07.25 */

void	I2C_ACK_Check_Polling(u32 ch)
{
	u32	uTmp, uCnt = 0;
		
	uTmp = I2C_Inp32(I2C_REG[ch].CON);	
	while( (uTmp & I2CCON_INT_PEND) != I2CCON_INT_PEND )  //Interrupt Pending (ACK check)
	{
		uTmp = I2C_Inp32(I2C_REG[ch].CON);
		uCnt++;
		if (uCnt > 0x10000) {
			UART_Printf("\n[Warning] ACK Checktime Overflow..!!!!(Polling)\n");
			break;
		}
	}
}

void I2C_Pending_Bit_Clear(u32 ch)
{
	u32 uTmp;
	uTmp = I2C_Inp32(I2C_REG[ch].CON);
	uTmp &= ~(I2CCON_INT_PEND);		//	Clear pending bit to resume
	I2C_Outp8(I2C_REG[ch].CON, uTmp);			
}

/**
* {I2C_SlaveAddressMatch_Polling}
*
* @brief			This function do Master TX->SlaveRx mode by polling operation with tx_ch,rx_ch.
* 				Master TX -->   Slave RX	
* @param[in]	(u32 ch)
* @return		void
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void	I2C_SlaveAddressMatch_Polling(u32 ch)
{
	u32	uTmp;
	u8	uData;

	/* Once 'I2CSTAT' is read, Slave Address Match flag will be cleared */
	
	uTmp = I2C_Inp32(I2C_REG[ch].STAT);
	while( !( uTmp & I2CSTAT_ADDR_SLAVE ) )		// Wait Slave Address Match.
		uTmp = I2C_Inp32(I2C_REG[ch].STAT);

	uData = I2C_Inp8(I2C_REG[ch].DS);			// Read data from I2CDS.

	I2C_Pending_Bit_Clear(ch);
}

/**
* {I2C_MTx_IRQHandler}
*
* @brief			Interrupt Handler between Channel 0 & Channel 2.
* @description	For Two Channel LoopBack Test in Evaluation B'd.
* @param[in]	VOID
* @return		VOID 
* @version		2009.07.25
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void __irq I2C_MTx_IRQHandler(void)
{
	u8		ucStatus;
	u8		ucReg;
	s32		sCont = TRUE;	
	bool	RxDone = false;
	
	// check interrupt reason
	ucStatus = I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON);

	if(ucStatus & I2CCON_INT_PEND)
	{
		// check Status flags
		ucReg = I2C_Inp8(I2C_REG[gI2c_Tx_Ch].STAT);

		while (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_SLAVE | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR) )
		{
			sCont = FALSE;
			if (ucReg & I2CSTAT_ARBITER)
			{
				UART_Printf ("I2CSTAT_ARBITER\n");
			}
			else if (ucReg & I2CSTAT_ADDR_SLAVE)
			{
				UART_Printf ("I2CSTAT_ADDR_SLAVE\n");
				sCont = TRUE;
				break;
			}
			else if (ucReg & I2CSTAT_ADDR_ZERO)
			{
				UART_Printf ("I2CSTAT_ADDR_ZERO\n");
			}
			else if (ucReg & I2CSTAT_ACK_ERROR)
			{
				if (	I2C_Context.eState == I2C_MASTER_TX_STOP_STATE )
				{
					sCont = TRUE;
					break;
				}
				else
				{
					UART_Printf ("I2C_STAT_ACKERROR : %d\n", I2C_Context.eState);
				}
			}

			I2C_Release(gI2c_Tx_Ch);

			gI2C_EventStatus = EVENT_I2C_ERROR;
			break;
		}

		if (sCont)
		{
			switch (I2C_Context.eState)
			{
				case I2C_MASTER_TX_ADDR_STATE:

					// Slave RX check
					RxDone = false;
					while( !RxDone )
					{
						if( I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & I2CCON_INT_PEND )
						{
							RxDone = true;
							I2C_Pending_Bit_Clear(gI2c_Rx_Ch);	//	RX : Clear pending bit to resume
						}
					}
					
					I2C_Context.eState = I2C_MASTER_TX_DATA_STATE;

					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;
					--(I2C_Context.bytes);
			
					if(I2C_Context.bytes > 0)
					{
						//Clear pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE); // 0xE0
					}
					else
					{
						I2C_Context.eState = I2C_MASTER_TX_STOP_STATE;
						I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR); // 0x60
					}
					break;
					
				case I2C_MASTER_TX_DATA_STATE:

					// Slave RX check
					RxDone = false;
					while( !RxDone )
					{
						if( I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & I2CCON_INT_PEND )
						{
							*(I2C_Context.Rx_buffer) = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].DS);
							I2C_Context.Rx_buffer++;
							RxDone = true;
							I2C_Pending_Bit_Clear(gI2c_Rx_Ch);	//	RX : Clear pending bit to resume
						}
					}
					
					I2C_Context.eState = I2C_MASTER_TX_DATA_STATE;

					// write data to DS
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;
					--(I2C_Context.bytes);
			
					if(I2C_Context.bytes > 0)
					{
						//Clear pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE); // 0xE0
					}
					else
					{
						I2C_Context.eState = I2C_MASTER_TX_STOP_STATE;
						I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR); // 0x60
					}
					break;

				case I2C_MASTER_TX_STOP_STATE:
					// Slave RX check
					RxDone = false;
					while( !RxDone )
					{
						if( I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & I2CCON_INT_PEND ) 
						{
							*(I2C_Context.Rx_buffer) = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].DS);
							I2C_Context.Rx_buffer++;
							RxDone = true;
							I2C_Pending_Bit_Clear(gI2c_Rx_Ch);	//	RX : Clear pending bit to resume
						}
					}
					
					I2C_Context.eState = I2C_IDLE_STATE;

					// Master Tx Stop
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].STAT, I2CSTAT_MASTER_TX_MODE | I2CSTAT_ENABLE); // 0xD0					
					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);// 0x60


					// Slave Rx Stop
					I2C_Outp8(I2C_REG[gI2c_Rx_Ch].STAT, I2CSTAT_IDLE);	 //	Stop Slave Rx condition
					I2C_Pending_Bit_Clear(gI2c_Rx_Ch);	//	RX : Clear pending bit to resume

					// wait til the stop condition takes effect.
					while( I2C_Inp8(I2C_REG[gI2c_Tx_Ch].STAT) & I2CSTAT_START );

					gI2C_EventStatus = EVENT_I2C_DONE;
					break;

				case I2C_IDLE_STATE:
				default:
					UART_Printf ("Unknown state!!!\n");
					gI2C_EventStatus = EVENT_I2C_ERROR;
			}
		}
	}
	else
	{
		UART_Printf ("*** UNHANDLED ***\n");
	}

	INTC_ClearVectAddr();
}

/**
* {I2C_MRx_IRQHandler}
*
* @brief			Interrupt Handler between Channel 0 & Channel 2.
* @description	For Two Channel LoopBack Test in Evaluation B'd.
* @param[in]	VOID
* @return		VOID 
* @version		2009.07.25
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void __irq I2C_MRx_IRQHandler(void)
{
	u8		ucStatus;
	u8		ucReg;
	s32		sCont = TRUE;	
	bool	RxDone = false;
	
	// check interrupt reason
	ucStatus = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON);

	if(ucStatus & I2CCON_INT_PEND)
	{
		// check Status flags
		ucReg = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].STAT);

		while (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_SLAVE | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR) )
		{
			sCont = FALSE;
			if (ucReg & I2CSTAT_ARBITER)
			{
				UART_Printf ("I2CSTAT_ARBITER\n");
			}
			else if (ucReg & I2CSTAT_ADDR_SLAVE)
			{
				UART_Printf ("I2CSTAT_ADDR_SLAVE\n");
				sCont = TRUE;
				break;
			}
			else if (ucReg & I2CSTAT_ADDR_ZERO)
			{
				UART_Printf ("I2CSTAT_ADDR_ZERO\n");
			}
			else if (ucReg & I2CSTAT_ACK_ERROR)
			{
				if (	I2C_Context.eState == I2C_MASTER_RX_STOP_STATE )
				{
					sCont = TRUE;
					break;
				}
				else
				{
					UART_Printf ("I2C_STAT_ACKERROR : %d\n", I2C_Context.eState);
				}
			}

			I2C_Release(gI2c_Rx_Ch);

			gI2C_EventStatus = EVENT_I2C_ERROR;
			break;
		}

		if (sCont)
		{
			switch (I2C_Context.eState)
			{
				case I2C_MASTER_RX_ADDR_STATE:				
					I2C_Context.eState = I2C_MASTER_RX_DATA_STATE;

					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;

						// clear interrupt pending bit  to resume : Master Rx
						I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );
					}
					else
					{
						// clear interrupt pending bit  to resume : Master Rx
						I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
					}
					
					// Slave Tx Write Data to I2CDS
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;

					// Slave Tx Clear pending bit to resume
					I2C_Pending_Bit_Clear(gI2c_Tx_Ch);
					I2C_ACK_Check_Polling(gI2c_Tx_Ch);
					break;
					
				case I2C_MASTER_RX_DATA_STATE:
					// read byte
					*(I2C_Context.Rx_buffer) = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].DS);
					I2C_Context.Rx_buffer++;
					--(I2C_Context.bytes);
					        
					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;
						
						//slave address   next data reading NO ACK߻ϱ  ̸ disableŴ
						// Disable Ack generation (for EEPROM reading)							
						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR ); //0x60
					}
					else
					{
						// clear interrupt pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE ); // 0xE0
					}

					// Slave Tx Write Data to I2CDS
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;

					// Slave Tx Clear pending bit to resume
					I2C_Pending_Bit_Clear(gI2c_Tx_Ch);					
					break;

				case I2C_MASTER_RX_STOP_STATE:
					I2C_Context.eState = I2C_IDLE_STATE;

					*(I2C_Context.Rx_buffer) = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].DS);

					I2C_Outp8(I2C_REG[gI2c_Rx_Ch].STAT, I2CSTAT_MASTER_RX_MODE | I2CSTAT_ENABLE); //0x90
					
					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
					
					// Master Rx Stop set to "not busy" - "STOP" sequence generation
					I2C_Outp8(I2C_REG[gI2c_Rx_Ch].STAT, I2CSTAT_MASTER_RX_MODE); // 0x80

					// Slave Tx Stop
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].STAT, I2CSTAT_IDLE);	 //	Stop Slave Tx condition
					I2C_Pending_Bit_Clear(gI2c_Tx_Ch);	//	Tx : Clear pending bit to resume

					// wait til the stop condition takes effect.
					while( I2C_Inp8(I2C_REG[gI2c_Rx_Ch].STAT) & I2CSTAT_START );
					
					gI2C_EventStatus = EVENT_I2C_DONE;
					break;

				case I2C_IDLE_STATE:
				default:
					UART_Printf ("Unknown state!!!\n");
					gI2C_EventStatus = EVENT_I2C_ERROR;
			}
		}
	}
	else
	{
		UART_Printf ("*** UNHANDLED ***\n");
	}

	INTC_ClearVectAddr();
}	

/**
* {I2C_STx_IRQHandler}
*
* @brief			Interrupt Handler between Channel 0 & Channel 2.
* @description	For Two Channel LoopBack Test in Evaluation B'd.
* @param[in]	VOID
* @return		VOID 
* @version		2009.07.25
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
void __irq I2C_STx_IRQHandler(void)
{
	u8		ucStatus;
	u8		ucReg;
	s32		sCont = TRUE;	
	bool	RxDone = false;
	
	// check interrupt reason
	ucStatus = I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON);

	if(ucStatus & I2CCON_INT_PEND)
	{
		// check Status flags
		ucReg = I2C_Inp8(I2C_REG[gI2c_Tx_Ch].STAT);

		while (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_SLAVE | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR) )
		{
			sCont = FALSE;
			if (ucReg & I2CSTAT_ARBITER)
			{
				UART_Printf ("I2CSTAT_ARBITER\n");
			}
			else if (ucReg & I2CSTAT_ADDR_SLAVE)
			{
				UART_Printf ("I2CSTAT_ADDR_SLAVE\n");
				sCont = TRUE;
				break;
			}
			else if (ucReg & I2CSTAT_ADDR_ZERO)
			{
				UART_Printf ("I2CSTAT_ADDR_ZERO\n");
			}
			else if (ucReg & I2CSTAT_ACK_ERROR)
			{
				if (	I2C_Context.eState == I2C_SLAVE_TX_STOP_STATE )
				{
					sCont = TRUE;
					break;
				}
				else
				{
					UART_Printf ("I2C_STAT_ACKERROR : %d\n", I2C_Context.eState);
				}
			}

			I2C_Release(gI2c_Tx_Ch);

			gI2C_EventStatus = EVENT_I2C_ERROR;
			break;
		}

		if (sCont)
		{
			switch (I2C_Context.eState)
			{
				case I2C_SLAVE_TX_ADDR_STATE:
					I2C_Context.eState = I2C_SLAVE_TX_DATA_STATE;

					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;
					--(I2C_Context.bytes);
			
					if(I2C_Context.bytes > 0)
					{
						//Clear pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE); // 0xE0
					}
					else
					{
						I2C_Context.eState = I2C_SLAVE_TX_STOP_STATE;
						I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR); // 0x60
					}

					// Master Rx Clear pending bit to resume
					I2C_Pending_Bit_Clear(gI2c_Rx_Ch);
					I2C_ACK_Check_Polling(gI2c_Rx_Ch);
					break;
					
				case I2C_SLAVE_TX_DATA_STATE:
					// Master Rx Read
					*(I2C_Context.Rx_buffer) = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].DS);
					I2C_Context.Rx_buffer++;
					
					
					I2C_Context.eState = I2C_SLAVE_TX_DATA_STATE;

					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;
					--(I2C_Context.bytes);
			
					if(I2C_Context.bytes > 0)
					{
						//Clear pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE); // 0xE0
					}
					else
					{
						I2C_Context.eState = I2C_SLAVE_TX_STOP_STATE;
						I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR); // 0x60
					}

					// Master Rx Clear pending bit to resume
					I2C_Pending_Bit_Clear(gI2c_Rx_Ch);
					I2C_ACK_Check_Polling(gI2c_Rx_Ch);					
					break;

				case I2C_SLAVE_TX_STOP_STATE:
					// Master Rx Read
					*(I2C_Context.Rx_buffer) = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].DS);
					I2C_Context.Rx_buffer++;

					// Master Rx Stop
					I2C_Outp8(I2C_REG[gI2c_Rx_Ch].STAT, I2CSTAT_MASTER_RX_MODE | I2CSTAT_ENABLE); //0x90
					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);					
					// Master Rx Stop set to "not busy" - "STOP" sequence generation
					I2C_Outp8(I2C_REG[gI2c_Rx_Ch].STAT, I2CSTAT_MASTER_RX_MODE); // 0x80

					
					I2C_Context.eState = I2C_IDLE_STATE;

					// Slave Tx Stop
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].STAT, I2CSTAT_IDLE);	 //	Stop Slave Tx condition
					I2C_Pending_Bit_Clear(gI2c_Tx_Ch);	//	Tx : Clear pending bit to resume
					
					// wait til the stop condition takes effect.
					while( I2C_Inp8(I2C_REG[gI2c_Tx_Ch].STAT) & I2CSTAT_START );

					gI2C_EventStatus = EVENT_I2C_DONE;
					break;
					
				case I2C_IDLE_STATE:
				default:
					UART_Printf ("Unknown state!!!\n");
					gI2C_EventStatus = EVENT_I2C_ERROR;
			}
		}
	}
	else
	{
		UART_Printf ("*** UNHANDLED ***\n");
	}

	INTC_ClearVectAddr();
}	

void __irq I2C_SRx_IRQHandler(void)
{
	u8 ucStatus;
	u8 ucReg;
	s32 sCont = TRUE;	
	u32 uTmp;
		
	// check interrupt reason
	ucStatus = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON);

	if(ucStatus & I2CCON_INT_PEND)
	{
		// check Status flags
		ucReg = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].STAT);

		while (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_SLAVE | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR) )
		{
			sCont = FALSE;
			if (ucReg & I2CSTAT_ARBITER)
			{
				UART_Printf ("I2CSTAT_ARBITER\n");
			}
			else if (ucReg & I2CSTAT_ADDR_SLAVE)
			{
				UART_Printf ("I2CSTAT_ADDR_SLAVE\n");
				sCont = TRUE;
				break;
			}
			else if (ucReg & I2CSTAT_ADDR_ZERO)
			{
				UART_Printf ("I2CSTAT_ADDR_ZERO\n");
			}
			else if (ucReg & I2CSTAT_ACK_ERROR)
			{
				if ( I2C_Context.eState == I2C_SLAVE_RX_STOP_STATE )
				{
					sCont = TRUE;
					break;
				}
				else
				{
					UART_Printf ("I2C_STAT_ACKERROR : %d\n", I2C_Context.eState);
				}
			}

			I2C_Release(gI2c_Rx_Ch);

			gI2C_EventStatus = EVENT_I2C_ERROR;
			break;
		}

		if (sCont)
		{
			switch (I2C_Context.eState)
			{					
				case I2C_SLAVE_RX_ADDR_STATE:
					I2C_Context.eState = I2C_SLAVE_RX_DATA_STATE;

					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_SLAVE_RX_STOP_STATE;

						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR );
					}
					else
					{
						// clear interrupt pending bit  to resume
						I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
					}
					
					// Master Tx : write data to DS
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;

					// Master Tx Clear pending bit to resume
					I2C_Pending_Bit_Clear(gI2c_Tx_Ch);
					I2C_ACK_Check_Polling(gI2c_Tx_Ch);
					break;
					
				case I2C_SLAVE_RX_DATA_STATE:
					// read byte
					*(I2C_Context.Rx_buffer) = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].DS);
					I2C_Context.Rx_buffer++;
					--(I2C_Context.bytes);
					        
					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_SLAVE_RX_STOP_STATE;
						
						//slave address   next data reading NO ACK߻ϱ  ̸ disableŴ
						// Disable Ack generation (for EEPROM reading)
						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR ); //0x60
					}
					else
					{
						// clear interrupt pending bit to resume
						I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Rx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE| I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE ); // 0xE0
					}
					
					// Master Tx : write data to DS
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].DS, *(I2C_Context.buffer));
					I2C_Context.buffer++;

					// Master Tx Clear pending bit to resume
					I2C_Pending_Bit_Clear(gI2c_Tx_Ch);
					I2C_ACK_Check_Polling(gI2c_Tx_Ch);					
					break;
					
				case I2C_SLAVE_RX_STOP_STATE:
					I2C_Context.eState = I2C_IDLE_STATE;

					*(I2C_Context.Rx_buffer) = I2C_Inp8(I2C_REG[gI2c_Rx_Ch].DS);

					// Slave Rx Stop
					I2C_Outp8(I2C_REG[gI2c_Rx_Ch].STAT, I2CSTAT_IDLE);	 //	Stop Slave Rx condition

					//I2C_Pending_Bit_Clear(gI2c_Rx_Ch);	//	Rx : Clear pending bit
					uTmp = I2C_Inp32(I2C_REG[gI2c_Rx_Ch].CON);
					uTmp &= ~(I2CCON_INT_PEND);		//	Clear pending bit to resume
					I2C_Outp8(I2C_REG[gI2c_Rx_Ch].CON, uTmp);

					// Master Tx Stop
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].STAT, I2CSTAT_MASTER_TX_MODE | I2CSTAT_ENABLE); // 0xD0					
					// clear interrupt pending bit
					I2C_Outp8(I2C_REG[gI2c_Tx_Ch].CON, (I2C_Inp8(I2C_REG[gI2c_Tx_Ch].CON) & (I2CCON_CLK | I2CCON_PRESCALER ) ) | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);// 0x60

					// wait til the stop condition takes effect.
					while( I2C_Inp8(I2C_REG[gI2c_Rx_Ch].STAT) & I2CSTAT_START );
					
					gI2C_EventStatus = EVENT_I2C_DONE;					
					break;
					
				case I2C_IDLE_STATE:
				default:
					UART_Printf ("Unknown state!!!\n");
					gI2C_EventStatus = EVENT_I2C_ERROR;
			}
		}
	}
	else
	{
		UART_Printf ("*** UNHANDLED ***\n");
	}

	INTC_ClearVectAddr();
}


/*******************************************************************************
 * HDMI PHY : START
 ******************************************************************************/
/**
* {I2C_HDMI_PHY_Release}
*
* @brief			 Release I2C line
* 
* @param[in]	void
* @return		s32
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
static s32 I2C_HDMI_PHY_Release(void)
{
	// clear interrupt pending flag and interrupt clear bit
	I2C_Outp8(rI2CCON_HDMI_PHY, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
	// disable 
	I2C_Outp8(rI2CSTAT_HDMI_PHY, I2CSTAT_IDLE);

	return OK;
}

/**
* {I2C_HDMI_PHY_InterruptWait}
*
* @brief			 Polling interrupt. On success, return OK;Otherwise return, ERROR.
* 
* @param[in]	void
* @return		s32
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
static s32 I2C_HDMI_PHY_InterruptWait(void)
{
	u8 ucStatus, ucReg;
	s32 retval = OK;

	do
	{
		ucStatus = I2C_Inp8(rI2CCON_HDMI_PHY);
		if (ucStatus & I2CCON_INT_PEND)
		{
			// check status flags
			ucReg = I2C_Inp8(rI2CSTAT_HDMI_PHY);

			if (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_SLAVE | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR))
			{
				if (ucReg & I2CSTAT_ARBITER)
				{
					UART_Printf("I2CSTAT_ARBITER\n");
				}
				else if (ucReg & I2CSTAT_ADDR_SLAVE)
				{
					UART_Printf("I2CSTAT_ADDR_SLAVE\n");
				}
				else if (ucReg & I2CSTAT_ADDR_ZERO)
				{
					UART_Printf("I2CSTAT_ADDR_ZERO\n");
				}
				else if (ucReg & I2CSTAT_ACK_ERROR)
				{
					UART_Printf("I2CSTAT_ACK_ERROR\n");
				}
				
				I2C_HDMI_PHY_Release();
				retval = ERROR;
			}
			break;
		}
	}while (1);

	return retval;
}

/**
* {I2C_HDMI_PHY_Init}
*
* @brief			  Initialize I2C library
* 
* @param[in]	void
* @return		s32
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_HDMI_PHY_Init(void)
{
	UART_Printf("[I2C_HDMI_PHY_Init]\n");

	gI2c_CH_Index = I2C_CH_HDMI_PHY;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);
	
	return OK;
}

/**
* {I2C_HDMI_PHY_DeInit}
*
* @brief			Deinitialize I2C library
* @param[in]	VOID
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/  
s32 I2C_HDMI_PHY_DeInit(void)
{
	u32 ch = gI2c_CH_Index;
	Assert(gI2c_CH_Index == I2C_CH_HDMI_PHY);
	
	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);
	gI2c_CH_Index = I2C_CH0;
	
	return OK;
}

/**
* {I2C_HDMI_PHY_Read}
*
* @brief			Read data through I2C
* @param[in]	VOID
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/   
s32	I2C_HDMI_PHY_Read(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	iRetval = I2C_Master_Polling_Read( I2C_CH_HDMI_PHY, ucAddr, pBuffer, (s32) ucBytes );
	return iRetval;
}

/**
* {I2C_HDMI_PHY_Write}
*
* @brief			Write data through I2C
* @param[in]	VOID
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_HDMI_PHY_Write(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	iRetval = I2C_Master_Polling_Write( I2C_CH_HDMI_PHY, ucAddr, pBuffer, (s32) ucBytes );
	return iRetval;
}
/*******************************************************************************
 * HDMI PHY : END
 ******************************************************************************/

/*******************************************************************************
 * HDMI DDC : START
 ******************************************************************************/
/**
 * Release DDC line
 */
static s32 I2C_HDMI_DDC_Release(void)
{
	// clear interrupt pending flag and interrupt clear bit
	I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
	// disable 
	I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_IDLE);

	return OK;
}

/**
 * DDC interrupt service routine
 */
void __irq I2C_HDMI_DDC_IRQHandler(void)
{
	u8 ucStatus;
	u8 ucReg;
	s32 sCont = TRUE;	

	// check interrupt reason
	ucStatus = I2C_Inp8(rI2CCON_HDMI_DDC);

	if (ucStatus & I2CCON_INT_PEND)
	{
		// check Status flags
		ucReg = I2C_Inp8(rI2CSTAT_HDMI_DDC);

		while (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_SLAVE | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR))
		{
			sCont = FALSE;
			if (ucReg & I2CSTAT_ARBITER)
			{
				UART_Printf ("I2CSTAT_ARBITER\n");
			}
			else if (ucReg & I2CSTAT_ADDR_SLAVE)
			{
				UART_Printf ("I2CSTAT_ADDR_SLAVE\n");
			}
			else if (ucReg & I2CSTAT_ADDR_ZERO)
			{
				UART_Printf ("I2CSTAT_ADDR_ZERO\n");
			}
			else if (ucReg & I2CSTAT_ACK_ERROR)
			{
				if (	I2C_Context.eState == I2C_MASTER_RX_STOP_STATE ||
					I2C_Context.eState == I2C_MASTER_TX_EDDC_SEGADDR_STATE ||
					((I2C_Context.eState == I2C_MASTER_TX_EDDC_SEGNUM_STATE ) && (I2C_Context.segnum == 0)))
				{
					sCont = TRUE;
					break;
				}
				else
				{
					UART_Printf ("I2CSTAT_ACK_ERROR : %d\n", I2C_Context.eState);
				}
			}
			I2C_HDMI_DDC_Release();
			gI2C_EventStatus = EVENT_I2C_ERROR;
			break;
		}

		if (sCont)
		{
			switch (I2C_Context.eState)
			{
				case I2C_MASTER_TX_EDDC_SEGADDR_STATE:
					I2C_Context.eState = I2C_MASTER_TX_EDDC_SEGNUM_STATE;

					// write segment number
					I2C_Outp8(rI2CDS_HDMI_DDC, I2C_Context.segnum);
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
					break;

				case I2C_MASTER_TX_EDDC_SEGNUM_STATE:
					I2C_Context.eState = I2C_MASTER_TX_DDC_ADDR_STATE;

					// put address to shift register
					I2C_Outp8(rI2CDS_HDMI_DDC, I2C_Context.addr & 0xFE);
					// master tx mode + start + enable - "START" sequence generation
					I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_TX_MODE); // 0xF0
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );
					break;

				case I2C_MASTER_TX_DDC_ADDR_STATE:
					I2C_Context.eState = I2C_MASTER_TX_DDC_OFFSET_STATE;

					// write offset
					I2C_Outp8(rI2CDS_HDMI_DDC, I2C_Context.offset);
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
					break;

				case I2C_MASTER_TX_DDC_OFFSET_STATE:
					I2C_Context.eState = I2C_MASTER_RX_DDC_ADDR_STATE;

					// put address to shift register
					I2C_Outp8(rI2CDS_HDMI_DDC, I2C_Context.addr & 0xFE);
					// master rx mode + start + enable - "START" sequence generation
					I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_RX_MODE); // 0xB0
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );
					break;

				case I2C_MASTER_TX_ADDR_STATE:
				case I2C_MASTER_TX_DATA_STATE:
					I2C_Context.eState = I2C_MASTER_TX_DATA_STATE;

					I2C_Outp8(rI2CDS_HDMI_DDC, *(I2C_Context.buffer));
					I2C_Context.buffer++;
					--(I2C_Context.bytes);

					if (I2C_Context.bytes == 0)
					{
						I2C_Context.eState = I2C_MASTER_TX_STOP_STATE;
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
					}
					else
					{
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
					}
					break;

				case I2C_MASTER_TX_STOP_STATE:
					I2C_Context.eState = I2C_IDLE_STATE;

					I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_MASTER_TX_MODE | I2CSTAT_ENABLE); 
					
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);

					while( I2C_Inp8(rI2CSTAT_HDMI_DDC) & (1<<5) );

					gI2C_EventStatus = EVENT_I2C_DONE;
					break;
					
				case I2C_MASTER_RX_DDC_ADDR_STATE:
					I2C_Context.eState = I2C_MASTER_RX_DDC_DATA_STATE;

					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;

						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );  
					}
					else
					{
						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
					}
					break;

				case I2C_MASTER_RX_ADDR_STATE:
					I2C_Context.eState = I2C_MASTER_RX_DATA_STATE;

					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;

						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );  
					}
					else
					{
						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
					}
					break;
					
				case I2C_MASTER_RX_DATA_STATE:
				case I2C_MASTER_RX_DDC_DATA_STATE:
					// read byte
					*(I2C_Context.buffer) = I2C_Inp8(rI2CDS_HDMI_DDC);
					I2C_Context.buffer++;
					--(I2C_Context.bytes);
					        
					if (I2C_Context.bytes == 1)
					{
						I2C_Context.eState = I2C_MASTER_RX_STOP_STATE;

						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR );
					}
					else
					{
						// clear interrupt pending flag and interrupt clear bit
						I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR | I2CCON_ACK_ENABLE);
					}
					break;

				case I2C_MASTER_RX_STOP_STATE:
					I2C_Context.eState = I2C_IDLE_STATE;

					*(I2C_Context.buffer) = I2C_Inp8(rI2CDS_HDMI_DDC);

					I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_MASTER_RX_MODE | I2CSTAT_ENABLE); 
					
					// clear interrupt pending flag and interrupt clear bit
					I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
					
					// disable Rx/Tx and set to "not busy" - "STOP" sequence generation
					I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_MASTER_RX_MODE); // 0x80

					while( I2C_Inp8(rI2CSTAT_HDMI_DDC) & (1<<5) );
					
					gI2C_EventStatus = EVENT_I2C_DONE;
					break;

				case I2C_IDLE_STATE:
				default:
					UART_Printf ("Unknown state!!!\n");
					gI2C_EventStatus = EVENT_I2C_ERROR;
			}
		}
	}
	else
	{
		UART_Printf ("*** UNHANDLED ***\n");
	}

	INTC_ClearVectAddr();
}

/**
 * Initialize DDC library
 *
 * @return  On success, return OK;Otherwise return ERROR.
 */
s32 I2C_HDMI_DDC_Init(void)
{
	gI2c_CH_Index = I2C_CH_HDMI_DDC;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_Register_Interrupt(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);
	
	return OK;
}

/**
* {I2C_HDMI_DDC_DeInit}
*
* @brief			Deinitialize I2C library
* @param[in]	VOID
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/  
s32 I2C_HDMI_DDC_DeInit(void)
{
	u32 ch = gI2c_CH_Index;
	Assert(gI2c_CH_Index == I2C_CH_HDMI_DDC);
	
	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);
	gI2c_CH_Index = I2C_CH0;
	
	return OK;
}

/**
* {I2C_HDMI_DDC_Read}
*
* @brief			Read data through I2C
* @param[in]	(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_HDMI_DDC_Read(u8 addr, u8 bytes, u8 *buffer)
{
	s32 iRetval = OK;
	iRetval = I2C_Master_Interrupt_Read(I2C_CH_HDMI_DDC, addr, buffer, (s32) bytes); 
	return iRetval;
}

/**
* {I2C_HDMI_DDC_Write}
*
* @brief			Write data through I2C
* @param[in]	(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_HDMI_DDC_Write(u8 addr, u8 bytes, u8 *buffer)
{
	s32 iRetval = OK;
	iRetval = I2C_Master_Interrupt_Write(I2C_CH_HDMI_DDC, addr, buffer, (s32) bytes); 
	return iRetval;
}

/**
* {I2C_HDMI_DDC_ReadHDCP}
*
* @brief			Process fast I2C read through DDC
* @param[in]	(u8 addr, u8 offset, u8 bytes, u8 *buffer)
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32	I2C_HDMI_DDC_ReadHDCP(u8 addr, u8 offset, u8 bytes, u8 *buffer)
{
#if 1
	s32 iRetval = OK;
	iRetval = I2C_Master_Interrupt_ReadHDCP(I2C_CH_HDMI_DDC, addr, offset, buffer, (s32) bytes); 
	return iRetval;
#else	
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_MASTER_TX_DDC_ADDR_STATE;

	// set parameters
	I2C_Context.offset = offset;
	I2C_Context.buffer = buffer;
	I2C_Context.bytes = bytes;
	I2C_Context.addr = addr;

	// set clock, enable interrupts and ack generation
	I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_ENABLE); // 0x10
	// put address to shift register
	I2C_Outp8(rI2CDS_HDMI_DDC, addr & 0xFE);

	// master tx mode + start + enable - "START" sequence generation
	I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_TX_MODE); // 0xF0

	//waiting DDC status
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf ("ERROR!!!\n");
		return ERROR;
	}

	return OK;
#endif	
}

/**
* {I2C_HDMI_DDC_ReadEDDC}
*
* @brief			Process E-DDC read.
* @param[in]	(u8 segaddr, u8 segnum, u8 addr, u8 offset, u8 bytes, u8* buffer)
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/
s32 I2C_HDMI_DDC_ReadEDDC(u8 segaddr, u8 segnum, u8 addr, u8 offset, u8 bytes, u8* buffer)
{
#if 1
	s32 iRetval = OK;
	iRetval = I2C_Master_Interrupt_ReadEDDC(I2C_CH_HDMI_DDC, segaddr, segnum, addr, offset, buffer, (s32) bytes); 
	return iRetval;
#else	
	gI2C_EventStatus = EVENT_I2C_NO;
	// set state
	I2C_Context.eState = I2C_MASTER_TX_EDDC_SEGADDR_STATE;

	// set parameters
	I2C_Context.segnum = segnum;
	I2C_Context.addr = addr;
	I2C_Context.offset = offset;
	I2C_Context.buffer = buffer;
	I2C_Context.bytes = bytes;

	// set clock, enable interrupts and ack generation
	I2C_Outp8(rI2CCON_HDMI_DDC, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_ACK_ENABLE); // 0xE1
	// to make IDSR write-enabled
	I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_ENABLE); // 0x10
	// put address to shift register
	I2C_Outp8(rI2CDS_HDMI_DDC, segaddr & 0xFE);

	// master tx mode + start + enable - "START" sequence generation
	I2C_Outp8(rI2CSTAT_HDMI_DDC, I2CSTAT_ENABLE | I2CSTAT_START | I2CSTAT_MASTER_TX_MODE); // 0xF0

	//waiting DDC status
	while(gI2C_EventStatus == EVENT_I2C_NO);
	
	if (gI2C_EventStatus == EVENT_I2C_ERROR)
	{
		UART_Printf("ERROR!!!\n");
		return ERROR;
	}

	return OK;
#endif
}
/*******************************************************************************
 * HDMI DDC : END
 ******************************************************************************/

/*******************************************************************************
 * PMIC : START
 ******************************************************************************/
/**
 * Release I2C line
 */
s32	I2C_PMIC_Release(void)
{
	// clear interrupt pending flag and interrupt clear bit
	I2C_Outp8(rI2CCON2, I2CCON_CLK | I2CCON_INT_ENABLE | I2CCON_INT_CLEAR);
	// disable 
	I2C_Outp8(rI2CSTAT2, I2CSTAT_IDLE);

	return OK;
}

/**
 * Polling interrupt 
 *
 * @return  On success, return OK;Otherwise return, ERROR.
 */
static s32 I2C_PMIC_InterruptWait(void)
{
	u8 ucStatus, ucReg;
	s32 retval = OK;
	u8 i=0;

	do
	{
		ucStatus = I2C_Inp8(rI2CCON2);
		
		if (ucStatus & I2CCON_INT_PEND)
		{
			// check status flags
			ucReg = I2C_Inp8(rI2CSTAT2);

			if (ucReg & (I2CSTAT_ARBITER | I2CSTAT_ADDR_SLAVE | I2CSTAT_ADDR_ZERO | I2CSTAT_ACK_ERROR) )
			{
				if (ucReg & I2CSTAT_ARBITER)
				{
					UART_Printf("I2CSTAT_ARBITER\n");
				}
				else if (ucReg & I2CSTAT_ADDR_SLAVE)
				{
					UART_Printf("I2CSTAT_ADDR_SLAVE\n");
				}
				else if (ucReg & I2CSTAT_ADDR_ZERO)
				{
					UART_Printf("I2CSTAT_ADDR_ZERO\n");
				}
				else if (ucReg & I2CSTAT_ACK_ERROR)
				{
					UART_Printf("I2CSTAT_ACK_ERROR\n");
				}
				
				I2C_PMIC_Release();
				retval = ERROR;
			}
			break;
		}
	}while (1);

	return retval;
}

/**
* {I2C_PMIC_Init}
*
* @brief			Initialize I2C library
* @param[in]	VOID
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/  
s32 I2C_PMIC_Init(void)
{
	gI2c_CH_Index = I2C_CH2;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);

	return OK;
}

/**
* {I2C_PMIC_DeInit}
*
* @brief			Deinitialize I2C library
* @param[in]	VOID
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/  
s32 I2C_PMIC_DeInit(void)
{
	u32 ch = gI2c_CH_Index;
	Assert(gI2c_CH_Index == I2C_CH2);
	
	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);
	gI2c_CH_Index = I2C_CH0;
	
	return OK;
}

/**
* {I2C_PMIC_Read}
*
* @brief			Read data through I2C
* @param[in]	(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/ 
s32	I2C_PMIC_Read(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	iRetval = I2C_Master_Polling_Read( I2C_CH2, ucAddr, pBuffer, (s32) ucBytes );
	return iRetval;
}

/**
* {I2C_PMIC_Write}
*
* @brief			Write data through I2C
* @param[in]	(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/ 
s32 I2C_PMIC_Write(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	iRetval = I2C_Master_Polling_Write( I2C_CH2, ucAddr, pBuffer, (s32) ucBytes );
	return iRetval;
}
/*******************************************************************************
 * PMIC : END
 ******************************************************************************/

/*******************************************************************************
 * OTHER IPs such as WM8580, CAMERA I/F etc... : START
 ******************************************************************************/
/**
* {I2C_Common_Ch_Init}
*
* @brief			Initialize I2C library
* @param[in]	VOID
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/  
//s32 I2C_General_IP_Init(u32 ufrequency)
s32 I2C_General_IP_Init(void)
{
	gI2c_CH_Index = I2C_CH0;
	I2C_GPIO_Open(gI2c_CH_Index);
	I2C_SFR_Mapping_Init(gI2c_CH_Index);
	I2C_Initialize_Controller(gI2c_CH_Index);
//	I2C_TxClockPrescaler(gI2c_CH_Index, ufrequency);
	
	return OK;
}

/**
* {I2C_Common_Ch_DeInit}
*
* @brief			Deinitialize I2C library
* @param[in]	VOID
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/  
s32 I2C_General_IP_DeInit(void)
{
	u32 ch = gI2c_CH_Index;
	Assert(gI2c_CH_Index == I2C_CH0);
	
	// deinitialize
	I2C_DeInitialize_Controller(ch);
	I2C_UnRegister_Interrupt(ch);
	I2C_GPIO_Close(ch);
	gI2c_CH_Index = I2C_CH0;
	
	return OK;
}

/**
* {I2C_Common_Ch_Read}
*
* @brief			Read data through I2C
* @param[in]	(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/ 
s32	I2C_General_IP_Read(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	iRetval = I2C_Master_Polling_Read( I2C_CH0, ucAddr, pBuffer, (s32) ucBytes );
	return iRetval;
}

/**
* {I2C_Common_Ch_Write}
*
* @brief			Write data through I2C
* @param[in]	(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
* @return		s32(On success, return OK;Otherwise return ERROR).
* @version		2009.07.07
* @author		Ho-June Byun(hjune.byun@samsung.com)
* 
*/ 
s32 I2C_General_IP_Write(u8 ucAddr, u8 ucBytes, u8 *pBuffer)
{
	s32 iRetval = OK;
	iRetval = I2C_Master_Polling_Write( I2C_CH0, ucAddr, pBuffer, (s32) ucBytes );
	return iRetval;
}
/*******************************************************************************
 * OTHER IPs such as WM8580, CAMERA I/F etc... : END
 ******************************************************************************/
 
/*******************************************************************************
 * End of file Iic.c
 ******************************************************************************/
