/*******************************************************************************

	File Name: i2s_hdmi.cpp
	Description: SMDKv210 Board I2S(S5PV210) Controller Function Test Code For HDMI Audio

   	Version: i2s v5.1 
   	History:
             R0.0(2009): HakSong.Kim First draft

        Memo:     	    
                
********************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "def.h"
#include "option.h"
#include "util.h"
#include "v210_sfr.h"
#include "system.h"
#include "intc.h"
#include "dma.h"
#include "gpio.h"
#include "sysc.h"
#include "i2s_hdmi.h"
#include "audiocodecs.h"
#include "spdif.h"
#include "rpetc.h"
#include "pcmutils.h"
#include "audiocodecs.h"
#include "assclkcon.h"
#include "i2sv5_1.h"

#ifndef DBG_I2S
#define i2sDbg(x) Dbg x
#else
#define i2sDbg(x)    0
#endif

#ifndef DISP_I2S_REG
#define I2sOutp32(a, d) Outp32(a, d)
#define I2sInp32(a, d) Inp32(a, d)
#else
#define I2sOutp32(a, d) Disp("Outp1w(\'h%08X, \'h%08x);\n", a, d), Outp32(a, d)
#define I2sInp32(a, d) (Disp("Inp1w (\'h%08x); ", a), Inp32(a, d), Disp("// d=0x%08x\n", d))
#endif

#define WM8580_ID 		(0x34)		// Chip ID, Device Address[7Bit] +RD/WR[1Bit] 
#define WM8580_ID_RD 	(0x35)		// Chip Read ID, Device Address[7Bit] +RD/WR[1Bit] 

enum I2S_SFR
{
	I2SCON = I2S0_BASE + 0x00,	 // ResetValue 0xE00
	I2SMOD = I2S0_BASE + 0x04,
	I2SFIC = I2S0_BASE + 0x08, 	// FIFO control
	I2SPSR = I2S0_BASE + 0x0C, 	// Clock divide control
	I2STXD = I2S0_BASE + 0x10, 	// W transmit data
	I2SRXD = I2S0_BASE + 0x14,  	// R receive data
	I2SFICS = I2S0_BASE + 0x18,  	// Secondary FIFO
	I2STXDS = I2S0_BASE + 0x1C,  	// Secondary W transmit data
	I2SAHB = I2S0_BASE + 0x20,  	// 
	I2SSTR0 = I2S0_BASE + 0x24,  	// 
	I2SSIZE = I2S0_BASE + 0x28,  	// 
	I2STRNCNT = I2S0_BASE + 0x2C,  	// 	
	I2SLVL0ADDR = I2S0_BASE + 0x30,  	// 	
	I2SLVL1ADDR = I2S0_BASE + 0x34,  	// 	
	I2SLVL2ADDR = I2S0_BASE + 0x38,  	// 	
	I2SLVL3ADDR = I2S0_BASE + 0x3C,  	// 	
	I2SSTR1 = I2S0_BASE + 0x40  	// 	
};


/*
	I2SCON
		[17]  0:Interrupt didn't be occurred, 1: Interrupt was occurred	
		[16]  0:TXFIFO Under-run INT disable, 1: TXFIFO Under-run INT enable
		[15]  0:Rx FIFO2 not Empty, 1: RX FIFO2 is Empty
		[14]  0:Rx FIFO1 not Empty, 1: RX FIFO1 is Empty
		[13]  0:Tx FIFO2 is not full, 1: TX FIFO2 is full	
		[12]  0:Tx FIFO1 is not full, 1: TX FIFO1 is full	
		[11] 0:Left(low) / Right, 1: Right(low) / Left
		[10] 0:Tx FIFO0 not Empty, 1: TX FIFO0 is Empty
		[9]  0:Rx FIFO0 not Empty, 1: RX FIFO0 is Empty
		[8]  0:Tx FIFO0 is not full, 1: TX FIFO0 is full
		[7]  0:Rx FIFO0 is not full, 1: RX FIFO0 is full
		[6]  0:Tx No pause DMA operation, 1: TX pause DMA operation
		[5]  0:Rx No pause DMA operation, 1: RX pause DMA operation
		[4]  0:no pause TX channels, 1: puase TX channels
		[3]  0:no pause Rx channels, 1: puase RX channels
		[2]  0:Tx DMA INACTIVE, 1: Tx DMA ACTIVE
		[1]  0:Rx DMA INACTIVE, 1: Rx DMA ACTIVE
		[0]  0:I2S Interface INACTIVE, 1: I2S Interface ACTIVE
	I2SMOD 
	        [21:20] Channel-2, 0: No Discard, 1: I2STXD[15:0] Discard, 2: I2STXD[31:16] Discard, 3: Reserved
	        [19:18] Channel-1, 0: No Discard, 1: I2STXD[15:0] Discard, 2: I2STXD[31:16] Discard, 3: Reserved
		[17]    SD2 channel enalbe 
		[16]    SD1 channel enalbe 
		[14:13] 0: 16bit per channel, 1: 8bit per channel, 2: 24bit per channel, 3: Reserved
		[12]    0:CODCLKOEN is Zero, 1:CODCLKOEN is one
		[11:10] 0:internal master, 1:External master, 2:slave mode(PCLK), 3:slave(CODCLKI)
		[9:8]   0:Transimit only, 1:Receive Only, 2:T&R Simultaneous Mode, 3:No Operation
		[7]     0:low for left, 1: Hig for left
		[6:5]   0:I2S format, 1:MSB, 2:LSB, 3:Reserved
		[4:3]   I2S Codec Clock. 0:256fs, 1:512fs, 2:384fs, 3:768fs
		[2:1]   bit Clock Freq  0:32fs, 1:48fs, 2:16fs, 3:24fs  
	I2SFIC
		[28:24] TxFIfO2 Data Count.	
		[20:16] TxFIfO1 Data Count.	
		[15]   TxFifo Flush.
		[12:8] TxFIfO0 Data Count.
		[7]    RxFifo Flush
		[4:0]  RxFifo Data Count
*/

// clock controller Reg Definition
#define rEPLL_LOCK		0xE0100010
#define rEPLL_CON		0xE0100110
#define rCLK_SRC0		0xE0100200
#define rCLK_SRC6		0xE0100218
#define rCLK_SRC_MASK0	0xE0100280
#define rCLK_GATE_IP3			0xE010046C
#define rCLK_DIV6		0xE0100318

// Global value
DMAC	oI2s0Dma0;
u32 uSaveFsVal;
extern u32 IIC0_Write_Polling(u32 ch, u32 SlaveAddr,u8* data, u32 TxSize,u32 StopGen);
extern CLK_SEL eUsedAudioMainClk;
extern FS_SEL eUsedAudioFs;

/******************************************************************************
*					I2S I/O Functions									
 ******************************************************************************/	
void I2S_InitIp(I2S_MODE eMode, I2S_SRCCLK eSrcClk, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample, AUDIO_MODE eAudioMode, I2S_FORMAT eSdf,u32 uSrcClk , AUDIO_CODEC eCodec,I2S *sCh)
{
	u32 uChSet, uRfs, uCalRfs;
	u32 uModReg, uPsrReg;

	if(eUsedAudioMainClk==EXTERNAL_CLK)
		I2sOutp32(0xEEE10000, 1<<2);		// Audio sub system clockcon source register external setting
	else if(eUsedAudioMainClk==INTERNAL_CLK)
		;								// I2S_SetAudioClock() will do internal clock setting.  
	else
		Assert(0);

	/// 1. I2S s/w reset
	I2sOutp32(0xeee30000, 0x00000000);
	I2sOutp32(0xeee30000, 0x80000000);

#if 0	
	//// Initialize WM8580 with I2C
	IIC0_OpenPolling(200000);	//	200Khz
	I2S_InitWm8580(eSrcClk, eAudioMode, uChannelType, uSampleRate, uBitsPerSample, eSdf, uSrcClk );

	WM8580_CodecInitPCMOut(I2SFormat, uSampleRate, Master, Word16, SECONDARY_PORT, RFS_512fs);
#endif 

	uPsrReg = Inp32(I2SPSR);
	//// I2S Prescaler Register Setting
	uPsrReg = (uPsrReg & ~(1<<15)) | (0<<15);	// Prescaler(Clock divide) A active, 0: Inactive, 1: Active

	if (uPsrReg & 1<<15)		// Only prescaler(Clock divide) A active
	{
		if (eSrcClk==INTERNAL_MASTER || eSrcClk == INTERNAL_SLAVE)
		{
			switch (uSampleRate) //PCLK
			{
				case 44100: uPsrReg |= 2<<8; break;
				case 22050: uPsrReg |= 4<<8; break;
				case 11025: uPsrReg |= 9<<8; break;
				case 5513 : uPsrReg |= 19<<8; break;
				default: Assert(0);
			}
		}	
		else if (eSrcClk==EXTERNAL_MASTER || eSrcClk == EXTERNAL_SLAVE)
		{
			switch (uSampleRate) // 16.9344Mhz 
			{
				case 44100: uPsrReg |= 0<<8; break;
				case 22050: uPsrReg |= 1<<8; break;
				case 11025: uPsrReg |= 3<<8; break;
				case 5513 : uPsrReg |= 7<<8; break;
				default: Assert(0);
			}
		}
		else
			Assert(0);
	}

	//// I2S MOD Register Data Channel Enable Setting 
	if (uChannelType == 1)
		uChSet = 0;			// I2SSDO_0 Enable, Mono 
	else if (uChannelType == 2)
		uChSet = 0;			// I2SSDO_0 Enable, stereo 
	else if (uChannelType == 4)
		uChSet = 1;			// I2SSDO_0, I2SSDO_1 Enable, 4 Ch 
	else if (uChannelType == 6)
		uChSet = 3;			// I2SSDO_0, I2SSDO_1, I2SSDO_2 Enable, 5.1 Ch 
	else
		Assert(0);

	//// I2S MOD Register Bit Per Length Setting
	sCh->m_uBitsPerSample = uBitsPerSample;

	//// I2S MOD Register Transmit or Receive mode Setting
	sCh->m_uMode = (eMode == TX_MODE) ? 0 :
			(eMode == RX_MODE) ? 1 : 2;

	if (eUsedAudioFs == FS_256)
		uRfs = 0;			// 256fs
	else if (eUsedAudioFs == FS_512)
		uRfs = 1;			// 512fs
	else if (eUsedAudioFs == FS_384)
		uRfs = 2;			// 384fs
	else if (eUsedAudioFs == FS_768)
		uRfs = 3;			// 768fs		
	else
		Assert(0);

	if (sCh->m_uBitsPerSample == 24)
	{
		// 24bit, External Marster, Trans or Receive,low, I2S format, 384fs, 32fs
		//uModReg = (2<<26)|(2<<24)|(2<<13)|(1<<12)|(1<<10)|(sCh->m_uMode<<8)|(eSdf<<5)|(uRfs<<3)|(1<<1)|(0<<0);
		uModReg = (2<<26)|(2<<24)|(uChSet<<16)|(2<<13)|(1<<10)|(sCh->m_uMode<<8)|(eSdf<<5)|(uRfs<<3)|(1<<1)|(0<<0);
	}
	else if (sCh->m_uBitsPerSample == 16)
	{
		// 16bit, External Master, Trans or Receive,low, I2S format, 384fs, 32fs
		//uModReg = (0<<13)|(0<<12)|(eSrcClk<<10)|(sCh->m_uMode<<8)|(eSdf<<5)|(uRfs<<3)|(0<<1)|(0<<0);
		uModReg = (0<<26)|(0<<24)|(uChSet<<16)|(0<<13)|(1<<10)|(sCh->m_uMode<<8)|(eSdf<<5)|(uRfs<<3)|(0<<1)|(0<<0);
	}	
	else if (sCh->m_uBitsPerSample == 8)
	{
		// 8bit, External Marster, Trans or Receive,low, I2S format, 384fs, 32fs
		uModReg = (1<<26)|(1<<24)|(uChSet<<16)|(1<<13)|(eSrcClk<<10)|(sCh->m_uMode<<8)|(eSdf<<5)|(uRfs<<3)|(2<<1)|(1<<0);
	}	
	else
		Assert(0);

	if(eUsedAudioMainClk==EXTERNAL_CLK)
		uModReg |= 1<<12; 		// Key point external clock
	else if(eUsedAudioMainClk==INTERNAL_CLK)
		uModReg &= ~(1<<12); 		// Key point Internal clock
	else 
		Assert(0);

	I2sOutp32(I2SMOD, uModReg);
	I2sOutp32(I2SPSR, uPsrReg); // Prescaler Active
	I2sOutp32(I2SFIC, 0xFFFFFFFF);
	I2sOutp32(I2SFIC, 0x0);
}

void I2S_PlayByCpu(u32 uPcmDataAddr, u32 uPcmDataSize, u32 uChannelType, I2S *sCh)
{
	u32 uSfr;
	u32 *pPcmAddr;
	u16 *p16PcmAddr;
	u32 uConReg;

	i2sDbg(("\nI2S PCM play polling!\n"));

	I2sOutp32(I2SFIC, 1<<15|1<<7);	//Fifo Flush
	I2sOutp32(I2SFIC, 0x0);
 	
	uConReg = Inp32(I2SCON);
	uConReg |= (1<<0);
	I2sOutp32(I2SCON, uConReg);   

	pPcmAddr = (u32*)uPcmDataAddr;
	p16PcmAddr = (u16*)uPcmDataAddr;

	if (sCh->m_uBitsPerSample == 24)
	{
		while (uPcmDataSize > 0)
		{
			do {
				uSfr = Inp32(I2SCON);
			} while (uSfr&(1<<8));			// FIFO 0
			I2sOutp32(I2STXD, *pPcmAddr++);
			uPcmDataSize -= 4;	

			if ((uChannelType == 6)||(uChannelType == 4))
			{
				do {
				uSfr = Inp32(I2SCON);
				} while (uSfr&(1<<12));			// FIFO 1
				I2sOutp32(I2STXD, *pPcmAddr++);
				uPcmDataSize -= 4;
			}

			if (uChannelType == 6)
			{
				do {
				uSfr = Inp32(I2SCON);
				} while (uSfr&(1<<13));			// FIFO 2
				I2sOutp32(I2STXD, *pPcmAddr++);
				uPcmDataSize -= 4;
			}
		}
	}	

	else if (sCh->m_uBitsPerSample == 16)
	{
		while (uPcmDataSize > 0)
		{
			do {
				uSfr = Inp32(I2SCON);
			} while (uSfr&(1<<8));			// FIFO 0
			I2sOutp32(I2STXD, *pPcmAddr++);
			uPcmDataSize -= 4;	

			if ((uChannelType == 6)||(uChannelType == 4))
			{
				do {
				uSfr = Inp32(I2SCON);
				} while (uSfr&(1<<12));			// FIFO 1
				I2sOutp32(I2STXD, *pPcmAddr++);
				uPcmDataSize -= 4;
			}

			if (uChannelType == 6)
			{
				do {
				uSfr = Inp32(I2SCON);
				} while (uSfr&(1<<13));			// FIFO 2
				I2sOutp32(I2STXD, *pPcmAddr++);
				uPcmDataSize -= 4;
			}
		}
	}
	
	else if (sCh->m_uBitsPerSample == 8)
	{
		while (uPcmDataSize > 0 )
		{
 			do {
				uSfr = Inp32(I2SCON);
			} while (uSfr&(1<<8));			// FIFO 0			
			uSfr = (*p16PcmAddr)&0xFF; // L[7:0]
			uSfr |= ((*p16PcmAddr)&0xFF00)<<8; // R[23:16]
			I2sOutp32(I2STXD, uSfr);
			p16PcmAddr++;
			uPcmDataSize -= 2;
		}
	}
	
	else
		Assert(0);

	uConReg = Inp32(I2SCON);
	uConReg &= ~0x1;
	I2sOutp32(I2SCON, uConReg);
	UART_Printf( "uConReg: 0x08%\n", uConReg);
	
}

void I2S_PlayByDma(u32 uPcmStartAddr, u32 uPcmDataSize, u32 uBitsPerSample, I2S *sCh)
{
	u32 wavdatasize;
	u32 uConReg;
	DATA_SIZE eDataSize;
	BURST_MODE eBurstMode;

	/// 1. Wave file size calculation
	if(uBitsPerSample == 24)
	{
		eDataSize = WORD;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 16)
	{
		eDataSize = WORD;
		eBurstMode = SINGLE;
			wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 8)
	{
		eDataSize = BYTE;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else
		Assert(0);
	
	I2sOutp32(I2SFIC, 1<<15|1<<7);	//Fifo Flush
	I2sOutp32(I2SFIC, 0x0);

	/// 1. Wave file playing start by DMA
	DMA_SetCh(DMA_00, &oI2s0Dma0);//ok. peri 0
	DMA_InitCh(eDataSize , I2S0_TX, DMA_M2P, eBurstMode, &oI2s0Dma0);
	DMA_StartCh(uPcmStartAddr, I2STXD, wavdatasize, &oI2s0Dma0);
	uConReg = Inp32(I2SCON);
	uConReg = (uConReg & ~((1<18)|(1<<2)|(1<<0)))|((1<<18)|(1<<2)|(1<<0));	// Tx DMA active , I2S active
	I2sOutp32(I2SCON, uConReg);	
	
	while(!DMA_IsTransferDone(&oI2s0Dma0));

	DMA_StopCh(&oI2s0Dma0);	
}

void I2S_PlayByDmaForInterrupt(u32 uPcmStartAddr, u32 uPcmDataSize, u32 uBitsPerSample, I2S *sCh)
{
	u32 wavdatasize;
	u32 uConReg;
	DATA_SIZE eDataSize;
	BURST_MODE eBurstMode;

	/// 1. Wave file size calculation
	if(uBitsPerSample == 24)
	{
		eDataSize = WORD;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 16)
	{
		eDataSize = WORD;
		eBurstMode = SINGLE;
			wavdatasize = uPcmDataSize;
	}
	else if(uBitsPerSample == 8)
	{
		eDataSize = BYTE;
		eBurstMode = SINGLE;
		wavdatasize = uPcmDataSize;
	}
	else
		Assert(0);
	
	I2sOutp32(I2SFIC, 1<<15|1<<7);	//Fifo Flush
	I2sOutp32(I2SFIC, 0x0);

	/// 1. Wave file playing start by DMA
	DMA_SetCh(DMA_00, &oI2s0Dma0);//ok. peri 0
	DMA_InitCh(eDataSize , I2S0_TX, DMA_M2P, eBurstMode, &oI2s0Dma0);
	DMA_StartCh(uPcmStartAddr, I2STXD, wavdatasize, &oI2s0Dma0);
#if 0
	uConReg = Inp32(I2SCON);
	uConReg = (uConReg & ~((1<18)|(1<<2)|(1<<0)))|((1<<18)|(1<<2)|(1<<0));	// Tx DMA active , I2S active
	I2sOutp32(I2SCON, uConReg);	
	
	while(!DMA_IsTransferDone(&oI2s0Dma0));

	DMA_StopCh(&oI2s0Dma0);	
#endif
}

void I2S_InitWm8580(I2S_SRCCLK eSrcClk, AUDIO_MODE eAudioMode, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample, I2S_FORMAT eSdf, u32 uSrcClk )
{
	u32 i,uCalRfs;
	u8 x;
	u16 uBclkRate, uSamplingFreq, uBitSpl, uPaif3, uPaif1, uPaif2, uPaif4, uI2sFmt;

	if (uBitsPerSample == 16 || uBitsPerSample == 8)
		uBitSpl = 0;
	else if (uBitsPerSample == 24)
		uBitSpl = 2;
	else
		Assert(0);

	if (eSdf == I2S_FMT)
		uI2sFmt = 2;
	else if (eSdf == MSB_FMT)
		uI2sFmt = 1;
	else if (eSdf == LSB_FMT)
		uI2sFmt = 0;
	else
		Assert(0);

		uPaif3 = (3<<7)|(uBitSpl<<2)|(uI2sFmt<<0);
		uPaif4 = (1<<7)|(uBitSpl<<2)|(uI2sFmt<<0);

	if (eSrcClk == INTERNAL_MASTER || eSrcClk == EXTERNAL_MASTER)
	{
		/// 1. Master Play Setting
		if (eAudioMode == SPEAKER_MODE)
		{
			I2S_SendWm8580Data(53, 0x000);				// Reset WM8580
			I2S_SendWm8580Data(50, 0x03a);				// All Digital running, ADC disable, DAC1,2,3 enable, DACs under control of DACPD[3:0]
			I2S_SendWm8580Data(51, 0x038);
			I2S_SendWm8580Data(8, 0x01c);
			I2S_SendWm8580Data(11, 3<<6);
			I2S_SendWm8580Data(14, (1<<7|1<<6|uPaif3));
			I2S_SendWm8580Data(12, (2<<7)|uPaif3);
			I2S_SendWm8580Data(16, 0x009);
			I2S_SendWm8580Data(28, 0x1bf);
		}

		/// 2. Master Iine-In Record Setting
		else if (eAudioMode == LINE_IN_MODE)
		{
			I2S_SendWm8580Data(53, 0x000);				// Reset WM8580
			I2S_SendWm8580Data(50, 0x07c);				// PWRDN1, All DACs disable, DAC1,2,3 disabled, ADC enabled, All digital circuit running.
			I2S_SendWm8580Data(8, 0x01c);				// DAC clock sourcen MCLK pin
			I2S_SendWm8580Data(13, uPaif4);				// I2S FMT, Bit rate Setting
		}

		/// 3. AUDIO_MODE setting error
		else
			Assert(0);		
			
	}
	else if (eSrcClk == INTERNAL_SLAVE || eSrcClk == EXTERNAL_SLAVE) 
	{
		uCalRfs = uSrcClk/uSampleRate;

		if(uCalRfs == 128)
			uSamplingFreq = 0;
		else if(uCalRfs == 192)
			uSamplingFreq = 1;
		else if(uCalRfs == 256)
			uSamplingFreq = 2;
		else if(uCalRfs == 384)
			uSamplingFreq = 3;
		else if(uCalRfs == 512)
			uSamplingFreq = 4;
		else if(uCalRfs == 768)
			uSamplingFreq = 5;
		else if(uCalRfs == 1152)
			uSamplingFreq = 6;
		else
			Assert(0);
	
		if(uBitsPerSample==16)
			uBclkRate = 1;
		else if(uBitsPerSample==8)
			uBclkRate = 1;
		else if(uBitsPerSample==24)
			uBclkRate = 0;
		else 	
			Assert(0);

		uPaif1 = 1<<5|uBclkRate<<3 |uSamplingFreq<<0;		// Wm8580 Master Mode		
		uPaif2 = 1<<5|uBclkRate<<3 |uSamplingFreq<<0;		// Wm8580 Master Mode		
	
		/// 1. Slave Play Setting
		if(eAudioMode == SPEAKER_MODE)
		{
			I2S_SendWm8580Data(53, 0x000);			// Reset WM8580
			I2S_SendWm8580Data(50, 0x022);			// All Digital running, ADC disable, DAC1,2,3 enable, DACs under control of DACPD[3:0]
			I2S_SendWm8580Data(9 , uPaif1);			// slave mode setting
			I2S_SendWm8580Data(26, 0x0FF);			// need for volume setting
			I2S_SendWm8580Data(27, 0x0FF);			// nedd for volume setting
			I2S_SendWm8580Data(28, 0x1cF);  			//volume control
		}
		
		/// 2. Slave Iine-In Record Setting
		else if(eAudioMode == LINE_IN_MODE)
		{
			I2S_SendWm8580Data(53, 0x000);				// Reset WM8580
			I2S_SendWm8580Data(50, 0x07c);				// PWRDN1, All DACs disable, DAC1,2,3 disabled, ADC enabled, All digital circuit running.
			I2S_SendWm8580Data(8, 0x01c);				// DAC clock sourcen MCLK pin
			I2S_SendWm8580Data(10, uPaif2);				// In AP, Rx => In Wm8580, PAIFTX, slave mode setting
		}

		/// 3. AUDIO_MODE setting error
		else
			Assert(0);		
	}

	else
		Assert(0);
		//	I2S_SendWm8580Data(19, 0x0ff);					//Mute
}

/*****************************************************************
*
*	WM8580 Data Format For IIC
*
*	- Device Address[7Bit] +RD/WR[1Bit] : 0x34 or 0x36
*	- Control Byte1[8bit] : Register Addr[7:1] + Data[0MSB]
*	- Control Byte2[8bit] : Data[7:0]
*
*	WM8580 Data type For C Coding
*
*	- slv_addr : 8bit
*	- reg_addr : 7bit
*	- data : 9bit
*
******************************************************************/
void I2S_SendWm8580Data(u8 reg_addr, u16 data)
{
	u8 ucReg_addr, ucData;

	ucReg_addr = (reg_addr<<1 | (u8)(data>>8));
	ucData = (u8)data;
	I2S_IIC_SendData(WM8580_ID, ucReg_addr, ucData);
}

void I2S_IIC_SendData( unsigned char SlaveAddr, char reg_addr, unsigned short data)
{
#ifdef 1
	char D[2+1];

	#if 1 //jspark
	D[0]=reg_addr;
	D[1]=(unsigned char)(data);
 	IIC0_Write_Polling(IIC_CH0, SlaveAddr,D, 2, 1);
	#else
	IIC_Write_Pol( slave_addr, D, 2);
	IIC_Wait();
	#endif
	
#endif
}

void GPIO_SetFunctionForI2s(void)
{
	I2sOutp32(0xe0200220, 0x02222222);		// I2S_0 setting
//	I2sOutp32(0xe020022c, 0x03333333);		// GPIO Drive strength
}

void I2S_SetAudioClock(u32 uSampleRate, FS_SEL eFsSel)
{

	volatile u32 uRead;
	u32 uTrg_Fout, uVSEL=0, uP_Value=0, uM_Value=0, uS_Value=0;
	u32 uCal, uInp_Clk, uDiv, uFsVal;

	/// 1. Clock Cal
	if (eUsedAudioFs == FS_256)
		uFsVal = 256;
	else if (eUsedAudioFs == FS_384)
		uFsVal = 384;
	else if (eUsedAudioFs == FS_512)
		uFsVal = 512;
	else if (eUsedAudioFs == FS_768)
		uFsVal = 768;		
	else 
		Assert(0);
		
	uTrg_Fout = uSampleRate*uFsVal;   
	i2sDbg(("uTrg_Fout: %d\n",uTrg_Fout));

	/// 2. Target FOUT(MHz), EPLL Table
	switch(uTrg_Fout)
	{
		case	 48000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 96;	uS_Value = 3; 
			if(uTrg_Fout== 48000000) uDiv=0; 
			break;	
		case	 96000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 96;	uS_Value = 2; 
			if(uTrg_Fout== 96000000) uDiv=0; 
			break;	
		case	 144000000:	
			uVSEL=1;	uP_Value = 6;	uM_Value = 144;	uS_Value = 2; 
			if(uTrg_Fout== 144000000) uDiv=0; 
			break;	
		case	 192000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 96;	uS_Value = 1; 
			if(uTrg_Fout== 192000000) uDiv=0; 
			break;		
		case	 288000000:	
			uVSEL=1;	uP_Value = 6;	uM_Value = 144;	uS_Value = 1; 
			if(uTrg_Fout== 288000000) uDiv=0; 
			break;		
		case	 84000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 84;	uS_Value = 2; 
			if(uTrg_Fout== 84000000) uDiv=0; 
			break;	
		case	 50000000:	
			uVSEL=0;	uP_Value = 6;	uM_Value = 100;	uS_Value = 3; 
			if(uTrg_Fout== 50000000) uDiv=0; 
			break;	
		case	 80000000:	
			uVSEL=1;	uP_Value = 6;	uM_Value = 160;	uS_Value = 3; 
			if(uTrg_Fout== 80000000) uDiv=0; 
			break;
		case 4096000:	
		case 16384000:	
		case 8192000:	
		case	 32768000:	
			uVSEL=1;	uP_Value = 6;	uM_Value = 131;	uS_Value = 4; 
			if (uTrg_Fout== 32768000) uDiv=0; 
			else if (uTrg_Fout== 8192000) uDiv=3; 
			else if (uTrg_Fout== 16384000) uDiv=1; 
			else if (uTrg_Fout== 4096000) uDiv=7;
			break;
		case 12288000:	
		case 24576000:	
		case 49152000:
			uVSEL=0;	uP_Value = 8;	uM_Value = 131;	uS_Value = 3; 
			if(uTrg_Fout== 49152000) uDiv=0; 
			else if(uTrg_Fout == 24576000) uDiv=1; 
			else if(uTrg_Fout == 12288000) uDiv=3; 			
			break;	
		case 16934400:		
		case 33868800:	
		case 67737600:
			uVSEL=1;	uP_Value = 12;	uM_Value = 271;	uS_Value = 3; 
			if (uTrg_Fout== 67738000) uDiv=0; 
			else if (uTrg_Fout== 33868800) uDiv=1; 
			else if (uTrg_Fout== 16934400) uDiv=3; 
			break;
		case 36864000:
		case 18432000:
		case 73728000:
			uVSEL=1;	uP_Value = 12;	uM_Value = 295;	uS_Value = 3; 
			if (uTrg_Fout== 73728000) uDiv=0;
			else if (uTrg_Fout== 18432000) uDiv=3; 
			else if (uTrg_Fout== 36864000) uDiv=1; 			
			break;	
		case 11289600:	
		case 22579200:	
		case 45158400:
			uVSEL=0;	uP_Value = 18;	uM_Value = 271;	uS_Value = 3; 
			if (uTrg_Fout== 45158400) uDiv=0; 
			else if (uTrg_Fout== 22579200) uDiv=1;
			else if (uTrg_Fout== 11289600) uDiv=3;
			break;	
		default:
			Assert(0); break;
	}

	//system setting
	SYSC_SetAudioBlockPowerOn();//I2S0 v5.1, RP
	SYSC_SetAudioEndian(RP_R_BIG,RP_W_BIG,ARM_R_LITTLE,ARM_W_LITTLE);//to access i2sv5.1, clkcon.(commbox)

	///3.  EPLL CLK Settting
	/// 3.1 EPLL Lock Time, 300us
	uRead = Inp32(rEPLL_LOCK);
	uRead = (uRead & ~(0xffff<<0)) | (0xe10<<0);
	Outp32(rEPLL_LOCK, uRead);	

	/// 3.2 EPLL control setting
	uRead = Inp32(rEPLL_CON);
	uRead = (uRead & ~((1<<31)|(1<<27)|(0x3ff<<16)|(0x3f<<8)|(7<<0))) | ((1<<31)|(uVSEL<<27)|(uM_Value<<16)|(uP_Value<<8)|(uS_Value<<0));
	Outp32(rEPLL_CON, uRead);	
	while(!(Inp32(rEPLL_CON)>>29)&0x1);		// lock time is done

	/// 4. Audio Sub System Clock Setting
	AudioSS_CLKCON_SetGate(HCLK_I2SV5_1_PASS|	I2SCLK_PASS);
	AudioSS_CLKCON_SetSource(MUX_FOUT_EPLL, Main_CLK, MUXI2SA_Main_CLK);
	AudioSS_CLKCON_SetDivider(uDiv,0);//(i2sclk, audio bus) // peater
}


