 #include <string.h>
#include "system.h"
#include "library.h"
#include "def.h"
#include "util.h"
#include "mixer.h"
#include "vp_v210.h"
#include "fimc.h"
#include "timer.h"
#include "glib.h"
#include "hdmi.h"
#include "phy.h"
#include "hpd.h"
#include "hdcp.h"
#include "i2c_hdmi.h"
#include "edid.h"
#include "tvout_common.h"
#include "cec.h"

#include "hdmi_audio.h"
#include "spdif.h"
#include "wavefile.h"
#include "ac3file.h"
#include "i2s_hdmi.h"
#include "intc.h"
#include "sysc.h"

extern void TZPC_NonSecureInit(void);

#define WAVE_BUF_START		CODEC_MEM_ST
#define PCM_BUF_START		(WAVE_BUF_START+WAVFILE_HEADER_SZ)
#define AC3_BUF_START		CODEC_MEM_ST

static u32 uPcmSize;
static u32 uAc3Size;
extern CLK_SEL eUsedAudioMainClk;
extern FS_SEL eUsedAudioFs;

static I2S oI2s0;
static I2S_FORMAT eSerialDataFmt;
static u32 uPcmSize;
static u32 uCdclk=24576000;

static SCAN_MODE eScanMode;
static IMG_RESOLUTION eImgSz;
extern volatile DMAC	oSPDIFDma0;

static void __irq Isr_HDMI_ADUIO(void)
{
	volatile u32 uRead, uStatus, uI2sIrQStaus, uSpdifinIrqStatus; 
	INTC_Disable(NUM_HDMI);
	INTC_ClearVectAddr();

	uRead = HDMI_AUDIO_ReadIntFlag();

	UART_Printf("\nIntFlag: 0x%02x \n", uRead);

	if(uRead&(1<<0))
	{
		UART_Printf("IntrHDCP_flag interrupt is generated!\n");
		HDMI_AUDIO_DisableIntCon(IntrHDCP_flag);
	}

	if(uRead&(1<<1))
	{
		uStatus = HDMI_AUDIO_ReadStatusFlag();
		uSpdifinIrqStatus = HDMI_AUDIO_ReadSpdifinIrqStatus();
		UART_Printf("IntrSPDIF_flag interrupt!\n");
		UART_Printf("HDMI Systme status : 0x%02x\n", uStatus);
		UART_Printf("SPDIFIN IRQ Status : 0x%02x\n", uSpdifinIrqStatus);	
		
		if(uSpdifinIrqStatus&(1<<0))
		{
			UART_Printf("wrong_signal_ir_flag!\n");	
			HDMI_AUDIO_ClearSpdifIrqStatusFlag(wrong_signal_ir_flag);
		}

		if(uSpdifinIrqStatus&(1<<1))
		{
			//UART_Printf("ch_status_recovered_ir_flag!\n");
			UART_Printf("Detected Preamble Signal of SPDIF!\n");
			HDMI_AUDIO_ClearSpdifIrqStatusFlag(ch_status_recovered_ir_flag);
			HDMI_AUDIO_DisableSpdifIrqMask(ch_status_recovered_ir_flag);
		}

		if(uSpdifinIrqStatus&(1<<2))
		{
			UART_Printf("wrong_preamble_ir_flag!\n");	
			HDMI_AUDIO_ClearSpdifIrqStatusFlag(wrong_preamble_ir_flag);
		}

		if(uSpdifinIrqStatus&(1<<3))
		{
			UART_Printf("stream_header_not_detected_ir_flag!\n");	
			HDMI_AUDIO_ClearSpdifIrqStatusFlag(stream_header_not_detected_ir_flag);
		}

		if(uSpdifinIrqStatus&(1<<4))
		{
			UART_Printf("stream_header_detected_ir_flag!\n");	
			HDMI_AUDIO_ClearSpdifIrqStatusFlag(stream_header_detected_ir_flag);
			HDMI_AUDIO_DisableSpdifIrqMask(stream_header_detected_ir_flag);			
		}		

		if(uSpdifinIrqStatus&(1<<7))
		{
			UART_Printf("buf_overflow_ir_flag!\n");	
			HDMI_AUDIO_ClearSpdifIrqStatusFlag(buf_overflow_ir_flag);
		}
	}
	
	if(uRead&(1<<2))
	{
		UART_Printf("IntrHPDunpluag_flag interrupt is generated!\n");
		HDMI_AUDIO_DisableIntCon(IntrHPDunpluag_flag);
	}		

	if(uRead&(1<<3))
	{
		UART_Printf("IntrHPDpluag_flag interrupt is generated!\n");
		HDMI_AUDIO_DisableIntCon(IntrHPDpluag_flag);
	}		

	if(uRead&(1<<4))
	{
		UART_Printf("IntrCEC_flag interrupt is generated!\n");
		HDMI_AUDIO_DisableIntCon(IntrCEC_flag);
	}		
		
	if(uRead&(1<<5))
	{
		uStatus = HDMI_AUDIO_ReadStatusFlag();
		uI2sIrQStaus = HDMI_AUDIO_ReadI2sIrqStatus();
		UART_Printf("IntrI2S_flag interrupt!\n");
		UART_Printf("HDMI Systme status : 0x%02x\n", uStatus);		
		UART_Printf("I2S IRQ Status : 0x%02x\n", uI2sIrQStaus);	
		#if 0
		if(uI2sIrQStaus&(1<<0))
		{
			UART_Printf("Bit_per_channel_mismatch_flag!\n");	
			HDMI_AUDIO_ClearI2sIrqStatusFlag(Bit_per_channel_mismatch_flag);
		}
		#endif
		if(uI2sIrQStaus&(1<<1))
		{
			UART_Printf("Wrong_register_setting_flag!\n");	
			HDMI_AUDIO_ClearI2sIrqStatusFlag(Wrong_register_setting_flag);
		}		
	}		

	INTC_Enable(NUM_HDMI);
}

static void HDMIT_DrawGraphicForMixer(u32 uWin0StAddr, u32 uWin1StAddr)
{
	GLIB_InitInstance(uWin0StAddr, IMG_SRC_HSZ_240, IMG_SRC_VSZ_320, RGB16);
	GLIB_DrawPattern(IMG_SRC_HSZ_240, IMG_SRC_VSZ_320);
	//GLIB_DrawPattern_Blue(IMG_SRC_HSZ_240, IMG_SRC_VSZ_320);
	GLIB_InitFont();
	GLIB_Printf(90,160, "Window 0");
	
	GLIB_InitInstance(uWin1StAddr, IMG_SRC_HSZ_240, IMG_SRC_VSZ_320, RGB16);
	GLIB_DrawPattern2(IMG_SRC_HSZ_240, IMG_SRC_VSZ_320);
	GLIB_Printf(90,160, "Window 1");
}

extern cec_state g_eCECRxStatus;
extern oCec_Rx_struct oCec_Rx;

void HDMI_HW_Init(void)
{
	u32 hpd;

	HDMI_Init();

	if (HPD_Init() != OK)

	{
		UART_Printf("fail to init HPD\n");
		return;
	}

	I2CHDMI_Init();
    
	if (EDDC_Init() != OK)
	{
		UART_Printf("fail to init EDDC\n");
		return;
	}

	if (HDCP_Init() != OK)
	{
		UART_Printf("fail to init HDMI\n");
		return;
	}
	    
	UART_Printf("waiting for HPD...\n");
	while(!UART_GetKey())
	{
		hpd = HPD_Detect();
		if(hpd == HDMI_HPD_CABLE_IN)
		{
			UART_Printf("HDMI_HPD_CABLE_IN\n");
			break;
		}
	}
}

static void InitBaseHWForHdout(void)
{
	u32 uTemp;

	// PMU Block : HDMI PHY Enable
	uTemp = Inp32(0xE010E804);
	uTemp |= (1<<0);
	Outp32(0xE010E804, uTemp);		
}

void HDMI_Audio_SetVideo(void)
{
	u32 uResolution, uScanMode, uHdmiMode, uColorFormat, uBitPerColor, uWideColor;
	u32 uVpSrcAddr, uRGBSrcAddr,uBMPSrcAddr, uWin0StAddr, uWin1StAddr;
	HDMIMode eHDMI_DVI;
	VideoFormat eVideoFmt;
	ColorDepth eColorDepth;
	PixelAspectRatio ePixelRatio;
	ColorSpace eColorSpace;
	PixelLimit ePixelLimit;
	MX_RGB_FORMAT eMxOutRgb;

	/// 3. HDMI Video Setting
	uHdmiMode = 0;		// peater
	eHDMI_DVI = HDMI;
	uResolution = 2;		// peater
	eImgSz = SD480;
	
	uScanMode = 0;		// peater
		
	eScanMode = PROGRESSIVE ;
	
	eVideoFmt = 	(eImgSz == HD1080)? ((eScanMode== PROGRESSIVE)? v1920x1080p_60Hz : v1920x1080i_60Hz) :
				(eImgSz == HD720)? 	v1280x720p_60Hz:
				(eImgSz == SD480)? 	((eScanMode== PROGRESSIVE)? v720x480p_60Hz : v720x480i_60Hz) :
									((eScanMode== PROGRESSIVE)? v720x576p_50Hz : v720x576i_50Hz);
									
	ePixelRatio = 	(eImgSz == HD1080)? HDMI_PIXEL_RATIO_16_9:
				(eImgSz == HD720)? 	HDMI_PIXEL_RATIO_16_9:
				(eImgSz == SD480)?	HDMI_PIXEL_RATIO_4_3: HDMI_PIXEL_RATIO_4_3;

	uColorFormat = 0;	// peater

	if(uColorFormat>1)
		uColorFormat = 0;

	eColorSpace = (uColorFormat == 0)? HDMI_CS_YCBCR444 : HDMI_CS_RGB;

	uBitPerColor = 0;		// peater		

	if(uBitPerColor>2)
		uBitPerColor = 0;
		eColorDepth = 	(uBitPerColor == 0)? HDMI_CD_24 :
						(uBitPerColor == 1)? HDMI_CD_30 : HDMI_CD_36 ;

	uWideColor = 0;
	
	if(uWideColor>3)
		uWideColor = 0;	
	if(eColorSpace == HDMI_CS_RGB)
	{
		eMxOutRgb = 	(uWideColor == 0)? RGB601_16_235 : 
					(uWideColor == 1)? RGB601_0_255 : 
					(uWideColor == 2)? RGB709_16_235 : RGB709_0_255 ;

		if( (eMxOutRgb == RGB601_0_255) || (eMxOutRgb == RGB709_0_255) )
			ePixelLimit = HDMI_FULL_RANGE;
		else
			ePixelLimit = HDMI_RGB_LIMIT_RANGE;
	}
	else
	{
		eMxOutRgb = 	(uWideColor == 0)? RGB601_16_235 : 
					(uWideColor == 1)? RGB601_0_255 : 
					(uWideColor == 2)? RGB709_16_235 : RGB709_0_255 ;
						
		ePixelLimit = (uWideColor == 0 ||uWideColor == 2 )? HDMI_YCBCR_LIMIT_RANGE : HDMI_FULL_RANGE;		
	}

	uVpSrcAddr = CODEC_MEM_ST;
	uRGBSrcAddr = uVpSrcAddr + 0x00800000;
	uBMPSrcAddr = uRGBSrcAddr + 0x00800000;
	uWin0StAddr = uBMPSrcAddr + 0x00800000;
	uWin1StAddr = uWin0StAddr + 0x00800000;

	HDMIT_DrawGraphicForMixer(uWin0StAddr, uWin1StAddr);

	HDMI_SetMode(eHDMI_DVI);
	HDMI_SetColorSpace(eColorSpace);
	HDMI_SetPixelLimit(ePixelLimit); 	
	HDMI_SetVideoMode(eVideoFmt, eColorDepth, ePixelRatio);
}

void TestHdmiSpdifByCpu(void)
{
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1.1 Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	Disp("\nChannelType: %d\n", uChannelType);
	Disp("SampleRate: %d\n", uSampleRate);
	Disp("BitsPerSample: %d\n", uBitsPerSample);
	Disp("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}

	/// 2. SPDIF Main Clock Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_InitBaseOfIp(HDMI_MODE);		// peater add for audio error
	HDMI_AUDIO_Spdif_InitAudioPath(PCM, uSampleRate, uBitsPerSample, 0);		// Because of PCM, uFrameSizeCode is 0.

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Wave file playing start by CPU
	SPDIF_StartPlayingPcmByCpu(PCM_BUF_START, uPcmSize, PCM, uChannelType, uSampleRate, uBitsPerSample);		//size: uPcmSize*2 byte

	/// 6.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
}

void TestHdmiSpdifByDma(void)
{
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1.1 Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	Disp("\nChannelType: %d\n", uChannelType);
	Disp("SampleRate: %d\n", uSampleRate);
	Disp("BitsPerSample: %d\n", uBitsPerSample);
	Disp("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}
	
	/// 2. SPDIF Main Clock Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_Spdif_InitAudioPath(PCM, uSampleRate, uBitsPerSample, 0);		// Because of PCM, uFrameSizeCode is 0.

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Wave file playing start by CPU
	SPDIF_StartPlayingPcmByDma(PCM_BUF_START, uPcmSize, PCM, uChannelType, uSampleRate, uBitsPerSample);		//size: uPcmSize*2 byte

	/// 6.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
	
}

void TestHdmiSpdifAc3ByCpu(void)
{
	u32 uSyncword, uSampleRate, uBitRate, uFrameSizeCode, Save_Ac3Size;	
	u32 uEpllValue;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");
	
	uAc3Size = 0x634420;
	//uAc3Size = 8133120;

	/// 1.1 Wave file parsing
	AC3FILE_ParseHeader(AC3_BUF_START, &uSyncword, &uSampleRate, &uBitRate, &uFrameSizeCode);

	UART_Printf("\nSyncword: 0x%04x\n", uSyncword);
	UART_Printf("SampleRate(Hz): %d\n", uSampleRate);
	UART_Printf("BitRate(kbps): %d\n", uBitRate);
	UART_Printf("FrameSizeCode: %d\n", uFrameSizeCode);
	Save_Ac3Size = uAc3Size;
	UART_Printf("Ac3FileSize : %d\n", Save_Ac3Size);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}
	
	/// 2. SPDIF Main Clock Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_Spdif_InitAudioPath(AC3, uSampleRate, 0, uFrameSizeCode);		// Because of PCM, uFrameSizeCode is 0.

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Wave file playing start by CPU
	SPDIF_StartPlayingAc3ByCpu(AC3_BUF_START, Save_Ac3Size, AC3, uSampleRate, uBitRate, uFrameSizeCode);		//size 0xfffff*2 byte

	/// 6.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
	
}

void TestHdmiSpdifAc3ByDma(void)
{
	u32 uSyncword, uSampleRate, uBitRate, uFrameSizeCode, Save_Ac3Size;	
	u32 uEpllValue;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	//uAc3Size = 2177000;
	uAc3Size = 0x634420;

	/// 1.1 Wave file parsing
	AC3FILE_ParseHeader(AC3_BUF_START, &uSyncword, &uSampleRate, &uBitRate, &uFrameSizeCode);

	UART_Printf("\nSyncword: 0x%04x\n", uSyncword);
	UART_Printf("SampleRate(Hz): %d\n", uSampleRate);
	UART_Printf("BitRate(kbps): %d\n", uBitRate);
	UART_Printf("FrameSizeCode: %d\n", uFrameSizeCode);
	Save_Ac3Size = uAc3Size;
	UART_Printf("Ac3FileSize : %d\n", Save_Ac3Size);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}
	
	/// 2. SPDIF Main Clock Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_Spdif_InitAudioPath(AC3, uSampleRate, 0, uFrameSizeCode);		// Because of PCM, uFrameSizeCode is 0.

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Wave file playing start by CPU
	SPDIF_StartPlayingAc3ByDma(AC3_BUF_START, Save_Ac3Size, AC3, uSampleRate, uBitRate, uFrameSizeCode);		//size 0xfffff*2 byte

	/// 6.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
	
}

void TestHdmiI2sByCpu(void)
{
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

#if 0
	uPcmSize = 0xac8ba0;
	uChannelType = 6;
	uSampleRate = 48000;
	uBitsPerSample = 16;

	WAVEFILE_PutHeader(WAVE_BUF_START, uPcmSize, uChannelType, uSampleRate, uBitsPerSample);
#endif

	/// 1.1 Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);


	/// 1.2 Protection Code, 512fs/24bit 
	if((eUsedAudioFs==FS_512)&&(uBitsPerSample==24))
	{
		UART_Printf("I2S[Tx] is not supported 512fs/24bit!");		
		Assert(0);
	}

	/// 2. I2S 5.1 Main Clock Setting, Syscon & I2S Init
	if (eUsedAudioMainClk == EXTERNAL_CLK)
	{
		I2S_InitIp(TX_MODE, EXTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	{
		I2S_SetAudioClock(uSampleRate, eUsedAudioFs);
		I2S_InitIp(TX_MODE, INTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_InitBaseOfIp(HDMI_MODE);		// peater add for audio error
	HDMI_AUDIO_I2s_InitAudioPath(PCM, uChannelType,uSampleRate, uBitsPerSample, 0);		

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Wave file playing start by CPU
	I2S_PlayByCpu(PCM_BUF_START, uPcmSize, uChannelType, &oI2s0);

	/// 6.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
}

void TestHdmiI2sByDma(void)
{
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1. Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 512fs/24bit 
	if((eUsedAudioFs==FS_512)&&(uBitsPerSample==24))
	{
		UART_Printf("I2S[Tx] is not supported 512fs/24bit!");		
		Assert(0);
	}

	/// 2. I2S 5.1 Main Clock Setting, Syscon & I2S Init
	if (eUsedAudioMainClk == EXTERNAL_CLK)
	{
		I2S_InitIp(TX_MODE, EXTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	{
		I2S_SetAudioClock(uSampleRate, eUsedAudioFs);
		I2S_InitIp(TX_MODE, INTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_InitBaseOfIp(HDMI_MODE);		// peater add for audio error
	HDMI_AUDIO_I2s_InitAudioPath(PCM, uChannelType,uSampleRate, uBitsPerSample, 0);		

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Wave file playing start by CPU
	I2S_PlayByDma(PCM_BUF_START, uPcmSize, uBitsPerSample, &oI2s0);

	/// 6.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);

}

void TestHdmiInterruptForSpdif(void)
{
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	volatile u32 uRead;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1.1 Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	Disp("\nChannelType: %d\n", uChannelType);
	Disp("SampleRate: %d\n", uSampleRate);
	Disp("BitsPerSample: %d\n", uBitsPerSample);
	Disp("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}

	/// 2. SPDIF Main Clock Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_InitBaseOfIp(HDMI_MODE);		// peater add for audio error
	HDMI_AUDIO_Spdif_InitAudioPath(PCM, uSampleRate, uBitsPerSample, 0);		// Because of PCM, uFrameSizeCode is 0.

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Interrupt
	HDMI_AUDIO_DisableAllIntCon();	
	INTC_Init();
	INTC_SetVectAddr(NUM_HDMI, Isr_HDMI_ADUIO);
	INTC_Enable(NUM_HDMI);	

	uRead = Inp32(0xfa110020);			// STATUS_EN
	uRead = (1<<6);   // // Audio buffer overflow interrupt enable
	Outp32(0xfa110020, uRead); 
	
	uRead = Inp32(0xfa130008);			// SPDIFIN_IRQ_MASK		
	uRead = (uRead &~((1<<7)|(1<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0))) |((1<<7)|(1<<4)|(1<<3)|(1<<2)|(0<<1)|(1<<0));   
	Outp32(0xfa130008, uRead); 	

	HDMI_AUDIO_EnableIntCon(IntrSPDIF_flag);

	/// 6. Wave file playing start by CPU
	SPDIF_StartPlayingPcmByCpu(PCM_BUF_START, uPcmSize, PCM, uChannelType, uSampleRate, uBitsPerSample);		//size: uPcmSize*2 byte

	/// 7.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
}

void TestHdmiInterruptForSpdifByDma(void)
{
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	u32 uRead;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1.1 Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	Disp("\nChannelType: %d\n", uChannelType);
	Disp("SampleRate: %d\n", uSampleRate);
	Disp("BitsPerSample: %d\n", uBitsPerSample);
	Disp("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}
	
	/// 2. SPDIF Main Clock Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_Spdif_InitAudioPathForInterrupt(PCM, uSampleRate, uBitsPerSample, 0);		// Because of PCM, uFrameSizeCode is 0.

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Interrupt
	HDMI_AUDIO_DisableAllIntCon();	
	INTC_Init();
	INTC_SetVectAddr(NUM_HDMI, Isr_HDMI_ADUIO);
	INTC_Enable(NUM_HDMI);	

	uRead = Inp32(0xfa110020);			// STATUS_EN
	uRead = (1<<6);   // // Audio buffer overflow interrupt enable
	Outp32(0xfa110020, uRead); 
	
	uRead = Inp32(0xfa130008);			// SPDIFIN_IRQ_MASK		
	uRead = (uRead &~((1<<7)|(1<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0))) |((1<<7)|(0<<4)|(0<<3)|(1<<2)|(1<<1)|(1<<0));   
	Outp32(0xfa130008, uRead); 	

	HDMI_AUDIO_EnableIntCon(IntrSPDIF_flag);
	
	/// 6. Wave file playing start by DMA
	SPDIF_StartPlayingPcmByDmaForInterrupt(PCM_BUF_START, uPcmSize, PCM, uChannelType, uSampleRate, uBitsPerSample);		//size: uPcmSize*2 byte

	/// 7. SPDIF Receiver Operation Setting
	Outp32(0xfa130004, 0x01);		// detect signal
	while((Inp32(0xfa13000c) & 0x03) != 0x02);
	Outp32(0xfa130004, 0x03);		// run

	while(!DMA_IsTransferDone(&oSPDIFDma0));

	/// 8. SPDIF Clock Power off
	SPDIF_TurnOffSpdifoutClock();
	DMA_StopCh(&oSPDIFDma0);

	/// 9.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);	

	/// 10. Interrupt Disable
	INTC_Disable(NUM_HDMI);
	HDMI_AUDIO_DisableAllIntCon();		
}

void TestHdmiInterruptForSpdifAC3ByDma(void)
{
	u32 uSyncword, uSampleRate, uBitRate, uFrameSizeCode, Save_Ac3Size;	
	u32 uEpllValue, uRead;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	//uAc3Size = 2177000;
	uAc3Size = 0x634420;

	/// 1.1 Wave file parsing
	AC3FILE_ParseHeader(AC3_BUF_START, &uSyncword, &uSampleRate, &uBitRate, &uFrameSizeCode);

	UART_Printf("\nSyncword: 0x%04x\n", uSyncword);
	UART_Printf("SampleRate(Hz): %d\n", uSampleRate);
	UART_Printf("BitRate(kbps): %d\n", uBitRate);
	UART_Printf("FrameSizeCode: %d\n", uFrameSizeCode);
	Save_Ac3Size = uAc3Size;
	UART_Printf("Ac3FileSize : %d\n", Save_Ac3Size);

	/// 1.2 Protection Code, 768fs
	if(eUsedAudioFs==FS_768)
	{
		UART_Printf("SPDIF is not supported 768fs!");		
		Assert(0);
	}
	
	/// 2. SPDIF Main Clock Setting
	if (eUsedAudioMainClk == EXTERNAL_CLK)
		;
	else if (eUsedAudioMainClk == INTERNAL_CLK)
		SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_Spdif_InitAudioPathForInterrupt(AC3, uSampleRate, 0, uFrameSizeCode);		// Because of PCM, uFrameSizeCode is 0.

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Interrupt
	HDMI_AUDIO_DisableAllIntCon();	
	INTC_Init();
	INTC_SetVectAddr(NUM_HDMI, Isr_HDMI_ADUIO);
	INTC_Enable(NUM_HDMI);	

	uRead = Inp32(0xfa110020);			// STATUS_EN
	uRead = (1<<6);   // // Audio buffer overflow interrupt enable
	Outp32(0xfa110020, uRead); 
	
	uRead = Inp32(0xfa130008);			// SPDIFIN_IRQ_MASK		
	uRead = (uRead &~((1<<7)|(1<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0))) |((1<<7)|(1<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0));   
	Outp32(0xfa130008, uRead); 	

	HDMI_AUDIO_EnableIntCon(IntrSPDIF_flag);

	/// 6. Wave file playing start by CPU
	SPDIF_StartPlayingAc3ByDmaForInterrupt(AC3_BUF_START, Save_Ac3Size, AC3, uSampleRate, uBitRate, uFrameSizeCode);		//size 0xfffff*2 byte

	/// 7. SPDIF Receiver Operation Setting
	Outp32(0xfa130004, 0x01);		// detect signal
	while((Inp32(0xfa13000c) & 0x73) != 0x12);
	Outp32(0xfa130004, 0x03);		// run

	while(!DMA_IsTransferDone(&oSPDIFDma0));

	/// 8. Interrupt Disable
	INTC_Disable(NUM_HDMI);
	HDMI_AUDIO_DisableAllIntCon();		
	
	/// 9. SPDIF Clock Power off
	SPDIF_TurnOffSpdifoutClock();
	DMA_StopCh(&oSPDIFDma0);

	/// 10.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);	
}

#if 0
void TestHdmiInterruptForI2s(void)
{
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	volatile u32 uRead;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1.1 Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 512fs/24bit 
	if((eUsedAudioFs==FS_512)&&(uBitsPerSample==24))
	{
		UART_Printf("I2S[Tx] is not supported 512fs/24bit!");		
		Assert(0);
	}

	/// 2. I2S 5.1 Main Clock Setting, Syscon & I2S Init
	if (eUsedAudioMainClk == EXTERNAL_CLK)
	{
		I2S_InitIp(TX_MODE, EXTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	{
		I2S_SetAudioClock(uSampleRate, eUsedAudioFs);
		I2S_InitIp(TX_MODE, INTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_InitBaseOfIp(HDMI_MODE);		// peater add for audio error
	HDMI_AUDIO_I2s_InitAudioPath(PCM, uChannelType,uSampleRate, uBitsPerSample, 0);		

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// PCLK_DSYS?
	/*
	uRead = Inp32(0xE0100300);	
	uRead = (uRead&~(7<<20))|(0<<20);	
	Outp32(0xE0100300, uRead);	

	SYSC_UpdateClkInform();
		UART_Printf(" ARMCLK: %.2fMHz\n HCLK_Msys: %.2fMHz  PCLK_Msys: %.2fMHz\n HCLK_Dsys: %.2fMHz  PCLK_Dsys: %.2fMHz\n HCLK_Psys: %.2fMHz  PCLK_Psys: %.2fMHz\n\n",
				(float)g_uARMCLK/1.0e6, (float)g_uHclkMsys/1.0e6, (float)g_uPclkMsys/1.0e6, (float)g_uHclkDsys/1.0e6, (float)g_uPclkDsys/1.0e6, (float)g_uHclkPsys/1.0e6, (float)g_uPclkPsys/1.0e6);

	Outp32(0xe0100500, 0x0000c001);		// for debugging PCLK_DYSY, 
	*/
	/// Interrupt
	HDMI_AUDIO_DisableAllIntCon();	
	INTC_Init();
	INTC_SetVectAddr(NUM_HDMI, Isr_HDMI_ADUIO);
	INTC_Enable(NUM_HDMI);	

	/// 5. I2S Interrupt,   // HDMI_IRQHandler
	/*
	uRead = Inp32(0xfa100000);			// INTC_CON
	//uRead = (uRead &~((1<<6)|(1<<5))) |((1<<6)|(1<<5));   // I2S interrupt enable
	uRead = (1<<6)|(1<<5);   // I2S interrupt enable
	Outp32(0xfa100000, uRead); 
	*/
	HDMI_AUDIO_EnableIntCon(IntrI2S_flag);

	uRead = Inp32(0xfa14005c);			// I2S_IRQ_MASK
	uRead = (uRead &~((1<<1)|(1<<0))) |((1<<1)|(1<<0));   // int_1,int_2 enable
	Outp32(0xfa14005c, uRead); 	

	uRead = Inp32(0xfa110020);			// STATUS_EN
	//uRead = (uRead &~((1<<6)|(1<<5))) |((1<<6)|(1<<5));   // Audio buffer overflow interrupt enable
	uRead = (0<<6);   // I2S interrupt disable
	Outp32(0xfa110020, uRead); 

	/// 6. Wave file playing start by CPU
	I2S_PlayByCpu(PCM_BUF_START, uPcmSize, uChannelType, &oI2s0);

	/// 7.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
}
#endif 

void TestHdmiInterruptForI2sByDma(void)
{
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	volatile u32 uRead;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1. Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 512fs/24bit 
	if((eUsedAudioFs==FS_512)&&(uBitsPerSample==24))
	{
		UART_Printf("I2S[Tx] is not supported 512fs/24bit!");		
		Assert(0);
	}

	/// 2. I2S 5.1 Main Clock Setting, Syscon & I2S Init
	if (eUsedAudioMainClk == EXTERNAL_CLK)
	{
		I2S_InitIp(TX_MODE, EXTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	{
		I2S_SetAudioClock(uSampleRate, eUsedAudioFs);
		I2S_InitIp(TX_MODE, INTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_InitBaseOfIp(HDMI_MODE);		// peater add for audio error
	HDMI_AUDIO_I2s_InitAudioPath(PCM, uChannelType,uSampleRate, uBitsPerSample, 0);		

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Interrupt
	HDMI_AUDIO_DisableAllIntCon();	
	INTC_Init();
	INTC_SetVectAddr(NUM_HDMI, Isr_HDMI_ADUIO);
	INTC_Enable(NUM_HDMI);	

	uRead = Inp32(0xfa110020);			// STATUS_EN
	uRead = (1<<6);   // // Audio buffer overflow interrupt enable
	Outp32(0xfa110020, uRead); 

	uRead = Inp32(0xfa14005c);			// I2S_IRQ_MASK
	uRead = (uRead &~((1<<1)|(1<<0))) |((1<<1)|(0<<0));   // wrong register interrupt enable. Bit per channel mismatch deleted in manual
	Outp32(0xfa14005c, uRead); 	

	HDMI_AUDIO_EnableIntCon(IntrI2S_flag);

	/// 6. Wave file playing start by DMA
	I2S_PlayByDma(PCM_BUF_START, uPcmSize, uBitsPerSample, &oI2s0);

	/// 7. Interrupt Disable
	INTC_Disable(NUM_HDMI);
	HDMI_AUDIO_DisableAllIntCon();		

	/// 8.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
}


void HDMI_ADUIO_InitSysconForHD(void)
{
/*
	// T-Block Power Off and On
	SYSC_SetBlockPowerOFF_IP(eHDTV);
	DelayfrTimer(milli, 100);
	SYSC_SetBlockPowerON_IP(eHDTV);

	// IP Clock Off and On
	SYSC_SetClkOFF_IP(eVP);
	SYSC_SetClkOFF_IP(eMIXER);
	SYSC_SetClkOFF_IP(eHDMI);
	DelayfrTimer(milli, 100);
	SYSC_SetClkON_IP(eVP);
	SYSC_SetClkON_IP(eMIXER);
	
	SYSC_SetClkON_IP(eHDMI);	
*/	
	SYSC_SetDivHDMI(0x96);
	SYSC_EnHDMIPHY();
}

void TestHdmiI2sUsingTimerByCpu(void)
{
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue,uMicroSec,uSec;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1.1 Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);


	/// 1.2 Protection Code, 512fs/24bit 
	if((eUsedAudioFs==FS_512)&&(uBitsPerSample==24))
	{
		UART_Printf("I2S[Tx] is not supported 512fs/24bit!");		
		Assert(0);
	}

	/// 2. I2S 5.1 Main Clock Setting, Syscon & I2S Init
	if (eUsedAudioMainClk == EXTERNAL_CLK)
	{
		I2S_InitIp(TX_MODE, EXTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	{
		I2S_SetAudioClock(uSampleRate, eUsedAudioFs);
		I2S_InitIp(TX_MODE, INTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_InitBaseOfIp(HDMI_MODE);		// peater add for audio error
	HDMI_AUDIO_I2s_InitAudioPath(PCM, uChannelType,uSampleRate, uBitsPerSample, 0);		

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Wave file playing start by CPU
	StartTimer(0);
	I2S_PlayByCpu(PCM_BUF_START, uPcmSize, uChannelType, &oI2s0);
   	uMicroSec = StopTimer(0);
	UART_Printf("%d microsecond has passed!\n",uMicroSec);
	uSec= uMicroSec/1000000;	
	UART_Printf("%d sec has passed!\n",uSec);

	/// 6.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
}


void HdmiAudioAgingTest(void)
{
#if 0
	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1.1 Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 512fs/24bit 
	if((eUsedAudioFs==FS_512)&&(uBitsPerSample==24))
	{
		UART_Printf("I2S[Tx] is not supported 512fs/24bit!");		
		Assert(0);
	}

	/// 2. I2S 5.1 Main Clock Setting, Syscon & I2S Init
	if (eUsedAudioMainClk == EXTERNAL_CLK)
	{
		I2S_InitIp(TX_MODE, EXTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	{
		I2S_SetAudioClock(uSampleRate, eUsedAudioFs);
		I2S_InitIp(TX_MODE, INTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_InitBaseOfIp(HDMI_MODE);		// peater add for audio error
	HDMI_AUDIO_I2s_InitAudioPath(PCM, uChannelType,uSampleRate, uBitsPerSample, 0);		

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Wave file playing start by CPU
	UART_Printf("If you want to exit, please press any key\n");	
	while(!UART_GetKey())
	{
		I2S_PlayByCpu(PCM_BUF_START, uPcmSize, uChannelType, &oI2s0);
	}

	/// 6.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);

#else

	u32 uDataAddr, uDataSize;
	u32 uChannelType, uSampleRate, uBitsPerSample;	
	u32 uEpllValue;
	
	UART_Printf("[HDMI_Audio Test!!!]\n");

	/// 1. Wave file parsing
	WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);

	UART_Printf("\nChannelType: %d\n", uChannelType);
	UART_Printf("SampleRate: %d\n", uSampleRate);
	UART_Printf("BitsPerSample: %d\n", uBitsPerSample);
	UART_Printf("PcmSize: %d\n", uPcmSize);

	/// 1.2 Protection Code, 512fs/24bit 
	if((eUsedAudioFs==FS_512)&&(uBitsPerSample==24))
	{
		UART_Printf("I2S[Tx] is not supported 512fs/24bit!");		
		Assert(0);
	}

	/// 2. I2S 5.1 Main Clock Setting, Syscon & I2S Init
	if (eUsedAudioMainClk == EXTERNAL_CLK)
	{
		I2S_InitIp(TX_MODE, EXTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}
	else if (eUsedAudioMainClk == INTERNAL_CLK)
	{
		I2S_SetAudioClock(uSampleRate, eUsedAudioFs);
		I2S_InitIp(TX_MODE, INTERNAL_MASTER, uChannelType, uSampleRate, uBitsPerSample, SPEAKER_MODE, eSerialDataFmt, uCdclk, AUDIO_WM8580, &oI2s0);	
	}

	/// 3. HDMI Video Setting
	HDMI_Audio_SetVideo();

	/// 4. HDMI Audio Setting
	HDMI_AUDIO_InitBaseOfIp(HDMI_MODE);		// peater add for audio error
	HDMI_AUDIO_I2s_InitAudioPath(PCM, uChannelType,uSampleRate, uBitsPerSample, 0);		

	HDMI_Enable(TRUE);	// Blues Screen on
	HDMI_TG_OnOff(TRUE);	
	HDMI_AUDIO_EnableBlueScreen(0xff, 0x00, 0x00);	
	HDMI_AUDIO_SetOnOff(1);		// keypoint

	/// 5. Wave file playing start by CPU
	UART_Printf("If you want to exit, please press any key\n");	
	while(!UART_GetKey())
	{
		I2S_PlayByDma(PCM_BUF_START, uPcmSize, uBitsPerSample, &oI2s0);
	}
	/// 6.  HDMI AUDIO off
	HDMI_AUDIO_SetOnOff(0);
#endif
}

void * func_hdmi_audio_test[][2]=
{
	//  									"123456789012345678901"
	0,  				  		         	  	"Exit",
	TestHdmiSpdifByCpu,					"Test Hdmi Spdif By Cpu",
	TestHdmiSpdifByDma,					"Test Hdmi Spdif By Dma",
	TestHdmiSpdifAc3ByCpu,				"Test Hdmi Spdif[AC3] By Cpu",		// Our TV was not supported 	
	TestHdmiSpdifAc3ByDma,				"Test Hdmi Spdif[AC3] By Dma",		// Our TV was not supported 
	TestHdmiI2sByCpu,					"Test Hdmi I2s By Cpu",	
	TestHdmiI2sByDma,					"Test Hdmi I2s By Dma",		
	TestHdmiInterruptForSpdif,				"TestHdmiInterruptForSpdif",	
	TestHdmiInterruptForSpdifByDma,		"TestHdmiInterruptForSpdif[DMA]",	
	TestHdmiInterruptForSpdifAC3ByDma,	"TestHdmiInterruptForSpdif_AC3[DMA]",		
	//TestHdmiInterruptForI2s,				"TestHdmiInterruptForI2s",
	TestHdmiInterruptForI2sByDma,			"TestHdmiInterruptForI2s[DMA]",
	TestHdmiI2sUsingTimerByCpu,			"TestHdmiI2sUsingTimerByCpu",
	HdmiAudioAgingTest,					"Hdmi Audio Aging Test",	
	0,0
};

void Test_HDMI_AUDIO(void)
{
	int i;
	hdmi_audio_clk_sel eClkSel;
	u32 uSampleRate, uFsVal;	
	u32 uChannelType, uBitsPerSample,uRead;
	while(1)
	{
		i=0;

		/// 1. HDMI PHY Initialize
		InitBaseHWForHdout();
		HDMI_HW_Init();
		UART_Printf("\n");
		
		/// 2. SPDIF Initialize
		/// 2.1 GPIO Setting for SPDIF 
		SPDIF_SetGpio(1);
		
		/// 3. Select SPDIF&I2S Main clock 
		eUsedAudioMainClk = INTERNAL_CLK;		
		eUsedAudioFs = FS_512;			// if you use HDMI Audio clock[SPDIF clock], you should select 256/384/512fs. 

		if (eUsedAudioMainClk == EXTERNAL_CLK)
			UART_Printf("\nUsed External Clock!");
		else if (eUsedAudioMainClk == INTERNAL_CLK)
			UART_Printf("\nUsed Internal Clock!");	
		
		UART_Printf(" Befoe excute SPDIF & I2S Playign test,\n");
		UART_Printf("Upload the sound file of playing at the Memory(addr: 0x%08x).\n", WAVE_BUF_START);

		/// 4. TZPC NonSecure Init
		TZPC_NonSecureInit();

		/// 5. I2S Initialize	
		/// 5.1 GPIO Setting for SPDIF 	
		 GPIO_SetFunctionForI2s();

#if 0
	uRead = Inp32(0xE0100300);	
	uRead = (uRead&~(7<<20))|(0<<20);	
	Outp32(0xE0100300, uRead);	

	SYSC_UpdateClkInform();
	UART_Printf("Used HDMI Audio System Clock[PCLK_Dsys]: %.2fMHz\n", (float)g_uPclkDsys/1.0e6);
#endif 
		
#if 1
		/// 6 HDMI Audio clock source selection
		/// 6.1 SysCon clock Setting
		eClkSel = PCLK;

		if(eClkSel==PCLK)
		{
			SYSC_UpdateClkInform();
			UART_Printf("Used HDMI Audio System Clock[PCLK_Dsys]: %.2fMHz\n", (float)g_uPclkDsys/1.0e6);
		}		
		else if(eClkSel==SPDIF_CLK)
		{
			WAVEFILE_ParseHeader(WAVE_BUF_START, &uPcmSize, &uChannelType, &uSampleRate, &uBitsPerSample);
			/// 1. Clock Cal
			if (eUsedAudioFs == FS_256)
				uFsVal = 256;
			else if (eUsedAudioFs == FS_384)
				uFsVal = 384;
			else if (eUsedAudioFs == FS_512)
				uFsVal = 512;
			else 
			Assert(0);	
			SPDIF_SetAudioClock(uSampleRate, eUsedAudioFs);
			UART_Printf("Used HDMI Audio System Clock[SPDIF Clock]: %.2fMHz\n", (float)((uSampleRate*uFsVal)/1.0e6));
		}
		else
			Assert(0);
		/// 6.2 HDMI Audio clock Select
		HDMI_AUDIO_SelectClkSourc(eClkSel);
#endif			
		UART_Printf("\n\n=============== HDMI Audio Function Test ==================\n\n");
		
		while(1)
		{   //UART_Printflay menu
			UART_Printf("%2d:%s",i,func_hdmi_audio_test[i][1]);
			i++;
			if((int)(func_hdmi_audio_test[i][0])==0)
			{
				UART_Printf("\n");
				break;
			}
			if((i%1)==0)
			UART_Printf("\n");
		}
		UART_Printf("\n==========================================================\n");
		UART_Printf("Select #Item or Press Enter key to exit : ");
		i = UART_GetIntNum(); 
		//if(i==-1) break;		// return.
		if(i==0)break;
		if(i>=0 && (i<((sizeof(func_hdmi_audio_test)-1)/8)) )	// select and execute...
			( (void (*)(void)) (func_hdmi_audio_test[i][0]) )();

		ResetEDID();
	}	
}

