/*******************************************************************************

	File Name: hdmi_audio.c
	Description: S5PV210 HDMI Audio Controller Function Test Code

   	Version:  
   	History:
             R0.0(2009): HakSong.Kim First draft

        Memo:
                
********************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "def.h"
#include "option.h"
#include "util.h"
#include "v210_sfr.h"
#include "system.h"
#include "intc.h"
#include "dma.h"
#include "gpio.h"

#include "sysc.h"
#include "hdmi_audio.h"
#include "hdmi_reg.h"
#include "intc.h"

#define DBG_HDMI

#ifndef DBG_SPDIF
#define hmDbg(x)    0
#else
#define hmDbg(x) Dbg x
#endif

#ifndef DISP_HDMI_REG
#define hmOutp32(a, d) Outp32(a, d)
#define hmInp32(a, d)  Inp32(a, d)
#define hmOutp16(a, d) Outp16(a, d)
#define hmInp16(a, d)  Inp16(a, d)
#define hmOutp8(a, d) Outp8(a, d)
#define hmInp8(a, d)  Inp8(a, d)
#define hmDispVector(x) 0
#else
#define hmOutp32(a, d) (Disp("Outp32(\'h%08x, \'h%08x);\n", a, d), Outp32(a, d))
#define hmInp32(a, d) (Disp("Inp32(\'h%08x, d); ", a), Inp32(a, d), Dbg("\/\/ d=0x%08x\n", d))
#define hmOutp16(a, d) Disp("Outp32(\'h%08x, \'h%08x);\n", a, d), Outp16(a, d)
#define hmInp16(a, d) (Disp("Inp32(\'h%08x, d); ", a), Inp16(a, d), Dbg("\/\/ d=0x%08x\n", d))
#define hmOutp8(a, d) Disp("Outp32(\'h%08x, \'h%08x);\n", a, d), Outp8(a, d)
#define hmInp8(a, d) (Disp("Inp32(\'h%08x, d); ", a), Inp8(a, d), Dbg("\/\/ d=0x%08x\n", d))
#define hmDispVector(x) Dbg x
#endif

#define hmOutp32Multi(uLowRegAddr, uWrData) { hmOutp32(uLowRegAddr, uWrData&0xff); \
                                                hmOutp32(uLowRegAddr+0x4, uWrData>>8); }

// AVI InfoFrame
#define AVI_RGB_IF                                       0x0<<5
#define AVI_YCBCR444_IF                                  0x2<<5
#define AVI_ITU709                                       0x2<<6
#define AVI_PAR_16_9                                     0x2<<4
#define AVI_NO_PIXEL_REPEAT                              0x0<<0

#define AVI_VIC_2                                        2<<0 // 720x480p@59.94/60Hz 4:3
#define AVI_VIC_3                                        3<<0 // 720x480p@59.94/60Hz 16:9
#define AVI_VIC_4                                        4<<0 // 1280x720p@59.94/60Hz 16:9
#define AVI_VIC_5                                        5<<0 // 1920x1080i@59.94/60Hz 16:9
#define AVI_VIC_17                                       17<<0 // 720x576p@50Hz 4:3
#define AVI_VIC_18                                       18<<0 // 720x576p@50Hz 16:9
#define AVI_VIC_19                                       19<<0 // 1280x720p@50Hz 16:9
#define AVI_VIC_20                                       20<<0 // 1920x1080i@50Hz 16:9

/******************************************************************************
*					HDMI Audio Global Value									
 ******************************************************************************/
IMG_RESOLUTION m_eImgSz;
SCAN_MODE m_eScanMode;

HDMI_RUN_MODE m_eRunMode;
HDCP_STATE m_eAuthState;
	
bool m_bIsRepeater;
	
u8 m_ucMode;
u8 m_ucAudioFreq;
u8 m_ucVideoFormat;
u8 m_ucColorSpace;
u8 m_ucCableCurrent;
u8 m_ucHdcpBuf[0x50];

// AVI info
u32 m_uImgIfTypeVal; // Y[1:0]
u32 m_uColorimetryVal; // C[1:0]
u32 m_uParVal; // M[1:0] : Picture Aspect Ratio
u32 m_uPixelRepVal; // PR[3:0]
u32 m_uVideoCodeVal; // VIC[6:0]

/******************************************************************************
*					HDMI Audio I/O Functions									
 ******************************************************************************/
// [Initialize base of IP]
void HDMI_AUDIO_InitBaseOfIp(HDMI_RUN_MODE eRunMode)
{
	m_eRunMode = eRunMode;

	if (eRunMode == DVI_MODE)
	{
		hmOutp32(rHDMI_MODE_SEL, 
			0x1<<0 | // Enable dvi mode
			0x0<<1); // Disalbe hdmi mode
		hmOutp32(rHDMI_CON_0, 
			0x0<<0 | // HDMI system disalbe
			0x0<<1 | // TMDS power down
			0x0<<2 | // Discard audio sample
			0x0<<5 ); // Disable blue screen
		hmOutp32(rHDMI_CON_1, 
			0x0<<0 | // 3 time repetition (why?)
			0x0<<3 | // aux mute
			0x2<<5); // Pixel value limitation: RGB mode
		hmOutp32(rHDMI_CON_2,
			0x0<<0 | // Every sync for interlace mode
			0x1<<1 | // In DVI mode, control signal is dvi
			0x0<<4 | // In authentication period, control signal is sent
			0x1<<5); // ?

		// Set info frame packet reg.
		hmOutp32(rHDMI_AUI_CON, 0x0); // Don't transmit AUI infoframe packet

		// Set audio related packet reg.
		hmOutp32(rHDMI_ACR_CON, 
			0x0<<0 | // Don't transmit
			0x0<<3); // Always CTS value 1
	}
	else // HDMI_MODE
	{
		hmOutp32(rHDMI_MODE_SEL, 
			0x0<<0 | // Disable dvi mode
			0x1<<1); // Enable hdmi mode
			
		hmOutp32(rHDMI_CON_0, 
			0x0<<0 | // HDMI system disalbe
			0x0<<1 | // TMDS power down
			0x1<<2 | // Audio sample packet is generated
			0x0<<5 ); // Disable blue screen
		hmOutp32(rHDMI_CON_1, 
			0x0<<0 | // 3 time repetition (why?)
			0x0<<2 | // TMDS pre-emphasis off
			0x0<<3 | // aux mute
			0x0<<5); // Pixel value limitation: Bypass
		hmOutp32(rHDMI_CON_2,
			0x0<<0 | // Every sync for interlace mode
			0x0<<1 | // In DVI mode, control signal is hdmi
			0x0<<4 | // In authentication period, control signal is sent
			0x0<<5); // ?

		// Set info frame packet reg.
		hmOutp32(rHDMI_AUI_CON, 0x2); // Transmit AUI infoframe packet in every sync.

		// Set audio related packet reg.
		hmOutp32(rHDMI_ACR_CON, 
			0x4<<0 | // Measured CTS mode
			0x0<<3); // Always CTS value 1
	}

	hmOutp32(rHDCP_ENC_EN, 0x0); // Disable encryption

	// RGB value in blue screen
	hmOutp32(rHDMI_BLUE_SCREEN_0, 0x0); // Blue
	hmOutp32(rHDMI_BLUE_SCREEN_1, 0x0); // Green
	hmOutp32(rHDMI_BLUE_SCREEN_2, 0x0); // Yellow 

	hmOutp32(rHDMI_V_BLANK_F_0, 0x49);
	hmOutp32(rHDMI_V_BLANK_F_1, 0x2a);
	hmOutp32(rHDMI_V_BLANK_F_2, 0x23);

	// Set full range of Y
	hmOutp32(rHDMI_YMAX, 254);
	hmOutp32(rHDMI_YMIN, 1);

	// Set full range of CbCr
	hmOutp32(rHDMI_CMAX, 254);
	hmOutp32(rHDMI_CMIN, 1);

	hmOutp32(rHDMI_VACT_ST_MG, 0x2b);
	hmOutp32(rHDMI_VACT_END_MG, 0x30);

	if (eRunMode == DVI_MODE)
	{
		m_uImgIfTypeVal = AVI_RGB_IF;
		m_uColorimetryVal = AVI_ITU709;		
		HDMI_AUDIO_SetCsc(YUV709_TO_RGB_LR);
	}
	else
	{
		m_uImgIfTypeVal = AVI_YCBCR444_IF	;
		m_uColorimetryVal = AVI_ITU709;
		HDMI_AUDIO_SetCsc(HDMI_BYPASS);
	}
}

// [Set a color space conversion]
void HDMI_AUDIO_SetCsc(CSC_TYPE eCscType)
{
	u32 uYGCoef, uYBCoef, uYRCoef;
	u32 uCbGCoef, uCbBCoef, uCbRCoef;
	u32 uCrGCoef, uCrBCoef, uCrRCoef;	
	u32 uCscConVal;

	// Set Y coefficient
	uYGCoef = 
		(eCscType == YUV601_TO_RGB_LR || eCscType == YUV709_TO_RGB_LR || eCscType == HDMI_BYPASS) ? 256 :
		298; // YUV601_TO_RGB_FR || YUV709_TO_RGB_FR
	uYBCoef =
		(eCscType == YUV601_TO_RGB_LR) ? 938 :
		(eCscType == YUV601_TO_RGB_FR) ? 924 :
		(eCscType == YUV709_TO_RGB_LR) ? 978 :	
		(eCscType == YUV709_TO_RGB_FR) ? 970 :
		0; // Mixer data bypass
	uYRCoef =
		(eCscType == YUV601_TO_RGB_LR) ? 846 :
		(eCscType == YUV601_TO_RGB_FR) ? 816 :
		(eCscType == YUV709_TO_RGB_LR) ? 907 :	
		(eCscType == YUV709_TO_RGB_FR) ? 888 :
		0; // Mixer data bypass

	// Set Cb coefficient		
	uCbGCoef =
		(eCscType == YUV601_TO_RGB_LR || eCscType == YUV709_TO_RGB_LR) ? 256 :
		(eCscType == YUV601_TO_RGB_FR || eCscType == YUV709_TO_RGB_FR) ? 298 :
		0; // Mixer data bypass
	uCbBCoef =
		(eCscType == YUV601_TO_RGB_LR) ? 443 :
		(eCscType == YUV601_TO_RGB_FR) ? 516 :
		(eCscType == YUV709_TO_RGB_LR) ? 464 :	
		(eCscType == YUV709_TO_RGB_FR) ? 540 :
		256; // Mixer data bypass
	uCbRCoef = 0;

	// Set Cr coefficient
	uCrGCoef = 
		(eCscType == YUV601_TO_RGB_LR || eCscType == YUV709_TO_RGB_LR) ? 256 :
		(eCscType == YUV601_TO_RGB_FR || eCscType == YUV709_TO_RGB_FR) ? 298 :
		0; // Mixer data bypass
	uCrBCoef = 0;
	uCrRCoef =
		(eCscType == YUV601_TO_RGB_LR) ? 350 :
		(eCscType == YUV601_TO_RGB_FR) ? 408 :
		(eCscType == YUV709_TO_RGB_LR) ? 394 :	
		(eCscType == YUV709_TO_RGB_FR) ? 458 :
		256; // Mixer data bypass
	// Set CSC offset
	uCscConVal = (eCscType == HDMI_BYPASS) ? 0x33 : 0x3;
}

// [Initialize spdif audio path]
void HDMI_AUDIO_Spdif_InitAudioPath(CODEC_TYPE eAudioCodec, u32 uSampleRate, u32 uBits, u32 uFrameSizeCode)
{
	u32 uDataType, uBps, wl, uRptCnt, uValueN, uCts, uType, uCh, uSample;
	u32 uBpsType, uSumOfBits, bytes1, bytes2, bytes3, uCheckSum, uBitrate;
	volatile u32 uRead;

	uDataType = 
		(eAudioCodec==PCM) ? 0 :
		(eAudioCodec==AC3) ? 1 : 0xff;
	Assert(uDataType!=0xff);

	hmOutp32(rHDMI_I2S_CLK_CON, 0x1);		// Set I2S Clcok Enable
	hmOutp32(rHDMI_SPDIF_CONFIG_1, (1<<6)|(1<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0));
	hmOutp32(rHDMI_SPDIF_CONFIG_2, 0); // 

	uBps = (eAudioCodec==PCM) ? uBits : 16; // Ac3 16bit only
	wl = 11; // Only 4'b1011 // 24bit
		
	uRptCnt = (eAudioCodec==AC3) ? 1536*2-1 : 0;
	hmOutp32(rHDMI_SPDIF_USER_VALUE_1,  ((uRptCnt&0xf)<<4)|wl);
	hmOutp32(rHDMI_SPDIF_USER_VALUE_2, (uRptCnt>>4)&0xff);
	hmOutp32(rHDMI_SPDIF_USER_VALUE_3, uFrameSizeCode&0xff);
	hmOutp32(rHDMI_SPDIF_USER_VALUE_4, (uFrameSizeCode>>8)&0xff);

	uRead = Inp32(rHDMI_I2S_IN_MUX_CON);
	uRead = (uRead &~((1<<4)|(3<<2)|(1<<1)|(1<<0))) |((1<<4)|(0<<2)|(0<<1)|(1<<0));	// Set audio_sel	, SPDIF CUV 	
	hmOutp32(rHDMI_I2S_IN_MUX_CON, uRead);		

	hmOutp32(rHDMI_I2S_MUX_CH, 0xff);		// enable all data
	hmOutp32(rHDMI_I2S_MUX_CUV, 0x03);	// enable all channel

	/// SPDIF Intterrupt
#if 0
	uRead = Inp32(rHDMI_SPDIF_IRQ_MASK);			// SPDIFIN_IRQ_MASK		
	uRead = (uRead &~((1<<7)|(1<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0))) |((1<<7)|(1<<4)|(1<<3)|(1<<2)|(0<<1)|(1<<0));   
	Outp32(rHDMI_SPDIF_IRQ_MASK, uRead); 	

	uRead = Inp32(rHDMI_INTC_CON);
	uRead = (uRead &~((1<<6)|(1<<1))) |((1<<6)|(1<<1));	 	
	hmOutp32(rHDMI_INTC_CON, 0x01); 	 
#endif 

	hmOutp32(rHDMI_SPDIF_CLK_CTRL, 0x0);
	hmOutp32(rHDMI_SPDIF_CLK_CTRL, 0x1);		// clock enable

	hmOutp32(rHDMI_SPDIF_OP_CTRL, 0x01);		// detect signal
	hmOutp32(rHDMI_SPDIF_OP_CTRL, 0x03);		// run

	uValueN = 	(uSampleRate==32000) ? 4096 :
				(uSampleRate== 44100) ? 6272 :
				(uSampleRate== 88200) ? 12544 :
				(uSampleRate==176400) ? 25088 :
				(uSampleRate== 48000) ? 6144 :
				(uSampleRate== 96000) ? 12288 :
				(uSampleRate==192000) ? 24576 : 0;	
	Assert(uValueN != 0);
	uCts =	 	(uSampleRate==32000) ? 27000 : 
				(uSampleRate== 44100) ? 30000 :
				(uSampleRate== 88200) ? 30000 :
				(uSampleRate==176400) ? 30000 :
				(uSampleRate== 48000) ? 27000 :
				(uSampleRate== 96000) ? 27000 :
				(uSampleRate==192000) ? 27000 : 0;
	Assert(uCts != 0);

	hmOutp32(rHDMI_ASP_CON, 0xf);		
//	hmOutp32(rHDMI_ASP_CON, 1);		// peater for protecting Audio mute error
//	hmOutp32(rHDMI_ASP_SP_FLAT, 0xf); // All Subpackets contain audio samples
	hmOutp32(rHDMI_ASP_SP_FLAT, 0x0); 		// peater for protecting Audio Mute error
	hmOutp32(rHDMI_ASP_CHCFG0, 1<<3|0);
	hmOutp32(rHDMI_ASP_CHCFG1, 1<<3|0);
	hmOutp32(rHDMI_ASP_CHCFG2, 1<<3|0);
	hmOutp32(rHDMI_ASP_CHCFG3, 1<<3|0);

	hmOutp32(rHDMI_ACR_N0, uValueN & 0xff);
	hmOutp32(rHDMI_ACR_N1, (uValueN>>8) & 0xff);
	hmOutp32(rHDMI_ACR_N2, (uValueN>>16) & 0xff);

	hmOutp32(rHDMI_ACR_MCTS0, uCts & 0xff);
	hmOutp32(rHDMI_ACR_MCTS1, (uCts>>8) & 0xff);
	hmOutp32(rHDMI_ACR_MCTS2, (uCts>>16) & 0xff);

	hmOutp32(rHDMI_ACR_CTS0, uCts & 0xff);
	hmOutp32(rHDMI_ACR_CTS1, (uCts>>8) & 0xff);
	hmOutp32(rHDMI_ACR_CTS2, (uCts>>16) & 0xff);

	hmOutp32(rHDMI_ACR_CON, 4); // Messaured

	// AUI Packet set.
	uType = (eAudioCodec==PCM) ? 1 :  // PCM
		(eAudioCodec==AC3) ? 2 : 0; //AC3 or Refer stream header
	uCh = (eAudioCodec==PCM) ? 1 : 0; // 2ch or refer to stream header
	uSample = (uSampleRate==32000) ? 1 :
		(uSampleRate== 44100) ? 2 :
		(uSampleRate== 48000) ? 3 :
		(uSampleRate== 88200) ? 4 :
		(uSampleRate== 96000) ? 5 :
		(uSampleRate==176400) ? 6 :
		(uSampleRate==192000) ? 7 : 0;
	uBpsType = (uBps==16) ? 1 :
		(uBps==20) ? 2 :
		(uBps==24) ? 3 : 0;
	uBpsType = (eAudioCodec==PCM) ? uBpsType : 0;
		
	uSumOfBits = (0x84+0x1+10);//header
	bytes1 = (uType<<4)|uCh;
	bytes2 = (uSample<<2)|uBpsType;
	uBitrate= 512; // AC3 Test peater
	bytes3 = (eAudioCodec==PCM) ? 0 : (uBitrate/8) ;
	uSumOfBits += (bytes1+bytes2+bytes3);
	uCheckSum = 256-uSumOfBits;

	hmOutp32(rHDMI_AUI_CHECK_SUM, uCheckSum); // 
	hmOutp32(rHDMI_AUI_BYTE1, bytes1);
	hmOutp32(rHDMI_AUI_BYTE2, bytes2);
	hmOutp32(rHDMI_AUI_BYTE3, bytes3); // Pcm or stream
	hmOutp32(rHDMI_AUI_BYTE4, 0x00); // 2ch pcm or Stream
	hmOutp32(rHDMI_AUI_BYTE5, 0x00);	 // 2ch pcm or Stream

	hmOutp32(rHDMI_ACP_CON, 2);	 
	hmOutp32(rHDMI_ACP_TYPE, 1);	 

	hmOutp32(rHDMI_GCP_BYTE1, 0x10); 
	hmOutp32(rHDMI_GCP_CON, 0x2); // packet will be transmitted within 384 cycles after active sync.	
}

// [Initialize I2S audio path]
void HDMI_AUDIO_I2s_InitAudioPath(CODEC_TYPE eAudioCodec, u32 uChannelType, u32 uSampleRate, u32 uBitsPerSample, u32 uFrameSizeCode)
{
	u32 uDataType, uBps, wl, uRptCnt, uValueN, uCts, uType, uCh, uSample;
	u32 uBpsType, uSumOfBits, bytes1, bytes2, bytes3, uCheckSum, uBitrate;
	u32 uBitCh, uDataNum, uSampleFrq;	
	volatile u32 uRead;

	/// 1. Bit per Channel Cal
	if(uBitsPerSample==16)
	{
		uDataNum = 1;
		uBitCh = 0;
	}
	else if(uBitsPerSample==20)
	{
		uDataNum = 2;
		uBitCh = 1;
	}
	else if(uBitsPerSample==24)
	{
		uDataNum = 3;
		uBitCh = 1;
	}
	else
		Assert(0);	

	/// 2. I2S Receiver Setting
	hmOutp32(rHDMI_I2S_CLK_CON, 0x00);	// Disable Audio Receiver. 
	hmOutp32(rHDMI_I2S_CLK_CON, 0x01);	// Enable Audio Receiver. Set I2S_en bit 
	
	uRead = Inp32(rHDMI_I2S_DSD_CON);
	uRead |= 1;
	hmOutp32(rHDMI_I2S_DSD_CON, uRead);	// Disable DSD module. Clear dsd_en bit.

	// Configuration I2S input ports. Configure I2S_PIN_SEL_0~4
	uRead = Inp32(rHDMI_I2S_PIN_SEL_0);			// SCLK(BCLK) => i_i2s_in[5]
	uRead = (uRead &~((7<<4)|(7<<0))) | (5<<4|6<<0);   // LRCLK => i_i2s_in[6]
	hmOutp32(rHDMI_I2S_PIN_SEL_0, uRead); 

	uRead = Inp32(rHDMI_I2S_PIN_SEL_1);			// SDO_1 => i_i2s_in[3]
	uRead = (uRead &~((7<<4)|(7<<0))) |(1<<4|4<<0);    // SDO_0 => i_i2s_in[4] 
	hmOutp32(rHDMI_I2S_PIN_SEL_1, uRead); 

	uRead = Inp32(rHDMI_I2S_PIN_SEL_2);			// SDO_3 => i_i2s_in[1]
	uRead = (uRead &~((7<<4)|(7<<0))) |(1<<4|2<<0);	// SDO_2 => i_i2s_in[2] 
	hmOutp32(rHDMI_I2S_PIN_SEL_2, uRead); 

	uRead = Inp32(rHDMI_I2S_PIN_SEL_3);			// SDO_4 => i_i2s_in[0]
	uRead = (uRead &~(7<<0)) |(0<<0);
	hmOutp32(rHDMI_I2S_PIN_SEL_3, uRead); 

	// I2S_CON_1 & 2
	uRead = Inp32(rHDMI_I2S_CON_1);
	uRead = (uRead &~((1<<1)|(1<<0))) |(1<<1|0<<0);
	hmOutp32(rHDMI_I2S_CON_1, uRead); 
	
	uRead = Inp32(rHDMI_I2S_CON_2);
	uRead = (uRead &~((1<<6)|(3<<4)|(3<<2)|(3<<0))) |((0<<6)|(uBitCh<<4)|uDataNum<<2|0<<0);
	hmOutp32(rHDMI_I2S_CON_2, uRead); 

	// Configure register related to CUV information
	uRead = Inp32(rHDMI_I2S_CH_ST_0);
	uRead = (uRead &~((3<<6)|(7<<3)|(1<<2)|(1<<1)|(1<<0))) |((0<<6)|(0<<3)|(0<<2)|(0<<1)|(1<<0));
	hmOutp32(rHDMI_I2S_CH_ST_0, uRead); 

	uRead = Inp32(rHDMI_I2S_CH_ST_1);
	uRead = (uRead &~(0xff<<0)) |(0<<0);
	hmOutp32(rHDMI_I2S_CH_ST_1, uRead); 
	
	uRead = Inp32(rHDMI_I2S_CH_ST_2);
	uRead = (uRead &~(0xff<<0)) |(0<<0);
	hmOutp32(rHDMI_I2S_CH_ST_2, uRead); 

	uSampleFrq = (uSampleRate==44100) ? 0 : 
				(uSampleRate==48000) ? 2 :
				(uSampleRate==32000) ? 3 :
				(uSampleRate==96000) ? 0xa : 0x0;
	
	uRead = Inp32(rHDMI_I2S_CH_ST_3);
	uRead = (uRead &~((3<<4)|(0xf<<0))) |(1<<4|uSampleFrq<<0);		// Sampling Frequency : 44.1Khz(00), 48Khz(10), 32KHz(11), 96Khz(1010) 
	hmOutp32(rHDMI_I2S_CH_ST_3, uRead); 

	uRead = Inp32(rHDMI_I2S_CH_ST_4);
	uRead = (uRead &~((0xf<<4)|(7<<1)|(1<<0))) |((0xf<<4)|(5<<1)|(1<<0));	// Original Sampling Frequency
	hmOutp32(rHDMI_I2S_CH_ST_4, uRead); 

	hmOutp32(rHDMI_I2S_CH_ST_SH_0, 0x0);	
	hmOutp32(rHDMI_I2S_CH_ST_SH_1, 0x0);	
	hmOutp32(rHDMI_I2S_CH_ST_SH_2, 0x0);	
	hmOutp32(rHDMI_I2S_CH_ST_SH_3, 0x2);	
	hmOutp32(rHDMI_I2S_CH_ST_SH_4, 0x0);	

	// I2S_CH_ST_CON
	uRead = Inp32(rHDMI_I2S_CH_ST_CON);
	uRead = (uRead &~(1<<0)) |(1<<0);
	hmOutp32(rHDMI_I2S_CH_ST_CON, uRead); 

	// I2S_IN_MUX_CON
	uRead = Inp32(rHDMI_I2S_IN_MUX_CON);
	uRead = (uRead &~((1<<4)|(3<<2)|(1<<1)|(1<<0))) |((1<<4)|(1<<2)|(1<<1)|(1<<0));	
	hmOutp32(rHDMI_I2S_IN_MUX_CON, uRead); 

	// I2S_MUX_CH
	uRead = Inp32(rHDMI_I2S_MUX_CH);
	uRead = (uRead &~(0xff<<0)) |(0x3f<<0);
	hmOutp32(rHDMI_I2S_MUX_CH, uRead); 

	// I2S_MUX_CUV
	uRead = Inp32(rHDMI_I2S_MUX_CUV);
	uRead = (uRead &~(0x3<<0)) |(0x3<<0);
	hmOutp32(rHDMI_I2S_MUX_CUV, uRead); 

	uValueN = 	(uSampleRate==32000) ? 4096 :
				(uSampleRate== 44100) ? 6272 :
				(uSampleRate== 88200) ? 12544 :
				(uSampleRate==176400) ? 25088 :
				(uSampleRate== 48000) ? 6144 :
				(uSampleRate== 96000) ? 12288 :
				(uSampleRate==192000) ? 24576 : 0;	
	Assert(uValueN != 0);
	uCts =	 	(uSampleRate==32000) ? 27000 : 
				(uSampleRate== 44100) ? 30000 :
				(uSampleRate== 88200) ? 30000 :
				(uSampleRate==176400) ? 30000 :
				(uSampleRate== 48000) ? 27000 :
				(uSampleRate== 96000) ? 27000 :
				(uSampleRate==192000) ? 27000 : 0;
	Assert(uCts != 0);

	hmOutp32(rHDMI_ASP_SP_FLAT, 0x0); 		// peater for protecting Audio Mute error
	hmOutp32(rHDMI_ASP_CHCFG0, 0x08);
	hmOutp32(rHDMI_ASP_CHCFG1, 0x1a);
	hmOutp32(rHDMI_ASP_CHCFG2, 0x2c);
	hmOutp32(rHDMI_ASP_CHCFG3, 0x3e);

	hmOutp32(rHDMI_ACR_N0, uValueN & 0xff);
	hmOutp32(rHDMI_ACR_N1, (uValueN>>8) & 0xff);
	hmOutp32(rHDMI_ACR_N2, (uValueN>>16) & 0xff);

	hmOutp32(rHDMI_ACR_MCTS0, uCts & 0xff);
	hmOutp32(rHDMI_ACR_MCTS1, (uCts>>8) & 0xff);
	hmOutp32(rHDMI_ACR_MCTS2, (uCts>>16) & 0xff);

	hmOutp32(rHDMI_ACR_CTS0, uCts & 0xff);
	hmOutp32(rHDMI_ACR_CTS1, (uCts>>8) & 0xff);
	hmOutp32(rHDMI_ACR_CTS2, (uCts>>16) & 0xff);

	// ASP_CON
	uRead = Inp32(rHDMI_ASP_CON);
	uRead = (uRead &~((1<<7)|(3<<5)|(1<<4)|(0xf<<0))) |((0<<7)|(0<<5)|(1<<4)|(0x7<<0));
	hmOutp32(rHDMI_ASP_CON, uRead); 

	// ACR_CON
	uRead = Inp32(rHDMI_ACR_CON);
	uRead = (uRead &~((3<<3)|(7<<0))) |((0<<3)|(4<<0));
	hmOutp32(rHDMI_ACR_CON, uRead); 

	uRead = Inp32(rHDMI_CON_0);
	uRead = (uRead &~((1<<7)|(1<<6)|(1<<2)|1<<0)) |((1<<7)|(1<<6)|(1<<2)|(1<<0));
	hmOutp32(rHDMI_CON_0, uRead); 

	HDMI_AUDIO_SetAudioClockRecovery(uSampleRate);	
}

// [Enable a blue screen]
void HDMI_AUDIO_EnableBlueScreen(u8 ucCh0Color, u8 ucCh1Color, u8 ucCh2Color)
{
	u32 uConReg;

	hmOutp32(rHDMI_BLUE_SCREEN_0, ucCh0Color); // B
	hmOutp32(rHDMI_BLUE_SCREEN_1, ucCh1Color); // G
	hmOutp32(rHDMI_BLUE_SCREEN_2, ucCh2Color); // R

	uConReg = Inp32(rHDMI_CON_0);
	uConReg |= (0x1<<5); // Enable Blue screen
	hmOutp32(rHDMI_CON_0, uConReg);
}

// [Set on/off]
void HDMI_AUDIO_SetOnOff(u32 uOnOff)
{
	u32 uConReg;

	Assert(uOnOff == 1 || uOnOff == 0);

	uConReg = Inp32(rHDMI_CON_0);

	if (uOnOff == 1)
	{
		HDMI_AUDIO_SendAviInfoFrame();
		uConReg |= (0x7<<0);
		HDMI_AUDIO_SetTgOn(m_eImgSz, m_eScanMode, 60);		// compare original source!!!
	}
	else 
	{
		uConReg &= ~(0x7<<0);
		HDMI_AUDIO_SetTgOff();
	}
	hmOutp32(rHDMI_CON_0, uConReg);
}

// [Set AVI_InfoFrame]
void HDMI_AUDIO_SendAviInfoFrame(void)
{
	u32 uCheckSum;
	u32 uData1, uData2, uData4, uData5;

	uData1 = m_uImgIfTypeVal;
	uData2 = m_uColorimetryVal | m_uParVal;
	uData4 = m_uVideoCodeVal;
	uData5 = m_uPixelRepVal;

	// check sum = 0x100 - (0x91 + byte1 + byte2 + ... byte11)&0xff
	uCheckSum = 0x100 - (0x91+uData1+uData2+uData4+uData5)&0xff;

	hmOutp32(rHDMI_AVI_CHECK_SUM, uCheckSum);
	hmOutp32(rHDMI_AVI_BYTE1, uData1);
	hmOutp32(rHDMI_AVI_BYTE2, uData2);
	hmOutp32(rHDMI_AVI_BYTE3, 0x00); 		
	hmOutp32(rHDMI_AVI_BYTE4, uData4);
	hmOutp32(rHDMI_AVI_BYTE5, uData5);
	hmOutp32(rHDMI_AVI_BYTE6, 0x00); 
	hmOutp32(rHDMI_AVI_BYTE7, 0x00); 	
	hmOutp32(rHDMI_AVI_BYTE8, 0x00);
	hmOutp32(rHDMI_AVI_BYTE9, 0x00); 
	hmOutp32(rHDMI_AVI_BYTE10, 0x00);		
	hmOutp32(rHDMI_AVI_BYTE11, 0x00);
	hmOutp32(rHDMI_AVI_BYTE12, 0x00);	
	hmOutp32(rHDMI_AVI_BYTE13, 0x00);	
	hmOutp32(rHDMI_AVI_CON, 0x2);
}

// [Set timing generator on]
void HDMI_AUDIO_SetTgOn(IMG_RESOLUTION eImgSize, SCAN_MODE eScanMode, u32 uFreq)
{
	u32 uHFSzReg, uHActStReg, uHActSzReg;
	u32 uVFSzReg, uVSyncReg, uVActStReg, uVActSzReg;

	u32 uCmdVal=0;

	eImgSize = SD480; // peater for debugging error

	Assert(
		( (eImgSize==SD480 || eImgSize==SD576 || eImgSize==HD720)
		&& eScanMode==PROGRESSIVE )  // SD480p, SD576p, HD720p
		||
		( (eImgSize==HD1080) && eScanMode==INTERLACED ) // SD480i, HD1080i
		);

	if (eScanMode==PROGRESSIVE)
	{
		if (uFreq == 60)
		{
			if (eImgSize == SD480)
			{
				uHFSzReg = 858; uHActStReg = 138; uHActSzReg = 720;
				uVFSzReg = 525; uVSyncReg = 1; uVActStReg = 45; uVActSzReg = 480;
			}
			else if (eImgSize == WWSD480)
			{
				// Below data should be modified
				uHFSzReg = 858; uHActStReg = 123; uHActSzReg = 720;
				uVFSzReg = 525; uVSyncReg = 7; uVActStReg = 43; uVActSzReg = 480;
			}
			else if (eImgSize == HD720)
			{
				uHFSzReg = 1650; uHActStReg = 370; uHActSzReg = 1280;
				uVFSzReg = 750; uVSyncReg = 1; uVActStReg = 30; uVActSzReg = 720;
			}
			else
				Assert(0);
		}
		else // (uFreq == 50)
		{
			if (eImgSize == SD576)
			{
				// Below data should be modified
				uHFSzReg = 858; uHActStReg = 123; uHActSzReg = 720;
				uVFSzReg = 525; uVSyncReg = 7; uVActStReg = 43; uVActSzReg = 480;
			}
			else if (eImgSize == WWSD576)
			{
				// Below data should be modified
				uHFSzReg = 858; uHActStReg = 123; uHActSzReg = 720;
				uVFSzReg = 525; uVSyncReg = 7; uVActStReg = 43; uVActSzReg = 480;
			}
			else if (eImgSize == HD720)
			{
				// Below data should be modified
				uHFSzReg = 1650; uHActStReg = 370; uHActSzReg = 1280;
				uVFSzReg = 750; uVSyncReg = 1; uVActStReg = 30; uVActSzReg = 720;
			}
			else
				Assert(0);
		}
		uCmdVal = 0x0<<1;
	}
	else // INTERLACE
	{
		if (uFreq == 60)
		{
			if (eImgSize==SD480)
			{
				; // Do nothing.
			}
			else if (eImgSize==HD1080)
			{
				uHFSzReg = 2200; uHActStReg = 280; uHActSzReg = 1920;
				uVFSzReg = 1125; uVActStReg = 22; uVActSzReg = 540;
				uVSyncReg = 1;
				hmOutp32Multi(rTG_FIELD_CHG_L, 563);
				hmOutp32Multi(rTG_VACT_ST2_L, 584);
				hmOutp32Multi(rTG_VSYNC_TOP_HDMI_L, 1);
				hmOutp32Multi(rTG_VSYNC_BOT_HDMI_L, 563);
				hmOutp32Multi(rTG_FIELD_TOP_HDMI_L, 1);
				hmOutp32Multi(rTG_FIELD_BOT_HDMI_L, 563);
			}
			else
				Assert(0);
		}
		else // (uFreq == 50)
		{
			if (eImgSize==HD1080)
			{
				uHFSzReg = 2640; uHActStReg = 192; uHActSzReg = 1920;
				uVFSzReg = 1125; uVSyncReg = 1; uVActStReg = 21; uVActSzReg = 540;
			}
			else
				Assert(0);
		}
		uCmdVal = 0x1<<1;		
	}

	hmOutp32Multi(rTG_H_FSZ_L, uHFSzReg);
	hmOutp32Multi(rTG_HACT_ST_L, uHActStReg);
	hmOutp32Multi(rTG_HACT_SZ_L, uHActSzReg);
	hmOutp32Multi(rTG_V_FSZ_L, uVFSzReg);
	hmOutp32Multi(rTG_VSYNC_L, uVSyncReg);
	hmOutp32Multi(rTG_VACT_ST_L, uVActStReg);
	hmOutp32Multi(rTG_VACT_SZ_L, uVActSzReg);
	uCmdVal |= 0x1<<0;
	hmOutp32(rTG_CMD, uCmdVal); // Enable TG	
}

// [Set timing generator off]
void HDMI_AUDIO_SetTgOff(void)
{
	u32 uCmdVal;
	
	uCmdVal = Inp32(rTG_CMD);
	uCmdVal &= ~(0x1<<0);
	hmOutp32(rTG_CMD, uCmdVal);
}

u32 HDMI_AUDIO_ReadIntFlag(void)
{
	volatile u32 uRead;

	uRead = Inp32(rHDMI_INTC_FLAG);

	return uRead; 
}

u32 HDMI_AUDIO_ReadStatusFlag(void)
{
	volatile u32 uRead;

	uRead = Inp32(rHDMI_STATUS);

	return uRead; 
}

u32 HDMI_AUDIO_ReadI2sIrqStatus(void)
{
	volatile u32 uRead;

	uRead = Inp32(rHDMI_I2S_IRQ_STATUS);

	return uRead; 
}

u32 HDMI_AUDIO_ReadSpdifinIrqStatus(void)
{
	volatile u32 uRead;

	uRead = Inp32(rHDMI_SPDIF_IRQ_STATUS);

	return uRead; 
}

void HDMI_AUDIO_DisableAllIntCon(void)
{
	hmOutp32(rHDMI_INTC_CON, 0x00);	
}

void HDMI_AUDIO_DisableIntCon(hdmi_audio_int_flag eHdmiAudioIntFlag)
{
	volatile u32 uRead;

	uRead = Inp32(rHDMI_INTC_CON);	
	uRead = (uRead&~(1<<eHdmiAudioIntFlag))|(0<<eHdmiAudioIntFlag);	
	hmOutp32(rHDMI_INTC_CON, uRead);	
}

void HDMI_AUDIO_EnableIntCon(hdmi_audio_int_flag eHdmiAudioIntFlag)
{
	volatile u32 uRead;

	uRead = Inp32(rHDMI_INTC_CON);	
	uRead = (uRead&~((1<<6)|(1<<eHdmiAudioIntFlag)))|((1<<6)|(1<<eHdmiAudioIntFlag));	
	hmOutp32(rHDMI_INTC_CON, uRead);	
}

void HDMI_AUDIO_ClearI2sIrqStatusFlag(hdmi_audio_i2s_irq_status_flag eHdmiAudioI2sIrqStatusFlag)
{
	volatile u32 uRead;

	uRead = Inp32(rHDMI_I2S_IRQ_STATUS);	
	uRead = (uRead&~(1<<eHdmiAudioI2sIrqStatusFlag))|(1<<eHdmiAudioI2sIrqStatusFlag);	
	hmOutp32(rHDMI_I2S_IRQ_STATUS, uRead);	
}

void HDMI_AUDIO_ClearSpdifIrqStatusFlag(hdmi_audio_spdif_irq_status_flag eHdmiAudioSpdifIrqStatusFlag)
{
	volatile u32 uRead;

	uRead = Inp32(rHDMI_SPDIF_IRQ_STATUS);	
	uRead = (uRead&~(1<<eHdmiAudioSpdifIrqStatusFlag))|(1<<eHdmiAudioSpdifIrqStatusFlag);	
	hmOutp32(rHDMI_SPDIF_IRQ_STATUS, uRead);	
}

void HDMI_AUDIO_DisableSpdifIrqMask(hdmi_audio_spdif_irq_status_flag eHdmiAudioSpdifIrqMask)
{
	volatile u32 uRead;

	uRead = Inp32(rHDMI_SPDIF_IRQ_MASK);	
	uRead = (uRead&~(1<<eHdmiAudioSpdifIrqMask))|(0<<eHdmiAudioSpdifIrqMask);	
	hmOutp32(rHDMI_SPDIF_IRQ_MASK, uRead);	
}

void HDMI_AUDIO_SelectClkSourc(hdmi_audio_clk_sel eClkSel)
{
	if(eClkSel == PCLK)
		hmOutp32(rHDMI_AUDIO_CLKSEL, 2);		// always [1]bit => 1 
	else if(eClkSel == SPDIF_CLK)
		hmOutp32(rHDMI_AUDIO_CLKSEL, 3);
	else
		Assert(0);
}

void HDMI_AUDIO_SetAudioClockRecovery(u32 freq)
{
	u32 uN;
	
	// ACR N register
	if(freq == 32000)
		uN = 4096;
	else if(freq == 44100)
		uN = 6272;
	else if(freq == 48000)
		uN = 6144;
	else if(freq == 88000)
		uN = 12544;	
	else if(freq == 96000)
		uN = 12288;	
	else if(freq == 176000)
		uN = 25088;	
	else if(freq == 192000)	
		uN = 24576;	
	else
		UART_Printf("Audio clock recovery is not support at SampleRate[%d Hz].\n", freq);
	
	hmOutp32(rHDMI_ACR_N0, 0xff & uN);
	hmOutp32(rHDMI_ACR_N1, (uN >> 8) & 0xff);
	hmOutp32(rHDMI_ACR_N2, (uN >> 16) & 0xff);

	// ACR packet (Measured)
	// ACR CON register
	hmOutp32(rHDMI_ACR_CON, 0x04);		//ACR_MEASURED_CTS_MODE
}

void HDMI_AUDIO_Spdif_InitAudioPathForInterrupt(CODEC_TYPE eAudioCodec, u32 uSampleRate, u32 uBits, u32 uFrameSizeCode)
{
	u32 uDataType, uBps, wl, uRptCnt, uValueN, uCts, uType, uCh, uSample;
	u32 uBpsType, uSumOfBits, bytes1, bytes2, bytes3, uCheckSum, uBitrate;
	volatile u32 uRead;

	uDataType = 
		(eAudioCodec==PCM) ? 0 :
		(eAudioCodec==AC3) ? 1 : 0xff;
	Assert(uDataType!=0xff);

	hmOutp32(rHDMI_I2S_CLK_CON, 0x1);		// Set I2S Clcok Enable
#if 1	 // rHDMI_SPDIF_CONFIG_1 manual set
	hmOutp32(rHDMI_SPDIF_CONFIG_1, (1<<6)|(1<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0));
#else // rHDMI_SPDIF_CONFIG_1 
	hmOutp32(rHDMI_SPDIF_CONFIG_1, (0<<6)|(0<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0));
#endif 
	hmOutp32(rHDMI_SPDIF_CONFIG_2, 0); // 

	uBps = (eAudioCodec==PCM) ? uBits : 16; // Ac3 16bit only
	wl = 11; // Only 4'b1011 // 24bit
		
	uRptCnt = (eAudioCodec==AC3) ? 1536*2-1 : 0;
#if 1	
	hmOutp32(rHDMI_SPDIF_USER_VALUE_1,  ((uRptCnt&0xf)<<4)|wl);
	hmOutp32(rHDMI_SPDIF_USER_VALUE_2, (uRptCnt>>4)&0xff);
	hmOutp32(rHDMI_SPDIF_USER_VALUE_3, uFrameSizeCode&0xff);
	hmOutp32(rHDMI_SPDIF_USER_VALUE_4, (uFrameSizeCode>>8)&0xff);
#else
	hmOutp32(rHDMI_SPDIF_USER_VALUE_1, 0x0b);
#endif 

	uRead = Inp32(rHDMI_I2S_IN_MUX_CON);
	uRead = (uRead &~((1<<4)|(3<<2)|(1<<1)|(1<<0))) |((1<<4)|(0<<2)|(0<<1)|(1<<0));	// Set audio_sel	, SPDIF CUV 	
	hmOutp32(rHDMI_I2S_IN_MUX_CON, uRead);		

	hmOutp32(rHDMI_I2S_MUX_CH, 0xff);		// enable all data
	hmOutp32(rHDMI_I2S_MUX_CUV, 0x03);	// enable all channel

	/// SPDIF Intterrupt
#if 0
	uRead = Inp32(rHDMI_SPDIF_IRQ_MASK);			// SPDIFIN_IRQ_MASK		
	uRead = (uRead &~((1<<7)|(1<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0))) |((1<<7)|(1<<4)|(1<<3)|(1<<2)|(0<<1)|(1<<0));   
	Outp32(rHDMI_SPDIF_IRQ_MASK, uRead); 	

	uRead = Inp32(rHDMI_INTC_CON);
	uRead = (uRead &~((1<<6)|(1<<1))) |((1<<6)|(1<<1));	 	
	hmOutp32(rHDMI_INTC_CON, 0x01); 	 
#endif 

	hmOutp32(rHDMI_SPDIF_CLK_CTRL, 0x0);
	hmOutp32(rHDMI_SPDIF_CLK_CTRL, 0x1);		// clock enable
#if 0
	hmOutp32(rHDMI_SPDIF_OP_CTRL, 0x01);		// detect signal
	if(eAudioCodec==PCM)
		while((Inp32(rHDMI_SPDIF_IRQ_STATUS) & 0x03) != 0x2);
	else 	// ac3
		while((Inp32(rHDMI_SPDIF_IRQ_STATUS) & 0x73) != 0x12);
	hmOutp32(rHDMI_SPDIF_OP_CTRL, 0x03);		// run
#endif 

#if 1
	uValueN = 	(uSampleRate==32000) ? 4096 :
				(uSampleRate== 44100) ? 6272 :
				(uSampleRate== 88200) ? 12544 :
				(uSampleRate==176400) ? 25088 :
				(uSampleRate== 48000) ? 6144 :
				(uSampleRate== 96000) ? 12288 :
				(uSampleRate==192000) ? 24576 : 0;	
	Assert(uValueN != 0);
	uCts =	 	(uSampleRate==32000) ? 27000 : 
				(uSampleRate== 44100) ? 30000 :
				(uSampleRate== 88200) ? 30000 :
				(uSampleRate==176400) ? 30000 :
				(uSampleRate== 48000) ? 27000 :
				(uSampleRate== 96000) ? 27000 :
				(uSampleRate==192000) ? 27000 : 0;
	Assert(uCts != 0);

	hmOutp32(rHDMI_ASP_CON, 0xf);		
//	hmOutp32(rHDMI_ASP_CON, 1);		// peater for protecting Audio mute error
//	hmOutp32(rHDMI_ASP_SP_FLAT, 0xf); // All Subpackets contain audio samples
	hmOutp32(rHDMI_ASP_SP_FLAT, 0x0); 		// peater for protecting Audio Mute error
	hmOutp32(rHDMI_ASP_CHCFG0, 1<<3|0);
	hmOutp32(rHDMI_ASP_CHCFG1, 1<<3|0);
	hmOutp32(rHDMI_ASP_CHCFG2, 1<<3|0);
	hmOutp32(rHDMI_ASP_CHCFG3, 1<<3|0);

	hmOutp32(rHDMI_ACR_N0, uValueN & 0xff);
	hmOutp32(rHDMI_ACR_N1, (uValueN>>8) & 0xff);
	hmOutp32(rHDMI_ACR_N2, (uValueN>>16) & 0xff);

	hmOutp32(rHDMI_ACR_MCTS0, uCts & 0xff);
	hmOutp32(rHDMI_ACR_MCTS1, (uCts>>8) & 0xff);
	hmOutp32(rHDMI_ACR_MCTS2, (uCts>>16) & 0xff);

	hmOutp32(rHDMI_ACR_CTS0, uCts & 0xff);
	hmOutp32(rHDMI_ACR_CTS1, (uCts>>8) & 0xff);
	hmOutp32(rHDMI_ACR_CTS2, (uCts>>16) & 0xff);

	hmOutp32(rHDMI_ACR_CON, 4); // Messaured

	// AUI Packet set.
	uType = (eAudioCodec==PCM) ? 1 :  // PCM
		(eAudioCodec==AC3) ? 2 : 0; //AC3 or Refer stream header
	uCh = (eAudioCodec==PCM) ? 1 : 0; // 2ch or refer to stream header
	uSample = (uSampleRate==32000) ? 1 :
		(uSampleRate== 44100) ? 2 :
		(uSampleRate== 48000) ? 3 :
		(uSampleRate== 88200) ? 4 :
		(uSampleRate== 96000) ? 5 :
		(uSampleRate==176400) ? 6 :
		(uSampleRate==192000) ? 7 : 0;
	uBpsType = (uBps==16) ? 1 :
		(uBps==20) ? 2 :
		(uBps==24) ? 3 : 0;
	uBpsType = (eAudioCodec==PCM) ? uBpsType : 0;
		
	uSumOfBits = (0x84+0x1+10);//header
	bytes1 = (uType<<4)|uCh;
	bytes2 = (uSample<<2)|uBpsType;
	uBitrate= 512; // AC3 Test peater
	bytes3 = (eAudioCodec==PCM) ? 0 : (uBitrate/8) ;
	uSumOfBits += (bytes1+bytes2+bytes3);
	uCheckSum = 256-uSumOfBits;

	hmOutp32(rHDMI_AUI_CHECK_SUM, uCheckSum); // 
	hmOutp32(rHDMI_AUI_BYTE1, bytes1);
	hmOutp32(rHDMI_AUI_BYTE2, bytes2);
	hmOutp32(rHDMI_AUI_BYTE3, bytes3); // Pcm or stream
	hmOutp32(rHDMI_AUI_BYTE4, 0x00); // 2ch pcm or Stream
	hmOutp32(rHDMI_AUI_BYTE5, 0x00);	 // 2ch pcm or Stream

	hmOutp32(rHDMI_ACP_CON, 2);	 
	hmOutp32(rHDMI_ACP_TYPE, 1);	 

	hmOutp32(rHDMI_GCP_BYTE1, 0x10); 
	hmOutp32(rHDMI_GCP_CON, 0x2); // packet will be transmitted within 384 cycles after active sync.	

//	HDMI_AUDIO_SetAudioClockRecovery(uSampleRate);
#endif 	
}


