/**************************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2009 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : DSIM.c
*  
*	File Description : This file describe Specification of MIPI DSI Master v1.01r11
*
*	Author : Sunghyun,Na
*	Dept. : AP Development Team
*	Created Date : 2009/07/18
*	Version : 0.2 
* 
*	History
*	- 0. Created(Sunghyun,Na 2009/07/18)		
*   
**************************************************************************************/
#include <string.h>
#include "v210_sfr.h"
#include "system.h"
#include "option.h"

#include "library.h"
#include "def.h"
#include "util.h"

#include "sysc.h"			//MIPI_DPHY_CON Register
#include "gpio.h"			//Enable Power for MIPI LCD
#include "intc.h"			//Interrupt Controller

#include "dsim.h"

#define DBG_DSI
#ifdef DBG_DSI
#define DbgDSI	Disp
#else
#define DbgDSI(...)
#endif

#define DBG_DSIPrototocol
#ifdef DBG_DSIPrototocol
#define DbgDSIP	Disp
#else
#define DbgDSIP(...)
#endif

//#define DBG_DSIMInterrupt
#ifdef DBG_DSIMInterrupt
#define DbgDSIIntr	Disp
#else
#define DbgDSIIntr(...)
#endif

static DSIM_Configuration g_oDSIDefault;
static DSIM_Infor* g_oDSIMInfor;

#define Vector_DSI		0
#if (Vector_DSI)
#define DSIM_Outp32(sfr, uTemp)	{Disp("Outp1w(32'h%x,32'h%x);\n", (u32)(&g_oDSIMInfor->m_pBaseAddr->sfr), (uTemp)); Outp32(&g_oDSIMInfor->m_pBaseAddr->sfr, uTemp);}
static u32 Vector_Inp32(u32 sfr)
{
	u32 uTemp = (*(volatile u32 *)(sfr));
	Disp("Inp1w(32'h%x,32'h%x);\n",sfr,uTemp);
	return uTemp;
}
#define DSIM_Inp32(sfr)			Vector_Inp32((u32)(&g_oDSIMInfor->m_pBaseAddr->sfr))
#else
#define DSIM_Outp32(sfr, uTemp)	Outp32(&g_oDSIMInfor->m_pBaseAddr->sfr, uTemp)	
#define DSIM_Outp16(sfr, uTemp)	Outp16(&g_oDSIMInfor->m_pBaseAddr->sfr, uTemp)
#define DSIM_Outp8(sfr, uTemp)	Outp8(&g_oDSIMInfor->m_pBaseAddr->sfr, uTemp)	
#define DSIM_Inp32(sfr)			Inp32(&g_oDSIMInfor->m_pBaseAddr->sfr)
#define DSIM_Inp16(sfr)			Inp16(&g_oDSIMInfor->m_pBaseAddr->sfr)
#define DSIM_Inp8(sfr)			Inp8(&g_oDSIMInfor->m_pBaseAddr->sfr)
#endif

/*------------------------------ APIs of SMDK_S5PV210 for MIPI DSI--------------------*/

//////////
// Function Name : DSIM_EnBackLite
// Function Description : 
//   This function Enable/Disable D-Phy
// Input : 	bEn - True/False
// Output : None
// Version : v0.0
void DSIM_EnBackLite(u8 bEn)
{
	#ifdef S5PC100
	// Back-lite on // GPD 
	GPIO_SetFunctionEach(eGPIO_D, eGPIO_0, 1);
	GPIO_SetDataEach(eGPIO_D, eGPIO_0, bEn);	

	// Slave reset(High Active)  
	GPIO_SetFunctionEach(eGPIO_H0, eGPIO_6, 1);
	GPIO_SetDataEach(eGPIO_H0, eGPIO_6, bEn);
	#endif
	
	#ifdef S5PV210
	// Back-lite on 
	GPIO_SetFunctionEach(eGPIO_D0, eGPIO_3, 1);
	GPIO_SetDataEach(eGPIO_D0, eGPIO_3, bEn);	
	#endif
}

//////////
// Function Name : DSIM_HWResetModule
// Function Description : 
//   This function Enable/Disable D-Phy
// Input : 	bEn - True/False
// Output : None
// Version : v0.0
void DSIM_HWResetModule(void)
{
	//Reset LDI or Slave reset(High Active)  
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_1, 1); 	//GPN[5] -> Output
	GPIO_SetDataEach(eGPIO_H2, eGPIO_1 , 0); 		//GPN[5] -> Low 
	Delay(100);
	GPIO_SetDataEach(eGPIO_H2, eGPIO_1 , 1); 		//GPN[5] -> High
	Delay(100);

}
/*------------------------------ APIs of Basic MIPI DSI--------------------*/
//////////
// Function Name : DSIM_DefaultLinkAndPhy
// Function Description : 
//   This function reads Status of DSIM
// Input : 	DDI_Spec : Configuration of DDI to Initialize DSI 
// Output : DSIMStatus : DSIM Status Structure 
// Version : v0.0
DSIM_Configuration* DSIM_DefaultLinkAndPhy(void)
{
	static u8 ucInit = 1;
	if (ucInit)
	{
		ucInit = 0;
		#if ( DSIM_Version == DSIM_1_02)
		g_oDSIDefault.m_ucEOTDisable = false;					//EOT Packet Genration 		
		#endif	
		//Porch Option
		g_oDSIDefault.m_ucAutoVerticalCnt = false;			//Auto Vertical Cnt Mode
		g_oDSIDefault.m_ucHSE = true;						//Horizontal Sync Evnet Mode
		g_oDSIDefault.m_ucHFP = false;						//Discard Horizontal Front Porch Time
		g_oDSIDefault.m_ucHBP = false;						//Discard Horizontal Back Porch Time
		g_oDSIDefault.m_ucHSA = false;						//Discard Horizontal Sync Area Timing	
		//Data Lane  	
		g_oDSIDefault.m_eNoDataLane = DataLane_2;			//Number of Data Lane using DSI Master

		//Pll PMS & Frequency Band
		g_oDSIDefault.m_bPllBypass = false;					//Pll Bypass
		#if (1 == 0)					// 100Mbps
		g_oDSIDefault.m_ucP = 3;g_oDSIDefault.m_usM = 100;g_oDSIDefault.m_ucS = 3;
		#elif( 1== 0)				// 150Mbps
		g_oDSIDefault.m_ucP = 4;g_oDSIDefault.m_usM = 100;g_oDSIDefault.m_ucS = 2;
		#elif( 1 == 0)				// 200Mbps				
		g_oDSIDefault.m_ucP = 3;g_oDSIDefault.m_usM = 100;g_oDSIDefault.m_ucS = 2;
		#elif( 1== 0)				// 252Mbps
		g_oDSIDefault.m_ucP = 3;g_oDSIDefault.m_usM = 63;g_oDSIDefault.m_ucS = 1;
		#elif( 1== 1)				// 300Mbps
		g_oDSIDefault.m_ucP = 4;g_oDSIDefault.m_usM = 100;g_oDSIDefault.m_ucS = 1;
		#elif( 1== 0)				// 330Mbps
		g_oDSIDefault.m_ucP = 4;g_oDSIDefault.m_usM = 110;g_oDSIDefault.m_ucS = 1;
		#elif( 1== 0)				// 350Mbps
		g_oDSIDefault.m_ucP = 12;g_oDSIDefault.m_usM = 350;g_oDSIDefault.m_ucS = 1;
		#elif( 1== 0)				// 400Mbps
		g_oDSIDefault.m_ucP = 3;g_oDSIDefault.m_usM = 100;g_oDSIDefault.m_ucS = 1;
		#elif( 1== 0)				// 450Mbps
		g_oDSIDefault.m_ucP = 4;g_oDSIDefault.m_usM = 150;g_oDSIDefault.m_ucS = 1;
		#elif( 1== 0 )				// 500Mbps
		g_oDSIDefault.m_ucP = 12;g_oDSIDefault.m_usM = 250;g_oDSIDefault.m_ucS = 0;
		#elif( 1== 0 )				// 600Mbps
		g_oDSIDefault.m_ucP = 4;g_oDSIDefault.m_usM = 100;g_oDSIDefault.m_ucS = 0;
		#elif( 1== 0 )				// 648Mbps
		g_oDSIDefault.m_ucP = 3;g_oDSIDefault.m_usM = 81;g_oDSIDefault.m_ucS = 0;
		#elif( 1== 0 )				// 704Mbps
		g_oDSIDefault.m_ucP = 3;g_oDSIDefault.m_usM = 88;g_oDSIDefault.m_ucS = 0;
		#elif( 1== 0 )				// 720Mbps
		g_oDSIDefault.m_ucP = 3;g_oDSIDefault.m_usM = 90;g_oDSIDefault.m_ucS = 0;
		#elif( 1== 0 )				// 800Mbps
		g_oDSIDefault.m_ucP = 3;g_oDSIDefault.m_usM = 100;g_oDSIDefault.m_ucS = 0;
		#elif( 1== 0 )				// 850Mbps
		g_oDSIDefault.m_ucP = 12;g_oDSIDefault.m_usM = 425;g_oDSIDefault.m_ucS = 0;
		#elif( 1== 0 )				// 900Mbps
		g_oDSIDefault.m_ucP = 4;g_oDSIDefault.m_usM = 150;g_oDSIDefault.m_ucS = 0;	
		#elif( 1== 0 )				// 950Mbps
		g_oDSIDefault.m_ucP = 12;g_oDSIDefault.m_usM = 475;g_oDSIDefault.m_ucS = 0;
		#else						// 1000Mbps
		g_oDSIDefault.m_ucP = 6;g_oDSIDefault.m_usM = 250;g_oDSIDefault.m_ucS = 0;
		#endif	
		
		//Pll Stable Time
		#if (1 == 1)
		g_oDSIDefault.m_uPllStableTime = 0x2f59*20;		// D-PHY PLL Stable Time Spec : Min=200usec ~ Max 400usec
		//g_oDSIDefault.m_uPllStableTime = 200;		// D-PHY PLL Stable Time Spec : Min=200usec ~ Max 400usec
		#endif
		//Escape clk		
		g_oDSIDefault.m_dEscClk = 9 * 1000000;				//Escape clk : 10MHz
		//BTA Counter
		g_oDSIDefault.m_usStopHoldingCnt = 0x0f;			//Stop State Holding Counter After BTA Change Count 0 ~ 0xfff
		g_oDSIDefault.m_ucBTATimeOut = 0xff;				//BTA Time Out 0 ~ 0xff
		g_oDSIDefault.m_usRxTimeOut = 0xffff;				//LP Rx Time Out 0 ~ 0xffff		
		
		g_oDSIDefault.m_eIsrPriority[0] = ErrContentLP0;
		g_oDSIDefault.m_eIsrPriority[1] = ErrContentLP1;
		g_oDSIDefault.m_eIsrPriority[2] = ErrSync0;
		g_oDSIDefault.m_eIsrPriority[3] = ErrSync1;
		g_oDSIDefault.m_eIsrPriority[4] = ErrSync2;
		g_oDSIDefault.m_eIsrPriority[5] = ErrControl0;
		g_oDSIDefault.m_eIsrPriority[6] = ErrControl1;
		g_oDSIDefault.m_eIsrPriority[7] = ErrControl2;
		g_oDSIDefault.m_eIsrPriority[8] = ErrEscLane0;
		g_oDSIDefault.m_eIsrPriority[9] = ErrEscLane1;
		g_oDSIDefault.m_eIsrPriority[10] = ErrEscLane2;
		
		g_oDSIDefault.m_eIsrPriority[11] = ErrRxECC;
		g_oDSIDefault.m_eIsrPriority[12] = ErrRxCRC;

		g_oDSIDefault.m_eIsrPriority[13] = BusTunrOver;		
		g_oDSIDefault.m_eIsrPriority[14] = RxAck;
		g_oDSIDefault.m_eIsrPriority[15] = RxTE;
		g_oDSIDefault.m_eIsrPriority[16] = RxDatDone;

		g_oDSIDefault.m_eIsrPriority[17] = BTAAckTimeOut;
		g_oDSIDefault.m_eIsrPriority[18] = LPDRTimeOut;
		
		g_oDSIDefault.m_eIsrPriority[19] = SwRstRelease;
		g_oDSIDefault.m_eIsrPriority[20] = PllStable;
		g_oDSIDefault.m_eIsrPriority[21] = FrameDone;
		g_oDSIDefault.m_eIsrPriority[22] = SFRFifoEmpty;
		g_oDSIDefault.m_eIsrPriority[23] = ErrControl3;
		g_oDSIDefault.m_eIsrPriority[24] = ErrSync3;
		g_oDSIDefault.m_eIsrPriority[25] = ErrEscLane3;
	}
	
	return &(g_oDSIDefault);
}

//////////
// Function Name : DSIM_GetStatus
// Function Description : 
//   This function reads Status of DSIM
// Input : 	NONE
// Output : DSIMStatus : DSIM Status Structure 
// Version : v0.0
DSIM_Infor* DSIM_Create(DSIM_Configuration* pDSIConfiguration, u8 ucIntrEn)
{
	DSIM_Infor* pInfor = (DSIM_Infor*) malloc(sizeof(DSIM_Infor));		
	pInfor = memset(pInfor, 0, sizeof(DSIM_Infor));
	 
	g_oDSIMInfor = pInfor;
	//SFR
	pInfor->m_pBaseAddr = (DSIM_SFR *)MIPI_DSIM_BASE;
	
	//Configuration
	pInfor->m_oDSIConfiguration = (DSIM_Configuration*) malloc(sizeof(DSIM_Configuration));
	memcpy(pInfor->m_oDSIConfiguration, pDSIConfiguration, sizeof(DSIM_Configuration));	
	
	pInfor->m_eState = DSIM_State_Reset;
	
	// Lane Selection
	if( pDSIConfiguration->m_eNoDataLane == DataLane_1)
		pInfor->m_eDataLane = Lane_Data0;
	else if( pDSIConfiguration->m_eNoDataLane == DataLane_2)
		pInfor->m_eDataLane = Lane_Data0 | Lane_Data1;
	else if( pDSIConfiguration->m_eNoDataLane == DataLane_3)
		pInfor->m_eDataLane = Lane_Data0 | Lane_Data1 | Lane_Data2;
	else if( pDSIConfiguration->m_eNoDataLane == DataLane_4)
		pInfor->m_eDataLane = Lane_Data0 | Lane_Data1 | Lane_Data2 | Lane_Data3;

	pInfor->m_eByteclk = PLLOutDiv8;
	
	SYSC_EnDPHY( (DPHY_HW | DPHY_Master), 1);
	
	DSIM_HWResetModule();
	DSIM_InitHeaderFifo(pInfor);
	
	DSIM_Reset(pInfor, DSIM_SwRst);
	
	if( ucIntrEn )			// Interrupt Enable
	{
		DSIM_EnIntr(pInfor, ucIntrEn);
	}
	else					// Mask all interrupt source
	{
		DSIM_SetIntrMask(AllofIntr, 1);			
	}
	
	return pInfor;
}

//////////
// Function Name : DSIM_GetStatus
// Function Description : 
//   This function reads Status of DSIM
// Input : 	NONE
// Output : NONE
// Version : v0.0
void DSIM_Destory(DSIM_Infor* pInfor)
{
	INTC_Disable(NUM_MIPI_DSI);
	
	#if 0
	//Enter ULPS State
	if( pInfor->m_eState != DSIM_State_Stop )
		DSIM_SetStopState(pInfor);
	DSIM_SetULPS(pInfor);
	#endif	
	DSIM_Reset(pInfor, DSIM_AllRst);
	DSIM_EnBackLite(0);
	
	SYSC_EnDPHY( (DPHY_HW | DPHY_Master), 0);	

	free(pInfor->m_oDSIConfiguration);
	free(pInfor);
	g_oDSIMInfor = NULL;
}
/*---------------------------------- APIs of State Transition ---------------------------------*/
//////////
// Function Name : DSIM_Reset
// Function Description : 
//   This function reset DSIM Link
// Input : 	DSIM_Infor* pInfor : DSIM Information
// Output : None
// Version : v0.0
void DSIM_Reset(DSIM_Infor* pInfor, DSIM_Rst eRst)
{
	if ( eRst == DSIM_AllRst)
	{
		//Register init
		pInfor->m_pBaseAddr->rDSIM_CLKCTRL = 0xffff;
		pInfor->m_pBaseAddr->rDSIM_PLLCTRL = 0;
		pInfor->m_pBaseAddr->rDSIM_PLLTMR = 0;
		pInfor->m_pBaseAddr->rDSIM_PHYACCHR = 0;
		pInfor->m_pBaseAddr->rDSIM_PHYACCHR1 = 0;
		DSIM_SWReset(DSIM_SwRst);
		pInfor->m_eState = DSIM_State_Reset;
	}
	else
	{
		DSIM_SWReset(eRst);
		while(1)
		{	
			if(DSIM_IsSwRstRelease());
				break;
		}
		if ( eRst == DSIM_SwRst )
			pInfor->m_eState = DSIM_State_Init;
	}	
}

//////////
// Function Name : DSIM_InitLink
// Function Description : 
//   This function reads Status of DSIM
// Input : 	DDI_Spec : Configuration of DDI to Initialize DSI 
// Output : DSIMStatus : DSIM Status Structure 
// Version : v0.0
u8  DSIM_InitLink(DSIM_Infor* pInfor)
{
	s32 uTimeOut = 100;
	u16 usEscDiv = 0;
	switch(pInfor->m_eState)
	{
		case DSIM_State_Reset:
			DSIM_SWReset(DSIM_SwRst);
		case DSIM_State_Init:
			DSIM_InitFifoPointer(0x1f);
			//DSI Configuartion
			DSIM_InitConfiguration(pInfor->m_oDSIConfiguration);	
			DSIM_EnLane(Lane_Clock, 1);
			DSIM_EnLane(pInfor->m_eDataLane, 1);
			//Set Clock Configuration
			DSIM_SetClock(pInfor, pInfor->m_eByteclk, true);
			//make Phy Stop state and wait 100us ~ 1ms
			#if 0
			DSIM_ForceDPhyStopState(1);
			Delay(100);				
			DSIM_ForceDPhyStopState(0);	
			#endif
			//Check Clock and Data Lane State is Stop State
			while(!(DSIM_IsLaneState(Lane_Clock) == LaneState_Stop) && !(DSIM_IsLaneState(pInfor->m_eDataLane) == LaneState_Stop))
			{
				uTimeOut--;
				if (uTimeOut == 0)
				{
					Disp("DSI Master State is not Stop State!!!\nPlease Check initialization process\n");
					return FALSE;
					break;
				}
			}
			if (uTimeOut != 0)
				Disp("Initialization of DSI Master is successful\nDSI Master State is Stop State\n");		
			pInfor->m_eState = DSIM_State_Stop;
			//Set Counters for BTA Sequence 
			DSIM_SetStopStateCounter(pInfor->m_oDSIConfiguration->m_usStopHoldingCnt);
			DSIM_SetBTATimeOut(pInfor->m_oDSIConfiguration->m_ucBTATimeOut);
			DSIM_SetLPDRTimeOut(pInfor->m_oDSIConfiguration->m_usRxTimeOut);
			//Defalut LPDT by both CPU and LCD Controller
			DSIM_SetDataMode(Transfer_Both, DSIM_State_Stop);
			return TRUE;
		default:
			Disp("DSI master is already init!\n");
			return false;
	}
}

//////////
// Function Name : DSIM_SetHSEn
// Function Description : 
//   This function enable High Speed Clock Lane for HSDT and change Status of DSIM STM to High Speed Data Transmission State
// Input : 	DSIM_Infor* pInfor 	- Address of Control Structure for DSIM
// Output : DSIMStatus : DSIM Status Structure 
// Version : v0.0
u8 DSIM_SetHSEn(DSIM_Infor* pInfor)
{
	if( pInfor->m_eState == DSIM_State_Stop)
	{
		pInfor->m_eState = DSIM_State_HSClkEn;
		DSIM_SetDataMode(Transfer_Both, DSIM_State_HSClkEn);
		DSIM_EnHSClk(1);		
		return true;
	}
	else
		return false;
}

//////////
// Function Name : DSIM_SetULPS
// Function Description : 
//   This function change Status of DSIM STM to Ultra Low Power State(ULPS)
// Input : 	DSIM_Infor* pInfor 	- Address of Control Structure for DSIM
// Output : bo0l					- Pass/Fail
// Version : v0.0
u8 DSIM_SetULPS(DSIM_Infor* pInfor)
{
	s32 uTimeOut = 50;
	if( pInfor->m_eState == DSIM_State_Stop)
	{
		do
		{
			DSIM_ULPSRequest(ULPS_Entry, 1);	
			Delay(10);
			DSIM_ULPSRequest(ULPS_Entry, 0);			
			Delay(10);			//wait for what?
			if (uTimeOut-- == 0)
			{
				Disp("Time Out Error!!\n");
				return false;
			}
		}while(!(DSIM_IsLaneState(Lane_Clock) == LaneState_ULPS) || !(DSIM_IsLaneState(pInfor->m_eDataLane) == LaneState_ULPS));
		DSIM_SetClock(pInfor, pInfor->m_eByteclk, false);
		 pInfor->m_eState = DSIM_State_ULPS;
		return true;
	}
	else
		return false;
}


//////////
// Function Name : DSIM_SetStopState
// Function Description : 
//   This function change Status of DSIM STM to Stop State
// Input : 	DSIM_Infor* pInfor 	- Address of Control Structure for DSIM
// Output : DSIMStatus : DSIM Status Structure 
// Version : v0.0
u8 DSIM_SetStopState(DSIM_Infor* pInfor)
{
	u32 uTimeout = 100;
	switch(pInfor->m_eState)
	{
		case DSIM_State_HSClkEn:
			pInfor->m_eState = DSIM_State_Stop;
			DSIM_EnHSClk(0);			
			return true;	
		case DSIM_State_ULPS:
			do
			{
				DSIM_SetClock(pInfor, pInfor->m_eByteclk, true);
				DSIM_ULPSRequest(ULPS_Exit, 1);			
				Delay(10);
				DSIM_ULPSRequest(ULPS_Exit, 0);				
				Delay(1000);
				uTimeout--;
				if( uTimeout == 0)
					break;
			}while(!(DSIM_IsLaneState(pInfor->m_eDataLane) == LaneState_Stop));
			if( uTimeout != 0)
			{
				pInfor->m_eState = DSIM_State_Stop;
				return true;
			}
			else 
			{
				return false;
			} 
		default:
			return false;
	}
}

//////////
// Function Name : DSIM_SetDataTransferMode
// Function Description : 
//   This function change Status of DSIM STM to Low Power Data Transmission State
// Input : 	DSIM_Infor* pInfor 	- Address of Control Structure for DSIM
// Output : DSIMStatus : DSIM Status Structure 
// Version : v0.0
u8 DSIM_SetDataTransferMode(DSIM_Infor* pInfor, DSIM_Transfer eDataPath, u8 bHSEn)
{
	if ( bHSEn )
	{
		if ( pInfor->m_eState == DSIM_State_HSClkEn ) 
		{
			DSIM_SetDataMode(eDataPath, DSIM_State_HSClkEn);
			return true;		
		}
		else 
		{	
			Disp("HS Clock Lane is not enabled!!\n");
			return false;
		}
	}
	else 
	{
		if ( pInfor->m_eState == DSIM_State_Init  || pInfor->m_eState == DSIM_State_ULPS ) 
		{
			Disp("DSI Master is not STOP or HSDT State!!\n");
			return false;
		}
		else
		{
			DSIM_SetDataMode(eDataPath, DSIM_State_Stop);
			return true;		
		}
		
	}		
}

//////////
// Function Name : DSIM_SetClock
// Function Description : 
//   This function Command Tranfer /Disable D-Phy
// Input : 	DSIM_Infor* pInfor 	- Address of Control Structure for DSIM
//			ucEn - Short Packet : Data0/Long Packet : Start Address of Transfer Data
// Output : u8 
// Version : v0.0
u8 DSIM_SetClock(DSIM_Infor* pInfor, DSIM_ByteClkSrc eByteClkSel, u8 ucEn)
{
	if( ucEn )
	{
		u32 usEscDiv;
		//Byte Clock Source		
		DSIM_SetByteClockSrc(eByteClkSel);
		if (pInfor->m_eByteclk != eByteClkSel)
			pInfor->m_eByteclk = eByteClkSel;
		if ( eByteClkSel == PLLOutDiv8 )		//DPHY, DSIM Link : D-PHY Clock Out
		{
			pInfor->m_dHSClk = DSIM_ChangePLL(pInfor,pInfor->m_oDSIConfiguration->m_ucP,pInfor->m_oDSIConfiguration->m_usM,pInfor->m_oDSIConfiguration->m_ucS);
			pInfor->m_dByteClk =  pInfor->m_dHSClk / 8;
			DSIM_PLLOn(pInfor, 1);
		}
		else if ( eByteClkSel == ExtclkDiv8 )		//DPHY : D-PHY Clock Out, DSIM Link : External Clock Out
		{
			#if 1
			Disp("Sorry S5PV210 is not Supproted External Clock source Divided by 8 for MIPI DSIM\n");
			Assert('0');
			#else
			pInfor->m_dHSClk = MIPI_FIN;
			pInfor->m_dByteClk =  pInfor->m_dHSClk / 8;
			#endif
		}
		else if ( eByteClkSel == ExtclkBypass )	//DPHY : HSDT Off (LPDT), DSIM Link: External Clock Out
		{
			#if 1
			Disp("Sorry S5PV210 is not Supproted External Clock source for MIPI DSIM\n");
			Assert('0');
			#else
			pInfor->m_dHSClk = MIPI_FIN;
			pInfor->m_dByteClk =  pInfor->m_dHSClk / 8;
			#endif
		}				
		//Escape clock divider
		usEscDiv = (u32)(pInfor->m_dByteClk / (pInfor->m_oDSIConfiguration->m_dEscClk * 2));
		if (pInfor->m_dByteClk/(2 * usEscDiv) > pInfor->m_oDSIConfiguration->m_dEscClk)
			usEscDiv += 1;
		pInfor->m_dEscapeClk = pInfor->m_dByteClk/(2*usEscDiv);
		//enable EscClk on Lane
		DSIM_EnByteClock(TRUE);		
		//Enable Byte clk & Escape clock
		DSIM_SetEscClkPrs(1, usEscDiv);
		//Escaspe Clock on Lane
		DSIM_EnEscClkOnLane(Lane_Clock,1);
		DSIM_EnEscClkOnLane(pInfor->m_eDataLane,1);
		Disp("Byte Clock is %2.3fMHz\n", pInfor->m_dByteClk / 1000000);
		Disp("Escape clock that User's need is %2.3fMHz\n",pInfor->m_oDSIConfiguration->m_dEscClk / 1000000);			
		Disp("Escape clock Divider is 0x%x\n",usEscDiv);
		Disp("Escape clock is %2.3fMHz\n",(pInfor->m_dByteClk/(2*usEscDiv))/1000000);		
	}
	
	else	
	{
		DSIM_EnEscClkOnLane(pInfor->m_eDataLane,0);	
		DSIM_EnEscClkOnLane(Lane_Clock,0);		
		DSIM_SetEscClkPrs(0, 0);
		DSIM_EnByteClock(0);
		if ( eByteClkSel == PLLOutDiv8 )		//DPHY, DSIM Link : D-PHY Clock Out
			DSIM_PLLOn(pInfor, 0);
	}	
}

//////////
// Function Name : DSIM_ChangePLL
// Function Description : 
//   This function Command Tranfer /Disable D-Phy
// Input : 	DSIM_Infor* pInfor 	- Address of Control Structure for DSIM
//			ucEn - Short Packet : Data0/Long Packet : Start Address of Transfer Data
// Output : double _ output frequency of PLL
// Version : v0.0
double DSIM_ChangePLL(DSIM_Infor* pInfor, u8 ucPredivider, u16 ucMaindivider, u8 ucScaler)
{
	//PLL Bypass Selection
	DSIM_EnPllBypass(pInfor->m_oDSIConfiguration->m_bPllBypass);
	if ( pInfor->m_oDSIConfiguration->m_bPllBypass == true )
	{
		#if 1
		Disp("S5PV210 does not support PLL By Pass Mode \n");
		Assert('0');
		#else
		DSIM_EnPll(0);
		return (double)MIPI_FIN;
		#endif
	}
	else
	{
		double dFin_Pll;
		double dFvco;
		double dPllout;
		u8 ucFreqBand;

		dFin_Pll = (double)((double)(MIPI_FIN) / ((double)ucPredivider));
		
		#if (DPHY_Version == DPHY_65n)
		if (dFin_Pll < 2.5*1000*1000 || dFin_Pll > 5*1000*1000 )		
		{
			Disp("Warning!!\n");
			Disp("Fin_PLL Range is 2.5MHz ~ 5MHz\n");
			Disp("Fin_PLL of MIPI DPHY Pll is %3.3fMHz\n",(dFin_Pll/1000000));
		}
		#elif ( DPHY_Version == DPHY_45n )
		if (dFin_Pll < 6*1000*1000 || dFin_Pll > 12*1000*1000 )
		{
			Disp("Warning!!\n");
			Disp("Fin_PLL Range is 6MHz ~ 12MHz\n");
			Disp("Fin_PLL of MIPI DPHY Pll is %3.3fMHz\n",(dFin_Pll/1000000));
			DSIM_EnAFC(0,0);
		}
		else
		{
			if ((u32) dFin_Pll < 7*1000000)			DSIM_EnAFC(1, 0x1);			
			else if ((u32) dFin_Pll < 8*1000000)		DSIM_EnAFC(1, 0x0);
			else if ((u32) dFin_Pll < 9*1000000)		DSIM_EnAFC(1, 0x3);
			else if ((u32) dFin_Pll < 10*1000000)		DSIM_EnAFC(1, 0x2);
			else if ((u32) dFin_Pll < 11*1000000)		DSIM_EnAFC(1, 0x5);
			else										DSIM_EnAFC(1, 0x4);
		}
		#endif
		dFvco = (double)(dFin_Pll * (double)(ucMaindivider));
		if( dFvco < 500000000 || dFvco > 1000000000)
		{
			Disp("Caution!!\n");
			Disp("Fvco Range is 500MHz ~ 1000MHz\n");
			Disp("Fvco of MIPI DPHY Pll is %3.3fMHz\n",(dFvco/1000000));
			Assert('0');
		}
		dPllout = (double)(dFvco / ((double) (1 << ucScaler)));
		if ((u32) dPllout < 100*1000000)			ucFreqBand = 0x0;
		else if ((u32) dPllout < 120*1000000)		ucFreqBand = 0x1;
		else if ((u32) dPllout < 170*1000000)		ucFreqBand = 0x2;
		else if ((u32) dPllout < 220*1000000)		ucFreqBand = 0x3;
		else if ((u32) dPllout < 270*1000000)		ucFreqBand = 0x4;
		else if ((u32) dPllout < 320*1000000)		ucFreqBand = 0x5;
		else if ((u32) dPllout < 390*1000000)		ucFreqBand = 0x6;
		else if ((u32) dPllout < 450*1000000)		ucFreqBand = 0x7;
		else if ((u32) dPllout < 510*1000000)		ucFreqBand = 0x8;
		else if ((u32) dPllout < 560*1000000)		ucFreqBand = 0x9;
		else if ((u32) dPllout < 640*1000000)		ucFreqBand = 0xa;
		else if ((u32) dPllout < 690*1000000)		ucFreqBand = 0xb;
		else if ((u32) dPllout < 770*1000000)		ucFreqBand = 0xc;
		else if ((u32) dPllout < 870*1000000)		ucFreqBand = 0xd;
		else if ((u32) dPllout < 950*1000000)		ucFreqBand = 0xe;
		else 									ucFreqBand = 0xf;
		//Set PMS
		DSIM_PllFreq(ucPredivider, ucMaindivider, ucScaler);
		//Freq Band
		DSIM_PllFreqBand(ucFreqBand);
		//Stable Time
		DSIM_PllStatbleTime(pInfor->m_oDSIConfiguration->m_uPllStableTime);		
		//Enable Pll
		Disp("Fout of MIPI DPHY PLL is %3.2fMHz\n",dPllout / 1000000);
		return dPllout;
	}
}


//////////
// Function Name : DSIM_PLLOn
// Function Description : 
//   This function Command Tranfer /Disable D-Phy
// Input : 	DSIM_Infor* pInfor 	- Address of Control Structure for DSIM
//			ucEn - Short Packet : Data0/Long Packet : Start Address of Transfer Data
// Output : double _ output frequency of PLL
// Version : v0.0
u8 DSIM_PLLOn(DSIM_Infor* pInfor, u8 bEn)
{
	if (bEn)
	{
		s32 sSwTimeOut = 1000;
		DSIM_ClrIntr(PllStable);
		DSIM_EnPll(1);
		while(1)
		{
			sSwTimeOut--;
			if(DSIM_IsPllStable())
				return true;
			if ( sSwTimeOut == 0 )
				return false;
		}		
	}
	else
	{
		DSIM_EnPll(0);
	}
}

//////////
// Function Name : DSIM_SetDisplayMode
// Function Description : 
//   This function change Status of DSIM STM to Low Power Data Transmission State
// Input : 	DSIM_Infor* pInfor 	- Address of Control Structure for DSIM
// Output : DSIMStatus : DSIM Status Structure 
// Version : v0.0
u8 DSIM_SetDisplayMode(DSIM_Infor* pInfor, DDI_Spec* pMainDisplay, DDI_Spec* pSubDisplay)
{
	// Main Display 	
	DSIM_SetMainDispResolution(pMainDisplay->m_01uVertiSize, pMainDisplay->m_00uHoriSize);	// Display Resolution	
	if( pMainDisplay->m_03Interface == (u32) DSIM_Video )			//Video Mode Transfer
	{
		if ( pInfor->m_oDSIConfiguration->m_ucAutoVerticalCnt == false )
		{
			DSIM_SetMainDispVPorch((u16)pMainDisplay->m_uParaMeter[VFrontPorch], (u16)pMainDisplay->m_uParaMeter[VBackPorch], (u16)pMainDisplay->m_uParaMeter[DSI_CmdAllowLen]);	// Vporch
			DSIM_SetMainDispHPorch((u16)pMainDisplay->m_uParaMeter[HFrontPorch], (u16)pMainDisplay->m_uParaMeter[HBackPorch]);		// Hporch
			DSIM_SetMainDispSyncArea((u16)pMainDisplay->m_uParaMeter[VSyncArea], (u16)pMainDisplay->m_uParaMeter[HSyncArea]);		// Sync 
		}
	}
	else		//Command Mode Transfer															
	{
		//Sub Display
		if( pSubDisplay != NULL)
			DSIM_SetSubDispResolution(pSubDisplay->m_01uVertiSize, pSubDisplay->m_00uHoriSize);
	}
	//DSI Configuartion	
	DSIM_DisplayConfiguration(pMainDisplay, pSubDisplay);
}

/*---------------------------------- APIs of DSI Command Transfer and Response------------------------*/
//////////
// Function Name : DSIM_WrData
// Function Description : 
//   This function Command Tranfer by CPU.
// Input : 	DSIM_Infor* pInfor 	- Address of Control Structure for DSIM
//			DSIM_DataID eDataID	- Data ID(DSI Spec)
//			u32 uData0				- short packet : 1st Parameter,  Long Packet : Address of Parameter 
//			u32 uData1				- short packet : 2nd Parameter, Long Packet : size of Parameter
//			u8 bCheckRxAck		- option enable to check Rx Ack.								//add on v0.1 09.06.17
// Output : None
// Version : v0.1
//#define WAIT_SFR_EMPTY
u32 DSIM_WrData(DSIM_Infor* pInfor, DSIM_DataID eDataID, u32 uData0, u32 uData1, u8 bCheckRxAck)
{
	u32 uTimeOut = DSIM_TIMEOUT_MS;
	u8 bReturn = 0;
	if ( pInfor == NULL )
		return false;
	if ( pInfor->m_eState == DSIM_State_ULPS )
		return false;
	{
		double dDelay = 1000000 / pInfor->m_oDSIConfiguration->m_dEscClk;
		u32 uDelay = (u32) (10 * dDelay);
		Delay(uDelay);		
	}

	#ifdef WAIT_SFR_EMPTY	
	while(!(DSIM_GetFifoState() & SFRHeaderEmpty))
	{
		if (( uTimeOut--) >0 ) 
			Delay(1);
		else
		{
			DbgDSIP(("SFR Hearder Fifo is not empty!!\n"));
			return false;
		}
	}
	#endif
	
	switch(eDataID)
	{
		//Short Packet Type///////////////////////////////////////////////////////	
		//Packet Types for Command
		case GenShortWrNoPara:
		case GenShortWr1Para:
		case GenShortWr2Para:		
		case DCSWrNoPara:
		case DCSWr1para:
		case SetMaxRtnPktsize:			
			DSIM_WrTxHeader((u8)eDataID, (u8) uData0,  (u8) uData1);
			if ( bCheckRxAck )		//Check Rx Ack
				return DSIM_ProcessResponse(pInfor, eDataID);
			else
				return TRUE;			
		//General Command
		case CMOff:
		case CMOn:
		case ShutDown:
		case TurnOn:			
			DSIM_WrTxHeader((u8)eDataID, (u8)uData0, (u8)uData1);
			if ( bCheckRxAck )		//Check Rx Ack
				return DSIM_ProcessResponse(pInfor, eDataID);
			else
				return TRUE;
		//Packet Types for Video Data - for test Data transfer through Bus
		case VSyncStart:
		case VSyncEnd:
		case HSyncStart:
		case HSyncEnd:
		case EotPkt:	
			return TRUE;
		//short and responce
		//Packet Types for Command
		case GenRd1Para:
		case GenRd2Para:
		case GenRdNoPara:		
		case DCSRdNoPara:
			DSIM_ClrIntr(AllofIntr);
			DSIM_WrTxHeader((u8)eDataID, (u8) uData0,  (u8) uData1);
			return DSIM_ProcessResponse(pInfor, eDataID);
		//Long Packet Type///////////////////////////////////////////////////////
		//Null Packet 
		case NullPkt:
		case BlankingPkt:	
			return TRUE;			
		//Packet Types for Command
		case GenLongWr:
		case DCSLongWr:
			{
				//Aling Data Word 
				u32 uCnt;
				u32* pWordPtr = (u32*)malloc((uData1+3)/4*4); 		
				memset(pWordPtr,0,(uData1+3)/4*4*4);		
				memcpy(pWordPtr,(u8 *) uData0,uData1);		
				// Put data into Payload FIFO
				for( uCnt = 0;uCnt < uData1/4;uCnt++)
				{
					DSIM_WrTxData(pWordPtr[uCnt]);					
				}
				if(uData1%4)
					DSIM_WrTxData(pWordPtr[uCnt]);					
				free(pWordPtr);			
				// Put Data into Header FIFO
				DSIM_WrTxHeader((u8)eDataID, (u8) (((u16)uData1) & 0xff), (u8) (((u16)uData1 & 0xff00) >> 8));
				//DSIM_WaitState(SFRFifoEmpty);
			};		
			if ( bCheckRxAck )		//Check Rx Ack
				return DSIM_ProcessResponse(pInfor, eDataID);
			else
				return TRUE;
		//Packet Types for Video Data - for test Data transfer through Bus
		case RGB565Packed:
		case RGB666Packed:
		case RGB666Loosly:
		case RGB888Packed:
			if ( bCheckRxAck )		//Check Rx Ack
				return DSIM_ProcessResponse(pInfor, eDataID);
			else
				return TRUE;
		default:
			Disp("The Data ID 0x%2x  is not supported current DSI Spec\n", eDataID);
			return false;		
	}
}

//////////
// Function Name : DSIM_ForceRead
// Function Description : 
//   This function read Slave to force BTA(without Read response command).
// Input : 	DSIM_DataID - Packet ID 
//			uReturnedAddress - Short Packet : Data0/Long Packet : Start Address of Transfer Data
//			uSize - Short Packet : Data0/Long Packet : Size of Transfer Data
// Output : DSIM_IntSrc - RX State
// Version : v0.1
DSIM_IntSrc DSIM_ForceRead(DSIM_Infor* pInfor, u32 uReturnedAddress, u32 uSize)
{
	DSIM_ShortPktHeader eAckPkt;
	s32 sTimeOut = DSIM_TIMEOUT_MS;
	DSIM_ForceBTA();
	while(1)
	{
		if ( DSIM_IsState(RxDatDone) )						// Error Report
		{
			DbgDSIP("Error Report\n");
			if (DSIM_PopRxHeader(pInfor, &eAckPkt) == sizeof(eAckPkt)) 
			{
				if (DSIM_GetRxHeaderType(&eAckPkt) == Ack ) 
				{
					DSIM_ShowErrorReportInPacket(&eAckPkt);
					// multi bit ecc
					if (DSIM_GetErrorReportInPacket(&eAckPkt) & EccMulti) 
					{
						if (DSIM_WaitState(ErrRxECC)) 
						{
							if ( uReturnedAddress != NULL && uSize != NULL )
								*(u8*) uReturnedAddress = false;
						}
					}
					// all flag except single ecc means failed
					else if (DSIM_GetErrorReportInPacket(&eAckPkt) & ~(EccSingle) ) 
					{
						if ( uReturnedAddress != NULL && uSize != NULL )
							*(u8*) uReturnedAddress = false;
					}
						
					else if (DSIM_GetErrorReportInPacket(&eAckPkt) & (EccSingle)) 
					{
						if ( uReturnedAddress != NULL && uSize != NULL )
							*(u8*) uReturnedAddress = true;
					}	
				} 
				else
				{
					DSIM_PushHD(pInfor, &eAckPkt);	// valid data
					if ( uReturnedAddress != NULL && uSize >= 2 )
					{
						memcpy((void *)uReturnedAddress,(void*)&(eAckPkt._inter.data.data0),2);
					}
				}
			}
			return RxDatDone;
		}
		else if ( DSIM_IsState(RxAck) )
		{
			DbgDSIP("Rx Ack!\n");
			return RxAck;
		}
		else if ( DSIM_IsState(RxTE) )			//Tearing Effect On
		{
			DbgDSIP("TE Signal\n");
			return RxTE;
		}
		// case 3: Read Time Out
		else if (DSIM_IsState(LPDRTimeOut))
		{
			DbgDSIP( ("RX timeout \n") );
			return false;
		}
		// case 4: BTA Time Out
		else if (DSIM_IsState(BTAAckTimeOut))
		{
			DbgDSIP( ("BTA Time Out\n") );
			return false;			
		}
		else
		{
			if(!(sTimeOut--))
			{
				Assert('0');
			}
		}
	}
}

//////////
// Function Name : DSIM_ProcessResponse
// Function Description : 
//   This function shows Resonse error flag.
// Input : 	DSIM_DataID : ID of Data type
// Output : u8 -	True : valid sequence and data
//					False : invalid data or invalid sequence
// Version : v0.1	
u32 DSIM_ProcessResponse(DSIM_Infor* pInfor, DSIM_DataID eId) 
{
	DSIM_ShortPktHeader eAckPkt;
	u32 uTimeOut = DSIM_TIMEOUT_MS;	
	u8 bForceBTA = 1;
	
	//Read Packet
	if ( eId == GenRdNoPara ||eId == GenRd1Para ||eId == GenRd2Para || eId == DCSRdNoPara)
	{
		while(1)
		{
			// case 1: read case
			if (DSIM_IsState(RxDatDone))
			{
				if (DSIM_PopRxHeader(pInfor, &eAckPkt) == sizeof(eAckPkt)) 
				{
					if (DSIM_GetRxHeaderType(&eAckPkt) == Ack ) 
					{
						DSIM_ShowErrorReportInPacket(&eAckPkt);
						// multi bit ecc
						if (DSIM_GetErrorReportInPacket(&eAckPkt) & EccMulti) 
						{
							if (DSIM_WaitState(ErrRxECC)) 
								return false;						
						}
						// all flag except single ecc means failed
						else if (DSIM_GetErrorReportInPacket(&eAckPkt) & ~(EccSingle) ) 
							return false;
						else if (DSIM_GetErrorReportInPacket(&eAckPkt) & (EccSingle)) 
							return true;
						
					} 
					else
					{
						DSIM_PushHD(pInfor, &eAckPkt);	// valid data
						return true;
					}
				}
			}
			// case 2: Read Time Out
			if (DSIM_IsState(LPDRTimeOut))
			{
				DbgDSIP( ("RX timeout \n") );
				return false;
			}
			// case 3: BTA Time Out
			if (DSIM_IsState(BTAAckTimeOut))
			{
				DbgDSIP( ("BTA Time Out\n") );				
				return false;			
			}
			if (!((uTimeOut--)>0)) 
				break;
		}
	}
	
	//Non Read Packet
	else 
	{
		DSIM_ForceBTA();
		while(1)
		{
			// case 0: end without error : see spec v1.01.00 r05 1344 line
			if (DSIM_IsState(RxAck))
			{
				DbgDSIP("RX ACK!!\n");
				return true;
			}		
			// case 1: Tearing Effect Signal --> Tearing Effect signal On using Set_Tear_On  Command 
			if (DSIM_IsState(RxTE))			//Tearing Effect On
			{
				Disp("TE Signal\n");
				return true;
			}
			// case 2: non-read but, any data ? --> bad news 
			if (DSIM_IsState(RxDatDone)) 
			{
				if (DSIM_PopRxHeader(pInfor, &eAckPkt) == sizeof(eAckPkt)) 
				{
					if (DSIM_GetRxHeaderType(&eAckPkt)==Ack ) 
					{
						DSIM_ShowErrorReportInPacket(&eAckPkt);
						return true;
					} 
					else
					{
						DbgDSIP(("Can't be here\n"));
						Assert(0);
					}
				}
			}			
			// case 3: Read Time Out
			if (DSIM_IsState(LPDRTimeOut))
			{
				DbgDSIP( ("RX timeout \n") );
				return false;
			}
			// case 4: BTA Time Out
			if (DSIM_IsState(BTAAckTimeOut))
			{
				DbgDSIP( ("BTA Time Out\n") );				
				return false;			
			}
			if (!((uTimeOut--)>0)) 
				break;
		}
	}
	DbgDSIP( ("SW timeout \n") );
	DSIM_ForceDPhyStopState(1);
	Delay(100);
	DSIM_ForceDPhyStopState(0);
	return false;	
}
/*---------------------------------- Header Parser of DSI Rx Data ---------------------------------*/
//////////
// Function Name : DSIM_PopRxPayload
// Function Description : 
//   This function extracts Rx Data from Rx Short Packet from DSIM_PopRxHeader function called.
// Input : 	DSIM_ShortPktHeader* : Address of Rx Short Packet poped from FIFO
//			u8* : Buffer address for Returned Data extrated from Rx Short Packet
//			u32 :
// Output : u32 : size of Returned data
// Version : v0.0		
u32 DSIM_PopRxPayload(DSIM_ShortPktHeader *pHead,u8* pBuf,u32 size)
{
	if(pHead == NULL)
		Assert(0);
	
	switch(DSIM_GetRxHeaderType(pHead)) 
	{
		case Ack:
			return false;
		case GenShort1B:
		case DcsShort1B:
			pBuf[0] = pHead->_inter.data.data0;
			return 1;
		case GenShort2B:
		case DcsShort2B:
			pBuf[0] = pHead->_inter.data.data0;
			pBuf[1] = pHead->_inter.data.data1;
			return 2;
		case GenLong:
		case DcsLong:
			{
				u32 toRead = DSIM_GetRxPayloadDataSize(pHead);
				u32 nread=0;
				u32 temp;				
				if (toRead > size) 
				{
					DbgDSIP("Buffer small! remainder will be thrown away buddy!\n");
				}				
				if (toRead % (sizeof(u32))) 
				{
					toRead = (toRead+sizeof(u32)) & 0xFFFFFFFC;
				}				
				while(toRead != nread || !(DSIM_GetFifoState() & DSIM_RxDataEmpty)) 
				{		
					nread+=sizeof(u32);
					// fifo rx not empty
					temp = DSIM_RdRxData();
					if (nread >= size) 
					{
						*((u32*)pBuf) = temp;
						DbgDSIP("Payload : 0x%.8x\n",temp);
						pBuf+=sizeof(u32);
						Delay(1);		//wait 1ms
					}					
				}
				return nread;
			}
	}
	return 0;
}


//////////
// Function Name : DSIM_GetRxHeaderType
// Function Description : 
//   This function extracts Rx Response Type from Rx Short Packet from DSIM_PopRxHeader function called.
// Input : 	DSIM_ShortPktHeader* : Address of Rx Short Packet poped from FIFO
// Output : DSIM_Response : Type of Rx Data
// Version : v0.0		
DSIM_Response DSIM_GetRxHeaderType(DSIM_ShortPktHeader *pHead) 
{	
	if( pHead == NULL)
		Assert(0);
	
	return (DSIM_Response)(pHead->id & 0x3F);
}

//////////
// Function Name : DSIM_GetRxPayloadDataSize
// Function Description : 
//   This function extracts Recieving word count from Packet Header.
// Input : 	DSIM_ShortPktHeader* : Address of Rx Short Packet poped from FIFO
// Output : u16 : size of Returned data
// Version : v0.0	
u16 DSIM_GetRxPayloadDataSize(DSIM_ShortPktHeader *pHead) 
{
	if( pHead == NULL)
		Assert(0);
	switch(DSIM_GetRxHeaderType(pHead))
	{
		case Ack:
		case GenShort1B:
		case GenShort2B:
		case DcsShort1B:
		case DcsShort2B:
			return 0;
		case GenLong:
		case DcsLong:
			return (((*(u32 *)pHead) & 0xffff00) >> 8);
			//return pHead->_inter.word_count;
		default:
			return 0;
	}	
}

//////////
// Function Name : DSIM_ShowResponseFlagInfo
// Function Description : 
//   This function show Resonse flag.
// Input : 	DSIM_ShortPktHeader* : Address of Rx Short Packet poped from FIFO
// Output : None
// Version : v0.0	
void DSIM_ShowResponseFlagInfo(DSIM_ShortPktHeader *pPacket)
{
	DbgDSIP("Response Flag is ");
	if ( (pPacket->id & 0x3F) == Ack ) 
		DbgDSIP( ("ERROR_REPORT \n") );

	if ( (pPacket->id & 0x3F) == EoTp ) 
		DbgDSIP( ("End of Packet \n") );
	
	if ( (pPacket->id & 0x3F) == GenShort1B ) 
		DbgDSIP( ("GenShort1B \n") );
	
	if ( (pPacket->id & 0x3F) == GenShort2B ) 
		DbgDSIP( ("GenShort2B \n") );

	if ( (pPacket->id & 0x3F) == GenLong ) 
		DbgDSIP( ("GenLong \n") );
	
	if ( (pPacket->id & 0x3F) == DcsShort2B ) 
		DbgDSIP( ("DcsShort2B \n") );
	
	if ( (pPacket->id & 0x3F) == DcsShort1B ) 
		DbgDSIP( ("DcsShort1B \n") );
	
	if ( (pPacket->id & 0x3F) == DcsShort2B ) 
		DbgDSIP( ("DcsShort2B \n") );
	
	if ( (pPacket->id & 0x3F) == DcsLong ) 
		DbgDSIP( ("DCSLong \n") );
	
}

//////////
// Function Name : DSIM_GetErrorReportInPacket
// Function Description : 
//   This function extracts Error Report In Packet count from Packet Header.
// Input : 	DSIM_ShortPktHeader* : Address of Rx Short Packet poped from FIFO
// Output : u16 : size of Returned data
// Version : v0.0	
DSIM_Error DSIM_GetErrorReportInPacket(DSIM_ShortPktHeader *pHead) 
{
	if( pHead == NULL)
		Assert(0);
	switch(DSIM_GetRxHeaderType(pHead))
	{
		case GenLong:
		case DcsLong:
		case GenShort1B:
		case GenShort2B:
		case DcsShort1B:
		case DcsShort2B:
			return 0;		
		case Ack:
			return (DSIM_Error)(((*(u32 *)pHead) & 0xffff00) >> 8);
			//return pHead->_inter.word_count;
		default:
			return 0;
	}	
}


//////////
// Function Name : DSIM_ShowResponseFlagInfo
// Function Description : 
//   This function shows Resonse error flag.
// Input : 	DSIM_ShortPktHeader* : Address of Rx Short Packet poped from FIFO
// Output : None
// Version : v0.0	
void DSIM_ShowErrorReportInPacket(DSIM_ShortPktHeader *pPacket) 
{
	if (DSIM_GetErrorReportInPacket(pPacket) & Sot) 
		DbgDSIP( ("SOT_ERROR\n") );
	
	if (DSIM_GetErrorReportInPacket(pPacket) & SotTSync) 
		DbgDSIP( ("SOT_SYNC_ERROR\n") );
	
	if (DSIM_GetErrorReportInPacket(pPacket) & EotSync) 
		DbgDSIP( ("EOT_SYNC_ERROR\n") );
	
	if (DSIM_GetErrorReportInPacket(pPacket) & ESCEntry) {
		DbgDSIP( ("ESCAPE_MODE_ENTRY_COMMAND_ERROR\n") );
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & LPTSync) {
		DbgDSIP( ("LOWPOWER_TRANSIT_SYNC_ERROR\n") );
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & HSTimeout) {
		DbgDSIP( ("HS_RECEIVE_TIMEOUT_ERROR\n") );
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & FalseControl) {
		DbgDSIP( ("FALSE_CONTROL_ERROR\n") );
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & EccSingle) {
		DbgDSIP( ("ECC_ERROR_SINGLE\n") );		
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & EccMulti) {
		DbgDSIP( ("ECC_ERROR_MULTI\n") );
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & CheckSum) {
		DbgDSIP( ("CHECKSUM_ERROR\n") );
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & DataType) {
		DbgDSIP( ("DATATYPE_NOT_RECOGNIZED\n") );
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & VC) {
		DbgDSIP( ("VCHANNEL_INVALID\n") );
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & Length) {
		DbgDSIP( ("VCHANNEL_INVALID\n") );
	}
	if (DSIM_GetErrorReportInPacket(pPacket) & Protocol) {
		DbgDSIP( ("PROTOCOL_VIOLATION\n") );
	}

}
/*---------------------------------- Fifo Apis of DSI Rx Data ---------------------------------*/
// Function Name : DSIM_PopRxHeader
// Function Description : 
//   This function pop Rx Header from S/W contructed FIFO  in memory or DSI H/W Rx FIFO
// Input : 	DSIM_ShortPktHeader* : memory Address of Data poped from FIFO
// Output : u32 : size of Header
// Version : v0.0
u32 DSIM_PopRxHeader(DSIM_Infor* pInfor, DSIM_ShortPktHeader *pHead) 
{
	if ( !DSIM_IsHeaderFifoEmpty(pInfor)) 
	{
		DSIM_PopHD(pInfor, pHead);
		return sizeof(u32);
	}	
	
	if ( !(DSIM_GetFifoState() & DSIM_RxDataEmpty) ) 
	{
		// fifo rx not empty
		*((u32*)pHead) = DSIM_RdRxData();
		DbgDSIP("Header : 0x%.8x\n",*((u32*)(pHead)));
		DSIM_ShowResponseFlagInfo(pHead);		
		return sizeof(u32);
	}
	else 		// Fifo is Empty
		return 0;
}

//////////
// Function Name : DSIM_PushHD
// Function Description : 
//   This function push Rx Short Packet Data into fifo.
// Input : 	DSIM_Infor* : informaition data of DSI 
//			DSIM_ShortPktHeader* : Data to store fifo
// Output : u8 -	True : success 
//					False : fail
// Version : v0.0	
u8 DSIM_PushHD(DSIM_Infor* pInfor, DSIM_ShortPktHeader *pHead)
{
	s32 maxIdx = -1;
	s32 emptyIdx = 0;
	u32 uCnt0;	
	for (uCnt0 = 0;uCnt0 < DSIM_HEADER_FIFO_SZ;uCnt0++) 
	{
		if (pInfor->m_HeaderFifoIndex[uCnt0] >= 0) 
		{
			if (maxIdx < pInfor->m_HeaderFifoIndex[uCnt0]) 
				maxIdx = pInfor->m_HeaderFifoIndex[uCnt0];
		}
		else if (emptyIdx < 0) 
		{
			emptyIdx = uCnt0;
		}
	}	
	if (emptyIdx < 0) 
	{
		DbgDSIP(("Header fifo full\n"));
		return false;
	}
	pInfor->m_HeaderFifoIndex[emptyIdx] = maxIdx+1;	
	pInfor->m_oHeaderFifo[emptyIdx] = *pHead;
	return true;
}

//////////
// Function Name : DSIM_PushHD
// Function Description : 
//   This function push Rx Short Packet Data into fifo.
// Input : 	DSIM_Infor* : informaition data of DSI 
//			DSIM_ShortPktHeader* : Data to store fifo
// Output : u8 -	True : success 
//					False : fail
// Version : v0.0	
u8 DSIM_PopHD(DSIM_Infor* pInfor, DSIM_ShortPktHeader *pHead)
{
	s32 minIdx = -1;
	u32 uCnt0;
	for (uCnt0 = 0;uCnt0 < DSIM_HEADER_FIFO_SZ; uCnt0++)	//check Current Header fifo index
	{				
		if (pInfor->m_HeaderFifoIndex[uCnt0] >= 0)
		{
			if (minIdx < 0 || minIdx > pInfor->m_HeaderFifoIndex[uCnt0]) 
				minIdx = uCnt0;								// Get max index			
		}
	}
	if (minIdx<0) 				// if upper process is failed, return false
	{									
		DbgDSIP(("Empty Header FIFO\n"));
		return false;
	}	
	*pHead = pInfor->m_oHeaderFifo[minIdx];	
	pInfor->m_HeaderFifoIndex[minIdx] = -1;					// set index -1 : fifo directed that index is empty
	return true;
}

//////////
// Function Name : DSIM_PushHD
// Function Description : 
//   This function push Rx Short Packet Data into fifo.
// Input : 	DSIM_Infor* : informaition data of DSI 
//			DSIM_ShortPktHeader* : Data to store fifo
// Output : u8 -	True : success 
//					False : fail
// Version : v0.0	
u8 DSIM_InitHeaderFifo(DSIM_Infor* pInfor)
{
	u32 uCnt0;
	for (uCnt0 = 0;uCnt0 <DSIM_HEADER_FIFO_SZ;uCnt0++) 
	{
		pInfor->m_HeaderFifoIndex[uCnt0] = -1;
	}
	return true;
}

//////////
// Function Name : DSIM_PushHD
// Function Description : 
//   This function push Rx Short Packet Data into fifo.
// Input : 	DSIM_Infor* : informaition data of DSI 
//			DSIM_ShortPktHeader* : Data to store fifo
// Output : u8 -	True : success 
//					False : fail
// Version : v0.0	
u8 DSIM_IsHeaderFifoEmpty(DSIM_Infor* pInfor)
{
	u32 uCnt0;
	for (uCnt0 = 0;uCnt0 <DSIM_HEADER_FIFO_SZ;uCnt0++) 
	{
		if (pInfor->m_HeaderFifoIndex[uCnt0] >= 0) 
			return false;
	}
	return true;
}
/*---------------------------------- APIs of DSIM Interrupt --------------------------------*/
//////////
// Function Name : Isr_DSIM
// Function Description : 
//   This function is ISR of DSI Master.
// Input : 	None
// Output : None
// Version : v0.0	
void __irq Isr_DSIM(void)
{
	DSIM_Infor* pInfor = g_oDSIMInfor;
	u32 uCnt0;	
	u32 uState = (u32) DSIM_GetIntrState();		//Mask State
	u32 uMask = (u32) DSIM_GetIntrMask();
	INTC_Disable(NUM_MIPI_DSI);			
	// For Interrupt 	
	for ( uCnt0 = 0;uCnt0 < DSIM_NoOfInterrupt;uCnt0++)
	{
		if (  uState  & ~uMask & (u32) pInfor->m_oDSIConfiguration->m_eIsrPriority[uCnt0])
		{
			DSIM_ClrIntr(pInfor->m_oDSIConfiguration->m_eIsrPriority[uCnt0]);
			switch(pInfor->m_oDSIConfiguration->m_eIsrPriority[uCnt0])
			{
				case PllStable:
					DbgDSIIntr(("\n$$$\tPll Statble Interrupt\n"));					
					pInfor->m_bSetState[31] = 1;
					if ( pInfor->m_fnCallback[31] != NULL )	pInfor->m_fnCallback[31](pInfor->m_uCallbackVar[31]);break;
				case SwRstRelease:	
					DbgDSIIntr(("\n$$$\tSoftware Rest Release\n"));
					pInfor->m_bSetState[30] = 1;
					if ( pInfor->m_fnCallback[30] != NULL )pInfor->m_fnCallback[30](pInfor->m_uCallbackVar[30]);break;
				case SFRFifoEmpty:	
					DbgDSIIntr(("\n$$$\tSFR Fifo Empty\n"));
					pInfor->m_bSetState[29] = 1;
					if ( pInfor->m_fnCallback[29] != NULL )pInfor->m_fnCallback[29](pInfor->m_uCallbackVar[29]);break;
				case BusTunrOver:
					DbgDSIIntr(("\n$$$\tBus Turn over\n"));
					pInfor->m_bSetState[25] = 1;
					if ( pInfor->m_fnCallback[25] != NULL )pInfor->m_fnCallback[25](pInfor->m_uCallbackVar[25]);break;
				case FrameDone:
					DbgDSIIntr(("\n$$$\tFrame Done\n"));
					pInfor->m_bSetState[24] = 1;
					if ( pInfor->m_fnCallback[24] != NULL )pInfor->m_fnCallback[24](pInfor->m_uCallbackVar[24]);break;
				case LPDRTimeOut:
					DbgDSIIntr(("\n###\tRx Time Out\n"));
					pInfor->m_bSetState[21] = 1;
					if ( pInfor->m_fnCallback[21] != NULL )pInfor->m_fnCallback[21](pInfor->m_uCallbackVar[21]);break;
				case BTAAckTimeOut:
					DbgDSIIntr(("\n###\tBTA Time Out\n"));
					pInfor->m_bSetState[20] = 1;
					if ( pInfor->m_fnCallback[20] != NULL )pInfor->m_fnCallback[20](pInfor->m_uCallbackVar[20]);break;
				case RxDatDone:
					DbgDSIIntr(("\n@@@\tRX Data Done\n"));					
					pInfor->m_bSetState[18] = 1;
					if ( pInfor->m_fnCallback[18] != NULL )pInfor->m_fnCallback[18](pInfor->m_uCallbackVar[18]);break;
				case RxTE:
					DbgDSIIntr(("\n@@@\tRX Tearing Effect\n"));
					pInfor->m_bSetState[17] = 1;
					if ( pInfor->m_fnCallback[17] != NULL )pInfor->m_fnCallback[17](pInfor->m_uCallbackVar[17]);break;
				case RxAck:
					DbgDSIIntr(("\n@@@\tRX Ack\n"));
					pInfor->m_bSetState[16] = 1;
					if ( pInfor->m_fnCallback[16] != NULL )pInfor->m_fnCallback[16](pInfor->m_uCallbackVar[16]);break;
				case ErrRxECC:
					DbgDSIIntr(("\n!!!\tRx Ecc Error\n"));
					pInfor->m_bSetState[15] = 1;
					if ( pInfor->m_fnCallback[15] != NULL )pInfor->m_fnCallback[15](pInfor->m_uCallbackVar[15]);break;
				case ErrRxCRC:
					DbgDSIIntr(("\n!!!\tRx CRC Error\n"));
					pInfor->m_bSetState[14] = 1;
					if ( pInfor->m_fnCallback[14] != NULL )pInfor->m_fnCallback[14](pInfor->m_uCallbackVar[14]);break;
				case ErrEscLane3:
					DbgDSIIntr(("\n!!!\tEsc Entry Lane 3 Error\n"));
					pInfor->m_bSetState[13] = 1;
					if ( pInfor->m_fnCallback[13] != NULL )pInfor->m_fnCallback[13](pInfor->m_uCallbackVar[13]);break;
				case ErrEscLane2:
					DbgDSIIntr(("\n!!!\tEsc Entry Lane 2 Error\n"));
					pInfor->m_bSetState[12] = 1;
					if ( pInfor->m_fnCallback[12] != NULL )pInfor->m_fnCallback[12](pInfor->m_uCallbackVar[12]);break;
				case ErrEscLane1:
					DbgDSIIntr(("\n!!!\tEsc Entry Lane 1 Error\n"));
					pInfor->m_bSetState[11] = 1;
					if ( pInfor->m_fnCallback[11] != NULL )pInfor->m_fnCallback[11](pInfor->m_uCallbackVar[11]);break;
				case ErrEscLane0:
					DbgDSIIntr(("\n!!!\tEsc Entry Lane 2 Error\n"));
					pInfor->m_bSetState[10] = 1;
					if ( pInfor->m_fnCallback[10] != NULL )pInfor->m_fnCallback[10](pInfor->m_uCallbackVar[10]);break;
				case ErrSync3:
					DbgDSIIntr(("\n!!!\tLPDT Sync Lane 3 Error\n"));
					pInfor->m_bSetState[9] = 1;
					if ( pInfor->m_fnCallback[9] != NULL ) pInfor->m_fnCallback[9](pInfor->m_uCallbackVar[9]);break;
				case ErrSync2:
					DbgDSIIntr(("\n!!!\tLPDT Sync Lane 2 Error\n"));
					pInfor->m_bSetState[8] = 1;
					if ( pInfor->m_fnCallback[8] != NULL )pInfor->m_fnCallback[8](pInfor->m_uCallbackVar[8]);break;
				case ErrSync1:
					DbgDSIIntr(("\n!!!\tLPDT Sync Lane 1 Error\n"));
					pInfor->m_bSetState[7] = 1;
					if ( pInfor->m_fnCallback[7] != NULL )pInfor->m_fnCallback[7](pInfor->m_uCallbackVar[7]);break;
				case ErrSync0:
					DbgDSIIntr(("\n!!!\tLPDT Sync Lane 0 Error\n"));
					pInfor->m_bSetState[6] = 1;
					if ( pInfor->m_fnCallback[6] != NULL )pInfor->m_fnCallback[6](pInfor->m_uCallbackVar[6]);break;
				case ErrControl3:
					DbgDSIIntr(("\n!!!\tFalse Control Error 3\n"));
					pInfor->m_bSetState[5] = 1;
					if ( pInfor->m_fnCallback[5] != NULL )pInfor->m_fnCallback[5](pInfor->m_uCallbackVar[5]);break;
				case ErrControl2:
					DbgDSIIntr(("\n!!!\tFalse Control Error 2\n"));
					pInfor->m_bSetState[4] = 1;
					if ( pInfor->m_fnCallback[4] != NULL )pInfor->m_fnCallback[4](pInfor->m_uCallbackVar[4]);break;
				case ErrControl1:
					DbgDSIIntr(("\n!!!\tFalse Control Error 1\n"));
					pInfor->m_bSetState[3] = 1;
					if ( pInfor->m_fnCallback[3] != NULL )pInfor->m_fnCallback[3](pInfor->m_uCallbackVar[3]);break;
				case ErrControl0:
					DbgDSIIntr(("\n!!!\tFalse Control Error 0\n"));
					pInfor->m_bSetState[2] = 1;
					if ( pInfor->m_fnCallback[2] != NULL )pInfor->m_fnCallback[2](pInfor->m_uCallbackVar[2]);break;
				case ErrContentLP0:
					DbgDSIIntr(("\n!!!\tContention LP0\n"));
					pInfor->m_bSetState[1] = 1;
					if ( pInfor->m_fnCallback[1] != NULL )pInfor->m_fnCallback[1](pInfor->m_uCallbackVar[1]);break;
				case ErrContentLP1:
					DbgDSIIntr(("\n!!!\tContention LP1\n"));
					pInfor->m_bSetState[0] = 1;
					if ( pInfor->m_fnCallback[0] != NULL )pInfor->m_fnCallback[0](pInfor->m_uCallbackVar[0]);break;
			}		
		}
	}	
	INTC_ClearVectAddr();
	INTC_Enable(NUM_MIPI_DSI);	
 }

//////////
// Function Name : DSIM_EnIntr
// Function Description : 
//   This function enables interrupt controller to interrupt of DSIM.
// Input : 	
// Output : 
// Version : v0.0	
void DSIM_EnIntr(DSIM_Infor* pInfor, u8 ucEn)
{
	if (ucEn)
	{
		INTC_SetVectAddr(NUM_MIPI_DSI,Isr_DSIM);
		DSIM_ClrIntr(AllofIntr);
		DSIM_SetIntrMask(AllofIntr, 1);			//Mask all interrupt source
		INTC_Enable(NUM_MIPI_DSI);
	}
	else
	{
		INTC_Disable(NUM_MIPI_DSI);
	}
}
	
//////////
// Function Name : DSIM_SetIsrCallBack
// Function Description : 
//   This function registers Call back Function of Interrupt source
// Input : 	None
// Output : None
// Version : v0.0	
void DSIM_SetIsrCallBack(DSIM_Infor* pInfor, DSIM_IntSrc eIntSrc, void (*fnCallBack)(u32), u32 CallbackVar)
{
	s32 uCnt0;
	s32 uInterruptEnable = 0;
	for ( uCnt0 = 0;uCnt0 < 32;uCnt0++)
	{
		uInterruptEnable = 1 << uCnt0;
		if(uInterruptEnable & (s32) eIntSrc)
		{
			DSIM_ClrIntr(uInterruptEnable);
			pInfor->m_fnCallback[uCnt0] = fnCallBack;
			pInfor->m_uCallbackVar[uCnt0] = CallbackVar;		
			if ( fnCallBack )
			{
				DSIM_SetIntrMask(uInterruptEnable, 0);		// Unmasked 	
			}
			else
			{
				DSIM_SetIntrMask(uInterruptEnable, 1);		// Masked 	
			}
		
		}
	}
}
/*---------------------------------- APIs of DSIM State---------------------------------*/
//////////
// Function Name : DSIM_IsState
// Function Description : 
//   This function query state.
// Input : 	DSIM_IntSrc - Status Bit 
// Output : None
// Version : v0.0
u8 DSIM_IsState(DSIM_IntSrc eIntSrc)
{
	DSIM_Infor* pInfor = g_oDSIMInfor;
	u32 uCnt0;
	u32 uState = (u32) DSIM_GetIntrMask();		//Mask State
	if ( DSIM_GetIntrState() & eIntSrc)
	{
		DSIM_ClrIntr(eIntSrc);							//clear
		return true;
	}
	
	for ( uCnt0 = 0;uCnt0 < 32;uCnt0++)			//when state bit is cleared in the isr, some sequence is wrong.
	{
		if (  eIntSrc & (1 << uCnt0))
		{
			if (pInfor->m_bSetState[uCnt0])
			{
				pInfor->m_bSetState[uCnt0] = false;		//clear
				return true;
			}
			else
				return false;
		}
	}
	
}

//////////
// Function Name : DSIM_WaitState
// Function Description : 
//   This function wait Interrupt.
// Input : 	DSIM_IntSrc - Clear Interrupt bit
// Output : None
// Version : v0.0
u8 DSIM_WaitState(DSIM_IntSrc  eIntSrc)
{
	DSIM_Infor* pInfor = g_oDSIMInfor;
	u32 uTimeOut = DSIM_TIMEOUT_MS;
	while (1) 
	{
		if ( DSIM_IsState(eIntSrc) ) 
		{
			return true;
		}		
		if ((uTimeOut--)==0) 
		{
			Disp( ("Wait State -Timeout\n") );
			return false;
		}
		else 
			Delay(1);		
	}	
	/* timeout */
	return false;
}
/*---------------------------------- APIs of DSIM_STATUS ---------------------------------*/
//////////
// Function Name : DSIM_IsPllStable
// Function Description : 
//   This function return Pll Stable status
// Input : 	NONE
// Output : u8 - D-Phy Pll Stable
// Version : v0.0
u8 DSIM_IsPllStable(void)
{
	return (u8) (DSIM_Inp32(rDSIM_STATUS) & (1 << 31) >> 31);
}

//////////
// Function Name : DSIM_IsSwResetRelease
// Function Description : 
//   This function return Pll Stable status
// Input : 	NONE
// Output : u8 - SW Reset Release
// Version : v0.0
u8 DSIM_IsSwRstRelease(void)
{
	return (u8) (DSIM_Inp32(rDSIM_STATUS) & (1 << 29) >> 29);
}

// Function Name : DSIM_IsDirection
// Function Description : 
//   This function return Data Direction 
// Input : 	NONE
// Output : u8 - SW Reset Release
// Version : v0.0
u8 DSIM_IsDirection(void)
{
	return (u8) (DSIM_Inp32(rDSIM_STATUS) & (1 << 16) >> 16);
}

//////////
// Function Name : DSIM_IsLaneState
// Function Description : 
//   This function return Pll Stable status
// Input : 	DSIM_LANE - one of DSI Lane
// Output : DSIM_LaneState - Lane State
// Version : v0.0
DSIM_LaneState DSIM_IsLaneState(DSIM_LANE eLane)
{
	u32 uTemp = DSIM_Inp32(rDSIM_STATUS);
	if(eLane == Lane_Clock)		//Clock Lane
	{
		if (uTemp & (1 << 9))
			return LaneState_ULPS;
		else if (uTemp & (1 << 8))
			return LaneState_Stop;
		else if (uTemp & (1 << 10))
			return LaneState_HsReady;
		else 
		{
			Disp("Clcock Lane State is Unkown\n");
			return (DSIM_LaneState) (-1);
		}
	}	
	else if ( eLane & Lane_All )
	{
		
		if ((uTemp & 0x7ff) ^ (((eLane & 0xf) << 4) | (1 << 9)))
			return LaneState_ULPS;
		else if ((uTemp & 0x7ff) ^ (((eLane & 0xf) << 0) | (1 << 8)))
			return LaneState_Stop;		
		else 
		{
			Disp("Lane State is Unkown\n");
			return (DSIM_LaneState) (-1);
		}
	}
	else	 if ( eLane & Lane_DataAll )//Data Lane
	{
		if (uTemp & (eLane << 4))
			return LaneState_ULPS;
		else if (uTemp & (eLane << 0))
			return LaneState_Stop;		
		else 
		{
			Disp("Data Lane State is Unkown\n");
			return (DSIM_LaneState) (-1);
		}
	}	
	
}

/*---------------------------------- APIs of DSIM_SWRST ---------------------------------*/
//////////
// Function Name : DSIM_SWReset
// Function Description : 
//   This function reset DSIM SFR Resgister.
// Input : 	eRst - kind of reset : Function/All SFR
// Output : None
// Version : v0.0
void DSIM_SWReset(DSIM_Rst eRst)
{
	DSIM_Outp32(rDSIM_SWRST, (u32) eRst);
}

/*---------------------------------- APIs of DSIM_CLKCTRL ---------------------------------*/
//////////
// Function Name : DSIM_EnHSClk
// Function Description : 
//   This function enable High Speed Clock on the Clock Lane
// Input : 	usEn - Enable
// Output : None
// Version : v0.0
void DSIM_EnHSClk(u8 bEn)
{
	u32 uTemp = DSIM_Inp32(rDSIM_CLKCTRL) & ~(1 << 31);
	uTemp |= bEn << 31;
	DSIM_Outp32(rDSIM_CLKCTRL, uTemp);	
}

//////////
// Function Name : DSIM_SetEscapeClkPrs
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetEscClkPrs(u8 bEn, u16 usPrsVal)
{
	u32 uTemp = DSIM_Inp32(rDSIM_CLKCTRL) & ~(1 << 28) & ~(0xffff);
	uTemp |= bEn << 28;
	if( bEn ) 
		uTemp |= usPrsVal;	
	DSIM_Outp32(rDSIM_CLKCTRL, uTemp);
}

//////////
// Function Name : DSIM_EnEscapeClkPrs
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_EnEscClkPrs(u8 bEn)
{
	u32 uTemp = DSIM_Inp32(rDSIM_CLKCTRL) & ~(1 << 28);
	uTemp |= bEn << 28;	
	DSIM_Outp32(rDSIM_CLKCTRL, uTemp);
}

//////////
// Function Name : DSIM_EnEscapeClkPrs
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetEscClkPrsVal(u16 usPrsVal)
{
	u32 uTemp = DSIM_Inp32(rDSIM_CLKCTRL) & ~(0xffff);
	uTemp |= usPrsVal;	
	DSIM_Outp32(rDSIM_CLKCTRL, uTemp);
}

//////////
// Function Name : DSIM_EnPllBypass
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_EnPllBypass(u8 bEn)
{
	u32 uTemp = DSIM_Inp32(rDSIM_CLKCTRL) & ~(1 << 27);
	uTemp |= bEn << 27;
	DSIM_Outp32(rDSIM_CLKCTRL, uTemp);
}

//////////
// Function Name : DSIM_SelByteClock
// Function Description : 
//   This function select Source of  Byte Clock.
// Input : 	eSrc - Source of Byte Clock
// Output : None
// Version : v0.0
void DSIM_SetByteClockSrc(DSIM_ByteClkSrc eSrc)
{
	u32 uTemp = DSIM_Inp32(rDSIM_CLKCTRL) & ~(0x3 << 25);
	uTemp |= (u32) eSrc << 25;
	DSIM_Outp32(rDSIM_CLKCTRL, uTemp);
}

//////////
// Function Name : DSIM_EnByteClock
// Function Description : 
//   This function enable Byte Clock.
// Input : 	bEn - enable Byte Clock
// Output : None
// Version : v0.0
void DSIM_EnByteClock(u8 bEn)
{
	u32 uTemp = DSIM_Inp32(rDSIM_CLKCTRL) & ~(1 << 24);
	uTemp |= bEn << 24;
	DSIM_Outp32(rDSIM_CLKCTRL, uTemp);
}


//////////
// Function Name : DSIM_EnByteClock
// Function Description : 
//   This function enable Byte Clock.
// Input : 	bEn - enable Byte Clock
// Output : None
// Version : v0.0
void DSIM_EnEscClkOnLane(DSIM_LANE eLaneSel, u8 ucEn)
{
	u32 uTemp = DSIM_Inp32(rDSIM_CLKCTRL);
	if (ucEn){
		if ( eLaneSel == Lane_Clock ) uTemp |= 1 << 19;
		if ( eLaneSel & Lane_Data0 ) uTemp |= 1 << 20;
		if ( eLaneSel & Lane_Data1 ) uTemp |= 1 << 21;
		if ( eLaneSel & Lane_Data2 ) uTemp |= 1 << 22;
		if ( eLaneSel & Lane_Data2 ) uTemp |= 1 << 23;	
	}
	else{
		if ( eLaneSel == Lane_Clock ) uTemp &= ~(1 << 19);
		if ( eLaneSel & Lane_Data0 ) uTemp &= ~(1 << 20);
		if ( eLaneSel & Lane_Data1 ) uTemp &= ~(1 << 21);
		if ( eLaneSel & Lane_Data2 ) uTemp &= ~(1 << 22);
		if ( eLaneSel & Lane_Data2 ) uTemp &= ~(1 << 23);	
	}
	DSIM_Outp32(rDSIM_CLKCTRL, uTemp);
}


/*---------------------------------- APIs of DSIM_TIMEOUT---------------------------------*/
//////////
// Function Name : DSIM_SetBTATimeOut
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetBTATimeOut(u8 ucTimeOut)
{
	u32 uTemp = DSIM_Inp32(rDSIM_TIMEOUT) & ~(0xff << 16);
	uTemp |= (ucTimeOut << 16);
	DSIM_Outp32(rDSIM_TIMEOUT, uTemp);
}

//////////
// Function Name : DSIM_SetLPDRTimeOut
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetLPDRTimeOut(u16 usTimeOut)
{
	u32 uTemp = DSIM_Inp32(rDSIM_TIMEOUT) & ~(0xffff << 0);
	uTemp |= (usTimeOut << 0);
	DSIM_Outp32(rDSIM_TIMEOUT, uTemp);
}
/*---------------------------------- APIs of DSIM_CONFIG---------------------------------*/
//////////
// Function Name : DSIM_InitConfiguration
// Function Description : 
//   This function init DSIM Configration except Display.
// Input : 	oConfiguration - Configuration Structure
// Output : None
// Version : v0.0
void DSIM_InitConfiguration(DSIM_Configuration* oConfiguration)
{
	u32 uDSIM_CONFIG = DSIM_Inp32(rDSIM_CONFIG) & ~(0x1 << 28) & ~(0x1f << 20) & ~(0x3 << 5);
	uDSIM_CONFIG = 	
						#if ( DSIM_Version == DSIM_1_02 )
						(oConfiguration->m_ucEOTDisable << 28 ) |	//EOT Packet Genration						
						#endif						
						(oConfiguration->m_ucAutoVerticalCnt << 24) |	//Auto Vertical Cnt Mode
						(oConfiguration->m_ucHSE << 23) |			//Horizontal Sync Event
						(oConfiguration->m_ucHFP << 22) |			//Discard Horizontal Front Porch Time
						(oConfiguration->m_ucHBP << 21) |			//Discard Horizontal Back Porch Time
						(oConfiguration->m_ucHSA << 20) |			//Discard Horizontal Sync Area Timing
						((u8) oConfiguration->m_eNoDataLane << 5);	//Number of Data Lane using DSI Master
	DSIM_Outp32(rDSIM_CONFIG, uDSIM_CONFIG);
}

//////////
// Function Name : DSIM_EnLane
// Function Description : 
//   This function enable Lanes.
// Input : 	eLane - Lane Selection
//			ucEn - Enable / Disable
// Output : None
// Version : v0.0
void DSIM_EnLane(DSIM_LANE eLane, u8 ucEn)
{
	u32 uTemp = DSIM_Inp32(rDSIM_CONFIG);
	if( eLane == Lane_Clock )
	{
		if ( ucEn )
			uTemp |= (1 << 0);
		else
			uTemp &=  ~(1 << 0);
	}
	else 
	{
		if ( ucEn )
			uTemp |= (eLane << 1);
		else
			uTemp &=  ~(eLane << 1);
	}	
	DSIM_Outp32(rDSIM_CONFIG, uTemp);
}

//////////
// Function Name : DSIM_DisplayConfiguration
// Function Description : 
//   This function enable Lanes.
// Input : 	eLane - Lane Selection
//			ucEn - Enable / Disable
// Output : None
// Version : v0.0
void DSIM_DisplayConfiguration(DDI_Spec* pMainDispConf, DDI_Spec* pSubDispConf)
{
	u32 uDSIM_CONFIG = DSIM_Inp32(rDSIM_CONFIG) & ~(0x3 << 26) & ~(1 << 25) & ~(0x3 << 18) & ~(0x7 << 12) & ~(0x3 << 16) & ~(0x7 << 8);
	u8 ucCnt0 = 0;
	if ( pMainDispConf->m_03Interface == DSIM_Video )
		uDSIM_CONFIG  |= (1 << 25);
	else if ( pMainDispConf->m_03Interface == DSIM_Command )
		uDSIM_CONFIG  &= ~(1 << 25);
	else
	{
		Disp("This DDI is not MIPI Interface!\n");
		Assert('0');
	}
		
	// Main Display 
	uDSIM_CONFIG |=	((u8) (pMainDispConf->m_uParaMeter[DSI_VideoModeSel]) & 0x3) << 26 |	//VideoMode Type
						((u8) (pMainDispConf->m_uParaMeter[DSI_VirtualChID]) & 0x3) << 18 |	//Main Virtual Channel Number
						((u8) (pMainDispConf->m_uParaMeter[DSI_Format]) & 0x7) << 12;	//Main Virtual Channel Pixel Format
	// Sub Display						
	if ( pMainDispConf->m_03Interface == DSIM_Command && pSubDispConf != NULL)
	{
		uDSIM_CONFIG |=	((u8) (pSubDispConf->m_uParaMeter[DSI_VirtualChID]) & 0x3) << 16 |	//Sub Virtual Channel Number
							((u8) (pSubDispConf->m_uParaMeter[DSI_Format]) & 0x7) << 8;	//Sub Virtual Channel Pixel Format
	}	
	DSIM_Outp32(rDSIM_CONFIG, uDSIM_CONFIG);
}


/*---------------------------------- APIs of DSIM_ESCMODE---------------------------------*/
//////////
// Function Name : DSIM_SetStopStateCounter
// Function Description : 
//  This function set counter value to enter BTA State from Stop State when Read Packet is transmitted 
//	or "Set_Tear_On" Command that want data from Peri
// Input : 	uCntVal - Stop Cnt Value
// Output : None
// Version : v0.0
void DSIM_SetStopStateCounter(u16 uCntVal)
{
	u32 uTemp = DSIM_Inp32(rDSIM_ESCMODE) & ~( 0x7ff << 21 );
	uTemp |= ((uCntVal & 0x7ff) << 21);
	DSIM_Outp32(rDSIM_ESCMODE, uTemp);	
}

//////////
// Function Name : DSIM_SetStopDPhy
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_ForceDPhyStopState(u8 ucEn)
{
	u32 uTemp = DSIM_Inp32(rDSIM_ESCMODE) & ~( 0x1 << 20 );
	uTemp |= ((ucEn & 0x1 )<< 20);
	DSIM_Outp32(rDSIM_ESCMODE, uTemp);	

}

//////////
// Function Name : DSIM_SetStopDPhy
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_ForceBTA(void)
{
	DSIM_Outp32(rDSIM_ESCMODE, (DSIM_Inp32(rDSIM_ESCMODE) | ( 0x1 << 16 )));	
	//Delay(1000);
	//DSIM_Outp32(rDSIM_ESCMODE, (DSIM_Inp32(rDSIM_ESCMODE) & ~( 0x1 << 16 )));	
}

//////////
// Function Name : DSIM_SetDataMode
// Function Description : 
//   This function change Data Transfer Mode LPDT bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetDataMode(DSIM_Transfer eData,DSIM_State eState)
{
	u32 uTemp = DSIM_Inp32(rDSIM_ESCMODE);
	if ( eState == DSIM_State_HSClkEn)
		uTemp &= ~eData;
	else 
		uTemp |= eData;
	DSIM_Outp32(rDSIM_ESCMODE, uTemp);	
}

//////////
// Function Name : DSIM_SetFifoLPDTMode
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_ResetTrigger(void)
{
	DSIM_Outp32(rDSIM_ESCMODE, (DSIM_Inp32(rDSIM_ESCMODE) | ( 0x1 << 4 )));	
}

//////////
// Function Name : DSIM_SetFifoLPDTMode
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_ULPSRequest(DSIM_UlpsReques ucRequest, u8 ucBitSet)
{
	u32 uTemp = DSIM_Inp32(rDSIM_ESCMODE);
	
	if ( ucRequest == ULPS_Entry)		// Entry request
	{
		if ( ucBitSet )
			uTemp |= (1 << 1) | (1 << 3);
		else 
			uTemp &= ~(1 << 1) & ~(1 << 3);
	}
	else 				// Exit request
	{
		if ( ucBitSet )
			uTemp |= (1 << 0) |(1 << 2);
		else 
			uTemp &= ~(1 << 0) & ~(1 << 2); 
	}
	

	DSIM_Outp32(rDSIM_ESCMODE, uTemp);	
}

/*---------------------------------- APIs of DSIM_MDRESOL---------------------------------*/
//////////
// Function Name : DSIM_SetMainDispResolution
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetMainDispResolution(u16 usVertiResol, u16 usHorResol)
{
	u32 uTemp = DSIM_Inp32(rDSIM_MDRESOL) & ~( 1 << 31) & ~(0x7ff << 16) & ~(0x7ff << 0);	
	uTemp |=(((usVertiResol) & 0x7ff) << 16) | (((usHorResol )& 0x7ff) << 0);
	DSIM_Outp32(rDSIM_MDRESOL, uTemp);
	uTemp |= ( 1 << 31 );
	DSIM_Outp32(rDSIM_MDRESOL, uTemp);	
}

/*---------------------------------- APIs of DSIM_MVPORCH---------------------------------*/
//////////
// Function Name : DSIM_SetMainDispVPorch
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetMainDispVPorch(u16 usFront, u16 usBack, u16 usCmdAllow)
{
	u32 uTemp = DSIM_Inp32(rDSIM_MVPORCH) & ~(0xf << 28) & ~(0x7ff << 16) & ~(0x7ff << 0);
	uTemp |=((usCmdAllow & 0xf) << 28) | ((usFront & 0x7ff) << 16) | ((usBack & 0x7ff) << 0);
	DSIM_Outp32(rDSIM_MVPORCH, uTemp);	
}

/*---------------------------------- APIs of DSIM_MHPORCH---------------------------------*/
//////////
// Function Name : DSIM_SetMainDispVPorch
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetMainDispHPorch(u16 usFront, u16 usBack)
{
	u32 uTemp = DSIM_Inp32(rDSIM_MHPORCH) & ~(0xffff << 16) & ~(0xffff << 0);
	uTemp |=(usFront  << 16) | (usBack << 0);
	DSIM_Outp32(rDSIM_MHPORCH, uTemp);	
}

/*---------------------------------- APIs of DSIM_MSYNC---------------------------------*/
//////////
// Function Name : DSIM_SetMainDispVPorch
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetMainDispSyncArea(u16 usVeri, u16 usHori)
{
	u32 uTemp = DSIM_Inp32(rDSIM_MSYNC) & ~(0x3ff << 22) & ~(0xffff << 0);
	uTemp |=((usVeri  & 0x3ff)<< 22) | (usHori << 0);
	DSIM_Outp32(rDSIM_MSYNC, uTemp);	
}

/*---------------------------------- APIs of DSIM_SDRESOL---------------------------------*/
//////////
// Function Name : DSIM_SetSubDispResolution
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetSubDispResolution(u16 usVerti, u16 usHori)
{
	u32 uTemp = DSIM_Inp32(rDSIM_SDRESOL) & ~( 1 << 31);
	DSIM_Outp32(rDSIM_SDRESOL, uTemp);
	uTemp &= ~(0x7ff << 16) & ~(0x7ff << 0);
	uTemp |=((usVerti & 0x7ff) << 16) | ((usHori & 0x7ff) << 0);
	DSIM_Outp32(rDSIM_SDRESOL ,uTemp);
	uTemp |= ( 1 << 31 );
	DSIM_Outp32(rDSIM_SDRESOL, uTemp);	
}

/*---------------------------------- APIs of DSIM_INTSRC---------------------------------*/
//////////
// Function Name : DSIM_GetIntrState
// Function Description : 
//   This function get Interrupt conditionof DSI master.
// Input : 	None
// Output : DSIM_IntSrc - Interrupt Condition
// Version : v0.0
DSIM_IntSrc DSIM_GetIntrState(void)
{
	return (DSIM_IntSrc) DSIM_Inp32(rDSIM_INTSRC);	
}

//////////
// Function Name : DSIM_ClrIntr
// Function Description : 
//   This function clear Interrupt pending bit that is caused.
// Input : 	DSIM_IntSrc - Clear Interrupt bit
// Output : None
// Version : v0.0
void DSIM_ClrIntr(DSIM_IntSrc eIntSrc)
{
	DSIM_Outp32(rDSIM_INTSRC, (u32) eIntSrc);	
}
/*---------------------------------- APIs of DSIM_INTMSK---------------------------------*/
//////////
// Function Name : DSIM_SetIntrMask
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_SetIntrMask(DSIM_IntSrc eIntSel, u8 ucMask)
{
	u32 uTemp = DSIM_Inp32(rDSIM_INTMASK);
	if ( ucMask )

		uTemp |= (u32) eIntSel;
	else
		uTemp &= ~((u32)eIntSel);
	DSIM_Outp32(rDSIM_INTMASK, uTemp);	
}

//////////
// Function Name : DSIM_GetIntrMask
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
u32 DSIM_GetIntrMask(void)
{
	return DSIM_Inp32(rDSIM_INTMASK);	
}

/*---------------------------------- APIs of DSIM_PKTHDR---------------------------------*/
//////////
// Function Name : DSIM_WrTxHeader
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_WrTxHeader( u8 ucDI, u8 ucDat0,u8 ucDat1)
{
	u32 uTemp = (ucDat1 << 16) | (ucDat0 << 8) | ((ucDI & 0x3f) << 0);
	DSIM_Outp32(rDSIM_PKTHDR, uTemp);	
}

/*---------------------------------- APIs of DSIM_PAYLOAD---------------------------------*/
//////////
// Function Name : DSIM_WrTxData
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_WrTxData(u32 uTxData)
{
	DSIM_Outp32(rDSIM_PAYLOAD, uTxData);	
}

/*---------------------------------- APIs of DSIM_RXFIFO---------------------------------*/
//////////
// Function Name : DSIM_WrPacketHeader
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
u32 DSIM_RdRxData(void)
{
	return DSIM_Inp32(rDSIM_RXFIFO);	
}

/*---------------------------------- Hidden SFR--------------------------------------------*/
/*---------------------------------- APIs of DSIM_FIFOTHLD---------------------------------*/
//////////
// Function Name : DSIM_WrPacketHeader
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
DSIM_FifoState DSIM_GetFifoState(void)
{
	return (DSIM_FifoState) (DSIM_Inp32(rDSIM_FIFOCTRL) & ~(0x1f));	
}

//////////
// Function Name : DSIM_InitFifoPointer
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_InitFifoPointer(DSIM_Fifo eFifo)
{
	u32 uTemp =  DSIM_Inp32(rDSIM_FIFOCTRL);
	DSIM_Outp32(rDSIM_FIFOCTRL, uTemp & ~(eFifo));
	Delay(10);					//check need to delay
	uTemp |= eFifo;
	DSIM_Outp32(rDSIM_FIFOCTRL, uTemp);	
}

/*---------------------------------- Hidden SFR--------------------------------------------*/
/*---------------------------------- APIs of DSIM_MEMACCHR-------------------------------*/
//////////
// Function Name : DSIM_MemControl
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_MemControl(DSIM_Fifo eFifo)
{

}

/*---------------------------------- APIs of DSIM_PLLCTRL-------------------------------*/
//////////
// Function Name : DSIM_PllFreqBand
// Function Description : 
//   This function enable bypass input clock of D-PHY PLL.
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_PllFreqBand(u8 ucFreqBand)
{
	u32 uTemp =  DSIM_Inp32(rDSIM_PLLCTRL) & ~(0x1f << 24);
	uTemp |= ((ucFreqBand & 0x1f) << 24);
	DSIM_Outp32(rDSIM_PLLCTRL, uTemp);
}
//////////
// Function Name : DSIM_EnPll
// Function Description : 
//   This function enable Pll in D-PHY
// Input : 	usEn - Enable bypass
// Output : None
// Version : v0.0
void DSIM_EnPll(u8 ucEn)
{
	u32 uTemp =  DSIM_Inp32(rDSIM_PLLCTRL) & ~(0x1 << 23);
	uTemp |= ((ucEn & 0x1) << 23);
	DSIM_Outp32(rDSIM_PLLCTRL, uTemp);
}

//////////
// Function Name : DSIM_EnPll
// Function Description : 
//   This function set Pll P,M and S value in D-PHY
// Input : 	ucPredivider - 1 ~ 63
//			ucMaindivider - 100 ~ 511		//45n Phy
//			ucScaler - 0, 1, 2, 3, 4, 5
// Output : None
// Version : v0.0
void DSIM_SetPllPMS(u8 ucPredivider, u16 ucMaindivider, u8 ucScaler)
{
	u32 uTemp =  DSIM_Inp32(rDSIM_PLLCTRL) & ~(0x7ffff << 1);
	#if (DPHY_Version == DPHY_65n)
	uTemp |= ((ucPredivider & 0x3f) << 14) | ((ucMaindivider & 0x3ff) << 4) | ((ucScaler & 0x7) << 1);
	#elif (DPHY_Version == DPHY_45n)
	uTemp |= ((ucPredivider & 0x3f) << 13) | ((ucMaindivider & 0x1ff) << 4) | ((ucScaler & 0x7) << 1);
	#endif
	DSIM_Outp32(rDSIM_PLLCTRL, uTemp);
}

//////////
// Function Name : DSIM_EnPll
// Function Description : 
//   This function set Pll P,M and S value in D-PHY
// Input : 	ucPredivider - 1 ~ 63
//			ucMaindivider - 100 ~ 1023		//65n Phy
//			ucScaler - 0, 1, 2, 3, 4, 5
// Output : None
// Version : v0.0
void DSIM_PllFreq(u8 ucPredivider, u16 ucMaindivider, u8 ucScaler)
{
	u32 uTemp =  DSIM_Inp32(rDSIM_PLLCTRL) & ~(0x7ffff << 1);
	#if (DPHY_Version == DPHY_65n)
	uTemp |= ((ucPredivider & 0x3f) << 14) | ((ucMaindivider & 0x3ff) << 4) | ((ucScaler & 0x7) << 1);
	#elif (DPHY_Version == DPHY_45n)
	uTemp |= ((ucPredivider & 0x3f) << 13) | ((ucMaindivider & 0x1ff) << 4) | ((ucScaler & 0x7) << 1);
	#endif
	DSIM_Outp32(rDSIM_PLLCTRL, uTemp);
}
#if (DSIM_Version == DSIM_1_00)
//////////
// Function Name : DSIM_EnPll
// Function Description : 
//   This function set Pll P,M and S value in D-PHY
// Input : 	ucPredivider - 1 ~ 63
//			ucMaindivider - 100 ~ 1023
//			ucScaler - 0, 1, 2, 3, 4, 5
// Output : None
// Version : v0.0
void DSIM_SwapDpDn(u8 ucSwapEn)
{
	u32 uTemp =  DSIM_Inp32(rDSIM_PLLCTRL) & ~(1 << 0);
	uTemp |= ucSwapEn & 1;
	DSIM_Outp32(rDSIM_PLLCTRL, uTemp);
}
#endif
/*---------------------------------- APIs of DSIM_PLLTMR-------------------------------*/
//////////
// Function Name : DSIM_PllLockTime
// Function Description : 
//   This function set Pll P,M and S value in D-PHY
// Input : 	uLockTime - 1 ~ 63
//			ucMaindivider - 100 ~ 1023
//			ucScaler - 0, 1, 2, 3, 4, 5
// Output : None
// Version : v0.0
void DSIM_PllStatbleTime(u32 uLockTime)
{
	DSIM_Outp32(rDSIM_PLLTMR, uLockTime);
}

#if (DPHY_Version == DPHY_45n)
/*---------------------------------- APIs of DSIM_PHYACCHR-------------------------------*/
//////////
// Function Name : DSIM_SetPhyTunning
// Function Description : 
//   This function set Pll P,M and S value in D-PHY
// Input : 	uLockTime - 1 ~ 63
//			ucMaindivider - 100 ~ 1023
//			ucScaler - 0, 1, 2, 3, 4, 5
// Output : None
// Version : v0.0
void DSIM_SetPhyTunning(u32 uTunnigValue)
{
	DSIM_Outp32(rDSIM_PHYACCHR, uTunnigValue);
}

//////////
// Function Name : DSIM_EnAFC
// Function Description : 
//   This function En AFC 
// Input : 	ucEn : Enable/Disable
// Output : None
// Version : v0.0
void DSIM_EnAFC(u8 ucEn, u8 ucAFCCode)
{
	u32 uTemp =  DSIM_Inp32(rDSIM_PHYACCHR);
	if (ucEn)
	{
		uTemp |= (1 << 14);
		uTemp &= ~(0x7 << 5);
		uTemp |= (ucAFCCode & 0x7) << 5;
	}
	else
		uTemp &= ~(1 << 14);	
	DSIM_Outp32(rDSIM_PHYACCHR, uTemp);
}
/*---------------------------------- APIs of DSIM_PHYACCHR1------------------------------*/
//////////
// Function Name : DSIM_PllLockTime
// Function Description : 
//   This function set Pll P,M and S value in D-PHY
// Input : 	uLockTime - 1 ~ 63
//			ucMaindivider - 100 ~ 1023
//			ucScaler - 0, 1, 2, 3, 4, 5
// Output : None
// Version : v0.0
void DSIM_DpDnSwap(DSIM_LANE eLane, u8 ucSwapEn)
{
	u32 uTemp = DSIM_Inp32(rDSIM_PHYACCHR1);
	if ( eLane & Lane_Clock )
	{
		uTemp &= ~(1 << 1);
		uTemp |= ucSwapEn << 1;
	}

	if ( eLane & (Lane_Data1|Lane_Data2|Lane_Data3|Lane_Data0))
	{
		uTemp &= ~(1 << 0);
		uTemp |= ucSwapEn << 0;
	}
	DSIM_Outp32(rDSIM_PHYACCHR1, uTemp);
}
#endif
