/********************************************************************************
* 
*	Project Name : S5PV210 Validation
*
*	Copyright 2008 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for verifying functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------
* 
*	File Name : dcs.c
*  
*	File Description :
*
*	Author	: Sung-Hyun, Na
*	Dept. : AP Development Team
*	Created Date : 2009/03/10
*	Version : 0.1
* 
*	History
*	- Version 0.0 (Sung-Hyun,Na, 2009/03/10)
*	  	-> convert from Media Team DCSProtocol.cpp 
*	- Version 0.1 (Sung-Hyn,Na 2009/06/17)
*		-> Check Rx Ack Enable/Disable using DSI Protocol
*		-> Seperate H/W Link and DCS Command
*
********************************************************************************/

#include <stdlib.h>
#include <string.h>

#include "library.h"

#include "dcs.h"

#include "dsim.h"
#include "lcd.h"

/*------------------------------------ macros for dcs.c----------------------------------------*/
#define READBUFFER		8
#define HI_BYTE(x)		((x&0xFF00)>>8)
#define LO_BYTE(x)		((x)&0xFF)
#define BITBYTE(b7,b6,b5,b4,b3,b2,b1,b0)	( (b7)<<7 | (b6)<<6 | (b5)<<5 | (b4)<<4 | (b3)<<3 | (b2)<<2 | (b1)<<1 | (b0) )

//static DCS_Infor* g_pDCS;

/*---------------------------------- APIs of DCS Object------------------------*/
//////////
// Function Name : DCS_Create
// Function Description : 
//   This function creates Display Command Set(DCS) information structure
// Input : 	DSIM_Infor* : Information structure of Display Serial Interface(DSI) Driver.
// Output : DCS_Infor* : Information structure of DCS Driver
// Version : v0.0
DCS_Infor* DCS_Create(Connectivity_Link eProtocol, u32 uDriver, u8 ucVcID) 
{
	DCS_Infor* pInfor = (DCS_Infor*) malloc(sizeof(DCS_Infor));
	if (pInfor)
	{
		memset(pInfor,0,sizeof(DCS_Infor));
		pInfor->m_eProtocol = eProtocol;
		pInfor->m_uLinkProtocol = uDriver;
		pInfor->m_ucVcID = ucVcID;
		pInfor->m_bCheckRxAck = false;
		return pInfor;
	}
	else
	{
		return false;
	}
}

//////////
// Function Name : DCS_Destory
// Function Description : 
//   This function destroys Display Command Set(DCS) information structure
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : None
// Version : v0.0
void DCS_Destory(DCS_Infor* pInfor)
{
	if( pInfor != NULL)
	{
		memset(pInfor,0,sizeof(DCS_Infor));
		free(pInfor);
	}
}

//////////
// Function Name : DCS_EnCheckRxAck
// Function Description : 
//   This function enable Rx Ack for Non read type Rx Display Command Set(DCS)
// Input : 	DSIM_Infor* pInfor	- Information structure of DCS Driver
//			u8 bEn				- Enable Rx Check
// Output : None
// Version : v0.1
void DCS_EnCheckRxAck(DCS_Infor* pInfor, u8 bEn)
{
	if( bEn )
		pInfor->m_bCheckRxAck = true;
	else
		pInfor->m_bCheckRxAck = false;
}

/*---------------------------------- APIs of DCS Read/Write------------------------*/
//////////
// Function Name : DCS_SimpleWrite
// Function Description : 
//   This function writer command with 1 parameter.
// Input : 	DSIM_Infor* pInfor	- Information structure of DCS Driver
//			DCS_CmdSet eId	- Command
//			u8 ucPara			- parameter
// Output : None
// Version : v0.1
u32 DCS_SimpleWrite(DCS_Infor* pInfor, DCS_CmdSet eId, u8 ucPara)
{
	if ( pInfor->m_eProtocol == Link_DSIM )
	{
		DSIM_Infor* pDSILink = (DSIM_Infor*)(pInfor->m_uLinkProtocol);
		return DSIM_WrData(pDSILink, DCSWrNoPara,eId,ucPara, pInfor->m_bCheckRxAck);
	}
	else if ( pInfor->m_eProtocol == Link_LCDC_I80 )
	{
		CPUIF_LDI eLdiMode = (CPUIF_LDI)(pInfor->m_uLinkProtocol);
		u32 uAlignBuffer;
		uAlignBuffer = (ucPara << 16) | ((u8)eId << 8) | ((u8) DCSWrNoPara);
		LCDC_PutDataToLdiByManual(uAlignBuffer,eLdiMode);
		return true;
	}
	else
	{
		Disp("Other Link is not embodied for DCS\n");
		return false;
	}
}

//////////
// Function Name : DCS_LongWrite
// Function Description : 
//   This function write Long command 
// Input : 	DSIM_Infor* pInfor	- Information structure of DCS Driver
//			u8* pAddr			- Address of Parameter
//			u32 uSize			- size of Parameter
// Output : None
// Version : v0.1
u32 DCS_LongWrite(DCS_Infor* pInfor, u8* pAddr, u32 uSize)
{
	if ( pInfor->m_eProtocol == Link_DSIM )
	{
		DSIM_Infor* pDSILink = (DSIM_Infor*)(pInfor->m_uLinkProtocol);
		return DSIM_WrData(pDSILink, DCSLongWr,(u32) pAddr,uSize, pInfor->m_bCheckRxAck);
	}
	else if ( pInfor->m_eProtocol == Link_LCDC_I80 )
	{
		CPUIF_LDI eLdiMode = (CPUIF_LDI)(pInfor->m_uLinkProtocol);
		u32 uAlignBuffer;		
		u32 uCnt;
		u32* pWordPtr = (u32*)malloc((uSize+3)/4*4);

		uAlignBuffer = ((u16)uSize << 8) |((u8)DCSLongWr);
		//uAlignBuffer = ((u8)(uSize & 0xff) << 16)  |((u8)(uSize & 0xff00) << 8) |((u8)DCSLongWr);
		LCDC_PutDataToLdiByManual(uAlignBuffer,eLdiMode);
	
		memset(pWordPtr,0,(uSize+3)/4*4*4);		
		memcpy(pWordPtr,(u8 *) pAddr,uSize);		
		// Put data into Payload
		for( uCnt = 0;uCnt < uSize/4;uCnt++)
		{
			LCDC_PutPayLoad(pWordPtr[uCnt], eLdiMode);
		}
		if(uSize%4)
			LCDC_PutPayLoad(pWordPtr[uCnt], eLdiMode);
		free(pWordPtr);			

		return true;
	}
	else
	{
		Disp("Other Link is not embodied for DCS\n");
		return false;
	}
}

// Function Name : DCS_Read
// Function Description : 
//   This function read DCS Register.
// Input : 	pInfor : Information structure of DCS Driver
//			cmd : read command 
//			pBuf : Address of Buffer for returned Data 
//			bufSize : size of buffer
//			pnRead : Actual size of returned Data
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_Read(DCS_Infor* pInfor, DCS_CmdSet cmd,u8 *pBuf,u32 nBufsize,u32* pRead)
{
	if ( pInfor->m_eProtocol == Link_DSIM )
	{
		DSIM_ShortPktHeader header;
		DSIM_Infor* pDSILink = (DSIM_Infor*)(pInfor->m_uLinkProtocol);
		u8 ret = DSIM_WrData(pDSILink,DCSRdNoPara,cmd,0, pInfor->m_bCheckRxAck);		
		if (ret) 
		{
			// check rx 
			if (DSIM_PopRxHeader(pDSILink, &header)) 
			{
				DSIM_Response eResponse = DSIM_GetRxHeaderType(&header);
				if ( eResponse == DcsShort1B || eResponse == GenShort1B) 
				{
					memcpy(pBuf,(void*)&(header._inter.data.data0),1);
					if ( pRead ) 
						*pRead=1;
					return true;
				}
				else if (eResponse == DcsShort2B || eResponse == GenShort2B) 
				{
					memcpy(pBuf,(void*)&(header._inter.data.data0),2);
					if ( pRead ) 
						*pRead=2;
					return true;
				}
				else if (eResponse == DcsLong || eResponse == GenLong)
				{
					u32 toReadActual = (*((u32 *)&header) & 0xff00) >> 8;
					if ( pRead ) 
					{
						if (toReadActual > nBufsize) 
						{
							// TODO : warn user
							Disp("Warnning!! Read data size is bigger than user set buffer size\n");
							Disp("Read Data size is %d",toReadActual);
							Disp("Press Any key\n");
							UART_Getc();
							toReadActual = nBufsize;
						}					
						*pRead = DSIM_PopRxPayload(&header,pBuf,toReadActual);
						if (*pRead == toReadActual)
							return true;				
					}
				}
				else 
					return false;
			}
			else 
				return false;
		}
		else 
			return false;
	}
	else if ( pInfor->m_eProtocol == Link_LCDC_I80 )
	{
		Disp("It is impossible to get data on LCDC(I80) Protocol\n");
		return false;
	}
	else
	{
		Disp("Other Link is not embodied for DCS\n");
		return false;
	}
}

// Function Name : DCS_WriteMemoryStart
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_WriteLongPacketWithDCSCommand(DCS_Infor* pInfor, DCS_CmdSet eCmd,u8* pBuf,u32 uSize) //u8 useHs,u8 dummy
{	
	if (pInfor->m_eProtocol == Link_DSIM)
	{
		DSIM_Infor* pDSILink = (DSIM_Infor*)(pInfor->m_uLinkProtocol);
		u8 bPass;
		u8* pDatatoSend = (u8 *)malloc(2048);			//2KB(4Byte * 512 Depth) : Maximum SFR Payload FIFO Size + command 
		u8* pStartAddr = pDatatoSend;
		u32 uCnt;
		
		if (pDatatoSend == NULL) 
			return false;
		
		memset(pDatatoSend,0,2048);
		*(pDatatoSend++) = (u8) eCmd;
		if ( uSize > 2047 )
		{
			Disp("Maximum Payload FIFO  size is 2KB.\n");			
		}
		else
		{
			// byte memcpy
			for(uCnt =0; uCnt < uSize; uCnt++)
			{
				*(pDatatoSend++) = *(pBuf++);
			}
			bPass = DSIM_WrData(pDSILink, DCSLongWr, (u32)(pStartAddr), uSize+1,pInfor->m_bCheckRxAck);
		}
		free(pStartAddr);
		return bPass;
	}
	else if ( pInfor->m_eProtocol == Link_LCDC_I80 )
	{
		CPUIF_LDI eLdiMode = (CPUIF_LDI)(pInfor->m_uLinkProtocol);
		u32 uAlignBuffer = 0;	
		u8* pDatatoSend = pBuf;		
		u32 uCnt;	
		if ( uSize > 2047 )
		{
			Disp("Maximum Payload FIFO  size is 2KB.\n");
			return false;
		}
		else			// byte memcpy
		{
			uAlignBuffer = ((u16)(uSize+1) << 8) |((u8)DCSLongWr);
			//uAlignBuffer = ((u8)((u16)(uSize+1) & 0xff) << 16)  |((u8)((u16)(uSize+1) & 0xff00) << 8) |((u8)DCSLongWr);
			LCDC_PutDataToLdiByManual(uAlignBuffer,eLdiMode);
			
			uAlignBuffer = (u8) eCmd;
			for(uCnt = 1; uCnt < uSize; uCnt++)
			{
				uAlignBuffer |=  (*(pDatatoSend++)) << ((uCnt % 4) * 8);
				if ( (uCnt != 0) && (uCnt % 4 == 3) )
				{
					LCDC_PutPayLoad(uAlignBuffer, eLdiMode);
					uAlignBuffer = 0;
				}					
			}
			if (uAlignBuffer)
			{
				LCDC_PutPayLoad(uAlignBuffer, eLdiMode);
				uAlignBuffer = 0;
			}			
			return true;
		}

	}
	else
	{
		Disp("Other Link is not embodied yet!!\n");
		return false;
	}
	
}
/*---------------------------------- DCS Commands------------------------*/
//////////
// Function Name : DCS_EnterIdle
// Function Description : 
//
// 6.1 enter_idle_mode
// This command causes the display module to enter Idle Mode.
// In Idle Mode, color expression is reduced. Colors are shown on the display device using the MSB of each
// of the R, G and B color components in the frame memory.
//
// Paramter : NONE
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_EnterIdle(DCS_Infor* pInfor) 
{
	return DCS_SimpleWrite(pInfor, ENTER_IDLE,0);
}

//////////
// Function Name : DCS_EnterInvertMode
// Function Description : 
//
// 6.2 enter_invert_mode
// This command causes the display module to invert the image data only on the display device. 
// The frame memory contents remain unchanged. No status bits are changed.
//
// Paramter : NONE
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_EnterInvertMode(DCS_Infor* pInfor) 
{
	return DCS_SimpleWrite(pInfor, ENTER_INVERT,0);
}

//////////
// Function Name : DCS_EnterNormalMode
// Function Description : 
//
// 6.3 enter_normal_mode
// This command causes the display module to enter the Normal mode.
// Normal Mode is defined as Partial Display mode and Scroll mode are off.
// The host processor sends PCLK, HS and VS information to Type 2 display modules two frames before this
// command is sent when the display module is in Partial Display Mode.
//
//
// Paramter : NONE
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_EnterNormalMode(DCS_Infor* pInfor) 
{
	return DCS_SimpleWrite(pInfor, ENTER_NORMAL,0);	
}

//////////
// Function Name : DCS_EnterPartialMode
// Function Description : 

// 6.4 enter_partial_mode
// This command causes the display module to enter the Partial Display Mode. The Partial Display Mode
// window is described by the set_partial_area command. See section 6.30 for details.
// To leave Partial Display Mode, the enter_normal_mode command should be written.
// The host processor continues to send PCLK, HS and VS information to Type 2 display modules for two
// frames after this command is sent when the display module is in Normal Display Mode.
//
// Paramter : NONE
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_EnterPartialMode(DCS_Infor* pInfor)
{
	return DCS_SimpleWrite(pInfor, ENTER_PARTIAL,0);
}


//////////
// Function Name : DCS_EnterSleepMode
// Function Description : 
//
// 6.5 enter_sleep_mode
// This command causes the display module to enter the Sleep mode.
// In this mode, all unnecessary blocks inside the display module are disabled except interface
// communication. This is the lowest power mode the display module supports.
// DBI or DSI Command Mode remains operational and the frame memory maintains its contents. The host
// processor continues to send PCLK, HS and VS information to Type 2 and Type 3 display modules for two
// frames after this command is sent when the display module is in Normal mode.
//
// Paramter : NONE
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_EnterSleepMode(DCS_Infor* pInfor)
{
	u8 ucRet = DCS_SimpleWrite(pInfor, ENTER_SLEEP,0);
	// see DCS spec v01-01-00 563 line 
	if (ucRet)
	{
		Delay(60);		
	}
	return ucRet;
}

//////////
// Function Name : DCS_ExitIdleMode
// Function Description : 
// 
// 6.6 exit_idle_mode
// This command causes the display module to exit Idle mode.
//
// Paramter : NONE
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_ExitIdleMode(DCS_Infor* pInfor) 
{
	return DCS_SimpleWrite(pInfor, EXIT_IDLE,0);
}

//////////
// Function Name : DCS_ExitInvertMode
// Function Description : 
//
// 6.7 exit_invert_mode
// This command causes the display module to stop inverting the image data on the display device. The frame
// memory contents remain unchanged. No status bits are changed.
//
// Paramter : NONE
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_ExitInvertMode(DCS_Infor* pInfor)
{
	return DCS_SimpleWrite(pInfor, EXIT_INVERT,0);	
}

//////////
// Function Name : DCS_ExitSleepMode
// Function Description : 
// 
// 6.8 exit_sleep_mode
// This command causes the display module to exit Sleep mode. All blocks inside the display module are enabled.
// The host processor sends PCLK, HS and VS information to Type 2 and Type 3 display modules two frames
// before this command is sent when the display module is in Normal Mode.
//
// Paramter : NONE
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_ExitSleepMode(DCS_Infor* pInfor) 
{
	u8 ucRet=DCS_SimpleWrite(pInfor, EXIT_SLEEP,0);
	// see DCS spec v01-01-00 610 line 
	if (ucRet)
		Delay(600);			//Wait 60ms
	return ucRet;
}

//////////
// Function Name : DCS_ExitSleepMode
// Function Description : 
//
// 6.9 get_address_mode
// The display module returns the current status. 
// Parameters : 1 
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
//			u8* : Address of Returned Parameter from Display.
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetAddressMode(DCS_Infor* pInfor, u8 *pMode)
{
	// Write GET_ADDRESS_MODE
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_ADDRESS_MODE, ucReadBuffer, READBUFFER, NULL)) 
	{
		*pMode = ucReadBuffer[0];
		return TRUE;
	}
	else
		return FALSE;	
}

//////////
// Function Name : DCS_GetBlueChannel
// Function Description : 

// 6.10 get_blue_channel
// The display module returns the blue component value of the first pixel in the active frame. 
// This command is only valid for Type 2 and Type 3 display modules.
//
// Paramter : 1
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetBlueChannel(DCS_Infor* pInfor, u8* pBlueCh)
{
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_BLUE_CHANNEL, ucReadBuffer, READBUFFER, NULL)) 
	{
		*pBlueCh = ucReadBuffer[0];
		return true;
	}
	return false;
}

//////////
// Function Name : DCS_GetDisanosticResult
// Function Description : 
// 6.11 get_diagnostic_result
//
// The display module returns the self-diagnostic results following a Sleep Out command. 
// See section 5.3 for a description of the status results.
//
// Paramter : 1
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetDisanosticResult(DCS_Infor* pInfor, u8* pResult)
{
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_DIAGNOSTIC_RESULT, ucReadBuffer, READBUFFER, NULL)) 
	{
		*pResult = ucReadBuffer[0];
		return true;
	}
	return false;
}


//////////
// Function Name : GetDisplayMode
// Function Description : 
//
// 6.12 get_display_mode
// The display module returns the Display Image Mode status.
//
// Parameters : 1
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetDisplayMode(DCS_Infor* pInfor, u8* pMode) 
{
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_DISPLAY_MODE, ucReadBuffer, READBUFFER, NULL)) 	
	{
		*pMode = ucReadBuffer[0];
		return true;
	}
	return false;	
}

//////////
// Function Name : GetGreenChannel
// Function Description : 
//
// 6.13 get_green_channel
// The display module returns the green component value of the first pixel in the active frame. 
// This command is only valid for Type 2 and Type 3 display modules.
//
// Parameters : 1
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetGreenChannel(DCS_Infor* pInfor, u8 *pGreenCh) 
{
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_GREEN_CHANNEL, ucReadBuffer, READBUFFER, NULL)) 	
	{
		*pGreenCh = ucReadBuffer[0];
		return true;
	}
	return false;	
}

//////////
// Function Name : GetGreenChannel
// Function Description : 
//
// 6.14 get_pixel_format
// This command gets the pixel format for the RGB image data used by the interface.
// Bits D[6:4] . DPI Pixel Format Definition
// Bits D[2:0] . DBI Pixel Format Definition
// Bits D7 and D3 are not used.
// 
// Parameters : 1
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetPixelFormat(DCS_Infor* pInfor, u8 *pFmt) 
{
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_PIXEL_FORMAT, ucReadBuffer, READBUFFER, NULL)) 	
	{
		*pFmt = ucReadBuffer[0];
		return true;
	}
	return false;	
}

//////////
// Function Name : GetGreenChannel
// Function Description : 
//
// 6.15 get_power_mode
// The display module returns the current power mode.
//
// Parameters : 1
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
//			u8* : Address of Returned Data
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetPowerMode(DCS_Infor* pInfor, u8* pMode) 
{
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_POWER_MODE, ucReadBuffer, READBUFFER, NULL)) 	
	{
		*pMode = ucReadBuffer[0];
		return true;
	}
	return false;	
}

//////////
// Function Name : DCS_GetRedChannel
// Function Description : 
//
// 6.16 get_red_channel
// The display module returns the red component value of the first pixel in the active frame. 
// This command is only valid for Type 2 and Type 3 display modules.
//
// Parameters : 1
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetRedChannel(DCS_Infor* pInfor, u8* pRedCh) 
{
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_RED_CHANNEL, ucReadBuffer, READBUFFER, NULL)) 	
	{
		*pRedCh = ucReadBuffer[0];
		return true;
	}
	return false;	
}

//////////
// Function Name : DCS_GetScanLine
// Function Description : 
//   
// 6.17 get_scanline
// The display module returns the current scanline, N, used to update the display device. 
// The total number of scanlines on a display device is defined as VSYNC + VBP + VACT + VFP. 
// The first scanline is defined as the first line of V Sync and is denoted as Line 0.
// When in Sleep Mode, the value returned by get_scanline is undefined.
// See MIPI Alliance Standard for Display Pixel Interface (DPI-2) [1] for definitions of VSYNC, VBP, VACT, and VFP.
//
// Parameters : 2
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
//			u16* : Address of Returnd Data
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetScanLine(DCS_Infor* pInfor, u16* line) 
{
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_SCANLINE, ucReadBuffer, READBUFFER, NULL)) 	
	{
		*line = (ucReadBuffer[1] << 8) | (ucReadBuffer[0] << 0);
		return true;
	}
	return false;	
}

//////////
// Function Name : DCS_GetScanLine
// Function Description : 
// 
// 6.18 get_signal_mode
// The display module returns the Display Signal Mode.
//
// Parameters : 1
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
//			u8* : Address of Returned Data
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_GetSignalMode(DCS_Infor* pInfor, u8* pSig) 
{
	u8 ucReadBuffer[READBUFFER];
	if (DCS_Read(pInfor, GET_SIGNAL_MODE, ucReadBuffer, READBUFFER, NULL)) 	
	{
		*pSig = (ucReadBuffer[0]);
		return true;
	}
	return false;	
}

//////////
// Function Name : DCS_Nop
// Function Description : 
//  
// 6.19 nop
// This command does not have any effect on the display module.
// It can be used to terminate a Frame  Memory Write or Read 
// as described in the descriptions for write_memory_continue and read_memory_continue.
//
// Parameters : NONE
//
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_Nop(DCS_Infor* pInfor) 
{
	return DCS_SimpleWrite(pInfor, NOP,0);
}

// Function Name : DCS_ReadDDBStart
// Function Description : 
// 
// 6.20 read_DDB_continue
// Input : 	pInfor : Information structure of DCS Driver
//			pBuf : Address of Buffer for returned Data 
//			bufSize : size of buffer
//			pnRead : Actual size of returned Data
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_ReadDDBStart(DCS_Infor* pInfor, u8* pBuf,u32 bufSize,u32* pnRead)
{
	return DCS_Read(pInfor, READ_DDB_START,pBuf,bufSize,pnRead);
}

// Function Name : DCS_ReadDDBContinue
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	pInfor : Information structure of DCS Driver
//			pBuf : Address of Buffer for returned Data 
//			bufSize : size of buffer
//			pnRead : Actual size of returned Data
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_ReadDDBContinue(DCS_Infor* pInfor, u8* pBuf,u32 bufSize,u32* pnRead) 
{
	return DCS_Read(pInfor, READ_DDB_CONTINUE,pBuf,bufSize,pnRead);
}

// Function Name : DCS_ReadMemoryStart
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	pInfor : Information structure of DCS Driver
//			pBuf : Address of Buffer for returned Data 
//			bufSize : size of buffer
//			pnRead : Actual size of returned Data
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_ReadMemoryStart(DCS_Infor* pInfor, u8* pBuf,u32 bufSize,u32* pnRead) 
{
	return DCS_Read(pInfor, READ_MEMORY_START,pBuf,bufSize,pnRead);
}

// Function Name : DCS_ReadMemoryContinue
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	pInfor : Information structure of DCS Driver
//			pBuf : Address of Buffer for returned Data 
//			bufSize : size of buffer
//			pnRead : Actual size of returned Data
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_ReadMemoryContinue(DCS_Infor* pInfor, u8* pBuf,u32 bufSize,u32* pnRead)
{
	return DCS_Read(pInfor, READ_MEMORY_CONTINUE,pBuf,bufSize,pnRead);
}
/*---------------------------------- wrappers of ------------------------*/

// Function Name : DCS_SetAddressMode
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetAddressMode(DCS_Infor* pInfor, u8 mode) 
{
	return DCS_SimpleWrite(pInfor, SET_ADDRESS_MODE, mode);
}

// Function Name : DCS_SetPageAddress
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetPageAddress(DCS_Infor* pInfor, u16 startPage,u16 endPage)
{
	u8 dataToSend[5];
	dataToSend[0] =	SET_PAGE_ADDRESS;
	dataToSend[1] =	HI_BYTE(startPage);dataToSend[2] =	LO_BYTE(startPage);
	dataToSend[3] =	HI_BYTE(endPage);	dataToSend[4] =	LO_BYTE(endPage);
	
	if (startPage > endPage) 
		return false;
	return DCS_LongWrite(pInfor, dataToSend, 5);
}

// Function Name : DCS_SetColumnAddress
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetColumnAddress(DCS_Infor* pInfor, u16 sc,u16 ec) 
{
	
	u8 dataToSend[5];
	dataToSend[0] =	SET_COLUMN_ADDRESS;
	dataToSend[1] =	HI_BYTE(sc);dataToSend[2] = LO_BYTE(sc);
	dataToSend[3] =	HI_BYTE(ec);dataToSend[4] = LO_BYTE(ec);
	
	if (sc > ec)
		return false;
	return DCS_LongWrite(pInfor, dataToSend, 5);
}

// Function Name : DCS_SetDisplayOff
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetDisplayOff(DCS_Infor* pInfor) 
{
	return DCS_SimpleWrite(pInfor, SET_DISPLAY_OFF,0);
}


// Function Name : DCS_SetDisplayOn
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetDisplayOn(DCS_Infor* pInfor) 
{
	return DCS_SimpleWrite(pInfor,SET_DISPLAY_ON,0);
}


// Function Name : DCS_SetGammaCurve
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetGammaCurve(DCS_Infor* pInfor, GAMMA_CURVE_t curve)
{
	return DCS_SimpleWrite(pInfor, SET_GAMMA_CURVE,(u8)curve);
}

// Function Name : DCS_SetPartialArea
// Function Description : 
//  This command defines the partial mode's display area. 
//	There are 2 parameters associated with this command, 
//	the first defines the Start Row (SR) and the second the End Row (ER).
// 	SR and ER refer to the Frame Memory Line Pointer.
// Input : 	DSIM_Infor* : Information structure of DCS Driver
//			startRow : Start Row Address
//			endRow : End Row Address
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetPartialArea(DCS_Infor* pInfor, u16 startRow,u16 endRow) 
{
	u8 dataToSend[5];
	dataToSend[0] = SET_PARTIAL_AREA;
	dataToSend[1] = HI_BYTE(startRow); dataToSend[2] = LO_BYTE(startRow);
	dataToSend[3] = HI_BYTE(endRow); dataToSend[4] = LO_BYTE(endRow);
	return DCS_LongWrite(pInfor, dataToSend, 5);
}

// Function Name : DCS_SetPixelFormat
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetPixelFormat(DCS_Infor* pInfor, u8 format)
{
    return DCS_SimpleWrite(pInfor, SET_PIXEL_FORMAT,format);
}


// Function Name : DCS_SetScrollArea
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetScrollArea(DCS_Infor* pInfor, u16 top_fixed_area_lines,u16 vertical_scrolling_area_lines,	u16 bottom_fixed_area_lines) 
{
	/*if (top_fixed_area_lines + vertical_scrolling_area_lines + bottom_fixed_area_lines !=
		m_uLinkProtocol->GetDisplayHeight()) 
	{
		// the sum of parameter must equal the number of display device's horizontal lines
		return false;
	}*/
	u8 dataToSend[7] ;
	dataToSend[0] = SET_SCROLL_AREA;
	dataToSend[1] = HI_BYTE(top_fixed_area_lines);		dataToSend[2] = LO_BYTE(top_fixed_area_lines);
	dataToSend[3] = HI_BYTE(vertical_scrolling_area_lines);	dataToSend[4] = LO_BYTE(vertical_scrolling_area_lines);
	dataToSend[5] = HI_BYTE(bottom_fixed_area_lines);	dataToSend[6] = LO_BYTE(bottom_fixed_area_lines);	
	return DCS_LongWrite(pInfor, dataToSend, 7);
}

// Function Name : DCS_SetScrollStart
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetScrollStart(DCS_Infor* pInfor, u16 vertical_scroll_pointer) 
{
	u8 dataToSend[3];
	dataToSend[0] = SET_SCROLL_START;
	dataToSend[1] = HI_BYTE(vertical_scroll_pointer); dataToSend[2] =LO_BYTE(vertical_scroll_pointer);
	return DCS_LongWrite(pInfor, dataToSend, 3);
}

// Function Name : DCS_SetTearOff
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetTearOff(DCS_Infor* pInfor)
{
	return DCS_SimpleWrite(pInfor,SET_TEAR_OFF,0);
}

// Function Name : DCS_SetTearOn
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetTearOn(DCS_Infor* pInfor, u8 bothBlanking) 
{
	return DCS_SimpleWrite(pInfor, SET_TEAR_ON,(u8)bothBlanking);
}

// Function Name : DCS_SetTearScanline
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SetTearScanline(DCS_Infor* pInfor, u16 line) 
{
	u8 dataToSend[3];
	dataToSend[0] =SET_TEAR_SCANLINE,
	dataToSend[1] = HI_BYTE(line); dataToSend[2] =LO_BYTE(line);
	return DCS_LongWrite(pInfor, dataToSend, 3);
}

// Function Name : DCS_SetTearScanline
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_SoftReset(DCS_Infor* pInfor) 
{
	return DCS_SimpleWrite(pInfor,SOFT_RESET,0);
}

// Function Name : DCS_WriteLUT
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_WriteLUT(DCS_Infor* pInfor, u8* pBuf,u32 size)
{
	return DCS_WriteLongPacketWithDCSCommand(pInfor, WRITE_LUT,pBuf,size);
}

// Function Name : DCS_WriteMemoryContinue
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_WriteMemoryContinue(DCS_Infor* pInfor, u8* pBuf,u32 size) 
{
	return DCS_WriteLongPacketWithDCSCommand(pInfor, WRITE_MEMORY_CONTINUE,pBuf,size);
}

// Function Name : DCS_WriteMemoryStart
// Function Description : 
//   This function is Display module to operate nothing
// Input : 	DSIM_Infor* : Information structure of DCS Driver
// Output : u8 (Bool) : true/false
// Version : v0.0
u8 DCS_WriteMemoryStart(DCS_Infor* pInfor, u8* pBuf,u32 size) 
{
	return DCS_WriteLongPacketWithDCSCommand(pInfor, WRITE_MEMORY_START,pBuf,size);
}






