/**************************************************************************************
* 
*	Project Name : S5PV210 Silicon Validation
*
*	Copyright 2008 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : csi.c
*  
*	File Description : This file implements CSI-Slave functions.
*
*	Author : Yuanfeng,chen
*	Dept. : AP Development Team
*	Created Date : 
*	Version : 0.1 
* 
*	History
*	- Created(Yuanfeng,chen ????/??/??)
*	- Modified(Jun,Kim 2009/04/17)
**************************************************************************************/

#include "option.h"
#include "system.h"
#include "library.h"
#include "sysc.h"
#include "lcd.h"
#include "fimc.h"
#include "intc.h"
#include "glib.h"
#include "rtc.h"
#include "csi.h"
#include "mipi_camera.h"

#include "max8698.h"			//to control PMIC for MIPI 1.8V
#include "dsim.h"
#include "dcs.h"		//Display Command Set Driver

static volatile u32 bProcessingDoneA;
static volatile u32 gCaptureCnt;
static volatile u32 gBufferIdx=0;

static CIS_MODEL eCisModel;
static IMG_FMT eCameraFmtType;
static IMG_RESOLUTION eCameraSize;
static MIPI_DATA_ALIGN eMIPIDataAlign;
static CSIS_WCLK_SRC eWCLKSrc;
static CSIS_INTV_SETTING eCSISIntvSetting;
static CSIRaw_Color eCsiRawColorType=CSIRaw_NoColor;
static u8 eHSsettle;

CSIS_DATA_LANE_NUM eCSISDataLaneNum;

// BayerRGB converting address
#define RGB_IMG_ADDR			0x42000000	// Convert raw to RGB24(5M*4=20M)
#define RAW_EXPAND_IMG_ADDR	0x44000000	// Raw10, Raw12 -> Expand 2 byte per pixel
#define RAW_IMG_ADDR			0x45000000	// Raw data from Camera module

#define DBG_CSIINT

#ifdef DBG_CSIINT
#define DbgCsi(x) Dbg x
#else
#define DbgCsi(x) 0;
#endif

static u32 FrameDoneCnt=0;
static u32 FrameErrorCnt=0;
static u32 uStartRTCTick;

static void __irq Isr_CsisIntSrc(void)
{
	u32 uCsisInt;
	u32 uIntStatus;
	u32 uIntMask;

	CSI_ChkIntStatus(&uIntStatus);
	CSI_GetIntMask(&uIntMask);

	uCsisInt = uIntMask&uIntStatus;

	if(uCsisInt & (0x1<<0)) {
		FrameErrorCnt++;
		DbgCsi(("Unknown ID error\n"));
	}

	if(uCsisInt & (0x1<<1)) {
		FrameErrorCnt++;
		DbgCsi(("CRC error\n"));
	}

	if(uCsisInt & (0x1<<2)) {
		FrameErrorCnt++;
		DbgCsi(("ECC error\n"));
	}

	if(uCsisInt & (0x1<<3)) {
		FrameErrorCnt++;
		DbgCsi(("Overflow error\n"));
	}

	// Datalane 0~3 SOT error
	if(uCsisInt & (0xF<<12)) {
		FrameErrorCnt++;
		DbgCsi(("Start of transmission error\n"));
	}
/*	
	if(uCsisInt & (0x1<<28))
		DbgCsi(("Embedded Data : OddAfter\n"));

	if(uCsisInt & (0x1<<29))
		DbgCsi(("Embedded Data : OddBefore\n"));

	if(uCsisInt & (0x1<<30))
		DbgCsi(("Embedded Data : EvenAfter\n"));

	if(uCsisInt & (0x1u<<31))
		DbgCsi(("Embedded Data : EvenBefore\n"));
*/
	CSI_ClearInterrupt(uCsisInt);
	INTC_ClearVectAddr();
}

static void __irq Isr_Fimd(void)
{
	u32 uIntFlag;

	LCDC_GetIntFlag(&uIntFlag);	
	Disp("[u]");
	LCDC_ClearPending(uIntFlag);	
	INTC_ClearVectAddr();	
}

static void __irq Isr_YUV422FrameDone(void)
{
	u32 uCapturedFrm;
	u32 uCurRTCTick;
	u32 uTotalSec;
	float fCurFPS;
	
	FIMC_INTR_SRC eSrc;
	FIMC_GetIntrSrc(FIMC_A, &eSrc);
	switch(eSrc)
	{
		case FIMC_SUCCESS:
			bProcessingDoneA = true;
			if(!(FrameDoneCnt++ & 0xF)) {
				uCurRTCTick = Inp32(rCURTICCNT)>>15;	// 32768 Hz (RTC)
				uTotalSec = uCurRTCTick-uStartRTCTick;
				Disp("[%.1ffps] : %d (Error : %d)\r", (float)(FrameDoneCnt/(float)uTotalSec), FrameDoneCnt, FrameErrorCnt); 
			}
			break;
		case FIFO_Y_OVF:
			bProcessingDoneA = false;
			Disp("\n FIFO Y overflow \n");
			break;
		case FIFO_CB_OVF:
			bProcessingDoneA = false;
			Disp("\n FIFO Cb overflow \n");
			break;
		case FIFO_CR_OVF:
			bProcessingDoneA = false;
			Disp("\n FIFO Cr overflow \n");
			break;
		case ROT_LINE_BUF_OVF:
			bProcessingDoneA = false;
			Disp("\n Rotation line buffer overflow \n");
			break;
		default:
			bProcessingDoneA = false;
			Assert(0);
			break;		
	}
	
	//IRQ is generated before image capturing and FrameCnt which is read in ISR, means next fraem count
	uCapturedFrm = (gBufferIdx%2)? 0 : 1;

	FIMC_ClearInterrupt(FIMC_A);
	INTC_ClearVectAddr();
	LCDC_SetActiveFrameBuf(uCapturedFrm, WIN0);

	bProcessingDoneA = true;
	gBufferIdx ++;
}

static void CSIS_SetParameter(void)
{
	u32 uParam;

	while (true)
	{
		Disp("Select Sensor output Image format[Cur_eImgFmt : ");
		switch(eCameraFmtType){
			case BayerRGB8: 
				Disp("BayerRGB8]\n");
				break;
			case BayerRGB10: 
				Disp("BayerRGB10]\n");
				break;
			case BayerRGB12: 
				Disp("BayerRGB12]\n");
				break;
			case CBYCRY: 
				Disp("CBYCRY]\n");
				break;

			default :
				Assert(0);
				break;
		}

		Disp("(0:Exit  1:BayerRGB8  2:BayerRGB10  3:BayerRGB12 4:CBYCRY[YUV422-8bit])\n");
		Disp("Select the setting value : ");
		uParam = UART_GetIntNum();

		if (uParam == 0)
			break;
		else if(uParam == 1)
			eCameraFmtType = BayerRGB8;
		else if(uParam == 2)
			eCameraFmtType = BayerRGB10;
		else if(uParam == 3)
			eCameraFmtType = BayerRGB12;
		else if(uParam == 4)
			eCameraFmtType = CBYCRY;
		else
			Assert(0);
	}
}

static void CSIS_DispTestParameter(void)
{
	Disp("=============================================\n");
	Disp("g_eCisModel\t\t: ");
	switch(eCisModel){	
		case EXTCAM:		Disp("EXTCAM\n");	break;
		case S5K4CAGX:		Disp("S5K4CAGX\n");	break;
		case S5K4E1GX:		Disp("S5K4E1GX\n");	break;
		case S5K4EAG:		Disp("S5K4EAG\n");	break;
		case S5K6AAFX:		Disp("S5K6AAFX\n");	break;
		default : Assert(0); break;
	}

	Disp("eCameraFmtType\t\t: ");
	switch(eCameraFmtType){	
		case CBYCRY:		Disp("CBYCRY\n");	break;
		case BayerRGB8:		Disp("BayerRGB8\n");	break;
		case BayerRGB10:	Disp("BayerRGB10\n");	break;
		case BayerRGB12:	Disp("BayerRGB12\n");	break;
		default : Assert(0); break;
	}

	if(eCameraFmtType != CBYCRY) {
		Disp("eCsiRawColorType\t: ");
		switch(eCsiRawColorType){	
			case CSIRaw_NoColor:		Disp("CSIRaw_NoColor\n");	break;
			case CSIRaw_oRGeGB:	Disp("CSIRaw_oRGeGB\n");	break;
			case CSIRaw_oGReBG:	Disp("CSIRaw_oGReBG\n");	break;
			case CSIRaw_oBGeGR:	Disp("CSIRaw_oBGeGR\n");	break;
			case CSIRaw_oGBeRG:	Disp("CSIRaw_oGBeRG\n");	break;
			default : Assert(0); break;
		}
	}

	Disp("eCSISDataLaneNum\t: ");
	switch(eCSISDataLaneNum){	
		case DATALANE1:		Disp("1 Lane\n");	break;
		case DATALANE2:		Disp("2 Lane\n");	break;
		case DATALANE3:		Disp("3 Lane\n");	break;
		case DATALANE4:		Disp("4 Lane\n");	break;
		default : Assert(0); break;
	}

	Disp("eCameraSize\t\t: ");
	switch(eCameraSize){	
		case USER_DEFINE_SIZE:		
			Disp("USER_DEFINE(");	
			Disp("Width:%d/Height:%d)\n", USER_DEFINE_SIZE_HSIZE, USER_DEFINE_SIZE_VSIZE);	
			break;
		case QVGA:				Disp("QVGA\n");	break;
		case VGA:				Disp("VGA\n");	break;
		case QXGA:				Disp("QXGA\n");	break;
		case HD720:				Disp("HD720\n");	break;
		case HD1080:			Disp("HD1080\n");	break;
		default : Assert(0); break;
	}
}

static void CSIS_TestCSIS2FIMC_YUV422(void)
{
	u32 uOutImgAddr1 = CODEC_MEM_ST;
	u32 uOutImgAddr2 = CODEC_MEM_ST+(CODEC_MEM_LMT-CODEC_MEM_ST)/2;
	
	u32 uCamSrcHsize;
	u32 uCamSrcVsize;
	u32 uDoubleBufAddr;
	u32 uBufferIdx = 0;
	
	Assert(eCameraFmtType == CBYCRY);
	bProcessingDoneA = false;
	FrameDoneCnt=0;
	FrameErrorCnt=0;
	
	INTC_Init();
	INTC_SetVectAddr(NUM_MIPI_CSI,Isr_CsisIntSrc);
	INTC_SetVectAddr(NUM_FIMC_A,Isr_YUV422FrameDone);
	INTC_Enable(NUM_MIPI_CSI);
	INTC_Enable(NUM_FIMC_A);	

	MIPICAM_Prepare(eCisModel, eCameraFmtType, eCameraSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eCameraFmtType, eCameraSize, eMIPIDataAlign, eWCLKSrc, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(eCisModel, eCameraFmtType, eCameraSize);

	FIMC_InitCamera(eCisModel, eCameraFmtType, eCameraSize);
	FIMC_GetFinImgSize(&uCamSrcHsize,    &uCamSrcVsize);
	FIMC_InitIpForMFinDout(FIMC_A,
		0, 0, uCamSrcHsize, uCamSrcVsize,
	#if 1
		320, 240,
		0, 0, 320, 240, 0,
	#else
		640, 480,
		0, 0, 640, 480, 0,
	#endif
		uOutImgAddr1, uOutImgAddr1, eCameraFmtType, 
		ROT_NO_FLIP_NO, CSC_ITU709, YCBCR_NARROW,
		eMIPIDataAlign); 

	FIMC_StartProcessing(FIMC_A, false, 0, 0, false, 0);			

	while(!bProcessingDoneA);
	bProcessingDoneA = false;

	uStartRTCTick = Inp32(rCURTICCNT)>>15;	// 32768 Hz (RTC)

	Disp("Output Address : 0x%8X, 0x%8X\n", uOutImgAddr1, uOutImgAddr2);
	Disp("Enter any key to end it!\n");
	UART_Getc();

	bProcessingDoneA = false;
	FIMC_PreProcessForLastFrmCpt(FIMC_A);
	while(!bProcessingDoneA);
	FIMC_StopProcessing(FIMC_A);

	CSI_DisableCSI();
	MIPICAM_Disable();	
}

// [Test MIPI(BayerRGB)-FIFO-in DMA-out for scaler bypass and Convert BayerRGB->RGB ]
static void CSIS_TestCSIS2FIMC_BayerRGB(void)
{
	u32 uLcdHsz, uLcdVsz;
	u32 uOutImgAddr1 = CODEC_MEM_ST;

	u8 inputChar;
	u32 uFinImgHsz, uFinImgVsz;
	u32 OutputImgHSz, OutputImgVSz;	
	
	Assert((eCameraFmtType == BayerRGB8) ||	(eCameraFmtType == BayerRGB10) || (eCameraFmtType == BayerRGB12));
	bProcessingDoneA = false;
	FrameDoneCnt=0;
	FrameErrorCnt=0;
	
	INTC_Init();
	INTC_SetVectAddr(NUM_MIPI_CSI,Isr_CsisIntSrc);
	INTC_SetVectAddr(NUM_FIMC_A,Isr_YUV422FrameDone);
	INTC_Enable(NUM_MIPI_CSI);
	INTC_Enable(NUM_FIMC_A);	

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(RGB24, uOutImgAddr1, WIN0, false);	
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	LCDC_SetOutputPath(LCD_TVRGB);
	LCDC_Start();	

	memset((void *)uOutImgAddr1, 0, uLcdHsz*uLcdVsz*4);

	MIPICAM_Prepare(eCisModel, eCameraFmtType, eCameraSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eCameraFmtType, eCameraSize, eMIPIDataAlign, eWCLKSrc, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(eCisModel, eCameraFmtType, eCameraSize);

	Disp("\n Sensor initialization! please wait.......... \n");
	FIMC_InitCamera(eCisModel, eCameraFmtType, eCameraSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	Raw2RGBInit((u8*)RAW_EXPAND_IMG_ADDR, uFinImgHsz, uFinImgVsz, eCameraFmtType);

	while(1) {
		FIMC_InitIpForMFinDout(FIMC_A,
			0, 0, uFinImgHsz, uFinImgVsz,
			uFinImgHsz, uFinImgVsz,
			0, 0, uFinImgHsz, uFinImgVsz, 0,
			RAW_IMG_ADDR, RAW_IMG_ADDR, eCameraFmtType,
			ROT_NO_FLIP_NO, CSC_ITU709, YCBCR_NARROW, 
			eMIPIDataAlign);

		// ============ Notice !!! ============
		Disp("[FIMC]ScalerByPass\n");
		FIMC_StartProcessingForScalerBypass(FIMC_A);
		// ================================

		while(!bProcessingDoneA);
		bProcessingDoneA = false;

		uStartRTCTick = Inp32(rCURTICCNT)>>15;	// 32768 Hz (RTC)

		Disp("Output Address : 0x%8X\n", uOutImgAddr1);
		Disp("Press any key to capture image![q:exit]\n");

		inputChar = UART_Getc();

		bProcessingDoneA = false;
		FIMC_PreProcessForLastFrmCpt(FIMC_A);
		while(!bProcessingDoneA);
		FIMC_StopProcessing(FIMC_A);

		if(inputChar == 'q')
			break;

		Disp("Start convert BayerRGB -> RGB\n");

		// Expand bayer pixel size to 2 byte(10bit -> 2 byte, 12bit -> 2 byte for convenience)
		ExpandRawPixel((u8*)RAW_IMG_ADDR, (u16*)RAW_EXPAND_IMG_ADDR, uFinImgHsz*uFinImgVsz, eCameraFmtType);

		// Convert BayerRGB->RGB
		ConvertRawToRGB((u8*)RAW_EXPAND_IMG_ADDR, (u32*)RGB_IMG_ADDR, uFinImgHsz ,uFinImgVsz, eCameraFmtType, eCsiRawColorType);

		// Resize RGB
		// Rotation (Assume image's width is longer then height)
		Disp("[FIMC]Rotation\n");
		if(uFinImgHsz < uLcdVsz)
			OutputImgHSz = uFinImgHsz;
		else
			OutputImgHSz = uLcdVsz;

		if(uFinImgVsz < uLcdHsz)
			OutputImgVSz = uFinImgVsz;
		else
			OutputImgVSz = uLcdHsz;

		FIMC_InitIpForDinDout(FIMC_A,
			uFinImgHsz, uFinImgVsz, 0,
			0, 0, uFinImgHsz, uFinImgVsz,
			RGB_IMG_ADDR, RGB24,
		#if 1
			uLcdHsz, uLcdVsz,
			0, 0, OutputImgHSz, OutputImgVSz, 0, // Scaler	
			uOutImgAddr1, RGB24,		
			ROT_90_FLIP_NO, CSC_ITU709, YCBCR_NARROW, false);			
		#else
			320, 240,
			0, 0, 320, 240, 0, // Scaler	
			uOutImgAddr1, RGB24,		
			ROT_NO_FLIP_NO, CSC_ITU709, YCBCR_NARROW, false);			
		#endif
		bProcessingDoneA = false;
		FIMC_StartProcessing(FIMC_A, false, 0, 0, false, 0);	
		while(!bProcessingDoneA);
		FIMC_StopProcessing(FIMC_A);

		Disp("Convert Raw to RGB Done.\n");
	}

	CSI_DisableCSI();
	MIPICAM_Disable();	
}


// [Test MIPI-FIFO-in DMA-out for scaler bypass]
static void CSIS_TestCSIS2FIMC_ScalerBypass(void)
{
	u32 uLcdHsz, uLcdVsz;
	u32 uOutImgAddr1 = CODEC_MEM_ST;	
	u32 uFinImgHsz, uFinImgVsz;
	
	Assert((eCameraFmtType == CBYCRY) || (eCameraFmtType == BayerRGB8) ||
		(eCameraFmtType == BayerRGB10) || (eCameraFmtType == BayerRGB12));
	bProcessingDoneA = false;
	FrameDoneCnt=0;
	FrameErrorCnt=0;

	// [Scaler bypass]
	//--------------
	Disp("\n Scaler bypass \n");	
	
	INTC_Init();
	INTC_SetVectAddr(NUM_MIPI_CSI,Isr_CsisIntSrc);
	INTC_SetVectAddr(NUM_FIMC_A,Isr_YUV422FrameDone);
	INTC_Enable(NUM_MIPI_CSI);
	INTC_Enable(NUM_FIMC_A);	

	MIPICAM_Prepare(eCisModel, eCameraFmtType, eCameraSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eCameraFmtType, eCameraSize, eMIPIDataAlign, eWCLKSrc, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(eCisModel, eCameraFmtType, eCameraSize);

	Disp("\n Sensor initialization! please wait.......... \n");
	FIMC_InitCamera(eCisModel, eCameraFmtType, eCameraSize);
	FIMC_GetFinImgSize(&uFinImgHsz, &uFinImgVsz);

	FIMC_InitIpForMFinDout(FIMC_A,
		0, 0, uFinImgHsz, uFinImgVsz,
		uFinImgHsz, uFinImgVsz,
		0, 0, uFinImgHsz, uFinImgVsz, 0,
		uOutImgAddr1, uOutImgAddr1, eCameraFmtType,
		ROT_NO_FLIP_NO, CSC_ITU709, YCBCR_NARROW, 
		eMIPIDataAlign);

	// ============ Notice !!! ============
	Disp("[FIMC]ScalerByPass\n");
	FIMC_StartProcessingForScalerBypass(FIMC_A);
	// ================================

	while(!bProcessingDoneA);
	bProcessingDoneA = false;

	uStartRTCTick = Inp32(rCURTICCNT)>>15;	// 32768 Hz (RTC)

	Disp("Output Address : 0x%8X\n", uOutImgAddr1);
	Disp("Enter any key to end it!\n");
	UART_Getc();

	bProcessingDoneA = false;
	FIMC_PreProcessForLastFrmCpt(FIMC_A);
	while(!bProcessingDoneA);
	FIMC_StopProcessing(FIMC_A);

	CSI_DisableCSI();
	MIPICAM_Disable();	
}

static void CSIS_TestCSIS2FIMD_YUV422(void)
{
	u32 uLcdHsz, uLcdVsz;
	u32 uOutImgAddr1 = CODEC_MEM_ST;
	u32 uOutImgAddr2 = CODEC_MEM_ST+(CODEC_MEM_LMT-CODEC_MEM_ST)/2;
	
	u32 uCamSrcHsize;
	u32 uCamSrcVsize;
	u32 OutputImgHSz, OutputImgVSz;	
	u32 uDoubleBufAddr;
	u32 uBufferIdx = 0;
	
	Assert(eCameraFmtType == CBYCRY);
	bProcessingDoneA = false;
	FrameDoneCnt=0;
	FrameErrorCnt=0;
	
	INTC_Init();
	INTC_SetVectAddr(NUM_MIPI_CSI,Isr_CsisIntSrc);
	INTC_SetVectAddr(NUM_FIMC_A,Isr_YUV422FrameDone);
	INTC_SetVectAddr(NUM_LCD0, Isr_Fimd);
	INTC_Enable(NUM_MIPI_CSI);
	INTC_Enable(NUM_FIMC_A);	
	INTC_Enable(NUM_LCD0);

	// Initialize LCDC v6.0
	LCDC_InitLdi(MAIN);
	LCDC_InitIp(RGB24, uOutImgAddr1, WIN0, false);	
	LCDC_GetLcdSz(&uLcdHsz, &uLcdVsz);
	LCDC_SetOutputPath(LCD_TVRGB);
	LCDC_Start();	

	memset((void *)uOutImgAddr1, 0, uLcdHsz*uLcdVsz*4);

	MIPICAM_Prepare(eCisModel, eCameraFmtType, eCameraSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eCameraFmtType, eCameraSize, eMIPIDataAlign, eWCLKSrc, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(eCisModel, eCameraFmtType, eCameraSize);
	
	FIMC_InitCamera(eCisModel, eCameraFmtType, eCameraSize);
	FIMC_GetFinImgSize(&uCamSrcHsize,    &uCamSrcVsize);

#if 1
	// No Rotation
	Disp("[FIMC]No Rotation\n");	
	if(uCamSrcHsize < uLcdHsz)
		OutputImgHSz = uCamSrcHsize;
	else
		OutputImgHSz = uLcdHsz;

	if(uCamSrcVsize < uLcdVsz)
		OutputImgVSz = uCamSrcVsize;
	else
		OutputImgVSz = uLcdVsz;
		
	FIMC_InitIpForMFinDout(FIMC_A,
		0, 0, uCamSrcHsize, uCamSrcVsize,
		uLcdHsz, uLcdVsz,
		0, 0, OutputImgHSz, OutputImgVSz, 0,
		uOutImgAddr1, uOutImgAddr2, RGB24,
		ROT_NO_FLIP_NO, CSC_ITU709, YCBCR_NARROW,
		eMIPIDataAlign); 
#elif (1)
	// Rotation (Assume image's width is longer then height)
	Disp("[FIMC]Rotation\n");
	if(uCamSrcHsize < uLcdVsz)
		OutputImgHSz = uCamSrcHsize;
	else
		OutputImgHSz = uLcdVsz;

	if(uCamSrcVsize < uLcdHsz)
		OutputImgVSz = uCamSrcVsize;
	else
		OutputImgVSz = uLcdHsz;

	FIMC_InitIpForMFinDout(FIMC_A,
		0, 0, uCamSrcHsize, uCamSrcVsize,
		uLcdHsz, uLcdVsz,
		0, 0, OutputImgHSz, OutputImgVSz, 0,
		uOutImgAddr1, uOutImgAddr2, RGB24,
		ROT_90_FLIP_NO, CSC_ITU709, YCBCR_NARROW,
		eMIPIDataAlign); 		
#endif

	FIMC_StartProcessing(FIMC_A, false, 0, 0, false, 0);			

	while(!bProcessingDoneA);
	bProcessingDoneA = false;

	uStartRTCTick = Inp32(rCURTICCNT)>>15;	// 32768 Hz (RTC)

	Disp("Output Address : 0x%8X, 0x%8X\n", uOutImgAddr1, uOutImgAddr2);
	Disp("Enter any key to end it!\n");
	UART_Getc();

	bProcessingDoneA = false;
	FIMC_PreProcessForLastFrmCpt(FIMC_A);
	while(!bProcessingDoneA);
	FIMC_StopProcessing(FIMC_A);

	CSI_DisableCSI();
	MIPICAM_Disable();
}

extern DDI_MODEL g_eDSIMMainDisp;
static u8 bInterrtupEnable = false;
#define PRESS_ANY(msg)		UART_Printf(msg);UART_Getc();

static void CSIS_TestCSIS2DSIM_YUV422(void)
{
	static u8 uDSIPll=0;

	DDI_Spec* pMainDisp = DDI_GetSpec(g_eDSIMMainDisp);
	DSIM_Infor* pDSIInfor = DSIM_Create(DSIM_DefaultLinkAndPhy(), bInterrtupEnable);
	DCS_Infor* pDCSInfor = DCS_Create(Link_DSIM, (u32)pDSIInfor, pMainDisp->m_uParaMeter[DSI_VirtualChID]);
	DDI_Handler* pDDI_Handler  = DDI_Create(g_eDSIMMainDisp, Link_DSIM, (u32)pDSIInfor,(u32)pDCSInfor, 0);
	
	u32 uLcdHsz, uLcdVsz;
	u32 uOutImgAddr1 = CODEC_MEM_ST;
	u32 uOutImgAddr2 = CODEC_MEM_ST+(CODEC_MEM_LMT-CODEC_MEM_ST)/2;
	
	u32 uCamSrcHsize;
	u32 uCamSrcVsize;
	u32 OutputImgHSz, OutputImgVSz;	
	u32 uDoubleBufAddr;
	u32 uBufferIdx = 0;
	
	Assert(eCameraFmtType == CBYCRY);
	bProcessingDoneA = false;
	FrameDoneCnt=0;
	FrameErrorCnt=0;
	
	INTC_Init();
	INTC_SetVectAddr(NUM_MIPI_CSI,Isr_CsisIntSrc);
	INTC_SetVectAddr(NUM_FIMC_A,Isr_YUV422FrameDone);
	INTC_SetVectAddr(NUM_LCD0, Isr_Fimd);
	INTC_Enable(NUM_MIPI_CSI);
	INTC_Enable(NUM_FIMC_A);	
	INTC_Enable(NUM_LCD0);

	// Initialize LCDC v6.0
	LCDC_InitIp(RGB24, uOutImgAddr1, WIN0, false);
	DSIM_InitLink(pDSIInfor);
	DSIM_SetDisplayMode(pDSIInfor, pMainDisp, NULL);
	LCDC_Start();			
	DSIM_SetHSEn(pDSIInfor);
	DSIM_SetDataTransferMode(pDSIInfor, Transfer_ByCPU, false);
	pDDI_Handler->m_fnInit(0,0);
	DSIM_EnBackLite(1);		
	
	uLcdHsz = pDDI_Handler->m_pDDISpec->m_00uHoriSize;
	uLcdVsz = pDDI_Handler->m_pDDISpec->m_01uVertiSize;
	memset((void *)uOutImgAddr1, 0, uLcdHsz*uLcdVsz*4);

	MIPICAM_Prepare(eCisModel, eCameraFmtType, eCameraSize, eCSISDataLaneNum);
	CSI_Start(eCSISDataLaneNum, eCameraFmtType, eCameraSize, eMIPIDataAlign, eWCLKSrc, &eCSISIntvSetting, eHSsettle);
	MIPICAM_Start(eCisModel, eCameraFmtType, eCameraSize);
	
	FIMC_InitCamera(eCisModel, eCameraFmtType, eCameraSize);
	FIMC_GetFinImgSize(&uCamSrcHsize,    &uCamSrcVsize);

#if 1
	// No Rotation
	Disp("[FIMC]No Rotation\n");	
	if(uCamSrcHsize < uLcdHsz)
		OutputImgHSz = uCamSrcHsize;
	else
		OutputImgHSz = uLcdHsz;

	if(uCamSrcVsize < uLcdVsz)
		OutputImgVSz = uCamSrcVsize;
	else
		OutputImgVSz = uLcdVsz;
		
	FIMC_InitIpForMFinDout(FIMC_A,
		0, 0, uCamSrcHsize, uCamSrcVsize,
		uLcdHsz, uLcdVsz,
		0, 0, OutputImgHSz, OutputImgVSz, 0,
//		uOutImgAddr1, uOutImgAddr2, RGB24,
		uOutImgAddr1, uOutImgAddr1, RGB24,
		ROT_NO_FLIP_NO, CSC_ITU709, YCBCR_NARROW,
		eMIPIDataAlign); 
#elif (1)
	// Rotation (Assume image's width is longer then height)
	Disp("[FIMC]Rotation\n");
	if(uCamSrcHsize < uLcdVsz)
		OutputImgHSz = uCamSrcHsize;
	else
		OutputImgHSz = uLcdVsz;

	if(uCamSrcVsize < uLcdHsz)
		OutputImgVSz = uCamSrcVsize;
	else
		OutputImgVSz = uLcdHsz;

	FIMC_InitIpForMFinDout(FIMC_A,
		0, 0, uCamSrcHsize, uCamSrcVsize,
		uLcdHsz, uLcdVsz,
		0, 0, OutputImgHSz, OutputImgVSz, 0,
		uOutImgAddr1, uOutImgAddr2, RGB24,
		ROT_90_FLIP_NO, CSC_ITU709, YCBCR_NARROW,
		eMIPIDataAlign); 		
#endif

	FIMC_StartProcessing(FIMC_A, false, 0, 0, false, 0);			

	while(!bProcessingDoneA);
	bProcessingDoneA = false;

	uStartRTCTick = Inp32(rCURTICCNT)>>15;	// 32768 Hz (RTC)

	Disp("Output Address : 0x%8X, 0x%8X\n", uOutImgAddr1, uOutImgAddr2);
	Disp("Enter any key to end it!\n");
	UART_Getc();
	Disp("Start spread spectrum!\n");


#if 1
	GPIO_SetFunctionEach(eGPIO_B, eGPIO_0, eGPO); // XEINT3
	GPIO_SetPullUpDownEach(eGPIO_B, eGPIO_0, eGPUDdis); // XEINT3
//	DSIM_IsLaneState(Lane_DataAll)

	while(1) {
//		UART_Getc();
//		Delay(100);
		if(uDSIPll == 0) {
//			Disp("[PMS]4,101,1\n");

#if 1
			while(1) {
//				if((*((u32*)0xFA500000)&0xf) == 0xf)
			}
			while(1) {
//				if((*((u32*)0xFA500000)&0xf) == 0xc)
					break;
			}
#endif			

			DSIM_PllFreq(4, 101, 1);
			GPIO_SetDataEach(eGPIO_B, eGPIO_0, 0);
			uDSIPll = 1;
		}
		else if(uDSIPll == 1) {
//			Disp("[PMS]4,100,1\n");
#if 1
			while(1) {
//				if((*((u32*)0xFA500000)&0xf) == 0xf)
					break;
			}
			while(1) {
//				if((*((u32*)0xFA500000)&0xf) == 0xc)
					break;
			}
#endif
			DSIM_PllFreq(4, 100, 1);
			GPIO_SetDataEach(eGPIO_B, eGPIO_0, 1);
			uDSIPll = 0;
		}
	}
#endif	

	UART_Getc();

	bProcessingDoneA = false;
	FIMC_PreProcessForLastFrmCpt(FIMC_A);
	while(!bProcessingDoneA);
	FIMC_StopProcessing(FIMC_A);

	CSI_DisableCSI();
	MIPICAM_Disable();

	pDDI_Handler->m_fnDown(0,0);
	LCDC_Stop();	
	DDI_Destory(pDDI_Handler, 0,0,0);
	DCS_Destory(pDCSInfor);
	DSIM_Destory(pDSIInfor);
}

void Test_MIPICSI(void)
{
	int sel, i;

	const testFuncMenu menu[]=
	{
		0,                   						"Exit",
		CSIS_SetParameter,						"Set CSIS Parameter",
		CSIS_TestCSIS2FIMC_YUV422,			"Test MIPI YUV422(CSI->FIMC)",
		CSIS_TestCSIS2FIMC_BayerRGB,			"Test MIPI BayerRGB(Capture Display)",
		CSIS_TestCSIS2FIMC_ScalerBypass,		"Test MIPI Data YUV422, Raw8/10/12(CSI-> FIMC, ByPass)",
		CSIS_TestCSIS2FIMD_YUV422,			"Test Display MIPI YUV422(CSI->FIMC->FIMD(LCD))",
		CSIS_TestCSIS2DSIM_YUV422,			"Test Display MIPI YUV422(CSI->FIMC->FIMD(DSI))",
		0,                      					0
	};

	GPIO_Init();

	SYSC_InitClkForMM();
	InitBaseHwForI2C();
	InitBaseHwForFIMC();
	InitBaseHwForCSIS();

	// KJ_090722 : To mesure FPS
	RTC_Enable(true);
	RTC_TickTimeEnable(true);

	SYSC_SetDispPath(eDISP_BYPASS);

	// Set initial values for global variables
#if 1
	eCisModel = S5K4EAG;
	eHSsettle = 12;
	eCameraFmtType = CBYCRY; 
	eCSISDataLaneNum = DATALANE2;
	eCameraSize = HD1080;
#elif (0)
	eCisModel = S5K6AAFX;
	eHSsettle = 6;
	eCameraFmtType = CBYCRY; 
	eCSISDataLaneNum = DATALANE1;
	eCameraSize = VGA;
#elif (0)
	eCisModel = S5K4E1GX;
	eHSsettle = 12;	
	eCameraFmtType = BayerRGB10;
	eCsiRawColorType = CSIRaw_NoColor;
	eCSISDataLaneNum = DATALANE1;
	eCameraSize = USER_DEFINE_SIZE;	
#elif (0)
	eCisModel = EXTCAM;
	eHSsettle = 2;		// 24C0
	eCameraFmtType = CBYCRY;
	eCSISDataLaneNum = DATALANE4;
	eCameraSize = VGA;		
#endif

	eWCLKSrc = WCLKSRC_PCLK;
	eMIPIDataAlign = MIPI_32BIT_ALIGN;

	eCSISIntvSetting.CSIS_INTV_Lintv = 0x0;
	eCSISIntvSetting.CSIS_INTV_Sintv = 0x0;
	eCSISIntvSetting.CSIS_INTV_Eintv = 0x0;

	CSI_Reset();
	bProcessingDoneA = false;
	
	while(1)
	{
		Disp("\n");
		for (i=0; (int)(menu[i].desc)!=0; i++)
			Disp("%2d: %s\n", i, menu[i].desc);

		CSIS_DispTestParameter();

		Disp("\nSelect the function to test : ");
		sel = UART_GetIntNum();
		Disp("\n");

		if (sel == 0)
			break;
		else if (sel>0 && sel<(sizeof(menu)/8-1))
			(menu[sel].func)();
	}
}

