/**************************************************************************************
* 
*	Project Name : S5PV210 FPGA Validation
*
*	Copyright 2008 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S3C6400.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : csi.c
*  
*	File Description : This file implements CSI-Slave functions.
*
*	Author : Yuanfeng,chen
*	Dept. : AP Development Team
*	Created Date : 
*	Version : 0.1 
* 
*	History
*	- Created(Yuanfeng,chen ????/??/??)
*	- Modified(Jun,Kim 2009/04/17)
**************************************************************************************/

#include "system.h"
#include "library.h"
#include "sysc.h"
#include "gpio.h"
#include "csi.h"

#ifdef DBG_CSI
#define	CsiOutp32(a, d) (Dbg("  Outp1w(\'h%08x, \'h%08x);\n", a, d), Outp32(a, d))
#define CsiInp32(a, d)  (Dbg("Inp1w (\'h%08x);\n", a), Inp32(a, d))
#else
#define CsiOutp32 Outp32
#define CsiInp32 Inp32
#endif

#define rMIPI_DPHY_CONTROL		0xE010E814
#define rCLK_SRC1			0xE0100204
#define rCLK_DIV1			0xE0100304

#define CSIS_BASE			MIPI_CSI_BASE

#define rCSIS_CONTROL		(CSIS_BASE+0x00)
#define rCSIS_DPHYCTRL		(CSIS_BASE+0x04)
#define rCSIS_CONFIG		(CSIS_BASE+0x08)
#define rCSIS_DPHYSTS		(CSIS_BASE+0x0c)
#define rCSIS_INTMSK		(CSIS_BASE+0x10)
#define rCSIS_INTSRC			(CSIS_BASE+0x14)
#define rCSIS_DPHYCTRL_1	(CSIS_BASE+0x24)
#define rCSIS_RESOL			(CSIS_BASE+0x2C)

#define DnDpSWAP			(1u<<31)
#define SWRESET				(1<<4)
#define CSI2_ENABLE			(1<<0)

#define	MSK_EVENBEFORE	(1<<31)
#define	MSK_EVENAFTER		(1<<30)
#define	MSK_ODDBEFORE		(1<<29)
#define	MSK_ODDAFTER		(1<<28)
#define	MSK_ERR_SOT_HS	(1<<12)
#define	MSK_ERR_ESC		(1<<8)
#define	MSK_ERR_CTRL		(1<<4)
#define	MSK_ERR_ECC		(1<<2)
#define	MSK_ERR_CRC		(1<<1)
#define	MSK_ERR_ID			(1<<0)

void CSI_S_RESETN_LP11(void)
{
#ifdef S5PV210
	// KJ_090728
	// 6AA initialization sequence 
	// LP00->DP hi-z ->LP10->LP11(>20ms)
	// D-PHY state is unknown after hi-z state.
	// So we should re-reset S_RESETN during LP11 state.
	while(((CsiInp32(rCSIS_DPHYSTS)&1) == 0));	// Wait Stop State
	CsiOutp32(rMIPI_DPHY_CONTROL, 0x1);	// S_RESETN = 0
	CsiOutp32(rMIPI_DPHY_CONTROL, 0x3);	// S_RESETN = 1
#endif
}

// [uOnOff]
//		1 : S_RESETN = High
//		0 : S_RESETN = Low
void CSI_S_RESETNEn(u8 uOnOff)
{
	u32 u32RegVal;
	
#ifdef S5PV210
	u32RegVal = CsiInp32(rMIPI_DPHY_CONTROL); 
	u32RegVal &= ~(0x2);

	u32RegVal |= (uOnOff << 1);
	CsiOutp32(rMIPI_DPHY_CONTROL, u32RegVal);	
#endif
}

void CSI_Reset(void)
{
	// User have to assert software reset when camera module is turned off.
	u32 u32RegVal;
	
	u32RegVal = CsiInp32(rCSIS_CONTROL); 
	u32RegVal |= SWRESET;
	CsiOutp32(rCSIS_CONTROL, u32RegVal);
}

void CSI_UpdateShadow(void)
{
	// User have to assert software reset when camera module is turned off.
	u32 u32RegVal;
	
	u32RegVal = CsiInp32(rCSIS_CONTROL); 
	u32RegVal |= (1 << 16);
	CsiOutp32(rCSIS_CONTROL, u32RegVal);
}

void CSI_Enable(void)
{
	u32 u32RegVal;

	u32RegVal = CsiInp32(rCSIS_CONTROL); 
	u32RegVal |= CSI2_ENABLE;
	CsiOutp32(rCSIS_CONTROL, u32RegVal);
}

void CSI_DPHYOn(CSIS_DATA_LANE_NUM eDataLane)
{
	u32 u32RegVal;

	CSI_S_RESETNEn(1);

	u32RegVal = CsiInp32(rCSIS_DPHYCTRL); 
	u32RegVal &= ~(0xF);

	if(eDataLane == DATALANE1) {
		Disp("[CSIS]Data lane 1\n");
		u32RegVal |= 0x1;
	}
	else if(eDataLane == DATALANE2) {
		Disp("[CSIS]Data lane 2\n");
		u32RegVal |= 0x3;
	}
	else if(eDataLane == DATALANE3) {
		Disp("[CSIS]Data lane 3\n");
		u32RegVal |= 0x7;
	}
	else if(eDataLane == DATALANE4) {
		Disp("[CSIS]Data lane 4\n");
		u32RegVal |= 0xF;
	}

	CsiOutp32(rCSIS_DPHYCTRL, u32RegVal); // D-Phy On.

}

void CSI_DPHYOff(void)
{
	u32 u32RegVal;

	CSI_S_RESETNEn(0);
	
	u32RegVal = CsiInp32(rCSIS_DPHYCTRL); 
	u32RegVal &= ~(0xF);
	CsiOutp32(rCSIS_DPHYCTRL, u32RegVal); // D-Phy Off.	
}

void CSI_SetDataAlign(MIPI_DATA_ALIGN eDataAlign)
{
	u32 u32RegVal;

	u32RegVal = CsiInp32(rCSIS_CONTROL);

	if(eDataAlign == MIPI_24BIT_ALIGN) {
		Disp("[CSIS]24bits align\n");
		u32RegVal &= ~(0x1 << 20);
	}
	else if(eDataAlign == MIPI_32BIT_ALIGN) {
		Disp("[CSIS]32bits align\n");
		u32RegVal |= (0x1 << 20);
	}
	else
		Assert(0);

	CsiOutp32(rCSIS_CONTROL, u32RegVal);
}

void CSI_SetWCLKSrc(CSIS_WCLK_SRC WCLK_Src)
{
	u32 u32RegVal;

	u32RegVal = CsiInp32(rCSIS_CONTROL); 

	if(WCLK_Src == WCLKSRC_PCLK) {
		Disp("[CSIS]Set Pclk\n");
		u32RegVal &= ~(0x1 << 8);
	}
	else if(WCLK_Src == WCLKSRC_EXTCLK) {
		Disp("[CSIS]Set ExtClk\n");
		u32RegVal |= (0x1 << 8);
	}
	else
		Assert(0);

	CsiOutp32(rCSIS_CONTROL, u32RegVal);	
}

void CSI_SetImgFmt(IMG_FMT eFinImgType)
{
	u32 u32RegVal;

	u32RegVal = CsiInp32(rCSIS_CONFIG);
	u32RegVal &= ~(0xFC);

	if(eFinImgType == CBYCRY) {
		Disp("[CSIS]CBYCRY\n");	
		u32RegVal |= (0x1E << 2);
	}
	else if(eFinImgType == BayerRGB8) {
		Disp("[CSIS]BayerRGB8\n");
		u32RegVal |= (0x2A << 2);
	}
	else if(eFinImgType == BayerRGB10) {
		Disp("[CSIS]BayerRGB10\n");
		u32RegVal |= (0x2B << 2);
	}
	else if(eFinImgType == BayerRGB12) {
		Disp("[CSIS]BayerRGB12\n");
		u32RegVal |= (0x2C << 2);
	}

	CsiOutp32(rCSIS_CONFIG, u32RegVal);
}

void CSI_SetResolution(IMG_RESOLUTION eSize)
{
	u32 u32RegVal;

	switch(eSize)
	{
	case VGA:
		u32RegVal = (VGA_HSIZE << 16)|(VGA_VSIZE << 0);
		break;
	case WVGA:
		u32RegVal = (WVGA_HSIZE << 16)|(WVGA_VSIZE << 0);
		break;
	case WSVGA:
		u32RegVal = (WSVGA_HSIZE << 16)|(WSVGA_VSIZE << 0);
		break;		
	case HD720:
		u32RegVal = (HD720_HSIZE << 16)|(HD720_VSIZE << 0);
		break;
	case HD1080:
		u32RegVal = (HD1080_HSIZE << 16)|(HD1080_VSIZE << 0);
		break;
	case QXGA:
		u32RegVal = (QXGA_HSIZE << 16)|(QXGA_VSIZE << 0);
		break;
	case USER_DEFINE_SIZE:
		u32RegVal = (USER_DEFINE_SIZE_HSIZE << 16)|(USER_DEFINE_SIZE_VSIZE << 0);
		break;		
	default :
		Assert(0);
	}

	Disp("[CSIS]Size : %d*%d\n", (u32RegVal & 0xFFFF0000) >> 16, u32RegVal & 0xFFFF);
	CsiOutp32(rCSIS_RESOL, u32RegVal);
}

void CSI_SetWrapperTiming(CSIS_INTV_SETTING *eCSISIntvSetting)
{
	u32 u32RegVal;

	u32RegVal = CsiInp32(rCSIS_CONFIG);
	u32RegVal &= ~(0xFFFFFF00);

	Disp("[CSIS]Lintv : %d, Sintv: %d, Eintv : %d\n", 
		eCSISIntvSetting->CSIS_INTV_Lintv, 
		eCSISIntvSetting->CSIS_INTV_Sintv, 
		eCSISIntvSetting->CSIS_INTV_Eintv);
	u32RegVal |= (eCSISIntvSetting->CSIS_INTV_Lintv << 26);
	u32RegVal |= (eCSISIntvSetting->CSIS_INTV_Sintv << 20);
	u32RegVal |= (eCSISIntvSetting->CSIS_INTV_Eintv << 8);

	CsiOutp32(rCSIS_CONFIG, u32RegVal);
}

void CSI_SwapDnDp(void)
{
	u32 u32RegVal;

	u32RegVal=CsiInp32(rCSIS_CONTROL);
	u32RegVal |= (1u<<31);
	CsiOutp32(rCSIS_CONTROL, u32RegVal);
}

void CSI_SetNumOfDataLane(CSIS_DATA_LANE_NUM eDataLane)
{
	u32 u32RegVal;

	u32RegVal = CsiInp32(rCSIS_CONFIG);
	u32RegVal &= ~(0x3);

	if(eDataLane == DATALANE1)
		u32RegVal |= 0x0;
	else if(eDataLane == DATALANE2)
		u32RegVal |= 0x1;
	else if(eDataLane == DATALANE3)
		u32RegVal |= 0x2;
	else if(eDataLane == DATALANE4)
		u32RegVal |= 0x3;

	CsiOutp32(rCSIS_CONFIG, u32RegVal);
}

void CSI_SetHSsettle(u8 eHSsettle)
{
	u32 u32RegVal;

	Disp("[CSIS]HSsettle : %d\n", eHSsettle);
	u32RegVal = CsiInp32(rCSIS_DPHYCTRL);
	u32RegVal &= ~(0xF8000000);
	u32RegVal |= (eHSsettle<<27);
	CsiOutp32(rCSIS_DPHYCTRL, u32RegVal);
}

void CSI_SetClockLaneDelay(u8 uValue)
{
	u32 u32RegVal;

	if(uValue == 0x0)
		Disp("[CSIS]ClockLaneDelay : No Delay\n");
	else if(uValue == 0x1)
		Disp("[CSIS]ClockLaneDelay : 100ps Delay\n");
	else if(uValue == 0x2)
		Disp("[CSIS]ClockLaneDelay : 200ps Delay\n");
	else if(uValue == 0x3)
		Disp("[CSIS]ClockLaneDelay : 300ps Delay\n");
		
	u32RegVal = CsiInp32(rCSIS_DPHYCTRL_1);
	u32RegVal &= ~(0xC);
	u32RegVal |= (uValue<<2);
	CsiOutp32(rCSIS_DPHYCTRL_1, u32RegVal);
}

void CSI_EnableInterrupt(void)
{
	u32 u32RegVal;
	
	u32RegVal = 0x0;
	u32RegVal |= (0xfu<<28)|(1<<12)|(0xfu<<0);

	CsiOutp32(rCSIS_INTMSK, u32RegVal);
}

void CSI_DisableInterrupt(void)
{
	u32 u32RegVal;
	
	u32RegVal = 0x0;
	CsiOutp32(rCSIS_INTMSK, u32RegVal);
}

void CSI_ChkIntStatus(u32* eIntsec)
{
	*eIntsec=CsiInp32(rCSIS_INTSRC);
}

void CSI_ClearInterrupt(u32 eIntsec)
{
	CsiOutp32(rCSIS_INTSRC, eIntsec);
}

void CSI_GetIntMask(u32* uIntMask)
{
	*uIntMask=CsiInp32(rCSIS_INTMSK);
}

void CSI_DisableCSI(void)
{
	u32 u32RegVal;
	
	CsiOutp32(rCSIS_CONTROL, 0x0);
	CSI_DPHYOff();
}

void CSI_Start(CSIS_DATA_LANE_NUM eDataLane, IMG_FMT eFinImgType,
				IMG_RESOLUTION eSize, MIPI_DATA_ALIGN eDataAlign, 
				CSIS_WCLK_SRC eWCLKSrc, CSIS_INTV_SETTING *eCSISIntvSetting, 
				u8 eHSsettle)
{
	CSI_Reset();

	CSI_SetHSsettle(eHSsettle);
	CSI_SetClockLaneDelay(0x00);
	CSI_SetNumOfDataLane(eDataLane);
	CSI_SetDataAlign(eDataAlign);
	CSI_SetWCLKSrc(eWCLKSrc);
	CSI_SetImgFmt(eFinImgType);
	CSI_SetResolution(eSize);
	CSI_SetWrapperTiming(eCSISIntvSetting);

	CSI_UpdateShadow();

	CSI_EnableInterrupt();

	CSI_DPHYOn(eDataLane);
	CSI_Enable();
}

#define CSI_EXTCLK_XusbXTI		0x1
#define CSI_EXTCLK_SCLKMPLL	0x6
#define CSI_EXTCLK_SCLKEPLL		0x7

// [Initialize base H/W for CSIS]
void InitBaseHwForCSIS(void)
{
#ifdef S5PV210
	u32 uRegVal;

	// KJ_CHECK : I think that external clock is not neccesary in silicon.
	// XusbXTI is used by CSIS wrapper clk.
	// Because XXTI(PCLK) is so slow in FPGA environment. (Overflow Error is occured)
	uRegVal = Inp32(rCLK_SRC1); 
	uRegVal = uRegVal |(CSI_EXTCLK_SCLKMPLL<<24);		// 660
	CsiOutp32(rCLK_SRC1, uRegVal);

	uRegVal = Inp32(rCLK_DIV1); 
	uRegVal = uRegVal |(0x3<<28);		// 660/4
	CsiOutp32(rCLK_DIV1, uRegVal);

	// MIPI DPHY Power enable
	CsiOutp32(rMIPI_DPHY_CONTROL, 0x1);
#endif
}


