/**************************************************************************************
* 
*	Project Name : S5PV210 FPGA Validation
*
*	Copyright 2008 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PV210.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : RawConverter.c
*  
*	File Description : This file implements converting BayerRGB to RGB24(For CSI-Slave functions).
*
*	Author : Jun,Kim
*	Dept. : AP Development Team
*	Created Date : 
*	Version : 0.1 
* 
*	History
*	- Created(Jun,Kim 2009/07/24)
**************************************************************************************/
#include "library.h"
#include "def.h"
#include "csi.h"

// RGB24(RGBX888LE)
#define Set8bRawToRGBValue(r,g,b)	((r<<16)|(g<<8)|(b))
#define Set10bRawToRGBValue(r,g,b)	(((r&0x3FC)<<14)|((g&0x3FC)<<6)|((b&0x3FC)>>2))
#define Set12bRawToRGBValue(r,g,b)	(((r&0xFF0)<<12)|((g&0xFF0)<<4)|((b&0xFF0)>>4))

static u8 uRawPixelSize;
static u8 *uRawSrcImgAddr;
static u32 uRawImgHSz;
static u32 uRawImgVSz;

u32 GetPixelValue(u32 PosX, u32 PosY)
{
	u32 pixelValue;

	if(uRawPixelSize == 1) {
		pixelValue = *((u8*)uRawSrcImgAddr+((uRawImgHSz*PosY)+PosX));
	}
	else if(uRawPixelSize == 2) {
		pixelValue = *((u16*)uRawSrcImgAddr+((uRawImgHSz*PosY)+PosX));
	}
	else
		Assert(0);

	return pixelValue;
}

// Get Average left, right
u32 GetAvgHorizontal(u32 PosX, u32 PosY)
{
	u32 avgValue;

	if(PosX == 0) {
		avgValue = GetPixelValue(PosX+1, PosY);
	}
	else if(PosX == (uRawImgHSz-1)) {
		avgValue = GetPixelValue(PosX-1, PosY);
	}
	else {
		avgValue = (GetPixelValue(PosX-1, PosY) + GetPixelValue(PosX+1, PosY)) >> 1;
	}

	return avgValue;
}

// Get Average up, down
u32 GetAvgVertical(u32 PosX, u32 PosY)
{
	u32 avgValue;

	if(PosY == 0) {
		avgValue = GetPixelValue(PosX, PosY+1);
	}
	else if(PosY == (uRawImgVSz-1)) {
		avgValue = GetPixelValue(PosX, PosY-1);
	}
	else {
		avgValue = (GetPixelValue(PosX, PosY-1) + GetPixelValue(PosX, PosY+1)) >> 1;
	}

	return avgValue;
}

// Get Average up, down, left, right (4 value)
u32 GetAvgAllDirection(u32 PosX, u32 PosY)
{
	u32 avgValue;

	if(PosX == 0) {
		if(PosY == 0)
			avgValue = (GetPixelValue(PosX+1, PosY) + GetPixelValue(PosX, PosY+1)) >> 1;
		else if(PosY == (uRawImgVSz-1))
			avgValue = (GetPixelValue(PosX+1, PosY) + GetPixelValue(PosX, PosY-1)) >> 1;
		else
			avgValue = (GetPixelValue(PosX, PosY+1) + GetPixelValue(PosX, PosY-1) + GetPixelValue(PosX+1, PosY))/3;
	}
	else if(PosX == (uRawImgHSz-1)) {
		if(PosY == 0)
			avgValue = (GetPixelValue(PosX-1, PosY) + GetPixelValue(PosX, PosY+1)) >> 1;
		else if(PosY == (uRawImgVSz-1))
			avgValue = (GetPixelValue(PosX-1, PosY) + GetPixelValue(PosX, PosY-1)) >> 1;
		else
			avgValue = (GetPixelValue(PosX, PosY+1) + GetPixelValue(PosX, PosY-1) + GetPixelValue(PosX-1, PosY))/3;
	}
	else {
		if(PosY == 0) {
			avgValue = (GetPixelValue(PosX-1, PosY) + GetPixelValue(PosX+1, PosY) + GetPixelValue(PosX, PosY+1))/3;
		}
		else if(PosY == (uRawImgVSz-1)) {
			avgValue = (GetPixelValue(PosX-1, PosY) + GetPixelValue(PosX+1, PosY) + GetPixelValue(PosX, PosY-1))/3;
		}
		else {
			avgValue = (GetPixelValue(PosX-1, PosY) + GetPixelValue(PosX+1, PosY) +
						GetPixelValue(PosX, PosY-1) + GetPixelValue(PosX, PosY+1)) >> 2;
		}
	}

	return avgValue;
}

// Get Average diagonal(4 value)
u32 GetAvgDiagonal(u32 PosX, u32 PosY)
{
	u32 avgValue;

	if(PosX == 0) {
		if(PosY == 0)
			avgValue = GetPixelValue(PosX+1, PosY+1);
		else if(PosY == (uRawImgVSz-1))
			avgValue = GetPixelValue(PosX+1, PosY-1);
		else
			avgValue = (GetPixelValue(PosX+1, PosY-1) + GetPixelValue(PosX+1, PosY+1)) >> 1;
	}
	else if(PosX == (uRawImgHSz-1)) {
		if(PosY == 0)
			avgValue = GetPixelValue(PosX-1, PosY+1);
		else if(PosY == (uRawImgVSz-1))
			avgValue = GetPixelValue(PosX-1, PosY-1);
		else
			avgValue = (GetPixelValue(PosX-1, PosY-1) + GetPixelValue(PosX-1, PosY+1)) >> 1;
	}
	else {
		if(PosY == 0) {
			avgValue = (GetPixelValue(PosX-1, PosY+1) + GetPixelValue(PosX+1, PosY+1)) >> 1;
		}
		else if(PosY == (uRawImgVSz-1)) {
			avgValue = (GetPixelValue(PosX-1, PosY-1) + GetPixelValue(PosX+1, PosY-1)) >> 1;
		}
		else {
			avgValue = (GetPixelValue(PosX-1, PosY-1) + GetPixelValue(PosX+1, PosY-1) +
						GetPixelValue(PosX-1, PosY+1) + GetPixelValue(PosX+1, PosY+1)) >> 2;
		}
	}

	return avgValue;
}

// No color(Achromatic color)
static void ConvertRaw8ToRGB_Fast(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;
	u8 *skipLine;

	// 3.7
	while(curPixelY < imgHeight)
	{
		// odd row(RGRGRG..)
		skipLine = (u8*)uRawSrcImgAddr+(uRawImgHSz*curPixelY);
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// R pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(GBGBGB...)
		skipLine = (u8*)uRawSrcImgAddr+(uRawImgHSz*curPixelY);
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// B pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// No color(Achromatic color)
static void ConvertRaw10ToRGB_Fast(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;
	u16 *skipLine;

	// 3.7
	while(curPixelY < imgHeight)
	{
		// odd row(RGRGRG..)
		skipLine = (u16*)uRawSrcImgAddr+(uRawImgHSz*curPixelY);
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// R pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(GBGBGB...)
		skipLine = (u16*)uRawSrcImgAddr+(uRawImgHSz*curPixelY);
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// B pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// No color(Achromatic color)
static void ConvertRaw12ToRGB_Fast(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;
	u16 *skipLine;

	// 2.97
	while(curPixelY < imgHeight)
	{
		// odd row(RGRGRG..)
		skipLine = (u16*)uRawSrcImgAddr+(uRawImgHSz*curPixelY);
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// R pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(GBGBGB...)
		skipLine = (u16*)uRawSrcImgAddr+(uRawImgHSz*curPixelY);
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// B pixel
			avgB=avgG=avgR=*(skipLine+curPixelX);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// RGRG...
// GBGB...
static void ConvertRaw8ToRGB_1(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(RGRGRG..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(GBGBGB...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// GRGR...
// BGBG...
static void ConvertRaw8ToRGB_2(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(GRGRGR..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(BGBGBG...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// BGBG...
// GRGR...
static void ConvertRaw8ToRGB_3(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(BGBGBG..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(GRGRGR...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// GBGB...
// RGRG...
static void ConvertRaw8ToRGB_4(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(GBGBGB..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(RGRGRG...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set8bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// RGRG...
// GBGB...
static void ConvertRaw10ToRGB_1(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(RGRGRG..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(GBGBGB...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// GRGR...
// BGBG...
static void ConvertRaw10ToRGB_2(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(GRGRGR..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(BGBGBG...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// BGBG...
// GRGR...
static void ConvertRaw10ToRGB_3(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(BGBGBG..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(GRGRGR...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// GBGB...
// RGRG...
static void ConvertRaw10ToRGB_4(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(GBGBGB..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(RGRGRG...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set10bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// RGRG...
// GBGB...
static void ConvertRaw12ToRGB_1(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(RGRGRG..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(GBGBGB...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// GRGR...
// BGBG...
static void ConvertRaw12ToRGB_2(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(GRGRGR..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(BGBGBG...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// BGBG...
// GRGR...
static void ConvertRaw12ToRGB_3(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(BGBGBG..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(GRGRGR...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

// GBGB...
// RGRG...
static void ConvertRaw12ToRGB_4(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight)
{
	u16 avgR, avgG, avgB;
	int curPixelX=0, curPixelY=0;
	u32 *dstRGBImgAddr=(u32*)dstImgAddr;

	while(curPixelY < imgHeight)
	{
		// odd row(GBGBGB..)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// G pixel
			avgR = GetAvgVertical(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgHorizontal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// B pixel
			avgR = GetAvgDiagonal(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetPixelValue(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;

		// even row(RGRGRG...)
		curPixelX = 0;
		while(curPixelX < imgWidth)
		{
			// R pixel
			avgR = GetPixelValue(curPixelX, curPixelY);
			avgG = GetAvgAllDirection(curPixelX, curPixelY);
			avgB = GetAvgDiagonal(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;

			// G pixel
			avgR = GetAvgHorizontal(curPixelX, curPixelY);
			avgG = GetPixelValue(curPixelX, curPixelY);
			avgB = GetAvgVertical(curPixelX, curPixelY);
			*dstRGBImgAddr++ =  Set12bRawToRGBValue(avgR, avgG, avgB);
			curPixelX++;
		}
		curPixelY++;
	}
}

void 
Raw2RGBInit(u8 *srcImgAddr, u32 imgWidth, u32 imgHeight, IMG_FMT inImgType)
{
	uRawPixelSize = (inImgType==BayerRGB8)?1:2;
	uRawSrcImgAddr = srcImgAddr;
	uRawImgHSz = imgWidth;
	uRawImgVSz = imgHeight;
}

void 
ExpandRawPixel(u8 *srcImgAddr, u16 *dstImgAddr, u32 InImgSize, IMG_FMT inImgType)
{
	u8 *curPixel;
	u8 *srcImgAddrEnd=srcImgAddr+InImgSize;

	if(inImgType == BayerRGB8) {	
		// RGB8 -> 1 Byte(There is no need expand pixel)
		memcpy(dstImgAddr, srcImgAddr, InImgSize);
	}
	else if(inImgType == BayerRGB10) {
		u8 *srcImgAddrEnd=srcImgAddr+((InImgSize*10)>>3);
		
		for(curPixel = srcImgAddr; curPixel < srcImgAddrEnd; curPixel+=5){
			*dstImgAddr++ = (curPixel[0] << 2)|(curPixel[4]&0x03);
			*dstImgAddr++ = (curPixel[1] << 2)|((curPixel[4]&0x0C) >> 2);
			*dstImgAddr++ = (curPixel[2] << 2)|((curPixel[4]&0x30) >> 4);
			*dstImgAddr++ = (curPixel[3] << 2)|((curPixel[4]&0xC0) >> 6);
		}
	}
	else if(inImgType == BayerRGB12) {
		u8 *srcImgAddrEnd=srcImgAddr+((InImgSize*12)>>3);
		
		for(curPixel = srcImgAddr; curPixel < srcImgAddrEnd; curPixel+=3){
			*dstImgAddr++ = (curPixel[0] << 4)|(curPixel[2]&0x0F);
			*dstImgAddr++ = (curPixel[1] << 4)|((curPixel[2]&0xF0) >> 4);
		}
	}
	else
		Assert(0);
}

void 
ConvertRawToRGB(u8 *srcImgAddr, void *dstImgAddr, u32 imgWidth, u32 imgHeight, 
					IMG_FMT eCameraFmtType, CSIRaw_Color eCsiRawColorType)
{
	switch(eCameraFmtType) {
		case BayerRGB8:
			switch(eCsiRawColorType) {
				case CSIRaw_NoColor:
					ConvertRaw8ToRGB_Fast((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oRGeGB:
					ConvertRaw8ToRGB_1((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oGReBG:
					ConvertRaw8ToRGB_2((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oBGeGR:
					ConvertRaw8ToRGB_3((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oGBeRG:
					ConvertRaw8ToRGB_4((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				default :
					Assert(0);
					break;
			}
			break;
		case BayerRGB10:
			switch(eCsiRawColorType) {
				case CSIRaw_NoColor:
					ConvertRaw10ToRGB_Fast((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oRGeGB:
					ConvertRaw10ToRGB_1((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oGReBG:
					ConvertRaw10ToRGB_2((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oBGeGR:
					ConvertRaw10ToRGB_3((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oGBeRG:
					ConvertRaw10ToRGB_4((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				default :
					Assert(0);
					break;
			}
			break;
		case BayerRGB12:
			switch(eCsiRawColorType) {
				case CSIRaw_NoColor:
					ConvertRaw12ToRGB_Fast((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oRGeGB:
					ConvertRaw12ToRGB_1((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oGReBG:
					ConvertRaw12ToRGB_2((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oBGeGR:
					ConvertRaw12ToRGB_3((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				case CSIRaw_oGBeRG:
					ConvertRaw12ToRGB_4((u8*)srcImgAddr, (u32*)dstImgAddr, imgWidth, imgHeight);	break;
				default :
					Assert(0);
					break;
			}
			break;			
		default:
			Assert(0);
			break;
	}
}

