#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdarg.h>


#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "assaddr.h"
#include "assclkcon.h"


////////////////////////////////////////////////////////////////////////
//
//  CLKCON Setting  for Audio sub system.
//
//  Order of Setting(first time) : does not matter.
//  Moment of Setting(run time) : Divider can changes at any time, others have limits
//  warning : 1. To reduce leakage current, gates blocks which are not used.
//            2. Set CLKCON properly for adequate usage of AudioSS. => utilizes AudioSS_CLKCON_SetTypicalConf()


////////////////////////////////////////////////////////////////////////
// This function set clock source of Audio Sub system
//
// INPUT : eCLKMUX_ASS[1:0], eMUXBUS[1:0], eMUXI2SA[2:0]
//  
//  CLKMUX select input clock, Selected clock is referred as 'Main Clk'.
//	 0 : MUX_XXTI
//	 1 : MUX_FOUT_EPLL
//
//  eMUXBUS select input clock of Audio Sub System BUS
//	 0 : Main Clk
//	 1 : IISCDCLK0
//
//  eMUXI2SA select input clock of I2S v5.1 I2SCLK
//	 0 : Main Clk
//	 1 : IISCDCLK0
//	 1 : SCLK_AUDIO0
//
//	USAGE : 
//	AudioSS_CLKCON_SetSource(FOUT_EPLL, Main_CLK, MUXI2SA_Main_CLK);
//	AudioSS_CLKCON_SetSource(MUX_XXTI, Main_CLK, MUXI2SA_Main_CLK);
//			//AudioSS bus use Main Clk(which is FOUT_EPLL)
//			//I2SCLK use Main Clk(which is FOUT_EPLL)
//
//   refer more tables at User's manual.(master / RCLK / CDCLKCON)
void AudioSS_CLKCON_SetSource(CLKMUX_ASS eCLKMUX_ASS, MUXBUS eMUXBUS, MUXI2SA eMUXI2SA)
{
//	Assert(eCLKMUX_ASS<=FOUT_EPLL && eMUXBUS<=I2S_iCLK&& eMUXI2SA<=MUXI2SA_SCLK_AUDIO0);

	Outp32( ASS_CLK_SRC,eMUXI2SA<<2|eMUXBUS<<1 | eCLKMUX_ASS);
}

////////////////////////////////////////////////////////////////////////
// This function set divider, can change at any time
//
// INPUT :  uDIVI2S_A[3:0], uDIVAUD_BUS[3:0]
//  
//  input 0~15 value mean clock source is divided by 1~16.
//  DIVI2S_A divide to 'I2S v5.1 I2SCLK'
//  DIVAUD_BUS divide to 'Audio Sub system BUS Clock'
//
//	USAGE : 
//	AudioSS_CLKCON_SetDivider(7-1,1-1);//	normal power mode (EPLL 79Mhz for AudioSS, 11.28Mhz for codec chip)
//	AudioSS_CLKCON_SetDivider(7-1,9-1);//	deep idle power mode (EPLL 8.8Mhz for AudioSS, 11.28Mhz for codec chip)
void AudioSS_CLKCON_SetDivider(unsigned char uDIVI2S_A, unsigned char uDIVAUD_BUS)
{
//	Assert(uDIVCLK<16 && uDIVI2S_A<16);
	Outp32(ASS_CLK_DIV,uDIVI2S_A<<4 | uDIVAUD_BUS);
}


////////////////////////////////////////////////////////////////////////
// This function set divider, can change at any time
//
// INPUT : uDIVCLK[3:0] , uDIVI2S_A[3:0]
//  
//  input 0~15 value mean clock source is divided by 1~16.
//
//	USAGE : 
void AudioSS_CLKCON_ChangeDiv_AudBus(unsigned char uDIVAUD_BUS)
{
	unsigned temp = Inp32( ASS_CLK_DIV);

//	Assert(uDIVAUD_BUS<16);
	temp &=~(0xf);
	Outp32( ASS_CLK_DIV, temp |  uDIVAUD_BUS);
}

////////////////////////////////////////////////////////////////////////
// This function change CLKMUX, can change at any time
//
//  CLKMUX select input clock, Selected clock is referred as 'Main Clk'.
//	 0 : MUX_XXTI
//	 1 : MUX_FOUT_EPLL
void AudioSS_CLKCON_ChangeCLKMUX(CLKMUX_ASS eCLKMUX_ASS)
{
	unsigned temp = Inp32( ASS_CLK_SRC);

//	Assert(uDIVAUD_BUS<16);
	temp &=~(0x1);
	Outp32( ASS_CLK_DIV, temp |  eCLKMUX_ASS);
}


////////////////////////////////////////////////////////////////////////
// This function set gate passing/masking of Audio Sub system by one function call.
//
// INPUT : uGateBlocks[6:0]
//
// if bit is 1, then set pass at certain location
// if bit is 0, then set mask at certain location
// [0] : HCLK_RPCORE_IMEM_DMEM (including IMEM_DMEM)
// [1] : HCLK_InOutBUF (IBUF0,1 / OBUF0,1)
// [2] : HCLK_DMAX_DMA
// [3] : HCLK_ODO_HWA
// [4] : HCLK_UART
// [5] : HCLK_I2SV5_1
// [6] : I2SCLK
//
// USAGE :
// AudioSS_CLKCON_SetGate(ALL_BLOCKS_PASS); // set all blocks pass when developing
// AudioSS_CLKCON_SetGate(ALL_BLOCKS_PASS|~(HCLK_UART_PASS)); 
//										// set running block when RP decode
// AudioSS_CLKCON_SetGate(HCLK_RPCORE_IMEM_DMEM_PASS|HCLK_I2SV5_1_PASS|I2SCLK_PASS);
//										// set running block when ARM decode
// temp = AudioSS_CLKCON_GetGate();
// AudioSS_CLKCON_SetGate(temp|~I2SCLK_PASS);
void AudioSS_CLKCON_SetGate(unsigned char uGateBlocks)
{
//	Assert(uGateBlocks<=0x7F);
	Outp32( ASS_CLK_GATE,uGateBlocks);
}

unsigned char AudioSS_CLKCON_GetGate()
{
	return Inp32( ASS_CLK_GATE);
}


////////////////////////////////////////////////////////////////////////
//Set AudioSS CLKCON sources, Gating
//
//Typical configuration(I2S Opmode : MOUT_I2SA) are as follows
// [1] Master : EPLL out <-MOUTI2SA_MainCLK_FOUTEPLL
// [2] Master : CDCLK in <- MOUTI2SA_IISCDCLK0
// [3] Slave : gating I2SCLK, not connect to CDCLK <-MOUTI2SA_GATING
//
//Notice : Audio Gating should be done before calling this function[especially for config [3] 
void AudioSS_CLKCON_SetTypicalConf(I2SOPMode eI2SOpmode, MOUTI2SA eMoutI2SA )
{
	unsigned int temp;

	if(eI2SOpmode==I2S_Master)
	{
		if(eMoutI2SA==MOUTI2SA_MainCLK_FOUTEPLL)
			AudioSS_CLKCON_SetSource(MUX_FOUT_EPLL, Main_CLK, MUXI2SA_Main_CLK);
		else	 if(eMoutI2SA==MOUTI2SA_IISCDCLK0)
			AudioSS_CLKCON_SetSource(MUX_FOUT_EPLL, Main_CLK, MUXI2SA_I2S_iCLK);
		else
			Disp("no typical cases\n");
	}else {
		if(eMoutI2SA==MOUTI2SA_GATING) 
		{
			 temp = AudioSS_CLKCON_GetGate();
			AudioSS_CLKCON_SetGate(temp|~I2SCLK_PASS);
			AudioSS_CLKCON_SetSource(MUX_FOUT_EPLL, Main_CLK, MUXI2SA_Main_CLK);
		}else
			Disp("no typical cases\n");
	}

}

